using DataContracts;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Runtime.CompilerServices;
using System.Text.RegularExpressions;

namespace Ato.EN.IntegrationServices.CodeGenerationTRT
{

    public partial class TRT2024Validator
    {
        /// <summary>
        /// The max parameter name length is restricted by the SBR1 and SBR2 schemas - the lowest common denominator of SBR 1 is used as the default
        /// </summary>
        private int _maxParameterNameLength;

        /// <summary>
        /// The max parameter value length is restricted by the SBR1 and SBR2 schemas
        /// </summary>
        private int _maxParameterValueLength;

        /// <summary>
        /// The SBR1 and ebms3 schemas do not allow parameter names or values to be the empty string
        /// </summary>
        private string _emptyParameterValue;

        /// <summary>
        /// The report parameter
        /// </summary>
        private TRT2024 report;

        /// <summary>
        /// The response parameter
        /// </summary>
        public List<ProcessMessageDocument> response {get; private set;}

        /// <summary>
        /// Initializes a new instance of the <see cref="TRT2024Validator" /> class.
        /// </summary>
        /// <param name="maxParameterNameLength">Maximum length of the parameter name.</param>
        /// <param name="maxParameterValueLength">Maximum length of the parameter value.</param>
        /// <param name="emptyParameterValue">This value will be used in place of any parameter values that result in a null or empty value.</param>
        public TRT2024Validator(int maxParameterNameLength = 20, int maxParameterValueLength = 4096, string emptyParameterValue = "EMPTY")
        {
            _maxParameterNameLength = maxParameterNameLength;
            _maxParameterValueLength = maxParameterValueLength;
            _emptyParameterValue = emptyParameterValue;
            this.response = new List<ProcessMessageDocument>();
        }

        /// <summary>
        /// Initializes a new instance of the <see cref="TRT2024Validator" /> class.
        /// </summary>
        /// <param name="report">Report variable </param>                
        /// <param name="maxParameterNameLength">Maximum length of the parameter name.</param>
        /// <param name="maxParameterValueLength">Maximum length of the parameter value.</param>
        /// <param name="emptyParameterValue">This value will be used in place of any parameter values that result in a null or empty value.</param>
        public TRT2024Validator(TRT2024 reportIn, int maxParameterNameLength = 20, int maxParameterValueLength = 4096, string emptyParameterValue = "EMPTY"):this(maxParameterNameLength,maxParameterValueLength,emptyParameterValue)
        {                   
            this.ConsumedReport = reportIn;
        }

        #region Functions
        private static IEnumerable<string> Union(IEnumerable<string> list1, IEnumerable<string> list2)
        {
            IEnumerable<string> response;

            if (list1 == null && list2 == null)
            {
                response = null;
            }
            else if (list1 == null)
            {
                response = list2.Distinct();
            }
            else if (list2 == null)
            {
                response = list1.Distinct();
            }
            else
            {
                response = list1.Union(list2);
            }

            return response;
        }

        private static bool IsMatch(int? field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(Convert.ToString(field.Value), expression, options);
        }

        private static bool IsMatch(string field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(field, expression, options);
        }

        // This is just context and tuple counts where they are integer values - easier that changing the parsing logic to just return the value
        private static int Count(int count)
        {
            return count;
        }


        private static int Count<T>(IEnumerable<T> values)
        {
            return values == null ? 0 : values.Where(f => f != null).Count();
        }


        private static int Count<T>(ICollection<T> values)
        {
            return values == null ? 0 : values.Where(f => f != null).Count();
        }

        private static bool exists(bool value)
        {
            return value;
        }

        public static string GetValueOrEmpty(bool? val)
        {
            return (val.HasValue) ? val.ToString().ToLower() : string.Empty;
        }

        public static string GetValueOrEmptyDate(DateTime? val)
        {
            return (val.HasValue) ? val.Value.ToString("yyyy-MM-dd") : string.Empty;
        }

        private static string GetValueOrEmptyDate(string val)
        {
            DateTime dateTimeOut;
            if (DateTime.TryParse(val, out dateTimeOut))
            {
                return dateTimeOut.ToString("yyyy-MM-dd");
            }
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmptyDateTime(string val)
        {
            DateTime dateTimeOut;
            if (DateTime.TryParse(val, out dateTimeOut))
            {
                return dateTimeOut.ToString("yyyy-MM-ddTHH:mm:ssZ");
            }
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmptyDateTime(DateTime? val)
        {
            return (val.HasValue) ? val.Value.ToString("yyyy-MM-ddTHH:mm:ssZ") : string.Empty;
        }

        public static string GetValueOrEmpty(string val)
        {
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        public static string GetValueOrEmpty(decimal? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        private static string GetValueOrEmpty(int? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

         /// <summary>
         /// Get string value between [first] a and [last] b.
         /// </summary>
        public static string Between(string value, string a, string b)
        {
            int posA = value.IndexOf(a);
            int posB = value.LastIndexOf(b);
            if (posA == -1)
            {
                return "";
            }
            if (posB == -1)
            {
                return "";
            }
            int adjustedPosA = posA + a.Length;
            if (adjustedPosA >= posB)
            {
                return "";
            }
            return value.Substring(adjustedPosA, posB - adjustedPosA);
        }

         /// <summary>
         /// Get string value after [first] a.
         /// </summary>
        public static string Before(string value, string a)
        {
            int posA = value.IndexOf(a);
            if (posA == -1)
            {
                return "";
            }
            return value.Substring(0, posA);
        }

         /// <summary>
         /// Get string value after [last] a.
         /// </summary>
        public static string After(string value, string a)
        {
            int posA = value.LastIndexOf(a);
            if (posA == -1)
            {
                return "";
            }
            int adjustedPosA = posA + a.Length;
            if (adjustedPosA >= value.Length)
            {
                return "";
            }
            return value.Substring(adjustedPosA);
        }

        private static int Length(object field)
        {
            if (field == null)
            return 0;
            else
            return field.ToString().Trim().Length;
        }

        private static bool NotSameValues(IEnumerable<object> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool NotSameValues(params object[] nodes)
        {
            if (nodes == null)
                return false;

            return (from x in nodes select x).Distinct().Count() != 1;
        }

        private static bool HasDuplicateValues(IEnumerable<DateTime> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool HasDuplicateValues(IEnumerable<DateTime?> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool NotSameValues(IEnumerable<DateTime> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool NotSameValues(IEnumerable<DateTime?> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool HasDuplicateValues(IEnumerable<object> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool HasDuplicateValues(params object[] nodes)
        {
           if (nodes == null)
                return false;

            nodes = nodes.Where(x => x != null).ToArray();
            return !((from x in nodes select x).Distinct().Count() == nodes.Count());
        
        }

        private int DuplicateValueIndex(IEnumerable<object> values)
        {
            int response = 0;
            var hashset = new HashSet<object>();
            foreach (var value in values)
            {
                if (!hashset.Add(value))
                {
                    return response;
                }
                response++;
            }
            return response;
        }

        private int DuplicateValueIndex<T>(IEnumerable<T?> values) where T : struct
        {
            int response = 0;
            var hashset = new HashSet<T?>();
            foreach (var value in values)
            {
                if (!hashset.Add(value))
                {
                    return response;
                }
                response++;
            }
            return response;
        }

        private static bool IsDate(object value)
        {
            DateTime dateValue;
            return (value != null && DateTime.TryParse(value.ToString(), out dateValue));
        }

        public DateTime AsDate(string dateAsString)
        {
            DateTime response = DateTime.MinValue;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date;
            }

            return response;
        }

        public DateTime Date(DateTime? datetime)
        {                   
            return datetime.GetValueOrDefault().Date;
        }

				//The logic in After function expects "---" for day and "--" for month. 
				//Since hyphen was missing the date always returned null
        public DateTime? ConvertToDate(int day, int month, int year)
        {
					return ConvertToDate(day == 0 ? null : "---" + day.ToString(), month == 0 ? null : "--" + month.ToString(), year == 0 ? null : year.ToString());
        }

        public DateTime? ConvertToDate(string day, string month, string year)
        {
            DateTime? response;
            DateTime result;

            if (year == null || month == null || day == null)
            {
                return null;
            }
            string dateAsString = year + "-" + After(month, "--") + "-" + After(day, "---");
            if (DateTime.TryParse(dateAsString, out result))
            {
                response = result;
            }
            else
            {
                response = null;
            }
            return response;
        }


        public DateTime? ConvertToDate(string day, string month, int year)
        {
            DateTime? response;
            DateTime result;

            if (year == 0 || month == null || day == null)
            {
                return null;
            }
            string dateAsString = year.ToString() + "-" + After(month, "--") + "-" + After(day, "---");
            if (DateTime.TryParse(dateAsString, out result))
            {
                response = result;
            }
            else
            {
                response = null;
            }
            return response;
        }


        private static int Day(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Day;
            }

            return response;
        }


        private static int Day(DateTime? date)
        {
            if (date == null)
                return 0;
            else
                return date.Value.Day;

        }


        private static string Month(string dateAsString)
        {
            string response = null;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.ToString("MMMM");
            }
            else
            {
                return "NotAMonth";
            }

            return response;
        }

        private static string Month(DateTime? date)
        {
            if (date == null)
                return "NotAMonth";
            else
                return date.Value.ToString("MMMM");
        }


        private static int MonthAsInt(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Month;
            }

            return response;
        }

        private static int MonthAsInt(DateTime? date)
        {
            if (date == null)
            {
                return 0;
            }
            return date.Value.Month;
        }


        private static int Year(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Year;
            }

            return response;
        }

        private static int Year(DateTime? date)
        {
            if (date == null)
                return 0;
            else
                return date.Value.Year;
        }


        private static int CurrentFinancialYear()
        {
            return DateToFinancialYear(DateTime.Now, 7);
        }

        private static int FinancialYear(string dateAsString)
        {
            return DateToFinancialYear(dateAsString, 7);
        }

        private static int FinancialYear(DateTime? date)
        {
            return DateToFinancialYear(date, 7);
        }

        private static int DateToFinancialYear(string dateAsString, int startingMonth)
        {
            int response = 0;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
            {
                response = DateToFinancialYear(date, startingMonth);
            }

            return response;
        }
        private static int DateToFinancialYear(DateTime? date, int startingMonth)
        {
            int response;
            if (date == null)
            {
                response = 0;
            }
            else
            {
                int year = date.Value.Year;
                int month = date.Value.Month;

                if (startingMonth > month)
                    response = year;
                else
                    response = year + 1;
            }
            return response;
        }


        private static int FBTYear(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
                response = FBTYear(date);
            return response;
        }

        private static int FBTYear(DateTime? date)
        {
            if (date == null)
            {
                return 0;
            }
            else
            {
                if (date.Value.Month > 3)
                    return date.Value.Year + 1;
                else
                    return date.Value.Year;
            }
        }

        private static DateTime? AddMonthsToDate(DateTime? dateTime, int months)
        {
            return dateTime == null ? null : (DateTime?)dateTime.Value.AddMonths(months);
        }

        private static bool IsNumeric(object value)
        {
            decimal numbervalue;
            return (value != null && decimal.TryParse(value.ToString(), out numbervalue));
        }

        private static bool NotMonetary(decimal? field, string sign, int digits, int decimals)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                string decimalExpression;
                int digitsToUse = digits - decimals;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                if (decimals > 0)
                    decimalExpression = @"(\.\d{1," + decimals + "})?$";
                else
                    decimalExpression = @"$";

                return !(Regex.IsMatch(field.Value.ToString("0.#########################"), signExpression + @"\d{1," + digitsToUse + "}" + decimalExpression));
            }
        }

        private static bool NotNumeric(decimal? field, string sign, int digits, int decimals)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                string decimalExpression;
                int digitsToUse = digits - decimals;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                if (decimals > 0)
                    decimalExpression = @"(\.\d{1," + decimals + "})?$";
                else
                    decimalExpression = @"$";

                return !(Regex.IsMatch(field.Value.ToString("0.#########################"), signExpression + @"\d{1," + digitsToUse + "}" + decimalExpression));
            }
        }

        private static bool NotNumeric(int? field, string sign, int digits, int decimals = 0)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                return !(Regex.IsMatch(field.Value.ToString(), signExpression + @"\d{1," + digits + "}$"));
            }
        }

        private static bool NotNumeric(long? field, string sign, int digits, int decimals = 0)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                return !(Regex.IsMatch(field.Value.ToString(), signExpression + @"\d{1," + digits + "}$"));
            }
        }


        private static bool OutsideRange(decimal field, decimal expression, int range)
        {
            bool response;

            response = (field < (expression - range)) || (field > (expression + range));

            return response;
        }


        private static bool FailsUSIAlgorithm(string usi, string abn)
        {
            bool response = true;
            if (usi == null || abn == null)
            {
                response = false;
            }
            else
            {
                usi = usi.Trim();
                abn = abn.Trim();
                if (usi.Length == 14)
                {
                    int numeric;
                    if (usi.Substring(0, 11) == abn && int.TryParse(usi.Substring(11, 3), out numeric))
                        response = false;

                }
                else if (usi.Length == 9)
                {

                    if (Regex.IsMatch(usi, @"^([a-zA-Z]{3}\d{4}[a-zA-Z]{2})$"))
                        response = false;
                }
            }
            return response;
        }

        private static bool FailsTANAlgorithm(string tan)
        {
            bool response;
            decimal decimalTan;

            if (tan == null)
                return false;

            tan = tan.Trim();

            if (!decimal.TryParse(tan, out decimalTan))
                return true;

            if (tan.Length != 8)
                return true;

            decimal tanSum =
                7 * int.Parse(tan.Substring(0, 1)) +
                9 * int.Parse(tan.Substring(1, 1)) +
                8 * int.Parse(tan.Substring(2, 1)) +
                4 * int.Parse(tan.Substring(3, 1)) +
                6 * int.Parse(tan.Substring(4, 1)) +
                3 * int.Parse(tan.Substring(5, 1)) +
                5 * int.Parse(tan.Substring(6, 1)) +
                1 * int.Parse(tan.Substring(7, 1));

            if ((tanSum % 11) == 0)
                response = false;
            else
                response = true;

            return response;
        }


        private static bool FailsABNAlgorithm(string abn)
        {
            bool response;
            decimal decimalAbn;

            if (abn == null)
                return false;

            abn = abn.Trim();

            if (!decimal.TryParse(abn, out decimalAbn))
                return true;

            if (abn.Length != 11)
                return true;

            decimal abnSum =
                10 * (int.Parse(abn.Substring(0, 1)) - 1) +
                1 * int.Parse(abn.Substring(1, 1)) +
                3 * int.Parse(abn.Substring(2, 1)) +
                5 * int.Parse(abn.Substring(3, 1)) +
                7 * int.Parse(abn.Substring(4, 1)) +
                9 * int.Parse(abn.Substring(5, 1)) +
                11 * int.Parse(abn.Substring(6, 1)) +
                13 * int.Parse(abn.Substring(7, 1)) +
                15 * int.Parse(abn.Substring(8, 1)) +
                17 * int.Parse(abn.Substring(9, 1)) +
                19 * int.Parse(abn.Substring(10, 1));

            if ((abnSum % 89) == 0)
                response = false;
            else
                response = true;

            return response;
        }
        private static bool FailsACNAlgorithm(string acn)
        {
            bool response;
            decimal decimalAbn;
            if (acn == null)
                return false;

            acn = acn.Trim();

            if (!decimal.TryParse(acn, out decimalAbn))
                return true;

            if (acn.Length != 9)
                return true;

            decimal abnSum =
                8 * int.Parse(acn.Substring(0, 1)) +
                7 * int.Parse(acn.Substring(1, 1)) +
                6 * int.Parse(acn.Substring(2, 1)) +
                5 * int.Parse(acn.Substring(3, 1)) +
                4 * int.Parse(acn.Substring(4, 1)) +
                3 * int.Parse(acn.Substring(5, 1)) +
                2 * int.Parse(acn.Substring(6, 1)) +
                1 * int.Parse(acn.Substring(7, 1));

            decimal checkDigit = int.Parse(acn.Substring(8, 1));
            decimal acnRemainder = abnSum % 10;

            if (((10 - acnRemainder) % 10) == checkDigit)
                response = false;
            else
                response = true;

            return response;
        }


        private static bool FailsTFNAlgorithm(string tfn)
        {
            bool response;
            decimal decimalTfn;

            if (tfn == null)
                return false;

            tfn = tfn.Trim();
            tfn = Regex.Replace(tfn, "^0+", "");

            if (!decimal.TryParse(tfn, out decimalTfn))
                return true;

            if (tfn.Length < 8)
                return true;


            decimal tfn1To7Sum =
                1 * int.Parse(tfn.Substring(0, 1)) +
                4 * int.Parse(tfn.Substring(1, 1)) +
                3 * int.Parse(tfn.Substring(2, 1)) +
                7 * int.Parse(tfn.Substring(3, 1)) +
                5 * int.Parse(tfn.Substring(4, 1)) +
                8 * int.Parse(tfn.Substring(5, 1)) +
                6 * int.Parse(tfn.Substring(6, 1));

            decimal tfn8 = 9 * int.Parse(tfn.Substring(7, 1));

            if (tfn.Length == 8)
            {
                decimal tFNLg8WSum9 = 10 * int.Parse(tfn.Substring(7, 1));
                decimal tFNLg8WSum = tfn1To7Sum + tFNLg8WSum9;

                if ((tFNLg8WSum % 11) == 0)
                    response = false;
                else
                    response = true;
            }
            else if (tfn.Length == 9)
            {
                decimal tfn9 = 10 * int.Parse(tfn.Substring(8, 1));
                decimal tFNLg9WSum = tfn1To7Sum + tfn8 + tfn9;

                if ((tFNLg9WSum % 11) == 0)
                    response = false;
                else
                    response = true;
            }
            else
            {
                response = true;
            }

            return response;
        }


        private static decimal ConditionalValue(bool expression, decimal? trueVal, decimal? falseVal)
        {
             return expression ? trueVal.GetValueOrDefault() : falseVal.GetValueOrDefault();
        }

        private static decimal AsNumeric(string value)
        {
             decimal numberValue;
             decimal.TryParse(value, out numberValue);
             return numberValue;
        }

        private static bool RegexMatch(int? field, string expression, string flags="")
        {
            return IsMatch(field, expression, GetRegexOption(flags));
        }

        private static bool RegexMatch(string field, string expression, string flags="")
        {
            return IsMatch(field, expression, GetRegexOption(flags));
        }

        private static RegexOptions GetRegexOption(string flags)
        {
            RegexOptions options = RegexOptions.None;

            char[] characters = flags.ToCharArray();

            foreach (char character in characters)
            {
                switch (character)
                {
                    case 'i':                        
                        options = options | RegexOptions.IgnoreCase;                       
                        break;
                    case 'm':                        
                        options = options | RegexOptions.Multiline;                       
                        break;
                    case 's':                        
                            options = options | RegexOptions.Singleline;                       
                        break;
                    case 'n':                        
                            options = options | RegexOptions.ExplicitCapture;                       
                        break;
                    case 'x':                       
                            options = options | RegexOptions.IgnorePatternWhitespace;                       
                        break;
                }
            }

            return options;
        }

        private static string Substring(object field, int start, int length)
        {
            return field.ToString().Substring(start, length);
        }
        
        /// <summary>
        /// Returns an occurrence index as [occurrenceIndex] of occurrenceIndex > 0, otherwise the empty string
        /// </summary>
        /// <param name="occurrenceIndex">Index of the occurrence.</param>
        /// <returns>Occurrence in XPath [#] format</returns>
        public string OccurrenceIndex(int occurrenceIndex)
        {
            return occurrenceIndex > 0 ? "[" + occurrenceIndex + "]" : "";
        }

        #endregion // Functions

        public TRT2024 ConsumedReport { get {return report;} private set {report = value;}}

        private static Dictionary<string, ProcessMessageDocument> _processMessageDocuments = new Dictionary<string,ProcessMessageDocument>();				

        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public List<ProcessMessageDocument> ValidateReport(TRT2024 reportIn)
        {
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            bool assertion;

            this.ConsumedReport = reportIn;

    #region Manual Rules - Rules marked for manual coding
    
        #region VR.ATO.TRT.432089

        /*  VR.ATO.TRT.432089
            The sum of 'Credit for tax withheld where ABN not quoted' recorded in the Statements of distribution must equal the sum of 'Credit for tax withheld where ABN not quoted' plus 'Share of credit for tax withheld where ABN not quoted'

        Legacy Rule Format:
            (Sum(^TRT220) + ^TRT236) <> (^TRT73 + ^TRT84)
    
        Technical Business Rule Format:
            (Sum(^TRT220) + ^TRT236) <> (^TRT73 + ^TRT84)

        Data Elements:

        ^TRT73 = TRT:RP:BusinessDetails:Income:PayAsYouGoWithholding:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount

        ^TRT84 = TRT:RP:BusinessDetails:Income:ShareOfCredits:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuotedShareReceivedIndirectly.Amount

        ^TRT220 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount

        ^TRT236 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount
        */
    VRATOTRT432089();
    #endregion // VR.ATO.TRT.432089

    #region VR.ATO.TRT.432093

/*  VR.ATO.TRT.432093
            The sum of 'Credit for tax withheld - foreign resident withholding (excluding capital gains)' recorded in the Statements of distribution must equal the sum of 'Credit for tax withheld - foreign resident withholding (excluding capital gains)' plus 'Share of credit for tax withheld - foreign resident withholding (excluding capital gains)'

        Legacy Rule Format:
            (Sum(^TRT476) + ^TRT477) <> (^TRT474 + ^TRT87)

    Technical Business Rule Format:
            (Sum(^TRT476) + ^TRT477) <> (^TRT474 + ^TRT87)

Data Elements:
    
    ^TRT474 = TRT:RP:BusinessDetails:Income:PayAsYouGoWithholding:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromForeignResidents.Amount
    
    ^TRT87 = TRT:RP:BusinessDetails:Income:ShareOfCredits:InternationalDealings.CreditForTaxWithheldFromForeignResidentWithholdingShareReceivedIndirectly.Amount
    
    ^TRT476 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromForeignResidents.Amount
    
    ^TRT477 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromForeignResidents.Amount
    */
    VRATOTRT432093();
    #endregion // VR.ATO.TRT.432093

    #region VR.ATO.TRT.432110

/*  VR.ATO.TRT.432110
            If Total net income is greater than zero, then the sum of 'Franking credit' recorded in the Statements of distribution must equal the sum of 'Share of franking credit from franked dividends' and 'Dividends:Franking credit'

        Legacy Rule Format:
            (^TRT130 + ^TRT133 + ^TRT134 + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144) > 0 AND OutsideRange(Sum(^TRT221) + ^TRT237, ^TRT85 + ^TRT98, 1)

    Technical Business Rule Format:
            (^TRT130 + ^TRT133 + ^TRT134 + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144) > 0 AND OutsideRange(Sum(^TRT221) + ^TRT237, ^TRT85 + ^TRT98, 1)

Data Elements:
    
    ^TRT85 = TRT:RP:BusinessDetails:Income:ShareOfCredits:IncomeTax.FrankingCredits.FrankingCreditShareReceivedIndirectly.Amount
    
    ^TRT98 = TRT:RP:BusinessDetails:Income:Dividends:IncomeTax.FrankingCredits.Amount
    
    ^TRT130 = TRT:RP:BusinessDetails:Income:Income.Net.Amount
    
    ^TRT133 = TRT:RP:BusinessDetails:Income:CapitalGains:Income.CapitalGainsNet.Amount
    
    ^TRT134 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:ListedCountry:Income.InternationalDealings.Attributed.Amount
    
    ^TRT136 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:UnlistedCountry:Income.InternationalDealings.Attributed.Amount
    
    ^TRT141 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:Income.Net.Amount
    
    ^TRT143 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
    
    ^TRT144 = TRT:RP:BusinessDetails:Income:TaxLossesDeducted:IncomeTax.Deduction.TaxLossesDeducted.Amount
    
    ^TRT221 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.FrankingCredits.ReceivedFromAustralianCompanies.Amount
    
    ^TRT237 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.FrankingCredits.ReceivedFromAustralianCompanies.Amount
    */
    VRATOTRT432110();
    #endregion // VR.ATO.TRT.432110

    #region VR.ATO.TRT.432112

/*  VR.ATO.TRT.432112
            The sum of 'TFN amounts withheld' recorded in the Statements of distribution must equal the sum of 'Share of credit for TFN amounts withheld from interest...' plus 'TFN amounts withheld from gross interest' plus 'TFN amounts withheld from dividends'

        Legacy Rule Format:
            (Sum(^TRT222) + ^TRT238) <> (^TRT86 + ^TRT95 + ^TRT99)

    Technical Business Rule Format:
            (Sum(^TRT222) + ^TRT238) <> (^TRT86 + ^TRT95 + ^TRT99)

Data Elements:
    
    ^TRT86 = TRT:RP:BusinessDetails:Income:ShareOfCredits:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromInterestAndDividendsWhereTFNNotQuotedShareReceivedIndirectly.Amount
    
    ^TRT95 = TRT:RP:BusinessDetails:Income:GrossInterest:IncomeTax.TFNAmountsWithheldFromGrossInterest.Amount
    
    ^TRT99 = TRT:RP:BusinessDetails:Income:Dividends:IncomeTax.DividendsTFNAmountsWithheld.Amount
    
    ^TRT222 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldWhereTFNNotQuoted.Amount
    
    ^TRT238 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldWhereTFNNotQuoted.Amount
    */
    VRATOTRT432112();
    #endregion // VR.ATO.TRT.432112

    #region VR.ATO.TRT.432179

/*  VR.ATO.TRT.432179
            The overall net trust income is greater than zero (the sum of net Australian income/loss, net capital gains, attributed foreign income other assessable foreign source income minus allowable tax losses of earlier income years deducted this year) AND the sum of the income amounts in the distribution statement is not equal to the overall net trust income.

        Legacy Rule Format:
            (^TRT130 + ^TRT133 + ^TRT134 + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144) > 0 AND (^TRT130 + ^TRT133 + ^TRT134 + ^TRT136 + ^TRT141 - ^TRT144) <> (Sum(^TRT218) + ^TRT234 + Sum(^TRT219) + ^TRT235 + Sum(^TRT442) + ^TRT444 + Sum(^TRT223) + ^TRT239 + ^TRT446 + Sum(^TRT224) + ^TRT240 + Sum(^TRT225) + ^TRT241)

    Technical Business Rule Format:
            (^TRT130 + ^TRT133 + ^TRT134 + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144) > 0 AND (^TRT130 + ^TRT133 + ^TRT134 + ^TRT136 + ^TRT141 - ^TRT144) <> (Sum(^TRT218) + ^TRT234 + Sum(^TRT219) + ^TRT235 + Sum(^TRT442) + ^TRT444 + Sum(^TRT223) + ^TRT239 + ^TRT446 + Sum(^TRT224) + ^TRT240 + Sum(^TRT225) + ^TRT241)

Data Elements:
    
    ^TRT130 = TRT:RP:BusinessDetails:Income:Income.Net.Amount
    
    ^TRT133 = TRT:RP:BusinessDetails:Income:CapitalGains:Income.CapitalGainsNet.Amount
    
    ^TRT134 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:ListedCountry:Income.InternationalDealings.Attributed.Amount
    
    ^TRT136 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:UnlistedCountry:Income.InternationalDealings.Attributed.Amount
    
    ^TRT141 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:Income.Net.Amount
    
    ^TRT143 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
    
    ^TRT144 = TRT:RP:BusinessDetails:Income:TaxLossesDeducted:IncomeTax.Deduction.TaxLossesDeducted.Amount
    
    ^TRT218 = TRT:RP:StatementOfDistributionToBeneficiary:Information:ShareOfIncomePrimaryProduction:Income.BeneficiaryShare.Amount
    
    ^TRT219 = TRT:RP:StatementOfDistributionToBeneficiary:Information:ShareOfIncomeNonPrimaryProduction:Income.BeneficiaryShare.Amount
    
    ^TRT223 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Income.CapitalGainsNet.Amount
    
    ^TRT224 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.AttributedForeignIncome.Amount
    
    ^TRT225 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Income.InternationalDealings.Net.Amount
    
    ^TRT234 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:PrimaryProductionNoBeneficiaryEntitled:IncomeTax.IncomeOrLossTaxable.Amount
    
    ^TRT235 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:NonPrimaryProductionNoBeneficiaryEntitled:IncomeTax.IncomeOrLossTaxable.Amount
    
    ^TRT239 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:Income.CapitalGainsNet.Amount
    
    ^TRT240 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.AttributedForeignIncome.Amount
    
    ^TRT241 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:Income.InternationalDealings.Net.Amount
    
    ^TRT442 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Income.DistributionFranked.Amount
    
    ^TRT444 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:Income.DistributionFranked.Amount
    
    ^TRT446 = TRT:RP:CapitalGainsAssessed:CapitalGainsTax.TrusteeAssessedCapitalGain.Amount
    */
    VRATOTRT432179();
    #endregion // VR.ATO.TRT.432179

    #region VR.ATO.TRT.432181

/*  VR.ATO.TRT.432181
            There is an overall trust loss (i.e. the sum of net Australian income/loss plus net capital gains, plus attributed foreign income plus other assessable foreign source income minus allowable tax losses of earlier income years deducted this year is less than zero) AND income distribution amount/s is/are present in the distribution statement (distribution to a beneficiary or no beneficiary presently entitled). Income cannot be distributed when there is an overall trust loss. Delete the income from the distribution statement.

        Legacy Rule Format:
            (^TRT130 + ^TRT133 + ^TRT134 + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144) <= 0 AND (Sum(^TRT218) + Sum(^TRT219) + Sum(^TRT442) + Sum(^TRT223) + Sum(^TRT224) + Sum(^TRT225) + Sum(^TRT217) + ^TRT233 + ^TRT234 + ^TRT235 + ^TRT444 + ^TRT239 + ^TRT240 + ^TRT241 + ^TRT446) <> 0

    Technical Business Rule Format:
            (^TRT130 + ^TRT133 + ^TRT134 + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144) <= 0 AND (Sum(^TRT218) + Sum(^TRT219) + Sum(^TRT442) + Sum(^TRT223) + Sum(^TRT224) + Sum(^TRT225) + Sum(^TRT217) + ^TRT233 + ^TRT234 + ^TRT235 + ^TRT444 + ^TRT239 + ^TRT240 + ^TRT241 + ^TRT446) <> 0

Data Elements:
    
    ^TRT130 = TRT:RP:BusinessDetails:Income:Income.Net.Amount
    
    ^TRT133 = TRT:RP:BusinessDetails:Income:CapitalGains:Income.CapitalGainsNet.Amount
    
    ^TRT134 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:ListedCountry:Income.InternationalDealings.Attributed.Amount
    
    ^TRT136 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:UnlistedCountry:Income.InternationalDealings.Attributed.Amount
    
    ^TRT141 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:Income.Net.Amount
    
    ^TRT143 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
    
    ^TRT144 = TRT:RP:BusinessDetails:Income:TaxLossesDeducted:IncomeTax.Deduction.TaxLossesDeducted.Amount
    
    ^TRT217 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
    
    ^TRT218 = TRT:RP:StatementOfDistributionToBeneficiary:Information:ShareOfIncomePrimaryProduction:Income.BeneficiaryShare.Amount
    
    ^TRT219 = TRT:RP:StatementOfDistributionToBeneficiary:Information:ShareOfIncomeNonPrimaryProduction:Income.BeneficiaryShare.Amount
    
    ^TRT223 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Income.CapitalGainsNet.Amount
    
    ^TRT224 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.AttributedForeignIncome.Amount
    
    ^TRT225 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Income.InternationalDealings.Net.Amount
    
    ^TRT233 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
    
    ^TRT234 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:PrimaryProductionNoBeneficiaryEntitled:IncomeTax.IncomeOrLossTaxable.Amount
    
    ^TRT235 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:NonPrimaryProductionNoBeneficiaryEntitled:IncomeTax.IncomeOrLossTaxable.Amount
    
    ^TRT239 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:Income.CapitalGainsNet.Amount
    
    ^TRT240 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.AttributedForeignIncome.Amount
    
    ^TRT241 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:Income.InternationalDealings.Net.Amount
    
    ^TRT442 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Income.DistributionFranked.Amount
    
    ^TRT444 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:Income.DistributionFranked.Amount
    
    ^TRT446 = TRT:RP:CapitalGainsAssessed:CapitalGainsTax.TrusteeAssessedCapitalGain.Amount
    */
    VRATOTRT432181();
    #endregion // VR.ATO.TRT.432181

    #region VR.ATO.TRT.432193

/*  VR.ATO.TRT.432193
            The overall net trust income is greater than zero (the sum of net Australian income/loss, net capital gains, attributed foreign income other assessable foreign source income minus allowable tax losses of earlier income years deducted this year), AND the sum of Attributed Foreign Income (including no beneficiary presently entitled) in the distribution statement, is not equal to the sum of Attributed Foreign Income amounts of the trust.

        Legacy Rule Format:
            (^TRT130 + ^TRT133 + ^TRT134 + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144) > 0 AND (Sum(^TRT224) + ^TRT240) <> (^TRT134 + ^TRT136)

    Technical Business Rule Format:
            (^TRT130 + ^TRT133 + ^TRT134 + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144) > 0 AND (Sum(^TRT224) + ^TRT240) <> (^TRT134 + ^TRT136)

Data Elements:
    
    ^TRT134 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:ListedCountry:Income.InternationalDealings.Attributed.Amount
    
    ^TRT130 = TRT:RP:BusinessDetails:Income:Income.Net.Amount
    
    ^TRT133 = TRT:RP:BusinessDetails:Income:CapitalGains:Income.CapitalGainsNet.Amount
    
    ^TRT136 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:UnlistedCountry:Income.InternationalDealings.Attributed.Amount
    
    ^TRT141 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:Income.Net.Amount
    
    ^TRT143 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
    
    ^TRT144 = TRT:RP:BusinessDetails:Income:TaxLossesDeducted:IncomeTax.Deduction.TaxLossesDeducted.Amount
    
    ^TRT224 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.AttributedForeignIncome.Amount
    
    ^TRT240 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.AttributedForeignIncome.Amount
    */
    VRATOTRT432193();
    #endregion // VR.ATO.TRT.432193

    #region VR.ATO.TRT.432298

/*  VR.ATO.TRT.432298
            If the trust is in an income position, and Australian franking credits from a New Zealand franking company is present, then this amount must be greater than or equal to the amount of Australian franking credits from a New Zealand franking company in the distribution statement (beneficiary and non-beneficiary).

        Legacy Rule Format:
            (^TRT130 + ^TRT133 + ^TRT134 + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144) > 0 AND ^TRT143 > 0 AND ^TRT143 < (Sum(^TRT217) + ^TRT233)

    Technical Business Rule Format:
            (^TRT130 + ^TRT133 + ^TRT134 + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144) > 0 AND ^TRT143 > 0 AND ^TRT143 < (Sum(^TRT217) + ^TRT233)

Data Elements:
    
    ^TRT217 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
    
    ^TRT130 = TRT:RP:BusinessDetails:Income:Income.Net.Amount
    
    ^TRT133 = TRT:RP:BusinessDetails:Income:CapitalGains:Income.CapitalGainsNet.Amount
    
    ^TRT134 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:ListedCountry:Income.InternationalDealings.Attributed.Amount
    
    ^TRT136 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:UnlistedCountry:Income.InternationalDealings.Attributed.Amount
    
    ^TRT141 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:Income.Net.Amount
    
    ^TRT143 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
    
    ^TRT144 = TRT:RP:BusinessDetails:Income:TaxLossesDeducted:IncomeTax.Deduction.TaxLossesDeducted.Amount
    
    ^TRT233 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
    */
    VRATOTRT432298();
    #endregion // VR.ATO.TRT.432298

    #region VR.ATO.TRT.432315

/*  VR.ATO.TRT.432315
            The overall net trust income is greater than zero (the sum of net Australian income/loss, net capital gains, attributed foreign income other assessable foreign source income minus allowable tax losses of earlier income years deducted this year) and the sum of Foreign income tax offsets (including no beneficiary presently entitled) is not equal to the Foreign income tax offsets.

        Legacy Rule Format:
            (^TRT130 + ^TRT133 + ^TRT134 + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144) > 0 AND (Sum(^TRT226) + ^TRT242) <> ^TRT142

    Technical Business Rule Format:
            (^TRT130 + ^TRT133 + ^TRT134 + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144) > 0 AND (Sum(^TRT226) + ^TRT242) <> ^TRT142

Data Elements:
    
    ^TRT226 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Income.InternationalDealings.TaxOffset.Amount
    
    ^TRT130 = TRT:RP:BusinessDetails:Income:Income.Net.Amount
    
    ^TRT133 = TRT:RP:BusinessDetails:Income:CapitalGains:Income.CapitalGainsNet.Amount
    
    ^TRT134 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:ListedCountry:Income.InternationalDealings.Attributed.Amount
    
    ^TRT136 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:UnlistedCountry:Income.InternationalDealings.Attributed.Amount
    
    ^TRT141 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:Income.Net.Amount
    
    ^TRT142 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:Income.InternationalDealings.TaxOffset.Amount
    
    ^TRT143 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
    
    ^TRT144 = TRT:RP:BusinessDetails:Income:TaxLossesDeducted:IncomeTax.Deduction.TaxLossesDeducted.Amount
    
    ^TRT242 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:Income.InternationalDealings.TaxOffset.Amount
    */
    VRATOTRT432315();
    #endregion // VR.ATO.TRT.432315

    #region VR.ATO.TRT.432342

/*  VR.ATO.TRT.432342
            The total amount at National rental affordability scheme tax offset (beneficiary and non beneficiary) must equal the amount at National rental affordability scheme tax offset entitlement.

        Legacy Rule Format:
            (Sum(^TRT342) + ^TRT343) <> ^TRT340

    Technical Business Rule Format:
            (Sum(^TRT342) + ^TRT343) <> ^TRT340

Data Elements:
    
    ^TRT340 = TRT:RP:TaxOffsets:TaxOffsetClaim.NationalRentalAffordabilitySchemeEntitlement.Amount
    
    ^TRT342 = TRT:RP:StatementOfDistributionToBeneficiary:Information:TaxOffsetClaim.NationalRentalAffordabilitySchemeEntitlement.Amount
    
    ^TRT343 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:TaxOffsetClaim.NationalRentalAffordabilitySchemeEntitlement.Amount
    */
    VRATOTRT432342();
    #endregion // VR.ATO.TRT.432342

    #region VR.ATO.TRT.432578

/*  VR.ATO.TRT.432578
            In the distribution statement the sum of Share of credit for TFN amounts withheld from payments from closely held trusts to which beneficiaries are entitled to and to which beneficiaries are not entitled to is not equal to the Credit for TFN amounts withheld from payments from closely held trust.

        Legacy Rule Format:
            ^TRT418 <> (Sum(^TRT419) + ^TRT435)

    Technical Business Rule Format:
            ^TRT418 <> (Sum(^TRT419) + ^TRT435)

Data Elements:
    
    ^TRT418 = TRT:RP:BusinessDetails:Income:ShareOfCredits:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldFromCloselyHeldTrustShareReceivedIndirectly.Amount
    
    ^TRT419 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldFromCloselyHeldTrust.Amount
    
    ^TRT435 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldFromCloselyHeldTrust.Amount
    */
    VRATOTRT432578();
    #endregion // VR.ATO.TRT.432578

    #region VR.ATO.TRT.432632

/*  VR.ATO.TRT.432632
            The sum of Share of income of the trust estate (beneficary entitled) and the Share of income of the trust estate (no beneficiary entitled) in the Statement of distribution, must equal the Income of the Trust Estate.

        Legacy Rule Format:
            (AnyOccurrence(^TRT1070, ^TRT441 > 0) OR ^TRT443 > 0) AND ((Sum(^TRT441) + ^TRT443) <> ^TRT440)

    Technical Business Rule Format:
            (AnyOccurrence(^TRT1070, ^TRT441 > 0) OR ^TRT443 > 0) AND ((Sum(^TRT441) + ^TRT443) <> ^TRT440)

Data Elements:
    
    ^TRT440 = TRT:RP:Income.TrustEstateIncome.Amount
    
    ^TRT441 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Income.TrustEstateIncomeShare.Amount
    
    ^TRT443 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:Income.TrustEstateIncomeShare.Amount
    
    ^TRT1070 = TRT:RP:StatementOfDistributionToBeneficiary
    */
    VRATOTRT432632();
    #endregion // VR.ATO.TRT.432632

    #region VR.ATO.TRT.432714

/*  VR.ATO.TRT.432714
            The Share of Other Refundable Tax Offsets plus the sum of Exploration Credits is not equal to Other Refundable Tax Offset

        Legacy Rule Format:
            (Sum(^TRT465) + ^TRT466) <> ^TRT463

    Technical Business Rule Format:
            (Sum(^TRT465) + ^TRT466) <> ^TRT463

Data Elements:
    
    ^TRT463 = TRT:RP:TaxOffsets:IncomeTax.TaxOffsetRefundableOther.Amount
    
    ^TRT465 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.ExplorationCreditsDistributed.Amount
    
    ^TRT466 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.TaxOffsetRefundableOther.Amount
    */
    VRATOTRT432714();
    #endregion // VR.ATO.TRT.432714

    #region VR.ATO.TRT.432765

/*  VR.ATO.TRT.432765
            The total amount at Credit for amounts withheld from foreign resident capital gains withholding (beneficiary and non beneficiary) must equal the amount at Credit for amounts withheld from foreign resident capital gains withholding.

        Legacy Rule Format:
            ^TRT480 <> (Sum(^TRT481) + ^TRT482)

    Technical Business Rule Format:
            ^TRT480 <> (Sum(^TRT481) + ^TRT482)

Data Elements:
    
    ^TRT480 = TRT:RP:BusinessDetails:Income:CapitalGains:IncomeTax.PayAsYouGoWithholding.CreditForCapitalGainsWithheldFromForeignResidents.Amount
    
    ^TRT481 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.PayAsYouGoWithholding.CreditForCapitalGainsWithheldFromForeignResidents.Amount
    
    ^TRT482 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.PayAsYouGoWithholding.CreditForCapitalGainsWithheldFromForeignResidents.Amount
    */
    VRATOTRT432765();
    #endregion // VR.ATO.TRT.432765

    #region VR.ATO.TRT.432784

/*  VR.ATO.TRT.432784
            The total amount at Early stage venture capital limited partnership tax offset (beneficiary and no beneficiary presently entitled) must equal the amount at Early stage venture capital limited partnership tax offset.

        Legacy Rule Format:
            ^TRT500 <> (Sum(^TRT502) + ^TRT504)

    Technical Business Rule Format:
            ^TRT500 <> (Sum(^TRT502) + ^TRT504)

Data Elements:
    
    ^TRT500 = TRT:RP:TaxOffsets:LimitedPartnershipTaxOffset:TaxOffsetClaim.NonRefundableOther.Amount
    
    ^TRT502 = TRT:RP:StatementOfDistributionToBeneficiary:Information:EarlyStageVentureTaxOffset:TaxOffsetClaim.NonRefundableOther.Amount
    
    ^TRT504 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:EarlyStageVentureNoBeneficiary:TaxOffsetClaim.NonRefundableOther.Amount
    */
    VRATOTRT432784();
    #endregion // VR.ATO.TRT.432784

    #region VR.ATO.TRT.432787

/*  VR.ATO.TRT.432787
            The total of amounts at Early stage investor tax offset (beneficiary and no beneficiary presently entitled) must equal the amount at Early stage investor tax offset.

        Legacy Rule Format:
            ^TRT501 <> (Sum(^TRT503) + ^TRT506)

    Technical Business Rule Format:
            ^TRT501 <> (Sum(^TRT503) + ^TRT506)

Data Elements:
    
    ^TRT501 = TRT:RP:TaxOffsets:InvestorTaxOffset:TaxOffsetClaim.NonRefundableOther.Amount
    
    ^TRT503 = TRT:RP:StatementOfDistributionToBeneficiary:Information:EarlyStageInvestorTaxOffset:TaxOffsetClaim.NonRefundableOther.Amount
    
    ^TRT506 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:InvestorNoBeneficiary:TaxOffsetClaim.NonRefundableOther.Amount
    */
    VRATOTRT432787();
    #endregion // VR.ATO.TRT.432787

    #region VR.ATO.TRT.433083

/*  VR.ATO.TRT.433083
            Capital gains must be equal to the sum of Gross capital gain minus Capital losses applied, CGT discounts applied and CGT small business concessions applied with a tolerance of $5

        Legacy Rule Format:
            AnyOccurrence(^TRT1070, OutsideRange(^TRT223, ^TRT569 - ^TRT570 - ^TRT571 - ^TRT572, 5))

    Technical Business Rule Format:
            AnyOccurrence(^TRT1070, OutsideRange(^TRT223, ^TRT569 - ^TRT570 - ^TRT571 - ^TRT572, 5))

Data Elements:
    
    ^TRT223 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Income.CapitalGainsNet.Amount
    
    ^TRT569 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Capital.Gains.Total.Amount
    
    ^TRT570 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Capital.Losses.Total.Amount
    
    ^TRT571 = TRT:RP:StatementOfDistributionToBeneficiary:Information:TaxConcession.CapitalGains.DiscountTotal.Amount
    
    ^TRT572 = TRT:RP:StatementOfDistributionToBeneficiary:Information:TaxConcession.CapitalGains.DiscountAndSmallBusinessConcessionsTotal.Amount
    
    ^TRT1070 = TRT:RP:StatementOfDistributionToBeneficiary
    */
    VRATOTRT433083();
    #endregion // VR.ATO.TRT.433083

    #region VR.ATO.TRT.433084

/*  VR.ATO.TRT.433084
            Net small business income must not be less than the sum of Share of net small business income with a tolerance of $5

        Legacy Rule Format:
            ^TRT478 + 5 <  Sum(^TRT479)

    Technical Business Rule Format:
            ^TRT478 + 5 <  Sum(^TRT479)

Data Elements:
    
    ^TRT478 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:Income.SmallBusinessEntityNet.Amount
    
    ^TRT479 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Income.SmallBusinessEntityNet.Amount
    */
    VRATOTRT433084();
    #endregion // VR.ATO.TRT.433084

    #region VR.ATO.TRT.433095

/*  VR.ATO.TRT.433095
            The sum of Australian franking credits from a New Zealand franking company - Other assessable foreign source income, excluding income shown at Attributed foreign income, Share of franking credit from franked distributions, and Foreign income tax offset doesn't equal the sum of all Australian franking credits from a New Zealand company - Beneficiary share, Dividends - Franking credit, Australian franking credits from a New Zealand company - No beneficiary presently entitled, all Franking credit - Beneficiary share, Franking credit - No beneficiary presently entitled, all Foreign income tax offset - Beneficiary share, and Foreign income tax offsets - No beneficiary presently entitled

        Legacy Rule Format:
            (^TRT130 + ^TRT133 + ^TRT134 + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144) > 0 AND OutsideRange((^TRT143 + ^TRT85 + ^TRT98 + ^TRT142), (Sum(^TRT217) + ^TRT233 + Sum(^TRT221) + ^TRT237 + Sum(^TRT226) + ^TRT242), 1)

    Technical Business Rule Format:
            (^TRT130 + ^TRT133 + ^TRT134 + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144) > 0 AND OutsideRange((^TRT143 + ^TRT85 + ^TRT98 + ^TRT142), (Sum(^TRT217) + ^TRT233 + Sum(^TRT221) + ^TRT237 + Sum(^TRT226) + ^TRT242), 1)

Data Elements:
    
    ^TRT130 = TRT:RP:BusinessDetails:Income:Income.Net.Amount
    
    ^TRT85 = TRT:RP:BusinessDetails:Income:ShareOfCredits:IncomeTax.FrankingCredits.FrankingCreditShareReceivedIndirectly.Amount
    
    ^TRT98 = TRT:RP:BusinessDetails:Income:Dividends:IncomeTax.FrankingCredits.Amount
    
    ^TRT133 = TRT:RP:BusinessDetails:Income:CapitalGains:Income.CapitalGainsNet.Amount
    
    ^TRT134 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:ListedCountry:Income.InternationalDealings.Attributed.Amount
    
    ^TRT136 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:UnlistedCountry:Income.InternationalDealings.Attributed.Amount
    
    ^TRT141 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:Income.Net.Amount
    
    ^TRT142 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:Income.InternationalDealings.TaxOffset.Amount
    
    ^TRT143 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
    
    ^TRT144 = TRT:RP:BusinessDetails:Income:TaxLossesDeducted:IncomeTax.Deduction.TaxLossesDeducted.Amount
    
    ^TRT217 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
    
    ^TRT221 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.FrankingCredits.ReceivedFromAustralianCompanies.Amount
    
    ^TRT226 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Income.InternationalDealings.TaxOffset.Amount
    
    ^TRT233 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
    
    ^TRT237 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.FrankingCredits.ReceivedFromAustralianCompanies.Amount
    
    ^TRT242 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:Income.InternationalDealings.TaxOffset.Amount
    */
    VRATOTRT433095();
    #endregion // VR.ATO.TRT.433095

    #region VR.ATO.TRT.433096

/*  VR.ATO.TRT.433096
            When the sum of all Australian franking credits from a New Zealand company - Beneficiary share, Share of income - Primary production - Beneficiary share, Share of income - Non-primary production - Beneficiary share, Franked distributions - Beneficiary share, Capital gains - Beneficiary share, Attributed foreign income - Beneficiary share, and Other assessable foreign source income - Beneficiary share is less than 0, then Amount of capital gains on which the trustee has chosen to be assessed on behalf of beneficiaries must be provided

        Legacy Rule Format:
            (Sum(^TRT217) + Sum(^TRT218) + Sum(^TRT219) + Sum(^TRT442) + Sum(^TRT223) + Sum(^TRT224) + Sum(^TRT225)) < 0 AND ^TRT446 = NULL

    Technical Business Rule Format:
            (Sum(^TRT217) + Sum(^TRT218) + Sum(^TRT219) + Sum(^TRT442) + Sum(^TRT223) + Sum(^TRT224) + Sum(^TRT225)) < 0 AND ^TRT446 = NULL

Data Elements:
    
    ^TRT446 = TRT:RP:CapitalGainsAssessed:CapitalGainsTax.TrusteeAssessedCapitalGain.Amount
    
    ^TRT217 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
    
    ^TRT218 = TRT:RP:StatementOfDistributionToBeneficiary:Information:ShareOfIncomePrimaryProduction:Income.BeneficiaryShare.Amount
    
    ^TRT219 = TRT:RP:StatementOfDistributionToBeneficiary:Information:ShareOfIncomeNonPrimaryProduction:Income.BeneficiaryShare.Amount
    
    ^TRT223 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Income.CapitalGainsNet.Amount
    
    ^TRT224 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.AttributedForeignIncome.Amount
    
    ^TRT225 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Income.InternationalDealings.Net.Amount
    
    ^TRT442 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Income.DistributionFranked.Amount
    */
    VRATOTRT433096();
    #endregion // VR.ATO.TRT.433096

    #region VR.ATO.TRT.433098

/*  VR.ATO.TRT.433098
            When sum of net Australian income/loss plus net capital gains, plus attributed foreign income plus other assessable foreign source income minus allowable tax losses of earlier income years deducted this year is less than or equal to zero, then the sum of all Franked distributions - Beneficiary shares plus Franked distributions - No beneficiary presently entitled must be less than or equal to zero

        Legacy Rule Format:
            ^TRT130 + ^TRT133 + ^TRT134  + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144 <= 0 AND Sum(^TRT442) + ^TRT444 > 0

    Technical Business Rule Format:
            ^TRT130 + ^TRT133 + ^TRT134  + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144 <= 0 AND Sum(^TRT442) + ^TRT444 > 0

Data Elements:
    
    ^TRT444 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:Income.DistributionFranked.Amount
    
    ^TRT130 = TRT:RP:BusinessDetails:Income:Income.Net.Amount
    
    ^TRT133 = TRT:RP:BusinessDetails:Income:CapitalGains:Income.CapitalGainsNet.Amount
    
    ^TRT134 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:ListedCountry:Income.InternationalDealings.Attributed.Amount
    
    ^TRT136 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:UnlistedCountry:Income.InternationalDealings.Attributed.Amount
    
    ^TRT141 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:Income.Net.Amount
    
    ^TRT143 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
    
    ^TRT144 = TRT:RP:BusinessDetails:Income:TaxLossesDeducted:IncomeTax.Deduction.TaxLossesDeducted.Amount
    
    ^TRT442 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Income.DistributionFranked.Amount
    */
    VRATOTRT433098();
    #endregion // VR.ATO.TRT.433098

    #region VR.ATO.TRT.433100

/*  VR.ATO.TRT.433100
            When sum of net Australian income/loss plus net capital gains, plus attributed foreign income plus other assessable foreign source income minus allowable tax losses of earlier income years deducted this year is less than or equal to zero, then the sum of all Capital gains - Beneficiary share, plus Net capital gain - No beneficiary presently entitled, plus Amount of capital gains on which the trustee has chosen to be assessed on behalf of beneficiaries must be less than or equal to zero

        Legacy Rule Format:
            (^TRT130 + ^TRT133 + ^TRT134 + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144) <= 0 AND (Sum(^TRT223) + ^TRT239 + ^TRT446) > 0

    Technical Business Rule Format:
            (^TRT130 + ^TRT133 + ^TRT134 + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144) <= 0 AND (Sum(^TRT223) + ^TRT239 + ^TRT446) > 0

Data Elements:
    
    ^TRT239 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:Income.CapitalGainsNet.Amount
    
    ^TRT130 = TRT:RP:BusinessDetails:Income:Income.Net.Amount
    
    ^TRT133 = TRT:RP:BusinessDetails:Income:CapitalGains:Income.CapitalGainsNet.Amount
    
    ^TRT134 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:ListedCountry:Income.InternationalDealings.Attributed.Amount
    
    ^TRT136 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:UnlistedCountry:Income.InternationalDealings.Attributed.Amount
    
    ^TRT141 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:Income.Net.Amount
    
    ^TRT143 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
    
    ^TRT144 = TRT:RP:BusinessDetails:Income:TaxLossesDeducted:IncomeTax.Deduction.TaxLossesDeducted.Amount
    
    ^TRT223 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Income.CapitalGainsNet.Amount
    
    ^TRT446 = TRT:RP:CapitalGainsAssessed:CapitalGainsTax.TrusteeAssessedCapitalGain.Amount
    */
    VRATOTRT433100();
    #endregion // VR.ATO.TRT.433100

    #region VR.ATO.TRT.433105

/*  VR.ATO.TRT.433105
            When the sum of net Australian income/loss plus net capital gains, plus attributed foreign income plus other assessable foreign source income minus allowable tax losses of earlier income years deducted this year is greater than zero, then the sum of all Attributed foreign income - Beneficiary share, plus Attributed foreign income - No beneficiary presently entitled must equal the sum of Attributed foreign income- Listed country plus Attributed foreign income- Unlisted country, with a tolerance of 1

        Legacy Rule Format:
            ^TRT130 + ^TRT133 + ^TRT134  + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144 > 0 AND OutsideRange(Sum(^TRT224) + ^TRT240, ^TRT134 + ^TRT136, 1)

    Technical Business Rule Format:
            ^TRT130 + ^TRT133 + ^TRT134  + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144 > 0 AND OutsideRange(Sum(^TRT224) + ^TRT240, ^TRT134 + ^TRT136, 1)

Data Elements:
    
    ^TRT134 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:ListedCountry:Income.InternationalDealings.Attributed.Amount
    
    ^TRT130 = TRT:RP:BusinessDetails:Income:Income.Net.Amount
    
    ^TRT133 = TRT:RP:BusinessDetails:Income:CapitalGains:Income.CapitalGainsNet.Amount
    
    ^TRT136 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:UnlistedCountry:Income.InternationalDealings.Attributed.Amount
    
    ^TRT141 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:Income.Net.Amount
    
    ^TRT143 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
    
    ^TRT144 = TRT:RP:BusinessDetails:Income:TaxLossesDeducted:IncomeTax.Deduction.TaxLossesDeducted.Amount
    
    ^TRT224 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.AttributedForeignIncome.Amount
    
    ^TRT240 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.AttributedForeignIncome.Amount
    */
    VRATOTRT433105();
    #endregion // VR.ATO.TRT.433105

    #region VR.ATO.TRT.433106

/*  VR.ATO.TRT.433106
            When sum of net Australian income/loss plus net capital gains, plus attributed foreign income plus other assessable foreign source income minus allowable tax losses of earlier income years deducted this year is less than or equal to zero, then the sum of all Attributed foreign income - Beneficiary shares plus Attributed foreign income - No beneficiary presently entitled must be less than or equal to zero

        Legacy Rule Format:
            ^TRT130 + ^TRT133 + ^TRT134  + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144 <= 0 AND Sum(^TRT224) + ^TRT240 > 0

    Technical Business Rule Format:
            ^TRT130 + ^TRT133 + ^TRT134  + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144 <= 0 AND Sum(^TRT224) + ^TRT240 > 0

Data Elements:
    
    ^TRT240 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.AttributedForeignIncome.Amount
    
    ^TRT130 = TRT:RP:BusinessDetails:Income:Income.Net.Amount
    
    ^TRT133 = TRT:RP:BusinessDetails:Income:CapitalGains:Income.CapitalGainsNet.Amount
    
    ^TRT134 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:ListedCountry:Income.InternationalDealings.Attributed.Amount
    
    ^TRT136 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:UnlistedCountry:Income.InternationalDealings.Attributed.Amount
    
    ^TRT141 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:Income.Net.Amount
    
    ^TRT143 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
    
    ^TRT144 = TRT:RP:BusinessDetails:Income:TaxLossesDeducted:IncomeTax.Deduction.TaxLossesDeducted.Amount
    
    ^TRT224 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.AttributedForeignIncome.Amount
    */
    VRATOTRT433106();
    #endregion // VR.ATO.TRT.433106

    #region VR.ATO.TRT.433107

/*  VR.ATO.TRT.433107
            When sum of net Australian income/loss plus net capital gains, plus attributed foreign income plus other assessable foreign source income minus allowable tax losses of earlier income years deducted this year is less than or equal to zero, then the sum of all Other assessable foreign source income - Beneficiary shares plus Other assessable foreign source income - No beneficiary presently entitled must be less than or equal to zero

        Legacy Rule Format:
            ^TRT130 + ^TRT133 + ^TRT134  + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144 <= 0 AND Sum(^TRT225) + ^TRT241 > 0

    Technical Business Rule Format:
            ^TRT130 + ^TRT133 + ^TRT134  + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144 <= 0 AND Sum(^TRT225) + ^TRT241 > 0

Data Elements:
    
    ^TRT241 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:Income.InternationalDealings.Net.Amount
    
    ^TRT130 = TRT:RP:BusinessDetails:Income:Income.Net.Amount
    
    ^TRT133 = TRT:RP:BusinessDetails:Income:CapitalGains:Income.CapitalGainsNet.Amount
    
    ^TRT134 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:ListedCountry:Income.InternationalDealings.Attributed.Amount
    
    ^TRT136 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:UnlistedCountry:Income.InternationalDealings.Attributed.Amount
    
    ^TRT141 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:Income.Net.Amount
    
    ^TRT143 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
    
    ^TRT144 = TRT:RP:BusinessDetails:Income:TaxLossesDeducted:IncomeTax.Deduction.TaxLossesDeducted.Amount
    
    ^TRT225 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Income.InternationalDealings.Net.Amount
    */
    VRATOTRT433107();
    #endregion // VR.ATO.TRT.433107

    #region VR.ATO.TRT.W00009

/*  VR.ATO.TRT.W00009
            The total amount at NCMI from primary production (beneficiary and no beneficiary presently entitled) cannot exceed the sum of NCMI from primary production - Business income, NCMI from primary production - Distribution from partnerships and NCMI from primary production - Share of net income from trusts.

        Legacy Rule Format:
            (Sum(^TRT537) + ^TRT543) > (^TRT523 + ^TRT527 + ^TRT529)

    Technical Business Rule Format:
            (Sum(^TRT537) + ^TRT543) > (^TRT523 + ^TRT527 + ^TRT529)

Data Elements:
    
    ^TRT523 = TRT:RP:BusinessDetails:Income:NCMIBusinessIncome:NCMIBusinessIncomePrimaryProduction:Income.NonConcessionalManagedInvestmentTrust.Amount
    
    ^TRT527 = TRT:RP:BusinessDetails:Income:NCMIPartnershipsAndTrusts:NCMIPartnershipsAndTrustsPrimaryProduction:Income.NonConcessionalManagedInvestmentTrust.Amount
    
    ^TRT529 = TRT:RP:BusinessDetails:Income:NCMIPartnershipsAndTrusts:NCMIPartnershipsAndTrustsPrimaryProduction:PrimaryProductionShareOfNetIncome:Income.NonConcessionalManagedInvestmentTrust.Amount
    
    ^TRT537 = TRT:RP:StatementOfDistributionToBeneficiary:Information:ShareOfIncomePrimaryProduction:Income.NonConcessionalManagedInvestmentTrust.Amount
    
    ^TRT543 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:PrimaryProductionNoBeneficiaryEntitled:Income.NonConcessionalManagedInvestmentTrust.Amount
    */
    VRATOTRTW00009();
    #endregion // VR.ATO.TRT.W00009

    #region VR.ATO.TRT.W00010

/*  VR.ATO.TRT.W00010
            The total amount at Excluded from NCMI from primary production (beneficiary and no beneficiary presently entitled) cannot exceed the sum of excluded from NCMI from primary production - Business income, excluded from NCMI from primary production - Distribution from partnerships and excluded from NCMI from primary production - Share of net income from trusts.

        Legacy Rule Format:
            (Sum(^TRT538) + ^TRT544) > (^TRT524 + ^TRT528 + ^TRT530)

    Technical Business Rule Format:
            (Sum(^TRT538) + ^TRT544) > (^TRT524 + ^TRT528 + ^TRT530)

Data Elements:
    
    ^TRT524 = TRT:RP:BusinessDetails:Income:NCMIBusinessIncome:NCMIBusinessIncomePrimaryProduction:Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount
    
    ^TRT528 = TRT:RP:BusinessDetails:Income:NCMIPartnershipsAndTrusts:NCMIPartnershipsAndTrustsPrimaryProduction:Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount
    
    ^TRT530 = TRT:RP:BusinessDetails:Income:NCMIPartnershipsAndTrusts:NCMIPartnershipsAndTrustsPrimaryProduction:PrimaryProductionShareOfNetIncome:Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount
    
    ^TRT538 = TRT:RP:StatementOfDistributionToBeneficiary:Information:ShareOfIncomePrimaryProduction:Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount
    
    ^TRT544 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:PrimaryProductionNoBeneficiaryEntitled:Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount
    */
    VRATOTRTW00010();
    #endregion // VR.ATO.TRT.W00010

    #region VR.ATO.TRT.W00011

/*  VR.ATO.TRT.W00011
            The total amount at NCMI from non-primary production (beneficiary and no beneficiary presently entitled) cannot exceed the sum of NCMI from non-primary production - Business income, NCMI from non-primary production - Distribution from partnerships less foreign income and NCMI from non-primary production - Share of net income from trusts less net capital gain, foreign income and franked distributions.

        Legacy Rule Format:
            (Sum(^TRT539) + ^TRT545) > (^TRT525 + ^TRT531 + ^TRT533)

    Technical Business Rule Format:
            (Sum(^TRT539) + ^TRT545) > (^TRT525 + ^TRT531 + ^TRT533)

Data Elements:
    
    ^TRT525 = TRT:RP:BusinessDetails:Income:NCMIBusinessIncome:NCMIBusinessIncomeNonPrimaryProduction:Income.NonConcessionalManagedInvestmentTrust.Amount
    
    ^TRT531 = TRT:RP:BusinessDetails:Income:NCMIPartnershipsAndTrusts:NCMIPartnershipsAndTrustsNonPrimaryProduction:Income.NonConcessionalManagedInvestmentTrust.Amount
    
    ^TRT533 = TRT:RP:BusinessDetails:Income:NCMIPartnershipsAndTrusts:NCMIPartnershipsAndTrustsNonPrimaryProduction:NonPrimaryProductionShareOfNetIncome:Income.NonConcessionalManagedInvestmentTrust.Amount
    
    ^TRT539 = TRT:RP:StatementOfDistributionToBeneficiary:Information:ShareOfIncomeNonPrimaryProduction:Income.NonConcessionalManagedInvestmentTrust.Amount
    
    ^TRT545 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:NonPrimaryProductionNoBeneficiaryEntitled:Income.NonConcessionalManagedInvestmentTrust.Amount
    */
    VRATOTRTW00011();
    #endregion // VR.ATO.TRT.W00011

    #region VR.ATO.TRT.W00012

/*  VR.ATO.TRT.W00012
            The total amount at Excluded from NCMI from non-primary production (beneficiary and no beneficiary presently entitled) cannot exceed the sum of excluded from NCMI from non-primary production - Business income, excluded from NCMI from non-primary production - Distribution from partnerships less foreign income and excluded from NCMI from non-primary production - Share of net income from trusts less net capital gain, foreign income and franked distributions.

        Legacy Rule Format:
            (Sum(^TRT540) + ^TRT546) > (^TRT526 + ^TRT532 + ^TRT534)

    Technical Business Rule Format:
            (Sum(^TRT540) + ^TRT546) > (^TRT526 + ^TRT532 + ^TRT534)

Data Elements:
    
    ^TRT526 = TRT:RP:BusinessDetails:Income:NCMIBusinessIncome:NCMIBusinessIncomeNonPrimaryProduction:Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount
    
    ^TRT532 = TRT:RP:BusinessDetails:Income:NCMIPartnershipsAndTrusts:NCMIPartnershipsAndTrustsNonPrimaryProduction:Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount
    
    ^TRT534 = TRT:RP:BusinessDetails:Income:NCMIPartnershipsAndTrusts:NCMIPartnershipsAndTrustsNonPrimaryProduction:NonPrimaryProductionShareOfNetIncome:Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount
    
    ^TRT540 = TRT:RP:StatementOfDistributionToBeneficiary:Information:ShareOfIncomeNonPrimaryProduction:Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount
    
    ^TRT546 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:NonPrimaryProductionNoBeneficiaryEntitled:Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount
    */
    VRATOTRTW00012();
    #endregion // VR.ATO.TRT.W00012

    #region VR.ATO.TRT.W00013

/*  VR.ATO.TRT.W00013
            The total amount at capital gains NCMI (beneficiary and no beneficiary presently entitled) cannot exceed net capital gains NCMI.

        Legacy Rule Format:
            (Sum(^TRT541) + ^TRT547) > ^TRT535

    Technical Business Rule Format:
            (Sum(^TRT541) + ^TRT547) > ^TRT535

Data Elements:
    
    ^TRT535 = TRT:RP:BusinessDetails:Income:NCMICapitalGains:Capital.Gains.NonConcessionalManagedInvestmentTrustIncome.Amount
    
    ^TRT541 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Capital.Gains.NonConcessionalManagedInvestmentTrustIncome.Amount
    
    ^TRT547 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:Capital.Gains.NonConcessionalManagedInvestmentTrustIncome.Amount
    */
    VRATOTRTW00013();
    #endregion // VR.ATO.TRT.W00013

    #region VR.ATO.TRT.W00014

/*  VR.ATO.TRT.W00014
            The total amount at capital gains Excluded from NCMI (beneficiary and no beneficiary presently entitled) cannot exceed net capital gains Excluded from NCMI.

        Legacy Rule Format:
            (Sum(^TRT542) + ^TRT548) > ^TRT536

    Technical Business Rule Format:
            (Sum(^TRT542) + ^TRT548) > ^TRT536

Data Elements:
    
    ^TRT536 = TRT:RP:BusinessDetails:Income:NCMICapitalGains:Capital.Gains.ExcludedFromNonConcessionalManagedInvestmentTrustIncome.Amount
    
    ^TRT542 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Capital.Gains.ExcludedFromNonConcessionalManagedInvestmentTrustIncome.Amount
    
    ^TRT548 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:Capital.Gains.ExcludedFromNonConcessionalManagedInvestmentTrustIncome.Amount
    */
    VRATOTRTW00014();
    #endregion // VR.ATO.TRT.W00014
#endregion Manual Rules - Rules marked for manual coding
                VRATOTRT432013();
                VRATOTRT432014();
                VRATOTRT432015();
                VRATOTRT432024();
                VRATOTRT432025();
                VRATOTRT432026();
                VRATOTRT432027();
                VRATOTRT432028();
                VRATOTRT432029();
                VRATOTRT432030();
                VRATOTRT432031();
                VRATOTRT432038();
                VRATOTRT432039();
                VRATOTRT432041();
                VRATOTRT432045();
                VRATOTRT432046();
                VRATOTRT432048();
                VRATOTRT432052();
                VRATOTRT432054();
                VRATOTRT432055();
                VRATOTRT432070();
                VRATOTRT432078();
                VRATOTRT432079();
                VRATOTRT432080();
                VRATOTRT432081();
                VRATOTRT432083();
                VRATOTRT432085();
                VRATOTRT432086();
                VRATOTRT432087();
                VRATOTRT432091();
                VRATOTRT432092();
                VRATOTRT432097();
                VRATOTRT432098();
                VRATOTRT432100();
                VRATOTRT432103();
                VRATOTRT432105();
                VRATOTRT432107();
                VRATOTRT432118();
                VRATOTRT432119();
                VRATOTRT432129();
                VRATOTRT432131();
                VRATOTRT432133();
                VRATOTRT432134();
                VRATOTRT432137();
                VRATOTRT432138();
                VRATOTRT432140();
                VRATOTRT432148();
                VRATOTRT432180();
                VRATOTRT432192();
                VRATOTRT432199();
                VRATOTRT432202();
                VRATOTRT432203();
                VRATOTRT432207();
                VRATOTRT432227();
                VRATOTRT432230();
                VRATOTRT432234();
                VRATOTRT432235();
                VRATOTRT432238();
                VRATOTRT432240();
                VRATOTRT432241();
                VRATOTRT432243();
                VRATOTRT432245();
                VRATOTRT432247();
                VRATOTRT432259();
                VRATOTRT432263();
                VRATOTRT432277();
                VRATOTRT432285();
                VRATOTRT432290();
                VRATOTRT432307();
                VRATOTRT432311();
                VRATOTRT432314();
                VRATOTRT432321();
                VRATOTRT432325();
                VRATOTRT432335();
                VRATOTRT432337();
                VRATOTRT432339();
                VRATOTRT432352();
                VRATOTRT432353();
                VRATOTRT432384();
                VRATOTRT432419();
                VRATOTRT432421();
                VRATOTRT432562();
                VRATOTRT432567();
                VRATOTRT432572();
                VRATOTRT432593();
                VRATOTRT432600();
                VRATOTRT432607();
                VRATOTRT432609();
                VRATOTRT432610();
                VRATOTRT432612();
                VRATOTRT432631();
                VRATOTRT432633();
                VRATOTRT432634();
                VRATOTRT432694();
                VRATOTRT432713();
                VRATOTRT432730();
                VRATOTRT432731();
                VRATOTRT432786();
                VRATOTRT432801();
                VRATOTRT432802();
                VRATOTRT432805();
                VRATOTRT432863();
                VRATOTRT432888();
                VRATOTRT432894();
                VRATOTRT432896();
                VRATOTRT432920();
                VRATOTRT432921();
                VRATOTRT432924();
                VRATOTRT432928();
                VRATOTRT432929();
                VRATOTRT432930();
                VRATOTRT432931();
                VRATOTRT432932();
                VRATOTRT432933();
                VRATOTRT432934();
                VRATOTRT432935();
                VRATOTRT432936();
                VRATOTRT432937();
                VRATOTRT432938();
                VRATOTRT432939();
                VRATOTRT432940();
                VRATOTRT432941();
                VRATOTRT432942();
                VRATOTRT432943();
                VRATOTRT432944();
                VRATOTRT432945();
                VRATOTRT432946();
                VRATOTRT432947();
                VRATOTRT432948();
                VRATOTRT432949();
                VRATOTRT432950();
                VRATOTRT432951();
                VRATOTRT432952();
                VRATOTRT432953();
                VRATOTRT432954();
                VRATOTRT432955();
                VRATOTRT432956();
                VRATOTRT432958();
                VRATOTRT432959();
                VRATOTRT432960();
                VRATOTRT432961();
                VRATOTRT432962();
                VRATOTRT432963();
                VRATOTRT432964();
                VRATOTRT432965();
                VRATOTRT432966();
                VRATOTRT432967();
                VRATOTRT432969();
                VRATOTRT432970();
                VRATOTRT432971();
                VRATOTRT432972();
                VRATOTRT432973();
                VRATOTRT432974();
                VRATOTRT432975();
                VRATOTRT432976();
                VRATOTRT432977();
                VRATOTRT432979();
                VRATOTRT432980();
                VRATOTRT432981();
                VRATOTRT432982();
                VRATOTRT432983();
                VRATOTRT432984();
                VRATOTRT432985();
                VRATOTRT432986();
                VRATOTRT432987();
                VRATOTRT432988();
                VRATOTRT432989();
                VRATOTRT432990();
                VRATOTRT432991();
                VRATOTRT432992();
                VRATOTRT432993();
                VRATOTRT432994();
                VRATOTRT432995();
                VRATOTRT432996();
                VRATOTRT432997();
                VRATOTRT432999();
                VRATOTRT433041();
                VRATOTRT433042();
                VRATOTRT433051();
                VRATOTRT433052();
                VRATOTRT433053();
                VRATOTRT433054();
                VRATOTRT433055();
                VRATOTRT433056();
                VRATOTRT433057();
                VRATOTRT433058();
                VRATOTRT433059();
                VRATOTRT433060();
                VRATOTRT433061();
                VRATOTRT433062();
                VRATOTRT433063();
                VRATOTRT433064();
                VRATOTRT433065();
                VRATOTRT433066();
                VRATOTRT433067();
                VRATOTRT433068();
                VRATOTRT433069();
                VRATOTRT433070();
                VRATOTRT433071();
                VRATOTRT433072();
                VRATOTRT433073();
                VRATOTRT433074();
                VRATOTRT433075();
                VRATOTRT433076();
                VRATOTRT433077();
                VRATOTRT433078();
                VRATOTRT433079();
                VRATOTRT433080();
                VRATOTRT433085();
                VRATOTRT433086();
                VRATOTRT433087();
                VRATOTRT433088();
                VRATOTRT433089();
                VRATOTRT433090();
                VRATOTRT433091();
                VRATOTRT433093();
                VRATOTRT433094();
                VRATOTRT433097();
                VRATOTRT433104();
                VRATOTRT433108();
                VRATOTRT433109();
                VRATOTRT433110();
                VRATOTRT442893();
                VRATOTRTW00001();
                VRATOTRTW00002();
                VRATOTRTW00003();
                VRATOTRTW00004();
                VRATOTRTW00005();
                VRATOTRTW00006();
                VRATOTRTW00007();
                VRATOTRTW00008();
                VRATOTRTW00015();
                VRATOTRTW00016();
                VRATOTRTW00019();

            #region Repeating report.RP_StatementOfDistributionToBeneficiaryCollection  
            if (report.RP_StatementOfDistributionToBeneficiaryCollection != null)
            {    
                int itemIndex2 = 0;
                foreach (TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary in report.RP_StatementOfDistributionToBeneficiaryCollection)
                {
                    itemIndex2++;
                        VRATOTRT432022(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT432280(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT432281(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT432282(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT432283(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT432284(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT432286(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT432287(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT432289(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT432292(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT432293(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT432294(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT432295(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT432296(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT432300(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT432302(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT432303(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT432318(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT432347(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT432348(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT432582(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT432583(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT432584(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT432585(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT432586(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT433002(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT433003(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT433004(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT433005(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT433006(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT433007(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT433008(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT433009(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT433010(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT433011(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT433012(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT433013(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT433014(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT433015(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT433016(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT433017(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT433018(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT433019(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT433020(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT433021(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT433022(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT433023(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT433024(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT433025(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT433026(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT433027(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT433028(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT433029(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT433030(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT433031(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT433032(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT433033(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT433034(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT433035(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT433036(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT433037(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT433038(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT433039(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT433040(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT433101(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT433102(statementOfDistributionToBeneficiary, itemIndex2);
                        VRATOTRT433103(statementOfDistributionToBeneficiary, itemIndex2);
                    }
                }
        
                #endregion // Foreach loop

            #region Repeating report.RP_BusinessDetails_Income_EmploymentTerminationPaymentDetailsCollection  
            if (report.RP_BusinessDetails_Income_EmploymentTerminationPaymentDetailsCollection != null)
            {    
                int itemIndex4 = 0;
                foreach (TRT2024.RP_BusinessDetails_Income_EmploymentTerminationPaymentDetails employmentTerminationPaymentDetails in report.RP_BusinessDetails_Income_EmploymentTerminationPaymentDetailsCollection)
                {
                    itemIndex4++;
                        VRATOTRT432420(employmentTerminationPaymentDetails, itemIndex4);
                    }
                }
        
                #endregion // Foreach loop

            #region Repeating report.RP_BusinessDetails_Income_IndividualNonBusinessWithholdingPaymentDetailsCollection  
            if (report.RP_BusinessDetails_Income_IndividualNonBusinessWithholdingPaymentDetailsCollection != null)
            {    
                int itemIndex4 = 0;
                foreach (TRT2024.RP_BusinessDetails_Income_IndividualNonBusinessWithholdingPaymentDetails individualNonBusinessWithholdingPaymentDetails in report.RP_BusinessDetails_Income_IndividualNonBusinessWithholdingPaymentDetailsCollection)
                {
                    itemIndex4++;
                        VRATOTRT432425(individualNonBusinessWithholdingPaymentDetails, itemIndex4);
                        VRATOTRT433000(individualNonBusinessWithholdingPaymentDetails, itemIndex4);
                    }
                }
        
                #endregion // Foreach loop

            #region Repeating report.InT_Declaration_IntermediaryContactCollection  
            if (report.InT_Declaration_IntermediaryContactCollection != null)
            {    
                int itemIndex3 = 0;
                foreach (TRT2024.InT_Declaration_IntermediaryContact intermediaryContact in report.InT_Declaration_IntermediaryContactCollection)
                {
                    itemIndex3++;
                        VRATOTRT433081(intermediaryContact, itemIndex3);
                    }
                }
        
                #endregion // Foreach loop

            foreach (ProcessMessageDocument currentProcessMessage in response)
            {
                if (currentProcessMessage.Parameters != null)
                {
                    foreach (ProcessMessageParameter currentParameter in currentProcessMessage.Parameters)
                    {
                        if (string.IsNullOrEmpty(currentParameter.Name))
                        {
                            currentParameter.Name = _emptyParameterValue;
                        }

                        if (currentParameter.Name.Length > _maxParameterNameLength)
                        {
                            currentParameter.Name = currentParameter.Name.Substring(0, _maxParameterNameLength - 1);
                        }

                        if (string.IsNullOrEmpty(currentParameter.Value))
                        {
                            currentParameter.Value = _emptyParameterValue;
                        }

                        if (currentParameter.Value.Length > _maxParameterValueLength)
                        {
                            currentParameter.Value = currentParameter.Value.Substring(0, _maxParameterValueLength - 1);
                        }
                    }
                }
            }

            return response;
        }
				
			
    
      #region VR.ATO.TRT.432013
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432013()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432013
            Non-individual name AND Family name are both present. Only one name can be present.
    
            Legacy Rule Format:
            ^TRT24 <> NULL AND ^TRT20 <> NULL

            Technical Business Rule Format:
            ^TRT24 <> NULL AND ^TRT20 <> NULL
    
            Data Elements:
    
            ^TRT20 = TRT:RP:MailRecipient:PersonNameDetails:PersonNameDetails.FamilyName.Text
    
            ^TRT24 = TRT:RP:MailRecipient:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.TRT24 != null && report.TRT20 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432013", Severity = ProcessMessageSeverity.Error,
                    Description = @"Name of individual AND name of company must not both be present",
                    Location = "/tns:TRT/tns:RP/tns:MailRecipient/tns:PersonNameDetails/tns:FamilyNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432013"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT24", Value = GetValueOrEmpty(report.TRT24) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT20", Value = report.TRT20 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432014
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432014()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432014
            Previous postal address is only required when an address change has occurred since the last return. Delete previous postal address or input the old address if a change of address has occurred.
    
            Legacy Rule Format:
            ^TRT14 = ^TRT9 AND ^TRT312 = ^TRT306 AND ^TRT15 = ^TRT10 AND ^TRT16 = ^TRT11 AND ^TRT17 = ^TRT12

            Technical Business Rule Format:
            ^TRT14 = ^TRT9 AND ^TRT312 = ^TRT306 AND ^TRT15 = ^TRT10 AND ^TRT16 = ^TRT11 AND ^TRT17 = ^TRT12
    
            Data Elements:
    
            ^TRT14 = TRT:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.Line1.Text
    
            ^TRT9 = TRT:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.Line1.Text
    
            ^TRT10 = TRT:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.LocalityName.Text
    
            ^TRT11 = TRT:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.StateOrTerritory.Code
    
            ^TRT12 = TRT:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.Postcode.Text
    
            ^TRT15 = TRT:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.LocalityName.Text
    
            ^TRT16 = TRT:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.StateOrTerritory.Code
    
            ^TRT17 = TRT:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.Postcode.Text
    
            ^TRT306 = TRT:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.Line2.Text
    
            ^TRT312 = TRT:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.Line2.Text
            */
            assertion = (report.TRT14 == report.TRT9 && report.TRT312 == report.TRT306 && report.TRT15 == report.TRT10 && report.TRT16 == report.TRT11 && report.TRT17 == report.TRT12);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430009", Severity = ProcessMessageSeverity.Error,
                    Description = @"Previous postal address must not be the same as current postal address",
                    LongDescription = @"Previous postal address is only required when an address change has occurred since the last return.  Delete previous postal address or input the old address if a change of address has occurred.",
                    Location = "/tns:TRT/tns:RP/tns:PreviousTrust/tns:PreviousPostalAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432014"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT14", Value = report.TRT14 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT9", Value = report.TRT9 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT312", Value = GetValueOrEmpty(report.TRT312) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT306", Value = GetValueOrEmpty(report.TRT306) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT15", Value = report.TRT15 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT10", Value = report.TRT10 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT16", Value = GetValueOrEmpty(report.TRT16) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT11", Value = GetValueOrEmpty(report.TRT11) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT17", Value = GetValueOrEmpty(report.TRT17) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT12", Value = GetValueOrEmpty(report.TRT12) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432015
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432015()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432015
            Family name or non-individual name must be present.
    
            Legacy Rule Format:
            ^TRT20 = NULL AND ^TRT24 = NULL

            Technical Business Rule Format:
            ^TRT20 = NULL AND ^TRT24 = NULL
    
            Data Elements:
    
            ^TRT24 = TRT:RP:MailRecipient:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text
    
            ^TRT20 = TRT:RP:MailRecipient:PersonNameDetails:PersonNameDetails.FamilyName.Text
            */
            assertion = (report.TRT20 == null && report.TRT24 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432015", Severity = ProcessMessageSeverity.Error,
                    Description = @"Name of the trustee to whom notices should be sent must be present",
                    LongDescription = @"If the trustee is an individual, provide the name of the individual. If the trustee is a company, provide the name of the company",
                    Location = "/tns:TRT/tns:RP/tns:MailRecipient/tns:OrganisationNameDetails/tns:OrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432015"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT20", Value = report.TRT20 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT24", Value = GetValueOrEmpty(report.TRT24) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432022
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432022(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.432022
                    In a Statement of Distribution, if the Assessment calculation code is 120 or 140, then there must be an amount present at 's98(4) assessable amount'
    
                    Legacy Rule Format:
                    (^TRT214 = '120' OR ^TRT214 = '140') AND ^TRT229 = NULL
        
                    Technical Business Rule Format:
                    (^TRT214 = '120' OR ^TRT214 = '140') AND ^TRT229 = NULL
            
                    Data Elements:
            
                    ^TRT229 = TRT:RP:StatementOfDistributionToBeneficiary:Information:NonResidentAdditionalInformation:InternationalDealings.AssessableIncomeNonResidentBeneficiaryTrustee.Amount
            
                    ^TRT214 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.AssessmentCalculation.Code
                    */
                    assertion = ((statementOfDistributionToBeneficiary.TRT214 == @"120" || statementOfDistributionToBeneficiary.TRT214 == @"140") && statementOfDistributionToBeneficiary.TRT229 == null);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.TRT.442022", Severity = ProcessMessageSeverity.Error,
                            Description = @"s98(4) assessable amount must be present",
                            LongDescription = @"In a Statement of Distribution, if the Assessment calculation code is 120 or 140, then there must be an amount present at 's98(4) assessable amount'",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:Information/tns:NonResidentAdditionalInformation/tns:InternationalDealingsAssessableIncomeNonResidentBeneficiaryTrusteeA",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432022"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT214", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT214) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT229", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT229) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.432024
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432024()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432024
            If the Type of trust is '059' (deceased estate), then the assessment calculation code for each Statement of Distribution must be 11,12,13,14, 111, 118, 119 or 120 and the assessment calculation code in the distribution statement where no beneficiary is entitled, if present, must be 15, 16 or 17.
    
            Legacy Rule Format:
            ^TRT32 = '059' AND (AnyOccurrence(^TRT1070, InSet(^TRT214, '"25", "26", "27", "28", "29", "30", "34", "35", "125", "126", "127", "128", "129", "134", "138", "139", "140"')) OR AnyOccurrence(^TRT1070, InSet(^TRT230, '"36", "37"')))

            Technical Business Rule Format:
            ^TRT32 = '059' AND (AnyOccurrence(^TRT1070, InSet(^TRT214, '"25", "26", "27", "28", "29", "30", "34", "35", "125", "126", "127", "128", "129", "134", "138", "139", "140"')) OR AnyOccurrence(^TRT1070, InSet(^TRT230, '"36", "37"')))
    
            Data Elements:
    
            ^TRT32 = TRT:RP:TypeOfTrust:OrganisationDetails.OrganisationType.Code
    
            ^TRT214 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.AssessmentCalculation.Code
    
            ^TRT230 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.AssessmentCalculation.Code
    
            ^TRT1070 = TRT:RP:StatementOfDistributionToBeneficiary
            */
            assertion = (report.TRT32 == @"059" && ((report.RP_StatementOfDistributionToBeneficiaryCollection == null ? false : report.RP_StatementOfDistributionToBeneficiaryCollection.Any(TRT1070Repeat => IsMatch(TRT1070Repeat.TRT214, @"^(25|26|27|28|29|30|34|35|125|126|127|128|129|134|138|139|140)$"))) || (report.RP_StatementOfDistributionToBeneficiaryCollection == null ? false : report.RP_StatementOfDistributionToBeneficiaryCollection.Any(TRT1070Repeat => IsMatch(report.TRT230, @"^(36|37)$")))));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432725", Severity = ProcessMessageSeverity.Error,
                    Description = @"Assessment calculation code(s) must be a Deceased trust estate code",
                    LongDescription = @"If the Type of trust is '059' (deceased estate), then the assessment calculation code for each Statement of Distribution must be 11 to 17, 119, 120 or 134 and the assessment calculation code  in the distribution statement where no beneficiary is entitled, if present, must be 36 or 37.",
                    Location = "/tns:TRT/tns:RP/tns:TypeOfTrust/tns:OrganisationDetailsOrganisationTypeC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432024"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT32", Value = report.TRT32 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT214", Value = "Count(TRT214)" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT230", Value = GetValueOrEmpty(report.TRT230) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432025
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432025()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432025
            The type of trust estate is other than D (deceased estate) on the front cover AND the assessment calculation code in the distribution statement is 11,12,13,14, 111, 118, 119, 120 or the assessment calculation code in the distribution statement (no beneficiary presently entitled) is 15 to 17. The assessment calculation code is invalid for this type of trust.
    
            Legacy Rule Format:
            ^TRT32 <> '059' AND AnyOccurrence(^TRT1070, InSet(^TRT214, '"11", "12", "13", "14", "111", "118", "119", "120"') OR InSet(^TRT230, '"15", "16", "17"'))

            Technical Business Rule Format:
            ^TRT32 <> '059' AND AnyOccurrence(^TRT1070, InSet(^TRT214, '"11", "12", "13", "14", "111", "118", "119", "120"') OR InSet(^TRT230, '"15", "16", "17"'))
    
            Data Elements:
    
            ^TRT214 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.AssessmentCalculation.Code
    
            ^TRT32 = TRT:RP:TypeOfTrust:OrganisationDetails.OrganisationType.Code
    
            ^TRT230 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.AssessmentCalculation.Code
    
            ^TRT1070 = TRT:RP:StatementOfDistributionToBeneficiary
            */
            assertion = (report.TRT32 != @"059" && (report.RP_StatementOfDistributionToBeneficiaryCollection == null ? false : report.RP_StatementOfDistributionToBeneficiaryCollection.Any(TRT1070Repeat => IsMatch(TRT1070Repeat.TRT214, @"^(11|12|13|14|111|118|119|120)$") || IsMatch(report.TRT230, @"^(15|16|17)$"))));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432025", Severity = ProcessMessageSeverity.Error,
                    Description = @"Assessment calculation code must be an Inter vivos trust estate code",
                    LongDescription = @"If the Type of trust estate is not a Deceased estate, (code '059') then the Assessment calculation codes in a beneficiary distribution statement must not be a deceased estate code",
                    Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary/tns:Information/tns:IncomeTaxAssessmentCalculationC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432025"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT32", Value = report.TRT32 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT214", Value = "Count(TRT214)" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT230", Value = GetValueOrEmpty(report.TRT230) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432026
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432026()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432026
            If the Type of trust is a special disability trust (code 203), then there must not be any 'income to which no beneficiary is presently entitled' and that Assessment calculation code must be blank.
    
            Legacy Rule Format:
            ^TRT230 <>  NULL AND ^TRT32 = '203'

            Technical Business Rule Format:
            ^TRT230 <>  NULL AND ^TRT32 = '203'
    
            Data Elements:
    
            ^TRT32 = TRT:RP:TypeOfTrust:OrganisationDetails.OrganisationType.Code
    
            ^TRT230 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.AssessmentCalculation.Code
            */
            assertion = (report.TRT230 != null && report.TRT32 == @"203");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432026", Severity = ProcessMessageSeverity.Error,
                    Description = @"Special disability trust must not have income to which no beneficiary is presently entitled",
                    LongDescription = @"If the Type of trust is a special disability trust (code 203), then there must not be any 'income to which no beneficiary is presently entitled' and that Assessment calculation code must be blank",
                    Location = "/tns:TRT/tns:RP/tns:TypeOfTrust/tns:OrganisationDetailsOrganisationTypeC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432026"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT230", Value = GetValueOrEmpty(report.TRT230) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT32", Value = report.TRT32 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432027
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432027()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432027
            If there is a Date of death, then the Type of trust must be a 'deceased estate' - code '059'
    
            Legacy Rule Format:
            ^TRT34 <> NULL AND ^TRT32 <> '059'

            Technical Business Rule Format:
            ^TRT34 <> NULL AND ^TRT32 <> '059'
    
            Data Elements:
    
            ^TRT32 = TRT:RP:TypeOfTrust:OrganisationDetails.OrganisationType.Code
    
            ^TRT34 = TRT:RP:TypeOfTrust:PersonDemographicDetails.Death.Date
            */
            assertion = (report.TRT34 != null && report.TRT32 != @"059");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432027", Severity = ProcessMessageSeverity.Error,
                    Description = @"If there is a Date of death, then the Type of trust must be a 'deceased estate' - code '059'",
                    Location = "/tns:TRT/tns:RP/tns:TypeOfTrust/tns:OrganisationDetailsOrganisationTypeC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432027"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT34", Value = GetValueOrEmptyDate(report.TRT34) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT32", Value = report.TRT32 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432028
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432028()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432028
            Assessment calculation code 45 or 145 can only be used by Special Disability Trusts. (code 203)
    
            Legacy Rule Format:
            ^TRT32 = '203' AND AnyOccurrence(^TRT1070, NotInSet(^TRT214, '"45", "145"'))

            Technical Business Rule Format:
            ^TRT32 = '203' AND AnyOccurrence(^TRT1070, NotInSet(^TRT214, '"45", "145"'))
    
            Data Elements:
    
            ^TRT32 = TRT:RP:TypeOfTrust:OrganisationDetails.OrganisationType.Code
    
            ^TRT214 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.AssessmentCalculation.Code
    
            ^TRT1070 = TRT:RP:StatementOfDistributionToBeneficiary
            */
            assertion = (report.TRT32 == @"203" && (report.RP_StatementOfDistributionToBeneficiaryCollection == null ? false : report.RP_StatementOfDistributionToBeneficiaryCollection.Any(TRT1070Repeat => !(IsMatch(TRT1070Repeat.TRT214, @"^(45|145)$",RegexOptions.IgnoreCase)))));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432028", Severity = ProcessMessageSeverity.Error,
                    Description = @"Beneficiary assessment calculation code must be '45' or '145' for special disability trust",
                    LongDescription = @"Assessment calculation code 45 or 145 can only be used by Special Disability Trusts.",
                    Location = "/tns:TRT/tns:RP/tns:TypeOfTrust/tns:OrganisationDetailsOrganisationTypeC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432028"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT32", Value = report.TRT32 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT214", Value = "Count(TRT214)" });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432029
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432029()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432029
            A beneficiary must NOT use a special disability Trust Assessment calculation code (code 45 or 145) unless the Type of trust is a special disability trust (code 203)
    
            Legacy Rule Format:
            ^TRT32 <> '203' AND AnyOccurrence(^TRT1070, InSet(^TRT214, '"45", "145"'))

            Technical Business Rule Format:
            ^TRT32 <> '203' AND AnyOccurrence(^TRT1070, InSet(^TRT214, '"45", "145"'))
    
            Data Elements:
    
            ^TRT32 = TRT:RP:TypeOfTrust:OrganisationDetails.OrganisationType.Code
    
            ^TRT214 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.AssessmentCalculation.Code
    
            ^TRT1070 = TRT:RP:StatementOfDistributionToBeneficiary
            */
            assertion = (report.TRT32 != @"203" && (report.RP_StatementOfDistributionToBeneficiaryCollection == null ? false : report.RP_StatementOfDistributionToBeneficiaryCollection.Any(TRT1070Repeat => IsMatch(TRT1070Repeat.TRT214, @"^(45|145)$"))));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432029", Severity = ProcessMessageSeverity.Error,
                    Description = @"Beneficiary assessment calculation code must not be special disability trust code",
                    LongDescription = @"A beneficiary must NOT use a special disability Trust Assessment calculation code (code 45 or 145) unless the Type of trust is a special disability trust (code 230)",
                    Location = "/tns:TRT/tns:RP/tns:TypeOfTrust/tns:OrganisationDetailsOrganisationTypeC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432029"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT32", Value = report.TRT32 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT214", Value = "Count(TRT214)" });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432030
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432030()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432030
            Date of death must not be after the financial year of the return
    
            Legacy Rule Format:
            ^TRT34 <> NULL AND ^TRT34 > ConvertToDate(30,6,^TRT1)

            Technical Business Rule Format:
            ^TRT34 <> NULL AND ^TRT34 > ConvertToDate(30,6,^TRT1)
    
            Data Elements:
    
            ^TRT34 = TRT:RP:TypeOfTrust:PersonDemographicDetails.Death.Date
    
            ^TRT1 = TRT:RP:Report.TargetFinancial.Year
            */
            assertion = (report.TRT34 != null && report.TRT34.GetValueOrDefault() > ConvertToDate(30, 6, report.TRT1.GetValueOrDefault()));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432030", Severity = ProcessMessageSeverity.Error,
                    Description = @"Date of death must not be after the financial year of the return",
                    Location = "/tns:TRT/tns:RP/tns:TypeOfTrust/tns:PersonDemographicDetailsDeathD",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432030"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT34", Value = GetValueOrEmptyDate(report.TRT34) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT1", Value = GetValueOrEmpty(report.TRT1) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432031
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432031()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432031
            Date of death must be present if the Type of trust estate is a deceased estate (code 059)
    
            Legacy Rule Format:
            ^TRT32 = '059' AND ^TRT34 = NULL

            Technical Business Rule Format:
            ^TRT32 = '059' AND ^TRT34 = NULL
    
            Data Elements:
    
            ^TRT34 = TRT:RP:TypeOfTrust:PersonDemographicDetails.Death.Date
    
            ^TRT32 = TRT:RP:TypeOfTrust:OrganisationDetails.OrganisationType.Code
            */
            assertion = (report.TRT32 == @"059" && report.TRT34 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432031", Severity = ProcessMessageSeverity.Error,
                    Description = @"Date of death must be present if the Type of trust estate is a deceased estate (code 059)",
                    Location = "/tns:TRT/tns:RP/tns:TypeOfTrust/tns:PersonDemographicDetailsDeathD",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432031"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT32", Value = report.TRT32 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT34", Value = GetValueOrEmptyDate(report.TRT34) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432038
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432038()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432038
            If Status of Business code is present, the Description of main business activity AND Industry code are also required
    
            Legacy Rule Format:
            ^TRT43 <> NULL AND (^TRT42 = NULL OR ^TRT41 = NULL)

            Technical Business Rule Format:
            ^TRT43 <> NULL AND (^TRT42 = NULL OR ^TRT41 = NULL)
    
            Data Elements:
    
            ^TRT41 = TRT:RP:BusinessDetails:OrganisationDetails.MainIncomeActivity.Description
    
            ^TRT42 = TRT:RP:BusinessDetails:OrganisationDetails.OrganisationIndustry2006Extended.Code
    
            ^TRT43 = TRT:RP:BusinessDetails:OrganisationDetails.ActivityEvent.Code
            */
            assertion = (report.TRT43 != null && (report.TRT42 == null || report.TRT41 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430016", Severity = ProcessMessageSeverity.Error,
                    Description = @"Description of main business and industry code must be present",
                    LongDescription = @"If Status of Business code is present, the Description of main business activity and Industry code are also required",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:OrganisationDetailsMainIncomeActivityDe",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432038"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT43", Value = GetValueOrEmpty(report.TRT43) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT42", Value = GetValueOrEmpty(report.TRT42) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT41", Value = GetValueOrEmpty(report.TRT41) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432039
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432039()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432039
            Description of main business activity AND total business income must be present
    
            Legacy Rule Format:
            ^TRT42 <> NULL AND (^TRT41 = NULL OR (^TRT46 = NULL AND ^TRT47 = NULL AND ^TRT49 = NULL AND ^TRT50 = NULL AND ^TRT52 = NULL AND ^TRT54 = NULL AND ^TRT55 = NULL))

            Technical Business Rule Format:
            ^TRT42 <> NULL AND (^TRT41 = NULL OR (^TRT46 = NULL AND ^TRT47 = NULL AND ^TRT49 = NULL AND ^TRT50 = NULL AND ^TRT52 = NULL AND ^TRT54 = NULL AND ^TRT55 = NULL))
    
            Data Elements:
    
            ^TRT41 = TRT:RP:BusinessDetails:OrganisationDetails.MainIncomeActivity.Description
    
            ^TRT42 = TRT:RP:BusinessDetails:OrganisationDetails.OrganisationIndustry2006Extended.Code
    
            ^TRT46 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^TRT47 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^TRT49 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Remuneration.PaymentToForeignResidentGross.Amount
    
            ^TRT50 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^TRT52 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^TRT54 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Income.Other.Amount
    
            ^TRT55 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Income.Other.Amount
            */
            assertion = (report.TRT42 != null && (report.TRT41 == null || report.TRT46 == null && report.TRT47 == null && report.TRT49 == null && report.TRT50 == null && report.TRT52 == null && report.TRT54 == null && report.TRT55 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430015", Severity = ProcessMessageSeverity.Error,
                    Description = @"Description of main business activity and total business income must be present",
                    LongDescription = @"If an Industry code is present, then the 'Description of main business activity' and business income (Primary Production and/or Non-Primary Production) must be present",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:OrganisationDetailsMainIncomeActivityDe",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432039"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT42", Value = GetValueOrEmpty(report.TRT42) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT41", Value = GetValueOrEmpty(report.TRT41) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT46", Value = GetValueOrEmpty(report.TRT46) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT47", Value = GetValueOrEmpty(report.TRT47) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT49", Value = GetValueOrEmpty(report.TRT49) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT50", Value = GetValueOrEmpty(report.TRT50) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT52", Value = GetValueOrEmpty(report.TRT52) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT54", Value = GetValueOrEmpty(report.TRT54) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT55", Value = GetValueOrEmpty(report.TRT55) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432041
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432041()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432041
            If business income (primary production or non-primary production) is present, an Industry code (ANZSIC) must be present
    
            Legacy Rule Format:
            (^TRT46 <> NULL OR ^TRT47 <> NULL OR ^TRT49 <> NULL OR ^TRT50 <> NULL OR ^TRT52 <> NULL OR ^TRT54 <> NULL OR ^TRT55 <> NULL) AND ^TRT42 = NULL

            Technical Business Rule Format:
            (^TRT46 <> NULL OR ^TRT47 <> NULL OR ^TRT49 <> NULL OR ^TRT50 <> NULL OR ^TRT52 <> NULL OR ^TRT54 <> NULL OR ^TRT55 <> NULL) AND ^TRT42 = NULL
    
            Data Elements:
    
            ^TRT42 = TRT:RP:BusinessDetails:OrganisationDetails.OrganisationIndustry2006Extended.Code
    
            ^TRT46 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^TRT47 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^TRT49 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Remuneration.PaymentToForeignResidentGross.Amount
    
            ^TRT50 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^TRT52 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^TRT54 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Income.Other.Amount
    
            ^TRT55 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Income.Other.Amount
            */
            assertion = ((report.TRT46 != null || report.TRT47 != null || report.TRT49 != null || report.TRT50 != null || report.TRT52 != null || report.TRT54 != null || report.TRT55 != null) && report.TRT42 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432041", Severity = ProcessMessageSeverity.Error,
                    Description = @"If business income (primary production or non-primary production) is present, an Industry code (ANZSIC) must be present",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:OrganisationDetailsOrganisationIndustry2006ExtendedC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432041"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT46", Value = GetValueOrEmpty(report.TRT46) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT47", Value = GetValueOrEmpty(report.TRT47) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT49", Value = GetValueOrEmpty(report.TRT49) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT50", Value = GetValueOrEmpty(report.TRT50) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT52", Value = GetValueOrEmpty(report.TRT52) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT54", Value = GetValueOrEmpty(report.TRT54) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT55", Value = GetValueOrEmpty(report.TRT55) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT42", Value = GetValueOrEmpty(report.TRT42) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432045
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432045()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432045
            If Net primary production income/loss amount is present, then total business income from primary production (Gross payments where ABN not quoted, Assessable government industry payments or Other business income) must be present
    
            Legacy Rule Format:
            ^TRT71 <> NULL AND ^TRT46 = NULL AND ^TRT50 = NULL AND ^TRT54 = NULL

            Technical Business Rule Format:
            ^TRT71 <> NULL AND ^TRT46 = NULL AND ^TRT50 = NULL AND ^TRT54 = NULL
    
            Data Elements:
    
            ^TRT46 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^TRT50 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^TRT54 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Income.Other.Amount
    
            ^TRT71 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Income.Net.Amount
            */
            assertion = (report.TRT71 != null && report.TRT46 == null && report.TRT50 == null && report.TRT54 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432686", Severity = ProcessMessageSeverity.Error,
                    Description = @"PP business income must be present",
                    LongDescription = @"Net primary production income/loss amount is present without total business income from primary production (Gross payments where ABN not quoted, Assessable government industry payments, Other business income).",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:BusinessIncomeAndExpenses/tns:IncomePrimaryProduction/tns:RemunerationABNNotQuotedPaymentGrossA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432045"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT71", Value = GetValueOrEmpty(report.TRT71) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT46", Value = GetValueOrEmpty(report.TRT46) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT50", Value = GetValueOrEmpty(report.TRT50) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT54", Value = GetValueOrEmpty(report.TRT54) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432046
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432046()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432046
            If Business name of main business or street address of main business is present, then total business income (primary production or non-primary production) must be present
    
            Legacy Rule Format:
            (^TRT161 <> NULL OR ^TRT162 <> NULL) AND ^TRT46 = NULL AND ^TRT47 = NULL AND ^TRT49 = NULL AND ^TRT50 = NULL AND ^TRT52 = NULL AND ^TRT54 = NULL AND ^TRT55 = NULL

            Technical Business Rule Format:
            (^TRT161 <> NULL OR ^TRT162 <> NULL) AND ^TRT46 = NULL AND ^TRT47 = NULL AND ^TRT49 = NULL AND ^TRT50 = NULL AND ^TRT52 = NULL AND ^TRT54 = NULL AND ^TRT55 = NULL
    
            Data Elements:
    
            ^TRT46 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^TRT47 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^TRT49 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Remuneration.PaymentToForeignResidentGross.Amount
    
            ^TRT50 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^TRT52 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^TRT54 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Income.Other.Amount
    
            ^TRT55 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Income.Other.Amount
    
            ^TRT161 = TRT:RP:BusinessAndProfessionalInformation:OrganisationNameDetails.OrganisationalName.Text
    
            ^TRT162 = TRT:RP:BusinessAndProfessionalInformation:BusinessAddressDetails:AddressDetails.Line1.Text
            */
            assertion = ((report.TRT161 != null || report.TRT162 != null) && report.TRT46 == null && report.TRT47 == null && report.TRT49 == null && report.TRT50 == null && report.TRT52 == null && report.TRT54 == null && report.TRT55 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430018", Severity = ProcessMessageSeverity.Error,
                    Description = @"Total business income amount required",
                    LongDescription = @"Business name of main business or street address of main business is present and total business income (primary production or non-primary production) is missing at business income and expenses.",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:BusinessIncomeAndExpenses/tns:IncomePrimaryProduction/tns:RemunerationABNNotQuotedPaymentGrossA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432046"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT161", Value = report.TRT161 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT162", Value = report.TRT162 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT46", Value = GetValueOrEmpty(report.TRT46) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT47", Value = GetValueOrEmpty(report.TRT47) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT49", Value = GetValueOrEmpty(report.TRT49) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT50", Value = GetValueOrEmpty(report.TRT50) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT52", Value = GetValueOrEmpty(report.TRT52) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT54", Value = GetValueOrEmpty(report.TRT54) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT55", Value = GetValueOrEmpty(report.TRT55) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432048
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432048()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432048
            Net non-primary production income/loss amount is present without total business income non-primary production (Gross payments where ABN not quoted, Gross payments subject to foreign resident withholding (excluding capital gains), Assessable government industry payments, Other business income).
    
            Legacy Rule Format:
            ^TRT72 <> NULL AND ^TRT47 = NULL AND ^TRT49 = NULL AND ^TRT52 = NULL AND ^TRT55 = NULL

            Technical Business Rule Format:
            ^TRT72 <> NULL AND ^TRT47 = NULL AND ^TRT49 = NULL AND ^TRT52 = NULL AND ^TRT55 = NULL
    
            Data Elements:
    
            ^TRT47 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^TRT49 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Remuneration.PaymentToForeignResidentGross.Amount
    
            ^TRT52 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^TRT55 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Income.Other.Amount
    
            ^TRT72 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Income.Net.Amount
            */
            assertion = (report.TRT72 != null && report.TRT47 == null && report.TRT49 == null && report.TRT52 == null && report.TRT55 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432748", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-PP business income must be present",
                    LongDescription = @"Net non-primary production income/loss amount is present without total business income non-primary production (Gross payments where ABN not quoted, Gross payments subject to foreign resident withholding (excluding capital gains), Assessable government industry payments, Other business income).",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:BusinessIncomeAndExpenses/tns:IncomeNonPrimaryProduction/tns:RemunerationABNNotQuotedPaymentGrossA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432048"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT72", Value = GetValueOrEmpty(report.TRT72) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT47", Value = GetValueOrEmpty(report.TRT47) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT49", Value = GetValueOrEmpty(report.TRT49) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT52", Value = GetValueOrEmpty(report.TRT52) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT55", Value = GetValueOrEmpty(report.TRT55) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432052
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432052()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432052
            If Credit for tax withheld - foreign resident withholding (excluding capital gains) is present, then Non Primary production Gross payment subject to foreign resident withholding (excluding capital gains) must be present
    
            Legacy Rule Format:
            ^TRT474 <> NULL AND ^TRT49 = NULL

            Technical Business Rule Format:
            ^TRT474 <> NULL AND ^TRT49 = NULL
    
            Data Elements:
    
            ^TRT49 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Remuneration.PaymentToForeignResidentGross.Amount
    
            ^TRT474 = TRT:RP:BusinessDetails:Income:PayAsYouGoWithholding:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromForeignResidents.Amount
            */
            assertion = (report.TRT474 != null && report.TRT49 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432750", Severity = ProcessMessageSeverity.Error,
                    Description = @"NPP Gross payments subject to foreign resident withholding (excluding capital gains) must be present",
                    LongDescription = @"If Credit for tax withheld - foreign resident withholding (excluding capital gains) is present, then Non Primary production Gross payment subject to foreign resident withholding (excluding capital gains) must be present.",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:BusinessIncomeAndExpenses/tns:IncomeNonPrimaryProduction/tns:RemunerationPaymentToForeignResidentGrossA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432052"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT474", Value = GetValueOrEmpty(report.TRT474) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT49", Value = GetValueOrEmpty(report.TRT49) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432054
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432054()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432054
            If Primary production - Assessable government industry payments action code is present, Primary production - Assessable government industry payments amount must be present AND not zero
    
            Legacy Rule Format:
            ^TRT361 = TRUE AND (^TRT50 = 0 OR ^TRT50 = NULL)

            Technical Business Rule Format:
            ^TRT361 = TRUE AND (^TRT50 = 0 OR ^TRT50 = NULL)
    
            Data Elements:
    
            ^TRT50 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^TRT361 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Indicator
            */
            assertion = (report.TRT361 == true && (report.TRT50 == 0 || report.TRT50 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430030", Severity = ProcessMessageSeverity.Error,
                    Description = @"PP Assessable government industry payments amount must be present",
                    LongDescription = @"If Primary production - Assessable government industry payments action code is present, Primary production - Assessable government industry payments amount must be present and not zero",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:BusinessIncomeAndExpenses/tns:IncomePrimaryProduction/tns:GovernmentFundingGovernmentIndustryPaymentsAssessableA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432054"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT361", Value = GetValueOrEmpty(report.TRT361) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT50", Value = GetValueOrEmpty(report.TRT50) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432055
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432055()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432055
            If Non-primary production - Assessable government industry payments action code is present, Non-primary production - Assessable government industry payments must be present AND not zero
    
            Legacy Rule Format:
            ^TRT362 = TRUE AND (^TRT52 = 0 OR ^TRT52 = NULL)

            Technical Business Rule Format:
            ^TRT362 = TRUE AND (^TRT52 = 0 OR ^TRT52 = NULL)
    
            Data Elements:
    
            ^TRT52 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^TRT362 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Indicator
            */
            assertion = (report.TRT362 == true && (report.TRT52 == 0 || report.TRT52 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430032", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-PP Assessable government industry payments amount must be present",
                    LongDescription = @"If Non-primary production - Assessable government industry payments action code is present, Non-primary production - Assessable government industry payments must be present and not zero",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:BusinessIncomeAndExpenses/tns:IncomeNonPrimaryProduction/tns:GovernmentFundingGovernmentIndustryPaymentsAssessableA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432055"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT362", Value = GetValueOrEmpty(report.TRT362) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT52", Value = GetValueOrEmpty(report.TRT52) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432070
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432070()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432070
            The sum of deductions for 'Small business entity simplified depreciation' must not exceed the 'Depreciation expenses' amount
    
            Legacy Rule Format:
            ^TRT65 < (^TRT189 + ^TRT190)

            Technical Business Rule Format:
            ^TRT65 < (^TRT189 + ^TRT190)
    
            Data Elements:
    
            ^TRT65 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:BusinessExpenses:Expense.DepreciationAndAmortisation.Amount
    
            ^TRT189 = TRT:RP:BusinessAndProfessionalInformation:LowCostDeduction:Expense.DepreciatingAssets.SmallBusinessEntity.Amount
    
            ^TRT190 = TRT:RP:BusinessAndProfessionalInformation:GeneralPoolDeduction:Expense.DepreciatingAssets.SmallBusinessEntity.Amount
            */
            assertion = (report.TRT65.GetValueOrDefault() < report.TRT189.GetValueOrDefault() + report.TRT190.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432687", Severity = ProcessMessageSeverity.Error,
                    Description = @"Small business entity simplified depreciation must not exceed Depreciation expenses",
                    LongDescription = @"The sum of deductions for 'Small business entity simplified depreciation' must not exceed the 'Depreciation expenses' amount",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:BusinessIncomeAndExpenses/tns:BusinessExpenses/tns:DepreciationAndAmortisationA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432070"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT65", Value = GetValueOrEmpty(report.TRT65) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT189", Value = GetValueOrEmpty(report.TRT189) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT190", Value = GetValueOrEmpty(report.TRT190) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432078
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432078()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432078
            If Total business income from primary production (Assessable government industry payments, other business income) is present, Primary production Net primary production income or loss from business amount must be present
    
            Legacy Rule Format:
            (^TRT46 <> NULL OR ^TRT50 <> NULL OR ^TRT54 <> NULL) AND ^TRT71 = NULL

            Technical Business Rule Format:
            (^TRT46 <> NULL OR ^TRT50 <> NULL OR ^TRT54 <> NULL) AND ^TRT71 = NULL
    
            Data Elements:
    
            ^TRT71 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Income.Net.Amount
    
            ^TRT46 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^TRT50 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^TRT54 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Income.Other.Amount
            */
            assertion = ((report.TRT46 != null || report.TRT50 != null || report.TRT54 != null) && report.TRT71 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432688", Severity = ProcessMessageSeverity.Error,
                    Description = @"Primary Production Net income or loss from business must be present",
                    LongDescription = @"If Total business income from primary production (Assessable government industry payments, other business income) is present, Primary production Net primary production income or loss from business amount must be present",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:BusinessIncomeAndExpenses/tns:IncomePrimaryProduction/tns:NetA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432078"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT46", Value = GetValueOrEmpty(report.TRT46) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT50", Value = GetValueOrEmpty(report.TRT50) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT54", Value = GetValueOrEmpty(report.TRT54) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT71", Value = GetValueOrEmpty(report.TRT71) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432079
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432079()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432079
            The total of Primary production and Non-primary production Net business income must equal the total of all income amounts, plus Reconciliation items, minus the sum of all expenses amounts
    
            Legacy Rule Format:
            ((^TRT46 + ^TRT47 + ^TRT49 + ^TRT50 + ^TRT52 + ^TRT54 + ^TRT55) - (^TRT56 + ^TRT57 + ^TRT58 + ^TRT467 + ^TRT468 + ^TRT61 + ^TRT469 + ^TRT470 + ^TRT471 + ^TRT65 + ^TRT472 + ^TRT473 + ^TRT68) + ^TRT69 + ^TRT447) <> (^TRT71 + ^TRT72)

            Technical Business Rule Format:
            ((^TRT46 + ^TRT47 + ^TRT49 + ^TRT50 + ^TRT52 + ^TRT54 + ^TRT55) - (^TRT56 + ^TRT57 + ^TRT58 + ^TRT467 + ^TRT468 + ^TRT61 + ^TRT469 + ^TRT470 + ^TRT471 + ^TRT65 + ^TRT472 + ^TRT473 + ^TRT68) + ^TRT69 + ^TRT447) <> (^TRT71 + ^TRT72)
    
            Data Elements:
    
            ^TRT71 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Income.Net.Amount
    
            ^TRT46 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^TRT47 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^TRT49 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Remuneration.PaymentToForeignResidentGross.Amount
    
            ^TRT50 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^TRT52 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^TRT54 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Income.Other.Amount
    
            ^TRT55 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Income.Other.Amount
    
            ^TRT56 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:BusinessExpenses:Expense.ForeignResidentWithholding.Amount
    
            ^TRT57 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:BusinessExpenses:Remuneration.WagesAndSalaries.PaymentsContractorAndConsultant.Amount
    
            ^TRT58 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:BusinessExpenses:SuperannuationContribution.EmployerContributions.Amount
    
            ^TRT61 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:BusinessExpenses:Expense.Operating.LeaseTotal.Amount
    
            ^TRT65 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:BusinessExpenses:Expense.DepreciationAndAmortisation.Amount
    
            ^TRT68 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:BusinessExpenses:Expense.DeductibleOtherTotal.Amount
    
            ^TRT69 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:Income.ReconciliationAdjustmentTotal.Amount
    
            ^TRT72 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Income.Net.Amount
    
            ^TRT447 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:BusinessExpenses:Expense.ReconciliationAdjustmentTotal.Amount
    
            ^TRT467 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:BusinessExpenses:Expense.Operating.CostOfSales.Amount
    
            ^TRT468 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:BusinessExpenses:Expense.Operating.BadDebts.Amount
    
            ^TRT469 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:BusinessExpenses:Expense.Operating.Rent.Amount
    
            ^TRT470 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:BusinessExpenses:Expense.Operating.Interest.Amount
    
            ^TRT471 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:BusinessExpenses:Expense.Royalties.Amount
    
            ^TRT472 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:BusinessExpenses:Expense.Operating.MotorVehicle.Amount
    
            ^TRT473 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:BusinessExpenses:Expense.Operating.RepairsAndMaintenance.Amount
            */
            assertion = (report.TRT46.GetValueOrDefault() + report.TRT47.GetValueOrDefault() + report.TRT49.GetValueOrDefault() + report.TRT50.GetValueOrDefault() + report.TRT52.GetValueOrDefault() + report.TRT54.GetValueOrDefault() + report.TRT55.GetValueOrDefault() - (report.TRT56.GetValueOrDefault() + report.TRT57.GetValueOrDefault() + report.TRT58.GetValueOrDefault() + report.TRT467.GetValueOrDefault() + report.TRT468.GetValueOrDefault() + report.TRT61.GetValueOrDefault() + report.TRT469.GetValueOrDefault() + report.TRT470.GetValueOrDefault() + report.TRT471.GetValueOrDefault() + report.TRT65.GetValueOrDefault() + report.TRT472.GetValueOrDefault() + report.TRT473.GetValueOrDefault() + report.TRT68.GetValueOrDefault()) + report.TRT69.GetValueOrDefault() + report.TRT447.GetValueOrDefault() != report.TRT71.GetValueOrDefault() + report.TRT72.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430050", Severity = ProcessMessageSeverity.Error,
                    Description = @"Total Net income or loss must equal total Income, plus Reconciliation items, minus total Expenses",
                    LongDescription = @"The total of Primary production and Non-primary production Net business income must equal the total of all income amounts, plus Reconciliation items, minus the sum of all expenses amounts",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:BusinessIncomeAndExpenses/tns:IncomePrimaryProduction/tns:NetA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432079"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT46", Value = GetValueOrEmpty(report.TRT46) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT47", Value = GetValueOrEmpty(report.TRT47) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT49", Value = GetValueOrEmpty(report.TRT49) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT50", Value = GetValueOrEmpty(report.TRT50) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT52", Value = GetValueOrEmpty(report.TRT52) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT54", Value = GetValueOrEmpty(report.TRT54) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT55", Value = GetValueOrEmpty(report.TRT55) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT56", Value = GetValueOrEmpty(report.TRT56) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT57", Value = GetValueOrEmpty(report.TRT57) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT58", Value = GetValueOrEmpty(report.TRT58) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT467", Value = GetValueOrEmpty(report.TRT467) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT468", Value = GetValueOrEmpty(report.TRT468) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT61", Value = GetValueOrEmpty(report.TRT61) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT469", Value = GetValueOrEmpty(report.TRT469) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT470", Value = GetValueOrEmpty(report.TRT470) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT471", Value = GetValueOrEmpty(report.TRT471) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT65", Value = GetValueOrEmpty(report.TRT65) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT472", Value = GetValueOrEmpty(report.TRT472) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT473", Value = GetValueOrEmpty(report.TRT473) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT68", Value = GetValueOrEmpty(report.TRT68) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT69", Value = GetValueOrEmpty(report.TRT69) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT447", Value = GetValueOrEmpty(report.TRT447) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT71", Value = GetValueOrEmpty(report.TRT71) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT72", Value = GetValueOrEmpty(report.TRT72) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432080
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432080()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432080
            If Tax withheld where ABN not quoted is present or Credit for tax withheld - Foreign resident withholding (excluding capital gains) is present, then Net income or loss from business must be present
    
            Legacy Rule Format:
            (^TRT73 <> NULL OR ^TRT474 <> NULL) AND ^TRT71 = NULL AND ^TRT72 = NULL

            Technical Business Rule Format:
            (^TRT73 <> NULL OR ^TRT474 <> NULL) AND ^TRT71 = NULL AND ^TRT72 = NULL
    
            Data Elements:
    
            ^TRT71 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Income.Net.Amount
    
            ^TRT72 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Income.Net.Amount
    
            ^TRT73 = TRT:RP:BusinessDetails:Income:PayAsYouGoWithholding:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount
    
            ^TRT474 = TRT:RP:BusinessDetails:Income:PayAsYouGoWithholding:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromForeignResidents.Amount
            */
            assertion = ((report.TRT73 != null || report.TRT474 != null) && report.TRT71 == null && report.TRT72 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432752", Severity = ProcessMessageSeverity.Error,
                    Description = @"Net income or loss from business must be present",
                    LongDescription = @"If Tax withheld where ABN not quoted is present or Credit for tax withheld - Foreign resident withholding is present (excluding capital gains), then Net income or loss from business must be present.",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:BusinessIncomeAndExpenses/tns:IncomePrimaryProduction/tns:NetA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432080"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT73", Value = GetValueOrEmpty(report.TRT73) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT474", Value = GetValueOrEmpty(report.TRT474) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT71", Value = GetValueOrEmpty(report.TRT71) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT72", Value = GetValueOrEmpty(report.TRT72) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432081
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432081()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432081
            If an amount, greater than zero, is present in the distribution statement at share of income primary production, there must also be an amount greater than zero at primary production income or loss from business or primary production distribution from partnerships or primary production share of net income from trusts.
    
            Legacy Rule Format:
            (AnyOccurrence(^TRT1070, ^TRT218 > 0) OR AnyOccurrence(^TRT1070, ^TRT234 > 0)) AND ^TRT71 = NULL AND ^TRT76 = NULL AND ^TRT77 = NULL

            Technical Business Rule Format:
            (AnyOccurrence(^TRT1070, ^TRT218 > 0) OR AnyOccurrence(^TRT1070, ^TRT234 > 0)) AND ^TRT71 = NULL AND ^TRT76 = NULL AND ^TRT77 = NULL
    
            Data Elements:
    
            ^TRT71 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Income.Net.Amount
    
            ^TRT76 = TRT:RP:BusinessDetails:Income:PartnershipsTrustsPrimaryProduction:Income.PartnershipDistributionGross.Amount
    
            ^TRT77 = TRT:RP:BusinessDetails:Income:PartnershipsTrustsPrimaryProduction:Income.TrustShareNet.Amount
    
            ^TRT218 = TRT:RP:StatementOfDistributionToBeneficiary:Information:ShareOfIncomePrimaryProduction:Income.BeneficiaryShare.Amount
    
            ^TRT234 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:PrimaryProductionNoBeneficiaryEntitled:IncomeTax.IncomeOrLossTaxable.Amount
    
            ^TRT1070 = TRT:RP:StatementOfDistributionToBeneficiary
            */
            assertion = (((report.RP_StatementOfDistributionToBeneficiaryCollection == null ? false : report.RP_StatementOfDistributionToBeneficiaryCollection.Any(TRT1070Repeat => TRT1070Repeat.TRT218.GetValueOrDefault() > 0)) || (report.RP_StatementOfDistributionToBeneficiaryCollection == null ? false : report.RP_StatementOfDistributionToBeneficiaryCollection.Any(TRT1070Repeat => report.TRT234.GetValueOrDefault() > 0))) && report.TRT71 == null && report.TRT76 == null && report.TRT77 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432689", Severity = ProcessMessageSeverity.Error,
                    Description = @"PP income/loss from business and/or PP distribution from partnerships or share of net income from trusts must be present",
                    LongDescription = @"If an amount greater than zero is present in a distribution statement at Share of income: primary production, then there must also be an amount greater than zero at primary production income or loss from business, or primary production distribution from partnerships, or primary production share of net income from trusts",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:BusinessIncomeAndExpenses/tns:IncomePrimaryProduction/tns:NetA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432081"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT218", Value = "TRT218" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT234", Value = "TRT234" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT71", Value = GetValueOrEmpty(report.TRT71) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT76", Value = GetValueOrEmpty(report.TRT76) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT77", Value = GetValueOrEmpty(report.TRT77) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432083
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432083()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432083
            Total business income from non-primary production (Gross payments where ABN not quoted, Gross payments subject to foreign resident withholding (excluding capital gains), Assessable government industry payments, Other business income) is present without net non-primary production income/loss amount.
    
            Legacy Rule Format:
            ^TRT72 = NULL AND (^TRT47 <> NULL OR ^TRT49 <> NULL OR ^TRT52 <> NULL OR ^TRT55 <> NULL)

            Technical Business Rule Format:
            ^TRT72 = NULL AND (^TRT47 <> NULL OR ^TRT49 <> NULL OR ^TRT52 <> NULL OR ^TRT55 <> NULL)
    
            Data Elements:
    
            ^TRT72 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Income.Net.Amount
    
            ^TRT47 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^TRT49 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Remuneration.PaymentToForeignResidentGross.Amount
    
            ^TRT52 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^TRT55 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Income.Other.Amount
            */
            assertion = (report.TRT72 == null && (report.TRT47 != null || report.TRT49 != null || report.TRT52 != null || report.TRT55 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432751", Severity = ProcessMessageSeverity.Error,
                    Description = @"Net non-PP income must be present",
                    LongDescription = @"Total business income from non-primary production (Gross payments where ABN not quoted, Gross payments subject to foreign resident withholding (excluding capital gains), Assessable government industry payments, Other business income) is present without net non-primary production income/loss amount.",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:BusinessIncomeAndExpenses/tns:IncomeNonPrimaryProduction/tns:NetA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432083"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT72", Value = GetValueOrEmpty(report.TRT72) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT47", Value = GetValueOrEmpty(report.TRT47) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT49", Value = GetValueOrEmpty(report.TRT49) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT52", Value = GetValueOrEmpty(report.TRT52) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT55", Value = GetValueOrEmpty(report.TRT55) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432085
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432085()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432085
            If either Primary production or Non-primary production Gross payments where ABN not quoted are greater than zero, then an amount at Tax withheld where ABN not quoted must be present
    
            Legacy Rule Format:
            (^TRT46 > 0 OR ^TRT47 > 0) AND ^TRT73 = NULL

            Technical Business Rule Format:
            (^TRT46 > 0 OR ^TRT47 > 0) AND ^TRT73 = NULL
    
            Data Elements:
    
            ^TRT73 = TRT:RP:BusinessDetails:Income:PayAsYouGoWithholding:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount
    
            ^TRT46 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^TRT47 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Remuneration.ABNNotQuotedPaymentGross.Amount
            */
            assertion = ((report.TRT46.GetValueOrDefault() > 0 || report.TRT47.GetValueOrDefault() > 0) && report.TRT73 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430058", Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax withheld where ABN not quoted must be present",
                    LongDescription = @"If either Primary production or Non-primary production Gross payments where ABN not quoted are greater than zero, then an amount at Tax withheld where ABN not quoted must be present",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:PayAsYouGoWithholding/tns:TaxCreditForTaxWithheldWhereABNNotQuotedA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432085"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT46", Value = GetValueOrEmpty(report.TRT46) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT47", Value = GetValueOrEmpty(report.TRT47) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT73", Value = GetValueOrEmpty(report.TRT73) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432086
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432086()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432086
            Tax withheld where ABN not quoted cannot be greater than 50% of Gross payments (PP AND Non-PP) where ABN not quoted.
    
            Legacy Rule Format:
            ^TRT73 > 0 AND ^TRT73 > (((^TRT46 + ^TRT47) * 0.50) + 1)

            Technical Business Rule Format:
            ^TRT73 > 0 AND ^TRT73 > (((^TRT46 + ^TRT47) * 0.50) + 1)
    
            Data Elements:
    
            ^TRT73 = TRT:RP:BusinessDetails:Income:PayAsYouGoWithholding:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount
    
            ^TRT46 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^TRT47 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Remuneration.ABNNotQuotedPaymentGross.Amount
            */
            assertion = (report.TRT73.GetValueOrDefault() > 0 && report.TRT73.GetValueOrDefault() > (report.TRT46.GetValueOrDefault() + report.TRT47.GetValueOrDefault()) * 0.50M + 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430056", Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax withheld where ABN not quoted cannot be greater than 50% of Gross payments (PP and Non-PP) where ABN not quoted.",
                    LongDescription = @"Tax withheld where ABN not quoted cannot be greater than 50% of Gross payments (Primary production and Non-primary production) where ABN not quoted.",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:PayAsYouGoWithholding/tns:TaxCreditForTaxWithheldWhereABNNotQuotedA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432086"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT73", Value = GetValueOrEmpty(report.TRT73) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT46", Value = GetValueOrEmpty(report.TRT46) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT47", Value = GetValueOrEmpty(report.TRT47) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432087
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432087()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432087
            If Tax withheld where ABN not quoted is present there must be an amount at Gross payments where ABN not quoted for either Primary production or Non-primary production
    
            Legacy Rule Format:
            ^TRT73 <> NULL AND ^TRT46 = NULL AND ^TRT47 = NULL

            Technical Business Rule Format:
            ^TRT73 <> NULL AND ^TRT46 = NULL AND ^TRT47 = NULL
    
            Data Elements:
    
            ^TRT73 = TRT:RP:BusinessDetails:Income:PayAsYouGoWithholding:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount
    
            ^TRT46 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^TRT47 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Remuneration.ABNNotQuotedPaymentGross.Amount
            */
            assertion = (report.TRT73 != null && report.TRT46 == null && report.TRT47 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430057", Severity = ProcessMessageSeverity.Error,
                    Description = @"PP or Non-PP Gross payments where ABN not quoted must be present",
                    LongDescription = @"If Tax withheld where ABN not quoted is present there must be an amount at Gross payments where ABN not quoted for either Primary production or Non-primary production",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:PayAsYouGoWithholding/tns:TaxCreditForTaxWithheldWhereABNNotQuotedA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432087"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT73", Value = GetValueOrEmpty(report.TRT73) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT46", Value = GetValueOrEmpty(report.TRT46) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT47", Value = GetValueOrEmpty(report.TRT47) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432091
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432091()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432091
            Credit for tax withheld - foreign resident withholding (excluding capital gains) is greater than 50% of Non PP Gross payments subject to foreign resident withholding (excluding capital gains).
    
            Legacy Rule Format:
            ^TRT474 > 0 AND ^TRT474 > ((^TRT49 * 0.50) + 1)

            Technical Business Rule Format:
            ^TRT474 > 0 AND ^TRT474 > ((^TRT49 * 0.50) + 1)
    
            Data Elements:
    
            ^TRT474 = TRT:RP:BusinessDetails:Income:PayAsYouGoWithholding:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromForeignResidents.Amount
    
            ^TRT49 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Remuneration.PaymentToForeignResidentGross.Amount
            */
            assertion = (report.TRT474.GetValueOrDefault() > 0 && report.TRT474.GetValueOrDefault() > report.TRT49.GetValueOrDefault() * 0.50M + 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432744", Severity = ProcessMessageSeverity.Error,
                    Description = @"Credit for tax withheld - foreign resident withholding (excluding capital gains) must not exceed 50% of Non PP gross payments",
                    LongDescription = @"Credit for tax withheld - foreign resident withholding (excluding capital gains) must not exceed 50% of Non-primary production Gross payments subject to foreign resident withholding (excluding capital gains).",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:PayAsYouGoWithholding/tns:TaxCreditForAmountsWithheldFromForeignResidentsA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432091"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT474", Value = GetValueOrEmpty(report.TRT474) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT49", Value = GetValueOrEmpty(report.TRT49) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432092
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432092()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432092
            If Non-primary production Gross payments subject to foreign resident withholding (excluding capital gains) is greater than zero, then there must be an amount at Credit for tax withheld - foreign resident withholding (excluding capital gains)
    
            Legacy Rule Format:
            ^TRT49 > 0 AND ^TRT474 = NULL

            Technical Business Rule Format:
            ^TRT49 > 0 AND ^TRT474 = NULL
    
            Data Elements:
    
            ^TRT474 = TRT:RP:BusinessDetails:Income:PayAsYouGoWithholding:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromForeignResidents.Amount
    
            ^TRT49 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Remuneration.PaymentToForeignResidentGross.Amount
            */
            assertion = (report.TRT49.GetValueOrDefault() > 0 && report.TRT474 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432745", Severity = ProcessMessageSeverity.Error,
                    Description = @"Credit for tax withheld - foreign resident withholding (excluding capital gains) must be present",
                    LongDescription = @"If Non-primary production Gross payments subject to foreign resident withholding (excluding capital gains) is greater than zero, then there must be an amount at Credit for tax withheld - foreign resident withholding.",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:PayAsYouGoWithholding/tns:TaxCreditForAmountsWithheldFromForeignResidentsA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432092"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT49", Value = GetValueOrEmpty(report.TRT49) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT474", Value = GetValueOrEmpty(report.TRT474) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432097
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432097()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432097
            If there is an amount present at 'Primary production - Deductions relating to distribution', then there must also be an amount at 'Primary production - Distribution from partnerships' or 'Primary production - Share of net income from trusts'
    
            Legacy Rule Format:
            ^TRT79 > 0 AND ^TRT76 = NULL AND ^TRT77 = NULL

            Technical Business Rule Format:
            ^TRT79 > 0 AND ^TRT76 = NULL AND ^TRT77 = NULL
    
            Data Elements:
    
            ^TRT76 = TRT:RP:BusinessDetails:Income:PartnershipsTrustsPrimaryProduction:Income.PartnershipDistributionGross.Amount
    
            ^TRT77 = TRT:RP:BusinessDetails:Income:PartnershipsTrustsPrimaryProduction:Income.TrustShareNet.Amount
    
            ^TRT79 = TRT:RP:BusinessDetails:Income:PartnershipsTrustsPrimaryProduction:Expense.DistributionDeductionsOther.Amount
            */
            assertion = (report.TRT79.GetValueOrDefault() > 0 && report.TRT76 == null && report.TRT77 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432690", Severity = ProcessMessageSeverity.Error,
                    Description = @"PP - Income distribution from partnerships and/or share of net income from trusts must be present",
                    LongDescription = @"If there is an amount present at 'Primary production - Deductions relating to distribution', then there must also be an amount at 'Primary production - Distribution from partnerships' or 'Primary production - Share of net income from trusts'",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:PartnershipsTrustsPrimaryProduction/tns:PartnershipDistributionGrossA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432097"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT79", Value = GetValueOrEmpty(report.TRT79) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT76", Value = GetValueOrEmpty(report.TRT76) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT77", Value = GetValueOrEmpty(report.TRT77) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432098
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432098()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432098
            If an amount is present at 'Deductions relating to distribution', 'Share of credit for tax withheld where ABN not quoted', 'Deductions relating to franked distributions from trusts', 'Share of franking credit from franked dividends', 'Share of credit for TFN amounts withheld from interest dividends and unit trust distributions', or 'Share of credit for tax withheld from foreign resident withholding (excluding capital gains)'; then there must be an amount present at 'Primary production - Distribution from partnerships', 'Primary production - Share of net income from trusts', 'Non-primary production - Distribution from partnerships less foreign income', 'Non-primary production - Share of net income from trusts less capital gains, foreign income and franked distributions' and 'Non-primary production - Franked distributions from trusts'.
    
            Legacy Rule Format:
            (^TRT84 > 0 OR ^TRT438 > 0 OR ^TRT83 > 0 OR ^TRT85 > 0 OR ^TRT86 > 0 OR ^TRT87 > 0) AND ^TRT76 = NULL AND ^TRT77 = NULL AND ^TRT80 = NULL AND ^TRT81 = NULL AND ^TRT437 = NULL

            Technical Business Rule Format:
            (^TRT84 > 0 OR ^TRT438 > 0 OR ^TRT83 > 0 OR ^TRT85 > 0 OR ^TRT86 > 0 OR ^TRT87 > 0) AND ^TRT76 = NULL AND ^TRT77 = NULL AND ^TRT80 = NULL AND ^TRT81 = NULL AND ^TRT437 = NULL
    
            Data Elements:
    
            ^TRT76 = TRT:RP:BusinessDetails:Income:PartnershipsTrustsPrimaryProduction:Income.PartnershipDistributionGross.Amount
    
            ^TRT77 = TRT:RP:BusinessDetails:Income:PartnershipsTrustsPrimaryProduction:Income.TrustShareNet.Amount
    
            ^TRT80 = TRT:RP:BusinessDetails:Income:PartnershipTrustNonPrimaryProduction:Income.PartnershipDistributionLessForeignIncome.Amount
    
            ^TRT81 = TRT:RP:BusinessDetails:Income:PartnershipTrustNonPrimaryProduction:Income.TrustShareNetExcludeNetCapitalGainsAndForeignIncomeAndDistributionFranked.Amount
    
            ^TRT83 = TRT:RP:BusinessDetails:Income:PartnershipTrustNonPrimaryProduction:Expense.DistributionDeductionsOther.Amount
    
            ^TRT84 = TRT:RP:BusinessDetails:Income:ShareOfCredits:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuotedShareReceivedIndirectly.Amount
    
            ^TRT85 = TRT:RP:BusinessDetails:Income:ShareOfCredits:IncomeTax.FrankingCredits.FrankingCreditShareReceivedIndirectly.Amount
    
            ^TRT86 = TRT:RP:BusinessDetails:Income:ShareOfCredits:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromInterestAndDividendsWhereTFNNotQuotedShareReceivedIndirectly.Amount
    
            ^TRT87 = TRT:RP:BusinessDetails:Income:ShareOfCredits:InternationalDealings.CreditForTaxWithheldFromForeignResidentWithholdingShareReceivedIndirectly.Amount
    
            ^TRT437 = TRT:RP:BusinessDetails:Income:PartnershipTrustNonPrimaryProduction:Income.TrustDistributionFranked.Amount
    
            ^TRT438 = TRT:RP:BusinessDetails:Income:PartnershipTrustNonPrimaryProduction:IncomeTax.Deduction.TrustDistributionFranked.Amount
            */
            assertion = ((report.TRT84.GetValueOrDefault() > 0 || report.TRT438.GetValueOrDefault() > 0 || report.TRT83.GetValueOrDefault() > 0 || report.TRT85.GetValueOrDefault() > 0 || report.TRT86.GetValueOrDefault() > 0 || report.TRT87.GetValueOrDefault() > 0) && report.TRT76 == null && report.TRT77 == null && report.TRT80 == null && report.TRT81 == null && report.TRT437 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432753", Severity = ProcessMessageSeverity.Error,
                    Description = @"Distribution/share of net income from partnerships or trusts must be present",
                    LongDescription = @"If an amount is present at 'Deductions relating to distribution', 'Share of credit for tax withheld where ABN not quoted', 'Deductions relating to franked distributions from trusts', 'Share of franking credit from franked dividends', 'Share of credit for TFN amounts withheld from interest dividends and unit trust distributions', or 'Share of credit for tax withheld from foreign resident withholding (excluding capital gains)'; then there must be an amount present at 'Primary production - Distribution from partnerships', 'Primary production - Share of net income from trusts', 'Non-primary production - Distribution from partnerships less foreign income', 'Non-primary production - Share of net income from trusts less capital gains, foreign income and franked distributions' and 'Non-primary production - Franked distributions from trusts'.",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:PartnershipsTrustsPrimaryProduction/tns:PartnershipDistributionGrossA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432098"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT84", Value = GetValueOrEmpty(report.TRT84) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT438", Value = GetValueOrEmpty(report.TRT438) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT83", Value = GetValueOrEmpty(report.TRT83) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT85", Value = GetValueOrEmpty(report.TRT85) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT86", Value = GetValueOrEmpty(report.TRT86) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT87", Value = GetValueOrEmpty(report.TRT87) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT76", Value = GetValueOrEmpty(report.TRT76) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT77", Value = GetValueOrEmpty(report.TRT77) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT80", Value = GetValueOrEmpty(report.TRT80) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT81", Value = GetValueOrEmpty(report.TRT81) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT437", Value = GetValueOrEmpty(report.TRT437) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432100
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432100()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432100
            If the Primary production - Share of net income from trusts action code is present, then the Primary production - Share of net income from trusts must be present AND greater than zero. An action code is not required for negative or zero amounts
    
            Legacy Rule Format:
            ^TRT353 <> NULL AND (^TRT77 = NULL OR ^TRT77 <= 0)

            Technical Business Rule Format:
            ^TRT353 <> NULL AND (^TRT77 = NULL OR ^TRT77 <= 0)
    
            Data Elements:
    
            ^TRT77 = TRT:RP:BusinessDetails:Income:PartnershipsTrustsPrimaryProduction:Income.TrustShareNet.Amount
    
            ^TRT353 = TRT:RP:BusinessDetails:Income:PartnershipsTrustsPrimaryProduction:Party.TrustType.Code
            */
            assertion = (report.TRT353 != null && (report.TRT77 == null || report.TRT77.GetValueOrDefault() <= 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432659", Severity = ProcessMessageSeverity.Error,
                    Description = @"PP - Share of net income from trusts amount must be greater than zero if action code is present",
                    LongDescription = @"If the Primary production - Share of net income from trusts action code is present, then the Primary production - Share of net income from trusts must be present and greater than zero. An action code is not required for negative or zero amounts",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:PartnershipsTrustsPrimaryProduction/tns:TrustShareNetA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432100"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT353", Value = GetValueOrEmpty(report.TRT353) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT77", Value = GetValueOrEmpty(report.TRT77) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432103
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432103()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432103
            If there is an amount at Share of franking credits from franked dividends, and Share of credit for TFN amounts withheld from interest, dividends and unit trust distributions; then there must be an amount at Non-Primary Production Distribution from partnerships less foreign income; Share of net income from trusts less capital gains, foreign income and franked distributions; Deductions relating to distribution; Franked distributions from trusts or Deductions relating to franked distributions from trusts.
    
            Legacy Rule Format:
            ^TRT85 <> NULL AND ^TRT86 <> NULL AND ^TRT80 = NULL AND ^TRT81 = NULL AND ^TRT83 = NULL AND ^TRT437 = NULL AND ^TRT438 = NULL

            Technical Business Rule Format:
            ^TRT85 <> NULL AND ^TRT86 <> NULL AND ^TRT80 = NULL AND ^TRT81 = NULL AND ^TRT83 = NULL AND ^TRT437 = NULL AND ^TRT438 = NULL
    
            Data Elements:
    
            ^TRT80 = TRT:RP:BusinessDetails:Income:PartnershipTrustNonPrimaryProduction:Income.PartnershipDistributionLessForeignIncome.Amount
    
            ^TRT81 = TRT:RP:BusinessDetails:Income:PartnershipTrustNonPrimaryProduction:Income.TrustShareNetExcludeNetCapitalGainsAndForeignIncomeAndDistributionFranked.Amount
    
            ^TRT83 = TRT:RP:BusinessDetails:Income:PartnershipTrustNonPrimaryProduction:Expense.DistributionDeductionsOther.Amount
    
            ^TRT85 = TRT:RP:BusinessDetails:Income:ShareOfCredits:IncomeTax.FrankingCredits.FrankingCreditShareReceivedIndirectly.Amount
    
            ^TRT86 = TRT:RP:BusinessDetails:Income:ShareOfCredits:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromInterestAndDividendsWhereTFNNotQuotedShareReceivedIndirectly.Amount
    
            ^TRT437 = TRT:RP:BusinessDetails:Income:PartnershipTrustNonPrimaryProduction:Income.TrustDistributionFranked.Amount
    
            ^TRT438 = TRT:RP:BusinessDetails:Income:PartnershipTrustNonPrimaryProduction:IncomeTax.Deduction.TrustDistributionFranked.Amount
            */
            assertion = (report.TRT85 != null && report.TRT86 != null && report.TRT80 == null && report.TRT81 == null && report.TRT83 == null && report.TRT437 == null && report.TRT438 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432692", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-PP distribution amount must be present",
                    LongDescription = @"If there is an amount at Share of franking credits from franked dividends, and Share of credit for TFN amounts withheld from interest, dividends and unit trust distributions; then there must be an amount at Non-Primary Production Distribution from partnerships less foreign income; Share of net income from trusts less capital gains, foreign income and franked distributions; Deductions relating to distribution; Franked distributions from trusts or Deductions relating to franked distributions from trusts.",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:PartnershipTrustNonPrimaryProduction/tns:PartnershipDistributionLessForeignIncomeA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432103"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT85", Value = GetValueOrEmpty(report.TRT85) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT86", Value = GetValueOrEmpty(report.TRT86) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT80", Value = GetValueOrEmpty(report.TRT80) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT81", Value = GetValueOrEmpty(report.TRT81) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT83", Value = GetValueOrEmpty(report.TRT83) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT437", Value = GetValueOrEmpty(report.TRT437) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT438", Value = GetValueOrEmpty(report.TRT438) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432105
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432105()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432105
            If Non-primary production - Share of net income from trusts action code is present, then Non-primary production- Share of net income from trusts less capital gains, foreign income and franked distributions must be greater than zero. A trusts action code is not required for negative or zero amounts
    
            Legacy Rule Format:
            ^TRT436 <> NULL AND (^TRT81 = NULL OR ^TRT81 <= 0)

            Technical Business Rule Format:
            ^TRT436 <> NULL AND (^TRT81 = NULL OR ^TRT81 <= 0)
    
            Data Elements:
    
            ^TRT81 = TRT:RP:BusinessDetails:Income:PartnershipTrustNonPrimaryProduction:Income.TrustShareNetExcludeNetCapitalGainsAndForeignIncomeAndDistributionFranked.Amount
    
            ^TRT436 = TRT:RP:BusinessDetails:Income:PartnershipTrustNonPrimaryProduction:Party.TrustType.Code
            */
            assertion = (report.TRT436 != null && (report.TRT81 == null || report.TRT81.GetValueOrDefault() <= 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432662", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-PP - Share of net income from trusts amount must be present",
                    LongDescription = @"If Non-primary production - Share of net income from trusts action code is present, then Non-primary production- Share of net income from trusts less capital gains, foreign income and franked distributions must be greater than zero.  A trusts action code is not required for negative or zero amounts",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:PartnershipTrustNonPrimaryProduction/tns:TrustShareNetExcludeNetCapitalGainsAndForeignIncomeAndDistributionFrankedA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432105"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT436", Value = GetValueOrEmpty(report.TRT436) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT81", Value = GetValueOrEmpty(report.TRT81) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432107
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432107()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432107
            If there is an amount present at Non-primary production - Deductions relating to amounts shown at distribution from partnerships less foreign income and share of net income from trusts less capital gains, foreign income and franked distributions, there must be an amount at either Non-primary production - Distribution from partnerships less foreign income, or Non-primary production - Share of net income from trusts less capital gains, foreign income and franked distributions
    
            Legacy Rule Format:
            ^TRT83 > 0 AND ^TRT80 = NULL AND ^TRT81 = NULL

            Technical Business Rule Format:
            ^TRT83 > 0 AND ^TRT80 = NULL AND ^TRT81 = NULL
    
            Data Elements:
    
            ^TRT83 = TRT:RP:BusinessDetails:Income:PartnershipTrustNonPrimaryProduction:Expense.DistributionDeductionsOther.Amount
    
            ^TRT80 = TRT:RP:BusinessDetails:Income:PartnershipTrustNonPrimaryProduction:Income.PartnershipDistributionLessForeignIncome.Amount
    
            ^TRT81 = TRT:RP:BusinessDetails:Income:PartnershipTrustNonPrimaryProduction:Income.TrustShareNetExcludeNetCapitalGainsAndForeignIncomeAndDistributionFranked.Amount
            */
            assertion = (report.TRT83.GetValueOrDefault() > 0 && report.TRT80 == null && report.TRT81 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432664", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-PP - Deductions relating to distributions/share of net income can not be more than zero",
                    LongDescription = @"If there is an amount present at Non-primary production - Deductions relating to amounts shown at distribution from partnerships less foreign income and share of net income from trusts less capital gains, foreign income and franked distributions, there must be an amount at either Non-primary production - Distribution from partnerships less foreign income, or Non-primary production - Share of net income from trusts less capital gains, foreign income and franked distributions",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:PartnershipTrustNonPrimaryProduction/tns:ExpenseDistributionDeductionsOtherA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432107"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT83", Value = GetValueOrEmpty(report.TRT83) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT80", Value = GetValueOrEmpty(report.TRT80) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT81", Value = GetValueOrEmpty(report.TRT81) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432118
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432118()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432118
            Interest deductions, capital works deductions or other rental deductions are greater than zero AND gross rent is absent.
    
            Legacy Rule Format:
            (^TRT90 > 0 OR ^TRT91 > 0 OR ^TRT92 > 0) AND ^TRT89 = NULL

            Technical Business Rule Format:
            (^TRT90 > 0 OR ^TRT91 > 0 OR ^TRT92 > 0) AND ^TRT89 = NULL
    
            Data Elements:
    
            ^TRT89 = TRT:RP:BusinessDetails:Income:Rent:Income.Operating.RentalIncomeGross.Amount
    
            ^TRT90 = TRT:RP:BusinessDetails:Income:Rent:Expense.Interest.Amount
    
            ^TRT91 = TRT:RP:BusinessDetails:Income:Rent:Expense.CapitalWorksDeduction.Amount
    
            ^TRT92 = TRT:RP:BusinessDetails:Income:Rent:IncomeTax.Deduction.RentalIncomeDeductionsOtherThanInterestAndCapitalWorks.Amount
            */
            assertion = ((report.TRT90.GetValueOrDefault() > 0 || report.TRT91.GetValueOrDefault() > 0 || report.TRT92.GetValueOrDefault() > 0) && report.TRT89 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430079", Severity = ProcessMessageSeverity.Error,
                    Description = @"Gross rent must be present",
                    LongDescription = @"If Interest deductions, Capital works deductions, or Other rental deductions are greater than zero, Gross rent must be present",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:Rent/tns:OperatingIncomeGrossA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432118"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT90", Value = GetValueOrEmpty(report.TRT90) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT91", Value = GetValueOrEmpty(report.TRT91) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT92", Value = GetValueOrEmpty(report.TRT92) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT89", Value = GetValueOrEmpty(report.TRT89) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432119
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432119()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432119
            Gross rent is greater than zero AND no rental expense items are present. (Rental expense items are interest deductions, capital works deductions AND other rental deductions). If there are no rental expenses then zero is acceptable in one of these fields.
    
            Legacy Rule Format:
            ^TRT89 > 0 AND ^TRT90 = NULL AND ^TRT91 = NULL AND ^TRT92 = NULL

            Technical Business Rule Format:
            ^TRT89 > 0 AND ^TRT90 = NULL AND ^TRT91 = NULL AND ^TRT92 = NULL
    
            Data Elements:
    
            ^TRT89 = TRT:RP:BusinessDetails:Income:Rent:Income.Operating.RentalIncomeGross.Amount
    
            ^TRT90 = TRT:RP:BusinessDetails:Income:Rent:Expense.Interest.Amount
    
            ^TRT91 = TRT:RP:BusinessDetails:Income:Rent:Expense.CapitalWorksDeduction.Amount
    
            ^TRT92 = TRT:RP:BusinessDetails:Income:Rent:IncomeTax.Deduction.RentalIncomeDeductionsOtherThanInterestAndCapitalWorks.Amount
            */
            assertion = (report.TRT89.GetValueOrDefault() > 0 && report.TRT90 == null && report.TRT91 == null && report.TRT92 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430080", Severity = ProcessMessageSeverity.Error,
                    Description = @"Rent deductions must be present",
                    LongDescription = @"If Gross rent is greater than zero, then there must be an amount at Interest deductions, Capital works deductions or Other rental deductions. If there are no rental deductions, then zero must be present in at least one of these fields",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:Rent/tns:OperatingIncomeGrossA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432119"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT89", Value = GetValueOrEmpty(report.TRT89) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT90", Value = GetValueOrEmpty(report.TRT90) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT91", Value = GetValueOrEmpty(report.TRT91) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT92", Value = GetValueOrEmpty(report.TRT92) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432129
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432129()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432129
            Where there are Deductions relating to Australian investment income greater than zero there must be income amounts at either Gross interest, Unfranked dividend amount or franked dividends.
    
            Legacy Rule Format:
            ^TRT123 > 0 AND ^TRT94 = NULL AND ^TRT96 = NULL AND ^TRT97 = NULL

            Technical Business Rule Format:
            ^TRT123 > 0 AND ^TRT94 = NULL AND ^TRT96 = NULL AND ^TRT97 = NULL
    
            Data Elements:
    
            ^TRT94 = TRT:RP:BusinessDetails:Income:GrossInterest:Income.Interest.Gross.Amount
    
            ^TRT96 = TRT:RP:BusinessDetails:Income:Dividends:Income.DividendsUnfranked.Amount
    
            ^TRT97 = TRT:RP:BusinessDetails:Income:Dividends:Income.DividendsFranked.Amount
    
            ^TRT123 = TRT:RP:BusinessDetails:Income:Deductions:Expense.AustralianInvestmentAllowableDeduction.Amount
            */
            assertion = (report.TRT123.GetValueOrDefault() > 0 && report.TRT94 == null && report.TRT96 == null && report.TRT97 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430106", Severity = ProcessMessageSeverity.Error,
                    Description = @"Gross interest or dividend income must be present",
                    LongDescription = @"If Deductions relating to Australian investment income is greater than zero, then Gross interest, Unfranked amount, or Franked amount must be present",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:GrossInterest/tns:A",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432129"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT123", Value = GetValueOrEmpty(report.TRT123) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT94", Value = GetValueOrEmpty(report.TRT94) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT96", Value = GetValueOrEmpty(report.TRT96) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT97", Value = GetValueOrEmpty(report.TRT97) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432131
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432131()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432131
            TFN amounts deducted on interest are greater than or equal to 50% of gross interest.
    
            Legacy Rule Format:
            ^TRT95 > 0 AND (^TRT94 * 0.5) <= ^TRT95

            Technical Business Rule Format:
            ^TRT95 > 0 AND (^TRT94 * 0.5) <= ^TRT95
    
            Data Elements:
    
            ^TRT95 = TRT:RP:BusinessDetails:Income:GrossInterest:IncomeTax.TFNAmountsWithheldFromGrossInterest.Amount
    
            ^TRT94 = TRT:RP:BusinessDetails:Income:GrossInterest:Income.Interest.Gross.Amount
            */
            assertion = (report.TRT95.GetValueOrDefault() > 0 && report.TRT94.GetValueOrDefault() * 0.5M <= report.TRT95.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430093", Severity = ProcessMessageSeverity.Error,
                    Description = @"TFN amounts withheld from dividends must not exceed 50% of Gross interest",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:GrossInterest/tns:TaxTFNAmountsWithheldFromGrossInterestA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432131"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT95", Value = GetValueOrEmpty(report.TRT95) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT94", Value = GetValueOrEmpty(report.TRT94) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432133
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432133()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432133
            If an amount greater than zero is present at TFN amounts withheld from dividends, then there must also be an amount greater than zero at Unfranked amount or Franked amount.
    
            Legacy Rule Format:
            ^TRT99 > 0 AND (^TRT96 = 0 OR ^TRT96 = NULL) AND (^TRT97 = 0 OR ^TRT97 = NULL)

            Technical Business Rule Format:
            ^TRT99 > 0 AND (^TRT96 = 0 OR ^TRT96 = NULL) AND (^TRT97 = 0 OR ^TRT97 = NULL)
    
            Data Elements:
    
            ^TRT96 = TRT:RP:BusinessDetails:Income:Dividends:Income.DividendsUnfranked.Amount
    
            ^TRT97 = TRT:RP:BusinessDetails:Income:Dividends:Income.DividendsFranked.Amount
    
            ^TRT99 = TRT:RP:BusinessDetails:Income:Dividends:IncomeTax.DividendsTFNAmountsWithheld.Amount
            */
            assertion = (report.TRT99.GetValueOrDefault() > 0 && (report.TRT96 == 0 || report.TRT96 == null) && (report.TRT97 == 0 || report.TRT97 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430095", Severity = ProcessMessageSeverity.Error,
                    Description = @"Dividends amounts must be present",
                    LongDescription = @"If an amount greater than zero is present at TFN amounts withheld from dividends, then there must also be an amount greater than zero at Unfranked amount or Franked amount",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:Dividends/tns:UnfrankedA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432133"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT99", Value = GetValueOrEmpty(report.TRT99) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT96", Value = GetValueOrEmpty(report.TRT96) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT97", Value = GetValueOrEmpty(report.TRT97) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432134
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432134()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432134
            The amount at TFN amounts withheld from dividends cannot be greater than 50% of Franked amount plus Unfranked amounts.
    
            Legacy Rule Format:
            ^TRT99 > 0 AND ^TRT96 > 0 AND ^TRT97 > 0 AND (^TRT96 + ((^TRT97*0.5) + 1)) < ^TRT99

            Technical Business Rule Format:
            ^TRT99 > 0 AND ^TRT96 > 0 AND ^TRT97 > 0 AND (^TRT96 + ((^TRT97*0.5) + 1)) < ^TRT99
    
            Data Elements:
    
            ^TRT96 = TRT:RP:BusinessDetails:Income:Dividends:Income.DividendsUnfranked.Amount
    
            ^TRT97 = TRT:RP:BusinessDetails:Income:Dividends:Income.DividendsFranked.Amount
    
            ^TRT99 = TRT:RP:BusinessDetails:Income:Dividends:IncomeTax.DividendsTFNAmountsWithheld.Amount
            */
            assertion = (report.TRT99.GetValueOrDefault() > 0 && report.TRT96.GetValueOrDefault() > 0 && report.TRT97.GetValueOrDefault() > 0 && report.TRT96.GetValueOrDefault() + (report.TRT97.GetValueOrDefault() * 0.5M + 1) < report.TRT99.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430096", Severity = ProcessMessageSeverity.Error,
                    Description = @"TFN amount withheld must not exceed unfranked dividends plus 50% of franked dividends",
                    LongDescription = @"The amount at TFN amounts withheld from dividends must not be greater than Unfranked amount plus 50% of Franked amount",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:Dividends/tns:UnfrankedA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432134"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT99", Value = GetValueOrEmpty(report.TRT99) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT96", Value = GetValueOrEmpty(report.TRT96) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT97", Value = GetValueOrEmpty(report.TRT97) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432137
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432137()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432137
            Franking credit cannot be greater than 48% of dividends-franked amount.
    
            Legacy Rule Format:
            ^TRT98 > 0 AND (^TRT97 * 0.48) < (^TRT98 - 1)

            Technical Business Rule Format:
            ^TRT98 > 0 AND (^TRT97 * 0.48) < (^TRT98 - 1)
    
            Data Elements:
    
            ^TRT98 = TRT:RP:BusinessDetails:Income:Dividends:IncomeTax.FrankingCredits.Amount
    
            ^TRT97 = TRT:RP:BusinessDetails:Income:Dividends:Income.DividendsFranked.Amount
            */
            assertion = (report.TRT98.GetValueOrDefault() > 0 && report.TRT97.GetValueOrDefault() * 0.48M < report.TRT98.GetValueOrDefault() - 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430098", Severity = ProcessMessageSeverity.Error,
                    Description = @"Franking credit cannot be greater than 48% of dividends-franked amount.",
                    LongDescription = @"Franking credit must not exceed 48% of Dividends - Franked amount",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:Dividends/tns:TaxFrankingCreditsA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432137"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT98", Value = GetValueOrEmpty(report.TRT98) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT97", Value = GetValueOrEmpty(report.TRT97) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432138
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432138()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432138
            Dividends-franked amount has a value AND dividends-franking credit has no value.
    
            Legacy Rule Format:
            ^TRT97 > 0 AND ^TRT98 = NULL

            Technical Business Rule Format:
            ^TRT97 > 0 AND ^TRT98 = NULL
    
            Data Elements:
    
            ^TRT98 = TRT:RP:BusinessDetails:Income:Dividends:IncomeTax.FrankingCredits.Amount
    
            ^TRT97 = TRT:RP:BusinessDetails:Income:Dividends:Income.DividendsFranked.Amount
            */
            assertion = (report.TRT97.GetValueOrDefault() > 0 && report.TRT98 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430100", Severity = ProcessMessageSeverity.Error,
                    Description = @"Franking credit amount must be present",
                    LongDescription = @"If Dividends - Franked amount is present, then Franking credit must be present",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:Dividends/tns:TaxFrankingCreditsA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432138"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT97", Value = GetValueOrEmpty(report.TRT97) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT98", Value = GetValueOrEmpty(report.TRT98) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432140
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432140()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432140
            TFN amounts withheld from dividends is greater than 50% of the unfranked amount.
    
            Legacy Rule Format:
            ^TRT99 > 0 AND ((^TRT96*0.5) + 1) < ^TRT99

            Technical Business Rule Format:
            ^TRT99 > 0 AND ((^TRT96*0.5) + 1) < ^TRT99
    
            Data Elements:
    
            ^TRT99 = TRT:RP:BusinessDetails:Income:Dividends:IncomeTax.DividendsTFNAmountsWithheld.Amount
    
            ^TRT96 = TRT:RP:BusinessDetails:Income:Dividends:Income.DividendsUnfranked.Amount
            */
            assertion = (report.TRT99.GetValueOrDefault() > 0 && report.TRT96.GetValueOrDefault() * 0.5M + 1 < report.TRT99.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430102", Severity = ProcessMessageSeverity.Error,
                    Description = @"TFN amounts withheld from dividends must not exceed 50% of the Unfranked amount",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:Dividends/tns:TaxTFNAmountsWithheldA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432140"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT99", Value = GetValueOrEmpty(report.TRT99) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT96", Value = GetValueOrEmpty(report.TRT96) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432148
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432148()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432148
            Tax withheld amounts from Australian government pensions or allowances are greater than or equal to the amount of Australian government pensions. Check both amounts AND correct as necessary.
    
            Legacy Rule Format:
            ^TRT104 > 0 AND ^TRT105 <= ^TRT104

            Technical Business Rule Format:
            ^TRT104 > 0 AND ^TRT105 <= ^TRT104
    
            Data Elements:
    
            ^TRT105 = TRT:RP:BusinessDetails:Income:Pension.Total.Amount
    
            ^TRT104 = TRT:RP:BusinessDetails:Income:IncomeTax.AustralianGovernmentPensionsAndAllowancesTaxWithheld.Amount
            */
            assertion = (report.TRT104.GetValueOrDefault() > 0 && report.TRT105.GetValueOrDefault() <= report.TRT104.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432148", Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax withheld from pension must not be greater than the total pension amount",
                    LongDescription = @"Tax withheld amounts from Australian government pensions or allowances must not be greater than or equal to the amount of Australian government pensions",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:PensionTotalA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432148"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT104", Value = GetValueOrEmpty(report.TRT104) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT105", Value = GetValueOrEmpty(report.TRT105) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432180
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432180()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432180
            The amount of net Australian income/loss is not equal to the sum of all Australian income less all Australian source deductions.
    
            Legacy Rule Format:
            OutsideRange(^TRT130, ^TRT71 + ^TRT72 + ^TRT76 + ^TRT77 - ^TRT79 + ^TRT80 + ^TRT81 - ^TRT83 + ^TRT437 - ^TRT438 + ^TRT89 - ^TRT90 - ^TRT91 - ^TRT92 + ^TRT93 + ^TRT94 + ^TRT96 + ^TRT97 + ^TRT98 + ^TRT105 + Sum(^TRT109) + Sum(^TRT111) + (Sum(^TRT113) * 0.05) + ^TRT115 + ^TRT116 + ^TRT117 + ^TRT118 + ^TRT119 + ^TRT100 + ^TRT101 + Sum(^TRT102) +  Sum(^TRT121) - (^TRT123 + ^TRT439 + ^TRT124 + Sum(^TRT129)) , 1)

            Technical Business Rule Format:
            OutsideRange(^TRT130, ^TRT71 + ^TRT72 + ^TRT76 + ^TRT77 - ^TRT79 + ^TRT80 + ^TRT81 - ^TRT83 + ^TRT437 - ^TRT438 + ^TRT89 - ^TRT90 - ^TRT91 - ^TRT92 + ^TRT93 + ^TRT94 + ^TRT96 + ^TRT97 + ^TRT98 + ^TRT105 + Sum(^TRT109) + Sum(^TRT111) + (Sum(^TRT113) * 0.05) + ^TRT115 + ^TRT116 + ^TRT117 + ^TRT118 + ^TRT119 + ^TRT100 + ^TRT101 + Sum(^TRT102) +  Sum(^TRT121) - (^TRT123 + ^TRT439 + ^TRT124 + Sum(^TRT129)) , 1)
    
            Data Elements:
    
            ^TRT130 = TRT:RP:BusinessDetails:Income:Income.Net.Amount
    
            ^TRT71 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Income.Net.Amount
    
            ^TRT72 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Income.Net.Amount
    
            ^TRT76 = TRT:RP:BusinessDetails:Income:PartnershipsTrustsPrimaryProduction:Income.PartnershipDistributionGross.Amount
    
            ^TRT77 = TRT:RP:BusinessDetails:Income:PartnershipsTrustsPrimaryProduction:Income.TrustShareNet.Amount
    
            ^TRT79 = TRT:RP:BusinessDetails:Income:PartnershipsTrustsPrimaryProduction:Expense.DistributionDeductionsOther.Amount
    
            ^TRT80 = TRT:RP:BusinessDetails:Income:PartnershipTrustNonPrimaryProduction:Income.PartnershipDistributionLessForeignIncome.Amount
    
            ^TRT81 = TRT:RP:BusinessDetails:Income:PartnershipTrustNonPrimaryProduction:Income.TrustShareNetExcludeNetCapitalGainsAndForeignIncomeAndDistributionFranked.Amount
    
            ^TRT83 = TRT:RP:BusinessDetails:Income:PartnershipTrustNonPrimaryProduction:Expense.DistributionDeductionsOther.Amount
    
            ^TRT89 = TRT:RP:BusinessDetails:Income:Rent:Income.Operating.RentalIncomeGross.Amount
    
            ^TRT90 = TRT:RP:BusinessDetails:Income:Rent:Expense.Interest.Amount
    
            ^TRT91 = TRT:RP:BusinessDetails:Income:Rent:Expense.CapitalWorksDeduction.Amount
    
            ^TRT92 = TRT:RP:BusinessDetails:Income:Rent:IncomeTax.Deduction.RentalIncomeDeductionsOtherThanInterestAndCapitalWorks.Amount
    
            ^TRT93 = TRT:RP:BusinessDetails:Income:Income.ForestryManagedInvestmentScheme.Amount
    
            ^TRT94 = TRT:RP:BusinessDetails:Income:GrossInterest:Income.Interest.Gross.Amount
    
            ^TRT96 = TRT:RP:BusinessDetails:Income:Dividends:Income.DividendsUnfranked.Amount
    
            ^TRT97 = TRT:RP:BusinessDetails:Income:Dividends:Income.DividendsFranked.Amount
    
            ^TRT98 = TRT:RP:BusinessDetails:Income:Dividends:IncomeTax.FrankingCredits.Amount
    
            ^TRT100 = TRT:RP:BusinessDetails:Income:SuperannuationLumpSumWithholdingPaymentDetails:SuperannuationBenefit.DeathBenefitLumpSumPaymentTaxedElement.Amount
    
            ^TRT101 = TRT:RP:BusinessDetails:Income:SuperannuationLumpSumWithholdingPaymentDetails:SuperannuationBenefit.DeathBenefitLumpSumPaymentUntaxedElement.Amount
    
            ^TRT102 = TRT:RP:BusinessDetails:Income:EmploymentTerminationPaymentDetails:SuperannuationBenefit.DeathBenefitEmploymentTerminationPaymentTaxableComponent.Amount
    
            ^TRT105 = TRT:RP:BusinessDetails:Income:Pension.Total.Amount
    
            ^TRT109 = TRT:RP:BusinessDetails:Income:IndividualNonBusinessWithholdingPaymentDetails:Remuneration.IndividualNonBusinessGross.Amount
    
            ^TRT111 = TRT:RP:BusinessDetails:Income:IndividualNonBusinessWithholdingPaymentDetails:Remuneration.UnusedAnnualLongServiceLeavePaymentLumpSumA.Amount
    
            ^TRT113 = TRT:RP:BusinessDetails:Income:IndividualNonBusinessWithholdingPaymentDetails:Remuneration.UnusedAnnualLongServiceLeavePaymentLumpSumB.Amount
    
            ^TRT115 = TRT:RP:BusinessDetails:Income:SuperannuationIncomeStreamWithholdingPaymentDetails:IncomeTax.Superannuation.TaxableComponentTaxedElement.Amount
    
            ^TRT116 = TRT:RP:BusinessDetails:Income:SuperannuationIncomeStreamWithholdingPaymentDetails:IncomeTax.Superannuation.TaxableComponentUntaxedElement.Amount
    
            ^TRT117 = TRT:RP:BusinessDetails:Income:SuperannuationIncomeStreamWithholdingPaymentDetails:LumpSumInArrears:IncomeTax.Superannuation.TaxableComponentTaxedElement.Amount
    
            ^TRT118 = TRT:RP:BusinessDetails:Income:SuperannuationIncomeStreamWithholdingPaymentDetails:LumpSumInArrears:IncomeTax.Superannuation.TaxableComponentUntaxedElement.Amount
    
            ^TRT119 = TRT:RP:BusinessDetails:Income:Remuneration.BonusesAndAllowances.Amount
    
            ^TRT121 = TRT:RP:BusinessDetails:Income:OtherAustralianIncome:Income.Other.Amount
    
            ^TRT123 = TRT:RP:BusinessDetails:Income:Deductions:Expense.AustralianInvestmentAllowableDeduction.Amount
    
            ^TRT124 = TRT:RP:BusinessDetails:Income:Deductions:Expense.ForestryManagedInvestmentSchemeDeduction.Amount
    
            ^TRT129 = TRT:RP:BusinessDetails:Income:Deductions:OtherDeductions:Expense.DeductibleOther.Amount
    
            ^TRT437 = TRT:RP:BusinessDetails:Income:PartnershipTrustNonPrimaryProduction:Income.TrustDistributionFranked.Amount
    
            ^TRT438 = TRT:RP:BusinessDetails:Income:PartnershipTrustNonPrimaryProduction:IncomeTax.Deduction.TrustDistributionFranked.Amount
    
            ^TRT439 = TRT:RP:BusinessDetails:Income:Deductions:IncomeTax.Deduction.DistributionFranked.Amount
            */
            assertion = OutsideRange(report.TRT130.GetValueOrDefault(), report.TRT71.GetValueOrDefault() + report.TRT72.GetValueOrDefault() + report.TRT76.GetValueOrDefault() + report.TRT77.GetValueOrDefault() - report.TRT79.GetValueOrDefault() + report.TRT80.GetValueOrDefault() + report.TRT81.GetValueOrDefault() - report.TRT83.GetValueOrDefault() + report.TRT437.GetValueOrDefault() - report.TRT438.GetValueOrDefault() + report.TRT89.GetValueOrDefault() - report.TRT90.GetValueOrDefault() - report.TRT91.GetValueOrDefault() - report.TRT92.GetValueOrDefault() + report.TRT93.GetValueOrDefault() + report.TRT94.GetValueOrDefault() + report.TRT96.GetValueOrDefault() + report.TRT97.GetValueOrDefault() + report.TRT98.GetValueOrDefault() + report.TRT105.GetValueOrDefault() + (report.RP_BusinessDetails_Income_IndividualNonBusinessWithholdingPaymentDetailsCollection == null ? 0 : report.RP_BusinessDetails_Income_IndividualNonBusinessWithholdingPaymentDetailsCollection.Sum(f => f.TRT109.GetValueOrDefault())) + (report.RP_BusinessDetails_Income_IndividualNonBusinessWithholdingPaymentDetailsCollection == null ? 0 : report.RP_BusinessDetails_Income_IndividualNonBusinessWithholdingPaymentDetailsCollection.Sum(f => f.TRT111.GetValueOrDefault())) + (report.RP_BusinessDetails_Income_IndividualNonBusinessWithholdingPaymentDetailsCollection == null ? 0 : report.RP_BusinessDetails_Income_IndividualNonBusinessWithholdingPaymentDetailsCollection.Sum(f => f.TRT113.GetValueOrDefault())) * 0.05M + report.TRT115.GetValueOrDefault() + report.TRT116.GetValueOrDefault() + report.TRT117.GetValueOrDefault() + report.TRT118.GetValueOrDefault() + report.TRT119.GetValueOrDefault() + report.TRT100.GetValueOrDefault() + report.TRT101.GetValueOrDefault() + (report.RP_BusinessDetails_Income_EmploymentTerminationPaymentDetailsCollection == null ? 0 : report.RP_BusinessDetails_Income_EmploymentTerminationPaymentDetailsCollection.Sum(f => f.TRT102.GetValueOrDefault())) + (report.RP_BusinessDetails_Income_OtherAustralianIncomeCollection == null ? 0 : report.RP_BusinessDetails_Income_OtherAustralianIncomeCollection.Sum(f => f.TRT121.GetValueOrDefault())) - (report.TRT123.GetValueOrDefault() + report.TRT439.GetValueOrDefault() + report.TRT124.GetValueOrDefault() + (report.RP_BusinessDetails_Income_Deductions_OtherDeductionsCollection == null ? 0 : report.RP_BusinessDetails_Income_Deductions_OtherDeductionsCollection.Sum(f => f.TRT129.GetValueOrDefault()))), 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432613", Severity = ProcessMessageSeverity.Error,
                    Description = @"Net Australian income must equal income less deductions",
                    LongDescription = @"The amount of net Australian income/loss is not equal to the sum of all Australian income less all Australian source deductions.",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:NetA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432180"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT130", Value = GetValueOrEmpty(report.TRT130) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT71", Value = GetValueOrEmpty(report.TRT71) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT72", Value = GetValueOrEmpty(report.TRT72) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT76", Value = GetValueOrEmpty(report.TRT76) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT77", Value = GetValueOrEmpty(report.TRT77) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT79", Value = GetValueOrEmpty(report.TRT79) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT80", Value = GetValueOrEmpty(report.TRT80) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT81", Value = GetValueOrEmpty(report.TRT81) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT83", Value = GetValueOrEmpty(report.TRT83) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT437", Value = GetValueOrEmpty(report.TRT437) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT438", Value = GetValueOrEmpty(report.TRT438) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT89", Value = GetValueOrEmpty(report.TRT89) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT90", Value = GetValueOrEmpty(report.TRT90) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT91", Value = GetValueOrEmpty(report.TRT91) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT92", Value = GetValueOrEmpty(report.TRT92) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT93", Value = GetValueOrEmpty(report.TRT93) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT94", Value = GetValueOrEmpty(report.TRT94) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT96", Value = GetValueOrEmpty(report.TRT96) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT97", Value = GetValueOrEmpty(report.TRT97) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT98", Value = GetValueOrEmpty(report.TRT98) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT105", Value = GetValueOrEmpty(report.TRT105) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT109", Value = "TRT109" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT111", Value = "TRT111" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT113", Value = "TRT113" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT115", Value = GetValueOrEmpty(report.TRT115) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT116", Value = GetValueOrEmpty(report.TRT116) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT117", Value = GetValueOrEmpty(report.TRT117) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT118", Value = GetValueOrEmpty(report.TRT118) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT119", Value = GetValueOrEmpty(report.TRT119) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT100", Value = GetValueOrEmpty(report.TRT100) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT101", Value = GetValueOrEmpty(report.TRT101) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT102", Value = "TRT102" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT121", Value = "TRT121" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT123", Value = GetValueOrEmpty(report.TRT123) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT439", Value = GetValueOrEmpty(report.TRT439) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT124", Value = GetValueOrEmpty(report.TRT124) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT129", Value = "TRT129" });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432192
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432192()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432192
            Where an amount has been entered at Net capital gain, the answer to the question "Did you have a CGT event during the year" must be Yes.
    
            Legacy Rule Format:
            ^TRT133 > 0 AND ^TRT131 = FALSE

            Technical Business Rule Format:
            ^TRT133 > 0 AND ^TRT131 = FALSE
    
            Data Elements:
    
            ^TRT133 = TRT:RP:BusinessDetails:Income:CapitalGains:Income.CapitalGainsNet.Amount
    
            ^TRT131 = TRT:RP:BusinessDetails:Income:CapitalGains:CapitalGainsTax.Event.Indicator
            */
            assertion = (report.TRT133.GetValueOrDefault() > 0 && report.TRT131 == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432192", Severity = ProcessMessageSeverity.Error,
                    Description = @"Answer to the 'Did you have a CGT event..?' question must be 'yes'",
                    LongDescription = @"If an amount is present at Net capital gain, then the answer to the question 'Did you have a CGT event during the year?' must be 'yes' (true)",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:CapitalGains/tns:NetA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432192"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT133", Value = GetValueOrEmpty(report.TRT133) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT131", Value = GetValueOrEmpty(report.TRT131) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432199
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432199()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432199
            For attributed foreign income, a yes response to the question "did the entity have either a direct or indirect interest in a foreign trust, controlled foreign company or transferor trust?" is required when there are amounts present at either listed country or unlisted country.
    
            Legacy Rule Format:
            (^TRT134 > 0 OR ^TRT136 > 0) AND ^TRT138 <> TRUE

            Technical Business Rule Format:
            (^TRT134 > 0 OR ^TRT136 > 0) AND ^TRT138 <> TRUE
    
            Data Elements:
    
            ^TRT138 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:Income.InternationalDealings.AttributedForeignIncomeForeignTrustorControlledForeignCompanyorTransferorTrust.Indicator
    
            ^TRT134 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:ListedCountry:Income.InternationalDealings.Attributed.Amount
    
            ^TRT136 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:UnlistedCountry:Income.InternationalDealings.Attributed.Amount
            */
            assertion = ((report.TRT134.GetValueOrDefault() > 0 || report.TRT136.GetValueOrDefault() > 0) && report.TRT138 != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432712", Severity = ProcessMessageSeverity.Error,
                    Description = @"Answer to Attributed foreign income question must be 'yes'",
                    LongDescription = @"If Attributed foreign income - Listed country or Unlisted country is greater than zero, then a yes (true) response to the question ""Did you have either a direct or indirect interest in a foreign trust, controlled foreign company or transferor trust?"" is required",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:AttributedForeignIncome/tns:InternationalDealingsTrustorControlledCompanyorTransferorTrustI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432199"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT134", Value = GetValueOrEmpty(report.TRT134) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT136", Value = GetValueOrEmpty(report.TRT136) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT138", Value = GetValueOrEmpty(report.TRT138) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432202
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432202()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432202
            Other assessable foreign source income 'Gross' amount must not be less than 'Net' amount
    
            Legacy Rule Format:
            ^TRT140 < ^TRT141

            Technical Business Rule Format:
            ^TRT140 < ^TRT141
    
            Data Elements:
    
            ^TRT140 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:Income.Other.Amount
    
            ^TRT141 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:Income.Net.Amount
            */
            assertion = (report.TRT140.GetValueOrDefault() < report.TRT141.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430120", Severity = ProcessMessageSeverity.Error,
                    Description = @"Other assessable foreign source income 'Gross' amount must not be less than 'Net' amount",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:AttributedForeignIncome/tns:OtherAssessableForeignIncome/tns:A",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432202"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT140", Value = GetValueOrEmpty(report.TRT140) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT141", Value = GetValueOrEmpty(report.TRT141) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432203
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432203()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432203
            The sum of PP share of net income from trusts plus non-PP share of net income from trusts plus non-PP franked distributions from trusts plus net capital gains plus attributed foreign income plus other assessable net foreign source income plus Aust'n franking credits from a NZ coy is less than zero which indicates that an overall trust loss is being distributed between trusts. Adjust distributed amounts until the overall negative amount no longer exists (i.e. make the sum equal to zero).
    
            Legacy Rule Format:
            (^TRT141 = NULL OR ^TRT141 = 0) AND (^TRT77 + ^TRT81 + ^TRT437 + ^TRT133 + ^TRT134 + ^TRT136 + ^TRT141 + ^TRT143) < 0

            Technical Business Rule Format:
            (^TRT141 = NULL OR ^TRT141 = 0) AND (^TRT77 + ^TRT81 + ^TRT437 + ^TRT133 + ^TRT134 + ^TRT136 + ^TRT141 + ^TRT143) < 0
    
            Data Elements:
    
            ^TRT141 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:Income.Net.Amount
    
            ^TRT77 = TRT:RP:BusinessDetails:Income:PartnershipsTrustsPrimaryProduction:Income.TrustShareNet.Amount
    
            ^TRT81 = TRT:RP:BusinessDetails:Income:PartnershipTrustNonPrimaryProduction:Income.TrustShareNetExcludeNetCapitalGainsAndForeignIncomeAndDistributionFranked.Amount
    
            ^TRT133 = TRT:RP:BusinessDetails:Income:CapitalGains:Income.CapitalGainsNet.Amount
    
            ^TRT134 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:ListedCountry:Income.InternationalDealings.Attributed.Amount
    
            ^TRT136 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:UnlistedCountry:Income.InternationalDealings.Attributed.Amount
    
            ^TRT143 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
    
            ^TRT437 = TRT:RP:BusinessDetails:Income:PartnershipTrustNonPrimaryProduction:Income.TrustDistributionFranked.Amount
            */
            assertion = ((report.TRT141 == null || report.TRT141 == 0) && report.TRT77.GetValueOrDefault() + report.TRT81.GetValueOrDefault() + report.TRT437.GetValueOrDefault() + report.TRT133.GetValueOrDefault() + report.TRT134.GetValueOrDefault() + report.TRT136.GetValueOrDefault() + report.TRT141.GetValueOrDefault() + report.TRT143.GetValueOrDefault() < 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432683", Severity = ProcessMessageSeverity.Error,
                    Description = @"An overall trust loss must not be distributed between trusts",
                    LongDescription = @"The sum of Share of net income from trusts, plus Net capital gains, plus Attributed foreign income, plus Other assessable net foreign source income, plus Aust'n franking credits from a NZ coy, must not total less than zero. This indicates that an overall trust loss is being distributed between trusts.",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:AttributedForeignIncome/tns:OtherAssessableForeignIncome/tns:NetA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432203"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT141", Value = GetValueOrEmpty(report.TRT141) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT77", Value = GetValueOrEmpty(report.TRT77) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT81", Value = GetValueOrEmpty(report.TRT81) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT437", Value = GetValueOrEmpty(report.TRT437) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT133", Value = GetValueOrEmpty(report.TRT133) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT134", Value = GetValueOrEmpty(report.TRT134) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT136", Value = GetValueOrEmpty(report.TRT136) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT143", Value = GetValueOrEmpty(report.TRT143) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432207
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432207()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432207
            The amount at Tax losses Deducted cannot exceed the sum of 'Net Aust'n income or loss', plus 'Net capital gain', plus 'Attributed foreign income' amounts, plus 'Net - Other assessable foreign income', plus 'Aust'n franking credits from a NZ coy' .
    
            Legacy Rule Format:
            ^TRT144 > 0 AND ^TRT144 > (^TRT130 + ^TRT133 + ^TRT134 + ^TRT136 + ^TRT141 + ^TRT143)

            Technical Business Rule Format:
            ^TRT144 > 0 AND ^TRT144 > (^TRT130 + ^TRT133 + ^TRT134 + ^TRT136 + ^TRT141 + ^TRT143)
    
            Data Elements:
    
            ^TRT144 = TRT:RP:BusinessDetails:Income:TaxLossesDeducted:IncomeTax.Deduction.TaxLossesDeducted.Amount
    
            ^TRT130 = TRT:RP:BusinessDetails:Income:Income.Net.Amount
    
            ^TRT133 = TRT:RP:BusinessDetails:Income:CapitalGains:Income.CapitalGainsNet.Amount
    
            ^TRT134 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:ListedCountry:Income.InternationalDealings.Attributed.Amount
    
            ^TRT136 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:UnlistedCountry:Income.InternationalDealings.Attributed.Amount
    
            ^TRT141 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:Income.Net.Amount
    
            ^TRT143 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
            */
            assertion = (report.TRT144.GetValueOrDefault() > 0 && report.TRT144.GetValueOrDefault() > report.TRT130.GetValueOrDefault() + report.TRT133.GetValueOrDefault() + report.TRT134.GetValueOrDefault() + report.TRT136.GetValueOrDefault() + report.TRT141.GetValueOrDefault() + report.TRT143.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432207", Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax losses deducted must not exceed total income.",
                    LongDescription = @"The amount at 'Tax losses deducted' must not be greater than the sum of 'Net Aust'n income or loss', plus 'Net capital gain', plus 'Attributed foreign income' amounts, plus 'Net - Other assessable foreign income', plus 'Aust'n franking credits from a NZ coy'",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:TaxLossesDeductedTaxDeductionA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432207"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT144", Value = GetValueOrEmpty(report.TRT144) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT130", Value = GetValueOrEmpty(report.TRT130) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT133", Value = GetValueOrEmpty(report.TRT133) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT134", Value = GetValueOrEmpty(report.TRT134) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT136", Value = GetValueOrEmpty(report.TRT136) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT141", Value = GetValueOrEmpty(report.TRT141) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT143", Value = GetValueOrEmpty(report.TRT143) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432227
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432227()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432227
            All current assets is greater than total assets.
    
            Legacy Rule Format:
            ^TRT156 <> NULL AND ^TRT157 < ^TRT156

            Technical Business Rule Format:
            ^TRT156 <> NULL AND ^TRT157 < ^TRT156
    
            Data Elements:
    
            ^TRT156 = TRT:RP:KeyFinancialInformation:Assets.CurrentTotal.Amount
    
            ^TRT157 = TRT:RP:KeyFinancialInformation:Assets.Total.Amount
            */
            assertion = (report.TRT156 != null && report.TRT157.GetValueOrDefault() < report.TRT156.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430194", Severity = ProcessMessageSeverity.Error,
                    Description = @"All current assets must not exceed Total assets",
                    Location = "/tns:TRT/tns:RP/tns:KeyFinancialInformation/tns:AssetsCurrentTotalA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432227"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT156", Value = GetValueOrEmpty(report.TRT156) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT157", Value = GetValueOrEmpty(report.TRT157) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432230
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432230()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432230
            All current liabilities is greater than total liabilities.
    
            Legacy Rule Format:
            ^TRT158 <> NULL AND ^TRT159 < ^TRT158

            Technical Business Rule Format:
            ^TRT158 <> NULL AND ^TRT159 < ^TRT158
    
            Data Elements:
    
            ^TRT158 = TRT:RP:KeyFinancialInformation:Liabilities.CurrentTotal.Amount
    
            ^TRT159 = TRT:RP:KeyFinancialInformation:Liabilities.Total.Amount
            */
            assertion = (report.TRT158 != null && report.TRT159.GetValueOrDefault() < report.TRT158.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430195", Severity = ProcessMessageSeverity.Error,
                    Description = @"All current liabilities must not exceed Total liabilities",
                    Location = "/tns:TRT/tns:RP/tns:KeyFinancialInformation/tns:LiabilitiesCurrentTotalA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432230"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT158", Value = GetValueOrEmpty(report.TRT158) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT159", Value = GetValueOrEmpty(report.TRT159) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432234
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432234()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432234
            Total business income (primary production or non-primary production) is present at business income AND expenses AND either the business name or business street address is missing.
    
            Legacy Rule Format:
            (^TRT46 <> NULL OR ^TRT47 <> NULL OR ^TRT49 <> NULL OR ^TRT50 <> NULL OR ^TRT52 <> NULL OR ^TRT54 <> NULL OR ^TRT55 <> NULL) AND (^TRT161 = NULL OR ^TRT162 = NULL)

            Technical Business Rule Format:
            (^TRT46 <> NULL OR ^TRT47 <> NULL OR ^TRT49 <> NULL OR ^TRT50 <> NULL OR ^TRT52 <> NULL OR ^TRT54 <> NULL OR ^TRT55 <> NULL) AND (^TRT161 = NULL OR ^TRT162 = NULL)
    
            Data Elements:
    
            ^TRT161 = TRT:RP:BusinessAndProfessionalInformation:OrganisationNameDetails.OrganisationalName.Text
    
            ^TRT46 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^TRT47 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^TRT49 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Remuneration.PaymentToForeignResidentGross.Amount
    
            ^TRT50 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^TRT52 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^TRT54 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Income.Other.Amount
    
            ^TRT55 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Income.Other.Amount
    
            ^TRT162 = TRT:RP:BusinessAndProfessionalInformation:BusinessAddressDetails:AddressDetails.Line1.Text
            */
            assertion = ((report.TRT46 != null || report.TRT47 != null || report.TRT49 != null || report.TRT50 != null || report.TRT52 != null || report.TRT54 != null || report.TRT55 != null) && (report.TRT161 == null || report.TRT162 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430129", Severity = ProcessMessageSeverity.Error,
                    Description = @"Business name and business address of main business must be present",
                    LongDescription = @"If Total business income (Primary production or Non-primary production) is present at business income and expenses, then both the Business name and Business address of main business must be present",
                    Location = "/tns:TRT/tns:RP/tns:BusinessAndProfessionalInformation/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432234"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT46", Value = GetValueOrEmpty(report.TRT46) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT47", Value = GetValueOrEmpty(report.TRT47) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT49", Value = GetValueOrEmpty(report.TRT49) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT50", Value = GetValueOrEmpty(report.TRT50) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT52", Value = GetValueOrEmpty(report.TRT52) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT54", Value = GetValueOrEmpty(report.TRT54) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT55", Value = GetValueOrEmpty(report.TRT55) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT161", Value = report.TRT161 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT162", Value = report.TRT162 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432235
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432235()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432235
            Net business income (PP or non-PP) is present in business income AND deductions AND there are no business declaration items present.
    
            Legacy Rule Format:
            (^TRT71 <> NULL OR ^TRT72 <> NULL) AND ^TRT166 = NULL AND ^TRT167 = NULL AND ^TRT168 = NULL AND ^TRT170 = NULL AND ^TRT156 = NULL AND ^TRT157 = NULL AND ^TRT171 = NULL AND ^TRT158 = NULL AND ^TRT159 = NULL AND ^TRT172 = NULL AND ^TRT174 = NULL AND ^TRT181 = NULL

            Technical Business Rule Format:
            (^TRT71 <> NULL OR ^TRT72 <> NULL) AND ^TRT166 = NULL AND ^TRT167 = NULL AND ^TRT168 = NULL AND ^TRT170 = NULL AND ^TRT156 = NULL AND ^TRT157 = NULL AND ^TRT171 = NULL AND ^TRT158 = NULL AND ^TRT159 = NULL AND ^TRT172 = NULL AND ^TRT174 = NULL AND ^TRT181 = NULL
    
            Data Elements:
    
            ^TRT166 = TRT:RP:BusinessAndProfessionalInformation:BusinessAndProfessionaltems:Assets.Inventories.Total.Amount
    
            ^TRT71 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Income.Net.Amount
    
            ^TRT72 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Income.Net.Amount
    
            ^TRT156 = TRT:RP:KeyFinancialInformation:Assets.CurrentTotal.Amount
    
            ^TRT157 = TRT:RP:KeyFinancialInformation:Assets.Total.Amount
    
            ^TRT158 = TRT:RP:KeyFinancialInformation:Liabilities.CurrentTotal.Amount
    
            ^TRT159 = TRT:RP:KeyFinancialInformation:Liabilities.Total.Amount
    
            ^TRT167 = TRT:RP:BusinessAndProfessionalInformation:BusinessAndProfessionaltems:Expense.Purchases.AndOtherCosts.Amount
    
            ^TRT168 = TRT:RP:BusinessAndProfessionalInformation:BusinessAndProfessionaltems:Closing:Assets.Inventories.Total.Amount
    
            ^TRT170 = TRT:RP:BusinessAndProfessionalInformation:BusinessAndProfessionaltems:Closing:Assets.TradeandReceivablesOther.Amount
    
            ^TRT171 = TRT:RP:BusinessAndProfessionalInformation:BusinessAndProfessionaltems:Closing:Liabilities.TradeAndOtherPayablesTotal.Amount
    
            ^TRT172 = TRT:RP:BusinessAndProfessionalInformation:BusinessAndProfessionaltems:Remuneration.WagesAndSalaries.Total.Amount
    
            ^TRT174 = TRT:RP:BusinessAndProfessionalInformation:BusinessAndProfessionaltems:Remuneration.PaymentToRelatedPartiesGross.Amount
    
            ^TRT181 = TRT:RP:BusinessAndProfessionalInformation:BusinessAndProfessionaltems:Remuneration.FringeBenefits.EmployeeContribution.Amount
            */
            assertion = ((report.TRT71 != null || report.TRT72 != null) && report.TRT166 == null && report.TRT167 == null && report.TRT168 == null && report.TRT170 == null && report.TRT156 == null && report.TRT157 == null && report.TRT171 == null && report.TRT158 == null && report.TRT159 == null && report.TRT172 == null && report.TRT174 == null && report.TRT181 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432235", Severity = ProcessMessageSeverity.Error,
                    Description = @"Business and professional items amounts must be present",
                    LongDescription = @"If Net business income is present under Business income and deductions, then at least one Business and professional items amount must be present",
                    Location = "/tns:TRT/tns:RP/tns:BusinessAndProfessionalInformation/tns:BusinessAndProfessionaltems/tns:AssetsInventoriesTotalA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432235"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT71", Value = GetValueOrEmpty(report.TRT71) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT72", Value = GetValueOrEmpty(report.TRT72) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT166", Value = GetValueOrEmpty(report.TRT166) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT167", Value = GetValueOrEmpty(report.TRT167) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT168", Value = GetValueOrEmpty(report.TRT168) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT170", Value = GetValueOrEmpty(report.TRT170) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT156", Value = GetValueOrEmpty(report.TRT156) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT157", Value = GetValueOrEmpty(report.TRT157) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT171", Value = GetValueOrEmpty(report.TRT171) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT158", Value = GetValueOrEmpty(report.TRT158) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT159", Value = GetValueOrEmpty(report.TRT159) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT172", Value = GetValueOrEmpty(report.TRT172) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT174", Value = GetValueOrEmpty(report.TRT174) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT181", Value = GetValueOrEmpty(report.TRT181) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432238
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432238()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432238
            Closing stock amount must be present if Code is present
    
            Legacy Rule Format:
            ^TRT169 <> NULL AND (^TRT168 = NULL OR ^TRT168 = 0)

            Technical Business Rule Format:
            ^TRT169 <> NULL AND (^TRT168 = NULL OR ^TRT168 = 0)
    
            Data Elements:
    
            ^TRT168 = TRT:RP:BusinessAndProfessionalInformation:BusinessAndProfessionaltems:Closing:Assets.Inventories.Total.Amount
    
            ^TRT169 = TRT:RP:BusinessAndProfessionalInformation:BusinessAndProfessionaltems:Closing:Assets.Inventories.ValuationMethod.Code
            */
            assertion = (report.TRT169 != null && (report.TRT168 == null || report.TRT168 == 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430197", Severity = ProcessMessageSeverity.Error,
                    Description = @"Closing stock amount must be present if Code is present",
                    Location = "/tns:TRT/tns:RP/tns:BusinessAndProfessionalInformation/tns:BusinessAndProfessionaltems/tns:Closing/tns:AssetsInventoriesTotalA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432238"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT169", Value = GetValueOrEmpty(report.TRT169) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT168", Value = GetValueOrEmpty(report.TRT168) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432240
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432240()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432240
            Under business income and expenses, the closing stock amount is present; however the closing stock action code is missing. The code must be present to indicate the method used to value closing stock.
    
            Legacy Rule Format:
            ^TRT169 = NULL AND ^TRT168 >0

            Technical Business Rule Format:
            ^TRT169 = NULL AND ^TRT168 >0
    
            Data Elements:
    
            ^TRT169 = TRT:RP:BusinessAndProfessionalInformation:BusinessAndProfessionaltems:Closing:Assets.Inventories.ValuationMethod.Code
    
            ^TRT168 = TRT:RP:BusinessAndProfessionalInformation:BusinessAndProfessionaltems:Closing:Assets.Inventories.Total.Amount
            */
            assertion = (report.TRT169 == null && report.TRT168.GetValueOrDefault() > 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430198", Severity = ProcessMessageSeverity.Error,
                    Description = @"Closing stock action code must be present if Closing stock amount is present",
                    Location = "/tns:TRT/tns:RP/tns:BusinessAndProfessionalInformation/tns:BusinessAndProfessionaltems/tns:Closing/tns:AssetsInventoriesValuationMethodC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432240"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT169", Value = GetValueOrEmpty(report.TRT169) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT168", Value = GetValueOrEmpty(report.TRT168) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432241
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432241()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432241
            Trade debtors is greater than all current assets.
    
            Legacy Rule Format:
            ^TRT170 <> NULL AND (^TRT156 < ^TRT170)

            Technical Business Rule Format:
            ^TRT170 <> NULL AND (^TRT156 < ^TRT170)
    
            Data Elements:
    
            ^TRT170 = TRT:RP:BusinessAndProfessionalInformation:BusinessAndProfessionaltems:Closing:Assets.TradeandReceivablesOther.Amount
    
            ^TRT156 = TRT:RP:KeyFinancialInformation:Assets.CurrentTotal.Amount
            */
            assertion = (report.TRT170 != null && report.TRT156.GetValueOrDefault() < report.TRT170.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430199", Severity = ProcessMessageSeverity.Error,
                    Description = @"Trade debtors amount must not exceed All current assets amount",
                    Location = "/tns:TRT/tns:RP/tns:BusinessAndProfessionalInformation/tns:BusinessAndProfessionaltems/tns:Closing/tns:AssetsTradeandReceivablesOtherA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432241"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT170", Value = GetValueOrEmpty(report.TRT170) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT156", Value = GetValueOrEmpty(report.TRT156) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432243
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432243()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432243
            Trade creditors is greater than all current liabilities.
    
            Legacy Rule Format:
            ^TRT171 <> NULL AND ^TRT158 < ^TRT171

            Technical Business Rule Format:
            ^TRT171 <> NULL AND ^TRT158 < ^TRT171
    
            Data Elements:
    
            ^TRT171 = TRT:RP:BusinessAndProfessionalInformation:BusinessAndProfessionaltems:Closing:Liabilities.TradeAndOtherPayablesTotal.Amount
    
            ^TRT158 = TRT:RP:KeyFinancialInformation:Liabilities.CurrentTotal.Amount
            */
            assertion = (report.TRT171 != null && report.TRT158.GetValueOrDefault() < report.TRT171.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430200", Severity = ProcessMessageSeverity.Error,
                    Description = @"Trade creditors cannot exceed all current liabilities in Financial and other information",
                    Location = "/tns:TRT/tns:RP/tns:BusinessAndProfessionalInformation/tns:BusinessAndProfessionaltems/tns:Closing/tns:LiabilitiesTradeAndOtherPayablesTotalA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432243"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT171", Value = GetValueOrEmpty(report.TRT171) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT158", Value = GetValueOrEmpty(report.TRT158) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432245
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432245()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432245
            In other business and professional items, the type code indicating the expense component where the salary and wage expenses for total salary and wage expenses is present, however, total salary and wage expenses amount is missing.
    
            Legacy Rule Format:
            ^TRT173 <> NULL AND (^TRT172 = 0 OR ^TRT172 = NULL)

            Technical Business Rule Format:
            ^TRT173 <> NULL AND (^TRT172 = 0 OR ^TRT172 = NULL)
    
            Data Elements:
    
            ^TRT172 = TRT:RP:BusinessAndProfessionalInformation:BusinessAndProfessionaltems:Remuneration.WagesAndSalaries.Total.Amount
    
            ^TRT173 = TRT:RP:BusinessAndProfessionalInformation:BusinessAndProfessionaltems:Remuneration.WagesAndSalariesAction.Code
            */
            assertion = (report.TRT173 != null && (report.TRT172 == 0 || report.TRT172 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430131", Severity = ProcessMessageSeverity.Error,
                    Description = @"Total salary and wage expenses amount must be present if the action code is present",
                    LongDescription = @"If a Total salary and wage expenses action code is present an amount must be present. Either enter an amount or delete the action code",
                    Location = "/tns:TRT/tns:RP/tns:BusinessAndProfessionalInformation/tns:BusinessAndProfessionaltems/tns:RemunerationWagesAndSalariesTotalA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432245"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT173", Value = GetValueOrEmpty(report.TRT173) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT172", Value = GetValueOrEmpty(report.TRT172) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432247
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432247()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432247
            In other business and professional items, total salary and wage expenses amount is present; however, the type code indicating the expense component where the salary and wage expenses have been reported is missing.
    
            Legacy Rule Format:
            ^TRT172 > 0 AND ^TRT173 = NULL

            Technical Business Rule Format:
            ^TRT172 > 0 AND ^TRT173 = NULL
    
            Data Elements:
    
            ^TRT173 = TRT:RP:BusinessAndProfessionalInformation:BusinessAndProfessionaltems:Remuneration.WagesAndSalariesAction.Code
    
            ^TRT172 = TRT:RP:BusinessAndProfessionalInformation:BusinessAndProfessionaltems:Remuneration.WagesAndSalaries.Total.Amount
            */
            assertion = (report.TRT172.GetValueOrDefault() > 0 && report.TRT173 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430132", Severity = ProcessMessageSeverity.Error,
                    Description = @"Total salary and wage expenses action code must be present",
                    LongDescription = @"If a Total salary and wage expenses amount is present, then an action code must be present",
                    Location = "/tns:TRT/tns:RP/tns:BusinessAndProfessionalInformation/tns:BusinessAndProfessionaltems/tns:RemunerationWagesAndSalariesActionC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432247"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT172", Value = GetValueOrEmpty(report.TRT172) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT173", Value = GetValueOrEmpty(report.TRT173) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432259
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432259()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432259
            Overseas royalty expenses under business and professional items, is greater than total royalty expenses in the business income and expenses section.
    
            Legacy Rule Format:
            ^TRT183 > 0 AND ^TRT183 > ^TRT471

            Technical Business Rule Format:
            ^TRT183 > 0 AND ^TRT183 > ^TRT471
    
            Data Elements:
    
            ^TRT183 = TRT:RP:BusinessDetails:Income:ThinCapitalisation:Expense.Royalties.Amount
    
            ^TRT471 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:BusinessExpenses:Expense.Royalties.Amount
            */
            assertion = (report.TRT183.GetValueOrDefault() > 0 && report.TRT183.GetValueOrDefault() > report.TRT471.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430040", Severity = ProcessMessageSeverity.Error,
                    Description = @"Overseas royalty expenses must not exceed Total royalty expenses",
                    LongDescription = @"Overseas royalty expenses - under Business and professional items must not be greater than Total royalty expenses - under Business income and expenses",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:ThinCapitalisation/tns:ExpenseRoyaltiesA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432259"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT183", Value = GetValueOrEmpty(report.TRT183) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT471", Value = GetValueOrEmpty(report.TRT471) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432263
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432263()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432263
            Unpaid present entitlement to a private company does not contain an amount and code is D or X.
    
            Legacy Rule Format:
            InSet(^TRT187, '"D", "X"') AND (^TRT186 = 0 OR ^TRT186 = NULL)

            Technical Business Rule Format:
            InSet(^TRT187, '"D", "X"') AND (^TRT186 = 0 OR ^TRT186 = NULL)
    
            Data Elements:
    
            ^TRT186 = TRT:RP:BusinessAndProfessionalInformation:BusinessAndProfessionaltems:Closing:IncomeTax.PrivateCompanyUnpaidPresentEntitlement.Amount
    
            ^TRT187 = TRT:RP:BusinessAndProfessionalInformation:BusinessAndProfessionaltems:IncomeTax.PrivateCompanyUnpaidPresentEntitlement.Code
            */
            assertion = (IsMatch(report.TRT187, @"^(D|X)$") && (report.TRT186 == 0 || report.TRT186 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432263", Severity = ProcessMessageSeverity.Error,
                    Description = @"No amount in Unpaid present entitlement to a private company.",
                    LongDescription = @"Unpaid present entitlement to a private company does not contain an amount and code is D or X.",
                    Location = "/tns:TRT/tns:RP/tns:BusinessAndProfessionalInformation/tns:BusinessAndProfessionaltems/tns:Closing/tns:IncomeTaxPrivateCompanyUnpaidPresentEntitlementA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432263"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT187", Value = GetValueOrEmpty(report.TRT187) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT186", Value = GetValueOrEmpty(report.TRT186) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432277
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432277()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432277
            The number of days full levy exemption and/or the number of days half levy exemption exceed 365 (366 in leap years).
    
            Legacy Rule Format:
            (^TRT196 + ^TRT198) > (DaysInDuration(AsDate(Concat(^TRT1, '-06-30')) - AsDate(Concat(^TRT1-1, '-07-01'))) + 1)

            Technical Business Rule Format:
            (^TRT196 + ^TRT198) > (DaysInDuration(AsDate(Concat(^TRT1, '-06-30')) - AsDate(Concat(^TRT1-1, '-07-01'))) + 1)
    
            Data Elements:
    
            ^TRT196 = TRT:RP:MedicareLevy:IncomeTax.MedicareLevy.ExemptionFullDays.Count
    
            ^TRT1 = TRT:RP:Report.TargetFinancial.Year
    
            ^TRT198 = TRT:RP:MedicareLevy:IncomeTax.MedicareLevy.ExemptionHalfDays.Count
            */
            assertion = (report.TRT196.GetValueOrDefault() + report.TRT198.GetValueOrDefault() > (AsDate(string.Concat(report.TRT1.GetValueOrDefault(), @"-06-30")) - AsDate(string.Concat(report.TRT1.GetValueOrDefault() - 1, @"-07-01"))).Days + 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432614", Severity = ProcessMessageSeverity.Error,
                    Description = @"Levy exemption days exceeds limit",
                    LongDescription = @"The number of days full levy exemption and/or the number of days half levy exemption exceed 365 (366 in leap years).",
                    Location = "/tns:TRT/tns:RP/tns:MedicareLevy/tns:IncomeTaxExemptionFullDaysCt",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432277"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT196", Value = report.TRT196.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT198", Value = report.TRT198.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT1", Value = GetValueOrEmpty(report.TRT1) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432280
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432280(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.432280
                    A trust can only make an overall distribution of net income to each individual or non-individual beneficiary. Please adjust distributed income until the overall negative amount no longer exists.
    
                    Legacy Rule Format:
                    (^TRT200 <> NULL OR ^TRT203 <> NULL) AND (^TRT219 + ^TRT218 + ^TRT442 + ^TRT223 + ^TRT224 + ^TRT225 + ^TRT217 ) < 0
        
                    Technical Business Rule Format:
                    (^TRT200 <> NULL OR ^TRT203 <> NULL) AND (^TRT219 + ^TRT218 + ^TRT442 + ^TRT223 + ^TRT224 + ^TRT225 + ^TRT217 ) < 0
            
                    Data Elements:
            
                    ^TRT200 = TRT:RP:StatementOfDistributionToBeneficiary:IndividualNameDetails:PersonNameDetails.FamilyName.Text
            
                    ^TRT203 = TRT:RP:StatementOfDistributionToBeneficiary:NonIndividualNameDetails:OrganisationNameDetails.OrganisationalName.Text
            
                    ^TRT217 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
            
                    ^TRT218 = TRT:RP:StatementOfDistributionToBeneficiary:Information:ShareOfIncomePrimaryProduction:Income.BeneficiaryShare.Amount
            
                    ^TRT219 = TRT:RP:StatementOfDistributionToBeneficiary:Information:ShareOfIncomeNonPrimaryProduction:Income.BeneficiaryShare.Amount
            
                    ^TRT223 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Income.CapitalGainsNet.Amount
            
                    ^TRT224 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.AttributedForeignIncome.Amount
            
                    ^TRT225 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Income.InternationalDealings.Net.Amount
            
                    ^TRT442 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Income.DistributionFranked.Amount
                    */
                    assertion = ((statementOfDistributionToBeneficiary.TRT200 != null || statementOfDistributionToBeneficiary.TRT203 != null) && statementOfDistributionToBeneficiary.TRT219.GetValueOrDefault() + statementOfDistributionToBeneficiary.TRT218.GetValueOrDefault() + statementOfDistributionToBeneficiary.TRT442.GetValueOrDefault() + statementOfDistributionToBeneficiary.TRT223.GetValueOrDefault() + statementOfDistributionToBeneficiary.TRT224.GetValueOrDefault() + statementOfDistributionToBeneficiary.TRT225.GetValueOrDefault() + statementOfDistributionToBeneficiary.TRT217.GetValueOrDefault() < 0);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.TRT.432280", Severity = ProcessMessageSeverity.Error,
                            Description = @"Trust beneficiary cannot have a zero or a negative income distribution",
                            LongDescription = @"A trust can only make an overall distribution of net income to each individual or non-individual beneficiary.  Please adjust distributed income until the overall negative amount no longer exists.",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsFamilyNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432280"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT200", Value = statementOfDistributionToBeneficiary.TRT200 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT203", Value = statementOfDistributionToBeneficiary.TRT203 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT219", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT219) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT218", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT218) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT442", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT442) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT223", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT223) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT224", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT224) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT225", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT225) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT217", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT217) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.432281
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432281(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.432281
                    Either an individual or non-individual name must be present in each Statement of distribution
    
                    Legacy Rule Format:
                    ^TRT214 <> NULL AND ^TRT200 = NULL AND ^TRT203 = NULL
        
                    Technical Business Rule Format:
                    ^TRT214 <> NULL AND ^TRT200 = NULL AND ^TRT203 = NULL
            
                    Data Elements:
            
                    ^TRT200 = TRT:RP:StatementOfDistributionToBeneficiary:IndividualNameDetails:PersonNameDetails.FamilyName.Text
            
                    ^TRT203 = TRT:RP:StatementOfDistributionToBeneficiary:NonIndividualNameDetails:OrganisationNameDetails.OrganisationalName.Text
            
                    ^TRT214 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.AssessmentCalculation.Code
                    */
                    assertion = (statementOfDistributionToBeneficiary.TRT214 != null && statementOfDistributionToBeneficiary.TRT200 == null && statementOfDistributionToBeneficiary.TRT203 == null);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.TRT.432281", Severity = ProcessMessageSeverity.Error,
                            Description = @"Either an individual or non-individual name must be present in each Statement of distribution",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsFamilyNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432281"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT214", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT214) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT200", Value = statementOfDistributionToBeneficiary.TRT200 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT203", Value = statementOfDistributionToBeneficiary.TRT203 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.432282
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432282(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.432282
                    Family name and non-individual name line 1 are both present in the distribution statement. Please delete the incorrect name as only one individual or non-individual name may be present.
    
                    Legacy Rule Format:
                    ^TRT200 <> NULL AND ^TRT203 <> NULL
        
                    Technical Business Rule Format:
                    ^TRT200 <> NULL AND ^TRT203 <> NULL
            
                    Data Elements:
            
                    ^TRT203 = TRT:RP:StatementOfDistributionToBeneficiary:NonIndividualNameDetails:OrganisationNameDetails.OrganisationalName.Text
            
                    ^TRT200 = TRT:RP:StatementOfDistributionToBeneficiary:IndividualNameDetails:PersonNameDetails.FamilyName.Text
                    */
                    assertion = (statementOfDistributionToBeneficiary.TRT200 != null && statementOfDistributionToBeneficiary.TRT203 != null);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.TRT.432615", Severity = ProcessMessageSeverity.Error,
                            Description = @"Both family name and company name present",
                            LongDescription = @"Family name and non-individual name line 1 are both present in the distribution statement. Please delete the incorrect name as only one individual or non-individual name may be present.",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:NonIndividualNameDetailsOrganisationNameDetailsOrganisationalNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432282"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT200", Value = statementOfDistributionToBeneficiary.TRT200 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT203", Value = statementOfDistributionToBeneficiary.TRT203 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.432283
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432283(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.432283
                    Foreign income tax offsets are greater than zero in distribution statement and there is no Attributed foreign income or Other assessable foreign source income.
    
                    Legacy Rule Format:
                    ^TRT226 > 0 AND ^TRT224 = NULL AND ^TRT225 = NULL
        
                    Technical Business Rule Format:
                    ^TRT226 > 0 AND ^TRT224 = NULL AND ^TRT225 = NULL
            
                    Data Elements:
            
                    ^TRT224 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.AttributedForeignIncome.Amount
            
                    ^TRT225 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Income.InternationalDealings.Net.Amount
            
                    ^TRT226 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Income.InternationalDealings.TaxOffset.Amount
                    */
                    assertion = (statementOfDistributionToBeneficiary.TRT226.GetValueOrDefault() > 0 && statementOfDistributionToBeneficiary.TRT224 == null && statementOfDistributionToBeneficiary.TRT225 == null);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.TRT.432283", Severity = ProcessMessageSeverity.Error,
                            Description = @"Attributed foreign income or Other assessable foreign source income must be present",
                            LongDescription = @"If 'Foreign income tax offsets' is greater than zero in a Statement of Distribution',  then there must be 'Attributed foreign income' or 'Other assessable foreign source income' within a beneficiary's Statement of Distribution or against the 'Income to which no beneficiary is presently entitled'",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:Information/tns:IncomeTaxAttributedForeignIncomeA",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432283"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT226", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT226) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT224", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT224) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT225", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT225) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.432284
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432284(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.432284
                    TFN or address must be present in each Statement of Distribution
    
                    Legacy Rule Format:
                    ^TRT211 = NULL AND Count(^TRT1073) = 0
        
                    Technical Business Rule Format:
                    ^TRT211 = NULL AND Count(^TRT1073) = 0
            
                    Data Elements:
            
                    ^TRT211 = TRT:RP:StatementOfDistributionToBeneficiary:Identifiers.TaxFileNumber.Identifier
            
                    ^TRT1073 = TRT:RP:StatementOfDistributionToBeneficiary:BeneficiaryAddressDetails
                    */
                    assertion = (statementOfDistributionToBeneficiary.TRT211 == null && Count(statementOfDistributionToBeneficiary.RP_StatementOfDistributionToBeneficiary_BeneficiaryAddressDetailsCollectionCount) == 0);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.430141", Severity = ProcessMessageSeverity.Error,
                            Description = @"TFN or address must be present in each Statement of Distribution",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:TaxFileNumberId",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432284"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT211", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT211) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.432285
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432285()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432285
            The TFN provided for the Trustee Beneficiary or a Partner in the distribution statement should not be the same as the TFN of the Trust making the statement or the TFN of the Partnership return.
    
            Legacy Rule Format:
            AnyOccurrence(^TRT1070, ^TRT211 = ^TRT5)

            Technical Business Rule Format:
            AnyOccurrence(^TRT1070, ^TRT211 = ^TRT5)
    
            Data Elements:
    
            ^TRT211 = TRT:RP:StatementOfDistributionToBeneficiary:Identifiers.TaxFileNumber.Identifier
    
            ^TRT5 = TRT:RP:Identifiers.TaxFileNumber.Identifier
    
            ^TRT1070 = TRT:RP:StatementOfDistributionToBeneficiary
            */
            assertion = (report.RP_StatementOfDistributionToBeneficiaryCollection == null ? false : report.RP_StatementOfDistributionToBeneficiaryCollection.Any(TRT1070Repeat => TRT1070Repeat.TRT211 == report.TRT5));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430142", Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax file number must not be the TFN of the Trust or Partnership",
                    LongDescription = @"The TFN provided for a Trustee or Partner in a Statement of distribution must not be the same as the TFN of the Trust or Partnership",
                    Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary/tns:TaxFileNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432285"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT211", Value = "TRT211" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT5", Value = "TRT5" });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432286
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432286(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.432286
                    Beneficiary's identifier is present in the distribution statement without a beneficiary’s tax file number.
    
                    Legacy Rule Format:
                    ^TRT212 <> NULL AND ^TRT211 = NULL
        
                    Technical Business Rule Format:
                    ^TRT212 <> NULL AND ^TRT211 = NULL
            
                    Data Elements:
            
                    ^TRT211 = TRT:RP:StatementOfDistributionToBeneficiary:Identifiers.TaxFileNumber.Identifier
            
                    ^TRT212 = TRT:RP:StatementOfDistributionToBeneficiary:Identifiers.TaxFileNumberTrailingNumeric.Identifier
                    */
                    assertion = (statementOfDistributionToBeneficiary.TRT212 != null && statementOfDistributionToBeneficiary.TRT211 == null);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.TRT.432286", Severity = ProcessMessageSeverity.Error,
                            Description = @"Beneficiary's tax file number must be present if the TFN identifier is present",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:TaxFileNumberId",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432286"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT212", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT212) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT211", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT211) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.432287
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432287(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.432287
                    Resident beneficiary details must include TFN and Name. Only one individidual or non-individual name may be present.
    
                    Legacy Rule Format:
                    InSet(^TRT214, '"14", "35"') AND ^TRT344 = TRUE AND ^TRT211 = NULL AND ^TRT200 = NULL AND ^TRT203 = NULL
        
                    Technical Business Rule Format:
                    InSet(^TRT214, '"14", "35"') AND ^TRT344 = TRUE AND ^TRT211 = NULL AND ^TRT200 = NULL AND ^TRT203 = NULL
            
                    Data Elements:
            
                    ^TRT211 = TRT:RP:StatementOfDistributionToBeneficiary:Identifiers.TaxFileNumber.Identifier
            
                    ^TRT200 = TRT:RP:StatementOfDistributionToBeneficiary:IndividualNameDetails:PersonNameDetails.FamilyName.Text
            
                    ^TRT203 = TRT:RP:StatementOfDistributionToBeneficiary:NonIndividualNameDetails:OrganisationNameDetails.OrganisationalName.Text
            
                    ^TRT214 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.AssessmentCalculation.Code
            
                    ^TRT344 = TRT:RP:StatementOfDistributionToBeneficiary:Information:TBStatementInformation:IncomeTax.TrustBeneficiaryStatement.Indicator
                    */
                    assertion = (IsMatch(statementOfDistributionToBeneficiary.TRT214, @"^(14|35)$") && statementOfDistributionToBeneficiary.TRT344 == true && statementOfDistributionToBeneficiary.TRT211 == null && statementOfDistributionToBeneficiary.TRT200 == null && statementOfDistributionToBeneficiary.TRT203 == null);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.TRT.432287", Severity = ProcessMessageSeverity.Error,
                            Description = @"To make a correct TB statement, TFN must be present for resident trustee beneficiary",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:TaxFileNumberId",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432287"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT214", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT214) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT344", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT344) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT211", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT211) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT200", Value = statementOfDistributionToBeneficiary.TRT200 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT203", Value = statementOfDistributionToBeneficiary.TRT203 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.432289
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432289(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.432289
                    Assessment calculation code is 11, 25, 26, 27, 28, 29, 111, 125,126,127,128 129 in the distribution statement and beneficiary's birth date is missing.
    
                    Legacy Rule Format:
                    InSet(^TRT214, '"11", "25", "26", "27", "28", "29", "111", "125", "126", "127", "128", "129"') AND ^TRT213 = NULL
        
                    Technical Business Rule Format:
                    InSet(^TRT214, '"11", "25", "26", "27", "28", "29", "111", "125", "126", "127", "128", "129"') AND ^TRT213 = NULL
            
                    Data Elements:
            
                    ^TRT213 = TRT:RP:StatementOfDistributionToBeneficiary:PersonDemographicDetails.Birth.Date
            
                    ^TRT214 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.AssessmentCalculation.Code
                    */
                    assertion = (IsMatch(statementOfDistributionToBeneficiary.TRT214, @"^(11|25|26|27|28|29|111|125|126|127|128|129)$") && statementOfDistributionToBeneficiary.TRT213 == null);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.TRT.432289", Severity = ProcessMessageSeverity.Error,
                            Description = @"Beneficiary's date of birth must be present",
                            LongDescription = @"If the Assessment calculation code in the distribution statement is 11, 25 to 29, 111, 125 to 129, the beneficiary's Date of birth must be present",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:PersonDemographicDetailsBirthD",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432289"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT214", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT214) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT213", Value = GetValueOrEmptyDate(statementOfDistributionToBeneficiary.TRT213) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.432290
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432290()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432290
            Assessment calculation code is 111, 118, 119, 120, 125,126,127,128,129, 134, 138, 139 or 140 in the distribution statement and Other Attachments Schedule is missing.
    
            Legacy Rule Format:
            AnyOccurrence(^TRT1070, InSet(^TRT214, '"111", "118", "119", "120", "125", "126", "127", "128", "129", "134", "138", "139", "140"')) AND ^TRT405 = NULL

            Technical Business Rule Format:
            AnyOccurrence(^TRT1070, InSet(^TRT214, '"111", "118", "119", "120", "125", "126", "127", "128", "129", "134", "138", "139", "140"')) AND ^TRT405 = NULL
    
            Data Elements:
    
            ^TRT214 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.AssessmentCalculation.Code
    
            ^TRT405 = TRT:RP:OtherReportingInformation:RegulatoryDisclosures.GeneralInformationAboutFinancialStatements.Text
    
            ^TRT1070 = TRT:RP:StatementOfDistributionToBeneficiary
            */
            assertion = ((report.RP_StatementOfDistributionToBeneficiaryCollection == null ? false : report.RP_StatementOfDistributionToBeneficiaryCollection.Any(TRT1070Repeat => IsMatch(TRT1070Repeat.TRT214, @"^(111|118|119|120|125|126|127|128|129|134|138|139|140)$"))) && report.TRT405 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432723", Severity = ProcessMessageSeverity.Error,
                    Description = @"Further information must be provided at Other attachments schedule",
                    LongDescription = @"If the Assessment calculation code in a Statement of Distribution is a non-resident code (111, 118, 119, 120, 125 to 129, 134, 138, 139 or 140) then further details must be supplied using the 'Other Attachments' schedule",
                    Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary/tns:Information/tns:IncomeTaxAssessmentCalculationC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432290"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT214", Value = "Count(TRT214)" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT405", Value = GetValueOrEmpty(report.TRT405) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432292
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432292(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.432292
                    Primary production income, non-primary production income, credit for tax withheld where ABN not quoted, Div 6AA eligible income, franked distributions, franking credit, TFN credit or Australian franking credit from a NZ company is present in the distribution statement and the assessment calculation code is missing.
    
                    Legacy Rule Format:
                    (^TRT218 <> NULL OR ^TRT219 <> NULL OR ^TRT442 <> NULL OR ^TRT220 > 0 OR ^TRT549 > 0 OR ^TRT221 > 0 OR ^TRT222 > 0 OR ^TRT217 > 0) AND ^TRT214 = NULL
        
                    Technical Business Rule Format:
                    (^TRT218 <> NULL OR ^TRT219 <> NULL OR ^TRT442 <> NULL OR ^TRT220 > 0 OR ^TRT549 > 0 OR ^TRT221 > 0 OR ^TRT222 > 0 OR ^TRT217 > 0) AND ^TRT214 = NULL
            
                    Data Elements:
            
                    ^TRT214 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.AssessmentCalculation.Code
            
                    ^TRT217 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
            
                    ^TRT218 = TRT:RP:StatementOfDistributionToBeneficiary:Information:ShareOfIncomePrimaryProduction:Income.BeneficiaryShare.Amount
            
                    ^TRT219 = TRT:RP:StatementOfDistributionToBeneficiary:Information:ShareOfIncomeNonPrimaryProduction:Income.BeneficiaryShare.Amount
            
                    ^TRT220 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount
            
                    ^TRT221 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.FrankingCredits.ReceivedFromAustralianCompanies.Amount
            
                    ^TRT222 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldWhereTFNNotQuoted.Amount
            
                    ^TRT442 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Income.DistributionFranked.Amount
            
                    ^TRT549 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Income.TaxableMinorBeneficiary.Amount
                    */
                    assertion = ((statementOfDistributionToBeneficiary.TRT218 != null || statementOfDistributionToBeneficiary.TRT219 != null || statementOfDistributionToBeneficiary.TRT442 != null || statementOfDistributionToBeneficiary.TRT220.GetValueOrDefault() > 0 || statementOfDistributionToBeneficiary.TRT549.GetValueOrDefault() > 0 || statementOfDistributionToBeneficiary.TRT221.GetValueOrDefault() > 0 || statementOfDistributionToBeneficiary.TRT222.GetValueOrDefault() > 0 || statementOfDistributionToBeneficiary.TRT217.GetValueOrDefault() > 0) && statementOfDistributionToBeneficiary.TRT214 == null);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.TRT.432616", Severity = ProcessMessageSeverity.Error,
                            Description = @"Assessment calculation code must be present",
                            LongDescription = @"If a Statement of Distribution contains an amount at Primary production income, Non-primary production income, Credit for tax withheld where ABN not quoted, Franked distributions, Franking credit, TFN amounts withheld, or Australian franking credit from a NZ company, then an Assessment calculation code must be present in the statement.",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:Information/tns:IncomeTaxAssessmentCalculationC",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432292"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT218", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT218) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT219", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT219) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT442", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT442) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT220", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT220) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT549", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT549) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT221", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT221) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT222", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT222) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT217", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT217) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT214", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT214) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.432293
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432293(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.432293
                    If within a Statement of Distribution the answer to the TB statement question is 'no' (false), then an Assessment calculation code must be present
    
                    Legacy Rule Format:
                    (^TRT200 <> NULL OR ^TRT203 <> NULL) AND ^TRT344 <> TRUE AND ^TRT214 = NULL
        
                    Technical Business Rule Format:
                    (^TRT200 <> NULL OR ^TRT203 <> NULL) AND ^TRT344 <> TRUE AND ^TRT214 = NULL
            
                    Data Elements:
            
                    ^TRT214 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.AssessmentCalculation.Code
            
                    ^TRT200 = TRT:RP:StatementOfDistributionToBeneficiary:IndividualNameDetails:PersonNameDetails.FamilyName.Text
            
                    ^TRT203 = TRT:RP:StatementOfDistributionToBeneficiary:NonIndividualNameDetails:OrganisationNameDetails.OrganisationalName.Text
            
                    ^TRT344 = TRT:RP:StatementOfDistributionToBeneficiary:Information:TBStatementInformation:IncomeTax.TrustBeneficiaryStatement.Indicator
                    */
                    assertion = ((statementOfDistributionToBeneficiary.TRT200 != null || statementOfDistributionToBeneficiary.TRT203 != null) && statementOfDistributionToBeneficiary.TRT344 != true && statementOfDistributionToBeneficiary.TRT214 == null);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.TRT.432293", Severity = ProcessMessageSeverity.Error,
                            Description = @"Assessment calculation code must be present",
                            LongDescription = @"If within a Statement of Distribution the answer to the TB statement question is either 'no' (false) or absent, then an Assessment calculation code must be present",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:Information/tns:IncomeTaxAssessmentCalculationC",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432293"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT200", Value = statementOfDistributionToBeneficiary.TRT200 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT203", Value = statementOfDistributionToBeneficiary.TRT203 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT344", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT344) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT214", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT214) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.432294
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432294(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.432294
                    Address of beneficiary must be present and be an overseas address
    
                    Legacy Rule Format:
                    InSet(^TRT214, '"120", "140"') AND ^TRT344 = TRUE AND ((^TRT200 = NULL AND ^TRT203 = NULL) OR (^TRT205 = NULL OR ^TRT207 = NULL OR ^TRT382 = NULL OR ^TRT320 <> TRUE))
        
                    Technical Business Rule Format:
                    InSet(^TRT214, '"120", "140"') AND ^TRT344 = TRUE AND ((^TRT200 = NULL AND ^TRT203 = NULL) OR (^TRT205 = NULL OR ^TRT207 = NULL OR ^TRT382 = NULL OR ^TRT320 <> TRUE))
            
                    Data Elements:
            
                    ^TRT214 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.AssessmentCalculation.Code
            
                    ^TRT200 = TRT:RP:StatementOfDistributionToBeneficiary:IndividualNameDetails:PersonNameDetails.FamilyName.Text
            
                    ^TRT203 = TRT:RP:StatementOfDistributionToBeneficiary:NonIndividualNameDetails:OrganisationNameDetails.OrganisationalName.Text
            
                    ^TRT205 = TRT:RP:StatementOfDistributionToBeneficiary:BeneficiaryAddressDetails:AddressDetails.Line1.Text
            
                    ^TRT207 = TRT:RP:StatementOfDistributionToBeneficiary:BeneficiaryAddressDetails:AddressDetails.LocalityName.Text
            
                    ^TRT320 = TRT:RP:StatementOfDistributionToBeneficiary:BeneficiaryAddressDetails:AddressDetails.OverseasAddress.Indicator
            
                    ^TRT344 = TRT:RP:StatementOfDistributionToBeneficiary:Information:TBStatementInformation:IncomeTax.TrustBeneficiaryStatement.Indicator
            
                    ^TRT382 = TRT:RP:StatementOfDistributionToBeneficiary:BeneficiaryAddressDetails:AddressDetails.Country.Code
                    */
                    assertion = (IsMatch(statementOfDistributionToBeneficiary.TRT214, @"^(120|140)$") && statementOfDistributionToBeneficiary.TRT344 == true && (statementOfDistributionToBeneficiary.TRT200 == null && statementOfDistributionToBeneficiary.TRT203 == null || (statementOfDistributionToBeneficiary.TRT205 == null || statementOfDistributionToBeneficiary.TRT207 == null || statementOfDistributionToBeneficiary.TRT382 == null || statementOfDistributionToBeneficiary.TRT320 != true)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.TRT.432294", Severity = ProcessMessageSeverity.Error,
                            Description = @"Address of beneficiary must be present and be an overseas address",
                            LongDescription = @"If you are making a TB Statement for a non-resident beneficiary (using Assessment calculation code 120 or 140) the address of the beneficiary must be present in the Statement of Distribution. The overseas address indicator must be 'yes' (true)",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:Information/tns:IncomeTaxAssessmentCalculationC",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432294"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT214", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT214) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT344", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT344) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT200", Value = statementOfDistributionToBeneficiary.TRT200 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT203", Value = statementOfDistributionToBeneficiary.TRT203 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT205", Value = statementOfDistributionToBeneficiary.TRT205 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT207", Value = statementOfDistributionToBeneficiary.TRT207 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT382", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT382) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT320", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT320) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.432295
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432295(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.432295
                    Where a trust is making a TB statement by answering the TB question Yes, the Assessment calculation code should be blank, 35, 140, 14 or 120.
    
                    Legacy Rule Format:
                    ^TRT214 <> NULL AND NotInSet(^TRT214, '"14", "35", "120", "140"') AND ^TRT344 = TRUE
        
                    Technical Business Rule Format:
                    ^TRT214 <> NULL AND NotInSet(^TRT214, '"14", "35", "120", "140"') AND ^TRT344 = TRUE
            
                    Data Elements:
            
                    ^TRT214 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.AssessmentCalculation.Code
            
                    ^TRT344 = TRT:RP:StatementOfDistributionToBeneficiary:Information:TBStatementInformation:IncomeTax.TrustBeneficiaryStatement.Indicator
                    */
                    assertion = (statementOfDistributionToBeneficiary.TRT214 != null && !(IsMatch(statementOfDistributionToBeneficiary.TRT214, @"^(14|35|120|140)$",RegexOptions.IgnoreCase)) && statementOfDistributionToBeneficiary.TRT344 == true);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.TRT.432295", Severity = ProcessMessageSeverity.Error,
                            Description = @"TB statement must not be made if the distribution is not to a trustee beneficiary",
                            LongDescription = @"If within a Statement of Distribution the answer to the 'TB statement' question is 'yes' (true), then the 'Assessment calculation code' must be a trustee beneficiary code (14, 35, 120 or 140) or blank.",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:Information/tns:IncomeTaxAssessmentCalculationC",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432295"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT214", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT214) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT344", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT344) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.432296
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432296(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.432296
                    Where credit for tax withheld - foreign resident withholding (excluding capital gains) is greater than zero, there must be an amount at either share of income - primary or non-primary production or franked distributions. Beneficiary entitled.
    
                    Legacy Rule Format:
                    ^TRT476 > 0 AND ^TRT218 = NULL AND ^TRT219 = NULL AND ^TRT442 = NULL
        
                    Technical Business Rule Format:
                    ^TRT476 > 0 AND ^TRT218 = NULL AND ^TRT219 = NULL AND ^TRT442 = NULL
            
                    Data Elements:
            
                    ^TRT476 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromForeignResidents.Amount
            
                    ^TRT218 = TRT:RP:StatementOfDistributionToBeneficiary:Information:ShareOfIncomePrimaryProduction:Income.BeneficiaryShare.Amount
            
                    ^TRT219 = TRT:RP:StatementOfDistributionToBeneficiary:Information:ShareOfIncomeNonPrimaryProduction:Income.BeneficiaryShare.Amount
            
                    ^TRT442 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Income.DistributionFranked.Amount
                    */
                    assertion = (statementOfDistributionToBeneficiary.TRT476.GetValueOrDefault() > 0 && statementOfDistributionToBeneficiary.TRT218 == null && statementOfDistributionToBeneficiary.TRT219 == null && statementOfDistributionToBeneficiary.TRT442 == null);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.TRT.432738", Severity = ProcessMessageSeverity.Error,
                            Description = @"PP or Non PP Share of income or franked distributions must be present",
                            LongDescription = @"If within a Statement of Distribution there is an amount greater than zero at 'Credit for tax withheld - foreign resident withholding (excluding capital gains)', then there must be an amount at either 'Share of income - primary production'  or 'Share of income - non-primary production' or 'Franked distributions' within the Statement of Distribution.",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:Information/tns:IncomeTaxPayAsYouGoWithholdingCreditForAmountsWithheldFromForeignResidentsA",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432296"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT476", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT476) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT218", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT218) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT219", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT219) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT442", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT442) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.432300
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432300(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.432300
                    If value is included in entity type, TFN or any of the Beneficiary Name and address fields, then an amount must be included at Primary production (PP) or non PP income or Franked distributions or Franking credit or TFN amounts withheld or Australian franking credit from a NZ franking company or the TB Statement-Tax preferred amounts or TB Statement -Untaxed part of share of net income for each beneficiary listed in the distribution statement.
    
                    Legacy Rule Format:
                    (^TRT432 <> NULL OR ^TRT211 <> NULL OR ^TRT212 <> NULL OR ^TRT200 <> NULL OR ^TRT201 <> NULL OR ^TRT202 <> NULL OR ^TRT203 <> NULL OR ^TRT213 <> NULL OR ^TRT205 <> NULL OR ^TRT206 <> NULL OR ^TRT207 <> NULL OR ^TRT208 <> NULL OR ^TRT209 <> NULL OR ^TRT382 <> NULL) AND ^TRT218 = NULL AND ^TRT219 = NULL AND ^TRT442 = NULL AND ^TRT221 = NULL AND ^TRT222 = NULL AND ^TRT217 = NULL AND ^TRT345 = NULL AND ^TRT346 = NULL
        
                    Technical Business Rule Format:
                    (^TRT432 <> NULL OR ^TRT211 <> NULL OR ^TRT212 <> NULL OR ^TRT200 <> NULL OR ^TRT201 <> NULL OR ^TRT202 <> NULL OR ^TRT203 <> NULL OR ^TRT213 <> NULL OR ^TRT205 <> NULL OR ^TRT206 <> NULL OR ^TRT207 <> NULL OR ^TRT208 <> NULL OR ^TRT209 <> NULL OR ^TRT382 <> NULL) AND ^TRT218 = NULL AND ^TRT219 = NULL AND ^TRT442 = NULL AND ^TRT221 = NULL AND ^TRT222 = NULL AND ^TRT217 = NULL AND ^TRT345 = NULL AND ^TRT346 = NULL
            
                    Data Elements:
            
                    ^TRT218 = TRT:RP:StatementOfDistributionToBeneficiary:Information:ShareOfIncomePrimaryProduction:Income.BeneficiaryShare.Amount
            
                    ^TRT200 = TRT:RP:StatementOfDistributionToBeneficiary:IndividualNameDetails:PersonNameDetails.FamilyName.Text
            
                    ^TRT201 = TRT:RP:StatementOfDistributionToBeneficiary:IndividualNameDetails:PersonNameDetails.GivenName.Text
            
                    ^TRT202 = TRT:RP:StatementOfDistributionToBeneficiary:IndividualNameDetails:PersonNameDetails.OtherGivenName.Text
            
                    ^TRT203 = TRT:RP:StatementOfDistributionToBeneficiary:NonIndividualNameDetails:OrganisationNameDetails.OrganisationalName.Text
            
                    ^TRT205 = TRT:RP:StatementOfDistributionToBeneficiary:BeneficiaryAddressDetails:AddressDetails.Line1.Text
            
                    ^TRT206 = TRT:RP:StatementOfDistributionToBeneficiary:BeneficiaryAddressDetails:AddressDetails.Line2.Text
            
                    ^TRT207 = TRT:RP:StatementOfDistributionToBeneficiary:BeneficiaryAddressDetails:AddressDetails.LocalityName.Text
            
                    ^TRT208 = TRT:RP:StatementOfDistributionToBeneficiary:BeneficiaryAddressDetails:AddressDetails.StateOrTerritory.Code
            
                    ^TRT209 = TRT:RP:StatementOfDistributionToBeneficiary:BeneficiaryAddressDetails:AddressDetails.Postcode.Text
            
                    ^TRT211 = TRT:RP:StatementOfDistributionToBeneficiary:Identifiers.TaxFileNumber.Identifier
            
                    ^TRT212 = TRT:RP:StatementOfDistributionToBeneficiary:Identifiers.TaxFileNumberTrailingNumeric.Identifier
            
                    ^TRT213 = TRT:RP:StatementOfDistributionToBeneficiary:PersonDemographicDetails.Birth.Date
            
                    ^TRT217 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
            
                    ^TRT219 = TRT:RP:StatementOfDistributionToBeneficiary:Information:ShareOfIncomeNonPrimaryProduction:Income.BeneficiaryShare.Amount
            
                    ^TRT221 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.FrankingCredits.ReceivedFromAustralianCompanies.Amount
            
                    ^TRT222 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldWhereTFNNotQuoted.Amount
            
                    ^TRT345 = TRT:RP:StatementOfDistributionToBeneficiary:Information:TBStatementInformation:IncomeTax.Superannuation.TrusteeBeneficiaryTaxPreferred.Amount
            
                    ^TRT346 = TRT:RP:StatementOfDistributionToBeneficiary:Information:TBStatementInformation:IncomeTax.Superannuation.TrusteeBeneficiaryUntaxedPartShareOfNetIncome.Amount
            
                    ^TRT382 = TRT:RP:StatementOfDistributionToBeneficiary:BeneficiaryAddressDetails:AddressDetails.Country.Code
            
                    ^TRT432 = TRT:RP:StatementOfDistributionToBeneficiary:IncomeTax.BeneficiaryEntityType.Code
            
                    ^TRT442 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Income.DistributionFranked.Amount
                    */
                    assertion = ((statementOfDistributionToBeneficiary.TRT432 != null || statementOfDistributionToBeneficiary.TRT211 != null || statementOfDistributionToBeneficiary.TRT212 != null || statementOfDistributionToBeneficiary.TRT200 != null || statementOfDistributionToBeneficiary.TRT201 != null || statementOfDistributionToBeneficiary.TRT202 != null || statementOfDistributionToBeneficiary.TRT203 != null || statementOfDistributionToBeneficiary.TRT213 != null || statementOfDistributionToBeneficiary.TRT205 != null || statementOfDistributionToBeneficiary.TRT206 != null || statementOfDistributionToBeneficiary.TRT207 != null || statementOfDistributionToBeneficiary.TRT208 != null || statementOfDistributionToBeneficiary.TRT209 != null || statementOfDistributionToBeneficiary.TRT382 != null) && statementOfDistributionToBeneficiary.TRT218 == null && statementOfDistributionToBeneficiary.TRT219 == null && statementOfDistributionToBeneficiary.TRT442 == null && statementOfDistributionToBeneficiary.TRT221 == null && statementOfDistributionToBeneficiary.TRT222 == null && statementOfDistributionToBeneficiary.TRT217 == null && statementOfDistributionToBeneficiary.TRT345 == null && statementOfDistributionToBeneficiary.TRT346 == null);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.TRT.432637", Severity = ProcessMessageSeverity.Error,
                            Description = @"Each Statement of Distribution must contain distribution amounts",
                            LongDescription = @"A Statement of distribution must include an amount against at least one of: Primary production income, Non-primary production income, Franked distributions, Franking credit, TFN amounts withheld, Australian franking credit from a NZ company, TB Statement-Tax preferred amounts or TB Statement-Untaxed part of share of net income.",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:Information/tns:ShareOfIncomePrimaryProduction/tns:BeneficiaryShareA",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432300"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT432", Value = statementOfDistributionToBeneficiary.TRT432 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT211", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT211) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT212", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT212) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT200", Value = statementOfDistributionToBeneficiary.TRT200 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT201", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT201) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT202", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT202) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT203", Value = statementOfDistributionToBeneficiary.TRT203 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT213", Value = GetValueOrEmptyDate(statementOfDistributionToBeneficiary.TRT213) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT205", Value = statementOfDistributionToBeneficiary.TRT205 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT206", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT206) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT207", Value = statementOfDistributionToBeneficiary.TRT207 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT208", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT208) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT209", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT209) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT382", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT382) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT218", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT218) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT219", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT219) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT442", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT442) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT221", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT221) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT222", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT222) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT217", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT217) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT345", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT345) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT346", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT346) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.432302
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432302(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.432302
                    If an amount greater than zero is present at 'Credit for tax withheld where ABN not quoted' against a Statement of Distribution, then there must be Primary production or Non-primary production income or franked distributions amount against the beneficiary
    
                    Legacy Rule Format:
                    ^TRT220 > 0 AND ^TRT218 = NULL AND ^TRT219 = NULL AND ^TRT442 = NULL
        
                    Technical Business Rule Format:
                    ^TRT220 > 0 AND ^TRT218 = NULL AND ^TRT219 = NULL AND ^TRT442 = NULL
            
                    Data Elements:
            
                    ^TRT220 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount
            
                    ^TRT218 = TRT:RP:StatementOfDistributionToBeneficiary:Information:ShareOfIncomePrimaryProduction:Income.BeneficiaryShare.Amount
            
                    ^TRT219 = TRT:RP:StatementOfDistributionToBeneficiary:Information:ShareOfIncomeNonPrimaryProduction:Income.BeneficiaryShare.Amount
            
                    ^TRT442 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Income.DistributionFranked.Amount
                    */
                    assertion = (statementOfDistributionToBeneficiary.TRT220.GetValueOrDefault() > 0 && statementOfDistributionToBeneficiary.TRT218 == null && statementOfDistributionToBeneficiary.TRT219 == null && statementOfDistributionToBeneficiary.TRT442 == null);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.TRT.432619", Severity = ProcessMessageSeverity.Error,
                            Description = @"PP or Non-PP income distribution or franked distribution amount must be present in Statement of distribution",
                            LongDescription = @"If an amount greater than zero is present at 'Credit for tax withheld where ABN not quoted' against a Statement of Distribution, then there must be Primary production or Non-primary production income or franked distributions amount against the beneficiary.",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:Information/tns:IncomeTaxPayAsYouGoWithholdingCreditForTaxWithheldWhereABNNotQuotedA",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432302"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT220", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT220) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT218", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT218) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT219", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT219) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT442", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT442) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.432303
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432303(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.432303
                    Franking credit or TFN amount withheld is present in the distribution statement and there is no non-primary production income amount or franked distributions amount. Zero is acceptable when there is an overall trust loss.
    
                    Legacy Rule Format:
                    (^TRT221 > 0 OR ^TRT222 > 0) AND ^TRT219 = NULL AND ^TRT442 = NULL
        
                    Technical Business Rule Format:
                    (^TRT221 > 0 OR ^TRT222 > 0) AND ^TRT219 = NULL AND ^TRT442 = NULL
            
                    Data Elements:
            
                    ^TRT219 = TRT:RP:StatementOfDistributionToBeneficiary:Information:ShareOfIncomeNonPrimaryProduction:Income.BeneficiaryShare.Amount
            
                    ^TRT221 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.FrankingCredits.ReceivedFromAustralianCompanies.Amount
            
                    ^TRT222 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldWhereTFNNotQuoted.Amount
            
                    ^TRT442 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Income.DistributionFranked.Amount
                    */
                    assertion = ((statementOfDistributionToBeneficiary.TRT221.GetValueOrDefault() > 0 || statementOfDistributionToBeneficiary.TRT222.GetValueOrDefault() > 0) && statementOfDistributionToBeneficiary.TRT219 == null && statementOfDistributionToBeneficiary.TRT442 == null);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.TRT.432618", Severity = ProcessMessageSeverity.Error,
                            Description = @"Non-PP income or franked distributions amount must be present in Statement of distribution",
                            LongDescription = @"If within a Statement of Distribution ''Franking credit' or 'TFN amount withheld' is present, then there must be an amount at 'Non-primary production income' or 'Franked distributions'. Zero is acceptable if there is an overall trust loss.",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:Information/tns:ShareOfIncomeNonPrimaryProduction/tns:BeneficiaryShareA",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432303"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT221", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT221) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT222", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT222) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT219", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT219) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT442", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT442) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.432307
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432307()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432307
            There is an overall trust loss (the sum of net Australian income/loss, net capital gains, attributed foreign income other assessable foreign source income minus allowable tax losses of earlier income years deducted this year is less than or equal to zero) and franking credits are present in the distribution statement and Australian franking credits from a NZ company (distribution to a beneficiary or no beneficiary presently entitled). Franking credits cannot be distributed when there is an overall trust loss. Delete the franking credits from the distribution statement.
    
            Legacy Rule Format:
            (^TRT130 + ^TRT133 + ^TRT134 + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144) <= 0 AND (AnyOccurrence(^TRT1070, ^TRT217 <> NULL OR ^TRT221 <> NULL) OR ^TRT237 <> NULL OR ^TRT233 <> NULL)

            Technical Business Rule Format:
            (^TRT130 + ^TRT133 + ^TRT134 + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144) <= 0 AND (AnyOccurrence(^TRT1070, ^TRT217 <> NULL OR ^TRT221 <> NULL) OR ^TRT237 <> NULL OR ^TRT233 <> NULL)
    
            Data Elements:
    
            ^TRT221 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.FrankingCredits.ReceivedFromAustralianCompanies.Amount
    
            ^TRT130 = TRT:RP:BusinessDetails:Income:Income.Net.Amount
    
            ^TRT133 = TRT:RP:BusinessDetails:Income:CapitalGains:Income.CapitalGainsNet.Amount
    
            ^TRT134 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:ListedCountry:Income.InternationalDealings.Attributed.Amount
    
            ^TRT136 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:UnlistedCountry:Income.InternationalDealings.Attributed.Amount
    
            ^TRT141 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:Income.Net.Amount
    
            ^TRT143 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
    
            ^TRT144 = TRT:RP:BusinessDetails:Income:TaxLossesDeducted:IncomeTax.Deduction.TaxLossesDeducted.Amount
    
            ^TRT217 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
    
            ^TRT233 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
    
            ^TRT237 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.FrankingCredits.ReceivedFromAustralianCompanies.Amount
    
            ^TRT1070 = TRT:RP:StatementOfDistributionToBeneficiary
            */
            assertion = (report.TRT130.GetValueOrDefault() + report.TRT133.GetValueOrDefault() + report.TRT134.GetValueOrDefault() + report.TRT136.GetValueOrDefault() + report.TRT141.GetValueOrDefault() + report.TRT143.GetValueOrDefault() - report.TRT144.GetValueOrDefault() <= 0 && ((report.RP_StatementOfDistributionToBeneficiaryCollection == null ? false : report.RP_StatementOfDistributionToBeneficiaryCollection.Any(TRT1070Repeat => TRT1070Repeat.TRT217 != null || TRT1070Repeat.TRT221 != null)) || report.TRT237 != null || report.TRT233 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432307", Severity = ProcessMessageSeverity.Error,
                    Description = @"Franking credits must not be present when the trust has zero net income or incurs an overall loss",
                    LongDescription = @"If there is an overall trust loss (the sum of net Australian income/loss, net capital gains, attributed foreign income, other assessable foreign source income; minus allowable tax losses is less than or equal to zero) then franking credits and Australian franking credits from a NZ company must not be present in a Statement of Distribution. Franking credits cannot be distributed when there is an overall trust loss. Delete the franking credits from the Statement of Distribution",
                    Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary/tns:Information/tns:IncomeTaxFrankingCreditsReceivedFromAustralianCompaniesA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432307"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT130", Value = GetValueOrEmpty(report.TRT130) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT133", Value = GetValueOrEmpty(report.TRT133) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT134", Value = GetValueOrEmpty(report.TRT134) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT136", Value = GetValueOrEmpty(report.TRT136) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT141", Value = GetValueOrEmpty(report.TRT141) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT143", Value = GetValueOrEmpty(report.TRT143) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT144", Value = GetValueOrEmpty(report.TRT144) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT217", Value = "TRT217" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT221", Value = "TRT221" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT237", Value = GetValueOrEmpty(report.TRT237) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT233", Value = GetValueOrEmpty(report.TRT233) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432311
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432311()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432311
            Foreign income tax offsets are greater than zero where no beneficiary entitled and there is no Attributed foreign income or Other assessable foreign source income.
    
            Legacy Rule Format:
            ^TRT242 > 0  AND ^TRT240 = NULL AND ^TRT241 = NULL

            Technical Business Rule Format:
            ^TRT242 > 0  AND ^TRT240 = NULL AND ^TRT241 = NULL
    
            Data Elements:
    
            ^TRT242 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:Income.InternationalDealings.TaxOffset.Amount
    
            ^TRT240 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.AttributedForeignIncome.Amount
    
            ^TRT241 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:Income.InternationalDealings.Net.Amount
            */
            assertion = (report.TRT242.GetValueOrDefault() > 0 && report.TRT240 == null && report.TRT241 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432311", Severity = ProcessMessageSeverity.Error,
                    Description = @"Attributed foreign income or Other assessable foreign source income must be present",
                    LongDescription = @"If 'Foreign income tax offsets' is greater than zero in the 'Income to which no beneficiary is presently entitled', then there must be 'Attributed foreign income' or 'Other assessable foreign source income' against the 'Income to which no beneficiary is presently entitled'",
                    Location = "/tns:TRT/tns:RP/tns:IncomeNoBeneficiaryPresentlyEntitled/tns:InternationalDealingsTaxOffsetA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432311"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT242", Value = GetValueOrEmpty(report.TRT242) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT240", Value = GetValueOrEmpty(report.TRT240) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT241", Value = GetValueOrEmpty(report.TRT241) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432314
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432314()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432314
            There is zero net income or an overall trust loss (the sum of net Australian income/loss, net capital gains, attributed foreign income other assessable foreign source income minus allowable tax losses of earlier income years deducted this year is less than or equal to zero) and distributed Foreign income tax offsets are present in the distribution statements (distribution to a beneficiary or no beneficiary presently entitled). Foreign income tax offsets cannot be distributed when there is zero net income or an overall trust loss. Delete the Foreign income tax offsets from the distribution statement.
    
            Legacy Rule Format:
            (^TRT130 + ^TRT133 + ^TRT134 + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144) <= 0 AND (AnyOccurrence(^TRT1070, ^TRT226 <> NULL) OR ^TRT242 <> NULL)

            Technical Business Rule Format:
            (^TRT130 + ^TRT133 + ^TRT134 + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144) <= 0 AND (AnyOccurrence(^TRT1070, ^TRT226 <> NULL) OR ^TRT242 <> NULL)
    
            Data Elements:
    
            ^TRT226 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Income.InternationalDealings.TaxOffset.Amount
    
            ^TRT130 = TRT:RP:BusinessDetails:Income:Income.Net.Amount
    
            ^TRT133 = TRT:RP:BusinessDetails:Income:CapitalGains:Income.CapitalGainsNet.Amount
    
            ^TRT134 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:ListedCountry:Income.InternationalDealings.Attributed.Amount
    
            ^TRT136 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:UnlistedCountry:Income.InternationalDealings.Attributed.Amount
    
            ^TRT141 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:Income.Net.Amount
    
            ^TRT143 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
    
            ^TRT144 = TRT:RP:BusinessDetails:Income:TaxLossesDeducted:IncomeTax.Deduction.TaxLossesDeducted.Amount
    
            ^TRT242 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:Income.InternationalDealings.TaxOffset.Amount
    
            ^TRT1070 = TRT:RP:StatementOfDistributionToBeneficiary
            */
            assertion = (report.TRT130.GetValueOrDefault() + report.TRT133.GetValueOrDefault() + report.TRT134.GetValueOrDefault() + report.TRT136.GetValueOrDefault() + report.TRT141.GetValueOrDefault() + report.TRT143.GetValueOrDefault() - report.TRT144.GetValueOrDefault() <= 0 && ((report.RP_StatementOfDistributionToBeneficiaryCollection == null ? false : report.RP_StatementOfDistributionToBeneficiaryCollection.Any(TRT1070Repeat => TRT1070Repeat.TRT226 != null)) || report.TRT242 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432314", Severity = ProcessMessageSeverity.Error,
                    Description = @"Foreign income tax offsets must not be present if trust has zero net income or incurs a loss",
                    LongDescription = @"If there is zero net income or an overall trust loss (the sum of net Australian income/loss, net capital gains, attributed foreign income, other assessable foreign source income; minus allowable tax losses is less than or equal to zero) then a 'Foreign income tax offset' amount must not be present in a Statement of Distribution or at 'Income to which no beneficiary is presently entitled..'. Foreign income tax offsets cannot be distributed when there is zero net income or an overall trust loss. Delete the Foreign income tax offsets from the distribution statement.",
                    Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary/tns:Information/tns:IncomeInternationalDealingsTaxOffsetA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432314"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT130", Value = GetValueOrEmpty(report.TRT130) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT133", Value = GetValueOrEmpty(report.TRT133) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT134", Value = GetValueOrEmpty(report.TRT134) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT136", Value = GetValueOrEmpty(report.TRT136) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT141", Value = GetValueOrEmpty(report.TRT141) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT143", Value = GetValueOrEmpty(report.TRT143) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT144", Value = GetValueOrEmpty(report.TRT144) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT226", Value = "TRT226" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT242", Value = GetValueOrEmpty(report.TRT242) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432318
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432318(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.432318
                    In the statement of distribution, if the assessment calculation code is 118, 119, 134, 138, 139 or 145, there must be an S98(3) assessable amount.
    
                    Legacy Rule Format:
                    InSet(^TRT214, '"118", "119", "134","138", "139", "145"') AND ^TRT228 = NULL
        
                    Technical Business Rule Format:
                    InSet(^TRT214, '"118", "119", "134","138", "139", "145"') AND ^TRT228 = NULL
            
                    Data Elements:
            
                    ^TRT228 = TRT:RP:StatementOfDistributionToBeneficiary:Information:NonResidentAdditionalInformation:InternationalDealings.AssessableIncomeNonResidentBeneficiaryNonTrustee.Amount
            
                    ^TRT214 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.AssessmentCalculation.Code
                    */
                    assertion = (IsMatch(statementOfDistributionToBeneficiary.TRT214, @"^(118|119|134|138|139|145)$") && statementOfDistributionToBeneficiary.TRT228 == null);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.TRT.442318", Severity = ProcessMessageSeverity.Error,
                            Description = @"s98(3) assessable amount must be present",
                            LongDescription = @"In the statement of distribution, if the assessment calculation code is 118, 119, 134, 138, 139 or 145, there must be an S98(3) assessable amount.",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:Information/tns:NonResidentAdditionalInformation/tns:InternationalDealingsAssessableIncomeNonResidentBeneficiaryNonTrusteeA",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432318"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT214", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT214) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT228", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT228) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.432321
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432321()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432321
            PP income, Non-PP income, credit for tax withheld where no ABN quoted, franked distributions, franking credit, or TFN credit or Aust'n franking credit from a NZ coy is present in the distribution statement (no beneficiary presently entitled) and the assessment calculation code no beneficiary presently entitled is missing.
    
            Legacy Rule Format:
            (^TRT234 <> NULL OR ^TRT235 <> NULL OR ^TRT444 <> NULL OR ^TRT236 > 0 OR ^TRT237 > 0 OR ^TRT238 > 0 OR ^TRT233 > 0) AND ^TRT230 = NULL

            Technical Business Rule Format:
            (^TRT234 <> NULL OR ^TRT235 <> NULL OR ^TRT444 <> NULL OR ^TRT236 > 0 OR ^TRT237 > 0 OR ^TRT238 > 0 OR ^TRT233 > 0) AND ^TRT230 = NULL
    
            Data Elements:
    
            ^TRT230 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.AssessmentCalculation.Code
    
            ^TRT233 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
    
            ^TRT234 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:PrimaryProductionNoBeneficiaryEntitled:IncomeTax.IncomeOrLossTaxable.Amount
    
            ^TRT235 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:NonPrimaryProductionNoBeneficiaryEntitled:IncomeTax.IncomeOrLossTaxable.Amount
    
            ^TRT236 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount
    
            ^TRT237 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.FrankingCredits.ReceivedFromAustralianCompanies.Amount
    
            ^TRT238 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldWhereTFNNotQuoted.Amount
    
            ^TRT444 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:Income.DistributionFranked.Amount
            */
            assertion = ((report.TRT234 != null || report.TRT235 != null || report.TRT444 != null || report.TRT236.GetValueOrDefault() > 0 || report.TRT237.GetValueOrDefault() > 0 || report.TRT238.GetValueOrDefault() > 0 || report.TRT233.GetValueOrDefault() > 0) && report.TRT230 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432621", Severity = ProcessMessageSeverity.Error,
                    Description = @"Assessment calculation code must be present",
                    LongDescription = @"If the 'Income to which no beneficiary is presently entitled' Statement of Distribution contains an amount at Primary production income, Non-primary production income, Credit for tax withheld where ABN not quoted, Franked distributions, Franking credit, TFN amounts withheld, or Australian franking credit from a NZ company, then an Assessment calculation code must be present in the statement.",
                    Location = "/tns:TRT/tns:RP/tns:IncomeNoBeneficiaryPresentlyEntitled/tns:IncomeTaxAssessmentCalculationC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432321"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT234", Value = GetValueOrEmpty(report.TRT234) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT235", Value = GetValueOrEmpty(report.TRT235) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT444", Value = GetValueOrEmpty(report.TRT444) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT236", Value = GetValueOrEmpty(report.TRT236) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT237", Value = GetValueOrEmpty(report.TRT237) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT238", Value = GetValueOrEmpty(report.TRT238) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT233", Value = GetValueOrEmpty(report.TRT233) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT230", Value = GetValueOrEmpty(report.TRT230) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432325
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432325()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432325
            Franking credit or TFN amount withheld is present in the distribution statement (no beneficiary presently entitled) and there is no non-primary production income distribution or franked distributions amount (no beneficiary presently entitled). Zero is acceptable when there is an overall trust loss.
    
            Legacy Rule Format:
            (^TRT237 > 0 OR ^TRT238 > 0) AND ^TRT235 = NULL AND ^TRT444 = NULL

            Technical Business Rule Format:
            (^TRT237 > 0 OR ^TRT238 > 0) AND ^TRT235 = NULL AND ^TRT444 = NULL
    
            Data Elements:
    
            ^TRT235 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:NonPrimaryProductionNoBeneficiaryEntitled:IncomeTax.IncomeOrLossTaxable.Amount
    
            ^TRT237 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.FrankingCredits.ReceivedFromAustralianCompanies.Amount
    
            ^TRT238 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldWhereTFNNotQuoted.Amount
    
            ^TRT444 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:Income.DistributionFranked.Amount
            */
            assertion = ((report.TRT237.GetValueOrDefault() > 0 || report.TRT238.GetValueOrDefault() > 0) && report.TRT235 == null && report.TRT444 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432623", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-PP income or franked distributions amount must be present in 'no beneficiary entitled' distribution",
                    LongDescription = @"If within the 'Income to which no beneficiary is presently entitled' Statement of Distribution, a 'Franking credit' or 'TFN amount withheld' amount is present, then there must be an amount at 'Non-primary production income' or 'franked distributions'. Zero is acceptable if there is an overall trust loss.",
                    Location = "/tns:TRT/tns:RP/tns:IncomeNoBeneficiaryPresentlyEntitled/tns:NonPrimaryProductionNoBeneficiaryEntitled/tns:IncomeTaxIncomeOrLossTaxableA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432325"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT237", Value = GetValueOrEmpty(report.TRT237) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT238", Value = GetValueOrEmpty(report.TRT238) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT235", Value = GetValueOrEmpty(report.TRT235) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT444", Value = GetValueOrEmpty(report.TRT444) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432335
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432335()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432335
            There is a 'yes' response to one or more of the questions on the trust return at Overseas Transactions, Beneficiary under legal disability, or Non-resident trust and the Other Attachments Schedule is missing.
    
            Legacy Rule Format:
            (^TRT244 = TRUE OR ^TRT149 = TRUE OR ^TRT245 = 'NonResidentNoPermEstab') AND ^TRT405 = NULL

            Technical Business Rule Format:
            (^TRT244 = TRUE OR ^TRT149 = TRUE OR ^TRT245 = 'NonResidentNoPermEstab') AND ^TRT405 = NULL
    
            Data Elements:
    
            ^TRT244 = TRT:RP:IncomeTax.BeneficiaryUnderLegalDisabilityAlsoPresentlyEntitledToIncomeFromAnotherTrust.Indicator
    
            ^TRT149 = TRT:RP:BusinessDetails:Income:ThinCapitalisation:Residency.TaxPurposesPersonStatus.Indicator
    
            ^TRT245 = TRT:RP:NonResidentTrust:Residency.TaxPurposesOrganisationStatus.Code
    
            ^TRT405 = TRT:RP:OtherReportingInformation:RegulatoryDisclosures.GeneralInformationAboutFinancialStatements.Text
            */
            assertion = ((report.TRT244 == true || report.TRT149 == true || report.TRT245 == @"NonResidentNoPermEstab") && report.TRT405 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432335", Severity = ProcessMessageSeverity.Error,
                    Description = @"Other attachments schedule must be present",
                    LongDescription = @"If a non-resident beneficiary or a beneficiary under a legal disability is entitled to a share of trust income, or if the trust is a non-resident trust, then further information must be provided against the 'Other Attachments' element",
                    Location = "/tns:TRT/tns:RP/tns:IncomeTaxBeneficiaryUnderLegalDisabilityAlsoPresentlyEntitledToIncomeFromAnotherTrustI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432335"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT244", Value = GetValueOrEmpty(report.TRT244) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT149", Value = GetValueOrEmpty(report.TRT149) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT245", Value = report.TRT245 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT405", Value = GetValueOrEmpty(report.TRT405) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432337
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432337()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432337
            There is a 'yes' response to the question concerning whether the trust estate is a non-resident trust estate and the amount of income derived outside Australia to which no beneficiary is presently entitled is missing.
    
            Legacy Rule Format:
            ^TRT245 = 'NonResidentNoPermEstab' AND ^TRT246 = NULL

            Technical Business Rule Format:
            ^TRT245 = 'NonResidentNoPermEstab' AND ^TRT246 = NULL
    
            Data Elements:
    
            ^TRT246 = TRT:RP:NonResidentTrust:InternationalDealings.NonResidentTrustNonAustralianSourcedIncomeNoPresentEntitlement.Amount
    
            ^TRT245 = TRT:RP:NonResidentTrust:Residency.TaxPurposesOrganisationStatus.Code
            */
            assertion = (report.TRT245 == @"NonResidentNoPermEstab" && report.TRT246 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432705", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-resident trust - amount of income must be present",
                    LongDescription = @"If the answer to the question 'Is the trust a non-resident trust' is 'NonResidentNoPermEstab', indicating non-resident, then an amount must be present at '...the amount of income derived outside Australia to which no beneficiary is presently entitled...'",
                    Location = "/tns:TRT/tns:RP/tns:NonResidentTrust/tns:InternationalDealingsAustralianSourcedIncomeNoPresentEntitlementA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432337"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT245", Value = report.TRT245 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT246", Value = GetValueOrEmpty(report.TRT246) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432339
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432339()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432339
            There is a 'no' response to the question concerning whether the trust estate is a non resident trust estate and there is an amount of income derived outside Australia to which no beneficiary is presently entitled.
    
            Legacy Rule Format:
            ^TRT245 = 'Resident' AND ^TRT246 <> NULL

            Technical Business Rule Format:
            ^TRT245 = 'Resident' AND ^TRT246 <> NULL
    
            Data Elements:
    
            ^TRT246 = TRT:RP:NonResidentTrust:InternationalDealings.NonResidentTrustNonAustralianSourcedIncomeNoPresentEntitlement.Amount
    
            ^TRT245 = TRT:RP:NonResidentTrust:Residency.TaxPurposesOrganisationStatus.Code
            */
            assertion = (report.TRT245 == @"Resident" && report.TRT246 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432703", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-resident trust income amount must not be present",
                    LongDescription = @"If the answer to the question 'Is the trust a non-resident trust' is 'Resident', indicating 'resident', then an amount must NOT be present at '...the amount of income derived outside Australia to which no beneficiary is presently entitled...'",
                    Location = "/tns:TRT/tns:RP/tns:NonResidentTrust/tns:InternationalDealingsAustralianSourcedIncomeNoPresentEntitlementA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432339"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT245", Value = report.TRT245 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT246", Value = GetValueOrEmpty(report.TRT246) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432347
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432347(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.432347
                    Where the trustee has indicated that they are making a TB statement, by answering the TB question Yes, there must be a value greater than 0 at 'Tax preferred amounts' or Untaxed part of share of net income' .
    
                    Legacy Rule Format:
                    ^TRT344 = TRUE AND (^TRT345 = 0 OR ^TRT345 = NULL) AND (^TRT346 = 0 OR ^TRT346 = NULL)
        
                    Technical Business Rule Format:
                    ^TRT344 = TRUE AND (^TRT345 = 0 OR ^TRT345 = NULL) AND (^TRT346 = 0 OR ^TRT346 = NULL)
            
                    Data Elements:
            
                    ^TRT344 = TRT:RP:StatementOfDistributionToBeneficiary:Information:TBStatementInformation:IncomeTax.TrustBeneficiaryStatement.Indicator
            
                    ^TRT345 = TRT:RP:StatementOfDistributionToBeneficiary:Information:TBStatementInformation:IncomeTax.Superannuation.TrusteeBeneficiaryTaxPreferred.Amount
            
                    ^TRT346 = TRT:RP:StatementOfDistributionToBeneficiary:Information:TBStatementInformation:IncomeTax.Superannuation.TrusteeBeneficiaryUntaxedPartShareOfNetIncome.Amount
                    */
                    assertion = (statementOfDistributionToBeneficiary.TRT344 == true && (statementOfDistributionToBeneficiary.TRT345 == 0 || statementOfDistributionToBeneficiary.TRT345 == null) && (statementOfDistributionToBeneficiary.TRT346 == 0 || statementOfDistributionToBeneficiary.TRT346 == null));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.TRT.432347", Severity = ProcessMessageSeverity.Error,
                            Description = @"TB Statement information amount must be present",
                            LongDescription = @"If within a Statement of Distribution the answer to the question 'TB statement' is 'yes' (true), then there must be an amount greater than zero at 'Tax preferred amounts' or Untaxed part of share of net income'",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:Information/tns:TBStatementInformation/tns:IncomeTaxTrustBeneficiaryStatementI",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432347"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT344", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT344) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT345", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT345) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT346", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT346) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.432348
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432348(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.432348
                    Where the trust has indicated that they are not making a TB statement, by answering the TB question No, 'Tax preferred amounts' or Untaxed part of share of net income' must be blank or equal to zero.
    
                    Legacy Rule Format:
                    ^TRT344 <> TRUE AND (^TRT345 > 0 OR ^TRT346 > 0)
        
                    Technical Business Rule Format:
                    ^TRT344 <> TRUE AND (^TRT345 > 0 OR ^TRT346 > 0)
            
                    Data Elements:
            
                    ^TRT344 = TRT:RP:StatementOfDistributionToBeneficiary:Information:TBStatementInformation:IncomeTax.TrustBeneficiaryStatement.Indicator
            
                    ^TRT345 = TRT:RP:StatementOfDistributionToBeneficiary:Information:TBStatementInformation:IncomeTax.Superannuation.TrusteeBeneficiaryTaxPreferred.Amount
            
                    ^TRT346 = TRT:RP:StatementOfDistributionToBeneficiary:Information:TBStatementInformation:IncomeTax.Superannuation.TrusteeBeneficiaryUntaxedPartShareOfNetIncome.Amount
                    */
                    assertion = (statementOfDistributionToBeneficiary.TRT344 != true && (statementOfDistributionToBeneficiary.TRT345.GetValueOrDefault() > 0 || statementOfDistributionToBeneficiary.TRT346.GetValueOrDefault() > 0));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.TRT.432348", Severity = ProcessMessageSeverity.Error,
                            Description = @"TB Statement information amount must NOT be present",
                            LongDescription = @"If within a Statement of Distribution the answer to the question 'TB statement' is not 'yes' (false or blank), then there must NOT be an amount greater than zero at 'Tax preferred amounts' or Untaxed part of share of net income'",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:Information/tns:TBStatementInformation/tns:IncomeTaxTrustBeneficiaryStatementI",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432348"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT344", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT344) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT345", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT345) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT346", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT346) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.432352
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432352()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432352
            Primary production- Share of net income from trusts is greater than zero and there is no primary production-Distribution from trusts action code.
    
            Legacy Rule Format:
            ^TRT77 > 0 AND ^TRT353 = NULL

            Technical Business Rule Format:
            ^TRT77 > 0 AND ^TRT353 = NULL
    
            Data Elements:
    
            ^TRT353 = TRT:RP:BusinessDetails:Income:PartnershipsTrustsPrimaryProduction:Party.TrustType.Code
    
            ^TRT77 = TRT:RP:BusinessDetails:Income:PartnershipsTrustsPrimaryProduction:Income.TrustShareNet.Amount
            */
            assertion = (report.TRT77.GetValueOrDefault() > 0 && report.TRT353 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432660", Severity = ProcessMessageSeverity.Error,
                    Description = @"Share of net income from trusts action code must be present",
                    LongDescription = @"If Primary production- Share of net income from trusts is greater than zero, the Primary production - Share of net income from trusts action code must be present",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:PartnershipsTrustsPrimaryProduction/tns:PartyTrustTypeC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432352"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT77", Value = GetValueOrEmpty(report.TRT77) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT353", Value = GetValueOrEmpty(report.TRT353) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432353
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432353()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432353
            When there is an amount present for non-primary production distribution from trusts, there must be a trust action code.
    
            Legacy Rule Format:
            ^TRT81 > 0 AND ^TRT436 = NULL

            Technical Business Rule Format:
            ^TRT81 > 0 AND ^TRT436 = NULL
    
            Data Elements:
    
            ^TRT436 = TRT:RP:BusinessDetails:Income:PartnershipTrustNonPrimaryProduction:Party.TrustType.Code
    
            ^TRT81 = TRT:RP:BusinessDetails:Income:PartnershipTrustNonPrimaryProduction:Income.TrustShareNetExcludeNetCapitalGainsAndForeignIncomeAndDistributionFranked.Amount
            */
            assertion = (report.TRT81.GetValueOrDefault() > 0 && report.TRT436 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432663", Severity = ProcessMessageSeverity.Error,
                    Description = @"Distribution from trusts action code must be present",
                    LongDescription = @"If Non-primary production - Share of net income from trusts, less net capital gain, foreign income and franked distributions is greater than zero, the Non-primary production - Distribution from trusts action code must be present",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:PartnershipTrustNonPrimaryProduction/tns:PartyTrustTypeC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432353"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT81", Value = GetValueOrEmpty(report.TRT81) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT436", Value = GetValueOrEmpty(report.TRT436) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432384
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432384()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432384
            Death benefit superannuation lump sum must be complete
    
            Legacy Rule Format:
            ^TRT1024 <> NULL AND ((^TRT100 = NULL AND ^TRT101 = NULL) OR (^TRT408 <> TRUE))

            Technical Business Rule Format:
            ^TRT1024 <> NULL AND ((^TRT100 = NULL AND ^TRT101 = NULL) OR (^TRT408 <> TRUE))
    
            Data Elements:
    
            ^TRT101 = TRT:RP:BusinessDetails:Income:SuperannuationLumpSumWithholdingPaymentDetails:SuperannuationBenefit.DeathBenefitLumpSumPaymentUntaxedElement.Amount
    
            ^TRT100 = TRT:RP:BusinessDetails:Income:SuperannuationLumpSumWithholdingPaymentDetails:SuperannuationBenefit.DeathBenefitLumpSumPaymentTaxedElement.Amount
    
            ^TRT408 = TRT:RP:BusinessDetails:Income:SuperannuationLumpSumWithholdingPaymentDetails:InsuranceBenefits.DeathBenefit.Indicator
    
            ^TRT1024 = TRT:RP:BusinessDetails:Income:SuperannuationLumpSumWithholdingPaymentDetails
            */
            assertion = (report.RP_BusinessDetails_Income_SuperannuationLumpSumWithholdingPaymentDetailsCollectionExists != false && (report.TRT100 == null && report.TRT101 == null || report.TRT408 != true));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.442384", Severity = ProcessMessageSeverity.Error,
                    Description = @"Death benefit superannuation lump sum must be complete",
                    LongDescription = @"The 'Death benefit superannuation lump sum where the beneficiary is a non-dependant' must include a 'Taxed element' amount or 'Untaxed element' amount (or both), plus Death benefit indicator set to 'TRUE'",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:SuperannuationLumpSumWithholdingPaymentDetails/tns:SuperannuationBenefitDeathBenefitLumpSumPaymentUntaxedElementA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432384"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT100", Value = GetValueOrEmpty(report.TRT100) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT101", Value = GetValueOrEmpty(report.TRT101) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT408", Value = GetValueOrEmpty(report.TRT408) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432419
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432419()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432419
            The number of ETP payments for a dependent beneficiary must not exceed 1.
    
            Legacy Rule Format:
            CountOccurrence(^TRT1025, ^TRT411 = "D") > 1

            Technical Business Rule Format:
            CountOccurrence(^TRT1025, ^TRT411 = 'D') > 1
    
            Data Elements:
    
            ^TRT411 = TRT:RP:BusinessDetails:Income:EmploymentTerminationPaymentDetails:InsuranceBenefits.BeneficiaryType.Code
    
            ^TRT1025 = TRT:RP:BusinessDetails:Income:EmploymentTerminationPaymentDetails
            */
            assertion = ((report.RP_BusinessDetails_Income_EmploymentTerminationPaymentDetailsCollection == null ? 0 : report.RP_BusinessDetails_Income_EmploymentTerminationPaymentDetailsCollection.Count(TRT1025Repeat => TRT1025Repeat.TRT411 == @"D")) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432419", Severity = ProcessMessageSeverity.Error,
                    Description = @"The number of ETP payments for a dependent beneficiary must not exceed 1.",
                    LongDescription = @"You may provide a maximum of two instances of Eligible termination payments. One for where the beneficiary is a dependant and another where the beneficiary is a non-dependant.",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:EmploymentTerminationPaymentDetailsCollection/tns:EmploymentTerminationPaymentDetails/tns:InsuranceBenefitsBeneficiaryTypeC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432419"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432420
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432420(TRT2024.RP_BusinessDetails_Income_EmploymentTerminationPaymentDetails employmentTerminationPaymentDetails, int itemIndex4)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.432420
                    Amount, type code and indicator must be present for a death benefit ETP
    
                    Legacy Rule Format:
                    ^TRT411 = 'D' AND (^TRT102 = NULL OR ^TRT409 <> TRUE)
        
                    Technical Business Rule Format:
                    ^TRT411 = 'D' AND (^TRT102 = NULL OR ^TRT409 <> TRUE)
            
                    Data Elements:
            
                    ^TRT102 = TRT:RP:BusinessDetails:Income:EmploymentTerminationPaymentDetails:SuperannuationBenefit.DeathBenefitEmploymentTerminationPaymentTaxableComponent.Amount
            
                    ^TRT409 = TRT:RP:BusinessDetails:Income:EmploymentTerminationPaymentDetails:InsuranceBenefits.DeathBenefit.Indicator
            
                    ^TRT411 = TRT:RP:BusinessDetails:Income:EmploymentTerminationPaymentDetails:InsuranceBenefits.BeneficiaryType.Code
                    */
                    assertion = (employmentTerminationPaymentDetails.TRT411 == @"D" && (employmentTerminationPaymentDetails.TRT102 == null || employmentTerminationPaymentDetails.TRT409 != true));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.TRT.432420", Severity = ProcessMessageSeverity.Error,
                            Description = @"Amount, type code and indicator must be present for a death benefit ETP",
                            LongDescription = @"For each Death benefit employment termination payment, an amount, a type code - indicating 'd' - dependant or 'o' -non-dependant, and a death benefit indicator set to 'yes' (true), must all be present.",
                            Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:EmploymentTerminationPaymentDetailsCollection/tns:EmploymentTerminationPaymentDetails" + OccurrenceIndex(employmentTerminationPaymentDetails.OccurrenceIndex) + "/tns:SuperannuationBenefitDeathBenefitEmploymentTerminationPaymentTaxableComponentA",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432420"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT411", Value = employmentTerminationPaymentDetails.TRT411 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT102", Value = GetValueOrEmpty(employmentTerminationPaymentDetails.TRT102) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT409", Value = GetValueOrEmpty(employmentTerminationPaymentDetails.TRT409) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.432421
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432421()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432421
            If an amount greater than zero is present at TFN amounts withheld from gross interest, then an amount greater than zero must also be present at Gross interest.
    
            Legacy Rule Format:
            ^TRT95 > 0 AND (^TRT94 = 0 OR ^TRT94 = NULL)

            Technical Business Rule Format:
            ^TRT95 > 0 AND (^TRT94 = 0 OR ^TRT94 = NULL)
    
            Data Elements:
    
            ^TRT94 = TRT:RP:BusinessDetails:Income:GrossInterest:Income.Interest.Gross.Amount
    
            ^TRT95 = TRT:RP:BusinessDetails:Income:GrossInterest:IncomeTax.TFNAmountsWithheldFromGrossInterest.Amount
            */
            assertion = (report.TRT95.GetValueOrDefault() > 0 && (report.TRT94 == 0 || report.TRT94 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430092", Severity = ProcessMessageSeverity.Error,
                    Description = @"Gross interest amount greater than zero must be present",
                    LongDescription = @"If an amount greater than zero is present at TFN amounts withheld from gross interest, then an amount greater than zero must also be present at Gross interest",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:GrossInterest/tns:A",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432421"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT95", Value = GetValueOrEmpty(report.TRT95) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT94", Value = GetValueOrEmpty(report.TRT94) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432425
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432425(TRT2024.RP_BusinessDetails_Income_IndividualNonBusinessWithholdingPaymentDetails individualNonBusinessWithholdingPaymentDetails, int itemIndex4)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.432425
                    At least one amount must be present for Other Australian Income - withholding details
    
                    Legacy Rule Format:
                    ^TRT108 = NULL AND ^TRT109 = NULL AND ^TRT110 = NULL AND ^TRT111 = NULL AND ^TRT112 = NULL AND ^TRT113 = NULL
        
                    Technical Business Rule Format:
                    ^TRT108 = NULL AND ^TRT109 = NULL AND ^TRT110 = NULL AND ^TRT111 = NULL AND ^TRT112 = NULL AND ^TRT113 = NULL
            
                    Data Elements:
            
                    ^TRT108 = TRT:RP:BusinessDetails:Income:IndividualNonBusinessWithholdingPaymentDetails:IncomeTax.PayAsYouGoWithholding.TaxWithheld.Amount
            
                    ^TRT109 = TRT:RP:BusinessDetails:Income:IndividualNonBusinessWithholdingPaymentDetails:Remuneration.IndividualNonBusinessGross.Amount
            
                    ^TRT110 = TRT:RP:BusinessDetails:Income:IndividualNonBusinessWithholdingPaymentDetails:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldUnusedAnnualOrLongServiceLeavePaymentLumpSumA.Amount
            
                    ^TRT111 = TRT:RP:BusinessDetails:Income:IndividualNonBusinessWithholdingPaymentDetails:Remuneration.UnusedAnnualLongServiceLeavePaymentLumpSumA.Amount
            
                    ^TRT112 = TRT:RP:BusinessDetails:Income:IndividualNonBusinessWithholdingPaymentDetails:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldUnusedAnnualOrLongServiceLeavePaymentLumpSumB.Amount
            
                    ^TRT113 = TRT:RP:BusinessDetails:Income:IndividualNonBusinessWithholdingPaymentDetails:Remuneration.UnusedAnnualLongServiceLeavePaymentLumpSumB.Amount
                    */
                    assertion = (individualNonBusinessWithholdingPaymentDetails.TRT108 == null && individualNonBusinessWithholdingPaymentDetails.TRT109 == null && individualNonBusinessWithholdingPaymentDetails.TRT110 == null && individualNonBusinessWithholdingPaymentDetails.TRT111 == null && individualNonBusinessWithholdingPaymentDetails.TRT112 == null && individualNonBusinessWithholdingPaymentDetails.TRT113 == null);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.TRT.432425", Severity = ProcessMessageSeverity.Error,
                            Description = @"At least one amount must be present for Other Australian Income - withholding details",
                            LongDescription = @"For each ABN that is provided against Other Australian income - Withholding details, at least one payment amount must be present",
                            Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:IndividualNonBusinessWithholdingPaymentDetailsCollection/tns:IndividualNonBusinessWithholdingPaymentDetails" + OccurrenceIndex(individualNonBusinessWithholdingPaymentDetails.OccurrenceIndex) + "/tns:TaxPayAsYouGoWithholdingTaxWithheldA",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432425"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT108", Value = GetValueOrEmpty(individualNonBusinessWithholdingPaymentDetails.TRT108) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT109", Value = GetValueOrEmpty(individualNonBusinessWithholdingPaymentDetails.TRT109) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT110", Value = GetValueOrEmpty(individualNonBusinessWithholdingPaymentDetails.TRT110) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT111", Value = GetValueOrEmpty(individualNonBusinessWithholdingPaymentDetails.TRT111) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT112", Value = GetValueOrEmpty(individualNonBusinessWithholdingPaymentDetails.TRT112) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT113", Value = GetValueOrEmpty(individualNonBusinessWithholdingPaymentDetails.TRT113) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.432562
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432562()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432562
            Reporting period start date must be prior to Reporting period end date
    
            Legacy Rule Format:
            ^TRT573 >= ^TRT574

            Technical Business Rule Format:
            ^TRT573 >= ^TRT574
    
            Data Elements:
    
            ^TRT573 = TRT:RP:Period.Start.Date
    
            ^TRT574 = TRT:RP:Period.End.Date
            */
            assertion = (report.TRT573.GetValueOrDefault() >= report.TRT574.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009", Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/tns:TRT/tns:RP/tns:PeriodStartD",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432562"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT573", Value = GetValueOrEmptyDate(report.TRT573) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT574", Value = GetValueOrEmptyDate(report.TRT574) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432567
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432567()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432567
            The number of ETP payments for a non-dependent beneficiary must not exceed 1.
    
            Legacy Rule Format:
            CountOccurrence(^TRT1025, ^TRT411 = "O") > 1

            Technical Business Rule Format:
            CountOccurrence(^TRT1025, ^TRT411 = 'O') > 1
    
            Data Elements:
    
            ^TRT411 = TRT:RP:BusinessDetails:Income:EmploymentTerminationPaymentDetails:InsuranceBenefits.BeneficiaryType.Code
    
            ^TRT1025 = TRT:RP:BusinessDetails:Income:EmploymentTerminationPaymentDetails
            */
            assertion = ((report.RP_BusinessDetails_Income_EmploymentTerminationPaymentDetailsCollection == null ? 0 : report.RP_BusinessDetails_Income_EmploymentTerminationPaymentDetailsCollection.Count(TRT1025Repeat => TRT1025Repeat.TRT411 == @"O")) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432567", Severity = ProcessMessageSeverity.Error,
                    Description = @"The number of ETP payments for a non-dependent beneficiary must not exceed 1.",
                    LongDescription = @"You may provide a maximum of two instances of Eligible termination payments. One for where the beneficiary is a dependant and another where the beneficiary is a non-dependant.",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:EmploymentTerminationPaymentDetailsCollection/tns:EmploymentTerminationPaymentDetails/tns:InsuranceBenefitsBeneficiaryTypeC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432567"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432572
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432572()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432572
            Telephone must include area code for landline numbers.
    
            Legacy Rule Format:
            ^TRT334 = '02' AND ^TRT249 <> NULL AND ^TRT248 = NULL

            Technical Business Rule Format:
            ^TRT334 = '02' AND ^TRT249 <> NULL AND ^TRT248 = NULL
    
            Data Elements:
    
            ^TRT248 = TRT:INT:IntermediaryContactTelephone:ElectronicContact.Telephone.Area.Code
    
            ^TRT249 = TRT:INT:IntermediaryContactTelephone:ElectronicContact.Telephone.Minimal.Number
    
            ^TRT334 = TRT:INT:IntermediaryContactTelephone:ElectronicContact.Telephone.ServiceLine.Code
            */
            assertion = (report.TRT334 == @"02" && report.TRT249 != null && report.TRT248 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432572", Severity = ProcessMessageSeverity.Error,
                    Description = @"Telephone must include area code for landline numbers.",
                    LongDescription = @"Telephone must include area code for landline numbers.",
                    Location = "/tns:TRT/tns:INT/tns:IntermediaryContactTelephone/tns:ElectronicContactAreaC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432572"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT334", Value = report.TRT334 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT249", Value = report.TRT249 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT248", Value = GetValueOrEmpty(report.TRT248) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432582
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432582(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.432582
                    Individual name must be present for individual beneficiaries.
    
                    Legacy Rule Format:
                    ^TRT432 = 'I' AND Count(^TRT1072) = 0
        
                    Technical Business Rule Format:
                    ^TRT432 = 'I' AND Count(^TRT1072) = 0
            
                    Data Elements:
            
                    ^TRT432 = TRT:RP:StatementOfDistributionToBeneficiary:IncomeTax.BeneficiaryEntityType.Code
            
                    ^TRT1072 = TRT:RP:StatementOfDistributionToBeneficiary:IndividualNameDetails
                    */
                    assertion = (statementOfDistributionToBeneficiary.TRT432 == @"I" && Count(statementOfDistributionToBeneficiary.RP_StatementOfDistributionToBeneficiary_IndividualNameDetailsCollectionCount) == 0);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.TRT.432582", Severity = ProcessMessageSeverity.Error,
                            Description = @"Individual name must be present for individual beneficiaries.",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:IncomeTaxBeneficiaryEntityTypeC",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432582"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT432", Value = statementOfDistributionToBeneficiary.TRT432 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.432583
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432583(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.432583
                    Non-individual name must be present for non-individual beneficiaries.
    
                    Legacy Rule Format:
                    ^TRT432 <> 'I' AND Count(^TRT1071) = 0
        
                    Technical Business Rule Format:
                    ^TRT432 <> 'I' AND Count(^TRT1071) = 0
            
                    Data Elements:
            
                    ^TRT432 = TRT:RP:StatementOfDistributionToBeneficiary:IncomeTax.BeneficiaryEntityType.Code
            
                    ^TRT1071 = TRT:RP:StatementOfDistributionToBeneficiary:NonIndividualNameDetails
                    */
                    assertion = (statementOfDistributionToBeneficiary.TRT432 != @"I" && Count(statementOfDistributionToBeneficiary.RP_StatementOfDistributionToBeneficiary_NonIndividualNameDetailsCollectionCount) == 0);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.TRT.432583", Severity = ProcessMessageSeverity.Error,
                            Description = @"Non-individual name must be present for non-individual beneficiaries.",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:IncomeTaxBeneficiaryEntityTypeC",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432583"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT432", Value = statementOfDistributionToBeneficiary.TRT432 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.432584
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432584(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.432584
                    Date of birth must be present for individual beneficiaries.
    
                    Legacy Rule Format:
                    ^TRT213 = NULL AND ^TRT432 = 'I'
        
                    Technical Business Rule Format:
                    ^TRT213 = NULL AND ^TRT432 = 'I'
            
                    Data Elements:
            
                    ^TRT432 = TRT:RP:StatementOfDistributionToBeneficiary:IncomeTax.BeneficiaryEntityType.Code
            
                    ^TRT213 = TRT:RP:StatementOfDistributionToBeneficiary:PersonDemographicDetails.Birth.Date
                    */
                    assertion = (statementOfDistributionToBeneficiary.TRT213 == null && statementOfDistributionToBeneficiary.TRT432 == @"I");
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.TRT.432584", Severity = ProcessMessageSeverity.Error,
                            Description = @"Date of birth must be present for individual beneficiaries.",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:IncomeTaxBeneficiaryEntityTypeC",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432584"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT213", Value = GetValueOrEmptyDate(statementOfDistributionToBeneficiary.TRT213) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT432", Value = statementOfDistributionToBeneficiary.TRT432 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.432585
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432585(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.432585
                    Assessment calculation codes are not correct for company beneficiaries.
    
                    Legacy Rule Format:
                    NotInSet(^TRT214, '"13","34","119","134","139"') AND ^TRT432 = 'C'
        
                    Technical Business Rule Format:
                    NotInSet(^TRT214, '"13","34","119","134","139"') AND ^TRT432 = 'C'
            
                    Data Elements:
            
                    ^TRT432 = TRT:RP:StatementOfDistributionToBeneficiary:IncomeTax.BeneficiaryEntityType.Code
            
                    ^TRT214 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.AssessmentCalculation.Code
                    */
                    assertion = (!(IsMatch(statementOfDistributionToBeneficiary.TRT214, @"^(13|34|119|134|139)$",RegexOptions.IgnoreCase)) && statementOfDistributionToBeneficiary.TRT432 == @"C");
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.TRT.432585", Severity = ProcessMessageSeverity.Error,
                            Description = @"Assessment calculation codes are not correct for company beneficiaries.",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:IncomeTaxBeneficiaryEntityTypeC",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432585"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT214", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT214) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT432", Value = statementOfDistributionToBeneficiary.TRT432 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.432586
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432586(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.432586
                    Assessment calculation codes are not correct for trustee beneficiaries
    
                    Legacy Rule Format:
                    NotInSet(^TRT214, '"14","35","45","120","140","145"') AND ^TRT432 = 'T'
        
                    Technical Business Rule Format:
                    NotInSet(^TRT214, '"14","35","45","120","140","145"') AND ^TRT432 = 'T'
            
                    Data Elements:
            
                    ^TRT432 = TRT:RP:StatementOfDistributionToBeneficiary:IncomeTax.BeneficiaryEntityType.Code
            
                    ^TRT214 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.AssessmentCalculation.Code
                    */
                    assertion = (!(IsMatch(statementOfDistributionToBeneficiary.TRT214, @"^(14|35|45|120|140|145)$",RegexOptions.IgnoreCase)) && statementOfDistributionToBeneficiary.TRT432 == @"T");
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.TRT.432586", Severity = ProcessMessageSeverity.Error,
                            Description = @"Assessment calculation codes are not correct for trustee beneficiaries",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:IncomeTaxBeneficiaryEntityTypeC",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432586"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT214", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT214) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT432", Value = statementOfDistributionToBeneficiary.TRT432 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.432593
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432593()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432593
            The beneficiary TFN reported must not be reported more than once.
    
            Legacy Rule Format:
            HasDuplicateValues(^TRT211)

            Technical Business Rule Format:
            HasDuplicateValues(^TRT211)
    
            Data Elements:
    
            ^TRT211 = TRT:RP:StatementOfDistributionToBeneficiary:Identifiers.TaxFileNumber.Identifier
            */
            assertion = HasDuplicateValues(report.RP_StatementOfDistributionToBeneficiaryCollection == null ? null : report.RP_StatementOfDistributionToBeneficiaryCollection.Select(f => f.TRT211).Cast<object>().ToArray());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432593", Severity = ProcessMessageSeverity.Error,
                    Description = @"The beneficiary TFN reported must not be reported more than once.",
                    Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary[" + report.RP_StatementOfDistributionToBeneficiaryCollection[DuplicateValueIndex(report.RP_StatementOfDistributionToBeneficiaryCollection.Select(f => f.TRT211))].OccurrenceIndex + "]/tns:TaxFileNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432593"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432600
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432600()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432600
            An amount greater than zero is present at credit for tax withheld where ABN not quoted (beneficiary not entitled) and there is no primary production or non-primary production income distribution or franked distributions amount.
    
            Legacy Rule Format:
            ^TRT236 > 0  AND ^TRT234 = NULL AND ^TRT235 = NULL AND ^TRT444 = NULL

            Technical Business Rule Format:
            ^TRT236 > 0  AND ^TRT234 = NULL AND ^TRT235 = NULL AND ^TRT444 = NULL
    
            Data Elements:
    
            ^TRT236 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount
    
            ^TRT234 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:PrimaryProductionNoBeneficiaryEntitled:IncomeTax.IncomeOrLossTaxable.Amount
    
            ^TRT235 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:NonPrimaryProductionNoBeneficiaryEntitled:IncomeTax.IncomeOrLossTaxable.Amount
    
            ^TRT444 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:Income.DistributionFranked.Amount
            */
            assertion = (report.TRT236.GetValueOrDefault() > 0 && report.TRT234 == null && report.TRT235 == null && report.TRT444 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432620", Severity = ProcessMessageSeverity.Error,
                    Description = @"PP or Non-PP income distribution or franked distributions must be present in Statement of distribution",
                    LongDescription = @"If an amount greater than zero is present at 'Credit for tax withheld where ABN not quoted' against 'Income to which no beneficiary is presently entitled...', then there must be Primary production or Non-primary production income or franked distributions amount for 'income to which no beneficiary is presently entitled'.",
                    Location = "/tns:TRT/tns:RP/tns:IncomeNoBeneficiaryPresentlyEntitled/tns:IncomeTaxPayAsYouGoWithholdingCreditForTaxWithheldWhereABNNotQuotedA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432600"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT236", Value = GetValueOrEmpty(report.TRT236) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT234", Value = GetValueOrEmpty(report.TRT234) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT235", Value = GetValueOrEmpty(report.TRT235) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT444", Value = GetValueOrEmpty(report.TRT444) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432607
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432607()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432607
            Where credit for tax withheld - foreign resident withholding (excluding capital gains) is greater than zero, there must be an amount at either share of income - primary or non-primary production or franked distributions. No beneficiary entitled.
    
            Legacy Rule Format:
            ^TRT477 > 0 AND ^TRT234 = NULL AND ^TRT235 = NULL AND ^TRT444 = NULL

            Technical Business Rule Format:
            ^TRT477 > 0 AND ^TRT234 = NULL AND ^TRT235 = NULL AND ^TRT444 = NULL
    
            Data Elements:
    
            ^TRT477 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromForeignResidents.Amount
    
            ^TRT234 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:PrimaryProductionNoBeneficiaryEntitled:IncomeTax.IncomeOrLossTaxable.Amount
    
            ^TRT235 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:NonPrimaryProductionNoBeneficiaryEntitled:IncomeTax.IncomeOrLossTaxable.Amount
    
            ^TRT444 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:Income.DistributionFranked.Amount
            */
            assertion = (report.TRT477.GetValueOrDefault() > 0 && report.TRT234 == null && report.TRT235 == null && report.TRT444 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432741", Severity = ProcessMessageSeverity.Error,
                    Description = @"PP or Non PP Share of income or franked distributions must be present",
                    LongDescription = @"If within a Statement of Distribution there is an amount greater than zero at 'Credit for tax withheld - foreign resident withholding (excluding capital gains)', then there must be an amount at either 'Share of income - primary production'  or 'Share of income - non-primary production' or 'Franked distributions' where 'Income to which no beneficiary is presently entitled'.",
                    Location = "/tns:TRT/tns:RP/tns:IncomeNoBeneficiaryPresentlyEntitled/tns:IncomeTaxPayAsYouGoWithholdingCreditForAmountsWithheldFromForeignResidentsA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432607"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT477", Value = GetValueOrEmpty(report.TRT477) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT234", Value = GetValueOrEmpty(report.TRT234) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT235", Value = GetValueOrEmpty(report.TRT235) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT444", Value = GetValueOrEmpty(report.TRT444) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432609
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432609()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432609
            Family Trust Election Status - Specified income year must not be less than 1995 or greater than the current reporting year.
    
            Legacy Rule Format:
            ^TRT28 <> NULL AND (^TRT28 < 1995 OR ^TRT28 > ^TRT1)

            Technical Business Rule Format:
            ^TRT28 <> NULL AND (^TRT28 < 1995 OR ^TRT28 > ^TRT1)
    
            Data Elements:
    
            ^TRT28 = TRT:RP:ElectionStatus:Elections.FamilyTrustElectionStatus.Year
    
            ^TRT1 = TRT:RP:Report.TargetFinancial.Year
            */
            assertion = (report.TRT28 != null && (report.TRT28.GetValueOrDefault() < 1995 || report.TRT28.GetValueOrDefault() > report.TRT1.GetValueOrDefault()));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432609", Severity = ProcessMessageSeverity.Error,
                    Description = @"Family Trust Election Status - Specified income year must not be less than 1995 or greater than the current reporting year.",
                    Location = "/tns:TRT/tns:RP/tns:ElectionStatus/tns:ElectionsFamilyTrustElectionStatusY",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432609"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT28", Value = GetValueOrEmpty(report.TRT28) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT1", Value = GetValueOrEmpty(report.TRT1) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432610
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432610()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432610
            Interposed Entity Election Status - Specified income year must not be less than 1995 or greater than the current reporting year.
    
            Legacy Rule Format:
            ^TRT30 <> NULL AND (^TRT30 < 1995 OR ^TRT30 > ^TRT1)

            Technical Business Rule Format:
            ^TRT30 <> NULL AND (^TRT30 < 1995 OR ^TRT30 > ^TRT1)
    
            Data Elements:
    
            ^TRT30 = TRT:RP:InterposedEntityElectionStatus:Elections.InterposedEntityElectionStatus.Year
    
            ^TRT1 = TRT:RP:Report.TargetFinancial.Year
            */
            assertion = (report.TRT30 != null && (report.TRT30.GetValueOrDefault() < 1995 || report.TRT30.GetValueOrDefault() > report.TRT1.GetValueOrDefault()));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432610", Severity = ProcessMessageSeverity.Error,
                    Description = @"Interposed Entity Election Status - Specified income year must not be less than 1995 or greater than the current reporting year.",
                    Location = "/tns:TRT/tns:RP/tns:InterposedEntityElectionStatus/tns:ElectionsY",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432610"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT30", Value = GetValueOrEmpty(report.TRT30) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT1", Value = GetValueOrEmpty(report.TRT1) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432612
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432612()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432612
            If tax offset code is present then the tax offset amount must be present and greater than zero
    
            Legacy Rule Format:
            (^TRT463 = 0 OR ^TRT463 = NULL) AND ^TRT464 <> NULL

            Technical Business Rule Format:
            (^TRT463 = 0 OR ^TRT463 = NULL) AND ^TRT464 <> NULL
    
            Data Elements:
    
            ^TRT464 = TRT:RP:TaxOffsets:IncomeTax.TaxOffsetRefundableOther.Code
    
            ^TRT463 = TRT:RP:TaxOffsets:IncomeTax.TaxOffsetRefundableOther.Amount
            */
            assertion = ((report.TRT463 == 0 || report.TRT463 == null) && report.TRT464 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432640", Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax offset amount is equal to zero or not present but code is present",
                    Location = "/tns:TRT/tns:RP/tns:TaxOffsets/tns:IncomeTaxTaxOffsetRefundableOtherC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432612"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT463", Value = GetValueOrEmpty(report.TRT463) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT464", Value = GetValueOrEmpty(report.TRT464) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432631
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432631()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432631
            If there is an amount present at non-primary production deductions relating to franked distributions from trusts, then there must be an amount at non-primary production franked distributions from trusts.
    
            Legacy Rule Format:
            ^TRT438 > 0 AND ^TRT437 = NULL

            Technical Business Rule Format:
            ^TRT438 > 0 AND ^TRT437 = NULL
    
            Data Elements:
    
            ^TRT437 = TRT:RP:BusinessDetails:Income:PartnershipTrustNonPrimaryProduction:Income.TrustDistributionFranked.Amount
    
            ^TRT438 = TRT:RP:BusinessDetails:Income:PartnershipTrustNonPrimaryProduction:IncomeTax.Deduction.TrustDistributionFranked.Amount
            */
            assertion = (report.TRT438.GetValueOrDefault() > 0 && report.TRT437 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432631", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-primary production franked distributions from trusts amount must be present",
                    LongDescription = @"Non-primary production 'Deductions relating to franked distributions from trusts' is greater than zero, however no non-primary production 'Franked distributions from trusts' amount exists.",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:PartnershipTrustNonPrimaryProduction/tns:TrustDistributionFrankedA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432631"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT438", Value = GetValueOrEmpty(report.TRT438) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT437", Value = GetValueOrEmpty(report.TRT437) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432633
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432633()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432633
            If No beneficiary presently entitled assessment calculation code and Choice for resident trustee to be assessed to capital gains on behalf of beneficiaries assessment calculation code both exist, the assessment calculation code value for both must be identical.
    
            Legacy Rule Format:
            ^TRT230 <> NULL AND ^TRT445 <> NULL AND ^TRT230 <> ^TRT445

            Technical Business Rule Format:
            ^TRT230 <> NULL AND ^TRT445 <> NULL AND ^TRT230 <> ^TRT445
    
            Data Elements:
    
            ^TRT445 = TRT:RP:CapitalGainsAssessed:CapitalGainsTax.AssessmentCalculation.Code
    
            ^TRT230 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.AssessmentCalculation.Code
            */
            assertion = (report.TRT230 != null && report.TRT445 != null && report.TRT230 != report.TRT445);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432633", Severity = ProcessMessageSeverity.Error,
                    Description = @"Assessment calculation code is not correct",
                    LongDescription = @"No beneficiary presently entitled assessment calculation code and Choice for resident trusteee to be assessed to capital gains on behalf of beneficiaries assessment calculation code are present, but the assessment calculation code value of each are not identical.",
                    Location = "/tns:TRT/tns:RP/tns:CapitalGainsAssessed/tns:CapitalGainsTaxAssessmentCalculationC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432633"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT230", Value = GetValueOrEmpty(report.TRT230) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT445", Value = report.TRT445 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432634
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432634()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432634
            Where there are deductions relating to franked distributions greater than zero, then there must be an income amount at franked dividends.
    
            Legacy Rule Format:
            ^TRT439 > 0 AND ^TRT97 = NULL

            Technical Business Rule Format:
            ^TRT439 > 0 AND ^TRT97 = NULL
    
            Data Elements:
    
            ^TRT97 = TRT:RP:BusinessDetails:Income:Dividends:Income.DividendsFranked.Amount
    
            ^TRT439 = TRT:RP:BusinessDetails:Income:Deductions:IncomeTax.Deduction.DistributionFranked.Amount
            */
            assertion = (report.TRT439.GetValueOrDefault() > 0 && report.TRT97 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432634", Severity = ProcessMessageSeverity.Error,
                    Description = @"Franked Dividends Amount must be present",
                    LongDescription = @"If 'Deductions relating to franked distributions' is greater than zero, then there must be a 'Dividends - Franked Amount'",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:Dividends/tns:FrankedA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432634"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT439", Value = GetValueOrEmpty(report.TRT439) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT97", Value = GetValueOrEmpty(report.TRT97) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432694
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432694()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432694
            Exemption or rollover code must be unique
    
            Legacy Rule Format:
            HasDuplicateValues(^TRT448)

            Technical Business Rule Format:
            HasDuplicateValues(^TRT448)
    
            Data Elements:
    
            ^TRT448 = TRT:RP:BusinessDetails:Income:CapitalGains:CapitalGainsTax.ExemptionOrRolloverApplied.Code
            */
            assertion = HasDuplicateValues(report.TRT448Collection);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432694", Severity = ProcessMessageSeverity.Error,
                    Description = @"Exemption or rollover code must be unique",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:CapitalGains/tns:TaxExemptionOrRolloverAppliedCCollection/tns:TaxExemptionOrRolloverAppliedC" + OccurrenceIndex(DuplicateValueIndex(report.TRT448Collection) + 1) + "",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432694"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432713
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432713()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432713
            If the amount in tax offset is greater than zero then tax offsets code must be present
    
            Legacy Rule Format:
            ^TRT463 > 0 AND ^TRT464 = NULL

            Technical Business Rule Format:
            ^TRT463 > 0 AND ^TRT464 = NULL
    
            Data Elements:
    
            ^TRT463 = TRT:RP:TaxOffsets:IncomeTax.TaxOffsetRefundableOther.Amount
    
            ^TRT464 = TRT:RP:TaxOffsets:IncomeTax.TaxOffsetRefundableOther.Code
            */
            assertion = (report.TRT463.GetValueOrDefault() > 0 && report.TRT464 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432638", Severity = ProcessMessageSeverity.Error,
                    Description = @"Code not present for tax offset amount greater than zero",
                    Location = "/tns:TRT/tns:RP/tns:TaxOffsets/tns:IncomeTaxTaxOffsetRefundableOtherA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432713"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT463", Value = GetValueOrEmpty(report.TRT463) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT464", Value = GetValueOrEmpty(report.TRT464) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432730
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432730()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432730
            Exploration credit refundable tax offset cannot be claimed by non-resident entities
    
            Legacy Rule Format:
            ^TRT245 = 'NonResidentNoPermEstab' AND ^TRT464 = 'ExplorationCreditRefundableTaxOffset'

            Technical Business Rule Format:
            ^TRT245 = 'NonResidentNoPermEstab' AND ^TRT464 = 'ExplorationCreditRefundableTaxOffset'
    
            Data Elements:
    
            ^TRT245 = TRT:RP:NonResidentTrust:Residency.TaxPurposesOrganisationStatus.Code
    
            ^TRT464 = TRT:RP:TaxOffsets:IncomeTax.TaxOffsetRefundableOther.Code
            */
            assertion = (report.TRT245 == @"NonResidentNoPermEstab" && report.TRT464 == @"ExplorationCreditRefundableTaxOffset");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432730", Severity = ProcessMessageSeverity.Error,
                    Description = @"Exploration credit tax offset cannot be claimed",
                    LongDescription = @"Exploration credit refundable tax offset cannot be claimed by non-resident entities",
                    Location = "/tns:TRT/tns:RP/tns:NonResidentTrust/tns:ResidencyTaxPurposesOrganisationStatusC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432730"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT245", Value = report.TRT245 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT464", Value = GetValueOrEmpty(report.TRT464) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432731
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432731()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432731
            Year of return must be either the current or next (future) lodgment year associated with this service. If a future year lodgment is required, then it must be lodged no later than the 15th of June of the current year, unless it is an amendment. After the 15th of June, a return for that lodgment year must be submitted from the 1st of July, using the new service associated to that specific lodgment year.
    
            Legacy Rule Format:
            NotInSet(^TRT1, '"2024", "2025"') OR (^TRT1 = 2025 AND Today() > AsDate('2025-06-15') AND ^TRT414 <> TRUE)

            Technical Business Rule Format:
            NotInSet(^TRT1, '"2024", "2025"') OR (^TRT1 = 2025 AND Today() > AsDate('2025-06-15') AND ^TRT414 <> TRUE)
    
            Data Elements:
    
            ^TRT1 = TRT:RP:Report.TargetFinancial.Year
    
            ^TRT414 = TRT:RP:ReportAmendment:Report.Amendment.Indicator
            */
            assertion = (!(IsMatch(report.TRT1.GetValueOrDefault().ToString(), @"^(2024|2025)$",RegexOptions.IgnoreCase)) || report.TRT1.GetValueOrDefault() == 2025 && DateTime.Now.Date > AsDate(@"2025-06-15") && report.TRT414 != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500127", Severity = ProcessMessageSeverity.Error,
                    Description = @"Year of return is invalid",
                    LongDescription = @"Year of return must be either the current or next (future) lodgment year associated with this service. If a future year lodgment is required, then it must be lodged no later than the 15th of June of the current year, unless it is an amendment. After the 15th of June, a return for that lodgment year must be submitted from the 1st of July, using the new service associated to that specific lodgment year.",
                    Location = "/tns:TRT/tns:RP/tns:TargetFinancialY",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432731"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT1", Value = GetValueOrEmpty(report.TRT1) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT414", Value = GetValueOrEmpty(report.TRT414) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432786
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432786()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432786
            The sum of Early stage investor tax offset and Early stage investor tax offset carried forward from previous year- No beneficiary presently entitled cannot exceed $200,000.
    
            Legacy Rule Format:
            (^TRT501 <> NULL OR ^TRT507 <> NULL) AND ^TRT501 + ^TRT507 > 200000

            Technical Business Rule Format:
            (^TRT501 <> NULL OR ^TRT507 <> NULL) AND ^TRT501 + ^TRT507 > 200000
    
            Data Elements:
    
            ^TRT501 = TRT:RP:TaxOffsets:InvestorTaxOffset:TaxOffsetClaim.NonRefundableOther.Amount
    
            ^TRT507 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:InvestorNoBeneficiary:InvestorNoBeneficiaryCarriedForward:TaxOffsetClaim.NonRefundableOther.Amount
            */
            assertion = ((report.TRT501 != null || report.TRT507 != null) && report.TRT501.GetValueOrDefault() + report.TRT507.GetValueOrDefault() > 200000);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432786", Severity = ProcessMessageSeverity.Error,
                    Description = @"The sum of Early stage investor tax offset and Early stage investor tax offset carried forward from previous year cannot exceed $200,000.",
                    LongDescription = @"This offset is capped at $200,000 per investor and their affiliates combined each income year. This cap applies to the sum of the current year entitlement and tax offset carried forward from the previous year.",
                    Location = "/tns:TRT/tns:RP/tns:TaxOffsets/tns:InvestorTaxOffsetTaxOffsetClaimNonRefundableOtherA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432786"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT501", Value = GetValueOrEmpty(report.TRT501) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT507", Value = GetValueOrEmpty(report.TRT507) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432801
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432801()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432801
            If the trust is not a Managed Investment Trust (MIT), then MIT type code and election into capital account treatment must be left blank.
    
            Legacy Rule Format:
            ^TRT520 = FALSE AND (^TRT521 <> NULL OR ^TRT352 <> NULL)

            Technical Business Rule Format:
            ^TRT520 = FALSE AND (^TRT521 <> NULL OR ^TRT352 <> NULL)
    
            Data Elements:
    
            ^TRT520 = TRT:RP:TypeOfTrust:OrganisationDetails.ManagedInvestmentTrust.Indicator
    
            ^TRT352 = TRT:RP:TypeOfTrust:Elections.ManagedInvestmentTrusts.Indicator
    
            ^TRT521 = TRT:RP:TypeOfTrust:OrganisationDetails.ManagedInvestmentTrustType.Code
            */
            assertion = (report.TRT520 == false && (report.TRT521 != null || report.TRT352 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432801", Severity = ProcessMessageSeverity.Error,
                    Description = @"Managed investment trust information not required.",
                    Location = "/tns:TRT/tns:RP/tns:TypeOfTrust/tns:OrganisationDetailsManagedInvestmentTrustI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432801"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT520", Value = GetValueOrEmpty(report.TRT520) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT521", Value = GetValueOrEmpty(report.TRT521) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT352", Value = GetValueOrEmpty(report.TRT352) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432802
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432802()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432802
            If the trust is a Managed Investment Trust (MIT) and the type of trust is either Cash Management Unit Trust (code 039), or Public Unit Trust (listed) - other than a cash management unit trust (code 179), or Public Unit Trust (unlisted) - other than a cash management unit trust (code 184), or Fixed Unit Trust - other than a public unit trust described in '179' or '184' (code 089), or Fixed Trust - other than a fixed unit trust or public unit trust described in '089', '179', or '184' (code 090), then MIT type code must be provided.
    
            Legacy Rule Format:
            ^TRT521 = NULL AND ^TRT520 = TRUE AND InSet(^TRT32, '"039", "179", "184", "089", "090"')

            Technical Business Rule Format:
            ^TRT521 = NULL AND ^TRT520 = TRUE AND InSet(^TRT32, '"039", "179", "184", "089", "090"')
    
            Data Elements:
    
            ^TRT521 = TRT:RP:TypeOfTrust:OrganisationDetails.ManagedInvestmentTrustType.Code
    
            ^TRT32 = TRT:RP:TypeOfTrust:OrganisationDetails.OrganisationType.Code
    
            ^TRT520 = TRT:RP:TypeOfTrust:OrganisationDetails.ManagedInvestmentTrust.Indicator
            */
            assertion = (report.TRT521 == null && report.TRT520 == true && IsMatch(report.TRT32, @"^(039|179|184|089|090)$"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432802", Severity = ProcessMessageSeverity.Error,
                    Description = @"Managed investment trust type code required.",
                    Location = "/tns:TRT/tns:RP/tns:TypeOfTrust/tns:OrganisationDetailsManagedInvestmentTrustTypeC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432802"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT521", Value = GetValueOrEmpty(report.TRT521) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT520", Value = GetValueOrEmpty(report.TRT520) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT32", Value = report.TRT32 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432805
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432805()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432805
            If the trust is a managed investment trust (MIT), has the trustee made an election into capital account treatment?' must be completed if the trust is a Managed Investment Trust (MIT).
    
            Legacy Rule Format:
            ^TRT352 = NULL AND ^TRT520 = TRUE

            Technical Business Rule Format:
            ^TRT352 = NULL AND ^TRT520 = TRUE
    
            Data Elements:
    
            ^TRT352 = TRT:RP:TypeOfTrust:Elections.ManagedInvestmentTrusts.Indicator
    
            ^TRT520 = TRT:RP:TypeOfTrust:OrganisationDetails.ManagedInvestmentTrust.Indicator
            */
            assertion = (report.TRT352 == null && report.TRT520 == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432805", Severity = ProcessMessageSeverity.Error,
                    Description = @"Capital account treatment election required.",
                    Location = "/tns:TRT/tns:RP/tns:TypeOfTrust/tns:ElectionsManagedInvestmentTrustsI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432805"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT352", Value = GetValueOrEmpty(report.TRT352) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT520", Value = GetValueOrEmpty(report.TRT520) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432863
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432863()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432863
            If the type of trust is '232' (testamentary trust), then the assessment calculation code for each Statement of Distribution must be 25, 26, 27, 28, 29, 30, 34, 35, 125, 126, 127, 128, 129, 134, 138, 139 or 140 and the assessment calculation code in the distribution statement where no beneficiary is entitled, if present, must be 36 or 37.
    
            Legacy Rule Format:
            ^TRT32 = '232' AND (AnyOccurrence(^TRT1070, NotInSet(^TRT214, '"25", "26", "27", "28", "29", "30", "34", "35", "125", "126", "127", "128", "129", "134", "138", "139", "140"')) OR (AnyOccurrence(^TRT1070, ^TRT230 <> NULL AND NotInSet(^TRT230, '"36", "37"'))))

            Technical Business Rule Format:
            ^TRT32 = '232' AND (AnyOccurrence(^TRT1070, NotInSet(^TRT214, '"25", "26", "27", "28", "29", "30", "34", "35", "125", "126", "127", "128", "129", "134", "138", "139", "140"')) OR (AnyOccurrence(^TRT1070, ^TRT230 <> NULL AND NotInSet(^TRT230, '"36", "37"'))))
    
            Data Elements:
    
            ^TRT32 = TRT:RP:TypeOfTrust:OrganisationDetails.OrganisationType.Code
    
            ^TRT214 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.AssessmentCalculation.Code
    
            ^TRT230 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.AssessmentCalculation.Code
    
            ^TRT1070 = TRT:RP:StatementOfDistributionToBeneficiary
            */
            assertion = (report.TRT32 == @"232" && ((report.RP_StatementOfDistributionToBeneficiaryCollection == null ? false : report.RP_StatementOfDistributionToBeneficiaryCollection.Any(TRT1070Repeat => !(IsMatch(TRT1070Repeat.TRT214, @"^(25|26|27|28|29|30|34|35|125|126|127|128|129|134|138|139|140)$",RegexOptions.IgnoreCase)))) || (report.RP_StatementOfDistributionToBeneficiaryCollection == null ? false : report.RP_StatementOfDistributionToBeneficiaryCollection.Any(TRT1070Repeat => report.TRT230 != null && !(IsMatch(report.TRT230, @"^(36|37)$",RegexOptions.IgnoreCase))))));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432863", Severity = ProcessMessageSeverity.Error,
                    Description = @"Assessment calculation code(s) must be a testamentary trust code.",
                    Location = "/tns:TRT/tns:RP/tns:TypeOfTrust/tns:OrganisationDetailsOrganisationTypeC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432863"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT32", Value = report.TRT32 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT214", Value = "Count(TRT214)" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT230", Value = "TRT230" });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432888
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432888()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432888
            Subsequent year accelerated depreciation deductions for assets using Backing Business Investment must be greater than 0
    
            Legacy Rule Format:
            ^TRT561 <> NULL AND ^TRT561 = 0

            Technical Business Rule Format:
            ^TRT561 <> NULL AND ^TRT561 = 0
    
            Data Elements:
    
            ^TRT561 = TRT:RP:BusinessAndProfessionalInformation:CapitalAllowances:Expense.SubsequentYearAcceleratedDepreciationDeductionsForAssets.Amount
            */
            assertion = (report.TRT561 != null && report.TRT561 == 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432888", Severity = ProcessMessageSeverity.Error,
                    Description = @"Subsequent year accelerated depreciation deductions for assets using Backing Business Investment must be greater than 0",
                    Location = "/tns:TRT/tns:RP/tns:BusinessAndProfessionalInformation/tns:CapitalAllowances/tns:ExpenseSubsequentYearAcceleratedDepreciationDeductionsForAssetsA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432888"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT561", Value = GetValueOrEmpty(report.TRT561) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432894
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432894()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432894
            When P is selected at aggregated turnover range, Aggregated turnover must be provided
    
            Legacy Rule Format:
            ^TRT564 = NULL AND ^TRT563 = "P"

            Technical Business Rule Format:
            ^TRT564 = NULL AND ^TRT563 = 'P'
    
            Data Elements:
    
            ^TRT564 = TRT:RP:BusinessAndProfessionalInformation:AggregatedTurnover:Income.AggregatedTurnover.Amount
    
            ^TRT563 = TRT:RP:BusinessAndProfessionalInformation:AggregatedTurnover:Income.AggregatedTurnoverRange.Code
            */
            assertion = (report.TRT564 == null && report.TRT563 == @"P");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432912", Severity = ProcessMessageSeverity.Error,
                    Description = @"Aggregated turnover must be provided",
                    LongDescription = @"When P is selected at aggregated turnover range, Aggregated turnover must be provided",
                    Location = "/tns:TRT/tns:RP/tns:BusinessAndProfessionalInformation/tns:AggregatedTurnover/tns:IncomeA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432894"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT564", Value = GetValueOrEmpty(report.TRT564) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT563", Value = report.TRT563 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432896
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432896()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432896
            If the company has self-assessed that they are a Country by Country Reporting Entity (CbCRE), then the Significant Global Entity (SGE) indicator must be true.
    
            Legacy Rule Format:
            ^TRT550 = TRUE AND (^TRT487 = NULL OR ^TRT487 = FALSE)

            Technical Business Rule Format:
            ^TRT550 = TRUE AND (^TRT487 = NULL OR ^TRT487 = FALSE)
    
            Data Elements:
    
            ^TRT550 = TRT:RP:BusinessDetails:InternationalDealings.CountryByCountryReportingEntityStatus.Indicator
    
            ^TRT487 = TRT:RP:BusinessDetails:OrganisationDetails.SignificantGlobalEntityStatus.Indicator
            */
            assertion = (report.TRT550 == true && (report.TRT487 == null || report.TRT487 == false));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432896", Severity = ProcessMessageSeverity.Error,
                    Description = @"A reporting party who indicates that they are a Country-by-country reporting entity must also indicate that they are an SGE",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:InternationalDealingsCountryByCountryReportingEntityStatusI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432896"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT550", Value = GetValueOrEmpty(report.TRT550) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT487", Value = GetValueOrEmpty(report.TRT487) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432920
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432920()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432920
            'Do you own shares that are referable to a sub-fund within the same CCIV as you?' must be completed if the trust is a CCIV sub-fund trust.
    
            Legacy Rule Format:
            ^TRT32 = '233' AND ^TRT566 = NULL

            Technical Business Rule Format:
            ^TRT32 = '233' AND ^TRT566 = NULL
    
            Data Elements:
    
            ^TRT566 = TRT:RP:TypeOfTrust:Assets.Investment.CrossOtherCCIVSubfund.Indicator
    
            ^TRT32 = TRT:RP:TypeOfTrust:OrganisationDetails.OrganisationType.Code
            */
            assertion = (report.TRT32 == @"233" && report.TRT566 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432920", Severity = ProcessMessageSeverity.Error,
                    Description = @"'Do you own shares that are referable to a sub-fund within the same CCIV as you?' must be completed when type of trust is CCIV sub-fund trust.",
                    Location = "/tns:TRT/tns:RP/tns:TypeOfTrust/tns:AssetsInvestmentCrossOtherCCIVSubfundI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432920"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT32", Value = report.TRT32 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT566", Value = GetValueOrEmpty(report.TRT566) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432921
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432921()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432921
            Do you own shares that are referable to a sub-fund within the same CCIV as you?' must be left blank if when the trust is not a CCIV sub-fund trust.
    
            Legacy Rule Format:
            ^TRT32 <> '233' AND ^TRT566 <> NULL

            Technical Business Rule Format:
            ^TRT32 <> '233' AND ^TRT566 <> NULL
    
            Data Elements:
    
            ^TRT566 = TRT:RP:TypeOfTrust:Assets.Investment.CrossOtherCCIVSubfund.Indicator
    
            ^TRT32 = TRT:RP:TypeOfTrust:OrganisationDetails.OrganisationType.Code
            */
            assertion = (report.TRT32 != @"233" && report.TRT566 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432921", Severity = ProcessMessageSeverity.Error,
                    Description = @"'Do you own shares that are referable to a sub-fund within the same CCIV as you?' must not be completed when trust type is not a CCIV sub-fund trust.",
                    Location = "/tns:TRT/tns:RP/tns:TypeOfTrust/tns:AssetsInvestmentCrossOtherCCIVSubfundI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432921"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT32", Value = report.TRT32 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT566", Value = GetValueOrEmpty(report.TRT566) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432924
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432924()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432924
            If there is an amount greater than zero at Small business skills and training boost, there must be an amount at Reconciliation items- Expense reconciliation adjustments- Australian business expenses
    
            Legacy Rule Format:
            ^TRT567 > 0 AND ^TRT447 = NULL

            Technical Business Rule Format:
            ^TRT567 > 0 AND ^TRT447 = NULL
    
            Data Elements:
    
            ^TRT567 = TRT:RP:BusinessAndProfessionalInformation:SmallBusinessBoost:Expense.EmployeeTrainingBonus.Amount
    
            ^TRT447 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:BusinessExpenses:Expense.ReconciliationAdjustmentTotal.Amount
            */
            assertion = (report.TRT567.GetValueOrDefault() > 0 && report.TRT447 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432924", Severity = ProcessMessageSeverity.Error,
                    Description = @"Reconciliation items- Expense reconciliation adjustments- Australian business expenses must be present",
                    LongDescription = @"If there is an amount greater than zero at Small business skills and training boost, there must be an amount at Reconciliation items- Expense reconciliation adjustments- Australian business expenses",
                    Location = "/tns:TRT/tns:RP/tns:BusinessAndProfessionalInformation/tns:SmallBusinessBoost/tns:ExpenseEmployeeTrainingBonusA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432924"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT567", Value = GetValueOrEmpty(report.TRT567) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT447", Value = GetValueOrEmpty(report.TRT447) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432928
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432928()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432928
            ABN must pass the ABN algorithm
    
            Legacy Rule Format:
            ^TRT7 <> NULL and FailsABNAlgorithm(^TRT7)

            Technical Business Rule Format:
            ^TRT7 <> NULL and FailsABNAlgorithm(^TRT7)
    
            Data Elements:
    
            ^TRT7 = TRT:RP:Identifiers.AustralianBusinessNumber.Identifier
            */
            assertion = (report.TRT7 != null && FailsABNAlgorithm(report.TRT7));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000477", Severity = ProcessMessageSeverity.Error,
                    Description = @"ABN is invalid",
                    Location = "/tns:TRT/tns:RP/tns:AustralianBusinessNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432928"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT7", Value = GetValueOrEmpty(report.TRT7) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432929
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432929()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432929
            Tax File Number must pass the TFN algorithm check.
    
            Legacy Rule Format:
            FailsTFNAlgorithm(^TRT5)

            Technical Business Rule Format:
            FailsTFNAlgorithm(^TRT5)
    
            Data Elements:
    
            ^TRT5 = TRT:RP:Identifiers.TaxFileNumber.Identifier
            */
            assertion = FailsTFNAlgorithm(report.TRT5);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.428016", Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax File Number has failed the algorithm check",
                    Location = "/tns:TRT/tns:RP/tns:TaxFileNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432929"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT5", Value = report.TRT5 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432930
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432930()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432930
            If the return is an amendment, sequence number, type and reason must be present
    
            Legacy Rule Format:
            ^TRT414 = TRUE AND (^TRT417 = NULL OR ^TRT415 = NULL OR ^TRT416 = NULL)

            Technical Business Rule Format:
            ^TRT414 = TRUE AND (^TRT417 = NULL OR ^TRT415 = NULL OR ^TRT416 = NULL)
    
            Data Elements:
    
            ^TRT414 = TRT:RP:ReportAmendment:Report.Amendment.Indicator
    
            ^TRT415 = TRT:RP:ReportAmendment:Report.AmendmentType.Code
    
            ^TRT416 = TRT:RP:ReportAmendment:Report.AmendmentReason.Text
    
            ^TRT417 = TRT:RP:ReportAmendment:Report.AmendmentSequence.Number
            */
            assertion = (report.TRT414 == true && (report.TRT417 == null || report.TRT415 == null || report.TRT416 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010028", Severity = ProcessMessageSeverity.Error,
                    Description = @"If the return is an amendment, sequence number, type and reason must be present",
                    LongDescription = @"If the Amendment Indicator is 'yes' (true), then the Amendment Sequence Number, Amendment Type and Amendment Reason must be present and not blank.",
                    Location = "/tns:TRT/tns:RP/tns:ReportAmendment/tns:I",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432930"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT414", Value = GetValueOrEmpty(report.TRT414) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT417", Value = report.TRT417.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT415", Value = GetValueOrEmpty(report.TRT415) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT416", Value = GetValueOrEmpty(report.TRT416) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432931
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432931()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432931
            If the return is an amendment, Amendment Indicator must be 'yes' (true)
    
            Legacy Rule Format:
            ^TRT414 <> TRUE AND (^TRT417 <> NULL OR ^TRT415 <> NULL OR ^TRT416 <> NULL)

            Technical Business Rule Format:
            ^TRT414 <> TRUE AND (^TRT417 <> NULL OR ^TRT415 <> NULL OR ^TRT416 <> NULL)
    
            Data Elements:
    
            ^TRT414 = TRT:RP:ReportAmendment:Report.Amendment.Indicator
    
            ^TRT415 = TRT:RP:ReportAmendment:Report.AmendmentType.Code
    
            ^TRT416 = TRT:RP:ReportAmendment:Report.AmendmentReason.Text
    
            ^TRT417 = TRT:RP:ReportAmendment:Report.AmendmentSequence.Number
            */
            assertion = (report.TRT414 != true && (report.TRT417 != null || report.TRT415 != null || report.TRT416 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010029", Severity = ProcessMessageSeverity.Error,
                    Description = @"If the return is an amendment, Amendment Indicator must be 'yes' (true)",
                    LongDescription = @"If an Amendment Sequence Number, Amendment Type or Amendment Reason is present, then the Amendment Indicator must be 'yes' (true)",
                    Location = "/tns:TRT/tns:RP/tns:ReportAmendment/tns:I",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432931"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT414", Value = GetValueOrEmpty(report.TRT414) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT417", Value = report.TRT417.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT415", Value = GetValueOrEmpty(report.TRT415) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT416", Value = GetValueOrEmpty(report.TRT416) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432932
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432932()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432932
            Non-individual name field cannot contain 'Pship', 'P'ship or 'P/Ship'
    
            Legacy Rule Format:
            StartsWith(^TRT6, 'T/A ') AND EndsWithSet(^TRT6, '" Pship"," P'ship"," P/ship"')

            Technical Business Rule Format:
            StartsWith(^TRT6, 'T/A ') AND EndsWithSet(^TRT6, '" Pship"," P'ship"," P/ship"')
    
            Data Elements:
    
            ^TRT6 = TRT:RP:CurrentTrust:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (IsMatch(report.TRT6, @"^(T/A )",RegexOptions.IgnoreCase) && IsMatch(report.TRT6, @"( Pship| P'ship| P/ship)$",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410038", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-individual name field cannot start with 'T/A' and end with 'Pship', 'P'ship or 'P/Ship'",
                    Location = "/tns:TRT/tns:RP/tns:CurrentTrust/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432932"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT6", Value = report.TRT6 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432933
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432933()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432933
            Organisation name must contain at least one alpha or numeric character
    
            Legacy Rule Format:
            NotContainsSet(^TRT6, '"a-z", "A-Z", "0-9"')

            Technical Business Rule Format:
            NotContainsSet(^TRT6, '"a-z", "A-Z", "0-9"')
    
            Data Elements:
    
            ^TRT6 = TRT:RP:CurrentTrust:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = !(IsMatch(report.TRT6, @"[a-z]|[A-Z]|[0-9]",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410039", Severity = ProcessMessageSeverity.Error,
                    Description = @"Organisation name must contain at least one alpha or numeric character",
                    Location = "/tns:TRT/tns:RP/tns:CurrentTrust/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432933"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT6", Value = report.TRT6 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432934
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432934()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432934
            Non Individual Name cannot contain space hyphen space combinations
    
            Legacy Rule Format:
            Contains(^TRT6, ' - ')

            Technical Business Rule Format:
            Contains(^TRT6, ' - ')
    
            Data Elements:
    
            ^TRT6 = TRT:RP:CurrentTrust:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.TRT6 == null ? false : report.TRT6.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000413", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain space hyphen space combinations",
                    Location = "/tns:TRT/tns:RP/tns:CurrentTrust/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432934"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT6", Value = report.TRT6 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432935
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432935()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432935
            Non Individual Name cannot contain "P/L"
    
            Legacy Rule Format:
            Contains(^TRT6, 'P/L')

            Technical Business Rule Format:
            Contains(^TRT6, 'P/L')
    
            Data Elements:
    
            ^TRT6 = TRT:RP:CurrentTrust:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.TRT6 == null ? false : report.TRT6.ToUpperInvariant().Contains(@"P/L"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000414", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain ""P/L""",
                    Location = "/tns:TRT/tns:RP/tns:CurrentTrust/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432935"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT6", Value = report.TRT6 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432936
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432936()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432936
            Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            ContainsSet(^TRT6, '"--","''","  "')

            Technical Business Rule Format:
            ContainsSet(^TRT6, '"--","''","  "')
    
            Data Elements:
    
            ^TRT6 = TRT:RP:CurrentTrust:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.TRT6, @"--|''|  ",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000416", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:TRT/tns:RP/tns:CurrentTrust/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432936"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT6", Value = report.TRT6 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432937
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432937()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432937
            Non Individual Name cannot end with "T/A", "T/A P'ship", "T/A Pship", "T/A P/Ship" or "T/A Partnership"
    
            Legacy Rule Format:
            EndsWithSet(^TRT6, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"')

            Technical Business Rule Format:
            EndsWithSet(^TRT6, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"')
    
            Data Elements:
    
            ^TRT6 = TRT:RP:CurrentTrust:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.TRT6, @"( T/A| T/A P'ship| T/A Pship| T/A P/Ship| T/A Partnership)$",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000417", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot end with ""T/A"", ""T/A P'ship"", ""T/A Pship"", ""T/A P/Ship"" or ""T/A Partnership""",
                    Location = "/tns:TRT/tns:RP/tns:CurrentTrust/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432937"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT6", Value = report.TRT6 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432938
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432938()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432938
            Non-individual name field cannot contain 'Pship', 'P'ship or 'P/Ship'
    
            Legacy Rule Format:
            StartsWith(^TRT8, 'T/A ') AND EndsWithSet(^TRT8, '" Pship"," P'ship"," P/ship"')

            Technical Business Rule Format:
            StartsWith(^TRT8, 'T/A ') AND EndsWithSet(^TRT8, '" Pship"," P'ship"," P/ship"')
    
            Data Elements:
    
            ^TRT8 = TRT:RP:PreviousTrust:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (IsMatch(report.TRT8, @"^(T/A )",RegexOptions.IgnoreCase) && IsMatch(report.TRT8, @"( Pship| P'ship| P/ship)$",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410038", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-individual name field cannot start with 'T/A' and end with 'Pship', 'P'ship or 'P/Ship'",
                    Location = "/tns:TRT/tns:RP/tns:PreviousTrust/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432938"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT8", Value = GetValueOrEmpty(report.TRT8) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432939
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432939()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432939
            Organisation name must contain at least one alpha or numeric character
    
            Legacy Rule Format:
            ^TRT8  <> NULL AND NotContainsSet(^TRT8, '"a-z", "A-Z", "0-9"')

            Technical Business Rule Format:
            ^TRT8  <> NULL AND NotContainsSet(^TRT8, '"a-z", "A-Z", "0-9"')
    
            Data Elements:
    
            ^TRT8 = TRT:RP:PreviousTrust:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.TRT8 != null && !(IsMatch(report.TRT8, @"[a-z]|[A-Z]|[0-9]",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410039", Severity = ProcessMessageSeverity.Error,
                    Description = @"Organisation name must contain at least one alpha or numeric character",
                    Location = "/tns:TRT/tns:RP/tns:PreviousTrust/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432939"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT8", Value = GetValueOrEmpty(report.TRT8) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432940
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432940()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432940
            Non Individual Name cannot contain space hyphen space combinations
    
            Legacy Rule Format:
            Contains(^TRT8, ' - ')

            Technical Business Rule Format:
            Contains(^TRT8, ' - ')
    
            Data Elements:
    
            ^TRT8 = TRT:RP:PreviousTrust:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.TRT8 == null ? false : report.TRT8.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000413", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain space hyphen space combinations",
                    Location = "/tns:TRT/tns:RP/tns:PreviousTrust/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432940"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT8", Value = GetValueOrEmpty(report.TRT8) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432941
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432941()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432941
            Non Individual Name cannot contain "P/L"
    
            Legacy Rule Format:
            Contains(^TRT8, 'P/L')

            Technical Business Rule Format:
            Contains(^TRT8, 'P/L')
    
            Data Elements:
    
            ^TRT8 = TRT:RP:PreviousTrust:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.TRT8 == null ? false : report.TRT8.ToUpperInvariant().Contains(@"P/L"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000414", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain ""P/L""",
                    Location = "/tns:TRT/tns:RP/tns:PreviousTrust/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432941"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT8", Value = GetValueOrEmpty(report.TRT8) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432942
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432942()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432942
            Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            ContainsSet(^TRT8, '"--","''","  "')

            Technical Business Rule Format:
            ContainsSet(^TRT8, '"--","''","  "')
    
            Data Elements:
    
            ^TRT8 = TRT:RP:PreviousTrust:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.TRT8, @"--|''|  ",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000416", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:TRT/tns:RP/tns:PreviousTrust/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432942"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT8", Value = GetValueOrEmpty(report.TRT8) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432943
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432943()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432943
            Non Individual Name cannot end with "T/A", "T/A P'ship", "T/A Pship", "T/A P/Ship" or "T/A Partnership"
    
            Legacy Rule Format:
            EndsWithSet(^TRT8, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"')

            Technical Business Rule Format:
            EndsWithSet(^TRT8, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"')
    
            Data Elements:
    
            ^TRT8 = TRT:RP:PreviousTrust:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.TRT8, @"( T/A| T/A P'ship| T/A Pship| T/A P/Ship| T/A Partnership)$",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000417", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot end with ""T/A"", ""T/A P'ship"", ""T/A Pship"", ""T/A P/Ship"" or ""T/A Partnership""",
                    Location = "/tns:TRT/tns:RP/tns:PreviousTrust/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432943"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT8", Value = GetValueOrEmpty(report.TRT8) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432944
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432944()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432944
            State and post code must be present for an Australian address
    
            Legacy Rule Format:
            ^TRT315 = FALSE AND (^TRT11 = NULL OR ^TRT12 = NULL)

            Technical Business Rule Format:
            ^TRT315 = FALSE AND (^TRT11 = NULL OR ^TRT12 = NULL)
    
            Data Elements:
    
            ^TRT315 = TRT:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.OverseasAddress.Indicator
    
            ^TRT11 = TRT:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.StateOrTerritory.Code
    
            ^TRT12 = TRT:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.Postcode.Text
            */
            assertion = (report.TRT315 == false && (report.TRT11 == null || report.TRT12 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.434147", Severity = ProcessMessageSeverity.Error,
                    Description = @"State and post code must be present for an Australian address",
                    LongDescription = @"If the overseas indicator is no (false), then an Australian state code and post code must be present",
                    Location = "/tns:TRT/tns:RP/tns:CurrentTrust/tns:CurrentPostalAddressDetails/tns:OverseasAddressI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432944"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT315", Value = GetValueOrEmpty(report.TRT315) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT11", Value = GetValueOrEmpty(report.TRT11) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT12", Value = GetValueOrEmpty(report.TRT12) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432945
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432945()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432945
            Address line 1 contains incorrect 'care of' reference. The only acceptable 'care of' reference is C/-
    
            Legacy Rule Format:
            StartsWithSet(^TRT9,'"C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^TRT9, '" C/O "," C/ "," Care Of "')

            Technical Business Rule Format:
            StartsWithSet(^TRT9,'"C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^TRT9, '" C/O "," C/ "," Care Of "')
    
            Data Elements:
    
            ^TRT9 = TRT:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.Line1.Text
            */
            assertion = (IsMatch(report.TRT9, @"^(C/O |C/ |Care Of |CO )",RegexOptions.IgnoreCase) || IsMatch(report.TRT9, @" C/O | C/ | Care Of ",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000406", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 contains incorrect 'care of' reference. The only acceptable 'care of' reference is C/-",
                    Location = "/tns:TRT/tns:RP/tns:CurrentTrust/tns:CurrentPostalAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432945"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT9", Value = report.TRT9 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432946
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432946()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432946
            Address Line 2 must be present if Address Line 1 contains 'C/-'
    
            Legacy Rule Format:
            Contains(^TRT9, 'C/-') AND ^TRT306 = NULL

            Technical Business Rule Format:
            Contains(^TRT9, 'C/-') AND ^TRT306 = NULL
    
            Data Elements:
    
            ^TRT9 = TRT:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.Line1.Text
    
            ^TRT306 = TRT:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.Line2.Text
            */
            assertion = ((report.TRT9 == null ? false : report.TRT9.ToUpperInvariant().Contains(@"C/-")) && report.TRT306 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410002", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 2 must be present if Address Line 1 contains 'C/-'",
                    Location = "/tns:TRT/tns:RP/tns:CurrentTrust/tns:CurrentPostalAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432946"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT9", Value = report.TRT9 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT306", Value = GetValueOrEmpty(report.TRT306) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432947
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432947()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432947
            Address Line 1 must not contain 'AS ABOVE'
    
            Legacy Rule Format:
            FoundSet(^TRT9, '"AS ABOVE"')

            Technical Business Rule Format:
            FoundSet(^TRT9, '"AS ABOVE"')
    
            Data Elements:
    
            ^TRT9 = TRT:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.Line1.Text
            */
            assertion = IsMatch(report.TRT9, @"(^(AS ABOVE)$)|(^(.*)( AS ABOVE )(.*)$)|(^(AS ABOVE )(.*)$)|(^(.*)( AS ABOVE)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410001", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 1 must not contain 'AS ABOVE'",
                    Location = "/tns:TRT/tns:RP/tns:CurrentTrust/tns:CurrentPostalAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432947"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT9", Value = report.TRT9 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432948
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432948()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432948
            Address line 1 cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            Contains(^TRT9, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^TRT9, 'UNKNOWN')
    
            Data Elements:
    
            ^TRT9 = TRT:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.Line1.Text
            */
            assertion = (report.TRT9 == null ? false : report.TRT9.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000405", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 cannot contain ""UNKNOWN""",
                    Location = "/tns:TRT/tns:RP/tns:CurrentTrust/tns:CurrentPostalAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432948"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT9", Value = report.TRT9 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432949
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432949()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432949
            Care of' is not valid for address line 2
    
            Legacy Rule Format:
            StartsWithSet(^TRT306,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^TRT306, '" C/- "," C/O "," C/ "," Care Of "')

            Technical Business Rule Format:
            StartsWithSet(^TRT306,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^TRT306, '" C/- "," C/O "," C/ "," Care Of "')
    
            Data Elements:
    
            ^TRT306 = TRT:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.Line2.Text
            */
            assertion = (IsMatch(report.TRT306, @"^(C/- |C/O |C/ |Care Of |CO )",RegexOptions.IgnoreCase) || IsMatch(report.TRT306, @" C/- | C/O | C/ | Care Of ",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000408", Severity = ProcessMessageSeverity.Error,
                    Description = @"Care of' is not valid for address line 2",
                    Location = "/tns:TRT/tns:RP/tns:CurrentTrust/tns:CurrentPostalAddressDetails/tns:Line2T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432949"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT306", Value = GetValueOrEmpty(report.TRT306) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432950
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432950()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432950
            Address line 2 cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            Contains(^TRT306, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^TRT306, 'UNKNOWN')
    
            Data Elements:
    
            ^TRT306 = TRT:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.Line2.Text
            */
            assertion = (report.TRT306 == null ? false : report.TRT306.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000407", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 2 cannot contain ""UNKNOWN""",
                    Location = "/tns:TRT/tns:RP/tns:CurrentTrust/tns:CurrentPostalAddressDetails/tns:Line2T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432950"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT306", Value = GetValueOrEmpty(report.TRT306) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432951
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432951()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432951
            Suburb/Town/City cannot contain state code
    
            Legacy Rule Format:
            FoundSet(^TRT10, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"')

            Technical Business Rule Format:
            FoundSet(^TRT10, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"')
    
            Data Elements:
    
            ^TRT10 = TRT:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.LocalityName.Text
            */
            assertion = IsMatch(report.TRT10, @"(^(QLD|NSW|VIC|SA|WA|NT|ACT|TAS)$)|(^(.*)( QLD | NSW | VIC | SA | WA | NT | ACT | TAS )(.*)$)|(^(QLD |NSW |VIC |SA |WA |NT |ACT |TAS )(.*)$)|(^(.*)( QLD| NSW| VIC| SA| WA| NT| ACT| TAS)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000411", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain state code",
                    Location = "/tns:TRT/tns:RP/tns:CurrentTrust/tns:CurrentPostalAddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432951"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT10", Value = report.TRT10 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432952
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432952()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432952
            Suburb/Town/City cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            Contains(^TRT10, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^TRT10, 'UNKNOWN')
    
            Data Elements:
    
            ^TRT10 = TRT:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.LocalityName.Text
            */
            assertion = (report.TRT10 == null ? false : report.TRT10.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000409", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain ""UNKNOWN""",
                    Location = "/tns:TRT/tns:RP/tns:CurrentTrust/tns:CurrentPostalAddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432952"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT10", Value = report.TRT10 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432953
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432953()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432953
            Suburb/Town/City cannot contain numeric characters
    
            Legacy Rule Format:
            ^TRT315 = FALSE AND ContainsSet(^TRT10, '0-9')

            Technical Business Rule Format:
            ^TRT315 = FALSE AND ContainsSet(^TRT10, '0-9')
    
            Data Elements:
    
            ^TRT10 = TRT:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.LocalityName.Text
    
            ^TRT315 = TRT:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.OverseasAddress.Indicator
            */
            assertion = (report.TRT315 == false && IsMatch(report.TRT10, @"[0-9]",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000410", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain numeric characters",
                    Location = "/tns:TRT/tns:RP/tns:CurrentTrust/tns:CurrentPostalAddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432953"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT315", Value = GetValueOrEmpty(report.TRT315) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT10", Value = report.TRT10 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432954
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432954()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432954
            Country code must be present for overseas address
    
            Legacy Rule Format:
            ^TRT315 = TRUE AND ^TRT367 = NULL

            Technical Business Rule Format:
            ^TRT315 = TRUE AND ^TRT367 = NULL
    
            Data Elements:
    
            ^TRT367 = TRT:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.Country.Code
    
            ^TRT315 = TRT:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.OverseasAddress.Indicator
            */
            assertion = (report.TRT315 == true && report.TRT367 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410191", Severity = ProcessMessageSeverity.Error,
                    Description = @"Country code must be present for overseas address",
                    LongDescription = @"When the overseas address indicator is 'TRUE', the corresponding Country Code must be supplied",
                    Location = "/tns:TRT/tns:RP/tns:CurrentTrust/tns:CurrentPostalAddressDetails/tns:CountryC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432954"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT315", Value = GetValueOrEmpty(report.TRT315) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT367", Value = GetValueOrEmpty(report.TRT367) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432955
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432955()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432955
            Overseas address indicator must not be 'TRUE' if Country Code is 'au'
    
            Legacy Rule Format:
            ^TRT367 = 'au' AND ^TRT315 = TRUE

            Technical Business Rule Format:
            ^TRT367 = 'au' AND ^TRT315 = TRUE
    
            Data Elements:
    
            ^TRT367 = TRT:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.Country.Code
    
            ^TRT315 = TRT:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.OverseasAddress.Indicator
            */
            assertion = (report.TRT367 == @"au" && report.TRT315 == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410211", Severity = ProcessMessageSeverity.Error,
                    Description = @"Overseas address indicator must not be 'TRUE' if Country Code is 'au'",
                    LongDescription = @"The overseas address indicator must be 'FALSE' if the Country Code is 'au' - representing Australia. If the address is an overseas address please supply the correct Country Code",
                    Location = "/tns:TRT/tns:RP/tns:CurrentTrust/tns:CurrentPostalAddressDetails/tns:CountryC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432955"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT367", Value = GetValueOrEmpty(report.TRT367) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT315", Value = GetValueOrEmpty(report.TRT315) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432956
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432956()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432956
            If an overseas Country Code is used, the Overseas Address indicator must be 'TRUE'
    
            Legacy Rule Format:
            ^TRT367 <> NULL AND ^TRT367 <> 'au' AND ^TRT315 = FALSE

            Technical Business Rule Format:
            ^TRT367 <> NULL AND ^TRT367 <> 'au' AND ^TRT315 = FALSE
    
            Data Elements:
    
            ^TRT367 = TRT:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.Country.Code
    
            ^TRT315 = TRT:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.OverseasAddress.Indicator
            */
            assertion = (report.TRT367 != null && report.TRT367 != @"au" && report.TRT315 == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410212", Severity = ProcessMessageSeverity.Error,
                    Description = @"If an overseas Country Code is used, the Overseas Address indicator must be 'TRUE'",
                    LongDescription = @"If the address is an Australian address, the overseas address indicator must be set to 'FALSE' and you do not need to supply a Country Code. If the address is an overseas address, the overseas address indicator must be set to 'TRUE' and you must supply a Country Code other than 'au'",
                    Location = "/tns:TRT/tns:RP/tns:CurrentTrust/tns:CurrentPostalAddressDetails/tns:CountryC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432956"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT367", Value = GetValueOrEmpty(report.TRT367) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT315", Value = GetValueOrEmpty(report.TRT315) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432958
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432958()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432958
            State and post code must be present for an Australian address
    
            Legacy Rule Format:
            ^TRT317 = FALSE AND (^TRT16 = NULL OR ^TRT17 = NULL)

            Technical Business Rule Format:
            ^TRT317 = FALSE AND (^TRT16 = NULL OR ^TRT17 = NULL)
    
            Data Elements:
    
            ^TRT317 = TRT:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.OverseasAddress.Indicator
    
            ^TRT16 = TRT:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.StateOrTerritory.Code
    
            ^TRT17 = TRT:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.Postcode.Text
            */
            assertion = (report.TRT317 == false && (report.TRT16 == null || report.TRT17 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.434147", Severity = ProcessMessageSeverity.Error,
                    Description = @"State and post code must be present for an Australian address",
                    LongDescription = @"If the overseas indicator is no (false), then an Australian state code and post code must be present",
                    Location = "/tns:TRT/tns:RP/tns:PreviousTrust/tns:PreviousPostalAddressDetails/tns:OverseasAddressI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432958"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT317", Value = GetValueOrEmpty(report.TRT317) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT16", Value = GetValueOrEmpty(report.TRT16) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT17", Value = GetValueOrEmpty(report.TRT17) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432959
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432959()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432959
            Address line 1 contains incorrect 'care of' reference. The only acceptable 'care of' reference is C/-
    
            Legacy Rule Format:
            StartsWithSet(^TRT14,'"C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^TRT14, '" C/O "," C/ "," Care Of "')

            Technical Business Rule Format:
            StartsWithSet(^TRT14,'"C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^TRT14, '" C/O "," C/ "," Care Of "')
    
            Data Elements:
    
            ^TRT14 = TRT:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.Line1.Text
            */
            assertion = (IsMatch(report.TRT14, @"^(C/O |C/ |Care Of |CO )",RegexOptions.IgnoreCase) || IsMatch(report.TRT14, @" C/O | C/ | Care Of ",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000406", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 contains incorrect 'care of' reference. The only acceptable 'care of' reference is C/-",
                    Location = "/tns:TRT/tns:RP/tns:PreviousTrust/tns:PreviousPostalAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432959"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT14", Value = report.TRT14 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432960
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432960()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432960
            Address Line 2 must be present if Address Line 1 contains 'C/-'
    
            Legacy Rule Format:
            Contains(^TRT14, 'C/-') AND ^TRT312 = NULL

            Technical Business Rule Format:
            Contains(^TRT14, 'C/-') AND ^TRT312 = NULL
    
            Data Elements:
    
            ^TRT14 = TRT:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.Line1.Text
    
            ^TRT312 = TRT:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.Line2.Text
            */
            assertion = ((report.TRT14 == null ? false : report.TRT14.ToUpperInvariant().Contains(@"C/-")) && report.TRT312 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410002", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 2 must be present if Address Line 1 contains 'C/-'",
                    Location = "/tns:TRT/tns:RP/tns:PreviousTrust/tns:PreviousPostalAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432960"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT14", Value = report.TRT14 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT312", Value = GetValueOrEmpty(report.TRT312) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432961
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432961()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432961
            Address Line 1 must not contain 'AS ABOVE'
    
            Legacy Rule Format:
            FoundSet(^TRT14, '"AS ABOVE"')

            Technical Business Rule Format:
            FoundSet(^TRT14, '"AS ABOVE"')
    
            Data Elements:
    
            ^TRT14 = TRT:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.Line1.Text
            */
            assertion = IsMatch(report.TRT14, @"(^(AS ABOVE)$)|(^(.*)( AS ABOVE )(.*)$)|(^(AS ABOVE )(.*)$)|(^(.*)( AS ABOVE)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410001", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 1 must not contain 'AS ABOVE'",
                    Location = "/tns:TRT/tns:RP/tns:PreviousTrust/tns:PreviousPostalAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432961"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT14", Value = report.TRT14 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432962
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432962()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432962
            Address line 1 cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            Contains(^TRT14, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^TRT14, 'UNKNOWN')
    
            Data Elements:
    
            ^TRT14 = TRT:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.Line1.Text
            */
            assertion = (report.TRT14 == null ? false : report.TRT14.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000405", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 cannot contain ""UNKNOWN""",
                    Location = "/tns:TRT/tns:RP/tns:PreviousTrust/tns:PreviousPostalAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432962"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT14", Value = report.TRT14 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432963
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432963()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432963
            Care of' is not valid for address line 2
    
            Legacy Rule Format:
            StartsWithSet(^TRT312,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^TRT312, '" C/- "," C/O "," C/ "," Care Of "')

            Technical Business Rule Format:
            StartsWithSet(^TRT312,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^TRT312, '" C/- "," C/O "," C/ "," Care Of "')
    
            Data Elements:
    
            ^TRT312 = TRT:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.Line2.Text
            */
            assertion = (IsMatch(report.TRT312, @"^(C/- |C/O |C/ |Care Of |CO )",RegexOptions.IgnoreCase) || IsMatch(report.TRT312, @" C/- | C/O | C/ | Care Of ",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000408", Severity = ProcessMessageSeverity.Error,
                    Description = @"Care of' is not valid for address line 2",
                    Location = "/tns:TRT/tns:RP/tns:PreviousTrust/tns:PreviousPostalAddressDetails/tns:Line2T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432963"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT312", Value = GetValueOrEmpty(report.TRT312) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432964
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432964()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432964
            Address line 2 cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            Contains(^TRT312, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^TRT312, 'UNKNOWN')
    
            Data Elements:
    
            ^TRT312 = TRT:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.Line2.Text
            */
            assertion = (report.TRT312 == null ? false : report.TRT312.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000407", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 2 cannot contain ""UNKNOWN""",
                    Location = "/tns:TRT/tns:RP/tns:PreviousTrust/tns:PreviousPostalAddressDetails/tns:Line2T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432964"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT312", Value = GetValueOrEmpty(report.TRT312) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432965
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432965()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432965
            Suburb/Town/City cannot contain state code
    
            Legacy Rule Format:
            FoundSet(^TRT15, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"')

            Technical Business Rule Format:
            FoundSet(^TRT15, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"')
    
            Data Elements:
    
            ^TRT15 = TRT:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.LocalityName.Text
            */
            assertion = IsMatch(report.TRT15, @"(^(QLD|NSW|VIC|SA|WA|NT|ACT|TAS)$)|(^(.*)( QLD | NSW | VIC | SA | WA | NT | ACT | TAS )(.*)$)|(^(QLD |NSW |VIC |SA |WA |NT |ACT |TAS )(.*)$)|(^(.*)( QLD| NSW| VIC| SA| WA| NT| ACT| TAS)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000411", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain state code",
                    Location = "/tns:TRT/tns:RP/tns:PreviousTrust/tns:PreviousPostalAddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432965"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT15", Value = report.TRT15 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432966
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432966()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432966
            Suburb/Town/City cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            Contains(^TRT15, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^TRT15, 'UNKNOWN')
    
            Data Elements:
    
            ^TRT15 = TRT:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.LocalityName.Text
            */
            assertion = (report.TRT15 == null ? false : report.TRT15.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000409", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain ""UNKNOWN""",
                    Location = "/tns:TRT/tns:RP/tns:PreviousTrust/tns:PreviousPostalAddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432966"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT15", Value = report.TRT15 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432967
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432967()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432967
            Suburb/Town/City cannot contain numeric characters
    
            Legacy Rule Format:
            ^TRT317 = FALSE AND ContainsSet(^TRT15, '0-9')

            Technical Business Rule Format:
            ^TRT317 = FALSE AND ContainsSet(^TRT15, '0-9')
    
            Data Elements:
    
            ^TRT15 = TRT:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.LocalityName.Text
    
            ^TRT317 = TRT:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.OverseasAddress.Indicator
            */
            assertion = (report.TRT317 == false && IsMatch(report.TRT15, @"[0-9]",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000410", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain numeric characters",
                    Location = "/tns:TRT/tns:RP/tns:PreviousTrust/tns:PreviousPostalAddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432967"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT317", Value = GetValueOrEmpty(report.TRT317) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT15", Value = report.TRT15 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432969
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432969()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432969
            Country code must be present for overseas address
    
            Legacy Rule Format:
            ^TRT317 = TRUE AND ^TRT368 = NULL

            Technical Business Rule Format:
            ^TRT317 = TRUE AND ^TRT368 = NULL
    
            Data Elements:
    
            ^TRT368 = TRT:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.Country.Code
    
            ^TRT317 = TRT:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.OverseasAddress.Indicator
            */
            assertion = (report.TRT317 == true && report.TRT368 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410191", Severity = ProcessMessageSeverity.Error,
                    Description = @"Country code must be present for overseas address",
                    LongDescription = @"When the overseas address indicator is 'TRUE', the corresponding Country Code must be supplied",
                    Location = "/tns:TRT/tns:RP/tns:PreviousTrust/tns:PreviousPostalAddressDetails/tns:CountryC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432969"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT317", Value = GetValueOrEmpty(report.TRT317) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT368", Value = GetValueOrEmpty(report.TRT368) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432970
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432970()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432970
            Overseas address indicator must not be 'TRUE' if Country Code is 'au'
    
            Legacy Rule Format:
            ^TRT368 = 'au' AND ^TRT317 = TRUE

            Technical Business Rule Format:
            ^TRT368 = 'au' AND ^TRT317 = TRUE
    
            Data Elements:
    
            ^TRT368 = TRT:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.Country.Code
    
            ^TRT317 = TRT:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.OverseasAddress.Indicator
            */
            assertion = (report.TRT368 == @"au" && report.TRT317 == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410211", Severity = ProcessMessageSeverity.Error,
                    Description = @"Overseas address indicator must not be 'TRUE' if Country Code is 'au'",
                    LongDescription = @"The overseas address indicator must be 'FALSE' if the Country Code is 'au' - representing Australia. If the address is an overseas address please supply the correct Country Code",
                    Location = "/tns:TRT/tns:RP/tns:PreviousTrust/tns:PreviousPostalAddressDetails/tns:CountryC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432970"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT368", Value = GetValueOrEmpty(report.TRT368) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT317", Value = GetValueOrEmpty(report.TRT317) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432971
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432971()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432971
            If an overseas Country Code is used, the Overseas Address indicator must be 'TRUE'
    
            Legacy Rule Format:
            ^TRT368 <> NULL AND ^TRT368 <> 'au' AND ^TRT317 = FALSE

            Technical Business Rule Format:
            ^TRT368 <> NULL AND ^TRT368 <> 'au' AND ^TRT317 = FALSE
    
            Data Elements:
    
            ^TRT368 = TRT:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.Country.Code
    
            ^TRT317 = TRT:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.OverseasAddress.Indicator
            */
            assertion = (report.TRT368 != null && report.TRT368 != @"au" && report.TRT317 == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410212", Severity = ProcessMessageSeverity.Error,
                    Description = @"If an overseas Country Code is used, the Overseas Address indicator must be 'TRUE'",
                    LongDescription = @"If the address is an Australian address, the overseas address indicator must be set to 'FALSE' and you do not need to supply a Country Code. If the address is an overseas address, the overseas address indicator must be set to 'TRUE' and you must supply a Country Code other than 'au'",
                    Location = "/tns:TRT/tns:RP/tns:PreviousTrust/tns:PreviousPostalAddressDetails/tns:CountryC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432971"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT368", Value = GetValueOrEmpty(report.TRT368) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT317", Value = GetValueOrEmpty(report.TRT317) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432972
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432972()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432972
            Family name cannot contain "Exec for", "Rep for" or "Trustee for"
    
            Legacy Rule Format:
            FoundSet(^TRT20, '"Exec for","Rep for","Trustee for"')

            Technical Business Rule Format:
            FoundSet(^TRT20, '"Exec for","Rep for","Trustee for"')
    
            Data Elements:
    
            ^TRT20 = TRT:RP:MailRecipient:PersonNameDetails:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.TRT20, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000424", Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/tns:TRT/tns:RP/tns:MailRecipient/tns:PersonNameDetails/tns:FamilyNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432972"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT20", Value = report.TRT20 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432973
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432973()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432973
            Family name cannot contain "MR", "MRS", "MISS" or "MS"
    
            Legacy Rule Format:
            FoundSet(^TRT20, '"MR","MRS","MISS","MS"')

            Technical Business Rule Format:
            FoundSet(^TRT20, '"MR","MRS","MISS","MS"')
    
            Data Elements:
    
            ^TRT20 = TRT:RP:MailRecipient:PersonNameDetails:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.TRT20, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000426", Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/tns:TRT/tns:RP/tns:MailRecipient/tns:PersonNameDetails/tns:FamilyNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432973"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT20", Value = report.TRT20 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432974
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432974()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432974
            Family Name cannot contain "space hyphen space"
    
            Legacy Rule Format:
            Contains(^TRT20, ' - ')

            Technical Business Rule Format:
            Contains(^TRT20, ' - ')
    
            Data Elements:
    
            ^TRT20 = TRT:RP:MailRecipient:PersonNameDetails:PersonNameDetails.FamilyName.Text
            */
            assertion = (report.TRT20 == null ? false : report.TRT20.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000423", Severity = ProcessMessageSeverity.Error,
                    Description = @"Family Name cannot contain ""space hyphen space""",
                    Location = "/tns:TRT/tns:RP/tns:MailRecipient/tns:PersonNameDetails/tns:FamilyNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432974"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT20", Value = report.TRT20 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432975
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432975()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432975
            Family name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            ContainsSet(^TRT20, '"--","''","  "')

            Technical Business Rule Format:
            ContainsSet(^TRT20, '"--","''","  "')
    
            Data Elements:
    
            ^TRT20 = TRT:RP:MailRecipient:PersonNameDetails:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.TRT20, @"--|''|  ",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000427", Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:TRT/tns:RP/tns:MailRecipient/tns:PersonNameDetails/tns:FamilyNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432975"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT20", Value = report.TRT20 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432976
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432976()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432976
            Family name must contain at least one alphabetical character
    
            Legacy Rule Format:
            ^TRT20 <> NULL AND NotContainsSet(^TRT20, '"A-Z","a-z"')

            Technical Business Rule Format:
            ^TRT20 <> NULL AND NotContainsSet(^TRT20, '"A-Z","a-z"')
    
            Data Elements:
    
            ^TRT20 = TRT:RP:MailRecipient:PersonNameDetails:PersonNameDetails.FamilyName.Text
            */
            assertion = (report.TRT20 != null && !(IsMatch(report.TRT20, @"[A-Z]|[a-z]",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500142", Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name must contain at least one alphabetical character.",
                    Location = "/tns:TRT/tns:RP/tns:MailRecipient/tns:PersonNameDetails/tns:FamilyNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432976"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT20", Value = report.TRT20 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432977
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432977()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432977
            Family name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
            Legacy Rule Format:
            FoundSet(^TRT20, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')

            Technical Business Rule Format:
            FoundSet(^TRT20, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')
    
            Data Elements:
    
            ^TRT20 = TRT:RP:MailRecipient:PersonNameDetails:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.TRT20, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500153", Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/tns:TRT/tns:RP/tns:MailRecipient/tns:PersonNameDetails/tns:FamilyNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432977"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT20", Value = report.TRT20 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432979
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432979()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432979
            Given Name must contain at least one alphabetical character
    
            Legacy Rule Format:
            ^TRT22 <> NULL AND NotContainsSet(^TRT22, '"A-Z","a-z"')

            Technical Business Rule Format:
            ^TRT22 <> NULL AND NotContainsSet(^TRT22, '"A-Z","a-z"')
    
            Data Elements:
    
            ^TRT22 = TRT:RP:MailRecipient:PersonNameDetails:PersonNameDetails.GivenName.Text
            */
            assertion = (report.TRT22 != null && !(IsMatch(report.TRT22, @"[A-Z]|[a-z]",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410063", Severity = ProcessMessageSeverity.Error,
                    Description = @"Given Name must contain at least one alphabetical character",
                    Location = "/tns:TRT/tns:RP/tns:MailRecipient/tns:PersonNameDetails/tns:GivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432979"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT22", Value = GetValueOrEmpty(report.TRT22) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432980
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432980()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432980
            First name cannot contain "Exec for", "Rep for" or "Trustee for"
    
            Legacy Rule Format:
            FoundSet(^TRT22, '"Exec for","Rep for","Trustee for"')

            Technical Business Rule Format:
            FoundSet(^TRT22, '"Exec for","Rep for","Trustee for"')
    
            Data Elements:
    
            ^TRT22 = TRT:RP:MailRecipient:PersonNameDetails:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.TRT22, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000437", Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/tns:TRT/tns:RP/tns:MailRecipient/tns:PersonNameDetails/tns:GivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432980"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT22", Value = GetValueOrEmpty(report.TRT22) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432981
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432981()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432981
            First name cannot contain "MR", "MRS", "MISS" or "MS"
    
            Legacy Rule Format:
            FoundSet(^TRT22, '"MR","MRS","MISS","MS"')

            Technical Business Rule Format:
            FoundSet(^TRT22, '"MR","MRS","MISS","MS"')
    
            Data Elements:
    
            ^TRT22 = TRT:RP:MailRecipient:PersonNameDetails:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.TRT22, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000438", Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/tns:TRT/tns:RP/tns:MailRecipient/tns:PersonNameDetails/tns:GivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432981"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT22", Value = GetValueOrEmpty(report.TRT22) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432982
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432982()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432982
            FirstName must be entered if OtherGivenNames is entered
    
            Legacy Rule Format:
            ^TRT23 <> NULL AND ^TRT22 = NULL

            Technical Business Rule Format:
            ^TRT23 <> NULL AND ^TRT22 = NULL
    
            Data Elements:
    
            ^TRT22 = TRT:RP:MailRecipient:PersonNameDetails:PersonNameDetails.GivenName.Text
    
            ^TRT23 = TRT:RP:MailRecipient:PersonNameDetails:PersonNameDetails.OtherGivenName.Text
            */
            assertion = (report.TRT23 != null && report.TRT22 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410218", Severity = ProcessMessageSeverity.Error,
                    Description = @"FirstName must be entered if OtherGivenNames is entered",
                    Location = "/tns:TRT/tns:RP/tns:MailRecipient/tns:PersonNameDetails/tns:GivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432982"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT23", Value = GetValueOrEmpty(report.TRT23) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT22", Value = GetValueOrEmpty(report.TRT22) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432983
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432983()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432983
            First name cannot contain space hyphen space
    
            Legacy Rule Format:
            Contains(^TRT22, ' - ')

            Technical Business Rule Format:
            Contains(^TRT22, ' - ')
    
            Data Elements:
    
            ^TRT22 = TRT:RP:MailRecipient:PersonNameDetails:PersonNameDetails.GivenName.Text
            */
            assertion = (report.TRT22 == null ? false : report.TRT22.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000434", Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain space hyphen space",
                    Location = "/tns:TRT/tns:RP/tns:MailRecipient/tns:PersonNameDetails/tns:GivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432983"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT22", Value = GetValueOrEmpty(report.TRT22) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432984
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432984()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432984
            First name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            ContainsSet(^TRT22, '"--","''","  "')

            Technical Business Rule Format:
            ContainsSet(^TRT22, '"--","''","  "')
    
            Data Elements:
    
            ^TRT22 = TRT:RP:MailRecipient:PersonNameDetails:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.TRT22, @"--|''|  ",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000439", Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:TRT/tns:RP/tns:MailRecipient/tns:PersonNameDetails/tns:GivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432984"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT22", Value = GetValueOrEmpty(report.TRT22) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432985
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432985()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432985
            First name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
            Legacy Rule Format:
            FoundSet(^TRT22, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')

            Technical Business Rule Format:
            FoundSet(^TRT22, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')
    
            Data Elements:
    
            ^TRT22 = TRT:RP:MailRecipient:PersonNameDetails:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.TRT22, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010436", Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/tns:TRT/tns:RP/tns:MailRecipient/tns:PersonNameDetails/tns:GivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432985"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT22", Value = GetValueOrEmpty(report.TRT22) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432986
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432986()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432986
            Other Given Name must contain at least one alphabetical character
    
            Legacy Rule Format:
            ^TRT23 <> NULL AND NotContainsSet(^TRT23, '"A-Z","a-z"')

            Technical Business Rule Format:
            ^TRT23 <> NULL AND NotContainsSet(^TRT23, '"A-Z","a-z"')
    
            Data Elements:
    
            ^TRT23 = TRT:RP:MailRecipient:PersonNameDetails:PersonNameDetails.OtherGivenName.Text
            */
            assertion = (report.TRT23 != null && !(IsMatch(report.TRT23, @"[A-Z]|[a-z]",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410131", Severity = ProcessMessageSeverity.Error,
                    Description = @"Other Given Name must contain at least one alphabetical character",
                    Location = "/tns:TRT/tns:RP/tns:MailRecipient/tns:PersonNameDetails/tns:OtherGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432986"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT23", Value = GetValueOrEmpty(report.TRT23) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432987
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432987()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432987
            Other given name cannot contain "Exec for", "Rep for" or "Trustee for"
    
            Legacy Rule Format:
            FoundSet(^TRT23, '"Exec for","Rep for","Trustee for"')

            Technical Business Rule Format:
            FoundSet(^TRT23, '"Exec for","Rep for","Trustee for"')
    
            Data Elements:
    
            ^TRT23 = TRT:RP:MailRecipient:PersonNameDetails:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.TRT23, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000449", Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/tns:TRT/tns:RP/tns:MailRecipient/tns:PersonNameDetails/tns:OtherGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432987"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT23", Value = GetValueOrEmpty(report.TRT23) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432988
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432988()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432988
            Other given name cannot contain "MR", "MRS", "MISS" or "MS"
    
            Legacy Rule Format:
            FoundSet(^TRT23, '"MR","MRS","MISS","MS"')

            Technical Business Rule Format:
            FoundSet(^TRT23, '"MR","MRS","MISS","MS"')
    
            Data Elements:
    
            ^TRT23 = TRT:RP:MailRecipient:PersonNameDetails:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.TRT23, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000450", Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/tns:TRT/tns:RP/tns:MailRecipient/tns:PersonNameDetails/tns:OtherGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432988"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT23", Value = GetValueOrEmpty(report.TRT23) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432989
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432989()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432989
            Other given name cannot contain space hyphen space
    
            Legacy Rule Format:
            Contains(^TRT23, ' - ')

            Technical Business Rule Format:
            Contains(^TRT23, ' - ')
    
            Data Elements:
    
            ^TRT23 = TRT:RP:MailRecipient:PersonNameDetails:PersonNameDetails.OtherGivenName.Text
            */
            assertion = (report.TRT23 == null ? false : report.TRT23.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000446", Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain space hyphen space",
                    Location = "/tns:TRT/tns:RP/tns:MailRecipient/tns:PersonNameDetails/tns:OtherGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432989"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT23", Value = GetValueOrEmpty(report.TRT23) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432990
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432990()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432990
            Other given name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            ContainsSet(^TRT23, '"--","''","  "')

            Technical Business Rule Format:
            ContainsSet(^TRT23, '"--","''","  "')
    
            Data Elements:
    
            ^TRT23 = TRT:RP:MailRecipient:PersonNameDetails:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.TRT23, @"--|''|  ",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000451", Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:TRT/tns:RP/tns:MailRecipient/tns:PersonNameDetails/tns:OtherGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432990"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT23", Value = GetValueOrEmpty(report.TRT23) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432991
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432991()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432991
            Other given name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
            Legacy Rule Format:
            FoundSet(^TRT23, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')

            Technical Business Rule Format:
            FoundSet(^TRT23, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')
    
            Data Elements:
    
            ^TRT23 = TRT:RP:MailRecipient:PersonNameDetails:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.TRT23, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010448", Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/tns:TRT/tns:RP/tns:MailRecipient/tns:PersonNameDetails/tns:OtherGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432991"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT23", Value = GetValueOrEmpty(report.TRT23) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432992
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432992()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432992
            Non-individual name field cannot contain 'Pship', 'P'ship or 'P/Ship'
    
            Legacy Rule Format:
            StartsWith(^TRT24, 'T/A ') AND EndsWithSet(^TRT24, '" Pship"," P'ship"," P/ship"')

            Technical Business Rule Format:
            StartsWith(^TRT24, 'T/A ') AND EndsWithSet(^TRT24, '" Pship"," P'ship"," P/ship"')
    
            Data Elements:
    
            ^TRT24 = TRT:RP:MailRecipient:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (IsMatch(report.TRT24, @"^(T/A )",RegexOptions.IgnoreCase) && IsMatch(report.TRT24, @"( Pship| P'ship| P/ship)$",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410038", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-individual name field cannot start with 'T/A' and end with 'Pship', 'P'ship or 'P/Ship'",
                    Location = "/tns:TRT/tns:RP/tns:MailRecipient/tns:OrganisationNameDetails/tns:OrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432992"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT24", Value = GetValueOrEmpty(report.TRT24) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432993
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432993()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432993
            Organisation name must contain at least one alpha or numeric character
    
            Legacy Rule Format:
            ^TRT24 <> NULL AND NotContainsSet(^TRT24, '"a-z", "A-Z", "0-9"')

            Technical Business Rule Format:
            ^TRT24 <> NULL AND NotContainsSet(^TRT24, '"a-z", "A-Z", "0-9"')
    
            Data Elements:
    
            ^TRT24 = TRT:RP:MailRecipient:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.TRT24 != null && !(IsMatch(report.TRT24, @"[a-z]|[A-Z]|[0-9]",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410039", Severity = ProcessMessageSeverity.Error,
                    Description = @"Organisation name must contain at least one alpha or numeric character",
                    Location = "/tns:TRT/tns:RP/tns:MailRecipient/tns:OrganisationNameDetails/tns:OrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432993"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT24", Value = GetValueOrEmpty(report.TRT24) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432994
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432994()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432994
            Non Individual Name cannot contain space hyphen space combinations
    
            Legacy Rule Format:
            Contains(^TRT24, ' - ')

            Technical Business Rule Format:
            Contains(^TRT24, ' - ')
    
            Data Elements:
    
            ^TRT24 = TRT:RP:MailRecipient:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.TRT24 == null ? false : report.TRT24.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000413", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain space hyphen space combinations",
                    Location = "/tns:TRT/tns:RP/tns:MailRecipient/tns:OrganisationNameDetails/tns:OrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432994"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT24", Value = GetValueOrEmpty(report.TRT24) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432995
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432995()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432995
            Non Individual Name cannot contain "P/L"
    
            Legacy Rule Format:
            Contains(^TRT24, 'P/L')

            Technical Business Rule Format:
            Contains(^TRT24, 'P/L')
    
            Data Elements:
    
            ^TRT24 = TRT:RP:MailRecipient:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.TRT24 == null ? false : report.TRT24.ToUpperInvariant().Contains(@"P/L"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000414", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain ""P/L""",
                    Location = "/tns:TRT/tns:RP/tns:MailRecipient/tns:OrganisationNameDetails/tns:OrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432995"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT24", Value = GetValueOrEmpty(report.TRT24) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432996
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432996()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432996
            Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            ContainsSet(^TRT24, '"--","''","  "')

            Technical Business Rule Format:
            ContainsSet(^TRT24, '"--","''","  "')
    
            Data Elements:
    
            ^TRT24 = TRT:RP:MailRecipient:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.TRT24, @"--|''|  ",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000416", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:TRT/tns:RP/tns:MailRecipient/tns:OrganisationNameDetails/tns:OrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432996"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT24", Value = GetValueOrEmpty(report.TRT24) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432997
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432997()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432997
            Non Individual Name cannot end with "T/A", "T/A P'ship", "T/A Pship", "T/A P/Ship" or "T/A Partnership"
    
            Legacy Rule Format:
            EndsWithSet(^TRT24, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"')

            Technical Business Rule Format:
            EndsWithSet(^TRT24, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"')
    
            Data Elements:
    
            ^TRT24 = TRT:RP:MailRecipient:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.TRT24, @"( T/A| T/A P'ship| T/A Pship| T/A P/Ship| T/A Partnership)$",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000417", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot end with ""T/A"", ""T/A P'ship"", ""T/A Pship"", ""T/A P/Ship"" or ""T/A Partnership""",
                    Location = "/tns:TRT/tns:RP/tns:MailRecipient/tns:OrganisationNameDetails/tns:OrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432997"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT24", Value = GetValueOrEmpty(report.TRT24) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.432999
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT432999()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.432999
            ABN must pass the ABN algorithm
    
            Legacy Rule Format:
            ^TRT25 <> NULL and FailsABNAlgorithm(^TRT25)

            Technical Business Rule Format:
            ^TRT25 <> NULL and FailsABNAlgorithm(^TRT25)
    
            Data Elements:
    
            ^TRT25 = TRT:RP:MailRecipient:OrganisationNameDetails:Identifiers.AustralianBusinessNumber.Identifier
            */
            assertion = (report.TRT25 != null && FailsABNAlgorithm(report.TRT25));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000477", Severity = ProcessMessageSeverity.Error,
                    Description = @"ABN is invalid",
                    Location = "/tns:TRT/tns:RP/tns:MailRecipient/tns:OrganisationNameDetails/tns:AustralianBusinessNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.432999"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT25", Value = GetValueOrEmpty(report.TRT25) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.433000
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433000(TRT2024.RP_BusinessDetails_Income_IndividualNonBusinessWithholdingPaymentDetails individualNonBusinessWithholdingPaymentDetails, int itemIndex4)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.433000
                    ABN must pass the ABN algorithm
    
                    Legacy Rule Format:
                    ^TRT107 <> NULL AND FailsABNAlgorithm(^TRT107)
        
                    Technical Business Rule Format:
                    ^TRT107 <> NULL AND FailsABNAlgorithm(^TRT107)
            
                    Data Elements:
            
                    ^TRT107 = TRT:RP:BusinessDetails:Income:IndividualNonBusinessWithholdingPaymentDetails:Identifiers.AustralianBusinessNumber.Identifier
                    */
                    assertion = (individualNonBusinessWithholdingPaymentDetails.TRT107 != null && FailsABNAlgorithm(individualNonBusinessWithholdingPaymentDetails.TRT107));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000477", Severity = ProcessMessageSeverity.Error,
                            Description = @"ABN is invalid",
                            Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:IndividualNonBusinessWithholdingPaymentDetailsCollection/tns:IndividualNonBusinessWithholdingPaymentDetails" + OccurrenceIndex(individualNonBusinessWithholdingPaymentDetails.OccurrenceIndex) + "/tns:AustralianBusinessNumberId",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433000"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT107", Value = GetValueOrEmpty(individualNonBusinessWithholdingPaymentDetails.TRT107) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.433002
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433002(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.433002
                    The Tax File Number has failed the Tax File Number algorithm check.
    
                    Legacy Rule Format:
                    FailsTFNAlgorithm(^TRT211)
        
                    Technical Business Rule Format:
                    FailsTFNAlgorithm(^TRT211)
            
                    Data Elements:
            
                    ^TRT211 = TRT:RP:StatementOfDistributionToBeneficiary:Identifiers.TaxFileNumber.Identifier
                    */
                    assertion = FailsTFNAlgorithm(statementOfDistributionToBeneficiary.TRT211);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410031", Severity = ProcessMessageSeverity.Error,
                            Description = @"TFN must be a valid TFN",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:TaxFileNumberId",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433002"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT211", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT211) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.433003
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433003(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.433003
                    Non-individual name field cannot contain 'Pship', 'P'ship or 'P/Ship'
    
                    Legacy Rule Format:
                    StartsWith(^TRT203, 'T/A ') AND EndsWithSet(^TRT203, '" Pship"," P'ship"," P/ship"')
        
                    Technical Business Rule Format:
                    StartsWith(^TRT203, 'T/A ') AND EndsWithSet(^TRT203, '" Pship"," P'ship"," P/ship"')
            
                    Data Elements:
            
                    ^TRT203 = TRT:RP:StatementOfDistributionToBeneficiary:NonIndividualNameDetails:OrganisationNameDetails.OrganisationalName.Text
                    */
                    assertion = (IsMatch(statementOfDistributionToBeneficiary.TRT203, @"^(T/A )",RegexOptions.IgnoreCase) && IsMatch(statementOfDistributionToBeneficiary.TRT203, @"( Pship| P'ship| P/ship)$",RegexOptions.IgnoreCase));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410038", Severity = ProcessMessageSeverity.Error,
                            Description = @"Non-individual name field cannot start with 'T/A' and end with 'Pship', 'P'ship or 'P/Ship'",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:NonIndividualNameDetailsOrganisationNameDetailsOrganisationalNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433003"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT203", Value = statementOfDistributionToBeneficiary.TRT203 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.433004
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433004(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.433004
                    Organisation name must contain at least one alpha or numeric character
    
                    Legacy Rule Format:
                    ^TRT203 <> NULL AND NotContainsSet(^TRT203, '"a-z", "A-Z", "0-9"')
        
                    Technical Business Rule Format:
                    ^TRT203 <> NULL AND NotContainsSet(^TRT203, '"a-z", "A-Z", "0-9"')
            
                    Data Elements:
            
                    ^TRT203 = TRT:RP:StatementOfDistributionToBeneficiary:NonIndividualNameDetails:OrganisationNameDetails.OrganisationalName.Text
                    */
                    assertion = (statementOfDistributionToBeneficiary.TRT203 != null && !(IsMatch(statementOfDistributionToBeneficiary.TRT203, @"[a-z]|[A-Z]|[0-9]",RegexOptions.IgnoreCase)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410039", Severity = ProcessMessageSeverity.Error,
                            Description = @"Organisation name must contain at least one alpha or numeric character",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:NonIndividualNameDetailsOrganisationNameDetailsOrganisationalNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433004"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT203", Value = statementOfDistributionToBeneficiary.TRT203 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.433005
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433005(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.433005
                    Non Individual Name cannot contain space hyphen space combinations
    
                    Legacy Rule Format:
                    Contains(^TRT203, ' - ')
        
                    Technical Business Rule Format:
                    Contains(^TRT203, ' - ')
            
                    Data Elements:
            
                    ^TRT203 = TRT:RP:StatementOfDistributionToBeneficiary:NonIndividualNameDetails:OrganisationNameDetails.OrganisationalName.Text
                    */
                    assertion = (statementOfDistributionToBeneficiary.TRT203 == null ? false : statementOfDistributionToBeneficiary.TRT203.ToUpperInvariant().Contains(@" - "));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000413", Severity = ProcessMessageSeverity.Error,
                            Description = @"Non Individual Name cannot contain space hyphen space combinations",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:NonIndividualNameDetailsOrganisationNameDetailsOrganisationalNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433005"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT203", Value = statementOfDistributionToBeneficiary.TRT203 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.433006
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433006(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.433006
                    Non Individual Name cannot contain "P/L"
    
                    Legacy Rule Format:
                    Contains(^TRT203, 'P/L')
        
                    Technical Business Rule Format:
                    Contains(^TRT203, 'P/L')
            
                    Data Elements:
            
                    ^TRT203 = TRT:RP:StatementOfDistributionToBeneficiary:NonIndividualNameDetails:OrganisationNameDetails.OrganisationalName.Text
                    */
                    assertion = (statementOfDistributionToBeneficiary.TRT203 == null ? false : statementOfDistributionToBeneficiary.TRT203.ToUpperInvariant().Contains(@"P/L"));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000414", Severity = ProcessMessageSeverity.Error,
                            Description = @"Non Individual Name cannot contain ""P/L""",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:NonIndividualNameDetailsOrganisationNameDetailsOrganisationalNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433006"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT203", Value = statementOfDistributionToBeneficiary.TRT203 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.433007
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433007(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.433007
                    Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters
    
                    Legacy Rule Format:
                    ContainsSet(^TRT203, '"--","''","  "')
        
                    Technical Business Rule Format:
                    ContainsSet(^TRT203, '"--","''","  "')
            
                    Data Elements:
            
                    ^TRT203 = TRT:RP:StatementOfDistributionToBeneficiary:NonIndividualNameDetails:OrganisationNameDetails.OrganisationalName.Text
                    */
                    assertion = IsMatch(statementOfDistributionToBeneficiary.TRT203, @"--|''|  ",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000416", Severity = ProcessMessageSeverity.Error,
                            Description = @"Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:NonIndividualNameDetailsOrganisationNameDetailsOrganisationalNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433007"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT203", Value = statementOfDistributionToBeneficiary.TRT203 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.433008
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433008(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.433008
                    Non Individual Name cannot end with "T/A", "T/A P'ship", "T/A Pship", "T/A P/Ship" or "T/A Partnership"
    
                    Legacy Rule Format:
                    EndsWithSet(^TRT203, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"')
        
                    Technical Business Rule Format:
                    EndsWithSet(^TRT203, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"')
            
                    Data Elements:
            
                    ^TRT203 = TRT:RP:StatementOfDistributionToBeneficiary:NonIndividualNameDetails:OrganisationNameDetails.OrganisationalName.Text
                    */
                    assertion = IsMatch(statementOfDistributionToBeneficiary.TRT203, @"( T/A| T/A P'ship| T/A Pship| T/A P/Ship| T/A Partnership)$",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000417", Severity = ProcessMessageSeverity.Error,
                            Description = @"Non Individual Name cannot end with ""T/A"", ""T/A P'ship"", ""T/A Pship"", ""T/A P/Ship"" or ""T/A Partnership""",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:NonIndividualNameDetailsOrganisationNameDetailsOrganisationalNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433008"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT203", Value = statementOfDistributionToBeneficiary.TRT203 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.433009
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433009(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.433009
                    Family name cannot contain "Exec for", "Rep for" or "Trustee for"
    
                    Legacy Rule Format:
                    FoundSet(^TRT200, '"Exec for","Rep for","Trustee for"')
        
                    Technical Business Rule Format:
                    FoundSet(^TRT200, '"Exec for","Rep for","Trustee for"')
            
                    Data Elements:
            
                    ^TRT200 = TRT:RP:StatementOfDistributionToBeneficiary:IndividualNameDetails:PersonNameDetails.FamilyName.Text
                    */
                    assertion = IsMatch(statementOfDistributionToBeneficiary.TRT200, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000424", Severity = ProcessMessageSeverity.Error,
                            Description = @"Family name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsFamilyNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433009"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT200", Value = statementOfDistributionToBeneficiary.TRT200 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.433010
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433010(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.433010
                    Family name cannot contain "MR", "MRS", "MISS" or "MS"
    
                    Legacy Rule Format:
                    FoundSet(^TRT200, '"MR","MRS","MISS","MS"')
        
                    Technical Business Rule Format:
                    FoundSet(^TRT200, '"MR","MRS","MISS","MS"')
            
                    Data Elements:
            
                    ^TRT200 = TRT:RP:StatementOfDistributionToBeneficiary:IndividualNameDetails:PersonNameDetails.FamilyName.Text
                    */
                    assertion = IsMatch(statementOfDistributionToBeneficiary.TRT200, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000426", Severity = ProcessMessageSeverity.Error,
                            Description = @"Family name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsFamilyNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433010"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT200", Value = statementOfDistributionToBeneficiary.TRT200 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.433011
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433011(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.433011
                    Family Name cannot contain "space hyphen space"
    
                    Legacy Rule Format:
                    Contains(^TRT200, ' - ')
        
                    Technical Business Rule Format:
                    Contains(^TRT200, ' - ')
            
                    Data Elements:
            
                    ^TRT200 = TRT:RP:StatementOfDistributionToBeneficiary:IndividualNameDetails:PersonNameDetails.FamilyName.Text
                    */
                    assertion = (statementOfDistributionToBeneficiary.TRT200 == null ? false : statementOfDistributionToBeneficiary.TRT200.ToUpperInvariant().Contains(@" - "));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000423", Severity = ProcessMessageSeverity.Error,
                            Description = @"Family Name cannot contain ""space hyphen space""",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsFamilyNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433011"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT200", Value = statementOfDistributionToBeneficiary.TRT200 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.433012
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433012(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.433012
                    Family name cannot contain repeated hyphen, apostrophe, or space characters
    
                    Legacy Rule Format:
                    ContainsSet(^TRT200, '"--","''","  "')
        
                    Technical Business Rule Format:
                    ContainsSet(^TRT200, '"--","''","  "')
            
                    Data Elements:
            
                    ^TRT200 = TRT:RP:StatementOfDistributionToBeneficiary:IndividualNameDetails:PersonNameDetails.FamilyName.Text
                    */
                    assertion = IsMatch(statementOfDistributionToBeneficiary.TRT200, @"--|''|  ",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000427", Severity = ProcessMessageSeverity.Error,
                            Description = @"Family name cannot contain repeated hyphen, apostrophe, or space characters",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsFamilyNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433012"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT200", Value = statementOfDistributionToBeneficiary.TRT200 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.433013
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433013(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.433013
                    Family name must contain at least one alphabetical character
    
                    Legacy Rule Format:
                    ^TRT200 <> NULL AND NotContainsSet(^TRT200, '"A-Z","a-z"')
        
                    Technical Business Rule Format:
                    ^TRT200 <> NULL AND NotContainsSet(^TRT200, '"A-Z","a-z"')
            
                    Data Elements:
            
                    ^TRT200 = TRT:RP:StatementOfDistributionToBeneficiary:IndividualNameDetails:PersonNameDetails.FamilyName.Text
                    */
                    assertion = (statementOfDistributionToBeneficiary.TRT200 != null && !(IsMatch(statementOfDistributionToBeneficiary.TRT200, @"[A-Z]|[a-z]",RegexOptions.IgnoreCase)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.500142", Severity = ProcessMessageSeverity.Error,
                            Description = @"Family name must contain at least one alphabetical character.",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsFamilyNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433013"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT200", Value = statementOfDistributionToBeneficiary.TRT200 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.433014
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433014(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.433014
                    Family name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
                    Legacy Rule Format:
                    FoundSet(^TRT200, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')
        
                    Technical Business Rule Format:
                    FoundSet(^TRT200, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')
            
                    Data Elements:
            
                    ^TRT200 = TRT:RP:StatementOfDistributionToBeneficiary:IndividualNameDetails:PersonNameDetails.FamilyName.Text
                    */
                    assertion = IsMatch(statementOfDistributionToBeneficiary.TRT200, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.500153", Severity = ProcessMessageSeverity.Error,
                            Description = @"Family name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsFamilyNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433014"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT200", Value = statementOfDistributionToBeneficiary.TRT200 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.433015
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433015(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.433015
                    Given Name must contain at least one alphabetical character
    
                    Legacy Rule Format:
                    ^TRT201 <> NULL AND NotContainsSet(^TRT201, '"A-Z","a-z"')
        
                    Technical Business Rule Format:
                    ^TRT201 <> NULL AND NotContainsSet(^TRT201, '"A-Z","a-z"')
            
                    Data Elements:
            
                    ^TRT201 = TRT:RP:StatementOfDistributionToBeneficiary:IndividualNameDetails:PersonNameDetails.GivenName.Text
                    */
                    assertion = (statementOfDistributionToBeneficiary.TRT201 != null && !(IsMatch(statementOfDistributionToBeneficiary.TRT201, @"[A-Z]|[a-z]",RegexOptions.IgnoreCase)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410063", Severity = ProcessMessageSeverity.Error,
                            Description = @"Given Name must contain at least one alphabetical character",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433015"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT201", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT201) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.433016
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433016(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.433016
                    First name cannot contain "Exec for", "Rep for" or "Trustee for"
    
                    Legacy Rule Format:
                    FoundSet(^TRT201, '"Exec for","Rep for","Trustee for"')
        
                    Technical Business Rule Format:
                    FoundSet(^TRT201, '"Exec for","Rep for","Trustee for"')
            
                    Data Elements:
            
                    ^TRT201 = TRT:RP:StatementOfDistributionToBeneficiary:IndividualNameDetails:PersonNameDetails.GivenName.Text
                    */
                    assertion = IsMatch(statementOfDistributionToBeneficiary.TRT201, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000437", Severity = ProcessMessageSeverity.Error,
                            Description = @"First name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433016"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT201", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT201) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.433017
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433017(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.433017
                    First name cannot contain "MR", "MRS", "MISS" or "MS"
    
                    Legacy Rule Format:
                    FoundSet(^TRT201, '"MR","MRS","MISS","MS"')
        
                    Technical Business Rule Format:
                    FoundSet(^TRT201, '"MR","MRS","MISS","MS"')
            
                    Data Elements:
            
                    ^TRT201 = TRT:RP:StatementOfDistributionToBeneficiary:IndividualNameDetails:PersonNameDetails.GivenName.Text
                    */
                    assertion = IsMatch(statementOfDistributionToBeneficiary.TRT201, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000438", Severity = ProcessMessageSeverity.Error,
                            Description = @"First name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433017"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT201", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT201) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.433018
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433018(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.433018
                    FirstName must be entered if OtherGivenNames is entered
    
                    Legacy Rule Format:
                    ^TRT202 <> NULL AND ^TRT201 = NULL
        
                    Technical Business Rule Format:
                    ^TRT202 <> NULL AND ^TRT201 = NULL
            
                    Data Elements:
            
                    ^TRT201 = TRT:RP:StatementOfDistributionToBeneficiary:IndividualNameDetails:PersonNameDetails.GivenName.Text
            
                    ^TRT202 = TRT:RP:StatementOfDistributionToBeneficiary:IndividualNameDetails:PersonNameDetails.OtherGivenName.Text
                    */
                    assertion = (statementOfDistributionToBeneficiary.TRT202 != null && statementOfDistributionToBeneficiary.TRT201 == null);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410218", Severity = ProcessMessageSeverity.Error,
                            Description = @"FirstName must be entered if OtherGivenNames is entered",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433018"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT202", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT202) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT201", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT201) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.433019
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433019(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.433019
                    First name cannot contain space hyphen space
    
                    Legacy Rule Format:
                    Contains(^TRT201, ' - ')
        
                    Technical Business Rule Format:
                    Contains(^TRT201, ' - ')
            
                    Data Elements:
            
                    ^TRT201 = TRT:RP:StatementOfDistributionToBeneficiary:IndividualNameDetails:PersonNameDetails.GivenName.Text
                    */
                    assertion = (statementOfDistributionToBeneficiary.TRT201 == null ? false : statementOfDistributionToBeneficiary.TRT201.ToUpperInvariant().Contains(@" - "));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000434", Severity = ProcessMessageSeverity.Error,
                            Description = @"First name cannot contain space hyphen space",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433019"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT201", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT201) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.433020
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433020(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.433020
                    First name cannot contain repeated hyphen, apostrophe, or space characters
    
                    Legacy Rule Format:
                    ContainsSet(^TRT201, '"--","''","  "')
        
                    Technical Business Rule Format:
                    ContainsSet(^TRT201, '"--","''","  "')
            
                    Data Elements:
            
                    ^TRT201 = TRT:RP:StatementOfDistributionToBeneficiary:IndividualNameDetails:PersonNameDetails.GivenName.Text
                    */
                    assertion = IsMatch(statementOfDistributionToBeneficiary.TRT201, @"--|''|  ",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000439", Severity = ProcessMessageSeverity.Error,
                            Description = @"First name cannot contain repeated hyphen, apostrophe, or space characters",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433020"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT201", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT201) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.433021
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433021(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.433021
                    First name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
                    Legacy Rule Format:
                    FoundSet(^TRT201, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')
        
                    Technical Business Rule Format:
                    FoundSet(^TRT201, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')
            
                    Data Elements:
            
                    ^TRT201 = TRT:RP:StatementOfDistributionToBeneficiary:IndividualNameDetails:PersonNameDetails.GivenName.Text
                    */
                    assertion = IsMatch(statementOfDistributionToBeneficiary.TRT201, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.010436", Severity = ProcessMessageSeverity.Error,
                            Description = @"First name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433021"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT201", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT201) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.433022
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433022(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.433022
                    Other Given Name must contain at least one alphabetical character
    
                    Legacy Rule Format:
                    ^TRT202 <> NULL AND NotContainsSet(^TRT202, '"A-Z","a-z"')
        
                    Technical Business Rule Format:
                    ^TRT202 <> NULL AND NotContainsSet(^TRT202, '"A-Z","a-z"')
            
                    Data Elements:
            
                    ^TRT202 = TRT:RP:StatementOfDistributionToBeneficiary:IndividualNameDetails:PersonNameDetails.OtherGivenName.Text
                    */
                    assertion = (statementOfDistributionToBeneficiary.TRT202 != null && !(IsMatch(statementOfDistributionToBeneficiary.TRT202, @"[A-Z]|[a-z]",RegexOptions.IgnoreCase)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410131", Severity = ProcessMessageSeverity.Error,
                            Description = @"Other Given Name must contain at least one alphabetical character",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsOtherGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433022"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT202", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT202) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.433023
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433023(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.433023
                    Other given name cannot contain "Exec for", "Rep for" or "Trustee for"
    
                    Legacy Rule Format:
                    FoundSet(^TRT202, '"Exec for","Rep for","Trustee for"')
        
                    Technical Business Rule Format:
                    FoundSet(^TRT202, '"Exec for","Rep for","Trustee for"')
            
                    Data Elements:
            
                    ^TRT202 = TRT:RP:StatementOfDistributionToBeneficiary:IndividualNameDetails:PersonNameDetails.OtherGivenName.Text
                    */
                    assertion = IsMatch(statementOfDistributionToBeneficiary.TRT202, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000449", Severity = ProcessMessageSeverity.Error,
                            Description = @"Other given name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsOtherGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433023"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT202", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT202) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.433024
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433024(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.433024
                    Other given name cannot contain "MR", "MRS", "MISS" or "MS"
    
                    Legacy Rule Format:
                    FoundSet(^TRT202, '"MR","MRS","MISS","MS"')
        
                    Technical Business Rule Format:
                    FoundSet(^TRT202, '"MR","MRS","MISS","MS"')
            
                    Data Elements:
            
                    ^TRT202 = TRT:RP:StatementOfDistributionToBeneficiary:IndividualNameDetails:PersonNameDetails.OtherGivenName.Text
                    */
                    assertion = IsMatch(statementOfDistributionToBeneficiary.TRT202, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000450", Severity = ProcessMessageSeverity.Error,
                            Description = @"Other given name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsOtherGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433024"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT202", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT202) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.433025
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433025(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.433025
                    Other given name cannot contain space hyphen space
    
                    Legacy Rule Format:
                    Contains(^TRT202, ' - ')
        
                    Technical Business Rule Format:
                    Contains(^TRT202, ' - ')
            
                    Data Elements:
            
                    ^TRT202 = TRT:RP:StatementOfDistributionToBeneficiary:IndividualNameDetails:PersonNameDetails.OtherGivenName.Text
                    */
                    assertion = (statementOfDistributionToBeneficiary.TRT202 == null ? false : statementOfDistributionToBeneficiary.TRT202.ToUpperInvariant().Contains(@" - "));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000446", Severity = ProcessMessageSeverity.Error,
                            Description = @"Other given name cannot contain space hyphen space",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsOtherGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433025"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT202", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT202) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.433026
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433026(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.433026
                    Other given name cannot contain repeated hyphen, apostrophe, or space characters
    
                    Legacy Rule Format:
                    ContainsSet(^TRT202, '"--","''","  "')
        
                    Technical Business Rule Format:
                    ContainsSet(^TRT202, '"--","''","  "')
            
                    Data Elements:
            
                    ^TRT202 = TRT:RP:StatementOfDistributionToBeneficiary:IndividualNameDetails:PersonNameDetails.OtherGivenName.Text
                    */
                    assertion = IsMatch(statementOfDistributionToBeneficiary.TRT202, @"--|''|  ",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000451", Severity = ProcessMessageSeverity.Error,
                            Description = @"Other given name cannot contain repeated hyphen, apostrophe, or space characters",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsOtherGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433026"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT202", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT202) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.433027
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433027(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.433027
                    Other given name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
                    Legacy Rule Format:
                    FoundSet(^TRT202, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')
        
                    Technical Business Rule Format:
                    FoundSet(^TRT202, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')
            
                    Data Elements:
            
                    ^TRT202 = TRT:RP:StatementOfDistributionToBeneficiary:IndividualNameDetails:PersonNameDetails.OtherGivenName.Text
                    */
                    assertion = IsMatch(statementOfDistributionToBeneficiary.TRT202, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.010448", Severity = ProcessMessageSeverity.Error,
                            Description = @"Other given name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:IndividualNameDetails/tns:PersonNameDetailsOtherGivenNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433027"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT202", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT202) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.433028
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433028(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.433028
                    State and post code must be present for an Australian address
    
                    Legacy Rule Format:
                    ^TRT320 = FALSE AND (^TRT209 = NULL OR ^TRT208 = NULL)
        
                    Technical Business Rule Format:
                    ^TRT320 = FALSE AND (^TRT209 = NULL OR ^TRT208 = NULL)
            
                    Data Elements:
            
                    ^TRT320 = TRT:RP:StatementOfDistributionToBeneficiary:BeneficiaryAddressDetails:AddressDetails.OverseasAddress.Indicator
            
                    ^TRT208 = TRT:RP:StatementOfDistributionToBeneficiary:BeneficiaryAddressDetails:AddressDetails.StateOrTerritory.Code
            
                    ^TRT209 = TRT:RP:StatementOfDistributionToBeneficiary:BeneficiaryAddressDetails:AddressDetails.Postcode.Text
                    */
                    assertion = (statementOfDistributionToBeneficiary.TRT320 == false && (statementOfDistributionToBeneficiary.TRT209 == null || statementOfDistributionToBeneficiary.TRT208 == null));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.434147", Severity = ProcessMessageSeverity.Error,
                            Description = @"State and post code must be present for an Australian address",
                            LongDescription = @"If the overseas indicator is no (false), then an Australian state code and post code must be present",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:BeneficiaryAddressDetails/tns:OverseasAddressI",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433028"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT320", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT320) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT209", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT209) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT208", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT208) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.433029
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433029(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.433029
                    Address line 1 contains incorrect 'care of' reference. The only acceptable 'care of' reference is C/-
    
                    Legacy Rule Format:
                    StartsWithSet(^TRT205,'"C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^TRT205, '" C/O "," C/ "," Care Of "')
        
                    Technical Business Rule Format:
                    StartsWithSet(^TRT205,'"C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^TRT205, '" C/O "," C/ "," Care Of "')
            
                    Data Elements:
            
                    ^TRT205 = TRT:RP:StatementOfDistributionToBeneficiary:BeneficiaryAddressDetails:AddressDetails.Line1.Text
                    */
                    assertion = (IsMatch(statementOfDistributionToBeneficiary.TRT205, @"^(C/O |C/ |Care Of |CO )",RegexOptions.IgnoreCase) || IsMatch(statementOfDistributionToBeneficiary.TRT205, @" C/O | C/ | Care Of ",RegexOptions.IgnoreCase));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000406", Severity = ProcessMessageSeverity.Error,
                            Description = @"Address line 1 contains incorrect 'care of' reference. The only acceptable 'care of' reference is C/-",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:BeneficiaryAddressDetails/tns:Line1T",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433029"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT205", Value = statementOfDistributionToBeneficiary.TRT205 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.433030
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433030(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.433030
                    Address Line 2 must be present if Address Line 1 contains 'C/-'
    
                    Legacy Rule Format:
                    Contains(^TRT205, 'C/-') AND ^TRT206 = NULL
        
                    Technical Business Rule Format:
                    Contains(^TRT205, 'C/-') AND ^TRT206 = NULL
            
                    Data Elements:
            
                    ^TRT205 = TRT:RP:StatementOfDistributionToBeneficiary:BeneficiaryAddressDetails:AddressDetails.Line1.Text
            
                    ^TRT206 = TRT:RP:StatementOfDistributionToBeneficiary:BeneficiaryAddressDetails:AddressDetails.Line2.Text
                    */
                    assertion = ((statementOfDistributionToBeneficiary.TRT205 == null ? false : statementOfDistributionToBeneficiary.TRT205.ToUpperInvariant().Contains(@"C/-")) && statementOfDistributionToBeneficiary.TRT206 == null);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410002", Severity = ProcessMessageSeverity.Error,
                            Description = @"Address Line 2 must be present if Address Line 1 contains 'C/-'",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:BeneficiaryAddressDetails/tns:Line1T",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433030"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT205", Value = statementOfDistributionToBeneficiary.TRT205 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT206", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT206) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.433031
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433031(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.433031
                    Address Line 1 must not contain 'AS ABOVE'
    
                    Legacy Rule Format:
                    FoundSet(^TRT205, '"AS ABOVE"')
        
                    Technical Business Rule Format:
                    FoundSet(^TRT205, '"AS ABOVE"')
            
                    Data Elements:
            
                    ^TRT205 = TRT:RP:StatementOfDistributionToBeneficiary:BeneficiaryAddressDetails:AddressDetails.Line1.Text
                    */
                    assertion = IsMatch(statementOfDistributionToBeneficiary.TRT205, @"(^(AS ABOVE)$)|(^(.*)( AS ABOVE )(.*)$)|(^(AS ABOVE )(.*)$)|(^(.*)( AS ABOVE)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410001", Severity = ProcessMessageSeverity.Error,
                            Description = @"Address Line 1 must not contain 'AS ABOVE'",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:BeneficiaryAddressDetails/tns:Line1T",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433031"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT205", Value = statementOfDistributionToBeneficiary.TRT205 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.433032
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433032(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.433032
                    Address line 1 cannot contain "UNKNOWN"
    
                    Legacy Rule Format:
                    Contains(^TRT205, 'UNKNOWN')
        
                    Technical Business Rule Format:
                    Contains(^TRT205, 'UNKNOWN')
            
                    Data Elements:
            
                    ^TRT205 = TRT:RP:StatementOfDistributionToBeneficiary:BeneficiaryAddressDetails:AddressDetails.Line1.Text
                    */
                    assertion = (statementOfDistributionToBeneficiary.TRT205 == null ? false : statementOfDistributionToBeneficiary.TRT205.ToUpperInvariant().Contains(@"UNKNOWN"));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000405", Severity = ProcessMessageSeverity.Error,
                            Description = @"Address line 1 cannot contain ""UNKNOWN""",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:BeneficiaryAddressDetails/tns:Line1T",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433032"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT205", Value = statementOfDistributionToBeneficiary.TRT205 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.433033
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433033(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.433033
                    Care of' is not valid for address line 2
    
                    Legacy Rule Format:
                    StartsWithSet(^TRT206,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^TRT206, '" C/- "," C/O "," C/ "," Care Of "')
        
                    Technical Business Rule Format:
                    StartsWithSet(^TRT206,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^TRT206, '" C/- "," C/O "," C/ "," Care Of "')
            
                    Data Elements:
            
                    ^TRT206 = TRT:RP:StatementOfDistributionToBeneficiary:BeneficiaryAddressDetails:AddressDetails.Line2.Text
                    */
                    assertion = (IsMatch(statementOfDistributionToBeneficiary.TRT206, @"^(C/- |C/O |C/ |Care Of |CO )",RegexOptions.IgnoreCase) || IsMatch(statementOfDistributionToBeneficiary.TRT206, @" C/- | C/O | C/ | Care Of ",RegexOptions.IgnoreCase));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000408", Severity = ProcessMessageSeverity.Error,
                            Description = @"Care of' is not valid for address line 2",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:BeneficiaryAddressDetails/tns:Line2T",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433033"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT206", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT206) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.433034
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433034(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.433034
                    Address line 2 cannot contain "UNKNOWN"
    
                    Legacy Rule Format:
                    Contains(^TRT206, 'UNKNOWN')
        
                    Technical Business Rule Format:
                    Contains(^TRT206, 'UNKNOWN')
            
                    Data Elements:
            
                    ^TRT206 = TRT:RP:StatementOfDistributionToBeneficiary:BeneficiaryAddressDetails:AddressDetails.Line2.Text
                    */
                    assertion = (statementOfDistributionToBeneficiary.TRT206 == null ? false : statementOfDistributionToBeneficiary.TRT206.ToUpperInvariant().Contains(@"UNKNOWN"));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000407", Severity = ProcessMessageSeverity.Error,
                            Description = @"Address line 2 cannot contain ""UNKNOWN""",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:BeneficiaryAddressDetails/tns:Line2T",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433034"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT206", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT206) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.433035
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433035(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.433035
                    Suburb/Town/City cannot contain numeric characters
    
                    Legacy Rule Format:
                    ^TRT320 = FALSE AND ContainsSet(^TRT207, '0-9')
        
                    Technical Business Rule Format:
                    ^TRT320 = FALSE AND ContainsSet(^TRT207, '0-9')
            
                    Data Elements:
            
                    ^TRT207 = TRT:RP:StatementOfDistributionToBeneficiary:BeneficiaryAddressDetails:AddressDetails.LocalityName.Text
            
                    ^TRT320 = TRT:RP:StatementOfDistributionToBeneficiary:BeneficiaryAddressDetails:AddressDetails.OverseasAddress.Indicator
                    */
                    assertion = (statementOfDistributionToBeneficiary.TRT320 == false && IsMatch(statementOfDistributionToBeneficiary.TRT207, @"[0-9]",RegexOptions.IgnoreCase));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000410", Severity = ProcessMessageSeverity.Error,
                            Description = @"Suburb/Town/City cannot contain numeric characters",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:BeneficiaryAddressDetails/tns:LocalityNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433035"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT320", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT320) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT207", Value = statementOfDistributionToBeneficiary.TRT207 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.433036
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433036(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.433036
                    Suburb/Town/City cannot contain state code
    
                    Legacy Rule Format:
                    FoundSet(^TRT207, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"')
        
                    Technical Business Rule Format:
                    FoundSet(^TRT207, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"')
            
                    Data Elements:
            
                    ^TRT207 = TRT:RP:StatementOfDistributionToBeneficiary:BeneficiaryAddressDetails:AddressDetails.LocalityName.Text
                    */
                    assertion = IsMatch(statementOfDistributionToBeneficiary.TRT207, @"(^(QLD|NSW|VIC|SA|WA|NT|ACT|TAS)$)|(^(.*)( QLD | NSW | VIC | SA | WA | NT | ACT | TAS )(.*)$)|(^(QLD |NSW |VIC |SA |WA |NT |ACT |TAS )(.*)$)|(^(.*)( QLD| NSW| VIC| SA| WA| NT| ACT| TAS)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000411", Severity = ProcessMessageSeverity.Error,
                            Description = @"Suburb/Town/City cannot contain state code",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:BeneficiaryAddressDetails/tns:LocalityNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433036"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT207", Value = statementOfDistributionToBeneficiary.TRT207 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.433037
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433037(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.433037
                    Suburb/Town/City cannot contain "UNKNOWN"
    
                    Legacy Rule Format:
                    Contains(^TRT207, 'UNKNOWN')
        
                    Technical Business Rule Format:
                    Contains(^TRT207, 'UNKNOWN')
            
                    Data Elements:
            
                    ^TRT207 = TRT:RP:StatementOfDistributionToBeneficiary:BeneficiaryAddressDetails:AddressDetails.LocalityName.Text
                    */
                    assertion = (statementOfDistributionToBeneficiary.TRT207 == null ? false : statementOfDistributionToBeneficiary.TRT207.ToUpperInvariant().Contains(@"UNKNOWN"));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000409", Severity = ProcessMessageSeverity.Error,
                            Description = @"Suburb/Town/City cannot contain ""UNKNOWN""",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:BeneficiaryAddressDetails/tns:LocalityNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433037"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT207", Value = statementOfDistributionToBeneficiary.TRT207 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.433038
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433038(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.433038
                    Country code must be present for overseas address
    
                    Legacy Rule Format:
                    ^TRT320 = TRUE AND ^TRT382 = NULL
        
                    Technical Business Rule Format:
                    ^TRT320 = TRUE AND ^TRT382 = NULL
            
                    Data Elements:
            
                    ^TRT382 = TRT:RP:StatementOfDistributionToBeneficiary:BeneficiaryAddressDetails:AddressDetails.Country.Code
            
                    ^TRT320 = TRT:RP:StatementOfDistributionToBeneficiary:BeneficiaryAddressDetails:AddressDetails.OverseasAddress.Indicator
                    */
                    assertion = (statementOfDistributionToBeneficiary.TRT320 == true && statementOfDistributionToBeneficiary.TRT382 == null);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410191", Severity = ProcessMessageSeverity.Error,
                            Description = @"Country code must be present for overseas address",
                            LongDescription = @"When the overseas address indicator is 'TRUE', the corresponding Country Code must be supplied",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:BeneficiaryAddressDetails/tns:CountryC",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433038"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT320", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT320) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT382", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT382) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.433039
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433039(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.433039
                    Overseas address indicator must not be 'TRUE' if Country Code is 'au'
    
                    Legacy Rule Format:
                    ^TRT382 = 'au' AND ^TRT320 = TRUE
        
                    Technical Business Rule Format:
                    ^TRT382 = 'au' AND ^TRT320 = TRUE
            
                    Data Elements:
            
                    ^TRT382 = TRT:RP:StatementOfDistributionToBeneficiary:BeneficiaryAddressDetails:AddressDetails.Country.Code
            
                    ^TRT320 = TRT:RP:StatementOfDistributionToBeneficiary:BeneficiaryAddressDetails:AddressDetails.OverseasAddress.Indicator
                    */
                    assertion = (statementOfDistributionToBeneficiary.TRT382 == @"au" && statementOfDistributionToBeneficiary.TRT320 == true);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410211", Severity = ProcessMessageSeverity.Error,
                            Description = @"Overseas address indicator must not be 'TRUE' if Country Code is 'au'",
                            LongDescription = @"The overseas address indicator must be 'FALSE' if the Country Code is 'au' - representing Australia. If the address is an overseas address please supply the correct Country Code",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:BeneficiaryAddressDetails/tns:CountryC",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433039"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT382", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT382) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT320", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT320) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.433040
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433040(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.433040
                    If an overseas Country Code is used, the Overseas Address indicator must be 'TRUE'
    
                    Legacy Rule Format:
                    ^TRT382 <> NULL AND ^TRT382 <> 'au' AND ^TRT320 = FALSE
        
                    Technical Business Rule Format:
                    ^TRT382 <> NULL AND ^TRT382 <> 'au' AND ^TRT320 = FALSE
            
                    Data Elements:
            
                    ^TRT382 = TRT:RP:StatementOfDistributionToBeneficiary:BeneficiaryAddressDetails:AddressDetails.Country.Code
            
                    ^TRT320 = TRT:RP:StatementOfDistributionToBeneficiary:BeneficiaryAddressDetails:AddressDetails.OverseasAddress.Indicator
                    */
                    assertion = (statementOfDistributionToBeneficiary.TRT382 != null && statementOfDistributionToBeneficiary.TRT382 != @"au" && statementOfDistributionToBeneficiary.TRT320 == false);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410212", Severity = ProcessMessageSeverity.Error,
                            Description = @"If an overseas Country Code is used, the Overseas Address indicator must be 'TRUE'",
                            LongDescription = @"If the address is an Australian address, the overseas address indicator must be set to 'FALSE' and you do not need to supply a Country Code. If the address is an overseas address, the overseas address indicator must be set to 'TRUE' and you must supply a Country Code other than 'au'",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:BeneficiaryAddressDetails/tns:CountryC",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433040"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT382", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT382) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT320", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT320) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.433041
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433041()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.433041
            If an exemption or rollover has been applied then the exemption or rollover code(s) must be supplied.
    
            Legacy Rule Format:
            ^TRT454 = TRUE AND Count(^TRT448) = 0

            Technical Business Rule Format:
            ^TRT454 = TRUE AND Count(^TRT448) = 0
    
            Data Elements:
    
            ^TRT454 = TRT:RP:BusinessDetails:Income:CapitalGains:CapitalGainsTax.ExemptionOrRolloverApplied.Indicator
    
            ^TRT448 = TRT:RP:BusinessDetails:Income:CapitalGains:CapitalGainsTax.ExemptionOrRolloverApplied.Code
            */
            assertion = (report.TRT454 == true && Count(report.TRT448Collection) == 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438019", Severity = ProcessMessageSeverity.Error,
                    Description = @"Exemption or rollover code not supplied",
                    LongDescription = @"If an exemption or rollover has been applied then the exemption or rollover code(s) must be supplied",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:CapitalGains/tns:TaxExemptionOrRolloverAppliedI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433041"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT454", Value = GetValueOrEmpty(report.TRT454) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.433042
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433042()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.433042
            If an exemption or rollover code is provided then the exemption or rollover applied indicator must be true.
    
            Legacy Rule Format:
            Count(^TRT448) > 0 AND ^TRT454 <> TRUE

            Technical Business Rule Format:
            Count(^TRT448) > 0 AND ^TRT454 <> TRUE
    
            Data Elements:
    
            ^TRT454 = TRT:RP:BusinessDetails:Income:CapitalGains:CapitalGainsTax.ExemptionOrRolloverApplied.Indicator
    
            ^TRT448 = TRT:RP:BusinessDetails:Income:CapitalGains:CapitalGainsTax.ExemptionOrRolloverApplied.Code
            */
            assertion = (Count(report.TRT448Collection) > 0 && report.TRT454 != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438021", Severity = ProcessMessageSeverity.Error,
                    Description = @"Exemption or rollover applied indicator must be true",
                    LongDescription = @"If an exemption or rollover code(s) is provided then the exemption or rollover applied indicator must be true",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:CapitalGains/tns:TaxExemptionOrRolloverAppliedI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433042"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT454", Value = GetValueOrEmpty(report.TRT454) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.433051
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433051()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.433051
            If the income includes personal services income (PSI) then total amount of income and deductions, results test satisfied and business determination indicators must all be completed
    
            Legacy Rule Format:
            ^TRT155 = TRUE AND (^TRT450 = NULL OR ^TRT451 = NULL OR ^TRT452 = NULL OR ^TRT453 = NULL)

            Technical Business Rule Format:
            ^TRT155 = TRUE AND (^TRT450 = NULL OR ^TRT451 = NULL OR ^TRT452 = NULL OR ^TRT453 = NULL)
    
            Data Elements:
    
            ^TRT155 = TRT:RP:BusinessDetails:Income:PersonalServicesIncome:Income.PersonalServicesIncomeIncluded.Indicator
    
            ^TRT450 = TRT:RP:BusinessDetails:Income:PersonalServicesIncome:Income.PersonalServicesIncome.Total.Amount
    
            ^TRT451 = TRT:RP:BusinessDetails:Income:PersonalServicesIncome:Expense.PersonalServicesIncome.Deduction.Amount
    
            ^TRT452 = TRT:RP:BusinessDetails:Income:PersonalServicesIncome:IncomeTax.PersonalServicesIncome.IndividualResultsTestSatisfied.Indicator
    
            ^TRT453 = TRT:RP:BusinessDetails:Income:PersonalServicesIncome:IncomeTax.PersonalServicesIncome.IndividualPersonalServiceBusinessDeterminationHeld.Indicator
            */
            assertion = (report.TRT155 == true && (report.TRT450 == null || report.TRT451 == null || report.TRT452 == null || report.TRT453 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438010", Severity = ProcessMessageSeverity.Error,
                    Description = @"Personal services income section completed incorrectly",
                    LongDescription = @"If the income includes personal services income (PSI) then total amount of income and deductions, results test satisfied and business determination indicators must all be completed",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:PersonalServicesIncome/tns:IncludedI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433051"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT155", Value = GetValueOrEmpty(report.TRT155) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT450", Value = GetValueOrEmpty(report.TRT450) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT451", Value = GetValueOrEmpty(report.TRT451) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT452", Value = GetValueOrEmpty(report.TRT452) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT453", Value = GetValueOrEmpty(report.TRT453) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.433052
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433052()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.433052
            If the income does not include personal services income (PSI) then total amount of income and deductions,results test satisfied,business determination,unrelated clients test,employment test and business premises test indicators must not be completed
    
            Legacy Rule Format:
            ^TRT155 = FALSE AND (^TRT450 <> NULL OR ^TRT451 <> NULL OR ^TRT452 <> NULL OR ^TRT453 <> NULL OR ^TRT455 <> NULL OR ^TRT456 <> NULL OR ^TRT457 <> NULL)

            Technical Business Rule Format:
            ^TRT155 = FALSE AND (^TRT450 <> NULL OR ^TRT451 <> NULL OR ^TRT452 <> NULL OR ^TRT453 <> NULL OR ^TRT455 <> NULL OR ^TRT456 <> NULL OR ^TRT457 <> NULL)
    
            Data Elements:
    
            ^TRT155 = TRT:RP:BusinessDetails:Income:PersonalServicesIncome:Income.PersonalServicesIncomeIncluded.Indicator
    
            ^TRT450 = TRT:RP:BusinessDetails:Income:PersonalServicesIncome:Income.PersonalServicesIncome.Total.Amount
    
            ^TRT451 = TRT:RP:BusinessDetails:Income:PersonalServicesIncome:Expense.PersonalServicesIncome.Deduction.Amount
    
            ^TRT452 = TRT:RP:BusinessDetails:Income:PersonalServicesIncome:IncomeTax.PersonalServicesIncome.IndividualResultsTestSatisfied.Indicator
    
            ^TRT453 = TRT:RP:BusinessDetails:Income:PersonalServicesIncome:IncomeTax.PersonalServicesIncome.IndividualPersonalServiceBusinessDeterminationHeld.Indicator
    
            ^TRT455 = TRT:RP:BusinessDetails:Income:PersonalServicesIncome:IncomeTax.PersonalServicesIncome.UnrelatedClientsTestSatisfied.Indicator
    
            ^TRT456 = TRT:RP:BusinessDetails:Income:PersonalServicesIncome:IncomeTax.PersonalServicesIncome.EmploymentTestSatisfied.Indicator
    
            ^TRT457 = TRT:RP:BusinessDetails:Income:PersonalServicesIncome:IncomeTax.PersonalServicesIncome.BusinessPremisesTestSatisfied.Indicator
            */
            assertion = (report.TRT155 == false && (report.TRT450 != null || report.TRT451 != null || report.TRT452 != null || report.TRT453 != null || report.TRT455 != null || report.TRT456 != null || report.TRT457 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Personal services income section completed incorrectly",
                    LongDescription = @"If the income does not include personal services income (PSI) then total amount of income and deductions,results test satisfied,business determination,unrelated clients test,employment test and business premises test indicators must not be completed",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:PersonalServicesIncome/tns:IncludedI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433052"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT155", Value = GetValueOrEmpty(report.TRT155) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT450", Value = GetValueOrEmpty(report.TRT450) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT451", Value = GetValueOrEmpty(report.TRT451) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT452", Value = GetValueOrEmpty(report.TRT452) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT453", Value = GetValueOrEmpty(report.TRT453) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT455", Value = GetValueOrEmpty(report.TRT455) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT456", Value = GetValueOrEmpty(report.TRT456) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT457", Value = GetValueOrEmpty(report.TRT457) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.433053
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433053()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.433053
            Non-individual name field cannot contain 'Pship', 'P'ship or 'P/Ship'
    
            Legacy Rule Format:
            StartsWith(^TRT161, 'T/A ') AND EndsWithSet(^TRT161, '" Pship"," P'ship"," P/ship"')

            Technical Business Rule Format:
            StartsWith(^TRT161, 'T/A ') AND EndsWithSet(^TRT161, '" Pship"," P'ship"," P/ship"')
    
            Data Elements:
    
            ^TRT161 = TRT:RP:BusinessAndProfessionalInformation:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (IsMatch(report.TRT161, @"^(T/A )",RegexOptions.IgnoreCase) && IsMatch(report.TRT161, @"( Pship| P'ship| P/ship)$",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410038", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-individual name field cannot start with 'T/A' and end with 'Pship', 'P'ship or 'P/Ship'",
                    Location = "/tns:TRT/tns:RP/tns:BusinessAndProfessionalInformation/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433053"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT161", Value = report.TRT161 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.433054
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433054()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.433054
            Organisation name must contain at least one alpha or numeric character
    
            Legacy Rule Format:
            ^TRT161 <> NULL AND NotContainsSet(^TRT161, '"a-z", "A-Z", "0-9"')

            Technical Business Rule Format:
            ^TRT161 <> NULL AND NotContainsSet(^TRT161, '"a-z", "A-Z", "0-9"')
    
            Data Elements:
    
            ^TRT161 = TRT:RP:BusinessAndProfessionalInformation:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.TRT161 != null && !(IsMatch(report.TRT161, @"[a-z]|[A-Z]|[0-9]",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410039", Severity = ProcessMessageSeverity.Error,
                    Description = @"Organisation name must contain at least one alpha or numeric character",
                    Location = "/tns:TRT/tns:RP/tns:BusinessAndProfessionalInformation/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433054"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT161", Value = report.TRT161 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.433055
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433055()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.433055
            Non Individual Name cannot contain space hyphen space combinations
    
            Legacy Rule Format:
            Contains(^TRT161, ' - ')

            Technical Business Rule Format:
            Contains(^TRT161, ' - ')
    
            Data Elements:
    
            ^TRT161 = TRT:RP:BusinessAndProfessionalInformation:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.TRT161 == null ? false : report.TRT161.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000413", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain space hyphen space combinations",
                    Location = "/tns:TRT/tns:RP/tns:BusinessAndProfessionalInformation/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433055"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT161", Value = report.TRT161 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.433056
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433056()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.433056
            Non Individual Name cannot contain "P/L"
    
            Legacy Rule Format:
            Contains(^TRT161, 'P/L')

            Technical Business Rule Format:
            Contains(^TRT161, 'P/L')
    
            Data Elements:
    
            ^TRT161 = TRT:RP:BusinessAndProfessionalInformation:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.TRT161 == null ? false : report.TRT161.ToUpperInvariant().Contains(@"P/L"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000414", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain ""P/L""",
                    Location = "/tns:TRT/tns:RP/tns:BusinessAndProfessionalInformation/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433056"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT161", Value = report.TRT161 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.433057
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433057()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.433057
            Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            ContainsSet(^TRT161, '"--","''","  "')

            Technical Business Rule Format:
            ContainsSet(^TRT161, '"--","''","  "')
    
            Data Elements:
    
            ^TRT161 = TRT:RP:BusinessAndProfessionalInformation:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.TRT161, @"--|''|  ",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000416", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:TRT/tns:RP/tns:BusinessAndProfessionalInformation/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433057"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT161", Value = report.TRT161 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.433058
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433058()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.433058
            Non Individual Name cannot end with "T/A", "T/A P'ship", "T/A Pship", "T/A P/Ship" or "T/A Partnership"
    
            Legacy Rule Format:
            EndsWithSet(^TRT161, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"')

            Technical Business Rule Format:
            EndsWithSet(^TRT161, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"')
    
            Data Elements:
    
            ^TRT161 = TRT:RP:BusinessAndProfessionalInformation:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.TRT161, @"( T/A| T/A P'ship| T/A Pship| T/A P/Ship| T/A Partnership)$",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000417", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot end with ""T/A"", ""T/A P'ship"", ""T/A Pship"", ""T/A P/Ship"" or ""T/A Partnership""",
                    Location = "/tns:TRT/tns:RP/tns:BusinessAndProfessionalInformation/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433058"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT161", Value = report.TRT161 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.433059
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433059()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.433059
            State and post code must be present for an Australian address
    
            Legacy Rule Format:
            ^TRT316 = FALSE AND (^TRT164 = NULL OR ^TRT165 = NULL)

            Technical Business Rule Format:
            ^TRT316 = FALSE AND (^TRT164 = NULL OR ^TRT165 = NULL)
    
            Data Elements:
    
            ^TRT316 = TRT:RP:BusinessAndProfessionalInformation:BusinessAddressDetails:AddressDetails.OverseasAddress.Indicator
    
            ^TRT164 = TRT:RP:BusinessAndProfessionalInformation:BusinessAddressDetails:AddressDetails.StateOrTerritory.Code
    
            ^TRT165 = TRT:RP:BusinessAndProfessionalInformation:BusinessAddressDetails:AddressDetails.Postcode.Text
            */
            assertion = (report.TRT316 == false && (report.TRT164 == null || report.TRT165 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.434147", Severity = ProcessMessageSeverity.Error,
                    Description = @"State and post code must be present for an Australian address",
                    LongDescription = @"If the overseas indicator is no (false), then an Australian state code and post code must be present",
                    Location = "/tns:TRT/tns:RP/tns:BusinessAndProfessionalInformation/tns:BusinessAddressDetails/tns:OverseasAddressI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433059"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT316", Value = GetValueOrEmpty(report.TRT316) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT164", Value = GetValueOrEmpty(report.TRT164) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT165", Value = GetValueOrEmpty(report.TRT165) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.433060
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433060()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.433060
            Address line 1 contains incorrect 'care of' reference. The only acceptable 'care of' reference is C/-
    
            Legacy Rule Format:
            StartsWithSet(^TRT162,'"C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^TRT162, '" C/O "," C/ "," Care Of "')

            Technical Business Rule Format:
            StartsWithSet(^TRT162,'"C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^TRT162, '" C/O "," C/ "," Care Of "')
    
            Data Elements:
    
            ^TRT162 = TRT:RP:BusinessAndProfessionalInformation:BusinessAddressDetails:AddressDetails.Line1.Text
            */
            assertion = (IsMatch(report.TRT162, @"^(C/O |C/ |Care Of |CO )",RegexOptions.IgnoreCase) || IsMatch(report.TRT162, @" C/O | C/ | Care Of ",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000406", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 contains incorrect 'care of' reference. The only acceptable 'care of' reference is C/-",
                    Location = "/tns:TRT/tns:RP/tns:BusinessAndProfessionalInformation/tns:BusinessAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433060"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT162", Value = report.TRT162 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.433061
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433061()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.433061
            Address Line 2 must be present if Address Line 1 contains 'C/-'
    
            Legacy Rule Format:
            Contains(^TRT162, 'C/-') AND ^TRT309 = NULL

            Technical Business Rule Format:
            Contains(^TRT162, 'C/-') AND ^TRT309 = NULL
    
            Data Elements:
    
            ^TRT162 = TRT:RP:BusinessAndProfessionalInformation:BusinessAddressDetails:AddressDetails.Line1.Text
    
            ^TRT309 = TRT:RP:BusinessAndProfessionalInformation:BusinessAddressDetails:AddressDetails.Line2.Text
            */
            assertion = ((report.TRT162 == null ? false : report.TRT162.ToUpperInvariant().Contains(@"C/-")) && report.TRT309 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410002", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 2 must be present if Address Line 1 contains 'C/-'",
                    Location = "/tns:TRT/tns:RP/tns:BusinessAndProfessionalInformation/tns:BusinessAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433061"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT162", Value = report.TRT162 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT309", Value = GetValueOrEmpty(report.TRT309) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.433062
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433062()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.433062
            Address Line 1 must not contain 'AS ABOVE'
    
            Legacy Rule Format:
            FoundSet(^TRT162, '"AS ABOVE"')

            Technical Business Rule Format:
            FoundSet(^TRT162, '"AS ABOVE"')
    
            Data Elements:
    
            ^TRT162 = TRT:RP:BusinessAndProfessionalInformation:BusinessAddressDetails:AddressDetails.Line1.Text
            */
            assertion = IsMatch(report.TRT162, @"(^(AS ABOVE)$)|(^(.*)( AS ABOVE )(.*)$)|(^(AS ABOVE )(.*)$)|(^(.*)( AS ABOVE)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410001", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 1 must not contain 'AS ABOVE'",
                    Location = "/tns:TRT/tns:RP/tns:BusinessAndProfessionalInformation/tns:BusinessAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433062"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT162", Value = report.TRT162 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.433063
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433063()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.433063
            Address line 1 cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            Contains(^TRT162, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^TRT162, 'UNKNOWN')
    
            Data Elements:
    
            ^TRT162 = TRT:RP:BusinessAndProfessionalInformation:BusinessAddressDetails:AddressDetails.Line1.Text
            */
            assertion = (report.TRT162 == null ? false : report.TRT162.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000405", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 cannot contain ""UNKNOWN""",
                    Location = "/tns:TRT/tns:RP/tns:BusinessAndProfessionalInformation/tns:BusinessAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433063"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT162", Value = report.TRT162 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.433064
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433064()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.433064
            Care of' is not valid for address line 2
    
            Legacy Rule Format:
            StartsWithSet(^TRT309,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^TRT309, '" C/- "," C/O "," C/ "," Care Of "')

            Technical Business Rule Format:
            StartsWithSet(^TRT309,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^TRT309, '" C/- "," C/O "," C/ "," Care Of "')
    
            Data Elements:
    
            ^TRT309 = TRT:RP:BusinessAndProfessionalInformation:BusinessAddressDetails:AddressDetails.Line2.Text
            */
            assertion = (IsMatch(report.TRT309, @"^(C/- |C/O |C/ |Care Of |CO )",RegexOptions.IgnoreCase) || IsMatch(report.TRT309, @" C/- | C/O | C/ | Care Of ",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000408", Severity = ProcessMessageSeverity.Error,
                    Description = @"Care of' is not valid for address line 2",
                    Location = "/tns:TRT/tns:RP/tns:BusinessAndProfessionalInformation/tns:BusinessAddressDetails/tns:Line2T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433064"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT309", Value = GetValueOrEmpty(report.TRT309) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.433065
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433065()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.433065
            Address line 2 cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            Contains(^TRT309, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^TRT309, 'UNKNOWN')
    
            Data Elements:
    
            ^TRT309 = TRT:RP:BusinessAndProfessionalInformation:BusinessAddressDetails:AddressDetails.Line2.Text
            */
            assertion = (report.TRT309 == null ? false : report.TRT309.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000407", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 2 cannot contain ""UNKNOWN""",
                    Location = "/tns:TRT/tns:RP/tns:BusinessAndProfessionalInformation/tns:BusinessAddressDetails/tns:Line2T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433065"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT309", Value = GetValueOrEmpty(report.TRT309) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.433066
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433066()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.433066
            Suburb/Town/City cannot contain state code
    
            Legacy Rule Format:
            FoundSet(^TRT163, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"')

            Technical Business Rule Format:
            FoundSet(^TRT163, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"')
    
            Data Elements:
    
            ^TRT163 = TRT:RP:BusinessAndProfessionalInformation:BusinessAddressDetails:AddressDetails.LocalityName.Text
            */
            assertion = IsMatch(report.TRT163, @"(^(QLD|NSW|VIC|SA|WA|NT|ACT|TAS)$)|(^(.*)( QLD | NSW | VIC | SA | WA | NT | ACT | TAS )(.*)$)|(^(QLD |NSW |VIC |SA |WA |NT |ACT |TAS )(.*)$)|(^(.*)( QLD| NSW| VIC| SA| WA| NT| ACT| TAS)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000411", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain state code",
                    Location = "/tns:TRT/tns:RP/tns:BusinessAndProfessionalInformation/tns:BusinessAddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433066"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT163", Value = report.TRT163 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.433067
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433067()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.433067
            Suburb/Town/City cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            Contains(^TRT163, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^TRT163, 'UNKNOWN')
    
            Data Elements:
    
            ^TRT163 = TRT:RP:BusinessAndProfessionalInformation:BusinessAddressDetails:AddressDetails.LocalityName.Text
            */
            assertion = (report.TRT163 == null ? false : report.TRT163.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000409", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain ""UNKNOWN""",
                    Location = "/tns:TRT/tns:RP/tns:BusinessAndProfessionalInformation/tns:BusinessAddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433067"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT163", Value = report.TRT163 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.433068
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433068()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.433068
            Suburb/Town/City cannot contain numeric characters
    
            Legacy Rule Format:
            ^TRT316 = FALSE AND ContainsSet(^TRT163, '0-9')

            Technical Business Rule Format:
            ^TRT316 = FALSE AND ContainsSet(^TRT163, '0-9')
    
            Data Elements:
    
            ^TRT163 = TRT:RP:BusinessAndProfessionalInformation:BusinessAddressDetails:AddressDetails.LocalityName.Text
    
            ^TRT316 = TRT:RP:BusinessAndProfessionalInformation:BusinessAddressDetails:AddressDetails.OverseasAddress.Indicator
            */
            assertion = (report.TRT316 == false && IsMatch(report.TRT163, @"[0-9]",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000410", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain numeric characters",
                    Location = "/tns:TRT/tns:RP/tns:BusinessAndProfessionalInformation/tns:BusinessAddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433068"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT316", Value = GetValueOrEmpty(report.TRT316) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT163", Value = report.TRT163 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.433069
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433069()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.433069
            Country code must be present for overseas address
    
            Legacy Rule Format:
            ^TRT316 = TRUE AND ^TRT374 = NULL

            Technical Business Rule Format:
            ^TRT316 = TRUE AND ^TRT374 = NULL
    
            Data Elements:
    
            ^TRT374 = TRT:RP:BusinessAndProfessionalInformation:BusinessAddressDetails:AddressDetails.Country.Code
    
            ^TRT316 = TRT:RP:BusinessAndProfessionalInformation:BusinessAddressDetails:AddressDetails.OverseasAddress.Indicator
            */
            assertion = (report.TRT316 == true && report.TRT374 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410191", Severity = ProcessMessageSeverity.Error,
                    Description = @"Country code must be present for overseas address",
                    LongDescription = @"When the overseas address indicator is 'TRUE', the corresponding Country Code must be supplied",
                    Location = "/tns:TRT/tns:RP/tns:BusinessAndProfessionalInformation/tns:BusinessAddressDetails/tns:CountryC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433069"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT316", Value = GetValueOrEmpty(report.TRT316) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT374", Value = GetValueOrEmpty(report.TRT374) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.433070
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433070()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.433070
            Overseas address indicator must not be 'TRUE' if Country Code is 'au'
    
            Legacy Rule Format:
            ^TRT374 = "au" AND ^TRT316 = TRUE

            Technical Business Rule Format:
            ^TRT374 = 'au' AND ^TRT316 = TRUE
    
            Data Elements:
    
            ^TRT374 = TRT:RP:BusinessAndProfessionalInformation:BusinessAddressDetails:AddressDetails.Country.Code
    
            ^TRT316 = TRT:RP:BusinessAndProfessionalInformation:BusinessAddressDetails:AddressDetails.OverseasAddress.Indicator
            */
            assertion = (report.TRT374 == @"au" && report.TRT316 == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410211", Severity = ProcessMessageSeverity.Error,
                    Description = @"Overseas address indicator must not be 'TRUE' if Country Code is 'au'",
                    LongDescription = @"The overseas address indicator must be 'FALSE' if the Country Code is 'au' - representing Australia. If the address is an overseas address please supply the correct Country Code",
                    Location = "/tns:TRT/tns:RP/tns:BusinessAndProfessionalInformation/tns:BusinessAddressDetails/tns:CountryC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433070"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT374", Value = GetValueOrEmpty(report.TRT374) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT316", Value = GetValueOrEmpty(report.TRT316) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.433071
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433071()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.433071
            If an overseas Country Code is used, the Overseas Address indicator must be 'TRUE'
    
            Legacy Rule Format:
            ^TRT374 <> NULL AND ^TRT374 <> "au" AND ^TRT316 = FALSE

            Technical Business Rule Format:
            ^TRT374 <> NULL AND ^TRT374 <> 'au' AND ^TRT316 = FALSE
    
            Data Elements:
    
            ^TRT374 = TRT:RP:BusinessAndProfessionalInformation:BusinessAddressDetails:AddressDetails.Country.Code
    
            ^TRT316 = TRT:RP:BusinessAndProfessionalInformation:BusinessAddressDetails:AddressDetails.OverseasAddress.Indicator
            */
            assertion = (report.TRT374 != null && report.TRT374 != @"au" && report.TRT316 == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410212", Severity = ProcessMessageSeverity.Error,
                    Description = @"If an overseas Country Code is used, the Overseas Address indicator must be 'TRUE'",
                    LongDescription = @"If the address is an Australian address, the overseas address indicator must be set to 'FALSE' and you do not need to supply a Country Code. If the address is an overseas address, the overseas address indicator must be set to 'TRUE' and you must supply a Country Code other than 'au'",
                    Location = "/tns:TRT/tns:RP/tns:BusinessAndProfessionalInformation/tns:BusinessAddressDetails/tns:CountryC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433071"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT374", Value = GetValueOrEmpty(report.TRT374) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT316", Value = GetValueOrEmpty(report.TRT316) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.433072
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433072()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.433072
            If an Intermediary is not present, Reporting Party declaration statement accepted indicator must be supplied and must be “True”.
    
            Legacy Rule Format:
            ^TRT1086 = NULL AND ^TRT384 <> TRUE

            Technical Business Rule Format:
            ^TRT1086 = NULL AND ^TRT384 <> TRUE
    
            Data Elements:
    
            ^TRT384 = TRT:RP:ReportingPartyDeclaration:Declaration.StatementAccepted.Indicator
    
            ^TRT1086 = TRT:INT
            */
            assertion = (report.IntCollectionExists == false && report.TRT384 != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438064", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declaration statement accepted indicator is required.",
                    LongDescription = @"If an Intermediary is not present, Reporting Party declaration statement accepted indicator must be supplied and must be ""True"".",
                    Location = "/tns:TRT/tns:RP/tns:ReportingPartyDeclaration/tns:StatementAcceptedI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433072"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT384", Value = GetValueOrEmpty(report.TRT384) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.433073
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433073()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.433073
            Signature date must not be later than today
    
            Legacy Rule Format:
            ^TRT386 > Today()

            Technical Business Rule Format:
            ^TRT386 > Today()
    
            Data Elements:
    
            ^TRT386 = TRT:RP:ReportingPartyDeclaration:Declaration.Signature.Date
            */
            assertion = (report.TRT386.GetValueOrDefault() > DateTime.Now.Date);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430255", Severity = ProcessMessageSeverity.Error,
                    Description = @"Signature date must not be later than today",
                    Location = "/tns:TRT/tns:RP/tns:ReportingPartyDeclaration/tns:SignatureD",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433073"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT386", Value = GetValueOrEmptyDate(report.TRT386) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.433074
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433074()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.433074
            If an Intermediary is not present, Reporting Party declaration signature date must be supplied.
    
            Legacy Rule Format:
            ^TRT1086 = NULL AND ^TRT386 = NULL

            Technical Business Rule Format:
            ^TRT1086 = NULL AND ^TRT386 = NULL
    
            Data Elements:
    
            ^TRT386 = TRT:RP:ReportingPartyDeclaration:Declaration.Signature.Date
    
            ^TRT1086 = TRT:INT
            */
            assertion = (report.IntCollectionExists == false && report.TRT386 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438068", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declaration signature date is required.",
                    LongDescription = @"If an Intermediary is not present, Reporting Party declaration signature date must be supplied.",
                    Location = "/tns:TRT/tns:RP/tns:ReportingPartyDeclaration/tns:SignatureD",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433074"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT386", Value = GetValueOrEmptyDate(report.TRT386) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.433075
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433075()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.433075
            If an Intermediary is not present, Reporting Party declaration signatory identifier must be supplied.
    
            Legacy Rule Format:
            ^TRT1086 = NULL AND ^TRT387 = NULL

            Technical Business Rule Format:
            ^TRT1086 = NULL AND ^TRT387 = NULL
    
            Data Elements:
    
            ^TRT387 = TRT:RP:ReportingPartyDeclaration:Declaration.SignatoryIdentifier.Text
    
            ^TRT1086 = TRT:INT
            */
            assertion = (report.IntCollectionExists == false && report.TRT387 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438066", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declaration signatory identifier is required.",
                    LongDescription = @"If an Intermediary is not present, Reporting Party declaration signatory identifier must be supplied.",
                    Location = "/tns:TRT/tns:RP/tns:ReportingPartyDeclaration/tns:SignatoryIdentifierT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433075"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT387", Value = GetValueOrEmpty(report.TRT387) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.433076
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433076()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.433076
            Name field cannot contain 'space hyphen space'.
    
            Legacy Rule Format:
            Contains(^TRT389, ' - ')

            Technical Business Rule Format:
            Contains(^TRT389, ' - ')
    
            Data Elements:
    
            ^TRT389 = TRT:RP:ReportingPartyDeclaration:PersonUnstructuredName.FullName.Text
            */
            assertion = (report.TRT389 == null ? false : report.TRT389.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.408010", Severity = ProcessMessageSeverity.Error,
                    Description = @"Name field cannot contain 'space hyphen space'.",
                    LongDescription = @"Name field cannot contain 'space hyphen space'.",
                    Location = "/tns:TRT/tns:RP/tns:ReportingPartyDeclaration/tns:PersonUnstructuredNameFullNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433076"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT389", Value = report.TRT389 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.433077
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433077()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.433077
            The Tax Agent number has failed the Tax Agent number algorithm check
    
            Legacy Rule Format:
            FailsTANAlgorithm(^TRT2)

            Technical Business Rule Format:
            FailsTANAlgorithm(^TRT2)
    
            Data Elements:
    
            ^TRT2 = TRT:INT:Identifiers.TaxAgentNumber.Identifier
            */
            assertion = FailsTANAlgorithm(report.TRT2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410009", Severity = ProcessMessageSeverity.Error,
                    Description = @"The Tax Agent number has failed the Tax Agent number algorithm check",
                    Location = "/tns:TRT/tns:INT/tns:TaxAgentNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433077"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT2", Value = GetValueOrEmpty(report.TRT2) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.433078
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433078()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.433078
            Invalid ABN
    
            Legacy Rule Format:
            FailsABNAlgorithm(^TRT575)

            Technical Business Rule Format:
            FailsABNAlgorithm(^TRT575)
    
            Data Elements:
    
            ^TRT575 = TRT:INT:Identifiers.AustralianBusinessNumber.Identifier
            */
            assertion = FailsABNAlgorithm(report.TRT575);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid ABN",
                    Location = "/tns:TRT/tns:INT/tns:AustralianBusinessNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433078"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT575", Value = report.TRT575 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.433079
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433079()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.433079
            If an Intermediary is present, Intermediary declaration statement accepted indicator must be supplied and must be “True”.
    
            Legacy Rule Format:
            ^TRT1086 <> NULL AND ^TRT391 <> TRUE

            Technical Business Rule Format:
            ^TRT1086 <> NULL AND ^TRT391 <> TRUE
    
            Data Elements:
    
            ^TRT391 = TRT:INT:Declaration:Declaration.StatementAccepted.Indicator
    
            ^TRT1086 = TRT:INT
            */
            assertion = (report.IntCollectionExists != false && report.TRT391 != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438063", Severity = ProcessMessageSeverity.Error,
                    Description = @"Declaration statement accepted indicator is required.",
                    LongDescription = @"If an Intermediary is present, Intermediary declaration statement accepted indicator must be supplied and must be ""True"".",
                    Location = "/tns:TRT/tns:INT/tns:Declaration/tns:StatementAcceptedI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433079"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT391", Value = GetValueOrEmpty(report.TRT391) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.433080
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433080()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.433080
            Signature date must not be later than today
    
            Legacy Rule Format:
            ^TRT393 > Today()

            Technical Business Rule Format:
            ^TRT393 > Today()
    
            Data Elements:
    
            ^TRT393 = TRT:INT:Declaration:Declaration.Signature.Date
            */
            assertion = (report.TRT393.GetValueOrDefault() > DateTime.Now.Date);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430255", Severity = ProcessMessageSeverity.Error,
                    Description = @"Signature date must not be later than today",
                    Location = "/tns:TRT/tns:INT/tns:Declaration/tns:SignatureD",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433080"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT393", Value = GetValueOrEmptyDate(report.TRT393) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.433081
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433081(TRT2024.InT_Declaration_IntermediaryContact intermediaryContact, int itemIndex3)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.433081
                    Name field cannot contain 'space hyphen space'.
    
                    Legacy Rule Format:
                    Contains(^TRT396, ' - ')
        
                    Technical Business Rule Format:
                    Contains(^TRT396, ' - ')
            
                    Data Elements:
            
                    ^TRT396 = TRT:INT:Declaration:IntermediaryContact:PersonUnstructuredName.FullName.Text
                    */
                    assertion = (intermediaryContact.TRT396 == null ? false : intermediaryContact.TRT396.ToUpperInvariant().Contains(@" - "));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.408010", Severity = ProcessMessageSeverity.Error,
                            Description = @"Name field cannot contain 'space hyphen space'.",
                            LongDescription = @"Name field cannot contain 'space hyphen space'.",
                            Location = "/tns:TRT/tns:INT/tns:Declaration/tns:IntermediaryContactCollection/tns:IntermediaryContact" + OccurrenceIndex(intermediaryContact.OccurrenceIndex) + "/tns:PersonUnstructuredNameFullNameT",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433081"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT396", Value = intermediaryContact.TRT396 });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.433085
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433085()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.433085
            Tax losses carried forward to later income years and the sum of Net Australian income or loss - excluding capital gains, Net capital gain, Attributed foreign income - Listed country, Attributed foreign income - Unlisted country, Net amount of Other assessable foreign source income and Australian franking credits from a New Zealand franking company minus Tax losses deducted cannot be both greater than 0
    
            Legacy Rule Format:
            (^TRT130 + ^TRT133 + ^TRT134 + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144) > 0 AND ^TRT145 > 0

            Technical Business Rule Format:
            (^TRT130 + ^TRT133 + ^TRT134 + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144) > 0 AND ^TRT145 > 0
    
            Data Elements:
    
            ^TRT145 = TRT:RP:BusinessDetails:Income:Tax.Losses.CarriedForward.LaterIncomeYearsTotal.Amount
    
            ^TRT130 = TRT:RP:BusinessDetails:Income:Income.Net.Amount
    
            ^TRT133 = TRT:RP:BusinessDetails:Income:CapitalGains:Income.CapitalGainsNet.Amount
    
            ^TRT134 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:ListedCountry:Income.InternationalDealings.Attributed.Amount
    
            ^TRT136 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:UnlistedCountry:Income.InternationalDealings.Attributed.Amount
    
            ^TRT141 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:Income.Net.Amount
    
            ^TRT143 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
    
            ^TRT144 = TRT:RP:BusinessDetails:Income:TaxLossesDeducted:IncomeTax.Deduction.TaxLossesDeducted.Amount
            */
            assertion = (report.TRT130.GetValueOrDefault() + report.TRT133.GetValueOrDefault() + report.TRT134.GetValueOrDefault() + report.TRT136.GetValueOrDefault() + report.TRT141.GetValueOrDefault() + report.TRT143.GetValueOrDefault() - report.TRT144.GetValueOrDefault() > 0 && report.TRT145.GetValueOrDefault() > 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.433085", Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax losses carried forward to later income years and Total net income or loss can not both be greater than 0",
                    LongDescription = @"Tax losses carried forward to later income years and the sum of Net Australian income or loss - excluding capital gains, Net capital gain, Attributed foreign income - Listed country, Attributed foreign income - Unlisted country, Net amount of Other assessable foreign source income and Australian franking credits from a New Zealand franking company minus Tax losses deducted cannot be both greater than 0",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:TaxLossesCarriedForwardLaterIncomeYearsTotalA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433085"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT130", Value = GetValueOrEmpty(report.TRT130) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT133", Value = GetValueOrEmpty(report.TRT133) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT134", Value = GetValueOrEmpty(report.TRT134) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT136", Value = GetValueOrEmpty(report.TRT136) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT141", Value = GetValueOrEmpty(report.TRT141) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT143", Value = GetValueOrEmpty(report.TRT143) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT144", Value = GetValueOrEmpty(report.TRT144) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT145", Value = GetValueOrEmpty(report.TRT145) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.433086
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433086()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.433086
            When the sum of Share of credit for tax withheld - foreign resident withholding (excluding capital gains), Australian franking credits from a New Zealand company, Share of income - Primary production, Share of income - Non-primary production, Credit for tax withheld where ABN not quoted, Franked distributions, Franking credit, TFN amounts withheld, Share of credit for TFN amounts withheld from payments from closely held trusts, Capital gains, Share of credit for foreign resident capital gains withholding amounts, Attributed foreign income, Other assessable foreign source income, Foreign income tax offset, Share of National rental affordability scheme tax offset, Exploration credits distributed, Share of net small business income, Early stage venture capital limited partnership tax offset and Early stage investor tax offset is greater 0, Assessment calculation code of beneficiary must be provided
    
            Legacy Rule Format:
            AnyOccurrence(^TRT1070, ^TRT214 = NULL AND (^TRT476 + ^TRT217 + ^TRT218 + ^TRT219 + ^TRT220 + ^TRT442 + ^TRT221 + ^TRT222 + ^TRT419 + ^TRT223 + ^TRT481 + ^TRT224 + ^TRT225 + ^TRT226 + ^TRT342 + ^TRT465 + ^TRT479 + ^TRT502 + ^TRT503) > 0)

            Technical Business Rule Format:
            AnyOccurrence(^TRT1070, ^TRT214 = NULL AND (^TRT476 + ^TRT217 + ^TRT218 + ^TRT219 + ^TRT220 + ^TRT442 + ^TRT221 + ^TRT222 + ^TRT419 + ^TRT223 + ^TRT481 + ^TRT224 + ^TRT225 + ^TRT226 + ^TRT342 + ^TRT465 + ^TRT479 + ^TRT502 + ^TRT503) > 0)
    
            Data Elements:
    
            ^TRT214 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.AssessmentCalculation.Code
    
            ^TRT217 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
    
            ^TRT218 = TRT:RP:StatementOfDistributionToBeneficiary:Information:ShareOfIncomePrimaryProduction:Income.BeneficiaryShare.Amount
    
            ^TRT219 = TRT:RP:StatementOfDistributionToBeneficiary:Information:ShareOfIncomeNonPrimaryProduction:Income.BeneficiaryShare.Amount
    
            ^TRT220 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount
    
            ^TRT221 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.FrankingCredits.ReceivedFromAustralianCompanies.Amount
    
            ^TRT222 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldWhereTFNNotQuoted.Amount
    
            ^TRT223 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Income.CapitalGainsNet.Amount
    
            ^TRT224 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.AttributedForeignIncome.Amount
    
            ^TRT225 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Income.InternationalDealings.Net.Amount
    
            ^TRT226 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Income.InternationalDealings.TaxOffset.Amount
    
            ^TRT342 = TRT:RP:StatementOfDistributionToBeneficiary:Information:TaxOffsetClaim.NationalRentalAffordabilitySchemeEntitlement.Amount
    
            ^TRT419 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldFromCloselyHeldTrust.Amount
    
            ^TRT442 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Income.DistributionFranked.Amount
    
            ^TRT465 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.ExplorationCreditsDistributed.Amount
    
            ^TRT476 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromForeignResidents.Amount
    
            ^TRT479 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Income.SmallBusinessEntityNet.Amount
    
            ^TRT481 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.PayAsYouGoWithholding.CreditForCapitalGainsWithheldFromForeignResidents.Amount
    
            ^TRT502 = TRT:RP:StatementOfDistributionToBeneficiary:Information:EarlyStageVentureTaxOffset:TaxOffsetClaim.NonRefundableOther.Amount
    
            ^TRT503 = TRT:RP:StatementOfDistributionToBeneficiary:Information:EarlyStageInvestorTaxOffset:TaxOffsetClaim.NonRefundableOther.Amount
    
            ^TRT1070 = TRT:RP:StatementOfDistributionToBeneficiary
            */
            assertion = (report.RP_StatementOfDistributionToBeneficiaryCollection == null ? false : report.RP_StatementOfDistributionToBeneficiaryCollection.Any(TRT1070Repeat => TRT1070Repeat.TRT214 == null && TRT1070Repeat.TRT476.GetValueOrDefault() + TRT1070Repeat.TRT217.GetValueOrDefault() + TRT1070Repeat.TRT218.GetValueOrDefault() + TRT1070Repeat.TRT219.GetValueOrDefault() + TRT1070Repeat.TRT220.GetValueOrDefault() + TRT1070Repeat.TRT442.GetValueOrDefault() + TRT1070Repeat.TRT221.GetValueOrDefault() + TRT1070Repeat.TRT222.GetValueOrDefault() + TRT1070Repeat.TRT419.GetValueOrDefault() + TRT1070Repeat.TRT223.GetValueOrDefault() + TRT1070Repeat.TRT481.GetValueOrDefault() + TRT1070Repeat.TRT224.GetValueOrDefault() + TRT1070Repeat.TRT225.GetValueOrDefault() + TRT1070Repeat.TRT226.GetValueOrDefault() + TRT1070Repeat.TRT342.GetValueOrDefault() + TRT1070Repeat.TRT465.GetValueOrDefault() + TRT1070Repeat.TRT479.GetValueOrDefault() + TRT1070Repeat.TRT502.GetValueOrDefault() + TRT1070Repeat.TRT503.GetValueOrDefault() > 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.433086", Severity = ProcessMessageSeverity.Error,
                    Description = @"Assessment calculation code of beneficiary must be provided",
                    LongDescription = @"When the sum of Share of credit for tax withheld - foreign resident withholding (excluding capital gains), Australian franking credits from a New Zealand company, Share of income - Primary production, Share of income - Non-primary production, Credit for tax withheld where ABN not quoted, Franked distributions, Franking credit, TFN amounts withheld, Share of credit for TFN amounts withheld from payments from closely held trusts, Capital gains, Share of credit for foreign resident capital gains withholding amounts, Attributed foreign income, Other assessable foreign source income, Foreign income tax offset, Share of National rental affordability scheme tax offset, Exploration credits distributed, Share of net small business income, Early stage venture capital limited partnership tax offset and Early stage investor tax offset is greater 0, Assessment calculation code of beneficiary must be provided",
                    Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary/tns:Information/tns:IncomeTaxAssessmentCalculationC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433086"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT214", Value = "TRT214" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT476", Value = "TRT476" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT217", Value = "TRT217" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT218", Value = "TRT218" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT219", Value = "TRT219" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT220", Value = "TRT220" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT442", Value = "TRT442" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT221", Value = "TRT221" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT222", Value = "TRT222" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT419", Value = "TRT419" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT223", Value = "TRT223" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT481", Value = "TRT481" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT224", Value = "TRT224" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT225", Value = "TRT225" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT226", Value = "TRT226" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT342", Value = "TRT342" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT465", Value = "TRT465" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT479", Value = "TRT479" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT502", Value = "TRT502" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT503", Value = "TRT503" });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.433087
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433087()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.433087
            When the sum of Share of credit for tax withheld, Australian franking credits from a New Zealand company, Primary production share of income, Non-primary production share of income, Credit for tax withheld where ABN not quoted, Franked distributions, Franking credit, TFN amounts withheld, Share of credit for TFN amounts withheld from payments from closely held trusts, Net capital gain, Share of credit for foreign resident capital gains withholding amounts, Attributed foreign income, Other assessable foreign source income, Foreign income tax offsets, Share of National rental affordability scheme tax offset, Amount of capital gains on which the trustee has chosen to be assessed on behalf of beneficiaries, Early stage venture capital limited partnership tax offset - No beneficiary presently entitled, Early stage venture capital limited partnership tax offset carried forward from previous year, Early stage investor tax offset - No beneficiary presently entitled and Early stage investor tax offset carried forward from previous year is greater than 0, Assesment calculation code must be provided
    
            Legacy Rule Format:
            ^TRT230 = NULL AND (^TRT477 + ^TRT233 + ^TRT234 + ^TRT235 + ^TRT236 + ^TRT444 + ^TRT237 + ^TRT238 + ^TRT435 + ^TRT239 + ^TRT482 + ^TRT240 + ^TRT241 + ^TRT242 + ^TRT343 + ^TRT446 + ^TRT504 + ^TRT505 + ^TRT506 + ^TRT507) > 0

            Technical Business Rule Format:
            ^TRT230 = NULL AND (^TRT477 + ^TRT233 + ^TRT234 + ^TRT235 + ^TRT236 + ^TRT444 + ^TRT237 + ^TRT238 + ^TRT435 + ^TRT239 + ^TRT482 + ^TRT240 + ^TRT241 + ^TRT242 + ^TRT343 + ^TRT446 + ^TRT504 + ^TRT505 + ^TRT506 + ^TRT507) > 0
    
            Data Elements:
    
            ^TRT230 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.AssessmentCalculation.Code
    
            ^TRT233 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
    
            ^TRT234 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:PrimaryProductionNoBeneficiaryEntitled:IncomeTax.IncomeOrLossTaxable.Amount
    
            ^TRT235 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:NonPrimaryProductionNoBeneficiaryEntitled:IncomeTax.IncomeOrLossTaxable.Amount
    
            ^TRT236 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount
    
            ^TRT237 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.FrankingCredits.ReceivedFromAustralianCompanies.Amount
    
            ^TRT238 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldWhereTFNNotQuoted.Amount
    
            ^TRT239 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:Income.CapitalGainsNet.Amount
    
            ^TRT240 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.AttributedForeignIncome.Amount
    
            ^TRT241 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:Income.InternationalDealings.Net.Amount
    
            ^TRT242 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:Income.InternationalDealings.TaxOffset.Amount
    
            ^TRT343 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:TaxOffsetClaim.NationalRentalAffordabilitySchemeEntitlement.Amount
    
            ^TRT435 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldFromCloselyHeldTrust.Amount
    
            ^TRT444 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:Income.DistributionFranked.Amount
    
            ^TRT446 = TRT:RP:CapitalGainsAssessed:CapitalGainsTax.TrusteeAssessedCapitalGain.Amount
    
            ^TRT477 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromForeignResidents.Amount
    
            ^TRT482 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.PayAsYouGoWithholding.CreditForCapitalGainsWithheldFromForeignResidents.Amount
    
            ^TRT504 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:EarlyStageVentureNoBeneficiary:TaxOffsetClaim.NonRefundableOther.Amount
    
            ^TRT505 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:EarlyStageVentureNoBeneficiary:EarlyStageVentureNoBeneficiaryCarriedForward:TaxOffsetClaim.NonRefundableOther.Amount
    
            ^TRT506 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:InvestorNoBeneficiary:TaxOffsetClaim.NonRefundableOther.Amount
    
            ^TRT507 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:InvestorNoBeneficiary:InvestorNoBeneficiaryCarriedForward:TaxOffsetClaim.NonRefundableOther.Amount
            */
            assertion = (report.TRT230 == null && report.TRT477.GetValueOrDefault() + report.TRT233.GetValueOrDefault() + report.TRT234.GetValueOrDefault() + report.TRT235.GetValueOrDefault() + report.TRT236.GetValueOrDefault() + report.TRT444.GetValueOrDefault() + report.TRT237.GetValueOrDefault() + report.TRT238.GetValueOrDefault() + report.TRT435.GetValueOrDefault() + report.TRT239.GetValueOrDefault() + report.TRT482.GetValueOrDefault() + report.TRT240.GetValueOrDefault() + report.TRT241.GetValueOrDefault() + report.TRT242.GetValueOrDefault() + report.TRT343.GetValueOrDefault() + report.TRT446.GetValueOrDefault() + report.TRT504.GetValueOrDefault() + report.TRT505.GetValueOrDefault() + report.TRT506.GetValueOrDefault() + report.TRT507.GetValueOrDefault() > 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.433087", Severity = ProcessMessageSeverity.Error,
                    Description = @"Assessment calculation code must be provided",
                    LongDescription = @"When the sum of Share of credit for tax withheld, Australian franking credits from a New Zealand company, Primary production share of income, Non-primary production share of income, Credit for tax withheld where ABN not quoted, Franked distributions, Franking credit, TFN amounts withheld, Share of credit for TFN amounts withheld from payments from closely held trusts, Net capital gain, Share of credit for foreign resident capital gains withholding amounts, Attributed foreign income, Other assessable foreign source income, Foreign income tax offsets, Share of National rental affordability scheme tax offset, Amount of capital gains on which the trustee has chosen to be assessed on behalf of beneficiaries, Early stage venture capital limited partnership tax offset - No beneficiary presently entitled, Early stage venture capital limited partnership tax offset carried forward from previous year, Early stage investor tax offset - No beneficiary presently entitled and Early stage investor tax offset carried forward from previous year is greater than 0, Assesment calculation code must be provided",
                    Location = "/tns:TRT/tns:RP/tns:IncomeNoBeneficiaryPresentlyEntitled/tns:IncomeTaxAssessmentCalculationC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433087"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT230", Value = GetValueOrEmpty(report.TRT230) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT477", Value = GetValueOrEmpty(report.TRT477) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT233", Value = GetValueOrEmpty(report.TRT233) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT234", Value = GetValueOrEmpty(report.TRT234) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT235", Value = GetValueOrEmpty(report.TRT235) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT236", Value = GetValueOrEmpty(report.TRT236) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT444", Value = GetValueOrEmpty(report.TRT444) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT237", Value = GetValueOrEmpty(report.TRT237) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT238", Value = GetValueOrEmpty(report.TRT238) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT435", Value = GetValueOrEmpty(report.TRT435) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT239", Value = GetValueOrEmpty(report.TRT239) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT482", Value = GetValueOrEmpty(report.TRT482) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT240", Value = GetValueOrEmpty(report.TRT240) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT241", Value = GetValueOrEmpty(report.TRT241) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT242", Value = GetValueOrEmpty(report.TRT242) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT343", Value = GetValueOrEmpty(report.TRT343) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT446", Value = GetValueOrEmpty(report.TRT446) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT504", Value = GetValueOrEmpty(report.TRT504) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT505", Value = GetValueOrEmpty(report.TRT505) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT506", Value = GetValueOrEmpty(report.TRT506) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT507", Value = GetValueOrEmpty(report.TRT507) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.433088
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433088()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.433088
            When Family Trust Election Status and Interposed Entity Election Status are not provided, the Assessment calculation code of beneficiary is '35', For each trustee beneficiary indicate whether you will be making a TB Statement is not equal to false, Tax preferred amounts and Untaxed part of share of net income are either not provided or have a value of 0, Type of trust must be a deceased estate' - code '059'
    
            Legacy Rule Format:
            ^TRT32 <> '059' AND ^TRT28 = NULL AND ^TRT30 = NULL AND AnyOccurrence(^TRT1070, ^TRT214 = '35' AND ^TRT344 <> FALSE AND (^TRT345 = NULL OR ^TRT345 = 0) AND (^TRT346 = NULL OR ^TRT346 = 0))

            Technical Business Rule Format:
            ^TRT32 <> '059' AND ^TRT28 = NULL AND ^TRT30 = NULL AND AnyOccurrence(^TRT1070, ^TRT214 = '35' AND ^TRT344 <> FALSE AND (^TRT345 = NULL OR ^TRT345 = 0) AND (^TRT346 = NULL OR ^TRT346 = 0))
    
            Data Elements:
    
            ^TRT32 = TRT:RP:TypeOfTrust:OrganisationDetails.OrganisationType.Code
    
            ^TRT28 = TRT:RP:ElectionStatus:Elections.FamilyTrustElectionStatus.Year
    
            ^TRT30 = TRT:RP:InterposedEntityElectionStatus:Elections.InterposedEntityElectionStatus.Year
    
            ^TRT214 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.AssessmentCalculation.Code
    
            ^TRT344 = TRT:RP:StatementOfDistributionToBeneficiary:Information:TBStatementInformation:IncomeTax.TrustBeneficiaryStatement.Indicator
    
            ^TRT345 = TRT:RP:StatementOfDistributionToBeneficiary:Information:TBStatementInformation:IncomeTax.Superannuation.TrusteeBeneficiaryTaxPreferred.Amount
    
            ^TRT346 = TRT:RP:StatementOfDistributionToBeneficiary:Information:TBStatementInformation:IncomeTax.Superannuation.TrusteeBeneficiaryUntaxedPartShareOfNetIncome.Amount
    
            ^TRT1070 = TRT:RP:StatementOfDistributionToBeneficiary
            */
            assertion = (report.TRT32 != @"059" && report.TRT28 == null && report.TRT30 == null && (report.RP_StatementOfDistributionToBeneficiaryCollection == null ? false : report.RP_StatementOfDistributionToBeneficiaryCollection.Any(TRT1070Repeat => TRT1070Repeat.TRT214 == @"35" && TRT1070Repeat.TRT344 != false && (TRT1070Repeat.TRT345 == null || TRT1070Repeat.TRT345 == 0) && (TRT1070Repeat.TRT346 == null || TRT1070Repeat.TRT346 == 0))));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.433088", Severity = ProcessMessageSeverity.Error,
                    Description = @"Type of trust must be a 'deceased estate' - code '059'",
                    LongDescription = @"When Family Trust Election Status and Interposed Entity Election Status are not provided, the Assessment calculation code of beneficiary is '35', For each trustee beneficiary indicate whether you will be making a TB Statement is not equal to false, Tax preferred amounts and Untaxed part of share of net income are either not provided or have a value of 0, Type of trust must be a deceased estate' - code '059'",
                    Location = "/tns:TRT/tns:RP/tns:TypeOfTrust/tns:OrganisationDetailsOrganisationTypeC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433088"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT32", Value = report.TRT32 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT28", Value = GetValueOrEmpty(report.TRT28) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT30", Value = GetValueOrEmpty(report.TRT30) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT214", Value = "TRT214" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT344", Value = "TRT344" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT345", Value = "TRT345" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT346", Value = "TRT346" });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.433089
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433089()
      {
        ProcessMessageDocument processMessage;
        bool assertion;

            /*  VR.ATO.TRT.433089
            When Date of death is more than 6 years prior to the financial year, Family Trust Election Status, Interposed Entity Election Status are not provided, Assessment calculation code of beneficiary is "14", For each trustee beneficiary indicate whether you will be making a TB Statement is not equal to false and Tax preferred amounts, Untaxed part of share of net income are either not provided or have a value of 0 or both Surname/family name of individual beneficiary and Name of non-individual beneficiary are not provided or TFN of beneficiary is not provided, Type of trust must not be a deceased estate' - code '059'
    
            Legacy Rule Format:
            ^TRT32 = '059' AND ^TRT34 < ConvertToDate(1, 7, ^TRT1 - 6) AND ^TRT28 = NULL AND ^TRT30 = NULL AND AnyOccurrence(^TRT1070, ^TRT214 = '14' AND ^TRT344 <> FALSE AND (((^TRT345 = NULL OR ^TRT345 = 0) AND (^TRT346 = NULL OR ^TRT346 = 0)) OR (^TRT200 = NULL AND ^TRT203 = NULL) OR (^TRT211 = NULL)))

            Technical Business Rule Format:
            ^TRT32 = '059' AND ^TRT34 < ConvertToDate(1, 7, ^TRT1 - 6) AND ^TRT28 = NULL AND ^TRT30 = NULL AND AnyOccurrence(^TRT1070, ^TRT214 = '14' AND ^TRT344 <> FALSE AND (((^TRT345 = NULL OR ^TRT345 = 0) AND (^TRT346 = NULL OR ^TRT346 = 0)) OR (^TRT200 = NULL AND ^TRT203 = NULL) OR (^TRT211 = NULL)))
    
            Data Elements:
    
            ^TRT32 = TRT:RP:TypeOfTrust:OrganisationDetails.OrganisationType.Code
    
            ^TRT1 = TRT:RP:Report.TargetFinancial.Year
    
            ^TRT28 = TRT:RP:ElectionStatus:Elections.FamilyTrustElectionStatus.Year
    
            ^TRT30 = TRT:RP:InterposedEntityElectionStatus:Elections.InterposedEntityElectionStatus.Year
    
            ^TRT34 = TRT:RP:TypeOfTrust:PersonDemographicDetails.Death.Date
    
            ^TRT200 = TRT:RP:StatementOfDistributionToBeneficiary:IndividualNameDetails:PersonNameDetails.FamilyName.Text
    
            ^TRT203 = TRT:RP:StatementOfDistributionToBeneficiary:NonIndividualNameDetails:OrganisationNameDetails.OrganisationalName.Text
    
            ^TRT211 = TRT:RP:StatementOfDistributionToBeneficiary:Identifiers.TaxFileNumber.Identifier
    
            ^TRT214 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.AssessmentCalculation.Code
    
            ^TRT344 = TRT:RP:StatementOfDistributionToBeneficiary:Information:TBStatementInformation:IncomeTax.TrustBeneficiaryStatement.Indicator
    
            ^TRT345 = TRT:RP:StatementOfDistributionToBeneficiary:Information:TBStatementInformation:IncomeTax.Superannuation.TrusteeBeneficiaryTaxPreferred.Amount
    
            ^TRT346 = TRT:RP:StatementOfDistributionToBeneficiary:Information:TBStatementInformation:IncomeTax.Superannuation.TrusteeBeneficiaryUntaxedPartShareOfNetIncome.Amount
    
            ^TRT1070 = TRT:RP:StatementOfDistributionToBeneficiary
            */
            assertion = (report.TRT32 == @"059" && report.TRT34.GetValueOrDefault() < ConvertToDate(1, 7, report.TRT1.GetValueOrDefault() - 6) && report.TRT28 == null && report.TRT30 == null && (report.RP_StatementOfDistributionToBeneficiaryCollection == null ? false : report.RP_StatementOfDistributionToBeneficiaryCollection.Any(TRT1070Repeat => TRT1070Repeat.TRT214 == @"14" && TRT1070Repeat.TRT344 != false && ((TRT1070Repeat.TRT345 == null || TRT1070Repeat.TRT345 == 0) && (TRT1070Repeat.TRT346 == null || TRT1070Repeat.TRT346 == 0) || TRT1070Repeat.TRT200 == null && TRT1070Repeat.TRT203 == null || TRT1070Repeat.TRT211 == null))));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.433089", Severity = ProcessMessageSeverity.Error,
                    Description = @"Type of trust must not be a 'deceased estate' - code '059'",
                    LongDescription = @"When Date of death is more than 6 years prior to the financial year, Family Trust Election Status, Interposed Entity Election Status are not provided, Assessment calculation code of beneficiary is ""14"", For each trustee beneficiary indicate whether you will be making a TB Statement is not equal to false and Tax preferred amounts, Untaxed part of share of net income are either not provided or have a value of 0 or both Surname/family name of individual beneficiary and Name of non-individual beneficiary are not provided or TFN of beneficiary is not provided, Type of trust must not be a deceased estate' - code '059'",
                    Location = "/tns:TRT/tns:RP/tns:TypeOfTrust/tns:OrganisationDetailsOrganisationTypeC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433089"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT32", Value = report.TRT32 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT34", Value = GetValueOrEmptyDate(report.TRT34) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT1", Value = GetValueOrEmpty(report.TRT1) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT28", Value = GetValueOrEmpty(report.TRT28) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT30", Value = GetValueOrEmpty(report.TRT30) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT214", Value = "TRT214" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT344", Value = "TRT344" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT345", Value = "TRT345" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT346", Value = "TRT346" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT200", Value = "TRT200" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT203", Value = "TRT203" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT211", Value = "TRT211" });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.433090
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433090()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.433090
            When Family Trust Election Status and Interposed Entity Election Status are not provided, the Assessment calculation code of beneficiary is '35', For each trustee beneficiary indicate whether you will be making a TB Statement is equal to true and Tax preferred amounts, Untaxed part of share of net income are either not provided or have a value of 0 or both Surname/family name of individual beneficiary and Name of non-individual beneficiary are not provided or TFN of beneficiary is not provided, Type of trust must be a 'deceased estate' - code '059'
    
            Legacy Rule Format:
            ^TRT32 <> '059' AND ^TRT28 = NULL AND ^TRT30 = NULL AND AnyOccurrence(^TRT1070, ^TRT214 = '35' AND ^TRT344 = TRUE AND (((^TRT345 = NULL OR ^TRT345 = 0) AND (^TRT346 = NULL OR ^TRT346 = 0)) OR (^TRT200 = NULL AND ^TRT203 = NULL) OR ^TRT211 = NULL))

            Technical Business Rule Format:
            ^TRT32 <> '059' AND ^TRT28 = NULL AND ^TRT30 = NULL AND AnyOccurrence(^TRT1070, ^TRT214 = '35' AND ^TRT344 = TRUE AND (((^TRT345 = NULL OR ^TRT345 = 0) AND (^TRT346 = NULL OR ^TRT346 = 0)) OR (^TRT200 = NULL AND ^TRT203 = NULL) OR ^TRT211 = NULL))
    
            Data Elements:
    
            ^TRT32 = TRT:RP:TypeOfTrust:OrganisationDetails.OrganisationType.Code
    
            ^TRT28 = TRT:RP:ElectionStatus:Elections.FamilyTrustElectionStatus.Year
    
            ^TRT30 = TRT:RP:InterposedEntityElectionStatus:Elections.InterposedEntityElectionStatus.Year
    
            ^TRT200 = TRT:RP:StatementOfDistributionToBeneficiary:IndividualNameDetails:PersonNameDetails.FamilyName.Text
    
            ^TRT203 = TRT:RP:StatementOfDistributionToBeneficiary:NonIndividualNameDetails:OrganisationNameDetails.OrganisationalName.Text
    
            ^TRT211 = TRT:RP:StatementOfDistributionToBeneficiary:Identifiers.TaxFileNumber.Identifier
    
            ^TRT214 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.AssessmentCalculation.Code
    
            ^TRT344 = TRT:RP:StatementOfDistributionToBeneficiary:Information:TBStatementInformation:IncomeTax.TrustBeneficiaryStatement.Indicator
    
            ^TRT345 = TRT:RP:StatementOfDistributionToBeneficiary:Information:TBStatementInformation:IncomeTax.Superannuation.TrusteeBeneficiaryTaxPreferred.Amount
    
            ^TRT346 = TRT:RP:StatementOfDistributionToBeneficiary:Information:TBStatementInformation:IncomeTax.Superannuation.TrusteeBeneficiaryUntaxedPartShareOfNetIncome.Amount
    
            ^TRT1070 = TRT:RP:StatementOfDistributionToBeneficiary
            */
            assertion = (report.TRT32 != @"059" && report.TRT28 == null && report.TRT30 == null && (report.RP_StatementOfDistributionToBeneficiaryCollection == null ? false : report.RP_StatementOfDistributionToBeneficiaryCollection.Any(TRT1070Repeat => TRT1070Repeat.TRT214 == @"35" && TRT1070Repeat.TRT344 == true && ((TRT1070Repeat.TRT345 == null || TRT1070Repeat.TRT345 == 0) && (TRT1070Repeat.TRT346 == null || TRT1070Repeat.TRT346 == 0) || TRT1070Repeat.TRT200 == null && TRT1070Repeat.TRT203 == null || TRT1070Repeat.TRT211 == null))));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.433090", Severity = ProcessMessageSeverity.Error,
                    Description = @"Type of trust must be a 'deceased estate' - code '059'",
                    LongDescription = @"When Family Trust Election Status and Interposed Entity Election Status are not provided, the Assessment calculation code of beneficiary is '35', For each trustee beneficiary indicate whether you will be making a TB Statement is equal to true and Tax preferred amounts, Untaxed part of share of net income are either not provided or have a value of 0 or both Surname/family name of individual beneficiary and Name of non-individual beneficiary are not provided or TFN of beneficiary is not provided, Type of trust must be a 'deceased estate' - code '059'",
                    Location = "/tns:TRT/tns:RP/tns:TypeOfTrust/tns:OrganisationDetailsOrganisationTypeC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433090"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT32", Value = report.TRT32 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT28", Value = GetValueOrEmpty(report.TRT28) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT30", Value = GetValueOrEmpty(report.TRT30) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT214", Value = "TRT214" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT344", Value = "TRT344" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT345", Value = "TRT345" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT346", Value = "TRT346" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT200", Value = "TRT200" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT203", Value = "TRT203" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT211", Value = "TRT211" });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.433091
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433091()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.433091
            When Date of death is more than 6 years prior to the financial year, Family Trust Election Status and Interposed Entity Election Status are not provided, the Assessment calculation code of beneficiary is '14', For each trustee beneficiary indicate whether you will be making a TB Statement is equal to true and Tax preferred amounts, Untaxed part of share of net income are either not provided or have a value of 0 or both Surname/family name of individual beneficiary and Name of non-individual beneficiary are not provided or TFN of beneficiary is not provided, Type of trust must not be a 'deceased estate' - code '059'
    
            Legacy Rule Format:
            ^TRT32 = '059' AND ^TRT34 <> NULL AND ^TRT34 < ConvertToDate(1, 7, ^TRT1 - 6) AND ^TRT28 = NULL AND ^TRT30 = NULL AND AnyOccurrence(^TRT1070, ^TRT214 = '14' AND ^TRT344 = TRUE AND (((^TRT345 = NULL OR ^TRT345 = 0) AND (^TRT346 = NULL OR ^TRT346 = 0)) OR (^TRT200 = NULL AND ^TRT203 = NULL) OR ^TRT211 = NULL))

            Technical Business Rule Format:
            ^TRT32 = '059' AND ^TRT34 <> NULL AND ^TRT34 < ConvertToDate(1, 7, ^TRT1 - 6) AND ^TRT28 = NULL AND ^TRT30 = NULL AND AnyOccurrence(^TRT1070, ^TRT214 = '14' AND ^TRT344 = TRUE AND (((^TRT345 = NULL OR ^TRT345 = 0) AND (^TRT346 = NULL OR ^TRT346 = 0)) OR (^TRT200 = NULL AND ^TRT203 = NULL) OR ^TRT211 = NULL))
    
            Data Elements:
    
            ^TRT32 = TRT:RP:TypeOfTrust:OrganisationDetails.OrganisationType.Code
    
            ^TRT1 = TRT:RP:Report.TargetFinancial.Year
    
            ^TRT28 = TRT:RP:ElectionStatus:Elections.FamilyTrustElectionStatus.Year
    
            ^TRT30 = TRT:RP:InterposedEntityElectionStatus:Elections.InterposedEntityElectionStatus.Year
    
            ^TRT34 = TRT:RP:TypeOfTrust:PersonDemographicDetails.Death.Date
    
            ^TRT200 = TRT:RP:StatementOfDistributionToBeneficiary:IndividualNameDetails:PersonNameDetails.FamilyName.Text
    
            ^TRT203 = TRT:RP:StatementOfDistributionToBeneficiary:NonIndividualNameDetails:OrganisationNameDetails.OrganisationalName.Text
    
            ^TRT211 = TRT:RP:StatementOfDistributionToBeneficiary:Identifiers.TaxFileNumber.Identifier
    
            ^TRT214 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.AssessmentCalculation.Code
    
            ^TRT344 = TRT:RP:StatementOfDistributionToBeneficiary:Information:TBStatementInformation:IncomeTax.TrustBeneficiaryStatement.Indicator
    
            ^TRT345 = TRT:RP:StatementOfDistributionToBeneficiary:Information:TBStatementInformation:IncomeTax.Superannuation.TrusteeBeneficiaryTaxPreferred.Amount
    
            ^TRT346 = TRT:RP:StatementOfDistributionToBeneficiary:Information:TBStatementInformation:IncomeTax.Superannuation.TrusteeBeneficiaryUntaxedPartShareOfNetIncome.Amount
    
            ^TRT1070 = TRT:RP:StatementOfDistributionToBeneficiary
            */
            assertion = (report.TRT32 == @"059" && report.TRT34 != null && report.TRT34.GetValueOrDefault() < ConvertToDate(1, 7, report.TRT1.GetValueOrDefault() - 6) && report.TRT28 == null && report.TRT30 == null && (report.RP_StatementOfDistributionToBeneficiaryCollection == null ? false : report.RP_StatementOfDistributionToBeneficiaryCollection.Any(TRT1070Repeat => TRT1070Repeat.TRT214 == @"14" && TRT1070Repeat.TRT344 == true && ((TRT1070Repeat.TRT345 == null || TRT1070Repeat.TRT345 == 0) && (TRT1070Repeat.TRT346 == null || TRT1070Repeat.TRT346 == 0) || TRT1070Repeat.TRT200 == null && TRT1070Repeat.TRT203 == null || TRT1070Repeat.TRT211 == null))));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.433091", Severity = ProcessMessageSeverity.Error,
                    Description = @"Type of trust must not be a 'deceased estate' - code '059'",
                    LongDescription = @"When Date of death is more than 6 years prior to the financial year, Family Trust Election Status and Interposed Entity Election Status are not provided, the Assessment calculation code of beneficiary is '14', For each trustee beneficiary indicate whether you will be making a TB Statement is equal to true and Tax preferred amounts, Untaxed part of share of net income are either not provided or have a value of 0 or both Surname/family name of individual beneficiary and Name of non-individual beneficiary are not provided or TFN of beneficiary is not provided, Type of trust must not be a 'deceased estate' - code '059'",
                    Location = "/tns:TRT/tns:RP/tns:TypeOfTrust/tns:OrganisationDetailsOrganisationTypeC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433091"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT32", Value = report.TRT32 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT34", Value = GetValueOrEmptyDate(report.TRT34) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT1", Value = GetValueOrEmpty(report.TRT1) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT28", Value = GetValueOrEmpty(report.TRT28) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT30", Value = GetValueOrEmpty(report.TRT30) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT214", Value = "TRT214" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT344", Value = "TRT344" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT345", Value = "TRT345" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT346", Value = "TRT346" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT200", Value = "TRT200" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT203", Value = "TRT203" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT211", Value = "TRT211" });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.433093
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433093()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.433093
            When the type of trust is '232' for Testamentary trust and the sum of Net Australian income or loss, Net capital gain, Attributed foreign income - Listed country, Attributed foreign income - Unlisted country, Net amount of Other assessable foreign source income, Australian franking credits from a New Zealand franking company minus Tax losses deducted is greater than 0, the Assessment calculation code of beneficiary must not be 11, 12, 13, 14, 45, 111, 118, 119, 120 or 145
    
            Legacy Rule Format:
            ^TRT32 = '232' AND (^TRT130 + ^TRT133 + ^TRT134 + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144) > 0 AND AnyOccurrence(^TRT1070, InSet(^TRT214, '"11", "12", "13", "14", "45", "111", "118", "119", "120", "145"'))

            Technical Business Rule Format:
            ^TRT32 = '232' AND (^TRT130 + ^TRT133 + ^TRT134 + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144) > 0 AND AnyOccurrence(^TRT1070, InSet(^TRT214, '"11", "12", "13", "14", "45", "111", "118", "119", "120", "145"'))
    
            Data Elements:
    
            ^TRT214 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.AssessmentCalculation.Code
    
            ^TRT32 = TRT:RP:TypeOfTrust:OrganisationDetails.OrganisationType.Code
    
            ^TRT130 = TRT:RP:BusinessDetails:Income:Income.Net.Amount
    
            ^TRT133 = TRT:RP:BusinessDetails:Income:CapitalGains:Income.CapitalGainsNet.Amount
    
            ^TRT134 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:ListedCountry:Income.InternationalDealings.Attributed.Amount
    
            ^TRT136 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:UnlistedCountry:Income.InternationalDealings.Attributed.Amount
    
            ^TRT141 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:Income.Net.Amount
    
            ^TRT143 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
    
            ^TRT144 = TRT:RP:BusinessDetails:Income:TaxLossesDeducted:IncomeTax.Deduction.TaxLossesDeducted.Amount
    
            ^TRT1070 = TRT:RP:StatementOfDistributionToBeneficiary
            */
            assertion = (report.TRT32 == @"232" && report.TRT130.GetValueOrDefault() + report.TRT133.GetValueOrDefault() + report.TRT134.GetValueOrDefault() + report.TRT136.GetValueOrDefault() + report.TRT141.GetValueOrDefault() + report.TRT143.GetValueOrDefault() - report.TRT144.GetValueOrDefault() > 0 && (report.RP_StatementOfDistributionToBeneficiaryCollection == null ? false : report.RP_StatementOfDistributionToBeneficiaryCollection.Any(TRT1070Repeat => IsMatch(TRT1070Repeat.TRT214, @"^(11|12|13|14|45|111|118|119|120|145)$"))));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.433093", Severity = ProcessMessageSeverity.Error,
                    Description = @"Assessment calculation code of beneficiary is incorrect",
                    LongDescription = @"When the type of trust is '232' for Testamentary trust and the sum of Net Australian income or loss, Net capital gain, Attributed foreign income - Listed country, Attributed foreign income - Unlisted country, Net amount of Other assessable foreign source income, Australian franking credits from a New Zealand franking company minus Tax losses deducted is greater than 0, the Assessment calculation code of beneficiary must not be 11, 12, 13, 14, 45, 111, 118, 119, 120 or 145",
                    Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary/tns:Information/tns:IncomeTaxAssessmentCalculationC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433093"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT32", Value = report.TRT32 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT130", Value = GetValueOrEmpty(report.TRT130) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT133", Value = GetValueOrEmpty(report.TRT133) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT134", Value = GetValueOrEmpty(report.TRT134) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT136", Value = GetValueOrEmpty(report.TRT136) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT141", Value = GetValueOrEmpty(report.TRT141) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT143", Value = GetValueOrEmpty(report.TRT143) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT144", Value = GetValueOrEmpty(report.TRT144) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT214", Value = "Count(TRT214)" });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.433094
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433094()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.433094
            BSB Number must be greater than 012000
    
            Legacy Rule Format:
            (^TRT38 <> NULL) AND (^TRT38 < 012001 OR ^TRT38 > 999999)

            Technical Business Rule Format:
            (^TRT38 <> NULL) AND (^TRT38 < 012001 OR ^TRT38 > 999999)
    
            Data Elements:
    
            ^TRT38 = TRT:RP:FinancialInstitutionAccount:FinancialInstitutionAccount.BankStateBranch.Number
            */
            assertion = (report.TRT38 != null && (AsNumeric(report.TRT38) < 12001 || AsNumeric(report.TRT38) > 999999));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500136", Severity = ProcessMessageSeverity.Error,
                    Description = @"BSB Number must be greater than 012000.",
                    Location = "/tns:TRT/tns:RP/tns:FinancialInstitutionAccount/tns:BankStateBranchN",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433094"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT38", Value = report.TRT38 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.433097
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433097()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.433097
            When the sum of Australian franking credits from a New Zealand company - No beneficiary presently entitled, Share of income - Primary production - No beneficiary presently entitled, Share of income - Non-primary production - No beneficiary presently entitled, Franked distributions - No beneficiary presently entitled, Capital gains - No beneficiary presently entitled, Attributed foreign income - No beneficiary presently entitled, and Other assessable foreign source income - No beneficiary presently entitled is less than 0, then Amount of capital gains on which the trustee has chosen to be assessed on behalf of beneficiaries must be provided
    
            Legacy Rule Format:
            (^TRT233+ ^TRT234 + ^TRT235 + ^TRT444 + ^TRT239 + ^TRT240 + ^TRT241) < 0 AND ^TRT446 = NULL

            Technical Business Rule Format:
            (^TRT233+ ^TRT234 + ^TRT235 + ^TRT444 + ^TRT239 + ^TRT240 + ^TRT241) < 0 AND ^TRT446 = NULL
    
            Data Elements:
    
            ^TRT446 = TRT:RP:CapitalGainsAssessed:CapitalGainsTax.TrusteeAssessedCapitalGain.Amount
    
            ^TRT233 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
    
            ^TRT234 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:PrimaryProductionNoBeneficiaryEntitled:IncomeTax.IncomeOrLossTaxable.Amount
    
            ^TRT235 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:NonPrimaryProductionNoBeneficiaryEntitled:IncomeTax.IncomeOrLossTaxable.Amount
    
            ^TRT239 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:Income.CapitalGainsNet.Amount
    
            ^TRT240 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.AttributedForeignIncome.Amount
    
            ^TRT241 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:Income.InternationalDealings.Net.Amount
    
            ^TRT444 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:Income.DistributionFranked.Amount
            */
            assertion = (report.TRT233.GetValueOrDefault() + report.TRT234.GetValueOrDefault() + report.TRT235.GetValueOrDefault() + report.TRT444.GetValueOrDefault() + report.TRT239.GetValueOrDefault() + report.TRT240.GetValueOrDefault() + report.TRT241.GetValueOrDefault() < 0 && report.TRT446 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.433097", Severity = ProcessMessageSeverity.Error,
                    Description = @"Amount of capital gains on which the trustee has chosen to be assessed on behalf of beneficiaries must be provided",
                    LongDescription = @"When the sum of Australian franking credits from a New Zealand company - No beneficiary presently entitled, Share of income - Primary production - No beneficiary presently entitled, Share of income - Non-primary production - No beneficiary presently entitled, Franked distributions - No beneficiary presently entitled, Capital gains - No beneficiary presently entitled, Attributed foreign income - No beneficiary presently entitled, and Other assessable foreign source income - No beneficiary presently entitled is less than 0, then Amount of capital gains on which the trustee has chosen to be assessed on behalf of beneficiaries must be provided",
                    Location = "/tns:TRT/tns:RP/tns:CapitalGainsAssessed/tns:CapitalGainsTaxTrusteeAssessedCapitalGainA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433097"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT233", Value = GetValueOrEmpty(report.TRT233) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT234", Value = GetValueOrEmpty(report.TRT234) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT235", Value = GetValueOrEmpty(report.TRT235) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT444", Value = GetValueOrEmpty(report.TRT444) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT239", Value = GetValueOrEmpty(report.TRT239) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT240", Value = GetValueOrEmpty(report.TRT240) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT241", Value = GetValueOrEmpty(report.TRT241) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT446", Value = GetValueOrEmpty(report.TRT446) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.433101
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433101(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.433101
                    When the Assessment calculation code of beneficiary is; 11, 25, 26, 27, 28, or 29, Date of birth of beneficiary must not be more recent than the Reporting period end date
    
                    Legacy Rule Format:
                    InSet(^TRT214, '"11", "25", "26", "27", "28", "29"') AND (^TRT213 > ^TRT574)
        
                    Technical Business Rule Format:
                    InSet(^TRT214, '"11", "25", "26", "27", "28", "29"') AND (^TRT213 > ^TRT574)
            
                    Data Elements:
            
                    ^TRT214 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.AssessmentCalculation.Code
            
                    ^TRT213 = TRT:RP:StatementOfDistributionToBeneficiary:PersonDemographicDetails.Birth.Date
            
                    ^TRT574 = TRT:RP:Period.End.Date
                    */
                    assertion = (IsMatch(statementOfDistributionToBeneficiary.TRT214, @"^(11|25|26|27|28|29)$") && statementOfDistributionToBeneficiary.TRT213.GetValueOrDefault() > report.TRT574.GetValueOrDefault());
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.TRT.433101", Severity = ProcessMessageSeverity.Error,
                            Description = @"Date of birth of beneficiary must not be more recent than the Reporting period end date",
                            LongDescription = @"When the Assessment calculation code of beneficiary is; 11, 25, 26, 27, 28, or 29, Date of birth of beneficiary must not be more recent than the Reporting period end date",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:Information/tns:IncomeTaxAssessmentCalculationC",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433101"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT214", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT214) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT213", Value = GetValueOrEmptyDate(statementOfDistributionToBeneficiary.TRT213) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT574", Value = GetValueOrEmptyDate(report.TRT574) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.433102
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433102(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.433102
                    When the Assessment calculation code of beneficiary is '140' and Type of trust - code is not '059' and Family Trust Election Status - Specified Income Year isn't provided and Interposed Entity Election Status - Earliest income year specified isn't provided and For each trustee beneficiary indicate whether you will be making a TB Statement - TB statement information is not false, then Tax preferred amounts - TB statement information or Untaxed part of share of net income - TB statement information must be greater than 0 and the beneficiary address must be provided
    
                    Legacy Rule Format:
                    ^TRT214 = '140' AND ^TRT32 <> '059' AND ^TRT28 = NULL AND ^TRT30 = NULL AND  ^TRT344 <> FALSE AND (((^TRT345 = NULL OR ^TRT345 = 0) AND (^TRT346 = NULL OR ^TRT346 = 0)) OR (^TRT382 = NULL))
        
                    Technical Business Rule Format:
                    ^TRT214 = '140' AND ^TRT32 <> '059' AND ^TRT28 = NULL AND ^TRT30 = NULL AND  ^TRT344 <> FALSE AND (((^TRT345 = NULL OR ^TRT345 = 0) AND (^TRT346 = NULL OR ^TRT346 = 0)) OR (^TRT382 = NULL))
            
                    Data Elements:
            
                    ^TRT214 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.AssessmentCalculation.Code
            
                    ^TRT28 = TRT:RP:ElectionStatus:Elections.FamilyTrustElectionStatus.Year
            
                    ^TRT30 = TRT:RP:InterposedEntityElectionStatus:Elections.InterposedEntityElectionStatus.Year
            
                    ^TRT32 = TRT:RP:TypeOfTrust:OrganisationDetails.OrganisationType.Code
            
                    ^TRT344 = TRT:RP:StatementOfDistributionToBeneficiary:Information:TBStatementInformation:IncomeTax.TrustBeneficiaryStatement.Indicator
            
                    ^TRT345 = TRT:RP:StatementOfDistributionToBeneficiary:Information:TBStatementInformation:IncomeTax.Superannuation.TrusteeBeneficiaryTaxPreferred.Amount
            
                    ^TRT346 = TRT:RP:StatementOfDistributionToBeneficiary:Information:TBStatementInformation:IncomeTax.Superannuation.TrusteeBeneficiaryUntaxedPartShareOfNetIncome.Amount
            
                    ^TRT382 = TRT:RP:StatementOfDistributionToBeneficiary:BeneficiaryAddressDetails:AddressDetails.Country.Code
                    */
                    assertion = (statementOfDistributionToBeneficiary.TRT214 == @"140" && report.TRT32 != @"059" && report.TRT28 == null && report.TRT30 == null && statementOfDistributionToBeneficiary.TRT344 != false && ((statementOfDistributionToBeneficiary.TRT345 == null || statementOfDistributionToBeneficiary.TRT345 == 0) && (statementOfDistributionToBeneficiary.TRT346 == null || statementOfDistributionToBeneficiary.TRT346 == 0) || statementOfDistributionToBeneficiary.TRT382 == null));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.TRT.433102", Severity = ProcessMessageSeverity.Error,
                            Description = @"The beneficiary address and at least one positive TB statement information amount must be provided",
                            LongDescription = @"When the Assessment calculation code of beneficiary is '140' and Type of trust - code is not '059' and Family Trust Election Status - Specified Income Year isn't provided and Interposed Entity Election Status - Earliest income year specified isn't provided and For each trustee beneficiary indicate whether you will be making a TB Statement - TB statement information is not false, then Tax preferred amounts - TB statement information or Untaxed part of share of net income - TB statement information must be greater than 0 and the beneficiary address must be provided",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:Information/tns:IncomeTaxAssessmentCalculationC",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433102"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT214", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT214) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT32", Value = report.TRT32 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT28", Value = GetValueOrEmpty(report.TRT28) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT30", Value = GetValueOrEmpty(report.TRT30) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT344", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT344) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT345", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT345) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT346", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT346) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT382", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT382) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.433103
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433103(TRT2024.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary, int itemIndex2)
      {
        ProcessMessageDocument processMessage;
        bool assertion;
            
                    /*  VR.ATO.TRT.433103
                    When the Assessment calculation code of beneficiary is '120' and Type of trust - code is '059' and Date of death is 6 years prior to the financial year and Family Trust Election Status - Specified Income Year isn't provided and Interposed Entity Election Status - Earliest income year specified isn't provided and For each trustee beneficiary indicate whether you will be making a TB Statement - TB statement information is not false, then Tax preferred amounts - TB statement information or Untaxed part of share of net income - TB statement information must be greater than 0 and the beneficiary address must be provided
    
                    Legacy Rule Format:
                    ^TRT214 = '120' AND ^TRT32 = '059' AND ^TRT34 < ConvertToDate(1, 7, ^TRT1 - 6) AND ^TRT28 = NULL AND ^TRT30 = NULL AND ^TRT344 <> FALSE AND (((^TRT345 = NULL OR ^TRT345 = 0) AND (^TRT346 = NULL OR ^TRT346 = 0)) OR (^TRT382 = NULL))
        
                    Technical Business Rule Format:
                    ^TRT214 = '120' AND ^TRT32 = '059' AND ^TRT34 < ConvertToDate(1, 7, ^TRT1 - 6) AND ^TRT28 = NULL AND ^TRT30 = NULL AND ^TRT344 <> FALSE AND (((^TRT345 = NULL OR ^TRT345 = 0) AND (^TRT346 = NULL OR ^TRT346 = 0)) OR (^TRT382 = NULL))
            
                    Data Elements:
            
                    ^TRT214 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.AssessmentCalculation.Code
            
                    ^TRT1 = TRT:RP:Report.TargetFinancial.Year
            
                    ^TRT28 = TRT:RP:ElectionStatus:Elections.FamilyTrustElectionStatus.Year
            
                    ^TRT30 = TRT:RP:InterposedEntityElectionStatus:Elections.InterposedEntityElectionStatus.Year
            
                    ^TRT32 = TRT:RP:TypeOfTrust:OrganisationDetails.OrganisationType.Code
            
                    ^TRT34 = TRT:RP:TypeOfTrust:PersonDemographicDetails.Death.Date
            
                    ^TRT344 = TRT:RP:StatementOfDistributionToBeneficiary:Information:TBStatementInformation:IncomeTax.TrustBeneficiaryStatement.Indicator
            
                    ^TRT345 = TRT:RP:StatementOfDistributionToBeneficiary:Information:TBStatementInformation:IncomeTax.Superannuation.TrusteeBeneficiaryTaxPreferred.Amount
            
                    ^TRT346 = TRT:RP:StatementOfDistributionToBeneficiary:Information:TBStatementInformation:IncomeTax.Superannuation.TrusteeBeneficiaryUntaxedPartShareOfNetIncome.Amount
            
                    ^TRT382 = TRT:RP:StatementOfDistributionToBeneficiary:BeneficiaryAddressDetails:AddressDetails.Country.Code
                    */
                    assertion = (statementOfDistributionToBeneficiary.TRT214 == @"120" && report.TRT32 == @"059" && report.TRT34.GetValueOrDefault() < ConvertToDate(1, 7, report.TRT1.GetValueOrDefault() - 6) && report.TRT28 == null && report.TRT30 == null && statementOfDistributionToBeneficiary.TRT344 != false && ((statementOfDistributionToBeneficiary.TRT345 == null || statementOfDistributionToBeneficiary.TRT345 == 0) && (statementOfDistributionToBeneficiary.TRT346 == null || statementOfDistributionToBeneficiary.TRT346 == 0) || statementOfDistributionToBeneficiary.TRT382 == null));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.TRT.433103", Severity = ProcessMessageSeverity.Error,
                            Description = @"The beneficiary address and at least one positive TB statement information amount must be provided",
                            LongDescription = @"When the Assessment calculation code of beneficiary is '120' and Type of trust - code is '059' and Date of death is 6 years prior to the financial year and Family Trust Election Status - Specified Income Year isn't provided and Interposed Entity Election Status - Earliest income year specified isn't provided and For each trustee beneficiary indicate whether you will be making a TB Statement - TB statement information is not false, then Tax preferred amounts - TB statement information or Untaxed part of share of net income - TB statement information must be greater than 0 and the beneficiary address must be provided",
                            Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary" + OccurrenceIndex(statementOfDistributionToBeneficiary.OccurrenceIndex) + "/tns:Information/tns:IncomeTaxAssessmentCalculationC",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433103"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT214", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT214) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT32", Value = report.TRT32 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT34", Value = GetValueOrEmptyDate(report.TRT34) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT1", Value = GetValueOrEmpty(report.TRT1) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT28", Value = GetValueOrEmpty(report.TRT28) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT30", Value = GetValueOrEmpty(report.TRT30) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT344", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT344) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT345", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT345) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT346", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT346) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "TRT382", Value = GetValueOrEmpty(statementOfDistributionToBeneficiary.TRT382) });
            
                        response.Add(processMessage);
                    }
              }
              #endregion 
    
      #region VR.ATO.TRT.433104
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433104()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.433104
            When Net amount of Other assessable foreign source income, excluding income shown at Attributed foreign income is greater than 0, then the sum of Foreign income tax offset, Non-primary production - share of net income from trusts less net capital gain, foreign income and franked distributions, Franked distributions from trusts, Net capital gain, Attributed foreign income - Listed country, Attributed foreign income - Unlisted country, Net amount of Other assessable foreign source income, excluding income shown at Attributed foreign income, and Australian franking credits from a New Zealand franking company - Other assessable foreign source income, excluding income shown at Attributed foreign income must be greater than or equal to 0
    
            Legacy Rule Format:
            ^TRT141 > 0 AND (^TRT142 + ^TRT81 + ^TRT437 + ^TRT133 + ^TRT134 + ^TRT136 + ^TRT141 + ^TRT143) < 0

            Technical Business Rule Format:
            ^TRT141 > 0 AND (^TRT142 + ^TRT81 + ^TRT437 + ^TRT133 + ^TRT134 + ^TRT136 + ^TRT141 + ^TRT143) < 0
    
            Data Elements:
    
            ^TRT141 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:Income.Net.Amount
    
            ^TRT81 = TRT:RP:BusinessDetails:Income:PartnershipTrustNonPrimaryProduction:Income.TrustShareNetExcludeNetCapitalGainsAndForeignIncomeAndDistributionFranked.Amount
    
            ^TRT133 = TRT:RP:BusinessDetails:Income:CapitalGains:Income.CapitalGainsNet.Amount
    
            ^TRT134 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:ListedCountry:Income.InternationalDealings.Attributed.Amount
    
            ^TRT136 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:UnlistedCountry:Income.InternationalDealings.Attributed.Amount
    
            ^TRT142 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:Income.InternationalDealings.TaxOffset.Amount
    
            ^TRT143 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
    
            ^TRT437 = TRT:RP:BusinessDetails:Income:PartnershipTrustNonPrimaryProduction:Income.TrustDistributionFranked.Amount
            */
            assertion = (report.TRT141.GetValueOrDefault() > 0 && report.TRT142.GetValueOrDefault() + report.TRT81.GetValueOrDefault() + report.TRT437.GetValueOrDefault() + report.TRT133.GetValueOrDefault() + report.TRT134.GetValueOrDefault() + report.TRT136.GetValueOrDefault() + report.TRT141.GetValueOrDefault() + report.TRT143.GetValueOrDefault() < 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.433104", Severity = ProcessMessageSeverity.Error,
                    Description = @"Other assessable foreign source income is incorrect",
                    LongDescription = @"When Net amount of Other assessable foreign source income, excluding income shown at Attributed foreign income is greater than 0, then the sum of Foreign income tax offset, Non-primary production - share of net income from trusts less net capital gain, foreign income and franked distributions, Franked distributions from trusts, Net capital gain, Attributed foreign income - Listed country, Attributed foreign income - Unlisted country, Net amount of Other assessable foreign source income, excluding income shown at Attributed foreign income, and Australian franking credits from a New Zealand franking company - Other assessable foreign source income, excluding income shown at Attributed foreign income must be greater than or equal to 0",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:AttributedForeignIncome/tns:OtherAssessableForeignIncome/tns:NetA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433104"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT141", Value = GetValueOrEmpty(report.TRT141) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT142", Value = GetValueOrEmpty(report.TRT142) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT81", Value = GetValueOrEmpty(report.TRT81) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT437", Value = GetValueOrEmpty(report.TRT437) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT133", Value = GetValueOrEmpty(report.TRT133) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT134", Value = GetValueOrEmpty(report.TRT134) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT136", Value = GetValueOrEmpty(report.TRT136) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT143", Value = GetValueOrEmpty(report.TRT143) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.433108
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433108()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.433108
            Date of death must not be after the Report period end date
    
            Legacy Rule Format:
            ^TRT34 <> NULL AND ^TRT34 > ^TRT574

            Technical Business Rule Format:
            ^TRT34 <> NULL AND ^TRT34 > ^TRT574
    
            Data Elements:
    
            ^TRT34 = TRT:RP:TypeOfTrust:PersonDemographicDetails.Death.Date
    
            ^TRT574 = TRT:RP:Period.End.Date
            */
            assertion = (report.TRT34 != null && report.TRT34.GetValueOrDefault() > report.TRT574.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.433108", Severity = ProcessMessageSeverity.Error,
                    Description = @"Date of death must not be after the Report period end date",
                    Location = "/tns:TRT/tns:RP/tns:TypeOfTrust/tns:PersonDemographicDetailsDeathD",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433108"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT34", Value = GetValueOrEmptyDate(report.TRT34) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT574", Value = GetValueOrEmptyDate(report.TRT574) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.433109
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433109()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.433109
            When the Type of trust - code is 203, then there can be no Income No Beneficiary Presently Entitled
    
            Legacy Rule Format:
            ^TRT32 = '203' AND Count(^TRT1079) > 0

            Technical Business Rule Format:
            ^TRT32 = '203' AND Count(^TRT1079) > 0
    
            Data Elements:
    
            ^TRT32 = TRT:RP:TypeOfTrust:OrganisationDetails.OrganisationType.Code
    
            ^TRT1079 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled
            */
            assertion = (report.TRT32 == @"203" && Count(report.RP_IncomeNoBeneficiaryPresentlyEntitledCollectionCount) > 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.433109", Severity = ProcessMessageSeverity.Error,
                    Description = @"When the Type of trust - code is 203, then there can be no Income No Beneficiary Presently Entitled",
                    Location = "/tns:TRT/tns:RP/tns:TypeOfTrust/tns:OrganisationDetailsOrganisationTypeC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433109"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT32", Value = report.TRT32 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.433110
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT433110()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.433110
            Intermediary Declaration Name usage code must be unique
    
            Legacy Rule Format:
            HasDuplicateValues(^TRT395)

            Technical Business Rule Format:
            HasDuplicateValues(^TRT395)
    
            Data Elements:
    
            ^TRT395 = TRT:INT:Declaration:IntermediaryContact:PersonUnstructuredName.Usage.Code
            */
            assertion = HasDuplicateValues(report.InT_Declaration_IntermediaryContactCollection == null ? null : report.InT_Declaration_IntermediaryContactCollection.Select(f => f.TRT395).Cast<object>().ToArray());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.433110", Severity = ProcessMessageSeverity.Error,
                    Description = @"Intermediary Declaration Name usage code must be unique",
                    Location = "/tns:TRT/tns:INT/tns:Declaration/tns:IntermediaryContactCollection/tns:IntermediaryContact[" + report.InT_Declaration_IntermediaryContactCollection[DuplicateValueIndex(report.InT_Declaration_IntermediaryContactCollection.Select(f => f.TRT395))].OccurrenceIndex + "]/tns:PersonUnstructuredNameUsageC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.433110"} },
                };
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.442893
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRT442893()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.442893
            If there is an amount greater than zero at Small business energy incentive, there must be an amount at Reconciliation items- Expense reconciliation adjustments- Australian business expenses
    
            Legacy Rule Format:
            ^TRT576 > 0 AND ^TRT447 = NULL

            Technical Business Rule Format:
            ^TRT576 > 0 AND ^TRT447 = NULL
    
            Data Elements:
    
            ^TRT576 = TRT:RP:BusinessAndProfessionalInformation:SmallBusinessBoost:Expense.EnergyIncentive.Amount
    
            ^TRT447 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:BusinessExpenses:Expense.ReconciliationAdjustmentTotal.Amount
            */
            assertion = (report.TRT576.GetValueOrDefault() > 0 && report.TRT447 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.442893", Severity = ProcessMessageSeverity.Error,
                    Description = @"Reconciliation items- Expense reconciliation adjustments- Australian business expenses must be present",
                    LongDescription = @"If there is an amount greater than zero at Small business energy incentive, there must be an amount at Reconciliation items- Expense reconciliation adjustments- Australian business expenses",
                    Location = "/tns:TRT/tns:RP/tns:BusinessAndProfessionalInformation/tns:SmallBusinessBoost/tns:ExpenseEnergyIncentiveA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.442893"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT576", Value = GetValueOrEmpty(report.TRT576) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT447", Value = GetValueOrEmpty(report.TRT447) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.W00001
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRTW00001()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.W00001
            If the 'Exemption or rollover code' is 'T - Small Business Restructure Roll-over' then the value entered in 'total business income' cannot be more than 10 million dollars.
    
            Legacy Rule Format:
            AnyOccurrence(^TRT448, ^TRT448 = 'T - Small Business Restructure Roll-over') AND (^TRT46 + ^TRT50 + ^TRT54 + ^TRT47 + ^TRT49 + ^TRT52 + ^TRT55) > 10000000

            Technical Business Rule Format:
            AnyOccurrence(^TRT448, ^TRT448 = 'T - Small Business Restructure Roll-over') AND (^TRT46 + ^TRT50 + ^TRT54 + ^TRT47 + ^TRT49 + ^TRT52 + ^TRT55) > 10000000
    
            Data Elements:
    
            ^TRT448 = TRT:RP:BusinessDetails:Income:CapitalGains:CapitalGainsTax.ExemptionOrRolloverApplied.Code
    
            ^TRT46 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^TRT47 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^TRT49 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Remuneration.PaymentToForeignResidentGross.Amount
    
            ^TRT50 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^TRT52 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^TRT54 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Income.Other.Amount
    
            ^TRT55 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Income.Other.Amount
            */
            assertion = ((report.TRT448Collection == null ? false : report.TRT448Collection.Any(TRT448Repeat => TRT448Repeat == @"T - Small Business Restructure Roll-over")) && report.TRT46.GetValueOrDefault() + report.TRT50.GetValueOrDefault() + report.TRT54.GetValueOrDefault() + report.TRT47.GetValueOrDefault() + report.TRT49.GetValueOrDefault() + report.TRT52.GetValueOrDefault() + report.TRT55.GetValueOrDefault() > 10000000);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.W00001", Severity = ProcessMessageSeverity.Warning,
                    Description = @"Total business income amount incorrect",
                    LongDescription = @"To apply the small business restructure roll-over, each party to the transfer must be a small business entity, or an entity that is either (a) an affiliate, (b) connected with an entity, or (c) a partner in a partnership, that is a small business entity.",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:CapitalGains/tns:TaxExemptionOrRolloverAppliedCCollection/tns:TaxExemptionOrRolloverAppliedC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.W00001"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT448", Value = "TRT448" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT46", Value = GetValueOrEmpty(report.TRT46) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT50", Value = GetValueOrEmpty(report.TRT50) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT54", Value = GetValueOrEmpty(report.TRT54) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT47", Value = GetValueOrEmpty(report.TRT47) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT49", Value = GetValueOrEmpty(report.TRT49) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT52", Value = GetValueOrEmpty(report.TRT52) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT55", Value = GetValueOrEmpty(report.TRT55) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.W00002
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRTW00002()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.W00002
            If the rollover applied code is "T - Small Business Restructure Roll-over", then the residency status of the trust must be 'Resident'.
    
            Legacy Rule Format:
            AnyOccurrence(^TRT448, ^TRT448 = 'T - Small Business Restructure Roll-over') AND ^TRT245 <> 'Resident'

            Technical Business Rule Format:
            AnyOccurrence(^TRT448, ^TRT448 = 'T - Small Business Restructure Roll-over') AND ^TRT245 <> 'Resident'
    
            Data Elements:
    
            ^TRT448 = TRT:RP:BusinessDetails:Income:CapitalGains:CapitalGainsTax.ExemptionOrRolloverApplied.Code
    
            ^TRT245 = TRT:RP:NonResidentTrust:Residency.TaxPurposesOrganisationStatus.Code
            */
            assertion = ((report.TRT448Collection == null ? false : report.TRT448Collection.Any(TRT448Repeat => TRT448Repeat == @"T - Small Business Restructure Roll-over")) && report.TRT245 != @"Resident");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.W00002", Severity = ProcessMessageSeverity.Warning,
                    Description = @"Residency status of the trust must be 'Resident', unless eligible under certain exceptions.",
                    LongDescription = @"If the rollover applied code is ""T - Small Business Restructure Roll-over"", then the residency status of the trust must be 'Resident'.",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:CapitalGains/tns:TaxExemptionOrRolloverAppliedCCollection/tns:TaxExemptionOrRolloverAppliedC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.W00002"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT448", Value = "TRT448" });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT245", Value = report.TRT245 });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.W00003
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRTW00003()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.W00003
            NCMI from non-primary production - Business income and excluded from NCMI from non-primary production - Business income should be included in non-PP income - Other business income.
    
            Legacy Rule Format:
            ^TRT55 < (^TRT525 + ^TRT526)

            Technical Business Rule Format:
            ^TRT55 < (^TRT525 + ^TRT526)
    
            Data Elements:
    
            ^TRT55 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Income.Other.Amount
    
            ^TRT525 = TRT:RP:BusinessDetails:Income:NCMIBusinessIncome:NCMIBusinessIncomeNonPrimaryProduction:Income.NonConcessionalManagedInvestmentTrust.Amount
    
            ^TRT526 = TRT:RP:BusinessDetails:Income:NCMIBusinessIncome:NCMIBusinessIncomeNonPrimaryProduction:Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount
            */
            assertion = (report.TRT55.GetValueOrDefault() < report.TRT525.GetValueOrDefault() + report.TRT526.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.W00003", Severity = ProcessMessageSeverity.Warning,
                    Description = @"Ensure NCMI and Excluded from NCMI - business income from non-primary production are included at non-PP income - Other business income.",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:BusinessIncomeAndExpenses/tns:IncomeNonPrimaryProduction/tns:OtherA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.W00003"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT55", Value = GetValueOrEmpty(report.TRT55) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT525", Value = GetValueOrEmpty(report.TRT525) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT526", Value = GetValueOrEmpty(report.TRT526) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.W00004
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRTW00004()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.W00004
            NCMI from non-primary production - Distribution from partnerships less foreign income and excluded from NCMI from non-primary production - Distribution from partnerships less foreign income should be included in non-PP income - Distribution from partnerships less foreign income.
    
            Legacy Rule Format:
            ^TRT80 < (^TRT531 + ^TRT532)

            Technical Business Rule Format:
            ^TRT80 < (^TRT531 + ^TRT532)
    
            Data Elements:
    
            ^TRT80 = TRT:RP:BusinessDetails:Income:PartnershipTrustNonPrimaryProduction:Income.PartnershipDistributionLessForeignIncome.Amount
    
            ^TRT531 = TRT:RP:BusinessDetails:Income:NCMIPartnershipsAndTrusts:NCMIPartnershipsAndTrustsNonPrimaryProduction:Income.NonConcessionalManagedInvestmentTrust.Amount
    
            ^TRT532 = TRT:RP:BusinessDetails:Income:NCMIPartnershipsAndTrusts:NCMIPartnershipsAndTrustsNonPrimaryProduction:Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount
            */
            assertion = (report.TRT80.GetValueOrDefault() < report.TRT531.GetValueOrDefault() + report.TRT532.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.W00004", Severity = ProcessMessageSeverity.Warning,
                    Description = @"Ensure NCMI and Excluded from NCMI - partnership income from non-primary production are included at non-PP income - Distribution from partnerships less foreign income.",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:PartnershipTrustNonPrimaryProduction/tns:PartnershipDistributionLessForeignIncomeA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.W00004"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT80", Value = GetValueOrEmpty(report.TRT80) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT531", Value = GetValueOrEmpty(report.TRT531) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT532", Value = GetValueOrEmpty(report.TRT532) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.W00005
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRTW00005()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.W00005
            NCMI from non-primary production - Share of net income from trusts less net capital gain, foreign income and franked distributions and excluded from NCMI from non-primary production - Share of net income from trusts less net capital gain, foreign income and franked distributions should be included in non-PP income - Share of net income from trusts less net capital gain, foreign income and franked distributions.
    
            Legacy Rule Format:
            ^TRT81 < (^TRT533 + ^TRT534)

            Technical Business Rule Format:
            ^TRT81 < (^TRT533 + ^TRT534)
    
            Data Elements:
    
            ^TRT81 = TRT:RP:BusinessDetails:Income:PartnershipTrustNonPrimaryProduction:Income.TrustShareNetExcludeNetCapitalGainsAndForeignIncomeAndDistributionFranked.Amount
    
            ^TRT533 = TRT:RP:BusinessDetails:Income:NCMIPartnershipsAndTrusts:NCMIPartnershipsAndTrustsNonPrimaryProduction:NonPrimaryProductionShareOfNetIncome:Income.NonConcessionalManagedInvestmentTrust.Amount
    
            ^TRT534 = TRT:RP:BusinessDetails:Income:NCMIPartnershipsAndTrusts:NCMIPartnershipsAndTrustsNonPrimaryProduction:NonPrimaryProductionShareOfNetIncome:Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount
            */
            assertion = (report.TRT81.GetValueOrDefault() < report.TRT533.GetValueOrDefault() + report.TRT534.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.W00005", Severity = ProcessMessageSeverity.Warning,
                    Description = @"Ensure NCMI and Excluded from NCMI - share of net income from trusts from non-primary production are included at non-PP income - Share of net income from trusts less net capital gains, foreign income and franked distributions.",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:PartnershipTrustNonPrimaryProduction/tns:TrustShareNetExcludeNetCapitalGainsAndForeignIncomeAndDistributionFrankedA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.W00005"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT81", Value = GetValueOrEmpty(report.TRT81) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT533", Value = GetValueOrEmpty(report.TRT533) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT534", Value = GetValueOrEmpty(report.TRT534) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.W00006
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRTW00006()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.W00006
            NCMI from primary production - Business income and excluded from NCMI from primary production - Business income should be included in PP income - Other business income.
    
            Legacy Rule Format:
            ^TRT54 < (^TRT523 + ^TRT524)

            Technical Business Rule Format:
            ^TRT54 < (^TRT523 + ^TRT524)
    
            Data Elements:
    
            ^TRT54 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Income.Other.Amount
    
            ^TRT523 = TRT:RP:BusinessDetails:Income:NCMIBusinessIncome:NCMIBusinessIncomePrimaryProduction:Income.NonConcessionalManagedInvestmentTrust.Amount
    
            ^TRT524 = TRT:RP:BusinessDetails:Income:NCMIBusinessIncome:NCMIBusinessIncomePrimaryProduction:Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount
            */
            assertion = (report.TRT54.GetValueOrDefault() < report.TRT523.GetValueOrDefault() + report.TRT524.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.W00006", Severity = ProcessMessageSeverity.Warning,
                    Description = @"Ensure NCMI and Excluded from NCMI - business income from primary production are included at PP income - Other business income.",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:BusinessIncomeAndExpenses/tns:IncomePrimaryProduction/tns:OtherA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.W00006"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT54", Value = GetValueOrEmpty(report.TRT54) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT523", Value = GetValueOrEmpty(report.TRT523) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT524", Value = GetValueOrEmpty(report.TRT524) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.W00007
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRTW00007()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.W00007
            NCMI from primary production - Distribution from partnerships and excluded from NCMI from primary production - Distribution from partnerships should be included in PP income - Distribution from partnerships.
    
            Legacy Rule Format:
            ^TRT76 < (^TRT527 + ^TRT528)

            Technical Business Rule Format:
            ^TRT76 < (^TRT527 + ^TRT528)
    
            Data Elements:
    
            ^TRT76 = TRT:RP:BusinessDetails:Income:PartnershipsTrustsPrimaryProduction:Income.PartnershipDistributionGross.Amount
    
            ^TRT527 = TRT:RP:BusinessDetails:Income:NCMIPartnershipsAndTrusts:NCMIPartnershipsAndTrustsPrimaryProduction:Income.NonConcessionalManagedInvestmentTrust.Amount
    
            ^TRT528 = TRT:RP:BusinessDetails:Income:NCMIPartnershipsAndTrusts:NCMIPartnershipsAndTrustsPrimaryProduction:Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount
            */
            assertion = (report.TRT76.GetValueOrDefault() < report.TRT527.GetValueOrDefault() + report.TRT528.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.W00007", Severity = ProcessMessageSeverity.Warning,
                    Description = @"Ensure NCMI and Excluded from NCMI - partnership income from primary production are included at PP income - Distribution from partnerships.",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:PartnershipsTrustsPrimaryProduction/tns:PartnershipDistributionGrossA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.W00007"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT76", Value = GetValueOrEmpty(report.TRT76) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT527", Value = GetValueOrEmpty(report.TRT527) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT528", Value = GetValueOrEmpty(report.TRT528) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.W00008
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRTW00008()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.W00008
            NCMI from primary production - Share of net income from trusts and excluded from NCMI from primary production - Share of net income from trusts should be included in PP income - Share of net income from trusts.
    
            Legacy Rule Format:
            ^TRT77 < (^TRT529 + ^TRT530)

            Technical Business Rule Format:
            ^TRT77 < (^TRT529 + ^TRT530)
    
            Data Elements:
    
            ^TRT77 = TRT:RP:BusinessDetails:Income:PartnershipsTrustsPrimaryProduction:Income.TrustShareNet.Amount
    
            ^TRT529 = TRT:RP:BusinessDetails:Income:NCMIPartnershipsAndTrusts:NCMIPartnershipsAndTrustsPrimaryProduction:PrimaryProductionShareOfNetIncome:Income.NonConcessionalManagedInvestmentTrust.Amount
    
            ^TRT530 = TRT:RP:BusinessDetails:Income:NCMIPartnershipsAndTrusts:NCMIPartnershipsAndTrustsPrimaryProduction:PrimaryProductionShareOfNetIncome:Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount
            */
            assertion = (report.TRT77.GetValueOrDefault() < report.TRT529.GetValueOrDefault() + report.TRT530.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.W00008", Severity = ProcessMessageSeverity.Warning,
                    Description = @"Ensure NCMI and Excluded from NCMI - share of net income from trusts from primary production are included at PP income - Share of net income from trusts.",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:PartnershipsTrustsPrimaryProduction/tns:TrustShareNetA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.W00008"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT77", Value = GetValueOrEmpty(report.TRT77) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT529", Value = GetValueOrEmpty(report.TRT529) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT530", Value = GetValueOrEmpty(report.TRT530) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.W00015
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRTW00015()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.W00015
            Net capital gain cannot be blank if either of capital gains NCMI or capital gains excluded from NCMI are greater than zero.
    
            Legacy Rule Format:
            ^TRT133 = NULL AND (^TRT535 > 0 OR ^TRT536 > 0)

            Technical Business Rule Format:
            ^TRT133 = NULL AND (^TRT535 > 0 OR ^TRT536 > 0)
    
            Data Elements:
    
            ^TRT133 = TRT:RP:BusinessDetails:Income:CapitalGains:Income.CapitalGainsNet.Amount
    
            ^TRT535 = TRT:RP:BusinessDetails:Income:NCMICapitalGains:Capital.Gains.NonConcessionalManagedInvestmentTrustIncome.Amount
    
            ^TRT536 = TRT:RP:BusinessDetails:Income:NCMICapitalGains:Capital.Gains.ExcludedFromNonConcessionalManagedInvestmentTrustIncome.Amount
            */
            assertion = (report.TRT133 == null && (report.TRT535.GetValueOrDefault() > 0 || report.TRT536.GetValueOrDefault() > 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.W00015", Severity = ProcessMessageSeverity.Warning,
                    Description = @"Ensure net capital gains is not blank if either of capital gains NCMI or capital gains Excluded from NCMI are greater than zero.",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:CapitalGains/tns:NetA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.W00015"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT133", Value = GetValueOrEmpty(report.TRT133) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT535", Value = GetValueOrEmpty(report.TRT535) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT536", Value = GetValueOrEmpty(report.TRT536) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.W00016
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRTW00016()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.W00016
            If the company has self-assessed that they are a Significant Global Entity (SGE), please confirm if they are a Country by Country Reporting Entity (CbCRE).
    
            Legacy Rule Format:
            ^TRT487 = TRUE AND ^TRT550 = NULL

            Technical Business Rule Format:
            ^TRT487 = TRUE AND ^TRT550 = NULL
    
            Data Elements:
    
            ^TRT487 = TRT:RP:BusinessDetails:OrganisationDetails.SignificantGlobalEntityStatus.Indicator
    
            ^TRT550 = TRT:RP:BusinessDetails:InternationalDealings.CountryByCountryReportingEntityStatus.Indicator
            */
            assertion = (report.TRT487 == true && report.TRT550 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.W00016", Severity = ProcessMessageSeverity.Warning,
                    Description = @"Country by Country Reporting Entity (CBCRE) indicator should be supplied.",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:OrganisationDetailsSignificantGlobalEntityStatusI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.W00016"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT487", Value = GetValueOrEmpty(report.TRT487) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT550", Value = GetValueOrEmpty(report.TRT550) });
    
                response.Add(processMessage);
            }
      }
      #endregion 
    
      #region VR.ATO.TRT.W00019
      [MethodImpl(MethodImplOptions.AggressiveInlining)]
      public virtual void VRATOTRTW00019()
      {
        ProcessMessageDocument processMessage;
        bool assertion;
    
            /*  VR.ATO.TRT.W00019
            If 'P' is selected at aggregated turnover range, then the aggregated turnover value should be more than $1 billion
    
            Legacy Rule Format:
            ^TRT564 < 1000000000 AND ^TRT563 = "P"

            Technical Business Rule Format:
            ^TRT564 < 1000000000 AND ^TRT563 = 'P'
    
            Data Elements:
    
            ^TRT564 = TRT:RP:BusinessAndProfessionalInformation:AggregatedTurnover:Income.AggregatedTurnover.Amount
    
            ^TRT563 = TRT:RP:BusinessAndProfessionalInformation:AggregatedTurnover:Income.AggregatedTurnoverRange.Code
            */
            assertion = (report.TRT564.GetValueOrDefault() < 1000000000 && report.TRT563 == @"P");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.W00020", Severity = ProcessMessageSeverity.Warning,
                    Description = @"Aggregated turnover value should be more than $1 billion",
                    LongDescription = @"If 'P' is selected at aggregated turnover range, then the aggregated turnover value should be more than $1 billion",
                    Location = "/tns:TRT/tns:RP/tns:BusinessAndProfessionalInformation/tns:AggregatedTurnover/tns:IncomeA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.TRT.W00019"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT564", Value = GetValueOrEmpty(report.TRT564) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "TRT563", Value = report.TRT563 });
    
                response.Add(processMessage);
            }
      }
      #endregion 

    }
} 
