using System;
using System.Collections.Generic;
using System.Linq;
using System.Xml;
using System.Text.RegularExpressions;
using Au.Gov.Sbr.Xbrl.Document;
using DataContracts;
using Au.Gov.Sbr.Xbrl.Metadata;

namespace Ato.EN.IntegrationServices.CodeGenerationPSS
{
    public class PSS2018Consumer
    {

        private static Regex StripWhitespace = new Regex(@"\s*");

        #region Helper/Validation Methods
        private static bool? ToBoolean(string str)
        {
            bool returnValue;
            string value = (str ?? "").Trim();
            if (value == "0" || value == "1")
                return Convert.ToBoolean(Convert.ToInt32(value));
            
            if (Boolean.TryParse(value, out returnValue))
                return returnValue;
            return null;
        }

        private static string GetValueOrEmpty(bool? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        private static string GetValueOrEmpty(DateTime? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        /// <summary>
        /// Derives the SBR element location.
        /// </summary>
        /// <param name="sbrElement">The SBR element.</param>
        /// <param name="addOccurrenceIndex">if set to <c>true</c> [add occurrence the occurrence index to the path].</param>
        /// <returns></returns>
        private static string DeriveSbrElementLocation(SBRElement sbrElement, bool addOccurrenceIndex = false)
        {
            if (sbrElement == null) return "/xbrli:xbrl";
            int xpathOccurrence = sbrElement.OccurrenceIndex + 1;
            string location;

            if (addOccurrenceIndex)
            {
                location = "/" + sbrElement.NamespacePrefix + ":" + sbrElement.Name + "[" +
                           xpathOccurrence + "]";
            }
            else
            {
                location = "/" + sbrElement.NamespacePrefix + ":" + sbrElement.Name;
            }

            if (sbrElement.IsInTuple)
            {
                location = DeriveSbrElementLocation(sbrElement.ParentTuple, true) + location;
            }
            else
            {
                location = "/xbrli:xbrl" + location;
            }

            return location;
        }

        private static bool IsMatch(int? field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(Convert.ToString(field.Value), expression, options);
        }

        private static bool IsMatch(string field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(field, expression, options);
        }

        public DateTime AsDate(string dateAsString)
        {
            DateTime response = DateTime.MinValue;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date;
            }

            return response;
        }

         #endregion

        public PSS2018 Consume(SBRDocument sbrDocument)
        {
            List<ProcessMessageDocument> errors;
            return Consume(sbrDocument, false, out errors);
        }

        /// <summary>
        /// Allows for validation of rules that can not be generated and must run at a lower level.
        /// A sub-class will need to be created with an override for this method - which does nothing in its base implementation.
        /// </summary>
        /// <param name="sbrDocument">The SBR document.</param>
        /// <param name="populatedReportElementsMap">The populated report elements map can be used to find data elements and tuples.</param>
        /// <param name="contextArray">The context array can be used to find context values.</param>
        /// <param name="errors">The errors collection is to be populated with validation results.</param>
        protected virtual void ValidateUnsupportedRules(SBRDocument sbrDocument, IDictionary<string, SBRElement> populatedReportElementsMap, SBRContext[] contextArray, List<ProcessMessageDocument> errors)
        {

        }

        public PSS2018 Consume(SBRDocument sbrDocument, bool validationMode, out List<ProcessMessageDocument> errors)
        {
            PSS2018 report;
            report = new PSS2018();
            errors = new List<ProcessMessageDocument>();

            SBRContext[] contextArray;
            IDictionary<string, SBRElement> populatedReportElementsMap;

            contextArray = sbrDocument.ContextsMap.Values.ToArray();
            populatedReportElementsMap = sbrDocument.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();

            if (validationMode) ValidateUnsupportedRules(sbrDocument, populatedReportElementsMap, contextArray, errors);

            /// <summary>
            /// Current element being extracted
            /// </summary>
            SBRElement currentValue = null;
            /// <summary>
            /// Current elements being extracted
            /// </summary>
            IList<SBRElement> currentValues = null;
            /// <summary>
            /// Current occurrances being extracted
            /// </summary>
            IList<SBRElement> currentOccurrences = null;
            /// <summary>
            /// Working Variable for date time variables
            /// </summary>
            DateTime currentDateTimeValue;
            /// <summary>
            /// Working Variable for boolean variables
            /// </summary>
            bool currentBooleanValue;
            /// <summary>
            /// Working Variable for decimal variables
            /// </summary>
            decimal currentDecimalValue;
            /// <summary>
            /// Working Variable for integer variables
            /// </summary>
            int currentIntValue;
            /// <summary>
            /// Working Variable for long variables
            /// </summary>
            long currentLongValue;
            XmlDocumentFragment currentSegment;
            XmlNamespaceManager currentNamespaceManager;
            XmlNode currentNode;
    
            #region sbrRPPayerPSSeqNumContext
            IEnumerable<SBRContext> sbrRPPayerPSSeqNumContexts = null;
            SBRContext sbrRPPayerPSSeqNumContext = null;
    
            sbrRPPayerPSSeqNumContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 3 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}SecondaryPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Payer"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.TYPED && dd.Dimension.EndsWith("}SequenceNumberDimension")) ))));
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPPayerPSSeqNumElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPPayerPSSeqNumContexts != null && sbrRPPayerPSSeqNumContexts.Count() > 0)
            {
        
                report.RPPayerPSSeqNumContextCollection = new List<PSS2018.RPPayerPSSeqNumContext>();
                for (int i = 0; i < sbrRPPayerPSSeqNumContexts.Count(); i++)
                {
                    sbrRPPayerPSSeqNumContext = sbrRPPayerPSSeqNumContexts.ElementAt(i);
                    PSS2018.RPPayerPSSeqNumContext rpPayerPSSeqNumContext = new PSS2018.RPPayerPSSeqNumContext();
                    report.RPPayerPSSeqNumContextCollection.Add(rpPayerPSSeqNumContext);
            
                    rpPayerPSSeqNumContext.Id = sbrRPPayerPSSeqNumContext.Id;
                    rpPayerPSSeqNumContext.Identifier = sbrRPPayerPSSeqNumContext.EntityIdentifier;
                    rpPayerPSSeqNumContext.IdentifierScheme = sbrRPPayerPSSeqNumContext.EntityScheme;
            
                    rpPayerPSSeqNumContext.Count = sbrRPPayerPSSeqNumContexts.Count();
                    rpPayerPSSeqNumContext.Exists = true;
                    rpPayerPSSeqNumContext.OccurrenceIndex = Array.IndexOf(contextArray, sbrRPPayerPSSeqNumContext) + 1;
                    rpPayerPSSeqNumContext.LastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPPayerPSSeqNumContexts.Last()) + 1;
            
    
                    if (rpPayerPSSeqNumContext.IdentifierScheme == "http://www.ato.gov.au/tfn") rpPayerPSSeqNumContext.IdentifierTFN = rpPayerPSSeqNumContext.Identifier;
        
                    rpPayerPSSeqNumContext.StartDate = ((SBRDurationPeriod)sbrRPPayerPSSeqNumContext.Period).DurationStartAsDateObject;
                    rpPayerPSSeqNumContext.EndDate = ((SBRDurationPeriod)sbrRPPayerPSSeqNumContext.Period).DurationEndAsDateObject;
            
                    currentSegment = sbrRPPayerPSSeqNumContext.Segment.Segment;
            
                    currentNamespaceManager = new XmlNamespaceManager(currentSegment.OwnerDocument.NameTable);
                    currentNamespaceManager.AddNamespace("xbrldi", "http://xbrl.org/2006/xbrldi");
            
                    currentNode = currentSegment.SelectSingleNode("xbrldi:typedMember[contains(@dimension, 'SequenceNumberDimension')]", currentNamespaceManager);
            
                    if (currentNode != null)
                    {
                        rpPayerPSSeqNumContext.SequenceNumberDimensionValue = currentNode.InnerText.Trim();
                    }
            
                    sbrPopulatedRPPayerPSSeqNumElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPPayerPSSeqNumContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPPayerPSSeqNumElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPPayerPSSeqNumElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPPayerPSSeqNumElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
                    rpPayerPSSeqNumContext.PSS30 = rpPayerPSSeqNumContext.Identifier;
                    rpPayerPSSeqNumContext.PSS50 = rpPayerPSSeqNumContext.SequenceNumberDimensionValue;
            
                    if (sbrPopulatedRPPayerPSSeqNumElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/pss}IncomeTax.IncomeType.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) rpPayerPSSeqNumContext.PSS15 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPPayerPSSeqNumElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/pss}Identifiers.AustralianBusinessNumber.Identifier", out currentValues))
                    {
                        if (!currentValues[0].IsNil) rpPayerPSSeqNumContext.PSS16 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPPayerPSSeqNumElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/pss}Identifiers.WithholdingPayerNumber.Identifier", out currentValues))
                    {
                        if (!currentValues[0].IsNil) rpPayerPSSeqNumContext.PSS17 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPPayerPSSeqNumElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/pss}OrganisationNameDetails.OrganisationalName.Text", out currentValues))
                    {
                        if (!currentValues[0].IsNil) rpPayerPSSeqNumContext.PSS26 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPPayerPSSeqNumElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/pss}PersonNameDetails.FamilyName.Text", out currentValues))
                    {
                        if (!currentValues[0].IsNil) rpPayerPSSeqNumContext.PSS29 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPPayerPSSeqNumElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/pss}PersonNameDetails.GivenName.Text", out currentValues))
                    {
                        if (!currentValues[0].IsNil) rpPayerPSSeqNumContext.PSS27 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPPayerPSSeqNumElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/pss}PersonNameDetails.OtherGivenName.Text", out currentValues))
                    {
                        if (!currentValues[0].IsNil) rpPayerPSSeqNumContext.PSS28 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPPayerPSSeqNumElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/pss}Remuneration.PaymentToForeignResidentGross.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) rpPayerPSSeqNumContext.PSS18 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPPayerPSSeqNumElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/pss}Remuneration.ABNNotQuotedPaymentGross.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) rpPayerPSSeqNumContext.PSS19 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPPayerPSSeqNumElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/pss}Remuneration.VoluntaryAgreementGross.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) rpPayerPSSeqNumContext.PSS20 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPPayerPSSeqNumElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/pss}Remuneration.LabourHireArrangementPaymentGross.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) rpPayerPSSeqNumContext.PSS21 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPPayerPSSeqNumElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/pss}Remuneration.PersonalServicesIncome.AttributedGross.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) rpPayerPSSeqNumContext.PSS31 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPPayerPSSeqNumElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/pss}IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldBusinessForeignResident.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) rpPayerPSSeqNumContext.PSS22 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPPayerPSSeqNumElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/pss}IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldBusinessABNNotQuoted.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) rpPayerPSSeqNumContext.PSS23 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPPayerPSSeqNumElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/pss}IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldBusinessVoluntaryAgreement.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) rpPayerPSSeqNumContext.PSS24 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPPayerPSSeqNumElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/pss}IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldBusinessLabourHireOrOtherPayments.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) rpPayerPSSeqNumContext.PSS25 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPPayerPSSeqNumElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/pss}IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldPersonalServicesIncome.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) rpPayerPSSeqNumContext.PSS32 = decimal.Parse(currentValues[0].Value);
                    }
        } 
            } 
            #endregion End of Context sbrRPPayerPSSeqNumContext
            return report;

        } // Of Consume Method
    } // Of Class
} // Of Namespace

