using System;
using System.Collections.Generic;
using System.Linq;
using System.Xml;
using System.Text.RegularExpressions;
using Au.Gov.Sbr.Xbrl.Document;
using DataContracts;
using Au.Gov.Sbr.Xbrl.Metadata;

namespace Ato.EN.IntegrationServices.CodeGenerationRS
{
    public class RS2018Consumer
    {

        private static Regex StripWhitespace = new Regex(@"\s*");

        #region Helper/Validation Methods
        private static bool? ToBoolean(string str)
        {
            bool returnValue;
            string value = (str ?? "").Trim();
            if (value == "0" || value == "1")
                return Convert.ToBoolean(Convert.ToInt32(value));
            
            if (Boolean.TryParse(value, out returnValue))
                return returnValue;
            return null;
        }

        private static string GetValueOrEmpty(bool? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        private static string GetValueOrEmpty(DateTime? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        /// <summary>
        /// Derives the SBR element location.
        /// </summary>
        /// <param name="sbrElement">The SBR element.</param>
        /// <param name="addOccurrenceIndex">if set to <c>true</c> [add occurrence the occurrence index to the path].</param>
        /// <returns></returns>
        private static string DeriveSbrElementLocation(SBRElement sbrElement, bool addOccurrenceIndex = false)
        {
            if (sbrElement == null) return "/xbrli:xbrl";
            int xpathOccurrence = sbrElement.OccurrenceIndex + 1;
            string location;

            if (addOccurrenceIndex)
            {
                location = "/" + sbrElement.NamespacePrefix + ":" + sbrElement.Name + "[" +
                           xpathOccurrence + "]";
            }
            else
            {
                location = "/" + sbrElement.NamespacePrefix + ":" + sbrElement.Name;
            }

            if (sbrElement.IsInTuple)
            {
                location = DeriveSbrElementLocation(sbrElement.ParentTuple, true) + location;
            }
            else
            {
                location = "/xbrli:xbrl" + location;
            }

            return location;
        }

        private static bool IsMatch(int? field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(Convert.ToString(field.Value), expression, options);
        }

        private static bool IsMatch(string field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(field, expression, options);
        }

        public DateTime AsDate(string dateAsString)
        {
            DateTime response = DateTime.MinValue;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date;
            }

            return response;
        }

         #endregion

        public RS2018 Consume(SBRDocument sbrDocument)
        {
            List<ProcessMessageDocument> errors;
            return Consume(sbrDocument, false, out errors);
        }

        /// <summary>
        /// Allows for validation of rules that can not be generated and must run at a lower level.
        /// A sub-class will need to be created with an override for this method - which does nothing in its base implementation.
        /// </summary>
        /// <param name="sbrDocument">The SBR document.</param>
        /// <param name="populatedReportElementsMap">The populated report elements map can be used to find data elements and tuples.</param>
        /// <param name="contextArray">The context array can be used to find context values.</param>
        /// <param name="errors">The errors collection is to be populated with validation results.</param>
        protected virtual void ValidateUnsupportedRules(SBRDocument sbrDocument, IDictionary<string, SBRElement> populatedReportElementsMap, SBRContext[] contextArray, List<ProcessMessageDocument> errors)
        {

        }

        public RS2018 Consume(SBRDocument sbrDocument, bool validationMode, out List<ProcessMessageDocument> errors)
        {
            RS2018 report;
            report = new RS2018();
            errors = new List<ProcessMessageDocument>();

            SBRContext[] contextArray;
            IDictionary<string, SBRElement> populatedReportElementsMap;

            contextArray = sbrDocument.ContextsMap.Values.ToArray();
            populatedReportElementsMap = sbrDocument.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();

            if (validationMode) ValidateUnsupportedRules(sbrDocument, populatedReportElementsMap, contextArray, errors);

            /// <summary>
            /// Current element being extracted
            /// </summary>
            SBRElement currentValue = null;
            /// <summary>
            /// Current elements being extracted
            /// </summary>
            IList<SBRElement> currentValues = null;
            /// <summary>
            /// Current occurrances being extracted
            /// </summary>
            IList<SBRElement> currentOccurrences = null;
            /// <summary>
            /// Working Variable for date time variables
            /// </summary>
            DateTime currentDateTimeValue;
            /// <summary>
            /// Working Variable for boolean variables
            /// </summary>
            bool currentBooleanValue;
            /// <summary>
            /// Working Variable for decimal variables
            /// </summary>
            decimal currentDecimalValue;
            /// <summary>
            /// Working Variable for integer variables
            /// </summary>
            int currentIntValue;
            /// <summary>
            /// Working Variable for long variables
            /// </summary>
            long currentLongValue;
            XmlDocumentFragment currentSegment;
            XmlNamespaceManager currentNamespaceManager;
            XmlNode currentNode;
    
            #region sbrRPContext
            IEnumerable<SBRContext> sbrRPContexts = null;
            SBRContext sbrRPContext = null;
    
            sbrRPContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 1 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty")) ));
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPContexts != null && sbrRPContexts.Count() > 0)
            {
                sbrRPContext = sbrRPContexts.First();
        
                report.RPId = sbrRPContext.Id;
                report.RPIdentifier = sbrRPContext.EntityIdentifier;
                report.RPIdentifierScheme = sbrRPContext.EntityScheme;
        
                report.RPCount = sbrRPContexts.Count();
                report.RPExists = true;
                report.RPOccurrenceIndex = Array.IndexOf(contextArray, sbrRPContext) + 1;
                report.RPLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPContexts.Last()) + 1;
        

                if (report.RPIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPIdentifierTFN = report.RPIdentifier;
    
                report.RPStartDate = ((SBRDurationPeriod)sbrRPContext.Period).DurationStartAsDateObject;
                report.RPEndDate = ((SBRDurationPeriod)sbrRPContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPContexts.Count(); i++)
                {
                    sbrRPContext = sbrRPContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
                    report.RS11 = report.RPIdentifier;
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/rs}Party.Type.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.RS12 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/rs}AddressDetails.Line1.Text", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.RS17 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/rs}AddressDetails.Line2.Text", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.RS18 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/rs}AddressDetails.LocalityName.Text", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.RS19 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/rs}AddressDetails.Postcode.Text", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.RS20 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/rs}AddressDetails.StateOrTerritory.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.RS21 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/rs}RealEstateProperty.RentalIncomeFirstEarned.Date", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.RS24 = DateTime.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/rs}RealEstateProperty.RentalWeeks.Count", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.RS25 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/rs}RealEstateProperty.RentalWeeksAvailable.Count", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.RS27 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/rs}RealEstateProperty.Ownership.Percent", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.RS28 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/rs}RealEstateProperty.Acquisition.Date", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.RS29 = DateTime.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/rs}Assets.Investment.RealEstatePropertyPurchase.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.RS30 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/rs}RealEstateProperty.Disposal.Date", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.RS31 = DateTime.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/rs}Assets.Investment.RealEstatePropertySale.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.RS32 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/rs}Income.RealEstateProperty.CapitalGainsNet.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.RS33 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/rs}Expense.RealEstateProperty.PlantRecoupedDepreciationOnSale.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.RS34 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/rs}Expense.RealEstateProperty.CapitalWorksDeductionRecouped.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.RS35 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/rs}RealEstateProperty.LoanRenegotiated.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.RS36 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/rs}OrganisationNameDetails.OrganisationalName.Text", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.RS14 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/rs}Income.RealEstateProperty.Rental.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.RS38 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/rs}Income.RealEstateProperty.RentalRelatedOther.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.RS39 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/rs}Expense.RealEstateProperty.Marketing.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.RS40 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/rs}Expense.RealEstateProperty.BodyCorporate.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.RS41 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/rs}Expense.RealEstateProperty.Borrowing.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.RS42 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/rs}Expense.RealEstateProperty.Cleaning.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.RS43 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/rs}Expense.RealEstateProperty.CouncilRates.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.RS44 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/rs}Expense.DepreciationandAmortisation.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.RS45 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/rs}Expense.RealEstateProperty.Gardening.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.RS46 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/rs}Expense.RealEstateProperty.Insurance.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.RS47 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/rs}Expense.RealEstateProperty.LoanInterest.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.RS48 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/rs}Expense.LandTax.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.RS49 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/rs}Expense.RealEstateProperty.LegalFees.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.RS50 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/rs}Expense.RealEstateProperty.PestControl.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.RS51 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/rs}Expense.RealEstateProperty.AgentFeesCommission.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.RS52 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/rs}Expense.RealEstateProperty.RepairsAndMaintenance.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.RS53 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/rs}Expense.RealEstateProperty.CapitalWorksDeduction.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.RS54 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/rs}Expense.RealEstateProperty.OfficeSupplies.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.RS55 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/rs}Expense.Travel.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.RS56 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/rs}Expense.RealEstateProperty.Water.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.RS57 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/rs}Expense.RealEstateProperty.Sundry.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.RS58 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/rs}Income.RealEstateProperty.RentalNet.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.RS59 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPContext
            return report;

        } // Of Consume Method
    } // Of Class
} // Of Namespace

