using System;
using System.IO;
using System.Runtime.CompilerServices;
using System.Text;
using System.Xml;
using System.Linq;
using System.Collections.Generic;
using Ato.EN.IntegrationServices.Document.DataContracts;
using System.Text.RegularExpressions;

namespace Ato.EN.IntegrationServices.CodeGenerationTRTAMIS
{

    public partial class TRTAMIS2022Validator
    {
        /// <summary>
        /// The max parameter name length is restricted by the SBR1 and SBR2 schemas - the lowest common denominator of SBR 1 is used as the default
        /// </summary>
        private int _maxParameterNameLength;

        /// <summary>
        /// The max parameter value length is restricted by the SBR1 and SBR2 schemas
        /// </summary>
        private int _maxParameterValueLength;

        /// <summary>
        /// The SBR1 and ebms3 schemas do not allow parameter names or values to be the empty string
        /// </summary>
        private string _emptyParameterValue;

        /// <summary>
        /// Initializes a new instance of the <see cref="TRTAMIS2022Validator" /> class.
        /// </summary>
        /// <param name="maxParameterNameLength">Maximum length of the parameter name.</param>
        /// <param name="maxParameterValueLength">Maximum length of the parameter value.</param>
        /// <param name="emptyParameterValue">This value will be used in place of any parameter values that result in a null or empty value.</param>
        public TRTAMIS2022Validator(int maxParameterNameLength = 20, int maxParameterValueLength = 4096, string emptyParameterValue = "EMPTY")
        {
            _maxParameterNameLength = maxParameterNameLength;
            _maxParameterValueLength = maxParameterValueLength;
            _emptyParameterValue = emptyParameterValue;
        }

        #region Functions
        private static IEnumerable<string> Union(IEnumerable<string> list1, IEnumerable<string> list2)
        {
            IEnumerable<string> response;

            if (list1 == null && list2 == null)
            {
                response = null;
            }
            else if (list1 == null)
            {
                response = list2.Distinct();
            }
            else if (list2 == null)
            {
                response = list1.Distinct();
            }
            else
            {
                response = list1.Union(list2);
            }

            return response;
        }

        private static bool IsMatch(int? field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(Convert.ToString(field.Value), expression, options);
        }

        private static bool IsMatch(string field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(field, expression, options);
        }

        // This is just context and tuple counts where they are integer values - easier that changing the parsing logic to just return the value
        private static int Count(int count)
        {
            return count;
        }


        private static int Count<T>(IEnumerable<T> values)
        {
            return values == null ? 0 : values.Where(f => f != null).Count();
        }


        private static int Count<T>(ICollection<T> values)
        {
            return values == null ? 0 : values.Where(f => f != null).Count();
        }


        private static bool exists(bool value)
        {
            return value;
        }

        private static string GetValueOrEmpty(bool? val)
        {
            return (val.HasValue) ? val.ToString().ToLower() : string.Empty;
        }

        private static string GetValueOrEmptyDate(DateTime? val)
        {
            return (val.HasValue) ? val.Value.ToString("yyyy-MM-dd") : string.Empty;
        }

        private static string GetValueOrEmptyDate(string val)
        {
            DateTime dateTimeOut;
            if (DateTime.TryParse(val, out dateTimeOut))
            {
                return dateTimeOut.ToString("yyyy-MM-dd");
            }
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmptyDateTime(string val)
        {
            DateTime dateTimeOut;
            if (DateTime.TryParse(val, out dateTimeOut))
            {
                return dateTimeOut.ToString("yyyy-MM-ddTHH:mm:ssZ");
            }
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmptyDateTime(DateTime? val)
        {
            return (val.HasValue) ? val.Value.ToString("yyyy-MM-ddTHH:mm:ssZ") : string.Empty;
        }

        private static string GetValueOrEmpty(string val)
        {
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmpty(decimal? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        private static string GetValueOrEmpty(int? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        /// <summary>
        /// Get string value between [first] a and [last] b.
        /// </summary>
        public static string Between(string value, string a, string b)
        {
            int posA = value.IndexOf(a);
            int posB = value.LastIndexOf(b);
            if (posA == -1)
            {
                return "";
            }
            if (posB == -1)
            {
                return "";
            }
            int adjustedPosA = posA + a.Length;
            if (adjustedPosA >= posB)
            {
                return "";
            }
            return value.Substring(adjustedPosA, posB - adjustedPosA);
        }

        /// <summary>
        /// Get string value after [first] a.
        /// </summary>
        public static string Before(string value, string a)
        {
            int posA = value.IndexOf(a);
            if (posA == -1)
            {
                return "";
            }
            return value.Substring(0, posA);
        }

        /// <summary>
        /// Get string value after [last] a.
        /// </summary>
        public static string After(string value, string a)
        {
            int posA = value.LastIndexOf(a);
            if (posA == -1)
            {
                return "";
            }
            int adjustedPosA = posA + a.Length;
            if (adjustedPosA >= value.Length)
            {
                return "";
            }
            return value.Substring(adjustedPosA);
        }

        private static int Length(object field)
        {
            if (field == null)
                return 0;
            else
                return field.ToString().Trim().Length;
        }

        private static bool NotSameValues(IEnumerable<object> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool NotSameValues(params object[] nodes)
        {
            if (nodes == null)
                return false;

            return (from x in nodes select x).Distinct().Count() != 1;
        }

        private static bool HasDuplicateValues(IEnumerable<DateTime> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool HasDuplicateValues(IEnumerable<DateTime?> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool NotSameValues(IEnumerable<DateTime> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool NotSameValues(IEnumerable<DateTime?> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool HasDuplicateValues(IEnumerable<object> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool HasDuplicateValues(params object[] nodes)
        {
            if (nodes == null)
                return false;

            nodes = nodes.Where(x => x != null).ToArray();
            return !((from x in nodes select x).Distinct().Count() == nodes.Count());

        }

        private int DuplicateValueIndex(IEnumerable<object> values)
        {
            int response = 0;
            var hashset = new HashSet<object>();
            foreach (var value in values)
            {
                if (!hashset.Add(value))
                {
                    return response;
                }
                response++;
            }
            return response;
        }

        private int DuplicateValueIndex<T>(IEnumerable<T?> values) where T : struct
        {
            int response = 0;
            var hashset = new HashSet<T?>();
            foreach (var value in values)
            {
                if (!hashset.Add(value))
                {
                    return response;
                }
                response++;
            }
            return response;
        }

        private static bool IsDate(object value)
        {
            DateTime dateValue;
            return (value != null && DateTime.TryParse(value.ToString(), out dateValue));
        }

        public DateTime AsDate(string dateAsString)
        {
            DateTime response = DateTime.MinValue;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date;
            }

            return response;
        }

        public DateTime Date(DateTime? datetime)
        {
            return datetime.GetValueOrDefault().Date;
        }

        //The logic in After function expects "---" for day and "--" for month. 
        //Since hyphen was missing the date always returned null
        public DateTime? ConvertToDate(int day, int month, int year)
        {
            return ConvertToDate(day == 0 ? null : "---" + day.ToString(), month == 0 ? null : "--" + month.ToString(), year == 0 ? null : year.ToString());
        }

        public DateTime? ConvertToDate(string day, string month, string year)
        {
            DateTime? response;
            DateTime result;

            if (year == null || month == null || day == null)
            {
                return null;
            }
            string dateAsString = year + "-" + After(month, "--") + "-" + After(day, "---");
            if (DateTime.TryParse(dateAsString, out result))
            {
                response = result;
            }
            else
            {
                response = null;
            }
            return response;
        }


        public DateTime? ConvertToDate(string day, string month, int year)
        {
            DateTime? response;
            DateTime result;

            if (year == 0 || month == null || day == null)
            {
                return null;
            }
            string dateAsString = year.ToString() + "-" + After(month, "--") + "-" + After(day, "---");
            if (DateTime.TryParse(dateAsString, out result))
            {
                response = result;
            }
            else
            {
                response = null;
            }
            return response;
        }


        private static int Day(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Day;
            }

            return response;
        }


        private static int Day(DateTime? date)
        {
            if (date == null)
                return 0;
            else
                return date.Value.Day;

        }


        private static string Month(string dateAsString)
        {
            string response = null;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.ToString("MMMM");
            }
            else
            {
                return "NotAMonth";
            }

            return response;
        }

        private static string Month(DateTime? date)
        {
            if (date == null)
                return "NotAMonth";
            else
                return date.Value.ToString("MMMM");
        }


        private static int MonthAsInt(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Month;
            }

            return response;
        }

        private static int MonthAsInt(DateTime? date)
        {
            if (date == null)
            {
                return 0;
            }
            return date.Value.Month;
        }


        private static int Year(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Year;
            }

            return response;
        }

        private static int Year(DateTime? date)
        {
            if (date == null)
                return 0;
            else
                return date.Value.Year;
        }


        private static int CurrentFinancialYear()
        {
            return DateToFinancialYear(DateTime.Now, 7);
        }

        private static int FinancialYear(string dateAsString)
        {
            return DateToFinancialYear(dateAsString, 7);
        }

        private static int FinancialYear(DateTime? date)
        {
            return DateToFinancialYear(date, 7);
        }

        private static int DateToFinancialYear(string dateAsString, int startingMonth)
        {
            int response = 0;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
            {
                response = DateToFinancialYear(date, startingMonth);
            }

            return response;
        }
        private static int DateToFinancialYear(DateTime? date, int startingMonth)
        {
            int response;
            if (date == null)
            {
                response = 0;
            }
            else
            {
                int year = date.Value.Year;
                int month = date.Value.Month;

                if (startingMonth > month)
                    response = year;
                else
                    response = year + 1;
            }
            return response;
        }


        private static int FBTYear(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
                response = FBTYear(date);
            return response;
        }

        private static int FBTYear(DateTime? date)
        {
            if (date == null)
            {
                return 0;
            }
            else
            {
                if (date.Value.Month > 3)
                    return date.Value.Year + 1;
                else
                    return date.Value.Year;
            }
        }

        private static DateTime? AddMonthsToDate(DateTime? dateTime, int months)
        {
            return dateTime == null ? null : (DateTime?)dateTime.Value.AddMonths(months);
        }

        private static bool IsNumeric(object value)
        {
            decimal numbervalue;
            return (value != null && decimal.TryParse(value.ToString(), out numbervalue));
        }

        private static bool NotMonetary(decimal? field, string sign, int digits, int decimals)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                string decimalExpression;
                int digitsToUse = digits - decimals;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                if (decimals > 0)
                    decimalExpression = @"(\.\d{1," + decimals + "})?$";
                else
                    decimalExpression = @"$";

                return !(Regex.IsMatch(field.Value.ToString("0.#########################"), signExpression + @"\d{1," + digitsToUse + "}" + decimalExpression));
            }
        }

        private static bool NotNumeric(decimal? field, string sign, int digits, int decimals)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                string decimalExpression;
                int digitsToUse = digits - decimals;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                if (decimals > 0)
                    decimalExpression = @"(\.\d{1," + decimals + "})?$";
                else
                    decimalExpression = @"$";

                return !(Regex.IsMatch(field.Value.ToString("0.#########################"), signExpression + @"\d{1," + digitsToUse + "}" + decimalExpression));
            }
        }

        private static bool NotNumeric(int? field, string sign, int digits, int decimals = 0)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                return !(Regex.IsMatch(field.Value.ToString(), signExpression + @"\d{1," + digits + "}$"));
            }
        }

        private static bool NotNumeric(long? field, string sign, int digits, int decimals = 0)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                return !(Regex.IsMatch(field.Value.ToString(), signExpression + @"\d{1," + digits + "}$"));
            }
        }


        private static bool OutsideRange(decimal field, decimal expression, int range)
        {
            bool response;

            response = (field < (expression - range)) || (field > (expression + range));

            return response;
        }


        private static bool FailsUSIAlgorithm(string usi, string abn)
        {
            bool response = true;
            if (usi == null || abn == null)
            {
                response = false;
            }
            else
            {
                usi = usi.Trim();
                abn = abn.Trim();
                if (usi.Length == 14)
                {
                    int numeric;
                    if (usi.Substring(0, 11) == abn && int.TryParse(usi.Substring(11, 3), out numeric))
                        response = false;

                }
                else if (usi.Length == 9)
                {

                    if (Regex.IsMatch(usi, @"^([a-zA-Z]{3}\d{4}[a-zA-Z]{2})$"))
                        response = false;
                }
            }
            return response;
        }

        private static bool FailsTANAlgorithm(string tan)
        {
            bool response;
            decimal decimalTan;

            if (tan == null)
                return false;

            tan = tan.Trim();

            if (!decimal.TryParse(tan, out decimalTan))
                return true;

            if (tan.Length != 8)
                return true;

            decimal tanSum =
                7 * int.Parse(tan.Substring(0, 1)) +
                9 * int.Parse(tan.Substring(1, 1)) +
                8 * int.Parse(tan.Substring(2, 1)) +
                4 * int.Parse(tan.Substring(3, 1)) +
                6 * int.Parse(tan.Substring(4, 1)) +
                3 * int.Parse(tan.Substring(5, 1)) +
                5 * int.Parse(tan.Substring(6, 1)) +
                1 * int.Parse(tan.Substring(7, 1));

            if ((tanSum % 11) == 0)
                response = false;
            else
                response = true;

            return response;
        }


        private static bool FailsABNAlgorithm(string abn)
        {
            bool response;
            decimal decimalAbn;

            if (abn == null)
                return false;

            abn = abn.Trim();

            if (!decimal.TryParse(abn, out decimalAbn))
                return true;

            if (abn.Length != 11)
                return true;

            decimal abnSum =
                10 * (int.Parse(abn.Substring(0, 1)) - 1) +
                1 * int.Parse(abn.Substring(1, 1)) +
                3 * int.Parse(abn.Substring(2, 1)) +
                5 * int.Parse(abn.Substring(3, 1)) +
                7 * int.Parse(abn.Substring(4, 1)) +
                9 * int.Parse(abn.Substring(5, 1)) +
                11 * int.Parse(abn.Substring(6, 1)) +
                13 * int.Parse(abn.Substring(7, 1)) +
                15 * int.Parse(abn.Substring(8, 1)) +
                17 * int.Parse(abn.Substring(9, 1)) +
                19 * int.Parse(abn.Substring(10, 1));

            if ((abnSum % 89) == 0)
                response = false;
            else
                response = true;

            return response;
        }
        private static bool FailsACNAlgorithm(string acn)
        {
            bool response;
            decimal decimalAbn;
            if (acn == null)
                return false;

            acn = acn.Trim();

            if (!decimal.TryParse(acn, out decimalAbn))
                return true;

            if (acn.Length != 9)
                return true;

            decimal abnSum =
                8 * int.Parse(acn.Substring(0, 1)) +
                7 * int.Parse(acn.Substring(1, 1)) +
                6 * int.Parse(acn.Substring(2, 1)) +
                5 * int.Parse(acn.Substring(3, 1)) +
                4 * int.Parse(acn.Substring(4, 1)) +
                3 * int.Parse(acn.Substring(5, 1)) +
                2 * int.Parse(acn.Substring(6, 1)) +
                1 * int.Parse(acn.Substring(7, 1));

            decimal checkDigit = int.Parse(acn.Substring(8, 1));
            decimal acnRemainder = abnSum % 10;

            if (((10 - acnRemainder) % 10) == checkDigit)
                response = false;
            else
                response = true;

            return response;
        }


        private static bool FailsTFNAlgorithm(string tfn)
        {
            bool response;
            decimal decimalTfn;

            if (tfn == null)
                return false;

            tfn = tfn.Trim();
            tfn = Regex.Replace(tfn, "^0+", "");

            if (!decimal.TryParse(tfn, out decimalTfn))
                return true;

            if (tfn.Length < 8)
                return true;


            decimal tfn1To7Sum =
                1 * int.Parse(tfn.Substring(0, 1)) +
                4 * int.Parse(tfn.Substring(1, 1)) +
                3 * int.Parse(tfn.Substring(2, 1)) +
                7 * int.Parse(tfn.Substring(3, 1)) +
                5 * int.Parse(tfn.Substring(4, 1)) +
                8 * int.Parse(tfn.Substring(5, 1)) +
                6 * int.Parse(tfn.Substring(6, 1));

            decimal tfn8 = 9 * int.Parse(tfn.Substring(7, 1));

            if (tfn.Length == 8)
            {
                decimal tFNLg8WSum9 = 10 * int.Parse(tfn.Substring(7, 1));
                decimal tFNLg8WSum = tfn1To7Sum + tFNLg8WSum9;

                if ((tFNLg8WSum % 11) == 0)
                    response = false;
                else
                    response = true;
            }
            else if (tfn.Length == 9)
            {
                decimal tfn9 = 10 * int.Parse(tfn.Substring(8, 1));
                decimal tFNLg9WSum = tfn1To7Sum + tfn8 + tfn9;

                if ((tFNLg9WSum % 11) == 0)
                    response = false;
                else
                    response = true;
            }
            else
            {
                response = true;
            }

            return response;
        }


        private static decimal ConditionalValue(bool expression, decimal? trueVal, decimal? falseVal)
        {
            return expression ? trueVal.GetValueOrDefault() : falseVal.GetValueOrDefault();
        }

        private static decimal AsNumeric(string value)
        {
            decimal numberValue;
            decimal.TryParse(value, out numberValue);
            return numberValue;
        }

        private static bool RegexMatch(int? field, string expression, string flags = "")
        {
            return IsMatch(field, expression, GetRegexOption(flags));
        }

        private static bool RegexMatch(string field, string expression, string flags = "")
        {
            return IsMatch(field, expression, GetRegexOption(flags));
        }

        private static RegexOptions GetRegexOption(string flags)
        {
            RegexOptions options = RegexOptions.None;

            char[] characters = flags.ToCharArray();

            foreach (char character in characters)
            {
                switch (character)
                {
                    case 'i':
                        options = options | RegexOptions.IgnoreCase;
                        break;
                    case 'm':
                        options = options | RegexOptions.Multiline;
                        break;
                    case 's':
                        options = options | RegexOptions.Singleline;
                        break;
                    case 'n':
                        options = options | RegexOptions.ExplicitCapture;
                        break;
                    case 'x':
                        options = options | RegexOptions.IgnorePatternWhitespace;
                        break;
                }
            }

            return options;
        }

        private static string Substring(object field, int start, int length)
        {
            return field.ToString().Substring(start, length);
        }

        /// <summary>
        /// Returns an occurrence index as [occurrenceIndex] of occurrenceIndex > 0, otherwise the empty string
        /// </summary>
        /// <param name="occurrenceIndex">Index of the occurrence.</param>
        /// <returns>Occurrence in XPath [#] format</returns>
        public string OccurrenceIndex(int occurrenceIndex)
        {
            return occurrenceIndex > 0 ? "[" + occurrenceIndex + "]" : "";
        }

        #endregion // Functions

        public TRTAMIS2022 ConsumedReport { get; private set; }

        private static Dictionary<string, ProcessMessageDocument> _processMessageDocuments = new Dictionary<string, ProcessMessageDocument>();

        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public List<ProcessMessageDocument> ValidateReport(TRTAMIS2022 report)
        {

            List<ProcessMessageDocument> response = new List<ProcessMessageDocument>();
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            bool assertion;

            this.ConsumedReport = report;

            // ------------------------------------------------------------------------------
            // Validations are now done in code.
            // This version supports full validation of a report - for those rules that have successfully parsed and could be generated.
            // Generated Validations include:
            //   - Validation logic in C#.
            //   - Production of errors in EBMS, SBR and SWS format.
            // 
            // The generation at this stage does not support the following - and has to be completed manually (later versions of the generation will do this for you)
            //   - Business Rules that did not parse.  The ESR team will help support you where this happens as an effort is being made to rectify these
            //   - You will get TODO tasks for those that the parser could not cope with
            // ------------------------------------------------------------------------------


            #region VR.ATO.GEN.000209

            /*  VR.ATO.GEN.000209
            Invalid context. The number of ReportingParty contexts must equal 1
    
            Legacy Rule Format:
            IF COUNT(RP) <> 1
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) <> 1
    
            Data Elements:
    
            ^Context = RP
            */
            assertion = (Count(report.RPCount) != 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430296",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Reporting party contexts must equal 1",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.000209" } },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.000209

            #region VR.ATO.GEN.001023

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP(Instant)
    
            ^OtherTFN = RP(Instant)
    
            ^TFN = RP
            */
            assertion = (report.RPInstantExists != false && report.RPInstantIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPInstantOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001023

            #region VR.ATO.GEN.001023

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.AssessableIncomeCapitalGains
    
            ^OtherTFN = RP:RP.AssessableIncomeCapitalGains
    
            ^TFN = RP
            */
            assertion = (report.RPAssessableIncomeCapitalGainsExists != false && report.RPAssessableIncomeCapitalGainsIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPAssessableIncomeCapitalGainsOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001023

            #region VR.ATO.GEN.001023

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP.AssessableIncomeCapitalGains:RP.AssessableIncomeCapitalGains.Direct
    
            ^OtherTFN = RP.AssessableIncomeCapitalGains:RP.AssessableIncomeCapitalGains.Direct
    
            ^TFN = RP
            */
            assertion = (report.RPAssessableIncomeCapitalGainsDirectExists != false && report.RPAssessableIncomeCapitalGainsDirectIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPAssessableIncomeCapitalGainsDirectOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001023

            #region VR.ATO.GEN.001023

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP.AssessableIncomeCapitalGains:RP.AssessableIncomeCapitalGains.Indirect
    
            ^OtherTFN = RP.AssessableIncomeCapitalGains:RP.AssessableIncomeCapitalGains.Indirect
    
            ^TFN = RP
            */
            assertion = (report.RPAssessableIncomeCapitalGainsIndirectExists != false && report.RPAssessableIncomeCapitalGainsIndirectIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPAssessableIncomeCapitalGainsIndirectOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001023

            #region VR.ATO.GEN.001023

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.AssessableIncomeOtherThanCapitalGains
    
            ^OtherTFN = RP:RP.AssessableIncomeOtherThanCapitalGains
    
            ^TFN = RP
            */
            assertion = (report.RPAssessableIncomeOtherThanCapitalGainsExists != false && report.RPAssessableIncomeOtherThanCapitalGainsIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPAssessableIncomeOtherThanCapitalGainsOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001023

            #region VR.ATO.GEN.001023

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP.AssessableIncomeOtherThanCapitalGains:RP.AssessableIncomeOtherThanCapitalGains.Direct
    
            ^OtherTFN = RP.AssessableIncomeOtherThanCapitalGains:RP.AssessableIncomeOtherThanCapitalGains.Direct
    
            ^TFN = RP
            */
            assertion = (report.RPAssessableIncomeOtherThanCapitalGainsDirectExists != false && report.RPAssessableIncomeOtherThanCapitalGainsDirectIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPAssessableIncomeOtherThanCapitalGainsDirectOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001023

            #region VR.ATO.GEN.001023

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP.AssessableIncomeOtherThanCapitalGains:RP.AssessableIncomeOtherThanCapitalGains.Indirect
    
            ^OtherTFN = RP.AssessableIncomeOtherThanCapitalGains:RP.AssessableIncomeOtherThanCapitalGains.Indirect
    
            ^TFN = RP
            */
            assertion = (report.RPAssessableIncomeOtherThanCapitalGainsIndirectExists != false && report.RPAssessableIncomeOtherThanCapitalGainsIndirectIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPAssessableIncomeOtherThanCapitalGainsIndirectOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001023

            #region VR.ATO.GEN.001023

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.NetExemptionIncome
    
            ^OtherTFN = RP:RP.NetExemptionIncome
    
            ^TFN = RP
            */
            assertion = (report.RPNetExemptionIncomeExists != false && report.RPNetExemptionIncomeIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPNetExemptionIncomeOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001023

            #region VR.ATO.GEN.001023

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.NonAssessableNonExemptAmounts
    
            ^OtherTFN = RP:RP.NonAssessableNonExemptAmounts
    
            ^TFN = RP
            */
            assertion = (report.RPNonAssessableNonExemptAmountsExists != false && report.RPNonAssessableNonExemptAmountsIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPNonAssessableNonExemptAmountsOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001023

            #region VR.ATO.GEN.001023

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.Offset
    
            ^OtherTFN = RP:RP.Offset
    
            ^TFN = RP
            */
            assertion = (report.RPOffsetExists != false && report.RPOffsetIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOffsetOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001023

            #region VR.ATO.GEN.438029

            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be"http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP
    
            ^Context = RP
            */
            assertion = (report.RPExists != false && report.RPIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029" } },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.438029

            #region VR.ATO.GEN.438029

            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be"http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP(Instant)
    
            ^Context = RP(Instant)
            */
            assertion = (report.RPInstantExists != false && report.RPInstantIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPInstantOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029" } },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.438029

            #region VR.ATO.GEN.438029

            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be"http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.AssessableIncomeCapitalGains
    
            ^Context = RP:RP.AssessableIncomeCapitalGains
            */
            assertion = (report.RPAssessableIncomeCapitalGainsExists != false && report.RPAssessableIncomeCapitalGainsIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPAssessableIncomeCapitalGainsOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029" } },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.438029

            #region VR.ATO.GEN.438029

            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be"http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP.AssessableIncomeCapitalGains:RP.AssessableIncomeCapitalGains.Direct
    
            ^Context = RP.AssessableIncomeCapitalGains:RP.AssessableIncomeCapitalGains.Direct
            */
            assertion = (report.RPAssessableIncomeCapitalGainsDirectExists != false && report.RPAssessableIncomeCapitalGainsDirectIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPAssessableIncomeCapitalGainsDirectOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029" } },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.438029

            #region VR.ATO.GEN.438029

            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be"http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP.AssessableIncomeCapitalGains:RP.AssessableIncomeCapitalGains.Indirect
    
            ^Context = RP.AssessableIncomeCapitalGains:RP.AssessableIncomeCapitalGains.Indirect
            */
            assertion = (report.RPAssessableIncomeCapitalGainsIndirectExists != false && report.RPAssessableIncomeCapitalGainsIndirectIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPAssessableIncomeCapitalGainsIndirectOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029" } },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.438029

            #region VR.ATO.GEN.438029

            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be"http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.AssessableIncomeOtherThanCapitalGains
    
            ^Context = RP:RP.AssessableIncomeOtherThanCapitalGains
            */
            assertion = (report.RPAssessableIncomeOtherThanCapitalGainsExists != false && report.RPAssessableIncomeOtherThanCapitalGainsIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPAssessableIncomeOtherThanCapitalGainsOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029" } },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.438029

            #region VR.ATO.GEN.438029

            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be"http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP.AssessableIncomeOtherThanCapitalGains:RP.AssessableIncomeOtherThanCapitalGains.Direct
    
            ^Context = RP.AssessableIncomeOtherThanCapitalGains:RP.AssessableIncomeOtherThanCapitalGains.Direct
            */
            assertion = (report.RPAssessableIncomeOtherThanCapitalGainsDirectExists != false && report.RPAssessableIncomeOtherThanCapitalGainsDirectIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPAssessableIncomeOtherThanCapitalGainsDirectOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029" } },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.438029

            #region VR.ATO.GEN.438029

            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be"http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP.AssessableIncomeOtherThanCapitalGains:RP.AssessableIncomeOtherThanCapitalGains.Indirect
    
            ^Context = RP.AssessableIncomeOtherThanCapitalGains:RP.AssessableIncomeOtherThanCapitalGains.Indirect
            */
            assertion = (report.RPAssessableIncomeOtherThanCapitalGainsIndirectExists != false && report.RPAssessableIncomeOtherThanCapitalGainsIndirectIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPAssessableIncomeOtherThanCapitalGainsIndirectOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029" } },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.438029

            #region VR.ATO.GEN.438029

            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be"http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.NetExemptionIncome
    
            ^Context = RP:RP.NetExemptionIncome
            */
            assertion = (report.RPNetExemptionIncomeExists != false && report.RPNetExemptionIncomeIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPNetExemptionIncomeOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029" } },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.438029

            #region VR.ATO.GEN.438029

            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be"http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.NonAssessableNonExemptAmounts
    
            ^Context = RP:RP.NonAssessableNonExemptAmounts
            */
            assertion = (report.RPNonAssessableNonExemptAmountsExists != false && report.RPNonAssessableNonExemptAmountsIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPNonAssessableNonExemptAmountsOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029" } },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.438029

            #region VR.ATO.GEN.438029

            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be"http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.Offset
    
            ^Context = RP:RP.Offset
            */
            assertion = (report.RPOffsetExists != false && report.RPOffsetIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOffsetOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029" } },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.438029

            #region VR.ATO.TRTAMIS.000002

            /*  VR.ATO.TRTAMIS.000002
            Trust Australian Business Number is mandatory
    
            Legacy Rule Format:
            IF [TRTAMIS5] = NULL
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [TRTAMIS5] = TRTAMIS:RP:pyid.02.00:Identifiers.AustralianBusinessNumber.Identifier

            Technical Business Rule Format:
            ^TRTAMIS5 = NULL
    
            Data Elements:
    
            ^TRTAMIS5 = RP:Identifiers.AustralianBusinessNumber.Identifier
            */
            assertion = (report.TRTAMIS5 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000002",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Trust Australian Business Number must be present",
                    Location = "/xbrli:xbrl/tns:Identifiers.AustralianBusinessNumber.Identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000002" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS5", Value = report.TRTAMIS5 });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000002

            #region VR.ATO.TRTAMIS.000429

            /*  VR.ATO.TRTAMIS.000429
            Invalid context. The number of RP.NetExemptionIncome contexts cannot exceed 1.
    
            Legacy Rule Format:
            IF COUNT(RP.NetExemptionIncome) > 1
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.NetExemptionIncome
            */
            assertion = (Count(report.RPNetExemptionIncomeCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000429",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of RP.NetExemptionIncome contexts cannot exceed 1.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPNetExemptionIncomeLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000429" } },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000429

            #region VR.ATO.TRTAMIS.000430

            /*  VR.ATO.TRTAMIS.000430
            Invalid context. The number of RP.NonAssessableNonExemptAmounts contexts cannot exceed 1.
    
            Legacy Rule Format:
            IF COUNT(RP.NonAssessableNonExemptAmounts) > 1
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.NonAssessableNonExemptAmounts
            */
            assertion = (Count(report.RPNonAssessableNonExemptAmountsCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000430",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of RP.NonAssessableNonExemptAmounts contexts cannot exceed 1.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPNonAssessableNonExemptAmountsLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000430" } },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000430

            #region VR.ATO.TRTAMIS.000435

            /*  VR.ATO.TRTAMIS.000435
            Invalid context. The number of RP.Offset contexts cannot exceed 1.
    
            Legacy Rule Format:
            IF COUNT(RP.Offset) > 1
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.Offset
            */
            assertion = (Count(report.RPOffsetCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000435",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of RP.Offset contexts cannot exceed 1.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOffsetLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000435" } },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000435

            #region VR.ATO.TRTAMIS.000436

            /*  VR.ATO.TRTAMIS.000436
            Monetary field, must be unsigned, maximum of 11 chars, no decimals
    
            Legacy Rule Format:
            IF [TRTAMIS260] <> NULL AND [TRTAMIS260] <> MONETARY(U,11,0)
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [TRTAMIS260] = TRTAMIS:RP.AssessableIncomeOtherThanCapitalGains:bafpr1.02.23:Income.Taxable.Amount

            Technical Business Rule Format:
            NotMonetary(^TRTAMIS260, 'U', 11, 0)
    
            Data Elements:
    
            ^TRTAMIS260 = RP.AssessableIncomeOtherThanCapitalGains:Income.Taxable.Amount
            */
            assertion = NotMonetary(report.TRTAMIS260, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000436",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Assessable Income not in a valid monetary format",
                    LongDescription = @"Assessable Income must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Income.Taxable.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000436" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS260", Value = GetValueOrEmpty(report.TRTAMIS260) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000436

            #region VR.ATO.TRTAMIS.000437

            /*  VR.ATO.TRTAMIS.000437
            Monetary field, must be unsigned, maximum of 11 chars, no decimals
    
            Legacy Rule Format:
            IF [TRTAMIS261] <> NULL AND [TRTAMIS261] <> MONETARY(U,11,0)
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [TRTAMIS261] = TRTAMIS:RP.AssessableIncomeOtherThanCapitalGains.Direct:bafpr1.02.12:Expense.DeductionsTotal.Amount

            Technical Business Rule Format:
            NotMonetary(^TRTAMIS261, 'U', 11, 0)
    
            Data Elements:
    
            ^TRTAMIS261 = RP.AssessableIncomeOtherThanCapitalGains.Direct:Expense.DeductionsTotal.Amount
            */
            assertion = NotMonetary(report.TRTAMIS261, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000437",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Direct Deductions is not in a valid monetary format",
                    LongDescription = @"Direct Deductions must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Expense.DeductionsTotal.Amount[@contextRef='" + report.RPAssessableIncomeOtherThanCapitalGainsDirectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000437" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS261", Value = GetValueOrEmpty(report.TRTAMIS261) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000437

            #region VR.ATO.TRTAMIS.000438

            /*  VR.ATO.TRTAMIS.000438
            Monetary field, must be unsigned, maximum of 11 chars, no decimals
    
            Legacy Rule Format:
            IF [TRTAMIS262] <> NULL AND [TRTAMIS262] <> MONETARY(U,11,0)
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [TRTAMIS262] = TRTAMIS:RP.AssessableIncomeOtherThanCapitalGains.Indirect:bafpr1.02.29:Expense.DeductionsOther.Amount

            Technical Business Rule Format:
            NotMonetary(^TRTAMIS262, 'U', 11, 0)
    
            Data Elements:
    
            ^TRTAMIS262 = RP.AssessableIncomeOtherThanCapitalGains.Indirect:Expense.DeductionsOther.Amount
            */
            assertion = NotMonetary(report.TRTAMIS262, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000438",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other Deductions is not in a valid monetary format",
                    LongDescription = @"Other Deductions must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Expense.DeductionsOther.Amount[@contextRef='" + report.RPAssessableIncomeOtherThanCapitalGainsIndirectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000438" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS262", Value = GetValueOrEmpty(report.TRTAMIS262) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000438

            #region VR.ATO.TRTAMIS.000439

            /*  VR.ATO.TRTAMIS.000439
            Monetary field, must be unsigned, maximum of 11 chars, no decimals
    
            Legacy Rule Format:
            IF [TRTAMIS314] <> NULL AND [TRTAMIS314] <> MONETARY(U,11,0)
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [TRTAMIS314] = TRTAMIS:RP.AssessableIncomeOtherThanCapitalGains:bafpr1.02.29:Income.TrustComponent.Amount

            Technical Business Rule Format:
            NotMonetary(^TRTAMIS314, 'U', 11, 0)
    
            Data Elements:
    
            ^TRTAMIS314 = RP.AssessableIncomeOtherThanCapitalGains:Income.TrustComponent.Amount
            */
            assertion = NotMonetary(report.TRTAMIS314, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000439",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Trust Components is not in a valid monetary format",
                    LongDescription = @"Trust Components must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Income.TrustComponent.Amount[@contextRef='" + report.RPAssessableIncomeOtherThanCapitalGainsId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000439" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS314", Value = GetValueOrEmpty(report.TRTAMIS314) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000439

            #region VR.ATO.TRTAMIS.000440

            /*  VR.ATO.TRTAMIS.000440
            Monetary field, must be unsigned, maximum of 11 chars, no decimals
    
            Legacy Rule Format:
            IF [TRTAMIS315] <> NULL AND [TRTAMIS315] <> MONETARY(U,11,0)
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [TRTAMIS315] = TRTAMIS:RP.AssessableIncomeOtherThanCapitalGains:bafpr1.02.27:Income.UnderstatedEntitlement.Amount

            Technical Business Rule Format:
            NotMonetary(^TRTAMIS315, 'U', 11, 0)
    
            Data Elements:
    
            ^TRTAMIS315 = RP.AssessableIncomeOtherThanCapitalGains:Income.UnderstatedEntitlement.Amount
            */
            assertion = NotMonetary(report.TRTAMIS315, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000440",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total Unders is not in a valid monetary format",
                    LongDescription = @"Total Unders must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Income.UnderstatedEntitlement.Amount[@contextRef='" + report.RPAssessableIncomeOtherThanCapitalGainsId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000440" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS315", Value = GetValueOrEmpty(report.TRTAMIS315) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000440

            #region VR.ATO.TRTAMIS.000441

            /*  VR.ATO.TRTAMIS.000441
            Monetary field, must be unsigned, maximum of 11 chars, no decimals
    
            Legacy Rule Format:
            IF [TRTAMIS316] <> NULL AND [TRTAMIS316] <> MONETARY(U,11,0)
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [TRTAMIS316] = TRTAMIS:RP.AssessableIncomeOtherThanCapitalGains:bafpr1.02.27:Expense.Overstated.Amount

            Technical Business Rule Format:
            NotMonetary(^TRTAMIS316, 'U', 11, 0)
    
            Data Elements:
    
            ^TRTAMIS316 = RP.AssessableIncomeOtherThanCapitalGains:Expense.Overstated.Amount
            */
            assertion = NotMonetary(report.TRTAMIS316, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000441",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total Overs is not in a valid monetary format",
                    LongDescription = @"Total Overs must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Expense.Overstated.Amount[@contextRef='" + report.RPAssessableIncomeOtherThanCapitalGainsId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000441" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS316", Value = GetValueOrEmpty(report.TRTAMIS316) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000441

            #region VR.ATO.TRTAMIS.000461

            /*  VR.ATO.TRTAMIS.000461
            Monetary field, must be unsigned, maximum of 11 chars, no decimals
    
            Legacy Rule Format:
            IF [TRTAMIS285] <> NULL AND [TRTAMIS285] <> MONETARY(U,11,0)
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [TRTAMIS285] = TRTAMIS:RP.NetExemptionIncome:bafpr1.02.23:Income.Net.Amount

            Technical Business Rule Format:
            NotMonetary(^TRTAMIS285, 'U', 11, 0)
    
            Data Elements:
    
            ^TRTAMIS285 = RP.NetExemptionIncome:Income.Net.Amount
            */
            assertion = NotMonetary(report.TRTAMIS285, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000461",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Determined Trust Components is not in a valid monetary format",
                    LongDescription = @"Determined Trust Components must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Income.Net.Amount[@contextRef='" + report.RPNetExemptionIncomeId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000461" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS285", Value = GetValueOrEmpty(report.TRTAMIS285) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000461

            #region VR.ATO.TRTAMIS.000467

            /*  VR.ATO.TRTAMIS.000467
            Monetary field, must be unsigned, maximum of 11 chars, no decimals
    
            Legacy Rule Format:
            IF [TRTAMIS295] <> NULL AND [TRTAMIS295] <> MONETARY(U,11,0)
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [TRTAMIS295] = TRTAMIS:RP.Offset:bafpr1.02.29:Income.TrustComponent.Amount

            Technical Business Rule Format:
            NotMonetary(^TRTAMIS295, 'U', 11, 0)
    
            Data Elements:
    
            ^TRTAMIS295 = RP.Offset:Income.TrustComponent.Amount
            */
            assertion = NotMonetary(report.TRTAMIS295, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000467",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Trust Components is not in a valid monetary format",
                    LongDescription = @"Trust Components must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Income.TrustComponent.Amount[@contextRef='" + report.RPOffsetId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000467" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS295", Value = GetValueOrEmpty(report.TRTAMIS295) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000467

            #region VR.ATO.TRTAMIS.000468

            /*  VR.ATO.TRTAMIS.000468
            Monetary field, must be unsigned, maximum of 11 chars, no decimals
    
            Legacy Rule Format:
            IF [TRTAMIS296] <> NULL AND [TRTAMIS296] <> MONETARY(U,11,0)
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [TRTAMIS296] = TRTAMIS:RP.Offset:bafpr1.02.27:Income.UnderstatedEntitlement.Amount

            Technical Business Rule Format:
            NotMonetary(^TRTAMIS296, 'U', 11, 0)
    
            Data Elements:
    
            ^TRTAMIS296 = RP.Offset:Income.UnderstatedEntitlement.Amount
            */
            assertion = NotMonetary(report.TRTAMIS296, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000468",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total Unders is not in a valid monetary format",
                    LongDescription = @"Total Unders must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Income.UnderstatedEntitlement.Amount[@contextRef='" + report.RPOffsetId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000468" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS296", Value = GetValueOrEmpty(report.TRTAMIS296) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000468

            #region VR.ATO.TRTAMIS.000469

            /*  VR.ATO.TRTAMIS.000469
            Monetary field, must be unsigned, maximum of 11 chars, no decimals
    
            Legacy Rule Format:
            IF [TRTAMIS297] <> NULL AND [TRTAMIS297] <> MONETARY(U,11,0)
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [TRTAMIS297] = TRTAMIS:RP.Offset:bafpr1.02.27:Expense.Overstated.Amount

            Technical Business Rule Format:
            NotMonetary(^TRTAMIS297, 'U', 11, 0)
    
            Data Elements:
    
            ^TRTAMIS297 = RP.Offset:Expense.Overstated.Amount
            */
            assertion = NotMonetary(report.TRTAMIS297, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000469",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total Overs is not in a valid monetary format",
                    LongDescription = @"Total Overs must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Expense.Overstated.Amount[@contextRef='" + report.RPOffsetId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000469" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS297", Value = GetValueOrEmpty(report.TRTAMIS297) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000469

            #region VR.ATO.TRTAMIS.000471

            /*  VR.ATO.TRTAMIS.000471
            Monetary field, may be unsigned maximum of 11 chars, no decimals
    
            Legacy Rule Format:
            IF [TRTAMIS299] <> NULL AND [TRTAMIS299] <> MONETARY(U,11,0)
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [TRTAMIS299] = TRTAMIS:RP.Offset:bafpr1.02.23:Income.Net.Amount

            Technical Business Rule Format:
            NotMonetary(^TRTAMIS299, 'U', 11, 0)
    
            Data Elements:
    
            ^TRTAMIS299 = RP.Offset:Income.Net.Amount
            */
            assertion = NotMonetary(report.TRTAMIS299, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000471",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Determined Trust Components is not in a valid monetary format",
                    LongDescription = @"Determined Trust Components must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Income.Net.Amount[@contextRef='" + report.RPOffsetId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000471" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS299", Value = GetValueOrEmpty(report.TRTAMIS299) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000471

            #region VR.ATO.TRTAMIS.000472

            /*  VR.ATO.TRTAMIS.000472
            Monetary field, must be unsigned, maximum of 11 chars, no decimals
    
            Legacy Rule Format:
            IF [TRTAMIS300] <> NULL AND [TRTAMIS300] <> MONETARY(U,11,0)
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [TRTAMIS300] = TRTAMIS:RP.Offset:bafpr1.02.27:Income.TrustDeficitComponent.Amount

            Technical Business Rule Format:
            NotMonetary(^TRTAMIS300, 'U', 11, 0)
    
            Data Elements:
    
            ^TRTAMIS300 = RP.Offset:Income.TrustDeficitComponent.Amount
            */
            assertion = NotMonetary(report.TRTAMIS300, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000472",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Trust Component Deficits is not in a valid monetary format",
                    LongDescription = @"Trust Component Deficits must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Income.TrustDeficitComponent.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000472" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS300", Value = GetValueOrEmpty(report.TRTAMIS300) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000472

            #region VR.ATO.TRTAMIS.000480

            /*  VR.ATO.TRTAMIS.000480
            Monetary field, must be unsigned, maximum of 11 chars, no decimals
    
            Legacy Rule Format:
            IF [TRTAMIS309] <> NULL AND [TRTAMIS309] <> MONETARY(U,11,0)
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [TRTAMIS309] = TRTAMIS:RP:rvctc3.02.02:Tax.Losses.BalanceOfTaxLossesBroughtForwardFromPriorYear.Amount

            Technical Business Rule Format:
            NotMonetary(^TRTAMIS309, 'U', 11, 0)
    
            Data Elements:
    
            ^TRTAMIS309 = RP:Tax.Losses.BalanceOfTaxLossesBroughtForwardFromPriorYear.Amount
            */
            assertion = NotMonetary(report.TRTAMIS309, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000480",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Balance of Tax Losses Brought Forward from Prior Income Years is not in a valid monetary format",
                    LongDescription = @"Balance of Tax Losses Brought Forward from Prior Income Years must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.BalanceOfTaxLossesBroughtForwardFromPriorYear.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000480" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS309", Value = GetValueOrEmpty(report.TRTAMIS309) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000480

            #region VR.ATO.TRTAMIS.000481

            /*  VR.ATO.TRTAMIS.000481
            Monetary field, must be unsigned, maximum of 11 chars, no decimals
    
            Legacy Rule Format:
            IF [TRTAMIS310] <> NULL AND [TRTAMIS310] <> MONETARY(U,11,0)
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [TRTAMIS310] = TRTAMIS:RP:rvctc3.02.15:Tax.Losses.Deducted.Amount

            Technical Business Rule Format:
            NotMonetary(^TRTAMIS310, 'U', 11, 0)
    
            Data Elements:
    
            ^TRTAMIS310 = RP:Tax.Losses.Deducted.Amount
            */
            assertion = NotMonetary(report.TRTAMIS310, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000481",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax Losses Deducted is not in a valid monetary format",
                    LongDescription = @"Tax Losses Deducted must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.Deducted.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000481" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS310", Value = GetValueOrEmpty(report.TRTAMIS310) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000481

            #region VR.ATO.TRTAMIS.000482

            /*  VR.ATO.TRTAMIS.000482
            Monetary field, must be unsigned, maximum of 11 chars, no decimals
    
            Legacy Rule Format:
            IF [TRTAMIS311] <> NULL AND [TRTAMIS311] <> MONETARY(U,11,0)
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [TRTAMIS311] = TRTAMIS:RP:rvctc3.02.02:Tax.Losses.CarriedForward.LaterIncomeYearsTotal.Amount

            Technical Business Rule Format:
            NotMonetary(^TRTAMIS311, 'U', 11, 0)
    
            Data Elements:
    
            ^TRTAMIS311 = RP:Tax.Losses.CarriedForward.LaterIncomeYearsTotal.Amount
            */
            assertion = NotMonetary(report.TRTAMIS311, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000482",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax Losses Carried Forward to Later Income Years is not in a valid monetary format",
                    LongDescription = @"Tax Losses Carried Forward to Later Income Years must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.CarriedForward.LaterIncomeYearsTotal.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000482" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS311", Value = GetValueOrEmpty(report.TRTAMIS311) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000482

            #region VR.ATO.TRTAMIS.000507

            /*  VR.ATO.TRTAMIS.000507
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character
    
            Legacy Rule Format:
            IF ([TRTAMIS313] <> NULLORBLANK) AND (ANY CHARACTER OF [TRTAMIS313] <> SET("a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "))
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [TRTAMIS313] = TRTAMIS:RP:rvctc1.02.22:IncomeTax.AttributionManagedInvestmentTrustClass.Description

            Technical Business Rule Format:
            (^TRTAMIS313 <> BLANK) AND (NotCharacterInSet(^TRTAMIS313, '"a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))
    
            Data Elements:
    
            ^TRTAMIS313 = RP:IncomeTax.AttributionManagedInvestmentTrustClass.Description
            */
            assertion = (string.IsNullOrWhiteSpace(report.TRTAMIS313) != true && !(IsMatch(report.TRTAMIS313, @"^[a-zA-Z0-9!@\$%&\*\(\)\-=\[\];:'"",\.\?/ ]*$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000507",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Name of Attribution Managed Investment Trust (AMIT) Class contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character",
                    Location = "/xbrli:xbrl/tns:IncomeTax.AttributionManagedInvestmentTrustClass.Description",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000507" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS313", Value = GetValueOrEmpty(report.TRTAMIS313) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000507

            #region VR.ATO.TRTAMIS.000513

            /*  VR.ATO.TRTAMIS.000513
            Monetary field, must be unsigned, maximum of 11 chars, no decimals
    
            Legacy Rule Format:
            IF [TRTAMIS318] <> NULL AND [TRTAMIS318] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [TRTAMIS318] = TRTAMIS:RP:rvctc3.02.14:Tax.Losses.Uplift.Amount

            Technical Business Rule Format:
            NotMonetary(^TRTAMIS318, 'U', 11, 0)
    
            Data Elements:
    
            ^TRTAMIS318 = RP:Tax.Losses.Uplift.Amount
            */
            assertion = NotMonetary(report.TRTAMIS318, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000513",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Uplift of tax losses of designated infrastructure project entities is not in a valid monetary format",
                    LongDescription = @"Uplift of tax losses of designated infrastructure project entities must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.Uplift.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000513" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS318", Value = GetValueOrEmpty(report.TRTAMIS318) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000513

            #region VR.ATO.TRTAMIS.000514

            /*  VR.ATO.TRTAMIS.000514
            Monetary field, must be unsigned, maximum of 11 chars, no decimals
    
            Legacy Rule Format:
            IF [TRTAMIS319] <> NULL AND [TRTAMIS319] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [TRTAMIS319] = TRTAMIS:RP:rvctc3.02.02:Tax.Losses.ForgivenDebtNet.Amount

            Technical Business Rule Format:
            NotMonetary(^TRTAMIS319, 'U', 11, 0)
    
            Data Elements:
    
            ^TRTAMIS319 = RP:Tax.Losses.ForgivenDebtNet.Amount
            */
            assertion = NotMonetary(report.TRTAMIS319, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000514",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Net Forgiven Amount of Debt is not in a valid monetary format",
                    LongDescription = @"Net Forgiven Amount of Debt must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.ForgivenDebtNet.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000514" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS319", Value = GetValueOrEmpty(report.TRTAMIS319) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000514

            #region VR.ATO.TRTAMIS.000515

            /*  VR.ATO.TRTAMIS.000515
            Monetary field, must be unsigned, maximum of 11 chars, no decimals
    
            Legacy Rule Format:
            IF [TRTAMIS320] <> NULL AND [TRTAMIS320] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [TRTAMIS320] = TRTAMIS:RP:rvctc3.02.02:Tax.Losses.Incurred.Amount

            Technical Business Rule Format:
            NotMonetary(^TRTAMIS320, 'U', 11, 0)
    
            Data Elements:
    
            ^TRTAMIS320 = RP:Tax.Losses.Incurred.Amount
            */
            assertion = NotMonetary(report.TRTAMIS320, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000515",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax Losses Incurred (if any) During the Current Year is not in a valid monetary format",
                    LongDescription = @"Tax Losses Incurred (if any) During the Current Year must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.Incurred.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000515" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS320", Value = GetValueOrEmpty(report.TRTAMIS320) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000515

            #region VR.ATO.TRTAMIS.000516

            /*  VR.ATO.TRTAMIS.000516
            Monetary field, must be unsigned, maximum of 11 chars, no decimals
    
            Legacy Rule Format:
            IF [TRTAMIS321] <> NULL AND [TRTAMIS321] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [TRTAMIS321] = TRTAMIS:RP:rvctc3.02.02:Tax.Losses.ExemptIncomeNet.Amount

            Technical Business Rule Format:
            NotMonetary(^TRTAMIS321, 'U', 11, 0)
    
            Data Elements:
    
            ^TRTAMIS321 = RP:Tax.Losses.ExemptIncomeNet.Amount
            */
            assertion = NotMonetary(report.TRTAMIS321, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000516",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Net exempt Income is not in a valid monetary format",
                    LongDescription = @"Net exempt Income must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.ExemptIncomeNet.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000516" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS321", Value = GetValueOrEmpty(report.TRTAMIS321) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000516

            #region VR.ATO.TRTAMIS.000517

            /*  VR.ATO.TRTAMIS.000517
            Monetary field, must be unsigned, maximum of 11 chars, no decimals
    
            Legacy Rule Format:
            IF [TRTAMIS322] <> NULL AND [TRTAMIS322] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [TRTAMIS322] = TRTAMIS:RP:rvctc3.02.34:Tax.Losses.Forgone.Amount

            Technical Business Rule Format:
            NotMonetary(^TRTAMIS322, 'U', 11, 0)
    
            Data Elements:
    
            ^TRTAMIS322 = RP:Tax.Losses.Forgone.Amount
            */
            assertion = NotMonetary(report.TRTAMIS322, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000517",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax losses forgone is not in a valid monetary format",
                    LongDescription = @"Tax Losses forgone must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.Forgone.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000517" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS322", Value = GetValueOrEmpty(report.TRTAMIS322) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000517

            #region VR.ATO.TRTAMIS.000524

            /*  VR.ATO.TRTAMIS.000524
            Tax Losses Carried Forward to Later Income Years must equal the sum of Balance of Tax Losses Brought Forward from Prior Income Years, Uplift of Tax Losses of Designated Infrastructure Project Entities and Tax Loss Incurred (if any) During the Current Year, less Net Forgiven Amount of Debt, Net Exempt Income, Tax Losses Forgone and Tax Losses Deducted
    
            Legacy Rule Format:
            IF [TRTAMIS311] <> ([TRTAMIS309] + [TRTAMIS318] - [TRTAMIS319] + [TRTAMIS320] - [TRTAMIS321] - [TRTAMIS322] - [TRTAMIS310])
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [TRTAMIS309] = TRTAMIS:RP:rvctc3.02.02:Tax.Losses.BalanceOfTaxLossesBroughtForwardFromPriorYear.Amount
            [TRTAMIS310] = TRTAMIS:RP:rvctc3.02.15:Tax.Losses.Deducted.Amount
            [TRTAMIS311] = TRTAMIS:RP:rvctc3.02.02:Tax.Losses.CarriedForward.LaterIncomeYearsTotal.Amount
            [TRTAMIS318] = TRTAMIS:RP:rvctc3.02.14:Tax.Losses.Uplift.Amount
            [TRTAMIS319] = TRTAMIS:RP:rvctc3.02.02:Tax.Losses.ForgivenDebtNet.Amount
            [TRTAMIS320] = TRTAMIS:RP:rvctc3.02.02:Tax.Losses.Incurred.Amount
            [TRTAMIS321] = TRTAMIS:RP:rvctc3.02.02:Tax.Losses.ExemptIncomeNet.Amount
            [TRTAMIS322] = TRTAMIS:RP:rvctc3.02.34:Tax.Losses.Forgone.Amount

            Technical Business Rule Format:
            ^TRTAMIS311 <> (^TRTAMIS309 + ^TRTAMIS318 - ^TRTAMIS319 + ^TRTAMIS320 - ^TRTAMIS321 - ^TRTAMIS322 - ^TRTAMIS310)
    
            Data Elements:
    
            ^TRTAMIS311 = RP:Tax.Losses.CarriedForward.LaterIncomeYearsTotal.Amount
    
            ^TRTAMIS309 = RP:Tax.Losses.BalanceOfTaxLossesBroughtForwardFromPriorYear.Amount
    
            ^TRTAMIS310 = RP:Tax.Losses.Deducted.Amount
    
            ^TRTAMIS318 = RP:Tax.Losses.Uplift.Amount
    
            ^TRTAMIS319 = RP:Tax.Losses.ForgivenDebtNet.Amount
    
            ^TRTAMIS320 = RP:Tax.Losses.Incurred.Amount
    
            ^TRTAMIS321 = RP:Tax.Losses.ExemptIncomeNet.Amount
    
            ^TRTAMIS322 = RP:Tax.Losses.Forgone.Amount
            */
            assertion = (report.TRTAMIS311.GetValueOrDefault() != report.TRTAMIS309.GetValueOrDefault() + report.TRTAMIS318.GetValueOrDefault() - report.TRTAMIS319.GetValueOrDefault() + report.TRTAMIS320.GetValueOrDefault() - report.TRTAMIS321.GetValueOrDefault() - report.TRTAMIS322.GetValueOrDefault() - report.TRTAMIS310.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000524",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax Losses Carried Forward to Later Income Years amount is incorrect",
                    LongDescription = @"Tax Losses Carried Forward to Later Income Years for Tax Losses Information must equal the sum of Balance of Tax Losses Brought Forward from Prior Income Years, Uplift of Tax Losses of Designated Infrastructure Project Entities and Tax Loss Incurred (if any) During the Current Year, less Net Forgiven Amount of Debt, Net Exempt Income, Tax Losses Forgone and Tax Losses Deducted",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.CarriedForward.LaterIncomeYearsTotal.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000524" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS311", Value = GetValueOrEmpty(report.TRTAMIS311) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS309", Value = GetValueOrEmpty(report.TRTAMIS309) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS318", Value = GetValueOrEmpty(report.TRTAMIS318) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS319", Value = GetValueOrEmpty(report.TRTAMIS319) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS320", Value = GetValueOrEmpty(report.TRTAMIS320) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS321", Value = GetValueOrEmpty(report.TRTAMIS321) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS322", Value = GetValueOrEmpty(report.TRTAMIS322) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS310", Value = GetValueOrEmpty(report.TRTAMIS310) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000524

            #region VR.ATO.TRTAMIS.000527

            /*  VR.ATO.TRTAMIS.000527
            Monetary field, must be unsigned, maximum of 11 chars, no decimals
    
            Legacy Rule Format:
            IF [TRTAMIS265] <> NULL AND [TRTAMIS265] <> MONETARY(U,11,0)
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [TRTAMIS265] = TRTAMIS:RP.AssessableIncomeOtherThanCapitalGains:bafpr1.02.29:Expense.CarryForwardTrustComponentDeficit.Amount

            Technical Business Rule Format:
            NotMonetary(^TRTAMIS265, 'U', 11, 0)
    
            Data Elements:
    
            ^TRTAMIS265 = RP.AssessableIncomeOtherThanCapitalGains:Expense.CarryForwardTrustComponentDeficit.Amount
            */
            assertion = NotMonetary(report.TRTAMIS265, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000527",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Carry-Forward Trust Component Deficits is not in a valid monetary format",
                    LongDescription = @"Carry-Forward Trust Component Deficits must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Expense.CarryForwardTrustComponentDeficit.Amount[@contextRef='" + report.RPAssessableIncomeOtherThanCapitalGainsId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000527" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS265", Value = GetValueOrEmpty(report.TRTAMIS265) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000527

            #region VR.ATO.TRTAMIS.000528

            /*  VR.ATO.TRTAMIS.000528
            Monetary field, must be unsigned, maximum of 11 chars, no decimals
    
            Legacy Rule Format:
            IF [TRTAMIS266] <> NULL AND [TRTAMIS266] <> MONETARY(U,11,0)
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [TRTAMIS266] = TRTAMIS:RP.AssessableIncomeOtherThanCapitalGains:bafpr1.02.23:Income.Net.Amount

            Technical Business Rule Format:
            NotMonetary(^TRTAMIS266, 'U', 11, 0)
    
            Data Elements:
    
            ^TRTAMIS266 = RP.AssessableIncomeOtherThanCapitalGains:Income.Net.Amount
            */
            assertion = NotMonetary(report.TRTAMIS266, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000528",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Determined Trust Components is not in a valid monetary format",
                    LongDescription = @"Determined Trust Components must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Income.Net.Amount[@contextRef='" + report.RPAssessableIncomeOtherThanCapitalGainsId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000528" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS266", Value = GetValueOrEmpty(report.TRTAMIS266) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000528

            #region VR.ATO.TRTAMIS.000532

            /*  VR.ATO.TRTAMIS.000532
            Name of AMIT class has a maximum field length of 200
    
            Legacy Rule Format:
            IF LENGTH([TRTAMIS313]) > 200
                 RETURN VALIDATION MESSAGE
            ENDIF
            
            [TRTAMIS313] = TRTAMIS:RP:rvctc1.02.22:IncomeTax.AttributionManagedInvestmentTrustClass.Description

            Technical Business Rule Format:
            Length(^TRTAMIS313) > 200
    
            Data Elements:
    
            ^TRTAMIS313 = RP:IncomeTax.AttributionManagedInvestmentTrustClass.Description
            */
            assertion = (Length(report.TRTAMIS313) > 200);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000532",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Name of Attribution Managed Investment Trust (AMIT) Class must not exceed 200 characters",
                    LongDescription = @"For Name of Attribution Managed Investment Trust (AMIT) Class, the maximum allowable characters for this form is 200",
                    Location = "/xbrli:xbrl/tns:IncomeTax.AttributionManagedInvestmentTrustClass.Description",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000532" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS313", Value = GetValueOrEmpty(report.TRTAMIS313) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000532

            #region VR.ATO.TRTAMIS.000601

            /*  VR.ATO.TRTAMIS.000601
            Monetary field, must be unsigned, maximum of 11 chars, no decimals
    
            Legacy Rule Format:
            IF [TRTAMIS332] <> NULL AND [TRTAMIS332] <> MONETARY(U,11,0)
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [TRTAMIS332] = TRTAMIS:RP.NonAssessableNonExemptAmounts:bafpr1.02.23:Income.Net.Amount

            Technical Business Rule Format:
            NotMonetary(^TRTAMIS332, 'U', 11, 0)
    
            Data Elements:
    
            ^TRTAMIS332 = RP.NonAssessableNonExemptAmounts:Income.Net.Amount
            */
            assertion = NotMonetary(report.TRTAMIS332, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000601",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Determined Trust Components is not in a valid monetary format",
                    LongDescription = @"Determined Trust Components must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Income.Net.Amount[@contextRef='" + report.RPNonAssessableNonExemptAmountsId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000601" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS332", Value = GetValueOrEmpty(report.TRTAMIS332) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000601

            #region VR.ATO.TRTAMIS.000602

            /*  VR.ATO.TRTAMIS.000602
            Number of Members in the AMIT Class at the end of the Income Year and must be provided when Name of Attribution Managed Investment Trust (AMIT) Class is provided
    
            Legacy Rule Format:
            IF [TRTAMIS323] = NULL AND [TRTAMIS313] <> NULLORBLANK
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [TRTAMIS313] = TRTAMIS:RP:rvctc1.02.22:IncomeTax.AttributionManagedInvestmentTrustClass.Description
            [TRTAMIS323] = TRTAMIS:RP:rvctc1.02.24:IncomeTax.AttributionManagedInvestmentTrustClassMembers.Count

            Technical Business Rule Format:
            ^TRTAMIS323 = NULL AND ^TRTAMIS313 <> BLANK
    
            Data Elements:
    
            ^TRTAMIS323 = RP:IncomeTax.AttributionManagedInvestmentTrustClassMembers.Count
    
            ^TRTAMIS313 = RP:IncomeTax.AttributionManagedInvestmentTrustClass.Description
            */
            assertion = (report.TRTAMIS323 == null && string.IsNullOrWhiteSpace(report.TRTAMIS313) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000602",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Number of Members in the AMIT Class at the End of the Income Year must be provided",
                    LongDescription = @"Number of Members in the AMIT Class at the end of the Income Year and must be provided when Name of Attribution Managed Investment Trust (AMIT) Class is provided",
                    Location = "/xbrli:xbrl/tns:IncomeTax.AttributionManagedInvestmentTrustClassMembers.Count",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000602" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS323", Value = report.TRTAMIS323.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS313", Value = GetValueOrEmpty(report.TRTAMIS313) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000602

            #region VR.ATO.TRTAMIS.000603

            /*  VR.ATO.TRTAMIS.000603
            For assessable income details, Trust Components amounts must not be provided when the sum of all Direct Deductions and Other Deductions amounts exceeds the sum of all Assessable Income amounts.
    
            Legacy Rule Format:
            IF ([TRTAMIS314] > 0 OR [TRTAMIS327] > 0) AND ([TRTAMIS261] + [TRTAMIS325] + [TRTAMIS262] + [TRTAMIS326]) > ([TRTAMIS260] + [TRTAMIS336] + [TRTAMIS337] + [TRTAMIS324] + [TRTAMIS338] + [TRTAMIS339])
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [TRTAMIS260] = TRTAMIS:RP.AssessableIncomeOtherThanCapitalGains:bafpr1.02.23:Income.Taxable.Amount
            [TRTAMIS261] = TRTAMIS:RP.AssessableIncomeOtherThanCapitalGains.Direct:bafpr1.02.12:Expense.DeductionsTotal.Amount
            [TRTAMIS262] = TRTAMIS:RP.AssessableIncomeOtherThanCapitalGains.Indirect:bafpr1.02.29:Expense.DeductionsOther.Amount
            [TRTAMIS314] = TRTAMIS:RP.AssessableIncomeOtherThanCapitalGains:bafpr1.02.29:Income.TrustComponent.Amount
            [TRTAMIS324] = TRTAMIS:RP.AssessableIncomeCapitalGains:bafpr1.02.30:Income.CapitalGainsNetCredit.Amount
            [TRTAMIS325] = TRTAMIS:RP.AssessableIncomeCapitalGains.Direct:bafpr1.02.12:Expense.DeductionsTotal.Amount
            [TRTAMIS326] = TRTAMIS:RP.AssessableIncomeCapitalGains.Indirect:bafpr1.02.29:Expense.DeductionsOther.Amount
            [TRTAMIS327] = TRTAMIS:RP.AssessableIncomeCapitalGains:bafpr1.02.29:Income.TrustComponent.Amount
            [TRTAMIS336] = TRTAMIS:RP.AssessableIncomeOtherThanCapitalGains:bafpr1.02.40:Income.NonConcessionalManagedInvestmentTrust.Amount
            [TRTAMIS337] = TRTAMIS:RP.AssessableIncomeOtherThanCapitalGains:bafpr1.02.40:Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount
            [TRTAMIS338] = TRTAMIS:RP.AssessableIncomeCapitalGains:bafpr1.02.40:Income.NonConcessionalManagedInvestmentTrust.Amount
            [TRTAMIS339] = TRTAMIS:RP.AssessableIncomeCapitalGains:bafpr1.02.40:Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount

            Technical Business Rule Format:
            (^TRTAMIS314 > 0 OR ^TRTAMIS327 > 0) AND (^TRTAMIS261 + ^TRTAMIS325 + ^TRTAMIS262 + ^TRTAMIS326) > (^TRTAMIS260 + ^TRTAMIS336 + ^TRTAMIS337 + ^TRTAMIS324 + ^TRTAMIS338 + ^TRTAMIS339)
    
            Data Elements:
    
            ^TRTAMIS314 = RP.AssessableIncomeOtherThanCapitalGains:Income.TrustComponent.Amount
    
            ^TRTAMIS260 = RP.AssessableIncomeOtherThanCapitalGains:Income.Taxable.Amount
    
            ^TRTAMIS261 = RP.AssessableIncomeOtherThanCapitalGains.Direct:Expense.DeductionsTotal.Amount
    
            ^TRTAMIS262 = RP.AssessableIncomeOtherThanCapitalGains.Indirect:Expense.DeductionsOther.Amount
    
            ^TRTAMIS324 = RP.AssessableIncomeCapitalGains:Income.CapitalGainsNetCredit.Amount
    
            ^TRTAMIS325 = RP.AssessableIncomeCapitalGains.Direct:Expense.DeductionsTotal.Amount
    
            ^TRTAMIS326 = RP.AssessableIncomeCapitalGains.Indirect:Expense.DeductionsOther.Amount
    
            ^TRTAMIS327 = RP.AssessableIncomeCapitalGains:Income.TrustComponent.Amount
    
            ^TRTAMIS336 = RP.AssessableIncomeOtherThanCapitalGains:Income.NonConcessionalManagedInvestmentTrust.Amount
    
            ^TRTAMIS337 = RP.AssessableIncomeOtherThanCapitalGains:Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount
    
            ^TRTAMIS338 = RP.AssessableIncomeCapitalGains:Income.NonConcessionalManagedInvestmentTrust.Amount
    
            ^TRTAMIS339 = RP.AssessableIncomeCapitalGains:Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount
            */
            assertion = ((report.TRTAMIS314.GetValueOrDefault() > 0 || report.TRTAMIS327.GetValueOrDefault() > 0) && report.TRTAMIS261.GetValueOrDefault() + report.TRTAMIS325.GetValueOrDefault() + report.TRTAMIS262.GetValueOrDefault() + report.TRTAMIS326.GetValueOrDefault() > report.TRTAMIS260.GetValueOrDefault() + report.TRTAMIS336.GetValueOrDefault() + report.TRTAMIS337.GetValueOrDefault() + report.TRTAMIS324.GetValueOrDefault() + report.TRTAMIS338.GetValueOrDefault() + report.TRTAMIS339.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000603",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Trust Components amount is incorrect",
                    LongDescription = @"For assessable income details, Trust Components amounts must not be provided when the sum of all Direct Deductions and Other Deductions amounts exceeds the sum of all Assessable Income amounts.",
                    Location = "/xbrli:xbrl/tns:Income.TrustComponent.Amount[@contextRef='" + report.RPAssessableIncomeOtherThanCapitalGainsId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000603" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS314", Value = GetValueOrEmpty(report.TRTAMIS314) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS327", Value = GetValueOrEmpty(report.TRTAMIS327) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS261", Value = GetValueOrEmpty(report.TRTAMIS261) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS325", Value = GetValueOrEmpty(report.TRTAMIS325) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS262", Value = GetValueOrEmpty(report.TRTAMIS262) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS326", Value = GetValueOrEmpty(report.TRTAMIS326) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS260", Value = GetValueOrEmpty(report.TRTAMIS260) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS336", Value = GetValueOrEmpty(report.TRTAMIS336) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS337", Value = GetValueOrEmpty(report.TRTAMIS337) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS324", Value = GetValueOrEmpty(report.TRTAMIS324) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS338", Value = GetValueOrEmpty(report.TRTAMIS338) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS339", Value = GetValueOrEmpty(report.TRTAMIS339) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000603

            #region VR.ATO.TRTAMIS.000604

            /*  VR.ATO.TRTAMIS.000604
            Monetary field, must be unsigned, maximum of 11 chars, no decimals
    
            Legacy Rule Format:
            IF [TRTAMIS324] <> NULL AND [TRTAMIS324] <> MONETARY(U,11,0)
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [TRTAMIS324] = TRTAMIS:RP.AssessableIncomeCapitalGains:bafpr1.02.30:Income.CapitalGainsNetCredit.Amount

            Technical Business Rule Format:
            NotMonetary(^TRTAMIS324, 'U', 11, 0)
    
            Data Elements:
    
            ^TRTAMIS324 = RP.AssessableIncomeCapitalGains:Income.CapitalGainsNetCredit.Amount
            */
            assertion = NotMonetary(report.TRTAMIS324, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000604",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Net Capital Gain is not in a valid monetary format",
                    LongDescription = @"Net Capital Gain must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Income.CapitalGainsNetCredit.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000604" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS324", Value = GetValueOrEmpty(report.TRTAMIS324) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000604

            #region VR.ATO.TRTAMIS.000605

            /*  VR.ATO.TRTAMIS.000605
            Monetary field, must be unsigned, maximum of 11 chars, no decimals
    
            Legacy Rule Format:
            IF [TRTAMIS325] <> NULL AND [TRTAMIS325] <> MONETARY(U,11,0)
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [TRTAMIS325] = TRTAMIS:RP.AssessableIncomeCapitalGains.Direct:bafpr1.02.12:Expense.DeductionsTotal.Amount

            Technical Business Rule Format:
            NotMonetary(^TRTAMIS325, 'U', 11, 0)
    
            Data Elements:
    
            ^TRTAMIS325 = RP.AssessableIncomeCapitalGains.Direct:Expense.DeductionsTotal.Amount
            */
            assertion = NotMonetary(report.TRTAMIS325, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000605",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Direct Deductions is not in a valid monetary format",
                    LongDescription = @"Direct Deductions must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Expense.DeductionsTotal.Amount[@contextRef='" + report.RPAssessableIncomeCapitalGainsDirectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000605" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS325", Value = GetValueOrEmpty(report.TRTAMIS325) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000605

            #region VR.ATO.TRTAMIS.000606

            /*  VR.ATO.TRTAMIS.000606
            Monetary field, must be unsigned, maximum of 11 chars, no decimals
    
            Legacy Rule Format:
            IF [TRTAMIS326] <> NULL AND [TRTAMIS326] <> MONETARY(U,11,0)
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [TRTAMIS326] = TRTAMIS:RP.AssessableIncomeCapitalGains.Indirect:bafpr1.02.29:Expense.DeductionsOther.Amount

            Technical Business Rule Format:
            NotMonetary(^TRTAMIS326, 'U', 11, 0)
    
            Data Elements:
    
            ^TRTAMIS326 = RP.AssessableIncomeCapitalGains.Indirect:Expense.DeductionsOther.Amount
            */
            assertion = NotMonetary(report.TRTAMIS326, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000606",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other Deductions is not in a valid monetary format",
                    LongDescription = @"Other Deductions must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Expense.DeductionsOther.Amount[@contextRef='" + report.RPAssessableIncomeCapitalGainsIndirectId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000606" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS326", Value = GetValueOrEmpty(report.TRTAMIS326) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000606

            #region VR.ATO.TRTAMIS.000607

            /*  VR.ATO.TRTAMIS.000607
            Monetary field, must be unsigned, maximum of 11 chars, no decimals
    
            Legacy Rule Format:
            IF [TRTAMIS327] <> NULL AND [TRTAMIS327] <> MONETARY(U,11,0)
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [TRTAMIS327] = TRTAMIS:RP.AssessableIncomeCapitalGains:bafpr1.02.29:Income.TrustComponent.Amount

            Technical Business Rule Format:
            NotMonetary(^TRTAMIS327, 'U', 11, 0)
    
            Data Elements:
    
            ^TRTAMIS327 = RP.AssessableIncomeCapitalGains:Income.TrustComponent.Amount
            */
            assertion = NotMonetary(report.TRTAMIS327, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000607",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Trust Components is not in a valid monetary format",
                    LongDescription = @"Trust Components must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Income.TrustComponent.Amount[@contextRef='" + report.RPAssessableIncomeCapitalGainsId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000607" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS327", Value = GetValueOrEmpty(report.TRTAMIS327) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000607

            #region VR.ATO.TRTAMIS.000608

            /*  VR.ATO.TRTAMIS.000608
            Monetary field, must be unsigned, maximum of 11 chars, no decimals
    
            Legacy Rule Format:
            IF [TRTAMIS328] <> NULL AND [TRTAMIS328] <> MONETARY(U,11,0)
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [TRTAMIS328] = TRTAMIS:RP.AssessableIncomeCapitalGains:bafpr1.02.27:Income.UnderstatedEntitlement.Amount

            Technical Business Rule Format:
            NotMonetary(^TRTAMIS328, 'U', 11, 0)
    
            Data Elements:
    
            ^TRTAMIS328 = RP.AssessableIncomeCapitalGains:Income.UnderstatedEntitlement.Amount
            */
            assertion = NotMonetary(report.TRTAMIS328, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000608",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total Unders is not in a valid monetary format",
                    LongDescription = @"Total Unders must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Income.UnderstatedEntitlement.Amount[@contextRef='" + report.RPAssessableIncomeCapitalGainsId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000608" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS328", Value = GetValueOrEmpty(report.TRTAMIS328) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000608

            #region VR.ATO.TRTAMIS.000609

            /*  VR.ATO.TRTAMIS.000609
            Monetary field, must be unsigned, maximum of 11 chars, no decimals
    
            Legacy Rule Format:
            IF [TRTAMIS329] <> NULL AND [TRTAMIS329] <> MONETARY(U,11,0)
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [TRTAMIS329] = TRTAMIS:RP.AssessableIncomeCapitalGains:bafpr1.02.27:Expense.Overstated.Amount

            Technical Business Rule Format:
            NotMonetary(^TRTAMIS329, 'U', 11, 0)
    
            Data Elements:
    
            ^TRTAMIS329 = RP.AssessableIncomeCapitalGains:Expense.Overstated.Amount
            */
            assertion = NotMonetary(report.TRTAMIS329, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000609",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total Overs is not in a valid monetary format",
                    LongDescription = @"Total Overs must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Expense.Overstated.Amount[@contextRef='" + report.RPAssessableIncomeCapitalGainsId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000609" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS329", Value = GetValueOrEmpty(report.TRTAMIS329) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000609

            #region VR.ATO.TRTAMIS.000610

            /*  VR.ATO.TRTAMIS.000610
            Monetary field, must be unsigned, maximum of 11 chars, no decimals
    
            Legacy Rule Format:
            IF [TRTAMIS330] <> NULL AND [TRTAMIS330] <> MONETARY(U,11,0)
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [TRTAMIS330] = TRTAMIS:RP.AssessableIncomeCapitalGains:bafpr1.02.23:Income.Net.Amount

            Technical Business Rule Format:
            NotMonetary(^TRTAMIS330, 'U', 11, 0)
    
            Data Elements:
    
            ^TRTAMIS330 = RP.AssessableIncomeCapitalGains:Income.Net.Amount
            */
            assertion = NotMonetary(report.TRTAMIS330, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000610",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Determined Trust Components is not in a valid monetary format",
                    LongDescription = @"Determined Trust Components must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Income.Net.Amount[@contextRef='" + report.RPAssessableIncomeCapitalGainsId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000610" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS330", Value = GetValueOrEmpty(report.TRTAMIS330) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000610

            #region VR.ATO.TRTAMIS.000611

            /*  VR.ATO.TRTAMIS.000611
            Monetary field, must be unsigned, maximum of 11 chars, no decimals
    
            Legacy Rule Format:
            IF [TRTAMIS331] <> NULL AND [TRTAMIS331] <> MONETARY(U,11,0)
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [TRTAMIS331] = TRTAMIS:RP.AssessableIncomeCapitalGains:bafpr1.02.29:Expense.CarryForwardTrustComponentDeficit.Amount

            Technical Business Rule Format:
            NotMonetary(^TRTAMIS331, 'U', 11, 0)
    
            Data Elements:
    
            ^TRTAMIS331 = RP.AssessableIncomeCapitalGains:Expense.CarryForwardTrustComponentDeficit.Amount
            */
            assertion = NotMonetary(report.TRTAMIS331, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000611",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Carry-Forward Trust Component Deficits is not in a valid monetary format",
                    LongDescription = @"Carry-Forward Trust Component Deficit must be unsigned, not exceed 11 digits and not contain a decimal point or any other characters",
                    Location = "/xbrli:xbrl/tns:Expense.CarryForwardTrustComponentDeficit.Amount[@contextRef='" + report.RPAssessableIncomeCapitalGainsId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000611" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS331", Value = GetValueOrEmpty(report.TRTAMIS331) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000611

            #region VR.ATO.TRTAMIS.000612

            /*  VR.ATO.TRTAMIS.000612
            Invalid context. The number of RP.AssessableIncomeOtherThanCapitalGains contexts cannot exceed 1.
    
            Legacy Rule Format:
            IF COUNT(RP.AssessableIncomeOtherThanCapitalGains) > 1
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.AssessableIncomeOtherThanCapitalGains
            */
            assertion = (Count(report.RPAssessableIncomeOtherThanCapitalGainsCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000612",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of RP.AssessableIncomeOtherThanCapitalGains contexts cannot exceed 1.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPAssessableIncomeOtherThanCapitalGainsLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000612" } },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000612

            #region VR.ATO.TRTAMIS.000613

            /*  VR.ATO.TRTAMIS.000613
            Invalid context. The number of RP.AssessableIncomeOtherThanCapitalGains.Direct contexts cannot exceed 1.
    
            Legacy Rule Format:
            IF COUNT(RP.AssessableIncomeOtherThanCapitalGains.Direct) > 1
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP.AssessableIncomeOtherThanCapitalGains:RP.AssessableIncomeOtherThanCapitalGains.Direct
            */
            assertion = (Count(report.RPAssessableIncomeOtherThanCapitalGainsDirectCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000613",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of RP.AssessableIncomeOtherThanCapitalGains.Direct contexts cannot exceed 1.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPAssessableIncomeOtherThanCapitalGainsDirectLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000613" } },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000613

            #region VR.ATO.TRTAMIS.000614

            /*  VR.ATO.TRTAMIS.000614
            Invalid context. The number of RP.AssessableIncomeOtherThanCapitalGains.Indirect contexts cannot exceed 1.
    
            Legacy Rule Format:
            IF COUNT(RP.AssessableIncomeOtherThanCapitalGains.Indirect) > 1
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP.AssessableIncomeOtherThanCapitalGains:RP.AssessableIncomeOtherThanCapitalGains.Indirect
            */
            assertion = (Count(report.RPAssessableIncomeOtherThanCapitalGainsIndirectCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000614",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of RP.AssessableIncomeOtherThanCapitalGains.Indirect contexts cannot exceed 1.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPAssessableIncomeOtherThanCapitalGainsIndirectLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000614" } },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000614

            #region VR.ATO.TRTAMIS.000615

            /*  VR.ATO.TRTAMIS.000615
            Invalid context. The number of RP.AssessableIncomeCapitalGains contexts cannot exceed 1.
    
            Legacy Rule Format:
            IF COUNT(RP.AssessableIncomeCapitalGains) > 1
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.AssessableIncomeCapitalGains
            */
            assertion = (Count(report.RPAssessableIncomeCapitalGainsCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000615",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of RP.AssessableIncomeCapitalGains contexts cannot exceed 1.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPAssessableIncomeCapitalGainsLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000615" } },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000615

            #region VR.ATO.TRTAMIS.000616

            /*  VR.ATO.TRTAMIS.000616
            Invalid context. The number of RP.AssessableIncomeCapitalGains.Direct contexts cannot exceed 1.
    
            Legacy Rule Format:
            IF COUNT(RP.AssessableIncomeCapitalGains.Direct) > 1
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP.AssessableIncomeCapitalGains:RP.AssessableIncomeCapitalGains.Direct
            */
            assertion = (Count(report.RPAssessableIncomeCapitalGainsDirectCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000616",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of RP.AssessableIncomeCapitalGains.Direct contexts cannot exceed 1.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPAssessableIncomeCapitalGainsDirectLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000616" } },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000616

            #region VR.ATO.TRTAMIS.000617

            /*  VR.ATO.TRTAMIS.000617
            Invalid context. The number of RP.AssessableIncomeCapitalGains.Indirect contexts cannot exceed 1.
    
            Legacy Rule Format:
            IF COUNT(RP.AssessableIncomeCapitalGains.Indirect) > 1
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP.AssessableIncomeCapitalGains:RP.AssessableIncomeCapitalGains.Indirect
            */
            assertion = (Count(report.RPAssessableIncomeCapitalGainsIndirectCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000617",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of RP.AssessableIncomeCapitalGains.Indirect contexts cannot exceed 1.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPAssessableIncomeCapitalGainsIndirectLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000617" } },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000617

            #region VR.ATO.TRTAMIS.000618

            /*  VR.ATO.TRTAMIS.000618
            Is this the Final Schedule for this Class (Where the Schedule Relates to a Separate Class)? must be provided when Name of Attribution Managed Investment Trust (AMIT) Class is provided
    
            Legacy Rule Format:
            IF [TRTAMIS317] = NULL AND [TRTAMIS313] <> NULLORBLANK
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [TRTAMIS313] = TRTAMIS:RP:rvctc1.02.22:IncomeTax.AttributionManagedInvestmentTrustClass.Description
            [TRTAMIS317] = TRTAMIS:RP:pyin.02.28:Lodgment.FinalReturn.Indicator

            Technical Business Rule Format:
            ^TRTAMIS317 = NULL AND ^TRTAMIS313 <> BLANK
    
            Data Elements:
    
            ^TRTAMIS317 = RP:Lodgment.FinalReturn.Indicator
    
            ^TRTAMIS313 = RP:IncomeTax.AttributionManagedInvestmentTrustClass.Description
            */
            assertion = (report.TRTAMIS317 == null && string.IsNullOrWhiteSpace(report.TRTAMIS313) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000618",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Is this the Final Schedule for this class (where the schedule relates to a separate class)?' must be provided",
                    LongDescription = @"Is this the Final Schedule for this Class (Where the Schedule Relates to a Separate Class)? must be provided when Name of Attribution Managed Investment Trust (AMIT) Class is provided",
                    Location = "/xbrli:xbrl/tns:Lodgment.FinalReturn.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000618" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS317", Value = GetValueOrEmpty(report.TRTAMIS317) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS313", Value = GetValueOrEmpty(report.TRTAMIS313) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000618

            #region VR.ATO.TRTAMIS.000619

            /*  VR.ATO.TRTAMIS.000619
            Monetary field, must be unsigned, maximum of 15 chars, no decimals
    
            Legacy Rule Format:
            IF [TRTAMIS333] <> NULL AND [TRTAMIS333] <> MONETARY(U,15,0)
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [TRTAMIS333] = TRTAMIS:RP(Instant):bafpo1.02.13:Assets.Total.Amount

            Technical Business Rule Format:
            NotMonetary(^TRTAMIS333, 'U', 15, 0)
    
            Data Elements:
    
            ^TRTAMIS333 = RP(Instant):Assets.Total.Amount
            */
            assertion = NotMonetary(report.TRTAMIS333, @"U", 15, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000619",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total Assets is not in a valid monetary format",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Assets.Total.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000619" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS333", Value = GetValueOrEmpty(report.TRTAMIS333) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000619

            #region VR.ATO.TRTAMIS.000621

            /*  VR.ATO.TRTAMIS.000621
            Monetary field, must be unsigned, maximum of 15 chars, no decimals
    
            Legacy Rule Format:
            IF [TRTAMIS334] <> NULL AND [TRTAMIS334] <> MONETARY(U,15,0)
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [TRTAMIS334] = TRTAMIS:RP(Instant):bafpo1.02.13:Liabilities.Total.Amount

            Technical Business Rule Format:
            NotMonetary(^TRTAMIS334, 'U', 15, 0)
    
            Data Elements:
    
            ^TRTAMIS334 = RP(Instant):Liabilities.Total.Amount
            */
            assertion = NotMonetary(report.TRTAMIS334, @"U", 15, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000621",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total Liabilities is not in a valid monetary format",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Liabilities.Total.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000621" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS334", Value = GetValueOrEmpty(report.TRTAMIS334) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000621

            #region VR.ATO.TRTAMIS.000623

            /*  VR.ATO.TRTAMIS.000623
            Monetary field, must be unsigned, maximum of 11 chars, no decimals
    
            Legacy Rule Format:
            IF [TRTAMIS335] <> NULL AND [TRTAMIS335] <> MONETARY(U,11,0)
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [TRTAMIS335] = TRTAMIS:RP:bafpo6.02.09:Liabilities.ThinCapitalisation.DeductionDebt.Amount

            Technical Business Rule Format:
            NotMonetary(^TRTAMIS335, 'U', 11, 0)
    
            Data Elements:
    
            ^TRTAMIS335 = RP:Liabilities.ThinCapitalisation.DeductionDebt.Amount
            */
            assertion = NotMonetary(report.TRTAMIS335, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000623",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Debt Deduction is not in a valid monetary format",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Liabilities.ThinCapitalisation.DeductionDebt.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000623" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS335", Value = GetValueOrEmpty(report.TRTAMIS335) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000623

            #region VR.ATO.TRTAMIS.000630

            /*  VR.ATO.TRTAMIS.000630
            Monetary field may be signed, maximum of 11 chars, no decimals
    
            Legacy Rule Format:
            IF [TRTAMIS336] <> NULL AND [TRTAMIS336] <> MONETARY(S,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [TRTAMIS336] = TRTAMIS:RP.AssessableIncomeOtherThanCapitalGains:bafpr1.02.40:Income.NonConcessionalManagedInvestmentTrust.Amount

            Technical Business Rule Format:
            NotMonetary(^TRTAMIS336, 'S', 11, 0)
    
            Data Elements:
    
            ^TRTAMIS336 = RP.AssessableIncomeOtherThanCapitalGains:Income.NonConcessionalManagedInvestmentTrust.Amount
            */
            assertion = NotMonetary(report.TRTAMIS336, @"S", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000630",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-Concessional MIT Income not in a valid monetary format.",
                    Location = "/xbrli:xbrl/tns:Income.NonConcessionalManagedInvestmentTrust.Amount[@contextRef='" + report.RPAssessableIncomeOtherThanCapitalGainsId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000630" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS336", Value = GetValueOrEmpty(report.TRTAMIS336) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000630

            #region VR.ATO.TRTAMIS.000631

            /*  VR.ATO.TRTAMIS.000631
            Monetary field may be signed, maximum of 11 chars, no decimals
    
            Legacy Rule Format:
            IF [TRTAMIS337] <> NULL AND [TRTAMIS337] <> MONETARY(S,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [TRTAMIS337] = TRTAMIS:RP.AssessableIncomeOtherThanCapitalGains:bafpr1.02.40:Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount

            Technical Business Rule Format:
            NotMonetary(^TRTAMIS337, 'S', 11, 0)
    
            Data Elements:
    
            ^TRTAMIS337 = RP.AssessableIncomeOtherThanCapitalGains:Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount
            */
            assertion = NotMonetary(report.TRTAMIS337, @"S", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000631",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Excluded from non-Concessional MIT Income not in a valid monetary format.",
                    Location = "/xbrli:xbrl/tns:Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount[@contextRef='" + report.RPAssessableIncomeOtherThanCapitalGainsId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000631" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS337", Value = GetValueOrEmpty(report.TRTAMIS337) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000631

            #region VR.ATO.TRTAMIS.000632

            /*  VR.ATO.TRTAMIS.000632
            Monetary field may be signed, maximum of 11 chars, no decimals
    
            Legacy Rule Format:
            IF [TRTAMIS338] <> NULL AND [TRTAMIS338] <> MONETARY(S,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [TRTAMIS338] = TRTAMIS:RP.AssessableIncomeCapitalGains:bafpr1.02.40:Income.NonConcessionalManagedInvestmentTrust.Amount

            Technical Business Rule Format:
            NotMonetary(^TRTAMIS338, 'S', 11, 0)
    
            Data Elements:
    
            ^TRTAMIS338 = RP.AssessableIncomeCapitalGains:Income.NonConcessionalManagedInvestmentTrust.Amount
            */
            assertion = NotMonetary(report.TRTAMIS338, @"S", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000632",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-Concessional MIT Income not in a valid monetary format.",
                    Location = "/xbrli:xbrl/tns:Income.NonConcessionalManagedInvestmentTrust.Amount[@contextRef='" + report.RPAssessableIncomeCapitalGainsId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000632" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS338", Value = GetValueOrEmpty(report.TRTAMIS338) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000632

            #region VR.ATO.TRTAMIS.000633

            /*  VR.ATO.TRTAMIS.000633
            Monetary field may be signed, maximum of 11 chars, no decimals
    
            Legacy Rule Format:
            IF [TRTAMIS339] <> NULL AND [TRTAMIS339] <> MONETARY(S,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF
            
            [TRTAMIS339] = TRTAMIS:RP.AssessableIncomeCapitalGains:bafpr1.02.40:Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount

            Technical Business Rule Format:
            NotMonetary(^TRTAMIS339, 'S', 11, 0)
    
            Data Elements:
    
            ^TRTAMIS339 = RP.AssessableIncomeCapitalGains:Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount
            */
            assertion = NotMonetary(report.TRTAMIS339, @"S", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMIS.000633",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Excluded from non-Concessional MIT Income not in a valid monetary format.",
                    Location = "/xbrli:xbrl/tns:Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount[@contextRef='" + report.RPAssessableIncomeCapitalGainsId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000633" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRTAMIS339", Value = GetValueOrEmpty(report.TRTAMIS339) });

                response.Add(processMessage);
            }
            #endregion // VR.ATO.TRTAMIS.000633

            foreach (ProcessMessageDocument currentProcessMessage in response)
            {
                if (currentProcessMessage.Parameters != null)
                {
                    foreach (ProcessMessageParameter currentParameter in currentProcessMessage.Parameters)
                    {
                        if (string.IsNullOrEmpty(currentParameter.Name))
                        {
                            currentParameter.Name = _emptyParameterValue;
                        }

                        if (currentParameter.Name.Length > _maxParameterNameLength)
                        {
                            currentParameter.Name = currentParameter.Name.Substring(0, _maxParameterNameLength - 1);
                        }

                        if (string.IsNullOrEmpty(currentParameter.Value))
                        {
                            currentParameter.Value = _emptyParameterValue;
                        }

                        if (currentParameter.Value.Length > _maxParameterValueLength)
                        {
                            currentParameter.Value = currentParameter.Value.Substring(0, _maxParameterValueLength - 1);
                        }
                    }
                }
            }

            return response;
        }


    }
}