
using System;
using System.Collections.Generic;
using Au.Gov.Sbr.Xbrl.Document;
using Au.Gov.Sbr.Xbrl.Metadata;
using System.Linq;
using System.Xml;
using System.Text.RegularExpressions;
using DataContracts;

namespace Ato.EN.IntegrationServices.CodeGenerationCTR
{
    public class CTR2025Consumer
    {

        private static Regex StripWhitespace = new Regex(@"\s*");

        #region Helper/Validation Methods
        private static bool? ToBoolean(string str)
        {
            bool returnValue;
            string value = (str ?? "").Trim();
            if (value == "0" || value == "1")
                return Convert.ToBoolean(Convert.ToInt32(value));
            
            if (Boolean.TryParse(value, out returnValue))
                return returnValue;
            return null;
        }

        private static string GetValueOrEmpty(bool? val)
        {
            return (val.HasValue) ? val.ToString().ToLower() : string.Empty;
        }

        private static string GetValueOrEmptyDate(DateTime? val)
        {
            return (val.HasValue) ? val.Value.ToString("yyyy-MM-dd") : string.Empty;
        }

        private static string GetValueOrEmptyDateTime(DateTime? val)
        {
            return (val.HasValue) ? val.Value.ToString("yyyy-MM-ddTHH:mm:ssZ") : string.Empty;
        }
				
        private static string GetValueOrEmptyDate(string val)
        {
            DateTime dateTimeOut;
            if (DateTime.TryParse(val, out dateTimeOut))
            {
                return dateTimeOut.ToString("yyyy-MM-dd");
            }
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmptyDateTime(string val)
        {
            DateTime dateTimeOut;
            if (DateTime.TryParse(val, out dateTimeOut))
            {
                return dateTimeOut.ToString("yyyy-MM-ddTHH:mm:ssZ");
            }
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmpty(string val)
        {
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmpty(decimal? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        private static string GetValueOrEmpty(int? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        /// <summary>
        /// Derives the SBR element location.
        /// </summary>
        /// <param name="sbrElement">The SBR element.</param>
        /// <param name="addOccurrenceIndex">if set to <c>true</c> [add occurrence the occurrence index to the path].</param>
        /// <returns></returns>
        private static string DeriveSbrElementLocation(SBRElement sbrElement, bool addOccurrenceIndex = false)
        {
            if (sbrElement == null) return "/xbrli:xbrl";
            int xpathOccurrence = sbrElement.OccurrenceIndex + 1;
            string location;

            if (addOccurrenceIndex)
            {
                location = "/" + sbrElement.NamespacePrefix + ":" + sbrElement.Name + "[" +
                           xpathOccurrence + "]";
            }
            else
            {
                location = "/" + sbrElement.NamespacePrefix + ":" + sbrElement.Name;
            }

            if (sbrElement.IsInTuple)
            {
                location = DeriveSbrElementLocation(sbrElement.ParentTuple, true) + location;
            }
            else
            {
                location = "/xbrli:xbrl" + location;
            }

            return location;
        }

        private static bool IsMatch(int? field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(Convert.ToString(field.Value), expression, options);
        }

        private static bool IsMatch(string field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(field, expression, options);
        }

        public DateTime AsDate(string dateAsString)
        {
            DateTime response = DateTime.MinValue;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date;
            }

            return response;
        }

         #endregion

        public CTR2025 Consume(SBRDocument sbrDocument)
        {
            List<ProcessMessageDocument> errors;
            return Consume(sbrDocument, false, out errors);
        }

        /// <summary>
        /// Allows for validation of rules that can not be generated and must run at a lower level.
        /// A sub-class will need to be created with an override for this method - which does nothing in its base implementation.
        /// </summary>
        /// <param name="sbrDocument">The SBR document.</param>
        /// <param name="populatedReportElementsMap">The populated report elements map can be used to find data elements and tuples.</param>
        /// <param name="contextArray">The context array can be used to find context values.</param>
        /// <param name="errors">The errors collection is to be populated with validation results.</param>
        protected virtual void ValidateUnsupportedRules(SBRDocument sbrDocument, IDictionary<string, SBRElement> populatedReportElementsMap, SBRContext[] contextArray, List<ProcessMessageDocument> errors)
        {

        }

        public CTR2025 Consume(SBRDocument sbrDocument, bool validationMode, out List<ProcessMessageDocument> errors)
        {
            CTR2025 report;
            report = new CTR2025();
            errors = new List<ProcessMessageDocument>();

            SBRContext[] contextArray;
            IDictionary<string, SBRElement> populatedReportElementsMap;

            contextArray = sbrDocument.ContextsMap.Values.ToArray();
            populatedReportElementsMap = sbrDocument.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();

            if (validationMode) ValidateUnsupportedRules(sbrDocument, populatedReportElementsMap, contextArray, errors);

            /// <summary>
            /// Current element being extracted
            /// </summary>
            SBRElement currentValue = null;
            /// <summary>
            /// Current elements being extracted
            /// </summary>
            IList<SBRElement> currentValues = null;
            /// <summary>
            /// Current occurrences being extracted
            /// </summary>
            IList<SBRElement> currentOccurrences = null;
            /// <summary>
            /// Working Variable for date time variables
            /// </summary>
            DateTime currentDateTimeValue;
            /// <summary>
            /// Working Variable for boolean variables
            /// </summary>
            bool currentBooleanValue;
            /// <summary>
            /// Working Variable for decimal variables
            /// </summary>
            decimal currentDecimalValue;
            /// <summary>
            /// Working Variable for integer variables
            /// </summary>
            int currentIntValue;
            /// <summary>
            /// Working Variable for long variables
            /// </summary>
            long currentLongValue;
            XmlDocumentFragment currentSegment;
            XmlNamespaceManager currentNamespaceManager;
            XmlNode currentNode;
    
            #region sbrINTContext
            IEnumerable<SBRContext> sbrINTContexts = null;
            SBRContext sbrINTContext = null;
    
            sbrINTContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 1 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Intermediary")) )).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedINTElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrINTContexts != null && sbrINTContexts.Count() > 0)
            {
                sbrINTContext = sbrINTContexts.First();
        
                report.INTId = sbrINTContext.Id;
                report.INTIdentifier = sbrINTContext.EntityIdentifier;
                report.INTIdentifierScheme = sbrINTContext.EntityScheme;
        
                report.INTCount = sbrINTContexts.Count();
                report.INTExists = true;
                report.INTOccurrenceIndex = Array.IndexOf(contextArray, sbrINTContext) + 1;
                report.INTLastOccurrenceIndex = Array.IndexOf(contextArray, sbrINTContexts.Last()) + 1;
        

                if (report.INTIdentifierScheme == "http://www.ato.gov.au/abn") report.INTIdentifierABN = report.INTIdentifier;
    
                report.INTStartDate = ((SBRDurationPeriod)sbrINTContext.Period).DurationStartAsDateObject;
                report.INTEndDate = ((SBRDurationPeriod)sbrINTContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedINTElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrINTContexts.Count(); i++)
                {
                    sbrINTContext = sbrINTContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrINTContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedINTElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedINTElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedINTElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
        
                    IEnumerable<SBRElement> sbrDeclarationCollection;
                    if (populatedReportElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Declaration", out currentValue))
                        sbrDeclarationCollection = currentValue.Occurrences;
                    else
                        sbrDeclarationCollection = null;
            
                    #region sbrDeclarationCollection
            
                    if (sbrDeclarationCollection != null)
                    {
                        report.DeclarationCollectionExists = true;
                        report.DeclarationCollectionCount = sbrDeclarationCollection.Count();
                
                        IEnumerable<SBRElement> sbrDeclarationCollection_TrueAndCorrect_INTCollection;
                        sbrDeclarationCollection_TrueAndCorrect_INTCollection = sbrDeclarationCollection.Where(
                            ce => ce.ChildElements != null && 
                            ce.ChildElements.Any(te => te.Name == "Declaration.StatementType.Code" && te.Value == "TrueAndCorrect" &&
                            sbrINTContexts.Contains(te.Context)));
                        if (sbrDeclarationCollection_TrueAndCorrect_INTCollection != null && sbrDeclarationCollection_TrueAndCorrect_INTCollection.Count() > 0)
                        {
                            SBRElement sbrDeclarationCollection_TrueAndCorrect_INT;
                            sbrDeclarationCollection_TrueAndCorrect_INT = sbrDeclarationCollection_TrueAndCorrect_INTCollection.ElementAt(0);
                
                            IDictionary<string, SBRElement> sbrDeclarationCollection_TrueAndCorrect_INTTupleElementMap;
                            sbrDeclarationCollection_TrueAndCorrect_INTTupleElementMap = sbrDeclarationCollection_TrueAndCorrect_INT.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                
                            report.DeclarationCollection_TrueAndCorrect_INTExists = true;
                            report.DeclarationCollection_TrueAndCorrect_INTCount = sbrDeclarationCollection_TrueAndCorrect_INTCollection.Count();
                            report.DeclarationCollection_TrueAndCorrect_INTOccurrenceIndex = sbrDeclarationCollection_TrueAndCorrect_INT.OccurrenceIndex + 1;
                            report.DeclarationCollection_TrueAndCorrect_INTLastOccurrenceIndex = sbrDeclarationCollection_TrueAndCorrect_INTCollection.Last().OccurrenceIndex + 1;
                    
                            if (sbrDeclarationCollection_TrueAndCorrect_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Declaration.StatementType.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR261 = currentValue.Value;
                            }
                    
                            if (sbrDeclarationCollection_TrueAndCorrect_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Declaration.StatementAccepted.Indicator", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR237 = ToBoolean(currentValue.Value);
                            }
                    
                            if (sbrDeclarationCollection_TrueAndCorrect_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Declaration.Statement.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR238 = currentValue.Value;
                            }
                    
                            if (sbrDeclarationCollection_TrueAndCorrect_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Declaration.Signature.Date", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR234 = DateTime.Parse(currentValue.Value);
                            }
                    
                            if (sbrDeclarationCollection_TrueAndCorrect_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Declaration.SignatoryIdentifier.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR281 = currentValue.Value;
                            }
                
                            IEnumerable<SBRElement> sbrDeclaration_PersonUnstructuredNameCollection;
                            if (sbrDeclarationCollection_TrueAndCorrect_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}PersonUnstructuredName", out currentValue))
                                sbrDeclaration_PersonUnstructuredNameCollection = currentValue.Occurrences;
                            else
                                sbrDeclaration_PersonUnstructuredNameCollection = null;
                    
                            #region sbrDeclaration_PersonUnstructuredNameCollection
                    
                            if (sbrDeclaration_PersonUnstructuredNameCollection != null)
                            {
                                report.Declaration_PersonUnstructuredNameCollectionExists = true;
                                report.Declaration_PersonUnstructuredNameCollectionCount = sbrDeclaration_PersonUnstructuredNameCollection.Count();
                        
                                IEnumerable<SBRElement> sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTCollection;
                                sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTCollection = sbrDeclaration_PersonUnstructuredNameCollection.Where(
                                    ce => ce.ChildElements != null && 
                                    ce.ChildElements.Any(te => te.Name == "PersonUnstructuredName.Usage.Code" && te.Value == "DeclarationSignatory" &&
                                    sbrINTContexts.Contains(te.Context)));
                                if (sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTCollection != null && sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTCollection.Count() > 0)
                                {
                                    SBRElement sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INT;
                                    sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INT = sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTCollection.ElementAt(0);
                        
                                    IDictionary<string, SBRElement> sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTTupleElementMap;
                                    sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTTupleElementMap = sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INT.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                        
                                    report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTExists = true;
                                    report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTCount = sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTCollection.Count();
                                    report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTOccurrenceIndex = sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INT.OccurrenceIndex + 1;
                                    report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTLastOccurrenceIndex = sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTCollection.Last().OccurrenceIndex + 1;
                            
                                    if (sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}PersonUnstructuredName.Usage.Code", out currentValue))
                                    {
                                        if (!currentValue.IsNil) report.CTR292 = currentValue.Value;
                                    }
                            
                                    if (sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}PersonUnstructuredName.FullName.Text", out currentValue))
                                    {
                                        if (!currentValue.IsNil) report.CTR291 = currentValue.Value;
                                    }
                                }  
                            } 
                            #endregion End of Tuple sbrDeclaration_PersonUnstructuredNameCollection
                            if (validationMode) ValidateINTDeclaration_PersonUnstructuredNameCollection(errors, sbrDeclaration_PersonUnstructuredNameCollection, sbrINTContexts, report);
                    
                        }  
                    } 
                    #endregion End of Tuple sbrDeclarationCollection
                    if (validationMode) ValidateINTDeclarationCollection(errors, sbrDeclarationCollection, sbrINTContexts, report);
            
            
                    if (sbrPopulatedINTElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Identifiers.TaxAgentClientReference.Text", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR213 = currentValues[0].Value;
                    }
        
                    IEnumerable<SBRElement> sbrElectronicContactTelephoneCollection;
                    if (populatedReportElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}ElectronicContactTelephone", out currentValue))
                        sbrElectronicContactTelephoneCollection = currentValue.Occurrences;
                    else
                        sbrElectronicContactTelephoneCollection = null;
            
                    #region sbrElectronicContactTelephoneCollection
            
                    if (sbrElectronicContactTelephoneCollection != null)
                    {
                        report.ElectronicContactTelephoneCollectionExists = true;
                        report.ElectronicContactTelephoneCollectionCount = sbrElectronicContactTelephoneCollection.Count();
                
                        IEnumerable<SBRElement> sbrElectronicContactTelephoneCollection_03_INTCollection;
                        sbrElectronicContactTelephoneCollection_03_INTCollection = sbrElectronicContactTelephoneCollection.Where(
                            ce => ce.ChildElements != null && 
                            ce.ChildElements.Any(te => te.Name == "ElectronicContact.Telephone.Usage.Code" && te.Value == "03" &&
                            sbrINTContexts.Contains(te.Context)));
                        if (sbrElectronicContactTelephoneCollection_03_INTCollection != null && sbrElectronicContactTelephoneCollection_03_INTCollection.Count() > 0)
                        {
                            SBRElement sbrElectronicContactTelephoneCollection_03_INT;
                            sbrElectronicContactTelephoneCollection_03_INT = sbrElectronicContactTelephoneCollection_03_INTCollection.ElementAt(0);
                
                            IDictionary<string, SBRElement> sbrElectronicContactTelephoneCollection_03_INTTupleElementMap;
                            sbrElectronicContactTelephoneCollection_03_INTTupleElementMap = sbrElectronicContactTelephoneCollection_03_INT.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                
                            report.ElectronicContactTelephoneCollection_03_INTExists = true;
                            report.ElectronicContactTelephoneCollection_03_INTCount = sbrElectronicContactTelephoneCollection_03_INTCollection.Count();
                            report.ElectronicContactTelephoneCollection_03_INTOccurrenceIndex = sbrElectronicContactTelephoneCollection_03_INT.OccurrenceIndex + 1;
                            report.ElectronicContactTelephoneCollection_03_INTLastOccurrenceIndex = sbrElectronicContactTelephoneCollection_03_INTCollection.Last().OccurrenceIndex + 1;
                    
                            if (sbrElectronicContactTelephoneCollection_03_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}ElectronicContact.Telephone.Usage.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR276 = currentValue.Value;
                            }
                    
                            if (sbrElectronicContactTelephoneCollection_03_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}ElectronicContact.Telephone.ServiceLine.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR274 = currentValue.Value;
                            }
                    
                            if (sbrElectronicContactTelephoneCollection_03_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}ElectronicContact.Telephone.Area.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR215 = currentValue.Value;
                            }
                    
                            if (sbrElectronicContactTelephoneCollection_03_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}ElectronicContact.Telephone.Minimal.Number", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR216 = currentValue.Value;
                            }
                        }  
                    } 
                    #endregion End of Tuple sbrElectronicContactTelephoneCollection
                    if (validationMode) ValidateINTElectronicContactTelephoneCollection(errors, sbrElectronicContactTelephoneCollection, sbrINTContexts, report);
            
        
                    IEnumerable<SBRElement> sbrPersonUnstructuredNameCollection;
                    if (populatedReportElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}PersonUnstructuredName", out currentValue))
                        sbrPersonUnstructuredNameCollection = currentValue.Occurrences;
                    else
                        sbrPersonUnstructuredNameCollection = null;
            
                    #region sbrPersonUnstructuredNameCollection
            
                    if (sbrPersonUnstructuredNameCollection != null)
                    {
                        report.PersonUnstructuredNameCollectionExists = true;
                        report.PersonUnstructuredNameCollectionCount = sbrPersonUnstructuredNameCollection.Count();
                
                        IEnumerable<SBRElement> sbrPersonUnstructuredNameCollection_Contact_INTCollection;
                        sbrPersonUnstructuredNameCollection_Contact_INTCollection = sbrPersonUnstructuredNameCollection.Where(
                            ce => ce.ChildElements != null && 
                            ce.ChildElements.Any(te => te.Name == "PersonUnstructuredName.Usage.Code" && te.Value == "Contact" &&
                            sbrINTContexts.Contains(te.Context)));
                        if (sbrPersonUnstructuredNameCollection_Contact_INTCollection != null && sbrPersonUnstructuredNameCollection_Contact_INTCollection.Count() > 0)
                        {
                            SBRElement sbrPersonUnstructuredNameCollection_Contact_INT;
                            sbrPersonUnstructuredNameCollection_Contact_INT = sbrPersonUnstructuredNameCollection_Contact_INTCollection.ElementAt(0);
                
                            IDictionary<string, SBRElement> sbrPersonUnstructuredNameCollection_Contact_INTTupleElementMap;
                            sbrPersonUnstructuredNameCollection_Contact_INTTupleElementMap = sbrPersonUnstructuredNameCollection_Contact_INT.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                
                            report.PersonUnstructuredNameCollection_Contact_INTExists = true;
                            report.PersonUnstructuredNameCollection_Contact_INTCount = sbrPersonUnstructuredNameCollection_Contact_INTCollection.Count();
                            report.PersonUnstructuredNameCollection_Contact_INTOccurrenceIndex = sbrPersonUnstructuredNameCollection_Contact_INT.OccurrenceIndex + 1;
                            report.PersonUnstructuredNameCollection_Contact_INTLastOccurrenceIndex = sbrPersonUnstructuredNameCollection_Contact_INTCollection.Last().OccurrenceIndex + 1;
                    
                            if (sbrPersonUnstructuredNameCollection_Contact_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}PersonUnstructuredName.Usage.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR278 = currentValue.Value;
                            }
                    
                            if (sbrPersonUnstructuredNameCollection_Contact_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}PersonUnstructuredName.FullName.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR214 = currentValue.Value;
                            }
                        }  
                    } 
                    #endregion End of Tuple sbrPersonUnstructuredNameCollection
                    if (validationMode) ValidateINTPersonUnstructuredNameCollection(errors, sbrPersonUnstructuredNameCollection, sbrINTContexts, report);
            
            
                    if (sbrPopulatedINTElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Identifiers.TaxAgentNumber.Identifier", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR3 = currentValues[0].Value;
                    }
                } 
            } 
            #endregion End of Context sbrINTContext
    
            #region sbrRPContext
            IEnumerable<SBRContext> sbrRPContexts = null;
            SBRContext sbrRPContext = null;
    
            sbrRPContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 1 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty")) )).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPContexts != null && sbrRPContexts.Count() > 0)
            {
                sbrRPContext = sbrRPContexts.First();
        
                report.RPId = sbrRPContext.Id;
                report.RPIdentifier = sbrRPContext.EntityIdentifier;
                report.RPIdentifierScheme = sbrRPContext.EntityScheme;
        
                report.RPCount = sbrRPContexts.Count();
                report.RPExists = true;
                report.RPOccurrenceIndex = Array.IndexOf(contextArray, sbrRPContext) + 1;
                report.RPLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPContexts.Last()) + 1;
        

                if (report.RPIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPIdentifierTFN = report.RPIdentifier;
    
                report.RPStartDate = ((SBRDurationPeriod)sbrRPContext.Period).DurationStartAsDateObject;
                report.RPEndDate = ((SBRDurationPeriod)sbrRPContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPContexts.Count(); i++)
                {
                    sbrRPContext = sbrRPContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
                    report.CTR8 = report.RPIdentifier;
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Report.TargetFinancial.Year", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR2 = int.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Report.Amendment.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR298 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Report.AmendmentType.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR306 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Report.AmendmentSequence.Number", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR305 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Report.AmendmentReason.Text", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR307 = currentValues[0].Value;
                    }
        
                    IEnumerable<SBRElement> sbrOrganisationNameDetailsCollection;
                    if (populatedReportElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}OrganisationNameDetails", out currentValue))
                        sbrOrganisationNameDetailsCollection = currentValue.Occurrences;
                    else
                        sbrOrganisationNameDetailsCollection = null;
            
                    #region sbrOrganisationNameDetailsCollection
            
                    if (sbrOrganisationNameDetailsCollection != null)
                    {
                        report.OrganisationNameDetailsCollectionExists = true;
                        report.OrganisationNameDetailsCollectionCount = sbrOrganisationNameDetailsCollection.Count();
                
                        IEnumerable<SBRElement> sbrOrganisationNameDetailsCollection_MN_C_RPCollection;
                        sbrOrganisationNameDetailsCollection_MN_C_RPCollection = sbrOrganisationNameDetailsCollection.Where(
                            ce => ce.ChildElements != null && 
                            ce.ChildElements.Any(te => te.Name == "OrganisationNameDetails.OrganisationalNameType.Code" && te.Value == "MN") &&
                            ce.ChildElements.Any(te => te.Name == "OrganisationNameDetails.Currency.Code" && te.Value == "C" &&
                            sbrRPContexts.Contains(te.Context)));
                        if (sbrOrganisationNameDetailsCollection_MN_C_RPCollection != null && sbrOrganisationNameDetailsCollection_MN_C_RPCollection.Count() > 0)
                        {
                            SBRElement sbrOrganisationNameDetailsCollection_MN_C_RP;
                            sbrOrganisationNameDetailsCollection_MN_C_RP = sbrOrganisationNameDetailsCollection_MN_C_RPCollection.ElementAt(0);
                
                            IDictionary<string, SBRElement> sbrOrganisationNameDetailsCollection_MN_C_RPTupleElementMap;
                            sbrOrganisationNameDetailsCollection_MN_C_RPTupleElementMap = sbrOrganisationNameDetailsCollection_MN_C_RP.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                
                            report.OrganisationNameDetailsCollection_MN_C_RPExists = true;
                            report.OrganisationNameDetailsCollection_MN_C_RPCount = sbrOrganisationNameDetailsCollection_MN_C_RPCollection.Count();
                            report.OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex = sbrOrganisationNameDetailsCollection_MN_C_RP.OccurrenceIndex + 1;
                            report.OrganisationNameDetailsCollection_MN_C_RPLastOccurrenceIndex = sbrOrganisationNameDetailsCollection_MN_C_RPCollection.Last().OccurrenceIndex + 1;
                    
                            if (sbrOrganisationNameDetailsCollection_MN_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}OrganisationNameDetails.OrganisationalNameType.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR228 = currentValue.Value;
                            }
                    
                            if (sbrOrganisationNameDetailsCollection_MN_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}OrganisationNameDetails.Currency.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR285 = currentValue.Value;
                            }
                    
                            if (sbrOrganisationNameDetailsCollection_MN_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}OrganisationNameDetails.OrganisationalName.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR9 = currentValue.Value;
                            }
                        }  
                    } 
                    #endregion End of Tuple sbrOrganisationNameDetailsCollection
                    if (validationMode) ValidateRPOrganisationNameDetailsCollection(errors, sbrOrganisationNameDetailsCollection, sbrRPContexts, report);
            
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}OrganisationDetails.OrganisationActivityStatus.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR10 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Identifiers.AustralianBusinessNumber.Identifier", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR11 = currentValues[0].Value;
                    }
        
                    IEnumerable<SBRElement> sbrAddressDetailsCollection;
                    if (populatedReportElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}AddressDetails", out currentValue))
                        sbrAddressDetailsCollection = currentValue.Occurrences;
                    else
                        sbrAddressDetailsCollection = null;
            
                    #region sbrAddressDetailsCollection
            
                    if (sbrAddressDetailsCollection != null)
                    {
                        report.AddressDetailsCollectionExists = true;
                        report.AddressDetailsCollectionCount = sbrAddressDetailsCollection.Count();
                
                        IEnumerable<SBRElement> sbrAddressDetailsCollection_POS_C_RPCollection;
                        sbrAddressDetailsCollection_POS_C_RPCollection = sbrAddressDetailsCollection.Where(
                            ce => ce.ChildElements != null && 
                            ce.ChildElements.Any(te => te.Name == "AddressDetails.Usage.Code" && te.Value == "POS") &&
                            ce.ChildElements.Any(te => te.Name == "AddressDetails.Currency.Code" && te.Value == "C" &&
                            sbrRPContexts.Contains(te.Context)));
                        if (sbrAddressDetailsCollection_POS_C_RPCollection != null && sbrAddressDetailsCollection_POS_C_RPCollection.Count() > 0)
                        {
                            SBRElement sbrAddressDetailsCollection_POS_C_RP;
                            sbrAddressDetailsCollection_POS_C_RP = sbrAddressDetailsCollection_POS_C_RPCollection.ElementAt(0);
                
                            IDictionary<string, SBRElement> sbrAddressDetailsCollection_POS_C_RPTupleElementMap;
                            sbrAddressDetailsCollection_POS_C_RPTupleElementMap = sbrAddressDetailsCollection_POS_C_RP.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                
                            report.AddressDetailsCollection_POS_C_RPExists = true;
                            report.AddressDetailsCollection_POS_C_RPCount = sbrAddressDetailsCollection_POS_C_RPCollection.Count();
                            report.AddressDetailsCollection_POS_C_RPOccurrenceIndex = sbrAddressDetailsCollection_POS_C_RP.OccurrenceIndex + 1;
                            report.AddressDetailsCollection_POS_C_RPLastOccurrenceIndex = sbrAddressDetailsCollection_POS_C_RPCollection.Last().OccurrenceIndex + 1;
                    
                            if (sbrAddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}AddressDetails.OverseasAddress.Indicator", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR22 = ToBoolean(currentValue.Value);
                            }
                    
                            if (sbrAddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}AddressDetails.Usage.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR267 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}AddressDetails.Currency.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR268 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}AddressDetails.Line1.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR15 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}AddressDetails.Line2.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR16 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}AddressDetails.Line3.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR17 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}AddressDetails.Line4.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR18 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}AddressDetails.LocalityName.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR19 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}AddressDetails.Postcode.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR21 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}AddressDetails.StateOrTerritory.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR20 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}AddressDetails.Country.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR23 = currentValue.Value;
                            }
                        }  
                    } 
                    #endregion End of Tuple sbrAddressDetailsCollection
                    if (validationMode) ValidateRPAddressDetailsCollection(errors, sbrAddressDetailsCollection, sbrRPContexts, report);
            
            
                    #region sbrOrganisationNameDetailsCollection
            
                    if (sbrOrganisationNameDetailsCollection != null)
                    {
                
                        IEnumerable<SBRElement> sbrOrganisationNameDetailsCollection_MN_P_RPCollection;
                        sbrOrganisationNameDetailsCollection_MN_P_RPCollection = sbrOrganisationNameDetailsCollection.Where(
                            ce => ce.ChildElements != null && 
                            ce.ChildElements.Any(te => te.Name == "OrganisationNameDetails.OrganisationalNameType.Code" && te.Value == "MN") &&
                            ce.ChildElements.Any(te => te.Name == "OrganisationNameDetails.Currency.Code" && te.Value == "P" &&
                            sbrRPContexts.Contains(te.Context)));
                        if (sbrOrganisationNameDetailsCollection_MN_P_RPCollection != null && sbrOrganisationNameDetailsCollection_MN_P_RPCollection.Count() > 0)
                        {
                            SBRElement sbrOrganisationNameDetailsCollection_MN_P_RP;
                            sbrOrganisationNameDetailsCollection_MN_P_RP = sbrOrganisationNameDetailsCollection_MN_P_RPCollection.ElementAt(0);
                
                            IDictionary<string, SBRElement> sbrOrganisationNameDetailsCollection_MN_P_RPTupleElementMap;
                            sbrOrganisationNameDetailsCollection_MN_P_RPTupleElementMap = sbrOrganisationNameDetailsCollection_MN_P_RP.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                
                            report.OrganisationNameDetailsCollection_MN_P_RPExists = true;
                            report.OrganisationNameDetailsCollection_MN_P_RPCount = sbrOrganisationNameDetailsCollection_MN_P_RPCollection.Count();
                            report.OrganisationNameDetailsCollection_MN_P_RPOccurrenceIndex = sbrOrganisationNameDetailsCollection_MN_P_RP.OccurrenceIndex + 1;
                            report.OrganisationNameDetailsCollection_MN_P_RPLastOccurrenceIndex = sbrOrganisationNameDetailsCollection_MN_P_RPCollection.Last().OccurrenceIndex + 1;
                    
                            if (sbrOrganisationNameDetailsCollection_MN_P_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}OrganisationNameDetails.OrganisationalNameType.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR229 = currentValue.Value;
                            }
                    
                            if (sbrOrganisationNameDetailsCollection_MN_P_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}OrganisationNameDetails.Currency.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR286 = currentValue.Value;
                            }
                    
                            if (sbrOrganisationNameDetailsCollection_MN_P_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}OrganisationNameDetails.OrganisationalName.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR12 = currentValue.Value;
                            }
                        }  
                    } 
                    #endregion End of Tuple sbrOrganisationNameDetailsCollection
                    if (validationMode) ValidateRPOrganisationNameDetailsCollection1(errors, sbrOrganisationNameDetailsCollection, sbrRPContexts, report);
            
            
                    #region sbrAddressDetailsCollection
            
                    if (sbrAddressDetailsCollection != null)
                    {
                
                        IEnumerable<SBRElement> sbrAddressDetailsCollection_POS_P_RPCollection;
                        sbrAddressDetailsCollection_POS_P_RPCollection = sbrAddressDetailsCollection.Where(
                            ce => ce.ChildElements != null && 
                            ce.ChildElements.Any(te => te.Name == "AddressDetails.Usage.Code" && te.Value == "POS") &&
                            ce.ChildElements.Any(te => te.Name == "AddressDetails.Currency.Code" && te.Value == "P" &&
                            sbrRPContexts.Contains(te.Context)));
                        if (sbrAddressDetailsCollection_POS_P_RPCollection != null && sbrAddressDetailsCollection_POS_P_RPCollection.Count() > 0)
                        {
                            SBRElement sbrAddressDetailsCollection_POS_P_RP;
                            sbrAddressDetailsCollection_POS_P_RP = sbrAddressDetailsCollection_POS_P_RPCollection.ElementAt(0);
                
                            IDictionary<string, SBRElement> sbrAddressDetailsCollection_POS_P_RPTupleElementMap;
                            sbrAddressDetailsCollection_POS_P_RPTupleElementMap = sbrAddressDetailsCollection_POS_P_RP.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                
                            report.AddressDetailsCollection_POS_P_RPExists = true;
                            report.AddressDetailsCollection_POS_P_RPCount = sbrAddressDetailsCollection_POS_P_RPCollection.Count();
                            report.AddressDetailsCollection_POS_P_RPOccurrenceIndex = sbrAddressDetailsCollection_POS_P_RP.OccurrenceIndex + 1;
                            report.AddressDetailsCollection_POS_P_RPLastOccurrenceIndex = sbrAddressDetailsCollection_POS_P_RPCollection.Last().OccurrenceIndex + 1;
                    
                            if (sbrAddressDetailsCollection_POS_P_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}AddressDetails.OverseasAddress.Indicator", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR31 = ToBoolean(currentValue.Value);
                            }
                    
                            if (sbrAddressDetailsCollection_POS_P_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}AddressDetails.Usage.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR273 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_POS_P_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}AddressDetails.Currency.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR270 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_POS_P_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}AddressDetails.Line1.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR24 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_POS_P_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}AddressDetails.Line2.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR25 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_POS_P_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}AddressDetails.Line3.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR26 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_POS_P_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}AddressDetails.Line4.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR27 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_POS_P_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}AddressDetails.LocalityName.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR28 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_POS_P_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}AddressDetails.Postcode.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR30 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_POS_P_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}AddressDetails.StateOrTerritory.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR29 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_POS_P_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}AddressDetails.Country.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR32 = currentValue.Value;
                            }
                        }  
                    } 
                    #endregion End of Tuple sbrAddressDetailsCollection
                    if (validationMode) ValidateRPAddressDetailsCollection1(errors, sbrAddressDetailsCollection, sbrRPContexts, report);
            
            
                    #region sbrAddressDetailsCollection
            
                    if (sbrAddressDetailsCollection != null)
                    {
                
                        IEnumerable<SBRElement> sbrAddressDetailsCollection_BUS_C_RPCollection;
                        sbrAddressDetailsCollection_BUS_C_RPCollection = sbrAddressDetailsCollection.Where(
                            ce => ce.ChildElements != null && 
                            ce.ChildElements.Any(te => te.Name == "AddressDetails.Usage.Code" && te.Value == "BUS") &&
                            ce.ChildElements.Any(te => te.Name == "AddressDetails.Currency.Code" && te.Value == "C" &&
                            sbrRPContexts.Contains(te.Context)));
                        if (sbrAddressDetailsCollection_BUS_C_RPCollection != null && sbrAddressDetailsCollection_BUS_C_RPCollection.Count() > 0)
                        {
                            SBRElement sbrAddressDetailsCollection_BUS_C_RP;
                            sbrAddressDetailsCollection_BUS_C_RP = sbrAddressDetailsCollection_BUS_C_RPCollection.ElementAt(0);
                
                            IDictionary<string, SBRElement> sbrAddressDetailsCollection_BUS_C_RPTupleElementMap;
                            sbrAddressDetailsCollection_BUS_C_RPTupleElementMap = sbrAddressDetailsCollection_BUS_C_RP.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                
                            report.AddressDetailsCollection_BUS_C_RPExists = true;
                            report.AddressDetailsCollection_BUS_C_RPCount = sbrAddressDetailsCollection_BUS_C_RPCollection.Count();
                            report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex = sbrAddressDetailsCollection_BUS_C_RP.OccurrenceIndex + 1;
                            report.AddressDetailsCollection_BUS_C_RPLastOccurrenceIndex = sbrAddressDetailsCollection_BUS_C_RPCollection.Last().OccurrenceIndex + 1;
                    
                            if (sbrAddressDetailsCollection_BUS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}AddressDetails.OverseasAddress.Indicator", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR40 = ToBoolean(currentValue.Value);
                            }
                    
                            if (sbrAddressDetailsCollection_BUS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}AddressDetails.Usage.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR269 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_BUS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}AddressDetails.Currency.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR283 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_BUS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}AddressDetails.Line1.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR33 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_BUS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}AddressDetails.Line2.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR34 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_BUS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}AddressDetails.Line3.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR35 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_BUS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}AddressDetails.Line4.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR36 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_BUS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}AddressDetails.LocalityName.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR37 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_BUS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}AddressDetails.Postcode.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR39 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_BUS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}AddressDetails.StateOrTerritory.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR38 = currentValue.Value;
                            }
                    
                            if (sbrAddressDetailsCollection_BUS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}AddressDetails.Country.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR41 = currentValue.Value;
                            }
                        }  
                    } 
                    #endregion End of Tuple sbrAddressDetailsCollection
                    if (validationMode) ValidateRPAddressDetailsCollection2(errors, sbrAddressDetailsCollection, sbrRPContexts, report);
            
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Lodgment.FinalReturn.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR42 = ToBoolean(currentValues[0].Value);
                    }
        
                    IEnumerable<SBRElement> sbrFinancialInstitutionAccountCollection;
                    if (populatedReportElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}FinancialInstitutionAccount", out currentValue))
                        sbrFinancialInstitutionAccountCollection = currentValue.Occurrences;
                    else
                        sbrFinancialInstitutionAccountCollection = null;
            
                    #region sbrFinancialInstitutionAccountCollection
            
                    if (sbrFinancialInstitutionAccountCollection != null)
                    {
                        report.FinancialInstitutionAccountCollectionExists = true;
                        report.FinancialInstitutionAccountCollectionCount = sbrFinancialInstitutionAccountCollection.Count();
                        SBRElement sbrFinancialInstitutionAccount;
                        sbrFinancialInstitutionAccount = sbrFinancialInstitutionAccountCollection.ElementAt(0);
                
                        IDictionary<string, SBRElement> sbrFinancialInstitutionAccountTupleElementMap;
                        sbrFinancialInstitutionAccountTupleElementMap = sbrFinancialInstitutionAccount.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                
                        if (sbrFinancialInstitutionAccountTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}FinancialInstitutionAccount.BankStateBranch.Number", out currentValue))
                        {
                            if (!currentValue.IsNil) report.CTR293 = currentValue.Value;
                        }
                
                        if (sbrFinancialInstitutionAccountTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}FinancialInstitutionAccount.FinancialInstitutionAccount.Number", out currentValue))
                        {
                            if (!currentValue.IsNil) report.CTR294 = currentValue.Value;
                        }
                
                        if (sbrFinancialInstitutionAccountTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}FinancialInstitutionAccount.FinancialInstitutionAccountName.Text", out currentValue))
                        {
                            if (!currentValue.IsNil) report.CTR295 = currentValue.Value;
                        }
                
                        if (sbrFinancialInstitutionAccountTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}FinancialInstitutionAccount.FinancialInstitutionBranchName.Text", out currentValue))
                        {
                            if (!currentValue.IsNil) report.CTR296 = currentValue.Value;
                        }
                
                        if (sbrFinancialInstitutionAccountTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}FinancialInstitutionAccount.FinancialInstitutionName.Text", out currentValue))
                        {
                            if (!currentValue.IsNil) report.CTR297 = currentValue.Value;
                        }
                    } 
                    #endregion End of Tuple sbrFinancialInstitutionAccountCollection
        
                    IEnumerable<SBRElement> sbrUltimateHoldingCompanyCollection;
                    if (populatedReportElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}UltimateHoldingCompany", out currentValue))
                        sbrUltimateHoldingCompanyCollection = currentValue.Occurrences;
                    else
                        sbrUltimateHoldingCompanyCollection = null;
            
                    #region sbrUltimateHoldingCompanyCollection
            
                    if (sbrUltimateHoldingCompanyCollection != null)
                    {
                        report.UltimateHoldingCompanyCollectionExists = true;
                        report.UltimateHoldingCompanyCollectionCount = sbrUltimateHoldingCompanyCollection.Count();
                        SBRElement sbrUltimateHoldingCompany;
                        sbrUltimateHoldingCompany = sbrUltimateHoldingCompanyCollection.ElementAt(0);
                
                        IDictionary<string, SBRElement> sbrUltimateHoldingCompanyTupleElementMap;
                        sbrUltimateHoldingCompanyTupleElementMap = sbrUltimateHoldingCompany.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
            
                        IEnumerable<SBRElement> sbrUltimateHoldingCompany_OrganisationNameDetailsCollection;
                        if (sbrUltimateHoldingCompanyTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}OrganisationNameDetails", out currentValue))
                            sbrUltimateHoldingCompany_OrganisationNameDetailsCollection = currentValue.Occurrences;
                        else
                            sbrUltimateHoldingCompany_OrganisationNameDetailsCollection = null;
                
                        #region sbrUltimateHoldingCompany_OrganisationNameDetailsCollection
                
                        if (sbrUltimateHoldingCompany_OrganisationNameDetailsCollection != null)
                        {
                            report.UltimateHoldingCompany_OrganisationNameDetailsCollectionExists = true;
                            report.UltimateHoldingCompany_OrganisationNameDetailsCollectionCount = sbrUltimateHoldingCompany_OrganisationNameDetailsCollection.Count();
                    
                            IEnumerable<SBRElement> sbrUltimateHoldingCompany_OrganisationNameDetailsCollection_MN_C_RPCollection;
                            sbrUltimateHoldingCompany_OrganisationNameDetailsCollection_MN_C_RPCollection = sbrUltimateHoldingCompany_OrganisationNameDetailsCollection.Where(
                                ce => ce.ChildElements != null && 
                                ce.ChildElements.Any(te => te.Name == "OrganisationNameDetails.OrganisationalNameType.Code" && te.Value == "MN") &&
                                ce.ChildElements.Any(te => te.Name == "OrganisationNameDetails.Currency.Code" && te.Value == "C" &&
                                sbrRPContexts.Contains(te.Context)));
                            if (sbrUltimateHoldingCompany_OrganisationNameDetailsCollection_MN_C_RPCollection != null && sbrUltimateHoldingCompany_OrganisationNameDetailsCollection_MN_C_RPCollection.Count() > 0)
                            {
                                SBRElement sbrUltimateHoldingCompany_OrganisationNameDetailsCollection_MN_C_RP;
                                sbrUltimateHoldingCompany_OrganisationNameDetailsCollection_MN_C_RP = sbrUltimateHoldingCompany_OrganisationNameDetailsCollection_MN_C_RPCollection.ElementAt(0);
                    
                                IDictionary<string, SBRElement> sbrUltimateHoldingCompany_OrganisationNameDetailsCollection_MN_C_RPTupleElementMap;
                                sbrUltimateHoldingCompany_OrganisationNameDetailsCollection_MN_C_RPTupleElementMap = sbrUltimateHoldingCompany_OrganisationNameDetailsCollection_MN_C_RP.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                    
                                report.UltimateHoldingCompany_OrganisationNameDetailsCollection_MN_C_RPExists = true;
                                report.UltimateHoldingCompany_OrganisationNameDetailsCollection_MN_C_RPCount = sbrUltimateHoldingCompany_OrganisationNameDetailsCollection_MN_C_RPCollection.Count();
                                report.UltimateHoldingCompany_OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex = sbrUltimateHoldingCompany_OrganisationNameDetailsCollection_MN_C_RP.OccurrenceIndex + 1;
                                report.UltimateHoldingCompany_OrganisationNameDetailsCollection_MN_C_RPLastOccurrenceIndex = sbrUltimateHoldingCompany_OrganisationNameDetailsCollection_MN_C_RPCollection.Last().OccurrenceIndex + 1;
                        
                                if (sbrUltimateHoldingCompany_OrganisationNameDetailsCollection_MN_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}OrganisationNameDetails.OrganisationalNameType.Code", out currentValue))
                                {
                                    if (!currentValue.IsNil) report.CTR271 = currentValue.Value;
                                }
                        
                                if (sbrUltimateHoldingCompany_OrganisationNameDetailsCollection_MN_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}OrganisationNameDetails.Currency.Code", out currentValue))
                                {
                                    if (!currentValue.IsNil) report.CTR287 = currentValue.Value;
                                }
                        
                                if (sbrUltimateHoldingCompany_OrganisationNameDetailsCollection_MN_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}OrganisationNameDetails.OrganisationalName.Text", out currentValue))
                                {
                                    if (!currentValue.IsNil) report.CTR43 = currentValue.Value;
                                }
                            }  
                        } 
                        #endregion End of Tuple sbrUltimateHoldingCompany_OrganisationNameDetailsCollection
                        if (validationMode) ValidateRPUltimateHoldingCompany_OrganisationNameDetailsCollection(errors, sbrUltimateHoldingCompany_OrganisationNameDetailsCollection, sbrRPContexts, report);
                
                
                        if (sbrUltimateHoldingCompanyTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Identifiers.AustralianBusinessNumber.Identifier", out currentValue))
                        {
                            if (!currentValue.IsNil) report.CTR44 = currentValue.Value;
                        }
                
                        if (sbrUltimateHoldingCompanyTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Residency.TaxPurposesCountry.Code", out currentValue))
                        {
                            if (!currentValue.IsNil) report.CTR45 = currentValue.Value;
                        }
                    } 
                    #endregion End of Tuple sbrUltimateHoldingCompanyCollection
        
                    IEnumerable<SBRElement> sbrImmediateHoldingCompanyCollection;
                    if (populatedReportElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}ImmediateHoldingCompany", out currentValue))
                        sbrImmediateHoldingCompanyCollection = currentValue.Occurrences;
                    else
                        sbrImmediateHoldingCompanyCollection = null;
            
                    #region sbrImmediateHoldingCompanyCollection
            
                    if (sbrImmediateHoldingCompanyCollection != null)
                    {
                        report.ImmediateHoldingCompanyCollectionExists = true;
                        report.ImmediateHoldingCompanyCollectionCount = sbrImmediateHoldingCompanyCollection.Count();
                        SBRElement sbrImmediateHoldingCompany;
                        sbrImmediateHoldingCompany = sbrImmediateHoldingCompanyCollection.ElementAt(0);
                
                        IDictionary<string, SBRElement> sbrImmediateHoldingCompanyTupleElementMap;
                        sbrImmediateHoldingCompanyTupleElementMap = sbrImmediateHoldingCompany.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
            
                        IEnumerable<SBRElement> sbrImmediateHoldingCompany_OrganisationNameDetailsCollection;
                        if (sbrImmediateHoldingCompanyTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}OrganisationNameDetails", out currentValue))
                            sbrImmediateHoldingCompany_OrganisationNameDetailsCollection = currentValue.Occurrences;
                        else
                            sbrImmediateHoldingCompany_OrganisationNameDetailsCollection = null;
                
                        #region sbrImmediateHoldingCompany_OrganisationNameDetailsCollection
                
                        if (sbrImmediateHoldingCompany_OrganisationNameDetailsCollection != null)
                        {
                            report.ImmediateHoldingCompany_OrganisationNameDetailsCollectionExists = true;
                            report.ImmediateHoldingCompany_OrganisationNameDetailsCollectionCount = sbrImmediateHoldingCompany_OrganisationNameDetailsCollection.Count();
                    
                            IEnumerable<SBRElement> sbrImmediateHoldingCompany_OrganisationNameDetailsCollection_MN_C_RPCollection;
                            sbrImmediateHoldingCompany_OrganisationNameDetailsCollection_MN_C_RPCollection = sbrImmediateHoldingCompany_OrganisationNameDetailsCollection.Where(
                                ce => ce.ChildElements != null && 
                                ce.ChildElements.Any(te => te.Name == "OrganisationNameDetails.OrganisationalNameType.Code" && te.Value == "MN") &&
                                ce.ChildElements.Any(te => te.Name == "OrganisationNameDetails.Currency.Code" && te.Value == "C" &&
                                sbrRPContexts.Contains(te.Context)));
                            if (sbrImmediateHoldingCompany_OrganisationNameDetailsCollection_MN_C_RPCollection != null && sbrImmediateHoldingCompany_OrganisationNameDetailsCollection_MN_C_RPCollection.Count() > 0)
                            {
                                SBRElement sbrImmediateHoldingCompany_OrganisationNameDetailsCollection_MN_C_RP;
                                sbrImmediateHoldingCompany_OrganisationNameDetailsCollection_MN_C_RP = sbrImmediateHoldingCompany_OrganisationNameDetailsCollection_MN_C_RPCollection.ElementAt(0);
                    
                                IDictionary<string, SBRElement> sbrImmediateHoldingCompany_OrganisationNameDetailsCollection_MN_C_RPTupleElementMap;
                                sbrImmediateHoldingCompany_OrganisationNameDetailsCollection_MN_C_RPTupleElementMap = sbrImmediateHoldingCompany_OrganisationNameDetailsCollection_MN_C_RP.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                    
                                report.ImmediateHoldingCompany_OrganisationNameDetailsCollection_MN_C_RPExists = true;
                                report.ImmediateHoldingCompany_OrganisationNameDetailsCollection_MN_C_RPCount = sbrImmediateHoldingCompany_OrganisationNameDetailsCollection_MN_C_RPCollection.Count();
                                report.ImmediateHoldingCompany_OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex = sbrImmediateHoldingCompany_OrganisationNameDetailsCollection_MN_C_RP.OccurrenceIndex + 1;
                                report.ImmediateHoldingCompany_OrganisationNameDetailsCollection_MN_C_RPLastOccurrenceIndex = sbrImmediateHoldingCompany_OrganisationNameDetailsCollection_MN_C_RPCollection.Last().OccurrenceIndex + 1;
                        
                                if (sbrImmediateHoldingCompany_OrganisationNameDetailsCollection_MN_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}OrganisationNameDetails.OrganisationalNameType.Code", out currentValue))
                                {
                                    if (!currentValue.IsNil) report.CTR272 = currentValue.Value;
                                }
                        
                                if (sbrImmediateHoldingCompany_OrganisationNameDetailsCollection_MN_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}OrganisationNameDetails.Currency.Code", out currentValue))
                                {
                                    if (!currentValue.IsNil) report.CTR288 = currentValue.Value;
                                }
                        
                                if (sbrImmediateHoldingCompany_OrganisationNameDetailsCollection_MN_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}OrganisationNameDetails.OrganisationalName.Text", out currentValue))
                                {
                                    if (!currentValue.IsNil) report.CTR46 = currentValue.Value;
                                }
                            }  
                        } 
                        #endregion End of Tuple sbrImmediateHoldingCompany_OrganisationNameDetailsCollection
                        if (validationMode) ValidateRPImmediateHoldingCompany_OrganisationNameDetailsCollection(errors, sbrImmediateHoldingCompany_OrganisationNameDetailsCollection, sbrRPContexts, report);
                
                
                        if (sbrImmediateHoldingCompanyTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Identifiers.AustralianBusinessNumber.Identifier", out currentValue))
                        {
                            if (!currentValue.IsNil) report.CTR47 = currentValue.Value;
                        }
                    } 
                    #endregion End of Tuple sbrImmediateHoldingCompanyCollection
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}OrganisationDetails.MainIncomeActivity.Description", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR48 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}OrganisationDetails.OrganisationIndustry2006Extended.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR49 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Equity.ContributedEquity.NonAustralianShareholding.Percent", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR50 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Residency.TaxPurposesOrganisationStatus.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR51 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}OrganisationDetails.OrganisationType.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR52 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}OrganisationDetails.TaxConsolidationStatus.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR53 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}OrganisationDetails.ActivityEvent.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR54 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Elections.SmallBusinessEntitiesElection.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR363 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Elections.BaseRateEntityElection.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR375 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}OrganisationDetails.SignificantGlobalEntityStatus.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR370 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}InternationalDealings.CountryByCountryReportingEntityStatus.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR378 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Elections.InterposedEntityElectionStatus.Year", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR55 = int.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Elections.InterposedEntityElectionOrRevocation.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR56 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}RegulatoryDisclosures.FinancialStatementLodged.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR371 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}RegulatoryDisclosures.FinancialStatementLodgmentPending.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR372 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Remuneration.PaymentToForeignResidentGross.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR57 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Remuneration.ABNNotQuotedPaymentGross.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR58 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Income.SaleOfGoodsAndServices.Other.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR59 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Income.PartnershipDistributionGross.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR60 = decimal.Parse(currentValues[0].Value);
                    }
        
                    IEnumerable<SBRElement> sbrTrustDistributionIncomeCollection;
                    if (populatedReportElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}TrustDistributionIncome", out currentValue))
                        sbrTrustDistributionIncomeCollection = currentValue.Occurrences;
                    else
                        sbrTrustDistributionIncomeCollection = null;
            
                    #region sbrTrustDistributionIncomeCollection
            
                    if (sbrTrustDistributionIncomeCollection != null)
                    {
                        report.TrustDistributionIncomeCollectionExists = true;
                        report.TrustDistributionIncomeCollectionCount = sbrTrustDistributionIncomeCollection.Count();
                        SBRElement sbrTrustDistributionIncome;
                        sbrTrustDistributionIncome = sbrTrustDistributionIncomeCollection.ElementAt(0);
                
                        IDictionary<string, SBRElement> sbrTrustDistributionIncomeTupleElementMap;
                        sbrTrustDistributionIncomeTupleElementMap = sbrTrustDistributionIncome.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                
                        if (sbrTrustDistributionIncomeTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Income.TrustDistributionGross.Amount", out currentValue))
                        {
                            if (!currentValue.IsNil) report.CTR61 = decimal.Parse(currentValue.Value);
                        }
                
                        if (sbrTrustDistributionIncomeTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Party.TrustType.Code", out currentValue))
                        {
                            if (!currentValue.IsNil) report.CTR379 = currentValue.Value;
                        }
                    } 
                    #endregion End of Tuple sbrTrustDistributionIncomeCollection
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Income.ForestryManagedInvestmentScheme.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR63 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Income.Interest.Gross.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR64 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Income.RentLeasingHiringGross.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR65 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Income.DividendsTotal.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR66 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Remuneration.FringeBenefits.EmployeeContribution.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR67 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR68 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}GovernmentFunding.GovernmentIndustryPaymentsIncludesFuel.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR69 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Income.AssetRevaluationsUnrealisedGains.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR70 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Income.OtherGross.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR71 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Expense.ForeignResidentWithholding.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR72 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Expense.Operating.CostOfSales.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR73 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Remuneration.WagesAndSalaries.PaymentsContractorAndConsultant.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR364 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}SuperannuationContribution.EmployerContributions.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR75 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Expense.Operating.BadDebts.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR76 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Expense.Operating.Rent.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR79 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Expense.DepreciationAndAmortisation.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR84 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Expense.Operating.MotorVehicle.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR85 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Expense.Operating.RepairsAndMaintenance.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR86 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Expense.AssetRevaluationsUnrealisedLosses.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR87 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Expense.Other.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR88 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}CapitalGainsTax.Event.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR89 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}CapitalGainsTax.ExemptionOrRolloverApplied.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR329 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}CapitalGainsTax.ExemptionOrRolloverApplied.Code", out currentValues))
                    {
                        report.CTR330Collection = new List<string>();
                        try
                        {
                                foreach (SBRElement occurrence in currentValues)
                                {
                                    if (!occurrence.IsNil)
                                        report.CTR330Collection.Add(occurrence.Value);
                                }
                        }
                        catch (Exception)
                        {
                            // The above code is known to throw exceptions in some cases but not others.
                            // We don't really know why but think that catching them like this will result in correct behaviour
                        }
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Income.CapitalGainsNet.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR91 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Expense.NonDeductibleExemptIncome.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR92 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}IncomeTax.FrankingCredits.ReceivedFromAustralianCompanies.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR93 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR94 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Income.AssessableOther.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR95 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Expense.NonDeductible.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR96 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}TaxConcession.ResearchAndDevelopment.WriteBackOfAccountingExpenditure.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR97 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Expense.FlowOnDividendDeduction.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR98 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Expense.DepreciationAllowableDeduction.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR99 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Expense.ForestryManagedInvestmentSchemeDeduction.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR100 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Expense.CapitalExpenditureImmediateDeduction.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR101 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Expense.ProjectPoolAllowableDeduction.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR102 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Expense.CapitalWorksDeduction.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR103 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Expense.BuildToRentCapitalWorksDeduction.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR456 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Expense.CapitalExpenditureSpecifiedAllowableDeduction.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR104 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Expense.EnvironmentalProtectionDeduction.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR112 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Income.Exempt.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR114 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Income.NotincludedInAssessableOther.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR115 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Expense.DeductibleOther.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR116 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}IncomeTax.Deduction.TaxLossesDeducted.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR117 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}IncomeTax.TaxLossesTransferredIn.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR118 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Income.Taxable.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR120 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Miscellaneous.FunctionalCurrency.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR244 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Expense.Purchases.AndOtherCosts.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR124 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Elections.TradingStock.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR127 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Liabilities.DebtTotal.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR134 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Liabilities.CommercialDebtForgiveness.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR135 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Equity.Dividends.FrankedPaid.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR137 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Equity.Dividends.UnfrankedPaid.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR138 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Income.AggregatedTurnoverRange.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR392 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Income.AggregatedTurnover.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR393 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}IncomeTax.FrankingCredits.ExcessFrankingOffsets.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR142 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Assets.LoansAndReceivables.RelatedParties.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR145 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Assets.LoansAndReceivablesAction.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR146 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Remuneration.WagesAndSalaries.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR151 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Remuneration.WagesAndSalariesAction.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR152 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Remuneration.PaymentToRelatedPartiesGross.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR153 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}InternationalDealings.ForeignIncomeGross.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR318 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Income.InternationalDealings.Net.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR154 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}InternationalDealings.TransferorTrustAttributedForeignIncome.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR159 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Expense.DepreciatingAssets.IntangibleFirstDeducted.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR147 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Expense.DepreciatingAssets.OtherFirstDeducted.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR148 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Assets.PropertyPlantAndEquipment.DepreciatingAssets.FirstDeductedSelfAssessedEffectiveLife.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR331 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Assets.PropertyPlantAndEquipment.DepreciatingAssets.EffectiveLifeRecalculation.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR332 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Assets.DepreciatingAssets.AdjustableValuesTotal.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR333 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Income.DepreciatingAssets.IntangibleBalancingAdjustment.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR334 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Expense.DepreciatingAssets.IntangibleBalancingAdjustment.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR335 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Assets.DepreciatingAssets.IntangibleTerminationValue.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR149 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Assets.DepreciatingAssets.OtherTerminationValue.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR150 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Expense.SubsequentYearAcceleratedDepreciationDeductionsForAssets.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR390 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Expense.DepreciatingAssets.ProjectPoolAllocationsForMiningAndTransportCapitalExpenditure.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR336 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Expense.DepreciatingAssets.IntangibleAssetsDeclineInValueOfExplorationOrProspecting.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR337 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Expense.DepreciatingAssets.OtherAssetsDeclineInValueOfExplorationOrProspecting.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR338 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}TaxOffsetClaim.NationalRentalAffordabilitySchemeEntitlement.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR246 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Tax.Losses.CarriedForward.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR175 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Income.PersonalServicesIncomeIncluded.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR177 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Income.PersonalServicesIncome.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR346 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Expense.PersonalServicesIncome.Deduction.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR347 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}IncomeTax.PersonalServicesIncome.IndividualResultsTestSatisfied.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR348 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}IncomeTax.PersonalServicesIncome.IndividualPersonalServiceBusinessDeterminationHeld.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR349 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}IncomeTax.PersonalServicesIncome.UnrelatedClientsTestSatisfied.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR350 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}IncomeTax.PersonalServicesIncome.EmploymentTestSatisfied.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR351 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}IncomeTax.PersonalServicesIncome.BusinessPremisesTestSatisfied.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR352 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Income.NonMembers.Percent", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR178 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Income.Insurance.Taxable.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR182 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Income.Insurance.FeesAndCharges.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR183 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Income.PooledDevelopmentFundsSmallToMediumEnterprises.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR184 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Income.PooledDevelopmentFundsUnregulatedInvestment.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR185 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Income.RetirementSavingsAccountTFNNotQuotedContributionsIncome.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR308 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}IncomeTax.RetirementSavingsAccountTFNNotQuotedContributionsIncomeTaxPayable.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR309 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Income.RetirementSavingsAccountsTaxableNet.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR190 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Income.InternationalDealings.TaxOffset.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR201 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}TaxConcession.ResearchAndDevelopment.TaxOffsetNonRefundable.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR328 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}TaxConcession.ResearchAndDevelopment.TaxOffsetNonRefundableUtilised.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR300 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}TaxConcession.ResearchAndDevelopment.TaxOffsetRefundable.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR210 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Income.ResearchAndDevelopment.Clawback.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR450 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Expense.ResearchAndDevelopment.AdjustmentBalance.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR451 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}TaxOffsetClaim.DigitalGamesTaxOffset.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR454 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}TaxOffsetClaim.DigitalGamesTaxOffsetAlreadyClaimed.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR455 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}RegulatoryDisclosures.ReportableTaxPosition.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR319 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}InternationalDealings.RelatedPartiesTransactions.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR193 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}InternationalDealings.RelatedPartiesTransactionsExcessAggregateValue.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR194 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}InternationalDealings.DirectOrIndirectOverseasInterest.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR195 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Liabilities.ThinCapitalisation.ProvisionsApplied.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR196 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}InternationalDealings.TransactionsWithTaxHavenCountries.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR198 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}IncomeTax.TotalTaxOnTaxableIncome.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR320 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}IncomeTax.TaxableGross.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR199 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}IncomeTax.TaxOffsetNonRefundableNonCarryForward.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR327 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}IncomeTax.TaxOffsetNonRefundableSubTotal.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR322 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}IncomeTax.TaxOffsetNonRefundableCarryForward.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR323 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}IncomeTax.TaxOffsetCarriedForwardSubTotal.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR324 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}IncomeTax.TotalRefundableTaxOffsets.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR316 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}IncomeTax.TaxOffsetRefundableSubTotal.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR325 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}IncomeTax.Payable.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR203 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}InternationalDealings.AAMInterestSection102.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR204 = decimal.Parse(currentValues[0].Value);
                    }
        
                    IEnumerable<SBRElement> sbrITPAYGWithholdingDetailsCollection;
                    if (populatedReportElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}ITPAYGWithholdingDetails", out currentValue))
                        sbrITPAYGWithholdingDetailsCollection = currentValue.Occurrences;
                    else
                        sbrITPAYGWithholdingDetailsCollection = null;
            
                    #region sbrITPAYGWithholdingDetailsCollection
            
                    if (sbrITPAYGWithholdingDetailsCollection != null)
                    {
                        report.ItpaygWithholdingDetailsCollectionExists = true;
                        report.ItpaygWithholdingDetailsCollectionCount = sbrITPAYGWithholdingDetailsCollection.Count();
                
                        IEnumerable<SBRElement> sbrITPAYGWithholdingDetailsCollection_DFRW_RPCollection;
                        sbrITPAYGWithholdingDetailsCollection_DFRW_RPCollection = sbrITPAYGWithholdingDetailsCollection.Where(
                            ce => ce.ChildElements != null && 
                            ce.ChildElements.Any(te => te.Name == "IncomeTax.PayAsYouGoWithholding.PaymentType.Code" && te.Value == "DFRW" &&
                            sbrRPContexts.Contains(te.Context)));
                        if (sbrITPAYGWithholdingDetailsCollection_DFRW_RPCollection != null && sbrITPAYGWithholdingDetailsCollection_DFRW_RPCollection.Count() > 0)
                        {
                            SBRElement sbrITPAYGWithholdingDetailsCollection_DFRW_RP;
                            sbrITPAYGWithholdingDetailsCollection_DFRW_RP = sbrITPAYGWithholdingDetailsCollection_DFRW_RPCollection.ElementAt(0);
                
                            IDictionary<string, SBRElement> sbrITPAYGWithholdingDetailsCollection_DFRW_RPTupleElementMap;
                            sbrITPAYGWithholdingDetailsCollection_DFRW_RPTupleElementMap = sbrITPAYGWithholdingDetailsCollection_DFRW_RP.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                
                            report.ItpaygWithholdingDetailsCollection_DFRW_RPExists = true;
                            report.ItpaygWithholdingDetailsCollection_DFRW_RPCount = sbrITPAYGWithholdingDetailsCollection_DFRW_RPCollection.Count();
                            report.ItpaygWithholdingDetailsCollection_DFRW_RPOccurrenceIndex = sbrITPAYGWithholdingDetailsCollection_DFRW_RP.OccurrenceIndex + 1;
                            report.ItpaygWithholdingDetailsCollection_DFRW_RPLastOccurrenceIndex = sbrITPAYGWithholdingDetailsCollection_DFRW_RPCollection.Last().OccurrenceIndex + 1;
                    
                            if (sbrITPAYGWithholdingDetailsCollection_DFRW_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}IncomeTax.PayAsYouGoWithholding.PaymentType.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR241 = currentValue.Value;
                            }
                    
                            if (sbrITPAYGWithholdingDetailsCollection_DFRW_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}IncomeTax.PayAsYouGoWithholding.TaxWithheld.Amount", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR207 = decimal.Parse(currentValue.Value);
                            }
                        }  
                    } 
                    #endregion End of Tuple sbrITPAYGWithholdingDetailsCollection
                    if (validationMode) ValidateRPItpaygWithholdingDetailsCollection(errors, sbrITPAYGWithholdingDetailsCollection, sbrRPContexts, report);
            
            
                    #region sbrITPAYGWithholdingDetailsCollection
            
                    if (sbrITPAYGWithholdingDetailsCollection != null)
                    {
                
                        IEnumerable<SBRElement> sbrITPAYGWithholdingDetailsCollection_DNOABN_RPCollection;
                        sbrITPAYGWithholdingDetailsCollection_DNOABN_RPCollection = sbrITPAYGWithholdingDetailsCollection.Where(
                            ce => ce.ChildElements != null && 
                            ce.ChildElements.Any(te => te.Name == "IncomeTax.PayAsYouGoWithholding.PaymentType.Code" && te.Value == "DNOABN" &&
                            sbrRPContexts.Contains(te.Context)));
                        if (sbrITPAYGWithholdingDetailsCollection_DNOABN_RPCollection != null && sbrITPAYGWithholdingDetailsCollection_DNOABN_RPCollection.Count() > 0)
                        {
                            SBRElement sbrITPAYGWithholdingDetailsCollection_DNOABN_RP;
                            sbrITPAYGWithholdingDetailsCollection_DNOABN_RP = sbrITPAYGWithholdingDetailsCollection_DNOABN_RPCollection.ElementAt(0);
                
                            IDictionary<string, SBRElement> sbrITPAYGWithholdingDetailsCollection_DNOABN_RPTupleElementMap;
                            sbrITPAYGWithholdingDetailsCollection_DNOABN_RPTupleElementMap = sbrITPAYGWithholdingDetailsCollection_DNOABN_RP.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                
                            report.ItpaygWithholdingDetailsCollection_DNOABN_RPExists = true;
                            report.ItpaygWithholdingDetailsCollection_DNOABN_RPCount = sbrITPAYGWithholdingDetailsCollection_DNOABN_RPCollection.Count();
                            report.ItpaygWithholdingDetailsCollection_DNOABN_RPOccurrenceIndex = sbrITPAYGWithholdingDetailsCollection_DNOABN_RP.OccurrenceIndex + 1;
                            report.ItpaygWithholdingDetailsCollection_DNOABN_RPLastOccurrenceIndex = sbrITPAYGWithholdingDetailsCollection_DNOABN_RPCollection.Last().OccurrenceIndex + 1;
                    
                            if (sbrITPAYGWithholdingDetailsCollection_DNOABN_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}IncomeTax.PayAsYouGoWithholding.PaymentType.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR242 = currentValue.Value;
                            }
                    
                            if (sbrITPAYGWithholdingDetailsCollection_DNOABN_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}IncomeTax.PayAsYouGoWithholding.TaxWithheld.Amount", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR208 = decimal.Parse(currentValue.Value);
                            }
                        }  
                    } 
                    #endregion End of Tuple sbrITPAYGWithholdingDetailsCollection
                    if (validationMode) ValidateRPItpaygWithholdingDetailsCollection1(errors, sbrITPAYGWithholdingDetailsCollection, sbrRPContexts, report);
            
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}IncomeTax.PayAsYouGoWithholding.TaxWithheldFromInvestmentIncomeAndInterest.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR209 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldFromCloselyHeldTrustTFNNotQuoted.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR299 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}IncomeTax.PayAsYouGoWithholding.CreditForCapitalGainsWithheldFromForeignResidents.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR367 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}IncomeTax.PayAsYouGoWithholding.CreditsRefundableOther.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR211 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}IncomeTax.Deduction.CreditsEligibleTotal.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR317 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}IncomeTax.TotalRemainderOfRefundableTaxOffsets.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR326 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}IncomeTax.LiabilityInstalmentsTotal.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR205 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}IncomeTax.PayableOrRefundableTotal.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR360 = decimal.Parse(currentValues[0].Value);
                    }
        
                    IEnumerable<SBRElement> sbrDeclarationCollection;
                    if (populatedReportElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Declaration", out currentValue))
                        sbrDeclarationCollection = currentValue.Occurrences;
                    else
                        sbrDeclarationCollection = null;
            
                    #region sbrDeclarationCollection
            
                    if (sbrDeclarationCollection != null)
                    {
                
                        IEnumerable<SBRElement> sbrDeclarationCollection_TrueAndCorrect_RPCollection;
                        sbrDeclarationCollection_TrueAndCorrect_RPCollection = sbrDeclarationCollection.Where(
                            ce => ce.ChildElements != null && 
                            ce.ChildElements.Any(te => te.Name == "Declaration.StatementType.Code" && te.Value == "TrueAndCorrect" &&
                            sbrRPContexts.Contains(te.Context)));
                        if (sbrDeclarationCollection_TrueAndCorrect_RPCollection != null && sbrDeclarationCollection_TrueAndCorrect_RPCollection.Count() > 0)
                        {
                            SBRElement sbrDeclarationCollection_TrueAndCorrect_RP;
                            sbrDeclarationCollection_TrueAndCorrect_RP = sbrDeclarationCollection_TrueAndCorrect_RPCollection.ElementAt(0);
                
                            IDictionary<string, SBRElement> sbrDeclarationCollection_TrueAndCorrect_RPTupleElementMap;
                            sbrDeclarationCollection_TrueAndCorrect_RPTupleElementMap = sbrDeclarationCollection_TrueAndCorrect_RP.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                
                            report.DeclarationCollection_TrueAndCorrect_RPExists = true;
                            report.DeclarationCollection_TrueAndCorrect_RPCount = sbrDeclarationCollection_TrueAndCorrect_RPCollection.Count();
                            report.DeclarationCollection_TrueAndCorrect_RPOccurrenceIndex = sbrDeclarationCollection_TrueAndCorrect_RP.OccurrenceIndex + 1;
                            report.DeclarationCollection_TrueAndCorrect_RPLastOccurrenceIndex = sbrDeclarationCollection_TrueAndCorrect_RPCollection.Last().OccurrenceIndex + 1;
                    
                            if (sbrDeclarationCollection_TrueAndCorrect_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Declaration.StatementType.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR262 = currentValue.Value;
                            }
                    
                            if (sbrDeclarationCollection_TrueAndCorrect_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Declaration.StatementAccepted.Indicator", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR239 = ToBoolean(currentValue.Value);
                            }
                    
                            if (sbrDeclarationCollection_TrueAndCorrect_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Declaration.Statement.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR240 = currentValue.Value;
                            }
                    
                            if (sbrDeclarationCollection_TrueAndCorrect_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Declaration.Signature.Date", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR236 = DateTime.Parse(currentValue.Value);
                            }
                    
                            if (sbrDeclarationCollection_TrueAndCorrect_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Declaration.SignatoryIdentifier.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR282 = currentValue.Value;
                            }
                
                            IEnumerable<SBRElement> sbrDeclaration_PersonUnstructuredNameCollection;
                            if (sbrDeclarationCollection_TrueAndCorrect_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}PersonUnstructuredName", out currentValue))
                                sbrDeclaration_PersonUnstructuredNameCollection = currentValue.Occurrences;
                            else
                                sbrDeclaration_PersonUnstructuredNameCollection = null;
                    
                            #region sbrDeclaration_PersonUnstructuredNameCollection
                    
                            if (sbrDeclaration_PersonUnstructuredNameCollection != null)
                            {
                                SBRElement sbrDeclaration_PersonUnstructuredName;
                                sbrDeclaration_PersonUnstructuredName = sbrDeclaration_PersonUnstructuredNameCollection.ElementAt(0);
                        
                                IDictionary<string, SBRElement> sbrDeclaration_PersonUnstructuredNameTupleElementMap;
                                sbrDeclaration_PersonUnstructuredNameTupleElementMap = sbrDeclaration_PersonUnstructuredName.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                        
                                if (sbrDeclaration_PersonUnstructuredNameTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}PersonUnstructuredName.Usage.Code", out currentValue))
                                {
                                    if (!currentValue.IsNil) report.CTR314 = currentValue.Value;
                                }
                        
                                if (sbrDeclaration_PersonUnstructuredNameTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}PersonUnstructuredName.FullName.Text", out currentValue))
                                {
                                    if (!currentValue.IsNil) report.CTR313 = currentValue.Value;
                                }
                            } 
                            #endregion End of Tuple sbrDeclaration_PersonUnstructuredNameCollection
                        }  
                    } 
                    #endregion End of Tuple sbrDeclarationCollection
                    if (validationMode) ValidateRPDeclarationCollection(errors, sbrDeclarationCollection, sbrRPContexts, report);
            
        
                    IEnumerable<SBRElement> sbrPersonNameDetailsCollection;
                    if (populatedReportElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}PersonNameDetails", out currentValue))
                        sbrPersonNameDetailsCollection = currentValue.Occurrences;
                    else
                        sbrPersonNameDetailsCollection = null;
            
                    #region sbrPersonNameDetailsCollection
            
                    if (sbrPersonNameDetailsCollection != null)
                    {
                        report.PersonNameDetailsCollectionExists = true;
                        report.PersonNameDetailsCollectionCount = sbrPersonNameDetailsCollection.Count();
                
                        IEnumerable<SBRElement> sbrPersonNameDetailsCollection_LGL_DeclarationSignatory_RPCollection;
                        sbrPersonNameDetailsCollection_LGL_DeclarationSignatory_RPCollection = sbrPersonNameDetailsCollection.Where(
                            ce => ce.ChildElements != null && 
                            ce.ChildElements.Any(te => te.Name == "PersonNameDetails.PersonNameType.Code" && te.Value == "LGL") &&
                            ce.ChildElements.Any(te => te.Name == "PersonNameDetails.Usage.Code" && te.Value == "DeclarationSignatory" &&
                            sbrRPContexts.Contains(te.Context)));
                        if (sbrPersonNameDetailsCollection_LGL_DeclarationSignatory_RPCollection != null && sbrPersonNameDetailsCollection_LGL_DeclarationSignatory_RPCollection.Count() > 0)
                        {
                            SBRElement sbrPersonNameDetailsCollection_LGL_DeclarationSignatory_RP;
                            sbrPersonNameDetailsCollection_LGL_DeclarationSignatory_RP = sbrPersonNameDetailsCollection_LGL_DeclarationSignatory_RPCollection.ElementAt(0);
                
                            IDictionary<string, SBRElement> sbrPersonNameDetailsCollection_LGL_DeclarationSignatory_RPTupleElementMap;
                            sbrPersonNameDetailsCollection_LGL_DeclarationSignatory_RPTupleElementMap = sbrPersonNameDetailsCollection_LGL_DeclarationSignatory_RP.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                
                            report.PersonNameDetailsCollection_LGL_DeclarationSignatory_RPExists = true;
                            report.PersonNameDetailsCollection_LGL_DeclarationSignatory_RPCount = sbrPersonNameDetailsCollection_LGL_DeclarationSignatory_RPCollection.Count();
                            report.PersonNameDetailsCollection_LGL_DeclarationSignatory_RPOccurrenceIndex = sbrPersonNameDetailsCollection_LGL_DeclarationSignatory_RP.OccurrenceIndex + 1;
                            report.PersonNameDetailsCollection_LGL_DeclarationSignatory_RPLastOccurrenceIndex = sbrPersonNameDetailsCollection_LGL_DeclarationSignatory_RPCollection.Last().OccurrenceIndex + 1;
                    
                            if (sbrPersonNameDetailsCollection_LGL_DeclarationSignatory_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}PersonNameDetails.PersonNameType.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR280 = currentValue.Value;
                            }
                    
                            if (sbrPersonNameDetailsCollection_LGL_DeclarationSignatory_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}PersonNameDetails.Usage.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR279 = currentValue.Value;
                            }
                    
                            if (sbrPersonNameDetailsCollection_LGL_DeclarationSignatory_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}PersonNameDetails.Currency.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR284 = currentValue.Value;
                            }
                    
                            if (sbrPersonNameDetailsCollection_LGL_DeclarationSignatory_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}PersonNameDetails.Title.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR218 = currentValue.Value;
                            }
                    
                            if (sbrPersonNameDetailsCollection_LGL_DeclarationSignatory_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}PersonNameDetails.NameSuffix.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR220 = currentValue.Value;
                            }
                    
                            if (sbrPersonNameDetailsCollection_LGL_DeclarationSignatory_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}PersonNameDetails.FamilyName.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR219 = currentValue.Value;
                            }
                    
                            if (sbrPersonNameDetailsCollection_LGL_DeclarationSignatory_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}PersonNameDetails.GivenName.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR221 = currentValue.Value;
                            }
                    
                            if (sbrPersonNameDetailsCollection_LGL_DeclarationSignatory_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}PersonNameDetails.OtherGivenName.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR222 = currentValue.Value;
                            }
                        }  
                    } 
                    #endregion End of Tuple sbrPersonNameDetailsCollection
                    if (validationMode) ValidateRPPersonNameDetailsCollection(errors, sbrPersonNameDetailsCollection, sbrRPContexts, report);
            
        
                    IEnumerable<SBRElement> sbrElectronicContactTelephoneCollection;
                    if (populatedReportElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}ElectronicContactTelephone", out currentValue))
                        sbrElectronicContactTelephoneCollection = currentValue.Occurrences;
                    else
                        sbrElectronicContactTelephoneCollection = null;
            
                    #region sbrElectronicContactTelephoneCollection
            
                    if (sbrElectronicContactTelephoneCollection != null)
                    {
                
                        IEnumerable<SBRElement> sbrElectronicContactTelephoneCollection_03_RPCollection;
                        sbrElectronicContactTelephoneCollection_03_RPCollection = sbrElectronicContactTelephoneCollection.Where(
                            ce => ce.ChildElements != null && 
                            ce.ChildElements.Any(te => te.Name == "ElectronicContact.Telephone.Usage.Code" && te.Value == "03" &&
                            sbrRPContexts.Contains(te.Context)));
                        if (sbrElectronicContactTelephoneCollection_03_RPCollection != null && sbrElectronicContactTelephoneCollection_03_RPCollection.Count() > 0)
                        {
                            SBRElement sbrElectronicContactTelephoneCollection_03_RP;
                            sbrElectronicContactTelephoneCollection_03_RP = sbrElectronicContactTelephoneCollection_03_RPCollection.ElementAt(0);
                
                            IDictionary<string, SBRElement> sbrElectronicContactTelephoneCollection_03_RPTupleElementMap;
                            sbrElectronicContactTelephoneCollection_03_RPTupleElementMap = sbrElectronicContactTelephoneCollection_03_RP.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();
                
                            report.ElectronicContactTelephoneCollection_03_RPExists = true;
                            report.ElectronicContactTelephoneCollection_03_RPCount = sbrElectronicContactTelephoneCollection_03_RPCollection.Count();
                            report.ElectronicContactTelephoneCollection_03_RPOccurrenceIndex = sbrElectronicContactTelephoneCollection_03_RP.OccurrenceIndex + 1;
                            report.ElectronicContactTelephoneCollection_03_RPLastOccurrenceIndex = sbrElectronicContactTelephoneCollection_03_RPCollection.Last().OccurrenceIndex + 1;
                    
                            if (sbrElectronicContactTelephoneCollection_03_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}ElectronicContact.Telephone.Usage.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR275 = currentValue.Value;
                            }
                    
                            if (sbrElectronicContactTelephoneCollection_03_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}ElectronicContact.Telephone.ServiceLine.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR230 = currentValue.Value;
                            }
                    
                            if (sbrElectronicContactTelephoneCollection_03_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}ElectronicContact.Telephone.Area.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR223 = currentValue.Value;
                            }
                    
                            if (sbrElectronicContactTelephoneCollection_03_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}ElectronicContact.Telephone.Minimal.Number", out currentValue))
                            {
                                if (!currentValue.IsNil) report.CTR224 = currentValue.Value;
                            }
                        }  
                    } 
                    #endregion End of Tuple sbrElectronicContactTelephoneCollection
                    if (validationMode) ValidateRPElectronicContactTelephoneCollection(errors, sbrElectronicContactTelephoneCollection, sbrRPContexts, report);
            
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Report.CompletionHours.Number", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR217 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}RegulatoryDisclosures.GeneralInformationAboutFinancialStatements.Text", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR263 = currentValues[0].Value;
                    }
                } 
            } 
            #endregion End of Context sbrRPContext
    
            #region sbrRPInstantContext
            IEnumerable<SBRContext> sbrRPInstantContexts = null;
            SBRContext sbrRPInstantContext = null;
    
            sbrRPInstantContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.INSTANT &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 1 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty")) )).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPInstantElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPInstantContexts != null && sbrRPInstantContexts.Count() > 0)
            {
                sbrRPInstantContext = sbrRPInstantContexts.First();
        
                report.RPInstantId = sbrRPInstantContext.Id;
                report.RPInstantIdentifier = sbrRPInstantContext.EntityIdentifier;
                report.RPInstantIdentifierScheme = sbrRPInstantContext.EntityScheme;
        
                report.RPInstantCount = sbrRPInstantContexts.Count();
                report.RPInstantExists = true;
                report.RPInstantOccurrenceIndex = Array.IndexOf(contextArray, sbrRPInstantContext) + 1;
                report.RPInstantLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPInstantContexts.Last()) + 1;
        

                if (report.RPInstantIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPInstantIdentifierTFN = report.RPInstantIdentifier;
    
                report.RPInstantInstantDate = ((SBRInstantPeriod)sbrRPInstantContext.Period).InstantAsDateObject;
        
                sbrPopulatedRPInstantElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPInstantContexts.Count(); i++)
                {
                    sbrRPInstantContext = sbrRPInstantContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPInstantContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPInstantElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPInstantElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPInstantElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPInstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Capital.Losses.CarriedForward.Net.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR176 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPInstantContext
    
            #region sbrRPJAUSContext
            IEnumerable<SBRContext> sbrRPJAUSContexts = null;
            SBRContext sbrRPJAUSContext = null;
    
            sbrRPJAUSContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}InternationalJurisdictionDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Australian")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPJAUSElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPJAUSContexts != null && sbrRPJAUSContexts.Count() > 0)
            {
                sbrRPJAUSContext = sbrRPJAUSContexts.First();
        
                report.RPJAUSId = sbrRPJAUSContext.Id;
                report.RPJAUSIdentifier = sbrRPJAUSContext.EntityIdentifier;
                report.RPJAUSIdentifierScheme = sbrRPJAUSContext.EntityScheme;
        
                report.RPJAUSCount = sbrRPJAUSContexts.Count();
                report.RPJAUSExists = true;
                report.RPJAUSOccurrenceIndex = Array.IndexOf(contextArray, sbrRPJAUSContext) + 1;
                report.RPJAUSLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPJAUSContexts.Last()) + 1;
        

                if (report.RPJAUSIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPJAUSIdentifierTFN = report.RPJAUSIdentifier;
    
                report.RPJAUSStartDate = ((SBRDurationPeriod)sbrRPJAUSContext.Period).DurationStartAsDateObject;
                report.RPJAUSEndDate = ((SBRDurationPeriod)sbrRPJAUSContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPJAUSElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPJAUSContexts.Count(); i++)
                {
                    sbrRPJAUSContext = sbrRPJAUSContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPJAUSContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPJAUSElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPJAUSElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPJAUSElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Expense.Operating.Lease.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR77 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Expense.Operating.Interest.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR80 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Expense.Royalties.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR365 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPJAUSContext
    
            #region sbrRPJFOREIGNContext
            IEnumerable<SBRContext> sbrRPJFOREIGNContexts = null;
            SBRContext sbrRPJFOREIGNContext = null;
    
            sbrRPJFOREIGNContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}InternationalJurisdictionDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Foreign")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPJFOREIGNElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPJFOREIGNContexts != null && sbrRPJFOREIGNContexts.Count() > 0)
            {
                sbrRPJFOREIGNContext = sbrRPJFOREIGNContexts.First();
        
                report.RPJFOREIGNId = sbrRPJFOREIGNContext.Id;
                report.RPJFOREIGNIdentifier = sbrRPJFOREIGNContext.EntityIdentifier;
                report.RPJFOREIGNIdentifierScheme = sbrRPJFOREIGNContext.EntityScheme;
        
                report.RPJFOREIGNCount = sbrRPJFOREIGNContexts.Count();
                report.RPJFOREIGNExists = true;
                report.RPJFOREIGNOccurrenceIndex = Array.IndexOf(contextArray, sbrRPJFOREIGNContext) + 1;
                report.RPJFOREIGNLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPJFOREIGNContexts.Last()) + 1;
        

                if (report.RPJFOREIGNIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPJFOREIGNIdentifierTFN = report.RPJFOREIGNIdentifier;
    
                report.RPJFOREIGNStartDate = ((SBRDurationPeriod)sbrRPJFOREIGNContext.Period).DurationStartAsDateObject;
                report.RPJFOREIGNEndDate = ((SBRDurationPeriod)sbrRPJFOREIGNContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPJFOREIGNElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPJFOREIGNContexts.Count(); i++)
                {
                    sbrRPJFOREIGNContext = sbrRPJFOREIGNContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPJFOREIGNContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPJFOREIGNElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPJFOREIGNElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPJFOREIGNElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPJFOREIGNElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Expense.Operating.Lease.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR78 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJFOREIGNElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Expense.Operating.Interest.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR81 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPJFOREIGNElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Expense.Royalties.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR366 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPJFOREIGNContext
    
            #region sbrRPY0Context
            IEnumerable<SBRContext> sbrRPY0Contexts = null;
            SBRContext sbrRPY0Context = null;
    
            sbrRPY0Contexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}RelativePeriodDurationDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Y0")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPY0ElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPY0Contexts != null && sbrRPY0Contexts.Count() > 0)
            {
                sbrRPY0Context = sbrRPY0Contexts.First();
        
                report.RPY0Id = sbrRPY0Context.Id;
                report.RPY0Identifier = sbrRPY0Context.EntityIdentifier;
                report.RPY0IdentifierScheme = sbrRPY0Context.EntityScheme;
        
                report.RPY0Count = sbrRPY0Contexts.Count();
                report.RPY0Exists = true;
                report.RPY0OccurrenceIndex = Array.IndexOf(contextArray, sbrRPY0Context) + 1;
                report.RPY0LastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPY0Contexts.Last()) + 1;
        

                if (report.RPY0IdentifierScheme == "http://www.ato.gov.au/tfn") report.RPY0IdentifierTFN = report.RPY0Identifier;
    
                report.RPY0StartDate = ((SBRDurationPeriod)sbrRPY0Context.Period).DurationStartAsDateObject;
                report.RPY0EndDate = ((SBRDurationPeriod)sbrRPY0Context.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPY0ElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPY0Contexts.Count(); i++)
                {
                    sbrRPY0Context = sbrRPY0Contexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPY0Context.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPY0ElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPY0ElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPY0ElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPY0ElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Expense.LandcareOperationsAcceleratedDepreciation.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR111 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPY0Context
    
            #region sbrRPY0Minus1Context
            IEnumerable<SBRContext> sbrRPY0Minus1Contexts = null;
            SBRContext sbrRPY0Minus1Context = null;
    
            sbrRPY0Minus1Contexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}RelativePeriodDurationDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Y0Minus1")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPY0Minus1ElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPY0Minus1Contexts != null && sbrRPY0Minus1Contexts.Count() > 0)
            {
                sbrRPY0Minus1Context = sbrRPY0Minus1Contexts.First();
        
                report.RPY0Minus1Id = sbrRPY0Minus1Context.Id;
                report.RPY0Minus1Identifier = sbrRPY0Minus1Context.EntityIdentifier;
                report.RPY0Minus1IdentifierScheme = sbrRPY0Minus1Context.EntityScheme;
        
                report.RPY0Minus1Count = sbrRPY0Minus1Contexts.Count();
                report.RPY0Minus1Exists = true;
                report.RPY0Minus1OccurrenceIndex = Array.IndexOf(contextArray, sbrRPY0Minus1Context) + 1;
                report.RPY0Minus1LastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPY0Minus1Contexts.Last()) + 1;
        

                if (report.RPY0Minus1IdentifierScheme == "http://www.ato.gov.au/tfn") report.RPY0Minus1IdentifierTFN = report.RPY0Minus1Identifier;
    
                report.RPY0Minus1StartDate = ((SBRDurationPeriod)sbrRPY0Minus1Context.Period).DurationStartAsDateObject;
                report.RPY0Minus1EndDate = ((SBRDurationPeriod)sbrRPY0Minus1Context.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPY0Minus1ElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPY0Minus1Contexts.Count(); i++)
                {
                    sbrRPY0Minus1Context = sbrRPY0Minus1Contexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPY0Minus1Context.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPY0Minus1ElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPY0Minus1ElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPY0Minus1ElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPY0Minus1ElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Identifiers.AustralianRegisteredBodyNumber.Identifier", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR13 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPY0Minus1ElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Identifiers.AustralianCompanyNumber.Identifier", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR14 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPY0Minus1ElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}TaxOffsetClaim.FrankingDeficit.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR202 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPY0Minus1Context
    
            #region sbrRPYPreviousContext
            IEnumerable<SBRContext> sbrRPYPreviousContexts = null;
            SBRContext sbrRPYPreviousContext = null;
    
            sbrRPYPreviousContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}RelativePeriodDurationDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}YPrevious")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPYPreviousElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPYPreviousContexts != null && sbrRPYPreviousContexts.Count() > 0)
            {
                sbrRPYPreviousContext = sbrRPYPreviousContexts.First();
        
                report.RPYPreviousId = sbrRPYPreviousContext.Id;
                report.RPYPreviousIdentifier = sbrRPYPreviousContext.EntityIdentifier;
                report.RPYPreviousIdentifierScheme = sbrRPYPreviousContext.EntityScheme;
        
                report.RPYPreviousCount = sbrRPYPreviousContexts.Count();
                report.RPYPreviousExists = true;
                report.RPYPreviousOccurrenceIndex = Array.IndexOf(contextArray, sbrRPYPreviousContext) + 1;
                report.RPYPreviousLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPYPreviousContexts.Last()) + 1;
        

                if (report.RPYPreviousIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPYPreviousIdentifierTFN = report.RPYPreviousIdentifier;
    
                report.RPYPreviousStartDate = ((SBRDurationPeriod)sbrRPYPreviousContext.Period).DurationStartAsDateObject;
                report.RPYPreviousEndDate = ((SBRDurationPeriod)sbrRPYPreviousContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPYPreviousElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPYPreviousContexts.Count(); i++)
                {
                    sbrRPYPreviousContext = sbrRPYPreviousContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPYPreviousContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPYPreviousElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPYPreviousElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPYPreviousElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPYPreviousElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}TaxConcession.ResearchAndDevelopment.TaxOffsetNonRefundable.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR353 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPYPreviousContext
    
            #region sbrRPY0Plus1Context
            IEnumerable<SBRContext> sbrRPY0Plus1Contexts = null;
            SBRContext sbrRPY0Plus1Context = null;
    
            sbrRPY0Plus1Contexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}RelativePeriodDurationDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Y0Plus1")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPY0Plus1ElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPY0Plus1Contexts != null && sbrRPY0Plus1Contexts.Count() > 0)
            {
                sbrRPY0Plus1Context = sbrRPY0Plus1Contexts.First();
        
                report.RPY0Plus1Id = sbrRPY0Plus1Context.Id;
                report.RPY0Plus1Identifier = sbrRPY0Plus1Context.EntityIdentifier;
                report.RPY0Plus1IdentifierScheme = sbrRPY0Plus1Context.EntityScheme;
        
                report.RPY0Plus1Count = sbrRPY0Plus1Contexts.Count();
                report.RPY0Plus1Exists = true;
                report.RPY0Plus1OccurrenceIndex = Array.IndexOf(contextArray, sbrRPY0Plus1Context) + 1;
                report.RPY0Plus1LastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPY0Plus1Contexts.Last()) + 1;
        

                if (report.RPY0Plus1IdentifierScheme == "http://www.ato.gov.au/tfn") report.RPY0Plus1IdentifierTFN = report.RPY0Plus1Identifier;
    
                report.RPY0Plus1StartDate = ((SBRDurationPeriod)sbrRPY0Plus1Context.Period).DurationStartAsDateObject;
                report.RPY0Plus1EndDate = ((SBRDurationPeriod)sbrRPY0Plus1Context.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPY0Plus1ElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPY0Plus1Contexts.Count(); i++)
                {
                    sbrRPY0Plus1Context = sbrRPY0Plus1Contexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPY0Plus1Context.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPY0Plus1ElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPY0Plus1ElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPY0Plus1ElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPY0Plus1ElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}TaxConcession.ResearchAndDevelopment.TaxOffsetNonRefundable.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR302 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPY0Plus1Context
    
            #region sbrRPCFCLCContext
            IEnumerable<SBRContext> sbrRPCFCLCContexts = null;
            SBRContext sbrRPCFCLCContext = null;
    
            sbrRPCFCLCContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}CountryOfResidencyCFCClassificationDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ListedCountry")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPCFCLCElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPCFCLCContexts != null && sbrRPCFCLCContexts.Count() > 0)
            {
                sbrRPCFCLCContext = sbrRPCFCLCContexts.First();
        
                report.RPCFCLCId = sbrRPCFCLCContext.Id;
                report.RPCFCLCIdentifier = sbrRPCFCLCContext.EntityIdentifier;
                report.RPCFCLCIdentifierScheme = sbrRPCFCLCContext.EntityScheme;
        
                report.RPCFCLCCount = sbrRPCFCLCContexts.Count();
                report.RPCFCLCExists = true;
                report.RPCFCLCOccurrenceIndex = Array.IndexOf(contextArray, sbrRPCFCLCContext) + 1;
                report.RPCFCLCLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPCFCLCContexts.Last()) + 1;
        

                if (report.RPCFCLCIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPCFCLCIdentifierTFN = report.RPCFCLCIdentifier;
    
                report.RPCFCLCStartDate = ((SBRDurationPeriod)sbrRPCFCLCContext.Period).DurationStartAsDateObject;
                report.RPCFCLCEndDate = ((SBRDurationPeriod)sbrRPCFCLCContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPCFCLCElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPCFCLCContexts.Count(); i++)
                {
                    sbrRPCFCLCContext = sbrRPCFCLCContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPCFCLCContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPCFCLCElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPCFCLCElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPCFCLCElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPCFCLCElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Income.InternationalDealings.Attributed.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR156 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPCFCLCContext
    
            #region sbrRPCFCUCContext
            IEnumerable<SBRContext> sbrRPCFCUCContexts = null;
            SBRContext sbrRPCFCUCContext = null;
    
            sbrRPCFCUCContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}CountryOfResidencyCFCClassificationDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}UnlistedCountry")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPCFCUCElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPCFCUCContexts != null && sbrRPCFCUCContexts.Count() > 0)
            {
                sbrRPCFCUCContext = sbrRPCFCUCContexts.First();
        
                report.RPCFCUCId = sbrRPCFCUCContext.Id;
                report.RPCFCUCIdentifier = sbrRPCFCUCContext.EntityIdentifier;
                report.RPCFCUCIdentifierScheme = sbrRPCFCUCContext.EntityScheme;
        
                report.RPCFCUCCount = sbrRPCFCUCContexts.Count();
                report.RPCFCUCExists = true;
                report.RPCFCUCOccurrenceIndex = Array.IndexOf(contextArray, sbrRPCFCUCContext) + 1;
                report.RPCFCUCLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPCFCUCContexts.Last()) + 1;
        

                if (report.RPCFCUCIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPCFCUCIdentifierTFN = report.RPCFCUCIdentifier;
    
                report.RPCFCUCStartDate = ((SBRDurationPeriod)sbrRPCFCUCContext.Period).DurationStartAsDateObject;
                report.RPCFCUCEndDate = ((SBRDurationPeriod)sbrRPCFCUCContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPCFCUCElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPCFCUCContexts.Count(); i++)
                {
                    sbrRPCFCUCContext = sbrRPCFCUCContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPCFCUCContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPCFCUCElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPCFCUCElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPCFCUCElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPCFCUCElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Income.InternationalDealings.Attributed.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR158 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPCFCUCContext
    
            #region sbrRPTOFAContext
            IEnumerable<SBRContext> sbrRPTOFAContexts = null;
            SBRContext sbrRPTOFAContext = null;
    
            sbrRPTOFAContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}FinancialArrangementTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}TOFA")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPTOFAElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPTOFAContexts != null && sbrRPTOFAContexts.Count() > 0)
            {
                sbrRPTOFAContext = sbrRPTOFAContexts.First();
        
                report.RPTOFAId = sbrRPTOFAContext.Id;
                report.RPTOFAIdentifier = sbrRPTOFAContext.EntityIdentifier;
                report.RPTOFAIdentifierScheme = sbrRPTOFAContext.EntityScheme;
        
                report.RPTOFACount = sbrRPTOFAContexts.Count();
                report.RPTOFAExists = true;
                report.RPTOFAOccurrenceIndex = Array.IndexOf(contextArray, sbrRPTOFAContext) + 1;
                report.RPTOFALastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPTOFAContexts.Last()) + 1;
        

                if (report.RPTOFAIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPTOFAIdentifierTFN = report.RPTOFAIdentifier;
    
                report.RPTOFAStartDate = ((SBRDurationPeriod)sbrRPTOFAContext.Period).DurationStartAsDateObject;
                report.RPTOFAEndDate = ((SBRDurationPeriod)sbrRPTOFAContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPTOFAElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPTOFAContexts.Count(); i++)
                {
                    sbrRPTOFAContext = sbrRPTOFAContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPTOFAContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPTOFAElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPTOFAElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPTOFAElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPTOFAElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Income.OtherTotal.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR255 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPTOFAElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Expense.DeductibleOther.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR256 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPTOFAElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Income.GainsTotal.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR310 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPTOFAElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Expense.LossesTotal.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR311 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPTOFAElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Income.FinancialArrangementsUnrealisedGains.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR290 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPTOFAContext
    
            #region sbrRPCMPLYContext
            IEnumerable<SBRContext> sbrRPCMPLYContexts = null;
            SBRContext sbrRPCMPLYContext = null;
    
            sbrRPCMPLYContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}LifeInsurancePolicyClassTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ComplyingSuperannuationClass")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPCMPLYElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPCMPLYContexts != null && sbrRPCMPLYContexts.Count() > 0)
            {
                sbrRPCMPLYContext = sbrRPCMPLYContexts.First();
        
                report.RPCMPLYId = sbrRPCMPLYContext.Id;
                report.RPCMPLYIdentifier = sbrRPCMPLYContext.EntityIdentifier;
                report.RPCMPLYIdentifierScheme = sbrRPCMPLYContext.EntityScheme;
        
                report.RPCMPLYCount = sbrRPCMPLYContexts.Count();
                report.RPCMPLYExists = true;
                report.RPCMPLYOccurrenceIndex = Array.IndexOf(contextArray, sbrRPCMPLYContext) + 1;
                report.RPCMPLYLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPCMPLYContexts.Last()) + 1;
        

                if (report.RPCMPLYIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPCMPLYIdentifierTFN = report.RPCMPLYIdentifier;
    
                report.RPCMPLYStartDate = ((SBRDurationPeriod)sbrRPCMPLYContext.Period).DurationStartAsDateObject;
                report.RPCMPLYEndDate = ((SBRDurationPeriod)sbrRPCMPLYContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPCMPLYElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPCMPLYContexts.Count(); i++)
                {
                    sbrRPCMPLYContext = sbrRPCMPLYContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPCMPLYContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPCMPLYElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPCMPLYElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPCMPLYElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPCMPLYElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Income.Insurance.Taxable.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR179 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPCMPLYElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Income.CapitalGainsNet.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR180 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPCMPLYContext
    
            #region sbrRPORDContext
            IEnumerable<SBRContext> sbrRPORDContexts = null;
            SBRContext sbrRPORDContext = null;
    
            sbrRPORDContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}LifeInsurancePolicyClassTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}OrdinaryClass")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPORDElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPORDContexts != null && sbrRPORDContexts.Count() > 0)
            {
                sbrRPORDContext = sbrRPORDContexts.First();
        
                report.RPORDId = sbrRPORDContext.Id;
                report.RPORDIdentifier = sbrRPORDContext.EntityIdentifier;
                report.RPORDIdentifierScheme = sbrRPORDContext.EntityScheme;
        
                report.RPORDCount = sbrRPORDContexts.Count();
                report.RPORDExists = true;
                report.RPORDOccurrenceIndex = Array.IndexOf(contextArray, sbrRPORDContext) + 1;
                report.RPORDLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPORDContexts.Last()) + 1;
        

                if (report.RPORDIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPORDIdentifierTFN = report.RPORDIdentifier;
    
                report.RPORDStartDate = ((SBRDurationPeriod)sbrRPORDContext.Period).DurationStartAsDateObject;
                report.RPORDEndDate = ((SBRDurationPeriod)sbrRPORDContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPORDElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPORDContexts.Count(); i++)
                {
                    sbrRPORDContext = sbrRPORDContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPORDContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPORDElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPORDElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPORDElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPORDElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Income.CapitalGainsNet.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR181 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPORDContext
    
            #region sbrRPClosingContext
            IEnumerable<SBRContext> sbrRPClosingContexts = null;
            SBRContext sbrRPClosingContext = null;
    
            sbrRPClosingContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}OpeningAndClosingBalanceDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Closing")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPClosingElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPClosingContexts != null && sbrRPClosingContexts.Count() > 0)
            {
                sbrRPClosingContext = sbrRPClosingContexts.First();
        
                report.RPClosingId = sbrRPClosingContext.Id;
                report.RPClosingIdentifier = sbrRPClosingContext.EntityIdentifier;
                report.RPClosingIdentifierScheme = sbrRPClosingContext.EntityScheme;
        
                report.RPClosingCount = sbrRPClosingContexts.Count();
                report.RPClosingExists = true;
                report.RPClosingOccurrenceIndex = Array.IndexOf(contextArray, sbrRPClosingContext) + 1;
                report.RPClosingLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPClosingContexts.Last()) + 1;
        

                if (report.RPClosingIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPClosingIdentifierTFN = report.RPClosingIdentifier;
    
                report.RPClosingStartDate = ((SBRDurationPeriod)sbrRPClosingContext.Period).DurationStartAsDateObject;
                report.RPClosingEndDate = ((SBRDurationPeriod)sbrRPClosingContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPClosingElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPClosingContexts.Count(); i++)
                {
                    sbrRPClosingContext = sbrRPClosingContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPClosingContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPClosingElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPClosingElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPClosingElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPClosingElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Equity.Dividends.FrankingAccountBalance.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR139 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPClosingContext
    
            #region sbrRPOpeningContext
            IEnumerable<SBRContext> sbrRPOpeningContexts = null;
            SBRContext sbrRPOpeningContext = null;
    
            sbrRPOpeningContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}OpeningAndClosingBalanceDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Opening")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPOpeningElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPOpeningContexts != null && sbrRPOpeningContexts.Count() > 0)
            {
                sbrRPOpeningContext = sbrRPOpeningContexts.First();
        
                report.RPOpeningId = sbrRPOpeningContext.Id;
                report.RPOpeningIdentifier = sbrRPOpeningContext.EntityIdentifier;
                report.RPOpeningIdentifierScheme = sbrRPOpeningContext.EntityScheme;
        
                report.RPOpeningCount = sbrRPOpeningContexts.Count();
                report.RPOpeningExists = true;
                report.RPOpeningOccurrenceIndex = Array.IndexOf(contextArray, sbrRPOpeningContext) + 1;
                report.RPOpeningLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPOpeningContexts.Last()) + 1;
        

                if (report.RPOpeningIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPOpeningIdentifierTFN = report.RPOpeningIdentifier;
    
                report.RPOpeningStartDate = ((SBRDurationPeriod)sbrRPOpeningContext.Period).DurationStartAsDateObject;
                report.RPOpeningEndDate = ((SBRDurationPeriod)sbrRPOpeningContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPOpeningElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPOpeningContexts.Count(); i++)
                {
                    sbrRPOpeningContext = sbrRPOpeningContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPOpeningContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPOpeningElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPOpeningElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPOpeningElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPOpeningElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Equity.Dividends.FrankingAccountBalance.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR396 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPOpeningContext
    
            #region sbrRPClosingInstantContext
            IEnumerable<SBRContext> sbrRPClosingInstantContexts = null;
            SBRContext sbrRPClosingInstantContext = null;
    
            sbrRPClosingInstantContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.INSTANT &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}OpeningAndClosingBalanceDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Closing")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPClosingInstantElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPClosingInstantContexts != null && sbrRPClosingInstantContexts.Count() > 0)
            {
                sbrRPClosingInstantContext = sbrRPClosingInstantContexts.First();
        
                report.RPClosingInstantId = sbrRPClosingInstantContext.Id;
                report.RPClosingInstantIdentifier = sbrRPClosingInstantContext.EntityIdentifier;
                report.RPClosingInstantIdentifierScheme = sbrRPClosingInstantContext.EntityScheme;
        
                report.RPClosingInstantCount = sbrRPClosingInstantContexts.Count();
                report.RPClosingInstantExists = true;
                report.RPClosingInstantOccurrenceIndex = Array.IndexOf(contextArray, sbrRPClosingInstantContext) + 1;
                report.RPClosingInstantLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPClosingInstantContexts.Last()) + 1;
        

                if (report.RPClosingInstantIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPClosingInstantIdentifierTFN = report.RPClosingInstantIdentifier;
    
                report.RPClosingInstantInstantDate = ((SBRInstantPeriod)sbrRPClosingInstantContext.Period).InstantAsDateObject;
        
                sbrPopulatedRPClosingInstantElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPClosingInstantContexts.Count(); i++)
                {
                    sbrRPClosingInstantContext = sbrRPClosingInstantContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPClosingInstantContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPClosingInstantElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPClosingInstantElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPClosingInstantElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPClosingInstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Miscellaneous.CurrencyTranslation.Rate", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR121 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPClosingInstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Assets.Inventories.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR125 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPClosingInstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Assets.Inventories.ValuationMethod.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR126 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPClosingInstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Assets.TradeandReceivablesOther.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR128 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPClosingInstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Assets.CurrentTotal.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR129 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPClosingInstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Assets.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR130 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPClosingInstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Liabilities.NonCurrentTradeAndOtherPayables.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR131 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPClosingInstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Liabilities.CurrentTotal.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR132 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPClosingInstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Liabilities.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR133 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPClosingInstantContext
    
            #region sbrRPOpeningInstantContext
            IEnumerable<SBRContext> sbrRPOpeningInstantContexts = null;
            SBRContext sbrRPOpeningInstantContext = null;
    
            sbrRPOpeningInstantContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.INSTANT &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}OpeningAndClosingBalanceDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Opening")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPOpeningInstantElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPOpeningInstantContexts != null && sbrRPOpeningInstantContexts.Count() > 0)
            {
                sbrRPOpeningInstantContext = sbrRPOpeningInstantContexts.First();
        
                report.RPOpeningInstantId = sbrRPOpeningInstantContext.Id;
                report.RPOpeningInstantIdentifier = sbrRPOpeningInstantContext.EntityIdentifier;
                report.RPOpeningInstantIdentifierScheme = sbrRPOpeningInstantContext.EntityScheme;
        
                report.RPOpeningInstantCount = sbrRPOpeningInstantContexts.Count();
                report.RPOpeningInstantExists = true;
                report.RPOpeningInstantOccurrenceIndex = Array.IndexOf(contextArray, sbrRPOpeningInstantContext) + 1;
                report.RPOpeningInstantLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPOpeningInstantContexts.Last()) + 1;
        

                if (report.RPOpeningInstantIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPOpeningInstantIdentifierTFN = report.RPOpeningInstantIdentifier;
    
                report.RPOpeningInstantInstantDate = ((SBRInstantPeriod)sbrRPOpeningInstantContext.Period).InstantAsDateObject;
        
                sbrPopulatedRPOpeningInstantElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPOpeningInstantContexts.Count(); i++)
                {
                    sbrRPOpeningInstantContext = sbrRPOpeningInstantContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPOpeningInstantContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPOpeningInstantElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPOpeningInstantElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPOpeningInstantElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPOpeningInstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Assets.Inventories.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR123 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPOpeningInstantContext
    
            #region sbrRPLowCostContext
            IEnumerable<SBRContext> sbrRPLowCostContexts = null;
            SBRContext sbrRPLowCostContext = null;
    
            sbrRPLowCostContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}SmallBusinessDepreciatingAssetTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}LowCost")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPLowCostElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPLowCostContexts != null && sbrRPLowCostContexts.Count() > 0)
            {
                sbrRPLowCostContext = sbrRPLowCostContexts.First();
        
                report.RPLowCostId = sbrRPLowCostContext.Id;
                report.RPLowCostIdentifier = sbrRPLowCostContext.EntityIdentifier;
                report.RPLowCostIdentifierScheme = sbrRPLowCostContext.EntityScheme;
        
                report.RPLowCostCount = sbrRPLowCostContexts.Count();
                report.RPLowCostExists = true;
                report.RPLowCostOccurrenceIndex = Array.IndexOf(contextArray, sbrRPLowCostContext) + 1;
                report.RPLowCostLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPLowCostContexts.Last()) + 1;
        

                if (report.RPLowCostIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPLowCostIdentifierTFN = report.RPLowCostIdentifier;
    
                report.RPLowCostStartDate = ((SBRDurationPeriod)sbrRPLowCostContext.Period).DurationStartAsDateObject;
                report.RPLowCostEndDate = ((SBRDurationPeriod)sbrRPLowCostContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPLowCostElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPLowCostContexts.Count(); i++)
                {
                    sbrRPLowCostContext = sbrRPLowCostContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPLowCostContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPLowCostElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPLowCostElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPLowCostElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPLowCostElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Expense.DepreciatingAssets.SmallBusinessEntity.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR168 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPLowCostContext
    
            #region sbrRPGeneralPoolContext
            IEnumerable<SBRContext> sbrRPGeneralPoolContexts = null;
            SBRContext sbrRPGeneralPoolContext = null;
    
            sbrRPGeneralPoolContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}SmallBusinessDepreciatingAssetTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}GeneralPool")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPGeneralPoolElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPGeneralPoolContexts != null && sbrRPGeneralPoolContexts.Count() > 0)
            {
                sbrRPGeneralPoolContext = sbrRPGeneralPoolContexts.First();
        
                report.RPGeneralPoolId = sbrRPGeneralPoolContext.Id;
                report.RPGeneralPoolIdentifier = sbrRPGeneralPoolContext.EntityIdentifier;
                report.RPGeneralPoolIdentifierScheme = sbrRPGeneralPoolContext.EntityScheme;
        
                report.RPGeneralPoolCount = sbrRPGeneralPoolContexts.Count();
                report.RPGeneralPoolExists = true;
                report.RPGeneralPoolOccurrenceIndex = Array.IndexOf(contextArray, sbrRPGeneralPoolContext) + 1;
                report.RPGeneralPoolLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPGeneralPoolContexts.Last()) + 1;
        

                if (report.RPGeneralPoolIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPGeneralPoolIdentifierTFN = report.RPGeneralPoolIdentifier;
    
                report.RPGeneralPoolStartDate = ((SBRDurationPeriod)sbrRPGeneralPoolContext.Period).DurationStartAsDateObject;
                report.RPGeneralPoolEndDate = ((SBRDurationPeriod)sbrRPGeneralPoolContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPGeneralPoolElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPGeneralPoolContexts.Count(); i++)
                {
                    sbrRPGeneralPoolContext = sbrRPGeneralPoolContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPGeneralPoolContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPGeneralPoolElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPGeneralPoolElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPGeneralPoolElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPGeneralPoolElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Expense.DepreciatingAssets.SmallBusinessEntity.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR169 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPGeneralPoolContext
    
            #region sbrRPProspectiveRulesContext
            IEnumerable<SBRContext> sbrRPProspectiveRulesContexts = null;
            SBRContext sbrRPProspectiveRulesContext = null;
    
            sbrRPProspectiveRulesContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ConsolidationDeductionTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ProspectiveRules")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPProspectiveRulesElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPProspectiveRulesContexts != null && sbrRPProspectiveRulesContexts.Count() > 0)
            {
                sbrRPProspectiveRulesContext = sbrRPProspectiveRulesContexts.First();
        
                report.RPProspectiveRulesId = sbrRPProspectiveRulesContext.Id;
                report.RPProspectiveRulesIdentifier = sbrRPProspectiveRulesContext.EntityIdentifier;
                report.RPProspectiveRulesIdentifierScheme = sbrRPProspectiveRulesContext.EntityScheme;
        
                report.RPProspectiveRulesCount = sbrRPProspectiveRulesContexts.Count();
                report.RPProspectiveRulesExists = true;
                report.RPProspectiveRulesOccurrenceIndex = Array.IndexOf(contextArray, sbrRPProspectiveRulesContext) + 1;
                report.RPProspectiveRulesLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPProspectiveRulesContexts.Last()) + 1;
        

                if (report.RPProspectiveRulesIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPProspectiveRulesIdentifierTFN = report.RPProspectiveRulesIdentifier;
    
                report.RPProspectiveRulesStartDate = ((SBRDurationPeriod)sbrRPProspectiveRulesContext.Period).DurationStartAsDateObject;
                report.RPProspectiveRulesEndDate = ((SBRDurationPeriod)sbrRPProspectiveRulesContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPProspectiveRulesElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPProspectiveRulesContexts.Count(); i++)
                {
                    sbrRPProspectiveRulesContext = sbrRPProspectiveRulesContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPProspectiveRulesContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPProspectiveRulesElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPProspectiveRulesElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPProspectiveRulesElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPProspectiveRulesElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Expense.DeductionsTotal.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR341 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPProspectiveRulesContext
    
            #region sbrRPInvestorContext
            IEnumerable<SBRContext> sbrRPInvestorContexts = null;
            SBRContext sbrRPInvestorContext = null;
    
            sbrRPInvestorContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}EarlyInvestorDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Investor")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPInvestorElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPInvestorContexts != null && sbrRPInvestorContexts.Count() > 0)
            {
                sbrRPInvestorContext = sbrRPInvestorContexts.First();
        
                report.RPInvestorId = sbrRPInvestorContext.Id;
                report.RPInvestorIdentifier = sbrRPInvestorContext.EntityIdentifier;
                report.RPInvestorIdentifierScheme = sbrRPInvestorContext.EntityScheme;
        
                report.RPInvestorCount = sbrRPInvestorContexts.Count();
                report.RPInvestorExists = true;
                report.RPInvestorOccurrenceIndex = Array.IndexOf(contextArray, sbrRPInvestorContext) + 1;
                report.RPInvestorLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPInvestorContexts.Last()) + 1;
        

                if (report.RPInvestorIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPInvestorIdentifierTFN = report.RPInvestorIdentifier;
    
                report.RPInvestorStartDate = ((SBRDurationPeriod)sbrRPInvestorContext.Period).DurationStartAsDateObject;
                report.RPInvestorEndDate = ((SBRDurationPeriod)sbrRPInvestorContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPInvestorElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPInvestorContexts.Count(); i++)
                {
                    sbrRPInvestorContext = sbrRPInvestorContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPInvestorContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPInvestorElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPInvestorElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPInvestorElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPInvestorElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}TaxOffsetClaim.NonRefundableOther.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR368 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPInvestorContext
    
            #region sbrRPLimitedPartnersContext
            IEnumerable<SBRContext> sbrRPLimitedPartnersContexts = null;
            SBRContext sbrRPLimitedPartnersContext = null;
    
            sbrRPLimitedPartnersContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}EarlyInvestorDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}LimitedPartners")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPLimitedPartnersElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPLimitedPartnersContexts != null && sbrRPLimitedPartnersContexts.Count() > 0)
            {
                sbrRPLimitedPartnersContext = sbrRPLimitedPartnersContexts.First();
        
                report.RPLimitedPartnersId = sbrRPLimitedPartnersContext.Id;
                report.RPLimitedPartnersIdentifier = sbrRPLimitedPartnersContext.EntityIdentifier;
                report.RPLimitedPartnersIdentifierScheme = sbrRPLimitedPartnersContext.EntityScheme;
        
                report.RPLimitedPartnersCount = sbrRPLimitedPartnersContexts.Count();
                report.RPLimitedPartnersExists = true;
                report.RPLimitedPartnersOccurrenceIndex = Array.IndexOf(contextArray, sbrRPLimitedPartnersContext) + 1;
                report.RPLimitedPartnersLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPLimitedPartnersContexts.Last()) + 1;
        

                if (report.RPLimitedPartnersIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPLimitedPartnersIdentifierTFN = report.RPLimitedPartnersIdentifier;
    
                report.RPLimitedPartnersStartDate = ((SBRDurationPeriod)sbrRPLimitedPartnersContext.Period).DurationStartAsDateObject;
                report.RPLimitedPartnersEndDate = ((SBRDurationPeriod)sbrRPLimitedPartnersContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPLimitedPartnersElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPLimitedPartnersContexts.Count(); i++)
                {
                    sbrRPLimitedPartnersContext = sbrRPLimitedPartnersContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPLimitedPartnersContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPLimitedPartnersElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPLimitedPartnersElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPLimitedPartnersElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPLimitedPartnersElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}TaxOffsetClaim.NonRefundableOther.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR369 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPLimitedPartnersContext
    
            #region sbrRPInvestorYPreviousContext
            IEnumerable<SBRContext> sbrRPInvestorYPreviousContexts = null;
            SBRContext sbrRPInvestorYPreviousContext = null;
    
            sbrRPInvestorYPreviousContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 3 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}EarlyInvestorDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Investor"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}RelativePeriodDurationDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}YPrevious")) )))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPInvestorYPreviousElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPInvestorYPreviousContexts != null && sbrRPInvestorYPreviousContexts.Count() > 0)
            {
                sbrRPInvestorYPreviousContext = sbrRPInvestorYPreviousContexts.First();
        
                report.RPInvestorYPreviousId = sbrRPInvestorYPreviousContext.Id;
                report.RPInvestorYPreviousIdentifier = sbrRPInvestorYPreviousContext.EntityIdentifier;
                report.RPInvestorYPreviousIdentifierScheme = sbrRPInvestorYPreviousContext.EntityScheme;
        
                report.RPInvestorYPreviousCount = sbrRPInvestorYPreviousContexts.Count();
                report.RPInvestorYPreviousExists = true;
                report.RPInvestorYPreviousOccurrenceIndex = Array.IndexOf(contextArray, sbrRPInvestorYPreviousContext) + 1;
                report.RPInvestorYPreviousLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPInvestorYPreviousContexts.Last()) + 1;
        

                if (report.RPInvestorYPreviousIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPInvestorYPreviousIdentifierTFN = report.RPInvestorYPreviousIdentifier;
    
                report.RPInvestorYPreviousStartDate = ((SBRDurationPeriod)sbrRPInvestorYPreviousContext.Period).DurationStartAsDateObject;
                report.RPInvestorYPreviousEndDate = ((SBRDurationPeriod)sbrRPInvestorYPreviousContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPInvestorYPreviousElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPInvestorYPreviousContexts.Count(); i++)
                {
                    sbrRPInvestorYPreviousContext = sbrRPInvestorYPreviousContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPInvestorYPreviousContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPInvestorYPreviousElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPInvestorYPreviousElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPInvestorYPreviousElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPInvestorYPreviousElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}TaxOffsetClaim.NonRefundableOther.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR374 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPInvestorYPreviousContext
    
            #region sbrRPLimitedPartnersYPreviousContext
            IEnumerable<SBRContext> sbrRPLimitedPartnersYPreviousContexts = null;
            SBRContext sbrRPLimitedPartnersYPreviousContext = null;
    
            sbrRPLimitedPartnersYPreviousContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 3 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}EarlyInvestorDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}LimitedPartners"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}RelativePeriodDurationDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}YPrevious")) )))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPLimitedPartnersYPreviousElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPLimitedPartnersYPreviousContexts != null && sbrRPLimitedPartnersYPreviousContexts.Count() > 0)
            {
                sbrRPLimitedPartnersYPreviousContext = sbrRPLimitedPartnersYPreviousContexts.First();
        
                report.RPLimitedPartnersYPreviousId = sbrRPLimitedPartnersYPreviousContext.Id;
                report.RPLimitedPartnersYPreviousIdentifier = sbrRPLimitedPartnersYPreviousContext.EntityIdentifier;
                report.RPLimitedPartnersYPreviousIdentifierScheme = sbrRPLimitedPartnersYPreviousContext.EntityScheme;
        
                report.RPLimitedPartnersYPreviousCount = sbrRPLimitedPartnersYPreviousContexts.Count();
                report.RPLimitedPartnersYPreviousExists = true;
                report.RPLimitedPartnersYPreviousOccurrenceIndex = Array.IndexOf(contextArray, sbrRPLimitedPartnersYPreviousContext) + 1;
                report.RPLimitedPartnersYPreviousLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPLimitedPartnersYPreviousContexts.Last()) + 1;
        

                if (report.RPLimitedPartnersYPreviousIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPLimitedPartnersYPreviousIdentifierTFN = report.RPLimitedPartnersYPreviousIdentifier;
    
                report.RPLimitedPartnersYPreviousStartDate = ((SBRDurationPeriod)sbrRPLimitedPartnersYPreviousContext.Period).DurationStartAsDateObject;
                report.RPLimitedPartnersYPreviousEndDate = ((SBRDurationPeriod)sbrRPLimitedPartnersYPreviousContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPLimitedPartnersYPreviousElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPLimitedPartnersYPreviousContexts.Count(); i++)
                {
                    sbrRPLimitedPartnersYPreviousContext = sbrRPLimitedPartnersYPreviousContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPLimitedPartnersYPreviousContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPLimitedPartnersYPreviousElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPLimitedPartnersYPreviousElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPLimitedPartnersYPreviousElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPLimitedPartnersYPreviousElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}TaxOffsetClaim.NonRefundableOther.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.CTR373 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPLimitedPartnersYPreviousContext
            if (validationMode) ValidCountOccurrenceForExplicitTuples(errors, populatedReportElementsMap);
            return report;

        } // Of Consume Method

        #region ValidCountOccurrenceForExplicitTuples

        /// <summary>
        /// Finds SBR Elements by name - down to two levels at xbrl root and under a top level tuple
        /// </summary>
        /// <param name="populatedReportElementsMap">The populated report elements map.</param>
        /// <param name="elementName">Name of the element.</param>
        /// <returns></returns>
        private static List<SBRElement> FindElementsByName(IDictionary<string, SBRElement> populatedReportElementsMap, string elementName)
        {
            string elementNameWithPartOfNamespacePrefix = "}" + elementName;

            IEnumerable<KeyValuePair<string, SBRElement>> addressesMap = populatedReportElementsMap.Where(
                em => em.Key.EndsWith(elementNameWithPartOfNamespacePrefix) ||
                      (em.Value.ChildElements != null && em.Value.ChildElements.Any(ce => ce.Name.EndsWith(elementName))));


            List<SBRElement> foundElements = new List<SBRElement>();
            foreach (KeyValuePair<string, SBRElement> addressEntry in addressesMap)
            {
                if (addressEntry.Key.EndsWith(elementNameWithPartOfNamespacePrefix))
                {
                    if (addressEntry.Value.Occurrences == null)
                        foundElements.Add(addressEntry.Value);
                    else
                        foundElements.AddRange(addressEntry.Value.Occurrences);
                }
                else
                {
                    foreach (SBRElement childElement in addressEntry.Value.ChildElements)
                    {
                        if (childElement.Name.EndsWith(elementName))
                        {
                            if (childElement.Occurrences == null)
                                foundElements.Add(childElement);
                            else
                                foundElements.AddRange(childElement.Occurrences);
                        }
                    }
                }
            }
            return foundElements;
        }

        /// <summary>
        /// Valids the count occurrence rules for explicit tuples.
        /// </summary>
        /// <param name="errors">The errors.</param>
        /// <param name="populatedReportElementsMap">The populated report elements map.</param>
        protected virtual void ValidCountOccurrenceForExplicitTuples(List<ProcessMessageDocument> errors, IDictionary<string, SBRElement> populatedReportElementsMap)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;

            List<SBRElement> currentElements;

            #region VR.ATO.GEN.428211

/*  VR.ATO.GEN.428211
            Pay As You Go Withholding Payment type code "DFRW" is only required once

        Legacy Rule Format:
            IF COUNT (rvctc2.xx.xx:IncomeTax.PayAsYouGoWithholding.Payment Type.Code = "DFRW") > 1
               RETURN VALIDATION MESSAGE
            ENDIF
    
        Technical Business Rule Format:
            CountOccurrence(^CTR241, ^CTR241 = 'DFRW') > 1

        Data Elements:

        ^CTR241 = RP:ITPAYGWithholdingDetails:IncomeTax.PayAsYouGoWithholding.PaymentType.Code WHERE [IncomeTax.PayAsYouGoWithholding.PaymentType.Code='DFRW']
        */
            currentElements = FindElementsByName(populatedReportElementsMap, "ITPAYGWithholdingDetails");
            assertion = (currentElements.Count(el => el.ChildElements != null && el.ChildElements.Any(ce => ce.Name == "IncomeTax.PayAsYouGoWithholding.PaymentType.Code" && ce.Value  == @"DFRW")) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.428211", Severity = ProcessMessageSeverity.Error,
                    Description = @"Pay As You Go Withholding Payment type code ""DFRW"" is only required once",
                    Location = DeriveSbrElementLocation(currentElements.Last().ChildElements.First(ce => ce.Name == "IncomeTax.PayAsYouGoWithholding.PaymentType.Code")),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428211"} },
                };
                errors.Add(processMessage);
            }
            #endregion

            #region VR.ATO.GEN.428212

/*  VR.ATO.GEN.428212
            Pay As You Go Withholding Payment type code "DNOABN" is only required once

        Legacy Rule Format:
            IF COUNT (rvctc2.xx.xx:Income Tax.PayAsYouGoWithholding.PaymentType.Code = "DNOABN") > 1
               RETURN VALIDATION MESSAGE
            ENDIF
    
        Technical Business Rule Format:
            CountOccurrence(^CTR242, ^CTR242 = 'DNOABN') > 1

        Data Elements:

        ^CTR242 = RP:ITPAYGWithholdingDetails:IncomeTax.PayAsYouGoWithholding.PaymentType.Code WHERE [IncomeTax.PayAsYouGoWithholding.PaymentType.Code='DNOABN']
        */
            currentElements = FindElementsByName(populatedReportElementsMap, "ITPAYGWithholdingDetails");
            assertion = (currentElements.Count(el => el.ChildElements != null && el.ChildElements.Any(ce => ce.Name == "IncomeTax.PayAsYouGoWithholding.PaymentType.Code" && ce.Value  == @"DNOABN")) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.428212", Severity = ProcessMessageSeverity.Error,
                    Description = @"Pay As You Go Withholding Payment type code ""DNOABN"" is only required once",
                    Location = DeriveSbrElementLocation(currentElements.Last().ChildElements.First(ce => ce.Name == "IncomeTax.PayAsYouGoWithholding.PaymentType.Code")),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428212"} },
                };
                errors.Add(processMessage);
            }
            #endregion
        }
        #endregion
        #region ValidateINTDeclaration_PersonUnstructuredNameCollection
        protected virtual void ValidateINTDeclaration_PersonUnstructuredNameCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, CTR2025 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the INT Context exit out otherwise
                     if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;
                    bool Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTExists = true;

                    SBRElement CTR292Value = null;
                    string CTR292 = null;
                    SBRElement CTR291Value = null;
                    string CTR291 = null;

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}PersonUnstructuredName.Usage.Code", out CTR292Value))
                    {
                        if (!CTR292Value.IsNil) CTR292 = CTR292Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}PersonUnstructuredName.FullName.Text", out CTR291Value))
                    {
                        if (!CTR291Value.IsNil) CTR291 = CTR291Value.Value;
                    }
        
                    #region VR.ATO.GEN.436279

        /*  VR.ATO.GEN.436279
            The unstructured person name tuple must include the Usage Code 'DeclarationSignatory'.

        Legacy Rule Format:
            WHERE IN TUPLE (personunstructuredname1.xx.xx:PersonUnstructuredName) IN TUPLE(declaration2.xx.xx:Declaration)
            IF pyde.xx.xx:PersonUnstructuredName.Usage.Code <> "DeclarationSignatory"
               RETURN VALIDATION MESSAGE
            ENDIF
    
        Technical Business Rule Format:
            (^PersonUnstructuredName <> NULL) AND (^CTR292 <> 'DeclarationSignatory')

        Data Elements:

        ^CTR292 = INT:Declaration:PersonUnstructuredName:PersonUnstructuredName.Usage.Code WHERE [PersonUnstructuredName.Usage.Code='DeclarationSignatory']

        ^PersonUnstructuredName = INT:Declaration:PersonUnstructuredName
        */
                    assertion = (Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTExists != false && CTR292 != @"DeclarationSignatory");
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.436279", Severity = ProcessMessageSeverity.Error,
                            Description = @"Declaration name must have a usage code of 'DeclarationSignatory'",
                            Location = CTR292Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:PersonUnstructuredName.Usage.Code" : DeriveSbrElementLocation(CTR292Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.436279"} },
                        };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "CTR292", Value = CTR292 });
        
                        errors.Add(processMessage);
                    }
                    #endregion
                }
            }
        }
        #endregion

        #region ValidateINTDeclarationCollection
        protected virtual void ValidateINTDeclarationCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, CTR2025 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the INT Context exit out otherwise
                     if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;

                    SBRElement CTR261Value = null;
                    string CTR261 = null;
                    SBRElement CTR237Value = null;
                    bool? CTR237 = null;
                    SBRElement CTR238Value = null;
                    string CTR238 = null;
                    SBRElement CTR234Value = null;
                    DateTime? CTR234 = null;
                    SBRElement CTR281Value = null;
                    string CTR281 = null;

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Declaration.StatementType.Code", out CTR261Value))
                    {
                        if (!CTR261Value.IsNil) CTR261 = CTR261Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Declaration.StatementAccepted.Indicator", out CTR237Value))
                    {
                        if (!CTR237Value.IsNil) CTR237 = ToBoolean(CTR237Value.Value);
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Declaration.Statement.Text", out CTR238Value))
                    {
                        if (!CTR238Value.IsNil) CTR238 = CTR238Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Declaration.Signature.Date", out CTR234Value))
                    {
                        if (!CTR234Value.IsNil) CTR234 = DateTime.Parse(CTR234Value.Value);
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Declaration.SignatoryIdentifier.Text", out CTR281Value))
                    {
                        if (!CTR281Value.IsNil) CTR281 = CTR281Value.Value;
                    }
        
                    #region VR.ATO.GEN.438061

        /*  VR.ATO.GEN.438061
            If an Intermediary is present, Intermediary declaration statement type code must be supplied and must be "TrueAndCorrect".

        Legacy Rule Format:
            IF INT<>NULL AND INT:pyin.xx.xx:Declaration.StatementType.Code <> "TrueAndCorrect"
               RETURN VALIDATION MESSAGE
            ENDIF
    
        Technical Business Rule Format:
            ^INT <> NULL AND ^CTR261 <> 'TrueAndCorrect'

        Data Elements:

        ^CTR261 = INT:Declaration:Declaration.StatementType.Code WHERE [Declaration.StatementType.Code='TrueAndCorrect']

        ^INT = INT
        */
                    assertion = (report.INTExists != false && CTR261 != @"TrueAndCorrect");
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.438061", Severity = ProcessMessageSeverity.Error,
                            Description = @"Declaration statement type code is required.",
                            LongDescription = @"If an Intermediary is present, Intermediary declaration statement type code must be supplied and must be ""TrueAndCorrect"".",
                            Location = CTR261Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:Declaration.StatementType.Code" : DeriveSbrElementLocation(CTR261Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438061"} },
                        };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "CTR261", Value = CTR261 });
        
                        errors.Add(processMessage);
                    }
                    #endregion
                }
            }
        }
        #endregion

        #region ValidateINTElectronicContactTelephoneCollection
        protected virtual void ValidateINTElectronicContactTelephoneCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, CTR2025 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the INT Context exit out otherwise
                     if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;
                    bool ElectronicContactTelephoneCollection_03_INTExists = true;

                    SBRElement CTR276Value = null;
                    string CTR276 = null;

                    SBRElement CTR274Value = null;
                    string CTR274 = null;
                    SBRElement CTR215Value = null;
                    string CTR215 = null;
                    SBRElement CTR216Value = null;
                    string CTR216 = null;

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}ElectronicContact.Telephone.Usage.Code", out CTR276Value))
                    {
                        if (!CTR276Value.IsNil) CTR276 = CTR276Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}ElectronicContact.Telephone.ServiceLine.Code", out CTR274Value))
                    {
                        if (!CTR274Value.IsNil) CTR274 = CTR274Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}ElectronicContact.Telephone.Area.Code", out CTR215Value))
                    {
                        if (!CTR215Value.IsNil) CTR215 = CTR215Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}ElectronicContact.Telephone.Minimal.Number", out CTR216Value))
                    {
                        if (!CTR216Value.IsNil) CTR216 = CTR216Value.Value;
                    }
        
                    #region VR.ATO.GEN.500131

        /*  VR.ATO.GEN.500131
            Telephone must include usage code '03' and service line code '01' or '02'.

        Legacy Rule Format:
            WHERE IN TUPLE(electroniccontacttelephone1.xx.xx:ElectronicContactTelephone)
            IF pyde.xx.xx:ElectronicContact.Telephone.Usage.Code <> "03" OR pyde.xx.xx:ElectronicContact.Telephone.ServiceLine.Code <> SET ("01", "02")
               RETURN VALIDATION MESSAGE
            ENDIF
    
        Technical Business Rule Format:
            (^ElectronicContactTelephone <> NULL) AND (^CTR276 <> '03' OR NotInSet(^CTR274, '"01", "02"'))

        Data Elements:

        ^CTR276 = INT:ElectronicContactTelephone:ElectronicContact.Telephone.Usage.Code WHERE [ElectronicContact.Telephone.Usage.Code='03']

        ^CTR274 = INT:ElectronicContactTelephone:ElectronicContact.Telephone.ServiceLine.Code WHERE [ElectronicContact.Telephone.Usage.Code='03']

        ^ElectronicContactTelephone = INT:ElectronicContactTelephone
        */
                    assertion = (ElectronicContactTelephoneCollection_03_INTExists != false && (CTR276 != @"03" || !(IsMatch(CTR274, @"^(01|02)$",RegexOptions.IgnoreCase))));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.500131", Severity = ProcessMessageSeverity.Error,
                            Description = @"Telephone must include usage code '03' and service line code '01' or '02'.",
                            Location = CTR276Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:ElectronicContact.Telephone.Usage.Code" : DeriveSbrElementLocation(CTR276Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500131"} },
                        };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "CTR276", Value = CTR276 });
        
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "CTR274", Value = CTR274 });
        
                        errors.Add(processMessage);
                    }
                    #endregion
                }
            }
        }
        #endregion

        #region ValidateINTPersonUnstructuredNameCollection
        protected virtual void ValidateINTPersonUnstructuredNameCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, CTR2025 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the INT Context exit out otherwise
                     if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;

                    SBRElement CTR278Value = null;
                    string CTR278 = null;
                    SBRElement CTR214Value = null;
                    string CTR214 = null;

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}PersonUnstructuredName.Usage.Code", out CTR278Value))
                    {
                        if (!CTR278Value.IsNil) CTR278 = CTR278Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}PersonUnstructuredName.FullName.Text", out CTR214Value))
                    {
                        if (!CTR214Value.IsNil) CTR214 = CTR214Value.Value;
                    }
        
                    #region VR.ATO.GEN.438080

        /*  VR.ATO.GEN.438080
            The unstructured person name for the Intermediary contact must include a Usage Code of 'Contact'.

        Legacy Rule Format:
            WHERE IN TUPLE (xbrli\personunstructuredname1.xx.xx:PersonUnstructuredName)
            IF INT:pyde.xx.xx:PersonUnstructuredName.Usage.Code <> "Contact"
               RETURN VALIDATION MESSAGE
            ENDIF
    
        Technical Business Rule Format:
            ^CTR278 <> 'Contact'

        Data Elements:

        ^CTR278 = INT:PersonUnstructuredName:PersonUnstructuredName.Usage.Code WHERE [PersonUnstructuredName.Usage.Code='Contact']
        */
                    assertion = (CTR278 != @"Contact");
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.438080", Severity = ProcessMessageSeverity.Error,
                            Description = @"Intermediary contact name must have a Usage code of 'Contact'",
                            LongDescription = @"The unstructured person name for the Intermediary contact must include a Usage Code of 'Contact'. Please contact your software vendor in order to rectify this error.",
                            Location = CTR278Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:PersonUnstructuredName.Usage.Code" : DeriveSbrElementLocation(CTR278Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438080"} },
                        };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "CTR278", Value = CTR278 });
        
                        errors.Add(processMessage);
                    }
                    #endregion
                }
            }
        }
        #endregion

        #region ValidateRPOrganisationNameDetailsCollection
        protected virtual void ValidateRPOrganisationNameDetailsCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, CTR2025 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the RP Context exit out otherwise
                     if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;

                    SBRElement CTR228Value = null;
                    string CTR228 = null;
                    SBRElement CTR285Value = null;
                    string CTR285 = null;
                    SBRElement CTR9Value = null;
                    string CTR9 = null;

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}OrganisationNameDetails.OrganisationalNameType.Code", out CTR228Value))
                    {
                        if (!CTR228Value.IsNil) CTR228 = CTR228Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}OrganisationNameDetails.Currency.Code", out CTR285Value))
                    {
                        if (!CTR285Value.IsNil) CTR285 = CTR285Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}OrganisationNameDetails.OrganisationalName.Text", out CTR9Value))
                    {
                        if (!CTR9Value.IsNil) CTR9 = CTR9Value.Value;
                    }
        
                    #region VR.ATO.GEN.410049

        /*  VR.ATO.GEN.410049
            Organisation Name Type Code must be 'MN'

        Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalNameType.Code <> NULLORBLANK) AND (pyde.xx.xx:OrganisationNameDetails.OrganisationalNameType.Code <> "MN")
               RETURN VALIDATION MESSAGE
            ENDIF
    
        Technical Business Rule Format:
            (^CTR228 <> BLANK) AND (^CTR228 <> 'MN')

        Data Elements:

        ^CTR228 = RP:OrganisationNameDetails:OrganisationNameDetails.OrganisationalNameType.Code WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
        */
                    assertion = (string.IsNullOrWhiteSpace(CTR228) != true && CTR228 != @"MN");
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410049", Severity = ProcessMessageSeverity.Error,
                            Description = @"Organisation Name Type Code must be 'MN'",
                            Location = CTR228Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:OrganisationNameDetails.OrganisationalNameType.Code" : DeriveSbrElementLocation(CTR228Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410049"} },
                        };
                        errors.Add(processMessage);
                    }
                    #endregion
                }
            }
        }
        #endregion

        #region ValidateRPAddressDetailsCollection
        protected virtual void ValidateRPAddressDetailsCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, CTR2025 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the RP Context exit out otherwise
                     if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;

                    SBRElement CTR15Value = null;
                    string CTR15 = null;

                    SBRElement CTR267Value = null;
                    string CTR267 = null;
                    SBRElement CTR22Value = null;
                    bool? CTR22 = null;
                    SBRElement CTR268Value = null;
                    string CTR268 = null;
                    SBRElement CTR16Value = null;
                    string CTR16 = null;
                    SBRElement CTR17Value = null;
                    string CTR17 = null;
                    SBRElement CTR18Value = null;
                    string CTR18 = null;
                    SBRElement CTR19Value = null;
                    string CTR19 = null;
                    SBRElement CTR21Value = null;
                    string CTR21 = null;
                    SBRElement CTR20Value = null;
                    string CTR20 = null;
                    SBRElement CTR23Value = null;
                    string CTR23 = null;

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}AddressDetails.Line1.Text", out CTR15Value))
                    {
                        if (!CTR15Value.IsNil) CTR15 = CTR15Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}AddressDetails.Usage.Code", out CTR267Value))
                    {
                        if (!CTR267Value.IsNil) CTR267 = CTR267Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}AddressDetails.OverseasAddress.Indicator", out CTR22Value))
                    {
                        if (!CTR22Value.IsNil) CTR22 = ToBoolean(CTR22Value.Value);
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}AddressDetails.Currency.Code", out CTR268Value))
                    {
                        if (!CTR268Value.IsNil) CTR268 = CTR268Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}AddressDetails.Line2.Text", out CTR16Value))
                    {
                        if (!CTR16Value.IsNil) CTR16 = CTR16Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}AddressDetails.Line3.Text", out CTR17Value))
                    {
                        if (!CTR17Value.IsNil) CTR17 = CTR17Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}AddressDetails.Line4.Text", out CTR18Value))
                    {
                        if (!CTR18Value.IsNil) CTR18 = CTR18Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}AddressDetails.LocalityName.Text", out CTR19Value))
                    {
                        if (!CTR19Value.IsNil) CTR19 = CTR19Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}AddressDetails.Postcode.Text", out CTR21Value))
                    {
                        if (!CTR21Value.IsNil) CTR21 = CTR21Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}AddressDetails.StateOrTerritory.Code", out CTR20Value))
                    {
                        if (!CTR20Value.IsNil) CTR20 = CTR20Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}AddressDetails.Country.Code", out CTR23Value))
                    {
                        if (!CTR23Value.IsNil) CTR23 = CTR23Value.Value;
                    }
        
                    #region VR.ATO.GEN.000467

        /*  VR.ATO.GEN.000467
            Address usage code must be "POS" or "BUS"

        Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line1.Text <> NULLORBLANK)
            AND (pyde.xx.xx:AddressDetails.Usage.Code <> SET("POS","BUS"))
               RETURN VALIDATION MESSAGE
            ENDIF
    
        Technical Business Rule Format:
            (^CTR15 <> BLANK) AND (NotInSet(^CTR267, '"POS","BUS"'))

        Data Elements:

        ^CTR267 = RP:AddressDetails:AddressDetails.Usage.Code WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']

        ^CTR15 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
        */
                    assertion = (string.IsNullOrWhiteSpace(CTR15) != true && !(IsMatch(CTR267, @"^(POS|BUS)$",RegexOptions.IgnoreCase)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000467", Severity = ProcessMessageSeverity.Error,
                            Description = @"Address usage code must be ""POS"" or ""BUS""",
                            Location = CTR267Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:AddressDetails.Usage.Code" : DeriveSbrElementLocation(CTR267Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000467"} },
                        };
                        errors.Add(processMessage);
                    }
                    #endregion
                }
            }
        }
        #endregion

        #region ValidateRPOrganisationNameDetailsCollection1
        protected virtual void ValidateRPOrganisationNameDetailsCollection1(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, CTR2025 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the RP Context exit out otherwise
                     if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;
                }
            }
        }
        #endregion

        #region ValidateRPAddressDetailsCollection1
        protected virtual void ValidateRPAddressDetailsCollection1(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, CTR2025 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the RP Context exit out otherwise
                     if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;
                }
            }
        }
        #endregion

        #region ValidateRPAddressDetailsCollection2
        protected virtual void ValidateRPAddressDetailsCollection2(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, CTR2025 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the RP Context exit out otherwise
                     if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;
                }
            }
        }
        #endregion

        #region ValidateRPUltimateHoldingCompany_OrganisationNameDetailsCollection
        protected virtual void ValidateRPUltimateHoldingCompany_OrganisationNameDetailsCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, CTR2025 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the RP Context exit out otherwise
                     if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;
                    bool UltimateHoldingCompany_OrganisationNameDetailsCollection_MN_C_RPExists = true;

                    SBRElement CTR287Value = null;
                    string CTR287 = null;
                    SBRElement CTR271Value = null;
                    string CTR271 = null;
                    SBRElement CTR43Value = null;
                    string CTR43 = null;

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}OrganisationNameDetails.Currency.Code", out CTR287Value))
                    {
                        if (!CTR287Value.IsNil) CTR287 = CTR287Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}OrganisationNameDetails.OrganisationalNameType.Code", out CTR271Value))
                    {
                        if (!CTR271Value.IsNil) CTR271 = CTR271Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}OrganisationNameDetails.OrganisationalName.Text", out CTR43Value))
                    {
                        if (!CTR43Value.IsNil) CTR43 = CTR43Value.Value;
                    }
        
                    #region VR.ATO.CTR.428290

        /*  VR.ATO.CTR.428290
            Only current name to be provided for Immediate or Ultimate Holding Company

        Legacy Rule Format:
            IF (IN TUPLE(ctr.lodge.req.xx.xx:UltimateHoldingCompany) OR IN TUPLE(ctr.lodge.req.xx.xx:ImmediateHoldingCompany)) AND (pyde.xx.xx:OrganisationNameDetails.Currency.Code <> "C")
               RETURN VALIDATION MESSAGE
            ENDIF
    
        Technical Business Rule Format:
            (^OrganisationNameDetails <> NULL) AND (^CTR287 <> 'C')

        Data Elements:

        ^CTR287 = RP:UltimateHoldingCompany:OrganisationNameDetails:OrganisationNameDetails.Currency.Code WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']

        ^OrganisationNameDetails = RP:UltimateHoldingCompany:OrganisationNameDetails
        */
                    assertion = (UltimateHoldingCompany_OrganisationNameDetailsCollection_MN_C_RPExists != false && CTR287 != @"C");
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CTR.428290", Severity = ProcessMessageSeverity.Error,
                            Description = @"Only current name to be provided for Immediated or Ultimate Holding Company",
                            Location = CTR287Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:OrganisationNameDetails.Currency.Code" : DeriveSbrElementLocation(CTR287Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CTR.428290"} },
                        };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "CTR287", Value = CTR287 });
        
                        errors.Add(processMessage);
                    }
                    #endregion
                }
            }
        }
        #endregion

        #region ValidateRPImmediateHoldingCompany_OrganisationNameDetailsCollection
        protected virtual void ValidateRPImmediateHoldingCompany_OrganisationNameDetailsCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, CTR2025 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the RP Context exit out otherwise
                     if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;
                    bool ImmediateHoldingCompany_OrganisationNameDetailsCollection_MN_C_RPExists = true;

                    SBRElement CTR288Value = null;
                    string CTR288 = null;
                    SBRElement CTR272Value = null;
                    string CTR272 = null;
                    SBRElement CTR46Value = null;
                    string CTR46 = null;

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}OrganisationNameDetails.Currency.Code", out CTR288Value))
                    {
                        if (!CTR288Value.IsNil) CTR288 = CTR288Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}OrganisationNameDetails.OrganisationalNameType.Code", out CTR272Value))
                    {
                        if (!CTR272Value.IsNil) CTR272 = CTR272Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}OrganisationNameDetails.OrganisationalName.Text", out CTR46Value))
                    {
                        if (!CTR46Value.IsNil) CTR46 = CTR46Value.Value;
                    }
        
                    #region VR.ATO.CTR.428290

        /*  VR.ATO.CTR.428290
            Only current name to be provided for Immediate or Ultimate Holding Company

        Legacy Rule Format:
            IF (IN TUPLE(ctr.lodge.req.xx.xx:UltimateHoldingCompany) OR IN TUPLE(ctr.lodge.req.xx.xx:ImmediateHoldingCompany)) AND (pyde.xx.xx:OrganisationNameDetails.Currency.Code <> "C")
               RETURN VALIDATION MESSAGE
            ENDIF
    
        Technical Business Rule Format:
            (^OrganisationNameDetails <> NULL) AND (^CTR288 <> 'C')

        Data Elements:

        ^CTR288 = RP:ImmediateHoldingCompany:OrganisationNameDetails:OrganisationNameDetails.Currency.Code WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']

        ^OrganisationNameDetails = RP:ImmediateHoldingCompany:OrganisationNameDetails
        */
                    assertion = (ImmediateHoldingCompany_OrganisationNameDetailsCollection_MN_C_RPExists != false && CTR288 != @"C");
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CTR.428290", Severity = ProcessMessageSeverity.Error,
                            Description = @"Only current name to be provided for Immediated or Ultimate Holding Company",
                            Location = CTR288Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:OrganisationNameDetails.Currency.Code" : DeriveSbrElementLocation(CTR288Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CTR.428290"} },
                        };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "CTR288", Value = CTR288 });
        
                        errors.Add(processMessage);
                    }
                    #endregion
                }
            }
        }
        #endregion

        #region ValidateRPItpaygWithholdingDetailsCollection
        protected virtual void ValidateRPItpaygWithholdingDetailsCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, CTR2025 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the RP Context exit out otherwise
                     if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;

                    SBRElement CTR241Value = null;
                    string CTR241 = null;
                    SBRElement CTR207Value = null;
                    decimal? CTR207 = null;

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}IncomeTax.PayAsYouGoWithholding.PaymentType.Code", out CTR241Value))
                    {
                        if (!CTR241Value.IsNil) CTR241 = CTR241Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}IncomeTax.PayAsYouGoWithholding.TaxWithheld.Amount", out CTR207Value))
                    {
                        if (!CTR207Value.IsNil) CTR207 = decimal.Parse(CTR207Value.Value);
                    }
        
                    #region VR.ATO.CTR.428213

        /*  VR.ATO.CTR.428213
            PAYG Withholding Payment Type Code must be "DFRW" or "DNOABN"

        Legacy Rule Format:
            IF  (rvctc2.xx.xx:IncomeTax.PayAsYouGoWithholding.PaymentType.Code <> NULLORBLANK) AND (rvctc2.xx.xx:IncomeTax.PayAsYouGoWithholding.PaymentType.Code) <> SET("DFRW","DNOABN")
               RETURN VALIDATION MESSAGE
            ENDIF
    
        Technical Business Rule Format:
            (^CTR241 <> BLANK) AND NotInSet(^CTR241, '"DFRW","DNOABN"')

        Data Elements:

        ^CTR241 = RP:ITPAYGWithholdingDetails:IncomeTax.PayAsYouGoWithholding.PaymentType.Code WHERE [IncomeTax.PayAsYouGoWithholding.PaymentType.Code='DFRW']
        */
                    assertion = (string.IsNullOrWhiteSpace(CTR241) != true && !(IsMatch(CTR241, @"^(DFRW|DNOABN)$",RegexOptions.IgnoreCase)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.CTR.428213", Severity = ProcessMessageSeverity.Error,
                            Description = @"PAYG Withholding Payment Type Code must be ""DFRW"" or ""DNOABN""",
                            Location = CTR241Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:IncomeTax.PayAsYouGoWithholding.PaymentType.Code" : DeriveSbrElementLocation(CTR241Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.CTR.428213"} },
                        };
                        errors.Add(processMessage);
                    }
                    #endregion
                }
            }
        }
        #endregion

        #region ValidateRPItpaygWithholdingDetailsCollection1
        protected virtual void ValidateRPItpaygWithholdingDetailsCollection1(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, CTR2025 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the RP Context exit out otherwise
                     if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;
                }
            }
        }
        #endregion

        #region ValidateRPDeclarationCollection
        protected virtual void ValidateRPDeclarationCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, CTR2025 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the RP Context exit out otherwise
                     if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;

                    SBRElement CTR262Value = null;
                    string CTR262 = null;
                    SBRElement CTR239Value = null;
                    bool? CTR239 = null;
                    SBRElement CTR240Value = null;
                    string CTR240 = null;
                    SBRElement CTR236Value = null;
                    DateTime? CTR236 = null;
                    SBRElement CTR282Value = null;
                    string CTR282 = null;

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Declaration.StatementType.Code", out CTR262Value))
                    {
                        if (!CTR262Value.IsNil) CTR262 = CTR262Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Declaration.StatementAccepted.Indicator", out CTR239Value))
                    {
                        if (!CTR239Value.IsNil) CTR239 = ToBoolean(CTR239Value.Value);
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Declaration.Statement.Text", out CTR240Value))
                    {
                        if (!CTR240Value.IsNil) CTR240 = CTR240Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Declaration.Signature.Date", out CTR236Value))
                    {
                        if (!CTR236Value.IsNil) CTR236 = DateTime.Parse(CTR236Value.Value);
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}Declaration.SignatoryIdentifier.Text", out CTR282Value))
                    {
                        if (!CTR282Value.IsNil) CTR282 = CTR282Value.Value;
                    }
        
                    #region VR.ATO.GEN.438062

        /*  VR.ATO.GEN.438062
            If an Intermediary is not present, Reporting Party Declaration Statement Type Code must be supplied and must be "TrueAndCorrect".

        Legacy Rule Format:
            IF INT=NULL AND RP:pyin.xx.xx:Declaration.StatementType.Code <> "TrueAndCorrect"
               RETURN VALIDATION MESSAGE
            ENDIF
    
        Technical Business Rule Format:
            ^INT = NULL AND ^CTR262 <> 'TrueAndCorrect'

        Data Elements:

        ^CTR262 = RP:Declaration:Declaration.StatementType.Code WHERE [Declaration.StatementType.Code='TrueAndCorrect']

        ^INT = INT
        */
                    assertion = (report.INTExists == false && CTR262 != @"TrueAndCorrect");
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.438062", Severity = ProcessMessageSeverity.Error,
                            Description = @"Declaration statement type code is required.",
                            LongDescription = @"If an Intermediary is not present, Reporting Party Declaration Statement Type Code must be supplied and must be ""TrueAndCorrect"".",
                            Location = CTR262Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:Declaration.StatementType.Code" : DeriveSbrElementLocation(CTR262Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438062"} },
                        };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "CTR262", Value = CTR262 });
        
                        errors.Add(processMessage);
                    }
                    #endregion
                }
            }
        }
        #endregion

        #region ValidateRPPersonNameDetailsCollection
        protected virtual void ValidateRPPersonNameDetailsCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, CTR2025 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the RP Context exit out otherwise
                     if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;

                    SBRElement CTR279Value = null;
                    string CTR279 = null;

                    SBRElement CTR280Value = null;
                    string CTR280 = null;
                    SBRElement CTR284Value = null;
                    string CTR284 = null;
                    SBRElement CTR218Value = null;
                    string CTR218 = null;
                    SBRElement CTR220Value = null;
                    string CTR220 = null;
                    SBRElement CTR219Value = null;
                    string CTR219 = null;
                    SBRElement CTR221Value = null;
                    string CTR221 = null;
                    SBRElement CTR222Value = null;
                    string CTR222 = null;

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}PersonNameDetails.Usage.Code", out CTR279Value))
                    {
                        if (!CTR279Value.IsNil) CTR279 = CTR279Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}PersonNameDetails.PersonNameType.Code", out CTR280Value))
                    {
                        if (!CTR280Value.IsNil) CTR280 = CTR280Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}PersonNameDetails.Currency.Code", out CTR284Value))
                    {
                        if (!CTR284Value.IsNil) CTR284 = CTR284Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}PersonNameDetails.Title.Text", out CTR218Value))
                    {
                        if (!CTR218Value.IsNil) CTR218 = CTR218Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}PersonNameDetails.NameSuffix.Text", out CTR220Value))
                    {
                        if (!CTR220Value.IsNil) CTR220 = CTR220Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}PersonNameDetails.FamilyName.Text", out CTR219Value))
                    {
                        if (!CTR219Value.IsNil) CTR219 = CTR219Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}PersonNameDetails.GivenName.Text", out CTR221Value))
                    {
                        if (!CTR221Value.IsNil) CTR221 = CTR221Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}PersonNameDetails.OtherGivenName.Text", out CTR222Value))
                    {
                        if (!CTR222Value.IsNil) CTR222 = CTR222Value.Value;
                    }
        
                    #region VR.ATO.GEN.428219

        /*  VR.ATO.GEN.428219
            Declaration signatory must be supplied

        Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.Usage.Code <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.Usage.Code <> "DeclarationSignatory")
               RETURN VALIDATION MESSAGE
            ENDIF
    
        Technical Business Rule Format:
            (^CTR279 <> BLANK) AND (^CTR279 <> 'DeclarationSignatory')

        Data Elements:

        ^CTR279 = RP:PersonNameDetails:PersonNameDetails.Usage.Code WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='DeclarationSignatory']
        */
                    assertion = (string.IsNullOrWhiteSpace(CTR279) != true && CTR279 != @"DeclarationSignatory");
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.428219", Severity = ProcessMessageSeverity.Error,
                            Description = @"Declaration signatory must be supplied",
                            Location = CTR279Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:PersonNameDetails.Usage.Code" : DeriveSbrElementLocation(CTR279Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428219"} },
                        };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "CTR279", Value = CTR279 });
        
                        errors.Add(processMessage);
                    }
                    #endregion
        
                    #region VR.ATO.GEN.428221

        /*  VR.ATO.GEN.428221
            Invalid Reporting Party Name Type code.  Valid code is "LGL"

        Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.PersonNameType.Code <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.PersonNameType.Code <> "LGL")
               RETURN VALIDATION MESSAGE
            ENDIF
    
        Technical Business Rule Format:
            (^CTR280 <> BLANK) AND (^CTR280 <> 'LGL')

        Data Elements:

        ^CTR280 = RP:PersonNameDetails:PersonNameDetails.PersonNameType.Code WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='DeclarationSignatory']
        */
                    assertion = (string.IsNullOrWhiteSpace(CTR280) != true && CTR280 != @"LGL");
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.428221", Severity = ProcessMessageSeverity.Error,
                            Description = @"Invalid Person Name Type code. Valid code is ""LGL""",
                            Location = CTR280Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:PersonNameDetails.PersonNameType.Code" : DeriveSbrElementLocation(CTR280Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428221"} },
                        };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "CTR280", Value = CTR280 });
        
                        errors.Add(processMessage);
                    }
                    #endregion
                }
            }
        }
        #endregion

        #region ValidateRPElectronicContactTelephoneCollection
        protected virtual void ValidateRPElectronicContactTelephoneCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, CTR2025 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the RP Context exit out otherwise
                     if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;
                    bool ElectronicContactTelephoneCollection_03_RPExists = true;

                    SBRElement CTR275Value = null;
                    string CTR275 = null;

                    SBRElement CTR230Value = null;
                    string CTR230 = null;
                    SBRElement CTR223Value = null;
                    string CTR223 = null;
                    SBRElement CTR224Value = null;
                    string CTR224 = null;

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}ElectronicContact.Telephone.Usage.Code", out CTR275Value))
                    {
                        if (!CTR275Value.IsNil) CTR275 = CTR275Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}ElectronicContact.Telephone.ServiceLine.Code", out CTR230Value))
                    {
                        if (!CTR230Value.IsNil) CTR230 = CTR230Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}ElectronicContact.Telephone.Area.Code", out CTR223Value))
                    {
                        if (!CTR223Value.IsNil) CTR223 = CTR223Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/ctr}ElectronicContact.Telephone.Minimal.Number", out CTR224Value))
                    {
                        if (!CTR224Value.IsNil) CTR224 = CTR224Value.Value;
                    }
        
                    #region VR.ATO.GEN.500131

        /*  VR.ATO.GEN.500131
            Telephone must include usage code '03' and service line code '01' or '02'.

        Legacy Rule Format:
            WHERE IN TUPLE(electroniccontacttelephone1.xx.xx:ElectronicContactTelephone)
            IF pyde.xx.xx:ElectronicContact.Telephone.Usage.Code <> "03" OR pyde.xx.xx:ElectronicContact.Telephone.ServiceLine.Code <> SET ("01", "02")
               RETURN VALIDATION MESSAGE
            ENDIF
    
        Technical Business Rule Format:
            (^ElectronicContactTelephone <> NULL) AND (^CTR275 <> '03' OR NotInSet(^CTR230, '"01", "02"'))

        Data Elements:

        ^CTR275 = RP:ElectronicContactTelephone:ElectronicContact.Telephone.Usage.Code WHERE [ElectronicContact.Telephone.Usage.Code='03']

        ^CTR230 = RP:ElectronicContactTelephone:ElectronicContact.Telephone.ServiceLine.Code WHERE [ElectronicContact.Telephone.Usage.Code='03']

        ^ElectronicContactTelephone = RP:ElectronicContactTelephone
        */
                    assertion = (ElectronicContactTelephoneCollection_03_RPExists != false && (CTR275 != @"03" || !(IsMatch(CTR230, @"^(01|02)$",RegexOptions.IgnoreCase))));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.500131", Severity = ProcessMessageSeverity.Error,
                            Description = @"Telephone must include usage code '03' and service line code '01' or '02'.",
                            Location = CTR275Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:ElectronicContact.Telephone.Usage.Code" : DeriveSbrElementLocation(CTR275Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500131"} },
                        };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "CTR275", Value = CTR275 });
        
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "CTR230", Value = CTR230 });
        
                        errors.Add(processMessage);
                    }
                    #endregion
                }
            }
        }
        #endregion

    } // Of Class
} // Of Namespace