using System;
using System.Collections.Generic;
using Au.Gov.Sbr.Xbrl.Document;
using Au.Gov.Sbr.Xbrl.Metadata;
using System.Linq;
using System.Xml;
using System.Text.RegularExpressions;
using DataContracts;

namespace Ato.EN.IntegrationServices.CodeGenerationSMSFAR
{
    public class SMSFAR2025Consumer
    {

        private static Regex StripWhitespace = new Regex(@"\s*");

        #region Helper/Validation Methods
        private static bool? ToBoolean(string str)
        {
            bool returnValue;
            string value = (str ?? "").Trim();
            if (value == "0" || value == "1")
                return Convert.ToBoolean(Convert.ToInt32(value));

            if (Boolean.TryParse(value, out returnValue))
                return returnValue;
            return null;
        }

        private static string GetValueOrEmpty(bool? val)
        {
            return (val.HasValue) ? val.ToString().ToLower() : string.Empty;
        }

        private static string GetValueOrEmptyDate(DateTime? val)
        {
            return (val.HasValue) ? val.Value.ToString("yyyy-MM-dd") : string.Empty;
        }

        private static string GetValueOrEmptyDateTime(DateTime? val)
        {
            return (val.HasValue) ? val.Value.ToString("yyyy-MM-ddTHH:mm:ssZ") : string.Empty;
        }

        private static string GetValueOrEmptyDate(string val)
        {
            DateTime dateTimeOut;
            if (DateTime.TryParse(val, out dateTimeOut))
            {
                return dateTimeOut.ToString("yyyy-MM-dd");
            }
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmptyDateTime(string val)
        {
            DateTime dateTimeOut;
            if (DateTime.TryParse(val, out dateTimeOut))
            {
                return dateTimeOut.ToString("yyyy-MM-ddTHH:mm:ssZ");
            }
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmpty(string val)
        {
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmpty(decimal? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        private static string GetValueOrEmpty(int? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        /// <summary>
        /// Derives the SBR element location.
        /// </summary>
        /// <param name="sbrElement">The SBR element.</param>
        /// <param name="addOccurrenceIndex">if set to <c>true</c> [add occurrence the occurrence index to the path].</param>
        /// <returns></returns>
        private static string DeriveSbrElementLocation(SBRElement sbrElement, bool addOccurrenceIndex = false)
        {
            if (sbrElement == null) return "/xbrli:xbrl";
            int xpathOccurrence = sbrElement.OccurrenceIndex + 1;
            string location;

            if (addOccurrenceIndex)
            {
                location = "/" + sbrElement.NamespacePrefix + ":" + sbrElement.Name + "[" +
                           xpathOccurrence + "]";
            }
            else
            {
                location = "/" + sbrElement.NamespacePrefix + ":" + sbrElement.Name;
            }

            if (sbrElement.IsInTuple)
            {
                location = DeriveSbrElementLocation(sbrElement.ParentTuple, true) + location;
            }
            else
            {
                location = "/xbrli:xbrl" + location;
            }

            return location;
        }

        private static bool IsMatch(int? field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(Convert.ToString(field.Value), expression, options);
        }

        private static bool IsMatch(string field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(field, expression, options);
        }

        public DateTime AsDate(string dateAsString)
        {
            DateTime response = DateTime.MinValue;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date;
            }

            return response;
        }

        #endregion

        public SMSFAR2025 Consume(SBRDocument sbrDocument)
        {
            List<ProcessMessageDocument> errors;
            return Consume(sbrDocument, false, out errors);
        }

        /// <summary>
        /// Allows for validation of rules that can not be generated and must run at a lower level.
        /// A sub-class will need to be created with an override for this method - which does nothing in its base implementation.
        /// </summary>
        /// <param name="sbrDocument">The SBR document.</param>
        /// <param name="populatedReportElementsMap">The populated report elements map can be used to find data elements and tuples.</param>
        /// <param name="contextArray">The context array can be used to find context values.</param>
        /// <param name="errors">The errors collection is to be populated with validation results.</param>
        protected virtual void ValidateUnsupportedRules(SBRDocument sbrDocument, IDictionary<string, SBRElement> populatedReportElementsMap, SBRContext[] contextArray, List<ProcessMessageDocument> errors)
        {

        }

        public SMSFAR2025 Consume(SBRDocument sbrDocument, bool validationMode, out List<ProcessMessageDocument> errors)
        {
            SMSFAR2025 report;
            report = new SMSFAR2025();
            errors = new List<ProcessMessageDocument>();

            SBRContext[] contextArray;
            IDictionary<string, SBRElement> populatedReportElementsMap;

            contextArray = sbrDocument.ContextsMap.Values.ToArray();
            populatedReportElementsMap = sbrDocument.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();

            if (validationMode) ValidateUnsupportedRules(sbrDocument, populatedReportElementsMap, contextArray, errors);

            /// <summary>
            /// Current element being extracted
            /// </summary>
            SBRElement currentValue = null;
            /// <summary>
            /// Current elements being extracted
            /// </summary>
            IList<SBRElement> currentValues = null;
            /// <summary>
            /// Current occurrences being extracted
            /// </summary>
            IList<SBRElement> currentOccurrences = null;
            /// <summary>
            /// Working Variable for date time variables
            /// </summary>
            DateTime currentDateTimeValue;
            /// <summary>
            /// Working Variable for boolean variables
            /// </summary>
            bool currentBooleanValue;
            /// <summary>
            /// Working Variable for decimal variables
            /// </summary>
            decimal currentDecimalValue;
            /// <summary>
            /// Working Variable for integer variables
            /// </summary>
            int currentIntValue;
            /// <summary>
            /// Working Variable for long variables
            /// </summary>
            long currentLongValue;
            XmlDocumentFragment currentSegment;
            XmlNamespaceManager currentNamespaceManager;
            XmlNode currentNode;

            #region sbrRPContext
            IEnumerable<SBRContext> sbrRPContexts = null;
            SBRContext sbrRPContext = null;

            sbrRPContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 1 &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty")))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPContexts != null && sbrRPContexts.Count() > 0)
            {
                sbrRPContext = sbrRPContexts.First();

                report.RPId = sbrRPContext.Id;
                report.RPIdentifier = sbrRPContext.EntityIdentifier;
                report.RPIdentifierScheme = sbrRPContext.EntityScheme;

                report.RPCount = sbrRPContexts.Count();
                report.RPExists = true;
                report.RPOccurrenceIndex = Array.IndexOf(contextArray, sbrRPContext) + 1;
                report.RPLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPContexts.Last()) + 1;


                if (report.RPIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPIdentifierTFN = report.RPIdentifier;

                report.RPStartDate = ((SBRDurationPeriod)sbrRPContext.Period).DurationStartAsDateObject;
                report.RPEndDate = ((SBRDurationPeriod)sbrRPContext.Period).DurationEndAsDateObject;

                sbrPopulatedRPElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPContexts.Count(); i++)
                {
                    sbrRPContext = sbrRPContexts.ElementAt(i);

                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement });
                            }
                        }
                    }
                    report.SMSFAR3 = report.RPIdentifier;

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Report.TargetFinancial.Year", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR1 = int.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Identifiers.AustralianBusinessNumber.Identifier", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR5 = currentValues[0].Value;
                    }

                    IEnumerable<SBRElement> sbrOrganisationNameDetailsCollection;
                    if (populatedReportElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}OrganisationNameDetails", out currentValue))
                        sbrOrganisationNameDetailsCollection = currentValue.Occurrences;
                    else
                        sbrOrganisationNameDetailsCollection = null;

                    #region sbrOrganisationNameDetailsCollection

                    if (sbrOrganisationNameDetailsCollection != null)
                    {
                        report.OrganisationNameDetailsCollectionExists = true;
                        report.OrganisationNameDetailsCollectionCount = sbrOrganisationNameDetailsCollection.Count();

                        IEnumerable<SBRElement> sbrOrganisationNameDetailsCollection_MN_C_RPCollection;
                        sbrOrganisationNameDetailsCollection_MN_C_RPCollection = sbrOrganisationNameDetailsCollection.Where(
                            ce => ce.ChildElements != null &&
                            ce.ChildElements.Any(te => te.Name == "OrganisationNameDetails.OrganisationalNameType.Code" && te.Value == "MN") &&
                            ce.ChildElements.Any(te => te.Name == "OrganisationNameDetails.Currency.Code" && te.Value == "C" &&
                            sbrRPContexts.Contains(te.Context)));
                        if (sbrOrganisationNameDetailsCollection_MN_C_RPCollection != null && sbrOrganisationNameDetailsCollection_MN_C_RPCollection.Count() > 0)
                        {
                            SBRElement sbrOrganisationNameDetailsCollection_MN_C_RP;
                            sbrOrganisationNameDetailsCollection_MN_C_RP = sbrOrganisationNameDetailsCollection_MN_C_RPCollection.ElementAt(0);

                            IDictionary<string, SBRElement> sbrOrganisationNameDetailsCollection_MN_C_RPTupleElementMap;
                            sbrOrganisationNameDetailsCollection_MN_C_RPTupleElementMap = sbrOrganisationNameDetailsCollection_MN_C_RP.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();

                            report.OrganisationNameDetailsCollection_MN_C_RPExists = true;
                            report.OrganisationNameDetailsCollection_MN_C_RPCount = sbrOrganisationNameDetailsCollection_MN_C_RPCollection.Count();
                            report.OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex = sbrOrganisationNameDetailsCollection_MN_C_RP.OccurrenceIndex + 1;
                            report.OrganisationNameDetailsCollection_MN_C_RPLastOccurrenceIndex = sbrOrganisationNameDetailsCollection_MN_C_RPCollection.Last().OccurrenceIndex + 1;

                            if (sbrOrganisationNameDetailsCollection_MN_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}OrganisationNameDetails.OrganisationalNameType.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR234 = currentValue.Value;
                            }

                            if (sbrOrganisationNameDetailsCollection_MN_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}OrganisationNameDetails.Currency.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR319 = currentValue.Value;
                            }

                            if (sbrOrganisationNameDetailsCollection_MN_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}OrganisationNameDetails.OrganisationalName.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR4 = currentValue.Value;
                            }
                        }
                    }
                    #endregion End of Tuple sbrOrganisationNameDetailsCollection
                    if (validationMode) ValidateRPOrganisationNameDetailsCollection(errors, sbrOrganisationNameDetailsCollection, sbrRPContexts, report);


                    IEnumerable<SBRElement> sbrAddressDetailsCollection;
                    if (populatedReportElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}AddressDetails", out currentValue))
                        sbrAddressDetailsCollection = currentValue.Occurrences;
                    else
                        sbrAddressDetailsCollection = null;

                    #region sbrAddressDetailsCollection

                    if (sbrAddressDetailsCollection != null)
                    {
                        report.AddressDetailsCollectionExists = true;
                        report.AddressDetailsCollectionCount = sbrAddressDetailsCollection.Count();

                        IEnumerable<SBRElement> sbrAddressDetailsCollection_POS_C_RPCollection;
                        sbrAddressDetailsCollection_POS_C_RPCollection = sbrAddressDetailsCollection.Where(
                            ce => ce.ChildElements != null &&
                            ce.ChildElements.Any(te => te.Name == "AddressDetails.Usage.Code" && te.Value == "POS") &&
                            ce.ChildElements.Any(te => te.Name == "AddressDetails.Currency.Code" && te.Value == "C" &&
                            sbrRPContexts.Contains(te.Context)));
                        if (sbrAddressDetailsCollection_POS_C_RPCollection != null && sbrAddressDetailsCollection_POS_C_RPCollection.Count() > 0)
                        {
                            SBRElement sbrAddressDetailsCollection_POS_C_RP;
                            sbrAddressDetailsCollection_POS_C_RP = sbrAddressDetailsCollection_POS_C_RPCollection.ElementAt(0);

                            IDictionary<string, SBRElement> sbrAddressDetailsCollection_POS_C_RPTupleElementMap;
                            sbrAddressDetailsCollection_POS_C_RPTupleElementMap = sbrAddressDetailsCollection_POS_C_RP.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();

                            report.AddressDetailsCollection_POS_C_RPExists = true;
                            report.AddressDetailsCollection_POS_C_RPCount = sbrAddressDetailsCollection_POS_C_RPCollection.Count();
                            report.AddressDetailsCollection_POS_C_RPOccurrenceIndex = sbrAddressDetailsCollection_POS_C_RP.OccurrenceIndex + 1;
                            report.AddressDetailsCollection_POS_C_RPLastOccurrenceIndex = sbrAddressDetailsCollection_POS_C_RPCollection.Last().OccurrenceIndex + 1;

                            if (sbrAddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}AddressDetails.OverseasAddress.Indicator", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR281 = ToBoolean(currentValue.Value);
                            }

                            if (sbrAddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}AddressDetails.Usage.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR237 = currentValue.Value;
                            }

                            if (sbrAddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}AddressDetails.Currency.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR282 = currentValue.Value;
                            }

                            if (sbrAddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}AddressDetails.Line1.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR6 = currentValue.Value;
                            }

                            if (sbrAddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}AddressDetails.Line2.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR239 = currentValue.Value;
                            }

                            if (sbrAddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}AddressDetails.Line3.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR241 = currentValue.Value;
                            }

                            if (sbrAddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}AddressDetails.Line4.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR243 = currentValue.Value;
                            }

                            if (sbrAddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}AddressDetails.LocalityName.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR7 = currentValue.Value;
                            }

                            if (sbrAddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}AddressDetails.Postcode.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR9 = currentValue.Value;
                            }

                            if (sbrAddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}AddressDetails.StateOrTerritory.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR8 = currentValue.Value;
                            }

                            if (sbrAddressDetailsCollection_POS_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}AddressDetails.Country.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR245 = currentValue.Value;
                            }
                        }
                    }
                    #endregion End of Tuple sbrAddressDetailsCollection
                    if (validationMode) ValidateRPAddressDetailsCollection(errors, sbrAddressDetailsCollection, sbrRPContexts, report);


                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Report.Amendment.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR186 = ToBoolean(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Report.AmendmentReason.Text", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR290 = currentValues[0].Value;
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Report.AmendmentSequence.Number", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR291 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Report.AmendmentType.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR289 = currentValues[0].Value;
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Lodgment.FirstReturn.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR360 = ToBoolean(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}AuditorReport.Signature.Date", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR23 = DateTime.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}AuditorReport.PartAQualified.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR405 = ToBoolean(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}AuditorReport.OpinionQualified.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR24 = ToBoolean(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}AuditorReport.QualificationsRectified.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR320 = ToBoolean(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}ElectronicContact.ElectronicServiceAddressAlias.Text", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR369 = currentValues[0].Value;
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}FinancialInstitutionAccount.RefundPreference.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR406 = ToBoolean(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}SuperannuationFundDetails.AustralianSuperannuationFundStatus.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR25 = ToBoolean(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}SuperannuationFundDetails.FundBenefitStructure.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR26 = currentValues[0].Value;
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}SuperannuationFundDetails.FundAcceptGovernmentContributions.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR27 = ToBoolean(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}OrganisationDetails.OperationCease.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR28 = ToBoolean(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}OrganisationDetails.OrganisationActivityStatusEnd.Date", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR29 = DateTime.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}SuperannuationFundDetails.TaxObligationsMet.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR30 = ToBoolean(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}SuperannuationFundDetails.IncomeStreamPaid.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR321 = ToBoolean(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}TaxConcession.PensionIncomeExemptDeduction.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR301 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}TaxConcession.PensionIncomeExemptCalculationMethod.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR323 = currentValues[0].Value;
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}SuperannuationFundDetails.ActuarialCertificate.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR324 = ToBoolean(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Income.AssessableOther.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR352 = ToBoolean(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}CapitalGainsTax.Event.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR31 = ToBoolean(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}CapitalGainsTax.ExemptionOrRolloverApplied.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR327 = ToBoolean(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}CapitalGainsTax.ExemptionOrRolloverApplied.Code", out currentValues))
                    {
                        report.SMSFAR328Collection = new List<string>();
                        try
                        {
                            foreach (SBRElement occurrence in currentValues)
                            {
                                if (!occurrence.IsNil)
                                    report.SMSFAR328Collection.Add(occurrence.Value);
                            }
                        }
                        catch (Exception)
                        {
                            // The above code is known to throw exceptions in some cases but not others.
                            // We don't really know why but think that catching them like this will result in correct behaviour
                        }
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Income.CapitalGainsNet.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR33 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Income.RentLeasingHiringGross.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR34 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Income.Interest.Gross.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR35 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Income.ForestryManagedInvestmentScheme.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR36 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR39 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Equity.InternationalDealings.ForeignFundTransfers.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR40 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}InternationalDealings.ForeignFundTransfers.Count", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR41 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Remuneration.ABNNotQuotedPaymentGross.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR42 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Income.PartnershipDistributionGross.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR43 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Income.DividendsUnfranked.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR44 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Income.DividendsFranked.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR45 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}IncomeTax.FrankingCredits.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR46 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Income.TrustDistributionGross.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR47 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Party.TrustType.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR48 = currentValues[0].Value;
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Income.Other.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR53 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Income.Other.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR54 = currentValues[0].Value;
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Income.AssessableIncomeDueToFundTaxStatusChange.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR55 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}SuperannuationContribution.EmployerAssessable.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR49 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}SuperannuationContribution.PersonalAssessable.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR50 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}SuperannuationContribution.EmployerContributionsTaxFileNumberNotQuoted.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR51 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}SuperannuationContribution.LiabilityTransferToLifeInsuranceCompanyOrPooledSuperannuationTrustAssessable.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR52 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Income.NonArmsLengthPrivateCompanyDividendsNet.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR56 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Income.NonArmsLengthTrustDistributionsNet.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR57 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Income.NonArmsLengthOtherNet.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR58 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Income.Taxable.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR73 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Expense.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR338 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}IncomeTax.TotalTaxOnTaxableIncome.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR315 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}IncomeTax.TFNNotQuotedContributionsTax.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR316 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}IncomeTax.TaxableGross.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR74 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Income.InternationalDealings.TaxOffset.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR75 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}IncomeTax.TaxOffsetsAndRebatesTotal.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR76 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}IncomeTax.TaxOffsetNonRefundableNonCarryForward.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR305 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}IncomeTax.SubtotalTaxPayable.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR304 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}IncomeTax.TaxOffsetNonRefundableCarryForward.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR389 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}IncomeTax.TaxOffsetCarriedForwardSubTotal.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR390 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}IncomeTax.FrankingCredits.Refundable.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR81 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}IncomeTax.TaxOffsetTFNNotQuoted.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR225 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}IncomeTax.TaxOffsetEntitlementForNationalRentalAffordabilityScheme.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR227 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}IncomeTax.TaxOffsetExplorationCredits.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR370 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}IncomeTax.TotalRefundableTaxOffsets.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR306 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}IncomeTax.Payable.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR307 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}InternationalDealings.AAMInterestSection102.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR77 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromForeignResidents.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR79 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldTFNNotQuotedAndABNNotQuoted.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR80 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldFromCloselyHeldTrust.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR298 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}IncomeTax.TaxOffsetCalculatedInterestTFNNotQuoted.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR226 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}IncomeTax.PayAsYouGoWithholding.CreditForCapitalGainsWithheldFromForeignResidents.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR388 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}IncomeTax.Deduction.CreditsEligibleTotal.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR308 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}IncomeTax.TotalRemainderOfRefundableTaxOffsets.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR309 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}IncomeTax.LiabilityInstalmentsTotal.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR82 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}IncomeTax.SupervisoryLevy.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR83 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}IncomeTax.SupervisoryLevyWoundupAdjustment.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR361 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}IncomeTax.SupervisoryLevyNewlyRegisteredAdjustment.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR362 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}IncomeTax.PayableOrRefundableTotal.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR84 = decimal.Parse(currentValues[0].Value);
                    }

                    IEnumerable<SBRElement> sbrMemberInformationCollection;
                    if (populatedReportElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}MemberInformation", out currentValue))
                        sbrMemberInformationCollection = currentValue.Occurrences;
                    else
                        sbrMemberInformationCollection = null;

                    #region sbrMemberInformationCollection

                    if (sbrMemberInformationCollection != null)
                    {
                        report.MemberInformationCollectionExists = true;
                        report.MemberInformationCollectionCount = sbrMemberInformationCollection.Count();
                        if (sbrMemberInformationCollection != null && sbrMemberInformationCollection.Count() > 0)
                        {
                            report.MemberInformationCollection = new List<SMSFAR2025.MemberInformation>();
                            for (int tupleIndex = 0; tupleIndex < sbrMemberInformationCollection.Count(); tupleIndex++)
                            {
                                SBRElement sbrMemberInformation;
                                sbrMemberInformation = sbrMemberInformationCollection.ElementAt(tupleIndex);

                                IDictionary<string, SBRElement> sbrMemberInformationTupleElementMap;
                                sbrMemberInformationTupleElementMap = sbrMemberInformation.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();

                                SMSFAR2025.MemberInformation memberInformation = new SMSFAR2025.MemberInformation();
                                report.MemberInformationCollection.Add(memberInformation);
                                memberInformation.OccurrenceIndex = tupleIndex + 1;

                                if (sbrMemberInformationTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Report.ItemOrder.Number", out currentValue))
                                {
                                    if (!currentValue.IsNil) memberInformation.SMSFAR195 = currentValue.Value;
                                }

                                IEnumerable<SBRElement> sbrMemberInformation_PersonNameDetailsCollection;
                                if (sbrMemberInformationTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails", out currentValue))
                                    sbrMemberInformation_PersonNameDetailsCollection = currentValue.Occurrences;
                                else
                                    sbrMemberInformation_PersonNameDetailsCollection = null;

                                #region sbrMemberInformation_PersonNameDetailsCollection

                                if (sbrMemberInformation_PersonNameDetailsCollection != null)
                                {
                                    memberInformation.MemberInformation_PersonNameDetailsCollectionExists = true;
                                    memberInformation.MemberInformation_PersonNameDetailsCollectionCount = sbrMemberInformation_PersonNameDetailsCollection.Count();

                                    IEnumerable<SBRElement> sbrMemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RPCollection;
                                    sbrMemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RPCollection = sbrMemberInformation_PersonNameDetailsCollection.Where(
                                        ce => ce.ChildElements != null &&
                                        ce.ChildElements.Any(te => te.Name == "PersonNameDetails.PersonNameType.Code" && te.Value == "LGL") &&
                                        ce.ChildElements.Any(te => te.Name == "PersonNameDetails.Usage.Code" && te.Value == "Identifier") &&
                                        ce.ChildElements.Any(te => te.Name == "PersonNameDetails.Currency.Code" && te.Value == "C" &&
                                        sbrRPContexts.Contains(te.Context)));
                                    if (sbrMemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RPCollection != null && sbrMemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RPCollection.Count() > 0)
                                    {
                                        SBRElement sbrMemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RP;
                                        sbrMemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RP = sbrMemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RPCollection.ElementAt(0);

                                        IDictionary<string, SBRElement> sbrMemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RPTupleElementMap;
                                        sbrMemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RPTupleElementMap = sbrMemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RP.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();

                                        memberInformation.MemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RPExists = true;
                                        memberInformation.MemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RPCount = sbrMemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RPCollection.Count();
                                        memberInformation.MemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RPOccurrenceIndex = sbrMemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RP.OccurrenceIndex + 1;
                                        memberInformation.MemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RPLastOccurrenceIndex = sbrMemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RPCollection.Last().OccurrenceIndex + 1;

                                        if (sbrMemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.PersonNameType.Code", out currentValue))
                                        {
                                            if (!currentValue.IsNil) memberInformation.SMSFAR251 = currentValue.Value;
                                        }

                                        if (sbrMemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.Usage.Code", out currentValue))
                                        {
                                            if (!currentValue.IsNil) memberInformation.SMSFAR255 = currentValue.Value;
                                        }

                                        if (sbrMemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.Currency.Code", out currentValue))
                                        {
                                            if (!currentValue.IsNil) memberInformation.SMSFAR259 = currentValue.Value;
                                        }

                                        if (sbrMemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.Title.Text", out currentValue))
                                        {
                                            if (!currentValue.IsNil) memberInformation.SMSFAR196 = currentValue.Value;
                                        }

                                        if (sbrMemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.NameSuffix.Text", out currentValue))
                                        {
                                            if (!currentValue.IsNil) memberInformation.SMSFAR198 = currentValue.Value;
                                        }

                                        if (sbrMemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.FamilyName.Text", out currentValue))
                                        {
                                            if (!currentValue.IsNil) memberInformation.SMSFAR197 = currentValue.Value;
                                        }

                                        if (sbrMemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.GivenName.Text", out currentValue))
                                        {
                                            if (!currentValue.IsNil) memberInformation.SMSFAR199 = currentValue.Value;
                                        }

                                        if (sbrMemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.OtherGivenName.Text", out currentValue))
                                        {
                                            if (!currentValue.IsNil) memberInformation.SMSFAR200 = currentValue.Value;
                                        }
                                    }
                                }
                                #endregion End of Tuple sbrMemberInformation_PersonNameDetailsCollection
                                if (validationMode) ValidateRPMemberInformation_PersonNameDetailsCollection(errors, sbrMemberInformation_PersonNameDetailsCollection, sbrRPContexts, report);


                                if (sbrMemberInformationTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Identifiers.TaxFileNumber.Identifier", out currentValue))
                                {
                                    if (!currentValue.IsNil) memberInformation.SMSFAR201 = currentValue.Value;
                                }

                                if (sbrMemberInformationTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonDemographicDetails.Birth.Date", out currentValue))
                                {
                                    if (!currentValue.IsNil) memberInformation.SMSFAR202 = DateTime.Parse(currentValue.Value);
                                }

                                if (sbrMemberInformationTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonDemographicDetails.Death.Date", out currentValue))
                                {
                                    if (!currentValue.IsNil) memberInformation.SMSFAR95 = DateTime.Parse(currentValue.Value);
                                }

                                if (sbrMemberInformationTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}SuperannuationFundDetails.MemberAccountStatus.Code", out currentValue))
                                {
                                    if (!currentValue.IsNil) memberInformation.SMSFAR354 = currentValue.Value;
                                }

                                if (sbrMemberInformationTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}SuperannuationContribution.EmployerContributions.Amount", out currentValue))
                                {
                                    if (!currentValue.IsNil) memberInformation.SMSFAR203 = decimal.Parse(currentValue.Value);
                                }

                                if (sbrMemberInformationTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Identifiers.AustralianBusinessNumber.Identifier", out currentValue))
                                {
                                    if (!currentValue.IsNil) memberInformation.SMSFAR204 = currentValue.Value;
                                }

                                if (sbrMemberInformationTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}SuperannuationContribution.PersonalContributions.Amount", out currentValue))
                                {
                                    if (!currentValue.IsNil) memberInformation.SMSFAR205 = decimal.Parse(currentValue.Value);
                                }

                                if (sbrMemberInformationTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}SuperannuationContribution.CapitalGainsTaxSmallBusinessRetirementExemption.Amount", out currentValue))
                                {
                                    if (!currentValue.IsNil) memberInformation.SMSFAR206 = decimal.Parse(currentValue.Value);
                                }

                                if (sbrMemberInformationTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}SuperannuationContribution.CapitalGainsTaxSmallBusinessExemption.Amount", out currentValue))
                                {
                                    if (!currentValue.IsNil) memberInformation.SMSFAR207 = decimal.Parse(currentValue.Value);
                                }

                                if (sbrMemberInformationTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}SuperannuationContribution.PersonalInjuryElection.Amount", out currentValue))
                                {
                                    if (!currentValue.IsNil) memberInformation.SMSFAR208 = decimal.Parse(currentValue.Value);
                                }

                                if (sbrMemberInformationTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}SuperannuationContribution.SpouseAndChildContributions.Amount", out currentValue))
                                {
                                    if (!currentValue.IsNil) memberInformation.SMSFAR209 = decimal.Parse(currentValue.Value);
                                }

                                if (sbrMemberInformationTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}SuperannuationContribution.OtherThirdPartyContributions.Amount", out currentValue))
                                {
                                    if (!currentValue.IsNil) memberInformation.SMSFAR210 = decimal.Parse(currentValue.Value);
                                }

                                if (sbrMemberInformationTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}SuperannuationContribution.ProceedsFromPrimaryResidenceDisposal.Amount", out currentValue))
                                {
                                    if (!currentValue.IsNil) memberInformation.SMSFAR400 = decimal.Parse(currentValue.Value);
                                }

                                if (sbrMemberInformationTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}SuperannuationContribution.Receipt.Date", out currentValue))
                                {
                                    if (!currentValue.IsNil) memberInformation.SMSFAR401 = DateTime.Parse(currentValue.Value);
                                }

                                if (sbrMemberInformationTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}SuperannuationContribution.ForeignSuperannuationFundAssessable.Amount", out currentValue))
                                {
                                    if (!currentValue.IsNil) memberInformation.SMSFAR212 = decimal.Parse(currentValue.Value);
                                }

                                if (sbrMemberInformationTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}SuperannuationContribution.ForeignSuperannuationFundNonAssessable.Amount", out currentValue))
                                {
                                    if (!currentValue.IsNil) memberInformation.SMSFAR213 = decimal.Parse(currentValue.Value);
                                }

                                if (sbrMemberInformationTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}SuperannuationContribution.TransferFromReserveAssessable.Amount", out currentValue))
                                {
                                    if (!currentValue.IsNil) memberInformation.SMSFAR214 = decimal.Parse(currentValue.Value);
                                }

                                if (sbrMemberInformationTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}SuperannuationContribution.TransferFromReserveNonAssessable.Amount", out currentValue))
                                {
                                    if (!currentValue.IsNil) memberInformation.SMSFAR215 = decimal.Parse(currentValue.Value);
                                }

                                if (sbrMemberInformationTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}SuperannuationContribution.NonComplyingFundsAndPreviouslyNonComplyingFunds.Amount", out currentValue))
                                {
                                    if (!currentValue.IsNil) memberInformation.SMSFAR310 = decimal.Parse(currentValue.Value);
                                }

                                if (sbrMemberInformationTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}SuperannuationContribution.TotalOther.Amount", out currentValue))
                                {
                                    if (!currentValue.IsNil) memberInformation.SMSFAR216 = decimal.Parse(currentValue.Value);
                                }

                                if (sbrMemberInformationTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}SuperannuationContribution.MemberTotal.Amount", out currentValue))
                                {
                                    if (!currentValue.IsNil) memberInformation.SMSFAR217 = decimal.Parse(currentValue.Value);
                                }

                                if (sbrMemberInformationTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}SuperannuationContribution.MemberAllocatedEarningsOrLosses.Amount", out currentValue))
                                {
                                    if (!currentValue.IsNil) memberInformation.SMSFAR218 = decimal.Parse(currentValue.Value);
                                }

                                if (sbrMemberInformationTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}SuperannuationRollover.Received.Amount", out currentValue))
                                {
                                    if (!currentValue.IsNil) memberInformation.SMSFAR219 = decimal.Parse(currentValue.Value);
                                }

                                if (sbrMemberInformationTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}SuperannuationRollover.Paid.Amount", out currentValue))
                                {
                                    if (!currentValue.IsNil) memberInformation.SMSFAR220 = decimal.Parse(currentValue.Value);
                                }

                                if (sbrMemberInformationTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}SuperannuationPayment.BenefitLumpSum.Amount", out currentValue))
                                {
                                    if (!currentValue.IsNil) memberInformation.SMSFAR340 = decimal.Parse(currentValue.Value);
                                }

                                if (sbrMemberInformationTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}SuperannuationPayment.BenefitLumpSum.Code", out currentValue))
                                {
                                    if (!currentValue.IsNil) memberInformation.SMSFAR341 = currentValue.Value;
                                }

                                if (sbrMemberInformationTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}SuperannuationPayment.BenefitPensions.Amount", out currentValue))
                                {
                                    if (!currentValue.IsNil) memberInformation.SMSFAR342 = decimal.Parse(currentValue.Value);
                                }

                                if (sbrMemberInformationTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}SuperannuationPayment.BenefitPensions.Code", out currentValue))
                                {
                                    if (!currentValue.IsNil) memberInformation.SMSFAR343 = currentValue.Value;
                                }

                                if (sbrMemberInformationTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}SuperannuationBenefit.AccountBalance.Amount", out currentValue))
                                {
                                    foreach (var sbrElementItem in currentValue.Occurrences)
                                    {
                                        if (!sbrElementItem.IsNil)
                                        {
                                            if (sbrElementItem.Context.Segment.DimensionDomains.Count == 3)
                                            {
                                                if (sbrElementItem.Context.Segment.DimensionDomains.ToList().FindIndex(p => p.Domain.EndsWith("}Closing") || p.Domain.EndsWith(":Closing")) > -1 && sbrElementItem.Context.Segment.DimensionDomains.ToList().FindIndex(p => p.Domain.EndsWith("}Accumulation") || p.Domain.EndsWith(":Accumulation")) > -1)
                                                {
                                                    memberInformation.SMSFAR393 = decimal.Parse(sbrElementItem.Value);
                                                }
                                            }
                                        }
                                    }
                                }

                                if (sbrMemberInformationTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}SuperannuationBenefit.NonCappedDefinedBenefitIncomeStream.Amount", out currentValue))
                                {
                                    foreach (var sbrElementItem in currentValue.Occurrences)
                                    {
                                        if (!sbrElementItem.IsNil)
                                        {
                                            if (sbrElementItem.Context.Segment.DimensionDomains.Count == 3)
                                            {
                                                if (sbrElementItem.Context.Segment.DimensionDomains.ToList().FindIndex(p => p.Domain.EndsWith("}Closing") || p.Domain.EndsWith(":Closing")) > -1 && sbrElementItem.Context.Segment.DimensionDomains.ToList().FindIndex(p => p.Domain.EndsWith("}Retirement") || p.Domain.EndsWith(":Retirement")) > -1)
                                                {
                                                    memberInformation.SMSFAR394 = decimal.Parse(sbrElementItem.Value);
                                                }
                                            }
                                        }
                                    }
                                }

                                if (sbrMemberInformationTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}SuperannuationBenefit.CappedDefinedBenefitIncomeStream.Amount", out currentValue))
                                {
                                    foreach (var sbrElementItem in currentValue.Occurrences)
                                    {
                                        if (!sbrElementItem.IsNil)
                                        {
                                            if (sbrElementItem.Context.Segment.DimensionDomains.Count == 3)
                                            {
                                                if (sbrElementItem.Context.Segment.DimensionDomains.ToList().FindIndex(p => p.Domain.EndsWith("}Closing") || p.Domain.EndsWith(":Closing")) > -1 && sbrElementItem.Context.Segment.DimensionDomains.ToList().FindIndex(p => p.Domain.EndsWith("}Retirement") || p.Domain.EndsWith(":Retirement")) > -1)
                                                {
                                                    memberInformation.SMSFAR395 = decimal.Parse(sbrElementItem.Value);
                                                }
                                            }
                                        }
                                    }
                                }

                                if (sbrMemberInformationTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}SuperannuationBenefit.AccountsTransitioningToRetirementIncomeStream.Count", out currentValue))
                                {
                                    if (!currentValue.IsNil) memberInformation.SMSFAR396 = decimal.Parse(currentValue.Value);
                                }

                                if (sbrMemberInformationTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}SuperannuationBenefit.AccountBalance.Amount", out currentValue))
                                {
                                    foreach (var sbrElementItem in currentValue.Occurrences)
                                    {
                                        if (!sbrElementItem.IsNil)
                                        {
                                            if (sbrElementItem.Context.Segment.DimensionDomains.Count == 2)
                                            {
                                                if (sbrElementItem.Context.Segment.DimensionDomains.ToList().FindIndex(p => p.Domain.EndsWith("}Closing") || p.Domain.EndsWith(":Closing")) > -1)
                                                {
                                                    memberInformation.SMSFAR223 = decimal.Parse(sbrElementItem.Value);
                                                }
                                            }
                                        }
                                    }
                                }

                                if (sbrMemberInformationTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}SuperannuationBenefit.AccountBalance.Amount", out currentValue))
                                {
                                    foreach (var sbrElementItem in currentValue.Occurrences)
                                    {
                                        if (!sbrElementItem.IsNil)
                                        {
                                            if (sbrElementItem.Context.Segment.DimensionDomains.Count == 2)
                                            {
                                                if (sbrElementItem.Context.Segment.DimensionDomains.ToList().FindIndex(p => p.Domain.EndsWith("}Accumulation") || p.Domain.EndsWith(":Accumulation")) > -1)
                                                {
                                                    memberInformation.SMSFAR397 = decimal.Parse(sbrElementItem.Value);
                                                }
                                            }
                                        }
                                    }
                                }

                                if (sbrMemberInformationTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}SuperannuationBenefit.AccountBalance.Amount", out currentValue))
                                {
                                    foreach (var sbrElementItem in currentValue.Occurrences)
                                    {
                                        if (!sbrElementItem.IsNil)
                                        {
                                            if (sbrElementItem.Context.Segment.DimensionDomains.Count == 2)
                                            {
                                                if (sbrElementItem.Context.Segment.DimensionDomains.ToList().FindIndex(p => p.Domain.EndsWith("}Retirement") || p.Domain.EndsWith(":Retirement")) > -1)
                                                {
                                                    memberInformation.SMSFAR398 = decimal.Parse(sbrElementItem.Value);
                                                }
                                            }
                                        }
                                    }
                                }

                                if (sbrMemberInformationTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Assets.OutstandingLimitedRecourseBorrowing.Amount", out currentValue))
                                {
                                    if (!currentValue.IsNil) memberInformation.SMSFAR402 = decimal.Parse(currentValue.Value);
                                }
                            }
                            #endregion End of Tuple sbrMemberInformationCollection
                        }
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Assets.BorrowingLicensedFinancialInstitution.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR384 = ToBoolean(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Assets.PersonalGuaranteesOtherSecurity.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR385 = ToBoolean(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Assets.Investment.CryptoCurrency.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR399 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Liabilities.BorrowingsLimited.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR381 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Liabilities.BorrowingsTemporary.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR382 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Liabilities.Borrowings.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR136 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Elections.FamilyTrustElectionStatus.Year", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR162 = int.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Elections.FamilyTrustElectionRevocation.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR163 = currentValues[0].Value;
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Elections.InterposedEntityElectionStatus.Year", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR164 = int.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Elections.InterposedEntityElectionRevocation.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR165 = currentValues[0].Value;
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}RegulatoryDisclosures.GeneralInformationAboutFinancialStatements.Text", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR355 = currentValues[0].Value;
                    }

                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Report.CompletionHours.Number", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR176 = decimal.Parse(currentValues[0].Value);
                    }

                    IEnumerable<SBRElement> sbrDeclarationCollection;
                    if (populatedReportElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Declaration", out currentValue))
                        sbrDeclarationCollection = currentValue.Occurrences;
                    else
                        sbrDeclarationCollection = null;

                    #region sbrDeclarationCollection

                    if (sbrDeclarationCollection != null)
                    {
                        report.DeclarationCollectionExists = true;
                        report.DeclarationCollectionCount = sbrDeclarationCollection.Count();

                        IEnumerable<SBRElement> sbrDeclarationCollection_TrueAndCorrect_RPCollection;
                        sbrDeclarationCollection_TrueAndCorrect_RPCollection = sbrDeclarationCollection.Where(
                            ce => ce.ChildElements != null &&
                            ce.ChildElements.Any(te => te.Name == "Declaration.StatementType.Code" && te.Value == "TrueAndCorrect" &&
                            sbrRPContexts.Contains(te.Context)));
                        if (sbrDeclarationCollection_TrueAndCorrect_RPCollection != null && sbrDeclarationCollection_TrueAndCorrect_RPCollection.Count() > 0)
                        {
                            SBRElement sbrDeclarationCollection_TrueAndCorrect_RP;
                            sbrDeclarationCollection_TrueAndCorrect_RP = sbrDeclarationCollection_TrueAndCorrect_RPCollection.ElementAt(0);

                            IDictionary<string, SBRElement> sbrDeclarationCollection_TrueAndCorrect_RPTupleElementMap;
                            sbrDeclarationCollection_TrueAndCorrect_RPTupleElementMap = sbrDeclarationCollection_TrueAndCorrect_RP.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();

                            report.DeclarationCollection_TrueAndCorrect_RPExists = true;
                            report.DeclarationCollection_TrueAndCorrect_RPCount = sbrDeclarationCollection_TrueAndCorrect_RPCollection.Count();
                            report.DeclarationCollection_TrueAndCorrect_RPOccurrenceIndex = sbrDeclarationCollection_TrueAndCorrect_RP.OccurrenceIndex + 1;
                            report.DeclarationCollection_TrueAndCorrect_RPLastOccurrenceIndex = sbrDeclarationCollection_TrueAndCorrect_RPCollection.Last().OccurrenceIndex + 1;

                            if (sbrDeclarationCollection_TrueAndCorrect_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Declaration.StatementType.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR267 = currentValue.Value;
                            }

                            if (sbrDeclarationCollection_TrueAndCorrect_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Declaration.StatementAccepted.Indicator", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR268 = ToBoolean(currentValue.Value);
                            }

                            if (sbrDeclarationCollection_TrueAndCorrect_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Declaration.Statement.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR191 = currentValue.Value;
                            }

                            if (sbrDeclarationCollection_TrueAndCorrect_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Declaration.Signature.Date", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR192 = DateTime.Parse(currentValue.Value);
                            }

                            if (sbrDeclarationCollection_TrueAndCorrect_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Declaration.SignatoryIdentifier.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR269 = currentValue.Value;
                            }

                            IEnumerable<SBRElement> sbrDeclaration_PersonUnstructuredNameCollection;
                            if (sbrDeclarationCollection_TrueAndCorrect_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonUnstructuredName", out currentValue))
                                sbrDeclaration_PersonUnstructuredNameCollection = currentValue.Occurrences;
                            else
                                sbrDeclaration_PersonUnstructuredNameCollection = null;

                            #region sbrDeclaration_PersonUnstructuredNameCollection

                            if (sbrDeclaration_PersonUnstructuredNameCollection != null)
                            {
                                report.Declaration_PersonUnstructuredNameCollectionExists = true;
                                report.Declaration_PersonUnstructuredNameCollectionCount = sbrDeclaration_PersonUnstructuredNameCollection.Count();

                                IEnumerable<SBRElement> sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPCollection;
                                sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPCollection = sbrDeclaration_PersonUnstructuredNameCollection.Where(
                                    ce => ce.ChildElements != null &&
                                    ce.ChildElements.Any(te => te.Name == "PersonUnstructuredName.Usage.Code" && te.Value == "DeclarationSignatory" &&
                                    sbrRPContexts.Contains(te.Context)));
                                if (sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPCollection != null && sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPCollection.Count() > 0)
                                {
                                    SBRElement sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_RP;
                                    sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_RP = sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPCollection.ElementAt(0);

                                    IDictionary<string, SBRElement> sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPTupleElementMap;
                                    sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPTupleElementMap = sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_RP.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();

                                    report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPExists = true;
                                    report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPCount = sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPCollection.Count();
                                    report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPOccurrenceIndex = sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_RP.OccurrenceIndex + 1;
                                    report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPLastOccurrenceIndex = sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPCollection.Last().OccurrenceIndex + 1;

                                    if (sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonUnstructuredName.Usage.Code", out currentValue))
                                    {
                                        if (!currentValue.IsNil) report.SMSFAR277 = currentValue.Value;
                                    }

                                    if (sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonUnstructuredName.FullName.Text", out currentValue))
                                    {
                                        if (!currentValue.IsNil) report.SMSFAR278 = currentValue.Value;
                                    }
                                }
                            }
                            #endregion End of Tuple sbrDeclaration_PersonUnstructuredNameCollection
                            if (validationMode) ValidateRPDeclaration_PersonUnstructuredNameCollection(errors, sbrDeclaration_PersonUnstructuredNameCollection, sbrRPContexts, report);

                        }
                    }
                    #endregion End of Tuple sbrDeclarationCollection
                    if (validationMode) ValidateRPDeclarationCollection(errors, sbrDeclarationCollection, sbrRPContexts, report);


                    IEnumerable<SBRElement> sbrTrusteeOrDirectorCollection;
                    if (populatedReportElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}TrusteeOrDirector", out currentValue))
                        sbrTrusteeOrDirectorCollection = currentValue.Occurrences;
                    else
                        sbrTrusteeOrDirectorCollection = null;

                    #region sbrTrusteeOrDirectorCollection

                    if (sbrTrusteeOrDirectorCollection != null)
                    {
                        report.TrusteeOrDirectorCollectionExists = true;
                        report.TrusteeOrDirectorCollectionCount = sbrTrusteeOrDirectorCollection.Count();
                        SBRElement sbrTrusteeOrDirector;
                        sbrTrusteeOrDirector = sbrTrusteeOrDirectorCollection.ElementAt(0);

                        IDictionary<string, SBRElement> sbrTrusteeOrDirectorTupleElementMap;
                        sbrTrusteeOrDirectorTupleElementMap = sbrTrusteeOrDirector.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();

                        IEnumerable<SBRElement> sbrTrusteeOrDirector_PersonNameDetailsCollection;
                        if (sbrTrusteeOrDirectorTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails", out currentValue))
                            sbrTrusteeOrDirector_PersonNameDetailsCollection = currentValue.Occurrences;
                        else
                            sbrTrusteeOrDirector_PersonNameDetailsCollection = null;

                        #region sbrTrusteeOrDirector_PersonNameDetailsCollection

                        if (sbrTrusteeOrDirector_PersonNameDetailsCollection != null)
                        {
                            report.TrusteeOrDirector_PersonNameDetailsCollectionExists = true;
                            report.TrusteeOrDirector_PersonNameDetailsCollectionCount = sbrTrusteeOrDirector_PersonNameDetailsCollection.Count();

                            IEnumerable<SBRElement> sbrTrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RPCollection;
                            sbrTrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RPCollection = sbrTrusteeOrDirector_PersonNameDetailsCollection.Where(
                                ce => ce.ChildElements != null &&
                                ce.ChildElements.Any(te => te.Name == "PersonNameDetails.PersonNameType.Code" && te.Value == "LGL") &&
                                ce.ChildElements.Any(te => te.Name == "PersonNameDetails.Usage.Code" && te.Value == "Contact") &&
                                ce.ChildElements.Any(te => te.Name == "PersonNameDetails.Currency.Code" && te.Value == "C" &&
                                sbrRPContexts.Contains(te.Context)));
                            if (sbrTrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RPCollection != null && sbrTrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RPCollection.Count() > 0)
                            {
                                SBRElement sbrTrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RP;
                                sbrTrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RP = sbrTrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RPCollection.ElementAt(0);

                                IDictionary<string, SBRElement> sbrTrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RPTupleElementMap;
                                sbrTrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RPTupleElementMap = sbrTrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RP.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();

                                report.TrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RPExists = true;
                                report.TrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RPCount = sbrTrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RPCollection.Count();
                                report.TrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RPOccurrenceIndex = sbrTrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RP.OccurrenceIndex + 1;
                                report.TrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RPLastOccurrenceIndex = sbrTrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RPCollection.Last().OccurrenceIndex + 1;

                                if (sbrTrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.PersonNameType.Code", out currentValue))
                                {
                                    if (!currentValue.IsNil) report.SMSFAR252 = currentValue.Value;
                                }

                                if (sbrTrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.Usage.Code", out currentValue))
                                {
                                    if (!currentValue.IsNil) report.SMSFAR256 = currentValue.Value;
                                }

                                if (sbrTrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.Currency.Code", out currentValue))
                                {
                                    if (!currentValue.IsNil) report.SMSFAR260 = currentValue.Value;
                                }

                                if (sbrTrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.Title.Text", out currentValue))
                                {
                                    if (!currentValue.IsNil) report.SMSFAR166 = currentValue.Value;
                                }

                                if (sbrTrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.NameSuffix.Text", out currentValue))
                                {
                                    if (!currentValue.IsNil) report.SMSFAR168 = currentValue.Value;
                                }

                                if (sbrTrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.FamilyName.Text", out currentValue))
                                {
                                    if (!currentValue.IsNil) report.SMSFAR167 = currentValue.Value;
                                }

                                if (sbrTrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.GivenName.Text", out currentValue))
                                {
                                    if (!currentValue.IsNil) report.SMSFAR169 = currentValue.Value;
                                }

                                if (sbrTrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.OtherGivenName.Text", out currentValue))
                                {
                                    if (!currentValue.IsNil) report.SMSFAR170 = currentValue.Value;
                                }
                            }
                        }
                        #endregion End of Tuple sbrTrusteeOrDirector_PersonNameDetailsCollection
                        if (validationMode) ValidateRPTrusteeOrDirector_PersonNameDetailsCollection(errors, sbrTrusteeOrDirector_PersonNameDetailsCollection, sbrRPContexts, report);


                        IEnumerable<SBRElement> sbrTrusteeOrDirector_ElectronicContactTelephoneCollection;
                        if (sbrTrusteeOrDirectorTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}ElectronicContactTelephone", out currentValue))
                            sbrTrusteeOrDirector_ElectronicContactTelephoneCollection = currentValue.Occurrences;
                        else
                            sbrTrusteeOrDirector_ElectronicContactTelephoneCollection = null;

                        #region sbrTrusteeOrDirector_ElectronicContactTelephoneCollection

                        if (sbrTrusteeOrDirector_ElectronicContactTelephoneCollection != null)
                        {
                            report.TrusteeOrDirector_ElectronicContactTelephoneCollectionExists = true;
                            report.TrusteeOrDirector_ElectronicContactTelephoneCollectionCount = sbrTrusteeOrDirector_ElectronicContactTelephoneCollection.Count();

                            IEnumerable<SBRElement> sbrTrusteeOrDirector_ElectronicContactTelephoneCollection_03_RPCollection;
                            sbrTrusteeOrDirector_ElectronicContactTelephoneCollection_03_RPCollection = sbrTrusteeOrDirector_ElectronicContactTelephoneCollection.Where(
                                ce => ce.ChildElements != null &&
                                ce.ChildElements.Any(te => te.Name == "ElectronicContact.Telephone.Usage.Code" && te.Value == "03" &&
                                sbrRPContexts.Contains(te.Context)));
                            if (sbrTrusteeOrDirector_ElectronicContactTelephoneCollection_03_RPCollection != null && sbrTrusteeOrDirector_ElectronicContactTelephoneCollection_03_RPCollection.Count() > 0)
                            {
                                SBRElement sbrTrusteeOrDirector_ElectronicContactTelephoneCollection_03_RP;
                                sbrTrusteeOrDirector_ElectronicContactTelephoneCollection_03_RP = sbrTrusteeOrDirector_ElectronicContactTelephoneCollection_03_RPCollection.ElementAt(0);

                                IDictionary<string, SBRElement> sbrTrusteeOrDirector_ElectronicContactTelephoneCollection_03_RPTupleElementMap;
                                sbrTrusteeOrDirector_ElectronicContactTelephoneCollection_03_RPTupleElementMap = sbrTrusteeOrDirector_ElectronicContactTelephoneCollection_03_RP.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();

                                report.TrusteeOrDirector_ElectronicContactTelephoneCollection_03_RPExists = true;
                                report.TrusteeOrDirector_ElectronicContactTelephoneCollection_03_RPCount = sbrTrusteeOrDirector_ElectronicContactTelephoneCollection_03_RPCollection.Count();
                                report.TrusteeOrDirector_ElectronicContactTelephoneCollection_03_RPOccurrenceIndex = sbrTrusteeOrDirector_ElectronicContactTelephoneCollection_03_RP.OccurrenceIndex + 1;
                                report.TrusteeOrDirector_ElectronicContactTelephoneCollection_03_RPLastOccurrenceIndex = sbrTrusteeOrDirector_ElectronicContactTelephoneCollection_03_RPCollection.Last().OccurrenceIndex + 1;

                                if (sbrTrusteeOrDirector_ElectronicContactTelephoneCollection_03_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}ElectronicContact.Telephone.Usage.Code", out currentValue))
                                {
                                    if (!currentValue.IsNil) report.SMSFAR262 = currentValue.Value;
                                }

                                if (sbrTrusteeOrDirector_ElectronicContactTelephoneCollection_03_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}ElectronicContact.Telephone.ServiceLine.Code", out currentValue))
                                {
                                    if (!currentValue.IsNil) report.SMSFAR265 = currentValue.Value;
                                }

                                if (sbrTrusteeOrDirector_ElectronicContactTelephoneCollection_03_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}ElectronicContact.Telephone.Area.Code", out currentValue))
                                {
                                    if (!currentValue.IsNil) report.SMSFAR171 = currentValue.Value;
                                }

                                if (sbrTrusteeOrDirector_ElectronicContactTelephoneCollection_03_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}ElectronicContact.Telephone.Minimal.Number", out currentValue))
                                {
                                    if (!currentValue.IsNil) report.SMSFAR172 = currentValue.Value;
                                }
                            }
                        }
                        #endregion End of Tuple sbrTrusteeOrDirector_ElectronicContactTelephoneCollection
                        if (validationMode) ValidateRPTrusteeOrDirector_ElectronicContactTelephoneCollection(errors, sbrTrusteeOrDirector_ElectronicContactTelephoneCollection, sbrRPContexts, report);


                        IEnumerable<SBRElement> sbrTrusteeOrDirector_ElectronicContactElectronicMailCollection;
                        if (sbrTrusteeOrDirectorTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}ElectronicContactElectronicMail", out currentValue))
                            sbrTrusteeOrDirector_ElectronicContactElectronicMailCollection = currentValue.Occurrences;
                        else
                            sbrTrusteeOrDirector_ElectronicContactElectronicMailCollection = null;

                        #region sbrTrusteeOrDirector_ElectronicContactElectronicMailCollection

                        if (sbrTrusteeOrDirector_ElectronicContactElectronicMailCollection != null)
                        {
                            report.TrusteeOrDirector_ElectronicContactElectronicMailCollectionExists = true;
                            report.TrusteeOrDirector_ElectronicContactElectronicMailCollectionCount = sbrTrusteeOrDirector_ElectronicContactElectronicMailCollection.Count();

                            IEnumerable<SBRElement> sbrTrusteeOrDirector_ElectronicContactElectronicMailCollection_03_RPCollection;
                            sbrTrusteeOrDirector_ElectronicContactElectronicMailCollection_03_RPCollection = sbrTrusteeOrDirector_ElectronicContactElectronicMailCollection.Where(
                                ce => ce.ChildElements != null &&
                                ce.ChildElements.Any(te => te.Name == "ElectronicContact.ElectronicMail.Usage.Code" && te.Value == "03" &&
                                sbrRPContexts.Contains(te.Context)));
                            if (sbrTrusteeOrDirector_ElectronicContactElectronicMailCollection_03_RPCollection != null && sbrTrusteeOrDirector_ElectronicContactElectronicMailCollection_03_RPCollection.Count() > 0)
                            {
                                SBRElement sbrTrusteeOrDirector_ElectronicContactElectronicMailCollection_03_RP;
                                sbrTrusteeOrDirector_ElectronicContactElectronicMailCollection_03_RP = sbrTrusteeOrDirector_ElectronicContactElectronicMailCollection_03_RPCollection.ElementAt(0);

                                IDictionary<string, SBRElement> sbrTrusteeOrDirector_ElectronicContactElectronicMailCollection_03_RPTupleElementMap;
                                sbrTrusteeOrDirector_ElectronicContactElectronicMailCollection_03_RPTupleElementMap = sbrTrusteeOrDirector_ElectronicContactElectronicMailCollection_03_RP.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();

                                report.TrusteeOrDirector_ElectronicContactElectronicMailCollection_03_RPExists = true;
                                report.TrusteeOrDirector_ElectronicContactElectronicMailCollection_03_RPCount = sbrTrusteeOrDirector_ElectronicContactElectronicMailCollection_03_RPCollection.Count();
                                report.TrusteeOrDirector_ElectronicContactElectronicMailCollection_03_RPOccurrenceIndex = sbrTrusteeOrDirector_ElectronicContactElectronicMailCollection_03_RP.OccurrenceIndex + 1;
                                report.TrusteeOrDirector_ElectronicContactElectronicMailCollection_03_RPLastOccurrenceIndex = sbrTrusteeOrDirector_ElectronicContactElectronicMailCollection_03_RPCollection.Last().OccurrenceIndex + 1;

                                if (sbrTrusteeOrDirector_ElectronicContactElectronicMailCollection_03_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}ElectronicContact.ElectronicMail.Usage.Code", out currentValue))
                                {
                                    if (!currentValue.IsNil) report.SMSFAR284 = currentValue.Value;
                                }

                                if (sbrTrusteeOrDirector_ElectronicContactElectronicMailCollection_03_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}ElectronicContact.ElectronicMail.Address.Text", out currentValue))
                                {
                                    if (!currentValue.IsNil) report.SMSFAR173 = currentValue.Value;
                                }
                            }
                        }
                        #endregion End of Tuple sbrTrusteeOrDirector_ElectronicContactElectronicMailCollection
                        if (validationMode) ValidateRPTrusteeOrDirector_ElectronicContactElectronicMailCollection(errors, sbrTrusteeOrDirector_ElectronicContactElectronicMailCollection, sbrRPContexts, report);


                        IEnumerable<SBRElement> sbrTrusteeOrDirector_OrganisationNameDetailsCollection;
                        if (sbrTrusteeOrDirectorTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}OrganisationNameDetails", out currentValue))
                            sbrTrusteeOrDirector_OrganisationNameDetailsCollection = currentValue.Occurrences;
                        else
                            sbrTrusteeOrDirector_OrganisationNameDetailsCollection = null;

                        #region sbrTrusteeOrDirector_OrganisationNameDetailsCollection

                        if (sbrTrusteeOrDirector_OrganisationNameDetailsCollection != null)
                        {
                            report.TrusteeOrDirector_OrganisationNameDetailsCollectionExists = true;
                            report.TrusteeOrDirector_OrganisationNameDetailsCollectionCount = sbrTrusteeOrDirector_OrganisationNameDetailsCollection.Count();

                            IEnumerable<SBRElement> sbrTrusteeOrDirector_OrganisationNameDetailsCollection_MN_C_RPCollection;
                            sbrTrusteeOrDirector_OrganisationNameDetailsCollection_MN_C_RPCollection = sbrTrusteeOrDirector_OrganisationNameDetailsCollection.Where(
                                ce => ce.ChildElements != null &&
                                ce.ChildElements.Any(te => te.Name == "OrganisationNameDetails.OrganisationalNameType.Code" && te.Value == "MN") &&
                                ce.ChildElements.Any(te => te.Name == "OrganisationNameDetails.Currency.Code" && te.Value == "C" &&
                                sbrRPContexts.Contains(te.Context)));
                            if (sbrTrusteeOrDirector_OrganisationNameDetailsCollection_MN_C_RPCollection != null && sbrTrusteeOrDirector_OrganisationNameDetailsCollection_MN_C_RPCollection.Count() > 0)
                            {
                                SBRElement sbrTrusteeOrDirector_OrganisationNameDetailsCollection_MN_C_RP;
                                sbrTrusteeOrDirector_OrganisationNameDetailsCollection_MN_C_RP = sbrTrusteeOrDirector_OrganisationNameDetailsCollection_MN_C_RPCollection.ElementAt(0);

                                IDictionary<string, SBRElement> sbrTrusteeOrDirector_OrganisationNameDetailsCollection_MN_C_RPTupleElementMap;
                                sbrTrusteeOrDirector_OrganisationNameDetailsCollection_MN_C_RPTupleElementMap = sbrTrusteeOrDirector_OrganisationNameDetailsCollection_MN_C_RP.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();

                                report.TrusteeOrDirector_OrganisationNameDetailsCollection_MN_C_RPExists = true;
                                report.TrusteeOrDirector_OrganisationNameDetailsCollection_MN_C_RPCount = sbrTrusteeOrDirector_OrganisationNameDetailsCollection_MN_C_RPCollection.Count();
                                report.TrusteeOrDirector_OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex = sbrTrusteeOrDirector_OrganisationNameDetailsCollection_MN_C_RP.OccurrenceIndex + 1;
                                report.TrusteeOrDirector_OrganisationNameDetailsCollection_MN_C_RPLastOccurrenceIndex = sbrTrusteeOrDirector_OrganisationNameDetailsCollection_MN_C_RPCollection.Last().OccurrenceIndex + 1;

                                if (sbrTrusteeOrDirector_OrganisationNameDetailsCollection_MN_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}OrganisationNameDetails.OrganisationalNameType.Code", out currentValue))
                                {
                                    if (!currentValue.IsNil) report.SMSFAR236 = currentValue.Value;
                                }

                                if (sbrTrusteeOrDirector_OrganisationNameDetailsCollection_MN_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}OrganisationNameDetails.Currency.Code", out currentValue))
                                {
                                    if (!currentValue.IsNil) report.SMSFAR317 = currentValue.Value;
                                }

                                if (sbrTrusteeOrDirector_OrganisationNameDetailsCollection_MN_C_RPTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}OrganisationNameDetails.OrganisationalName.Text", out currentValue))
                                {
                                    if (!currentValue.IsNil) report.SMSFAR174 = currentValue.Value;
                                }
                            }
                        }
                        #endregion End of Tuple sbrTrusteeOrDirector_OrganisationNameDetailsCollection
                        if (validationMode) ValidateRPTrusteeOrDirector_OrganisationNameDetailsCollection(errors, sbrTrusteeOrDirector_OrganisationNameDetailsCollection, sbrRPContexts, report);


                        if (sbrTrusteeOrDirectorTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Identifiers.AustralianBusinessNumber.Identifier", out currentValue))
                        {
                            if (!currentValue.IsNil) report.SMSFAR175 = currentValue.Value;
                        }
                    }
                    #endregion End of Tuple sbrTrusteeOrDirectorCollection
                }
            }
            #endregion End of Context sbrRPContext

            #region sbrRPInstantContext
            IEnumerable<SBRContext> sbrRPInstantContexts = null;
            SBRContext sbrRPInstantContext = null;

            sbrRPInstantContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.INSTANT &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 1 &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty")))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPInstantElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPInstantContexts != null && sbrRPInstantContexts.Count() > 0)
            {
                sbrRPInstantContext = sbrRPInstantContexts.First();

                report.RPInstantId = sbrRPInstantContext.Id;
                report.RPInstantIdentifier = sbrRPInstantContext.EntityIdentifier;
                report.RPInstantIdentifierScheme = sbrRPInstantContext.EntityScheme;

                report.RPInstantCount = sbrRPInstantContexts.Count();
                report.RPInstantExists = true;
                report.RPInstantOccurrenceIndex = Array.IndexOf(contextArray, sbrRPInstantContext) + 1;
                report.RPInstantLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPInstantContexts.Last()) + 1;


                if (report.RPInstantIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPInstantIdentifierTFN = report.RPInstantIdentifier;

                report.RPInstantInstantDate = ((SBRInstantPeriod)sbrRPInstantContext.Period).InstantAsDateObject;

                sbrPopulatedRPInstantElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPInstantContexts.Count(); i++)
                {
                    sbrRPInstantContext = sbrRPInstantContexts.ElementAt(i);

                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPInstantContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPInstantElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPInstantElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPInstantElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement });
                            }
                        }
                    }

                    if (sbrPopulatedRPInstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Liabilities.Other.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR139 = decimal.Parse(currentValues[0].Value);
                    }
                }
            }
            #endregion End of Context sbrRPInstantContext

            #region sbrINTContext
            IEnumerable<SBRContext> sbrINTContexts = null;
            SBRContext sbrINTContext = null;

            sbrINTContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 1 &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Intermediary")))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedINTElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrINTContexts != null && sbrINTContexts.Count() > 0)
            {
                sbrINTContext = sbrINTContexts.First();

                report.INTId = sbrINTContext.Id;
                report.INTIdentifier = sbrINTContext.EntityIdentifier;
                report.INTIdentifierScheme = sbrINTContext.EntityScheme;

                report.INTCount = sbrINTContexts.Count();
                report.INTExists = true;
                report.INTOccurrenceIndex = Array.IndexOf(contextArray, sbrINTContext) + 1;
                report.INTLastOccurrenceIndex = Array.IndexOf(contextArray, sbrINTContexts.Last()) + 1;


                if (report.INTIdentifierScheme == "http://www.ato.gov.au/abn") report.INTIdentifierABN = report.INTIdentifier;

                report.INTStartDate = ((SBRDurationPeriod)sbrINTContext.Period).DurationStartAsDateObject;
                report.INTEndDate = ((SBRDurationPeriod)sbrINTContext.Period).DurationEndAsDateObject;

                sbrPopulatedINTElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrINTContexts.Count(); i++)
                {
                    sbrINTContext = sbrINTContexts.ElementAt(i);

                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrINTContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedINTElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedINTElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedINTElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement });
                            }
                        }
                    }

                    if (sbrPopulatedINTElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Identifiers.TaxAgentClientReference.Text", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR185 = currentValues[0].Value;
                    }

                    if (sbrPopulatedINTElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Identifiers.TaxAgentNumber.Identifier", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR2 = currentValues[0].Value;
                    }

                    IEnumerable<SBRElement> sbrDeclarationCollection;
                    if (populatedReportElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Declaration", out currentValue))
                        sbrDeclarationCollection = currentValue.Occurrences;
                    else
                        sbrDeclarationCollection = null;

                    #region sbrDeclarationCollection

                    if (sbrDeclarationCollection != null)
                    {

                        IEnumerable<SBRElement> sbrDeclarationCollection_TrueAndCorrect_INTCollection;
                        sbrDeclarationCollection_TrueAndCorrect_INTCollection = sbrDeclarationCollection.Where(
                            ce => ce.ChildElements != null &&
                            ce.ChildElements.Any(te => te.Name == "Declaration.StatementType.Code" && te.Value == "TrueAndCorrect" &&
                            sbrINTContexts.Contains(te.Context)));
                        if (sbrDeclarationCollection_TrueAndCorrect_INTCollection != null && sbrDeclarationCollection_TrueAndCorrect_INTCollection.Count() > 0)
                        {
                            SBRElement sbrDeclarationCollection_TrueAndCorrect_INT;
                            sbrDeclarationCollection_TrueAndCorrect_INT = sbrDeclarationCollection_TrueAndCorrect_INTCollection.ElementAt(0);

                            IDictionary<string, SBRElement> sbrDeclarationCollection_TrueAndCorrect_INTTupleElementMap;
                            sbrDeclarationCollection_TrueAndCorrect_INTTupleElementMap = sbrDeclarationCollection_TrueAndCorrect_INT.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();

                            report.DeclarationCollection_TrueAndCorrect_INTExists = true;
                            report.DeclarationCollection_TrueAndCorrect_INTCount = sbrDeclarationCollection_TrueAndCorrect_INTCollection.Count();
                            report.DeclarationCollection_TrueAndCorrect_INTOccurrenceIndex = sbrDeclarationCollection_TrueAndCorrect_INT.OccurrenceIndex + 1;
                            report.DeclarationCollection_TrueAndCorrect_INTLastOccurrenceIndex = sbrDeclarationCollection_TrueAndCorrect_INTCollection.Last().OccurrenceIndex + 1;

                            if (sbrDeclarationCollection_TrueAndCorrect_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Declaration.StatementType.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR270 = currentValue.Value;
                            }

                            if (sbrDeclarationCollection_TrueAndCorrect_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Declaration.StatementAccepted.Indicator", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR271 = ToBoolean(currentValue.Value);
                            }

                            if (sbrDeclarationCollection_TrueAndCorrect_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Declaration.Statement.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR193 = currentValue.Value;
                            }

                            if (sbrDeclarationCollection_TrueAndCorrect_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Declaration.Signature.Date", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR194 = DateTime.Parse(currentValue.Value);
                            }

                            if (sbrDeclarationCollection_TrueAndCorrect_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Declaration.SignatoryIdentifier.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR272 = currentValue.Value;
                            }

                            IEnumerable<SBRElement> sbrDeclaration_PersonUnstructuredNameCollection;
                            if (sbrDeclarationCollection_TrueAndCorrect_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonUnstructuredName", out currentValue))
                                sbrDeclaration_PersonUnstructuredNameCollection = currentValue.Occurrences;
                            else
                                sbrDeclaration_PersonUnstructuredNameCollection = null;

                            #region sbrDeclaration_PersonUnstructuredNameCollection

                            if (sbrDeclaration_PersonUnstructuredNameCollection != null)
                            {

                                IEnumerable<SBRElement> sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTCollection;
                                sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTCollection = sbrDeclaration_PersonUnstructuredNameCollection.Where(
                                    ce => ce.ChildElements != null &&
                                    ce.ChildElements.Any(te => te.Name == "PersonUnstructuredName.Usage.Code" && te.Value == "DeclarationSignatory" &&
                                    sbrINTContexts.Contains(te.Context)));
                                if (sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTCollection != null && sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTCollection.Count() > 0)
                                {
                                    SBRElement sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INT;
                                    sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INT = sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTCollection.ElementAt(0);

                                    IDictionary<string, SBRElement> sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTTupleElementMap;
                                    sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTTupleElementMap = sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INT.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();

                                    report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTExists = true;
                                    report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTCount = sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTCollection.Count();
                                    report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTOccurrenceIndex = sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INT.OccurrenceIndex + 1;
                                    report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTLastOccurrenceIndex = sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTCollection.Last().OccurrenceIndex + 1;

                                    if (sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonUnstructuredName.Usage.Code", out currentValue))
                                    {
                                        if (!currentValue.IsNil) report.SMSFAR279 = currentValue.Value;
                                    }

                                    if (sbrDeclaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonUnstructuredName.FullName.Text", out currentValue))
                                    {
                                        if (!currentValue.IsNil) report.SMSFAR280 = currentValue.Value;
                                    }
                                }
                            }
                            #endregion End of Tuple sbrDeclaration_PersonUnstructuredNameCollection
                            if (validationMode) ValidateINTDeclaration_PersonUnstructuredNameCollection(errors, sbrDeclaration_PersonUnstructuredNameCollection, sbrINTContexts, report);

                        }
                    }
                    #endregion End of Tuple sbrDeclarationCollection
                    if (validationMode) ValidateINTDeclarationCollection(errors, sbrDeclarationCollection, sbrINTContexts, report);


                    IEnumerable<SBRElement> sbrPersonNameDetailsCollection;
                    if (populatedReportElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails", out currentValue))
                        sbrPersonNameDetailsCollection = currentValue.Occurrences;
                    else
                        sbrPersonNameDetailsCollection = null;

                    #region sbrPersonNameDetailsCollection

                    if (sbrPersonNameDetailsCollection != null)
                    {
                        report.PersonNameDetailsCollectionExists = true;
                        report.PersonNameDetailsCollectionCount = sbrPersonNameDetailsCollection.Count();

                        IEnumerable<SBRElement> sbrPersonNameDetailsCollection_LGL_Contact_C_INTCollection;
                        sbrPersonNameDetailsCollection_LGL_Contact_C_INTCollection = sbrPersonNameDetailsCollection.Where(
                            ce => ce.ChildElements != null &&
                            ce.ChildElements.Any(te => te.Name == "PersonNameDetails.PersonNameType.Code" && te.Value == "LGL") &&
                            ce.ChildElements.Any(te => te.Name == "PersonNameDetails.Usage.Code" && te.Value == "Contact") &&
                            ce.ChildElements.Any(te => te.Name == "PersonNameDetails.Currency.Code" && te.Value == "C" &&
                            sbrINTContexts.Contains(te.Context)));
                        if (sbrPersonNameDetailsCollection_LGL_Contact_C_INTCollection != null && sbrPersonNameDetailsCollection_LGL_Contact_C_INTCollection.Count() > 0)
                        {
                            SBRElement sbrPersonNameDetailsCollection_LGL_Contact_C_INT;
                            sbrPersonNameDetailsCollection_LGL_Contact_C_INT = sbrPersonNameDetailsCollection_LGL_Contact_C_INTCollection.ElementAt(0);

                            IDictionary<string, SBRElement> sbrPersonNameDetailsCollection_LGL_Contact_C_INTTupleElementMap;
                            sbrPersonNameDetailsCollection_LGL_Contact_C_INTTupleElementMap = sbrPersonNameDetailsCollection_LGL_Contact_C_INT.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();

                            report.PersonNameDetailsCollection_LGL_Contact_C_INTExists = true;
                            report.PersonNameDetailsCollection_LGL_Contact_C_INTCount = sbrPersonNameDetailsCollection_LGL_Contact_C_INTCollection.Count();
                            report.PersonNameDetailsCollection_LGL_Contact_C_INTOccurrenceIndex = sbrPersonNameDetailsCollection_LGL_Contact_C_INT.OccurrenceIndex + 1;
                            report.PersonNameDetailsCollection_LGL_Contact_C_INTLastOccurrenceIndex = sbrPersonNameDetailsCollection_LGL_Contact_C_INTCollection.Last().OccurrenceIndex + 1;

                            if (sbrPersonNameDetailsCollection_LGL_Contact_C_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.PersonNameType.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR273 = currentValue.Value;
                            }

                            if (sbrPersonNameDetailsCollection_LGL_Contact_C_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.Usage.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR274 = currentValue.Value;
                            }

                            if (sbrPersonNameDetailsCollection_LGL_Contact_C_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.Currency.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR275 = currentValue.Value;
                            }

                            if (sbrPersonNameDetailsCollection_LGL_Contact_C_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.Title.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR177 = currentValue.Value;
                            }

                            if (sbrPersonNameDetailsCollection_LGL_Contact_C_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.NameSuffix.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR179 = currentValue.Value;
                            }

                            if (sbrPersonNameDetailsCollection_LGL_Contact_C_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.FamilyName.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR178 = currentValue.Value;
                            }

                            if (sbrPersonNameDetailsCollection_LGL_Contact_C_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.GivenName.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR180 = currentValue.Value;
                            }

                            if (sbrPersonNameDetailsCollection_LGL_Contact_C_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.OtherGivenName.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR181 = currentValue.Value;
                            }
                        }
                    }
                    #endregion End of Tuple sbrPersonNameDetailsCollection
                    if (validationMode) ValidateINTPersonNameDetailsCollection(errors, sbrPersonNameDetailsCollection, sbrINTContexts, report);


                    IEnumerable<SBRElement> sbrOrganisationNameDetailsCollection;
                    if (populatedReportElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}OrganisationNameDetails", out currentValue))
                        sbrOrganisationNameDetailsCollection = currentValue.Occurrences;
                    else
                        sbrOrganisationNameDetailsCollection = null;

                    #region sbrOrganisationNameDetailsCollection

                    if (sbrOrganisationNameDetailsCollection != null)
                    {

                        IEnumerable<SBRElement> sbrOrganisationNameDetailsCollection_MN_C_INTCollection;
                        sbrOrganisationNameDetailsCollection_MN_C_INTCollection = sbrOrganisationNameDetailsCollection.Where(
                            ce => ce.ChildElements != null &&
                            ce.ChildElements.Any(te => te.Name == "OrganisationNameDetails.OrganisationalNameType.Code" && te.Value == "MN") &&
                            ce.ChildElements.Any(te => te.Name == "OrganisationNameDetails.Currency.Code" && te.Value == "C" &&
                            sbrINTContexts.Contains(te.Context)));
                        if (sbrOrganisationNameDetailsCollection_MN_C_INTCollection != null && sbrOrganisationNameDetailsCollection_MN_C_INTCollection.Count() > 0)
                        {
                            SBRElement sbrOrganisationNameDetailsCollection_MN_C_INT;
                            sbrOrganisationNameDetailsCollection_MN_C_INT = sbrOrganisationNameDetailsCollection_MN_C_INTCollection.ElementAt(0);

                            IDictionary<string, SBRElement> sbrOrganisationNameDetailsCollection_MN_C_INTTupleElementMap;
                            sbrOrganisationNameDetailsCollection_MN_C_INTTupleElementMap = sbrOrganisationNameDetailsCollection_MN_C_INT.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();

                            report.OrganisationNameDetailsCollection_MN_C_INTExists = true;
                            report.OrganisationNameDetailsCollection_MN_C_INTCount = sbrOrganisationNameDetailsCollection_MN_C_INTCollection.Count();
                            report.OrganisationNameDetailsCollection_MN_C_INTOccurrenceIndex = sbrOrganisationNameDetailsCollection_MN_C_INT.OccurrenceIndex + 1;
                            report.OrganisationNameDetailsCollection_MN_C_INTLastOccurrenceIndex = sbrOrganisationNameDetailsCollection_MN_C_INTCollection.Last().OccurrenceIndex + 1;

                            if (sbrOrganisationNameDetailsCollection_MN_C_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}OrganisationNameDetails.OrganisationalNameType.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR276 = currentValue.Value;
                            }

                            if (sbrOrganisationNameDetailsCollection_MN_C_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}OrganisationNameDetails.Currency.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR318 = currentValue.Value;
                            }

                            if (sbrOrganisationNameDetailsCollection_MN_C_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}OrganisationNameDetails.OrganisationalName.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR182 = currentValue.Value;
                            }
                        }
                    }
                    #endregion End of Tuple sbrOrganisationNameDetailsCollection
                    if (validationMode) ValidateINTOrganisationNameDetailsCollection(errors, sbrOrganisationNameDetailsCollection, sbrINTContexts, report);


                    IEnumerable<SBRElement> sbrElectronicContactTelephoneCollection;
                    if (populatedReportElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}ElectronicContactTelephone", out currentValue))
                        sbrElectronicContactTelephoneCollection = currentValue.Occurrences;
                    else
                        sbrElectronicContactTelephoneCollection = null;

                    #region sbrElectronicContactTelephoneCollection

                    if (sbrElectronicContactTelephoneCollection != null)
                    {
                        report.ElectronicContactTelephoneCollectionExists = true;
                        report.ElectronicContactTelephoneCollectionCount = sbrElectronicContactTelephoneCollection.Count();

                        IEnumerable<SBRElement> sbrElectronicContactTelephoneCollection_03_INTCollection;
                        sbrElectronicContactTelephoneCollection_03_INTCollection = sbrElectronicContactTelephoneCollection.Where(
                            ce => ce.ChildElements != null &&
                            ce.ChildElements.Any(te => te.Name == "ElectronicContact.Telephone.Usage.Code" && te.Value == "03" &&
                            sbrINTContexts.Contains(te.Context)));
                        if (sbrElectronicContactTelephoneCollection_03_INTCollection != null && sbrElectronicContactTelephoneCollection_03_INTCollection.Count() > 0)
                        {
                            SBRElement sbrElectronicContactTelephoneCollection_03_INT;
                            sbrElectronicContactTelephoneCollection_03_INT = sbrElectronicContactTelephoneCollection_03_INTCollection.ElementAt(0);

                            IDictionary<string, SBRElement> sbrElectronicContactTelephoneCollection_03_INTTupleElementMap;
                            sbrElectronicContactTelephoneCollection_03_INTTupleElementMap = sbrElectronicContactTelephoneCollection_03_INT.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();

                            report.ElectronicContactTelephoneCollection_03_INTExists = true;
                            report.ElectronicContactTelephoneCollection_03_INTCount = sbrElectronicContactTelephoneCollection_03_INTCollection.Count();
                            report.ElectronicContactTelephoneCollection_03_INTOccurrenceIndex = sbrElectronicContactTelephoneCollection_03_INT.OccurrenceIndex + 1;
                            report.ElectronicContactTelephoneCollection_03_INTLastOccurrenceIndex = sbrElectronicContactTelephoneCollection_03_INTCollection.Last().OccurrenceIndex + 1;

                            if (sbrElectronicContactTelephoneCollection_03_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}ElectronicContact.Telephone.Usage.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR263 = currentValue.Value;
                            }

                            if (sbrElectronicContactTelephoneCollection_03_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}ElectronicContact.Telephone.ServiceLine.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR266 = currentValue.Value;
                            }

                            if (sbrElectronicContactTelephoneCollection_03_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}ElectronicContact.Telephone.Area.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR183 = currentValue.Value;
                            }

                            if (sbrElectronicContactTelephoneCollection_03_INTTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}ElectronicContact.Telephone.Minimal.Number", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR184 = currentValue.Value;
                            }
                        }
                    }
                    #endregion End of Tuple sbrElectronicContactTelephoneCollection
                    if (validationMode) ValidateINTElectronicContactTelephoneCollection(errors, sbrElectronicContactTelephoneCollection, sbrINTContexts, report);

                }
            }
            #endregion End of Context sbrINTContext

            #region sbrAuditorContext
            IEnumerable<SBRContext> sbrAuditorContexts = null;
            SBRContext sbrAuditorContext = null;

            sbrAuditorContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 1 &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Auditor")))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedAuditorElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrAuditorContexts != null && sbrAuditorContexts.Count() > 0)
            {
                sbrAuditorContext = sbrAuditorContexts.First();

                report.AuditorId = sbrAuditorContext.Id;
                report.AuditorIdentifier = sbrAuditorContext.EntityIdentifier;
                report.AuditorIdentifierScheme = sbrAuditorContext.EntityScheme;

                report.AuditorCount = sbrAuditorContexts.Count();
                report.AuditorExists = true;
                report.AuditorOccurrenceIndex = Array.IndexOf(contextArray, sbrAuditorContext) + 1;
                report.AuditorLastOccurrenceIndex = Array.IndexOf(contextArray, sbrAuditorContexts.Last()) + 1;


                if (report.AuditorIdentifierScheme == "http://www.asic.gov.au/san") report.AuditorIdentifierSAN = report.AuditorIdentifier;

                report.AuditorStartDate = ((SBRDurationPeriod)sbrAuditorContext.Period).DurationStartAsDateObject;
                report.AuditorEndDate = ((SBRDurationPeriod)sbrAuditorContext.Period).DurationEndAsDateObject;

                sbrPopulatedAuditorElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrAuditorContexts.Count(); i++)
                {
                    sbrAuditorContext = sbrAuditorContexts.ElementAt(i);

                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrAuditorContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedAuditorElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedAuditorElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedAuditorElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement });
                            }
                        }
                    }
                    report.SMSFAR300 = report.AuditorIdentifier;

                    IEnumerable<SBRElement> sbrPersonNameDetailsCollection;
                    if (populatedReportElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails", out currentValue))
                        sbrPersonNameDetailsCollection = currentValue.Occurrences;
                    else
                        sbrPersonNameDetailsCollection = null;

                    #region sbrPersonNameDetailsCollection

                    if (sbrPersonNameDetailsCollection != null)
                    {

                        IEnumerable<SBRElement> sbrPersonNameDetailsCollection_LGL_Contact_C_AuditorCollection;
                        sbrPersonNameDetailsCollection_LGL_Contact_C_AuditorCollection = sbrPersonNameDetailsCollection.Where(
                            ce => ce.ChildElements != null &&
                            ce.ChildElements.Any(te => te.Name == "PersonNameDetails.PersonNameType.Code" && te.Value == "LGL") &&
                            ce.ChildElements.Any(te => te.Name == "PersonNameDetails.Usage.Code" && te.Value == "Contact") &&
                            ce.ChildElements.Any(te => te.Name == "PersonNameDetails.Currency.Code" && te.Value == "C" &&
                            sbrAuditorContexts.Contains(te.Context)));
                        if (sbrPersonNameDetailsCollection_LGL_Contact_C_AuditorCollection != null && sbrPersonNameDetailsCollection_LGL_Contact_C_AuditorCollection.Count() > 0)
                        {
                            SBRElement sbrPersonNameDetailsCollection_LGL_Contact_C_Auditor;
                            sbrPersonNameDetailsCollection_LGL_Contact_C_Auditor = sbrPersonNameDetailsCollection_LGL_Contact_C_AuditorCollection.ElementAt(0);

                            IDictionary<string, SBRElement> sbrPersonNameDetailsCollection_LGL_Contact_C_AuditorTupleElementMap;
                            sbrPersonNameDetailsCollection_LGL_Contact_C_AuditorTupleElementMap = sbrPersonNameDetailsCollection_LGL_Contact_C_Auditor.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();

                            report.PersonNameDetailsCollection_LGL_Contact_C_AuditorExists = true;
                            report.PersonNameDetailsCollection_LGL_Contact_C_AuditorCount = sbrPersonNameDetailsCollection_LGL_Contact_C_AuditorCollection.Count();
                            report.PersonNameDetailsCollection_LGL_Contact_C_AuditorOccurrenceIndex = sbrPersonNameDetailsCollection_LGL_Contact_C_Auditor.OccurrenceIndex + 1;
                            report.PersonNameDetailsCollection_LGL_Contact_C_AuditorLastOccurrenceIndex = sbrPersonNameDetailsCollection_LGL_Contact_C_AuditorCollection.Last().OccurrenceIndex + 1;

                            if (sbrPersonNameDetailsCollection_LGL_Contact_C_AuditorTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.PersonNameType.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR249 = currentValue.Value;
                            }

                            if (sbrPersonNameDetailsCollection_LGL_Contact_C_AuditorTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.Usage.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR253 = currentValue.Value;
                            }

                            if (sbrPersonNameDetailsCollection_LGL_Contact_C_AuditorTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.Currency.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR257 = currentValue.Value;
                            }

                            if (sbrPersonNameDetailsCollection_LGL_Contact_C_AuditorTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.Title.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR10 = currentValue.Value;
                            }

                            if (sbrPersonNameDetailsCollection_LGL_Contact_C_AuditorTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.NameSuffix.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR12 = currentValue.Value;
                            }

                            if (sbrPersonNameDetailsCollection_LGL_Contact_C_AuditorTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.FamilyName.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR11 = currentValue.Value;
                            }

                            if (sbrPersonNameDetailsCollection_LGL_Contact_C_AuditorTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.GivenName.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR13 = currentValue.Value;
                            }

                            if (sbrPersonNameDetailsCollection_LGL_Contact_C_AuditorTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.OtherGivenName.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR14 = currentValue.Value;
                            }
                        }
                    }
                    #endregion End of Tuple sbrPersonNameDetailsCollection
                    if (validationMode) ValidateAuditorPersonNameDetailsCollection(errors, sbrPersonNameDetailsCollection, sbrAuditorContexts, report);


                    IEnumerable<SBRElement> sbrElectronicContactTelephoneCollection;
                    if (populatedReportElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}ElectronicContactTelephone", out currentValue))
                        sbrElectronicContactTelephoneCollection = currentValue.Occurrences;
                    else
                        sbrElectronicContactTelephoneCollection = null;

                    #region sbrElectronicContactTelephoneCollection

                    if (sbrElectronicContactTelephoneCollection != null)
                    {

                        IEnumerable<SBRElement> sbrElectronicContactTelephoneCollection_03_AuditorCollection;
                        sbrElectronicContactTelephoneCollection_03_AuditorCollection = sbrElectronicContactTelephoneCollection.Where(
                            ce => ce.ChildElements != null &&
                            ce.ChildElements.Any(te => te.Name == "ElectronicContact.Telephone.Usage.Code" && te.Value == "03" &&
                            sbrAuditorContexts.Contains(te.Context)));
                        if (sbrElectronicContactTelephoneCollection_03_AuditorCollection != null && sbrElectronicContactTelephoneCollection_03_AuditorCollection.Count() > 0)
                        {
                            SBRElement sbrElectronicContactTelephoneCollection_03_Auditor;
                            sbrElectronicContactTelephoneCollection_03_Auditor = sbrElectronicContactTelephoneCollection_03_AuditorCollection.ElementAt(0);

                            IDictionary<string, SBRElement> sbrElectronicContactTelephoneCollection_03_AuditorTupleElementMap;
                            sbrElectronicContactTelephoneCollection_03_AuditorTupleElementMap = sbrElectronicContactTelephoneCollection_03_Auditor.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();

                            report.ElectronicContactTelephoneCollection_03_AuditorExists = true;
                            report.ElectronicContactTelephoneCollection_03_AuditorCount = sbrElectronicContactTelephoneCollection_03_AuditorCollection.Count();
                            report.ElectronicContactTelephoneCollection_03_AuditorOccurrenceIndex = sbrElectronicContactTelephoneCollection_03_Auditor.OccurrenceIndex + 1;
                            report.ElectronicContactTelephoneCollection_03_AuditorLastOccurrenceIndex = sbrElectronicContactTelephoneCollection_03_AuditorCollection.Last().OccurrenceIndex + 1;

                            if (sbrElectronicContactTelephoneCollection_03_AuditorTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}ElectronicContact.Telephone.Usage.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR261 = currentValue.Value;
                            }

                            if (sbrElectronicContactTelephoneCollection_03_AuditorTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}ElectronicContact.Telephone.ServiceLine.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR264 = currentValue.Value;
                            }

                            if (sbrElectronicContactTelephoneCollection_03_AuditorTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}ElectronicContact.Telephone.Area.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR17 = currentValue.Value;
                            }

                            if (sbrElectronicContactTelephoneCollection_03_AuditorTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}ElectronicContact.Telephone.Minimal.Number", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR18 = currentValue.Value;
                            }
                        }
                    }
                    #endregion End of Tuple sbrElectronicContactTelephoneCollection
                    if (validationMode) ValidateAuditorElectronicContactTelephoneCollection(errors, sbrElectronicContactTelephoneCollection, sbrAuditorContexts, report);


                    IEnumerable<SBRElement> sbrAddressDetailsCollection;
                    if (populatedReportElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}AddressDetails", out currentValue))
                        sbrAddressDetailsCollection = currentValue.Occurrences;
                    else
                        sbrAddressDetailsCollection = null;

                    #region sbrAddressDetailsCollection

                    if (sbrAddressDetailsCollection != null)
                    {

                        IEnumerable<SBRElement> sbrAddressDetailsCollection_POS_C_AuditorCollection;
                        sbrAddressDetailsCollection_POS_C_AuditorCollection = sbrAddressDetailsCollection.Where(
                            ce => ce.ChildElements != null &&
                            ce.ChildElements.Any(te => te.Name == "AddressDetails.Usage.Code" && te.Value == "POS") &&
                            ce.ChildElements.Any(te => te.Name == "AddressDetails.Currency.Code" && te.Value == "C" &&
                            sbrAuditorContexts.Contains(te.Context)));
                        if (sbrAddressDetailsCollection_POS_C_AuditorCollection != null && sbrAddressDetailsCollection_POS_C_AuditorCollection.Count() > 0)
                        {
                            SBRElement sbrAddressDetailsCollection_POS_C_Auditor;
                            sbrAddressDetailsCollection_POS_C_Auditor = sbrAddressDetailsCollection_POS_C_AuditorCollection.ElementAt(0);

                            IDictionary<string, SBRElement> sbrAddressDetailsCollection_POS_C_AuditorTupleElementMap;
                            sbrAddressDetailsCollection_POS_C_AuditorTupleElementMap = sbrAddressDetailsCollection_POS_C_Auditor.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();

                            report.AddressDetailsCollection_POS_C_AuditorExists = true;
                            report.AddressDetailsCollection_POS_C_AuditorCount = sbrAddressDetailsCollection_POS_C_AuditorCollection.Count();
                            report.AddressDetailsCollection_POS_C_AuditorOccurrenceIndex = sbrAddressDetailsCollection_POS_C_Auditor.OccurrenceIndex + 1;
                            report.AddressDetailsCollection_POS_C_AuditorLastOccurrenceIndex = sbrAddressDetailsCollection_POS_C_AuditorCollection.Last().OccurrenceIndex + 1;

                            if (sbrAddressDetailsCollection_POS_C_AuditorTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}AddressDetails.OverseasAddress.Indicator", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR286 = ToBoolean(currentValue.Value);
                            }

                            if (sbrAddressDetailsCollection_POS_C_AuditorTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}AddressDetails.Usage.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR238 = currentValue.Value;
                            }

                            if (sbrAddressDetailsCollection_POS_C_AuditorTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}AddressDetails.Currency.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR285 = currentValue.Value;
                            }

                            if (sbrAddressDetailsCollection_POS_C_AuditorTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}AddressDetails.Line1.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR19 = currentValue.Value;
                            }

                            if (sbrAddressDetailsCollection_POS_C_AuditorTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}AddressDetails.Line2.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR240 = currentValue.Value;
                            }

                            if (sbrAddressDetailsCollection_POS_C_AuditorTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}AddressDetails.Line3.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR242 = currentValue.Value;
                            }

                            if (sbrAddressDetailsCollection_POS_C_AuditorTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}AddressDetails.Line4.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR244 = currentValue.Value;
                            }

                            if (sbrAddressDetailsCollection_POS_C_AuditorTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}AddressDetails.LocalityName.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR20 = currentValue.Value;
                            }

                            if (sbrAddressDetailsCollection_POS_C_AuditorTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}AddressDetails.Postcode.Text", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR22 = currentValue.Value;
                            }

                            if (sbrAddressDetailsCollection_POS_C_AuditorTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}AddressDetails.StateOrTerritory.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR21 = currentValue.Value;
                            }

                            if (sbrAddressDetailsCollection_POS_C_AuditorTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}AddressDetails.Country.Code", out currentValue))
                            {
                                if (!currentValue.IsNil) report.SMSFAR246 = currentValue.Value;
                            }
                        }
                    }
                    #endregion End of Tuple sbrAddressDetailsCollection
                    if (validationMode) ValidateAuditorAddressDetailsCollection(errors, sbrAddressDetailsCollection, sbrAuditorContexts, report);

                }
            }
            #endregion End of Context sbrAuditorContext

            #region sbrRPJAUSContext
            IEnumerable<SBRContext> sbrRPJAUSContexts = null;
            SBRContext sbrRPJAUSContext = null;

            sbrRPJAUSContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty")) &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}InternationalJurisdictionDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Australian"))))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPJAUSElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPJAUSContexts != null && sbrRPJAUSContexts.Count() > 0)
            {
                sbrRPJAUSContext = sbrRPJAUSContexts.First();

                report.RPJAUSId = sbrRPJAUSContext.Id;
                report.RPJAUSIdentifier = sbrRPJAUSContext.EntityIdentifier;
                report.RPJAUSIdentifierScheme = sbrRPJAUSContext.EntityScheme;

                report.RPJAUSCount = sbrRPJAUSContexts.Count();
                report.RPJAUSExists = true;
                report.RPJAUSOccurrenceIndex = Array.IndexOf(contextArray, sbrRPJAUSContext) + 1;
                report.RPJAUSLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPJAUSContexts.Last()) + 1;


                if (report.RPJAUSIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPJAUSIdentifierTFN = report.RPJAUSIdentifier;

                report.RPJAUSStartDate = ((SBRDurationPeriod)sbrRPJAUSContext.Period).DurationStartAsDateObject;
                report.RPJAUSEndDate = ((SBRDurationPeriod)sbrRPJAUSContext.Period).DurationEndAsDateObject;

                sbrPopulatedRPJAUSElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPJAUSContexts.Count(); i++)
                {
                    sbrRPJAUSContext = sbrRPJAUSContexts.ElementAt(i);

                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPJAUSContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPJAUSElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPJAUSElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPJAUSElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement });
                            }
                        }
                    }

                    if (sbrPopulatedRPJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Assets.Investment.UnitTrustsListed.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR117 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Assets.Investment.UnitTrustsUnlisted.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR118 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Assets.Investment.EquitySecuritiesListed.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR124 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Assets.Investment.EquitySecuritiesUnlisted.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR125 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Assets.LimitedRecourseBorrowingRealProperty.Count", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR407 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Assets.LimitedRecourseBorrowing.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR311 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Assets.Investment.PropertyNonResidential.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR127 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Assets.Investment.PropertyResidential.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR128 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Assets.Investment.Collectables.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR312 = decimal.Parse(currentValues[0].Value);
                    }
                }
            }
            #endregion End of Context sbrRPJAUSContext

            #region sbrRPJFOREIGNContext
            IEnumerable<SBRContext> sbrRPJFOREIGNContexts = null;
            SBRContext sbrRPJFOREIGNContext = null;

            sbrRPJFOREIGNContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty")) &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}InternationalJurisdictionDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Foreign"))))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPJFOREIGNElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPJFOREIGNContexts != null && sbrRPJFOREIGNContexts.Count() > 0)
            {
                sbrRPJFOREIGNContext = sbrRPJFOREIGNContexts.First();

                report.RPJFOREIGNId = sbrRPJFOREIGNContext.Id;
                report.RPJFOREIGNIdentifier = sbrRPJFOREIGNContext.EntityIdentifier;
                report.RPJFOREIGNIdentifierScheme = sbrRPJFOREIGNContext.EntityScheme;

                report.RPJFOREIGNCount = sbrRPJFOREIGNContexts.Count();
                report.RPJFOREIGNExists = true;
                report.RPJFOREIGNOccurrenceIndex = Array.IndexOf(contextArray, sbrRPJFOREIGNContext) + 1;
                report.RPJFOREIGNLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPJFOREIGNContexts.Last()) + 1;


                if (report.RPJFOREIGNIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPJFOREIGNIdentifierTFN = report.RPJFOREIGNIdentifier;

                report.RPJFOREIGNStartDate = ((SBRDurationPeriod)sbrRPJFOREIGNContext.Period).DurationStartAsDateObject;
                report.RPJFOREIGNEndDate = ((SBRDurationPeriod)sbrRPJFOREIGNContext.Period).DurationEndAsDateObject;

                sbrPopulatedRPJFOREIGNElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPJFOREIGNContexts.Count(); i++)
                {
                    sbrRPJFOREIGNContext = sbrRPJFOREIGNContexts.ElementAt(i);

                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPJFOREIGNContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPJFOREIGNElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPJFOREIGNElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPJFOREIGNElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement });
                            }
                        }
                    }

                    if (sbrPopulatedRPJFOREIGNElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Income.Gross.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR37 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPJFOREIGNElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Income.Net.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR38 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPJFOREIGNElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Assets.Investment.PropertyNonResidential.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR132 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPJFOREIGNElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Assets.Investment.PropertyResidential.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR133 = decimal.Parse(currentValues[0].Value);
                    }
                }
            }
            #endregion End of Context sbrRPJFOREIGNContext

            #region sbrRPClosingContext
            IEnumerable<SBRContext> sbrRPClosingContexts = null;
            SBRContext sbrRPClosingContext = null;

            sbrRPClosingContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty")) &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}OpeningAndClosingBalanceDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Closing"))))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPClosingElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPClosingContexts != null && sbrRPClosingContexts.Count() > 0)
            {
                sbrRPClosingContext = sbrRPClosingContexts.First();

                report.RPClosingId = sbrRPClosingContext.Id;
                report.RPClosingIdentifier = sbrRPClosingContext.EntityIdentifier;
                report.RPClosingIdentifierScheme = sbrRPClosingContext.EntityScheme;

                report.RPClosingCount = sbrRPClosingContexts.Count();
                report.RPClosingExists = true;
                report.RPClosingOccurrenceIndex = Array.IndexOf(contextArray, sbrRPClosingContext) + 1;
                report.RPClosingLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPClosingContexts.Last()) + 1;


                if (report.RPClosingIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPClosingIdentifierTFN = report.RPClosingIdentifier;

                report.RPClosingStartDate = ((SBRDurationPeriod)sbrRPClosingContext.Period).DurationStartAsDateObject;
                report.RPClosingEndDate = ((SBRDurationPeriod)sbrRPClosingContext.Period).DurationEndAsDateObject;

                sbrPopulatedRPClosingElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPClosingContexts.Count(); i++)
                {
                    sbrRPClosingContext = sbrRPClosingContexts.ElementAt(i);

                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPClosingContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPClosingElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPClosingElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPClosingElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement });
                            }
                        }
                    }

                    if (sbrPopulatedRPClosingElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Tax.Losses.CarriedForward.LaterIncomeYearsTotal.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR85 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPClosingElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Assets.RelatedParties.InHouse.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR350 = ToBoolean(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPClosingElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Equity.MemberAccountBalance.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR137 = decimal.Parse(currentValues[0].Value);
                    }
                }
            }
            #endregion End of Context sbrRPClosingContext

            #region sbrRPClosingInstantContext
            IEnumerable<SBRContext> sbrRPClosingInstantContexts = null;
            SBRContext sbrRPClosingInstantContext = null;

            sbrRPClosingInstantContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.INSTANT &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty")) &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}OpeningAndClosingBalanceDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Closing"))))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPClosingInstantElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPClosingInstantContexts != null && sbrRPClosingInstantContexts.Count() > 0)
            {
                sbrRPClosingInstantContext = sbrRPClosingInstantContexts.First();

                report.RPClosingInstantId = sbrRPClosingInstantContext.Id;
                report.RPClosingInstantIdentifier = sbrRPClosingInstantContext.EntityIdentifier;
                report.RPClosingInstantIdentifierScheme = sbrRPClosingInstantContext.EntityScheme;

                report.RPClosingInstantCount = sbrRPClosingInstantContexts.Count();
                report.RPClosingInstantExists = true;
                report.RPClosingInstantOccurrenceIndex = Array.IndexOf(contextArray, sbrRPClosingInstantContext) + 1;
                report.RPClosingInstantLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPClosingInstantContexts.Last()) + 1;


                if (report.RPClosingInstantIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPClosingInstantIdentifierTFN = report.RPClosingInstantIdentifier;

                report.RPClosingInstantInstantDate = ((SBRInstantPeriod)sbrRPClosingInstantContext.Period).InstantAsDateObject;

                sbrPopulatedRPClosingInstantElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPClosingInstantContexts.Count(); i++)
                {
                    sbrRPClosingInstantContext = sbrRPClosingInstantContexts.ElementAt(i);

                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPClosingInstantContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPClosingInstantElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPClosingInstantElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPClosingInstantElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement });
                            }
                        }
                    }

                    if (sbrPopulatedRPClosingInstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Assets.RelatedParties.InHouse.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR351 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPClosingInstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Liabilities.BorrowingsOther.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR383 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPClosingInstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Equity.NetAssetsNotAllocatedToMembers.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR138 = decimal.Parse(currentValues[0].Value);
                    }
                }
            }
            #endregion End of Context sbrRPClosingInstantContext

            #region sbrRPTOFAContext
            IEnumerable<SBRContext> sbrRPTOFAContexts = null;
            SBRContext sbrRPTOFAContext = null;

            sbrRPTOFAContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty")) &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}FinancialArrangementTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}TOFA"))))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPTOFAElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPTOFAContexts != null && sbrRPTOFAContexts.Count() > 0)
            {
                sbrRPTOFAContext = sbrRPTOFAContexts.First();

                report.RPTOFAId = sbrRPTOFAContext.Id;
                report.RPTOFAIdentifier = sbrRPTOFAContext.EntityIdentifier;
                report.RPTOFAIdentifierScheme = sbrRPTOFAContext.EntityScheme;

                report.RPTOFACount = sbrRPTOFAContexts.Count();
                report.RPTOFAExists = true;
                report.RPTOFAOccurrenceIndex = Array.IndexOf(contextArray, sbrRPTOFAContext) + 1;
                report.RPTOFALastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPTOFAContexts.Last()) + 1;


                if (report.RPTOFAIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPTOFAIdentifierTFN = report.RPTOFAIdentifier;

                report.RPTOFAStartDate = ((SBRDurationPeriod)sbrRPTOFAContext.Period).DurationStartAsDateObject;
                report.RPTOFAEndDate = ((SBRDurationPeriod)sbrRPTOFAContext.Period).DurationEndAsDateObject;

                sbrPopulatedRPTOFAElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPTOFAContexts.Count(); i++)
                {
                    sbrRPTOFAContext = sbrRPTOFAContexts.ElementAt(i);

                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPTOFAContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPTOFAElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPTOFAElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPTOFAElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement });
                            }
                        }
                    }

                    if (sbrPopulatedRPTOFAElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Income.GainsTotal.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR228 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPTOFAElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Expense.LossesTotal.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR230 = decimal.Parse(currentValues[0].Value);
                    }
                }
            }
            #endregion End of Context sbrRPTOFAContext

            #region sbrRPDeductibleContext
            IEnumerable<SBRContext> sbrRPDeductibleContexts = null;
            SBRContext sbrRPDeductibleContext = null;

            sbrRPDeductibleContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty")) &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}TaxDeductibleStatusDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Deductible"))))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPDeductibleElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPDeductibleContexts != null && sbrRPDeductibleContexts.Count() > 0)
            {
                sbrRPDeductibleContext = sbrRPDeductibleContexts.First();

                report.RPDeductibleId = sbrRPDeductibleContext.Id;
                report.RPDeductibleIdentifier = sbrRPDeductibleContext.EntityIdentifier;
                report.RPDeductibleIdentifierScheme = sbrRPDeductibleContext.EntityScheme;

                report.RPDeductibleCount = sbrRPDeductibleContexts.Count();
                report.RPDeductibleExists = true;
                report.RPDeductibleOccurrenceIndex = Array.IndexOf(contextArray, sbrRPDeductibleContext) + 1;
                report.RPDeductibleLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPDeductibleContexts.Last()) + 1;


                if (report.RPDeductibleIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPDeductibleIdentifierTFN = report.RPDeductibleIdentifier;

                report.RPDeductibleStartDate = ((SBRDurationPeriod)sbrRPDeductibleContext.Period).DurationStartAsDateObject;
                report.RPDeductibleEndDate = ((SBRDurationPeriod)sbrRPDeductibleContext.Period).DurationEndAsDateObject;

                sbrPopulatedRPDeductibleElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPDeductibleContexts.Count(); i++)
                {
                    sbrRPDeductibleContext = sbrRPDeductibleContexts.ElementAt(i);

                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPDeductibleContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPDeductibleElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPDeductibleElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPDeductibleElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement });
                            }
                        }
                    }

                    if (sbrPopulatedRPDeductibleElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Expense.CapitalWorks.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR62 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPDeductibleElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Expense.DepreciationAndAmortisation.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR63 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPDeductibleElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Expense.PremiumsDeathOrDisability.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR64 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPDeductibleElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Expense.Operating.ApprovedAuditorFee.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR66 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPDeductibleElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Expense.Investment.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR67 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPDeductibleElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Expense.Operating.GeneralManagementAndAdministrationFees.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR302 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPDeductibleElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Expense.ForestryManagedInvestmentScheme.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR69 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPDeductibleElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Expense.Other.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR70 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPDeductibleElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Expense.Other.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR71 = currentValues[0].Value;
                    }

                    if (sbrPopulatedRPDeductibleElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}IncomeTax.Deduction.TaxLossesDeducted.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR72 = decimal.Parse(currentValues[0].Value);
                    }
                }
            }
            #endregion End of Context sbrRPDeductibleContext

            #region sbrRPNonDeductibleContext
            IEnumerable<SBRContext> sbrRPNonDeductibleContexts = null;
            SBRContext sbrRPNonDeductibleContext = null;

            sbrRPNonDeductibleContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty")) &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}TaxDeductibleStatusDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}NonDeductible"))))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPNonDeductibleElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPNonDeductibleContexts != null && sbrRPNonDeductibleContexts.Count() > 0)
            {
                sbrRPNonDeductibleContext = sbrRPNonDeductibleContexts.First();

                report.RPNonDeductibleId = sbrRPNonDeductibleContext.Id;
                report.RPNonDeductibleIdentifier = sbrRPNonDeductibleContext.EntityIdentifier;
                report.RPNonDeductibleIdentifierScheme = sbrRPNonDeductibleContext.EntityScheme;

                report.RPNonDeductibleCount = sbrRPNonDeductibleContexts.Count();
                report.RPNonDeductibleExists = true;
                report.RPNonDeductibleOccurrenceIndex = Array.IndexOf(contextArray, sbrRPNonDeductibleContext) + 1;
                report.RPNonDeductibleLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPNonDeductibleContexts.Last()) + 1;


                if (report.RPNonDeductibleIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPNonDeductibleIdentifierTFN = report.RPNonDeductibleIdentifier;

                report.RPNonDeductibleStartDate = ((SBRDurationPeriod)sbrRPNonDeductibleContext.Period).DurationStartAsDateObject;
                report.RPNonDeductibleEndDate = ((SBRDurationPeriod)sbrRPNonDeductibleContext.Period).DurationEndAsDateObject;

                sbrPopulatedRPNonDeductibleElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPNonDeductibleContexts.Count(); i++)
                {
                    sbrRPNonDeductibleContext = sbrRPNonDeductibleContexts.ElementAt(i);

                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPNonDeductibleContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPNonDeductibleElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPNonDeductibleElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPNonDeductibleElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement });
                            }
                        }
                    }

                    if (sbrPopulatedRPNonDeductibleElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Expense.CapitalWorks.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR331 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPNonDeductibleElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Expense.DepreciationAndAmortisation.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR332 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPNonDeductibleElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Expense.PremiumsDeathOrDisability.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR353 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPNonDeductibleElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Expense.Operating.ApprovedAuditorFee.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR333 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPNonDeductibleElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Expense.Investment.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR334 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPNonDeductibleElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Expense.Operating.GeneralManagementAndAdministrationFees.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR349 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPNonDeductibleElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Expense.ForestryManagedInvestmentScheme.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR335 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPNonDeductibleElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Expense.Other.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR336 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPNonDeductibleElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Expense.Other.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR337 = currentValues[0].Value;
                    }
                }
            }
            #endregion End of Context sbrRPNonDeductibleContext

            #region sbrRPClosingJAUSInstantContext
            IEnumerable<SBRContext> sbrRPClosingJAUSInstantContexts = null;
            SBRContext sbrRPClosingJAUSInstantContext = null;

            sbrRPClosingJAUSInstantContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.INSTANT &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 3 &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty")) &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}OpeningAndClosingBalanceDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Closing")) &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}InternationalJurisdictionDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Australian")))))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPClosingJAUSInstantElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPClosingJAUSInstantContexts != null && sbrRPClosingJAUSInstantContexts.Count() > 0)
            {
                sbrRPClosingJAUSInstantContext = sbrRPClosingJAUSInstantContexts.First();

                report.RPClosingJAUSInstantId = sbrRPClosingJAUSInstantContext.Id;
                report.RPClosingJAUSInstantIdentifier = sbrRPClosingJAUSInstantContext.EntityIdentifier;
                report.RPClosingJAUSInstantIdentifierScheme = sbrRPClosingJAUSInstantContext.EntityScheme;

                report.RPClosingJAUSInstantCount = sbrRPClosingJAUSInstantContexts.Count();
                report.RPClosingJAUSInstantExists = true;
                report.RPClosingJAUSInstantOccurrenceIndex = Array.IndexOf(contextArray, sbrRPClosingJAUSInstantContext) + 1;
                report.RPClosingJAUSInstantLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPClosingJAUSInstantContexts.Last()) + 1;


                if (report.RPClosingJAUSInstantIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPClosingJAUSInstantIdentifierTFN = report.RPClosingJAUSInstantIdentifier;

                report.RPClosingJAUSInstantInstantDate = ((SBRInstantPeriod)sbrRPClosingJAUSInstantContext.Period).InstantAsDateObject;

                sbrPopulatedRPClosingJAUSInstantElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPClosingJAUSInstantContexts.Count(); i++)
                {
                    sbrRPClosingJAUSInstantContext = sbrRPClosingJAUSInstantContexts.ElementAt(i);

                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPClosingJAUSInstantContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPClosingJAUSInstantElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPClosingJAUSInstantElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPClosingJAUSInstantElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement });
                            }
                        }
                    }

                    if (sbrPopulatedRPClosingJAUSInstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Assets.Investment.SecuritiesAndOrInsurancePoliciesHeldInLifeCompanies.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR119 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPClosingJAUSInstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Assets.ManagedInvestments.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR120 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPClosingJAUSInstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Assets.CashAndLiquidAssets.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR121 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPClosingJAUSInstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Assets.DebtSecurities.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR122 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPClosingJAUSInstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Assets.LoansAndReceivables.Held.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR123 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPClosingJAUSInstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Assets.Other.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR130 = decimal.Parse(currentValues[0].Value);
                    }
                }
            }
            #endregion End of Context sbrRPClosingJAUSInstantContext

            #region sbrRPClosingJFOREIGNInstantContext
            IEnumerable<SBRContext> sbrRPClosingJFOREIGNInstantContexts = null;
            SBRContext sbrRPClosingJFOREIGNInstantContext = null;

            sbrRPClosingJFOREIGNInstantContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.INSTANT &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 3 &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty")) &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}OpeningAndClosingBalanceDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Closing")) &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}InternationalJurisdictionDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Foreign")))))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPClosingJFOREIGNInstantElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPClosingJFOREIGNInstantContexts != null && sbrRPClosingJFOREIGNInstantContexts.Count() > 0)
            {
                sbrRPClosingJFOREIGNInstantContext = sbrRPClosingJFOREIGNInstantContexts.First();

                report.RPClosingJFOREIGNInstantId = sbrRPClosingJFOREIGNInstantContext.Id;
                report.RPClosingJFOREIGNInstantIdentifier = sbrRPClosingJFOREIGNInstantContext.EntityIdentifier;
                report.RPClosingJFOREIGNInstantIdentifierScheme = sbrRPClosingJFOREIGNInstantContext.EntityScheme;

                report.RPClosingJFOREIGNInstantCount = sbrRPClosingJFOREIGNInstantContexts.Count();
                report.RPClosingJFOREIGNInstantExists = true;
                report.RPClosingJFOREIGNInstantOccurrenceIndex = Array.IndexOf(contextArray, sbrRPClosingJFOREIGNInstantContext) + 1;
                report.RPClosingJFOREIGNInstantLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPClosingJFOREIGNInstantContexts.Last()) + 1;


                if (report.RPClosingJFOREIGNInstantIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPClosingJFOREIGNInstantIdentifierTFN = report.RPClosingJFOREIGNInstantIdentifier;

                report.RPClosingJFOREIGNInstantInstantDate = ((SBRInstantPeriod)sbrRPClosingJFOREIGNInstantContext.Period).InstantAsDateObject;

                sbrPopulatedRPClosingJFOREIGNInstantElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPClosingJFOREIGNInstantContexts.Count(); i++)
                {
                    sbrRPClosingJFOREIGNInstantContext = sbrRPClosingJFOREIGNInstantContexts.ElementAt(i);

                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPClosingJFOREIGNInstantContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPClosingJFOREIGNInstantElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPClosingJFOREIGNInstantElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPClosingJFOREIGNInstantElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement });
                            }
                        }
                    }

                    if (sbrPopulatedRPClosingJFOREIGNInstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Assets.Investment.EquitySecurities.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR131 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPClosingJFOREIGNInstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Assets.ManagedInvestments.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR134 = decimal.Parse(currentValues[0].Value);
                    }

                    if (sbrPopulatedRPClosingJFOREIGNInstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Assets.Other.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR135 = decimal.Parse(currentValues[0].Value);
                    }
                }
            }
            #endregion End of Context sbrRPClosingJFOREIGNInstantContext

            #region sbrRPY0PlusClosingInstantContext
            IEnumerable<SBRContext> sbrRPY0PlusClosingInstantContexts = null;
            SBRContext sbrRPY0PlusClosingInstantContext = null;

            sbrRPY0PlusClosingInstantContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.INSTANT &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 3 &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty")) &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}RelativePeriodDurationDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Y0Plus")) &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}OpeningAndClosingBalanceDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Closing")))))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPY0PlusClosingInstantElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPY0PlusClosingInstantContexts != null && sbrRPY0PlusClosingInstantContexts.Count() > 0)
            {
                sbrRPY0PlusClosingInstantContext = sbrRPY0PlusClosingInstantContexts.First();

                report.RPY0PlusClosingInstantId = sbrRPY0PlusClosingInstantContext.Id;
                report.RPY0PlusClosingInstantIdentifier = sbrRPY0PlusClosingInstantContext.EntityIdentifier;
                report.RPY0PlusClosingInstantIdentifierScheme = sbrRPY0PlusClosingInstantContext.EntityScheme;

                report.RPY0PlusClosingInstantCount = sbrRPY0PlusClosingInstantContexts.Count();
                report.RPY0PlusClosingInstantExists = true;
                report.RPY0PlusClosingInstantOccurrenceIndex = Array.IndexOf(contextArray, sbrRPY0PlusClosingInstantContext) + 1;
                report.RPY0PlusClosingInstantLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPY0PlusClosingInstantContexts.Last()) + 1;


                if (report.RPY0PlusClosingInstantIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPY0PlusClosingInstantIdentifierTFN = report.RPY0PlusClosingInstantIdentifier;

                report.RPY0PlusClosingInstantInstantDate = ((SBRInstantPeriod)sbrRPY0PlusClosingInstantContext.Period).InstantAsDateObject;

                sbrPopulatedRPY0PlusClosingInstantElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPY0PlusClosingInstantContexts.Count(); i++)
                {
                    sbrRPY0PlusClosingInstantContext = sbrRPY0PlusClosingInstantContexts.ElementAt(i);

                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPY0PlusClosingInstantContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPY0PlusClosingInstantElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPY0PlusClosingInstantElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPY0PlusClosingInstantElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement });
                            }
                        }
                    }

                    if (sbrPopulatedRPY0PlusClosingInstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Capital.Losses.CarriedForward.Net.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR380 = decimal.Parse(currentValues[0].Value);
                    }
                }
            }
            #endregion End of Context sbrRPY0PlusClosingInstantContext

            #region sbrRPDeductibleJAUSContext
            IEnumerable<SBRContext> sbrRPDeductibleJAUSContexts = null;
            SBRContext sbrRPDeductibleJAUSContext = null;

            sbrRPDeductibleJAUSContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 3 &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty")) &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}TaxDeductibleStatusDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Deductible")) &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}InternationalJurisdictionDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Australian")))))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPDeductibleJAUSElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPDeductibleJAUSContexts != null && sbrRPDeductibleJAUSContexts.Count() > 0)
            {
                sbrRPDeductibleJAUSContext = sbrRPDeductibleJAUSContexts.First();

                report.RPDeductibleJAUSId = sbrRPDeductibleJAUSContext.Id;
                report.RPDeductibleJAUSIdentifier = sbrRPDeductibleJAUSContext.EntityIdentifier;
                report.RPDeductibleJAUSIdentifierScheme = sbrRPDeductibleJAUSContext.EntityScheme;

                report.RPDeductibleJAUSCount = sbrRPDeductibleJAUSContexts.Count();
                report.RPDeductibleJAUSExists = true;
                report.RPDeductibleJAUSOccurrenceIndex = Array.IndexOf(contextArray, sbrRPDeductibleJAUSContext) + 1;
                report.RPDeductibleJAUSLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPDeductibleJAUSContexts.Last()) + 1;


                if (report.RPDeductibleJAUSIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPDeductibleJAUSIdentifierTFN = report.RPDeductibleJAUSIdentifier;

                report.RPDeductibleJAUSStartDate = ((SBRDurationPeriod)sbrRPDeductibleJAUSContext.Period).DurationStartAsDateObject;
                report.RPDeductibleJAUSEndDate = ((SBRDurationPeriod)sbrRPDeductibleJAUSContext.Period).DurationEndAsDateObject;

                sbrPopulatedRPDeductibleJAUSElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPDeductibleJAUSContexts.Count(); i++)
                {
                    sbrRPDeductibleJAUSContext = sbrRPDeductibleJAUSContexts.ElementAt(i);

                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPDeductibleJAUSContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPDeductibleJAUSElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPDeductibleJAUSElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPDeductibleJAUSElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement });
                            }
                        }
                    }

                    if (sbrPopulatedRPDeductibleJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Expense.Interest.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR60 = decimal.Parse(currentValues[0].Value);
                    }
                }
            }
            #endregion End of Context sbrRPDeductibleJAUSContext

            #region sbrRPDeductibleJFOREIGNContext
            IEnumerable<SBRContext> sbrRPDeductibleJFOREIGNContexts = null;
            SBRContext sbrRPDeductibleJFOREIGNContext = null;

            sbrRPDeductibleJFOREIGNContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 3 &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty")) &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}TaxDeductibleStatusDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Deductible")) &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}InternationalJurisdictionDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Foreign")))))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPDeductibleJFOREIGNElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPDeductibleJFOREIGNContexts != null && sbrRPDeductibleJFOREIGNContexts.Count() > 0)
            {
                sbrRPDeductibleJFOREIGNContext = sbrRPDeductibleJFOREIGNContexts.First();

                report.RPDeductibleJFOREIGNId = sbrRPDeductibleJFOREIGNContext.Id;
                report.RPDeductibleJFOREIGNIdentifier = sbrRPDeductibleJFOREIGNContext.EntityIdentifier;
                report.RPDeductibleJFOREIGNIdentifierScheme = sbrRPDeductibleJFOREIGNContext.EntityScheme;

                report.RPDeductibleJFOREIGNCount = sbrRPDeductibleJFOREIGNContexts.Count();
                report.RPDeductibleJFOREIGNExists = true;
                report.RPDeductibleJFOREIGNOccurrenceIndex = Array.IndexOf(contextArray, sbrRPDeductibleJFOREIGNContext) + 1;
                report.RPDeductibleJFOREIGNLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPDeductibleJFOREIGNContexts.Last()) + 1;


                if (report.RPDeductibleJFOREIGNIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPDeductibleJFOREIGNIdentifierTFN = report.RPDeductibleJFOREIGNIdentifier;

                report.RPDeductibleJFOREIGNStartDate = ((SBRDurationPeriod)sbrRPDeductibleJFOREIGNContext.Period).DurationStartAsDateObject;
                report.RPDeductibleJFOREIGNEndDate = ((SBRDurationPeriod)sbrRPDeductibleJFOREIGNContext.Period).DurationEndAsDateObject;

                sbrPopulatedRPDeductibleJFOREIGNElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPDeductibleJFOREIGNContexts.Count(); i++)
                {
                    sbrRPDeductibleJFOREIGNContext = sbrRPDeductibleJFOREIGNContexts.ElementAt(i);

                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPDeductibleJFOREIGNContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPDeductibleJFOREIGNElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPDeductibleJFOREIGNElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPDeductibleJFOREIGNElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement });
                            }
                        }
                    }

                    if (sbrPopulatedRPDeductibleJFOREIGNElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Expense.Interest.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR61 = decimal.Parse(currentValues[0].Value);
                    }
                }
            }
            #endregion End of Context sbrRPDeductibleJFOREIGNContext

            #region sbrRPNonDeductibleJAUSContext
            IEnumerable<SBRContext> sbrRPNonDeductibleJAUSContexts = null;
            SBRContext sbrRPNonDeductibleJAUSContext = null;

            sbrRPNonDeductibleJAUSContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 3 &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty")) &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}TaxDeductibleStatusDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}NonDeductible")) &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}InternationalJurisdictionDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Australian")))))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPNonDeductibleJAUSElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPNonDeductibleJAUSContexts != null && sbrRPNonDeductibleJAUSContexts.Count() > 0)
            {
                sbrRPNonDeductibleJAUSContext = sbrRPNonDeductibleJAUSContexts.First();

                report.RPNonDeductibleJAUSId = sbrRPNonDeductibleJAUSContext.Id;
                report.RPNonDeductibleJAUSIdentifier = sbrRPNonDeductibleJAUSContext.EntityIdentifier;
                report.RPNonDeductibleJAUSIdentifierScheme = sbrRPNonDeductibleJAUSContext.EntityScheme;

                report.RPNonDeductibleJAUSCount = sbrRPNonDeductibleJAUSContexts.Count();
                report.RPNonDeductibleJAUSExists = true;
                report.RPNonDeductibleJAUSOccurrenceIndex = Array.IndexOf(contextArray, sbrRPNonDeductibleJAUSContext) + 1;
                report.RPNonDeductibleJAUSLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPNonDeductibleJAUSContexts.Last()) + 1;


                if (report.RPNonDeductibleJAUSIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPNonDeductibleJAUSIdentifierTFN = report.RPNonDeductibleJAUSIdentifier;

                report.RPNonDeductibleJAUSStartDate = ((SBRDurationPeriod)sbrRPNonDeductibleJAUSContext.Period).DurationStartAsDateObject;
                report.RPNonDeductibleJAUSEndDate = ((SBRDurationPeriod)sbrRPNonDeductibleJAUSContext.Period).DurationEndAsDateObject;

                sbrPopulatedRPNonDeductibleJAUSElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPNonDeductibleJAUSContexts.Count(); i++)
                {
                    sbrRPNonDeductibleJAUSContext = sbrRPNonDeductibleJAUSContexts.ElementAt(i);

                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPNonDeductibleJAUSContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPNonDeductibleJAUSElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPNonDeductibleJAUSElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPNonDeductibleJAUSElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement });
                            }
                        }
                    }

                    if (sbrPopulatedRPNonDeductibleJAUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Expense.Interest.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR329 = decimal.Parse(currentValues[0].Value);
                    }
                }
            }
            #endregion End of Context sbrRPNonDeductibleJAUSContext

            #region sbrRPNonDeductibleJFOREIGNContext
            IEnumerable<SBRContext> sbrRPNonDeductibleJFOREIGNContexts = null;
            SBRContext sbrRPNonDeductibleJFOREIGNContext = null;

            sbrRPNonDeductibleJFOREIGNContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 3 &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty")) &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}TaxDeductibleStatusDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}NonDeductible")) &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}InternationalJurisdictionDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Foreign")))))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPNonDeductibleJFOREIGNElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPNonDeductibleJFOREIGNContexts != null && sbrRPNonDeductibleJFOREIGNContexts.Count() > 0)
            {
                sbrRPNonDeductibleJFOREIGNContext = sbrRPNonDeductibleJFOREIGNContexts.First();

                report.RPNonDeductibleJFOREIGNId = sbrRPNonDeductibleJFOREIGNContext.Id;
                report.RPNonDeductibleJFOREIGNIdentifier = sbrRPNonDeductibleJFOREIGNContext.EntityIdentifier;
                report.RPNonDeductibleJFOREIGNIdentifierScheme = sbrRPNonDeductibleJFOREIGNContext.EntityScheme;

                report.RPNonDeductibleJFOREIGNCount = sbrRPNonDeductibleJFOREIGNContexts.Count();
                report.RPNonDeductibleJFOREIGNExists = true;
                report.RPNonDeductibleJFOREIGNOccurrenceIndex = Array.IndexOf(contextArray, sbrRPNonDeductibleJFOREIGNContext) + 1;
                report.RPNonDeductibleJFOREIGNLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPNonDeductibleJFOREIGNContexts.Last()) + 1;


                if (report.RPNonDeductibleJFOREIGNIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPNonDeductibleJFOREIGNIdentifierTFN = report.RPNonDeductibleJFOREIGNIdentifier;

                report.RPNonDeductibleJFOREIGNStartDate = ((SBRDurationPeriod)sbrRPNonDeductibleJFOREIGNContext.Period).DurationStartAsDateObject;
                report.RPNonDeductibleJFOREIGNEndDate = ((SBRDurationPeriod)sbrRPNonDeductibleJFOREIGNContext.Period).DurationEndAsDateObject;

                sbrPopulatedRPNonDeductibleJFOREIGNElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPNonDeductibleJFOREIGNContexts.Count(); i++)
                {
                    sbrRPNonDeductibleJFOREIGNContext = sbrRPNonDeductibleJFOREIGNContexts.ElementAt(i);

                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPNonDeductibleJFOREIGNContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPNonDeductibleJFOREIGNElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPNonDeductibleJFOREIGNElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPNonDeductibleJFOREIGNElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement });
                            }
                        }
                    }

                    if (sbrPopulatedRPNonDeductibleJFOREIGNElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Expense.Interest.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR330 = decimal.Parse(currentValues[0].Value);
                    }
                }
            }
            #endregion End of Context sbrRPNonDeductibleJFOREIGNContext

            #region sbrRPJAUSAustNonResidentialRealPropertyContext
            IEnumerable<SBRContext> sbrRPJAUSAustNonResidentialRealPropertyContexts = null;
            SBRContext sbrRPJAUSAustNonResidentialRealPropertyContext = null;

            sbrRPJAUSAustNonResidentialRealPropertyContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 3 &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty")) &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}InternationalJurisdictionDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Australian")) &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}AssetTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}AustralianNonResidentialRealProperty")))))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPJAUSAustNonResidentialRealPropertyElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPJAUSAustNonResidentialRealPropertyContexts != null && sbrRPJAUSAustNonResidentialRealPropertyContexts.Count() > 0)
            {
                sbrRPJAUSAustNonResidentialRealPropertyContext = sbrRPJAUSAustNonResidentialRealPropertyContexts.First();

                report.RPJAUSAustNonResidentialRealPropertyId = sbrRPJAUSAustNonResidentialRealPropertyContext.Id;
                report.RPJAUSAustNonResidentialRealPropertyIdentifier = sbrRPJAUSAustNonResidentialRealPropertyContext.EntityIdentifier;
                report.RPJAUSAustNonResidentialRealPropertyIdentifierScheme = sbrRPJAUSAustNonResidentialRealPropertyContext.EntityScheme;

                report.RPJAUSAustNonResidentialRealPropertyCount = sbrRPJAUSAustNonResidentialRealPropertyContexts.Count();
                report.RPJAUSAustNonResidentialRealPropertyExists = true;
                report.RPJAUSAustNonResidentialRealPropertyOccurrenceIndex = Array.IndexOf(contextArray, sbrRPJAUSAustNonResidentialRealPropertyContext) + 1;
                report.RPJAUSAustNonResidentialRealPropertyLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPJAUSAustNonResidentialRealPropertyContexts.Last()) + 1;


                if (report.RPJAUSAustNonResidentialRealPropertyIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPJAUSAustNonResidentialRealPropertyIdentifierTFN = report.RPJAUSAustNonResidentialRealPropertyIdentifier;

                report.RPJAUSAustNonResidentialRealPropertyStartDate = ((SBRDurationPeriod)sbrRPJAUSAustNonResidentialRealPropertyContext.Period).DurationStartAsDateObject;
                report.RPJAUSAustNonResidentialRealPropertyEndDate = ((SBRDurationPeriod)sbrRPJAUSAustNonResidentialRealPropertyContext.Period).DurationEndAsDateObject;

                sbrPopulatedRPJAUSAustNonResidentialRealPropertyElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPJAUSAustNonResidentialRealPropertyContexts.Count(); i++)
                {
                    sbrRPJAUSAustNonResidentialRealPropertyContext = sbrRPJAUSAustNonResidentialRealPropertyContexts.ElementAt(i);

                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPJAUSAustNonResidentialRealPropertyContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPJAUSAustNonResidentialRealPropertyElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPJAUSAustNonResidentialRealPropertyElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPJAUSAustNonResidentialRealPropertyElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement });
                            }
                        }
                    }

                    if (sbrPopulatedRPJAUSAustNonResidentialRealPropertyElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Assets.LimitedRecourseBorrowing.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR344 = decimal.Parse(currentValues[0].Value);
                    }
                }
            }
            #endregion End of Context sbrRPJAUSAustNonResidentialRealPropertyContext

            #region sbrRPJAUSAustralianSharesContext
            IEnumerable<SBRContext> sbrRPJAUSAustralianSharesContexts = null;
            SBRContext sbrRPJAUSAustralianSharesContext = null;

            sbrRPJAUSAustralianSharesContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 3 &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty")) &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}InternationalJurisdictionDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Australian")) &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}AssetTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}AustralianShares")))))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPJAUSAustralianSharesElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPJAUSAustralianSharesContexts != null && sbrRPJAUSAustralianSharesContexts.Count() > 0)
            {
                sbrRPJAUSAustralianSharesContext = sbrRPJAUSAustralianSharesContexts.First();

                report.RPJAUSAustralianSharesId = sbrRPJAUSAustralianSharesContext.Id;
                report.RPJAUSAustralianSharesIdentifier = sbrRPJAUSAustralianSharesContext.EntityIdentifier;
                report.RPJAUSAustralianSharesIdentifierScheme = sbrRPJAUSAustralianSharesContext.EntityScheme;

                report.RPJAUSAustralianSharesCount = sbrRPJAUSAustralianSharesContexts.Count();
                report.RPJAUSAustralianSharesExists = true;
                report.RPJAUSAustralianSharesOccurrenceIndex = Array.IndexOf(contextArray, sbrRPJAUSAustralianSharesContext) + 1;
                report.RPJAUSAustralianSharesLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPJAUSAustralianSharesContexts.Last()) + 1;


                if (report.RPJAUSAustralianSharesIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPJAUSAustralianSharesIdentifierTFN = report.RPJAUSAustralianSharesIdentifier;

                report.RPJAUSAustralianSharesStartDate = ((SBRDurationPeriod)sbrRPJAUSAustralianSharesContext.Period).DurationStartAsDateObject;
                report.RPJAUSAustralianSharesEndDate = ((SBRDurationPeriod)sbrRPJAUSAustralianSharesContext.Period).DurationEndAsDateObject;

                sbrPopulatedRPJAUSAustralianSharesElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPJAUSAustralianSharesContexts.Count(); i++)
                {
                    sbrRPJAUSAustralianSharesContext = sbrRPJAUSAustralianSharesContexts.ElementAt(i);

                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPJAUSAustralianSharesContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPJAUSAustralianSharesElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPJAUSAustralianSharesElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPJAUSAustralianSharesElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement });
                            }
                        }
                    }

                    if (sbrPopulatedRPJAUSAustralianSharesElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Assets.LimitedRecourseBorrowing.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR346 = decimal.Parse(currentValues[0].Value);
                    }
                }
            }
            #endregion End of Context sbrRPJAUSAustralianSharesContext

            #region sbrRPJAUSAustResidentialRealPropertyContext
            IEnumerable<SBRContext> sbrRPJAUSAustResidentialRealPropertyContexts = null;
            SBRContext sbrRPJAUSAustResidentialRealPropertyContext = null;

            sbrRPJAUSAustResidentialRealPropertyContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 3 &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty")) &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}InternationalJurisdictionDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Australian")) &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}AssetTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}AustralianResidentialRealProperty")))))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPJAUSAustResidentialRealPropertyElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPJAUSAustResidentialRealPropertyContexts != null && sbrRPJAUSAustResidentialRealPropertyContexts.Count() > 0)
            {
                sbrRPJAUSAustResidentialRealPropertyContext = sbrRPJAUSAustResidentialRealPropertyContexts.First();

                report.RPJAUSAustResidentialRealPropertyId = sbrRPJAUSAustResidentialRealPropertyContext.Id;
                report.RPJAUSAustResidentialRealPropertyIdentifier = sbrRPJAUSAustResidentialRealPropertyContext.EntityIdentifier;
                report.RPJAUSAustResidentialRealPropertyIdentifierScheme = sbrRPJAUSAustResidentialRealPropertyContext.EntityScheme;

                report.RPJAUSAustResidentialRealPropertyCount = sbrRPJAUSAustResidentialRealPropertyContexts.Count();
                report.RPJAUSAustResidentialRealPropertyExists = true;
                report.RPJAUSAustResidentialRealPropertyOccurrenceIndex = Array.IndexOf(contextArray, sbrRPJAUSAustResidentialRealPropertyContext) + 1;
                report.RPJAUSAustResidentialRealPropertyLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPJAUSAustResidentialRealPropertyContexts.Last()) + 1;


                if (report.RPJAUSAustResidentialRealPropertyIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPJAUSAustResidentialRealPropertyIdentifierTFN = report.RPJAUSAustResidentialRealPropertyIdentifier;

                report.RPJAUSAustResidentialRealPropertyStartDate = ((SBRDurationPeriod)sbrRPJAUSAustResidentialRealPropertyContext.Period).DurationStartAsDateObject;
                report.RPJAUSAustResidentialRealPropertyEndDate = ((SBRDurationPeriod)sbrRPJAUSAustResidentialRealPropertyContext.Period).DurationEndAsDateObject;

                sbrPopulatedRPJAUSAustResidentialRealPropertyElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPJAUSAustResidentialRealPropertyContexts.Count(); i++)
                {
                    sbrRPJAUSAustResidentialRealPropertyContext = sbrRPJAUSAustResidentialRealPropertyContexts.ElementAt(i);

                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPJAUSAustResidentialRealPropertyContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPJAUSAustResidentialRealPropertyElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPJAUSAustResidentialRealPropertyElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPJAUSAustResidentialRealPropertyElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement });
                            }
                        }
                    }

                    if (sbrPopulatedRPJAUSAustResidentialRealPropertyElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Assets.LimitedRecourseBorrowing.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR322 = decimal.Parse(currentValues[0].Value);
                    }
                }
            }
            #endregion End of Context sbrRPJAUSAustResidentialRealPropertyContext

            #region sbrRPJAUSOtherContext
            IEnumerable<SBRContext> sbrRPJAUSOtherContexts = null;
            SBRContext sbrRPJAUSOtherContext = null;

            sbrRPJAUSOtherContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 3 &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty")) &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}InternationalJurisdictionDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Australian")) &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}AssetTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Other")))))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPJAUSOtherElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPJAUSOtherContexts != null && sbrRPJAUSOtherContexts.Count() > 0)
            {
                sbrRPJAUSOtherContext = sbrRPJAUSOtherContexts.First();

                report.RPJAUSOtherId = sbrRPJAUSOtherContext.Id;
                report.RPJAUSOtherIdentifier = sbrRPJAUSOtherContext.EntityIdentifier;
                report.RPJAUSOtherIdentifierScheme = sbrRPJAUSOtherContext.EntityScheme;

                report.RPJAUSOtherCount = sbrRPJAUSOtherContexts.Count();
                report.RPJAUSOtherExists = true;
                report.RPJAUSOtherOccurrenceIndex = Array.IndexOf(contextArray, sbrRPJAUSOtherContext) + 1;
                report.RPJAUSOtherLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPJAUSOtherContexts.Last()) + 1;


                if (report.RPJAUSOtherIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPJAUSOtherIdentifierTFN = report.RPJAUSOtherIdentifier;

                report.RPJAUSOtherStartDate = ((SBRDurationPeriod)sbrRPJAUSOtherContext.Period).DurationStartAsDateObject;
                report.RPJAUSOtherEndDate = ((SBRDurationPeriod)sbrRPJAUSOtherContext.Period).DurationEndAsDateObject;

                sbrPopulatedRPJAUSOtherElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPJAUSOtherContexts.Count(); i++)
                {
                    sbrRPJAUSOtherContext = sbrRPJAUSOtherContexts.ElementAt(i);

                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPJAUSOtherContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPJAUSOtherElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPJAUSOtherElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPJAUSOtherElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement });
                            }
                        }
                    }

                    if (sbrPopulatedRPJAUSOtherElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Assets.LimitedRecourseBorrowing.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR348 = decimal.Parse(currentValues[0].Value);
                    }
                }
            }
            #endregion End of Context sbrRPJAUSOtherContext

            #region sbrRPJAUSOtherSharesContext
            IEnumerable<SBRContext> sbrRPJAUSOtherSharesContexts = null;
            SBRContext sbrRPJAUSOtherSharesContext = null;

            sbrRPJAUSOtherSharesContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 3 &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty")) &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}InternationalJurisdictionDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Australian")) &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}AssetTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}OtherShares")))))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPJAUSOtherSharesElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPJAUSOtherSharesContexts != null && sbrRPJAUSOtherSharesContexts.Count() > 0)
            {
                sbrRPJAUSOtherSharesContext = sbrRPJAUSOtherSharesContexts.First();

                report.RPJAUSOtherSharesId = sbrRPJAUSOtherSharesContext.Id;
                report.RPJAUSOtherSharesIdentifier = sbrRPJAUSOtherSharesContext.EntityIdentifier;
                report.RPJAUSOtherSharesIdentifierScheme = sbrRPJAUSOtherSharesContext.EntityScheme;

                report.RPJAUSOtherSharesCount = sbrRPJAUSOtherSharesContexts.Count();
                report.RPJAUSOtherSharesExists = true;
                report.RPJAUSOtherSharesOccurrenceIndex = Array.IndexOf(contextArray, sbrRPJAUSOtherSharesContext) + 1;
                report.RPJAUSOtherSharesLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPJAUSOtherSharesContexts.Last()) + 1;


                if (report.RPJAUSOtherSharesIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPJAUSOtherSharesIdentifierTFN = report.RPJAUSOtherSharesIdentifier;

                report.RPJAUSOtherSharesStartDate = ((SBRDurationPeriod)sbrRPJAUSOtherSharesContext.Period).DurationStartAsDateObject;
                report.RPJAUSOtherSharesEndDate = ((SBRDurationPeriod)sbrRPJAUSOtherSharesContext.Period).DurationEndAsDateObject;

                sbrPopulatedRPJAUSOtherSharesElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPJAUSOtherSharesContexts.Count(); i++)
                {
                    sbrRPJAUSOtherSharesContext = sbrRPJAUSOtherSharesContexts.ElementAt(i);

                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPJAUSOtherSharesContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPJAUSOtherSharesElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPJAUSOtherSharesElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPJAUSOtherSharesElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement });
                            }
                        }
                    }

                    if (sbrPopulatedRPJAUSOtherSharesElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Assets.LimitedRecourseBorrowing.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR347 = decimal.Parse(currentValues[0].Value);
                    }
                }
            }
            #endregion End of Context sbrRPJAUSOtherSharesContext

            #region sbrRPJAUSOverseasRealPropertyContext
            IEnumerable<SBRContext> sbrRPJAUSOverseasRealPropertyContexts = null;
            SBRContext sbrRPJAUSOverseasRealPropertyContext = null;

            sbrRPJAUSOverseasRealPropertyContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 3 &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty")) &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}InternationalJurisdictionDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Australian")) &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}AssetTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}OverseasRealProperty")))))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPJAUSOverseasRealPropertyElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPJAUSOverseasRealPropertyContexts != null && sbrRPJAUSOverseasRealPropertyContexts.Count() > 0)
            {
                sbrRPJAUSOverseasRealPropertyContext = sbrRPJAUSOverseasRealPropertyContexts.First();

                report.RPJAUSOverseasRealPropertyId = sbrRPJAUSOverseasRealPropertyContext.Id;
                report.RPJAUSOverseasRealPropertyIdentifier = sbrRPJAUSOverseasRealPropertyContext.EntityIdentifier;
                report.RPJAUSOverseasRealPropertyIdentifierScheme = sbrRPJAUSOverseasRealPropertyContext.EntityScheme;

                report.RPJAUSOverseasRealPropertyCount = sbrRPJAUSOverseasRealPropertyContexts.Count();
                report.RPJAUSOverseasRealPropertyExists = true;
                report.RPJAUSOverseasRealPropertyOccurrenceIndex = Array.IndexOf(contextArray, sbrRPJAUSOverseasRealPropertyContext) + 1;
                report.RPJAUSOverseasRealPropertyLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPJAUSOverseasRealPropertyContexts.Last()) + 1;


                if (report.RPJAUSOverseasRealPropertyIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPJAUSOverseasRealPropertyIdentifierTFN = report.RPJAUSOverseasRealPropertyIdentifier;

                report.RPJAUSOverseasRealPropertyStartDate = ((SBRDurationPeriod)sbrRPJAUSOverseasRealPropertyContext.Period).DurationStartAsDateObject;
                report.RPJAUSOverseasRealPropertyEndDate = ((SBRDurationPeriod)sbrRPJAUSOverseasRealPropertyContext.Period).DurationEndAsDateObject;

                sbrPopulatedRPJAUSOverseasRealPropertyElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPJAUSOverseasRealPropertyContexts.Count(); i++)
                {
                    sbrRPJAUSOverseasRealPropertyContext = sbrRPJAUSOverseasRealPropertyContexts.ElementAt(i);

                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPJAUSOverseasRealPropertyContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPJAUSOverseasRealPropertyElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPJAUSOverseasRealPropertyElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPJAUSOverseasRealPropertyElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement });
                            }
                        }
                    }

                    if (sbrPopulatedRPJAUSOverseasRealPropertyElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Assets.LimitedRecourseBorrowing.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR345 = decimal.Parse(currentValues[0].Value);
                    }
                }
            }
            #endregion End of Context sbrRPJAUSOverseasRealPropertyContext

            #region sbrRPIncomeTaxRefundContext
            IEnumerable<SBRContext> sbrRPIncomeTaxRefundContexts = null;
            SBRContext sbrRPIncomeTaxRefundContext = null;

            sbrRPIncomeTaxRefundContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty")) &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}FinancialAccountUsageTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}IncomeTaxRefund"))))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPIncomeTaxRefundElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPIncomeTaxRefundContexts != null && sbrRPIncomeTaxRefundContexts.Count() > 0)
            {
                sbrRPIncomeTaxRefundContext = sbrRPIncomeTaxRefundContexts.First();

                report.RPIncomeTaxRefundId = sbrRPIncomeTaxRefundContext.Id;
                report.RPIncomeTaxRefundIdentifier = sbrRPIncomeTaxRefundContext.EntityIdentifier;
                report.RPIncomeTaxRefundIdentifierScheme = sbrRPIncomeTaxRefundContext.EntityScheme;

                report.RPIncomeTaxRefundCount = sbrRPIncomeTaxRefundContexts.Count();
                report.RPIncomeTaxRefundExists = true;
                report.RPIncomeTaxRefundOccurrenceIndex = Array.IndexOf(contextArray, sbrRPIncomeTaxRefundContext) + 1;
                report.RPIncomeTaxRefundLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPIncomeTaxRefundContexts.Last()) + 1;


                if (report.RPIncomeTaxRefundIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPIncomeTaxRefundIdentifierTFN = report.RPIncomeTaxRefundIdentifier;

                report.RPIncomeTaxRefundStartDate = ((SBRDurationPeriod)sbrRPIncomeTaxRefundContext.Period).DurationStartAsDateObject;
                report.RPIncomeTaxRefundEndDate = ((SBRDurationPeriod)sbrRPIncomeTaxRefundContext.Period).DurationEndAsDateObject;

                sbrPopulatedRPIncomeTaxRefundElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPIncomeTaxRefundContexts.Count(); i++)
                {
                    sbrRPIncomeTaxRefundContext = sbrRPIncomeTaxRefundContexts.ElementAt(i);

                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPIncomeTaxRefundContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPIncomeTaxRefundElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPIncomeTaxRefundElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPIncomeTaxRefundElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement });
                            }
                        }
                    }

                    IEnumerable<SBRElement> sbrFinancialInstitutionAccountCollection;
                    if (populatedReportElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}FinancialInstitutionAccount", out currentValue))
                        sbrFinancialInstitutionAccountCollection = currentValue.Occurrences;
                    else
                        sbrFinancialInstitutionAccountCollection = null;

                    #region sbrFinancialInstitutionAccountCollection

                    if (sbrFinancialInstitutionAccountCollection != null)
                    {
                        sbrFinancialInstitutionAccountCollection =
                            sbrFinancialInstitutionAccountCollection.Where(
                                t =>
                                    t.ChildElements != null && t.ChildElements.Count > 0 &&
                                    sbrRPIncomeTaxRefundContexts.Contains(t.ChildElements[0].Context));
                    }


                    if (sbrFinancialInstitutionAccountCollection != null && sbrFinancialInstitutionAccountCollection.Count() > 0)
                    {
                        report.FinancialInstitutionAccountCollectionExists = true;
                        report.FinancialInstitutionAccountCollectionCount = sbrFinancialInstitutionAccountCollection.Count();
                        report.RpIncomeTaxRefundFinancialInstitutionAccountCollectionExists = true;
                        report.RpIncomeTaxRefundFinancialInstitutionAccountCollectionCount = sbrFinancialInstitutionAccountCollection.Count();
                        SBRElement sbrFinancialInstitutionAccount;
                        sbrFinancialInstitutionAccount = sbrFinancialInstitutionAccountCollection.ElementAt(0);

                        IDictionary<string, SBRElement> sbrFinancialInstitutionAccountTupleElementMap;
                        sbrFinancialInstitutionAccountTupleElementMap = sbrFinancialInstitutionAccount.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();

                        if (sbrFinancialInstitutionAccountTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}FinancialInstitutionAccount.BankStateBranch.Number", out currentValue))
                        {
                            if (!currentValue.IsNil) report.SMSFAR293 = currentValue.Value;
                        }

                        if (sbrFinancialInstitutionAccountTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}FinancialInstitutionAccount.FinancialInstitutionAccount.Number", out currentValue))
                        {
                            if (!currentValue.IsNil) report.SMSFAR294 = currentValue.Value;
                        }

                        if (sbrFinancialInstitutionAccountTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}FinancialInstitutionAccount.FinancialInstitutionAccountName.Text", out currentValue))
                        {
                            if (!currentValue.IsNil) report.SMSFAR295 = currentValue.Value;
                        }

                        if (sbrFinancialInstitutionAccountTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}FinancialInstitutionAccount.FinancialInstitutionBranchName.Text", out currentValue))
                        {
                            if (!currentValue.IsNil) report.SMSFAR296 = currentValue.Value;
                        }

                        if (sbrFinancialInstitutionAccountTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}FinancialInstitutionAccount.FinancialInstitutionName.Text", out currentValue))
                        {
                            if (!currentValue.IsNil) report.SMSFAR297 = currentValue.Value;
                        }
                    }
                    #endregion End of Tuple sbrFinancialInstitutionAccountCollection
                }
            }
            #endregion End of Context sbrRPIncomeTaxRefundContext

            #region sbrRPSuperannuationContext
            IEnumerable<SBRContext> sbrRPSuperannuationContexts = null;
            SBRContext sbrRPSuperannuationContext = null;

            sbrRPSuperannuationContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty")) &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}FinancialAccountUsageTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Superannuation"))))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPSuperannuationElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPSuperannuationContexts != null && sbrRPSuperannuationContexts.Count() > 0)
            {
                sbrRPSuperannuationContext = sbrRPSuperannuationContexts.First();

                report.RPSuperannuationId = sbrRPSuperannuationContext.Id;
                report.RPSuperannuationIdentifier = sbrRPSuperannuationContext.EntityIdentifier;
                report.RPSuperannuationIdentifierScheme = sbrRPSuperannuationContext.EntityScheme;

                report.RPSuperannuationCount = sbrRPSuperannuationContexts.Count();
                report.RPSuperannuationExists = true;
                report.RPSuperannuationOccurrenceIndex = Array.IndexOf(contextArray, sbrRPSuperannuationContext) + 1;
                report.RPSuperannuationLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPSuperannuationContexts.Last()) + 1;


                if (report.RPSuperannuationIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPSuperannuationIdentifierTFN = report.RPSuperannuationIdentifier;

                report.RPSuperannuationStartDate = ((SBRDurationPeriod)sbrRPSuperannuationContext.Period).DurationStartAsDateObject;
                report.RPSuperannuationEndDate = ((SBRDurationPeriod)sbrRPSuperannuationContext.Period).DurationEndAsDateObject;

                sbrPopulatedRPSuperannuationElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPSuperannuationContexts.Count(); i++)
                {
                    sbrRPSuperannuationContext = sbrRPSuperannuationContexts.ElementAt(i);

                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPSuperannuationContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPSuperannuationElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPSuperannuationElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPSuperannuationElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement });
                            }
                        }
                    }

                    IEnumerable<SBRElement> sbrFinancialInstitutionAccountCollection;
                    if (populatedReportElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}FinancialInstitutionAccount", out currentValue))
                        sbrFinancialInstitutionAccountCollection = currentValue.Occurrences;
                    else
                        sbrFinancialInstitutionAccountCollection = null;

                    #region sbrFinancialInstitutionAccountCollection

                    if (sbrFinancialInstitutionAccountCollection != null)
                    {
                        sbrFinancialInstitutionAccountCollection =
                            sbrFinancialInstitutionAccountCollection.Where(
                                t =>
                                    t.ChildElements != null && t.ChildElements.Count > 0 &&
                                    sbrRPSuperannuationContexts.Contains(t.ChildElements[0].Context));
                    }


                    if (sbrFinancialInstitutionAccountCollection != null && sbrFinancialInstitutionAccountCollection.Count() > 0)
                    {
                        report.RpSuperannuationFinancialInstitutionAccountCollectionExists = true;
                        report.RpSuperannuationFinancialInstitutionAccountCollectionCount = sbrFinancialInstitutionAccountCollection.Count();
                        SBRElement sbrFinancialInstitutionAccount;
                        sbrFinancialInstitutionAccount = sbrFinancialInstitutionAccountCollection.ElementAt(0);

                        IDictionary<string, SBRElement> sbrFinancialInstitutionAccountTupleElementMap;
                        sbrFinancialInstitutionAccountTupleElementMap = sbrFinancialInstitutionAccount.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();

                        if (sbrFinancialInstitutionAccountTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}FinancialInstitutionAccount.BankStateBranch.Number", out currentValue))
                        {
                            if (!currentValue.IsNil) report.SMSFAR364 = currentValue.Value;
                        }

                        if (sbrFinancialInstitutionAccountTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}FinancialInstitutionAccount.FinancialInstitutionAccount.Number", out currentValue))
                        {
                            if (!currentValue.IsNil) report.SMSFAR365 = currentValue.Value;
                        }

                        if (sbrFinancialInstitutionAccountTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}FinancialInstitutionAccount.FinancialInstitutionAccountName.Text", out currentValue))
                        {
                            if (!currentValue.IsNil) report.SMSFAR366 = currentValue.Value;
                        }

                        if (sbrFinancialInstitutionAccountTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}FinancialInstitutionAccount.FinancialInstitutionBranchName.Text", out currentValue))
                        {
                            if (!currentValue.IsNil) report.SMSFAR367 = currentValue.Value;
                        }

                        if (sbrFinancialInstitutionAccountTupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}FinancialInstitutionAccount.FinancialInstitutionName.Text", out currentValue))
                        {
                            if (!currentValue.IsNil) report.SMSFAR368 = currentValue.Value;
                        }
                    }
                    #endregion End of Tuple sbrFinancialInstitutionAccountCollection
                }
            }
            #endregion End of Context sbrRPSuperannuationContext

            #region sbrRPLimitedPartnersContext
            IEnumerable<SBRContext> sbrRPLimitedPartnersContexts = null;
            SBRContext sbrRPLimitedPartnersContext = null;

            sbrRPLimitedPartnersContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty")) &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}EarlyInvestorDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}LimitedPartners"))))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPLimitedPartnersElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPLimitedPartnersContexts != null && sbrRPLimitedPartnersContexts.Count() > 0)
            {
                sbrRPLimitedPartnersContext = sbrRPLimitedPartnersContexts.First();

                report.RPLimitedPartnersId = sbrRPLimitedPartnersContext.Id;
                report.RPLimitedPartnersIdentifier = sbrRPLimitedPartnersContext.EntityIdentifier;
                report.RPLimitedPartnersIdentifierScheme = sbrRPLimitedPartnersContext.EntityScheme;

                report.RPLimitedPartnersCount = sbrRPLimitedPartnersContexts.Count();
                report.RPLimitedPartnersExists = true;
                report.RPLimitedPartnersOccurrenceIndex = Array.IndexOf(contextArray, sbrRPLimitedPartnersContext) + 1;
                report.RPLimitedPartnersLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPLimitedPartnersContexts.Last()) + 1;


                if (report.RPLimitedPartnersIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPLimitedPartnersIdentifierTFN = report.RPLimitedPartnersIdentifier;

                report.RPLimitedPartnersStartDate = ((SBRDurationPeriod)sbrRPLimitedPartnersContext.Period).DurationStartAsDateObject;
                report.RPLimitedPartnersEndDate = ((SBRDurationPeriod)sbrRPLimitedPartnersContext.Period).DurationEndAsDateObject;

                sbrPopulatedRPLimitedPartnersElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPLimitedPartnersContexts.Count(); i++)
                {
                    sbrRPLimitedPartnersContext = sbrRPLimitedPartnersContexts.ElementAt(i);

                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPLimitedPartnersContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPLimitedPartnersElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPLimitedPartnersElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPLimitedPartnersElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement });
                            }
                        }
                    }

                    if (sbrPopulatedRPLimitedPartnersElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}TaxOffsetClaim.NonRefundableOther.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR387 = decimal.Parse(currentValues[0].Value);
                    }
                }
            }
            #endregion End of Context sbrRPLimitedPartnersContext

            #region sbrRPInvestorContext
            IEnumerable<SBRContext> sbrRPInvestorContexts = null;
            SBRContext sbrRPInvestorContext = null;

            sbrRPInvestorContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty")) &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}EarlyInvestorDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Investor"))))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPInvestorElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPInvestorContexts != null && sbrRPInvestorContexts.Count() > 0)
            {
                sbrRPInvestorContext = sbrRPInvestorContexts.First();

                report.RPInvestorId = sbrRPInvestorContext.Id;
                report.RPInvestorIdentifier = sbrRPInvestorContext.EntityIdentifier;
                report.RPInvestorIdentifierScheme = sbrRPInvestorContext.EntityScheme;

                report.RPInvestorCount = sbrRPInvestorContexts.Count();
                report.RPInvestorExists = true;
                report.RPInvestorOccurrenceIndex = Array.IndexOf(contextArray, sbrRPInvestorContext) + 1;
                report.RPInvestorLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPInvestorContexts.Last()) + 1;


                if (report.RPInvestorIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPInvestorIdentifierTFN = report.RPInvestorIdentifier;

                report.RPInvestorStartDate = ((SBRDurationPeriod)sbrRPInvestorContext.Period).DurationStartAsDateObject;
                report.RPInvestorEndDate = ((SBRDurationPeriod)sbrRPInvestorContext.Period).DurationEndAsDateObject;

                sbrPopulatedRPInvestorElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPInvestorContexts.Count(); i++)
                {
                    sbrRPInvestorContext = sbrRPInvestorContexts.ElementAt(i);

                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPInvestorContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPInvestorElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPInvestorElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPInvestorElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement });
                            }
                        }
                    }

                    if (sbrPopulatedRPInvestorElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}TaxOffsetClaim.NonRefundableOther.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR386 = decimal.Parse(currentValues[0].Value);
                    }
                }
            }
            #endregion End of Context sbrRPInvestorContext

            #region sbrRPLimitedPartnersYPreviousContext
            IEnumerable<SBRContext> sbrRPLimitedPartnersYPreviousContexts = null;
            SBRContext sbrRPLimitedPartnersYPreviousContext = null;

            sbrRPLimitedPartnersYPreviousContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 3 &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty")) &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}EarlyInvestorDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}LimitedPartners")) &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}RelativePeriodDurationDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}YPrevious")))))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPLimitedPartnersYPreviousElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPLimitedPartnersYPreviousContexts != null && sbrRPLimitedPartnersYPreviousContexts.Count() > 0)
            {
                sbrRPLimitedPartnersYPreviousContext = sbrRPLimitedPartnersYPreviousContexts.First();

                report.RPLimitedPartnersYPreviousId = sbrRPLimitedPartnersYPreviousContext.Id;
                report.RPLimitedPartnersYPreviousIdentifier = sbrRPLimitedPartnersYPreviousContext.EntityIdentifier;
                report.RPLimitedPartnersYPreviousIdentifierScheme = sbrRPLimitedPartnersYPreviousContext.EntityScheme;

                report.RPLimitedPartnersYPreviousCount = sbrRPLimitedPartnersYPreviousContexts.Count();
                report.RPLimitedPartnersYPreviousExists = true;
                report.RPLimitedPartnersYPreviousOccurrenceIndex = Array.IndexOf(contextArray, sbrRPLimitedPartnersYPreviousContext) + 1;
                report.RPLimitedPartnersYPreviousLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPLimitedPartnersYPreviousContexts.Last()) + 1;


                if (report.RPLimitedPartnersYPreviousIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPLimitedPartnersYPreviousIdentifierTFN = report.RPLimitedPartnersYPreviousIdentifier;

                report.RPLimitedPartnersYPreviousStartDate = ((SBRDurationPeriod)sbrRPLimitedPartnersYPreviousContext.Period).DurationStartAsDateObject;
                report.RPLimitedPartnersYPreviousEndDate = ((SBRDurationPeriod)sbrRPLimitedPartnersYPreviousContext.Period).DurationEndAsDateObject;

                sbrPopulatedRPLimitedPartnersYPreviousElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPLimitedPartnersYPreviousContexts.Count(); i++)
                {
                    sbrRPLimitedPartnersYPreviousContext = sbrRPLimitedPartnersYPreviousContexts.ElementAt(i);

                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPLimitedPartnersYPreviousContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPLimitedPartnersYPreviousElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPLimitedPartnersYPreviousElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPLimitedPartnersYPreviousElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement });
                            }
                        }
                    }

                    if (sbrPopulatedRPLimitedPartnersYPreviousElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}TaxOffsetClaim.NonRefundableOther.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR391 = decimal.Parse(currentValues[0].Value);
                    }
                }
            }
            #endregion End of Context sbrRPLimitedPartnersYPreviousContext

            #region sbrRPInvestorYPreviousContext
            IEnumerable<SBRContext> sbrRPInvestorYPreviousContexts = null;
            SBRContext sbrRPInvestorYPreviousContext = null;

            sbrRPInvestorYPreviousContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 3 &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty")) &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}EarlyInvestorDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Investor")) &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}RelativePeriodDurationDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}YPrevious")))))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPInvestorYPreviousElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPInvestorYPreviousContexts != null && sbrRPInvestorYPreviousContexts.Count() > 0)
            {
                sbrRPInvestorYPreviousContext = sbrRPInvestorYPreviousContexts.First();

                report.RPInvestorYPreviousId = sbrRPInvestorYPreviousContext.Id;
                report.RPInvestorYPreviousIdentifier = sbrRPInvestorYPreviousContext.EntityIdentifier;
                report.RPInvestorYPreviousIdentifierScheme = sbrRPInvestorYPreviousContext.EntityScheme;

                report.RPInvestorYPreviousCount = sbrRPInvestorYPreviousContexts.Count();
                report.RPInvestorYPreviousExists = true;
                report.RPInvestorYPreviousOccurrenceIndex = Array.IndexOf(contextArray, sbrRPInvestorYPreviousContext) + 1;
                report.RPInvestorYPreviousLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPInvestorYPreviousContexts.Last()) + 1;


                if (report.RPInvestorYPreviousIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPInvestorYPreviousIdentifierTFN = report.RPInvestorYPreviousIdentifier;

                report.RPInvestorYPreviousStartDate = ((SBRDurationPeriod)sbrRPInvestorYPreviousContext.Period).DurationStartAsDateObject;
                report.RPInvestorYPreviousEndDate = ((SBRDurationPeriod)sbrRPInvestorYPreviousContext.Period).DurationEndAsDateObject;

                sbrPopulatedRPInvestorYPreviousElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPInvestorYPreviousContexts.Count(); i++)
                {
                    sbrRPInvestorYPreviousContext = sbrRPInvestorYPreviousContexts.ElementAt(i);

                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPInvestorYPreviousContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPInvestorYPreviousElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPInvestorYPreviousElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPInvestorYPreviousElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement });
                            }
                        }
                    }

                    if (sbrPopulatedRPInvestorYPreviousElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}TaxOffsetClaim.NonRefundableOther.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.SMSFAR392 = decimal.Parse(currentValues[0].Value);
                    }
                }
            }
            #endregion End of Context sbrRPInvestorYPreviousContext

            #region sbrRPAccumulationContext
            IEnumerable<SBRContext> sbrRPAccumulationContexts = null;
            SBRContext sbrRPAccumulationContext = null;

            sbrRPAccumulationContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty")) &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}SuperannuationPhaseDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Accumulation"))))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPAccumulationElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPAccumulationContexts != null && sbrRPAccumulationContexts.Count() > 0)
            {
                sbrRPAccumulationContext = sbrRPAccumulationContexts.First();

                report.RPAccumulationId = sbrRPAccumulationContext.Id;
                report.RPAccumulationIdentifier = sbrRPAccumulationContext.EntityIdentifier;
                report.RPAccumulationIdentifierScheme = sbrRPAccumulationContext.EntityScheme;

                report.RPAccumulationCount = sbrRPAccumulationContexts.Count();
                report.RPAccumulationExists = true;
                report.RPAccumulationOccurrenceIndex = Array.IndexOf(contextArray, sbrRPAccumulationContext) + 1;
                report.RPAccumulationLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPAccumulationContexts.Last()) + 1;


                if (report.RPAccumulationIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPAccumulationIdentifierTFN = report.RPAccumulationIdentifier;

                report.RPAccumulationStartDate = ((SBRDurationPeriod)sbrRPAccumulationContext.Period).DurationStartAsDateObject;
                report.RPAccumulationEndDate = ((SBRDurationPeriod)sbrRPAccumulationContext.Period).DurationEndAsDateObject;

                sbrPopulatedRPAccumulationElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPAccumulationContexts.Count(); i++)
                {
                    sbrRPAccumulationContext = sbrRPAccumulationContexts.ElementAt(i);

                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPAccumulationContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPAccumulationElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPAccumulationElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPAccumulationElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement });
                            }
                        }
                    }
                }
            }
            #endregion End of Context sbrRPAccumulationContext

            #region sbrRPRetirementContext
            IEnumerable<SBRContext> sbrRPRetirementContexts = null;
            SBRContext sbrRPRetirementContext = null;

            sbrRPRetirementContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty")) &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}SuperannuationPhaseDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Retirement"))))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPRetirementElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPRetirementContexts != null && sbrRPRetirementContexts.Count() > 0)
            {
                sbrRPRetirementContext = sbrRPRetirementContexts.First();

                report.RPRetirementId = sbrRPRetirementContext.Id;
                report.RPRetirementIdentifier = sbrRPRetirementContext.EntityIdentifier;
                report.RPRetirementIdentifierScheme = sbrRPRetirementContext.EntityScheme;

                report.RPRetirementCount = sbrRPRetirementContexts.Count();
                report.RPRetirementExists = true;
                report.RPRetirementOccurrenceIndex = Array.IndexOf(contextArray, sbrRPRetirementContext) + 1;
                report.RPRetirementLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPRetirementContexts.Last()) + 1;


                if (report.RPRetirementIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPRetirementIdentifierTFN = report.RPRetirementIdentifier;

                report.RPRetirementStartDate = ((SBRDurationPeriod)sbrRPRetirementContext.Period).DurationStartAsDateObject;
                report.RPRetirementEndDate = ((SBRDurationPeriod)sbrRPRetirementContext.Period).DurationEndAsDateObject;

                sbrPopulatedRPRetirementElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPRetirementContexts.Count(); i++)
                {
                    sbrRPRetirementContext = sbrRPRetirementContexts.ElementAt(i);

                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPRetirementContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPRetirementElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPRetirementElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPRetirementElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement });
                            }
                        }
                    }
                }
            }
            #endregion End of Context sbrRPRetirementContext

            #region sbrRPClosingAccumulationContext
            IEnumerable<SBRContext> sbrRPClosingAccumulationContexts = null;
            SBRContext sbrRPClosingAccumulationContext = null;

            sbrRPClosingAccumulationContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 3 &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty")) &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}OpeningAndClosingBalanceDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Closing")) &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}SuperannuationPhaseDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Accumulation")))))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPClosingAccumulationElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPClosingAccumulationContexts != null && sbrRPClosingAccumulationContexts.Count() > 0)
            {
                sbrRPClosingAccumulationContext = sbrRPClosingAccumulationContexts.First();

                report.RPClosingAccumulationId = sbrRPClosingAccumulationContext.Id;
                report.RPClosingAccumulationIdentifier = sbrRPClosingAccumulationContext.EntityIdentifier;
                report.RPClosingAccumulationIdentifierScheme = sbrRPClosingAccumulationContext.EntityScheme;

                report.RPClosingAccumulationCount = sbrRPClosingAccumulationContexts.Count();
                report.RPClosingAccumulationExists = true;
                report.RPClosingAccumulationOccurrenceIndex = Array.IndexOf(contextArray, sbrRPClosingAccumulationContext) + 1;
                report.RPClosingAccumulationLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPClosingAccumulationContexts.Last()) + 1;


                if (report.RPClosingAccumulationIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPClosingAccumulationIdentifierTFN = report.RPClosingAccumulationIdentifier;

                report.RPClosingAccumulationStartDate = ((SBRDurationPeriod)sbrRPClosingAccumulationContext.Period).DurationStartAsDateObject;
                report.RPClosingAccumulationEndDate = ((SBRDurationPeriod)sbrRPClosingAccumulationContext.Period).DurationEndAsDateObject;

                sbrPopulatedRPClosingAccumulationElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPClosingAccumulationContexts.Count(); i++)
                {
                    sbrRPClosingAccumulationContext = sbrRPClosingAccumulationContexts.ElementAt(i);

                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPClosingAccumulationContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPClosingAccumulationElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPClosingAccumulationElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPClosingAccumulationElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement });
                            }
                        }
                    }
                }
            }
            #endregion End of Context sbrRPClosingAccumulationContext

            #region sbrRPClosingRetirementContext
            IEnumerable<SBRContext> sbrRPClosingRetirementContexts = null;
            SBRContext sbrRPClosingRetirementContext = null;

            sbrRPClosingRetirementContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 3 &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty")) &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}OpeningAndClosingBalanceDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Closing")) &&
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}SuperannuationPhaseDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Retirement")))))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPClosingRetirementElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPClosingRetirementContexts != null && sbrRPClosingRetirementContexts.Count() > 0)
            {
                sbrRPClosingRetirementContext = sbrRPClosingRetirementContexts.First();

                report.RPClosingRetirementId = sbrRPClosingRetirementContext.Id;
                report.RPClosingRetirementIdentifier = sbrRPClosingRetirementContext.EntityIdentifier;
                report.RPClosingRetirementIdentifierScheme = sbrRPClosingRetirementContext.EntityScheme;

                report.RPClosingRetirementCount = sbrRPClosingRetirementContexts.Count();
                report.RPClosingRetirementExists = true;
                report.RPClosingRetirementOccurrenceIndex = Array.IndexOf(contextArray, sbrRPClosingRetirementContext) + 1;
                report.RPClosingRetirementLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPClosingRetirementContexts.Last()) + 1;


                if (report.RPClosingRetirementIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPClosingRetirementIdentifierTFN = report.RPClosingRetirementIdentifier;

                report.RPClosingRetirementStartDate = ((SBRDurationPeriod)sbrRPClosingRetirementContext.Period).DurationStartAsDateObject;
                report.RPClosingRetirementEndDate = ((SBRDurationPeriod)sbrRPClosingRetirementContext.Period).DurationEndAsDateObject;

                sbrPopulatedRPClosingRetirementElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPClosingRetirementContexts.Count(); i++)
                {
                    sbrRPClosingRetirementContext = sbrRPClosingRetirementContexts.ElementAt(i);

                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPClosingRetirementContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPClosingRetirementElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPClosingRetirementElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPClosingRetirementElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement });
                            }
                        }
                    }
                }
            }
            #endregion End of Context sbrRPClosingRetirementContext
            return report;

        } // Of Consume Method
        #region ValidateRPOrganisationNameDetailsCollection
        protected virtual void ValidateRPOrganisationNameDetailsCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, SMSFAR2025 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the RP Context exit out otherwise
                    if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;
                }
            }
        }
        #endregion

        #region ValidateRPAddressDetailsCollection
        protected virtual void ValidateRPAddressDetailsCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, SMSFAR2025 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the RP Context exit out otherwise
                    if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;
                }
            }
        }
        #endregion

        #region ValidateRPMemberInformation_PersonNameDetailsCollection
        protected virtual void ValidateRPMemberInformation_PersonNameDetailsCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, SMSFAR2025 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the RP Context exit out otherwise
                    if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;
                    bool MemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RPExists = true;

                    SBRElement SMSFAR251Value = null;
                    string SMSFAR251 = null;

                    SBRElement SMSFAR259Value = null;
                    string SMSFAR259 = null;

                    SBRElement SMSFAR255Value = null;
                    string SMSFAR255 = null;
                    SBRElement SMSFAR196Value = null;
                    string SMSFAR196 = null;
                    SBRElement SMSFAR198Value = null;
                    string SMSFAR198 = null;
                    SBRElement SMSFAR197Value = null;
                    string SMSFAR197 = null;
                    SBRElement SMSFAR199Value = null;
                    string SMSFAR199 = null;
                    SBRElement SMSFAR200Value = null;
                    string SMSFAR200 = null;

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.PersonNameType.Code", out SMSFAR251Value))
                    {
                        if (!SMSFAR251Value.IsNil) SMSFAR251 = SMSFAR251Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.Currency.Code", out SMSFAR259Value))
                    {
                        if (!SMSFAR259Value.IsNil) SMSFAR259 = SMSFAR259Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.Usage.Code", out SMSFAR255Value))
                    {
                        if (!SMSFAR255Value.IsNil) SMSFAR255 = SMSFAR255Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.Title.Text", out SMSFAR196Value))
                    {
                        if (!SMSFAR196Value.IsNil) SMSFAR196 = SMSFAR196Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.NameSuffix.Text", out SMSFAR198Value))
                    {
                        if (!SMSFAR198Value.IsNil) SMSFAR198 = SMSFAR198Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.FamilyName.Text", out SMSFAR197Value))
                    {
                        if (!SMSFAR197Value.IsNil) SMSFAR197 = SMSFAR197Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.GivenName.Text", out SMSFAR199Value))
                    {
                        if (!SMSFAR199Value.IsNil) SMSFAR199 = SMSFAR199Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.OtherGivenName.Text", out SMSFAR200Value))
                    {
                        if (!SMSFAR200Value.IsNil) SMSFAR200 = SMSFAR200Value.Value;
                    }

                    #region VR.ATO.SMSFAR.436297

                    /*  VR.ATO.SMSFAR.436297
                        Each member name must have the Name Type code 'LGL' (legal), Currency Code 'C' (current) and Usage code 'Identifier'

                    Legacy Rule Format:
                        WHERE IN TUPLE(prsnstrcnm3.xx.xx:PersonNameDetails) IN TUPLE (smsfar.0013.lodge.req.xx.xx:MemberInformation)
                        IF (pyde.xx.xx:PersonNameDetails.PersonNameType.Code <> "LGL") OR (pyde.xx.xx:PersonNameDetails.Currency.Code <> "C") OR (pyde.xx.xx:PersonNameDetails.Usage.Code <> "Identifier")
                           RETURN VALIDATION MESSAGE
                        ENDIF

                    Technical Business Rule Format:
                        (^PersonNameDetails <> NULL) AND ((^SMSFAR251 <> 'LGL') OR (^SMSFAR259 <> 'C') OR (^SMSFAR255 <> 'Identifier'))

                    Data Elements:

                    ^SMSFAR251 = RP:MemberInformation:PersonNameDetails:PersonNameDetails.PersonNameType.Code WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Identifier'].[PersonNameDetails.Currency.Code='C']

                    ^SMSFAR259 = RP:MemberInformation:PersonNameDetails:PersonNameDetails.Currency.Code WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Identifier'].[PersonNameDetails.Currency.Code='C']

                    ^SMSFAR255 = RP:MemberInformation:PersonNameDetails:PersonNameDetails.Usage.Code WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Identifier'].[PersonNameDetails.Currency.Code='C']

                    ^PersonNameDetails = RP:MemberInformation:PersonNameDetails
                    */
                    assertion = (MemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RPExists != false && (SMSFAR251 != @"LGL" || SMSFAR259 != @"C" || SMSFAR255 != @"Identifier"));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.SMSFAR.436297",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Member name Type code, Currency code and Usage code must be present and correct",
                            LongDescription = @"Each member name must have the Name Type code 'LGL' (legal) , Currency Code 'C' (current) and Usage code 'Identifier'. Please contact your software vendor in order to rectify this error",
                            Location = SMSFAR197Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:PersonNameDetails.FamilyName.Text" : DeriveSbrElementLocation(SMSFAR197Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436297" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "SMSFAR251", Value = SMSFAR251 });

                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "SMSFAR259", Value = SMSFAR259 });

                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "SMSFAR255", Value = SMSFAR255 });

                        errors.Add(processMessage);
                    }
                    #endregion
                }
            }
        }
        #endregion

        #region ValidateRPDeclaration_PersonUnstructuredNameCollection
        protected virtual void ValidateRPDeclaration_PersonUnstructuredNameCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, SMSFAR2025 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the RP Context exit out otherwise
                    if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;
                    bool Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPExists = true;

                    SBRElement SMSFAR277Value = null;
                    string SMSFAR277 = null;
                    SBRElement SMSFAR278Value = null;
                    string SMSFAR278 = null;

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonUnstructuredName.Usage.Code", out SMSFAR277Value))
                    {
                        if (!SMSFAR277Value.IsNil) SMSFAR277 = SMSFAR277Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonUnstructuredName.FullName.Text", out SMSFAR278Value))
                    {
                        if (!SMSFAR278Value.IsNil) SMSFAR278 = SMSFAR278Value.Value;
                    }

                    #region VR.ATO.GEN.436279

                    /*  VR.ATO.GEN.436279
                        The unstructured person name tuple must include the Usage Code 'DeclarationSignatory'.

                    Legacy Rule Format:
                        WHERE IN TUPLE (personunstructuredname1.xx.xx:PersonUnstructuredName) IN TUPLE(declaration2.xx.xx:Declaration)
                        IF pyde.xx.xx:PersonUnstructuredName.Usage.Code <> "DeclarationSignatory"
                           RETURN VALIDATION MESSAGE
                        ENDIF

                    Technical Business Rule Format:
                        (^PersonUnstructuredName <> NULL) AND (^SMSFAR277 <> 'DeclarationSignatory')

                    Data Elements:

                    ^SMSFAR277 = RP:Declaration:PersonUnstructuredName:PersonUnstructuredName.Usage.Code WHERE [PersonUnstructuredName.Usage.Code='DeclarationSignatory']

                    ^PersonUnstructuredName = RP:Declaration:PersonUnstructuredName
                    */
                    assertion = (Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPExists != false && SMSFAR277 != @"DeclarationSignatory");
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.436279",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Declaration name must have a usage code of 'DeclarationSignatory'",
                            Location = SMSFAR277Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:PersonUnstructuredName.Usage.Code" : DeriveSbrElementLocation(SMSFAR277Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.436279" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "SMSFAR277", Value = SMSFAR277 });

                        errors.Add(processMessage);
                    }
                    #endregion
                }
            }
        }
        #endregion

        #region ValidateRPDeclarationCollection
        protected virtual void ValidateRPDeclarationCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, SMSFAR2025 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the RP Context exit out otherwise
                    if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;

                    SBRElement SMSFAR267Value = null;
                    string SMSFAR267 = null;
                    SBRElement SMSFAR268Value = null;
                    bool? SMSFAR268 = null;
                    SBRElement SMSFAR191Value = null;
                    string SMSFAR191 = null;
                    SBRElement SMSFAR192Value = null;
                    DateTime? SMSFAR192 = null;
                    SBRElement SMSFAR269Value = null;
                    string SMSFAR269 = null;

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Declaration.StatementType.Code", out SMSFAR267Value))
                    {
                        if (!SMSFAR267Value.IsNil) SMSFAR267 = SMSFAR267Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Declaration.StatementAccepted.Indicator", out SMSFAR268Value))
                    {
                        if (!SMSFAR268Value.IsNil) SMSFAR268 = ToBoolean(SMSFAR268Value.Value);
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Declaration.Statement.Text", out SMSFAR191Value))
                    {
                        if (!SMSFAR191Value.IsNil) SMSFAR191 = SMSFAR191Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Declaration.Signature.Date", out SMSFAR192Value))
                    {
                        if (!SMSFAR192Value.IsNil) SMSFAR192 = DateTime.Parse(SMSFAR192Value.Value);
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Declaration.SignatoryIdentifier.Text", out SMSFAR269Value))
                    {
                        if (!SMSFAR269Value.IsNil) SMSFAR269 = SMSFAR269Value.Value;
                    }

                    #region VR.ATO.GEN.438062

                    /*  VR.ATO.GEN.438062
                        If an Intermediary is not present, Reporting Party Declaration Statement Type Code must be supplied and must be “TrueAndCorrect”.

                    Legacy Rule Format:
                        IF INT=NULL AND RP:pyin.xx.xx:Declaration.StatementType.Code <> "TrueAndCorrect"
                           RETURN VALIDATION MESSAGE
                        ENDIF

                    Technical Business Rule Format:
                        ^INT = NULL AND ^SMSFAR267 <> 'TrueAndCorrect'

                    Data Elements:

                    ^SMSFAR267 = RP:Declaration:Declaration.StatementType.Code WHERE [Declaration.StatementType.Code='TrueAndCorrect']

                    ^INT = INT
                    */
                    assertion = (report.INTExists == false && SMSFAR267 != @"TrueAndCorrect");
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.438062",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Declaration statement type code is required.",
                            LongDescription = @"If an Intermediary is not present, Reporting Party Declaration Statement Type Code must be supplied and must be ""TrueAndCorrect"".",
                            Location = SMSFAR267Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:Declaration.StatementType.Code" : DeriveSbrElementLocation(SMSFAR267Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438062" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "SMSFAR267", Value = SMSFAR267 });

                        errors.Add(processMessage);
                    }
                    #endregion
                }
            }
        }
        #endregion

        #region ValidateRPTrusteeOrDirector_PersonNameDetailsCollection
        protected virtual void ValidateRPTrusteeOrDirector_PersonNameDetailsCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, SMSFAR2025 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the RP Context exit out otherwise
                    if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;
                    bool TrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RPExists = true;

                    SBRElement SMSFAR252Value = null;
                    string SMSFAR252 = null;

                    SBRElement SMSFAR260Value = null;
                    string SMSFAR260 = null;

                    SBRElement SMSFAR256Value = null;
                    string SMSFAR256 = null;
                    SBRElement SMSFAR166Value = null;
                    string SMSFAR166 = null;
                    SBRElement SMSFAR168Value = null;
                    string SMSFAR168 = null;
                    SBRElement SMSFAR167Value = null;
                    string SMSFAR167 = null;
                    SBRElement SMSFAR169Value = null;
                    string SMSFAR169 = null;
                    SBRElement SMSFAR170Value = null;
                    string SMSFAR170 = null;

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.PersonNameType.Code", out SMSFAR252Value))
                    {
                        if (!SMSFAR252Value.IsNil) SMSFAR252 = SMSFAR252Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.Currency.Code", out SMSFAR260Value))
                    {
                        if (!SMSFAR260Value.IsNil) SMSFAR260 = SMSFAR260Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.Usage.Code", out SMSFAR256Value))
                    {
                        if (!SMSFAR256Value.IsNil) SMSFAR256 = SMSFAR256Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.Title.Text", out SMSFAR166Value))
                    {
                        if (!SMSFAR166Value.IsNil) SMSFAR166 = SMSFAR166Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.NameSuffix.Text", out SMSFAR168Value))
                    {
                        if (!SMSFAR168Value.IsNil) SMSFAR168 = SMSFAR168Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.FamilyName.Text", out SMSFAR167Value))
                    {
                        if (!SMSFAR167Value.IsNil) SMSFAR167 = SMSFAR167Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.GivenName.Text", out SMSFAR169Value))
                    {
                        if (!SMSFAR169Value.IsNil) SMSFAR169 = SMSFAR169Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.OtherGivenName.Text", out SMSFAR170Value))
                    {
                        if (!SMSFAR170Value.IsNil) SMSFAR170 = SMSFAR170Value.Value;
                    }

                    #region VR.ATO.SMSFAR.436296

                    /*  VR.ATO.SMSFAR.436296
                        Each person name must have the Name Type code 'LGL' (legal), Currency Code 'C' (current) and Usage code 'Contact'

                    Legacy Rule Format:
                        WHERE IN TUPLE(prsnstrcnm3.xx.xx:PersonNameDetails) IN TUPLE (smsfar.0013.lodge.req.xx.xx:TrusteeOrDirector)
                        IF (pyde.xx.xx:PersonNameDetails.PersonNameType.Code <> "LGL") OR (pyde.xx.xx:PersonNameDetails.Currency.Code <> "C") OR (pyde.xx.xx:PersonNameDetails.Usage.Code <> "Contact")
                           RETURN VALIDATION MESSAGE
                        ENDIF

                    Technical Business Rule Format:
                        (^PersonNameDetails <> NULL) AND ((^SMSFAR252 <> 'LGL') OR (^SMSFAR260 <> 'C') OR (^SMSFAR256 <> 'Contact'))

                    Data Elements:

                    ^SMSFAR252 = RP:TrusteeOrDirector:PersonNameDetails:PersonNameDetails.PersonNameType.Code WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']

                    ^SMSFAR260 = RP:TrusteeOrDirector:PersonNameDetails:PersonNameDetails.Currency.Code WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']

                    ^SMSFAR256 = RP:TrusteeOrDirector:PersonNameDetails:PersonNameDetails.Usage.Code WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']

                    ^PersonNameDetails = RP:TrusteeOrDirector:PersonNameDetails
                    */
                    assertion = (TrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RPExists != false && (SMSFAR252 != @"LGL" || SMSFAR260 != @"C" || SMSFAR256 != @"Contact"));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.SMSFAR.436296",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Individual name Type code, Currency code and Usage code must be present and correct",
                            LongDescription = @"Each person name must have the Name Type code 'LGL' (legal) , Currency Code 'C' (current) and Usage code 'Contact'. Please contact your software vendor in order to rectify this error",
                            Location = SMSFAR167Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:PersonNameDetails.FamilyName.Text" : DeriveSbrElementLocation(SMSFAR167Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436296" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "SMSFAR252", Value = SMSFAR252 });

                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "SMSFAR260", Value = SMSFAR260 });

                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "SMSFAR256", Value = SMSFAR256 });

                        errors.Add(processMessage);
                    }
                    #endregion
                }
            }
        }
        #endregion

        #region ValidateRPTrusteeOrDirector_ElectronicContactTelephoneCollection
        protected virtual void ValidateRPTrusteeOrDirector_ElectronicContactTelephoneCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, SMSFAR2025 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the RP Context exit out otherwise
                    if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;
                    bool TrusteeOrDirector_ElectronicContactTelephoneCollection_03_RPExists = true;

                    SBRElement SMSFAR262Value = null;
                    string SMSFAR262 = null;

                    SBRElement SMSFAR265Value = null;
                    string SMSFAR265 = null;
                    SBRElement SMSFAR171Value = null;
                    string SMSFAR171 = null;
                    SBRElement SMSFAR172Value = null;
                    string SMSFAR172 = null;

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}ElectronicContact.Telephone.Usage.Code", out SMSFAR262Value))
                    {
                        if (!SMSFAR262Value.IsNil) SMSFAR262 = SMSFAR262Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}ElectronicContact.Telephone.ServiceLine.Code", out SMSFAR265Value))
                    {
                        if (!SMSFAR265Value.IsNil) SMSFAR265 = SMSFAR265Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}ElectronicContact.Telephone.Area.Code", out SMSFAR171Value))
                    {
                        if (!SMSFAR171Value.IsNil) SMSFAR171 = SMSFAR171Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}ElectronicContact.Telephone.Minimal.Number", out SMSFAR172Value))
                    {
                        if (!SMSFAR172Value.IsNil) SMSFAR172 = SMSFAR172Value.Value;
                    }

                    #region VR.ATO.GEN.500131

                    /*  VR.ATO.GEN.500131
                        Telephone must include usage code '03' and service line code '01' or '02'.

                    Legacy Rule Format:
                        WHERE IN TUPLE(electroniccontacttelephone1.xx.xx:ElectronicContactTelephone)
                        IF pyde.xx.xx:ElectronicContact.Telephone.Usage.Code <> "03" OR pyde.xx.xx:ElectronicContact.Telephone.ServiceLine.Code <> SET ("01", "02")
                           RETURN VALIDATION MESSAGE
                        ENDIF

                    Technical Business Rule Format:
                        (^ElectronicContactTelephone <> NULL) AND (^SMSFAR262 <> '03' OR NotInSet(^SMSFAR265, '"01", "02"'))

                    Data Elements:

                    ^SMSFAR262 = RP:TrusteeOrDirector:ElectronicContactTelephone:ElectronicContact.Telephone.Usage.Code WHERE [ElectronicContact.Telephone.Usage.Code='03']

                    ^SMSFAR265 = RP:TrusteeOrDirector:ElectronicContactTelephone:ElectronicContact.Telephone.ServiceLine.Code WHERE [ElectronicContact.Telephone.Usage.Code='03']

                    ^ElectronicContactTelephone = RP:TrusteeOrDirector:ElectronicContactTelephone
                    */
                    assertion = (TrusteeOrDirector_ElectronicContactTelephoneCollection_03_RPExists != false && (SMSFAR262 != @"03" || !(IsMatch(SMSFAR265, @"^(01|02)$", RegexOptions.IgnoreCase))));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.500131",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Telephone must include usage code '03' and service line code '01' or '02'.",
                            Location = SMSFAR172Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:ElectronicContact.Telephone.Minimal.Number" : DeriveSbrElementLocation(SMSFAR172Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500131" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "SMSFAR262", Value = SMSFAR262 });

                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "SMSFAR265", Value = SMSFAR265 });

                        errors.Add(processMessage);
                    }
                    #endregion
                }
            }
        }
        #endregion

        #region ValidateRPTrusteeOrDirector_ElectronicContactElectronicMailCollection
        protected virtual void ValidateRPTrusteeOrDirector_ElectronicContactElectronicMailCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, SMSFAR2025 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the RP Context exit out otherwise
                    if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;
                    bool TrusteeOrDirector_ElectronicContactElectronicMailCollection_03_RPExists = true;

                    SBRElement SMSFAR173Value = null;
                    string SMSFAR173 = null;

                    SBRElement SMSFAR284Value = null;
                    string SMSFAR284 = null;

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}ElectronicContact.ElectronicMail.Address.Text", out SMSFAR173Value))
                    {
                        if (!SMSFAR173Value.IsNil) SMSFAR173 = SMSFAR173Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}ElectronicContact.ElectronicMail.Usage.Code", out SMSFAR284Value))
                    {
                        if (!SMSFAR284Value.IsNil) SMSFAR284 = SMSFAR284Value.Value;
                    }

                    #region VR.ATO.GEN.500140

                    /*  VR.ATO.GEN.500140
                        Email address must include an address and a usage code '03'

                    Legacy Rule Format:
                        WHERE IN TUPLE(ElectronicContactElectronicMail1.xx.xx:ElectronicContactElectronicMail)
                        IF (pyde.xx.xx:ElectronicContact.ElectronicMail.Address.Text = BLANK) OR (pyde.xx.xx:ElectronicContact.ElectronicMail.Usage.Code <> "03")
                            RETURN VALIDATION MESSAGE
                        ENDIF

                    Technical Business Rule Format:
                        (^ElectronicContactElectronicMail <> NULL) AND ((^SMSFAR173 = BLANK) OR (^SMSFAR284 <> '03'))

                    Data Elements:

                    ^SMSFAR284 = RP:TrusteeOrDirector:ElectronicContactElectronicMail:ElectronicContact.ElectronicMail.Usage.Code WHERE [ElectronicContact.ElectronicMail.Usage.Code='03']

                    ^SMSFAR173 = RP:TrusteeOrDirector:ElectronicContactElectronicMail:ElectronicContact.ElectronicMail.Address.Text WHERE [ElectronicContact.ElectronicMail.Usage.Code='03']

                    ^ElectronicContactElectronicMail = RP:TrusteeOrDirector:ElectronicContactElectronicMail
                    */
                    assertion = (TrusteeOrDirector_ElectronicContactElectronicMailCollection_03_RPExists != false && (string.IsNullOrWhiteSpace(SMSFAR173) == true || SMSFAR284 != @"03"));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.500140",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Email address must include an address and a usage code '03'.",
                            Location = SMSFAR284Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:ElectronicContact.ElectronicMail.Usage.Code" : DeriveSbrElementLocation(SMSFAR284Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500140" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "SMSFAR173", Value = SMSFAR173 });

                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "SMSFAR284", Value = SMSFAR284 });

                        errors.Add(processMessage);
                    }
                    #endregion
                }
            }
        }
        #endregion

        #region ValidateRPTrusteeOrDirector_OrganisationNameDetailsCollection
        protected virtual void ValidateRPTrusteeOrDirector_OrganisationNameDetailsCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, SMSFAR2025 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the RP Context exit out otherwise
                    if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;
                    bool TrusteeOrDirector_OrganisationNameDetailsCollection_MN_C_RPExists = true;

                    SBRElement SMSFAR236Value = null;
                    string SMSFAR236 = null;

                    SBRElement SMSFAR317Value = null;
                    string SMSFAR317 = null;
                    SBRElement SMSFAR174Value = null;
                    string SMSFAR174 = null;

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}OrganisationNameDetails.OrganisationalNameType.Code", out SMSFAR236Value))
                    {
                        if (!SMSFAR236Value.IsNil) SMSFAR236 = SMSFAR236Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}OrganisationNameDetails.Currency.Code", out SMSFAR317Value))
                    {
                        if (!SMSFAR317Value.IsNil) SMSFAR317 = SMSFAR317Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}OrganisationNameDetails.OrganisationalName.Text", out SMSFAR174Value))
                    {
                        if (!SMSFAR174Value.IsNil) SMSFAR174 = SMSFAR174Value.Value;
                    }

                    #region VR.ATO.GEN.438006

                    /*  VR.ATO.GEN.438006
                        Organisation Name Type Code must be 'MN' and Organisation Name Currency Code must be 'C'.

                    Legacy Rule Format:
                        WHERE IN TUPLE(orgname2.xx.xx:OrganisationNameDetails)
                        IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalNameType.Code <> "MN") OR (pyin.xx.xx:OrganisationNameDetails.Currency.Code <> "C")
                           RETURN VALIDATION MESSAGE
                        ENDIF

                    Technical Business Rule Format:
                        (^OrganisationNameDetails <> NULL) AND ((^SMSFAR236 <> 'MN') OR (^SMSFAR317 <> 'C'))

                    Data Elements:

                    ^SMSFAR236 = RP:TrusteeOrDirector:OrganisationNameDetails:OrganisationNameDetails.OrganisationalNameType.Code WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']

                    ^SMSFAR317 = RP:TrusteeOrDirector:OrganisationNameDetails:OrganisationNameDetails.Currency.Code WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']

                    ^OrganisationNameDetails = RP:TrusteeOrDirector:OrganisationNameDetails
                    */
                    assertion = (TrusteeOrDirector_OrganisationNameDetailsCollection_MN_C_RPExists != false && (SMSFAR236 != @"MN" || SMSFAR317 != @"C"));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.438006",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Organisation Name Type Code must be 'MN' and Organisation Name Currency Code must be 'C'.",
                            Location = SMSFAR174Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:OrganisationNameDetails.OrganisationalName.Text" : DeriveSbrElementLocation(SMSFAR174Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438006" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "SMSFAR236", Value = SMSFAR236 });

                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "SMSFAR317", Value = SMSFAR317 });

                        errors.Add(processMessage);
                    }
                    #endregion
                }
            }
        }
        #endregion

        #region ValidateINTDeclaration_PersonUnstructuredNameCollection
        protected virtual void ValidateINTDeclaration_PersonUnstructuredNameCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, SMSFAR2025 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the INT Context exit out otherwise
                    if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;
                    bool Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTExists = true;

                    SBRElement SMSFAR279Value = null;
                    string SMSFAR279 = null;
                    SBRElement SMSFAR280Value = null;
                    string SMSFAR280 = null;

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonUnstructuredName.Usage.Code", out SMSFAR279Value))
                    {
                        if (!SMSFAR279Value.IsNil) SMSFAR279 = SMSFAR279Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonUnstructuredName.FullName.Text", out SMSFAR280Value))
                    {
                        if (!SMSFAR280Value.IsNil) SMSFAR280 = SMSFAR280Value.Value;
                    }

                    #region VR.ATO.GEN.436279

                    /*  VR.ATO.GEN.436279
                        The unstructured person name tuple must include the Usage Code 'DeclarationSignatory'.

                    Legacy Rule Format:
                        WHERE IN TUPLE (personunstructuredname1.xx.xx:PersonUnstructuredName) IN TUPLE(declaration2.xx.xx:Declaration)
                        IF pyde.xx.xx:PersonUnstructuredName.Usage.Code <> "DeclarationSignatory"
                           RETURN VALIDATION MESSAGE
                        ENDIF

                    Technical Business Rule Format:
                        (^PersonUnstructuredName <> NULL) AND (^SMSFAR279 <> 'DeclarationSignatory')

                    Data Elements:

                    ^SMSFAR279 = INT:Declaration:PersonUnstructuredName:PersonUnstructuredName.Usage.Code WHERE [PersonUnstructuredName.Usage.Code='DeclarationSignatory']

                    ^PersonUnstructuredName = INT:Declaration:PersonUnstructuredName
                    */
                    assertion = (Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTExists != false && SMSFAR279 != @"DeclarationSignatory");
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.436279",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Declaration name must have a usage code of 'DeclarationSignatory'",
                            Location = SMSFAR279Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:PersonUnstructuredName.Usage.Code" : DeriveSbrElementLocation(SMSFAR279Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.436279" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "SMSFAR279", Value = SMSFAR279 });

                        errors.Add(processMessage);
                    }
                    #endregion
                }
            }
        }
        #endregion

        #region ValidateINTDeclarationCollection
        protected virtual void ValidateINTDeclarationCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, SMSFAR2025 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the INT Context exit out otherwise
                    if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;

                    SBRElement SMSFAR270Value = null;
                    string SMSFAR270 = null;
                    SBRElement SMSFAR271Value = null;
                    bool? SMSFAR271 = null;
                    SBRElement SMSFAR193Value = null;
                    string SMSFAR193 = null;
                    SBRElement SMSFAR194Value = null;
                    DateTime? SMSFAR194 = null;
                    SBRElement SMSFAR272Value = null;
                    string SMSFAR272 = null;

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Declaration.StatementType.Code", out SMSFAR270Value))
                    {
                        if (!SMSFAR270Value.IsNil) SMSFAR270 = SMSFAR270Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Declaration.StatementAccepted.Indicator", out SMSFAR271Value))
                    {
                        if (!SMSFAR271Value.IsNil) SMSFAR271 = ToBoolean(SMSFAR271Value.Value);
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Declaration.Statement.Text", out SMSFAR193Value))
                    {
                        if (!SMSFAR193Value.IsNil) SMSFAR193 = SMSFAR193Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Declaration.Signature.Date", out SMSFAR194Value))
                    {
                        if (!SMSFAR194Value.IsNil) SMSFAR194 = DateTime.Parse(SMSFAR194Value.Value);
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}Declaration.SignatoryIdentifier.Text", out SMSFAR272Value))
                    {
                        if (!SMSFAR272Value.IsNil) SMSFAR272 = SMSFAR272Value.Value;
                    }

                    #region VR.ATO.GEN.438061

                    /*  VR.ATO.GEN.438061
                        If an Intermediary is present, Intermediary declaration statement type code must be supplied and must be 
                        “TrueAndCorrect”.

                    Legacy Rule Format:
                        IF INT<>NULL AND INT:pyin.xx.xx:Declaration.StatementType.Code <> "TrueAndCorrect"
                           RETURN VALIDATION MESSAGE
                        ENDIF

                    Technical Business Rule Format:
                        ^INT <> NULL AND ^SMSFAR270 <> 'TrueAndCorrect'

                    Data Elements:

                    ^SMSFAR270 = INT:Declaration:Declaration.StatementType.Code WHERE [Declaration.StatementType.Code='TrueAndCorrect']

                    ^INT = INT
                    */
                    assertion = (report.INTExists != false && SMSFAR270 != @"TrueAndCorrect");
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.438061",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Declaration statement type code is required.",
                            LongDescription = @"If an Intermediary is present, Intermediary declaration statement type code must be supplied and must be ""TrueAndCorrect"".",
                            Location = SMSFAR270Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:Declaration.StatementType.Code" : DeriveSbrElementLocation(SMSFAR270Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438061" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "SMSFAR270", Value = SMSFAR270 });

                        errors.Add(processMessage);
                    }
                    #endregion

                    #region VR.ATO.SMSFAR.436288

                    /*  VR.ATO.SMSFAR.436288
                        If there is an intermediary, there must be a single declaration by them.

                    Legacy Rule Format:
                        IF CONTEXT(INT) <> NULL AND COUNT([SMSFAR270]) <> 1
                          RETURN VALIDATION MESSAGE
                        ENDIF

                    Technical Business Rule Format:
                        ^Context <> NULL AND Count(^SMSFAR270) <> 1

                    Data Elements:

                    ^SMSFAR270 = INT:Declaration:Declaration.StatementType.Code WHERE [Declaration.StatementType.Code='TrueAndCorrect']

                    ^Context = INT
                    */
                    assertion = (!alreadyChecked.Contains("VR.ATO.SMSFAR.436288") && (report.INTExists != false && report.DeclarationCollection_TrueAndCorrect_INTCount != 1));
                    if (assertion)
                    {
                        alreadyChecked.Add("VR.ATO.SMSFAR.436288");
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.SMSFAR.436288",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"If there is an intermediary, there must be a single declaration by them.",
                            Location = DeriveSbrElementLocation(tupleElement) + "/tns:Declaration.StatementType.Code",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436288" } },
                        };
                        errors.Add(processMessage);
                    }
                    #endregion
                }
            }
        }
        #endregion

        #region ValidateINTPersonNameDetailsCollection
        protected virtual void ValidateINTPersonNameDetailsCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, SMSFAR2025 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the INT Context exit out otherwise
                    if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;
                    bool PersonNameDetailsCollection_LGL_Contact_C_INTExists = true;

                    SBRElement SMSFAR273Value = null;
                    string SMSFAR273 = null;

                    SBRElement SMSFAR275Value = null;
                    string SMSFAR275 = null;

                    SBRElement SMSFAR274Value = null;
                    string SMSFAR274 = null;
                    SBRElement SMSFAR177Value = null;
                    string SMSFAR177 = null;
                    SBRElement SMSFAR179Value = null;
                    string SMSFAR179 = null;
                    SBRElement SMSFAR178Value = null;
                    string SMSFAR178 = null;
                    SBRElement SMSFAR180Value = null;
                    string SMSFAR180 = null;
                    SBRElement SMSFAR181Value = null;
                    string SMSFAR181 = null;

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.PersonNameType.Code", out SMSFAR273Value))
                    {
                        if (!SMSFAR273Value.IsNil) SMSFAR273 = SMSFAR273Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.Currency.Code", out SMSFAR275Value))
                    {
                        if (!SMSFAR275Value.IsNil) SMSFAR275 = SMSFAR275Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.Usage.Code", out SMSFAR274Value))
                    {
                        if (!SMSFAR274Value.IsNil) SMSFAR274 = SMSFAR274Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.Title.Text", out SMSFAR177Value))
                    {
                        if (!SMSFAR177Value.IsNil) SMSFAR177 = SMSFAR177Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.NameSuffix.Text", out SMSFAR179Value))
                    {
                        if (!SMSFAR179Value.IsNil) SMSFAR179 = SMSFAR179Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.FamilyName.Text", out SMSFAR178Value))
                    {
                        if (!SMSFAR178Value.IsNil) SMSFAR178 = SMSFAR178Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.GivenName.Text", out SMSFAR180Value))
                    {
                        if (!SMSFAR180Value.IsNil) SMSFAR180 = SMSFAR180Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.OtherGivenName.Text", out SMSFAR181Value))
                    {
                        if (!SMSFAR181Value.IsNil) SMSFAR181 = SMSFAR181Value.Value;
                    }

                    #region VR.ATO.SMSFAR.436294

                    /*  VR.ATO.SMSFAR.436294
                        Each person name must have the Name Type code 'LGL' (legal), Currency Code 'C' (current) and Usage code 'Contact'

                    Legacy Rule Format:
                        WHERE IN TUPLE(xbrli\prsnstrcnm3.xx.xx:PersonNameDetails)
                        IF (pyde.xx.xx:PersonNameDetails.PersonNameType.Code <> "LGL") OR (pyde.xx.xx:PersonNameDetails.Currency.Code <> "C") OR (pyde.xx.xx:PersonNameDetails.Usage.Code <> "Contact")
                           RETURN VALIDATION MESSAGE
                        ENDIF

                    Technical Business Rule Format:
                        (^PersonNameDetails <> NULL) AND ((^SMSFAR273 <> 'LGL') OR (^SMSFAR275 <> 'C') OR (^SMSFAR274 <> 'Contact'))

                    Data Elements:

                    ^SMSFAR273 = INT:PersonNameDetails:PersonNameDetails.PersonNameType.Code WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']

                    ^SMSFAR275 = INT:PersonNameDetails:PersonNameDetails.Currency.Code WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']

                    ^SMSFAR274 = INT:PersonNameDetails:PersonNameDetails.Usage.Code WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']

                    ^PersonNameDetails = INT:PersonNameDetails
                    */
                    assertion = (PersonNameDetailsCollection_LGL_Contact_C_INTExists != false && (SMSFAR273 != @"LGL" || SMSFAR275 != @"C" || SMSFAR274 != @"Contact"));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.SMSFAR.436294",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Individual name Type code, Currency code and Usage code must be present and correct",
                            LongDescription = @"Each person name must have the Name Type code 'LGL' (legal) , Currency Code 'C' (current) and Usage code 'Contact'. Please contact your software vendor in order to rectify this error",
                            Location = SMSFAR178Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:PersonNameDetails.FamilyName.Text" : DeriveSbrElementLocation(SMSFAR178Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436294" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "SMSFAR273", Value = SMSFAR273 });

                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "SMSFAR275", Value = SMSFAR275 });

                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "SMSFAR274", Value = SMSFAR274 });

                        errors.Add(processMessage);
                    }
                    #endregion
                }
            }
        }
        #endregion

        #region ValidateINTOrganisationNameDetailsCollection
        protected virtual void ValidateINTOrganisationNameDetailsCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, SMSFAR2025 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the INT Context exit out otherwise
                    if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;
                    bool OrganisationNameDetailsCollection_MN_C_INTExists = true;

                    SBRElement SMSFAR276Value = null;
                    string SMSFAR276 = null;

                    SBRElement SMSFAR318Value = null;
                    string SMSFAR318 = null;
                    SBRElement SMSFAR182Value = null;
                    string SMSFAR182 = null;

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}OrganisationNameDetails.OrganisationalNameType.Code", out SMSFAR276Value))
                    {
                        if (!SMSFAR276Value.IsNil) SMSFAR276 = SMSFAR276Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}OrganisationNameDetails.Currency.Code", out SMSFAR318Value))
                    {
                        if (!SMSFAR318Value.IsNil) SMSFAR318 = SMSFAR318Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}OrganisationNameDetails.OrganisationalName.Text", out SMSFAR182Value))
                    {
                        if (!SMSFAR182Value.IsNil) SMSFAR182 = SMSFAR182Value.Value;
                    }

                    #region VR.ATO.GEN.438006

                    /*  VR.ATO.GEN.438006
                        Organisation Name Type Code must be 'MN' and Organisation Name Currency Code must be 'C'.

                    Legacy Rule Format:
                        WHERE IN TUPLE(orgname2.xx.xx:OrganisationNameDetails)
                        IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalNameType.Code <> "MN") OR (pyin.xx.xx:OrganisationNameDetails.Currency.Code <> "C")
                           RETURN VALIDATION MESSAGE
                        ENDIF

                    Technical Business Rule Format:
                        (^OrganisationNameDetails <> NULL) AND ((^SMSFAR276 <> 'MN') OR (^SMSFAR318 <> 'C'))

                    Data Elements:

                    ^SMSFAR276 = INT:OrganisationNameDetails:OrganisationNameDetails.OrganisationalNameType.Code WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']

                    ^SMSFAR318 = INT:OrganisationNameDetails:OrganisationNameDetails.Currency.Code WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']

                    ^OrganisationNameDetails = INT:OrganisationNameDetails
                    */
                    assertion = (OrganisationNameDetailsCollection_MN_C_INTExists != false && (SMSFAR276 != @"MN" || SMSFAR318 != @"C"));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.438006",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Organisation Name Type Code must be 'MN' and Organisation Name Currency Code must be 'C'.",
                            Location = SMSFAR182Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:OrganisationNameDetails.OrganisationalName.Text" : DeriveSbrElementLocation(SMSFAR182Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438006" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "SMSFAR276", Value = SMSFAR276 });

                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "SMSFAR318", Value = SMSFAR318 });

                        errors.Add(processMessage);
                    }
                    #endregion
                }
            }
        }
        #endregion

        #region ValidateINTElectronicContactTelephoneCollection
        protected virtual void ValidateINTElectronicContactTelephoneCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, SMSFAR2025 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the INT Context exit out otherwise
                    if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;
                    bool ElectronicContactTelephoneCollection_03_INTExists = true;

                    SBRElement SMSFAR263Value = null;
                    string SMSFAR263 = null;

                    SBRElement SMSFAR266Value = null;
                    string SMSFAR266 = null;
                    SBRElement SMSFAR183Value = null;
                    string SMSFAR183 = null;
                    SBRElement SMSFAR184Value = null;
                    string SMSFAR184 = null;

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}ElectronicContact.Telephone.Usage.Code", out SMSFAR263Value))
                    {
                        if (!SMSFAR263Value.IsNil) SMSFAR263 = SMSFAR263Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}ElectronicContact.Telephone.ServiceLine.Code", out SMSFAR266Value))
                    {
                        if (!SMSFAR266Value.IsNil) SMSFAR266 = SMSFAR266Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}ElectronicContact.Telephone.Area.Code", out SMSFAR183Value))
                    {
                        if (!SMSFAR183Value.IsNil) SMSFAR183 = SMSFAR183Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}ElectronicContact.Telephone.Minimal.Number", out SMSFAR184Value))
                    {
                        if (!SMSFAR184Value.IsNil) SMSFAR184 = SMSFAR184Value.Value;
                    }

                    #region VR.ATO.GEN.500131

                    /*  VR.ATO.GEN.500131
                        Telephone must include usage code '03' and service line code '01' or '02'.

                    Legacy Rule Format:
                        WHERE IN TUPLE(electroniccontacttelephone1.xx.xx:ElectronicContactTelephone)
                        IF pyde.xx.xx:ElectronicContact.Telephone.Usage.Code <> "03" OR pyde.xx.xx:ElectronicContact.Telephone.ServiceLine.Code <> SET ("01", "02")
                           RETURN VALIDATION MESSAGE
                        ENDIF

                    Technical Business Rule Format:
                        (^ElectronicContactTelephone <> NULL) AND (^SMSFAR263 <> '03' OR NotInSet(^SMSFAR266, '"01", "02"'))

                    Data Elements:

                    ^SMSFAR263 = INT:ElectronicContactTelephone:ElectronicContact.Telephone.Usage.Code WHERE [ElectronicContact.Telephone.Usage.Code='03']

                    ^SMSFAR266 = INT:ElectronicContactTelephone:ElectronicContact.Telephone.ServiceLine.Code WHERE [ElectronicContact.Telephone.Usage.Code='03']

                    ^ElectronicContactTelephone = INT:ElectronicContactTelephone
                    */
                    assertion = (ElectronicContactTelephoneCollection_03_INTExists != false && (SMSFAR263 != @"03" || !(IsMatch(SMSFAR266, @"^(01|02)$", RegexOptions.IgnoreCase))));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.500131",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Telephone must include usage code '03' and service line code '01' or '02'.",
                            Location = SMSFAR184Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:ElectronicContact.Telephone.Minimal.Number" : DeriveSbrElementLocation(SMSFAR184Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500131" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "SMSFAR263", Value = SMSFAR263 });

                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "SMSFAR266", Value = SMSFAR266 });

                        errors.Add(processMessage);
                    }
                    #endregion
                }
            }
        }
        #endregion

        #region ValidateAuditorPersonNameDetailsCollection
        protected virtual void ValidateAuditorPersonNameDetailsCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, SMSFAR2025 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the Auditor Context exit out otherwise
                    if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;
                    bool PersonNameDetailsCollection_LGL_Contact_C_AuditorExists = true;

                    SBRElement SMSFAR249Value = null;
                    string SMSFAR249 = null;

                    SBRElement SMSFAR257Value = null;
                    string SMSFAR257 = null;

                    SBRElement SMSFAR253Value = null;
                    string SMSFAR253 = null;
                    SBRElement SMSFAR10Value = null;
                    string SMSFAR10 = null;
                    SBRElement SMSFAR12Value = null;
                    string SMSFAR12 = null;
                    SBRElement SMSFAR11Value = null;
                    string SMSFAR11 = null;
                    SBRElement SMSFAR13Value = null;
                    string SMSFAR13 = null;
                    SBRElement SMSFAR14Value = null;
                    string SMSFAR14 = null;

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.PersonNameType.Code", out SMSFAR249Value))
                    {
                        if (!SMSFAR249Value.IsNil) SMSFAR249 = SMSFAR249Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.Currency.Code", out SMSFAR257Value))
                    {
                        if (!SMSFAR257Value.IsNil) SMSFAR257 = SMSFAR257Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.Usage.Code", out SMSFAR253Value))
                    {
                        if (!SMSFAR253Value.IsNil) SMSFAR253 = SMSFAR253Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.Title.Text", out SMSFAR10Value))
                    {
                        if (!SMSFAR10Value.IsNil) SMSFAR10 = SMSFAR10Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.NameSuffix.Text", out SMSFAR12Value))
                    {
                        if (!SMSFAR12Value.IsNil) SMSFAR12 = SMSFAR12Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.FamilyName.Text", out SMSFAR11Value))
                    {
                        if (!SMSFAR11Value.IsNil) SMSFAR11 = SMSFAR11Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.GivenName.Text", out SMSFAR13Value))
                    {
                        if (!SMSFAR13Value.IsNil) SMSFAR13 = SMSFAR13Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}PersonNameDetails.OtherGivenName.Text", out SMSFAR14Value))
                    {
                        if (!SMSFAR14Value.IsNil) SMSFAR14 = SMSFAR14Value.Value;
                    }

                    #region VR.ATO.SMSFAR.436294

                    /*  VR.ATO.SMSFAR.436294
                        Each person name must have the Name Type code 'LGL' (legal), Currency Code 'C' (current) and Usage code 'Contact'

                    Legacy Rule Format:
                        WHERE IN TUPLE(xbrli\prsnstrcnm3.xx.xx:PersonNameDetails)
                        IF (pyde.xx.xx:PersonNameDetails.PersonNameType.Code <> "LGL") OR (pyde.xx.xx:PersonNameDetails.Currency.Code <> "C") OR (pyde.xx.xx:PersonNameDetails.Usage.Code <> "Contact")
                           RETURN VALIDATION MESSAGE
                        ENDIF

                    Technical Business Rule Format:
                        (^PersonNameDetails <> NULL) AND ((^SMSFAR249 <> 'LGL') OR (^SMSFAR257 <> 'C') OR (^SMSFAR253 <> 'Contact'))

                    Data Elements:

                    ^SMSFAR249 = Auditor:PersonNameDetails:PersonNameDetails.PersonNameType.Code WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']

                    ^SMSFAR257 = Auditor:PersonNameDetails:PersonNameDetails.Currency.Code WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']

                    ^SMSFAR253 = Auditor:PersonNameDetails:PersonNameDetails.Usage.Code WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']

                    ^PersonNameDetails = Auditor:PersonNameDetails
                    */
                    assertion = (PersonNameDetailsCollection_LGL_Contact_C_AuditorExists != false && (SMSFAR249 != @"LGL" || SMSFAR257 != @"C" || SMSFAR253 != @"Contact"));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.SMSFAR.436294",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Individual name Type code, Currency code and Usage code must be present and correct",
                            LongDescription = @"Each person name must have the Name Type code 'LGL' (legal) , Currency Code 'C' (current) and Usage code 'Contact'. Please contact your software vendor in order to rectify this error",
                            Location = SMSFAR11Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:PersonNameDetails.FamilyName.Text" : DeriveSbrElementLocation(SMSFAR11Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436294" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "SMSFAR249", Value = SMSFAR249 });

                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "SMSFAR257", Value = SMSFAR257 });

                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "SMSFAR253", Value = SMSFAR253 });

                        errors.Add(processMessage);
                    }
                    #endregion
                }
            }
        }
        #endregion

        #region ValidateAuditorElectronicContactTelephoneCollection
        protected virtual void ValidateAuditorElectronicContactTelephoneCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, SMSFAR2025 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the Auditor Context exit out otherwise
                    if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;
                    bool ElectronicContactTelephoneCollection_03_AuditorExists = true;

                    SBRElement SMSFAR261Value = null;
                    string SMSFAR261 = null;

                    SBRElement SMSFAR264Value = null;
                    string SMSFAR264 = null;
                    SBRElement SMSFAR17Value = null;
                    string SMSFAR17 = null;
                    SBRElement SMSFAR18Value = null;
                    string SMSFAR18 = null;

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}ElectronicContact.Telephone.Usage.Code", out SMSFAR261Value))
                    {
                        if (!SMSFAR261Value.IsNil) SMSFAR261 = SMSFAR261Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}ElectronicContact.Telephone.ServiceLine.Code", out SMSFAR264Value))
                    {
                        if (!SMSFAR264Value.IsNil) SMSFAR264 = SMSFAR264Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}ElectronicContact.Telephone.Area.Code", out SMSFAR17Value))
                    {
                        if (!SMSFAR17Value.IsNil) SMSFAR17 = SMSFAR17Value.Value;
                    }

                    if (tupleElementMap.TryGetValue("{http://www.sbr.gov.au/ato/smsfar}ElectronicContact.Telephone.Minimal.Number", out SMSFAR18Value))
                    {
                        if (!SMSFAR18Value.IsNil) SMSFAR18 = SMSFAR18Value.Value;
                    }

                    #region VR.ATO.GEN.500131

                    /*  VR.ATO.GEN.500131
                        Telephone must include usage code '03' and service line code '01' or '02'.

                    Legacy Rule Format:
                        WHERE IN TUPLE(electroniccontacttelephone1.xx.xx:ElectronicContactTelephone)
                        IF pyde.xx.xx:ElectronicContact.Telephone.Usage.Code <> "03" OR pyde.xx.xx:ElectronicContact.Telephone.ServiceLine.Code <> SET ("01", "02")
                           RETURN VALIDATION MESSAGE
                        ENDIF

                    Technical Business Rule Format:
                        (^ElectronicContactTelephone <> NULL) AND (^SMSFAR261 <> '03' OR NotInSet(^SMSFAR264, '"01", "02"'))

                    Data Elements:

                    ^SMSFAR261 = Auditor:ElectronicContactTelephone:ElectronicContact.Telephone.Usage.Code WHERE [ElectronicContact.Telephone.Usage.Code='03']

                    ^SMSFAR264 = Auditor:ElectronicContactTelephone:ElectronicContact.Telephone.ServiceLine.Code WHERE [ElectronicContact.Telephone.Usage.Code='03']

                    ^ElectronicContactTelephone = Auditor:ElectronicContactTelephone
                    */
                    assertion = (ElectronicContactTelephoneCollection_03_AuditorExists != false && (SMSFAR261 != @"03" || !(IsMatch(SMSFAR264, @"^(01|02)$", RegexOptions.IgnoreCase))));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.500131",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Telephone must include usage code '03' and service line code '01' or '02'.",
                            Location = SMSFAR18Value == null ? DeriveSbrElementLocation(tupleElement, true) + "/tns:ElectronicContact.Telephone.Minimal.Number" : DeriveSbrElementLocation(SMSFAR18Value),
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500131" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "SMSFAR261", Value = SMSFAR261 });

                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "SMSFAR264", Value = SMSFAR264 });

                        errors.Add(processMessage);
                    }
                    #endregion
                }
            }
        }
        #endregion

        #region ValidateAuditorAddressDetailsCollection
        protected virtual void ValidateAuditorAddressDetailsCollection(List<ProcessMessageDocument> errors, IEnumerable<SBRElement> tupleCollection, IEnumerable<SBRContext> contexts, SMSFAR2025 report)
        {
            bool assertion;
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            HashSet<string> alreadyChecked = new HashSet<string>();

            if (tupleCollection != null)
            {
                foreach (SBRElement tupleElement in tupleCollection)
                {
                    //Only check this for the Auditor Context exit out otherwise
                    if (tupleElement.ChildElements == null || tupleElement.ChildElements.Count == 0 || contexts == null || !contexts.Contains(tupleElement.ChildElements[0].Context))
                        continue;

                    IDictionary<string, SBRElement> tupleElementMap = tupleElement.PopulatedChildElementsMap;
                }
            }
        }
        #endregion

    } // Of Class
} // Of Namespace
