
using System;
using System.Xml;
using System.IO;
using System.Collections.Generic;
using DataContracts;
using System.Text.RegularExpressions;
using System.Reflection;
using System.Linq;
using System.Xml.Schema;
using System.Text;

namespace Ato.EN.IntegrationServices.CodeGenerationTRTAMI
{
    /// <summary>
    /// XML Consumer for TRTAMI
    /// </summary>
    public class TRTAMI2025XmlConsumer
    {
        const string MissingMandatoryMessage = "A mandatory field has not been completed.";
        const string ContactYourProviderMsg = "The message did not pass XML validation. Please contact your software provider.";
        const string InvalidDataValidation = "A field contains invalid data (such as letters in numeric or date field).";
        private const string ErrorCode1 = "CMN.ATO.GEN.XML01";
        private const string ErrorCode3 = "CMN.ATO.GEN.XML03";
        private const string ErrorCode4 = "CMN.ATO.GEN.XML04";
        private const string ErrorCode6 = "CMN.ATO.GEN.XML06";
        private bool _found = false;
        private int _parentCollectionCount;
        private int _childCollectionCount;
        private bool _isExiting = false;
        private bool _isValidationError = false;
        private string _lastPath = string.Empty;
        private ErrorMessageType _validationError = new ErrorMessageType(ErrorDescriptor.NoError);


        #region Error Messages

        public List<ProcessMessageDocument> ErrorMessages { get; set; }

        public bool HasErrors
        {
            get
            {
                return ErrorMessages != null && ErrorMessages.Count > 0;
            }
        }

        private string GetCurrentLocation()
        {
            StringBuilder location = new StringBuilder();
            string[] paths = _currentXPath.ToArray<string>();
            for (int i = paths.Length - 1; i > -1; i--)
            {
                location.Append(paths[i]);
            }
            return location.ToString();
        }

        private void MissingElementError()
        {
            ProcessMessageDocument processMessage;
            processMessage = new ProcessMessageDocument()
            {
                Code = "CMN.ATO.GEN.XML04",
                Description = MissingMandatoryMessage,
                SeverityAsString = "Error",
                Location = GetCurrentLocation(),
            };
            this.ErrorMessages.Add(processMessage);
        }

        #endregion  Error Messages

        #region Embedded Schema

        private static readonly XmlSchemaSet EmbeddedXmlSchemaSet;
        static TRTAMI2025XmlConsumer()
        {
            Assembly executingAssembly = Assembly.GetExecutingAssembly();
            string[] manifestResourceNames = executingAssembly.GetManifestResourceNames();
            string resourceName;
            Stream embeddedSchemaStream;
            XmlSchema embeddedXmlSchema;

            if (manifestResourceNames != null)
            {
                resourceName = manifestResourceNames.FirstOrDefault(rn => rn.Contains("ato.trtami.0007.2025.01.00.xsd") && rn.EndsWith(".xsd"));
                if (resourceName != null)
                {
                    embeddedSchemaStream = executingAssembly.GetManifestResourceStream(resourceName);
                    embeddedXmlSchema = XmlSchema.Read(embeddedSchemaStream, SchemaCallback);
                    EmbeddedXmlSchemaSet = new XmlSchemaSet();
                    EmbeddedXmlSchemaSet.Add(embeddedXmlSchema);
                    EmbeddedXmlSchemaSet.Compile();
                }
            }
        }

        private static void SchemaCallback(object sender, ValidationEventArgs args)
        {
            if (args.Severity == XmlSeverityType.Error)
                throw new XmlSchemaException(args.Message);
        }

        #endregion  Embedded Schema

        #region Xml Reader Settings

        private static XmlReaderSettings ReaderSettings = new XmlReaderSettings()
        {
            CloseInput = false,
            ConformanceLevel = ConformanceLevel.Fragment,
            IgnoreWhitespace = true,
            IgnoreComments = true,
            IgnoreProcessingInstructions = true,
        };

        private XmlReaderSettings GetValidatingReaderSettings()
        {
            XmlReaderSettings validatingReaderSettings = new XmlReaderSettings()
            {
                CloseInput = false,
                ConformanceLevel = ConformanceLevel.Fragment,
                IgnoreWhitespace = true,
                IgnoreComments = true,
                IgnoreProcessingInstructions = true,
                ValidationType = ValidationType.Schema,
            };
            if (EmbeddedXmlSchemaSet == null)
            {
                throw new XmlSchemaException("Embedded Schema is Null");
            }
            else
            {
                validatingReaderSettings.Schemas.Add(EmbeddedXmlSchemaSet);
                validatingReaderSettings.ValidationEventHandler += new ValidationEventHandler(ValidationCallBack);
            }
            return validatingReaderSettings;
        }

        private static Regex DataTypeFailureExpression = new Regex("'(?<Uniqueid>.*?)' element is invalid.*value '(?<Value>.*?)' is invalid.*datatype '(?<DataType>.*?)'", RegexOptions.IgnoreCase | RegexOptions.Compiled | RegexOptions.Singleline);

        private void ValidationCallBack(object sender, ValidationEventArgs args)
        {
            var msg = args.Message;
            var msgToLower = msg.ToLower();

            ProcessMessageDocument processMessage = null;

            if (msgToLower.Contains("invalid according to its datatype"))
            {
                Match match = DataTypeFailureExpression.Match(args.Message);

                if (match.Success && match.Groups.Count == 4)
                {
                    var inner = args.Exception.InnerException;
                    var hint = (inner != null && !string.IsNullOrEmpty(inner.Message)) ? " Hint: " + inner.Message : string.Empty;
                    processMessage = BuildProcessMessageDocument(InvalidDataValidation, string.Empty, ErrorCode3);

                    string uniqueID = match.Groups["Uniqueid"].Value;
                    string value = match.Groups["Value"].Value;
                    string dataType = match.Groups["DataType"].Value;


                    var longDescription = string.Format(@"The value specified for an item does not match the item type (value = ""{0}"", item type = {1}, uniqueID = {2})", value, dataType, uniqueID);
                    processMessage.LongDescription = longDescription + hint;
                    processMessage.Parameters = new ProcessMessageParameters();
                    processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "uniqueID", Value = uniqueID });
                    processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "value", Value = value });
                    processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "dataType", Value = dataType });
                }
            }
            else if (msgToLower.Contains("has incomplete content"))
            {
                _validationError = GetErrorDescriptorForHasIncompleteContent(msg);
                processMessage = BuildProcessMessageDocument(MissingMandatoryMessage, msg + _validationError.Hint, _validationError.Code);
                _isValidationError = true;
            }
            else if (msgToLower.Contains("has invalid child element"))
            {
                _validationError = GetErrorDescriptorForHasInvalidChildElement(msg);
                var shortMessage = _validationError.Code == ErrorCode4 ? MissingMandatoryMessage : ContactYourProviderMsg;
                processMessage = BuildProcessMessageDocument(shortMessage, msg + _validationError.Hint, _validationError.Code);
                _isValidationError = true;
            }
            else
            {
                processMessage = BuildProcessMessageDocument(ContactYourProviderMsg, msg, ErrorCode1);
                _isValidationError = true;
            }

            this.ErrorMessages.Add(processMessage);
        }

        private ProcessMessageDocument BuildProcessMessageDocument(string shortDescription, string longDescription, string code)
        {
            return new ProcessMessageDocument()
            {
                Code = code,
                Description = shortDescription,
                LongDescription = longDescription,
                SeverityAsString = "Error",
                Location = GetCurrentLocation()
            };
        }

        private ProcessMessageDocument BuildProcessMessageDocument(ProcessMessageDocument processMessage, string newLocation)
        {
            return new ProcessMessageDocument()
            {
                Code = processMessage.Code,
                Description = processMessage.Description,
                LongDescription = processMessage.LongDescription,
                SeverityAsString = "Error",
                Location = newLocation
            };
        }

        private enum ErrorDescriptor
        {
            NoError = 0,
            MandatoryElementError,
            NonMandatoryElementError,
            UnknownElement,
            Duplicate
        }

        private struct ErrorMessageType
        {
            public string Hint, Code, ExpectedNextValidElement;
            public ErrorDescriptor Errno;

            public ErrorMessageType(ErrorDescriptor errorDescriptor)
            {
                Errno = errorDescriptor;
                Hint = "";
                Code = "";
                ExpectedNextValidElement = "";
            }

            public ErrorMessageType(ErrorDescriptor errorDescriptor, string hint, string code, string elementName)
            {
                Errno = errorDescriptor;
                Hint = hint;
                Code = code;
                ExpectedNextValidElement = elementName;
            }
        }

        private ErrorMessageType GetErrorDescriptorForHasIncompleteContent(string validationErrorMessage)
        {
            ErrorMessageType retval = new ErrorMessageType();
            if (string.IsNullOrWhiteSpace(validationErrorMessage)) return retval;

            validationErrorMessage = validationErrorMessage.Replace(" ", string.Empty);
            validationErrorMessage = validationErrorMessage.Replace("'", " ");
            var parts = validationErrorMessage.Split(' ');
            if (parts.Length != 9) return retval;

            var elementBeingProcessed = parts[1];
            var csvList = parts[5];

            retval = ParseHasIncompleteContentCVSList(csvList, elementBeingProcessed);
            return retval;
        }

        private ErrorMessageType GetErrorDescriptorForHasInvalidChildElement(string validationErrorMessage)
        {
            string msg = string.Empty;
            ErrorMessageType retval = new ErrorMessageType();
            if (string.IsNullOrWhiteSpace(validationErrorMessage)) return retval;

            validationErrorMessage = validationErrorMessage.Replace(" ", string.Empty);
            validationErrorMessage = validationErrorMessage.Replace("'", " ");
            var parts = validationErrorMessage.Split(' ');

            //decide which message format to process
            if (validationErrorMessage.ToLower().Contains("listofpossibleelements"))
            {
                if (parts.Length == 13 || parts.Length == 12 || parts.Length == 10)
                {
                    var csvList = parts[9];
                    var elementBeingProcessed = parts[1];
                    var invalidChildElementName = parts[5];
                    retval = ParseCvsListForHasInvalidChildElementLongFormat(csvList, invalidChildElementName, elementBeingProcessed);
                }

                return retval;
            }
            else
            {
                if (parts.Length != 9) return retval;
                var csvList = parts[5];
                var elementBeingProcessed = parts[1];
                var invalidChildElementName = parts[5];

                retval = ParseCvsListForHasInvalidChildElementLongFormat(csvList, invalidChildElementName, elementBeingProcessed);
            }
            return retval;
        }

        private ErrorMessageType ParseCvsListForHasInvalidChildElementShortFormat(string csvList, string invalidChildElementName, string elementBeingProcessed)
        {
            ErrorMessageType retval = new ErrorMessageType();

            if (!string.IsNullOrEmpty(csvList))
            {
                var lastElementInListOfPossibleElements = GetLastElementInCsvList(csvList);

                retval.Hint =
                    string.Format(
                        " Hint: while processing parent element [{0}] child element [{1}] was unexpected",
                        elementBeingProcessed, invalidChildElementName);
                retval.Code = ErrorCode1;
                retval.Errno = ErrorDescriptor.UnknownElement;
                retval.ExpectedNextValidElement = invalidChildElementName;
            }

            return retval;
        }

        private string GetLastElementInCsvList(string csvList)
        {
            var list = (csvList.Contains(','))
                ? new List<string>(csvList.Split(','))
                : new List<string> { csvList };

            string lastElement = list.Last();
            if (lastElement.EndsWith("...."))
            {
                list.RemoveAt(list.Count - 1);
            }

            return list.Last();
        }

        private ErrorMessageType ParseCvsListForHasInvalidChildElementLongFormat(string csvList, string invalidChildElementName, string elementBeingProcessed)
        {
            ErrorMessageType retval = new ErrorMessageType();

            if (!string.IsNullOrEmpty(csvList))
            {
                var lastElementInListOfPossibleElements = GetLastElementInCsvList(csvList);
                var elementList = new List<string>(_elementCsvList.Split(','));

                if (ContainsElementForElementBeingProcessed(elementList, elementBeingProcessed, invalidChildElementName))
                {
                    var indexOfElementBeingProcessed = GetIndexOfElementBeingProcessed(elementList, elementBeingProcessed);
                    var indexOfinvalidChildElementName = GetIndexOfElementUnderElementBeingProcessed(elementList, elementBeingProcessed, invalidChildElementName);
                    var indexOflastElementInListOfPossibleElements = GetIndexOfElementUnderElementBeingProcessed(elementList, elementBeingProcessed, lastElementInListOfPossibleElements);

                    if (indexOfinvalidChildElementName > indexOflastElementInListOfPossibleElements)
                    {
                        retval.Hint =
                            string.Format(
                                " Hint: while processing parent element [{0}] child mandatory element [{1}] was expected but not found",
                                elementBeingProcessed, lastElementInListOfPossibleElements);
                        retval.Code = ErrorCode4;
                        retval.Errno = ErrorDescriptor.MandatoryElementError;
                        retval.ExpectedNextValidElement = lastElementInListOfPossibleElements;
                    }
                    else
                    {
                        retval.Hint =
                            string.Format(
                                " Hint: while processing parent element [{0}] child non mandatory element [{1}] was unexpected",
                                elementBeingProcessed, invalidChildElementName);
                        retval.Code = ErrorCode1;
                        retval.Errno = (indexOfinvalidChildElementName != indexOflastElementInListOfPossibleElements) ? ErrorDescriptor.NonMandatoryElementError : ErrorDescriptor.Duplicate;
                        retval.ExpectedNextValidElement = invalidChildElementName;
                    }
                }
                else
                {
                    //this element is unknown to the schema.
                    retval.Hint =
                        string.Format(
                            " Hint: while processing parent element [{0}] child element [{1}] was unexpected",
                            elementBeingProcessed, invalidChildElementName);
                    retval.Code = ErrorCode1;
                    retval.Errno = ErrorDescriptor.UnknownElement;
                    retval.ExpectedNextValidElement = invalidChildElementName;
                }
            }

            return retval;
        }

        private int GetIndexOfElementUnderElementBeingProcessed(List<string> validElementList,
                                                                string elementBeingProcessed,
                                                                string elementName)
        {
            int result;

            string elementWithParentNameMatch = elementBeingProcessed + "#" + elementName;
            result = validElementList.IndexOf(elementWithParentNameMatch);

            if (result < 0)
            {
                result = validElementList.IndexOf(elementName);
            }

            return result;
        }

        private int GetIndexOfElementBeingProcessed(List<string> validElementList,
                                                    string elementBeingProcessed)
        {
            return validElementList.IndexOf(elementBeingProcessed);
        }

        private bool ContainsElementForElementBeingProcessed(List<string> validElementList,
                                                             string elementBeingProcessed,
                                                             string elementName)
        {
            return validElementList.Contains(elementBeingProcessed + "#" + elementName) || validElementList.Contains(elementName);
        }

        private ErrorMessageType ParseHasIncompleteContentCVSList(string csvList, string elementBeingProcessed)
        {
            ErrorMessageType retval = new ErrorMessageType();

            if (!string.IsNullOrEmpty(csvList))
            {
                var lastElementInListOfPossibleElements = GetLastElementInCsvList(csvList);

                retval.Hint =
                    string.Format(
                        " Hint: while processing parent element [{0}] child mandatory element [{1}] was not found",
                        elementBeingProcessed, lastElementInListOfPossibleElements);

                retval.Code = ErrorCode4;
                retval.Errno = ErrorDescriptor.MandatoryElementError;
                retval.ExpectedNextValidElement = lastElementInListOfPossibleElements;
            }
            return retval;
        }

        private void StartEndDateError(DateTime startDate, DateTime endDate)
        {
            ProcessMessageDocument processMessage;

            string longDescription = string.Format("End date is earlier than start date (state date = {0}, end date = {1})", startDate.ToString("yyyy-MM-dd"), endDate.ToString("yyyy-MM-dd"));

            processMessage = new ProcessMessageDocument()
            {
                Code = "CMN.ATO.GEN.XML06",
                Description = "End date is earlier than start date.",
                LongDescription = longDescription,
                SeverityAsString = "Error",
                Location = GetCurrentLocation(),
            };
            processMessage.Parameters = new ProcessMessageParameters();
            processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "startDate", Value = startDate.ToString("yyyy-MM-dd") });
            processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "endDate", Value = endDate.ToString("yyyy-MM-dd") });
            this.ErrorMessages.Add(processMessage);
        }

        private void StartEndDateError(DateTime? startDate, DateTime? endDate)
        {
            ProcessMessageDocument processMessage;

            if (startDate == null || endDate == null)
                return;

            string longDescription = string.Format("End date is earlier than start date (state date = {0}, end date = {1})", startDate.Value.ToString("yyyy-MM-dd"), endDate.Value.ToString("yyyy-MM-dd"));

            processMessage = new ProcessMessageDocument()
            {
                Code = "CMN.ATO.GEN.XML06",
                Description = "End date is earlier than start date.",
                LongDescription = longDescription,
                SeverityAsString = "Error",
                Location = GetCurrentLocation(),
            };
            processMessage.Parameters = new ProcessMessageParameters();
            processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "startDate", Value = startDate.Value.ToString("yyyy-MM-dd") });
            processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "endDate", Value = endDate.Value.ToString("yyyy-MM-dd") });
            this.ErrorMessages.Add(processMessage);
        }

        #endregion  Xml Reader Settings

        #region IsEmptyOrNilElement
        private static bool IsEmptyOrNilElement(XmlReader reader)
        {
            bool emptyOrNil = false;
            if (reader.IsEmptyElement)
            {
                emptyOrNil = true;
            }
            else
            {
                if (reader.HasAttributes)
                {
                    string nilValue = reader.GetAttribute("nil", "http://www.w3.org/2001/XMLSchema-instance");
                    if (nilValue != null && nilValue.ToLowerInvariant() == "true" || nilValue == "1")
                    {
                        emptyOrNil = true;
                    }
                }
            }
            return emptyOrNil;
        }
        #endregion  IsEmptyOrNilElement

        #region ToBoolean
        private bool? ToBoolean(string str)
        {
            bool returnValue;
            string value = (str ?? "").Trim();
            if (value == "0" || value == "1")
                return Convert.ToBoolean(Convert.ToInt32(value));

            if (Boolean.TryParse(value, out returnValue))
                return returnValue;
            return null;
        }
        #endregion  ToBoolean

        #region MoveToContent
        public bool MoveToContent(XmlReader reader)
        {
            try
            {
                reader.MoveToContent();
                return true;
            }
            catch (XmlException ex)
            {
                ProcessMessageDocument processMessage;
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.XML01",
                    Description = ContactYourProviderMsg,
                    LongDescription = ex.Message,
                    SeverityAsString = "Error",
                };
                this.ErrorMessages.Add(processMessage);
                return false;
            }
        }
        #endregion  MoveToContent

        #region ReadToNextElement


        public bool ReadToNextElement(XmlReader reader, bool isMandatory)
        {
            return ReadToNextElement(reader);
        }

        public bool ReadToNextElement(XmlReader reader)
        {
            bool retval = false;

            if ((reader.EOF && !_isValidationError) || _isExiting)
            {
                return false; //do nothing
            }

            if (!_isValidationError)
            {
                retval = SetReaderToNextElement(reader); //position the reader on the next valid element
            }

            if (_isValidationError)
            {
                //process validation errors
                var expectedNextElement = (!string.IsNullOrEmpty(_validationError.ExpectedNextValidElement)) ? _validationError.ExpectedNextValidElement : reader.LocalName;
                var actualNextValidElement = reader.LocalName;
                var location = GetCurrentLocation();

                if (_validationError.Errno == ErrorDescriptor.Duplicate)
                {
                    location = _lastPath;
                }

                var xpath = RemoveLeadingAndTrailingSlashes(location);

                var element = GetElementDescriptor(xpath);

                if (_validationError.Errno == ErrorDescriptor.UnknownElement)
                {
                    //set the xpath for elements that are not known to the schema
                    location = "/" + element.Path + "/" + element.Prefix + ":" + _validationError.ExpectedNextValidElement;
                    _isExiting = true;
                }
                else if (_validationError.Errno == ErrorDescriptor.NonMandatoryElementError)
                {
                    //set the xpath for the optional element that caused a validation
                    location = "/" + element.Path + "/" + element.Prefix + ":" + reader.LocalName;
                    _isExiting = true;
                }
                else
                {
                    //decide if we have spooled to the missing mandatory element. 
                    _isExiting = expectedNextElement == element.PathLeafNode;
                }

                if (_isExiting)
                {
                    //The location has changed since detection of the validation error - so update it.
                    UpdateErrorMessageWithLocation(location);
                }

                return false;
            }

            //Save previous path so we can detect duplicate elements
            _lastPath = GetCurrentLocation();

            return retval;
        }

        private bool ReadNext(XmlReader reader)
        {
            return !reader.EOF && reader.Read();
        }

        private void UpdateErrorMessageWithLocation(string newLocation)
        {
            var lastProcessMessage = ErrorMessages.Last();
            ErrorMessages.Remove(lastProcessMessage);
            var newProcessMessage = BuildProcessMessageDocument(lastProcessMessage, newLocation);
            ErrorMessages.Add(newProcessMessage);
        }

        private string RemoveLeadingAndTrailingSlashes(string xpath)
        {
            var xpathLength = xpath.Length;
            if (xpathLength <= 0) return string.Empty;
            var firstChar = xpath.Substring(0, 1);
            var lastChar = xpath.Substring(xpathLength - 1, 1);
            if (firstChar == "/")
            {
                xpath = xpath.Remove(0, 1);
                xpathLength--;
            }
            if (lastChar == "/")
            {
                xpath = xpath.Remove(xpathLength - 1, 1);
            }
            return xpath;
        }

        private struct ElementDescriptor
        {
            public string Prefix, PathLeafNode, Path;
            public ElementDescriptor(string prefix, string leaf, string path)
            {
                Prefix = prefix;
                PathLeafNode = leaf;
                Path = path;
            }
        }

        private ElementDescriptor GetElementDescriptor(string xpath)
        {
            var retval = new ElementDescriptor();

            var parts = xpath.Split('/');
            var numberOfParts = parts.Length;
            if (numberOfParts > 0)
            {
                var leafNode = parts[numberOfParts - 1];
                var p = leafNode.Split(':');
                numberOfParts = p.Length;
                if (numberOfParts == 1)
                {
                    retval.PathLeafNode = p[0];
                    retval.Prefix = string.Empty;
                }
                else if (numberOfParts == 2)
                {
                    retval.PathLeafNode = p[1];
                    retval.Prefix = p[0];
                }
                var leafNodeLength = leafNode.Length;
                var path = xpath.Remove(xpath.Length - leafNodeLength, leafNodeLength);
                retval.Path = RemoveLeadingAndTrailingSlashes(path);
            }
            return retval;
        }

        private bool SetReaderToNextElement(XmlReader reader)
        {
            try
            {
                if (reader.EOF || _isExiting) return false;
                if (_found)
                {
                    reader.Read();
                    _found = false;
                }
                while (!reader.EOF && reader.NodeType != XmlNodeType.Element)
                {
                    reader.Read();
                }

                return reader.NodeType == XmlNodeType.Element && !reader.EOF;

            }
            catch (XmlException ex)
            {
                ProcessMessageDocument processMessage;
                processMessage = new ProcessMessageDocument()
                {
                    Code = ErrorCode1,
                    Description = ContactYourProviderMsg,
                    LongDescription = ex.Message,
                    SeverityAsString = "Error",
                };
                this.ErrorMessages.Add(processMessage);
                return false;
            }
        }

        #endregion  ReadToNextElement

        private Stack<string> _currentXPath = new Stack<string>();

        public TRTAMI2025 Consume(Stream streamToLoad, bool validateDataTypes = false)
        {
            TRTAMI2025 report = new TRTAMI2025();

            // Working Variable for if can still read from the xml stream
            bool reading;

            // Working Variables for current values
            string currentValue;
            DateTime currentDateTimeValue;
            bool? currentBooleanValue;
            decimal currentDecimalValue;
            double currentDoubleValue;
            float currentFloatValue;
            sbyte currentsByteValue;
            byte currentByteValue;
            short currentShortValue;
            ushort currentuShortValue;
            uint currentuIntValue;
            int currentIntValue;
            long currentLongValue;
            ulong currentuLongValue;

            this.ErrorMessages = new List<ProcessMessageDocument>();
            _currentXPath.Push("/tns:TRTAMI");

            if (streamToLoad == null || streamToLoad.Length == 0 || !streamToLoad.CanRead)
            {
                MissingElementError();
                return report;
            }

            streamToLoad.Position = 0;
            XmlReader reader;

            if (validateDataTypes)
                reader = XmlReader.Create(streamToLoad, GetValidatingReaderSettings());
            else
                reader = XmlReader.Create(streamToLoad, ReaderSettings);

            if (!MoveToContent(reader))
                return report;

            reading = !reader.EOF;

            if (ReadToNextElement(reader) && reader.LocalName == "TRTAMI" && reader.NamespaceURI == "http://www.sbr.gov.au/ato/trtami")
            {
                _found = true;
                ReadToNextElement(reader);
            }
            else
            {
                MissingElementError();
                return report;
            }


            #region RP
            _currentXPath.Push("/tns:RP");
            //3. use case
            if (ReadToNextElement(reader, true) && reader.LocalName == "RP" && reader.Depth == _currentXPath.Count - 1)
            {
                report.RpCollectionExists = true;
                report.RpCollectionCount += 1;
                _found = true;

                #region TargetFinancialY
                _currentXPath.Push("/tns:TargetFinancialY");
                //6. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "TargetFinancialY" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Int32.TryParse(currentValue, out currentIntValue))
                            {
                                report.TRTAMI1 = currentIntValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion TargetFinancialY

                #region PeriodStartD
                _currentXPath.Push("/tns:PeriodStartD");
                //6. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "PeriodStartD" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                            {
                                report.TRTAMI2 = currentDateTimeValue.Date;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion PeriodStartD

                #region PeriodEndD
                _currentXPath.Push("/tns:PeriodEndD");
                //6. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "PeriodEndD" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                            {
                                report.TRTAMI3 = currentDateTimeValue.Date;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion PeriodEndD

                #region TaxFileNumberId
                _currentXPath.Push("/tns:TaxFileNumberId");
                //6. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "TaxFileNumberId" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.TRTAMI4 = currentValue;
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion TaxFileNumberId

                #region AustralianBusinessNumberId
                _currentXPath.Push("/tns:AustralianBusinessNumberId");
                //6. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "AustralianBusinessNumberId" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.TRTAMI5 = currentValue;
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion AustralianBusinessNumberId

                #region CorporateCollectiveInvestmentVehicleSubFundI
                _currentXPath.Push("/tns:CorporateCollectiveInvestmentVehicleSubFundI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "CorporateCollectiveInvestmentVehicleSubFundI" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.TRTAMI192 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion CorporateCollectiveInvestmentVehicleSubFundI

                #region AustralianRegisteredFundNumberId
                _currentXPath.Push("/tns:AustralianRegisteredFundNumberId");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "AustralianRegisteredFundNumberId" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.TRTAMI180 = currentValue;
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion AustralianRegisteredFundNumberId

                #region ReportAmendmentDetails
                _currentXPath.Push("/tns:ReportAmendmentDetails");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "ReportAmendmentDetails" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_ReportAmendmentDetailsCollectionExists = true;
                    report.RP_ReportAmendmentDetailsCollectionCount += 1;
                    _found = true;

                    #region I
                    _currentXPath.Push("/tns:I");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "I" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.TRTAMI6 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion I

                    #region AmendmentTypeC
                    _currentXPath.Push("/tns:AmendmentTypeC");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "AmendmentTypeC" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRTAMI7 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion AmendmentTypeC

                    #region AmendmentReasonT
                    _currentXPath.Push("/tns:AmendmentReasonT");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "AmendmentReasonT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRTAMI8 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion AmendmentReasonT

                    #region AmendmentSequenceN
                    _currentXPath.Push("/tns:AmendmentSequenceN");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "AmendmentSequenceN" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRTAMI9 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion AmendmentSequenceN
                } // End of if ReportAmendmentDetails node exists

                _currentXPath.Pop();
                #endregion ReportAmendmentDetails

                #region CurrentTrust
                _currentXPath.Push("/tns:CurrentTrust");
                //3. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "CurrentTrust" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_CurrentTrustCollectionExists = true;
                    report.RP_CurrentTrustCollectionCount += 1;
                    _found = true;

                    #region OrganisationNameDetailsOrganisationalNameT
                    _currentXPath.Push("/tns:OrganisationNameDetailsOrganisationalNameT");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "OrganisationNameDetailsOrganisationalNameT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRTAMI10 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion OrganisationNameDetailsOrganisationalNameT

                    #region CurrentPostalAddressDetails
                    _currentXPath.Push("/tns:CurrentPostalAddressDetails");
                    //3. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "CurrentPostalAddressDetails" && reader.Depth == _currentXPath.Count - 1)
                    {
                        report.RP_CurrentTrust_CurrentPostalAddressDetailsCollectionExists = true;
                        report.RP_CurrentTrust_CurrentPostalAddressDetailsCollectionCount += 1;
                        _found = true;

                        #region Line1T
                        _currentXPath.Push("/tns:Line1T");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "Line1T" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.TRTAMI12 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion Line1T

                        #region Line2T
                        _currentXPath.Push("/tns:Line2T");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "Line2T" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.TRTAMI13 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion Line2T

                        #region LocalityNameT
                        _currentXPath.Push("/tns:LocalityNameT");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "LocalityNameT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.TRTAMI14 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LocalityNameT

                        #region StateOrTerritoryC
                        _currentXPath.Push("/tns:StateOrTerritoryC");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "StateOrTerritoryC" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.TRTAMI15 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion StateOrTerritoryC

                        #region PostcodeT
                        _currentXPath.Push("/tns:PostcodeT");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "PostcodeT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.TRTAMI16 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion PostcodeT

                        #region CountryC
                        _currentXPath.Push("/tns:CountryC");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "CountryC" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.TRTAMI17 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion CountryC
                    } // End of if CurrentPostalAddressDetails node exists

                    _currentXPath.Pop();
                    #endregion CurrentPostalAddressDetails
                } // End of if CurrentTrust node exists

                _currentXPath.Pop();
                #endregion CurrentTrust

                #region PreviousTrust
                _currentXPath.Push("/tns:PreviousTrust");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "PreviousTrust" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_PreviousTrustCollectionExists = true;
                    report.RP_PreviousTrustCollectionCount += 1;
                    _found = true;

                    #region OrganisationNameDetailsOrganisationalNameT
                    _currentXPath.Push("/tns:OrganisationNameDetailsOrganisationalNameT");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "OrganisationNameDetailsOrganisationalNameT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRTAMI11 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion OrganisationNameDetailsOrganisationalNameT

                    #region PreviousPostalAddressDetails
                    _currentXPath.Push("/tns:PreviousPostalAddressDetails");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "PreviousPostalAddressDetails" && reader.Depth == _currentXPath.Count - 1)
                    {
                        report.RP_PreviousTrust_PreviousPostalAddressDetailsCollectionExists = true;
                        report.RP_PreviousTrust_PreviousPostalAddressDetailsCollectionCount += 1;
                        _found = true;

                        #region Line1T
                        _currentXPath.Push("/tns:Line1T");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "Line1T" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.TRTAMI18 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion Line1T

                        #region Line2T
                        _currentXPath.Push("/tns:Line2T");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "Line2T" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.TRTAMI19 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion Line2T

                        #region LocalityNameT
                        _currentXPath.Push("/tns:LocalityNameT");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "LocalityNameT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.TRTAMI20 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion LocalityNameT

                        #region StateOrTerritoryC
                        _currentXPath.Push("/tns:StateOrTerritoryC");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "StateOrTerritoryC" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.TRTAMI21 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion StateOrTerritoryC

                        #region PostcodeT
                        _currentXPath.Push("/tns:PostcodeT");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "PostcodeT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.TRTAMI22 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion PostcodeT

                        #region CountryC
                        _currentXPath.Push("/tns:CountryC");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "CountryC" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.TRTAMI23 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion CountryC
                    } // End of if PreviousPostalAddressDetails node exists

                    _currentXPath.Pop();
                    #endregion PreviousPostalAddressDetails
                } // End of if PreviousTrust node exists

                _currentXPath.Pop();
                #endregion PreviousTrust

                #region TrusteeDetails
                _currentXPath.Push("/tns:TrusteeDetails");
                //3. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "TrusteeDetails" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_TrusteeDetailsCollectionExists = true;
                    report.RP_TrusteeDetailsCollectionCount += 1;
                    _found = true;

                    #region OrganisationNameDetailsOrganisationalNameT
                    _currentXPath.Push("/tns:OrganisationNameDetailsOrganisationalNameT");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "OrganisationNameDetailsOrganisationalNameT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRTAMI28 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion OrganisationNameDetailsOrganisationalNameT

                    #region AustralianBusinessNumberId
                    _currentXPath.Push("/tns:AustralianBusinessNumberId");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "AustralianBusinessNumberId" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRTAMI29 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion AustralianBusinessNumberId

                    #region AustralianCompanyNumberId
                    _currentXPath.Push("/tns:AustralianCompanyNumberId");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "AustralianCompanyNumberId" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRTAMI191 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion AustralianCompanyNumberId

                    #region DirectorDetails
                    _currentXPath.Push("/tns:DirectorDetails");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "DirectorDetails" && reader.Depth == _currentXPath.Count - 1)
                    {
                        report.RP_TrusteeDetails_DirectorDetailsCollectionExists = true;
                        report.RP_TrusteeDetails_DirectorDetailsCollectionCount += 1;
                        _found = true;

                        #region OrganisationNameDetailsOrganisationalNameT
                        _currentXPath.Push("/tns:OrganisationNameDetailsOrganisationalNameT");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "OrganisationNameDetailsOrganisationalNameT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.TRTAMI197 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion OrganisationNameDetailsOrganisationalNameT

                        #region AustralianBusinessNumberId
                        _currentXPath.Push("/tns:AustralianBusinessNumberId");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "AustralianBusinessNumberId" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.TRTAMI198 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion AustralianBusinessNumberId

                        #region DirectorElectronicContactDetails
                        _currentXPath.Push("/tns:DirectorElectronicContactDetails");
                        //3. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "DirectorElectronicContactDetails" && reader.Depth == _currentXPath.Count - 1)
                        {
                            report.RP_TrusteeDetails_DirectorDetails_DirectorElectronicContactDetailsCollectionExists = true;
                            report.RP_TrusteeDetails_DirectorDetails_DirectorElectronicContactDetailsCollectionCount += 1;
                            _found = true;

                            #region TelephoneAreaC
                            _currentXPath.Push("/tns:TelephoneAreaC");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "TelephoneAreaC" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        report.TRTAMI199 = currentValue;
                                    }
                                }
                                _found = true;
                            }

                            _currentXPath.Pop();
                            #endregion TelephoneAreaC

                            #region TelephoneMinimalN
                            _currentXPath.Push("/tns:TelephoneMinimalN");
                            //6. use case
                            if (ReadToNextElement(reader, true) && reader.LocalName == "TelephoneMinimalN" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        report.TRTAMI200 = currentValue;
                                    }
                                }
                                _found = true;
                            }

                            _currentXPath.Pop();
                            #endregion TelephoneMinimalN
                        } // End of if DirectorElectronicContactDetails node exists

                        _currentXPath.Pop();
                        #endregion DirectorElectronicContactDetails
                    } // End of if DirectorDetails node exists

                    _currentXPath.Pop();
                    #endregion DirectorDetails

                    #region PersonNameDetails
                    _currentXPath.Push("/tns:PersonNameDetails");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "PersonNameDetails" && reader.Depth == _currentXPath.Count - 1)
                    {
                        report.RP_TrusteeDetails_PersonNameDetailsCollectionExists = true;
                        report.RP_TrusteeDetails_PersonNameDetailsCollectionCount += 1;
                        _found = true;

                        #region TitleT
                        _currentXPath.Push("/tns:TitleT");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "TitleT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.TRTAMI24 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion TitleT

                        #region FamilyNameT
                        _currentXPath.Push("/tns:FamilyNameT");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "FamilyNameT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.TRTAMI25 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion FamilyNameT

                        #region GivenNameT
                        _currentXPath.Push("/tns:GivenNameT");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "GivenNameT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.TRTAMI26 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion GivenNameT

                        #region OtherGivenNameT
                        _currentXPath.Push("/tns:OtherGivenNameT");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "OtherGivenNameT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.TRTAMI27 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion OtherGivenNameT

                        #region TrusteeElectronicContactDetails
                        _currentXPath.Push("/tns:TrusteeElectronicContactDetails");
                        //3. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "TrusteeElectronicContactDetails" && reader.Depth == _currentXPath.Count - 1)
                        {
                            report.RP_TrusteeDetails_PersonNameDetails_TrusteeElectronicContactDetailsCollectionExists = true;
                            report.RP_TrusteeDetails_PersonNameDetails_TrusteeElectronicContactDetailsCollectionCount += 1;
                            _found = true;

                            #region TelephoneAreaC
                            _currentXPath.Push("/tns:TelephoneAreaC");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "TelephoneAreaC" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        report.TRTAMI30 = currentValue;
                                    }
                                }
                                _found = true;
                            }

                            _currentXPath.Pop();
                            #endregion TelephoneAreaC

                            #region TelephoneMinimalN
                            _currentXPath.Push("/tns:TelephoneMinimalN");
                            //6. use case
                            if (ReadToNextElement(reader, true) && reader.LocalName == "TelephoneMinimalN" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        report.TRTAMI31 = currentValue;
                                    }
                                }
                                _found = true;
                            }

                            _currentXPath.Pop();
                            #endregion TelephoneMinimalN
                        } // End of if TrusteeElectronicContactDetails node exists

                        _currentXPath.Pop();
                        #endregion TrusteeElectronicContactDetails
                    } // End of if PersonNameDetails node exists

                    _currentXPath.Pop();
                    #endregion PersonNameDetails
                } // End of if TrusteeDetails node exists

                _currentXPath.Pop();
                #endregion TrusteeDetails

                #region AMITEligibility
                _currentXPath.Push("/tns:AMITEligibility");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "AMITEligibility" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_AMITEligibilityCollectionExists = true;
                    report.RP_AMITEligibilityCollectionCount += 1;
                    _found = true;

                    #region IncomeTaxAttributionManagedInvestmentEligibilityI
                    _currentXPath.Push("/tns:IncomeTaxAttributionManagedInvestmentEligibilityI");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "IncomeTaxAttributionManagedInvestmentEligibilityI" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.TRTAMI201 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeTaxAttributionManagedInvestmentEligibilityI

                    #region IncomeTaxEligibleInvestmentBusinessSafeHarbourI
                    _currentXPath.Push("/tns:IncomeTaxEligibleInvestmentBusinessSafeHarbourI");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "IncomeTaxEligibleInvestmentBusinessSafeHarbourI" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.TRTAMI181 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeTaxEligibleInvestmentBusinessSafeHarbourI

                    #region IncomeTaxSafeHarbourPeriodC
                    _currentXPath.Push("/tns:IncomeTaxSafeHarbourPeriodC");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxSafeHarbourPeriodC" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRTAMI182 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeTaxSafeHarbourPeriodC
                } // End of if AMITEligibility node exists

                _currentXPath.Pop();
                #endregion AMITEligibility

                #region TrusteeLiabilities
                _currentXPath.Push("/tns:TrusteeLiabilities");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "TrusteeLiabilities" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_TrusteeLiabilitiesCollectionExists = true;
                    report.RP_TrusteeLiabilitiesCollectionCount += 1;
                    _found = true;

                    #region IncomeTaxTrustComponentDeficitOffsetA
                    _currentXPath.Push("/tns:IncomeTaxTrustComponentDeficitOffsetA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxTrustComponentDeficitOffsetA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRTAMI35 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeTaxTrustComponentDeficitOffsetA

                    #region IncomeTaxTrustShortfallIncomeA
                    _currentXPath.Push("/tns:IncomeTaxTrustShortfallIncomeA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxTrustShortfallIncomeA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRTAMI36 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeTaxTrustShortfallIncomeA

                    #region IncomeTaxTrustExcessOffsetA
                    _currentXPath.Push("/tns:IncomeTaxTrustExcessOffsetA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxTrustExcessOffsetA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRTAMI39 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeTaxTrustExcessOffsetA

                    #region IncomeTaxTrustComponentNotReflectedA
                    _currentXPath.Push("/tns:IncomeTaxTrustComponentNotReflectedA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxTrustComponentNotReflectedA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRTAMI40 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeTaxTrustComponentNotReflectedA
                } // End of if TrusteeLiabilities node exists

                _currentXPath.Pop();
                #endregion TrusteeLiabilities

                #region FinancialInstitutionAccount
                _currentXPath.Push("/tns:FinancialInstitutionAccount");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "FinancialInstitutionAccount" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_FinancialInstitutionAccountCollectionExists = true;
                    report.RP_FinancialInstitutionAccountCollectionCount += 1;
                    _found = true;

                    #region BankStateBranchN
                    _currentXPath.Push("/tns:BankStateBranchN");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "BankStateBranchN" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRTAMI41 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion BankStateBranchN

                    #region N
                    _currentXPath.Push("/tns:N");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "N" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRTAMI42 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion N

                    #region NameT
                    _currentXPath.Push("/tns:NameT");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "NameT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRTAMI43 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion NameT
                } // End of if FinancialInstitutionAccount node exists

                _currentXPath.Pop();
                #endregion FinancialInstitutionAccount

                #region TrustDetails
                _currentXPath.Push("/tns:TrustDetails");
                //3. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "TrustDetails" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_TrustDetailsCollectionExists = true;
                    report.RP_TrustDetailsCollectionCount += 1;
                    _found = true;

                    #region OrganisationDetailsSignificantGlobalEntityStatusI
                    _currentXPath.Push("/tns:OrganisationDetailsSignificantGlobalEntityStatusI");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "OrganisationDetailsSignificantGlobalEntityStatusI" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.TRTAMI162 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion OrganisationDetailsSignificantGlobalEntityStatusI

                    #region InternationalDealingsCountryByCountryReportingEntityStatusI
                    _currentXPath.Push("/tns:InternationalDealingsCountryByCountryReportingEntityStatusI");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsCountryByCountryReportingEntityStatusI" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.TRTAMI163 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion InternationalDealingsCountryByCountryReportingEntityStatusI

                    #region OrganisationControlRegisteredMemberCt
                    _currentXPath.Push("/tns:OrganisationControlRegisteredMemberCt");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "OrganisationControlRegisteredMemberCt" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRTAMI193 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion OrganisationControlRegisteredMemberCt

                    #region OrganisationDetailsMainIncomeActivityDe
                    _currentXPath.Push("/tns:OrganisationDetailsMainIncomeActivityDe");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "OrganisationDetailsMainIncomeActivityDe" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRTAMI44 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion OrganisationDetailsMainIncomeActivityDe

                    #region OrganisationDetailsOrganisationIndustry2006ExtendedC
                    _currentXPath.Push("/tns:OrganisationDetailsOrganisationIndustry2006ExtendedC");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "OrganisationDetailsOrganisationIndustry2006ExtendedC" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRTAMI45 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion OrganisationDetailsOrganisationIndustry2006ExtendedC

                    #region InternationalDealingsRelatedPartiesTransactionsExcessAggregateValueI
                    _currentXPath.Push("/tns:InternationalDealingsRelatedPartiesTransactionsExcessAggregateValueI");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "InternationalDealingsRelatedPartiesTransactionsExcessAggregateValueI" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.TRTAMI47 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion InternationalDealingsRelatedPartiesTransactionsExcessAggregateValueI

                    #region LiabilitiesThinCapitalisationProvisionsAppliedI
                    _currentXPath.Push("/tns:LiabilitiesThinCapitalisationProvisionsAppliedI");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "LiabilitiesThinCapitalisationProvisionsAppliedI" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.TRTAMI48 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion LiabilitiesThinCapitalisationProvisionsAppliedI

                    #region ExpenseInterestA
                    _currentXPath.Push("/tns:ExpenseInterestA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseInterestA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRTAMI49 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion ExpenseInterestA

                    #region ExpenseRoyaltiesA
                    _currentXPath.Push("/tns:ExpenseRoyaltiesA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseRoyaltiesA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRTAMI50 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion ExpenseRoyaltiesA

                    #region InternationalDealingsTransactionsWithTaxHavenCountriesI
                    _currentXPath.Push("/tns:InternationalDealingsTransactionsWithTaxHavenCountriesI");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "InternationalDealingsTransactionsWithTaxHavenCountriesI" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.TRTAMI137 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion InternationalDealingsTransactionsWithTaxHavenCountriesI

                    #region InternationalDealingsTransactionControlTaxHavenCountriesI
                    _currentXPath.Push("/tns:InternationalDealingsTransactionControlTaxHavenCountriesI");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "InternationalDealingsTransactionControlTaxHavenCountriesI" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.TRTAMI138 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion InternationalDealingsTransactionControlTaxHavenCountriesI
                } // End of if TrustDetails node exists

                _currentXPath.Pop();
                #endregion TrustDetails

                #region KeyFinancialInformation
                _currentXPath.Push("/tns:KeyFinancialInformation");
                //3. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "KeyFinancialInformation" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_KeyFinancialInformationCollectionExists = true;
                    report.RP_KeyFinancialInformationCollectionCount += 1;
                    _found = true;

                    #region AssetsCurrentTotalA
                    _currentXPath.Push("/tns:AssetsCurrentTotalA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "AssetsCurrentTotalA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRTAMI57 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion AssetsCurrentTotalA

                    #region AssetsTotalA
                    _currentXPath.Push("/tns:AssetsTotalA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "AssetsTotalA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRTAMI58 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion AssetsTotalA

                    #region LiabilitiesCurrentTotalA
                    _currentXPath.Push("/tns:LiabilitiesCurrentTotalA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesCurrentTotalA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRTAMI59 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion LiabilitiesCurrentTotalA

                    #region LiabilitiesTotalA
                    _currentXPath.Push("/tns:LiabilitiesTotalA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesTotalA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRTAMI60 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion LiabilitiesTotalA

                    #region IncomeTrustEstateIncomeA
                    _currentXPath.Push("/tns:IncomeTrustEstateIncomeA");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "IncomeTrustEstateIncomeA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRTAMI73 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeTrustEstateIncomeA

                    #region IncomeAggregatedTurnoverRangeC
                    _currentXPath.Push("/tns:IncomeAggregatedTurnoverRangeC");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeAggregatedTurnoverRangeC" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRTAMI160 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeAggregatedTurnoverRangeC

                    #region IncomeAggregatedTurnoverA
                    _currentXPath.Push("/tns:IncomeAggregatedTurnoverA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeAggregatedTurnoverA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRTAMI161 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeAggregatedTurnoverA
                } // End of if KeyFinancialInformation node exists

                _currentXPath.Pop();
                #endregion KeyFinancialInformation

                #region MultiClassTreatment
                _currentXPath.Push("/tns:MultiClassTreatment");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "MultiClassTreatment" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_MultiClassTreatmentCollectionExists = true;
                    report.RP_MultiClassTreatmentCollectionCount += 1;
                    _found = true;

                    #region IncomeTaxI
                    _currentXPath.Push("/tns:IncomeTaxI");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "IncomeTaxI" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.TRTAMI102 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeTaxI

                    #region IncomeTaxCt
                    _currentXPath.Push("/tns:IncomeTaxCt");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxCt" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRTAMI103 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeTaxCt

                    #region IncomeTaxMultiClassTransferI
                    _currentXPath.Push("/tns:IncomeTaxMultiClassTransferI");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxMultiClassTransferI" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.TRTAMI104 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeTaxMultiClassTransferI
                } // End of if MultiClassTreatment node exists

                _currentXPath.Pop();
                #endregion MultiClassTreatment

                #region ElectionsManagedInvestmentTrustsI
                _currentXPath.Push("/tns:ElectionsManagedInvestmentTrustsI");
                //6. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "ElectionsManagedInvestmentTrustsI" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.TRTAMI32 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion ElectionsManagedInvestmentTrustsI

                #region IncomeTaxStapledEntities
                _currentXPath.Push("/tns:IncomeTaxStapledEntities");
                //3. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "IncomeTaxStapledEntities" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_IncomeTaxStapledEntitiesCollectionExists = true;
                    report.RP_IncomeTaxStapledEntitiesCollectionCount += 1;
                    _found = true;

                    #region AttributionManagedInvestmentTrustStapledStructureI
                    _currentXPath.Push("/tns:AttributionManagedInvestmentTrustStapledStructureI");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "AttributionManagedInvestmentTrustStapledStructureI" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.TRTAMI112 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion AttributionManagedInvestmentTrustStapledStructureI

                    #region AustralianBusinessNumberIdCollection
                    _currentXPath.Push("/tns:AustralianBusinessNumberIdCollection");
                    if (ReadToNextElement(reader, false) && reader.LocalName == "AustralianBusinessNumberIdCollection" && reader.Depth == _currentXPath.Count - 1)
                    {
                        report.TRTAMI139Collection = new List<string>();
                        ReadNext(reader);
                        _currentXPath.Push("/tns:AustralianBusinessNumberId");
                        // 5. use case
                        while (ReadToNextElement(reader, false) && reader.LocalName == "AustralianBusinessNumberId" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before the currentXPath is updated.
                                    report.TRTAMI139Collection.Add(currentValue);
                                }
                            }
                            _found = true;
                        }
                        _currentXPath.Pop();
                    }
                    _currentXPath.Pop();
                    #endregion AustralianBusinessNumberIdCollection
                } // End of if IncomeTaxStapledEntities node exists

                _currentXPath.Pop();
                #endregion IncomeTaxStapledEntities

                #region AssetsInvestmentCrossOtherCCIVSubfundI
                _currentXPath.Push("/tns:AssetsInvestmentCrossOtherCCIVSubfundI");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "AssetsInvestmentCrossOtherCCIVSubfundI" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.TRTAMI189 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion AssetsInvestmentCrossOtherCCIVSubfundI

                #region CapitalAllowances
                _currentXPath.Push("/tns:CapitalAllowances");
                //3. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "CapitalAllowances" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_CapitalAllowancesCollectionExists = true;
                    report.RP_CapitalAllowancesCollectionCount += 1;
                    _found = true;

                    #region AssetsPropertyPlantAndEquipmentDepreciatingAssetsFirstDeductedSelfAssessedEffectiveLifeI
                    _currentXPath.Push("/tns:AssetsPropertyPlantAndEquipmentDepreciatingAssetsFirstDeductedSelfAssessedEffectiveLifeI");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "AssetsPropertyPlantAndEquipmentDepreciatingAssetsFirstDeductedSelfAssessedEffectiveLifeI" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.TRTAMI63 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion AssetsPropertyPlantAndEquipmentDepreciatingAssetsFirstDeductedSelfAssessedEffectiveLifeI

                    #region AssetsPropertyPlantAndEquipmentDepreciatingAssetsEffectiveLifeRecalculationI
                    _currentXPath.Push("/tns:AssetsPropertyPlantAndEquipmentDepreciatingAssetsEffectiveLifeRecalculationI");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "AssetsPropertyPlantAndEquipmentDepreciatingAssetsEffectiveLifeRecalculationI" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.TRTAMI64 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion AssetsPropertyPlantAndEquipmentDepreciatingAssetsEffectiveLifeRecalculationI

                    #region ExpenseDepreciationAllowableDeductionA
                    _currentXPath.Push("/tns:ExpenseDepreciationAllowableDeductionA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseDepreciationAllowableDeductionA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRTAMI118 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion ExpenseDepreciationAllowableDeductionA

                    #region ExpenseCapitalExpenditureSpecifiedAllowableDeductionA
                    _currentXPath.Push("/tns:ExpenseCapitalExpenditureSpecifiedAllowableDeductionA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseCapitalExpenditureSpecifiedAllowableDeductionA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRTAMI71 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion ExpenseCapitalExpenditureSpecifiedAllowableDeductionA

                    #region ExpenseCapitalWorksDeductionA
                    _currentXPath.Push("/tns:ExpenseCapitalWorksDeductionA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseCapitalWorksDeductionA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRTAMI140 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion ExpenseCapitalWorksDeductionA

                    #region ExpenseBuildToRentCapitalWorksDeductionA
                    _currentXPath.Push("/tns:ExpenseBuildToRentCapitalWorksDeductionA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseBuildToRentCapitalWorksDeductionA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRTAMI210 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion ExpenseBuildToRentCapitalWorksDeductionA
                } // End of if CapitalAllowances node exists

                _currentXPath.Pop();
                #endregion CapitalAllowances

                #region WithholdingObligations
                _currentXPath.Push("/tns:WithholdingObligations");
                //3. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "WithholdingObligations" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_WithholdingObligationsCollectionExists = true;
                    report.RP_WithholdingObligationsCollectionCount += 1;
                    _found = true;

                    #region OrganisationDetailsWithholdingManagedInvestmentTrustI
                    _currentXPath.Push("/tns:OrganisationDetailsWithholdingManagedInvestmentTrustI");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "OrganisationDetailsWithholdingManagedInvestmentTrustI" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.TRTAMI211 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion OrganisationDetailsWithholdingManagedInvestmentTrustI

                    #region ExpenseDividendInterestRoyaltyA
                    _currentXPath.Push("/tns:ExpenseDividendInterestRoyaltyA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseDividendInterestRoyaltyA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRTAMI121 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion ExpenseDividendInterestRoyaltyA

                    #region IncomeTrustDistributionsA
                    _currentXPath.Push("/tns:IncomeTrustDistributionsA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTrustDistributionsA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRTAMI108 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeTrustDistributionsA
                } // End of if WithholdingObligations node exists

                _currentXPath.Pop();
                #endregion WithholdingObligations

                #region ExpenseReturnsPaidDeductionA
                _currentXPath.Push("/tns:ExpenseReturnsPaidDeductionA");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseReturnsPaidDeductionA" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                            {
                                report.TRTAMI111 = currentDecimalValue;
                            }
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion ExpenseReturnsPaidDeductionA

                #region Division6CAmounts
                _currentXPath.Push("/tns:Division6CAmounts");
                //3. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "Division6CAmounts" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_Division6CAmountsCollectionExists = true;
                    report.RP_Division6CAmountsCollectionCount += 1;
                    _found = true;

                    #region IncomeTaxDeductionEligibleInvestmentBusinessC
                    _currentXPath.Push("/tns:IncomeTaxDeductionEligibleInvestmentBusinessC");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxDeductionEligibleInvestmentBusinessC" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRTAMI113 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeTaxDeductionEligibleInvestmentBusinessC

                    #region IncomeTaxDeductionCreditsEligibleTotalA
                    _currentXPath.Push("/tns:IncomeTaxDeductionCreditsEligibleTotalA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxDeductionCreditsEligibleTotalA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRTAMI114 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeTaxDeductionCreditsEligibleTotalA

                    #region IncomeTaxRentalSafeHarbourI
                    _currentXPath.Push("/tns:IncomeTaxRentalSafeHarbourI");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "IncomeTaxRentalSafeHarbourI" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.TRTAMI116 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeTaxRentalSafeHarbourI

                    #region IncomeTaxEligibleInvestmentBusinessSafeHarbourI
                    _currentXPath.Push("/tns:IncomeTaxEligibleInvestmentBusinessSafeHarbourI");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "IncomeTaxEligibleInvestmentBusinessSafeHarbourI" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.TRTAMI115 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeTaxEligibleInvestmentBusinessSafeHarbourI

                    #region IncomeTaxDeductionNonEligibleInvestmentBusinessPercentageC
                    _currentXPath.Push("/tns:IncomeTaxDeductionNonEligibleInvestmentBusinessPercentageC");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxDeductionNonEligibleInvestmentBusinessPercentageC" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRTAMI141 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeTaxDeductionNonEligibleInvestmentBusinessPercentageC
                } // End of if Division6CAmounts node exists

                _currentXPath.Pop();
                #endregion Division6CAmounts

                #region RelatedEntitiesPayments
                _currentXPath.Push("/tns:RelatedEntitiesPayments");
                //3. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "RelatedEntitiesPayments" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_RelatedEntitiesPaymentsCollectionExists = true;
                    report.RP_RelatedEntitiesPaymentsCollectionCount += 1;
                    _found = true;

                    #region IncomeTaxAttributionManagedInvestmentTrustRelatedEntitiesPaymentI
                    _currentXPath.Push("/tns:IncomeTaxAttributionManagedInvestmentTrustRelatedEntitiesPaymentI");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "IncomeTaxAttributionManagedInvestmentTrustRelatedEntitiesPaymentI" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.TRTAMI117 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeTaxAttributionManagedInvestmentTrustRelatedEntitiesPaymentI

                    #region IncomeTaxReceivedA
                    _currentXPath.Push("/tns:IncomeTaxReceivedA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxReceivedA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRTAMI142 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion IncomeTaxReceivedA
                } // End of if RelatedEntitiesPayments node exists

                _currentXPath.Pop();
                #endregion RelatedEntitiesPayments

                #region OtherReportingInformation
                _currentXPath.Push("/tns:OtherReportingInformation");
                //3. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "OtherReportingInformation" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_OtherReportingInformationCollectionExists = true;
                    report.RP_OtherReportingInformationCollectionCount += 1;
                    _found = true;

                    #region StatementSummaryPaymentRefundOrNilC
                    _currentXPath.Push("/tns:StatementSummaryPaymentRefundOrNilC");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "StatementSummaryPaymentRefundOrNilC" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRTAMI33 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion StatementSummaryPaymentRefundOrNilC

                    #region LodgmentFinalReturnI
                    _currentXPath.Push("/tns:LodgmentFinalReturnI");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "LodgmentFinalReturnI" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.TRTAMI34 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion LodgmentFinalReturnI

                    #region RegulatoryDisclosuresGeneralInformationAboutFinancialStatementsT
                    _currentXPath.Push("/tns:RegulatoryDisclosuresGeneralInformationAboutFinancialStatementsT");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "RegulatoryDisclosuresGeneralInformationAboutFinancialStatementsT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRTAMI101 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion RegulatoryDisclosuresGeneralInformationAboutFinancialStatementsT

                    #region CompletionHoursN
                    _currentXPath.Push("/tns:CompletionHoursN");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "CompletionHoursN" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRTAMI74 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion CompletionHoursN
                } // End of if OtherReportingInformation node exists

                _currentXPath.Pop();
                #endregion OtherReportingInformation

                #region StatementOfAttributionDetails

                _currentXPath.Push("/tns:StatementOfAttributionDetailsCollection");
                // 4. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "StatementOfAttributionDetailsCollection" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_StatementOfAttributionDetailsCollection = new List<TRTAMI2025.RP_StatementOfAttributionDetails>();
                    report.RP_StatementOfAttributionDetailsCollectionExists = true;

                    _found = true;

                    while (ReadToNextElement(reader, false) && reader.LocalName == "StatementOfAttributionDetails")
                    {
                        _found = true;

                        TRTAMI2025.RP_StatementOfAttributionDetails statementOfAttributionDetails = new TRTAMI2025.RP_StatementOfAttributionDetails();
                        report.RP_StatementOfAttributionDetailsCollection.Add(statementOfAttributionDetails);
                        report.RP_StatementOfAttributionDetailsCollectionCount += 1;
                        statementOfAttributionDetails.OccurrenceIndex = report.RP_StatementOfAttributionDetailsCollectionCount;

                        _currentXPath.Push("/tns:StatementOfAttributionDetails[" + report.RP_StatementOfAttributionDetailsCollectionCount + "]");

                        #region TaxFileNumberId
                        _currentXPath.Push("/tns:TaxFileNumberId");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "TaxFileNumberId" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    statementOfAttributionDetails.TRTAMI75 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion TaxFileNumberId

                        #region IndividualPersonNameDetails
                        _currentXPath.Push("/tns:IndividualPersonNameDetails");
                        //3. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "IndividualPersonNameDetails" && reader.Depth == _currentXPath.Count - 1)
                        {
                            statementOfAttributionDetails.RP_StatementOfAttributionDetails_IndividualPersonNameDetailsCollectionExists = true;
                            statementOfAttributionDetails.RP_StatementOfAttributionDetails_IndividualPersonNameDetailsCollectionCount += 1;
                            _found = true;

                            #region TitleT
                            _currentXPath.Push("/tns:TitleT");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "TitleT" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        statementOfAttributionDetails.TRTAMI76 = currentValue;
                                    }
                                }
                                _found = true;
                            }

                            _currentXPath.Pop();
                            #endregion TitleT

                            #region FamilyNameT
                            _currentXPath.Push("/tns:FamilyNameT");
                            //6. use case
                            if (ReadToNextElement(reader, true) && reader.LocalName == "FamilyNameT" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        statementOfAttributionDetails.TRTAMI77 = currentValue;
                                    }
                                }
                                _found = true;
                            }

                            _currentXPath.Pop();
                            #endregion FamilyNameT

                            #region GivenNameT
                            _currentXPath.Push("/tns:GivenNameT");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "GivenNameT" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        statementOfAttributionDetails.TRTAMI78 = currentValue;
                                    }
                                }
                                _found = true;
                            }

                            _currentXPath.Pop();
                            #endregion GivenNameT

                            #region OtherGivenNameT
                            _currentXPath.Push("/tns:OtherGivenNameT");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "OtherGivenNameT" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        statementOfAttributionDetails.TRTAMI79 = currentValue;
                                    }
                                }
                                _found = true;
                            }

                            _currentXPath.Pop();
                            #endregion OtherGivenNameT

                            #region PersonDemographicDetailsBirthD
                            _currentXPath.Push("/tns:PersonDemographicDetailsBirthD");
                            //6. use case
                            if (ReadToNextElement(reader, true) && reader.LocalName == "PersonDemographicDetailsBirthD" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                                        {
                                            statementOfAttributionDetails.TRTAMI124 = currentDateTimeValue.Date;
                                        }
                                    }
                                }
                                _found = true;
                            }

                            _currentXPath.Pop();
                            #endregion PersonDemographicDetailsBirthD

                            #region IndividualAddressDetails
                            _currentXPath.Push("/tns:IndividualAddressDetails");
                            //3. use case
                            if (ReadToNextElement(reader, true) && reader.LocalName == "IndividualAddressDetails" && reader.Depth == _currentXPath.Count - 1)
                            {
                                statementOfAttributionDetails.RP_StatementOfAttributionDetails_IndividualPersonNameDetails_IndividualAddressDetailsCollectionExists = true;
                                statementOfAttributionDetails.RP_StatementOfAttributionDetails_IndividualPersonNameDetails_IndividualAddressDetailsCollectionCount += 1;
                                _found = true;

                                #region Line1T
                                _currentXPath.Push("/tns:Line1T");
                                //6. use case
                                if (ReadToNextElement(reader, true) && reader.LocalName == "Line1T" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            statementOfAttributionDetails.TRTAMI125 = currentValue;
                                        }
                                    }
                                    _found = true;
                                }

                                _currentXPath.Pop();
                                #endregion Line1T

                                #region Line2T
                                _currentXPath.Push("/tns:Line2T");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "Line2T" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            statementOfAttributionDetails.TRTAMI126 = currentValue;
                                        }
                                    }
                                    _found = true;
                                }

                                _currentXPath.Pop();
                                #endregion Line2T

                                #region LocalityNameT
                                _currentXPath.Push("/tns:LocalityNameT");
                                //6. use case
                                if (ReadToNextElement(reader, true) && reader.LocalName == "LocalityNameT" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            statementOfAttributionDetails.TRTAMI127 = currentValue;
                                        }
                                    }
                                    _found = true;
                                }

                                _currentXPath.Pop();
                                #endregion LocalityNameT

                                #region PostalCodeT
                                _currentXPath.Push("/tns:PostalCodeT");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "PostalCodeT" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            statementOfAttributionDetails.TRTAMI143 = currentValue;
                                        }
                                    }
                                    _found = true;
                                }

                                _currentXPath.Pop();
                                #endregion PostalCodeT

                                #region CountryC
                                _currentXPath.Push("/tns:CountryC");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "CountryC" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            statementOfAttributionDetails.TRTAMI130 = currentValue;
                                        }
                                    }
                                    _found = true;
                                }

                                _currentXPath.Pop();
                                #endregion CountryC
                            } // End of if IndividualAddressDetails node exists

                            _currentXPath.Pop();
                            #endregion IndividualAddressDetails
                        } // End of if IndividualPersonNameDetails node exists

                        _currentXPath.Pop();
                        #endregion IndividualPersonNameDetails

                        #region NonIndividualDetails
                        _currentXPath.Push("/tns:NonIndividualDetails");
                        //3. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "NonIndividualDetails" && reader.Depth == _currentXPath.Count - 1)
                        {
                            statementOfAttributionDetails.RP_StatementOfAttributionDetails_NonIndividualDetailsCollectionExists = true;
                            statementOfAttributionDetails.RP_StatementOfAttributionDetails_NonIndividualDetailsCollectionCount += 1;
                            _found = true;

                            #region OrganisationNameDetailsOrganisationalNameT
                            _currentXPath.Push("/tns:OrganisationNameDetailsOrganisationalNameT");
                            //6. use case
                            if (ReadToNextElement(reader, true) && reader.LocalName == "OrganisationNameDetailsOrganisationalNameT" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        statementOfAttributionDetails.TRTAMI80 = currentValue;
                                    }
                                }
                                _found = true;
                            }

                            _currentXPath.Pop();
                            #endregion OrganisationNameDetailsOrganisationalNameT

                            #region NonIndividualAddressDetails
                            _currentXPath.Push("/tns:NonIndividualAddressDetails");
                            //3. use case
                            if (ReadToNextElement(reader, true) && reader.LocalName == "NonIndividualAddressDetails" && reader.Depth == _currentXPath.Count - 1)
                            {
                                statementOfAttributionDetails.RP_StatementOfAttributionDetails_NonIndividualDetails_NonIndividualAddressDetailsCollectionExists = true;
                                statementOfAttributionDetails.RP_StatementOfAttributionDetails_NonIndividualDetails_NonIndividualAddressDetailsCollectionCount += 1;
                                _found = true;

                                #region Line1T
                                _currentXPath.Push("/tns:Line1T");
                                //6. use case
                                if (ReadToNextElement(reader, true) && reader.LocalName == "Line1T" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            statementOfAttributionDetails.TRTAMI131 = currentValue;
                                        }
                                    }
                                    _found = true;
                                }

                                _currentXPath.Pop();
                                #endregion Line1T

                                #region Line2T
                                _currentXPath.Push("/tns:Line2T");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "Line2T" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            statementOfAttributionDetails.TRTAMI132 = currentValue;
                                        }
                                    }
                                    _found = true;
                                }

                                _currentXPath.Pop();
                                #endregion Line2T

                                #region LocalityNameT
                                _currentXPath.Push("/tns:LocalityNameT");
                                //6. use case
                                if (ReadToNextElement(reader, true) && reader.LocalName == "LocalityNameT" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            statementOfAttributionDetails.TRTAMI133 = currentValue;
                                        }
                                    }
                                    _found = true;
                                }

                                _currentXPath.Pop();
                                #endregion LocalityNameT

                                #region PostalCodeT
                                _currentXPath.Push("/tns:PostalCodeT");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "PostalCodeT" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            statementOfAttributionDetails.TRTAMI144 = currentValue;
                                        }
                                    }
                                    _found = true;
                                }

                                _currentXPath.Pop();
                                #endregion PostalCodeT

                                #region CountryC
                                _currentXPath.Push("/tns:CountryC");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "CountryC" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            statementOfAttributionDetails.TRTAMI136 = currentValue;
                                        }
                                    }
                                    _found = true;
                                }

                                _currentXPath.Pop();
                                #endregion CountryC
                            } // End of if NonIndividualAddressDetails node exists

                            _currentXPath.Pop();
                            #endregion NonIndividualAddressDetails
                        } // End of if NonIndividualDetails node exists

                        _currentXPath.Pop();
                        #endregion NonIndividualDetails

                        #region AssessmentDetails
                        _currentXPath.Push("/tns:AssessmentDetails");
                        //3. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "AssessmentDetails" && reader.Depth == _currentXPath.Count - 1)
                        {
                            statementOfAttributionDetails.RP_StatementOfAttributionDetails_AssessmentDetailsCollectionExists = true;
                            statementOfAttributionDetails.RP_StatementOfAttributionDetails_AssessmentDetailsCollectionCount += 1;
                            _found = true;

                            #region IncomeTaxAssessmentCalculationC
                            _currentXPath.Push("/tns:IncomeTaxAssessmentCalculationC");
                            //6. use case
                            if (ReadToNextElement(reader, true) && reader.LocalName == "IncomeTaxAssessmentCalculationC" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        statementOfAttributionDetails.TRTAMI81 = currentValue;
                                    }
                                }
                                _found = true;
                            }

                            _currentXPath.Pop();
                            #endregion IncomeTaxAssessmentCalculationC

                            #region InternationalDealingsAssessableIncomeNonResidentBeneficiaryA
                            _currentXPath.Push("/tns:InternationalDealingsAssessableIncomeNonResidentBeneficiaryA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsAssessableIncomeNonResidentBeneficiaryA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            statementOfAttributionDetails.TRTAMI82 = currentDecimalValue;
                                        }
                                    }
                                }
                                _found = true;
                            }

                            _currentXPath.Pop();
                            #endregion InternationalDealingsAssessableIncomeNonResidentBeneficiaryA
                        } // End of if AssessmentDetails node exists

                        _currentXPath.Pop();
                        #endregion AssessmentDetails
                        _currentXPath.Pop();
                    } // End of while loop for StatementOfAttributionDetails
                } // End of if StatementOfAttributionDetailsCollection node exists

                _currentXPath.Pop();
                #endregion StatementOfAttributionDetails

                #region ContactDetails
                _currentXPath.Push("/tns:ContactDetails");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "ContactDetails" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_ContactDetailsCollectionExists = true;
                    report.RP_ContactDetailsCollectionCount += 1;
                    _found = true;

                    #region PersonUnstructuredNameFullNameT
                    _currentXPath.Push("/tns:PersonUnstructuredNameFullNameT");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "PersonUnstructuredNameFullNameT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRTAMI94 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion PersonUnstructuredNameFullNameT

                    #region PersonNameDetailsPositionT
                    _currentXPath.Push("/tns:PersonNameDetailsPositionT");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "PersonNameDetailsPositionT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRTAMI95 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion PersonNameDetailsPositionT

                    #region ElectronicContactTelephoneAreaC
                    _currentXPath.Push("/tns:ElectronicContactTelephoneAreaC");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "ElectronicContactTelephoneAreaC" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRTAMI96 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion ElectronicContactTelephoneAreaC

                    #region ElectronicContactTelephoneMinimalN
                    _currentXPath.Push("/tns:ElectronicContactTelephoneMinimalN");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "ElectronicContactTelephoneMinimalN" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRTAMI97 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion ElectronicContactTelephoneMinimalN

                    #region Declaration
                    _currentXPath.Push("/tns:Declaration");
                    //3. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "Declaration" && reader.Depth == _currentXPath.Count - 1)
                    {
                        report.RP_ContactDetails_DeclarationCollectionExists = true;
                        report.RP_ContactDetails_DeclarationCollectionCount += 1;
                        _found = true;

                        #region SignatureD
                        _currentXPath.Push("/tns:SignatureD");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "SignatureD" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                                    {
                                        report.TRTAMI98 = currentDateTimeValue.Date;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion SignatureD

                        #region StatementTypeC
                        _currentXPath.Push("/tns:StatementTypeC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "StatementTypeC" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.TRTAMI171 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion StatementTypeC

                        #region StatementAcceptedI
                        _currentXPath.Push("/tns:StatementAcceptedI");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "StatementAcceptedI" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    currentBooleanValue = ToBoolean(currentValue);
                                    if (currentBooleanValue != null)
                                    {
                                        report.TRTAMI99 = currentBooleanValue;
                                    }
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion StatementAcceptedI

                        #region SignatoryIdentifierT
                        _currentXPath.Push("/tns:SignatoryIdentifierT");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "SignatoryIdentifierT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.TRTAMI100 = currentValue;
                                }
                            }
                            _found = true;
                        }

                        _currentXPath.Pop();
                        #endregion SignatoryIdentifierT
                    } // End of if Declaration node exists

                    _currentXPath.Pop();
                    #endregion Declaration
                } // End of if ContactDetails node exists

                _currentXPath.Pop();
                #endregion ContactDetails
            } // End of if RP node exists

            _currentXPath.Pop();
            #endregion RP

            #region INT
            _currentXPath.Push("/tns:INT");
            //3. use case
            if (ReadToNextElement(reader, false) && reader.LocalName == "INT" && reader.Depth == _currentXPath.Count - 1)
            {
                report.IntCollectionExists = true;
                report.IntCollectionCount += 1;
                _found = true;

                #region TaxAgentClientReferenceT
                _currentXPath.Push("/tns:TaxAgentClientReferenceT");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "TaxAgentClientReferenceT" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.TRTAMI90 = currentValue;
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion TaxAgentClientReferenceT

                #region TaxAgentNumberId
                _currentXPath.Push("/tns:TaxAgentNumberId");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "TaxAgentNumberId" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.TRTAMI91 = currentValue;
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion TaxAgentNumberId

                #region AustralianBusinessNumberId
                _currentXPath.Push("/tns:AustralianBusinessNumberId");
                //6. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "AustralianBusinessNumberId" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.TRTAMI92 = currentValue;
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion AustralianBusinessNumberId

                #region PersonUnstructuredNameFullNameT
                _currentXPath.Push("/tns:PersonUnstructuredNameFullNameT");
                //6. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "PersonUnstructuredNameFullNameT" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.TRTAMI83 = currentValue;
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion PersonUnstructuredNameFullNameT

                #region PersonNameDetailsPositionT
                _currentXPath.Push("/tns:PersonNameDetailsPositionT");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "PersonNameDetailsPositionT" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.TRTAMI84 = currentValue;
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion PersonNameDetailsPositionT

                #region ElectronicContactTelephoneAreaC
                _currentXPath.Push("/tns:ElectronicContactTelephoneAreaC");
                //6. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "ElectronicContactTelephoneAreaC" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.TRTAMI85 = currentValue;
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion ElectronicContactTelephoneAreaC

                #region ElectronicContactTelephoneMinimalN
                _currentXPath.Push("/tns:ElectronicContactTelephoneMinimalN");
                //6. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "ElectronicContactTelephoneMinimalN" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.TRTAMI86 = currentValue;
                        }
                    }
                    _found = true;
                }

                _currentXPath.Pop();
                #endregion ElectronicContactTelephoneMinimalN

                #region Declaration
                _currentXPath.Push("/tns:Declaration");
                //3. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "Declaration" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.InT_DeclarationCollectionExists = true;
                    report.InT_DeclarationCollectionCount += 1;
                    _found = true;

                    #region SignatureD
                    _currentXPath.Push("/tns:SignatureD");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "SignatureD" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                                {
                                    report.TRTAMI87 = currentDateTimeValue.Date;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion SignatureD

                    #region StatementTypeC
                    _currentXPath.Push("/tns:StatementTypeC");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "StatementTypeC" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRTAMI170 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion StatementTypeC

                    #region StatementAcceptedI
                    _currentXPath.Push("/tns:StatementAcceptedI");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "StatementAcceptedI" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.TRTAMI88 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion StatementAcceptedI

                    #region SignatoryIdentifierT
                    _currentXPath.Push("/tns:SignatoryIdentifierT");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "SignatoryIdentifierT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRTAMI89 = currentValue;
                            }
                        }
                        _found = true;
                    }

                    _currentXPath.Pop();
                    #endregion SignatoryIdentifierT
                } // End of if Declaration node exists

                _currentXPath.Pop();
                #endregion Declaration
            } // End of if INT node exists

            _currentXPath.Pop();
            #endregion INT

            while (reader.EOF != true)
                reader.Read();

            return report;
        }

        private string _elementCsvList = "RP#TargetFinancialY,RP#PeriodStartD,RP#PeriodEndD,RP#TaxFileNumberId,RP#AustralianBusinessNumberId,RP#CorporateCollectiveInvestmentVehicleSubFundI,RP#AustralianRegisteredFundNumberId,ReportAmendmentDetails#I,ReportAmendmentDetails#AmendmentTypeC,ReportAmendmentDetails#AmendmentReasonT,ReportAmendmentDetails#AmendmentSequenceN,ReportAmendmentDetails,CurrentTrust#OrganisationNameDetailsOrganisationalNameT,CurrentPostalAddressDetails#Line1T,CurrentPostalAddressDetails#Line2T,CurrentPostalAddressDetails#LocalityNameT,CurrentPostalAddressDetails#StateOrTerritoryC,CurrentPostalAddressDetails#PostcodeT,CurrentPostalAddressDetails#CountryC,CurrentPostalAddressDetails,CurrentTrust,PreviousTrust#OrganisationNameDetailsOrganisationalNameT,PreviousPostalAddressDetails#Line1T,PreviousPostalAddressDetails#Line2T,PreviousPostalAddressDetails#LocalityNameT,PreviousPostalAddressDetails#StateOrTerritoryC,PreviousPostalAddressDetails#PostcodeT,PreviousPostalAddressDetails#CountryC,PreviousPostalAddressDetails,PreviousTrust,TrusteeDetails#OrganisationNameDetailsOrganisationalNameT,TrusteeDetails#AustralianBusinessNumberId,TrusteeDetails#AustralianCompanyNumberId,DirectorDetails#OrganisationNameDetailsOrganisationalNameT,DirectorDetails#AustralianBusinessNumberId,DirectorElectronicContactDetails#TelephoneAreaC,DirectorElectronicContactDetails#TelephoneMinimalN,DirectorElectronicContactDetails,DirectorDetails,PersonNameDetails#TitleT,PersonNameDetails#FamilyNameT,PersonNameDetails#GivenNameT,PersonNameDetails#OtherGivenNameT,TrusteeElectronicContactDetails#TelephoneAreaC,TrusteeElectronicContactDetails#TelephoneMinimalN,TrusteeElectronicContactDetails,PersonNameDetails,TrusteeDetails,AMITEligibility#IncomeTaxAttributionManagedInvestmentEligibilityI,AMITEligibility#IncomeTaxEligibleInvestmentBusinessSafeHarbourI,AMITEligibility#IncomeTaxSafeHarbourPeriodC,AMITEligibility,TrusteeLiabilities#IncomeTaxTrustComponentDeficitOffsetA,TrusteeLiabilities#IncomeTaxTrustShortfallIncomeA,TrusteeLiabilities#IncomeTaxTrustExcessOffsetA,TrusteeLiabilities#IncomeTaxTrustComponentNotReflectedA,TrusteeLiabilities,FinancialInstitutionAccount#BankStateBranchN,FinancialInstitutionAccount#N,FinancialInstitutionAccount#NameT,FinancialInstitutionAccount,TrustDetails#OrganisationDetailsSignificantGlobalEntityStatusI,TrustDetails#InternationalDealingsCountryByCountryReportingEntityStatusI,TrustDetails#OrganisationControlRegisteredMemberCt,TrustDetails#OrganisationDetailsMainIncomeActivityDe,TrustDetails#OrganisationDetailsOrganisationIndustry2006ExtendedC,TrustDetails#InternationalDealingsRelatedPartiesTransactionsExcessAggregateValueI,TrustDetails#LiabilitiesThinCapitalisationProvisionsAppliedI,TrustDetails#ExpenseInterestA,TrustDetails#ExpenseRoyaltiesA,TrustDetails#InternationalDealingsTransactionsWithTaxHavenCountriesI,TrustDetails#InternationalDealingsTransactionControlTaxHavenCountriesI,TrustDetails,KeyFinancialInformation#AssetsCurrentTotalA,KeyFinancialInformation#AssetsTotalA,KeyFinancialInformation#LiabilitiesCurrentTotalA,KeyFinancialInformation#LiabilitiesTotalA,KeyFinancialInformation#IncomeTrustEstateIncomeA,KeyFinancialInformation#IncomeAggregatedTurnoverRangeC,KeyFinancialInformation#IncomeAggregatedTurnoverA,KeyFinancialInformation,MultiClassTreatment#IncomeTaxI,MultiClassTreatment#IncomeTaxCt,MultiClassTreatment#IncomeTaxMultiClassTransferI,MultiClassTreatment,RP#ElectionsManagedInvestmentTrustsI,IncomeTaxStapledEntities#AttributionManagedInvestmentTrustStapledStructureI,AustralianBusinessNumberId,AustralianBusinessNumberIdCollection,IncomeTaxStapledEntities,RP#AssetsInvestmentCrossOtherCCIVSubfundI,CapitalAllowances#AssetsPropertyPlantAndEquipmentDepreciatingAssetsFirstDeductedSelfAssessedEffectiveLifeI,CapitalAllowances#AssetsPropertyPlantAndEquipmentDepreciatingAssetsEffectiveLifeRecalculationI,CapitalAllowances#ExpenseDepreciationAllowableDeductionA,CapitalAllowances#ExpenseCapitalExpenditureSpecifiedAllowableDeductionA,CapitalAllowances#ExpenseCapitalWorksDeductionA,CapitalAllowances#ExpenseBuildToRentCapitalWorksDeductionA,CapitalAllowances,WithholdingObligations#OrganisationDetailsWithholdingManagedInvestmentTrustI,WithholdingObligations#ExpenseDividendInterestRoyaltyA,WithholdingObligations#IncomeTrustDistributionsA,WithholdingObligations,RP#ExpenseReturnsPaidDeductionA,Division6CAmounts#IncomeTaxDeductionEligibleInvestmentBusinessC,Division6CAmounts#IncomeTaxDeductionCreditsEligibleTotalA,Division6CAmounts#IncomeTaxRentalSafeHarbourI,Division6CAmounts#IncomeTaxEligibleInvestmentBusinessSafeHarbourI,Division6CAmounts#IncomeTaxDeductionNonEligibleInvestmentBusinessPercentageC,Division6CAmounts,RelatedEntitiesPayments#IncomeTaxAttributionManagedInvestmentTrustRelatedEntitiesPaymentI,RelatedEntitiesPayments#IncomeTaxReceivedA,RelatedEntitiesPayments,OtherReportingInformation#StatementSummaryPaymentRefundOrNilC,OtherReportingInformation#LodgmentFinalReturnI,OtherReportingInformation#RegulatoryDisclosuresGeneralInformationAboutFinancialStatementsT,OtherReportingInformation#CompletionHoursN,OtherReportingInformation,StatementOfAttributionDetails#TaxFileNumberId,StatementOfAttributionDetails#TitleT,StatementOfAttributionDetails#FamilyNameT,StatementOfAttributionDetails#GivenNameT,StatementOfAttributionDetails#OtherGivenNameT,StatementOfAttributionDetails#PersonDemographicDetailsBirthD,StatementOfAttributionDetails#Line1T,StatementOfAttributionDetails#Line2T,StatementOfAttributionDetails#LocalityNameT,StatementOfAttributionDetails#PostalCodeT,StatementOfAttributionDetails#CountryC,IndividualAddressDetails,IndividualPersonNameDetails,StatementOfAttributionDetails#OrganisationNameDetailsOrganisationalNameT,StatementOfAttributionDetails#Line1T,StatementOfAttributionDetails#Line2T,StatementOfAttributionDetails#LocalityNameT,StatementOfAttributionDetails#PostalCodeT,StatementOfAttributionDetails#CountryC,NonIndividualAddressDetails,NonIndividualDetails,StatementOfAttributionDetails#IncomeTaxAssessmentCalculationC,StatementOfAttributionDetails#InternationalDealingsAssessableIncomeNonResidentBeneficiaryA,AssessmentDetails,StatementOfAttributionDetails,StatementOfAttributionDetailsCollection,ContactDetails#PersonUnstructuredNameFullNameT,ContactDetails#PersonNameDetailsPositionT,ContactDetails#ElectronicContactTelephoneAreaC,ContactDetails#ElectronicContactTelephoneMinimalN,Declaration#SignatureD,Declaration#StatementTypeC,Declaration#StatementAcceptedI,Declaration#SignatoryIdentifierT,Declaration,ContactDetails,RP,INT#TaxAgentClientReferenceT,INT#TaxAgentNumberId,INT#AustralianBusinessNumberId,INT#PersonUnstructuredNameFullNameT,INT#PersonNameDetailsPositionT,INT#ElectronicContactTelephoneAreaC,INT#ElectronicContactTelephoneMinimalN,Declaration#SignatureD,Declaration#StatementTypeC,Declaration#StatementAcceptedI,Declaration#SignatoryIdentifierT,Declaration,INT";

    }
}