using System.Collections.Generic;
using DataContracts;
using System;
using System.IO;
using System.Runtime.CompilerServices;
using System.Text;
using System.Xml;
using System.Linq;
using System.Text.RegularExpressions;

namespace Ato.EN.IntegrationServices.CodeGenerationDIS
{
    public partial class DIS2018Validator
    {
        #region Manual Rules - Rules marked for manual coding

        #region VR.ATO.DIS.500004

        /*  VR.ATO.DIS.500004
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF ([DIS48] <> NULLORBLANK) AND (ANY CHARACTER OF [DIS48] <> SET(a-z,A-Z,0-9,"!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "))
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^DIS48 <> BLANK) AND (NotCharacterInSet(^DIS48, '"a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))
    
            Data Elements:
    
            ^DIS48 = RP.TrueAndCorrect:PersonNameDetails.Position.Text WHERE RP.[AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            */

        protected void VRATODIS500004(List<ProcessMessageDocument> response, DIS2018 report, bool isStandalone)
        {
            bool assertion;
            assertion = (string.IsNullOrWhiteSpace(report.DIS48) != true && (!(IsMatch(report.DIS48, @"^[a-zA-Z0-9!@\$%&\*\(\)\-=\[\];:'"",\.\?/ ]*$", RegexOptions.IgnoreCase)) || report.DIS48.Contains(Environment.NewLine) || IsMatch(report.DIS48, "\n")));
            if (assertion)
            {
                var processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.DIS.500004",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Business address of Company - Street name and number line 1 contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.DIS.500004" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "DIS48", Value = report.DIS48
                });

                response.Add(processMessage);
            }}
        #endregion

        #region VR.ATO.DIS.500005

        /*  VR.ATO.DIS.500005
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF ([DIS64] <> NULLORBLANK) AND (ANY CHARACTER OF [DIS64] <> SET(a-z,A-Z,0-9,"!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "))
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^DIS64 <> BLANK) AND (NotCharacterInSet(^DIS64, '"a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))
    
            Data Elements:
    
            ^DIS64 = RP.TrueAndCorrect:PersonNameDetails.Position.Text WHERE RP.[AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */

        protected void VRATODIS500005(List<ProcessMessageDocument> response, DIS2018 report, bool isStandalone)
        {
            bool assertion;
            assertion = (string.IsNullOrWhiteSpace(report.DIS64) != true && (!(IsMatch(report.DIS64, @"^[a-zA-Z0-9!@\$%&\*\(\)\-=\[\];:'"",\.\?/ ]*$", RegexOptions.IgnoreCase)) || report.DIS64.Contains(Environment.NewLine) || IsMatch(report.DIS64, "\n")));
            if (assertion)
            {
                var processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.DIS.500005",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Postal address of Company - Street name and number line 1 contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.DIS.500005" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                {
                    Name = "DIS64",
                    Value = report.DIS64
                });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.DIS.500006

        /*  VR.ATO.DIS.500006
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF ([DIS7] <> NULLORBLANK) AND (ANY CHARACTER OF [DIS7] <> SET(a-z,A-Z,0-9,"!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "))
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^DIS7 <> BLANK) AND (NotCharacterInSet(^DIS7, '"a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))
    
            Data Elements:
    
            ^DIS7 = RP:Investor:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */

        protected void VRATODIS500006(List<ProcessMessageDocument> response, DIS2018 report, bool isStandalone)
        {
            bool assertion;

            if (report.InvestorCollection != null)
            {
                foreach (DIS2018.Investor investor in report.InvestorCollection)
                {

                    assertion = (!string.IsNullOrWhiteSpace(investor.DIS7) && (!(IsMatch(investor.DIS7, @"^[a-zA-Z0-9!@\$%&\*\(\)\-=\[\];:'"",\.\?/ ]*$", RegexOptions.IgnoreCase)) || investor.DIS7.Contains(Environment.NewLine) || IsMatch(investor.DIS7, "\n")));
                    if (assertion)
                    {
                        var processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.DIS.500006",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Investor address - Street name and number - Address line 1 contains invalid text",
                            LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character.",
                            Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.DIS.500006" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        {
                            Name = "DIS7",
                            Value = investor.DIS7
                        });

                        response.Add(processMessage);
                    }
                }
            }
        }
        #endregion

        #region VR.ATO.DIS.500007

        /*  VR.ATO.DIS.500007
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF ([DIS49] <> NULLORBLANK) AND (ANY CHARACTER OF [DIS49] <> SET(a-z,A-Z,0-9,"!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "))
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^DIS49 <> BLANK) AND (NotCharacterInSet(^DIS49, '"a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))
    
            Data Elements:
    
            ^DIS49 = RP.AddressDetails.Line2.Text WHERE RP.[AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            */

        protected void VRATODIS500007(List<ProcessMessageDocument> response, DIS2018 report, bool isStandalone)
        {
            bool assertion;
            assertion = (string.IsNullOrWhiteSpace(report.DIS49) != true && (!(IsMatch(report.DIS49, @"^[a-zA-Z0-9!@\$%&\*\(\)\-=\[\];:'"",\.\?/ ]*$", RegexOptions.IgnoreCase)) || report.DIS49.Contains(Environment.NewLine) || IsMatch(report.DIS49, "\n")));
            if (assertion)
            {
                var processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.DIS.500007",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Business address of Company - Street name and number line 2 contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line2.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.DIS.500007" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                {
                    Name = "DIS49",
                    Value = report.DIS49
                });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.DIS.500008

        /*  VR.ATO.DIS.500008
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF ([DIS65] <> NULLORBLANK) AND (ANY CHARACTER OF [DIS65] <> SET(a-z,A-Z,0-9,"!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "))
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^DIS65 <> BLANK) AND (NotCharacterInSet(^DIS65, '"a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))
    
            Data Elements:
    
            ^DIS65 = RP.AddressDetails.Line2.Text WHERE RP.[AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */

        protected void VRATODIS500008(List<ProcessMessageDocument> response, DIS2018 report, bool isStandalone)
        {
            bool assertion;
            assertion = (string.IsNullOrWhiteSpace(report.DIS65) != true && (!(IsMatch(report.DIS65, @"^[a-zA-Z0-9!@\$%&\*\(\)\-=\[\];:'"",\.\?/ ]*$", RegexOptions.IgnoreCase)) || report.DIS65.Contains(Environment.NewLine) || IsMatch(report.DIS65, "\n")));
            if (assertion)
            {
                var processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.DIS.500008",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Postal address of Company - Street name and number line 2 contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line2.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.DIS.500008" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                {
                    Name = "DIS65",
                    Value = report.DIS65
                });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.DIS.500009

        /*  VR.ATO.DIS.500009
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF ([DIS8] <> NULLORBLANK) AND (ANY CHARACTER OF [DIS8] <> SET(a-z,A-Z,0-9,"!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "))
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^DIS8 <> BLANK) AND (NotCharacterInSet(^DIS8, '"a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))
    
            Data Elements:
    
            ^DIS8 = RP.TrueAndCorrect:PersonNameDetails.Position.Text WHERE RP.[AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */

        protected void VRATODIS500009(List<ProcessMessageDocument> response, DIS2018 report, bool isStandalone)
        {
            bool assertion;

            if (report.InvestorCollection != null)
            {
                foreach (DIS2018.Investor investor in report.InvestorCollection)
                {

                    assertion = (string.IsNullOrWhiteSpace(investor.DIS8) != true && (!(IsMatch(investor.DIS8, @"^[a-zA-Z0-9!@\$%&\*\(\)\-=\[\];:'"",\.\?/ ]*$", RegexOptions.IgnoreCase)) || investor.DIS8.Contains(Environment.NewLine) || IsMatch(investor.DIS8, "\n")));
                    if (assertion)
                    {
                        var processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.DIS.500009",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Investor address - Street name and number - Address line 2 contains invalid text",
                            LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character.",
                            Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line2.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.DIS.500009" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        {
                            Name = "DIS8",
                            Value = investor.DIS8
                        });

                        response.Add(processMessage);
                    }
                }
            }
        }
        #endregion

        #region VR.ATO.DIS.500010

        /*  VR.ATO.DIS.500010
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF ([DIS52] <> NULLORBLANK) AND (ANY CHARACTER OF [DIS52] <> SET(a-z,A-Z,0-9,"!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "))
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^DIS52 <> BLANK) AND (NotCharacterInSet(^DIS52, '"a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))
    
            Data Elements:
    
            ^DIS52 = RP.AddressDetails.LocalityName.Text WHERE RP.[AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            */

        protected void VRATODIS500010(List<ProcessMessageDocument> response, DIS2018 report, bool isStandalone)
        {
            bool assertion;
            assertion = (string.IsNullOrWhiteSpace(report.DIS52) != true && (!(IsMatch(report.DIS52, @"^[a-zA-Z0-9!@\$%&\*\(\)\-=\[\];:'"",\.\?/ ]*$", RegexOptions.IgnoreCase)) || report.DIS52.Contains(Environment.NewLine) || IsMatch(report.DIS52, "\n")));
            if (assertion)
            {
                var processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.DIS.500010",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Business address of Company - Suburb/town of Company contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.LocalityName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.DIS.500010" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                {
                    Name = "DIS52",
                    Value = report.DIS52
                });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.DIS.500011

        /*  VR.ATO.DIS.500011
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF ([DIS69] <> NULLORBLANK) AND (ANY CHARACTER OF [DIS69] <> SET(a-z,A-Z,0-9,"!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "))
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^DIS69 <> BLANK) AND (NotCharacterInSet(^DIS69, '"a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))
    
            Data Elements:
    
            ^DIS69 = RP.AddressDetails.LocalityName.Text WHERE RP.[AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */

        protected void VRATODIS500011(List<ProcessMessageDocument> response, DIS2018 report, bool isStandalone)
        {
            bool assertion;
            assertion = (string.IsNullOrWhiteSpace(report.DIS69) != true && (!(IsMatch(report.DIS69, @"^[a-zA-Z0-9!@\$%&\*\(\)\-=\[\];:'"",\.\?/ ]*$", RegexOptions.IgnoreCase)) || report.DIS69.Contains(Environment.NewLine) || IsMatch(report.DIS69, "\n")));
            if (assertion)
            {
                var processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.DIS.500011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Postal address of Company - Suburb/town of Company contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.LocalityName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.DIS.500011" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                {
                    Name = "DIS69",
                    Value = report.DIS69
                });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.DIS.500012

        /*  VR.ATO.DIS.500012
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF ([DIS9] <> NULLORBLANK) AND (ANY CHARACTER OF [DIS9] <> SET(a-z,A-Z,0-9,"!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "))
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^DIS9 <> BLANK) AND (NotCharacterInSet(^DIS9, '"a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))
    
            Data Elements:
    
            ^DIS9 = RP.AddressDetails.LocalityName.Text WHERE RP.[AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */

        protected void VRATODIS500012(List<ProcessMessageDocument> response, DIS2018 report, bool isStandalone)
        {
            bool assertion;

            if (report.InvestorCollection != null)
            {
                foreach (DIS2018.Investor investor in report.InvestorCollection)
                {

                    assertion = (string.IsNullOrWhiteSpace(investor.DIS9) != true && (!(IsMatch(investor.DIS9, @"^[a-zA-Z0-9!@\$%&\*\(\)\-=\[\];:'"",\.\?/ ]*$", RegexOptions.IgnoreCase)) || investor.DIS9.Contains(Environment.NewLine) || IsMatch(investor.DIS9, "\n")));
                    if (assertion)
                    {
                        var processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.DIS.500012",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Investor address - Suburb or town contains invalid text",
                            LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character.",
                            Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.LocalityName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.DIS.500012" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        {
                            Name = "DIS9",
                            Value = investor.DIS9
                        });

                        response.Add(processMessage);
                    }
                }
            }
        }
        #endregion

        #region VR.ATO.DIS.500013

        /*  VR.ATO.DIS.500013
        Business address of Company - Suburb/town of Company has a maximum length of 27

        Legacy Rule Format:
        IF LENGTH([DIS52]) > 27
            RETURN VALIDATION MESSAGE
        ENDIF

        Technical Business Rule Format:
        Length(^DIS52) > 27

        Data Elements:

            ^DIS52 = RP:AddressDetails:AddressDetails.LocalityName.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
        */
        protected void VRATODIS500013(List<ProcessMessageDocument> response, DIS2018 report, bool isStandalone)
        {
            bool assertion;
            assertion = (Length(report.DIS52) > 27);
            if (assertion)
            {
                var processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.DIS.500013",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Business address of Company - Suburb/town of Company must not exceed 27 characters",
                    LongDescription = @"For Business address of Company - Suburb/town of Company, the maximum allowable characters for this form is 27",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.LocalityName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.DIS.500013" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                {
                    Name = "DIS52",
                    Value = report.DIS52
                });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.DIS.500014

        /*  VR.ATO.DIS.500014
        Postal address of Company - Suburb/town of Company has a maximum length of 27

        Legacy Rule Format:
        IF LENGTH([DIS69]) > 27
            RETURN VALIDATION MESSAGE
        ENDIF

        Technical Business Rule Format:
        Length(^DIS69) > 27

        Data Elements:

            ^DIS52 = RP:AddressDetails:AddressDetails.LocalityName.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
        */
        protected void VRATODIS500014(List<ProcessMessageDocument> response, DIS2018 report, bool isStandalone)
        {
            bool assertion;
            assertion = (Length(report.DIS69) > 27);
            if (assertion)
            {
                var processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.DIS.500014",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Postal address of Company - Suburb/town of Company must not exceed 27 characters",
                    LongDescription = @"For Postal address of Company - Suburb/town of Company, the maximum allowable characters for this form is 27",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.LocalityName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.DIS.500014" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                {
                    Name = "DIS69",
                    Value = report.DIS69
                });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.DIS.500015

        /*  VR.ATO.DIS.500015
        Investor address - Suburb or town must not exceed 27 characters

        Legacy Rule Format:
        IF LENGTH([DIS9]) > 27
            RETURN VALIDATION MESSAGE
        ENDIF

        Technical Business Rule Format:
        Length(^DIS9) > 27

        Data Elements:

            ^DIS9 = RP:AddressDetails:AddressDetails.LocalityName.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
        */
        protected void VRATODIS500015(List<ProcessMessageDocument> response, DIS2018 report, bool isStandalone)
        {
            bool assertion;
            if (report.InvestorCollection != null)
            {
                foreach (DIS2018.Investor investor in report.InvestorCollection)
                {

                    assertion = (Length(investor.DIS9) > 27);
                    if (assertion)
                    {
                        var processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.DIS.500015",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Investor address - Suburb or town must not exceed 27 characters",
                            LongDescription = @"For Investor address - Suburb or town, the maximum allowable characters for this form is 27",
                            Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.LocalityName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.DIS.500015" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        {
                            Name = "DIS9",
                            Value = investor.DIS9
                        });

                        response.Add(processMessage);
                    }
                }
            }
        }
        #endregion

        #region VR.ATO.DIS.500016

        /*  VR.ATO.DIS.500016
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.

            Legacy Rule Format:
            IF ([DIS78] <> NULLORBLANK) AND (ANY CHARACTER OF [DIS78] <> SET(a-z,A-Z,0-9,"!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "))
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^DIS78 <> BLANK) AND (NotCharacterInSet(^DIS78, '"a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))

            Data Elements:

            ^DIS89 = RP.Declaration.SignatoryIdentifier.Text WHERE RP.[Declaration.StatementType.Code='TrueAndCorrect']
            */

        protected void VRATODIS500016(List<ProcessMessageDocument> response, DIS2018 report, bool isStandalone)
        {
            bool assertion;
            assertion = (string.IsNullOrWhiteSpace(report.DIS78) != true && (!(IsMatch(report.DIS78, @"^[a-zA-Z0-9!@\$%&\*\(\)\-=\[\];:'"",\.\?/ ]*$", RegexOptions.IgnoreCase)) || report.DIS78.Contains(Environment.NewLine) || IsMatch(report.DIS78, "\n")));
            if (assertion)
            {
                var processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.DIS.500016",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Reporting Party Person Declaration Signatory Identifier Text contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:Declaration" + OccurrenceIndex(report.DeclarationCollection_TrueAndCorrect_RPCount) + "/tns:Declaration.SignatoryIdentifier.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.DIS.500016" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                {
                    Name = "DIS78",
                    Value = report.DIS78
                });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.DIS.500017

        /*  VR.ATO.DIS.500017
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF ([DIS85] <> NULLORBLANK) AND (ANY CHARACTER OF [DIS85] <> SET(a-z,A-Z,0-9,"!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "))
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^DIS85 <> BLANK) AND (NotCharacterInSet(^DIS85, '"a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))
    
            Data Elements:
    
            ^DIS85 = INT.Declaration.SignatoryIdentifier.Text WHERE INT.[Declaration.StatementType.Code='TrueAndCorrect']
            */

        protected void VRATODIS500017(List<ProcessMessageDocument> response, DIS2018 report, bool isStandalone)
        {
            bool assertion;
            assertion = (string.IsNullOrWhiteSpace(report.DIS85) != true && (!(IsMatch(report.DIS85, @"^[a-zA-Z0-9!@\$%&\*\(\)\-=\[\];:'"",\.\?/ ]*$", RegexOptions.IgnoreCase)) || report.DIS85.Contains(Environment.NewLine) || IsMatch(report.DIS85, "\n")));
            if (assertion)
            {
                var processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.DIS.500017",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Intermediary Declaration Signatory Identifier Text contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:Declaration" + OccurrenceIndex(report.DeclarationCollection_TrueAndCorrect_INTCount) + "/tns:Declaration.SignatoryIdentifier.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.DIS.500017" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                {
                    Name = "DIS85",
                    Value = report.DIS85
                });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.DIS.500018

        /*  VR.ATO.DIS.500018
        Registered name of company must not exceed 161 characters

        Legacy Rule Format:
        IF LENGTH([DIS24]) > 161
            RETURN VALIDATION MESSAGE
        ENDIF

        Technical Business Rule Format:
        Length(^DIS24) > 161

        Data Elements:

            ^DIS24 = RP:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN']
        */
        protected void VRATODIS500018(List<ProcessMessageDocument> response, DIS2018 report, bool isStandalone)
        {
            bool assertion;
            assertion = (Length(report.DIS24) > 161);
            if (assertion)
            {
                var processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.DIS.500018",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Registered name of company must not exceed 161 characters",
                    LongDescription = @"For Registered name of company, the maximum allowable characters for this form is 161",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_MN_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.DIS.500018" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                {
                    Name = "DIS24",
                    Value = report.DIS24
                });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.DIS.500019

        /*  VR.ATO.DIS.500019
        Individual investor - Surname or Family name must not exceed 30 characters

        Legacy Rule Format:
        IF LENGTH([DIS4]) > 30
            RETURN VALIDATION MESSAGE
        ENDIF

        Technical Business Rule Format:
        Length(^DIS4) > 30

        Data Elements:

        ^DIS4 = RP:Investor:PersonNameDetails:PersonNameDetails.FamilyName.Text
        */
        protected void VRATODIS500019(List<ProcessMessageDocument> response, DIS2018 report, bool isStandalone)
        {
            bool assertion;
            if (report.InvestorCollection != null)
            {
                foreach (DIS2018.Investor investor in report.InvestorCollection)
                {

                    assertion = (Length(investor.DIS4) > 30);
                    if (assertion)
                    {
                        var processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.DIS.500019",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Individual investor - Surname or Family name must not exceed 30 characters",
                            LongDescription = @"For Individual investor - Surname or Family name, the maximum allowable characters for this form is 30",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:PersonNameDetails/tns:PersonNameDetails.FamilyName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.DIS.500019" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        {
                            Name = "DIS4",
                            Value = investor.DIS4
                        });

                        response.Add(processMessage);
                    }
                }
            }
        }
        #endregion

        #region VR.ATO.DIS.500020

        /*  VR.ATO.DIS.500020
        Individual investor -  First given name must not exceed 15 characters

        Legacy Rule Format:
        IF LENGTH([DIS5]) > 15
            RETURN VALIDATION MESSAGE
        ENDIF

        Technical Business Rule Format:
        Length(^DIS5) > 15

        Data Elements:

         ^DIS5 = RP:Investor:PersonNameDetails:PersonNameDetails.GivenName.Text
        */
        protected void VRATODIS500020(List<ProcessMessageDocument> response, DIS2018 report, bool isStandalone)
        {
            bool assertion;
            if (report.InvestorCollection != null)
            {
                foreach (DIS2018.Investor investor in report.InvestorCollection)
                {

                    assertion = (Length(investor.DIS5) > 15);
                    if (assertion)
                    {
                        var processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.DIS.500020",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Individual investor - First given name must not exceed 15 characters",
                            LongDescription = @"For Individual investor - First given name, the maximum allowable characters for this form is 15",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:PersonNameDetails/tns:PersonNameDetails.GivenName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.DIS.500020" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        {
                            Name = "DIS5",
                            Value = investor.DIS5
                        });

                        response.Add(processMessage);
                    }
                }
            }
        }
        #endregion

        #region VR.ATO.DIS.500021

        /*  VR.ATO.DIS.500021
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF ([DIS25] <> NULLORBLANK) AND ([DIS25] CONTAINS SET("{","}","|","_","\","#"))
                RETURN VALIDATION MESSAGE
            ENDIF

            [DIS25] = DIS:RP:pyde.02.00:PersonUnstructuredName.FullName.Text WHERE (pyde.02.37:PersonUnstructuredName.Usage.Code = "Contact") IN TUPLE(prsnunstrcnm1.02.01:PersonUnstructuredName)
            
            Technical Business Rule Format:
            (^DIS25 <> BLANK) AND (NotCharacterInSet(^DIS25, '"{","}","|","_","\","#"'))
    
            Data Elements:
    
            ^DIS25 = RP:PersonUnstructuredName:PersonUnstructuredName.FullName.Text WHERE [PersonUnstructuredName.Usage.Code='Contact']
            */

        protected void VRATODIS500021(List<ProcessMessageDocument> response, DIS2018 report, bool isStandalone)
        {
            bool assertion;
            assertion = IsMatch(report.DIS25, @"\{|\}|\||_|\\|#|\n", RegexOptions.IgnoreCase);
            if (assertion)
            {
                var processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.DIS.500021",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Reporting Party - Contact name contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:PersonUnstructuredName" + OccurrenceIndex(report.PersonUnstructuredNameCollection_Contact_RPOccurrenceIndex) + "/tns:PersonUnstructuredName.FullName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.DIS.500021" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                {
                    Name = "DIS25",
                    Value = report.DIS25
                });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.DIS.500022

        /*  VR.ATO.DIS.500022
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF ([DIS80] <> NULLORBLANK) AND ([DIS80] CONTAINS SET("{","}","|","_","\","#"))
                RETURN VALIDATION MESSAGE
            ENDIF

            [DIS80] = DIS:RP:pyde.02.00:PersonUnstructuredName.FullName.Text WHERE (pyde.02.37:PersonUnstructuredName.Usage.Code = "Contact") IN TUPLE(prsnunstrcnm1.02.01:PersonUnstructuredName)
            
            Technical Business Rule Format:
            (^DIS80 <> BLANK) AND (NotCharacterInSet(^DIS80, '"{","}","|","_","\","#"'))
    
            Data Elements:
    
            ^DIS80 = RP:Declaration:PersonUnstructuredName:PersonUnstructuredName.FullName.Text WHERE [PersonUnstructuredName.Usage.Code='DeclarationSignatory']
            */

        protected void VRATODIS500022(List<ProcessMessageDocument> response, DIS2018 report, bool isStandalone)
        {
            bool assertion;
            assertion = IsMatch(report.DIS80, @"\{|\}|\||_|\\|#|\n", RegexOptions.IgnoreCase);
            if (assertion)
            {
                var processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.DIS.500022",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Reporting Party Declaration Person Name contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:Declaration" + OccurrenceIndex(report.DeclarationCollection_TrueAndCorrect_RPOccurrenceIndex) + "/tns:PersonUnstructuredName" + OccurrenceIndex(report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPOccurrenceIndex) + "/tns:PersonUnstructuredName.FullName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.DIS.500022" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                {
                    Name = "DIS80",
                    Value = report.DIS80
                });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.DIS.500023

        /*  VR.ATO.DIS.500023
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF ([DIS87] <> NULLORBLANK) AND ([DIS87] CONTAINS SET("{","}","|","_","\","#"))
                RETURN VALIDATION MESSAGE
            ENDIF

            [DIS87] = INT:Declaration:PersonUnstructuredName:PersonUnstructuredName.FullName.Text WHERE [PersonUnstructuredName.Usage.Code='DeclarationSignatory']
            
            Technical Business Rule Format:
            (^DIS87 <> BLANK) AND (NotCharacterInSet(^DIS80, '"{","}","|","_","\","#"'))
    
            Data Elements:
    
            ^DIS87 = INT:Declaration:PersonUnstructuredName:PersonUnstructuredName.FullName.Text WHERE [PersonUnstructuredName.Usage.Code='DeclarationSignatory']
            */

        protected void VRATODIS500023(List<ProcessMessageDocument> response, DIS2018 report, bool isStandalone)
        {
            bool assertion;
            assertion = IsMatch(report.DIS87, @"\{|\}|\||_|\\|#|\n", RegexOptions.IgnoreCase);
            if (assertion)
            {
                var processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.DIS.500023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Intermediary Declaration Person Name contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:Declaration" + OccurrenceIndex(report.DeclarationCollection_TrueAndCorrect_INTOccurrenceIndex) + "/tns:PersonUnstructuredName" + OccurrenceIndex(report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTOccurrenceIndex) + "/tns:PersonUnstructuredName.FullName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.DIS.500023" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                {
                    Name = "DIS87",
                    Value = report.DIS87
                });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.DIS.500024

        /*  VR.ATO.DIS.500024
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF ([DIS89] <> NULLORBLANK) AND ([DIS89] CONTAINS SET("{","}","|","_","\","#"))
                RETURN VALIDATION MESSAGE
            ENDIF

            [DIS89] = INT:Declaration:PersonUnstructuredName:PersonUnstructuredName.FullName.Text WHERE [PersonUnstructuredName.Usage.Code='DeclarationSignatory']
            
            Technical Business Rule Format:
            (^DIS89 <> BLANK) AND (NotCharacterInSet(^DIS80, '"{","}","|","_","\","#"'))
    
            Data Elements:
    
            ^DIS89 = INT:Declaration:PersonUnstructuredName:PersonUnstructuredName.FullName.Text WHERE [PersonUnstructuredName.Usage.Code='DeclarationSignatory']
            */

        protected void VRATODIS500024(List<ProcessMessageDocument> response, DIS2018 report, bool isStandalone)
        {
            bool assertion;
            assertion = IsMatch(report.DIS89, @"\{|\}|\||_|\\|#|\n", RegexOptions.IgnoreCase);
            if (assertion)
            {
                var processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.DIS.500024",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Intermediary contact name contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:PersonUnstructuredName" + OccurrenceIndex(report.PersonUnstructuredNameCollection_Contact_INTOccurrenceIndex) + "/tns:PersonUnstructuredName.FullName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.DIS.500024" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                {
                    Name = "DIS89",
                    Value = report.DIS89
                });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.DIS.500025

        /*  VR.ATO.DIS.500025
        Individual investor- Other given names must not exceed 15 characters

        Legacy Rule Format:
        IF LENGTH([DIS6]) > 15
            RETURN VALIDATION MESSAGE
        ENDIF

        Technical Business Rule Format:
        Length(^DIS6) > 15

        Data Elements:

        ^DIS6 = RP:Investor:PersonNameDetails:PersonNameDetails.OtherGivenName.Text
        */
        protected void VRATODIS500025(List<ProcessMessageDocument> response, DIS2018 report, bool isStandalone)
        {
            bool assertion;
            if (report.InvestorCollection != null)
            {
                foreach (DIS2018.Investor investor in report.InvestorCollection)
                {

                    assertion = (Length(investor.DIS6) > 15);
                    if (assertion)
                    {
                        var processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.DIS.500025",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Individual investor- Other given names must not exceed 15 characters",
                            LongDescription = @"For Individual investor- Other given names, the maximum allowable characters for this form is 15",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:PersonNameDetails/tns:PersonNameDetails.OtherGivenName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.DIS.500025" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        {
                            Name = "DIS6",
                            Value = investor.DIS6
                        });

                        response.Add(processMessage);
                    }
                }
            }
        }
        #endregion

        #region VR.ATO.DIS.500026

        /*  VR.ATO.DIS.500026
        Individual investor -  Position must not exceed 50 characters

        Legacy Rule Format:
        IF LENGTH([DIS55]) > 50
            RETURN VALIDATION MESSAGE
        ENDIF

        Technical Business Rule Format:
        Length(^DIS55) > 50

        Data Elements:

        ^DIS55 = RP:Investor:PersonNameDetails:Position.Text
        */
        protected void VRATODIS500026(List<ProcessMessageDocument> response, DIS2018 report, bool isStandalone)
        {
            bool assertion;
            if (report.InvestorCollection != null)
            {
                foreach (DIS2018.Investor investor in report.InvestorCollection)
                {

                    assertion = (Length(investor.DIS55) > 50);
                    if (assertion)
                    {
                        var processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.DIS.500026",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Individual investor -  Position must not exceed 50 characters",
                            LongDescription = @"For Individual investor -  Position, the maximum allowable characters for this form is 50",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:PersonNameDetails/tns:PersonNameDetails.Position.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.DIS.500026" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        {
                            Name = "DIS55",
                            Value = investor.DIS55
                        });

                        response.Add(processMessage);
                    }
                }
            }
        }
        #endregion

        #region VR.ATO.GEN.428041 (DIS56)

        /*  VR.ATO.GEN.428041
            Postcode must be in the range 0200 - 9999
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Postcode.Text <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.Postcode.Text  <> SET(0200-9999))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^DIS56 <> BLANK) AND (IsNumeric(^DIS56) = 'FALSE' OR AsNumeric(^DIS56) < 200 OR AsNumeric(^DIS56) > 9999)
    
            Data Elements:
    
            ^DIS56 = RP:AddressDetails:AddressDetails.Postcode.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            */

        protected void VRATOGEN428041_01(List<ProcessMessageDocument> response, DIS2018 report, bool isStandalone)
        {
            bool assertion;
            assertion = (string.IsNullOrWhiteSpace(report.DIS56) != true && (IsNumeric(report.DIS56) == false || AsNumeric(report.DIS56) < 200 || AsNumeric(report.DIS56) > 9999));
            if (assertion)
            {
                var processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.428041",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"For an Australian address, postcode is invalid. Valid range is 0200 - 9999",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Postcode.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428041" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "DIS56", Value = report.DIS56 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.428041 (DIS70)

        /*  VR.ATO.GEN.428041
            Postcode must be in the range 0200 - 9999
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Postcode.Text <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.Postcode.Text  <> SET(0200-9999))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^DIS70 <> BLANK) AND (IsNumeric(^DIS70) = 'FALSE' OR AsNumeric(^DIS70) < 200 OR AsNumeric(^DIS70) > 9999)
    
            Data Elements:
    
            ^DIS70 = RP:AddressDetails:AddressDetails.Postcode.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */

        protected void VRATOGEN428041_02(List<ProcessMessageDocument> response, DIS2018 report, bool isStandalone)
        {
            bool assertion;
            assertion = (string.IsNullOrWhiteSpace(report.DIS70) != true && (IsNumeric(report.DIS70) == false || AsNumeric(report.DIS70) < 200 || AsNumeric(report.DIS70) > 9999));
            if (assertion)
            {
                var processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.428041",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"For an Australian address, postcode is invalid. Valid range is 0200 - 9999",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Postcode.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428041" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "DIS70", Value = report.DIS70 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.428041 (DIS11)

        /*  VR.ATO.GEN.428041
            Postcode must be in the range 0200 - 9999
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Postcode.Text <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.Postcode.Text  <> SET(0200-9999))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^DIS11 <> BLANK) AND (IsNumeric(^DIS11) = 'FALSE' OR AsNumeric(^DIS11) < 200 OR AsNumeric(^DIS11) > 9999)
    
            Data Elements:
    
            ^DIS11 = RP:AddressDetails:AddressDetails.Postcode.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */

        protected void VRATOGEN428041_03(List<ProcessMessageDocument> response, DIS2018 report, bool isStandalone)
        {
            bool assertion;

            if (report.InvestorCollection != null)
            {
                foreach (DIS2018.Investor investor in report.InvestorCollection)
                {

                    assertion = (string.IsNullOrWhiteSpace(investor.DIS11) != true && (IsNumeric(investor.DIS11) == false || AsNumeric(investor.DIS11) < 200 || AsNumeric(investor.DIS11) > 9999));
                    if (assertion)
                    {
                        var processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.428041",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"For an Australian address, postcode is invalid. Valid range is 0200 - 9999",
                            Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Postcode.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428041" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "DIS11", Value = investor.DIS11 });

                        response.Add(processMessage);
                    }
                }
            }
        }
        #endregion

        #region VR.ATO.GEN.500023 (DIS24)

        /*  VR.ATO.GEN.500023
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text <> NULLORBLANK) AND (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text CONTAINS SET("{","}","|","_","\","#"))
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ContainsSet(^DIS24, '"{","}","|","_","\","#"')
    
            Data Elements:
    
            RP.^DIS24 = tns:OrganisationNameDetails.OrganisationalName.Text WHERE RP.[OrganisationNameDetails.OrganisationalNameType.Code='MN']
            */

        protected void VRATOGEN500023_DIS24(List<ProcessMessageDocument> response, DIS2018 report, bool isStandalone)
        {
            bool assertion;
            assertion = (string.IsNullOrWhiteSpace(report.DIS24) != true && (!(IsMatch(report.DIS24, @"^[a-zA-Z0-9!@\$%&\*\(\)\-=\[\];:'"",\.\?/ ]*$", RegexOptions.IgnoreCase)) || report.DIS24.Contains(Environment.NewLine) || IsMatch(report.DIS24, "\n")));
            if (assertion)
            {
                var processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-Individual Name contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = ; : ' "", . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_MN_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500023" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "DIS24", Value = report.DIS24 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500023 (DIS3)

        /*  VR.ATO.GEN.500023
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text <> NULLORBLANK) AND (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text CONTAINS SET("{","}","|","_","\","#"))
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ContainsSet(^DIS3, '"{","}","|","_","\","#"')
    
            Data Elements:
    
            RP.^DIS3 = tns:OrganisationNameDetails.OrganisationalName.Text WHERE RP.[OrganisationNameDetails.OrganisationalNameType.Code='MN']
            */

        protected void VRATOGEN500023_DIS3(List<ProcessMessageDocument> response, DIS2018 report, bool isStandalone)
        {
            bool assertion;
            if (report.InvestorCollection != null)
            {
                foreach (DIS2018.Investor investor in report.InvestorCollection)
                {

                    assertion = (string.IsNullOrWhiteSpace(investor.DIS3) != true && (!(IsMatch(investor.DIS3, @"^[a-zA-Z0-9!@\$%&\*\(\)\-=\[\];:'"",\.\?/ ]*$", RegexOptions.IgnoreCase)) || investor.DIS3.Contains(Environment.NewLine) || IsMatch(investor.DIS3, "\n")));
                    if (assertion)
                    {
                        var processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.500023",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Non-Individual Name contains invalid text",
                            LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = ; : ' "", . ? / or a space character.",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:OrganisationNameDetails" + OccurrenceIndex(investor.Investor_OrganisationNameDetailsCollection_MN_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500023" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        {
                            Name = "DIS3",
                            Value = investor.DIS3
                        });

                        response.Add(processMessage);
                    }
                }
            }
        }
        #endregion

        #region VR.ATO.GEN.500023 (DIS91)

        /*  VR.ATO.GEN.500023
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text <> NULLORBLANK) AND (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text CONTAINS SET("{","}","|","_","\","#"))
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ContainsSet(^DIS91, '"{","}","|","_","\","#"')
    
            Data Elements:
    
            ^DIS91 = INT:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN']
            */

        protected void VRATOGEN500023_DIS91(List<ProcessMessageDocument> response, DIS2018 report, bool isStandalone)
        {
            bool assertion;
            assertion = (string.IsNullOrWhiteSpace(report.DIS91) != true && (!(IsMatch(report.DIS91, @"^[a-zA-Z0-9!@\$%&\*\(\)\-=\[\];:'"",\.\?/ ]*$", RegexOptions.IgnoreCase)) || report.DIS91.Contains(Environment.NewLine) || IsMatch(report.DIS91, "\n")));
            if (assertion)
            {
                var processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-Individual Name contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = ; : ' "", . ? / or a space character.",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_MN_INTOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500023" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "DIS91", Value = report.DIS91 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500024

        /*  VR.ATO.GEN.500024
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.FamilyName.Text CONTAINS SET("{","}","|","_","\","#"))
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^DIS4 <> BLANK) AND (NotCharacterInSet(^DIS4, '"{","}","|","_","\","#"'))
    
            Data Elements:
    
            ^DIS4 = RP:Investor:PersonNameDetails:PersonNameDetails.FamilyName.Text
            */

        protected void VRATOGEN500024(List<ProcessMessageDocument> response, DIS2018 report, bool isStandalone)
        {
            bool assertion;

            if (report.InvestorCollection != null)
            {
                foreach (DIS2018.Investor investor in report.InvestorCollection)
                {

                    assertion = IsMatch(investor.DIS4, @"\{|\}|\||_|\\|#|\n", RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        var processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.500024",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Individual Family name contains invalid text",
                            LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character.",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:PersonNameDetails/tns:PersonNameDetails.FamilyName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500024" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        {
                            Name = "DIS4",
                            Value = investor.DIS4
                        });

                        response.Add(processMessage);
                    }
                }
            }
        }
        #endregion

        #region VR.ATO.GEN.500025

        /*  VR.ATO.GEN.500025
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.GivenName.Text CONTAINS SET("{","}","|","_","\","#"))
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^DIS5 <> BLANK) AND (NotCharacterInSet(^DIS5, '"{","}","|","_","\","#"'))
    
            Data Elements:
    
            ^DIS5 = RP:Investor:PersonNameDetails:PersonNameDetails.GivenName.Text
            */

        protected void VRATOGEN500025(List<ProcessMessageDocument> response, DIS2018 report, bool isStandalone)
        {
            bool assertion;

            if (report.InvestorCollection != null)
            {
                foreach (DIS2018.Investor investor in report.InvestorCollection)
                {

                    assertion = IsMatch(investor.DIS5, @"\{|\}|\||_|\\|#|\n", RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        var processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.500025",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Individual First given name contains invalid text",
                            LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character.",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:PersonNameDetails/tns:PersonNameDetails.GivenName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500025" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        {
                            Name = "DIS5",
                            Value = investor.DIS5
                        });

                        response.Add(processMessage);
                    }
                }
            }
        }
        #endregion

        #region VR.ATO.GEN.500042

        /*  VR.ATO.GEN.500042
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text CONTAINS SET("{","}","|","_","\","#"))
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^DIS6 <> BLANK) AND (NotCharacterInSet(^DIS6, '"{","}","|","_","\","#"'))
    
            Data Elements:
    
            ^DIS6 = RP:Investor:PersonNameDetails:PersonNameDetails.OtherGivenName.Text
            */

        protected void VRATOGEN500042(List<ProcessMessageDocument> response, DIS2018 report, bool isStandalone)
        {
            bool assertion;

            if (report.InvestorCollection != null)
            {
                foreach (DIS2018.Investor investor in report.InvestorCollection)
                {

                    assertion = IsMatch(investor.DIS6, @"\{|\}|\||_|\\|#|\n", RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        var processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.500042",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Individual other given names contains invalid text",
                            LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character.",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:PersonNameDetails/tns:PersonNameDetails.OtherGivenName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500042" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        {
                            Name = "DIS6",
                            Value = investor.DIS6
                        });

                        response.Add(processMessage);
                    }
                }
            }
        }
        #endregion

        #region VR.ATO.GEN.500059

        /*  VR.ATO.GEN.500059
            A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' " , . ? / or a space character.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.Position.Text <> NULLORBLANK) AND (ANY CHARACTER OF pyde.xx.xx:PersonNameDetails.Position.Text <> SET("a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "))
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^DIS55 <> BLANK) AND (NotCharacterInSet(^DIS55, '"a-z","A-Z","0-9","!","@","$","%","&","*","(",")","-","=","[","]",";",":","'",""",",",".","?","/"," "'))
    
            Data Elements:
    
            ^DIS55 = RP:Investor:PersonNameDetails:Position.Text
            */

        protected void VRATOGEN500059(List<ProcessMessageDocument> response, DIS2018 report, bool isStandalone)
        {
            bool assertion;

            if (report.InvestorCollection != null)
            {
                foreach (DIS2018.Investor investor in report.InvestorCollection)
                {

                    assertion = (string.IsNullOrWhiteSpace(investor.DIS55) != true && (!(IsMatch(investor.DIS55, @"^[a-zA-Z0-9!@\$%&\*\(\)\-=\[\];:'"",\.\?/ ]*$", RegexOptions.IgnoreCase)) || investor.DIS55.Contains(Environment.NewLine) || IsMatch(investor.DIS55, "\n")));
                    if (assertion)
                    {
                        var processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.500059",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Declarer position held of person making the declaration contains invalid text",
                            LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - = [ ] ; : ' "" , . ? / or a space character.",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:PersonNameDetails/tns:PersonNameDetails.OtherGivenName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500059" } },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                        {
                            Name = "DIS55",
                            Value = investor.DIS55
                        });

                        response.Add(processMessage);
                    }
                }
            }
        }
        #endregion

        #region VR.ATO.DIS.406041

        /*  VR.ATO.DIS.406041
                    Reporting party declaration must not be supplied when submitted with a parent return

                Legacy Rule Format:
                    WHERE PARENT RETURN EXISTS
                    IF [DIS74] <> NULLORBLANK
                        RETURN VALIDATION MESSAGE
                    ENDIF

                    [DIS74] = DIS:RP:pyin.02.03:Declaration.StatementType.Code WHERE (pyin.02.03:Declaration.StatementType.Code = "TrueAndCorrect") IN TUPLE(declaration2.02.01:Declaration)

            Technical Business Rule Format:
                    ^DIS74 <> BLANK

        Data Elements:

                    RP:^DIS74 = tns:Declaration.StatementType.Code WHERE [Declaration.StatementType.Code='TrueAndCorrect']
                    */
        protected void VRATODIS406041(List<ProcessMessageDocument> response, DIS2018 report, bool isStandalone)
        {
            bool assertion = !isStandalone && !string.IsNullOrWhiteSpace(report.DIS74);

            if (assertion)
            {
                var processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.DIS.406041",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Reporting party declaration must not be supplied when submitted with a parent return.",
                    Location = "/xbrli:xbrl/tns:Declaration" + OccurrenceIndex(report.DeclarationCollection_TrueAndCorrect_RPOccurrenceIndex) + "/tns:Declaration.StatementType.Code",
                    Parameters = new ProcessMessageParameters { new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.DIS.406041" }, new ProcessMessageParameter { Name = "DIS74", Value = report.DIS74 } }
                };

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.DIS.406041

        #region VR.ATO.DIS.406047

        /*  VR.ATO.DIS.406047
                    Intermediary declaration must not be supplied when submitted with a parent return

                Legacy Rule Format:
                    WHERE PARENT RETURN EXISTS
                    IF [DIS81] <> NULLORBLANK
                        RETURN VALIDATION MESSAGE
                    ENDIF
                    
                    [DIS81] = DIS:INT:pyin.02.03:Declaration.StatementType.Code WHERE (pyin.02.03:Declaration.StatementType.Code = "TrueAndCorrect") IN TUPLE(declaration2.02.01:Declaration)

            Technical Business Rule Format:
                    ^DIS81 <> BLANK

        Data Elements:
            
                    INT:^DIS81 = tns:Declaration.StatementType.Code WHERE [Declaration.StatementType.Code='TrueAndCorrect']
                    */
        protected void VRATODIS406047(List<ProcessMessageDocument> response, DIS2018 report, bool isStandalone)
        {
            bool assertion = !isStandalone && !string.IsNullOrWhiteSpace(report.DIS81);

            if (assertion)
            {
                var processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.DIS.406047",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Intermediary declaration must not be supplied when submitted with a parent return",
                    Location = "/xbrli:xbrl/tns:Declaration" + OccurrenceIndex(report.DeclarationCollection_TrueAndCorrect_INTOccurrenceIndex) + "/tns:Declaration.StatementType.Code",
                    Parameters = new ProcessMessageParameters { new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.DIS.406047" }, new ProcessMessageParameter { Name = "DIS81", Value = report.DIS81 } }
                };

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.DIS.406047

        #endregion Manual Rules - Rules marked for manual coding
    }
}