
using System;
using System.IO;
using System.Runtime.CompilerServices;
using System.Text;
using System.Xml;
using System.Linq;
using System.Collections.Generic;
using DataContracts;
using System.Text.RegularExpressions;

namespace Ato.EN.IntegrationServices.CodeGenerationDIS
{

    public partial class DIS2018Validator
    {
        /// <summary>
        /// The max parameter name length is restricted by the SBR1 and SBR2 schemas - the lowest common denominator of SBR 1 is used as the default
        /// </summary>
        private int _maxParameterNameLength;

        /// <summary>
        /// The max parameter value length is restricted by the SBR1 and SBR2 schemas
        /// </summary>
        private int _maxParameterValueLength;

        /// <summary>
        /// The SBR1 and ebms3 schemas do not allow parameter names or values to be the empty string
        /// </summary>
        private string _emptyParameterValue;

        /// <summary>
        /// Initializes a new instance of the <see cref="DIS2018Validator" /> class.
        /// </summary>
        /// <param name="maxParameterNameLength">Maximum length of the parameter name.</param>
        /// <param name="maxParameterValueLength">Maximum length of the parameter value.</param>
        /// <param name="emptyParameterValue">This value will be used in place of any parameter values that result in a null or empty value.</param>
        public DIS2018Validator(int maxParameterNameLength = 20, int maxParameterValueLength = 4096, string emptyParameterValue = "EMPTY")
        {
            _maxParameterNameLength = maxParameterNameLength;
            _maxParameterValueLength = maxParameterValueLength;
            _emptyParameterValue = emptyParameterValue;
        }

        #region Functions
        private static IEnumerable<string> Union(IEnumerable<string> list1, IEnumerable<string> list2)
        {
            IEnumerable<string> response;

            if (list1 == null && list2 == null)
            {
                response = null;
            }
            else if (list1 == null)
            {
                response = list2.Distinct();
            }
            else if (list2 == null)
            {
                response = list1.Distinct();
            }
            else
            {
                response = list1.Union(list2);
            }

            return response;
        }

        private static bool IsMatch(int? field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(Convert.ToString(field.Value), expression, options);
        }

        private static bool IsMatch(string field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(field, expression, options);
        }

        // This is just context and tuple counts where they are integer values - easier that changing the parsing logic to just return the value
        private static int Count(int count)
        {
            return count;
        }


        private static int Count<T>(IEnumerable<T> values)
        {
            return values == null ? 0 : values.Where(f => f != null).Count();
        }


        private static int Count<T>(ICollection<T> values)
        {
            return values == null ? 0 : values.Where(f => f != null).Count();
        }


        private static bool exists(bool value)
        {
            return value;
        }

        private static string GetValueOrEmpty(bool? val)
        {
            return (val.HasValue) ? val.ToString().ToLower() : string.Empty;
        }

        private static string GetValueOrEmpty(DateTime? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        private static string GetValueOrEmpty(string val)
        {
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmpty(decimal? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        private static string GetValueOrEmpty(int? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

         /// <summary>
         /// Get string value between [first] a and [last] b.
         /// </summary>
        public static string Between(string value, string a, string b)
        {
            int posA = value.IndexOf(a);
            int posB = value.LastIndexOf(b);
            if (posA == -1)
            {
                return "";
            }
            if (posB == -1)
            {
                return "";
            }
            int adjustedPosA = posA + a.Length;
            if (adjustedPosA >= posB)
            {
                return "";
            }
            return value.Substring(adjustedPosA, posB - adjustedPosA);
        }

         /// <summary>
         /// Get string value after [first] a.
         /// </summary>
        public static string Before(string value, string a)
        {
            int posA = value.IndexOf(a);
            if (posA == -1)
            {
                return "";
            }
            return value.Substring(0, posA);
        }

         /// <summary>
         /// Get string value after [last] a.
         /// </summary>
        public static string After(string value, string a)
        {
            int posA = value.LastIndexOf(a);
            if (posA == -1)
            {
                return "";
            }
            int adjustedPosA = posA + a.Length;
            if (adjustedPosA >= value.Length)
            {
                return "";
            }
            return value.Substring(adjustedPosA);
        }

        private static int Length(object field)
        {
            if (field == null)
            return 0;
            else
            return field.ToString().Trim().Length;
        }

        private static bool NotSameValues(IEnumerable<object> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool NotSameValues(params object[] nodes)
        {
            if (nodes == null)
                return false;

            return ((from x in nodes select x).Distinct().Count() == nodes.Count());
        }

        private static bool HasDuplicateValues(IEnumerable<object> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool HasDuplicateValues(params object[] nodes)
        {
           if (nodes == null)
                return false;

            nodes = nodes.Where(x => x != null).ToArray();
            return !((from x in nodes select x).Distinct().Count() == nodes.Count());
        
        }

        private int DuplicateValueIndex(IEnumerable<object> values)
        {
            int response = 0;
            var hashset = new HashSet<object>();
            foreach (var value in values)
            {
                if (!hashset.Add(value))
                {
                    return response;
                }
                response++;
            }
            return response;
        }

        private int DuplicateValueIndex<T>(IEnumerable<T?> values) where T : struct
        {
            int response = 0;
            var hashset = new HashSet<T?>();
            foreach (var value in values)
            {
                if (!hashset.Add(value))
                {
                    return response;
                }
                response++;
            }
            return response;
        }

        private static bool IsDate(object value)
        {
            DateTime dateValue;
            return (value != null && DateTime.TryParse(value.ToString(), out dateValue));
        }

        public DateTime AsDate(string dateAsString)
        {
            DateTime response = DateTime.MinValue;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date;
            }

            return response;
        }

        public DateTime Date(DateTime? datetime)
        {                   
            return datetime.GetValueOrDefault().Date;
        }

				//The logic in After function expects "---" for day and "--" for month. 
				//Since hyphen was missing the date always returned null
        public DateTime? ConvertToDate(int day, int month, int year)
        {
					return ConvertToDate(day == 0 ? null : "---" + day.ToString(), month == 0 ? null : "--" + month.ToString(), year == 0 ? null : year.ToString());
        }

        public DateTime? ConvertToDate(string day, string month, string year)
        {
            DateTime? response;
            DateTime result;

            if (year == null || month == null || day == null)
            {
                return null;
            }
            string dateAsString = year + "-" + After(month, "--") + "-" + After(day, "---");
            if (DateTime.TryParse(dateAsString, out result))
            {
                response = result;
            }
            else
            {
                response = null;
            }
            return response;
        }


        public DateTime? ConvertToDate(string day, string month, int year)
        {
            DateTime? response;
            DateTime result;

            if (year == 0 || month == null || day == null)
            {
                return null;
            }
            string dateAsString = year.ToString() + "-" + After(month, "--") + "-" + After(day, "---");
            if (DateTime.TryParse(dateAsString, out result))
            {
                response = result;
            }
            else
            {
                response = null;
            }
            return response;
        }


        private static int Day(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Day;
            }

            return response;
        }


        private static int Day(DateTime? date)
        {
            if (date == null)
                return 0;
            else
                return date.Value.Day;

        }


        private static string Month(string dateAsString)
        {
            string response = null;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.ToString("MMMM");
            }
            else
            {
                return "NotAMonth";
            }

            return response;
        }

        private static string Month(DateTime? date)
        {
            if (date == null)
                return "NotAMonth";
            else
                return date.Value.ToString("MMMM");
        }


        private static int MonthAsInt(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Month;
            }

            return response;
        }

        private static int MonthAsInt(DateTime? date)
        {
            if (date == null)
            {
                return 0;
            }
            return date.Value.Month;
        }


        private static int Year(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Year;
            }

            return response;
        }

        private static int Year(DateTime? date)
        {
            if (date == null)
                return 0;
            else
                return date.Value.Year;
        }


        private static int CurrentFinancialYear()
        {
            return DateToFinancialYear(DateTime.Now, 7);
        }

        private static int FinancialYear(string dateAsString)
        {
            return DateToFinancialYear(dateAsString, 7);
        }

        private static int FinancialYear(DateTime? date)
        {
            return DateToFinancialYear(date, 7);
        }

        private static int DateToFinancialYear(string dateAsString, int startingMonth)
        {
            int response = 0;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
            {
                response = DateToFinancialYear(date, startingMonth);
            }

            return response;
        }
        private static int DateToFinancialYear(DateTime? date, int startingMonth)
        {
            int response;
            if (date == null)
            {
                response = 0;
            }
            else
            {
                int year = date.Value.Year;
                int month = date.Value.Month;

                if (startingMonth > month)
                    response = year;
                else
                    response = year + 1;
            }
            return response;
        }


        private static int FBTYear(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
                response = FBTYear(date);
            return response;
        }

        private static int FBTYear(DateTime? date)
        {
            if (date == null)
            {
                return 0;
            }
            else
            {
                if (date.Value.Month > 3)
                    return date.Value.Year + 1;
                else
                    return date.Value.Year;
            }
        }

        private static DateTime? AddMonthsToDate(DateTime? dateTime, int months)
        {
            return dateTime == null ? null : (DateTime?)dateTime.Value.AddMonths(months);
        }

        private static bool IsNumeric(object value)
        {
            decimal numbervalue;
            return (value != null && decimal.TryParse(value.ToString(), out numbervalue));
        }

        private static bool NotMonetary(decimal? field, string sign, int digits, int decimals)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                string decimalExpression;
                int digitsToUse = digits - decimals;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                if (decimals > 0)
                    decimalExpression = @"(\.\d{1," + decimals + "})?$";
                else
                    decimalExpression = @"$";

                return !(Regex.IsMatch(field.Value.ToString("0.#########################"), signExpression + @"\d{1," + digitsToUse + "}" + decimalExpression));
            }
        }

        private static bool NotNumeric(decimal? field, string sign, int digits, int decimals)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                string decimalExpression;
                int digitsToUse = digits - decimals;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                if (decimals > 0)
                    decimalExpression = @"(\.\d{1," + decimals + "})?$";
                else
                    decimalExpression = @"$";

                return !(Regex.IsMatch(field.Value.ToString("0.#########################"), signExpression + @"\d{1," + digitsToUse + "}" + decimalExpression));
            }
        }

        private static bool NotNumeric(int? field, string sign, int digits, int decimals = 0)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                return !(Regex.IsMatch(field.Value.ToString(), signExpression + @"\d{1," + digits + "}$"));
            }
        }

        private static bool NotNumeric(long? field, string sign, int digits, int decimals = 0)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                return !(Regex.IsMatch(field.Value.ToString(), signExpression + @"\d{1," + digits + "}$"));
            }
        }


        private static bool OutsideRange(decimal field, decimal expression, int range)
        {
            bool response;

            response = (field < (expression - range)) || (field > (expression + range));

            return response;
        }


        private static bool FailsUSIAlgorithm(string usi, string abn)
        {
            bool response = true;
            if (usi == null || abn == null)
            {
                response = false;
            }
            else
            {
                usi = usi.Trim();
                abn = abn.Trim();
                if (usi.Length == 14)
                {
                    int numeric;
                    if (usi.Substring(0, 11) == abn && int.TryParse(usi.Substring(11, 3), out numeric))
                        response = false;

                }
                else if (usi.Length == 9)
                {

                    if (Regex.IsMatch(usi, @"^([a-zA-Z]{3}\d{4}[a-zA-Z]{2})$"))
                        response = false;
                }
            }
            return response;
        }

        private static bool FailsTANAlgorithm(string tan)
        {
            bool response;
            decimal decimalTan;

            if (tan == null)
                return false;

            tan = tan.Trim();

            if (!decimal.TryParse(tan, out decimalTan))
                return true;

            if (tan.Length != 8)
                return true;

            decimal tanSum =
                7 * int.Parse(tan.Substring(0, 1)) +
                9 * int.Parse(tan.Substring(1, 1)) +
                8 * int.Parse(tan.Substring(2, 1)) +
                4 * int.Parse(tan.Substring(3, 1)) +
                6 * int.Parse(tan.Substring(4, 1)) +
                3 * int.Parse(tan.Substring(5, 1)) +
                5 * int.Parse(tan.Substring(6, 1)) +
                1 * int.Parse(tan.Substring(7, 1));

            if ((tanSum % 11) == 0)
                response = false;
            else
                response = true;

            return response;
        }


        private static bool FailsABNAlgorithm(string abn)
        {
            bool response;
            decimal decimalAbn;

            if (abn == null)
                return false;

            abn = abn.Trim();

            if (!decimal.TryParse(abn, out decimalAbn))
                return true;

            if (abn.Length != 11)
                return true;

            decimal abnSum =
                10 * (int.Parse(abn.Substring(0, 1)) - 1) +
                1 * int.Parse(abn.Substring(1, 1)) +
                3 * int.Parse(abn.Substring(2, 1)) +
                5 * int.Parse(abn.Substring(3, 1)) +
                7 * int.Parse(abn.Substring(4, 1)) +
                9 * int.Parse(abn.Substring(5, 1)) +
                11 * int.Parse(abn.Substring(6, 1)) +
                13 * int.Parse(abn.Substring(7, 1)) +
                15 * int.Parse(abn.Substring(8, 1)) +
                17 * int.Parse(abn.Substring(9, 1)) +
                19 * int.Parse(abn.Substring(10, 1));

            if ((abnSum % 89) == 0)
                response = false;
            else
                response = true;

            return response;
        }
        private static bool FailsACNAlgorithm(string acn)
        {
            bool response;
            decimal decimalAbn;
            if (acn == null)
                return false;

            acn = acn.Trim();

            if (!decimal.TryParse(acn, out decimalAbn))
                return true;

            if (acn.Length != 9)
                return true;

            decimal abnSum =
                8 * int.Parse(acn.Substring(0, 1)) +
                7 * int.Parse(acn.Substring(1, 1)) +
                6 * int.Parse(acn.Substring(2, 1)) +
                5 * int.Parse(acn.Substring(3, 1)) +
                4 * int.Parse(acn.Substring(4, 1)) +
                3 * int.Parse(acn.Substring(5, 1)) +
                2 * int.Parse(acn.Substring(6, 1)) +
                1 * int.Parse(acn.Substring(7, 1));

            decimal checkDigit = int.Parse(acn.Substring(8, 1));
            decimal acnRemainder = abnSum % 10;

            if (((10 - acnRemainder) % 10) == checkDigit)
                response = false;
            else
                response = true;

            return response;
        }


        private static bool FailsTFNAlgorithm(string tfn)
        {
            bool response;
            decimal decimalTfn;

            if (tfn == null)
                return false;

            tfn = tfn.Trim();
            tfn = Regex.Replace(tfn, "^0+", "");

            if (!decimal.TryParse(tfn, out decimalTfn))
                return true;

            if (tfn.Length < 8)
                return true;


            decimal tfn1To7Sum =
                1 * int.Parse(tfn.Substring(0, 1)) +
                4 * int.Parse(tfn.Substring(1, 1)) +
                3 * int.Parse(tfn.Substring(2, 1)) +
                7 * int.Parse(tfn.Substring(3, 1)) +
                5 * int.Parse(tfn.Substring(4, 1)) +
                8 * int.Parse(tfn.Substring(5, 1)) +
                6 * int.Parse(tfn.Substring(6, 1));

            decimal tfn8 = 9 * int.Parse(tfn.Substring(7, 1));

            if (tfn.Length == 8)
            {
                decimal tFNLg8WSum9 = 10 * int.Parse(tfn.Substring(7, 1));
                decimal tFNLg8WSum = tfn1To7Sum + tFNLg8WSum9;

                if ((tFNLg8WSum % 11) == 0)
                    response = false;
                else
                    response = true;
            }
            else if (tfn.Length == 9)
            {
                decimal tfn9 = 10 * int.Parse(tfn.Substring(8, 1));
                decimal tFNLg9WSum = tfn1To7Sum + tfn8 + tfn9;

                if ((tFNLg9WSum % 11) == 0)
                    response = false;
                else
                    response = true;
            }
            else
            {
                response = true;
            }

            return response;
        }


        private static decimal ConditionalValue(bool expression, decimal? trueVal, decimal? falseVal)
        {
             return expression ? trueVal.GetValueOrDefault() : falseVal.GetValueOrDefault();
        }

        private static decimal AsNumeric(string value)
        {
             decimal numberValue;
             decimal.TryParse(value, out numberValue);
             return numberValue;
        }

        private static bool RegexMatch(int? field, string expression, string flags="")
        {
            return IsMatch(field, expression, GetRegexOption(flags));
        }

        private static bool RegexMatch(string field, string expression, string flags="")
        {
            return IsMatch(field, expression, GetRegexOption(flags));
        }

        private static RegexOptions GetRegexOption(string flags)
        {
            RegexOptions options = RegexOptions.None;

            char[] characters = flags.ToCharArray();

            foreach (char character in characters)
            {
                switch (character)
                {
                    case 'i':                        
                        options = options | RegexOptions.IgnoreCase;                       
                        break;
                    case 'm':                        
                        options = options | RegexOptions.Multiline;                       
                        break;
                    case 's':                        
                            options = options | RegexOptions.Singleline;                       
                        break;
                    case 'n':                        
                            options = options | RegexOptions.ExplicitCapture;                       
                        break;
                    case 'x':                       
                            options = options | RegexOptions.IgnorePatternWhitespace;                       
                        break;
                }
            }

            return options;
        }

        /// <summary>
        /// Returns an occurrence index as [occurrenceIndex] of occurrenceIndex > 0, otherwise the empty string
        /// </summary>
        /// <param name="occurrenceIndex">Index of the occurrence.</param>
        /// <returns>Occurrence in XPath [#] format</returns>
        public string OccurrenceIndex(int occurrenceIndex)
        {
            return occurrenceIndex > 0 ? "[" + occurrenceIndex + "]" : "";
        }

        #endregion // Functions

        public DIS2018 ConsumedReport { get; private set; }

        private static Dictionary<string, ProcessMessageDocument> _processMessageDocuments = new Dictionary<string,ProcessMessageDocument>();

        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public List<ProcessMessageDocument> ValidateReport(DIS2018 report, bool isStandalone)
        {

            List<ProcessMessageDocument> response = new List<ProcessMessageDocument>();
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            bool assertion;

            this.ConsumedReport = report;


    #region Manual Rules - Rules marked for manual coding
    
        #region VR.ATO.DIS.406041

        /*  VR.ATO.DIS.406041
            Reporting party declaration must not be supplied when submitted with a parent return

        Legacy Rule Format:
            WHERE PARENT RETURN EXISTS
            IF [DIS74] <> NULLORBLANK
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [DIS74] = DIS:RP:pyin.02.03:Declaration.StatementType.Code WHERE (pyin.02.03:Declaration.StatementType.Code = "TrueAndCorrect") IN TUPLE(declaration2.02.01:Declaration)
    
        Technical Business Rule Format:
            ^DIS74 <> BLANK

        Data Elements:

        ^DIS74 = RP:Declaration:Declaration.StatementType.Code WHERE [Declaration.StatementType.Code='TrueAndCorrect']
        */
    VRATODIS406041(response, report, isStandalone);
    #endregion // VR.ATO.DIS.406041

        #region VR.ATO.DIS.406047

/*  VR.ATO.DIS.406047
            Intermediary declaration must not be supplied when submitted with a parent return

        Legacy Rule Format:
            WHERE PARENT RETURN EXISTS
            IF [DIS81] <> NULLORBLANK
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [DIS81] = DIS:INT:pyin.02.03:Declaration.StatementType.Code WHERE (pyin.02.03:Declaration.StatementType.Code = "TrueAndCorrect") IN TUPLE(declaration2.02.01:Declaration)

    Technical Business Rule Format:
            ^DIS81 <> BLANK

Data Elements:
    
    ^DIS81 = INT:Declaration:Declaration.StatementType.Code WHERE [Declaration.StatementType.Code='TrueAndCorrect']
    */
    VRATODIS406047(response, report, isStandalone);
            #endregion // VR.ATO.DIS.406047

            

            #region VR.ATO.DIS.500004
            VRATODIS500004(response, report, isStandalone);
            #endregion

            #region VR.ATO.DIS.500005
            VRATODIS500005(response, report, isStandalone);
            #endregion

            #region VR.ATO.DIS.500006
            VRATODIS500006(response, report, isStandalone);
            #endregion

            #region VR.ATO.DIS.500007
            VRATODIS500007(response, report, isStandalone);
            #endregion

            #region VR.ATO.DIS.500008
            VRATODIS500008(response, report, isStandalone);
            #endregion

            #region VR.ATO.DIS.500009
            VRATODIS500009(response, report, isStandalone);
            #endregion

            #region VR.ATO.DIS.500010
            VRATODIS500010(response, report, isStandalone);
            #endregion

            #region VR.ATO.DIS.500011
            VRATODIS500011(response, report, isStandalone);
            #endregion

            #region VR.ATO.DIS.500012
            VRATODIS500012(response, report, isStandalone);
            #endregion

            #region VR.ATO.DIS.500013
            VRATODIS500013(response, report, isStandalone);
            #endregion

            #region VR.ATO.DIS.500014
            VRATODIS500014(response, report, isStandalone);
            #endregion

            #region VR.ATO.DIS.500015
            VRATODIS500015(response, report, isStandalone);
            #endregion

            #region VR.ATO.DIS.500016
            VRATODIS500016(response, report, isStandalone);
            #endregion

            #region VR.ATO.DIS.500017
            VRATODIS500017(response, report, isStandalone);
            #endregion

            #region VR.ATO.DIS.500018
            VRATODIS500018(response, report, isStandalone);
            #endregion

            #region VR.ATO.DIS.500019
            VRATODIS500019(response, report, isStandalone);
            #endregion

            #region VR.ATO.DIS.500020
            VRATODIS500020(response, report, isStandalone);
            #endregion

            #region VR.ATO.DIS.500021
            VRATODIS500021(response, report, isStandalone);
            #endregion

            #region VR.ATO.DIS.500022
            VRATODIS500022(response, report, isStandalone);
            #endregion

            #region VR.ATO.DIS.500023
            VRATODIS500023(response, report, isStandalone);
            #endregion

            #region VR.ATO.DIS.500024
            VRATODIS500024(response, report, isStandalone);
            #endregion

            #region VR.ATO.DIS.500025
            VRATODIS500025(response, report, isStandalone);
            #endregion

            #region VR.ATO.DIS.500026
            VRATODIS500026(response, report, isStandalone);
            #endregion

            #region VR.ATO.GEN.428041 (DIS56)
            VRATOGEN428041_01(response, report, isStandalone);
            #endregion

            #region VR.ATO.GEN.428041 (DIS70)
            VRATOGEN428041_02(response, report, isStandalone);
            #endregion

            #region VR.ATO.GEN.428041 (DIS11)
            VRATOGEN428041_03(response, report, isStandalone);
            #endregion

            #region VR.ATO.GEN.500023 (DIS24)
            VRATOGEN500023_DIS24(response, report, isStandalone);
            #endregion

            #region VR.ATO.GEN.500023 (DIS3)
            VRATOGEN500023_DIS3(response, report, isStandalone);
            #endregion

            #region VR.ATO.GEN.500023 (DIS91)
            VRATOGEN500023_DIS91(response, report, isStandalone);
            #endregion

            #region VR.ATO.GEN.500024
            VRATOGEN500024(response, report, isStandalone);
            #endregion

            #region VR.ATO.GEN.500025
            VRATOGEN500025(response, report, isStandalone);
            #endregion

            #region VR.ATO.GEN.500042
            VRATOGEN500042(response, report, isStandalone);
            #endregion

            #region VR.ATO.GEN.500059
            VRATOGEN500059(response, report, isStandalone);
            #endregion



            #endregion Manual Rules - Rules marked for manual coding

            #region Repeating report.InvestorCollection
            if (report.InvestorCollection != null)
            {    
                int itemIndex1 = 0;
                foreach (DIS2018.Investor investor in report.InvestorCollection)
                {
                    itemIndex1++;
            
                    #region VR.ATO.DIS.406002
            
                    /*  VR.ATO.DIS.406002
                    Franking Credit must be entered when a value for Franked Dividends is supplied.
    
                    Legacy Rule Format:
                    IF ([DIS17] > 2 AND [DIS16] < 1)
                     RETURN VALIDATION MESSAGE
                    ENDIF
                    
                    
                    
                    
                    [DIS16] = DIS:RP:rvctc2.02.00:IncomeTax.FrankingCredits.ReceivedFromAustralianCompanies.Amount IN TUPLE(dis.0003.lodge.req.02.01:Investor)
                    [DIS17] = DIS:RP:bafpo2.02.02:Equity.Dividends.FrankedPaid.Amount IN TUPLE(dis.0003.lodge.req.02.01:Investor)
        
                    Technical Business Rule Format:
                    ^DIS17 > 2 AND ^DIS16 < 1
            
                    Data Elements:
            
                    ^DIS16 = RP:Investor:IncomeTax.FrankingCredits.ReceivedFromAustralianCompanies.Amount
            
                    ^DIS17 = RP:Investor:Equity.Dividends.FrankedPaid.Amount
                    */
                    assertion = (investor.DIS17.GetValueOrDefault() > 2 && investor.DIS16.GetValueOrDefault() < 1);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.DIS.406002", Severity = ProcessMessageSeverity.Error,
                            Description = @"Franking Credit must be entered when a value for Franked Dividends is supplied",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:IncomeTax.FrankingCredits.ReceivedFromAustralianCompanies.Amount",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DIS.406002"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS17", Value = GetValueOrEmpty(investor.DIS17) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS16", Value = GetValueOrEmpty(investor.DIS16) });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.DIS.406002
            
                    #region VR.ATO.DIS.406003
            
                    /*  VR.ATO.DIS.406003
                    Franked Dividends must be entered  when a value for Franked Credit is supplied.
    
                    Legacy Rule Format:
                    IF ([DIS16] > 0 AND [DIS17] = NULL)
                    RETURN VALIDATION MESSAGE
                    ENDIF
                    
                    
                    
                    [DIS16] = DIS:RP:rvctc2.02.00:IncomeTax.FrankingCredits.ReceivedFromAustralianCompanies.Amount IN TUPLE(dis.0003.lodge.req.02.01:Investor)
                    [DIS17] = DIS:RP:bafpo2.02.02:Equity.Dividends.FrankedPaid.Amount IN TUPLE(dis.0003.lodge.req.02.01:Investor)
        
                    Technical Business Rule Format:
                    ^DIS16 > 0 AND ^DIS17 = NULL
            
                    Data Elements:
            
                    ^DIS16 = RP:Investor:IncomeTax.FrankingCredits.ReceivedFromAustralianCompanies.Amount
            
                    ^DIS17 = RP:Investor:Equity.Dividends.FrankedPaid.Amount
                    */
                    assertion = (investor.DIS16.GetValueOrDefault() > 0 && investor.DIS17 == null);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.DIS.406003", Severity = ProcessMessageSeverity.Error,
                            Description = @"Franked Dividends must be entered when a value for Franked Credit is supplied",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:IncomeTax.FrankingCredits.ReceivedFromAustralianCompanies.Amount",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DIS.406003"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS16", Value = GetValueOrEmpty(investor.DIS16) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS17", Value = GetValueOrEmpty(investor.DIS17) });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.DIS.406003
            
                    #region VR.ATO.DIS.406004
            
                    /*  VR.ATO.DIS.406004
                    At least one Amounts paid field should be more than or equal to one dollar.
    
                    Legacy Rule Format:
                    IF ([DIS16] < 1 AND [DIS17] < 1 AND [DIS18] < 1 AND [DIS19] < 1 AND [DIS20] < 1)
                     RETURN VALIDATION MESSAGE
                    ENDIF
                    
                    
                    
                    [DIS16] = DIS:RP:rvctc2.02.00:IncomeTax.FrankingCredits.ReceivedFromAustralianCompanies.Amount IN TUPLE(dis.0003.lodge.req.02.01:Investor)
                    [DIS17] = DIS:RP:bafpo2.02.02:Equity.Dividends.FrankedPaid.Amount IN TUPLE(dis.0003.lodge.req.02.01:Investor)
                    [DIS18] = DIS:RP.NONCFI:bafpo2.02.02:Equity.Dividends.UnfrankedPaid.Amount IN TUPLE(dis.0003.lodge.req.02.01:Investor)
                    [DIS19] = DIS:RP.CFI:bafpo2.02.02:Equity.Dividends.UnfrankedPaid.Amount IN TUPLE(dis.0003.lodge.req.02.01:Investor)
                    [DIS20] = DIS:RP:bafpr2.02.02:Expense.Interest.Paid.Amount IN TUPLE(dis.0003.lodge.req.02.01:Investor)
        
                    Technical Business Rule Format:
                    ^DIS16 < 1 AND ^DIS17 < 1 AND ^DIS18 < 1 AND ^DIS19 < 1 AND ^DIS20 < 1
            
                    Data Elements:
            
                    ^DIS16 = RP:Investor:IncomeTax.FrankingCredits.ReceivedFromAustralianCompanies.Amount
            
                    ^DIS17 = RP:Investor:Equity.Dividends.FrankedPaid.Amount
            
                    ^DIS18 = RP.NONCFI:Investor:Equity.Dividends.UnfrankedPaid.Amount
            
                    ^DIS19 = RP.CFI:Investor:Equity.Dividends.UnfrankedPaid.Amount
            
                    ^DIS20 = RP:Investor:Expense.Interest.Paid.Amount
                    */
                    assertion = (investor.DIS16.GetValueOrDefault() < 1 && investor.DIS17.GetValueOrDefault() < 1 && investor.DIS18.GetValueOrDefault() < 1 && investor.DIS19.GetValueOrDefault() < 1 && investor.DIS20.GetValueOrDefault() < 1);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.DIS.406004", Severity = ProcessMessageSeverity.Error,
                            Description = @"At least one positive amount must be provided",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:IncomeTax.FrankingCredits.ReceivedFromAustralianCompanies.Amount",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DIS.406004"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS16", Value = GetValueOrEmpty(investor.DIS16) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS17", Value = GetValueOrEmpty(investor.DIS17) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS18", Value = GetValueOrEmpty(investor.DIS18) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS19", Value = GetValueOrEmpty(investor.DIS19) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS20", Value = GetValueOrEmpty(investor.DIS20) });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.DIS.406004
            
                    #region VR.ATO.DIS.406006
            
                    /*  VR.ATO.DIS.406006
                    If deceased indicator = True then Investor must be an individual, not an organisation
    
                    Legacy Rule Format:
                    IF ([DIS44] = TRUE AND [DIS15] <> “060”)
                    RETURN VALIDATION MESSAGE
                    ENDIF
                    
                    
                    
                    
                    [DIS15] = DIS:RP:pyde.02.21:Party.Type.Code IN TUPLE(dis.0003.lodge.req.02.01:Investor)
                    [DIS44] = DIS:RP:pyde.02.00:PersonDemographicDetails.Deceased.Indicator IN TUPLE(dis.0003.lodge.req.02.01:Investor)
        
                    Technical Business Rule Format:
                    ^DIS44 = TRUE AND ^DIS15 <> '060'
            
                    Data Elements:
            
                    ^DIS44 = RP:Investor:PersonDemographicDetails.Deceased.Indicator
            
                    ^DIS15 = RP:Investor:Party.Type.Code
                    */
                    assertion = (investor.DIS44 == true && investor.DIS15 != @"060");
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.DIS.406006", Severity = ProcessMessageSeverity.Error,
                            Description = @"Deceased flagged when the investor is not an Individual",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:PersonDemographicDetails.Deceased.Indicator",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DIS.406006"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS44", Value = GetValueOrEmpty(investor.DIS44) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS15", Value = investor.DIS15 });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.DIS.406006
            
                    #region VR.ATO.DIS.406010
            
                    /*  VR.ATO.DIS.406010
                    If indicator = False then Country Name must be blank
    
                    Legacy Rule Format:
                    IF ([DIS41] = FALSE AND [DIS12] <> NULLORBLANK)
                    RETURN VALIDATION MESSAGE
                    ENDIF
                    
                    
                    
                    [DIS12] = DIS:RP:pyde.02.08:AddressDetails.CountryName.Text WHERE (pyde.02.01:AddressDetails.Usage.Code = "POS" AND pyde.02.00:AddressDetails.Currency.Code = "C") IN TUPLE(address2.02.02:AddressDetails) IN TUPLE(dis.0003.lodge.req.02.01:Investor)
                    [DIS41] = DIS:RP:pyde.02.00:AddressDetails.OverseasAddress.Indicator WHERE (pyde.02.01:AddressDetails.Usage.Code = "POS" AND pyde.02.00:AddressDetails.Currency.Code = "C") IN TUPLE(address2.02.02:AddressDetails) IN TUPLE(dis.0003.lodge.req.02.01:Investor)
        
                    Technical Business Rule Format:
                    ^DIS41 = FALSE AND ^DIS12 <> BLANK
            
                    Data Elements:
            
                    ^DIS41 = RP:Investor:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            
                    ^DIS12 = RP:Investor:AddressDetails:AddressDetails.CountryName.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = (investor.DIS41 == false && string.IsNullOrWhiteSpace(investor.DIS12) != true);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.DIS.406010", Severity = ProcessMessageSeverity.Error,
                            Description = @"If the Overseas Indicator is blank or False the country must be blank",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(investor.Investor_AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.OverseasAddress.Indicator",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DIS.406010"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS41", Value = GetValueOrEmpty(investor.DIS41) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS12", Value = GetValueOrEmpty(investor.DIS12) });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.DIS.406010
            
                    #region VR.ATO.DIS.406012
            
                    /*  VR.ATO.DIS.406012
    
            Legacy Rule Format:
            IF ([DIS16] <> NULL) AND ([DIS16] <> MONETARY(U,12,0))
             RETURN VALIDATION ERROR
            ENDIF
            
            
            
            
            [DIS16] = DIS:RP:rvctc2.02.00:IncomeTax.FrankingCredits.ReceivedFromAustralianCompanies.Amount IN TUPLE(dis.0003.lodge.req.02.01:Investor)
        
                    Technical Business Rule Format:
                    NotMonetary(^DIS16, 'U', 12, 0)
            
                    Data Elements:
            
                    ^DIS16 = RP:Investor:IncomeTax.FrankingCredits.ReceivedFromAustralianCompanies.Amount
                    */
                    assertion = NotMonetary(investor.DIS16, @"U", 12, 0);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                            Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                            LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:IncomeTax.FrankingCredits.ReceivedFromAustralianCompanies.Amount",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DIS.406012"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS16", Value = GetValueOrEmpty(investor.DIS16) });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.DIS.406012
            
                    #region VR.ATO.DIS.406013
            
                    /*  VR.ATO.DIS.406013
    
            Legacy Rule Format:
            IF ([DIS17] <> NULL) AND ([DIS17] <> MONETARY(U,12,0))
             RETURN VALIDATION ERROR
            ENDIF
            
            
            
            
            [DIS17] = DIS:RP:bafpo2.02.02:Equity.Dividends.FrankedPaid.Amount IN TUPLE(dis.0003.lodge.req.02.01:Investor)
        
                    Technical Business Rule Format:
                    NotMonetary(^DIS17, 'U', 12, 0)
            
                    Data Elements:
            
                    ^DIS17 = RP:Investor:Equity.Dividends.FrankedPaid.Amount
                    */
                    assertion = NotMonetary(investor.DIS17, @"U", 12, 0);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                            Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                            LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:Equity.Dividends.FrankedPaid.Amount",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DIS.406013"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS17", Value = GetValueOrEmpty(investor.DIS17) });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.DIS.406013
            
                    #region VR.ATO.DIS.406014
            
                    /*  VR.ATO.DIS.406014
    
            Legacy Rule Format:
            IF ([DIS20] <> NULL) AND ([DIS20] <> MONETARY(U,12,0))
             RETURN VALIDATION ERROR
            ENDIF
            
            
            
            
            [DIS20] = DIS:RP:bafpr2.02.02:Expense.Interest.Paid.Amount IN TUPLE(dis.0003.lodge.req.02.01:Investor)
        
                    Technical Business Rule Format:
                    NotMonetary(^DIS20, 'U', 12, 0)
            
                    Data Elements:
            
                    ^DIS20 = RP:Investor:Expense.Interest.Paid.Amount
                    */
                    assertion = NotMonetary(investor.DIS20, @"U", 12, 0);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                            Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                            LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:Expense.Interest.Paid.Amount",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DIS.406014"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS20", Value = GetValueOrEmpty(investor.DIS20) });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.DIS.406014
            
                    #region VR.ATO.DIS.406015
            
                    /*  VR.ATO.DIS.406015
                    Non-individual investors are required to enter their investor name details
    
                    Legacy Rule Format:
                    IF ([DIS15] <>SET ("117", "060") AND [DIS3] = BLANK)
                    RETURN VALIDATION ERROR
                    ENDIF
                    
                    
                    
                    
                    [DIS3] = DIS:RP:pyde.02.00:OrganisationNameDetails.OrganisationalName.Text WHERE (pyde.02.00:OrganisationNameDetails.OrganisationalNameType.Code = "MN") IN TUPLE(orgname1.02.00:OrganisationNameDetails) IN TUPLE(dis.0003.lodge.req.02.01:Investor)
                    [DIS15] = DIS:RP:pyde.02.21:Party.Type.Code IN TUPLE(dis.0003.lodge.req.02.01:Investor)
        
                    Technical Business Rule Format:
                    (NotInSet(^DIS15, '"117", "060"') AND ^DIS3 = BLANK)
            
                    Data Elements:
            
                    ^DIS15 = RP:Investor:Party.Type.Code
            
                    ^DIS3 = RP:Investor:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN']
                    */
                    assertion = (!(IsMatch(investor.DIS15, @"^(117|060)$",RegexOptions.IgnoreCase)) && string.IsNullOrWhiteSpace(investor.DIS3) == true);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.DIS.406012", Severity = ProcessMessageSeverity.Error,
                            Description = @"Non-individual investor name details missing",
                            LongDescription = @"Non-individual investors are required to enter their investor name details",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:Party.Type.Code",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DIS.406015"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS15", Value = investor.DIS15 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS3", Value = investor.DIS3 });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.DIS.406015
            
                    #region VR.ATO.DIS.406016
            
                    /*  VR.ATO.DIS.406016
                    Individual or Deceased individual investor's surname details should be entered
    
                    Legacy Rule Format:
                    IF ([DIS15] = SET ("117", "060") AND [DIS4] = BLANK)
                    RETURN VALIDATION ERROR
                    ENDIF
                    
                    
                    
                    
                    
                    [DIS4] = DIS:RP:pyde.02.00:PersonNameDetails.FamilyName.Text IN TUPLE(prsnstrcnm1.02.00:PersonNameDetails) IN TUPLE(dis.0003.lodge.req.02.01:Investor)
                    [DIS15] = DIS:RP:pyde.02.21:Party.Type.Code IN TUPLE(dis.0003.lodge.req.02.01:Investor)
        
                    Technical Business Rule Format:
                    (InSet(^DIS15, '"117", "060"') AND ^DIS4 = BLANK)
            
                    Data Elements:
            
                    ^DIS15 = RP:Investor:Party.Type.Code
            
                    ^DIS4 = RP:Investor:PersonNameDetails:PersonNameDetails.FamilyName.Text
                    */
                    assertion = (IsMatch(investor.DIS15, @"^(117|060)$") && string.IsNullOrWhiteSpace(investor.DIS4) == true);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.DIS.406013", Severity = ProcessMessageSeverity.Error,
                            Description = @"Surname details have to be entered",
                            LongDescription = @"Individual or Deceased individual investor's surname details should be entered",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:Party.Type.Code",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DIS.406016"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS15", Value = investor.DIS15 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS4", Value = investor.DIS4 });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.DIS.406016
            
                    #region VR.ATO.DIS.406017
            
                    /*  VR.ATO.DIS.406017
    
            Legacy Rule Format:
            IF ([DIS15] <>SET ("117","051","212","161","210","060"))
            RETURN VALIDATION ERROR
            ENDIF
            
            
            
            
            [DIS15] = DIS:RP:pyde.02.21:Party.Type.Code IN TUPLE(dis.0003.lodge.req.02.01:Investor)
        
                    Technical Business Rule Format:
                    (NotInSet(^DIS15, '"117","051","212","161","210","060"'))
            
                    Data Elements:
            
                    ^DIS15 = RP:Investor:Party.Type.Code
                    */
                    assertion = !(IsMatch(investor.DIS15, @"^(117|051|212|161|210|060)$",RegexOptions.IgnoreCase));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.DIS.406014", Severity = ProcessMessageSeverity.Error,
                            Description = @"Incorrect Investor Type Code received",
                            LongDescription = @"Investor Type Code valid values must be within the set ""I, C, T, P, S, D""",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:Party.Type.Code",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DIS.406017"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS15", Value = investor.DIS15 });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.DIS.406017
            
                    #region VR.ATO.DIS.406018
            
                    /*  VR.ATO.DIS.406018
    
            Legacy Rule Format:
            IF ([DIS18] <> NULL) AND ([DIS18] <> MONETARY(U,12,0))
             RETURN VALIDATION ERROR
            ENDIF
            
            
            
            
            [DIS18] = DIS:RP.NONCFI:bafpo2.02.02:Equity.Dividends.UnfrankedPaid.Amount IN TUPLE(dis.0003.lodge.req.02.01:Investor)
        
                    Technical Business Rule Format:
                    NotMonetary(^DIS18, 'U', 12, 0)
            
                    Data Elements:
            
                    ^DIS18 = RP.NONCFI:Investor:Equity.Dividends.UnfrankedPaid.Amount
                    */
                    assertion = NotMonetary(investor.DIS18, @"U", 12, 0);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                            Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                            LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:Equity.Dividends.UnfrankedPaid.Amount[@contextRef='" + report.RPNONCFIId + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DIS.406018"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS18", Value = GetValueOrEmpty(investor.DIS18) });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.DIS.406018
            
                    #region VR.ATO.DIS.406019
            
                    /*  VR.ATO.DIS.406019
    
            Legacy Rule Format:
            IF ([DIS19] <> NULL) AND ([DIS19] <> MONETARY(U,12,0))
             RETURN VALIDATION ERROR
            ENDIF
            
            
            
            
            [DIS19] = DIS:RP.CFI:bafpo2.02.02:Equity.Dividends.UnfrankedPaid.Amount IN TUPLE(dis.0003.lodge.req.02.01:Investor)
        
                    Technical Business Rule Format:
                    NotMonetary(^DIS19, 'U', 12, 0)
            
                    Data Elements:
            
                    ^DIS19 = RP.CFI:Investor:Equity.Dividends.UnfrankedPaid.Amount
                    */
                    assertion = NotMonetary(investor.DIS19, @"U", 12, 0);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                            Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                            LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:Equity.Dividends.UnfrankedPaid.Amount[@contextRef='" + report.RPCFIId + "']",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DIS.406019"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS19", Value = GetValueOrEmpty(investor.DIS19) });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.DIS.406019
            
                    #region VR.ATO.DIS.406020
            
                    /*  VR.ATO.DIS.406020
                    Date must not be in the future
    
                    Legacy Rule Format:
                    IF ([DIS13] > CURRENTDATE)
                    RETURN VALIDATION MESSAGE
                    ENDIF
                    
                    
                    [DIS13] = DIS:RP:pyde.02.00:PersonDemographicDetails.Birth.Date IN TUPLE(dis.0003.lodge.req.02.01:Investor)
        
                    Technical Business Rule Format:
                    (^DIS13 > Today())
            
                    Data Elements:
            
                    ^DIS13 = RP:Investor:PersonDemographicDetails.Birth.Date
                    */
                    assertion = (investor.DIS13.GetValueOrDefault() > DateTime.Now.Date);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.DIS.406015", Severity = ProcessMessageSeverity.Error,
                            Description = @"Date of birth can not be a future date",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:PersonDemographicDetails.Birth.Date",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DIS.406020"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS13", Value = GetValueOrEmpty(investor.DIS13) });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.DIS.406020
            
                    #region VR.ATO.DIS.406022
            
                    /*  VR.ATO.DIS.406022
    
            Legacy Rule Format:
            IF ([DIS7] = NULLORBLANK)
             RETURN VALIDATION MESSAGE
            ENDIF
            
            
            
            [DIS7] = DIS:RP:pyde.02.00:AddressDetails.Line1.Text WHERE (pyde.02.01:AddressDetails.Usage.Code = "POS" AND pyde.02.00:AddressDetails.Currency.Code = "C") IN TUPLE(address2.02.02:AddressDetails) IN TUPLE(dis.0003.lodge.req.02.01:Investor)
        
                    Technical Business Rule Format:
                    (^AddressDetails <> NULL) AND (^DIS7 = BLANK)
            
                    Data Elements:
            
                    ^DIS7 = RP:Investor:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            
                    ^AddressDetails = RP:Investor:AddressDetails
                    */
                    assertion = (investor.Investor_AddressDetailsCollection_POS_C_RPExists != false && string.IsNullOrWhiteSpace(investor.DIS7) == true);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.DIS.406016", Severity = ProcessMessageSeverity.Error,
                            Description = @"Investor Address Line 1 is required",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(investor.Investor_AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DIS.406022"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS7", Value = investor.DIS7 });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.DIS.406022
            
                    #region VR.ATO.DIS.406023
            
                    /*  VR.ATO.DIS.406023
                    Both Investor Organisation Name Details and Person Name Details are present. Only one is allowed.
    
                    Legacy Rule Format:
                    IF [DIS3] <> NULLORBLANK AND ([DIS4] <> NULLORBLANK OR [DIS5] <> NULLORBLANK OR [DIS6] <> NULLORBLANK)
                      RETURN VALIDATION MESSAGE
                    ENDIF
                    
                    
                    
                    [DIS3] = DIS:RP:pyde.02.00:OrganisationNameDetails.OrganisationalName.Text WHERE (pyde.02.00:OrganisationNameDetails.OrganisationalNameType.Code = "MN") IN TUPLE(orgname1.02.00:OrganisationNameDetails) IN TUPLE(dis.0003.lodge.req.02.01:Investor)
                    [DIS4] = DIS:RP:pyde.02.00:PersonNameDetails.FamilyName.Text IN TUPLE(prsnstrcnm1.02.00:PersonNameDetails) IN TUPLE(dis.0003.lodge.req.02.01:Investor)
                    [DIS5] = DIS:RP:pyde.02.00:PersonNameDetails.GivenName.Text IN TUPLE(prsnstrcnm1.02.00:PersonNameDetails) IN TUPLE(dis.0003.lodge.req.02.01:Investor)
                    [DIS6] = DIS:RP:pyde.02.00:PersonNameDetails.OtherGivenName.Text IN TUPLE(prsnstrcnm1.02.00:PersonNameDetails) IN TUPLE(dis.0003.lodge.req.02.01:Investor)
        
                    Technical Business Rule Format:
                    ^DIS3 <> BLANK AND (^DIS4 <> BLANK OR ^DIS5 <> BLANK OR ^DIS6 <> BLANK)
            
                    Data Elements:
            
                    ^DIS3 = RP:Investor:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN']
            
                    ^DIS4 = RP:Investor:PersonNameDetails:PersonNameDetails.FamilyName.Text
            
                    ^DIS5 = RP:Investor:PersonNameDetails:PersonNameDetails.GivenName.Text
            
                    ^DIS6 = RP:Investor:PersonNameDetails:PersonNameDetails.OtherGivenName.Text
                    */
                    assertion = (string.IsNullOrWhiteSpace(investor.DIS3) != true && (string.IsNullOrWhiteSpace(investor.DIS4) != true || string.IsNullOrWhiteSpace(investor.DIS5) != true || string.IsNullOrWhiteSpace(investor.DIS6) != true));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.DIS.406023", Severity = ProcessMessageSeverity.Error,
                            Description = @"Both Investor Organisation Name Details and Person Name Details are present. Only one is allowed.",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:OrganisationNameDetails" + OccurrenceIndex(investor.Investor_OrganisationNameDetailsCollection_MN_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DIS.406023"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS3", Value = investor.DIS3 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS4", Value = investor.DIS4 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS5", Value = investor.DIS5 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS6", Value = GetValueOrEmpty(investor.DIS6) });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.DIS.406023
            
                    #region VR.ATO.GEN.000406
            
                    /*  VR.ATO.GEN.000406
                    Address line 1 contains incorrect 'care of' reference.  The only acceptable 'care of' reference is C/-
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:AddressDetails.Line1.Text <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.Line1.Text STARTSWITH SET("C/O ","C/ ","Care Of ","CO ")) OR (pyde.xx.xx:AddressDetails.Line1.Text CONTAINS SET(" C/O "," C/ "," Care Of "))
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^DIS7 <> BLANK) AND (StartsWithSet(^DIS7,'"C/O ","C/ ","Care Of ","CO "')) OR (ContainsSet(^DIS7, '" C/O "," C/ "," Care Of "'))
            
                    Data Elements:
            
                    ^DIS7 = RP:Investor:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = (string.IsNullOrWhiteSpace(investor.DIS7) != true && IsMatch(investor.DIS7, @"^(C/O |C/ |Care Of |CO )",RegexOptions.IgnoreCase) || IsMatch(investor.DIS7, @" C/O | C/ | Care Of ",RegexOptions.IgnoreCase));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000406", Severity = ProcessMessageSeverity.Error,
                            Description = @"Address line 1 contains incorrect 'care of' reference.  The only acceptable 'care of' reference is C/-",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(investor.Investor_AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000406"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS7", Value = investor.DIS7 });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.000406
            
                    #region VR.ATO.GEN.000408
            
                    /*  VR.ATO.GEN.000408
                    Care of' is not valid for address line 2
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:AddressDetails.Line2.Text <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.Line2.Text STARTSWITH SET("C/- ","C/O ","C/ ","Care Of ","CO ") OR pyde.xx.xx:AddressDetails.Line2.Text CONTAINS SET(" C/- "," C/O "," C/ "," Care Of "))
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^DIS8 <> BLANK) AND (StartsWithSet(^DIS8,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^DIS8, '" C/- "," C/O "," C/ "," Care Of "'))
            
                    Data Elements:
            
                    ^DIS8 = RP:Investor:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = (string.IsNullOrWhiteSpace(investor.DIS8) != true && (IsMatch(investor.DIS8, @"^(C/- |C/O |C/ |Care Of |CO )",RegexOptions.IgnoreCase) || IsMatch(investor.DIS8, @" C/- | C/O | C/ | Care Of ",RegexOptions.IgnoreCase)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000408", Severity = ProcessMessageSeverity.Error,
                            Description = @"Care of' is not valid for address line 2",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(investor.Investor_AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line2.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000408"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS8", Value = GetValueOrEmpty(investor.DIS8) });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.000408
            
                    #region VR.ATO.GEN.000423
            
                    /*  VR.ATO.GEN.000423
                    Family name cannot contain " - "
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text CONTAINS " - ")
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    Contains(^DIS4, ' - ')
            
                    Data Elements:
            
                    ^DIS4 = RP:Investor:PersonNameDetails:PersonNameDetails.FamilyName.Text
                    */
                    assertion = (investor.DIS4 == null ? false : investor.DIS4.ToUpperInvariant().Contains(@" - "));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000423", Severity = ProcessMessageSeverity.Error,
                            Description = @"Family Name cannot contain ""space hyphen space""",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:PersonNameDetails/tns:PersonNameDetails.FamilyName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000423"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS4", Value = investor.DIS4 });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.000423
            
                    #region VR.ATO.GEN.000427
            
                    /*  VR.ATO.GEN.000427
                    Family name cannot contain "--", """", "  "
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text CONTAINS SET("--", """", "  "))
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (ContainsSet(^DIS4, '"--", """", "  "'))
            
                    Data Elements:
            
                    ^DIS4 = RP:Investor:PersonNameDetails:PersonNameDetails.FamilyName.Text
                    */
                    assertion = IsMatch(investor.DIS4, @"--|""""|  ",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000427", Severity = ProcessMessageSeverity.Error,
                            Description = @"Family name cannot contain repeated hyphen, apostrophe, or space characters",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:PersonNameDetails/tns:PersonNameDetails.FamilyName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000427"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS4", Value = investor.DIS4 });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.000427
            
                    #region VR.ATO.GEN.000434
            
                    /*  VR.ATO.GEN.000434
                    invalid Given name
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:PersonNameDetails.GivenName.Text CONTAINS " - ")
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    Contains(^DIS5, ' - ')
            
                    Data Elements:
            
                    ^DIS5 = RP:Investor:PersonNameDetails:PersonNameDetails.GivenName.Text
                    */
                    assertion = (investor.DIS5 == null ? false : investor.DIS5.ToUpperInvariant().Contains(@" - "));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000434", Severity = ProcessMessageSeverity.Error,
                            Description = @"First name cannot contain space hyphen space",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:PersonNameDetails/tns:PersonNameDetails.GivenName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000434"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS5", Value = investor.DIS5 });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.000434
            
                    #region VR.ATO.GEN.000439
            
                    /*  VR.ATO.GEN.000439
                    Given name cannot contain "--", """", "  "
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:PersonNameDetails.GivenName.Text CONTAINS SET("--", """", "  "))
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (ContainsSet(^DIS5, '"--", """", "  "'))
            
                    Data Elements:
            
                    ^DIS5 = RP:Investor:PersonNameDetails:PersonNameDetails.GivenName.Text
                    */
                    assertion = IsMatch(investor.DIS5, @"--|""""|  ",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000439", Severity = ProcessMessageSeverity.Error,
                            Description = @"First name cannot contain repeated hyphen, apostrophe, or space characters",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:PersonNameDetails/tns:PersonNameDetails.GivenName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000439"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS5", Value = investor.DIS5 });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.000439
            
                    #region VR.ATO.GEN.000446
            
                    /*  VR.ATO.GEN.000446
                    OtherGivenName cannot contain  " - "
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text CONTAINS " - ")
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    Contains(^DIS6, ' - ')
            
                    Data Elements:
            
                    ^DIS6 = RP:Investor:PersonNameDetails:PersonNameDetails.OtherGivenName.Text
                    */
                    assertion = (investor.DIS6 == null ? false : investor.DIS6.ToUpperInvariant().Contains(@" - "));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000446", Severity = ProcessMessageSeverity.Error,
                            Description = @"Other given name cannot contain space hyphen space",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:PersonNameDetails/tns:PersonNameDetails.OtherGivenName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000446"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS6", Value = GetValueOrEmpty(investor.DIS6) });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.000446
            
                    #region VR.ATO.GEN.000451
            
                    /*  VR.ATO.GEN.000451
                    OtherGivenName cannot contain "--", """", "  "
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text CONTAINS SET("--", """", "  "))
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (ContainsSet(^DIS6, '"--", """", "  "'))
            
                    Data Elements:
            
                    ^DIS6 = RP:Investor:PersonNameDetails:PersonNameDetails.OtherGivenName.Text
                    */
                    assertion = IsMatch(investor.DIS6, @"--|""""|  ",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000451", Severity = ProcessMessageSeverity.Error,
                            Description = @"Other given name cannot contain repeated hyphen, apostrophe, or space characters",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:PersonNameDetails/tns:PersonNameDetails.OtherGivenName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000451"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS6", Value = GetValueOrEmpty(investor.DIS6) });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.000451
            
                    #region VR.ATO.GEN.000530
            
                    /*  VR.ATO.GEN.000530
                    Family name cannot be null or blank
    
                    Legacy Rule Format:
                    WHERE IN TUPLE(prsnstrcnm1.xx.xx:PersonNameDetails)
                    IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text = NULLORBLANK) 
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^PersonNameDetails <> NULL) AND (^DIS4 = BLANK)
            
                    Data Elements:
            
                    ^DIS4 = RP:Investor:PersonNameDetails:PersonNameDetails.FamilyName.Text
            
                    ^PersonNameDetails = RP:Investor:PersonNameDetails
                    */
                    assertion = (investor.Investor_PersonNameDetailsCollectionExists != false && string.IsNullOrWhiteSpace(investor.DIS4) == true);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410217", Severity = ProcessMessageSeverity.Error,
                            Description = @"Family Name must be present",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:PersonNameDetails/tns:PersonNameDetails.FamilyName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000530"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS4", Value = investor.DIS4 });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.000530
            
                    #region VR.ATO.GEN.000531
            
                    /*  VR.ATO.GEN.000531
                    GivenName cannot be null or blank
    
                    Legacy Rule Format:
                    WHERE IN TUPLE(prsnstrcnm1.xx.xx:PersonNameDetails) 
                    IF (pyde.xx.xx:PersonNameDetails.GivenName.Text = NULLORBLANK) 
                        RETURN VALIDATION MESSAGE 
                     ENDIF
        
                    Technical Business Rule Format:
                    (^PersonNameDetails <> NULL) AND (^DIS5 = BLANK)
            
                    Data Elements:
            
                    ^DIS5 = RP:Investor:PersonNameDetails:PersonNameDetails.GivenName.Text
            
                    ^PersonNameDetails = RP:Investor:PersonNameDetails
                    */
                    assertion = (investor.Investor_PersonNameDetailsCollectionExists != false && string.IsNullOrWhiteSpace(investor.DIS5) == true);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410218", Severity = ProcessMessageSeverity.Error,
                            Description = @"FirstName must be entered if OtherGivenNames is entered",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:PersonNameDetails/tns:PersonNameDetails.GivenName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000531"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS5", Value = investor.DIS5 });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.000531
            
                    #region VR.ATO.GEN.410002
            
                    /*  VR.ATO.GEN.410002
                    Address Line 2 must be present if Address Line 1 contains 'C/-'
    
                    Legacy Rule Format:
                    WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
                    IF ((pyde.xx.xx:AddressDetails.Line1.Text CONTAINS "C/-") AND (pyde.xx.xx:AddressDetails.Line2.Text = NULLORBLANK))  
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    ((Contains(^DIS7, 'C/-')) AND (^DIS8 = BLANK))
            
                    Data Elements:
            
                    ^DIS7 = RP:Investor:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            
                    ^DIS8 = RP:Investor:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = ((investor.DIS7 == null ? false : investor.DIS7.ToUpperInvariant().Contains(@"C/-")) && string.IsNullOrWhiteSpace(investor.DIS8) == true);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410002", Severity = ProcessMessageSeverity.Error,
                            Description = @"Address Line 2 must be present if Address Line 1 contains 'C/-'",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(investor.Investor_AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410002"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS7", Value = investor.DIS7 });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS8", Value = GetValueOrEmpty(investor.DIS8) });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.410002
            
                    #region VR.ATO.GEN.410008
            
                    /*  VR.ATO.GEN.410008
                    Address Line 3 must be left blank
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:AddressDetails.Line3.Text <> NULLORBLANK) 
                      RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    ^DIS38 <> BLANK
            
                    Data Elements:
            
                    ^DIS38 = RP:Investor:AddressDetails:AddressDetails.Line3.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = (string.IsNullOrWhiteSpace(investor.DIS38) != true);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410008", Severity = ProcessMessageSeverity.Error,
                            Description = @"Address Line 3 must be left blank",
                            LongDescription = @"Lines 3 and 4 of the address are not used on this form",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(investor.Investor_AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line3.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410008"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS38", Value = GetValueOrEmpty(investor.DIS38) });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.410008
            
                    #region VR.ATO.GEN.410013
            
                    /*  VR.ATO.GEN.410013
                    Address Line 4 must be left blank
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:AddressDetails.Line4.Text <> NULLORBLANK)
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    ^DIS39 <> BLANK
            
                    Data Elements:
            
                    ^DIS39 = RP:Investor:AddressDetails:AddressDetails.Line4.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = (string.IsNullOrWhiteSpace(investor.DIS39) != true);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410013", Severity = ProcessMessageSeverity.Error,
                            Description = @"Address Line 4 must be left blank",
                            LongDescription = @"Lines 3 and 4 of the address are not used on this form",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(investor.Investor_AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line4.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410013"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS39", Value = GetValueOrEmpty(investor.DIS39) });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.410013
            
                    #region VR.ATO.GEN.410031
            
                    /*  VR.ATO.GEN.410031
    
            Legacy Rule Format:
            IF (pyid.xx.xx:Identifiers.TaxFileNumber.Identifier <> NULLORBLANK) AND (TFNALGORITHM (pyid.xx.xx:Identifiers.TaxFileNumber.Identifier) = FALSE)
              RETURN VALIDATION MESSAGE
            ENDIF
        
                    Technical Business Rule Format:
                    (^DIS1 <> BLANK) AND (FailsTFNAlgorithm(^DIS1))
            
                    Data Elements:
            
                    ^DIS1 = RP:Investor:Identifiers.TaxFileNumber.Identifier
                    */
                    assertion = (string.IsNullOrWhiteSpace(investor.DIS1) != true && FailsTFNAlgorithm(investor.DIS1));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410031", Severity = ProcessMessageSeverity.Error,
                            Description = @"TFN must be a valid TFN",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:Identifiers.TaxFileNumber.Identifier",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410031"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS1", Value = GetValueOrEmpty(investor.DIS1) });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.410031
            
                    #region VR.ATO.GEN.410039
            
                    /*  VR.ATO.GEN.410039
                    Organisation name must contain at least one alpha or numeric character
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text  <> NULLORBLANK) AND (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text  DOES NOT CONTAIN SET("a-z", "A-Z", "0-9"))
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^DIS3  <> BLANK) AND (NotContainsSet(^DIS3, '"a-z", "A-Z", "0-9"'))
            
                    Data Elements:
            
                    ^DIS3 = RP:Investor:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN']
                    */
                    assertion = (string.IsNullOrWhiteSpace(investor.DIS3) != true && !(IsMatch(investor.DIS3, @"[a-z]|[A-Z]|[0-9]",RegexOptions.IgnoreCase)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410039", Severity = ProcessMessageSeverity.Error,
                            Description = @"Organisation name must contain at least one alpha or numeric character",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:OrganisationNameDetails" + OccurrenceIndex(investor.Investor_OrganisationNameDetailsCollection_MN_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410039"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS3", Value = investor.DIS3 });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.410039
            
                    #region VR.ATO.GEN.410040
            
                    /*  VR.ATO.GEN.410040
                    Invalid family name
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.FamilyName.Text DOES NOT CONTAIN SET("A-Z","a-z"))
                       RETURN VALIDATION MESSSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^DIS4 <> BLANK) AND (NotContainsSet(^DIS4, '"A-Z","a-z"'))
            
                    Data Elements:
            
                    ^DIS4 = RP:Investor:PersonNameDetails:PersonNameDetails.FamilyName.Text
                    */
                    assertion = (string.IsNullOrWhiteSpace(investor.DIS4) != true && !(IsMatch(investor.DIS4, @"[A-Z]|[a-z]",RegexOptions.IgnoreCase)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410040", Severity = ProcessMessageSeverity.Error,
                            Description = @"Family name must contain at least one alphabetical character",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:PersonNameDetails/tns:PersonNameDetails.FamilyName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410040"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS4", Value = investor.DIS4 });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.410040
            
                    #region VR.ATO.GEN.410063
            
                    /*  VR.ATO.GEN.410063
                    invalid Given name
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:PersonNameDetails.GivenName.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.GivenName.Text DOES NOT CONTAIN SET("A-Z","a-z"))
                       RETURN VALIDATION MESSSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^DIS5 <> BLANK) AND (NotContainsSet(^DIS5, '"A-Z","a-z"'))
            
                    Data Elements:
            
                    ^DIS5 = RP:Investor:PersonNameDetails:PersonNameDetails.GivenName.Text
                    */
                    assertion = (string.IsNullOrWhiteSpace(investor.DIS5) != true && !(IsMatch(investor.DIS5, @"[A-Z]|[a-z]",RegexOptions.IgnoreCase)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410063", Severity = ProcessMessageSeverity.Error,
                            Description = @"Given Name must contain at least one alphabetical character",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:PersonNameDetails/tns:PersonNameDetails.GivenName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410063"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS5", Value = investor.DIS5 });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.410063
            
                    #region VR.ATO.GEN.410167
            
                    /*  VR.ATO.GEN.410167
                    Overseas address indicator must be present
    
                    Legacy Rule Format:
                    WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
                    IF (pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = NULLORBLANK)
                      RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^AddressDetails <> NULL) AND (^DIS41 = BLANK)
            
                    Data Elements:
            
                    ^DIS41 = RP:Investor:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            
                    ^AddressDetails = RP:Investor:AddressDetails
                    */
                    assertion = (investor.Investor_AddressDetailsCollection_POS_C_RPExists != false && investor.DIS41 == null);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410167", Severity = ProcessMessageSeverity.Error,
                            Description = @"Overseas address indicator must be present",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(investor.Investor_AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.OverseasAddress.Indicator",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410167"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS41", Value = GetValueOrEmpty(investor.DIS41) });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.410167
            
                    #region VR.ATO.GEN.410191
            
                    /*  VR.ATO.GEN.410191
                    Country code must be present for overseas address
    
                    Legacy Rule Format:
                    WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
                    IF (pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = TRUE) AND (pyde.xx.xx:AddressDetails.Country.Code = NULLORBLANK)
                      RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^DIS41 = TRUE) AND (^DIS40 = BLANK)
            
                    Data Elements:
            
                    ^DIS40 = RP:Investor:AddressDetails:AddressDetails.Country.Code WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            
                    ^DIS41 = RP:Investor:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = (investor.DIS41 == true && string.IsNullOrWhiteSpace(investor.DIS40) == true);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410191", Severity = ProcessMessageSeverity.Error,
                            Description = @"Country code must be present for overseas address",
                            LongDescription = @"When the overseas address indicator is 'TRUE', the corresponding Country Code must be supplied",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(investor.Investor_AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Country.Code",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410191"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS41", Value = GetValueOrEmpty(investor.DIS41) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS40", Value = GetValueOrEmpty(investor.DIS40) });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.410191
            
                    #region VR.ATO.GEN.410194
            
                    /*  VR.ATO.GEN.410194
                    Address Line 1 must not exceed 38 characters
    
                    Legacy Rule Format:
                    IF LENGTH(pyde.xx.xx:AddressDetails.Line1.Text) > 38 
                        RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    Length(^DIS7) > 38
            
                    Data Elements:
            
                    ^DIS7 = RP:Investor:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = (Length(investor.DIS7) > 38);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410194", Severity = ProcessMessageSeverity.Error,
                            Description = @"Address Line 1 must not exceed 38 characters",
                            LongDescription = @"Although SBR sets a limit of 50 characters for the Line 1 of an address, the maximum allowable for this form is 38",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(investor.Investor_AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410194"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS7", Value = investor.DIS7 });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.410194
            
                    #region VR.ATO.GEN.410195
            
                    /*  VR.ATO.GEN.410195
                    Address Line 2 must not exceed 38 characters
    
                    Legacy Rule Format:
                    IF LENGTH(pyde.xx.xx:AddressDetails.Line2.Text) > 38
                         RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    Length(^DIS8) > 38
            
                    Data Elements:
            
                    ^DIS8 = RP:Investor:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = (Length(investor.DIS8) > 38);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410195", Severity = ProcessMessageSeverity.Error,
                            Description = @"Address Line 2 must not exceed 38 characters",
                            LongDescription = @"Although SBR sets a limit of 50 characters for the Line 2 of an address, the maximum allowable for this form is 38",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(investor.Investor_AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line2.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410195"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS8", Value = GetValueOrEmpty(investor.DIS8) });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.410195
            
                    #region VR.ATO.GEN.410200
            
                    /*  VR.ATO.GEN.410200
                    Family name cannot contain "Exec for", "Rep for" or "Trustee for"
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text = FOUND("Exec for","Rep for","Trustee for"))
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (FoundSet(^DIS4, '"Exec for","Rep for","Trustee for"'))
            
                    Data Elements:
            
                    ^DIS4 = RP:Investor:PersonNameDetails:PersonNameDetails.FamilyName.Text
                    */
                    assertion = IsMatch(investor.DIS4, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000424", Severity = ProcessMessageSeverity.Error,
                            Description = @"Family name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:PersonNameDetails/tns:PersonNameDetails.FamilyName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410200"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS4", Value = investor.DIS4 });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.410200
            
                    #region VR.ATO.GEN.410201
            
                    /*  VR.ATO.GEN.410201
                    Family name cannot contain "MR", "MRS", "MISS" or "MS"
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text = FOUND("MR","MRS","MISS","MS"))
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (FoundSet(^DIS4, '"MR","MRS","MISS","MS"'))
            
                    Data Elements:
            
                    ^DIS4 = RP:Investor:PersonNameDetails:PersonNameDetails.FamilyName.Text
                    */
                    assertion = IsMatch(investor.DIS4, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000426", Severity = ProcessMessageSeverity.Error,
                            Description = @"Family name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:PersonNameDetails/tns:PersonNameDetails.FamilyName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410201"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS4", Value = investor.DIS4 });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.410201
            
                    #region VR.ATO.GEN.410202
            
                    /*  VR.ATO.GEN.410202
                    Given name cannot contain "ESQ","II","III","IV","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:PersonNameDetails.GivenName.Text = FOUND("ESQ","II","III","IV","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"))
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (FoundSet(^DIS5, '"ESQ","II","III","IV","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
            
                    Data Elements:
            
                    ^DIS5 = RP:Investor:PersonNameDetails:PersonNameDetails.GivenName.Text
                    */
                    assertion = IsMatch(investor.DIS5, @"(^(ESQ|II|III|IV|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | II | III | IV | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |II |III |IV |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| II| III| IV| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000436", Severity = ProcessMessageSeverity.Error,
                            Description = @"First name cannot contain suffix ESQ, II, III, IV, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:PersonNameDetails/tns:PersonNameDetails.GivenName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410202"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS5", Value = investor.DIS5 });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.410202
            
                    #region VR.ATO.GEN.410203
            
                    /*  VR.ATO.GEN.410203
                    Given name cannot contain "Exec for","Rep for","Trustee for"
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:PersonNameDetails.GivenName.Text = FOUND("Exec for","Rep for","Trustee for"))
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (FoundSet(^DIS5, '"Exec for","Rep for","Trustee for"'))
            
                    Data Elements:
            
                    ^DIS5 = RP:Investor:PersonNameDetails:PersonNameDetails.GivenName.Text
                    */
                    assertion = IsMatch(investor.DIS5, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000437", Severity = ProcessMessageSeverity.Error,
                            Description = @"First name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:PersonNameDetails/tns:PersonNameDetails.GivenName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410203"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS5", Value = investor.DIS5 });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.410203
            
                    #region VR.ATO.GEN.410204
            
                    /*  VR.ATO.GEN.410204
                    Given name cannot contain "MR","MRS","MISS","MS"
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:PersonNameDetails.GivenName.Text = FOUND("MR","MRS","MISS","MS"))
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (FoundSet(^DIS5, '"MR","MRS","MISS","MS"'))
            
                    Data Elements:
            
                    ^DIS5 = RP:Investor:PersonNameDetails:PersonNameDetails.GivenName.Text
                    */
                    assertion = IsMatch(investor.DIS5, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000438", Severity = ProcessMessageSeverity.Error,
                            Description = @"First name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:PersonNameDetails/tns:PersonNameDetails.GivenName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410204"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS5", Value = investor.DIS5 });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.410204
            
                    #region VR.ATO.GEN.410205
            
                    /*  VR.ATO.GEN.410205
                    Address Line 1 must not contain 'AS ABOVE'
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:AddressDetails.Line1.Text = FOUND("AS ABOVE"))
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (FoundSet(^DIS7, '"AS ABOVE"'))
            
                    Data Elements:
            
                    ^DIS7 = RP:Investor:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = IsMatch(investor.DIS7, @"(^(AS ABOVE)$)|(^(.*)( AS ABOVE )(.*)$)|(^(AS ABOVE )(.*)$)|(^(.*)( AS ABOVE)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410001", Severity = ProcessMessageSeverity.Error,
                            Description = @"Address Line 1 must not contain 'AS ABOVE'",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(investor.Investor_AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410205"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS7", Value = investor.DIS7 });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.410205
            
                    //VR.ATO.GEN.410206 removed for CR2900 | INC000029748406
                   
                    #region VR.ATO.GEN.410207
            
                    /*  VR.ATO.GEN.410207
                    OtherGivenName cannot contain "ESQ","II","III","IV","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text = FOUND("ESQ","II","III","IV","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"))
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (FoundSet(^DIS6, '"ESQ","II","III","IV","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
            
                    Data Elements:
            
                    ^DIS6 = RP:Investor:PersonNameDetails:PersonNameDetails.OtherGivenName.Text
                    */
                    assertion = IsMatch(investor.DIS6, @"(^(ESQ|II|III|IV|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | II | III | IV | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |II |III |IV |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| II| III| IV| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000448", Severity = ProcessMessageSeverity.Error,
                            Description = @"Other given name cannot contain suffix ESQ, II, III, IV, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:PersonNameDetails/tns:PersonNameDetails.OtherGivenName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410207"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS6", Value = GetValueOrEmpty(investor.DIS6) });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.410207
            
                    #region VR.ATO.GEN.410208
            
                    /*  VR.ATO.GEN.410208
                    OtherGivenName cannot contain "Exec for","Rep for","Trustee for"
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text = FOUND("Exec for","Rep for","Trustee for"))
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (FoundSet(^DIS6, '"Exec for","Rep for","Trustee for"'))
            
                    Data Elements:
            
                    ^DIS6 = RP:Investor:PersonNameDetails:PersonNameDetails.OtherGivenName.Text
                    */
                    assertion = IsMatch(investor.DIS6, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000449", Severity = ProcessMessageSeverity.Error,
                            Description = @"Other given name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:PersonNameDetails/tns:PersonNameDetails.OtherGivenName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410208"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS6", Value = GetValueOrEmpty(investor.DIS6) });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.410208
            
                    #region VR.ATO.GEN.410209
            
                    /*  VR.ATO.GEN.410209
                    OtherGivenName cannot contain "MR","MRS","MISS","MS"
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text = FOUND("MR","MRS","MISS","MS"))
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (FoundSet(^DIS6, '"MR","MRS","MISS","MS"'))
            
                    Data Elements:
            
                    ^DIS6 = RP:Investor:PersonNameDetails:PersonNameDetails.OtherGivenName.Text
                    */
                    assertion = IsMatch(investor.DIS6, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000450", Severity = ProcessMessageSeverity.Error,
                            Description = @"Other given name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:PersonNameDetails/tns:PersonNameDetails.OtherGivenName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410209"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS6", Value = GetValueOrEmpty(investor.DIS6) });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.410209
            
                    #region VR.ATO.GEN.410211
            
                    /*  VR.ATO.GEN.410211
                    Overseas address indicator must not be 'TRUE' if Country Code is 'au'
    
                    Legacy Rule Format:
                    WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
                    IF (pyde.xx.xx:AddressDetails.Country.Code = "au") AND  (pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = TRUE)   
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^DIS40 = 'au') AND  (^DIS41 = TRUE)
            
                    Data Elements:
            
                    ^DIS40 = RP:Investor:AddressDetails:AddressDetails.Country.Code WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            
                    ^DIS41 = RP:Investor:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = (investor.DIS40 == @"au" && investor.DIS41 == true);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410211", Severity = ProcessMessageSeverity.Error,
                            Description = @"Overseas address indicator must not be 'TRUE' if Country Code is 'au'",
                            LongDescription = @"The overseas address indicator must be 'FALSE' if the Country Code is 'au' - representing Australia. If the address is an overseas address please supply the correct Country Code",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(investor.Investor_AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Country.Code",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410211"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS40", Value = GetValueOrEmpty(investor.DIS40) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS41", Value = GetValueOrEmpty(investor.DIS41) });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.410211
            
                    #region VR.ATO.GEN.410212
            
                    /*  VR.ATO.GEN.410212
                    If an overseas Country Code is used, the Overseas Address indicator must be 'TRUE'
    
                    Legacy Rule Format:
                    WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)IF (pyde.xx.xx:AddressDetails.Country.Code <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.Country.Code <> "au") AND  (pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = FALSE)
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^DIS40 <> BLANK) AND (^DIS40 <> 'au') AND  (^DIS41 = FALSE)
            
                    Data Elements:
            
                    ^DIS40 = RP:Investor:AddressDetails:AddressDetails.Country.Code WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            
                    ^DIS41 = RP:Investor:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = (string.IsNullOrWhiteSpace(investor.DIS40) != true && investor.DIS40 != @"au" && investor.DIS41 == false);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410212", Severity = ProcessMessageSeverity.Error,
                            Description = @"If an overseas Country Code is used, the Overseas Address indicator must be 'TRUE'",
                            LongDescription = @"If the address is an Australian address, the overseas address indicator must be set to 'FALSE' and you do not need to supply a Country Code. If the address is an overseas address, the overseas address indicator must be set to 'TRUE' and you must supply a Country Code other than 'au'",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(investor.Investor_AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Country.Code",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410212"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS40", Value = GetValueOrEmpty(investor.DIS40) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS41", Value = GetValueOrEmpty(investor.DIS41) });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.410212
            
                    #region VR.ATO.GEN.410213
            
                    /*  VR.ATO.GEN.410213
                    Country Name must be left blank
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:AddressDetails.CountryName.Text <> NULLORBLANK)
                        RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    ^DIS12 <> BLANK
            
                    Data Elements:
            
                    ^DIS12 = RP:Investor:AddressDetails:AddressDetails.CountryName.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = (string.IsNullOrWhiteSpace(investor.DIS12) != true);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410213", Severity = ProcessMessageSeverity.Error,
                            Description = @"Country Name must be left blank",
                            LongDescription = @"For SBR submissions to the ATO, the Country Name must be left blank. To supply the country for an overseas address please supply a Country Code",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(investor.Investor_AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.CountryName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410213"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS12", Value = GetValueOrEmpty(investor.DIS12) });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.410213
            
                    #region VR.ATO.GEN.410214
            
                    /*  VR.ATO.GEN.410214
                    Address Line 1 must be present
    
                    Legacy Rule Format:
                    WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
                    IF (pyde.xx.xx:AddressDetails.Line1.Text = NULLORBLANK)
                      RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^AddressDetails <> NULL) AND (^DIS7 = BLANK)
            
                    Data Elements:
            
                    ^DIS7 = RP:Investor:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            
                    ^AddressDetails = RP:Investor:AddressDetails
                    */
                    assertion = (investor.Investor_AddressDetailsCollection_POS_C_RPExists != false && string.IsNullOrWhiteSpace(investor.DIS7) == true);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410214", Severity = ProcessMessageSeverity.Error,
                            Description = @"Address Line 1 must be present",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(investor.Investor_AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410214"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS7", Value = investor.DIS7 });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.410214
            
                    #region VR.ATO.GEN.428040
            
                    /*  VR.ATO.GEN.428040
                    ABN is invalid
    
                    Legacy Rule Format:
                    IF (pyid.xx.xx:IdentifiersAustralianBusinessNumberIdentifier <> NULLORBLANK) and (ABNALGORITHM(pyid.xx.xx:IdentifiersAustralianBusinessNumber.Identifier) = FALSE)
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^DIS2 <> BLANK) and (FailsABNAlgorithm(^DIS2))
            
                    Data Elements:
            
                    ^DIS2 = RP:Investor:Identifiers.AustralianBusinessNumber.Identifier
                    */
                    assertion = (string.IsNullOrWhiteSpace(investor.DIS2) != true && FailsABNAlgorithm(investor.DIS2));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000477", Severity = ProcessMessageSeverity.Error,
                            Description = @"ABN is invalid",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:Identifiers.AustralianBusinessNumber.Identifier",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428040"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS2", Value = GetValueOrEmpty(investor.DIS2) });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.428040
            
                    #region VR.ATO.GEN.428230
            
                    /*  VR.ATO.GEN.428230
                    Suburb/Town/City cannot contain state code
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:AddressDetails.LocalityName.Text = FOUND("QLD","NSW","VIC","SA","WA","NT","ACT","TAS"))
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (FoundSet(^DIS9, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))
            
                    Data Elements:
            
                    ^DIS9 = RP:Investor:AddressDetails:AddressDetails.LocalityName.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = IsMatch(investor.DIS9, @"(^(QLD|NSW|VIC|SA|WA|NT|ACT|TAS)$)|(^(.*)( QLD | NSW | VIC | SA | WA | NT | ACT | TAS )(.*)$)|(^(QLD |NSW |VIC |SA |WA |NT |ACT |TAS )(.*)$)|(^(.*)( QLD| NSW| VIC| SA| WA| NT| ACT| TAS)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000411", Severity = ProcessMessageSeverity.Error,
                            Description = @"Suburb/Town/City cannot contain state code",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(investor.Investor_AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.LocalityName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428230"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS9", Value = GetValueOrEmpty(investor.DIS9) });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.428230
            
                    #region VR.ATO.GEN.428231
            
                    /*  VR.ATO.GEN.428231
                    Famaliy name cannot contain"ESQ","II","III","IV","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text = FOUND("ESQ","II","III","IV","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"))
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (FoundSet(^DIS4, '"ESQ","II","III","IV","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
            
                    Data Elements:
            
                    ^DIS4 = RP:Investor:PersonNameDetails:PersonNameDetails.FamilyName.Text
                    */
                    assertion = IsMatch(investor.DIS4, @"(^(ESQ|II|III|IV|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | II | III | IV | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |II |III |IV |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| II| III| IV| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000422", Severity = ProcessMessageSeverity.Error,
                            Description = @"Family name cannot contain suffix ESQ, II, III, IV, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:PersonNameDetails/tns:PersonNameDetails.FamilyName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428231"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS4", Value = investor.DIS4 });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.428231
            
                    #region VR.ATO.GEN.428254
            
                    /*  VR.ATO.GEN.428254
                    Address line 1 cannot contain "UNKNOWN"
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:AddressDetails.Line1.Text CONTAINS "UNKNOWN")
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    Contains(^DIS7, 'UNKNOWN')
            
                    Data Elements:
            
                    ^DIS7 = RP:Investor:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = (investor.DIS7 == null ? false : investor.DIS7.ToUpperInvariant().Contains(@"UNKNOWN"));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000405", Severity = ProcessMessageSeverity.Error,
                            Description = @"Address line 1 cannot contain ""UNKNOWN""",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(investor.Investor_AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428254"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS7", Value = investor.DIS7 });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.428254
            
                    #region VR.ATO.GEN.428255
            
                    /*  VR.ATO.GEN.428255
                    Address line 2 cannot contain "UNKNOWN"
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:AddressDetails.Line2.Text CONTAINS "UNKNOWN")
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    Contains(^DIS8, 'UNKNOWN')
            
                    Data Elements:
            
                    ^DIS8 = RP:Investor:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = (investor.DIS8 == null ? false : investor.DIS8.ToUpperInvariant().Contains(@"UNKNOWN"));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000407", Severity = ProcessMessageSeverity.Error,
                            Description = @"Address line 2 cannot contain ""UNKNOWN""",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(investor.Investor_AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line2.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428255"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS8", Value = GetValueOrEmpty(investor.DIS8) });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.428255
            
                    #region VR.ATO.GEN.428256
            
                    /*  VR.ATO.GEN.428256
                    Suburb/Town/City cannot contain "UNKNOWN"
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:AddressDetails.LocalityName.Text CONTAINS "UNKNOWN")
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    Contains(^DIS9, 'UNKNOWN')
            
                    Data Elements:
            
                    ^DIS9 = RP:Investor:AddressDetails:AddressDetails.LocalityName.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = (investor.DIS9 == null ? false : investor.DIS9.ToUpperInvariant().Contains(@"UNKNOWN"));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000409", Severity = ProcessMessageSeverity.Error,
                            Description = @"Suburb/Town/City cannot contain ""UNKNOWN""",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(investor.Investor_AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.LocalityName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428256"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS9", Value = GetValueOrEmpty(investor.DIS9) });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.428256
            
                    #region VR.ATO.GEN.428258
            
                    /*  VR.ATO.GEN.428258
                    Non Individual Name cannot contain space hyphen space combinations
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text CONTAINS  " - " )
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    Contains(^DIS3, ' - ')
            
                    Data Elements:
            
                    ^DIS3 = RP:Investor:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN']
                    */
                    assertion = (investor.DIS3 == null ? false : investor.DIS3.ToUpperInvariant().Contains(@" - "));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000413", Severity = ProcessMessageSeverity.Error,
                            Description = @"Non Individual Name cannot contain space hyphen space combinations",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:OrganisationNameDetails" + OccurrenceIndex(investor.Investor_OrganisationNameDetailsCollection_MN_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428258"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS3", Value = investor.DIS3 });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.428258
            
                    #region VR.ATO.GEN.428259
            
                    /*  VR.ATO.GEN.428259
                    Non Individual Name cannot contain "P/L"
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text CONTAINS "P/L")
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    Contains(^DIS3, 'P/L')
            
                    Data Elements:
            
                    ^DIS3 = RP:Investor:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN']
                    */
                    assertion = (investor.DIS3 == null ? false : investor.DIS3.ToUpperInvariant().Contains(@"P/L"));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000414", Severity = ProcessMessageSeverity.Error,
                            Description = @"Non Individual Name cannot contain ""P/L""",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:OrganisationNameDetails" + OccurrenceIndex(investor.Investor_OrganisationNameDetailsCollection_MN_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428259"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS3", Value = investor.DIS3 });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.428259
            
                    #region VR.ATO.GEN.428260
            
                    /*  VR.ATO.GEN.428260
                    Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text CONTAINS SET("--","''","  ") )
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (ContainsSet(^DIS3, '"--","''","  "'))
            
                    Data Elements:
            
                    ^DIS3 = RP:Investor:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN']
                    */
                    assertion = IsMatch(investor.DIS3, @"--|''|  ",RegexOptions.IgnoreCase);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000416", Severity = ProcessMessageSeverity.Error,
                            Description = @"Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:OrganisationNameDetails" + OccurrenceIndex(investor.Investor_OrganisationNameDetailsCollection_MN_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428260"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS3", Value = investor.DIS3 });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.428260
            
                    #region VR.ATO.GEN.430245
            
                    /*  VR.ATO.GEN.430245
                    Suburb/Town/City must be present in an address
    
                    Legacy Rule Format:
                    WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
                    IF (pyde.xx.xx:AddressDetails.LocalityName.Text = NULLORBLANK)
                      RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^AddressDetails <> NULL) AND (^DIS9 = BLANK)
            
                    Data Elements:
            
                    ^DIS9 = RP:Investor:AddressDetails:AddressDetails.LocalityName.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            
                    ^AddressDetails = RP:Investor:AddressDetails
                    */
                    assertion = (investor.Investor_AddressDetailsCollection_POS_C_RPExists != false && string.IsNullOrWhiteSpace(investor.DIS9) == true);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.430245", Severity = ProcessMessageSeverity.Error,
                            Description = @"Suburb/Town/City must be present in an address",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(investor.Investor_AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.LocalityName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.430245"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS9", Value = GetValueOrEmpty(investor.DIS9) });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.430245
            
                    #region VR.ATO.GEN.430246
            
                    /*  VR.ATO.GEN.430246
                    Suburb/Town/City cannot contain numeric characters
    
                    Legacy Rule Format:
                    WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
                    IF (pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = FALSE) AND (pyde.xx.xx:AddressDetails.LocalityName.Text CONTAINS NUMERICS SET(0-9))
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^DIS41 = FALSE) AND (ContainsSet(^DIS9, '0-9'))
            
                    Data Elements:
            
                    ^DIS9 = RP:Investor:AddressDetails:AddressDetails.LocalityName.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            
                    ^DIS41 = RP:Investor:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = (investor.DIS41 == false && IsMatch(investor.DIS9, @"[0-9]",RegexOptions.IgnoreCase));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.000410", Severity = ProcessMessageSeverity.Error,
                            Description = @"Suburb/Town/City cannot contain numeric characters",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(investor.Investor_AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.LocalityName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.430246"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS41", Value = GetValueOrEmpty(investor.DIS41) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS9", Value = GetValueOrEmpty(investor.DIS9) });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.430246
            
                    #region VR.ATO.GEN.434147
            
                    /*  VR.ATO.GEN.434147
                    State and post code must be present for an Australian address
    
                    Legacy Rule Format:
                    WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
                    IF pyde.xx.xx:AddressDetails.OverseasAddressIndicator = FALSE AND (pyde.xx.xx:AddressDetails.StateOrTerritory.Code = NULLORBLANK OR pyde.xx.xx:AddressDetails.Postcode.Text = NULLORBLANK)
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^AddressDetails <> NULL) AND (^DIS41 = FALSE AND (^DIS10 = BLANK OR ^DIS11 = BLANK))
            
                    Data Elements:
            
                    ^DIS41 = RP:Investor:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            
                    ^DIS10 = RP:Investor:AddressDetails:AddressDetails.StateOrTerritory.Code WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            
                    ^DIS11 = RP:Investor:AddressDetails:AddressDetails.Postcode.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            
                    ^AddressDetails = RP:Investor:AddressDetails
                    */
                    assertion = (investor.Investor_AddressDetailsCollection_POS_C_RPExists != false && (investor.DIS41 == false && (string.IsNullOrWhiteSpace(investor.DIS10) == true || string.IsNullOrWhiteSpace(investor.DIS11) == true)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.434147", Severity = ProcessMessageSeverity.Error,
                            Description = @"State and post code must be present for an Australian address",
                            LongDescription = @"If the overseas indicator is no (false), then an Australian state code and post code must be present",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(investor.Investor_AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.OverseasAddress.Indicator",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.434147"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS41", Value = GetValueOrEmpty(investor.DIS41) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS10", Value = GetValueOrEmpty(investor.DIS10) });
            
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS11", Value = GetValueOrEmpty(investor.DIS11) });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.434147
            
                    #region VR.ATO.GEN.436003
            
                    /*  VR.ATO.GEN.436003
                    Organisation name must be present
    
                    Legacy Rule Format:
                    WHERE IN TUPLE(organisationname1.xx.xx:OrganisationNameDetails)
                    IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text = NULLORBLANK)
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^OrganisationNameDetails <> NULL) AND (^DIS3 = BLANK)
            
                    Data Elements:
            
                    ^DIS3 = RP:Investor:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN']
            
                    ^OrganisationNameDetails = RP:Investor:OrganisationNameDetails
                    */
                    assertion = (investor.Investor_OrganisationNameDetailsCollection_MN_RPExists != false && string.IsNullOrWhiteSpace(investor.DIS3) == true);
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.410215", Severity = ProcessMessageSeverity.Error,
                            Description = @"Organisation name must be present",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:OrganisationNameDetails" + OccurrenceIndex(investor.Investor_OrganisationNameDetailsCollection_MN_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.436003"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS3", Value = investor.DIS3 });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.436003
            
                    #region VR.ATO.GEN.500141
            
                    /*  VR.ATO.GEN.500141
                    For an Australian address, state or territory must be one of: ACT, NSW, NT, QLD, SA, VIC, WA, TAS
    
                    Legacy Rule Format:
                    IF (pyde.xx.xx:AddressDetails.StateOrTerritory.Code <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.StateOrTerritory.Code <> SET ("ACT","NSW","NT","QLD","SA","VIC","WA","TAS"))
                       RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    (^DIS10 <> BLANK) AND (NotInSet(^DIS10, '"ACT","NSW","NT","QLD","SA","VIC","WA","TAS"'))
            
                    Data Elements:
            
                    ^DIS10 = RP:Investor:AddressDetails:AddressDetails.StateOrTerritory.Code WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
                    */
                    assertion = (string.IsNullOrWhiteSpace(investor.DIS10) != true && !(IsMatch(investor.DIS10, @"^(ACT|NSW|NT|QLD|SA|VIC|WA|TAS)$",RegexOptions.IgnoreCase)));
                    if (assertion)
                    {
                        processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.GEN.500141", Severity = ProcessMessageSeverity.Error,
                            Description = @"For an Australian address, state or territory must be one of: ACT, NSW, NT, QLD, SA, VIC, WA, TAS",
                            Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(investor.OccurrenceIndex) + "/tns:AddressDetails" + OccurrenceIndex(investor.Investor_AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.StateOrTerritory.Code",
                            Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500141"} },
                        };
                        processMessage.Parameters.Add(new ProcessMessageParameter
                            { Name = "DIS10", Value = GetValueOrEmpty(investor.DIS10) });
            
                        response.Add(processMessage);
                    }
                    #endregion // VR.ATO.GEN.500141
                    }
                }
        
                #endregion // Foreach loop
    
            #region VR.ATO.DIS.406005
    
            /*  VR.ATO.DIS.406005
            Company name must be provided
    
            Legacy Rule Format:
            IF ([DIS24] = NULLORBLANK)
            RETURN VALIDATION MESSAGE
            ENDIF
            
            
            
            [DIS24] = DIS:RP:pyde.02.00:OrganisationNameDetails.OrganisationalName.Text WHERE (pyde.02.00:OrganisationNameDetails.OrganisationalNameType.Code = "MN") IN TUPLE(orgname1.02.00:OrganisationNameDetails)

            Technical Business Rule Format:
            (^OrganisationNameDetails <> NULL) AND (^DIS24 = BLANK)
    
            Data Elements:
    
            ^DIS24 = RP:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN']
    
            ^OrganisationNameDetails = RP:OrganisationNameDetails
            */
            assertion = (report.OrganisationNameDetailsCollection_MN_RPExists != false && string.IsNullOrWhiteSpace(report.DIS24) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.DIS.406005", Severity = ProcessMessageSeverity.Error,
                    Description = @"Company Name not provided",
                    LongDescription = @"Company name must be provided",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_MN_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DIS.406005"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS24", Value = report.DIS24 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.DIS.406005
    
            #region VR.ATO.DIS.406021
    
            /*  VR.ATO.DIS.406021
            Standard name format rules apply
    
            Legacy Rule Format:
            IF ([DIS25] <> NULL) AND ([DIS25] > TEXT (200))
            RETURN VALIDATION MESSAGE
            ENDIF
            
            
            
            
            [DIS25] = DIS:RP:pyde.02.00:PersonUnstructuredName.FullName.Text WHERE (pyde.02.37:PersonUnstructuredName.Usage.Code = "Contact") IN TUPLE(prsnunstrcnm1.02.01:PersonUnstructuredName)

            Technical Business Rule Format:
            Length(^DIS25) > 200
    
            Data Elements:
    
            ^DIS25 = RP:PersonUnstructuredName:PersonUnstructuredName.FullName.Text WHERE [PersonUnstructuredName.Usage.Code='Contact']
            */
            assertion = (Length(report.DIS25) > 200);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:PersonUnstructuredName" + OccurrenceIndex(report.PersonUnstructuredNameCollection_Contact_RPOccurrenceIndex) + "/tns:PersonUnstructuredName.FullName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DIS.406021"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS25", Value = report.DIS25 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.DIS.406021
    
            #region VR.ATO.DIS.406024
    
            /*  VR.ATO.DIS.406024
            The number of Reporting party: Conduit Foreign Income must not exceed 1
    
            Legacy Rule Format:
            IF (COUNT(RP.CFI) > 1)
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (Count(^Context) > 1)
    
            Data Elements:
    
            ^Context = RP:RP.CFI
            */
            assertion = (Count(report.RPCFICount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.DIS.406024", Severity = ProcessMessageSeverity.Error,
                    Description = @"The number of Reporting party: Conduit Foreign Income must not exceed 1",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPCFILastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DIS.406024"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.DIS.406024
    
            #region VR.ATO.DIS.406025
    
            /*  VR.ATO.DIS.406025
            The number of Reporting party: Non-Conduit Foreign Income must not exceed 1
    
            Legacy Rule Format:
            IF (COUNT(RP.NONCFI) > 1)
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (Count(^Context) > 1)
    
            Data Elements:
    
            ^Context = RP:RP.NONCFI
            */
            assertion = (Count(report.RPNONCFICount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.DIS.406025", Severity = ProcessMessageSeverity.Error,
                    Description = @"The number of Reporting party: Non-Conduit Foreign Income must not exceed 1",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPNONCFILastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DIS.406025"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.DIS.406025
    
            #region VR.ATO.DIS.406026
    
            /*  VR.ATO.DIS.406026
            Year of return must be between 2015 and current lodgement year when dividend and interest schedule is submitted as a stand alone form.
    
            Legacy Rule Format:
            WHERE PARENT RETURN DOES NOT EXIST
            IF ([DIS42] < 2015) OR ([DIS42] > CURRENT FINANCIAL YEAR)
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [DIS42] = DIS:RP:pyin.02.37:Report.TargetFinancial.Year

            Technical Business Rule Format:
            (^DIS42 < 2015) OR (^DIS42 > CurrentFinancialYear())
    
            Data Elements:
    
            ^DIS42 = RP:Report.TargetFinancial.Year
            */
            if (isStandalone)
            {
                assertion = (report.DIS42.GetValueOrDefault() < 2015 || report.DIS42.GetValueOrDefault() > CurrentFinancialYear());
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.DIS.406026", Severity = ProcessMessageSeverity.Error,
                        Description = @"Year of return must be between 2015 and current lodgement year when dividend and interest schedule is submitted as a stand alone form.",
                        LongDescription = @"Year of return must be between 2015 and current lodgement year when dividend and interest schedule is submitted as a stand alone form.",
                        Location = "/xbrli:xbrl/tns:Report.TargetFinancial.Year",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DIS.406026"} },
                    };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "DIS42", Value = GetValueOrEmpty(report.DIS42) });
        
                    response.Add(processMessage);
                }
                }
            #endregion // VR.ATO.DIS.406026
    
            #region VR.ATO.DIS.406027
    
            /*  VR.ATO.DIS.406027
            Year of Return must not be supplied when submitted with a parent return
    
            Legacy Rule Format:
            WHERE PARENT RETURN EXISTS
            IF [DIS42] <> NULLORBLANK
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [DIS42] = DIS:RP:pyin.02.37:Report.TargetFinancial.Year

            Technical Business Rule Format:
            ^DIS42 <> BLANK
    
            Data Elements:
    
            ^DIS42 = RP:Report.TargetFinancial.Year
            */
            if (!isStandalone)
            {
                assertion = (report.DIS42 != null);
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.DIS.406027", Severity = ProcessMessageSeverity.Error,
                        Description = @"Year of Return must not be supplied when submitted with a parent return",
                        LongDescription = @"Year of Return must not be supplied when submitted with a parent return",
                        Location = "/xbrli:xbrl/tns:Report.TargetFinancial.Year",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DIS.406027"} },
                    };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "DIS42", Value = GetValueOrEmpty(report.DIS42) });
        
                    response.Add(processMessage);
                }
                }
            #endregion // VR.ATO.DIS.406027
    
            #region VR.ATO.DIS.406029
    
            /*  VR.ATO.DIS.406029
            Company Tax File Number must pass the TFN algorithm check.
    
            Legacy Rule Format:
            WHERE PARENT RETURN DOES NOT EXIST
            IF (RP:entity.identifier.TFN <> NULLORBLANK) AND (TFNALGORITHM (RP:entity.identifier.TFN) = FALSE)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^TFN <> BLANK) AND (FailsTFNAlgorithm(^TFN))
    
            Data Elements:
    
            ^TFN = RP
            */
            if (isStandalone)
            {
                assertion = (string.IsNullOrWhiteSpace(report.RPIdentifierTFN) != true && FailsTFNAlgorithm(report.RPIdentifierTFN));
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.DIS.406029", Severity = ProcessMessageSeverity.Error,
                        Description = @"Company Tax File Number has failed the algorithm check",
                        Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DIS.406029"} },
                    };
                    response.Add(processMessage);
                }
                }
            #endregion // VR.ATO.DIS.406029
    
            #region VR.ATO.DIS.406030
    
            /*  VR.ATO.DIS.406030
            Overseas address indicator can only be false
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.OverseasAddress.Indicator <> NULL) AND (pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = TRUE)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^DIS45 <> NULL) AND (^DIS45 = TRUE)
    
            Data Elements:
    
            ^DIS45 = RP:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (report.DIS45 != null && report.DIS45 == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.DIS.406030", Severity = ProcessMessageSeverity.Error,
                    Description = @"Overseas address indicator must be false.",
                    LongDescription = @"Current address must be an Australian address. The overseas indicator must therefore be false.",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.OverseasAddress.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DIS.406030"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS45", Value = GetValueOrEmpty(report.DIS45) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.DIS.406030
    
            #region VR.ATO.DIS.406030
    
            /*  VR.ATO.DIS.406030
            Overseas address indicator can only be false
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.OverseasAddress.Indicator <> NULL) AND (pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = TRUE)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^DIS60 <> NULL) AND (^DIS60 = TRUE)
    
            Data Elements:
    
            ^DIS60 = RP:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (report.DIS60 != null && report.DIS60 == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.DIS.406030", Severity = ProcessMessageSeverity.Error,
                    Description = @"Overseas address indicator must be false.",
                    LongDescription = @"Current address must be an Australian address. The overseas indicator must therefore be false.",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.OverseasAddress.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DIS.406030"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS60", Value = GetValueOrEmpty(report.DIS60) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.DIS.406030
    
            #region VR.ATO.DIS.406031
    
            /*  VR.ATO.DIS.406031
            Current business address - Line 1 is required, it cannot be blank
    
            Legacy Rule Format:
            WHERE PARENT RETURN DOES NOT EXIST
            IF COUNT(((pyde.xx.xx:AddressDetails.Usage.Code = "BUS") AND (pyde.xx.xx:AddressDetails.Currency.Code = "C") AND (pyde.xx.xx:AddressDetails.Line1.Text <> NULLORBLANK)) IN TUPLE(address2.xx.xx:AddressDetails)) <> 1
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Not(Count(^AddressDetails) = 1 AND ^DIS48 <> BLANK)
    
            Data Elements:
    
            ^DIS48 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
    
            ^AddressDetails = RP:AddressDetails
            */
            if (isStandalone)
            {
                assertion =  !((Count(report.AddressDetailsCollection_BUS_C_RPCount) == 1 && string.IsNullOrWhiteSpace(report.DIS48) != true) );
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.DIS.406031", Severity = ProcessMessageSeverity.Error,
                        Description = @"Current business address must be supplied, it cannot be blank.",
                        Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPLastOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DIS.406031"} },
                    };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "DIS48", Value = report.DIS48 });
        
                    response.Add(processMessage);
                }
                }
            #endregion // VR.ATO.DIS.406031
    
            #region VR.ATO.DIS.406033
    
            /*  VR.ATO.DIS.406033
            Current postal address - Line 1 is required, it cannot be blank.
    
            Legacy Rule Format:
            WHERE PARENT RETURN DOES NOT EXIST
            IF COUNT(((pyde.xx.xx:AddressDetails.Usage.Code = "POS") AND (pyde.xx.xx:AddressDetails.Currency.Code = "C") AND (pyde.xx.xx:AddressDetails.Line1.Text <> NULLORBLANK)) IN TUPLE(address2.xx.xx:AddressDetails)) <> 1
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^AddressDetails <> NULL) AND (^DIS64 = BLANK)
    
            Data Elements:
    
            ^DIS64 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
    
            ^AddressDetails = RP:AddressDetails
            */
            if (isStandalone)
            {
                assertion = (report.AddressDetailsCollection_POS_C_RPExists != false && string.IsNullOrWhiteSpace(report.DIS64) == true);
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.DIS.406033", Severity = ProcessMessageSeverity.Error,
                        Description = @"Current postal address - Line 1 is required, it cannot be blank.",
                        Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DIS.406033"} },
                    };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "DIS64", Value = report.DIS64 });
        
                    response.Add(processMessage);
                }
                }
            #endregion // VR.ATO.DIS.406033
    
            #region VR.ATO.DIS.406034
    
            /*  VR.ATO.DIS.406034
            If an Intermediary context exists, then a Tax Agent or Intermediary contact telephone must be present
    
            Legacy Rule Format:
            WHERE PARENT RETURN DOES NOT EXIST
            IF CONTEXT(INT) <> NULLORBLANK AND [DIS95] = NULL
               RETURN VALIDATION MESSAGE
            ENDIF
            
            
            
            [DIS95] = DIS:INT:pyde.02.00:ElectronicContact.Telephone.Minimal.Number WHERE (pyde.02.00:ElectronicContact.Telephone.Usage.Code = "03") IN TUPLE(phone1.02.00:ElectronicContactTelephone)

            Technical Business Rule Format:
            ^Context <> BLANK AND ^DIS95 = NULL
    
            Data Elements:
    
            ^DIS95 = INT:ElectronicContactTelephone:ElectronicContact.Telephone.Minimal.Number WHERE [ElectronicContact.Telephone.Usage.Code='03']
    
            ^Context = INT
            */
            if (isStandalone)
            {
                assertion = (report.INTExists != false && report.DIS95 == null);
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.DIS.406034", Severity = ProcessMessageSeverity.Error,
                        Description = @"Tax Agent (Intermediary) contact telephone must be present",
                        LongDescription = @"If an Intermediary context exists, then a Tax Agent (Intermediary) contact telephone must be present",
                        Location = "/xbrli:xbrl/tns:ElectronicContactTelephone" + OccurrenceIndex(report.ElectronicContactTelephoneCollection_03_INTOccurrenceIndex) + "/tns:ElectronicContact.Telephone.Minimal.Number",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DIS.406034"} },
                    };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "DIS95", Value = report.DIS95 });
        
                    response.Add(processMessage);
                }
                }
            #endregion // VR.ATO.DIS.406034
    
            #region VR.ATO.DIS.406037
    
            /*  VR.ATO.DIS.406037
            The number of Intermediary context must not exceed 1
    
            Legacy Rule Format:
            WHERE PARENT RETURN DOES NOT EXIST
            IF (COUNT(INT) > 1)
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (Count(^Context) > 1)
    
            Data Elements:
    
            ^Context = INT
            */
            if (isStandalone)
            {
                assertion = (Count(report.INTCount) > 1);
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.DIS.406037", Severity = ProcessMessageSeverity.Error,
                        Description = @"Invalid context. The number of Intermediary context must not exceed 1",
                        Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.INTLastOccurrenceIndex),
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DIS.406037"} },
                    };
                    response.Add(processMessage);
                }
                }
            #endregion // VR.ATO.DIS.406037
    
            #region VR.ATO.DIS.406039
    
            /*  VR.ATO.DIS.406039
            Intermediary is not required when Dividend and Interest Schedule is submitted with a parent return.
    
            Legacy Rule Format:
            WHERE PARENT RETURN EXISTS
            IF Context(INT) <> NULLORBLANK 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> BLANK
    
            Data Elements:
    
            ^Context = INT
            */
            if (!isStandalone)
            {
                assertion = (report.INTExists != false);
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.DIS.406039", Severity = ProcessMessageSeverity.Error,
                        Description = @"Intermediary is not required when Dividend and Interest Schedule is submitted with a parent return.",
                        Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.INTOccurrenceIndex),
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DIS.406039"} },
                    };
                    response.Add(processMessage);
                }
                }
            #endregion // VR.ATO.DIS.406039
    
            #region VR.ATO.DIS.406040
    
            /*  VR.ATO.DIS.406040
            Address must not be supplied when submitted with the parent return
    
            Legacy Rule Format:
            WHERE PARENT RETURN EXISTS
            IF ([DIS48] <> NULL OR [DIS64] <> NULL)
                   RETURN VALIDATION MESSAGE
            ENDIF
            
            
            
            
            [DIS48] = DIS:RP:pyde.02.00:AddressDetails.Line1.Text WHERE (pyde.02.01:AddressDetails.Usage.Code = "BUS" AND pyde.02.00:AddressDetails.Currency.Code = "C") IN TUPLE(address2.02.02:AddressDetails)
            [DIS64] = DIS:RP:pyde.02.00:AddressDetails.Line1.Text WHERE (pyde.02.01:AddressDetails.Usage.Code = "POS" AND pyde.02.00:AddressDetails.Currency.Code = "C") IN TUPLE(address2.02.02:AddressDetails)

            Technical Business Rule Format:
            ^DIS48 <> NULL OR ^DIS64 <> NULL
    
            Data Elements:
    
            ^DIS48 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
    
            ^DIS64 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            if (!isStandalone)
            {
                assertion = (report.DIS48 != null || report.DIS64 != null);
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.DIS.406040", Severity = ProcessMessageSeverity.Error,
                        Description = @"Address must not be supplied when submitted with the parent return.",
                        Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DIS.406040"} },
                    };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "DIS48", Value = report.DIS48 });
        
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "DIS64", Value = report.DIS64 });
        
                    response.Add(processMessage);
                }
                }
            #endregion // VR.ATO.DIS.406040
    
            #region VR.ATO.DIS.406042
    
            /*  VR.ATO.DIS.406042
            If a Parent Return  does not exist and an Intermediary is present, then the Intermediary contact name must be present.
    
            Legacy Rule Format:
            WHERE PARENT RETURN DOES NOT EXIST
            IF INT <> NULLORBLANK
            AND [DIS89] = NULL
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [DIS89] = DIS:INT:pyde.02.00:PersonUnstructuredName.FullName.Text WHERE (pyde.02.37:PersonUnstructuredName.Usage.Code = "Contact") IN TUPLE(prsnunstrcnm1.02.01:PersonUnstructuredName)

            Technical Business Rule Format:
            ^INT <> BLANK AND ^DIS89 = NULL
    
            Data Elements:
    
            ^DIS89 = INT:PersonUnstructuredName:PersonUnstructuredName.FullName.Text WHERE [PersonUnstructuredName.Usage.Code='Contact']
    
            ^INT = INT
            */
            if (isStandalone)
            {
                assertion = (report.INTExists != false && report.DIS89 == null);
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.DIS.406042", Severity = ProcessMessageSeverity.Error,
                        Description = @"If a Parent Return  does not exist and an Intermediary is present, then the Intermediary contact name must be present.",
                        Location = "/xbrli:xbrl/tns:PersonUnstructuredName" + OccurrenceIndex(report.PersonUnstructuredNameCollection_Contact_INTOccurrenceIndex) + "/tns:PersonUnstructuredName.FullName.Text",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DIS.406042"} },
                    };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "DIS89", Value = report.DIS89 });
        
                    response.Add(processMessage);
                }
                }
            #endregion // VR.ATO.DIS.406042
    
            #region VR.ATO.DIS.406043
    
            /*  VR.ATO.DIS.406043
            If a Parent Return does not exist and an Intermediary is present, then the Intermediary registered name of company must be present.
    
            Legacy Rule Format:
            WHERE PARENT RETURN DOES NOT EXIST
            IF INT <> NULLORBLANK
            AND [DIS91] = NULL
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [DIS91] = DIS:INT:pyde.02.00:OrganisationNameDetails.OrganisationalName.Text WHERE (pyde.02.00:OrganisationNameDetails.OrganisationalNameType.Code = "MN") IN TUPLE(orgname1.02.00:OrganisationNameDetails)

            Technical Business Rule Format:
            ^INT <> BLANK AND ^DIS91 = NULL
    
            Data Elements:
    
            ^DIS91 = INT:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN']
    
            ^INT = INT
            */
            if (isStandalone)
            {
                assertion = (report.INTExists != false && report.DIS91 == null);
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.DIS.406043", Severity = ProcessMessageSeverity.Error,
                        Description = @"If a Parent Return does not exist and an Intermediary is present, then the Intermediary registered name of company must be present.",
                        Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_MN_INTOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DIS.406043"} },
                    };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "DIS91", Value = report.DIS91 });
        
                    response.Add(processMessage);
                }
                }
            #endregion // VR.ATO.DIS.406043
    
            #region VR.ATO.DIS.406044
    
            /*  VR.ATO.DIS.406044
            Declaration by the person submitting the return must be present when it does not have a parent return.
    
            Legacy Rule Format:
            WHERE PARENT RETURN DOES NOT EXIST
            IF (INT = NULLORBLANK AND RP:pyin.xx.xx:Declaration.StatementAccepted.Indicator = NULL)  OR (INT <> NULLORBLANK AND INT:pyin.xx.xx:Declaration.StatementAccepted.Indicator = NULL)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^INT = BLANK AND ^DIS75 = NULL)  OR (^INT <> BLANK AND ^DIS82 = NULL)
    
            Data Elements:
    
            ^DIS82 = INT:Declaration:Declaration.StatementAccepted.Indicator WHERE [Declaration.StatementType.Code='TrueAndCorrect']
    
            ^DIS75 = RP:Declaration:Declaration.StatementAccepted.Indicator WHERE [Declaration.StatementType.Code='TrueAndCorrect']
    
            ^INT = INT
            */
            if (isStandalone)
            {
                assertion = (report.INTExists == false && report.DIS75 == null || report.INTExists != false && report.DIS82 == null);
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.DIS.406044", Severity = ProcessMessageSeverity.Error,
                        Description = @"Declaration by the person submitting the return must be present when it does not have a parent return.",
                        LongDescription = @"If an Intermediary is present, there must be a Declaration by the Intermediary. If an Intermediary is not present, there must be a Declaration by the Reporting Party",
                        Location = "/xbrli:xbrl/tns:Declaration" + OccurrenceIndex(report.DeclarationCollection_TrueAndCorrect_INTOccurrenceIndex) + "/tns:Declaration.StatementAccepted.Indicator",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DIS.406044"} },
                    };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "DIS75", Value = GetValueOrEmpty(report.DIS75) });
        
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "DIS82", Value = GetValueOrEmpty(report.DIS82) });
        
                    response.Add(processMessage);
                }
                }
            #endregion // VR.ATO.DIS.406044
    
            #region VR.ATO.DIS.406045
    
            /*  VR.ATO.DIS.406045
            Reporting party's contact name must be present if parent return and intermediary does not exist.
    
            Legacy Rule Format:
            WHERE PARENT RETURN DOES NOT EXIST
            IF Context(INT) = NULL AND [DIS25] = NULL
            RETURN VALIDATION MESSAGE
            ENDIF
            
            
            [DIS25] = DIS:RP:pyde.02.00:PersonUnstructuredName.FullName.Text WHERE (pyde.02.37:PersonUnstructuredName.Usage.Code = "Contact") IN TUPLE(prsnunstrcnm1.02.01:PersonUnstructuredName)

            Technical Business Rule Format:
            ^Context = NULL AND ^DIS25 = NULL
    
            Data Elements:
    
            ^DIS25 = RP:PersonUnstructuredName:PersonUnstructuredName.FullName.Text WHERE [PersonUnstructuredName.Usage.Code='Contact']
    
            ^Context = INT
            */
            if (isStandalone)
            {
                assertion = (report.INTExists == false && report.DIS25 == null);
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.DIS.406045", Severity = ProcessMessageSeverity.Error,
                        Description = @"Reporting party's contact name must be present if parent return and intermediary does not exist.",
                        Location = "/xbrli:xbrl/tns:PersonUnstructuredName" + OccurrenceIndex(report.PersonUnstructuredNameCollection_Contact_RPOccurrenceIndex) + "/tns:PersonUnstructuredName.FullName.Text",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DIS.406045"} },
                    };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "DIS25", Value = report.DIS25 });
        
                    response.Add(processMessage);
                }
                }
            #endregion // VR.ATO.DIS.406045
    
            #region VR.ATO.DIS.406046
    
            /*  VR.ATO.DIS.406046
            Reporting party's telephone number must be present if parent return and intermediary does not exist.
    
            Legacy Rule Format:
            WHERE PARENT RETURN DOES NOT EXIST
            IF Context(INT) = NULL AND [DIS27] = NULL
            RETURN VALIDATION MESSAGE
            ENDIF
            
            
            [DIS27] = DIS:RP:pyde.02.00:ElectronicContact.Telephone.Minimal.Number WHERE (pyde.02.00:ElectronicContact.Telephone.Usage.Code = "03") IN TUPLE(phone1.02.00:ElectronicContactTelephone)

            Technical Business Rule Format:
            ^Context = NULL AND ^DIS27 = NULL
    
            Data Elements:
    
            ^DIS27 = RP:ElectronicContactTelephone:ElectronicContact.Telephone.Minimal.Number WHERE [ElectronicContact.Telephone.Usage.Code='03']
    
            ^Context = INT
            */
            if (isStandalone)
            {
                assertion = (report.INTExists == false && report.DIS27 == null);
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.DIS.406046", Severity = ProcessMessageSeverity.Error,
                        Description = @"Reporting party's contact name must be present if parent return and intermediary does not exist.",
                        Location = "/xbrli:xbrl/tns:ElectronicContactTelephone" + OccurrenceIndex(report.ElectronicContactTelephoneCollection_03_RPOccurrenceIndex) + "/tns:ElectronicContact.Telephone.Minimal.Number",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DIS.406046"} },
                    };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                        { Name = "DIS27", Value = report.DIS27 });
        
                    response.Add(processMessage);
                }
                }
            #endregion // VR.ATO.DIS.406046
    
            #region VR.ATO.DIS.406049
    
            /*  VR.ATO.DIS.406049
            Submission must contain at least one investor.
    
            Legacy Rule Format:
            IF COUNT(TUPLE(dis.0003.lodge.req.xx.xx:Investor)) < 1 
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Investor) < 1
    
            Data Elements:
    
            ^Investor = RP:Investor
            */
            assertion = (Count(report.InvestorCollectionCount) < 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.DIS.438030", Severity = ProcessMessageSeverity.Error,
                    Description = @"Submission must contain at least one investor.",
                    Location = "/xbrli:xbrl/tns:Investor" + OccurrenceIndex(report.InvestorCollection == null ? 0 : (report.InvestorCollection.Count() == 0 ? 0 : report.InvestorCollection.Last().OccurrenceIndex)) + "/tns:Identifiers.TaxFileNumber.Identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.DIS.406049"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.DIS.406049
    
            #region VR.ATO.GEN.000209
    
            /*  VR.ATO.GEN.000209
            Invalid context. The number of Reporting party contexts must equal 1
    
            Legacy Rule Format:
            IF COUNT(RP) <> 1
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) <> 1
    
            Data Elements:
    
            ^Context = RP
            */
            assertion = (Count(report.RPCount) != 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430296", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Reporting party contexts must equal 1",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000209"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.000209
    
            #region VR.ATO.GEN.000406
    
            /*  VR.ATO.GEN.000406
            Address line 1 contains incorrect 'care of' reference.  The only acceptable 'care of' reference is C/-
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line1.Text <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.Line1.Text STARTSWITH SET("C/O ","C/ ","Care Of ","CO ")) OR (pyde.xx.xx:AddressDetails.Line1.Text CONTAINS SET(" C/O "," C/ "," Care Of "))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^DIS48 <> BLANK) AND (StartsWithSet(^DIS48,'"C/O ","C/ ","Care Of ","CO "')) OR (ContainsSet(^DIS48, '" C/O "," C/ "," Care Of "'))
    
            Data Elements:
    
            ^DIS48 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.DIS48) != true && IsMatch(report.DIS48, @"^(C/O |C/ |Care Of |CO )",RegexOptions.IgnoreCase) || IsMatch(report.DIS48, @" C/O | C/ | Care Of ",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000406", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 contains incorrect 'care of' reference.  The only acceptable 'care of' reference is C/-",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000406"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS48", Value = report.DIS48 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.000406
    
            #region VR.ATO.GEN.000406
    
            /*  VR.ATO.GEN.000406
            Address line 1 contains incorrect 'care of' reference.  The only acceptable 'care of' reference is C/-
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line1.Text <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.Line1.Text STARTSWITH SET("C/O ","C/ ","Care Of ","CO ")) OR (pyde.xx.xx:AddressDetails.Line1.Text CONTAINS SET(" C/O "," C/ "," Care Of "))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^DIS64 <> BLANK) AND (StartsWithSet(^DIS64,'"C/O ","C/ ","Care Of ","CO "')) OR (ContainsSet(^DIS64, '" C/O "," C/ "," Care Of "'))
    
            Data Elements:
    
            ^DIS64 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.DIS64) != true && IsMatch(report.DIS64, @"^(C/O |C/ |Care Of |CO )",RegexOptions.IgnoreCase) || IsMatch(report.DIS64, @" C/O | C/ | Care Of ",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000406", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 contains incorrect 'care of' reference.  The only acceptable 'care of' reference is C/-",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000406"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS64", Value = report.DIS64 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.000406
    
            #region VR.ATO.GEN.000408
    
            /*  VR.ATO.GEN.000408
            Care of' is not valid for address line 2
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line2.Text <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.Line2.Text STARTSWITH SET("C/- ","C/O ","C/ ","Care Of ","CO ") OR pyde.xx.xx:AddressDetails.Line2.Text CONTAINS SET(" C/- "," C/O "," C/ "," Care Of "))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^DIS49 <> BLANK) AND (StartsWithSet(^DIS49,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^DIS49, '" C/- "," C/O "," C/ "," Care Of "'))
    
            Data Elements:
    
            ^DIS49 = RP:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.DIS49) != true && (IsMatch(report.DIS49, @"^(C/- |C/O |C/ |Care Of |CO )",RegexOptions.IgnoreCase) || IsMatch(report.DIS49, @" C/- | C/O | C/ | Care Of ",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000408", Severity = ProcessMessageSeverity.Error,
                    Description = @"Care of' is not valid for address line 2",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line2.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000408"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS49", Value = GetValueOrEmpty(report.DIS49) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.000408
    
            #region VR.ATO.GEN.000408
    
            /*  VR.ATO.GEN.000408
            Care of' is not valid for address line 2
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line2.Text <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.Line2.Text STARTSWITH SET("C/- ","C/O ","C/ ","Care Of ","CO ") OR pyde.xx.xx:AddressDetails.Line2.Text CONTAINS SET(" C/- "," C/O "," C/ "," Care Of "))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^DIS65 <> BLANK) AND (StartsWithSet(^DIS65,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^DIS65, '" C/- "," C/O "," C/ "," Care Of "'))
    
            Data Elements:
    
            ^DIS65 = RP:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.DIS65) != true && (IsMatch(report.DIS65, @"^(C/- |C/O |C/ |Care Of |CO )",RegexOptions.IgnoreCase) || IsMatch(report.DIS65, @" C/- | C/O | C/ | Care Of ",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000408", Severity = ProcessMessageSeverity.Error,
                    Description = @"Care of' is not valid for address line 2",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line2.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000408"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS65", Value = GetValueOrEmpty(report.DIS65) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.000408
    
            #region VR.ATO.GEN.408010
    
            /*  VR.ATO.GEN.408010
            Name field cannot contain 'space hyphen space'.
    
            Legacy Rule Format:
            WHERE IN TUPLE(personunstructuredname1.xx.xx:PersonUnstructuredName)
            IF (pyde.xx.xx:PersonUnstructuredName.FullName.Text CONTAINS "" - "")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^DIS89, ' - ')
    
            Data Elements:
    
            ^DIS89 = INT:PersonUnstructuredName:PersonUnstructuredName.FullName.Text WHERE [PersonUnstructuredName.Usage.Code='Contact']
            */
            assertion = (report.DIS89 == null ? false : report.DIS89.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.408010", Severity = ProcessMessageSeverity.Error,
                    Description = @"Name field cannot contain 'space hyphen space'.",
                    LongDescription = @"Name field cannot contain 'space hyphen space'.",
                    Location = "/xbrli:xbrl/tns:PersonUnstructuredName" + OccurrenceIndex(report.PersonUnstructuredNameCollection_Contact_INTOccurrenceIndex) + "/tns:PersonUnstructuredName.FullName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.408010"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS89", Value = report.DIS89 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.408010
    
            #region VR.ATO.GEN.408010
    
            /*  VR.ATO.GEN.408010
            Name field cannot contain 'space hyphen space'.
    
            Legacy Rule Format:
            WHERE IN TUPLE(personunstructuredname1.xx.xx:PersonUnstructuredName)
            IF (pyde.xx.xx:PersonUnstructuredName.FullName.Text CONTAINS "" - "")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^DIS87, ' - ')
    
            Data Elements:
    
            ^DIS87 = INT:Declaration:PersonUnstructuredName:PersonUnstructuredName.FullName.Text WHERE [PersonUnstructuredName.Usage.Code='DeclarationSignatory']
            */
            assertion = (report.DIS87 == null ? false : report.DIS87.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.408010", Severity = ProcessMessageSeverity.Error,
                    Description = @"Name field cannot contain 'space hyphen space'.",
                    LongDescription = @"Name field cannot contain 'space hyphen space'.",
                    Location = "/xbrli:xbrl/tns:Declaration" + OccurrenceIndex(report.DeclarationCollection_TrueAndCorrect_INTOccurrenceIndex) + "/tns:PersonUnstructuredName" + OccurrenceIndex(report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTOccurrenceIndex) + "/tns:PersonUnstructuredName.FullName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.408010"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS87", Value = report.DIS87 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.408010
    
            #region VR.ATO.GEN.408010
    
            /*  VR.ATO.GEN.408010
            Name field cannot contain 'space hyphen space'.
    
            Legacy Rule Format:
            WHERE IN TUPLE(personunstructuredname1.xx.xx:PersonUnstructuredName)
            IF (pyde.xx.xx:PersonUnstructuredName.FullName.Text CONTAINS "" - "")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^DIS25, ' - ')
    
            Data Elements:
    
            ^DIS25 = RP:PersonUnstructuredName:PersonUnstructuredName.FullName.Text WHERE [PersonUnstructuredName.Usage.Code='Contact']
            */
            assertion = (report.DIS25 == null ? false : report.DIS25.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.408010", Severity = ProcessMessageSeverity.Error,
                    Description = @"Name field cannot contain 'space hyphen space'.",
                    LongDescription = @"Name field cannot contain 'space hyphen space'.",
                    Location = "/xbrli:xbrl/tns:PersonUnstructuredName" + OccurrenceIndex(report.PersonUnstructuredNameCollection_Contact_RPOccurrenceIndex) + "/tns:PersonUnstructuredName.FullName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.408010"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS25", Value = report.DIS25 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.408010
    
            #region VR.ATO.GEN.408010
    
            /*  VR.ATO.GEN.408010
            Name field cannot contain 'space hyphen space'.
    
            Legacy Rule Format:
            WHERE IN TUPLE(personunstructuredname1.xx.xx:PersonUnstructuredName)
            IF (pyde.xx.xx:PersonUnstructuredName.FullName.Text CONTAINS "" - "")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^DIS80, ' - ')
    
            Data Elements:
    
            ^DIS80 = RP:Declaration:PersonUnstructuredName:PersonUnstructuredName.FullName.Text WHERE [PersonUnstructuredName.Usage.Code='DeclarationSignatory']
            */
            assertion = (report.DIS80 == null ? false : report.DIS80.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.408010", Severity = ProcessMessageSeverity.Error,
                    Description = @"Name field cannot contain 'space hyphen space'.",
                    LongDescription = @"Name field cannot contain 'space hyphen space'.",
                    Location = "/xbrli:xbrl/tns:Declaration" + OccurrenceIndex(report.DeclarationCollection_TrueAndCorrect_RPOccurrenceIndex) + "/tns:PersonUnstructuredName" + OccurrenceIndex(report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPOccurrenceIndex) + "/tns:PersonUnstructuredName.FullName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.408010"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS80", Value = report.DIS80 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.408010
    
            #region VR.ATO.GEN.410002
    
            /*  VR.ATO.GEN.410002
            Address Line 2 must be present if Address Line 1 contains 'C/-'
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF ((pyde.xx.xx:AddressDetails.Line1.Text CONTAINS "C/-") AND (pyde.xx.xx:AddressDetails.Line2.Text = NULLORBLANK))  
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ((Contains(^DIS48, 'C/-')) AND (^DIS49 = BLANK))
    
            Data Elements:
    
            ^DIS48 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
    
            ^DIS49 = RP:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            */
            assertion = ((report.DIS48 == null ? false : report.DIS48.ToUpperInvariant().Contains(@"C/-")) && string.IsNullOrWhiteSpace(report.DIS49) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410002", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 2 must be present if Address Line 1 contains 'C/-'",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410002"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS48", Value = report.DIS48 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS49", Value = GetValueOrEmpty(report.DIS49) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410002
    
            #region VR.ATO.GEN.410002
    
            /*  VR.ATO.GEN.410002
            Address Line 2 must be present if Address Line 1 contains 'C/-'
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF ((pyde.xx.xx:AddressDetails.Line1.Text CONTAINS "C/-") AND (pyde.xx.xx:AddressDetails.Line2.Text = NULLORBLANK))  
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ((Contains(^DIS64, 'C/-')) AND (^DIS65 = BLANK))
    
            Data Elements:
    
            ^DIS64 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
    
            ^DIS65 = RP:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = ((report.DIS64 == null ? false : report.DIS64.ToUpperInvariant().Contains(@"C/-")) && string.IsNullOrWhiteSpace(report.DIS65) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410002", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 2 must be present if Address Line 1 contains 'C/-'",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410002"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS64", Value = report.DIS64 });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS65", Value = GetValueOrEmpty(report.DIS65) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410002
    
            #region VR.ATO.GEN.410008
    
            /*  VR.ATO.GEN.410008
            Address Line 3 must be left blank
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line3.Text <> NULLORBLANK) 
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^DIS50 <> BLANK
    
            Data Elements:
    
            ^DIS50 = RP:AddressDetails:AddressDetails.Line3.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.DIS50) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410008", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 3 must be left blank",
                    LongDescription = @"Lines 3 and 4 of the address are not used on this form",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line3.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410008"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS50", Value = GetValueOrEmpty(report.DIS50) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410008
    
            #region VR.ATO.GEN.410008
    
            /*  VR.ATO.GEN.410008
            Address Line 3 must be left blank
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line3.Text <> NULLORBLANK) 
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^DIS66 <> BLANK
    
            Data Elements:
    
            ^DIS66 = RP:AddressDetails:AddressDetails.Line3.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.DIS66) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410008", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 3 must be left blank",
                    LongDescription = @"Lines 3 and 4 of the address are not used on this form",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line3.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410008"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS66", Value = GetValueOrEmpty(report.DIS66) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410008
    
            #region VR.ATO.GEN.410013
    
            /*  VR.ATO.GEN.410013
            Address Line 4 must be left blank
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line4.Text <> NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^DIS51 <> BLANK
    
            Data Elements:
    
            ^DIS51 = RP:AddressDetails:AddressDetails.Line4.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.DIS51) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410013", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 4 must be left blank",
                    LongDescription = @"Lines 3 and 4 of the address are not used on this form",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line4.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410013"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS51", Value = GetValueOrEmpty(report.DIS51) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410013
    
            #region VR.ATO.GEN.410013
    
            /*  VR.ATO.GEN.410013
            Address Line 4 must be left blank
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line4.Text <> NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^DIS67 <> BLANK
    
            Data Elements:
    
            ^DIS67 = RP:AddressDetails:AddressDetails.Line4.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.DIS67) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410013", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 4 must be left blank",
                    LongDescription = @"Lines 3 and 4 of the address are not used on this form",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line4.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410013"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS67", Value = GetValueOrEmpty(report.DIS67) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410013
    
            #region VR.ATO.GEN.410039
    
            /*  VR.ATO.GEN.410039
            Organisation name must contain at least one alpha or numeric character
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text  <> NULLORBLANK) AND (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text  DOES NOT CONTAIN SET("a-z", "A-Z", "0-9"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^DIS91  <> BLANK) AND (NotContainsSet(^DIS91, '"a-z", "A-Z", "0-9"'))
    
            Data Elements:
    
            ^DIS91 = INT:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN']
            */
            assertion = (string.IsNullOrWhiteSpace(report.DIS91) != true && !(IsMatch(report.DIS91, @"[a-z]|[A-Z]|[0-9]",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410039", Severity = ProcessMessageSeverity.Error,
                    Description = @"Organisation name must contain at least one alpha or numeric character",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_MN_INTOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410039"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS91", Value = report.DIS91 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410039
    
            #region VR.ATO.GEN.410039
    
            /*  VR.ATO.GEN.410039
            Organisation name must contain at least one alpha or numeric character
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text  <> NULLORBLANK) AND (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text  DOES NOT CONTAIN SET("a-z", "A-Z", "0-9"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^DIS24  <> BLANK) AND (NotContainsSet(^DIS24, '"a-z", "A-Z", "0-9"'))
    
            Data Elements:
    
            ^DIS24 = RP:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN']
            */
            assertion = (string.IsNullOrWhiteSpace(report.DIS24) != true && !(IsMatch(report.DIS24, @"[a-z]|[A-Z]|[0-9]",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410039", Severity = ProcessMessageSeverity.Error,
                    Description = @"Organisation name must contain at least one alpha or numeric character",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_MN_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410039"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS24", Value = report.DIS24 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410039
    
            #region VR.ATO.GEN.410167
    
            /*  VR.ATO.GEN.410167
            Overseas address indicator must be present
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF (pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = NULLORBLANK)
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^AddressDetails <> NULL) AND (^DIS45 = BLANK)
    
            Data Elements:
    
            ^DIS45 = RP:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
    
            ^AddressDetails = RP:AddressDetails
            */
            assertion = (report.AddressDetailsCollection_BUS_C_RPExists != false && report.DIS45 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410167", Severity = ProcessMessageSeverity.Error,
                    Description = @"Overseas address indicator must be present",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.OverseasAddress.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410167"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS45", Value = GetValueOrEmpty(report.DIS45) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410167
    
            #region VR.ATO.GEN.410167
    
            /*  VR.ATO.GEN.410167
            Overseas address indicator must be present
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF (pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = NULLORBLANK)
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^AddressDetails <> NULL) AND (^DIS60 = BLANK)
    
            Data Elements:
    
            ^DIS60 = RP:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
    
            ^AddressDetails = RP:AddressDetails
            */
            assertion = (report.AddressDetailsCollection_POS_C_RPExists != false && report.DIS60 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410167", Severity = ProcessMessageSeverity.Error,
                    Description = @"Overseas address indicator must be present",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.OverseasAddress.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410167"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS60", Value = GetValueOrEmpty(report.DIS60) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410167
    
            #region VR.ATO.GEN.410191
    
            /*  VR.ATO.GEN.410191
            Country code must be present for overseas address
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF (pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = TRUE) AND (pyde.xx.xx:AddressDetails.Country.Code = NULLORBLANK)
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^DIS45 = TRUE) AND (^DIS59 = BLANK)
    
            Data Elements:
    
            ^DIS59 = RP:AddressDetails:AddressDetails.Country.Code WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
    
            ^DIS45 = RP:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (report.DIS45 == true && string.IsNullOrWhiteSpace(report.DIS59) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410191", Severity = ProcessMessageSeverity.Error,
                    Description = @"Country code must be present for overseas address",
                    LongDescription = @"When the overseas address indicator is 'TRUE', the corresponding Country Code must be supplied",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Country.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410191"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS45", Value = GetValueOrEmpty(report.DIS45) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS59", Value = GetValueOrEmpty(report.DIS59) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410191
    
            #region VR.ATO.GEN.410191
    
            /*  VR.ATO.GEN.410191
            Country code must be present for overseas address
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF (pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = TRUE) AND (pyde.xx.xx:AddressDetails.Country.Code = NULLORBLANK)
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^DIS60 = TRUE) AND (^DIS73 = BLANK)
    
            Data Elements:
    
            ^DIS73 = RP:AddressDetails:AddressDetails.Country.Code WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
    
            ^DIS60 = RP:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (report.DIS60 == true && string.IsNullOrWhiteSpace(report.DIS73) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410191", Severity = ProcessMessageSeverity.Error,
                    Description = @"Country code must be present for overseas address",
                    LongDescription = @"When the overseas address indicator is 'TRUE', the corresponding Country Code must be supplied",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Country.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410191"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS60", Value = GetValueOrEmpty(report.DIS60) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS73", Value = GetValueOrEmpty(report.DIS73) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410191
    
            #region VR.ATO.GEN.410194
    
            /*  VR.ATO.GEN.410194
            Address Line 1 must not exceed 38 characters
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:AddressDetails.Line1.Text) > 38 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^DIS48) > 38
    
            Data Elements:
    
            ^DIS48 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (Length(report.DIS48) > 38);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410194", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 1 must not exceed 38 characters",
                    LongDescription = @"Although SBR sets a limit of 50 characters for the Line 1 of an address, the maximum allowable for this form is 38",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410194"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS48", Value = report.DIS48 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410194
    
            #region VR.ATO.GEN.410194
    
            /*  VR.ATO.GEN.410194
            Address Line 1 must not exceed 38 characters
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:AddressDetails.Line1.Text) > 38 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^DIS64) > 38
    
            Data Elements:
    
            ^DIS64 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (Length(report.DIS64) > 38);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410194", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 1 must not exceed 38 characters",
                    LongDescription = @"Although SBR sets a limit of 50 characters for the Line 1 of an address, the maximum allowable for this form is 38",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410194"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS64", Value = report.DIS64 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410194
    
            #region VR.ATO.GEN.410195
    
            /*  VR.ATO.GEN.410195
            Address Line 2 must not exceed 38 characters
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:AddressDetails.Line2.Text) > 38
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^DIS49) > 38
    
            Data Elements:
    
            ^DIS49 = RP:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (Length(report.DIS49) > 38);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410195", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 2 must not exceed 38 characters",
                    LongDescription = @"Although SBR sets a limit of 50 characters for the Line 2 of an address, the maximum allowable for this form is 38",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line2.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410195"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS49", Value = GetValueOrEmpty(report.DIS49) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410195
    
            #region VR.ATO.GEN.410195
    
            /*  VR.ATO.GEN.410195
            Address Line 2 must not exceed 38 characters
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:AddressDetails.Line2.Text) > 38
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^DIS65) > 38
    
            Data Elements:
    
            ^DIS65 = RP:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (Length(report.DIS65) > 38);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410195", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 2 must not exceed 38 characters",
                    LongDescription = @"Although SBR sets a limit of 50 characters for the Line 2 of an address, the maximum allowable for this form is 38",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line2.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410195"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS65", Value = GetValueOrEmpty(report.DIS65) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410195
    
            #region VR.ATO.GEN.410205
    
            /*  VR.ATO.GEN.410205
            Address Line 1 must not contain 'AS ABOVE'
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line1.Text = FOUND("AS ABOVE"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^DIS48, '"AS ABOVE"'))
    
            Data Elements:
    
            ^DIS48 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.DIS48, @"(^(AS ABOVE)$)|(^(.*)( AS ABOVE )(.*)$)|(^(AS ABOVE )(.*)$)|(^(.*)( AS ABOVE)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410001", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 1 must not contain 'AS ABOVE'",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410205"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS48", Value = report.DIS48 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410205
    
            #region VR.ATO.GEN.410205
    
            /*  VR.ATO.GEN.410205
            Address Line 1 must not contain 'AS ABOVE'
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line1.Text = FOUND("AS ABOVE"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^DIS64, '"AS ABOVE"'))
    
            Data Elements:
    
            ^DIS64 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.DIS64, @"(^(AS ABOVE)$)|(^(.*)( AS ABOVE )(.*)$)|(^(AS ABOVE )(.*)$)|(^(.*)( AS ABOVE)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410001", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 1 must not contain 'AS ABOVE'",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410205"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS64", Value = report.DIS64 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410205
    
            #region VR.ATO.GEN.410211
    
            /*  VR.ATO.GEN.410211
            Overseas address indicator must not be 'TRUE' if Country Code is 'au'
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF (pyde.xx.xx:AddressDetails.Country.Code = "au") AND  (pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = TRUE)   
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^DIS59 = 'au') AND  (^DIS45 = TRUE)
    
            Data Elements:
    
            ^DIS59 = RP:AddressDetails:AddressDetails.Country.Code WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
    
            ^DIS45 = RP:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (report.DIS59 == @"au" && report.DIS45 == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410211", Severity = ProcessMessageSeverity.Error,
                    Description = @"Overseas address indicator must not be 'TRUE' if Country Code is 'au'",
                    LongDescription = @"The overseas address indicator must be 'FALSE' if the Country Code is 'au' - representing Australia. If the address is an overseas address please supply the correct Country Code",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Country.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410211"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS59", Value = GetValueOrEmpty(report.DIS59) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS45", Value = GetValueOrEmpty(report.DIS45) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410211
    
            #region VR.ATO.GEN.410211
    
            /*  VR.ATO.GEN.410211
            Overseas address indicator must not be 'TRUE' if Country Code is 'au'
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF (pyde.xx.xx:AddressDetails.Country.Code = "au") AND  (pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = TRUE)   
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^DIS73 = 'au') AND  (^DIS60 = TRUE)
    
            Data Elements:
    
            ^DIS73 = RP:AddressDetails:AddressDetails.Country.Code WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
    
            ^DIS60 = RP:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (report.DIS73 == @"au" && report.DIS60 == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410211", Severity = ProcessMessageSeverity.Error,
                    Description = @"Overseas address indicator must not be 'TRUE' if Country Code is 'au'",
                    LongDescription = @"The overseas address indicator must be 'FALSE' if the Country Code is 'au' - representing Australia. If the address is an overseas address please supply the correct Country Code",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Country.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410211"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS73", Value = GetValueOrEmpty(report.DIS73) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS60", Value = GetValueOrEmpty(report.DIS60) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410211
    
            #region VR.ATO.GEN.410212
    
            /*  VR.ATO.GEN.410212
            If an overseas Country Code is used, the Overseas Address indicator must be 'TRUE'
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)IF (pyde.xx.xx:AddressDetails.Country.Code <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.Country.Code <> "au") AND  (pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = FALSE)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^DIS59 <> BLANK) AND (^DIS59 <> 'au') AND  (^DIS45 = FALSE)
    
            Data Elements:
    
            ^DIS59 = RP:AddressDetails:AddressDetails.Country.Code WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
    
            ^DIS45 = RP:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.DIS59) != true && report.DIS59 != @"au" && report.DIS45 == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410212", Severity = ProcessMessageSeverity.Error,
                    Description = @"If an overseas Country Code is used, the Overseas Address indicator must be 'TRUE'",
                    LongDescription = @"If the address is an Australian address, the overseas address indicator must be set to 'FALSE' and you do not need to supply a Country Code. If the address is an overseas address, the overseas address indicator must be set to 'TRUE' and you must supply a Country Code other than 'au'",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Country.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410212"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS59", Value = GetValueOrEmpty(report.DIS59) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS45", Value = GetValueOrEmpty(report.DIS45) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410212
    
            #region VR.ATO.GEN.410212
    
            /*  VR.ATO.GEN.410212
            If an overseas Country Code is used, the Overseas Address indicator must be 'TRUE'
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)IF (pyde.xx.xx:AddressDetails.Country.Code <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.Country.Code <> "au") AND  (pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = FALSE)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^DIS73 <> BLANK) AND (^DIS73 <> 'au') AND  (^DIS60 = FALSE)
    
            Data Elements:
    
            ^DIS73 = RP:AddressDetails:AddressDetails.Country.Code WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
    
            ^DIS60 = RP:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.DIS73) != true && report.DIS73 != @"au" && report.DIS60 == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410212", Severity = ProcessMessageSeverity.Error,
                    Description = @"If an overseas Country Code is used, the Overseas Address indicator must be 'TRUE'",
                    LongDescription = @"If the address is an Australian address, the overseas address indicator must be set to 'FALSE' and you do not need to supply a Country Code. If the address is an overseas address, the overseas address indicator must be set to 'TRUE' and you must supply a Country Code other than 'au'",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Country.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410212"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS73", Value = GetValueOrEmpty(report.DIS73) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS60", Value = GetValueOrEmpty(report.DIS60) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410212
    
            #region VR.ATO.GEN.410213
    
            /*  VR.ATO.GEN.410213
            Country Name must be left blank
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.CountryName.Text <> NULLORBLANK)
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^DIS58 <> BLANK
    
            Data Elements:
    
            ^DIS58 = RP:AddressDetails:AddressDetails.CountryName.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.DIS58) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410213", Severity = ProcessMessageSeverity.Error,
                    Description = @"Country Name must be left blank",
                    LongDescription = @"For SBR submissions to the ATO, the Country Name must be left blank. To supply the country for an overseas address please supply a Country Code",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.CountryName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410213"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS58", Value = GetValueOrEmpty(report.DIS58) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410213
    
            #region VR.ATO.GEN.410213
    
            /*  VR.ATO.GEN.410213
            Country Name must be left blank
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.CountryName.Text <> NULLORBLANK)
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^DIS72 <> BLANK
    
            Data Elements:
    
            ^DIS72 = RP:AddressDetails:AddressDetails.CountryName.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.DIS72) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410213", Severity = ProcessMessageSeverity.Error,
                    Description = @"Country Name must be left blank",
                    LongDescription = @"For SBR submissions to the ATO, the Country Name must be left blank. To supply the country for an overseas address please supply a Country Code",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.CountryName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410213"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS72", Value = GetValueOrEmpty(report.DIS72) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410213
    
            #region VR.ATO.GEN.410214
    
            /*  VR.ATO.GEN.410214
            Address Line 1 must be present
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF (pyde.xx.xx:AddressDetails.Line1.Text = NULLORBLANK)
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^AddressDetails <> NULL) AND (^DIS48 = BLANK)
    
            Data Elements:
    
            ^DIS48 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
    
            ^AddressDetails = RP:AddressDetails
            */
            assertion = (report.AddressDetailsCollection_BUS_C_RPExists != false && string.IsNullOrWhiteSpace(report.DIS48) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410214", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 1 must be present",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410214"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS48", Value = report.DIS48 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410214
    
            #region VR.ATO.GEN.410214
    
            /*  VR.ATO.GEN.410214
            Address Line 1 must be present
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF (pyde.xx.xx:AddressDetails.Line1.Text = NULLORBLANK)
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^AddressDetails <> NULL) AND (^DIS64 = BLANK)
    
            Data Elements:
    
            ^DIS64 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
    
            ^AddressDetails = RP:AddressDetails
            */
            assertion = (report.AddressDetailsCollection_POS_C_RPExists != false && string.IsNullOrWhiteSpace(report.DIS64) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410214", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 1 must be present",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.410214"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS64", Value = report.DIS64 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.410214
    
            #region VR.ATO.GEN.428040
    
            /*  VR.ATO.GEN.428040
            ABN is invalid
    
            Legacy Rule Format:
            IF (pyid.xx.xx:IdentifiersAustralianBusinessNumberIdentifier <> NULLORBLANK) and (ABNALGORITHM(pyid.xx.xx:IdentifiersAustralianBusinessNumber.Identifier) = FALSE)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^DIS23 <> BLANK) and (FailsABNAlgorithm(^DIS23))
    
            Data Elements:
    
            ^DIS23 = RP:Identifiers.AustralianBusinessNumber.Identifier
            */
            assertion = (string.IsNullOrWhiteSpace(report.DIS23) != true && FailsABNAlgorithm(report.DIS23));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000477", Severity = ProcessMessageSeverity.Error,
                    Description = @"ABN is invalid",
                    Location = "/xbrli:xbrl/tns:Identifiers.AustralianBusinessNumber.Identifier[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428040"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS23", Value = GetValueOrEmpty(report.DIS23) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428040
    
            #region VR.ATO.GEN.428230
    
            /*  VR.ATO.GEN.428230
            Suburb/Town/City cannot contain state code
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.LocalityName.Text = FOUND("QLD","NSW","VIC","SA","WA","NT","ACT","TAS"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^DIS52, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))
    
            Data Elements:
    
            ^DIS52 = RP:AddressDetails:AddressDetails.LocalityName.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.DIS52, @"(^(QLD|NSW|VIC|SA|WA|NT|ACT|TAS)$)|(^(.*)( QLD | NSW | VIC | SA | WA | NT | ACT | TAS )(.*)$)|(^(QLD |NSW |VIC |SA |WA |NT |ACT |TAS )(.*)$)|(^(.*)( QLD| NSW| VIC| SA| WA| NT| ACT| TAS)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000411", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain state code",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.LocalityName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428230"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS52", Value = GetValueOrEmpty(report.DIS52) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428230
    
            #region VR.ATO.GEN.428230
    
            /*  VR.ATO.GEN.428230
            Suburb/Town/City cannot contain state code
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.LocalityName.Text = FOUND("QLD","NSW","VIC","SA","WA","NT","ACT","TAS"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^DIS69, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))
    
            Data Elements:
    
            ^DIS69 = RP:AddressDetails:AddressDetails.LocalityName.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.DIS69, @"(^(QLD|NSW|VIC|SA|WA|NT|ACT|TAS)$)|(^(.*)( QLD | NSW | VIC | SA | WA | NT | ACT | TAS )(.*)$)|(^(QLD |NSW |VIC |SA |WA |NT |ACT |TAS )(.*)$)|(^(.*)( QLD| NSW| VIC| SA| WA| NT| ACT| TAS)$)",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000411", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain state code",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.LocalityName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428230"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS69", Value = GetValueOrEmpty(report.DIS69) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428230
    
            #region VR.ATO.GEN.428247
    
            /*  VR.ATO.GEN.428247
            The Tax Agent number must pass the Tax Agent Number algorithm check
    
            Legacy Rule Format:
            IF ((pyid.xx.xx:Identifiers.TaxAgentNumber.Identifier) <> NULLORBLANK) AND (TANALGORITHM(pyid.xx.xx:Identifiers.TaxAgentNumber.Identifier) = FALSE)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^DIS68 <> BLANK) AND (FailsTANAlgorithm(^DIS68))
    
            Data Elements:
    
            ^DIS68 = INT:Identifiers.TaxAgentNumber.Identifier
            */
            assertion = (string.IsNullOrWhiteSpace(report.DIS68) != true && FailsTANAlgorithm(report.DIS68));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410009", Severity = ProcessMessageSeverity.Error,
                    Description = @"The Tax Agent number has failed the Tax Agent number algorithm check",
                    Location = "/xbrli:xbrl/tns:Identifiers.TaxAgentNumber.Identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428247"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS68", Value = GetValueOrEmpty(report.DIS68) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428247
    
            #region VR.ATO.GEN.428254
    
            /*  VR.ATO.GEN.428254
            Address line 1 cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line1.Text CONTAINS "UNKNOWN")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^DIS48, 'UNKNOWN')
    
            Data Elements:
    
            ^DIS48 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (report.DIS48 == null ? false : report.DIS48.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000405", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 cannot contain ""UNKNOWN""",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428254"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS48", Value = report.DIS48 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428254
    
            #region VR.ATO.GEN.428254
    
            /*  VR.ATO.GEN.428254
            Address line 1 cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line1.Text CONTAINS "UNKNOWN")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^DIS64, 'UNKNOWN')
    
            Data Elements:
    
            ^DIS64 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (report.DIS64 == null ? false : report.DIS64.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000405", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 cannot contain ""UNKNOWN""",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428254"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS64", Value = report.DIS64 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428254
    
            #region VR.ATO.GEN.428255
    
            /*  VR.ATO.GEN.428255
            Address line 2 cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line2.Text CONTAINS "UNKNOWN")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^DIS49, 'UNKNOWN')
    
            Data Elements:
    
            ^DIS49 = RP:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (report.DIS49 == null ? false : report.DIS49.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000407", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 2 cannot contain ""UNKNOWN""",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line2.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428255"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS49", Value = GetValueOrEmpty(report.DIS49) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428255
    
            #region VR.ATO.GEN.428255
    
            /*  VR.ATO.GEN.428255
            Address line 2 cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line2.Text CONTAINS "UNKNOWN")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^DIS65, 'UNKNOWN')
    
            Data Elements:
    
            ^DIS65 = RP:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (report.DIS65 == null ? false : report.DIS65.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000407", Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 2 cannot contain ""UNKNOWN""",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line2.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428255"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS65", Value = GetValueOrEmpty(report.DIS65) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428255
    
            #region VR.ATO.GEN.428256
    
            /*  VR.ATO.GEN.428256
            Suburb/Town/City cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.LocalityName.Text CONTAINS "UNKNOWN")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^DIS52, 'UNKNOWN')
    
            Data Elements:
    
            ^DIS52 = RP:AddressDetails:AddressDetails.LocalityName.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (report.DIS52 == null ? false : report.DIS52.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000409", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain ""UNKNOWN""",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.LocalityName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428256"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS52", Value = GetValueOrEmpty(report.DIS52) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428256
    
            #region VR.ATO.GEN.428256
    
            /*  VR.ATO.GEN.428256
            Suburb/Town/City cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.LocalityName.Text CONTAINS "UNKNOWN")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^DIS69, 'UNKNOWN')
    
            Data Elements:
    
            ^DIS69 = RP:AddressDetails:AddressDetails.LocalityName.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (report.DIS69 == null ? false : report.DIS69.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000409", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain ""UNKNOWN""",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.LocalityName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428256"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS69", Value = GetValueOrEmpty(report.DIS69) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428256
    
            #region VR.ATO.GEN.428258
    
            /*  VR.ATO.GEN.428258
            Non Individual Name cannot contain space hyphen space combinations
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text CONTAINS  " - " )
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^DIS91, ' - ')
    
            Data Elements:
    
            ^DIS91 = INT:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN']
            */
            assertion = (report.DIS91 == null ? false : report.DIS91.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000413", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain space hyphen space combinations",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_MN_INTOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428258"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS91", Value = report.DIS91 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428258
    
            #region VR.ATO.GEN.428258
    
            /*  VR.ATO.GEN.428258
            Non Individual Name cannot contain space hyphen space combinations
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text CONTAINS  " - " )
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^DIS24, ' - ')
    
            Data Elements:
    
            ^DIS24 = RP:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN']
            */
            assertion = (report.DIS24 == null ? false : report.DIS24.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000413", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain space hyphen space combinations",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_MN_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428258"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS24", Value = report.DIS24 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428258
    
            #region VR.ATO.GEN.428259
    
            /*  VR.ATO.GEN.428259
            Non Individual Name cannot contain "P/L"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text CONTAINS "P/L")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^DIS91, 'P/L')
    
            Data Elements:
    
            ^DIS91 = INT:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN']
            */
            assertion = (report.DIS91 == null ? false : report.DIS91.ToUpperInvariant().Contains(@"P/L"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000414", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain ""P/L""",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_MN_INTOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428259"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS91", Value = report.DIS91 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428259
    
            #region VR.ATO.GEN.428259
    
            /*  VR.ATO.GEN.428259
            Non Individual Name cannot contain "P/L"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text CONTAINS "P/L")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^DIS24, 'P/L')
    
            Data Elements:
    
            ^DIS24 = RP:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN']
            */
            assertion = (report.DIS24 == null ? false : report.DIS24.ToUpperInvariant().Contains(@"P/L"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000414", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain ""P/L""",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_MN_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428259"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS24", Value = report.DIS24 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428259
    
            #region VR.ATO.GEN.428260
    
            /*  VR.ATO.GEN.428260
            Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text CONTAINS SET("--","''","  ") )
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (ContainsSet(^DIS91, '"--","''","  "'))
    
            Data Elements:
    
            ^DIS91 = INT:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN']
            */
            assertion = IsMatch(report.DIS91, @"--|''|  ",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000416", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_MN_INTOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428260"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS91", Value = report.DIS91 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428260
    
            #region VR.ATO.GEN.428260
    
            /*  VR.ATO.GEN.428260
            Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text CONTAINS SET("--","''","  ") )
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (ContainsSet(^DIS24, '"--","''","  "'))
    
            Data Elements:
    
            ^DIS24 = RP:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN']
            */
            assertion = IsMatch(report.DIS24, @"--|''|  ",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000416", Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_MN_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.428260"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS24", Value = report.DIS24 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.428260
    
            #region VR.ATO.GEN.430245
    
            /*  VR.ATO.GEN.430245
            Suburb/Town/City must be present in an address
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF (pyde.xx.xx:AddressDetails.LocalityName.Text = NULLORBLANK)
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^AddressDetails <> NULL) AND (^DIS52 = BLANK)
    
            Data Elements:
    
            ^DIS52 = RP:AddressDetails:AddressDetails.LocalityName.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
    
            ^AddressDetails = RP:AddressDetails
            */
            assertion = (report.AddressDetailsCollection_BUS_C_RPExists != false && string.IsNullOrWhiteSpace(report.DIS52) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430245", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City must be present in an address",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.LocalityName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.430245"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS52", Value = GetValueOrEmpty(report.DIS52) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.430245
    
            #region VR.ATO.GEN.430245
    
            /*  VR.ATO.GEN.430245
            Suburb/Town/City must be present in an address
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF (pyde.xx.xx:AddressDetails.LocalityName.Text = NULLORBLANK)
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^AddressDetails <> NULL) AND (^DIS69 = BLANK)
    
            Data Elements:
    
            ^DIS69 = RP:AddressDetails:AddressDetails.LocalityName.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
    
            ^AddressDetails = RP:AddressDetails
            */
            assertion = (report.AddressDetailsCollection_POS_C_RPExists != false && string.IsNullOrWhiteSpace(report.DIS69) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430245", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City must be present in an address",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.LocalityName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.430245"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS69", Value = GetValueOrEmpty(report.DIS69) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.430245
    
            #region VR.ATO.GEN.430246
    
            /*  VR.ATO.GEN.430246
            Suburb/Town/City cannot contain numeric characters
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF (pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = FALSE) AND (pyde.xx.xx:AddressDetails.LocalityName.Text CONTAINS NUMERICS SET(0-9))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^DIS45 = FALSE) AND (ContainsSet(^DIS52, '0-9'))
    
            Data Elements:
    
            ^DIS52 = RP:AddressDetails:AddressDetails.LocalityName.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
    
            ^DIS45 = RP:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (report.DIS45 == false && IsMatch(report.DIS52, @"[0-9]",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000410", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain numeric characters",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.LocalityName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.430246"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS45", Value = GetValueOrEmpty(report.DIS45) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS52", Value = GetValueOrEmpty(report.DIS52) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.430246
    
            #region VR.ATO.GEN.430246
    
            /*  VR.ATO.GEN.430246
            Suburb/Town/City cannot contain numeric characters
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF (pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = FALSE) AND (pyde.xx.xx:AddressDetails.LocalityName.Text CONTAINS NUMERICS SET(0-9))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^DIS60 = FALSE) AND (ContainsSet(^DIS69, '0-9'))
    
            Data Elements:
    
            ^DIS69 = RP:AddressDetails:AddressDetails.LocalityName.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
    
            ^DIS60 = RP:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (report.DIS60 == false && IsMatch(report.DIS69, @"[0-9]",RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000410", Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain numeric characters",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.LocalityName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.430246"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS60", Value = GetValueOrEmpty(report.DIS60) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS69", Value = GetValueOrEmpty(report.DIS69) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.430246
    
            #region VR.ATO.GEN.430252
    
            /*  VR.ATO.GEN.430252
            Unstructured name must not be more than 200 characters
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:PersonUnstructuredName.FullName.Text) > 200
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^DIS89) > 200
    
            Data Elements:
    
            ^DIS89 = INT:PersonUnstructuredName:PersonUnstructuredName.FullName.Text WHERE [PersonUnstructuredName.Usage.Code='Contact']
            */
            assertion = (Length(report.DIS89) > 200);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430252", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unstructured name must not be more than 200 characters",
                    Location = "/xbrli:xbrl/tns:PersonUnstructuredName" + OccurrenceIndex(report.PersonUnstructuredNameCollection_Contact_INTOccurrenceIndex) + "/tns:PersonUnstructuredName.FullName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.430252"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS89", Value = report.DIS89 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.430252
    
            #region VR.ATO.GEN.430252
    
            /*  VR.ATO.GEN.430252
            Unstructured name must not be more than 200 characters
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:PersonUnstructuredName.FullName.Text) > 200
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^DIS87) > 200
    
            Data Elements:
    
            ^DIS87 = INT:Declaration:PersonUnstructuredName:PersonUnstructuredName.FullName.Text WHERE [PersonUnstructuredName.Usage.Code='DeclarationSignatory']
            */
            assertion = (Length(report.DIS87) > 200);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430252", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unstructured name must not be more than 200 characters",
                    Location = "/xbrli:xbrl/tns:Declaration" + OccurrenceIndex(report.DeclarationCollection_TrueAndCorrect_INTOccurrenceIndex) + "/tns:PersonUnstructuredName" + OccurrenceIndex(report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTOccurrenceIndex) + "/tns:PersonUnstructuredName.FullName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.430252"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS87", Value = report.DIS87 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.430252
    
            #region VR.ATO.GEN.430252
    
            /*  VR.ATO.GEN.430252
            Unstructured name must not be more than 200 characters
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:PersonUnstructuredName.FullName.Text) > 200
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^DIS25) > 200
    
            Data Elements:
    
            ^DIS25 = RP:PersonUnstructuredName:PersonUnstructuredName.FullName.Text WHERE [PersonUnstructuredName.Usage.Code='Contact']
            */
            assertion = (Length(report.DIS25) > 200);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430252", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unstructured name must not be more than 200 characters",
                    Location = "/xbrli:xbrl/tns:PersonUnstructuredName" + OccurrenceIndex(report.PersonUnstructuredNameCollection_Contact_RPOccurrenceIndex) + "/tns:PersonUnstructuredName.FullName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.430252"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS25", Value = report.DIS25 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.430252
    
            #region VR.ATO.GEN.430252
    
            /*  VR.ATO.GEN.430252
            Unstructured name must not be more than 200 characters
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:PersonUnstructuredName.FullName.Text) > 200
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^DIS80) > 200
    
            Data Elements:
    
            ^DIS80 = RP:Declaration:PersonUnstructuredName:PersonUnstructuredName.FullName.Text WHERE [PersonUnstructuredName.Usage.Code='DeclarationSignatory']
            */
            assertion = (Length(report.DIS80) > 200);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430252", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unstructured name must not be more than 200 characters",
                    Location = "/xbrli:xbrl/tns:Declaration" + OccurrenceIndex(report.DeclarationCollection_TrueAndCorrect_RPOccurrenceIndex) + "/tns:PersonUnstructuredName" + OccurrenceIndex(report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPOccurrenceIndex) + "/tns:PersonUnstructuredName.FullName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.430252"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS80", Value = report.DIS80 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.430252
    
            #region VR.ATO.GEN.430255
    
            /*  VR.ATO.GEN.430255
            Signature date must not be later than today
    
            Legacy Rule Format:
            IF pyde.xx.xx:Declaration.Signature.Date <> NULL AND pyde.xx.xx:Declaration.Signature.Date > DATE(TODAY)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^DIS84 <> NULL AND ^DIS84 > Today()
    
            Data Elements:
    
            ^DIS84 = INT:Declaration:Declaration.Signature.Date WHERE [Declaration.StatementType.Code='TrueAndCorrect']
            */
            assertion = (report.DIS84 != null && report.DIS84.GetValueOrDefault() > DateTime.Now.Date);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430255", Severity = ProcessMessageSeverity.Error,
                    Description = @"Signature date must not be later than today",
                    Location = "/xbrli:xbrl/tns:Declaration" + OccurrenceIndex(report.DeclarationCollection_TrueAndCorrect_INTOccurrenceIndex) + "/tns:Declaration.Signature.Date",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.430255"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS84", Value = GetValueOrEmpty(report.DIS84) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.430255
    
            #region VR.ATO.GEN.430255
    
            /*  VR.ATO.GEN.430255
            Signature date must not be later than today
    
            Legacy Rule Format:
            IF pyde.xx.xx:Declaration.Signature.Date <> NULL AND pyde.xx.xx:Declaration.Signature.Date > DATE(TODAY)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^DIS77 <> NULL AND ^DIS77 > Today()
    
            Data Elements:
    
            ^DIS77 = RP:Declaration:Declaration.Signature.Date WHERE [Declaration.StatementType.Code='TrueAndCorrect']
            */
            assertion = (report.DIS77 != null && report.DIS77.GetValueOrDefault() > DateTime.Now.Date);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430255", Severity = ProcessMessageSeverity.Error,
                    Description = @"Signature date must not be later than today",
                    Location = "/xbrli:xbrl/tns:Declaration" + OccurrenceIndex(report.DeclarationCollection_TrueAndCorrect_RPOccurrenceIndex) + "/tns:Declaration.Signature.Date",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.430255"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS77", Value = GetValueOrEmpty(report.DIS77) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.430255
    
            #region VR.ATO.GEN.434147
    
            /*  VR.ATO.GEN.434147
            State and post code must be present for an Australian address
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF pyde.xx.xx:AddressDetails.OverseasAddressIndicator = FALSE AND (pyde.xx.xx:AddressDetails.StateOrTerritory.Code = NULLORBLANK OR pyde.xx.xx:AddressDetails.Postcode.Text = NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^AddressDetails <> NULL) AND (^DIS45 = FALSE AND (^DIS57 = BLANK OR ^DIS56 = BLANK))
    
            Data Elements:
    
            ^DIS45 = RP:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
    
            ^DIS57 = RP:AddressDetails:AddressDetails.StateOrTerritory.Code WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
    
            ^DIS56 = RP:AddressDetails:AddressDetails.Postcode.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
    
            ^AddressDetails = RP:AddressDetails
            */
            assertion = (report.AddressDetailsCollection_BUS_C_RPExists != false && (report.DIS45 == false && (string.IsNullOrWhiteSpace(report.DIS57) == true || string.IsNullOrWhiteSpace(report.DIS56) == true)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.434147", Severity = ProcessMessageSeverity.Error,
                    Description = @"State and post code must be present for an Australian address",
                    LongDescription = @"If the overseas indicator is no (false), then an Australian state code and post code must be present",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.OverseasAddress.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.434147"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS45", Value = GetValueOrEmpty(report.DIS45) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS57", Value = GetValueOrEmpty(report.DIS57) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS56", Value = GetValueOrEmpty(report.DIS56) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.434147
    
            #region VR.ATO.GEN.434147
    
            /*  VR.ATO.GEN.434147
            State and post code must be present for an Australian address
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF pyde.xx.xx:AddressDetails.OverseasAddressIndicator = FALSE AND (pyde.xx.xx:AddressDetails.StateOrTerritory.Code = NULLORBLANK OR pyde.xx.xx:AddressDetails.Postcode.Text = NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^AddressDetails <> NULL) AND (^DIS60 = FALSE AND (^DIS71 = BLANK OR ^DIS70 = BLANK))
    
            Data Elements:
    
            ^DIS60 = RP:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
    
            ^DIS71 = RP:AddressDetails:AddressDetails.StateOrTerritory.Code WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
    
            ^DIS70 = RP:AddressDetails:AddressDetails.Postcode.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
    
            ^AddressDetails = RP:AddressDetails
            */
            assertion = (report.AddressDetailsCollection_POS_C_RPExists != false && (report.DIS60 == false && (string.IsNullOrWhiteSpace(report.DIS71) == true || string.IsNullOrWhiteSpace(report.DIS70) == true)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.434147", Severity = ProcessMessageSeverity.Error,
                    Description = @"State and post code must be present for an Australian address",
                    LongDescription = @"If the overseas indicator is no (false), then an Australian state code and post code must be present",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.OverseasAddress.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.434147"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS60", Value = GetValueOrEmpty(report.DIS60) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS71", Value = GetValueOrEmpty(report.DIS71) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS70", Value = GetValueOrEmpty(report.DIS70) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.434147
    
            #region VR.ATO.GEN.436003
    
            /*  VR.ATO.GEN.436003
            Organisation name must be present
    
            Legacy Rule Format:
            WHERE IN TUPLE(organisationname1.xx.xx:OrganisationNameDetails)
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text = NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^OrganisationNameDetails <> NULL) AND (^DIS91 = BLANK)
    
            Data Elements:
    
            ^DIS91 = INT:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN']
    
            ^OrganisationNameDetails = INT:OrganisationNameDetails
            */
            assertion = (report.OrganisationNameDetailsCollection_MN_INTExists != false && string.IsNullOrWhiteSpace(report.DIS91) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410215", Severity = ProcessMessageSeverity.Error,
                    Description = @"Organisation name must be present",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_MN_INTOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.436003"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS91", Value = report.DIS91 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.436003
    
            #region VR.ATO.GEN.436003
    
            /*  VR.ATO.GEN.436003
            Organisation name must be present
    
            Legacy Rule Format:
            WHERE IN TUPLE(organisationname1.xx.xx:OrganisationNameDetails)
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text = NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^OrganisationNameDetails <> NULL) AND (^DIS24 = BLANK)
    
            Data Elements:
    
            ^DIS24 = RP:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN']
    
            ^OrganisationNameDetails = RP:OrganisationNameDetails
            */
            assertion = (report.OrganisationNameDetailsCollection_MN_RPExists != false && string.IsNullOrWhiteSpace(report.DIS24) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410215", Severity = ProcessMessageSeverity.Error,
                    Description = @"Organisation name must be present",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_MN_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.436003"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS24", Value = report.DIS24 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.436003
    
            #region VR.ATO.GEN.438029
    
            /*  VR.ATO.GEN.438029
            Entity identifier scheme for Reporting Party must be"www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP
    
            ^Context = RP
            */
            assertion = (report.RPExists != false && report.RPIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029", Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.438029
    
            #region VR.ATO.GEN.438029
    
            /*  VR.ATO.GEN.438029
            Entity identifier scheme for Reporting Party must be"www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.CFI
    
            ^Context = RP:RP.CFI
            */
            assertion = (report.RPCFIExists != false && report.RPCFIIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029", Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPCFIOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.438029
    
            #region VR.ATO.GEN.438029
    
            /*  VR.ATO.GEN.438029
            Entity identifier scheme for Reporting Party must be"www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.NONCFI
    
            ^Context = RP:RP.NONCFI
            */
            assertion = (report.RPNONCFIExists != false && report.RPNONCFIIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029", Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPNONCFIOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.438029
    
            #region VR.ATO.GEN.438030
    
            /*  VR.ATO.GEN.438030
            Scheme for Intermediary must be set to http://www.ato.gov.au/abn
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = “RprtPyType.xx.xx:Intermediary”) AND (entity.identifier.scheme <> “http://www.ato.gov.au/abn”)
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^IdentifierScheme <> 'http://www.ato.gov.au/abn'
    
            Data Elements:
    
            ^IdentifierScheme = INT
    
            ^Context = INT
            */
            assertion = (report.INTExists != false && report.INTIdentifierScheme != @"http://www.ato.gov.au/abn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438030", Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for Intermediary must be ""http://www.ato.gov.au/abn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.INTOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438030"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.438030
    
            #region VR.ATO.GEN.438035
    
            /*  VR.ATO.GEN.438035
            Unstructured name must not  contain invalid characters ~ ^ < > + , * 
            Invalid characters ~^<>+  have been removed from the techincal business rule as they will be picked by the XBRL validator.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonUnstructuredName.FullName.Text <> NULLORBLANK AND pyde.xx.xx:PersonUnstructuredName.FullName.Text CONTAINS SET(",","*"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ContainsSet(^DIS89, '",","*"')
    
            Data Elements:
    
            ^DIS89 = INT:PersonUnstructuredName:PersonUnstructuredName.FullName.Text WHERE [PersonUnstructuredName.Usage.Code='Contact']
            */
            assertion = IsMatch(report.DIS89, @",|\*",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438035", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unstructured name must not contain invalid characters ~ ^ < > + , *",
                    Location = "/xbrli:xbrl/tns:PersonUnstructuredName" + OccurrenceIndex(report.PersonUnstructuredNameCollection_Contact_INTOccurrenceIndex) + "/tns:PersonUnstructuredName.FullName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438035"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS89", Value = report.DIS89 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.438035
    
            #region VR.ATO.GEN.438035
    
            /*  VR.ATO.GEN.438035
            Unstructured name must not  contain invalid characters ~ ^ < > + , * 
            Invalid characters ~^<>+  have been removed from the techincal business rule as they will be picked by the XBRL validator.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonUnstructuredName.FullName.Text <> NULLORBLANK AND pyde.xx.xx:PersonUnstructuredName.FullName.Text CONTAINS SET(",","*"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ContainsSet(^DIS87, '",","*"')
    
            Data Elements:
    
            ^DIS87 = INT:Declaration:PersonUnstructuredName:PersonUnstructuredName.FullName.Text WHERE [PersonUnstructuredName.Usage.Code='DeclarationSignatory']
            */
            assertion = IsMatch(report.DIS87, @",|\*",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438035", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unstructured name must not contain invalid characters ~ ^ < > + , *",
                    Location = "/xbrli:xbrl/tns:Declaration" + OccurrenceIndex(report.DeclarationCollection_TrueAndCorrect_INTOccurrenceIndex) + "/tns:PersonUnstructuredName" + OccurrenceIndex(report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTOccurrenceIndex) + "/tns:PersonUnstructuredName.FullName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438035"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS87", Value = report.DIS87 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.438035
    
            #region VR.ATO.GEN.438035
    
            /*  VR.ATO.GEN.438035
            Unstructured name must not  contain invalid characters ~ ^ < > + , * 
            Invalid characters ~^<>+  have been removed from the techincal business rule as they will be picked by the XBRL validator.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonUnstructuredName.FullName.Text <> NULLORBLANK AND pyde.xx.xx:PersonUnstructuredName.FullName.Text CONTAINS SET(",","*"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ContainsSet(^DIS25, '",","*"')
    
            Data Elements:
    
            ^DIS25 = RP:PersonUnstructuredName:PersonUnstructuredName.FullName.Text WHERE [PersonUnstructuredName.Usage.Code='Contact']
            */
            assertion = IsMatch(report.DIS25, @",|\*",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438035", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unstructured name must not contain invalid characters ~ ^ < > + , *",
                    Location = "/xbrli:xbrl/tns:PersonUnstructuredName" + OccurrenceIndex(report.PersonUnstructuredNameCollection_Contact_RPOccurrenceIndex) + "/tns:PersonUnstructuredName.FullName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438035"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS25", Value = report.DIS25 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.438035
    
            #region VR.ATO.GEN.438035
    
            /*  VR.ATO.GEN.438035
            Unstructured name must not  contain invalid characters ~ ^ < > + , * 
            Invalid characters ~^<>+  have been removed from the techincal business rule as they will be picked by the XBRL validator.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonUnstructuredName.FullName.Text <> NULLORBLANK AND pyde.xx.xx:PersonUnstructuredName.FullName.Text CONTAINS SET(",","*"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ContainsSet(^DIS80, '",","*"')
    
            Data Elements:
    
            ^DIS80 = RP:Declaration:PersonUnstructuredName:PersonUnstructuredName.FullName.Text WHERE [PersonUnstructuredName.Usage.Code='DeclarationSignatory']
            */
            assertion = IsMatch(report.DIS80, @",|\*",RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438035", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unstructured name must not contain invalid characters ~ ^ < > + , *",
                    Location = "/xbrli:xbrl/tns:Declaration" + OccurrenceIndex(report.DeclarationCollection_TrueAndCorrect_RPOccurrenceIndex) + "/tns:PersonUnstructuredName" + OccurrenceIndex(report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPOccurrenceIndex) + "/tns:PersonUnstructuredName.FullName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.438035"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS80", Value = report.DIS80 });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.438035
    
            #region VR.ATO.GEN.500141
    
            /*  VR.ATO.GEN.500141
            For an Australian address, state or territory must be one of: ACT, NSW, NT, QLD, SA, VIC, WA, TAS
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.StateOrTerritory.Code <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.StateOrTerritory.Code <> SET ("ACT","NSW","NT","QLD","SA","VIC","WA","TAS"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^DIS57 <> BLANK) AND (NotInSet(^DIS57, '"ACT","NSW","NT","QLD","SA","VIC","WA","TAS"'))
    
            Data Elements:
    
            ^DIS57 = RP:AddressDetails:AddressDetails.StateOrTerritory.Code WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.DIS57) != true && !(IsMatch(report.DIS57, @"^(ACT|NSW|NT|QLD|SA|VIC|WA|TAS)$",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500141", Severity = ProcessMessageSeverity.Error,
                    Description = @"For an Australian address, state or territory must be one of: ACT, NSW, NT, QLD, SA, VIC, WA, TAS",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.StateOrTerritory.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500141"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS57", Value = GetValueOrEmpty(report.DIS57) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500141
    
            #region VR.ATO.GEN.500141
    
            /*  VR.ATO.GEN.500141
            For an Australian address, state or territory must be one of: ACT, NSW, NT, QLD, SA, VIC, WA, TAS
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.StateOrTerritory.Code <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.StateOrTerritory.Code <> SET ("ACT","NSW","NT","QLD","SA","VIC","WA","TAS"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^DIS71 <> BLANK) AND (NotInSet(^DIS71, '"ACT","NSW","NT","QLD","SA","VIC","WA","TAS"'))
    
            Data Elements:
    
            ^DIS71 = RP:AddressDetails:AddressDetails.StateOrTerritory.Code WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.DIS71) != true && !(IsMatch(report.DIS71, @"^(ACT|NSW|NT|QLD|SA|VIC|WA|TAS)$",RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500141", Severity = ProcessMessageSeverity.Error,
                    Description = @"For an Australian address, state or territory must be one of: ACT, NSW, NT, QLD, SA, VIC, WA, TAS",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.StateOrTerritory.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.500141"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "DIS71", Value = GetValueOrEmpty(report.DIS71) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.500141



            foreach (ProcessMessageDocument currentProcessMessage in response)
            {
                if (currentProcessMessage.Parameters != null)
                {
                    foreach (ProcessMessageParameter currentParameter in currentProcessMessage.Parameters)
                    {
                        if (string.IsNullOrEmpty(currentParameter.Name))
                        {
                            currentParameter.Name = _emptyParameterValue;
                        }

                        if (currentParameter.Name.Length > _maxParameterNameLength)
                        {
                            currentParameter.Name = currentParameter.Name.Substring(0, _maxParameterNameLength - 1);
                        }

                        if (string.IsNullOrEmpty(currentParameter.Value))
                        {
                            currentParameter.Value = _emptyParameterValue;
                        }

                        if (currentParameter.Value.Length > _maxParameterValueLength)
                        {
                            currentParameter.Value = currentParameter.Value.Substring(0, _maxParameterValueLength - 1);
                        }
                    }
                }
            }

            return response;
        }


    }
} 