
using System;
using System.IO;
using System.Runtime.CompilerServices;
using System.Text;
using System.Xml;
using System.Linq;
using System.Collections.Generic;
using DataContracts;
using System.Text.RegularExpressions;

namespace Ato.EN.IntegrationServices.CodeGenerationLS
{

    public partial class LS2017Validator
    {
        /// <summary>
        /// The max parameter name length is restricted by the SBR1 and SBR2 schemas - the lowest common denominator of SBR 1 is used as the default
        /// </summary>
        private int _maxParameterNameLength;

        /// <summary>
        /// The max parameter value length is restricted by the SBR1 and SBR2 schemas
        /// </summary>
        private int _maxParameterValueLength;

        /// <summary>
        /// The SBR1 and ebms3 schemas do not allow parameter names or values to be the empty string
        /// </summary>
        private string _emptyParameterValue;

        /// <summary>
        /// Initializes a new instance of the <see cref="LS2017Validator" /> class.
        /// </summary>
        /// <param name="maxParameterNameLength">Maximum length of the parameter name.</param>
        /// <param name="maxParameterValueLength">Maximum length of the parameter value.</param>
        /// <param name="emptyParameterValue">This value will be used in place of any parameter values that result in a null or empty value.</param>
        public LS2017Validator(int maxParameterNameLength = 20, int maxParameterValueLength = 4096, string emptyParameterValue = "EMPTY")
        {
            _maxParameterNameLength = maxParameterNameLength;
            _maxParameterValueLength = maxParameterValueLength;
            _emptyParameterValue = emptyParameterValue;
        }

        #region Functions
        private static IEnumerable<string> Union(IEnumerable<string> list1, IEnumerable<string> list2)
        {
            IEnumerable<string> response;

            if (list1 == null && list2 == null)
            {
                response = null;
            }
            else if (list1 == null)
            {
                response = list2.Distinct();
            }
            else if (list2 == null)
            {
                response = list1.Distinct();
            }
            else
            {
                response = list1.Union(list2);
            }

            return response;
        }

        private static bool IsMatch(int? field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(Convert.ToString(field.Value), expression, options);
        }

        private static bool IsMatch(string field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(field, expression, options);
        }

        // This is just context and tuple counts where they are integer values - easier that changing the parsing logic to just return the value
        private static int Count(int count)
        {
            return count;
        }


        private static int Count<T>(IEnumerable<T> values)
        {
            return values == null ? 0 : values.Where(f => f != null).Count();
        }


        private static int Count<T>(ICollection<T> values)
        {
            return values == null ? 0 : values.Where(f => f != null).Count();
        }


        private static bool exists(bool value)
        {
            return value;
        }

        private static string GetValueOrEmpty(bool? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        private static string GetValueOrEmpty(DateTime? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

         /// <summary>
         /// Get string value between [first] a and [last] b.
         /// </summary>
        public static string Between(string value, string a, string b)
        {
            int posA = value.IndexOf(a);
            int posB = value.LastIndexOf(b);
            if (posA == -1)
            {
                return "";
            }
            if (posB == -1)
            {
                return "";
            }
            int adjustedPosA = posA + a.Length;
            if (adjustedPosA >= posB)
            {
                return "";
            }
            return value.Substring(adjustedPosA, posB - adjustedPosA);
        }

         /// <summary>
         /// Get string value after [first] a.
         /// </summary>
        public static string Before(string value, string a)
        {
            int posA = value.IndexOf(a);
            if (posA == -1)
            {
                return "";
            }
            return value.Substring(0, posA);
        }

         /// <summary>
         /// Get string value after [last] a.
         /// </summary>
        public static string After(string value, string a)
        {
            int posA = value.LastIndexOf(a);
            if (posA == -1)
            {
                return "";
            }
            int adjustedPosA = posA + a.Length;
            if (adjustedPosA >= value.Length)
            {
                return "";
            }
            return value.Substring(adjustedPosA);
        }

        private static int Length(object field)
        {
            if (field == null)
            return 0;
            else
            return field.ToString().Trim().Length;
        }

        private static bool NotSameValues(IEnumerable<object> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool NotSameValues(params object[] nodes)
        {
            if (nodes == null)
                return false;

            return ((from x in nodes select x).Distinct().Count() == nodes.Count());
        }

        private static bool HasDuplicateValues(IEnumerable<object> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool HasDuplicateValues(params object[] nodes)
        {
            if (nodes == null)
                return false;

            return !((from x in nodes select x).Distinct().Count() == nodes.Count());
        }

        private int DuplicateValueIndex(IEnumerable<object> values)
        {
            int response = 0;
            var hashset = new HashSet<object>();
            foreach (var value in values)
            {
                if (!hashset.Add(value))
                {
                    return response;
                }
                response++;
            }
            return response;
        }

        private int DuplicateValueIndex<T>(IEnumerable<T?> values) where T : struct
        {
            int response = 0;
            var hashset = new HashSet<T?>();
            foreach (var value in values)
            {
                if (!hashset.Add(value))
                {
                    return response;
                }
                response++;
            }
            return response;
        }

        private static bool IsDate(object value)
        {
            DateTime dateValue;
            return (value != null && DateTime.TryParse(value.ToString(), out dateValue));
        }

        public DateTime AsDate(string dateAsString)
        {
            DateTime response = DateTime.MinValue;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date;
            }

            return response;
        }

        public DateTime? ConvertToDate(int day, int month, int year)
        {
            return ConvertToDate(day == 0 ? null : day.ToString(), month == 0 ? null : month.ToString(), year == 0 ? null : year.ToString());
        }

        public DateTime? ConvertToDate(string day, string month, string year)
        {
            DateTime? response;
            DateTime result;

            if (year == null || month == null || day == null)
            {
                return null;
            }
            string dateAsString = year + "-" + After(month, "--") + "-" + After(day, "---");
            if (DateTime.TryParse(dateAsString, out result))
            {
                response = result;
            }
            else
            {
                response = null;
            }
            return response;
        }


        public DateTime? ConvertToDate(string day, string month, int year)
        {
            DateTime? response;
            DateTime result;

            if (year == 0 || month == null || day == null)
            {
                return null;
            }
            string dateAsString = year.ToString() + "-" + After(month, "--") + "-" + After(day, "---");
            if (DateTime.TryParse(dateAsString, out result))
            {
                response = result;
            }
            else
            {
                response = null;
            }
            return response;
        }


        private static int Day(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Day;
            }

            return response;
        }


        private static int Day(DateTime? date)
        {
            if (date == null)
                return 0;
            else
                return date.Value.Day;

        }


        private static string Month(string dateAsString)
        {
            string response = null;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.ToString("MMMM");
            }
            else
            {
                return "NotAMonth";
            }

            return response;
        }

        private static string Month(DateTime? date)
        {
            if (date == null)
                return "NotAMonth";
            else
                return date.Value.ToString("MMMM");
        }


        private static int MonthAsInt(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Month;
            }

            return response;
        }

        private static int MonthAsInt(DateTime? date)
        {
            if (date == null)
            {
                return 0;
            }
            return date.Value.Month;
        }


        private static int Year(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Year;
            }

            return response;
        }

        private static int Year(DateTime? date)
        {
            if (date == null)
                return 0;
            else
                return date.Value.Year;
        }


        private static int CurrentFinancialYear()
        {
            return DateToFinancialYear(DateTime.Now, 7);
        }

        private static int FinancialYear(string dateAsString)
        {
            return DateToFinancialYear(dateAsString, 7);
        }

        private static int FinancialYear(DateTime? date)
        {
            return DateToFinancialYear(date, 7);
        }

        private static int DateToFinancialYear(string dateAsString, int startingMonth)
        {
            int response = 0;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
            {
                response = DateToFinancialYear(date, startingMonth);
            }

            return response;
        }
        private static int DateToFinancialYear(DateTime? date, int startingMonth)
        {
            int response;
            if (date == null)
            {
                response = 0;
            }
            else
            {
                int year = date.Value.Year;
                int month = date.Value.Month;

                if (startingMonth > month)
                    response = year;
                else
                    response = year + 1;
            }
            return response;
        }


        private static int FBTYear(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
                response = FBTYear(date);
            return response;
        }

        private static int FBTYear(DateTime? date)
        {
            if (date == null)
            {
                return 0;
            }
            else
            {
                if (date.Value.Month > 3)
                    return date.Value.Year + 1;
                else
                    return date.Value.Year;
            }
        }


        private static bool IsNumeric(object value)
        {
            decimal numbervalue;
            return (value != null && decimal.TryParse(value.ToString(), out numbervalue));
        }

        private static bool NotMonetary(decimal? field, string sign, int digits, int decimals)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                string decimalExpression;
                int digitsToUse = digits - decimals;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                if (decimals > 0)
                    decimalExpression = @"(\.\d{1," + decimals + "})?$";
                else
                    decimalExpression = @"$";

                return !(Regex.IsMatch(field.Value.ToString("0.#########################"), signExpression + @"\d{1," + digitsToUse + "}" + decimalExpression));
            }
        }

        private static bool NotNumeric(decimal? field, string sign, int digits, int decimals)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                string decimalExpression;
                int digitsToUse = digits - decimals;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                if (decimals > 0)
                    decimalExpression = @"(\.\d{1," + decimals + "})?$";
                else
                    decimalExpression = @"$";

                return !(Regex.IsMatch(field.Value.ToString("0.#########################"), signExpression + @"\d{1," + digitsToUse + "}" + decimalExpression));
            }
        }

        private static bool NotNumeric(int? field, string sign, int digits, int decimals = 0)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                return !(Regex.IsMatch(field.Value.ToString(), signExpression + @"\d{1," + digits + "}$"));
            }
        }

        private static bool NotNumeric(long? field, string sign, int digits, int decimals = 0)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                return !(Regex.IsMatch(field.Value.ToString(), signExpression + @"\d{1," + digits + "}$"));
            }
        }


        private static bool OutsideRange(decimal field, decimal expression, int range)
        {
            bool response;

            response = (field < (expression - range)) || (field > (expression + range));

            return response;
        }


        private static bool FailsUSIAlgorithm(string usi, string abn)
        {
            bool response;
            if (usi == null || abn == null)
            {
                response = false;
            }
            else
            {
                usi = usi.Trim();
                abn = abn.Trim();
                if (usi.Length < 13 || abn.Length < 11)
                {
                    response = false;
                }
                else
                {
                    int numeric;
                    if (usi.Substring(0, 11) == abn && int.TryParse(usi.Substring(11, 2), out numeric))
                        response = false;
                    else if (Regex.IsMatch(usi, @"^[a-zA-Z]{3}\d{4}[a-zA-Z]{2}"))
                        response = false;
                    else
                        response = true;
                }
            }
            return response;
        }


        private static bool FailsTANAlgorithm(string tan)
        {
            bool response;
            decimal decimalTan;

            if (tan == null)
                return false;

            tan = tan.Trim();

            if (!decimal.TryParse(tan, out decimalTan))
                return true;

            if (tan.Length != 8)
                return true;

            decimal tanSum =
                7 * int.Parse(tan.Substring(0, 1)) +
                9 * int.Parse(tan.Substring(1, 1)) +
                8 * int.Parse(tan.Substring(2, 1)) +
                4 * int.Parse(tan.Substring(3, 1)) +
                6 * int.Parse(tan.Substring(4, 1)) +
                3 * int.Parse(tan.Substring(5, 1)) +
                5 * int.Parse(tan.Substring(6, 1)) +
                1 * int.Parse(tan.Substring(7, 1));

            if ((tanSum % 11) == 0)
                response = false;
            else
                response = true;

            return response;
        }


        private static bool FailsABNAlgorithm(string abn)
        {
            bool response;
            decimal decimalAbn;

            if (abn == null)
                return false;

            abn = abn.Trim();

            if (!decimal.TryParse(abn, out decimalAbn))
                return true;

            if (abn.Length != 11)
                return true;

            decimal abnSum =
                10 * (int.Parse(abn.Substring(0, 1)) - 1) +
                1 * int.Parse(abn.Substring(1, 1)) +
                3 * int.Parse(abn.Substring(2, 1)) +
                5 * int.Parse(abn.Substring(3, 1)) +
                7 * int.Parse(abn.Substring(4, 1)) +
                9 * int.Parse(abn.Substring(5, 1)) +
                11 * int.Parse(abn.Substring(6, 1)) +
                13 * int.Parse(abn.Substring(7, 1)) +
                15 * int.Parse(abn.Substring(8, 1)) +
                17 * int.Parse(abn.Substring(9, 1)) +
                19 * int.Parse(abn.Substring(10, 1));

            if ((abnSum % 89) == 0)
                response = false;
            else
                response = true;

            return response;
        }
        private static bool FailsACNAlgorithm(string acn)
        {
            bool response;
            decimal decimalAbn;
            if (acn == null)
                return false;

            acn = acn.Trim();

            if (!decimal.TryParse(acn, out decimalAbn))
                return true;

            if (acn.Length != 9)
                return true;

            decimal abnSum =
                8 * int.Parse(acn.Substring(0, 1)) +
                7 * int.Parse(acn.Substring(1, 1)) +
                6 * int.Parse(acn.Substring(2, 1)) +
                5 * int.Parse(acn.Substring(3, 1)) +
                4 * int.Parse(acn.Substring(4, 1)) +
                3 * int.Parse(acn.Substring(5, 1)) +
                2 * int.Parse(acn.Substring(6, 1)) +
                1 * int.Parse(acn.Substring(7, 1));

            decimal checkDigit = int.Parse(acn.Substring(8, 1));
            decimal acnRemainder = abnSum % 10;

            if (((10 - acnRemainder) % 10) == checkDigit)
                response = false;
            else
                response = true;

            return response;
        }


        private static bool FailsTFNAlgorithm(string tfn)
        {
            bool response;
            decimal decimalTfn;

            if (tfn == null)
                return false;

            tfn = tfn.Trim();
            tfn = Regex.Replace(tfn, "^0+", "");

            if (!decimal.TryParse(tfn, out decimalTfn))
                return true;

            if (tfn.Length < 8)
                return true;


            decimal tfn1To7Sum =
                1 * int.Parse(tfn.Substring(0, 1)) +
                4 * int.Parse(tfn.Substring(1, 1)) +
                3 * int.Parse(tfn.Substring(2, 1)) +
                7 * int.Parse(tfn.Substring(3, 1)) +
                5 * int.Parse(tfn.Substring(4, 1)) +
                8 * int.Parse(tfn.Substring(5, 1)) +
                6 * int.Parse(tfn.Substring(6, 1));

            decimal tfn8 = 9 * int.Parse(tfn.Substring(7, 1));

            if (tfn.Length == 8)
            {
                decimal tFNLg8WSum9 = 10 * int.Parse(tfn.Substring(7, 1));
                decimal tFNLg8WSum = tfn1To7Sum + tFNLg8WSum9;

                if ((tFNLg8WSum % 11) == 0)
                    response = false;
                else
                    response = true;
            }
            else if (tfn.Length == 9)
            {
                decimal tfn9 = 10 * int.Parse(tfn.Substring(8, 1));
                decimal tFNLg9WSum = tfn1To7Sum + tfn8 + tfn9;

                if ((tFNLg9WSum % 11) == 0)
                    response = false;
                else
                    response = true;
            }
            else
            {
                response = true;
            }

            return response;
        }


        private static decimal ConditionalValue(bool expression, decimal? trueVal, decimal? falseVal)
        {
             return expression ? trueVal.GetValueOrDefault() : falseVal.GetValueOrDefault();
        }

        private static decimal AsNumeric(string value)
        {
             decimal numberValue;
             decimal.TryParse(value, out numberValue);
             return numberValue;
        }

        private static bool RegexMatch(int? field, string expression, string flags="")
        {
            return IsMatch(field, expression, GetRegexOption(flags));
        }

        private static bool RegexMatch(string field, string expression, string flags="")
        {
            return IsMatch(field, expression, GetRegexOption(flags));
        }

        private static RegexOptions GetRegexOption(string flags)
        {
            RegexOptions options = RegexOptions.None;

            char[] characters = flags.ToCharArray();

            foreach (char character in characters)
            {
                switch (character)
                {
                    case 'i':                        
                        options = options | RegexOptions.IgnoreCase;                       
                        break;
                    case 'm':                        
                        options = options | RegexOptions.Multiline;                       
                        break;
                    case 's':                        
                            options = options | RegexOptions.Singleline;                       
                        break;
                    case 'n':                        
                            options = options | RegexOptions.ExplicitCapture;                       
                        break;
                    case 'x':                       
                            options = options | RegexOptions.IgnorePatternWhitespace;                       
                        break;
                }
            }

            return options;
        }

        /// <summary>
        /// Returns an occurrence index as [occurrenceIndex] of occurrenceIndex > 0, otherwise the empty string
        /// </summary>
        /// <param name="occurrenceIndex">Index of the occurrence.</param>
        /// <returns>Occurrence in XPath [#] format</returns>
        public string OccurrenceIndex(int occurrenceIndex)
        {
            return occurrenceIndex > 0 ? "[" + occurrenceIndex + "]" : "";
        }

        #endregion // Functions

        public LS2017 ConsumedReport { get; private set; }

        private static Dictionary<string, ProcessMessageDocument> _processMessageDocuments = new Dictionary<string,ProcessMessageDocument>();

        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public List<ProcessMessageDocument> ValidateReport(LS2017 report)
        {

            List<ProcessMessageDocument> response = new List<ProcessMessageDocument>();
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            bool assertion;

            this.ConsumedReport = report;

            // Validations can now be done in code rather than just schematron.

    
            #region VR.ATO.GEN.000209
    
            /*  VR.ATO.GEN.000209
            Invalid context. The number of Reporting party contexts must equal 1
    
            Legacy Rule Format:
            IF COUNT(RP) <> 1
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) <> 1
    
            Data Elements:
    
            ^Context = RP
            */
            assertion = (Count(report.RPCount) != 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430296", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Reporting party contexts must equal 1",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.000209"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.000209
    
            #region VR.ATO.GEN.001021
    
            /*  VR.ATO.GEN.001021
            Entity Identifier Scheme is not set to http://www.ato.gov.au/tfn
    
            Legacy Rule Format:
            IF Identifier Scheme <> “http://www.ato.gov.au/tfn”
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP
    
            ^Context = RP
            */
            assertion = (report.RPExists != false && report.RPIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001021", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid identifier scheme.",
                    LongDescription = @"Invalid identifier scheme.  Indentifier scheme must be set to ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001021"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001021
    
            #region VR.ATO.GEN.001021
    
            /*  VR.ATO.GEN.001021
            Entity Identifier Scheme is not set to http://www.ato.gov.au/tfn
    
            Legacy Rule Format:
            IF Identifier Scheme <> “http://www.ato.gov.au/tfn”
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.ALL
    
            ^Context = RP:RP.ALL
            */
            assertion = (report.RPALLExists != false && report.RPALLIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001021", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid identifier scheme.",
                    LongDescription = @"Invalid identifier scheme.  Indentifier scheme must be set to ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPALLOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001021"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001021
    
            #region VR.ATO.GEN.001021
    
            /*  VR.ATO.GEN.001021
            Entity Identifier Scheme is not set to http://www.ato.gov.au/tfn
    
            Legacy Rule Format:
            IF Identifier Scheme <> “http://www.ato.gov.au/tfn”
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP.ALL(Instant)
    
            ^Context = RP.ALL(Instant)
            */
            assertion = (report.RPALLInstantExists != false && report.RPALLInstantIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001021", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid identifier scheme.",
                    LongDescription = @"Invalid identifier scheme.  Indentifier scheme must be set to ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPALLInstantOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001021"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001021
    
            #region VR.ATO.GEN.001021
    
            /*  VR.ATO.GEN.001021
            Entity Identifier Scheme is not set to http://www.ato.gov.au/tfn
    
            Legacy Rule Format:
            IF Identifier Scheme <> “http://www.ato.gov.au/tfn”
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.Y0
    
            ^Context = RP:RP.Y0
            */
            assertion = (report.RPY0Exists != false && report.RPY0IdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001021", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid identifier scheme.",
                    LongDescription = @"Invalid identifier scheme.  Indentifier scheme must be set to ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0OccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001021"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001021
    
            #region VR.ATO.GEN.001021
    
            /*  VR.ATO.GEN.001021
            Entity Identifier Scheme is not set to http://www.ato.gov.au/tfn
    
            Legacy Rule Format:
            IF Identifier Scheme <> “http://www.ato.gov.au/tfn”
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP.Y0(Instant)
    
            ^Context = RP.Y0(Instant)
            */
            assertion = (report.RPY0InstantExists != false && report.RPY0InstantIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001021", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid identifier scheme.",
                    LongDescription = @"Invalid identifier scheme.  Indentifier scheme must be set to ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0InstantOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001021"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001021
    
            #region VR.ATO.GEN.001021
    
            /*  VR.ATO.GEN.001021
            Entity Identifier Scheme is not set to http://www.ato.gov.au/tfn
    
            Legacy Rule Format:
            IF Identifier Scheme <> “http://www.ato.gov.au/tfn”
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.Y0-1
    
            ^Context = RP:RP.Y0-1
            */
            assertion = (report.RPY0Minus1Exists != false && report.RPY0Minus1IdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001021", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid identifier scheme.",
                    LongDescription = @"Invalid identifier scheme.  Indentifier scheme must be set to ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0Minus1OccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001021"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001021
    
            #region VR.ATO.GEN.001021
    
            /*  VR.ATO.GEN.001021
            Entity Identifier Scheme is not set to http://www.ato.gov.au/tfn
    
            Legacy Rule Format:
            IF Identifier Scheme <> “http://www.ato.gov.au/tfn”
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP.Y0-1(Instant)
    
            ^Context = RP.Y0-1(Instant)
            */
            assertion = (report.RPY0Minus1InstantExists != false && report.RPY0Minus1InstantIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001021", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid identifier scheme.",
                    LongDescription = @"Invalid identifier scheme.  Indentifier scheme must be set to ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0Minus1InstantOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001021"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001021
    
            #region VR.ATO.GEN.001021
    
            /*  VR.ATO.GEN.001021
            Entity Identifier Scheme is not set to http://www.ato.gov.au/tfn
    
            Legacy Rule Format:
            IF Identifier Scheme <> “http://www.ato.gov.au/tfn”
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.Y0-2
    
            ^Context = RP:RP.Y0-2
            */
            assertion = (report.RPY0Minus2Exists != false && report.RPY0Minus2IdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001021", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid identifier scheme.",
                    LongDescription = @"Invalid identifier scheme.  Indentifier scheme must be set to ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0Minus2OccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001021"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001021
    
            #region VR.ATO.GEN.001021
    
            /*  VR.ATO.GEN.001021
            Entity Identifier Scheme is not set to http://www.ato.gov.au/tfn
    
            Legacy Rule Format:
            IF Identifier Scheme <> “http://www.ato.gov.au/tfn”
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP.Y0-2(Instant)
    
            ^Context = RP.Y0-2(Instant)
            */
            assertion = (report.RPY0Minus2InstantExists != false && report.RPY0Minus2InstantIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001021", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid identifier scheme.",
                    LongDescription = @"Invalid identifier scheme.  Indentifier scheme must be set to ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0Minus2InstantOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001021"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001021
    
            #region VR.ATO.GEN.001021
    
            /*  VR.ATO.GEN.001021
            Entity Identifier Scheme is not set to http://www.ato.gov.au/tfn
    
            Legacy Rule Format:
            IF Identifier Scheme <> “http://www.ato.gov.au/tfn”
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.Y0-3
    
            ^Context = RP:RP.Y0-3
            */
            assertion = (report.RPY0Minus3Exists != false && report.RPY0Minus3IdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001021", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid identifier scheme.",
                    LongDescription = @"Invalid identifier scheme.  Indentifier scheme must be set to ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0Minus3OccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001021"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001021
    
            #region VR.ATO.GEN.001021
    
            /*  VR.ATO.GEN.001021
            Entity Identifier Scheme is not set to http://www.ato.gov.au/tfn
    
            Legacy Rule Format:
            IF Identifier Scheme <> “http://www.ato.gov.au/tfn”
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP.Y0-3(Instant)
    
            ^Context = RP.Y0-3(Instant)
            */
            assertion = (report.RPY0Minus3InstantExists != false && report.RPY0Minus3InstantIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001021", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid identifier scheme.",
                    LongDescription = @"Invalid identifier scheme.  Indentifier scheme must be set to ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0Minus3InstantOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001021"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001021
    
            #region VR.ATO.GEN.001021
    
            /*  VR.ATO.GEN.001021
            Entity Identifier Scheme is not set to http://www.ato.gov.au/tfn
    
            Legacy Rule Format:
            IF Identifier Scheme <> “http://www.ato.gov.au/tfn”
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.Y0-4
    
            ^Context = RP:RP.Y0-4
            */
            assertion = (report.RPY0Minus4Exists != false && report.RPY0Minus4IdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001021", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid identifier scheme.",
                    LongDescription = @"Invalid identifier scheme.  Indentifier scheme must be set to ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0Minus4OccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001021"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001021
    
            #region VR.ATO.GEN.001021
    
            /*  VR.ATO.GEN.001021
            Entity Identifier Scheme is not set to http://www.ato.gov.au/tfn
    
            Legacy Rule Format:
            IF Identifier Scheme <> “http://www.ato.gov.au/tfn”
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP.Y0-4(Instant)
    
            ^Context = RP.Y0-4(Instant)
            */
            assertion = (report.RPY0Minus4InstantExists != false && report.RPY0Minus4InstantIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001021", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid identifier scheme.",
                    LongDescription = @"Invalid identifier scheme.  Indentifier scheme must be set to ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0Minus4InstantOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001021"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001021
    
            #region VR.ATO.GEN.001021
    
            /*  VR.ATO.GEN.001021
            Entity Identifier Scheme is not set to http://www.ato.gov.au/tfn
    
            Legacy Rule Format:
            IF Identifier Scheme <> “http://www.ato.gov.au/tfn”
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.Y0-5P
    
            ^Context = RP:RP.Y0-5P
            */
            assertion = (report.RPY0Minus5PExists != false && report.RPY0Minus5PIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001021", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid identifier scheme.",
                    LongDescription = @"Invalid identifier scheme.  Indentifier scheme must be set to ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0Minus5POccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001021"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001021
    
            #region VR.ATO.GEN.001021
    
            /*  VR.ATO.GEN.001021
            Entity Identifier Scheme is not set to http://www.ato.gov.au/tfn
    
            Legacy Rule Format:
            IF Identifier Scheme <> “http://www.ato.gov.au/tfn”
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP.Y0-5P(Instant)
    
            ^Context = RP.Y0-5P(Instant)
            */
            assertion = (report.RPY0Minus5PInstantExists != false && report.RPY0Minus5PInstantIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001021", Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid identifier scheme.",
                    LongDescription = @"Invalid identifier scheme.  Indentifier scheme must be set to ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0Minus5PInstantOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001021"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001021
    
            #region VR.ATO.GEN.001023
    
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.ALL
    
            ^OtherTFN = RP:RP.ALL
    
            ^TFN = RP
            */
            assertion = (report.RPALLExists != false && report.RPALLIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPALLOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001023
    
            #region VR.ATO.GEN.001023
    
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP.ALL(Instant)
    
            ^OtherTFN = RP.ALL(Instant)
    
            ^TFN = RP
            */
            assertion = (report.RPALLInstantExists != false && report.RPALLInstantIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPALLInstantOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001023
    
            #region VR.ATO.GEN.001023
    
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.Y0
    
            ^OtherTFN = RP:RP.Y0
    
            ^TFN = RP
            */
            assertion = (report.RPY0Exists != false && report.RPY0IdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0OccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001023
    
            #region VR.ATO.GEN.001023
    
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP.Y0(Instant)
    
            ^OtherTFN = RP.Y0(Instant)
    
            ^TFN = RP
            */
            assertion = (report.RPY0InstantExists != false && report.RPY0InstantIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0InstantOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001023
    
            #region VR.ATO.GEN.001023
    
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.Y0-1
    
            ^OtherTFN = RP:RP.Y0-1
    
            ^TFN = RP
            */
            assertion = (report.RPY0Minus1Exists != false && report.RPY0Minus1IdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0Minus1OccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001023
    
            #region VR.ATO.GEN.001023
    
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP.Y0-1(Instant)
    
            ^OtherTFN = RP.Y0-1(Instant)
    
            ^TFN = RP
            */
            assertion = (report.RPY0Minus1InstantExists != false && report.RPY0Minus1InstantIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0Minus1InstantOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001023
    
            #region VR.ATO.GEN.001023
    
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.Y0-2
    
            ^OtherTFN = RP:RP.Y0-2
    
            ^TFN = RP
            */
            assertion = (report.RPY0Minus2Exists != false && report.RPY0Minus2IdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0Minus2OccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001023
    
            #region VR.ATO.GEN.001023
    
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP.Y0-2(Instant)
    
            ^OtherTFN = RP.Y0-2(Instant)
    
            ^TFN = RP
            */
            assertion = (report.RPY0Minus2InstantExists != false && report.RPY0Minus2InstantIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0Minus2InstantOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001023
    
            #region VR.ATO.GEN.001023
    
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.Y0-3
    
            ^OtherTFN = RP:RP.Y0-3
    
            ^TFN = RP
            */
            assertion = (report.RPY0Minus3Exists != false && report.RPY0Minus3IdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0Minus3OccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001023
    
            #region VR.ATO.GEN.001023
    
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP.Y0-3(Instant)
    
            ^OtherTFN = RP.Y0-3(Instant)
    
            ^TFN = RP
            */
            assertion = (report.RPY0Minus3InstantExists != false && report.RPY0Minus3InstantIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0Minus3InstantOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001023
    
            #region VR.ATO.GEN.001023
    
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.Y0-4
    
            ^OtherTFN = RP:RP.Y0-4
    
            ^TFN = RP
            */
            assertion = (report.RPY0Minus4Exists != false && report.RPY0Minus4IdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0Minus4OccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001023
    
            #region VR.ATO.GEN.001023
    
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP.Y0-4(Instant)
    
            ^OtherTFN = RP.Y0-4(Instant)
    
            ^TFN = RP
            */
            assertion = (report.RPY0Minus4InstantExists != false && report.RPY0Minus4InstantIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0Minus4InstantOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001023
    
            #region VR.ATO.GEN.001023
    
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.Y0-5P
    
            ^OtherTFN = RP:RP.Y0-5P
    
            ^TFN = RP
            */
            assertion = (report.RPY0Minus5PExists != false && report.RPY0Minus5PIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0Minus5POccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001023
    
            #region VR.ATO.GEN.001023
    
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP.Y0-5P(Instant)
    
            ^OtherTFN = RP.Y0-5P(Instant)
    
            ^TFN = RP
            */
            assertion = (report.RPY0Minus5PInstantExists != false && report.RPY0Minus5PInstantIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023", Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0Minus5PInstantOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.GEN.001023
    
            #region VR.ATO.LS.414000
    
            /*  VR.ATO.LS.414000
            If an amount has been entered for tax losses carried forward in the current income year and the continuity of majority ownership test was not passed in the current income year then an amount must be entered for tax losses carried forward for which the business continuity test must be satisfied before they can be deducted in later years.
    
            Legacy Rule Format:
            IF ([LS22] = NULL OR [LS22] = 0) AND [LS5] >0 AND [LS15] = FALSE
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS5] = LS:RP.Y0:rvctc3.02.00:Tax.Losses.CarriedForward.Total.Amount
            [LS15] = LS:RP.Y0:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS22] = LS:RP:rvctc3.02.34:Tax.Losses.CarriedForward.BusinessContinuityTestPassed.Amount

            Technical Business Rule Format:
            (^LS22 = NULL OR ^LS22 = 0) AND ^LS5 >0 AND ^LS15 = FALSE
    
            Data Elements:
    
            RP.Y0:^LS5 = tns:Tax.Losses.CarriedForward.Total.Amount
    
            RP.Y0:^LS15 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP:^LS22 = tns:Tax.Losses.CarriedForward.BusinessContinuityTestPassed.Amount
            */
            assertion = ((report.LS22 == null || report.LS22 == 0) && report.LS5.GetValueOrDefault() > 0 && report.LS15 == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.LS.414207", Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax losses at ownership and business continuity test must be supplied.",
                    LongDescription = @"If an amount has been entered for tax losses carried forward in the current income year and the continuity of majority ownership test was not passed in the current income year then an amount must be supplied for tax losses carried forward for which the business continuity test must be satisfied before they can be deducted in later years.",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.CarriedForward.Total.Amount[@contextRef='" + report.RPY0Id + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414000"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS22", Value = report.LS22.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS5", Value = report.LS5.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS15", Value = GetValueOrEmpty(report.LS15) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414000
    
            #region VR.ATO.LS.414001
    
            /*  VR.ATO.LS.414001
            The sum of tax losses carried forward to later income years , listed against each income year, must equal the total amount of tax losses carried forward to later income years.
    
            Legacy Rule Format:
            IF ([LS5] <> NULL OR [LS6] <> NULL OR [LS7] <> NULL OR [LS8] <> NULL OR [LS9] <> NULL OR [LS49] <> NULL OR [LS50] <> NULL) AND [LS50] <> [LS5] + [LS6] + [LS7] + [LS8] + [LS9] + [LS49]
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS5] = LS:RP.Y0:rvctc3.02.00:Tax.Losses.CarriedForward.Total.Amount
            [LS6] = LS:RP.Y0-1:rvctc3.02.00:Tax.Losses.CarriedForward.Total.Amount
            [LS7] = LS:RP.Y0-2:rvctc3.02.00:Tax.Losses.CarriedForward.Total.Amount
            [LS8] = LS:RP.Y0-3:rvctc3.02.00:Tax.Losses.CarriedForward.Total.Amount
            [LS9] = LS:RP.Y0-4:rvctc3.02.00:Tax.Losses.CarriedForward.Total.Amount
            [LS49] = LS:RP.Y0-5P:rvctc3.02.00:Tax.Losses.CarriedForward.Total.Amount
            [LS50] = LS:RP.ALL:rvctc3.02.00:Tax.Losses.CarriedForward.Total.Amount

            Technical Business Rule Format:
            (^LS5 <> NULL OR ^LS6 <> NULL OR ^LS7 <> NULL OR ^LS8 <> NULL OR ^LS9 <> NULL OR ^LS49 <> NULL OR ^LS50 <> NULL) AND ^LS50 <> ^LS5 + ^LS6 + ^LS7 + ^LS8 + ^LS9 + ^LS49
    
            Data Elements:
    
            RP.ALL:^LS50 = tns:Tax.Losses.CarriedForward.Total.Amount
    
            RP.Y0:^LS5 = tns:Tax.Losses.CarriedForward.Total.Amount
    
            RP.Y0-1:^LS6 = tns:Tax.Losses.CarriedForward.Total.Amount
    
            RP.Y0-2:^LS7 = tns:Tax.Losses.CarriedForward.Total.Amount
    
            RP.Y0-3:^LS8 = tns:Tax.Losses.CarriedForward.Total.Amount
    
            RP.Y0-4:^LS9 = tns:Tax.Losses.CarriedForward.Total.Amount
    
            RP.Y0-5P:^LS49 = tns:Tax.Losses.CarriedForward.Total.Amount
            */
            assertion = ((report.LS5 != null || report.LS6 != null || report.LS7 != null || report.LS8 != null || report.LS9 != null || report.LS49 != null || report.LS50 != null) && report.LS50.GetValueOrDefault() != report.LS5.GetValueOrDefault() + report.LS6.GetValueOrDefault() + report.LS7.GetValueOrDefault() + report.LS8.GetValueOrDefault() + report.LS9.GetValueOrDefault() + report.LS49.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.LS.414001", Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax losses carried forward to later income years - Total is incorrect",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.CarriedForward.Total.Amount[@contextRef='" + report.RPALLId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414001"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS5", Value = report.LS5.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS6", Value = report.LS6.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS7", Value = report.LS7.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS8", Value = report.LS8.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS9", Value = report.LS9.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS49", Value = report.LS49.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS50", Value = report.LS50.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414001
    
            #region VR.ATO.LS.414002
    
            /*  VR.ATO.LS.414002
            If an amount has been entered for tax losses carried forward in the current year - 1 income year and the continuity of majority ownership test was not passed in the current year - 1 income year or in later years then an amount must be entered for tax losses carried forward for which the business continuity test must be satisfied before they can be deducted in later years.
    
            Legacy Rule Format:
            IF ([LS22] = NULL OR [LS22] = 0) AND [LS6]>0 AND ([LS15] = FALSE OR [LS16] = FALSE)
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS6] = LS:RP.Y0-1:rvctc3.02.00:Tax.Losses.CarriedForward.Total.Amount
            [LS15] = LS:RP.Y0:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS16] = LS:RP.Y0-1:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS22] = LS:RP:rvctc3.02.34:Tax.Losses.CarriedForward.BusinessContinuityTestPassed.Amount

            Technical Business Rule Format:
            (^LS22 = NULL OR ^LS22 = 0) AND ^LS6>0 AND (^LS15 = FALSE OR ^LS16 = FALSE)
    
            Data Elements:
    
            RP.Y0-1:^LS6 = tns:Tax.Losses.CarriedForward.Total.Amount
    
            RP.Y0:^LS15 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP.Y0-1:^LS16 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP:^LS22 = tns:Tax.Losses.CarriedForward.BusinessContinuityTestPassed.Amount
            */
            assertion = ((report.LS22 == null || report.LS22 == 0) && report.LS6.GetValueOrDefault() > 0 && (report.LS15 == false || report.LS16 == false));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.LS.414211", Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax losses at ownership and business continuity test must be supplied.",
                    LongDescription = @"If an amount has been entered for tax losses carried forward in the current year - 1 income year and the continuity of majority ownership test was not passed in the current year - 1 income year or in later years then an amount must be supplied for tax losses carried forward for which the business continuity test must be satisfied before they can be deducted in later years.",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.CarriedForward.Total.Amount[@contextRef='" + report.RPY0Minus1Id + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414002"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS22", Value = report.LS22.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS6", Value = report.LS6.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS15", Value = GetValueOrEmpty(report.LS15) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS16", Value = GetValueOrEmpty(report.LS16) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414002
    
            #region VR.ATO.LS.414003
    
            /*  VR.ATO.LS.414003
            If an amount has been entered for tax losses carried forward in the current year - 2 income year and the continuity of majority ownership test was not passed in the current year - 2 income year or in later years then an amount must be entered for tax losses carried forward for which the business continuity test must be satisfied before they can be deducted in later years.
    
            Legacy Rule Format:
            IF ([LS22] = NULL OR [LS22] = 0) AND [LS7] > 0 AND ([LS15] = FALSE OR [LS16] = FALSE OR [LS17] = FALSE)
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS7] = LS:RP.Y0-2:rvctc3.02.00:Tax.Losses.CarriedForward.Total.Amount
            [LS15] = LS:RP.Y0:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS16] = LS:RP.Y0-1:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS17] = LS:RP.Y0-2:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS22] = LS:RP:rvctc3.02.34:Tax.Losses.CarriedForward.BusinessContinuityTestPassed.Amount

            Technical Business Rule Format:
            (^LS22 = NULL OR ^LS22 = 0) AND ^LS7 > 0 AND (^LS15 = FALSE OR ^LS16 = FALSE OR ^LS17 = FALSE)
    
            Data Elements:
    
            RP.Y0-2:^LS7 = tns:Tax.Losses.CarriedForward.Total.Amount
    
            RP.Y0:^LS15 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP.Y0-1:^LS16 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP.Y0-2:^LS17 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP:^LS22 = tns:Tax.Losses.CarriedForward.BusinessContinuityTestPassed.Amount
            */
            assertion = ((report.LS22 == null || report.LS22 == 0) && report.LS7.GetValueOrDefault() > 0 && (report.LS15 == false || report.LS16 == false || report.LS17 == false));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.LS.414213", Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax losses at ownership and business continuity test must be supplied.",
                    LongDescription = @"If an amount has been entered for tax losses carried forward in the current year - 2 income year and the continuity of majority ownership test was not passed in the current year - 2 income year or in later years then an amount must be supplied for tax losses carried forward for which the business continuity test must be satisfied before they can be deducted in later years.",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.CarriedForward.Total.Amount[@contextRef='" + report.RPY0Minus2Id + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414003"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS22", Value = report.LS22.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS7", Value = report.LS7.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS15", Value = GetValueOrEmpty(report.LS15) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS16", Value = GetValueOrEmpty(report.LS16) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS17", Value = GetValueOrEmpty(report.LS17) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414003
    
            #region VR.ATO.LS.414004
    
            /*  VR.ATO.LS.414004
            If an amount has been entered for tax losses carried forward in the current year - 3 income year and the continuity of majority ownership test was not passed in the current year - 3 income year or in later years then an amount must be entered for tax losses carried forward for which the business continuity test must be satisfied before they can be deducted in later years.
    
            Legacy Rule Format:
            IF ([LS22] = NULL OR [LS22] = 0) AND [LS8] >0 AND ([LS15] = FALSE OR [LS16] = FALSE OR [LS17] = FALSE OR [LS18] = FALSE)
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS8] = LS:RP.Y0-3:rvctc3.02.00:Tax.Losses.CarriedForward.Total.Amount
            [LS15] = LS:RP.Y0:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS16] = LS:RP.Y0-1:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS17] = LS:RP.Y0-2:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS18] = LS:RP.Y0-3:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS22] = LS:RP:rvctc3.02.34:Tax.Losses.CarriedForward.BusinessContinuityTestPassed.Amount

            Technical Business Rule Format:
            (^LS22 = NULL OR ^LS22 = 0) AND ^LS8 >0 AND (^LS15 = FALSE OR ^LS16 = FALSE OR ^LS17 = FALSE OR ^LS18 = FALSE)
    
            Data Elements:
    
            RP.Y0-3:^LS8 = tns:Tax.Losses.CarriedForward.Total.Amount
    
            RP.Y0:^LS15 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP.Y0-1:^LS16 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP.Y0-2:^LS17 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP.Y0-3:^LS18 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP:^LS22 = tns:Tax.Losses.CarriedForward.BusinessContinuityTestPassed.Amount
            */
            assertion = ((report.LS22 == null || report.LS22 == 0) && report.LS8.GetValueOrDefault() > 0 && (report.LS15 == false || report.LS16 == false || report.LS17 == false || report.LS18 == false));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.LS.414215", Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax losses at ownership and business continuity test must be supplied.",
                    LongDescription = @"If an amount has been entered for tax losses carried forward in the current year - 3 income year and the continuity of majority ownership test was not passed in the current year - 3 income year or in later years then an amount must be supplied for tax losses carried forward for which the business continuity test must be satisfied before they can be deducted in later years.",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.CarriedForward.Total.Amount[@contextRef='" + report.RPY0Minus3Id + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414004"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS22", Value = report.LS22.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS8", Value = report.LS8.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS15", Value = GetValueOrEmpty(report.LS15) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS16", Value = GetValueOrEmpty(report.LS16) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS17", Value = GetValueOrEmpty(report.LS17) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS18", Value = GetValueOrEmpty(report.LS18) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414004
    
            #region VR.ATO.LS.414005
    
            /*  VR.ATO.LS.414005
            If an amount has been entered for tax losses carried forward in the current year - 4 income year and the continuity of majority ownership test was not passed in the current year - 4 income year or in later years then an amount must be entered for tax losses carried forward for which the business continuity test must be satisfied before they can be deducted in later years.
    
            Legacy Rule Format:
            IF ([LS22] = NULL OR [LS22] = 0) AND [LS9]>0 AND ([LS15] = FALSE OR [LS16] = FALSE OR [LS17] = FALSE OR [LS18] = FALSE OR [LS19] = FALSE)
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS9] = LS:RP.Y0-4:rvctc3.02.00:Tax.Losses.CarriedForward.Total.Amount
            [LS15] = LS:RP.Y0:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS16] = LS:RP.Y0-1:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS17] = LS:RP.Y0-2:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS18] = LS:RP.Y0-3:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS19] = LS:RP.Y0-4:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS22] = LS:RP:rvctc3.02.34:Tax.Losses.CarriedForward.BusinessContinuityTestPassed.Amount

            Technical Business Rule Format:
            (^LS22 = NULL OR ^LS22 = 0) AND ^LS9>0 AND (^LS15 = FALSE OR ^LS16 = FALSE OR ^LS17 = FALSE OR ^LS18 = FALSE OR ^LS19 = FALSE)
    
            Data Elements:
    
            RP.Y0-4:^LS9 = tns:Tax.Losses.CarriedForward.Total.Amount
    
            RP.Y0:^LS15 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP.Y0-1:^LS16 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP.Y0-2:^LS17 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP.Y0-3:^LS18 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP.Y0-4:^LS19 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP:^LS22 = tns:Tax.Losses.CarriedForward.BusinessContinuityTestPassed.Amount
            */
            assertion = ((report.LS22 == null || report.LS22 == 0) && report.LS9.GetValueOrDefault() > 0 && (report.LS15 == false || report.LS16 == false || report.LS17 == false || report.LS18 == false || report.LS19 == false));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.LS.414217", Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax losses at ownership and business continuity test must be supplied.",
                    LongDescription = @"If an amount has been entered for tax losses carried forward in the current year - 4 income year and the continuity of majority ownership test was not passed in the current year - 4 income year or in later years then an amount must be supplied for tax losses carried forward for which the business continuity test must be satisfied before they can be deducted in later years.",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.CarriedForward.Total.Amount[@contextRef='" + report.RPY0Minus4Id + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414005"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS22", Value = report.LS22.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS9", Value = report.LS9.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS15", Value = GetValueOrEmpty(report.LS15) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS16", Value = GetValueOrEmpty(report.LS16) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS17", Value = GetValueOrEmpty(report.LS17) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS18", Value = GetValueOrEmpty(report.LS18) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS19", Value = GetValueOrEmpty(report.LS19) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414005
    
            #region VR.ATO.LS.414006
    
            /*  VR.ATO.LS.414006
            If an amount has been entered for tax losses carried forward in the current year - 5 income year and the continuity of majority ownership test was not passed in the current year - 5 income year or in later years then an amount must be entered for tax losses carried forward for which the business continuity test must be satisfied before they can be deducted in later years.
    
            Legacy Rule Format:
            IF ([LS22] = NULL OR [LS22] = 0) AND [LS49] > 0 AND ([LS15] = FALSE OR [LS16] = FALSE OR [LS17] = FALSE OR [LS18] = FALSE OR [LS19] = FALSE OR [LS53] = FALSE)
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS15] = LS:RP.Y0:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS16] = LS:RP.Y0-1:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS17] = LS:RP.Y0-2:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS18] = LS:RP.Y0-3:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS19] = LS:RP.Y0-4:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS22] = LS:RP:rvctc3.02.34:Tax.Losses.CarriedForward.BusinessContinuityTestPassed.Amount
            [LS49] = LS:RP.Y0-5P:rvctc3.02.00:Tax.Losses.CarriedForward.Total.Amount
            [LS53] = LS:RP.Y0-5P:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator

            Technical Business Rule Format:
            (^LS22 = NULL OR ^LS22 = 0) AND ^LS49 > 0 AND (^LS15 = FALSE OR ^LS16 = FALSE OR ^LS17 = FALSE OR ^LS18 = FALSE OR ^LS19 = FALSE OR ^LS53 = FALSE)
    
            Data Elements:
    
            RP.Y0-5P:^LS49 = tns:Tax.Losses.CarriedForward.Total.Amount
    
            RP.Y0:^LS15 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP.Y0-1:^LS16 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP.Y0-2:^LS17 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP.Y0-3:^LS18 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP.Y0-4:^LS19 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP:^LS22 = tns:Tax.Losses.CarriedForward.BusinessContinuityTestPassed.Amount
    
            RP.Y0-5P:^LS53 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            */
            assertion = ((report.LS22 == null || report.LS22 == 0) && report.LS49.GetValueOrDefault() > 0 && (report.LS15 == false || report.LS16 == false || report.LS17 == false || report.LS18 == false || report.LS19 == false || report.LS53 == false));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.LS.414219", Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax losses at ownership and business continuity test must be supplied.",
                    LongDescription = @"If an amount has been entered for tax losses carried forward in the current year - 5 income year and the continuity of majority ownership test was not passed in the current year - 5 income year or in later years then an amount must be supplied for tax losses carried forward for which the business continuity test must be satisfied before they can be deducted in later years.",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.CarriedForward.Total.Amount[@contextRef='" + report.RPY0Minus5PId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414006"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS22", Value = report.LS22.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS49", Value = report.LS49.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS15", Value = GetValueOrEmpty(report.LS15) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS16", Value = GetValueOrEmpty(report.LS16) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS17", Value = GetValueOrEmpty(report.LS17) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS18", Value = GetValueOrEmpty(report.LS18) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS19", Value = GetValueOrEmpty(report.LS19) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS53", Value = GetValueOrEmpty(report.LS53) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414006
    
            #region VR.ATO.LS.414007
    
            /*  VR.ATO.LS.414007
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [LS5] <> NULL AND [LS5] <> MONETARY(S,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS5] = LS:RP.Y0:rvctc3.02.00:Tax.Losses.CarriedForward.Total.Amount

            Technical Business Rule Format:
            NotMonetary(^LS5, 'S', 11, 0)
    
            Data Elements:
    
            RP.Y0:^LS5 = tns:Tax.Losses.CarriedForward.Total.Amount
            */
            assertion = NotMonetary(report.LS5, @"S", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.CarriedForward.Total.Amount[@contextRef='" + report.RPY0Id + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414007"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS5", Value = report.LS5.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414007
    
            #region VR.ATO.LS.414008
    
            /*  VR.ATO.LS.414008
            Tax losses carried forward to later income years - Total must be equal to the Total tax losses carried forward to later income years.
    
            Legacy Rule Format:
            IF ([LS50] <> NULL OR [LS74] <> NULL) AND [LS50] <> [LS74]
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS50] = LS:RP.ALL:rvctc3.02.00:Tax.Losses.CarriedForward.Total.Amount
            [LS74] = LS:RP:rvctc3.02.02:Tax.Losses.CarriedForward.LaterIncomeYearsTotal.Amount

            Technical Business Rule Format:
            (^LS50 <> NULL OR ^LS74 <> NULL) AND ^LS50 <> ^LS74
    
            Data Elements:
    
            RP.ALL:^LS50 = tns:Tax.Losses.CarriedForward.Total.Amount
    
            RP:^LS74 = tns:Tax.Losses.CarriedForward.LaterIncomeYearsTotal.Amount
            */
            assertion = ((report.LS50 != null || report.LS74 != null) && report.LS50.GetValueOrDefault() != report.LS74.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.LS.414008", Severity = ProcessMessageSeverity.Error,
                    Description = @"(Tax losses carried forward to later income years, Total) is not equal to Part (F), label (J)",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.CarriedForward.Total.Amount[@contextRef='" + report.RPALLId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414008"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS50", Value = report.LS50.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS74", Value = report.LS74.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414008
    
            #region VR.ATO.LS.414009
    
            /*  VR.ATO.LS.414009
            If an amount has been entered for net capital losses carried forward in the current income year and the continuity of majority ownership test was not passed in the current income year then an amount must be entered for net capital losses carried forward for which the business continuity test must be satisfied before they can be applied in later years.
    
            Legacy Rule Format:
            IF ([LS80] = NULL OR [LS80] = 0) AND [LS10] > 0 AND [LS15] = FALSE
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS10] = LS:RP.Y0(Instant):rvctc3.02.16:Capital.Losses.CarriedForward.Net.Amount
            [LS15] = LS:RP.Y0:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS80] = LS:RP:rvctc3.02.34:Capital.Losses.CarriedForward.BusinessContinuityTestPassedNet.Amount

            Technical Business Rule Format:
            (^LS80 = NULL OR ^LS80 = 0) AND ^LS10 > 0 AND ^LS15 = FALSE
    
            Data Elements:
    
            RP.Y0(Instant):^LS10 = tns:Capital.Losses.CarriedForward.Net.Amount
    
            RP.Y0:^LS15 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP:^LS80 = tns:Capital.Losses.CarriedForward.BusinessContinuityTestPassedNet.Amount
            */
            assertion = ((report.LS80 == null || report.LS80 == 0) && report.LS10.GetValueOrDefault() > 0 && report.LS15 == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.LS.414210", Severity = ProcessMessageSeverity.Error,
                    Description = @"Net capital losses at ownership and business continuity test must be supplied.",
                    LongDescription = @"Net capital losses at ownership and business continuity test must be supplied.
If an amount has been entered for net capital losses carried forward in the current income year and the continuity of majority ownership test was not passed in the current income year then an amount must be entered for net capital losses carried forward for which the business continuity test must be satisfied before they can be applied in later years.",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.CarriedForward.Net.Amount[@contextRef='" + report.RPY0InstantId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414009"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS80", Value = report.LS80.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS10", Value = report.LS10.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS15", Value = GetValueOrEmpty(report.LS15) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414009
    
            #region VR.ATO.LS.414010
    
            /*  VR.ATO.LS.414010
            The sum of net capital losses carried forward to later income years , listed against each income year, must equal the total amount of net capital losses carried forward to later income years.
    
            Legacy Rule Format:
            IF ([LS10] <> NULL OR [LS11] <> NULL OR [LS12] <> NULL OR [LS13] <> NULL OR [LS14] <> NULL OR [LS51] <> NULL OR [LS52] <> NULL) AND [LS52] <> [LS10] + [LS11] + [LS12] + [LS13] + [LS14] + [LS51]
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS10] = LS:RP.Y0(Instant):rvctc3.02.16:Capital.Losses.CarriedForward.Net.Amount
            [LS11] = LS:RP.Y0-1(Instant):rvctc3.02.16:Capital.Losses.CarriedForward.Net.Amount
            [LS12] = LS:RP.Y0-2(Instant):rvctc3.02.16:Capital.Losses.CarriedForward.Net.Amount
            [LS13] = LS:RP.Y0-3(Instant):rvctc3.02.16:Capital.Losses.CarriedForward.Net.Amount
            [LS14] = LS:RP.Y0-4(Instant):rvctc3.02.16:Capital.Losses.CarriedForward.Net.Amount
            [LS51] = LS:RP.Y0-5P(Instant):rvctc3.02.16:Capital.Losses.CarriedForward.Net.Amount
            [LS52] = LS:RP.ALL(Instant):rvctc3.02.16:Capital.Losses.CarriedForward.Net.Amount

            Technical Business Rule Format:
            (^LS10 <> NULL OR ^LS11 <> NULL OR ^LS12 <> NULL OR ^LS13 <> NULL OR ^LS14 <> NULL OR ^LS51 <> NULL OR ^LS52 <> NULL) AND ^LS52 <> ^LS10 + ^LS11 + ^LS12 + ^LS13 + ^LS14 + ^LS51
    
            Data Elements:
    
            RP.Y0(Instant):^LS10 = tns:Capital.Losses.CarriedForward.Net.Amount
    
            RP.Y0-1(Instant):^LS11 = tns:Capital.Losses.CarriedForward.Net.Amount
    
            RP.Y0-2(Instant):^LS12 = tns:Capital.Losses.CarriedForward.Net.Amount
    
            RP.Y0-3(Instant):^LS13 = tns:Capital.Losses.CarriedForward.Net.Amount
    
            RP.Y0-4(Instant):^LS14 = tns:Capital.Losses.CarriedForward.Net.Amount
    
            RP.Y0-5P(Instant):^LS51 = tns:Capital.Losses.CarriedForward.Net.Amount
    
            RP.ALL(Instant):^LS52 = tns:Capital.Losses.CarriedForward.Net.Amount
            */
            assertion = ((report.LS10 != null || report.LS11 != null || report.LS12 != null || report.LS13 != null || report.LS14 != null || report.LS51 != null || report.LS52 != null) && report.LS52.GetValueOrDefault() != report.LS10.GetValueOrDefault() + report.LS11.GetValueOrDefault() + report.LS12.GetValueOrDefault() + report.LS13.GetValueOrDefault() + report.LS14.GetValueOrDefault() + report.LS51.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.LS.414010", Severity = ProcessMessageSeverity.Error,
                    Description = @"(Net capital losses carried forward to later income years - Total) is incorrect",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.CarriedForward.Net.Amount[@contextRef='" + report.RPY0InstantId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414010"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS10", Value = report.LS10.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS11", Value = report.LS11.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS12", Value = report.LS12.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS13", Value = report.LS13.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS14", Value = report.LS14.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS51", Value = report.LS51.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS52", Value = report.LS52.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414010
    
            #region VR.ATO.LS.414011
    
            /*  VR.ATO.LS.414011
            If an amount has been entered for net capital losses carried forward in the current income year - 1 and the continuity of majority ownership test was not passed in the current income year - 1 income year or in later years then an amount must be entered for net capital losses carried forward for which the business continuity test must be satisfied before they can be applied in later years.
    
            Legacy Rule Format:
            IF ([LS80] = NULL OR [LS80] = 0) AND [LS11] > 0 AND ([LS15] = FALSE OR [LS16] = FALSE)
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS11] = LS:RP.Y0-1(Instant):rvctc3.02.16:Capital.Losses.CarriedForward.Net.Amount
            [LS15] = LS:RP.Y0:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS16] = LS:RP.Y0-1:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS80] = LS:RP:rvctc3.02.34:Capital.Losses.CarriedForward.BusinessContinuityTestPassedNet.Amount

            Technical Business Rule Format:
            (^LS80 = NULL OR ^LS80 = 0) AND ^LS11 > 0 AND (^LS15 = FALSE OR ^LS16 = FALSE)
    
            Data Elements:
    
            RP.Y0-1(Instant):^LS11 = tns:Capital.Losses.CarriedForward.Net.Amount
    
            RP.Y0:^LS15 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP.Y0-1:^LS16 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP:^LS80 = tns:Capital.Losses.CarriedForward.BusinessContinuityTestPassedNet.Amount
            */
            assertion = ((report.LS80 == null || report.LS80 == 0) && report.LS11.GetValueOrDefault() > 0 && (report.LS15 == false || report.LS16 == false));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.LS.414212", Severity = ProcessMessageSeverity.Error,
                    Description = @"Net capital losses at ownership and business continuity test must be supplied.",
                    LongDescription = @"If an amount has been entered for net capital losses carried forward in the current income year - 1 and the continuity of majority ownership test was not passed in the current income year - 1 income year or in later years then an amount must be supplied for net capital losses carried forward for which the business continuity test must be satisfied before they can be applied in later years.",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.CarriedForward.Net.Amount[@contextRef='" + report.RPY0Minus1InstantId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414011"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS80", Value = report.LS80.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS11", Value = report.LS11.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS15", Value = GetValueOrEmpty(report.LS15) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS16", Value = GetValueOrEmpty(report.LS16) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414011
    
            #region VR.ATO.LS.414012
    
            /*  VR.ATO.LS.414012
            If an amount has been entered for net capital losses carried forward in the current income year - 2 and the continuity of majority ownership test was not passed in the current income year - 2 income year or in later years then an amount must be entered for net capital losses carried forward for which the business continuity test must be satisfied before they can be applied in later years.
    
            Legacy Rule Format:
            IF ([LS80] = NULL OR [LS80] = 0) AND [LS12] >0 AND ([LS15] = FALSE OR [LS16] = FALSE OR [LS17] = FALSE)
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS12] = LS:RP.Y0-2(Instant):rvctc3.02.16:Capital.Losses.CarriedForward.Net.Amount
            [LS15] = LS:RP.Y0:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS16] = LS:RP.Y0-1:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS17] = LS:RP.Y0-2:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS80] = LS:RP:rvctc3.02.34:Capital.Losses.CarriedForward.BusinessContinuityTestPassedNet.Amount

            Technical Business Rule Format:
            (^LS80 = NULL OR ^LS80 = 0) AND ^LS12 >0 AND (^LS15 = FALSE OR ^LS16 = FALSE OR ^LS17 = FALSE)
    
            Data Elements:
    
            RP.Y0-2(Instant):^LS12 = tns:Capital.Losses.CarriedForward.Net.Amount
    
            RP.Y0:^LS15 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP.Y0-1:^LS16 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP.Y0-2:^LS17 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP:^LS80 = tns:Capital.Losses.CarriedForward.BusinessContinuityTestPassedNet.Amount
            */
            assertion = ((report.LS80 == null || report.LS80 == 0) && report.LS12.GetValueOrDefault() > 0 && (report.LS15 == false || report.LS16 == false || report.LS17 == false));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.LS.414214", Severity = ProcessMessageSeverity.Error,
                    Description = @"Net capital losses at ownership and business continuity test must be supplied.",
                    LongDescription = @"If an amount has been entered for net capital losses carried forward in the current income year - 2 and the continuity of majority ownership test was not passed in the current income year - 2 income year or in later years then an amount must be supplied for net capital losses carried forward for which the business continuity test must be satisfied before they can be applied in later years.",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.CarriedForward.Net.Amount[@contextRef='" + report.RPY0Minus2InstantId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414012"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS80", Value = report.LS80.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS12", Value = report.LS12.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS15", Value = GetValueOrEmpty(report.LS15) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS16", Value = GetValueOrEmpty(report.LS16) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS17", Value = GetValueOrEmpty(report.LS17) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414012
    
            #region VR.ATO.LS.414013
    
            /*  VR.ATO.LS.414013
            If an amount has been entered for net capital losses carried forward in the current income year - 3 and the continuity of majority ownership test was not passed in the current income year - 3 income year or in later years then an amount must be entered for net capital losses carried forward for which the business continuity test must be satisfied before they can be applied in later years.
    
            Legacy Rule Format:
            IF ([LS80] = NULL OR [LS80] = 0) AND [LS13] >0 AND ([LS15] = FALSE OR [LS16] = FALSE OR [LS17] = FALSE OR [LS18] = FALSE)
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS13] = LS:RP.Y0-3(Instant):rvctc3.02.16:Capital.Losses.CarriedForward.Net.Amount
            [LS15] = LS:RP.Y0:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS16] = LS:RP.Y0-1:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS17] = LS:RP.Y0-2:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS18] = LS:RP.Y0-3:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS80] = LS:RP:rvctc3.02.34:Capital.Losses.CarriedForward.BusinessContinuityTestPassedNet.Amount

            Technical Business Rule Format:
            (^LS80 = NULL OR ^LS80 = 0) AND ^LS13 >0 AND (^LS15 = FALSE OR ^LS16 = FALSE OR ^LS17 = FALSE OR ^LS18 = FALSE)
    
            Data Elements:
    
            RP.Y0-3(Instant):^LS13 = tns:Capital.Losses.CarriedForward.Net.Amount
    
            RP.Y0:^LS15 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP.Y0-1:^LS16 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP.Y0-2:^LS17 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP.Y0-3:^LS18 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP:^LS80 = tns:Capital.Losses.CarriedForward.BusinessContinuityTestPassedNet.Amount
            */
            assertion = ((report.LS80 == null || report.LS80 == 0) && report.LS13.GetValueOrDefault() > 0 && (report.LS15 == false || report.LS16 == false || report.LS17 == false || report.LS18 == false));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.LS.414216", Severity = ProcessMessageSeverity.Error,
                    Description = @"Net capital losses at ownership and business continuity test must be supplied.",
                    LongDescription = @"If an amount has been entered for net capital losses carried forward in the current income year - 3 and the continuity of majority ownership test was not passed in the current income year - 3 income year or in later years then an amount must be supplied for net capital losses carried forward for which the business continuity test must be satisfied before they can be applied in later years.",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.CarriedForward.Net.Amount[@contextRef='" + report.RPY0Minus3InstantId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414013"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS80", Value = report.LS80.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS13", Value = report.LS13.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS15", Value = GetValueOrEmpty(report.LS15) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS16", Value = GetValueOrEmpty(report.LS16) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS17", Value = GetValueOrEmpty(report.LS17) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS18", Value = GetValueOrEmpty(report.LS18) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414013
    
            #region VR.ATO.LS.414014
    
            /*  VR.ATO.LS.414014
            If an amount has been entered for net capital losses carried forward in the current income year - 4 and the continuity of majority ownership test was not passed in the current income year - 4 income year or in later years then an amount must be entered for net capital losses carried forward for which the business continuity test must be satisfied before they can be applied in later years.
    
            Legacy Rule Format:
            IF ([LS80] = NULL OR [LS80] = 0) AND [LS14] >0 AND ([LS15] = FALSE OR [LS16] = FALSE OR [LS17] = FALSE OR [LS18] = FALSE OR [LS19] = FALSE)
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS14] = LS:RP.Y0-4(Instant):rvctc3.02.16:Capital.Losses.CarriedForward.Net.Amount
            [LS15] = LS:RP.Y0:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS16] = LS:RP.Y0-1:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS17] = LS:RP.Y0-2:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS18] = LS:RP.Y0-3:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS19] = LS:RP.Y0-4:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS80] = LS:RP:rvctc3.02.34:Capital.Losses.CarriedForward.BusinessContinuityTestPassedNet.Amount

            Technical Business Rule Format:
            (^LS80 = NULL OR ^LS80 = 0) AND ^LS14 >0 AND (^LS15 = FALSE OR ^LS16 = FALSE OR ^LS17 = FALSE OR ^LS18 = FALSE OR ^LS19 = FALSE)
    
            Data Elements:
    
            RP.Y0-4(Instant):^LS14 = tns:Capital.Losses.CarriedForward.Net.Amount
    
            RP.Y0:^LS15 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP.Y0-1:^LS16 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP.Y0-2:^LS17 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP.Y0-3:^LS18 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP.Y0-4:^LS19 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP:^LS80 = tns:Capital.Losses.CarriedForward.BusinessContinuityTestPassedNet.Amount
            */
            assertion = ((report.LS80 == null || report.LS80 == 0) && report.LS14.GetValueOrDefault() > 0 && (report.LS15 == false || report.LS16 == false || report.LS17 == false || report.LS18 == false || report.LS19 == false));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.LS.414218", Severity = ProcessMessageSeverity.Error,
                    Description = @"Net capital losses at ownership and business continuity test must be supplied.",
                    LongDescription = @"If an amount has been entered for net capital losses carried forward in the current income year - 4 and the continuity of majority ownership test was not passed in the current income year - 4 income year or in later years then an amount must be supplied for net capital losses carried forward for which the business continuity test must be satisfied before they can be applied in later years.",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.CarriedForward.Net.Amount[@contextRef='" + report.RPY0Minus4InstantId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414014"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS80", Value = report.LS80.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS14", Value = report.LS14.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS15", Value = GetValueOrEmpty(report.LS15) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS16", Value = GetValueOrEmpty(report.LS16) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS17", Value = GetValueOrEmpty(report.LS17) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS18", Value = GetValueOrEmpty(report.LS18) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS19", Value = GetValueOrEmpty(report.LS19) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414014
    
            #region VR.ATO.LS.414015
    
            /*  VR.ATO.LS.414015
            If an amount has been entered for net capital losses carried forward in the current income year - 5 and the continuity of majority ownership test was not passed in the current income year - 5 income year or in later years then an amount must be entered for net capital losses carried forward for which the business continuity test must be satisfied before they can be applied in later years.
    
            Legacy Rule Format:
            IF ([LS80] = NULL OR [LS80] = 0) AND [LS51] >0 AND ([LS15] = FALSE OR [LS16] = FALSE OR [LS17] = FALSE OR [LS18] = FALSE OR [LS19] = FALSE OR [LS53] = FALSE)
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS15] = LS:RP.Y0:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS16] = LS:RP.Y0-1:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS17] = LS:RP.Y0-2:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS18] = LS:RP.Y0-3:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS19] = LS:RP.Y0-4:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS51] = LS:RP.Y0-5P(Instant):rvctc3.02.16:Capital.Losses.CarriedForward.Net.Amount
            [LS53] = LS:RP.Y0-5P:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS80] = LS:RP:rvctc3.02.34:Capital.Losses.CarriedForward.BusinessContinuityTestPassedNet.Amount

            Technical Business Rule Format:
            (^LS80 = NULL OR ^LS80 = 0) AND ^LS51 >0 AND (^LS15 = FALSE OR ^LS16 = FALSE OR ^LS17 = FALSE OR ^LS18 = FALSE OR ^LS19 = FALSE OR ^LS53 = FALSE)
    
            Data Elements:
    
            RP.Y0-5P(Instant):^LS51 = tns:Capital.Losses.CarriedForward.Net.Amount
    
            RP.Y0:^LS15 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP.Y0-1:^LS16 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP.Y0-2:^LS17 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP.Y0-3:^LS18 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP.Y0-4:^LS19 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP.Y0-5P:^LS53 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP:^LS80 = tns:Capital.Losses.CarriedForward.BusinessContinuityTestPassedNet.Amount
            */
            assertion = ((report.LS80 == null || report.LS80 == 0) && report.LS51.GetValueOrDefault() > 0 && (report.LS15 == false || report.LS16 == false || report.LS17 == false || report.LS18 == false || report.LS19 == false || report.LS53 == false));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.LS.414220", Severity = ProcessMessageSeverity.Error,
                    Description = @"Net capital losses at ownership and business continuity test must be supplied.",
                    LongDescription = @"If an amount has been entered for net capital losses carried forward in the current income year - 5 and the continuity of majority ownership test was not passed in the current income year - 5 income year or in later years then an amount must be supplied for net capital losses carried forward for which the business continuity test must be satisfied before they can be applied in later years.",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.CarriedForward.Net.Amount[@contextRef='" + report.RPY0Minus5PInstantId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414015"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS80", Value = report.LS80.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS51", Value = report.LS51.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS15", Value = GetValueOrEmpty(report.LS15) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS16", Value = GetValueOrEmpty(report.LS16) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS17", Value = GetValueOrEmpty(report.LS17) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS18", Value = GetValueOrEmpty(report.LS18) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS19", Value = GetValueOrEmpty(report.LS19) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS53", Value = GetValueOrEmpty(report.LS53) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414015
    
            #region VR.ATO.LS.414018
    
            /*  VR.ATO.LS.414018
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [LS6] <> NULL AND [LS6] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS6] = LS:RP.Y0-1:rvctc3.02.00:Tax.Losses.CarriedForward.Total.Amount

            Technical Business Rule Format:
            NotMonetary(^LS6, 'U', 11, 0)
    
            Data Elements:
    
            RP.Y0-1:^LS6 = tns:Tax.Losses.CarriedForward.Total.Amount
            */
            assertion = NotMonetary(report.LS6, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.CarriedForward.Total.Amount[@contextRef='" + report.RPY0Minus1Id + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414018"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS6", Value = report.LS6.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414018
    
            #region VR.ATO.LS.414019
    
            /*  VR.ATO.LS.414019
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [LS7] <> NULL AND [LS7] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS7] = LS:RP.Y0-2:rvctc3.02.00:Tax.Losses.CarriedForward.Total.Amount

            Technical Business Rule Format:
            NotMonetary(^LS7, 'U', 11, 0)
    
            Data Elements:
    
            RP.Y0-2:^LS7 = tns:Tax.Losses.CarriedForward.Total.Amount
            */
            assertion = NotMonetary(report.LS7, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.CarriedForward.Total.Amount[@contextRef='" + report.RPY0Minus2Id + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414019"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS7", Value = report.LS7.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414019
    
            #region VR.ATO.LS.414020
    
            /*  VR.ATO.LS.414020
            If the continuity of majority ownership test is not passed in any income year listed then amounts of losses deducted for which the continuity of majority ownership test is not passed but the business continuity test is satisfied (tax losses and net capital losses) and losses carried forward for which the  business continuity test must be satisfied before they can be deducted in later years (tax losses and net capital losses) must be entered and whether the current year loss provisions apply need to be indicated.
    
            Legacy Rule Format:
            IF ([LS15] = FALSE OR [LS16] = FALSE OR [LS17] = FALSE OR [LS18] = FALSE OR [LS19] = FALSE OR [LS53] = FALSE) AND ([LS20] = NULL OR [LS21] = NULL OR [LS22] = NULL OR [LS80] = NULL OR [LS24] = NULL)
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS15] = LS:RP.Y0:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS16] = LS:RP.Y0-1:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS17] = LS:RP.Y0-2:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS18] = LS:RP.Y0-3:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS19] = LS:RP.Y0-4:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS20] = LS:RP:rvctc3.02.34:Tax.Losses.DeductedOwnershipTestFailContinuityTestPass.Amount
            [LS21] = LS:RP:rvctc3.02.34:Capital.Losses.DeductedOwnershipTestFailContinuityTestPass.Amount
            [LS22] = LS:RP:rvctc3.02.34:Tax.Losses.CarriedForward.BusinessContinuityTestPassed.Amount
            [LS24] = LS:RP:rvctc3.02.00:Tax.Losses.CurrentYearLossProvisionsApply.Indicator
            [LS53] = LS:RP.Y0-5P:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS80] = LS:RP:rvctc3.02.34:Capital.Losses.CarriedForward.BusinessContinuityTestPassedNet.Amount

            Technical Business Rule Format:
            (^LS15 = FALSE OR ^LS16 = FALSE OR ^LS17 = FALSE OR ^LS18 = FALSE OR ^LS19 = FALSE OR ^LS53 = FALSE) AND (^LS20 = NULL OR ^LS21 = NULL OR ^LS22 = NULL OR ^LS80 = NULL OR ^LS24 = NULL)
    
            Data Elements:
    
            RP.Y0:^LS15 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP.Y0-1:^LS16 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP.Y0-2:^LS17 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP.Y0-3:^LS18 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP.Y0-4:^LS19 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP:^LS20 = tns:Tax.Losses.DeductedOwnershipTestFailContinuityTestPass.Amount
    
            RP:^LS21 = tns:Capital.Losses.DeductedOwnershipTestFailContinuityTestPass.Amount
    
            RP:^LS22 = tns:Tax.Losses.CarriedForward.BusinessContinuityTestPassed.Amount
    
            RP:^LS24 = tns:Tax.Losses.CurrentYearLossProvisionsApply.Indicator
    
            RP.Y0-5P:^LS53 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP:^LS80 = tns:Capital.Losses.CarriedForward.BusinessContinuityTestPassedNet.Amount
            */
            assertion = ((report.LS15 == false || report.LS16 == false || report.LS17 == false || report.LS18 == false || report.LS19 == false || report.LS53 == false) && (report.LS20 == null || report.LS21 == null || report.LS22 == null || report.LS80 == null || report.LS24 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.LS.414208", Severity = ProcessMessageSeverity.Error,
                    Description = @"Ownership and business continuity test information must be completed.",
                    LongDescription = @"If the continuity of majority ownership test is not passed in any income year listed then amounts of losses deducted for which the continuity of majority ownership test is not passed but the business continuity test is satisfied (tax losses and net capital losses) and losses carried forward for which the  business continuity test must be satisfied before they can be deducted in later years (tax losses and net capital losses) must be supplied and whether the current year loss provisions apply need to be indicated.",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator[@contextRef='" + report.RPY0Id + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414020"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS15", Value = GetValueOrEmpty(report.LS15) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS16", Value = GetValueOrEmpty(report.LS16) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS17", Value = GetValueOrEmpty(report.LS17) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS18", Value = GetValueOrEmpty(report.LS18) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS19", Value = GetValueOrEmpty(report.LS19) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS53", Value = GetValueOrEmpty(report.LS53) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS20", Value = report.LS20.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS21", Value = report.LS21.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS22", Value = report.LS22.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS80", Value = report.LS80.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS24", Value = GetValueOrEmpty(report.LS24) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414020
    
            #region VR.ATO.LS.414021
    
            /*  VR.ATO.LS.414021
            If the continuity of majority ownership test is not passed in any income year listed then the entity will have to indicate whether the current year loss provisions apply.
    
            Legacy Rule Format:
            IF ([LS15] = FALSE OR [LS16] = FALSE OR [LS17] = FALSE OR [LS18] = FALSE OR [LS19] = FALSE OR [LS53] = FALSE) AND [LS24] = NULL
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS15] = LS:RP.Y0:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS16] = LS:RP.Y0-1:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS17] = LS:RP.Y0-2:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS18] = LS:RP.Y0-3:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS19] = LS:RP.Y0-4:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            [LS24] = LS:RP:rvctc3.02.00:Tax.Losses.CurrentYearLossProvisionsApply.Indicator
            [LS53] = LS:RP.Y0-5P:rvctc3.02.00:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator

            Technical Business Rule Format:
            (^LS15 = FALSE OR ^LS16 = FALSE OR ^LS17 = FALSE OR ^LS18 = FALSE OR ^LS19 = FALSE OR ^LS53 = FALSE) AND ^LS24 = NULL
    
            Data Elements:
    
            RP.Y0:^LS15 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP.Y0-1:^LS16 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP.Y0-2:^LS17 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP.Y0-3:^LS18 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP.Y0-4:^LS19 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
    
            RP:^LS24 = tns:Tax.Losses.CurrentYearLossProvisionsApply.Indicator
    
            RP.Y0-5P:^LS53 = tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator
            */
            assertion = ((report.LS15 == false || report.LS16 == false || report.LS17 == false || report.LS18 == false || report.LS19 == false || report.LS53 == false) && report.LS24 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.LS.414209", Severity = ProcessMessageSeverity.Error,
                    Description = @"Ownership and business continuity test current year loss provisions information must be supplied.",
                    LongDescription = @"If the continuity of majority ownership test is not passed in any income year listed then the entity will have to indicate whether the current year loss provisions apply.",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.CarriedForward.ContinuityOfMajorityOwnershipTestPassed.Indicator[@contextRef='" + report.RPY0Id + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414021"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS15", Value = GetValueOrEmpty(report.LS15) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS16", Value = GetValueOrEmpty(report.LS16) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS17", Value = GetValueOrEmpty(report.LS17) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS18", Value = GetValueOrEmpty(report.LS18) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS19", Value = GetValueOrEmpty(report.LS19) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS53", Value = GetValueOrEmpty(report.LS53) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS24", Value = GetValueOrEmpty(report.LS24) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414021
    
            #region VR.ATO.LS.414024
    
            /*  VR.ATO.LS.414024
            In regard to unrealised losses, if a changeover time occurred in relation to a company then the company must indicate whether the maximum asset test was satisfied at the changeover time.
    
            Legacy Rule Format:
            IF [LS25] = TRUE AND [LS26] = NULL
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS25] = LS:RP:rvctc3.02.00:Tax.Losses.ChangeOfOwnershipOrControlAfterReferenceTime.Indicator
            [LS26] = LS:RP:rvctc3.02.00:Tax.Losses.MaximumNetAssetValueTestPassedAtChangeoverTime.Indicator

            Technical Business Rule Format:
            ^LS25 = TRUE AND ^LS26 = NULL
    
            Data Elements:
    
            RP:^LS25 = tns:Tax.Losses.ChangeOfOwnershipOrControlAfterReferenceTime.Indicator
    
            RP:^LS26 = tns:Tax.Losses.MaximumNetAssetValueTestPassedAtChangeoverTime.Indicator
            */
            assertion = (report.LS25 == true && report.LS26 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.LS.414024", Severity = ProcessMessageSeverity.Error,
                    Description = @"You must answer (Unrealised losses), label (M)",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.ChangeOfOwnershipOrControlAfterReferenceTime.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414024"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS25", Value = GetValueOrEmpty(report.LS25) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS26", Value = GetValueOrEmpty(report.LS26) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414024
    
            #region VR.ATO.LS.414025
    
            /*  VR.ATO.LS.414025
            In regard to unrealised losses, if a changeover time did not occur in relation to a company then the company must not answer any further questions relating to unrealised losses.
    
            Legacy Rule Format:
            IF [LS25] = FALSE AND ([LS26] <> NULL OR [LS27] <> NULL OR [LS28] <> NULL)
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS25] = LS:RP:rvctc3.02.00:Tax.Losses.ChangeOfOwnershipOrControlAfterReferenceTime.Indicator
            [LS26] = LS:RP:rvctc3.02.00:Tax.Losses.MaximumNetAssetValueTestPassedAtChangeoverTime.Indicator
            [LS27] = LS:RP:rvctc3.02.00:Tax.Losses.UnrealisedNetLossAtChangeoverTime.Indicator
            [LS28] = LS:RP:rvctc3.02.00:Tax.Losses.UnrealisedNetLossAtChangeoverTime.Amount

            Technical Business Rule Format:
            ^LS25 = FALSE AND (^LS26 <> NULL OR ^LS27 <> NULL OR ^LS28 <> NULL)
    
            Data Elements:
    
            RP:^LS25 = tns:Tax.Losses.ChangeOfOwnershipOrControlAfterReferenceTime.Indicator
    
            RP:^LS26 = tns:Tax.Losses.MaximumNetAssetValueTestPassedAtChangeoverTime.Indicator
    
            RP:^LS27 = tns:Tax.Losses.UnrealisedNetLossAtChangeoverTime.Indicator
    
            RP:^LS28 = tns:Tax.Losses.UnrealisedNetLossAtChangeoverTime.Amount
            */
            assertion = (report.LS25 == false && (report.LS26 != null || report.LS27 != null || report.LS28 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.LS.414025", Severity = ProcessMessageSeverity.Error,
                    Description = @"Answers to questions at (Unrealised losses), labels (M), (N), or (O) are not required",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.ChangeOfOwnershipOrControlAfterReferenceTime.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414025"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS25", Value = GetValueOrEmpty(report.LS25) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS26", Value = GetValueOrEmpty(report.LS26) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS27", Value = GetValueOrEmpty(report.LS27) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS28", Value = report.LS28.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414025
    
            #region VR.ATO.LS.414026
    
            /*  VR.ATO.LS.414026
            In regard to unrealised losses, if the company has satisfied the maximum asset test then the company must not answer any further questions relating to unrealised losses.
    
            Legacy Rule Format:
            IF [LS26] = TRUE AND ([LS27] <> NULL OR [LS28] <> NULL)
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS26] = LS:RP:rvctc3.02.00:Tax.Losses.MaximumNetAssetValueTestPassedAtChangeoverTime.Indicator
            [LS27] = LS:RP:rvctc3.02.00:Tax.Losses.UnrealisedNetLossAtChangeoverTime.Indicator
            [LS28] = LS:RP:rvctc3.02.00:Tax.Losses.UnrealisedNetLossAtChangeoverTime.Amount

            Technical Business Rule Format:
            ^LS26 = TRUE AND (^LS27 <> NULL OR ^LS28 <> NULL)
    
            Data Elements:
    
            RP:^LS26 = tns:Tax.Losses.MaximumNetAssetValueTestPassedAtChangeoverTime.Indicator
    
            RP:^LS27 = tns:Tax.Losses.UnrealisedNetLossAtChangeoverTime.Indicator
    
            RP:^LS28 = tns:Tax.Losses.UnrealisedNetLossAtChangeoverTime.Amount
            */
            assertion = (report.LS26 == true && (report.LS27 != null || report.LS28 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.LS.414026", Severity = ProcessMessageSeverity.Error,
                    Description = @"Answers to questions at (Unrealised losses), labels (N) or (O) are not required",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.MaximumNetAssetValueTestPassedAtChangeoverTime.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414026"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS26", Value = GetValueOrEmpty(report.LS26) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS27", Value = GetValueOrEmpty(report.LS27) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS28", Value = report.LS28.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414026
    
            #region VR.ATO.LS.414027
    
            /*  VR.ATO.LS.414027
            In regard to unrealised losses, if the company did not satisfy the maximum asset test at the changeover time then either "yes" or "no" is to be indicated as to whether the company determined it had an unrealised loss at the changeover time.
    
            Legacy Rule Format:
            IF [LS26] = FALSE AND [LS27] = NULL
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS26] = LS:RP:rvctc3.02.00:Tax.Losses.MaximumNetAssetValueTestPassedAtChangeoverTime.Indicator
            [LS27] = LS:RP:rvctc3.02.00:Tax.Losses.UnrealisedNetLossAtChangeoverTime.Indicator

            Technical Business Rule Format:
            ^LS26 = FALSE AND ^LS27 = NULL
    
            Data Elements:
    
            RP:^LS26 = tns:Tax.Losses.MaximumNetAssetValueTestPassedAtChangeoverTime.Indicator
    
            RP:^LS27 = tns:Tax.Losses.UnrealisedNetLossAtChangeoverTime.Indicator
            */
            assertion = (report.LS26 == false && report.LS27 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.LS.414027", Severity = ProcessMessageSeverity.Error,
                    Description = @"You must answer (Unrealised losses), label (N)",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.MaximumNetAssetValueTestPassedAtChangeoverTime.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414027"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS26", Value = GetValueOrEmpty(report.LS26) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS27", Value = GetValueOrEmpty(report.LS27) });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414027
    
            #region VR.ATO.LS.414028
    
            /*  VR.ATO.LS.414028
            If the company determined that it had an unrealised net loss at the changeover time then the amount of the unrealised loss must be entered.
    
            Legacy Rule Format:
            IF [LS27] = TRUE AND ([LS28] = NULL OR [LS28] = 0)
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS27] = LS:RP:rvctc3.02.00:Tax.Losses.UnrealisedNetLossAtChangeoverTime.Indicator
            [LS28] = LS:RP:rvctc3.02.00:Tax.Losses.UnrealisedNetLossAtChangeoverTime.Amount

            Technical Business Rule Format:
            ^LS27 = TRUE AND (^LS28 = NULL OR ^LS28 = 0)
    
            Data Elements:
    
            RP:^LS27 = tns:Tax.Losses.UnrealisedNetLossAtChangeoverTime.Indicator
    
            RP:^LS28 = tns:Tax.Losses.UnrealisedNetLossAtChangeoverTime.Amount
            */
            assertion = (report.LS27 == true && (report.LS28 == null || report.LS28 == 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.LS.414028", Severity = ProcessMessageSeverity.Error,
                    Description = @"The amount of unrealised net loss calculated under section 165-115E of ITAA 1997 is required",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.UnrealisedNetLossAtChangeoverTime.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414028"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS27", Value = GetValueOrEmpty(report.LS27) });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS28", Value = report.LS28.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414028
    
            #region VR.ATO.LS.414038
    
            /*  VR.ATO.LS.414038
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [LS8] <> NULL AND [LS8] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS8] = LS:RP.Y0-3:rvctc3.02.00:Tax.Losses.CarriedForward.Total.Amount

            Technical Business Rule Format:
            NotMonetary(^LS8, 'U', 11, 0)
    
            Data Elements:
    
            RP.Y0-3:^LS8 = tns:Tax.Losses.CarriedForward.Total.Amount
            */
            assertion = NotMonetary(report.LS8, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.CarriedForward.Total.Amount[@contextRef='" + report.RPY0Minus3Id + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414038"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS8", Value = report.LS8.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414038
    
            #region VR.ATO.LS.414039
    
            /*  VR.ATO.LS.414039
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [LS9] <> NULL AND [LS9] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS9] = LS:RP.Y0-4:rvctc3.02.00:Tax.Losses.CarriedForward.Total.Amount

            Technical Business Rule Format:
            NotMonetary(^LS9, 'U', 11, 0)
    
            Data Elements:
    
            RP.Y0-4:^LS9 = tns:Tax.Losses.CarriedForward.Total.Amount
            */
            assertion = NotMonetary(report.LS9, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.CarriedForward.Total.Amount[@contextRef='" + report.RPY0Minus4Id + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414039"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS9", Value = report.LS9.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414039
    
            #region VR.ATO.LS.414040
    
            /*  VR.ATO.LS.414040
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [LS49] <> NULL AND [LS49] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS49] = LS:RP.Y0-5P:rvctc3.02.00:Tax.Losses.CarriedForward.Total.Amount

            Technical Business Rule Format:
            NotMonetary(^LS49, 'U', 11, 0)
    
            Data Elements:
    
            RP.Y0-5P:^LS49 = tns:Tax.Losses.CarriedForward.Total.Amount
            */
            assertion = NotMonetary(report.LS49, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.CarriedForward.Total.Amount[@contextRef='" + report.RPY0Minus5PId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414040"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS49", Value = report.LS49.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414040
    
            #region VR.ATO.LS.414041
    
            /*  VR.ATO.LS.414041
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [LS50] <> NULL AND [LS50] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS50] = LS:RP.ALL:rvctc3.02.00:Tax.Losses.CarriedForward.Total.Amount

            Technical Business Rule Format:
            NotMonetary(^LS50, 'U', 11, 0)
    
            Data Elements:
    
            RP.ALL:^LS50 = tns:Tax.Losses.CarriedForward.Total.Amount
            */
            assertion = NotMonetary(report.LS50, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.CarriedForward.Total.Amount[@contextRef='" + report.RPALLId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414041"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS50", Value = report.LS50.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414041
    
            #region VR.ATO.LS.414042
    
            /*  VR.ATO.LS.414042
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [LS10] <> NULL AND [LS10] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS10] = LS:RP.Y0(Instant):rvctc3.02.16:Capital.Losses.CarriedForward.Net.Amount

            Technical Business Rule Format:
            NotMonetary(^LS10, 'U', 11, 0)
    
            Data Elements:
    
            RP.Y0(Instant):^LS10 = tns:Capital.Losses.CarriedForward.Net.Amount
            */
            assertion = NotMonetary(report.LS10, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.CarriedForward.Net.Amount[@contextRef='" + report.RPY0InstantId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414042"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS10", Value = report.LS10.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414042
    
            #region VR.ATO.LS.414043
    
            /*  VR.ATO.LS.414043
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [LS11] <> NULL AND [LS11] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS11] = LS:RP.Y0-1(Instant):rvctc3.02.16:Capital.Losses.CarriedForward.Net.Amount

            Technical Business Rule Format:
            NotMonetary(^LS11, 'U', 11, 0)
    
            Data Elements:
    
            RP.Y0-1(Instant):^LS11 = tns:Capital.Losses.CarriedForward.Net.Amount
            */
            assertion = NotMonetary(report.LS11, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.CarriedForward.Net.Amount[@contextRef='" + report.RPY0Minus1InstantId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414043"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS11", Value = report.LS11.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414043
    
            #region VR.ATO.LS.414044
    
            /*  VR.ATO.LS.414044
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [LS12] <> NULL AND [LS12] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS12] = LS:RP.Y0-2(Instant):rvctc3.02.16:Capital.Losses.CarriedForward.Net.Amount

            Technical Business Rule Format:
            NotMonetary(^LS12, 'U', 11, 0)
    
            Data Elements:
    
            RP.Y0-2(Instant):^LS12 = tns:Capital.Losses.CarriedForward.Net.Amount
            */
            assertion = NotMonetary(report.LS12, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.CarriedForward.Net.Amount[@contextRef='" + report.RPY0Minus2InstantId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414044"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS12", Value = report.LS12.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414044
    
            #region VR.ATO.LS.414045
    
            /*  VR.ATO.LS.414045
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [LS13] <> NULL AND [LS13] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS13] = LS:RP.Y0-3(Instant):rvctc3.02.16:Capital.Losses.CarriedForward.Net.Amount

            Technical Business Rule Format:
            NotMonetary(^LS13, 'U', 11, 0)
    
            Data Elements:
    
            RP.Y0-3(Instant):^LS13 = tns:Capital.Losses.CarriedForward.Net.Amount
            */
            assertion = NotMonetary(report.LS13, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.CarriedForward.Net.Amount[@contextRef='" + report.RPY0Minus3InstantId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414045"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS13", Value = report.LS13.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414045
    
            #region VR.ATO.LS.414046
    
            /*  VR.ATO.LS.414046
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [LS14] <> NULL AND [LS14] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS14] = LS:RP.Y0-4(Instant):rvctc3.02.16:Capital.Losses.CarriedForward.Net.Amount

            Technical Business Rule Format:
            NotMonetary(^LS14, 'U', 11, 0)
    
            Data Elements:
    
            RP.Y0-4(Instant):^LS14 = tns:Capital.Losses.CarriedForward.Net.Amount
            */
            assertion = NotMonetary(report.LS14, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.CarriedForward.Net.Amount[@contextRef='" + report.RPY0Minus4InstantId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414046"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS14", Value = report.LS14.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414046
    
            #region VR.ATO.LS.414047
    
            /*  VR.ATO.LS.414047
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [LS51] <> NULL AND [LS51] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS51] = LS:RP.Y0-5P(Instant):rvctc3.02.16:Capital.Losses.CarriedForward.Net.Amount

            Technical Business Rule Format:
            NotMonetary(^LS51, 'U', 11, 0)
    
            Data Elements:
    
            RP.Y0-5P(Instant):^LS51 = tns:Capital.Losses.CarriedForward.Net.Amount
            */
            assertion = NotMonetary(report.LS51, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.CarriedForward.Net.Amount[@contextRef='" + report.RPY0Minus5PInstantId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414047"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS51", Value = report.LS51.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414047
    
            #region VR.ATO.LS.414048
    
            /*  VR.ATO.LS.414048
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [LS52] <> NULL AND [LS52] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS52] = LS:RP.ALL(Instant):rvctc3.02.16:Capital.Losses.CarriedForward.Net.Amount

            Technical Business Rule Format:
            NotMonetary(^LS52, 'U', 11, 0)
    
            Data Elements:
    
            RP.ALL(Instant):^LS52 = tns:Capital.Losses.CarriedForward.Net.Amount
            */
            assertion = NotMonetary(report.LS52, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.CarriedForward.Net.Amount[@contextRef='" + report.RPALLInstantId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414048"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS52", Value = report.LS52.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414048
    
            #region VR.ATO.LS.414049
    
            /*  VR.ATO.LS.414049
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [LS20] <> NULL AND [LS20] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS20] = LS:RP:rvctc3.02.34:Tax.Losses.DeductedOwnershipTestFailContinuityTestPass.Amount

            Technical Business Rule Format:
            NotMonetary(^LS20, 'U', 11, 0)
    
            Data Elements:
    
            RP:^LS20 = tns:Tax.Losses.DeductedOwnershipTestFailContinuityTestPass.Amount
            */
            assertion = NotMonetary(report.LS20, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.DeductedOwnershipTestFailContinuityTestPass.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414049"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS20", Value = report.LS20.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414049
    
            #region VR.ATO.LS.414050
    
            /*  VR.ATO.LS.414050
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [LS21] <> NULL AND [LS21] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS21] = LS:RP:rvctc3.02.34:Capital.Losses.DeductedOwnershipTestFailContinuityTestPass.Amount

            Technical Business Rule Format:
            NotMonetary(^LS21, 'U', 11, 0)
    
            Data Elements:
    
            RP:^LS21 = tns:Capital.Losses.DeductedOwnershipTestFailContinuityTestPass.Amount
            */
            assertion = NotMonetary(report.LS21, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.DeductedOwnershipTestFailContinuityTestPass.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414050"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS21", Value = report.LS21.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414050
    
            #region VR.ATO.LS.414051
    
            /*  VR.ATO.LS.414051
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [LS22] <> NULL AND [LS22] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS22] = LS:RP:rvctc3.02.34:Tax.Losses.CarriedForward.BusinessContinuityTestPassed.Amount

            Technical Business Rule Format:
            NotMonetary(^LS22, 'U', 11, 0)
    
            Data Elements:
    
            RP:^LS22 = tns:Tax.Losses.CarriedForward.BusinessContinuityTestPassed.Amount
            */
            assertion = NotMonetary(report.LS22, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.CarriedForward.BusinessContinuityTestPassed.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414051"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS22", Value = report.LS22.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414051
    
            #region VR.ATO.LS.414053
    
            /*  VR.ATO.LS.414053
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [LS28] <> NULL AND [LS28] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS28] = LS:RP:rvctc3.02.00:Tax.Losses.UnrealisedNetLossAtChangeoverTime.Amount

            Technical Business Rule Format:
            NotMonetary(^LS28, 'U', 11, 0)
    
            Data Elements:
    
            RP:^LS28 = tns:Tax.Losses.UnrealisedNetLossAtChangeoverTime.Amount
            */
            assertion = NotMonetary(report.LS28, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.UnrealisedNetLossAtChangeoverTime.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414053"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS28", Value = report.LS28.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414053
    
            #region VR.ATO.LS.414066
    
            /*  VR.ATO.LS.414066
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [LS64] <> NULL AND [LS64] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS64] = LS:RP:rvctc3.02.04:Tax.Losses.ControlledForeignCompany.Amount

            Technical Business Rule Format:
            NotMonetary(^LS64, 'U', 11, 0)
    
            Data Elements:
    
            RP:^LS64 = tns:Tax.Losses.ControlledForeignCompany.Amount
            */
            assertion = NotMonetary(report.LS64, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.ControlledForeignCompany.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414066"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS64", Value = report.LS64.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414066
    
            #region VR.ATO.LS.414067
    
            /*  VR.ATO.LS.414067
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [LS44] <> NULL AND [LS44] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS44] = LS:RP:rvctc3.02.02:Tax.Losses.ControlledForeignCompanyDeducted.Amount

            Technical Business Rule Format:
            NotMonetary(^LS44, 'U', 11, 0)
    
            Data Elements:
    
            RP:^LS44 = tns:Tax.Losses.ControlledForeignCompanyDeducted.Amount
            */
            assertion = NotMonetary(report.LS44, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.ControlledForeignCompanyDeducted.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414067"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS44", Value = report.LS44.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414067
    
            #region VR.ATO.LS.414068
    
            /*  VR.ATO.LS.414068
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [LS45] <> NULL AND [LS45] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS45] = LS:RP:rvctc3.02.02:Tax.Losses.CarriedForward.ControlledForeignCompany.Amount

            Technical Business Rule Format:
            NotMonetary(^LS45, 'U', 11, 0)
    
            Data Elements:
    
            RP:^LS45 = tns:Tax.Losses.CarriedForward.ControlledForeignCompany.Amount
            */
            assertion = NotMonetary(report.LS45, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.CarriedForward.ControlledForeignCompany.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414068"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS45", Value = report.LS45.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414068
    
            #region VR.ATO.LS.414069
    
            /*  VR.ATO.LS.414069
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [LS65] <> NULL AND [LS65] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS65] = LS:RP:rvctc3.02.02:Tax.Losses.BalanceOfTaxLossesBroughtForwardFromPriorYear.Amount

            Technical Business Rule Format:
            NotMonetary(^LS65, 'U', 11, 0)
    
            Data Elements:
    
            RP:^LS65 = tns:Tax.Losses.BalanceOfTaxLossesBroughtForwardFromPriorYear.Amount
            */
            assertion = NotMonetary(report.LS65, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.BalanceOfTaxLossesBroughtForwardFromPriorYear.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414069"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS65", Value = report.LS65.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414069
    
            #region VR.ATO.LS.414071
    
            /*  VR.ATO.LS.414071
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [LS67] <> NULL AND [LS67] <> MONETARY(S,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS67] = LS:RP:rvctc3.02.02:Tax.Losses.ForgivenDebtNet.Amount

            Technical Business Rule Format:
            NotMonetary(^LS67, 'S', 11, 0)
    
            Data Elements:
    
            RP:^LS67 = tns:Tax.Losses.ForgivenDebtNet.Amount
            */
            assertion = NotMonetary(report.LS67, @"S", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.ForgivenDebtNet.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414071"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS67", Value = report.LS67.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414071
    
            #region VR.ATO.LS.414072
    
            /*  VR.ATO.LS.414072
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [LS68] <> NULL AND [LS68] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS68] = LS:RP:rvctc3.02.02:Tax.Losses.Incurred.Amount

            Technical Business Rule Format:
            NotMonetary(^LS68, 'U', 11, 0)
    
            Data Elements:
    
            RP:^LS68 = tns:Tax.Losses.Incurred.Amount
            */
            assertion = NotMonetary(report.LS68, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.Incurred.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414072"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS68", Value = report.LS68.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414072
    
            #region VR.ATO.LS.414073
    
            /*  VR.ATO.LS.414073
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [LS69] <> NULL AND [LS69] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS69] = LS:RP:rvctc3.02.02:Tax.Losses.ConversionOfExcessFrankingOffsets.Amount

            Technical Business Rule Format:
            NotMonetary(^LS69, 'U', 11, 0)
    
            Data Elements:
    
            RP:^LS69 = tns:Tax.Losses.ConversionOfExcessFrankingOffsets.Amount
            */
            assertion = NotMonetary(report.LS69, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.ConversionOfExcessFrankingOffsets.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414073"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS69", Value = report.LS69.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414073
    
            #region VR.ATO.LS.414074
    
            /*  VR.ATO.LS.414074
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [LS70] <> NULL AND [LS70] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS70] = LS:RP:rvctc3.02.02:Tax.Losses.ExemptIncomeNet.Amount

            Technical Business Rule Format:
            NotMonetary(^LS70, 'U', 11, 0)
    
            Data Elements:
    
            RP:^LS70 = tns:Tax.Losses.ExemptIncomeNet.Amount
            */
            assertion = NotMonetary(report.LS70, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.ExemptIncomeNet.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414074"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS70", Value = report.LS70.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414074
    
            #region VR.ATO.LS.414075
    
            /*  VR.ATO.LS.414075
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [LS71] <> NULL AND [LS71] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS71] = LS:RP:rvctc3.02.34:Tax.Losses.Forgone.Amount

            Technical Business Rule Format:
            NotMonetary(^LS71, 'U', 11, 0)
    
            Data Elements:
    
            RP:^LS71 = tns:Tax.Losses.Forgone.Amount
            */
            assertion = NotMonetary(report.LS71, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.Forgone.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414075"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS71", Value = report.LS71.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414075
    
            #region VR.ATO.LS.414076
    
            /*  VR.ATO.LS.414076
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [LS72] <> NULL AND [LS72] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS72] = LS:RP:rvctc3.02.00:Tax.Losses.Total.Amount

            Technical Business Rule Format:
            NotMonetary(^LS72, 'U', 11, 0)
    
            Data Elements:
    
            RP:^LS72 = tns:Tax.Losses.Total.Amount
            */
            assertion = NotMonetary(report.LS72, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.Total.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414076"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS72", Value = report.LS72.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414076
    
            #region VR.ATO.LS.414077
    
            /*  VR.ATO.LS.414077
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [LS73] <> NULL AND [LS73] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS73] = LS:RP:rvctc3.02.02:Tax.Losses.TransferredOut.Amount

            Technical Business Rule Format:
            NotMonetary(^LS73, 'U', 11, 0)
    
            Data Elements:
    
            RP:^LS73 = tns:Tax.Losses.TransferredOut.Amount
            */
            assertion = NotMonetary(report.LS73, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.TransferredOut.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414077"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS73", Value = report.LS73.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414077
    
            #region VR.ATO.LS.414078
    
            /*  VR.ATO.LS.414078
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [LS74] <> NULL AND [LS74] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS74] = LS:RP:rvctc3.02.02:Tax.Losses.CarriedForward.LaterIncomeYearsTotal.Amount

            Technical Business Rule Format:
            NotMonetary(^LS74, 'U', 11, 0)
    
            Data Elements:
    
            RP:^LS74 = tns:Tax.Losses.CarriedForward.LaterIncomeYearsTotal.Amount
            */
            assertion = NotMonetary(report.LS74, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.CarriedForward.LaterIncomeYearsTotal.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414078"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS74", Value = report.LS74.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414078
    
            #region VR.ATO.LS.414094
    
            /*  VR.ATO.LS.414094
            The total number RP.Y0 supplied cannot exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.Y0)) > 1
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.Y0
            */
            assertion = (Count(report.RPY0Count) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.LS.414105", Severity = ProcessMessageSeverity.Error,
                    Description = @"The number of RP.Y0 contexts must not exceed 1.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0LastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414094"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414094
    
            #region VR.ATO.LS.414095
    
            /*  VR.ATO.LS.414095
            The total number RP.Y0-1 supplied cannot exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.Y0-1)) > 1
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.Y0-1
            */
            assertion = (Count(report.RPY0Minus1Count) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.LS.414106", Severity = ProcessMessageSeverity.Error,
                    Description = @"The number of RP.Y0-1 contexts must not exceed 1.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0Minus1LastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414095"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414095
    
            #region VR.ATO.LS.414096
    
            /*  VR.ATO.LS.414096
            The total number RP.Y0-2 supplied cannot exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.Y0-2)) > 1
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.Y0-2
            */
            assertion = (Count(report.RPY0Minus2Count) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.LS.414107", Severity = ProcessMessageSeverity.Error,
                    Description = @"The number of RP.Y0-2 contexts must not exceed 1",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0Minus2LastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414096"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414096
    
            #region VR.ATO.LS.414097
    
            /*  VR.ATO.LS.414097
            The total number RP.Y0-3 supplied cannot exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.Y0-3)) > 1
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.Y0-3
            */
            assertion = (Count(report.RPY0Minus3Count) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.LS.414108", Severity = ProcessMessageSeverity.Error,
                    Description = @"The number of RP.Y0-3 contexts must not exceed 1",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0Minus3LastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414097"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414097
    
            #region VR.ATO.LS.414098
    
            /*  VR.ATO.LS.414098
            The total number RP.Y0-4 supplied cannot exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.Y0-4)) > 1
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.Y0-4
            */
            assertion = (Count(report.RPY0Minus4Count) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.LS.414109", Severity = ProcessMessageSeverity.Error,
                    Description = @"The number of RP.Y0-4 contexts must not exceed 1.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0Minus4LastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414098"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414098
    
            #region VR.ATO.LS.414099
    
            /*  VR.ATO.LS.414099
            The total number RP.Y0-5P supplied cannot exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.Y0-5P)) > 1
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.Y0-5P
            */
            assertion = (Count(report.RPY0Minus5PCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.LS.414110", Severity = ProcessMessageSeverity.Error,
                    Description = @"The number of RP.Y0-5 contexts must not exceed 1.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0Minus5PLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414099"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414099
    
            #region VR.ATO.LS.414100
    
            /*  VR.ATO.LS.414100
            The total number RP.All supplied cannot exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.ALL)) > 1
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.ALL
            */
            assertion = (Count(report.RPALLCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.LS.414111", Severity = ProcessMessageSeverity.Error,
                    Description = @"The number of RP.All contexts must not exceed 1.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPALLLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414100"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414100
    
            #region VR.ATO.LS.414112
    
            /*  VR.ATO.LS.414112
            Total tax losses carried forward to later income years must be equal to the equation of all amounts in the Tax losses reconciliation statement and must be a loss.
    
            Legacy Rule Format:
            IF ([LS65] <> NULL OR [LS67] <> NULL OR [LS68] <> NULL OR [LS69] <> NULL OR [LS70] <> NULL OR [LS71] <> NULL OR [LS72] <> NULL OR [LS73] <> NULL OR [LS74] <> NULL OR [LS79] <> NULL) AND 
            [LS74] <> [LS65] + [LS79] - [LS67] + [LS68] + [LS69] - [LS70] - [LS71] - [LS72] - [LS73]
              RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS65] = LS:RP:rvctc3.02.02:Tax.Losses.BalanceOfTaxLossesBroughtForwardFromPriorYear.Amount
            [LS67] = LS:RP:rvctc3.02.02:Tax.Losses.ForgivenDebtNet.Amount
            [LS68] = LS:RP:rvctc3.02.02:Tax.Losses.Incurred.Amount
            [LS69] = LS:RP:rvctc3.02.02:Tax.Losses.ConversionOfExcessFrankingOffsets.Amount
            [LS70] = LS:RP:rvctc3.02.02:Tax.Losses.ExemptIncomeNet.Amount
            [LS71] = LS:RP:rvctc3.02.34:Tax.Losses.Forgone.Amount
            [LS72] = LS:RP:rvctc3.02.00:Tax.Losses.Total.Amount
            [LS73] = LS:RP:rvctc3.02.02:Tax.Losses.TransferredOut.Amount
            [LS74] = LS:RP:rvctc3.02.02:Tax.Losses.CarriedForward.LaterIncomeYearsTotal.Amount
            [LS79] = LS:RP:rvctc3.02.14:Tax.Losses.Uplift.Amount

            Technical Business Rule Format:
            (^LS65 <> NULL OR ^LS67 <> NULL OR ^LS68 <> NULL OR ^LS69 <> NULL OR ^LS70 <> NULL OR ^LS71 <> NULL OR ^LS72 <> NULL OR ^LS73 <> NULL OR ^LS74 <> NULL OR ^LS79 <> NULL) AND ^LS74 <> ^LS65 + ^LS79 - ^LS67 + ^LS68 + ^LS69 - ^LS70 - ^LS71 - ^LS72 - ^LS73
    
            Data Elements:
    
            RP:^LS65 = tns:Tax.Losses.BalanceOfTaxLossesBroughtForwardFromPriorYear.Amount
    
            RP:^LS67 = tns:Tax.Losses.ForgivenDebtNet.Amount
    
            RP:^LS68 = tns:Tax.Losses.Incurred.Amount
    
            RP:^LS69 = tns:Tax.Losses.ConversionOfExcessFrankingOffsets.Amount
    
            RP:^LS70 = tns:Tax.Losses.ExemptIncomeNet.Amount
    
            RP:^LS71 = tns:Tax.Losses.Forgone.Amount
    
            RP:^LS72 = tns:Tax.Losses.Total.Amount
    
            RP:^LS73 = tns:Tax.Losses.TransferredOut.Amount
    
            RP:^LS74 = tns:Tax.Losses.CarriedForward.LaterIncomeYearsTotal.Amount
    
            RP:^LS79 = tns:Tax.Losses.Uplift.Amount
            */
            assertion = ((report.LS65 != null || report.LS67 != null || report.LS68 != null || report.LS69 != null || report.LS70 != null || report.LS71 != null || report.LS72 != null || report.LS73 != null || report.LS74 != null || report.LS79 != null) && report.LS74.GetValueOrDefault() != report.LS65.GetValueOrDefault() + report.LS79.GetValueOrDefault() - report.LS67.GetValueOrDefault() + report.LS68.GetValueOrDefault() + report.LS69.GetValueOrDefault() - report.LS70.GetValueOrDefault() - report.LS71.GetValueOrDefault() - report.LS72.GetValueOrDefault() - report.LS73.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.LS.414112", Severity = ProcessMessageSeverity.Error,
                    Description = @"Total tax losses carried forward to later income years must be equal to the equation of all amounts in the Tax losses reconciliation statement and must be a loss.",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.BalanceOfTaxLossesBroughtForwardFromPriorYear.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414112"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS65", Value = report.LS65.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS67", Value = report.LS67.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS68", Value = report.LS68.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS69", Value = report.LS69.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS70", Value = report.LS70.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS71", Value = report.LS71.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS72", Value = report.LS72.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS73", Value = report.LS73.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS74", Value = report.LS74.GetValueOrDefault().ToString() });
    
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS79", Value = report.LS79.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414112
    
            #region VR.ATO.LS.414113
    
            /*  VR.ATO.LS.414113
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [LS79] <> NULL AND [LS79] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS79] = LS:RP:rvctc3.02.14:Tax.Losses.Uplift.Amount

            Technical Business Rule Format:
            NotMonetary(^LS79, 'U', 11, 0)
    
            Data Elements:
    
            RP:^LS79 = tns:Tax.Losses.Uplift.Amount
            */
            assertion = NotMonetary(report.LS79, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.Uplift.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414113"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS79", Value = report.LS79.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414113
    
            #region VR.ATO.LS.414114
    
            /*  VR.ATO.LS.414114
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [LS80] <> NULL AND [LS80] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS80] = LS:RP:rvctc3.02.34:Capital.Losses.CarriedForward.BusinessContinuityTestPassedNet.Amount

            Technical Business Rule Format:
            NotMonetary(^LS80, 'U', 11, 0)
    
            Data Elements:
    
            RP:^LS80 = tns:Capital.Losses.CarriedForward.BusinessContinuityTestPassedNet.Amount
            */
            assertion = NotMonetary(report.LS80, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.CarriedForward.BusinessContinuityTestPassedNet.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414114"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS80", Value = report.LS80.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414114
    
            #region VR.ATO.LS.414170
    
            /*  VR.ATO.LS.414170
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [LS81] <> NULL AND [LS81] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS81] = LS:RP:rvctc3.02.22:Tax.Losses.CarriedForward.ComplyingSuperannuationClass.Amount

            Technical Business Rule Format:
            NotMonetary(^LS81, 'U', 11, 0)
    
            Data Elements:
    
            RP:^LS81 = tns:Tax.Losses.CarriedForward.ComplyingSuperannuationClass.Amount
            */
            assertion = NotMonetary(report.LS81, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.CarriedForward.ComplyingSuperannuationClass.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414170"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS81", Value = report.LS81.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414170
    
            #region VR.ATO.LS.414171
    
            /*  VR.ATO.LS.414171
            If this monetary field is populated ensure it complies with the monetary field format rules and also contains a maximum of 11 digits with no decimal places.
    
            Legacy Rule Format:
            IF [LS82] <> NULL AND [LS82] <> MONETARY(U,11,0) 
                RETURN VALIDATION MESSAGE
            ENDIF
            
            [LS82] = LS:RP:rvctc3.02.22:Capital.Losses.CarriedForward.ComplyingSuperannuationClassNet.Amount

            Technical Business Rule Format:
            NotMonetary(^LS82, 'U', 11, 0)
    
            Data Elements:
    
            RP:^LS82 = tns:Capital.Losses.CarriedForward.ComplyingSuperannuationClassNet.Amount
            */
            assertion = NotMonetary(report.LS82, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011", Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.CarriedForward.ComplyingSuperannuationClassNet.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414171"} },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "LS82", Value = report.LS82.GetValueOrDefault().ToString() });
    
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414171
    
            #region VR.ATO.LS.414200
    
            /*  VR.ATO.LS.414200
            The total number RP.Y0(Instant) supplied cannot exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.Y0(Instant))) > 1
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP.Y0(Instant)
            */
            assertion = (Count(report.RPY0InstantCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.LS.414200", Severity = ProcessMessageSeverity.Error,
                    Description = @"The number of RP.Y0(Instant) contexts must not exceed 1.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0InstantLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414200"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414200
    
            #region VR.ATO.LS.414201
    
            /*  VR.ATO.LS.414201
            The total number RP.Y0-1(Instant) supplied cannot exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.Y0-1(Instant))) > 1
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP.Y0-1(Instant)
            */
            assertion = (Count(report.RPY0Minus1InstantCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.LS.414201", Severity = ProcessMessageSeverity.Error,
                    Description = @"The number of RP.Y0-1(Instant) contexts must not exceed 1.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0Minus1InstantLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414201"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414201
    
            #region VR.ATO.LS.414202
    
            /*  VR.ATO.LS.414202
            The total number RP.Y0-2(Instant) supplied cannot exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.Y0-2(Instant))) > 1
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP.Y0-2(Instant)
            */
            assertion = (Count(report.RPY0Minus2InstantCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.LS.414202", Severity = ProcessMessageSeverity.Error,
                    Description = @"The number of RP.Y0-2(Instant) contexts must not exceed 1.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0Minus2InstantLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414202"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414202
    
            #region VR.ATO.LS.414203
    
            /*  VR.ATO.LS.414203
            The total number RP.Y0-3(Instant) supplied cannot exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.Y0-3(Instant))) > 1
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP.Y0-3(Instant)
            */
            assertion = (Count(report.RPY0Minus3InstantCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.LS.414203", Severity = ProcessMessageSeverity.Error,
                    Description = @"The number of RP.Y0-3(Instant) contexts must not exceed 1.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0Minus3InstantLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414203"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414203
    
            #region VR.ATO.LS.414204
    
            /*  VR.ATO.LS.414204
            The total number RP.Y0-4(Instant) supplied cannot exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.Y0-4(Instant))) > 1
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP.Y0-4(Instant)
            */
            assertion = (Count(report.RPY0Minus4InstantCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.LS.414204", Severity = ProcessMessageSeverity.Error,
                    Description = @"The number of RP.Y0-4(Instant) contexts must not exceed 1.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0Minus4InstantLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414204"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414204
    
            #region VR.ATO.LS.414205
    
            /*  VR.ATO.LS.414205
            The total number RP.Y0-5P(Instant) supplied cannot exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.Y0-5P(Instant))) > 1
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP.Y0-5P(Instant)
            */
            assertion = (Count(report.RPY0Minus5PInstantCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.LS.414205", Severity = ProcessMessageSeverity.Error,
                    Description = @"The number of RP.Y0-5(Instant) contexts must not exceed 1.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0Minus5PInstantLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414205"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414205
    
            #region VR.ATO.LS.414206
    
            /*  VR.ATO.LS.414206
            The total number RP.All(Instant) supplied cannot exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.ALL(Instant))) > 1
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP.ALL(Instant)
            */
            assertion = (Count(report.RPALLInstantCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.LS.414206", Severity = ProcessMessageSeverity.Error,
                    Description = @"The number of RP.All(Instant) contexts must not exceed 1.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPALLInstantLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() {Name = "RuleIdentifier", Value = "VR.ATO.LS.414206"} },
                };
                response.Add(processMessage);
            }
            #endregion // VR.ATO.LS.414206

            foreach (ProcessMessageDocument currentProcessMessage in response)
            {
                if (currentProcessMessage.Parameters != null)
                {
                    foreach (ProcessMessageParameter currentParameter in currentProcessMessage.Parameters)
                    {
                        if (string.IsNullOrEmpty(currentParameter.Name))
                        {
                            currentParameter.Name = _emptyParameterValue;
                        }

                        if (currentParameter.Name.Length > _maxParameterNameLength)
                        {
                            currentParameter.Name = currentParameter.Name.Substring(0, _maxParameterNameLength - 1);
                        }

                        if (string.IsNullOrEmpty(currentParameter.Value))
                        {
                            currentParameter.Value = _emptyParameterValue;
                        }

                        if (currentParameter.Value.Length > _maxParameterValueLength)
                        {
                            currentParameter.Value = currentParameter.Value.Substring(0, _maxParameterValueLength - 1);
                        }
                    }
                }
            }

            return response;
        }


    }
} 