using System;
using System.Runtime.CompilerServices;
using System.Linq;
using System.Collections.Generic;
using DataContracts;
using System.Text.RegularExpressions;
using static Ato.CD.Inbound.Shared.ValidatorHelpers;

namespace Ato.EN.IntegrationServices.CodeGenerationPTR
{

    public partial class PTR2025RequestValidator
    {
        /// <summary>
        /// The max parameter name length is restricted by the SBR1 and SBR2 schemas - the lowest common denominator of SBR 1 is used as the default
        /// </summary>
        private int _maxParameterNameLength;

        /// <summary>
        /// The max parameter value length is restricted by the SBR1 and SBR2 schemas
        /// </summary>
        private int _maxParameterValueLength;

        /// <summary>
        /// The SBR1 and ebms3 schemas do not allow parameter names or values to be the empty string
        /// </summary>
        private string _emptyParameterValue;

        /// <summary>
        /// The report parameter
        /// </summary>
        private PTR2025 report;

        /// <summary>
        /// The response parameter
        /// </summary>
        public List<ProcessMessageDocument> response { get; private set; }

        /// <summary>
        /// Initializes a new instance of the <see cref="PTR2025RequestValidator" /> class.
        /// </summary>
        /// <param name="maxParameterNameLength">Maximum length of the parameter name.</param>
        /// <param name="maxParameterValueLength">Maximum length of the parameter value.</param>
        /// <param name="emptyParameterValue">This value will be used in place of any parameter values that result in a null or empty value.</param>
        public PTR2025RequestValidator(int maxParameterNameLength = 20, int maxParameterValueLength = 4096, string emptyParameterValue = "EMPTY")
        {
            _maxParameterNameLength = maxParameterNameLength;
            _maxParameterValueLength = maxParameterValueLength;
            _emptyParameterValue = emptyParameterValue;
            this.response = new List<ProcessMessageDocument>();
        }

        /// <summary>
        /// Initializes a new instance of the <see cref="PTR2025RequestValidator" /> class.
        /// </summary>
        /// <param name="report">Report variable </param>                
        /// <param name="maxParameterNameLength">Maximum length of the parameter name.</param>
        /// <param name="maxParameterValueLength">Maximum length of the parameter value.</param>
        /// <param name="emptyParameterValue">This value will be used in place of any parameter values that result in a null or empty value.</param>
        public PTR2025RequestValidator(PTR2025 reportIn, int maxParameterNameLength = 20, int maxParameterValueLength = 4096, string emptyParameterValue = "EMPTY") : this(maxParameterNameLength, maxParameterValueLength, emptyParameterValue)
        {
            ConsumedReport = reportIn;
        }

        public PTR2025 ConsumedReport { get { return report; } private set { report = value; } }

        private static Dictionary<string, ProcessMessageDocument> _processMessageDocuments = new Dictionary<string, ProcessMessageDocument>();

        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public List<ProcessMessageDocument> ValidateReport(PTR2025 reportIn)
        {
            ConsumedReport = reportIn;

            VRATOPTR430001();
            VRATOPTR430009();
            VRATOPTR430010();
            VRATOPTR430012();
            VRATOPTR430013();
            VRATOPTR430015();
            VRATOPTR430016();
            VRATOPTR430017();
            VRATOPTR430018();
            VRATOPTR430022();
            VRATOPTR430023();
            VRATOPTR430027();
            VRATOPTR430030();
            VRATOPTR430032();
            VRATOPTR430040();
            VRATOPTR430044();
            VRATOPTR430046();
            VRATOPTR430049();
            VRATOPTR430050();
            VRATOPTR430051();
            VRATOPTR430052();
            VRATOPTR430056();
            VRATOPTR430057();
            VRATOPTR430058();
            VRATOPTR430059();
            VRATOPTR430061();
            VRATOPTR430063();
            VRATOPTR430068();
            VRATOPTR430079();
            VRATOPTR430080();
            VRATOPTR430092();
            VRATOPTR430093();
            VRATOPTR430095();
            VRATOPTR430096();
            VRATOPTR430098();
            VRATOPTR430100();
            VRATOPTR430102();
            VRATOPTR430106();
            VRATOPTR430112();
            VRATOPTR430116();
            VRATOPTR430120();
            VRATOPTR430129();
            VRATOPTR430130();
            VRATOPTR430131();
            VRATOPTR430132();
            VRATOPTR430142();
            VRATOPTR430144();
            VRATOPTR430145();
            VRATOPTR430151();
            VRATOPTR430153();
            VRATOPTR430155();
            VRATOPTR430158();
            VRATOPTR430166();
            VRATOPTR430171();
            VRATOPTR430172();
            VRATOPTR430177();
            VRATOPTR430194();
            VRATOPTR430195();
            VRATOPTR430197();
            VRATOPTR430198();
            VRATOPTR430199();
            VRATOPTR430200();
            VRATOPTR430207();
            VRATOPTR430212();
            VRATOPTR430213();
            VRATOPTR430314();
            VRATOPTR430705();
            VRATOPTR431006();
            VRATOPTR431007();
            VRATOPTR431008();
            VRATOPTR431009();
            VRATOPTR431030();
            VRATOPTR431031();
            VRATOPTR431043();
            VRATOPTR432001();
            VRATOPTR438080();
            VRATOPTR439030();
            VRATOPTR440028();
            VRATOPTR440100();
            VRATOPTR440101();
            VRATOPTR440102();
            VRATOPTR440103();
            VRATOPTR440104();
            VRATOPTR440105();
            VRATOPTR440106();
            VRATOPTR440107();
            VRATOPTR440108();
            VRATOPTR440109();
            VRATOPTR440110();
            VRATOPTR440111();
            VRATOPTR440112();
            VRATOPTR440113();
            VRATOPTR440114();
            VRATOPTR440115();
            VRATOPTR440116();
            VRATOPTR440117();
            VRATOPTR440118();
            VRATOPTR440119();
            VRATOPTR440120();
            VRATOPTR440121();
            VRATOPTR440122();
            VRATOPTR440123();
            VRATOPTR440124();
            VRATOPTR440125();
            VRATOPTR440126();
            VRATOPTR440127();
            VRATOPTR440128();
            VRATOPTR440129();
            VRATOPTR440130();
            VRATOPTR440131();
            VRATOPTR440132();
            VRATOPTR440133();
            VRATOPTR440134();
            VRATOPTR440135();
            VRATOPTR440136();
            VRATOPTR440137();
            VRATOPTR440138();
            VRATOPTR440139();
            VRATOPTR440140();
            VRATOPTR440141();
            VRATOPTR440142();
            VRATOPTR440143();
            VRATOPTR440144();
            VRATOPTR440145();
            VRATOPTR440146();
            VRATOPTR440147();
            VRATOPTR440148();
            VRATOPTR440149();
            VRATOPTR440150();
            VRATOPTR440151();
            VRATOPTR440152();
            VRATOPTR440153();
            VRATOPTR440154();
            VRATOPTR440155();
            VRATOPTR440156();
            VRATOPTR440157();
            VRATOPTR440158();
            VRATOPTR440159();
            VRATOPTR440160();
            VRATOPTR440161();
            VRATOPTR440162();
            VRATOPTR440163();
            VRATOPTR440164();
            VRATOPTR440165();
            VRATOPTR440166();
            VRATOPTR440167();
            VRATOPTR440168();
            VRATOPTR440169();
            VRATOPTR440170();
            VRATOPTR440300();
            VRATOPTR440301();
            VRATOPTR440302();
            VRATOPTR440303();
            VRATOPTR440304();
            VRATOPTR440305();
            VRATOPTR440306();
            VRATOPTR440307();
            VRATOPTR440308();
            VRATOPTR440309();
            VRATOPTR440310();
            VRATOPTR440311();
            VRATOPTR440312();
            VRATOPTR440313();
            VRATOPTR440314();
            VRATOPTR440315();
            VRATOPTR440316();
            VRATOPTR440317();
            VRATOPTR440318();
            VRATOPTR440319();
            VRATOPTR440320();
            VRATOPTR440361();
            VRATOPTR440362();
            VRATOPTR440363();
            VRATOPTR440364();
            VRATOPTR440365();
            VRATOPTR440366();
            VRATOPTR440367();
            VRATOPTR440368();
            VRATOPTR440369();
            VRATOPTR440370();
            VRATOPTR440371();
            VRATOPTR440372();
            VRATOPTR440377();
            VRATOPTRW00001();
            VRATOPTRW00002();
            VRATOPTRW00003();
            VRATOPTRW00004();
            VRATOPTRW00005();
            VRATOPTRW00006();
            VRATOPTRW00007();
            VRATOPTRW00008();
            VRATOPTRW00009();
            VRATOPTRW00010();
            VRATOPTRW00011();
            VRATOPTRW00012();
            VRATOPTRW01014();
            VRATOPTRW00015();

            #region Repeating report.RP_StatementOfDistributionCollection  
            if (report.RP_StatementOfDistributionCollection != null)
            {
                int itemIndex2 = 0;
                foreach (PTR2025.RP_StatementOfDistribution statementOfDistribution in report.RP_StatementOfDistributionCollection)
                {
                    itemIndex2++;
                    VRATOPTR430136(statementOfDistribution, itemIndex2);
                    VRATOPTR430139(statementOfDistribution, itemIndex2);
                    VRATOPTR430141(statementOfDistribution, itemIndex2);
                    VRATOPTR430143(statementOfDistribution, itemIndex2);
                    VRATOPTR430146(statementOfDistribution, itemIndex2);
                    VRATOPTR430148(statementOfDistribution, itemIndex2);
                    VRATOPTR430149(statementOfDistribution, itemIndex2);
                    VRATOPTR431049(statementOfDistribution, itemIndex2);
                    VRATOPTR431050(statementOfDistribution, itemIndex2);
                    VRATOPTR431051(statementOfDistribution, itemIndex2);
                    VRATOPTR438036(statementOfDistribution, itemIndex2);
                    VRATOPTR440321(statementOfDistribution, itemIndex2);
                    VRATOPTR440322(statementOfDistribution, itemIndex2);
                    VRATOPTR440323(statementOfDistribution, itemIndex2);
                    VRATOPTR440324(statementOfDistribution, itemIndex2);
                    VRATOPTR440325(statementOfDistribution, itemIndex2);
                    VRATOPTR440326(statementOfDistribution, itemIndex2);
                    VRATOPTR440327(statementOfDistribution, itemIndex2);
                    VRATOPTR440328(statementOfDistribution, itemIndex2);
                    VRATOPTR440329(statementOfDistribution, itemIndex2);
                    VRATOPTR440330(statementOfDistribution, itemIndex2);
                    VRATOPTR440331(statementOfDistribution, itemIndex2);
                    VRATOPTR440332(statementOfDistribution, itemIndex2);
                    VRATOPTR440333(statementOfDistribution, itemIndex2);
                    VRATOPTR440334(statementOfDistribution, itemIndex2);
                    VRATOPTR440335(statementOfDistribution, itemIndex2);
                    VRATOPTR440336(statementOfDistribution, itemIndex2);
                    VRATOPTR440337(statementOfDistribution, itemIndex2);
                    VRATOPTR440338(statementOfDistribution, itemIndex2);
                    VRATOPTR440339(statementOfDistribution, itemIndex2);
                    VRATOPTR440340(statementOfDistribution, itemIndex2);
                    VRATOPTR440341(statementOfDistribution, itemIndex2);
                    VRATOPTR440342(statementOfDistribution, itemIndex2);
                    VRATOPTR440343(statementOfDistribution, itemIndex2);
                    VRATOPTR440344(statementOfDistribution, itemIndex2);
                    VRATOPTR440345(statementOfDistribution, itemIndex2);
                    VRATOPTR440346(statementOfDistribution, itemIndex2);
                    VRATOPTR440347(statementOfDistribution, itemIndex2);
                    VRATOPTR440348(statementOfDistribution, itemIndex2);
                    VRATOPTR440349(statementOfDistribution, itemIndex2);
                    VRATOPTR440350(statementOfDistribution, itemIndex2);
                    VRATOPTR440351(statementOfDistribution, itemIndex2);
                    VRATOPTR440352(statementOfDistribution, itemIndex2);
                    VRATOPTR440353(statementOfDistribution, itemIndex2);
                    VRATOPTR440354(statementOfDistribution, itemIndex2);
                    VRATOPTR440355(statementOfDistribution, itemIndex2);
                    VRATOPTR440356(statementOfDistribution, itemIndex2);
                    VRATOPTR440357(statementOfDistribution, itemIndex2);
                    VRATOPTR440358(statementOfDistribution, itemIndex2);
                    VRATOPTR440359(statementOfDistribution, itemIndex2);
                    VRATOPTR440360(statementOfDistribution, itemIndex2);
                }
            }

            #endregion Repeating report.RP_StatementOfDistributionCollection   

            foreach (ProcessMessageDocument currentProcessMessage in response)
            {
                if (currentProcessMessage.Parameters != null)
                {
                    foreach (ProcessMessageParameter currentParameter in currentProcessMessage.Parameters.Cast<ProcessMessageParameter>())
                    {
                        if (string.IsNullOrEmpty(currentParameter.Name))
                        {
                            currentParameter.Name = _emptyParameterValue;
                        }

                        if (currentParameter.Name.Length > _maxParameterNameLength)
                        {
                            currentParameter.Name = currentParameter.Name.Substring(0, _maxParameterNameLength - 1);
                        }

                        if (string.IsNullOrEmpty(currentParameter.Value))
                        {
                            currentParameter.Value = _emptyParameterValue;
                        }

                        if (currentParameter.Value.Length > _maxParameterValueLength)
                        {
                            currentParameter.Value = currentParameter.Value.Substring(0, _maxParameterValueLength - 1);
                        }
                    }
                }
            }

            return response;
        }



        #region VR.ATO.PTR.430001
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430001()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430001
            Depreciation expenses must not exceed total Small business entity depreciating assets
    
            Legacy Rule Format:
            ^PTR54 < (^PTR141 + ^PTR142)

            Technical Business Rule Format:
            ^PTR54 < (^PTR141 + ^PTR142)
    
            Data Elements:
    
            ^PTR54 = PTR:RP:Income:BusinessIncomeAndExpenses:BusinessExpenses:Expense.DepreciationAndAmortisation.Amount
    
            ^PTR141 = PTR:RP:LowCostDeduction:Expense.DepreciatingAssets.SmallBusinessEntity.Amount
    
            ^PTR142 = PTR:RP:GeneralPoolDeduction:Expense.DepreciatingAssets.SmallBusinessEntity.Amount
            */
            assertion = (report.PTR54.GetValueOrDefault() < report.PTR141.GetValueOrDefault() + report.PTR142.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Small business entity depreciating assets must not exceed Depreciation expenses",
                    LongDescription = @"The sum of deductions for 'Small business entity depreciating assets' must not exceed the 'Depreciation expenses' amount",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:BusinessIncomeAndExpenses/tns:BusinessExpenses/tns:DepreciationAndAmortisationA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430001" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR54", Value = GetValueOrEmpty(report.PTR54) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR141", Value = GetValueOrEmpty(report.PTR141) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR142", Value = GetValueOrEmpty(report.PTR142) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430009
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430009()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430009
            Previous postal address must not be the same as current postal address
    
            Legacy Rule Format:
            ^PTR14 = ^PTR9 AND ^PTR212 = ^PTR211 AND ^PTR15 = ^PTR10 AND ^PTR16 = ^PTR11 AND ^PTR17 = ^PTR12

            Technical Business Rule Format:
            ^PTR14 = ^PTR9 AND ^PTR212 = ^PTR211 AND ^PTR15 = ^PTR10 AND ^PTR16 = ^PTR11 AND ^PTR17 = ^PTR12
    
            Data Elements:
    
            ^PTR14 = PTR:RP:PreviousPostalAddressDetails:AddressDetails.Line1.Text
    
            ^PTR9 = PTR:RP:CurrentPostalAddressDetails:AddressDetails.Line1.Text
    
            ^PTR10 = PTR:RP:CurrentPostalAddressDetails:AddressDetails.LocalityName.Text
    
            ^PTR11 = PTR:RP:CurrentPostalAddressDetails:AddressDetails.StateOrTerritory.Code
    
            ^PTR12 = PTR:RP:CurrentPostalAddressDetails:AddressDetails.Postcode.Text
    
            ^PTR15 = PTR:RP:PreviousPostalAddressDetails:AddressDetails.LocalityName.Text
    
            ^PTR16 = PTR:RP:PreviousPostalAddressDetails:AddressDetails.StateOrTerritory.Code
    
            ^PTR17 = PTR:RP:PreviousPostalAddressDetails:AddressDetails.Postcode.Text
    
            ^PTR211 = PTR:RP:CurrentPostalAddressDetails:AddressDetails.Line2.Text
    
            ^PTR212 = PTR:RP:PreviousPostalAddressDetails:AddressDetails.Line2.Text
            */
            assertion = (report.PTR14 == report.PTR9 && report.PTR212 == report.PTR211 && report.PTR15 == report.PTR10 && report.PTR16 == report.PTR11 && report.PTR17 == report.PTR12);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Previous postal address must not be the same as current postal address",
                    LongDescription = @"Previous postal address is only required when an address change has occurred since the last return.  Delete previous postal address or input the old address if a change of address has occurred.",
                    Location = "/tns:PTR/tns:RP/tns:PreviousPostalAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430009" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR14", Value = report.PTR14 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR9", Value = report.PTR9 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR212", Value = GetValueOrEmpty(report.PTR212) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR211", Value = GetValueOrEmpty(report.PTR211) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR15", Value = report.PTR15 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR10", Value = report.PTR10 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR16", Value = GetValueOrEmpty(report.PTR16) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR11", Value = GetValueOrEmpty(report.PTR11) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR17", Value = GetValueOrEmpty(report.PTR17) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR12", Value = GetValueOrEmpty(report.PTR12) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430010
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430010()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430010
            Name of company or trust and name of individual must not both be present
    
            Legacy Rule Format:
            ^PTR20 <> NULL AND ^PTR24 <> NULL

            Technical Business Rule Format:
            ^PTR20 <> NULL AND ^PTR24 <> NULL
    
            Data Elements:
    
            ^PTR20 = PTR:RP:MailRecipient:PersonNameDetails:PersonNameDetails.FamilyName.Text
    
            ^PTR24 = PTR:RP:MailRecipient:OrganisationDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.PTR20 != null && report.PTR24 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.430010",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Name of company or trust and name of individual must not both be present",
                    Location = "/tns:PTR/tns:RP/tns:MailRecipient/tns:PersonNameDetails/tns:FamilyNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430010" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR20", Value = report.PTR20 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR24", Value = report.PTR24 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430012
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430012()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430012
            Name of the partner to whom notices should be sent must be present
    
            Legacy Rule Format:
            ^PTR24 = NULL AND ^PTR20 = NULL

            Technical Business Rule Format:
            ^PTR24 = NULL AND ^PTR20 = NULL
    
            Data Elements:
    
            ^PTR24 = PTR:RP:MailRecipient:OrganisationDetails:OrganisationNameDetails.OrganisationalName.Text
    
            ^PTR20 = PTR:RP:MailRecipient:PersonNameDetails:PersonNameDetails.FamilyName.Text
            */
            assertion = (report.PTR24 == null && report.PTR20 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.430012",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Name of the partner to whom notices should be sent must be present",
                    LongDescription = @"If the partner is an individual, provide the name of the individual. If the partner is a company or trust, provide the name of the company or trust",
                    Location = "/tns:PTR/tns:RP/tns:MailRecipient/tns:OrganisationDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430012" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR24", Value = report.PTR24 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR20", Value = report.PTR20 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430013
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430013()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430013
            Name of the company or trust must be present if the ABN is present
    
            Legacy Rule Format:
            ^PTR24 = NULL AND ^PTR25 <> NULL

            Technical Business Rule Format:
            ^PTR24 = NULL AND ^PTR25 <> NULL
    
            Data Elements:
    
            ^PTR24 = PTR:RP:MailRecipient:OrganisationDetails:OrganisationNameDetails.OrganisationalName.Text
    
            ^PTR25 = PTR:RP:MailRecipient:OrganisationDetails:Identifiers.AustralianBusinessNumber.Identifier
            */
            assertion = (report.PTR24 == null && report.PTR25 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.430013",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Name of the company or trust must be present if the ABN is present",
                    Location = "/tns:PTR/tns:RP/tns:MailRecipient/tns:OrganisationDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430013" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR24", Value = report.PTR24 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR25", Value = GetValueOrEmpty(report.PTR25) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430015
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430015()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430015
            Description of main business activity and total business income must be present
    
            Legacy Rule Format:
            ^PTR31 <> NULL AND (^PTR30 = NULL OR (^PTR35 = NULL AND ^PTR36 = NULL AND ^PTR38 = NULL AND ^PTR39 = NULL AND ^PTR41 = NULL AND ^PTR43 = NULL AND ^PTR44 = NULL))

            Technical Business Rule Format:
            ^PTR31 <> NULL AND (^PTR30 = NULL OR (^PTR35 = NULL AND ^PTR36 = NULL AND ^PTR38 = NULL AND ^PTR39 = NULL AND ^PTR41 = NULL AND ^PTR43 = NULL AND ^PTR44 = NULL))
    
            Data Elements:
    
            ^PTR30 = PTR:RP:OrganisationDetails.MainIncomeActivity.Description
    
            ^PTR31 = PTR:RP:OrganisationDetails.OrganisationIndustry2006Extended.Code
    
            ^PTR35 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^PTR36 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^PTR38 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Remuneration.PaymentToForeignResidentGross.Amount
    
            ^PTR39 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^PTR41 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^PTR43 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Income.Other.Amount
    
            ^PTR44 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Income.Other.Amount
            */
            assertion = (report.PTR31 != null && (report.PTR30 == null || report.PTR35 == null && report.PTR36 == null && report.PTR38 == null && report.PTR39 == null && report.PTR41 == null && report.PTR43 == null && report.PTR44 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430015",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Description of main business activity and total business income must be present",
                    LongDescription = @"If an Industry code is present, then the 'Description of main business activity' and business income (Primary Production and/or Non-Primary Production) must be present",
                    Location = "/tns:PTR/tns:RP/tns:OrganisationDetailsMainIncomeActivityDe",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430015" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR31", Value = GetValueOrEmpty(report.PTR31) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR30", Value = GetValueOrEmpty(report.PTR30) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR35", Value = GetValueOrEmpty(report.PTR35) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR36", Value = GetValueOrEmpty(report.PTR36) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR38", Value = GetValueOrEmpty(report.PTR38) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR39", Value = GetValueOrEmpty(report.PTR39) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR41", Value = GetValueOrEmpty(report.PTR41) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR43", Value = GetValueOrEmpty(report.PTR43) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR44", Value = GetValueOrEmpty(report.PTR44) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430016
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430016()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430016
            Description of main business and industry code must be present
    
            Legacy Rule Format:
            ^PTR32 <> NULL AND (^PTR31 = NULL OR ^PTR30 = NULL)

            Technical Business Rule Format:
            ^PTR32 <> NULL AND (^PTR31 = NULL OR ^PTR30 = NULL)
    
            Data Elements:
    
            ^PTR30 = PTR:RP:OrganisationDetails.MainIncomeActivity.Description
    
            ^PTR31 = PTR:RP:OrganisationDetails.OrganisationIndustry2006Extended.Code
    
            ^PTR32 = PTR:RP:OrganisationDetails.ActivityEvent.Code
            */
            assertion = (report.PTR32 != null && (report.PTR31 == null || report.PTR30 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430016",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Description of main business and industry code must be present",
                    LongDescription = @"If Status of Business code is present, the Description of main business activity and Industry code are also required",
                    Location = "/tns:PTR/tns:RP/tns:OrganisationDetailsMainIncomeActivityDe",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430016" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR32", Value = GetValueOrEmpty(report.PTR32) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR31", Value = GetValueOrEmpty(report.PTR31) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR30", Value = GetValueOrEmpty(report.PTR30) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430017
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430017()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430017
            PP business income must be present
    
            Legacy Rule Format:
            ^PTR35 = NULL AND ^PTR39 = NULL AND ^PTR43 = NULL AND ^PTR61 <> NULL

            Technical Business Rule Format:
            ^PTR35 = NULL AND ^PTR39 = NULL AND ^PTR43 = NULL AND ^PTR61 <> NULL
    
            Data Elements:
    
            ^PTR35 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^PTR39 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^PTR43 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Income.Other.Amount
    
            ^PTR61 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Income.Net.Amount
            */
            assertion = (report.PTR35 == null && report.PTR39 == null && report.PTR43 == null && report.PTR61 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430017",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"PP business income must be present",
                    LongDescription = @"Net primary production income/loss amount is present without total business income from primary production (Gross payments subject to Foreign Resident Withholding, Gross payments where ABN not quoted, Assessable government industry payments, Other business income).",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:BusinessIncomeAndExpenses/tns:IncomePrimaryProduction/tns:RemunerationABNNotQuotedPaymentGrossA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430017" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR35", Value = GetValueOrEmpty(report.PTR35) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR39", Value = GetValueOrEmpty(report.PTR39) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR43", Value = GetValueOrEmpty(report.PTR43) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR61", Value = GetValueOrEmpty(report.PTR61) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430018
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430018()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430018
            Total business income amount required
    
            Legacy Rule Format:
            ^PTR35 = NULL AND ^PTR36 = NULL AND ^PTR38 = NULL AND ^PTR39 = NULL AND ^PTR41 = NULL AND ^PTR43 = NULL AND ^PTR44 = NULL AND (^PTR116 <> NULL OR ^PTR117 <> NULL)

            Technical Business Rule Format:
            ^PTR35 = NULL AND ^PTR36 = NULL AND ^PTR38 = NULL AND ^PTR39 = NULL AND ^PTR41 = NULL AND ^PTR43 = NULL AND ^PTR44 = NULL AND (^PTR116 <> NULL OR ^PTR117 <> NULL)
    
            Data Elements:
    
            ^PTR35 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^PTR36 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^PTR38 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Remuneration.PaymentToForeignResidentGross.Amount
    
            ^PTR39 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^PTR41 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^PTR43 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Income.Other.Amount
    
            ^PTR44 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Income.Other.Amount
    
            ^PTR116 = PTR:RP:BusinessName:OrganisationNameDetails.OrganisationalName.Text
    
            ^PTR117 = PTR:RP:BusinessAddressDetails:AddressDetails.Line1.Text
            */
            assertion = (report.PTR35 == null && report.PTR36 == null && report.PTR38 == null && report.PTR39 == null && report.PTR41 == null && report.PTR43 == null && report.PTR44 == null && (report.PTR116 != null || report.PTR117 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430018",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total business income amount required",
                    LongDescription = @"Business name of main business or street address of main business is present and total business income (primary production or non-primary production) is missing at business income and expenses.",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:BusinessIncomeAndExpenses/tns:IncomePrimaryProduction/tns:RemunerationABNNotQuotedPaymentGrossA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430018" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR35", Value = GetValueOrEmpty(report.PTR35) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR36", Value = GetValueOrEmpty(report.PTR36) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR38", Value = GetValueOrEmpty(report.PTR38) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR39", Value = GetValueOrEmpty(report.PTR39) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR41", Value = GetValueOrEmpty(report.PTR41) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR43", Value = GetValueOrEmpty(report.PTR43) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR44", Value = GetValueOrEmpty(report.PTR44) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR116", Value = report.PTR116 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR117", Value = report.PTR117 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430022
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430022()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430022
            Non-PP business income must be present
    
            Legacy Rule Format:
            ^PTR36 = NULL AND ^PTR38 = NULL AND ^PTR41 = NULL AND ^PTR44 = NULL AND ^PTR62 <> NULL

            Technical Business Rule Format:
            ^PTR36 = NULL AND ^PTR38 = NULL AND ^PTR41 = NULL AND ^PTR44 = NULL AND ^PTR62 <> NULL
    
            Data Elements:
    
            ^PTR36 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^PTR38 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Remuneration.PaymentToForeignResidentGross.Amount
    
            ^PTR41 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^PTR44 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Income.Other.Amount
    
            ^PTR62 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Income.Net.Amount
            */
            assertion = (report.PTR36 == null && report.PTR38 == null && report.PTR41 == null && report.PTR44 == null && report.PTR62 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430022",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-PP business income must be present",
                    LongDescription = @"Net non-primary production income/loss amount is present without total business income non-primary production (Gross payments where ABN not quoted, Gross payments subject to foreign resident withholding, Assessable government industry payments, Other business income).",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:BusinessIncomeAndExpenses/tns:IncomeNonPrimaryProduction/tns:RemunerationABNNotQuotedPaymentGrossA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430022" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR36", Value = GetValueOrEmpty(report.PTR36) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR38", Value = GetValueOrEmpty(report.PTR38) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR41", Value = GetValueOrEmpty(report.PTR41) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR44", Value = GetValueOrEmpty(report.PTR44) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR62", Value = GetValueOrEmpty(report.PTR62) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430023()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430023
            Net non-PP income must be present
    
            Legacy Rule Format:
            ^PTR62 = NULL AND (^PTR36 <> NULL OR ^PTR38 <> NULL OR ^PTR41 <> NULL OR ^PTR44 <> NULL)

            Technical Business Rule Format:
            ^PTR62 = NULL AND (^PTR36 <> NULL OR ^PTR38 <> NULL OR ^PTR41 <> NULL OR ^PTR44 <> NULL)
    
            Data Elements:
    
            ^PTR62 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Income.Net.Amount
    
            ^PTR36 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^PTR38 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Remuneration.PaymentToForeignResidentGross.Amount
    
            ^PTR41 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^PTR44 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Income.Other.Amount
            */
            assertion = (report.PTR62 == null && (report.PTR36 != null || report.PTR38 != null || report.PTR41 != null || report.PTR44 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Net non-PP income must be present",
                    LongDescription = @"Total business income from non-primary production (Gross payments where ABN not quoted, Gross payments subject to foreign resident withholding, Assessable government industry payments, Other business income) is present without net non-primary production income/loss amount.",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:BusinessIncomeAndExpenses/tns:IncomeNonPrimaryProduction/tns:NetA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430023" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR62", Value = GetValueOrEmpty(report.PTR62) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR36", Value = GetValueOrEmpty(report.PTR36) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR38", Value = GetValueOrEmpty(report.PTR38) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR41", Value = GetValueOrEmpty(report.PTR41) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR44", Value = GetValueOrEmpty(report.PTR44) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430027
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430027()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430027
            Credit for tax withheld - foreign resident withholding (exlcluding capital gains) must not exceed 50% of gross payments
    
            Legacy Rule Format:
            ^PTR350 > 0 AND ^PTR350 > ((^PTR38 * 0.50) + 1)

            Technical Business Rule Format:
            ^PTR350 > 0 AND ^PTR350 > ((^PTR38 * 0.50) + 1)
    
            Data Elements:
    
            ^PTR350 = PTR:RP:Income:BusinessIncomeAndExpenses:PayAsYouGoWithholding:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromForeignResidents.Amount
    
            ^PTR38 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Remuneration.PaymentToForeignResidentGross.Amount
            */
            assertion = (report.PTR350.GetValueOrDefault() > 0 && report.PTR350.GetValueOrDefault() > report.PTR38.GetValueOrDefault() * 0.50M + 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.430027",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Credit for tax withheld - foreign resident withholding (excluding capital gains) must not exceed 50% of gross payments",
                    LongDescription = @"Credit for tax withheld - foreign resident withholding (excluding capital gains) must not exceed 50% of total of Primary production and Non-primary production Gross payments subject to foreign resident withholding",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:BusinessIncomeAndExpenses/tns:PayAsYouGoWithholding/tns:TaxCreditForAmountsWithheldFromForeignResidentsA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430027" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR350", Value = GetValueOrEmpty(report.PTR350) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR38", Value = GetValueOrEmpty(report.PTR38) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430030
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430030()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430030
            PP Assessable government industry payments amount must be present
    
            Legacy Rule Format:
            (^PTR39 = 0 OR ^PTR39 = NULL) AND ^PTR258 = TRUE

            Technical Business Rule Format:
            (^PTR39 = 0 OR ^PTR39 = NULL) AND ^PTR258 = TRUE
    
            Data Elements:
    
            ^PTR39 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^PTR258 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Indicator
            */
            assertion = ((report.PTR39 == 0 || report.PTR39 == null) && report.PTR258 == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430030",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"PP Assessable government industry payments amount must be present",
                    LongDescription = @"If Primary production - Assessable government industry payments action code is present, Primary production - Assessable government industry payments amount must be present and not zero",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:BusinessIncomeAndExpenses/tns:IncomePrimaryProduction/tns:GovernmentFundingGovernmentIndustryPaymentsAssessableA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430030" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR39", Value = GetValueOrEmpty(report.PTR39) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR258", Value = GetValueOrEmpty(report.PTR258) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430032
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430032()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430032
            Non-PP Assessable government industry payments amount must be present
    
            Legacy Rule Format:
            (^PTR41 = 0 OR ^PTR41 = NULL) AND ^PTR259 = TRUE

            Technical Business Rule Format:
            (^PTR41 = 0 OR ^PTR41 = NULL) AND ^PTR259 = TRUE
    
            Data Elements:
    
            ^PTR41 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^PTR259 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Indicator
            */
            assertion = ((report.PTR41 == 0 || report.PTR41 == null) && report.PTR259 == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430032",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-PP Assessable government industry payments amount must be present",
                    LongDescription = @"If Non-primary production - Assessable government industry payments action code is present, Non-primary production - Assessable government industry payments must be present and not zero",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:BusinessIncomeAndExpenses/tns:IncomeNonPrimaryProduction/tns:GovernmentFundingGovernmentIndustryPaymentsAssessableA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430032" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR41", Value = GetValueOrEmpty(report.PTR41) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR259", Value = GetValueOrEmpty(report.PTR259) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430040
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430040()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430040
            Overseas royalty expenses must not exceed Total royalty expenses
    
            Legacy Rule Format:
            ^PTR138 > 0 AND ^PTR138 > ^PTR347

            Technical Business Rule Format:
            ^PTR138 > 0 AND ^PTR138 > ^PTR347
    
            Data Elements:
    
            ^PTR138 = PTR:RP:OverseasTransactions:Expense.Royalties.Amount
    
            ^PTR347 = PTR:RP:Income:BusinessIncomeAndExpenses:BusinessExpenses:Expense.Royalties.Amount
            */
            assertion = (report.PTR138.GetValueOrDefault() > 0 && report.PTR138.GetValueOrDefault() > report.PTR347.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430040",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Overseas royalty expenses must not exceed Total royalty expenses",
                    LongDescription = @"Overseas royalty expenses - under Business and professional items must not be greater than Total royalty expenses - under Business income and expenses",
                    Location = "/tns:PTR/tns:RP/tns:OverseasTransactions/tns:ExpenseRoyaltiesA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430040" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR138", Value = GetValueOrEmpty(report.PTR138) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR347", Value = GetValueOrEmpty(report.PTR347) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430044
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430044()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430044
            Motor vehicle expenses amount must be present if action code is present
    
            Legacy Rule Format:
            (^PTR348 = 0 OR ^PTR348 = NULL) AND ^PTR56 <> NULL

            Technical Business Rule Format:
            (^PTR348 = 0 OR ^PTR348 = NULL) AND ^PTR56 <> NULL
    
            Data Elements:
    
            ^PTR348 = PTR:RP:Income:BusinessIncomeAndExpenses:BusinessExpenses:Expense.Operating.MotorVehicle.Amount
    
            ^PTR56 = PTR:RP:Income:BusinessIncomeAndExpenses:BusinessExpenses:Expense.Operating.MotorVehicleTypeOrClaimMethod.Code
            */
            assertion = ((report.PTR348 == 0 || report.PTR348 == null) && report.PTR56 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.430044",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Motor vehicle expenses amount must be present if action code is present",
                    LongDescription = @"If Motor vehicle expenses action code is present, there must be a motor vehicle expense amount",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:BusinessIncomeAndExpenses/tns:BusinessExpenses/tns:OperatingMotorVehicleA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430044" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR348", Value = GetValueOrEmpty(report.PTR348) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR56", Value = GetValueOrEmpty(report.PTR56) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430046
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430046()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430046
            Motor vehicle expenses action code must be present
    
            Legacy Rule Format:
            ^PTR56 = NULL AND ^PTR348 > 0

            Technical Business Rule Format:
            ^PTR56 = NULL AND ^PTR348 > 0
    
            Data Elements:
    
            ^PTR56 = PTR:RP:Income:BusinessIncomeAndExpenses:BusinessExpenses:Expense.Operating.MotorVehicleTypeOrClaimMethod.Code
    
            ^PTR348 = PTR:RP:Income:BusinessIncomeAndExpenses:BusinessExpenses:Expense.Operating.MotorVehicle.Amount
            */
            assertion = (report.PTR56 == null && report.PTR348.GetValueOrDefault() > 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.440046",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Motor vehicle expenses action code must be present",
                    LongDescription = @"If a motor vehicle expense amount is present, a motor vehicle expense action code must be present. Codes are: S - Cents per Kilometre, B - Logbook, or N - Expenses of other vehicles",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:BusinessIncomeAndExpenses/tns:BusinessExpenses/tns:OperatingMotorVehicleTypeOrClaimMethodC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430046" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR56", Value = GetValueOrEmpty(report.PTR56) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR348", Value = GetValueOrEmpty(report.PTR348) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430049
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430049()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430049
            PP Net income or loss from business must be present
    
            Legacy Rule Format:
            ^PTR61 = NULL AND (^PTR35 <> NULL OR ^PTR39 <> NULL OR ^PTR43 <> NULL)

            Technical Business Rule Format:
            ^PTR61 = NULL AND (^PTR35 <> NULL OR ^PTR39 <> NULL OR ^PTR43 <> NULL)
    
            Data Elements:
    
            ^PTR61 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Income.Net.Amount
    
            ^PTR35 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^PTR39 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^PTR43 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Income.Other.Amount
            */
            assertion = (report.PTR61 == null && (report.PTR35 != null || report.PTR39 != null || report.PTR43 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430049",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"PP Net income or loss from business must be present",
                    LongDescription = @"If Total business income from primary production (Gross payments, Assessable government industry payments, other business income) is present, Primary production Net primary production income or loss from business amount must be present",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:BusinessIncomeAndExpenses/tns:IncomePrimaryProduction/tns:NetA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430049" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR61", Value = GetValueOrEmpty(report.PTR61) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR35", Value = GetValueOrEmpty(report.PTR35) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR39", Value = GetValueOrEmpty(report.PTR39) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR43", Value = GetValueOrEmpty(report.PTR43) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430050
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430050()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430050
            Total Net income or loss from business must equal total Income minus total Expenses
    
            Legacy Rule Format:
            (^PTR61 + ^PTR62) <> (^PTR35 + ^PTR36 + ^PTR38 + ^PTR39 + ^PTR41 + ^PTR43 + ^PTR44 - (^PTR45 + ^PTR46 + ^PTR47 + ^PTR343 + ^PTR344 + ^PTR50 + ^PTR345 + ^PTR346 + ^PTR347 + ^PTR54 + ^PTR348 + ^PTR349 + ^PTR58) + ^PTR59 + ^PTR328)

            Technical Business Rule Format:
            (^PTR61 + ^PTR62) <> (^PTR35 + ^PTR36 + ^PTR38 + ^PTR39 + ^PTR41 + ^PTR43 + ^PTR44 - (^PTR45 + ^PTR46 + ^PTR47 + ^PTR343 + ^PTR344 + ^PTR50 + ^PTR345 + ^PTR346 + ^PTR347 + ^PTR54 + ^PTR348 + ^PTR349 + ^PTR58) + ^PTR59 + ^PTR328)
    
            Data Elements:
    
            ^PTR61 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Income.Net.Amount
    
            ^PTR35 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^PTR36 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^PTR38 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Remuneration.PaymentToForeignResidentGross.Amount
    
            ^PTR39 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^PTR41 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^PTR43 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Income.Other.Amount
    
            ^PTR44 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Income.Other.Amount
    
            ^PTR45 = PTR:RP:Income:BusinessIncomeAndExpenses:BusinessExpenses:Expense.ForeignResidentWithholding.Amount
    
            ^PTR46 = PTR:RP:Income:BusinessIncomeAndExpenses:BusinessExpenses:Remuneration.WagesAndSalaries.PaymentsContractorAndConsultant.Amount
    
            ^PTR47 = PTR:RP:Income:BusinessIncomeAndExpenses:BusinessExpenses:SuperannuationContribution.EmployerContributions.Amount
    
            ^PTR50 = PTR:RP:Income:BusinessIncomeAndExpenses:BusinessExpenses:Expense.Operating.LeaseTotal.Amount
    
            ^PTR54 = PTR:RP:Income:BusinessIncomeAndExpenses:BusinessExpenses:Expense.DepreciationAndAmortisation.Amount
    
            ^PTR58 = PTR:RP:Income:BusinessIncomeAndExpenses:BusinessExpenses:Expense.DeductibleOtherTotal.Amount
    
            ^PTR59 = PTR:RP:Income:BusinessIncomeAndExpenses:Income.ReconciliationAdjustmentTotal.Amount
    
            ^PTR62 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Income.Net.Amount
    
            ^PTR328 = PTR:RP:Income:BusinessIncomeAndExpenses:BusinessExpenses:Expense.ReconciliationAdjustmentTotal.Amount
    
            ^PTR343 = PTR:RP:Income:BusinessIncomeAndExpenses:BusinessExpenses:Expense.Operating.CostOfSales.Amount
    
            ^PTR344 = PTR:RP:Income:BusinessIncomeAndExpenses:BusinessExpenses:Expense.Operating.BadDebts.Amount
    
            ^PTR345 = PTR:RP:Income:BusinessIncomeAndExpenses:BusinessExpenses:Expense.Operating.Rent.Amount
    
            ^PTR346 = PTR:RP:Income:BusinessIncomeAndExpenses:BusinessExpenses:Expense.Operating.Interest.Amount
    
            ^PTR347 = PTR:RP:Income:BusinessIncomeAndExpenses:BusinessExpenses:Expense.Royalties.Amount
    
            ^PTR348 = PTR:RP:Income:BusinessIncomeAndExpenses:BusinessExpenses:Expense.Operating.MotorVehicle.Amount
    
            ^PTR349 = PTR:RP:Income:BusinessIncomeAndExpenses:BusinessExpenses:Expense.Operating.RepairsAndMaintenance.Amount
            */
            assertion = (report.PTR61.GetValueOrDefault() + report.PTR62.GetValueOrDefault() != report.PTR35.GetValueOrDefault() + report.PTR36.GetValueOrDefault() + report.PTR38.GetValueOrDefault() + report.PTR39.GetValueOrDefault() + report.PTR41.GetValueOrDefault() + report.PTR43.GetValueOrDefault() + report.PTR44.GetValueOrDefault() - (report.PTR45.GetValueOrDefault() + report.PTR46.GetValueOrDefault() + report.PTR47.GetValueOrDefault() + report.PTR343.GetValueOrDefault() + report.PTR344.GetValueOrDefault() + report.PTR50.GetValueOrDefault() + report.PTR345.GetValueOrDefault() + report.PTR346.GetValueOrDefault() + report.PTR347.GetValueOrDefault() + report.PTR54.GetValueOrDefault() + report.PTR348.GetValueOrDefault() + report.PTR349.GetValueOrDefault() + report.PTR58.GetValueOrDefault()) + report.PTR59.GetValueOrDefault() + report.PTR328.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430050",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total Net income or loss must equal total Income, plus Reconciliation items, minus total Expenses",
                    LongDescription = @"The total of Primary production and Non-primary production Net business income must equal the total of all income amounts, plus Reconciliation items, minus the sum of all expenses amounts",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:BusinessIncomeAndExpenses/tns:IncomePrimaryProduction/tns:NetA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430050" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR61", Value = GetValueOrEmpty(report.PTR61) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR62", Value = GetValueOrEmpty(report.PTR62) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR35", Value = GetValueOrEmpty(report.PTR35) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR36", Value = GetValueOrEmpty(report.PTR36) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR38", Value = GetValueOrEmpty(report.PTR38) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR39", Value = GetValueOrEmpty(report.PTR39) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR41", Value = GetValueOrEmpty(report.PTR41) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR43", Value = GetValueOrEmpty(report.PTR43) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR44", Value = GetValueOrEmpty(report.PTR44) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR45", Value = GetValueOrEmpty(report.PTR45) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR46", Value = GetValueOrEmpty(report.PTR46) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR47", Value = GetValueOrEmpty(report.PTR47) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR343", Value = GetValueOrEmpty(report.PTR343) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR344", Value = GetValueOrEmpty(report.PTR344) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR50", Value = GetValueOrEmpty(report.PTR50) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR345", Value = GetValueOrEmpty(report.PTR345) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR346", Value = GetValueOrEmpty(report.PTR346) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR347", Value = GetValueOrEmpty(report.PTR347) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR54", Value = GetValueOrEmpty(report.PTR54) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR348", Value = GetValueOrEmpty(report.PTR348) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR349", Value = GetValueOrEmpty(report.PTR349) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR58", Value = GetValueOrEmpty(report.PTR58) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR59", Value = GetValueOrEmpty(report.PTR59) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR328", Value = GetValueOrEmpty(report.PTR328) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430051
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430051()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430051
            Net income or loss from business must be present
    
            Legacy Rule Format:
            ^PTR61 = NULL AND ^PTR62 = NULL AND (^PTR63 <> NULL OR ^PTR350 <> NULL)

            Technical Business Rule Format:
            ^PTR61 = NULL AND ^PTR62 = NULL AND (^PTR63 <> NULL OR ^PTR350 <> NULL)
    
            Data Elements:
    
            ^PTR61 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Income.Net.Amount
    
            ^PTR62 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Income.Net.Amount
    
            ^PTR63 = PTR:RP:Income:BusinessIncomeAndExpenses:PayAsYouGoWithholding:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount
    
            ^PTR350 = PTR:RP:Income:BusinessIncomeAndExpenses:PayAsYouGoWithholding:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromForeignResidents.Amount
            */
            assertion = (report.PTR61 == null && report.PTR62 == null && (report.PTR63 != null || report.PTR350 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430051",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Net income or loss from business must be present",
                    LongDescription = @"If Tax withheld where ABN not quoted is present or Credit for tax withheld - Foreign resident withholding is present, then Net income or loss from business must be present",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:BusinessIncomeAndExpenses/tns:IncomePrimaryProduction/tns:NetA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430051" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR61", Value = GetValueOrEmpty(report.PTR61) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR62", Value = GetValueOrEmpty(report.PTR62) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR63", Value = GetValueOrEmpty(report.PTR63) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR350", Value = GetValueOrEmpty(report.PTR350) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430052
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430052()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430052
            PP Net income/loss from business or distribution from partnerships or trusts must be present
    
            Legacy Rule Format:
            AnyOccurrence(^PTR158, ^PTR158 > 0) AND ^PTR61 = NULL AND ^PTR65 = NULL AND ^PTR66 = NULL

            Technical Business Rule Format:
            AnyOccurrence(^PTR158, ^PTR158 > 0) AND ^PTR61 = NULL AND ^PTR65 = NULL AND ^PTR66 = NULL
    
            Data Elements:
    
            ^PTR61 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Income.Net.Amount
    
            ^PTR65 = PTR:RP:Income:PartnershipsTrustsPrimaryProduction:Income.PartnershipDistributionGross.Amount
    
            ^PTR66 = PTR:RP:Income:PartnershipsTrustsPrimaryProduction:Income.TrustShareNet.Amount
    
            ^PTR158 = PTR:RP:StatementOfDistribution:IncomeOrLoss:ShareOfIncomePrimaryProduction:Income.BeneficiaryShare.Amount
            */
            assertion = (report.RP_StatementOfDistributionCollection == null ? false : report.RP_StatementOfDistributionCollection.Any(PTR158Repeat => PTR158Repeat.PTR158.GetValueOrDefault() > 0)) && report.PTR61 == null && report.PTR65 == null && report.PTR66 == null;
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.430052",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"PP Net income or loss from business or a distribution from partnerships or trusts must be present",
                    LongDescription = @"If an amount, greater than zero, is present for a Partner in the distribution statement at Share of income  - Primary production, there must also be an amount greater than zero at Primary production Net income or loss from business, or Primary production - Distribution from partnerships, or Primary production - Distribution from trusts",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:BusinessIncomeAndExpenses/tns:IncomePrimaryProduction/tns:NetA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430052" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR158", Value = report.RP_StatementOfDistributionCollection.First(PTR158Repeat => PTR158Repeat.PTR158.GetValueOrDefault() > 0).ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR61", Value = GetValueOrEmpty(report.PTR61) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR65", Value = GetValueOrEmpty(report.PTR65) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR66", Value = GetValueOrEmpty(report.PTR66) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430056
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430056()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430056
            Tax withheld where ABN not quoted cannot be greater than 50% of Gross payments (PP and Non-PP) where ABN not quoted.
    
            Legacy Rule Format:
            ^PTR63 > 0 AND ^PTR63 > (((^PTR35 + ^PTR36) * 0.50) + 1)

            Technical Business Rule Format:
            ^PTR63 > 0 AND ^PTR63 > (((^PTR35 + ^PTR36) * 0.50) + 1)
    
            Data Elements:
    
            ^PTR63 = PTR:RP:Income:BusinessIncomeAndExpenses:PayAsYouGoWithholding:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount
    
            ^PTR35 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^PTR36 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Remuneration.ABNNotQuotedPaymentGross.Amount
            */
            assertion = (report.PTR63.GetValueOrDefault() > 0 && report.PTR63.GetValueOrDefault() > (report.PTR35.GetValueOrDefault() + report.PTR36.GetValueOrDefault()) * 0.50M + 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430056",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax withheld where ABN not quoted cannot be greater than 50% of Gross payments (PP and Non-PP) where ABN not quoted.",
                    LongDescription = @"Tax withheld where ABN not quoted cannot be greater than 50% of Gross payments (Primary production and Non-primary production) where ABN not quoted.",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:BusinessIncomeAndExpenses/tns:PayAsYouGoWithholding/tns:TaxCreditForTaxWithheldWhereABNNotQuotedA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430056" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR63", Value = GetValueOrEmpty(report.PTR63) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR35", Value = GetValueOrEmpty(report.PTR35) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR36", Value = GetValueOrEmpty(report.PTR36) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430057
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430057()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430057
            PP or Non-PP Gross payments where ABN not quoted must be present
    
            Legacy Rule Format:
            ^PTR63 <> NULL AND ^PTR35 = NULL AND ^PTR36 = NULL

            Technical Business Rule Format:
            ^PTR63 <> NULL AND ^PTR35 = NULL AND ^PTR36 = NULL
    
            Data Elements:
    
            ^PTR63 = PTR:RP:Income:BusinessIncomeAndExpenses:PayAsYouGoWithholding:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount
    
            ^PTR35 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^PTR36 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Remuneration.ABNNotQuotedPaymentGross.Amount
            */
            assertion = (report.PTR63 != null && report.PTR35 == null && report.PTR36 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430057",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"PP or Non-PP Gross payments where ABN not quoted must be present",
                    LongDescription = @"If Tax withheld where ABN not quoted is present there must be an amount at Gross payments where ABN not quoted for either Primary production or Non-primary production",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:BusinessIncomeAndExpenses/tns:PayAsYouGoWithholding/tns:TaxCreditForTaxWithheldWhereABNNotQuotedA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430057" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR63", Value = GetValueOrEmpty(report.PTR63) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR35", Value = GetValueOrEmpty(report.PTR35) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR36", Value = GetValueOrEmpty(report.PTR36) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430058
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430058()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430058
            Tax withheld where ABN not quoted must be present
    
            Legacy Rule Format:
            ^PTR63 = NULL AND (^PTR35 > 0 OR ^PTR36 > 0)

            Technical Business Rule Format:
            ^PTR63 = NULL AND (^PTR35 > 0 OR ^PTR36 > 0)
    
            Data Elements:
    
            ^PTR63 = PTR:RP:Income:BusinessIncomeAndExpenses:PayAsYouGoWithholding:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount
    
            ^PTR35 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^PTR36 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Remuneration.ABNNotQuotedPaymentGross.Amount
            */
            assertion = (report.PTR63 == null && (report.PTR35.GetValueOrDefault() > 0 || report.PTR36.GetValueOrDefault() > 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430058",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax withheld where ABN not quoted must be present",
                    LongDescription = @"If either Primary production or Non-primary production Gross payments where ABN not quoted are greater than zero, then an amount at Tax withheld where ABN not quoted must be present",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:BusinessIncomeAndExpenses/tns:PayAsYouGoWithholding/tns:TaxCreditForTaxWithheldWhereABNNotQuotedA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430058" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR63", Value = GetValueOrEmpty(report.PTR63) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR35", Value = GetValueOrEmpty(report.PTR35) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR36", Value = GetValueOrEmpty(report.PTR36) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430059
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430059()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430059
            Credit for tax withheld - foreign resident withholding (excluding capital gains) must be present
    
            Legacy Rule Format:
            ^PTR350 = NULL AND ^PTR38 > 0

            Technical Business Rule Format:
            ^PTR350 = NULL AND ^PTR38 > 0
    
            Data Elements:
    
            ^PTR350 = PTR:RP:Income:BusinessIncomeAndExpenses:PayAsYouGoWithholding:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromForeignResidents.Amount
    
            ^PTR38 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Remuneration.PaymentToForeignResidentGross.Amount
            */
            assertion = (report.PTR350 == null && report.PTR38.GetValueOrDefault() > 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.430059",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Credit for tax withheld - foreign resident withholding (excluding capital gains) must be present",
                    LongDescription = @"If Primary production or Non-primary production Gross payments subject to foreign resident withholding is greater than zero, then there must be an amount at Credit for tax withheld - foreign resident withholding (excluding capital gains)",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:BusinessIncomeAndExpenses/tns:PayAsYouGoWithholding/tns:TaxCreditForAmountsWithheldFromForeignResidentsA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430059" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR350", Value = GetValueOrEmpty(report.PTR350) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR38", Value = GetValueOrEmpty(report.PTR38) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430061
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430061()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430061
            PP - Income distribution from partnerships and/or trusts must be present
    
            Legacy Rule Format:
            ^PTR65 = NULL AND ^PTR66 = NULL AND ^PTR68 > 0

            Technical Business Rule Format:
            ^PTR65 = NULL AND ^PTR66 = NULL AND ^PTR68 > 0
    
            Data Elements:
    
            ^PTR65 = PTR:RP:Income:PartnershipsTrustsPrimaryProduction:Income.PartnershipDistributionGross.Amount
    
            ^PTR66 = PTR:RP:Income:PartnershipsTrustsPrimaryProduction:Income.TrustShareNet.Amount
    
            ^PTR68 = PTR:RP:Income:PartnershipsTrustsPrimaryProduction:Expense.DistributionDeductionsOther.Amount
            */
            assertion = (report.PTR65 == null && report.PTR66 == null && report.PTR68.GetValueOrDefault() > 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430061",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"PP - Income distribution from partnerships and/or trusts must be present",
                    LongDescription = @"If there is an amount present at 'Primary production - Deductions relating to distribution', then there must also be an amount at 'Primary production - Distribution from partnerships' or 'Primary production - Distribution from trusts'",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:PartnershipsTrustsPrimaryProduction/tns:PartnershipDistributionGrossA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430061" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR65", Value = GetValueOrEmpty(report.PTR65) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR66", Value = GetValueOrEmpty(report.PTR66) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR68", Value = GetValueOrEmpty(report.PTR68) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430063
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430063()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430063
            Distribution from partnerships or trusts must be present
    
            Legacy Rule Format:
            ^PTR65 = NULL AND ^PTR66 = NULL AND ^PTR69 = NULL AND ^PTR70 = NULL AND ^PTR322 = NULL AND (^PTR72 > 0 OR ^PTR323 > 0 OR ^PTR74 > 0 OR ^PTR75 > 0 OR ^PTR76 > 0)

            Technical Business Rule Format:
            ^PTR65 = NULL AND ^PTR66 = NULL AND ^PTR69 = NULL AND ^PTR70 = NULL AND ^PTR322 = NULL AND (^PTR72 > 0 OR ^PTR323 > 0 OR ^PTR74 > 0 OR ^PTR75 > 0 OR ^PTR76 > 0)
    
            Data Elements:
    
            ^PTR65 = PTR:RP:Income:PartnershipsTrustsPrimaryProduction:Income.PartnershipDistributionGross.Amount
    
            ^PTR66 = PTR:RP:Income:PartnershipsTrustsPrimaryProduction:Income.TrustShareNet.Amount
    
            ^PTR69 = PTR:RP:Income:PartnershipsTrustsNonPrimaryProduction:Income.PartnershipDistributionLessForeignIncome.Amount
    
            ^PTR70 = PTR:RP:Income:PartnershipsTrustsNonPrimaryProduction:Income.TrustShareNetExcludeNetCapitalGainsAndForeignIncomeAndDistributionFranked.Amount
    
            ^PTR72 = PTR:RP:Income:PartnershipsTrustsNonPrimaryProduction:Expense.DistributionDeductionsOther.Amount
    
            ^PTR74 = PTR:RP:Income:ShareOfCreditsFromIncome:IncomeTax.FrankingCredits.FrankingCreditShareReceivedIndirectly.Amount
    
            ^PTR75 = PTR:RP:Income:ShareOfCreditsFromIncome:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromInterestAndDividendsWhereTFNNotQuotedShareReceivedIndirectly.Amount
    
            ^PTR76 = PTR:RP:Income:ShareOfCreditsFromIncome:InternationalDealings.CreditForTaxWithheldFromForeignResidentWithholdingShareReceivedIndirectly.Amount
    
            ^PTR322 = PTR:RP:Income:PartnershipsTrustsNonPrimaryProduction:Income.TrustDistributionFranked.Amount
    
            ^PTR323 = PTR:RP:Income:PartnershipsTrustsNonPrimaryProduction:IncomeTax.Deduction.TrustDistributionFranked.Amount
            */
            assertion = (report.PTR65 == null && report.PTR66 == null && report.PTR69 == null && report.PTR70 == null && report.PTR322 == null && (report.PTR72.GetValueOrDefault() > 0 || report.PTR323.GetValueOrDefault() > 0 || report.PTR74.GetValueOrDefault() > 0 || report.PTR75.GetValueOrDefault() > 0 || report.PTR76.GetValueOrDefault() > 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430063",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Distribution from partnerships or trusts must be present",
                    LongDescription = @"If an amount is present at 'Deductions relating to distribution', 'Share of credit for tax withheld where ABN not quoted', 'Share of franking credit from franked dividends', 'Share of credit for TFN amounts withheld from interest dividends and unit trust distributions', or 'Share of credit for tax withheld from foreign resident withholding'; then there must be an amount present at 'Primary production - Distribution from partnerships', 'Primary production - Distribution from trusts', 'Non-primary production - Distribution from partnerships less foreign income', or 'Distribution from trusts less net capital gain and foreign income'",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:PartnershipsTrustsPrimaryProduction/tns:PartnershipDistributionGrossA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430063" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR65", Value = GetValueOrEmpty(report.PTR65) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR66", Value = GetValueOrEmpty(report.PTR66) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR69", Value = GetValueOrEmpty(report.PTR69) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR70", Value = GetValueOrEmpty(report.PTR70) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR322", Value = GetValueOrEmpty(report.PTR322) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR72", Value = GetValueOrEmpty(report.PTR72) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR323", Value = GetValueOrEmpty(report.PTR323) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR74", Value = GetValueOrEmpty(report.PTR74) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR75", Value = GetValueOrEmpty(report.PTR75) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR76", Value = GetValueOrEmpty(report.PTR76) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430068
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430068()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430068
            Non-PP distribution amount must be present
    
            Legacy Rule Format:
            ^PTR69 = NULL AND ^PTR70 = NULL AND ^PTR72 = NULL AND ^PTR322 = NULL AND ^PTR323 = NULL AND ^PTR74 <> NULL AND ^PTR75 <> NULL

            Technical Business Rule Format:
            ^PTR69 = NULL AND ^PTR70 = NULL AND ^PTR72 = NULL AND ^PTR322 = NULL AND ^PTR323 = NULL AND ^PTR74 <> NULL AND ^PTR75 <> NULL
    
            Data Elements:
    
            ^PTR69 = PTR:RP:Income:PartnershipsTrustsNonPrimaryProduction:Income.PartnershipDistributionLessForeignIncome.Amount
    
            ^PTR70 = PTR:RP:Income:PartnershipsTrustsNonPrimaryProduction:Income.TrustShareNetExcludeNetCapitalGainsAndForeignIncomeAndDistributionFranked.Amount
    
            ^PTR72 = PTR:RP:Income:PartnershipsTrustsNonPrimaryProduction:Expense.DistributionDeductionsOther.Amount
    
            ^PTR74 = PTR:RP:Income:ShareOfCreditsFromIncome:IncomeTax.FrankingCredits.FrankingCreditShareReceivedIndirectly.Amount
    
            ^PTR75 = PTR:RP:Income:ShareOfCreditsFromIncome:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromInterestAndDividendsWhereTFNNotQuotedShareReceivedIndirectly.Amount
    
            ^PTR322 = PTR:RP:Income:PartnershipsTrustsNonPrimaryProduction:Income.TrustDistributionFranked.Amount
    
            ^PTR323 = PTR:RP:Income:PartnershipsTrustsNonPrimaryProduction:IncomeTax.Deduction.TrustDistributionFranked.Amount
            */
            assertion = (report.PTR69 == null && report.PTR70 == null && report.PTR72 == null && report.PTR322 == null && report.PTR323 == null && report.PTR74 != null && report.PTR75 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430068",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-PP distribution amount must be present",
                    LongDescription = @"If there is an amount at Share of franking credits from franked dividends, and Share of credit for TFN amounts withheld from interest, dividends and unit trust distributions; then there must be an amount at Non-Primary Production Distribution from partnerships less foreign income; Distribution from trusts less net capital gain and foreign income; or Deductions relating to distribution",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:PartnershipsTrustsNonPrimaryProduction/tns:PartnershipDistributionLessForeignIncomeA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430068" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR69", Value = GetValueOrEmpty(report.PTR69) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR70", Value = GetValueOrEmpty(report.PTR70) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR72", Value = GetValueOrEmpty(report.PTR72) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR322", Value = GetValueOrEmpty(report.PTR322) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR323", Value = GetValueOrEmpty(report.PTR323) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR74", Value = GetValueOrEmpty(report.PTR74) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR75", Value = GetValueOrEmpty(report.PTR75) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430079
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430079()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430079
            Gross rent must be present
    
            Legacy Rule Format:
            ^PTR77 = NULL AND (^PTR78 > 0 OR ^PTR79 > 0 OR ^PTR80 > 0)

            Technical Business Rule Format:
            ^PTR77 = NULL AND (^PTR78 > 0 OR ^PTR79 > 0 OR ^PTR80 > 0)
    
            Data Elements:
    
            ^PTR77 = PTR:RP:Income:Rent:Income.Operating.RentalIncomeGross.Amount
    
            ^PTR78 = PTR:RP:Income:Rent:Expense.Interest.Amount
    
            ^PTR79 = PTR:RP:Income:Rent:Expense.CapitalWorksDeduction.Amount
    
            ^PTR80 = PTR:RP:Income:Rent:IncomeTax.Deduction.RentalIncomeDeductionsOtherThanInterestAndCapitalWorks.Amount
            */
            assertion = (report.PTR77 == null && (report.PTR78.GetValueOrDefault() > 0 || report.PTR79.GetValueOrDefault() > 0 || report.PTR80.GetValueOrDefault() > 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430079",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Gross rent must be present",
                    LongDescription = @"If Interest deductions, Capital works deductions, or Other rental deductions are greater than zero, Gross rent must be present",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:Rent/tns:OperatingIncomeGrossA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430079" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR77", Value = GetValueOrEmpty(report.PTR77) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR78", Value = GetValueOrEmpty(report.PTR78) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR79", Value = GetValueOrEmpty(report.PTR79) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR80", Value = GetValueOrEmpty(report.PTR80) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430080
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430080()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430080
            Rent deductions must be present
    
            Legacy Rule Format:
            ^PTR78 = NULL AND ^PTR79 = NULL AND ^PTR80 = NULL AND ^PTR77 > 0

            Technical Business Rule Format:
            ^PTR78 = NULL AND ^PTR79 = NULL AND ^PTR80 = NULL AND ^PTR77 > 0
    
            Data Elements:
    
            ^PTR78 = PTR:RP:Income:Rent:Expense.Interest.Amount
    
            ^PTR77 = PTR:RP:Income:Rent:Income.Operating.RentalIncomeGross.Amount
    
            ^PTR79 = PTR:RP:Income:Rent:Expense.CapitalWorksDeduction.Amount
    
            ^PTR80 = PTR:RP:Income:Rent:IncomeTax.Deduction.RentalIncomeDeductionsOtherThanInterestAndCapitalWorks.Amount
            */
            assertion = (report.PTR78 == null && report.PTR79 == null && report.PTR80 == null && report.PTR77.GetValueOrDefault() > 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430080",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Rent deductions must be present",
                    LongDescription = @"If Gross rent is greater than zero, then there must be an amount at Interest deductions, Capital works deductions or Other rental deductions. If there are no rental deductions, then zero must be present in at least one of these fields",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:Rent/tns:ExpenseInterestA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430080" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR78", Value = GetValueOrEmpty(report.PTR78) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR79", Value = GetValueOrEmpty(report.PTR79) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR80", Value = GetValueOrEmpty(report.PTR80) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR77", Value = GetValueOrEmpty(report.PTR77) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430092
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430092()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430092
            Gross interest amount greater than zero must be present
    
            Legacy Rule Format:
            (^PTR82 = 0 OR ^PTR82 = NULL) AND ^PTR83 > 0

            Technical Business Rule Format:
            (^PTR82 = 0 OR ^PTR82 = NULL) AND ^PTR83 > 0
    
            Data Elements:
    
            ^PTR82 = PTR:RP:Income:GrossInterest:Income.Interest.Gross.Amount
    
            ^PTR83 = PTR:RP:Income:GrossInterest:IncomeTax.TFNAmountsWithheldFromGrossInterest.Amount
            */
            assertion = ((report.PTR82 == 0 || report.PTR82 == null) && report.PTR83.GetValueOrDefault() > 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430092",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Gross interest amount greater than zero must be present",
                    LongDescription = @"If an amount greater than zero is present at TFN amounts withheld from gross interest, then an amount greater than zero must also be present at Gross interest",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:GrossInterest/tns:A",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430092" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR82", Value = GetValueOrEmpty(report.PTR82) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR83", Value = GetValueOrEmpty(report.PTR83) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430093
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430093()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430093
            TFN amounts withheld from dividends must not exceed 50% of Gross interest
    
            Legacy Rule Format:
            ^PTR83 > 0 AND (^PTR82 * 0.5) <= ^PTR83

            Technical Business Rule Format:
            ^PTR83 > 0 AND (^PTR82 * 0.5) <= ^PTR83
    
            Data Elements:
    
            ^PTR83 = PTR:RP:Income:GrossInterest:IncomeTax.TFNAmountsWithheldFromGrossInterest.Amount
    
            ^PTR82 = PTR:RP:Income:GrossInterest:Income.Interest.Gross.Amount
            */
            assertion = (report.PTR83.GetValueOrDefault() > 0 && report.PTR82.GetValueOrDefault() * 0.5M <= report.PTR83.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430093",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"TFN amounts withheld from dividends must not exceed 50% of Gross interest",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:GrossInterest/tns:TaxTFNAmountsWithheldFromGrossInterestA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430093" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR83", Value = GetValueOrEmpty(report.PTR83) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR82", Value = GetValueOrEmpty(report.PTR82) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430095
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430095()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430095
            Dividends amounts must be present
    
            Legacy Rule Format:
            (^PTR84 = 0 OR ^PTR84 = NULL) AND (^PTR85 = 0 OR ^PTR85 = NULL) AND ^PTR87 > 0

            Technical Business Rule Format:
            (^PTR84 = 0 OR ^PTR84 = NULL) AND (^PTR85 = 0 OR ^PTR85 = NULL) AND ^PTR87 > 0
    
            Data Elements:
    
            ^PTR84 = PTR:RP:Income:Dividends:Income.DividendsUnfranked.Amount
    
            ^PTR85 = PTR:RP:Income:Dividends:Income.DividendsFranked.Amount
    
            ^PTR87 = PTR:RP:Income:Dividends:IncomeTax.DividendsTFNAmountsWithheld.Amount
            */
            assertion = ((report.PTR84 == 0 || report.PTR84 == null) && (report.PTR85 == 0 || report.PTR85 == null) && report.PTR87.GetValueOrDefault() > 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430095",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Dividends amounts must be present",
                    LongDescription = @"If an amount greater than zero is present at TFN amounts withheld from dividends, then there must also be an amount greater than zero at Unfranked amount or Franked amount",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:Dividends/tns:UnfrankedA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430095" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR84", Value = GetValueOrEmpty(report.PTR84) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR85", Value = GetValueOrEmpty(report.PTR85) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR87", Value = GetValueOrEmpty(report.PTR87) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430096
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430096()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430096
            TFN amount withheld must not exceed unfranked dividends plus 50% of franked dividends
    
            Legacy Rule Format:
            ^PTR84 > 0 AND ^PTR87 > 0 AND ^PTR85 > 0 AND (^PTR84 + (^PTR85 * 0.5) + 1) < ^PTR87

            Technical Business Rule Format:
            ^PTR84 > 0 AND ^PTR87 > 0 AND ^PTR85 > 0 AND (^PTR84 + (^PTR85 * 0.5) + 1) < ^PTR87
    
            Data Elements:
    
            ^PTR84 = PTR:RP:Income:Dividends:Income.DividendsUnfranked.Amount
    
            ^PTR85 = PTR:RP:Income:Dividends:Income.DividendsFranked.Amount
    
            ^PTR87 = PTR:RP:Income:Dividends:IncomeTax.DividendsTFNAmountsWithheld.Amount
            */
            assertion = (report.PTR84.GetValueOrDefault() > 0 && report.PTR87.GetValueOrDefault() > 0 && report.PTR85.GetValueOrDefault() > 0 && report.PTR84.GetValueOrDefault() + report.PTR85.GetValueOrDefault() * 0.5M + 1 < report.PTR87.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430096",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"TFN amount withheld must not exceed unfranked dividends plus 50% of franked dividends",
                    LongDescription = @"The amount at TFN amounts withheld from dividends must not be greater than Unfranked amount plus 50% of Franked amount",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:Dividends/tns:UnfrankedA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430096" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR84", Value = GetValueOrEmpty(report.PTR84) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR87", Value = GetValueOrEmpty(report.PTR87) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR85", Value = GetValueOrEmpty(report.PTR85) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430098
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430098()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430098
            Franking credit cannot be greater than 48% of dividends-franked amount.
    
            Legacy Rule Format:
            ^PTR86 > 0 AND (^PTR85 * 0.48) < (^PTR86 - 1)

            Technical Business Rule Format:
            ^PTR86 > 0 AND (^PTR85 * 0.48) < (^PTR86 - 1)
    
            Data Elements:
    
            ^PTR86 = PTR:RP:Income:Dividends:IncomeTax.FrankingCredits.Amount
    
            ^PTR85 = PTR:RP:Income:Dividends:Income.DividendsFranked.Amount
            */
            assertion = (report.PTR86.GetValueOrDefault() > 0 && report.PTR85.GetValueOrDefault() * 0.48M < report.PTR86.GetValueOrDefault() - 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430098",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Franking credit cannot be greater than 48% of dividends-franked amount.",
                    LongDescription = @"Franking credit must not exceed 48% of Dividends - Franked amount",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:Dividends/tns:TaxFrankingCreditsA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430098" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR86", Value = GetValueOrEmpty(report.PTR86) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR85", Value = GetValueOrEmpty(report.PTR85) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430100
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430100()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430100
            Franking credit amount must be present
    
            Legacy Rule Format:
            ^PTR86 = NULL AND ^PTR85 > 0

            Technical Business Rule Format:
            ^PTR86 = NULL AND ^PTR85 > 0
    
            Data Elements:
    
            ^PTR86 = PTR:RP:Income:Dividends:IncomeTax.FrankingCredits.Amount
    
            ^PTR85 = PTR:RP:Income:Dividends:Income.DividendsFranked.Amount
            */
            assertion = (report.PTR86 == null && report.PTR85.GetValueOrDefault() > 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430100",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Franking credit amount must be present",
                    LongDescription = @"If Dividends - Franked amount is present, then Franking credit must be present",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:Dividends/tns:TaxFrankingCreditsA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430100" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR86", Value = GetValueOrEmpty(report.PTR86) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR85", Value = GetValueOrEmpty(report.PTR85) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430102
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430102()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430102
            TFN amounts withheld from dividends must not exceed 50% of the Unfranked amount
    
            Legacy Rule Format:
            ^PTR87 > 0 AND ((^PTR84 * 0.5) + 1) < ^PTR87

            Technical Business Rule Format:
            ^PTR87 > 0 AND ((^PTR84 * 0.5) + 1) < ^PTR87
    
            Data Elements:
    
            ^PTR87 = PTR:RP:Income:Dividends:IncomeTax.DividendsTFNAmountsWithheld.Amount
    
            ^PTR84 = PTR:RP:Income:Dividends:Income.DividendsUnfranked.Amount
            */
            assertion = (report.PTR87.GetValueOrDefault() > 0 && report.PTR84.GetValueOrDefault() * 0.5M + 1 < report.PTR87.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430102",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"TFN amounts withheld from dividends must not exceed 50% of the Unfranked amount",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:Dividends/tns:TaxTFNAmountsWithheldA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430102" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR87", Value = GetValueOrEmpty(report.PTR87) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR84", Value = GetValueOrEmpty(report.PTR84) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430106
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430106()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430106
            Gross interest or dividend income must be present
    
            Legacy Rule Format:
            ^PTR82 = NULL AND ^PTR84 = NULL AND ^PTR85 = NULL AND ^PTR90 > 0

            Technical Business Rule Format:
            ^PTR82 = NULL AND ^PTR84 = NULL AND ^PTR85 = NULL AND ^PTR90 > 0
    
            Data Elements:
    
            ^PTR82 = PTR:RP:Income:GrossInterest:Income.Interest.Gross.Amount
    
            ^PTR84 = PTR:RP:Income:Dividends:Income.DividendsUnfranked.Amount
    
            ^PTR85 = PTR:RP:Income:Dividends:Income.DividendsFranked.Amount
    
            ^PTR90 = PTR:RP:Expense.AustralianInvestmentAllowableDeduction.Amount
            */
            assertion = (report.PTR82 == null && report.PTR84 == null && report.PTR85 == null && report.PTR90.GetValueOrDefault() > 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430106",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Gross interest or dividend income must be present",
                    LongDescription = @"If Deductions relating to Australian investment income is greater than zero, then Gross interest, Unfranked amount, or Franked amount must be present",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:GrossInterest/tns:A",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430106" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR82", Value = GetValueOrEmpty(report.PTR82) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR84", Value = GetValueOrEmpty(report.PTR84) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR85", Value = GetValueOrEmpty(report.PTR85) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR90", Value = GetValueOrEmpty(report.PTR90) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430112
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430112()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430112
            Net Australian income must equal income less deductions
    
            Legacy Rule Format:
            ^PTR97 <> (^PTR61 + ^PTR62 + ^PTR65 + ^PTR66 - ^PTR68 + ^PTR69 + ^PTR70 - ^PTR72 + ^PTR322 - ^PTR323 + ^PTR77 - ^PTR78 - ^PTR79 - ^PTR80 + ^PTR81 + ^PTR82 + ^PTR84 + ^PTR85 + ^PTR86 + Sum(^PTR89) - (^PTR90 + ^PTR91 + Sum(^PTR96)))

            Technical Business Rule Format:
            ^PTR97 <> (^PTR61 + ^PTR62 + ^PTR65 + ^PTR66 - ^PTR68 + ^PTR69 + ^PTR70 - ^PTR72 + ^PTR322 - ^PTR323 + ^PTR77 - ^PTR78 - ^PTR79 - ^PTR80 + ^PTR81 + ^PTR82 + ^PTR84 + ^PTR85 + ^PTR86 + Sum(^PTR89) - (^PTR90 + ^PTR91 + Sum(^PTR96)))
    
            Data Elements:
    
            ^PTR97 = PTR:RP:Income.Net.Amount
    
            ^PTR61 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Income.Net.Amount
    
            ^PTR62 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Income.Net.Amount
    
            ^PTR65 = PTR:RP:Income:PartnershipsTrustsPrimaryProduction:Income.PartnershipDistributionGross.Amount
    
            ^PTR66 = PTR:RP:Income:PartnershipsTrustsPrimaryProduction:Income.TrustShareNet.Amount
    
            ^PTR68 = PTR:RP:Income:PartnershipsTrustsPrimaryProduction:Expense.DistributionDeductionsOther.Amount
    
            ^PTR69 = PTR:RP:Income:PartnershipsTrustsNonPrimaryProduction:Income.PartnershipDistributionLessForeignIncome.Amount
    
            ^PTR70 = PTR:RP:Income:PartnershipsTrustsNonPrimaryProduction:Income.TrustShareNetExcludeNetCapitalGainsAndForeignIncomeAndDistributionFranked.Amount
    
            ^PTR72 = PTR:RP:Income:PartnershipsTrustsNonPrimaryProduction:Expense.DistributionDeductionsOther.Amount
    
            ^PTR77 = PTR:RP:Income:Rent:Income.Operating.RentalIncomeGross.Amount
    
            ^PTR78 = PTR:RP:Income:Rent:Expense.Interest.Amount
    
            ^PTR79 = PTR:RP:Income:Rent:Expense.CapitalWorksDeduction.Amount
    
            ^PTR80 = PTR:RP:Income:Rent:IncomeTax.Deduction.RentalIncomeDeductionsOtherThanInterestAndCapitalWorks.Amount
    
            ^PTR81 = PTR:RP:Income:Income.ForestryManagedInvestmentScheme.Amount
    
            ^PTR82 = PTR:RP:Income:GrossInterest:Income.Interest.Gross.Amount
    
            ^PTR84 = PTR:RP:Income:Dividends:Income.DividendsUnfranked.Amount
    
            ^PTR85 = PTR:RP:Income:Dividends:Income.DividendsFranked.Amount
    
            ^PTR86 = PTR:RP:Income:Dividends:IncomeTax.FrankingCredits.Amount
    
            ^PTR89 = PTR:RP:Income:OtherAustralianIncome:Income.Other.Amount
    
            ^PTR90 = PTR:RP:Expense.AustralianInvestmentAllowableDeduction.Amount
    
            ^PTR91 = PTR:RP:Expense.ForestryManagedInvestmentSchemeDeduction.Amount
    
            ^PTR96 = PTR:RP:OtherDeductions:Expense.DeductibleOther.Amount
    
            ^PTR322 = PTR:RP:Income:PartnershipsTrustsNonPrimaryProduction:Income.TrustDistributionFranked.Amount
    
            ^PTR323 = PTR:RP:Income:PartnershipsTrustsNonPrimaryProduction:IncomeTax.Deduction.TrustDistributionFranked.Amount
            */
            assertion = (report.PTR97.GetValueOrDefault() != report.PTR61.GetValueOrDefault() + report.PTR62.GetValueOrDefault() + report.PTR65.GetValueOrDefault() + report.PTR66.GetValueOrDefault() - report.PTR68.GetValueOrDefault() + report.PTR69.GetValueOrDefault() + report.PTR70.GetValueOrDefault() - report.PTR72.GetValueOrDefault() + report.PTR322.GetValueOrDefault() - report.PTR323.GetValueOrDefault() + report.PTR77.GetValueOrDefault() - report.PTR78.GetValueOrDefault() - report.PTR79.GetValueOrDefault() - report.PTR80.GetValueOrDefault() + report.PTR81.GetValueOrDefault() + report.PTR82.GetValueOrDefault() + report.PTR84.GetValueOrDefault() + report.PTR85.GetValueOrDefault() + report.PTR86.GetValueOrDefault() + (report.RP_Income_OtherAustralianIncomeCollection == null ? 0 : report.RP_Income_OtherAustralianIncomeCollection.Sum(f => f.PTR89.GetValueOrDefault())) - (report.PTR90.GetValueOrDefault() + report.PTR91.GetValueOrDefault() + (report.RP_OtherDeductionsCollection == null ? 0 : report.RP_OtherDeductionsCollection.Sum(f => f.PTR96.GetValueOrDefault()))));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.430112",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Net Australian income must equal income less deductions",
                    LongDescription = @"The Net Australian income or loss amount must equal the sum of all income less the sum of all deductions",
                    Location = "/tns:PTR/tns:RP/tns:IncomeNetA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430112" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR97", Value = GetValueOrEmpty(report.PTR97) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR61", Value = GetValueOrEmpty(report.PTR61) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR62", Value = GetValueOrEmpty(report.PTR62) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR65", Value = GetValueOrEmpty(report.PTR65) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR66", Value = GetValueOrEmpty(report.PTR66) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR68", Value = GetValueOrEmpty(report.PTR68) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR69", Value = GetValueOrEmpty(report.PTR69) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR70", Value = GetValueOrEmpty(report.PTR70) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR72", Value = GetValueOrEmpty(report.PTR72) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR322", Value = GetValueOrEmpty(report.PTR322) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR323", Value = GetValueOrEmpty(report.PTR323) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR77", Value = GetValueOrEmpty(report.PTR77) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR78", Value = GetValueOrEmpty(report.PTR78) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR79", Value = GetValueOrEmpty(report.PTR79) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR80", Value = GetValueOrEmpty(report.PTR80) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR81", Value = GetValueOrEmpty(report.PTR81) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR82", Value = GetValueOrEmpty(report.PTR82) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR84", Value = GetValueOrEmpty(report.PTR84) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR85", Value = GetValueOrEmpty(report.PTR85) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR86", Value = GetValueOrEmpty(report.PTR86) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Sum(PTR89)", Value = report.RP_Income_OtherAustralianIncomeCollection == null ? "0" : report.RP_Income_OtherAustralianIncomeCollection.Sum(f => f.PTR89.GetValueOrDefault()).ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR90", Value = GetValueOrEmpty(report.PTR90) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR91", Value = GetValueOrEmpty(report.PTR91) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Sum(PTR96)", Value = report.RP_OtherDeductionsCollection == null ? "0" : report.RP_OtherDeductionsCollection.Sum(f => f.PTR96.GetValueOrDefault()).ToString() });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430116
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430116()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430116
            Answer to Attributed foreign income question must be 'yes'
    
            Legacy Rule Format:
            (^PTR98 > 0 OR ^PTR100 > 0) AND ^PTR102 <> TRUE

            Technical Business Rule Format:
            (^PTR98 > 0 OR ^PTR100 > 0) AND ^PTR102 <> TRUE
    
            Data Elements:
    
            ^PTR102 = PTR:RP:AttributedForeignIncome:Income.InternationalDealings.AttributedForeignIncomeForeignTrustorControlledForeignCompanyorTransferorTrust.Indicator
    
            ^PTR98 = PTR:RP:AttributedForeignIncome:ListedCountry:Income.InternationalDealings.Attributed.Amount
    
            ^PTR100 = PTR:RP:AttributedForeignIncome:UnlistedCountry:Income.InternationalDealings.Attributed.Amount
            */
            assertion = ((report.PTR98.GetValueOrDefault() > 0 || report.PTR100.GetValueOrDefault() > 0) && report.PTR102 != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430116",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Answer to Attributed foreign income question must be 'yes'",
                    LongDescription = @"If Attributed foreign income - Listed country, Section 404 country, or Unlisted country is greater than zero, then a yes (true) response to the question ""Did you have either a direct or indirect interest in a foreign trust, controlled foreign company or transferor trust?"" is required",
                    Location = "/tns:PTR/tns:RP/tns:AttributedForeignIncome/tns:InternationalDealingsTrustorControlledCompanyorTransferorTrustI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430116" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR98", Value = GetValueOrEmpty(report.PTR98) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR100", Value = GetValueOrEmpty(report.PTR100) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR102", Value = GetValueOrEmpty(report.PTR102) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430120
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430120()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430120
            Other assessable foreign source income 'Gross' amount must not be less than 'Net' amount
    
            Legacy Rule Format:
            ^PTR104 < ^PTR105

            Technical Business Rule Format:
            ^PTR104 < ^PTR105
    
            Data Elements:
    
            ^PTR104 = PTR:RP:OtherAssessableForeignSourceIncome:Income.Other.Amount
    
            ^PTR105 = PTR:RP:OtherAssessableForeignSourceIncome:Income.Net.Amount
            */
            assertion = (report.PTR104.GetValueOrDefault() < report.PTR105.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430120",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other assessable foreign source income 'Gross' amount must not be less than 'Net' amount",
                    Location = "/tns:PTR/tns:RP/tns:OtherAssessableForeignSourceIncome/tns:A",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430120" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR104", Value = GetValueOrEmpty(report.PTR104) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR105", Value = GetValueOrEmpty(report.PTR105) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430129
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430129()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430129
            Business name and Business address of main business must be present
    
            Legacy Rule Format:
            (^PTR35 <> NULL OR ^PTR36 <> NULL OR ^PTR38 <> NULL OR ^PTR39 <> NULL OR ^PTR41 <> NULL OR ^PTR43 <> NULL OR ^PTR44 <> NULL) AND (^PTR116 = NULL OR ^PTR117 = NULL)

            Technical Business Rule Format:
            (^PTR35 <> NULL OR ^PTR36 <> NULL OR ^PTR38 <> NULL OR ^PTR39 <> NULL OR ^PTR41 <> NULL OR ^PTR43 <> NULL OR ^PTR44 <> NULL) AND (^PTR116 = NULL OR ^PTR117 = NULL)
    
            Data Elements:
    
            ^PTR116 = PTR:RP:BusinessName:OrganisationNameDetails.OrganisationalName.Text
    
            ^PTR35 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^PTR36 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^PTR38 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Remuneration.PaymentToForeignResidentGross.Amount
    
            ^PTR39 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^PTR41 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^PTR43 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Income.Other.Amount
    
            ^PTR44 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Income.Other.Amount
    
            ^PTR117 = PTR:RP:BusinessAddressDetails:AddressDetails.Line1.Text
            */
            assertion = ((report.PTR35 != null || report.PTR36 != null || report.PTR38 != null || report.PTR39 != null || report.PTR41 != null || report.PTR43 != null || report.PTR44 != null) && (report.PTR116 == null || report.PTR117 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430129",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Business name and business address of main business must be present",
                    LongDescription = @"If Total business income (Primary production or Non-primary production) is present at business income and expenses, then both the Business name and Business address of main business must be present",
                    Location = "/tns:PTR/tns:RP/tns:BusinessNameOrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430129" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR35", Value = GetValueOrEmpty(report.PTR35) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR36", Value = GetValueOrEmpty(report.PTR36) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR38", Value = GetValueOrEmpty(report.PTR38) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR39", Value = GetValueOrEmpty(report.PTR39) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR41", Value = GetValueOrEmpty(report.PTR41) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR43", Value = GetValueOrEmpty(report.PTR43) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR44", Value = GetValueOrEmpty(report.PTR44) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR116", Value = report.PTR116 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR117", Value = report.PTR117 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430130
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430130()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430130
            Business and professional amounts must be present
    
            Legacy Rule Format:
            (^PTR61 <> NULL OR ^PTR62 <> NULL) AND ^PTR111 = NULL AND ^PTR112 = NULL AND ^PTR113 = NULL AND ^PTR114 = NULL AND ^PTR121 = NULL AND ^PTR122 = NULL AND ^PTR123 = NULL AND ^PTR125 = NULL AND ^PTR126 = NULL AND ^PTR127 = NULL AND ^PTR129 = NULL AND ^PTR136 = NULL

            Technical Business Rule Format:
            (^PTR61 <> NULL OR ^PTR62 <> NULL) AND ^PTR111 = NULL AND ^PTR112 = NULL AND ^PTR113 = NULL AND ^PTR114 = NULL AND ^PTR121 = NULL AND ^PTR122 = NULL AND ^PTR123 = NULL AND ^PTR125 = NULL AND ^PTR126 = NULL AND ^PTR127 = NULL AND ^PTR129 = NULL AND ^PTR136 = NULL
    
            Data Elements:
    
            ^PTR121 = PTR:RP:BusinessAndProfessionaltems:Assets.Inventories.Total.Amount
    
            ^PTR61 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Income.Net.Amount
    
            ^PTR62 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Income.Net.Amount
    
            ^PTR111 = PTR:RP:KeyFinancialInformation:Assets.CurrentTotal.Amount
    
            ^PTR112 = PTR:RP:KeyFinancialInformation:Assets.Total.Amount
    
            ^PTR113 = PTR:RP:KeyFinancialInformation:Liabilities.CurrentTotal.Amount
    
            ^PTR114 = PTR:RP:KeyFinancialInformation:Liabilities.Total.Amount
    
            ^PTR122 = PTR:RP:BusinessAndProfessionaltems:Expense.Purchases.AndOtherCosts.Amount
    
            ^PTR123 = PTR:RP:BusinessAndProfessionaltems:Closing:Assets.Inventories.Total.Amount
    
            ^PTR125 = PTR:RP:BusinessAndProfessionaltems:Closing:Assets.TradeandReceivablesOther.Amount
    
            ^PTR126 = PTR:RP:BusinessAndProfessionaltems:Closing:Liabilities.TradeAndOtherPayablesTotal.Amount
    
            ^PTR127 = PTR:RP:BusinessAndProfessionaltems:Remuneration.WagesAndSalaries.Total.Amount
    
            ^PTR129 = PTR:RP:BusinessAndProfessionaltems:Remuneration.PaymentToRelatedPartiesGross.Amount
    
            ^PTR136 = PTR:RP:BusinessAndProfessionaltems:Remuneration.FringeBenefits.EmployeeContribution.Amount
            */
            assertion = ((report.PTR61 != null || report.PTR62 != null) && report.PTR111 == null && report.PTR112 == null && report.PTR113 == null && report.PTR114 == null && report.PTR121 == null && report.PTR122 == null && report.PTR123 == null && report.PTR125 == null && report.PTR126 == null && report.PTR127 == null && report.PTR129 == null && report.PTR136 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.439000",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Key financial information amounts and/or Business and professional amounts must be present",
                    LongDescription = @"If Net income or loss from business (either Primary production or Non-primary production) is present, then there must be an amount included under Assets, Liabilities or under Business and professional items",
                    Location = "/tns:PTR/tns:RP/tns:BusinessAndProfessionaltems/tns:AssetsInventoriesTotalA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430130" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR61", Value = GetValueOrEmpty(report.PTR61) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR62", Value = GetValueOrEmpty(report.PTR62) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR111", Value = GetValueOrEmpty(report.PTR111) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR112", Value = GetValueOrEmpty(report.PTR112) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR113", Value = GetValueOrEmpty(report.PTR113) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR114", Value = GetValueOrEmpty(report.PTR114) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR121", Value = GetValueOrEmpty(report.PTR121) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR122", Value = GetValueOrEmpty(report.PTR122) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR123", Value = GetValueOrEmpty(report.PTR123) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR125", Value = GetValueOrEmpty(report.PTR125) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR126", Value = GetValueOrEmpty(report.PTR126) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR127", Value = GetValueOrEmpty(report.PTR127) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR129", Value = GetValueOrEmpty(report.PTR129) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR136", Value = GetValueOrEmpty(report.PTR136) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430131
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430131()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430131
            Total salary and wage expenses amount must be present if the action code is present
    
            Legacy Rule Format:
            ^PTR128 <> NULL AND (^PTR127 = 0 OR ^PTR127 = NULL)

            Technical Business Rule Format:
            ^PTR128 <> NULL AND (^PTR127 = 0 OR ^PTR127 = NULL)
    
            Data Elements:
    
            ^PTR127 = PTR:RP:BusinessAndProfessionaltems:Remuneration.WagesAndSalaries.Total.Amount
    
            ^PTR128 = PTR:RP:BusinessAndProfessionaltems:Remuneration.WagesAndSalariesAction.Code
            */
            assertion = (report.PTR128 != null && (report.PTR127 == 0 || report.PTR127 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430131",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total salary and wage expenses amount must be present if the action code is present",
                    LongDescription = @"If a Total salary and wage expenses action code is present an amount must be present. Either enter an amount or delete the action code",
                    Location = "/tns:PTR/tns:RP/tns:BusinessAndProfessionaltems/tns:RemunerationWagesAndSalariesTotalA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430131" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR128", Value = GetValueOrEmpty(report.PTR128) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR127", Value = GetValueOrEmpty(report.PTR127) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430132
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430132()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430132
            Total salary and wage expenses action code must be present
    
            Legacy Rule Format:
            ^PTR127 > 0 AND ^PTR128 = NULL

            Technical Business Rule Format:
            ^PTR127 > 0 AND ^PTR128 = NULL
    
            Data Elements:
    
            ^PTR128 = PTR:RP:BusinessAndProfessionaltems:Remuneration.WagesAndSalariesAction.Code
    
            ^PTR127 = PTR:RP:BusinessAndProfessionaltems:Remuneration.WagesAndSalaries.Total.Amount
            */
            assertion = (report.PTR127.GetValueOrDefault() > 0 && report.PTR128 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430132",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total salary and wage expenses action code must be present",
                    LongDescription = @"If a Total salary and wage expenses amount is present, then an action code must be present",
                    Location = "/tns:PTR/tns:RP/tns:BusinessAndProfessionaltems/tns:RemunerationWagesAndSalariesActionC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430132" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR127", Value = GetValueOrEmpty(report.PTR127) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR128", Value = GetValueOrEmpty(report.PTR128) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430136
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430136(PTR2025.RP_StatementOfDistribution statementOfDistribution, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430136
            Statement of distribution - name must be present

            Legacy Rule Format:
            ^PTR146 = NULL AND ^PTR149 = NULL

            Technical Business Rule Format:
            ^PTR146 = NULL AND ^PTR149 = NULL

            Data Elements:

            ^PTR146 = PTR:RP:StatementOfDistribution:IndividualPartnerDetails:PersonNameDetails.FamilyName.Text

            ^PTR149 = PTR:RP:StatementOfDistribution:NonIndividualPartnerDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (statementOfDistribution.PTR146 == null && statementOfDistribution.PTR149 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430136",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Statement of distribution - name must be present",
                    LongDescription = @"If the partner/beneficiary tax file number or current postal address is present in the Statement of distribution, then an individual  name or non-individual name must be present",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:IndividualPartnerDetails/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430136" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR146", Value = statementOfDistribution.PTR146 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR149", Value = statementOfDistribution.PTR149 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430139
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430139(PTR2025.RP_StatementOfDistribution statementOfDistribution, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430139
            Statement of distribution must not contain both an individual name and a non-individual name

            Legacy Rule Format:
            ^PTR149 <> NULL AND ^PTR146 <> NULL

            Technical Business Rule Format:
            ^PTR149 <> NULL AND ^PTR146 <> NULL

            Data Elements:

            ^PTR149 = PTR:RP:StatementOfDistribution:NonIndividualPartnerDetails:OrganisationNameDetails.OrganisationalName.Text

            ^PTR146 = PTR:RP:StatementOfDistribution:IndividualPartnerDetails:PersonNameDetails.FamilyName.Text
            */
            assertion = (statementOfDistribution.PTR149 != null && statementOfDistribution.PTR146 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430139",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Both family name and company name present",
                    LongDescription = @"Family name and non-individual name line 1 are both present in the distribution statement (Item 65). Please delete the incorrect name as only one individual or non-individual name may be present.",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:NonIndividualPartnerDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430139" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR149", Value = statementOfDistribution.PTR149 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR146", Value = statementOfDistribution.PTR146 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430141
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430141(PTR2025.RP_StatementOfDistribution statementOfDistribution, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430141
            Statement of distribution - Individual's residential address or Non-Individual's business address or Tax File Number or Australian Business Number must be present

            Legacy Rule Format:
            ^PTR157 = NULL AND ^PTR151 = NULL AND ^PTR285 = NULL

            Technical Business Rule Format:
            ^PTR157 = NULL AND ^PTR151 = NULL AND ^PTR285 = NULL

            Data Elements:

            ^PTR157 = PTR:RP:StatementOfDistribution:Identifiers.TaxFileNumber.Identifier

            ^PTR151 = PTR:RP:StatementOfDistribution:PartnerAddressDetails:AddressDetails.Line1.Text

            ^PTR285 = PTR:RP:StatementOfDistribution:NonIndividualPartnerDetails:Identifiers.AustralianBusinessNumber.Identifier
            */
            assertion = (statementOfDistribution.PTR157 == null && statementOfDistribution.PTR151 == null && statementOfDistribution.PTR285 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.431053",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Individual's  residential address  or Non-Individual's business address or Tax File Number or Australian Business Number  must be present in each Statement of Distribution",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:TaxFileNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430141" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR157", Value = GetValueOrEmpty(statementOfDistribution.PTR157) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR151", Value = statementOfDistribution.PTR151 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR285", Value = GetValueOrEmpty(statementOfDistribution.PTR285) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430142
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430142()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430142
            Tax file number must not be the TFN of the Trust or Partnership
    
            Legacy Rule Format:
            AnyOccurrence(^PTR157, ^PTR157 = ^PTR5)

            Technical Business Rule Format:
            AnyOccurrence(^PTR157, ^PTR157 = ^PTR5)
    
            Data Elements:
    
            ^PTR157 = PTR:RP:StatementOfDistribution:Identifiers.TaxFileNumber.Identifier
    
            ^PTR5 = PTR:RP:Identifiers.TaxFileNumber.Identifier
            */
            assertion = (report.RP_StatementOfDistributionCollection == null ? false : report.RP_StatementOfDistributionCollection.Any(PTR157Repeat => PTR157Repeat.PTR157 == report.PTR5));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430142",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax file number must not be the TFN of the Trust or Partnership",
                    LongDescription = @"The TFN provided for a Trustee or Partner in a Statement of distribution must not be the same as the TFN of the Trust or Partnership",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution/tns:TaxFileNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430142" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR157", Value = report.RP_StatementOfDistributionCollection.First(PTR157Repeat => PTR157Repeat.PTR157 == report.PTR5).PTR157 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR5", Value = report.PTR5 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430143
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430143(PTR2025.RP_StatementOfDistribution statementOfDistribution, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430143
            PP, Non-PP or Franked distribution required.

            Legacy Rule Format:
            ^PTR160 > 0 AND ^PTR158 = NULL AND ^PTR159 = NULL AND ^PTR324 = NULL

            Technical Business Rule Format:
            ^PTR160 > 0 AND ^PTR158 = NULL AND ^PTR159 = NULL AND ^PTR324 = NULL

            Data Elements:

            ^PTR158 = PTR:RP:StatementOfDistribution:IncomeOrLoss:ShareOfIncomePrimaryProduction:Income.BeneficiaryShare.Amount

            ^PTR159 = PTR:RP:StatementOfDistribution:IncomeOrLoss:ShareOfIncomeNonPrimaryProduction:Income.BeneficiaryShare.Amount

            ^PTR160 = PTR:RP:StatementOfDistribution:IncomeOrLoss:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount

            ^PTR324 = PTR:RP:StatementOfDistribution:IncomeOrLoss:Income.DistributionFranked.Amount
            */
            assertion = (statementOfDistribution.PTR160.GetValueOrDefault() > 0 && statementOfDistribution.PTR158 == null && statementOfDistribution.PTR159 == null && statementOfDistribution.PTR324 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.430707",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"PP, Non-PP or Franked distributions required",
                    LongDescription = @"Credit for tax withheld where ABN not quoted is greater than zero and there is no Primary production or Non-primary production income distribution, or Franked Distributions",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:IncomeOrLoss/tns:ShareOfIncomePrimaryProduction/tns:BeneficiaryShareA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430143" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR160", Value = GetValueOrEmpty(statementOfDistribution.PTR160) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR158", Value = GetValueOrEmpty(statementOfDistribution.PTR158) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR159", Value = GetValueOrEmpty(statementOfDistribution.PTR159) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR324", Value = GetValueOrEmpty(statementOfDistribution.PTR324) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430144
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430144()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430144
            Distribution of PP income cannot exceed net PP income
    
            Legacy Rule Format:
            Sum(^PTR158) > (^PTR61 + ^PTR65 + ^PTR66 - ^PTR68)

            Technical Business Rule Format:
            Sum(^PTR158) > (^PTR61 + ^PTR65 + ^PTR66 - ^PTR68)
    
            Data Elements:
    
            ^PTR158 = PTR:RP:StatementOfDistribution:IncomeOrLoss:ShareOfIncomePrimaryProduction:Income.BeneficiaryShare.Amount
    
            ^PTR61 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Income.Net.Amount
    
            ^PTR65 = PTR:RP:Income:PartnershipsTrustsPrimaryProduction:Income.PartnershipDistributionGross.Amount
    
            ^PTR66 = PTR:RP:Income:PartnershipsTrustsPrimaryProduction:Income.TrustShareNet.Amount
    
            ^PTR68 = PTR:RP:Income:PartnershipsTrustsPrimaryProduction:Expense.DistributionDeductionsOther.Amount
            */
            assertion = ((report.RP_StatementOfDistributionCollection == null ? 0 : report.RP_StatementOfDistributionCollection.Sum(f => f.PTR158.GetValueOrDefault())) > report.PTR61.GetValueOrDefault() + report.PTR65.GetValueOrDefault() + report.PTR66.GetValueOrDefault() - report.PTR68.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.440144",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Distribution of PP income cannot exceed net PP income",
                    LongDescription = @"The amount of primary production income in the distribution statement is greater than the net primary production income received - net primary production income plus primary production distribution from partnerships plus primary production distribution from trusts less primary production deductions relating to distribution.",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution/tns:IncomeOrLoss/tns:ShareOfIncomePrimaryProduction/tns:BeneficiaryShareA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430144" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Sum(PTR158)", Value = report.RP_StatementOfDistributionCollection == null ? "0" : report.RP_StatementOfDistributionCollection.Sum(f => f.PTR158.GetValueOrDefault()).ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR61", Value = GetValueOrEmpty(report.PTR61) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR65", Value = GetValueOrEmpty(report.PTR65) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR66", Value = GetValueOrEmpty(report.PTR66) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR68", Value = GetValueOrEmpty(report.PTR68) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430145
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430145()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430145
            Net Australian income/loss must equal total Distributed PP and non-PP income
    
            Legacy Rule Format:
            ^PTR97 <> (Sum(^PTR158) + Sum(^PTR159) + Sum(^PTR324))

            Technical Business Rule Format:
            ^PTR97 <> (Sum(^PTR158) + Sum(^PTR159) + Sum(^PTR324))
    
            Data Elements:
    
            ^PTR97 = PTR:RP:Income.Net.Amount
    
            ^PTR158 = PTR:RP:StatementOfDistribution:IncomeOrLoss:ShareOfIncomePrimaryProduction:Income.BeneficiaryShare.Amount
    
            ^PTR159 = PTR:RP:StatementOfDistribution:IncomeOrLoss:ShareOfIncomeNonPrimaryProduction:Income.BeneficiaryShare.Amount
    
            ^PTR324 = PTR:RP:StatementOfDistribution:IncomeOrLoss:Income.DistributionFranked.Amount
            */
            assertion = report.PTR97.GetValueOrDefault() != (report.RP_StatementOfDistributionCollection == null ? 0 : report.RP_StatementOfDistributionCollection.Sum(f => f.PTR158.GetValueOrDefault() + f.PTR159.GetValueOrDefault() + f.PTR324.GetValueOrDefault()));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.430706",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Net Australian income/loss must equal total Distributed PP, non-PP income and Franked distributions",
                    LongDescription = @"The Net Australian income/loss amount must equal the sum of Share of income - Primary production plus Share of income - Non-primary production in all Statements of distribution plus Franked distributions",
                    Location = "/tns:PTR/tns:RP/tns:IncomeNetA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430145" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR97", Value = GetValueOrEmpty(report.PTR97) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Sum(PTR158)", Value = report.RP_StatementOfDistributionCollection == null ? "0" : report.RP_StatementOfDistributionCollection.Sum(f => f.PTR158.GetValueOrDefault()).ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Sum(PTR159)", Value = report.RP_StatementOfDistributionCollection == null ? "0" : report.RP_StatementOfDistributionCollection.Sum(f => f.PTR159.GetValueOrDefault()).ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Sum(PTR324)", Value = report.RP_StatementOfDistributionCollection == null ? "0" : report.RP_StatementOfDistributionCollection.Sum(f => f.PTR324.GetValueOrDefault()).ToString() });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430146
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430146(PTR2025.RP_StatementOfDistribution statementOfDistribution, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430146
            Partner distribution amount(s) missing

            Legacy Rule Format:
            (^PTR146 <> NULL OR ^PTR149 <> NULL OR ^PTR151 <> NULL OR ^PTR157 <> NULL) AND ^PTR158 = NULL AND ^PTR159 = NULL AND ^PTR324 = NULL AND ^PTR161 = NULL AND ^PTR162 = NULL AND ^PTR164 = NULL

            Technical Business Rule Format:
            (^PTR146 <> NULL OR ^PTR149 <> NULL OR ^PTR151 <> NULL OR ^PTR157 <> NULL) AND ^PTR158 = NULL AND ^PTR159 = NULL AND ^PTR324 = NULL AND ^PTR161 = NULL AND ^PTR162 = NULL AND ^PTR164 = NULL

            Data Elements:

            ^PTR158 = PTR:RP:StatementOfDistribution:IncomeOrLoss:ShareOfIncomePrimaryProduction:Income.BeneficiaryShare.Amount

            ^PTR146 = PTR:RP:StatementOfDistribution:IndividualPartnerDetails:PersonNameDetails.FamilyName.Text

            ^PTR149 = PTR:RP:StatementOfDistribution:NonIndividualPartnerDetails:OrganisationNameDetails.OrganisationalName.Text

            ^PTR151 = PTR:RP:StatementOfDistribution:PartnerAddressDetails:AddressDetails.Line1.Text

            ^PTR157 = PTR:RP:StatementOfDistribution:Identifiers.TaxFileNumber.Identifier

            ^PTR159 = PTR:RP:StatementOfDistribution:IncomeOrLoss:ShareOfIncomeNonPrimaryProduction:Income.BeneficiaryShare.Amount

            ^PTR161 = PTR:RP:StatementOfDistribution:IncomeOrLoss:IncomeTax.FrankingCredits.ReceivedFromAustralianCompanies.Amount

            ^PTR162 = PTR:RP:StatementOfDistribution:IncomeOrLoss:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldWhereTFNNotQuoted.Amount

            ^PTR164 = PTR:RP:StatementOfDistribution:IncomeOrLoss:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount

            ^PTR324 = PTR:RP:StatementOfDistribution:IncomeOrLoss:Income.DistributionFranked.Amount
            */
            assertion = ((statementOfDistribution.PTR146 != null || statementOfDistribution.PTR149 != null || statementOfDistribution.PTR151 != null || statementOfDistribution.PTR157 != null) && statementOfDistribution.PTR158 == null && statementOfDistribution.PTR159 == null && statementOfDistribution.PTR324 == null && statementOfDistribution.PTR161 == null && statementOfDistribution.PTR162 == null && statementOfDistribution.PTR164 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.431137",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Partner distribution amount(s) must be provided",
                    LongDescription = @"There must be an amount provided for the Partner distribution amount(s) - Share of income Primary Production, Share of income Non-primary production, Franked Distributions, Franking credit, TFN amounts withheld or Australian franking credits from a New Zealand company, for each partner listed in the Statement of distribution.",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:IncomeOrLoss/tns:ShareOfIncomePrimaryProduction/tns:BeneficiaryShareA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430146" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR146", Value = statementOfDistribution.PTR146 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR149", Value = statementOfDistribution.PTR149 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR151", Value = statementOfDistribution.PTR151 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR157", Value = GetValueOrEmpty(statementOfDistribution.PTR157) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR158", Value = GetValueOrEmpty(statementOfDistribution.PTR158) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR159", Value = GetValueOrEmpty(statementOfDistribution.PTR159) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR324", Value = GetValueOrEmpty(statementOfDistribution.PTR324) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR161", Value = GetValueOrEmpty(statementOfDistribution.PTR161) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR162", Value = GetValueOrEmpty(statementOfDistribution.PTR162) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR164", Value = GetValueOrEmpty(statementOfDistribution.PTR164) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430148
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430148(PTR2025.RP_StatementOfDistribution statementOfDistribution, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430148
            Share of income - Non-PP or Franked distributions must be present

            Legacy Rule Format:
            (^PTR160 <> NULL OR ^PTR161 <> NULL OR ^PTR162 <> NULL OR ^PTR351 <> NULL) AND ^PTR159 = NULL AND ^PTR324 = NULL

            Technical Business Rule Format:
            (^PTR160 <> NULL OR ^PTR161 <> NULL OR ^PTR162 <> NULL OR ^PTR351 <> NULL) AND ^PTR159 = NULL AND ^PTR324 = NULL

            Data Elements:

            ^PTR159 = PTR:RP:StatementOfDistribution:IncomeOrLoss:ShareOfIncomeNonPrimaryProduction:Income.BeneficiaryShare.Amount

            ^PTR160 = PTR:RP:StatementOfDistribution:IncomeOrLoss:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount

            ^PTR161 = PTR:RP:StatementOfDistribution:IncomeOrLoss:IncomeTax.FrankingCredits.ReceivedFromAustralianCompanies.Amount

            ^PTR162 = PTR:RP:StatementOfDistribution:IncomeOrLoss:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldWhereTFNNotQuoted.Amount

            ^PTR324 = PTR:RP:StatementOfDistribution:IncomeOrLoss:Income.DistributionFranked.Amount

            ^PTR351 = PTR:RP:StatementOfDistribution:IncomeOrLoss:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromForeignResidents.Amount
            */
            assertion = ((statementOfDistribution.PTR160 != null || statementOfDistribution.PTR161 != null || statementOfDistribution.PTR162 != null || statementOfDistribution.PTR351 != null) && statementOfDistribution.PTR159 == null && statementOfDistribution.PTR324 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.430710",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Share of income - Non-PP income distribution or Franked distributions must be present",
                    LongDescription = @"In the statement of distribution, if there is an amount at either Credit for tax withheld where ABN not quoted, Franking Credit, TFN amounts withheld, or Credit for tax withheld - foreign resident withholding, then there must be an amount at either Share of income - Non-primary production or Franked distributions",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:IncomeOrLoss/tns:ShareOfIncomeNonPrimaryProduction/tns:BeneficiaryShareA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430148" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR160", Value = GetValueOrEmpty(statementOfDistribution.PTR160) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR161", Value = GetValueOrEmpty(statementOfDistribution.PTR161) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR162", Value = GetValueOrEmpty(statementOfDistribution.PTR162) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR351", Value = GetValueOrEmpty(statementOfDistribution.PTR351) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR159", Value = GetValueOrEmpty(statementOfDistribution.PTR159) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR324", Value = GetValueOrEmpty(statementOfDistribution.PTR324) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430149
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430149(PTR2025.RP_StatementOfDistribution statementOfDistribution, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430149
            Share of income must be present

            Legacy Rule Format:
            ^PTR351 > 0 AND ^PTR158 = NULL AND ^PTR159 = NULL AND ^PTR324 = NULL

            Technical Business Rule Format:
            ^PTR351 > 0 AND ^PTR158 = NULL AND ^PTR159 = NULL AND ^PTR324 = NULL

            Data Elements:

            ^PTR158 = PTR:RP:StatementOfDistribution:IncomeOrLoss:ShareOfIncomePrimaryProduction:Income.BeneficiaryShare.Amount

            ^PTR159 = PTR:RP:StatementOfDistribution:IncomeOrLoss:ShareOfIncomeNonPrimaryProduction:Income.BeneficiaryShare.Amount

            ^PTR324 = PTR:RP:StatementOfDistribution:IncomeOrLoss:Income.DistributionFranked.Amount

            ^PTR351 = PTR:RP:StatementOfDistribution:IncomeOrLoss:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromForeignResidents.Amount
            */
            assertion = (statementOfDistribution.PTR351.GetValueOrDefault() > 0 && statementOfDistribution.PTR158 == null && statementOfDistribution.PTR159 == null && statementOfDistribution.PTR324 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.430709",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Share of income must be present",
                    LongDescription = @"In a Statement of distribution, if Credit for tax withheld - foreign resident withholding is greater than zero, then there must be an amount at Share of income - primary production or Share of income - non-primary production or Franked distibutions",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:IncomeOrLoss/tns:ShareOfIncomePrimaryProduction/tns:BeneficiaryShareA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430149" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR351", Value = GetValueOrEmpty(statementOfDistribution.PTR351) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR158", Value = GetValueOrEmpty(statementOfDistribution.PTR158) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR159", Value = GetValueOrEmpty(statementOfDistribution.PTR159) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR324", Value = GetValueOrEmpty(statementOfDistribution.PTR324) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430151
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430151()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430151
            Tax withheld where ABN not quoted must equal distributed amounts
    
            Legacy Rule Format:
            (^PTR63 + ^PTR73) <> Sum(^PTR160)

            Technical Business Rule Format:
            (^PTR63 + ^PTR73) <> Sum(^PTR160)
    
            Data Elements:
    
            ^PTR63 = PTR:RP:Income:BusinessIncomeAndExpenses:PayAsYouGoWithholding:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount
    
            ^PTR73 = PTR:RP:Income:ShareOfCreditsFromIncome:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuotedShareReceivedIndirectly.Amount
    
            ^PTR160 = PTR:RP:StatementOfDistribution:IncomeOrLoss:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount
            */
            assertion = (report.PTR63.GetValueOrDefault() + report.PTR73.GetValueOrDefault() != (report.RP_StatementOfDistributionCollection == null ? 0 : report.RP_StatementOfDistributionCollection.Sum(f => f.PTR160.GetValueOrDefault())));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.430151",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax withheld where ABN not quoted must equal distributed amounts",
                    LongDescription = @"The total amount of Credit for tax withheld where ABN not quoted in the Statements of distribution, must equal the sum of Tax withheld where ABN not quoted, and Share of credit for tax withheld where ABN not quoted",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:BusinessIncomeAndExpenses/tns:PayAsYouGoWithholding/tns:TaxCreditForTaxWithheldWhereABNNotQuotedA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430151" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR63", Value = GetValueOrEmpty(report.PTR63) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR73", Value = GetValueOrEmpty(report.PTR73) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR160", Value = report.RP_StatementOfDistributionCollection == null ? "0" : report.RP_StatementOfDistributionCollection.Sum(f => f.PTR160.GetValueOrDefault()).ToString() });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430153
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430153()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430153
            Franking credit amounts must equal distributed amounts
    
            Legacy Rule Format:
            (^PTR74 + ^PTR86) <> Sum(^PTR161)

            Technical Business Rule Format:
            (^PTR74 + ^PTR86) <> Sum(^PTR161)
    
            Data Elements:
    
            ^PTR74 = PTR:RP:Income:ShareOfCreditsFromIncome:IncomeTax.FrankingCredits.FrankingCreditShareReceivedIndirectly.Amount
    
            ^PTR86 = PTR:RP:Income:Dividends:IncomeTax.FrankingCredits.Amount
    
            ^PTR161 = PTR:RP:StatementOfDistribution:IncomeOrLoss:IncomeTax.FrankingCredits.ReceivedFromAustralianCompanies.Amount
            */
            assertion = (report.PTR74.GetValueOrDefault() + report.PTR86.GetValueOrDefault() != (report.RP_StatementOfDistributionCollection == null ? 0 : report.RP_StatementOfDistributionCollection.Sum(f => f.PTR161.GetValueOrDefault())));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.440153",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Franking credit amounts must equal distributed amounts",
                    LongDescription = @"The sum of all Franking Credit in the Statements of distribution must equal the sum of Share of franking credit from franked distributions, plus Franking credit",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:ShareOfCreditsFromIncome/tns:TaxFrankingCreditsFrankingCreditShareReceivedIndirectlyA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430153" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR74", Value = GetValueOrEmpty(report.PTR74) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR86", Value = GetValueOrEmpty(report.PTR86) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR161", Value = report.RP_StatementOfDistributionCollection == null ? "0" : report.RP_StatementOfDistributionCollection.Sum(f => f.PTR161.GetValueOrDefault()).ToString() });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430155
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430155()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430155
            TFN amounts withheld must equal distributed amounts
    
            Legacy Rule Format:
            (^PTR75 + ^PTR83 + ^PTR87) <> Sum(^PTR162)

            Technical Business Rule Format:
            (^PTR75 + ^PTR83 + ^PTR87) <> Sum(^PTR162)
    
            Data Elements:
    
            ^PTR75 = PTR:RP:Income:ShareOfCreditsFromIncome:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromInterestAndDividendsWhereTFNNotQuotedShareReceivedIndirectly.Amount
    
            ^PTR83 = PTR:RP:Income:GrossInterest:IncomeTax.TFNAmountsWithheldFromGrossInterest.Amount
    
            ^PTR87 = PTR:RP:Income:Dividends:IncomeTax.DividendsTFNAmountsWithheld.Amount
    
            ^PTR162 = PTR:RP:StatementOfDistribution:IncomeOrLoss:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldWhereTFNNotQuoted.Amount
            */
            assertion = report.PTR75.GetValueOrDefault() + report.PTR83.GetValueOrDefault() + report.PTR87.GetValueOrDefault() != (report.RP_StatementOfDistributionCollection == null ? 0 : report.RP_StatementOfDistributionCollection.Sum(f => f.PTR162.GetValueOrDefault()));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.430155",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"TFN amounts withheld must equal distributed amounts",
                    LongDescription = @"The sum of TFN amounts withheld in the Statements of distribution must equal the sum of Share of credit for TFN amounts withheld from interest, dividends and unit trust distributions, TFN amounts withheld from gross interest, and TFN amounts withheld from dividends",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:ShareOfCreditsFromIncome/tns:TaxPayAsYouGoWithholdingCreditForAmountsWithheldFromInterestAndDividendsWhereTFNNotQuotedShareReceivedIndirectlyA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430155" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR75", Value = GetValueOrEmpty(report.PTR75) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR83", Value = GetValueOrEmpty(report.PTR83) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR87", Value = GetValueOrEmpty(report.PTR87) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR162", Value = report.RP_StatementOfDistributionCollection == null ? "0" : report.RP_StatementOfDistributionCollection.Sum(f => f.PTR162.GetValueOrDefault()).ToString() });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430158
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430158()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430158
            Credit for tax withheld - foreign resident withholding (excluding capital gains) amounts must equal distributed amounts
    
            Legacy Rule Format:
            (^PTR350 + ^PTR76) <> Sum(^PTR351)

            Technical Business Rule Format:
            (^PTR350 + ^PTR76) <> Sum(^PTR351)
    
            Data Elements:
    
            ^PTR350 = PTR:RP:Income:BusinessIncomeAndExpenses:PayAsYouGoWithholding:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromForeignResidents.Amount
    
            ^PTR76 = PTR:RP:Income:ShareOfCreditsFromIncome:InternationalDealings.CreditForTaxWithheldFromForeignResidentWithholdingShareReceivedIndirectly.Amount
    
            ^PTR351 = PTR:RP:StatementOfDistribution:IncomeOrLoss:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromForeignResidents.Amount
            */
            assertion = report.PTR350.GetValueOrDefault() + report.PTR76.GetValueOrDefault() != (report.RP_StatementOfDistributionCollection == null ? 0 : report.RP_StatementOfDistributionCollection.Sum(f => f.PTR351.GetValueOrDefault()));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.430158",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Credit for tax withheld - foreign resident withholding (excluding capital gains) amounts must equal distributed amounts",
                    LongDescription = @"The total of Credit for tax withheld - foreign resident withholding (excluding capital gains) in the Statements of distribution, must equal the sum of Credit for tax withheld - foreign resident withholding (excluding capital gains), plus Share of credit for tax withheld from foreign resident withholding (excluding capital gains)",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:BusinessIncomeAndExpenses/tns:PayAsYouGoWithholding/tns:TaxCreditForAmountsWithheldFromForeignResidentsA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430158" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR350", Value = GetValueOrEmpty(report.PTR350) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR76", Value = GetValueOrEmpty(report.PTR76) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR351", Value = report.RP_StatementOfDistributionCollection == null ? "0" : report.RP_StatementOfDistributionCollection.Sum(f => f.PTR351.GetValueOrDefault()).ToString() });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430166
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430166()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430166
            The National rental affordability scheme tax offset entitlement amount must equal the sum of the Share of National rental affordability scheme tax offset amounts in the Statements of distribution
    
            Legacy Rule Format:
            ^PTR248 <> Sum(^PTR249)

            Technical Business Rule Format:
            ^PTR248 <> Sum(^PTR249)
    
            Data Elements:
    
            ^PTR248 = PTR:RP:TaxOffsetClaim.NationalRentalAffordabilitySchemeEntitlement.Amount
    
            ^PTR249 = PTR:RP:StatementOfDistribution:IncomeOrLoss:TaxOffsetClaim.NationalRentalAffordabilitySchemeEntitlement.Amount
            */
            assertion = (report.PTR248.GetValueOrDefault() != (report.RP_StatementOfDistributionCollection == null ? 0 : report.RP_StatementOfDistributionCollection.Sum(f => f.PTR249.GetValueOrDefault())));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.430166",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total National rental affordability scheme tax offset amount must equal sum of distributed amounts",
                    LongDescription = @"The National rental affordability scheme tax offset entitlement amount must equal the sum of the Share of National rental affordability scheme tax offset amounts in the Statements of distribution",
                    Location = "/tns:PTR/tns:RP/tns:TaxOffsetClaimNationalRentalAffordabilitySchemeEntitlementA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430166" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR248", Value = GetValueOrEmpty(report.PTR248) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Sum(PTR249)", Value = report.RP_StatementOfDistributionCollection == null ? "0" : report.RP_StatementOfDistributionCollection.Sum(f => f.PTR249.GetValueOrDefault()).ToString() });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430171
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430171()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430171
            Net rental income must equal the sum of all partners' Share of net rental income
    
            Legacy Rule Format:
            ^PTR255 <> Sum(^PTR257)

            Technical Business Rule Format:
            ^PTR255 <> Sum(^PTR257)
    
            Data Elements:
    
            ^PTR255 = PTR:RP:IncomeTests:Income.RentalIncomeOrLossNet.Amount
    
            ^PTR257 = PTR:RP:StatementOfDistribution:IncomeOrLoss:Income.RentalIncomeOrLossNet.Amount
            */
            assertion = (report.PTR255.GetValueOrDefault() != (report.RP_StatementOfDistributionCollection == null ? 0 : report.RP_StatementOfDistributionCollection.Sum(f => f.PTR257.GetValueOrDefault())));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.430171",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Net rental income must equal the sum of all partners' Share of net rental income",
                    LongDescription = @"The amount at Net rental property income or loss must equal the sum of all amounts recorded in the Statement of distribution - Share of net rental property income or loss",
                    Location = "/tns:PTR/tns:RP/tns:IncomeTests/tns:RentalIncomeOrLossNetA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430171" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR255", Value = GetValueOrEmpty(report.PTR255) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Sum(PTR257)", Value = report.RP_StatementOfDistributionCollection == null ? "0" : report.RP_StatementOfDistributionCollection.Sum(f => f.PTR257.GetValueOrDefault()).ToString() });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430172
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430172()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430172
            Net financial investment income must equal the sum of all partners' Share of net financial income
    
            Legacy Rule Format:
            ^PTR254 <> Sum(^PTR256)

            Technical Business Rule Format:
            ^PTR254 <> Sum(^PTR256)
    
            Data Elements:
    
            ^PTR254 = PTR:RP:IncomeTests:Income.Investment.FinancialIncomeOrLossNet.Amount
    
            ^PTR256 = PTR:RP:StatementOfDistribution:IncomeOrLoss:Income.Investment.FinancialIncomeOrLossNet.Amount
            */
            assertion = (report.PTR254.GetValueOrDefault() != (report.RP_StatementOfDistributionCollection == null ? 0 : report.RP_StatementOfDistributionCollection.Sum(f => f.PTR256.GetValueOrDefault())));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.430172",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Net financial investment income must equal the sum of all partners' Share of net financial income",
                    LongDescription = @"The amount at Net financial investment income or loss must equal the sum of all amounts recorded in the Statement of distribution - Share of net financial investment income or loss",
                    Location = "/tns:PTR/tns:RP/tns:IncomeTests/tns:InvestmentFinancialIncomeOrLossNetA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430172" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR254", Value = GetValueOrEmpty(report.PTR254) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Sum(PTR256)", Value = report.RP_StatementOfDistributionCollection == null ? "0" : report.RP_StatementOfDistributionCollection.Sum(f => f.PTR256.GetValueOrDefault()).ToString() });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430177
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430177()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430177
            Interposed Entity Election Status - Specified income year must be in range 1995 to the current income year
    
            Legacy Rule Format:
            ^PTR26 <> NULL AND (^PTR26 < 1995 OR ^PTR26 > ^PTR1)

            Technical Business Rule Format:
            ^PTR26 <> NULL AND (^PTR26 < 1995 OR ^PTR26 > ^PTR1)
    
            Data Elements:
    
            ^PTR26 = PTR:RP:Elections.InterposedEntityElectionStatus.Year
    
            ^PTR1 = PTR:RP:Report.TargetFinancial.Year
            */
            assertion = (report.PTR26 != null && (report.PTR26.GetValueOrDefault() < 1995 || report.PTR26.GetValueOrDefault() > report.PTR1.GetValueOrDefault()));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430702",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Interposed Entity Election Status - Specified income year must be in range 1995 to current year",
                    Location = "/tns:PTR/tns:RP/tns:ElectionsInterposedEntityElectionStatusY",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430177" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR26", Value = GetValueOrEmpty(report.PTR26) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR1", Value = GetValueOrEmpty(report.PTR1) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430194
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430194()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430194
            All current assets must not exceed Total assets
    
            Legacy Rule Format:
            ^PTR111 > ^PTR112

            Technical Business Rule Format:
            ^PTR111 > ^PTR112
    
            Data Elements:
    
            ^PTR111 = PTR:RP:KeyFinancialInformation:Assets.CurrentTotal.Amount
    
            ^PTR112 = PTR:RP:KeyFinancialInformation:Assets.Total.Amount
            */
            assertion = (report.PTR111.GetValueOrDefault() > report.PTR112.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430194",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"All current assets must not exceed Total assets",
                    Location = "/tns:PTR/tns:RP/tns:KeyFinancialInformation/tns:AssetsCurrentTotalA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430194" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR111", Value = GetValueOrEmpty(report.PTR111) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR112", Value = GetValueOrEmpty(report.PTR112) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430195
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430195()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430195
            All current liabilities must not exceed Total liabilities
    
            Legacy Rule Format:
            ^PTR113 > ^PTR114

            Technical Business Rule Format:
            ^PTR113 > ^PTR114
    
            Data Elements:
    
            ^PTR113 = PTR:RP:KeyFinancialInformation:Liabilities.CurrentTotal.Amount
    
            ^PTR114 = PTR:RP:KeyFinancialInformation:Liabilities.Total.Amount
            */
            assertion = (report.PTR113.GetValueOrDefault() > report.PTR114.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430195",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"All current liabilities must not exceed Total liabilities",
                    Location = "/tns:PTR/tns:RP/tns:KeyFinancialInformation/tns:LiabilitiesCurrentTotalA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430195" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR113", Value = GetValueOrEmpty(report.PTR113) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR114", Value = GetValueOrEmpty(report.PTR114) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430197
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430197()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430197
            Closing stock amount must be present if Code is present
    
            Legacy Rule Format:
            ^PTR124 <> NULL AND ^PTR123 = NULL

            Technical Business Rule Format:
            ^PTR124 <> NULL AND ^PTR123 = NULL
    
            Data Elements:
    
            ^PTR123 = PTR:RP:BusinessAndProfessionaltems:Closing:Assets.Inventories.Total.Amount
    
            ^PTR124 = PTR:RP:BusinessAndProfessionaltems:Closing:Assets.Inventories.ValuationMethod.Code
            */
            assertion = (report.PTR124 != null && report.PTR123 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430197",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Closing stock amount must be present if Code is present",
                    Location = "/tns:PTR/tns:RP/tns:BusinessAndProfessionaltems/tns:Closing/tns:AssetsInventoriesTotalA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430197" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR124", Value = GetValueOrEmpty(report.PTR124) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR123", Value = GetValueOrEmpty(report.PTR123) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430198
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430198()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430198
            Closing stock action code must be present if Closing stock amount is present
    
            Legacy Rule Format:
            ^PTR124 = NULL AND ^PTR123 > 0

            Technical Business Rule Format:
            ^PTR124 = NULL AND ^PTR123 > 0
    
            Data Elements:
    
            ^PTR124 = PTR:RP:BusinessAndProfessionaltems:Closing:Assets.Inventories.ValuationMethod.Code
    
            ^PTR123 = PTR:RP:BusinessAndProfessionaltems:Closing:Assets.Inventories.Total.Amount
            */
            assertion = (report.PTR124 == null && report.PTR123.GetValueOrDefault() > 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430198",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Closing stock action code must be present if Closing stock amount is present",
                    Location = "/tns:PTR/tns:RP/tns:BusinessAndProfessionaltems/tns:Closing/tns:AssetsInventoriesValuationMethodC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430198" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR124", Value = GetValueOrEmpty(report.PTR124) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR123", Value = GetValueOrEmpty(report.PTR123) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430199
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430199()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430199
            Trade debtors amount must not exceed All current assets amount
    
            Legacy Rule Format:
            ^PTR111 < ^PTR125

            Technical Business Rule Format:
            ^PTR111 < ^PTR125
    
            Data Elements:
    
            ^PTR125 = PTR:RP:BusinessAndProfessionaltems:Closing:Assets.TradeandReceivablesOther.Amount
    
            ^PTR111 = PTR:RP:KeyFinancialInformation:Assets.CurrentTotal.Amount
            */
            assertion = (report.PTR111.GetValueOrDefault() < report.PTR125.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430199",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Trade debtors amount must not exceed All current assets amount",
                    Location = "/tns:PTR/tns:RP/tns:BusinessAndProfessionaltems/tns:Closing/tns:AssetsTradeandReceivablesOtherA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430199" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR111", Value = GetValueOrEmpty(report.PTR111) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR125", Value = GetValueOrEmpty(report.PTR125) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430200
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430200()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430200
            Trade creditors cannot exceed all current liabilities in Financial and other information
    
            Legacy Rule Format:
            ^PTR113 < ^PTR126

            Technical Business Rule Format:
            ^PTR113 < ^PTR126
    
            Data Elements:
    
            ^PTR126 = PTR:RP:BusinessAndProfessionaltems:Closing:Liabilities.TradeAndOtherPayablesTotal.Amount
    
            ^PTR113 = PTR:RP:KeyFinancialInformation:Liabilities.CurrentTotal.Amount
            */
            assertion = (report.PTR113.GetValueOrDefault() < report.PTR126.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430200",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Trade creditors cannot exceed all current liabilities in Financial and other information",
                    Location = "/tns:PTR/tns:RP/tns:BusinessAndProfessionaltems/tns:Closing/tns:LiabilitiesTradeAndOtherPayablesTotalA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430200" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR113", Value = GetValueOrEmpty(report.PTR113) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR126", Value = GetValueOrEmpty(report.PTR126) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430207
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430207()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430207
            Industry code must be present
    
            Legacy Rule Format:
            ^PTR31 = NULL AND (^PTR35 <> NULL OR ^PTR36 <> NULL OR ^PTR38 <> NULL OR ^PTR39 <> NULL OR ^PTR41 <> NULL OR ^PTR43 <> NULL OR ^PTR44 <> NULL)

            Technical Business Rule Format:
            ^PTR31 = NULL AND (^PTR35 <> NULL OR ^PTR36 <> NULL OR ^PTR38 <> NULL OR ^PTR39 <> NULL OR ^PTR41 <> NULL OR ^PTR43 <> NULL OR ^PTR44 <> NULL)
    
            Data Elements:
    
            ^PTR31 = PTR:RP:OrganisationDetails.OrganisationIndustry2006Extended.Code
    
            ^PTR35 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^PTR36 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^PTR38 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Remuneration.PaymentToForeignResidentGross.Amount
    
            ^PTR39 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^PTR41 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^PTR43 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Income.Other.Amount
    
            ^PTR44 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Income.Other.Amount
            */
            assertion = (report.PTR31 == null && (report.PTR35 != null || report.PTR36 != null || report.PTR38 != null || report.PTR39 != null || report.PTR41 != null || report.PTR43 != null || report.PTR44 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430207",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"If  business income (PP or non-PP) is present, an Industry code (ANZSIC) must be present",
                    Location = "/tns:PTR/tns:RP/tns:OrganisationDetailsOrganisationIndustry2006ExtendedC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430207" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR31", Value = GetValueOrEmpty(report.PTR31) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR35", Value = GetValueOrEmpty(report.PTR35) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR36", Value = GetValueOrEmpty(report.PTR36) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR38", Value = GetValueOrEmpty(report.PTR38) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR39", Value = GetValueOrEmpty(report.PTR39) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR41", Value = GetValueOrEmpty(report.PTR41) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR43", Value = GetValueOrEmpty(report.PTR43) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR44", Value = GetValueOrEmpty(report.PTR44) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430212
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430212()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430212
            Net rental property income or loss, and shared distributions must be present
    
            Legacy Rule Format:
            AnyOccurrence(^PTR257 , ^PTR257 = NULL) AND (^PTR77 + ^PTR78 + ^PTR79 + ^PTR80) > 0 AND ^PTR255 = NULL

            Technical Business Rule Format:
            AnyOccurrence(^PTR257 , ^PTR257 = NULL) AND (^PTR77 + ^PTR78 + ^PTR79 + ^PTR80) > 0 AND ^PTR255 = NULL
    
            Data Elements:
    
            ^PTR255 = PTR:RP:IncomeTests:Income.RentalIncomeOrLossNet.Amount
    
            ^PTR77 = PTR:RP:Income:Rent:Income.Operating.RentalIncomeGross.Amount
    
            ^PTR78 = PTR:RP:Income:Rent:Expense.Interest.Amount
    
            ^PTR79 = PTR:RP:Income:Rent:Expense.CapitalWorksDeduction.Amount
    
            ^PTR80 = PTR:RP:Income:Rent:IncomeTax.Deduction.RentalIncomeDeductionsOtherThanInterestAndCapitalWorks.Amount
    
            ^PTR257 = PTR:RP:StatementOfDistribution:IncomeOrLoss:Income.RentalIncomeOrLossNet.Amount
            */
            assertion = ((report.RP_StatementOfDistributionCollection == null ? false : report.RP_StatementOfDistributionCollection.Any(PTR257Repeat => PTR257Repeat.PTR257 == null)) && report.PTR77.GetValueOrDefault() + report.PTR78.GetValueOrDefault() + report.PTR79.GetValueOrDefault() + report.PTR80.GetValueOrDefault() > 0 && report.PTR255 == null);

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.430212",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Net rental property income or loss, and shared distributions must be present",
                    LongDescription = @"If Rent amounts greater than zero (including Gross rent and deductions) are present, then Net rental property income or loss must be present, and Share of net rental property income or loss must be present in each partner's Statement of distribution",
                    Location = "/tns:PTR/tns:RP/tns:IncomeTests/tns:RentalIncomeOrLossNetA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430212" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR257", Value = string.Empty });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR77", Value = GetValueOrEmpty(report.PTR77) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR78", Value = GetValueOrEmpty(report.PTR78) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR79", Value = GetValueOrEmpty(report.PTR79) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR80", Value = GetValueOrEmpty(report.PTR80) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR255", Value = GetValueOrEmpty(report.PTR255) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430213
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430213()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430213
            Net financial investment income or loss, and shared distributions must be present
    
            Legacy Rule Format:
            AnyOccurrence(^PTR256, ^PTR256 = NULL) AND (^PTR81 > 0 OR ^PTR84 > 0 OR ^PTR85 > 0 OR ^PTR86 > 0 OR ((^PTR82 = NULL OR ^PTR82 = 0) AND ^PTR90 > 0) OR ^PTR91 > 0) AND ^PTR254 = NULL

            Technical Business Rule Format:
            AnyOccurrence(^PTR256, ^PTR256 = NULL) AND (^PTR81 > 0 OR ^PTR84 > 0 OR ^PTR85 > 0 OR ^PTR86 > 0 OR ((^PTR82 = NULL OR ^PTR82 = 0) AND ^PTR90 > 0) OR ^PTR91 > 0) AND ^PTR254 = NULL
    
            Data Elements:
    
            ^PTR254 = PTR:RP:IncomeTests:Income.Investment.FinancialIncomeOrLossNet.Amount
    
            ^PTR81 = PTR:RP:Income:Income.ForestryManagedInvestmentScheme.Amount
    
            ^PTR82 = PTR:RP:Income:GrossInterest:Income.Interest.Gross.Amount
    
            ^PTR84 = PTR:RP:Income:Dividends:Income.DividendsUnfranked.Amount
    
            ^PTR85 = PTR:RP:Income:Dividends:Income.DividendsFranked.Amount
    
            ^PTR86 = PTR:RP:Income:Dividends:IncomeTax.FrankingCredits.Amount
    
            ^PTR90 = PTR:RP:Expense.AustralianInvestmentAllowableDeduction.Amount
    
            ^PTR91 = PTR:RP:Expense.ForestryManagedInvestmentSchemeDeduction.Amount
    
            ^PTR256 = PTR:RP:StatementOfDistribution:IncomeOrLoss:Income.Investment.FinancialIncomeOrLossNet.Amount
            */
            assertion = ((report.RP_StatementOfDistributionCollection == null ? false : report.RP_StatementOfDistributionCollection.Any(PTR256Repeat => PTR256Repeat.PTR256 == null)) && (report.PTR81.GetValueOrDefault() > 0 || report.PTR84.GetValueOrDefault() > 0 || report.PTR85.GetValueOrDefault() > 0 || report.PTR86.GetValueOrDefault() > 0 || (report.PTR82 == null || report.PTR82 == 0) && report.PTR90.GetValueOrDefault() > 0 || report.PTR91.GetValueOrDefault() > 0) && report.PTR254 == null);

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.430213",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Net financial investment income or loss, and shared distributions must be present",
                    LongDescription = @"If an amount greater than zero is present at FMIS income, or Dividend income, or Deductions relating to Australian investment income and Gross interest is not present, then Net financial investment income or loss must be present, and Share of net financial investment income or loss must be present in each partner's Statement of Distribution",
                    Location = "/tns:PTR/tns:RP/tns:IncomeTests/tns:InvestmentFinancialIncomeOrLossNetA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430213" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR256", Value = string.Empty });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR81", Value = GetValueOrEmpty(report.PTR81) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR84", Value = GetValueOrEmpty(report.PTR84) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR85", Value = GetValueOrEmpty(report.PTR85) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR86", Value = GetValueOrEmpty(report.PTR86) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR82", Value = GetValueOrEmpty(report.PTR82) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR90", Value = GetValueOrEmpty(report.PTR90) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR91", Value = GetValueOrEmpty(report.PTR91) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR254", Value = GetValueOrEmpty(report.PTR254) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430314
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430314()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430314
            The 'Share of credit for TFN amounts withheld from payments from closely held trusts' must equal the sum all 'Partner share - Share of credit for TFN amounts withheld from payments from closely held trusts'
    
            Legacy Rule Format:
            ^PTR320 <> Sum(^PTR321)

            Technical Business Rule Format:
            ^PTR320 <> Sum(^PTR321)
    
            Data Elements:
    
            ^PTR320 = PTR:RP:Income:ShareOfCreditsFromIncome:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldFromCloselyHeldTrustShareReceivedIndirectly.Amount
    
            ^PTR321 = PTR:RP:StatementOfDistribution:IncomeOrLoss:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldFromCloselyHeldTrust.Amount
            */
            assertion = report.PTR320.GetValueOrDefault() != (report.RP_StatementOfDistributionCollection == null ? 0 : report.RP_StatementOfDistributionCollection.Sum(f => f.PTR321.GetValueOrDefault()));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.430314",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Amount withheld from closely held trusts must equal sum of all partners' shares",
                    LongDescription = @"The 'Share of credit for TFN amounts withheld from payments from closely held trusts' must equal the sum all 'Partner share - Share of credit for TFN amounts withheld from payments from closely held trusts' in the Statements of Distribution",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:ShareOfCreditsFromIncome/tns:TaxPayAsYouGoWithholdingCreditForTaxWithheldFromCloselyHeldTrustShareReceivedIndirectlyA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430314" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR320", Value = GetValueOrEmpty(report.PTR320) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR321", Value = report.RP_StatementOfDistributionCollection == null ? "0" : report.RP_StatementOfDistributionCollection.Sum(f => f.PTR321.GetValueOrDefault()).ToString() });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.430705
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR430705()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.430705
            When there is an amount present at non-primary production deductions relating to franked distributions from trusts, there must be an amount at non-primary production franked distributions from trusts.
    
            Legacy Rule Format:
            ^PTR323 > 0 AND ^PTR322 = NULL

            Technical Business Rule Format:
            ^PTR323 > 0 AND ^PTR322 = NULL
    
            Data Elements:
    
            ^PTR323 = PTR:RP:Income:PartnershipsTrustsNonPrimaryProduction:IncomeTax.Deduction.TrustDistributionFranked.Amount
    
            ^PTR322 = PTR:RP:Income:PartnershipsTrustsNonPrimaryProduction:Income.TrustDistributionFranked.Amount
            */
            assertion = (report.PTR323.GetValueOrDefault() > 0 && report.PTR322 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.430705",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-PP franked distributions from trusts income must be present",
                    LongDescription = @"When there is an amount present at non-primary production deductions relating to franked distributions from trusts, there must be an amount at non-primary production franked distributions from trusts",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:PartnershipsTrustsNonPrimaryProduction/tns:TaxDeductionTrustDistributionFrankedA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.430705" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR323", Value = GetValueOrEmpty(report.PTR323) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR322", Value = GetValueOrEmpty(report.PTR322) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.431006
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR431006()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.431006
            Share of net income from trusts, less capital gains, foreign income and franked distributions action code must be present
    
            Legacy Rule Format:
            ^PTR327 = NULL AND ^PTR70 > 0

            Technical Business Rule Format:
            ^PTR327 = NULL AND ^PTR70 > 0
    
            Data Elements:
    
            ^PTR327 = PTR:RP:Income:PartnershipsTrustsNonPrimaryProduction:Party.TrustType.Code
    
            ^PTR70 = PTR:RP:Income:PartnershipsTrustsNonPrimaryProduction:Income.TrustShareNetExcludeNetCapitalGainsAndForeignIncomeAndDistributionFranked.Amount
            */
            assertion = (report.PTR327 == null && report.PTR70.GetValueOrDefault() > 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.431006",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Share of net income from trusts, less capital gains, foreign income and franked distributions action code must be present",
                    LongDescription = @"If Non-primary production - Share of net income from trusts, less capital gains, foreign income and franked distributions is greater than zero, the Non-primary production - Share of net income from trusts, less capital gains, foreign income and franked distributions iaction code must be present",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:PartnershipsTrustsNonPrimaryProduction/tns:PartyTrustTypeC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.431006" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR327", Value = GetValueOrEmpty(report.PTR327) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR70", Value = GetValueOrEmpty(report.PTR70) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.431007
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR431007()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.431007
            Non-PP - Deductions relating to distribution amounts can not be more than zero
    
            Legacy Rule Format:
            ^PTR72 > 0 AND ^PTR69 = NULL AND ^PTR70 = NULL

            Technical Business Rule Format:
            ^PTR72 > 0 AND ^PTR69 = NULL AND ^PTR70 = NULL
    
            Data Elements:
    
            ^PTR72 = PTR:RP:Income:PartnershipsTrustsNonPrimaryProduction:Expense.DistributionDeductionsOther.Amount
    
            ^PTR69 = PTR:RP:Income:PartnershipsTrustsNonPrimaryProduction:Income.PartnershipDistributionLessForeignIncome.Amount
    
            ^PTR70 = PTR:RP:Income:PartnershipsTrustsNonPrimaryProduction:Income.TrustShareNetExcludeNetCapitalGainsAndForeignIncomeAndDistributionFranked.Amount
            */
            assertion = (report.PTR72.GetValueOrDefault() > 0 && report.PTR69 == null && report.PTR70 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.431007",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-PP - Deductions relating to distribution amounts can not be more than zero",
                    LongDescription = @"If there is an amount present at Non-primary production- Deductions relating to amounts shown at Distribution from partnerships less foreign income or Non-primary production - Share of net income from trusts less capital gains, foreign income and franked distributions there must be an amount at either Non-primary production - Distribution from partnerships less foreign income or Non-primary production - Share of net income from trusts less capital gains, foreign income and franked distributions",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:PartnershipsTrustsNonPrimaryProduction/tns:ExpenseDistributionDeductionsOtherA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.431007" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR72", Value = GetValueOrEmpty(report.PTR72) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR69", Value = GetValueOrEmpty(report.PTR69) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR70", Value = GetValueOrEmpty(report.PTR70) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.431008
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR431008()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.431008
            Franking credits from a New Zealand franking company must not be less than distributed amounts
    
            Legacy Rule Format:
            Sum(^PTR164) > ^PTR107

            Technical Business Rule Format:
            Sum(^PTR164) > ^PTR107
    
            Data Elements:
    
            ^PTR107 = PTR:RP:OtherAssessableForeignSourceIncome:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
    
            ^PTR164 = PTR:RP:StatementOfDistribution:IncomeOrLoss:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
            */
            assertion = ((report.RP_StatementOfDistributionCollection == null ? 0 : report.RP_StatementOfDistributionCollection.Sum(f => f.PTR164.GetValueOrDefault())) > report.PTR107.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.431008",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Franking credits from a New Zealand franking company must not be less than distributed amounts",
                    LongDescription = @"The sum of Australian franking credits from a New Zealand franking company in the Statements of distribution must not be greater than Australian franking credits from a New Zealand franking company reported at Other assessable foreign income",
                    Location = "/tns:PTR/tns:RP/tns:OtherAssessableForeignSourceIncome/tns:IncomeTaxFrankingCreditsReceivedFromNewZealandCompaniesA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.431008" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Sum(PTR164)", Value = report.RP_StatementOfDistributionCollection.Sum(f => f.PTR164.GetValueOrDefault()).ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR107", Value = GetValueOrEmpty(report.PTR107) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.431009
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR431009()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.431009
            Share of net income from trusts action code must be present
    
            Legacy Rule Format:
            ^PTR299 = NULL AND ^PTR66 > 0

            Technical Business Rule Format:
            ^PTR299 = NULL AND ^PTR66 > 0
    
            Data Elements:
    
            ^PTR299 = PTR:RP:Income:PartnershipsTrustsPrimaryProduction:Party.TrustType.Code
    
            ^PTR66 = PTR:RP:Income:PartnershipsTrustsPrimaryProduction:Income.TrustShareNet.Amount
            */
            assertion = (report.PTR299 == null && report.PTR66.GetValueOrDefault() > 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.431009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Share of net income from trusts action code must be present",
                    LongDescription = @"If Primary production - Share of net income  from trusts is greater than zero, the Primary production - Share of net income trusts action code must be present",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:PartnershipsTrustsPrimaryProduction/tns:PartyTrustTypeC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.431009" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR299", Value = GetValueOrEmpty(report.PTR299) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR66", Value = GetValueOrEmpty(report.PTR66) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.431030
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR431030()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.431030
            PP - Distribution from trusts amount must be greater than zero if action code is present
    
            Legacy Rule Format:
            (^PTR66 = NULL OR ^PTR66 <= 0) AND ^PTR299 <> NULL

            Technical Business Rule Format:
            (^PTR66 = NULL OR ^PTR66 <= 0) AND ^PTR299 <> NULL
    
            Data Elements:
    
            ^PTR66 = PTR:RP:Income:PartnershipsTrustsPrimaryProduction:Income.TrustShareNet.Amount
    
            ^PTR299 = PTR:RP:Income:PartnershipsTrustsPrimaryProduction:Party.TrustType.Code
            */
            assertion = ((report.PTR66 == null || report.PTR66.GetValueOrDefault() <= 0) && report.PTR299 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.431030",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"PP - Share of net income from trusts amount must be greater than zero if action code is present",
                    LongDescription = @"If the Primary production - Share of net income  from trusts action code is present, then the Partnership and Trusts - Primary production - Share of net income  from trusts must be present and greater than zero. An action code is not required for negative or zero amounts",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:PartnershipsTrustsPrimaryProduction/tns:TrustShareNetA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.431030" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR66", Value = GetValueOrEmpty(report.PTR66) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR299", Value = GetValueOrEmpty(report.PTR299) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.431031
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR431031()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.431031
            Total distributions and income from a trust must not be less than zero
    
            Legacy Rule Format:
            (^PTR66 + ^PTR70 + ^PTR322 + ^PTR98 + ^PTR100 + ^PTR106 + ^PTR107) < 0 AND (^PTR105 = NULL OR ^PTR105 = 0)

            Technical Business Rule Format:
            (^PTR66 + ^PTR70 + ^PTR322 + ^PTR98 + ^PTR100 + ^PTR106 + ^PTR107) < 0 AND (^PTR105 = NULL OR ^PTR105 = 0)
    
            Data Elements:
    
            ^PTR66 = PTR:RP:Income:PartnershipsTrustsPrimaryProduction:Income.TrustShareNet.Amount
    
            ^PTR70 = PTR:RP:Income:PartnershipsTrustsNonPrimaryProduction:Income.TrustShareNetExcludeNetCapitalGainsAndForeignIncomeAndDistributionFranked.Amount
    
            ^PTR98 = PTR:RP:AttributedForeignIncome:ListedCountry:Income.InternationalDealings.Attributed.Amount
    
            ^PTR100 = PTR:RP:AttributedForeignIncome:UnlistedCountry:Income.InternationalDealings.Attributed.Amount
    
            ^PTR105 = PTR:RP:OtherAssessableForeignSourceIncome:Income.Net.Amount
    
            ^PTR106 = PTR:RP:OtherAssessableForeignSourceIncome:Income.InternationalDealings.TaxOffset.Amount
    
            ^PTR107 = PTR:RP:OtherAssessableForeignSourceIncome:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
    
            ^PTR322 = PTR:RP:Income:PartnershipsTrustsNonPrimaryProduction:Income.TrustDistributionFranked.Amount
            */
            assertion = (report.PTR66.GetValueOrDefault() + report.PTR70.GetValueOrDefault() + report.PTR322.GetValueOrDefault() + report.PTR98.GetValueOrDefault() + report.PTR100.GetValueOrDefault() + report.PTR106.GetValueOrDefault() + report.PTR107.GetValueOrDefault() < 0 && (report.PTR105 == null || report.PTR105 == 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.431031",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total distributions and income from a trust must not be less than zero",
                    LongDescription = @"The sum of Primary production - Share of net income from trusts, Non-primary production - Share of net income from trusts, less capital gains, foreign income and franked distributions, Franked distributions from trusts, Attributed foreign income, General net foreign source income, Australian franking credits from a New Zealand franking company can not be less than zero. This would indicate that an overall trust loss is being distributed from a trust to a partnership. Adjust distributed amounts until the overall negative amount no longer exists (i.e. make the sum equal to zero)",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:PartnershipsTrustsPrimaryProduction/tns:TrustShareNetA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.431031" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR66", Value = GetValueOrEmpty(report.PTR66) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR70", Value = GetValueOrEmpty(report.PTR70) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR322", Value = GetValueOrEmpty(report.PTR322) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR98", Value = GetValueOrEmpty(report.PTR98) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR100", Value = GetValueOrEmpty(report.PTR100) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR106", Value = GetValueOrEmpty(report.PTR106) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR107", Value = GetValueOrEmpty(report.PTR107) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR105", Value = GetValueOrEmpty(report.PTR105) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.431043
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR431043()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.431043
            Non-PP - Distribution from trusts amount must be present
    
            Legacy Rule Format:
            (^PTR70 = NULL OR ^PTR70 = 0) AND ^PTR327 <> NULL

            Technical Business Rule Format:
            (^PTR70 = NULL OR ^PTR70 = 0) AND ^PTR327 <> NULL
    
            Data Elements:
    
            ^PTR70 = PTR:RP:Income:PartnershipsTrustsNonPrimaryProduction:Income.TrustShareNetExcludeNetCapitalGainsAndForeignIncomeAndDistributionFranked.Amount
    
            ^PTR327 = PTR:RP:Income:PartnershipsTrustsNonPrimaryProduction:Party.TrustType.Code
            */
            assertion = ((report.PTR70 == null || report.PTR70 == 0) && report.PTR327 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.431043",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-PP - Distribution from trusts amount must be present",
                    LongDescription = @"If Non-primary production - Distribution from trusts action code is present, then Non-primary production - Share of net income from trusts, less capital gains, foreign income and franked distributions must be greater than zero.  A trusts action code is not required for negative or zero amounts",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:PartnershipsTrustsNonPrimaryProduction/tns:TrustShareNetExcludeNetCapitalGainsAndForeignIncomeAndDistributionFrankedA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.431043" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR70", Value = GetValueOrEmpty(report.PTR70) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR327", Value = GetValueOrEmpty(report.PTR327) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.431049
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR431049(PTR2025.RP_StatementOfDistribution statementOfDistribution, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.431049
            Values for at least one of the Identifying fields must be provided - Date of Birth, TFN, Individual name, Address (Business and Residential), ABN, Non-Individual Name

            Legacy Rule Format:
            ^PTR146 = NULL AND ^PTR157 = NULL AND ^PTR284 = NULL AND ^PTR149 = NULL AND ^PTR285 = NULL AND ^PTR151 = NULL

            Technical Business Rule Format:
            ^PTR146 = NULL AND ^PTR157 = NULL AND ^PTR284 = NULL AND ^PTR149 = NULL AND ^PTR285 = NULL AND ^PTR151 = NULL

            Data Elements:

            ^PTR157 = PTR:RP:StatementOfDistribution:Identifiers.TaxFileNumber.Identifier

            ^PTR146 = PTR:RP:StatementOfDistribution:IndividualPartnerDetails:PersonNameDetails.FamilyName.Text

            ^PTR149 = PTR:RP:StatementOfDistribution:NonIndividualPartnerDetails:OrganisationNameDetails.OrganisationalName.Text

            ^PTR151 = PTR:RP:StatementOfDistribution:PartnerAddressDetails:AddressDetails.Line1.Text

            ^PTR284 = PTR:RP:StatementOfDistribution:IndividualPartnerDetails:PersonDemographicDetails.Birth.Date

            ^PTR285 = PTR:RP:StatementOfDistribution:NonIndividualPartnerDetails:Identifiers.AustralianBusinessNumber.Identifier
            */
            assertion = (statementOfDistribution.PTR146 == null && statementOfDistribution.PTR157 == null && statementOfDistribution.PTR284 == null && statementOfDistribution.PTR149 == null && statementOfDistribution.PTR285 == null && statementOfDistribution.PTR151 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.431049",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Identifying fields missing in the statement of distribution",
                    LongDescription = @"Values for at least one of the Identifying fields must be provided - Date of Birth, TFN, Individual name, Address (Business and Residential), ABN, Non-Individual Name.",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:TaxFileNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.431049" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR146", Value = statementOfDistribution.PTR146 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR157", Value = GetValueOrEmpty(statementOfDistribution.PTR157) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR284", Value = GetValueOrEmptyDate(statementOfDistribution.PTR284) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR149", Value = statementOfDistribution.PTR149 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR285", Value = GetValueOrEmpty(statementOfDistribution.PTR285) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR151", Value = statementOfDistribution.PTR151 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.431050
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR431050(PTR2025.RP_StatementOfDistribution statementOfDistribution, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.431050
            Australian Business Number and
            Individual's Surname/family name must not be accepted together

            Legacy Rule Format:
            ^PTR285 <> NULL AND ^PTR146 <> NULL

            Technical Business Rule Format:
            ^PTR285 <> NULL AND ^PTR146 <> NULL

            Data Elements:

            ^PTR146 = PTR:RP:StatementOfDistribution:IndividualPartnerDetails:PersonNameDetails.FamilyName.Text

            ^PTR285 = PTR:RP:StatementOfDistribution:NonIndividualPartnerDetails:Identifiers.AustralianBusinessNumber.Identifier
            */
            assertion = (statementOfDistribution.PTR285 != null && statementOfDistribution.PTR146 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.431050",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Either Australian Business Number or
Individual's Surname/family name must not be supplied together",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:TaxFileNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.431050" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR285", Value = GetValueOrEmpty(statementOfDistribution.PTR285) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR146", Value = statementOfDistribution.PTR146 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.431051
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR431051(PTR2025.RP_StatementOfDistribution statementOfDistribution, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.431051
            Australian Business Number OR Non-Individual's company name 
            AND Individual's Date of Birth must not be accepted together

            Legacy Rule Format:
            (^PTR285 <> NULL OR ^PTR149 <> NULL) AND (^PTR284 <> NULL)

            Technical Business Rule Format:
            (^PTR285 <> NULL OR ^PTR149 <> NULL) AND (^PTR284 <> NULL)

            Data Elements:

            ^PTR149 = PTR:RP:StatementOfDistribution:NonIndividualPartnerDetails:OrganisationNameDetails.OrganisationalName.Text

            ^PTR284 = PTR:RP:StatementOfDistribution:IndividualPartnerDetails:PersonDemographicDetails.Birth.Date

            ^PTR285 = PTR:RP:StatementOfDistribution:NonIndividualPartnerDetails:Identifiers.AustralianBusinessNumber.Identifier
            */
            assertion = ((statementOfDistribution.PTR285 != null || statementOfDistribution.PTR149 != null) && statementOfDistribution.PTR284 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.431051",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Australian Business Number OR Non-Individual's company name AND Individual's Date of Birth must not be supplied together",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:TaxFileNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.431051" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR285", Value = GetValueOrEmpty(statementOfDistribution.PTR285) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR149", Value = statementOfDistribution.PTR149 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR284", Value = GetValueOrEmptyDate(statementOfDistribution.PTR284) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.432001
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR432001()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.432001
            The sum of Share of net small business income in the distribution statement cannot be greater than Net small business income
    
            Legacy Rule Format:
            Sum(^PTR353) > ^PTR352

            Technical Business Rule Format:
            Sum(^PTR353) > ^PTR352
    
            Data Elements:
    
            ^PTR353 = PTR:RP:StatementOfDistribution:IncomeOrLoss:Income.SmallBusinessEntityNet.Amount
    
            ^PTR352 = PTR:RP:Income:BusinessIncomeAndExpenses:Income.SmallBusinessEntityNet.Amount
            */
            assertion = ((report.RP_StatementOfDistributionCollection == null ? 0 : report.RP_StatementOfDistributionCollection.Sum(f => f.PTR353.GetValueOrDefault())) > report.PTR352.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.432001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Net small business income is incorrect",
                    LongDescription = @"The sum of Share of net small business income in the distribution statement cannot be greater than Net small business income",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution/tns:IncomeOrLoss/tns:SmallBusinessEntityNetA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.432001" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Sum(PTR353)", Value = report.RP_StatementOfDistributionCollection.Sum(f => f.PTR353.GetValueOrDefault()).ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR352", Value = GetValueOrEmpty(report.PTR352) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.438036
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR438036(PTR2025.RP_StatementOfDistribution statementOfDistribution, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.438036
            Date of Birth must not be later than today

            Legacy Rule Format:
            ^PTR284 <> NULL AND ^PTR284 > Today()

            Technical Business Rule Format:
            ^PTR284 <> NULL AND ^PTR284 > Today()

            Data Elements:

            ^PTR284 = PTR:RP:StatementOfDistribution:IndividualPartnerDetails:PersonDemographicDetails.Birth.Date
            */
            assertion = (statementOfDistribution.PTR284 != null && statementOfDistribution.PTR284.GetValueOrDefault() > DateTime.Now.Date);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.438036",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Date of Birth must not be later than today",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:IndividualPartnerDetails/tns:PersonDemographicDetailsBirthD",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.438036" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR284", Value = GetValueOrEmptyDate(statementOfDistribution.PTR284) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.438080
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR438080()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.438080
            Year of return must be either the current or next (future) lodgment year associated with this service. If a future year lodgment is required, then it must be lodged no later than the 15th of June of the current year, unless it is an amendment. After the 15th of June, a return for that lodgment year must be submitted from the 1st of July, using the new service associated to that specific lodgment year.
    
            Legacy Rule Format:
            NotInSet(^PTR1, '"2025","2026"') OR (^PTR1 = 2026 AND Today() > AsDate('2026-06-15') AND ^PTR316 <> TRUE)

            Technical Business Rule Format:
            NotInSet(^PTR1, '"2025","2026"') OR (^PTR1 = 2026 AND Today() > AsDate('2026-06-15') AND ^PTR316 <> TRUE)
    
            Data Elements:
    
            ^PTR1 = PTR:RP:Report.TargetFinancial.Year
    
            ^PTR316 = PTR:RP:ReportAmendment:Report.Amendment.Indicator
            */
            assertion = (!(IsMatch(report.PTR1.GetValueOrDefault().ToString(), @"^(2025|2026)$", RegexOptions.IgnoreCase)) || report.PTR1.GetValueOrDefault() == 2026 && DateTime.Now.Date > AsDate(@"2026-06-15") && report.PTR316 != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500127",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Year of return is invalid",
                    LongDescription = @"Year of return must be either the current or next (future) lodgment year associated with this service. If a future year lodgment is required, then it must be lodged no later than the 15th of June of the current year, unless it is an amendment. After the 15th of June, a return for that lodgment year must be submitted from the 1st of July, using the new service associated to that specific lodgment year.",
                    Location = "/tns:PTR/tns:RP/tns:TargetFinancialY",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.438080" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR1", Value = GetValueOrEmpty(report.PTR1) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR316", Value = GetValueOrEmpty(report.PTR316) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.439030
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR439030()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.439030
            If the company has self-assessed that they are a Country by Country Reporting Entity (CbCRE), then the Significant Global Entity (SGE) indicator must be true.
    
            Legacy Rule Format:
            ^PTR372 = TRUE AND (^PTR355 = NULL OR ^PTR355 = FALSE)

            Technical Business Rule Format:
            ^PTR372 = TRUE AND (^PTR355 = NULL OR ^PTR355 = FALSE)
    
            Data Elements:
    
            ^PTR372 = PTR:RP:InternationalDealings.CountryByCountryReportingEntityStatus.Indicator
    
            ^PTR355 = PTR:RP:OrganisationDetails.SignificantGlobalEntityStatus.Indicator
            */
            assertion = (report.PTR372 == true && (report.PTR355 == null || report.PTR355 == false));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.439030",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"A reporting party who indicates that they are a Country-by-country reporting entity must also indicate that they are an SGE",
                    Location = "/tns:PTR/tns:RP/tns:InternationalDealingsCountryByCountryReportingEntityStatusI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.439030" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR372", Value = GetValueOrEmpty(report.PTR372) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR355", Value = GetValueOrEmpty(report.PTR355) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440028
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440028()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440028
            When P is selected at Aggregated turnover range, Aggregated turnover must be provided
    
            Legacy Rule Format:
            ^PTR412 = NULL AND ^PTR411 = 'P'

            Technical Business Rule Format:
            ^PTR412 = NULL AND ^PTR411 = 'P'
    
            Data Elements:
    
            ^PTR412 = PTR:RP:AggregatedTurnover:Income.AggregatedTurnover.Amount
    
            ^PTR411 = PTR:RP:AggregatedTurnover:Income.AggregatedTurnoverRange.Code
            */
            assertion = (report.PTR412 == null && report.PTR411 == @"P");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.440028",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Aggregated turnover must be provided",
                    LongDescription = @"When P is selected at aggregated turnover range, Aggregated turnover must be provided",
                    Location = "/tns:PTR/tns:RP/tns:AggregatedTurnover/tns:IncomeA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440028" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR412", Value = GetValueOrEmpty(report.PTR412) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR411", Value = GetValueOrEmpty(report.PTR411) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440100
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440100()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440100
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            ^PTR500 >= ^PTR501

            Technical Business Rule Format:
            ^PTR500 >= ^PTR501
    
            Data Elements:
    
            ^PTR500 = PTR:RP:Period.Start.Date
    
            ^PTR501 = PTR:RP:Period.End.Date
            */
            assertion = (report.PTR500.GetValueOrDefault() >= report.PTR501.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/tns:PTR/tns:RP/tns:PeriodStartD",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440100" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR500", Value = GetValueOrEmptyDate(report.PTR500) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR501", Value = GetValueOrEmptyDate(report.PTR501) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440101
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440101()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440101
            Tax File Number has failed the algorithm check.
    
            Legacy Rule Format:
            FailsTFNAlgorithm(^PTR5)

            Technical Business Rule Format:
            FailsTFNAlgorithm(^PTR5)
    
            Data Elements:
    
            ^PTR5 = PTR:RP:Identifiers.TaxFileNumber.Identifier
            */
            assertion = FailsTFNAlgorithm(report.PTR5);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.428016",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax File Number has failed the algorithm check",
                    Location = "/tns:PTR/tns:RP/tns:TaxFileNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440101" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR5", Value = report.PTR5 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440102
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440102()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440102
            ABN for reporting party must be provided when there is no intermediary
    
            Legacy Rule Format:
            ^PTR7 = NULL AND Count(^PTR1060) = 0

            Technical Business Rule Format:
            ^PTR7 = NULL AND Count(^PTR1060) = 0
    
            Data Elements:
    
            ^PTR7 = PTR:RP:Identifiers.AustralianBusinessNumber.Identifier
    
            ^PTR1060 = PTR:INT
            */
            assertion = (report.PTR7 == null && Count(report.IntCollectionCount) == 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.432417",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"ABN for reporting party must be provided when there is no intermediary",
                    Location = "/tns:PTR/tns:RP/tns:AustralianBusinessNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440102" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR7", Value = GetValueOrEmpty(report.PTR7) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440103
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440103()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440103
            ABN must pass the ABN algorithm
    
            Legacy Rule Format:
            ^PTR7 <> NULL AND FailsABNAlgorithm(^PTR7)

            Technical Business Rule Format:
            ^PTR7 <> NULL AND FailsABNAlgorithm(^PTR7)
    
            Data Elements:
    
            ^PTR7 = PTR:RP:Identifiers.AustralianBusinessNumber.Identifier
            */
            assertion = (report.PTR7 != null && FailsABNAlgorithm(report.PTR7));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000477",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"ABN is invalid",
                    Location = "/tns:PTR/tns:RP/tns:AustralianBusinessNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440103" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR7", Value = GetValueOrEmpty(report.PTR7) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440104
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440104()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440104
            If the Amendment Indicator is 'yes' (true), then the Amendment Sequence Number, Amendment Type and Amendment Reason must be present and not blank
    
            Legacy Rule Format:
            ^PTR316 = TRUE AND (^PTR319 = NULL OR ^PTR317 = NULL OR ^PTR318 = BLANK)

            Technical Business Rule Format:
            ^PTR316 = TRUE AND (^PTR319 = NULL OR ^PTR317 = NULL OR ^PTR318 = BLANK)
    
            Data Elements:
    
            ^PTR316 = PTR:RP:ReportAmendment:Report.Amendment.Indicator
    
            ^PTR317 = PTR:RP:ReportAmendment:Report.AmendmentType.Code
    
            ^PTR318 = PTR:RP:ReportAmendment:Report.AmendmentReason.Text
    
            ^PTR319 = PTR:RP:ReportAmendment:Report.AmendmentSequence.Number
            */
            assertion = (report.PTR316 == true && (report.PTR319 == null || report.PTR317 == null || string.IsNullOrWhiteSpace(report.PTR318) == true));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010028",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"If the return is an amendment, sequence number, type and reason must be present",
                    LongDescription = @"If the Amendment Indicator is 'yes' (true), then the Amendment Sequence Number, Amendment Type and Amendment Reason must be present and not blank.",
                    Location = "/tns:PTR/tns:RP/tns:ReportAmendment/tns:I",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440104" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR316", Value = GetValueOrEmpty(report.PTR316) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR319", Value = report.PTR319.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR317", Value = GetValueOrEmpty(report.PTR317) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR318", Value = GetValueOrEmpty(report.PTR318) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440105
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440105()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440105
            If an Amendment Sequence Number, Amendment Type or Amendment Reason is present, then the Amendment Indicator must be 'yes' (true)
    
            Legacy Rule Format:
            ^PTR316 <> TRUE AND (^PTR319 <> NULL OR ^PTR317 <> NULL OR ^PTR318 <> BLANK)

            Technical Business Rule Format:
            ^PTR316 <> TRUE AND (^PTR319 <> NULL OR ^PTR317 <> NULL OR ^PTR318 <> BLANK)
    
            Data Elements:
    
            ^PTR316 = PTR:RP:ReportAmendment:Report.Amendment.Indicator
    
            ^PTR317 = PTR:RP:ReportAmendment:Report.AmendmentType.Code
    
            ^PTR318 = PTR:RP:ReportAmendment:Report.AmendmentReason.Text
    
            ^PTR319 = PTR:RP:ReportAmendment:Report.AmendmentSequence.Number
            */
            assertion = (report.PTR316 != true && (report.PTR319 != null || report.PTR317 != null || string.IsNullOrWhiteSpace(report.PTR318) != true));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"If the return is an amendment, Amendment Indicator must be 'yes' (true)",
                    LongDescription = @"If an Amendment Sequence Number, Amendment Type or Amendment Reason is present, then the Amendment Indicator must be 'yes' (true)",
                    Location = "/tns:PTR/tns:RP/tns:ReportAmendment/tns:I",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440105" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR316", Value = GetValueOrEmpty(report.PTR316) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR319", Value = report.PTR319.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR317", Value = GetValueOrEmpty(report.PTR317) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR318", Value = GetValueOrEmpty(report.PTR318) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440106
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440106()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440106
            Non-individual name field cannot contain 'Pship', 'P'ship or 'P/Ship'
    
            Legacy Rule Format:
            StartsWith(^PTR6, 'T/A ') AND EndsWithSet(^PTR6, '" Pship"," P'ship"," P/ship"')

            Technical Business Rule Format:
            StartsWith(^PTR6, 'T/A ') AND EndsWithSet(^PTR6, '" Pship"," P'ship"," P/ship"')
    
            Data Elements:
    
            ^PTR6 = PTR:RP:CurrentPartnershipName:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (IsMatch(report.PTR6, @"^(T/A )", RegexOptions.IgnoreCase) && IsMatch(report.PTR6, @"( Pship| P'ship| P/ship)$", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410038",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-individual name field cannot start with 'T/A' and end with 'Pship', 'P'ship or 'P/Ship'",
                    Location = "/tns:PTR/tns:RP/tns:CurrentPartnershipNameOrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440106" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR6", Value = report.PTR6 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440107
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440107()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440107
            Organisation name must contain at least one alpha or numeric character
    
            Legacy Rule Format:
            ^PTR6 <> NULL AND NotContainsSet(^PTR6, '"a-z", "A-Z", "0-9"')

            Technical Business Rule Format:
            ^PTR6 <> NULL AND NotContainsSet(^PTR6, '"a-z", "A-Z", "0-9"')
    
            Data Elements:
    
            ^PTR6 = PTR:RP:CurrentPartnershipName:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.PTR6 != null && !(IsMatch(report.PTR6, @"[a-z]|[A-Z]|[0-9]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410039",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Organisation name must contain at least one alpha or numeric character",
                    Location = "/tns:PTR/tns:RP/tns:CurrentPartnershipNameOrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440107" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR6", Value = report.PTR6 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440108
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440108()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440108
            Non Individual Name cannot contain space hyphen space combinations
    
            Legacy Rule Format:
            Contains(^PTR6, ' - ')

            Technical Business Rule Format:
            Contains(^PTR6, ' - ')
    
            Data Elements:
    
            ^PTR6 = PTR:RP:CurrentPartnershipName:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.PTR6 == null ? false : report.PTR6.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000413",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain space hyphen space combinations",
                    Location = "/tns:PTR/tns:RP/tns:CurrentPartnershipNameOrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440108" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR6", Value = report.PTR6 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440109
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440109()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440109
            Non Individual Name cannot contain "P/L"
    
            Legacy Rule Format:
            Contains(^PTR6, 'P/L')

            Technical Business Rule Format:
            Contains(^PTR6, 'P/L')
    
            Data Elements:
    
            ^PTR6 = PTR:RP:CurrentPartnershipName:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.PTR6 == null ? false : report.PTR6.ToUpperInvariant().Contains(@"P/L"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000414",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain ""P/L""",
                    Location = "/tns:PTR/tns:RP/tns:CurrentPartnershipNameOrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440109" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR6", Value = report.PTR6 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440110
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440110()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440110
            Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            ContainsSet(^PTR6, '"--","''","  "')

            Technical Business Rule Format:
            ContainsSet(^PTR6, '"--","''","  "')
    
            Data Elements:
    
            ^PTR6 = PTR:RP:CurrentPartnershipName:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.PTR6, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000416",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:PTR/tns:RP/tns:CurrentPartnershipNameOrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440110" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR6", Value = report.PTR6 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440111
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440111()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440111
            Non Individual Name cannot end with "T/A", "T/A P'ship", "T/A Pship", "T/A P/Ship" or "T/A Partnership"
    
            Legacy Rule Format:
            EndsWithSet(^PTR6, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"')

            Technical Business Rule Format:
            EndsWithSet(^PTR6, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"')
    
            Data Elements:
    
            ^PTR6 = PTR:RP:CurrentPartnershipName:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.PTR6, @"( T/A| T/A P'ship| T/A Pship| T/A P/Ship| T/A Partnership)$", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000417",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot end with ""T/A"", ""T/A P'ship"", ""T/A Pship"", ""T/A P/Ship"" or ""T/A Partnership""",
                    Location = "/tns:PTR/tns:RP/tns:CurrentPartnershipNameOrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440111" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR6", Value = report.PTR6 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440112
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440112()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440112
            State and post code must be present for an Australian address
    
            Legacy Rule Format:
            ^PTR216 = FALSE AND (^PTR11 = NULL OR ^PTR12 = NULL)

            Technical Business Rule Format:
            ^PTR216 = FALSE AND (^PTR11 = NULL OR ^PTR12 = NULL)
    
            Data Elements:
    
            ^PTR216 = PTR:RP:CurrentPostalAddressDetails:AddressDetails.OverseasAddress.Indicator
    
            ^PTR11 = PTR:RP:CurrentPostalAddressDetails:AddressDetails.StateOrTerritory.Code
    
            ^PTR12 = PTR:RP:CurrentPostalAddressDetails:AddressDetails.Postcode.Text
            */
            assertion = (report.PTR216 == false && (report.PTR11 == null || report.PTR12 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.434147",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"State and post code must be present for an Australian address",
                    LongDescription = @"If the overseas indicator is no (false), then an Australian state code and post code must be present",
                    Location = "/tns:PTR/tns:RP/tns:CurrentPostalAddressDetails/tns:OverseasAddressI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440112" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR216", Value = GetValueOrEmpty(report.PTR216) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR11", Value = GetValueOrEmpty(report.PTR11) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR12", Value = GetValueOrEmpty(report.PTR12) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440113
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440113()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440113
            Address Line 2 must be present if Address Line 1 contains 'C/-'
    
            Legacy Rule Format:
            Contains(^PTR9, 'C/-') AND ^PTR211 = NULL

            Technical Business Rule Format:
            Contains(^PTR9, 'C/-') AND ^PTR211 = NULL
    
            Data Elements:
    
            ^PTR9 = PTR:RP:CurrentPostalAddressDetails:AddressDetails.Line1.Text
    
            ^PTR211 = PTR:RP:CurrentPostalAddressDetails:AddressDetails.Line2.Text
            */
            assertion = ((report.PTR9 == null ? false : report.PTR9.ToUpperInvariant().Contains(@"C/-")) && report.PTR211 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410002",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 2 must be present if Address Line 1 contains 'C/-'",
                    Location = "/tns:PTR/tns:RP/tns:CurrentPostalAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440113" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR9", Value = report.PTR9 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR211", Value = GetValueOrEmpty(report.PTR211) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440114
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440114()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440114
            Address line 1 contains incorrect 'care of' reference. The only acceptable 'care of' reference is C/-
    
            Legacy Rule Format:
            ^PTR9 <> NULL AND (StartsWithSet(^PTR9,'"C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^PTR9, '" C/O "," C/ "," Care Of "'))

            Technical Business Rule Format:
            ^PTR9 <> NULL AND (StartsWithSet(^PTR9,'"C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^PTR9, '" C/O "," C/ "," Care Of "'))
    
            Data Elements:
    
            ^PTR9 = PTR:RP:CurrentPostalAddressDetails:AddressDetails.Line1.Text
            */
            assertion = (report.PTR9 != null && (IsMatch(report.PTR9, @"^(C/O |C/ |Care Of |CO )", RegexOptions.IgnoreCase) || IsMatch(report.PTR9, @" C/O | C/ | Care Of ", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000406",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 contains incorrect 'care of' reference.  The only acceptable 'care of' reference is C/-",
                    Location = "/tns:PTR/tns:RP/tns:CurrentPostalAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440114" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR9", Value = report.PTR9 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440115
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440115()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440115
            Address Line 1 must not contain 'AS ABOVE'
    
            Legacy Rule Format:
            FoundSet(^PTR9, '"AS ABOVE"')

            Technical Business Rule Format:
            FoundSet(^PTR9, '"AS ABOVE"')
    
            Data Elements:
    
            ^PTR9 = PTR:RP:CurrentPostalAddressDetails:AddressDetails.Line1.Text
            */
            assertion = IsMatch(report.PTR9, @"(^(AS ABOVE)$)|(^(.*)( AS ABOVE )(.*)$)|(^(AS ABOVE )(.*)$)|(^(.*)( AS ABOVE)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 1 must not contain 'AS ABOVE'",
                    Location = "/tns:PTR/tns:RP/tns:CurrentPostalAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440115" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR9", Value = report.PTR9 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440116
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440116()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440116
            Address line 1 cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            Contains(^PTR9, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^PTR9, 'UNKNOWN')
    
            Data Elements:
    
            ^PTR9 = PTR:RP:CurrentPostalAddressDetails:AddressDetails.Line1.Text
            */
            assertion = (report.PTR9 == null ? false : report.PTR9.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000405",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 cannot contain ""UNKNOWN""",
                    Location = "/tns:PTR/tns:RP/tns:CurrentPostalAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440116" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR9", Value = report.PTR9 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440117
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440117()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440117
            'Care of' is not valid for address line 2
    
            Legacy Rule Format:
            ^PTR211 <> NULL AND (StartsWithSet(^PTR211,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^PTR211, '" C/- "," C/O "," C/ "," Care Of "'))

            Technical Business Rule Format:
            ^PTR211 <> NULL AND (StartsWithSet(^PTR211,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^PTR211, '" C/- "," C/O "," C/ "," Care Of "'))
    
            Data Elements:
    
            ^PTR211 = PTR:RP:CurrentPostalAddressDetails:AddressDetails.Line2.Text
            */
            assertion = (report.PTR211 != null && (IsMatch(report.PTR211, @"^(C/- |C/O |C/ |Care Of |CO )", RegexOptions.IgnoreCase) || IsMatch(report.PTR211, @" C/- | C/O | C/ | Care Of ", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000408",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"'Care of' is not valid for address line 2",
                    Location = "/tns:PTR/tns:RP/tns:CurrentPostalAddressDetails/tns:Line2T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440117" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR211", Value = GetValueOrEmpty(report.PTR211) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440118
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440118()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440118
            Address line 2 cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            Contains(^PTR211, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^PTR211, 'UNKNOWN')
    
            Data Elements:
    
            ^PTR211 = PTR:RP:CurrentPostalAddressDetails:AddressDetails.Line2.Text
            */
            assertion = (report.PTR211 == null ? false : report.PTR211.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000407",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 2 cannot contain ""UNKNOWN""",
                    Location = "/tns:PTR/tns:RP/tns:CurrentPostalAddressDetails/tns:Line2T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440118" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR211", Value = GetValueOrEmpty(report.PTR211) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440119
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440119()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440119
            Suburb/Town/City cannot contain state code
    
            Legacy Rule Format:
            FoundSet(^PTR10, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"')

            Technical Business Rule Format:
            FoundSet(^PTR10, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"')
    
            Data Elements:
    
            ^PTR10 = PTR:RP:CurrentPostalAddressDetails:AddressDetails.LocalityName.Text
            */
            assertion = IsMatch(report.PTR10, @"(^(QLD|NSW|VIC|SA|WA|NT|ACT|TAS)$)|(^(.*)( QLD | NSW | VIC | SA | WA | NT | ACT | TAS )(.*)$)|(^(QLD |NSW |VIC |SA |WA |NT |ACT |TAS )(.*)$)|(^(.*)( QLD| NSW| VIC| SA| WA| NT| ACT| TAS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000411",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain state code",
                    Location = "/tns:PTR/tns:RP/tns:CurrentPostalAddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440119" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR10", Value = report.PTR10 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440120
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440120()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440120
            Suburb/Town/City cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            Contains(^PTR10, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^PTR10, 'UNKNOWN')
    
            Data Elements:
    
            ^PTR10 = PTR:RP:CurrentPostalAddressDetails:AddressDetails.LocalityName.Text
            */
            assertion = (report.PTR10 == null ? false : report.PTR10.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000409",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain ""UNKNOWN""",
                    Location = "/tns:PTR/tns:RP/tns:CurrentPostalAddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440120" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR10", Value = report.PTR10 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440121
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440121()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440121
            Suburb/Town/City cannot contain numeric characters
    
            Legacy Rule Format:
            ^PTR216 = FALSE AND ContainsSet(^PTR10, '0-9')

            Technical Business Rule Format:
            ^PTR216 = FALSE AND ContainsSet(^PTR10, '0-9')
    
            Data Elements:
    
            ^PTR10 = PTR:RP:CurrentPostalAddressDetails:AddressDetails.LocalityName.Text
    
            ^PTR216 = PTR:RP:CurrentPostalAddressDetails:AddressDetails.OverseasAddress.Indicator
            */
            assertion = (report.PTR216 == false && IsMatch(report.PTR10, @"[0-9]", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000410",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain numeric characters",
                    Location = "/tns:PTR/tns:RP/tns:CurrentPostalAddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440121" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR216", Value = GetValueOrEmpty(report.PTR216) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR10", Value = report.PTR10 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440122
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440122()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440122
            Country code must be present for overseas address
    
            Legacy Rule Format:
            ^PTR302 = NULL AND ^PTR216 = TRUE

            Technical Business Rule Format:
            ^PTR302 = NULL AND ^PTR216 = TRUE
    
            Data Elements:
    
            ^PTR302 = PTR:RP:CurrentPostalAddressDetails:AddressDetails.Country.Code
    
            ^PTR216 = PTR:RP:CurrentPostalAddressDetails:AddressDetails.OverseasAddress.Indicator
            */
            assertion = (report.PTR302 == null && report.PTR216 == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410191",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Country code must be present for overseas address",
                    LongDescription = @"When the overseas address indicator is 'TRUE', the corresponding Country Code must be supplied",
                    Location = "/tns:PTR/tns:RP/tns:CurrentPostalAddressDetails/tns:CountryC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440122" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR302", Value = GetValueOrEmpty(report.PTR302) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR216", Value = GetValueOrEmpty(report.PTR216) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440123
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440123()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440123
            Overseas address indicator must not be 'TRUE' if Country Code is 'au'
    
            Legacy Rule Format:
            ^PTR302 = 'au' AND ^PTR216 = TRUE

            Technical Business Rule Format:
            ^PTR302 = 'au' AND ^PTR216 = TRUE
    
            Data Elements:
    
            ^PTR302 = PTR:RP:CurrentPostalAddressDetails:AddressDetails.Country.Code
    
            ^PTR216 = PTR:RP:CurrentPostalAddressDetails:AddressDetails.OverseasAddress.Indicator
            */
            assertion = (report.PTR302 == @"au" && report.PTR216 == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410211",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Overseas address indicator must not be 'TRUE' if Country Code is 'au'",
                    LongDescription = @"The overseas address indicator must be 'FALSE' if the Country Code is 'au' - representing Australia. If the address is an overseas address please supply the correct Country Code",
                    Location = "/tns:PTR/tns:RP/tns:CurrentPostalAddressDetails/tns:CountryC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440123" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR302", Value = GetValueOrEmpty(report.PTR302) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR216", Value = GetValueOrEmpty(report.PTR216) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440124
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440124()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440124
            If an overseas Country Code is used, the Overseas Address indicator must be 'TRUE'
    
            Legacy Rule Format:
            ^PTR302 <> NULL AND ^PTR302 <> 'au' AND ^PTR216 = FALSE

            Technical Business Rule Format:
            ^PTR302 <> NULL AND ^PTR302 <> 'au' AND ^PTR216 = FALSE
    
            Data Elements:
    
            ^PTR302 = PTR:RP:CurrentPostalAddressDetails:AddressDetails.Country.Code
    
            ^PTR216 = PTR:RP:CurrentPostalAddressDetails:AddressDetails.OverseasAddress.Indicator
            */
            assertion = (report.PTR302 != null && report.PTR302 != @"au" && report.PTR216 == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410212",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"If an overseas Country Code is used, the Overseas Address indicator must be 'TRUE'",
                    LongDescription = @"If the address is an Australian address, the overseas address indicator must be set to 'FALSE' and you do not need to supply a Country Code. If the address is an overseas address, the overseas address indicator must be set to 'TRUE' and you must supply a Country Code other than 'au'",
                    Location = "/tns:PTR/tns:RP/tns:CurrentPostalAddressDetails/tns:CountryC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440124" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR302", Value = GetValueOrEmpty(report.PTR302) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR216", Value = GetValueOrEmpty(report.PTR216) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440125
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440125()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440125
            Non-individual name field cannot contain 'Pship', 'P'ship or 'P/Ship'
    
            Legacy Rule Format:
            StartsWith(^PTR8, 'T/A ') AND EndsWithSet(^PTR8, '" Pship"," P'ship"," P/ship"')

            Technical Business Rule Format:
            StartsWith(^PTR8, 'T/A ') AND EndsWithSet(^PTR8, '" Pship"," P'ship"," P/ship"')
    
            Data Elements:
    
            ^PTR8 = PTR:RP:PreviousPartnershipName:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (IsMatch(report.PTR8, @"^(T/A )", RegexOptions.IgnoreCase) && IsMatch(report.PTR8, @"( Pship| P'ship| P/ship)$", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410038",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-individual name field cannot start with 'T/A' and end with 'Pship', 'P'ship or 'P/Ship'",
                    Location = "/tns:PTR/tns:RP/tns:PreviousPartnershipNameOrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440125" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR8", Value = report.PTR8 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440126
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440126()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440126
            Organisation name must contain at least one alpha or numeric character
    
            Legacy Rule Format:
            ^PTR8 <> NULL AND NotContainsSet(^PTR8, '"a-z", "A-Z", "0-9"')

            Technical Business Rule Format:
            ^PTR8 <> NULL AND NotContainsSet(^PTR8, '"a-z", "A-Z", "0-9"')
    
            Data Elements:
    
            ^PTR8 = PTR:RP:PreviousPartnershipName:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.PTR8 != null && !(IsMatch(report.PTR8, @"[a-z]|[A-Z]|[0-9]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410039",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Organisation name must contain at least one alpha or numeric character",
                    Location = "/tns:PTR/tns:RP/tns:PreviousPartnershipNameOrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440126" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR8", Value = report.PTR8 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440127
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440127()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440127
            Non Individual Name cannot contain space hyphen space combinations
    
            Legacy Rule Format:
            Contains(^PTR8, ' - ')

            Technical Business Rule Format:
            Contains(^PTR8, ' - ')
    
            Data Elements:
    
            ^PTR8 = PTR:RP:PreviousPartnershipName:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.PTR8 == null ? false : report.PTR8.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000413",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain space hyphen space combinations",
                    Location = "/tns:PTR/tns:RP/tns:PreviousPartnershipNameOrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440127" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR8", Value = report.PTR8 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440128
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440128()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440128
            Non Individual Name cannot contain "P/L"
    
            Legacy Rule Format:
            Contains(^PTR8, 'P/L')

            Technical Business Rule Format:
            Contains(^PTR8, 'P/L')
    
            Data Elements:
    
            ^PTR8 = PTR:RP:PreviousPartnershipName:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.PTR8 == null ? false : report.PTR8.ToUpperInvariant().Contains(@"P/L"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000414",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain ""P/L""",
                    Location = "/tns:PTR/tns:RP/tns:PreviousPartnershipNameOrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440128" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR8", Value = report.PTR8 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440129
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440129()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440129
            Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            ContainsSet(^PTR8, '"--","''","  "')

            Technical Business Rule Format:
            ContainsSet(^PTR8, '"--","''","  "')
    
            Data Elements:
    
            ^PTR8 = PTR:RP:PreviousPartnershipName:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.PTR8, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000416",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:PTR/tns:RP/tns:PreviousPartnershipNameOrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440129" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR8", Value = report.PTR8 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440130
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440130()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440130
            Non Individual Name cannot end with "T/A", "T/A P'ship", "T/A Pship", "T/A P/Ship" or "T/A Partnership"
    
            Legacy Rule Format:
            EndsWithSet(^PTR8, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"')

            Technical Business Rule Format:
            EndsWithSet(^PTR8, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"')
    
            Data Elements:
    
            ^PTR8 = PTR:RP:PreviousPartnershipName:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.PTR8, @"( T/A| T/A P'ship| T/A Pship| T/A P/Ship| T/A Partnership)$", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000417",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot end with ""T/A"", ""T/A P'ship"", ""T/A Pship"", ""T/A P/Ship"" or ""T/A Partnership""",
                    Location = "/tns:PTR/tns:RP/tns:PreviousPartnershipNameOrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440130" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR8", Value = report.PTR8 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440131
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440131()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440131
            State and post code must be present for an Australian address
    
            Legacy Rule Format:
            ^PTR219 = FALSE AND (^PTR16 = NULL OR ^PTR17 = NULL)

            Technical Business Rule Format:
            ^PTR219 = FALSE AND (^PTR16 = NULL OR ^PTR17 = NULL)
    
            Data Elements:
    
            ^PTR219 = PTR:RP:PreviousPostalAddressDetails:AddressDetails.OverseasAddress.Indicator
    
            ^PTR16 = PTR:RP:PreviousPostalAddressDetails:AddressDetails.StateOrTerritory.Code
    
            ^PTR17 = PTR:RP:PreviousPostalAddressDetails:AddressDetails.Postcode.Text
            */
            assertion = (report.PTR219 == false && (report.PTR16 == null || report.PTR17 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.434147",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"State and post code must be present for an Australian address",
                    LongDescription = @"If the overseas indicator is no (false), then an Australian state code and post code must be present",
                    Location = "/tns:PTR/tns:RP/tns:PreviousPostalAddressDetails/tns:OverseasAddressI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440131" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR219", Value = GetValueOrEmpty(report.PTR219) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR16", Value = GetValueOrEmpty(report.PTR16) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR17", Value = GetValueOrEmpty(report.PTR17) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440132
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440132()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440132
            Address Line 2 must be present if Address Line 1 contains 'C/-'
    
            Legacy Rule Format:
            Contains(^PTR14, 'C/-') AND ^PTR212 = NULL

            Technical Business Rule Format:
            Contains(^PTR14, 'C/-') AND ^PTR212 = NULL
    
            Data Elements:
    
            ^PTR14 = PTR:RP:PreviousPostalAddressDetails:AddressDetails.Line1.Text
    
            ^PTR212 = PTR:RP:PreviousPostalAddressDetails:AddressDetails.Line2.Text
            */
            assertion = ((report.PTR14 == null ? false : report.PTR14.ToUpperInvariant().Contains(@"C/-")) && report.PTR212 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410002",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 2 must be present if Address Line 1 contains 'C/-'",
                    Location = "/tns:PTR/tns:RP/tns:PreviousPostalAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440132" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR14", Value = report.PTR14 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR212", Value = GetValueOrEmpty(report.PTR212) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440133
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440133()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440133
            Address line 1 contains incorrect 'care of' reference. The only acceptable 'care of' reference is C/-
    
            Legacy Rule Format:
            ^PTR14 <> NULL AND (StartsWithSet(^PTR14,'"C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^PTR14, '" C/O "," C/ "," Care Of "'))

            Technical Business Rule Format:
            ^PTR14 <> NULL AND (StartsWithSet(^PTR14,'"C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^PTR14, '" C/O "," C/ "," Care Of "'))
    
            Data Elements:
    
            ^PTR14 = PTR:RP:PreviousPostalAddressDetails:AddressDetails.Line1.Text
            */
            assertion = (report.PTR14 != null && (IsMatch(report.PTR14, @"^(C/O |C/ |Care Of |CO )", RegexOptions.IgnoreCase) || IsMatch(report.PTR14, @" C/O | C/ | Care Of ", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000406",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 contains incorrect 'care of' reference.  The only acceptable 'care of' reference is C/-",
                    Location = "/tns:PTR/tns:RP/tns:PreviousPostalAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440133" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR14", Value = report.PTR14 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440134
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440134()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440134
            Address Line 1 must not contain 'AS ABOVE'
    
            Legacy Rule Format:
            FoundSet(^PTR14, '"AS ABOVE"')

            Technical Business Rule Format:
            FoundSet(^PTR14, '"AS ABOVE"')
    
            Data Elements:
    
            ^PTR14 = PTR:RP:PreviousPostalAddressDetails:AddressDetails.Line1.Text
            */
            assertion = IsMatch(report.PTR14, @"(^(AS ABOVE)$)|(^(.*)( AS ABOVE )(.*)$)|(^(AS ABOVE )(.*)$)|(^(.*)( AS ABOVE)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 1 must not contain 'AS ABOVE'",
                    Location = "/tns:PTR/tns:RP/tns:PreviousPostalAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440134" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR14", Value = report.PTR14 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440135
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440135()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440135
            Address line 1 cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            Contains(^PTR14, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^PTR14, 'UNKNOWN')
    
            Data Elements:
    
            ^PTR14 = PTR:RP:PreviousPostalAddressDetails:AddressDetails.Line1.Text
            */
            assertion = (report.PTR14 == null ? false : report.PTR14.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000405",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 cannot contain ""UNKNOWN""",
                    Location = "/tns:PTR/tns:RP/tns:PreviousPostalAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440135" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR14", Value = report.PTR14 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440136
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440136()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440136
            'Care of' is not valid for address line 2
    
            Legacy Rule Format:
            ^PTR212 <> NULL AND (StartsWithSet(^PTR212,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^PTR212, '" C/- "," C/O "," C/ "," Care Of "'))

            Technical Business Rule Format:
            ^PTR212 <> NULL AND (StartsWithSet(^PTR212,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^PTR212, '" C/- "," C/O "," C/ "," Care Of "'))
    
            Data Elements:
    
            ^PTR212 = PTR:RP:PreviousPostalAddressDetails:AddressDetails.Line2.Text
            */
            assertion = (report.PTR212 != null && (IsMatch(report.PTR212, @"^(C/- |C/O |C/ |Care Of |CO )", RegexOptions.IgnoreCase) || IsMatch(report.PTR212, @" C/- | C/O | C/ | Care Of ", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000408",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"'Care of' is not valid for address line 2",
                    Location = "/tns:PTR/tns:RP/tns:PreviousPostalAddressDetails/tns:Line2T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440136" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR212", Value = GetValueOrEmpty(report.PTR212) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440137
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440137()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440137
            Address line 2 cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            Contains(^PTR212, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^PTR212, 'UNKNOWN')
    
            Data Elements:
    
            ^PTR212 = PTR:RP:PreviousPostalAddressDetails:AddressDetails.Line2.Text
            */
            assertion = (report.PTR212 == null ? false : report.PTR212.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000407",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 2 cannot contain ""UNKNOWN""",
                    Location = "/tns:PTR/tns:RP/tns:PreviousPostalAddressDetails/tns:Line2T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440137" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR212", Value = GetValueOrEmpty(report.PTR212) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440138
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440138()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440138
            Suburb/Town/City cannot contain state code
    
            Legacy Rule Format:
            FoundSet(^PTR15, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"')

            Technical Business Rule Format:
            FoundSet(^PTR15, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"')
    
            Data Elements:
    
            ^PTR15 = PTR:RP:PreviousPostalAddressDetails:AddressDetails.LocalityName.Text
            */
            assertion = IsMatch(report.PTR15, @"(^(QLD|NSW|VIC|SA|WA|NT|ACT|TAS)$)|(^(.*)( QLD | NSW | VIC | SA | WA | NT | ACT | TAS )(.*)$)|(^(QLD |NSW |VIC |SA |WA |NT |ACT |TAS )(.*)$)|(^(.*)( QLD| NSW| VIC| SA| WA| NT| ACT| TAS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000411",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain state code",
                    Location = "/tns:PTR/tns:RP/tns:PreviousPostalAddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440138" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR15", Value = report.PTR15 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440139
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440139()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440139
            Suburb/Town/City cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            Contains(^PTR15, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^PTR15, 'UNKNOWN')
    
            Data Elements:
    
            ^PTR15 = PTR:RP:PreviousPostalAddressDetails:AddressDetails.LocalityName.Text
            */
            assertion = (report.PTR15 == null ? false : report.PTR15.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000409",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain ""UNKNOWN""",
                    Location = "/tns:PTR/tns:RP/tns:PreviousPostalAddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440139" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR15", Value = report.PTR15 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440140
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440140()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440140
            Suburb/Town/City cannot contain numeric characters
    
            Legacy Rule Format:
            ^PTR219 = FALSE AND ContainsSet(^PTR15, '0-9')

            Technical Business Rule Format:
            ^PTR219 = FALSE AND ContainsSet(^PTR15, '0-9')
    
            Data Elements:
    
            ^PTR15 = PTR:RP:PreviousPostalAddressDetails:AddressDetails.LocalityName.Text
    
            ^PTR219 = PTR:RP:PreviousPostalAddressDetails:AddressDetails.OverseasAddress.Indicator
            */
            assertion = (report.PTR219 == false && IsMatch(report.PTR15, @"[0-9]", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000410",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain numeric characters",
                    Location = "/tns:PTR/tns:RP/tns:PreviousPostalAddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440140" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR219", Value = GetValueOrEmpty(report.PTR219) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR15", Value = report.PTR15 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440141
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440141()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440141
            Country code must be present for overseas address
    
            Legacy Rule Format:
            ^PTR303 = NULL AND ^PTR219 = TRUE

            Technical Business Rule Format:
            ^PTR303 = NULL AND ^PTR219 = TRUE
    
            Data Elements:
    
            ^PTR303 = PTR:RP:PreviousPostalAddressDetails:AddressDetails.Country.Code
    
            ^PTR219 = PTR:RP:PreviousPostalAddressDetails:AddressDetails.OverseasAddress.Indicator
            */
            assertion = (report.PTR303 == null && report.PTR219 == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410191",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Country code must be present for overseas address",
                    LongDescription = @"When the overseas address indicator is 'TRUE', the corresponding Country Code must be supplied",
                    Location = "/tns:PTR/tns:RP/tns:PreviousPostalAddressDetails/tns:CountryC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440141" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR303", Value = GetValueOrEmpty(report.PTR303) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR219", Value = GetValueOrEmpty(report.PTR219) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440142
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440142()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440142
            Overseas address indicator must not be 'TRUE' if Country Code is 'au'
    
            Legacy Rule Format:
            ^PTR303 = 'au' AND ^PTR219 = TRUE

            Technical Business Rule Format:
            ^PTR303 = 'au' AND ^PTR219 = TRUE
    
            Data Elements:
    
            ^PTR303 = PTR:RP:PreviousPostalAddressDetails:AddressDetails.Country.Code
    
            ^PTR219 = PTR:RP:PreviousPostalAddressDetails:AddressDetails.OverseasAddress.Indicator
            */
            assertion = (report.PTR303 == @"au" && report.PTR219 == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410211",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Overseas address indicator must not be 'TRUE' if Country Code is 'au'",
                    LongDescription = @"The overseas address indicator must be 'FALSE' if the Country Code is 'au' - representing Australia. If the address is an overseas address please supply the correct Country Code",
                    Location = "/tns:PTR/tns:RP/tns:PreviousPostalAddressDetails/tns:CountryC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440142" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR303", Value = GetValueOrEmpty(report.PTR303) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR219", Value = GetValueOrEmpty(report.PTR219) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440143
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440143()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440143
            If an overseas Country Code is used, the Overseas Address indicator must be 'TRUE'
    
            Legacy Rule Format:
            ^PTR303 <> NULL AND ^PTR303 <> 'au' AND ^PTR219 = FALSE

            Technical Business Rule Format:
            ^PTR303 <> NULL AND ^PTR303 <> 'au' AND ^PTR219 = FALSE
    
            Data Elements:
    
            ^PTR303 = PTR:RP:PreviousPostalAddressDetails:AddressDetails.Country.Code
    
            ^PTR219 = PTR:RP:PreviousPostalAddressDetails:AddressDetails.OverseasAddress.Indicator
            */
            assertion = (report.PTR303 != null && report.PTR303 != @"au" && report.PTR219 == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410212",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"If an overseas Country Code is used, the Overseas Address indicator must be 'TRUE'",
                    LongDescription = @"If the address is an Australian address, the overseas address indicator must be set to 'FALSE' and you do not need to supply a Country Code. If the address is an overseas address, the overseas address indicator must be set to 'TRUE' and you must supply a Country Code other than 'au'",
                    Location = "/tns:PTR/tns:RP/tns:PreviousPostalAddressDetails/tns:CountryC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440143" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR303", Value = GetValueOrEmpty(report.PTR303) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR219", Value = GetValueOrEmpty(report.PTR219) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440144
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440144()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440144
            Family name cannot contain "Exec for", "Rep for" or "Trustee for"
    
            Legacy Rule Format:
            FoundSet(^PTR20, '"Exec for","Rep for","Trustee for"')

            Technical Business Rule Format:
            FoundSet(^PTR20, '"Exec for","Rep for","Trustee for"')
    
            Data Elements:
    
            ^PTR20 = PTR:RP:MailRecipient:PersonNameDetails:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.PTR20, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000424",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/tns:PTR/tns:RP/tns:MailRecipient/tns:PersonNameDetails/tns:FamilyNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440144" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR20", Value = report.PTR20 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440145
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440145()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440145
            Family name cannot contain "MR", "MRS", "MISS" or "MS"
    
            Legacy Rule Format:
            FoundSet(^PTR20, '"MR","MRS","MISS","MS"')

            Technical Business Rule Format:
            FoundSet(^PTR20, '"MR","MRS","MISS","MS"')
    
            Data Elements:
    
            ^PTR20 = PTR:RP:MailRecipient:PersonNameDetails:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.PTR20, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000426",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/tns:PTR/tns:RP/tns:MailRecipient/tns:PersonNameDetails/tns:FamilyNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440145" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR20", Value = report.PTR20 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440146
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440146()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440146
            Family Name cannot contain "space hyphen space"
    
            Legacy Rule Format:
            Contains(^PTR20, ' - ')

            Technical Business Rule Format:
            Contains(^PTR20, ' - ')
    
            Data Elements:
    
            ^PTR20 = PTR:RP:MailRecipient:PersonNameDetails:PersonNameDetails.FamilyName.Text
            */
            assertion = (report.PTR20 == null ? false : report.PTR20.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000423",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family Name cannot contain ""space hyphen space""",
                    Location = "/tns:PTR/tns:RP/tns:MailRecipient/tns:PersonNameDetails/tns:FamilyNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440146" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR20", Value = report.PTR20 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440147
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440147()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440147
            Family name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            ContainsSet(^PTR20, '"--","''","  "')

            Technical Business Rule Format:
            ContainsSet(^PTR20, '"--","''","  "')
    
            Data Elements:
    
            ^PTR20 = PTR:RP:MailRecipient:PersonNameDetails:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.PTR20, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000427",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:PTR/tns:RP/tns:MailRecipient/tns:PersonNameDetails/tns:FamilyNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440147" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR20", Value = report.PTR20 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440148
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440148()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440148
            Family name must contain at least one alphabetical character
    
            Legacy Rule Format:
            ^PTR20 <> NULL AND NotContainsSet(^PTR20, '"A-Z","a-z"')

            Technical Business Rule Format:
            ^PTR20 <> NULL AND NotContainsSet(^PTR20, '"A-Z","a-z"')
    
            Data Elements:
    
            ^PTR20 = PTR:RP:MailRecipient:PersonNameDetails:PersonNameDetails.FamilyName.Text
            */
            assertion = (report.PTR20 != null && !(IsMatch(report.PTR20, @"[A-Z]|[a-z]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500142",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name must contain at least one alphabetical character.",
                    Location = "/tns:PTR/tns:RP/tns:MailRecipient/tns:PersonNameDetails/tns:FamilyNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440148" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR20", Value = report.PTR20 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440149
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440149()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440149
            Family name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
            Legacy Rule Format:
            FoundSet(^PTR20, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')

            Technical Business Rule Format:
            FoundSet(^PTR20, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')
    
            Data Elements:
    
            ^PTR20 = PTR:RP:MailRecipient:PersonNameDetails:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.PTR20, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500153",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/tns:PTR/tns:RP/tns:MailRecipient/tns:PersonNameDetails/tns:FamilyNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440149" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR20", Value = report.PTR20 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440150
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440150()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440150
            FirstName must be entered if OtherGivenNames is entered
    
            Legacy Rule Format:
            ^PTR22 = BLANK AND ^PTR23 <> BLANK

            Technical Business Rule Format:
            ^PTR22 = BLANK AND ^PTR23 <> BLANK
    
            Data Elements:
    
            ^PTR22 = PTR:RP:MailRecipient:PersonNameDetails:PersonNameDetails.GivenName.Text
    
            ^PTR23 = PTR:RP:MailRecipient:PersonNameDetails:PersonNameDetails.OtherGivenName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.PTR22) == true && string.IsNullOrWhiteSpace(report.PTR23) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410218",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"FirstName must be entered if OtherGivenNames is entered",
                    Location = "/tns:PTR/tns:RP/tns:MailRecipient/tns:PersonNameDetails/tns:GivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440150" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR22", Value = GetValueOrEmpty(report.PTR22) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR23", Value = GetValueOrEmpty(report.PTR23) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440151
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440151()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440151
            Given Name must contain at least one alphabetical character
    
            Legacy Rule Format:
            ^PTR22 <> BLANK AND NotContainsSet(^PTR22, '"A-Z","a-z"')

            Technical Business Rule Format:
            ^PTR22 <> BLANK AND NotContainsSet(^PTR22, '"A-Z","a-z"')
    
            Data Elements:
    
            ^PTR22 = PTR:RP:MailRecipient:PersonNameDetails:PersonNameDetails.GivenName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.PTR22) != true && !(IsMatch(report.PTR22, @"[A-Z]|[a-z]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410063",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Given Name must contain at least one alphabetical character",
                    Location = "/tns:PTR/tns:RP/tns:MailRecipient/tns:PersonNameDetails/tns:GivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440151" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR22", Value = GetValueOrEmpty(report.PTR22) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440152
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440152()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440152
            First name cannot contain "Exec for", "Rep for" or "Trustee for"
    
            Legacy Rule Format:
            FoundSet(^PTR22, '"Exec for","Rep for","Trustee for"')

            Technical Business Rule Format:
            FoundSet(^PTR22, '"Exec for","Rep for","Trustee for"')
    
            Data Elements:
    
            ^PTR22 = PTR:RP:MailRecipient:PersonNameDetails:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.PTR22, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000437",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/tns:PTR/tns:RP/tns:MailRecipient/tns:PersonNameDetails/tns:GivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440152" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR22", Value = GetValueOrEmpty(report.PTR22) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440153
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440153()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440153
            First name cannot contain "MR", "MRS", "MISS" or "MS"
    
            Legacy Rule Format:
            FoundSet(^PTR22, '"MR","MRS","MISS","MS"')

            Technical Business Rule Format:
            FoundSet(^PTR22, '"MR","MRS","MISS","MS"')
    
            Data Elements:
    
            ^PTR22 = PTR:RP:MailRecipient:PersonNameDetails:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.PTR22, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000438",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/tns:PTR/tns:RP/tns:MailRecipient/tns:PersonNameDetails/tns:GivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440153" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR22", Value = GetValueOrEmpty(report.PTR22) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440154
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440154()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440154
            First name cannot contain space hyphen space
    
            Legacy Rule Format:
            Contains(^PTR22, ' - ')

            Technical Business Rule Format:
            Contains(^PTR22, ' - ')
    
            Data Elements:
    
            ^PTR22 = PTR:RP:MailRecipient:PersonNameDetails:PersonNameDetails.GivenName.Text
            */
            assertion = (report.PTR22 == null ? false : report.PTR22.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000434",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain space hyphen space",
                    Location = "/tns:PTR/tns:RP/tns:MailRecipient/tns:PersonNameDetails/tns:GivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440154" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR22", Value = GetValueOrEmpty(report.PTR22) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440155
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440155()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440155
            First name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            ContainsSet(^PTR22, '"--","''","  "')

            Technical Business Rule Format:
            ContainsSet(^PTR22, '"--","''","  "')
    
            Data Elements:
    
            ^PTR22 = PTR:RP:MailRecipient:PersonNameDetails:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.PTR22, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000439",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:PTR/tns:RP/tns:MailRecipient/tns:PersonNameDetails/tns:GivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440155" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR22", Value = GetValueOrEmpty(report.PTR22) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440156
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440156()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440156
            First name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
            Legacy Rule Format:
            FoundSet(^PTR22, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')

            Technical Business Rule Format:
            FoundSet(^PTR22, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')
    
            Data Elements:
    
            ^PTR22 = PTR:RP:MailRecipient:PersonNameDetails:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.PTR22, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010436",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/tns:PTR/tns:RP/tns:MailRecipient/tns:PersonNameDetails/tns:GivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440156" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR22", Value = GetValueOrEmpty(report.PTR22) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440157
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440157()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440157
            Other Given Name must contain at least one alphabetical character
    
            Legacy Rule Format:
            ^PTR23 <> BLANK AND NotContainsSet(^PTR23, '"A-Z","a-z"')

            Technical Business Rule Format:
            ^PTR23 <> BLANK AND NotContainsSet(^PTR23, '"A-Z","a-z"')
    
            Data Elements:
    
            ^PTR23 = PTR:RP:MailRecipient:PersonNameDetails:PersonNameDetails.OtherGivenName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.PTR23) != true && !(IsMatch(report.PTR23, @"[A-Z]|[a-z]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410131",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other Given Name must contain at least one alphabetical character",
                    Location = "/tns:PTR/tns:RP/tns:MailRecipient/tns:PersonNameDetails/tns:OtherGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440157" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR23", Value = GetValueOrEmpty(report.PTR23) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440158
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440158()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440158
            Other given name cannot contain "Exec for", "Rep for" or "Trustee for"
    
            Legacy Rule Format:
            FoundSet(^PTR23, '"Exec for","Rep for","Trustee for"')

            Technical Business Rule Format:
            FoundSet(^PTR23, '"Exec for","Rep for","Trustee for"')
    
            Data Elements:
    
            ^PTR23 = PTR:RP:MailRecipient:PersonNameDetails:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.PTR23, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000449",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/tns:PTR/tns:RP/tns:MailRecipient/tns:PersonNameDetails/tns:OtherGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440158" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR23", Value = GetValueOrEmpty(report.PTR23) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440159
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440159()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440159
            Other given name cannot contain "MR", "MRS", "MISS" or "MS"
    
            Legacy Rule Format:
            FoundSet(^PTR23, '"MR","MRS","MISS","MS"')

            Technical Business Rule Format:
            FoundSet(^PTR23, '"MR","MRS","MISS","MS"')
    
            Data Elements:
    
            ^PTR23 = PTR:RP:MailRecipient:PersonNameDetails:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.PTR23, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000450",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/tns:PTR/tns:RP/tns:MailRecipient/tns:PersonNameDetails/tns:OtherGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440159" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR23", Value = GetValueOrEmpty(report.PTR23) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440160
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440160()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440160
            Other given name cannot contain space hyphen space
    
            Legacy Rule Format:
            Contains(^PTR23, ' - ')

            Technical Business Rule Format:
            Contains(^PTR23, ' - ')
    
            Data Elements:
    
            ^PTR23 = PTR:RP:MailRecipient:PersonNameDetails:PersonNameDetails.OtherGivenName.Text
            */
            assertion = (report.PTR23 == null ? false : report.PTR23.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000446",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain space hyphen space",
                    Location = "/tns:PTR/tns:RP/tns:MailRecipient/tns:PersonNameDetails/tns:OtherGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440160" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR23", Value = GetValueOrEmpty(report.PTR23) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440161
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440161()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440161
            Other given name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            ContainsSet(^PTR23, '"--","''","  "')

            Technical Business Rule Format:
            ContainsSet(^PTR23, '"--","''","  "')
    
            Data Elements:
    
            ^PTR23 = PTR:RP:MailRecipient:PersonNameDetails:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.PTR23, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000451",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:PTR/tns:RP/tns:MailRecipient/tns:PersonNameDetails/tns:OtherGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440161" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR23", Value = GetValueOrEmpty(report.PTR23) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440162
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440162()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440162
            Other given name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
            Legacy Rule Format:
            FoundSet(^PTR23, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')

            Technical Business Rule Format:
            FoundSet(^PTR23, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')
    
            Data Elements:
    
            ^PTR23 = PTR:RP:MailRecipient:PersonNameDetails:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.PTR23, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010448",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/tns:PTR/tns:RP/tns:MailRecipient/tns:PersonNameDetails/tns:OtherGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440162" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR23", Value = GetValueOrEmpty(report.PTR23) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440163
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440163()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440163
            Non-individual name field cannot contain 'Pship', 'P'ship or 'P/Ship'
    
            Legacy Rule Format:
            StartsWith(^PTR24, 'T/A ') AND EndsWithSet(^PTR24, '" Pship"," P'ship"," P/ship"')

            Technical Business Rule Format:
            StartsWith(^PTR24, 'T/A ') AND EndsWithSet(^PTR24, '" Pship"," P'ship"," P/ship"')
    
            Data Elements:
    
            ^PTR24 = PTR:RP:MailRecipient:OrganisationDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (IsMatch(report.PTR24, @"^(T/A )", RegexOptions.IgnoreCase) && IsMatch(report.PTR24, @"( Pship| P'ship| P/ship)$", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410038",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-individual name field cannot start with 'T/A' and end with 'Pship', 'P'ship or 'P/Ship'",
                    Location = "/tns:PTR/tns:RP/tns:MailRecipient/tns:OrganisationDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440163" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR24", Value = report.PTR24 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440164
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440164()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440164
            Organisation name must contain at least one alpha or numeric character
    
            Legacy Rule Format:
            ^PTR24 <> NULL AND NotContainsSet(^PTR24, '"a-z", "A-Z", "0-9"')

            Technical Business Rule Format:
            ^PTR24 <> NULL AND NotContainsSet(^PTR24, '"a-z", "A-Z", "0-9"')
    
            Data Elements:
    
            ^PTR24 = PTR:RP:MailRecipient:OrganisationDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.PTR24 != null && !(IsMatch(report.PTR24, @"[a-z]|[A-Z]|[0-9]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410039",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Organisation name must contain at least one alpha or numeric character",
                    Location = "/tns:PTR/tns:RP/tns:MailRecipient/tns:OrganisationDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440164" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR24", Value = report.PTR24 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440165
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440165()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440165
            Non Individual Name cannot contain space hyphen space combinations
    
            Legacy Rule Format:
            Contains(^PTR24, ' - ')

            Technical Business Rule Format:
            Contains(^PTR24, ' - ')
    
            Data Elements:
    
            ^PTR24 = PTR:RP:MailRecipient:OrganisationDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.PTR24 == null ? false : report.PTR24.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000413",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain space hyphen space combinations",
                    Location = "/tns:PTR/tns:RP/tns:MailRecipient/tns:OrganisationDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440165" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR24", Value = report.PTR24 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440166
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440166()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440166
            Non Individual Name cannot contain "P/L"
    
            Legacy Rule Format:
            Contains(^PTR24, 'P/L')

            Technical Business Rule Format:
            Contains(^PTR24, 'P/L')
    
            Data Elements:
    
            ^PTR24 = PTR:RP:MailRecipient:OrganisationDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.PTR24 == null ? false : report.PTR24.ToUpperInvariant().Contains(@"P/L"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000414",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain ""P/L""",
                    Location = "/tns:PTR/tns:RP/tns:MailRecipient/tns:OrganisationDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440166" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR24", Value = report.PTR24 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440167
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440167()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440167
            Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            ContainsSet(^PTR24, '"--","''","  "')

            Technical Business Rule Format:
            ContainsSet(^PTR24, '"--","''","  "')
    
            Data Elements:
    
            ^PTR24 = PTR:RP:MailRecipient:OrganisationDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.PTR24, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000416",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:PTR/tns:RP/tns:MailRecipient/tns:OrganisationDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440167" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR24", Value = report.PTR24 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440168
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440168()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440168
            Non Individual Name cannot end with "T/A", "T/A P'ship", "T/A Pship", "T/A P/Ship" or "T/A Partnership"
    
            Legacy Rule Format:
            EndsWithSet(^PTR24, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"')

            Technical Business Rule Format:
            EndsWithSet(^PTR24, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"')
    
            Data Elements:
    
            ^PTR24 = PTR:RP:MailRecipient:OrganisationDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.PTR24, @"( T/A| T/A P'ship| T/A Pship| T/A P/Ship| T/A Partnership)$", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000417",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot end with ""T/A"", ""T/A P'ship"", ""T/A Pship"", ""T/A P/Ship"" or ""T/A Partnership""",
                    Location = "/tns:PTR/tns:RP/tns:MailRecipient/tns:OrganisationDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440168" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR24", Value = report.PTR24 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440169
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440169()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440169
            ABN must pass the ABN algorithm
    
            Legacy Rule Format:
            ^PTR25 <> NULL AND FailsABNAlgorithm(^PTR25)

            Technical Business Rule Format:
            ^PTR25 <> NULL AND FailsABNAlgorithm(^PTR25)
    
            Data Elements:
    
            ^PTR25 = PTR:RP:MailRecipient:OrganisationDetails:Identifiers.AustralianBusinessNumber.Identifier
            */
            assertion = (report.PTR25 != null && FailsABNAlgorithm(report.PTR25));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000477",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"ABN is invalid",
                    Location = "/tns:PTR/tns:RP/tns:MailRecipient/tns:OrganisationDetails/tns:AustralianBusinessNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440169" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR25", Value = GetValueOrEmpty(report.PTR25) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440170
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440170()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440170
            Business Industry code is not a valid code
    
            Legacy Rule Format:
            ^PTR31 <> NULL AND NotInDomain(^PTR31, 'BUSINESS INDUSTRY CODES')

            Technical Business Rule Format:
            ^PTR31 <> NULL AND NotInDomain(^PTR31, 'BUSINESS INDUSTRY CODES')
    
            Data Elements:
    
            ^PTR31 = PTR:RP:OrganisationDetails.OrganisationIndustry2006Extended.Code
            */
            assertion = (report.PTR31 != null && !(IsMatch(report.PTR31, @"^(01110|01120|01130|01140|01150|01210|01220|01230|01310|01320|01330|01340|01350|01360|01370|01390|01410|01420|01430|01440|01450|01460|01490|01510|01520|01590|01600|01710|01720|01800|01910|01920|01930|01990|02011|02019|02020|02031|02039|03010|03020|04111|04112|04120|04130|04140|04191|04192|04199|04200|05100|05210|05220|05290|06000|07000|08010|08020|08030|08040|08050|08060|08070|08090|09110|09190|09901|09902|09903|09909|10111|10112|10121|10122|10900|11110|11120|11130|11200|11310|11320|11330|11400|11500|11610|11620|11710|11720|11730|11740|11810|11820|11910|11920|11990|12110|12120|12130|12140|12200|13110|13120|13130|13200|13310|13320|13330|13340|13400|13510|13520|14110|14120|14130|14910|14920|14930|14940|14990|15100|15210|15220|15230|15240|15290|16110|16120|16200|17010|17090|18110|18120|18130|18210|18290|18310|18320|18410|18420|18510|18520|18910|18920|18990|19110|19120|19130|19140|19150|19160|19190|19200|20100|20210|20290|20310|20320|20330|20340|20900|21100|21210|21220|21310|21320|21330|21390|21410|21420|21490|22100|22210|22220|22230|22240|22290|22310|22390|22400|22910|22920|22930|22990|23110|23120|23130|23190|23910|23920|23930|23940|23990|24110|24120|24190|24211|24219|24220|24290|24310|24320|24390|24410|24490|24510|24520|24610|24620|24630|24690|24910|24990|25110|25120|25130|25190|25910|25920|25990|26110|26120|26190|26200|26300|26400|27000|28110|28120|29110|29190|29210|29220|30110|30190|30200|31010|31091|31099|32110|32120|32210|32220|32230|32240|32310|32320|32330|32340|32390|32410|32420|32430|32440|32450|32910|32920|32990|33110|33120|33190|33210|33221|33229|33230|33310|33320|33390|34110|34190|34910|34920|34930|34940|34990|35010|35020|35030|35040|35050|36010|36020|36030|36040|36050|36061|36062|36090|37110|37120|37200|37310|37320|37330|37340|37350|37360|37390|38000|39110|39120|39130|39211|39219|39220|40000|41100|41211|41212|41213|41220|41230|41290|42110|42120|42130|42140|42210|42220|42290|42310|42320|42410|42420|42430|42441|42442|42450|42510|42520|42530|42590|42600|42711|42712|42720|42731|42732|42733|42740|42791|42792|42793|42794|42799|43101|43102|43109|43201|43202|43209|44000|45110|45120|45130|45200|45301|45302|46100|46210|46220|46231|46239|47100|47200|48100|48200|49001|49009|50100|50210|50290|51010|51020|52110|52120|52190|52200|52910|52920|52991|52999|53010|53090|54110|54120|54130|54140|54190|54200|55110|55120|55130|55140|55210|55220|56100|56210|56220|57000|58010|58020|58090|59100|59210|59220|60100|60200|62100|62210|62220|62230|62290|62300|62400|63100|63210|63220|63300|64110|64190|64200|66110|66190|66200|66310|66320|66390|66400|67110|67120|67200|69100|69210|69220|69230|69240|69250|69310|69320|69400|69500|69610|69621|69622|69629|69700|69910|69990|70000|72110|72120|72200|72910|72920|72930|72940|72990|73110|73120|73130|73200|75100|75200|75300|75400|75510|75520|76000|77110|77120|77130|77140|77190|77200|80100|80210|80220|80230|80240|81011|81019|81020|82110|82120|82191|82199|82200|84010|84020|85110|85121|85122|85123|85124|85125|85129|85201|85202|85203|85209|85310|85320|85330|85340|85391|85399|85910|85990|86011|86012|86090|87100|87900|89100|89210|89220|90010|90021|90029|90030|91110|91121|91122|91123|91124|91125|91126|91129|91131|91139|91140|91210|91291|91292|91299|91310|91390|92010|92020|92091|92092|92099|94110|94121|94129|94191|94192|94193|94199|94210|94220|94290|94910|94990|95110|95120|95200|95310|95320|95330|95340|95391|95392|95393|95399|95400|95510|95520|95591|95599|96010|96020|96030|98000|99010|99020|99030|99040|99050|99060|99070)$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.440170",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Business Industry code is not a valid code",
                    Location = "/tns:PTR/tns:RP/tns:OrganisationDetailsOrganisationIndustry2006ExtendedC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440170" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR31", Value = GetValueOrEmpty(report.PTR31) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440300
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440300()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440300
            If the income includes personal services income (PSI) then total amount of income and deductions, results test satisfied and business determination indicators must all be completed
    
            Legacy Rule Format:
            ^PTR110 = TRUE AND (^PTR265 = NULL OR ^PTR266 = NULL OR ^PTR268 = NULL OR ^PTR271 = NULL)

            Technical Business Rule Format:
            ^PTR110 = TRUE AND (^PTR265 = NULL OR ^PTR266 = NULL OR ^PTR268 = NULL OR ^PTR271 = NULL)
    
            Data Elements:
    
            ^PTR110 = PTR:RP:PersonalServicesIncome:Income.PersonalServicesIncomeIncluded.Indicator
    
            ^PTR265 = PTR:RP:PersonalServicesIncome:Income.PersonalServicesIncome.Total.Amount
    
            ^PTR266 = PTR:RP:PersonalServicesIncome:Expense.PersonalServicesIncome.Deduction.Amount
    
            ^PTR268 = PTR:RP:PersonalServicesIncome:IncomeTax.PersonalServicesIncome.IndividualResultsTestSatisfied.Indicator
    
            ^PTR271 = PTR:RP:PersonalServicesIncome:IncomeTax.PersonalServicesIncome.IndividualPersonalServiceBusinessDeterminationHeld.Indicator
            */
            assertion = (report.PTR110 == true && (report.PTR265 == null || report.PTR266 == null || report.PTR268 == null || report.PTR271 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438010",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Personal services income section completed incorrectly",
                    LongDescription = @"If the income includes personal services income (PSI) then total amount of income and deductions, results test satisfied and business determination indicators must all be completed",
                    Location = "/tns:PTR/tns:RP/tns:PersonalServicesIncome/tns:IncludedI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440300" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR110", Value = GetValueOrEmpty(report.PTR110) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR265", Value = GetValueOrEmpty(report.PTR265) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR266", Value = GetValueOrEmpty(report.PTR266) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR268", Value = GetValueOrEmpty(report.PTR268) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR271", Value = GetValueOrEmpty(report.PTR271) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440301
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440301()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440301
            If the income does not include personal services income (PSI) then total amount of income and deductions, results test satisfied , business determination, unrelated clients test, employment test and business premises test indicators must not be completed
    
            Legacy Rule Format:
            ^PTR110 = FALSE AND (^PTR265 <> NULL OR ^PTR266 <> NULL OR ^PTR268 <> NULL OR ^PTR271 <> NULL OR ^PTR276 <> NULL OR ^PTR272 <> NULL OR ^PTR275 <> NULL)

            Technical Business Rule Format:
            ^PTR110 = FALSE AND (^PTR265 <> NULL OR ^PTR266 <> NULL OR ^PTR268 <> NULL OR ^PTR271 <> NULL OR ^PTR276 <> NULL OR ^PTR272 <> NULL OR ^PTR275 <> NULL)
    
            Data Elements:
    
            ^PTR110 = PTR:RP:PersonalServicesIncome:Income.PersonalServicesIncomeIncluded.Indicator
    
            ^PTR265 = PTR:RP:PersonalServicesIncome:Income.PersonalServicesIncome.Total.Amount
    
            ^PTR266 = PTR:RP:PersonalServicesIncome:Expense.PersonalServicesIncome.Deduction.Amount
    
            ^PTR268 = PTR:RP:PersonalServicesIncome:IncomeTax.PersonalServicesIncome.IndividualResultsTestSatisfied.Indicator
    
            ^PTR271 = PTR:RP:PersonalServicesIncome:IncomeTax.PersonalServicesIncome.IndividualPersonalServiceBusinessDeterminationHeld.Indicator
    
            ^PTR272 = PTR:RP:PersonalServicesIncome:IncomeTax.PersonalServicesIncome.EmploymentTestSatisfied.Indicator
    
            ^PTR275 = PTR:RP:PersonalServicesIncome:IncomeTax.PersonalServicesIncome.BusinessPremisesTestSatisfied.Indicator
    
            ^PTR276 = PTR:RP:PersonalServicesIncome:IncomeTax.PersonalServicesIncome.UnrelatedClientsTestSatisfied.Indicator
            */
            assertion = (report.PTR110 == false && (report.PTR265 != null || report.PTR266 != null || report.PTR268 != null || report.PTR271 != null || report.PTR276 != null || report.PTR272 != null || report.PTR275 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Personal services income section completed incorrectly",
                    LongDescription = @"If the income does not include personal services income (PSI) then total amount of income and deductions,results test satisfied,business determination,unrelated clients test,employment test and business premises test indicators must not be completed",
                    Location = "/tns:PTR/tns:RP/tns:PersonalServicesIncome/tns:IncludedI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440301" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR110", Value = GetValueOrEmpty(report.PTR110) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR265", Value = GetValueOrEmpty(report.PTR265) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR266", Value = GetValueOrEmpty(report.PTR266) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR268", Value = GetValueOrEmpty(report.PTR268) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR271", Value = GetValueOrEmpty(report.PTR271) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR276", Value = GetValueOrEmpty(report.PTR276) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR272", Value = GetValueOrEmpty(report.PTR272) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR275", Value = GetValueOrEmpty(report.PTR275) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440302
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440302()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440302
            Non-individual name field cannot contain 'Pship', 'P'ship or 'P/Ship'
    
            Legacy Rule Format:
            StartsWith(^PTR116, 'T/A ') AND EndsWithSet(^PTR116, '" Pship"," P'ship"," P/ship"')

            Technical Business Rule Format:
            StartsWith(^PTR116, 'T/A ') AND EndsWithSet(^PTR116, '" Pship"," P'ship"," P/ship"')
    
            Data Elements:
    
            ^PTR116 = PTR:RP:BusinessName:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (IsMatch(report.PTR116, @"^(T/A )", RegexOptions.IgnoreCase) && IsMatch(report.PTR116, @"( Pship| P'ship| P/ship)$", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410038",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-individual name field cannot start with 'T/A' and end with 'Pship', 'P'ship or 'P/Ship'",
                    Location = "/tns:PTR/tns:RP/tns:BusinessNameOrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440302" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR116", Value = report.PTR116 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440303
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440303()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440303
            Organisation name must contain at least one alpha or numeric character
    
            Legacy Rule Format:
            ^PTR116 <> NULL AND NotContainsSet(^PTR116, '"a-z", "A-Z", "0-9"')

            Technical Business Rule Format:
            ^PTR116 <> NULL AND NotContainsSet(^PTR116, '"a-z", "A-Z", "0-9"')
    
            Data Elements:
    
            ^PTR116 = PTR:RP:BusinessName:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.PTR116 != null && !(IsMatch(report.PTR116, @"[a-z]|[A-Z]|[0-9]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410039",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Organisation name must contain at least one alpha or numeric character",
                    Location = "/tns:PTR/tns:RP/tns:BusinessNameOrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440303" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR116", Value = report.PTR116 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440304
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440304()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440304
            Non Individual Name cannot contain space hyphen space combinations
    
            Legacy Rule Format:
            Contains(^PTR116, ' - ')

            Technical Business Rule Format:
            Contains(^PTR116, ' - ')
    
            Data Elements:
    
            ^PTR116 = PTR:RP:BusinessName:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.PTR116 == null ? false : report.PTR116.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000413",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain space hyphen space combinations",
                    Location = "/tns:PTR/tns:RP/tns:BusinessNameOrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440304" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR116", Value = report.PTR116 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440305
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440305()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440305
            Non Individual Name cannot contain "P/L"
    
            Legacy Rule Format:
            Contains(^PTR116, 'P/L')

            Technical Business Rule Format:
            Contains(^PTR116, 'P/L')
    
            Data Elements:
    
            ^PTR116 = PTR:RP:BusinessName:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.PTR116 == null ? false : report.PTR116.ToUpperInvariant().Contains(@"P/L"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000414",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain ""P/L""",
                    Location = "/tns:PTR/tns:RP/tns:BusinessNameOrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440305" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR116", Value = report.PTR116 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440306
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440306()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440306
            Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            ContainsSet(^PTR116, '"--","''","  "')

            Technical Business Rule Format:
            ContainsSet(^PTR116, '"--","''","  "')
    
            Data Elements:
    
            ^PTR116 = PTR:RP:BusinessName:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.PTR116, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000416",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:PTR/tns:RP/tns:BusinessNameOrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440306" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR116", Value = report.PTR116 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440307
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440307()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440307
            Non Individual Name cannot end with "T/A", "T/A P'ship", "T/A Pship", "T/A P/Ship" or "T/A Partnership"
    
            Legacy Rule Format:
            EndsWithSet(^PTR116, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"')

            Technical Business Rule Format:
            EndsWithSet(^PTR116, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"')
    
            Data Elements:
    
            ^PTR116 = PTR:RP:BusinessName:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.PTR116, @"( T/A| T/A P'ship| T/A Pship| T/A P/Ship| T/A Partnership)$", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000417",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot end with ""T/A"", ""T/A P'ship"", ""T/A Pship"", ""T/A P/Ship"" or ""T/A Partnership""",
                    Location = "/tns:PTR/tns:RP/tns:BusinessNameOrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440307" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR116", Value = report.PTR116 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440308
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440308()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440308
            State and post code must be present for an Australian address
    
            Legacy Rule Format:
            ^PTR222 = FALSE AND (^PTR119 = NULL OR ^PTR120 = NULL)

            Technical Business Rule Format:
            ^PTR222 = FALSE AND (^PTR119 = NULL OR ^PTR120 = NULL)
    
            Data Elements:
    
            ^PTR222 = PTR:RP:BusinessAddressDetails:AddressDetails.OverseasAddress.Indicator
    
            ^PTR119 = PTR:RP:BusinessAddressDetails:AddressDetails.StateOrTerritory.Code
    
            ^PTR120 = PTR:RP:BusinessAddressDetails:AddressDetails.Postcode.Text
            */
            assertion = (report.PTR222 == false && (report.PTR119 == null || report.PTR120 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.434147",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"State and post code must be present for an Australian address",
                    LongDescription = @"If the overseas indicator is no (false), then an Australian state code and post code must be present",
                    Location = "/tns:PTR/tns:RP/tns:BusinessAddressDetails/tns:OverseasAddressI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440308" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR222", Value = GetValueOrEmpty(report.PTR222) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR119", Value = GetValueOrEmpty(report.PTR119) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR120", Value = GetValueOrEmpty(report.PTR120) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440309
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440309()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440309
            Address line 1 contains incorrect 'care of' reference. The only acceptable 'care of' reference is C/-
    
            Legacy Rule Format:
            ^PTR117 <> NULL AND (StartsWithSet(^PTR117,'"C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^PTR117, '" C/O "," C/ "," Care Of "'))

            Technical Business Rule Format:
            ^PTR117 <> NULL AND (StartsWithSet(^PTR117,'"C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^PTR117, '" C/O "," C/ "," Care Of "'))
    
            Data Elements:
    
            ^PTR117 = PTR:RP:BusinessAddressDetails:AddressDetails.Line1.Text
            */
            assertion = (report.PTR117 != null && (IsMatch(report.PTR117, @"^(C/O |C/ |Care Of |CO )", RegexOptions.IgnoreCase) || IsMatch(report.PTR117, @" C/O | C/ | Care Of ", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000406",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 contains incorrect 'care of' reference.  The only acceptable 'care of' reference is C/-",
                    Location = "/tns:PTR/tns:RP/tns:BusinessAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440309" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR117", Value = report.PTR117 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440310
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440310()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440310
            Address Line 2 must be present if Address Line 1 contains 'C/-'
    
            Legacy Rule Format:
            Contains(^PTR117, 'C/-') AND ^PTR213 = NULL

            Technical Business Rule Format:
            Contains(^PTR117, 'C/-') AND ^PTR213 = NULL
    
            Data Elements:
    
            ^PTR117 = PTR:RP:BusinessAddressDetails:AddressDetails.Line1.Text
    
            ^PTR213 = PTR:RP:BusinessAddressDetails:AddressDetails.Line2.Text
            */
            assertion = ((report.PTR117 == null ? false : report.PTR117.ToUpperInvariant().Contains(@"C/-")) && report.PTR213 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410002",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 2 must be present if Address Line 1 contains 'C/-'",
                    Location = "/tns:PTR/tns:RP/tns:BusinessAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440310" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR117", Value = report.PTR117 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR213", Value = GetValueOrEmpty(report.PTR213) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440311
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440311()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440311
            Address Line 1 must not contain 'AS ABOVE'
    
            Legacy Rule Format:
            FoundSet(^PTR117, '"AS ABOVE"')

            Technical Business Rule Format:
            FoundSet(^PTR117, '"AS ABOVE"')
    
            Data Elements:
    
            ^PTR117 = PTR:RP:BusinessAddressDetails:AddressDetails.Line1.Text
            */
            assertion = IsMatch(report.PTR117, @"(^(AS ABOVE)$)|(^(.*)( AS ABOVE )(.*)$)|(^(AS ABOVE )(.*)$)|(^(.*)( AS ABOVE)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 1 must not contain 'AS ABOVE'",
                    Location = "/tns:PTR/tns:RP/tns:BusinessAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440311" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR117", Value = report.PTR117 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440312
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440312()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440312
            Address line 1 cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            Contains(^PTR117, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^PTR117, 'UNKNOWN')
    
            Data Elements:
    
            ^PTR117 = PTR:RP:BusinessAddressDetails:AddressDetails.Line1.Text
            */
            assertion = (report.PTR117 == null ? false : report.PTR117.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000405",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 cannot contain ""UNKNOWN""",
                    Location = "/tns:PTR/tns:RP/tns:BusinessAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440312" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR117", Value = report.PTR117 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440313
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440313()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440313
            'Care of' is not valid for address line 2
    
            Legacy Rule Format:
            ^PTR213 <> NULL AND (StartsWithSet(^PTR213,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^PTR213, '" C/- "," C/O "," C/ "," Care Of "'))

            Technical Business Rule Format:
            ^PTR213 <> NULL AND (StartsWithSet(^PTR213,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^PTR213, '" C/- "," C/O "," C/ "," Care Of "'))
    
            Data Elements:
    
            ^PTR213 = PTR:RP:BusinessAddressDetails:AddressDetails.Line2.Text
            */
            assertion = (report.PTR213 != null && (IsMatch(report.PTR213, @"^(C/- |C/O |C/ |Care Of |CO )", RegexOptions.IgnoreCase) || IsMatch(report.PTR213, @" C/- | C/O | C/ | Care Of ", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000408",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"'Care of' is not valid for address line 2",
                    Location = "/tns:PTR/tns:RP/tns:BusinessAddressDetails/tns:Line2T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440313" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR213", Value = GetValueOrEmpty(report.PTR213) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440314
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440314()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440314
            Address line 2 cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            Contains(^PTR213, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^PTR213, 'UNKNOWN')
    
            Data Elements:
    
            ^PTR213 = PTR:RP:BusinessAddressDetails:AddressDetails.Line2.Text
            */
            assertion = (report.PTR213 == null ? false : report.PTR213.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000407",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 2 cannot contain ""UNKNOWN""",
                    Location = "/tns:PTR/tns:RP/tns:BusinessAddressDetails/tns:Line2T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440314" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR213", Value = GetValueOrEmpty(report.PTR213) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440315
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440315()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440315
            Suburb/Town/City cannot contain state code
    
            Legacy Rule Format:
            FoundSet(^PTR118, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"')

            Technical Business Rule Format:
            FoundSet(^PTR118, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"')
    
            Data Elements:
    
            ^PTR118 = PTR:RP:BusinessAddressDetails:AddressDetails.LocalityName.Text
            */
            assertion = IsMatch(report.PTR118, @"(^(QLD|NSW|VIC|SA|WA|NT|ACT|TAS)$)|(^(.*)( QLD | NSW | VIC | SA | WA | NT | ACT | TAS )(.*)$)|(^(QLD |NSW |VIC |SA |WA |NT |ACT |TAS )(.*)$)|(^(.*)( QLD| NSW| VIC| SA| WA| NT| ACT| TAS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000411",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain state code",
                    Location = "/tns:PTR/tns:RP/tns:BusinessAddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440315" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR118", Value = report.PTR118 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440316
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440316()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440316
            Suburb/Town/City cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            Contains(^PTR118, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^PTR118, 'UNKNOWN')
    
            Data Elements:
    
            ^PTR118 = PTR:RP:BusinessAddressDetails:AddressDetails.LocalityName.Text
            */
            assertion = (report.PTR118 == null ? false : report.PTR118.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000409",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain ""UNKNOWN""",
                    Location = "/tns:PTR/tns:RP/tns:BusinessAddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440316" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR118", Value = report.PTR118 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440317
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440317()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440317
            Suburb/Town/City cannot contain numeric characters
    
            Legacy Rule Format:
            ^PTR222 = FALSE AND ContainsSet(^PTR118, '0-9')

            Technical Business Rule Format:
            ^PTR222 = FALSE AND ContainsSet(^PTR118, '0-9')
    
            Data Elements:
    
            ^PTR118 = PTR:RP:BusinessAddressDetails:AddressDetails.LocalityName.Text
    
            ^PTR222 = PTR:RP:BusinessAddressDetails:AddressDetails.OverseasAddress.Indicator
            */
            assertion = (report.PTR222 == false && IsMatch(report.PTR118, @"[0-9]", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000410",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain numeric characters",
                    Location = "/tns:PTR/tns:RP/tns:BusinessAddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440317" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR222", Value = GetValueOrEmpty(report.PTR222) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR118", Value = report.PTR118 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440318
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440318()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440318
            Country code must be present for overseas address
    
            Legacy Rule Format:
            ^PTR304 = NULL AND ^PTR222 = TRUE

            Technical Business Rule Format:
            ^PTR304 = NULL AND ^PTR222 = TRUE
    
            Data Elements:
    
            ^PTR304 = PTR:RP:BusinessAddressDetails:AddressDetails.Country.Code
    
            ^PTR222 = PTR:RP:BusinessAddressDetails:AddressDetails.OverseasAddress.Indicator
            */
            assertion = (report.PTR304 == null && report.PTR222 == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410191",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Country code must be present for overseas address",
                    LongDescription = @"When the overseas address indicator is 'TRUE', the corresponding Country Code must be supplied",
                    Location = "/tns:PTR/tns:RP/tns:BusinessAddressDetails/tns:CountryC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440318" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR304", Value = GetValueOrEmpty(report.PTR304) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR222", Value = GetValueOrEmpty(report.PTR222) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440319
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440319()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440319
            Overseas address indicator must not be 'TRUE' if Country Code is 'au'
    
            Legacy Rule Format:
            ^PTR304 = 'au' AND ^PTR222 = TRUE

            Technical Business Rule Format:
            ^PTR304 = 'au' AND ^PTR222 = TRUE
    
            Data Elements:
    
            ^PTR304 = PTR:RP:BusinessAddressDetails:AddressDetails.Country.Code
    
            ^PTR222 = PTR:RP:BusinessAddressDetails:AddressDetails.OverseasAddress.Indicator
            */
            assertion = (report.PTR304 == @"au" && report.PTR222 == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410211",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Overseas address indicator must not be 'TRUE' if Country Code is 'au'",
                    LongDescription = @"The overseas address indicator must be 'FALSE' if the Country Code is 'au' - representing Australia. If the address is an overseas address please supply the correct Country Code",
                    Location = "/tns:PTR/tns:RP/tns:BusinessAddressDetails/tns:CountryC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440319" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR304", Value = GetValueOrEmpty(report.PTR304) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR222", Value = GetValueOrEmpty(report.PTR222) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440320
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440320()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440320
            If an overseas Country Code is used, the Overseas Address indicator must be 'TRUE'
    
            Legacy Rule Format:
            ^PTR304 <> NULL AND ^PTR304 <> 'au' AND ^PTR222 = FALSE

            Technical Business Rule Format:
            ^PTR304 <> NULL AND ^PTR304 <> 'au' AND ^PTR222 = FALSE
    
            Data Elements:
    
            ^PTR304 = PTR:RP:BusinessAddressDetails:AddressDetails.Country.Code
    
            ^PTR222 = PTR:RP:BusinessAddressDetails:AddressDetails.OverseasAddress.Indicator
            */
            assertion = (report.PTR304 != null && report.PTR304 != @"au" && report.PTR222 == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410212",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"If an overseas Country Code is used, the Overseas Address indicator must be 'TRUE'",
                    LongDescription = @"If the address is an Australian address, the overseas address indicator must be set to 'FALSE' and you do not need to supply a Country Code. If the address is an overseas address, the overseas address indicator must be set to 'TRUE' and you must supply a Country Code other than 'au'",
                    Location = "/tns:PTR/tns:RP/tns:BusinessAddressDetails/tns:CountryC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440320" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR304", Value = GetValueOrEmpty(report.PTR304) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR222", Value = GetValueOrEmpty(report.PTR222) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440321
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440321(PTR2025.RP_StatementOfDistribution statementOfDistribution, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440321
            The Tax File Number has failed the Tax File Number algorithm check.

            Legacy Rule Format:
            ^PTR157 <> NULL AND FailsTFNAlgorithm(^PTR157)

            Technical Business Rule Format:
            ^PTR157 <> NULL AND FailsTFNAlgorithm(^PTR157)

            Data Elements:

            ^PTR157 = PTR:RP:StatementOfDistribution:Identifiers.TaxFileNumber.Identifier
            */
            assertion = (statementOfDistribution.PTR157 != null && FailsTFNAlgorithm(statementOfDistribution.PTR157));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410031",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"TFN must be a valid TFN",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:TaxFileNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440321" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR157", Value = GetValueOrEmpty(statementOfDistribution.PTR157) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440322
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440322(PTR2025.RP_StatementOfDistribution statementOfDistribution, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440322
            Non-individual name field cannot contain 'Pship', 'P'ship or 'P/Ship'

            Legacy Rule Format:
            StartsWith(^PTR149, 'T/A ') AND EndsWithSet(^PTR149, '" Pship"," P'ship"," P/ship"')

            Technical Business Rule Format:
            StartsWith(^PTR149, 'T/A ') AND EndsWithSet(^PTR149, '" Pship"," P'ship"," P/ship"')

            Data Elements:

            ^PTR149 = PTR:RP:StatementOfDistribution:NonIndividualPartnerDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (IsMatch(statementOfDistribution.PTR149, @"^(T/A )", RegexOptions.IgnoreCase) && IsMatch(statementOfDistribution.PTR149, @"( Pship| P'ship| P/ship)$", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410038",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-individual name field cannot start with 'T/A' and end with 'Pship', 'P'ship or 'P/Ship'",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:NonIndividualPartnerDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440322" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR149", Value = statementOfDistribution.PTR149 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440323
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440323(PTR2025.RP_StatementOfDistribution statementOfDistribution, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440323
            Organisation name must contain at least one alpha or numeric character

            Legacy Rule Format:
            ^PTR149 <> NULL AND NotContainsSet(^PTR149, '"a-z", "A-Z", "0-9"')

            Technical Business Rule Format:
            ^PTR149 <> NULL AND NotContainsSet(^PTR149, '"a-z", "A-Z", "0-9"')

            Data Elements:

            ^PTR149 = PTR:RP:StatementOfDistribution:NonIndividualPartnerDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (statementOfDistribution.PTR149 != null && !(IsMatch(statementOfDistribution.PTR149, @"[a-z]|[A-Z]|[0-9]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410039",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Organisation name must contain at least one alpha or numeric character",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:NonIndividualPartnerDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440323" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR149", Value = statementOfDistribution.PTR149 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440324
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440324(PTR2025.RP_StatementOfDistribution statementOfDistribution, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440324
            Non Individual Name cannot contain space hyphen space combinations

            Legacy Rule Format:
            Contains(^PTR149, ' - ')

            Technical Business Rule Format:
            Contains(^PTR149, ' - ')

            Data Elements:

            ^PTR149 = PTR:RP:StatementOfDistribution:NonIndividualPartnerDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (statementOfDistribution.PTR149 == null ? false : statementOfDistribution.PTR149.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000413",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain space hyphen space combinations",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:NonIndividualPartnerDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440324" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR149", Value = statementOfDistribution.PTR149 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440325
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440325(PTR2025.RP_StatementOfDistribution statementOfDistribution, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440325
            Non Individual Name cannot contain "P/L"

            Legacy Rule Format:
            Contains(^PTR149, 'P/L')

            Technical Business Rule Format:
            Contains(^PTR149, 'P/L')

            Data Elements:

            ^PTR149 = PTR:RP:StatementOfDistribution:NonIndividualPartnerDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (statementOfDistribution.PTR149 == null ? false : statementOfDistribution.PTR149.ToUpperInvariant().Contains(@"P/L"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000414",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain ""P/L""",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:NonIndividualPartnerDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440325" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR149", Value = statementOfDistribution.PTR149 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440326
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440326(PTR2025.RP_StatementOfDistribution statementOfDistribution, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440326
            Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters

            Legacy Rule Format:
            ContainsSet(^PTR149, '"--","''","  "')

            Technical Business Rule Format:
            ContainsSet(^PTR149, '"--","''","  "')

            Data Elements:

            ^PTR149 = PTR:RP:StatementOfDistribution:NonIndividualPartnerDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(statementOfDistribution.PTR149, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000416",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:NonIndividualPartnerDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440326" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR149", Value = statementOfDistribution.PTR149 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440327
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440327(PTR2025.RP_StatementOfDistribution statementOfDistribution, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440327
            Non Individual Name cannot end with "T/A", "T/A P'ship", "T/A Pship", "T/A P/Ship" or "T/A Partnership"

            Legacy Rule Format:
            EndsWithSet(^PTR149, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"')

            Technical Business Rule Format:
            EndsWithSet(^PTR149, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"')

            Data Elements:

            ^PTR149 = PTR:RP:StatementOfDistribution:NonIndividualPartnerDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(statementOfDistribution.PTR149, @"( T/A| T/A P'ship| T/A Pship| T/A P/Ship| T/A Partnership)$", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000417",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot end with ""T/A"", ""T/A P'ship"", ""T/A Pship"", ""T/A P/Ship"" or ""T/A Partnership""",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:NonIndividualPartnerDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440327" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR149", Value = statementOfDistribution.PTR149 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440328
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440328(PTR2025.RP_StatementOfDistribution statementOfDistribution, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440328
            ABN must pass the ABN algorithm

            Legacy Rule Format:
            ^PTR285 <> NULL AND FailsABNAlgorithm(^PTR285)

            Technical Business Rule Format:
            ^PTR285 <> NULL AND FailsABNAlgorithm(^PTR285)

            Data Elements:

            ^PTR285 = PTR:RP:StatementOfDistribution:NonIndividualPartnerDetails:Identifiers.AustralianBusinessNumber.Identifier
            */
            assertion = (statementOfDistribution.PTR285 != null && FailsABNAlgorithm(statementOfDistribution.PTR285));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000477",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"ABN is invalid",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:NonIndividualPartnerDetails/tns:AustralianBusinessNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440328" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR285", Value = GetValueOrEmpty(statementOfDistribution.PTR285) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440329
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440329(PTR2025.RP_StatementOfDistribution statementOfDistribution, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440329
            Family name cannot contain "Exec for", "Rep for" or "Trustee for"

            Legacy Rule Format:
            FoundSet(^PTR146, '"Exec for","Rep for","Trustee for"')

            Technical Business Rule Format:
            FoundSet(^PTR146, '"Exec for","Rep for","Trustee for"')

            Data Elements:

            ^PTR146 = PTR:RP:StatementOfDistribution:IndividualPartnerDetails:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(statementOfDistribution.PTR146, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000424",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:IndividualPartnerDetails/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440329" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR146", Value = statementOfDistribution.PTR146 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440330
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440330(PTR2025.RP_StatementOfDistribution statementOfDistribution, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440330
            Family name cannot contain "MR", "MRS", "MISS" or "MS"

            Legacy Rule Format:
            FoundSet(^PTR146, '"MR","MRS","MISS","MS"')

            Technical Business Rule Format:
            FoundSet(^PTR146, '"MR","MRS","MISS","MS"')

            Data Elements:

            ^PTR146 = PTR:RP:StatementOfDistribution:IndividualPartnerDetails:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(statementOfDistribution.PTR146, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000426",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:IndividualPartnerDetails/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440330" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR146", Value = statementOfDistribution.PTR146 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440331
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440331(PTR2025.RP_StatementOfDistribution statementOfDistribution, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440331
            Family Name cannot contain "space hyphen space"

            Legacy Rule Format:
            Contains(^PTR146, ' - ')

            Technical Business Rule Format:
            Contains(^PTR146, ' - ')

            Data Elements:

            ^PTR146 = PTR:RP:StatementOfDistribution:IndividualPartnerDetails:PersonNameDetails.FamilyName.Text
            */
            assertion = (statementOfDistribution.PTR146 == null ? false : statementOfDistribution.PTR146.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000423",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family Name cannot contain ""space hyphen space""",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:IndividualPartnerDetails/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440331" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR146", Value = statementOfDistribution.PTR146 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440332
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440332(PTR2025.RP_StatementOfDistribution statementOfDistribution, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440332
            Family name cannot contain repeated hyphen, apostrophe, or space characters

            Legacy Rule Format:
            ContainsSet(^PTR146, '"--","''","  "')

            Technical Business Rule Format:
            ContainsSet(^PTR146, '"--","''","  "')

            Data Elements:

            ^PTR146 = PTR:RP:StatementOfDistribution:IndividualPartnerDetails:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(statementOfDistribution.PTR146, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000427",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:IndividualPartnerDetails/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440332" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR146", Value = statementOfDistribution.PTR146 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440333
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440333(PTR2025.RP_StatementOfDistribution statementOfDistribution, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440333
            Family name must contain at least one alphabetical character

            Legacy Rule Format:
            ^PTR146 <> NULL AND NotContainsSet(^PTR146, '"A-Z","a-z"')

            Technical Business Rule Format:
            ^PTR146 <> NULL AND NotContainsSet(^PTR146, '"A-Z","a-z"')

            Data Elements:

            ^PTR146 = PTR:RP:StatementOfDistribution:IndividualPartnerDetails:PersonNameDetails.FamilyName.Text
            */
            assertion = (statementOfDistribution.PTR146 != null && !(IsMatch(statementOfDistribution.PTR146, @"[A-Z]|[a-z]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500142",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name must contain at least one alphabetical character.",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:IndividualPartnerDetails/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440333" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR146", Value = statementOfDistribution.PTR146 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440334
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440334(PTR2025.RP_StatementOfDistribution statementOfDistribution, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440334
            Family name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR

            Legacy Rule Format:
            FoundSet(^PTR146, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')

            Technical Business Rule Format:
            FoundSet(^PTR146, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')

            Data Elements:

            ^PTR146 = PTR:RP:StatementOfDistribution:IndividualPartnerDetails:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(statementOfDistribution.PTR146, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500153",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:IndividualPartnerDetails/tns:PersonNameDetailsFamilyNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440334" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR146", Value = statementOfDistribution.PTR146 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440335
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440335(PTR2025.RP_StatementOfDistribution statementOfDistribution, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440335
            Given Name must contain at least one alphabetical character

            Legacy Rule Format:
            ^PTR147 <> BLANK AND NotContainsSet(^PTR147, '"A-Z","a-z"')

            Technical Business Rule Format:
            ^PTR147 <> BLANK AND NotContainsSet(^PTR147, '"A-Z","a-z"')

            Data Elements:

            ^PTR147 = PTR:RP:StatementOfDistribution:IndividualPartnerDetails:PersonNameDetails.GivenName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(statementOfDistribution.PTR147) != true && !(IsMatch(statementOfDistribution.PTR147, @"[A-Z]|[a-z]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410063",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Given Name must contain at least one alphabetical character",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:IndividualPartnerDetails/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440335" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR147", Value = GetValueOrEmpty(statementOfDistribution.PTR147) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440336
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440336(PTR2025.RP_StatementOfDistribution statementOfDistribution, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440336
            First name cannot contain "Exec for", "Rep for" or "Trustee for"

            Legacy Rule Format:
            FoundSet(^PTR147, '"Exec for","Rep for","Trustee for"')

            Technical Business Rule Format:
            FoundSet(^PTR147, '"Exec for","Rep for","Trustee for"')

            Data Elements:

            ^PTR147 = PTR:RP:StatementOfDistribution:IndividualPartnerDetails:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(statementOfDistribution.PTR147, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000437",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:IndividualPartnerDetails/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440336" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR147", Value = GetValueOrEmpty(statementOfDistribution.PTR147) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440337
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440337(PTR2025.RP_StatementOfDistribution statementOfDistribution, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440337
            First name cannot contain "MR", "MRS", "MISS" or "MS"

            Legacy Rule Format:
            FoundSet(^PTR147, '"MR","MRS","MISS","MS"')

            Technical Business Rule Format:
            FoundSet(^PTR147, '"MR","MRS","MISS","MS"')

            Data Elements:

            ^PTR147 = PTR:RP:StatementOfDistribution:IndividualPartnerDetails:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(statementOfDistribution.PTR147, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000438",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:IndividualPartnerDetails/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440337" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR147", Value = GetValueOrEmpty(statementOfDistribution.PTR147) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440338
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440338(PTR2025.RP_StatementOfDistribution statementOfDistribution, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440338
            FirstName must be entered if OtherGivenNames is entered

            Legacy Rule Format:
            ^PTR147 = BLANK AND ^PTR148 <> BLANK

            Technical Business Rule Format:
            ^PTR147 = BLANK AND ^PTR148 <> BLANK

            Data Elements:

            ^PTR147 = PTR:RP:StatementOfDistribution:IndividualPartnerDetails:PersonNameDetails.GivenName.Text

            ^PTR148 = PTR:RP:StatementOfDistribution:IndividualPartnerDetails:PersonNameDetails.OtherGivenName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(statementOfDistribution.PTR147) == true && string.IsNullOrWhiteSpace(statementOfDistribution.PTR148) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410218",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"FirstName must be entered if OtherGivenNames is entered",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:IndividualPartnerDetails/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440338" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR147", Value = GetValueOrEmpty(statementOfDistribution.PTR147) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR148", Value = GetValueOrEmpty(statementOfDistribution.PTR148) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440339
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440339(PTR2025.RP_StatementOfDistribution statementOfDistribution, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440339
            First name cannot contain space hyphen space

            Legacy Rule Format:
            Contains(^PTR147, ' - ')

            Technical Business Rule Format:
            Contains(^PTR147, ' - ')

            Data Elements:

            ^PTR147 = PTR:RP:StatementOfDistribution:IndividualPartnerDetails:PersonNameDetails.GivenName.Text
            */
            assertion = (statementOfDistribution.PTR147 == null ? false : statementOfDistribution.PTR147.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000434",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain space hyphen space",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:IndividualPartnerDetails/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440339" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR147", Value = GetValueOrEmpty(statementOfDistribution.PTR147) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440340
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440340(PTR2025.RP_StatementOfDistribution statementOfDistribution, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440340
            First name cannot contain repeated hyphen, apostrophe, or space characters

            Legacy Rule Format:
            ContainsSet(^PTR147, '"--","''","  "')

            Technical Business Rule Format:
            ContainsSet(^PTR147, '"--","''","  "')

            Data Elements:

            ^PTR147 = PTR:RP:StatementOfDistribution:IndividualPartnerDetails:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(statementOfDistribution.PTR147, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000439",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:IndividualPartnerDetails/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440340" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR147", Value = GetValueOrEmpty(statementOfDistribution.PTR147) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440341
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440341(PTR2025.RP_StatementOfDistribution statementOfDistribution, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440341
            First name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR

            Legacy Rule Format:
            FoundSet(^PTR147, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')

            Technical Business Rule Format:
            FoundSet(^PTR147, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')

            Data Elements:

            ^PTR147 = PTR:RP:StatementOfDistribution:IndividualPartnerDetails:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(statementOfDistribution.PTR147, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010436",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:IndividualPartnerDetails/tns:PersonNameDetailsGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440341" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR147", Value = GetValueOrEmpty(statementOfDistribution.PTR147) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440342
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440342(PTR2025.RP_StatementOfDistribution statementOfDistribution, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440342
            Other Given Name must contain at least one alphabetical character

            Legacy Rule Format:
            ^PTR148 <> BLANK AND NotContainsSet(^PTR148, '"A-Z","a-z"')

            Technical Business Rule Format:
            ^PTR148 <> BLANK AND NotContainsSet(^PTR148, '"A-Z","a-z"')

            Data Elements:

            ^PTR148 = PTR:RP:StatementOfDistribution:IndividualPartnerDetails:PersonNameDetails.OtherGivenName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(statementOfDistribution.PTR148) != true && !(IsMatch(statementOfDistribution.PTR148, @"[A-Z]|[a-z]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410131",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other Given Name must contain at least one alphabetical character",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:IndividualPartnerDetails/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440342" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR148", Value = GetValueOrEmpty(statementOfDistribution.PTR148) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440343
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440343(PTR2025.RP_StatementOfDistribution statementOfDistribution, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440343
            Other given name cannot contain "Exec for", "Rep for" or "Trustee for"

            Legacy Rule Format:
            FoundSet(^PTR148, '"Exec for","Rep for","Trustee for"')

            Technical Business Rule Format:
            FoundSet(^PTR148, '"Exec for","Rep for","Trustee for"')

            Data Elements:

            ^PTR148 = PTR:RP:StatementOfDistribution:IndividualPartnerDetails:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(statementOfDistribution.PTR148, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000449",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:IndividualPartnerDetails/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440343" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR148", Value = GetValueOrEmpty(statementOfDistribution.PTR148) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440344
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440344(PTR2025.RP_StatementOfDistribution statementOfDistribution, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440344
            Other given name cannot contain "MR", "MRS", "MISS" or "MS"

            Legacy Rule Format:
            FoundSet(^PTR148, '"MR","MRS","MISS","MS"')

            Technical Business Rule Format:
            FoundSet(^PTR148, '"MR","MRS","MISS","MS"')

            Data Elements:

            ^PTR148 = PTR:RP:StatementOfDistribution:IndividualPartnerDetails:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(statementOfDistribution.PTR148, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000450",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:IndividualPartnerDetails/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440344" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR148", Value = GetValueOrEmpty(statementOfDistribution.PTR148) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440345
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440345(PTR2025.RP_StatementOfDistribution statementOfDistribution, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440345
            Other given name cannot contain space hyphen space

            Legacy Rule Format:
            Contains(^PTR148, ' - ')

            Technical Business Rule Format:
            Contains(^PTR148, ' - ')

            Data Elements:

            ^PTR148 = PTR:RP:StatementOfDistribution:IndividualPartnerDetails:PersonNameDetails.OtherGivenName.Text
            */
            assertion = (statementOfDistribution.PTR148 == null ? false : statementOfDistribution.PTR148.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000446",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain space hyphen space",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:IndividualPartnerDetails/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440345" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR148", Value = GetValueOrEmpty(statementOfDistribution.PTR148) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440346
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440346(PTR2025.RP_StatementOfDistribution statementOfDistribution, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440346
            Other given name cannot contain repeated hyphen, apostrophe, or space characters

            Legacy Rule Format:
            ContainsSet(^PTR148, '"--","''","  "')

            Technical Business Rule Format:
            ContainsSet(^PTR148, '"--","''","  "')

            Data Elements:

            ^PTR148 = PTR:RP:StatementOfDistribution:IndividualPartnerDetails:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(statementOfDistribution.PTR148, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000451",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:IndividualPartnerDetails/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440346" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR148", Value = GetValueOrEmpty(statementOfDistribution.PTR148) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440347
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440347(PTR2025.RP_StatementOfDistribution statementOfDistribution, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440347
            Other given name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR

            Legacy Rule Format:
            FoundSet(^PTR148, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')

            Technical Business Rule Format:
            FoundSet(^PTR148, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')

            Data Elements:

            ^PTR148 = PTR:RP:StatementOfDistribution:IndividualPartnerDetails:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(statementOfDistribution.PTR148, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010448",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:IndividualPartnerDetails/tns:PersonNameDetailsOtherGivenNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440347" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR148", Value = GetValueOrEmpty(statementOfDistribution.PTR148) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440348
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440348(PTR2025.RP_StatementOfDistribution statementOfDistribution, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440348
            State and post code must be present for an Australian address

            Legacy Rule Format:
            ^PTR240 = FALSE AND (^PTR154 = NULL OR ^PTR155 = NULL)

            Technical Business Rule Format:
            ^PTR240 = FALSE AND (^PTR154 = NULL OR ^PTR155 = NULL)

            Data Elements:

            ^PTR240 = PTR:RP:StatementOfDistribution:PartnerAddressDetails:AddressDetails.OverseasAddress.Indicator

            ^PTR154 = PTR:RP:StatementOfDistribution:PartnerAddressDetails:AddressDetails.StateOrTerritory.Code

            ^PTR155 = PTR:RP:StatementOfDistribution:PartnerAddressDetails:AddressDetails.Postcode.Text
            */
            assertion = (statementOfDistribution.PTR240 == false && (statementOfDistribution.PTR154 == null || statementOfDistribution.PTR155 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.434147",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"State and post code must be present for an Australian address",
                    LongDescription = @"If the overseas indicator is no (false), then an Australian state code and post code must be present",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:PartnerAddressDetails/tns:OverseasAddressI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440348" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR240", Value = GetValueOrEmpty(statementOfDistribution.PTR240) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR154", Value = GetValueOrEmpty(statementOfDistribution.PTR154) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR155", Value = GetValueOrEmpty(statementOfDistribution.PTR155) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440349
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440349(PTR2025.RP_StatementOfDistribution statementOfDistribution, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440349
            Address line 1 contains incorrect 'care of' reference. The only acceptable 'care of' reference is C/-

            Legacy Rule Format:
            ^PTR151 <> NULL AND (StartsWithSet(^PTR151,'"C/O ","C/ ","Care Of ","CO "')) OR (ContainsSet(^PTR151, '" C/O "," C/ "," Care Of "'))

            Technical Business Rule Format:
            ^PTR151 <> NULL AND (StartsWithSet(^PTR151,'"C/O ","C/ ","Care Of ","CO "')) OR (ContainsSet(^PTR151, '" C/O "," C/ "," Care Of "'))

            Data Elements:

            ^PTR151 = PTR:RP:StatementOfDistribution:PartnerAddressDetails:AddressDetails.Line1.Text
            */
            assertion = (statementOfDistribution.PTR151 != null && IsMatch(statementOfDistribution.PTR151, @"^(C/O |C/ |Care Of |CO )", RegexOptions.IgnoreCase) || IsMatch(statementOfDistribution.PTR151, @" C/O | C/ | Care Of ", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000406",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 contains incorrect 'care of' reference.  The only acceptable 'care of' reference is C/-",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:PartnerAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440349" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR151", Value = statementOfDistribution.PTR151 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440350
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440350(PTR2025.RP_StatementOfDistribution statementOfDistribution, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440350
            Address Line 2 must be present if Address Line 1 contains 'C/-'

            Legacy Rule Format:
            Contains(^PTR151, 'C/-') AND ^PTR152 = NULL

            Technical Business Rule Format:
            Contains(^PTR151, 'C/-') AND ^PTR152 = NULL

            Data Elements:

            ^PTR151 = PTR:RP:StatementOfDistribution:PartnerAddressDetails:AddressDetails.Line1.Text

            ^PTR152 = PTR:RP:StatementOfDistribution:PartnerAddressDetails:AddressDetails.Line2.Text
            */
            assertion = ((statementOfDistribution.PTR151 == null ? false : statementOfDistribution.PTR151.ToUpperInvariant().Contains(@"C/-")) && statementOfDistribution.PTR152 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410002",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 2 must be present if Address Line 1 contains 'C/-'",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:PartnerAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440350" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR151", Value = statementOfDistribution.PTR151 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR152", Value = GetValueOrEmpty(statementOfDistribution.PTR152) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440351
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440351(PTR2025.RP_StatementOfDistribution statementOfDistribution, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440351
            Address Line 1 must not contain 'AS ABOVE'

            Legacy Rule Format:
            FoundSet(^PTR151, '"AS ABOVE"')

            Technical Business Rule Format:
            FoundSet(^PTR151, '"AS ABOVE"')

            Data Elements:

            ^PTR151 = PTR:RP:StatementOfDistribution:PartnerAddressDetails:AddressDetails.Line1.Text
            */
            assertion = IsMatch(statementOfDistribution.PTR151, @"(^(AS ABOVE)$)|(^(.*)( AS ABOVE )(.*)$)|(^(AS ABOVE )(.*)$)|(^(.*)( AS ABOVE)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 1 must not contain 'AS ABOVE'",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:PartnerAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440351" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR151", Value = statementOfDistribution.PTR151 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440352
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440352(PTR2025.RP_StatementOfDistribution statementOfDistribution, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440352
            Address line 1 cannot contain "UNKNOWN"

            Legacy Rule Format:
            Contains(^PTR151, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^PTR151, 'UNKNOWN')

            Data Elements:

            ^PTR151 = PTR:RP:StatementOfDistribution:PartnerAddressDetails:AddressDetails.Line1.Text
            */
            assertion = (statementOfDistribution.PTR151 == null ? false : statementOfDistribution.PTR151.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000405",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 cannot contain ""UNKNOWN""",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:PartnerAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440352" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR151", Value = statementOfDistribution.PTR151 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440353
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440353(PTR2025.RP_StatementOfDistribution statementOfDistribution, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440353
            'Care of' is not valid for address line 2

            Legacy Rule Format:
            ^PTR152 <> NULL AND (StartsWithSet(^PTR152,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^PTR152, '" C/- "," C/O "," C/ "," Care Of "'))

            Technical Business Rule Format:
            ^PTR152 <> NULL AND (StartsWithSet(^PTR152,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^PTR152, '" C/- "," C/O "," C/ "," Care Of "'))

            Data Elements:

            ^PTR152 = PTR:RP:StatementOfDistribution:PartnerAddressDetails:AddressDetails.Line2.Text
            */
            assertion = (statementOfDistribution.PTR152 != null && (IsMatch(statementOfDistribution.PTR152, @"^(C/- |C/O |C/ |Care Of |CO )", RegexOptions.IgnoreCase) || IsMatch(statementOfDistribution.PTR152, @" C/- | C/O | C/ | Care Of ", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000408",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"'Care of' is not valid for address line 2",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:PartnerAddressDetails/tns:Line2T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440353" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR152", Value = GetValueOrEmpty(statementOfDistribution.PTR152) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440354
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440354(PTR2025.RP_StatementOfDistribution statementOfDistribution, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440354
            Address line 2 cannot contain "UNKNOWN"

            Legacy Rule Format:
            Contains(^PTR152, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^PTR152, 'UNKNOWN')

            Data Elements:

            ^PTR152 = PTR:RP:StatementOfDistribution:PartnerAddressDetails:AddressDetails.Line2.Text
            */
            assertion = (statementOfDistribution.PTR152 == null ? false : statementOfDistribution.PTR152.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000407",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 2 cannot contain ""UNKNOWN""",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:PartnerAddressDetails/tns:Line2T",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440354" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR152", Value = GetValueOrEmpty(statementOfDistribution.PTR152) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440355
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440355(PTR2025.RP_StatementOfDistribution statementOfDistribution, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440355
            Suburb/Town/City cannot contain state code

            Legacy Rule Format:
            FoundSet(^PTR153, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"')

            Technical Business Rule Format:
            FoundSet(^PTR153, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"')

            Data Elements:

            ^PTR153 = PTR:RP:StatementOfDistribution:PartnerAddressDetails:AddressDetails.LocalityName.Text
            */
            assertion = IsMatch(statementOfDistribution.PTR153, @"(^(QLD|NSW|VIC|SA|WA|NT|ACT|TAS)$)|(^(.*)( QLD | NSW | VIC | SA | WA | NT | ACT | TAS )(.*)$)|(^(QLD |NSW |VIC |SA |WA |NT |ACT |TAS )(.*)$)|(^(.*)( QLD| NSW| VIC| SA| WA| NT| ACT| TAS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000411",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain state code",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:PartnerAddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440355" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR153", Value = statementOfDistribution.PTR153 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440356
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440356(PTR2025.RP_StatementOfDistribution statementOfDistribution, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440356
            Suburb/Town/City cannot contain "UNKNOWN"

            Legacy Rule Format:
            Contains(^PTR153, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^PTR153, 'UNKNOWN')

            Data Elements:

            ^PTR153 = PTR:RP:StatementOfDistribution:PartnerAddressDetails:AddressDetails.LocalityName.Text
            */
            assertion = (statementOfDistribution.PTR153 == null ? false : statementOfDistribution.PTR153.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000409",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain ""UNKNOWN""",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:PartnerAddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440356" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR153", Value = statementOfDistribution.PTR153 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440357
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440357(PTR2025.RP_StatementOfDistribution statementOfDistribution, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440357
            Suburb/Town/City cannot contain numeric characters

            Legacy Rule Format:
            ContainsSet(^PTR153, '0-9') AND ^PTR240 = FALSE

            Technical Business Rule Format:
            ContainsSet(^PTR153, '0-9') AND ^PTR240 = FALSE

            Data Elements:

            ^PTR153 = PTR:RP:StatementOfDistribution:PartnerAddressDetails:AddressDetails.LocalityName.Text

            ^PTR240 = PTR:RP:StatementOfDistribution:PartnerAddressDetails:AddressDetails.OverseasAddress.Indicator
            */
            assertion = (IsMatch(statementOfDistribution.PTR153, @"[0-9]", RegexOptions.IgnoreCase) && statementOfDistribution.PTR240 == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000410",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain numeric characters",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:PartnerAddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440357" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR153", Value = statementOfDistribution.PTR153 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR240", Value = GetValueOrEmpty(statementOfDistribution.PTR240) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440358
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440358(PTR2025.RP_StatementOfDistribution statementOfDistribution, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440358
            Country code must be present for overseas address

            Legacy Rule Format:
            ^PTR308 = NULL AND ^PTR240 = TRUE

            Technical Business Rule Format:
            ^PTR308 = NULL AND ^PTR240 = TRUE

            Data Elements:

            ^PTR308 = PTR:RP:StatementOfDistribution:PartnerAddressDetails:AddressDetails.Country.Code

            ^PTR240 = PTR:RP:StatementOfDistribution:PartnerAddressDetails:AddressDetails.OverseasAddress.Indicator
            */
            assertion = (statementOfDistribution.PTR308 == null && statementOfDistribution.PTR240 == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410191",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Country code must be present for overseas address",
                    LongDescription = @"When the overseas address indicator is 'TRUE', the corresponding Country Code must be supplied",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:PartnerAddressDetails/tns:CountryC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440358" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR308", Value = GetValueOrEmpty(statementOfDistribution.PTR308) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR240", Value = GetValueOrEmpty(statementOfDistribution.PTR240) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440359
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440359(PTR2025.RP_StatementOfDistribution statementOfDistribution, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440359
            Overseas address indicator must not be 'TRUE' if Country Code is 'au'

            Legacy Rule Format:
            ^PTR308 = 'au' AND ^PTR240 = TRUE

            Technical Business Rule Format:
            ^PTR308 = 'au' AND ^PTR240 = TRUE

            Data Elements:

            ^PTR308 = PTR:RP:StatementOfDistribution:PartnerAddressDetails:AddressDetails.Country.Code

            ^PTR240 = PTR:RP:StatementOfDistribution:PartnerAddressDetails:AddressDetails.OverseasAddress.Indicator
            */
            assertion = (statementOfDistribution.PTR308 == @"au" && statementOfDistribution.PTR240 == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410211",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Overseas address indicator must not be 'TRUE' if Country Code is 'au'",
                    LongDescription = @"The overseas address indicator must be 'FALSE' if the Country Code is 'au' - representing Australia. If the address is an overseas address please supply the correct Country Code",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:PartnerAddressDetails/tns:CountryC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440359" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR308", Value = GetValueOrEmpty(statementOfDistribution.PTR308) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR240", Value = GetValueOrEmpty(statementOfDistribution.PTR240) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440360
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440360(PTR2025.RP_StatementOfDistribution statementOfDistribution, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440360
            If an overseas Country Code is used, the Overseas Address indicator must be 'TRUE'

            Legacy Rule Format:
            ^PTR308 <> NULL AND ^PTR308 <> 'au' AND ^PTR240 = FALSE

            Technical Business Rule Format:
            ^PTR308 <> NULL AND ^PTR308 <> 'au' AND ^PTR240 = FALSE

            Data Elements:

            ^PTR308 = PTR:RP:StatementOfDistribution:PartnerAddressDetails:AddressDetails.Country.Code

            ^PTR240 = PTR:RP:StatementOfDistribution:PartnerAddressDetails:AddressDetails.OverseasAddress.Indicator
            */
            assertion = (statementOfDistribution.PTR308 != null && statementOfDistribution.PTR308 != @"au" && statementOfDistribution.PTR240 == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410212",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"If an overseas Country Code is used, the Overseas Address indicator must be 'TRUE'",
                    LongDescription = @"If the address is an Australian address, the overseas address indicator must be set to 'FALSE' and you do not need to supply a Country Code. If the address is an overseas address, the overseas address indicator must be set to 'TRUE' and you must supply a Country Code other than 'au'",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution" + OccurrenceIndex(statementOfDistribution.OccurrenceIndex) + "/tns:PartnerAddressDetails/tns:CountryC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440360" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR308", Value = GetValueOrEmpty(statementOfDistribution.PTR308) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR240", Value = GetValueOrEmpty(statementOfDistribution.PTR240) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440361
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440361()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440361
            If an Intermediary is not present, Reporting Party Declaration Statement Type Code must be supplied and must be “TrueAndCorrect”.
    
            Legacy Rule Format:
            ^PTR291 <> 'TrueAndCorrect' AND ^PTR1060 = NULL

            Technical Business Rule Format:
            ^PTR291 <> 'TrueAndCorrect' AND ^PTR1060 = NULL
    
            Data Elements:
    
            ^PTR291 = PTR:RP:ReportingPartyDeclaration:Declaration.StatementType.Code
    
            ^PTR1060 = PTR:INT
            */
            assertion = (report.PTR291 != @"TrueAndCorrect" && report.IntCollectionExists == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438062",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Declaration statement type code is required.",
                    LongDescription = @"If an Intermediary is not present, Reporting Party Declaration Statement Type Code must be supplied and must be ""TrueAndCorrect"".",
                    Location = "/tns:PTR/tns:RP/tns:ReportingPartyDeclaration/tns:StatementTypeC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440361" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR291", Value = report.PTR291 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440362
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440362()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440362
            If an Intermediary is not present, Reporting Party declaration statement accepted indicator must be supplied and must be “True”.
    
            Legacy Rule Format:
            ^PTR290 <> TRUE AND ^PTR1060 = NULL

            Technical Business Rule Format:
            ^PTR290 <> TRUE AND ^PTR1060 = NULL
    
            Data Elements:
    
            ^PTR290 = PTR:RP:ReportingPartyDeclaration:Declaration.StatementAccepted.Indicator
    
            ^PTR1060 = PTR:INT
            */
            assertion = (report.PTR290 != true && report.IntCollectionExists == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438064",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Declaration statement accepted indicator is required.",
                    LongDescription = @"If an Intermediary is not present, Reporting Party declaration statement accepted indicator must be supplied and must be ""True"".",
                    Location = "/tns:PTR/tns:RP/tns:ReportingPartyDeclaration/tns:StatementAcceptedI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440362" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR290", Value = GetValueOrEmpty(report.PTR290) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440363
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440363()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440363
            If an Intermediary is not present, Reporting Party declaration signature date must be supplied.
    
            Legacy Rule Format:
            ^PTR289 = NULL AND ^PTR1060 = NULL

            Technical Business Rule Format:
            ^PTR289 = NULL AND ^PTR1060 = NULL
    
            Data Elements:
    
            ^PTR289 = PTR:RP:ReportingPartyDeclaration:Declaration.Signature.Date
    
            ^PTR1060 = PTR:INT
            */
            assertion = (report.PTR289 == null && report.IntCollectionExists == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438068",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Declaration signature date is required.",
                    LongDescription = @"If an Intermediary is not present, Reporting Party declaration signature date must be supplied.",
                    Location = "/tns:PTR/tns:RP/tns:ReportingPartyDeclaration/tns:SignatureD",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440363" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR289", Value = GetValueOrEmptyDate(report.PTR289) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440364
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440364()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440364
            Signature date must not be later than today
    
            Legacy Rule Format:
            ^PTR289 > Today()

            Technical Business Rule Format:
            ^PTR289 > Today()
    
            Data Elements:
    
            ^PTR289 = PTR:RP:ReportingPartyDeclaration:Declaration.Signature.Date
            */
            assertion = (report.PTR289.GetValueOrDefault() > DateTime.Now.Date);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430255",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Signature date must not be later than today",
                    Location = "/tns:PTR/tns:RP/tns:ReportingPartyDeclaration/tns:SignatureD",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440364" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR289", Value = GetValueOrEmptyDate(report.PTR289) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440365
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440365()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440365
            If an Intermediary is not present, Reporting Party declaration signatory identifier must be supplied.
    
            Legacy Rule Format:
            ^PTR292 = NULL AND ^PTR1060 = NULL

            Technical Business Rule Format:
            ^PTR292 = NULL AND ^PTR1060 = NULL
    
            Data Elements:
    
            ^PTR292 = PTR:RP:ReportingPartyDeclaration:Declaration.SignatoryIdentifier.Text
    
            ^PTR1060 = PTR:INT
            */
            assertion = (report.PTR292 == null && report.IntCollectionExists == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438066",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Declaration signatory identifier is required.",
                    LongDescription = @"If an Intermediary is not present, Reporting Party declaration signatory identifier must be supplied.",
                    Location = "/tns:PTR/tns:RP/tns:ReportingPartyDeclaration/tns:SignatoryIdentifierT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440365" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR292", Value = report.PTR292 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440366
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440366()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440366
            Name field cannot contain 'space hyphen space'.
    
            Legacy Rule Format:
            Contains(^PTR264, ' - ')

            Technical Business Rule Format:
            Contains(^PTR264, ' - ')
    
            Data Elements:
    
            ^PTR264 = PTR:RP:ReportingPartyDeclaration:PersonUnstructuredName.FullName.Text
            */
            assertion = (report.PTR264 == null ? false : report.PTR264.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.408010",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Name field cannot contain 'space hyphen space'.",
                    LongDescription = @"Name field cannot contain 'space hyphen space'.",
                    Location = "/tns:PTR/tns:RP/tns:ReportingPartyDeclaration/tns:PersonUnstructuredNameFullNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440366" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR264", Value = report.PTR264 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440367
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440367()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440367
            Invalid ABN
    
            Legacy Rule Format:
            ^PTR502 <> NULL AND FailsABNAlgorithm(^PTR502)

            Technical Business Rule Format:
            ^PTR502 <> NULL AND FailsABNAlgorithm(^PTR502)
    
            Data Elements:
    
            ^PTR502 = PTR:INT:Identifiers.AustralianBusinessNumber.Identifier
            */
            assertion = (report.PTR502 != null && FailsABNAlgorithm(report.PTR502));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid ABN",
                    Location = "/tns:PTR/tns:INT/tns:AustralianBusinessNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440367" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR502", Value = report.PTR502 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440368
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440368()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440368
            The Tax Agent number has failed the Tax Agent number algorithm check
    
            Legacy Rule Format:
            ^PTR2 <> NULL AND FailsTANAlgorithm(^PTR2)

            Technical Business Rule Format:
            ^PTR2 <> NULL AND FailsTANAlgorithm(^PTR2)
    
            Data Elements:
    
            ^PTR2 = PTR:INT:Identifiers.TaxAgentNumber.Identifier
            */
            assertion = (report.PTR2 != null && FailsTANAlgorithm(report.PTR2));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The Tax Agent number has failed the Tax Agent number algorithm check",
                    Location = "/tns:PTR/tns:INT/tns:TaxAgentNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440368" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR2", Value = GetValueOrEmpty(report.PTR2) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440369
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440369()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440369
            If an Intermediary is present, Intermediary declaration statement accepted indicator must be supplied and must be “True”.
    
            Legacy Rule Format:
            ^PTR1060 <> NULL AND ^PTR295 <> TRUE

            Technical Business Rule Format:
            ^PTR1060 <> NULL AND ^PTR295 <> TRUE
    
            Data Elements:
    
            ^PTR295 = PTR:INT:IntermediaryDeclaration:Declaration.StatementAccepted.Indicator
    
            ^PTR1060 = PTR:INT
            */
            assertion = (report.IntCollectionExists != false && report.PTR295 != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438063",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Declaration statement accepted indicator is required.",
                    LongDescription = @"If an Intermediary is present, Intermediary declaration statement accepted indicator must be supplied and must be ""True"".",
                    Location = "/tns:PTR/tns:INT/tns:IntermediaryDeclaration/tns:StatementAcceptedI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440369" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR295", Value = GetValueOrEmpty(report.PTR295) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440370
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440370()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440370
            Signature date must not be later than today
    
            Legacy Rule Format:
            ^PTR294 > Today()

            Technical Business Rule Format:
            ^PTR294 > Today()
    
            Data Elements:
    
            ^PTR294 = PTR:INT:IntermediaryDeclaration:Declaration.Signature.Date
            */
            assertion = (report.PTR294.GetValueOrDefault() > DateTime.Now.Date);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430255",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Signature date must not be later than today",
                    Location = "/tns:PTR/tns:INT/tns:IntermediaryDeclaration/tns:SignatureD",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440370" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR294", Value = GetValueOrEmptyDate(report.PTR294) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440371
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440371()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440371
            Name field cannot contain 'space hyphen space'.
    
            Legacy Rule Format:
            Contains(^PTR314, ' - ')

            Technical Business Rule Format:
            Contains(^PTR314, ' - ')
    
            Data Elements:
    
            ^PTR314 = PTR:INT:IntermediaryDeclaration:PersonUnstructuredName.FullName.Text
            */
            assertion = (report.PTR314 == null ? false : report.PTR314.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.408010",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Name field cannot contain 'space hyphen space'.",
                    LongDescription = @"Name field cannot contain 'space hyphen space'.",
                    Location = "/tns:PTR/tns:INT/tns:IntermediaryDeclaration/tns:PersonUnstructuredNameFullNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440371" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR314", Value = report.PTR314 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440372
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440372()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440372
            Name field cannot contain 'space hyphen space'.
    
            Legacy Rule Format:
            Contains(^PTR242, ' - ')

            Technical Business Rule Format:
            Contains(^PTR242, ' - ')
    
            Data Elements:
    
            ^PTR242 = PTR:INT:IntermediaryContactDetails:PersonUnstructuredName.FullName.Text
            */
            assertion = (report.PTR242 == null ? false : report.PTR242.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.408010",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Name field cannot contain 'space hyphen space'.",
                    LongDescription = @"Name field cannot contain 'space hyphen space'.",
                    Location = "/tns:PTR/tns:INT/tns:IntermediaryContactDetails/tns:PersonUnstructuredNameFullNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440372" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR242", Value = report.PTR242 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.440377
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTR440377()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.440377
            You must enter a valid amount of $0 or more in the 'Build to Rent Capital Works Deductions at 4%' when Capital Works Deductions is $250,000 or more or Business Entity Industry Code indicates a residential property-related industry
    
            Legacy Rule Format:
            ^PTR503 = NULL AND (^PTR79 >= 250000 OR FoundSet(^PTR31, '"67110","30190"'))

            Technical Business Rule Format:
            ^PTR503 = NULL AND (^PTR79 >= 250000 OR FoundSet(^PTR31, '"67110","30190"'))
    
            Data Elements:
    
            ^PTR503 = PTR:RP:Income:Rent:Expense.BuildToRentCapitalWorksDeduction.Amount
    
            ^PTR31 = PTR:RP:OrganisationDetails.OrganisationIndustry2006Extended.Code
    
            ^PTR79 = PTR:RP:Income:Rent:Expense.CapitalWorksDeduction.Amount
            */
            assertion = (report.PTR503 == null && (report.PTR79.GetValueOrDefault() >= 250000 || IsMatch(report.PTR31, @"(^(67110|30190)$)|(^(.*)( 67110 | 30190 )(.*)$)|(^(67110 |30190 )(.*)$)|(^(.*)( 67110| 30190)$)", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.440377",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Build to rent capital works deductions at 4% amount must be provided",
                    LongDescription = @"You must enter a valid amount of $0 or more in the 'Build to Rent Capital Works Deductions at 4%' when Capital Works Deductions is $250,000 or more or Business Entity Industry Code indicates a residential property-related industry",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:Rent/tns:ExpenseBuildToRentCapitalWorksDeductionA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.440377" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR503", Value = GetValueOrEmpty(report.PTR503) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR79", Value = GetValueOrEmpty(report.PTR79) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR31", Value = GetValueOrEmpty(report.PTR31) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.W00001
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTRW00001()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.W00001
            NCMI from non-primary production - Business income and excluded from NCMI from non-primary production - Business income should be included in non-PP income - Other business income.
    
            Legacy Rule Format:
            ^PTR44 < (^PTR358 + ^PTR359)

            Technical Business Rule Format:
            ^PTR44 < (^PTR358 + ^PTR359)
    
            Data Elements:
    
            ^PTR44 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Income.Other.Amount
    
            ^PTR358 = PTR:RP:NCMIBusinessIncome:NCMIBusinessIncomeNonPrimaryProduction:Income.NonConcessionalManagedInvestmentTrust.Amount
    
            ^PTR359 = PTR:RP:NCMIBusinessIncome:NCMIBusinessIncomeNonPrimaryProduction:Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount
            */
            assertion = (report.PTR44.GetValueOrDefault() < report.PTR358.GetValueOrDefault() + report.PTR359.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.W00001",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Ensure NCMI and Excluded from NCMI - business income from non-primary production are included at non-PP income - Other business income.",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:BusinessIncomeAndExpenses/tns:IncomeNonPrimaryProduction/tns:OtherA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.W00001" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR44", Value = GetValueOrEmpty(report.PTR44) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR358", Value = GetValueOrEmpty(report.PTR358) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR359", Value = GetValueOrEmpty(report.PTR359) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.W00002
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTRW00002()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.W00002
            NCMI from non-primary production - Distribution from partnerships less foreign income and excluded from NCMI from non-primary production - Distribution from partnerships less foreign income should be included in non-PP income - Distribution from partnerships less foreign income.
    
            Legacy Rule Format:
            ^PTR69 < (^PTR364 + ^PTR365)

            Technical Business Rule Format:
            ^PTR69 < (^PTR364 + ^PTR365)
    
            Data Elements:
    
            ^PTR69 = PTR:RP:Income:PartnershipsTrustsNonPrimaryProduction:Income.PartnershipDistributionLessForeignIncome.Amount
    
            ^PTR364 = PTR:RP:NCMIPartnershipsAndTrusts:NCMIPartnershipsAndTrustsNonPrimaryProduction:Income.NonConcessionalManagedInvestmentTrust.Amount
    
            ^PTR365 = PTR:RP:NCMIPartnershipsAndTrusts:NCMIPartnershipsAndTrustsNonPrimaryProduction:Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount
            */
            assertion = (report.PTR69.GetValueOrDefault() < report.PTR364.GetValueOrDefault() + report.PTR365.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.W00002",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Ensure NCMI and Excluded from NCMI - partnership income from non-primary production are included at non-PP income - Distribution from partnerships less foreign income.",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:PartnershipsTrustsNonPrimaryProduction/tns:PartnershipDistributionLessForeignIncomeA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.W00002" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR69", Value = GetValueOrEmpty(report.PTR69) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR364", Value = GetValueOrEmpty(report.PTR364) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR365", Value = GetValueOrEmpty(report.PTR365) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.W00003
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTRW00003()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.W00003
            NCMI from non-primary production - Share of net income from trusts less net capital gain, foreign income and franked distributions and excluded from NCMI from non-primary production - Share of net income from trusts less net capital gain, foreign income and franked distributions should be included in non-PP income - Share of net income from trusts less net capital gain, foreign income and franked distributions.
    
            Legacy Rule Format:
            ^PTR70 < (^PTR366 + ^PTR367)

            Technical Business Rule Format:
            ^PTR70 < (^PTR366 + ^PTR367)
    
            Data Elements:
    
            ^PTR70 = PTR:RP:Income:PartnershipsTrustsNonPrimaryProduction:Income.TrustShareNetExcludeNetCapitalGainsAndForeignIncomeAndDistributionFranked.Amount
    
            ^PTR366 = PTR:RP:NCMIPartnershipsAndTrusts:NCMIPartnershipsAndTrustsNonPrimaryProduction:NonPrimaryProductionShareOfNetIncome:Income.NonConcessionalManagedInvestmentTrust.Amount
    
            ^PTR367 = PTR:RP:NCMIPartnershipsAndTrusts:NCMIPartnershipsAndTrustsNonPrimaryProduction:NonPrimaryProductionShareOfNetIncome:Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount
            */
            assertion = (report.PTR70.GetValueOrDefault() < report.PTR366.GetValueOrDefault() + report.PTR367.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.W00003",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Ensure NCMI and Excluded from NCMI - share of net income from trusts from non-primary production  are included at non-PP income - Share of net income from trusts less net capital gains, foreign income and franked distributions.",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:PartnershipsTrustsNonPrimaryProduction/tns:TrustShareNetExcludeNetCapitalGainsAndForeignIncomeAndDistributionFrankedA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.W00003" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR70", Value = GetValueOrEmpty(report.PTR70) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR366", Value = GetValueOrEmpty(report.PTR366) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR367", Value = GetValueOrEmpty(report.PTR367) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.W00004
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTRW00004()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.W00004
            NCMI from primary production - Business income and excluded from NCMI from primary production - Business income should be included in PP income - Other business income.
    
            Legacy Rule Format:
            ^PTR43 < (^PTR356 + ^PTR357)

            Technical Business Rule Format:
            ^PTR43 < (^PTR356 + ^PTR357)
    
            Data Elements:
    
            ^PTR43 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Income.Other.Amount
    
            ^PTR356 = PTR:RP:NCMIBusinessIncome:NCMIBusinessIncomePrimaryProduction:Income.NonConcessionalManagedInvestmentTrust.Amount
    
            ^PTR357 = PTR:RP:NCMIBusinessIncome:NCMIBusinessIncomePrimaryProduction:Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount
            */
            assertion = (report.PTR43.GetValueOrDefault() < report.PTR356.GetValueOrDefault() + report.PTR357.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.W00004",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Ensure NCMI and Excluded from NCMI - business income from primary production are included at PP income - Other business income.",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:BusinessIncomeAndExpenses/tns:IncomePrimaryProduction/tns:OtherA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.W00004" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR43", Value = GetValueOrEmpty(report.PTR43) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR356", Value = GetValueOrEmpty(report.PTR356) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR357", Value = GetValueOrEmpty(report.PTR357) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.W00005
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTRW00005()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.W00005
            NCMI from primary production - Distribution from partnerships and excluded from NCMI from primary production - Distribution from partnerships should be included in PP income - Distribution from partnerships.
    
            Legacy Rule Format:
            ^PTR65 < (^PTR360 + ^PTR361)

            Technical Business Rule Format:
            ^PTR65 < (^PTR360 + ^PTR361)
    
            Data Elements:
    
            ^PTR65 = PTR:RP:Income:PartnershipsTrustsPrimaryProduction:Income.PartnershipDistributionGross.Amount
    
            ^PTR360 = PTR:RP:NCMIPartnershipsAndTrusts:NCMIPartnershipsAndTrustsPrimaryProduction:Income.NonConcessionalManagedInvestmentTrust.Amount
    
            ^PTR361 = PTR:RP:NCMIPartnershipsAndTrusts:NCMIPartnershipsAndTrustsPrimaryProduction:Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount
            */
            assertion = (report.PTR65.GetValueOrDefault() < report.PTR360.GetValueOrDefault() + report.PTR361.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.W00005",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Ensure NCMI and Excluded from NCMI - partnership income from primary production are included at PP income - Distribution from partnerships.",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:PartnershipsTrustsPrimaryProduction/tns:PartnershipDistributionGrossA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.W00005" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR65", Value = GetValueOrEmpty(report.PTR65) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR360", Value = GetValueOrEmpty(report.PTR360) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR361", Value = GetValueOrEmpty(report.PTR361) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.W00006
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTRW00006()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.W00006
            NCMI from primary production - Share of net income from trusts and excluded from NCMI from primary production - Share of net income from trusts should be included in PP income - Share of net income from trusts.
    
            Legacy Rule Format:
            ^PTR66 < (^PTR362 + ^PTR363)

            Technical Business Rule Format:
            ^PTR66 < (^PTR362 + ^PTR363)
    
            Data Elements:
    
            ^PTR66 = PTR:RP:Income:PartnershipsTrustsPrimaryProduction:Income.TrustShareNet.Amount
    
            ^PTR362 = PTR:RP:NCMIPartnershipsAndTrusts:NCMIPartnershipsAndTrustsPrimaryProduction:PrimaryProductionShareOfNetIncome:Income.NonConcessionalManagedInvestmentTrust.Amount
    
            ^PTR363 = PTR:RP:NCMIPartnershipsAndTrusts:NCMIPartnershipsAndTrustsPrimaryProduction:PrimaryProductionShareOfNetIncome:Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount
            */
            assertion = (report.PTR66.GetValueOrDefault() < report.PTR362.GetValueOrDefault() + report.PTR363.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.W00006",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Ensure NCMI and Excluded from NCMI - share of net income from trusts from primary production are included at PP income - Share of net income from trusts.",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:PartnershipsTrustsPrimaryProduction/tns:TrustShareNetA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.W00006" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR66", Value = GetValueOrEmpty(report.PTR66) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR362", Value = GetValueOrEmpty(report.PTR362) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR363", Value = GetValueOrEmpty(report.PTR363) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.W00007
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTRW00007()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.W00007
            Distribution of NCMI from primary production cannot exceed the sum of NCMI from primary production - Business income, NCMI from primary production - Distribution from partnerships and NCMI from primary production - Share of net income from trusts.
    
            Legacy Rule Format:
            Sum(^PTR368) > (^PTR356 + ^PTR360 + ^PTR362)

            Technical Business Rule Format:
            Sum(^PTR368) > (^PTR356 + ^PTR360 + ^PTR362)
    
            Data Elements:
    
            ^PTR368 = PTR:RP:StatementOfDistribution:IncomeOrLoss:ShareOfIncomePrimaryProduction:Income.NonConcessionalManagedInvestmentTrust.Amount
    
            ^PTR356 = PTR:RP:NCMIBusinessIncome:NCMIBusinessIncomePrimaryProduction:Income.NonConcessionalManagedInvestmentTrust.Amount
    
            ^PTR360 = PTR:RP:NCMIPartnershipsAndTrusts:NCMIPartnershipsAndTrustsPrimaryProduction:Income.NonConcessionalManagedInvestmentTrust.Amount
    
            ^PTR362 = PTR:RP:NCMIPartnershipsAndTrusts:NCMIPartnershipsAndTrustsPrimaryProduction:PrimaryProductionShareOfNetIncome:Income.NonConcessionalManagedInvestmentTrust.Amount
            */
            assertion = ((report.RP_StatementOfDistributionCollection == null ? 0 : report.RP_StatementOfDistributionCollection.Sum(f => f.PTR368.GetValueOrDefault())) > report.PTR356.GetValueOrDefault() + report.PTR360.GetValueOrDefault() + report.PTR362.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.W00007",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Ensure the distribution of NCMI from primary production does not exceed net NCMI from primary production.",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution/tns:IncomeOrLoss/tns:ShareOfIncomePrimaryProduction/tns:NonConcessionalManagedInvestmentTrustA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.W00007" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Sum(PTR368)", Value = report.RP_StatementOfDistributionCollection.Sum(f => f.PTR368.GetValueOrDefault()).ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR356", Value = GetValueOrEmpty(report.PTR356) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR360", Value = GetValueOrEmpty(report.PTR360) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR362", Value = GetValueOrEmpty(report.PTR362) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.W00008
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTRW00008()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.W00008
            Distribution of excluded from NCMI from primary production cannot exceed the sum of excluded from NCMI from primary production - Business income, excluded from NCMI from primary production - Distribution from partnerships and excluded from NCMI from primary production - Share of net income from trusts.
    
            Legacy Rule Format:
            Sum(^PTR369) > (^PTR357 + ^PTR361 + ^PTR363)

            Technical Business Rule Format:
            Sum(^PTR369) > (^PTR357 + ^PTR361 + ^PTR363)
    
            Data Elements:
    
            ^PTR369 = PTR:RP:StatementOfDistribution:IncomeOrLoss:ShareOfIncomePrimaryProduction:Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount
    
            ^PTR357 = PTR:RP:NCMIBusinessIncome:NCMIBusinessIncomePrimaryProduction:Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount
    
            ^PTR361 = PTR:RP:NCMIPartnershipsAndTrusts:NCMIPartnershipsAndTrustsPrimaryProduction:Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount
    
            ^PTR363 = PTR:RP:NCMIPartnershipsAndTrusts:NCMIPartnershipsAndTrustsPrimaryProduction:PrimaryProductionShareOfNetIncome:Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount
            */
            assertion = ((report.RP_StatementOfDistributionCollection == null ? 0 : report.RP_StatementOfDistributionCollection.Sum(f => f.PTR369.GetValueOrDefault())) > report.PTR357.GetValueOrDefault() + report.PTR361.GetValueOrDefault() + report.PTR363.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.W00008",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Ensure the distribution of Excluded from NCMI from primary production does not exceed net Excluded from NCMI from primary production.",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution/tns:IncomeOrLoss/tns:ShareOfIncomePrimaryProduction/tns:ExcludedFromNonConcessionalManagedInvestmentTrustA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.W00008" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Sum(PTR369)", Value = report.RP_StatementOfDistributionCollection.Sum(f => f.PTR369.GetValueOrDefault()).ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR357", Value = GetValueOrEmpty(report.PTR357) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR361", Value = GetValueOrEmpty(report.PTR361) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR363", Value = GetValueOrEmpty(report.PTR363) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.W00009
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTRW00009()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.W00009
            Distribution of NCMI from non-primary production cannot exceed the sum of NCMI from non-primary production - Business income, NCMI from non-primary production - Distribution from partnerships less foreign income and NCMI from non-primary production - Share of net income from trusts less net capital gain, foreign income and franked distributions.
    
            Legacy Rule Format:
            Sum(^PTR370) > (^PTR358 + ^PTR364 + ^PTR366)

            Technical Business Rule Format:
            Sum(^PTR370) > (^PTR358 + ^PTR364 + ^PTR366)
    
            Data Elements:
    
            ^PTR370 = PTR:RP:StatementOfDistribution:IncomeOrLoss:ShareOfIncomeNonPrimaryProduction:Income.NonConcessionalManagedInvestmentTrust.Amount
    
            ^PTR358 = PTR:RP:NCMIBusinessIncome:NCMIBusinessIncomeNonPrimaryProduction:Income.NonConcessionalManagedInvestmentTrust.Amount
    
            ^PTR364 = PTR:RP:NCMIPartnershipsAndTrusts:NCMIPartnershipsAndTrustsNonPrimaryProduction:Income.NonConcessionalManagedInvestmentTrust.Amount
    
            ^PTR366 = PTR:RP:NCMIPartnershipsAndTrusts:NCMIPartnershipsAndTrustsNonPrimaryProduction:NonPrimaryProductionShareOfNetIncome:Income.NonConcessionalManagedInvestmentTrust.Amount
            */
            assertion = ((report.RP_StatementOfDistributionCollection == null ? 0 : report.RP_StatementOfDistributionCollection.Sum(f => f.PTR370.GetValueOrDefault())) > report.PTR358.GetValueOrDefault() + report.PTR364.GetValueOrDefault() + report.PTR366.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.W00009",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Ensure the distribution of NCMI from non-primary production does not exceed net NCMI from non-primary production.",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution/tns:IncomeOrLoss/tns:ShareOfIncomeNonPrimaryProduction/tns:NonConcessionalManagedInvestmentTrustA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.W00009" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Sum(PTR370)", Value = report.RP_StatementOfDistributionCollection.Sum(f => f.PTR370.GetValueOrDefault()).ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR358", Value = GetValueOrEmpty(report.PTR358) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR364", Value = GetValueOrEmpty(report.PTR364) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR366", Value = GetValueOrEmpty(report.PTR366) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.W00010
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTRW00010()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.W00010
            Distribution of excluded from NCMI from non-primary production cannot exceed the sum of excluded from NCMI from non-primary production - Business income, excluded from NCMI from non-primary production - Distribution from partnerships less foreign income and excluded from NCMI from non-primary production - Share of net income from trusts less net capital gain, foreign income and franked distributions.
    
            Legacy Rule Format:
            Sum(^PTR371) > (^PTR359 + ^PTR365 + ^PTR367)

            Technical Business Rule Format:
            Sum(^PTR371) > (^PTR359 + ^PTR365 + ^PTR367)
    
            Data Elements:
    
            ^PTR371 = PTR:RP:StatementOfDistribution:IncomeOrLoss:ShareOfIncomeNonPrimaryProduction:Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount
    
            ^PTR359 = PTR:RP:NCMIBusinessIncome:NCMIBusinessIncomeNonPrimaryProduction:Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount
    
            ^PTR365 = PTR:RP:NCMIPartnershipsAndTrusts:NCMIPartnershipsAndTrustsNonPrimaryProduction:Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount
    
            ^PTR367 = PTR:RP:NCMIPartnershipsAndTrusts:NCMIPartnershipsAndTrustsNonPrimaryProduction:NonPrimaryProductionShareOfNetIncome:Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount
            */
            assertion = ((report.RP_StatementOfDistributionCollection == null ? 0 : report.RP_StatementOfDistributionCollection.Sum(f => f.PTR371.GetValueOrDefault())) > report.PTR359.GetValueOrDefault() + report.PTR365.GetValueOrDefault() + report.PTR367.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.W00010",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Ensure the distribution of Excluded from NCMI from non-primary production does not exceed net Excluded from NCMI from non-primary production.",
                    Location = "/tns:PTR/tns:RP/tns:StatementOfDistributionCollection/tns:StatementOfDistribution/tns:IncomeOrLoss/tns:ShareOfIncomeNonPrimaryProduction/tns:ExcludedFromNonConcessionalManagedInvestmentTrustA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.W00010" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Sum(PTR371)", Value = report.RP_StatementOfDistributionCollection.Sum(f => f.PTR371.GetValueOrDefault()).ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR359", Value = GetValueOrEmpty(report.PTR359) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR365", Value = GetValueOrEmpty(report.PTR365) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR367", Value = GetValueOrEmpty(report.PTR367) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.W00011
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTRW00011()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.W00011
            If the company has self-assessed that they are a Significant Global Entity (SGE), please confirm if they are a Country by Country Reporting Entity (CbCRE).
    
            Legacy Rule Format:
            ^PTR355 = TRUE AND ^PTR372 = NULL

            Technical Business Rule Format:
            ^PTR355 = TRUE AND ^PTR372 = NULL
    
            Data Elements:
    
            ^PTR355 = PTR:RP:OrganisationDetails.SignificantGlobalEntityStatus.Indicator
    
            ^PTR372 = PTR:RP:InternationalDealings.CountryByCountryReportingEntityStatus.Indicator
            */
            assertion = (report.PTR355 == true && report.PTR372 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.W00011",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Country by Country Reporting Entity (CBCRE) indicator should be supplied.",
                    Location = "/tns:PTR/tns:RP/tns:OrganisationDetailsSignificantGlobalEntityStatusI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.W00011" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR355", Value = GetValueOrEmpty(report.PTR355) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR372", Value = GetValueOrEmpty(report.PTR372) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.W00012
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTRW00012()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.W00012
            If Total business income is greater than zero, then Depreciation expenses should be completed and the amount should be at least 0 or higher.
    
            Legacy Rule Format:
            ^PTR54 = NULL AND (^PTR35 + ^PTR36 + ^PTR38 + ^PTR39 + ^PTR41 + ^PTR43 + ^PTR44) > 0

            Technical Business Rule Format:
            ^PTR54 = NULL AND (^PTR35 + ^PTR36 + ^PTR38 + ^PTR39 + ^PTR41 + ^PTR43 + ^PTR44) > 0
    
            Data Elements:
    
            ^PTR54 = PTR:RP:Income:BusinessIncomeAndExpenses:BusinessExpenses:Expense.DepreciationAndAmortisation.Amount
    
            ^PTR35 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^PTR36 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^PTR38 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Remuneration.PaymentToForeignResidentGross.Amount
    
            ^PTR39 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^PTR41 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^PTR43 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Income.Other.Amount
    
            ^PTR44 = PTR:RP:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Income.Other.Amount
            */
            assertion = (report.PTR54 == null && report.PTR35.GetValueOrDefault() + report.PTR36.GetValueOrDefault() + report.PTR38.GetValueOrDefault() + report.PTR39.GetValueOrDefault() + report.PTR41.GetValueOrDefault() + report.PTR43.GetValueOrDefault() + report.PTR44.GetValueOrDefault() > 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.W00050",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Depreciation expenses required",
                    LongDescription = @"If Total business income is greater than zero, then Depreciation expenses should be completed and the amount should be at least 0 or higher.",
                    Location = "/tns:PTR/tns:RP/tns:Income/tns:BusinessIncomeAndExpenses/tns:BusinessExpenses/tns:DepreciationAndAmortisationA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.W00012" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR54", Value = GetValueOrEmpty(report.PTR54) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR35", Value = GetValueOrEmpty(report.PTR35) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR36", Value = GetValueOrEmpty(report.PTR36) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR38", Value = GetValueOrEmpty(report.PTR38) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR39", Value = GetValueOrEmpty(report.PTR39) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR41", Value = GetValueOrEmpty(report.PTR41) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR43", Value = GetValueOrEmpty(report.PTR43) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR44", Value = GetValueOrEmpty(report.PTR44) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.W01014
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTRW01014()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.W01014
            If 'P' is selected at Aggregated turnover range, then the Aggregated turnover value should be more than $1 billion
    
            Legacy Rule Format:
            ^PTR412 < 1000000000 AND ^PTR411 = 'P'

            Technical Business Rule Format:
            ^PTR412 < 1000000000 AND ^PTR411 = 'P'
    
            Data Elements:
    
            ^PTR412 = PTR:RP:AggregatedTurnover:Income.AggregatedTurnover.Amount
    
            ^PTR411 = PTR:RP:AggregatedTurnover:Income.AggregatedTurnoverRange.Code
            */
            assertion = (report.PTR412.GetValueOrDefault() < 1000000000 && report.PTR411 == @"P");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.W01014",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Aggregated turnover value should be more than $1 billion",
                    LongDescription = @"If 'P' is selected at aggregated turnover range, then the aggregated turnover value should be more than $1 billion",
                    Location = "/tns:PTR/tns:RP/tns:AggregatedTurnover/tns:IncomeA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.W01014" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR412", Value = GetValueOrEmpty(report.PTR412) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR411", Value = GetValueOrEmpty(report.PTR411) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PTR.W00015
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOPTRW00015()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.PTR.W00015
            Consider if you are a Significant global entity (SGE). An entity is an SGE for a period if it is any of the following:
            - a global parent entity (GPE) with an annual global income of A$1 billion or more
            - a member of a group of entities consolidated for accounting purposes as a single group and one of the other group members is a GPE with an annual global income of A$1 billion or more
            - a member of a notional listed company group and one of the other group members is a GPE with an annual global income of A$1 billion or more.
            
            If you are an SGE, you also need to consider whether you are a country-by-country (CBC) reporting entity. CBC reporting entities must complete label G2, and may have additional reporting obligations. 
            
            For more information on SGEs and CBC reporting, visit our website at ato.gov.au.
    
            Legacy Rule Format:
            ^PTR355 = TRUE

            Technical Business Rule Format:
            ^PTR355 = TRUE
    
            Data Elements:
    
            ^PTR355 = PTR:RP:OrganisationDetails.SignificantGlobalEntityStatus.Indicator
            */
            assertion = (report.PTR355 == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PTR.W00015",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Confirm whether taxpayer is a Significant Global Entity (SGE)",
                    LongDescription = @"Consider if you are a Significant global entity (SGE). An entity is an SGE for a period if it is any of the following:
- a global parent entity (GPE) with an annual global income of A$1 billion or more
- a member of a group of entities consolidated for accounting purposes as a single group and one of the other group members is a GPE with an annual global income of A$1 billion or more
- a member of a notional listed company group and one of the other group members is a GPE with an annual global income of A$1 billion or more.

If you are an SGE, you also need to consider whether you are a country-by-country (CBC) reporting entity. CBC reporting entities must complete label G2, and may have additional reporting obligations. 

For more information on SGEs and CBC reporting, visit our website at ato.gov.au.",
                    Location = "/tns:PTR/tns:RP/tns:OrganisationDetailsSignificantGlobalEntityStatusI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PTR.W00015" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PTR355", Value = GetValueOrEmpty(report.PTR355) });

                response.Add(processMessage);
            }
        }
        #endregion

    }
}