using Ato.CD.Inbound.Shared;
using Ato.EN.IntegrationServices.CodeGenerationTRTAMI;
using Ato.EN.IntegrationServices.CodeGenerationTRTAMIS;
using DataContracts;
using System.Collections.Generic;
using System.Linq;

namespace Ato.CD.Inbound.TRTAMI202502
{
    public class CrossFormValidatorTRTAMIS : ICrossFormValidator
    {
        private TRTAMIS2025 childDocument { get; set; }

        private List<Context> contexts { get; set; }

        public List<ProcessMessageDocument> response { get; private set; }

        private TRTAMI2025 parentDocument { get; set; }

        public CrossFormValidatorTRTAMIS(TRTAMI2025 parent, object child)
        {
            childDocument = (TRTAMIS2025)child;
            contexts = childDocument.GetContexts();
            parentDocument = parent;
            response = new List<ProcessMessageDocument>();
        }

        public List<ProcessMessageDocument> ValidateCrossFormRules()
        {
            VRATOGEN402009();
            VRATOGEN402010();
            VRATOGEN438000();
            VRATOGEN438001();
            VRATOTRTAMIS000001();

            return response;
        }

        #region Validation Rules

        #region VR.ATO.GEN.402009

        /*  VR.ATO.GEN.402009
        Your supplied TFN does not match the TFN supplied on the form it was submitted with

        Legacy Rule Format:
            IF (RP:entity.identifier.TFN <> PARENT RETURN:RP:entity.identifier.TFN)
              RETURN VALIDATION MESSAGE
            ENDIF

        Technical Business Rule Format:
            IF (RP:entity.identifier.TFN <> PARENT RETURN:RP:entity.identifier.TFN)
              RETURN VALIDATION MESSAGE
            ENDIF
        */
        public void VRATOGEN402009()
        {
            bool assertion = !parentDocument.TRTAMI4.Equals(childDocument.RPIdentifierTFN);

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.402009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your supplied TFN does not match the TFN supplied on the form it was submitted with",
                    LongDescription = @"Your supplied TFN does not match the TFN supplied on the parent form it was submitted with",
                    Location = "/xbrli:xbrl/xbrli:context/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.GEN.402009" },
                        new ProcessMessageParameter { Name = "TRTAMI4", Value = TRTAMI2025Validator.GetValueOrEmpty(parentDocument.TRTAMI4) },
                        new ProcessMessageParameter { Name = "TRTAMISTFN", Value = TRTAMI2025Validator.GetValueOrEmpty(childDocument.RPIdentifierTFN) }
                    }
                });
            }
        }

        #endregion

        #region VR.ATO.GEN.402010

        /*  VR.ATO.GEN.402010
        Your supplied ABN does not match the ABN supplied on the form it was submitted with

        Legacy Rule Format:
            IF (RP:pyid.xx.xx:Identifiers.AustralianBusinessNumber.Identifier <> NULLORBLANK) AND (PARENT RETURN:RP:pyid.xx.xx:Identifiers.AustralianBusinessNumber.Identifier <> NULLORBLANK) AND (RP:pyid.xx.xx:Identifiers.AustralianBusinessNumber.Identifier <> PARENT RETURN:RP:pyid.xx.xx:Identifiers.AustralianBusinessNumber.Identifier)
              RETURN VALIDATION MESSAGE
            ENDIF

        Technical Business Rule Format:
            IF (RP:pyid.xx.xx:Identifiers.AustralianBusinessNumber.Identifier <> NULLORBLANK) AND (PARENT RETURN:RP:pyid.xx.xx:Identifiers.AustralianBusinessNumber.Identifier <> NULLORBLANK) AND (RP:pyid.xx.xx:Identifiers.AustralianBusinessNumber.Identifier <> PARENT RETURN:RP:pyid.xx.xx:Identifiers.AustralianBusinessNumber.Identifier)
              RETURN VALIDATION MESSAGE
            ENDIF
        */
        public void VRATOGEN402010()
        {
            bool assertion = !string.IsNullOrWhiteSpace(parentDocument.TRTAMI5) &&
                             !string.IsNullOrWhiteSpace(childDocument.TRTAMIS5) &&
                             !parentDocument.TRTAMI5.Equals(childDocument.TRTAMIS5);
            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.402010",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your supplied ABN does not match the ABN supplied on the form it was submitted with",
                    LongDescription = @"Your supplied ABN does not match the ABN supplied on the parent form it was submitted with",
                    Location = "/xbrli:xbrl/tns:Identifiers.AustralianBusinessNumber.Identifier",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.GEN.402010" },
                        new ProcessMessageParameter { Name = "TRTAMI5", Value = TRTAMI2025Validator.GetValueOrEmpty(parentDocument.TRTAMI5) },
                        new ProcessMessageParameter { Name = "TRTAMIS5", Value = TRTAMI2025Validator.GetValueOrEmpty(childDocument.TRTAMIS5) }
                    }
                });
            }
        }

        #endregion

        #region VR.ATO.GEN.438000

        /*  VR.ATO.GEN.438000
        Period start date within context on schedule doesn't match start date within PARENT RETURN:reporting party context

        Legacy Rule Format:
            IF (period.startDate WHERE CONTEXT(ALL)) <> PARENT RETURN:RP:period.startDate
              RETURN VALIDATION MESSAGE
            ENDIF

        Technical Business Rule Format:
            IF (period.startDate WHERE CONTEXT(ALL)) <> PARENT RETURN:RP:period.startDate
              RETURN VALIDATION MESSAGE
            ENDIF
        */
        public void VRATOGEN438000()
        {
            response.AddRange(from context in contexts
                              where context.StartDate.GetValueOrDefault() != parentDocument.TRTAMI2.GetValueOrDefault()
                              select new ProcessMessageDocument
                              {
                                  Code = "CMN.ATO.GEN.438000",
                                  Severity = ProcessMessageSeverity.Error,
                                  Description = @"The context period start date is incorrect.",
                                  LongDescription = @"Period start date within context on the schedule must match period start date within the parent return reporting party context",
                                  Location = $"/xbrli:xbrl/xbrli:context[{context.Id}]/xbrli:period/xbrli:startDate",
                                  Parameters = new ProcessMessageParameters {
                                      new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438000" },
                                      new ProcessMessageParameter { Name = "TRTAMI2", Value = TRTAMI2025Validator.GetValueOrEmptyDate(parentDocument.TRTAMI2) },
                                      new ProcessMessageParameter { Name = "ContextStartDate", Value = TRTAMI2025Validator.GetValueOrEmptyDate(context.StartDate) }
                                  }
                              });
        }

        #endregion

        #region VR.ATO.GEN.438001

        /*  VR.ATO.GEN.438001
        Period end date within context on schedule doesn't match end date within PARENT RETURN:reporting party context

        Legacy Rule Format:
            IF (period.endDate WHERE CONTEXT(ALL)) <> PARENT RETURN:RP:period.endDate
              RETURN VALIDATION MESSAGE
            ENDIF

        Technical Business Rule Format:
            IF (period.endDate WHERE CONTEXT(ALL)) <> PARENT RETURN:RP:period.endDate
              RETURN VALIDATION MESSAGE
            ENDIF
        */
        public void VRATOGEN438001()
        {
            response.AddRange(from context in contexts
                              where context.EndDate.GetValueOrDefault() != parentDocument.TRTAMI3.GetValueOrDefault()
                              select new ProcessMessageDocument
                              {
                                  Code = "CMN.ATO.GEN.438001",
                                  Severity = ProcessMessageSeverity.Error,
                                  Description = @"The context period end date is incorrect.",
                                  LongDescription = @"Period end date within context on the schedule must match period end date within the parent return reporting party context",
                                  Location = $"/xbrli:xbrl/xbrli:context[{context.Id}]/xbrli:period/xbrli:endDate",
                                  Parameters = new ProcessMessageParameters {
                                      new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438001" },
                                      new ProcessMessageParameter { Name = "TRTAMI3", Value = TRTAMI2025Validator.GetValueOrEmptyDate(parentDocument.TRTAMI3) },
                                      new ProcessMessageParameter { Name = "ContextEndDate", Value = TRTAMI2025Validator.GetValueOrEmptyDate(context.EndDate) }
                                  }
                              });
        }

        #endregion

        #region VR.ATO.TRTAMIS.000001

        /*  VR.ATO.TRTAMIS.000001
        Invalid context. Target Financial Year must match the target year of the parent form.

        Legacy Rule Format:
            IF [TRTAMIS4] <> NULL AND [TRTAMIS4] <> PARENT RETURN:pyin.xx.xx:Report.TargetFinancial.Year
              RETURN VALIDATION MESSAGE
            ENDIF

        Technical Business Rule Format:
            IF [TRTAMIS4] <> NULL AND [TRTAMIS4] <> PARENT RETURN:pyin.xx.xx:Report.TargetFinancial.Year
              RETURN VALIDATION MESSAGE
            ENDIF
        */
        public void VRATOTRTAMIS000001()
        {
            bool assertion = childDocument.TRTAMIS4 != null && childDocument.TRTAMIS4 != parentDocument.TRTAMI1;

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.TRTAMIS.000001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The Target Financial Year must match the Year shown on the parent form.",
                    LongDescription = @"The date present in the Target Financial Year must match the date that is shown in the same field in the parent form",
                    Location = "/xbrli:xbrl/tns:Report.TargetFinancial.Year",
                    Parameters = new ProcessMessageParameters {
                                      new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000001" },
                                      new ProcessMessageParameter { Name = "TRTAMI1", Value = TRTAMI2025Validator.GetValueOrEmpty(parentDocument.TRTAMI1) },
                                      new ProcessMessageParameter { Name = "TRTAMIS4", Value = TRTAMI2025Validator.GetValueOrEmpty(childDocument.TRTAMIS4) }
                                  }
                });
            }
        }

        #endregion

        #endregion
    }
}