using System;
using System.IO;
using System.Runtime.CompilerServices;
using System.Text;
using System.Xml;
using System.Linq;
using System.Collections.Generic;
using DataContracts;
using System.Text.RegularExpressions;
using Ato.EN.IntegrationServices.CodeGenerationPAYEVNTEMPRequest2020;
using Ato.EN.IntegrationServices.CodeGenerationPAYEVNTPersist2020;

namespace Ato.EN.IntegrationServices.CodeGenerationPAYEVNTRequest2020
{

    public partial class PAYEVNT2020ValidatorAdjust : PAYEVNT2020RequestValidatorBase
    {
        /// <summary>
        /// The max parameter name length is restricted by the SBR1 and SBR2 schemas - the lowest common denominator of SBR 1 is used as the default
        /// </summary>
        private int _maxParameterNameLength;

        /// <summary>
        /// The max parameter value length is restricted by the SBR1 and SBR2 schemas
        /// </summary>
        private int _maxParameterValueLength;

        /// <summary>
        /// The SBR1 and ebms3 schemas do not allow parameter names or values to be the empty string
        /// </summary>
        private string _emptyParameterValue;

        /// <summary>
        /// Initializes a new instance of the <see cref="PAYEVNT2020Validator" /> class.
        /// </summary>
        /// <param name="maxParameterNameLength">Maximum length of the parameter name.</param>
        /// <param name="maxParameterValueLength">Maximum length of the parameter value.</param>
        /// <param name="emptyParameterValue">This value will be used in place of any parameter values that result in a null or empty value.</param>
        public PAYEVNT2020ValidatorAdjust(int maxParameterNameLength = 20, int maxParameterValueLength = 4096, string emptyParameterValue = "EMPTY")
        {
            _maxParameterNameLength = maxParameterNameLength;
            _maxParameterValueLength = maxParameterValueLength;
            _emptyParameterValue = emptyParameterValue;
        }
        
        public PAYEVNT2020 ConsumedReport { get; private set; }       

        private static Dictionary<string, ProcessMessageDocument> _processMessageDocuments = new Dictionary<string, ProcessMessageDocument>();

        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public override List<ProcessMessageDocument> ValidateReport(PAYEVNT2020 report, DateTime? CreatedAt = null)
        {
            List<ProcessMessageDocument> response = new List<ProcessMessageDocument>();
           ProcessMessageDocument processMessage;
           ProcessMessageParameter parameter;
           bool assertion;

            this.ConsumedReport = report;

            this.MaximumSeverity = ProcessMessageSeverity.Information;

            // ------------------------------------------------------------------------------
            // Validations are now done in code.
            // This version supports full validation of a report - for those rules that have successfully parsed and could be generated.
            // Generated Validations include:
            //   - Validation logic in C#.
            //   - Production of errors in EBMS, SBR and SWS format.
            // 
            // The generation at this stage does not support the following - and has to be completed manually (later versions of the generation will do this for you)
            //   - Business Rules that did not parse.  The ESR team will help support you where this happens as an effort is being made to rectify these
            //   - You will get TODO tasks for those that the parser could not cope with
            // ------------------------------------------------------------------------------

            VRATOPAYEVNT000015(report, response);
            VRATOPAYEVNT000110(report, response);
            VRATOPAYEVNT000170(report, response);
            VRATOPAYEVNT000172(report, response);
            VRATOPAYEVNT000173(report, response);
            VRATOPAYEVNT000177(report, response);
            VRATOPAYEVNT000179(report, response);
            VRATOPAYEVNT000180(report, response);
            VRATOPAYEVNT000183(report, response);
            VRATOPAYEVNT000184(report, response);
            VRATOPAYEVNT000185(report, response);           
            VRATOPAYEVNT000194(report, response, CreatedAt);
            VRATOPAYEVNT000195(report, response);            
            VRATOPAYEVNT000199(report, response);  
            VRATOPAYEVNT000200(report, response);
            VRATOPAYEVNT000201(report, response);
            VRATOPAYEVNT000202(report, response);
            VRATOPAYEVNT000203(report, response);
            VRATOPAYEVNT000210(report, response);                
            VRATOPAYEVNT000212(report, response);                                          
            VRATOPAYEVNT000220(report, response);
            VRATOPAYEVNT000221(report, response);
            VRATOPAYEVNT000223(report, response);
            VRATOPAYEVNT000224(report, response);
            VRATOPAYEVNT000225(report, response);          
           
            foreach (ProcessMessageDocument currentProcessMessage in response)
            {
                if (currentProcessMessage.Parameters != null)
                {
                    foreach (ProcessMessageParameter currentParameter in currentProcessMessage.Parameters)
                    {
                        if (string.IsNullOrEmpty(currentParameter.Name))
                        {
                            currentParameter.Name = _emptyParameterValue;
                        }

                        if (currentParameter.Name.Length > _maxParameterNameLength)
                        {
                            currentParameter.Name = currentParameter.Name.Substring(0, _maxParameterNameLength - 1);
                        }

                        if (string.IsNullOrEmpty(currentParameter.Value))
                        {
                            currentParameter.Value = _emptyParameterValue;
                        }

                        if (currentParameter.Value.Length > _maxParameterValueLength)
                        {
                            currentParameter.Value = currentParameter.Value.Substring(0, _maxParameterValueLength - 1);
                        }
                    }
                }                
            }           

            return response; 
        }
  
        public override List<ProcessMessageDocument> ValidateCrossformReport(PAYEVNT2020ValidatorPersistParent report, PAYEVNTEMP2020 childReport, List<PAYEVNTEMP2020> childDocuments, HashSet<string> failedDocumentIDs, bool generateWarnings = true, string ParentDocumentPartIdentifier = "", string ChildDocumentPartIdentifier = "", bool isLastOrAll = false)
        {

            List<ProcessMessageDocument> response = new List<ProcessMessageDocument>();            

            int childCount = report.childCount;

            VRATOPAYEVNT000226(response, childCount, this, ParentDocumentPartIdentifier, failedDocumentIDs);

            foreach (ProcessMessageDocument currentProcessMessage in response)
            {
                if (currentProcessMessage.Parameters != null)
                {
                    foreach (ProcessMessageParameter currentParameter in currentProcessMessage.Parameters)
                    {
                        if (string.IsNullOrEmpty(currentParameter.Name))
                        {
                            currentParameter.Name = _emptyParameterValue;
                        }

                        if (currentParameter.Name.Length > _maxParameterNameLength)
                        {
                            currentParameter.Name = currentParameter.Name.Substring(0, _maxParameterNameLength - 1);
                        }

                        if (string.IsNullOrEmpty(currentParameter.Value))
                        {
                            currentParameter.Value = _emptyParameterValue;
                        }

                        if (currentParameter.Value.Length > _maxParameterValueLength)
                        {
                            currentParameter.Value = currentParameter.Value.Substring(0, _maxParameterValueLength - 1);
                        }
                    }
                }
            }

            return response;
        }                     
         
        #region VR.ATO.PAYEVNT.000220

        /*  VR.ATO.PAYEVNT.000220
        Payee Record Count must be equal to zero.

        Legacy Rule Format:
        ^PAYEVNT70 <> 0

        Technical Business Rule Format:
        ^PAYEVNT70 <> 0

        Data Elements:

        ^PAYEVNT70 = PAYEVNT:Rp:Payroll:Interaction.Record.Count
        */
        public static void VRATOPAYEVNT000220(PAYEVNT2020 report, List<ProcessMessageDocument> response)
        {
            bool assertion = (report.PAYEVNT70 != 0);
            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PAYEVNT.000220",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Payee Record Count must be equal to zero.",
                    LongDescription = @"Payee Record Count must be equal to zero.",
                    Location = "/tns:PAYEVNT/tns:Rp/tns:Payroll/tns:InteractionRecordCt",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNT.000220" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PAYEVNT70", Value = report.PAYEVNT70.GetValueOrDefault().ToString() });

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.PAYEVNT.000220     

        #region VR.ATO.PAYEVNT.000221

        /*  VR.ATO.PAYEVNT.000221
        Both Payer Total PAYGW Amount and Payer Total Gross Payments Amount cannot be blank or zero.

        Legacy Rule Format:
        ((^PAYEVNT20 = NULL AND ^PAYEVNT22 = NULL) OR (^PAYEVNT20 = 0 AND ^PAYEVNT22 = 0))

        Technical Business Rule Format:
        ((^PAYEVNT20 = NULL AND ^PAYEVNT22 = NULL) OR (^PAYEVNT20 = 0 AND ^PAYEVNT22 = 0))

        Data Elements:

        ^PAYEVNT20 = PAYEVNT:Rp:Payroll:IncomeTaxAndRemuneration:IncomeTax.PayAsYouGoWithholding.TaxWithheld.Amount

        ^PAYEVNT22 = PAYEVNT:Rp:Payroll:IncomeTaxAndRemuneration:Remuneration.TotalGrossPaymentsWithholding.Amount
        */
        public static void VRATOPAYEVNT000221(PAYEVNT2020 report, List<ProcessMessageDocument> response)
        {
            bool assertion = (report.PAYEVNT20 == null && report.PAYEVNT22 == null || report.PAYEVNT20 == 0 && report.PAYEVNT22 == 0);
            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PAYEVNT.000221",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Both Payer Total PAYGW Amount and Payer Total Gross Payments Amount cannot be blank or zero.",
                    LongDescription = @"Both Payer Total PAYGW Amount and Payer Total Gross Payments Amount cannot be blank or zero.",
                    Location = "/tns:PAYEVNT/tns:Rp/tns:Payroll/tns:IncomeTaxAndRemuneration/tns:PayAsYouGoWithholdingTaxWithheldA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNT.000221" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PAYEVNT20", Value = GetValueOrEmpty(report.PAYEVNT20) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PAYEVNT22", Value = GetValueOrEmpty(report.PAYEVNT22) });

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.PAYEVNT.000221

        #region VR.ATO.PAYEVNT.000223

        /*  VR.ATO.PAYEVNT.000223
        The date entered for Pay/Update Date must not be prior to the previous financial year.

        Legacy Rule Format:
        FinancialYear(^PAYEVNT69) < (CurrentFinancialYear() - 1)

        Technical Business Rule Format:
        FinancialYear(^PAYEVNT69) < (CurrentFinancialYear() - 1)

        Data Elements:

        ^PAYEVNT69 = PAYEVNT:Rp:Payroll:PaymentRecord.Transaction.Date
        */
        public static void VRATOPAYEVNT000223(PAYEVNT2020 report, List<ProcessMessageDocument> response)
        {            
            bool assertion = (FinancialYear(report.PAYEVNT69.GetValueOrDefault()) < CurrentFinancialYear() - 1);
            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PAYEVNT.000223",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The date entered for Pay/Update Date must not be prior to the previous financial year.",
                    LongDescription = @"The date entered for Pay/Update Date must not be prior to the previous financial year.",
                    Location = "/tns:PAYEVNT/tns:Rp/tns:Payroll/tns:PaymentRecordTransactionD",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNT.000223" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PAYEVNT69", Value = GetValueOrEmptyDate(report.PAYEVNT69) });

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.PAYEVNT.000223

        #region VR.ATO.PAYEVNT.000224

        /*  VR.ATO.PAYEVNT.000224
        Child support amounts must not be provided.

        Legacy Rule Format:
        ^PAYEVNT103 <> NULL OR ^PAYEVNT103 <> 0 OR ^PAYEVNT104 <> NULL OR ^PAYEVNT104 <> 0

        Technical Business Rule Format:
        ^PAYEVNT103 <> NULL OR ^PAYEVNT103 <> 0 OR ^PAYEVNT104 <> NULL OR ^PAYEVNT104 <> 0

        Data Elements:

        ^PAYEVNT103 = PAYEVNT:Rp:Payroll:IncomeTaxAndRemuneration:ChildSupport.Garnishee.Amount

        ^PAYEVNT104 = PAYEVNT:Rp:Payroll:IncomeTaxAndRemuneration:ChildSupport.Withholding.Amount
        */
        public static void VRATOPAYEVNT000224(PAYEVNT2020 report, List<ProcessMessageDocument> response)
        {           
            bool assertion = (report.PAYEVNT103 != null || report.PAYEVNT104 != null);
            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PAYEVNT.000224",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Child support amounts must not be provided.",
                    LongDescription = @"Child support amounts must not be provided.",
                    Location = "/tns:PAYEVNT/tns:Rp/tns:Payroll/tns:IncomeTaxAndRemuneration/tns:ChildSupportGarnisheeA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNT.000224" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PAYEVNT103", Value = GetValueOrEmpty(report.PAYEVNT103) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PAYEVNT104", Value = GetValueOrEmpty(report.PAYEVNT104) });

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.PAYEVNT.000224

        #region VR.ATO.PAYEVNT.000225

        /*  VR.ATO.PAYEVNT.000225
        The date entered for Pay/Update Date cannot be more than 3 months after today's date.

        Legacy Rule Format:
        ^PAYEVNT69 > AddMonthsToDate(Today(), 3)

        Technical Business Rule Format:
        ^PAYEVNT69 > AddMonthsToDate(Today(), 3)

        Data Elements:

        ^PAYEVNT69 = PAYEVNT:Rp:Payroll:PaymentRecord.Transaction.Date
        */
        public static void VRATOPAYEVNT000225(PAYEVNT2020 report, List<ProcessMessageDocument> response)
        {
            bool assertion = (report.PAYEVNT69.GetValueOrDefault() > AddMonthsToDate(DateTime.Now.Date, 3));
            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PAYEVNT.000225",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The date entered for Pay/Update Date cannot be more than 3 months after today's date.",
                    LongDescription = @"The date entered for Pay/Update Date cannot be more than 3 months after today's date.",
                    Location = "/tns:PAYEVNT/tns:Rp/tns:Payroll/tns:PaymentRecordTransactionD",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNT.000225" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PAYEVNT69", Value = GetValueOrEmptyDate(report.PAYEVNT69) });

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.PAYEVNT.000225

        #region VR.ATO.PAYEVNT.000226

        /*  VR.ATO.PAYEVNT.000226
        PAYEVNTEMP child must not be provided.

        Legacy Rule Format:
        Count(^PAYEVNTEMP) <> 0

        Technical Business Rule Format:
        Count(^PAYEVNTEMP) <> 0
        */
        public static void VRATOPAYEVNT000226(List<ProcessMessageDocument> response, int childcount, PAYEVNT2020RequestValidatorBase validator, string ParentDocumentPartIdentifier, HashSet<string> failedDocumentIDs)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (childcount != 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PAYEVNT.000226",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"PAYEVNTEMP child must not be provided.",
                    Location = null,
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNT.000226" } },
                };
                
                response.Add(processMessage);
                
            }
        }
        #endregion VR.ATO.PAYEVNT.000226

    }
} 
