using System;
using System.IO;
using System.Runtime.CompilerServices;
using System.Text;
using System.Xml;
using System.Linq;
using System.Collections.Generic;
using DataContracts;
using System.Text.RegularExpressions;
using Ato.EN.IntegrationServices.CodeGenerationPAYEVNTEMPRequest2020;

namespace Ato.EN.IntegrationServices.CodeGenerationPAYEVNTRequest2020
{

    public partial class PAYEVNT2020ValidatorSubmit : PAYEVNT2020RequestValidatorBase
    {
        /// <summary>
        /// The max parameter name length is restricted by the SBR1 and SBR2 schemas - the lowest common denominator of SBR 1 is used as the default
        /// </summary>
        private int _maxParameterNameLength;

        /// <summary>
        /// The max parameter value length is restricted by the SBR1 and SBR2 schemas
        /// </summary>
        private int _maxParameterValueLength;

        /// <summary>
        /// The SBR1 and ebms3 schemas do not allow parameter names or values to be the empty string
        /// </summary>
        private string _emptyParameterValue;

        /// <summary>
        /// Initializes a new instance of the <see cref="PAYEVNT2018ValidatorSubmit" /> class.
        /// </summary>
        /// <param name="maxParameterNameLength">Maximum length of the parameter name.</param>
        /// <param name="maxParameterValueLength">Maximum length of the parameter value.</param>
        /// <param name="emptyParameterValue">This value will be used in place of any parameter values that result in a null or empty value.</param>
        public PAYEVNT2020ValidatorSubmit(int maxParameterNameLength = 20, int maxParameterValueLength = 4096, string emptyParameterValue = "EMPTY")
        {
            _maxParameterNameLength = maxParameterNameLength;
            _maxParameterValueLength = maxParameterValueLength;
            _emptyParameterValue = emptyParameterValue;
        }

        public PAYEVNT2020 ConsumedReport { get; private set; }
        public PAYEVNTEMP2020 ConsumedChildReport { get; private set; }

        private static Dictionary<string, ProcessMessageDocument> _processMessageDocuments = new Dictionary<string, ProcessMessageDocument>();

        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public override List<ProcessMessageDocument> ValidateReport(PAYEVNT2020 report, DateTime? CreatedAt = null)
        {
            List<ProcessMessageDocument> response = new List<ProcessMessageDocument>();
            this.ConsumedReport = report;

            // ------------------------------------------------------------------------------
            // Validations are now done in code.
            // This version supports full validation of a report - for those rules that have successfully parsed and could be generated.
            // Generated Validations include:
            //   - Validation logic in C#.
            //   - Production of errors in EBMS, SBR and SWS format.
            // 
            // The generation at this stage does not support the following - and has to be completed manually (later versions of the generation will do this for you)
            //   - Business Rules that did not parse.  The ESR team will help support you where this happens as an effort is being made to rectify these
            //   - You will get TODO tasks for those that the parser could not cope with
            // ------------------------------------------------------------------------------

            VRATOPAYEVNT000015(report, response);
            VRATOPAYEVNT000110(report, response);
            VRATOPAYEVNT000170(report, response);
            VRATOPAYEVNT000172(report, response);
            VRATOPAYEVNT000173(report, response);
            VRATOPAYEVNT000177(report, response);
            VRATOPAYEVNT000179(report, response);
            VRATOPAYEVNT000180(report, response);
            VRATOPAYEVNT000183(report, response);
            VRATOPAYEVNT000184(report, response);
            VRATOPAYEVNT000185(report, response);
            VRATOPAYEVNT000199(report, response);
            VRATOPAYEVNT000200(report, response);
            VRATOPAYEVNT000201(report, response);
            VRATOPAYEVNT000202(report, response);
            VRATOPAYEVNT000203(report, response);
            VRATOPAYEVNT000210(report, response);
            VRATOPAYEVNT000212(report, response);
            VRATOPAYEVNT000213(report, response);
            VRATOPAYEVNT000215(report, response);
            VRATOPAYEVNT000217(report, response);
			VRATOPAYEVNT000219(report, response);
            VRATOPAYEVNT000194(report, response, CreatedAt);

            foreach (ProcessMessageDocument currentProcessMessage in response)
            {
                if (currentProcessMessage.Parameters != null)
                {
                    foreach (ProcessMessageParameter currentParameter in currentProcessMessage.Parameters)
                    {
                        if (string.IsNullOrEmpty(currentParameter.Name))
                        {
                            currentParameter.Name = _emptyParameterValue;
                        }

                        if (currentParameter.Name.Length > _maxParameterNameLength)
                        {
                            currentParameter.Name = currentParameter.Name.Substring(0, _maxParameterNameLength - 1);
                        }

                        if (string.IsNullOrEmpty(currentParameter.Value))
                        {
                            currentParameter.Value = _emptyParameterValue;
                        }

                        if (currentParameter.Value.Length > _maxParameterValueLength)
                        {
                            currentParameter.Value = currentParameter.Value.Substring(0, _maxParameterValueLength - 1);
                        }
                    }
                }
            }

            return response;
        }

        public override List<ProcessMessageDocument> ValidateCrossformReport(PAYEVNT2020 report, PAYEVNTEMP2020 childReport, List<PAYEVNTEMP2020> childDocuments, int childcount = 0)
        {
            List<ProcessMessageDocument> response = new List<ProcessMessageDocument>();

            this.ConsumedChildReport = childReport;

            VRATOPAYEVNT000191(report, response, childcount);
            VRATOPAYEVNT000192(response, childcount);

            VRATOPAYEVNTEMP000180(report, response, childReport);
            VRATOPAYEVNTEMP000269(report, response, childReport);
            VRATOPAYEVNTEMP000271(report, response, childReport);
            VRATOPAYEVNTEMP000282(report, response, childReport);
            VRATOPAYEVNTEMP000286(report, response, childReport);
            VRATOPAYEVNTEMP000350(report, response, childReport);
            VRATOPAYEVNTEMP000351(report, response, childReport);
            VRATOPAYEVNTEMP000354(report, response, childReport);
            VRATOPAYEVNTEMP000355(report, response, childReport);
            VRATOPAYEVNTEMP000358(report, response, childReport);
            VRATOPAYEVNTEMP000365(report, response, childReport);
            VRATOPAYEVNTEMP000366(report, response, childReport);
            VRATOPAYEVNTEMP000371(report, response, childReport);
            VRATOPAYEVNTEMP000372(report, response, childReport);
            VRATOPAYEVNTEMP000378(report, response, childReport);

            foreach (ProcessMessageDocument currentProcessMessage in response)
            {
                if (currentProcessMessage.Parameters != null)
                {
                    foreach (ProcessMessageParameter currentParameter in currentProcessMessage.Parameters)
                    {
                        if (string.IsNullOrEmpty(currentParameter.Name))
                        {
                            currentParameter.Name = _emptyParameterValue;
                        }

                        if (currentParameter.Name.Length > _maxParameterNameLength)
                        {
                            currentParameter.Name = currentParameter.Name.Substring(0, _maxParameterNameLength - 1);
                        }

                        if (string.IsNullOrEmpty(currentParameter.Value))
                        {
                            currentParameter.Value = _emptyParameterValue;
                        }

                        if (currentParameter.Value.Length > _maxParameterValueLength)
                        {
                            currentParameter.Value = currentParameter.Value.Substring(0, _maxParameterValueLength - 1);
                        }
                    }
                }
            }

            return response;
        }
		
			 #region VR.ATO.PAYEVNT.000213

        /*  VR.ATO.PAYEVNT.000213
        Child support amounts must not be provided for Pay/Update Date prior to 1st October 2020
    
        Legacy Rule Format:
        ^PAYEVNT69 < ConvertToDate(1,10, 2020) AND (^PAYEVNT103 <> NULL OR ^PAYEVNT104 <> NULL)

        Technical Business Rule Format:
        ^PAYEVNT69 < ConvertToDate(1,10, 2020) AND (^PAYEVNT103 <> NULL OR ^PAYEVNT104 <> NULL)
    
        Data Elements:
    
        ^PAYEVNT103 = PAYEVNT:Rp:Payroll:IncomeTaxAndRemuneration:ChildSupport.Garnishee.Amount
    
        ^PAYEVNT69 = PAYEVNT:Rp:Payroll:PaymentRecord.Transaction.Date
    
        ^PAYEVNT104 = PAYEVNT:Rp:Payroll:IncomeTaxAndRemuneration:ChildSupport.Withholding.Amount
        */
        public static void VRATOPAYEVNT000213(PAYEVNT2020 report, List<ProcessMessageDocument> response)
        {
            bool assertion = (report.PAYEVNT69.GetValueOrDefault() < ConvertToDate(1, 10, 2020) && (report.PAYEVNT103 != null || report.PAYEVNT104 != null));
            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PAYEVNT.000213",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Child support amounts must not be provided before 1/10/2020",
                    LongDescription = @"Child support amounts must not be provided for Pay/Update Date prior to 1st October 2020",
                    Location = "/tns:PAYEVNT/tns:Rp/tns:Payroll/tns:IncomeTaxAndRemuneration/tns:ChildSupportGarnisheeA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNT.000213" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PAYEVNT69", Value = GetValueOrEmpty(report.PAYEVNT69) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PAYEVNT103", Value = GetValueOrEmpty(report.PAYEVNT103) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PAYEVNT104", Value = GetValueOrEmpty(report.PAYEVNT104) });

                response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.PAYEVNT.000213

        #region VR.ATO.PAYEVNT.000215

        /*  VR.ATO.PAYEVNT.000215
        Run Date/Time Stamp is 350 or more days from the Pay/Update Date

        Legacy Rule Format:
        Abs(DaysInDuration(^PAYEVNT71 - ^PAYEVNT69)) >= 350

        Technical Business Rule Format:
        Abs(DaysInDuration(^PAYEVNT71 - ^PAYEVNT69)) >= 350

        Data Elements:

        ^PAYEVNT71 = PAYEVNT:Rp:Payroll:Message.Timestamp.Generation.Datetime

        ^PAYEVNT69 = PAYEVNT:Rp:Payroll:PaymentRecord.Transaction.Date
        */
        public static void VRATOPAYEVNT000215(PAYEVNT2020 report, List<ProcessMessageDocument> response)
        {
            bool assertion = (Math.Abs((report.PAYEVNT71.GetValueOrDefault() - report.PAYEVNT69.GetValueOrDefault()).Days) >= 350);
            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PAYEVNT.000215",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"The Pay/Update Date is 350 or more days from the Run Date/Time Stamp",
                    Location = "/tns:PAYEVNT/tns:Rp/tns:Payroll/tns:MessageTimestampGenerationDt",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNT.000215" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PAYEVNT71", Value = GetValueOrEmpty(report.PAYEVNT71) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PAYEVNT69", Value = GetValueOrEmpty(report.PAYEVNT69) });

                response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.PAYEVNT.000215
       
        #region VR.ATO.PAYEVNT.000217

        /*  VR.ATO.PAYEVNT.000217
        Child Support Deduction Amount must not be provided when Payer Withholding Payer Number is provided
    
        Legacy Rule Format:
        ^PAYEVNT104 <> NULL AND ^PAYEVNT3 <> NULL

        Technical Business Rule Format:
        ^PAYEVNT104 <> NULL AND ^PAYEVNT3 <> NULL
    
        Data Elements:
    
        ^PAYEVNT104 = PAYEVNT:Rp:Payroll:IncomeTaxAndRemuneration:ChildSupport.Withholding.Amount
    
        ^PAYEVNT3 = PAYEVNT:Rp:Identifiers.WithholdingPayerNumber.Identifier
        */
        public static void VRATOPAYEVNT000217(PAYEVNT2020 report, List<ProcessMessageDocument> response)
        {
            bool assertion = (report.PAYEVNT104 != null && report.PAYEVNT3 != null);
            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PAYEVNT.000217",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Child Support Deduction Amount must not be provided",
                    LongDescription = @"Child Support Deduction Amount must not be provided when Payer Withholding Payer Number is provided",
                    Location = "/tns:PAYEVNT/tns:Rp/tns:Payroll/tns:IncomeTaxAndRemuneration/tns:ChildSupportWithholdingA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNT.000217" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PAYEVNT104", Value = GetValueOrEmpty(report.PAYEVNT104) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PAYEVNT3", Value = GetValueOrEmpty(report.PAYEVNT3) });

                response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.PAYEVNT.000217
    }
}