using System;
using System.IO;
using System.Runtime.CompilerServices;
using System.Text;
using System.Xml;
using System.Linq;
using System.Collections.Generic;
using DataContracts;
using System.Text.RegularExpressions;
using Ato.EN.IntegrationServices.CodeGenerationPAYEVNTEMPRequest2020;

namespace Ato.EN.IntegrationServices.CodeGenerationPAYEVNTRequest2020
{
    public partial class PAYEVNT2020ValidatorUpdate : PAYEVNT2020RequestValidatorBase
    {
        /// <summary>
        /// The max parameter name length is restricted by the SBR1 and SBR2 schemas - the lowest common denominator of SBR 1 is used as the default
        /// </summary>
        private int _maxParameterNameLength;

        /// <summary>
        /// The max parameter value length is restricted by the SBR1 and SBR2 schemas
        /// </summary>
        private int _maxParameterValueLength;

        /// <summary>
        /// The SBR1 and ebms3 schemas do not allow parameter names or values to be the empty string
        /// </summary>
        private string _emptyParameterValue;

        /// <summary>
        /// Initializes a new instance of the <see cref="PAYEVNT2018ValidatorUpdate" /> class.
        /// </summary>
        /// <param name="maxParameterNameLength">Maximum length of the parameter name.</param>
        /// <param name="maxParameterValueLength">Maximum length of the parameter value.</param>
        /// <param name="emptyParameterValue">This value will be used in place of any parameter values that result in a null or empty value.</param>
        public PAYEVNT2020ValidatorUpdate(int maxParameterNameLength = 20, int maxParameterValueLength = 4096, string emptyParameterValue = "EMPTY")
        {
            _maxParameterNameLength = maxParameterNameLength;
            _maxParameterValueLength = maxParameterValueLength;
            _emptyParameterValue = emptyParameterValue;
        }

        public PAYEVNT2020 ConsumedReport { get; private set; }
        public PAYEVNTEMP2020 ConsumedChildReport { get; private set; }

        private static Dictionary<string, ProcessMessageDocument> _processMessageDocuments = new Dictionary<string, ProcessMessageDocument>();

        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public override List<ProcessMessageDocument> ValidateReport(PAYEVNT2020 report, DateTime? CreatedAt = null)
        {

            List<ProcessMessageDocument> response = new List<ProcessMessageDocument>();

            this.ConsumedReport = report;

            // ------------------------------------------------------------------------------
            // Validations are now done in code.
            // This version supports full validation of a report - for those rules that have successfully parsed and could be generated.
            // Generated Validations include:
            //   - Validation logic in C#.
            //   - Production of errors in EBMS, SBR and SWS format.
            // 
            // The generation at this stage does not support the following - and has to be completed manually (later versions of the generation will do this for you)
            //   - Business Rules that did not parse.  The ESR team will help support you where this happens as an effort is being made to rectify these
            //   - You will get TODO tasks for those that the parser could not cope with
            // ------------------------------------------------------------------------------

            VRATOPAYEVNT000015(report, response);
            VRATOPAYEVNT000110(report, response);
            VRATOPAYEVNT000170(report, response);
            VRATOPAYEVNT000172(report, response);
            VRATOPAYEVNT000173(report, response);
            VRATOPAYEVNT000177(report, response);
            VRATOPAYEVNT000179(report, response);
            VRATOPAYEVNT000180(report, response);
            VRATOPAYEVNT000183(report, response);
            VRATOPAYEVNT000184(report, response);
            VRATOPAYEVNT000185(report, response);
            VRATOPAYEVNT000193(report, response);
            VRATOPAYEVNT000195(report, response);
            VRATOPAYEVNT000196(report, response);
            VRATOPAYEVNT000200(report, response);
            VRATOPAYEVNT000201(report, response);
            VRATOPAYEVNT000202(report, response);
            VRATOPAYEVNT000203(report, response);
            VRATOPAYEVNT000211(report, response);
            VRATOPAYEVNT000212(report, response);
			VRATOPAYEVNT000219(report, response);
            VRATOPAYEVNT000194(report, response, CreatedAt);

            foreach (ProcessMessageDocument currentProcessMessage in response)
            {
                if (currentProcessMessage.Parameters != null)
                {
                    foreach (ProcessMessageParameter currentParameter in currentProcessMessage.Parameters)
                    {
                        if (string.IsNullOrEmpty(currentParameter.Name))
                        {
                            currentParameter.Name = _emptyParameterValue;
                        }

                        if (currentParameter.Name.Length > _maxParameterNameLength)
                        {
                            currentParameter.Name = currentParameter.Name.Substring(0, _maxParameterNameLength - 1);
                        }

                        if (string.IsNullOrEmpty(currentParameter.Value))
                        {
                            currentParameter.Value = _emptyParameterValue;
                        }

                        if (currentParameter.Value.Length > _maxParameterValueLength)
                        {
                            currentParameter.Value = currentParameter.Value.Substring(0, _maxParameterValueLength - 1);
                        }
                    }
                }
                }

            return response;
        }

        public override List<ProcessMessageDocument> ValidateCrossformReport(PAYEVNT2020 report, PAYEVNTEMP2020 childReport, List<PAYEVNTEMP2020> childDocuments, int childcount = 0)
        {
            List<ProcessMessageDocument> response = new List<ProcessMessageDocument>();

            //this.ConsumedReport = report;
            this.ConsumedChildReport = childReport;

            VRATOPAYEVNT000191(report, response, childcount);
            VRATOPAYEVNT000192(response, childcount);
            VRATOPAYEVNT000218(report, childDocuments, response);

            VRATOPAYEVNTEMP000179(report, response, childReport);
            VRATOPAYEVNTEMP000180(report, response, childReport);
            VRATOPAYEVNTEMP000238(report, response, childReport);
            VRATOPAYEVNTEMP000239(report, response, childReport);
            VRATOPAYEVNTEMP000269(report, response, childReport);
            VRATOPAYEVNTEMP000271(report, response, childReport);
            VRATOPAYEVNTEMP000280(report, response, childReport);
            VRATOPAYEVNTEMP000281(report, response, childReport);
            VRATOPAYEVNTEMP000282(report, response, childReport);
            VRATOPAYEVNTEMP000286(report, response, childReport);
            VRATOPAYEVNTEMP000287(report, response, childReport);
            VRATOPAYEVNTEMP000289(report, response, childReport);
            VRATOPAYEVNTEMP000290(report, response, childReport);
            VRATOPAYEVNTEMP000291(report, response, childReport);
            VRATOPAYEVNTEMP000350(report, response, childReport);
            VRATOPAYEVNTEMP000351(report, response, childReport);
            VRATOPAYEVNTEMP000354(report, response, childReport);
            VRATOPAYEVNTEMP000355(report, response, childReport);
            VRATOPAYEVNTEMP000358(report, response, childReport);
            VRATOPAYEVNTEMP000365(report, response, childReport);
            VRATOPAYEVNTEMP000366(report, response, childReport);
            VRATOPAYEVNTEMP000371(report, response, childReport);
            VRATOPAYEVNTEMP000372(report, response, childReport);
            VRATOPAYEVNTEMP000378(report, response, childReport);
            VRATOPAYEVNTEMP000383(report, response, childReport);

            foreach (ProcessMessageDocument currentProcessMessage in response)
            {
                if (currentProcessMessage.Parameters != null)
                {
                    foreach (ProcessMessageParameter currentParameter in currentProcessMessage.Parameters)
                    {
                        if (string.IsNullOrEmpty(currentParameter.Name))
                        {
                            currentParameter.Name = _emptyParameterValue;
                        }

                        if (currentParameter.Name.Length > _maxParameterNameLength)
                        {
                            currentParameter.Name = currentParameter.Name.Substring(0, _maxParameterNameLength - 1);
                        }

                        if (string.IsNullOrEmpty(currentParameter.Value))
                        {
                            currentParameter.Value = _emptyParameterValue;
                        }

                        if (currentParameter.Value.Length > _maxParameterValueLength)
                        {
                            currentParameter.Value = currentParameter.Value.Substring(0, _maxParameterValueLength - 1);
                        }
                    }
                }
            }

            return response;
        }       
		 #region VR.ATO.PAYEVNT.000193

        /*  VR.ATO.PAYEVNT.000193
        The Pay/Update Date cannot be later than today.

        Legacy Rule Format:
        ((Today() >= ConvertToDate(1, 7, 2017)) AND (^PAYEVNT69 > Today()))

        Technical Business Rule Format:
        ((Today() >= ConvertToDate(1, 7, 2017)) AND (^PAYEVNT69 > Today()))

        Data Elements:

        ^PAYEVNT69 = PAYEVNT:Rp:Payroll:PaymentRecord.Transaction.Date
        */
        public static void VRATOPAYEVNT000193(PAYEVNT2020 report, List<ProcessMessageDocument> response)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (DateTime.Now.Date >= ConvertToDate(1, 7, 2017) && report.PAYEVNT69.GetValueOrDefault() > DateTime.Now.Date);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PAYEVNT.000199",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The date entered for Pay/Update Date must not be later than today's date",
                    Location = "/tns:PAYEVNT/tns:Rp/tns:Payroll/tns:PaymentRecordTransactionD",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNT.000193" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PAYEVNT69", Value = GetValueOrEmpty(report.PAYEVNT69) });

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.PAYEVNT.000193

        #region VR.ATO.PAYEVNT.000196

        /*  VR.ATO.PAYEVNT.000196
        Payer Period Totals must not be provided.

        Legacy Rule Format:
        Count(^PAYEVNT79) > 0

        Technical Business Rule Format:
        Count(^PAYEVNT79) > 0

        Data Elements:

        ^PAYEVNT79 = PAYEVNT:Rp:Payroll:IncomeTaxAndRemuneration
        */
        public static void VRATOPAYEVNT000196(PAYEVNT2020 report, List<ProcessMessageDocument> response)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (Count(report.Rp_Payroll_IncomeTaxAndRemunerationCollectionCount) > 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PAYEVNT.000202",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The Payer Period Totals tuple must not be provided",
                    Location = "/tns:PAYEVNT/tns:Rp/tns:Payroll/tns:IncomeTaxAndRemuneration",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNT.000196" } },
                };

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.PAYEVNT.000196

        #region VR.ATO.PAYEVNT.000211

        /*  VR.ATO.PAYEVNT.000211
        Previous BMS Identifier can not be the same as BMS Identifier
    
        Legacy Rule Format:
        ToUppercase(Trim(^PAYEVNT101)) = ToUppercase(Trim(^PAYEVNT63))

        Technical Business Rule Format:
        ToUppercase(Trim(^PAYEVNT101)) = ToUppercase(Trim(^PAYEVNT63))
    
        Data Elements:
    
        ^PAYEVNT101 = PAYEVNT:Rp:Previous:SoftwareInformation.BusinessManagementSystem.Identifier
    
        ^PAYEVNT63 = PAYEVNT:Rp:SoftwareInformation.BusinessManagementSystem.Identifier
        */
        public static void VRATOPAYEVNT000211(PAYEVNT2020 report, List<ProcessMessageDocument> response)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = string.Equals(report.PAYEVNT101?.Trim(), report.PAYEVNT63?.Trim(), StringComparison.CurrentCultureIgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PAYEVNT.000211",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Previous BMS Identifier can not be the same as BMS Identifier",
                    Location = "/tns:PAYEVNT/tns:Rp/tns:PreviousSoftwareInformationBusinessManagementSystemId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNT.000211" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PAYEVNT101", Value = report.PAYEVNT101 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PAYEVNT63", Value = report.PAYEVNT63 });

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.PAYEVNT.000211

        #region VR.ATO.PAYEVNT.000218

        /*  VR.ATO.PAYEVNT.000218
        When Pay/Update Date is in July with more than 10 PAYEVNTEMP children and of which more than half have Final Event Indicator set to true, check if financial year is correct
    
        Technical Business Rule Format:
        (Month(^PAYEVNT69) = "July" AND COUNT(PAYEVNTEMP) > 10) AND (CountOccurrence(PAYEVNTEMP110, PAYEVNTEMP110 = "true") > Count(PAYEVNTEMP) * 0.5)
        */
        public static void VRATOPAYEVNT000218(PAYEVNT2020 report, List<PAYEVNTEMP2020> childDocuments, List<ProcessMessageDocument> response)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (report.PAYEVNT69.Value.Month == 7 && Count(childDocuments) > 10 && childDocuments.Count(child => child.PAYEVNTEMP110.Value == true) > childDocuments.Count * 0.5);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PAYEVNT.000218",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"You are finalising more than 50 percent of your records in the current financial year",
                    LongDescription = @"The Pay/Update Date is in July of the current financial year and you are finalising more than 50 percent of payees. Check you have finalised for the correct financial year",
                    Location = "/tns:PAYEVNT/tns:Rp/tns:Payroll/tns:InteractionRecordCt",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNT.000218" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PAYEVNT69", Value = report.PAYEVNT69.Value.ToString("yyyy-MM-dd") });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Count(PAYEVNTEMP)", Value = childDocuments.Count.ToString() });
                
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Count(PAYEVNTEMP110 = 'true')", Value = childDocuments.Count(child => child.PAYEVNTEMP110.Value == true).ToString() });

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.PAYEVNT.000218

        #region VR.ATO.PAYEVNTEMP.000179

        /*  VR.ATO.PAYEVNTEMP.000179
        Both the Period Start Date and the Pay/Update Date must be within the same financial year.

        Technical Business Rule Format:
        (FinancialYear(^PAYEVNTEMP29) <> FinancialYear(^PAYEVNT69))
        */
        public static void VRATOPAYEVNTEMP000179(PAYEVNT2020 report, List<ProcessMessageDocument> response, PAYEVNTEMP2020 childReport)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = false;

            if (childReport != null)
            {
                assertion = ((DateToFinancialYear(report.PAYEVNT69, 7) != (DateToFinancialYear(childReport.PAYEVNTEMP29, 7))));

                if (assertion)
                {
					{
						processMessage = new ProcessMessageDocument()
						{
							Code = "CMN.ATO.PAYEVNTEMP.000179",
							Severity = ProcessMessageSeverity.Error,
							Description = @"The Period Start Date must be within the same financial year as the Pay/Update Date",
							Location = "/tns:PAYEVNTEMP/tns:Payee/tns:PayrollPeriod/tns:StartD",
							Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNTEMP.000179" } },
						};
					
							processMessage.Parameters.Add(new ProcessMessageParameter { Name = "PAYEVNT69", Value = report.PAYEVNT69.ToString() });
							processMessage.Parameters.Add(new ProcessMessageParameter { Name = "PAYEVNTEMP29", Value = childReport.PAYEVNTEMP29.ToString() });

							response.Add(processMessage);
					}
                }

            }
        }

        #endregion

        #region VR.ATO.PAYEVNTEMP.000238

        /*  VR.ATO.PAYEVNTEMP.000238
        Payee Cessation Date must be provided when Cessation Reason Code is also provided, when the Pay/Update Date is after 30/6/2020
        
        Technical Business Rule Format:
        (^PAYEVNT69 > 30/6/2020 AND (^PAYEVNTEMP93 = NULL AND ^PAYEVNTEMP253 <> NULL))
        */
        public static void VRATOPAYEVNTEMP000238(PAYEVNT2020 report, List<ProcessMessageDocument> response, PAYEVNTEMP2020 childReport)
        {
            if (childReport != null)
            {
                bool assertion = (report.PAYEVNT69.GetValueOrDefault() > DateTime.Parse("2020-06-30") && childReport.PAYEVNTEMP93 == null && childReport.PAYEVNTEMP253 != null);
                if (assertion)
                {
                    ProcessMessageDocument processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.PAYEVNTEMP.000238",
                        Severity = ProcessMessageSeverity.Error,
                        Description = @"Payee Cessation Date must be provided",
                        LongDescription = @"Payee Cessation Date must be provided when Cessation Reason Code is also provided, when the Pay/Update Date is after 30/6/2020",
                        Location = "/tns:PAYEVNTEMP/tns:Payee/tns:EmployerConditions/tns:EmploymentEndD",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNTEMP.000238" } },
                    };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PAYEVNT69", Value = report.PAYEVNT69.GetValueOrDefault().ToString("yyyy-MM-dd") });

                    processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PAYEVNTEMP93", Value = GetValueOrEmpty(childReport.PAYEVNTEMP93) });

                    processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PAYEVNTEMP253", Value = GetValueOrEmpty(childReport.PAYEVNTEMP253) });

                    response.Add(processMessage);
                }
            }
        }
        #endregion // VR.ATO.PAYEVNTEMP.000238

        #region VR.ATO.PAYEVNTEMP.000239

        /*  VR.ATO.PAYEVNTEMP.000239
        Cessation Reason Code must be provided when Payee Cessation Date is also provided, when the Pay/Update Date is after 30/6/2020
        
        Technical Business Rule Format:
        (^PAYEVNT69 > 30/6/2020 AND (^PAYEVNTEMP253 = NULL AND ^PAYEVNTEMP93 <> NULL))
        */
        public static void VRATOPAYEVNTEMP000239(PAYEVNT2020 report, List<ProcessMessageDocument> response, PAYEVNTEMP2020 childReport)
        {
            if (childReport != null)
            {
                bool assertion = (report.PAYEVNT69.GetValueOrDefault() > DateTime.Parse("2020-06-30") && childReport.PAYEVNTEMP253 == null && childReport.PAYEVNTEMP93 != null);
                if (assertion)
                {
                    ProcessMessageDocument processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.PAYEVNTEMP.000239",
                        Severity = ProcessMessageSeverity.Error,
                        Description = @"Cessation Reason Code must be provided",
                        LongDescription = @"Cessation Reason Code must be provided when Payee Cessation Date is also provided, when the Pay/Update Date is after 30/6/2020",
                        Location = "/tns:PAYEVNTEMP/tns:Payee/tns:EmployerConditions/tns:CessationTypeC",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNTEMP.000239" } },
                    };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PAYEVNT69", Value = report.PAYEVNT69.GetValueOrDefault().ToString("yyyy-MM-dd") });

                    processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PAYEVNTEMP253", Value = GetValueOrEmpty(childReport.PAYEVNTEMP253) });

                    processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "PAYEVNTEMP93", Value = GetValueOrEmpty(childReport.PAYEVNTEMP93) });

                    response.Add(processMessage);
                }
            }
        }

        #endregion // VR.ATO.PAYEVNTEMP.000239

        #region VR.ATO.PAYEVNTEMP.000280

        /*  VR.ATO.PAYEVNTEMP.000280
        When the Pay/Update Date is after 30/6/2020, Payee Commencement Date must be provided

        Technical Business Rule Format:
        (^PAYEVNTEMP92 = NULL AND ^PAYEVNT69 > 30/6/2020)
        */
        public static void VRATOPAYEVNTEMP000280(PAYEVNT2020 report, List<ProcessMessageDocument> response, PAYEVNTEMP2020 childReport)
        {
            if (childReport != null)
            {
                bool assertion = childReport.PAYEVNTEMP92 == null && report.PAYEVNT69.GetValueOrDefault() > DateTime.Parse("2020-06-30");

                if (assertion)
                {
                    ProcessMessageDocument processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.PAYEVNTEMP.000280",
                        Severity = ProcessMessageSeverity.Error,
                        Description = @"Payee Commencement Date must be provided",
                        LongDescription = @"When the Pay/Update Date is after 30/6/2020, Payee Commencement Date must be provided",
                        Location = "/tns:PAYEVNTEMP/tns:Payee/tns:EmployerConditions/tns:EmploymentStartD",
                        Parameters = new ProcessMessageParameters() {
                            new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNTEMP.000280" },
                            new ProcessMessageParameter() { Name = "PAYEVNT69", Value = report.PAYEVNT69.GetValueOrDefault().ToString("yyyy-MM-dd") },
                            new ProcessMessageParameter() { Name = "PAYEVNTEMP92", Value = "null" }
                        }
                    };

                    response.Add(processMessage);
                }
            }
        }

        #endregion

        #region VR.ATO.PAYEVNTEMP.000281

        /*  VR.ATO.PAYEVNTEMP.000281
        When the Pay/Update Date is after 30/6/2020, Employment Basis Code must be provided

        Technical Business Rule Format:
        (^PAYEVNTEMP282 = NULL AND ^PAYEVNT69 > 30/6/2020)
        */
        public static void VRATOPAYEVNTEMP000281(PAYEVNT2020 report, List<ProcessMessageDocument> response, PAYEVNTEMP2020 childReport)
        {
            if (childReport != null)
            {
                bool assertion = childReport.PAYEVNTEMP282 == null && report.PAYEVNT69.GetValueOrDefault() > DateTime.Parse("2020-06-30");

                if (assertion)
                {
                    ProcessMessageDocument processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.PAYEVNTEMP.000281",
                        Severity = ProcessMessageSeverity.Error,
                        Description = @"Employment Basis Code must be provided",
                        LongDescription = @"When the Pay/Update Date is after 30/6/2020, Employment Basis Code must be provided",
                        Location = "/tns:PAYEVNTEMP/tns:Payee/tns:EmployerConditions/tns:PaymentBasisC",
                        Parameters = new ProcessMessageParameters() {
                            new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNTEMP.000281" },
                            new ProcessMessageParameter() { Name = "PAYEVNT69", Value = report.PAYEVNT69.GetValueOrDefault().ToString("yyyy-MM-dd") },
                            new ProcessMessageParameter() { Name = "PAYEVNTEMP282", Value = "null" }
                        }
                    };

                    response.Add(processMessage);
                }
            }
        }

        #endregion

        #region VR.ATO.PAYEVNTEMP.000287

        /*  VR.ATO.PAYEVNTEMP.000287
        When the Pay/Update Date is within the current financial year and the Tax Treatment Code is DBXXXX, a Cessation Type Code of D and an ETP Type Code of D, B, N, or T must be provided
    
        Technical Business Rule Format:
        (^PAYEVNT69 > ConvertToDate(30, 6, CurrentFinancialYear() - 1) AND ^PAYEVNTEMP254 <> NULL AND (RegexMatch(^PAYEVNTEMP254, 'DBXXXX', 'i') AND RegexMatch(^PAYEVNTEMP253, 'D', 'i') = FALSE AND CountOccurrence(^PAYEVNTEMP223, RegexMatch(^PAYEVNTEMP34, 'D|B|N|T', 'i')) = 0))
        */
        public static void VRATOPAYEVNTEMP000287(PAYEVNT2020 report, List<ProcessMessageDocument> response, PAYEVNTEMP2020 childReport)
        {
            if (childReport != null)
            {
                bool assertion = (report.PAYEVNT69.GetValueOrDefault() > ConvertToDate(30, 6, CurrentFinancialYear() - 1) && childReport.PAYEVNTEMP254 != null && RegexMatch(childReport.PAYEVNTEMP254, "DBXXXX", @"i") && childReport.PAYEVNTEMP253 != null && RegexMatch(childReport.PAYEVNTEMP253, "D", @"i") == false && childReport.Payee_PayrollPeriod_RemunerationCollection != null && childReport.Payee_PayrollPeriod_RemunerationCollection.Count(remuneration => remuneration.Payee_PayrollPeriod_Remuneration_EmploymentTerminationPaymentCollection != null && remuneration.Payee_PayrollPeriod_Remuneration_EmploymentTerminationPaymentCollection.Exists(etp => RegexMatch(etp.PAYEVNTEMP34, "[DBNT]", @"i"))) == 0);
                if (assertion)
                {
                    ProcessMessageDocument processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.PAYEVNTEMP.000287",
                        Severity = ProcessMessageSeverity.Error,
                        Description = @"The Tax Treatment code Death Benefit must have a valid Cessation Type Code and ETP code",
                        LongDescription = @"When the Pay/Update Date is within the current financial year and the Tax Treatment Code is DBXXXX, a Cessation Type Code of D and an ETP Type Code of D, B, N, or T must be provided",
                        Location = "/tns:PAYEVNTEMP/tns:Payee/tns:EmployerConditions/tns:TaxTreatmentC",
                        Parameters = new ProcessMessageParameters() {
                            new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNTEMP.000287" },
                            new ProcessMessageParameter() { Name = "PAYEVNT69", Value = report.PAYEVNT69.GetValueOrDefault().ToString("yyyy-MM-dd") },
                            new ProcessMessageParameter() { Name = "PAYEVNTEMP253", Value = GetValueOrEmpty(childReport.PAYEVNTEMP253) },
                            new ProcessMessageParameter() { Name = "PAYEVNTEMP254", Value = GetValueOrEmpty(childReport.PAYEVNTEMP254) }
                        }
                    };

                    response.Add(processMessage);
                }
            }
        }

        #endregion // VR.ATO.PAYEVNTEMP.000287

        #region VR.ATO.PAYEVNTEMP.000289

        /*  VR.ATO.PAYEVNTEMP.000289
        When the Pay/Update Date is within the current financial year and the first two characters of the Tax Treatment Code are VO or VC, an Income Stream Type Code of VOL must be provided
    
        Technical Business Rule Format:
        (^PAYEVNT69 > ConvertToDate(30, 6, CurrentFinancialYear() - 1) AND ^PAYEVNTEMP254 <> NULL AND Length(^PAYEVNTEMP254) >1 AND (RegexMatch(Substring(^PAYEVNTEMP254, 0, 2), 'VO|VC', 'i') AND CountOccurrence(^PAYEVNTEMP256, RegexMatch(^PAYEVNTEMP257, 'VOL', 'i')) = 0))
        */
        public static void VRATOPAYEVNTEMP000289(PAYEVNT2020 report, List<ProcessMessageDocument> response, PAYEVNTEMP2020 childReport)
        {
            if (childReport != null)
            {
                bool assertion = (report.PAYEVNT69.GetValueOrDefault() > ConvertToDate(30, 6, CurrentFinancialYear() - 1) && childReport.PAYEVNTEMP254 != null && childReport.PAYEVNTEMP254.Length > 1 && RegexMatch(childReport.PAYEVNTEMP254.Substring(0, 2), "VO|VC", @"i") && (childReport.Payee_PayrollPeriod_RemunerationCollection == null || childReport.Payee_PayrollPeriod_RemunerationCollection.Count(remuneration => RegexMatch(remuneration.PAYEVNTEMP257, "VOL", @"i")) == 0));
                if (assertion)
                {
                    ProcessMessageDocument processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.PAYEVNTEMP.000289",
                        Severity = ProcessMessageSeverity.Error,
                        Description = @"The Tax Treatment code Voluntary Agreement must have a valid Income Stream Type",
                        LongDescription = @"When the Pay/Update Date is within the current financial year and the first two characters of the Tax Treatment Code are VO or VC, an Income Stream Type Code of VOL must be provided",
                        Location = "/tns:PAYEVNTEMP/tns:Payee/tns:EmployerConditions/tns:TaxTreatmentC",
                        Parameters = new ProcessMessageParameters() {
                            new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNTEMP.000289" },
                            new ProcessMessageParameter() { Name = "PAYEVNT69", Value = report.PAYEVNT69.GetValueOrDefault().ToString("yyyy-MM-dd") },
                            new ProcessMessageParameter() { Name = "PAYEVNTEMP254", Value = GetValueOrEmpty(childReport.PAYEVNTEMP254) }
                        }
                    };

                    response.Add(processMessage);
                }
            }
        }

        #endregion // VR.ATO.PAYEVNTEMP.000289

        #region VR.ATO.PAYEVNTEMP.000290

        /*  VR.ATO.PAYEVNTEMP.000290
        When the Pay/Update Date is within the current financial year and the first two characters of the Tax Treatment Code are HR, HU or HF, an Income Stream Type Code of WHM must be provided
    
        Technical Business Rule Format:
        (^PAYEVNT69 > ConvertToDate(30, 6, CurrentFinancialYear() - 1) AND ^PAYEVNTEMP254 <> NULL AND Length(^PAYEVNTEMP254) >1 AND (RegexMatch(Substring(^PAYEVNTEMP254, 0, 2), 'HR|HU|HF', 'i') AND CountOccurrence(^PAYEVNTEMP256, RegexMatch(^PAYEVNTEMP257, 'WHM', 'i')) = 0))
        */
        public static void VRATOPAYEVNTEMP000290(PAYEVNT2020 report, List<ProcessMessageDocument> response, PAYEVNTEMP2020 childReport)
        {
            if (childReport != null)
            {
                bool assertion = (report.PAYEVNT69.GetValueOrDefault() > ConvertToDate(30, 6, CurrentFinancialYear() - 1) && childReport.PAYEVNTEMP254 != null && childReport.PAYEVNTEMP254.Length > 1 && RegexMatch(childReport.PAYEVNTEMP254.Substring(0, 2), "HR|HU|HF", @"i") && (childReport.Payee_PayrollPeriod_RemunerationCollection == null || childReport.Payee_PayrollPeriod_RemunerationCollection.Count(remuneration => RegexMatch(remuneration.PAYEVNTEMP257, "WHM", @"i")) == 0));
                if (assertion)
                {
                    ProcessMessageDocument processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.PAYEVNTEMP.000290",
                        Severity = ProcessMessageSeverity.Error,
                        Description = @"The Tax Treatment code Working Holiday Maker must have a valid Income Stream Type",
                        LongDescription = @"When the Pay/Update Date is within the current financial year and the first two characters of the Tax Treatment Code are HR, HU or HF, an Income Stream Type Code of WHM must be provided",
                        Location = "/tns:PAYEVNTEMP/tns:Payee/tns:EmployerConditions/tns:TaxTreatmentC",
                        Parameters = new ProcessMessageParameters() {
                            new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNTEMP.000290" },
                            new ProcessMessageParameter() { Name = "PAYEVNT69", Value = report.PAYEVNT69.GetValueOrDefault().ToString("yyyy-MM-dd") },
                            new ProcessMessageParameter() { Name = "PAYEVNTEMP254", Value = GetValueOrEmpty(childReport.PAYEVNTEMP254) }
                        }
                    };

                    response.Add(processMessage);
                }
            }
        }

        #endregion // VR.ATO.PAYEVNTEMP.000290

        #region VR.ATO.PAYEVNTEMP.000291

        /*  VR.ATO.PAYEVNTEMP.000291
        When the Pay/Update Date is within the current financial year and the first two characters of the Tax Treatment Code are WP, an Income Stream Type Code of SWP must be provided
    
        Technical Business Rule Format:
        (^PAYEVNT69 > ConvertToDate(30, 6, CurrentFinancialYear() - 1) AND ^PAYEVNTEMP254 <> NULL AND Length(^PAYEVNTEMP254) >1 AND (RegexMatch(Substring(^PAYEVNTEMP254, 0, 2), 'WP', 'i') AND CountOccurrence(^PAYEVNTEMP256, RegexMatch(^PAYEVNTEMP257, 'SWP', 'i')) = 0))
        */
        public static void VRATOPAYEVNTEMP000291(PAYEVNT2020 report, List<ProcessMessageDocument> response, PAYEVNTEMP2020 childReport)
        {
            if (childReport != null)
            {
                bool assertion = (report.PAYEVNT69.GetValueOrDefault() > ConvertToDate(30, 6, CurrentFinancialYear() - 1) && childReport.PAYEVNTEMP254 != null && childReport.PAYEVNTEMP254.Length > 1 && RegexMatch(childReport.PAYEVNTEMP254.Substring(0, 2), "WP", @"i") && (childReport.Payee_PayrollPeriod_RemunerationCollection == null || childReport.Payee_PayrollPeriod_RemunerationCollection.Count(remuneration => RegexMatch(remuneration.PAYEVNTEMP257, "SWP", @"i")) == 0));
                if (assertion)
                {
                    ProcessMessageDocument processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.PAYEVNTEMP.000291",
                        Severity = ProcessMessageSeverity.Error,
                        Description = @"The Tax Treatment code Seasonal Worker Programme must have a valid Income Stream Type",
                        LongDescription = @"When the Pay/Update Date is within the current financial year and the first two characters of the Tax Treatment Code are WP, an Income Stream Type Code of SWP must be provided",
                        Location = "/tns:PAYEVNTEMP/tns:Payee/tns:EmployerConditions/tns:TaxTreatmentC",
                        Parameters = new ProcessMessageParameters() {
                            new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNTEMP.000291" },
                            new ProcessMessageParameter() { Name = "PAYEVNT69", Value = report.PAYEVNT69.GetValueOrDefault().ToString("yyyy-MM-dd") },
                            new ProcessMessageParameter() { Name = "PAYEVNTEMP254", Value = GetValueOrEmpty(childReport.PAYEVNTEMP254) }
                        }
                    };

                    response.Add(processMessage);
                }
            }
        }

        #endregion // VR.ATO.PAYEVNTEMP.000291

        #region VR.ATO.PAYEVNTEMP.000383

        /*  VR.ATO.PAYEVNTEMP.000383
        When the Pay/Update Date is in the current financial year and Payee Total ETP PAYGW Amount is provided, Payee Termination Payment Taxable Component must be provided
    
        Technical Business Rule Format:
        FinancialYear(^PAYEVNT69) = CurrentFinancialYear() AND ^PAYEVNTEMP117 <> NULL AND ^PAYEVNTEMP51 = NULL
        */
        public static void VRATOPAYEVNTEMP000383(PAYEVNT2020 report, List<ProcessMessageDocument> response, PAYEVNTEMP2020 childReport)
        {
            if (childReport != null)
            {
                if (childReport.Payee_PayrollPeriod_RemunerationCollection != null)
                {
                    foreach (PAYEVNTEMP2020.Payee_PayrollPeriod_Remuneration remuneration in childReport.Payee_PayrollPeriod_RemunerationCollection)
                    {
                        if (remuneration.Payee_PayrollPeriod_Remuneration_EmploymentTerminationPaymentCollection != null)
                        {
                            foreach (PAYEVNTEMP2020.Payee_PayrollPeriod_Remuneration.Payee_PayrollPeriod_Remuneration_EmploymentTerminationPayment employmentTerminationPayment in remuneration.Payee_PayrollPeriod_Remuneration_EmploymentTerminationPaymentCollection)
                            {
                                bool assertion = FinancialYear(report.PAYEVNT69) == CurrentFinancialYear() && employmentTerminationPayment.PAYEVNTEMP117 != null && employmentTerminationPayment.PAYEVNTEMP51 == null;
                                if (assertion)
                                {
                                    ProcessMessageDocument processMessage = new ProcessMessageDocument()
                                    {
                                        Code = "CMN.ATO.PAYEVNTEMP.000383",
                                        Severity = ProcessMessageSeverity.Error,
                                        Description = @"Payee Termination Payment Taxable Component must be provided",
                                        LongDescription = @"When the Pay/Update Date is in the current financial year and Payee Total ETP PAYGW Amount is provided, Payee Termination Payment Taxable Component must be provided",
                                        Location = "/tns:PAYEVNTEMP/tns:Payee/tns:PayrollPeriod/tns:RemunerationCollection/tns:Remuneration" + OccurrenceIndex(remuneration.OccurrenceIndex) + "/tns:EmploymentTerminationPaymentCollection/tns:EmploymentTerminationPayment" + OccurrenceIndex(employmentTerminationPayment.OccurrenceIndex) + "/tns:IncomeTaxableA",
                                        Parameters = new ProcessMessageParameters() {
                                            new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNTEMP.000383" },
                                            new ProcessMessageParameter() { Name = "PAYEVNT69", Value = report.PAYEVNT69.GetValueOrDefault().ToString("yyyy-MM-dd") },
                                            new ProcessMessageParameter() { Name = "PAYEVNTEMP117", Value = GetValueOrEmpty(employmentTerminationPayment.PAYEVNTEMP117) },
                                            new ProcessMessageParameter() { Name = "PAYEVNTEMP51", Value = GetValueOrEmpty(employmentTerminationPayment.PAYEVNTEMP51) }
                                        }
                                    };

                                    response.Add(processMessage);
                                }
                            }
                        }
                    }
                }
            }
        }

        #endregion // VR.ATO.PAYEVNTEMP.000383
    }
}