using System;
using System.Xml;
using System.IO;
using System.Collections.Generic;
using DataContracts;
using System.Text.RegularExpressions;
using System.Reflection;
using System.Linq;
using System.Xml.Schema;
using System.Text;

namespace Ato.EN.IntegrationServices.CodeGenerationDDCTNS
{
    /// <summary>
    /// XML Consumer for DDCTNS
    /// </summary>
    public class DDCTNS2026XmlConsumer
    {
        const string MissingMandatoryMessage = "A mandatory field has not been completed.";
        const string ContactYourProviderMsg = "The message did not pass XML validation. Please contact your software provider.";
        const string InvalidDataValidation = "A field contains invalid data (such as letters in numeric or date field).";
        private const string ErrorCode1 = "CMN.ATO.GEN.XML01";
        private const string ErrorCode3 = "CMN.ATO.GEN.XML03";
        private const string ErrorCode4 = "CMN.ATO.GEN.XML04";
        private const string ErrorCode6 = "CMN.ATO.GEN.XML06";
        private bool _found = false;
        private int _parentCollectionCount;
        private int _childCollectionCount;
        private bool _isExiting = false;
        private bool _isValidationError = false;
        private string _lastPath = string.Empty;
        private ErrorMessageType _validationError = new ErrorMessageType(ErrorDescriptor.NoError);


        #region Error Messages

        public List<ProcessMessageDocument> ErrorMessages { get; set; }

        public bool HasErrors
        {
            get
            {
                return ErrorMessages != null && ErrorMessages.Count > 0;
            }
        }

        private string GetCurrentLocation()
        {
            StringBuilder location = new StringBuilder();
            string[] paths = _currentXPath.ToArray<string>();
            for (int i = paths.Length - 1; i > -1; i--)
            {
                location.Append(paths[i]);
            }
            return location.ToString();
        }

        private void MissingElementError()
        {
            ProcessMessageDocument processMessage;
            processMessage = new ProcessMessageDocument()
            {
                Code = "CMN.ATO.GEN.XML04",
                Description = MissingMandatoryMessage,
                SeverityAsString = "Error",
                Location = GetCurrentLocation(),
            };
            this.ErrorMessages.Add(processMessage);
        }

        #endregion  Error Messages

        #region Embedded Schema

        private static readonly XmlSchemaSet EmbeddedXmlSchemaSet;
        static DDCTNS2026XmlConsumer()
        {
            Assembly executingAssembly = Assembly.GetExecutingAssembly();
            string[] manifestResourceNames = executingAssembly.GetManifestResourceNames();
            string resourceName;
            Stream embeddedSchemaStream;
            XmlSchema embeddedXmlSchema;

            if (manifestResourceNames != null)
            {
                resourceName = manifestResourceNames.FirstOrDefault(rn => rn.Contains("ato.ddctns.0002.2026.01.00.xsd") && rn.EndsWith(".xsd"));
                if (resourceName != null)
                {
                    embeddedSchemaStream = executingAssembly.GetManifestResourceStream(resourceName);
                    embeddedXmlSchema = XmlSchema.Read(embeddedSchemaStream, SchemaCallback);
                    EmbeddedXmlSchemaSet = new XmlSchemaSet();
                    EmbeddedXmlSchemaSet.Add(embeddedXmlSchema);
                    EmbeddedXmlSchemaSet.Compile();
                }
            }
        }

        private static void SchemaCallback(object sender, ValidationEventArgs args)
        {
            if (args.Severity == XmlSeverityType.Error)
                throw new XmlSchemaException(args.Message);
        }

        #endregion  Embedded Schema

        #region Xml Reader Settings

        private static XmlReaderSettings ReaderSettings = new XmlReaderSettings()
        {
            CloseInput = false,
            ConformanceLevel = ConformanceLevel.Fragment,
            IgnoreWhitespace = true,
            IgnoreComments = true,
            IgnoreProcessingInstructions = true,
        };

        private XmlReaderSettings GetValidatingReaderSettings()
        {
            XmlReaderSettings validatingReaderSettings = new XmlReaderSettings()
            {
                CloseInput = false,
                ConformanceLevel = ConformanceLevel.Fragment,
                IgnoreWhitespace = true,
                IgnoreComments = true,
                IgnoreProcessingInstructions = true,
                ValidationType = ValidationType.Schema,
            };
            if (EmbeddedXmlSchemaSet == null)
            {
                throw new XmlSchemaException("Embedded Schema is Null");
            }
            else
            {
                validatingReaderSettings.Schemas.Add(EmbeddedXmlSchemaSet);
                validatingReaderSettings.ValidationEventHandler += new ValidationEventHandler(ValidationCallBack);
            }
            return validatingReaderSettings;
        }

        private static Regex DataTypeFailureExpression = new Regex("'(?<Uniqueid>.*?)' element is invalid.*value '(?<Value>.*?)' is invalid.*datatype '(?<DataType>.*?)'", RegexOptions.IgnoreCase | RegexOptions.Compiled | RegexOptions.Singleline);

        private void ValidationCallBack(object sender, ValidationEventArgs args)
        {
            var msg = args.Message;
            var msgToLower = msg.ToLower();

            ProcessMessageDocument processMessage = null;
  
            if (msgToLower.Contains("invalid according to its datatype"))
            {
                Match match = DataTypeFailureExpression.Match(args.Message);

                if (match.Success && match.Groups.Count == 4)
                {
                    var inner = args.Exception.InnerException;
                    var hint = (inner != null && !string.IsNullOrEmpty(inner.Message)) ? " Hint: " + inner.Message : string.Empty;
                    processMessage = BuildProcessMessageDocument(InvalidDataValidation, string.Empty, ErrorCode3);

                    string uniqueID = match.Groups["Uniqueid"].Value;
                    string value = match.Groups["Value"].Value;
                    string dataType = match.Groups["DataType"].Value;


                    var longDescription = string.Format(@"The value specified for an item does not match the item type (value = ""{0}"", item type = {1}, uniqueID = {2})", value, dataType, uniqueID);
                    processMessage.LongDescription = longDescription + hint;
                    processMessage.Parameters = new ProcessMessageParameters();
                    processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "uniqueID", Value = uniqueID });
                    processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "value", Value = value });
                    processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "dataType", Value = dataType });
                }
            }
            else if (msgToLower.Contains("has incomplete content"))
            {
                _validationError = GetErrorDescriptorForHasIncompleteContent(msg);
                processMessage = BuildProcessMessageDocument(MissingMandatoryMessage, msg + _validationError.Hint, _validationError.Code);
                _isValidationError = true;
            }
            else if (msgToLower.Contains("has invalid child element"))
            {
                _validationError = GetErrorDescriptorForHasInvalidChildElement(msg);
                var shortMessage = _validationError.Code == ErrorCode4 ? MissingMandatoryMessage : ContactYourProviderMsg;
                processMessage = BuildProcessMessageDocument(shortMessage, msg + _validationError.Hint, _validationError.Code);
                _isValidationError = true;
            }
            else
            {
                processMessage = BuildProcessMessageDocument(ContactYourProviderMsg, msg, ErrorCode1);
                _isValidationError = true;
            }

            this.ErrorMessages.Add(processMessage);
        }

        private ProcessMessageDocument BuildProcessMessageDocument(string shortDescription, string longDescription, string code)
        {
            return new ProcessMessageDocument()
            {
                Code = code,
                Description = shortDescription,
                LongDescription = longDescription,
                SeverityAsString = "Error",
                Location = GetCurrentLocation()
            };
        }

        private ProcessMessageDocument BuildProcessMessageDocument(ProcessMessageDocument processMessage, string newLocation)
        {
            return new ProcessMessageDocument()
            {
                Code = processMessage.Code,
                Description = processMessage.Description,
                LongDescription = processMessage.LongDescription,
                SeverityAsString = "Error",
                Location = newLocation
            };
        }

        private enum ErrorDescriptor
        {
            NoError = 0,
            MandatoryElementError,
            NonMandatoryElementError,
            UnknownElement,
            Duplicate
        }

        private struct ErrorMessageType
        {
            public string Hint, Code, ExpectedNextValidElement;
            public ErrorDescriptor Errno;

            public ErrorMessageType(ErrorDescriptor errorDescriptor)
            {
                Errno = errorDescriptor;
                Hint = "";
                Code = "";
                ExpectedNextValidElement = "";
            }

            public ErrorMessageType(ErrorDescriptor errorDescriptor, string hint, string code, string elementName)
            {
                Errno = errorDescriptor;
                Hint = hint;
                Code = code;
                ExpectedNextValidElement = elementName;
            }
        }

        private ErrorMessageType GetErrorDescriptorForHasIncompleteContent(string validationErrorMessage)
        {
            ErrorMessageType retval = new ErrorMessageType();
            if (string.IsNullOrWhiteSpace(validationErrorMessage)) return retval;

            validationErrorMessage = validationErrorMessage.Replace(" ", string.Empty);
            validationErrorMessage = validationErrorMessage.Replace("'", " ");
            var parts = validationErrorMessage.Split(' ');
            if (parts.Length != 9) return retval;

            var elementBeingProcessed = parts[1];
            var csvList = parts[5];

            retval = ParseHasIncompleteContentCVSList(csvList, elementBeingProcessed);
            return retval;
        }

        private ErrorMessageType GetErrorDescriptorForHasInvalidChildElement(string validationErrorMessage)
        {
            string msg = string.Empty;
            ErrorMessageType retval = new ErrorMessageType();
            if (string.IsNullOrWhiteSpace(validationErrorMessage)) return retval;

            validationErrorMessage = validationErrorMessage.Replace(" ", string.Empty);
            validationErrorMessage = validationErrorMessage.Replace("'", " ");
            var parts = validationErrorMessage.Split(' ');

            //decide which message format to process
            if (validationErrorMessage.ToLower().Contains("listofpossibleelements"))
            {
                if (parts.Length == 13 || parts.Length == 12 || parts.Length == 10)
                {
                    var csvList = parts[9];
                    var elementBeingProcessed = parts[1];
                    var invalidChildElementName = parts[5];
                    retval = ParseCvsListForHasInvalidChildElementLongFormat(csvList, invalidChildElementName, elementBeingProcessed);
                }

                return retval;
            }
            else
            {
                if (parts.Length != 9) return retval;
                var csvList = parts[5];
                var elementBeingProcessed = parts[1];
                var invalidChildElementName = parts[5];
                
                retval = ParseCvsListForHasInvalidChildElementLongFormat(csvList, invalidChildElementName, elementBeingProcessed);
            }
            return retval;
        }

        private ErrorMessageType ParseCvsListForHasInvalidChildElementShortFormat(string csvList, string invalidChildElementName, string elementBeingProcessed)
        {
            ErrorMessageType retval = new ErrorMessageType();

            if (!string.IsNullOrEmpty(csvList))
            {
                var lastElementInListOfPossibleElements = GetLastElementInCsvList(csvList);

                retval.Hint =
                    string.Format(
                        " Hint: while processing parent element [{0}] child element [{1}] was unexpected",
                        elementBeingProcessed, invalidChildElementName);
                retval.Code = ErrorCode1;
                retval.Errno = ErrorDescriptor.UnknownElement;
                retval.ExpectedNextValidElement = invalidChildElementName;
            }

            return retval;
        }

        private string GetLastElementInCsvList(string csvList)
        {
            var list = (csvList.Contains(','))
                ? new List<string>(csvList.Split(','))
                : new List<string> { csvList };

            string lastElement = list.Last();        
            if (lastElement.EndsWith("...."))
            {
                list.RemoveAt(list.Count-1);
            }

            return list.Last();
        }

        private ErrorMessageType ParseCvsListForHasInvalidChildElementLongFormat(string csvList, string invalidChildElementName, string elementBeingProcessed)
        {
            ErrorMessageType retval = new ErrorMessageType();

            if (!string.IsNullOrEmpty(csvList))
            {
                var lastElementInListOfPossibleElements = GetLastElementInCsvList(csvList);
                var elementList = new List<string>(_elementCsvList.Split(','));

                if (ContainsElementForElementBeingProcessed(elementList, elementBeingProcessed, invalidChildElementName))
                {
                    var indexOfElementBeingProcessed = GetIndexOfElementBeingProcessed(elementList, elementBeingProcessed);
                    var indexOfinvalidChildElementName = GetIndexOfElementUnderElementBeingProcessed(elementList, elementBeingProcessed, invalidChildElementName);
                    var indexOflastElementInListOfPossibleElements = GetIndexOfElementUnderElementBeingProcessed(elementList, elementBeingProcessed, lastElementInListOfPossibleElements);

                    if (indexOfinvalidChildElementName > indexOflastElementInListOfPossibleElements) 
                    {
                        retval.Hint =
                            string.Format(
                                " Hint: while processing parent element [{0}] child mandatory element [{1}] was expected but not found",
                                elementBeingProcessed, lastElementInListOfPossibleElements);
                        retval.Code = ErrorCode4;
                        retval.Errno = ErrorDescriptor.MandatoryElementError;
                        retval.ExpectedNextValidElement = lastElementInListOfPossibleElements;
                    }
                    else
                    {
                        retval.Hint =
                            string.Format(
                                " Hint: while processing parent element [{0}] child non mandatory element [{1}] was unexpected",
                                elementBeingProcessed, invalidChildElementName);
                        retval.Code = ErrorCode1;
                        retval.Errno = (indexOfinvalidChildElementName != indexOflastElementInListOfPossibleElements) ? ErrorDescriptor.NonMandatoryElementError : ErrorDescriptor.Duplicate;
                        retval.ExpectedNextValidElement = invalidChildElementName;
                    }
                }
                else
                {
                    //this element is unknown to the schema.
                    retval.Hint =
                        string.Format(
                            " Hint: while processing parent element [{0}] child element [{1}] was unexpected",
                            elementBeingProcessed, invalidChildElementName);
                    retval.Code = ErrorCode1;
                    retval.Errno = ErrorDescriptor.UnknownElement;
                    retval.ExpectedNextValidElement = invalidChildElementName;
                }
            }

            return retval;
        }

        private int GetIndexOfElementUnderElementBeingProcessed(List<string> validElementList,
                                                                string elementBeingProcessed,
                                                                string elementName)
        {
            int result;

            string elementWithParentNameMatch = elementBeingProcessed + "#" + elementName;
            result = validElementList.IndexOf(elementWithParentNameMatch);

            if (result < 0)
            {
                result = validElementList.IndexOf(elementName);
            }

            return result;
        }

        private int GetIndexOfElementBeingProcessed(List<string> validElementList,
                                                    string elementBeingProcessed)
        {
           return validElementList.IndexOf(elementBeingProcessed);
        }

        private bool ContainsElementForElementBeingProcessed(List<string> validElementList,
                                                             string elementBeingProcessed,
                                                             string elementName)
        {
            return validElementList.Contains(elementBeingProcessed + "#" + elementName) || validElementList.Contains(elementName);
        }

        private ErrorMessageType ParseHasIncompleteContentCVSList(string csvList, string elementBeingProcessed)
        {
            ErrorMessageType retval = new ErrorMessageType();

            if (!string.IsNullOrEmpty(csvList))
            {
                var lastElementInListOfPossibleElements = GetLastElementInCsvList(csvList);

                retval.Hint =
                    string.Format(
                        " Hint: while processing parent element [{0}] child mandatory element [{1}] was not found",
                        elementBeingProcessed, lastElementInListOfPossibleElements);

                retval.Code = ErrorCode4;
                retval.Errno = ErrorDescriptor.MandatoryElementError;
                retval.ExpectedNextValidElement = lastElementInListOfPossibleElements;
            }
            return retval;
        }

        private void StartEndDateError(DateTime startDate, DateTime endDate)
        {
            ProcessMessageDocument processMessage;

            string longDescription = string.Format("End date is earlier than start date (state date = {0}, end date = {1})", startDate.ToString("yyyy-MM-dd"), endDate.ToString("yyyy-MM-dd"));

            processMessage = new ProcessMessageDocument()
            {
                Code = "CMN.ATO.GEN.XML06",
                Description = "End date is earlier than start date.",
                LongDescription = longDescription,
                SeverityAsString = "Error",
                Location = GetCurrentLocation(),
            };
            processMessage.Parameters = new ProcessMessageParameters();
            processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "startDate", Value = startDate.ToString("yyyy-MM-dd") });
            processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "endDate", Value = endDate.ToString("yyyy-MM-dd") });
            this.ErrorMessages.Add(processMessage);
        }

        private void StartEndDateError(DateTime? startDate, DateTime? endDate)
        {
            ProcessMessageDocument processMessage;

            if (startDate == null || endDate == null)
                return;

            string longDescription = string.Format("End date is earlier than start date (state date = {0}, end date = {1})", startDate.Value.ToString("yyyy-MM-dd"), endDate.Value.ToString("yyyy-MM-dd"));

            processMessage = new ProcessMessageDocument()
            {
                Code = "CMN.ATO.GEN.XML06",
                Description = "End date is earlier than start date.",
                LongDescription = longDescription,
                SeverityAsString = "Error",
                Location = GetCurrentLocation(),
            };
            processMessage.Parameters = new ProcessMessageParameters();
            processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "startDate", Value = startDate.Value.ToString("yyyy-MM-dd") });
            processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "endDate", Value = endDate.Value.ToString("yyyy-MM-dd") });
            this.ErrorMessages.Add(processMessage);
        }

        #endregion  Xml Reader Settings

        #region IsEmptyOrNilElement
        private static bool IsEmptyOrNilElement(XmlReader reader)
        {
            bool emptyOrNil = false;
            if (reader.IsEmptyElement)
            {
                emptyOrNil = true;
            }
            else
            {
                if (reader.HasAttributes)
                {
                    string nilValue = reader.GetAttribute("nil", "http://www.w3.org/2001/XMLSchema-instance");
                    if (nilValue != null && nilValue.ToLowerInvariant() == "true" || nilValue == "1")
                    {
                        emptyOrNil = true;
                    }
                }
            }
            return emptyOrNil;
        }
        #endregion  IsEmptyOrNilElement

        #region ToBoolean
        private bool? ToBoolean(string str)
        {
            bool returnValue;
            string value = (str ?? "").Trim();
            if (value == "0" || value == "1")
                return Convert.ToBoolean(Convert.ToInt32(value));
            
            if (Boolean.TryParse(value, out returnValue))
                return returnValue;
            return null;
        }
        #endregion  ToBoolean

        #region MoveToContent
        public bool MoveToContent(XmlReader reader)
        {
            try
            {
                reader.MoveToContent();
                return true;
            }
            catch (XmlException ex)
            {
                ProcessMessageDocument processMessage;
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.XML01",
                    Description = ContactYourProviderMsg,
                    LongDescription = ex.Message,
                    SeverityAsString = "Error",
                };
                this.ErrorMessages.Add(processMessage);
                return false;
            }
        }
        #endregion  MoveToContent

        #region ReadToNextElement


        public bool ReadToNextElement(XmlReader reader, bool isMandatory) 
        {
            return ReadToNextElement(reader);
        }

        public bool ReadToNextElement(XmlReader reader) 
        {
            bool retval = false;

            if ( (reader.EOF && !_isValidationError) || _isExiting)
            {
                return false; //do nothing
            }

            if (!_isValidationError)
            {
                retval = SetReaderToNextElement(reader); //position the reader on the next valid element
            }

            if (_isValidationError)
            {
                //process validation errors
                var expectedNextElement = (!string.IsNullOrEmpty(_validationError.ExpectedNextValidElement)) ? _validationError.ExpectedNextValidElement : reader.LocalName;
                var actualNextValidElement = reader.LocalName;
                var location = GetCurrentLocation();

                if (_validationError.Errno == ErrorDescriptor.Duplicate)
                {
                    location = _lastPath;
                }

                var xpath = RemoveLeadingAndTrailingSlashes(location);

                var element = GetElementDescriptor(xpath);

                if (_validationError.Errno == ErrorDescriptor.UnknownElement)
                {
                    //set the xpath for elements that are not known to the schema
                    location = "/" + element.Path + "/" + element.Prefix + ":" + _validationError.ExpectedNextValidElement;
                    _isExiting = true;
                }
                else if(_validationError.Errno == ErrorDescriptor.NonMandatoryElementError)
                {
                    //set the xpath for the optional element that caused a validation
                    location = "/" + element.Path + "/" + element.Prefix + ":" + reader.LocalName;
                    _isExiting = true;
                }
                else
                {
                    //decide if we have spooled to the missing mandatory element. 
                    _isExiting = expectedNextElement == element.PathLeafNode;
                }

                if(_isExiting)
                {
                    //The location has changed since detection of the validation error - so update it.
                    UpdateErrorMessageWithLocation(location);
                }

                return false;
             }

             //Save previous path so we can detect duplicate elements
            _lastPath = GetCurrentLocation();

            return retval;
        }

        private bool ReadNext(XmlReader reader)
        {
            return !reader.EOF && reader.Read();
        }

        private void UpdateErrorMessageWithLocation(string newLocation)
        {
            var lastProcessMessage = ErrorMessages.Last();
            ErrorMessages.Remove(lastProcessMessage);
            var newProcessMessage = BuildProcessMessageDocument(lastProcessMessage, newLocation);
            ErrorMessages.Add(newProcessMessage);
        }

        private string RemoveLeadingAndTrailingSlashes(string xpath)
        {
            var xpathLength = xpath.Length;
            if (xpathLength <= 0) return string.Empty;
            var firstChar = xpath.Substring(0, 1);
            var lastChar = xpath.Substring(xpathLength - 1, 1);
            if (firstChar == "/")
            {
                xpath = xpath.Remove(0, 1);
                xpathLength--;
            }
            if (lastChar == "/")
            {
                xpath = xpath.Remove(xpathLength - 1, 1);
            }
            return xpath;
        }

        private struct ElementDescriptor
        {
            public string Prefix, PathLeafNode, Path;
            public ElementDescriptor(string prefix, string leaf , string path)
            {
                Prefix = prefix;
                PathLeafNode = leaf;
                Path = path;
            }
        }

        private ElementDescriptor GetElementDescriptor(string xpath)
        {
            var retval = new ElementDescriptor();

            var parts = xpath.Split('/');
            var numberOfParts = parts.Length;
            if (numberOfParts > 0)
            {
                var leafNode = parts[numberOfParts - 1];
                var p = leafNode.Split(':');
                numberOfParts = p.Length;
                if(numberOfParts == 1)
                {
                    retval.PathLeafNode = p[0];
                    retval.Prefix = string.Empty;
                }
                else if(numberOfParts == 2)
                {
                    retval.PathLeafNode = p[1];
                    retval.Prefix = p[0];
                }
                var leafNodeLength = leafNode.Length;
                var path = xpath.Remove(xpath.Length - leafNodeLength, leafNodeLength);
                retval.Path = RemoveLeadingAndTrailingSlashes(path);
            }
            return retval;
        }

        private bool SetReaderToNextElement(XmlReader reader) 
        {
            try
            {
                if (reader.EOF || _isExiting) return false;
                if (_found)
                {
                    reader.Read();
                    _found = false;
                }
                while (!reader.EOF && reader.NodeType != XmlNodeType.Element)
                {
                    reader.Read();
                }
      
                return reader.NodeType == XmlNodeType.Element && !reader.EOF;

            }
            catch (XmlException ex)
            {
                ProcessMessageDocument processMessage;
                processMessage = new ProcessMessageDocument()
                {
                    Code = ErrorCode1,
                    Description = ContactYourProviderMsg,
                    LongDescription = ex.Message,
                    SeverityAsString = "Error",
                };
                this.ErrorMessages.Add(processMessage);
                return false;
            }
        }

        #endregion  ReadToNextElement

        private Stack<string> _currentXPath = new Stack<string>();

        public DDCTNS2026 Consume(Stream streamToLoad, bool validateDataTypes = false)
        {
            DDCTNS2026 report = new DDCTNS2026();

            // Working Variable for if can still read from the xml stream
            bool reading;

            // Working Variables for current values
            string currentValue;
            DateTime currentDateTimeValue;
            bool? currentBooleanValue;
            decimal currentDecimalValue;
            double currentDoubleValue;
            float currentFloatValue;
            sbyte currentsByteValue;
            byte currentByteValue;
            short currentShortValue;
            ushort currentuShortValue;
            uint currentuIntValue;
            int currentIntValue;
            long currentLongValue;
            ulong currentuLongValue;

            this.ErrorMessages = new List<ProcessMessageDocument>();
            _currentXPath.Push("/tns:DDCTNS");

            if (streamToLoad == null || streamToLoad.Length == 0 || !streamToLoad.CanRead)
            {
                MissingElementError();
                return report;
            }

            streamToLoad.Position = 0;
            XmlReader reader;

            if (validateDataTypes)
                reader = XmlReader.Create(streamToLoad, GetValidatingReaderSettings());
            else
                reader = XmlReader.Create(streamToLoad, ReaderSettings);

            if (!MoveToContent(reader))
                return report;

            reading = !reader.EOF;

            if (ReadToNextElement(reader) && reader.LocalName == "DDCTNS" && reader.NamespaceURI == "http://www.sbr.gov.au/ato/ddctns")
            {
                _found = true;
                ReadToNextElement(reader);
            }
            else
            {
                MissingElementError();
                return report;
            }

    
            #region Rp
            _currentXPath.Push("/tns:Rp");
            //3. use case
            if (ReadToNextElement(reader,true) && reader.LocalName == "Rp" && reader.Depth == _currentXPath.Count - 1)
            {
                report.RpCollectionExists = true;
                report.RpCollectionCount += 1;
                _found = true; 
        
                #region LodgmentPeriodStartD
                _currentXPath.Push("/tns:LodgmentPeriodStartD");
                //6. use case
                if (ReadToNextElement(reader,true) && reader.LocalName == "LodgmentPeriodStartD" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                            {
                                report.DDCTNS300 = currentDateTimeValue.Date;
                            }
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion LodgmentPeriodStartD
        
                #region LodgmentPeriodEndD
                _currentXPath.Push("/tns:LodgmentPeriodEndD");
                //6. use case
                if (ReadToNextElement(reader,true) && reader.LocalName == "LodgmentPeriodEndD" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                            {
                                report.DDCTNS301 = currentDateTimeValue.Date;
                            }
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion LodgmentPeriodEndD
        
                #region TaxFileNumberId
                _currentXPath.Push("/tns:TaxFileNumberId");
                //6. use case
                if (ReadToNextElement(reader,true) && reader.LocalName == "TaxFileNumberId" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.DDCTNS302 = currentValue;
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion TaxFileNumberId
        
                #region Car
        
                _currentXPath.Push("/tns:CarCollection");
                // 4. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "CarCollection" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.Rp_CarCollection = new List<DDCTNS2026.Rp_Car>();
                    report.Rp_CarCollectionExists = true;
        
                    _found = true;
        
                    while ( ReadToNextElement(reader,false) && reader.LocalName == "Car" ) 
                    {
                        _found = true;
        
                        DDCTNS2026.Rp_Car car = new DDCTNS2026.Rp_Car();
                        report.Rp_CarCollection.Add(car);
                        report.Rp_CarCollectionCount += 1;
                        car.OccurrenceIndex = report.Rp_CarCollectionCount;
        
                        _currentXPath.Push("/tns:Car[" + report.Rp_CarCollectionCount + "]");
                
                        #region AssetsMakeModelDe
                        _currentXPath.Push("/tns:AssetsMakeModelDe");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "AssetsMakeModelDe" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    car.DDCTNS101 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion AssetsMakeModelDe
                
                        #region IncomeTaxDeductionCalculationMethodC
                        _currentXPath.Push("/tns:IncomeTaxDeductionCalculationMethodC");
                        //6. use case
                        if (ReadToNextElement(reader,true) && reader.LocalName == "IncomeTaxDeductionCalculationMethodC" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    car.DDCTNS102 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion IncomeTaxDeductionCalculationMethodC
                
                        #region IncomeTaxDeductionBusinessKilometresN
                        _currentXPath.Push("/tns:IncomeTaxDeductionBusinessKilometresN");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "IncomeTaxDeductionBusinessKilometresN" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        car.DDCTNS103 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion IncomeTaxDeductionBusinessKilometresN
                
                        #region IncomeTaxDeductionLogBookMethodBusinessUseRecordedP
                        _currentXPath.Push("/tns:IncomeTaxDeductionLogBookMethodBusinessUseRecordedP");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "IncomeTaxDeductionLogBookMethodBusinessUseRecordedP" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        car.DDCTNS104 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion IncomeTaxDeductionLogBookMethodBusinessUseRecordedP
                
                        #region IncomeTaxDeductionLogBookMethodA
                        _currentXPath.Push("/tns:IncomeTaxDeductionLogBookMethodA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "IncomeTaxDeductionLogBookMethodA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        car.DDCTNS105 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion IncomeTaxDeductionLogBookMethodA
                
                        #region IncomeTaxDeductionLogBookMethodDepreciationA
                        _currentXPath.Push("/tns:IncomeTaxDeductionLogBookMethodDepreciationA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "IncomeTaxDeductionLogBookMethodDepreciationA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        car.DDCTNS106 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion IncomeTaxDeductionLogBookMethodDepreciationA
                    _currentXPath.Pop();
                    } // End of while loop for Car
                } // End of if CarCollection node exists
        
                _currentXPath.Pop();
                #endregion Car
        
                #region Travel
        
                _currentXPath.Push("/tns:TravelCollection");
                // 4. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "TravelCollection" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.Rp_TravelCollection = new List<DDCTNS2026.Rp_Travel>();
                    report.Rp_TravelCollectionExists = true;
        
                    _found = true;
        
                    while ( ReadToNextElement(reader,false) && reader.LocalName == "Travel" ) 
                    {
                        _found = true;
        
                        DDCTNS2026.Rp_Travel travel = new DDCTNS2026.Rp_Travel();
                        report.Rp_TravelCollection.Add(travel);
                        report.Rp_TravelCollectionCount += 1;
                        travel.OccurrenceIndex = report.Rp_TravelCollectionCount;
        
                        _currentXPath.Push("/tns:Travel[" + report.Rp_TravelCollectionCount + "]");
                
                        #region IncomeTaxDeductionReasonDe
                        _currentXPath.Push("/tns:IncomeTaxDeductionReasonDe");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "IncomeTaxDeductionReasonDe" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    travel.DDCTNS409 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion IncomeTaxDeductionReasonDe
                
                        #region IncomeTaxDeductionA
                        _currentXPath.Push("/tns:IncomeTaxDeductionA");
                        //6. use case
                        if (ReadToNextElement(reader,true) && reader.LocalName == "IncomeTaxDeductionA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        travel.DDCTNS410 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion IncomeTaxDeductionA
                    _currentXPath.Pop();
                    } // End of while loop for Travel
                } // End of if TravelCollection node exists
        
                _currentXPath.Pop();
                #endregion Travel
        
                #region Clothing
        
                _currentXPath.Push("/tns:ClothingCollection");
                // 4. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "ClothingCollection" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.Rp_ClothingCollection = new List<DDCTNS2026.Rp_Clothing>();
                    report.Rp_ClothingCollectionExists = true;
        
                    _found = true;
        
                    while ( ReadToNextElement(reader,false) && reader.LocalName == "Clothing" ) 
                    {
                        _found = true;
        
                        DDCTNS2026.Rp_Clothing clothing = new DDCTNS2026.Rp_Clothing();
                        report.Rp_ClothingCollection.Add(clothing);
                        report.Rp_ClothingCollectionCount += 1;
                        clothing.OccurrenceIndex = report.Rp_ClothingCollectionCount;
        
                        _currentXPath.Push("/tns:Clothing[" + report.Rp_ClothingCollectionCount + "]");
                
                        #region IncomeTaxDeductionTypeC
                        _currentXPath.Push("/tns:IncomeTaxDeductionTypeC");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "IncomeTaxDeductionTypeC" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    clothing.DDCTNS415 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion IncomeTaxDeductionTypeC
                
                        #region IncomeTaxDeductionA
                        _currentXPath.Push("/tns:IncomeTaxDeductionA");
                        //6. use case
                        if (ReadToNextElement(reader,true) && reader.LocalName == "IncomeTaxDeductionA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        clothing.DDCTNS416 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion IncomeTaxDeductionA
                    _currentXPath.Pop();
                    } // End of while loop for Clothing
                } // End of if ClothingCollection node exists
        
                _currentXPath.Pop();
                #endregion Clothing
        
                #region IncomeTaxDeductionEducationCircumstancesC
                _currentXPath.Push("/tns:IncomeTaxDeductionEducationCircumstancesC");
                //6. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "IncomeTaxDeductionEducationCircumstancesC" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.DDCTNS200 = currentValue;
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion IncomeTaxDeductionEducationCircumstancesC
        
                #region SelfEducation
        
                _currentXPath.Push("/tns:SelfEducationCollection");
                // 4. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "SelfEducationCollection" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.Rp_SelfEducationCollection = new List<DDCTNS2026.Rp_SelfEducation>();
                    report.Rp_SelfEducationCollectionExists = true;
        
                    _found = true;
        
                    while ( ReadToNextElement(reader,false) && reader.LocalName == "SelfEducation" ) 
                    {
                        _found = true;
        
                        DDCTNS2026.Rp_SelfEducation selfEducation = new DDCTNS2026.Rp_SelfEducation();
                        report.Rp_SelfEducationCollection.Add(selfEducation);
                        report.Rp_SelfEducationCollectionCount += 1;
                        selfEducation.OccurrenceIndex = report.Rp_SelfEducationCollectionCount;
        
                        _currentXPath.Push("/tns:SelfEducation[" + report.Rp_SelfEducationCollectionCount + "]");
                
                        #region IncomeTaxDeductionC
                        _currentXPath.Push("/tns:IncomeTaxDeductionC");
                        //6. use case
                        if (ReadToNextElement(reader,true) && reader.LocalName == "IncomeTaxDeductionC" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    selfEducation.DDCTNS202 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion IncomeTaxDeductionC
                
                        #region IncomeTaxDeductionEducationCategoryOtherDe
                        _currentXPath.Push("/tns:IncomeTaxDeductionEducationCategoryOtherDe");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "IncomeTaxDeductionEducationCategoryOtherDe" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    selfEducation.DDCTNS203 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion IncomeTaxDeductionEducationCategoryOtherDe
                
                        #region IncomeTaxDeductionA
                        _currentXPath.Push("/tns:IncomeTaxDeductionA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "IncomeTaxDeductionA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        selfEducation.DDCTNS204 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion IncomeTaxDeductionA
                
                        #region SelfEducationCarExpenses
                        _currentXPath.Push("/tns:SelfEducationCarExpenses");
                        //3. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "SelfEducationCarExpenses" && reader.Depth == _currentXPath.Count - 1)
                        {
                            selfEducation.Rp_SelfEducation_SelfEducationCarExpensesCollectionExists = true;
                            selfEducation.Rp_SelfEducation_SelfEducationCarExpensesCollectionCount += 1;
                            _found = true; 
                    
                            #region AssetsCarMakeModelDe
                            _currentXPath.Push("/tns:AssetsCarMakeModelDe");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "AssetsCarMakeModelDe" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        selfEducation.DDCTNS120 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion AssetsCarMakeModelDe
                    
                            #region IncomeTaxDeductionCarCalculationMethodC
                            _currentXPath.Push("/tns:IncomeTaxDeductionCarCalculationMethodC");
                            //6. use case
                            if (ReadToNextElement(reader,true) && reader.LocalName == "IncomeTaxDeductionCarCalculationMethodC" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        selfEducation.DDCTNS121 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion IncomeTaxDeductionCarCalculationMethodC
                    
                            #region IncomeTaxDeductionCarBusinessKilometresN
                            _currentXPath.Push("/tns:IncomeTaxDeductionCarBusinessKilometresN");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "IncomeTaxDeductionCarBusinessKilometresN" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            selfEducation.DDCTNS122 = currentDecimalValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion IncomeTaxDeductionCarBusinessKilometresN
                    
                            #region IncomeTaxDeductionCarLogBookMethodBusinessUseRecordedP
                            _currentXPath.Push("/tns:IncomeTaxDeductionCarLogBookMethodBusinessUseRecordedP");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "IncomeTaxDeductionCarLogBookMethodBusinessUseRecordedP" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            selfEducation.DDCTNS123 = currentDecimalValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion IncomeTaxDeductionCarLogBookMethodBusinessUseRecordedP
                    
                            #region IncomeTaxDeductionCarLogBookMethodA
                            _currentXPath.Push("/tns:IncomeTaxDeductionCarLogBookMethodA");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "IncomeTaxDeductionCarLogBookMethodA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            selfEducation.DDCTNS124 = currentDecimalValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion IncomeTaxDeductionCarLogBookMethodA
                    
                            #region IncomeTaxDeductionCarLogBookMethodDepreciationA
                            _currentXPath.Push("/tns:IncomeTaxDeductionCarLogBookMethodDepreciationA");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "IncomeTaxDeductionCarLogBookMethodDepreciationA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            selfEducation.DDCTNS125 = currentDecimalValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion IncomeTaxDeductionCarLogBookMethodDepreciationA
                        } // End of if SelfEducationCarExpenses node exists
                
                        _currentXPath.Pop();
                        #endregion SelfEducationCarExpenses
                    _currentXPath.Pop();
                    } // End of while loop for SelfEducation
                } // End of if SelfEducationCollection node exists
        
                _currentXPath.Pop();
                #endregion SelfEducation
        
                #region OtherWorkRelatedExpenses
        
                _currentXPath.Push("/tns:OtherWorkRelatedExpensesCollection");
                // 4. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "OtherWorkRelatedExpensesCollection" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.Rp_OtherWorkRelatedExpensesCollection = new List<DDCTNS2026.Rp_OtherWorkRelatedExpenses>();
                    report.Rp_OtherWorkRelatedExpensesCollectionExists = true;
        
                    _found = true;
        
                    while ( ReadToNextElement(reader,false) && reader.LocalName == "OtherWorkRelatedExpenses" ) 
                    {
                        _found = true;
        
                        DDCTNS2026.Rp_OtherWorkRelatedExpenses otherWorkRelatedExpenses = new DDCTNS2026.Rp_OtherWorkRelatedExpenses();
                        report.Rp_OtherWorkRelatedExpensesCollection.Add(otherWorkRelatedExpenses);
                        report.Rp_OtherWorkRelatedExpensesCollectionCount += 1;
                        otherWorkRelatedExpenses.OccurrenceIndex = report.Rp_OtherWorkRelatedExpensesCollectionCount;
        
                        _currentXPath.Push("/tns:OtherWorkRelatedExpenses[" + report.Rp_OtherWorkRelatedExpensesCollectionCount + "]");
                
                        #region IncomeTaxDeductionWorkRelatedReasonOtherDe
                        _currentXPath.Push("/tns:IncomeTaxDeductionWorkRelatedReasonOtherDe");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "IncomeTaxDeductionWorkRelatedReasonOtherDe" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    otherWorkRelatedExpenses.DDCTNS314 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion IncomeTaxDeductionWorkRelatedReasonOtherDe
                
                        #region IncomeTaxDeductionWorkRelatedOtherA
                        _currentXPath.Push("/tns:IncomeTaxDeductionWorkRelatedOtherA");
                        //6. use case
                        if (ReadToNextElement(reader,true) && reader.LocalName == "IncomeTaxDeductionWorkRelatedOtherA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        otherWorkRelatedExpenses.DDCTNS315 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion IncomeTaxDeductionWorkRelatedOtherA
                    _currentXPath.Pop();
                    } // End of while loop for OtherWorkRelatedExpenses
                } // End of if OtherWorkRelatedExpensesCollection node exists
        
                _currentXPath.Pop();
                #endregion OtherWorkRelatedExpenses
        
                #region GiftsorDonations
        
                _currentXPath.Push("/tns:GiftsorDonationsCollection");
                // 4. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "GiftsorDonationsCollection" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.Rp_GiftsorDonationsCollection = new List<DDCTNS2026.Rp_GiftsorDonations>();
                    report.Rp_GiftsorDonationsCollectionExists = true;
        
                    _found = true;
        
                    while ( ReadToNextElement(reader,false) && reader.LocalName == "GiftsorDonations" ) 
                    {
                        _found = true;
        
                        DDCTNS2026.Rp_GiftsorDonations giftsorDonations = new DDCTNS2026.Rp_GiftsorDonations();
                        report.Rp_GiftsorDonationsCollection.Add(giftsorDonations);
                        report.Rp_GiftsorDonationsCollectionCount += 1;
                        giftsorDonations.OccurrenceIndex = report.Rp_GiftsorDonationsCollectionCount;
        
                        _currentXPath.Push("/tns:GiftsorDonations[" + report.Rp_GiftsorDonationsCollectionCount + "]");
                
                        #region IncomeTaxDeductionGiftDonationDe
                        _currentXPath.Push("/tns:IncomeTaxDeductionGiftDonationDe");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "IncomeTaxDeductionGiftDonationDe" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    giftsorDonations.DDCTNS412 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion IncomeTaxDeductionGiftDonationDe
                
                        #region IncomeTaxDeductionGiftDonationA
                        _currentXPath.Push("/tns:IncomeTaxDeductionGiftDonationA");
                        //6. use case
                        if (ReadToNextElement(reader,true) && reader.LocalName == "IncomeTaxDeductionGiftDonationA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        giftsorDonations.DDCTNS413 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion IncomeTaxDeductionGiftDonationA
                    _currentXPath.Pop();
                    } // End of while loop for GiftsorDonations
                } // End of if GiftsorDonationsCollection node exists
        
                _currentXPath.Pop();
                #endregion GiftsorDonations
        
                #region InterestDeductions
        
                _currentXPath.Push("/tns:InterestDeductionsCollection");
                // 4. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "InterestDeductionsCollection" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.Rp_InterestDeductionsCollection = new List<DDCTNS2026.Rp_InterestDeductions>();
                    report.Rp_InterestDeductionsCollectionExists = true;
        
                    _found = true;
        
                    while ( ReadToNextElement(reader,false) && reader.LocalName == "InterestDeductions" ) 
                    {
                        _found = true;
        
                        DDCTNS2026.Rp_InterestDeductions interestDeductions = new DDCTNS2026.Rp_InterestDeductions();
                        report.Rp_InterestDeductionsCollection.Add(interestDeductions);
                        report.Rp_InterestDeductionsCollectionCount += 1;
                        interestDeductions.OccurrenceIndex = report.Rp_InterestDeductionsCollectionCount;
        
                        _currentXPath.Push("/tns:InterestDeductions[" + report.Rp_InterestDeductionsCollectionCount + "]");
                
                        #region IncomeTaxDe
                        _currentXPath.Push("/tns:IncomeTaxDe");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "IncomeTaxDe" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    interestDeductions.DDCTNS317 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion IncomeTaxDe
                
                        #region IncomeTaxA
                        _currentXPath.Push("/tns:IncomeTaxA");
                        //6. use case
                        if (ReadToNextElement(reader,true) && reader.LocalName == "IncomeTaxA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        interestDeductions.DDCTNS318 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion IncomeTaxA
                    _currentXPath.Pop();
                    } // End of while loop for InterestDeductions
                } // End of if InterestDeductionsCollection node exists
        
                _currentXPath.Pop();
                #endregion InterestDeductions
        
                #region DividendDeductions
        
                _currentXPath.Push("/tns:DividendDeductionsCollection");
                // 4. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "DividendDeductionsCollection" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.Rp_DividendDeductionsCollection = new List<DDCTNS2026.Rp_DividendDeductions>();
                    report.Rp_DividendDeductionsCollectionExists = true;
        
                    _found = true;
        
                    while ( ReadToNextElement(reader,false) && reader.LocalName == "DividendDeductions" ) 
                    {
                        _found = true;
        
                        DDCTNS2026.Rp_DividendDeductions dividendDeductions = new DDCTNS2026.Rp_DividendDeductions();
                        report.Rp_DividendDeductionsCollection.Add(dividendDeductions);
                        report.Rp_DividendDeductionsCollectionCount += 1;
                        dividendDeductions.OccurrenceIndex = report.Rp_DividendDeductionsCollectionCount;
        
                        _currentXPath.Push("/tns:DividendDeductions[" + report.Rp_DividendDeductionsCollectionCount + "]");
                
                        #region IncomeTaxDe
                        _currentXPath.Push("/tns:IncomeTaxDe");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "IncomeTaxDe" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    dividendDeductions.DDCTNS131 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion IncomeTaxDe
                
                        #region IncomeTaxA
                        _currentXPath.Push("/tns:IncomeTaxA");
                        //6. use case
                        if (ReadToNextElement(reader,true) && reader.LocalName == "IncomeTaxA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        dividendDeductions.DDCTNS132 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion IncomeTaxA
                    _currentXPath.Pop();
                    } // End of while loop for DividendDeductions
                } // End of if DividendDeductionsCollection node exists
        
                _currentXPath.Pop();
                #endregion DividendDeductions
        
                #region CostofManagingTaxAffairs
        
                _currentXPath.Push("/tns:CostofManagingTaxAffairsCollection");
                // 4. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "CostofManagingTaxAffairsCollection" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.Rp_CostofManagingTaxAffairsCollection = new List<DDCTNS2026.Rp_CostofManagingTaxAffairs>();
                    report.Rp_CostofManagingTaxAffairsCollectionExists = true;
        
                    _found = true;
        
                    while ( ReadToNextElement(reader,false) && reader.LocalName == "CostofManagingTaxAffairs" ) 
                    {
                        _found = true;
        
                        DDCTNS2026.Rp_CostofManagingTaxAffairs costofManagingTaxAffairs = new DDCTNS2026.Rp_CostofManagingTaxAffairs();
                        report.Rp_CostofManagingTaxAffairsCollection.Add(costofManagingTaxAffairs);
                        report.Rp_CostofManagingTaxAffairsCollectionCount += 1;
                        costofManagingTaxAffairs.OccurrenceIndex = report.Rp_CostofManagingTaxAffairsCollectionCount;
        
                        _currentXPath.Push("/tns:CostofManagingTaxAffairs[" + report.Rp_CostofManagingTaxAffairsCollectionCount + "]");
                
                        #region IncomeTaxDeductionTaxAffairManagementAdjustmentReasonC
                        _currentXPath.Push("/tns:IncomeTaxDeductionTaxAffairManagementAdjustmentReasonC");
                        //6. use case
                        if (ReadToNextElement(reader,true) && reader.LocalName == "IncomeTaxDeductionTaxAffairManagementAdjustmentReasonC" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    costofManagingTaxAffairs.DDCTNS304 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion IncomeTaxDeductionTaxAffairManagementAdjustmentReasonC
                
                        #region IncomeTaxDeductionTaxAffairManagementDe
                        _currentXPath.Push("/tns:IncomeTaxDeductionTaxAffairManagementDe");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "IncomeTaxDeductionTaxAffairManagementDe" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    costofManagingTaxAffairs.DDCTNS305 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion IncomeTaxDeductionTaxAffairManagementDe
                
                        #region IncomeTaxDeductionTaxAffairManagementA
                        _currentXPath.Push("/tns:IncomeTaxDeductionTaxAffairManagementA");
                        //6. use case
                        if (ReadToNextElement(reader,true) && reader.LocalName == "IncomeTaxDeductionTaxAffairManagementA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        costofManagingTaxAffairs.DDCTNS306 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion IncomeTaxDeductionTaxAffairManagementA
                    _currentXPath.Pop();
                    } // End of while loop for CostofManagingTaxAffairs
                } // End of if CostofManagingTaxAffairsCollection node exists
        
                _currentXPath.Pop();
                #endregion CostofManagingTaxAffairs
        
                #region PersonalSuperContributions
        
                _currentXPath.Push("/tns:PersonalSuperContributionsCollection");
                // 4. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "PersonalSuperContributionsCollection" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.Rp_PersonalSuperContributionsCollection = new List<DDCTNS2026.Rp_PersonalSuperContributions>();
                    report.Rp_PersonalSuperContributionsCollectionExists = true;
        
                    _found = true;
        
                    while ( ReadToNextElement(reader,false) && reader.LocalName == "PersonalSuperContributions" ) 
                    {
                        _found = true;
        
                        DDCTNS2026.Rp_PersonalSuperContributions personalSuperContributions = new DDCTNS2026.Rp_PersonalSuperContributions();
                        report.Rp_PersonalSuperContributionsCollection.Add(personalSuperContributions);
                        report.Rp_PersonalSuperContributionsCollectionCount += 1;
                        personalSuperContributions.OccurrenceIndex = report.Rp_PersonalSuperContributionsCollectionCount;
        
                        _currentXPath.Push("/tns:PersonalSuperContributions[" + report.Rp_PersonalSuperContributionsCollectionCount + "]");
                
                        #region SuperannuationContributionNOIProvidedAcknowledgmentReceivedI
                        _currentXPath.Push("/tns:SuperannuationContributionNOIProvidedAcknowledgmentReceivedI");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "SuperannuationContributionNOIProvidedAcknowledgmentReceivedI" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    currentBooleanValue = ToBoolean(currentValue);
                                    if (currentBooleanValue != null)
                                    {
                                        personalSuperContributions.DDCTNS401 = currentBooleanValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion SuperannuationContributionNOIProvidedAcknowledgmentReceivedI
                
                        #region OrganisationNameDetailsOrganisationalNameT
                        _currentXPath.Push("/tns:OrganisationNameDetailsOrganisationalNameT");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "OrganisationNameDetailsOrganisationalNameT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    personalSuperContributions.DDCTNS402 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion OrganisationNameDetailsOrganisationalNameT
                
                        #region IncomeTaxDeductionSuperannuationContributionEligibleD
                        _currentXPath.Push("/tns:IncomeTaxDeductionSuperannuationContributionEligibleD");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "IncomeTaxDeductionSuperannuationContributionEligibleD" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                                    {
                                        personalSuperContributions.DDCTNS403 = currentDateTimeValue.Date;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion IncomeTaxDeductionSuperannuationContributionEligibleD
                
                        #region SuperannuationFundDetailsMemberAccountId
                        _currentXPath.Push("/tns:SuperannuationFundDetailsMemberAccountId");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "SuperannuationFundDetailsMemberAccountId" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    personalSuperContributions.DDCTNS404 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion SuperannuationFundDetailsMemberAccountId
                
                        #region AustralianBusinessNumberId
                        _currentXPath.Push("/tns:AustralianBusinessNumberId");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "AustralianBusinessNumberId" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    personalSuperContributions.DDCTNS405 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion AustralianBusinessNumberId
                
                        #region TaxFileNumberId
                        _currentXPath.Push("/tns:TaxFileNumberId");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "TaxFileNumberId" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    personalSuperContributions.DDCTNS406 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion TaxFileNumberId
                
                        #region IncomeTaxDeductionSuperannuationContributionA
                        _currentXPath.Push("/tns:IncomeTaxDeductionSuperannuationContributionA");
                        //6. use case
                        if (ReadToNextElement(reader,true) && reader.LocalName == "IncomeTaxDeductionSuperannuationContributionA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        personalSuperContributions.DDCTNS407 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion IncomeTaxDeductionSuperannuationContributionA
                    _currentXPath.Pop();
                    } // End of while loop for PersonalSuperContributions
                } // End of if PersonalSuperContributionsCollection node exists
        
                _currentXPath.Pop();
                #endregion PersonalSuperContributions
        
                #region ProjectPool
        
                _currentXPath.Push("/tns:ProjectPoolCollection");
                // 4. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "ProjectPoolCollection" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.Rp_ProjectPoolCollection = new List<DDCTNS2026.Rp_ProjectPool>();
                    report.Rp_ProjectPoolCollectionExists = true;
        
                    _found = true;
        
                    while ( ReadToNextElement(reader,false) && reader.LocalName == "ProjectPool" ) 
                    {
                        _found = true;
        
                        DDCTNS2026.Rp_ProjectPool projectPool = new DDCTNS2026.Rp_ProjectPool();
                        report.Rp_ProjectPoolCollection.Add(projectPool);
                        report.Rp_ProjectPoolCollectionCount += 1;
                        projectPool.OccurrenceIndex = report.Rp_ProjectPoolCollectionCount;
        
                        _currentXPath.Push("/tns:ProjectPool[" + report.Rp_ProjectPoolCollectionCount + "]");
                
                        #region IncomeTaxDeductionDe
                        _currentXPath.Push("/tns:IncomeTaxDeductionDe");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "IncomeTaxDeductionDe" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    projectPool.DDCTNS211 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion IncomeTaxDeductionDe
                
                        #region IncomeTaxDeductionA
                        _currentXPath.Push("/tns:IncomeTaxDeductionA");
                        //6. use case
                        if (ReadToNextElement(reader,true) && reader.LocalName == "IncomeTaxDeductionA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        projectPool.DDCTNS212 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion IncomeTaxDeductionA
                    _currentXPath.Pop();
                    } // End of while loop for ProjectPool
                } // End of if ProjectPoolCollection node exists
        
                _currentXPath.Pop();
                #endregion ProjectPool
        
                #region ForestryManagedInvestment
        
                _currentXPath.Push("/tns:ForestryManagedInvestmentCollection");
                // 4. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "ForestryManagedInvestmentCollection" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.Rp_ForestryManagedInvestmentCollection = new List<DDCTNS2026.Rp_ForestryManagedInvestment>();
                    report.Rp_ForestryManagedInvestmentCollectionExists = true;
        
                    _found = true;
        
                    while ( ReadToNextElement(reader,false) && reader.LocalName == "ForestryManagedInvestment" ) 
                    {
                        _found = true;
        
                        DDCTNS2026.Rp_ForestryManagedInvestment forestryManagedInvestment = new DDCTNS2026.Rp_ForestryManagedInvestment();
                        report.Rp_ForestryManagedInvestmentCollection.Add(forestryManagedInvestment);
                        report.Rp_ForestryManagedInvestmentCollectionCount += 1;
                        forestryManagedInvestment.OccurrenceIndex = report.Rp_ForestryManagedInvestmentCollectionCount;
        
                        _currentXPath.Push("/tns:ForestryManagedInvestment[" + report.Rp_ForestryManagedInvestmentCollectionCount + "]");
                
                        #region ExpenseSchemeDeductionDe
                        _currentXPath.Push("/tns:ExpenseSchemeDeductionDe");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "ExpenseSchemeDeductionDe" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    forestryManagedInvestment.DDCTNS308 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion ExpenseSchemeDeductionDe
                
                        #region ExpenseSchemeDeductionA
                        _currentXPath.Push("/tns:ExpenseSchemeDeductionA");
                        //6. use case
                        if (ReadToNextElement(reader,true) && reader.LocalName == "ExpenseSchemeDeductionA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        forestryManagedInvestment.DDCTNS309 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion ExpenseSchemeDeductionA
                    _currentXPath.Pop();
                    } // End of while loop for ForestryManagedInvestment
                } // End of if ForestryManagedInvestmentCollection node exists
        
                _currentXPath.Pop();
                #endregion ForestryManagedInvestment
        
                #region OtherDeductions
        
                _currentXPath.Push("/tns:OtherDeductionsCollection");
                // 4. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "OtherDeductionsCollection" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.Rp_OtherDeductionsCollection = new List<DDCTNS2026.Rp_OtherDeductions>();
                    report.Rp_OtherDeductionsCollectionExists = true;
        
                    _found = true;
        
                    while ( ReadToNextElement(reader,false) && reader.LocalName == "OtherDeductions" ) 
                    {
                        _found = true;
        
                        DDCTNS2026.Rp_OtherDeductions otherDeductions = new DDCTNS2026.Rp_OtherDeductions();
                        report.Rp_OtherDeductionsCollection.Add(otherDeductions);
                        report.Rp_OtherDeductionsCollectionCount += 1;
                        otherDeductions.OccurrenceIndex = report.Rp_OtherDeductionsCollectionCount;
        
                        _currentXPath.Push("/tns:OtherDeductions[" + report.Rp_OtherDeductionsCollectionCount + "]");
                
                        #region ExpenseDeductibleOtherC
                        _currentXPath.Push("/tns:ExpenseDeductibleOtherC");
                        //6. use case
                        if (ReadToNextElement(reader,true) && reader.LocalName == "ExpenseDeductibleOtherC" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    otherDeductions.DDCTNS136 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion ExpenseDeductibleOtherC
                
                        #region ExpenseDeductibleOtherT
                        _currentXPath.Push("/tns:ExpenseDeductibleOtherT");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "ExpenseDeductibleOtherT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    otherDeductions.DDCTNS137 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion ExpenseDeductibleOtherT
                
                        #region ExpenseDeductibleOtherA
                        _currentXPath.Push("/tns:ExpenseDeductibleOtherA");
                        //6. use case
                        if (ReadToNextElement(reader,true) && reader.LocalName == "ExpenseDeductibleOtherA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        otherDeductions.DDCTNS138 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion ExpenseDeductibleOtherA
                    _currentXPath.Pop();
                    } // End of while loop for OtherDeductions
                } // End of if OtherDeductionsCollection node exists
        
                _currentXPath.Pop();
                #endregion OtherDeductions
        
                #region LowValuePool
                _currentXPath.Push("/tns:LowValuePool");
                //3. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "LowValuePool" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.Rp_LowValuePoolCollectionExists = true;
                    report.Rp_LowValuePoolCollectionCount += 1;
                    _found = true; 
            
                    #region IncomeTaxDeductionFinancialInvestmentsA
                    _currentXPath.Push("/tns:IncomeTaxDeductionFinancialInvestmentsA");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "IncomeTaxDeductionFinancialInvestmentsA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.DDCTNS310 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion IncomeTaxDeductionFinancialInvestmentsA
            
                    #region IncomeTaxDeductionRentalPropertiesA
                    _currentXPath.Push("/tns:IncomeTaxDeductionRentalPropertiesA");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "IncomeTaxDeductionRentalPropertiesA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.DDCTNS311 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion IncomeTaxDeductionRentalPropertiesA
            
                    #region IncomeTaxDeductionOtherA
                    _currentXPath.Push("/tns:IncomeTaxDeductionOtherA");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "IncomeTaxDeductionOtherA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.DDCTNS312 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion IncomeTaxDeductionOtherA
                } // End of if LowValuePool node exists
        
                _currentXPath.Pop();
                #endregion LowValuePool
        
                #region IncomeTaxDeductionElectionA
                _currentXPath.Push("/tns:IncomeTaxDeductionElectionA");
                //6. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "IncomeTaxDeductionElectionA" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                            {
                                report.DDCTNS417 = currentDecimalValue;
                            }
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion IncomeTaxDeductionElectionA
            } // End of if Rp node exists
    
            _currentXPath.Pop();
            #endregion Rp

            while (reader.EOF != true)
                reader.Read();

            return report;
        }

        private string _elementCsvList = "Rp#LodgmentPeriodStartD,Rp#LodgmentPeriodEndD,Rp#TaxFileNumberId,Car#AssetsMakeModelDe,Car#IncomeTaxDeductionCalculationMethodC,Car#IncomeTaxDeductionBusinessKilometresN,Car#IncomeTaxDeductionLogBookMethodBusinessUseRecordedP,Car#IncomeTaxDeductionLogBookMethodA,Car#IncomeTaxDeductionLogBookMethodDepreciationA,Car,CarCollection,Travel#IncomeTaxDeductionReasonDe,Travel#IncomeTaxDeductionA,Travel,TravelCollection,Clothing#IncomeTaxDeductionTypeC,Clothing#IncomeTaxDeductionA,Clothing,ClothingCollection,Rp#IncomeTaxDeductionEducationCircumstancesC,SelfEducation#IncomeTaxDeductionC,SelfEducation#IncomeTaxDeductionEducationCategoryOtherDe,SelfEducation#IncomeTaxDeductionA,SelfEducation#AssetsCarMakeModelDe,SelfEducation#IncomeTaxDeductionCarCalculationMethodC,SelfEducation#IncomeTaxDeductionCarBusinessKilometresN,SelfEducation#IncomeTaxDeductionCarLogBookMethodBusinessUseRecordedP,SelfEducation#IncomeTaxDeductionCarLogBookMethodA,SelfEducation#IncomeTaxDeductionCarLogBookMethodDepreciationA,SelfEducationCarExpenses,SelfEducation,SelfEducationCollection,OtherWorkRelatedExpenses#IncomeTaxDeductionWorkRelatedReasonOtherDe,OtherWorkRelatedExpenses#IncomeTaxDeductionWorkRelatedOtherA,OtherWorkRelatedExpenses,OtherWorkRelatedExpensesCollection,GiftsorDonations#IncomeTaxDeductionGiftDonationDe,GiftsorDonations#IncomeTaxDeductionGiftDonationA,GiftsorDonations,GiftsorDonationsCollection,InterestDeductions#IncomeTaxDe,InterestDeductions#IncomeTaxA,InterestDeductions,InterestDeductionsCollection,DividendDeductions#IncomeTaxDe,DividendDeductions#IncomeTaxA,DividendDeductions,DividendDeductionsCollection,CostofManagingTaxAffairs#IncomeTaxDeductionTaxAffairManagementAdjustmentReasonC,CostofManagingTaxAffairs#IncomeTaxDeductionTaxAffairManagementDe,CostofManagingTaxAffairs#IncomeTaxDeductionTaxAffairManagementA,CostofManagingTaxAffairs,CostofManagingTaxAffairsCollection,PersonalSuperContributions#SuperannuationContributionNOIProvidedAcknowledgmentReceivedI,PersonalSuperContributions#OrganisationNameDetailsOrganisationalNameT,PersonalSuperContributions#IncomeTaxDeductionSuperannuationContributionEligibleD,PersonalSuperContributions#SuperannuationFundDetailsMemberAccountId,PersonalSuperContributions#AustralianBusinessNumberId,PersonalSuperContributions#TaxFileNumberId,PersonalSuperContributions#IncomeTaxDeductionSuperannuationContributionA,PersonalSuperContributions,PersonalSuperContributionsCollection,ProjectPool#IncomeTaxDeductionDe,ProjectPool#IncomeTaxDeductionA,ProjectPool,ProjectPoolCollection,ForestryManagedInvestment#ExpenseSchemeDeductionDe,ForestryManagedInvestment#ExpenseSchemeDeductionA,ForestryManagedInvestment,ForestryManagedInvestmentCollection,OtherDeductions#ExpenseDeductibleOtherC,OtherDeductions#ExpenseDeductibleOtherT,OtherDeductions#ExpenseDeductibleOtherA,OtherDeductions,OtherDeductionsCollection,LowValuePool#IncomeTaxDeductionFinancialInvestmentsA,LowValuePool#IncomeTaxDeductionRentalPropertiesA,LowValuePool#IncomeTaxDeductionOtherA,LowValuePool,Rp#IncomeTaxDeductionElectionA,Rp";

    }
}
