using System;
using System.Collections.Generic;
using Au.Gov.Sbr.Xbrl.Document;
using Au.Gov.Sbr.Xbrl.Metadata;
using System.Linq;
using System.Xml;
using System.Text.RegularExpressions;
using DataContracts;

namespace Ato.EN.IntegrationServices.CodeGenerationIITR
{
    public class IITR2026Consumer
    {

        private static Regex StripWhitespace = new Regex(@"\s*");

        #region Helper/Validation Methods
        private static bool? ToBoolean(string str)
        {
            bool returnValue;
            string value = (str ?? "").Trim();
            if (value == "0" || value == "1")
                return Convert.ToBoolean(Convert.ToInt32(value));
            
            if (Boolean.TryParse(value, out returnValue))
                return returnValue;
            return null;
        }

        private static string GetValueOrEmpty(bool? val)
        {
            return (val.HasValue) ? val.ToString().ToLower() : string.Empty;
        }

        private static string GetValueOrEmptyDate(DateTime? val)
        {
            return (val.HasValue) ? val.Value.ToString("yyyy-MM-dd") : string.Empty;
        }

        private static string GetValueOrEmptyDateTime(DateTime? val)
        {
            return (val.HasValue) ? val.Value.ToString("yyyy-MM-ddTHH:mm:ssZ") : string.Empty;
        }
				
        private static string GetValueOrEmptyDate(string val)
        {
            DateTime dateTimeOut;
            if (DateTime.TryParse(val, out dateTimeOut))
            {
                return dateTimeOut.ToString("yyyy-MM-dd");
            }
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmptyDateTime(string val)
        {
            DateTime dateTimeOut;
            if (DateTime.TryParse(val, out dateTimeOut))
            {
                return dateTimeOut.ToString("yyyy-MM-ddTHH:mm:ssZ");
            }
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmpty(string val)
        {
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmpty(decimal? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        private static string GetValueOrEmpty(int? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        /// <summary>
        /// Derives the SBR element location.
        /// </summary>
        /// <param name="sbrElement">The SBR element.</param>
        /// <param name="addOccurrenceIndex">if set to <c>true</c> [add occurrence the occurrence index to the path].</param>
        /// <returns></returns>
        private static string DeriveSbrElementLocation(SBRElement sbrElement, bool addOccurrenceIndex = false)
        {
            if (sbrElement == null) return "/xbrli:xbrl";
            int xpathOccurrence = sbrElement.OccurrenceIndex + 1;
            string location;

            if (addOccurrenceIndex)
            {
                location = "/" + sbrElement.NamespacePrefix + ":" + sbrElement.Name + "[" +
                           xpathOccurrence + "]";
            }
            else
            {
                location = "/" + sbrElement.NamespacePrefix + ":" + sbrElement.Name;
            }

            if (sbrElement.IsInTuple)
            {
                location = DeriveSbrElementLocation(sbrElement.ParentTuple, true) + location;
            }
            else
            {
                location = "/xbrli:xbrl" + location;
            }

            return location;
        }

        private static bool IsMatch(int? field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(Convert.ToString(field.Value), expression, options);
        }

        private static bool IsMatch(string field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(field, expression, options);
        }

        public DateTime AsDate(string dateAsString)
        {
            DateTime response = DateTime.MinValue;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date;
            }

            return response;
        }

         #endregion

        public IITR2026 Consume(SBRDocument sbrDocument)
        {
            List<ProcessMessageDocument> errors;
            return Consume(sbrDocument, false, out errors);
        }

        /// <summary>
        /// Allows for validation of rules that can not be generated and must run at a lower level.
        /// A sub-class will need to be created with an override for this method - which does nothing in its base implementation.
        /// </summary>
        /// <param name="sbrDocument">The SBR document.</param>
        /// <param name="populatedReportElementsMap">The populated report elements map can be used to find data elements and tuples.</param>
        /// <param name="contextArray">The context array can be used to find context values.</param>
        /// <param name="errors">The errors collection is to be populated with validation results.</param>
        protected virtual void ValidateUnsupportedRules(SBRDocument sbrDocument, IDictionary<string, SBRElement> populatedReportElementsMap, SBRContext[] contextArray, List<ProcessMessageDocument> errors)
        {

        }

        public IITR2026 Consume(SBRDocument sbrDocument, bool validationMode, out List<ProcessMessageDocument> errors)
        {
            IITR2026 report;
            report = new IITR2026();
            errors = new List<ProcessMessageDocument>();

            SBRContext[] contextArray;
            IDictionary<string, SBRElement> populatedReportElementsMap;

            contextArray = sbrDocument.ContextsMap.Values.ToArray();
            populatedReportElementsMap = sbrDocument.PopulatedChildElementsMap ?? new Dictionary<string, SBRElement>();

            if (validationMode) ValidateUnsupportedRules(sbrDocument, populatedReportElementsMap, contextArray, errors);

            /// <summary>
            /// Current element being extracted
            /// </summary>
            SBRElement currentValue = null;
            /// <summary>
            /// Current elements being extracted
            /// </summary>
            IList<SBRElement> currentValues = null;
            /// <summary>
            /// Current occurrences being extracted
            /// </summary>
            IList<SBRElement> currentOccurrences = null;
            /// <summary>
            /// Working Variable for date time variables
            /// </summary>
            DateTime currentDateTimeValue;
            /// <summary>
            /// Working Variable for boolean variables
            /// </summary>
            bool currentBooleanValue;
            /// <summary>
            /// Working Variable for decimal variables
            /// </summary>
            decimal currentDecimalValue;
            /// <summary>
            /// Working Variable for integer variables
            /// </summary>
            int currentIntValue;
            /// <summary>
            /// Working Variable for long variables
            /// </summary>
            long currentLongValue;
            XmlDocumentFragment currentSegment;
            XmlNamespaceManager currentNamespaceManager;
            XmlNode currentNode;
    
            #region sbrRPContext
            IEnumerable<SBRContext> sbrRPContexts = null;
            SBRContext sbrRPContext = null;
    
            sbrRPContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 1 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty")) )).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPContexts != null && sbrRPContexts.Count() > 0)
            {
                sbrRPContext = sbrRPContexts.First();
        
                report.RPId = sbrRPContext.Id;
                report.RPIdentifier = sbrRPContext.EntityIdentifier;
                report.RPIdentifierScheme = sbrRPContext.EntityScheme;
        
                report.RPCount = sbrRPContexts.Count();
                report.RPExists = true;
                report.RPOccurrenceIndex = Array.IndexOf(contextArray, sbrRPContext) + 1;
                report.RPLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPContexts.Last()) + 1;
        

                if (report.RPIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPIdentifierTFN = report.RPIdentifier;
    
                report.RPStartDate = ((SBRDurationPeriod)sbrRPContext.Period).DurationStartAsDateObject;
                report.RPEndDate = ((SBRDurationPeriod)sbrRPContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPContexts.Count(); i++)
                {
                    sbrRPContext = sbrRPContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
                    report.IITR11 = report.RPStartDate;
                    report.IITR12 = report.RPEndDate;
                    report.IITR15 = report.RPIdentifier;
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Report.TargetFinancial.Year", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR10 = int.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.ReturnType.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR13 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Report.Amendment.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR16 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Report.AmendmentType.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR17 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Report.AmendmentSequence.Number", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR56 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Report.AmendmentReason.Text", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR18 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Residency.TaxPurposesPersonStatus.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR19 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Residency.TaxPurposesStart.Date", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR666 = DateTime.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Residency.TaxPurposesEnd.Date", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR667 = DateTime.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}PersonNameDetails.Title.Text", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR21 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}PersonNameDetails.FamilyName.Text", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR22 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}PersonNameDetails.NameSuffix.Text", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR23 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}PersonNameDetails.GivenName.Text", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR24 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}PersonNameDetails.OtherGivenName.Text", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR25 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Party.ChangeName.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR26 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}PersonDemographicDetails.Death.Date", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR28 = DateTime.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}PersonDemographicDetails.Birth.Date", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR29 = DateTime.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Lodgment.FinalReturn.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR30 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}RegulatoryDisclosures.GeneralInformationAboutFinancialStatements.Text", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR31 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}ElectronicContact.Telephone.Mobile.Number", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR48 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}ElectronicContact.Telephone.Country.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR579 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}ElectronicContact.Telephone.Minimal.Number", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR50 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}ElectronicContact.ElectronicMail.Address.Text", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR51 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}FinancialInstitutionAccount.BankStateBranch.Number", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR53 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}FinancialInstitutionAccount.FinancialInstitutionAccount.Number", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR54 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}FinancialInstitutionAccount.FinancialInstitutionAccountName.Text", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR55 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}PersonDemographicDetails.Occupation.Description", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR58 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}PersonDemographicDetails.Occupation.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR59 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldSalaryOrWages.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR600 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.SalaryOrWages.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR601 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.CommunityDevelopmentEmploymentProject.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR602 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldAllowancesEarningsTipsDirectorsFees.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR64 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.AllowancesEarningsTipsDirectorsFees.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR65 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldUnusedAnnualOrLongServiceLeavePaymentLumpSumA.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR606 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.UnusedAnnualOrLongServiceLeavePaymentLumpSumA.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR607 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldUnusedAnnualOrLongServiceLeavePaymentLumpSumB.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR608 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.UnusedAnnualOrLongServiceLeavePaymentLumpSumB.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR609 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldEmploymentTerminationPayment.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR625 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.EmploymentTerminationPaymentTaxable.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR626 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldAustralianGovernmentAllowancesAndPayments.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR85 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.AustralianGovernmentAllowancesAndPayments.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR86 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldAustralianGovernmentPensionsAndAllowances.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR88 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Pension.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR89 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldAustralianAnnuitiesAndSuperannuationIncomeStream.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR91 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.Superannuation.AnnuitiesAndSuperannuationTaxableComponentTaxedElement.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR92 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.Superannuation.AnnuitiesAndSuperannuationTaxableComponentUntaxedElement.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR93 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.Superannuation.AnnuitiesAndSuperannuationAssessableCappedBenefit.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR505 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.Superannuation.AnnuitiesAndSuperannuationLumpSumArrearsTaxableComponentTaxedElement.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR94 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.Superannuation.AnnuitiesAndSuperannuationLumpSumArrearsTaxableComponentUntaxedElement.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR95 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldSuperannuationLumpSumPayment.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR610 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.Superannuation.LumpSumPaymentTaxableComponentTaxedElement.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR611 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.Superannuation.LumpSumPaymentTaxableComponentUntaxedElement.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR612 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldPersonalServicesIncome.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR111 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.PersonalServicesIncome.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR112 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.PayAsYouGoWithholding.CreditTaxWithheld.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR113 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.Interest.Gross.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR115 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.TFNAmountsWithheldFromGrossInterest.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR116 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.DividendsUnfranked.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR118 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.DividendsFranked.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR119 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.FrankingCredits.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR120 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldDividendsTFNNotQuoted.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR121 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.EmployeeShareScheme.TaxUpfrontReducibleDiscount.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR123 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.EmployeeShareScheme.TaxUpfrontNonReducibleDiscount.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR124 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.EmployeeShareScheme.DeferralDiscount.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR125 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.EmployeeShareScheme.DiscountAssessable.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR127 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldEmployeeShareSchemeDiscountTFNNotQuoted.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR128 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.EmployeeShareScheme.ForeignSourceDiscount.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR129 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.Net.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR131 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.Deduction.Car.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR134 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.Deduction.Travel.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR136 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.Deduction.Clothing.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR137 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.Deduction.EducationNet.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR139 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.Deduction.WorkRelatedOther.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR141 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.Deduction.LowValuePool.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR145 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.Deduction.Interest.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR146 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.Deduction.Dividend.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR147 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.Deduction.GiftDonation.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR148 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.Deduction.TaxAffairManagementLitigation.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR503 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.Deduction.TaxAffairManagementOther.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR504 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Expense.DeductionsTotal.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR150 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.LossesBroughtForwardPrimaryProduction.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR153 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.LossesClaimedPrimaryProduction.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR154 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.LossesBroughtForwardNonPrimaryProduction.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR155 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.LossesClaimedNonPrimaryProduction.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR156 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.Taxable.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR157 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}TaxOffsetClaim.SeniorAndPensioner.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR163 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}TaxOffsetClaim.Veteran.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR164 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}TaxOffsetClaim.SuperannuationIncomeStream.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR166 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}TaxOffsetClaim.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR168 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.MedicareLevy.DependentChildrenAndStudents.Count", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR171 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.MedicareLevy.ExemptionFullDays.Count", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR172 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.MedicareLevy.ExemptionFullDays.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR173 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.MedicareLevy.ExemptionHalfDays.Count", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR174 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.MedicareLevy.PrivateHospitalCoverWholePeriod.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR176 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.MedicareLevy.SurchargeExemptionDays.Count", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR177 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.ExceptedNetIncomeUnder18.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR188 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.ExceptedIncomeUnder18.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR189 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.ExceptedPrimaryProductionIncomeUnder18.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR619 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.EligiblePrimaryProductionIncomeUnder18.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR620 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.ExceptedNonPrimaryProductionIncomeUnder18.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR621 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.PartYearTaxFreeThresholdEligibleMonths.Number", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR192 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.InvestmentPartnershipOtherSource.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR194 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.EligiblePersonalSuperannuationContribution.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR195 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.EmploymentBusinessOther.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR196 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.Deduction.BusinessOther.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR197 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}SuperannuationContribution.EmployerReportable.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR201 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Pension.TaxFree.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR202 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}InternationalDealings.ForeignIncomeTarget.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR203 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Tax.Losses.FinancialInvestmentNet.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR204 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Tax.Losses.RentalPropertyNet.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR205 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}ChildSupport.Payment.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR206 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}PersonDemographicDetails.DependentChildren.Count", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR207 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.TaxableAdjusted.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR208 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.Estimated.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR209 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.LowIncomeSuperannuationContributionEligible.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR210 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}PersonDemographicDetails.SpouseFullYear.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR220 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}PersonDemographicDetails.SpouseRelationshipStart.Date", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR221 = DateTime.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}PersonDemographicDetails.SpouseRelationshipEnd.Date", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR222 = DateTime.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.PartnershipPrimaryProductionShare.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR242 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.TrustPrimaryProductionNetShare.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR243 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.TrustPrimaryProductionNetShare.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR244 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.Deduction.PartnershipsAndTrustPrimaryProductionShare.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR246 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.Deduction.PartnershipsAndTrustPrimaryProductionShare.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR247 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.PartnershipInvestmentLessForeignIncomeNonPrimaryProductionShare.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR249 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.PartnershipRentalNonPrimaryProductionShare.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR250 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.PartnershipNonPrimaryProductionOtherShare.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR251 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.PartnershipLessForeignIncomeNonPrimaryProductionShare.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR252 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.TrustLessNetCapitalGainAndForeignIncomeNonPrimaryProductionShare.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR255 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.TrustLessNetCapitalGainAndForeignIncomeNonPrimaryProductionShare.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR256 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.TrustFrankedNonPrimaryProductionShare.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR259 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Expense.LandcareOperationsNonPrimaryProduction.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR260 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.Deduction.TrustExcludingCapitalGainsAndForeignIncomeNonPrimaryProduction.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR261 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.Deduction.PartnershipExcludingForeignIncomeNonPrimaryProduction.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR262 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.Deduction.PartnershipRentalNonPrimaryProduction.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR263 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.Deduction.TrustsPartnershipsOtherAllowableNonPrimaryProduction.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR264 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.Deduction.TrustPartnershipOtherNonPrimaryProductionNet.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR265 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.Deduction.TrustPartnershipOtherNonPrimaryProductionNet.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR266 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR268 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.FrankingCreditsShare.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR269 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldInvestmentsTFNNotQuotedShare.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR270 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldCloselyHeldTrustShare.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR271 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.CreditTrusteeTaxPaidShare.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR272 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldForeignResidentShare.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR273 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}TaxOffsetClaim.NationalRentalAffordabilitySchemeEntitlementShare.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR274 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldPersonalServicesIncomeVoluntaryAgreement.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR276 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldPersonalServicesIncomeABNNotQuoted.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR277 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldPersonalServicesIncomeLabourHireOrOtherPayments.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR278 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.PersonalServicesIncome.Net.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR279 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.PrimaryProductionNet.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR281 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.InvestmentNonPrimaryProductionNet.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR282 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.RentalPropertyNonPrimaryProductionNet.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR283 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.NonPrimaryProductionOtherNet.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR284 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.NonPrimaryProductionNet.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR285 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.SmallBusinessEntityNet.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR557 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldBusinessVoluntaryAgreement.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR286 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldBusinessABNNotQuoted.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR287 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldBusinessForeignResident.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR288 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldBusinessLabourHireOrOtherPayments.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR289 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Tax.Losses.DeferredNonCommercialPartnershipInvestmentShare.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR291 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Tax.Losses.DeferredNonCommercialPartnershipRentalShare.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR292 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Tax.Losses.DeferredNonCommercialPartnershipOtherShare.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR293 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Tax.Losses.DeferredNonCommercialPartnershipShare.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR294 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Tax.Losses.DeferredNonCommercialSoleTraderInvestment.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR295 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Tax.Losses.DeferredNonCommercialSoleTraderRental.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR296 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Tax.Losses.DeferredNonCommercialSoleTraderOther.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR297 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Tax.Losses.DeferredNonCommercialSoleTrader.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR298 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Tax.Losses.DeferredNonCommercialPrimaryProduction.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR299 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Tax.Losses.DeferredNonCommercialNonPrimaryProduction.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR300 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.Deduction.FarmManagementDeposits.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR302 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.FarmManagementEarlyRepaymentsNaturalDisaster.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR304 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.FarmManagementOtherRepayments.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR305 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.FarmManagementDepositsOrRepaymentsNet.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR306 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}CapitalGainsTax.Event.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR308 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}CapitalGainsTax.ExemptionOrRolloverApplied.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR309 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}CapitalGainsTax.ExemptionOrRolloverApplied.Code", out currentValues))
                    {
                        report.IITR310Collection = new List<string>();
                        try
                        {
                                foreach (SBRElement occurrence in currentValues)
                                {
                                    if (!occurrence.IsNil)
                                        report.IITR310Collection.Add(occurrence.Value);
                                }
                        }
                        catch (Exception)
                        {
                            // The above code is known to throw exceptions in some cases but not others.
                            // We don't really know why but think that catching them like this will result in correct behaviour
                        }
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Capital.Gains.Net.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR311 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Capital.Gains.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR312 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.PayAsYouGoWithholding.CreditForCapitalGainsWithheldFromForeignResidents.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR497 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}InternationalDealings.DirectOrIndirectOverseasInterest.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR315 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.ControlledForeignCompaniesAttributableIncome.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR316 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.InternationalDealings.AttributedForeignIncomeForeignTrustorControlledForeignCompanyorTransferorTrust.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR317 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}InternationalDealings.TransferorTrustAttributedForeignIncome.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR318 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}InternationalDealings.ForeignIncomeGross.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR320 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}InternationalDealings.ForeignEmploymentIncomeOther.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR321 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}InternationalDealings.ForeignPensionOrAnnuityIncomeWithoutUndeductedPurchasePrice.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR322 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}InternationalDealings.ForeignPensionOrAnnuityIncomeUndeductedPurchasePrice.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR323 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}InternationalDealings.ForeignRent.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR324 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}InternationalDealings.ForeignSourceCompanyIncomeOther.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR325 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}InternationalDealings.ForeignSourceIncomeOther.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR326 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.InternationalDealings.Net.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR327 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.FrankingCreditsNewZealand.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR328 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}InternationalDealings.ForeignEmploymentIncomePaymentSummary.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR329 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}InternationalDealings.ExemptForeignEmploymentIncome.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR330 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.InternationalDealings.TaxOffset.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR331 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}InternationalDealings.AssetsLocatedOutsideAustralia.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR332 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.RealEstateProperty.Rental.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR334 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.Deduction.RentalIncomeInterestDeductions.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR335 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.Deduction.CapitalWorksDeductions.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR336 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.Deduction.RentalIncomeDeductionsOtherThanInterestAndCapitalWorks.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR337 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.LifeInsuranceCompaniesAndFriendlySocietiesBonuses.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR339 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.ForestryManagedInvestmentScheme.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR341 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.DepreciatingAssets.IntangibleBalancingAdjustment.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR613 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.DepreciatingAssetsAssessableIncomeFromBalancingAdjustmentEventsTotal.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR614 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.IncomeCategory1Other.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR615 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.IncomeCategory2Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR616 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.AssessableFirstHomeSuperSaverReleased.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR650 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.AssessableFirstHomeSuperSaverReleasedTaxWithheld.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR651 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.FinancialInvestmentsOther.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR617 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.IncomeCategoryMiscellaneousOther.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR618 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldLumpSumPaymentsInArrears.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR354 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.SpecialProfessional.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR355 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.Deduction.ForeignPensionAnnuityIncomeUndeductedPurchasePrice.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR358 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.Deduction.ProjectPool.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR366 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Expense.ForestryManagedInvestmentSchemeDeduction.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR368 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}SuperannuationContribution.SpouseContributions.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR378 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}TaxOffsetClaim.SuperannuationContributionSpouse.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR379 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}TaxOffsetClaim.ZoneOrOverseasForces.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR381 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}TaxOffsetClaim.DependentInvalidAndCarer.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR386 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}TaxOffsetClaim.LandcareAndWaterFacilityBroughtForward.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR388 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.TaxOffsetRefundableOther.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR395 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.TaxOffsetRefundableOther.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR396 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.FamilyTrustDistributionWhereTaxPaid.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR399 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.PersonalServicesIncomeIncluded.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR404 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.PersonalServicesIncome.IndividualResultsTestSatisfied.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR405 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.PersonalServicesIncome.IndividualPersonalServiceBusinessDeterminationHeld.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR406 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.PersonalServicesIncome.OneSourceBenchmark.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR407 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.PersonalServicesIncome.UnrelatedClientsTestSatisfied.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR408 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.PersonalServicesIncome.EmploymentTestSatisfied.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR409 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.PersonalServicesIncome.BusinessPremisesTestSatisfied.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR410 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.PersonalServicesIncome.VoluntaryAgreement.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR411 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.PersonalServicesIncome.ABNNotQuoted.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR412 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.PersonalServicesIncome.LabourHireOrOtherSpecifiedPayment.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR413 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.PersonalServicesIncome.Other.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR414 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Expense.PersonalServicesIncome.PrincipalWorkPaymentsToAssociates.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR415 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Expense.PersonalServicesIncome.OtherDeductionsTotal.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR416 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}OrganisationDetails.MainIncomeActivity.Description", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR418 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}OrganisationDetails.OrganisationIndustry2006Extended.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR419 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.PersonalServicesIncome.BusinessActivities.Count", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR421 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}OrganisationDetails.ActivityEvent.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR423 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}OrganisationNameDetails.OrganisationalName.Text", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR425 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Identifiers.AustralianBusinessNumber.Identifier", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR426 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.ABNNotQuotedPrimaryProductionPaymentGross.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR436 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.ABNNotQuotedNonPrimaryProductionPaymentGross.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR437 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.ForeignResidentWithholdingNonPrimaryProduction.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR438 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.VoluntaryAgreementPrimaryProductionGross.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR439 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.VoluntaryAgreementNonPrimaryProductionGross.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR440 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.LabourHireArrangementPrimaryProductionPaymentGross.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR441 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.LabourHireArrangementNonPrimaryProductionPaymentGross.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR442 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.GovernmentIndustryPrimaryProductionPaymentsAssessable.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR443 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.GovernmentIndustryPrimaryProductionPaymentsAssessable.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR444 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.GovernmentIndustryNonPrimaryProductionPaymentsAssessable.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR445 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.GovernmentIndustryNonPrimaryProductionPaymentsAssessable.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR446 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.BusinessPrimaryProductionOther.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR447 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.BusinessNonPrimaryProductionOther.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR448 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Expense.Operating.MotorVehicleTypeOrClaimMethod.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR464 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Expense.Operating.BusinessPrimaryProduction.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR467 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Expense.Operating.BusinessNonPrimaryProduction.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR468 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.BusinessPrimaryProductionNet.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR474 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.BusinessNonPrimaryProductionNet.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR475 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Tax.Losses.BusinessDeferredNonCommercialLossPriorYearPrimaryProduction.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR476 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Tax.Losses.BusinessDeferredNonCommercialLossPriorYearNonPrimaryProduction.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR477 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.Deduction.LowCostAssetDepreciation.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR507 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.Deduction.SmallBusinessPoolDepreciation.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR508 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Remuneration.WagesAndSalaries.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR514 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Remuneration.WagesAndSalariesAction.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR516 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Remuneration.PaymentToRelatedPartiesGross.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR518 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Expense.DepreciatingAssets.IntangibleFirstDeducted.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR520 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Expense.DepreciatingAssets.OtherFirstDeducted.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR522 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Assets.DepreciatingAssets.IntangibleTerminationValue.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR524 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Assets.DepreciatingAssets.OtherTerminationValue.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR526 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Elections.TradingStock.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR528 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Report.CompletionHours.Number", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR530 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPContext
    
            #region sbrRPInstantContext
            IEnumerable<SBRContext> sbrRPInstantContexts = null;
            SBRContext sbrRPInstantContext = null;
    
            sbrRPInstantContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.INSTANT &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 1 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty")) )).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPInstantElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPInstantContexts != null && sbrRPInstantContexts.Count() > 0)
            {
                sbrRPInstantContext = sbrRPInstantContexts.First();
        
                report.RPInstantId = sbrRPInstantContext.Id;
                report.RPInstantIdentifier = sbrRPInstantContext.EntityIdentifier;
                report.RPInstantIdentifierScheme = sbrRPInstantContext.EntityScheme;
        
                report.RPInstantCount = sbrRPInstantContexts.Count();
                report.RPInstantExists = true;
                report.RPInstantOccurrenceIndex = Array.IndexOf(contextArray, sbrRPInstantContext) + 1;
                report.RPInstantLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPInstantContexts.Last()) + 1;
        

                if (report.RPInstantIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPInstantIdentifierTFN = report.RPInstantIdentifier;
    
                report.RPInstantInstantDate = ((SBRInstantPeriod)sbrRPInstantContext.Period).InstantAsDateObject;
        
                sbrPopulatedRPInstantElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPInstantContexts.Count(); i++)
                {
                    sbrRPInstantContext = sbrRPInstantContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPInstantContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPInstantElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPInstantElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPInstantElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPInstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Capital.Losses.CarriedForward.Net.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR313 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPInstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Assets.Inventories.ValuationMethod.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR453 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPInstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Assets.TradeandReceivablesOther.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR510 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPInstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Liabilities.TradeAndOtherPayablesTotal.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR512 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPInstantContext
    
            #region sbrRPPOSContext
            IEnumerable<SBRContext> sbrRPPOSContexts = null;
            SBRContext sbrRPPOSContext = null;
    
            sbrRPPOSContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}AddressUsageDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}POS")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPPOSElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPPOSContexts != null && sbrRPPOSContexts.Count() > 0)
            {
                sbrRPPOSContext = sbrRPPOSContexts.First();
        
                report.RPPOSId = sbrRPPOSContext.Id;
                report.RPPOSIdentifier = sbrRPPOSContext.EntityIdentifier;
                report.RPPOSIdentifierScheme = sbrRPPOSContext.EntityScheme;
        
                report.RPPOSCount = sbrRPPOSContexts.Count();
                report.RPPOSExists = true;
                report.RPPOSOccurrenceIndex = Array.IndexOf(contextArray, sbrRPPOSContext) + 1;
                report.RPPOSLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPPOSContexts.Last()) + 1;
        

                if (report.RPPOSIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPPOSIdentifierTFN = report.RPPOSIdentifier;
    
                report.RPPOSStartDate = ((SBRDurationPeriod)sbrRPPOSContext.Period).DurationStartAsDateObject;
                report.RPPOSEndDate = ((SBRDurationPeriod)sbrRPPOSContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPPOSElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPPOSContexts.Count(); i++)
                {
                    sbrRPPOSContext = sbrRPPOSContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPPOSContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPPOSElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPPOSElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPPOSElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPPOSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}AddressDetails.Line1.Text", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR33 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPPOSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}AddressDetails.Line2.Text", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR34 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPPOSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}AddressDetails.LocalityName.Text", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR35 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPPOSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}AddressDetails.StateOrTerritory.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR36 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPPOSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}AddressDetails.Postcode.Text", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR37 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPPOSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}AddressDetails.Country.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR38 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPPOSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}PersonDemographicDetails.AddressChange.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR39 = ToBoolean(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPPOSContext
    
            #region sbrRPRESContext
            IEnumerable<SBRContext> sbrRPRESContexts = null;
            SBRContext sbrRPRESContext = null;
    
            sbrRPRESContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}AddressUsageDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}RES")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPRESElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPRESContexts != null && sbrRPRESContexts.Count() > 0)
            {
                sbrRPRESContext = sbrRPRESContexts.First();
        
                report.RPRESId = sbrRPRESContext.Id;
                report.RPRESIdentifier = sbrRPRESContext.EntityIdentifier;
                report.RPRESIdentifierScheme = sbrRPRESContext.EntityScheme;
        
                report.RPRESCount = sbrRPRESContexts.Count();
                report.RPRESExists = true;
                report.RPRESOccurrenceIndex = Array.IndexOf(contextArray, sbrRPRESContext) + 1;
                report.RPRESLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPRESContexts.Last()) + 1;
        

                if (report.RPRESIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPRESIdentifierTFN = report.RPRESIdentifier;
    
                report.RPRESStartDate = ((SBRDurationPeriod)sbrRPRESContext.Period).DurationStartAsDateObject;
                report.RPRESEndDate = ((SBRDurationPeriod)sbrRPRESContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPRESElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPRESContexts.Count(); i++)
                {
                    sbrRPRESContext = sbrRPRESContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPRESContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPRESElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPRESElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPRESElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPRESElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}AddressDetails.Line1.Text", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR41 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPRESElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}AddressDetails.Line2.Text", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR42 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPRESElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}AddressDetails.LocalityName.Text", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR43 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPRESElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}AddressDetails.StateOrTerritory.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR44 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPRESElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}AddressDetails.Postcode.Text", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR45 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPRESElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}AddressDetails.Country.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR52 = currentValues[0].Value;
                    }
                } 
            } 
            #endregion End of Context sbrRPRESContext
    
            #region sbrRPBUSContext
            IEnumerable<SBRContext> sbrRPBUSContexts = null;
            SBRContext sbrRPBUSContext = null;
    
            sbrRPBUSContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}AddressUsageDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}BUS")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPBUSElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPBUSContexts != null && sbrRPBUSContexts.Count() > 0)
            {
                sbrRPBUSContext = sbrRPBUSContexts.First();
        
                report.RPBUSId = sbrRPBUSContext.Id;
                report.RPBUSIdentifier = sbrRPBUSContext.EntityIdentifier;
                report.RPBUSIdentifierScheme = sbrRPBUSContext.EntityScheme;
        
                report.RPBUSCount = sbrRPBUSContexts.Count();
                report.RPBUSExists = true;
                report.RPBUSOccurrenceIndex = Array.IndexOf(contextArray, sbrRPBUSContext) + 1;
                report.RPBUSLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPBUSContexts.Last()) + 1;
        

                if (report.RPBUSIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPBUSIdentifierTFN = report.RPBUSIdentifier;
    
                report.RPBUSStartDate = ((SBRDurationPeriod)sbrRPBUSContext.Period).DurationStartAsDateObject;
                report.RPBUSEndDate = ((SBRDurationPeriod)sbrRPBUSContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPBUSElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPBUSContexts.Count(); i++)
                {
                    sbrRPBUSContext = sbrRPBUSContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPBUSContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPBUSElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPBUSElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPBUSElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPBUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}AddressDetails.Line1.Text", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR428 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPBUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}AddressDetails.Line2.Text", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR429 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPBUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}AddressDetails.LocalityName.Text", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR430 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPBUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}AddressDetails.StateOrTerritory.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR431 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPBUSElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}AddressDetails.Postcode.Text", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR432 = currentValues[0].Value;
                    }
                } 
            } 
            #endregion End of Context sbrRPBUSContext
    
            #region sbrRPEligibleContext
            IEnumerable<SBRContext> sbrRPEligibleContexts = null;
            SBRContext sbrRPEligibleContext = null;
    
            sbrRPEligibleContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ExemptionEligibilityTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Eligible")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPEligibleElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPEligibleContexts != null && sbrRPEligibleContexts.Count() > 0)
            {
                sbrRPEligibleContext = sbrRPEligibleContexts.First();
        
                report.RPEligibleId = sbrRPEligibleContext.Id;
                report.RPEligibleIdentifier = sbrRPEligibleContext.EntityIdentifier;
                report.RPEligibleIdentifierScheme = sbrRPEligibleContext.EntityScheme;
        
                report.RPEligibleCount = sbrRPEligibleContexts.Count();
                report.RPEligibleExists = true;
                report.RPEligibleOccurrenceIndex = Array.IndexOf(contextArray, sbrRPEligibleContext) + 1;
                report.RPEligibleLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPEligibleContexts.Last()) + 1;
        

                if (report.RPEligibleIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPEligibleIdentifierTFN = report.RPEligibleIdentifier;
    
                report.RPEligibleStartDate = ((SBRDurationPeriod)sbrRPEligibleContext.Period).DurationStartAsDateObject;
                report.RPEligibleEndDate = ((SBRDurationPeriod)sbrRPEligibleContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPEligibleElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPEligibleContexts.Count(); i++)
                {
                    sbrRPEligibleContext = sbrRPEligibleContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPEligibleContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPEligibleElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPEligibleElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPEligibleElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPEligibleElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.FringeBenefitsReportable.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR490 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPEligibleContext
    
            #region sbrRPIneligibleContext
            IEnumerable<SBRContext> sbrRPIneligibleContexts = null;
            SBRContext sbrRPIneligibleContext = null;
    
            sbrRPIneligibleContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ExemptionEligibilityTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Ineligible")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPIneligibleElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPIneligibleContexts != null && sbrRPIneligibleContexts.Count() > 0)
            {
                sbrRPIneligibleContext = sbrRPIneligibleContexts.First();
        
                report.RPIneligibleId = sbrRPIneligibleContext.Id;
                report.RPIneligibleIdentifier = sbrRPIneligibleContext.EntityIdentifier;
                report.RPIneligibleIdentifierScheme = sbrRPIneligibleContext.EntityScheme;
        
                report.RPIneligibleCount = sbrRPIneligibleContexts.Count();
                report.RPIneligibleExists = true;
                report.RPIneligibleOccurrenceIndex = Array.IndexOf(contextArray, sbrRPIneligibleContext) + 1;
                report.RPIneligibleLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPIneligibleContexts.Last()) + 1;
        

                if (report.RPIneligibleIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPIneligibleIdentifierTFN = report.RPIneligibleIdentifier;
    
                report.RPIneligibleStartDate = ((SBRDurationPeriod)sbrRPIneligibleContext.Period).DurationStartAsDateObject;
                report.RPIneligibleEndDate = ((SBRDurationPeriod)sbrRPIneligibleContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPIneligibleElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPIneligibleContexts.Count(); i++)
                {
                    sbrRPIneligibleContext = sbrRPIneligibleContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPIneligibleContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPIneligibleElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPIneligibleElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPIneligibleElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPIneligibleElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.FringeBenefitsReportable.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR491 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPIneligibleContext
    
            #region sbrRPJFOREIGNContext
            IEnumerable<SBRContext> sbrRPJFOREIGNContexts = null;
            SBRContext sbrRPJFOREIGNContext = null;
    
            sbrRPJFOREIGNContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}JurisdictionDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Foreign")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPJFOREIGNElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPJFOREIGNContexts != null && sbrRPJFOREIGNContexts.Count() > 0)
            {
                sbrRPJFOREIGNContext = sbrRPJFOREIGNContexts.First();
        
                report.RPJFOREIGNId = sbrRPJFOREIGNContext.Id;
                report.RPJFOREIGNIdentifier = sbrRPJFOREIGNContext.EntityIdentifier;
                report.RPJFOREIGNIdentifierScheme = sbrRPJFOREIGNContext.EntityScheme;
        
                report.RPJFOREIGNCount = sbrRPJFOREIGNContexts.Count();
                report.RPJFOREIGNExists = true;
                report.RPJFOREIGNOccurrenceIndex = Array.IndexOf(contextArray, sbrRPJFOREIGNContext) + 1;
                report.RPJFOREIGNLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPJFOREIGNContexts.Last()) + 1;
        

                if (report.RPJFOREIGNIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPJFOREIGNIdentifierTFN = report.RPJFOREIGNIdentifier;
    
                report.RPJFOREIGNStartDate = ((SBRDurationPeriod)sbrRPJFOREIGNContext.Period).DurationStartAsDateObject;
                report.RPJFOREIGNEndDate = ((SBRDurationPeriod)sbrRPJFOREIGNContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPJFOREIGNElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPJFOREIGNContexts.Count(); i++)
                {
                    sbrRPJFOREIGNContext = sbrRPJFOREIGNContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPJFOREIGNContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPJFOREIGNElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPJFOREIGNElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPJFOREIGNElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPJFOREIGNElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.Net.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR494 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPJFOREIGNContext
    
            #region sbrRPSPOUSEContext
            IEnumerable<SBRContext> sbrRPSPOUSEContexts = null;
            SBRContext sbrRPSPOUSEContext = null;
    
            sbrRPSPOUSEContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}SecondaryPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Spouse")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPSPOUSEElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPSPOUSEContexts != null && sbrRPSPOUSEContexts.Count() > 0)
            {
                sbrRPSPOUSEContext = sbrRPSPOUSEContexts.First();
        
                report.RPSPOUSEId = sbrRPSPOUSEContext.Id;
                report.RPSPOUSEIdentifier = sbrRPSPOUSEContext.EntityIdentifier;
                report.RPSPOUSEIdentifierScheme = sbrRPSPOUSEContext.EntityScheme;
        
                report.RPSPOUSECount = sbrRPSPOUSEContexts.Count();
                report.RPSPOUSEExists = true;
                report.RPSPOUSEOccurrenceIndex = Array.IndexOf(contextArray, sbrRPSPOUSEContext) + 1;
                report.RPSPOUSELastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPSPOUSEContexts.Last()) + 1;
        

                if (report.RPSPOUSEIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPSPOUSEIdentifierTFN = report.RPSPOUSEIdentifier;
    
                report.RPSPOUSEStartDate = ((SBRDurationPeriod)sbrRPSPOUSEContext.Period).DurationStartAsDateObject;
                report.RPSPOUSEEndDate = ((SBRDurationPeriod)sbrRPSPOUSEContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPSPOUSEElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPSPOUSEContexts.Count(); i++)
                {
                    sbrRPSPOUSEContext = sbrRPSPOUSEContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPSPOUSEContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPSPOUSEElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPSPOUSEElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPSPOUSEElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPSPOUSEElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}PersonNameDetails.FamilyName.Text", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR213 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPSPOUSEElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}PersonNameDetails.GivenName.Text", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR214 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPSPOUSEElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}PersonNameDetails.OtherGivenName.Text", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR215 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPSPOUSEElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}PersonDemographicDetails.Birth.Date", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR217 = DateTime.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPSPOUSEElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}PersonDemographicDetails.Deceased.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR223 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPSPOUSEElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.Taxable.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR225 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPSPOUSEElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.TrustIncomeNotIncludedInTaxableIncome.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR226 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPSPOUSEElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.FamilyTrustDistributionWhereTaxPaid.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR227 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPSPOUSEElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Pension.Total.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR229 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPSPOUSEElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Pension.SpouseExempt.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR230 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPSPOUSEElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}SuperannuationContribution.EmployerReportable.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR231 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPSPOUSEElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Pension.TaxFree.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR232 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPSPOUSEElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}InternationalDealings.ForeignIncomeTarget.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR233 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPSPOUSEElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Tax.Losses.InvestmentNet.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR234 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPSPOUSEElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}ChildSupport.Payment.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR235 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPSPOUSEElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.TaxableAdjusted.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR237 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPSPOUSEElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Identifiers.FamilyAssistanceOfficeCustomerReferenceNumber.Identifier", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR533 = currentValues[0].Value;
                    }
                } 
            } 
            #endregion End of Context sbrRPSPOUSEContext
    
            #region sbrRPSPOUSEEligibleContext
            IEnumerable<SBRContext> sbrRPSPOUSEEligibleContexts = null;
            SBRContext sbrRPSPOUSEEligibleContext = null;
    
            sbrRPSPOUSEEligibleContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 3 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}SecondaryPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Spouse"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ExemptionEligibilityTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Eligible")) )))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPSPOUSEEligibleElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPSPOUSEEligibleContexts != null && sbrRPSPOUSEEligibleContexts.Count() > 0)
            {
                sbrRPSPOUSEEligibleContext = sbrRPSPOUSEEligibleContexts.First();
        
                report.RPSPOUSEEligibleId = sbrRPSPOUSEEligibleContext.Id;
                report.RPSPOUSEEligibleIdentifier = sbrRPSPOUSEEligibleContext.EntityIdentifier;
                report.RPSPOUSEEligibleIdentifierScheme = sbrRPSPOUSEEligibleContext.EntityScheme;
        
                report.RPSPOUSEEligibleCount = sbrRPSPOUSEEligibleContexts.Count();
                report.RPSPOUSEEligibleExists = true;
                report.RPSPOUSEEligibleOccurrenceIndex = Array.IndexOf(contextArray, sbrRPSPOUSEEligibleContext) + 1;
                report.RPSPOUSEEligibleLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPSPOUSEEligibleContexts.Last()) + 1;
        

                if (report.RPSPOUSEEligibleIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPSPOUSEEligibleIdentifierTFN = report.RPSPOUSEEligibleIdentifier;
    
                report.RPSPOUSEEligibleStartDate = ((SBRDurationPeriod)sbrRPSPOUSEEligibleContext.Period).DurationStartAsDateObject;
                report.RPSPOUSEEligibleEndDate = ((SBRDurationPeriod)sbrRPSPOUSEEligibleContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPSPOUSEEligibleElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPSPOUSEEligibleContexts.Count(); i++)
                {
                    sbrRPSPOUSEEligibleContext = sbrRPSPOUSEEligibleContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPSPOUSEEligibleContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPSPOUSEEligibleElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPSPOUSEEligibleElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPSPOUSEEligibleElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPSPOUSEEligibleElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.FringeBenefitsReportable.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR492 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPSPOUSEEligibleContext
    
            #region sbrRPSPOUSEIneligibleContext
            IEnumerable<SBRContext> sbrRPSPOUSEIneligibleContexts = null;
            SBRContext sbrRPSPOUSEIneligibleContext = null;
    
            sbrRPSPOUSEIneligibleContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 3 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}SecondaryPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Spouse"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ExemptionEligibilityTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Ineligible")) )))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPSPOUSEIneligibleElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPSPOUSEIneligibleContexts != null && sbrRPSPOUSEIneligibleContexts.Count() > 0)
            {
                sbrRPSPOUSEIneligibleContext = sbrRPSPOUSEIneligibleContexts.First();
        
                report.RPSPOUSEIneligibleId = sbrRPSPOUSEIneligibleContext.Id;
                report.RPSPOUSEIneligibleIdentifier = sbrRPSPOUSEIneligibleContext.EntityIdentifier;
                report.RPSPOUSEIneligibleIdentifierScheme = sbrRPSPOUSEIneligibleContext.EntityScheme;
        
                report.RPSPOUSEIneligibleCount = sbrRPSPOUSEIneligibleContexts.Count();
                report.RPSPOUSEIneligibleExists = true;
                report.RPSPOUSEIneligibleOccurrenceIndex = Array.IndexOf(contextArray, sbrRPSPOUSEIneligibleContext) + 1;
                report.RPSPOUSEIneligibleLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPSPOUSEIneligibleContexts.Last()) + 1;
        

                if (report.RPSPOUSEIneligibleIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPSPOUSEIneligibleIdentifierTFN = report.RPSPOUSEIneligibleIdentifier;
    
                report.RPSPOUSEIneligibleStartDate = ((SBRDurationPeriod)sbrRPSPOUSEIneligibleContext.Period).DurationStartAsDateObject;
                report.RPSPOUSEIneligibleEndDate = ((SBRDurationPeriod)sbrRPSPOUSEIneligibleContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPSPOUSEIneligibleElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPSPOUSEIneligibleContexts.Count(); i++)
                {
                    sbrRPSPOUSEIneligibleContext = sbrRPSPOUSEIneligibleContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPSPOUSEIneligibleContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPSPOUSEIneligibleElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPSPOUSEIneligibleElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPSPOUSEIneligibleElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPSPOUSEIneligibleElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.FringeBenefitsReportable.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR493 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPSPOUSEIneligibleContext
    
            #region sbrRPWHMContext
            IEnumerable<SBRContext> sbrRPWHMContexts = null;
            SBRContext sbrRPWHMContext = null;
    
            sbrRPWHMContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}WithholdingPaymentIncomeTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}WorkingHolidayMakers")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPWHMElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPWHMContexts != null && sbrRPWHMContexts.Count() > 0)
            {
                sbrRPWHMContext = sbrRPWHMContexts.First();
        
                report.RPWHMId = sbrRPWHMContext.Id;
                report.RPWHMIdentifier = sbrRPWHMContext.EntityIdentifier;
                report.RPWHMIdentifierScheme = sbrRPWHMContext.EntityScheme;
        
                report.RPWHMCount = sbrRPWHMContexts.Count();
                report.RPWHMExists = true;
                report.RPWHMOccurrenceIndex = Array.IndexOf(contextArray, sbrRPWHMContext) + 1;
                report.RPWHMLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPWHMContexts.Last()) + 1;
        

                if (report.RPWHMIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPWHMIdentifierTFN = report.RPWHMIdentifier;
    
                report.RPWHMStartDate = ((SBRDurationPeriod)sbrRPWHMContext.Period).DurationStartAsDateObject;
                report.RPWHMEndDate = ((SBRDurationPeriod)sbrRPWHMContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPWHMElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPWHMContexts.Count(); i++)
                {
                    sbrRPWHMContext = sbrRPWHMContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPWHMContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPWHMElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPWHMElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPWHMElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPWHMElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.Net.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR499 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPWHMElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Remuneration.WorkingHolidayMakerGross.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR622 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPWHMElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.Deduction.WorkingHolidayMaker.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR623 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPWHMElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}AddressDetails.Country.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR624 = currentValues[0].Value;
                    }
                } 
            } 
            #endregion End of Context sbrRPWHMContext
    
            #region sbrRPPHISeqNumContext
            IEnumerable<SBRContext> sbrRPPHISeqNumContexts = null;
            SBRContext sbrRPPHISeqNumContext = null;
    
            sbrRPPHISeqNumContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.TYPED && dd.Dimension.EndsWith("}PrivateHealthInsuranceSequenceDimension")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPPHISeqNumElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPPHISeqNumContexts != null && sbrRPPHISeqNumContexts.Count() > 0)
            {
        
                report.RPPHISeqNumContextCollection = new List<IITR2026.RPPHISeqNumContext>();
                for (int i = 0; i < sbrRPPHISeqNumContexts.Count(); i++)
                {
                    sbrRPPHISeqNumContext = sbrRPPHISeqNumContexts.ElementAt(i);
                    IITR2026.RPPHISeqNumContext rpphiSeqNumContext = new IITR2026.RPPHISeqNumContext();
                    report.RPPHISeqNumContextCollection.Add(rpphiSeqNumContext);
            
                    rpphiSeqNumContext.Id = sbrRPPHISeqNumContext.Id;
                    rpphiSeqNumContext.Identifier = sbrRPPHISeqNumContext.EntityIdentifier;
                    rpphiSeqNumContext.IdentifierScheme = sbrRPPHISeqNumContext.EntityScheme;
            
                    rpphiSeqNumContext.Count = sbrRPPHISeqNumContexts.Count();
                    rpphiSeqNumContext.Exists = true;
                    rpphiSeqNumContext.OccurrenceIndex = Array.IndexOf(contextArray, sbrRPPHISeqNumContext) + 1;
                    rpphiSeqNumContext.LastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPPHISeqNumContexts.Last()) + 1;
            
    
                    if (rpphiSeqNumContext.IdentifierScheme == "http://www.ato.gov.au/tfn") rpphiSeqNumContext.IdentifierTFN = rpphiSeqNumContext.Identifier;
        
                    rpphiSeqNumContext.StartDate = ((SBRDurationPeriod)sbrRPPHISeqNumContext.Period).DurationStartAsDateObject;
                    rpphiSeqNumContext.EndDate = ((SBRDurationPeriod)sbrRPPHISeqNumContext.Period).DurationEndAsDateObject;
            
                    currentSegment = sbrRPPHISeqNumContext.Segment.Segment;
            
                    currentNamespaceManager = new XmlNamespaceManager(currentSegment.OwnerDocument.NameTable);
                    currentNamespaceManager.AddNamespace("xbrldi", "http://xbrl.org/2006/xbrldi");
            
                    currentNode = currentSegment.SelectSingleNode("xbrldi:typedMember[contains(@dimension, 'PrivateHealthInsuranceSequenceDimension')]", currentNamespaceManager);
            
                    if (currentNode != null)
                    {
                        rpphiSeqNumContext.PrivateHealthInsuranceSequenceDimensionValue = currentNode.InnerText.Trim();
                    }
            
                    sbrPopulatedRPPHISeqNumElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPPHISeqNumContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPPHISeqNumElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPPHISeqNumElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPPHISeqNumElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
                           
                    rpphiSeqNumContext.IITR570 = decimal.Parse(rpphiSeqNumContext.PrivateHealthInsuranceSequenceDimensionValue);
            
                    if (sbrPopulatedRPPHISeqNumElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Identifiers.HealthInsuranceProvider.Identifier", out currentValues))
                    {
                        if (!currentValues[0].IsNil) rpphiSeqNumContext.IITR180 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPPHISeqNumElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Identifiers.HealthInsuranceMembership.Identifier", out currentValues))
                    {
                        if (!currentValues[0].IsNil) rpphiSeqNumContext.IITR181 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPPHISeqNumElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.HealthInsurance.RebatableComponent.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) rpphiSeqNumContext.IITR182 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPPHISeqNumElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.HealthInsurance.GovernmentRebateReceived.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) rpphiSeqNumContext.IITR183 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPPHISeqNumElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.HealthInsurance.Benefit.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) rpphiSeqNumContext.IITR184 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPPHISeqNumElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.HealthInsurance.TaxClaim.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) rpphiSeqNumContext.IITR185 = currentValues[0].Value;
                    }
        } 
            } 
            #endregion End of Context sbrRPPHISeqNumContext
    
            #region sbrRPLOSSSeqNumContext
            IEnumerable<SBRContext> sbrRPLOSSSeqNumContexts = null;
            SBRContext sbrRPLOSSSeqNumContext = null;
    
            sbrRPLOSSSeqNumContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.TYPED && dd.Dimension.EndsWith("}LossesSequenceDimension")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPLOSSSeqNumElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPLOSSSeqNumContexts != null && sbrRPLOSSSeqNumContexts.Count() > 0)
            {
        
                report.RPLOSSSeqNumContextCollection = new List<IITR2026.RPLOSSSeqNumContext>();
                for (int i = 0; i < sbrRPLOSSSeqNumContexts.Count(); i++)
                {
                    sbrRPLOSSSeqNumContext = sbrRPLOSSSeqNumContexts.ElementAt(i);
                    IITR2026.RPLOSSSeqNumContext rplossSeqNumContext = new IITR2026.RPLOSSSeqNumContext();
                    report.RPLOSSSeqNumContextCollection.Add(rplossSeqNumContext);
            
                    rplossSeqNumContext.Id = sbrRPLOSSSeqNumContext.Id;
                    rplossSeqNumContext.Identifier = sbrRPLOSSSeqNumContext.EntityIdentifier;
                    rplossSeqNumContext.IdentifierScheme = sbrRPLOSSSeqNumContext.EntityScheme;
            
                    rplossSeqNumContext.Count = sbrRPLOSSSeqNumContexts.Count();
                    rplossSeqNumContext.Exists = true;
                    rplossSeqNumContext.OccurrenceIndex = Array.IndexOf(contextArray, sbrRPLOSSSeqNumContext) + 1;
                    rplossSeqNumContext.LastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPLOSSSeqNumContexts.Last()) + 1;
            
    
                    if (rplossSeqNumContext.IdentifierScheme == "http://www.ato.gov.au/tfn") rplossSeqNumContext.IdentifierTFN = rplossSeqNumContext.Identifier;
        
                    rplossSeqNumContext.StartDate = ((SBRDurationPeriod)sbrRPLOSSSeqNumContext.Period).DurationStartAsDateObject;
                    rplossSeqNumContext.EndDate = ((SBRDurationPeriod)sbrRPLOSSSeqNumContext.Period).DurationEndAsDateObject;
            
                    currentSegment = sbrRPLOSSSeqNumContext.Segment.Segment;
            
                    currentNamespaceManager = new XmlNamespaceManager(currentSegment.OwnerDocument.NameTable);
                    currentNamespaceManager.AddNamespace("xbrldi", "http://xbrl.org/2006/xbrldi");
            
                    currentNode = currentSegment.SelectSingleNode("xbrldi:typedMember[contains(@dimension, 'LossesSequenceDimension')]", currentNamespaceManager);
            
                    if (currentNode != null)
                    {
                        rplossSeqNumContext.LossesSequenceDimensionValue = currentNode.InnerText.Trim();
                    }
            
                    sbrPopulatedRPLOSSSeqNumElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPLOSSSeqNumContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPLOSSSeqNumElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPLOSSSeqNumElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPLOSSSeqNumElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
                           
                    rplossSeqNumContext.IITR573 = decimal.Parse(rplossSeqNumContext.LossesSequenceDimensionValue);
            
                    if (sbrPopulatedRPLOSSSeqNumElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}OrganisationDetails.MainIncomeActivity.Description", out currentValues))
                    {
                        if (!currentValues[0].IsNil) rplossSeqNumContext.IITR479 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPLOSSSeqNumElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}OrganisationDetails.OrganisationIndustry2006Extended.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) rplossSeqNumContext.IITR480 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPLOSSSeqNumElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}OrganisationDetails.OrganisationType.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) rplossSeqNumContext.IITR481 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPLOSSSeqNumElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}IncomeTax.LossesType.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) rplossSeqNumContext.IITR482 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPLOSSSeqNumElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Tax.ProductRuling.Year", out currentValues))
                    {
                        if (!currentValues[0].IsNil) rplossSeqNumContext.IITR488 = int.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPLOSSSeqNumElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Tax.ProductRulingNumber.Identifier", out currentValues))
                    {
                        if (!currentValues[0].IsNil) rplossSeqNumContext.IITR489 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPLOSSSeqNumElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Tax.PrivateRulingNumber.Identifier", out currentValues))
                    {
                        if (!currentValues[0].IsNil) rplossSeqNumContext.IITR485 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPLOSSSeqNumElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Tax.Losses.PriorYearNonCommercial.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) rplossSeqNumContext.IITR486 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPLOSSSeqNumElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Tax.Losses.Net.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) rplossSeqNumContext.IITR487 = decimal.Parse(currentValues[0].Value);
                    }
        } 
            } 
            #endregion End of Context sbrRPLOSSSeqNumContext
    
            #region sbrRPPartnershipContext
            IEnumerable<SBRContext> sbrRPPartnershipContexts = null;
            SBRContext sbrRPPartnershipContext = null;
    
            sbrRPPartnershipContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}SecondaryPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Partnership")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPPartnershipElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPPartnershipContexts != null && sbrRPPartnershipContexts.Count() > 0)
            {
                sbrRPPartnershipContext = sbrRPPartnershipContexts.First();
        
                report.RPPartnershipId = sbrRPPartnershipContext.Id;
                report.RPPartnershipIdentifier = sbrRPPartnershipContext.EntityIdentifier;
                report.RPPartnershipIdentifierScheme = sbrRPPartnershipContext.EntityScheme;
        
                report.RPPartnershipCount = sbrRPPartnershipContexts.Count();
                report.RPPartnershipExists = true;
                report.RPPartnershipOccurrenceIndex = Array.IndexOf(contextArray, sbrRPPartnershipContext) + 1;
                report.RPPartnershipLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPPartnershipContexts.Last()) + 1;
        

                if (report.RPPartnershipIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPPartnershipIdentifierTFN = report.RPPartnershipIdentifier;
    
                report.RPPartnershipStartDate = ((SBRDurationPeriod)sbrRPPartnershipContext.Period).DurationStartAsDateObject;
                report.RPPartnershipEndDate = ((SBRDurationPeriod)sbrRPPartnershipContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPPartnershipElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPPartnershipContexts.Count(); i++)
                {
                    sbrRPPartnershipContext = sbrRPPartnershipContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPPartnershipContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPPartnershipElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPPartnershipElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPPartnershipElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPPartnershipElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Expense.LandcareOperationsAcceleratedDepreciation.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR245 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPPartnershipElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.SmallBusinessEntityNet.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR558 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPPartnershipContext
    
            #region sbrRPTrustContext
            IEnumerable<SBRContext> sbrRPTrustContexts = null;
            SBRContext sbrRPTrustContext = null;
    
            sbrRPTrustContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}SecondaryPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Trust")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPTrustElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPTrustContexts != null && sbrRPTrustContexts.Count() > 0)
            {
                sbrRPTrustContext = sbrRPTrustContexts.First();
        
                report.RPTrustId = sbrRPTrustContext.Id;
                report.RPTrustIdentifier = sbrRPTrustContext.EntityIdentifier;
                report.RPTrustIdentifierScheme = sbrRPTrustContext.EntityScheme;
        
                report.RPTrustCount = sbrRPTrustContexts.Count();
                report.RPTrustExists = true;
                report.RPTrustOccurrenceIndex = Array.IndexOf(contextArray, sbrRPTrustContext) + 1;
                report.RPTrustLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPTrustContexts.Last()) + 1;
        

                if (report.RPTrustIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPTrustIdentifierTFN = report.RPTrustIdentifier;
    
                report.RPTrustStartDate = ((SBRDurationPeriod)sbrRPTrustContext.Period).DurationStartAsDateObject;
                report.RPTrustEndDate = ((SBRDurationPeriod)sbrRPTrustContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPTrustElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPTrustContexts.Count(); i++)
                {
                    sbrRPTrustContext = sbrRPTrustContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPTrustContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPTrustElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPTrustElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPTrustElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPTrustElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.SmallBusinessEntityNet.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR559 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPTrustContext
    
            #region sbrRPPrimContext
            IEnumerable<SBRContext> sbrRPPrimContexts = null;
            SBRContext sbrRPPrimContext = null;
    
            sbrRPPrimContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}IndustryProductionTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}PrimaryProduction")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPPrimElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPPrimContexts != null && sbrRPPrimContexts.Count() > 0)
            {
                sbrRPPrimContext = sbrRPPrimContexts.First();
        
                report.RPPrimId = sbrRPPrimContext.Id;
                report.RPPrimIdentifier = sbrRPPrimContext.EntityIdentifier;
                report.RPPrimIdentifierScheme = sbrRPPrimContext.EntityScheme;
        
                report.RPPrimCount = sbrRPPrimContexts.Count();
                report.RPPrimExists = true;
                report.RPPrimOccurrenceIndex = Array.IndexOf(contextArray, sbrRPPrimContext) + 1;
                report.RPPrimLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPPrimContexts.Last()) + 1;
        

                if (report.RPPrimIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPPrimIdentifierTFN = report.RPPrimIdentifier;
    
                report.RPPrimStartDate = ((SBRDurationPeriod)sbrRPPrimContext.Period).DurationStartAsDateObject;
                report.RPPrimEndDate = ((SBRDurationPeriod)sbrRPPrimContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPPrimElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPPrimContexts.Count(); i++)
                {
                    sbrRPPrimContext = sbrRPPrimContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPPrimContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPPrimElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPPrimElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPPrimElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Expense.Purchases.AndOtherCosts.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR701 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Expense.Operating.EmploymentCostsTotal.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR703 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}SuperannuationContribution.EmployerContributions.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR704 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Expense.Operating.BadDebts.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR705 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Expense.Operating.Lease.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR706 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Expense.Operating.Rent.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR707 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Expense.Operating.InterestAustralia.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR708 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Expense.Operating.InterestForeign.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR709 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Expense.DepreciationandAmortisation.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR710 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Expense.Operating.MotorVehicle.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR711 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Expense.Operating.RepairsAndMaintenance.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR712 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Expense.Operating.Other.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR713 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Expense.CapitalExpenditureSpecifiedAllowableDeduction.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR714 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Expense.ProjectPoolAllowableDeduction.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR715 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Expense.LandcareOperationsAcceleratedDepreciation.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR716 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.ReconciliationAdjustmentTotal.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR717 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Expense.ReconciliationAdjustments.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR718 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPPrimContext
    
            #region sbrRPPrimInstantContext
            IEnumerable<SBRContext> sbrRPPrimInstantContexts = null;
            SBRContext sbrRPPrimInstantContext = null;
    
            sbrRPPrimInstantContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.INSTANT &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}IndustryProductionTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}PrimaryProduction")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPPrimInstantElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPPrimInstantContexts != null && sbrRPPrimInstantContexts.Count() > 0)
            {
                sbrRPPrimInstantContext = sbrRPPrimInstantContexts.First();
        
                report.RPPrimInstantId = sbrRPPrimInstantContext.Id;
                report.RPPrimInstantIdentifier = sbrRPPrimInstantContext.EntityIdentifier;
                report.RPPrimInstantIdentifierScheme = sbrRPPrimInstantContext.EntityScheme;
        
                report.RPPrimInstantCount = sbrRPPrimInstantContexts.Count();
                report.RPPrimInstantExists = true;
                report.RPPrimInstantOccurrenceIndex = Array.IndexOf(contextArray, sbrRPPrimInstantContext) + 1;
                report.RPPrimInstantLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPPrimInstantContexts.Last()) + 1;
        

                if (report.RPPrimInstantIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPPrimInstantIdentifierTFN = report.RPPrimInstantIdentifier;
    
                report.RPPrimInstantInstantDate = ((SBRInstantPeriod)sbrRPPrimInstantContext.Period).InstantAsDateObject;
        
                sbrPopulatedRPPrimInstantElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPPrimInstantContexts.Count(); i++)
                {
                    sbrRPPrimInstantContext = sbrRPPrimInstantContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPPrimInstantContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPPrimInstantElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPPrimInstantElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPPrimInstantElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPPrimInstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Assets.OpeningStock.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR700 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPPrimInstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Assets.ClosingStock.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR702 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPPrimInstantContext
    
            #region sbrRPNonPrimContext
            IEnumerable<SBRContext> sbrRPNonPrimContexts = null;
            SBRContext sbrRPNonPrimContext = null;
    
            sbrRPNonPrimContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}IndustryProductionTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}NonPrimaryProduction")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPNonPrimElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPNonPrimContexts != null && sbrRPNonPrimContexts.Count() > 0)
            {
                sbrRPNonPrimContext = sbrRPNonPrimContexts.First();
        
                report.RPNonPrimId = sbrRPNonPrimContext.Id;
                report.RPNonPrimIdentifier = sbrRPNonPrimContext.EntityIdentifier;
                report.RPNonPrimIdentifierScheme = sbrRPNonPrimContext.EntityScheme;
        
                report.RPNonPrimCount = sbrRPNonPrimContexts.Count();
                report.RPNonPrimExists = true;
                report.RPNonPrimOccurrenceIndex = Array.IndexOf(contextArray, sbrRPNonPrimContext) + 1;
                report.RPNonPrimLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPNonPrimContexts.Last()) + 1;
        

                if (report.RPNonPrimIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPNonPrimIdentifierTFN = report.RPNonPrimIdentifier;
    
                report.RPNonPrimStartDate = ((SBRDurationPeriod)sbrRPNonPrimContext.Period).DurationStartAsDateObject;
                report.RPNonPrimEndDate = ((SBRDurationPeriod)sbrRPNonPrimContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPNonPrimElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPNonPrimContexts.Count(); i++)
                {
                    sbrRPNonPrimContext = sbrRPNonPrimContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPNonPrimContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPNonPrimElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPNonPrimElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPNonPrimElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPNonPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Expense.Purchases.AndOtherCosts.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR720 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPNonPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Expense.ForeignResidentWithholding.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR722 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPNonPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Expense.Operating.EmploymentCostsTotal.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR723 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPNonPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}SuperannuationContribution.EmployerContributions.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR724 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPNonPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Expense.Operating.BadDebts.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR725 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPNonPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Expense.Operating.Lease.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR726 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPNonPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Expense.Operating.Rent.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR727 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPNonPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Expense.Operating.InterestAustralia.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR728 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPNonPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Expense.Operating.InterestForeign.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR729 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPNonPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Expense.DepreciationandAmortisation.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR730 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPNonPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Expense.Operating.MotorVehicle.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR731 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPNonPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Expense.Operating.RepairsAndMaintenance.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR732 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPNonPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Expense.Operating.Other.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR733 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPNonPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Expense.CapitalExpenditureSpecifiedAllowableDeduction.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR734 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPNonPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Expense.ProjectPoolAllowableDeduction.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR735 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPNonPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Expense.LandcareOperationsAcceleratedDepreciation.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR736 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPNonPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Income.ReconciliationAdjustmentTotal.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR737 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPNonPrimElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Expense.ReconciliationAdjustments.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR738 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPNonPrimContext
    
            #region sbrRPNonPrimInstantContext
            IEnumerable<SBRContext> sbrRPNonPrimInstantContexts = null;
            SBRContext sbrRPNonPrimInstantContext = null;
    
            sbrRPNonPrimInstantContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.INSTANT &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}IndustryProductionTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}NonPrimaryProduction")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPNonPrimInstantElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPNonPrimInstantContexts != null && sbrRPNonPrimInstantContexts.Count() > 0)
            {
                sbrRPNonPrimInstantContext = sbrRPNonPrimInstantContexts.First();
        
                report.RPNonPrimInstantId = sbrRPNonPrimInstantContext.Id;
                report.RPNonPrimInstantIdentifier = sbrRPNonPrimInstantContext.EntityIdentifier;
                report.RPNonPrimInstantIdentifierScheme = sbrRPNonPrimInstantContext.EntityScheme;
        
                report.RPNonPrimInstantCount = sbrRPNonPrimInstantContexts.Count();
                report.RPNonPrimInstantExists = true;
                report.RPNonPrimInstantOccurrenceIndex = Array.IndexOf(contextArray, sbrRPNonPrimInstantContext) + 1;
                report.RPNonPrimInstantLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPNonPrimInstantContexts.Last()) + 1;
        

                if (report.RPNonPrimInstantIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPNonPrimInstantIdentifierTFN = report.RPNonPrimInstantIdentifier;
    
                report.RPNonPrimInstantInstantDate = ((SBRInstantPeriod)sbrRPNonPrimInstantContext.Period).InstantAsDateObject;
        
                sbrPopulatedRPNonPrimInstantElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPNonPrimInstantContexts.Count(); i++)
                {
                    sbrRPNonPrimInstantContext = sbrRPNonPrimInstantContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPNonPrimInstantContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPNonPrimInstantElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPNonPrimInstantElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPNonPrimInstantElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPNonPrimInstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Assets.OpeningStock.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR719 = decimal.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPNonPrimInstantElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Assets.ClosingStock.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR721 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPNonPrimInstantContext
    
            #region sbrRPInvestorContext
            IEnumerable<SBRContext> sbrRPInvestorContexts = null;
            SBRContext sbrRPInvestorContext = null;
    
            sbrRPInvestorContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}EarlyInvestorDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Investor")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPInvestorElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPInvestorContexts != null && sbrRPInvestorContexts.Count() > 0)
            {
                sbrRPInvestorContext = sbrRPInvestorContexts.First();
        
                report.RPInvestorId = sbrRPInvestorContext.Id;
                report.RPInvestorIdentifier = sbrRPInvestorContext.EntityIdentifier;
                report.RPInvestorIdentifierScheme = sbrRPInvestorContext.EntityScheme;
        
                report.RPInvestorCount = sbrRPInvestorContexts.Count();
                report.RPInvestorExists = true;
                report.RPInvestorOccurrenceIndex = Array.IndexOf(contextArray, sbrRPInvestorContext) + 1;
                report.RPInvestorLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPInvestorContexts.Last()) + 1;
        

                if (report.RPInvestorIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPInvestorIdentifierTFN = report.RPInvestorIdentifier;
    
                report.RPInvestorStartDate = ((SBRDurationPeriod)sbrRPInvestorContext.Period).DurationStartAsDateObject;
                report.RPInvestorEndDate = ((SBRDurationPeriod)sbrRPInvestorContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPInvestorElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPInvestorContexts.Count(); i++)
                {
                    sbrRPInvestorContext = sbrRPInvestorContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPInvestorContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPInvestorElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPInvestorElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPInvestorElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPInvestorElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}TaxOffsetClaim.NonRefundableOther.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR496 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPInvestorContext
    
            #region sbrRPInvestorYPreviousContext
            IEnumerable<SBRContext> sbrRPInvestorYPreviousContexts = null;
            SBRContext sbrRPInvestorYPreviousContext = null;
    
            sbrRPInvestorYPreviousContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 3 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}EarlyInvestorDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Investor"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}RelativePeriodDurationDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}YPrevious")) )))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPInvestorYPreviousElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPInvestorYPreviousContexts != null && sbrRPInvestorYPreviousContexts.Count() > 0)
            {
                sbrRPInvestorYPreviousContext = sbrRPInvestorYPreviousContexts.First();
        
                report.RPInvestorYPreviousId = sbrRPInvestorYPreviousContext.Id;
                report.RPInvestorYPreviousIdentifier = sbrRPInvestorYPreviousContext.EntityIdentifier;
                report.RPInvestorYPreviousIdentifierScheme = sbrRPInvestorYPreviousContext.EntityScheme;
        
                report.RPInvestorYPreviousCount = sbrRPInvestorYPreviousContexts.Count();
                report.RPInvestorYPreviousExists = true;
                report.RPInvestorYPreviousOccurrenceIndex = Array.IndexOf(contextArray, sbrRPInvestorYPreviousContext) + 1;
                report.RPInvestorYPreviousLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPInvestorYPreviousContexts.Last()) + 1;
        

                if (report.RPInvestorYPreviousIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPInvestorYPreviousIdentifierTFN = report.RPInvestorYPreviousIdentifier;
    
                report.RPInvestorYPreviousStartDate = ((SBRDurationPeriod)sbrRPInvestorYPreviousContext.Period).DurationStartAsDateObject;
                report.RPInvestorYPreviousEndDate = ((SBRDurationPeriod)sbrRPInvestorYPreviousContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPInvestorYPreviousElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPInvestorYPreviousContexts.Count(); i++)
                {
                    sbrRPInvestorYPreviousContext = sbrRPInvestorYPreviousContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPInvestorYPreviousContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPInvestorYPreviousElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPInvestorYPreviousElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPInvestorYPreviousElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPInvestorYPreviousElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}TaxOffsetClaim.NonRefundableOther.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR501 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPInvestorYPreviousContext
    
            #region sbrRPLimitedPartnersContext
            IEnumerable<SBRContext> sbrRPLimitedPartnersContexts = null;
            SBRContext sbrRPLimitedPartnersContext = null;
    
            sbrRPLimitedPartnersContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}EarlyInvestorDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}LimitedPartners")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPLimitedPartnersElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPLimitedPartnersContexts != null && sbrRPLimitedPartnersContexts.Count() > 0)
            {
                sbrRPLimitedPartnersContext = sbrRPLimitedPartnersContexts.First();
        
                report.RPLimitedPartnersId = sbrRPLimitedPartnersContext.Id;
                report.RPLimitedPartnersIdentifier = sbrRPLimitedPartnersContext.EntityIdentifier;
                report.RPLimitedPartnersIdentifierScheme = sbrRPLimitedPartnersContext.EntityScheme;
        
                report.RPLimitedPartnersCount = sbrRPLimitedPartnersContexts.Count();
                report.RPLimitedPartnersExists = true;
                report.RPLimitedPartnersOccurrenceIndex = Array.IndexOf(contextArray, sbrRPLimitedPartnersContext) + 1;
                report.RPLimitedPartnersLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPLimitedPartnersContexts.Last()) + 1;
        

                if (report.RPLimitedPartnersIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPLimitedPartnersIdentifierTFN = report.RPLimitedPartnersIdentifier;
    
                report.RPLimitedPartnersStartDate = ((SBRDurationPeriod)sbrRPLimitedPartnersContext.Period).DurationStartAsDateObject;
                report.RPLimitedPartnersEndDate = ((SBRDurationPeriod)sbrRPLimitedPartnersContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPLimitedPartnersElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPLimitedPartnersContexts.Count(); i++)
                {
                    sbrRPLimitedPartnersContext = sbrRPLimitedPartnersContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPLimitedPartnersContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPLimitedPartnersElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPLimitedPartnersElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPLimitedPartnersElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPLimitedPartnersElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}TaxOffsetClaim.NonRefundableOther.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR495 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPLimitedPartnersContext
    
            #region sbrRPLimitedPartnersYPreviousContext
            IEnumerable<SBRContext> sbrRPLimitedPartnersYPreviousContexts = null;
            SBRContext sbrRPLimitedPartnersYPreviousContext = null;
    
            sbrRPLimitedPartnersYPreviousContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 3 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}EarlyInvestorDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}LimitedPartners"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}RelativePeriodDurationDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}YPrevious")) )))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPLimitedPartnersYPreviousElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPLimitedPartnersYPreviousContexts != null && sbrRPLimitedPartnersYPreviousContexts.Count() > 0)
            {
                sbrRPLimitedPartnersYPreviousContext = sbrRPLimitedPartnersYPreviousContexts.First();
        
                report.RPLimitedPartnersYPreviousId = sbrRPLimitedPartnersYPreviousContext.Id;
                report.RPLimitedPartnersYPreviousIdentifier = sbrRPLimitedPartnersYPreviousContext.EntityIdentifier;
                report.RPLimitedPartnersYPreviousIdentifierScheme = sbrRPLimitedPartnersYPreviousContext.EntityScheme;
        
                report.RPLimitedPartnersYPreviousCount = sbrRPLimitedPartnersYPreviousContexts.Count();
                report.RPLimitedPartnersYPreviousExists = true;
                report.RPLimitedPartnersYPreviousOccurrenceIndex = Array.IndexOf(contextArray, sbrRPLimitedPartnersYPreviousContext) + 1;
                report.RPLimitedPartnersYPreviousLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPLimitedPartnersYPreviousContexts.Last()) + 1;
        

                if (report.RPLimitedPartnersYPreviousIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPLimitedPartnersYPreviousIdentifierTFN = report.RPLimitedPartnersYPreviousIdentifier;
    
                report.RPLimitedPartnersYPreviousStartDate = ((SBRDurationPeriod)sbrRPLimitedPartnersYPreviousContext.Period).DurationStartAsDateObject;
                report.RPLimitedPartnersYPreviousEndDate = ((SBRDurationPeriod)sbrRPLimitedPartnersYPreviousContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPLimitedPartnersYPreviousElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPLimitedPartnersYPreviousContexts.Count(); i++)
                {
                    sbrRPLimitedPartnersYPreviousContext = sbrRPLimitedPartnersYPreviousContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPLimitedPartnersYPreviousContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPLimitedPartnersYPreviousElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPLimitedPartnersYPreviousElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPLimitedPartnersYPreviousElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
            
                    if (sbrPopulatedRPLimitedPartnersYPreviousElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}TaxOffsetClaim.NonRefundableOther.Amount", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR500 = decimal.Parse(currentValues[0].Value);
                    }
                } 
            } 
            #endregion End of Context sbrRPLimitedPartnersYPreviousContext
    
            #region sbrRPTrueAndCorrectContext
            IEnumerable<SBRContext> sbrRPTrueAndCorrectContexts = null;
            SBRContext sbrRPTrueAndCorrectContext = null;
    
            sbrRPTrueAndCorrectContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}ReportingParty"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}DeclarationTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}TrueAndCorrect")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedRPTrueAndCorrectElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrRPTrueAndCorrectContexts != null && sbrRPTrueAndCorrectContexts.Count() > 0)
            {
                sbrRPTrueAndCorrectContext = sbrRPTrueAndCorrectContexts.First();
        
                report.RPTrueAndCorrectId = sbrRPTrueAndCorrectContext.Id;
                report.RPTrueAndCorrectIdentifier = sbrRPTrueAndCorrectContext.EntityIdentifier;
                report.RPTrueAndCorrectIdentifierScheme = sbrRPTrueAndCorrectContext.EntityScheme;
        
                report.RPTrueAndCorrectCount = sbrRPTrueAndCorrectContexts.Count();
                report.RPTrueAndCorrectExists = true;
                report.RPTrueAndCorrectOccurrenceIndex = Array.IndexOf(contextArray, sbrRPTrueAndCorrectContext) + 1;
                report.RPTrueAndCorrectLastOccurrenceIndex = Array.IndexOf(contextArray, sbrRPTrueAndCorrectContexts.Last()) + 1;
        

                if (report.RPTrueAndCorrectIdentifierScheme == "http://www.ato.gov.au/tfn") report.RPTrueAndCorrectIdentifierTFN = report.RPTrueAndCorrectIdentifier;
    
                report.RPTrueAndCorrectStartDate = ((SBRDurationPeriod)sbrRPTrueAndCorrectContext.Period).DurationStartAsDateObject;
                report.RPTrueAndCorrectEndDate = ((SBRDurationPeriod)sbrRPTrueAndCorrectContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedRPTrueAndCorrectElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrRPTrueAndCorrectContexts.Count(); i++)
                {
                    sbrRPTrueAndCorrectContext = sbrRPTrueAndCorrectContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrRPTrueAndCorrectContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedRPTrueAndCorrectElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedRPTrueAndCorrectElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedRPTrueAndCorrectElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
                    report.IITR549 = report.RPTrueAndCorrectIdentifier;
            
                    if (sbrPopulatedRPTrueAndCorrectElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}PersonUnstructuredName.FullName.Text", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR550 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPTrueAndCorrectElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}PersonNameDetails.Position.Text", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR551 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPTrueAndCorrectElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}ElectronicContact.Telephone.Country.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR580 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPTrueAndCorrectElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}ElectronicContact.Telephone.Minimal.Number", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR553 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedRPTrueAndCorrectElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Declaration.Signature.Date", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR554 = DateTime.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPTrueAndCorrectElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Declaration.StatementAccepted.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR555 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedRPTrueAndCorrectElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Declaration.SignatoryIdentifier.Text", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR556 = currentValues[0].Value;
                    }
                } 
            } 
            #endregion End of Context sbrRPTrueAndCorrectContext
    
            #region sbrINTTrueAndCorrectContext
            IEnumerable<SBRContext> sbrINTTrueAndCorrectContexts = null;
            SBRContext sbrINTTrueAndCorrectContext = null;
    
            sbrINTTrueAndCorrectContexts = contextArray.Where(
                c => c.Period != null && c.Period.PeriodType == SBRPeriod_PeriodType.DURATION &&
                c.Segment != null && c.Segment.ExplicitOrTypedDimensionDomains.Count == 2 && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}ReportPartyTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}Intermediary"))  && 
                (c.Segment.ExplicitOrTypedDimensionDomains.Any(dd => dd.DimensionType == SBRExplicitOrTypedDimensionDomain_Type.EXPLICIT && dd.Dimension.EndsWith("}DeclarationTypeDimension") && ((SBRExplicitDimensionDomain)dd).Domain.EndsWith("}TrueAndCorrect")) ))).ToList();
            Dictionary<string, IList<SBRElement>> sbrPopulatedINTTrueAndCorrectElementsMap = new Dictionary<string, IList<SBRElement>>();
            if (sbrINTTrueAndCorrectContexts != null && sbrINTTrueAndCorrectContexts.Count() > 0)
            {
                sbrINTTrueAndCorrectContext = sbrINTTrueAndCorrectContexts.First();
        
                report.INTTrueAndCorrectId = sbrINTTrueAndCorrectContext.Id;
                report.INTTrueAndCorrectIdentifier = sbrINTTrueAndCorrectContext.EntityIdentifier;
                report.INTTrueAndCorrectIdentifierScheme = sbrINTTrueAndCorrectContext.EntityScheme;
        
                report.INTTrueAndCorrectCount = sbrINTTrueAndCorrectContexts.Count();
                report.INTTrueAndCorrectExists = true;
                report.INTTrueAndCorrectOccurrenceIndex = Array.IndexOf(contextArray, sbrINTTrueAndCorrectContext) + 1;
                report.INTTrueAndCorrectLastOccurrenceIndex = Array.IndexOf(contextArray, sbrINTTrueAndCorrectContexts.Last()) + 1;
        

                if (report.INTTrueAndCorrectIdentifierScheme == "http://www.abr.gov.au/abn") report.INTTrueAndCorrectIdentifierABN = report.INTTrueAndCorrectIdentifier;
    
                report.INTTrueAndCorrectStartDate = ((SBRDurationPeriod)sbrINTTrueAndCorrectContext.Period).DurationStartAsDateObject;
                report.INTTrueAndCorrectEndDate = ((SBRDurationPeriod)sbrINTTrueAndCorrectContext.Period).DurationEndAsDateObject;
        
                sbrPopulatedINTTrueAndCorrectElementsMap = new System.Collections.Generic.Dictionary<string, IList<SBRElement>>();
                for (int i = 0; i < sbrINTTrueAndCorrectContexts.Count(); i++)
                {
                    sbrINTTrueAndCorrectContext = sbrINTTrueAndCorrectContexts.ElementAt(i);
            
                    IList<SBRElement> elements = sbrDocument.ElementsByContextId[sbrINTTrueAndCorrectContext.Id];
                    if (elements != null)
                    {
                        for (int e = 0; e < elements.Count(); e++)
                        {
                            SBRElement currentElement = elements[e];
                            if (currentElement == null || currentElement.IsInTuple) continue; // Don't add tuple elements here, they will be found through a different map.
                            if (sbrPopulatedINTTrueAndCorrectElementsMap.ContainsKey("{" + currentElement.Namespace + "}" + currentElement.Name))
                            {
                                sbrPopulatedINTTrueAndCorrectElementsMap["{" + currentElement.Namespace + "}" + currentElement.Name].Add(currentElement);
                            }
                            else
                            {
                                sbrPopulatedINTTrueAndCorrectElementsMap.Add("{" + currentElement.Namespace + "}" + currentElement.Name, new List<SBRElement> { currentElement } );
                            }
                        }
                    }
                    report.IITR537 = report.INTTrueAndCorrectIdentifier;
            
                    if (sbrPopulatedINTTrueAndCorrectElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}PersonUnstructuredName.FullName.Text", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR538 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedINTTrueAndCorrectElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}PersonNameDetails.Position.Text", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR539 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedINTTrueAndCorrectElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}ElectronicContact.Telephone.Country.Code", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR581 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedINTTrueAndCorrectElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}ElectronicContact.Telephone.Minimal.Number", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR541 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedINTTrueAndCorrectElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Declaration.Signature.Date", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR543 = DateTime.Parse(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedINTTrueAndCorrectElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Declaration.StatementAccepted.Indicator", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR544 = ToBoolean(currentValues[0].Value);
                    }
            
                    if (sbrPopulatedINTTrueAndCorrectElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Declaration.SignatoryIdentifier.Text", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR545 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedINTTrueAndCorrectElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Identifiers.TaxAgentClientReference.Text", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR536 = currentValues[0].Value;
                    }
            
                    if (sbrPopulatedINTTrueAndCorrectElementsMap.TryGetValue("{http://www.sbr.gov.au/ato/iitr}Identifiers.TaxAgentNumber.Identifier", out currentValues))
                    {
                        if (!currentValues[0].IsNil) report.IITR542 = currentValues[0].Value;
                    }
                } 
            } 
            #endregion End of Context sbrINTTrueAndCorrectContext
            return report;

        } // Of Consume Method
    } // Of Class
} // Of Namespace
