using System;
using System.Collections.Generic;
using System.Linq;
using System.Runtime.CompilerServices;
using DataContracts;
namespace Ato.EN.IntegrationServices.CodeGenerationINCDTLS
{
    using Income = INCDTLS2026.Rp_Income;
    using GovernmentBenefit = INCDTLS2026.Rp_Income.Rp_Income_GovernmentBenefit;

    public partial class INCDTLS2026Validator
    {
        #region INCDTLS Helper Functions

        // Government Benefit
        internal static List<GovernmentBenefit> GetAllGovernmentBenefitTuples(INCDTLS2026 report)
        {
            List<GovernmentBenefit> response = new List<GovernmentBenefit>();

            if (report != null && Count(report.Rp_IncomeCollection) != 0)
            {
                response = report.Rp_IncomeCollection.SelectMany(income => income.Rp_Income_GovernmentBenefitCollection ?? new List<GovernmentBenefit>()).Distinct().ToList();
            }

            return response;
        }

        // Calculate amount of Total amount of Lump Sum In Arrears that occur within INCDTLS form
        internal static int GetTotalNumberOfLumpSumInArrears(INCDTLS2026 report)
        {
            int lumpSumTotalCounter = 0;
            int lumpSumSoWCounter = 0;
            int lumpSumFEICounter = 0;
            int lumpSumSISCounter = 0;
            int lumpSumFEINPSCounter = 0;
            int lumpSumFPACounter = 0;

            // Checks to see with the Income Collection exists
            if (report.Rp_IncomeCollection != null)
            {
                // Checks to see a Salary or Wage Lump Sum Collection exists
                if (report.Rp_IncomeCollection.Any(income => income.Rp_Income_OrganisationNameDetails_SalaryOrWages_LumpSumArrearsCollection != null))
                {
                    foreach (Income incomeTuple in report.Rp_IncomeCollection)
                    {
                        if (incomeTuple.Rp_Income_OrganisationNameDetails_SalaryOrWages_LumpSumArrearsCollection != null)
                            lumpSumSoWCounter += incomeTuple.Rp_Income_OrganisationNameDetails_SalaryOrWages_LumpSumArrearsCollection.Count(lumpSumTuple => lumpSumTuple.Rp_Income_OrganisationNameDetails_SalaryOrWages_LumpSumArrears_LumpSumArrearsPaymentCollection != null);
                    }
                }

                // Count all the FEIPS Lump Sum In Arrears
                lumpSumFEICounter = report.Rp_IncomeCollection.Count(income => income.Rp_Income_OrganisationNameDetails_ForeignEmployment_FEIPSLumpSumArrearsPaymentCollection != null);

                // Count all the SIS Lump Sum In Arrears
                lumpSumSISCounter = report.Rp_IncomeCollection.Count(income => income.Rp_Income_OrganisationNameDetails_AnnuitiesAndSuperannuation_SISLumpSumArrearsPaymentCollection != null);
            }

            // Checks to see whether FEINPS Collection exists
            if (report.Rp_ForeignIncomeCollection != null)
            {
                // Count all the FEINPS Lump Sum In Arrears
                lumpSumFEINPSCounter = report.Rp_ForeignIncomeCollection.Count(foreignIncome => foreignIncome.Rp_ForeignIncome_FEINPSLumpSumArrearsPaymentCollection != null);
            }

            // Checks to see whether Foreign PA Collection exists
            if (report.Rp_ForeignPensionsOrAnnuitiesCollection != null)
            {
                // Count all the FPA Lump Sum In Arrears
                lumpSumFPACounter = report.Rp_ForeignPensionsOrAnnuitiesCollection.Count(foreignPA => foreignPA.Rp_ForeignPensionsOrAnnuities_FPALumpSumArrearsPaymentCollection != null);
            }

            // Adds each of the Lump Sum Counters and set to a total amount
            lumpSumTotalCounter = lumpSumSoWCounter + lumpSumFEICounter + lumpSumSISCounter + lumpSumFEINPSCounter + lumpSumFPACounter;

            return lumpSumTotalCounter;
        }

        /// <summary>
        /// A method which returns the count of ALL INCDTLS150 tuples in the INCDTLS child report.
        /// </summary>
        /// <param name="report">The INCDTLS child report.</param>
        /// <returns>The count of ALL INCDTLS150 tuples in the INCDTLS child report.</returns>
        internal static int GetCountOfAllINCDTLS150Tuples(INCDTLS2026 reportIn)
        {
            // Count(^INCDTLS150)
            int INCDTLS150Count =
            (
                reportIn.Rp_IncomeCollection == null
                ? 0
                : reportIn.Rp_IncomeCollection.Sum
                (
                    // Salary or wages lump sum in arrears
                    INCDTLS150Repeat => INCDTLS150Repeat.Rp_Income_OrganisationNameDetails_SalaryOrWages_LumpSumArrearsCollection == null
                    ? 0
                    : INCDTLS150Repeat.Rp_Income_OrganisationNameDetails_SalaryOrWages_LumpSumArrearsCollection.Count()
                )
            );

            return INCDTLS150Count;
        }

        /// <summary>
        /// A method which returns the count of all INCDTLS160 tuples in the INCDTLS child report which contain at least one INCDTLS522 tuple.
        /// </summary>
        /// <param name="reportIn">The INCDTLS child report.</param>
        /// <returns>A count of all INCDTLS160 tuples in the INCDTLS child report which contain at least one INCDTLS522 tuple.</returns>
        internal static int GetCountOfAllINCDTLS160TuplesContainingINCDTLS522Tuples(INCDTLS2026 reportIn)
        {
            // CountOccurrence(^ INCDTLS160, Count(^ INCDTLS522) > 0)
            int INCDTLS160Count =
            (
                reportIn.Rp_IncomeCollection == null
                ? 0
                : reportIn.Rp_IncomeCollection.Count
                (
                    // Foreign employment income payment summary lump sum in arrears payment
                    INCDTLS160Repeat => Count(INCDTLS160Repeat.Rp_Income_OrganisationNameDetails_ForeignEmployment_FEIPSLumpSumArrearsPaymentCollectionCount) > 0
                )
            );

            return INCDTLS160Count;
        }


        /// <summary>
        /// A method which returns the count of all INCDTLS108 tuples in the INCDTLS child report which contain at least one INCDTLS525 tuple.
        /// </summary>
        /// <param name="reportIn">The INCDTLS child report.</param>
        /// <returns>A count of all INCDTLS108 tuples in the INCDTLS child report which contain at least one INCDTLS525 tuple.</returns>
        internal static int GetCountOfAllINCDTLS108TuplesContainingINCDTLS525Tuples(INCDTLS2026 reportIn)
        {
            // CountOccurrence(^ INCDTLS108, Count(^ INCDTLS525) > 0)
            int INCDTLS108Count =
            (
                reportIn.Rp_IncomeCollection == null
                ? 0
                : reportIn.Rp_IncomeCollection.Count
                (
                    // Australian superannuation income stream payment summary
                    INCDTLS108Repeat => Count(INCDTLS108Repeat.Rp_Income_OrganisationNameDetails_AnnuitiesAndSuperannuation_SISLumpSumArrearsPaymentCollectionCount) > 0
                )
            );

            return INCDTLS108Count;
        }

        /// <summary>
        /// A method which returns the count of all INCDTLS185 tuples in the INCDTLS child report which contain at least one INCDTLS528 tuple.
        /// </summary>
        /// <param name="reportIn">The INCDTLS child report.</param>
        /// <returns>A count of all INCDTLS185 tuples in the INCDTLS child report which contain at least one INCDTLS528 tuple.</returns>
        internal static int GetCountOfAllINCDTLS185TuplesContainingINCDTLS528Tuples(INCDTLS2026 reportIn)
        {
            // CountOccurrence(^ INCDTLS185, Count(^ INCDTLS528) > 0)
            int INCDTLS185Count =
            (
                // Foreign employment income non-payment summary
                reportIn.Rp_ForeignIncomeCollection == null
                ? 0
                : reportIn.Rp_ForeignIncomeCollection.Count
                (
                    // Foreign employment income payment summary lump sum in arrears payment
                    INCDTLS185Repeat => Count(INCDTLS185Repeat.Rp_ForeignIncome_FEINPSLumpSumArrearsPaymentCollectionCount) > 0
                )
            );

            return INCDTLS185Count;
        }

        /// <summary>
        /// A method which returns the count of all INCDTLS207 tuples in the INCDTLS child report which contain at least one INCDTLS531 tuple.
        /// </summary>
        /// <param name="reportIn">The INCDTLS child report.</param>
        /// <returns>A count of all INCDTLS207 tuples in the INCDTLS child report which contain at least one INCDTLS531 tuple.</returns>
        internal static int GetCountOfAllINCDTLS207TuplesContainingINCDTLS531Tuples(INCDTLS2026 reportIn)
        {
            // CountOccurrence(^ INCDTLS207, Count(^ INCDTLS531) > 0)) < 15
            int INCDTLS207Count =
            (
                // Foreign pensions or annuities
                reportIn.Rp_ForeignPensionsOrAnnuitiesCollection == null
                ? 0
                : reportIn.Rp_ForeignPensionsOrAnnuitiesCollection.Count
                (
                    // Foreign pensions or annuities lump sum in arrears payment
                    INCDTLS207Repeat => Count(INCDTLS207Repeat.Rp_ForeignPensionsOrAnnuities_FPALumpSumArrearsPaymentCollectionCount) > 0
                )
            );

            return INCDTLS207Count;
        }

        #endregion

        private const decimal EtpCapForLifeDeathBenefitThreshold = 245000;
        private const decimal EtpTaxableComponent = 0.32M;
        private readonly static string[] _australianCountryCodes = new[] { "au", "cc", "cx", "nf" };
        private readonly static Dictionary<string, string> _australianCountryCodesDictionary = new List<string>(_australianCountryCodes).ToDictionary(c => c);
        private readonly static Dictionary<string, string> _australianAndTlCountryCodesDictionary = new List<string>(_australianCountryCodes) { "tl" }.ToDictionary(c => c);
        private readonly static Dictionary<string, string> _australianAndHmCountryCodesDictionary = new List<string>(_australianCountryCodes) { "hm" }.ToDictionary(c => c);

        #region Manual Rules - Rules marked for manual coding

        #region VR.ATO.INCDTLS.000019

        /*  VR.ATO.INCDTLS.000019
                    The number of Australian superannuation income stream payment summary instances must not exceed 20

                Legacy Rule Format:
                    (Count(^INCDTLS108) > 20)

            Technical Business Rule Format:
                    (Count(^INCDTLS108) > 20)

        Data Elements:

            ^INCDTLS108 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation
            */
        protected void VRATOINCDTLS000019(List<ProcessMessageDocument> response, INCDTLS2026 report)
        {
            if (report.Rp_IncomeCollection != null)
            {
                ProcessMessageDocument processMessage;
                bool assertion;

                assertion = report.Rp_IncomeCollection.Count(t => t.Rp_Income_OrganisationNameDetails_AnnuitiesAndSuperannuationCollectionExists == true) > 20;
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.INCDTLS.000019",
                        Severity = ProcessMessageSeverity.Error,
                        Description = @"The number of Australian superannuation income stream payment summary instances must not exceed 20",
                        Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income/tns:OrganisationNameDetails/tns:AnnuitiesAndSuperannuation",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.INCDTLS.000019" } },
                    };

                    /*  Parameters are commented out or non-existant for manual rules 

                    */
                    response.Add(processMessage);
                }
            }
        }
        #endregion // VR.ATO.INCDTLS.000019

        #region VR.ATO.INCDTLS.000023

        /*  VR.ATO.INCDTLS.000023
            The number of Australian government benefit types must not exceed 10 per benefit type

            Legacy Rule Format:
            (CountOccurrence(^INCDTLS126, ^INCDTLS126 = "Allowance") > 10 OR CountOccurrence(^INCDTLS126, ^INCDTLS126 = "Pension") > 10 OR CountOccurrence(^INCDTLS126, ^INCDTLS126 = "Special") > 10)

            Technical Business Rule Format:
            (CountOccurrence(^INCDTLS126, ^INCDTLS126 = 'Allowance') > 10 OR CountOccurrence(^INCDTLS126, ^INCDTLS126 = 'Pension') > 10 OR CountOccurrence(^INCDTLS126, ^INCDTLS126 = 'Special') > 10)

            Data Elements:

            ^INCDTLS126 = INCDTLS:Rp:Income:GovernmentBenefit:Payment.Type.Code
        */
        protected void VRATOINCDTLS000023(List<ProcessMessageDocument> response, INCDTLS2026 report)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            List<GovernmentBenefit> gbTupleCollection = GetAllGovernmentBenefitTuples(report);
            int allowance = 0;
            int pension = 0;
            int special = 0;

            foreach (GovernmentBenefit gbTuple in gbTupleCollection)
            {
                if (gbTuple.INCDTLS126.Equals("Allowance"))
                    allowance++;
                else if (gbTuple.INCDTLS126.Equals("Pension"))
                    pension++;
                else if (gbTuple.INCDTLS126.Equals("Special"))
                    special++;
            }

            assertion = allowance > 10 || pension > 10 || special > 10;
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The number of Australian government benefit types must not exceed 10 per benefit type",
                    LongDescription = @"Where an Australian government benefit payment summary instance is present, Australian government benefit type must be provided",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income/tns:GovernmentBenefitCollection/tns:GovernmentBenefit/tns:PaymentTypeC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.INCDTLS.000023" } },
                };

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.INCDTLS.000023

        #region VR.ATO.INCDTLS.000100

        /*  VR.ATO.INCDTLS.000100
                    The total number of Salary or wages payment summary instances plus Australian annuities payment summary instances must not exceed 57

                Legacy Rule Format:
                    ((Count(^INCDTLS309) + Count(^INCDTLS145)) > 57)

            Technical Business Rule Format:
                    ((Count(^INCDTLS309) + Count(^INCDTLS145)) > 57)

        Data Elements:

            ^INCDTLS309 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages

            ^INCDTLS145 = INCDTLS:Rp:Income:OrganisationNameDetails:AustralianAnnuities
            */
        protected void VRATOINCDTLS000100(List<ProcessMessageDocument> response, INCDTLS2026 report)
        {
            if (report.Rp_IncomeCollection != null)
            {
                ProcessMessageDocument processMessage;
                bool assertion;

                assertion = (report.Rp_IncomeCollection.Count(t => t.Rp_Income_OrganisationNameDetails_SalaryOrWagesCollectionExists == true) + report.Rp_IncomeCollection.Count(t => t.Rp_Income_OrganisationNameDetails_AustralianAnnuitiesCollectionExists == true)) > 57;
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.INCDTLS.000100",
                        Severity = ProcessMessageSeverity.Error,
                        Description = @"Total number of Salary or wages and Australian annuities payment summary instances is incorrect",
                        LongDescription = @"The total number of Salary or wages payment summary instances plus Australian annuities payment summary instances must not exceed 57",
                        Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income/tns:OrganisationNameDetails/tns:SalaryOrWages",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.INCDTLS.000100" } },
                    };

                    /*  Parameters are commented out or non-existant for manual rules 

                    */
                    response.Add(processMessage);
                }
            }
        }
        #endregion // VR.ATO.INCDTLS.000100

        #region VR.ATO.INCDTLS.000123

        /*  VR.ATO.INCDTLS.000123
        The number of Australian Superannuation lump sum (SLS) payments instances must not exceed 25

        Legacy Rule Format:
        (Count(^INCDTLS327) > 25)

        Technical Business Rule Format:
        (Count(^INCDTLS327) > 25)

        Data Elements:

        ^INCDTLS327 = INCDTLS:Rp:Income:OrganisationNameDetails:SuperannuationLumpSumPayment
        */
        protected void VRATOINCDTLS000123(List<ProcessMessageDocument> response, INCDTLS2026 report)
        {
            if (report.Rp_IncomeCollection != null)
            {
                ProcessMessageDocument processMessage;
                bool assertion;

                assertion = report.Rp_IncomeCollection.Count(t => t.Rp_Income_OrganisationNameDetails_SuperannuationLumpSumPaymentCollectionExists == true) > 25;
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.INCDTLS.000123",
                        Severity = ProcessMessageSeverity.Error,
                        Description = @"The number of Australian Superannuation lump sum (SLS) payments instances must not exceed 25",
                        Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income/tns:OrganisationNameDetails/tns:SuperannuationLumpSumPayment",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.INCDTLS.000123" } },
                    };

                    /*  Parameters are commented out or non-existant for manual rules 

                    */
                    response.Add(processMessage);
                }
            }
        }
        #endregion // VR.ATO.INCDTLS.000123

        #region VR.ATO.INCDTLS.000129

        /*  VR.ATO.INCDTLS.000129
        The number of Attributed personal services (Attributed PSI) payments instances must not exceed 10

        Legacy Rule Format:
        (Count(^INCDTLS340) > 10)

        Technical Business Rule Format:
        (Count(^INCDTLS340) > 10)

        Data Elements:

        ^INCDTLS340 = INCDTLS:Rp:Income:OrganisationNameDetails:AttributedPersonalServicesIncome
        */
        protected void VRATOINCDTLS000129(List<ProcessMessageDocument> response, INCDTLS2026 report)
        {
            if (report.Rp_IncomeCollection != null)
            {
                ProcessMessageDocument processMessage;
                bool assertion;

                assertion = report.Rp_IncomeCollection.Count(t => t.Rp_Income_OrganisationNameDetails_AttributedPersonalServicesIncomeCollectionExists == true) > 10;
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.INCDTLS.000129",
                        Severity = ProcessMessageSeverity.Error,
                        Description = @"The number of Attributed personal services (Attributed PSI) payments instances must not exceed 10",
                        Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income/tns:OrganisationNameDetails/tns:AttributedPersonalServicesIncome",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.INCDTLS.000129" } },
                    };
                    response.Add(processMessage);
                }
            }
        }
        #endregion // VR.ATO.INCDTLS.000123

        #region VR.ATO.INCDTLS.000274

        /*  VR.ATO.INCDTLS.000274
                    The Foreign employment income payment summary lump sum in arrears year within the Foreign employment income payment summary instance must not be duplicated

                Legacy Rule Format:
                    (AnyOccurrence(^INCDTLS160, HasDuplicateValues(^INCDTLS523)))

            Technical Business Rule Format:
                    (AnyOccurrence(^INCDTLS160, HasDuplicateValues(^INCDTLS523)))

        Data Elements:

            ^INCDTLS523 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment:FEIPSLumpSumArrearsPayment:Report.TargetFinancial.Year

            ^INCDTLS160 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment
            */

        protected void VRATOINCDTLS000274(List<ProcessMessageDocument> response, INCDTLS2026 reportIn)
        {
            ProcessMessageDocument processMessage;
            bool assertion = false;

            // Checks to see if an income collection exists
            if (reportIn.Rp_IncomeCollection != null)
            {
                // Goes through each the income tuples within the income collection
                foreach (Income incomeTuple in reportIn.Rp_IncomeCollection)
                {
                    // Checks to see if the assertion is still false and the FEIPS Lump Sum Collection exists
                    if (assertion != true && incomeTuple.Rp_Income_OrganisationNameDetails_ForeignEmployment_FEIPSLumpSumArrearsPaymentCollection != null)
                    {
                        // Checks for any duplicate values with the tuple. If there is, set to true else false
                        assertion = HasDuplicateValues(incomeTuple.Rp_Income_OrganisationNameDetails_ForeignEmployment_FEIPSLumpSumArrearsPaymentCollection.Select(f => f.INCDTLS523).Cast<object>().ToArray());
                    }
                }
            }

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000274",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The Foreign employment income payment summary lump sum in arrears year must not be duplicated",
                    LongDescription = @"The Foreign employment income payment summary lump sum in arrears year within the Foreign employment income payment summary instance must not be duplicated",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income/tns:OrganisationNameDetails/tns:ForeignEmployment/tns:FEIPSLumpSumArrearsPaymentCollection/tns:FEIPSLumpSumArrearsPayment/tns:TargetFinancialY",
                    Parameters = new ProcessMessageParameters() {
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.INCDTLS.000274" } },
                };

                response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.INCDTLS.000274

        #region VR.ATO.INCDTLS.000275

        /*  VR.ATO.INCDTLS.000275
                    The Superannuation income stream lump sum in arrears year within the Australian superannuation income stream payment summary instance must not be duplicated

                Legacy Rule Format:
                    (AnyOccurrence(^INCDTLS108, HasDuplicateValues(^INCDTLS526)))

            Technical Business Rule Format:
                    (AnyOccurrence(^INCDTLS108, HasDuplicateValues(^INCDTLS526)))

        Data Elements:

            ^INCDTLS526 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:SISLumpSumArrearsPayment:Report.TargetFinancial.Year

            ^INCDTLS108 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation
            */

        protected void VRATOINCDTLS000275(List<ProcessMessageDocument> response, INCDTLS2026 reportIn)
        {
            ProcessMessageDocument processMessage;
            bool assertion = false;

            // Checks to see if an income collection exists
            if (reportIn.Rp_IncomeCollection != null)
            {
                // Goes through each the income tuples within the income collection
                foreach (Income incomeTuple in reportIn.Rp_IncomeCollection)
                {
                    // Checks to see if the assertion is still false and the SIS Lump Sum Collection exists
                    if (assertion != true && incomeTuple.Rp_Income_OrganisationNameDetails_AnnuitiesAndSuperannuation_SISLumpSumArrearsPaymentCollection != null)
                    {
                        // Checks for any duplicate values with the tuple. If there is, set to true else false
                        assertion = HasDuplicateValues(incomeTuple.Rp_Income_OrganisationNameDetails_AnnuitiesAndSuperannuation_SISLumpSumArrearsPaymentCollection.Select(f => f.INCDTLS526).Cast<object>().ToArray());
                    }
                }
            }

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000275",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The Superannuation income stream lump sum in arrears year must not be duplicated",
                    LongDescription = @"The Superannuation income stream lump sum in arrears year within the Australian superannuation income stream payment summary instance must not be duplicated",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income/tns:OrganisationNameDetails/tns:AnnuitiesAndSuperannuation/tns:SISLumpSumArrearsPaymentCollection/tns:SISLumpSumArrearsPayment/tns:TargetFinancialY",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.INCDTLS.000275" } },
                };

                response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.INCDTLS.000275

        #region VR.ATO.INCDTLS.000276

        /*  VR.ATO.INCDTLS.000276
                    The Foreign employment income non-payment summary lump sum in arrears year within the Foreign employment income non-payment summary instance must not be duplicated

                Legacy Rule Format:
                    (AnyOccurrence(^INCDTLS185, HasDuplicateValues(^INCDTLS529)))

            Technical Business Rule Format:
                    (AnyOccurrence(^INCDTLS185, HasDuplicateValues(^INCDTLS529)))

        Data Elements:

            ^INCDTLS529 = INCDTLS:Rp:ForeignIncome:FEINPSLumpSumArrearsPayment:Report.TargetFinancial.Year

            ^INCDTLS185 = INCDTLS:Rp:ForeignIncome
            */

        protected void VRATOINCDTLS000276(List<ProcessMessageDocument> response, INCDTLS2026 reportIn)
        {
            ProcessMessageDocument processMessage;
            bool assertion = false;

            if (reportIn.Rp_ForeignIncomeCollection != null)
            {
                // Goes through each the foreign income tuples within the foreign income collection
                foreach (INCDTLS2026.Rp_ForeignIncome foreignIncomeTuple in reportIn.Rp_ForeignIncomeCollection)
                {
                    // Checks to see if the assertion is still false and the FEINPS Lump Sum Collection exists
                    if (assertion != true && foreignIncomeTuple.Rp_ForeignIncome_FEINPSLumpSumArrearsPaymentCollection != null)
                    {
                        // Checks for any duplicate values with the tuple. If there is, set to true else false
                        assertion = HasDuplicateValues(foreignIncomeTuple.Rp_ForeignIncome_FEINPSLumpSumArrearsPaymentCollection.Select(f => f.INCDTLS529).Cast<object>().ToArray());
                    }
                }
            }

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000276",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The Foreign employment income non-payment summary lump sum in arrears year must not be duplicated",
                    LongDescription = @"The Foreign employment income non-payment summary lump sum in arrears year within the Foreign employment income non-payment summary instance must not be duplicated",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ForeignIncomeCollection/tns:ForeignIncome/tns:FEINPSLumpSumArrearsPaymentCollection/tns:FEINPSLumpSumArrearsPayment/tns:TargetFinancialY",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.INCDTLS.000276" } },
                };

                response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.INCDTLS.000276

        #region VR.ATO.INCDTLS.000277

        /*  VR.ATO.INCDTLS.000277
                    The Foreign pensions or annuities lump sum in arrears year within the Foreign pensions or annuities instance must not be duplicated

                Legacy Rule Format:
                    (AnyOccurrence(^INCDTLS207, HasDuplicateValues(^INCDTLS532)))

            Technical Business Rule Format:
                    (AnyOccurrence(^INCDTLS207, HasDuplicateValues(^INCDTLS532)))

        Data Elements:

            ^INCDTLS532 = INCDTLS:Rp:ForeignPensionsOrAnnuities:FPALumpSumArrearsPayment:Report.TargetFinancial.Year

            ^INCDTLS207 = INCDTLS:Rp:ForeignPensionsOrAnnuities
            */

        protected void VRATOINCDTLS000277(List<ProcessMessageDocument> response, INCDTLS2026 reportIn)
        {
            ProcessMessageDocument processMessage;
            bool assertion = false;

            if (reportIn.Rp_ForeignPensionsOrAnnuitiesCollection != null)
            {
                // Goes through each the FPA tuples within the FPA collection
                foreach (INCDTLS2026.Rp_ForeignPensionsOrAnnuities foreignPATuple in reportIn.Rp_ForeignPensionsOrAnnuitiesCollection)
                {
                    // Checks to see if the assertion is still false and the FPA Lump Sum Collection exists
                    if (assertion != true && foreignPATuple.Rp_ForeignPensionsOrAnnuities_FPALumpSumArrearsPaymentCollection != null)
                    {
                        // Checks for any duplicate values with the tuple. If there is, set to true else false
                        assertion = HasDuplicateValues(foreignPATuple.Rp_ForeignPensionsOrAnnuities_FPALumpSumArrearsPaymentCollection.Select(f => f.INCDTLS532).Cast<object>().ToArray());
                    }
                }
            }

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000277",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The Foreign pensions or annuities lump sum in arrears year must not be duplicated",
                    LongDescription = @"The Foreign pensions or annuities lump sum in arrears year within the Foreign pensions or annuities instance must not be duplicated",
                    Location = "/tns:INCDTLS/tns:Rp/tns:ForeignPensionsOrAnnuitiesCollection/tns:ForeignPensionsOrAnnuities/tns:FPALumpSumArrearsPaymentCollection/tns:FPALumpSumArrearsPayment/tns:TargetFinancialY",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.INCDTLS.000277" } },
                };

                response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.INCDTLS.000277

        #region VR.ATO.INCDTLS.000308

        /*  
        VR.ATO.INCDTLS.000308
        The number of Employment termination payment (ETP) summary instances must not exceed 25 for Australian ETP payments

        Legacy Rule Format:
        (CountOccurrence(^INCDTLS135, (InSet(^INCDTLS136, '"au","cc","cx","nf","hm"') OR ^INCDTLS136 = NULL)) > 25)

        Technical Business Rule Format:
        (CountOccurrence(^INCDTLS135, (InSet(^INCDTLS136, '"au","cc","cx","nf","hm"') OR ^INCDTLS136 = NULL)) > 25)

        Data Elements:

        ^INCDTLS136 = INCDTLS:Rp:Income:OrganisationNameDetails:EmploymentTerminationPayment:AddressDetails.Country.Code

        ^INCDTLS135 = INCDTLS:Rp:Income:OrganisationNameDetails:EmploymentTerminationPayment
        */
        protected void VRATOINCDTLS000308(List<ProcessMessageDocument> response, INCDTLS2026 report)
        {
            if (report.Rp_IncomeCollection != null)
            {
                bool assertion = report.Rp_IncomeCollection.Count(t => t.Rp_Income_OrganisationNameDetails_EmploymentTerminationPaymentCollectionExists && (t.INCDTLS136 == null || _australianAndHmCountryCodesDictionary.ContainsKey(t.INCDTLS136))) > 25;
                if (assertion)
                {
                    ProcessMessageDocument processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.INCDTLS.000308",
                        Severity = ProcessMessageSeverity.Error,
                        Description = @"The number of Employment termination payment (ETP) summary instances must not exceed 25 for Australian ETP payments",
                        Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income/tns:OrganisationNameDetails/tns:EmploymentTerminationPayment/tns:AddressDetailsCountryC",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.INCDTLS.000308" } },
                    };

                    response.Add(processMessage);
                }
            }
        }

        #endregion // VR.ATO.INCDTLS.000308

        #region VR.ATO.INCDTLS.000309

        /*
        VR.ATO.INCDTLS.000309
        The number of Employment termination payment (ETP) summary instances must not exceed 25 for Foreign ETP payments

        Legacy Rule Format:
        (CountOccurrence(^INCDTLS135, (NotInSet(^INCDTLS136, '"au","cc","cx","nf","hm"') AND ^INCDTLS136 <> NULL)) > 25)

        Technical Business Rule Format:
        (CountOccurrence(^INCDTLS135, (NotInSet(^INCDTLS136, '"au","cc","cx","nf","hm"') AND ^INCDTLS136 <> NULL)) > 25)

        Data Elements:

        ^INCDTLS136 = INCDTLS:Rp:Income:OrganisationNameDetails:EmploymentTerminationPayment:AddressDetails.Country.Code

        ^INCDTLS135 = INCDTLS:Rp:Income:OrganisationNameDetails:EmploymentTerminationPayment
        */
        protected void VRATOINCDTLS000309(List<ProcessMessageDocument> response, INCDTLS2026 report)
        {
            if (report.Rp_IncomeCollection != null)
            {
                bool assertion = report.Rp_IncomeCollection.Count(t => (t.INCDTLS136 != null && !_australianAndHmCountryCodesDictionary.ContainsKey(t.INCDTLS136))) > 25;
                if (assertion)
                {
                    ProcessMessageDocument processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.INCDTLS.000309",
                        Severity = ProcessMessageSeverity.Error,
                        Description = @"The number of Employment termination payment (ETP) summary instances must not exceed 25 for Foreign ETP payments",
                        Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income/tns:OrganisationNameDetails/tns:EmploymentTerminationPayment/tns:AddressDetailsCountryC",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.INCDTLS.000309" } },
                    };

                    response.Add(processMessage);
                }
            }
        }

        #endregion // VR.ATO.INCDTLS.000309

        #region VR.ATO.INCDTLS.000316

        /*  VR.ATO.INCDTLS.000316
                    The total number of sets of lump sum in arrear details across all payment types must not exceed 15 - this includes the following payment summaries/types; Salary or wages payment summary, Foreign employment income payment summary, Australian superannuation income stream payment summary, Foreign employment income non-payment summary, and Foreign pensions or annuities

                Legacy Rule Format:
                    ((Count(^INCDTLS150) + CountOccurrence(^INCDTLS160, Count(^INCDTLS522) > 0) + CountOccurrence(^INCDTLS108, Count(^INCDTLS525) > 0) + CountOccurrence(^INCDTLS185, Count(^INCDTLS528) > 0) + CountOccurrence(^INCDTLS207, Count(^INCDTLS531) > 0)) > 15)

            Technical Business Rule Format:
                    ((Count(^INCDTLS150) + CountOccurrence(^INCDTLS160, Count(^INCDTLS522) > 0) + CountOccurrence(^INCDTLS108, Count(^INCDTLS525) > 0) + CountOccurrence(^INCDTLS185, Count(^INCDTLS528) > 0) + CountOccurrence(^INCDTLS207, Count(^INCDTLS531) > 0)) > 15)

        Data Elements:

            ^INCDTLS150 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:LumpSumArrears

            ^INCDTLS108 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation

            ^INCDTLS160 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment

            ^INCDTLS185 = INCDTLS:Rp:ForeignIncome

            ^INCDTLS207 = INCDTLS:Rp:ForeignPensionsOrAnnuities

            ^INCDTLS522 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment:FEIPSLumpSumArrearsPayment

            ^INCDTLS525 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:SISLumpSumArrearsPayment

            ^INCDTLS528 = INCDTLS:Rp:ForeignIncome:FEINPSLumpSumArrearsPayment

            ^INCDTLS531 = INCDTLS:Rp:ForeignPensionsOrAnnuities:FPALumpSumArrearsPayment
            */

        protected void VRATOINCDTLS000316(List<ProcessMessageDocument> response, INCDTLS2026 reportIn)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            int lumpSumTotalCounter = 0;
            int lumpSumSoWCounter = 0;
            int lumpSumFEICounter = 0;
            int lumpSumSISCounter = 0;
            int lumpSumFEINPSCounter = 0;
            int lumpSumFPACounter = 0;

            // Checks to see with the Income Collection exists
            if (reportIn.Rp_IncomeCollection != null)
            {
                // Checks to see a Salary or Wage Lump Sum Collection exists
                if (reportIn.Rp_IncomeCollection.Any(income => income.Rp_Income_OrganisationNameDetails_SalaryOrWages_LumpSumArrearsCollection != null))
                {
                    foreach (Income incomeTuple in reportIn.Rp_IncomeCollection)
                    {
                        if (incomeTuple.Rp_Income_OrganisationNameDetails_SalaryOrWages_LumpSumArrearsCollection != null)
                            lumpSumSoWCounter += incomeTuple.Rp_Income_OrganisationNameDetails_SalaryOrWages_LumpSumArrearsCollection.Count(lumpSumTuple => lumpSumTuple.Rp_Income_OrganisationNameDetails_SalaryOrWages_LumpSumArrears_LumpSumArrearsPaymentCollection != null);
                    }
                }

                // Count all the FEIPS Lump Sum In Arrears
                lumpSumFEICounter = reportIn.Rp_IncomeCollection.Count(income => income.Rp_Income_OrganisationNameDetails_ForeignEmployment_FEIPSLumpSumArrearsPaymentCollection != null);

                // Count all the SIS Lump Sum In Arrears
                lumpSumSISCounter = reportIn.Rp_IncomeCollection.Count(income => income.Rp_Income_OrganisationNameDetails_AnnuitiesAndSuperannuation_SISLumpSumArrearsPaymentCollection != null);
            }

            // Checks to see whether FEINPS Collection exists
            if (reportIn.Rp_ForeignIncomeCollection != null)
            {
                // Count all the FEINPS Lump Sum In Arrears
                lumpSumFEINPSCounter = reportIn.Rp_ForeignIncomeCollection.Count(foreignIncome => foreignIncome.Rp_ForeignIncome_FEINPSLumpSumArrearsPaymentCollection != null);
            }

            // Checks to see whether Foreign PA Collection exists
            if (reportIn.Rp_ForeignPensionsOrAnnuitiesCollection != null)
            {
                // Count all the FPA Lump Sum In Arrears
                lumpSumFPACounter = reportIn.Rp_ForeignPensionsOrAnnuitiesCollection.Count(foreignPA => foreignPA.Rp_ForeignPensionsOrAnnuities_FPALumpSumArrearsPaymentCollection != null);
            }

            // Adds each of the Lump Sum Counters and set to a total amount
            lumpSumTotalCounter = lumpSumSoWCounter + lumpSumFEICounter + lumpSumSISCounter + lumpSumFEINPSCounter + lumpSumFPACounter;

            // Condition: ((Count(^ INCDTLS150) + CountOccurrence(^ INCDTLS160, Count(^ INCDTLS522) > 0) + CountOccurrence(^ INCDTLS108, Count(^ INCDTLS525) > 0) + CountOccurrence(^ INCDTLS185, Count(^ INCDTLS528) > 0) + CountOccurrence(^ INCDTLS207, Count(^ INCDTLS531) > 0)) > 15)
            assertion = lumpSumTotalCounter > 15;

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000316",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The total number of sets of lump sum in arrear details across all payment types must not exceed 15",
                    LongDescription = @"The total number of sets of lump sum in arrear details across all payment types must not exceed 15 - this includes the following payment summaries/types; Salary or wages payment summary, Foreign employment income payment summary, Australian superannuation income stream payment summary, Foreign employment income non-payment summary, and Foreign pensions or annuities",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income/tns:OrganisationNameDetails/tns:SalaryOrWages/tns:LumpSumArrearsCollection/tns:LumpSumArrears",
                    Parameters = new ProcessMessageParameters() {
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.INCDTLS.000316" },
                        new ProcessMessageParameter() { Name = "Total of Lump Sum In Arrears", Value = lumpSumTotalCounter.ToString() },
                        new ProcessMessageParameter() { Name = "Count(^ INCDTLS150)", Value = lumpSumSoWCounter.ToString() },
                        new ProcessMessageParameter() { Name = "CountOccurrence(^ INCDTLS160, Count(^ INCDTLS522) > 0))", Value = lumpSumFEICounter.ToString() },
                        new ProcessMessageParameter() { Name = "CountOccurrence(^ INCDTLS108, Count(^ INCDTLS525) > 0))", Value = lumpSumSISCounter.ToString() },
                        new ProcessMessageParameter() { Name = "CountOccurrence(^ INCDTLS185, Count(^ INCDTLS528) > 0))", Value = lumpSumFEINPSCounter.ToString() },
                        new ProcessMessageParameter() { Name = "CountOccurrence(^ INCDTLS207, Count(^ INCDTLS531) > 0))", Value = lumpSumFPACounter.ToString() },

                    },
                };
                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.INCDTLS.000316

        #region VR.ATO.INCDTLS.000317

        /*  VR.ATO.INCDTLS.000317
        The Salary or wages lump sum in arrears year within the Salary or wages lump sum in arrears instance must not be duplicated

        Legacy Rule Format:
        (AnyOccurrence(^INCDTLS150, HasDuplicateValues(^INCDTLS153)))

        Technical Business Rule Format:
        (AnyOccurrence(^INCDTLS150, HasDuplicateValues(^INCDTLS153)))

        Data Elements:

        ^INCDTLS153 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:LumpSumArrears:LumpSumArrearsPayment:Report.TargetFinancial.Year

        ^INCDTLS150 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:LumpSumArrears
        */

        protected void VRATOINCDTLS000317(List<ProcessMessageDocument> response, INCDTLS2026 report)
        {
            ProcessMessageDocument processMessage;
            bool assertion = false;

            // Checks to see if an income collection exists
            if (report.Rp_IncomeCollection != null)
            {
                // Goes through each the income tuples within the income collection
                foreach (Income incomeTuple in report.Rp_IncomeCollection)
                {
                    // Checks to see if the assertion is still false and the SoW Lump Sum Collection exists
                    if (assertion != true && incomeTuple.Rp_Income_OrganisationNameDetails_SalaryOrWages_LumpSumArrearsCollection != null)
                    {
                        // Goes through each of the LumpSumInArrears tuple within the Income Tuple
                        foreach (Income.Rp_Income_OrganisationNameDetails_SalaryOrWages_LumpSumArrears lumpSumTuple in incomeTuple.Rp_Income_OrganisationNameDetails_SalaryOrWages_LumpSumArrearsCollection)
                        {
                            // Checks to see if the assertion is still false and the SoW Lump Sum Payment Collection exists
                            if (assertion != true && lumpSumTuple.Rp_Income_OrganisationNameDetails_SalaryOrWages_LumpSumArrears_LumpSumArrearsPaymentCollection != null)
                            {
                                // Checks for any duplicate values with the tuple. If there is, set to true else false
                                assertion = HasDuplicateValues(lumpSumTuple.Rp_Income_OrganisationNameDetails_SalaryOrWages_LumpSumArrears_LumpSumArrearsPaymentCollection.Select(f => f.INCDTLS153).Cast<object>().ToArray());
                            }
                        }
                    }

                    // Checks to see if the assertion is still false and the FEIPS Lump Sum Collection exists
                    if (assertion != true && incomeTuple.Rp_Income_OrganisationNameDetails_ForeignEmployment_FEIPSLumpSumArrearsPaymentCollection != null)
                    {
                        // Checks for any duplicate values with the tuple. If there is, set to true else false
                        assertion = HasDuplicateValues(incomeTuple.Rp_Income_OrganisationNameDetails_ForeignEmployment_FEIPSLumpSumArrearsPaymentCollection.Select(f => f.INCDTLS523).Cast<object>().ToArray());
                    }
                }
            }

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000317",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The Salary or wages lump sum in arrears year must not be duplicated",
                    LongDescription = @"The Salary or wages lump sum in arrears year within the Salary or wages lump sum in arrears instance must not be duplicated",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter()
                        { Name = "RuleIdentifier", Value = "VR.ATO.INCDTLS.000317" }
                    },
                };

                response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.INCDTLS.000317

        #region VR.ATO.INCDTLS.000318

        /*  VR.ATO.INCDTLS.000318
        Where a positive amount is present for any of the following Superannuation income stream lump sum in arrears amounts for an Australian superannuation income stream payment summary, at least one Superannuation income stream lump sum in arrears payment instance must be provided; Superannuation income stream lump sum in arrears taxable component taxed element, Superannuation income stream lump sum in arrears taxable component untaxed element, or Superannuation income stream lump sum in arrears tax free component where the answer to "Are you under 60 years of age and a death benefits dependant, where the deceased died at 60 years or over?" is 'yes' (true)

        Legacy Rule Format:
        (^INCDTLS108 <> NULL AND (^INCDTLS117 > 0 OR ^INCDTLS118 > 0 OR (^INCDTLS109 = "true" AND ^INCDTLS119 > 0)) AND Count(^INCDTLS525) = 0) AND ((Count(^INCDTLS150) + CountOccurrence(^INCDTLS160, Count(^INCDTLS522) > 0) + CountOccurrence(^INCDTLS108, Count(^INCDTLS525) > 0) + CountOccurrence(^INCDTLS185, Count(^INCDTLS528) > 0) + CountOccurrence(^INCDTLS207, Count(^INCDTLS531) > 0)) < 15)

        Technical Business Rule Format:
        (^INCDTLS108 <> NULL AND (^INCDTLS117 > 0 OR ^INCDTLS118 > 0 OR (^INCDTLS109 = 'true' AND ^INCDTLS119 > 0)) AND Count(^INCDTLS525) = 0) AND ((Count(^INCDTLS150) + CountOccurrence(^INCDTLS160, Count(^INCDTLS522) > 0) + CountOccurrence(^INCDTLS108, Count(^INCDTLS525) > 0) + CountOccurrence(^INCDTLS185, Count(^INCDTLS528) > 0) + CountOccurrence(^INCDTLS207, Count(^INCDTLS531) > 0)) < 15)

        Data Elements:

        ^INCDTLS525 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:SISLumpSumArrearsPayment

        ^INCDTLS108 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation

        ^INCDTLS109 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:Income.Superannuation.AnnuitiesAndSuperannuationReversionaryIncomeStream.Indicator

        ^INCDTLS117 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:Income.Superannuation.AnnuitiesAndSuperannuationLumpSumArrearsTaxableComponentTaxedElement.Amount

        ^INCDTLS118 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:Income.Superannuation.AnnuitiesAndSuperannuationLumpSumArrearsTaxableComponentUntaxedElement.Amount

        ^INCDTLS119 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:Income.Superannuation.LumpSumArrearsTaxFree.Amount

        ^INCDTLS150 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:LumpSumArrears

        ^INCDTLS160 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment

        ^INCDTLS185 = INCDTLS:Rp:ForeignIncome

        ^INCDTLS207 = INCDTLS:Rp:ForeignPensionsOrAnnuities

        ^INCDTLS522 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment:FEIPSLumpSumArrearsPayment

        ^INCDTLS528 = INCDTLS:Rp:ForeignIncome:FEINPSLumpSumArrearsPayment

        ^INCDTLS531 = INCDTLS:Rp:ForeignPensionsOrAnnuities:FPALumpSumArrearsPayment
        */
        protected void VRATOINCDTLS000318(List<ProcessMessageDocument> response, INCDTLS2026 reportIn)
        {
            // Condition 1:
            // (^INCDTLS108 <> NULL AND (^INCDTLS117 > 0 OR ^INCDTLS118 > 0 OR (^INCDTLS109 = "true" AND ^INCDTLS119 > 0)) AND Count(^INCDTLS525) = 0)
            bool condition1 =
            (
                reportIn.Rp_IncomeCollection != null
                &&
                reportIn.Rp_IncomeCollection.Any
                (
                    INCDTLS002Repeat =>
                    (
                        // INCDTLS108 <> NULL
                        INCDTLS002Repeat.Rp_Income_OrganisationNameDetails_AnnuitiesAndSuperannuationCollectionExists == true
                        &&
                        (
                            (
                                // INCDTLS117 > 0
                                INCDTLS002Repeat.INCDTLS117 != null && INCDTLS002Repeat.INCDTLS117 > 0
                            )
                            ||
                            (
                                // INCDTLS118 > 0
                                INCDTLS002Repeat.INCDTLS118 != null && INCDTLS002Repeat.INCDTLS118 > 0
                            )
                            ||
                            (
                                (
                                    // INCDTLS109 = "true"
                                    INCDTLS002Repeat.INCDTLS109 != null && INCDTLS002Repeat.INCDTLS109 == true
                                )
                                &&
                                (
                                    // INCDTLS119 > 0
                                    INCDTLS002Repeat.INCDTLS119 != null && INCDTLS002Repeat.INCDTLS119 > 0
                                )
                            )
                        )
                        &&
                        (
                            // Count(^INCDTLS525) = 0)
                            INCDTLS002Repeat.Rp_Income_OrganisationNameDetails_AnnuitiesAndSuperannuation_SISLumpSumArrearsPaymentCollectionCount == 0
                        )
                    )
                )
            );

            if (condition1)
            {
                int INCDTLS150Count = GetCountOfAllINCDTLS150Tuples(reportIn);
                int INCDTLS160Count = GetCountOfAllINCDTLS160TuplesContainingINCDTLS522Tuples(reportIn);
                int INCDTLS108Count = GetCountOfAllINCDTLS108TuplesContainingINCDTLS525Tuples(reportIn);
                int INCDTLS185Count = GetCountOfAllINCDTLS185TuplesContainingINCDTLS528Tuples(reportIn);
                int INCDTLS207Count = GetCountOfAllINCDTLS207TuplesContainingINCDTLS531Tuples(reportIn);

                // Condition 2: 
                // (Count(^INCDTLS150) + CountOccurrence(^INCDTLS160, Count(^INCDTLS522) > 0) + CountOccurrence(^INCDTLS108, Count(^INCDTLS525) > 0) + CountOccurrence(^INCDTLS185, Count(^INCDTLS528) > 0) + CountOccurrence(^INCDTLS207, Count(^INCDTLS531) > 0)) < 15
                bool assertion = (INCDTLS150Count + INCDTLS160Count + INCDTLS108Count + INCDTLS185Count + INCDTLS207Count) < 15;

                if (assertion)
                {
                    ProcessMessageDocument processMessage = new ProcessMessageDocument();

                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.INCDTLS.000318",
                        Severity = ProcessMessageSeverity.Error,
                        Description = @"Superannuation income stream lump sum in arrears payment details must be provided",
                        LongDescription = @"Where a positive amount is present for any of the following Superannuation income stream lump sum in arrears amounts for an Australian superannuation income stream payment summary, at least one Superannuation income stream lump sum in arrears payment instance must be provided; Superannuation income stream lump sum in arrears taxable component taxed element, Superannuation income stream lump sum in arrears taxable component untaxed element, or Superannuation income stream lump sum in arrears tax free component where the answer to ""Are you under 60 years of age and a death benefits dependant, where the deceased died at 60 years or over?"" is 'yes' (true)",
                        Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income/tns:OrganisationNameDetails/tns:AnnuitiesAndSuperannuation/tns:SISLumpSumArrearsPaymentCollection/tns:SISLumpSumArrearsPayment",
                        Parameters = new ProcessMessageParameters()
                        {
                            new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.INCDTLS.000318" },
                            new ProcessMessageParameter() { Name = "Positive amount provided for any Superannuation income stream LSA amount", Value = condition1.ToString() },
                            new ProcessMessageParameter() { Name = "Count(^INCDTLS150)", Value = INCDTLS150Count.ToString() },
                            new ProcessMessageParameter() { Name = "CountOccurrence(^INCDTLS160, Count(^INCDTLS522) > 0)", Value = INCDTLS160Count.ToString() },
                            new ProcessMessageParameter() { Name = "CountOccurrence(^INCDTLS108, Count(^INCDTLS525) > 0)", Value = INCDTLS108Count.ToString() },
                            new ProcessMessageParameter() { Name = "CountOccurrence(^INCDTLS185, Count(^INCDTLS528) > 0)", Value = INCDTLS185Count.ToString() },
                            new ProcessMessageParameter() { Name = "CountOccurrence(^INCDTLS207, Count(^INCDTLS531) > 0)", Value = INCDTLS207Count.ToString() },
                        },
                    };

                    response.Add(processMessage);
                }
            }
        }

        #endregion // VR.ATO.INCDTLS.000318

        #region VR.ATO.INCDTLS.000321

        /*  VR.ATO.INCDTLS.000321
        Where a positive Salary or wages lump sum E amount is present for a Salary or wages payment summary, a Salary or wages lump sum in arrears instance with a Lump sum in arrears payment type code of "Salary or wage lump sum E" must be provided

        Legacy Rule Format:
        (^INCDTLS309 <> NULL AND ^INCDTLS326 > 0 AND CountOccurrence(^INCDTLS150, ^INCDTLS151 = "Salary or wage lump sum E") = 0) AND ((Count(^INCDTLS150) + CountOccurrence(^INCDTLS160, Count(^INCDTLS522) > 0) + CountOccurrence(^INCDTLS108, Count(^INCDTLS525) > 0) + CountOccurrence(^INCDTLS185, Count(^INCDTLS528) > 0) + CountOccurrence(^INCDTLS207, Count(^INCDTLS531) > 0)) < 15)

        Technical Business Rule Format:
        (^INCDTLS309 <> NULL AND ^INCDTLS326 > 0 AND CountOccurrence(^INCDTLS150, ^INCDTLS151 = 'Salary or wage lump sum E') = 0) AND ((Count(^INCDTLS150) + CountOccurrence(^INCDTLS160, Count(^INCDTLS522) > 0) + CountOccurrence(^INCDTLS108, Count(^INCDTLS525) > 0) + CountOccurrence(^INCDTLS185, Count(^INCDTLS528) > 0) + CountOccurrence(^INCDTLS207, Count(^INCDTLS531) > 0)) < 15)

        Data Elements:

        ^INCDTLS151 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:LumpSumArrears:Income.LumpSumArrearsPaymentType.Code

        ^INCDTLS108 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation

        ^INCDTLS150 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:LumpSumArrears

        ^INCDTLS160 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment

        ^INCDTLS185 = INCDTLS:Rp:ForeignIncome

        ^INCDTLS207 = INCDTLS:Rp:ForeignPensionsOrAnnuities

        ^INCDTLS309 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages

        ^INCDTLS326 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:Income.BackPaymentLumpSumE.Amount

        ^INCDTLS522 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment:FEIPSLumpSumArrearsPayment

        ^INCDTLS525 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:SISLumpSumArrearsPayment

        ^INCDTLS528 = INCDTLS:Rp:ForeignIncome:FEINPSLumpSumArrearsPayment

        ^INCDTLS531 = INCDTLS:Rp:ForeignPensionsOrAnnuities:FPALumpSumArrearsPayment
        */

        protected void VRATOINCDTLS000321(List<ProcessMessageDocument> response, INCDTLS2026 report)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            bool condition1_BothConditionsAreTrue = false;
            bool condition1a_INCDTLS326IsGreaterThan0 = false;
            bool condition1b_INCDTLS151NotLSE = false;
            bool condition2_LumpSumTotalLessThan15 = false;
            int lumpSumTotal = 0;

            // Condition 1: (^INCDTLS309 <> NULL AND ^INCDTLS326 > 0 AND CountOccurrence(^INCDTLS150, ^INCDTLS151 = "Salary or wage lump sum E") = 0)
            // Checks to see whether the Income Collection is provided
            if (report.Rp_IncomeCollection != null)
            {
                // Runs through each of the income tuple
                foreach (Income incomeTuple in report.Rp_IncomeCollection)
                {
                    // Only runs if both conditions aren't true, once they are both true in a single instance then it will skip over checking the other instances
                    if (!(condition1_BothConditionsAreTrue))
                    {
                        // Reset to condition 1s back to false as both need to be true in a single instance
                        condition1a_INCDTLS326IsGreaterThan0 = false;
                        condition1b_INCDTLS151NotLSE = false;

                        // Checks to see whether INCDTLS309 exists
                        if (incomeTuple.Rp_Income_OrganisationNameDetails_SalaryOrWagesCollectionExists)
                        {
                            condition1a_INCDTLS326IsGreaterThan0 = incomeTuple.INCDTLS326.GetValueOrDefault() > 0;

                            // Checks to see if a Lump Sum In Arrears exists within the Income Tuple instance, if it doesn't - condition1b will be set to turn since there is not instances of INCDTLS151 equals LSE
                            if (incomeTuple.Rp_Income_OrganisationNameDetails_SalaryOrWages_LumpSumArrearsCollection != null)
                            {
                                condition1b_INCDTLS151NotLSE = incomeTuple.Rp_Income_OrganisationNameDetails_SalaryOrWages_LumpSumArrearsCollection.Count(lumpSum => lumpSum.INCDTLS151 != null && lumpSum.INCDTLS151 == "Salary or wage lump sum E") == 0;
                            }
                            else
                            {
                                condition1b_INCDTLS151NotLSE = true;
                            }
                        }

                        condition1_BothConditionsAreTrue = condition1a_INCDTLS326IsGreaterThan0 && condition1b_INCDTLS151NotLSE;
                    }
                }
            }

            // Condition 2: ((Count(^INCDTLS150) + CountOccurrence(^INCDTLS160, Count(^INCDTLS522) > 0) + CountOccurrence(^INCDTLS108, Count(^INCDTLS525) > 0) + CountOccurrence(^INCDTLS185, Count(^INCDTLS528) > 0) + CountOccurrence(^INCDTLS207, Count(^INCDTLS531) > 0)) < 15)
            lumpSumTotal = GetTotalNumberOfLumpSumInArrears(report);
            condition2_LumpSumTotalLessThan15 = lumpSumTotal < 15;

            // (^INCDTLS309 <> NULL AND ^INCDTLS326 > 0 AND CountOccurrence(^INCDTLS150, ^INCDTLS151 = "Salary or wage lump sum E") = 0) AND ((Count(^INCDTLS150) + CountOccurrence(^INCDTLS160, Count(^INCDTLS522) > 0) + CountOccurrence(^INCDTLS108, Count(^INCDTLS525) > 0) + CountOccurrence(^INCDTLS185, Count(^INCDTLS528) > 0) + CountOccurrence(^INCDTLS207, Count(^INCDTLS531) > 0)) < 15)
            assertion = condition1_BothConditionsAreTrue && condition2_LumpSumTotalLessThan15;

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000321",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Salary or wages lump sum in arrears details must be provided",
                    LongDescription = "Where a positive Salary or wages lump sum E amount is present for a Salary or wages payment summary, a Salary or wages lump sum in arrears instance with a Lump sum in arrears payment type code of \"Salary or wage lump sum E\" must be provided",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.INCDTLS.000321" },
                        new ProcessMessageParameter() { Name = "Condition 1: INCDTLS326 is greater than 0 and no instances of INCDTLS151 equals 'Salary or wage lump sum E'", Value = condition1_BothConditionsAreTrue.ToString() },
                        new ProcessMessageParameter() { Name = "Condition 2: Count of all instances of Lump Sum In Arrears < 15 ", Value = condition2_LumpSumTotalLessThan15.ToString() },
                        new ProcessMessageParameter() { Name = "Total Number of Lump Sum In Arrears ", Value = lumpSumTotal.ToString() },
                    },
                };

                response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.INCDTLS.000321

        #region VR.ATO.INCDTLS.000323

        /*  VR.ATO.INCDTLS.000323
        Where a positive Exempt foreign employment income lump sum in arrears amount is present for a Salary or wages payment summary, a Salary or wages lump sum in arrears instance with a Lump sum in arrears payment type code of "Exempt FEI PS lump sum in arrears" must be provided

        Legacy Rule Format:
        (^INCDTLS309 <> NULL AND ^INCDTLS338 > 0 AND CountOccurrence(^INCDTLS150, ^INCDTLS151 = "Exempt FEI PS lump sum in arrears") = 0) AND ((Count(^INCDTLS150) + CountOccurrence(^INCDTLS160, Count(^INCDTLS522) > 0) + CountOccurrence(^INCDTLS108, Count(^INCDTLS525) > 0) + CountOccurrence(^INCDTLS185, Count(^INCDTLS528) > 0) + CountOccurrence(^INCDTLS207, Count(^INCDTLS531) > 0)) < 15)

        Technical Business Rule Format:
        (^INCDTLS309 <> NULL AND ^INCDTLS338 > 0 AND CountOccurrence(^INCDTLS150, ^INCDTLS151 = 'Exempt FEI PS lump sum in arrears') = 0) AND ((Count(^INCDTLS150) + CountOccurrence(^INCDTLS160, Count(^INCDTLS522) > 0) + CountOccurrence(^INCDTLS108, Count(^INCDTLS525) > 0) + CountOccurrence(^INCDTLS185, Count(^INCDTLS528) > 0) + CountOccurrence(^INCDTLS207, Count(^INCDTLS531) > 0)) < 15)

        Data Elements:

        ^INCDTLS151 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:LumpSumArrears:Income.LumpSumArrearsPaymentType.Code

        ^INCDTLS108 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation

        ^INCDTLS150 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:LumpSumArrears

        ^INCDTLS160 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment

        ^INCDTLS185 = INCDTLS:Rp:ForeignIncome

        ^INCDTLS207 = INCDTLS:Rp:ForeignPensionsOrAnnuities

        ^INCDTLS309 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages

        ^INCDTLS338 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:ExemptForeignEmployment:Income.LumpSumArrearsPayment.Amount

        ^INCDTLS522 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment:FEIPSLumpSumArrearsPayment

        ^INCDTLS525 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:SISLumpSumArrearsPayment

        ^INCDTLS528 = INCDTLS:Rp:ForeignIncome:FEINPSLumpSumArrearsPayment

        ^INCDTLS531 = INCDTLS:Rp:ForeignPensionsOrAnnuities:FPALumpSumArrearsPayment
        */

        protected void VRATOINCDTLS000323(List<ProcessMessageDocument> response, INCDTLS2026 report)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            bool condition1_BothConditionsAreTrue = false;
            bool condition1a_INCDTLS338IsGreaterThan0 = false;
            bool condition1b_INCDTLS151NotEFEIPS = false;
            bool condition2_LumpSumTotalLessThan15 = false;
            int lumpSumTotal = 0;

            // Condition 1: (^INCDTLS309 <> NULL AND ^INCDTLS338 > 0 AND CountOccurrence(^INCDTLS150, ^INCDTLS151 = "Exempt FEI PS lump sum in arrears") = 0)
            // Checks to see whether the Income Collection is provided
            if (report.Rp_IncomeCollection != null)
            {
                // Runs through each of the income tuple
                foreach (Income incomeTuple in report.Rp_IncomeCollection)
                {
                    // Only runs if both conditions aren't true, once they are both true in a single instance then it will skip over checking the other instances
                    if (!(condition1_BothConditionsAreTrue))
                    {
                        // Reset to condition 1s back to false as both need to be true in a single instance
                        condition1a_INCDTLS338IsGreaterThan0 = false;
                        condition1b_INCDTLS151NotEFEIPS = false;

                        // Checks to see whether INCDTLS309 exists
                        if (incomeTuple.Rp_Income_OrganisationNameDetails_SalaryOrWagesCollectionExists)
                        {
                            condition1a_INCDTLS338IsGreaterThan0 = incomeTuple.INCDTLS338.GetValueOrDefault() > 0;

                            // Checks to see if a Lump Sum In Arrears exists within the Income Tuple instance, if it doesn't - condition1b will be set to turn since there is not instances of INCDTLS151 equals Exempt FEI PS lump sum in arrears
                            if (incomeTuple.Rp_Income_OrganisationNameDetails_SalaryOrWages_LumpSumArrearsCollection != null)
                            {
                                condition1b_INCDTLS151NotEFEIPS = incomeTuple.Rp_Income_OrganisationNameDetails_SalaryOrWages_LumpSumArrearsCollection.Count(lumpSum => lumpSum.INCDTLS151 != null && lumpSum.INCDTLS151 == "Exempt FEI PS lump sum in arrears") == 0;
                            }
                            else
                            {
                                condition1b_INCDTLS151NotEFEIPS = true;
                            }
                        }

                        condition1_BothConditionsAreTrue = condition1a_INCDTLS338IsGreaterThan0 && condition1b_INCDTLS151NotEFEIPS;
                    }
                }
            }

            // Condition 2: ((Count(^INCDTLS150) + CountOccurrence(^INCDTLS160, Count(^INCDTLS522) > 0) + CountOccurrence(^INCDTLS108, Count(^INCDTLS525) > 0) + CountOccurrence(^INCDTLS185, Count(^INCDTLS528) > 0) + CountOccurrence(^INCDTLS207, Count(^INCDTLS531) > 0)) < 15)
            lumpSumTotal = GetTotalNumberOfLumpSumInArrears(report);
            condition2_LumpSumTotalLessThan15 = lumpSumTotal < 15;

            // (^INCDTLS309 <> NULL AND ^INCDTLS338 > 0 AND CountOccurrence(^INCDTLS150, ^INCDTLS151 = "Exempt FEI PS lump sum in arrears") = 0) AND ((Count(^INCDTLS150) + CountOccurrence(^INCDTLS160, Count(^INCDTLS522) > 0) + CountOccurrence(^INCDTLS108, Count(^INCDTLS525) > 0) + CountOccurrence(^INCDTLS185, Count(^INCDTLS528) > 0) + CountOccurrence(^INCDTLS207, Count(^INCDTLS531) > 0)) < 15)
            assertion = condition1_BothConditionsAreTrue && condition2_LumpSumTotalLessThan15;

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000323",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Salary or wages lump sum in arrears details must be provided",
                    LongDescription = "Where a positive Exempt foreign employment income lump sum in arrears amount is present for a Salary or wages payment summary, a Salary or wages lump sum in arrears instance with a Lump sum in arrears payment type code of \"Exempt FEI PS lump sum in arrears\" must be provided",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.INCDTLS.000323" },
                        new ProcessMessageParameter() { Name = "Condition 1: INCDTLS338 is greater than 0 and no instances of INCDTLS151 equals 'Exempt FEI PS lump sum in arrears'", Value = condition1_BothConditionsAreTrue.ToString() },
                        new ProcessMessageParameter() { Name = "Condition 2: Count of all instances of Lump Sum In Arrears < 15 ", Value = condition2_LumpSumTotalLessThan15.ToString() },
                        new ProcessMessageParameter() { Name = "Total Number of Lump Sum In Arrears ", Value = lumpSumTotal.ToString() },
                    },
                };

                response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.INCDTLS.000323

        #region VR.ATO.INCDTLS.000333

        /*  VR.ATO.INCDTLS.000333
            The number of Foreign employment payment summary instances must not exceed 3

            Legacy Rule Format:
            (Count(^INCDTLS160) > 3)

            Technical Business Rule Format:
            (Count(^INCDTLS160) > 3)

            Data Elements:

            ^INCDTLS160 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment
            */
        protected void VRATOINCDTLS000333(List<ProcessMessageDocument> response, INCDTLS2026 report)
        {
            if (report.Rp_IncomeCollection != null)
            {
                ProcessMessageDocument processMessage;

                bool assertion = report.Rp_IncomeCollection.Count(t => t.Rp_Income_OrganisationNameDetails_ForeignEmploymentCollectionExists == true) > 3;
                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.INCDTLS.000333",
                        Severity = ProcessMessageSeverity.Error,
                        Description = @"The number of Foreign employment payment summary instances must not exceed 3",
                        Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income/tns:OrganisationNameDetails/tns:ForeignEmployment",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.INCDTLS.000333" } },
                    };

                    response.Add(processMessage);
                }
            }
        }
        #endregion // VR.ATO.INCDTLS.000333

        #region VR.ATO.INCDTLS.000362

        /*  VR.ATO.INCDTLS.000362
        Where a positive Foreign employment income lump sum E amount is present for a Foreign employment income payment summary, at least one Foreign employment income payment summary lump sum in arrears payment instance must be provided

        Legacy Rule Format:
        (^INCDTLS160 <> NULL AND ^INCDTLS177 > 0 AND Count(^INCDTLS522) = 0) AND ((Count(^INCDTLS150) + CountOccurrence(^INCDTLS160, Count(^INCDTLS522) > 0) + CountOccurrence(^INCDTLS108, Count(^INCDTLS525) > 0) + CountOccurrence(^INCDTLS185, Count(^INCDTLS528) > 0) + CountOccurrence(^INCDTLS207, Count(^INCDTLS531) > 0)) < 15)

        Technical Business Rule Format:
        (^INCDTLS160 <> NULL AND ^INCDTLS177 > 0 AND Count(^INCDTLS522) = 0) AND ((Count(^INCDTLS150) + CountOccurrence(^INCDTLS160, Count(^INCDTLS522) > 0) + CountOccurrence(^INCDTLS108, Count(^INCDTLS525) > 0) + CountOccurrence(^INCDTLS185, Count(^INCDTLS528) > 0) + CountOccurrence(^INCDTLS207, Count(^INCDTLS531) > 0)) < 15)

        Data Elements:

        ^INCDTLS522 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment:FEIPSLumpSumArrearsPayment

        ^INCDTLS108 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation

        ^INCDTLS150 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:LumpSumArrears

        ^INCDTLS160 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment

        ^INCDTLS177 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment:Income.BackPaymentLumpSumE.Amount

        ^INCDTLS185 = INCDTLS:Rp:ForeignIncome

        ^INCDTLS207 = INCDTLS:Rp:ForeignPensionsOrAnnuities

        ^INCDTLS525 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:SISLumpSumArrearsPayment

        ^INCDTLS528 = INCDTLS:Rp:ForeignIncome:FEINPSLumpSumArrearsPayment

        ^INCDTLS531 = INCDTLS:Rp:ForeignPensionsOrAnnuities:FPALumpSumArrearsPayment
        */

        protected void VRATOINCDTLS000362(List<ProcessMessageDocument> response, INCDTLS2026 report)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            bool condition1_ConditionsAreTrue = false;
            bool condition1a_INCDTLS177IsGreaterThan0 = false;
            bool condition1b_INCDTLS522NoInstance = false;
            bool condition2_LumpSumTotalLessThan15 = false;
            int lumpSumTotal = 0;

            // Condition 1: (^INCDTLS160 <> NULL AND ^INCDTLS177 > 0 AND Count(^INCDTLS522) = 0)
            // Checks to see whether the Income Collection is provided
            if (report.Rp_IncomeCollection != null)
            {
                // Runs through each of the income tuple
                foreach (Income incomeTuple in report.Rp_IncomeCollection)
                {
                    // Only runs if both conditions aren't true, once they are both true in a single instance then it will skip over checking the other instances
                    if (!(condition1_ConditionsAreTrue))
                    {
                        // Reset to condition 1s back to false as both need to be true in a single instance
                        condition1a_INCDTLS177IsGreaterThan0 = false;
                        condition1b_INCDTLS522NoInstance = false;

                        // Checks to see whether INCDTLS160 exists
                        if (incomeTuple.Rp_Income_OrganisationNameDetails_ForeignEmploymentCollectionExists)
                        {
                            // ^INCDTLS177 > 0
                            condition1a_INCDTLS177IsGreaterThan0 = incomeTuple.INCDTLS177.GetValueOrDefault() > 0;

                            // Count(^INCDTLS522) = 0)
                            condition1b_INCDTLS522NoInstance = incomeTuple.Rp_Income_OrganisationNameDetails_ForeignEmployment_FEIPSLumpSumArrearsPaymentCollectionCount > 0 ? false : true;
                        }

                        // (^INCDTLS160 <> NULL AND ^INCDTLS177 > 0 AND Count(^INCDTLS522) = 0)
                        condition1_ConditionsAreTrue = condition1a_INCDTLS177IsGreaterThan0 && condition1b_INCDTLS522NoInstance;
                    }
                }
            }

            // Only bother running the second condition if the first half is true
            if (condition1_ConditionsAreTrue)
            {
                // Condition 2: ((Count(^INCDTLS150) + CountOccurrence(^INCDTLS160, Count(^INCDTLS522) > 0) + CountOccurrence(^INCDTLS108, Count(^INCDTLS525) > 0) + CountOccurrence(^INCDTLS185, Count(^INCDTLS528) > 0) + CountOccurrence(^INCDTLS207, Count(^INCDTLS531) > 0)) < 15)
                lumpSumTotal = GetTotalNumberOfLumpSumInArrears(report);
                condition2_LumpSumTotalLessThan15 = lumpSumTotal < 15;

                // (^INCDTLS160 <> NULL AND ^INCDTLS177 > 0 AND Count(^INCDTLS522) = 0) AND ((Count(^INCDTLS150) + CountOccurrence(^INCDTLS160, Count(^INCDTLS522) > 0) + CountOccurrence(^INCDTLS108, Count(^INCDTLS525) > 0) + CountOccurrence(^INCDTLS185, Count(^INCDTLS528) > 0) + CountOccurrence(^INCDTLS207, Count(^INCDTLS531) > 0)) < 15)
                assertion = condition2_LumpSumTotalLessThan15;

                if (assertion)
                {
                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.INCDTLS.000362",
                        Severity = ProcessMessageSeverity.Error,
                        Description = @"Foreign employment income payment summary lump sum in arrears payment details must be provided",
                        LongDescription = @"Where a positive Foreign employment income lump sum E amount is present for a Foreign employment income payment summary, at least one Foreign employment income payment summary lump sum in arrears payment instance must be provided",
                        Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income/tns:OrganisationNameDetails/tns:ForeignEmployment/tns:FEIPSLumpSumArrearsPaymentCollection/tns:FEIPSLumpSumArrearsPayment",
                        Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.INCDTLS.000362" },
                        new ProcessMessageParameter() { Name = "Condition 1: INCDTLS117 or INCDTLS118 or INCDTLS119 (where INCDTLS109 is true) is greater than 0 and no instances of INCDTLS525", Value = condition1_ConditionsAreTrue.ToString() },
                        new ProcessMessageParameter() { Name = "Condition 2: Count of all instances of Lump Sum In Arrears < 15 ", Value = condition2_LumpSumTotalLessThan15.ToString() },
                        new ProcessMessageParameter() { Name = "Total Number of Lump Sum In Arrears ", Value = lumpSumTotal.ToString() },
                    },
                    };

                    response.Add(processMessage);
                }
            }
        }

        #endregion // VR.ATO.INCDTLS.000362

        #region VR.ATO.INCDTLS.000363

        /*  VR.ATO.INCDTLS.000363
        Where a positive Exempt foreign income lump sum in arrears amount is present for a Foreign employment income non-payment summary, at least one Foreign employment income non-payment summary lump sum in arrears payment instance must be provided

        Legacy Rule Format:
        (^INCDTLS185 <> NULL AND ^INCDTLS195 > 0 AND Count(^INCDTLS528) = 0) AND ((Count(^INCDTLS150) + CountOccurrence(^INCDTLS160, Count(^INCDTLS522) > 0) + CountOccurrence(^INCDTLS108, Count(^INCDTLS525) > 0) + CountOccurrence(^INCDTLS185, Count(^INCDTLS528) > 0) + CountOccurrence(^INCDTLS207, Count(^INCDTLS531) > 0)) < 15)

        Technical Business Rule Format:
        (^INCDTLS185 <> NULL AND ^INCDTLS195 > 0 AND Count(^INCDTLS528) = 0) AND ((Count(^INCDTLS150) + CountOccurrence(^INCDTLS160, Count(^INCDTLS522) > 0) + CountOccurrence(^INCDTLS108, Count(^INCDTLS525) > 0) + CountOccurrence(^INCDTLS185, Count(^INCDTLS528) > 0) + CountOccurrence(^INCDTLS207, Count(^INCDTLS531) > 0)) < 15)

        Data Elements:

        ^INCDTLS528 = INCDTLS:Rp:ForeignIncome:FEINPSLumpSumArrearsPayment

        ^INCDTLS108 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation

        ^INCDTLS150 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:LumpSumArrears

        ^INCDTLS160 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment

        ^INCDTLS185 = INCDTLS:Rp:ForeignIncome

        ^INCDTLS195 = INCDTLS:Rp:ForeignIncome:ExemptForeignIncome:Income.LumpSumArrearsPayment.Amount

        ^INCDTLS207 = INCDTLS:Rp:ForeignPensionsOrAnnuities

        ^INCDTLS522 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment:FEIPSLumpSumArrearsPayment

        ^INCDTLS525 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:SISLumpSumArrearsPayment

        ^INCDTLS531 = INCDTLS:Rp:ForeignPensionsOrAnnuities:FPALumpSumArrearsPayment
        */

        protected void VRATOINCDTLS000363(List<ProcessMessageDocument> response, INCDTLS2026 reportIn)
        {
            // Condition 1:
            // (^INCDTLS185 <> NULL AND ^INCDTLS195 > 0 AND Count(^INCDTLS528) = 0)
            bool condition1 =
            (
                // ^INCDTLS185 <> NULL
                reportIn.Rp_ForeignIncomeCollection != null
                &&
                reportIn.Rp_ForeignIncomeCollection.Any
                (
                        INCDTLS195Repeat =>
                        (
                            // INCDTLS195 > 0
                            INCDTLS195Repeat.INCDTLS195.GetValueOrDefault() > 0
                        )
                        &&
                        (
                            // Count(^INCDTLS528) = 0
                            INCDTLS195Repeat.Rp_ForeignIncome_FEINPSLumpSumArrearsPaymentCollectionCount == 0
                        )
                )
            );

            if (condition1)
            {
                int INCDTLS150Count = GetCountOfAllINCDTLS150Tuples(reportIn);
                int INCDTLS160Count = GetCountOfAllINCDTLS160TuplesContainingINCDTLS522Tuples(reportIn);
                int INCDTLS108Count = GetCountOfAllINCDTLS108TuplesContainingINCDTLS525Tuples(reportIn);
                int INCDTLS185Count = GetCountOfAllINCDTLS185TuplesContainingINCDTLS528Tuples(reportIn);
                int INCDTLS207Count = GetCountOfAllINCDTLS207TuplesContainingINCDTLS531Tuples(reportIn);

                // Condition 2: 
                // (Count(^INCDTLS150) + CountOccurrence(^INCDTLS160, Count(^INCDTLS522) > 0) + CountOccurrence(^INCDTLS108, Count(^INCDTLS525) > 0) + CountOccurrence(^INCDTLS185, Count(^INCDTLS528) > 0) + CountOccurrence(^INCDTLS207, Count(^INCDTLS531) > 0)) < 15
                bool assertion = (INCDTLS150Count + INCDTLS160Count + INCDTLS108Count + INCDTLS185Count + INCDTLS207Count) < 15;

                if (assertion)
                {
                    ProcessMessageDocument processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.INCDTLS.000363",
                        Severity = ProcessMessageSeverity.Error,
                        Description = @"Foreign employment income non-payment summary lump sum in arrears payment details must be provided",
                        LongDescription = @"Where a positive Exempt foreign income lump sum in arrears amount is present for a Foreign employment income non-payment summary, at least one Foreign employment income non-payment summary lump sum in arrears payment instance must be provided",
                        Location = "/tns:INCDTLS/tns:Rp/tns:ForeignIncomeCollection/tns:ForeignIncome/tns:FEINPSLumpSumArrearsPaymentCollection/tns:FEINPSLumpSumArrearsPayment",
                        Parameters = new ProcessMessageParameters()
                        {
                            new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.INCDTLS.000363" },
                            new ProcessMessageParameter() { Name = "(^INCDTLS185 <> NULL AND ^INCDTLS195 > 0 AND Count(^INCDTLS528) = 0)", Value = condition1.ToString() },
                            new ProcessMessageParameter() { Name = "Count(^INCDTLS150)", Value = INCDTLS150Count.ToString() },
                            new ProcessMessageParameter() { Name = "CountOccurrence(^INCDTLS160, Count(^INCDTLS522) > 0)", Value = INCDTLS160Count.ToString() },
                            new ProcessMessageParameter() { Name = "CountOccurrence(^INCDTLS108, Count(^INCDTLS525) > 0)", Value = INCDTLS108Count.ToString() },
                            new ProcessMessageParameter() { Name = "CountOccurrence(^INCDTLS185, Count(^INCDTLS528) > 0)", Value = INCDTLS185Count.ToString() },
                            new ProcessMessageParameter() { Name = "CountOccurrence(^INCDTLS207, Count(^INCDTLS531) > 0)", Value = INCDTLS207Count.ToString() },
                        }
                    };

                    response.Add(processMessage);
                }
            }
        }
        #endregion // VR.ATO.INCDTLS.000363

        #region VR.ATO.INCDTLS.W00311

        /*  VR.ATO.INCDTLS.W00311
        Where the total ETP payments for Death benefit ETP paid to a non-dependant of the deceased, from the same payer ABN, is greater than $235,000 (low CAP rate), and total tax withheld from ETP payments for Death benefit ETP paid to a non-dependant of the deceased from the same payer ABN is less than ETP taxable component * 32% (30% is ETP rate 2 + 2% Medicare), check ETP Tax withheld amount
    
        Legacy Rule Format:
        (^INCDTLS138 = 'N') AND (((^INCDTLS308 <> NULL) AND (Sum(ConditionalValue(^INCDTLS308 = [INCDTLS308:N],^INCDTLS140,0)) > 235000) AND (Sum(ConditionalValue(^INCDTLS308 = [INCDTLS308:N],^INCDTLS139,0)) < (Sum(ConditionalValue(^INCDTLS308 = [INCDTLS308:N],^INCDTLS140,0)) * 0.32))) OR ((^INCDTLS307 <> NULL) AND (Sum(ConditionalValue(^INCDTLS307 = [INCDTLS307:N],^INCDTLS140,0)) > 235000) AND (Sum(ConditionalValue(^INCDTLS307 = [INCDTLS307:N],^INCDTLS139,0)) < (Sum(ConditionalValue(^INCDTLS307 = [INCDTLS307:N],^INCDTLS140,0)) * 0.32))))

        [INCDTLS308:N] = ANY OTHER OCCURRENCE OF ConditionalValue((^INCDTLS138 = 'N'),^INCDTLS308,NULL) 
        [INCDTLS307:N] = ANY OTHER OCCURRENCE OF ConditionalValue((^INCDTLS138 = 'N'),^INCDTLS307,NULL)

        Technical Business Rule Format:
        (^INCDTLS138 = 'N') AND (((^INCDTLS308 <> NULL) AND (Sum(ConditionalValue(^INCDTLS308 = [INCDTLS308:N],^INCDTLS140,0)) > 235000) AND (Sum(ConditionalValue(^INCDTLS308 = [INCDTLS308:N],^INCDTLS139,0)) < (Sum(ConditionalValue(^INCDTLS308 = [INCDTLS308:N],^INCDTLS140,0)) * 0.32))) OR ((^INCDTLS307 <> NULL) AND (Sum(ConditionalValue(^INCDTLS307 = [INCDTLS307:N],^INCDTLS140,0)) > 235000) AND (Sum(ConditionalValue(^INCDTLS307 = [INCDTLS307:N],^INCDTLS139,0)) < (Sum(ConditionalValue(^INCDTLS307 = [INCDTLS307:N],^INCDTLS140,0)) * 0.32))))

        [INCDTLS308:N] = ANY OTHER OCCURRENCE OF ConditionalValue((^INCDTLS138 = 'N'),^INCDTLS308,NULL) 
        [INCDTLS307:N] = ANY OTHER OCCURRENCE OF ConditionalValue((^INCDTLS138 = 'N'),^INCDTLS307,NULL)
            
        Data Elements:
            
        ^INCDTLS139 = INCDTLS:Rp:Income:OrganisationNameDetails:EmploymentTerminationPayment:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldEmploymentTerminationPayment.Amount
            
        ^INCDTLS138 = INCDTLS:Rp:Income:OrganisationNameDetails:EmploymentTerminationPayment:IncomeTax.PayAsYouGoWithholding.EmploymentTerminationPaymentType.Code
            
        ^INCDTLS140 = INCDTLS:Rp:Income:OrganisationNameDetails:EmploymentTerminationPayment:Income.EmploymentTerminationPaymentTaxable.Amount
            
        ^INCDTLS307 = INCDTLS:Rp:Income:OrganisationNameDetails:Identifiers.AustralianBusinessNumber.Identifier
            
        ^INCDTLS308 = INCDTLS:Rp:Income:OrganisationNameDetails:Identifiers.WithholdingPayerNumber.Identifier
        */
        public void VRATOINCDTLSW00311(List<ProcessMessageDocument> response, INCDTLS2026 report)
        {
            IEnumerable<Income> incomeCollectionsWithETP = report.Rp_IncomeCollection?.Where(tuple => tuple.Rp_Income_OrganisationNameDetails_EmploymentTerminationPaymentCollectionExists == true);
            if (incomeCollectionsWithETP == null) return;

            foreach (Income incomeCollectionWithETP in incomeCollectionsWithETP.ToList())
            {
                decimal sumETPTaxableAmtMatchingPayerNumber = 0;
                decimal sumCreditTaxWitheldMatchingPayerNumber = 0;
                decimal sumETPTaxableAmtMatchingAbn = 0;
                decimal sumCreditTaxWitheldMatchingAbn = 0;

                report.Rp_IncomeCollection
                    .Where(income => income.INCDTLS138 == "N")
                    .ToList()
                    .ForEach(income =>
                    {
                        if (income.INCDTLS307 != null && income.INCDTLS307 == incomeCollectionWithETP.INCDTLS307)
                        {
                            sumETPTaxableAmtMatchingAbn += income.INCDTLS140.GetValueOrDefault();
                            sumCreditTaxWitheldMatchingAbn = +income.INCDTLS139.GetValueOrDefault();
                        }

                        if (income.INCDTLS308 != null && income.INCDTLS308 == incomeCollectionWithETP.INCDTLS308)
                        {
                            sumETPTaxableAmtMatchingPayerNumber += income.INCDTLS140.GetValueOrDefault();
                            sumCreditTaxWitheldMatchingPayerNumber += income.INCDTLS139.GetValueOrDefault();
                        }
                    });

                bool assertion = incomeCollectionWithETP.INCDTLS138 == "N" &&

                    (incomeCollectionWithETP.INCDTLS308 != null &&
                        sumETPTaxableAmtMatchingPayerNumber > EtpCapForLifeDeathBenefitThreshold &&
                        sumCreditTaxWitheldMatchingPayerNumber < sumETPTaxableAmtMatchingPayerNumber * EtpTaxableComponent) ||

                    (incomeCollectionWithETP.INCDTLS307 != null &&
                        sumETPTaxableAmtMatchingAbn > EtpCapForLifeDeathBenefitThreshold &&
                        sumCreditTaxWitheldMatchingAbn < sumETPTaxableAmtMatchingAbn * EtpTaxableComponent);

                if (assertion)
                {
                    ProcessMessageDocument processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.INCDTLS.W50311",
                        Severity = ProcessMessageSeverity.Warning,
                        Description = @"Check the ETP tax withheld amount and ETP code provided",
                        LongDescription = @"Where the total ETP payments for Death benefit ETP paid to a non-dependant of the deceased, from the same payer ABN, is greater than $245,000 (low CAP rate), and total tax withheld from ETP payments for Death benefit ETP paid to a non-dependant of the deceased from the same payer ABN is less than ETP taxable component * 32% (30% is ETP rate 2 + 2% Medicare), check ETP Tax withheld amount",
                        Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income" + OccurrenceIndex(incomeCollectionWithETP.OccurrenceIndex) + "/tns:OrganisationNameDetails/tns:EmploymentTerminationPayment/tns:TaxPayAsYouGoWithholdingCreditTaxWithheldEmploymentTerminationPaymentA",
                        Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.INCDTLS.W00311" } },
                    };
                    processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "INCDTLS139", Value = incomeCollectionWithETP.INCDTLS139.GetValueOrDefault().ToString() });
                    processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "INCDTLS140", Value = incomeCollectionWithETP.INCDTLS140.GetValueOrDefault().ToString() });
                    processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "INCDTLS138", Value = incomeCollectionWithETP.INCDTLS138 });
                    processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "INCDTLS307", Value = incomeCollectionWithETP.INCDTLS307 });
                    processMessage.Parameters.Add(new ProcessMessageParameter
                    { Name = "INCDTLS308", Value = incomeCollectionWithETP.INCDTLS308 });

                    response.Add(processMessage);
                }
            }
        }
        #endregion // VR.ATO.INCDTLS.W00311

        #region VR.ATO.INCDTLS.000278

        /*  VR.ATO.INCDTLS.000278
        The Lump sum in arrears payment type code within the Salary or wages payment summary instance must not be duplicated

        Legacy Rule Format:
        (AnyOccurrence(^INCDTLS309, HasDuplicateValues(^INCDTLS151)))

        Technical Business Rule Format:
        (AnyOccurrence(^INCDTLS309, HasDuplicateValues(^INCDTLS151)))

        Data Elements:

        ^INCDTLS151 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:LumpSumArrears:Income.LumpSumArrearsPaymentType.Code

        ^INCDTLS309 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages
        */

        protected void VRATOINCDTLS000278(List<ProcessMessageDocument> response, INCDTLS2026 reportIn)
        {
            if (reportIn.Rp_IncomeCollection != null)
            {
                bool assertion =
                (
                    reportIn.Rp_IncomeCollection.Any
                    (
                            INCDTLS150Repeat => HasDuplicateValues
                            (
                                INCDTLS150Repeat.Rp_Income_OrganisationNameDetails_SalaryOrWages_LumpSumArrearsCollection == null
                                ? null
                                : INCDTLS150Repeat.Rp_Income_OrganisationNameDetails_SalaryOrWages_LumpSumArrearsCollection.Select
                                (
                                    f => f.INCDTLS151
                                ).Cast<object>().ToArray()
                            )
                    )
                );

                if (assertion)
                {
                    ProcessMessageDocument processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.INCDTLS.000278",
                        Severity = ProcessMessageSeverity.Error,
                        Description = @"The Lump sum in arrears payment type code within the Salary or wages payment summary instance must not be duplicated",
                        Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income/tns:OrganisationNameDetails/tns:SalaryOrWages/tns:LumpSumArrearsCollection/tns:LumpSumArrears/tns:PaymentTypeC",
                        Parameters = new ProcessMessageParameters() {
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.INCDTLS.000278" }
                    },
                    };
                    response.Add(processMessage);
                }
            }
        }
        #endregion // VR.ATO.INCDTLS.000278

        #region VR.ATO.INCDTLS.000364

        /*  VR.ATO.INCDTLS.000364
        Where a positive Foreign pension or annuity lump sum in arrears amount is present for a Foreign pensions or annuities, at least one Foreign pensions or annuities lump sum in arrears payment instance must be provided

        Legacy Rule Format:
        (^INCDTLS207 <> NULL AND ^INCDTLS213 > 0 AND Count(^INCDTLS531) = 0) AND ((Count(^INCDTLS150) + CountOccurrence(^INCDTLS160, Count(^INCDTLS522) > 0) + CountOccurrence(^INCDTLS108, Count(^INCDTLS525) > 0) + CountOccurrence(^INCDTLS185, Count(^INCDTLS528) > 0) + CountOccurrence(^INCDTLS207, Count(^INCDTLS531) > 0)) < 15)

        Technical Business Rule Format:
        (^INCDTLS207 <> NULL AND ^INCDTLS213 > 0 AND Count(^INCDTLS531) = 0) AND ((Count(^INCDTLS150) + CountOccurrence(^INCDTLS160, Count(^INCDTLS522) > 0) + CountOccurrence(^INCDTLS108, Count(^INCDTLS525) > 0) + CountOccurrence(^INCDTLS185, Count(^INCDTLS528) > 0) + CountOccurrence(^INCDTLS207, Count(^INCDTLS531) > 0)) < 15)

        Data Elements:

        ^INCDTLS531 = INCDTLS:Rp:ForeignPensionsOrAnnuities:FPALumpSumArrearsPayment

        ^INCDTLS108 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation

        ^INCDTLS150 = INCDTLS:Rp:Income:OrganisationNameDetails:SalaryOrWages:LumpSumArrears

        ^INCDTLS160 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment

        ^INCDTLS185 = INCDTLS:Rp:ForeignIncome

        ^INCDTLS207 = INCDTLS:Rp:ForeignPensionsOrAnnuities

        ^INCDTLS213 = INCDTLS:Rp:ForeignPensionsOrAnnuities:Income.LumpSumArrearsPayment.Amount

        ^INCDTLS522 = INCDTLS:Rp:Income:OrganisationNameDetails:ForeignEmployment:FEIPSLumpSumArrearsPayment

        ^INCDTLS525 = INCDTLS:Rp:Income:OrganisationNameDetails:AnnuitiesAndSuperannuation:SISLumpSumArrearsPayment

        ^INCDTLS528 = INCDTLS:Rp:ForeignIncome:FEINPSLumpSumArrearsPayment
        */
        protected void VRATOINCDTLS000364(List<ProcessMessageDocument> response, INCDTLS2026 reportIn)
        {
            // Condition 1:
            // (^INCDTLS207 <> NULL AND ^INCDTLS213 > 0 AND Count(^INCDTLS531) = 0)
            bool condition1 =
            (
                // ^INCDTLS207 <> NULL
                reportIn.Rp_ForeignPensionsOrAnnuitiesCollection != null
                &&
                reportIn.Rp_ForeignPensionsOrAnnuitiesCollection.Any
                (
                        INCDTLS213Repeat =>
                        (
                            // INCDTLS213 > 0
                            INCDTLS213Repeat.INCDTLS213.GetValueOrDefault() > 0
                        )
                        &&
                        (
                            // Count(^INCDTLS531) = 0
                            INCDTLS213Repeat.Rp_ForeignPensionsOrAnnuities_FPALumpSumArrearsPaymentCollectionCount == 0
                        )
                )
            );

            if (condition1)
            {
                int INCDTLS150Count = GetCountOfAllINCDTLS150Tuples(reportIn);
                int INCDTLS160Count = GetCountOfAllINCDTLS160TuplesContainingINCDTLS522Tuples(reportIn);
                int INCDTLS108Count = GetCountOfAllINCDTLS108TuplesContainingINCDTLS525Tuples(reportIn);
                int INCDTLS185Count = GetCountOfAllINCDTLS185TuplesContainingINCDTLS528Tuples(reportIn);
                int INCDTLS207Count = GetCountOfAllINCDTLS207TuplesContainingINCDTLS531Tuples(reportIn);

                // Condition 2:
                // (Count(^INCDTLS150) + CountOccurrence(^INCDTLS160, Count(^INCDTLS522) > 0) + CountOccurrence(^INCDTLS108, Count(^INCDTLS525) > 0) + CountOccurrence(^INCDTLS185, Count(^INCDTLS528) > 0) + CountOccurrence(^INCDTLS207, Count(^INCDTLS531) > 0)) < 15
                bool assertion = (INCDTLS150Count + INCDTLS160Count + INCDTLS108Count + INCDTLS185Count + INCDTLS207Count) < 15;

                if (assertion)
                {
                    ProcessMessageDocument processMessage = new ProcessMessageDocument();

                    processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.INCDTLS.000364",
                        Severity = ProcessMessageSeverity.Error,
                        Description = @"Foreign pensions or annuities lump sum in arrears payment details must be provided",
                        LongDescription = @"Where a positive Foreign pension or annuity lump sum in arrears amount is present for a Foreign pensions or annuities, at least one Foreign pensions or annuities lump sum in arrears payment instance must be provided",
                        Location = "/tns:INCDTLS/tns:Rp/tns:ForeignPensionsOrAnnuitiesCollection/tns:ForeignPensionsOrAnnuities/tns:FPALumpSumArrearsPaymentCollection/tns:FPALumpSumArrearsPayment",
                        Parameters = new ProcessMessageParameters()
                        {
                            new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.INCDTLS.000364" },
                            new ProcessMessageParameter() { Name = "(^INCDTLS207 <> NULL AND ^INCDTLS213 > 0 AND Count(^INCDTLS531) = 0)", Value = condition1.ToString() },
                            new ProcessMessageParameter() { Name = "Count(^INCDTLS150)", Value = INCDTLS150Count.ToString() },
                            new ProcessMessageParameter() { Name = "CountOccurrence(^INCDTLS160, Count(^INCDTLS522) > 0)", Value = INCDTLS160Count.ToString() },
                            new ProcessMessageParameter() { Name = "CountOccurrence(^INCDTLS108, Count(^INCDTLS525) > 0)", Value = INCDTLS108Count.ToString() },
                            new ProcessMessageParameter() { Name = "CountOccurrence(^INCDTLS185, Count(^INCDTLS528) > 0)", Value = INCDTLS185Count.ToString() },
                            new ProcessMessageParameter() { Name = "CountOccurrence(^INCDTLS207, Count(^INCDTLS531) > 0)", Value = INCDTLS207Count.ToString() },
                        },
                    };

                    response.Add(processMessage);
                }
            }
        }

        #endregion // VR.ATO.INCDTLS.000364

        #region VR.ATO.INCDTLS.000412

        /*  VR.ATO.INCDTLS.000412
        A positive ETP taxable component amount must be provided. When the Employment Termination Payment (ETP) only includes an ETP tax free component, it must not to be included in the lodgment

        Legacy Rule Format:
            (AnyOccurrence(^INCDTLS140, ^INCDTLS140 = 0))

        Technical Business Rule Format:
                (AnyOccurrence(^INCDTLS140, ^INCDTLS140 = 0))

        Data Elements:
    
        ^INCDTLS140 = INCDTLS:Rp:Income:OrganisationNameDetails:EmploymentTerminationPayment:Income.EmploymentTerminationPaymentTaxable.Amount
        
        */
        protected void VRATOINCDTLS000412(List<ProcessMessageDocument> response, INCDTLS2026 report)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            assertion = (report.Rp_IncomeCollection == null ? false : report.Rp_IncomeCollection.Any(INCDTLS140Repeat => INCDTLS140Repeat.INCDTLS140 == 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000412",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"A positive ETP taxable component amount must be provided",
                    LongDescription = @"A positive ETP taxable component amount must be provided. When the Employment Termination Payment (ETP) only includes an ETP tax free component, it must not to be included in the lodgment.",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeCollection/tns:Income/tns:OrganisationNameDetails/tns:EmploymentTerminationPayment/tns:TaxableA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.INCDTLS.000412" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "INCDTLS140", Value = "INCDTLS140" });

                response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.INCDTLS.000412

        #region VR.ATO.INCDTLS.000460
        /*  VR.ATO.INCDTLS.000460
            When any Taxable payments income classification decision is "Other Income", one of the Other income types must be "Taxable Payments and Other Third Party Reported Income"

            Legacy Rule Format:
            AnyOccurrence(^INCDTLS581, ^INCDTLS581 = "Other Income") AND CountOccurrence(^INCDTLS449, ^INCDTLS449 = "Taxable Payments and Other Third Party Reported Income") = 0

            Technical Business Rule Format:
            AnyOccurrence(^INCDTLS581, ^INCDTLS581 = "Other Income") AND CountOccurrence(^INCDTLS449, ^INCDTLS449 = "Taxable Payments and Other Third Party Reported Income") = 0

            Data Elements:

            ^INCDTLS581 = INCDTLS:Rp:TaxablePayment:Income.ClassificationDecision.Code

            ^INCDTLS449 = INCDTLS:Rp:IncomeOther:Income.OtherIncomeType.Code
            */

        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000460(List<ProcessMessageDocument> response, INCDTLS2026 reportIn)
        {
            ProcessMessageDocument processMessage;
            bool assertion = false;
            if (report.Rp_TaxablePaymentCollection != null && report.Rp_TaxablePaymentCollection.Any(tuple => tuple.INCDTLS581 == "Other Income"))
            {
                assertion = report.Rp_IncomeOtherCollection != null && report.Rp_IncomeOtherCollection.Count(tuple => tuple.INCDTLS449 == "Taxable Payments and Other Third Party Reported Income") == 0;
            };

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000460",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other income type must be provided",
                    LongDescription = @"When any Taxable payments income classification decision is ""Other Income"", one of the Other income types must be ""Taxable Payments and Other Third Party Reported Income""",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeOtherCollection/tns:IncomeOther/tns:OtherIncomeTypeC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.INCDTLS.000460" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "INCDTLS449", Value = "INCDTLS449" });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.INCDTLS.000461
        /*  VR.ATO.INCDTLS.000461
            Other income amount where Other income type is "Taxable Payments and Other Third Party Reported Income" must not be less than the sum of all Taxable payments gross amount (including GST plus any tax withheld) minus Taxable payments total GST amount where Taxable payments income classification decision is "Other Income"

            Legacy Rule Format:
            AnyOccurrence(^INCDTLS449, ^INCDTLS449 = "Taxable Payments and Other Third Party Reported Income") AND (Sum(ConditionalValue(^INCDTLS449 = "Taxable Payments and Other Third Party Reported Income", ^INCDTLS451, 0)) < Sum(ConditionalValue(^INCDTLS581 = "Other Income", ^INCDTLS575 - ^INCDTLS577), 0) - 5)

            Technical Business Rule Format:
            AnyOccurrence(^INCDTLS449, ^INCDTLS449 = "Taxable Payments and Other Third Party Reported Income") AND (Sum(ConditionalValue(^INCDTLS449 = "Taxable Payments and Other Third Party Reported Income", ^INCDTLS451, 0)) < Sum(ConditionalValue(^INCDTLS581 = "Other Income", ^INCDTLS575 - ^INCDTLS577), 0) - 5)

            Data Elements:

            ^INCDTLS449 = INCDTLS:Rp:IncomeOther:Income.OtherIncomeType.Code

            ^INCDTLS451 = INCDTLS:Rp:IncomeOther:Income.Other.Amount
        
            ^INCDTLS575 = INCDTLS:Rp:TaxablePayment:Income.Gross.Amount

            ^INCDTLS577 = INCDTLS:Rp:TaxablePayment:GoodsAndServicesTax.Liability.Amount

            ^INCDTLS581 = INCDTLS:Rp:TaxablePayment:Income.ClassificationDecision.Code
            
            */

        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOINCDTLS000461(List<ProcessMessageDocument> response, INCDTLS2026 reportIn)
        {
            ProcessMessageDocument processMessage;
            bool assertion = false;
            decimal? otherIncomeSum=null;
            decimal taxableOtherIncome = 0;

            if (report.Rp_IncomeOtherCollection != null && report.Rp_IncomeOtherCollection.Any(tuple => tuple.INCDTLS449 == "Taxable Payments and Other Third Party Reported Income" && tuple.INCDTLS451.HasValue))
            {
                otherIncomeSum = (report.Rp_IncomeOtherCollection.Where(tuple => tuple.INCDTLS449 == "Taxable Payments and Other Third Party Reported Income" && tuple.INCDTLS451.HasValue)).Sum(tuple => tuple.INCDTLS451).Value;
            };

            if (report.Rp_TaxablePaymentCollection != null && report.Rp_TaxablePaymentCollection.Any(tuple => tuple.INCDTLS581 == "Other Income" && (tuple.INCDTLS575.HasValue || tuple.INCDTLS577.HasValue)))
            {
                taxableOtherIncome = (report.Rp_TaxablePaymentCollection.Where(tuple => tuple.INCDTLS581 == "Other Income")).Sum(tuple => (tuple.INCDTLS575.HasValue ? tuple.INCDTLS575.Value : 0) - (tuple.INCDTLS577.HasValue ? tuple.INCDTLS577.Value : 0));
            };

            assertion = otherIncomeSum.HasValue ? otherIncomeSum.Value < (taxableOtherIncome - 5) : false;

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.INCDTLS.000461",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other income amount is incorrect",
                    LongDescription = @"Other income amount where Other income type is ""Taxable Payments and Other Third Party Reported Income"" must not be less than the sum of all Taxable payments gross amount (including GST plus any tax withheld) minus Taxable payments total GST amount where Taxable payments income classification decision is ""Other Income""",
                    Location = "/tns:INCDTLS/tns:Rp/tns:IncomeOtherCollection/tns:IncomeOther/tns:A",
                    Parameters = new ProcessMessageParameters() {
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.INCDTLS.000461" },
                        new ProcessMessageParameter() { Name = "SUM([INCDTLS451])", Value =  otherIncomeSum.ToString() },
                        new ProcessMessageParameter() { Name = "SUM([INCDTLS575 - INCDTLS577])", Value =  taxableOtherIncome.ToString() }
                    }
                };
                if (report.Rp_IncomeOtherCollection != null)
                {
                    foreach (var item in report.Rp_IncomeOtherCollection.Where(tuple => tuple.INCDTLS449 == "Taxable Payments and Other Third Party Reported Income"))
                    {
                        processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "INCDTLS451", Value = item.INCDTLS451.GetValueOrDefault().ToString() });
                    }
                }
                if (report.Rp_TaxablePaymentCollection != null)
                {
                    foreach (var item in report.Rp_TaxablePaymentCollection.Where(tuple => tuple.INCDTLS581 == "Other Income"))
                    {
                        processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "INCDTLS575", Value = item.INCDTLS575.GetValueOrDefault().ToString() });
                        processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "INCDTLS577", Value = item.INCDTLS577.GetValueOrDefault().ToString() });
                    }
                }
                response.Add(processMessage);
            }
        }
        #endregion

        #endregion Manual Rules - Rules marked for manual coding

    }
}