using System;
using System.Xml;
using System.IO;
using System.Collections.Generic;
using DataContracts;
using System.Text.RegularExpressions;
using System.Reflection;
using System.Linq;
using System.Xml.Schema;
using System.Text;

namespace Ato.EN.IntegrationServices.CodeGenerationINCDTLS
{
    /// <summary>
    /// XML Consumer for INCDTLS
    /// </summary>
    public class INCDTLS2026XmlConsumer
    {
        const string MissingMandatoryMessage = "A mandatory field has not been completed.";
        const string ContactYourProviderMsg = "The message did not pass XML validation. Please contact your software provider.";
        const string InvalidDataValidation = "A field contains invalid data (such as letters in numeric or date field).";
        private const string ErrorCode1 = "CMN.ATO.GEN.XML01";
        private const string ErrorCode3 = "CMN.ATO.GEN.XML03";
        private const string ErrorCode4 = "CMN.ATO.GEN.XML04";
        private const string ErrorCode6 = "CMN.ATO.GEN.XML06";
        private bool _found = false;
        private int _parentCollectionCount;
        private int _childCollectionCount;
        private bool _isExiting = false;
        private bool _isValidationError = false;
        private string _lastPath = string.Empty;
        private ErrorMessageType _validationError = new ErrorMessageType(ErrorDescriptor.NoError);
#region Error Messages
        public List<ProcessMessageDocument> ErrorMessages { get; set; }

        public bool HasErrors
        {
            get
            {
                return ErrorMessages != null && ErrorMessages.Count > 0;
            }
        }

        private string GetCurrentLocation()
        {
            StringBuilder location = new StringBuilder();
            string[] paths = _currentXPath.ToArray<string>();
            for (int i = paths.Length - 1; i > -1; i--)
            {
                location.Append(paths[i]);
            }

            return location.ToString();
        }

        private void MissingElementError()
        {
            ProcessMessageDocument processMessage;
            processMessage = new ProcessMessageDocument()
            {
                Code = "CMN.ATO.GEN.XML04",
                Description = MissingMandatoryMessage,
                SeverityAsString = "Error",
                Location = GetCurrentLocation(),
            };
            this.ErrorMessages.Add(processMessage);
        }

#endregion Error Messages
#region Embedded Schema
        private static readonly XmlSchemaSet EmbeddedXmlSchemaSet;
        static INCDTLS2026XmlConsumer()
        {
            Assembly executingAssembly = Assembly.GetExecutingAssembly();
            string[] manifestResourceNames = executingAssembly.GetManifestResourceNames();
            string resourceName;
            Stream embeddedSchemaStream;
            XmlSchema embeddedXmlSchema;
            if (manifestResourceNames != null)
            {
                resourceName = manifestResourceNames.FirstOrDefault(rn => rn.Contains("ato.incdtls.0007.2026.01.00.xsd") && rn.EndsWith(".xsd"));
                if (resourceName != null)
                {
                    embeddedSchemaStream = executingAssembly.GetManifestResourceStream(resourceName);
                    embeddedXmlSchema = XmlSchema.Read(embeddedSchemaStream, SchemaCallback);
                    EmbeddedXmlSchemaSet = new XmlSchemaSet();
                    EmbeddedXmlSchemaSet.Add(embeddedXmlSchema);
                    EmbeddedXmlSchemaSet.Compile();
                }
            }
        }

        private static void SchemaCallback(object sender, ValidationEventArgs args)
        {
            if (args.Severity == XmlSeverityType.Error)
                throw new XmlSchemaException(args.Message);
        }

#endregion Embedded Schema
#region Xml Reader Settings
        private static XmlReaderSettings ReaderSettings = new XmlReaderSettings()
        {
            CloseInput = false,
            ConformanceLevel = ConformanceLevel.Fragment,
            IgnoreWhitespace = true,
            IgnoreComments = true,
            IgnoreProcessingInstructions = true,
        };
        private XmlReaderSettings GetValidatingReaderSettings()
        {
            XmlReaderSettings validatingReaderSettings = new XmlReaderSettings()
            {
                CloseInput = false,
                ConformanceLevel = ConformanceLevel.Fragment,
                IgnoreWhitespace = true,
                IgnoreComments = true,
                IgnoreProcessingInstructions = true,
                ValidationType = ValidationType.Schema,
            };
            if (EmbeddedXmlSchemaSet == null)
            {
                throw new XmlSchemaException("Embedded Schema is Null");
            }
            else
            {
                validatingReaderSettings.Schemas.Add(EmbeddedXmlSchemaSet);
                validatingReaderSettings.ValidationEventHandler += new ValidationEventHandler(ValidationCallBack);
            }

            return validatingReaderSettings;
        }

        private static Regex DataTypeFailureExpression = new Regex("'(?<Uniqueid>.*?)' element is invalid.*value '(?<Value>.*?)' is invalid.*datatype '(?<DataType>.*?)'", RegexOptions.IgnoreCase | RegexOptions.Compiled | RegexOptions.Singleline);
        private void ValidationCallBack(object sender, ValidationEventArgs args)
        {
            var msg = args.Message;
            var msgToLower = msg.ToLower();
            ProcessMessageDocument processMessage = null;
            if (msgToLower.Contains("invalid according to its datatype"))
            {
                Match match = DataTypeFailureExpression.Match(args.Message);
                if (match.Success && match.Groups.Count == 4)
                {
                    var inner = args.Exception.InnerException;
                    var hint = (inner != null && !string.IsNullOrEmpty(inner.Message)) ? " Hint: " + inner.Message : string.Empty;
                    processMessage = BuildProcessMessageDocument(InvalidDataValidation, string.Empty, ErrorCode3);
                    string uniqueID = match.Groups["Uniqueid"].Value;
                    string value = match.Groups["Value"].Value;
                    string dataType = match.Groups["DataType"].Value;
                    var longDescription = string.Format(@"The value specified for an item does not match the item type (value = ""{0}"", item type = {1}, uniqueID = {2})", value, dataType, uniqueID);
                    processMessage.LongDescription = longDescription + hint;
                    processMessage.Parameters = new ProcessMessageParameters();
                    processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "uniqueID", Value = uniqueID });
                    processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "value", Value = value });
                    processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "dataType", Value = dataType });
                }
            }
            else if (msgToLower.Contains("has incomplete content"))
            {
                _validationError = GetErrorDescriptorForHasIncompleteContent(msg);
                processMessage = BuildProcessMessageDocument(MissingMandatoryMessage, msg + _validationError.Hint, _validationError.Code);
                _isValidationError = true;
            }
            else if (msgToLower.Contains("has invalid child element"))
            {
                _validationError = GetErrorDescriptorForHasInvalidChildElement(msg);
                var shortMessage = _validationError.Code == ErrorCode4 ? MissingMandatoryMessage : ContactYourProviderMsg;
                processMessage = BuildProcessMessageDocument(shortMessage, msg + _validationError.Hint, _validationError.Code);
                _isValidationError = true;
            }
            else
            {
                processMessage = BuildProcessMessageDocument(ContactYourProviderMsg, msg, ErrorCode1);
                _isValidationError = true;
            }

            this.ErrorMessages.Add(processMessage);
        }

        private ProcessMessageDocument BuildProcessMessageDocument(string shortDescription, string longDescription, string code)
        {
            return new ProcessMessageDocument()
            {
                Code = code,
                Description = shortDescription,
                LongDescription = longDescription,
                SeverityAsString = "Error",
                Location = GetCurrentLocation()
            };
        }

        private ProcessMessageDocument BuildProcessMessageDocument(ProcessMessageDocument processMessage, string newLocation)
        {
            return new ProcessMessageDocument()
            {
                Code = processMessage.Code,
                Description = processMessage.Description,
                LongDescription = processMessage.LongDescription,
                SeverityAsString = "Error",
                Location = newLocation
            };
        }

        private enum ErrorDescriptor
        {
            NoError = 0,
            MandatoryElementError,
            NonMandatoryElementError,
            UnknownElement,
            Duplicate
        }

        private struct ErrorMessageType
        {
            public string Hint, Code, ExpectedNextValidElement;
            public ErrorDescriptor Errno;
            public ErrorMessageType(ErrorDescriptor errorDescriptor)
            {
                Errno = errorDescriptor;
                Hint = "";
                Code = "";
                ExpectedNextValidElement = "";
            }

            public ErrorMessageType(ErrorDescriptor errorDescriptor, string hint, string code, string elementName)
            {
                Errno = errorDescriptor;
                Hint = hint;
                Code = code;
                ExpectedNextValidElement = elementName;
            }
        }

        private ErrorMessageType GetErrorDescriptorForHasIncompleteContent(string validationErrorMessage)
        {
            ErrorMessageType retval = new ErrorMessageType();
            if (string.IsNullOrWhiteSpace(validationErrorMessage))
                return retval;
            validationErrorMessage = validationErrorMessage.Replace(" ", string.Empty);
            validationErrorMessage = validationErrorMessage.Replace("'", " ");
            var parts = validationErrorMessage.Split(' ');
            if (parts.Length != 9)
                return retval;
            var elementBeingProcessed = parts[1];
            var csvList = parts[5];
            retval = ParseHasIncompleteContentCVSList(csvList, elementBeingProcessed);
            return retval;
        }

        private ErrorMessageType GetErrorDescriptorForHasInvalidChildElement(string validationErrorMessage)
        {
            string msg = string.Empty;
            ErrorMessageType retval = new ErrorMessageType();
            if (string.IsNullOrWhiteSpace(validationErrorMessage))
                return retval;
            validationErrorMessage = validationErrorMessage.Replace(" ", string.Empty);
            validationErrorMessage = validationErrorMessage.Replace("'", " ");
            var parts = validationErrorMessage.Split(' ');
            //decide which message format to process
            if (validationErrorMessage.ToLower().Contains("listofpossibleelements"))
            {
                if (parts.Length == 13 || parts.Length == 12 || parts.Length == 10)
                {
                    var csvList = parts[9];
                    var elementBeingProcessed = parts[1];
                    var invalidChildElementName = parts[5];
                    retval = ParseCvsListForHasInvalidChildElementLongFormat(csvList, invalidChildElementName, elementBeingProcessed);
                }

                return retval;
            }
            else
            {
                if (parts.Length != 9)
                    return retval;
                var csvList = parts[5];
                var elementBeingProcessed = parts[1];
                var invalidChildElementName = parts[5];
                retval = ParseCvsListForHasInvalidChildElementLongFormat(csvList, invalidChildElementName, elementBeingProcessed);
            }

            return retval;
        }

        private ErrorMessageType ParseCvsListForHasInvalidChildElementShortFormat(string csvList, string invalidChildElementName, string elementBeingProcessed)
        {
            ErrorMessageType retval = new ErrorMessageType();
            if (!string.IsNullOrEmpty(csvList))
            {
                var lastElementInListOfPossibleElements = GetLastElementInCsvList(csvList);
                retval.Hint = string.Format(" Hint: while processing parent element [{0}] child element [{1}] was unexpected", elementBeingProcessed, invalidChildElementName);
                retval.Code = ErrorCode1;
                retval.Errno = ErrorDescriptor.UnknownElement;
                retval.ExpectedNextValidElement = invalidChildElementName;
            }

            return retval;
        }

        private string GetLastElementInCsvList(string csvList)
        {
            var list = (csvList.Contains(',')) ? new List<string>(csvList.Split(',')) : new List<string>
            {
                csvList
            };
            string lastElement = list.Last();
            if (lastElement.EndsWith("...."))
            {
                list.RemoveAt(list.Count - 1);
            }

            return list.Last();
        }

        private ErrorMessageType ParseCvsListForHasInvalidChildElementLongFormat(string csvList, string invalidChildElementName, string elementBeingProcessed)
        {
            ErrorMessageType retval = new ErrorMessageType();
            if (!string.IsNullOrEmpty(csvList))
            {
                var lastElementInListOfPossibleElements = GetLastElementInCsvList(csvList);
                var elementList = new List<string>(_elementCsvList.Split(','));
                if (ContainsElementForElementBeingProcessed(elementList, elementBeingProcessed, invalidChildElementName))
                {
                    var indexOfElementBeingProcessed = GetIndexOfElementBeingProcessed(elementList, elementBeingProcessed);
                    var indexOfinvalidChildElementName = GetIndexOfElementUnderElementBeingProcessed(elementList, elementBeingProcessed, invalidChildElementName);
                    var indexOflastElementInListOfPossibleElements = GetIndexOfElementUnderElementBeingProcessed(elementList, elementBeingProcessed, lastElementInListOfPossibleElements);
                    if (indexOfinvalidChildElementName > indexOflastElementInListOfPossibleElements)
                    {
                        retval.Hint = string.Format(" Hint: while processing parent element [{0}] child mandatory element [{1}] was expected but not found", elementBeingProcessed, lastElementInListOfPossibleElements);
                        retval.Code = ErrorCode4;
                        retval.Errno = ErrorDescriptor.MandatoryElementError;
                        retval.ExpectedNextValidElement = lastElementInListOfPossibleElements;
                    }
                    else
                    {
                        retval.Hint = string.Format(" Hint: while processing parent element [{0}] child non mandatory element [{1}] was unexpected", elementBeingProcessed, invalidChildElementName);
                        retval.Code = ErrorCode1;
                        retval.Errno = (indexOfinvalidChildElementName != indexOflastElementInListOfPossibleElements) ? ErrorDescriptor.NonMandatoryElementError : ErrorDescriptor.Duplicate;
                        retval.ExpectedNextValidElement = invalidChildElementName;
                    }
                }
                else
                {
                    //this element is unknown to the schema.
                    retval.Hint = string.Format(" Hint: while processing parent element [{0}] child element [{1}] was unexpected", elementBeingProcessed, invalidChildElementName);
                    retval.Code = ErrorCode1;
                    retval.Errno = ErrorDescriptor.UnknownElement;
                    retval.ExpectedNextValidElement = invalidChildElementName;
                }
            }

            return retval;
        }

        private int GetIndexOfElementUnderElementBeingProcessed(List<string> validElementList, string elementBeingProcessed, string elementName)
        {
            int result;
            string elementWithParentNameMatch = elementBeingProcessed + "#" + elementName;
            result = validElementList.IndexOf(elementWithParentNameMatch);
            if (result < 0)
            {
                result = validElementList.IndexOf(elementName);
            }

            return result;
        }

        private int GetIndexOfElementBeingProcessed(List<string> validElementList, string elementBeingProcessed)
        {
            return validElementList.IndexOf(elementBeingProcessed);
        }

        private bool ContainsElementForElementBeingProcessed(List<string> validElementList, string elementBeingProcessed, string elementName)
        {
            return validElementList.Contains(elementBeingProcessed + "#" + elementName) || validElementList.Contains(elementName);
        }

        private ErrorMessageType ParseHasIncompleteContentCVSList(string csvList, string elementBeingProcessed)
        {
            ErrorMessageType retval = new ErrorMessageType();
            if (!string.IsNullOrEmpty(csvList))
            {
                var lastElementInListOfPossibleElements = GetLastElementInCsvList(csvList);
                retval.Hint = string.Format(" Hint: while processing parent element [{0}] child mandatory element [{1}] was not found", elementBeingProcessed, lastElementInListOfPossibleElements);
                retval.Code = ErrorCode4;
                retval.Errno = ErrorDescriptor.MandatoryElementError;
                retval.ExpectedNextValidElement = lastElementInListOfPossibleElements;
            }

            return retval;
        }

        private void StartEndDateError(DateTime startDate, DateTime endDate)
        {
            ProcessMessageDocument processMessage;
            string longDescription = string.Format("End date is earlier than start date (state date = {0}, end date = {1})", startDate.ToString("yyyy-MM-dd"), endDate.ToString("yyyy-MM-dd"));
            processMessage = new ProcessMessageDocument()
            {
                Code = "CMN.ATO.GEN.XML06",
                Description = "End date is earlier than start date.",
                LongDescription = longDescription,
                SeverityAsString = "Error",
                Location = GetCurrentLocation(),
            };
            processMessage.Parameters = new ProcessMessageParameters();
            processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "startDate", Value = startDate.ToString("yyyy-MM-dd") });
            processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "endDate", Value = endDate.ToString("yyyy-MM-dd") });
            this.ErrorMessages.Add(processMessage);
        }

        private void StartEndDateError(DateTime? startDate, DateTime? endDate)
        {
            ProcessMessageDocument processMessage;
            if (startDate == null || endDate == null)
                return;
            string longDescription = string.Format("End date is earlier than start date (state date = {0}, end date = {1})", startDate.Value.ToString("yyyy-MM-dd"), endDate.Value.ToString("yyyy-MM-dd"));
            processMessage = new ProcessMessageDocument()
            {
                Code = "CMN.ATO.GEN.XML06",
                Description = "End date is earlier than start date.",
                LongDescription = longDescription,
                SeverityAsString = "Error",
                Location = GetCurrentLocation(),
            };
            processMessage.Parameters = new ProcessMessageParameters();
            processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "startDate", Value = startDate.Value.ToString("yyyy-MM-dd") });
            processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "endDate", Value = endDate.Value.ToString("yyyy-MM-dd") });
            this.ErrorMessages.Add(processMessage);
        }

#endregion Xml Reader Settings
#region IsEmptyOrNilElement
        private static bool IsEmptyOrNilElement(XmlReader reader)
        {
            bool emptyOrNil = false;
            if (reader.IsEmptyElement)
            {
                emptyOrNil = true;
            }
            else
            {
                if (reader.HasAttributes)
                {
                    string nilValue = reader.GetAttribute("nil", "http://www.w3.org/2001/XMLSchema-instance");
                    if (nilValue != null && nilValue.ToLowerInvariant() == "true" || nilValue == "1")
                    {
                        emptyOrNil = true;
                    }
                }
            }

            return emptyOrNil;
        }

#endregion IsEmptyOrNilElement
#region ToBoolean
        private bool? ToBoolean(string str)
        {
            bool returnValue;
            string value = (str ?? "").Trim();
            if (value == "0" || value == "1")
                return Convert.ToBoolean(Convert.ToInt32(value));
            if (Boolean.TryParse(value, out returnValue))
                return returnValue;
            return null;
        }

#endregion ToBoolean
#region MoveToContent
        public bool MoveToContent(XmlReader reader)
        {
            try
            {
                reader.MoveToContent();
                return true;
            }
            catch (XmlException ex)
            {
                ProcessMessageDocument processMessage;
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.XML01",
                    Description = ContactYourProviderMsg,
                    LongDescription = ex.Message,
                    SeverityAsString = "Error",
                };
                this.ErrorMessages.Add(processMessage);
                return false;
            }
        }

#endregion MoveToContent
#region ReadToNextElement
        public bool ReadToNextElement(XmlReader reader, bool isMandatory)
        {
            return ReadToNextElement(reader);
        }

        public bool ReadToNextElement(XmlReader reader)
        {
            bool retval = false;
            if ((reader.EOF && !_isValidationError) || _isExiting)
            {
                return false; //do nothing
            }

            if (!_isValidationError)
            {
                retval = SetReaderToNextElement(reader); //position the reader on the next valid element
            }

            if (_isValidationError)
            {
                //process validation errors
                var expectedNextElement = (!string.IsNullOrEmpty(_validationError.ExpectedNextValidElement)) ? _validationError.ExpectedNextValidElement : reader.LocalName;
                var actualNextValidElement = reader.LocalName;
                var location = GetCurrentLocation();
                if (_validationError.Errno == ErrorDescriptor.Duplicate)
                {
                    location = _lastPath;
                }

                var xpath = RemoveLeadingAndTrailingSlashes(location);
                var element = GetElementDescriptor(xpath);
                if (_validationError.Errno == ErrorDescriptor.UnknownElement)
                {
                    //set the xpath for elements that are not known to the schema
                    location = "/" + element.Path + "/" + element.Prefix + ":" + _validationError.ExpectedNextValidElement;
                    _isExiting = true;
                }
                else if (_validationError.Errno == ErrorDescriptor.NonMandatoryElementError)
                {
                    //set the xpath for the optional element that caused a validation
                    location = "/" + element.Path + "/" + element.Prefix + ":" + reader.LocalName;
                    _isExiting = true;
                }
                else
                {
                    //decide if we have spooled to the missing mandatory element. 
                    _isExiting = expectedNextElement == element.PathLeafNode;
                }

                if (_isExiting)
                {
                    //The location has changed since detection of the validation error - so update it.
                    UpdateErrorMessageWithLocation(location);
                }

                return false;
            }

            //Save previous path so we can detect duplicate elements
            _lastPath = GetCurrentLocation();
            return retval;
        }

        private bool ReadNext(XmlReader reader)
        {
            return !reader.EOF && reader.Read();
        }

        private void UpdateErrorMessageWithLocation(string newLocation)
        {
            var lastProcessMessage = ErrorMessages.Last();
            ErrorMessages.Remove(lastProcessMessage);
            var newProcessMessage = BuildProcessMessageDocument(lastProcessMessage, newLocation);
            ErrorMessages.Add(newProcessMessage);
        }

        private string RemoveLeadingAndTrailingSlashes(string xpath)
        {
            var xpathLength = xpath.Length;
            if (xpathLength <= 0)
                return string.Empty;
            var firstChar = xpath.Substring(0, 1);
            var lastChar = xpath.Substring(xpathLength - 1, 1);
            if (firstChar == "/")
            {
                xpath = xpath.Remove(0, 1);
                xpathLength--;
            }

            if (lastChar == "/")
            {
                xpath = xpath.Remove(xpathLength - 1, 1);
            }

            return xpath;
        }

        private struct ElementDescriptor
        {
            public string Prefix, PathLeafNode, Path;
            public ElementDescriptor(string prefix, string leaf, string path)
            {
                Prefix = prefix;
                PathLeafNode = leaf;
                Path = path;
            }
        }

        private ElementDescriptor GetElementDescriptor(string xpath)
        {
            var retval = new ElementDescriptor();
            var parts = xpath.Split('/');
            var numberOfParts = parts.Length;
            if (numberOfParts > 0)
            {
                var leafNode = parts[numberOfParts - 1];
                var p = leafNode.Split(':');
                numberOfParts = p.Length;
                if (numberOfParts == 1)
                {
                    retval.PathLeafNode = p[0];
                    retval.Prefix = string.Empty;
                }
                else if (numberOfParts == 2)
                {
                    retval.PathLeafNode = p[1];
                    retval.Prefix = p[0];
                }

                var leafNodeLength = leafNode.Length;
                var path = xpath.Remove(xpath.Length - leafNodeLength, leafNodeLength);
                retval.Path = RemoveLeadingAndTrailingSlashes(path);
            }

            return retval;
        }

        private bool SetReaderToNextElement(XmlReader reader)
        {
            try
            {
                if (reader.EOF || _isExiting)
                    return false;
                if (_found)
                {
                    reader.Read();
                    _found = false;
                }

                while (!reader.EOF && reader.NodeType != XmlNodeType.Element)
                {
                    reader.Read();
                }

                return reader.NodeType == XmlNodeType.Element && !reader.EOF;
            }
            catch (XmlException ex)
            {
                ProcessMessageDocument processMessage;
                processMessage = new ProcessMessageDocument()
                {
                    Code = ErrorCode1,
                    Description = ContactYourProviderMsg,
                    LongDescription = ex.Message,
                    SeverityAsString = "Error",
                };
                this.ErrorMessages.Add(processMessage);
                return false;
            }
        }

#endregion ReadToNextElement
        private Stack<string> _currentXPath = new Stack<string>();
        public INCDTLS2026 Consume(Stream streamToLoad, bool validateDataTypes = false)
        {
            INCDTLS2026 report = new INCDTLS2026();
            // Working Variable for if can still read from the xml stream
            bool reading;
            // Working Variables for current values
            string currentValue;
            DateTime currentDateTimeValue;
            bool? currentBooleanValue;
            decimal currentDecimalValue;
            double currentDoubleValue;
            float currentFloatValue;
            sbyte currentsByteValue;
            byte currentByteValue;
            short currentShortValue;
            ushort currentuShortValue;
            uint currentuIntValue;
            int currentIntValue;
            long currentLongValue;
            ulong currentuLongValue;
            this.ErrorMessages = new List<ProcessMessageDocument>();
            _currentXPath.Push("/tns:INCDTLS");
            if (streamToLoad == null || streamToLoad.Length == 0 || !streamToLoad.CanRead)
            {
                MissingElementError();
                return report;
            }

            streamToLoad.Position = 0;
            XmlReader reader;
            if (validateDataTypes)
                reader = XmlReader.Create(streamToLoad, GetValidatingReaderSettings());
            else
                reader = XmlReader.Create(streamToLoad, ReaderSettings);
            if (!MoveToContent(reader))
                return report;
            reading = !reader.EOF;
            if (ReadToNextElement(reader) && reader.LocalName == "INCDTLS" && reader.NamespaceURI == "http://www.sbr.gov.au/ato/incdtls")
            {
                _found = true;
                ReadToNextElement(reader);
            }
            else
            {
                MissingElementError();
                return report;
            }

#region Rp
            _currentXPath.Push("/tns:Rp");
            //3. use case
            if (ReadToNextElement(reader, true) && reader.LocalName == "Rp" && reader.Depth == _currentXPath.Count - 1)
            {
                report.RpCollectionExists = true;
                report.RpCollectionCount += 1;
                _found = true;
#region LodgmentPeriodStartD
                _currentXPath.Push("/tns:LodgmentPeriodStartD");
                //6. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "LodgmentPeriodStartD" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                            {
                                report.INCDTLS100 = currentDateTimeValue.Date;
                            }
                        }
                    }

                    _found = true;
                }

                _currentXPath.Pop();
#endregion LodgmentPeriodStartD
#region LodgmentPeriodEndD
                _currentXPath.Push("/tns:LodgmentPeriodEndD");
                //6. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "LodgmentPeriodEndD" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                            {
                                report.INCDTLS101 = currentDateTimeValue.Date;
                            }
                        }
                    }

                    _found = true;
                }

                _currentXPath.Pop();
#endregion LodgmentPeriodEndD
#region TaxFileNumberId
                _currentXPath.Push("/tns:TaxFileNumberId");
                //6. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "TaxFileNumberId" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.INCDTLS102 = currentValue;
                        }
                    }

                    _found = true;
                }

                _currentXPath.Pop();
#endregion TaxFileNumberId
#region Income
                _currentXPath.Push("/tns:IncomeCollection");
                // 4. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeCollection" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.Rp_IncomeCollection = new List<INCDTLS2026.Rp_Income>();
                    report.Rp_IncomeCollectionExists = true;
                    _found = true;
                    while (ReadToNextElement(reader, false) && reader.LocalName == "Income")
                    {
                        _found = true;
                        INCDTLS2026.Rp_Income income = new INCDTLS2026.Rp_Income();
                        report.Rp_IncomeCollection.Add(income);
                        report.Rp_IncomeCollectionCount += 1;
                        income.OccurrenceIndex = report.Rp_IncomeCollectionCount;
                        _currentXPath.Push("/tns:Income[" + report.Rp_IncomeCollectionCount + "]");
#region OrganisationNameDetails
                        _currentXPath.Push("/tns:OrganisationNameDetails");
                        //3. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "OrganisationNameDetails" && reader.Depth == _currentXPath.Count - 1)
                        {
                            income.Rp_Income_OrganisationNameDetailsCollectionExists = true;
                            income.Rp_Income_OrganisationNameDetailsCollectionCount += 1;
                            _found = true;
#region OrganisationalNameT
                            _currentXPath.Push("/tns:OrganisationalNameT");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "OrganisationalNameT" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        income.INCDTLS306 = currentValue;
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion OrganisationalNameT
#region AustralianBusinessNumberId
                            _currentXPath.Push("/tns:AustralianBusinessNumberId");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "AustralianBusinessNumberId" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        income.INCDTLS307 = currentValue;
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion AustralianBusinessNumberId
#region WithholdingPayerNumberId
                            _currentXPath.Push("/tns:WithholdingPayerNumberId");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "WithholdingPayerNumberId" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        income.INCDTLS308 = currentValue;
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion WithholdingPayerNumberId
#region SalaryOrWages
                            _currentXPath.Push("/tns:SalaryOrWages");
                            //3. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "SalaryOrWages" && reader.Depth == _currentXPath.Count - 1)
                            {
                                income.Rp_Income_OrganisationNameDetails_SalaryOrWagesCollectionExists = true;
                                income.Rp_Income_OrganisationNameDetails_SalaryOrWagesCollectionCount += 1;
                                _found = true;
#region RemunerationWorkingHolidayMakerIncomeI
                                _currentXPath.Push("/tns:RemunerationWorkingHolidayMakerIncomeI");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationWorkingHolidayMakerIncomeI" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            currentBooleanValue = ToBoolean(currentValue);
                                            if (currentBooleanValue != null)
                                            {
                                                income.INCDTLS310 = currentBooleanValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion RemunerationWorkingHolidayMakerIncomeI
#region TaxPayAsYouGoWithholdingCreditTaxWithheldSalaryOrWagesA
                                _currentXPath.Push("/tns:TaxPayAsYouGoWithholdingCreditTaxWithheldSalaryOrWagesA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "TaxPayAsYouGoWithholdingCreditTaxWithheldSalaryOrWagesA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                income.INCDTLS311 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion TaxPayAsYouGoWithholdingCreditTaxWithheldSalaryOrWagesA
#region A
                                _currentXPath.Push("/tns:A");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "A" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                income.INCDTLS312 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion A
#region AllowancesEarningsTipsDirectorsFeesA
                                _currentXPath.Push("/tns:AllowancesEarningsTipsDirectorsFeesA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "AllowancesEarningsTipsDirectorsFeesA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                income.INCDTLS313 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion AllowancesEarningsTipsDirectorsFeesA
#region TaxPayAsYouGoWithholdingCreditTaxWithheldAllowancesEarningsTipsDirectorsFeesA
                                _currentXPath.Push("/tns:TaxPayAsYouGoWithholdingCreditTaxWithheldAllowancesEarningsTipsDirectorsFeesA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "TaxPayAsYouGoWithholdingCreditTaxWithheldAllowancesEarningsTipsDirectorsFeesA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                income.INCDTLS339 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion TaxPayAsYouGoWithholdingCreditTaxWithheldAllowancesEarningsTipsDirectorsFeesA
#region FringeBenefitsReportableA
                                _currentXPath.Push("/tns:FringeBenefitsReportableA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "FringeBenefitsReportableA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                income.INCDTLS314 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion FringeBenefitsReportableA
#region FringeBenefitsTaxExemptionI
                                _currentXPath.Push("/tns:FringeBenefitsTaxExemptionI");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "FringeBenefitsTaxExemptionI" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            currentBooleanValue = ToBoolean(currentValue);
                                            if (currentBooleanValue != null)
                                            {
                                                income.INCDTLS315 = currentBooleanValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion FringeBenefitsTaxExemptionI
#region CommunityDevelopmentEmploymentProjectA
                                _currentXPath.Push("/tns:CommunityDevelopmentEmploymentProjectA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "CommunityDevelopmentEmploymentProjectA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                income.INCDTLS316 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion CommunityDevelopmentEmploymentProjectA
#region SuperannuationContributionEmployerReportableA
                                _currentXPath.Push("/tns:SuperannuationContributionEmployerReportableA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "SuperannuationContributionEmployerReportableA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                income.INCDTLS319 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion SuperannuationContributionEmployerReportableA
#region TaxPayAsYouGoWithholdingCreditTaxWithheldUnusedAnnualOrLongServiceLeavePaymentLumpSumAA
                                _currentXPath.Push("/tns:TaxPayAsYouGoWithholdingCreditTaxWithheldUnusedAnnualOrLongServiceLeavePaymentLumpSumAA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "TaxPayAsYouGoWithholdingCreditTaxWithheldUnusedAnnualOrLongServiceLeavePaymentLumpSumAA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                income.INCDTLS320 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion TaxPayAsYouGoWithholdingCreditTaxWithheldUnusedAnnualOrLongServiceLeavePaymentLumpSumAA
#region UnusedAnnualOrLongServiceLeavePaymentLumpSumAA
                                _currentXPath.Push("/tns:UnusedAnnualOrLongServiceLeavePaymentLumpSumAA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "UnusedAnnualOrLongServiceLeavePaymentLumpSumAA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                income.INCDTLS321 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion UnusedAnnualOrLongServiceLeavePaymentLumpSumAA
#region RemunerationUnusedAnnualOrLongServiceLeavePaymentLumpSumAC
                                _currentXPath.Push("/tns:RemunerationUnusedAnnualOrLongServiceLeavePaymentLumpSumAC");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationUnusedAnnualOrLongServiceLeavePaymentLumpSumAC" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            income.INCDTLS322 = currentValue;
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion RemunerationUnusedAnnualOrLongServiceLeavePaymentLumpSumAC
#region TaxPayAsYouGoWithholdingCreditTaxWithheldUnusedAnnualOrLongServiceLeavePaymentLumpSumBA
                                _currentXPath.Push("/tns:TaxPayAsYouGoWithholdingCreditTaxWithheldUnusedAnnualOrLongServiceLeavePaymentLumpSumBA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "TaxPayAsYouGoWithholdingCreditTaxWithheldUnusedAnnualOrLongServiceLeavePaymentLumpSumBA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                income.INCDTLS323 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion TaxPayAsYouGoWithholdingCreditTaxWithheldUnusedAnnualOrLongServiceLeavePaymentLumpSumBA
#region UnusedAnnualOrLongServiceLeavePaymentLumpSumBA
                                _currentXPath.Push("/tns:UnusedAnnualOrLongServiceLeavePaymentLumpSumBA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "UnusedAnnualOrLongServiceLeavePaymentLumpSumBA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                income.INCDTLS324 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion UnusedAnnualOrLongServiceLeavePaymentLumpSumBA
#region GenuineRedundancyPaymentLumpSumDA
                                _currentXPath.Push("/tns:GenuineRedundancyPaymentLumpSumDA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "GenuineRedundancyPaymentLumpSumDA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                income.INCDTLS325 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion GenuineRedundancyPaymentLumpSumDA
#region BackPaymentLumpSumEA
                                _currentXPath.Push("/tns:BackPaymentLumpSumEA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "BackPaymentLumpSumEA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                income.INCDTLS326 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion BackPaymentLumpSumEA
#region ExemptForeignEmployment
                                _currentXPath.Push("/tns:ExemptForeignEmployment");
                                //3. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "ExemptForeignEmployment" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    income.Rp_Income_OrganisationNameDetails_SalaryOrWages_ExemptForeignEmploymentCollectionExists = true;
                                    income.Rp_Income_OrganisationNameDetails_SalaryOrWages_ExemptForeignEmploymentCollectionCount += 1;
                                    _found = true;
#region InternationalDealingsIncomeA
                                    _currentXPath.Push("/tns:InternationalDealingsIncomeA");
                                    //6. use case
                                    if (ReadToNextElement(reader, true) && reader.LocalName == "InternationalDealingsIncomeA" && reader.Depth == _currentXPath.Count - 1)
                                    {
                                        if (!IsEmptyOrNilElement(reader))
                                        {
                                            ReadNext(reader);
                                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                            {
                                                currentValue = reader.Value;
                                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                                {
                                                    income.INCDTLS335 = currentDecimalValue;
                                                }
                                            }
                                        }

                                        _found = true;
                                    }

                                    _currentXPath.Pop();
#endregion InternationalDealingsIncomeA
#region TaxForeignWithholdingA
                                    _currentXPath.Push("/tns:TaxForeignWithholdingA");
                                    //6. use case
                                    if (ReadToNextElement(reader, false) && reader.LocalName == "TaxForeignWithholdingA" && reader.Depth == _currentXPath.Count - 1)
                                    {
                                        if (!IsEmptyOrNilElement(reader))
                                        {
                                            ReadNext(reader);
                                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                            {
                                                currentValue = reader.Value;
                                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                                {
                                                    income.INCDTLS336 = currentDecimalValue;
                                                }
                                            }
                                        }

                                        _found = true;
                                    }

                                    _currentXPath.Pop();
#endregion TaxForeignWithholdingA
#region AddressDetailsCountryC
                                    _currentXPath.Push("/tns:AddressDetailsCountryC");
                                    //6. use case
                                    if (ReadToNextElement(reader, true) && reader.LocalName == "AddressDetailsCountryC" && reader.Depth == _currentXPath.Count - 1)
                                    {
                                        if (!IsEmptyOrNilElement(reader))
                                        {
                                            ReadNext(reader);
                                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                            {
                                                currentValue = reader.Value;
                                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                                income.INCDTLS337 = currentValue;
                                            }
                                        }

                                        _found = true;
                                    }

                                    _currentXPath.Pop();
#endregion AddressDetailsCountryC
#region LumpSumArrearsPaymentA
                                    _currentXPath.Push("/tns:LumpSumArrearsPaymentA");
                                    //6. use case
                                    if (ReadToNextElement(reader, false) && reader.LocalName == "LumpSumArrearsPaymentA" && reader.Depth == _currentXPath.Count - 1)
                                    {
                                        if (!IsEmptyOrNilElement(reader))
                                        {
                                            ReadNext(reader);
                                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                            {
                                                currentValue = reader.Value;
                                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                                {
                                                    income.INCDTLS338 = currentDecimalValue;
                                                }
                                            }
                                        }

                                        _found = true;
                                    }

                                    _currentXPath.Pop();
#endregion LumpSumArrearsPaymentA
                                } // End of if ExemptForeignEmployment node exists

                                _currentXPath.Pop();
#endregion ExemptForeignEmployment
#region LumpSumArrears
                                _currentXPath.Push("/tns:LumpSumArrearsCollection");
                                // 4. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "LumpSumArrearsCollection" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    income.Rp_Income_OrganisationNameDetails_SalaryOrWages_LumpSumArrearsCollection = new List<INCDTLS2026.Rp_Income.Rp_Income_OrganisationNameDetails_SalaryOrWages_LumpSumArrears>();
                                    income.Rp_Income_OrganisationNameDetails_SalaryOrWages_LumpSumArrearsCollectionExists = true;
                                    _found = true;
                                    while (ReadToNextElement(reader, false) && reader.LocalName == "LumpSumArrears")
                                    {
                                        _found = true;
                                        INCDTLS2026.Rp_Income.Rp_Income_OrganisationNameDetails_SalaryOrWages_LumpSumArrears lumpSumArrears = new INCDTLS2026.Rp_Income.Rp_Income_OrganisationNameDetails_SalaryOrWages_LumpSumArrears();
                                        income.Rp_Income_OrganisationNameDetails_SalaryOrWages_LumpSumArrearsCollection.Add(lumpSumArrears);
                                        income.Rp_Income_OrganisationNameDetails_SalaryOrWages_LumpSumArrearsCollectionCount += 1;
                                        lumpSumArrears.OccurrenceIndex = income.Rp_Income_OrganisationNameDetails_SalaryOrWages_LumpSumArrearsCollectionCount;
                                        _currentXPath.Push("/tns:LumpSumArrears[" + income.Rp_Income_OrganisationNameDetails_SalaryOrWages_LumpSumArrearsCollectionCount + "]");
#region PaymentTypeC
                                        _currentXPath.Push("/tns:PaymentTypeC");
                                        //6. use case
                                        if (ReadToNextElement(reader, true) && reader.LocalName == "PaymentTypeC" && reader.Depth == _currentXPath.Count - 1)
                                        {
                                            if (!IsEmptyOrNilElement(reader))
                                            {
                                                ReadNext(reader);
                                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                                {
                                                    currentValue = reader.Value;
                                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                                    lumpSumArrears.INCDTLS151 = currentValue;
                                                }
                                            }

                                            _found = true;
                                        }

                                        _currentXPath.Pop();
#endregion PaymentTypeC
#region LumpSumArrearsPayment
                                        _currentXPath.Push("/tns:LumpSumArrearsPaymentCollection");
                                        // 4. use case
                                        if (ReadToNextElement(reader, true) && reader.LocalName == "LumpSumArrearsPaymentCollection" && reader.Depth == _currentXPath.Count - 1)
                                        {
                                            lumpSumArrears.Rp_Income_OrganisationNameDetails_SalaryOrWages_LumpSumArrears_LumpSumArrearsPaymentCollection = new List<INCDTLS2026.Rp_Income.Rp_Income_OrganisationNameDetails_SalaryOrWages_LumpSumArrears.Rp_Income_OrganisationNameDetails_SalaryOrWages_LumpSumArrears_LumpSumArrearsPayment>();
                                            lumpSumArrears.Rp_Income_OrganisationNameDetails_SalaryOrWages_LumpSumArrears_LumpSumArrearsPaymentCollectionExists = true;
                                            _found = true;
                                            while (ReadToNextElement(reader, true) && reader.LocalName == "LumpSumArrearsPayment")
                                            {
                                                _found = true;
                                                INCDTLS2026.Rp_Income.Rp_Income_OrganisationNameDetails_SalaryOrWages_LumpSumArrears.Rp_Income_OrganisationNameDetails_SalaryOrWages_LumpSumArrears_LumpSumArrearsPayment lumpSumArrearsPayment = new INCDTLS2026.Rp_Income.Rp_Income_OrganisationNameDetails_SalaryOrWages_LumpSumArrears.Rp_Income_OrganisationNameDetails_SalaryOrWages_LumpSumArrears_LumpSumArrearsPayment();
                                                lumpSumArrears.Rp_Income_OrganisationNameDetails_SalaryOrWages_LumpSumArrears_LumpSumArrearsPaymentCollection.Add(lumpSumArrearsPayment);
                                                lumpSumArrears.Rp_Income_OrganisationNameDetails_SalaryOrWages_LumpSumArrears_LumpSumArrearsPaymentCollectionCount += 1;
                                                lumpSumArrearsPayment.OccurrenceIndex = lumpSumArrears.Rp_Income_OrganisationNameDetails_SalaryOrWages_LumpSumArrears_LumpSumArrearsPaymentCollectionCount;
                                                _currentXPath.Push("/tns:LumpSumArrearsPayment[" + lumpSumArrears.Rp_Income_OrganisationNameDetails_SalaryOrWages_LumpSumArrears_LumpSumArrearsPaymentCollectionCount + "]");
#region TargetFinancialY
                                                _currentXPath.Push("/tns:TargetFinancialY");
                                                //6. use case
                                                if (ReadToNextElement(reader, true) && reader.LocalName == "TargetFinancialY" && reader.Depth == _currentXPath.Count - 1)
                                                {
                                                    if (!IsEmptyOrNilElement(reader))
                                                    {
                                                        ReadNext(reader);
                                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                                        {
                                                            currentValue = reader.Value;
                                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                                            if (Int32.TryParse(currentValue, out currentIntValue))
                                                            {
                                                                lumpSumArrearsPayment.INCDTLS153 = currentIntValue;
                                                            }
                                                        }
                                                    }

                                                    _found = true;
                                                }

                                                _currentXPath.Pop();
#endregion TargetFinancialY
#region A
                                                _currentXPath.Push("/tns:A");
                                                //6. use case
                                                if (ReadToNextElement(reader, true) && reader.LocalName == "A" && reader.Depth == _currentXPath.Count - 1)
                                                {
                                                    if (!IsEmptyOrNilElement(reader))
                                                    {
                                                        ReadNext(reader);
                                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                                        {
                                                            currentValue = reader.Value;
                                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                                            {
                                                                lumpSumArrearsPayment.INCDTLS154 = currentDecimalValue;
                                                            }
                                                        }
                                                    }

                                                    _found = true;
                                                }

                                                _currentXPath.Pop();
#endregion A
                                                _currentXPath.Pop();
                                            } // End of while loop for LumpSumArrearsPayment
                                        } // End of if LumpSumArrearsPaymentCollection node exists

                                        _currentXPath.Pop();
#endregion LumpSumArrearsPayment
                                        _currentXPath.Pop();
                                    } // End of while loop for LumpSumArrears
                                } // End of if LumpSumArrearsCollection node exists

                                _currentXPath.Pop();
#endregion LumpSumArrears
                            } // End of if SalaryOrWages node exists

                            _currentXPath.Pop();
#endregion SalaryOrWages
#region ForeignEmployment
                            _currentXPath.Push("/tns:ForeignEmployment");
                            //3. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "ForeignEmployment" && reader.Depth == _currentXPath.Count - 1)
                            {
                                income.Rp_Income_OrganisationNameDetails_ForeignEmploymentCollectionExists = true;
                                income.Rp_Income_OrganisationNameDetails_ForeignEmploymentCollectionCount += 1;
                                _found = true;
#region TaxPayAsYouGoWithholdingCreditTaxWithheldForeignEmploymentIncomeA
                                _currentXPath.Push("/tns:TaxPayAsYouGoWithholdingCreditTaxWithheldForeignEmploymentIncomeA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "TaxPayAsYouGoWithholdingCreditTaxWithheldForeignEmploymentIncomeA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                income.INCDTLS167 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion TaxPayAsYouGoWithholdingCreditTaxWithheldForeignEmploymentIncomeA
#region InternationalDealingsForeignIncomeGrossA
                                _currentXPath.Push("/tns:InternationalDealingsForeignIncomeGrossA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsForeignIncomeGrossA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                income.INCDTLS168 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion InternationalDealingsForeignIncomeGrossA
#region InternationalDealingsTaxOffsetA
                                _currentXPath.Push("/tns:InternationalDealingsTaxOffsetA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsTaxOffsetA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                income.INCDTLS169 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion InternationalDealingsTaxOffsetA
#region FringeBenefitsReportableA
                                _currentXPath.Push("/tns:FringeBenefitsReportableA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "FringeBenefitsReportableA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                income.INCDTLS170 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion FringeBenefitsReportableA
#region FringeBenefitsTaxExemptionI
                                _currentXPath.Push("/tns:FringeBenefitsTaxExemptionI");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "FringeBenefitsTaxExemptionI" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            currentBooleanValue = ToBoolean(currentValue);
                                            if (currentBooleanValue != null)
                                            {
                                                income.INCDTLS180 = currentBooleanValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion FringeBenefitsTaxExemptionI
#region SuperannuationContributionEmployerReportableA
                                _currentXPath.Push("/tns:SuperannuationContributionEmployerReportableA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "SuperannuationContributionEmployerReportableA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                income.INCDTLS171 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion SuperannuationContributionEmployerReportableA
#region UnusedAnnualOrLongServiceLeavePaymentLumpSumAA
                                _currentXPath.Push("/tns:UnusedAnnualOrLongServiceLeavePaymentLumpSumAA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "UnusedAnnualOrLongServiceLeavePaymentLumpSumAA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                income.INCDTLS174 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion UnusedAnnualOrLongServiceLeavePaymentLumpSumAA
#region RemunerationUnusedAnnualOrLongServiceLeavePaymentLumpSumAC
                                _currentXPath.Push("/tns:RemunerationUnusedAnnualOrLongServiceLeavePaymentLumpSumAC");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationUnusedAnnualOrLongServiceLeavePaymentLumpSumAC" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            income.INCDTLS175 = currentValue;
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion RemunerationUnusedAnnualOrLongServiceLeavePaymentLumpSumAC
#region GenuineRedundancyPaymentLumpSumDA
                                _currentXPath.Push("/tns:GenuineRedundancyPaymentLumpSumDA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "GenuineRedundancyPaymentLumpSumDA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                income.INCDTLS176 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion GenuineRedundancyPaymentLumpSumDA
#region BackPaymentLumpSumEA
                                _currentXPath.Push("/tns:BackPaymentLumpSumEA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "BackPaymentLumpSumEA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                income.INCDTLS177 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion BackPaymentLumpSumEA
#region InternationalDealingsIncomePaymentSummaryA
                                _currentXPath.Push("/tns:InternationalDealingsIncomePaymentSummaryA");
                                //6. use case
                                if (ReadToNextElement(reader, true) && reader.LocalName == "InternationalDealingsIncomePaymentSummaryA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                income.INCDTLS178 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion InternationalDealingsIncomePaymentSummaryA
#region FEIPSLumpSumArrearsPayment
                                _currentXPath.Push("/tns:FEIPSLumpSumArrearsPaymentCollection");
                                // 4. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "FEIPSLumpSumArrearsPaymentCollection" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    income.Rp_Income_OrganisationNameDetails_ForeignEmployment_FEIPSLumpSumArrearsPaymentCollection = new List<INCDTLS2026.Rp_Income.Rp_Income_OrganisationNameDetails_ForeignEmployment_FEIPSLumpSumArrearsPayment>();
                                    income.Rp_Income_OrganisationNameDetails_ForeignEmployment_FEIPSLumpSumArrearsPaymentCollectionExists = true;
                                    _found = true;
                                    while (ReadToNextElement(reader, false) && reader.LocalName == "FEIPSLumpSumArrearsPayment")
                                    {
                                        _found = true;
                                        INCDTLS2026.Rp_Income.Rp_Income_OrganisationNameDetails_ForeignEmployment_FEIPSLumpSumArrearsPayment feipsLumpSumArrearsPayment = new INCDTLS2026.Rp_Income.Rp_Income_OrganisationNameDetails_ForeignEmployment_FEIPSLumpSumArrearsPayment();
                                        income.Rp_Income_OrganisationNameDetails_ForeignEmployment_FEIPSLumpSumArrearsPaymentCollection.Add(feipsLumpSumArrearsPayment);
                                        income.Rp_Income_OrganisationNameDetails_ForeignEmployment_FEIPSLumpSumArrearsPaymentCollectionCount += 1;
                                        feipsLumpSumArrearsPayment.OccurrenceIndex = income.Rp_Income_OrganisationNameDetails_ForeignEmployment_FEIPSLumpSumArrearsPaymentCollectionCount;
                                        _currentXPath.Push("/tns:FEIPSLumpSumArrearsPayment[" + income.Rp_Income_OrganisationNameDetails_ForeignEmployment_FEIPSLumpSumArrearsPaymentCollectionCount + "]");
#region TargetFinancialY
                                        _currentXPath.Push("/tns:TargetFinancialY");
                                        //6. use case
                                        if (ReadToNextElement(reader, true) && reader.LocalName == "TargetFinancialY" && reader.Depth == _currentXPath.Count - 1)
                                        {
                                            if (!IsEmptyOrNilElement(reader))
                                            {
                                                ReadNext(reader);
                                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                                {
                                                    currentValue = reader.Value;
                                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                                    if (Int32.TryParse(currentValue, out currentIntValue))
                                                    {
                                                        feipsLumpSumArrearsPayment.INCDTLS523 = currentIntValue;
                                                    }
                                                }
                                            }

                                            _found = true;
                                        }

                                        _currentXPath.Pop();
#endregion TargetFinancialY
#region A
                                        _currentXPath.Push("/tns:A");
                                        //6. use case
                                        if (ReadToNextElement(reader, true) && reader.LocalName == "A" && reader.Depth == _currentXPath.Count - 1)
                                        {
                                            if (!IsEmptyOrNilElement(reader))
                                            {
                                                ReadNext(reader);
                                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                                {
                                                    currentValue = reader.Value;
                                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                                    {
                                                        feipsLumpSumArrearsPayment.INCDTLS524 = currentDecimalValue;
                                                    }
                                                }
                                            }

                                            _found = true;
                                        }

                                        _currentXPath.Pop();
#endregion A
                                        _currentXPath.Pop();
                                    } // End of while loop for FEIPSLumpSumArrearsPayment
                                } // End of if FEIPSLumpSumArrearsPaymentCollection node exists

                                _currentXPath.Pop();
#endregion FEIPSLumpSumArrearsPayment
                            } // End of if ForeignEmployment node exists

                            _currentXPath.Pop();
#endregion ForeignEmployment
#region EmploymentTerminationPayment
                            _currentXPath.Push("/tns:EmploymentTerminationPayment");
                            //3. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "EmploymentTerminationPayment" && reader.Depth == _currentXPath.Count - 1)
                            {
                                income.Rp_Income_OrganisationNameDetails_EmploymentTerminationPaymentCollectionExists = true;
                                income.Rp_Income_OrganisationNameDetails_EmploymentTerminationPaymentCollectionCount += 1;
                                _found = true;
#region AddressDetailsCountryC
                                _currentXPath.Push("/tns:AddressDetailsCountryC");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "AddressDetailsCountryC" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            income.INCDTLS136 = currentValue;
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion AddressDetailsCountryC
#region D
                                _currentXPath.Push("/tns:D");
                                //6. use case
                                if (ReadToNextElement(reader, true) && reader.LocalName == "D" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                                            {
                                                income.INCDTLS137 = currentDateTimeValue.Date;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion D
#region TaxPayAsYouGoWithholdingTypeC
                                _currentXPath.Push("/tns:TaxPayAsYouGoWithholdingTypeC");
                                //6. use case
                                if (ReadToNextElement(reader, true) && reader.LocalName == "TaxPayAsYouGoWithholdingTypeC" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            income.INCDTLS138 = currentValue;
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion TaxPayAsYouGoWithholdingTypeC
#region TaxPayAsYouGoWithholdingCreditTaxWithheldEmploymentTerminationPaymentA
                                _currentXPath.Push("/tns:TaxPayAsYouGoWithholdingCreditTaxWithheldEmploymentTerminationPaymentA");
                                //6. use case
                                if (ReadToNextElement(reader, true) && reader.LocalName == "TaxPayAsYouGoWithholdingCreditTaxWithheldEmploymentTerminationPaymentA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                income.INCDTLS139 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion TaxPayAsYouGoWithholdingCreditTaxWithheldEmploymentTerminationPaymentA
#region TaxableA
                                _currentXPath.Push("/tns:TaxableA");
                                //6. use case
                                if (ReadToNextElement(reader, true) && reader.LocalName == "TaxableA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                income.INCDTLS140 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion TaxableA
#region TaxFreeA
                                _currentXPath.Push("/tns:TaxFreeA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "TaxFreeA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                income.INCDTLS141 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion TaxFreeA
                            } // End of if EmploymentTerminationPayment node exists

                            _currentXPath.Pop();
#endregion EmploymentTerminationPayment
#region AnnuitiesAndSuperannuation
                            _currentXPath.Push("/tns:AnnuitiesAndSuperannuation");
                            //3. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "AnnuitiesAndSuperannuation" && reader.Depth == _currentXPath.Count - 1)
                            {
                                income.Rp_Income_OrganisationNameDetails_AnnuitiesAndSuperannuationCollectionExists = true;
                                income.Rp_Income_OrganisationNameDetails_AnnuitiesAndSuperannuationCollectionCount += 1;
                                _found = true;
#region ReversionaryIncomeStreamI
                                _currentXPath.Push("/tns:ReversionaryIncomeStreamI");
                                //6. use case
                                if (ReadToNextElement(reader, true) && reader.LocalName == "ReversionaryIncomeStreamI" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            currentBooleanValue = ToBoolean(currentValue);
                                            if (currentBooleanValue != null)
                                            {
                                                income.INCDTLS109 = currentBooleanValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion ReversionaryIncomeStreamI
#region PaymentPeriodStartD
                                _currentXPath.Push("/tns:PaymentPeriodStartD");
                                //6. use case
                                if (ReadToNextElement(reader, true) && reader.LocalName == "PaymentPeriodStartD" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                                            {
                                                income.INCDTLS110 = currentDateTimeValue.Date;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion PaymentPeriodStartD
#region PaymentPeriodEndD
                                _currentXPath.Push("/tns:PaymentPeriodEndD");
                                //6. use case
                                if (ReadToNextElement(reader, true) && reader.LocalName == "PaymentPeriodEndD" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                                            {
                                                income.INCDTLS111 = currentDateTimeValue.Date;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion PaymentPeriodEndD
#region TaxPayAsYouGoWithholdingCreditTaxWithheldAustralianAnnuitiesAndSuperannuationIncomeStreamA
                                _currentXPath.Push("/tns:TaxPayAsYouGoWithholdingCreditTaxWithheldAustralianAnnuitiesAndSuperannuationIncomeStreamA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "TaxPayAsYouGoWithholdingCreditTaxWithheldAustralianAnnuitiesAndSuperannuationIncomeStreamA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                income.INCDTLS112 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion TaxPayAsYouGoWithholdingCreditTaxWithheldAustralianAnnuitiesAndSuperannuationIncomeStreamA
#region TaxableComponentTaxedElementA
                                _currentXPath.Push("/tns:TaxableComponentTaxedElementA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "TaxableComponentTaxedElementA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                income.INCDTLS113 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion TaxableComponentTaxedElementA
#region TaxableComponentUntaxedElementA
                                _currentXPath.Push("/tns:TaxableComponentUntaxedElementA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "TaxableComponentUntaxedElementA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                income.INCDTLS114 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion TaxableComponentUntaxedElementA
#region TaxTaxFreeComponentA
                                _currentXPath.Push("/tns:TaxTaxFreeComponentA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "TaxTaxFreeComponentA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                income.INCDTLS115 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion TaxTaxFreeComponentA
#region TaxOffsetClaimSuperannuationIncomeStreamA
                                _currentXPath.Push("/tns:TaxOffsetClaimSuperannuationIncomeStreamA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "TaxOffsetClaimSuperannuationIncomeStreamA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                income.INCDTLS116 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion TaxOffsetClaimSuperannuationIncomeStreamA
#region LumpSumArrearsTaxableComponentTaxedElementA
                                _currentXPath.Push("/tns:LumpSumArrearsTaxableComponentTaxedElementA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "LumpSumArrearsTaxableComponentTaxedElementA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                income.INCDTLS117 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion LumpSumArrearsTaxableComponentTaxedElementA
#region LumpSumArrearsTaxableComponentUntaxedElementA
                                _currentXPath.Push("/tns:LumpSumArrearsTaxableComponentUntaxedElementA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "LumpSumArrearsTaxableComponentUntaxedElementA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                income.INCDTLS118 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion LumpSumArrearsTaxableComponentUntaxedElementA
#region LumpSumArrearsTaxFreeA
                                _currentXPath.Push("/tns:LumpSumArrearsTaxFreeA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "LumpSumArrearsTaxFreeA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                income.INCDTLS119 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion LumpSumArrearsTaxFreeA
#region SISLumpSumArrearsPayment
                                _currentXPath.Push("/tns:SISLumpSumArrearsPaymentCollection");
                                // 4. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "SISLumpSumArrearsPaymentCollection" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    income.Rp_Income_OrganisationNameDetails_AnnuitiesAndSuperannuation_SISLumpSumArrearsPaymentCollection = new List<INCDTLS2026.Rp_Income.Rp_Income_OrganisationNameDetails_AnnuitiesAndSuperannuation_SISLumpSumArrearsPayment>();
                                    income.Rp_Income_OrganisationNameDetails_AnnuitiesAndSuperannuation_SISLumpSumArrearsPaymentCollectionExists = true;
                                    _found = true;
                                    while (ReadToNextElement(reader, false) && reader.LocalName == "SISLumpSumArrearsPayment")
                                    {
                                        _found = true;
                                        INCDTLS2026.Rp_Income.Rp_Income_OrganisationNameDetails_AnnuitiesAndSuperannuation_SISLumpSumArrearsPayment sisLumpSumArrearsPayment = new INCDTLS2026.Rp_Income.Rp_Income_OrganisationNameDetails_AnnuitiesAndSuperannuation_SISLumpSumArrearsPayment();
                                        income.Rp_Income_OrganisationNameDetails_AnnuitiesAndSuperannuation_SISLumpSumArrearsPaymentCollection.Add(sisLumpSumArrearsPayment);
                                        income.Rp_Income_OrganisationNameDetails_AnnuitiesAndSuperannuation_SISLumpSumArrearsPaymentCollectionCount += 1;
                                        sisLumpSumArrearsPayment.OccurrenceIndex = income.Rp_Income_OrganisationNameDetails_AnnuitiesAndSuperannuation_SISLumpSumArrearsPaymentCollectionCount;
                                        _currentXPath.Push("/tns:SISLumpSumArrearsPayment[" + income.Rp_Income_OrganisationNameDetails_AnnuitiesAndSuperannuation_SISLumpSumArrearsPaymentCollectionCount + "]");
#region TargetFinancialY
                                        _currentXPath.Push("/tns:TargetFinancialY");
                                        //6. use case
                                        if (ReadToNextElement(reader, true) && reader.LocalName == "TargetFinancialY" && reader.Depth == _currentXPath.Count - 1)
                                        {
                                            if (!IsEmptyOrNilElement(reader))
                                            {
                                                ReadNext(reader);
                                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                                {
                                                    currentValue = reader.Value;
                                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                                    if (Int32.TryParse(currentValue, out currentIntValue))
                                                    {
                                                        sisLumpSumArrearsPayment.INCDTLS526 = currentIntValue;
                                                    }
                                                }
                                            }

                                            _found = true;
                                        }

                                        _currentXPath.Pop();
#endregion TargetFinancialY
#region A
                                        _currentXPath.Push("/tns:A");
                                        //6. use case
                                        if (ReadToNextElement(reader, true) && reader.LocalName == "A" && reader.Depth == _currentXPath.Count - 1)
                                        {
                                            if (!IsEmptyOrNilElement(reader))
                                            {
                                                ReadNext(reader);
                                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                                {
                                                    currentValue = reader.Value;
                                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                                    {
                                                        sisLumpSumArrearsPayment.INCDTLS527 = currentDecimalValue;
                                                    }
                                                }
                                            }

                                            _found = true;
                                        }

                                        _currentXPath.Pop();
#endregion A
                                        _currentXPath.Pop();
                                    } // End of while loop for SISLumpSumArrearsPayment
                                } // End of if SISLumpSumArrearsPaymentCollection node exists

                                _currentXPath.Pop();
#endregion SISLumpSumArrearsPayment
                            } // End of if AnnuitiesAndSuperannuation node exists

                            _currentXPath.Pop();
#endregion AnnuitiesAndSuperannuation
#region AustralianAnnuities
                            _currentXPath.Push("/tns:AustralianAnnuities");
                            //3. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "AustralianAnnuities" && reader.Depth == _currentXPath.Count - 1)
                            {
                                income.Rp_Income_OrganisationNameDetails_AustralianAnnuitiesCollectionExists = true;
                                income.Rp_Income_OrganisationNameDetails_AustralianAnnuitiesCollectionCount += 1;
                                _found = true;
#region TaxPayAsYouGoWithholdingCreditTaxWithheldAustralianAnnuitiesAndSuperannuationIncomeStreamA
                                _currentXPath.Push("/tns:TaxPayAsYouGoWithholdingCreditTaxWithheldAustralianAnnuitiesAndSuperannuationIncomeStreamA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "TaxPayAsYouGoWithholdingCreditTaxWithheldAustralianAnnuitiesAndSuperannuationIncomeStreamA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                income.INCDTLS146 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion TaxPayAsYouGoWithholdingCreditTaxWithheldAustralianAnnuitiesAndSuperannuationIncomeStreamA
#region InsuranceLifePremiumDirectGrossA
                                _currentXPath.Push("/tns:InsuranceLifePremiumDirectGrossA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "InsuranceLifePremiumDirectGrossA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                income.INCDTLS147 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion InsuranceLifePremiumDirectGrossA
#region TaxDeductionPensionOrAnnuityPurchasePriceUndeductedA
                                _currentXPath.Push("/tns:TaxDeductionPensionOrAnnuityPurchasePriceUndeductedA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "TaxDeductionPensionOrAnnuityPurchasePriceUndeductedA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                income.INCDTLS148 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion TaxDeductionPensionOrAnnuityPurchasePriceUndeductedA
                            } // End of if AustralianAnnuities node exists

                            _currentXPath.Pop();
#endregion AustralianAnnuities
#region SuperannuationLumpSumPayment
                            _currentXPath.Push("/tns:SuperannuationLumpSumPayment");
                            //3. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "SuperannuationLumpSumPayment" && reader.Depth == _currentXPath.Count - 1)
                            {
                                income.Rp_Income_OrganisationNameDetails_SuperannuationLumpSumPaymentCollectionExists = true;
                                income.Rp_Income_OrganisationNameDetails_SuperannuationLumpSumPaymentCollectionCount += 1;
                                _found = true;
#region D
                                _currentXPath.Push("/tns:D");
                                //6. use case
                                if (ReadToNextElement(reader, true) && reader.LocalName == "D" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                                            {
                                                income.INCDTLS328 = currentDateTimeValue.Date;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion D
#region TaxPayAsYouGoWithholdingCreditTaxWithheldSuperannuationLumpSumPaymentA
                                _currentXPath.Push("/tns:TaxPayAsYouGoWithholdingCreditTaxWithheldSuperannuationLumpSumPaymentA");
                                //6. use case
                                if (ReadToNextElement(reader, true) && reader.LocalName == "TaxPayAsYouGoWithholdingCreditTaxWithheldSuperannuationLumpSumPaymentA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                income.INCDTLS329 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion TaxPayAsYouGoWithholdingCreditTaxWithheldSuperannuationLumpSumPaymentA
#region LumpSumPaymentTaxableComponentTaxedElementA
                                _currentXPath.Push("/tns:LumpSumPaymentTaxableComponentTaxedElementA");
                                //6. use case
                                if (ReadToNextElement(reader, true) && reader.LocalName == "LumpSumPaymentTaxableComponentTaxedElementA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                income.INCDTLS330 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion LumpSumPaymentTaxableComponentTaxedElementA
#region LumpSumPaymentTaxableComponentUntaxedElementA
                                _currentXPath.Push("/tns:LumpSumPaymentTaxableComponentUntaxedElementA");
                                //6. use case
                                if (ReadToNextElement(reader, true) && reader.LocalName == "LumpSumPaymentTaxableComponentUntaxedElementA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                income.INCDTLS331 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion LumpSumPaymentTaxableComponentUntaxedElementA
#region LumpSumPaymentTaxFreeA
                                _currentXPath.Push("/tns:LumpSumPaymentTaxFreeA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "LumpSumPaymentTaxFreeA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                income.INCDTLS332 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion LumpSumPaymentTaxFreeA
#region SuperannuationBenefitDeathBenefitI
                                _currentXPath.Push("/tns:SuperannuationBenefitDeathBenefitI");
                                //6. use case
                                if (ReadToNextElement(reader, true) && reader.LocalName == "SuperannuationBenefitDeathBenefitI" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            currentBooleanValue = ToBoolean(currentValue);
                                            if (currentBooleanValue != null)
                                            {
                                                income.INCDTLS333 = currentBooleanValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion SuperannuationBenefitDeathBenefitI
                            } // End of if SuperannuationLumpSumPayment node exists

                            _currentXPath.Pop();
#endregion SuperannuationLumpSumPayment
#region AttributedPersonalServicesIncome
                            _currentXPath.Push("/tns:AttributedPersonalServicesIncome");
                            //3. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "AttributedPersonalServicesIncome" && reader.Depth == _currentXPath.Count - 1)
                            {
                                income.Rp_Income_OrganisationNameDetails_AttributedPersonalServicesIncomeCollectionExists = true;
                                income.Rp_Income_OrganisationNameDetails_AttributedPersonalServicesIncomeCollectionCount += 1;
                                _found = true;
#region TaxPayAsYouGoWithholdingCreditTaxWithheldPersonalServicesIncomeA
                                _currentXPath.Push("/tns:TaxPayAsYouGoWithholdingCreditTaxWithheldPersonalServicesIncomeA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "TaxPayAsYouGoWithholdingCreditTaxWithheldPersonalServicesIncomeA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                income.INCDTLS341 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion TaxPayAsYouGoWithholdingCreditTaxWithheldPersonalServicesIncomeA
#region TotalA
                                _currentXPath.Push("/tns:TotalA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "TotalA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                income.INCDTLS342 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion TotalA
#region SuperannuationContributionEmployerReportableA
                                _currentXPath.Push("/tns:SuperannuationContributionEmployerReportableA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "SuperannuationContributionEmployerReportableA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                income.INCDTLS343 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion SuperannuationContributionEmployerReportableA
                            } // End of if AttributedPersonalServicesIncome node exists

                            _currentXPath.Pop();
#endregion AttributedPersonalServicesIncome
                        } // End of if OrganisationNameDetails node exists

                        _currentXPath.Pop();
#endregion OrganisationNameDetails
#region GovernmentBenefit
                        _currentXPath.Push("/tns:GovernmentBenefitCollection");
                        // 4. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "GovernmentBenefitCollection" && reader.Depth == _currentXPath.Count - 1)
                        {
                            income.Rp_Income_GovernmentBenefitCollection = new List<INCDTLS2026.Rp_Income.Rp_Income_GovernmentBenefit>();
                            income.Rp_Income_GovernmentBenefitCollectionExists = true;
                            _found = true;
                            while (ReadToNextElement(reader, false) && reader.LocalName == "GovernmentBenefit")
                            {
                                _found = true;
                                INCDTLS2026.Rp_Income.Rp_Income_GovernmentBenefit governmentBenefit = new INCDTLS2026.Rp_Income.Rp_Income_GovernmentBenefit();
                                income.Rp_Income_GovernmentBenefitCollection.Add(governmentBenefit);
                                income.Rp_Income_GovernmentBenefitCollectionCount += 1;
                                governmentBenefit.OccurrenceIndex = income.Rp_Income_GovernmentBenefitCollectionCount;
                                _currentXPath.Push("/tns:GovernmentBenefit[" + income.Rp_Income_GovernmentBenefitCollectionCount + "]");
#region PaymentTypeC
                                _currentXPath.Push("/tns:PaymentTypeC");
                                //6. use case
                                if (ReadToNextElement(reader, true) && reader.LocalName == "PaymentTypeC" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            governmentBenefit.INCDTLS126 = currentValue;
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion PaymentTypeC
#region PaymentTypeDe
                                _currentXPath.Push("/tns:PaymentTypeDe");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "PaymentTypeDe" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            governmentBenefit.INCDTLS127 = currentValue;
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion PaymentTypeDe
#region PaymentTaxableA
                                _currentXPath.Push("/tns:PaymentTaxableA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "PaymentTaxableA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                governmentBenefit.INCDTLS128 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion PaymentTaxableA
#region TaxPayAsYouGoWithholdingCreditTaxWithheldAustralianGovernmentPensionsAndAllowancesA
                                _currentXPath.Push("/tns:TaxPayAsYouGoWithholdingCreditTaxWithheldAustralianGovernmentPensionsAndAllowancesA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "TaxPayAsYouGoWithholdingCreditTaxWithheldAustralianGovernmentPensionsAndAllowancesA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                governmentBenefit.INCDTLS129 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion TaxPayAsYouGoWithholdingCreditTaxWithheldAustralianGovernmentPensionsAndAllowancesA
#region PaymentAdjustmentReasonC
                                _currentXPath.Push("/tns:PaymentAdjustmentReasonC");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "PaymentAdjustmentReasonC" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            governmentBenefit.INCDTLS131 = currentValue;
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion PaymentAdjustmentReasonC
#region PaymentAdjustmentReasonDe
                                _currentXPath.Push("/tns:PaymentAdjustmentReasonDe");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "PaymentAdjustmentReasonDe" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            governmentBenefit.INCDTLS132 = currentValue;
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion PaymentAdjustmentReasonDe
                                _currentXPath.Pop();
                            } // End of while loop for GovernmentBenefit
                        } // End of if GovernmentBenefitCollection node exists

                        _currentXPath.Pop();
#endregion GovernmentBenefit
                        _currentXPath.Pop();
                    } // End of while loop for Income
                } // End of if IncomeCollection node exists

                _currentXPath.Pop();
#endregion Income
#region EmployeeShareScheme
                _currentXPath.Push("/tns:EmployeeShareSchemeCollection");
                // 4. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "EmployeeShareSchemeCollection" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.Rp_EmployeeShareSchemeCollection = new List<INCDTLS2026.Rp_EmployeeShareScheme>();
                    report.Rp_EmployeeShareSchemeCollectionExists = true;
                    _found = true;
                    while (ReadToNextElement(reader, false) && reader.LocalName == "EmployeeShareScheme")
                    {
                        _found = true;
                        INCDTLS2026.Rp_EmployeeShareScheme employeeShareScheme = new INCDTLS2026.Rp_EmployeeShareScheme();
                        report.Rp_EmployeeShareSchemeCollection.Add(employeeShareScheme);
                        report.Rp_EmployeeShareSchemeCollectionCount += 1;
                        employeeShareScheme.OccurrenceIndex = report.Rp_EmployeeShareSchemeCollectionCount;
                        _currentXPath.Push("/tns:EmployeeShareScheme[" + report.Rp_EmployeeShareSchemeCollectionCount + "]");
#region OrganisationNameDetailsOrganisationalNameT
                        _currentXPath.Push("/tns:OrganisationNameDetailsOrganisationalNameT");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "OrganisationNameDetailsOrganisationalNameT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    employeeShareScheme.INCDTLS495 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion OrganisationNameDetailsOrganisationalNameT
#region AustralianBusinessNumberId
                        _currentXPath.Push("/tns:AustralianBusinessNumberId");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "AustralianBusinessNumberId" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    employeeShareScheme.INCDTLS496 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion AustralianBusinessNumberId
#region IncomeTaxUpfrontReducibleDiscountA
                        _currentXPath.Push("/tns:IncomeTaxUpfrontReducibleDiscountA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxUpfrontReducibleDiscountA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        employeeShareScheme.INCDTLS501 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion IncomeTaxUpfrontReducibleDiscountA
#region IncomeTaxUpfrontNonReducibleDiscountA
                        _currentXPath.Push("/tns:IncomeTaxUpfrontNonReducibleDiscountA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxUpfrontNonReducibleDiscountA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        employeeShareScheme.INCDTLS502 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion IncomeTaxUpfrontNonReducibleDiscountA
#region IncomeDeferralDiscountA
                        _currentXPath.Push("/tns:IncomeDeferralDiscountA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeDeferralDiscountA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        employeeShareScheme.INCDTLS503 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion IncomeDeferralDiscountA
#region IncomeTaxPayAsYouGoWithholdingCreditTaxWithheldEmployeeShareSchemeDiscountTFNNotQuotedA
                        _currentXPath.Push("/tns:IncomeTaxPayAsYouGoWithholdingCreditTaxWithheldEmployeeShareSchemeDiscountTFNNotQuotedA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxPayAsYouGoWithholdingCreditTaxWithheldEmployeeShareSchemeDiscountTFNNotQuotedA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        employeeShareScheme.INCDTLS505 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion IncomeTaxPayAsYouGoWithholdingCreditTaxWithheldEmployeeShareSchemeDiscountTFNNotQuotedA
#region IncomeForeignSourceDiscountA
                        _currentXPath.Push("/tns:IncomeForeignSourceDiscountA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeForeignSourceDiscountA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        employeeShareScheme.INCDTLS506 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion IncomeForeignSourceDiscountA
#region IncomeInternationalDealingsTaxOffsetA
                        _currentXPath.Push("/tns:IncomeInternationalDealingsTaxOffsetA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeInternationalDealingsTaxOffsetA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        employeeShareScheme.INCDTLS507 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion IncomeInternationalDealingsTaxOffsetA
                        _currentXPath.Pop();
                    } // End of while loop for EmployeeShareScheme
                } // End of if EmployeeShareSchemeCollection node exists

                _currentXPath.Pop();
#endregion EmployeeShareScheme
#region Interest
                _currentXPath.Push("/tns:InterestCollection");
                // 4. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "InterestCollection" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.Rp_InterestCollection = new List<INCDTLS2026.Rp_Interest>();
                    report.Rp_InterestCollectionExists = true;
                    _found = true;
                    while (ReadToNextElement(reader, false) && reader.LocalName == "Interest")
                    {
                        _found = true;
                        INCDTLS2026.Rp_Interest interest = new INCDTLS2026.Rp_Interest();
                        report.Rp_InterestCollection.Add(interest);
                        report.Rp_InterestCollectionCount += 1;
                        interest.OccurrenceIndex = report.Rp_InterestCollectionCount;
                        _currentXPath.Push("/tns:Interest[" + report.Rp_InterestCollectionCount + "]");
#region FinancialInstitutionAccountFinancialInstitutionNameT
                        _currentXPath.Push("/tns:FinancialInstitutionAccountFinancialInstitutionNameT");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "FinancialInstitutionAccountFinancialInstitutionNameT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    interest.INCDTLS351 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion FinancialInstitutionAccountFinancialInstitutionNameT
#region MiscellaneousInvestorReferenceNumberId
                        _currentXPath.Push("/tns:MiscellaneousInvestorReferenceNumberId");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "MiscellaneousInvestorReferenceNumberId" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    interest.INCDTLS352 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion MiscellaneousInvestorReferenceNumberId
#region RegulatoryDisclosuresAccountHolderCt
                        _currentXPath.Push("/tns:RegulatoryDisclosuresAccountHolderCt");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "RegulatoryDisclosuresAccountHolderCt" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        interest.INCDTLS353 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion RegulatoryDisclosuresAccountHolderCt
#region AddressDetailsCountryC
                        _currentXPath.Push("/tns:AddressDetailsCountryC");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "AddressDetailsCountryC" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    interest.INCDTLS354 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion AddressDetailsCountryC
#region IncomeAdjustmentReasonC
                        _currentXPath.Push("/tns:IncomeAdjustmentReasonC");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeAdjustmentReasonC" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    interest.INCDTLS534 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion IncomeAdjustmentReasonC
#region IncomeAdjustmentReasonDe
                        _currentXPath.Push("/tns:IncomeAdjustmentReasonDe");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeAdjustmentReasonDe" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    interest.INCDTLS535 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion IncomeAdjustmentReasonDe
#region InterestTotal
                        _currentXPath.Push("/tns:InterestTotal");
                        //3. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "InterestTotal" && reader.Depth == _currentXPath.Count - 1)
                        {
                            interest.Rp_Interest_InterestTotalCollectionExists = true;
                            interest.Rp_Interest_InterestTotalCollectionCount += 1;
                            _found = true;
#region IncomeTaxTFNAmountsWithheldFromGrossInterestA
                            _currentXPath.Push("/tns:IncomeTaxTFNAmountsWithheldFromGrossInterestA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxTFNAmountsWithheldFromGrossInterestA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            interest.INCDTLS358 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxTFNAmountsWithheldFromGrossInterestA
#region IncomeGrossA
                            _currentXPath.Push("/tns:IncomeGrossA");
                            //6. use case
                            if (ReadToNextElement(reader, true) && reader.LocalName == "IncomeGrossA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            interest.INCDTLS359 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeGrossA
                        } // End of if InterestTotal node exists

                        _currentXPath.Pop();
#endregion InterestTotal
#region InterestYourShare
                        _currentXPath.Push("/tns:InterestYourShare");
                        //3. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "InterestYourShare" && reader.Depth == _currentXPath.Count - 1)
                        {
                            interest.Rp_Interest_InterestYourShareCollectionExists = true;
                            interest.Rp_Interest_InterestYourShareCollectionCount += 1;
                            _found = true;
#region IncomeTaxTFNAmountsWithheldFromGrossInterestA
                            _currentXPath.Push("/tns:IncomeTaxTFNAmountsWithheldFromGrossInterestA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxTFNAmountsWithheldFromGrossInterestA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            interest.INCDTLS361 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxTFNAmountsWithheldFromGrossInterestA
#region IncomeGrossA
                            _currentXPath.Push("/tns:IncomeGrossA");
                            //6. use case
                            if (ReadToNextElement(reader, true) && reader.LocalName == "IncomeGrossA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            interest.INCDTLS362 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeGrossA
                        } // End of if InterestYourShare node exists

                        _currentXPath.Pop();
#endregion InterestYourShare
                        _currentXPath.Pop();
                    } // End of while loop for Interest
                } // End of if InterestCollection node exists

                _currentXPath.Pop();
#endregion Interest
#region Dividends
                _currentXPath.Push("/tns:DividendsCollection");
                // 4. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "DividendsCollection" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.Rp_DividendsCollection = new List<INCDTLS2026.Rp_Dividends>();
                    report.Rp_DividendsCollectionExists = true;
                    _found = true;
                    while (ReadToNextElement(reader, false) && reader.LocalName == "Dividends")
                    {
                        _found = true;
                        INCDTLS2026.Rp_Dividends dividends = new INCDTLS2026.Rp_Dividends();
                        report.Rp_DividendsCollection.Add(dividends);
                        report.Rp_DividendsCollectionCount += 1;
                        dividends.OccurrenceIndex = report.Rp_DividendsCollectionCount;
                        _currentXPath.Push("/tns:Dividends[" + report.Rp_DividendsCollectionCount + "]");
#region OrganisationNameDetailsOrganisationalNameT
                        _currentXPath.Push("/tns:OrganisationNameDetailsOrganisationalNameT");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "OrganisationNameDetailsOrganisationalNameT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    dividends.INCDTLS345 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion OrganisationNameDetailsOrganisationalNameT
#region MiscellaneousInvestorReferenceNumberId
                        _currentXPath.Push("/tns:MiscellaneousInvestorReferenceNumberId");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "MiscellaneousInvestorReferenceNumberId" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    dividends.INCDTLS346 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion MiscellaneousInvestorReferenceNumberId
#region RegulatoryDisclosuresAccountHolderCt
                        _currentXPath.Push("/tns:RegulatoryDisclosuresAccountHolderCt");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "RegulatoryDisclosuresAccountHolderCt" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        dividends.INCDTLS347 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion RegulatoryDisclosuresAccountHolderCt
#region AddressDetailsCountryC
                        _currentXPath.Push("/tns:AddressDetailsCountryC");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "AddressDetailsCountryC" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    dividends.INCDTLS348 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion AddressDetailsCountryC
#region DividendTotal
                        _currentXPath.Push("/tns:DividendTotal");
                        //3. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "DividendTotal" && reader.Depth == _currentXPath.Count - 1)
                        {
                            dividends.Rp_Dividends_DividendTotalCollectionExists = true;
                            dividends.Rp_Dividends_DividendTotalCollectionCount += 1;
                            _found = true;
#region IncomeUnfrankedA
                            _currentXPath.Push("/tns:IncomeUnfrankedA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeUnfrankedA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            dividends.INCDTLS371 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeUnfrankedA
#region IncomeFrankedA
                            _currentXPath.Push("/tns:IncomeFrankedA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeFrankedA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            dividends.INCDTLS372 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeFrankedA
#region IncomeTaxFrankingCreditsA
                            _currentXPath.Push("/tns:IncomeTaxFrankingCreditsA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxFrankingCreditsA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            dividends.INCDTLS373 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxFrankingCreditsA
#region IncomeTaxPayAsYouGoWithholdingCreditTaxWithheldDividendsTFNNotQuotedA
                            _currentXPath.Push("/tns:IncomeTaxPayAsYouGoWithholdingCreditTaxWithheldDividendsTFNNotQuotedA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxPayAsYouGoWithholdingCreditTaxWithheldDividendsTFNNotQuotedA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            dividends.INCDTLS374 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxPayAsYouGoWithholdingCreditTaxWithheldDividendsTFNNotQuotedA
#region IncomeTaxExplorationCreditsDistributedA
                            _currentXPath.Push("/tns:IncomeTaxExplorationCreditsDistributedA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxExplorationCreditsDistributedA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            dividends.INCDTLS375 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxExplorationCreditsDistributedA
#region CapitalGainsA
                            _currentXPath.Push("/tns:CapitalGainsA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "CapitalGainsA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            dividends.INCDTLS376 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion CapitalGainsA
                        } // End of if DividendTotal node exists

                        _currentXPath.Pop();
#endregion DividendTotal
#region DividendYourShare
                        _currentXPath.Push("/tns:DividendYourShare");
                        //3. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "DividendYourShare" && reader.Depth == _currentXPath.Count - 1)
                        {
                            dividends.Rp_Dividends_DividendYourShareCollectionExists = true;
                            dividends.Rp_Dividends_DividendYourShareCollectionCount += 1;
                            _found = true;
#region IncomeUnfrankedA
                            _currentXPath.Push("/tns:IncomeUnfrankedA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeUnfrankedA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            dividends.INCDTLS378 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeUnfrankedA
#region IncomeFrankedA
                            _currentXPath.Push("/tns:IncomeFrankedA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeFrankedA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            dividends.INCDTLS379 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeFrankedA
#region IncomeTaxFrankingCreditsA
                            _currentXPath.Push("/tns:IncomeTaxFrankingCreditsA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxFrankingCreditsA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            dividends.INCDTLS380 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxFrankingCreditsA
#region IncomeTaxPayAsYouGoWithholdingCreditTaxWithheldDividendsTFNNotQuotedA
                            _currentXPath.Push("/tns:IncomeTaxPayAsYouGoWithholdingCreditTaxWithheldDividendsTFNNotQuotedA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxPayAsYouGoWithholdingCreditTaxWithheldDividendsTFNNotQuotedA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            dividends.INCDTLS381 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxPayAsYouGoWithholdingCreditTaxWithheldDividendsTFNNotQuotedA
#region IncomeTaxExplorationCreditsDistributedA
                            _currentXPath.Push("/tns:IncomeTaxExplorationCreditsDistributedA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxExplorationCreditsDistributedA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            dividends.INCDTLS382 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxExplorationCreditsDistributedA
#region CapitalGainsTotalA
                            _currentXPath.Push("/tns:CapitalGainsTotalA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "CapitalGainsTotalA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            dividends.INCDTLS383 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion CapitalGainsTotalA
                        } // End of if DividendYourShare node exists

                        _currentXPath.Pop();
#endregion DividendYourShare
                        _currentXPath.Pop();
                    } // End of while loop for Dividends
                } // End of if DividendsCollection node exists

                _currentXPath.Pop();
#endregion Dividends
#region ManagedFunds
                _currentXPath.Push("/tns:ManagedFundsCollection");
                // 4. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "ManagedFundsCollection" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.Rp_ManagedFundsCollection = new List<INCDTLS2026.Rp_ManagedFunds>();
                    report.Rp_ManagedFundsCollectionExists = true;
                    _found = true;
                    while (ReadToNextElement(reader, false) && reader.LocalName == "ManagedFunds")
                    {
                        _found = true;
                        INCDTLS2026.Rp_ManagedFunds managedFunds = new INCDTLS2026.Rp_ManagedFunds();
                        report.Rp_ManagedFundsCollection.Add(managedFunds);
                        report.Rp_ManagedFundsCollectionCount += 1;
                        managedFunds.OccurrenceIndex = report.Rp_ManagedFundsCollectionCount;
                        _currentXPath.Push("/tns:ManagedFunds[" + report.Rp_ManagedFundsCollectionCount + "]");
#region OrganisationNameDetailsOrganisationalNameT
                        _currentXPath.Push("/tns:OrganisationNameDetailsOrganisationalNameT");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "OrganisationNameDetailsOrganisationalNameT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    managedFunds.INCDTLS385 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion OrganisationNameDetailsOrganisationalNameT
#region MiscellaneousInvestorReferenceNumberId
                        _currentXPath.Push("/tns:MiscellaneousInvestorReferenceNumberId");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "MiscellaneousInvestorReferenceNumberId" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    managedFunds.INCDTLS386 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion MiscellaneousInvestorReferenceNumberId
#region RegulatoryDisclosuresAccountHolderCt
                        _currentXPath.Push("/tns:RegulatoryDisclosuresAccountHolderCt");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "RegulatoryDisclosuresAccountHolderCt" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        managedFunds.INCDTLS387 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion RegulatoryDisclosuresAccountHolderCt
#region ManagedFundsTotal
                        _currentXPath.Push("/tns:ManagedFundsTotal");
                        //3. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "ManagedFundsTotal" && reader.Depth == _currentXPath.Count - 1)
                        {
                            managedFunds.Rp_ManagedFunds_ManagedFundsTotalCollectionExists = true;
                            managedFunds.Rp_ManagedFunds_ManagedFundsTotalCollectionCount += 1;
                            _found = true;
#region IncomeNonPrimaryProductionNetA
                            _currentXPath.Push("/tns:IncomeNonPrimaryProductionNetA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeNonPrimaryProductionNetA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            managedFunds.INCDTLS389 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeNonPrimaryProductionNetA
#region IncomeTrustFrankedNonPrimaryProductionShareA
                            _currentXPath.Push("/tns:IncomeTrustFrankedNonPrimaryProductionShareA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTrustFrankedNonPrimaryProductionShareA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            managedFunds.INCDTLS390 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTrustFrankedNonPrimaryProductionShareA
#region IncomeTaxDeductionTrustPartnershipOtherNonPrimaryProductionNetA
                            _currentXPath.Push("/tns:IncomeTaxDeductionTrustPartnershipOtherNonPrimaryProductionNetA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxDeductionTrustPartnershipOtherNonPrimaryProductionNetA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            managedFunds.INCDTLS391 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxDeductionTrustPartnershipOtherNonPrimaryProductionNetA
#region IncomeTaxFrankingCreditsShareA
                            _currentXPath.Push("/tns:IncomeTaxFrankingCreditsShareA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxFrankingCreditsShareA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            managedFunds.INCDTLS392 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxFrankingCreditsShareA
#region IncomeTaxPayAsYouGoWithholdingCreditForAmountsWithheldWhereTFNNotQuotedA
                            _currentXPath.Push("/tns:IncomeTaxPayAsYouGoWithholdingCreditForAmountsWithheldWhereTFNNotQuotedA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxPayAsYouGoWithholdingCreditForAmountsWithheldWhereTFNNotQuotedA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            managedFunds.INCDTLS393 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxPayAsYouGoWithholdingCreditForAmountsWithheldWhereTFNNotQuotedA
#region InternationalDealingsForeignIncomeGrossA
                            _currentXPath.Push("/tns:InternationalDealingsForeignIncomeGrossA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsForeignIncomeGrossA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            managedFunds.INCDTLS398 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion InternationalDealingsForeignIncomeGrossA
#region IncomeInternationalDealingsNetA
                            _currentXPath.Push("/tns:IncomeInternationalDealingsNetA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeInternationalDealingsNetA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            managedFunds.INCDTLS399 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeInternationalDealingsNetA
#region IncomeInternationalDealingsTaxOffsetA
                            _currentXPath.Push("/tns:IncomeInternationalDealingsTaxOffsetA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeInternationalDealingsTaxOffsetA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            managedFunds.INCDTLS400 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeInternationalDealingsTaxOffsetA
#region IncomeTaxIncomeCategoryMiscellaneousTotalA
                            _currentXPath.Push("/tns:IncomeTaxIncomeCategoryMiscellaneousTotalA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxIncomeCategoryMiscellaneousTotalA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            managedFunds.INCDTLS401 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxIncomeCategoryMiscellaneousTotalA
#region IncomeTrustPrimaryProductionNetShareA
                            _currentXPath.Push("/tns:IncomeTrustPrimaryProductionNetShareA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTrustPrimaryProductionNetShareA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            managedFunds.INCDTLS402 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTrustPrimaryProductionNetShareA
#region IncomeTaxDeductionPartnershipsAndTrustPrimaryProductionShareA
                            _currentXPath.Push("/tns:IncomeTaxDeductionPartnershipsAndTrustPrimaryProductionShareA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxDeductionPartnershipsAndTrustPrimaryProductionShareA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            managedFunds.INCDTLS403 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxDeductionPartnershipsAndTrustPrimaryProductionShareA
#region IncomeTaxPayAsYouGoWithholdingCreditForTaxWithheldWhereABNNotQuotedA
                            _currentXPath.Push("/tns:IncomeTaxPayAsYouGoWithholdingCreditForTaxWithheldWhereABNNotQuotedA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxPayAsYouGoWithholdingCreditForTaxWithheldWhereABNNotQuotedA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            managedFunds.INCDTLS404 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxPayAsYouGoWithholdingCreditForTaxWithheldWhereABNNotQuotedA
#region IncomeTaxCreditTrusteeTaxPaidTotalA
                            _currentXPath.Push("/tns:IncomeTaxCreditTrusteeTaxPaidTotalA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxCreditTrusteeTaxPaidTotalA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            managedFunds.INCDTLS405 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxCreditTrusteeTaxPaidTotalA
#region IncomeTaxCreditTrusteeTaxPaidReasonC
                            _currentXPath.Push("/tns:IncomeTaxCreditTrusteeTaxPaidReasonC");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxCreditTrusteeTaxPaidReasonC" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        managedFunds.INCDTLS406 = currentValue;
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxCreditTrusteeTaxPaidReasonC
#region IncomeTrustDistributionsA
                            _currentXPath.Push("/tns:IncomeTrustDistributionsA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTrustDistributionsA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            managedFunds.INCDTLS407 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTrustDistributionsA
#region IncomeTaxPayAsYouGoWithholdingCreditTaxWithheldForeignResidentShareA
                            _currentXPath.Push("/tns:IncomeTaxPayAsYouGoWithholdingCreditTaxWithheldForeignResidentShareA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxPayAsYouGoWithholdingCreditTaxWithheldForeignResidentShareA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            managedFunds.INCDTLS408 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxPayAsYouGoWithholdingCreditTaxWithheldForeignResidentShareA
#region TaxOffsetClaimNationalRentalAffordabilitySchemeEntitlementShareA
                            _currentXPath.Push("/tns:TaxOffsetClaimNationalRentalAffordabilitySchemeEntitlementShareA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "TaxOffsetClaimNationalRentalAffordabilitySchemeEntitlementShareA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            managedFunds.INCDTLS409 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion TaxOffsetClaimNationalRentalAffordabilitySchemeEntitlementShareA
#region IncomeTaxPayAsYouGoWithholdingCreditForCapitalGainsWithheldFromForeignResidentsA
                            _currentXPath.Push("/tns:IncomeTaxPayAsYouGoWithholdingCreditForCapitalGainsWithheldFromForeignResidentsA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxPayAsYouGoWithholdingCreditForCapitalGainsWithheldFromForeignResidentsA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            managedFunds.INCDTLS410 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxPayAsYouGoWithholdingCreditForCapitalGainsWithheldFromForeignResidentsA
#region InternationalDealingsTransferorTrustAttributedForeignIncomeA
                            _currentXPath.Push("/tns:InternationalDealingsTransferorTrustAttributedForeignIncomeA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsTransferorTrustAttributedForeignIncomeA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            managedFunds.INCDTLS411 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion InternationalDealingsTransferorTrustAttributedForeignIncomeA
#region IncomeControlledForeignCompaniesAttributableIncomeA
                            _currentXPath.Push("/tns:IncomeControlledForeignCompaniesAttributableIncomeA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeControlledForeignCompaniesAttributableIncomeA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            managedFunds.INCDTLS412 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeControlledForeignCompaniesAttributableIncomeA
#region InternationalDealingsForeignRentA
                            _currentXPath.Push("/tns:InternationalDealingsForeignRentA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsForeignRentA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            managedFunds.INCDTLS413 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion InternationalDealingsForeignRentA
#region IncomeTaxFrankingCreditsNewZealandA
                            _currentXPath.Push("/tns:IncomeTaxFrankingCreditsNewZealandA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxFrankingCreditsNewZealandA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            managedFunds.INCDTLS414 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxFrankingCreditsNewZealandA
#region IncomeTaxEarlyStageVentureCapitalLimitedPartnershipTaxOffsetA
                            _currentXPath.Push("/tns:IncomeTaxEarlyStageVentureCapitalLimitedPartnershipTaxOffsetA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxEarlyStageVentureCapitalLimitedPartnershipTaxOffsetA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            managedFunds.INCDTLS415 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxEarlyStageVentureCapitalLimitedPartnershipTaxOffsetA
#region IncomeTaxEarlyStageInvestorTaxOffsetA
                            _currentXPath.Push("/tns:IncomeTaxEarlyStageInvestorTaxOffsetA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxEarlyStageInvestorTaxOffsetA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            managedFunds.INCDTLS416 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxEarlyStageInvestorTaxOffsetA
#region IncomeTaxExplorationCreditsDistributedA
                            _currentXPath.Push("/tns:IncomeTaxExplorationCreditsDistributedA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxExplorationCreditsDistributedA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            managedFunds.INCDTLS417 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxExplorationCreditsDistributedA
#region CapitalGains
                            _currentXPath.Push("/tns:CapitalGains");
                            //3. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "CapitalGains" && reader.Depth == _currentXPath.Count - 1)
                            {
                                managedFunds.Rp_ManagedFunds_ManagedFundsTotal_CapitalGainsCollectionExists = true;
                                managedFunds.Rp_ManagedFunds_ManagedFundsTotal_CapitalGainsCollectionCount += 1;
                                _found = true;
#region NetA
                                _currentXPath.Push("/tns:NetA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "NetA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                managedFunds.INCDTLS394 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion NetA
#region A
                                _currentXPath.Push("/tns:A");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "A" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                managedFunds.INCDTLS395 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion A
#region DiscountedMethodA
                                _currentXPath.Push("/tns:DiscountedMethodA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "DiscountedMethodA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                managedFunds.INCDTLS396 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion DiscountedMethodA
#region OtherMethodA
                                _currentXPath.Push("/tns:OtherMethodA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "OtherMethodA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                managedFunds.INCDTLS397 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion OtherMethodA
                            } // End of if CapitalGains node exists

                            _currentXPath.Pop();
#endregion CapitalGains
                        } // End of if ManagedFundsTotal node exists

                        _currentXPath.Pop();
#endregion ManagedFundsTotal
#region ManagedFundsYourShare
                        _currentXPath.Push("/tns:ManagedFundsYourShare");
                        //3. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ManagedFundsYourShare" && reader.Depth == _currentXPath.Count - 1)
                        {
                            managedFunds.Rp_ManagedFunds_ManagedFundsYourShareCollectionExists = true;
                            managedFunds.Rp_ManagedFunds_ManagedFundsYourShareCollectionCount += 1;
                            _found = true;
#region IncomeNonPrimaryProductionNetA
                            _currentXPath.Push("/tns:IncomeNonPrimaryProductionNetA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeNonPrimaryProductionNetA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            managedFunds.INCDTLS419 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeNonPrimaryProductionNetA
#region IncomeTrustFrankedNonPrimaryProductionShareA
                            _currentXPath.Push("/tns:IncomeTrustFrankedNonPrimaryProductionShareA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTrustFrankedNonPrimaryProductionShareA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            managedFunds.INCDTLS420 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTrustFrankedNonPrimaryProductionShareA
#region IncomeTaxDeductionTrustPartnershipOtherNonPrimaryProductionNetA
                            _currentXPath.Push("/tns:IncomeTaxDeductionTrustPartnershipOtherNonPrimaryProductionNetA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxDeductionTrustPartnershipOtherNonPrimaryProductionNetA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            managedFunds.INCDTLS421 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxDeductionTrustPartnershipOtherNonPrimaryProductionNetA
#region IncomeTaxFrankingCreditsShareA
                            _currentXPath.Push("/tns:IncomeTaxFrankingCreditsShareA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxFrankingCreditsShareA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            managedFunds.INCDTLS422 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxFrankingCreditsShareA
#region IncomeTaxPayAsYouGoWithholdingCreditForAmountsWithheldWhereTFNNotQuotedA
                            _currentXPath.Push("/tns:IncomeTaxPayAsYouGoWithholdingCreditForAmountsWithheldWhereTFNNotQuotedA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxPayAsYouGoWithholdingCreditForAmountsWithheldWhereTFNNotQuotedA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            managedFunds.INCDTLS423 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxPayAsYouGoWithholdingCreditForAmountsWithheldWhereTFNNotQuotedA
#region InternationalDealingsForeignIncomeGrossA
                            _currentXPath.Push("/tns:InternationalDealingsForeignIncomeGrossA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsForeignIncomeGrossA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            managedFunds.INCDTLS428 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion InternationalDealingsForeignIncomeGrossA
#region IncomeInternationalDealingsNetA
                            _currentXPath.Push("/tns:IncomeInternationalDealingsNetA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeInternationalDealingsNetA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            managedFunds.INCDTLS429 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeInternationalDealingsNetA
#region IncomeInternationalDealingsTaxOffsetA
                            _currentXPath.Push("/tns:IncomeInternationalDealingsTaxOffsetA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeInternationalDealingsTaxOffsetA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            managedFunds.INCDTLS430 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeInternationalDealingsTaxOffsetA
#region IncomeTaxIncomeCategoryMiscellaneousTotalA
                            _currentXPath.Push("/tns:IncomeTaxIncomeCategoryMiscellaneousTotalA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxIncomeCategoryMiscellaneousTotalA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            managedFunds.INCDTLS431 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxIncomeCategoryMiscellaneousTotalA
#region IncomeTrustPrimaryProductionNetShareA
                            _currentXPath.Push("/tns:IncomeTrustPrimaryProductionNetShareA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTrustPrimaryProductionNetShareA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            managedFunds.INCDTLS432 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTrustPrimaryProductionNetShareA
#region IncomeTaxDeductionPartnershipsAndTrustPrimaryProductionShareA
                            _currentXPath.Push("/tns:IncomeTaxDeductionPartnershipsAndTrustPrimaryProductionShareA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxDeductionPartnershipsAndTrustPrimaryProductionShareA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            managedFunds.INCDTLS433 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxDeductionPartnershipsAndTrustPrimaryProductionShareA
#region IncomeTaxPayAsYouGoWithholdingCreditForTaxWithheldWhereABNNotQuotedA
                            _currentXPath.Push("/tns:IncomeTaxPayAsYouGoWithholdingCreditForTaxWithheldWhereABNNotQuotedA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxPayAsYouGoWithholdingCreditForTaxWithheldWhereABNNotQuotedA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            managedFunds.INCDTLS434 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxPayAsYouGoWithholdingCreditForTaxWithheldWhereABNNotQuotedA
#region IncomeTaxCreditTrusteeTaxPaidTotalA
                            _currentXPath.Push("/tns:IncomeTaxCreditTrusteeTaxPaidTotalA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxCreditTrusteeTaxPaidTotalA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            managedFunds.INCDTLS435 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxCreditTrusteeTaxPaidTotalA
#region IncomeTaxPayAsYouGoWithholdingCreditTaxWithheldForeignResidentShareA
                            _currentXPath.Push("/tns:IncomeTaxPayAsYouGoWithholdingCreditTaxWithheldForeignResidentShareA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxPayAsYouGoWithholdingCreditTaxWithheldForeignResidentShareA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            managedFunds.INCDTLS436 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxPayAsYouGoWithholdingCreditTaxWithheldForeignResidentShareA
#region TaxOffsetClaimNationalRentalAffordabilitySchemeEntitlementShareA
                            _currentXPath.Push("/tns:TaxOffsetClaimNationalRentalAffordabilitySchemeEntitlementShareA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "TaxOffsetClaimNationalRentalAffordabilitySchemeEntitlementShareA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            managedFunds.INCDTLS437 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion TaxOffsetClaimNationalRentalAffordabilitySchemeEntitlementShareA
#region IncomeTaxPayAsYouGoWithholdingCreditForCapitalGainsWithheldFromForeignResidentsA
                            _currentXPath.Push("/tns:IncomeTaxPayAsYouGoWithholdingCreditForCapitalGainsWithheldFromForeignResidentsA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxPayAsYouGoWithholdingCreditForCapitalGainsWithheldFromForeignResidentsA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            managedFunds.INCDTLS438 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxPayAsYouGoWithholdingCreditForCapitalGainsWithheldFromForeignResidentsA
#region InternationalDealingsTransferorTrustAttributedForeignIncomeA
                            _currentXPath.Push("/tns:InternationalDealingsTransferorTrustAttributedForeignIncomeA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsTransferorTrustAttributedForeignIncomeA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            managedFunds.INCDTLS439 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion InternationalDealingsTransferorTrustAttributedForeignIncomeA
#region IncomeControlledForeignCompaniesAttributableIncomeA
                            _currentXPath.Push("/tns:IncomeControlledForeignCompaniesAttributableIncomeA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeControlledForeignCompaniesAttributableIncomeA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            managedFunds.INCDTLS440 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeControlledForeignCompaniesAttributableIncomeA
#region InternationalDealingsForeignRentA
                            _currentXPath.Push("/tns:InternationalDealingsForeignRentA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsForeignRentA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            managedFunds.INCDTLS441 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion InternationalDealingsForeignRentA
#region IncomeTaxFrankingCreditsNewZealandA
                            _currentXPath.Push("/tns:IncomeTaxFrankingCreditsNewZealandA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxFrankingCreditsNewZealandA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            managedFunds.INCDTLS442 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxFrankingCreditsNewZealandA
#region IncomeTaxEarlyStageVentureCapitalLimitedPartnershipTaxOffsetA
                            _currentXPath.Push("/tns:IncomeTaxEarlyStageVentureCapitalLimitedPartnershipTaxOffsetA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxEarlyStageVentureCapitalLimitedPartnershipTaxOffsetA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            managedFunds.INCDTLS443 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxEarlyStageVentureCapitalLimitedPartnershipTaxOffsetA
#region IncomeTaxEarlyStageInvestorTaxOffsetA
                            _currentXPath.Push("/tns:IncomeTaxEarlyStageInvestorTaxOffsetA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxEarlyStageInvestorTaxOffsetA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            managedFunds.INCDTLS444 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxEarlyStageInvestorTaxOffsetA
#region IncomeTaxExplorationCreditsDistributedA
                            _currentXPath.Push("/tns:IncomeTaxExplorationCreditsDistributedA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxExplorationCreditsDistributedA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            managedFunds.INCDTLS445 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxExplorationCreditsDistributedA
#region YourShareCapitalGains
                            _currentXPath.Push("/tns:YourShareCapitalGains");
                            //3. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "YourShareCapitalGains" && reader.Depth == _currentXPath.Count - 1)
                            {
                                managedFunds.Rp_ManagedFunds_ManagedFundsYourShare_YourShareCapitalGainsCollectionExists = true;
                                managedFunds.Rp_ManagedFunds_ManagedFundsYourShare_YourShareCapitalGainsCollectionCount += 1;
                                _found = true;
#region NetA
                                _currentXPath.Push("/tns:NetA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "NetA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                managedFunds.INCDTLS424 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion NetA
#region TotalA
                                _currentXPath.Push("/tns:TotalA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "TotalA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                managedFunds.INCDTLS425 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion TotalA
#region DiscountedMethodA
                                _currentXPath.Push("/tns:DiscountedMethodA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "DiscountedMethodA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                managedFunds.INCDTLS426 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion DiscountedMethodA
#region OtherMethodA
                                _currentXPath.Push("/tns:OtherMethodA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "OtherMethodA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                managedFunds.INCDTLS427 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion OtherMethodA
                            } // End of if YourShareCapitalGains node exists

                            _currentXPath.Pop();
#endregion YourShareCapitalGains
                        } // End of if ManagedFundsYourShare node exists

                        _currentXPath.Pop();
#endregion ManagedFundsYourShare
                        _currentXPath.Pop();
                    } // End of while loop for ManagedFunds
                } // End of if ManagedFundsCollection node exists

                _currentXPath.Pop();
#endregion ManagedFunds
#region ForeignIncome
                _currentXPath.Push("/tns:ForeignIncomeCollection");
                // 4. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "ForeignIncomeCollection" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.Rp_ForeignIncomeCollection = new List<INCDTLS2026.Rp_ForeignIncome>();
                    report.Rp_ForeignIncomeCollectionExists = true;
                    _found = true;
                    while (ReadToNextElement(reader, false) && reader.LocalName == "ForeignIncome")
                    {
                        _found = true;
                        INCDTLS2026.Rp_ForeignIncome foreignIncome = new INCDTLS2026.Rp_ForeignIncome();
                        report.Rp_ForeignIncomeCollection.Add(foreignIncome);
                        report.Rp_ForeignIncomeCollectionCount += 1;
                        foreignIncome.OccurrenceIndex = report.Rp_ForeignIncomeCollectionCount;
                        _currentXPath.Push("/tns:ForeignIncome[" + report.Rp_ForeignIncomeCollectionCount + "]");
#region OrganisationNameDetailsOrganisationalNameT
                        _currentXPath.Push("/tns:OrganisationNameDetailsOrganisationalNameT");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "OrganisationNameDetailsOrganisationalNameT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    foreignIncome.INCDTLS186 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion OrganisationNameDetailsOrganisationalNameT
#region ExpenseDeductionsA
                        _currentXPath.Push("/tns:ExpenseDeductionsA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseDeductionsA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        foreignIncome.INCDTLS187 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion ExpenseDeductionsA
#region ForeignIncomeEmploymentDetails
                        _currentXPath.Push("/tns:ForeignIncomeEmploymentDetails");
                        //3. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ForeignIncomeEmploymentDetails" && reader.Depth == _currentXPath.Count - 1)
                        {
                            foreignIncome.Rp_ForeignIncome_ForeignIncomeEmploymentDetailsCollectionExists = true;
                            foreignIncome.Rp_ForeignIncome_ForeignIncomeEmploymentDetailsCollectionCount += 1;
                            _found = true;
#region InternationalDealingsGrossA
                            _currentXPath.Push("/tns:InternationalDealingsGrossA");
                            //6. use case
                            if (ReadToNextElement(reader, true) && reader.LocalName == "InternationalDealingsGrossA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            foreignIncome.INCDTLS189 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion InternationalDealingsGrossA
#region InternationalDealingsTaxOffsetA
                            _currentXPath.Push("/tns:InternationalDealingsTaxOffsetA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsTaxOffsetA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            foreignIncome.INCDTLS190 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion InternationalDealingsTaxOffsetA
                        } // End of if ForeignIncomeEmploymentDetails node exists

                        _currentXPath.Pop();
#endregion ForeignIncomeEmploymentDetails
#region ExemptForeignIncome
                        _currentXPath.Push("/tns:ExemptForeignIncome");
                        //3. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ExemptForeignIncome" && reader.Depth == _currentXPath.Count - 1)
                        {
                            foreignIncome.Rp_ForeignIncome_ExemptForeignIncomeCollectionExists = true;
                            foreignIncome.Rp_ForeignIncome_ExemptForeignIncomeCollectionCount += 1;
                            _found = true;
#region AddressDetailsCountryC
                            _currentXPath.Push("/tns:AddressDetailsCountryC");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "AddressDetailsCountryC" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        foreignIncome.INCDTLS192 = currentValue;
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion AddressDetailsCountryC
#region InternationalDealingsExemptForeignEmploymentIncomeA
                            _currentXPath.Push("/tns:InternationalDealingsExemptForeignEmploymentIncomeA");
                            //6. use case
                            if (ReadToNextElement(reader, true) && reader.LocalName == "InternationalDealingsExemptForeignEmploymentIncomeA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            foreignIncome.INCDTLS193 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion InternationalDealingsExemptForeignEmploymentIncomeA
#region InternationalDealingsTaxOffsetA
                            _currentXPath.Push("/tns:InternationalDealingsTaxOffsetA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsTaxOffsetA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            foreignIncome.INCDTLS194 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion InternationalDealingsTaxOffsetA
#region LumpSumArrearsPaymentA
                            _currentXPath.Push("/tns:LumpSumArrearsPaymentA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "LumpSumArrearsPaymentA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            foreignIncome.INCDTLS195 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion LumpSumArrearsPaymentA
                        } // End of if ExemptForeignIncome node exists

                        _currentXPath.Pop();
#endregion ExemptForeignIncome
#region FEINPSLumpSumArrearsPayment
                        _currentXPath.Push("/tns:FEINPSLumpSumArrearsPaymentCollection");
                        // 4. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "FEINPSLumpSumArrearsPaymentCollection" && reader.Depth == _currentXPath.Count - 1)
                        {
                            foreignIncome.Rp_ForeignIncome_FEINPSLumpSumArrearsPaymentCollection = new List<INCDTLS2026.Rp_ForeignIncome.Rp_ForeignIncome_FEINPSLumpSumArrearsPayment>();
                            foreignIncome.Rp_ForeignIncome_FEINPSLumpSumArrearsPaymentCollectionExists = true;
                            _found = true;
                            while (ReadToNextElement(reader, false) && reader.LocalName == "FEINPSLumpSumArrearsPayment")
                            {
                                _found = true;
                                INCDTLS2026.Rp_ForeignIncome.Rp_ForeignIncome_FEINPSLumpSumArrearsPayment feinpsLumpSumArrearsPayment = new INCDTLS2026.Rp_ForeignIncome.Rp_ForeignIncome_FEINPSLumpSumArrearsPayment();
                                foreignIncome.Rp_ForeignIncome_FEINPSLumpSumArrearsPaymentCollection.Add(feinpsLumpSumArrearsPayment);
                                foreignIncome.Rp_ForeignIncome_FEINPSLumpSumArrearsPaymentCollectionCount += 1;
                                feinpsLumpSumArrearsPayment.OccurrenceIndex = foreignIncome.Rp_ForeignIncome_FEINPSLumpSumArrearsPaymentCollectionCount;
                                _currentXPath.Push("/tns:FEINPSLumpSumArrearsPayment[" + foreignIncome.Rp_ForeignIncome_FEINPSLumpSumArrearsPaymentCollectionCount + "]");
#region TargetFinancialY
                                _currentXPath.Push("/tns:TargetFinancialY");
                                //6. use case
                                if (ReadToNextElement(reader, true) && reader.LocalName == "TargetFinancialY" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Int32.TryParse(currentValue, out currentIntValue))
                                            {
                                                feinpsLumpSumArrearsPayment.INCDTLS529 = currentIntValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion TargetFinancialY
#region A
                                _currentXPath.Push("/tns:A");
                                //6. use case
                                if (ReadToNextElement(reader, true) && reader.LocalName == "A" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                feinpsLumpSumArrearsPayment.INCDTLS530 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion A
                                _currentXPath.Pop();
                            } // End of while loop for FEINPSLumpSumArrearsPayment
                        } // End of if FEINPSLumpSumArrearsPaymentCollection node exists

                        _currentXPath.Pop();
#endregion FEINPSLumpSumArrearsPayment
                        _currentXPath.Pop();
                    } // End of while loop for ForeignIncome
                } // End of if ForeignIncomeCollection node exists

                _currentXPath.Pop();
#endregion ForeignIncome
#region ForeignEntities
                _currentXPath.Push("/tns:ForeignEntities");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "ForeignEntities" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.Rp_ForeignEntitiesCollectionExists = true;
                    report.Rp_ForeignEntitiesCollectionCount += 1;
                    _found = true;
#region OrganisationNameDetailsOrganisationalNameT
                    _currentXPath.Push("/tns:OrganisationNameDetailsOrganisationalNameT");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "OrganisationNameDetailsOrganisationalNameT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.INCDTLS201 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion OrganisationNameDetailsOrganisationalNameT
#region PersonUnstructuredNameFullNameT
                    _currentXPath.Push("/tns:PersonUnstructuredNameFullNameT");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "PersonUnstructuredNameFullNameT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.INCDTLS202 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion PersonUnstructuredNameFullNameT
#region IncomeControlledForeignCompaniesAttributableIncomeA
                    _currentXPath.Push("/tns:IncomeControlledForeignCompaniesAttributableIncomeA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeControlledForeignCompaniesAttributableIncomeA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.INCDTLS203 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion IncomeControlledForeignCompaniesAttributableIncomeA
#region InternationalDealingsTransferorTrustAttributedForeignIncomeA
                    _currentXPath.Push("/tns:InternationalDealingsTransferorTrustAttributedForeignIncomeA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsTransferorTrustAttributedForeignIncomeA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.INCDTLS204 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion InternationalDealingsTransferorTrustAttributedForeignIncomeA
                } // End of if ForeignEntities node exists

                _currentXPath.Pop();
#endregion ForeignEntities
#region ForeignPensionsOrAnnuities
                _currentXPath.Push("/tns:ForeignPensionsOrAnnuitiesCollection");
                // 4. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "ForeignPensionsOrAnnuitiesCollection" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.Rp_ForeignPensionsOrAnnuitiesCollection = new List<INCDTLS2026.Rp_ForeignPensionsOrAnnuities>();
                    report.Rp_ForeignPensionsOrAnnuitiesCollectionExists = true;
                    _found = true;
                    while (ReadToNextElement(reader, false) && reader.LocalName == "ForeignPensionsOrAnnuities")
                    {
                        _found = true;
                        INCDTLS2026.Rp_ForeignPensionsOrAnnuities foreignPensionsOrAnnuities = new INCDTLS2026.Rp_ForeignPensionsOrAnnuities();
                        report.Rp_ForeignPensionsOrAnnuitiesCollection.Add(foreignPensionsOrAnnuities);
                        report.Rp_ForeignPensionsOrAnnuitiesCollectionCount += 1;
                        foreignPensionsOrAnnuities.OccurrenceIndex = report.Rp_ForeignPensionsOrAnnuitiesCollectionCount;
                        _currentXPath.Push("/tns:ForeignPensionsOrAnnuities[" + report.Rp_ForeignPensionsOrAnnuitiesCollectionCount + "]");
#region InternationalDealingsDescriptionT
                        _currentXPath.Push("/tns:InternationalDealingsDescriptionT");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsDescriptionT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    foreignPensionsOrAnnuities.INCDTLS208 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion InternationalDealingsDescriptionT
#region IncomeInsuranceLifePremiumDirectGrossA
                        _currentXPath.Push("/tns:IncomeInsuranceLifePremiumDirectGrossA");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "IncomeInsuranceLifePremiumDirectGrossA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        foreignPensionsOrAnnuities.INCDTLS209 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion IncomeInsuranceLifePremiumDirectGrossA
#region ExpenseForeignIncomeDeductionsA
                        _currentXPath.Push("/tns:ExpenseForeignIncomeDeductionsA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseForeignIncomeDeductionsA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        foreignPensionsOrAnnuities.INCDTLS210 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion ExpenseForeignIncomeDeductionsA
#region InternationalDealingsForeignPensionOrAnnuityIncomeUndeductedPurchasePriceA
                        _currentXPath.Push("/tns:InternationalDealingsForeignPensionOrAnnuityIncomeUndeductedPurchasePriceA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsForeignPensionOrAnnuityIncomeUndeductedPurchasePriceA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        foreignPensionsOrAnnuities.INCDTLS211 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion InternationalDealingsForeignPensionOrAnnuityIncomeUndeductedPurchasePriceA
#region IncomeInternationalDealingsTaxOffsetA
                        _currentXPath.Push("/tns:IncomeInternationalDealingsTaxOffsetA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeInternationalDealingsTaxOffsetA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        foreignPensionsOrAnnuities.INCDTLS212 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion IncomeInternationalDealingsTaxOffsetA
#region IncomeLumpSumArrearsPaymentA
                        _currentXPath.Push("/tns:IncomeLumpSumArrearsPaymentA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeLumpSumArrearsPaymentA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        foreignPensionsOrAnnuities.INCDTLS213 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion IncomeLumpSumArrearsPaymentA
#region FPALumpSumArrearsPayment
                        _currentXPath.Push("/tns:FPALumpSumArrearsPaymentCollection");
                        // 4. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "FPALumpSumArrearsPaymentCollection" && reader.Depth == _currentXPath.Count - 1)
                        {
                            foreignPensionsOrAnnuities.Rp_ForeignPensionsOrAnnuities_FPALumpSumArrearsPaymentCollection = new List<INCDTLS2026.Rp_ForeignPensionsOrAnnuities.Rp_ForeignPensionsOrAnnuities_FPALumpSumArrearsPayment>();
                            foreignPensionsOrAnnuities.Rp_ForeignPensionsOrAnnuities_FPALumpSumArrearsPaymentCollectionExists = true;
                            _found = true;
                            while (ReadToNextElement(reader, false) && reader.LocalName == "FPALumpSumArrearsPayment")
                            {
                                _found = true;
                                INCDTLS2026.Rp_ForeignPensionsOrAnnuities.Rp_ForeignPensionsOrAnnuities_FPALumpSumArrearsPayment fpaLumpSumArrearsPayment = new INCDTLS2026.Rp_ForeignPensionsOrAnnuities.Rp_ForeignPensionsOrAnnuities_FPALumpSumArrearsPayment();
                                foreignPensionsOrAnnuities.Rp_ForeignPensionsOrAnnuities_FPALumpSumArrearsPaymentCollection.Add(fpaLumpSumArrearsPayment);
                                foreignPensionsOrAnnuities.Rp_ForeignPensionsOrAnnuities_FPALumpSumArrearsPaymentCollectionCount += 1;
                                fpaLumpSumArrearsPayment.OccurrenceIndex = foreignPensionsOrAnnuities.Rp_ForeignPensionsOrAnnuities_FPALumpSumArrearsPaymentCollectionCount;
                                _currentXPath.Push("/tns:FPALumpSumArrearsPayment[" + foreignPensionsOrAnnuities.Rp_ForeignPensionsOrAnnuities_FPALumpSumArrearsPaymentCollectionCount + "]");
#region TargetFinancialY
                                _currentXPath.Push("/tns:TargetFinancialY");
                                //6. use case
                                if (ReadToNextElement(reader, true) && reader.LocalName == "TargetFinancialY" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Int32.TryParse(currentValue, out currentIntValue))
                                            {
                                                fpaLumpSumArrearsPayment.INCDTLS532 = currentIntValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion TargetFinancialY
#region IncomeA
                                _currentXPath.Push("/tns:IncomeA");
                                //6. use case
                                if (ReadToNextElement(reader, true) && reader.LocalName == "IncomeA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                fpaLumpSumArrearsPayment.INCDTLS533 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion IncomeA
                                _currentXPath.Pop();
                            } // End of while loop for FPALumpSumArrearsPayment
                        } // End of if FPALumpSumArrearsPaymentCollection node exists

                        _currentXPath.Pop();
#endregion FPALumpSumArrearsPayment
                        _currentXPath.Pop();
                    } // End of while loop for ForeignPensionsOrAnnuities
                } // End of if ForeignPensionsOrAnnuitiesCollection node exists

                _currentXPath.Pop();
#endregion ForeignPensionsOrAnnuities
#region OtherForeignIncome
                _currentXPath.Push("/tns:OtherForeignIncomeCollection");
                // 4. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "OtherForeignIncomeCollection" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.Rp_OtherForeignIncomeCollection = new List<INCDTLS2026.Rp_OtherForeignIncome>();
                    report.Rp_OtherForeignIncomeCollectionExists = true;
                    _found = true;
                    while (ReadToNextElement(reader, false) && reader.LocalName == "OtherForeignIncome")
                    {
                        _found = true;
                        INCDTLS2026.Rp_OtherForeignIncome otherForeignIncome = new INCDTLS2026.Rp_OtherForeignIncome();
                        report.Rp_OtherForeignIncomeCollection.Add(otherForeignIncome);
                        report.Rp_OtherForeignIncomeCollectionCount += 1;
                        otherForeignIncome.OccurrenceIndex = report.Rp_OtherForeignIncomeCollectionCount;
                        _currentXPath.Push("/tns:OtherForeignIncome[" + report.Rp_OtherForeignIncomeCollectionCount + "]");
#region InternationalDealingsDescriptionT
                        _currentXPath.Push("/tns:InternationalDealingsDescriptionT");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsDescriptionT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    otherForeignIncome.INCDTLS216 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion InternationalDealingsDescriptionT
#region ExpenseForeignIncomeDeductionsA
                        _currentXPath.Push("/tns:ExpenseForeignIncomeDeductionsA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseForeignIncomeDeductionsA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        otherForeignIncome.INCDTLS217 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion ExpenseForeignIncomeDeductionsA
#region ForeignRentalIncomeDetails
                        _currentXPath.Push("/tns:ForeignRentalIncomeDetails");
                        //3. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ForeignRentalIncomeDetails" && reader.Depth == _currentXPath.Count - 1)
                        {
                            otherForeignIncome.Rp_OtherForeignIncome_ForeignRentalIncomeDetailsCollectionExists = true;
                            otherForeignIncome.Rp_OtherForeignIncome_ForeignRentalIncomeDetailsCollectionCount += 1;
                            _found = true;
#region InternationalDealingsA
                            _currentXPath.Push("/tns:InternationalDealingsA");
                            //6. use case
                            if (ReadToNextElement(reader, true) && reader.LocalName == "InternationalDealingsA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            otherForeignIncome.INCDTLS219 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion InternationalDealingsA
#region InternationalDealingsTaxOffsetA
                            _currentXPath.Push("/tns:InternationalDealingsTaxOffsetA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsTaxOffsetA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            otherForeignIncome.INCDTLS220 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion InternationalDealingsTaxOffsetA
                        } // End of if ForeignRentalIncomeDetails node exists

                        _currentXPath.Pop();
#endregion ForeignRentalIncomeDetails
#region ForeignFinancialInvestmentDetails
                        _currentXPath.Push("/tns:ForeignFinancialInvestmentDetails");
                        //3. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ForeignFinancialInvestmentDetails" && reader.Depth == _currentXPath.Count - 1)
                        {
                            otherForeignIncome.Rp_OtherForeignIncome_ForeignFinancialInvestmentDetailsCollectionExists = true;
                            otherForeignIncome.Rp_OtherForeignIncome_ForeignFinancialInvestmentDetailsCollectionCount += 1;
                            _found = true;
#region InternationalDealingsForeignSourceCompanyIncomeOtherA
                            _currentXPath.Push("/tns:InternationalDealingsForeignSourceCompanyIncomeOtherA");
                            //6. use case
                            if (ReadToNextElement(reader, true) && reader.LocalName == "InternationalDealingsForeignSourceCompanyIncomeOtherA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            otherForeignIncome.INCDTLS222 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion InternationalDealingsForeignSourceCompanyIncomeOtherA
#region IncomeTaxFrankingCreditsNewZealandA
                            _currentXPath.Push("/tns:IncomeTaxFrankingCreditsNewZealandA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxFrankingCreditsNewZealandA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            otherForeignIncome.INCDTLS223 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxFrankingCreditsNewZealandA
#region InternationalDealingsTaxOffsetA
                            _currentXPath.Push("/tns:InternationalDealingsTaxOffsetA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsTaxOffsetA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            otherForeignIncome.INCDTLS224 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion InternationalDealingsTaxOffsetA
                        } // End of if ForeignFinancialInvestmentDetails node exists

                        _currentXPath.Pop();
#endregion ForeignFinancialInvestmentDetails
#region OtherForeignIncomeDetails
                        _currentXPath.Push("/tns:OtherForeignIncomeDetails");
                        //3. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "OtherForeignIncomeDetails" && reader.Depth == _currentXPath.Count - 1)
                        {
                            otherForeignIncome.Rp_OtherForeignIncome_OtherForeignIncomeDetailsCollectionExists = true;
                            otherForeignIncome.Rp_OtherForeignIncome_OtherForeignIncomeDetailsCollectionCount += 1;
                            _found = true;
#region InternationalDealingsForeignSourceIncomeOtherA
                            _currentXPath.Push("/tns:InternationalDealingsForeignSourceIncomeOtherA");
                            //6. use case
                            if (ReadToNextElement(reader, true) && reader.LocalName == "InternationalDealingsForeignSourceIncomeOtherA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            otherForeignIncome.INCDTLS226 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion InternationalDealingsForeignSourceIncomeOtherA
#region InternationalDealingsTaxOffsetA
                            _currentXPath.Push("/tns:InternationalDealingsTaxOffsetA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsTaxOffsetA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            otherForeignIncome.INCDTLS227 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion InternationalDealingsTaxOffsetA
                        } // End of if OtherForeignIncomeDetails node exists

                        _currentXPath.Pop();
#endregion OtherForeignIncomeDetails
                        _currentXPath.Pop();
                    } // End of while loop for OtherForeignIncome
                } // End of if OtherForeignIncomeCollection node exists

                _currentXPath.Pop();
#endregion OtherForeignIncome
#region IncomeOther
                _currentXPath.Push("/tns:IncomeOtherCollection");
                // 4. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeOtherCollection" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.Rp_IncomeOtherCollection = new List<INCDTLS2026.Rp_IncomeOther>();
                    report.Rp_IncomeOtherCollectionExists = true;
                    _found = true;
                    while (ReadToNextElement(reader, false) && reader.LocalName == "IncomeOther")
                    {
                        _found = true;
                        INCDTLS2026.Rp_IncomeOther incomeOther = new INCDTLS2026.Rp_IncomeOther();
                        report.Rp_IncomeOtherCollection.Add(incomeOther);
                        report.Rp_IncomeOtherCollectionCount += 1;
                        incomeOther.OccurrenceIndex = report.Rp_IncomeOtherCollectionCount;
                        _currentXPath.Push("/tns:IncomeOther[" + report.Rp_IncomeOtherCollectionCount + "]");
#region OtherIncomeTypeC
                        _currentXPath.Push("/tns:OtherIncomeTypeC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "OtherIncomeTypeC" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    incomeOther.INCDTLS449 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion OtherIncomeTypeC
#region De
                        _currentXPath.Push("/tns:De");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "De" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    incomeOther.INCDTLS450 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion De
#region A
                        _currentXPath.Push("/tns:A");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "A" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        incomeOther.INCDTLS451 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion A
#region SpecialProfessionalExpense
                        _currentXPath.Push("/tns:SpecialProfessionalExpense");
                        //3. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "SpecialProfessionalExpense" && reader.Depth == _currentXPath.Count - 1)
                        {
                            incomeOther.Rp_IncomeOther_SpecialProfessionalExpenseCollectionExists = true;
                            incomeOther.Rp_IncomeOther_SpecialProfessionalExpenseCollectionCount += 1;
                            _found = true;
#region A
                            _currentXPath.Push("/tns:A");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "A" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            incomeOther.INCDTLS453 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion A
#region DeductibleOtherTotalA
                            _currentXPath.Push("/tns:DeductibleOtherTotalA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "DeductibleOtherTotalA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            incomeOther.INCDTLS454 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion DeductibleOtherTotalA
#region DeductibleIncludedElsewhereTotalA
                            _currentXPath.Push("/tns:DeductibleIncludedElsewhereTotalA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "DeductibleIncludedElsewhereTotalA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            incomeOther.INCDTLS455 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion DeductibleIncludedElsewhereTotalA
                        } // End of if SpecialProfessionalExpense node exists

                        _currentXPath.Pop();
#endregion SpecialProfessionalExpense
#region IncomeTaxAssessableLowValuePoolDeduction
                        _currentXPath.Push("/tns:IncomeTaxAssessableLowValuePoolDeduction");
                        //3. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxAssessableLowValuePoolDeduction" && reader.Depth == _currentXPath.Count - 1)
                        {
                            incomeOther.Rp_IncomeOther_IncomeTaxAssessableLowValuePoolDeductionCollectionExists = true;
                            incomeOther.Rp_IncomeOther_IncomeTaxAssessableLowValuePoolDeductionCollectionCount += 1;
                            _found = true;
#region DepreciatingAssetsIntangibleBalancingAdjustmentA
                            _currentXPath.Push("/tns:DepreciatingAssetsIntangibleBalancingAdjustmentA");
                            //6. use case
                            if (ReadToNextElement(reader, true) && reader.LocalName == "DepreciatingAssetsIntangibleBalancingAdjustmentA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            incomeOther.INCDTLS457 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion DepreciatingAssetsIntangibleBalancingAdjustmentA
#region DepreciatingAssetsAssessableIncomeFromBalancingAdjustmentEventsTotalA
                            _currentXPath.Push("/tns:DepreciatingAssetsAssessableIncomeFromBalancingAdjustmentEventsTotalA");
                            //6. use case
                            if (ReadToNextElement(reader, true) && reader.LocalName == "DepreciatingAssetsAssessableIncomeFromBalancingAdjustmentEventsTotalA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            incomeOther.INCDTLS458 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion DepreciatingAssetsAssessableIncomeFromBalancingAdjustmentEventsTotalA
#region BalancingAdjustmentTotalA
                            _currentXPath.Push("/tns:BalancingAdjustmentTotalA");
                            //6. use case
                            if (ReadToNextElement(reader, true) && reader.LocalName == "BalancingAdjustmentTotalA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            incomeOther.INCDTLS459 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion BalancingAdjustmentTotalA
                        } // End of if IncomeTaxAssessableLowValuePoolDeduction node exists

                        _currentXPath.Pop();
#endregion IncomeTaxAssessableLowValuePoolDeduction
                        _currentXPath.Pop();
                    } // End of while loop for IncomeOther
                } // End of if IncomeOtherCollection node exists

                _currentXPath.Pop();
#endregion IncomeOther
#region Partnerships
                _currentXPath.Push("/tns:PartnershipsCollection");
                // 4. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "PartnershipsCollection" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.Rp_PartnershipsCollection = new List<INCDTLS2026.Rp_Partnerships>();
                    report.Rp_PartnershipsCollectionExists = true;
                    _found = true;
                    while (ReadToNextElement(reader, false) && reader.LocalName == "Partnerships")
                    {
                        _found = true;
                        INCDTLS2026.Rp_Partnerships partnerships = new INCDTLS2026.Rp_Partnerships();
                        report.Rp_PartnershipsCollection.Add(partnerships);
                        report.Rp_PartnershipsCollectionCount += 1;
                        partnerships.OccurrenceIndex = report.Rp_PartnershipsCollectionCount;
                        _currentXPath.Push("/tns:Partnerships[" + report.Rp_PartnershipsCollectionCount + "]");
#region OrganisationNameDetailsOrganisationalNameT
                        _currentXPath.Push("/tns:OrganisationNameDetailsOrganisationalNameT");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "OrganisationNameDetailsOrganisationalNameT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    partnerships.INCDTLS461 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion OrganisationNameDetailsOrganisationalNameT
#region AustralianBusinessNumberId
                        _currentXPath.Push("/tns:AustralianBusinessNumberId");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "AustralianBusinessNumberId" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    partnerships.INCDTLS462 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion AustralianBusinessNumberId
#region PartnershipsCreditsIncomeTaxOffsets
                        _currentXPath.Push("/tns:PartnershipsCreditsIncomeTaxOffsets");
                        //3. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "PartnershipsCreditsIncomeTaxOffsets" && reader.Depth == _currentXPath.Count - 1)
                        {
                            partnerships.Rp_Partnerships_PartnershipsCreditsIncomeTaxOffsetsCollectionExists = true;
                            partnerships.Rp_Partnerships_PartnershipsCreditsIncomeTaxOffsetsCollectionCount += 1;
                            _found = true;
#region PayAsYouGoWithholdingCreditForTaxWithheldWhereABNNotQuotedA
                            _currentXPath.Push("/tns:PayAsYouGoWithholdingCreditForTaxWithheldWhereABNNotQuotedA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "PayAsYouGoWithholdingCreditForTaxWithheldWhereABNNotQuotedA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            partnerships.INCDTLS480 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion PayAsYouGoWithholdingCreditForTaxWithheldWhereABNNotQuotedA
#region FrankingCreditsShareA
                            _currentXPath.Push("/tns:FrankingCreditsShareA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "FrankingCreditsShareA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            partnerships.INCDTLS481 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion FrankingCreditsShareA
#region PayAsYouGoWithholdingCreditForTaxWithheldInvestmentsTFNNotQuotedShareA
                            _currentXPath.Push("/tns:PayAsYouGoWithholdingCreditForTaxWithheldInvestmentsTFNNotQuotedShareA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "PayAsYouGoWithholdingCreditForTaxWithheldInvestmentsTFNNotQuotedShareA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            partnerships.INCDTLS482 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion PayAsYouGoWithholdingCreditForTaxWithheldInvestmentsTFNNotQuotedShareA
#region PayAsYouGoWithholdingCreditTaxWithheldCloselyHeldTrustShareA
                            _currentXPath.Push("/tns:PayAsYouGoWithholdingCreditTaxWithheldCloselyHeldTrustShareA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "PayAsYouGoWithholdingCreditTaxWithheldCloselyHeldTrustShareA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            partnerships.INCDTLS483 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion PayAsYouGoWithholdingCreditTaxWithheldCloselyHeldTrustShareA
#region PayAsYouGoWithholdingCreditTaxWithheldForeignResidentShareA
                            _currentXPath.Push("/tns:PayAsYouGoWithholdingCreditTaxWithheldForeignResidentShareA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "PayAsYouGoWithholdingCreditTaxWithheldForeignResidentShareA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            partnerships.INCDTLS484 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion PayAsYouGoWithholdingCreditTaxWithheldForeignResidentShareA
#region TaxOffsetClaimNationalRentalAffordabilitySchemeEntitlementShareA
                            _currentXPath.Push("/tns:TaxOffsetClaimNationalRentalAffordabilitySchemeEntitlementShareA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "TaxOffsetClaimNationalRentalAffordabilitySchemeEntitlementShareA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            partnerships.INCDTLS485 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion TaxOffsetClaimNationalRentalAffordabilitySchemeEntitlementShareA
                        } // End of if PartnershipsCreditsIncomeTaxOffsets node exists

                        _currentXPath.Pop();
#endregion PartnershipsCreditsIncomeTaxOffsets
#region PartnershipsPrimary
                        _currentXPath.Push("/tns:PartnershipsPrimary");
                        //3. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "PartnershipsPrimary" && reader.Depth == _currentXPath.Count - 1)
                        {
                            partnerships.Rp_Partnerships_PartnershipsPrimaryCollectionExists = true;
                            partnerships.Rp_Partnerships_PartnershipsPrimaryCollectionCount += 1;
                            _found = true;
#region IncomeTrustsPrimaryProductionNetA
                            _currentXPath.Push("/tns:IncomeTrustsPrimaryProductionNetA");
                            //6. use case
                            if (ReadToNextElement(reader, true) && reader.LocalName == "IncomeTrustsPrimaryProductionNetA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            partnerships.INCDTLS464 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTrustsPrimaryProductionNetA
#region ExpenseLandcareOperationsAcceleratedDepreciationPrimaryProductionA
                            _currentXPath.Push("/tns:ExpenseLandcareOperationsAcceleratedDepreciationPrimaryProductionA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseLandcareOperationsAcceleratedDepreciationPrimaryProductionA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            partnerships.INCDTLS465 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion ExpenseLandcareOperationsAcceleratedDepreciationPrimaryProductionA
#region IncomeTaxDeductionAndTrustPrimaryProductionShareA
                            _currentXPath.Push("/tns:IncomeTaxDeductionAndTrustPrimaryProductionShareA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxDeductionAndTrustPrimaryProductionShareA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            partnerships.INCDTLS466 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxDeductionAndTrustPrimaryProductionShareA
#region IncomeTaxLossesNonCommercialTypeC
                            _currentXPath.Push("/tns:IncomeTaxLossesNonCommercialTypeC");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxLossesNonCommercialTypeC" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        partnerships.INCDTLS467 = currentValue;
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxLossesNonCommercialTypeC
                        } // End of if PartnershipsPrimary node exists

                        _currentXPath.Pop();
#endregion PartnershipsPrimary
#region PartnershipsNonPrimary
                        _currentXPath.Push("/tns:PartnershipsNonPrimary");
                        //3. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "PartnershipsNonPrimary" && reader.Depth == _currentXPath.Count - 1)
                        {
                            partnerships.Rp_Partnerships_PartnershipsNonPrimaryCollectionExists = true;
                            partnerships.Rp_Partnerships_PartnershipsNonPrimaryCollectionCount += 1;
                            _found = true;
#region PartnershipsNonPrimaryDistribution
                            _currentXPath.Push("/tns:PartnershipsNonPrimaryDistribution");
                            //3. use case
                            if (ReadToNextElement(reader, true) && reader.LocalName == "PartnershipsNonPrimaryDistribution" && reader.Depth == _currentXPath.Count - 1)
                            {
                                partnerships.Rp_Partnerships_PartnershipsNonPrimary_PartnershipsNonPrimaryDistributionCollectionExists = true;
                                partnerships.Rp_Partnerships_PartnershipsNonPrimary_PartnershipsNonPrimaryDistributionCollectionCount += 1;
                                _found = true;
#region IncomePartnershipInvestmentLessForeignIncomeNonPrimaryProductionShareA
                                _currentXPath.Push("/tns:IncomePartnershipInvestmentLessForeignIncomeNonPrimaryProductionShareA");
                                //6. use case
                                if (ReadToNextElement(reader, true) && reader.LocalName == "IncomePartnershipInvestmentLessForeignIncomeNonPrimaryProductionShareA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                partnerships.INCDTLS470 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion IncomePartnershipInvestmentLessForeignIncomeNonPrimaryProductionShareA
#region IncomePartnershipRentalNonPrimaryProductionShareA
                                _currentXPath.Push("/tns:IncomePartnershipRentalNonPrimaryProductionShareA");
                                //6. use case
                                if (ReadToNextElement(reader, true) && reader.LocalName == "IncomePartnershipRentalNonPrimaryProductionShareA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                partnerships.INCDTLS471 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion IncomePartnershipRentalNonPrimaryProductionShareA
#region IncomePartnershipNonPrimaryProductionOtherShareA
                                _currentXPath.Push("/tns:IncomePartnershipNonPrimaryProductionOtherShareA");
                                //6. use case
                                if (ReadToNextElement(reader, true) && reader.LocalName == "IncomePartnershipNonPrimaryProductionOtherShareA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                partnerships.INCDTLS472 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion IncomePartnershipNonPrimaryProductionOtherShareA
                            } // End of if PartnershipsNonPrimaryDistribution node exists

                            _currentXPath.Pop();
#endregion PartnershipsNonPrimaryDistribution
#region PartnershipsNonPrimaryDeduction
                            _currentXPath.Push("/tns:PartnershipsNonPrimaryDeduction");
                            //3. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "PartnershipsNonPrimaryDeduction" && reader.Depth == _currentXPath.Count - 1)
                            {
                                partnerships.Rp_Partnerships_PartnershipsNonPrimary_PartnershipsNonPrimaryDeductionCollectionExists = true;
                                partnerships.Rp_Partnerships_PartnershipsNonPrimary_PartnershipsNonPrimaryDeductionCollectionCount += 1;
                                _found = true;
#region ExpenseLandcareOperationsNonPrimaryProductionA
                                _currentXPath.Push("/tns:ExpenseLandcareOperationsNonPrimaryProductionA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseLandcareOperationsNonPrimaryProductionA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                partnerships.INCDTLS473 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion ExpenseLandcareOperationsNonPrimaryProductionA
#region IncomeTaxLossesNonCommercialTypeC
                                _currentXPath.Push("/tns:IncomeTaxLossesNonCommercialTypeC");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxLossesNonCommercialTypeC" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            partnerships.INCDTLS475 = currentValue;
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion IncomeTaxLossesNonCommercialTypeC
#region IncomeTaxPartnershipExcludingForeignIncomeNonPrimaryProductionA
                                _currentXPath.Push("/tns:IncomeTaxPartnershipExcludingForeignIncomeNonPrimaryProductionA");
                                //6. use case
                                if (ReadToNextElement(reader, true) && reader.LocalName == "IncomeTaxPartnershipExcludingForeignIncomeNonPrimaryProductionA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                partnerships.INCDTLS476 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion IncomeTaxPartnershipExcludingForeignIncomeNonPrimaryProductionA
#region IncomeTaxPartnershipRentalNonPrimaryProductionA
                                _currentXPath.Push("/tns:IncomeTaxPartnershipRentalNonPrimaryProductionA");
                                //6. use case
                                if (ReadToNextElement(reader, true) && reader.LocalName == "IncomeTaxPartnershipRentalNonPrimaryProductionA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                partnerships.INCDTLS477 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion IncomeTaxPartnershipRentalNonPrimaryProductionA
#region IncomeTaxTrustPartnershipOtherNonPrimaryProductionNetA
                                _currentXPath.Push("/tns:IncomeTaxTrustPartnershipOtherNonPrimaryProductionNetA");
                                //6. use case
                                if (ReadToNextElement(reader, true) && reader.LocalName == "IncomeTaxTrustPartnershipOtherNonPrimaryProductionNetA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                partnerships.INCDTLS478 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion IncomeTaxTrustPartnershipOtherNonPrimaryProductionNetA
                            } // End of if PartnershipsNonPrimaryDeduction node exists

                            _currentXPath.Pop();
#endregion PartnershipsNonPrimaryDeduction
                        } // End of if PartnershipsNonPrimary node exists

                        _currentXPath.Pop();
#endregion PartnershipsNonPrimary
                        _currentXPath.Pop();
                    } // End of while loop for Partnerships
                } // End of if PartnershipsCollection node exists

                _currentXPath.Pop();
#endregion Partnerships
#region TrustIncome
                _currentXPath.Push("/tns:TrustIncomeCollection");
                // 4. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "TrustIncomeCollection" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.Rp_TrustIncomeCollection = new List<INCDTLS2026.Rp_TrustIncome>();
                    report.Rp_TrustIncomeCollectionExists = true;
                    _found = true;
                    while (ReadToNextElement(reader, false) && reader.LocalName == "TrustIncome")
                    {
                        _found = true;
                        INCDTLS2026.Rp_TrustIncome trustIncome = new INCDTLS2026.Rp_TrustIncome();
                        report.Rp_TrustIncomeCollection.Add(trustIncome);
                        report.Rp_TrustIncomeCollectionCount += 1;
                        trustIncome.OccurrenceIndex = report.Rp_TrustIncomeCollectionCount;
                        _currentXPath.Push("/tns:TrustIncome[" + report.Rp_TrustIncomeCollectionCount + "]");
#region DistributingTrust
                        _currentXPath.Push("/tns:DistributingTrust");
                        //3. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "DistributingTrust" && reader.Depth == _currentXPath.Count - 1)
                        {
                            trustIncome.Rp_TrustIncome_DistributingTrustCollectionExists = true;
                            trustIncome.Rp_TrustIncome_DistributingTrustCollectionCount += 1;
                            _found = true;
#region AustralianBusinessNumberId
                            _currentXPath.Push("/tns:AustralianBusinessNumberId");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "AustralianBusinessNumberId" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        trustIncome.INCDTLS537 = currentValue;
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion AustralianBusinessNumberId
#region AustralianCompanyNumberId
                            _currentXPath.Push("/tns:AustralianCompanyNumberId");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "AustralianCompanyNumberId" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        trustIncome.INCDTLS538 = currentValue;
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion AustralianCompanyNumberId
#region OrganisationNameDetailsOrganisationalNameT
                            _currentXPath.Push("/tns:OrganisationNameDetailsOrganisationalNameT");
                            //6. use case
                            if (ReadToNextElement(reader, true) && reader.LocalName == "OrganisationNameDetailsOrganisationalNameT" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        trustIncome.INCDTLS539 = currentValue;
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion OrganisationNameDetailsOrganisationalNameT
                        } // End of if DistributingTrust node exists

                        _currentXPath.Pop();
#endregion DistributingTrust
#region TrustPrimaryProduction
                        _currentXPath.Push("/tns:TrustPrimaryProduction");
                        //3. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "TrustPrimaryProduction" && reader.Depth == _currentXPath.Count - 1)
                        {
                            trustIncome.Rp_TrustIncome_TrustPrimaryProductionCollectionExists = true;
                            trustIncome.Rp_TrustIncome_TrustPrimaryProductionCollectionCount += 1;
                            _found = true;
#region NetShareA
                            _currentXPath.Push("/tns:NetShareA");
                            //6. use case
                            if (ReadToNextElement(reader, true) && reader.LocalName == "NetShareA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            trustIncome.INCDTLS247 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion NetShareA
#region IncomeTaxDeductionPartnershipsAndTrustPrimaryProductionShareA
                            _currentXPath.Push("/tns:IncomeTaxDeductionPartnershipsAndTrustPrimaryProductionShareA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxDeductionPartnershipsAndTrustPrimaryProductionShareA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            trustIncome.INCDTLS248 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxDeductionPartnershipsAndTrustPrimaryProductionShareA
#region NonConcessionalManagedInvestmentTrustA
                            _currentXPath.Push("/tns:NonConcessionalManagedInvestmentTrustA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "NonConcessionalManagedInvestmentTrustA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            trustIncome.INCDTLS540 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion NonConcessionalManagedInvestmentTrustA
#region ExcludedFromNonConcessionalManagedInvestmentTrustA
                            _currentXPath.Push("/tns:ExcludedFromNonConcessionalManagedInvestmentTrustA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "ExcludedFromNonConcessionalManagedInvestmentTrustA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            trustIncome.INCDTLS541 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion ExcludedFromNonConcessionalManagedInvestmentTrustA
                        } // End of if TrustPrimaryProduction node exists

                        _currentXPath.Pop();
#endregion TrustPrimaryProduction
#region TrustNonPrimaryProduction
                        _currentXPath.Push("/tns:TrustNonPrimaryProduction");
                        //3. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "TrustNonPrimaryProduction" && reader.Depth == _currentXPath.Count - 1)
                        {
                            trustIncome.Rp_TrustIncome_TrustNonPrimaryProductionCollectionExists = true;
                            trustIncome.Rp_TrustIncome_TrustNonPrimaryProductionCollectionCount += 1;
                            _found = true;
#region TrustLessNetCapitalGainAndForeignIncomeNonPrimaryProductionShareA
                            _currentXPath.Push("/tns:TrustLessNetCapitalGainAndForeignIncomeNonPrimaryProductionShareA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "TrustLessNetCapitalGainAndForeignIncomeNonPrimaryProductionShareA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            trustIncome.INCDTLS250 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion TrustLessNetCapitalGainAndForeignIncomeNonPrimaryProductionShareA
#region TrustManagedInvestmentSchemeLessNetCapitalGainAndForeignIncomeNonPrimaryProductionShareA
                            _currentXPath.Push("/tns:TrustManagedInvestmentSchemeLessNetCapitalGainAndForeignIncomeNonPrimaryProductionShareA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "TrustManagedInvestmentSchemeLessNetCapitalGainAndForeignIncomeNonPrimaryProductionShareA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            trustIncome.INCDTLS251 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion TrustManagedInvestmentSchemeLessNetCapitalGainAndForeignIncomeNonPrimaryProductionShareA
#region TrustOtherNonPrimaryProductionShareA
                            _currentXPath.Push("/tns:TrustOtherNonPrimaryProductionShareA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "TrustOtherNonPrimaryProductionShareA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            trustIncome.INCDTLS252 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion TrustOtherNonPrimaryProductionShareA
#region TrustFrankedInvestmentNonPrimaryProductionShareA
                            _currentXPath.Push("/tns:TrustFrankedInvestmentNonPrimaryProductionShareA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "TrustFrankedInvestmentNonPrimaryProductionShareA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            trustIncome.INCDTLS254 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion TrustFrankedInvestmentNonPrimaryProductionShareA
#region TrustFrankedNonPrimaryProductionOtherShareA
                            _currentXPath.Push("/tns:TrustFrankedNonPrimaryProductionOtherShareA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "TrustFrankedNonPrimaryProductionOtherShareA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            trustIncome.INCDTLS255 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion TrustFrankedNonPrimaryProductionOtherShareA
#region IncomeTaxDeductionTrustExcludingCapitalGainsAndForeignIncomeNonPrimaryProductionA
                            _currentXPath.Push("/tns:IncomeTaxDeductionTrustExcludingCapitalGainsAndForeignIncomeNonPrimaryProductionA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxDeductionTrustExcludingCapitalGainsAndForeignIncomeNonPrimaryProductionA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            trustIncome.INCDTLS257 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxDeductionTrustExcludingCapitalGainsAndForeignIncomeNonPrimaryProductionA
#region IncomeTaxDeductionTrustsPartnershipsOtherAllowableNonPrimaryProductionA
                            _currentXPath.Push("/tns:IncomeTaxDeductionTrustsPartnershipsOtherAllowableNonPrimaryProductionA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxDeductionTrustsPartnershipsOtherAllowableNonPrimaryProductionA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            trustIncome.INCDTLS258 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxDeductionTrustsPartnershipsOtherAllowableNonPrimaryProductionA
#region NonConcessionalManagedInvestmentTrustA
                            _currentXPath.Push("/tns:NonConcessionalManagedInvestmentTrustA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "NonConcessionalManagedInvestmentTrustA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            trustIncome.INCDTLS542 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion NonConcessionalManagedInvestmentTrustA
#region ExcludedFromNonConcessionalManagedInvestmentTrustA
                            _currentXPath.Push("/tns:ExcludedFromNonConcessionalManagedInvestmentTrustA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "ExcludedFromNonConcessionalManagedInvestmentTrustA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            trustIncome.INCDTLS543 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion ExcludedFromNonConcessionalManagedInvestmentTrustA
                        } // End of if TrustNonPrimaryProduction node exists

                        _currentXPath.Pop();
#endregion TrustNonPrimaryProduction
#region TrustCreditsIncomeTaxOffsets
                        _currentXPath.Push("/tns:TrustCreditsIncomeTaxOffsets");
                        //3. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "TrustCreditsIncomeTaxOffsets" && reader.Depth == _currentXPath.Count - 1)
                        {
                            trustIncome.Rp_TrustIncome_TrustCreditsIncomeTaxOffsetsCollectionExists = true;
                            trustIncome.Rp_TrustIncome_TrustCreditsIncomeTaxOffsetsCollectionCount += 1;
                            _found = true;
#region PayAsYouGoWithholdingCreditForTaxWithheldWhereABNNotQuotedA
                            _currentXPath.Push("/tns:PayAsYouGoWithholdingCreditForTaxWithheldWhereABNNotQuotedA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "PayAsYouGoWithholdingCreditForTaxWithheldWhereABNNotQuotedA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            trustIncome.INCDTLS260 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion PayAsYouGoWithholdingCreditForTaxWithheldWhereABNNotQuotedA
#region TaxableMinorBeneficiaryA
                            _currentXPath.Push("/tns:TaxableMinorBeneficiaryA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "TaxableMinorBeneficiaryA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            trustIncome.INCDTLS544 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion TaxableMinorBeneficiaryA
#region FrankingCreditsShareA
                            _currentXPath.Push("/tns:FrankingCreditsShareA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "FrankingCreditsShareA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            trustIncome.INCDTLS261 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion FrankingCreditsShareA
#region PayAsYouGoWithholdingCreditForTaxWithheldInvestmentsTFNNotQuotedShareA
                            _currentXPath.Push("/tns:PayAsYouGoWithholdingCreditForTaxWithheldInvestmentsTFNNotQuotedShareA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "PayAsYouGoWithholdingCreditForTaxWithheldInvestmentsTFNNotQuotedShareA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            trustIncome.INCDTLS262 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion PayAsYouGoWithholdingCreditForTaxWithheldInvestmentsTFNNotQuotedShareA
#region PayAsYouGoWithholdingCreditTaxWithheldCloselyHeldTrustShareA
                            _currentXPath.Push("/tns:PayAsYouGoWithholdingCreditTaxWithheldCloselyHeldTrustShareA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "PayAsYouGoWithholdingCreditTaxWithheldCloselyHeldTrustShareA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            trustIncome.INCDTLS263 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion PayAsYouGoWithholdingCreditTaxWithheldCloselyHeldTrustShareA
#region CreditTrusteeTaxPaidTotalA
                            _currentXPath.Push("/tns:CreditTrusteeTaxPaidTotalA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "CreditTrusteeTaxPaidTotalA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            trustIncome.INCDTLS264 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion CreditTrusteeTaxPaidTotalA
#region CreditTrusteeTaxPaidReasonC
                            _currentXPath.Push("/tns:CreditTrusteeTaxPaidReasonC");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "CreditTrusteeTaxPaidReasonC" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        trustIncome.INCDTLS265 = currentValue;
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion CreditTrusteeTaxPaidReasonC
#region TrustDistributionsA
                            _currentXPath.Push("/tns:TrustDistributionsA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "TrustDistributionsA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            trustIncome.INCDTLS266 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion TrustDistributionsA
#region CreditTrusteeTaxPaidCreditsA
                            _currentXPath.Push("/tns:CreditTrusteeTaxPaidCreditsA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "CreditTrusteeTaxPaidCreditsA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            trustIncome.INCDTLS267 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion CreditTrusteeTaxPaidCreditsA
#region FrankingCreditsReceivedFromNewZealandCompaniesA
                            _currentXPath.Push("/tns:FrankingCreditsReceivedFromNewZealandCompaniesA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "FrankingCreditsReceivedFromNewZealandCompaniesA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            trustIncome.INCDTLS546 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion FrankingCreditsReceivedFromNewZealandCompaniesA
#region TaxOffsetClaimNationalRentalAffordabilitySchemeEntitlementShareA
                            _currentXPath.Push("/tns:TaxOffsetClaimNationalRentalAffordabilitySchemeEntitlementShareA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "TaxOffsetClaimNationalRentalAffordabilitySchemeEntitlementShareA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            trustIncome.INCDTLS269 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion TaxOffsetClaimNationalRentalAffordabilitySchemeEntitlementShareA
#region ExplorationCreditsDistributedA
                            _currentXPath.Push("/tns:ExplorationCreditsDistributedA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "ExplorationCreditsDistributedA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            trustIncome.INCDTLS547 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion ExplorationCreditsDistributedA
#region SmallBusinessEntityNetA
                            _currentXPath.Push("/tns:SmallBusinessEntityNetA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "SmallBusinessEntityNetA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            trustIncome.INCDTLS548 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion SmallBusinessEntityNetA
#region EarlyStageVentureCapitalLimitedPartnershipTaxOffsetA
                            _currentXPath.Push("/tns:EarlyStageVentureCapitalLimitedPartnershipTaxOffsetA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "EarlyStageVentureCapitalLimitedPartnershipTaxOffsetA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            trustIncome.INCDTLS549 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion EarlyStageVentureCapitalLimitedPartnershipTaxOffsetA
#region EarlyStageInvestorTaxOffsetA
                            _currentXPath.Push("/tns:EarlyStageInvestorTaxOffsetA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "EarlyStageInvestorTaxOffsetA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            trustIncome.INCDTLS550 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion EarlyStageInvestorTaxOffsetA
                        } // End of if TrustCreditsIncomeTaxOffsets node exists

                        _currentXPath.Pop();
#endregion TrustCreditsIncomeTaxOffsets
#region TrustCapitalGains
                        _currentXPath.Push("/tns:TrustCapitalGains");
                        //3. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "TrustCapitalGains" && reader.Depth == _currentXPath.Count - 1)
                        {
                            trustIncome.Rp_TrustIncome_TrustCapitalGainsCollectionExists = true;
                            trustIncome.Rp_TrustIncome_TrustCapitalGainsCollectionCount += 1;
                            _found = true;
#region CapitalGainsNetA
                            _currentXPath.Push("/tns:CapitalGainsNetA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "CapitalGainsNetA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            trustIncome.INCDTLS552 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion CapitalGainsNetA
#region TotalA
                            _currentXPath.Push("/tns:TotalA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "TotalA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            trustIncome.INCDTLS553 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion TotalA
#region LossesTotalA
                            _currentXPath.Push("/tns:LossesTotalA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "LossesTotalA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            trustIncome.INCDTLS554 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion LossesTotalA
#region TaxConcessionDiscountTotalA
                            _currentXPath.Push("/tns:TaxConcessionDiscountTotalA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "TaxConcessionDiscountTotalA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            trustIncome.INCDTLS555 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion TaxConcessionDiscountTotalA
#region TaxConcessionDiscountAndSmallBusinessConcessionsTotalA
                            _currentXPath.Push("/tns:TaxConcessionDiscountAndSmallBusinessConcessionsTotalA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "TaxConcessionDiscountAndSmallBusinessConcessionsTotalA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            trustIncome.INCDTLS556 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion TaxConcessionDiscountAndSmallBusinessConcessionsTotalA
#region NonConcessionalManagedInvestmentTrustIncomeA
                            _currentXPath.Push("/tns:NonConcessionalManagedInvestmentTrustIncomeA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "NonConcessionalManagedInvestmentTrustIncomeA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            trustIncome.INCDTLS557 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion NonConcessionalManagedInvestmentTrustIncomeA
#region ExcludedFromNonConcessionalManagedInvestmentTrustIncomeA
                            _currentXPath.Push("/tns:ExcludedFromNonConcessionalManagedInvestmentTrustIncomeA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "ExcludedFromNonConcessionalManagedInvestmentTrustIncomeA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            trustIncome.INCDTLS558 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion ExcludedFromNonConcessionalManagedInvestmentTrustIncomeA
                        } // End of if TrustCapitalGains node exists

                        _currentXPath.Pop();
#endregion TrustCapitalGains
#region ForeignAndNonResident
                        _currentXPath.Push("/tns:ForeignAndNonResident");
                        //3. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ForeignAndNonResident" && reader.Depth == _currentXPath.Count - 1)
                        {
                            trustIncome.Rp_TrustIncome_ForeignAndNonResidentCollectionExists = true;
                            trustIncome.Rp_TrustIncome_ForeignAndNonResidentCollectionCount += 1;
                            _found = true;
#region IncomeTaxPayAsYouGoWithholdingCreditTaxWithheldForeignResidentShareA
                            _currentXPath.Push("/tns:IncomeTaxPayAsYouGoWithholdingCreditTaxWithheldForeignResidentShareA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxPayAsYouGoWithholdingCreditTaxWithheldForeignResidentShareA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            trustIncome.INCDTLS268 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxPayAsYouGoWithholdingCreditTaxWithheldForeignResidentShareA
#region IncomeTaxPayAsYouGoWithholdingCreditForCapitalGainsWithheldFromForeignResidentsA
                            _currentXPath.Push("/tns:IncomeTaxPayAsYouGoWithholdingCreditForCapitalGainsWithheldFromForeignResidentsA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxPayAsYouGoWithholdingCreditForCapitalGainsWithheldFromForeignResidentsA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            trustIncome.INCDTLS561 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxPayAsYouGoWithholdingCreditForCapitalGainsWithheldFromForeignResidentsA
#region IncomeTaxAttributedForeignIncomeA
                            _currentXPath.Push("/tns:IncomeTaxAttributedForeignIncomeA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxAttributedForeignIncomeA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            trustIncome.INCDTLS562 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxAttributedForeignIncomeA
#region InternationalDealingsNetA
                            _currentXPath.Push("/tns:InternationalDealingsNetA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsNetA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            trustIncome.INCDTLS563 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion InternationalDealingsNetA
#region InternationalDealingsTaxOffsetA
                            _currentXPath.Push("/tns:InternationalDealingsTaxOffsetA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsTaxOffsetA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            trustIncome.INCDTLS564 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion InternationalDealingsTaxOffsetA
#region InternationalDealingsAssessableIncomeNonResidentBeneficiaryNonTrusteeA
                            _currentXPath.Push("/tns:InternationalDealingsAssessableIncomeNonResidentBeneficiaryNonTrusteeA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsAssessableIncomeNonResidentBeneficiaryNonTrusteeA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            trustIncome.INCDTLS565 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion InternationalDealingsAssessableIncomeNonResidentBeneficiaryNonTrusteeA
#region InternationalDealingsAssessableIncomeNonResidentBeneficiaryTrusteeA
                            _currentXPath.Push("/tns:InternationalDealingsAssessableIncomeNonResidentBeneficiaryTrusteeA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsAssessableIncomeNonResidentBeneficiaryTrusteeA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            trustIncome.INCDTLS566 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion InternationalDealingsAssessableIncomeNonResidentBeneficiaryTrusteeA
                        } // End of if ForeignAndNonResident node exists

                        _currentXPath.Pop();
#endregion ForeignAndNonResident
                        _currentXPath.Pop();
                    } // End of while loop for TrustIncome
                } // End of if TrustIncomeCollection node exists

                _currentXPath.Pop();
#endregion TrustIncome
#region BusinessIncomePaymentSummaries
                _currentXPath.Push("/tns:BusinessIncomePaymentSummariesCollection");
                // 4. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "BusinessIncomePaymentSummariesCollection" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.Rp_BusinessIncomePaymentSummariesCollection = new List<INCDTLS2026.Rp_BusinessIncomePaymentSummaries>();
                    report.Rp_BusinessIncomePaymentSummariesCollectionExists = true;
                    _found = true;
                    while (ReadToNextElement(reader, false) && reader.LocalName == "BusinessIncomePaymentSummaries")
                    {
                        _found = true;
                        INCDTLS2026.Rp_BusinessIncomePaymentSummaries businessIncomePaymentSummaries = new INCDTLS2026.Rp_BusinessIncomePaymentSummaries();
                        report.Rp_BusinessIncomePaymentSummariesCollection.Add(businessIncomePaymentSummaries);
                        report.Rp_BusinessIncomePaymentSummariesCollectionCount += 1;
                        businessIncomePaymentSummaries.OccurrenceIndex = report.Rp_BusinessIncomePaymentSummariesCollectionCount;
                        _currentXPath.Push("/tns:BusinessIncomePaymentSummaries[" + report.Rp_BusinessIncomePaymentSummariesCollectionCount + "]");
#region OrganisationNameDetailsOrganisationalNameT
                        _currentXPath.Push("/tns:OrganisationNameDetailsOrganisationalNameT");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "OrganisationNameDetailsOrganisationalNameT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    businessIncomePaymentSummaries.INCDTLS487 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion OrganisationNameDetailsOrganisationalNameT
#region AustralianBusinessNumberId
                        _currentXPath.Push("/tns:AustralianBusinessNumberId");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "AustralianBusinessNumberId" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    businessIncomePaymentSummaries.INCDTLS488 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion AustralianBusinessNumberId
#region RemunerationWithholdingPaymentIncomeTypeC
                        _currentXPath.Push("/tns:RemunerationWithholdingPaymentIncomeTypeC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "RemunerationWithholdingPaymentIncomeTypeC" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    businessIncomePaymentSummaries.INCDTLS489 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion RemunerationWithholdingPaymentIncomeTypeC
#region IncomeTaxIncomeTypeC
                        _currentXPath.Push("/tns:IncomeTaxIncomeTypeC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "IncomeTaxIncomeTypeC" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    businessIncomePaymentSummaries.INCDTLS490 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion IncomeTaxIncomeTypeC
#region IndustryProductionTypeC
                        _currentXPath.Push("/tns:IndustryProductionTypeC");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "IndustryProductionTypeC" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    businessIncomePaymentSummaries.INCDTLS491 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion IndustryProductionTypeC
#region BusinessGrossTotalA
                        _currentXPath.Push("/tns:BusinessGrossTotalA");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "BusinessGrossTotalA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        businessIncomePaymentSummaries.INCDTLS492 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion BusinessGrossTotalA
#region IncomeTaxPayAsYouGoWithholdingCreditTaxWithheldA
                        _currentXPath.Push("/tns:IncomeTaxPayAsYouGoWithholdingCreditTaxWithheldA");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "IncomeTaxPayAsYouGoWithholdingCreditTaxWithheldA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        businessIncomePaymentSummaries.INCDTLS493 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion IncomeTaxPayAsYouGoWithholdingCreditTaxWithheldA
#region SuperannuationContributionEmployerContributionsA
                        _currentXPath.Push("/tns:SuperannuationContributionEmployerContributionsA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "SuperannuationContributionEmployerContributionsA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        businessIncomePaymentSummaries.INCDTLS494 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion SuperannuationContributionEmployerContributionsA
                        _currentXPath.Pop();
                    } // End of while loop for BusinessIncomePaymentSummaries
                } // End of if BusinessIncomePaymentSummariesCollection node exists

                _currentXPath.Pop();
#endregion BusinessIncomePaymentSummaries
#region TaxablePayment
                _currentXPath.Push("/tns:TaxablePaymentCollection");
                // 4. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "TaxablePaymentCollection" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.Rp_TaxablePaymentCollection = new List<INCDTLS2026.Rp_TaxablePayment>();
                    report.Rp_TaxablePaymentCollectionExists = true;
                    _found = true;
                    while (ReadToNextElement(reader, false) && reader.LocalName == "TaxablePayment")
                    {
                        _found = true;
                        INCDTLS2026.Rp_TaxablePayment taxablePayment = new INCDTLS2026.Rp_TaxablePayment();
                        report.Rp_TaxablePaymentCollection.Add(taxablePayment);
                        report.Rp_TaxablePaymentCollectionCount += 1;
                        taxablePayment.OccurrenceIndex = report.Rp_TaxablePaymentCollectionCount;
                        _currentXPath.Push("/tns:TaxablePayment[" + report.Rp_TaxablePaymentCollectionCount + "]");
#region IncomeAdjustmentReasonC
                        _currentXPath.Push("/tns:IncomeAdjustmentReasonC");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeAdjustmentReasonC" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    taxablePayment.INCDTLS571 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion IncomeAdjustmentReasonC
#region IncomeAdjustmentReasonDe
                        _currentXPath.Push("/tns:IncomeAdjustmentReasonDe");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeAdjustmentReasonDe" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    taxablePayment.INCDTLS572 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion IncomeAdjustmentReasonDe
#region AustralianBusinessNumberId
                        _currentXPath.Push("/tns:AustralianBusinessNumberId");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "AustralianBusinessNumberId" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    taxablePayment.INCDTLS573 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion AustralianBusinessNumberId
#region OrganisationNameDetailsOrganisationalNameT
                        _currentXPath.Push("/tns:OrganisationNameDetailsOrganisationalNameT");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "OrganisationNameDetailsOrganisationalNameT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    taxablePayment.INCDTLS574 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion OrganisationNameDetailsOrganisationalNameT
#region IncomeGrossA
                        _currentXPath.Push("/tns:IncomeGrossA");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "IncomeGrossA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        taxablePayment.INCDTLS575 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion IncomeGrossA
#region IncomeTaxPayAsYouGoWithholdingTaxWithheldA
                        _currentXPath.Push("/tns:IncomeTaxPayAsYouGoWithholdingTaxWithheldA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxPayAsYouGoWithholdingTaxWithheldA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        taxablePayment.INCDTLS576 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion IncomeTaxPayAsYouGoWithholdingTaxWithheldA
#region GoodsAndServicesTaxLiabilityA
                        _currentXPath.Push("/tns:GoodsAndServicesTaxLiabilityA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "GoodsAndServicesTaxLiabilityA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        taxablePayment.INCDTLS577 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion GoodsAndServicesTaxLiabilityA
#region GovernmentFundingGrantPaymentD
                        _currentXPath.Push("/tns:GovernmentFundingGrantPaymentD");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "GovernmentFundingGrantPaymentD" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                                    {
                                        taxablePayment.INCDTLS582 = currentDateTimeValue.Date;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion GovernmentFundingGrantPaymentD
#region GovernmentFundingGrantProgramNameT
                        _currentXPath.Push("/tns:GovernmentFundingGrantProgramNameT");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "GovernmentFundingGrantProgramNameT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    taxablePayment.INCDTLS578 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion GovernmentFundingGrantProgramNameT
#region IncomeTaxPaymentTypeC
                        _currentXPath.Push("/tns:IncomeTaxPaymentTypeC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "IncomeTaxPaymentTypeC" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    taxablePayment.INCDTLS579 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion IncomeTaxPaymentTypeC
#region IncomeIndustryProductionTypeC
                        _currentXPath.Push("/tns:IncomeIndustryProductionTypeC");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeIndustryProductionTypeC" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    taxablePayment.INCDTLS580 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion IncomeIndustryProductionTypeC
#region IncomeClassificationDecisionC
                        _currentXPath.Push("/tns:IncomeClassificationDecisionC");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeClassificationDecisionC" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    taxablePayment.INCDTLS581 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion IncomeClassificationDecisionC
                        _currentXPath.Pop();
                    } // End of while loop for TaxablePayment
                } // End of if TaxablePaymentCollection node exists

                _currentXPath.Pop();
#endregion TaxablePayment
            } // End of if Rp node exists

            _currentXPath.Pop();
#endregion Rp
            while (reader.EOF != true)
                reader.Read();
            return report;
        }

        private string _elementCsvList = "Rp#LodgmentPeriodStartD,Rp#LodgmentPeriodEndD,Rp#TaxFileNumberId,Income#OrganisationalNameT,Income#AustralianBusinessNumberId,Income#WithholdingPayerNumberId,Income#RemunerationWorkingHolidayMakerIncomeI,Income#TaxPayAsYouGoWithholdingCreditTaxWithheldSalaryOrWagesA,Income#A,Income#AllowancesEarningsTipsDirectorsFeesA,Income#TaxPayAsYouGoWithholdingCreditTaxWithheldAllowancesEarningsTipsDirectorsFeesA,Income#FringeBenefitsReportableA,Income#FringeBenefitsTaxExemptionI,Income#CommunityDevelopmentEmploymentProjectA,Income#SuperannuationContributionEmployerReportableA,Income#TaxPayAsYouGoWithholdingCreditTaxWithheldUnusedAnnualOrLongServiceLeavePaymentLumpSumAA,Income#UnusedAnnualOrLongServiceLeavePaymentLumpSumAA,Income#RemunerationUnusedAnnualOrLongServiceLeavePaymentLumpSumAC,Income#TaxPayAsYouGoWithholdingCreditTaxWithheldUnusedAnnualOrLongServiceLeavePaymentLumpSumBA,Income#UnusedAnnualOrLongServiceLeavePaymentLumpSumBA,Income#GenuineRedundancyPaymentLumpSumDA,Income#BackPaymentLumpSumEA,Income#InternationalDealingsIncomeA,Income#TaxForeignWithholdingA,Income#AddressDetailsCountryC,Income#LumpSumArrearsPaymentA,ExemptForeignEmployment,LumpSumArrears#PaymentTypeC,LumpSumArrearsPayment#TargetFinancialY,LumpSumArrearsPayment#A,LumpSumArrearsPayment,LumpSumArrearsPaymentCollection,LumpSumArrears,LumpSumArrearsCollection,SalaryOrWages,Income#TaxPayAsYouGoWithholdingCreditTaxWithheldForeignEmploymentIncomeA,Income#InternationalDealingsForeignIncomeGrossA,Income#InternationalDealingsTaxOffsetA,Income#FringeBenefitsReportableA,Income#FringeBenefitsTaxExemptionI,Income#SuperannuationContributionEmployerReportableA,Income#UnusedAnnualOrLongServiceLeavePaymentLumpSumAA,Income#RemunerationUnusedAnnualOrLongServiceLeavePaymentLumpSumAC,Income#GenuineRedundancyPaymentLumpSumDA,Income#BackPaymentLumpSumEA,Income#InternationalDealingsIncomePaymentSummaryA,FEIPSLumpSumArrearsPayment#TargetFinancialY,FEIPSLumpSumArrearsPayment#A,FEIPSLumpSumArrearsPayment,FEIPSLumpSumArrearsPaymentCollection,ForeignEmployment,Income#AddressDetailsCountryC,Income#D,Income#TaxPayAsYouGoWithholdingTypeC,Income#TaxPayAsYouGoWithholdingCreditTaxWithheldEmploymentTerminationPaymentA,Income#TaxableA,Income#TaxFreeA,EmploymentTerminationPayment,Income#ReversionaryIncomeStreamI,Income#PaymentPeriodStartD,Income#PaymentPeriodEndD,Income#TaxPayAsYouGoWithholdingCreditTaxWithheldAustralianAnnuitiesAndSuperannuationIncomeStreamA,Income#TaxableComponentTaxedElementA,Income#TaxableComponentUntaxedElementA,Income#TaxTaxFreeComponentA,Income#TaxOffsetClaimSuperannuationIncomeStreamA,Income#LumpSumArrearsTaxableComponentTaxedElementA,Income#LumpSumArrearsTaxableComponentUntaxedElementA,Income#LumpSumArrearsTaxFreeA,SISLumpSumArrearsPayment#TargetFinancialY,SISLumpSumArrearsPayment#A,SISLumpSumArrearsPayment,SISLumpSumArrearsPaymentCollection,AnnuitiesAndSuperannuation,Income#TaxPayAsYouGoWithholdingCreditTaxWithheldAustralianAnnuitiesAndSuperannuationIncomeStreamA,Income#InsuranceLifePremiumDirectGrossA,Income#TaxDeductionPensionOrAnnuityPurchasePriceUndeductedA,AustralianAnnuities,Income#D,Income#TaxPayAsYouGoWithholdingCreditTaxWithheldSuperannuationLumpSumPaymentA,Income#LumpSumPaymentTaxableComponentTaxedElementA,Income#LumpSumPaymentTaxableComponentUntaxedElementA,Income#LumpSumPaymentTaxFreeA,Income#SuperannuationBenefitDeathBenefitI,SuperannuationLumpSumPayment,Income#TaxPayAsYouGoWithholdingCreditTaxWithheldPersonalServicesIncomeA,Income#TotalA,Income#SuperannuationContributionEmployerReportableA,AttributedPersonalServicesIncome,OrganisationNameDetails,GovernmentBenefit#PaymentTypeC,GovernmentBenefit#PaymentTypeDe,GovernmentBenefit#PaymentTaxableA,GovernmentBenefit#TaxPayAsYouGoWithholdingCreditTaxWithheldAustralianGovernmentPensionsAndAllowancesA,GovernmentBenefit#PaymentAdjustmentReasonC,GovernmentBenefit#PaymentAdjustmentReasonDe,GovernmentBenefit,GovernmentBenefitCollection,Income,IncomeCollection,EmployeeShareScheme#OrganisationNameDetailsOrganisationalNameT,EmployeeShareScheme#AustralianBusinessNumberId,EmployeeShareScheme#IncomeTaxUpfrontReducibleDiscountA,EmployeeShareScheme#IncomeTaxUpfrontNonReducibleDiscountA,EmployeeShareScheme#IncomeDeferralDiscountA,EmployeeShareScheme#IncomeTaxPayAsYouGoWithholdingCreditTaxWithheldEmployeeShareSchemeDiscountTFNNotQuotedA,EmployeeShareScheme#IncomeForeignSourceDiscountA,EmployeeShareScheme#IncomeInternationalDealingsTaxOffsetA,EmployeeShareScheme,EmployeeShareSchemeCollection,Interest#FinancialInstitutionAccountFinancialInstitutionNameT,Interest#MiscellaneousInvestorReferenceNumberId,Interest#RegulatoryDisclosuresAccountHolderCt,Interest#AddressDetailsCountryC,Interest#IncomeAdjustmentReasonC,Interest#IncomeAdjustmentReasonDe,Interest#IncomeTaxTFNAmountsWithheldFromGrossInterestA,Interest#IncomeGrossA,InterestTotal,Interest#IncomeTaxTFNAmountsWithheldFromGrossInterestA,Interest#IncomeGrossA,InterestYourShare,Interest,InterestCollection,Dividends#OrganisationNameDetailsOrganisationalNameT,Dividends#MiscellaneousInvestorReferenceNumberId,Dividends#RegulatoryDisclosuresAccountHolderCt,Dividends#AddressDetailsCountryC,Dividends#IncomeUnfrankedA,Dividends#IncomeFrankedA,Dividends#IncomeTaxFrankingCreditsA,Dividends#IncomeTaxPayAsYouGoWithholdingCreditTaxWithheldDividendsTFNNotQuotedA,Dividends#IncomeTaxExplorationCreditsDistributedA,Dividends#CapitalGainsA,DividendTotal,Dividends#IncomeUnfrankedA,Dividends#IncomeFrankedA,Dividends#IncomeTaxFrankingCreditsA,Dividends#IncomeTaxPayAsYouGoWithholdingCreditTaxWithheldDividendsTFNNotQuotedA,Dividends#IncomeTaxExplorationCreditsDistributedA,Dividends#CapitalGainsTotalA,DividendYourShare,Dividends,DividendsCollection,ManagedFunds#OrganisationNameDetailsOrganisationalNameT,ManagedFunds#MiscellaneousInvestorReferenceNumberId,ManagedFunds#RegulatoryDisclosuresAccountHolderCt,ManagedFunds#IncomeNonPrimaryProductionNetA,ManagedFunds#IncomeTrustFrankedNonPrimaryProductionShareA,ManagedFunds#IncomeTaxDeductionTrustPartnershipOtherNonPrimaryProductionNetA,ManagedFunds#IncomeTaxFrankingCreditsShareA,ManagedFunds#IncomeTaxPayAsYouGoWithholdingCreditForAmountsWithheldWhereTFNNotQuotedA,ManagedFunds#InternationalDealingsForeignIncomeGrossA,ManagedFunds#IncomeInternationalDealingsNetA,ManagedFunds#IncomeInternationalDealingsTaxOffsetA,ManagedFunds#IncomeTaxIncomeCategoryMiscellaneousTotalA,ManagedFunds#IncomeTrustPrimaryProductionNetShareA,ManagedFunds#IncomeTaxDeductionPartnershipsAndTrustPrimaryProductionShareA,ManagedFunds#IncomeTaxPayAsYouGoWithholdingCreditForTaxWithheldWhereABNNotQuotedA,ManagedFunds#IncomeTaxCreditTrusteeTaxPaidTotalA,ManagedFunds#IncomeTaxCreditTrusteeTaxPaidReasonC,ManagedFunds#IncomeTrustDistributionsA,ManagedFunds#IncomeTaxPayAsYouGoWithholdingCreditTaxWithheldForeignResidentShareA,ManagedFunds#TaxOffsetClaimNationalRentalAffordabilitySchemeEntitlementShareA,ManagedFunds#IncomeTaxPayAsYouGoWithholdingCreditForCapitalGainsWithheldFromForeignResidentsA,ManagedFunds#InternationalDealingsTransferorTrustAttributedForeignIncomeA,ManagedFunds#IncomeControlledForeignCompaniesAttributableIncomeA,ManagedFunds#InternationalDealingsForeignRentA,ManagedFunds#IncomeTaxFrankingCreditsNewZealandA,ManagedFunds#IncomeTaxEarlyStageVentureCapitalLimitedPartnershipTaxOffsetA,ManagedFunds#IncomeTaxEarlyStageInvestorTaxOffsetA,ManagedFunds#IncomeTaxExplorationCreditsDistributedA,ManagedFunds#NetA,ManagedFunds#A,ManagedFunds#DiscountedMethodA,ManagedFunds#OtherMethodA,CapitalGains,ManagedFundsTotal,ManagedFunds#IncomeNonPrimaryProductionNetA,ManagedFunds#IncomeTrustFrankedNonPrimaryProductionShareA,ManagedFunds#IncomeTaxDeductionTrustPartnershipOtherNonPrimaryProductionNetA,ManagedFunds#IncomeTaxFrankingCreditsShareA,ManagedFunds#IncomeTaxPayAsYouGoWithholdingCreditForAmountsWithheldWhereTFNNotQuotedA,ManagedFunds#InternationalDealingsForeignIncomeGrossA,ManagedFunds#IncomeInternationalDealingsNetA,ManagedFunds#IncomeInternationalDealingsTaxOffsetA,ManagedFunds#IncomeTaxIncomeCategoryMiscellaneousTotalA,ManagedFunds#IncomeTrustPrimaryProductionNetShareA,ManagedFunds#IncomeTaxDeductionPartnershipsAndTrustPrimaryProductionShareA,ManagedFunds#IncomeTaxPayAsYouGoWithholdingCreditForTaxWithheldWhereABNNotQuotedA,ManagedFunds#IncomeTaxCreditTrusteeTaxPaidTotalA,ManagedFunds#IncomeTaxPayAsYouGoWithholdingCreditTaxWithheldForeignResidentShareA,ManagedFunds#TaxOffsetClaimNationalRentalAffordabilitySchemeEntitlementShareA,ManagedFunds#IncomeTaxPayAsYouGoWithholdingCreditForCapitalGainsWithheldFromForeignResidentsA,ManagedFunds#InternationalDealingsTransferorTrustAttributedForeignIncomeA,ManagedFunds#IncomeControlledForeignCompaniesAttributableIncomeA,ManagedFunds#InternationalDealingsForeignRentA,ManagedFunds#IncomeTaxFrankingCreditsNewZealandA,ManagedFunds#IncomeTaxEarlyStageVentureCapitalLimitedPartnershipTaxOffsetA,ManagedFunds#IncomeTaxEarlyStageInvestorTaxOffsetA,ManagedFunds#IncomeTaxExplorationCreditsDistributedA,ManagedFunds#NetA,ManagedFunds#TotalA,ManagedFunds#DiscountedMethodA,ManagedFunds#OtherMethodA,YourShareCapitalGains,ManagedFundsYourShare,ManagedFunds,ManagedFundsCollection,ForeignIncome#OrganisationNameDetailsOrganisationalNameT,ForeignIncome#ExpenseDeductionsA,ForeignIncome#InternationalDealingsGrossA,ForeignIncome#InternationalDealingsTaxOffsetA,ForeignIncomeEmploymentDetails,ForeignIncome#AddressDetailsCountryC,ForeignIncome#InternationalDealingsExemptForeignEmploymentIncomeA,ForeignIncome#InternationalDealingsTaxOffsetA,ForeignIncome#LumpSumArrearsPaymentA,ExemptForeignIncome,FEINPSLumpSumArrearsPayment#TargetFinancialY,FEINPSLumpSumArrearsPayment#A,FEINPSLumpSumArrearsPayment,FEINPSLumpSumArrearsPaymentCollection,ForeignIncome,ForeignIncomeCollection,ForeignEntities#OrganisationNameDetailsOrganisationalNameT,ForeignEntities#PersonUnstructuredNameFullNameT,ForeignEntities#IncomeControlledForeignCompaniesAttributableIncomeA,ForeignEntities#InternationalDealingsTransferorTrustAttributedForeignIncomeA,ForeignEntities,ForeignPensionsOrAnnuities#InternationalDealingsDescriptionT,ForeignPensionsOrAnnuities#IncomeInsuranceLifePremiumDirectGrossA,ForeignPensionsOrAnnuities#ExpenseForeignIncomeDeductionsA,ForeignPensionsOrAnnuities#InternationalDealingsForeignPensionOrAnnuityIncomeUndeductedPurchasePriceA,ForeignPensionsOrAnnuities#IncomeInternationalDealingsTaxOffsetA,ForeignPensionsOrAnnuities#IncomeLumpSumArrearsPaymentA,FPALumpSumArrearsPayment#TargetFinancialY,FPALumpSumArrearsPayment#IncomeA,FPALumpSumArrearsPayment,FPALumpSumArrearsPaymentCollection,ForeignPensionsOrAnnuities,ForeignPensionsOrAnnuitiesCollection,OtherForeignIncome#InternationalDealingsDescriptionT,OtherForeignIncome#ExpenseForeignIncomeDeductionsA,OtherForeignIncome#InternationalDealingsA,OtherForeignIncome#InternationalDealingsTaxOffsetA,ForeignRentalIncomeDetails,OtherForeignIncome#InternationalDealingsForeignSourceCompanyIncomeOtherA,OtherForeignIncome#IncomeTaxFrankingCreditsNewZealandA,OtherForeignIncome#InternationalDealingsTaxOffsetA,ForeignFinancialInvestmentDetails,OtherForeignIncome#InternationalDealingsForeignSourceIncomeOtherA,OtherForeignIncome#InternationalDealingsTaxOffsetA,OtherForeignIncomeDetails,OtherForeignIncome,OtherForeignIncomeCollection,IncomeOther#OtherIncomeTypeC,IncomeOther#De,IncomeOther#A,IncomeOther#A,IncomeOther#DeductibleOtherTotalA,IncomeOther#DeductibleIncludedElsewhereTotalA,SpecialProfessionalExpense,IncomeOther#DepreciatingAssetsIntangibleBalancingAdjustmentA,IncomeOther#DepreciatingAssetsAssessableIncomeFromBalancingAdjustmentEventsTotalA,IncomeOther#BalancingAdjustmentTotalA,IncomeTaxAssessableLowValuePoolDeduction,IncomeOther,IncomeOtherCollection,Partnerships#OrganisationNameDetailsOrganisationalNameT,Partnerships#AustralianBusinessNumberId,Partnerships#PayAsYouGoWithholdingCreditForTaxWithheldWhereABNNotQuotedA,Partnerships#FrankingCreditsShareA,Partnerships#PayAsYouGoWithholdingCreditForTaxWithheldInvestmentsTFNNotQuotedShareA,Partnerships#PayAsYouGoWithholdingCreditTaxWithheldCloselyHeldTrustShareA,Partnerships#PayAsYouGoWithholdingCreditTaxWithheldForeignResidentShareA,Partnerships#TaxOffsetClaimNationalRentalAffordabilitySchemeEntitlementShareA,PartnershipsCreditsIncomeTaxOffsets,Partnerships#IncomeTrustsPrimaryProductionNetA,Partnerships#ExpenseLandcareOperationsAcceleratedDepreciationPrimaryProductionA,Partnerships#IncomeTaxDeductionAndTrustPrimaryProductionShareA,Partnerships#IncomeTaxLossesNonCommercialTypeC,PartnershipsPrimary,Partnerships#IncomePartnershipInvestmentLessForeignIncomeNonPrimaryProductionShareA,Partnerships#IncomePartnershipRentalNonPrimaryProductionShareA,Partnerships#IncomePartnershipNonPrimaryProductionOtherShareA,PartnershipsNonPrimaryDistribution,Partnerships#ExpenseLandcareOperationsNonPrimaryProductionA,Partnerships#IncomeTaxLossesNonCommercialTypeC,Partnerships#IncomeTaxPartnershipExcludingForeignIncomeNonPrimaryProductionA,Partnerships#IncomeTaxPartnershipRentalNonPrimaryProductionA,Partnerships#IncomeTaxTrustPartnershipOtherNonPrimaryProductionNetA,PartnershipsNonPrimaryDeduction,PartnershipsNonPrimary,Partnerships,PartnershipsCollection,TrustIncome#AustralianBusinessNumberId,TrustIncome#AustralianCompanyNumberId,TrustIncome#OrganisationNameDetailsOrganisationalNameT,DistributingTrust,TrustIncome#NetShareA,TrustIncome#IncomeTaxDeductionPartnershipsAndTrustPrimaryProductionShareA,TrustIncome#NonConcessionalManagedInvestmentTrustA,TrustIncome#ExcludedFromNonConcessionalManagedInvestmentTrustA,TrustPrimaryProduction,TrustIncome#TrustLessNetCapitalGainAndForeignIncomeNonPrimaryProductionShareA,TrustIncome#TrustManagedInvestmentSchemeLessNetCapitalGainAndForeignIncomeNonPrimaryProductionShareA,TrustIncome#TrustOtherNonPrimaryProductionShareA,TrustIncome#TrustFrankedInvestmentNonPrimaryProductionShareA,TrustIncome#TrustFrankedNonPrimaryProductionOtherShareA,TrustIncome#IncomeTaxDeductionTrustExcludingCapitalGainsAndForeignIncomeNonPrimaryProductionA,TrustIncome#IncomeTaxDeductionTrustsPartnershipsOtherAllowableNonPrimaryProductionA,TrustIncome#NonConcessionalManagedInvestmentTrustA,TrustIncome#ExcludedFromNonConcessionalManagedInvestmentTrustA,TrustNonPrimaryProduction,TrustIncome#PayAsYouGoWithholdingCreditForTaxWithheldWhereABNNotQuotedA,TrustIncome#TaxableMinorBeneficiaryA,TrustIncome#FrankingCreditsShareA,TrustIncome#PayAsYouGoWithholdingCreditForTaxWithheldInvestmentsTFNNotQuotedShareA,TrustIncome#PayAsYouGoWithholdingCreditTaxWithheldCloselyHeldTrustShareA,TrustIncome#CreditTrusteeTaxPaidTotalA,TrustIncome#CreditTrusteeTaxPaidReasonC,TrustIncome#TrustDistributionsA,TrustIncome#CreditTrusteeTaxPaidCreditsA,TrustIncome#FrankingCreditsReceivedFromNewZealandCompaniesA,TrustIncome#TaxOffsetClaimNationalRentalAffordabilitySchemeEntitlementShareA,TrustIncome#ExplorationCreditsDistributedA,TrustIncome#SmallBusinessEntityNetA,TrustIncome#EarlyStageVentureCapitalLimitedPartnershipTaxOffsetA,TrustIncome#EarlyStageInvestorTaxOffsetA,TrustCreditsIncomeTaxOffsets,TrustIncome#CapitalGainsNetA,TrustIncome#TotalA,TrustIncome#LossesTotalA,TrustIncome#TaxConcessionDiscountTotalA,TrustIncome#TaxConcessionDiscountAndSmallBusinessConcessionsTotalA,TrustIncome#NonConcessionalManagedInvestmentTrustIncomeA,TrustIncome#ExcludedFromNonConcessionalManagedInvestmentTrustIncomeA,TrustCapitalGains,TrustIncome#IncomeTaxPayAsYouGoWithholdingCreditTaxWithheldForeignResidentShareA,TrustIncome#IncomeTaxPayAsYouGoWithholdingCreditForCapitalGainsWithheldFromForeignResidentsA,TrustIncome#IncomeTaxAttributedForeignIncomeA,TrustIncome#InternationalDealingsNetA,TrustIncome#InternationalDealingsTaxOffsetA,TrustIncome#InternationalDealingsAssessableIncomeNonResidentBeneficiaryNonTrusteeA,TrustIncome#InternationalDealingsAssessableIncomeNonResidentBeneficiaryTrusteeA,ForeignAndNonResident,TrustIncome,TrustIncomeCollection,BusinessIncomePaymentSummaries#OrganisationNameDetailsOrganisationalNameT,BusinessIncomePaymentSummaries#AustralianBusinessNumberId,BusinessIncomePaymentSummaries#RemunerationWithholdingPaymentIncomeTypeC,BusinessIncomePaymentSummaries#IncomeTaxIncomeTypeC,BusinessIncomePaymentSummaries#IndustryProductionTypeC,BusinessIncomePaymentSummaries#BusinessGrossTotalA,BusinessIncomePaymentSummaries#IncomeTaxPayAsYouGoWithholdingCreditTaxWithheldA,BusinessIncomePaymentSummaries#SuperannuationContributionEmployerContributionsA,BusinessIncomePaymentSummaries,BusinessIncomePaymentSummariesCollection,TaxablePayment#IncomeAdjustmentReasonC,TaxablePayment#IncomeAdjustmentReasonDe,TaxablePayment#AustralianBusinessNumberId,TaxablePayment#OrganisationNameDetailsOrganisationalNameT,TaxablePayment#IncomeGrossA,TaxablePayment#IncomeTaxPayAsYouGoWithholdingTaxWithheldA,TaxablePayment#GoodsAndServicesTaxLiabilityA,TaxablePayment#GovernmentFundingGrantPaymentD,TaxablePayment#GovernmentFundingGrantProgramNameT,TaxablePayment#IncomeTaxPaymentTypeC,TaxablePayment#IncomeIndustryProductionTypeC,TaxablePayment#IncomeClassificationDecisionC,TaxablePayment,TaxablePaymentCollection,Rp";
    }
}