using System;
using System.Xml;
using System.IO;
using System.Collections.Generic;
using DataContracts;
using System.Text.RegularExpressions;
using System.Reflection;
using System.Linq;
using System.Xml.Schema;
using System.Text;

namespace Ato.EN.IntegrationServices.CodeGenerationPTR
{
    /// <summary>
    /// XML Consumer for PTR
    /// </summary>
    public class PTR2025XmlConsumer
    {
        const string MissingMandatoryMessage = "A mandatory field has not been completed.";
        const string ContactYourProviderMsg = "The message did not pass XML validation. Please contact your software provider.";
        const string InvalidDataValidation = "A field contains invalid data (such as letters in numeric or date field).";
        private const string ErrorCode1 = "CMN.ATO.GEN.XML01";
        private const string ErrorCode3 = "CMN.ATO.GEN.XML03";
        private const string ErrorCode4 = "CMN.ATO.GEN.XML04";
        private const string ErrorCode6 = "CMN.ATO.GEN.XML06";
        private bool _found = false;
        private int _parentCollectionCount;
        private int _childCollectionCount;
        private bool _isExiting = false;
        private bool _isValidationError = false;
        private string _lastPath = string.Empty;
        private ErrorMessageType _validationError = new ErrorMessageType(ErrorDescriptor.NoError);


        #region Error Messages

        public List<ProcessMessageDocument> ErrorMessages { get; set; }

        public bool HasErrors
        {
            get
            {
                return ErrorMessages != null && ErrorMessages.Count > 0;
            }
        }

        private string GetCurrentLocation()
        {
            StringBuilder location = new StringBuilder();
            string[] paths = _currentXPath.ToArray<string>();
            for (int i = paths.Length - 1; i > -1; i--)
            {
                location.Append(paths[i]);
            }
            return location.ToString();
        }

        private void MissingElementError()
        {
            ProcessMessageDocument processMessage;
            processMessage = new ProcessMessageDocument()
            {
                Code = "CMN.ATO.GEN.XML04",
                Description = MissingMandatoryMessage,
                SeverityAsString = "Error",
                Location = GetCurrentLocation(),
            };
            this.ErrorMessages.Add(processMessage);
        }

        #endregion  Error Messages

        #region Embedded Schema

        private static readonly XmlSchemaSet EmbeddedXmlSchemaSet;
        static PTR2025XmlConsumer()
        {
            Assembly executingAssembly = Assembly.GetExecutingAssembly();
            string[] manifestResourceNames = executingAssembly.GetManifestResourceNames();
            string resourceName;
            Stream embeddedSchemaStream;
            XmlSchema embeddedXmlSchema;

            if (manifestResourceNames != null)
            {
                resourceName = manifestResourceNames.FirstOrDefault(rn => rn.Contains("ato.ptr.0013.2025.01.00.xsd") && rn.EndsWith(".xsd"));
                if (resourceName != null)
                {
                    embeddedSchemaStream = executingAssembly.GetManifestResourceStream(resourceName);
                    embeddedXmlSchema = XmlSchema.Read(embeddedSchemaStream, SchemaCallback);
                    EmbeddedXmlSchemaSet = new XmlSchemaSet();
                    EmbeddedXmlSchemaSet.Add(embeddedXmlSchema);
                    EmbeddedXmlSchemaSet.Compile();
                }
            }
        }

        private static void SchemaCallback(object sender, ValidationEventArgs args)
        {
            if (args.Severity == XmlSeverityType.Error)
                throw new XmlSchemaException(args.Message);
        }

        #endregion  Embedded Schema

        #region Xml Reader Settings

        private static XmlReaderSettings ReaderSettings = new XmlReaderSettings()
        {
            CloseInput = false,
            ConformanceLevel = ConformanceLevel.Fragment,
            IgnoreWhitespace = true,
            IgnoreComments = true,
            IgnoreProcessingInstructions = true,
        };

        private XmlReaderSettings GetValidatingReaderSettings()
        {
            XmlReaderSettings validatingReaderSettings = new XmlReaderSettings()
            {
                CloseInput = false,
                ConformanceLevel = ConformanceLevel.Fragment,
                IgnoreWhitespace = true,
                IgnoreComments = true,
                IgnoreProcessingInstructions = true,
                ValidationType = ValidationType.Schema,
            };
            if (EmbeddedXmlSchemaSet == null)
            {
                throw new XmlSchemaException("Embedded Schema is Null");
            }
            else
            {
                validatingReaderSettings.Schemas.Add(EmbeddedXmlSchemaSet);
                validatingReaderSettings.ValidationEventHandler += new ValidationEventHandler(ValidationCallBack);
            }
            return validatingReaderSettings;
        }

        private static Regex DataTypeFailureExpression = new Regex("'(?<Uniqueid>.*?)' element is invalid.*value '(?<Value>.*?)' is invalid.*datatype '(?<DataType>.*?)'", RegexOptions.IgnoreCase | RegexOptions.Compiled | RegexOptions.Singleline);

        private void ValidationCallBack(object sender, ValidationEventArgs args)
        {
            var msg = args.Message;
            var msgToLower = msg.ToLower();

            ProcessMessageDocument processMessage = null;
  
            if (msgToLower.Contains("invalid according to its datatype"))
            {
                Match match = DataTypeFailureExpression.Match(args.Message);

                if (match.Success && match.Groups.Count == 4)
                {
                    var inner = args.Exception.InnerException;
                    var hint = (inner != null && !string.IsNullOrEmpty(inner.Message)) ? " Hint: " + inner.Message : string.Empty;
                    processMessage = BuildProcessMessageDocument(InvalidDataValidation, string.Empty, ErrorCode3);

                    string uniqueID = match.Groups["Uniqueid"].Value;
                    string value = match.Groups["Value"].Value;
                    string dataType = match.Groups["DataType"].Value;


                    var longDescription = string.Format(@"The value specified for an item does not match the item type (value = ""{0}"", item type = {1}, uniqueID = {2})", value, dataType, uniqueID);
                    processMessage.LongDescription = longDescription + hint;
                    processMessage.Parameters = new ProcessMessageParameters();
                    processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "uniqueID", Value = uniqueID });
                    processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "value", Value = value });
                    processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "dataType", Value = dataType });
                }
            }
            else if (msgToLower.Contains("has incomplete content"))
            {
                _validationError = GetErrorDescriptorForHasIncompleteContent(msg);
                processMessage = BuildProcessMessageDocument(MissingMandatoryMessage, msg + _validationError.Hint, _validationError.Code);
                _isValidationError = true;
            }
            else if (msgToLower.Contains("has invalid child element"))
            {
                _validationError = GetErrorDescriptorForHasInvalidChildElement(msg);
                var shortMessage = _validationError.Code == ErrorCode4 ? MissingMandatoryMessage : ContactYourProviderMsg;
                processMessage = BuildProcessMessageDocument(shortMessage, msg + _validationError.Hint, _validationError.Code);
                _isValidationError = true;
            }
            else
            {
                processMessage = BuildProcessMessageDocument(ContactYourProviderMsg, msg, ErrorCode1);
                _isValidationError = true;
            }

            this.ErrorMessages.Add(processMessage);
        }

        private ProcessMessageDocument BuildProcessMessageDocument(string shortDescription, string longDescription, string code)
        {
            return new ProcessMessageDocument()
            {
                Code = code,
                Description = shortDescription,
                LongDescription = longDescription,
                SeverityAsString = "Error",
                Location = GetCurrentLocation()
            };
        }

        private ProcessMessageDocument BuildProcessMessageDocument(ProcessMessageDocument processMessage, string newLocation)
        {
            return new ProcessMessageDocument()
            {
                Code = processMessage.Code,
                Description = processMessage.Description,
                LongDescription = processMessage.LongDescription,
                SeverityAsString = "Error",
                Location = newLocation
            };
        }

        private enum ErrorDescriptor
        {
            NoError = 0,
            MandatoryElementError,
            NonMandatoryElementError,
            UnknownElement,
            Duplicate
        }

        private struct ErrorMessageType
        {
            public string Hint, Code, ExpectedNextValidElement;
            public ErrorDescriptor Errno;

            public ErrorMessageType(ErrorDescriptor errorDescriptor)
            {
                Errno = errorDescriptor;
                Hint = "";
                Code = "";
                ExpectedNextValidElement = "";
            }

            public ErrorMessageType(ErrorDescriptor errorDescriptor, string hint, string code, string elementName)
            {
                Errno = errorDescriptor;
                Hint = hint;
                Code = code;
                ExpectedNextValidElement = elementName;
            }
        }

        private ErrorMessageType GetErrorDescriptorForHasIncompleteContent(string validationErrorMessage)
        {
            ErrorMessageType retval = new ErrorMessageType();
            if (string.IsNullOrWhiteSpace(validationErrorMessage)) return retval;

            validationErrorMessage = validationErrorMessage.Replace(" ", string.Empty);
            validationErrorMessage = validationErrorMessage.Replace("'", " ");
            var parts = validationErrorMessage.Split(' ');
            if (parts.Length != 9) return retval;

            var elementBeingProcessed = parts[1];
            var csvList = parts[5];

            retval = ParseHasIncompleteContentCVSList(csvList, elementBeingProcessed);
            return retval;
        }

        private ErrorMessageType GetErrorDescriptorForHasInvalidChildElement(string validationErrorMessage)
        {
            string msg = string.Empty;
            ErrorMessageType retval = new ErrorMessageType();
            if (string.IsNullOrWhiteSpace(validationErrorMessage)) return retval;

            validationErrorMessage = validationErrorMessage.Replace(" ", string.Empty);
            validationErrorMessage = validationErrorMessage.Replace("'", " ");
            var parts = validationErrorMessage.Split(' ');

            //decide which message format to process
            if (validationErrorMessage.ToLower().Contains("listofpossibleelements"))
            {
                if (parts.Length == 13 || parts.Length == 12 || parts.Length == 10)
                {
                    var csvList = parts[9];
                    var elementBeingProcessed = parts[1];
                    var invalidChildElementName = parts[5];
                    retval = ParseCvsListForHasInvalidChildElementLongFormat(csvList, invalidChildElementName, elementBeingProcessed);
                }

                return retval;
            }
            else
            {
                if (parts.Length != 9) return retval;
                var csvList = parts[5];
                var elementBeingProcessed = parts[1];
                var invalidChildElementName = parts[5];
                
                retval = ParseCvsListForHasInvalidChildElementLongFormat(csvList, invalidChildElementName, elementBeingProcessed);
            }
            return retval;
        }

        private ErrorMessageType ParseCvsListForHasInvalidChildElementShortFormat(string csvList, string invalidChildElementName, string elementBeingProcessed)
        {
            ErrorMessageType retval = new ErrorMessageType();

            if (!string.IsNullOrEmpty(csvList))
            {
                var lastElementInListOfPossibleElements = GetLastElementInCsvList(csvList);

                retval.Hint =
                    string.Format(
                        " Hint: while processing parent element [{0}] child element [{1}] was unexpected",
                        elementBeingProcessed, invalidChildElementName);
                retval.Code = ErrorCode1;
                retval.Errno = ErrorDescriptor.UnknownElement;
                retval.ExpectedNextValidElement = invalidChildElementName;
            }

            return retval;
        }

        private string GetLastElementInCsvList(string csvList)
        {
            var list = (csvList.Contains(','))
                ? new List<string>(csvList.Split(','))
                : new List<string> { csvList };

            string lastElement = list.Last();        
            if (lastElement.EndsWith("...."))
            {
                list.RemoveAt(list.Count-1);
            }

            return list.Last();
        }

        private ErrorMessageType ParseCvsListForHasInvalidChildElementLongFormat(string csvList, string invalidChildElementName, string elementBeingProcessed)
        {
            ErrorMessageType retval = new ErrorMessageType();

            if (!string.IsNullOrEmpty(csvList))
            {
                var lastElementInListOfPossibleElements = GetLastElementInCsvList(csvList);
                var elementList = new List<string>(_elementCsvList.Split(','));

                if (ContainsElementForElementBeingProcessed(elementList, elementBeingProcessed, invalidChildElementName))
                {
                    var indexOfElementBeingProcessed = GetIndexOfElementBeingProcessed(elementList, elementBeingProcessed);
                    var indexOfinvalidChildElementName = GetIndexOfElementUnderElementBeingProcessed(elementList, elementBeingProcessed, invalidChildElementName);
                    var indexOflastElementInListOfPossibleElements = GetIndexOfElementUnderElementBeingProcessed(elementList, elementBeingProcessed, lastElementInListOfPossibleElements);

                    if (indexOfinvalidChildElementName > indexOflastElementInListOfPossibleElements) 
                    {
                        retval.Hint =
                            string.Format(
                                " Hint: while processing parent element [{0}] child mandatory element [{1}] was expected but not found",
                                elementBeingProcessed, lastElementInListOfPossibleElements);
                        retval.Code = ErrorCode4;
                        retval.Errno = ErrorDescriptor.MandatoryElementError;
                        retval.ExpectedNextValidElement = lastElementInListOfPossibleElements;
                    }
                    else
                    {
                        retval.Hint =
                            string.Format(
                                " Hint: while processing parent element [{0}] child non mandatory element [{1}] was unexpected",
                                elementBeingProcessed, invalidChildElementName);
                        retval.Code = ErrorCode1;
                        retval.Errno = (indexOfinvalidChildElementName != indexOflastElementInListOfPossibleElements) ? ErrorDescriptor.NonMandatoryElementError : ErrorDescriptor.Duplicate;
                        retval.ExpectedNextValidElement = invalidChildElementName;
                    }
                }
                else
                {
                    //this element is unknown to the schema.
                    retval.Hint =
                        string.Format(
                            " Hint: while processing parent element [{0}] child element [{1}] was unexpected",
                            elementBeingProcessed, invalidChildElementName);
                    retval.Code = ErrorCode1;
                    retval.Errno = ErrorDescriptor.UnknownElement;
                    retval.ExpectedNextValidElement = invalidChildElementName;
                }
            }

            return retval;
        }

        private int GetIndexOfElementUnderElementBeingProcessed(List<string> validElementList,
                                                                string elementBeingProcessed,
                                                                string elementName)
        {
            int result;

            string elementWithParentNameMatch = elementBeingProcessed + "#" + elementName;
            result = validElementList.IndexOf(elementWithParentNameMatch);

            if (result < 0)
            {
                result = validElementList.IndexOf(elementName);
            }

            return result;
        }

        private int GetIndexOfElementBeingProcessed(List<string> validElementList,
                                                    string elementBeingProcessed)
        {
           return validElementList.IndexOf(elementBeingProcessed);
        }

        private bool ContainsElementForElementBeingProcessed(List<string> validElementList,
                                                             string elementBeingProcessed,
                                                             string elementName)
        {
            return validElementList.Contains(elementBeingProcessed + "#" + elementName) || validElementList.Contains(elementName);
        }

        private ErrorMessageType ParseHasIncompleteContentCVSList(string csvList, string elementBeingProcessed)
        {
            ErrorMessageType retval = new ErrorMessageType();

            if (!string.IsNullOrEmpty(csvList))
            {
                var lastElementInListOfPossibleElements = GetLastElementInCsvList(csvList);

                retval.Hint =
                    string.Format(
                        " Hint: while processing parent element [{0}] child mandatory element [{1}] was not found",
                        elementBeingProcessed, lastElementInListOfPossibleElements);

                retval.Code = ErrorCode4;
                retval.Errno = ErrorDescriptor.MandatoryElementError;
                retval.ExpectedNextValidElement = lastElementInListOfPossibleElements;
            }
            return retval;
        }

        private void StartEndDateError(DateTime startDate, DateTime endDate)
        {
            ProcessMessageDocument processMessage;

            string longDescription = string.Format("End date is earlier than start date (state date = {0}, end date = {1})", startDate.ToString("yyyy-MM-dd"), endDate.ToString("yyyy-MM-dd"));

            processMessage = new ProcessMessageDocument()
            {
                Code = "CMN.ATO.GEN.XML06",
                Description = "End date is earlier than start date.",
                LongDescription = longDescription,
                SeverityAsString = "Error",
                Location = GetCurrentLocation(),
            };
            processMessage.Parameters = new ProcessMessageParameters();
            processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "startDate", Value = startDate.ToString("yyyy-MM-dd") });
            processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "endDate", Value = endDate.ToString("yyyy-MM-dd") });
            this.ErrorMessages.Add(processMessage);
        }

        private void StartEndDateError(DateTime? startDate, DateTime? endDate)
        {
            ProcessMessageDocument processMessage;

            if (startDate == null || endDate == null)
                return;

            string longDescription = string.Format("End date is earlier than start date (state date = {0}, end date = {1})", startDate.Value.ToString("yyyy-MM-dd"), endDate.Value.ToString("yyyy-MM-dd"));

            processMessage = new ProcessMessageDocument()
            {
                Code = "CMN.ATO.GEN.XML06",
                Description = "End date is earlier than start date.",
                LongDescription = longDescription,
                SeverityAsString = "Error",
                Location = GetCurrentLocation(),
            };
            processMessage.Parameters = new ProcessMessageParameters();
            processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "startDate", Value = startDate.Value.ToString("yyyy-MM-dd") });
            processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "endDate", Value = endDate.Value.ToString("yyyy-MM-dd") });
            this.ErrorMessages.Add(processMessage);
        }

        #endregion  Xml Reader Settings

        #region IsEmptyOrNilElement
        private static bool IsEmptyOrNilElement(XmlReader reader)
        {
            bool emptyOrNil = false;
            if (reader.IsEmptyElement)
            {
                emptyOrNil = true;
            }
            else
            {
                if (reader.HasAttributes)
                {
                    string nilValue = reader.GetAttribute("nil", "http://www.w3.org/2001/XMLSchema-instance");
                    if (nilValue != null && nilValue.ToLowerInvariant() == "true" || nilValue == "1")
                    {
                        emptyOrNil = true;
                    }
                }
            }
            return emptyOrNil;
        }
        #endregion  IsEmptyOrNilElement

        #region ToBoolean
        private bool? ToBoolean(string str)
        {
            bool returnValue;
            string value = (str ?? "").Trim();
            if (value == "0" || value == "1")
                return Convert.ToBoolean(Convert.ToInt32(value));
            
            if (Boolean.TryParse(value, out returnValue))
                return returnValue;
            return null;
        }
        #endregion  ToBoolean

        #region MoveToContent
        public bool MoveToContent(XmlReader reader)
        {
            try
            {
                reader.MoveToContent();
                return true;
            }
            catch (XmlException ex)
            {
                ProcessMessageDocument processMessage;
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.XML01",
                    Description = ContactYourProviderMsg,
                    LongDescription = ex.Message,
                    SeverityAsString = "Error",
                };
                this.ErrorMessages.Add(processMessage);
                return false;
            }
        }
        #endregion  MoveToContent

        #region ReadToNextElement


        public bool ReadToNextElement(XmlReader reader, bool isMandatory) 
        {
            return ReadToNextElement(reader);
        }

        public bool ReadToNextElement(XmlReader reader) 
        {
            bool retval = false;

            if ( (reader.EOF && !_isValidationError) || _isExiting)
            {
                return false; //do nothing
            }

            if (!_isValidationError)
            {
                retval = SetReaderToNextElement(reader); //position the reader on the next valid element
            }

            if (_isValidationError)
            {
                //process validation errors
                var expectedNextElement = (!string.IsNullOrEmpty(_validationError.ExpectedNextValidElement)) ? _validationError.ExpectedNextValidElement : reader.LocalName;
                var actualNextValidElement = reader.LocalName;
                var location = GetCurrentLocation();

                if (_validationError.Errno == ErrorDescriptor.Duplicate)
                {
                    location = _lastPath;
                }

                var xpath = RemoveLeadingAndTrailingSlashes(location);

                var element = GetElementDescriptor(xpath);

                if (_validationError.Errno == ErrorDescriptor.UnknownElement)
                {
                    //set the xpath for elements that are not known to the schema
                    location = "/" + element.Path + "/" + element.Prefix + ":" + _validationError.ExpectedNextValidElement;
                    _isExiting = true;
                }
                else if(_validationError.Errno == ErrorDescriptor.NonMandatoryElementError)
                {
                    //set the xpath for the optional element that caused a validation
                    location = "/" + element.Path + "/" + element.Prefix + ":" + reader.LocalName;
                    _isExiting = true;
                }
                else
                {
                    //decide if we have spooled to the missing mandatory element. 
                    _isExiting = expectedNextElement == element.PathLeafNode;
                }

                if(_isExiting)
                {
                    //The location has changed since detection of the validation error - so update it.
                    UpdateErrorMessageWithLocation(location);
                }

                return false;
             }

             //Save previous path so we can detect duplicate elements
            _lastPath = GetCurrentLocation();

            return retval;
        }

        private bool ReadNext(XmlReader reader)
        {
            return !reader.EOF && reader.Read();
        }

        private void UpdateErrorMessageWithLocation(string newLocation)
        {
            var lastProcessMessage = ErrorMessages.Last();
            ErrorMessages.Remove(lastProcessMessage);
            var newProcessMessage = BuildProcessMessageDocument(lastProcessMessage, newLocation);
            ErrorMessages.Add(newProcessMessage);
        }

        private string RemoveLeadingAndTrailingSlashes(string xpath)
        {
            var xpathLength = xpath.Length;
            if (xpathLength <= 0) return string.Empty;
            var firstChar = xpath.Substring(0, 1);
            var lastChar = xpath.Substring(xpathLength - 1, 1);
            if (firstChar == "/")
            {
                xpath = xpath.Remove(0, 1);
                xpathLength--;
            }
            if (lastChar == "/")
            {
                xpath = xpath.Remove(xpathLength - 1, 1);
            }
            return xpath;
        }

        private struct ElementDescriptor
        {
            public string Prefix, PathLeafNode, Path;
            public ElementDescriptor(string prefix, string leaf , string path)
            {
                Prefix = prefix;
                PathLeafNode = leaf;
                Path = path;
            }
        }

        private ElementDescriptor GetElementDescriptor(string xpath)
        {
            var retval = new ElementDescriptor();

            var parts = xpath.Split('/');
            var numberOfParts = parts.Length;
            if (numberOfParts > 0)
            {
                var leafNode = parts[numberOfParts - 1];
                var p = leafNode.Split(':');
                numberOfParts = p.Length;
                if(numberOfParts == 1)
                {
                    retval.PathLeafNode = p[0];
                    retval.Prefix = string.Empty;
                }
                else if(numberOfParts == 2)
                {
                    retval.PathLeafNode = p[1];
                    retval.Prefix = p[0];
                }
                var leafNodeLength = leafNode.Length;
                var path = xpath.Remove(xpath.Length - leafNodeLength, leafNodeLength);
                retval.Path = RemoveLeadingAndTrailingSlashes(path);
            }
            return retval;
        }

        private bool SetReaderToNextElement(XmlReader reader) 
        {
            try
            {
                if (reader.EOF || _isExiting) return false;
                if (_found)
                {
                    reader.Read();
                    _found = false;
                }
                while (!reader.EOF && reader.NodeType != XmlNodeType.Element)
                {
                    reader.Read();
                }
      
                return reader.NodeType == XmlNodeType.Element && !reader.EOF;

            }
            catch (XmlException ex)
            {
                ProcessMessageDocument processMessage;
                processMessage = new ProcessMessageDocument()
                {
                    Code = ErrorCode1,
                    Description = ContactYourProviderMsg,
                    LongDescription = ex.Message,
                    SeverityAsString = "Error",
                };
                this.ErrorMessages.Add(processMessage);
                return false;
            }
        }

        #endregion  ReadToNextElement

        private Stack<string> _currentXPath = new Stack<string>();

        public PTR2025 Consume(Stream streamToLoad, bool validateDataTypes = false)
        {
            PTR2025 report = new PTR2025();

            // Working Variable for if can still read from the xml stream
            bool reading;

            // Working Variables for current values
            string currentValue;
            DateTime currentDateTimeValue;
            bool? currentBooleanValue;
            decimal currentDecimalValue;
            double currentDoubleValue;
            float currentFloatValue;
            sbyte currentsByteValue;
            byte currentByteValue;
            short currentShortValue;
            ushort currentuShortValue;
            uint currentuIntValue;
            int currentIntValue;
            long currentLongValue;
            ulong currentuLongValue;

            this.ErrorMessages = new List<ProcessMessageDocument>();
            _currentXPath.Push("/tns:PTR");

            if (streamToLoad == null || streamToLoad.Length == 0 || !streamToLoad.CanRead)
            {
                MissingElementError();
                return report;
            }

            streamToLoad.Position = 0;
            XmlReader reader;

            if (validateDataTypes)
                reader = XmlReader.Create(streamToLoad, GetValidatingReaderSettings());
            else
                reader = XmlReader.Create(streamToLoad, ReaderSettings);

            if (!MoveToContent(reader))
                return report;

            reading = !reader.EOF;

            if (ReadToNextElement(reader) && reader.LocalName == "PTR" && reader.NamespaceURI == "http://www.sbr.gov.au/ato/ptr")
            {
                _found = true;
                ReadToNextElement(reader);
            }
            else
            {
                MissingElementError();
                return report;
            }

    
            #region RP
            _currentXPath.Push("/tns:RP");
            //3. use case
            if (ReadToNextElement(reader,true) && reader.LocalName == "RP")
            {
                report.RpCollectionExists = true;
                report.RpCollectionCount += 1;
                _found = true; 
        
                #region TargetFinancialY
                _currentXPath.Push("/tns:TargetFinancialY");
                //6. use case
                if (ReadToNextElement(reader,true) && reader.LocalName == "TargetFinancialY")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Int32.TryParse(currentValue, out currentIntValue))
                            {
                                report.PTR1 = currentIntValue;
                            }
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion TargetFinancialY
        
                #region PeriodStartD
                _currentXPath.Push("/tns:PeriodStartD");
                //6. use case
                if (ReadToNextElement(reader,true) && reader.LocalName == "PeriodStartD")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                            {
                                report.PTR500 = currentDateTimeValue.Date;
                            }
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion PeriodStartD
        
                #region PeriodEndD
                _currentXPath.Push("/tns:PeriodEndD");
                //6. use case
                if (ReadToNextElement(reader,true) && reader.LocalName == "PeriodEndD")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                            {
                                report.PTR501 = currentDateTimeValue.Date;
                            }
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion PeriodEndD
        
                #region TaxFileNumberId
                _currentXPath.Push("/tns:TaxFileNumberId");
                //6. use case
                if (ReadToNextElement(reader,true) && reader.LocalName == "TaxFileNumberId")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.PTR5 = currentValue;
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion TaxFileNumberId
        
                #region AustralianBusinessNumberId
                _currentXPath.Push("/tns:AustralianBusinessNumberId");
                //6. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "AustralianBusinessNumberId")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.PTR7 = currentValue;
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion AustralianBusinessNumberId
        
                #region ReportAmendment
                _currentXPath.Push("/tns:ReportAmendment");
                //3. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "ReportAmendment")
                {
                    report.RP_ReportAmendmentCollectionExists = true;
                    report.RP_ReportAmendmentCollectionCount += 1;
                    _found = true; 
            
                    #region I
                    _currentXPath.Push("/tns:I");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "I")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.PTR316 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion I
            
                    #region AmendmentTypeC
                    _currentXPath.Push("/tns:AmendmentTypeC");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "AmendmentTypeC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.PTR317 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion AmendmentTypeC
            
                    #region AmendmentReasonT
                    _currentXPath.Push("/tns:AmendmentReasonT");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "AmendmentReasonT")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.PTR318 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion AmendmentReasonT
            
                    #region AmendmentSequenceN
                    _currentXPath.Push("/tns:AmendmentSequenceN");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "AmendmentSequenceN")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.PTR319 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion AmendmentSequenceN
                } // End of if ReportAmendment node exists
        
                _currentXPath.Pop();
                #endregion ReportAmendment
        
                #region CurrentPartnershipNameOrganisationNameDetailsOrganisationalNameT
                _currentXPath.Push("/tns:CurrentPartnershipNameOrganisationNameDetailsOrganisationalNameT");
                //6. use case
                if (ReadToNextElement(reader,true) && reader.LocalName == "CurrentPartnershipNameOrganisationNameDetailsOrganisationalNameT")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.PTR6 = currentValue;
                            report.RP_CurrentPartnershipNameCollectionExists = true;
                            report.RP_CurrentPartnershipNameCollectionCount += 1;
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion CurrentPartnershipNameOrganisationNameDetailsOrganisationalNameT
        
                #region PreviousPartnershipNameOrganisationNameDetailsOrganisationalNameT
                _currentXPath.Push("/tns:PreviousPartnershipNameOrganisationNameDetailsOrganisationalNameT");
                //6. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "PreviousPartnershipNameOrganisationNameDetailsOrganisationalNameT")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.PTR8 = currentValue;
                            report.RP_PreviousPartnershipNameCollectionExists = true;
                            report.RP_PreviousPartnershipNameCollectionCount += 1;
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion PreviousPartnershipNameOrganisationNameDetailsOrganisationalNameT
        
                #region CurrentPostalAddressDetails
                _currentXPath.Push("/tns:CurrentPostalAddressDetails");
                //3. use case
                if (ReadToNextElement(reader,true) && reader.LocalName == "CurrentPostalAddressDetails")
                {
                    report.RP_CurrentPostalAddressDetailsCollectionExists = true;
                    report.RP_CurrentPostalAddressDetailsCollectionCount += 1;
                    _found = true; 
            
                    #region OverseasAddressI
                    _currentXPath.Push("/tns:OverseasAddressI");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "OverseasAddressI")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.PTR216 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion OverseasAddressI
            
                    #region Line1T
                    _currentXPath.Push("/tns:Line1T");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "Line1T")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.PTR9 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion Line1T
            
                    #region Line2T
                    _currentXPath.Push("/tns:Line2T");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "Line2T")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.PTR211 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion Line2T
            
                    #region LocalityNameT
                    _currentXPath.Push("/tns:LocalityNameT");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "LocalityNameT")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.PTR10 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion LocalityNameT
            
                    #region PostcodeT
                    _currentXPath.Push("/tns:PostcodeT");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "PostcodeT")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.PTR12 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion PostcodeT
            
                    #region StateOrTerritoryC
                    _currentXPath.Push("/tns:StateOrTerritoryC");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "StateOrTerritoryC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.PTR11 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion StateOrTerritoryC
            
                    #region CountryC
                    _currentXPath.Push("/tns:CountryC");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "CountryC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.PTR302 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion CountryC
                } // End of if CurrentPostalAddressDetails node exists
        
                _currentXPath.Pop();
                #endregion CurrentPostalAddressDetails
        
                #region PreviousPostalAddressDetails
                _currentXPath.Push("/tns:PreviousPostalAddressDetails");
                //3. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "PreviousPostalAddressDetails")
                {
                    report.RP_PreviousPostalAddressDetailsCollectionExists = true;
                    report.RP_PreviousPostalAddressDetailsCollectionCount += 1;
                    _found = true; 
            
                    #region OverseasAddressI
                    _currentXPath.Push("/tns:OverseasAddressI");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "OverseasAddressI")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.PTR219 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion OverseasAddressI
            
                    #region Line1T
                    _currentXPath.Push("/tns:Line1T");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "Line1T")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.PTR14 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion Line1T
            
                    #region Line2T
                    _currentXPath.Push("/tns:Line2T");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "Line2T")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.PTR212 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion Line2T
            
                    #region LocalityNameT
                    _currentXPath.Push("/tns:LocalityNameT");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "LocalityNameT")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.PTR15 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion LocalityNameT
            
                    #region PostcodeT
                    _currentXPath.Push("/tns:PostcodeT");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "PostcodeT")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.PTR17 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion PostcodeT
            
                    #region StateOrTerritoryC
                    _currentXPath.Push("/tns:StateOrTerritoryC");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "StateOrTerritoryC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.PTR16 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion StateOrTerritoryC
            
                    #region CountryC
                    _currentXPath.Push("/tns:CountryC");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "CountryC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.PTR303 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion CountryC
                } // End of if PreviousPostalAddressDetails node exists
        
                _currentXPath.Pop();
                #endregion PreviousPostalAddressDetails
        
                #region MailRecipient
                _currentXPath.Push("/tns:MailRecipient");
                //3. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "MailRecipient")
                {
                    report.RP_MailRecipientCollectionExists = true;
                    report.RP_MailRecipientCollectionCount += 1;
                    _found = true; 
            
                    #region PersonNameDetails
                    _currentXPath.Push("/tns:PersonNameDetails");
                    //3. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "PersonNameDetails")
                    {
                        report.RP_MailRecipient_PersonNameDetailsCollectionExists = true;
                        report.RP_MailRecipient_PersonNameDetailsCollectionCount += 1;
                        _found = true; 
                
                        #region TitleT
                        _currentXPath.Push("/tns:TitleT");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "TitleT")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.PTR19 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion TitleT
                
                        #region NameSuffixT
                        _currentXPath.Push("/tns:NameSuffixT");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "NameSuffixT")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.PTR21 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion NameSuffixT
                
                        #region FamilyNameT
                        _currentXPath.Push("/tns:FamilyNameT");
                        //6. use case
                        if (ReadToNextElement(reader,true) && reader.LocalName == "FamilyNameT")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.PTR20 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion FamilyNameT
                
                        #region GivenNameT
                        _currentXPath.Push("/tns:GivenNameT");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "GivenNameT")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.PTR22 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion GivenNameT
                
                        #region OtherGivenNameT
                        _currentXPath.Push("/tns:OtherGivenNameT");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "OtherGivenNameT")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.PTR23 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion OtherGivenNameT
                    } // End of if PersonNameDetails node exists
            
                    _currentXPath.Pop();
                    #endregion PersonNameDetails
            
                    #region OrganisationDetails
                    _currentXPath.Push("/tns:OrganisationDetails");
                    //3. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "OrganisationDetails")
                    {
                        report.RP_MailRecipient_OrganisationDetailsCollectionExists = true;
                        report.RP_MailRecipient_OrganisationDetailsCollectionCount += 1;
                        _found = true; 
                
                        #region OrganisationNameDetailsOrganisationalNameT
                        _currentXPath.Push("/tns:OrganisationNameDetailsOrganisationalNameT");
                        //6. use case
                        if (ReadToNextElement(reader,true) && reader.LocalName == "OrganisationNameDetailsOrganisationalNameT")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.PTR24 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion OrganisationNameDetailsOrganisationalNameT
                
                        #region AustralianBusinessNumberId
                        _currentXPath.Push("/tns:AustralianBusinessNumberId");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "AustralianBusinessNumberId")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.PTR25 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion AustralianBusinessNumberId
                    } // End of if OrganisationDetails node exists
            
                    _currentXPath.Pop();
                    #endregion OrganisationDetails
                } // End of if MailRecipient node exists
        
                _currentXPath.Pop();
                #endregion MailRecipient
        
                #region ElectionsInterposedEntityElectionStatusY
                _currentXPath.Push("/tns:ElectionsInterposedEntityElectionStatusY");
                //6. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "ElectionsInterposedEntityElectionStatusY")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Int32.TryParse(currentValue, out currentIntValue))
                            {
                                report.PTR26 = currentIntValue;
                            }
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion ElectionsInterposedEntityElectionStatusY
        
                #region ElectionsInterposedEntityElectionRevocationC
                _currentXPath.Push("/tns:ElectionsInterposedEntityElectionRevocationC");
                //6. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "ElectionsInterposedEntityElectionRevocationC")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.PTR27 = currentValue;
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion ElectionsInterposedEntityElectionRevocationC
        
                #region LodgmentFinalReturnI
                _currentXPath.Push("/tns:LodgmentFinalReturnI");
                //6. use case
                if (ReadToNextElement(reader,true) && reader.LocalName == "LodgmentFinalReturnI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.PTR29 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion LodgmentFinalReturnI
        
                #region OrganisationDetailsMainIncomeActivityDe
                _currentXPath.Push("/tns:OrganisationDetailsMainIncomeActivityDe");
                //6. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "OrganisationDetailsMainIncomeActivityDe")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.PTR30 = currentValue;
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion OrganisationDetailsMainIncomeActivityDe
        
                #region OrganisationDetailsOrganisationIndustry2006ExtendedC
                _currentXPath.Push("/tns:OrganisationDetailsOrganisationIndustry2006ExtendedC");
                //6. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "OrganisationDetailsOrganisationIndustry2006ExtendedC")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.PTR31 = currentValue;
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion OrganisationDetailsOrganisationIndustry2006ExtendedC
        
                #region OrganisationDetailsActivityEventC
                _currentXPath.Push("/tns:OrganisationDetailsActivityEventC");
                //6. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "OrganisationDetailsActivityEventC")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.PTR32 = currentValue;
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion OrganisationDetailsActivityEventC
        
                #region OrganisationDetailsTaxConsolidationStatusI
                _currentXPath.Push("/tns:OrganisationDetailsTaxConsolidationStatusI");
                //6. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "OrganisationDetailsTaxConsolidationStatusI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.PTR33 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion OrganisationDetailsTaxConsolidationStatusI
        
                #region OrganisationDetailsSignificantGlobalEntityStatusI
                _currentXPath.Push("/tns:OrganisationDetailsSignificantGlobalEntityStatusI");
                //6. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "OrganisationDetailsSignificantGlobalEntityStatusI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.PTR355 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion OrganisationDetailsSignificantGlobalEntityStatusI
        
                #region InternationalDealingsCountryByCountryReportingEntityStatusI
                _currentXPath.Push("/tns:InternationalDealingsCountryByCountryReportingEntityStatusI");
                //6. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "InternationalDealingsCountryByCountryReportingEntityStatusI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.PTR372 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion InternationalDealingsCountryByCountryReportingEntityStatusI
        
                #region Income
                _currentXPath.Push("/tns:Income");
                //3. use case
                if (ReadToNextElement(reader,true) && reader.LocalName == "Income")
                {
                    report.RP_IncomeCollectionExists = true;
                    report.RP_IncomeCollectionCount += 1;
                    _found = true; 
            
                    #region BusinessIncomeAndExpenses
                    _currentXPath.Push("/tns:BusinessIncomeAndExpenses");
                    //3. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "BusinessIncomeAndExpenses")
                    {
                        report.RP_Income_BusinessIncomeAndExpensesCollectionExists = true;
                        report.RP_Income_BusinessIncomeAndExpensesCollectionCount += 1;
                        _found = true; 
                
                        #region SmallBusinessEntityNetA
                        _currentXPath.Push("/tns:SmallBusinessEntityNetA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "SmallBusinessEntityNetA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.PTR352 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion SmallBusinessEntityNetA
                
                        #region ReconciliationAdjustmentTotalA
                        _currentXPath.Push("/tns:ReconciliationAdjustmentTotalA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "ReconciliationAdjustmentTotalA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.PTR59 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion ReconciliationAdjustmentTotalA
                
                        #region IncomePrimaryProduction
                        _currentXPath.Push("/tns:IncomePrimaryProduction");
                        //3. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "IncomePrimaryProduction")
                        {
                            report.RP_Income_BusinessIncomeAndExpenses_IncomePrimaryProductionCollectionExists = true;
                            report.RP_Income_BusinessIncomeAndExpenses_IncomePrimaryProductionCollectionCount += 1;
                            _found = true; 
                    
                            #region RemunerationABNNotQuotedPaymentGrossA
                            _currentXPath.Push("/tns:RemunerationABNNotQuotedPaymentGrossA");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "RemunerationABNNotQuotedPaymentGrossA")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.PTR35 = currentDecimalValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion RemunerationABNNotQuotedPaymentGrossA
                    
                            #region GovernmentFundingGovernmentIndustryPaymentsAssessableA
                            _currentXPath.Push("/tns:GovernmentFundingGovernmentIndustryPaymentsAssessableA");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "GovernmentFundingGovernmentIndustryPaymentsAssessableA")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.PTR39 = currentDecimalValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion GovernmentFundingGovernmentIndustryPaymentsAssessableA
                    
                            #region GovernmentFundingGovernmentIndustryPaymentsAssessableI
                            _currentXPath.Push("/tns:GovernmentFundingGovernmentIndustryPaymentsAssessableI");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "GovernmentFundingGovernmentIndustryPaymentsAssessableI")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        currentBooleanValue = ToBoolean(currentValue);
                                        if (currentBooleanValue != null)
                                        {
                                            report.PTR258 = currentBooleanValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion GovernmentFundingGovernmentIndustryPaymentsAssessableI
                    
                            #region OtherA
                            _currentXPath.Push("/tns:OtherA");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "OtherA")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.PTR43 = currentDecimalValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion OtherA
                    
                            #region NetA
                            _currentXPath.Push("/tns:NetA");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "NetA")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.PTR61 = currentDecimalValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion NetA
                        } // End of if IncomePrimaryProduction node exists
                
                        _currentXPath.Pop();
                        #endregion IncomePrimaryProduction
                
                        #region IncomeNonPrimaryProduction
                        _currentXPath.Push("/tns:IncomeNonPrimaryProduction");
                        //3. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "IncomeNonPrimaryProduction")
                        {
                            report.RP_Income_BusinessIncomeAndExpenses_IncomeNonPrimaryProductionCollectionExists = true;
                            report.RP_Income_BusinessIncomeAndExpenses_IncomeNonPrimaryProductionCollectionCount += 1;
                            _found = true; 
                    
                            #region RemunerationABNNotQuotedPaymentGrossA
                            _currentXPath.Push("/tns:RemunerationABNNotQuotedPaymentGrossA");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "RemunerationABNNotQuotedPaymentGrossA")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.PTR36 = currentDecimalValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion RemunerationABNNotQuotedPaymentGrossA
                    
                            #region RemunerationPaymentToForeignResidentGrossA
                            _currentXPath.Push("/tns:RemunerationPaymentToForeignResidentGrossA");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "RemunerationPaymentToForeignResidentGrossA")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.PTR38 = currentDecimalValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion RemunerationPaymentToForeignResidentGrossA
                    
                            #region GovernmentFundingGovernmentIndustryPaymentsAssessableA
                            _currentXPath.Push("/tns:GovernmentFundingGovernmentIndustryPaymentsAssessableA");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "GovernmentFundingGovernmentIndustryPaymentsAssessableA")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.PTR41 = currentDecimalValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion GovernmentFundingGovernmentIndustryPaymentsAssessableA
                    
                            #region GovernmentFundingGovernmentIndustryPaymentsAssessableI
                            _currentXPath.Push("/tns:GovernmentFundingGovernmentIndustryPaymentsAssessableI");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "GovernmentFundingGovernmentIndustryPaymentsAssessableI")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        currentBooleanValue = ToBoolean(currentValue);
                                        if (currentBooleanValue != null)
                                        {
                                            report.PTR259 = currentBooleanValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion GovernmentFundingGovernmentIndustryPaymentsAssessableI
                    
                            #region OtherA
                            _currentXPath.Push("/tns:OtherA");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "OtherA")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.PTR44 = currentDecimalValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion OtherA
                    
                            #region NetA
                            _currentXPath.Push("/tns:NetA");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "NetA")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.PTR62 = currentDecimalValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion NetA
                        } // End of if IncomeNonPrimaryProduction node exists
                
                        _currentXPath.Pop();
                        #endregion IncomeNonPrimaryProduction
                
                        #region BusinessExpenses
                        _currentXPath.Push("/tns:BusinessExpenses");
                        //3. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "BusinessExpenses")
                        {
                            report.RP_Income_BusinessIncomeAndExpenses_BusinessExpensesCollectionExists = true;
                            report.RP_Income_BusinessIncomeAndExpenses_BusinessExpensesCollectionCount += 1;
                            _found = true; 
                    
                            #region ForeignResidentWithholdingA
                            _currentXPath.Push("/tns:ForeignResidentWithholdingA");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "ForeignResidentWithholdingA")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.PTR45 = currentDecimalValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion ForeignResidentWithholdingA
                    
                            #region RemunerationWagesAndSalariesPaymentsContractorAndConsultantA
                            _currentXPath.Push("/tns:RemunerationWagesAndSalariesPaymentsContractorAndConsultantA");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "RemunerationWagesAndSalariesPaymentsContractorAndConsultantA")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.PTR46 = currentDecimalValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion RemunerationWagesAndSalariesPaymentsContractorAndConsultantA
                    
                            #region SuperannuationContributionEmployerContributionsA
                            _currentXPath.Push("/tns:SuperannuationContributionEmployerContributionsA");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "SuperannuationContributionEmployerContributionsA")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.PTR47 = currentDecimalValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion SuperannuationContributionEmployerContributionsA
                    
                            #region OperatingCostOfSalesA
                            _currentXPath.Push("/tns:OperatingCostOfSalesA");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "OperatingCostOfSalesA")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.PTR343 = currentDecimalValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion OperatingCostOfSalesA
                    
                            #region OperatingBadDebtsA
                            _currentXPath.Push("/tns:OperatingBadDebtsA");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "OperatingBadDebtsA")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.PTR344 = currentDecimalValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion OperatingBadDebtsA
                    
                            #region OperatingLeaseTotalA
                            _currentXPath.Push("/tns:OperatingLeaseTotalA");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "OperatingLeaseTotalA")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.PTR50 = currentDecimalValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion OperatingLeaseTotalA
                    
                            #region OperatingRentA
                            _currentXPath.Push("/tns:OperatingRentA");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "OperatingRentA")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.PTR345 = currentDecimalValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion OperatingRentA
                    
                            #region OperatingInterestA
                            _currentXPath.Push("/tns:OperatingInterestA");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "OperatingInterestA")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.PTR346 = currentDecimalValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion OperatingInterestA
                    
                            #region RoyaltiesA
                            _currentXPath.Push("/tns:RoyaltiesA");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "RoyaltiesA")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.PTR347 = currentDecimalValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion RoyaltiesA
                    
                            #region DepreciationAndAmortisationA
                            _currentXPath.Push("/tns:DepreciationAndAmortisationA");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "DepreciationAndAmortisationA")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.PTR54 = currentDecimalValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion DepreciationAndAmortisationA
                    
                            #region OperatingMotorVehicleA
                            _currentXPath.Push("/tns:OperatingMotorVehicleA");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "OperatingMotorVehicleA")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.PTR348 = currentDecimalValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion OperatingMotorVehicleA
                    
                            #region OperatingMotorVehicleTypeOrClaimMethodC
                            _currentXPath.Push("/tns:OperatingMotorVehicleTypeOrClaimMethodC");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "OperatingMotorVehicleTypeOrClaimMethodC")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        report.PTR56 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion OperatingMotorVehicleTypeOrClaimMethodC
                    
                            #region OperatingRepairsAndMaintenanceA
                            _currentXPath.Push("/tns:OperatingRepairsAndMaintenanceA");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "OperatingRepairsAndMaintenanceA")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.PTR349 = currentDecimalValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion OperatingRepairsAndMaintenanceA
                    
                            #region DeductibleOtherTotalA
                            _currentXPath.Push("/tns:DeductibleOtherTotalA");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "DeductibleOtherTotalA")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.PTR58 = currentDecimalValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion DeductibleOtherTotalA
                    
                            #region ReconciliationAdjustmentTotalA
                            _currentXPath.Push("/tns:ReconciliationAdjustmentTotalA");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "ReconciliationAdjustmentTotalA")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.PTR328 = currentDecimalValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion ReconciliationAdjustmentTotalA
                        } // End of if BusinessExpenses node exists
                
                        _currentXPath.Pop();
                        #endregion BusinessExpenses
                
                        #region PayAsYouGoWithholding
                        _currentXPath.Push("/tns:PayAsYouGoWithholding");
                        //3. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "PayAsYouGoWithholding")
                        {
                            report.RP_Income_BusinessIncomeAndExpenses_PayAsYouGoWithholdingCollectionExists = true;
                            report.RP_Income_BusinessIncomeAndExpenses_PayAsYouGoWithholdingCollectionCount += 1;
                            _found = true; 
                    
                            #region TaxCreditForTaxWithheldWhereABNNotQuotedA
                            _currentXPath.Push("/tns:TaxCreditForTaxWithheldWhereABNNotQuotedA");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "TaxCreditForTaxWithheldWhereABNNotQuotedA")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.PTR63 = currentDecimalValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion TaxCreditForTaxWithheldWhereABNNotQuotedA
                    
                            #region TaxCreditForAmountsWithheldFromForeignResidentsA
                            _currentXPath.Push("/tns:TaxCreditForAmountsWithheldFromForeignResidentsA");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "TaxCreditForAmountsWithheldFromForeignResidentsA")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.PTR350 = currentDecimalValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion TaxCreditForAmountsWithheldFromForeignResidentsA
                        } // End of if PayAsYouGoWithholding node exists
                
                        _currentXPath.Pop();
                        #endregion PayAsYouGoWithholding
                    } // End of if BusinessIncomeAndExpenses node exists
            
                    _currentXPath.Pop();
                    #endregion BusinessIncomeAndExpenses
            
                    #region PartnershipsTrustsPrimaryProduction
                    _currentXPath.Push("/tns:PartnershipsTrustsPrimaryProduction");
                    //3. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "PartnershipsTrustsPrimaryProduction")
                    {
                        report.RP_Income_PartnershipsTrustsPrimaryProductionCollectionExists = true;
                        report.RP_Income_PartnershipsTrustsPrimaryProductionCollectionCount += 1;
                        _found = true; 
                
                        #region PartnershipDistributionGrossA
                        _currentXPath.Push("/tns:PartnershipDistributionGrossA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "PartnershipDistributionGrossA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.PTR65 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion PartnershipDistributionGrossA
                
                        #region TrustShareNetA
                        _currentXPath.Push("/tns:TrustShareNetA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "TrustShareNetA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.PTR66 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion TrustShareNetA
                
                        #region PartyTrustTypeC
                        _currentXPath.Push("/tns:PartyTrustTypeC");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "PartyTrustTypeC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.PTR299 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion PartyTrustTypeC
                
                        #region ExpenseDistributionDeductionsOtherA
                        _currentXPath.Push("/tns:ExpenseDistributionDeductionsOtherA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "ExpenseDistributionDeductionsOtherA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.PTR68 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion ExpenseDistributionDeductionsOtherA
                    } // End of if PartnershipsTrustsPrimaryProduction node exists
            
                    _currentXPath.Pop();
                    #endregion PartnershipsTrustsPrimaryProduction
            
                    #region PartnershipsTrustsNonPrimaryProduction
                    _currentXPath.Push("/tns:PartnershipsTrustsNonPrimaryProduction");
                    //3. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "PartnershipsTrustsNonPrimaryProduction")
                    {
                        report.RP_Income_PartnershipsTrustsNonPrimaryProductionCollectionExists = true;
                        report.RP_Income_PartnershipsTrustsNonPrimaryProductionCollectionCount += 1;
                        _found = true; 
                
                        #region PartnershipDistributionLessForeignIncomeA
                        _currentXPath.Push("/tns:PartnershipDistributionLessForeignIncomeA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "PartnershipDistributionLessForeignIncomeA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.PTR69 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion PartnershipDistributionLessForeignIncomeA
                
                        #region TrustShareNetExcludeNetCapitalGainsAndForeignIncomeAndDistributionFrankedA
                        _currentXPath.Push("/tns:TrustShareNetExcludeNetCapitalGainsAndForeignIncomeAndDistributionFrankedA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "TrustShareNetExcludeNetCapitalGainsAndForeignIncomeAndDistributionFrankedA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.PTR70 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion TrustShareNetExcludeNetCapitalGainsAndForeignIncomeAndDistributionFrankedA
                
                        #region PartyTrustTypeC
                        _currentXPath.Push("/tns:PartyTrustTypeC");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "PartyTrustTypeC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.PTR327 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion PartyTrustTypeC
                
                        #region ExpenseDistributionDeductionsOtherA
                        _currentXPath.Push("/tns:ExpenseDistributionDeductionsOtherA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "ExpenseDistributionDeductionsOtherA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.PTR72 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion ExpenseDistributionDeductionsOtherA
                
                        #region TrustDistributionFrankedA
                        _currentXPath.Push("/tns:TrustDistributionFrankedA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "TrustDistributionFrankedA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.PTR322 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion TrustDistributionFrankedA
                
                        #region TaxDeductionTrustDistributionFrankedA
                        _currentXPath.Push("/tns:TaxDeductionTrustDistributionFrankedA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "TaxDeductionTrustDistributionFrankedA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.PTR323 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion TaxDeductionTrustDistributionFrankedA
                    } // End of if PartnershipsTrustsNonPrimaryProduction node exists
            
                    _currentXPath.Pop();
                    #endregion PartnershipsTrustsNonPrimaryProduction
            
                    #region ShareOfCreditsFromIncome
                    _currentXPath.Push("/tns:ShareOfCreditsFromIncome");
                    //3. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "ShareOfCreditsFromIncome")
                    {
                        report.RP_Income_ShareOfCreditsFromIncomeCollectionExists = true;
                        report.RP_Income_ShareOfCreditsFromIncomeCollectionCount += 1;
                        _found = true; 
                
                        #region TaxPayAsYouGoWithholdingCreditForTaxWithheldWhereABNNotQuotedShareReceivedIndirectlyA
                        _currentXPath.Push("/tns:TaxPayAsYouGoWithholdingCreditForTaxWithheldWhereABNNotQuotedShareReceivedIndirectlyA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "TaxPayAsYouGoWithholdingCreditForTaxWithheldWhereABNNotQuotedShareReceivedIndirectlyA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.PTR73 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion TaxPayAsYouGoWithholdingCreditForTaxWithheldWhereABNNotQuotedShareReceivedIndirectlyA
                
                        #region TaxFrankingCreditsFrankingCreditShareReceivedIndirectlyA
                        _currentXPath.Push("/tns:TaxFrankingCreditsFrankingCreditShareReceivedIndirectlyA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "TaxFrankingCreditsFrankingCreditShareReceivedIndirectlyA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.PTR74 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion TaxFrankingCreditsFrankingCreditShareReceivedIndirectlyA
                
                        #region TaxPayAsYouGoWithholdingCreditForAmountsWithheldFromInterestAndDividendsWhereTFNNotQuotedShareReceivedIndirectlyA
                        _currentXPath.Push("/tns:TaxPayAsYouGoWithholdingCreditForAmountsWithheldFromInterestAndDividendsWhereTFNNotQuotedShareReceivedIndirectlyA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "TaxPayAsYouGoWithholdingCreditForAmountsWithheldFromInterestAndDividendsWhereTFNNotQuotedShareReceivedIndirectlyA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.PTR75 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion TaxPayAsYouGoWithholdingCreditForAmountsWithheldFromInterestAndDividendsWhereTFNNotQuotedShareReceivedIndirectlyA
                
                        #region TaxPayAsYouGoWithholdingCreditForTaxWithheldFromCloselyHeldTrustShareReceivedIndirectlyA
                        _currentXPath.Push("/tns:TaxPayAsYouGoWithholdingCreditForTaxWithheldFromCloselyHeldTrustShareReceivedIndirectlyA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "TaxPayAsYouGoWithholdingCreditForTaxWithheldFromCloselyHeldTrustShareReceivedIndirectlyA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.PTR320 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion TaxPayAsYouGoWithholdingCreditForTaxWithheldFromCloselyHeldTrustShareReceivedIndirectlyA
                
                        #region InternationalDealingsCreditForTaxWithheldFromForeignResidentWithholdingShareReceivedIndirectlyA
                        _currentXPath.Push("/tns:InternationalDealingsCreditForTaxWithheldFromForeignResidentWithholdingShareReceivedIndirectlyA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "InternationalDealingsCreditForTaxWithheldFromForeignResidentWithholdingShareReceivedIndirectlyA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.PTR76 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion InternationalDealingsCreditForTaxWithheldFromForeignResidentWithholdingShareReceivedIndirectlyA
                    } // End of if ShareOfCreditsFromIncome node exists
            
                    _currentXPath.Pop();
                    #endregion ShareOfCreditsFromIncome
            
                    #region Rent
                    _currentXPath.Push("/tns:Rent");
                    //3. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "Rent")
                    {
                        report.RP_Income_RentCollectionExists = true;
                        report.RP_Income_RentCollectionCount += 1;
                        _found = true; 
                
                        #region OperatingIncomeGrossA
                        _currentXPath.Push("/tns:OperatingIncomeGrossA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "OperatingIncomeGrossA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.PTR77 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion OperatingIncomeGrossA
                
                        #region ExpenseInterestA
                        _currentXPath.Push("/tns:ExpenseInterestA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "ExpenseInterestA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.PTR78 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion ExpenseInterestA
                
                        #region ExpenseCapitalWorksDeductionA
                        _currentXPath.Push("/tns:ExpenseCapitalWorksDeductionA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "ExpenseCapitalWorksDeductionA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.PTR79 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion ExpenseCapitalWorksDeductionA
                
                        #region TaxDeductionIncomeDeductionsOtherThanInterestAndCapitalWorksA
                        _currentXPath.Push("/tns:TaxDeductionIncomeDeductionsOtherThanInterestAndCapitalWorksA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "TaxDeductionIncomeDeductionsOtherThanInterestAndCapitalWorksA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.PTR80 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion TaxDeductionIncomeDeductionsOtherThanInterestAndCapitalWorksA
                
                        #region ExpenseBuildToRentCapitalWorksDeductionA
                        _currentXPath.Push("/tns:ExpenseBuildToRentCapitalWorksDeductionA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "ExpenseBuildToRentCapitalWorksDeductionA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.PTR503 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion ExpenseBuildToRentCapitalWorksDeductionA
                    } // End of if Rent node exists
            
                    _currentXPath.Pop();
                    #endregion Rent
            
                    #region ForestryManagedInvestmentSchemeA
                    _currentXPath.Push("/tns:ForestryManagedInvestmentSchemeA");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "ForestryManagedInvestmentSchemeA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.PTR81 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion ForestryManagedInvestmentSchemeA
            
                    #region GrossInterest
                    _currentXPath.Push("/tns:GrossInterest");
                    //3. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "GrossInterest")
                    {
                        report.RP_Income_GrossInterestCollectionExists = true;
                        report.RP_Income_GrossInterestCollectionCount += 1;
                        _found = true; 
                
                        #region A
                        _currentXPath.Push("/tns:A");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "A")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.PTR82 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion A
                
                        #region TaxTFNAmountsWithheldFromGrossInterestA
                        _currentXPath.Push("/tns:TaxTFNAmountsWithheldFromGrossInterestA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "TaxTFNAmountsWithheldFromGrossInterestA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.PTR83 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion TaxTFNAmountsWithheldFromGrossInterestA
                    } // End of if GrossInterest node exists
            
                    _currentXPath.Pop();
                    #endregion GrossInterest
            
                    #region Dividends
                    _currentXPath.Push("/tns:Dividends");
                    //3. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "Dividends")
                    {
                        report.RP_Income_DividendsCollectionExists = true;
                        report.RP_Income_DividendsCollectionCount += 1;
                        _found = true; 
                
                        #region UnfrankedA
                        _currentXPath.Push("/tns:UnfrankedA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "UnfrankedA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.PTR84 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion UnfrankedA
                
                        #region FrankedA
                        _currentXPath.Push("/tns:FrankedA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "FrankedA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.PTR85 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion FrankedA
                
                        #region TaxFrankingCreditsA
                        _currentXPath.Push("/tns:TaxFrankingCreditsA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "TaxFrankingCreditsA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.PTR86 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion TaxFrankingCreditsA
                
                        #region TaxTFNAmountsWithheldA
                        _currentXPath.Push("/tns:TaxTFNAmountsWithheldA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "TaxTFNAmountsWithheldA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.PTR87 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion TaxTFNAmountsWithheldA
                    } // End of if Dividends node exists
            
                    _currentXPath.Pop();
                    #endregion Dividends
            
                    #region OtherAustralianIncome
            
                    _currentXPath.Push("/tns:OtherAustralianIncomeCollection");
                    // 4. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "OtherAustralianIncomeCollection")
                    {
                        report.RP_Income_OtherAustralianIncomeCollection = new List<PTR2025.RP_Income_OtherAustralianIncome>();
                        report.RP_Income_OtherAustralianIncomeCollectionExists = true;
            
                        _found = true;
            
                        while ( ReadToNextElement(reader,false) && reader.LocalName == "OtherAustralianIncome" ) 
                        {
                            _found = true;
            
                            PTR2025.RP_Income_OtherAustralianIncome otherAustralianIncome = new PTR2025.RP_Income_OtherAustralianIncome();
                            report.RP_Income_OtherAustralianIncomeCollection.Add(otherAustralianIncome);
                            report.RP_Income_OtherAustralianIncomeCollectionCount += 1;
                            otherAustralianIncome.OccurrenceIndex = report.RP_Income_OtherAustralianIncomeCollectionCount;
            
                            _currentXPath.Push("/tns:OtherAustralianIncome[" + report.RP_Income_OtherAustralianIncomeCollectionCount + "]");
                    
                            #region De
                            _currentXPath.Push("/tns:De");
                            //6. use case
                            if (ReadToNextElement(reader,true) && reader.LocalName == "De")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        otherAustralianIncome.PTR88 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion De
                    
                            #region A
                            _currentXPath.Push("/tns:A");
                            //6. use case
                            if (ReadToNextElement(reader,true) && reader.LocalName == "A")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            otherAustralianIncome.PTR89 = currentDecimalValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion A
                        _currentXPath.Pop();
                        } // End of while loop for OtherAustralianIncome
                    } // End of if OtherAustralianIncomeCollection node exists
            
                    _currentXPath.Pop();
                    #endregion OtherAustralianIncome
                } // End of if Income node exists
        
                _currentXPath.Pop();
                #endregion Income
        
                #region ExpenseAustralianInvestmentAllowableDeductionA
                _currentXPath.Push("/tns:ExpenseAustralianInvestmentAllowableDeductionA");
                //6. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "ExpenseAustralianInvestmentAllowableDeductionA")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                            {
                                report.PTR90 = currentDecimalValue;
                            }
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion ExpenseAustralianInvestmentAllowableDeductionA
        
                #region ExpenseForestryManagedInvestmentSchemeDeductionA
                _currentXPath.Push("/tns:ExpenseForestryManagedInvestmentSchemeDeductionA");
                //6. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "ExpenseForestryManagedInvestmentSchemeDeductionA")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                            {
                                report.PTR91 = currentDecimalValue;
                            }
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion ExpenseForestryManagedInvestmentSchemeDeductionA
        
                #region OtherDeductions
        
                _currentXPath.Push("/tns:OtherDeductionsCollection");
                // 4. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "OtherDeductionsCollection")
                {
                    report.RP_OtherDeductionsCollection = new List<PTR2025.RP_OtherDeductions>();
                    report.RP_OtherDeductionsCollectionExists = true;
        
                    _found = true;
        
                    while ( ReadToNextElement(reader,false) && reader.LocalName == "OtherDeductions" ) 
                    {
                        _found = true;
        
                        PTR2025.RP_OtherDeductions otherDeductions = new PTR2025.RP_OtherDeductions();
                        report.RP_OtherDeductionsCollection.Add(otherDeductions);
                        report.RP_OtherDeductionsCollectionCount += 1;
                        otherDeductions.OccurrenceIndex = report.RP_OtherDeductionsCollectionCount;
        
                        _currentXPath.Push("/tns:OtherDeductions[" + report.RP_OtherDeductionsCollectionCount + "]");
                
                        #region ExpenseDeductibleOtherT
                        _currentXPath.Push("/tns:ExpenseDeductibleOtherT");
                        //6. use case
                        if (ReadToNextElement(reader,true) && reader.LocalName == "ExpenseDeductibleOtherT")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    otherDeductions.PTR95 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion ExpenseDeductibleOtherT
                
                        #region ExpenseDeductibleOtherA
                        _currentXPath.Push("/tns:ExpenseDeductibleOtherA");
                        //6. use case
                        if (ReadToNextElement(reader,true) && reader.LocalName == "ExpenseDeductibleOtherA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        otherDeductions.PTR96 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion ExpenseDeductibleOtherA
                    _currentXPath.Pop();
                    } // End of while loop for OtherDeductions
                } // End of if OtherDeductionsCollection node exists
        
                _currentXPath.Pop();
                #endregion OtherDeductions
        
                #region IncomeNetA
                _currentXPath.Push("/tns:IncomeNetA");
                //6. use case
                if (ReadToNextElement(reader,true) && reader.LocalName == "IncomeNetA")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                            {
                                report.PTR97 = currentDecimalValue;
                            }
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion IncomeNetA
        
                #region AttributedForeignIncome
                _currentXPath.Push("/tns:AttributedForeignIncome");
                //3. use case
                if (ReadToNextElement(reader,true) && reader.LocalName == "AttributedForeignIncome")
                {
                    report.RP_AttributedForeignIncomeCollectionExists = true;
                    report.RP_AttributedForeignIncomeCollectionCount += 1;
                    _found = true; 
            
                    #region InternationalDealingsTrustorControlledCompanyorTransferorTrustI
                    _currentXPath.Push("/tns:InternationalDealingsTrustorControlledCompanyorTransferorTrustI");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "InternationalDealingsTrustorControlledCompanyorTransferorTrustI")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.PTR102 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion InternationalDealingsTrustorControlledCompanyorTransferorTrustI
            
                    #region ListedCountryInternationalDealingsA
                    _currentXPath.Push("/tns:ListedCountryInternationalDealingsA");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "ListedCountryInternationalDealingsA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.PTR98 = currentDecimalValue;
                                }
                                report.RP_AttributedForeignIncome_ListedCountryCollectionExists = true;
                                report.RP_AttributedForeignIncome_ListedCountryCollectionCount += 1;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion ListedCountryInternationalDealingsA
            
                    #region UnlistedCountryInternationalDealingsA
                    _currentXPath.Push("/tns:UnlistedCountryInternationalDealingsA");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "UnlistedCountryInternationalDealingsA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.PTR100 = currentDecimalValue;
                                }
                                report.RP_AttributedForeignIncome_UnlistedCountryCollectionExists = true;
                                report.RP_AttributedForeignIncome_UnlistedCountryCollectionCount += 1;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion UnlistedCountryInternationalDealingsA
                } // End of if AttributedForeignIncome node exists
        
                _currentXPath.Pop();
                #endregion AttributedForeignIncome
        
                #region OtherAssessableForeignSourceIncome
                _currentXPath.Push("/tns:OtherAssessableForeignSourceIncome");
                //3. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "OtherAssessableForeignSourceIncome")
                {
                    report.RP_OtherAssessableForeignSourceIncomeCollectionExists = true;
                    report.RP_OtherAssessableForeignSourceIncomeCollectionCount += 1;
                    _found = true; 
            
                    #region A
                    _currentXPath.Push("/tns:A");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "A")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.PTR104 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion A
            
                    #region NetA
                    _currentXPath.Push("/tns:NetA");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "NetA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.PTR105 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion NetA
            
                    #region InternationalDealingsTaxOffsetA
                    _currentXPath.Push("/tns:InternationalDealingsTaxOffsetA");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "InternationalDealingsTaxOffsetA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.PTR106 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion InternationalDealingsTaxOffsetA
            
                    #region IncomeTaxFrankingCreditsReceivedFromNewZealandCompaniesA
                    _currentXPath.Push("/tns:IncomeTaxFrankingCreditsReceivedFromNewZealandCompaniesA");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "IncomeTaxFrankingCreditsReceivedFromNewZealandCompaniesA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.PTR107 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion IncomeTaxFrankingCreditsReceivedFromNewZealandCompaniesA
                } // End of if OtherAssessableForeignSourceIncome node exists
        
                _currentXPath.Pop();
                #endregion OtherAssessableForeignSourceIncome
        
                #region OverseasTransactions
                _currentXPath.Push("/tns:OverseasTransactions");
                //3. use case
                if (ReadToNextElement(reader,true) && reader.LocalName == "OverseasTransactions")
                {
                    report.RP_OverseasTransactionsCollectionExists = true;
                    report.RP_OverseasTransactionsCollectionCount += 1;
                    _found = true; 
            
                    #region InternationalDealingsRelatedPartiesTransactionsExcessAggregateValueI
                    _currentXPath.Push("/tns:InternationalDealingsRelatedPartiesTransactionsExcessAggregateValueI");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "InternationalDealingsRelatedPartiesTransactionsExcessAggregateValueI")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.PTR108 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion InternationalDealingsRelatedPartiesTransactionsExcessAggregateValueI
            
                    #region LiabilitiesThinCapitalisationProvisionsAppliedI
                    _currentXPath.Push("/tns:LiabilitiesThinCapitalisationProvisionsAppliedI");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "LiabilitiesThinCapitalisationProvisionsAppliedI")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.PTR277 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion LiabilitiesThinCapitalisationProvisionsAppliedI
            
                    #region ExpenseInterestA
                    _currentXPath.Push("/tns:ExpenseInterestA");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "ExpenseInterestA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.PTR137 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion ExpenseInterestA
            
                    #region ExpenseRoyaltiesA
                    _currentXPath.Push("/tns:ExpenseRoyaltiesA");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "ExpenseRoyaltiesA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.PTR138 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion ExpenseRoyaltiesA
                } // End of if OverseasTransactions node exists
        
                _currentXPath.Pop();
                #endregion OverseasTransactions
        
                #region InternationalDealingsTransactionsWithTaxHavenCountriesI
                _currentXPath.Push("/tns:InternationalDealingsTransactionsWithTaxHavenCountriesI");
                //6. use case
                if (ReadToNextElement(reader,true) && reader.LocalName == "InternationalDealingsTransactionsWithTaxHavenCountriesI")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.PTR109 = currentBooleanValue;
                            }
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion InternationalDealingsTransactionsWithTaxHavenCountriesI
        
                #region PersonalServicesIncome
                _currentXPath.Push("/tns:PersonalServicesIncome");
                //3. use case
                if (ReadToNextElement(reader,true) && reader.LocalName == "PersonalServicesIncome")
                {
                    report.RP_PersonalServicesIncomeCollectionExists = true;
                    report.RP_PersonalServicesIncomeCollectionCount += 1;
                    _found = true; 
            
                    #region IncludedI
                    _currentXPath.Push("/tns:IncludedI");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "IncludedI")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.PTR110 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion IncludedI
            
                    #region TotalA
                    _currentXPath.Push("/tns:TotalA");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "TotalA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.PTR265 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion TotalA
            
                    #region ExpenseDeductionA
                    _currentXPath.Push("/tns:ExpenseDeductionA");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "ExpenseDeductionA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.PTR266 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion ExpenseDeductionA
            
                    #region IncomeTaxIndividualResultsTestSatisfiedI
                    _currentXPath.Push("/tns:IncomeTaxIndividualResultsTestSatisfiedI");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "IncomeTaxIndividualResultsTestSatisfiedI")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.PTR268 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion IncomeTaxIndividualResultsTestSatisfiedI
            
                    #region IncomeTaxIndividualPersonalServiceBusinessDeterminationHeldI
                    _currentXPath.Push("/tns:IncomeTaxIndividualPersonalServiceBusinessDeterminationHeldI");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "IncomeTaxIndividualPersonalServiceBusinessDeterminationHeldI")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.PTR271 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion IncomeTaxIndividualPersonalServiceBusinessDeterminationHeldI
            
                    #region IncomeTaxUnrelatedClientsTestSatisfiedI
                    _currentXPath.Push("/tns:IncomeTaxUnrelatedClientsTestSatisfiedI");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "IncomeTaxUnrelatedClientsTestSatisfiedI")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.PTR276 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion IncomeTaxUnrelatedClientsTestSatisfiedI
            
                    #region IncomeTaxEmploymentTestSatisfiedI
                    _currentXPath.Push("/tns:IncomeTaxEmploymentTestSatisfiedI");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "IncomeTaxEmploymentTestSatisfiedI")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.PTR272 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion IncomeTaxEmploymentTestSatisfiedI
            
                    #region IncomeTaxBusinessPremisesTestSatisfiedI
                    _currentXPath.Push("/tns:IncomeTaxBusinessPremisesTestSatisfiedI");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "IncomeTaxBusinessPremisesTestSatisfiedI")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.PTR275 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion IncomeTaxBusinessPremisesTestSatisfiedI
                } // End of if PersonalServicesIncome node exists
        
                _currentXPath.Pop();
                #endregion PersonalServicesIncome
        
                #region TaxationOfFinancialArrangements
                _currentXPath.Push("/tns:TaxationOfFinancialArrangements");
                //3. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "TaxationOfFinancialArrangements")
                {
                    report.RP_TaxationOfFinancialArrangementsCollectionExists = true;
                    report.RP_TaxationOfFinancialArrangementsCollectionCount += 1;
                    _found = true; 
            
                    #region IncomeGainsTotalA
                    _currentXPath.Push("/tns:IncomeGainsTotalA");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "IncomeGainsTotalA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.PTR251 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion IncomeGainsTotalA
            
                    #region ExpenseLossesTotalA
                    _currentXPath.Push("/tns:ExpenseLossesTotalA");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "ExpenseLossesTotalA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.PTR252 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion ExpenseLossesTotalA
                } // End of if TaxationOfFinancialArrangements node exists
        
                _currentXPath.Pop();
                #endregion TaxationOfFinancialArrangements
        
                #region NCMIBusinessIncome
                _currentXPath.Push("/tns:NCMIBusinessIncome");
                //3. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "NCMIBusinessIncome")
                {
                    report.RP_NCMIBusinessIncomeCollectionExists = true;
                    report.RP_NCMIBusinessIncomeCollectionCount += 1;
                    _found = true; 
            
                    #region NCMIBusinessIncomePrimaryProduction
                    _currentXPath.Push("/tns:NCMIBusinessIncomePrimaryProduction");
                    //3. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "NCMIBusinessIncomePrimaryProduction")
                    {
                        report.RP_NCMIBusinessIncome_NCMIBusinessIncomePrimaryProductionCollectionExists = true;
                        report.RP_NCMIBusinessIncome_NCMIBusinessIncomePrimaryProductionCollectionCount += 1;
                        _found = true; 
                
                        #region NonConcessionalManagedInvestmentTrustA
                        _currentXPath.Push("/tns:NonConcessionalManagedInvestmentTrustA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "NonConcessionalManagedInvestmentTrustA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.PTR356 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion NonConcessionalManagedInvestmentTrustA
                
                        #region ExcludedFromNonConcessionalManagedInvestmentTrustA
                        _currentXPath.Push("/tns:ExcludedFromNonConcessionalManagedInvestmentTrustA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "ExcludedFromNonConcessionalManagedInvestmentTrustA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.PTR357 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion ExcludedFromNonConcessionalManagedInvestmentTrustA
                    } // End of if NCMIBusinessIncomePrimaryProduction node exists
            
                    _currentXPath.Pop();
                    #endregion NCMIBusinessIncomePrimaryProduction
            
                    #region NCMIBusinessIncomeNonPrimaryProduction
                    _currentXPath.Push("/tns:NCMIBusinessIncomeNonPrimaryProduction");
                    //3. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "NCMIBusinessIncomeNonPrimaryProduction")
                    {
                        report.RP_NCMIBusinessIncome_NCMIBusinessIncomeNonPrimaryProductionCollectionExists = true;
                        report.RP_NCMIBusinessIncome_NCMIBusinessIncomeNonPrimaryProductionCollectionCount += 1;
                        _found = true; 
                
                        #region NonConcessionalManagedInvestmentTrustA
                        _currentXPath.Push("/tns:NonConcessionalManagedInvestmentTrustA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "NonConcessionalManagedInvestmentTrustA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.PTR358 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion NonConcessionalManagedInvestmentTrustA
                
                        #region ExcludedFromNonConcessionalManagedInvestmentTrustA
                        _currentXPath.Push("/tns:ExcludedFromNonConcessionalManagedInvestmentTrustA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "ExcludedFromNonConcessionalManagedInvestmentTrustA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.PTR359 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion ExcludedFromNonConcessionalManagedInvestmentTrustA
                    } // End of if NCMIBusinessIncomeNonPrimaryProduction node exists
            
                    _currentXPath.Pop();
                    #endregion NCMIBusinessIncomeNonPrimaryProduction
                } // End of if NCMIBusinessIncome node exists
        
                _currentXPath.Pop();
                #endregion NCMIBusinessIncome
        
                #region NCMIPartnershipsAndTrusts
                _currentXPath.Push("/tns:NCMIPartnershipsAndTrusts");
                //3. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "NCMIPartnershipsAndTrusts")
                {
                    report.RP_NCMIPartnershipsAndTrustsCollectionExists = true;
                    report.RP_NCMIPartnershipsAndTrustsCollectionCount += 1;
                    _found = true; 
            
                    #region NCMIPartnershipsAndTrustsPrimaryProduction
                    _currentXPath.Push("/tns:NCMIPartnershipsAndTrustsPrimaryProduction");
                    //3. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "NCMIPartnershipsAndTrustsPrimaryProduction")
                    {
                        report.RP_NCMIPartnershipsAndTrusts_NCMIPartnershipsAndTrustsPrimaryProductionCollectionExists = true;
                        report.RP_NCMIPartnershipsAndTrusts_NCMIPartnershipsAndTrustsPrimaryProductionCollectionCount += 1;
                        _found = true; 
                
                        #region IncomeNonConcessionalManagedInvestmentTrustA
                        _currentXPath.Push("/tns:IncomeNonConcessionalManagedInvestmentTrustA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "IncomeNonConcessionalManagedInvestmentTrustA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.PTR360 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion IncomeNonConcessionalManagedInvestmentTrustA
                
                        #region IncomeExcludedFromNonConcessionalManagedInvestmentTrustA
                        _currentXPath.Push("/tns:IncomeExcludedFromNonConcessionalManagedInvestmentTrustA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "IncomeExcludedFromNonConcessionalManagedInvestmentTrustA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.PTR361 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion IncomeExcludedFromNonConcessionalManagedInvestmentTrustA
                
                        #region PrimaryProductionShareOfNetIncome
                        _currentXPath.Push("/tns:PrimaryProductionShareOfNetIncome");
                        //3. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "PrimaryProductionShareOfNetIncome")
                        {
                            report.RP_NCMIPartnershipsAndTrusts_NCMIPartnershipsAndTrustsPrimaryProduction_PrimaryProductionShareOfNetIncomeCollectionExists = true;
                            report.RP_NCMIPartnershipsAndTrusts_NCMIPartnershipsAndTrustsPrimaryProduction_PrimaryProductionShareOfNetIncomeCollectionCount += 1;
                            _found = true; 
                    
                            #region NonConcessionalManagedInvestmentTrustA
                            _currentXPath.Push("/tns:NonConcessionalManagedInvestmentTrustA");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "NonConcessionalManagedInvestmentTrustA")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.PTR362 = currentDecimalValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion NonConcessionalManagedInvestmentTrustA
                    
                            #region ExcludedFromNonConcessionalManagedInvestmentTrustA
                            _currentXPath.Push("/tns:ExcludedFromNonConcessionalManagedInvestmentTrustA");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "ExcludedFromNonConcessionalManagedInvestmentTrustA")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.PTR363 = currentDecimalValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion ExcludedFromNonConcessionalManagedInvestmentTrustA
                        } // End of if PrimaryProductionShareOfNetIncome node exists
                
                        _currentXPath.Pop();
                        #endregion PrimaryProductionShareOfNetIncome
                    } // End of if NCMIPartnershipsAndTrustsPrimaryProduction node exists
            
                    _currentXPath.Pop();
                    #endregion NCMIPartnershipsAndTrustsPrimaryProduction
            
                    #region NCMIPartnershipsAndTrustsNonPrimaryProduction
                    _currentXPath.Push("/tns:NCMIPartnershipsAndTrustsNonPrimaryProduction");
                    //3. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "NCMIPartnershipsAndTrustsNonPrimaryProduction")
                    {
                        report.RP_NCMIPartnershipsAndTrusts_NCMIPartnershipsAndTrustsNonPrimaryProductionCollectionExists = true;
                        report.RP_NCMIPartnershipsAndTrusts_NCMIPartnershipsAndTrustsNonPrimaryProductionCollectionCount += 1;
                        _found = true; 
                
                        #region IncomeNonConcessionalManagedInvestmentTrustA
                        _currentXPath.Push("/tns:IncomeNonConcessionalManagedInvestmentTrustA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "IncomeNonConcessionalManagedInvestmentTrustA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.PTR364 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion IncomeNonConcessionalManagedInvestmentTrustA
                
                        #region IncomeExcludedFromNonConcessionalManagedInvestmentTrustA
                        _currentXPath.Push("/tns:IncomeExcludedFromNonConcessionalManagedInvestmentTrustA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "IncomeExcludedFromNonConcessionalManagedInvestmentTrustA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.PTR365 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion IncomeExcludedFromNonConcessionalManagedInvestmentTrustA
                
                        #region NonPrimaryProductionShareOfNetIncome
                        _currentXPath.Push("/tns:NonPrimaryProductionShareOfNetIncome");
                        //3. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "NonPrimaryProductionShareOfNetIncome")
                        {
                            report.RP_NCMIPartnershipsAndTrusts_NCMIPartnershipsAndTrustsNonPrimaryProduction_NonPrimaryProductionShareOfNetIncomeCollectionExists = true;
                            report.RP_NCMIPartnershipsAndTrusts_NCMIPartnershipsAndTrustsNonPrimaryProduction_NonPrimaryProductionShareOfNetIncomeCollectionCount += 1;
                            _found = true; 
                    
                            #region NonConcessionalManagedInvestmentTrustA
                            _currentXPath.Push("/tns:NonConcessionalManagedInvestmentTrustA");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "NonConcessionalManagedInvestmentTrustA")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.PTR366 = currentDecimalValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion NonConcessionalManagedInvestmentTrustA
                    
                            #region ExcludedFromNonConcessionalManagedInvestmentTrustA
                            _currentXPath.Push("/tns:ExcludedFromNonConcessionalManagedInvestmentTrustA");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "ExcludedFromNonConcessionalManagedInvestmentTrustA")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.PTR367 = currentDecimalValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion ExcludedFromNonConcessionalManagedInvestmentTrustA
                        } // End of if NonPrimaryProductionShareOfNetIncome node exists
                
                        _currentXPath.Pop();
                        #endregion NonPrimaryProductionShareOfNetIncome
                    } // End of if NCMIPartnershipsAndTrustsNonPrimaryProduction node exists
            
                    _currentXPath.Pop();
                    #endregion NCMIPartnershipsAndTrustsNonPrimaryProduction
                } // End of if NCMIPartnershipsAndTrusts node exists
        
                _currentXPath.Pop();
                #endregion NCMIPartnershipsAndTrusts
        
                #region KeyFinancialInformation
                _currentXPath.Push("/tns:KeyFinancialInformation");
                //3. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "KeyFinancialInformation")
                {
                    report.RP_KeyFinancialInformationCollectionExists = true;
                    report.RP_KeyFinancialInformationCollectionCount += 1;
                    _found = true; 
            
                    #region AssetsCurrentTotalA
                    _currentXPath.Push("/tns:AssetsCurrentTotalA");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "AssetsCurrentTotalA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.PTR111 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion AssetsCurrentTotalA
            
                    #region AssetsTotalA
                    _currentXPath.Push("/tns:AssetsTotalA");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "AssetsTotalA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.PTR112 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion AssetsTotalA
            
                    #region LiabilitiesCurrentTotalA
                    _currentXPath.Push("/tns:LiabilitiesCurrentTotalA");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "LiabilitiesCurrentTotalA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.PTR113 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion LiabilitiesCurrentTotalA
            
                    #region LiabilitiesTotalA
                    _currentXPath.Push("/tns:LiabilitiesTotalA");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "LiabilitiesTotalA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.PTR114 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion LiabilitiesTotalA
                } // End of if KeyFinancialInformation node exists
        
                _currentXPath.Pop();
                #endregion KeyFinancialInformation
        
                #region BusinessNameOrganisationNameDetailsOrganisationalNameT
                _currentXPath.Push("/tns:BusinessNameOrganisationNameDetailsOrganisationalNameT");
                //6. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "BusinessNameOrganisationNameDetailsOrganisationalNameT")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.PTR116 = currentValue;
                            report.RP_BusinessNameCollectionExists = true;
                            report.RP_BusinessNameCollectionCount += 1;
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion BusinessNameOrganisationNameDetailsOrganisationalNameT
        
                #region BusinessAddressDetails
                _currentXPath.Push("/tns:BusinessAddressDetails");
                //3. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "BusinessAddressDetails")
                {
                    report.RP_BusinessAddressDetailsCollectionExists = true;
                    report.RP_BusinessAddressDetailsCollectionCount += 1;
                    _found = true; 
            
                    #region OverseasAddressI
                    _currentXPath.Push("/tns:OverseasAddressI");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "OverseasAddressI")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.PTR222 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion OverseasAddressI
            
                    #region Line1T
                    _currentXPath.Push("/tns:Line1T");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "Line1T")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.PTR117 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion Line1T
            
                    #region Line2T
                    _currentXPath.Push("/tns:Line2T");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "Line2T")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.PTR213 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion Line2T
            
                    #region LocalityNameT
                    _currentXPath.Push("/tns:LocalityNameT");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "LocalityNameT")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.PTR118 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion LocalityNameT
            
                    #region PostcodeT
                    _currentXPath.Push("/tns:PostcodeT");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "PostcodeT")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.PTR120 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion PostcodeT
            
                    #region StateOrTerritoryC
                    _currentXPath.Push("/tns:StateOrTerritoryC");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "StateOrTerritoryC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.PTR119 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion StateOrTerritoryC
            
                    #region CountryC
                    _currentXPath.Push("/tns:CountryC");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "CountryC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.PTR304 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion CountryC
                } // End of if BusinessAddressDetails node exists
        
                _currentXPath.Pop();
                #endregion BusinessAddressDetails
        
                #region BusinessAndProfessionaltems
                _currentXPath.Push("/tns:BusinessAndProfessionaltems");
                //3. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "BusinessAndProfessionaltems")
                {
                    report.RP_BusinessAndProfessionaltemsCollectionExists = true;
                    report.RP_BusinessAndProfessionaltemsCollectionCount += 1;
                    _found = true; 
            
                    #region AssetsInventoriesTotalA
                    _currentXPath.Push("/tns:AssetsInventoriesTotalA");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "AssetsInventoriesTotalA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.PTR121 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion AssetsInventoriesTotalA
            
                    #region ExpensePurchasesAndOtherCostsA
                    _currentXPath.Push("/tns:ExpensePurchasesAndOtherCostsA");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "ExpensePurchasesAndOtherCostsA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.PTR122 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion ExpensePurchasesAndOtherCostsA
            
                    #region RemunerationWagesAndSalariesTotalA
                    _currentXPath.Push("/tns:RemunerationWagesAndSalariesTotalA");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "RemunerationWagesAndSalariesTotalA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.PTR127 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion RemunerationWagesAndSalariesTotalA
            
                    #region RemunerationWagesAndSalariesActionC
                    _currentXPath.Push("/tns:RemunerationWagesAndSalariesActionC");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "RemunerationWagesAndSalariesActionC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.PTR128 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion RemunerationWagesAndSalariesActionC
            
                    #region RemunerationPaymentToRelatedPartiesGrossA
                    _currentXPath.Push("/tns:RemunerationPaymentToRelatedPartiesGrossA");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "RemunerationPaymentToRelatedPartiesGrossA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.PTR129 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion RemunerationPaymentToRelatedPartiesGrossA
            
                    #region RemunerationFringeBenefitsEmployeeContributionA
                    _currentXPath.Push("/tns:RemunerationFringeBenefitsEmployeeContributionA");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "RemunerationFringeBenefitsEmployeeContributionA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.PTR136 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion RemunerationFringeBenefitsEmployeeContributionA
            
                    #region ElectionsTradingStockI
                    _currentXPath.Push("/tns:ElectionsTradingStockI");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "ElectionsTradingStockI")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.PTR140 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion ElectionsTradingStockI
            
                    #region Closing
                    _currentXPath.Push("/tns:Closing");
                    //3. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "Closing")
                    {
                        report.RP_BusinessAndProfessionaltems_ClosingCollectionExists = true;
                        report.RP_BusinessAndProfessionaltems_ClosingCollectionCount += 1;
                        _found = true; 
                
                        #region AssetsInventoriesTotalA
                        _currentXPath.Push("/tns:AssetsInventoriesTotalA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "AssetsInventoriesTotalA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.PTR123 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion AssetsInventoriesTotalA
                
                        #region AssetsInventoriesValuationMethodC
                        _currentXPath.Push("/tns:AssetsInventoriesValuationMethodC");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "AssetsInventoriesValuationMethodC")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.PTR124 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion AssetsInventoriesValuationMethodC
                
                        #region AssetsTradeandReceivablesOtherA
                        _currentXPath.Push("/tns:AssetsTradeandReceivablesOtherA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "AssetsTradeandReceivablesOtherA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.PTR125 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion AssetsTradeandReceivablesOtherA
                
                        #region LiabilitiesTradeAndOtherPayablesTotalA
                        _currentXPath.Push("/tns:LiabilitiesTradeAndOtherPayablesTotalA");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "LiabilitiesTradeAndOtherPayablesTotalA")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.PTR126 = currentDecimalValue;
                                    }
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion LiabilitiesTradeAndOtherPayablesTotalA
                    } // End of if Closing node exists
            
                    _currentXPath.Pop();
                    #endregion Closing
                } // End of if BusinessAndProfessionaltems node exists
        
                _currentXPath.Pop();
                #endregion BusinessAndProfessionaltems
        
                #region AggregatedTurnover
                _currentXPath.Push("/tns:AggregatedTurnover");
                //3. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "AggregatedTurnover")
                {
                    report.RP_AggregatedTurnoverCollectionExists = true;
                    report.RP_AggregatedTurnoverCollectionCount += 1;
                    _found = true; 
            
                    #region IncomeRangeC
                    _currentXPath.Push("/tns:IncomeRangeC");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "IncomeRangeC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.PTR411 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion IncomeRangeC
            
                    #region IncomeA
                    _currentXPath.Push("/tns:IncomeA");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "IncomeA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.PTR412 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion IncomeA
                } // End of if AggregatedTurnover node exists
        
                _currentXPath.Pop();
                #endregion AggregatedTurnover
        
                #region CapitalAllowancesDepreciatingAssetsFirst
                _currentXPath.Push("/tns:CapitalAllowancesDepreciatingAssetsFirst");
                //3. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "CapitalAllowancesDepreciatingAssetsFirst")
                {
                    report.RP_CapitalAllowancesDepreciatingAssetsFirstCollectionExists = true;
                    report.RP_CapitalAllowancesDepreciatingAssetsFirstCollectionCount += 1;
                    _found = true; 
            
                    #region ExpenseIntangibleFirstDeductedA
                    _currentXPath.Push("/tns:ExpenseIntangibleFirstDeductedA");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "ExpenseIntangibleFirstDeductedA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.PTR130 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion ExpenseIntangibleFirstDeductedA
            
                    #region ExpenseOtherFirstDeductedA
                    _currentXPath.Push("/tns:ExpenseOtherFirstDeductedA");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "ExpenseOtherFirstDeductedA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.PTR131 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion ExpenseOtherFirstDeductedA
            
                    #region PropertyPlantAndEquipmentFirstDeductedSelfAssessedEffectiveLifeI
                    _currentXPath.Push("/tns:PropertyPlantAndEquipmentFirstDeductedSelfAssessedEffectiveLifeI");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "PropertyPlantAndEquipmentFirstDeductedSelfAssessedEffectiveLifeI")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.PTR283 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion PropertyPlantAndEquipmentFirstDeductedSelfAssessedEffectiveLifeI
                } // End of if CapitalAllowancesDepreciatingAssetsFirst node exists
        
                _currentXPath.Pop();
                #endregion CapitalAllowancesDepreciatingAssetsFirst
        
                #region CapitalAllowancesDepreciatingAssetsAll
                _currentXPath.Push("/tns:CapitalAllowancesDepreciatingAssetsAll");
                //3. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "CapitalAllowancesDepreciatingAssetsAll")
                {
                    report.RP_CapitalAllowancesDepreciatingAssetsAllCollectionExists = true;
                    report.RP_CapitalAllowancesDepreciatingAssetsAllCollectionCount += 1;
                    _found = true; 
            
                    #region PropertyPlantAndEquipmentEffectiveLifeRecalculationI
                    _currentXPath.Push("/tns:PropertyPlantAndEquipmentEffectiveLifeRecalculationI");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "PropertyPlantAndEquipmentEffectiveLifeRecalculationI")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.PTR278 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion PropertyPlantAndEquipmentEffectiveLifeRecalculationI
            
                    #region AdjustableValuesTotalA
                    _currentXPath.Push("/tns:AdjustableValuesTotalA");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "AdjustableValuesTotalA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.PTR279 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion AdjustableValuesTotalA
            
                    #region IncomeIntangibleBalancingAdjustmentA
                    _currentXPath.Push("/tns:IncomeIntangibleBalancingAdjustmentA");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "IncomeIntangibleBalancingAdjustmentA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.PTR282 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion IncomeIntangibleBalancingAdjustmentA
            
                    #region ExpenseIntangibleBalancingAdjustmentA
                    _currentXPath.Push("/tns:ExpenseIntangibleBalancingAdjustmentA");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "ExpenseIntangibleBalancingAdjustmentA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.PTR281 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion ExpenseIntangibleBalancingAdjustmentA
            
                    #region IntangibleTerminationValueA
                    _currentXPath.Push("/tns:IntangibleTerminationValueA");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "IntangibleTerminationValueA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.PTR132 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion IntangibleTerminationValueA
            
                    #region OtherTerminationValueA
                    _currentXPath.Push("/tns:OtherTerminationValueA");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "OtherTerminationValueA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.PTR133 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion OtherTerminationValueA
            
                    #region ExpenseSubsequentYearAcceleratedDepreciationDeductionsForAssetsA
                    _currentXPath.Push("/tns:ExpenseSubsequentYearAcceleratedDepreciationDeductionsForAssetsA");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "ExpenseSubsequentYearAcceleratedDepreciationDeductionsForAssetsA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.PTR401 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion ExpenseSubsequentYearAcceleratedDepreciationDeductionsForAssetsA
            
                    #region ExpenseProjectPoolAllowableDeductionA
                    _currentXPath.Push("/tns:ExpenseProjectPoolAllowableDeductionA");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "ExpenseProjectPoolAllowableDeductionA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.PTR134 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion ExpenseProjectPoolAllowableDeductionA
            
                    #region ExpenseCapitalExpenditureSpecifiedAllowableDeductionA
                    _currentXPath.Push("/tns:ExpenseCapitalExpenditureSpecifiedAllowableDeductionA");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "ExpenseCapitalExpenditureSpecifiedAllowableDeductionA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.PTR135 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion ExpenseCapitalExpenditureSpecifiedAllowableDeductionA
                } // End of if CapitalAllowancesDepreciatingAssetsAll node exists
        
                _currentXPath.Pop();
                #endregion CapitalAllowancesDepreciatingAssetsAll
        
                #region LowCostDeductionExpenseDepreciatingAssetsSmallBusinessEntityA
                _currentXPath.Push("/tns:LowCostDeductionExpenseDepreciatingAssetsSmallBusinessEntityA");
                //6. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "LowCostDeductionExpenseDepreciatingAssetsSmallBusinessEntityA")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                            {
                                report.PTR141 = currentDecimalValue;
                            }
                            report.RP_LowCostDeductionCollectionExists = true;
                            report.RP_LowCostDeductionCollectionCount += 1;
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion LowCostDeductionExpenseDepreciatingAssetsSmallBusinessEntityA
        
                #region GeneralPoolDeductionExpenseDepreciatingAssetsSmallBusinessEntityA
                _currentXPath.Push("/tns:GeneralPoolDeductionExpenseDepreciatingAssetsSmallBusinessEntityA");
                //6. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "GeneralPoolDeductionExpenseDepreciatingAssetsSmallBusinessEntityA")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                            {
                                report.PTR142 = currentDecimalValue;
                            }
                            report.RP_GeneralPoolDeductionCollectionExists = true;
                            report.RP_GeneralPoolDeductionCollectionCount += 1;
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion GeneralPoolDeductionExpenseDepreciatingAssetsSmallBusinessEntityA
        
                #region TaxOffsetClaimNationalRentalAffordabilitySchemeEntitlementA
                _currentXPath.Push("/tns:TaxOffsetClaimNationalRentalAffordabilitySchemeEntitlementA");
                //6. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "TaxOffsetClaimNationalRentalAffordabilitySchemeEntitlementA")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                            {
                                report.PTR248 = currentDecimalValue;
                            }
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion TaxOffsetClaimNationalRentalAffordabilitySchemeEntitlementA
        
                #region IncomeTests
                _currentXPath.Push("/tns:IncomeTests");
                //3. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "IncomeTests")
                {
                    report.RP_IncomeTestsCollectionExists = true;
                    report.RP_IncomeTestsCollectionCount += 1;
                    _found = true; 
            
                    #region InvestmentFinancialIncomeOrLossNetA
                    _currentXPath.Push("/tns:InvestmentFinancialIncomeOrLossNetA");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "InvestmentFinancialIncomeOrLossNetA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.PTR254 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion InvestmentFinancialIncomeOrLossNetA
            
                    #region RentalIncomeOrLossNetA
                    _currentXPath.Push("/tns:RentalIncomeOrLossNetA");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "RentalIncomeOrLossNetA")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.PTR255 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion RentalIncomeOrLossNetA
                } // End of if IncomeTests node exists
        
                _currentXPath.Pop();
                #endregion IncomeTests
        
                #region StatementOfDistribution
        
                _currentXPath.Push("/tns:StatementOfDistributionCollection");
                // 4. use case
                if (ReadToNextElement(reader,true) && reader.LocalName == "StatementOfDistributionCollection")
                {
                    report.RP_StatementOfDistributionCollection = new List<PTR2025.RP_StatementOfDistribution>();
                    report.RP_StatementOfDistributionCollectionExists = true;
        
                    _found = true;
        
                    while ( ReadToNextElement(reader,true) && reader.LocalName == "StatementOfDistribution" ) 
                    {
                        _found = true;
        
                        PTR2025.RP_StatementOfDistribution statementOfDistribution = new PTR2025.RP_StatementOfDistribution();
                        report.RP_StatementOfDistributionCollection.Add(statementOfDistribution);
                        report.RP_StatementOfDistributionCollectionCount += 1;
                        statementOfDistribution.OccurrenceIndex = report.RP_StatementOfDistributionCollectionCount;
        
                        _currentXPath.Push("/tns:StatementOfDistribution[" + report.RP_StatementOfDistributionCollectionCount + "]");
                
                        #region TaxFileNumberId
                        _currentXPath.Push("/tns:TaxFileNumberId");
                        //6. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "TaxFileNumberId")
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    statementOfDistribution.PTR157 = currentValue;
                                }
                            }
                            _found = true;
                        }
                
                        _currentXPath.Pop();
                        #endregion TaxFileNumberId
                
                        #region NonIndividualPartnerDetails
                        _currentXPath.Push("/tns:NonIndividualPartnerDetails");
                        //3. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "NonIndividualPartnerDetails")
                        {
                            statementOfDistribution.RP_StatementOfDistribution_NonIndividualPartnerDetailsCollectionExists = true;
                            statementOfDistribution.RP_StatementOfDistribution_NonIndividualPartnerDetailsCollectionCount += 1;
                            _found = true; 
                    
                            #region OrganisationNameDetailsOrganisationalNameT
                            _currentXPath.Push("/tns:OrganisationNameDetailsOrganisationalNameT");
                            //6. use case
                            if (ReadToNextElement(reader,true) && reader.LocalName == "OrganisationNameDetailsOrganisationalNameT")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        statementOfDistribution.PTR149 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion OrganisationNameDetailsOrganisationalNameT
                    
                            #region AustralianBusinessNumberId
                            _currentXPath.Push("/tns:AustralianBusinessNumberId");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "AustralianBusinessNumberId")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        statementOfDistribution.PTR285 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion AustralianBusinessNumberId
                        } // End of if NonIndividualPartnerDetails node exists
                
                        _currentXPath.Pop();
                        #endregion NonIndividualPartnerDetails
                
                        #region IndividualPartnerDetails
                        _currentXPath.Push("/tns:IndividualPartnerDetails");
                        //3. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "IndividualPartnerDetails")
                        {
                            statementOfDistribution.RP_StatementOfDistribution_IndividualPartnerDetailsCollectionExists = true;
                            statementOfDistribution.RP_StatementOfDistribution_IndividualPartnerDetailsCollectionCount += 1;
                            _found = true; 
                    
                            #region PersonNameDetailsTitleT
                            _currentXPath.Push("/tns:PersonNameDetailsTitleT");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "PersonNameDetailsTitleT")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        statementOfDistribution.PTR311 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion PersonNameDetailsTitleT
                    
                            #region PersonNameDetailsNameSuffixT
                            _currentXPath.Push("/tns:PersonNameDetailsNameSuffixT");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "PersonNameDetailsNameSuffixT")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        statementOfDistribution.PTR312 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion PersonNameDetailsNameSuffixT
                    
                            #region PersonNameDetailsFamilyNameT
                            _currentXPath.Push("/tns:PersonNameDetailsFamilyNameT");
                            //6. use case
                            if (ReadToNextElement(reader,true) && reader.LocalName == "PersonNameDetailsFamilyNameT")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        statementOfDistribution.PTR146 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion PersonNameDetailsFamilyNameT
                    
                            #region PersonNameDetailsGivenNameT
                            _currentXPath.Push("/tns:PersonNameDetailsGivenNameT");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "PersonNameDetailsGivenNameT")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        statementOfDistribution.PTR147 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion PersonNameDetailsGivenNameT
                    
                            #region PersonNameDetailsOtherGivenNameT
                            _currentXPath.Push("/tns:PersonNameDetailsOtherGivenNameT");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "PersonNameDetailsOtherGivenNameT")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        statementOfDistribution.PTR148 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion PersonNameDetailsOtherGivenNameT
                    
                            #region PersonDemographicDetailsBirthD
                            _currentXPath.Push("/tns:PersonDemographicDetailsBirthD");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "PersonDemographicDetailsBirthD")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                                        {
                                            statementOfDistribution.PTR284 = currentDateTimeValue.Date;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion PersonDemographicDetailsBirthD
                    
                            #region RemunerationPartnerWithNoControlAndShareDisposalInPartnershipIncomeI
                            _currentXPath.Push("/tns:RemunerationPartnerWithNoControlAndShareDisposalInPartnershipIncomeI");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "RemunerationPartnerWithNoControlAndShareDisposalInPartnershipIncomeI")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        currentBooleanValue = ToBoolean(currentValue);
                                        if (currentBooleanValue != null)
                                        {
                                            statementOfDistribution.PTR246 = currentBooleanValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion RemunerationPartnerWithNoControlAndShareDisposalInPartnershipIncomeI
                        } // End of if IndividualPartnerDetails node exists
                
                        _currentXPath.Pop();
                        #endregion IndividualPartnerDetails
                
                        #region PartnerAddressDetails
                        _currentXPath.Push("/tns:PartnerAddressDetails");
                        //3. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "PartnerAddressDetails")
                        {
                            statementOfDistribution.RP_StatementOfDistribution_PartnerAddressDetailsCollectionExists = true;
                            statementOfDistribution.RP_StatementOfDistribution_PartnerAddressDetailsCollectionCount += 1;
                            _found = true; 
                    
                            #region OverseasAddressI
                            _currentXPath.Push("/tns:OverseasAddressI");
                            //6. use case
                            if (ReadToNextElement(reader,true) && reader.LocalName == "OverseasAddressI")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        currentBooleanValue = ToBoolean(currentValue);
                                        if (currentBooleanValue != null)
                                        {
                                            statementOfDistribution.PTR240 = currentBooleanValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion OverseasAddressI
                    
                            #region Line1T
                            _currentXPath.Push("/tns:Line1T");
                            //6. use case
                            if (ReadToNextElement(reader,true) && reader.LocalName == "Line1T")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        statementOfDistribution.PTR151 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion Line1T
                    
                            #region Line2T
                            _currentXPath.Push("/tns:Line2T");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "Line2T")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        statementOfDistribution.PTR152 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion Line2T
                    
                            #region LocalityNameT
                            _currentXPath.Push("/tns:LocalityNameT");
                            //6. use case
                            if (ReadToNextElement(reader,true) && reader.LocalName == "LocalityNameT")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        statementOfDistribution.PTR153 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion LocalityNameT
                    
                            #region PostcodeT
                            _currentXPath.Push("/tns:PostcodeT");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "PostcodeT")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        statementOfDistribution.PTR155 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion PostcodeT
                    
                            #region StateOrTerritoryC
                            _currentXPath.Push("/tns:StateOrTerritoryC");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "StateOrTerritoryC")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        statementOfDistribution.PTR154 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion StateOrTerritoryC
                    
                            #region CountryC
                            _currentXPath.Push("/tns:CountryC");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "CountryC")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        statementOfDistribution.PTR308 = currentValue;
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion CountryC
                        } // End of if PartnerAddressDetails node exists
                
                        _currentXPath.Pop();
                        #endregion PartnerAddressDetails
                
                        #region IncomeOrLoss
                        _currentXPath.Push("/tns:IncomeOrLoss");
                        //3. use case
                        if (ReadToNextElement(reader,false) && reader.LocalName == "IncomeOrLoss")
                        {
                            statementOfDistribution.RP_StatementOfDistribution_IncomeOrLossCollectionExists = true;
                            statementOfDistribution.RP_StatementOfDistribution_IncomeOrLossCollectionCount += 1;
                            _found = true; 
                    
                            #region IncomeTaxPayAsYouGoWithholdingCreditForTaxWithheldWhereABNNotQuotedA
                            _currentXPath.Push("/tns:IncomeTaxPayAsYouGoWithholdingCreditForTaxWithheldWhereABNNotQuotedA");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "IncomeTaxPayAsYouGoWithholdingCreditForTaxWithheldWhereABNNotQuotedA")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            statementOfDistribution.PTR160 = currentDecimalValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion IncomeTaxPayAsYouGoWithholdingCreditForTaxWithheldWhereABNNotQuotedA
                    
                            #region DistributionFrankedA
                            _currentXPath.Push("/tns:DistributionFrankedA");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "DistributionFrankedA")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            statementOfDistribution.PTR324 = currentDecimalValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion DistributionFrankedA
                    
                            #region IncomeTaxFrankingCreditsReceivedFromAustralianCompaniesA
                            _currentXPath.Push("/tns:IncomeTaxFrankingCreditsReceivedFromAustralianCompaniesA");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "IncomeTaxFrankingCreditsReceivedFromAustralianCompaniesA")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            statementOfDistribution.PTR161 = currentDecimalValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion IncomeTaxFrankingCreditsReceivedFromAustralianCompaniesA
                    
                            #region IncomeTaxPayAsYouGoWithholdingCreditForAmountsWithheldWhereTFNNotQuotedA
                            _currentXPath.Push("/tns:IncomeTaxPayAsYouGoWithholdingCreditForAmountsWithheldWhereTFNNotQuotedA");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "IncomeTaxPayAsYouGoWithholdingCreditForAmountsWithheldWhereTFNNotQuotedA")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            statementOfDistribution.PTR162 = currentDecimalValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion IncomeTaxPayAsYouGoWithholdingCreditForAmountsWithheldWhereTFNNotQuotedA
                    
                            #region IncomeTaxPayAsYouGoWithholdingCreditForTaxWithheldFromCloselyHeldTrustA
                            _currentXPath.Push("/tns:IncomeTaxPayAsYouGoWithholdingCreditForTaxWithheldFromCloselyHeldTrustA");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "IncomeTaxPayAsYouGoWithholdingCreditForTaxWithheldFromCloselyHeldTrustA")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            statementOfDistribution.PTR321 = currentDecimalValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion IncomeTaxPayAsYouGoWithholdingCreditForTaxWithheldFromCloselyHeldTrustA
                    
                            #region IncomeTaxPayAsYouGoWithholdingCreditForAmountsWithheldFromForeignResidentsA
                            _currentXPath.Push("/tns:IncomeTaxPayAsYouGoWithholdingCreditForAmountsWithheldFromForeignResidentsA");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "IncomeTaxPayAsYouGoWithholdingCreditForAmountsWithheldFromForeignResidentsA")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            statementOfDistribution.PTR351 = currentDecimalValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion IncomeTaxPayAsYouGoWithholdingCreditForAmountsWithheldFromForeignResidentsA
                    
                            #region IncomeTaxFrankingCreditsReceivedFromNewZealandCompaniesA
                            _currentXPath.Push("/tns:IncomeTaxFrankingCreditsReceivedFromNewZealandCompaniesA");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "IncomeTaxFrankingCreditsReceivedFromNewZealandCompaniesA")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            statementOfDistribution.PTR164 = currentDecimalValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion IncomeTaxFrankingCreditsReceivedFromNewZealandCompaniesA
                    
                            #region SmallBusinessEntityNetA
                            _currentXPath.Push("/tns:SmallBusinessEntityNetA");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "SmallBusinessEntityNetA")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            statementOfDistribution.PTR353 = currentDecimalValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion SmallBusinessEntityNetA
                    
                            #region TaxOffsetClaimNationalRentalAffordabilitySchemeEntitlementA
                            _currentXPath.Push("/tns:TaxOffsetClaimNationalRentalAffordabilitySchemeEntitlementA");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "TaxOffsetClaimNationalRentalAffordabilitySchemeEntitlementA")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            statementOfDistribution.PTR249 = currentDecimalValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion TaxOffsetClaimNationalRentalAffordabilitySchemeEntitlementA
                    
                            #region InvestmentFinancialIncomeOrLossNetA
                            _currentXPath.Push("/tns:InvestmentFinancialIncomeOrLossNetA");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "InvestmentFinancialIncomeOrLossNetA")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            statementOfDistribution.PTR256 = currentDecimalValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion InvestmentFinancialIncomeOrLossNetA
                    
                            #region RentalIncomeOrLossNetA
                            _currentXPath.Push("/tns:RentalIncomeOrLossNetA");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "RentalIncomeOrLossNetA")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            statementOfDistribution.PTR257 = currentDecimalValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion RentalIncomeOrLossNetA
                    
                            #region IncomeTaxExplorationCreditsDistributedA
                            _currentXPath.Push("/tns:IncomeTaxExplorationCreditsDistributedA");
                            //6. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "IncomeTaxExplorationCreditsDistributedA")
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            statementOfDistribution.PTR342 = currentDecimalValue;
                                        }
                                    }
                                }
                                _found = true;
                            }
                    
                            _currentXPath.Pop();
                            #endregion IncomeTaxExplorationCreditsDistributedA
                    
                            #region ShareOfIncomePrimaryProduction
                            _currentXPath.Push("/tns:ShareOfIncomePrimaryProduction");
                            //3. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "ShareOfIncomePrimaryProduction")
                            {
                                statementOfDistribution.RP_StatementOfDistribution_IncomeOrLoss_ShareOfIncomePrimaryProductionCollectionExists = true;
                                statementOfDistribution.RP_StatementOfDistribution_IncomeOrLoss_ShareOfIncomePrimaryProductionCollectionCount += 1;
                                _found = true; 
                        
                                #region BeneficiaryShareA
                                _currentXPath.Push("/tns:BeneficiaryShareA");
                                //6. use case
                                if (ReadToNextElement(reader,false) && reader.LocalName == "BeneficiaryShareA")
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                statementOfDistribution.PTR158 = currentDecimalValue;
                                            }
                                        }
                                    }
                                    _found = true;
                                }
                        
                                _currentXPath.Pop();
                                #endregion BeneficiaryShareA
                        
                                #region NonConcessionalManagedInvestmentTrustA
                                _currentXPath.Push("/tns:NonConcessionalManagedInvestmentTrustA");
                                //6. use case
                                if (ReadToNextElement(reader,false) && reader.LocalName == "NonConcessionalManagedInvestmentTrustA")
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                statementOfDistribution.PTR368 = currentDecimalValue;
                                            }
                                        }
                                    }
                                    _found = true;
                                }
                        
                                _currentXPath.Pop();
                                #endregion NonConcessionalManagedInvestmentTrustA
                        
                                #region ExcludedFromNonConcessionalManagedInvestmentTrustA
                                _currentXPath.Push("/tns:ExcludedFromNonConcessionalManagedInvestmentTrustA");
                                //6. use case
                                if (ReadToNextElement(reader,false) && reader.LocalName == "ExcludedFromNonConcessionalManagedInvestmentTrustA")
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                statementOfDistribution.PTR369 = currentDecimalValue;
                                            }
                                        }
                                    }
                                    _found = true;
                                }
                        
                                _currentXPath.Pop();
                                #endregion ExcludedFromNonConcessionalManagedInvestmentTrustA
                            } // End of if ShareOfIncomePrimaryProduction node exists
                    
                            _currentXPath.Pop();
                            #endregion ShareOfIncomePrimaryProduction
                    
                            #region ShareOfIncomeNonPrimaryProduction
                            _currentXPath.Push("/tns:ShareOfIncomeNonPrimaryProduction");
                            //3. use case
                            if (ReadToNextElement(reader,false) && reader.LocalName == "ShareOfIncomeNonPrimaryProduction")
                            {
                                statementOfDistribution.RP_StatementOfDistribution_IncomeOrLoss_ShareOfIncomeNonPrimaryProductionCollectionExists = true;
                                statementOfDistribution.RP_StatementOfDistribution_IncomeOrLoss_ShareOfIncomeNonPrimaryProductionCollectionCount += 1;
                                _found = true; 
                        
                                #region BeneficiaryShareA
                                _currentXPath.Push("/tns:BeneficiaryShareA");
                                //6. use case
                                if (ReadToNextElement(reader,false) && reader.LocalName == "BeneficiaryShareA")
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                statementOfDistribution.PTR159 = currentDecimalValue;
                                            }
                                        }
                                    }
                                    _found = true;
                                }
                        
                                _currentXPath.Pop();
                                #endregion BeneficiaryShareA
                        
                                #region NonConcessionalManagedInvestmentTrustA
                                _currentXPath.Push("/tns:NonConcessionalManagedInvestmentTrustA");
                                //6. use case
                                if (ReadToNextElement(reader,false) && reader.LocalName == "NonConcessionalManagedInvestmentTrustA")
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                statementOfDistribution.PTR370 = currentDecimalValue;
                                            }
                                        }
                                    }
                                    _found = true;
                                }
                        
                                _currentXPath.Pop();
                                #endregion NonConcessionalManagedInvestmentTrustA
                        
                                #region ExcludedFromNonConcessionalManagedInvestmentTrustA
                                _currentXPath.Push("/tns:ExcludedFromNonConcessionalManagedInvestmentTrustA");
                                //6. use case
                                if (ReadToNextElement(reader,false) && reader.LocalName == "ExcludedFromNonConcessionalManagedInvestmentTrustA")
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                statementOfDistribution.PTR371 = currentDecimalValue;
                                            }
                                        }
                                    }
                                    _found = true;
                                }
                        
                                _currentXPath.Pop();
                                #endregion ExcludedFromNonConcessionalManagedInvestmentTrustA
                            } // End of if ShareOfIncomeNonPrimaryProduction node exists
                    
                            _currentXPath.Pop();
                            #endregion ShareOfIncomeNonPrimaryProduction
                        } // End of if IncomeOrLoss node exists
                
                        _currentXPath.Pop();
                        #endregion IncomeOrLoss
                    _currentXPath.Pop();
                    } // End of while loop for StatementOfDistribution
                } // End of if StatementOfDistributionCollection node exists
        
                _currentXPath.Pop();
                #endregion StatementOfDistribution
        
                #region OtherReportingInformation
                _currentXPath.Push("/tns:OtherReportingInformation");
                //3. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "OtherReportingInformation")
                {
                    report.RP_OtherReportingInformationCollectionExists = true;
                    report.RP_OtherReportingInformationCollectionCount += 1;
                    _found = true; 
            
                    #region RegulatoryDisclosuresGeneralInformationAboutFinancialStatementsT
                    _currentXPath.Push("/tns:RegulatoryDisclosuresGeneralInformationAboutFinancialStatementsT");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "RegulatoryDisclosuresGeneralInformationAboutFinancialStatementsT")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.PTR315 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion RegulatoryDisclosuresGeneralInformationAboutFinancialStatementsT
            
                    #region CompletionHoursN
                    _currentXPath.Push("/tns:CompletionHoursN");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "CompletionHoursN")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.PTR169 = currentDecimalValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion CompletionHoursN
                } // End of if OtherReportingInformation node exists
        
                _currentXPath.Pop();
                #endregion OtherReportingInformation
        
                #region ReportingPartyDeclaration
                _currentXPath.Push("/tns:ReportingPartyDeclaration");
                //3. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "ReportingPartyDeclaration")
                {
                    report.RP_ReportingPartyDeclarationCollectionExists = true;
                    report.RP_ReportingPartyDeclarationCollectionCount += 1;
                    _found = true; 
            
                    #region StatementTypeC
                    _currentXPath.Push("/tns:StatementTypeC");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "StatementTypeC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.PTR291 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion StatementTypeC
            
                    #region StatementAcceptedI
                    _currentXPath.Push("/tns:StatementAcceptedI");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "StatementAcceptedI")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.PTR290 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion StatementAcceptedI
            
                    #region StatementT
                    _currentXPath.Push("/tns:StatementT");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "StatementT")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.PTR293 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion StatementT
            
                    #region SignatureD
                    _currentXPath.Push("/tns:SignatureD");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "SignatureD")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                                {
                                    report.PTR289 = currentDateTimeValue.Date;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion SignatureD
            
                    #region SignatoryIdentifierT
                    _currentXPath.Push("/tns:SignatoryIdentifierT");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "SignatoryIdentifierT")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.PTR292 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion SignatoryIdentifierT
            
                    #region PersonUnstructuredNameFullNameT
                    _currentXPath.Push("/tns:PersonUnstructuredNameFullNameT");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "PersonUnstructuredNameFullNameT")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.PTR264 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion PersonUnstructuredNameFullNameT
                } // End of if ReportingPartyDeclaration node exists
        
                _currentXPath.Pop();
                #endregion ReportingPartyDeclaration
            } // End of if RP node exists
    
            _currentXPath.Pop();
            #endregion RP
    
            #region INT
            _currentXPath.Push("/tns:INT");
            //3. use case
            if (ReadToNextElement(reader,false) && reader.LocalName == "INT")
            {
                report.IntCollectionExists = true;
                report.IntCollectionCount += 1;
                _found = true; 
        
                #region AustralianBusinessNumberId
                _currentXPath.Push("/tns:AustralianBusinessNumberId");
                //6. use case
                if (ReadToNextElement(reader,true) && reader.LocalName == "AustralianBusinessNumberId")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.PTR502 = currentValue;
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion AustralianBusinessNumberId
        
                #region TaxAgentNumberId
                _currentXPath.Push("/tns:TaxAgentNumberId");
                //6. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "TaxAgentNumberId")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.PTR2 = currentValue;
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion TaxAgentNumberId
        
                #region TaxAgentClientReferenceT
                _currentXPath.Push("/tns:TaxAgentClientReferenceT");
                //6. use case
                if (ReadToNextElement(reader,false) && reader.LocalName == "TaxAgentClientReferenceT")
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.PTR170 = currentValue;
                        }
                    }
                    _found = true;
                }
        
                _currentXPath.Pop();
                #endregion TaxAgentClientReferenceT
        
                #region IntermediaryDeclaration
                _currentXPath.Push("/tns:IntermediaryDeclaration");
                //3. use case
                if (ReadToNextElement(reader,true) && reader.LocalName == "IntermediaryDeclaration")
                {
                    report.InT_IntermediaryDeclarationCollectionExists = true;
                    report.InT_IntermediaryDeclarationCollectionCount += 1;
                    _found = true; 
            
                    #region StatementTypeC
                    _currentXPath.Push("/tns:StatementTypeC");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "StatementTypeC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.PTR296 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion StatementTypeC
            
                    #region StatementAcceptedI
                    _currentXPath.Push("/tns:StatementAcceptedI");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "StatementAcceptedI")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.PTR295 = currentBooleanValue;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion StatementAcceptedI
            
                    #region StatementT
                    _currentXPath.Push("/tns:StatementT");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "StatementT")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.PTR298 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion StatementT
            
                    #region SignatureD
                    _currentXPath.Push("/tns:SignatureD");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "SignatureD")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                                {
                                    report.PTR294 = currentDateTimeValue.Date;
                                }
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion SignatureD
            
                    #region SignatoryIdentifierT
                    _currentXPath.Push("/tns:SignatoryIdentifierT");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "SignatoryIdentifierT")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.PTR297 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion SignatoryIdentifierT
            
                    #region PersonUnstructuredNameFullNameT
                    _currentXPath.Push("/tns:PersonUnstructuredNameFullNameT");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "PersonUnstructuredNameFullNameT")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.PTR314 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion PersonUnstructuredNameFullNameT
                } // End of if IntermediaryDeclaration node exists
        
                _currentXPath.Pop();
                #endregion IntermediaryDeclaration
        
                #region IntermediaryContactDetails
                _currentXPath.Push("/tns:IntermediaryContactDetails");
                //3. use case
                if (ReadToNextElement(reader,true) && reader.LocalName == "IntermediaryContactDetails")
                {
                    report.InT_IntermediaryContactDetailsCollectionExists = true;
                    report.InT_IntermediaryContactDetailsCollectionCount += 1;
                    _found = true; 
            
                    #region PersonUnstructuredNameFullNameT
                    _currentXPath.Push("/tns:PersonUnstructuredNameFullNameT");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "PersonUnstructuredNameFullNameT")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.PTR242 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion PersonUnstructuredNameFullNameT
            
                    #region ElectronicContactTelephoneServiceLineC
                    _currentXPath.Push("/tns:ElectronicContactTelephoneServiceLineC");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "ElectronicContactTelephoneServiceLineC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.PTR244 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion ElectronicContactTelephoneServiceLineC
            
                    #region ElectronicContactTelephoneAreaC
                    _currentXPath.Push("/tns:ElectronicContactTelephoneAreaC");
                    //6. use case
                    if (ReadToNextElement(reader,false) && reader.LocalName == "ElectronicContactTelephoneAreaC")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.PTR172 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion ElectronicContactTelephoneAreaC
            
                    #region ElectronicContactTelephoneMinimalN
                    _currentXPath.Push("/tns:ElectronicContactTelephoneMinimalN");
                    //6. use case
                    if (ReadToNextElement(reader,true) && reader.LocalName == "ElectronicContactTelephoneMinimalN")
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.PTR173 = currentValue;
                            }
                        }
                        _found = true;
                    }
            
                    _currentXPath.Pop();
                    #endregion ElectronicContactTelephoneMinimalN
                } // End of if IntermediaryContactDetails node exists
        
                _currentXPath.Pop();
                #endregion IntermediaryContactDetails
            } // End of if INT node exists
    
            _currentXPath.Pop();
            #endregion INT

            while (reader.EOF != true)
                reader.Read();

            return report;
        }

        private string _elementCsvList = "RP#TargetFinancialY,RP#PeriodStartD,RP#PeriodEndD,RP#TaxFileNumberId,RP#AustralianBusinessNumberId,ReportAmendment#I,ReportAmendment#AmendmentTypeC,ReportAmendment#AmendmentReasonT,ReportAmendment#AmendmentSequenceN,ReportAmendment,CurrentPartnershipNameOrganisationNameDetailsOrganisationalNameT,PreviousPartnershipNameOrganisationNameDetailsOrganisationalNameT,CurrentPostalAddressDetails#OverseasAddressI,CurrentPostalAddressDetails#Line1T,CurrentPostalAddressDetails#Line2T,CurrentPostalAddressDetails#LocalityNameT,CurrentPostalAddressDetails#PostcodeT,CurrentPostalAddressDetails#StateOrTerritoryC,CurrentPostalAddressDetails#CountryC,CurrentPostalAddressDetails,PreviousPostalAddressDetails#OverseasAddressI,PreviousPostalAddressDetails#Line1T,PreviousPostalAddressDetails#Line2T,PreviousPostalAddressDetails#LocalityNameT,PreviousPostalAddressDetails#PostcodeT,PreviousPostalAddressDetails#StateOrTerritoryC,PreviousPostalAddressDetails#CountryC,PreviousPostalAddressDetails,PersonNameDetails#TitleT,PersonNameDetails#NameSuffixT,PersonNameDetails#FamilyNameT,PersonNameDetails#GivenNameT,PersonNameDetails#OtherGivenNameT,PersonNameDetails,OrganisationDetails#OrganisationNameDetailsOrganisationalNameT,OrganisationDetails#AustralianBusinessNumberId,OrganisationDetails,MailRecipient,RP#ElectionsInterposedEntityElectionStatusY,RP#ElectionsInterposedEntityElectionRevocationC,RP#LodgmentFinalReturnI,RP#OrganisationDetailsMainIncomeActivityDe,RP#OrganisationDetailsOrganisationIndustry2006ExtendedC,RP#OrganisationDetailsActivityEventC,RP#OrganisationDetailsTaxConsolidationStatusI,RP#OrganisationDetailsSignificantGlobalEntityStatusI,RP#InternationalDealingsCountryByCountryReportingEntityStatusI,BusinessIncomeAndExpenses#SmallBusinessEntityNetA,BusinessIncomeAndExpenses#ReconciliationAdjustmentTotalA,IncomePrimaryProduction#RemunerationABNNotQuotedPaymentGrossA,IncomePrimaryProduction#GovernmentFundingGovernmentIndustryPaymentsAssessableA,IncomePrimaryProduction#GovernmentFundingGovernmentIndustryPaymentsAssessableI,IncomePrimaryProduction#OtherA,IncomePrimaryProduction#NetA,IncomePrimaryProduction,IncomeNonPrimaryProduction#RemunerationABNNotQuotedPaymentGrossA,IncomeNonPrimaryProduction#RemunerationPaymentToForeignResidentGrossA,IncomeNonPrimaryProduction#GovernmentFundingGovernmentIndustryPaymentsAssessableA,IncomeNonPrimaryProduction#GovernmentFundingGovernmentIndustryPaymentsAssessableI,IncomeNonPrimaryProduction#OtherA,IncomeNonPrimaryProduction#NetA,IncomeNonPrimaryProduction,BusinessExpenses#ForeignResidentWithholdingA,BusinessExpenses#RemunerationWagesAndSalariesPaymentsContractorAndConsultantA,BusinessExpenses#SuperannuationContributionEmployerContributionsA,BusinessExpenses#OperatingCostOfSalesA,BusinessExpenses#OperatingBadDebtsA,BusinessExpenses#OperatingLeaseTotalA,BusinessExpenses#OperatingRentA,BusinessExpenses#OperatingInterestA,BusinessExpenses#RoyaltiesA,BusinessExpenses#DepreciationAndAmortisationA,BusinessExpenses#OperatingMotorVehicleA,BusinessExpenses#OperatingMotorVehicleTypeOrClaimMethodC,BusinessExpenses#OperatingRepairsAndMaintenanceA,BusinessExpenses#DeductibleOtherTotalA,BusinessExpenses#ReconciliationAdjustmentTotalA,BusinessExpenses,PayAsYouGoWithholding#TaxCreditForTaxWithheldWhereABNNotQuotedA,PayAsYouGoWithholding#TaxCreditForAmountsWithheldFromForeignResidentsA,PayAsYouGoWithholding,BusinessIncomeAndExpenses,PartnershipsTrustsPrimaryProduction#PartnershipDistributionGrossA,PartnershipsTrustsPrimaryProduction#TrustShareNetA,PartnershipsTrustsPrimaryProduction#PartyTrustTypeC,PartnershipsTrustsPrimaryProduction#ExpenseDistributionDeductionsOtherA,PartnershipsTrustsPrimaryProduction,PartnershipsTrustsNonPrimaryProduction#PartnershipDistributionLessForeignIncomeA,PartnershipsTrustsNonPrimaryProduction#TrustShareNetExcludeNetCapitalGainsAndForeignIncomeAndDistributionFrankedA,PartnershipsTrustsNonPrimaryProduction#PartyTrustTypeC,PartnershipsTrustsNonPrimaryProduction#ExpenseDistributionDeductionsOtherA,PartnershipsTrustsNonPrimaryProduction#TrustDistributionFrankedA,PartnershipsTrustsNonPrimaryProduction#TaxDeductionTrustDistributionFrankedA,PartnershipsTrustsNonPrimaryProduction,ShareOfCreditsFromIncome#TaxPayAsYouGoWithholdingCreditForTaxWithheldWhereABNNotQuotedShareReceivedIndirectlyA,ShareOfCreditsFromIncome#TaxFrankingCreditsFrankingCreditShareReceivedIndirectlyA,ShareOfCreditsFromIncome#TaxPayAsYouGoWithholdingCreditForAmountsWithheldFromInterestAndDividendsWhereTFNNotQuotedShareReceivedIndirectlyA,ShareOfCreditsFromIncome#TaxPayAsYouGoWithholdingCreditForTaxWithheldFromCloselyHeldTrustShareReceivedIndirectlyA,ShareOfCreditsFromIncome#InternationalDealingsCreditForTaxWithheldFromForeignResidentWithholdingShareReceivedIndirectlyA,ShareOfCreditsFromIncome,Rent#OperatingIncomeGrossA,Rent#ExpenseInterestA,Rent#ExpenseCapitalWorksDeductionA,Rent#TaxDeductionIncomeDeductionsOtherThanInterestAndCapitalWorksA,Rent#ExpenseBuildToRentCapitalWorksDeductionA,Rent,Income#ForestryManagedInvestmentSchemeA,GrossInterest#A,GrossInterest#TaxTFNAmountsWithheldFromGrossInterestA,GrossInterest,Dividends#UnfrankedA,Dividends#FrankedA,Dividends#TaxFrankingCreditsA,Dividends#TaxTFNAmountsWithheldA,Dividends,OtherAustralianIncome#De,OtherAustralianIncome#A,OtherAustralianIncome,OtherAustralianIncomeCollection,Income,RP#ExpenseAustralianInvestmentAllowableDeductionA,RP#ExpenseForestryManagedInvestmentSchemeDeductionA,OtherDeductions#ExpenseDeductibleOtherT,OtherDeductions#ExpenseDeductibleOtherA,OtherDeductions,OtherDeductionsCollection,RP#IncomeNetA,AttributedForeignIncome#InternationalDealingsTrustorControlledCompanyorTransferorTrustI,ListedCountryInternationalDealingsA,UnlistedCountryInternationalDealingsA,AttributedForeignIncome,OtherAssessableForeignSourceIncome#A,OtherAssessableForeignSourceIncome#NetA,OtherAssessableForeignSourceIncome#InternationalDealingsTaxOffsetA,OtherAssessableForeignSourceIncome#IncomeTaxFrankingCreditsReceivedFromNewZealandCompaniesA,OtherAssessableForeignSourceIncome,OverseasTransactions#InternationalDealingsRelatedPartiesTransactionsExcessAggregateValueI,OverseasTransactions#LiabilitiesThinCapitalisationProvisionsAppliedI,OverseasTransactions#ExpenseInterestA,OverseasTransactions#ExpenseRoyaltiesA,OverseasTransactions,RP#InternationalDealingsTransactionsWithTaxHavenCountriesI,PersonalServicesIncome#IncludedI,PersonalServicesIncome#TotalA,PersonalServicesIncome#ExpenseDeductionA,PersonalServicesIncome#IncomeTaxIndividualResultsTestSatisfiedI,PersonalServicesIncome#IncomeTaxIndividualPersonalServiceBusinessDeterminationHeldI,PersonalServicesIncome#IncomeTaxUnrelatedClientsTestSatisfiedI,PersonalServicesIncome#IncomeTaxEmploymentTestSatisfiedI,PersonalServicesIncome#IncomeTaxBusinessPremisesTestSatisfiedI,PersonalServicesIncome,TaxationOfFinancialArrangements#IncomeGainsTotalA,TaxationOfFinancialArrangements#ExpenseLossesTotalA,TaxationOfFinancialArrangements,NCMIBusinessIncomePrimaryProduction#NonConcessionalManagedInvestmentTrustA,NCMIBusinessIncomePrimaryProduction#ExcludedFromNonConcessionalManagedInvestmentTrustA,NCMIBusinessIncomePrimaryProduction,NCMIBusinessIncomeNonPrimaryProduction#NonConcessionalManagedInvestmentTrustA,NCMIBusinessIncomeNonPrimaryProduction#ExcludedFromNonConcessionalManagedInvestmentTrustA,NCMIBusinessIncomeNonPrimaryProduction,NCMIBusinessIncome,NCMIPartnershipsAndTrustsPrimaryProduction#IncomeNonConcessionalManagedInvestmentTrustA,NCMIPartnershipsAndTrustsPrimaryProduction#IncomeExcludedFromNonConcessionalManagedInvestmentTrustA,PrimaryProductionShareOfNetIncome#NonConcessionalManagedInvestmentTrustA,PrimaryProductionShareOfNetIncome#ExcludedFromNonConcessionalManagedInvestmentTrustA,PrimaryProductionShareOfNetIncome,NCMIPartnershipsAndTrustsPrimaryProduction,NCMIPartnershipsAndTrustsNonPrimaryProduction#IncomeNonConcessionalManagedInvestmentTrustA,NCMIPartnershipsAndTrustsNonPrimaryProduction#IncomeExcludedFromNonConcessionalManagedInvestmentTrustA,NonPrimaryProductionShareOfNetIncome#NonConcessionalManagedInvestmentTrustA,NonPrimaryProductionShareOfNetIncome#ExcludedFromNonConcessionalManagedInvestmentTrustA,NonPrimaryProductionShareOfNetIncome,NCMIPartnershipsAndTrustsNonPrimaryProduction,NCMIPartnershipsAndTrusts,KeyFinancialInformation#AssetsCurrentTotalA,KeyFinancialInformation#AssetsTotalA,KeyFinancialInformation#LiabilitiesCurrentTotalA,KeyFinancialInformation#LiabilitiesTotalA,KeyFinancialInformation,BusinessNameOrganisationNameDetailsOrganisationalNameT,BusinessAddressDetails#OverseasAddressI,BusinessAddressDetails#Line1T,BusinessAddressDetails#Line2T,BusinessAddressDetails#LocalityNameT,BusinessAddressDetails#PostcodeT,BusinessAddressDetails#StateOrTerritoryC,BusinessAddressDetails#CountryC,BusinessAddressDetails,BusinessAndProfessionaltems#AssetsInventoriesTotalA,BusinessAndProfessionaltems#ExpensePurchasesAndOtherCostsA,BusinessAndProfessionaltems#RemunerationWagesAndSalariesTotalA,BusinessAndProfessionaltems#RemunerationWagesAndSalariesActionC,BusinessAndProfessionaltems#RemunerationPaymentToRelatedPartiesGrossA,BusinessAndProfessionaltems#RemunerationFringeBenefitsEmployeeContributionA,BusinessAndProfessionaltems#ElectionsTradingStockI,Closing#AssetsInventoriesTotalA,Closing#AssetsInventoriesValuationMethodC,Closing#AssetsTradeandReceivablesOtherA,Closing#LiabilitiesTradeAndOtherPayablesTotalA,Closing,BusinessAndProfessionaltems,AggregatedTurnover#IncomeRangeC,AggregatedTurnover#IncomeA,AggregatedTurnover,CapitalAllowancesDepreciatingAssetsFirst#ExpenseIntangibleFirstDeductedA,CapitalAllowancesDepreciatingAssetsFirst#ExpenseOtherFirstDeductedA,CapitalAllowancesDepreciatingAssetsFirst#PropertyPlantAndEquipmentFirstDeductedSelfAssessedEffectiveLifeI,CapitalAllowancesDepreciatingAssetsFirst,CapitalAllowancesDepreciatingAssetsAll#PropertyPlantAndEquipmentEffectiveLifeRecalculationI,CapitalAllowancesDepreciatingAssetsAll#AdjustableValuesTotalA,CapitalAllowancesDepreciatingAssetsAll#IncomeIntangibleBalancingAdjustmentA,CapitalAllowancesDepreciatingAssetsAll#ExpenseIntangibleBalancingAdjustmentA,CapitalAllowancesDepreciatingAssetsAll#IntangibleTerminationValueA,CapitalAllowancesDepreciatingAssetsAll#OtherTerminationValueA,CapitalAllowancesDepreciatingAssetsAll#ExpenseSubsequentYearAcceleratedDepreciationDeductionsForAssetsA,CapitalAllowancesDepreciatingAssetsAll#ExpenseProjectPoolAllowableDeductionA,CapitalAllowancesDepreciatingAssetsAll#ExpenseCapitalExpenditureSpecifiedAllowableDeductionA,CapitalAllowancesDepreciatingAssetsAll,LowCostDeductionExpenseDepreciatingAssetsSmallBusinessEntityA,GeneralPoolDeductionExpenseDepreciatingAssetsSmallBusinessEntityA,RP#TaxOffsetClaimNationalRentalAffordabilitySchemeEntitlementA,IncomeTests#InvestmentFinancialIncomeOrLossNetA,IncomeTests#RentalIncomeOrLossNetA,IncomeTests,StatementOfDistribution#TaxFileNumberId,StatementOfDistribution#OrganisationNameDetailsOrganisationalNameT,StatementOfDistribution#AustralianBusinessNumberId,NonIndividualPartnerDetails,StatementOfDistribution#PersonNameDetailsTitleT,StatementOfDistribution#PersonNameDetailsNameSuffixT,StatementOfDistribution#PersonNameDetailsFamilyNameT,StatementOfDistribution#PersonNameDetailsGivenNameT,StatementOfDistribution#PersonNameDetailsOtherGivenNameT,StatementOfDistribution#PersonDemographicDetailsBirthD,StatementOfDistribution#RemunerationPartnerWithNoControlAndShareDisposalInPartnershipIncomeI,IndividualPartnerDetails,StatementOfDistribution#OverseasAddressI,StatementOfDistribution#Line1T,StatementOfDistribution#Line2T,StatementOfDistribution#LocalityNameT,StatementOfDistribution#PostcodeT,StatementOfDistribution#StateOrTerritoryC,StatementOfDistribution#CountryC,PartnerAddressDetails,StatementOfDistribution#IncomeTaxPayAsYouGoWithholdingCreditForTaxWithheldWhereABNNotQuotedA,StatementOfDistribution#DistributionFrankedA,StatementOfDistribution#IncomeTaxFrankingCreditsReceivedFromAustralianCompaniesA,StatementOfDistribution#IncomeTaxPayAsYouGoWithholdingCreditForAmountsWithheldWhereTFNNotQuotedA,StatementOfDistribution#IncomeTaxPayAsYouGoWithholdingCreditForTaxWithheldFromCloselyHeldTrustA,StatementOfDistribution#IncomeTaxPayAsYouGoWithholdingCreditForAmountsWithheldFromForeignResidentsA,StatementOfDistribution#IncomeTaxFrankingCreditsReceivedFromNewZealandCompaniesA,StatementOfDistribution#SmallBusinessEntityNetA,StatementOfDistribution#TaxOffsetClaimNationalRentalAffordabilitySchemeEntitlementA,StatementOfDistribution#InvestmentFinancialIncomeOrLossNetA,StatementOfDistribution#RentalIncomeOrLossNetA,StatementOfDistribution#IncomeTaxExplorationCreditsDistributedA,StatementOfDistribution#BeneficiaryShareA,StatementOfDistribution#NonConcessionalManagedInvestmentTrustA,StatementOfDistribution#ExcludedFromNonConcessionalManagedInvestmentTrustA,ShareOfIncomePrimaryProduction,StatementOfDistribution#BeneficiaryShareA,StatementOfDistribution#NonConcessionalManagedInvestmentTrustA,StatementOfDistribution#ExcludedFromNonConcessionalManagedInvestmentTrustA,ShareOfIncomeNonPrimaryProduction,IncomeOrLoss,StatementOfDistribution,StatementOfDistributionCollection,OtherReportingInformation#RegulatoryDisclosuresGeneralInformationAboutFinancialStatementsT,OtherReportingInformation#CompletionHoursN,OtherReportingInformation,ReportingPartyDeclaration#StatementTypeC,ReportingPartyDeclaration#StatementAcceptedI,ReportingPartyDeclaration#StatementT,ReportingPartyDeclaration#SignatureD,ReportingPartyDeclaration#SignatoryIdentifierT,ReportingPartyDeclaration#PersonUnstructuredNameFullNameT,ReportingPartyDeclaration,RP,INT#AustralianBusinessNumberId,INT#TaxAgentNumberId,INT#TaxAgentClientReferenceT,IntermediaryDeclaration#StatementTypeC,IntermediaryDeclaration#StatementAcceptedI,IntermediaryDeclaration#StatementT,IntermediaryDeclaration#SignatureD,IntermediaryDeclaration#SignatoryIdentifierT,IntermediaryDeclaration#PersonUnstructuredNameFullNameT,IntermediaryDeclaration,IntermediaryContactDetails#PersonUnstructuredNameFullNameT,IntermediaryContactDetails#ElectronicContactTelephoneServiceLineC,IntermediaryContactDetails#ElectronicContactTelephoneAreaC,IntermediaryContactDetails#ElectronicContactTelephoneMinimalN,IntermediaryContactDetails,INT";

    }
}
