using DataContracts;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Runtime.CompilerServices;
using System.Text.RegularExpressions;

namespace Ato.EN.IntegrationServices.CodeGenerationCTR
{
    public partial class CTR2026Validator
    {
        /// <summary>
        /// The max parameter name length is restricted by the SBR1 and SBR2 schemas - the lowest common denominator of SBR 1 is used as the default
        /// </summary>
        private int _maxParameterNameLength;
        /// <summary>
        /// The max parameter value length is restricted by the SBR1 and SBR2 schemas
        /// </summary>
        private int _maxParameterValueLength;
        /// <summary>
        /// The SBR1 and ebms3 schemas do not allow parameter names or values to be the empty string
        /// </summary>
        private string _emptyParameterValue;
        /// <summary>
        /// The report parameter
        /// </summary>
        private CTR2026 report;
        /// <summary>
        /// The response parameter
        /// </summary>
        public List<ProcessMessageDocument> response { get; private set; }

        /// <summary>
        /// Initializes a new instance of the <see cref = "CTR2026Validator"/> class.
        /// </summary>
        /// <param name = "maxParameterNameLength">Maximum length of the parameter name.</param>
        /// <param name = "maxParameterValueLength">Maximum length of the parameter value.</param>
        /// <param name = "emptyParameterValue">This value will be used in place of any parameter values that result in a null or empty value.</param>
        public CTR2026Validator(int maxParameterNameLength = 20, int maxParameterValueLength = 4096, string emptyParameterValue = "EMPTY")
        {
            _maxParameterNameLength = maxParameterNameLength;
            _maxParameterValueLength = maxParameterValueLength;
            _emptyParameterValue = emptyParameterValue;
            this.response = new List<ProcessMessageDocument>();
        }

        /// <summary>
        /// Initializes a new instance of the <see cref = "CTR2026Validator"/> class.
        /// </summary>
        /// <param name = "report">Report variable </param>                
        /// <param name = "maxParameterNameLength">Maximum length of the parameter name.</param>
        /// <param name = "maxParameterValueLength">Maximum length of the parameter value.</param>
        /// <param name = "emptyParameterValue">This value will be used in place of any parameter values that result in a null or empty value.</param>
        public CTR2026Validator(CTR2026 reportIn, int maxParameterNameLength = 20, int maxParameterValueLength = 4096, string emptyParameterValue = "EMPTY") : this(maxParameterNameLength, maxParameterValueLength, emptyParameterValue)
        {
            this.ConsumedReport = reportIn;
        }

#region Functions
        private static IEnumerable<string> Union(IEnumerable<string> list1, IEnumerable<string> list2)
        {
            IEnumerable<string> response;
            if (list1 == null && list2 == null)
            {
                response = null;
            }
            else if (list1 == null)
            {
                response = list2.Distinct();
            }
            else if (list2 == null)
            {
                response = list1.Distinct();
            }
            else
            {
                response = list1.Union(list2);
            }

            return response;
        }

        private static bool IsMatch(int? field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(Convert.ToString(field.Value), expression, options);
        }

        private static bool IsMatch(string field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(field, expression, options);
        }

        // This is just context and tuple counts where they are integer values - easier that changing the parsing logic to just return the value
        private static int Count(int count)
        {
            return count;
        }

        private static int Count<T>(IEnumerable<T> values)
        {
            return values == null ? 0 : values.Where(f => f != null).Count();
        }

        private static int Count<T>(ICollection<T> values)
        {
            return values == null ? 0 : values.Where(f => f != null).Count();
        }

        private static bool exists(bool value)
        {
            return value;
        }

        public static string GetValueOrEmpty(bool? val)
        {
            return (val.HasValue) ? val.ToString().ToLower() : string.Empty;
        }

        public static string GetValueOrEmptyDate(DateTime? val)
        {
            return (val.HasValue) ? val.Value.ToString("yyyy-MM-dd") : string.Empty;
        }

        private static string GetValueOrEmptyDate(string val)
        {
            DateTime dateTimeOut;
            if (DateTime.TryParse(val, out dateTimeOut))
            {
                return dateTimeOut.ToString("yyyy-MM-dd");
            }

            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmptyDateTime(string val)
        {
            DateTime dateTimeOut;
            if (DateTime.TryParse(val, out dateTimeOut))
            {
                return dateTimeOut.ToString("yyyy-MM-ddTHH:mm:ssZ");
            }

            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmptyDateTime(DateTime? val)
        {
            return (val.HasValue) ? val.Value.ToString("yyyy-MM-ddTHH:mm:ssZ") : string.Empty;
        }

        private static string GetValueOrEmpty(string val)
        {
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        public static string GetValueOrEmpty(decimal? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        public static string GetValueOrEmpty(int? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        /// <summary>
        /// Get string value between [first] a and [last] b.
        /// </summary>
        public static string Between(string value, string a, string b)
        {
            int posA = value.IndexOf(a);
            int posB = value.LastIndexOf(b);
            if (posA == -1)
            {
                return "";
            }

            if (posB == -1)
            {
                return "";
            }

            int adjustedPosA = posA + a.Length;
            if (adjustedPosA >= posB)
            {
                return "";
            }

            return value.Substring(adjustedPosA, posB - adjustedPosA);
        }

        /// <summary>
        /// Get string value after [first] a.
        /// </summary>
        public static string Before(string value, string a)
        {
            int posA = value.IndexOf(a);
            if (posA == -1)
            {
                return "";
            }

            return value.Substring(0, posA);
        }

        /// <summary>
        /// Get string value after [last] a.
        /// </summary>
        public static string After(string value, string a)
        {
            int posA = value.LastIndexOf(a);
            if (posA == -1)
            {
                return "";
            }

            int adjustedPosA = posA + a.Length;
            if (adjustedPosA >= value.Length)
            {
                return "";
            }

            return value.Substring(adjustedPosA);
        }

        private static int Length(object field)
        {
            if (field == null)
                return 0;
            else
                return field.ToString().Trim().Length;
        }

        private static bool NotSameValues(IEnumerable<object> nodes)
        {
            if (nodes == null)
                return false;
            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool NotSameValues(params object[] nodes)
        {
            if (nodes == null)
                return false;
            return (
                from x in nodes
                select x).Distinct().Count() != 1;
        }

        private static bool HasDuplicateValues(IEnumerable<DateTime> nodes)
        {
            if (nodes == null)
                return false;
            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool HasDuplicateValues(IEnumerable<DateTime?> nodes)
        {
            if (nodes == null)
                return false;
            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool NotSameValues(IEnumerable<DateTime> nodes)
        {
            if (nodes == null)
                return false;
            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool NotSameValues(IEnumerable<DateTime?> nodes)
        {
            if (nodes == null)
                return false;
            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool HasDuplicateValues(IEnumerable<object> nodes)
        {
            if (nodes == null)
                return false;
            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool HasDuplicateValues(params object[] nodes)
        {
            if (nodes == null)
                return false;
            nodes = nodes.Where(x => x != null).ToArray();
            return !((
                from x in nodes
                select x).Distinct().Count() == nodes.Count());
        }

        private int DuplicateValueIndex(IEnumerable<object> values)
        {
            int response = 0;
            var hashset = new HashSet<object>();
            foreach (var value in values)
            {
                if (!hashset.Add(value))
                {
                    return response;
                }

                response++;
            }

            return response;
        }

        private int DuplicateValueIndex<T>(IEnumerable<T?> values)
            where T : struct
        {
            int response = 0;
            var hashset = new HashSet<T?>();
            foreach (var value in values)
            {
                if (!hashset.Add(value))
                {
                    return response;
                }

                response++;
            }

            return response;
        }

        private static bool IsDate(object value)
        {
            DateTime dateValue;
            return (value != null && DateTime.TryParse(value.ToString(), out dateValue));
        }

        public DateTime AsDate(string dateAsString)
        {
            DateTime response = DateTime.MinValue;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date;
            }

            return response;
        }

        public DateTime Date(DateTime? datetime)
        {
            return datetime.GetValueOrDefault().Date;
        }

        //The logic in After function expects "---" for day and "--" for month. 
        //Since hyphen was missing the date always returned null
        public DateTime? ConvertToDate(int day, int month, int year)
        {
            return ConvertToDate(day == 0 ? null : "---" + day.ToString(), month == 0 ? null : "--" + month.ToString(), year == 0 ? null : year.ToString());
        }

        public DateTime? ConvertToDate(string day, string month, string year)
        {
            DateTime? response;
            DateTime result;
            if (year == null || month == null || day == null)
            {
                return null;
            }

            string dateAsString = year + "-" + After(month, "--") + "-" + After(day, "---");
            if (DateTime.TryParse(dateAsString, out result))
            {
                response = result;
            }
            else
            {
                response = null;
            }

            return response;
        }

        public DateTime? ConvertToDate(string day, string month, int year)
        {
            DateTime? response;
            DateTime result;
            if (year == 0 || month == null || day == null)
            {
                return null;
            }

            string dateAsString = year.ToString() + "-" + After(month, "--") + "-" + After(day, "---");
            if (DateTime.TryParse(dateAsString, out result))
            {
                response = result;
            }
            else
            {
                response = null;
            }

            return response;
        }

        private static int Day(string dateAsString)
        {
            int response = 0;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Day;
            }

            return response;
        }

        private static int Day(DateTime? date)
        {
            if (date == null)
                return 0;
            else
                return date.Value.Day;
        }

        private static string Month(string dateAsString)
        {
            string response = null;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.ToString("MMMM");
            }
            else
            {
                return "NotAMonth";
            }

            return response;
        }

        private static string Month(DateTime? date)
        {
            if (date == null)
                return "NotAMonth";
            else
                return date.Value.ToString("MMMM");
        }

        private static int MonthAsInt(string dateAsString)
        {
            int response = 0;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Month;
            }

            return response;
        }

        private static int MonthAsInt(DateTime? date)
        {
            if (date == null)
            {
                return 0;
            }

            return date.Value.Month;
        }

        private static int Year(string dateAsString)
        {
            int response = 0;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Year;
            }

            return response;
        }

        private static int Year(DateTime? date)
        {
            if (date == null)
                return 0;
            else
                return date.Value.Year;
        }

        private static int CurrentFinancialYear()
        {
            return DateToFinancialYear(DateTime.Now, 7);
        }

        private static int FinancialYear(string dateAsString)
        {
            return DateToFinancialYear(dateAsString, 7);
        }

        private static int FinancialYear(DateTime? date)
        {
            return DateToFinancialYear(date, 7);
        }

        private static int DateToFinancialYear(string dateAsString, int startingMonth)
        {
            int response = 0;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
            {
                response = DateToFinancialYear(date, startingMonth);
            }

            return response;
        }

        private static int DateToFinancialYear(DateTime? date, int startingMonth)
        {
            int response;
            if (date == null)
            {
                response = 0;
            }
            else
            {
                int year = date.Value.Year;
                int month = date.Value.Month;
                if (startingMonth > month)
                    response = year;
                else
                    response = year + 1;
            }

            return response;
        }

        private static int FBTYear(string dateAsString)
        {
            int response = 0;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
                response = FBTYear(date);
            return response;
        }

        private static int FBTYear(DateTime? date)
        {
            if (date == null)
            {
                return 0;
            }
            else
            {
                if (date.Value.Month > 3)
                    return date.Value.Year + 1;
                else
                    return date.Value.Year;
            }
        }

        private static DateTime? AddMonthsToDate(DateTime? dateTime, int months)
        {
            return dateTime == null ? null : (DateTime? )dateTime.Value.AddMonths(months);
        }

        private static bool IsNumeric(object value)
        {
            decimal numbervalue;
            return (value != null && decimal.TryParse(value.ToString(), out numbervalue));
        }

        private static bool NotMonetary(decimal? field, string sign, int digits, int decimals)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                string decimalExpression;
                int digitsToUse = digits - decimals;
                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";
                if (decimals > 0)
                    decimalExpression = @"(\.\d{1," + decimals + "})?$";
                else
                    decimalExpression = @"$";
                return !(Regex.IsMatch(field.Value.ToString("0.#########################"), signExpression + @"\d{1," + digitsToUse + "}" + decimalExpression));
            }
        }

        private static bool NotNumeric(decimal? field, string sign, int digits, int decimals)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                string decimalExpression;
                int digitsToUse = digits - decimals;
                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";
                if (decimals > 0)
                    decimalExpression = @"(\.\d{1," + decimals + "})?$";
                else
                    decimalExpression = @"$";
                return !(Regex.IsMatch(field.Value.ToString("0.#########################"), signExpression + @"\d{1," + digitsToUse + "}" + decimalExpression));
            }
        }

        private static bool NotNumeric(int? field, string sign, int digits, int decimals = 0)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";
                return !(Regex.IsMatch(field.Value.ToString(), signExpression + @"\d{1," + digits + "}$"));
            }
        }

        private static bool NotNumeric(long? field, string sign, int digits, int decimals = 0)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";
                return !(Regex.IsMatch(field.Value.ToString(), signExpression + @"\d{1," + digits + "}$"));
            }
        }

        private static bool OutsideRange(decimal field, decimal expression, int range)
        {
            bool response;
            response = (field < (expression - range)) || (field > (expression + range));
            return response;
        }

        private static bool FailsUSIAlgorithm(string usi, string abn)
        {
            bool response = true;
            if (usi == null || abn == null)
            {
                response = false;
            }
            else
            {
                usi = usi.Trim();
                abn = abn.Trim();
                if (usi.Length == 14)
                {
                    int numeric;
                    if (usi.Substring(0, 11) == abn && int.TryParse(usi.Substring(11, 3), out numeric))
                        response = false;
                }
                else if (usi.Length == 9)
                {
                    if (Regex.IsMatch(usi, @"^([a-zA-Z]{3}\d{4}[a-zA-Z]{2})$"))
                        response = false;
                }
            }

            return response;
        }

        private static bool FailsTANAlgorithm(string tan)
        {
            bool response;
            decimal decimalTan;
            if (tan == null)
                return false;
            tan = tan.Trim();
            if (!decimal.TryParse(tan, out decimalTan))
                return true;
            if (tan.Length != 8)
                return true;
            decimal tanSum = 7 * int.Parse(tan.Substring(0, 1)) + 9 * int.Parse(tan.Substring(1, 1)) + 8 * int.Parse(tan.Substring(2, 1)) + 4 * int.Parse(tan.Substring(3, 1)) + 6 * int.Parse(tan.Substring(4, 1)) + 3 * int.Parse(tan.Substring(5, 1)) + 5 * int.Parse(tan.Substring(6, 1)) + 1 * int.Parse(tan.Substring(7, 1));
            if ((tanSum % 11) == 0)
                response = false;
            else
                response = true;
            return response;
        }

        private static bool FailsABNAlgorithm(string abn)
        {
            bool response;
            decimal decimalAbn;
            if (abn == null)
                return false;
            abn = abn.Trim();
            if (!decimal.TryParse(abn, out decimalAbn))
                return true;
            if (abn.Length != 11)
                return true;
            decimal abnSum = 10 * (int.Parse(abn.Substring(0, 1)) - 1) + 1 * int.Parse(abn.Substring(1, 1)) + 3 * int.Parse(abn.Substring(2, 1)) + 5 * int.Parse(abn.Substring(3, 1)) + 7 * int.Parse(abn.Substring(4, 1)) + 9 * int.Parse(abn.Substring(5, 1)) + 11 * int.Parse(abn.Substring(6, 1)) + 13 * int.Parse(abn.Substring(7, 1)) + 15 * int.Parse(abn.Substring(8, 1)) + 17 * int.Parse(abn.Substring(9, 1)) + 19 * int.Parse(abn.Substring(10, 1));
            if ((abnSum % 89) == 0)
                response = false;
            else
                response = true;
            return response;
        }

        private static bool FailsACNAlgorithm(string acn)
        {
            bool response;
            decimal decimalAbn;
            if (acn == null)
                return false;
            acn = acn.Trim();
            if (!decimal.TryParse(acn, out decimalAbn))
                return true;
            if (acn.Length != 9)
                return true;
            decimal abnSum = 8 * int.Parse(acn.Substring(0, 1)) + 7 * int.Parse(acn.Substring(1, 1)) + 6 * int.Parse(acn.Substring(2, 1)) + 5 * int.Parse(acn.Substring(3, 1)) + 4 * int.Parse(acn.Substring(4, 1)) + 3 * int.Parse(acn.Substring(5, 1)) + 2 * int.Parse(acn.Substring(6, 1)) + 1 * int.Parse(acn.Substring(7, 1));
            decimal checkDigit = int.Parse(acn.Substring(8, 1));
            decimal acnRemainder = abnSum % 10;
            if (((10 - acnRemainder) % 10) == checkDigit)
                response = false;
            else
                response = true;
            return response;
        }

        private static bool FailsTFNAlgorithm(string tfn)
        {
            bool response;
            decimal decimalTfn;
            if (tfn == null)
                return false;
            tfn = tfn.Trim();
            tfn = Regex.Replace(tfn, "^0+", "");
            if (!decimal.TryParse(tfn, out decimalTfn))
                return true;
            if (tfn.Length < 8)
                return true;
            decimal tfn1To7Sum = 1 * int.Parse(tfn.Substring(0, 1)) + 4 * int.Parse(tfn.Substring(1, 1)) + 3 * int.Parse(tfn.Substring(2, 1)) + 7 * int.Parse(tfn.Substring(3, 1)) + 5 * int.Parse(tfn.Substring(4, 1)) + 8 * int.Parse(tfn.Substring(5, 1)) + 6 * int.Parse(tfn.Substring(6, 1));
            decimal tfn8 = 9 * int.Parse(tfn.Substring(7, 1));
            if (tfn.Length == 8)
            {
                decimal tFNLg8WSum9 = 10 * int.Parse(tfn.Substring(7, 1));
                decimal tFNLg8WSum = tfn1To7Sum + tFNLg8WSum9;
                if ((tFNLg8WSum % 11) == 0)
                    response = false;
                else
                    response = true;
            }
            else if (tfn.Length == 9)
            {
                decimal tfn9 = 10 * int.Parse(tfn.Substring(8, 1));
                decimal tFNLg9WSum = tfn1To7Sum + tfn8 + tfn9;
                if ((tFNLg9WSum % 11) == 0)
                    response = false;
                else
                    response = true;
            }
            else
            {
                response = true;
            }

            return response;
        }

        private static decimal ConditionalValue(bool expression, decimal? trueVal, decimal? falseVal)
        {
            return expression ? trueVal.GetValueOrDefault() : falseVal.GetValueOrDefault();
        }

        private static decimal AsNumeric(string value)
        {
            decimal numberValue;
            decimal.TryParse(value, out numberValue);
            return numberValue;
        }

        private static bool RegexMatch(int? field, string expression, string flags = "")
        {
            return IsMatch(field, expression, GetRegexOption(flags));
        }

        private static bool RegexMatch(string field, string expression, string flags = "")
        {
            return IsMatch(field, expression, GetRegexOption(flags));
        }

        private static RegexOptions GetRegexOption(string flags)
        {
            RegexOptions options = RegexOptions.None;
            char[] characters = flags.ToCharArray();
            foreach (char character in characters)
            {
                switch (character)
                {
                    case 'i':
                        options = options | RegexOptions.IgnoreCase;
                        break;
                    case 'm':
                        options = options | RegexOptions.Multiline;
                        break;
                    case 's':
                        options = options | RegexOptions.Singleline;
                        break;
                    case 'n':
                        options = options | RegexOptions.ExplicitCapture;
                        break;
                    case 'x':
                        options = options | RegexOptions.IgnorePatternWhitespace;
                        break;
                }
            }

            return options;
        }

        private static string Substring(object field, int start, int length)
        {
            return field.ToString().Substring(start, length);
        }

        /// <summary>
        /// Returns an occurrence index as [occurrenceIndex] of occurrenceIndex > 0, otherwise the empty string
        /// </summary>
        /// <param name = "occurrenceIndex">Index of the occurrence.</param>
        /// <returns>Occurrence in XPath [#] format</returns>
        public string OccurrenceIndex(int occurrenceIndex)
        {
            return occurrenceIndex > 0 ? "[" + occurrenceIndex + "]" : "";
        }

#endregion // Functions

        public CTR2026 ConsumedReport
        {
            get
            {
                return report;
            }

            private set
            {
                report = value;
            }
        }

        private static Dictionary<string, ProcessMessageDocument> _processMessageDocuments = new Dictionary<string, ProcessMessageDocument>();
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public List<ProcessMessageDocument> ValidateReport(CTR2026 reportIn)
        {
            //ProcessMessageDocument processMessage;
            //ProcessMessageParameter parameter;
            //bool assertion;
            this.ConsumedReport = reportIn;
            // ------------------------------------------------------------------------------
            // Validations are now done in code.
            // This version supports full validation of a report - for those rules that have successfully parsed and could be generated.
            // Generated Validations include:
            //   - Validation logic in C#.
            //   - Production of errors in EBMS, SBR and SWS format.
            // 
            // The generation at this stage does not support the following - and has to be completed manually (later versions of the generation will do this for you)
            //   - Business Rules that did not parse.  The ESR team will help support you where this happens as an effort is being made to rectify these
            //   - You will get TODO tasks for those that the parser could not cope with
            // ------------------------------------------------------------------------------
#region Manual Rules - Rules marked for manual coding
#region VR.ATO.CTR.430216
            /*  VR.ATO.CTR.430216
            Check taxable/net income/loss for the Company, in Reconciliation to taxable income or loss

        Legacy Rule Format:
            IF  ([CTR120] <> 
              (([CTR57]+[CTR58]+[CTR59]+[CTR60]+[CTR61]+[CTR63]+
                 [CTR64]+[CTR65]+[CTR66]+
                 [CTR67]+[CTR68]+[CTR70]+[CTR71])
                 -
                 ([CTR72]+[CTR73]+[CTR364]+[CTR75]+[CTR76]+[CTR77]+
                  [CTR78]+[CTR79]+[CTR80]+[CTR81]+[CTR365]+[CTR366]+
                  [CTR84]+[CTR85]+[CTR86]+[CTR87]+[CTR88])  
                + 
                 ([CTR91]+[CTR92]+[CTR93]+[CTR94]+[CTR255]+[CTR95]+
                  [CTR96]+[CTR97])
                - 
                  ([CTR98]+[CTR99]+[CTR100]+[CTR101]+ 
                   [CTR102]+[CTR103]+[CTR104]+[CTR111]+[CTR112]+    
                   [CTR114]+[CTR115]+[CTR256]+[CTR116]+[CTR117]+   
                   [CTR118]) 
                ) +/- 30)
               RETURN VALIDATION MESSAGE
            ENDIF
    
        Technical Business Rule Format:
            (^CTR120 <>   ((^CTR57+^CTR58+^CTR59+^CTR60+^CTR61+^CTR63+     ^CTR64+^CTR65+^CTR66+     ^CTR67+^CTR68+^CTR70+^CTR71)     -     (^CTR72+^CTR73+^CTR364+^CTR75+^CTR76+^CTR77+      ^CTR78+^CTR79+^CTR80+^CTR81+^CTR365+^CTR366+      ^CTR84+^CTR85+^CTR86+^CTR87+^CTR88)      +      (^CTR91+^CTR92+^CTR93+^CTR94+^CTR255+^CTR95+      ^CTR96+^CTR97)    -       (^CTR98+^CTR99+^CTR100+^CTR101+        ^CTR102+^CTR103+^CTR104+^CTR111+^CTR112+           ^CTR114+^CTR115+^CTR256+^CTR116+^CTR117+          ^CTR118)    ) +/- 30)

        Data Elements:

        ^CTR120 = RP:Income.Taxable.Amount

        ^CTR57 = RP:Remuneration.PaymentToForeignResidentGross.Amount

        ^CTR58 = RP:Remuneration.ABNNotQuotedPaymentGross.Amount

        ^CTR59 = RP:Income.SaleOfGoodsAndServices.Other.Amount

        ^CTR60 = RP:Income.PartnershipDistributionGross.Amount

        ^CTR61 = RP:TrustDistributionIncome:Income.TrustDistributionGross.Amount

        ^CTR63 = RP:Income.ForestryManagedInvestmentScheme.Amount

        ^CTR64 = RP:Income.Interest.Gross.Amount

        ^CTR65 = RP:Income.RentLeasingHiringGross.Amount

        ^CTR66 = RP:Income.DividendsTotal.Amount

        ^CTR67 = RP:Remuneration.FringeBenefits.EmployeeContribution.Amount

        ^CTR68 = RP:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount

        ^CTR70 = RP:Income.AssetRevaluationsUnrealisedGains.Amount

        ^CTR71 = RP:Income.OtherGross.Amount

        ^CTR72 = RP:Expense.ForeignResidentWithholding.Amount

        ^CTR73 = RP:Expense.Operating.CostOfSales.Amount

        ^CTR364 = RP:Remuneration.WagesAndSalaries.PaymentsContractorAndConsultant.Amount

        ^CTR75 = RP:SuperannuationContribution.EmployerContributions.Amount

        ^CTR76 = RP:Expense.Operating.BadDebts.Amount

        ^CTR77 = RP.JAUS:Expense.Operating.Lease.Amount

        ^CTR78 = RP.JFOREIGN:Expense.Operating.Lease.Amount

        ^CTR79 = RP:Expense.Operating.Rent.Amount

        ^CTR80 = RP.JAUS:Expense.Operating.Interest.Amount

        ^CTR81 = RP.JFOREIGN:Expense.Operating.Interest.Amount

        ^CTR365 = RP.JAUS:Expense.Royalties.Amount

        ^CTR366 = RP.JFOREIGN:Expense.Royalties.Amount

        ^CTR84 = RP:Expense.DepreciationAndAmortisation.Amount

        ^CTR85 = RP:Expense.Operating.MotorVehicle.Amount

        ^CTR86 = RP:Expense.Operating.RepairsAndMaintenance.Amount

        ^CTR87 = RP:Expense.AssetRevaluationsUnrealisedLosses.Amount

        ^CTR88 = RP:Expense.Other.Amount

        ^CTR91 = RP:Income.CapitalGainsNet.Amount

        ^CTR92 = RP:Expense.NonDeductibleExemptIncome.Amount

        ^CTR93 = RP:IncomeTax.FrankingCredits.ReceivedFromAustralianCompanies.Amount

        ^CTR94 = RP:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount

        ^CTR255 = RP.TOFA:Income.OtherTotal.Amount

        ^CTR95 = RP:Income.AssessableOther.Amount

        ^CTR96 = RP:Expense.NonDeductible.Amount

        ^CTR97 = RP:TaxConcession.ResearchAndDevelopment.WriteBackOfAccountingExpenditure.Amount

        ^CTR98 = RP:Expense.FlowOnDividendDeduction.Amount

        ^CTR99 = RP:Expense.DepreciationAllowableDeduction.Amount

        ^CTR100 = RP:Expense.ForestryManagedInvestmentSchemeDeduction.Amount

        ^CTR101 = RP:Expense.CapitalExpenditureImmediateDeduction.Amount

        ^CTR102 = RP:Expense.ProjectPoolAllowableDeduction.Amount

        ^CTR103 = RP:Expense.CapitalWorksDeduction.Amount

        ^CTR104 = RP:Expense.CapitalExpenditureSpecifiedAllowableDeduction.Amount

        ^CTR111 = RP.Y0:Expense.LandcareOperationsAcceleratedDepreciation.Amount

        ^CTR112 = RP:Expense.EnvironmentalProtectionDeduction.Amount

        ^CTR114 = RP:Income.Exempt.Amount

        ^CTR115 = RP:Income.NotincludedInAssessableOther.Amount

        ^CTR256 = RP.TOFA:Expense.DeductibleOther.Amount

        ^CTR116 = RP:Expense.DeductibleOther.Amount

        ^CTR117 = RP:IncomeTax.Deduction.TaxLossesDeducted.Amount

        ^CTR118 = RP:IncomeTax.TaxLossesTransferredIn.Amount
        */
            VRATOCTR430216(response, report);
#endregion // VR.ATO.CTR.430216

#endregion Manual Rules - Rules marked for manual coding
            VRATOCTR428014();
            VRATOCTR428017();
            VRATOCTR428018();
            VRATOCTR428019();
            VRATOCTR428020();
            VRATOCTR428021();
            VRATOCTR428022();
            VRATOCTR428023();
            VRATOCTR428024();
            VRATOCTR428026();
            VRATOCTR428028();
            VRATOCTR428030();
            VRATOCTR428033();
            VRATOCTR428038();
            VRATOCTR428041();
            VRATOCTR428042();
            VRATOCTR428043();
            VRATOCTR428045();
            VRATOCTR428046();
            VRATOCTR428052();
            VRATOCTR428053();
            VRATOCTR428057();
            VRATOCTR428058();
            VRATOCTR428061();
            VRATOCTR428062();
            VRATOCTR428063();
            VRATOCTR428064();
            VRATOCTR428067();
            VRATOCTR428076();
            VRATOCTR428078();
            VRATOCTR428081();
            VRATOCTR428086();
            VRATOCTR428091();
            VRATOCTR428104();
            VRATOCTR428105();
            VRATOCTR428110();
            VRATOCTR428111();
            VRATOCTR428115();
            VRATOCTR428136();
            VRATOCTR428140();
            VRATOCTR428149();
            VRATOCTR428150();
            VRATOCTR428151();
            VRATOCTR428152();
            VRATOCTR428153();
            VRATOCTR428155();
            VRATOCTR428156();
            VRATOCTR428164();
            VRATOCTR428165();
            VRATOCTR428166();
            VRATOCTR428167();
            VRATOCTR428168();
            VRATOCTR428169();
            VRATOCTR428170();
            VRATOCTR428186();
            VRATOCTR428197();
            VRATOCTR428198();
            VRATOCTR428199();
            VRATOCTR428200();
            VRATOCTR428201();
            VRATOCTR428202();
            VRATOCTR428204();
            VRATOCTR428205();
            VRATOCTR428206();
            VRATOCTR428207();
            VRATOCTR428209();
            VRATOCTR428210();
            VRATOCTR428211();
            VRATOCTR428216();
            VRATOCTR428220();
            VRATOCTR428221();
            VRATOCTR428225();
            VRATOCTR428226();
            VRATOCTR428227();
            VRATOCTR428228();
            VRATOCTR428235();
            VRATOCTR428251();
            VRATOCTR428253();
            VRATOCTR428254();
            VRATOCTR428255();
            VRATOCTR428256();
            VRATOCTR428259();
            VRATOCTR428261();
            VRATOCTR428262();
            VRATOCTR428263();
            VRATOCTR428264();
            VRATOCTR428265();
            VRATOCTR428272();
            VRATOCTR428273();
            VRATOCTR428273_1();
            VRATOCTR428273_2();
            VRATOCTR428273_3();
            VRATOCTR428273_4();
            VRATOCTR428273_5();
            VRATOCTR428273_6();
            VRATOCTR428273_7();
            VRATOCTR428273_8();
            VRATOCTR428273_9();
            VRATOCTR428273_10();
            VRATOCTR428273_11();
            VRATOCTR428273_12();
            VRATOCTR428273_13();
            VRATOCTR428273_14();
            VRATOCTR428273_15();
            VRATOCTR428273_16();
            VRATOCTR428273_17();
            VRATOCTR428273_18();
            VRATOCTR428273_19();
            VRATOCTR428273_20();
            VRATOCTR428273_21();
            VRATOCTR428274();
            VRATOCTR428327();
            VRATOCTR428328();
            VRATOCTR428329();
            VRATOCTR428330();
            VRATOCTR428335();
            VRATOCTR428336();
            VRATOCTR428337();
            VRATOCTR428338();
            VRATOCTR428339();
            VRATOCTR428340();
            VRATOCTR428341();
            VRATOCTR428342();
            VRATOCTR428343();
            VRATOCTR428344();
            VRATOCTR428345();
            VRATOCTR428349();
            VRATOCTR428350();
            VRATOCTR428351();
            VRATOCTR428356();
            VRATOCTR428358();
            VRATOCTR428366();
            VRATOCTR428401();
            VRATOCTR428402();
            VRATOCTR428428();
            VRATOCTR429831();
            VRATOCTR429833();
            VRATOCTR429835();
            VRATOCTR429836();
            VRATOCTR429837();
            VRATOCTR429838();
            VRATOCTR429839();
            VRATOCTR429840();
            VRATOCTR429841();
            VRATOCTR429842();
            VRATOCTR429844();
            VRATOCTR429845();
            VRATOCTR429846();
            VRATOCTR429847();
            VRATOCTR429848();
            VRATOCTR429849();
            VRATOCTR429850();
            VRATOCTR429852();
            VRATOCTR429856();
            VRATOCTR429858();
            VRATOCTR429859();
            VRATOCTR429860();
            VRATOCTR429862();
            VRATOCTR429863();
            VRATOCTR429864();
            VRATOCTR429865();
            VRATOCTR429866();
            VRATOCTR429867();
            VRATOCTR429868();
            VRATOCTR429869();
            VRATOCTR429870();
            VRATOCTR429871();
            VRATOCTR429872();
            VRATOCTR429873();
            VRATOCTR429874();
            VRATOCTR429875();
            VRATOCTR429877();
            VRATOCTR429878();
            VRATOCTR429879();
            VRATOCTR429880();
            VRATOCTR429881();
            VRATOCTR429882();
            VRATOCTR429883();
            VRATOCTR429886();
            VRATOCTR429887();
            VRATOCTR429888();
            VRATOCTR429889();
            VRATOCTR429890();
            VRATOCTR429891();
            VRATOCTR429892();
            VRATOCTR429893();
            VRATOCTR429894();
            VRATOCTR429895();
            VRATOCTR429896();
            VRATOCTR429897();
            VRATOCTR429898();
            VRATOCTR429899();
            VRATOCTR429900();
            VRATOCTR429901();
            VRATOCTR429902();
            VRATOCTR429903();
            VRATOCTR429908();
            VRATOCTR429909();
            VRATOCTR429911();
            VRATOCTR429912();
            VRATOCTR429913();
            VRATOCTR429914();
            VRATOCTR429915();
            VRATOCTR429917();
            VRATOCTR429921();
            VRATOCTR429922();
            VRATOCTR429928();
            VRATOCTR429930();
            VRATOCTR429931();
            VRATOCTR429935();
            VRATOCTR429938();
            VRATOCTR429940();
            VRATOCTR429941();
            VRATOCTR429942();
            VRATOCTR429943();
            VRATOCTR429944();
            VRATOCTR429946();
            VRATOCTR429948();
            VRATOCTR429949();
            VRATOCTR429960();
            VRATOCTR429961();
            VRATOCTR429962();
            VRATOCTR429963();
            VRATOCTR429964();
            VRATOCTR429965();
            VRATOCTR429966();
            VRATOCTR429967();
            VRATOCTR429968();
            VRATOCTR429973();
            VRATOCTR429975();
            VRATOCTR429977();
            VRATOCTR429978();
            VRATOCTR429979();
            VRATOCTR429980();
            VRATOCTR429981();
            VRATOCTR429982();
            VRATOCTR429982_1();
            VRATOCTR429984();
            VRATOCTR429985();
            VRATOCTR429986();
            VRATOCTR429987();
            VRATOCTR429989();
            VRATOCTR429991();
            VRATOCTR429997();
            VRATOCTR429998();
            VRATOCTR430000();
            VRATOCTR430001();
            VRATOCTR430004();
            VRATOCTR430007();
            VRATOCTR430008();
            VRATOCTR430013();
            VRATOCTR430022();
            VRATOCTR430024();
            VRATOCTR430025();
            VRATOCTR430026();
            VRATOCTR430026_1();
            VRATOCTR430026_2();
            VRATOCTR430027();
            VRATOCTR430029();
            VRATOCTR430032();
            VRATOCTR430033();
            VRATOCTR430034();
            VRATOCTR430035();
            VRATOCTR430040();
            VRATOCTR430041();
            VRATOCTR430042();
            VRATOCTR430043();
            VRATOCTR430044();
            VRATOCTR430045();
            VRATOCTR430046();
            VRATOCTR430047();
            VRATOCTR430048();
            VRATOCTR430049();
            VRATOCTR430050();
            VRATOCTR430051();
            VRATOCTR430052();
            VRATOCTR430053();
            VRATOCTR430056();
            VRATOCTR430057();
            VRATOCTR430068();
            VRATOCTR430217();
            VRATOCTR438040();
            VRATOCTR438041();
            VRATOCTR438042();
            VRATOCTR438047();
            VRATOCTR438051();
            VRATOCTR438067();
            VRATOCTR438076();
            VRATOCTR438077();
            VRATOCTR438078();
            VRATOCTR438081();
            VRATOCTR438083();
            VRATOCTR438085();
            VRATOCTR438088();
            VRATOCTR438089();
            VRATOCTR438090();
            VRATOCTR438091();
            VRATOCTR438092();
            VRATOCTR438093();
            VRATOCTR438095();
            VRATOCTR438097();
            VRATOCTR500001();
            VRATOCTR500154();
            VRATOCTR500155();
            VRATOCTR500157();
            VRATOCTR500158();
            VRATOCTR500163();
            VRATOCTR500164();
            VRATOCTR500186();
            VRATOCTR500217();
            VRATOCTR500235();
            VRATOCTR500235_1();
            VRATOCTR500235_2();
            VRATOCTR500300();
            VRATOCTR500301();
            VRATOCTR500302();
            VRATOCTR500303();
            VRATOCTR500306();
            VRATOCTR500380();
            VRATOCTR500381();
            VRATOCTR500386();
            VRATOCTR500387();
            VRATOCTR500389();
            VRATOCTR500396();
            VRATOCTR500397();
            VRATOCTR500398();
            VRATOCTR500399();
            VRATOCTR500400();
            VRATOCTRW00003();
            VRATOCTRW00004();
            VRATOCTRW00005();
            VRATOCTRW00006();
            VRATOCTRW00008();
            VRATOCTRW00013();
            VRATOCTRW00018();
            VRATOCTRW00019();
            VRATOCTRW00020();
            VRATOCTRW00021();
            VRATOCTRW00023();
            VRATOCTRW00024();
            VRATOCTRW00025();
            VRATOCTRW00026();
            VRATOCTRW00029();
            VRATOGEN000208();
            VRATOGEN000226();
            VRATOGEN000406();
            VRATOGEN000406_1();
            VRATOGEN000408();
            VRATOGEN000408_1();
            VRATOGEN000458();
            VRATOGEN000459();
            VRATOGEN001023();
            VRATOGEN001023_1();
            VRATOGEN001023_2();
            VRATOGEN001023_3();
            VRATOGEN001023_4();
            VRATOGEN001023_5();
            VRATOGEN001023_6();
            VRATOGEN001023_7();
            VRATOGEN001023_8();
            VRATOGEN001023_9();
            VRATOGEN001023_10();
            VRATOGEN001023_11();
            VRATOGEN001023_12();
            VRATOGEN001023_13();
            VRATOGEN001023_14();
            VRATOGEN001023_15();
            VRATOGEN001023_16();
            VRATOGEN001023_17();
            VRATOGEN001023_18();
            VRATOGEN001023_19();
            VRATOGEN001023_20();
            VRATOGEN001023_21();
            VRATOGEN001023_22();
            VRATOGEN010028();
            VRATOGEN010029();
            VRATOGEN010030();
            VRATOGEN408010();
            VRATOGEN408010_1();
            VRATOGEN410002();
            VRATOGEN410002_1();
            VRATOGEN410002_2();
            VRATOGEN410008();
            VRATOGEN410008_1();
            VRATOGEN410008_2();
            VRATOGEN410013();
            VRATOGEN410013_1();
            VRATOGEN410013_2();
            VRATOGEN410038();
            VRATOGEN410038_1();
            VRATOGEN410039();
            VRATOGEN410039_1();
            VRATOGEN410039_2();
            VRATOGEN410039_3();
            VRATOGEN410105();
            VRATOGEN410131();
            VRATOGEN410191();
            VRATOGEN410191_1();
            VRATOGEN410191_2();
            VRATOGEN410194();
            VRATOGEN410194_1();
            VRATOGEN410194_2();
            VRATOGEN410195();
            VRATOGEN410195_1();
            VRATOGEN410195_2();
            VRATOGEN410200();
            VRATOGEN410201();
            VRATOGEN410203();
            VRATOGEN410204();
            VRATOGEN410205();
            VRATOGEN410205_1();
            VRATOGEN410208();
            VRATOGEN410209();
            VRATOGEN410211();
            VRATOGEN410211_1();
            VRATOGEN410211_2();
            VRATOGEN410212();
            VRATOGEN410212_1();
            VRATOGEN410212_2();
            VRATOGEN410218();
            VRATOGEN428016();
            VRATOGEN428021();
            VRATOGEN428040();
            VRATOGEN428040_1();
            VRATOGEN428040_2();
            VRATOGEN428223();
            VRATOGEN428230();
            VRATOGEN428230_1();
            VRATOGEN428230_2();
            VRATOGEN428247();
            VRATOGEN428254();
            VRATOGEN428254_1();
            VRATOGEN428254_2();
            VRATOGEN428255();
            VRATOGEN428255_1();
            VRATOGEN428255_2();
            VRATOGEN428256();
            VRATOGEN428256_1();
            VRATOGEN428256_2();
            VRATOGEN428258();
            VRATOGEN428258_1();
            VRATOGEN428258_2();
            VRATOGEN428258_3();
            VRATOGEN428259();
            VRATOGEN428259_1();
            VRATOGEN428259_2();
            VRATOGEN428259_3();
            VRATOGEN428260();
            VRATOGEN428260_1();
            VRATOGEN428260_2();
            VRATOGEN428260_3();
            VRATOGEN428261();
            VRATOGEN428261_1();
            VRATOGEN428262();
            VRATOGEN428263();
            VRATOGEN428264();
            VRATOGEN428265();
            VRATOGEN428266();
            VRATOGEN428267();
            VRATOGEN430246();
            VRATOGEN430246_1();
            VRATOGEN430246_2();
            VRATOGEN430252();
            VRATOGEN430252_1();
            VRATOGEN430255();
            VRATOGEN430255_1();
            VRATOGEN432037();
            VRATOGEN432040();
            VRATOGEN432041();
            VRATOGEN432267();
            VRATOGEN432268();
            VRATOGEN438010();
            VRATOGEN438011();
            VRATOGEN438012();
            VRATOGEN438013();
            VRATOGEN438015();
            VRATOGEN438016();
            VRATOGEN438018();
            VRATOGEN438018_1();
            VRATOGEN438019();
            VRATOGEN438021();
            VRATOGEN438029();
            VRATOGEN438029_1();
            VRATOGEN438029_2();
            VRATOGEN438029_3();
            VRATOGEN438029_4();
            VRATOGEN438029_5();
            VRATOGEN438029_6();
            VRATOGEN438029_7();
            VRATOGEN438029_8();
            VRATOGEN438029_9();
            VRATOGEN438029_10();
            VRATOGEN438029_11();
            VRATOGEN438029_12();
            VRATOGEN438029_13();
            VRATOGEN438029_14();
            VRATOGEN438029_15();
            VRATOGEN438029_16();
            VRATOGEN438029_17();
            VRATOGEN438029_18();
            VRATOGEN438029_19();
            VRATOGEN438029_20();
            VRATOGEN438029_21();
            VRATOGEN438029_22();
            VRATOGEN438029_23();
            VRATOGEN438030();
            VRATOGEN438035();
            VRATOGEN438035_1();
            VRATOGEN438051();
            VRATOGEN438052();
            VRATOGEN438053();
            VRATOGEN438053_1();
            VRATOGEN438053_2();
            VRATOGEN438053_3();
            VRATOGEN438053_4();
            VRATOGEN438053_5();
            VRATOGEN438053_6();
            VRATOGEN438053_7();
            VRATOGEN438053_8();
            VRATOGEN438053_9();
            VRATOGEN438053_10();
            VRATOGEN438053_11();
            VRATOGEN438053_12();
            VRATOGEN438053_13();
            VRATOGEN438053_14();
            VRATOGEN438053_15();
            VRATOGEN438053_16();
            VRATOGEN438053_17();
            VRATOGEN438053_18();
            VRATOGEN438053_19();
            VRATOGEN438053_20();
            VRATOGEN438054();
            VRATOGEN438054_1();
            VRATOGEN438054_2();
            VRATOGEN438054_3();
            VRATOGEN438054_4();
            VRATOGEN438054_5();
            VRATOGEN438054_6();
            VRATOGEN438054_7();
            VRATOGEN438054_8();
            VRATOGEN438054_9();
            VRATOGEN438054_10();
            VRATOGEN438054_11();
            VRATOGEN438054_12();
            VRATOGEN438054_13();
            VRATOGEN438054_14();
            VRATOGEN438054_15();
            VRATOGEN438054_16();
            VRATOGEN438054_17();
            VRATOGEN438054_18();
            VRATOGEN438054_19();
            VRATOGEN438054_20();
            VRATOGEN438058();
            VRATOGEN438059();
            VRATOGEN438063();
            VRATOGEN438064();
            VRATOGEN438065();
            VRATOGEN438066();
            VRATOGEN438081();
            VRATOGEN438082();
            VRATOGEN500130();
            VRATOGEN500134();
            VRATOGEN500134_1();
            VRATOGEN500134_2();
            VRATOGEN500136();
            VRATOGEN500137();
            VRATOGEN500141();
            VRATOGEN500141_1();
            VRATOGEN500142();
            VRATOGEN500143();
            VRATOGEN500145();
            VRATOGEN500151();
            VRATOGEN500152();
            VRATOGEN500153();
#region Repeating CTR330Collection  
            if (report.CTR330Collection != null)
            {
                int itemIndex1 = 0;
                foreach (string CTR330 in report.CTR330Collection)
                {
                    itemIndex1++;
                    VRATOCTR430028(CTR330, itemIndex1);
                }
            }

#endregion // Foreach loop

            foreach (ProcessMessageDocument currentProcessMessage in response)
            {
                if (currentProcessMessage.Parameters != null)
                {
                    foreach (ProcessMessageParameter currentParameter in currentProcessMessage.Parameters)
                    {
                        if (string.IsNullOrEmpty(currentParameter.Name))
                        {
                            currentParameter.Name = _emptyParameterValue;
                        }

                        if (currentParameter.Name.Length > _maxParameterNameLength)
                        {
                            currentParameter.Name = currentParameter.Name.Substring(0, _maxParameterNameLength - 1);
                        }

                        if (string.IsNullOrEmpty(currentParameter.Value))
                        {
                            currentParameter.Value = _emptyParameterValue;
                        }

                        if (currentParameter.Value.Length > _maxParameterValueLength)
                        {
                            currentParameter.Value = currentParameter.Value.Substring(0, _maxParameterValueLength - 1);
                        }
                    }
                }
            }

            return response;
        }

#region VR.ATO.CTR.428014
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428014()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428014
            Form type is not valid
    
            Legacy Rule Format:
            IF ([CTR3] = 11110047)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            AsNumeric(^CTR3) = 11110047
    
            Data Elements:
    
            ^CTR3 = INT:Identifiers.TaxAgentNumber.Identifier
            */
            assertion = (AsNumeric(report.CTR3) == 11110047);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428014",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Form type is not valid",
                    LongDescription = @"Form Type is not valid for this lodger",
                    Location = "/xbrli:xbrl/tns:Identifiers.TaxAgentNumber.Identifier",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428014"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR3", Value = GetValueOrEmpty(report.CTR3) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428017
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428017()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428017
            Invalid code provided for In Liquidation field
    
            Legacy Rule Format:
            IF ([CTR10] <> NULLORBLANK) AND ([CTR10] <> "05")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR10 <> BLANK) AND (^CTR10 <> '05')
    
            Data Elements:
    
            ^CTR10 = RP:OrganisationDetails.OrganisationActivityStatus.Code
            */
            assertion = (string.IsNullOrWhiteSpace(report.CTR10) != true && report.CTR10 != @"05");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428017",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"In Liquidation code must be ""05""",
                    LongDescription = @"If the company is in liquidation, then code ""05"" must be supplied.",
                    Location = "/xbrli:xbrl/tns:OrganisationDetails.OrganisationActivityStatus.Code",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428017"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR10", Value = GetValueOrEmpty(report.CTR10) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428018
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428018()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428018
            Previous ACN present without previous name
    
            Legacy Rule Format:
            IF ([CTR14] <> NULLORBLANK) AND ([CTR12] = NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR14 <> BLANK) AND (^CTR12 = BLANK)
    
            Data Elements:
    
            ^CTR14 = RP.Y0-1:Identifiers.AustralianCompanyNumber.Identifier
    
            ^CTR12 = RP:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='P']
            */
            assertion = (string.IsNullOrWhiteSpace(report.CTR14) != true && string.IsNullOrWhiteSpace(report.CTR12) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428018",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Previous ACN present without previous name",
                    LongDescription = @"Company Income Tax return has a previous Australia Company Number (ACN) without a previous name noted",
                    Location = "/xbrli:xbrl/tns:Identifiers.AustralianCompanyNumber.Identifier",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428018"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR14", Value = GetValueOrEmpty(report.CTR14) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR12", Value = report.CTR12 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428019
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428019()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428019
            Previous name without either ARBN or ACN
    
            Legacy Rule Format:
            IF ([CTR12] <> NULLORBLANK) AND (([CTR13] = NULLORBLANK) AND ([CTR14] = NULLORBLANK))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR12 <> BLANK) AND ((^CTR13 = BLANK) AND (^CTR14 = BLANK))
    
            Data Elements:
    
            ^CTR12 = RP:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='P']
    
            ^CTR13 = RP.Y0-1:Identifiers.AustralianRegisteredBodyNumber.Identifier
    
            ^CTR14 = RP.Y0-1:Identifiers.AustralianCompanyNumber.Identifier
            */
            assertion = (string.IsNullOrWhiteSpace(report.CTR12) != true && (string.IsNullOrWhiteSpace(report.CTR13) == true && string.IsNullOrWhiteSpace(report.CTR14) == true));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428019",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Previous name without either ARBN or ACN",
                    LongDescription = @"The Company Income Tax return must have either an ARBN or ACN if previous name noted on return",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_MN_P_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428019"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR12", Value = report.CTR12 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR13", Value = GetValueOrEmpty(report.CTR13) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR14", Value = GetValueOrEmpty(report.CTR14) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428020
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428020()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428020
            Previous ARBN present without name
    
            Legacy Rule Format:
            IF ([CTR13] <> NULLORBLANK) AND ([CTR12] = NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR13 <> BLANK) AND (^CTR12 = BLANK)
    
            Data Elements:
    
            ^CTR13 = RP.Y0-1:Identifiers.AustralianRegisteredBodyNumber.Identifier
    
            ^CTR12 = RP:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='P']
            */
            assertion = (string.IsNullOrWhiteSpace(report.CTR13) != true && string.IsNullOrWhiteSpace(report.CTR12) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428020",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Previous ARBN present without name",
                    LongDescription = @"Company Income Tax return has a previous Australia Registered Business Number (ARBN) without a previous name noted",
                    Location = "/xbrli:xbrl/tns:Identifiers.AustralianRegisteredBodyNumber.Identifier",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428020"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR13", Value = GetValueOrEmpty(report.CTR13) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR12", Value = report.CTR12 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428021
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428021()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428021
            Please provide only one of either ARBN or ACN
    
            Legacy Rule Format:
            IF ([CTR14] <> NULLORBLANK) AND ([CTR13] <> NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR14 <> BLANK) AND (^CTR13 <> BLANK)
    
            Data Elements:
    
            ^CTR14 = RP.Y0-1:Identifiers.AustralianCompanyNumber.Identifier
    
            ^CTR13 = RP.Y0-1:Identifiers.AustralianRegisteredBodyNumber.Identifier
            */
            assertion = (string.IsNullOrWhiteSpace(report.CTR14) != true && string.IsNullOrWhiteSpace(report.CTR13) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428021",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Please provide only one of either ARBN or ACN",
                    LongDescription = @"You may only provide a previous ARBN or ACN on the Company Income Tax return",
                    Location = "/xbrli:xbrl/tns:Identifiers.AustralianCompanyNumber.Identifier",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428021"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR14", Value = GetValueOrEmpty(report.CTR14) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR13", Value = GetValueOrEmpty(report.CTR13) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428022
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428022()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428022
            Ultimate holding company name details required when ABN or overseas location code is present
    
            Legacy Rule Format:
            IF (([CTR45] <> NULLORBLANK) OR ([CTR44] <> NULLORBLANK)) AND ([CTR43] = NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ((^CTR45 <> BLANK) OR (^CTR44 <> BLANK)) AND (^CTR43 = BLANK)
    
            Data Elements:
    
            ^CTR45 = RP:UltimateHoldingCompany:Residency.TaxPurposesCountry.Code
    
            ^CTR44 = RP:UltimateHoldingCompany:Identifiers.AustralianBusinessNumber.Identifier
    
            ^CTR43 = RP:UltimateHoldingCompany:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
            */
            assertion = ((string.IsNullOrWhiteSpace(report.CTR45) != true || string.IsNullOrWhiteSpace(report.CTR44) != true) && string.IsNullOrWhiteSpace(report.CTR43) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428022",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Ultimate holding company name details required when ABN or overseas location code is present",
                    LongDescription = @"Where an Ultimate holding company ABN or overseas location code is present, an ultimate holding company name must be provided",
                    Location = "/xbrli:xbrl/tns:UltimateHoldingCompany/tns:Residency.TaxPurposesCountry.Code",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428022"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR45", Value = GetValueOrEmpty(report.CTR45) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR44", Value = GetValueOrEmpty(report.CTR44) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR43", Value = report.CTR43 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428023()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428023
            Either ultimate holding company ABN or the country code is required. Both must not be present.
    
            Legacy Rule Format:
            IF([CTR44] <> NULLORBLANK) AND ([CTR45] <> NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR44 <> BLANK) AND (^CTR45 <> BLANK)
    
            Data Elements:
    
            ^CTR45 = RP:UltimateHoldingCompany:Residency.TaxPurposesCountry.Code
    
            ^CTR44 = RP:UltimateHoldingCompany:Identifiers.AustralianBusinessNumber.Identifier
            */
            assertion = (string.IsNullOrWhiteSpace(report.CTR44) != true && string.IsNullOrWhiteSpace(report.CTR45) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Either ultimate holding company ABN or the country code is required. Both must not be present.",
                    Location = "/xbrli:xbrl/tns:UltimateHoldingCompany/tns:Residency.TaxPurposesCountry.Code",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428023"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR44", Value = GetValueOrEmpty(report.CTR44) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR45", Value = GetValueOrEmpty(report.CTR45) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428024
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428024()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428024
            Immediate holding company name details required when immediate holding company ABN is present
    
            Legacy Rule Format:
            IF ([CTR47] <> NULLORBLANK) AND ([CTR46] = NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR47 <> BLANK) AND (^CTR46 = BLANK)
    
            Data Elements:
    
            ^CTR47 = RP:ImmediateHoldingCompany:Identifiers.AustralianBusinessNumber.Identifier
    
            ^CTR46 = RP:ImmediateHoldingCompany:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.CTR47) != true && string.IsNullOrWhiteSpace(report.CTR46) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428024",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Immediate holding company name details required when immediate holding company ABN is present",
                    LongDescription = @"Where an immediate holding company ABN is present at Item 1 on the form cover, then the immediate holding company name must be present",
                    Location = "/xbrli:xbrl/tns:ImmediateHoldingCompany/tns:Identifiers.AustralianBusinessNumber.Identifier",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428024"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR47", Value = GetValueOrEmpty(report.CTR47) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR46", Value = report.CTR46 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428026
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428026()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428026
            For corporate unit trusts or public trading trusts,  Business industry code 62400 must be provided. Residency and Status of business must not be provided.
    
            Legacy Rule Format:
            IF ([CTR52] = SET("054","174")) AND (([CTR51] <> NULLORBLANK) OR ([CTR54] <> NULLORBLANK) OR ([CTR49] <> "62400"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (InSet(^CTR52, '"054","174"')) AND ((^CTR51 <> BLANK) OR (^CTR54 <> BLANK) OR (^CTR49 <> '62400'))
    
            Data Elements:
    
            ^CTR52 = RP:OrganisationDetails.OrganisationType.Code
    
            ^CTR51 = RP:Residency.TaxPurposesOrganisationStatus.Code
    
            ^CTR54 = RP:OrganisationDetails.ActivityEvent.Code
    
            ^CTR49 = RP:OrganisationDetails.OrganisationIndustry2006Extended.Code
            */
            assertion = (IsMatch(report.CTR52, @"^(054|174)$") && (string.IsNullOrWhiteSpace(report.CTR51) != true || string.IsNullOrWhiteSpace(report.CTR54) != true || report.CTR49 != @"62400"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.430067",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"For corp. unit trusts or public trading trusts, you must provide Business industry code 62400. Residency and Status of business must not be provided.",
                    Location = "/xbrli:xbrl/tns:OrganisationDetails.OrganisationType.Code",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428026"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR52", Value = report.CTR52 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR51", Value = GetValueOrEmpty(report.CTR51) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR54", Value = GetValueOrEmpty(report.CTR54) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR49", Value = report.CTR49 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428028
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428028()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428028
            Invalid Percentage of foreign shareholding (valid values are 1-100%)
    
            Legacy Rule Format:
            IF [CTR50] <> NULL AND [CTR50] <> SET(1-100) 
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR50 <> NULL AND (^CTR50 < 1 OR ^CTR50 > 100)
    
            Data Elements:
    
            ^CTR50 = RP:Equity.ContributedEquity.NonAustralianShareholding.Percent
            */
            assertion = (report.CTR50 != null && (report.CTR50.GetValueOrDefault() < 1 || report.CTR50.GetValueOrDefault() > 100));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428028",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid Percentage of foreign shareholding (valid values are 1-100%)",
                    LongDescription = @"Valid Percentage of foreign shareholding are 1 to 100",
                    Location = "/xbrli:xbrl/tns:Equity.ContributedEquity.NonAustralianShareholding.Percent",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428028"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR50", Value = report.CTR50.GetValueOrDefault().ToString() });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428030
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428030()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428030
            Residency status of company required. Except corporate unit trust or public trading trust
    
            Legacy Rule Format:
            IF ([CTR52] <> SET("054","174")) AND ([CTR51] = NULL)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (NotInSet(^CTR52, '"054","174"')) AND (^CTR51 = NULL)
    
            Data Elements:
    
            ^CTR52 = RP:OrganisationDetails.OrganisationType.Code
    
            ^CTR51 = RP:Residency.TaxPurposesOrganisationStatus.Code
            */
            assertion = (!(IsMatch(report.CTR52, @"^(054|174)$", RegexOptions.IgnoreCase)) && report.CTR51 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428368",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Residency status of company required. Except corporate unit trust or public trading trust",
                    LongDescription = @"Residency is required unless the Company is a corporate unit trust or public trading trust",
                    Location = "/xbrli:xbrl/tns:OrganisationDetails.OrganisationType.Code",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428030"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR52", Value = report.CTR52 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR51", Value = GetValueOrEmpty(report.CTR51) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428033
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428033()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428033
            Invalid Company Type code - Valid Company Types codes are 027, 032, 045, 054,120,139,163, 174, 205
    
            Legacy Rule Format:
            IF (CTR:RP:pyde.xx.xx:OrganisationDetails.OrganisationType.Code <> NULLORBLANK) AND (CTR:RP:pyde.xx.xx:OrganisationDetails.OrganisationType.Code <> SET ("027","032","045","054","120","139","163","174","205"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR52 <> BLANK) AND (NotInSet(^CTR52, '"027","032","045","054","120","139","163","174","205"'))
    
            Data Elements:
    
            ^CTR52 = RP:OrganisationDetails.OrganisationType.Code
            */
            assertion = (string.IsNullOrWhiteSpace(report.CTR52) != true && !(IsMatch(report.CTR52, @"^(027|032|045|054|120|139|163|174|205)$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.430223",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid Company Type code - Valid Company Types codes are 027, 032, 045, 054,120,139,163, 174, 205",
                    Location = "/xbrli:xbrl/tns:OrganisationDetails.OrganisationType.Code",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428033"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR52", Value = report.CTR52 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428038
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428038()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428038
            Valid Interposed Entity Election revocation code is "R"
    
            Legacy Rule Format:
            IF ([CTR56] <> NULLORBLANK) AND ([CTR56] <> "R")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR56 <> BLANK) AND (^CTR56 <> 'R')
    
            Data Elements:
    
            ^CTR56 = RP:Elections.InterposedEntityElectionOrRevocation.Code
            */
            assertion = (string.IsNullOrWhiteSpace(report.CTR56) != true && report.CTR56 != @"R");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428038",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Valid Interposed Entity Election revocation code is ""R""",
                    Location = "/xbrli:xbrl/tns:Elections.InterposedEntityElectionOrRevocation.Code",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428038"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR56", Value = GetValueOrEmpty(report.CTR56) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428041
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428041()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428041
            In the Calculation statement, Gross payment where ABN not quoted amount, is incorrect
    
            Legacy Rule Format:
            IF ([CTR58] > 0) AND 
            ([CTR208] > (([CTR58] * 0.50) +1))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR58 > 0) AND (^CTR208 > ((^CTR58 * 0.50) +1))
    
            Data Elements:
    
            ^CTR58 = RP:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^CTR208 = RP:ITPAYGWithholdingDetails:IncomeTax.PayAsYouGoWithholding.TaxWithheld.Amount WHERE [IncomeTax.PayAsYouGoWithholding.PaymentType.Code='DNOABN']
            */
            assertion = (report.CTR58.GetValueOrDefault() > 0 && report.CTR208.GetValueOrDefault() > report.CTR58.GetValueOrDefault() * 0.50M + 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428369",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"In the Calculation statement, Gross payment where ABN not quoted amount, is incorrect",
                    LongDescription = @"Tax withheld where ABN not quoted in the calculation statement cannot be greater than 50% of Gross payments where ABN not quoted",
                    Location = "/xbrli:xbrl/tns:Remuneration.ABNNotQuotedPaymentGross.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428041"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR58", Value = GetValueOrEmpty(report.CTR58) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR208", Value = GetValueOrEmpty(report.CTR208) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428042
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428042()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428042
            In the Calculation statement, Tax withheld where ABN not quoted is required, where Item 6 Label A completed
    
            Legacy Rule Format:
            IF ([CTR58] > 0) AND ([CTR208] = NULL)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR58 > 0) AND (^CTR208 = NULL)
    
            Data Elements:
    
            ^CTR58 = RP:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^CTR208 = RP:ITPAYGWithholdingDetails:IncomeTax.PayAsYouGoWithholding.TaxWithheld.Amount WHERE [IncomeTax.PayAsYouGoWithholding.PaymentType.Code='DNOABN']
            */
            assertion = (report.CTR58.GetValueOrDefault() > 0 && report.CTR208 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428370",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Calculation statement, Tax withheld where ABN not quoted is required",
                    LongDescription = @"An amount has been entered at Gross payments where ABN not quoted and Tax withheld where ABN not quoted in the calculation statement is missing",
                    Location = "/xbrli:xbrl/tns:Remuneration.ABNNotQuotedPaymentGross.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428042"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR58", Value = GetValueOrEmpty(report.CTR58) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR208", Value = GetValueOrEmpty(report.CTR208) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428043
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428043()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428043
            Check other refundable credits including film tax offset in calculation statement
    
            Legacy Rule Format:
            IF ([CTR211] = NULL) AND (([CTR60] > 0) OR ([CTR61] > 0))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR211 = NULL) AND ((^CTR60 > 0) OR (^CTR61 > 0))
    
            Data Elements:
    
            ^CTR211 = RP:IncomeTax.PayAsYouGoWithholding.CreditsRefundableOther.Amount
    
            ^CTR60 = RP:Income.PartnershipDistributionGross.Amount
    
            ^CTR61 = RP:TrustDistributionIncome:Income.TrustDistributionGross.Amount
            */
            assertion = (report.CTR211 == null && (report.CTR60.GetValueOrDefault() > 0 || report.CTR61.GetValueOrDefault() > 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428371",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Check other refundable credits including film tax offset in calculation statement",
                    LongDescription = @"At the calculation of total profit or loss, an amount is present at gross distribution from partnerships or gross distribution from trusts and other refundable credits including film tax offset dismissing in the calculation statement. Please input the missing amount/s (if there are no credits, enter zero at the other refundable credits label in the calculation statement)",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditsRefundableOther.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428043"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR211", Value = GetValueOrEmpty(report.CTR211) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR60", Value = GetValueOrEmpty(report.CTR60) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR61", Value = GetValueOrEmpty(report.CTR61) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428045
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428045()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428045
            In Calculation statement, Credit for Tax withheld-Foreign resident withholding (excluding capital gains) amount is incorrect.
    
            Legacy Rule Format:
            IF (([CTR207] > 0 ) AND ([CTR60] = NULL) AND ([CTR61] = NULL)) AND 
            ([CTR207] > (([CTR57] * 0.50) + 1))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ((^CTR207 > 0) AND (^CTR60 = NULL) AND (^CTR61 = NULL)) AND (^CTR207 > ((^CTR57 * 0.50) + 1))
    
            Data Elements:
    
            ^CTR207 = RP:ITPAYGWithholdingDetails:IncomeTax.PayAsYouGoWithholding.TaxWithheld.Amount WHERE [IncomeTax.PayAsYouGoWithholding.PaymentType.Code='DFRW']
    
            ^CTR60 = RP:Income.PartnershipDistributionGross.Amount
    
            ^CTR61 = RP:TrustDistributionIncome:Income.TrustDistributionGross.Amount
    
            ^CTR57 = RP:Remuneration.PaymentToForeignResidentGross.Amount
            */
            assertion = (report.CTR207.GetValueOrDefault() > 0 && report.CTR60 == null && report.CTR61 == null && report.CTR207.GetValueOrDefault() > report.CTR57.GetValueOrDefault() * 0.50M + 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.438072",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"In Calculation statement, Credit for Tax withheld-Foreign resident withholding (excluding capital gains) amount is incorrect.",
                    LongDescription = @"Credit for tax withheld - Foreign resident withholding (excluding capital gains) (Calculation statement) cannot be greater than 50% of Gross payments subject to foreign resident withholding (excluding capital gains) unless relevant distributions from partnerships or trusts are shown.",
                    Location = "/xbrli:xbrl/tns:ITPAYGWithholdingDetails" + OccurrenceIndex(report.ItpaygWithholdingDetailsCollection_DFRW_RPOccurrenceIndex) + "/tns:IncomeTax.PayAsYouGoWithholding.TaxWithheld.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428045"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR207", Value = GetValueOrEmpty(report.CTR207) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR60", Value = GetValueOrEmpty(report.CTR60) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR61", Value = GetValueOrEmpty(report.CTR61) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR57", Value = GetValueOrEmpty(report.CTR57) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428046
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428046()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428046
            Gross payments subject to foreign resident withholding (excluding capital gains) amount required.
    
            Legacy Rule Format:
            IF ([CTR207] > 0 ) AND ([CTR57] = NULL) AND ([CTR60] = NULL) AND ([CTR61] = NULL)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR207 > 0) AND (^CTR57 = NULL) AND (^CTR60 = NULL) AND (^CTR61 = NULL)
    
            Data Elements:
    
            ^CTR207 = RP:ITPAYGWithholdingDetails:IncomeTax.PayAsYouGoWithholding.TaxWithheld.Amount WHERE [IncomeTax.PayAsYouGoWithholding.PaymentType.Code='DFRW']
    
            ^CTR57 = RP:Remuneration.PaymentToForeignResidentGross.Amount
    
            ^CTR60 = RP:Income.PartnershipDistributionGross.Amount
    
            ^CTR61 = RP:TrustDistributionIncome:Income.TrustDistributionGross.Amount
            */
            assertion = (report.CTR207.GetValueOrDefault() > 0 && report.CTR57 == null && report.CTR60 == null && report.CTR61 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.438073",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Gross payments subject to foreign resident withholding (excluding capital gains) amount required.",
                    LongDescription = @"Credit for tax withheld - Foreign resident withholding (excluding capital gains) (Calculation statement) has an amount greater than zero and Gross payments subject to foreign resident withholding (excluding capital gains) and gross distributions from partnerships and gross distributions from trusts are missing.",
                    Location = "/xbrli:xbrl/tns:ITPAYGWithholdingDetails" + OccurrenceIndex(report.ItpaygWithholdingDetailsCollection_DFRW_RPOccurrenceIndex) + "/tns:IncomeTax.PayAsYouGoWithholding.TaxWithheld.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428046"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR207", Value = GetValueOrEmpty(report.CTR207) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR57", Value = GetValueOrEmpty(report.CTR57) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR60", Value = GetValueOrEmpty(report.CTR60) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR61", Value = GetValueOrEmpty(report.CTR61) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428052
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428052()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428052
            Check assessable government industry payments amount, Calculation of total profit or loss (Label Q)
    
            Legacy Rule Format:
            IF ([CTR69] <> NULL) AND (([CTR68] <= 0) OR ([CTR68] = NULL))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR69 <> NULL) AND ((^CTR68 <= 0) OR (^CTR68 = NULL))
    
            Data Elements:
    
            ^CTR69 = RP:GovernmentFunding.GovernmentIndustryPaymentsIncludesFuel.Indicator
    
            ^CTR68 = RP:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
            */
            assertion = (report.CTR69 != null && (report.CTR68.GetValueOrDefault() <= 0 || report.CTR68 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428052",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Check assessable government industry payments amount, Calculation of total profit or loss (Label Q)",
                    LongDescription = @"Assessable government industry payments action code is present and the amount of assessable government industry payments is either absent or zero. (Item 6 Income Label Q)",
                    Location = "/xbrli:xbrl/tns:GovernmentFunding.GovernmentIndustryPaymentsIncludesFuel.Indicator",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428052"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR69", Value = GetValueOrEmpty(report.CTR69) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR68", Value = GetValueOrEmpty(report.CTR68) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428053
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428053()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428053
            Invalid Assessable government industry payments action code.  Valid code is "true"
    
            Legacy Rule Format:
            IF ([CTR69] <> NULL) AND ([CTR69] <> TRUE) 
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR69 <> NULL) AND (^CTR69 <> TRUE)
    
            Data Elements:
    
            ^CTR69 = RP:GovernmentFunding.GovernmentIndustryPaymentsIncludesFuel.Indicator
            */
            assertion = (report.CTR69 != null && report.CTR69 != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428053",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid Assessable government industry payments action code. Valid code is ""true""",
                    LongDescription = @"Valid Assessable government industry payments action code is ""true""",
                    Location = "/xbrli:xbrl/tns:GovernmentFunding.GovernmentIndustryPaymentsIncludesFuel.Indicator",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428053"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR69", Value = GetValueOrEmpty(report.CTR69) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428057
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428057()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428057
            Total salary and wage expenses cannot exceed total expenses
    
            Legacy Rule Format:
            IF([CTR151]<> NULL) AND ([CTR151] > ([CTR72] + [CTR73] + [CTR364] + [CTR75] + [CTR76] + [CTR77] + [CTR78] + [CTR79] + [CTR80] + [CTR81] + [CTR365] + [CTR366] + [CTR84] + [CTR85] + [CTR86] + [CTR87] + [CTR88]))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR151 <> NULL) AND (^CTR151 > (^CTR72 + ^CTR73 + ^CTR364 + ^CTR75 + ^CTR76 + ^CTR77 + ^CTR78 + ^CTR79 + ^CTR80 + ^CTR81 + ^CTR365 + ^CTR366 + ^CTR84 + ^CTR85 + ^CTR86 + ^CTR87 + ^CTR88))
    
            Data Elements:
    
            ^CTR151 = RP:Remuneration.WagesAndSalaries.Total.Amount
    
            ^CTR72 = RP:Expense.ForeignResidentWithholding.Amount
    
            ^CTR73 = RP:Expense.Operating.CostOfSales.Amount
    
            ^CTR364 = RP:Remuneration.WagesAndSalaries.PaymentsContractorAndConsultant.Amount
    
            ^CTR75 = RP:SuperannuationContribution.EmployerContributions.Amount
    
            ^CTR76 = RP:Expense.Operating.BadDebts.Amount
    
            ^CTR77 = RP.JAUS:Expense.Operating.Lease.Amount
    
            ^CTR78 = RP.JFOREIGN:Expense.Operating.Lease.Amount
    
            ^CTR79 = RP:Expense.Operating.Rent.Amount
    
            ^CTR80 = RP.JAUS:Expense.Operating.Interest.Amount
    
            ^CTR81 = RP.JFOREIGN:Expense.Operating.Interest.Amount
    
            ^CTR365 = RP.JAUS:Expense.Royalties.Amount
    
            ^CTR366 = RP.JFOREIGN:Expense.Royalties.Amount
    
            ^CTR84 = RP:Expense.DepreciationAndAmortisation.Amount
    
            ^CTR85 = RP:Expense.Operating.MotorVehicle.Amount
    
            ^CTR86 = RP:Expense.Operating.RepairsAndMaintenance.Amount
    
            ^CTR87 = RP:Expense.AssetRevaluationsUnrealisedLosses.Amount
    
            ^CTR88 = RP:Expense.Other.Amount
            */
            assertion = (report.CTR151 != null && report.CTR151.GetValueOrDefault() > report.CTR72.GetValueOrDefault() + report.CTR73.GetValueOrDefault() + report.CTR364.GetValueOrDefault() + report.CTR75.GetValueOrDefault() + report.CTR76.GetValueOrDefault() + report.CTR77.GetValueOrDefault() + report.CTR78.GetValueOrDefault() + report.CTR79.GetValueOrDefault() + report.CTR80.GetValueOrDefault() + report.CTR81.GetValueOrDefault() + report.CTR365.GetValueOrDefault() + report.CTR366.GetValueOrDefault() + report.CTR84.GetValueOrDefault() + report.CTR85.GetValueOrDefault() + report.CTR86.GetValueOrDefault() + report.CTR87.GetValueOrDefault() + report.CTR88.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428375",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total salary and wage expenses cannot exceed total expenses",
                    LongDescription = @"Total salary and wage expenses on a Company Income Tax Return cannot exceed the total of Profit or loss expenses",
                    Location = "/xbrli:xbrl/tns:Remuneration.WagesAndSalaries.Total.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428057"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR151", Value = GetValueOrEmpty(report.CTR151) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR72", Value = GetValueOrEmpty(report.CTR72) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR73", Value = GetValueOrEmpty(report.CTR73) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR364", Value = GetValueOrEmpty(report.CTR364) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR75", Value = GetValueOrEmpty(report.CTR75) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR76", Value = GetValueOrEmpty(report.CTR76) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR77", Value = GetValueOrEmpty(report.CTR77) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR78", Value = GetValueOrEmpty(report.CTR78) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR79", Value = GetValueOrEmpty(report.CTR79) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR80", Value = GetValueOrEmpty(report.CTR80) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR81", Value = GetValueOrEmpty(report.CTR81) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR365", Value = GetValueOrEmpty(report.CTR365) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR366", Value = GetValueOrEmpty(report.CTR366) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR84", Value = GetValueOrEmpty(report.CTR84) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR85", Value = GetValueOrEmpty(report.CTR85) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR86", Value = GetValueOrEmpty(report.CTR86) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR87", Value = GetValueOrEmpty(report.CTR87) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR88", Value = GetValueOrEmpty(report.CTR88) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428058
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428058()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428058
            Cost of Sales not Provided
    
            Legacy Rule Format:
            IF [CTR73] = NULL AND ([CTR59] <> NULL OR [CTR123] <> NULL OR [CTR124] <> NULL OR [CTR125] <> NULL)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR73 = NULL AND (^CTR59 <> NULL OR ^CTR123 <> NULL OR ^CTR124 <> NULL OR ^CTR125 <> NULL)
    
            Data Elements:
    
            ^CTR73 = RP:Expense.Operating.CostOfSales.Amount
    
            ^CTR59 = RP:Income.SaleOfGoodsAndServices.Other.Amount
    
            ^CTR123 = RP.Opening(Instant):Assets.Inventories.Total.Amount
    
            ^CTR124 = RP:Expense.Purchases.AndOtherCosts.Amount
    
            ^CTR125 = RP.Closing(Instant):Assets.Inventories.Total.Amount
            */
            assertion = (report.CTR73 == null && (report.CTR59 != null || report.CTR123 != null || report.CTR124 != null || report.CTR125 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428376",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Trading account items are present without cost of sales.  Include cost of sales to clear error.",
                    Location = "/xbrli:xbrl/tns:Expense.Operating.CostOfSales.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428058"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR73", Value = GetValueOrEmpty(report.CTR73) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR59", Value = GetValueOrEmpty(report.CTR59) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR123", Value = GetValueOrEmpty(report.CTR123) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR124", Value = GetValueOrEmpty(report.CTR124) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR125", Value = GetValueOrEmpty(report.CTR125) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428061
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428061()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428061
            An amount is present at deduction for decline in value of  depreciating assets and there is no amount at depreciation expenses or an amount is present at depreciation expenses and there is no amount at deduction for decline in value of  depreciating assets and the tax payer is not claiming any small business entity depreciating assets deductions
    
            Legacy Rule Format:
            IF ((([CTR99] > 0) AND ([CTR84] = NULL)) OR 
            (([CTR84] > 0) AND ([CTR99] = NULL))) AND 
            ([CTR168] = NULL) AND ([CTR169] = NULL)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (((^CTR99 > 0) AND (^CTR84 = NULL)) OR ((^CTR84 > 0) AND (^CTR99 = NULL))) AND (^CTR168 = NULL) AND (^CTR169 = NULL)
    
            Data Elements:
    
            ^CTR99 = RP:Expense.DepreciationAllowableDeduction.Amount
    
            ^CTR84 = RP:Expense.DepreciationAndAmortisation.Amount
    
            ^CTR168 = RP.LowCost:Expense.DepreciatingAssets.SmallBusinessEntity.Amount
    
            ^CTR169 = RP.GeneralPool:Expense.DepreciatingAssets.SmallBusinessEntity.Amount
            */
            assertion = ((report.CTR99.GetValueOrDefault() > 0 && report.CTR84 == null || report.CTR84.GetValueOrDefault() > 0 && report.CTR99 == null) && report.CTR168 == null && report.CTR169 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.430021",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Check depreciation expenses or deduction for decline in value of  depreciating assets amount",
                    LongDescription = @"An amount is present at deduction for decline in value of  depreciating assets and there is no amount at depreciation expenses or an amount is present at depreciation expenses and there is no amount at deduction for decline in value of  depreciating assets and the tax payer is not claiming any small business entity simplified depreciating assets deductions",
                    Location = "/xbrli:xbrl/tns:Expense.DepreciationAllowableDeduction.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428061"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR99", Value = GetValueOrEmpty(report.CTR99) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR84", Value = GetValueOrEmpty(report.CTR84) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR168", Value = GetValueOrEmpty(report.CTR168) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR169", Value = GetValueOrEmpty(report.CTR169) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428062
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428062()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428062
            Depreciation expenses cannot be greater than accounting expenditure subject to R&D tax incentive plus non deductible expenses for a taxpayer not claiming any small business entity depreciating assets deductions
    
            Legacy Rule Format:
            IF ([CTR84] > ([CTR97] + [CTR96])) AND ([CTR168] = NULL) AND ([CTR169] = NULL)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR84 > (^CTR97 + ^CTR96)) AND (^CTR168 = NULL) AND (^CTR169 = NULL)
    
            Data Elements:
    
            ^CTR84 = RP:Expense.DepreciationAndAmortisation.Amount
    
            ^CTR97 = RP:TaxConcession.ResearchAndDevelopment.WriteBackOfAccountingExpenditure.Amount
    
            ^CTR96 = RP:Expense.NonDeductible.Amount
    
            ^CTR168 = RP.LowCost:Expense.DepreciatingAssets.SmallBusinessEntity.Amount
    
            ^CTR169 = RP.GeneralPool:Expense.DepreciatingAssets.SmallBusinessEntity.Amount
            */
            assertion = (report.CTR84.GetValueOrDefault() > report.CTR97.GetValueOrDefault() + report.CTR96.GetValueOrDefault() && report.CTR168 == null && report.CTR169 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.430016",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Depreciation expenses is greater than accounting expenditure subject to R&D incentive plus non-deductible expenses",
                    LongDescription = @"Depreciation expenses cannot be greater than accounting expenditure subject to R&D tax incentive plus non deductible expenses for a taxpayer not claiming any small business entity simplified depreciating assets deductions",
                    Location = "/xbrli:xbrl/tns:Expense.DepreciationAndAmortisation.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428062"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR84", Value = GetValueOrEmpty(report.CTR84) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR97", Value = GetValueOrEmpty(report.CTR97) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR96", Value = GetValueOrEmpty(report.CTR96) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR168", Value = GetValueOrEmpty(report.CTR168) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR169", Value = GetValueOrEmpty(report.CTR169) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428063
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428063()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428063
            Depreciation expenses has a value and accounting expenditure subject to R&D tax incentive and non-deductible expenses are not supplied for a taxpayer not claiming any small business entity depreciating assets deductions
    
            Legacy Rule Format:
            IF ([CTR84] > 0) AND (([CTR96] = NULL) OR ([CTR96] = 0)) AND (([CTR97] = NULL) OR ([CTR97] = 0)) AND ([CTR168] = NULL) AND ([CTR169] = NULL) 
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR84 > 0) AND ((^CTR96 = NULL) OR (^CTR96 = 0)) AND ((^CTR97 = NULL) OR (^CTR97 = 0)) AND (^CTR168 = NULL) AND (^CTR169 = NULL)
    
            Data Elements:
    
            ^CTR84 = RP:Expense.DepreciationAndAmortisation.Amount
    
            ^CTR96 = RP:Expense.NonDeductible.Amount
    
            ^CTR97 = RP:TaxConcession.ResearchAndDevelopment.WriteBackOfAccountingExpenditure.Amount
    
            ^CTR168 = RP.LowCost:Expense.DepreciatingAssets.SmallBusinessEntity.Amount
    
            ^CTR169 = RP.GeneralPool:Expense.DepreciatingAssets.SmallBusinessEntity.Amount
            */
            assertion = (report.CTR84.GetValueOrDefault() > 0 && (report.CTR96 == null || report.CTR96 == 0) && (report.CTR97 == null || report.CTR97 == 0) && report.CTR168 == null && report.CTR169 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.430017",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Accounting expenditure subject to R&D tax incentive and non-deductible expenses not supplied",
                    LongDescription = @"Depreciation expenses has a value and accounting expenditure subject to R&D tax incentive and non-deductible expenses are not supplied for a taxpayer not claiming any small business entity simplified depreciating assets deductions",
                    Location = "/xbrli:xbrl/tns:Expense.DepreciationAndAmortisation.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428063"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR84", Value = GetValueOrEmpty(report.CTR84) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR96", Value = GetValueOrEmpty(report.CTR96) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR97", Value = GetValueOrEmpty(report.CTR97) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR168", Value = GetValueOrEmpty(report.CTR168) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR169", Value = GetValueOrEmpty(report.CTR169) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428064
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428064()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428064
            Response to CGT event question required, where amount noted at Net Capital gain (Item 7 Label A)
    
            Legacy Rule Format:
            IF ([CTR91] > 0) AND ([CTR89] <> TRUE)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR91 > 0) AND (^CTR89 <> TRUE)
    
            Data Elements:
    
            ^CTR91 = RP:Income.CapitalGainsNet.Amount
    
            ^CTR89 = RP:CapitalGainsTax.Event.Indicator
            */
            assertion = (report.CTR91.GetValueOrDefault() > 0 && report.CTR89 != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428378",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Response to CGT event question required, where amount noted at Net Capital gain",
                    LongDescription = @"Where an amount has been entered at net capital gain, the answer to the question ""Did you have a CGT event during the year"" must be Yes",
                    Location = "/xbrli:xbrl/tns:Income.CapitalGainsNet.Amount[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428064"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR91", Value = GetValueOrEmpty(report.CTR91) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR89", Value = GetValueOrEmpty(report.CTR89) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428067
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428067()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428067
            Invalid Closing stock action code
    
            Legacy Rule Format:
            IF ([CTR126] <> NULLORBLANK) AND ([CTR126] <> SET("C","M","R"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR126 <> BLANK) AND (NotInSet(^CTR126, '"C","M","R"'))
    
            Data Elements:
    
            ^CTR126 = RP.Closing(Instant):Assets.Inventories.ValuationMethod.Code
            */
            assertion = (string.IsNullOrWhiteSpace(report.CTR126) != true && !(IsMatch(report.CTR126, @"^(C|M|R)$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428067",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid Closing stock action code",
                    LongDescription = @"Valid Closing stock action codes are ""C"", ""M"" and ""R""",
                    Location = "/xbrli:xbrl/tns:Assets.Inventories.ValuationMethod.Code",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428067"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR126", Value = GetValueOrEmpty(report.CTR126) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428076
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428076()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428076
            Invalid Percentage of non-member income
    
            Legacy Rule Format:
            IF ([CTR178] <> NULL) AND ([CTR178] <> SET(1-100))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR178 <> NULL) AND ((^CTR178 < 1 OR ^CTR178 > 100))
    
            Data Elements:
    
            ^CTR178 = RP:Income.NonMembers.Percent
            */
            assertion = (report.CTR178 != null && (report.CTR178.GetValueOrDefault() < 1 || report.CTR178.GetValueOrDefault() > 100));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428076",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid Percentage of non-member income",
                    LongDescription = @"Valid Percentage of non-member income is 1 to 100",
                    Location = "/xbrli:xbrl/tns:Income.NonMembers.Percent",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428076"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR178", Value = report.CTR178.GetValueOrDefault().ToString() });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428078
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428078()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428078
            Invalid Hours taken to prepare and complete this form
    
            Legacy Rule Format:
            IF ([CTR217] <> NULL) AND ([CTR217] <> SET (1-999))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR217 <> NULL) AND ((^CTR217 < 1 OR ^CTR217 > 999))
    
            Data Elements:
    
            ^CTR217 = RP:Report.CompletionHours.Number
            */
            assertion = (report.CTR217 != null && (report.CTR217.GetValueOrDefault() < 1 || report.CTR217.GetValueOrDefault() > 999));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428078",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid Hours taken to prepare and complete this form",
                    LongDescription = @"Valid values for Hours taken to prepare and complete this form is 1 to 999",
                    Location = "/xbrli:xbrl/tns:Report.CompletionHours.Number",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428078"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR217", Value = report.CTR217.GetValueOrDefault().ToString() });
                response.Add(processMessage);
            }
        }

        #endregion
#region VR.ATO.CTR.428081
[MethodImpl(MethodImplOptions.AggressiveInlining)]
public virtual void VRATOCTR428081()
{
    ProcessMessageDocument processMessage;
    bool assertion;
    /*  VR.ATO.CTR.428081
    Invalid foreign currency code
    
    Legacy Rule Format:
    IF ([CTR244] <> NULL) AND ([CTR244] <> SET(DOMAIN(Foreign currency codes)))
        RETURN VALIDATION MESSAGE
    ENDIF

    Technical Business Rule Format:
    (^CTR244 <> NULL) AND (NotInDomain(^CTR244, 'Foreign Currency Codes'))
    
    Data Elements:
    
    ^CTR244 = RP:Miscellaneous.FunctionalCurrency.Code
    */
    assertion = (report.CTR244 != null && !(IsMatch(report.CTR244, @"^(AED|AFN|ALL|AMD|AOA|ARS|AWG|AZN|BAM|BBD|BDT|BGN|BHD|BIF|BMD|BND|BOB|BOV|BRL|BSD|BTN|BWP|BYN|BZD|CAD|CDF|CHF|CLP|CNY|COP|CRC|CUP|CVE|CZK|DJF|DKK|DOP|DZD|EGP|ERN|ETB|EUR|FJD|FKP|GBP|GEL|GHS|GIP|GMD|GNF|GTQ|GYD|HKD|HNL|HTG|HUF|IDR|ILS|INR|IQD|IRR|ISK|JMD|JOD|JPY|KES|KGS|KHR|KMF|KPW|KRW|KWD|KYD|KZT|LAK|LBP|LKR|LRD|LSL|LYD|MAD|MDL|MGA|MKD|MMK|MNT|MOP|MRU|MUR|MVR|MWK|MXN|MXV|MYR|MZN|NAD|NGN|NIO|NOK|NPR|NZD|OMR|OTH|PAB|PEN|PGK|PHP|PKR|PLN|PYG|QAR|RON|RSD|RUB|RWF|SAR|SBD|SCR|SDG|SEK|SGD|SHP|SLE|SOS|SRD|SSP|STN|SVC|SYP|SZL|THB|TJS|TMT|TND|TOP|TRY|TTD|TWD|TZS|UAH|UGX|USD|UYI|UYU|UZS|VES|VND|VUV|WST|XAF|XCD|XCG|XOF|XPF|YER|ZAR|ZMW|ZWG)$", RegexOptions.IgnoreCase)));
    if (assertion)
    {
        processMessage = new ProcessMessageDocument()
        {
            Code = "CMN.ATO.CTR.428081",
            Severity = ProcessMessageSeverity.Error,
            Description = @"Invalid foreign currency code",
            LongDescription = @"The currency code is not one of the valid codes that can be used for lodgment. You should be accessing valid codes from a lookup table provided with your software",
            Location = "/xbrli:xbrl/tns:Miscellaneous.FunctionalCurrency.Code",
            Parameters = new ProcessMessageParameters()
            {
                new ProcessMessageParameter()
                {
                    Name = "RuleIdentifier",
                    Value = "VR.ATO.CTR.428081"
                }
            },
        };
        processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR244", Value = GetValueOrEmpty(report.CTR244) });
        response.Add(processMessage);
    }
}

#endregion
#region VR.ATO.CTR.428086
[MethodImpl(MethodImplOptions.AggressiveInlining)]
public virtual void VRATOCTR428086()
{
    ProcessMessageDocument processMessage;
    bool assertion;
    /*  VR.ATO.CTR.428086
    Companies are not required to complete the Life insurance companies and friendly societies section
    
    Legacy Rule Format:
    IF [CTR49] <> SET("63100","62400") AND ([CTR179] <> NULL OR [CTR180] <> NULL OR [CTR181] <> NULL OR [CTR182] <> NULL OR [CTR183] <> NULL)
        RETURN VALIDATION MESSAGE
    ENDIF

    Technical Business Rule Format:
    NotInSet(^CTR49, '"63100","62400"') AND (^CTR179 <> NULL OR ^CTR180 <> NULL OR ^CTR181 <> NULL OR ^CTR182 <> NULL OR ^CTR183 <> NULL)
    
    Data Elements:
    
    ^CTR179 = RP.CMPLY:Income.Insurance.Taxable.Amount
    
    ^CTR49 = RP:OrganisationDetails.OrganisationIndustry2006Extended.Code
    
    ^CTR180 = RP.CMPLY:Income.CapitalGainsNet.Amount
    
    ^CTR181 = RP.ORD:Income.CapitalGainsNet.Amount
    
    ^CTR182 = RP:Income.Insurance.Taxable.Amount
    
    ^CTR183 = RP:Income.Insurance.FeesAndCharges.Amount
    */
    assertion = (!(IsMatch(report.CTR49, @"^(63100|62400)$", RegexOptions.IgnoreCase)) && (report.CTR179 != null || report.CTR180 != null || report.CTR181 != null || report.CTR182 != null || report.CTR183 != null));
    if (assertion)
    {
        processMessage = new ProcessMessageDocument()
        {
            Code = "CMN.ATO.CTR.428086",
            Severity = ProcessMessageSeverity.Error,
            Description = @"Companies are not required to complete the Life insurance companies and friendly societies section",
            LongDescription = @"The Life insurance companies and friendly societies section on the return form (Item 16 Label B) are only to be completed when the industry code (Item 2 Label B) is indicative of a life insurance company or a friendly society",
            Location = "/xbrli:xbrl/tns:Income.Insurance.Taxable.Amount[@contextRef='" + report.RPCMPLYId + "']",
            Parameters = new ProcessMessageParameters()
            {
                new ProcessMessageParameter()
                {
                    Name = "RuleIdentifier",
                    Value = "VR.ATO.CTR.428086"
                }
            },
        };
        processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR49", Value = report.CTR49 });
        processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR179", Value = GetValueOrEmpty(report.CTR179) });
        processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR180", Value = GetValueOrEmpty(report.CTR180) });
        processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR181", Value = GetValueOrEmpty(report.CTR181) });
        processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR182", Value = GetValueOrEmpty(report.CTR182) });
        processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR183", Value = GetValueOrEmpty(report.CTR183) });
        response.Add(processMessage);
    }
}

#endregion
#region VR.ATO.CTR.428091
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428091()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428091
            The Company is required to complete the Life insurance companies and friendly societies section
    
            Legacy Rule Format:
            IF [CTR49] = "63100" AND ([CTR179] = NULL OR [CTR180] = NULL OR [CTR181] = NULL OR [CTR182] = NULL OR [CTR183] = NULL)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR49 = '63100' AND (^CTR179 = NULL OR ^CTR180 = NULL OR ^CTR181 = NULL OR ^CTR182 = NULL OR ^CTR183 = NULL)
    
            Data Elements:
    
            ^CTR179 = RP.CMPLY:Income.Insurance.Taxable.Amount
    
            ^CTR49 = RP:OrganisationDetails.OrganisationIndustry2006Extended.Code
    
            ^CTR180 = RP.CMPLY:Income.CapitalGainsNet.Amount
    
            ^CTR181 = RP.ORD:Income.CapitalGainsNet.Amount
    
            ^CTR182 = RP:Income.Insurance.Taxable.Amount
    
            ^CTR183 = RP:Income.Insurance.FeesAndCharges.Amount
            */
            assertion = (report.CTR49 == @"63100" && (report.CTR179 == null || report.CTR180 == null || report.CTR181 == null || report.CTR182 == null || report.CTR183 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428091",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The Company is required to complete the Life insurance companies and friendly societies section",
                    LongDescription = @"The company is indicating it is a life insurance company (industry code 63100) and the Life insurance companies and friendly societies section is not completed (Item 16 Label B)",
                    Location = "/xbrli:xbrl/tns:Income.Insurance.Taxable.Amount[@contextRef='" + report.RPCMPLYId + "']",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428091"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR49", Value = report.CTR49 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR179", Value = GetValueOrEmpty(report.CTR179) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR180", Value = GetValueOrEmpty(report.CTR180) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR181", Value = GetValueOrEmpty(report.CTR181) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR182", Value = GetValueOrEmpty(report.CTR182) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR183", Value = GetValueOrEmpty(report.CTR183) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428104
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428104()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428104
            Gross payment where ABN not quoted amount required
    
            Legacy Rule Format:
            IF ([CTR208] > 0) AND ([CTR58] = NULL)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR208 > 0) AND (^CTR58 = NULL)
    
            Data Elements:
    
            ^CTR208 = RP:ITPAYGWithholdingDetails:IncomeTax.PayAsYouGoWithholding.TaxWithheld.Amount WHERE [IncomeTax.PayAsYouGoWithholding.PaymentType.Code='DNOABN']
    
            ^CTR58 = RP:Remuneration.ABNNotQuotedPaymentGross.Amount
            */
            assertion = (report.CTR208.GetValueOrDefault() > 0 && report.CTR58 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428381",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Gross payment where ABN not quoted amount required",
                    LongDescription = @"In the Calculation statement, where ""Credit for tax withheld where ABN not quoted is greater than zero, there must be an amount at 'Gross payments where ABN not quoted' in the calculation of total profit or loss",
                    Location = "/xbrli:xbrl/tns:ITPAYGWithholdingDetails" + OccurrenceIndex(report.ItpaygWithholdingDetailsCollection_DNOABN_RPOccurrenceIndex) + "/tns:IncomeTax.PayAsYouGoWithholding.TaxWithheld.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428104"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR208", Value = GetValueOrEmpty(report.CTR208) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR58", Value = GetValueOrEmpty(report.CTR58) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428105
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428105()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428105
            S102AAM interest amount incorrect in Calculation statement Label H
    
            Legacy Rule Format:
            IF ([CTR204] <> NULL) AND ([CTR204] > [CTR199])
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR204 <> NULL) AND (^CTR204 > ^CTR199)
    
            Data Elements:
    
            ^CTR204 = RP:InternationalDealings.AAMInterestSection102.Amount
    
            ^CTR199 = RP:IncomeTax.TaxableGross.Amount
            */
            assertion = (report.CTR204 != null && report.CTR204.GetValueOrDefault() > report.CTR199.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428382",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"S102AAM interest amount incorrect in Calculation statement Label H",
                    LongDescription = @"Calculation statement 'S102AAM Interest charge amount cannot be greater than calculation statement Gross tax amount",
                    Location = "/xbrli:xbrl/tns:InternationalDealings.AAMInterestSection102.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428105"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR204", Value = GetValueOrEmpty(report.CTR204) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR199", Value = GetValueOrEmpty(report.CTR199) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428110
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428110()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428110
            Response to international transaction question incorrect
    
            Legacy Rule Format:
            IF ([CTR194] = TRUE) AND 
            ([CTR193] <> TRUE)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR194 = TRUE) AND (^CTR193 <> TRUE)
    
            Data Elements:
    
            ^CTR194 = RP:InternationalDealings.RelatedPartiesTransactionsExcessAggregateValue.Indicator
    
            ^CTR193 = RP:InternationalDealings.RelatedPartiesTransactions.Indicator
            */
            assertion = (report.CTR194 == true && report.CTR193 != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428384",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Response to international transaction question incorrect",
                    LongDescription = @"There is a 'yes' response to the question concerning whether the value of property is greater than $2 million but there is no 'yes' response to question concerning whether the entity had international dealings, including transfer of tangible or intangible property. As these items relate to international related party dealings or transfer pricing, the answers can not conflict with one another.",
                    Location = "/xbrli:xbrl/tns:InternationalDealings.RelatedPartiesTransactionsExcessAggregateValue.Indicator",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428110"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR194", Value = GetValueOrEmpty(report.CTR194) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR193", Value = GetValueOrEmpty(report.CTR193) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428111
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428111()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428111
            Response to international aggregate amount transactions question incorrect
    
            Legacy Rule Format:
            IF ([CTR193] = FALSE) AND ([CTR194] <> FALSE)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR193 = FALSE) AND (^CTR194 <> FALSE)
    
            Data Elements:
    
            ^CTR194 = RP:InternationalDealings.RelatedPartiesTransactionsExcessAggregateValue.Indicator
    
            ^CTR193 = RP:InternationalDealings.RelatedPartiesTransactions.Indicator
            */
            assertion = (report.CTR193 == false && report.CTR194 != false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428385",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Response to international aggregate amount transactions question incorrect",
                    LongDescription = @"There is 'no' response to question concerning whether the entity had international dealings, including transfer of tangible or intangible property and there is a 'no' response to the question concerning whether the value of property is greater than $2 million. As these items relate to international related party dealings or transfer pricing, the answers can not conflict with one another",
                    Location = "/xbrli:xbrl/tns:InternationalDealings.RelatedPartiesTransactionsExcessAggregateValue.Indicator",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428111"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR193", Value = GetValueOrEmpty(report.CTR193) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR194", Value = GetValueOrEmpty(report.CTR194) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428115
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428115()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428115
            If there is amount greater than zero for 'Attributed foreign income' - Listed country, Unlisted country, or Transferor trust - there must be a 'yes' (true) response to 'Did you have an overseas branch or a direct or indirect interest in a foreign trust, foreign company, controlled foreign entity or transferor trust?
    
            Legacy Rule Format:
            IF [CTR195] = FALSE AND (([CTR156] > 0) OR ([CTR158] > 0) OR ([CTR159] > 0)) 
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR195 = FALSE AND ((^CTR156 > 0) OR (^CTR158 > 0) OR (^CTR159 > 0))
    
            Data Elements:
    
            ^CTR195 = RP:InternationalDealings.DirectOrIndirectOverseasInterest.Indicator
    
            ^CTR156 = RP.CFCLC:Income.InternationalDealings.Attributed.Amount
    
            ^CTR158 = RP.CFCUC:Income.InternationalDealings.Attributed.Amount
    
            ^CTR159 = RP:InternationalDealings.TransferorTrustAttributedForeignIncome.Amount
            */
            assertion = (report.CTR195 == false && (report.CTR156.GetValueOrDefault() > 0 || report.CTR158.GetValueOrDefault() > 0 || report.CTR159.GetValueOrDefault() > 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.430221",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Answer to Overseas interests question must be yes (true) when Attributed Foreign income is present",
                    LongDescription = @"If there is amount greater than zero for 'Attributed foreign income' - Listed country, Unlisted country, or Transferor trust - there must be a 'yes' (true) response to 'Did you have an overseas branch or a direct or indirect interest in a foreign trust, foreign company, controlled foreign entity or transferor trust?'",
                    Location = "/xbrli:xbrl/tns:InternationalDealings.DirectOrIndirectOverseasInterest.Indicator",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428115"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR195", Value = GetValueOrEmpty(report.CTR195) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR156", Value = GetValueOrEmpty(report.CTR156) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR158", Value = GetValueOrEmpty(report.CTR158) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR159", Value = GetValueOrEmpty(report.CTR159) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428136
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428136()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428136
            Check losses recouped or losses transferred in as these cannot create a taxable loss
    
            Legacy Rule Format:
            IF (([CTR117] > 0) OR ([CTR118] > 0)) AND ([CTR120] < 0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ((^CTR117 > 0) OR (^CTR118 > 0)) AND (^CTR120 < 0)
    
            Data Elements:
    
            ^CTR120 = RP:Income.Taxable.Amount
    
            ^CTR117 = RP:IncomeTax.Deduction.TaxLossesDeducted.Amount
    
            ^CTR118 = RP:IncomeTax.TaxLossesTransferredIn.Amount
            */
            assertion = ((report.CTR117.GetValueOrDefault() > 0 || report.CTR118.GetValueOrDefault() > 0) && report.CTR120.GetValueOrDefault() < 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428136",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Check losses recouped or losses transferred in as these cannot create a taxable loss",
                    LongDescription = @"At Item 7 losses recouped (Label R) or losses transferred in (from or to: a foreign bank branch or a PE of a foreign financial entity) (Label S) cannot add to or create a taxable loss (Label T). Adjust losses recouped or losses transferred in until the taxable income/loss figure is zero",
                    Location = "/xbrli:xbrl/tns:Income.Taxable.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428136"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR117", Value = GetValueOrEmpty(report.CTR117) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR118", Value = GetValueOrEmpty(report.CTR118) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR120", Value = GetValueOrEmpty(report.CTR120) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428140
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428140()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428140
            Check gross tax and company type
    
            Legacy Rule Format:
            IF ([CTR199] < (([CTR120] * 0.15) - 1)) AND (CTR:RP:pyde.xx.xx:OrganisationDetails.OrganisationType.Code = "163")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR199 < ((^CTR120 * 0.15) - 1)) AND (^CTR52 = '163')
    
            Data Elements:
    
            ^CTR199 = RP:IncomeTax.TaxableGross.Amount
    
            ^CTR120 = RP:Income.Taxable.Amount
    
            ^CTR52 = RP:OrganisationDetails.OrganisationType.Code
            */
            assertion = (report.CTR199.GetValueOrDefault() < report.CTR120.GetValueOrDefault() * 0.15M - 1 && report.CTR52 == @"163");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428390",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Check gross tax and company type",
                    LongDescription = @"The gross tax is less than 15% of taxable income and the company type is a pooled development fund. Check calculation statement gross tax and company type",
                    Location = "/xbrli:xbrl/tns:IncomeTax.TaxableGross.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428140"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR199", Value = GetValueOrEmpty(report.CTR199) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR120", Value = GetValueOrEmpty(report.CTR120) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR52", Value = report.CTR52 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428149
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428149()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428149
            Currency transalation rate and currency code must both be either present or not present
    
            Legacy Rule Format:
            IF (([CTR121] <> NULL) AND ([CTR244] = NULLORBLANK)) OR 
            (([CTR244] <> NULLORBLANK) AND ([CTR121] = NULL))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ((^CTR121 <> NULL) AND (^CTR244 = BLANK)) OR ((^CTR244 <> BLANK) AND (^CTR121 = NULL))
    
            Data Elements:
    
            ^CTR121 = RP.Closing(Instant):Miscellaneous.CurrencyTranslation.Rate
    
            ^CTR244 = RP:Miscellaneous.FunctionalCurrency.Code
            */
            assertion = (report.CTR121 != null && string.IsNullOrWhiteSpace(report.CTR244) == true || string.IsNullOrWhiteSpace(report.CTR244) != true && report.CTR121 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428149",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Currency translation rate and currency code must both be either present or not present",
                    Location = "/xbrli:xbrl/tns:Miscellaneous.CurrencyTranslation.Rate",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428149"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR121", Value = GetValueOrEmpty(report.CTR121) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR244", Value = GetValueOrEmpty(report.CTR244) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428150
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428150()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428150
            Check closing stock amount, where closing stock action code is present
    
            Legacy Rule Format:
            IF ([CTR126] <> NULLORBLANK) AND ([CTR125] = NULL OR [CTR125] = 0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR126 <> BLANK) AND (^CTR125 = NULL OR ^CTR125 = 0)
    
            Data Elements:
    
            ^CTR126 = RP.Closing(Instant):Assets.Inventories.ValuationMethod.Code
    
            ^CTR125 = RP.Closing(Instant):Assets.Inventories.Total.Amount
            */
            assertion = (string.IsNullOrWhiteSpace(report.CTR126) != true && (report.CTR125 == null || report.CTR125 == 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428150",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Check closing stock amount, where closing stock action code is present",
                    LongDescription = @"Under business income and expenses, the closing stock action code is present; however the closing stock amount is missing or equal to 0. The amount must be present as it is the total of the value of each trading stock calculated for tax purposes under s70-45 of the ITAA 1997.",
                    Location = "/xbrli:xbrl/tns:Assets.Inventories.ValuationMethod.Code",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428150"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR126", Value = GetValueOrEmpty(report.CTR126) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR125", Value = GetValueOrEmpty(report.CTR125) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428151
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428151()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428151
            Closing stock action code must be present, where closing stock amount is present
    
            Legacy Rule Format:
            IF ([CTR126] = NULLORBLANK) AND ([CTR125] >0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR126 = BLANK) AND (^CTR125 >0)
    
            Data Elements:
    
            ^CTR126 = RP.Closing(Instant):Assets.Inventories.ValuationMethod.Code
    
            ^CTR125 = RP.Closing(Instant):Assets.Inventories.Total.Amount
            */
            assertion = (string.IsNullOrWhiteSpace(report.CTR126) == true && report.CTR125.GetValueOrDefault() > 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428151",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Closing stock action code must be present, where closing stock amount is present",
                    LongDescription = @"Under business income and expenses, the closing stock amount is present (Item 8 Label B); however the closing stock action code is missing. The code must be present to indicate the method used to value closing stock",
                    Location = "/xbrli:xbrl/tns:Assets.Inventories.ValuationMethod.Code",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428151"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR126", Value = GetValueOrEmpty(report.CTR126) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR125", Value = GetValueOrEmpty(report.CTR125) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428152
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428152()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428152
            Trade debtors cannot exceed all current assets in Financial and other information
    
            Legacy Rule Format:
            IF ([CTR129] < [CTR128])
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR129 < ^CTR128
    
            Data Elements:
    
            ^CTR128 = RP.Closing(Instant):Assets.TradeandReceivablesOther.Amount
    
            ^CTR129 = RP.Closing(Instant):Assets.CurrentTotal.Amount
            */
            assertion = (report.CTR129.GetValueOrDefault() < report.CTR128.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428152",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Trade debtors cannot exceed all current assets in Financial and other information",
                    LongDescription = @"In Financial and other information, trade debtors is greater than all current assets (item 8 Labels C and D)",
                    Location = "/xbrli:xbrl/tns:Assets.TradeandReceivablesOther.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428152"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR129", Value = GetValueOrEmpty(report.CTR129) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR128", Value = GetValueOrEmpty(report.CTR128) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428153
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428153()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428153
            All current assets cannot exceed total assets in Financial and other information
    
            Legacy Rule Format:
            IF  ([CTR130]<[CTR129])
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR130<^CTR129
    
            Data Elements:
    
            ^CTR129 = RP.Closing(Instant):Assets.CurrentTotal.Amount
    
            ^CTR130 = RP.Closing(Instant):Assets.Total.Amount
            */
            assertion = (report.CTR130.GetValueOrDefault() < report.CTR129.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428153",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"All current assets cannot exceed total assets in Financial and other information",
                    LongDescription = @"In Financial and other information, all current assets is greater than total assets Item 8 Labels D and E)",
                    Location = "/xbrli:xbrl/tns:Assets.CurrentTotal.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428153"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR130", Value = GetValueOrEmpty(report.CTR130) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR129", Value = GetValueOrEmpty(report.CTR129) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428155
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428155()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428155
            Trade creditors cannot exceed all current liabilities in Financial and other information
    
            Legacy Rule Format:
            IF ([CTR131] <> NULL) AND ([CTR132] < [CTR131])
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR131 <> NULL) AND (^CTR132 < ^CTR131)
    
            Data Elements:
    
            ^CTR131 = RP.Closing(Instant):Liabilities.NonCurrentTradeAndOtherPayables.Amount
    
            ^CTR132 = RP.Closing(Instant):Liabilities.CurrentTotal.Amount
            */
            assertion = (report.CTR131 != null && report.CTR132.GetValueOrDefault() < report.CTR131.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428155",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Trade creditors cannot exceed all current liabilities in Financial and other information",
                    LongDescription = @"In Financial and other information, trade creditors (Item 8 Label F) is greater than all current liabilities (Item 8 Label G)",
                    Location = "/xbrli:xbrl/tns:Liabilities.NonCurrentTradeAndOtherPayables.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428155"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR131", Value = GetValueOrEmpty(report.CTR131) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR132", Value = GetValueOrEmpty(report.CTR132) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428156
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428156()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428156
            All current liabilities cannot exceed total liabilities in Financial and other information
    
            Legacy Rule Format:
            IF ([CTR132] <> NULL) AND ([CTR133] < [CTR132])
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR132 <> NULL) AND (^CTR133 < ^CTR132)
    
            Data Elements:
    
            ^CTR132 = RP.Closing(Instant):Liabilities.CurrentTotal.Amount
    
            ^CTR133 = RP.Closing(Instant):Liabilities.Total.Amount
            */
            assertion = (report.CTR132 != null && report.CTR133.GetValueOrDefault() < report.CTR132.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428156",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"All current liabilities cannot exceed total liabilities in Financial and other information",
                    LongDescription = @"In Financial and other information, All current liabilities (item 8 Label G) is greater than total liabilities (Item 8 Label H)",
                    Location = "/xbrli:xbrl/tns:Liabilities.CurrentTotal.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428156"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR132", Value = GetValueOrEmpty(report.CTR132) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR133", Value = GetValueOrEmpty(report.CTR133) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428164
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428164()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428164
            Check company type, where Loans to shareholders completed and entity is not a private company or closely held corporate limited partnerships
    
            Legacy Rule Format:
            IF [CTR145] > 0 AND ([CTR52] <> SET("027","120"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR145 > 0 AND (NotInSet(^CTR52, '"027","120"'))
    
            Data Elements:
    
            ^CTR145 = RP:Assets.LoansAndReceivables.RelatedParties.Amount
    
            ^CTR52 = RP:OrganisationDetails.OrganisationType.Code
            */
            assertion = (report.CTR145.GetValueOrDefault() > 0 && !(IsMatch(report.CTR52, @"^(027|120)$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.430231",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Check company type, where Loans to shareholders completed and entity is not a private company or closely held corporate limited partnerships",
                    LongDescription = @"At Financial and other information, Loans to shareholders can only be completed by a private company or closely held corporate limited partnerships",
                    Location = "/xbrli:xbrl/tns:Assets.LoansAndReceivables.RelatedParties.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428164"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR145", Value = GetValueOrEmpty(report.CTR145) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR52", Value = report.CTR52 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428165
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428165()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428165
            Loan to shareholders action code required, where amount field has been populated
    
            Legacy Rule Format:
            IF ([CTR146] = NULLORBLANK) AND ([CTR145] <> NULL)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR146 = BLANK) AND (^CTR145 <> NULL)
    
            Data Elements:
    
            ^CTR146 = RP:Assets.LoansAndReceivablesAction.Code
    
            ^CTR145 = RP:Assets.LoansAndReceivables.RelatedParties.Amount
            */
            assertion = (string.IsNullOrWhiteSpace(report.CTR146) == true && report.CTR145 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428165",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Loan to shareholders action code required, where amount field has been populated",
                    LongDescription = @"At Financial and other information, when there is an amount regarding loans to shareholders and their associates, the loan type action code must also be indicated. (Item 8 Label N)",
                    Location = "/xbrli:xbrl/tns:Assets.LoansAndReceivablesAction.Code",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428165"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR146", Value = GetValueOrEmpty(report.CTR146) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR145", Value = GetValueOrEmpty(report.CTR145) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428166
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428166()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428166
            Loans to shareholders amount required, where the loan type action code field has been populated
    
            Legacy Rule Format:
            IF ([CTR146] <> NULLORBLANK) AND ([CTR145] = NULL)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR146 <> BLANK) AND (^CTR145 = NULL)
    
            Data Elements:
    
            ^CTR146 = RP:Assets.LoansAndReceivablesAction.Code
    
            ^CTR145 = RP:Assets.LoansAndReceivables.RelatedParties.Amount
            */
            assertion = (string.IsNullOrWhiteSpace(report.CTR146) != true && report.CTR145 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428166",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Loans to shareholders amount required, where the loan type action code field has been populated",
                    LongDescription = @"At Financial and other information, when there is a loan type action code, there must be an amount present at loans to shareholders and their associates. (Item 8 Label N)",
                    Location = "/xbrli:xbrl/tns:Assets.LoansAndReceivablesAction.Code",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428166"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR146", Value = GetValueOrEmpty(report.CTR146) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR145", Value = GetValueOrEmpty(report.CTR145) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428167
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428167()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428167
            Total salary and wage expenses action code required, where amount field has been populated
    
            Legacy Rule Format:
            IF ([CTR152] = NULLORBLANK) AND ([CTR151] > 0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR152 = BLANK) AND (^CTR151 > 0)
    
            Data Elements:
    
            ^CTR152 = RP:Remuneration.WagesAndSalariesAction.Code
    
            ^CTR151 = RP:Remuneration.WagesAndSalaries.Total.Amount
            */
            assertion = (string.IsNullOrWhiteSpace(report.CTR152) == true && report.CTR151.GetValueOrDefault() > 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428167",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total salary and wage expenses action code required, where amount field has been populated",
                    LongDescription = @"At Financial and other information, when there is an amount present at Total salary and wage expenses, Item 8 Label D, the Total salary and wage expenses action code must also be indicated. Please check and input the relevant salary and wage expense action code",
                    Location = "/xbrli:xbrl/tns:Remuneration.WagesAndSalariesAction.Code",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428167"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR152", Value = GetValueOrEmpty(report.CTR152) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR151", Value = GetValueOrEmpty(report.CTR151) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428168
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428168()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428168
            Check total salary and wage expenses amount, where a valid action code has been input
    
            Legacy Rule Format:
            IF ([CTR152] <> NULLORBLANK) AND ([CTR151] = NULL)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR152 <> BLANK) AND (^CTR151 = NULL)
    
            Data Elements:
    
            ^CTR152 = RP:Remuneration.WagesAndSalariesAction.Code
    
            ^CTR151 = RP:Remuneration.WagesAndSalaries.Total.Amount
            */
            assertion = (string.IsNullOrWhiteSpace(report.CTR152) != true && report.CTR151 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428168",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Check total salary and wage expenses amount, where a valid action code has been input",
                    LongDescription = @"At Financial and other information, if there is a valid Total salary and wage expenses action code present, then an amount greater than zero must be present at Total salary and wage expenses, Item 8 Label D",
                    Location = "/xbrli:xbrl/tns:Remuneration.WagesAndSalariesAction.Code",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428168"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR152", Value = GetValueOrEmpty(report.CTR152) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR151", Value = GetValueOrEmpty(report.CTR151) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428169
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428169()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428169
            The amount of Foreign income tax offsets in the Calculation Statement must be less than or equal to the sum of Net foreign income plus all Attributed foreign income amounts
    
            Legacy Rule Format:
            IF [CTR201] > 1000 AND [CTR201] > ([CTR154] + [CTR156] + [CTR158] + [CTR159])
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR201 > 1000 AND ^CTR201 > (^CTR154 + ^CTR156 + ^CTR158 + ^CTR159)
    
            Data Elements:
    
            ^CTR201 = RP:Income.InternationalDealings.TaxOffset.Amount
    
            ^CTR154 = RP:Income.InternationalDealings.Net.Amount
    
            ^CTR156 = RP.CFCLC:Income.InternationalDealings.Attributed.Amount
    
            ^CTR158 = RP.CFCUC:Income.InternationalDealings.Attributed.Amount
    
            ^CTR159 = RP:InternationalDealings.TransferorTrustAttributedForeignIncome.Amount
            */
            assertion = (report.CTR201.GetValueOrDefault() > 1000 && report.CTR201.GetValueOrDefault() > report.CTR154.GetValueOrDefault() + report.CTR156.GetValueOrDefault() + report.CTR158.GetValueOrDefault() + report.CTR159.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428314",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Foreign income tax offsets must not exceed Net foreign and Attributed foreign income",
                    LongDescription = @"The amount of Foreign income tax offsets claimed must be less than or equal to the sum of Net foreign income and all Attributed foreign income.",
                    Location = "/xbrli:xbrl/tns:Income.InternationalDealings.TaxOffset.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428169"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR201", Value = GetValueOrEmpty(report.CTR201) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR154", Value = GetValueOrEmpty(report.CTR154) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR156", Value = GetValueOrEmpty(report.CTR156) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR158", Value = GetValueOrEmpty(report.CTR158) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR159", Value = GetValueOrEmpty(report.CTR159) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428170
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428170()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428170
            If there is an amount greater than zero at Foreign income tax offsets in the Calculation Statement, there must be an amount Net foreign income or at an Attributed foreign income item
    
            Legacy Rule Format:
            IF [CTR201] > 0 AND ([CTR154] = NULL) AND ([CTR156] = NULL) AND ([CTR158] = NULL) AND ([CTR159] = NULL)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR201 > 0 AND (^CTR154 = NULL) AND (^CTR156 = NULL) AND (^CTR158 = NULL) AND (^CTR159 = NULL)
    
            Data Elements:
    
            ^CTR154 = RP:Income.InternationalDealings.Net.Amount
    
            ^CTR201 = RP:Income.InternationalDealings.TaxOffset.Amount
    
            ^CTR156 = RP.CFCLC:Income.InternationalDealings.Attributed.Amount
    
            ^CTR158 = RP.CFCUC:Income.InternationalDealings.Attributed.Amount
    
            ^CTR159 = RP:InternationalDealings.TransferorTrustAttributedForeignIncome.Amount
            */
            assertion = (report.CTR201.GetValueOrDefault() > 0 && report.CTR154 == null && report.CTR156 == null && report.CTR158 == null && report.CTR159 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428313",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Net foreign income and/or Attributed foreign income amounts must be present",
                    LongDescription = @"There is an amount at the Foreign income tax offsets and no amounts at Net foreign income or Attributed foreign income are present.",
                    Location = "/xbrli:xbrl/tns:Income.InternationalDealings.Net.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428170"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR201", Value = GetValueOrEmpty(report.CTR201) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR154", Value = GetValueOrEmpty(report.CTR154) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR156", Value = GetValueOrEmpty(report.CTR156) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR158", Value = GetValueOrEmpty(report.CTR158) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR159", Value = GetValueOrEmpty(report.CTR159) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428186
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428186()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428186
            In Calculation statement, PAYG instalments raised amount should be greater than or equal to zero
    
            Legacy Rule Format:
            IF (([CTR53] = SET("1","2")) AND ([CTR205] = NULL))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ((InSet(^CTR53, '"1","2"')) AND (^CTR205 = NULL))
    
            Data Elements:
    
            ^CTR205 = RP:IncomeTax.LiabilityInstalmentsTotal.Amount
    
            ^CTR53 = RP:OrganisationDetails.TaxConsolidationStatus.Code
            */
            assertion = (IsMatch(report.CTR53, @"^(1|2)$") && report.CTR205 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428396",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"In Calculation statement, PAYG instalments raised amount should be greater than or equal to zero",
                    LongDescription = @"The Status of the Head company or Subsidiary member has been selected, however, there is no amount for PAYG instalments raised in the calculation statement",
                    Location = "/xbrli:xbrl/tns:IncomeTax.LiabilityInstalmentsTotal.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428186"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR53", Value = GetValueOrEmpty(report.CTR53) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR205", Value = GetValueOrEmpty(report.CTR205) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428197
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428197()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428197
            Mandatory field not supplied
    
            Legacy Rule Format:
            IF [CTR42] = NULL
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR42 = NULL
    
            Data Elements:
    
            ^CTR42 = RP:Lodgment.FinalReturn.Indicator
            */
            assertion = (report.CTR42 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Mandatory field not supplied",
                    Location = "/xbrli:xbrl/tns:Lodgment.FinalReturn.Indicator",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428197"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR42", Value = GetValueOrEmpty(report.CTR42) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428198
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428198()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428198
            Mandatory field not supplied
    
            Legacy Rule Format:
            IF [CTR48] = NULLORBLANK
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR48 = BLANK
    
            Data Elements:
    
            ^CTR48 = RP:OrganisationDetails.MainIncomeActivity.Description
            */
            assertion = (string.IsNullOrWhiteSpace(report.CTR48) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Mandatory field not supplied",
                    Location = "/xbrli:xbrl/tns:OrganisationDetails.MainIncomeActivity.Description",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428198"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR48", Value = report.CTR48 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428199
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428199()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428199
            Mandatory field not supplied
    
            Legacy Rule Format:
            IF [CTR49] = NULLORBLANK
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR49 = BLANK
    
            Data Elements:
    
            ^CTR49 = RP:OrganisationDetails.OrganisationIndustry2006Extended.Code
            */
            assertion = (string.IsNullOrWhiteSpace(report.CTR49) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Mandatory field not supplied",
                    Location = "/xbrli:xbrl/tns:OrganisationDetails.OrganisationIndustry2006Extended.Code",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428199"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR49", Value = report.CTR49 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428200
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428200()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428200
            Mandatory field not supplied
    
            Legacy Rule Format:
            IF [CTR52] = NULL
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR52 = NULL
    
            Data Elements:
    
            ^CTR52 = RP:OrganisationDetails.OrganisationType.Code
            */
            assertion = (report.CTR52 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Mandatory field not supplied",
                    Location = "/xbrli:xbrl/tns:OrganisationDetails.OrganisationType.Code",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428200"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR52", Value = report.CTR52 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428201
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428201()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428201
            Mandatory field not supplied
    
            Legacy Rule Format:
            IF [CTR89] = NULL
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR89 = NULL
    
            Data Elements:
    
            ^CTR89 = RP:CapitalGainsTax.Event.Indicator
            */
            assertion = (report.CTR89 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Mandatory field not supplied",
                    Location = "/xbrli:xbrl/tns:CapitalGainsTax.Event.Indicator",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428201"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR89", Value = GetValueOrEmpty(report.CTR89) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428202
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428202()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428202
            Mandatory field not supplied
    
            Legacy Rule Format:
            IF [CTR120] = NULL
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR120 = NULL
    
            Data Elements:
    
            ^CTR120 = RP:Income.Taxable.Amount
            */
            assertion = (report.CTR120 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Mandatory field not supplied",
                    Location = "/xbrli:xbrl/tns:Income.Taxable.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428202"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR120", Value = GetValueOrEmpty(report.CTR120) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428204
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428204()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428204
            Mandatory field not supplied
    
            Legacy Rule Format:
            IF [CTR193] = NULL
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR193 = NULL
    
            Data Elements:
    
            ^CTR193 = RP:InternationalDealings.RelatedPartiesTransactions.Indicator
            */
            assertion = (report.CTR193 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Mandatory field not supplied",
                    Location = "/xbrli:xbrl/tns:InternationalDealings.RelatedPartiesTransactions.Indicator",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428204"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR193", Value = GetValueOrEmpty(report.CTR193) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428205
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428205()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428205
            Mandatory field not supplied
    
            Legacy Rule Format:
            IF [CTR194] = NULL
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR194 = NULL
    
            Data Elements:
    
            ^CTR194 = RP:InternationalDealings.RelatedPartiesTransactionsExcessAggregateValue.Indicator
            */
            assertion = (report.CTR194 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Mandatory field not supplied",
                    Location = "/xbrli:xbrl/tns:InternationalDealings.RelatedPartiesTransactionsExcessAggregateValue.Indicator",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428205"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR194", Value = GetValueOrEmpty(report.CTR194) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428206
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428206()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428206
            Mandatory field not supplied
    
            Legacy Rule Format:
            IF [CTR195] = NULL
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR195 = NULL
    
            Data Elements:
    
            ^CTR195 = RP:InternationalDealings.DirectOrIndirectOverseasInterest.Indicator
            */
            assertion = (report.CTR195 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Mandatory field not supplied",
                    Location = "/xbrli:xbrl/tns:InternationalDealings.DirectOrIndirectOverseasInterest.Indicator",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428206"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR195", Value = GetValueOrEmpty(report.CTR195) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428207
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428207()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428207
            Mandatory field not supplied
    
            Legacy Rule Format:
            IF [CTR196] = NULL
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR196 = NULL
    
            Data Elements:
    
            ^CTR196 = RP:Liabilities.ThinCapitalisation.ProvisionsApplied.Indicator
            */
            assertion = (report.CTR196 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Mandatory field not supplied",
                    Location = "/xbrli:xbrl/tns:Liabilities.ThinCapitalisation.ProvisionsApplied.Indicator",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428207"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR196", Value = GetValueOrEmpty(report.CTR196) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428209
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428209()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428209
            Mandatory field not supplied
    
            Legacy Rule Format:
            IF [CTR198] = NULL
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR198 = NULL
    
            Data Elements:
    
            ^CTR198 = RP:InternationalDealings.TransactionsWithTaxHavenCountries.Indicator
            */
            assertion = (report.CTR198 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Mandatory field not supplied",
                    Location = "/xbrli:xbrl/tns:InternationalDealings.TransactionsWithTaxHavenCountries.Indicator",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428209"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR198", Value = GetValueOrEmpty(report.CTR198) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428210
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428210()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428210
            Mandatory field not supplied
    
            Legacy Rule Format:
            IF [CTR199] = NULL
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR199 = NULL
    
            Data Elements:
    
            ^CTR199 = RP:IncomeTax.TaxableGross.Amount
            */
            assertion = (report.CTR199 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Mandatory field not supplied",
                    Location = "/xbrli:xbrl/tns:IncomeTax.TaxableGross.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428210"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR199", Value = GetValueOrEmpty(report.CTR199) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428211
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428211()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428211
            Mandatory field not supplied
    
            Legacy Rule Format:
            IF [CTR203] = NULL
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR203 = NULL
    
            Data Elements:
    
            ^CTR203 = RP:IncomeTax.Payable.Amount
            */
            assertion = (report.CTR203 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Mandatory field not supplied",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Payable.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428211"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR203", Value = GetValueOrEmpty(report.CTR203) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428216
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428216()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428216
            Mandatory field not supplied
    
            Legacy Rule Format:
            IF [CTR9] = NULLORBLANK
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR9 = BLANK
    
            Data Elements:
    
            ^CTR9 = RP:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.CTR9) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Mandatory field not supplied",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428216"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR9", Value = report.CTR9 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428220
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428220()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428220
            Mandatory field not supplied
    
            Legacy Rule Format:
            IF (INT <> NULL)  AND [CTR216] = NULLORBLANK
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^INT <> NULL)  AND ^CTR216 = BLANK
    
            Data Elements:
    
            ^CTR216 = INT:ElectronicContactTelephone:ElectronicContact.Telephone.Minimal.Number WHERE [ElectronicContact.Telephone.Usage.Code='03']
    
            ^INT = INT
            */
            assertion = (report.INTExists != false && string.IsNullOrWhiteSpace(report.CTR216) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Mandatory field not supplied",
                    Location = "/xbrli:xbrl/tns:ElectronicContactTelephone" + OccurrenceIndex(report.ElectronicContactTelephoneCollection_03_INTOccurrenceIndex) + "/tns:ElectronicContact.Telephone.Minimal.Number",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428220"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR216", Value = report.CTR216 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428221
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428221()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428221
            Mandatory field not supplied
    
            Legacy Rule Format:
            IF [CTR33] = NULLORBLANK
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^AddressDetails <> NULL) AND (^CTR33 = BLANK)
    
            Data Elements:
    
            ^CTR33 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
    
            ^AddressDetails = RP:AddressDetails
            */
            assertion = (report.AddressDetailsCollection_BUS_C_RPExists != false && string.IsNullOrWhiteSpace(report.CTR33) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Mandatory field not supplied",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428221"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR33", Value = report.CTR33 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428225
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428225()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428225
            Mandatory field not supplied
    
            Legacy Rule Format:
            IF (CONTEXT(INT) <> NULL) AND [CTR214] = NULLORBLANK
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND ^CTR214 = BLANK
    
            Data Elements:
    
            ^CTR214 = INT:PersonUnstructuredName:PersonUnstructuredName.FullName.Text WHERE [PersonUnstructuredName.Usage.Code='Contact']
    
            ^Context = INT
            */
            assertion = (report.INTExists != false && string.IsNullOrWhiteSpace(report.CTR214) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Mandatory field not supplied",
                    Location = "/xbrli:xbrl/tns:PersonUnstructuredName" + OccurrenceIndex(report.PersonUnstructuredNameCollection_Contact_INTOccurrenceIndex) + "/tns:PersonUnstructuredName.FullName.Text",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428225"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR214", Value = report.CTR214 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428226
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428226()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428226
            Mandatory field not supplied
    
            Legacy Rule Format:
            IF [CTR219] = NULLORBLANK
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR219 = BLANK
    
            Data Elements:
    
            ^CTR219 = RP:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='DeclarationSignatory']
            */
            assertion = (string.IsNullOrWhiteSpace(report.CTR219) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Mandatory field not supplied",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_DeclarationSignatory_RPOccurrenceIndex) + "/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428226"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR219", Value = report.CTR219 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428227
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428227()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428227
            Mandatory field not supplied
    
            Legacy Rule Format:
            IF CTR:RP:pyde.xx.xx:ElectronicContact.Telephone.Area.Code = NULLORBLANK
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR223 = BLANK
    
            Data Elements:
    
            ^CTR223 = RP:ElectronicContactTelephone:ElectronicContact.Telephone.Area.Code WHERE [ElectronicContact.Telephone.Usage.Code='03']
            */
            assertion = (string.IsNullOrWhiteSpace(report.CTR223) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Mandatory field not supplied",
                    Location = "/xbrli:xbrl/tns:ElectronicContactTelephone" + OccurrenceIndex(report.ElectronicContactTelephoneCollection_03_RPOccurrenceIndex) + "/tns:ElectronicContact.Telephone.Area.Code",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428227"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR223", Value = GetValueOrEmpty(report.CTR223) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428228
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428228()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428228
            Mandatory field not supplied
    
            Legacy Rule Format:
            IF CTR:RP:pyde.xx.xx:ElectronicContact.Telephone.Minimal.Number = NULLORBLANK
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR224 = BLANK
    
            Data Elements:
    
            ^CTR224 = RP:ElectronicContactTelephone:ElectronicContact.Telephone.Minimal.Number WHERE [ElectronicContact.Telephone.Usage.Code='03']
            */
            assertion = (string.IsNullOrWhiteSpace(report.CTR224) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Mandatory field not supplied",
                    Location = "/xbrli:xbrl/tns:ElectronicContactTelephone" + OccurrenceIndex(report.ElectronicContactTelephoneCollection_03_RPOccurrenceIndex) + "/tns:ElectronicContact.Telephone.Minimal.Number",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428228"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR224", Value = report.CTR224 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428235
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428235()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428235
            Mandatory field not supplied
    
            Legacy Rule Format:
            IF(INT <> NULLORBLANK) AND ([CTR215] = NULL)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^INT <> BLANK) AND (^CTR215 = NULL)
    
            Data Elements:
    
            ^CTR215 = INT:ElectronicContactTelephone:ElectronicContact.Telephone.Area.Code WHERE [ElectronicContact.Telephone.Usage.Code='03']
    
            ^INT = INT
            */
            assertion = (report.INTExists != false && report.CTR215 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Mandatory field not supplied",
                    Location = "/xbrli:xbrl/tns:ElectronicContactTelephone" + OccurrenceIndex(report.ElectronicContactTelephoneCollection_03_INTOccurrenceIndex) + "/tns:ElectronicContact.Telephone.Area.Code",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428235"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR215", Value = GetValueOrEmpty(report.CTR215) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428251
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428251()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428251
            Incorrect number of XBRL contexts
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.Y0)) > 1
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.Y0
            */
            assertion = (Count(report.RPY0Count) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.428278",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Incorrect number of XBRL contexts",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0LastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428251"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428253
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428253()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428253
            Incorrect number of XBRL contexts
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.Y0-1)) > 1
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.Y0-1
            */
            assertion = (Count(report.RPY0Minus1Count) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.428278",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Incorrect number of XBRL contexts",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0Minus1LastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428253"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428254
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428254()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428254
            Incorrect number of XBRL contexts
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.TOFA)) > 1 
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.TOFA
            */
            assertion = (Count(report.RPTOFACount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.428278",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Incorrect number of XBRL contexts",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPTOFALastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428254"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428255
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428255()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428255
            Incorrect number of XBRL contexts
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.JAus)) > 1 
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.JAUS
            */
            assertion = (Count(report.RPJAUSCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.428278",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Incorrect number of XBRL contexts",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428255"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428256
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428256()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428256
            Incorrect number of XBRL contexts
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.JForeign)) > 1 
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.JFOREIGN
            */
            assertion = (Count(report.RPJFOREIGNCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.428278",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Incorrect number of XBRL contexts",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJFOREIGNLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428256"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428259
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428259()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428259
            Incorrect number of XBRL contexts
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.CFCLC)) > 1 
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.CFCLC
            */
            assertion = (Count(report.RPCFCLCCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.428278",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Incorrect number of XBRL contexts",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPCFCLCLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428259"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428261
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428261()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428261
            Incorrect number of XBRL contexts
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.CFCUC)) > 1 
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.CFCUC
            */
            assertion = (Count(report.RPCFCUCCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.428278",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Incorrect number of XBRL contexts",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPCFCUCLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428261"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428262
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428262()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428262
            Incorrect number of XBRL contexts
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.Opening(Instant))) > 1 
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP.Opening(Instant)
            */
            assertion = (Count(report.RPOpeningInstantCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.428278",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Incorrect number of XBRL contexts",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOpeningInstantLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428262"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428263
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428263()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428263
            Incorrect number of XBRL contexts
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.Closing(Instant))) > 1 
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP.Closing(Instant)
            */
            assertion = (Count(report.RPClosingInstantCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.428278",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Incorrect number of XBRL contexts",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPClosingInstantLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428263"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428264
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428264()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428264
            Incorrect number of XBRL contexts
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.CMPLY)) > 1 
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.CMPLY
            */
            assertion = (Count(report.RPCMPLYCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.428278",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Incorrect number of XBRL contexts",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPCMPLYLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428264"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428265
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428265()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428265
            Incorrect number of XBRL contexts
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.ORD)) > 1 
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.ORD
            */
            assertion = (Count(report.RPORDCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.428278",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Incorrect number of XBRL contexts",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPORDLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428265"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428272
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428272()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428272
            Incorrect number of XBRL contexts
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.Opening)) > 1 
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.Opening
            */
            assertion = (Count(report.RPOpeningCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.428278",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Incorrect number of XBRL contexts",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOpeningLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428272"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428273
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428273()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428273
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate = period.endDate WHERE NOT CONTEXT(SET(RP.Opening(Instant), RP.Closing(Instant), RP(Instant)))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate = ^EndDate
    
            Data Elements:
    
            ^StartDate = INT
    
            ^EndDate = INT
            */
            assertion = ((report.INTExists == true) && (report.INTStartDate == report.INTEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.INTOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428273"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428273
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428273_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428273
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate = period.endDate WHERE NOT CONTEXT(SET(RP.Opening(Instant), RP.Closing(Instant), RP(Instant)))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate = ^EndDate
    
            Data Elements:
    
            ^StartDate = RP
    
            ^EndDate = RP
            */
            assertion = ((report.RPExists == true) && (report.RPStartDate == report.RPEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428273"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428273
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428273_2()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428273
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate = period.endDate WHERE NOT CONTEXT(SET(RP.Opening(Instant), RP.Closing(Instant), RP(Instant)))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate = ^EndDate
    
            Data Elements:
    
            ^StartDate = RP:RP.CFCLC
    
            ^EndDate = RP:RP.CFCLC
            */
            assertion = ((report.RPCFCLCExists == true) && (report.RPCFCLCStartDate == report.RPCFCLCEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPCFCLCOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428273"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428273
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428273_3()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428273
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate = period.endDate WHERE NOT CONTEXT(SET(RP.Opening(Instant), RP.Closing(Instant), RP(Instant)))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate = ^EndDate
    
            Data Elements:
    
            ^StartDate = RP:RP.CFCUC
    
            ^EndDate = RP:RP.CFCUC
            */
            assertion = ((report.RPCFCUCExists == true) && (report.RPCFCUCStartDate == report.RPCFCUCEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPCFCUCOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428273"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428273
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428273_4()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428273
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate = period.endDate WHERE NOT CONTEXT(SET(RP.Opening(Instant), RP.Closing(Instant), RP(Instant)))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate = ^EndDate
    
            Data Elements:
    
            ^StartDate = RP:RP.Closing
    
            ^EndDate = RP:RP.Closing
            */
            assertion = ((report.RPClosingExists == true) && (report.RPClosingStartDate == report.RPClosingEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPClosingOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428273"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428273
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428273_5()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428273
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate = period.endDate WHERE NOT CONTEXT(SET(RP.Opening(Instant), RP.Closing(Instant), RP(Instant)))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate = ^EndDate
    
            Data Elements:
    
            ^StartDate = RP:RP.CMPLY
    
            ^EndDate = RP:RP.CMPLY
            */
            assertion = ((report.RPCMPLYExists == true) && (report.RPCMPLYStartDate == report.RPCMPLYEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPCMPLYOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428273"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428273
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428273_6()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428273
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate = period.endDate WHERE NOT CONTEXT(SET(RP.Opening(Instant), RP.Closing(Instant), RP(Instant)))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate = ^EndDate
    
            Data Elements:
    
            ^StartDate = RP:RP.GeneralPool
    
            ^EndDate = RP:RP.GeneralPool
            */
            assertion = ((report.RPGeneralPoolExists == true) && (report.RPGeneralPoolStartDate == report.RPGeneralPoolEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPGeneralPoolOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428273"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428273
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428273_7()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428273
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate = period.endDate WHERE NOT CONTEXT(SET(RP.Opening(Instant), RP.Closing(Instant), RP(Instant)))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate = ^EndDate
    
            Data Elements:
    
            ^StartDate = RP:RP.Investor
    
            ^EndDate = RP:RP.Investor
            */
            assertion = ((report.RPInvestorExists == true) && (report.RPInvestorStartDate == report.RPInvestorEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPInvestorOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428273"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428273
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428273_8()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428273
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate = period.endDate WHERE NOT CONTEXT(SET(RP.Opening(Instant), RP.Closing(Instant), RP(Instant)))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate = ^EndDate
    
            Data Elements:
    
            ^StartDate = RP.Investor:RP.Investor.YPrevious
    
            ^EndDate = RP.Investor:RP.Investor.YPrevious
            */
            assertion = ((report.RPInvestorYPreviousExists == true) && (report.RPInvestorYPreviousStartDate == report.RPInvestorYPreviousEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPInvestorYPreviousOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428273"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428273
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428273_9()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428273
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate = period.endDate WHERE NOT CONTEXT(SET(RP.Opening(Instant), RP.Closing(Instant), RP(Instant)))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate = ^EndDate
    
            Data Elements:
    
            ^StartDate = RP:RP.JAUS
    
            ^EndDate = RP:RP.JAUS
            */
            assertion = ((report.RPJAUSExists == true) && (report.RPJAUSStartDate == report.RPJAUSEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428273"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428273
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428273_10()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428273
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate = period.endDate WHERE NOT CONTEXT(SET(RP.Opening(Instant), RP.Closing(Instant), RP(Instant)))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate = ^EndDate
    
            Data Elements:
    
            ^StartDate = RP:RP.JFOREIGN
    
            ^EndDate = RP:RP.JFOREIGN
            */
            assertion = ((report.RPJFOREIGNExists == true) && (report.RPJFOREIGNStartDate == report.RPJFOREIGNEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJFOREIGNOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428273"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428273
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428273_11()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428273
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate = period.endDate WHERE NOT CONTEXT(SET(RP.Opening(Instant), RP.Closing(Instant), RP(Instant)))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate = ^EndDate
    
            Data Elements:
    
            ^StartDate = RP:RP.LimitedPartners
    
            ^EndDate = RP:RP.LimitedPartners
            */
            assertion = ((report.RPLimitedPartnersExists == true) && (report.RPLimitedPartnersStartDate == report.RPLimitedPartnersEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPLimitedPartnersOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428273"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428273
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428273_12()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428273
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate = period.endDate WHERE NOT CONTEXT(SET(RP.Opening(Instant), RP.Closing(Instant), RP(Instant)))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate = ^EndDate
    
            Data Elements:
    
            ^StartDate = RP.LimitedPartners:RP.LimitedPartners.YPrevious
    
            ^EndDate = RP.LimitedPartners:RP.LimitedPartners.YPrevious
            */
            assertion = ((report.RPLimitedPartnersYPreviousExists == true) && (report.RPLimitedPartnersYPreviousStartDate == report.RPLimitedPartnersYPreviousEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPLimitedPartnersYPreviousOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428273"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428273
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428273_13()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428273
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate = period.endDate WHERE NOT CONTEXT(SET(RP.Opening(Instant), RP.Closing(Instant), RP(Instant)))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate = ^EndDate
    
            Data Elements:
    
            ^StartDate = RP:RP.LowCost
    
            ^EndDate = RP:RP.LowCost
            */
            assertion = ((report.RPLowCostExists == true) && (report.RPLowCostStartDate == report.RPLowCostEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPLowCostOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428273"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428273
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428273_14()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428273
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate = period.endDate WHERE NOT CONTEXT(SET(RP.Opening(Instant), RP.Closing(Instant), RP(Instant)))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate = ^EndDate
    
            Data Elements:
    
            ^StartDate = RP:RP.Opening
    
            ^EndDate = RP:RP.Opening
            */
            assertion = ((report.RPOpeningExists == true) && (report.RPOpeningStartDate == report.RPOpeningEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOpeningOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428273"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428273
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428273_15()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428273
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate = period.endDate WHERE NOT CONTEXT(SET(RP.Opening(Instant), RP.Closing(Instant), RP(Instant)))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate = ^EndDate
    
            Data Elements:
    
            ^StartDate = RP:RP.ORD
    
            ^EndDate = RP:RP.ORD
            */
            assertion = ((report.RPORDExists == true) && (report.RPORDStartDate == report.RPORDEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPORDOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428273"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428273
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428273_16()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428273
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate = period.endDate WHERE NOT CONTEXT(SET(RP.Opening(Instant), RP.Closing(Instant), RP(Instant)))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate = ^EndDate
    
            Data Elements:
    
            ^StartDate = RP:RP.ProspectiveRules
    
            ^EndDate = RP:RP.ProspectiveRules
            */
            assertion = ((report.RPProspectiveRulesExists == true) && (report.RPProspectiveRulesStartDate == report.RPProspectiveRulesEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPProspectiveRulesOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428273"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428273
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428273_17()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428273
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate = period.endDate WHERE NOT CONTEXT(SET(RP.Opening(Instant), RP.Closing(Instant), RP(Instant)))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate = ^EndDate
    
            Data Elements:
    
            ^StartDate = RP:RP.TOFA
    
            ^EndDate = RP:RP.TOFA
            */
            assertion = ((report.RPTOFAExists == true) && (report.RPTOFAStartDate == report.RPTOFAEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPTOFAOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428273"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428273
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428273_18()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428273
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate = period.endDate WHERE NOT CONTEXT(SET(RP.Opening(Instant), RP.Closing(Instant), RP(Instant)))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate = ^EndDate
    
            Data Elements:
    
            ^StartDate = RP:RP.Y0
    
            ^EndDate = RP:RP.Y0
            */
            assertion = ((report.RPY0Exists == true) && (report.RPY0StartDate == report.RPY0EndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0OccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428273"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428273
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428273_19()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428273
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate = period.endDate WHERE NOT CONTEXT(SET(RP.Opening(Instant), RP.Closing(Instant), RP(Instant)))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate = ^EndDate
    
            Data Elements:
    
            ^StartDate = RP:RP.Y0-1
    
            ^EndDate = RP:RP.Y0-1
            */
            assertion = ((report.RPY0Minus1Exists == true) && (report.RPY0Minus1StartDate == report.RPY0Minus1EndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0Minus1OccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428273"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428273
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428273_20()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428273
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate = period.endDate WHERE NOT CONTEXT(SET(RP.Opening(Instant), RP.Closing(Instant), RP(Instant)))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate = ^EndDate
    
            Data Elements:
    
            ^StartDate = RP:RP.Y0Plus1
    
            ^EndDate = RP:RP.Y0Plus1
            */
            assertion = ((report.RPY0Plus1Exists == true) && (report.RPY0Plus1StartDate == report.RPY0Plus1EndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0Plus1OccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428273"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428273
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428273_21()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428273
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate = period.endDate WHERE NOT CONTEXT(SET(RP.Opening(Instant), RP.Closing(Instant), RP(Instant)))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate = ^EndDate
    
            Data Elements:
    
            ^StartDate = RP:RP.YPrevious
    
            ^EndDate = RP:RP.YPrevious
            */
            assertion = ((report.RPYPreviousExists == true) && (report.RPYPreviousStartDate == report.RPYPreviousEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPYPreviousOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428273"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428274
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428274()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428274
            Incorrect number of XBRL contexts
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.Closing)) > 1 
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.Closing
            */
            assertion = (Count(report.RPClosingCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.428278",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Incorrect number of XBRL contexts",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPClosingLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428274"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428327
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428327()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428327
            If RSA no-TFN contributions income is nil, then RSA income tax payable on no-TFN contributions amounts must also be nil
    
            Legacy Rule Format:
            IF [CTR309] > 0 AND( [CTR308] = 0 OR [CTR308] = NULL)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR309 > 0 AND (^CTR308 = 0 OR ^CTR308 = NULL)
    
            Data Elements:
    
            ^CTR309 = RP:IncomeTax.RetirementSavingsAccountTFNNotQuotedContributionsIncomeTaxPayable.Amount
    
            ^CTR308 = RP:Income.RetirementSavingsAccountTFNNotQuotedContributionsIncome.Amount
            */
            assertion = (report.CTR309.GetValueOrDefault() > 0 && (report.CTR308 == 0 || report.CTR308 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428327",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"RSA income tax payable on no-TFN contributions income should be nil",
                    LongDescription = @"If Retirement savings accounts no-TFN contributions income is nil, then Retirement savings accounts income tax payable on no-TFN contributions income should also be nil",
                    Location = "/xbrli:xbrl/tns:IncomeTax.RetirementSavingsAccountTFNNotQuotedContributionsIncomeTaxPayable.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428327"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR309", Value = GetValueOrEmpty(report.CTR309) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR308", Value = GetValueOrEmpty(report.CTR308) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428328
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428328()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428328
            Unstructured name of the Reporting Party should not be supplied
    
            Legacy Rule Format:
            IF [CTR313] <> NULL
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR313 <> NULL
    
            Data Elements:
    
            ^CTR313 = RP:Declaration:PersonUnstructuredName:PersonUnstructuredName.FullName.Text
            */
            assertion = (report.CTR313 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428328",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unstructured name of Reporting Party should not be supplied",
                    LongDescription = @"Unstructured name of Reporting Party should not be supplied. The Reporting Party's name should be supplied in a structured format",
                    Location = "/xbrli:xbrl/tns:Declaration" + OccurrenceIndex(report.DeclarationCollection_TrueAndCorrect_RPOccurrenceIndex) + "/tns:PersonUnstructuredName/tns:PersonUnstructuredName.FullName.Text",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428328"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR313", Value = report.CTR313 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428329
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428329()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428329
            RSA income tax payable on no-TFN contributions amounts must be equal to 32% of RSA no-TFN contributions income
    
            Legacy Rule Format:
            IF ([CTR309] <> ([CTR308] * 0.32) +/- 1)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (OutsideRange(^CTR309, ^CTR308 * 0.32, 1))
    
            Data Elements:
    
            ^CTR309 = RP:IncomeTax.RetirementSavingsAccountTFNNotQuotedContributionsIncomeTaxPayable.Amount
    
            ^CTR308 = RP:Income.RetirementSavingsAccountTFNNotQuotedContributionsIncome.Amount
            */
            assertion = OutsideRange(report.CTR309.GetValueOrDefault(), report.CTR308.GetValueOrDefault() * 0.32M, 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428329",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"RSA income tax payable on no-TFN contributions income is incorrect",
                    LongDescription = @"The amount of Retirement savings accounts income tax payable on no-TFN contributions income should be equal to the amount of Retirement savings accounts no-TFN contributions income multiplied by 0.32.",
                    Location = "/xbrli:xbrl/tns:IncomeTax.RetirementSavingsAccountTFNNotQuotedContributionsIncomeTaxPayable.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428329"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR309", Value = GetValueOrEmpty(report.CTR309) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR308", Value = GetValueOrEmpty(report.CTR308) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428330
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428330()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428330
            TOFA financial items cannot all be blank if TOFA income or expenses fields have been supplied
    
            Legacy Rule Format:
            IF ([CTR255] > 0 OR [CTR256] > 0) AND [CTR310] = NULL AND [CTR311] = NULL AND [CTR290] = NULL
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR255 > 0 OR ^CTR256 > 0) AND ^CTR310 = NULL AND ^CTR311 = NULL AND ^CTR290 = NULL
    
            Data Elements:
    
            ^CTR255 = RP.TOFA:Income.OtherTotal.Amount
    
            ^CTR256 = RP.TOFA:Expense.DeductibleOther.Amount
    
            ^CTR310 = RP.TOFA:Income.GainsTotal.Amount
    
            ^CTR311 = RP.TOFA:Expense.LossesTotal.Amount
    
            ^CTR290 = RP.TOFA:Income.FinancialArrangementsUnrealisedGains.Amount
            */
            assertion = ((report.CTR255.GetValueOrDefault() > 0 || report.CTR256.GetValueOrDefault() > 0) && report.CTR310 == null && report.CTR311 == null && report.CTR290 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.438051",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"TOFA financial items cannot all be blank if TOFA income or expenses fields have been supplied",
                    LongDescription = @"You have included an amount at one of the TOFA income or expense fields, therefore you must also include one or more of the following amounts: Total TOFA gains, Total TOFA losses, TOFA gains from unrealised movements in the value of financial arrangements",
                    Location = "/xbrli:xbrl/tns:Income.OtherTotal.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428330"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR255", Value = GetValueOrEmpty(report.CTR255) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR256", Value = GetValueOrEmpty(report.CTR256) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR310", Value = GetValueOrEmpty(report.CTR310) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR311", Value = GetValueOrEmpty(report.CTR311) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR290", Value = GetValueOrEmpty(report.CTR290) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428335
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428335()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428335
            Subtotal 1 must equal zero
    
            Legacy Rule Format:
            IF ([CTR327] >= [CTR199]) AND ([CTR322] <> 0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR327 >= ^CTR199) AND (^CTR322 <> 0)
    
            Data Elements:
    
            ^CTR322 = RP:IncomeTax.TaxOffsetNonRefundableSubTotal.Amount
    
            ^CTR327 = RP:IncomeTax.TaxOffsetNonRefundableNonCarryForward.Amount
    
            ^CTR199 = RP:IncomeTax.TaxableGross.Amount
            */
            assertion = (report.CTR327.GetValueOrDefault() >= report.CTR199.GetValueOrDefault() && report.CTR322 != 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428335",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Subtotal 1 must equal zero",
                    LongDescription = @"Where Gross Tax is less than or equal to non-refundable non-carry forward tax offsets; subtotal 1 must be zero",
                    Location = "/xbrli:xbrl/tns:IncomeTax.TaxOffsetNonRefundableSubTotal.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428335"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR327", Value = GetValueOrEmpty(report.CTR327) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR199", Value = GetValueOrEmpty(report.CTR199) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR322", Value = GetValueOrEmpty(report.CTR322) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428336
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428336()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428336

        Legacy Rule Format:
            IF [CTR322] = 0 AND [CTR324] <> 0
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR322 = 0 AND ^CTR324 <> 0
    
            Data Elements:
    
            ^CTR324 = RP:IncomeTax.TaxOffsetCarriedForwardSubTotal.Amount
    
            ^CTR322 = RP:IncomeTax.TaxOffsetNonRefundableSubTotal.Amount
            */
            assertion = (report.CTR322 == 0 && report.CTR324 != 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428336",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Where subtotal 1 equals zero - subtotal 2 all must also equal zero",
                    Location = "/xbrli:xbrl/tns:IncomeTax.TaxOffsetCarriedForwardSubTotal.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428336"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR322", Value = GetValueOrEmpty(report.CTR322) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR324", Value = GetValueOrEmpty(report.CTR324) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428337
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428337()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428337

        Legacy Rule Format:
            IF [CTR323] < [CTR322] AND ([CTR324] <> ([CTR322] - [CTR323]))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR323 < ^CTR322 AND (^CTR324 <> (^CTR322 - ^CTR323))
    
            Data Elements:
    
            ^CTR324 = RP:IncomeTax.TaxOffsetCarriedForwardSubTotal.Amount
    
            ^CTR323 = RP:IncomeTax.TaxOffsetNonRefundableCarryForward.Amount
    
            ^CTR322 = RP:IncomeTax.TaxOffsetNonRefundableSubTotal.Amount
            */
            assertion = (report.CTR323.GetValueOrDefault() < report.CTR322.GetValueOrDefault() && report.CTR324.GetValueOrDefault() != report.CTR322.GetValueOrDefault() - report.CTR323.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428337",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Subtotal 2 must equal Subtotal 1 less Non-refundable carry forward tax offsets",
                    LongDescription = @"Where non-refundable carry forward tax offsets is less than subtotal 1, subtotal 2 must equal subtotal 1 less non-refundable carry forward tax offsets",
                    Location = "/xbrli:xbrl/tns:IncomeTax.TaxOffsetCarriedForwardSubTotal.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428337"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR323", Value = GetValueOrEmpty(report.CTR323) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR322", Value = GetValueOrEmpty(report.CTR322) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR324", Value = GetValueOrEmpty(report.CTR324) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428338
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428338()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428338

        Legacy Rule Format:
            IF ([CTR323] >= [CTR322]) AND ([CTR324] <> 0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR323 >= ^CTR322) AND (^CTR324 <> 0)
    
            Data Elements:
    
            ^CTR324 = RP:IncomeTax.TaxOffsetCarriedForwardSubTotal.Amount
    
            ^CTR323 = RP:IncomeTax.TaxOffsetNonRefundableCarryForward.Amount
    
            ^CTR322 = RP:IncomeTax.TaxOffsetNonRefundableSubTotal.Amount
            */
            assertion = (report.CTR323.GetValueOrDefault() >= report.CTR322.GetValueOrDefault() && report.CTR324 != 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428338",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Subtotal 2 must equal zero",
                    LongDescription = @"Where non-refundable carry forward tax offsets greater than or equal to subtotal 1, subtotal 2 must equal zero",
                    Location = "/xbrli:xbrl/tns:IncomeTax.TaxOffsetCarriedForwardSubTotal.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428338"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR323", Value = GetValueOrEmpty(report.CTR323) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR322", Value = GetValueOrEmpty(report.CTR322) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR324", Value = GetValueOrEmpty(report.CTR324) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428339
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428339()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428339

        Legacy Rule Format:
            IF [CTR324] = 0 AND ([CTR325] <> 0 OR [CTR203] <> 0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR324 = 0 AND (^CTR325 <> 0 OR ^CTR203 <> 0)
    
            Data Elements:
    
            ^CTR324 = RP:IncomeTax.TaxOffsetCarriedForwardSubTotal.Amount
    
            ^CTR325 = RP:IncomeTax.TaxOffsetRefundableSubTotal.Amount
    
            ^CTR203 = RP:IncomeTax.Payable.Amount
            */
            assertion = (report.CTR324 == 0 && (report.CTR325 != 0 || report.CTR203 != 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428339",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Subtotal 3 and Tax Payable must equal zero",
                    LongDescription = @"Where subtotal 2 equals zero, subtotal 3 and Tax Payable all must also equal zero",
                    Location = "/xbrli:xbrl/tns:IncomeTax.TaxOffsetCarriedForwardSubTotal.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428339"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR324", Value = GetValueOrEmpty(report.CTR324) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR325", Value = GetValueOrEmpty(report.CTR325) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR203", Value = GetValueOrEmpty(report.CTR203) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428340
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428340()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428340
            If refundable tax offsets is less than subtal 2 then subtotal 3 equal subtotal 2 minus refundable tax offsets
    
            Legacy Rule Format:
            IF ([CTR316] < [CTR324]) AND ([CTR325] <> ([CTR324] - [CTR316]))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR316 < ^CTR324) AND (^CTR325 <> (^CTR324 - ^CTR316))
    
            Data Elements:
    
            ^CTR316 = RP:IncomeTax.TotalRefundableTaxOffsets.Amount
    
            ^CTR324 = RP:IncomeTax.TaxOffsetCarriedForwardSubTotal.Amount
    
            ^CTR325 = RP:IncomeTax.TaxOffsetRefundableSubTotal.Amount
            */
            assertion = (report.CTR316.GetValueOrDefault() < report.CTR324.GetValueOrDefault() && report.CTR325.GetValueOrDefault() != report.CTR324.GetValueOrDefault() - report.CTR316.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428340",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Subtotal 3 must equal Subtotal 2 less refundable tax offsets",
                    LongDescription = @"Where refundable tax offsets is less than subtotal 2, subtotal 3 must equal subtotal 2 less refundable tax offsets",
                    Location = "/xbrli:xbrl/tns:IncomeTax.TotalRefundableTaxOffsets.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428340"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR316", Value = GetValueOrEmpty(report.CTR316) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR324", Value = GetValueOrEmpty(report.CTR324) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR325", Value = GetValueOrEmpty(report.CTR325) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428341
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428341()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428341
            If refundable tax offsets is greater than or equal to subtotal 2 then subtotal 3 must be equal to zero
    
            Legacy Rule Format:
            IF [CTR316] >= [CTR324] AND ([CTR325] <> 0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR316 >= ^CTR324 AND (^CTR325 <> 0)
    
            Data Elements:
    
            ^CTR316 = RP:IncomeTax.TotalRefundableTaxOffsets.Amount
    
            ^CTR324 = RP:IncomeTax.TaxOffsetCarriedForwardSubTotal.Amount
    
            ^CTR325 = RP:IncomeTax.TaxOffsetRefundableSubTotal.Amount
            */
            assertion = (report.CTR316.GetValueOrDefault() >= report.CTR324.GetValueOrDefault() && report.CTR325 != 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428341",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Subtotal 3 must equal zero",
                    LongDescription = @"Where refundable tax offsets is greater than or equal to subtotal 2, subtotal 3 must equal zero",
                    Location = "/xbrli:xbrl/tns:IncomeTax.TotalRefundableTaxOffsets.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428341"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR316", Value = GetValueOrEmpty(report.CTR316) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR324", Value = GetValueOrEmpty(report.CTR324) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR325", Value = GetValueOrEmpty(report.CTR325) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428342
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428342()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428342
            Where subtotal 3 equals zero, tax offset refunds(Remainder of refundable tax offsets) must equal refundable tax offsets less subtotal 2
    
            Legacy Rule Format:
            IF [CTR325] = 0 AND ([CTR326] <> ([CTR316] - [CTR324]))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR325 = 0 AND (^CTR326 <> (^CTR316 - ^CTR324))
    
            Data Elements:
    
            ^CTR325 = RP:IncomeTax.TaxOffsetRefundableSubTotal.Amount
    
            ^CTR326 = RP:IncomeTax.TotalRemainderOfRefundableTaxOffsets.Amount
    
            ^CTR316 = RP:IncomeTax.TotalRefundableTaxOffsets.Amount
    
            ^CTR324 = RP:IncomeTax.TaxOffsetCarriedForwardSubTotal.Amount
            */
            assertion = (report.CTR325 == 0 && report.CTR326.GetValueOrDefault() != report.CTR316.GetValueOrDefault() - report.CTR324.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.430220",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax offset refunds (Remainder of refundable tax offsets) must equal refundable tax offsets less subtotal 2",
                    LongDescription = @"Where subtotal 3 equals zero, tax offset refunds (Remainder of refundable tax offsets) must equal refundable tax offsets less subtotal 2",
                    Location = "/xbrli:xbrl/tns:IncomeTax.TaxOffsetRefundableSubTotal.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428342"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR325", Value = GetValueOrEmpty(report.CTR325) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR326", Value = GetValueOrEmpty(report.CTR326) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR316", Value = GetValueOrEmpty(report.CTR316) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR324", Value = GetValueOrEmpty(report.CTR324) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428343
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428343()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428343
            Tax payable must equal zero
    
            Legacy Rule Format:
            IF [CTR325] = 0 AND [CTR203] <> 0
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR325 = 0 AND ^CTR203 <> 0
    
            Data Elements:
    
            ^CTR203 = RP:IncomeTax.Payable.Amount
    
            ^CTR325 = RP:IncomeTax.TaxOffsetRefundableSubTotal.Amount
            */
            assertion = (report.CTR325 == 0 && report.CTR203 != 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428343",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax payable must equal zero",
                    LongDescription = @"Where subtotal 3 equals zero, Tax Payable must also equal zero",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Payable.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428343"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR325", Value = GetValueOrEmpty(report.CTR325) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR203", Value = GetValueOrEmpty(report.CTR203) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428344
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428344()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428344
            If franking deficit tax offset is less than Subtotal 3 then tax payable is equal to Subotal 3 minus franking deficit tax offset
    
            Legacy Rule Format:
            IF ([CTR202] < [CTR325]) AND ([CTR203] <> ([CTR325] - [CTR202]))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR202 < ^CTR325) AND (^CTR203 <> (^CTR325 - ^CTR202))
    
            Data Elements:
    
            ^CTR203 = RP:IncomeTax.Payable.Amount
    
            ^CTR202 = RP.Y0-1:TaxOffsetClaim.FrankingDeficit.Amount
    
            ^CTR325 = RP:IncomeTax.TaxOffsetRefundableSubTotal.Amount
            */
            assertion = (report.CTR202.GetValueOrDefault() < report.CTR325.GetValueOrDefault() && report.CTR203.GetValueOrDefault() != report.CTR325.GetValueOrDefault() - report.CTR202.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428344",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax Payable must equal subtotal 3 less franking deficit tax offset",
                    LongDescription = @"Where franking deficit tax offset is less than subtotal 3, Tax Payable must equal subtotal 3 less franking deficit tax offset",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Payable.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428344"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR202", Value = GetValueOrEmpty(report.CTR202) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR325", Value = GetValueOrEmpty(report.CTR325) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR203", Value = GetValueOrEmpty(report.CTR203) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428345
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428345()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428345
            If franking deficit tax offset is greater than or equal to Subtotal 3 then tax payable should be zero
    
            Legacy Rule Format:
            IF [CTR202] >= [CTR325] AND [CTR203] <> 0
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR202 >= ^CTR325 AND ^CTR203 <> 0
    
            Data Elements:
    
            ^CTR203 = RP:IncomeTax.Payable.Amount
    
            ^CTR202 = RP.Y0-1:TaxOffsetClaim.FrankingDeficit.Amount
    
            ^CTR325 = RP:IncomeTax.TaxOffsetRefundableSubTotal.Amount
            */
            assertion = (report.CTR202.GetValueOrDefault() >= report.CTR325.GetValueOrDefault() && report.CTR203 != 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428345",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax payable must equal zero",
                    LongDescription = @"Where franking deficit tax offset is greater than or equal to subtotal 3, Tax Payable must equal zero",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Payable.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428345"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR202", Value = GetValueOrEmpty(report.CTR202) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR325", Value = GetValueOrEmpty(report.CTR325) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR203", Value = GetValueOrEmpty(report.CTR203) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428349
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428349()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428349
            Mandatory field not supplied
    
            Legacy Rule Format:
            IF [CTR319] = NULL
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR319 = NULL
    
            Data Elements:
    
            ^CTR319 = RP:RegulatoryDisclosures.ReportableTaxPosition.Indicator
            */
            assertion = (report.CTR319 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Mandatory field not supplied",
                    Location = "/xbrli:xbrl/tns:RegulatoryDisclosures.ReportableTaxPosition.Indicator",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428349"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR319", Value = GetValueOrEmpty(report.CTR319) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428350
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428350()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428350
            The amount entered at Gross foreign income must be greater than the Net foreign income.
    
            Legacy Rule Format:
            IF [CTR318] < [CTR154]
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR318 < ^CTR154
    
            Data Elements:
    
            ^CTR318 = RP:InternationalDealings.ForeignIncomeGross.Amount
    
            ^CTR154 = RP:Income.InternationalDealings.Net.Amount
            */
            assertion = (report.CTR318.GetValueOrDefault() < report.CTR154.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428350",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Gross foreign income is less than Net foreign income",
                    Location = "/xbrli:xbrl/tns:InternationalDealings.ForeignIncomeGross.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428350"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR318", Value = GetValueOrEmpty(report.CTR318) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR154", Value = GetValueOrEmpty(report.CTR154) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428351
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428351()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428351
            If Net foreign income is present then gross foreign income must also be present
    
            Legacy Rule Format:
            IF [CTR154] > 0 AND [CTR318] = NULL
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR154 > 0 AND ^CTR318 = NULL
    
            Data Elements:
    
            ^CTR318 = RP:InternationalDealings.ForeignIncomeGross.Amount
    
            ^CTR154 = RP:Income.InternationalDealings.Net.Amount
            */
            assertion = (report.CTR154.GetValueOrDefault() > 0 && report.CTR318 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428351",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Net foreign income is present and Gross foreign income is missing.",
                    Location = "/xbrli:xbrl/tns:InternationalDealings.ForeignIncomeGross.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428351"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR154", Value = GetValueOrEmpty(report.CTR154) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR318", Value = GetValueOrEmpty(report.CTR318) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428356
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428356()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428356
            Refundable tax offsets must not be less than sum of Refundable R&D tax offset and Current year refundable digital games tax offset
    
            Legacy Rule Format:
            IF ([CTR210] + [CTR454]) > [CTR316]
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR210 + ^CTR454) > ^CTR316
    
            Data Elements:
    
            ^CTR210 = RP:TaxConcession.ResearchAndDevelopment.TaxOffsetRefundable.Amount
    
            ^CTR454 = RP:TaxOffsetClaim.DigitalGamesTaxOffset.Amount
    
            ^CTR316 = RP:IncomeTax.TotalRefundableTaxOffsets.Amount
            */
            assertion = (report.CTR210.GetValueOrDefault() + report.CTR454.GetValueOrDefault() > report.CTR316.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.438356",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Sum of Refundable R&D tax offset and Current year refundable digital games tax offset amount being claimed is incorrect",
                    LongDescription = @"Refundable tax offsets must not be less than sum of Refundable R&D tax offset and Current year refundable digital games tax offset",
                    Location = "/xbrli:xbrl/tns:TaxConcession.ResearchAndDevelopment.TaxOffsetRefundable.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428356"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR210", Value = GetValueOrEmpty(report.CTR210) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR454", Value = GetValueOrEmpty(report.CTR454) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR316", Value = GetValueOrEmpty(report.CTR316) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428358
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428358()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428358
            Subtotal 1 must be equal to Gross Tax less non-refundable non-carry forward tax offsets
    
            Legacy Rule Format:
            IF ([CTR327] < [CTR199]) AND ([CTR322] <> [CTR199] - [CTR327])
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR327 < ^CTR199) AND (^CTR322 <> ^CTR199 - ^CTR327)
    
            Data Elements:
    
            ^CTR322 = RP:IncomeTax.TaxOffsetNonRefundableSubTotal.Amount
    
            ^CTR327 = RP:IncomeTax.TaxOffsetNonRefundableNonCarryForward.Amount
    
            ^CTR199 = RP:IncomeTax.TaxableGross.Amount
            */
            assertion = (report.CTR327.GetValueOrDefault() < report.CTR199.GetValueOrDefault() && report.CTR322.GetValueOrDefault() != report.CTR199.GetValueOrDefault() - report.CTR327.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428358",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Subtotal 1 must be equal to Gross Tax less non-refundable non-carry forward tax offsets",
                    LongDescription = @"Where Gross Tax is greater than non-refundable non-carry forward tax offsets; subtotal 1 must be equal to Gross Tax less non-refundable non-carry forward tax offsets",
                    Location = "/xbrli:xbrl/tns:IncomeTax.TaxOffsetNonRefundableSubTotal.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428358"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR327", Value = GetValueOrEmpty(report.CTR327) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR199", Value = GetValueOrEmpty(report.CTR199) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR322", Value = GetValueOrEmpty(report.CTR322) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428366
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428366()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428366
            Australia is an invalid Ultimate holding company Country code.
    
            Legacy Rule Format:
            IF [CTR45] = "au"
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR45 = 'au'
    
            Data Elements:
    
            ^CTR45 = RP:UltimateHoldingCompany:Residency.TaxPurposesCountry.Code
            */
            assertion = (report.CTR45 == @"au");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428366",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid Ultimate holding company country code - Australia is not a valid code",
                    Location = "/xbrli:xbrl/tns:UltimateHoldingCompany/tns:Residency.TaxPurposesCountry.Code",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428366"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR45", Value = GetValueOrEmpty(report.CTR45) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428401
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428401()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428401

        Legacy Rule Format:
            IF [CTR322] = 0 AND [CTR325] <> 0
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR322 = 0 AND ^CTR325 <> 0
    
            Data Elements:
    
            ^CTR325 = RP:IncomeTax.TaxOffsetRefundableSubTotal.Amount
    
            ^CTR322 = RP:IncomeTax.TaxOffsetNonRefundableSubTotal.Amount
            */
            assertion = (report.CTR322 == 0 && report.CTR325 != 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428401",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Where subtotal 1 equals zero - subtotal 3 all must also equal zero",
                    Location = "/xbrli:xbrl/tns:IncomeTax.TaxOffsetRefundableSubTotal.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428401"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR322", Value = GetValueOrEmpty(report.CTR322) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR325", Value = GetValueOrEmpty(report.CTR325) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428402
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428402()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428402

        Legacy Rule Format:
            IF [CTR322] = 0 AND [CTR203] <> 0
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR322 = 0 AND ^CTR203 <> 0
    
            Data Elements:
    
            ^CTR203 = RP:IncomeTax.Payable.Amount
    
            ^CTR322 = RP:IncomeTax.TaxOffsetNonRefundableSubTotal.Amount
            */
            assertion = (report.CTR322 == 0 && report.CTR203 != 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428402",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Where subtotal 1 equals zero - Tax Payable all must also equal zero",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Payable.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428402"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR322", Value = GetValueOrEmpty(report.CTR322) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR203", Value = GetValueOrEmpty(report.CTR203) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.428428
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428428()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.428428
            Tax offset refunds (Remainder of refundable tax offsets) must be zero
    
            Legacy Rule Format:
            IF ([CTR316] < [CTR324]) AND ([CTR326] > 0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR316 < ^CTR324) AND (^CTR326 > 0)
    
            Data Elements:
    
            ^CTR326 = RP:IncomeTax.TotalRemainderOfRefundableTaxOffsets.Amount
    
            ^CTR316 = RP:IncomeTax.TotalRefundableTaxOffsets.Amount
    
            ^CTR324 = RP:IncomeTax.TaxOffsetCarriedForwardSubTotal.Amount
            */
            assertion = (report.CTR316.GetValueOrDefault() < report.CTR324.GetValueOrDefault() && report.CTR326.GetValueOrDefault() > 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.430219",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax offset refunds (Remainder of refundable tax offsets) must be zero",
                    LongDescription = @"If Refundable tax offsets is less than Subtotal 2 then Tax offset refunds (Remainder of refundable tax offsets) must be zero",
                    Location = "/xbrli:xbrl/tns:IncomeTax.TotalRemainderOfRefundableTaxOffsets.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.428428"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR316", Value = GetValueOrEmpty(report.CTR316) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR324", Value = GetValueOrEmpty(report.CTR324) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR326", Value = GetValueOrEmpty(report.CTR326) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429831
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429831()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429831
            Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters
    
            Legacy Rule Format:
            IF [CTR300] <> NULL AND [CTR300] <> MONETARY(U,13,2)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR300, 'U', 13, 2)
    
            Data Elements:
    
            ^CTR300 = RP:TaxConcession.ResearchAndDevelopment.TaxOffsetNonRefundableUtilised.Amount
            */
            assertion = NotMonetary(report.CTR300, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:TaxConcession.ResearchAndDevelopment.TaxOffsetNonRefundableUtilised.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429831"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR300", Value = GetValueOrEmpty(report.CTR300) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429833
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429833()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429833
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR318] <> NULL AND [CTR318] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR318, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR318 = RP:InternationalDealings.ForeignIncomeGross.Amount
            */
            assertion = NotMonetary(report.CTR318, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:InternationalDealings.ForeignIncomeGross.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429833"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR318", Value = GetValueOrEmpty(report.CTR318) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429835
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429835()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429835
            Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters
    
            Legacy Rule Format:
            IF [CTR326] <> NULL AND [CTR326] <> MONETARY(U,13,2)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR326, 'U', 13, 2)
    
            Data Elements:
    
            ^CTR326 = RP:IncomeTax.TotalRemainderOfRefundableTaxOffsets.Amount
            */
            assertion = NotMonetary(report.CTR326, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:IncomeTax.TotalRemainderOfRefundableTaxOffsets.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429835"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR326", Value = GetValueOrEmpty(report.CTR326) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429836
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429836()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429836
            Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters
    
            Legacy Rule Format:
            IF [CTR317] <> NULL AND [CTR317] <> MONETARY(U,13,2)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR317, 'U', 13, 2)
    
            Data Elements:
    
            ^CTR317 = RP:IncomeTax.Deduction.CreditsEligibleTotal.Amount
            */
            assertion = NotMonetary(report.CTR317, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.CreditsEligibleTotal.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429836"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR317", Value = GetValueOrEmpty(report.CTR317) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429837
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429837()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429837
            Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters
    
            Legacy Rule Format:
            IF [CTR325] <> NULL AND [CTR325] <> MONETARY(U,13,2)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR325, 'U', 13, 2)
    
            Data Elements:
    
            ^CTR325 = RP:IncomeTax.TaxOffsetRefundableSubTotal.Amount
            */
            assertion = NotMonetary(report.CTR325, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:IncomeTax.TaxOffsetRefundableSubTotal.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429837"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR325", Value = GetValueOrEmpty(report.CTR325) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429838
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429838()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429838
            Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters
    
            Legacy Rule Format:
            IF [CTR316] <> NULL AND [CTR316] <> MONETARY(U,13,2)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR316, 'U', 13, 2)
    
            Data Elements:
    
            ^CTR316 = RP:IncomeTax.TotalRefundableTaxOffsets.Amount
            */
            assertion = NotMonetary(report.CTR316, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:IncomeTax.TotalRefundableTaxOffsets.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429838"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR316", Value = GetValueOrEmpty(report.CTR316) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429839
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429839()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429839
            Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters
    
            Legacy Rule Format:
            IF [CTR324] <> NULL AND [CTR324] <> MONETARY(U,13,2)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR324, 'U', 13, 2)
    
            Data Elements:
    
            ^CTR324 = RP:IncomeTax.TaxOffsetCarriedForwardSubTotal.Amount
            */
            assertion = NotMonetary(report.CTR324, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:IncomeTax.TaxOffsetCarriedForwardSubTotal.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429839"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR324", Value = GetValueOrEmpty(report.CTR324) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429840
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429840()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429840
            Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters
    
            Legacy Rule Format:
            IF [CTR323] <> NULL AND [CTR323] <> MONETARY(U,13,2)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR323, 'U', 13, 2)
    
            Data Elements:
    
            ^CTR323 = RP:IncomeTax.TaxOffsetNonRefundableCarryForward.Amount
            */
            assertion = NotMonetary(report.CTR323, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:IncomeTax.TaxOffsetNonRefundableCarryForward.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429840"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR323", Value = GetValueOrEmpty(report.CTR323) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429841
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429841()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429841
            Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters
    
            Legacy Rule Format:
            IF [CTR322] <> NULL AND [CTR322] <> MONETARY(U,13,2)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR322, 'U', 13, 2)
    
            Data Elements:
    
            ^CTR322 = RP:IncomeTax.TaxOffsetNonRefundableSubTotal.Amount
            */
            assertion = NotMonetary(report.CTR322, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:IncomeTax.TaxOffsetNonRefundableSubTotal.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429841"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR322", Value = GetValueOrEmpty(report.CTR322) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429842
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429842()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429842
            Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters
    
            Legacy Rule Format:
            IF [CTR327] <> NULL AND [CTR327] <> MONETARY(U,13,2)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR327, 'U', 13, 2)
    
            Data Elements:
    
            ^CTR327 = RP:IncomeTax.TaxOffsetNonRefundableNonCarryForward.Amount
            */
            assertion = NotMonetary(report.CTR327, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:IncomeTax.TaxOffsetNonRefundableNonCarryForward.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429842"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR327", Value = GetValueOrEmpty(report.CTR327) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429844
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429844()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429844
            Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters
    
            Legacy Rule Format:
            IF [CTR320] <> NULL AND [CTR320] <> MONETARY(U,13,2)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR320, 'U', 13, 2)
    
            Data Elements:
    
            ^CTR320 = RP:IncomeTax.TotalTaxOnTaxableIncome.Amount
            */
            assertion = NotMonetary(report.CTR320, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:IncomeTax.TotalTaxOnTaxableIncome.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429844"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR320", Value = GetValueOrEmpty(report.CTR320) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429845
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429845()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429845
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR310] <> NULL AND [CTR310] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR310, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR310 = RP.TOFA:Income.GainsTotal.Amount
            */
            assertion = NotMonetary(report.CTR310, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.GainsTotal.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429845"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR310", Value = GetValueOrEmpty(report.CTR310) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429846
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429846()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429846
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR311] <> NULL AND [CTR311] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR311, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR311 = RP.TOFA:Expense.LossesTotal.Amount
            */
            assertion = NotMonetary(report.CTR311, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.LossesTotal.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429846"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR311", Value = GetValueOrEmpty(report.CTR311) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429847
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429847()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429847
            Monetary field - unsigned, maximum of 13 characters, may include a decimal point and up to 2 decimals
    
            Legacy Rule Format:
            IF [CTR309] <> NULL AND [CTR309] <> MONETARY(U,13,2)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR309, 'U', 13, 2)
    
            Data Elements:
    
            ^CTR309 = RP:IncomeTax.RetirementSavingsAccountTFNNotQuotedContributionsIncomeTaxPayable.Amount
            */
            assertion = NotMonetary(report.CTR309, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:IncomeTax.RetirementSavingsAccountTFNNotQuotedContributionsIncomeTaxPayable.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429847"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR309", Value = GetValueOrEmpty(report.CTR309) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429848
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429848()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429848
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR308] <> NULL AND [CTR308] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR308, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR308 = RP:Income.RetirementSavingsAccountTFNNotQuotedContributionsIncome.Amount
            */
            assertion = NotMonetary(report.CTR308, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.RetirementSavingsAccountTFNNotQuotedContributionsIncome.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429848"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR308", Value = GetValueOrEmpty(report.CTR308) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429849
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429849()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429849
            Monetary field - unsigned, maximum of 13 characters, may include a decimal point and up to 2 decimals
    
            Legacy Rule Format:
            IF [CTR299] <> NULL AND [CTR299] <> MONETARY(U,13,2)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR299, 'U', 13, 2)
    
            Data Elements:
    
            ^CTR299 = RP:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldFromCloselyHeldTrustTFNNotQuoted.Amount
            */
            assertion = NotMonetary(report.CTR299, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldFromCloselyHeldTrustTFNNotQuoted.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429849"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR299", Value = GetValueOrEmpty(report.CTR299) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429850
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429850()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429850
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR290] <> NULL AND [CTR290] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR290, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR290 = RP.TOFA:Income.FinancialArrangementsUnrealisedGains.Amount
            */
            assertion = NotMonetary(report.CTR290, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.FinancialArrangementsUnrealisedGains.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429850"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR290", Value = GetValueOrEmpty(report.CTR290) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429852
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429852()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429852
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF CTR:RP:lrla.xx.xx:Remuneration.PaymentToForeignResidentGross.Amount <> NULL AND CTR:RP:lrla.xx.xx:Remuneration.PaymentToForeignResidentGross.Amount <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR57, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR57 = RP:Remuneration.PaymentToForeignResidentGross.Amount
            */
            assertion = NotMonetary(report.CTR57, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Remuneration.PaymentToForeignResidentGross.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429852"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR57", Value = GetValueOrEmpty(report.CTR57) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429856
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429856()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429856
            Field value supplied exceeds maximum field length
    
            Legacy Rule Format:
            IF CTR:RP:pyde.xx.xx:AddressDetails.LocalityName.Text <> NULLORBLANK AND (LENGTH(CTR:RP:pyde.xx.xx:AddressDetails.LocalityName.Text) > 60)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^CTR19) > 60
    
            Data Elements:
    
            ^CTR19 = RP:AddressDetails:AddressDetails.LocalityName.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (Length(report.CTR19) > 60);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field value supplied exceeds maximum field length",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.LocalityName.Text",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429856"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR19", Value = report.CTR19 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429858
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429858()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429858
            Field value supplied exceeds maximum field length
    
            Legacy Rule Format:
            IF [CTR43] <> NULLORBLANK AND (LENGTH([CTR43]) > 76)
             RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^CTR43) > 76
    
            Data Elements:
    
            ^CTR43 = RP:UltimateHoldingCompany:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
            */
            assertion = (Length(report.CTR43) > 76);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field value supplied exceeds maximum field length",
                    Location = "/xbrli:xbrl/tns:UltimateHoldingCompany/tns:OrganisationNameDetails" + OccurrenceIndex(report.UltimateHoldingCompany_OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429858"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR43", Value = report.CTR43 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429859
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429859()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429859
            Field value supplied exceeds maximum field length
    
            Legacy Rule Format:
            IF [CTR46] <> NULLORBLANK AND (LENGTH([CTR46]) > 76)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^CTR46) > 76
    
            Data Elements:
    
            ^CTR46 = RP:ImmediateHoldingCompany:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
            */
            assertion = (Length(report.CTR46) > 76);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field value supplied exceeds maximum field length",
                    Location = "/xbrli:xbrl/tns:ImmediateHoldingCompany/tns:OrganisationNameDetails" + OccurrenceIndex(report.ImmediateHoldingCompany_OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429859"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR46", Value = report.CTR46 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429860
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429860()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429860
            Field value supplied exceeds maximum field length
    
            Legacy Rule Format:
            IF [CTR48] <> NULLORBLANK AND (LENGTH([CTR48]) > 108)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^CTR48) > 108
    
            Data Elements:
    
            ^CTR48 = RP:OrganisationDetails.MainIncomeActivity.Description
            */
            assertion = (Length(report.CTR48) > 108);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field value supplied exceeds maximum field length",
                    Location = "/xbrli:xbrl/tns:OrganisationDetails.MainIncomeActivity.Description",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429860"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR48", Value = report.CTR48 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429862
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429862()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429862
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF CTR:RP:lrla.xx.xx:Remuneration.ABNNotQuotedPaymentGross.Amount <> NULL AND CTR:RP:lrla.xx.xx:Remuneration.ABNNotQuotedPaymentGross.Amount <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR58, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR58 = RP:Remuneration.ABNNotQuotedPaymentGross.Amount
            */
            assertion = NotMonetary(report.CTR58, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Remuneration.ABNNotQuotedPaymentGross.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429862"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR58", Value = GetValueOrEmpty(report.CTR58) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429863
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429863()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429863
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR59] <> NULL AND [CTR59] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR59, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR59 = RP:Income.SaleOfGoodsAndServices.Other.Amount
            */
            assertion = NotMonetary(report.CTR59, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.SaleOfGoodsAndServices.Other.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429863"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR59", Value = GetValueOrEmpty(report.CTR59) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429864
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429864()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429864
            Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters
    
            Legacy Rule Format:
            IF [CTR60] <> NULL AND [CTR60] <> MONETARY(S,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR60, 'S', 11, 0)
    
            Data Elements:
    
            ^CTR60 = RP:Income.PartnershipDistributionGross.Amount
            */
            assertion = NotMonetary(report.CTR60, @"S", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.PartnershipDistributionGross.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429864"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR60", Value = GetValueOrEmpty(report.CTR60) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429865
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429865()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429865
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR61] <> NULL AND [CTR61] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR61, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR61 = RP:TrustDistributionIncome:Income.TrustDistributionGross.Amount
            */
            assertion = NotMonetary(report.CTR61, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:TrustDistributionIncome/tns:Income.TrustDistributionGross.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429865"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR61", Value = GetValueOrEmpty(report.CTR61) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429866
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429866()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429866
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR63] <> NULL AND [CTR63] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR63, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR63 = RP:Income.ForestryManagedInvestmentScheme.Amount
            */
            assertion = NotMonetary(report.CTR63, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.ForestryManagedInvestmentScheme.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429866"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR63", Value = GetValueOrEmpty(report.CTR63) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429867
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429867()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429867
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR64] <> NULL AND [CTR64] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR64, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR64 = RP:Income.Interest.Gross.Amount
            */
            assertion = NotMonetary(report.CTR64, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.Interest.Gross.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429867"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR64", Value = GetValueOrEmpty(report.CTR64) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429868
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429868()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429868
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR65] <> NULL AND [CTR65] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR65, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR65 = RP:Income.RentLeasingHiringGross.Amount
            */
            assertion = NotMonetary(report.CTR65, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.RentLeasingHiringGross.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429868"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR65", Value = GetValueOrEmpty(report.CTR65) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429869
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429869()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429869
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR66] <> NULL AND [CTR66] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR66, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR66 = RP:Income.DividendsTotal.Amount
            */
            assertion = NotMonetary(report.CTR66, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.DividendsTotal.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429869"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR66", Value = GetValueOrEmpty(report.CTR66) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429870
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429870()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429870
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR67] <> NULL AND [CTR67] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR67, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR67 = RP:Remuneration.FringeBenefits.EmployeeContribution.Amount
            */
            assertion = NotMonetary(report.CTR67, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Remuneration.FringeBenefits.EmployeeContribution.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429870"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR67", Value = GetValueOrEmpty(report.CTR67) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429871
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429871()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429871
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR68] <> NULL AND [CTR68] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR68, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR68 = RP:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
            */
            assertion = NotMonetary(report.CTR68, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429871"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR68", Value = GetValueOrEmpty(report.CTR68) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429872
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429872()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429872
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR70] <> NULL AND [CTR70] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR70, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR70 = RP:Income.AssetRevaluationsUnrealisedGains.Amount
            */
            assertion = NotMonetary(report.CTR70, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.AssetRevaluationsUnrealisedGains.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429872"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR70", Value = GetValueOrEmpty(report.CTR70) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429873
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429873()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429873
            Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters
    
            Legacy Rule Format:
            IF [CTR71] <> NULL AND [CTR71] <> MONETARY(S,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR71, 'S', 11, 0)
    
            Data Elements:
    
            ^CTR71 = RP:Income.OtherGross.Amount
            */
            assertion = NotMonetary(report.CTR71, @"S", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.OtherGross.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429873"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR71", Value = GetValueOrEmpty(report.CTR71) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429874
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429874()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429874
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR72] <> NULL AND [CTR72] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR72, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR72 = RP:Expense.ForeignResidentWithholding.Amount
            */
            assertion = NotMonetary(report.CTR72, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.ForeignResidentWithholding.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429874"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR72", Value = GetValueOrEmpty(report.CTR72) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429875
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429875()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429875
            Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters
    
            Legacy Rule Format:
            IF [CTR73] <> NULL AND [CTR73] <> MONETARY(S,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR73, 'S', 11, 0)
    
            Data Elements:
    
            ^CTR73 = RP:Expense.Operating.CostOfSales.Amount
            */
            assertion = NotMonetary(report.CTR73, @"S", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.Operating.CostOfSales.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429875"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR73", Value = GetValueOrEmpty(report.CTR73) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429877
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429877()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429877
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR75] <> NULL AND [CTR75] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR75, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR75 = RP:SuperannuationContribution.EmployerContributions.Amount
            */
            assertion = NotMonetary(report.CTR75, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:SuperannuationContribution.EmployerContributions.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429877"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR75", Value = GetValueOrEmpty(report.CTR75) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429878
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429878()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429878
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR76] <> NULL AND [CTR76] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR76, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR76 = RP:Expense.Operating.BadDebts.Amount
            */
            assertion = NotMonetary(report.CTR76, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.Operating.BadDebts.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429878"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR76", Value = GetValueOrEmpty(report.CTR76) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429879
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429879()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429879
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR77] <> NULL AND [CTR77] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR77, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR77 = RP.JAUS:Expense.Operating.Lease.Amount
            */
            assertion = NotMonetary(report.CTR77, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.Operating.Lease.Amount[@contextRef='" + report.RPJAUSId + "']",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429879"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR77", Value = GetValueOrEmpty(report.CTR77) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429880
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429880()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429880
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR78] <> NULL AND [CTR78] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR78, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR78 = RP.JFOREIGN:Expense.Operating.Lease.Amount
            */
            assertion = NotMonetary(report.CTR78, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.Operating.Lease.Amount[@contextRef='" + report.RPJFOREIGNId + "']",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429880"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR78", Value = GetValueOrEmpty(report.CTR78) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429881
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429881()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429881
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR79] <> NULL AND [CTR79] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR79, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR79 = RP:Expense.Operating.Rent.Amount
            */
            assertion = NotMonetary(report.CTR79, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.Operating.Rent.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429881"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR79", Value = GetValueOrEmpty(report.CTR79) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429882
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429882()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429882
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR80] <> NULL AND [CTR80] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR80, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR80 = RP.JAUS:Expense.Operating.Interest.Amount
            */
            assertion = NotMonetary(report.CTR80, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.Operating.Interest.Amount[@contextRef='" + report.RPJAUSId + "']",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429882"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR80", Value = GetValueOrEmpty(report.CTR80) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429883
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429883()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429883
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR81] <> NULL AND [CTR81] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR81, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR81 = RP.JFOREIGN:Expense.Operating.Interest.Amount
            */
            assertion = NotMonetary(report.CTR81, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.Operating.Interest.Amount[@contextRef='" + report.RPJFOREIGNId + "']",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429883"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR81", Value = GetValueOrEmpty(report.CTR81) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429886
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429886()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429886
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR84] <> NULL AND [CTR84] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR84, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR84 = RP:Expense.DepreciationAndAmortisation.Amount
            */
            assertion = NotMonetary(report.CTR84, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.DepreciationAndAmortisation.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429886"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR84", Value = GetValueOrEmpty(report.CTR84) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429887
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429887()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429887
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR85] <> NULL AND [CTR85] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR85, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR85 = RP:Expense.Operating.MotorVehicle.Amount
            */
            assertion = NotMonetary(report.CTR85, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.Operating.MotorVehicle.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429887"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR85", Value = GetValueOrEmpty(report.CTR85) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429888
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429888()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429888
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR86] <> NULL AND [CTR86] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR86, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR86 = RP:Expense.Operating.RepairsAndMaintenance.Amount
            */
            assertion = NotMonetary(report.CTR86, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.Operating.RepairsAndMaintenance.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429888"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR86", Value = GetValueOrEmpty(report.CTR86) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429889
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429889()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429889
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR87] <> NULL AND [CTR87] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR87, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR87 = RP:Expense.AssetRevaluationsUnrealisedLosses.Amount
            */
            assertion = NotMonetary(report.CTR87, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.AssetRevaluationsUnrealisedLosses.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429889"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR87", Value = GetValueOrEmpty(report.CTR87) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429890
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429890()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429890
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR88] <> NULL AND [CTR88] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR88, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR88 = RP:Expense.Other.Amount
            */
            assertion = NotMonetary(report.CTR88, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.Other.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429890"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR88", Value = GetValueOrEmpty(report.CTR88) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429891
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429891()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429891
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR91] <> NULL AND [CTR91] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR91, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR91 = RP:Income.CapitalGainsNet.Amount
            */
            assertion = NotMonetary(report.CTR91, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.CapitalGainsNet.Amount[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429891"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR91", Value = GetValueOrEmpty(report.CTR91) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429892
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429892()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429892
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR92] <> NULL AND [CTR92] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR92, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR92 = RP:Expense.NonDeductibleExemptIncome.Amount
            */
            assertion = NotMonetary(report.CTR92, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.NonDeductibleExemptIncome.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429892"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR92", Value = GetValueOrEmpty(report.CTR92) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429893
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429893()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429893
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR93] <> NULL AND [CTR93] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR93, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR93 = RP:IncomeTax.FrankingCredits.ReceivedFromAustralianCompanies.Amount
            */
            assertion = NotMonetary(report.CTR93, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:IncomeTax.FrankingCredits.ReceivedFromAustralianCompanies.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429893"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR93", Value = GetValueOrEmpty(report.CTR93) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429894
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429894()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429894
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR94] <> NULL AND [CTR94] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR94, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR94 = RP:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
            */
            assertion = NotMonetary(report.CTR94, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429894"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR94", Value = GetValueOrEmpty(report.CTR94) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429895
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429895()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429895
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR95] <> NULL AND [CTR95] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR95, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR95 = RP:Income.AssessableOther.Amount
            */
            assertion = NotMonetary(report.CTR95, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.AssessableOther.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429895"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR95", Value = GetValueOrEmpty(report.CTR95) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429896
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429896()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429896
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR96] <> NULL AND [CTR96] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR96, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR96 = RP:Expense.NonDeductible.Amount
            */
            assertion = NotMonetary(report.CTR96, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.NonDeductible.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429896"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR96", Value = GetValueOrEmpty(report.CTR96) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429897
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429897()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429897
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR97] <> NULL AND [CTR97] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR97, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR97 = RP:TaxConcession.ResearchAndDevelopment.WriteBackOfAccountingExpenditure.Amount
            */
            assertion = NotMonetary(report.CTR97, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:TaxConcession.ResearchAndDevelopment.WriteBackOfAccountingExpenditure.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429897"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR97", Value = GetValueOrEmpty(report.CTR97) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429898
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429898()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429898
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR98] <> NULL AND [CTR98] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR98, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR98 = RP:Expense.FlowOnDividendDeduction.Amount
            */
            assertion = NotMonetary(report.CTR98, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.FlowOnDividendDeduction.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429898"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR98", Value = GetValueOrEmpty(report.CTR98) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429899
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429899()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429899
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR100] <> NULL AND [CTR100] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR100, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR100 = RP:Expense.ForestryManagedInvestmentSchemeDeduction.Amount
            */
            assertion = NotMonetary(report.CTR100, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.ForestryManagedInvestmentSchemeDeduction.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429899"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR100", Value = GetValueOrEmpty(report.CTR100) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429900
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429900()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429900
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR101] <> NULL AND [CTR101] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR101, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR101 = RP:Expense.CapitalExpenditureImmediateDeduction.Amount
            */
            assertion = NotMonetary(report.CTR101, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.CapitalExpenditureImmediateDeduction.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429900"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR101", Value = GetValueOrEmpty(report.CTR101) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429901
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429901()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429901
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR102] <> NULL AND [CTR102] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR102, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR102 = RP:Expense.ProjectPoolAllowableDeduction.Amount
            */
            assertion = NotMonetary(report.CTR102, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.ProjectPoolAllowableDeduction.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429901"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR102", Value = GetValueOrEmpty(report.CTR102) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429902
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429902()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429902
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR103] <> NULL AND [CTR103] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR103, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR103 = RP:Expense.CapitalWorksDeduction.Amount
            */
            assertion = NotMonetary(report.CTR103, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.CapitalWorksDeduction.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429902"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR103", Value = GetValueOrEmpty(report.CTR103) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429903
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429903()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429903
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR104] <> NULL AND [CTR104] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR104, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR104 = RP:Expense.CapitalExpenditureSpecifiedAllowableDeduction.Amount
            */
            assertion = NotMonetary(report.CTR104, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.CapitalExpenditureSpecifiedAllowableDeduction.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429903"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR104", Value = GetValueOrEmpty(report.CTR104) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429908
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429908()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429908
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR111] <> NULL AND [CTR111] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR111, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR111 = RP.Y0:Expense.LandcareOperationsAcceleratedDepreciation.Amount
            */
            assertion = NotMonetary(report.CTR111, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.LandcareOperationsAcceleratedDepreciation.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429908"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR111", Value = GetValueOrEmpty(report.CTR111) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429909
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429909()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429909
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR112] <> NULL AND [CTR112] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR112, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR112 = RP:Expense.EnvironmentalProtectionDeduction.Amount
            */
            assertion = NotMonetary(report.CTR112, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.EnvironmentalProtectionDeduction.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429909"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR112", Value = GetValueOrEmpty(report.CTR112) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429911
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429911()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429911
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR114] <> NULL AND [CTR114] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR114, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR114 = RP:Income.Exempt.Amount
            */
            assertion = NotMonetary(report.CTR114, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.Exempt.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429911"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR114", Value = GetValueOrEmpty(report.CTR114) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429912
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429912()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429912
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR115] <> NULL AND [CTR115] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR115, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR115 = RP:Income.NotincludedInAssessableOther.Amount
            */
            assertion = NotMonetary(report.CTR115, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.NotincludedInAssessableOther.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429912"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR115", Value = GetValueOrEmpty(report.CTR115) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429913
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429913()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429913
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR116] <> NULL AND [CTR116] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR116, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR116 = RP:Expense.DeductibleOther.Amount
            */
            assertion = NotMonetary(report.CTR116, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.DeductibleOther.Amount[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429913"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR116", Value = GetValueOrEmpty(report.CTR116) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429914
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429914()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429914
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR117] <> NULL AND [CTR117] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR117, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR117 = RP:IncomeTax.Deduction.TaxLossesDeducted.Amount
            */
            assertion = NotMonetary(report.CTR117, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.TaxLossesDeducted.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429914"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR117", Value = GetValueOrEmpty(report.CTR117) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429915
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429915()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429915
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR118] <> NULL AND [CTR118] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR118, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR118 = RP:IncomeTax.TaxLossesTransferredIn.Amount
            */
            assertion = NotMonetary(report.CTR118, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:IncomeTax.TaxLossesTransferredIn.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429915"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR118", Value = GetValueOrEmpty(report.CTR118) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429917
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429917()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429917
            Monetary field may be signed, maximum of 11 chars, no decimals
    
            Legacy Rule Format:
            IF [CTR120] <> NULL AND [CTR120] <> MONETARY(S,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR120, 'S', 11, 0)
    
            Data Elements:
    
            ^CTR120 = RP:Income.Taxable.Amount
            */
            assertion = NotMonetary(report.CTR120, @"S", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.Taxable.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429917"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR120", Value = GetValueOrEmpty(report.CTR120) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429921
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429921()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429921
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR125] <> NULL AND [CTR125] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR125, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR125 = RP.Closing(Instant):Assets.Inventories.Total.Amount
            */
            assertion = NotMonetary(report.CTR125, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Assets.Inventories.Total.Amount[@contextRef='" + report.RPClosingInstantId + "']",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429921"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR125", Value = GetValueOrEmpty(report.CTR125) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429922
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429922()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429922
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR128] <> NULL AND [CTR128] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR128, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR128 = RP.Closing(Instant):Assets.TradeandReceivablesOther.Amount
            */
            assertion = NotMonetary(report.CTR128, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Assets.TradeandReceivablesOther.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429922"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR128", Value = GetValueOrEmpty(report.CTR128) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429928
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429928()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429928
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR135] <> NULL AND [CTR135] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR135, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR135 = RP:Liabilities.CommercialDebtForgiveness.Amount
            */
            assertion = NotMonetary(report.CTR135, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Liabilities.CommercialDebtForgiveness.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429928"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR135", Value = GetValueOrEmpty(report.CTR135) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429930
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429930()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429930
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR137] <> NULL AND [CTR137] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR137, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR137 = RP:Equity.Dividends.FrankedPaid.Amount
            */
            assertion = NotMonetary(report.CTR137, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Equity.Dividends.FrankedPaid.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429930"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR137", Value = GetValueOrEmpty(report.CTR137) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429931
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429931()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429931
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR138] <> NULL AND [CTR138] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR138, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR138 = RP:Equity.Dividends.UnfrankedPaid.Amount
            */
            assertion = NotMonetary(report.CTR138, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Equity.Dividends.UnfrankedPaid.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429931"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR138", Value = GetValueOrEmpty(report.CTR138) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429935
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429935()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429935
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR142] <> NULL AND [CTR142] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR142, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR142 = RP:IncomeTax.FrankingCredits.ExcessFrankingOffsets.Amount
            */
            assertion = NotMonetary(report.CTR142, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:IncomeTax.FrankingCredits.ExcessFrankingOffsets.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429935"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR142", Value = GetValueOrEmpty(report.CTR142) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429938
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429938()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429938
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR147] <> NULL AND [CTR147] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR147, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR147 = RP:Expense.DepreciatingAssets.IntangibleFirstDeducted.Amount
            */
            assertion = NotMonetary(report.CTR147, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.DepreciatingAssets.IntangibleFirstDeducted.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429938"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR147", Value = GetValueOrEmpty(report.CTR147) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429940
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429940()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429940
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR149] <> NULL AND [CTR149] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR149, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR149 = RP:Assets.DepreciatingAssets.IntangibleTerminationValue.Amount
            */
            assertion = NotMonetary(report.CTR149, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Assets.DepreciatingAssets.IntangibleTerminationValue.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429940"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR149", Value = GetValueOrEmpty(report.CTR149) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429941
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429941()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429941
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR150] <> NULL AND [CTR150] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR150, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR150 = RP:Assets.DepreciatingAssets.OtherTerminationValue.Amount
            */
            assertion = NotMonetary(report.CTR150, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Assets.DepreciatingAssets.OtherTerminationValue.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429941"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR150", Value = GetValueOrEmpty(report.CTR150) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429942
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429942()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429942
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR151] <> NULL AND [CTR151] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR151, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR151 = RP:Remuneration.WagesAndSalaries.Total.Amount
            */
            assertion = NotMonetary(report.CTR151, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Remuneration.WagesAndSalaries.Total.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429942"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR151", Value = GetValueOrEmpty(report.CTR151) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429943
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429943()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429943
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR153] <> NULL AND [CTR153] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR153, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR153 = RP:Remuneration.PaymentToRelatedPartiesGross.Amount
            */
            assertion = NotMonetary(report.CTR153, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Remuneration.PaymentToRelatedPartiesGross.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429943"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR153", Value = GetValueOrEmpty(report.CTR153) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429944
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429944()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429944
            Monetary field may be signed, maximum of 11 chars, no decimals
    
            Legacy Rule Format:
            IF [CTR154] <> NULL AND [CTR154] <> MONETARY(S,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR154, 'S', 11, 0)
    
            Data Elements:
    
            ^CTR154 = RP:Income.InternationalDealings.Net.Amount
            */
            assertion = NotMonetary(report.CTR154, @"S", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.InternationalDealings.Net.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429944"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR154", Value = GetValueOrEmpty(report.CTR154) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429946
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429946()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429946
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR156] <> NULL AND [CTR156] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR156, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR156 = RP.CFCLC:Income.InternationalDealings.Attributed.Amount
            */
            assertion = NotMonetary(report.CTR156, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.InternationalDealings.Attributed.Amount[@contextRef='" + report.RPCFCLCId + "']",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429946"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR156", Value = GetValueOrEmpty(report.CTR156) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429948
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429948()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429948
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR158] <> NULL AND [CTR158] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR158, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR158 = RP.CFCUC:Income.InternationalDealings.Attributed.Amount
            */
            assertion = NotMonetary(report.CTR158, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.InternationalDealings.Attributed.Amount[@contextRef='" + report.RPCFCUCId + "']",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429948"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR158", Value = GetValueOrEmpty(report.CTR158) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429949
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429949()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429949
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR159] <> NULL AND [CTR159] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR159, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR159 = RP:InternationalDealings.TransferorTrustAttributedForeignIncome.Amount
            */
            assertion = NotMonetary(report.CTR159, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:InternationalDealings.TransferorTrustAttributedForeignIncome.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429949"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR159", Value = GetValueOrEmpty(report.CTR159) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429960
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429960()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429960
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR175] <> NULL AND [CTR175] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR175, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR175 = RP:Tax.Losses.CarriedForward.Total.Amount
            */
            assertion = NotMonetary(report.CTR175, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.CarriedForward.Total.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429960"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR175", Value = GetValueOrEmpty(report.CTR175) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429961
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429961()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429961
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR176] <> NULL AND [CTR176] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR176, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR176 = RP(Instant):Capital.Losses.CarriedForward.Net.Amount
            */
            assertion = NotMonetary(report.CTR176, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.CarriedForward.Net.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429961"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR176", Value = GetValueOrEmpty(report.CTR176) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429962
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429962()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429962
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR179] <> NULL AND [CTR179] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR179, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR179 = RP.CMPLY:Income.Insurance.Taxable.Amount
            */
            assertion = NotMonetary(report.CTR179, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.Insurance.Taxable.Amount[@contextRef='" + report.RPCMPLYId + "']",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429962"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR179", Value = GetValueOrEmpty(report.CTR179) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429963
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429963()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429963
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR180] <> NULL AND [CTR180] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR180, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR180 = RP.CMPLY:Income.CapitalGainsNet.Amount
            */
            assertion = NotMonetary(report.CTR180, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.CapitalGainsNet.Amount[@contextRef='" + report.RPCMPLYId + "']",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429963"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR180", Value = GetValueOrEmpty(report.CTR180) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429964
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429964()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429964
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR181] <> NULL AND [CTR181] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR181, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR181 = RP.ORD:Income.CapitalGainsNet.Amount
            */
            assertion = NotMonetary(report.CTR181, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.CapitalGainsNet.Amount[@contextRef='" + report.RPORDId + "']",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429964"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR181", Value = GetValueOrEmpty(report.CTR181) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429965
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429965()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429965
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR182] <> NULL AND [CTR182] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR182, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR182 = RP:Income.Insurance.Taxable.Amount
            */
            assertion = NotMonetary(report.CTR182, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.Insurance.Taxable.Amount[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429965"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR182", Value = GetValueOrEmpty(report.CTR182) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429966
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429966()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429966
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR183] <> NULL AND [CTR183] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR183, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR183 = RP:Income.Insurance.FeesAndCharges.Amount
            */
            assertion = NotMonetary(report.CTR183, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.Insurance.FeesAndCharges.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429966"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR183", Value = GetValueOrEmpty(report.CTR183) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429967
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429967()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429967
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR184] <> NULL AND [CTR184] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR184, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR184 = RP:Income.PooledDevelopmentFundsSmallToMediumEnterprises.Amount
            */
            assertion = NotMonetary(report.CTR184, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.PooledDevelopmentFundsSmallToMediumEnterprises.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429967"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR184", Value = GetValueOrEmpty(report.CTR184) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429968
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429968()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429968
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR185] <> NULL AND [CTR185] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR185, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR185 = RP:Income.PooledDevelopmentFundsUnregulatedInvestment.Amount
            */
            assertion = NotMonetary(report.CTR185, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.PooledDevelopmentFundsUnregulatedInvestment.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429968"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR185", Value = GetValueOrEmpty(report.CTR185) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429973
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429973()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429973
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR190] <> NULL AND [CTR190] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR190, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR190 = RP:Income.RetirementSavingsAccountsTaxableNet.Amount
            */
            assertion = NotMonetary(report.CTR190, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.RetirementSavingsAccountsTaxableNet.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429973"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR190", Value = GetValueOrEmpty(report.CTR190) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429975
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429975()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429975
            Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters
    
            Legacy Rule Format:
            IF [CTR199] <> NULL AND [CTR199] <> MONETARY(U,13,2)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR199, 'U', 13, 2)
    
            Data Elements:
    
            ^CTR199 = RP:IncomeTax.TaxableGross.Amount
            */
            assertion = NotMonetary(report.CTR199, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:IncomeTax.TaxableGross.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429975"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR199", Value = GetValueOrEmpty(report.CTR199) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429977
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429977()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429977
            Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters
    
            Legacy Rule Format:
            IF [CTR201] <> NULL AND [CTR201] <> MONETARY(U,13,2)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR201, 'U', 13, 2)
    
            Data Elements:
    
            ^CTR201 = RP:Income.InternationalDealings.TaxOffset.Amount
            */
            assertion = NotMonetary(report.CTR201, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.InternationalDealings.TaxOffset.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429977"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR201", Value = GetValueOrEmpty(report.CTR201) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429978
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429978()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429978
            Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters
    
            Legacy Rule Format:
            IF [CTR202] <> NULL AND [CTR202] <> MONETARY(U,13,2)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR202, 'U', 13, 2)
    
            Data Elements:
    
            ^CTR202 = RP.Y0-1:TaxOffsetClaim.FrankingDeficit.Amount
            */
            assertion = NotMonetary(report.CTR202, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:TaxOffsetClaim.FrankingDeficit.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429978"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR202", Value = GetValueOrEmpty(report.CTR202) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429979
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429979()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429979
            Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters
    
            Legacy Rule Format:
            IF [CTR203] <> NULL AND [CTR203] <> MONETARY(U,13,2)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR203, 'U', 13, 2)
    
            Data Elements:
    
            ^CTR203 = RP:IncomeTax.Payable.Amount
            */
            assertion = NotMonetary(report.CTR203, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Payable.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429979"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR203", Value = GetValueOrEmpty(report.CTR203) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429980
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429980()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429980
            Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters
    
            Legacy Rule Format:
            IF [CTR204] <> NULL AND [CTR204] <> MONETARY(U,13,2)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR204, 'U', 13, 2)
    
            Data Elements:
    
            ^CTR204 = RP:InternationalDealings.AAMInterestSection102.Amount
            */
            assertion = NotMonetary(report.CTR204, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:InternationalDealings.AAMInterestSection102.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429980"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR204", Value = GetValueOrEmpty(report.CTR204) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429981
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429981()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429981
            Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters
    
            Legacy Rule Format:
            IF [CTR205] <> NULL AND [CTR205] <> MONETARY(U,13,2)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR205, 'U', 13, 2)
    
            Data Elements:
    
            ^CTR205 = RP:IncomeTax.LiabilityInstalmentsTotal.Amount
            */
            assertion = NotMonetary(report.CTR205, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:IncomeTax.LiabilityInstalmentsTotal.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429981"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR205", Value = GetValueOrEmpty(report.CTR205) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429982
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429982()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429982
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF rvctc2.xx.xx:IncomeTax.PayAsYouGoWithholding.TaxWithheld.Amount <> NULL AND rvctc2.xx.xx:IncomeTax.PayAsYouGoWithholding.TaxWithheld.Amount <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR207, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR207 = RP:ITPAYGWithholdingDetails:IncomeTax.PayAsYouGoWithholding.TaxWithheld.Amount WHERE [IncomeTax.PayAsYouGoWithholding.PaymentType.Code='DFRW']
            */
            assertion = NotMonetary(report.CTR207, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:ITPAYGWithholdingDetails" + OccurrenceIndex(report.ItpaygWithholdingDetailsCollection_DFRW_RPOccurrenceIndex) + "/tns:IncomeTax.PayAsYouGoWithholding.TaxWithheld.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429982"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR207", Value = GetValueOrEmpty(report.CTR207) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429982
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429982_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429982
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF rvctc2.xx.xx:IncomeTax.PayAsYouGoWithholding.TaxWithheld.Amount <> NULL AND rvctc2.xx.xx:IncomeTax.PayAsYouGoWithholding.TaxWithheld.Amount <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR208, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR208 = RP:ITPAYGWithholdingDetails:IncomeTax.PayAsYouGoWithholding.TaxWithheld.Amount WHERE [IncomeTax.PayAsYouGoWithholding.PaymentType.Code='DNOABN']
            */
            assertion = NotMonetary(report.CTR208, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:ITPAYGWithholdingDetails" + OccurrenceIndex(report.ItpaygWithholdingDetailsCollection_DNOABN_RPOccurrenceIndex) + "/tns:IncomeTax.PayAsYouGoWithholding.TaxWithheld.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429982"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR208", Value = GetValueOrEmpty(report.CTR208) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429984
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429984()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429984
            Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters
    
            Legacy Rule Format:
            IF [CTR209] <> NULL AND [CTR209] <> MONETARY(U,13,2)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR209, 'U', 13, 2)
    
            Data Elements:
    
            ^CTR209 = RP:IncomeTax.PayAsYouGoWithholding.TaxWithheldFromInvestmentIncomeAndInterest.Amount
            */
            assertion = NotMonetary(report.CTR209, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.TaxWithheldFromInvestmentIncomeAndInterest.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429984"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR209", Value = GetValueOrEmpty(report.CTR209) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429985
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429985()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429985
            Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters
    
            Legacy Rule Format:
            IF [CTR210] <> NULL AND [CTR210] <> MONETARY(U,13,2)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR210, 'U', 13, 2)
    
            Data Elements:
    
            ^CTR210 = RP:TaxConcession.ResearchAndDevelopment.TaxOffsetRefundable.Amount
            */
            assertion = NotMonetary(report.CTR210, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:TaxConcession.ResearchAndDevelopment.TaxOffsetRefundable.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429985"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR210", Value = GetValueOrEmpty(report.CTR210) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429986
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429986()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429986
            Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters
    
            Legacy Rule Format:
            IF [CTR211] <> NULL AND [CTR211] <> MONETARY(U,13,2)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR211, 'U', 13, 2)
    
            Data Elements:
    
            ^CTR211 = RP:IncomeTax.PayAsYouGoWithholding.CreditsRefundableOther.Amount
            */
            assertion = NotMonetary(report.CTR211, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditsRefundableOther.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429986"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR211", Value = GetValueOrEmpty(report.CTR211) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429987
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429987()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429987
            Field value supplied exceeds maximum field length
    
            Legacy Rule Format:
            IF [CTR213] <> NULLORBLANK AND (LENGTH([CTR213]) > 12)   
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^CTR213) > 12
    
            Data Elements:
    
            ^CTR213 = INT:Identifiers.TaxAgentClientReference.Text
            */
            assertion = (Length(report.CTR213) > 12);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field value supplied exceeds maximum field length",
                    Location = "/xbrli:xbrl/tns:Identifiers.TaxAgentClientReference.Text",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429987"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR213", Value = GetValueOrEmpty(report.CTR213) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429989
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429989()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429989
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR99] <> NULL AND [CTR99] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR99, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR99 = RP:Expense.DepreciationAllowableDeduction.Amount
            */
            assertion = NotMonetary(report.CTR99, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.DepreciationAllowableDeduction.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429989"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR99", Value = GetValueOrEmpty(report.CTR99) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429991
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429991()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429991
            Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters
    
            Legacy Rule Format:
            IF [CTR246] <> NULL AND [CTR246] <> MONETARY(U,13,2)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR246, 'U', 13, 2)
    
            Data Elements:
    
            ^CTR246 = RP:TaxOffsetClaim.NationalRentalAffordabilitySchemeEntitlement.Amount
            */
            assertion = NotMonetary(report.CTR246, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:TaxOffsetClaim.NationalRentalAffordabilitySchemeEntitlement.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429991"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR246", Value = GetValueOrEmpty(report.CTR246) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429997
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429997()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429997
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR255] <> NULL AND [CTR255] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR255, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR255 = RP.TOFA:Income.OtherTotal.Amount
            */
            assertion = NotMonetary(report.CTR255, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.OtherTotal.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429997"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR255", Value = GetValueOrEmpty(report.CTR255) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.429998
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR429998()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.429998
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR256] <> NULL AND [CTR256] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR256, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR256 = RP.TOFA:Expense.DeductibleOther.Amount
            */
            assertion = NotMonetary(report.CTR256, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.DeductibleOther.Amount[@contextRef='" + report.RPTOFAId + "']",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.429998"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR256", Value = GetValueOrEmpty(report.CTR256) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.430000
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR430000()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.430000
            Invalid context. The number of Reporting party:LowCost contexts must not exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.LowCost)) > 1 
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.LowCost
            */
            assertion = (Count(report.RPLowCostCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.430000",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Reporting party:LowCost contexts must not exceed 1",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPLowCostLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.430000"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.430001
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR430001()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.430001
            Invalid context. The number of Reporting party:GeneralPool contexts must not exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.GeneralPool)) > 1 
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.GeneralPool
            */
            assertion = (Count(report.RPGeneralPoolCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.430001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Reporting party:GeneralPool contexts must not exceed 1",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPGeneralPoolLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.430001"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.430004
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR430004()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.430004
            Invalid context. The number of Reporting party:ProspectiveRules contexts must not exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.ProspectiveRules)) > 1 
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.ProspectiveRules
            */
            assertion = (Count(report.RPProspectiveRulesCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.430004",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Reporting party:ProspectiveRules contexts must not exceed 1",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPProspectiveRulesLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.430004"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.430007
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR430007()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.430007
            Invalid context. The number of Reporting party:Carried forward to next year contexts must not exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.Y0Plus1)) > 1 
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.Y0Plus1
            */
            assertion = (Count(report.RPY0Plus1Count) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.430007",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Reporting party:Carried forward to next year contexts must not exceed 1",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0Plus1LastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.430007"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.430008
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR430008()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.430008
            Invalid context. The number of Reporting party:Carried forward from previous year contexts must not exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.YPrevious)) > 1 
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.YPrevious
            */
            assertion = (Count(report.RPYPreviousCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.430008",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Reporting party:Carried forward from previous year contexts must not exceed 1",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPYPreviousLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.430008"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.430013
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR430013()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.430013
            Specified Income year must not be less than 1995 or greater than the current reporting year
    
            Legacy Rule Format:
            IF ([CTR55] <> NULL) AND (([CTR55] < "1995") OR ([CTR55] > [CTR2]))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR55 <> NULL) AND ((^CTR55 < 1995) OR (^CTR55 > ^CTR2))
    
            Data Elements:
    
            ^CTR55 = RP:Elections.InterposedEntityElectionStatus.Year
    
            ^CTR2 = RP:Report.TargetFinancial.Year
            */
            assertion = (report.CTR55 != null && (report.CTR55.GetValueOrDefault() < 1995 || report.CTR55.GetValueOrDefault() > report.CTR2.GetValueOrDefault()));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.430013",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Specified Income year must not be less than 1995 or greater than the current reporting year",
                    Location = "/xbrli:xbrl/tns:Elections.InterposedEntityElectionStatus.Year",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.430013"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR55", Value = GetValueOrEmpty(report.CTR55) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR2", Value = GetValueOrEmpty(report.CTR2) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.430022
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR430022()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.430022
            Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters
    
            Legacy Rule Format:
            IF [CTR121] <> NULL AND [CTR121] <> NUMERIC(U,13,5)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotNumeric(^CTR121, 'U', 13, 5)
    
            Data Elements:
    
            ^CTR121 = RP.Closing(Instant):Miscellaneous.CurrencyTranslation.Rate
            */
            assertion = NotNumeric(report.CTR121, @"U", 13, 5);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400012",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a numeric field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Miscellaneous.CurrencyTranslation.Rate",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.430022"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR121", Value = GetValueOrEmpty(report.CTR121) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.430024
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR430024()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.430024
            Mandatory field not supplied
    
            Legacy Rule Format:
            IF [CTR177] = NULL
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR177 = NULL
    
            Data Elements:
    
            ^CTR177 = RP:Income.PersonalServicesIncomeIncluded.Indicator
            */
            assertion = (report.CTR177 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Mandatory field not supplied",
                    Location = "/xbrli:xbrl/tns:Income.PersonalServicesIncomeIncluded.Indicator",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.430024"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR177", Value = GetValueOrEmpty(report.CTR177) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.430025
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR430025()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.430025
            Non-refundable R&D tax offset to be utilised in the current year cannot be greater than the total sum of Non-refundable R&D tax offset and Non-refundable R&D tax offset carried forward from previous year
    
            Legacy Rule Format:
            IF [CTR300] <> NULL AND [CTR300] > ([CTR328] + [CTR353])
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR300 <> NULL AND ^CTR300 > (^CTR328 + ^CTR353)
    
            Data Elements:
    
            ^CTR300 = RP:TaxConcession.ResearchAndDevelopment.TaxOffsetNonRefundableUtilised.Amount
    
            ^CTR328 = RP:TaxConcession.ResearchAndDevelopment.TaxOffsetNonRefundable.Amount
    
            ^CTR353 = RP.YPrevious:TaxConcession.ResearchAndDevelopment.TaxOffsetNonRefundable.Amount
            */
            assertion = (report.CTR300 != null && report.CTR300.GetValueOrDefault() > report.CTR328.GetValueOrDefault() + report.CTR353.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.430025",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-refundable R&D tax offset to be utilised amount incorrect",
                    LongDescription = @"Non-refundable R&D tax offset to be utilised in the current year cannot be greater than the total sum of Non-refundable R&D tax offset and Non-refundable R&D tax offset carried forward from previous year",
                    Location = "/xbrli:xbrl/tns:TaxConcession.ResearchAndDevelopment.TaxOffsetNonRefundableUtilised.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.430025"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR300", Value = GetValueOrEmpty(report.CTR300) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR328", Value = GetValueOrEmpty(report.CTR328) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR353", Value = GetValueOrEmpty(report.CTR353) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.430026
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR430026()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.430026
            Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character
    
            Legacy Rule Format:
            IF (gfagc.xx.xx:TaxConcession.ResearchAndDevelopment.TaxOffsetNonRefundable.Amount <> NULL) AND (gfagc.xx.xx:TaxConcession.ResearchAndDevelopment.TaxOffsetNonRefundable.Amount  <> MONETARY(U,13,2))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR328, 'U', 13, 2)
    
            Data Elements:
    
            ^CTR328 = RP:TaxConcession.ResearchAndDevelopment.TaxOffsetNonRefundable.Amount
            */
            assertion = NotMonetary(report.CTR328, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:TaxConcession.ResearchAndDevelopment.TaxOffsetNonRefundable.Amount[@contextRef='" + report.RPId + "']",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.430026"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR328", Value = GetValueOrEmpty(report.CTR328) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.430026
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR430026_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.430026
            Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character
    
            Legacy Rule Format:
            IF (gfagc.xx.xx:TaxConcession.ResearchAndDevelopment.TaxOffsetNonRefundable.Amount <> NULL) AND (gfagc.xx.xx:TaxConcession.ResearchAndDevelopment.TaxOffsetNonRefundable.Amount  <> MONETARY(U,13,2))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR353, 'U', 13, 2)
    
            Data Elements:
    
            ^CTR353 = RP.YPrevious:TaxConcession.ResearchAndDevelopment.TaxOffsetNonRefundable.Amount
            */
            assertion = NotMonetary(report.CTR353, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:TaxConcession.ResearchAndDevelopment.TaxOffsetNonRefundable.Amount[@contextRef='" + report.RPYPreviousId + "']",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.430026"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR353", Value = GetValueOrEmpty(report.CTR353) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.430026
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR430026_2()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.430026
            Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character
    
            Legacy Rule Format:
            IF (gfagc.xx.xx:TaxConcession.ResearchAndDevelopment.TaxOffsetNonRefundable.Amount <> NULL) AND (gfagc.xx.xx:TaxConcession.ResearchAndDevelopment.TaxOffsetNonRefundable.Amount  <> MONETARY(U,13,2))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR302, 'U', 13, 2)
    
            Data Elements:
    
            ^CTR302 = RP.Y0Plus1:TaxConcession.ResearchAndDevelopment.TaxOffsetNonRefundable.Amount
            */
            assertion = NotMonetary(report.CTR302, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:TaxConcession.ResearchAndDevelopment.TaxOffsetNonRefundable.Amount[@contextRef='" + report.RPY0Plus1Id + "']",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.430026"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR302", Value = GetValueOrEmpty(report.CTR302) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.430027
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR430027()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.430027
            Non-refundable R&D tax offset to be carried forward should be equal to Non-refundable R&D tax offset plus Non-refundable R&D tax offset carried forward from previous year minus Non-refundable R&D tax offset to be utilised in the current year
    
            Legacy Rule Format:
            IF [CTR302] <> NULL AND [CTR302] <> ([CTR328] + [CTR353] - [CTR300])
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR302 <> NULL AND ^CTR302 <> (^CTR328 + ^CTR353 - ^CTR300)
    
            Data Elements:
    
            ^CTR302 = RP.Y0Plus1:TaxConcession.ResearchAndDevelopment.TaxOffsetNonRefundable.Amount
    
            ^CTR328 = RP:TaxConcession.ResearchAndDevelopment.TaxOffsetNonRefundable.Amount
    
            ^CTR353 = RP.YPrevious:TaxConcession.ResearchAndDevelopment.TaxOffsetNonRefundable.Amount
    
            ^CTR300 = RP:TaxConcession.ResearchAndDevelopment.TaxOffsetNonRefundableUtilised.Amount
            */
            assertion = (report.CTR302 != null && report.CTR302.GetValueOrDefault() != report.CTR328.GetValueOrDefault() + report.CTR353.GetValueOrDefault() - report.CTR300.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.430027",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-refundable R&D tax offset to be carried forward amount is incorrect",
                    LongDescription = @"Non-refundable R&D tax offset to be carried forward should be equal to Non-refundable R&D tax offset plus Non-refundable R&D tax offset carried forward from previous year minus Non-refundable R&D tax offset to be utilised in the current year",
                    Location = "/xbrli:xbrl/tns:TaxConcession.ResearchAndDevelopment.TaxOffsetNonRefundable.Amount[@contextRef='" + report.RPY0Plus1Id + "']",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.430027"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR302", Value = GetValueOrEmpty(report.CTR302) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR328", Value = GetValueOrEmpty(report.CTR328) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR353", Value = GetValueOrEmpty(report.CTR353) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR300", Value = GetValueOrEmpty(report.CTR300) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.430028
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR430028(string CTR330, int itemIndex1)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.430028
                    Invalid exemption or rollover code
    
                    Legacy Rule Format:
                    IF [CTR330] <> NULL AND [CTR330] <> SET (DOMAIN(CTRCGTExemptionOrRolloverCodes))
                      RETURN VALIDATION MESSAGE
                    ENDIF
        
                    Technical Business Rule Format:
                    ^CTR330 <> NULL AND NotInDomain(^CTR330, 'CTRCGTExemptionOrRolloverCodes')
            
                    Data Elements:
            
                    ^CTR330 = RP:CapitalGainsTax.ExemptionOrRolloverApplied.Code
                    */
            assertion = (CTR330 != null && !(IsMatch(CTR330, @"^(A - Small business active asset reduction|B - Small business retirement exemption|C - Small business roll-over|D - Small business 15 year exemption|E - Foreign resident CGT exemption|F - Scrip for scrip roll-over|G - Inter-company roll-over|H - Demerger exemption|J - Capital gains disregarded as a result of the sale of a pre-CGT asset|K - Disposal or creation of assets in a wholly-owned company|L - Replacement asset roll-overs|M - Exchange of shares or units|N - Exchange of rights or options|O - Exchange of shares in one company for shares in another company|P - Exchange of units in a unit trust for shares in a company|R - Demerger roll-over|S - Same asset roll-overs|T - Small Business Restructure Roll-over|U - Early stage investor|V - Venture capital investment|X - Other exemptions and roll-overs)$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.430028",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid exemption or rollover code",
                    Location = "/xbrli:xbrl/tns:CapitalGainsTax.ExemptionOrRolloverApplied.Code[" + itemIndex1 + "]",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.430028"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR330", Value = GetValueOrEmpty(CTR330) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.430029
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR430029()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.430029
            Exemption or rollover code must be unique
    
            Legacy Rule Format:
            IF [CTR330] <> NULL AND [CTR330] = ANY OTHER OCCURRENCE OF [CTR330]
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            HasDuplicateValues(^CTR330)
    
            Data Elements:
    
            ^CTR330 = RP:CapitalGainsTax.ExemptionOrRolloverApplied.Code
            */
            assertion = HasDuplicateValues(report.CTR330Collection);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.430029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Exemption or rollover code must be unique",
                    Location = "/xbrli:xbrl/tns:CapitalGainsTax.ExemptionOrRolloverApplied.Code" + OccurrenceIndex(DuplicateValueIndex(report.CTR330Collection) + 1) + "",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.430029"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.430032
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR430032()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.430032
            Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character
    
            Legacy Rule Format:
            IF [CTR336] <> NULL AND [CTR336] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR336, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR336 = RP:Expense.DepreciatingAssets.ProjectPoolAllocationsForMiningAndTransportCapitalExpenditure.Amount
            */
            assertion = NotMonetary(report.CTR336, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.DepreciatingAssets.ProjectPoolAllocationsForMiningAndTransportCapitalExpenditure.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.430032"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR336", Value = GetValueOrEmpty(report.CTR336) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.430033
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR430033()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.430033
            Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character
    
            Legacy Rule Format:
            IF [CTR337] <> NULL AND [CTR337] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR337, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR337 = RP:Expense.DepreciatingAssets.IntangibleAssetsDeclineInValueOfExplorationOrProspecting.Amount
            */
            assertion = NotMonetary(report.CTR337, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.DepreciatingAssets.IntangibleAssetsDeclineInValueOfExplorationOrProspecting.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.430033"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR337", Value = GetValueOrEmpty(report.CTR337) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.430034
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR430034()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.430034
            Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character
    
            Legacy Rule Format:
            IF [CTR338] <> NULL AND [CTR338] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR338, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR338 = RP:Expense.DepreciatingAssets.OtherAssetsDeclineInValueOfExplorationOrProspecting.Amount
            */
            assertion = NotMonetary(report.CTR338, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.DepreciatingAssets.OtherAssetsDeclineInValueOfExplorationOrProspecting.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.430034"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR338", Value = GetValueOrEmpty(report.CTR338) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.430035
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR430035()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.430035
            Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character
    
            Legacy Rule Format:
            IF (bafpr1.xx.xx:Expense.DeductionsTotal.Amount <> NULL )AND (bafpr1.xx.xx:Expense.DeductionsTotal.Amount <> MONETARY(U,11,0))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR341, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR341 = RP.ProspectiveRules:Expense.DeductionsTotal.Amount
            */
            assertion = NotMonetary(report.CTR341, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.DeductionsTotal.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.430035"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR341", Value = GetValueOrEmpty(report.CTR341) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.430040
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR430040()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.430040
            Monetary field must be unsigned, maximum of 15 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR124] <> NULL AND [CTR124] <> MONETARY(U,15,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR124, 'U', 15, 0)
    
            Data Elements:
    
            ^CTR124 = RP:Expense.Purchases.AndOtherCosts.Amount
            */
            assertion = NotMonetary(report.CTR124, @"U", 15, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.Purchases.AndOtherCosts.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.430040"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR124", Value = GetValueOrEmpty(report.CTR124) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.430041
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR430041()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.430041
            Monetary field must be unsigned, maximum of 15 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR134] <> NULL AND [CTR134] <> MONETARY(U,15,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR134, 'U', 15, 0)
    
            Data Elements:
    
            ^CTR134 = RP:Liabilities.DebtTotal.Amount
            */
            assertion = NotMonetary(report.CTR134, @"U", 15, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Liabilities.DebtTotal.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.430041"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR134", Value = GetValueOrEmpty(report.CTR134) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.430042
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR430042()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.430042
            Monetary field must be unsigned, maximum of 15 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR396] <> NULL AND [CTR396] <> MONETARY(U,15,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR396, 'U', 15, 0)
    
            Data Elements:
    
            ^CTR396 = RP.Opening:Equity.Dividends.FrankingAccountBalance.Amount
            */
            assertion = NotMonetary(report.CTR396, @"U", 15, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Equity.Dividends.FrankingAccountBalance.Amount[@contextRef='" + report.RPOpeningId + "']",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.430042"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR396", Value = GetValueOrEmpty(report.CTR396) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.430043
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR430043()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.430043
            Monetary field must be unsigned, maximum of 15 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR139] <> NULL AND [CTR139] <> MONETARY(U,15,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR139, 'U', 15, 0)
    
            Data Elements:
    
            ^CTR139 = RP.Closing:Equity.Dividends.FrankingAccountBalance.Amount
            */
            assertion = NotMonetary(report.CTR139, @"U", 15, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Equity.Dividends.FrankingAccountBalance.Amount[@contextRef='" + report.RPClosingId + "']",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.430043"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR139", Value = GetValueOrEmpty(report.CTR139) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.430044
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR430044()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.430044
            Monetary field must be unsigned, maximum of 15 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR145] <> NULL AND [CTR145] <> MONETARY(U,15,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR145, 'U', 15, 0)
    
            Data Elements:
    
            ^CTR145 = RP:Assets.LoansAndReceivables.RelatedParties.Amount
            */
            assertion = NotMonetary(report.CTR145, @"U", 15, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Assets.LoansAndReceivables.RelatedParties.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.430044"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR145", Value = GetValueOrEmpty(report.CTR145) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.430045
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR430045()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.430045
            Monetary field must be unsigned, maximum of 15 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR148] <> NULL AND [CTR148] <> MONETARY(U,15,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR148, 'U', 15, 0)
    
            Data Elements:
    
            ^CTR148 = RP:Expense.DepreciatingAssets.OtherFirstDeducted.Amount
            */
            assertion = NotMonetary(report.CTR148, @"U", 15, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.DepreciatingAssets.OtherFirstDeducted.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.430045"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR148", Value = GetValueOrEmpty(report.CTR148) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.430046
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR430046()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.430046
            Monetary field must be unsigned, maximum of 15 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR333] <> NULL AND [CTR333] <> MONETARY(U,15,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR333, 'U', 15, 0)
    
            Data Elements:
    
            ^CTR333 = RP:Assets.DepreciatingAssets.AdjustableValuesTotal.Amount
            */
            assertion = NotMonetary(report.CTR333, @"U", 15, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Assets.DepreciatingAssets.AdjustableValuesTotal.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.430046"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR333", Value = GetValueOrEmpty(report.CTR333) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.430047
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR430047()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.430047
            Monetary field must be unsigned, maximum of 15 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR129] <> NULL AND [CTR129] <> MONETARY(U,15,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR129, 'U', 15, 0)
    
            Data Elements:
    
            ^CTR129 = RP.Closing(Instant):Assets.CurrentTotal.Amount
            */
            assertion = NotMonetary(report.CTR129, @"U", 15, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Assets.CurrentTotal.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.430047"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR129", Value = GetValueOrEmpty(report.CTR129) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.430048
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR430048()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.430048
            Monetary field must be unsigned, maximum of 15 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR130] <> NULL AND [CTR130] <> MONETARY(U,15,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR130, 'U', 15, 0)
    
            Data Elements:
    
            ^CTR130 = RP.Closing(Instant):Assets.Total.Amount
            */
            assertion = NotMonetary(report.CTR130, @"U", 15, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Assets.Total.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.430048"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR130", Value = GetValueOrEmpty(report.CTR130) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.430049
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR430049()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.430049
            Tax offset refunds (Remainder of refundable tax offsets) must be supplied
    
            Legacy Rule Format:
            IF [CTR326] = NULL 
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR326 = NULL
    
            Data Elements:
    
            ^CTR326 = RP:IncomeTax.TotalRemainderOfRefundableTaxOffsets.Amount
            */
            assertion = (report.CTR326 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Mandatory field not supplied",
                    Location = "/xbrli:xbrl/tns:IncomeTax.TotalRemainderOfRefundableTaxOffsets.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.430049"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR326", Value = GetValueOrEmpty(report.CTR326) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.430050
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR430050()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.430050
            Monetary field must be unsigned, maximum of 15 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR131] <> NULL AND [CTR131] <> MONETARY(U,15,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR131, 'U', 15, 0)
    
            Data Elements:
    
            ^CTR131 = RP.Closing(Instant):Liabilities.NonCurrentTradeAndOtherPayables.Amount
            */
            assertion = NotMonetary(report.CTR131, @"U", 15, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Liabilities.NonCurrentTradeAndOtherPayables.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.430050"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR131", Value = GetValueOrEmpty(report.CTR131) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.430051
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR430051()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.430051
            Monetary field must be unsigned, maximum of 15 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR132] <> NULL AND [CTR132] <> MONETARY(U,15,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR132, 'U', 15, 0)
    
            Data Elements:
    
            ^CTR132 = RP.Closing(Instant):Liabilities.CurrentTotal.Amount
            */
            assertion = NotMonetary(report.CTR132, @"U", 15, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Liabilities.CurrentTotal.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.430051"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR132", Value = GetValueOrEmpty(report.CTR132) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.430052
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR430052()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.430052
            Monetary field must be unsigned, maximum of 15 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR133] <> NULL AND [CTR133] <> MONETARY(U,15,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR133, 'U', 15, 0)
    
            Data Elements:
    
            ^CTR133 = RP.Closing(Instant):Liabilities.Total.Amount
            */
            assertion = NotMonetary(report.CTR133, @"U", 15, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Liabilities.Total.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.430052"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR133", Value = GetValueOrEmpty(report.CTR133) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.430053
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR430053()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.430053
            Monetary field must be unsigned, maximum of 15 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR123] <> NULL AND [CTR123] <> MONETARY(U,15,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR123, 'U', 15, 0)
    
            Data Elements:
    
            ^CTR123 = RP.Opening(Instant):Assets.Inventories.Total.Amount
            */
            assertion = NotMonetary(report.CTR123, @"U", 15, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Assets.Inventories.Total.Amount[@contextRef='" + report.RPOpeningInstantId + "']",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.430053"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR123", Value = GetValueOrEmpty(report.CTR123) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.430056
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR430056()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.430056
            Mandatory field not supplied
    
            Legacy Rule Format:
            IF [CTR360] = NULL
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR360 = NULL
    
            Data Elements:
    
            ^CTR360 = RP:IncomeTax.PayableOrRefundableTotal.Amount
            */
            assertion = (report.CTR360 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Mandatory field not supplied",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayableOrRefundableTotal.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.430056"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR360", Value = GetValueOrEmpty(report.CTR360) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.430057
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR430057()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.430057
            Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters
    
            Legacy Rule Format:
            IF [CTR360] <> NULL AND [CTR360] <> MONETARY(S,13,2)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR360, 'S', 13, 2)
    
            Data Elements:
    
            ^CTR360 = RP:IncomeTax.PayableOrRefundableTotal.Amount
            */
            assertion = NotMonetary(report.CTR360, @"S", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayableOrRefundableTotal.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.430057"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR360", Value = GetValueOrEmpty(report.CTR360) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.430068
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR430068()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.430068
            In the calculation statement, the Amount due or refundable must be equal to tax payable, plus Section 102AAM interest charge,  less eligible credits, less tax offset refunds (remainder of refundable tax offsets), less PAYG instalments raised.
    
            Legacy Rule Format:
            IF [CTR360] <> NULL AND [CTR360] <> (([CTR203] + [CTR204]) - ([CTR317] + [CTR326] + [CTR205]))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR360 <> NULL AND ^CTR360 <> ((^CTR203 + ^CTR204) - (^CTR317 + ^CTR326 + ^CTR205))
    
            Data Elements:
    
            ^CTR360 = RP:IncomeTax.PayableOrRefundableTotal.Amount
    
            ^CTR203 = RP:IncomeTax.Payable.Amount
    
            ^CTR204 = RP:InternationalDealings.AAMInterestSection102.Amount
    
            ^CTR317 = RP:IncomeTax.Deduction.CreditsEligibleTotal.Amount
    
            ^CTR326 = RP:IncomeTax.TotalRemainderOfRefundableTaxOffsets.Amount
    
            ^CTR205 = RP:IncomeTax.LiabilityInstalmentsTotal.Amount
            */
            assertion = (report.CTR360 != null && report.CTR360.GetValueOrDefault() != report.CTR203.GetValueOrDefault() + report.CTR204.GetValueOrDefault() - (report.CTR317.GetValueOrDefault() + report.CTR326.GetValueOrDefault() + report.CTR205.GetValueOrDefault()));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.430068",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Amount of payment/refund is incorrect",
                    LongDescription = @"In the calculation statement, the Amount due or refundable must be equal to tax payable, plus Section 102AAM interest charge,  less eligible credits, less tax offset refunds (remainder of refundable tax offsets), less PAYG instalments raised. Where the calculated figure is positive payment is required; if it is zero no payment is required; and if it is negative a refund is due.",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayableOrRefundableTotal.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.430068"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR360", Value = GetValueOrEmpty(report.CTR360) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR203", Value = GetValueOrEmpty(report.CTR203) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR204", Value = GetValueOrEmpty(report.CTR204) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR317", Value = GetValueOrEmpty(report.CTR317) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR326", Value = GetValueOrEmpty(report.CTR326) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR205", Value = GetValueOrEmpty(report.CTR205) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.430217
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR430217()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.430217
            Tax on taxable or net income must be supplied
    
            Legacy Rule Format:
            IF (RP:rvctc1.xx.xx:IncomeTax.TotalTaxOnTaxableIncome.Amount) = NULL
             RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR320 = NULL
    
            Data Elements:
    
            ^CTR320 = RP:IncomeTax.TotalTaxOnTaxableIncome.Amount
            */
            assertion = (report.CTR320 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.430217",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax on taxable or net income must be supplied. A value must be included in the field, even if the value is zero.",
                    Location = "/xbrli:xbrl/tns:IncomeTax.TotalTaxOnTaxableIncome.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.430217"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR320", Value = GetValueOrEmpty(report.CTR320) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.438040
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR438040()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.438040
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR364] <> NULL AND [CTR364] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR364, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR364 = RP:Remuneration.WagesAndSalaries.PaymentsContractorAndConsultant.Amount
            */
            assertion = NotMonetary(report.CTR364, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Remuneration.WagesAndSalaries.PaymentsContractorAndConsultant.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.438040"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR364", Value = GetValueOrEmpty(report.CTR364) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.438041
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR438041()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.438041
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR365] <> NULL AND [CTR365] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR365, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR365 = RP.JAUS:Expense.Royalties.Amount
            */
            assertion = NotMonetary(report.CTR365, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.Royalties.Amount[@contextRef='" + report.RPJAUSId + "']",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.438041"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR365", Value = GetValueOrEmpty(report.CTR365) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.438042
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR438042()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.438042
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR366] <> NULL AND [CTR366] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR366, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR366 = RP.JFOREIGN:Expense.Royalties.Amount
            */
            assertion = NotMonetary(report.CTR366, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.Royalties.Amount[@contextRef='" + report.RPJFOREIGNId + "']",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.438042"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR366", Value = GetValueOrEmpty(report.CTR366) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.438047
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR438047()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.438047
            Invalid context. The number of Reporting party: Instant period type contexts must not exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP(Instant))) > 1 
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP(Instant)
            */
            assertion = (Count(report.RPInstantCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200007",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid XBRL schema",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPInstantLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.438047"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.438051
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR438051()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.438051
            Monetary field - unsigned, maximum of 13 characters, may include a decimal point and up to 2 decimals
    
            Legacy Rule Format:
            IF [CTR367] <> NULL AND [CTR367] <> MONETARY(U,13,2)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR367, 'U', 13, 2)
    
            Data Elements:
    
            ^CTR367 = RP:IncomeTax.PayAsYouGoWithholding.CreditForCapitalGainsWithheldFromForeignResidents.Amount
            */
            assertion = NotMonetary(report.CTR367, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditForCapitalGainsWithheldFromForeignResidents.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.438051"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR367", Value = GetValueOrEmpty(report.CTR367) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.438067
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR438067()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.438067
            Year of return must be either the current or next (future) lodgment year associated with this service. If a future year lodgment is required, then it must be lodged no later than the 15th of June of the current year, unless it is an amendment. After the 15th of June, a return for that lodgment year must be submitted from the 1st of July, using the new service associated to that specific lodgment year.
    
            Legacy Rule Format:
            IF ([CTR2] <> SET(2026,2027))
            OR ([CTR2] = 2027 AND DATE(Today) > 2027-06-15 
            AND [CTR298] <> TRUE)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (NotInSet(^CTR2, '"2026","2027"')) OR (^CTR2 = 2027 AND Today() > AsDate('2027-06-15') AND ^CTR298 <> TRUE)
    
            Data Elements:
    
            ^CTR2 = RP:Report.TargetFinancial.Year
    
            ^CTR298 = RP:Report.Amendment.Indicator
            */
            assertion = (!(IsMatch(report.CTR2.GetValueOrDefault().ToString(), @"^(2026|2027)$", RegexOptions.IgnoreCase)) || report.CTR2.GetValueOrDefault() == 2027 && DateTime.Now.Date > AsDate(@"2027-06-15") && report.CTR298 != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500127",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Year of return is invalid",
                    LongDescription = @"Year of return must be either the current or next (future) lodgment year associated with this service. If a future year lodgment is required, then it must be lodged no later than the 15th of June of the current year, unless it is an amendment. After the 15th of June, a return for that lodgment year must be submitted from the 1st of July, using the new service associated to that specific lodgment year.",
                    Location = "/xbrli:xbrl/tns:Report.TargetFinancial.Year",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.438067"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR2", Value = GetValueOrEmpty(report.CTR2) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR298", Value = GetValueOrEmpty(report.CTR298) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.438076
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR438076()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.438076
            Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters
    
            Legacy Rule Format:
            IF [CTR369] <> NULL AND [CTR369] <> MONETARY(U,13,2)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR369, 'U', 13, 2)
    
            Data Elements:
    
            ^CTR369 = RP.LimitedPartners:TaxOffsetClaim.NonRefundableOther.Amount
            */
            assertion = NotMonetary(report.CTR369, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:TaxOffsetClaim.NonRefundableOther.Amount[@contextRef='" + report.RPLimitedPartnersId + "']",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.438076"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR369", Value = GetValueOrEmpty(report.CTR369) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.438077
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR438077()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.438077
            Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters
    
            Legacy Rule Format:
            IF [CTR368] <> NULL AND [CTR368] <> MONETARY(U,13,2)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR368, 'U', 13, 2)
    
            Data Elements:
    
            ^CTR368 = RP.Investor:TaxOffsetClaim.NonRefundableOther.Amount
            */
            assertion = NotMonetary(report.CTR368, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:TaxOffsetClaim.NonRefundableOther.Amount[@contextRef='" + report.RPInvestorId + "']",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.438077"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR368", Value = GetValueOrEmpty(report.CTR368) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.438078
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR438078()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.438078
            Amount claimed at the Early stage investor tax offset cannot exceed $200,000.
    
            Legacy Rule Format:
            IF ([CTR368] <> NULL OR [CTR374] <> NULL) AND (([CTR368] + [CTR374]) > 200000)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR368 <> NULL OR ^CTR374 <> NULL) AND ((^CTR368 + ^CTR374) > 200000)
    
            Data Elements:
    
            ^CTR368 = RP.Investor:TaxOffsetClaim.NonRefundableOther.Amount
    
            ^CTR374 = RP.Investor.YPrevious:TaxOffsetClaim.NonRefundableOther.Amount
            */
            assertion = ((report.CTR368 != null || report.CTR374 != null) && report.CTR368.GetValueOrDefault() + report.CTR374.GetValueOrDefault() > 200000);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.438078",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Early stage investor tax offset claim amount cannot exceed $200,000.",
                    Location = "/xbrli:xbrl/tns:TaxOffsetClaim.NonRefundableOther.Amount[@contextRef='" + report.RPInvestorId + "']",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.438078"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR368", Value = GetValueOrEmpty(report.CTR368) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR374", Value = GetValueOrEmpty(report.CTR374) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.438081
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR438081()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.438081
            If the company provides a ‘No’ response to question “Have you lodged a general purpose financial statement with ASIC in relation to this income year?”, a response is required to question “Will you be lodging a general purpose financial statement with ASIC in relation to this income year?”.
    
            Legacy Rule Format:
            IF [CTR371] = FALSE AND [CTR372] = NULL
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR371 = FALSE AND ^CTR372 = NULL
    
            Data Elements:
    
            ^CTR371 = RP:RegulatoryDisclosures.FinancialStatementLodged.Indicator
    
            ^CTR372 = RP:RegulatoryDisclosures.FinancialStatementLodgmentPending.Indicator
            */
            assertion = (report.CTR371 == false && report.CTR372 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.438081",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"General Purpose Financial Statement question must be supplied.",
                    LongDescription = @"If response to question ""Have you lodged a general purpose financial statement with ASIC in relation to this income year?"" is ""No"", then response to question ""Will you be lodging a general purpose financial statement with ASIC in relation to this income year?"" is required.",
                    Location = "/xbrli:xbrl/tns:RegulatoryDisclosures.FinancialStatementLodged.Indicator",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.438081"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR371", Value = GetValueOrEmpty(report.CTR371) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR372", Value = GetValueOrEmpty(report.CTR372) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.438083
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR438083()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.438083
            If the company provides a ‘Yes’ response to question “Have you lodged a general purpose financial statement with ASIC in relation to this income year?”, then the company is not required to provide a response at question “Will you be lodging a general purpose financial statement with ASIC in relation to this income year?”.
    
            Legacy Rule Format:
            IF [CTR371] = TRUE AND [CTR372] <> NULL
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR371 = TRUE AND ^CTR372 <> NULL
    
            Data Elements:
    
            ^CTR371 = RP:RegulatoryDisclosures.FinancialStatementLodged.Indicator
    
            ^CTR372 = RP:RegulatoryDisclosures.FinancialStatementLodgmentPending.Indicator
            */
            assertion = (report.CTR371 == true && report.CTR372 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.438083",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"General Purpose Financial Statement question must be supplied.",
                    LongDescription = @"If response to question ""Have you lodged a general purpose financial statement with ASIC in relation to this income year?"" is ""Yes"" then response to question ""Will you be lodging a general purpose financial statement with ASIC in relation to this income year?"" is not required.",
                    Location = "/xbrli:xbrl/tns:RegulatoryDisclosures.FinancialStatementLodged.Indicator",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.438083"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR371", Value = GetValueOrEmpty(report.CTR371) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR372", Value = GetValueOrEmpty(report.CTR372) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.438085
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR438085()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.438085
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR456] <> NULL AND [CTR456] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR456, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR456 = RP:Expense.BuildToRentCapitalWorksDeduction.Amount
            */
            assertion = NotMonetary(report.CTR456, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.BuildToRentCapitalWorksDeduction.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.438085"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR456", Value = GetValueOrEmpty(report.CTR456) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.438088
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR438088()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.438088
            Current postal address must be supplied.
    
            Legacy Rule Format:
            IF [CTR15] = NULLORBLANK
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR15 = BLANK
    
            Data Elements:
    
            ^CTR15 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.CTR15) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.438088",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Current postal address must be supplied.",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.438088"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR15", Value = report.CTR15 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.438089
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR438089()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.438089
            Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters
    
            Legacy Rule Format:
            IF [CTR373] <> NULL AND [CTR373] <> MONETARY(U,13,2)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR373, 'U', 13, 2)
    
            Data Elements:
    
            ^CTR373 = RP.LimitedPartners.YPrevious:TaxOffsetClaim.NonRefundableOther.Amount
            */
            assertion = NotMonetary(report.CTR373, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:TaxOffsetClaim.NonRefundableOther.Amount[@contextRef='" + report.RPLimitedPartnersYPreviousId + "']",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.438089"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR373", Value = GetValueOrEmpty(report.CTR373) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.438090
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR438090()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.438090
            Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters
    
            Legacy Rule Format:
            IF [CTR374] <> NULL AND [CTR374] <> MONETARY(U,13,2)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR374, 'U', 13, 2)
    
            Data Elements:
    
            ^CTR374 = RP.Investor.YPrevious:TaxOffsetClaim.NonRefundableOther.Amount
            */
            assertion = NotMonetary(report.CTR374, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:TaxOffsetClaim.NonRefundableOther.Amount[@contextRef='" + report.RPInvestorYPreviousId + "']",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.438090"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR374", Value = GetValueOrEmpty(report.CTR374) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.438091
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR438091()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.438091
            Non-refundable carry forward tax offsets must be at least equal to the sum of the amounts listed at R&D tax offsets, ESVCLP tax offsets and Early stage investor tax offset.s
    
            Legacy Rule Format:
            IF [CTR323] <> NULL AND [CTR323] < ([CTR328] + [CTR353] + [CTR369] + [CTR373] + [CTR368] + [CTR374])
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR323 <> NULL AND ^CTR323 < (^CTR328 + ^CTR353 + ^CTR369 + ^CTR373 + ^CTR368 + ^CTR374)
    
            Data Elements:
    
            ^CTR323 = RP:IncomeTax.TaxOffsetNonRefundableCarryForward.Amount
    
            ^CTR328 = RP:TaxConcession.ResearchAndDevelopment.TaxOffsetNonRefundable.Amount
    
            ^CTR353 = RP.YPrevious:TaxConcession.ResearchAndDevelopment.TaxOffsetNonRefundable.Amount
    
            ^CTR369 = RP.LimitedPartners:TaxOffsetClaim.NonRefundableOther.Amount
    
            ^CTR373 = RP.LimitedPartners.YPrevious:TaxOffsetClaim.NonRefundableOther.Amount
    
            ^CTR368 = RP.Investor:TaxOffsetClaim.NonRefundableOther.Amount
    
            ^CTR374 = RP.Investor.YPrevious:TaxOffsetClaim.NonRefundableOther.Amount
            */
            assertion = (report.CTR323 != null && report.CTR323.GetValueOrDefault() < report.CTR328.GetValueOrDefault() + report.CTR353.GetValueOrDefault() + report.CTR369.GetValueOrDefault() + report.CTR373.GetValueOrDefault() + report.CTR368.GetValueOrDefault() + report.CTR374.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.438091",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Check the calculation of non-refundable carry forward tax offsets.",
                    Location = "/xbrli:xbrl/tns:IncomeTax.TaxOffsetNonRefundableCarryForward.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.438091"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR323", Value = GetValueOrEmpty(report.CTR323) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR328", Value = GetValueOrEmpty(report.CTR328) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR353", Value = GetValueOrEmpty(report.CTR353) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR369", Value = GetValueOrEmpty(report.CTR369) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR373", Value = GetValueOrEmpty(report.CTR373) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR368", Value = GetValueOrEmpty(report.CTR368) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR374", Value = GetValueOrEmpty(report.CTR374) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.438092
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR438092()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.438092
            If the base rate entity indicator question is not answered or is false and company is a credit union, then tax on taxable or net income should equal 45% on any excess over $49,999 but below $150,000.
    
            Legacy Rule Format:
            IF (([CTR375] = NULL OR [CTR375] = FALSE) AND [CTR49] = "62230" AND 
            (([CTR120] > 49999) AND ([CTR120] < 150000))) AND 
            ([CTR320] <> (([CTR120] - 49999) * 0.45) +/- 1) 
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ((^CTR375 = NULL OR ^CTR375 = FALSE) AND ^CTR49 = '62230' AND ((^CTR120 > 49999) AND (^CTR120 < 150000))) AND (OutsideRange(^CTR320, (^CTR120 - 49999) * 0.45, 1))
    
            Data Elements:
    
            ^CTR120 = RP:Income.Taxable.Amount
    
            ^CTR375 = RP:Elections.BaseRateEntityElection.Indicator
    
            ^CTR49 = RP:OrganisationDetails.OrganisationIndustry2006Extended.Code
    
            ^CTR320 = RP:IncomeTax.TotalTaxOnTaxableIncome.Amount
            */
            assertion = ((report.CTR375 == null || report.CTR375 == false) && report.CTR49 == @"62230" && (report.CTR120.GetValueOrDefault() > 49999 && report.CTR120.GetValueOrDefault() < 150000) && OutsideRange(report.CTR320.GetValueOrDefault(), (report.CTR120.GetValueOrDefault() - 49999) * 0.45M, 1));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.430224",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax on taxable or net income is not equal to 45% of excess over $49,999 on Taxable/Net income for Credit Unions only",
                    LongDescription = @"Calculation statement Tax on taxable or net income is not equal to 45% of any excess over $49,999 on calculation statement Taxable/Net income for Credit Unions only",
                    Location = "/xbrli:xbrl/tns:Income.Taxable.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.438092"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR375", Value = GetValueOrEmpty(report.CTR375) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR49", Value = report.CTR49 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR120", Value = GetValueOrEmpty(report.CTR120) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR320", Value = GetValueOrEmpty(report.CTR320) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.438093
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR438093()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.438093
            If the base rate entity indicator response is TRUE and the company is a credit union, when lodging for the 2022 or future year of return, Tax on taxable or net income should equal to 37.5% of any excess over $49,999 (but below $150,000) of Taxable/Net income
    
            Legacy Rule Format:
            IF [CTR375] = TRUE AND [CTR49] = "62230" AND (([CTR120] > 49999) AND ([CTR120] < 150000)) 
            AND ([CTR320]<> (([CTR120] - 49999) * 0.375) +/- 1)
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR375 = TRUE AND ^CTR49 = '62230' AND ((^CTR120 > 49999) AND (^CTR120 < 150000)) AND (OutsideRange(^CTR320, (^CTR120 - 49999) * 0.375, 1))
    
            Data Elements:
    
            ^CTR120 = RP:Income.Taxable.Amount
    
            ^CTR375 = RP:Elections.BaseRateEntityElection.Indicator
    
            ^CTR49 = RP:OrganisationDetails.OrganisationIndustry2006Extended.Code
    
            ^CTR320 = RP:IncomeTax.TotalTaxOnTaxableIncome.Amount
            */
            assertion = (report.CTR375 == true && report.CTR49 == @"62230" && (report.CTR120.GetValueOrDefault() > 49999 && report.CTR120.GetValueOrDefault() < 150000) && OutsideRange(report.CTR320.GetValueOrDefault(), (report.CTR120.GetValueOrDefault() - 49999) * 0.375M, 1));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.458093",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax on taxable or net income incorrect. Check rate calculation and Credit Union operation entity validity.",
                    LongDescription = @"Calculation statement Tax on taxable or net income is not equal to 39% of any excess over $49,999 on calculation statement Taxable/Net income for Credit Unions only for the 2021 year of return or Tax on taxable or net income is not equal to 37.5% of excess over $49,999 on Taxable/Net income for Credit Unions only for the 2022 year of return.",
                    Location = "/xbrli:xbrl/tns:Income.Taxable.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.438093"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR375", Value = GetValueOrEmpty(report.CTR375) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR49", Value = report.CTR49 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR120", Value = GetValueOrEmpty(report.CTR120) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR320", Value = GetValueOrEmpty(report.CTR320) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.438095
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR438095()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.438095
            If the base rate entity indicator question is not answered or the response is FALSE and company is a Credit Union, then Tax on taxable or net income should equal to 30% of Taxable/Net income where Taxable/Net income is less than $50,000 or greater than $149,999.
    
            Legacy Rule Format:
            IF (([CTR375] = NULL OR [CTR375] = FALSE) AND [CTR49] = "62230" AND 
            (([CTR120] > 0 AND [CTR120] < 50000) OR ([CTR120] > 149999)))
             AND ([CTR320] <> (([CTR120] * 0.30) +/- 1))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ((^CTR375 = NULL OR ^CTR375 = FALSE) AND ^CTR49 = '62230' AND ((^CTR120 > 0 AND ^CTR120 < 50000) OR (^CTR120 > 149999))) AND (OutsideRange(^CTR320, ^CTR120  * 0.30, 1))
    
            Data Elements:
    
            ^CTR320 = RP:IncomeTax.TotalTaxOnTaxableIncome.Amount
    
            ^CTR375 = RP:Elections.BaseRateEntityElection.Indicator
    
            ^CTR49 = RP:OrganisationDetails.OrganisationIndustry2006Extended.Code
    
            ^CTR120 = RP:Income.Taxable.Amount
            */
            assertion = ((report.CTR375 == null || report.CTR375 == false) && report.CTR49 == @"62230" && (report.CTR120.GetValueOrDefault() > 0 && report.CTR120.GetValueOrDefault() < 50000 || report.CTR120.GetValueOrDefault() > 149999) && OutsideRange(report.CTR320.GetValueOrDefault(), report.CTR120.GetValueOrDefault() * 0.30M, 1));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.430226",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax on taxable or net income incorrect. Check rate calculation and Credit Union operation entity validity",
                    LongDescription = @"The Company is not an entity that has been identified as being allowed to have a tax rate of less than 30%. Check the rate calculation and company entity type. (Calculation statement Tax on taxable or net income)",
                    Location = "/xbrli:xbrl/tns:IncomeTax.TotalTaxOnTaxableIncome.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.438095"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR375", Value = GetValueOrEmpty(report.CTR375) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR49", Value = report.CTR49 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR120", Value = GetValueOrEmpty(report.CTR120) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR320", Value = GetValueOrEmpty(report.CTR320) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.438097
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR438097()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.438097
            If the base rate entity indicator response is TRUE and the company is a credit union, when lodging for the 2022, or future year of return Tax on taxable or net income should equal to 25% of Taxable/Net income, where Taxable/Net income is less than $50,000 or greater than $149,999
    
            Legacy Rule Format:
            IF [CTR375] = TRUE AND [CTR49] = "62230" AND (([CTR120] > 0 AND [CTR120] < 50000) OR ([CTR120] > 149999))
             AND ([CTR320] <> (([CTR120] * 0.25) +/- 1))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR375 = TRUE AND ^CTR49 = '62230' AND ((^CTR120 > 0 AND ^CTR120 < 50000) OR (^CTR120 > 149999)) AND (OutsideRange(^CTR320, ^CTR120  * 0.25, 1))
    
            Data Elements:
    
            ^CTR320 = RP:IncomeTax.TotalTaxOnTaxableIncome.Amount
    
            ^CTR375 = RP:Elections.BaseRateEntityElection.Indicator
    
            ^CTR49 = RP:OrganisationDetails.OrganisationIndustry2006Extended.Code
    
            ^CTR120 = RP:Income.Taxable.Amount
            */
            assertion = (report.CTR375 == true && report.CTR49 == @"62230" && (report.CTR120.GetValueOrDefault() > 0 && report.CTR120.GetValueOrDefault() < 50000 || report.CTR120.GetValueOrDefault() > 149999) && OutsideRange(report.CTR320.GetValueOrDefault(), report.CTR120.GetValueOrDefault() * 0.25M, 1));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.468097",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax on taxable or net income incorrect. Check rate calculation and Credit Union operation entity validity.",
                    LongDescription = @"The Company is not an entity that has been identified as being allowed to have a tax rate of less than 25% for the 2022, or future year of return. Check the rate calculation and company entity type. (Calculation statement Tax on taxable or net income)",
                    Location = "/xbrli:xbrl/tns:IncomeTax.TotalTaxOnTaxableIncome.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.438097"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR375", Value = GetValueOrEmpty(report.CTR375) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR49", Value = report.CTR49 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR120", Value = GetValueOrEmpty(report.CTR120) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR320", Value = GetValueOrEmpty(report.CTR320) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.500001
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR500001()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.500001
            Previous postal address is only required when an address change has occurred since the last return
    
            Legacy Rule Format:
            IF ([CTR24] = [CTR15] AND [CTR25] = [CTR16] AND [CTR28] = [CTR19] AND [CTR29] = [CTR20] AND [CTR30] = [CTR21]) 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR24 = ^CTR15 AND ^CTR25 = ^CTR16 AND ^CTR28 = ^CTR19 AND ^CTR29 = ^CTR20 AND ^CTR30 = ^CTR21
    
            Data Elements:
    
            ^CTR24 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
    
            ^CTR15 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
    
            ^CTR25 = RP:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
    
            ^CTR16 = RP:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
    
            ^CTR28 = RP:AddressDetails:AddressDetails.LocalityName.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
    
            ^CTR19 = RP:AddressDetails:AddressDetails.LocalityName.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
    
            ^CTR29 = RP:AddressDetails:AddressDetails.StateOrTerritory.Code WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
    
            ^CTR20 = RP:AddressDetails:AddressDetails.StateOrTerritory.Code WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
    
            ^CTR30 = RP:AddressDetails:AddressDetails.Postcode.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
    
            ^CTR21 = RP:AddressDetails:AddressDetails.Postcode.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (report.CTR24 == report.CTR15 && report.CTR25 == report.CTR16 && report.CTR28 == report.CTR19 && report.CTR29 == report.CTR20 && report.CTR30 == report.CTR21);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Previous postal address must not be the same as current postal address",
                    LongDescription = @"Previous postal address is only required when an address change has occurred since the last return.  Delete previous postal address or input the old address if a change of address has occurred.",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_P_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.500001"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR24", Value = report.CTR24 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR15", Value = report.CTR15 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR25", Value = GetValueOrEmpty(report.CTR25) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR16", Value = GetValueOrEmpty(report.CTR16) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR28", Value = report.CTR28 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR19", Value = report.CTR19 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR29", Value = GetValueOrEmpty(report.CTR29) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR20", Value = GetValueOrEmpty(report.CTR20) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR30", Value = GetValueOrEmpty(report.CTR30) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR21", Value = GetValueOrEmpty(report.CTR21) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.500154
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR500154()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.500154
            If the company has self-assessed that they are a Country by Country Reporting Entity (CbCRE), a response is required to question “Have you lodged a general purpose financial statement with ASIC in relation to this income year?”.
    
            Legacy Rule Format:
            IF [CTR378] = TRUE AND [CTR371] = NULL
             RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR378 = TRUE AND ^CTR371 = NULL
    
            Data Elements:
    
            ^CTR378 = RP:InternationalDealings.CountryByCountryReportingEntityStatus.Indicator
    
            ^CTR371 = RP:RegulatoryDisclosures.FinancialStatementLodged.Indicator
            */
            assertion = (report.CTR378 == true && report.CTR371 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.500154",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"General Purpose Financial Statement question must be supplied.",
                    Location = "/xbrli:xbrl/tns:InternationalDealings.CountryByCountryReportingEntityStatus.Indicator",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.500154"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR378", Value = GetValueOrEmpty(report.CTR378) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR371", Value = GetValueOrEmpty(report.CTR371) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.500155
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR500155()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.500155
            If the company has self-assessed that they are not a Country by Country Reporting Entity (CbCRE), then response to questions “Have you lodged a general purpose financial statement with ASIC in relation to this income year?” or “Will you be lodging a general purpose financial statement with ASIC in relation to this income year?” is not required.
    
            Legacy Rule Format:
            IF [CTR378] <> TRUE AND ([CTR371] <> NULL OR [CTR372] <> NULL)
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR378 <> TRUE AND (^CTR371 <> NULL OR ^CTR372 <> NULL)
    
            Data Elements:
    
            ^CTR378 = RP:InternationalDealings.CountryByCountryReportingEntityStatus.Indicator
    
            ^CTR371 = RP:RegulatoryDisclosures.FinancialStatementLodged.Indicator
    
            ^CTR372 = RP:RegulatoryDisclosures.FinancialStatementLodgmentPending.Indicator
            */
            assertion = (report.CTR378 != true && (report.CTR371 != null || report.CTR372 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.500155",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"General Purpose Financial Statement questions must not be supplied.",
                    LongDescription = @"General Purpose Financial Statement questions do not need to be supplied if you have not selected the Country-by-country reporting entity checkbox.",
                    Location = "/xbrli:xbrl/tns:InternationalDealings.CountryByCountryReportingEntityStatus.Indicator",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.500155"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR378", Value = GetValueOrEmpty(report.CTR378) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR371", Value = GetValueOrEmpty(report.CTR371) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR372", Value = GetValueOrEmpty(report.CTR372) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.500157
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR500157()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.500157
            Gross distribution from trusts action code is required when Gross distribution from trusts amount is present.
    
            Legacy Rule Format:
            IF ([CTR379] = NULL)  AND ([CTR61] > 0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR379 = NULL)  AND (^CTR61 > 0)
    
            Data Elements:
    
            ^CTR379 = RP:TrustDistributionIncome:Party.TrustType.Code
    
            ^CTR61 = RP:TrustDistributionIncome:Income.TrustDistributionGross.Amount
            */
            assertion = (report.CTR379 == null && report.CTR61.GetValueOrDefault() > 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.500157",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Gross distribution from trusts action code is required when Gross distribution from trusts amount is present.",
                    Location = "/xbrli:xbrl/tns:TrustDistributionIncome/tns:Party.TrustType.Code",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.500157"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR379", Value = GetValueOrEmpty(report.CTR379) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR61", Value = GetValueOrEmpty(report.CTR61) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.500158
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR500158()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.500158
            Gross distribution from trusts amount is required when Gross distribution from trusts action code is present.
    
            Legacy Rule Format:
            IF ([CTR379] <> NULLORBLANK) AND (([CTR61] = 0) or ([CTR61] = NULL))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR379 <> BLANK) AND ((^CTR61 = 0) or (^CTR61 = NULL))
    
            Data Elements:
    
            ^CTR379 = RP:TrustDistributionIncome:Party.TrustType.Code
    
            ^CTR61 = RP:TrustDistributionIncome:Income.TrustDistributionGross.Amount
            */
            assertion = (string.IsNullOrWhiteSpace(report.CTR379) != true && (report.CTR61 == 0 || report.CTR61 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.500158",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Gross distribution from trusts amount is required when Gross distribution from trusts action code is present.",
                    Location = "/xbrli:xbrl/tns:TrustDistributionIncome/tns:Party.TrustType.Code",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.500158"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR379", Value = GetValueOrEmpty(report.CTR379) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR61", Value = GetValueOrEmpty(report.CTR61) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.500163
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR500163()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.500163
            Monetary field must be unsigned, maximum of 12 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR393] <> NULL AND [CTR393] <> MONETARY(U,12,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR393, 'U', 12, 0)
    
            Data Elements:
    
            ^CTR393 = RP:Income.AggregatedTurnover.Amount
            */
            assertion = NotMonetary(report.CTR393, @"U", 12, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.AggregatedTurnover.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.500163"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR393", Value = GetValueOrEmpty(report.CTR393) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.500164
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR500164()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.500164
            Aggregated turnover must be provided if "P" is selected at Aggregated turnover range
    
            Legacy Rule Format:
            IF ([CTR393] = NULL OR [CTR393] = 0) AND [CTR392] = "P"
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR393 = NULL OR ^CTR393 = 0) AND ^CTR392 = 'P'
    
            Data Elements:
    
            ^CTR393 = RP:Income.AggregatedTurnover.Amount
    
            ^CTR392 = RP:Income.AggregatedTurnoverRange.Code
            */
            assertion = ((report.CTR393 == null || report.CTR393 == 0) && report.CTR392 == @"P");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.500164",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Aggregated turnover must be provided",
                    LongDescription = @"Aggregated turnover must be provided if the reported aggregated turnover range is more than $1 billion",
                    Location = "/xbrli:xbrl/tns:Income.AggregatedTurnover.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.500164"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR393", Value = GetValueOrEmpty(report.CTR393) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR392", Value = GetValueOrEmpty(report.CTR392) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.500186
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR500186()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.500186
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [CTR390] <> NULL AND [CTR390] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR390, 'U', 11, 0)
    
            Data Elements:
    
            ^CTR390 = RP:Expense.SubsequentYearAcceleratedDepreciationDeductionsForAssets.Amount
            */
            assertion = NotMonetary(report.CTR390, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.SubsequentYearAcceleratedDepreciationDeductionsForAssets.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.500186"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR390", Value = GetValueOrEmpty(report.CTR390) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.500217
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR500217()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.500217
            If the company has self-assessed that they are a Country by Country Reporting Entity (CbCRE), then the Significant Global Entity (SGE) indicator must be true.
    
            Legacy Rule Format:
            IF [CTR378] = TRUE AND ([CTR370] = NULL OR [CTR370] = FALSE)
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR378 = TRUE AND (^CTR370 = NULL OR ^CTR370 = FALSE)
    
            Data Elements:
    
            ^CTR378 = RP:InternationalDealings.CountryByCountryReportingEntityStatus.Indicator
    
            ^CTR370 = RP:OrganisationDetails.SignificantGlobalEntityStatus.Indicator
            */
            assertion = (report.CTR378 == true && (report.CTR370 == null || report.CTR370 == false));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.500217",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Significant Global Entity (SGE) indicator must be  'yes' (true) for a Country by Country Reporting Entity (CbCRE).",
                    Location = "/xbrli:xbrl/tns:InternationalDealings.CountryByCountryReportingEntityStatus.Indicator",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.500217"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR378", Value = GetValueOrEmpty(report.CTR378) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR370", Value = GetValueOrEmpty(report.CTR370) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.500235
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR500235()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.500235
            State and post code must be present for an Australian address and post code is in a valid range.
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = FALSE AND (pyde.xx.xx:AddressDetails.StateOrTerritory.Code = NULL OR (pyde.xx.xx:AddressDetails.Postcode.Text = NULL OR pyde.xx.xx:AddressDetails.Postcode.Text <> SET(0200-9999)))
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR40 = FALSE AND (^CTR38 = NULL OR (^CTR39 = NULL OR ^CTR39 <> BLANK AND (IsNumeric(^CTR39) = 'FALSE' OR AsNumeric(^CTR39) < 200 OR AsNumeric(^CTR39) > 9999)))
    
            Data Elements:
    
            ^CTR40 = RP:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
    
            ^CTR38 = RP:AddressDetails:AddressDetails.StateOrTerritory.Code WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
    
            ^CTR39 = RP:AddressDetails:AddressDetails.Postcode.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (report.CTR40 == false && (report.CTR38 == null || (report.CTR39 == null || string.IsNullOrWhiteSpace(report.CTR39) != true && (IsNumeric(report.CTR39) == false || AsNumeric(report.CTR39) < 200 || AsNumeric(report.CTR39) > 9999))));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.500235",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"For an Australian address, state code and post code must be present and a valid range for post code is 0200 - 9999.",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.OverseasAddress.Indicator",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.500235"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR40", Value = GetValueOrEmpty(report.CTR40) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR38", Value = GetValueOrEmpty(report.CTR38) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR39", Value = GetValueOrEmpty(report.CTR39) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.500235
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR500235_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.500235
            State and post code must be present for an Australian address and post code is in a valid range.
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = FALSE AND (pyde.xx.xx:AddressDetails.StateOrTerritory.Code = NULL OR (pyde.xx.xx:AddressDetails.Postcode.Text = NULL OR pyde.xx.xx:AddressDetails.Postcode.Text <> SET(0200-9999)))
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR22 = FALSE AND (^CTR20 = NULL OR (^CTR21 = NULL OR ^CTR21 <> BLANK AND (IsNumeric(^CTR21) = 'FALSE' OR AsNumeric(^CTR21) < 200 OR AsNumeric(^CTR21) > 9999)))
    
            Data Elements:
    
            ^CTR22 = RP:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
    
            ^CTR20 = RP:AddressDetails:AddressDetails.StateOrTerritory.Code WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
    
            ^CTR21 = RP:AddressDetails:AddressDetails.Postcode.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (report.CTR22 == false && (report.CTR20 == null || (report.CTR21 == null || string.IsNullOrWhiteSpace(report.CTR21) != true && (IsNumeric(report.CTR21) == false || AsNumeric(report.CTR21) < 200 || AsNumeric(report.CTR21) > 9999))));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.500235",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"For an Australian address, state code and post code must be present and a valid range for post code is 0200 - 9999.",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.OverseasAddress.Indicator",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.500235"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR22", Value = GetValueOrEmpty(report.CTR22) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR20", Value = GetValueOrEmpty(report.CTR20) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR21", Value = GetValueOrEmpty(report.CTR21) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.500235
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR500235_2()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.500235
            State and post code must be present for an Australian address and post code is in a valid range.
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = FALSE AND (pyde.xx.xx:AddressDetails.StateOrTerritory.Code = NULL OR (pyde.xx.xx:AddressDetails.Postcode.Text = NULL OR pyde.xx.xx:AddressDetails.Postcode.Text <> SET(0200-9999)))
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR31 = FALSE AND (^CTR29 = NULL OR (^CTR30 = NULL OR ^CTR30 <> BLANK AND (IsNumeric(^CTR30) = 'FALSE' OR AsNumeric(^CTR30) < 200 OR AsNumeric(^CTR30) > 9999)))
    
            Data Elements:
    
            ^CTR31 = RP:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
    
            ^CTR29 = RP:AddressDetails:AddressDetails.StateOrTerritory.Code WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
    
            ^CTR30 = RP:AddressDetails:AddressDetails.Postcode.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
            */
            assertion = (report.CTR31 == false && (report.CTR29 == null || (report.CTR30 == null || string.IsNullOrWhiteSpace(report.CTR30) != true && (IsNumeric(report.CTR30) == false || AsNumeric(report.CTR30) < 200 || AsNumeric(report.CTR30) > 9999))));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.500235",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"For an Australian address, state code and post code must be present and a valid range for post code is 0200 - 9999.",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_P_RPOccurrenceIndex) + "/tns:AddressDetails.OverseasAddress.Indicator",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.500235"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR31", Value = GetValueOrEmpty(report.CTR31) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR29", Value = GetValueOrEmpty(report.CTR29) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR30", Value = GetValueOrEmpty(report.CTR30) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.500300
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR500300()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.500300
            Monetary field - unsigned, maximum of 13 characters, may include a decimal point and up to 2 decimals
    
            Legacy Rule Format:
            IF [CTR450] <> NULL AND [CTR450] <> MONETARY(U,13,2)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR450, 'U', 13, 2)
    
            Data Elements:
    
            ^CTR450 = RP:Income.ResearchAndDevelopment.Clawback.Amount
            */
            assertion = NotMonetary(report.CTR450, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.ResearchAndDevelopment.Clawback.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.500300"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR450", Value = GetValueOrEmpty(report.CTR450) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.500301
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR500301()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.500301
            Other assessable income (7B) must be greater than or equal to Clawback amounts – additional assessable income (21W)
    
            Legacy Rule Format:
            IF [CTR450] <> NULL AND [CTR450] > [CTR95] + 1
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR450 <> NULL AND ^CTR450 > ^CTR95 + 1
    
            Data Elements:
    
            ^CTR450 = RP:Income.ResearchAndDevelopment.Clawback.Amount
    
            ^CTR95 = RP:Income.AssessableOther.Amount
            */
            assertion = (report.CTR450 != null && report.CTR450.GetValueOrDefault() > report.CTR95.GetValueOrDefault() + 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.500301",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Clawback amounts cannot be greater than other assessable income amount",
                    Location = "/xbrli:xbrl/tns:Income.ResearchAndDevelopment.Clawback.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.500301"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR450", Value = GetValueOrEmpty(report.CTR450) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR95", Value = GetValueOrEmpty(report.CTR95) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.500302
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR500302()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.500302
            Monetary field - unsigned, maximum of 13 characters, may include a decimal point and up to 2 decimals
    
            Legacy Rule Format:
            IF [CTR451] <> NULL AND [CTR451] <> MONETARY(U,13,2)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR451, 'U', 13, 2)
    
            Data Elements:
    
            ^CTR451 = RP:Expense.ResearchAndDevelopment.AdjustmentBalance.Amount
            */
            assertion = NotMonetary(report.CTR451, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.ResearchAndDevelopment.AdjustmentBalance.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.500302"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR451", Value = GetValueOrEmpty(report.CTR451) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.500303
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR500303()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.500303
            Other deductible expenses (7X) must be greater than or equal to Balancing adjustments – catch up deduction (21X)
    
            Legacy Rule Format:
            IF [CTR451] <> NULL AND [CTR451] > [CTR116] + 1
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR451 <> NULL AND ^CTR451 > ^CTR116 + 1
    
            Data Elements:
    
            ^CTR451 = RP:Expense.ResearchAndDevelopment.AdjustmentBalance.Amount
    
            ^CTR116 = RP:Expense.DeductibleOther.Amount
            */
            assertion = (report.CTR451 != null && report.CTR451.GetValueOrDefault() > report.CTR116.GetValueOrDefault() + 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.500303",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Balancing adjustments cannot be greater than Other deductible expenses amount",
                    Location = "/xbrli:xbrl/tns:Expense.ResearchAndDevelopment.AdjustmentBalance.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.500303"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR451", Value = GetValueOrEmpty(report.CTR451) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR116", Value = GetValueOrEmpty(report.CTR116) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.500306
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR500306()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.500306
            Non-refundable carry forward tax offset (calculation statement Label D) must be greater than or equal to Non-refundable R&D tax offset (21A) plus Non-refundable R&D tax offset carried forward from previous year (21B)
    
            Legacy Rule Format:
            IF [CTR323] < ([CTR328] + [CTR353])
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR323 < (^CTR328 + ^CTR353)
    
            Data Elements:
    
            ^CTR323 = RP:IncomeTax.TaxOffsetNonRefundableCarryForward.Amount
    
            ^CTR328 = RP:TaxConcession.ResearchAndDevelopment.TaxOffsetNonRefundable.Amount
    
            ^CTR353 = RP.YPrevious:TaxConcession.ResearchAndDevelopment.TaxOffsetNonRefundable.Amount
            */
            assertion = (report.CTR323.GetValueOrDefault() < report.CTR328.GetValueOrDefault() + report.CTR353.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.500306",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total amount of Non-refundable R&D tax offset and Non-refundable R&D tax offset carried forward from previous year must not be greater than Non-refundable carry forward tax offset in calculation statement.",
                    Location = "/xbrli:xbrl/tns:IncomeTax.TaxOffsetNonRefundableCarryForward.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.500306"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR323", Value = GetValueOrEmpty(report.CTR323) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR328", Value = GetValueOrEmpty(report.CTR328) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR353", Value = GetValueOrEmpty(report.CTR353) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.500380
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR500380()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.500380
            Gross Tax must be equal to Tax on taxable or net income plus RSA Income tax payable on no-TFN contributions income
    
            Legacy Rule Format:
            IF [CTR199] <> ([CTR320] + [CTR309])
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR199 <> (^CTR320 + ^CTR309)
    
            Data Elements:
    
            ^CTR199 = RP:IncomeTax.TaxableGross.Amount
    
            ^CTR320 = RP:IncomeTax.TotalTaxOnTaxableIncome.Amount
    
            ^CTR309 = RP:IncomeTax.RetirementSavingsAccountTFNNotQuotedContributionsIncomeTaxPayable.Amount
            */
            assertion = (report.CTR199.GetValueOrDefault() != report.CTR320.GetValueOrDefault() + report.CTR309.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.500380",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Gross tax calculation incorrect",
                    LongDescription = @"Gross Tax must be equal to Tax on taxable or net income plus RSA Income tax payable on no-TFN contributions income",
                    Location = "/xbrli:xbrl/tns:IncomeTax.TaxableGross.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.500380"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR199", Value = GetValueOrEmpty(report.CTR199) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR320", Value = GetValueOrEmpty(report.CTR320) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR309", Value = GetValueOrEmpty(report.CTR309) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.500381
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR500381()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.500381
            Eligible credits is equal to credit for tax withheld - foreign resident withholding (excluding capital gains) plus credit for tax withheld where ABN is not quoted plus tax withheld from interest or investments plus credit for TFN amounts withheld from payments from closely held trusts plus other credits plus Credit for amounts withheld from foreign resident capital gains withholding.
    
            Legacy Rule Format:
            IF [CTR317] <> ([CTR207] + [CTR208] + [CTR209] + [CTR299] + [CTR211] + [CTR367])
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR317 <> (^CTR207 + ^CTR208 + ^CTR209 + ^CTR299 + ^CTR211 + ^CTR367)
    
            Data Elements:
    
            ^CTR317 = RP:IncomeTax.Deduction.CreditsEligibleTotal.Amount
    
            ^CTR207 = RP:ITPAYGWithholdingDetails:IncomeTax.PayAsYouGoWithholding.TaxWithheld.Amount WHERE [IncomeTax.PayAsYouGoWithholding.PaymentType.Code='DFRW']
    
            ^CTR208 = RP:ITPAYGWithholdingDetails:IncomeTax.PayAsYouGoWithholding.TaxWithheld.Amount WHERE [IncomeTax.PayAsYouGoWithholding.PaymentType.Code='DNOABN']
    
            ^CTR209 = RP:IncomeTax.PayAsYouGoWithholding.TaxWithheldFromInvestmentIncomeAndInterest.Amount
    
            ^CTR299 = RP:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldFromCloselyHeldTrustTFNNotQuoted.Amount
    
            ^CTR211 = RP:IncomeTax.PayAsYouGoWithholding.CreditsRefundableOther.Amount
    
            ^CTR367 = RP:IncomeTax.PayAsYouGoWithholding.CreditForCapitalGainsWithheldFromForeignResidents.Amount
            */
            assertion = (report.CTR317.GetValueOrDefault() != report.CTR207.GetValueOrDefault() + report.CTR208.GetValueOrDefault() + report.CTR209.GetValueOrDefault() + report.CTR299.GetValueOrDefault() + report.CTR211.GetValueOrDefault() + report.CTR367.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.500381",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Eligible credits is incorrect in the Calculation statement",
                    LongDescription = @"Eligible credits is equal to credit for tax withheld - foreign resident withholding (excluding capital gains) plus credit for tax withheld where ABN is not quoted plus tax withheld from interest or investments plus credit for TFN amounts withheld from payments from closely held trusts plus other credits plus Credit for amounts withheld from foreign resident capital gains withholding.",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.CreditsEligibleTotal.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.500381"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR317", Value = GetValueOrEmpty(report.CTR317) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR207", Value = GetValueOrEmpty(report.CTR207) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR208", Value = GetValueOrEmpty(report.CTR208) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR209", Value = GetValueOrEmpty(report.CTR209) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR299", Value = GetValueOrEmpty(report.CTR299) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR211", Value = GetValueOrEmpty(report.CTR211) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR367", Value = GetValueOrEmpty(report.CTR367) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.500386
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR500386()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.500386
            Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters
    
            Legacy Rule Format:
            IF [CTR454] <> NULL AND [CTR454] <> MONETARY(U,11,2)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR454, 'U', 11, 2)
    
            Data Elements:
    
            ^CTR454 = RP:TaxOffsetClaim.DigitalGamesTaxOffset.Amount
            */
            assertion = NotMonetary(report.CTR454, @"U", 11, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:TaxOffsetClaim.DigitalGamesTaxOffset.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.500386"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR454", Value = GetValueOrEmpty(report.CTR454) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.500387
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR500387()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.500387
            Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters
    
            Legacy Rule Format:
            IF [CTR455] <> NULL AND [CTR455] <> MONETARY(U,11,2)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^CTR455, 'U', 11, 2)
    
            Data Elements:
    
            ^CTR455 = RP:TaxOffsetClaim.DigitalGamesTaxOffsetAlreadyClaimed.Amount
            */
            assertion = NotMonetary(report.CTR455, @"U", 11, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:TaxOffsetClaim.DigitalGamesTaxOffsetAlreadyClaimed.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.500387"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR455", Value = GetValueOrEmpty(report.CTR455) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.500389
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR500389()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.500389
            The sum of Current year refundable digital games tax offset amount being claimed and Total amount of current year refundable digital games tax offset already claimed or being claimed by related companies must not exceed $20,000,000
    
            Legacy Rule Format:
            IF ([CTR454] + [CTR455]) > 20000000
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR454 + ^CTR455) > 20000000
    
            Data Elements:
    
            ^CTR454 = RP:TaxOffsetClaim.DigitalGamesTaxOffset.Amount
    
            ^CTR455 = RP:TaxOffsetClaim.DigitalGamesTaxOffsetAlreadyClaimed.Amount
            */
            assertion = (report.CTR454.GetValueOrDefault() + report.CTR455.GetValueOrDefault() > 20000000);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.500389",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Refundable digital game tax offsets are greater than $20,000,000",
                    LongDescription = @"The sum of Current year refundable digital games tax offset amount being claimed and Total amount of current year refundable digital games tax offset already claimed or being claimed by related companies must not exceed $20,000,000.",
                    Location = "/xbrli:xbrl/tns:TaxOffsetClaim.DigitalGamesTaxOffset.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.500389"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR454", Value = GetValueOrEmpty(report.CTR454) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR455", Value = GetValueOrEmpty(report.CTR455) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.500396
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR500396()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.500396
            Gross tax amount is incorrect in the Calculation Statement
    
            Legacy Rule Format:
            IF (([CTR120] > 0) AND ([CTR49] <> "63100")) AND ([CTR320] > (([CTR120]* 0.55) +1))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ((^CTR120 > 0) AND (^CTR49 <> '63100')) AND (^CTR320 > ((^CTR120* 0.55) +1))
    
            Data Elements:
    
            ^CTR320 = RP:IncomeTax.TotalTaxOnTaxableIncome.Amount
    
            ^CTR120 = RP:Income.Taxable.Amount
    
            ^CTR49 = RP:OrganisationDetails.OrganisationIndustry2006Extended.Code
            */
            assertion = (report.CTR120.GetValueOrDefault() > 0 && report.CTR49 != @"63100" && report.CTR320.GetValueOrDefault() > report.CTR120.GetValueOrDefault() * 0.55M + 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.430227",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The tax on taxable or net income figure is incorrect in the Calculation Statement.",
                    LongDescription = @"The tax on taxable figure is excessive in the calculation statement.",
                    Location = "/xbrli:xbrl/tns:IncomeTax.TotalTaxOnTaxableIncome.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.500396"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR120", Value = GetValueOrEmpty(report.CTR120) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR49", Value = report.CTR49 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR320", Value = GetValueOrEmpty(report.CTR320) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.500397
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR500397()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.500397
            If the base rate entity indicator question is not answered or the response is FALSE, then Tax on taxable or net income should not be less than of 30% of Taxable/Net income.
    
            Legacy Rule Format:
            IF ([CTR375] = NULL OR [CTR375] =  FALSE) AND ((([CTR120] <= 0)  AND ([CTR320] > 0) AND ([CTR49] <> "63100")) OR (([CTR120] > 0) AND ([CTR52] <> SET("139","163")) AND ([CTR49] <> SET("62230","63100","87900","95599","95510","95591","99010","99020","99030","99040","99050","99060","99070")) AND ([CTR190] = NULL OR [CTR190] = 0) AND ([CTR320] < (([CTR120] * 0.30) – 1))))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR375 = NULL OR ^CTR375 =  FALSE) AND (((^CTR120 <= 0)  AND (^CTR320 > 0) AND (^CTR49 <> '63100')) OR ((^CTR120 > 0) AND (NotInSet(^CTR52, '"139","163"')) AND (NotInSet(^CTR49, '"62230","63100","87900","95599","95510","95591","99010","99020","99030","99040","99050","99060","99070"')) AND (^CTR190 = NULL OR ^CTR190 = 0) AND (^CTR320 < ((^CTR120 * 0.30) - 1))))
    
            Data Elements:
    
            ^CTR320 = RP:IncomeTax.TotalTaxOnTaxableIncome.Amount
    
            ^CTR375 = RP:Elections.BaseRateEntityElection.Indicator
    
            ^CTR120 = RP:Income.Taxable.Amount
    
            ^CTR49 = RP:OrganisationDetails.OrganisationIndustry2006Extended.Code
    
            ^CTR52 = RP:OrganisationDetails.OrganisationType.Code
    
            ^CTR190 = RP:Income.RetirementSavingsAccountsTaxableNet.Amount
            */
            assertion = ((report.CTR375 == null || report.CTR375 == false) && (report.CTR120.GetValueOrDefault() <= 0 && report.CTR320.GetValueOrDefault() > 0 && report.CTR49 != @"63100" || report.CTR120.GetValueOrDefault() > 0 && !(IsMatch(report.CTR52, @"^(139|163)$", RegexOptions.IgnoreCase)) && !(IsMatch(report.CTR49, @"^(62230|63100|87900|95599|95510|95591|99010|99020|99030|99040|99050|99060|99070)$", RegexOptions.IgnoreCase)) && (report.CTR190 == null || report.CTR190 == 0) && report.CTR320.GetValueOrDefault() < report.CTR120.GetValueOrDefault() * 0.30M - 1));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.430225",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax on taxable or net income incorrect. Check rate calculation and Company entity type",
                    LongDescription = @"The Company is not an entity that has been identified as being allowed to have a tax rate of less than 30%. Check the rate calculation and company entity type. (Calculation statement Tax on taxable or net income)",
                    Location = "/xbrli:xbrl/tns:IncomeTax.TotalTaxOnTaxableIncome.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.500397"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR375", Value = GetValueOrEmpty(report.CTR375) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR120", Value = GetValueOrEmpty(report.CTR120) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR320", Value = GetValueOrEmpty(report.CTR320) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR49", Value = report.CTR49 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR52", Value = report.CTR52 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR190", Value = GetValueOrEmpty(report.CTR190) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.500398
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR500398()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.500398
            If the base rate entity indicator response is TRUE, then Tax on taxable or net income should not be less than 25% of Taxable/Net income when lodging for the 2022, or future year of return.
    
            Legacy Rule Format:
            IF [CTR375] = TRUE AND ((([CTR120] <= 0)  AND ([CTR320] > 0) AND ([CTR49] <> "63100")) OR (([CTR120] > 0) AND ([CTR52] <> SET("139","163")) AND ([CTR49] <> SET("62230","63100","87900","95599","95510","95591","99010","99020","99030","99040","99050","99060","99070")) AND ([CTR320] < (([CTR120] * 0.25) – 1)))) 
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR375 = TRUE AND (((^CTR120 <= 0)  AND (^CTR320 > 0) AND (^CTR49 <> '63100')) OR ((^CTR120 > 0) AND (NotInSet(^CTR52, '"139","163"')) AND (NotInSet(^CTR49, '"62230","63100","87900","95599","95510","95591","99010","99020","99030","99040","99050","99060","99070"')) AND (^CTR320 < ((^CTR120 * 0.25) - 1))))
    
            Data Elements:
    
            ^CTR320 = RP:IncomeTax.TotalTaxOnTaxableIncome.Amount
    
            ^CTR375 = RP:Elections.BaseRateEntityElection.Indicator
    
            ^CTR120 = RP:Income.Taxable.Amount
    
            ^CTR49 = RP:OrganisationDetails.OrganisationIndustry2006Extended.Code
    
            ^CTR52 = RP:OrganisationDetails.OrganisationType.Code
            */
            assertion = (report.CTR375 == true && (report.CTR120.GetValueOrDefault() <= 0 && report.CTR320.GetValueOrDefault() > 0 && report.CTR49 != @"63100" || report.CTR120.GetValueOrDefault() > 0 && !(IsMatch(report.CTR52, @"^(139|163)$", RegexOptions.IgnoreCase)) && !(IsMatch(report.CTR49, @"^(62230|63100|87900|95599|95510|95591|99010|99020|99030|99040|99050|99060|99070)$", RegexOptions.IgnoreCase)) && report.CTR320.GetValueOrDefault() < report.CTR120.GetValueOrDefault() * 0.25M - 1));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.468096",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax on taxable or net income incorrect. Check rate calculation and Company entity type.",
                    LongDescription = @"The Company is not an entity that has been identified as being allowed to have a tax rate of less than 25% for the 2022, or future year of return. Check the rate calculation and company entity type. (Calculation statement Tax on taxable or net income)",
                    Location = "/xbrli:xbrl/tns:IncomeTax.TotalTaxOnTaxableIncome.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.500398"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR375", Value = GetValueOrEmpty(report.CTR375) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR120", Value = GetValueOrEmpty(report.CTR120) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR320", Value = GetValueOrEmpty(report.CTR320) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR49", Value = report.CTR49 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR52", Value = report.CTR52 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.500399
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR500399()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.500399
            Business Industry code is not a valid code
    
            Legacy Rule Format:
            IF ([CTR49] <> NULLORBLANK) AND ([CTR49] <> SET(DOMAIN(BUSINESS INDUSTRY CODES)))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR49 <> BLANK) AND (NotInDomain(^CTR49, 'BUSINESS INDUSTRY CODES'))
    
            Data Elements:
    
            ^CTR49 = RP:OrganisationDetails.OrganisationIndustry2006Extended.Code
            */
            assertion = (string.IsNullOrWhiteSpace(report.CTR49) != true && !(IsMatch(report.CTR49, @"^(01110|01120|01130|01140|01150|01210|01220|01230|01310|01320|01330|01340|01350|01360|01370|01390|01410|01420|01430|01440|01450|01460|01490|01510|01520|01590|01600|01710|01720|01800|01910|01920|01930|01990|02011|02019|02020|02031|02039|03010|03020|04111|04112|04120|04130|04140|04191|04192|04199|04200|05100|05210|05220|05290|06000|07000|08010|08020|08030|08040|08050|08060|08070|08090|09110|09190|09901|09902|09903|09909|10111|10112|10121|10122|10900|11110|11120|11130|11200|11310|11320|11330|11400|11500|11610|11620|11710|11720|11730|11740|11810|11820|11910|11920|11990|12110|12120|12130|12140|12200|13110|13120|13130|13200|13310|13320|13330|13340|13400|13510|13520|14110|14120|14130|14910|14920|14930|14940|14990|15100|15210|15220|15230|15240|15290|16110|16120|16200|17010|17090|18110|18120|18130|18210|18290|18310|18320|18410|18420|18510|18520|18910|18920|18990|19110|19120|19130|19140|19150|19160|19190|19200|20100|20210|20290|20310|20320|20330|20340|20900|21100|21210|21220|21310|21320|21330|21390|21410|21420|21490|22100|22210|22220|22230|22240|22290|22310|22390|22400|22910|22920|22930|22990|23110|23120|23130|23190|23910|23920|23930|23940|23990|24110|24120|24190|24211|24219|24220|24290|24310|24320|24390|24410|24490|24510|24520|24610|24620|24630|24690|24910|24990|25110|25120|25130|25190|25910|25920|25990|26110|26120|26190|26200|26300|26400|27000|28110|28120|29110|29190|29210|29220|30110|30190|30200|31010|31091|31099|32110|32120|32210|32220|32230|32240|32310|32320|32330|32340|32390|32410|32420|32430|32440|32450|32910|32920|32990|33110|33120|33190|33210|33221|33229|33230|33310|33320|33390|34110|34190|34910|34920|34930|34940|34990|35010|35020|35030|35040|35050|36010|36020|36030|36040|36050|36061|36062|36090|37110|37120|37200|37310|37320|37330|37340|37350|37360|37390|38000|39110|39120|39130|39211|39219|39220|40000|41100|41211|41212|41213|41220|41230|41290|42110|42120|42130|42140|42210|42220|42290|42310|42320|42410|42420|42430|42441|42442|42450|42510|42520|42530|42590|42600|42711|42712|42720|42731|42732|42733|42740|42791|42792|42793|42794|42799|43101|43102|43109|43201|43202|43209|44000|45110|45120|45130|45200|45301|45302|46100|46210|46220|46231|46239|47100|47200|48100|48200|49001|49009|50100|50210|50290|51010|51020|52110|52120|52190|52200|52910|52920|52991|52999|53010|53090|54110|54120|54130|54140|54190|54200|55110|55120|55130|55140|55210|55220|56100|56210|56220|57000|58010|58020|58090|59100|59210|59220|60100|60200|62100|62210|62220|62230|62290|62300|62400|63100|63210|63220|63300|64110|64190|64200|66110|66190|66200|66310|66320|66390|66400|67110|67120|67200|69100|69210|69220|69230|69240|69250|69310|69320|69400|69500|69610|69621|69622|69629|69700|69910|69990|70000|72110|72120|72200|72910|72920|72930|72940|72990|73110|73120|73130|73200|75100|75200|75300|75400|75510|75520|76000|77110|77120|77130|77140|77190|77200|80100|80210|80220|80230|80240|81011|81019|81020|82110|82120|82191|82199|82200|84010|84020|85110|85121|85122|85123|85124|85125|85129|85201|85202|85203|85209|85310|85320|85330|85340|85391|85399|85910|85990|86011|86012|86090|87100|87900|89100|89210|89220|90010|90021|90029|90030|91110|91121|91122|91123|91124|91125|91126|91129|91131|91139|91140|91210|91291|91292|91299|91310|91390|92010|92020|92091|92092|92099|94110|94121|94129|94191|94192|94193|94199|94210|94220|94290|94910|94990|95110|95120|95200|95310|95320|95330|95340|95391|95392|95393|95399|95400|95510|95520|95591|95599|96010|96020|96030|98000|99010|99020|99030|99040|99050|99060|99070)$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.500399",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Business Industry code is not a valid code",
                    Location = "/xbrli:xbrl/tns:OrganisationDetails.OrganisationIndustry2006Extended.Code",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.500399"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR49", Value = report.CTR49 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.500400
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR500400()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.500400
            You must enter a valid amount of $0 or more in the 'Build to Rent Capital Works Deductions at 4%' when Capital Works Deductions is $250,000 or more or Business Entity Industry Code indicates a residential property-related industry
    
            Legacy Rule Format:
            IF ([CTR103] >= 250000 OR [CTR49] = SET("67110","30190")) AND [CTR456] = NULL
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR103 >= 250000 OR InSet(^CTR49, '"67110","30190"')) AND ^CTR456 = NULL
    
            Data Elements:
    
            ^CTR456 = RP:Expense.BuildToRentCapitalWorksDeduction.Amount
    
            ^CTR103 = RP:Expense.CapitalWorksDeduction.Amount
    
            ^CTR49 = RP:OrganisationDetails.OrganisationIndustry2006Extended.Code
            */
            assertion = ((report.CTR103.GetValueOrDefault() >= 250000 || IsMatch(report.CTR49, @"^(67110|30190)$")) && report.CTR456 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.500400",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Build to rent capital works deductions at 4% amount must be provided",
                    LongDescription = @"You must enter a valid amount of $0 or more in the 'Build to Rent Capital Works Deductions at 4%' when Capital Works Deductions is $250,000 or more or Business Entity Industry Code indicates a residential property-related industry",
                    Location = "/xbrli:xbrl/tns:Expense.BuildToRentCapitalWorksDeduction.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.500400"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR103", Value = GetValueOrEmpty(report.CTR103) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR49", Value = report.CTR49 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR456", Value = GetValueOrEmpty(report.CTR456) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.W00003
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTRW00003()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.W00003
            If the rollover applied code is “T - Small Business Restructure Roll-over”, total business income must not exceed threshold.
    
            Legacy Rule Format:
            IF ANY OCCURRENCE OF ([CTR330]) = "T - Small Business Restructure Roll-over" AND (([CTR57] + [CTR58] + [CTR59] + [CTR60] + [CTR61] + [CTR63] + [CTR64] + [CTR65] + [CTR66] + [CTR67] + [CTR68] + [CTR70] + [CTR71]) > 10000000)
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            AnyOccurrence(^CTR330, ^CTR330 = 'T - Small Business Restructure Roll-over') AND ((^CTR57 + ^CTR58 + ^CTR59 + ^CTR60 + ^CTR61 + ^CTR63 + ^CTR64 + ^CTR65 + ^CTR66 + ^CTR67 + ^CTR68 + ^CTR70 + ^CTR71) > 10000000)
    
            Data Elements:
    
            ^CTR330 = RP:CapitalGainsTax.ExemptionOrRolloverApplied.Code
    
            ^CTR57 = RP:Remuneration.PaymentToForeignResidentGross.Amount
    
            ^CTR58 = RP:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^CTR59 = RP:Income.SaleOfGoodsAndServices.Other.Amount
    
            ^CTR60 = RP:Income.PartnershipDistributionGross.Amount
    
            ^CTR61 = RP:TrustDistributionIncome:Income.TrustDistributionGross.Amount
    
            ^CTR63 = RP:Income.ForestryManagedInvestmentScheme.Amount
    
            ^CTR64 = RP:Income.Interest.Gross.Amount
    
            ^CTR65 = RP:Income.RentLeasingHiringGross.Amount
    
            ^CTR66 = RP:Income.DividendsTotal.Amount
    
            ^CTR67 = RP:Remuneration.FringeBenefits.EmployeeContribution.Amount
    
            ^CTR68 = RP:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^CTR70 = RP:Income.AssetRevaluationsUnrealisedGains.Amount
    
            ^CTR71 = RP:Income.OtherGross.Amount
            */
            assertion = ((report.CTR330Collection == null ? false : report.CTR330Collection.Any(CTR330Repeat => CTR330Repeat == @"T - Small Business Restructure Roll-over")) && report.CTR57.GetValueOrDefault() + report.CTR58.GetValueOrDefault() + report.CTR59.GetValueOrDefault() + report.CTR60.GetValueOrDefault() + report.CTR61.GetValueOrDefault() + report.CTR63.GetValueOrDefault() + report.CTR64.GetValueOrDefault() + report.CTR65.GetValueOrDefault() + report.CTR66.GetValueOrDefault() + report.CTR67.GetValueOrDefault() + report.CTR68.GetValueOrDefault() + report.CTR70.GetValueOrDefault() + report.CTR71.GetValueOrDefault() > 10000000);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.W00003",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Total business income exceeds threshold.",
                    LongDescription = @"If the rollover applied code is ""T - Small Business Restructure Roll-over"", each party to the transfer both the transferor and transferee entities must be either a small business entity, or an entity that is either (a) an affiliate, (b) connected with an entity, or (c) a partner in a partnership, that is a small business entity.",
                    Location = "/xbrli:xbrl/tns:CapitalGainsTax.ExemptionOrRolloverApplied.Code",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.W00003"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR330", Value = "CTR330" });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR57", Value = GetValueOrEmpty(report.CTR57) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR58", Value = GetValueOrEmpty(report.CTR58) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR59", Value = GetValueOrEmpty(report.CTR59) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR60", Value = GetValueOrEmpty(report.CTR60) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR61", Value = GetValueOrEmpty(report.CTR61) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR63", Value = GetValueOrEmpty(report.CTR63) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR64", Value = GetValueOrEmpty(report.CTR64) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR65", Value = GetValueOrEmpty(report.CTR65) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR66", Value = GetValueOrEmpty(report.CTR66) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR67", Value = GetValueOrEmpty(report.CTR67) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR68", Value = GetValueOrEmpty(report.CTR68) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR70", Value = GetValueOrEmpty(report.CTR70) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR71", Value = GetValueOrEmpty(report.CTR71) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.W00004
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTRW00004()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.W00004
            Non-refundable carry forward tax offsets amount should be supplied if any of the innovation tax offset amounts are supplied and 'Subtotal 1' amount is not equal to zero.
    
            Legacy Rule Format:
            IF ([CTR323] = NULL AND (([CTR368] <> NULL OR [CTR374] <> NULL OR [CTR369] <> NULL OR [CTR373] <> NULL) AND ([CTR322] <> NULL AND [CTR322] <> 0)))
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR323 = NULL AND ((^CTR368 <> NULL OR ^CTR374 <> NULL OR ^CTR369 <> NULL OR ^CTR373 <> NULL) AND (^CTR322 <> NULL AND ^CTR322 <> 0)))
    
            Data Elements:
    
            ^CTR323 = RP:IncomeTax.TaxOffsetNonRefundableCarryForward.Amount
    
            ^CTR368 = RP.Investor:TaxOffsetClaim.NonRefundableOther.Amount
    
            ^CTR374 = RP.Investor.YPrevious:TaxOffsetClaim.NonRefundableOther.Amount
    
            ^CTR369 = RP.LimitedPartners:TaxOffsetClaim.NonRefundableOther.Amount
    
            ^CTR373 = RP.LimitedPartners.YPrevious:TaxOffsetClaim.NonRefundableOther.Amount
    
            ^CTR322 = RP:IncomeTax.TaxOffsetNonRefundableSubTotal.Amount
            */
            assertion = (report.CTR323 == null && ((report.CTR368 != null || report.CTR374 != null || report.CTR369 != null || report.CTR373 != null) && (report.CTR322 != null && report.CTR322 != 0)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.W00004",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Non-refundable carry forward tax offsets amount should be supplied.",
                    LongDescription = @"Non-refundable carry forward tax offsets amount should be supplied if there is an amount supplied at ""Early stage venture capital limited partnership tax offset"", ""Early stage investor tax offset"", ""Early Stage Venture Capital Limited Partnership (ESVCLP) tax offset carried forward from previous year"" or ""Early stage investor tax offset carried forward from previous year"" and ""Subtotal 1"" amount is not equal to zero.",
                    Location = "/xbrli:xbrl/tns:IncomeTax.TaxOffsetNonRefundableCarryForward.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.W00004"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR323", Value = GetValueOrEmpty(report.CTR323) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR368", Value = GetValueOrEmpty(report.CTR368) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR374", Value = GetValueOrEmpty(report.CTR374) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR369", Value = GetValueOrEmpty(report.CTR369) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR373", Value = GetValueOrEmpty(report.CTR373) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR322", Value = GetValueOrEmpty(report.CTR322) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.W00005
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTRW00005()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.W00005
            If the rollover applied code is “T - Small Business Restructure Roll-over”, then the taxpayer must be an Australian resident unless eligible under certain exceptions.
    
            Legacy Rule Format:
            IF ANY OCCURRENCE OF([CTR330]) = "T - Small Business Restructure Roll-over" AND [CTR51] <> "Resident"
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            AnyOccurrence(^CTR330, ^CTR330 = 'T - Small Business Restructure Roll-over') AND ^CTR51 <> 'Resident'
    
            Data Elements:
    
            ^CTR330 = RP:CapitalGainsTax.ExemptionOrRolloverApplied.Code
    
            ^CTR51 = RP:Residency.TaxPurposesOrganisationStatus.Code
            */
            assertion = ((report.CTR330Collection == null ? false : report.CTR330Collection.Any(CTR330Repeat => CTR330Repeat == @"T - Small Business Restructure Roll-over")) && report.CTR51 != @"Resident");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.W00005",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Taxpayer must be an Australian resident, unless eligible under certain exceptions.",
                    LongDescription = @"If the rollover applied code is ""T - Small Business Restructure Roll-over"", then the taxpayer must be an Australian resident unless eligible under certain exceptions.",
                    Location = "/xbrli:xbrl/tns:CapitalGainsTax.ExemptionOrRolloverApplied.Code",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.W00005"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR330", Value = "CTR330" });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR51", Value = GetValueOrEmpty(report.CTR51) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.W00006
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTRW00006()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.W00006
            If there are amounts present at 'Exempt income' and 'Non deductible exempt income expenditure', the carried forward ESVCLP and/or Early stage investor tax offset amounts from previous years may been to be adjusted.
    
            Legacy Rule Format:
            IF [CTR322] > 0 AND (([CTR114] > [CTR92]) AND ([CTR353] > 0 OR [CTR373] > 0 OR [CTR374] > 0))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR322 > 0 AND ((^CTR114 > ^CTR92) AND (^CTR353 > 0 OR ^CTR373 > 0 OR ^CTR374 > 0))
    
            Data Elements:
    
            ^CTR322 = RP:IncomeTax.TaxOffsetNonRefundableSubTotal.Amount
    
            ^CTR114 = RP:Income.Exempt.Amount
    
            ^CTR92 = RP:Expense.NonDeductibleExemptIncome.Amount
    
            ^CTR353 = RP.YPrevious:TaxConcession.ResearchAndDevelopment.TaxOffsetNonRefundable.Amount
    
            ^CTR373 = RP.LimitedPartners.YPrevious:TaxOffsetClaim.NonRefundableOther.Amount
    
            ^CTR374 = RP.Investor.YPrevious:TaxOffsetClaim.NonRefundableOther.Amount
            */
            assertion = (report.CTR322.GetValueOrDefault() > 0 && (report.CTR114.GetValueOrDefault() > report.CTR92.GetValueOrDefault() && (report.CTR353.GetValueOrDefault() > 0 || report.CTR373.GetValueOrDefault() > 0 || report.CTR374.GetValueOrDefault() > 0)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.W00006",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Ensure the calculation of non-refundable carry forward tax offsets has considered net exempt income.",
                    Location = "/xbrli:xbrl/tns:IncomeTax.TaxOffsetNonRefundableSubTotal.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.W00006"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR322", Value = GetValueOrEmpty(report.CTR322) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR114", Value = GetValueOrEmpty(report.CTR114) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR92", Value = GetValueOrEmpty(report.CTR92) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR353", Value = GetValueOrEmpty(report.CTR353) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR373", Value = GetValueOrEmpty(report.CTR373) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR374", Value = GetValueOrEmpty(report.CTR374) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.W00008
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTRW00008()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.W00008
            If the company has self-assessed that they are a Significant Global Entity (SGE), please confirm if they are a Country by Country Reporting Entity (CbCRE).
    
            Legacy Rule Format:
            IF [CTR378] = NULL AND [CTR370] = TRUE
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR378 = NULL AND ^CTR370 = TRUE
    
            Data Elements:
    
            ^CTR378 = RP:InternationalDealings.CountryByCountryReportingEntityStatus.Indicator
    
            ^CTR370 = RP:OrganisationDetails.SignificantGlobalEntityStatus.Indicator
            */
            assertion = (report.CTR378 == null && report.CTR370 == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.W00008",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Country by Country Reporting Entity (CBCRE) indicator should be supplied.",
                    Location = "/xbrli:xbrl/tns:InternationalDealings.CountryByCountryReportingEntityStatus.Indicator",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.W00008"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR378", Value = GetValueOrEmpty(report.CTR378) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR370", Value = GetValueOrEmpty(report.CTR370) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.W00013
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTRW00013()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.W00013
            If Aggregated turnover range is "P", then the Aggregated turnover should be $1 billion or more
    
            Legacy Rule Format:
            IF [CTR392] = "P" AND [CTR393] < 1000000000
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR392 = 'P' AND ^CTR393 < 1000000000
    
            Data Elements:
    
            ^CTR393 = RP:Income.AggregatedTurnover.Amount
    
            ^CTR392 = RP:Income.AggregatedTurnoverRange.Code
            */
            assertion = (report.CTR392 == @"P" && report.CTR393.GetValueOrDefault() < 1000000000);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.W00013",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Aggregated turnover value should be more than $1 billion",
                    LongDescription = @"If 'P' is selected at Aggregated turnover range, then the aggregated turnover value should be more than $1 billion",
                    Location = "/xbrli:xbrl/tns:Income.AggregatedTurnover.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.W00013"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR392", Value = GetValueOrEmpty(report.CTR392) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR393", Value = GetValueOrEmpty(report.CTR393) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.W00018
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTRW00018()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.W00018
            If Total income is greater than zero, then Depreciation expenses should be completed and the amount must be at least 0 or higher.
    
            Legacy Rule Format:
            IF ([CTR57]+[CTR58]+[CTR59]+[CTR60]+[CTR61]+[CTR63]+[CTR64]+[CTR65]+[CTR66]+[CTR67]+[CTR68]+[CTR70]+[CTR71]) > 0 AND [CTR84] = NULL
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR57+^CTR58+^CTR59+^CTR60+^CTR61+^CTR63+^CTR64+^CTR65+^CTR66+^CTR67+^CTR68+^CTR70+^CTR71) > 0 AND ^CTR84 = NULL
    
            Data Elements:
    
            ^CTR84 = RP:Expense.DepreciationAndAmortisation.Amount
    
            ^CTR57 = RP:Remuneration.PaymentToForeignResidentGross.Amount
    
            ^CTR58 = RP:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^CTR59 = RP:Income.SaleOfGoodsAndServices.Other.Amount
    
            ^CTR60 = RP:Income.PartnershipDistributionGross.Amount
    
            ^CTR61 = RP:TrustDistributionIncome:Income.TrustDistributionGross.Amount
    
            ^CTR63 = RP:Income.ForestryManagedInvestmentScheme.Amount
    
            ^CTR64 = RP:Income.Interest.Gross.Amount
    
            ^CTR65 = RP:Income.RentLeasingHiringGross.Amount
    
            ^CTR66 = RP:Income.DividendsTotal.Amount
    
            ^CTR67 = RP:Remuneration.FringeBenefits.EmployeeContribution.Amount
    
            ^CTR68 = RP:GovernmentFunding.GovernmentIndustryPaymentsAssessable.Amount
    
            ^CTR70 = RP:Income.AssetRevaluationsUnrealisedGains.Amount
    
            ^CTR71 = RP:Income.OtherGross.Amount
            */
            assertion = (report.CTR57.GetValueOrDefault() + report.CTR58.GetValueOrDefault() + report.CTR59.GetValueOrDefault() + report.CTR60.GetValueOrDefault() + report.CTR61.GetValueOrDefault() + report.CTR63.GetValueOrDefault() + report.CTR64.GetValueOrDefault() + report.CTR65.GetValueOrDefault() + report.CTR66.GetValueOrDefault() + report.CTR67.GetValueOrDefault() + report.CTR68.GetValueOrDefault() + report.CTR70.GetValueOrDefault() + report.CTR71.GetValueOrDefault() > 0 && report.CTR84 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.W00018",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Depreciation expenses should be completed if Total income is greater than zero.",
                    LongDescription = @"If Total income is greater than zero, then Depreciation expenses should be completed and the amount must be at least 0 or higher.",
                    Location = "/xbrli:xbrl/tns:Expense.DepreciationAndAmortisation.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.W00018"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR57", Value = GetValueOrEmpty(report.CTR57) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR58", Value = GetValueOrEmpty(report.CTR58) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR59", Value = GetValueOrEmpty(report.CTR59) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR60", Value = GetValueOrEmpty(report.CTR60) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR61", Value = GetValueOrEmpty(report.CTR61) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR63", Value = GetValueOrEmpty(report.CTR63) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR64", Value = GetValueOrEmpty(report.CTR64) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR65", Value = GetValueOrEmpty(report.CTR65) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR66", Value = GetValueOrEmpty(report.CTR66) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR67", Value = GetValueOrEmpty(report.CTR67) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR68", Value = GetValueOrEmpty(report.CTR68) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR70", Value = GetValueOrEmpty(report.CTR70) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR71", Value = GetValueOrEmpty(report.CTR71) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR84", Value = GetValueOrEmpty(report.CTR84) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.W00019
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTRW00019()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.W00019
            If Depreciation expenses is greater than zero, then Non deductible expenses should be completed and the amount must be at least 0 or higher.
    
            Legacy Rule Format:
            IF [CTR363] <> TRUE AND [CTR84] > 0 AND [CTR96] = NULL
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR363 <> TRUE AND ^CTR84 > 0 AND ^CTR96 = NULL
    
            Data Elements:
    
            ^CTR96 = RP:Expense.NonDeductible.Amount
    
            ^CTR363 = RP:Elections.SmallBusinessEntitiesElection.Indicator
    
            ^CTR84 = RP:Expense.DepreciationAndAmortisation.Amount
            */
            assertion = (report.CTR363 != true && report.CTR84.GetValueOrDefault() > 0 && report.CTR96 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.W00019",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Non deductible expenses should be completed if Depreciation expenses is greater than zero.",
                    LongDescription = @"If Depreciation expenses is greater than zero, then Non deductible expenses should be completed and the amount must be at least 0 or higher.",
                    Location = "/xbrli:xbrl/tns:Expense.NonDeductible.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.W00019"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR363", Value = GetValueOrEmpty(report.CTR363) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR84", Value = GetValueOrEmpty(report.CTR84) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR96", Value = GetValueOrEmpty(report.CTR96) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.W00020
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTRW00020()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.W00020
            If Depreciation expenses is completed, then Deduction for decline in value of depreciating assets should also be completed and the amount must be at least 0 or higher.
    
            Legacy Rule Format:
            IF [CTR363] <> TRUE AND [CTR84] <> NULL AND [CTR99] = NULL
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR363 <> TRUE AND ^CTR84 <> NULL AND ^CTR99 = NULL
    
            Data Elements:
    
            ^CTR99 = RP:Expense.DepreciationAllowableDeduction.Amount
    
            ^CTR363 = RP:Elections.SmallBusinessEntitiesElection.Indicator
    
            ^CTR84 = RP:Expense.DepreciationAndAmortisation.Amount
            */
            assertion = (report.CTR363 != true && report.CTR84 != null && report.CTR99 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.W00020",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Deduction for decline in value of depreciating assets should be completed if Depreciation expenses is completed.",
                    LongDescription = @"If Depreciation expenses is completed, then Deduction for decline in value of depreciating assets should also be completed and the amount must be at least 0 or higher.",
                    Location = "/xbrli:xbrl/tns:Expense.DepreciationAllowableDeduction.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.W00020"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR363", Value = GetValueOrEmpty(report.CTR363) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR84", Value = GetValueOrEmpty(report.CTR84) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR99", Value = GetValueOrEmpty(report.CTR99) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.W00021
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTRW00021()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.W00021
            If Deduction for decline in value of depreciating assets is completed, then Capital works deductions and Total adjustable values at end income year should be completed and the amount must be at least 0 or higher.
    
            Legacy Rule Format:
            IF [CTR363] <> TRUE AND [CTR99] <> NULL AND ([CTR103] = NULL OR [CTR333] = NULL)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR363 <> TRUE AND ^CTR99 <> NULL AND (^CTR103 = NULL OR ^CTR333 = NULL)
    
            Data Elements:
    
            ^CTR103 = RP:Expense.CapitalWorksDeduction.Amount
    
            ^CTR363 = RP:Elections.SmallBusinessEntitiesElection.Indicator
    
            ^CTR99 = RP:Expense.DepreciationAllowableDeduction.Amount
    
            ^CTR333 = RP:Assets.DepreciatingAssets.AdjustableValuesTotal.Amount
            */
            assertion = (report.CTR363 != true && report.CTR99 != null && (report.CTR103 == null || report.CTR333 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.W00021",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Capital works deductions and Total adjustable values at end income year should be completed if Deduction for decline in value of depreciating assets is completed.",
                    LongDescription = @"If Deduction for decline in value of depreciating assets is completed, then Capital works deductions and Total adjustable values at end income year should be completed and the amount must be at least 0 or higher.",
                    Location = "/xbrli:xbrl/tns:Expense.CapitalWorksDeduction.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.W00021"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR363", Value = GetValueOrEmpty(report.CTR363) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR99", Value = GetValueOrEmpty(report.CTR99) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR103", Value = GetValueOrEmpty(report.CTR103) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR333", Value = GetValueOrEmpty(report.CTR333) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.W00023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTRW00023()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.W00023
            If Deduction for decline in value of depreciating assets is greater than zero, then 'Have you self-assessed the effective life of any of these assets?' and 'Did you recalculate the effective life for any of your assets this income year?' fields should be answered Yes/No.
    
            Legacy Rule Format:
            IF [CTR363] <> TRUE AND [CTR99] > 0 AND ([CTR331] = NULL OR [CTR332] = NULL)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR363 <> TRUE AND ^CTR99 > 0 AND (^CTR331 = NULL OR ^CTR332 = NULL)
    
            Data Elements:
    
            ^CTR331 = RP:Assets.PropertyPlantAndEquipment.DepreciatingAssets.FirstDeductedSelfAssessedEffectiveLife.Indicator
    
            ^CTR363 = RP:Elections.SmallBusinessEntitiesElection.Indicator
    
            ^CTR99 = RP:Expense.DepreciationAllowableDeduction.Amount
    
            ^CTR332 = RP:Assets.PropertyPlantAndEquipment.DepreciatingAssets.EffectiveLifeRecalculation.Indicator
            */
            assertion = (report.CTR363 != true && report.CTR99.GetValueOrDefault() > 0 && (report.CTR331 == null || report.CTR332 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.W00023",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Have you self-assessed the effective life of any of these assets?' and 'Did you recalculate the effective life for any of your assets this income year?' fields should be answered if Deduction for decline in value of depreciating assets is greater than zero.",
                    LongDescription = @"If Deduction for decline in value of depreciating assets is greater than zero, then 'Have you self-assessed the effective life of any of these assets?' and 'Did you recalculate the effective life for any of your assets this income year?' fields should be answered Yes/No.",
                    Location = "/xbrli:xbrl/tns:Assets.PropertyPlantAndEquipment.DepreciatingAssets.FirstDeductedSelfAssessedEffectiveLife.Indicator",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.W00023"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR363", Value = GetValueOrEmpty(report.CTR363) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR99", Value = GetValueOrEmpty(report.CTR99) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR331", Value = GetValueOrEmpty(report.CTR331) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR332", Value = GetValueOrEmpty(report.CTR332) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.W00024
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTRW00024()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.W00024
            If Deduction for project pool is greater than zero and the Business Industry code is in the range of mining codes (06000 to 10900), then Total mining capital expenditure and/or transport capital that you allocated to a project pool and for which you can claim a deduction this income year should be completed.
    
            Legacy Rule Format:
            IF [CTR102] > 0 AND [CTR49] = SET ("06000","07000","08010","08020","08030","08040","08050","08060","08070","08090","09110","09190","09901","09902","09903","09909","10111","10112","10121","10122","10900") AND [CTR336] = NULL
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR102 > 0 AND InSet(^CTR49, '"06000","07000","08010","08020","08030","08040","08050","08060","08070","08090","09110","09190","09901","09902","09903","09909","10111","10112","10121","10122","10900"') AND ^CTR336 = NULL
    
            Data Elements:
    
            ^CTR336 = RP:Expense.DepreciatingAssets.ProjectPoolAllocationsForMiningAndTransportCapitalExpenditure.Amount
    
            ^CTR102 = RP:Expense.ProjectPoolAllowableDeduction.Amount
    
            ^CTR49 = RP:OrganisationDetails.OrganisationIndustry2006Extended.Code
            */
            assertion = (report.CTR102.GetValueOrDefault() > 0 && IsMatch(report.CTR49, @"^(06000|07000|08010|08020|08030|08040|08050|08060|08070|08090|09110|09190|09901|09902|09903|09909|10111|10112|10121|10122|10900)$") && report.CTR336 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.W00024",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Total mining capital expenditure and/or transport capital that you allocated to a project pool and for which you can claim a deduction this income year should be completed if Deduction for project pool is greater than zero and the Business Industry code is in the range of mining codes (06000 to 10900).",
                    LongDescription = @"If Deduction for project pool is greater than zero and the Business Industry code is in the range of mining codes (06000 to 10900), then Total mining capital expenditure and/or transport capital that you allocated to a project pool and for which you can claim a deduction this income year should be completed.",
                    Location = "/xbrli:xbrl/tns:Expense.DepreciatingAssets.ProjectPoolAllocationsForMiningAndTransportCapitalExpenditure.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.W00024"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR102", Value = GetValueOrEmpty(report.CTR102) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR49", Value = report.CTR49 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR336", Value = GetValueOrEmpty(report.CTR336) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.W00025
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTRW00025()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.W00025
            If Depreciation expenses is greater than zero, then the sum of Deduction for certain assets and Deduction for general small business pool should be equal to or less than the Depreciation expenses amount
    
            Legacy Rule Format:
            IF [CTR84] > 0 AND ([CTR84] < [CTR168] + [CTR169])
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR84 > 0 AND (^CTR84 < ^CTR168 + ^CTR169)
    
            Data Elements:
    
            ^CTR84 = RP:Expense.DepreciationAndAmortisation.Amount
    
            ^CTR168 = RP.LowCost:Expense.DepreciatingAssets.SmallBusinessEntity.Amount
    
            ^CTR169 = RP.GeneralPool:Expense.DepreciatingAssets.SmallBusinessEntity.Amount
            */
            assertion = (report.CTR84.GetValueOrDefault() > 0 && report.CTR84.GetValueOrDefault() < report.CTR168.GetValueOrDefault() + report.CTR169.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.W00025",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"The sum of Deduction for certain assets and Deduction for general small business pool should be equal to or less than the Depreciation expenses amount.",
                    LongDescription = @"If Depreciation expenses is greater than zero, then the sum of Deduction for certain assets and Deduction for general small business pool should be equal to or less than the Depreciation expenses amount",
                    Location = "/xbrli:xbrl/tns:Expense.DepreciationAndAmortisation.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.W00025"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR84", Value = GetValueOrEmpty(report.CTR84) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR168", Value = GetValueOrEmpty(report.CTR168) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR169", Value = GetValueOrEmpty(report.CTR169) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.W00026
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTRW00026()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.W00026
            If Deduction for decline in value of depreciating assets is greater than zero, then questions regarding depreciating assets should be answered.
    
            Legacy Rule Format:
            IF [CTR363] <> TRUE AND [CTR99] > 0 AND ([CTR147] = NULL OR [CTR148] = NULL OR [CTR331] = NULL OR [CTR332] = NULL OR [CTR333] = NULL OR [CTR334] = NULL OR [CTR335] = NULL OR [CTR149] = NULL OR [CTR150] = NULL)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR363 <> TRUE AND ^CTR99 > 0 AND (^CTR147 = NULL OR ^CTR148 = NULL OR ^CTR331 = NULL OR ^CTR332 = NULL OR ^CTR333 = NULL OR ^CTR334 = NULL OR ^CTR335 = NULL OR ^CTR149 = NULL OR ^CTR150 = NULL)
    
            Data Elements:
    
            ^CTR148 = RP:Expense.DepreciatingAssets.OtherFirstDeducted.Amount
    
            ^CTR363 = RP:Elections.SmallBusinessEntitiesElection.Indicator
    
            ^CTR99 = RP:Expense.DepreciationAllowableDeduction.Amount
    
            ^CTR147 = RP:Expense.DepreciatingAssets.IntangibleFirstDeducted.Amount
    
            ^CTR331 = RP:Assets.PropertyPlantAndEquipment.DepreciatingAssets.FirstDeductedSelfAssessedEffectiveLife.Indicator
    
            ^CTR332 = RP:Assets.PropertyPlantAndEquipment.DepreciatingAssets.EffectiveLifeRecalculation.Indicator
    
            ^CTR333 = RP:Assets.DepreciatingAssets.AdjustableValuesTotal.Amount
    
            ^CTR334 = RP:Income.DepreciatingAssets.IntangibleBalancingAdjustment.Amount
    
            ^CTR335 = RP:Expense.DepreciatingAssets.IntangibleBalancingAdjustment.Amount
    
            ^CTR149 = RP:Assets.DepreciatingAssets.IntangibleTerminationValue.Amount
    
            ^CTR150 = RP:Assets.DepreciatingAssets.OtherTerminationValue.Amount
            */
            assertion = (report.CTR363 != true && report.CTR99.GetValueOrDefault() > 0 && (report.CTR147 == null || report.CTR148 == null || report.CTR331 == null || report.CTR332 == null || report.CTR333 == null || report.CTR334 == null || report.CTR335 == null || report.CTR149 == null || report.CTR150 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.W00022",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Questions regarding depreciating assets should be answered if Deduction for decline in value of depreciating assets is greater than zero.",
                    LongDescription = @"If Deduction for decline in value of depreciating assets is greater than zero, then questions regarding depreciating assets should be answered.",
                    Location = "/xbrli:xbrl/tns:Expense.DepreciatingAssets.OtherFirstDeducted.Amount",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.W00026"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR363", Value = GetValueOrEmpty(report.CTR363) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR99", Value = GetValueOrEmpty(report.CTR99) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR147", Value = GetValueOrEmpty(report.CTR147) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR148", Value = GetValueOrEmpty(report.CTR148) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR331", Value = GetValueOrEmpty(report.CTR331) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR332", Value = GetValueOrEmpty(report.CTR332) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR333", Value = GetValueOrEmpty(report.CTR333) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR334", Value = GetValueOrEmpty(report.CTR334) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR335", Value = GetValueOrEmpty(report.CTR335) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR149", Value = GetValueOrEmpty(report.CTR149) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR150", Value = GetValueOrEmpty(report.CTR150) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.CTR.W00029
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTRW00029()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.CTR.W00029
            Consider if you are a Significant global entity (SGE). An entity is an SGE for a period if it is any of the following:
            - a global parent entity (GPE) with an annual global income of A$1 billion or more
            - a member of a group of entities consolidated for accounting purposes as a single group and one of the other group members is a GPE with an annual global income of A$1 billion or more
            - a member of a notional listed company group and one of the other group members is a GPE with an annual global income of A$1 billion or more.
            
            If you are an SGE, you also need to consider whether you are a country-by-country (CBC) reporting entity. CBC reporting entities must complete label G2, and may have additional reporting obligations. 
            
            For more information on SGEs and CBC reporting, visit our website at ato.gov.au.
    
            Legacy Rule Format:
            IF [CTR370] = TRUE
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR370 = TRUE
    
            Data Elements:
    
            ^CTR370 = RP:OrganisationDetails.SignificantGlobalEntityStatus.Indicator
            */
            assertion = (report.CTR370 == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.W00029",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Confirm whether taxpayer is a Significant Global Entity (SGE)",
                    LongDescription = @"Consider if you are a Significant global entity (SGE). An entity is an SGE for a period if it is any of the following:
- a global parent entity (GPE) with an annual global income of A$1 billion or more
- a member of a group of entities consolidated for accounting purposes as a single group and one of the other group members is a GPE with an annual global income of A$1 billion or more
- a member of a notional listed company group and one of the other group members is a GPE with an annual global income of A$1 billion or more.

If you are an SGE, you also need to consider whether you are a country-by-country (CBC) reporting entity. CBC reporting entities must complete label G2, and may have additional reporting obligations. 

For more information on SGEs and CBC reporting, visit our website at ato.gov.au.",
                    Location = "/xbrli:xbrl/tns:OrganisationDetails.SignificantGlobalEntityStatus.Indicator",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.CTR.W00029"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR370", Value = GetValueOrEmpty(report.CTR370) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.000208
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000208()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.000208
            Invalid context. The number of Reporting party contexts must equal 1
    
            Legacy Rule Format:
            IF COUNT(RP) <> 1
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) <> 1
    
            Data Elements:
    
            ^Context = RP
            */
            assertion = (Count(report.RPCount) != 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200007",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid XBRL schema",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.000208"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.000226
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000226()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.000226
            Invalid context. The number of Intermediary contexts must not exceed 1
    
            Legacy Rule Format:
            IF COUNT(INT) > 1
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = INT
            */
            assertion = (Count(report.INTCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200007",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid XBRL schema",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.INTLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.000226"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.000406
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000406()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.000406
            Address line 1 contains incorrect 'care of' reference.  The only acceptable 'care of' reference is C/-
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line1.Text <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.Line1.Text STARTSWITH SET("C/O ","C/ ","Care Of ","CO ")) OR (pyde.xx.xx:AddressDetails.Line1.Text CONTAINS SET(" C/O "," C/ "," Care Of "))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR15 <> BLANK) AND (StartsWithSet(^CTR15,'"C/O ","C/ ","Care Of ","CO "')) OR (ContainsSet(^CTR15, '" C/O "," C/ "," Care Of "'))
    
            Data Elements:
    
            ^CTR15 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.CTR15) != true && IsMatch(report.CTR15, @"^(C/O |C/ |Care Of |CO )", RegexOptions.IgnoreCase) || IsMatch(report.CTR15, @" C/O | C/ | Care Of ", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000406",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 contains incorrect 'care of' reference.  The only acceptable 'care of' reference is C/-",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.000406"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR15", Value = report.CTR15 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.000406
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000406_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.000406
            Address line 1 contains incorrect 'care of' reference.  The only acceptable 'care of' reference is C/-
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line1.Text <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.Line1.Text STARTSWITH SET("C/O ","C/ ","Care Of ","CO ")) OR (pyde.xx.xx:AddressDetails.Line1.Text CONTAINS SET(" C/O "," C/ "," Care Of "))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR24 <> BLANK) AND (StartsWithSet(^CTR24,'"C/O ","C/ ","Care Of ","CO "')) OR (ContainsSet(^CTR24, '" C/O "," C/ "," Care Of "'))
    
            Data Elements:
    
            ^CTR24 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
            */
            assertion = (string.IsNullOrWhiteSpace(report.CTR24) != true && IsMatch(report.CTR24, @"^(C/O |C/ |Care Of |CO )", RegexOptions.IgnoreCase) || IsMatch(report.CTR24, @" C/O | C/ | Care Of ", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000406",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 contains incorrect 'care of' reference.  The only acceptable 'care of' reference is C/-",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_P_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.000406"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR24", Value = report.CTR24 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.000408
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000408()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.000408
            Address line 2 must not contain a 'care of' reference
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line2.Text <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.Line2.Text STARTSWITH SET("C/- ","C/O ","C/ ","Care Of ","CO ") OR pyde.xx.xx:AddressDetails.Line2.Text CONTAINS SET(" C/- "," C/O "," C/ "," Care Of "))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR16 <> BLANK) AND (StartsWithSet(^CTR16,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^CTR16, '" C/- "," C/O "," C/ "," Care Of "'))
    
            Data Elements:
    
            ^CTR16 = RP:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.CTR16) != true && (IsMatch(report.CTR16, @"^(C/- |C/O |C/ |Care Of |CO )", RegexOptions.IgnoreCase) || IsMatch(report.CTR16, @" C/- | C/O | C/ | Care Of ", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000408",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Care of' is not valid for address line 2",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line2.Text",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.000408"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR16", Value = GetValueOrEmpty(report.CTR16) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.000408
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000408_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.000408
            Address line 2 must not contain a 'care of' reference
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line2.Text <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.Line2.Text STARTSWITH SET("C/- ","C/O ","C/ ","Care Of ","CO ") OR pyde.xx.xx:AddressDetails.Line2.Text CONTAINS SET(" C/- "," C/O "," C/ "," Care Of "))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR25 <> BLANK) AND (StartsWithSet(^CTR25,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^CTR25, '" C/- "," C/O "," C/ "," Care Of "'))
    
            Data Elements:
    
            ^CTR25 = RP:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
            */
            assertion = (string.IsNullOrWhiteSpace(report.CTR25) != true && (IsMatch(report.CTR25, @"^(C/- |C/O |C/ |Care Of |CO )", RegexOptions.IgnoreCase) || IsMatch(report.CTR25, @" C/- | C/O | C/ | Care Of ", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000408",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Care of' is not valid for address line 2",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_P_RPOccurrenceIndex) + "/tns:AddressDetails.Line2.Text",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.000408"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR25", Value = GetValueOrEmpty(report.CTR25) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.000458
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000458()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.000458
            Suffix is invalid
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.NameSuffix.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.NameSuffix.Text <> SET(DOMAIN(SUFFIX CODES)))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR220 <> BLANK) AND (NotInDomain(^CTR220, 'SUFFIX CODES'))
    
            Data Elements:
    
            ^CTR220 = RP:PersonNameDetails:PersonNameDetails.NameSuffix.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='DeclarationSignatory']
            */
            assertion = (string.IsNullOrWhiteSpace(report.CTR220) != true && !(IsMatch(report.CTR220, @"^(ESQ|II|III|IV|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000458",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suffix is invalid",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_DeclarationSignatory_RPOccurrenceIndex) + "/tns:PersonNameDetails.NameSuffix.Text",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.000458"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR220", Value = GetValueOrEmpty(report.CTR220) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.000459
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000459()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.000459
            Title is invalid
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.Title.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.Title.Text <> SET(DOMAIN(TITLE CODES)))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR218 <> BLANK) AND (NotInDomain(^CTR218, 'TITLE CODES'))
    
            Data Elements:
    
            ^CTR218 = RP:PersonNameDetails:PersonNameDetails.Title.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='DeclarationSignatory']
            */
            assertion = (string.IsNullOrWhiteSpace(report.CTR218) != true && !(IsMatch(report.CTR218, @"^(2LT|AB|ABBOT|AC|ACM|ADM|AIR CDRE|ALDERMAN|AM|ARCHBISHOP|ARCHDEACON|ASSOC PROF|AVM|BARON|BARONESS|BISHOP|BR|BRIG|CANON|CAPT|CARDINAL|CDRE|CDT|CHAP|CMDR|CMM|COL|CONST|COUNT|COUNTESS|CPL|CPO|DAME|DEACON|DEACONESS|DEAN|DEPUTY SUPT|DR|DUCHESS|DUKE|EARL|EF|FLGOFF|FLT LT|FR|FSGT|GEN|GNR|GP CAPT|HON JUDGE|HON JUST|HON|HRH|INSP|JUDGE|JUST|LAC|LACW|LADY|LBDR|LCPL|LORD|LS|LT CMDR|LT COL|LT GEN|LT|MADAM|MAJ GEN|MAJ|MASTER|MATRON|MAYOR|MAYORESS|MIDN|MISS|MON|MOST REV|MR|MRS|MS|PASTOR|PATRIARCH|PLT OFF|PO|PRIOR|PROF|PTE|RABBI|RADM|RECTOR|REV|RF|RT HON|RT REV MON|RT REV|RTREV BISHOP|SBLT|SEN|SGT|SIR|SMN|SNR CONST|SQN LDR|SR|SSGT|SUPR|SWAMI|TF|VADM|VERY REV|VICAR|VISCOUNT|WG CDR|WO|WO1|WO2)$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000459",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Title is invalid",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_DeclarationSignatory_RPOccurrenceIndex) + "/tns:PersonNameDetails.Title.Text",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.000459"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR218", Value = GetValueOrEmpty(report.CTR218) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP(Instant)
    
            ^OtherTFN = RP(Instant)
    
            ^TFN = RP
            */
            assertion = (report.RPInstantExists != false && report.RPInstantIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPInstantOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.001023"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.CFCLC
    
            ^OtherTFN = RP:RP.CFCLC
    
            ^TFN = RP
            */
            assertion = (report.RPCFCLCExists != false && report.RPCFCLCIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPCFCLCOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.001023"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_2()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.CFCUC
    
            ^OtherTFN = RP:RP.CFCUC
    
            ^TFN = RP
            */
            assertion = (report.RPCFCUCExists != false && report.RPCFCUCIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPCFCUCOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.001023"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_3()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.Closing
    
            ^OtherTFN = RP:RP.Closing
    
            ^TFN = RP
            */
            assertion = (report.RPClosingExists != false && report.RPClosingIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPClosingOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.001023"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_4()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP.Closing(Instant)
    
            ^OtherTFN = RP.Closing(Instant)
    
            ^TFN = RP
            */
            assertion = (report.RPClosingInstantExists != false && report.RPClosingInstantIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPClosingInstantOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.001023"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_5()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.CMPLY
    
            ^OtherTFN = RP:RP.CMPLY
    
            ^TFN = RP
            */
            assertion = (report.RPCMPLYExists != false && report.RPCMPLYIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPCMPLYOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.001023"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_6()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.GeneralPool
    
            ^OtherTFN = RP:RP.GeneralPool
    
            ^TFN = RP
            */
            assertion = (report.RPGeneralPoolExists != false && report.RPGeneralPoolIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPGeneralPoolOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.001023"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_7()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.Investor
    
            ^OtherTFN = RP:RP.Investor
    
            ^TFN = RP
            */
            assertion = (report.RPInvestorExists != false && report.RPInvestorIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPInvestorOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.001023"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_8()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP.Investor:RP.Investor.YPrevious
    
            ^OtherTFN = RP.Investor:RP.Investor.YPrevious
    
            ^TFN = RP
            */
            assertion = (report.RPInvestorYPreviousExists != false && report.RPInvestorYPreviousIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPInvestorYPreviousOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.001023"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_9()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.JAUS
    
            ^OtherTFN = RP:RP.JAUS
    
            ^TFN = RP
            */
            assertion = (report.RPJAUSExists != false && report.RPJAUSIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.001023"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_10()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.JFOREIGN
    
            ^OtherTFN = RP:RP.JFOREIGN
    
            ^TFN = RP
            */
            assertion = (report.RPJFOREIGNExists != false && report.RPJFOREIGNIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJFOREIGNOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.001023"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_11()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.LimitedPartners
    
            ^OtherTFN = RP:RP.LimitedPartners
    
            ^TFN = RP
            */
            assertion = (report.RPLimitedPartnersExists != false && report.RPLimitedPartnersIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPLimitedPartnersOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.001023"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_12()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP.LimitedPartners:RP.LimitedPartners.YPrevious
    
            ^OtherTFN = RP.LimitedPartners:RP.LimitedPartners.YPrevious
    
            ^TFN = RP
            */
            assertion = (report.RPLimitedPartnersYPreviousExists != false && report.RPLimitedPartnersYPreviousIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPLimitedPartnersYPreviousOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.001023"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_13()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.LowCost
    
            ^OtherTFN = RP:RP.LowCost
    
            ^TFN = RP
            */
            assertion = (report.RPLowCostExists != false && report.RPLowCostIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPLowCostOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.001023"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_14()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.Opening
    
            ^OtherTFN = RP:RP.Opening
    
            ^TFN = RP
            */
            assertion = (report.RPOpeningExists != false && report.RPOpeningIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOpeningOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.001023"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_15()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP.Opening(Instant)
    
            ^OtherTFN = RP.Opening(Instant)
    
            ^TFN = RP
            */
            assertion = (report.RPOpeningInstantExists != false && report.RPOpeningInstantIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOpeningInstantOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.001023"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_16()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.ORD
    
            ^OtherTFN = RP:RP.ORD
    
            ^TFN = RP
            */
            assertion = (report.RPORDExists != false && report.RPORDIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPORDOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.001023"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_17()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.ProspectiveRules
    
            ^OtherTFN = RP:RP.ProspectiveRules
    
            ^TFN = RP
            */
            assertion = (report.RPProspectiveRulesExists != false && report.RPProspectiveRulesIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPProspectiveRulesOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.001023"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_18()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.TOFA
    
            ^OtherTFN = RP:RP.TOFA
    
            ^TFN = RP
            */
            assertion = (report.RPTOFAExists != false && report.RPTOFAIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPTOFAOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.001023"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_19()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.Y0
    
            ^OtherTFN = RP:RP.Y0
    
            ^TFN = RP
            */
            assertion = (report.RPY0Exists != false && report.RPY0IdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0OccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.001023"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_20()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.Y0-1
    
            ^OtherTFN = RP:RP.Y0-1
    
            ^TFN = RP
            */
            assertion = (report.RPY0Minus1Exists != false && report.RPY0Minus1IdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0Minus1OccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.001023"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_21()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.Y0Plus1
    
            ^OtherTFN = RP:RP.Y0Plus1
    
            ^TFN = RP
            */
            assertion = (report.RPY0Plus1Exists != false && report.RPY0Plus1IdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0Plus1OccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.001023"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_22()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.YPrevious
    
            ^OtherTFN = RP:RP.YPrevious
    
            ^TFN = RP
            */
            assertion = (report.RPYPreviousExists != false && report.RPYPreviousIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPYPreviousOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.001023"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.010028
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN010028()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.010028
            If the return is an amendment, sequence number, type and reason must be present
    
            Legacy Rule Format:
            IF pyin.xx.xx:Report.Amendment.Indicator = TRUE AND (pyin.xx.xx:Report.AmendmentSequence.Number = NULL OR pyin.xx.xx:Report.AmendmentType.Code = NULLORBLANK OR pyin.xx.xx:Report.AmendmentReason.Text = NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR298 = TRUE AND (^CTR305 = NULL OR ^CTR306 = BLANK OR ^CTR307 = BLANK)
    
            Data Elements:
    
            ^CTR298 = RP:Report.Amendment.Indicator
    
            ^CTR305 = RP:Report.AmendmentSequence.Number
    
            ^CTR306 = RP:Report.AmendmentType.Code
    
            ^CTR307 = RP:Report.AmendmentReason.Text
            */
            assertion = (report.CTR298 == true && (report.CTR305 == null || string.IsNullOrWhiteSpace(report.CTR306) == true || string.IsNullOrWhiteSpace(report.CTR307) == true));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010028",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"If the return is an amendment, sequence number, type and reason must be present",
                    LongDescription = @"If the Amendment Indicator is 'yes' (true), then the Amendment Sequence Number, Amendment Type and Amendment Reason must be present and not blank.",
                    Location = "/xbrli:xbrl/tns:Report.Amendment.Indicator",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.010028"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR298", Value = GetValueOrEmpty(report.CTR298) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR305", Value = report.CTR305.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR306", Value = GetValueOrEmpty(report.CTR306) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR307", Value = GetValueOrEmpty(report.CTR307) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.010029
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN010029()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.010029
            If the return is an amendment, Amendment Indicator must be 'yes' (true)
    
            Legacy Rule Format:
            IF pyin.xx.xx:Report.Amendment.Indicator <> TRUE AND (pyin.xx.xx:Report.AmendmentSequence.Number <> NULL OR pyin.xx.xx:Report.AmendmentType.Code <> NULLORBLANK OR pyin.xx.xx:Report.AmendmentReason.Text <> NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR298 <> TRUE AND (^CTR305 <> NULL OR ^CTR306 <> BLANK OR ^CTR307 <> BLANK)
    
            Data Elements:
    
            ^CTR298 = RP:Report.Amendment.Indicator
    
            ^CTR305 = RP:Report.AmendmentSequence.Number
    
            ^CTR306 = RP:Report.AmendmentType.Code
    
            ^CTR307 = RP:Report.AmendmentReason.Text
            */
            assertion = (report.CTR298 != true && (report.CTR305 != null || string.IsNullOrWhiteSpace(report.CTR306) != true || string.IsNullOrWhiteSpace(report.CTR307) != true));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"If the return is an amendment, Amendment Indicator must be 'yes' (true)",
                    LongDescription = @"If an Amendment Sequence Number, Amendment Type or Amendment Reason is present, then the Amendment Indicator must be 'yes' (true)",
                    Location = "/xbrli:xbrl/tns:Report.Amendment.Indicator",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.010029"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR298", Value = GetValueOrEmpty(report.CTR298) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR305", Value = report.CTR305.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR306", Value = GetValueOrEmpty(report.CTR306) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR307", Value = GetValueOrEmpty(report.CTR307) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.010030
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN010030()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.010030
            Amendment sequence number must be a number between 1 and 9
    
            Legacy Rule Format:
            IF pyin.xx.xx:Report.AmendmentSequence.Number <> NULL AND pyin.xx.xx:Report.AmendmentSequence.Number <> SET(1-9)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR305 <> NULL AND (^CTR305 < 1 OR ^CTR305 > 9)
    
            Data Elements:
    
            ^CTR305 = RP:Report.AmendmentSequence.Number
            */
            assertion = (report.CTR305 != null && (report.CTR305.GetValueOrDefault() < 1 || report.CTR305.GetValueOrDefault() > 9));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010030",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Amendment sequence number must be a number between 1 and 9",
                    Location = "/xbrli:xbrl/tns:Report.AmendmentSequence.Number",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.010030"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR305", Value = report.CTR305.GetValueOrDefault().ToString() });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.408010
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN408010()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.408010
            Name field cannot contain 'space hyphen space'.
    
            Legacy Rule Format:
            WHERE IN TUPLE(personunstructuredname1.xx.xx:PersonUnstructuredName)
            IF (pyde.xx.xx:PersonUnstructuredName.FullName.Text CONTAINS " - ")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^CTR214, ' - ')
    
            Data Elements:
    
            ^CTR214 = INT:PersonUnstructuredName:PersonUnstructuredName.FullName.Text WHERE [PersonUnstructuredName.Usage.Code='Contact']
            */
            assertion = (report.CTR214 == null ? false : report.CTR214.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.408010",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Name field cannot contain 'space hyphen space'.",
                    LongDescription = @"Name field cannot contain 'space hyphen space'.",
                    Location = "/xbrli:xbrl/tns:PersonUnstructuredName" + OccurrenceIndex(report.PersonUnstructuredNameCollection_Contact_INTOccurrenceIndex) + "/tns:PersonUnstructuredName.FullName.Text",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.408010"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR214", Value = report.CTR214 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.408010
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN408010_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.408010
            Name field cannot contain 'space hyphen space'.
    
            Legacy Rule Format:
            WHERE IN TUPLE(personunstructuredname1.xx.xx:PersonUnstructuredName)
            IF (pyde.xx.xx:PersonUnstructuredName.FullName.Text CONTAINS " - ")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^CTR291, ' - ')
    
            Data Elements:
    
            ^CTR291 = INT:Declaration:PersonUnstructuredName:PersonUnstructuredName.FullName.Text WHERE [PersonUnstructuredName.Usage.Code='DeclarationSignatory']
            */
            assertion = (report.CTR291 == null ? false : report.CTR291.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.408010",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Name field cannot contain 'space hyphen space'.",
                    LongDescription = @"Name field cannot contain 'space hyphen space'.",
                    Location = "/xbrli:xbrl/tns:Declaration" + OccurrenceIndex(report.DeclarationCollection_TrueAndCorrect_INTOccurrenceIndex) + "/tns:PersonUnstructuredName" + OccurrenceIndex(report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTOccurrenceIndex) + "/tns:PersonUnstructuredName.FullName.Text",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.408010"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR291", Value = report.CTR291 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.410002
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410002()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.410002
            Address Line 2 must be present if Address Line 1 contains 'C/-'
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF ((pyde.xx.xx:AddressDetails.Line1.Text CONTAINS "C/-") AND (pyde.xx.xx:AddressDetails.Line2.Text = NULLORBLANK))
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ((Contains(^CTR33, 'C/-')) AND (^CTR34 = BLANK))
    
            Data Elements:
    
            ^CTR33 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
    
            ^CTR34 = RP:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            */
            assertion = ((report.CTR33 == null ? false : report.CTR33.ToUpperInvariant().Contains(@"C/-")) && string.IsNullOrWhiteSpace(report.CTR34) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410002",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 2 must be present if Address Line 1 contains 'C/-'",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.410002"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR33", Value = report.CTR33 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR34", Value = GetValueOrEmpty(report.CTR34) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.410002
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410002_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.410002
            Address Line 2 must be present if Address Line 1 contains 'C/-'
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF ((pyde.xx.xx:AddressDetails.Line1.Text CONTAINS "C/-") AND (pyde.xx.xx:AddressDetails.Line2.Text = NULLORBLANK))
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ((Contains(^CTR15, 'C/-')) AND (^CTR16 = BLANK))
    
            Data Elements:
    
            ^CTR15 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
    
            ^CTR16 = RP:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = ((report.CTR15 == null ? false : report.CTR15.ToUpperInvariant().Contains(@"C/-")) && string.IsNullOrWhiteSpace(report.CTR16) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410002",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 2 must be present if Address Line 1 contains 'C/-'",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.410002"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR15", Value = report.CTR15 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR16", Value = GetValueOrEmpty(report.CTR16) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.410002
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410002_2()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.410002
            Address Line 2 must be present if Address Line 1 contains 'C/-'
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF ((pyde.xx.xx:AddressDetails.Line1.Text CONTAINS "C/-") AND (pyde.xx.xx:AddressDetails.Line2.Text = NULLORBLANK))
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ((Contains(^CTR24, 'C/-')) AND (^CTR25 = BLANK))
    
            Data Elements:
    
            ^CTR24 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
    
            ^CTR25 = RP:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
            */
            assertion = ((report.CTR24 == null ? false : report.CTR24.ToUpperInvariant().Contains(@"C/-")) && string.IsNullOrWhiteSpace(report.CTR25) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410002",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 2 must be present if Address Line 1 contains 'C/-'",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_P_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.410002"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR24", Value = report.CTR24 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR25", Value = GetValueOrEmpty(report.CTR25) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.410008
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410008()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.410008
            Lines 3 and 4 of the address are not used on this form
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line3.Text <> NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR35 <> BLANK
    
            Data Elements:
    
            ^CTR35 = RP:AddressDetails:AddressDetails.Line3.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.CTR35) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410008",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 3 must be left blank",
                    LongDescription = @"Lines 3 and 4 of the address are not used on this form",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line3.Text",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.410008"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR35", Value = GetValueOrEmpty(report.CTR35) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.410008
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410008_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.410008
            Lines 3 and 4 of the address are not used on this form
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line3.Text <> NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR17 <> BLANK
    
            Data Elements:
    
            ^CTR17 = RP:AddressDetails:AddressDetails.Line3.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.CTR17) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410008",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 3 must be left blank",
                    LongDescription = @"Lines 3 and 4 of the address are not used on this form",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line3.Text",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.410008"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR17", Value = GetValueOrEmpty(report.CTR17) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.410008
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410008_2()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.410008
            Lines 3 and 4 of the address are not used on this form
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line3.Text <> NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR26 <> BLANK
    
            Data Elements:
    
            ^CTR26 = RP:AddressDetails:AddressDetails.Line3.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
            */
            assertion = (string.IsNullOrWhiteSpace(report.CTR26) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410008",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 3 must be left blank",
                    LongDescription = @"Lines 3 and 4 of the address are not used on this form",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_P_RPOccurrenceIndex) + "/tns:AddressDetails.Line3.Text",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.410008"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR26", Value = GetValueOrEmpty(report.CTR26) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.410013
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410013()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.410013
            Lines 3 and 4 of the address are not used on this form
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line4.Text <> NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR36 <> BLANK
    
            Data Elements:
    
            ^CTR36 = RP:AddressDetails:AddressDetails.Line4.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.CTR36) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410013",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 4 must be left blank",
                    LongDescription = @"Lines 3 and 4 of the address are not used on this form",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line4.Text",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.410013"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR36", Value = GetValueOrEmpty(report.CTR36) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.410013
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410013_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.410013
            Lines 3 and 4 of the address are not used on this form
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line4.Text <> NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR18 <> BLANK
    
            Data Elements:
    
            ^CTR18 = RP:AddressDetails:AddressDetails.Line4.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.CTR18) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410013",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 4 must be left blank",
                    LongDescription = @"Lines 3 and 4 of the address are not used on this form",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line4.Text",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.410013"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR18", Value = GetValueOrEmpty(report.CTR18) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.410013
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410013_2()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.410013
            Lines 3 and 4 of the address are not used on this form
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line4.Text <> NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^CTR27 <> BLANK
    
            Data Elements:
    
            ^CTR27 = RP:AddressDetails:AddressDetails.Line4.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
            */
            assertion = (string.IsNullOrWhiteSpace(report.CTR27) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410013",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 4 must be left blank",
                    LongDescription = @"Lines 3 and 4 of the address are not used on this form",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_P_RPOccurrenceIndex) + "/tns:AddressDetails.Line4.Text",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.410013"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR27", Value = GetValueOrEmpty(report.CTR27) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.410038
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410038()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.410038
            Non-individual name field cannot start with 'T/A' and end with 'Pship', 'P'ship or 'P/Ship'
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text STARTSWITH "T/A ") AND (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text ENDSWITH SET(" Pship"," P'ship"," P/ship"))
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (StartsWith(^CTR9, 'T/A ')) AND (EndsWithSet(^CTR9, '" Pship"," P'ship"," P/ship"'))
    
            Data Elements:
    
            ^CTR9 = RP:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
            */
            assertion = (IsMatch(report.CTR9, @"^(T/A )", RegexOptions.IgnoreCase) && IsMatch(report.CTR9, @"( Pship| P'ship| P/ship)$", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410038",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-individual name field cannot start with 'T/A' and end with 'Pship', 'P'ship or 'P/Ship'",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.410038"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR9", Value = report.CTR9 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.410038
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410038_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.410038
            Non-individual name field cannot start with 'T/A' and end with 'Pship', 'P'ship or 'P/Ship'
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text STARTSWITH "T/A ") AND (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text ENDSWITH SET(" Pship"," P'ship"," P/ship"))
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (StartsWith(^CTR12, 'T/A ')) AND (EndsWithSet(^CTR12, '" Pship"," P'ship"," P/ship"'))
    
            Data Elements:
    
            ^CTR12 = RP:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='P']
            */
            assertion = (IsMatch(report.CTR12, @"^(T/A )", RegexOptions.IgnoreCase) && IsMatch(report.CTR12, @"( Pship| P'ship| P/ship)$", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410038",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-individual name field cannot start with 'T/A' and end with 'Pship', 'P'ship or 'P/Ship'",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_MN_P_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.410038"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR12", Value = report.CTR12 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.410039
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410039()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.410039
            Organisation name must contain at least one alpha or numeric character.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text  <> NULLORBLANK) AND (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text  DOES NOT CONTAIN SET("a-z", "A-Z", "0-9"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR9  <> BLANK) AND (NotContainsSet(^CTR9, '"a-z", "A-Z", "0-9"'))
    
            Data Elements:
    
            ^CTR9 = RP:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.CTR9) != true && !(IsMatch(report.CTR9, @"[a-z]|[A-Z]|[0-9]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410039",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Organisation name must contain at least one alpha or numeric character",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.410039"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR9", Value = report.CTR9 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.410039
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410039_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.410039
            Organisation name must contain at least one alpha or numeric character.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text  <> NULLORBLANK) AND (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text  DOES NOT CONTAIN SET("a-z", "A-Z", "0-9"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR12  <> BLANK) AND (NotContainsSet(^CTR12, '"a-z", "A-Z", "0-9"'))
    
            Data Elements:
    
            ^CTR12 = RP:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='P']
            */
            assertion = (string.IsNullOrWhiteSpace(report.CTR12) != true && !(IsMatch(report.CTR12, @"[a-z]|[A-Z]|[0-9]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410039",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Organisation name must contain at least one alpha or numeric character",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_MN_P_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.410039"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR12", Value = report.CTR12 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.410039
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410039_2()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.410039
            Organisation name must contain at least one alpha or numeric character.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text  <> NULLORBLANK) AND (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text  DOES NOT CONTAIN SET("a-z", "A-Z", "0-9"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR46  <> BLANK) AND (NotContainsSet(^CTR46, '"a-z", "A-Z", "0-9"'))
    
            Data Elements:
    
            ^CTR46 = RP:ImmediateHoldingCompany:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.CTR46) != true && !(IsMatch(report.CTR46, @"[a-z]|[A-Z]|[0-9]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410039",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Organisation name must contain at least one alpha or numeric character",
                    Location = "/xbrli:xbrl/tns:ImmediateHoldingCompany/tns:OrganisationNameDetails" + OccurrenceIndex(report.ImmediateHoldingCompany_OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.410039"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR46", Value = report.CTR46 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.410039
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410039_3()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.410039
            Organisation name must contain at least one alpha or numeric character.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text  <> NULLORBLANK) AND (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text  DOES NOT CONTAIN SET("a-z", "A-Z", "0-9"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR43  <> BLANK) AND (NotContainsSet(^CTR43, '"a-z", "A-Z", "0-9"'))
    
            Data Elements:
    
            ^CTR43 = RP:UltimateHoldingCompany:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.CTR43) != true && !(IsMatch(report.CTR43, @"[a-z]|[A-Z]|[0-9]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410039",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Organisation name must contain at least one alpha or numeric character",
                    Location = "/xbrli:xbrl/tns:UltimateHoldingCompany/tns:OrganisationNameDetails" + OccurrenceIndex(report.UltimateHoldingCompany_OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.410039"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR43", Value = report.CTR43 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.410105
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410105()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.410105
            ARBN must be a valid number that passes the algorithm
    
            Legacy Rule Format:
            IF (pyid.xx.xx:Identifiers.AustralianRegisteredBodyNumber.Identifier <> NULLORBLANK) AND (ARBNALGORITHM (pyid.xx.xx:Identifiers.AustralianRegisteredBodyNumber.Identifier) = FALSE)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR13 <> BLANK) AND (FailsARBNAlgorithm(^CTR13))
    
            Data Elements:
    
            ^CTR13 = RP.Y0-1:Identifiers.AustralianRegisteredBodyNumber.Identifier
            */
            assertion = (string.IsNullOrWhiteSpace(report.CTR13) != true && FailsACNAlgorithm(report.CTR13));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410105",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"ARBN must be a valid number that passes the algorithm",
                    Location = "/xbrli:xbrl/tns:Identifiers.AustralianRegisteredBodyNumber.Identifier",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.410105"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR13", Value = GetValueOrEmpty(report.CTR13) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.410131
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410131()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.410131
            Other Given Name must contain at least one alphabetical character
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text DOES NOT CONTAIN SET("A-Z","a-z"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR222 <> BLANK) AND (NotContainsSet(^CTR222, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^CTR222 = RP:PersonNameDetails:PersonNameDetails.OtherGivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='DeclarationSignatory']
            */
            assertion = (string.IsNullOrWhiteSpace(report.CTR222) != true && !(IsMatch(report.CTR222, @"[A-Z]|[a-z]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410131",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other Given Name must contain at least one alphabetical character",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_DeclarationSignatory_RPOccurrenceIndex) + "/tns:PersonNameDetails.OtherGivenName.Text",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.410131"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR222", Value = GetValueOrEmpty(report.CTR222) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.410191
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410191()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.410191
            Country code must be present for overseas address
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF (pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = TRUE) AND (pyde.xx.xx:AddressDetails.Country.Code = NULL)
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR40 = TRUE) AND (^CTR41 = NULL)
    
            Data Elements:
    
            ^CTR41 = RP:AddressDetails:AddressDetails.Country.Code WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
    
            ^CTR40 = RP:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (report.CTR40 == true && report.CTR41 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410191",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Country code must be present for overseas address",
                    LongDescription = @"When the overseas address indicator is 'TRUE', the corresponding Country Code must be supplied",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Country.Code",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.410191"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR40", Value = GetValueOrEmpty(report.CTR40) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR41", Value = GetValueOrEmpty(report.CTR41) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.410191
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410191_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.410191
            Country code must be present for overseas address
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF (pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = TRUE) AND (pyde.xx.xx:AddressDetails.Country.Code = NULL)
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR22 = TRUE) AND (^CTR23 = NULL)
    
            Data Elements:
    
            ^CTR23 = RP:AddressDetails:AddressDetails.Country.Code WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
    
            ^CTR22 = RP:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (report.CTR22 == true && report.CTR23 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410191",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Country code must be present for overseas address",
                    LongDescription = @"When the overseas address indicator is 'TRUE', the corresponding Country Code must be supplied",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Country.Code",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.410191"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR22", Value = GetValueOrEmpty(report.CTR22) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR23", Value = GetValueOrEmpty(report.CTR23) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.410191
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410191_2()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.410191
            Country code must be present for overseas address
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF (pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = TRUE) AND (pyde.xx.xx:AddressDetails.Country.Code = NULL)
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR31 = TRUE) AND (^CTR32 = NULL)
    
            Data Elements:
    
            ^CTR32 = RP:AddressDetails:AddressDetails.Country.Code WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
    
            ^CTR31 = RP:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
            */
            assertion = (report.CTR31 == true && report.CTR32 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410191",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Country code must be present for overseas address",
                    LongDescription = @"When the overseas address indicator is 'TRUE', the corresponding Country Code must be supplied",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_P_RPOccurrenceIndex) + "/tns:AddressDetails.Country.Code",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.410191"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR31", Value = GetValueOrEmpty(report.CTR31) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR32", Value = GetValueOrEmpty(report.CTR32) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.410194
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410194()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.410194
            Although SBR sets a limit of 50 characters for the Line 1 of an address, the maximum allowable for this form is 38
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:AddressDetails.Line1.Text) > 38
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^CTR33) > 38
    
            Data Elements:
    
            ^CTR33 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (Length(report.CTR33) > 38);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410194",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 1 must not exceed 38 characters",
                    LongDescription = @"Although SBR sets a limit of 50 characters for the Line 1 of an address, the maximum allowable for this form is 38",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.410194"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR33", Value = report.CTR33 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.410194
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410194_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.410194
            Although SBR sets a limit of 50 characters for the Line 1 of an address, the maximum allowable for this form is 38
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:AddressDetails.Line1.Text) > 38
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^CTR15) > 38
    
            Data Elements:
    
            ^CTR15 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (Length(report.CTR15) > 38);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410194",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 1 must not exceed 38 characters",
                    LongDescription = @"Although SBR sets a limit of 50 characters for the Line 1 of an address, the maximum allowable for this form is 38",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.410194"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR15", Value = report.CTR15 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.410194
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410194_2()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.410194
            Although SBR sets a limit of 50 characters for the Line 1 of an address, the maximum allowable for this form is 38
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:AddressDetails.Line1.Text) > 38
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^CTR24) > 38
    
            Data Elements:
    
            ^CTR24 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
            */
            assertion = (Length(report.CTR24) > 38);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410194",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 1 must not exceed 38 characters",
                    LongDescription = @"Although SBR sets a limit of 50 characters for the Line 1 of an address, the maximum allowable for this form is 38",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_P_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.410194"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR24", Value = report.CTR24 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.410195
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410195()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.410195
            Although SBR sets a limit of 50 characters for the Line 1 of an address, the maximum allowable for this form is 38
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:AddressDetails.Line2.Text) > 38
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^CTR34) > 38
    
            Data Elements:
    
            ^CTR34 = RP:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (Length(report.CTR34) > 38);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410195",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 2 must not exceed 38 characters",
                    LongDescription = @"Although SBR sets a limit of 50 characters for the Line 2 of an address, the maximum allowable for this form is 38",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line2.Text",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.410195"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR34", Value = GetValueOrEmpty(report.CTR34) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.410195
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410195_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.410195
            Although SBR sets a limit of 50 characters for the Line 1 of an address, the maximum allowable for this form is 38
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:AddressDetails.Line2.Text) > 38
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^CTR16) > 38
    
            Data Elements:
    
            ^CTR16 = RP:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (Length(report.CTR16) > 38);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410195",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 2 must not exceed 38 characters",
                    LongDescription = @"Although SBR sets a limit of 50 characters for the Line 2 of an address, the maximum allowable for this form is 38",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line2.Text",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.410195"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR16", Value = GetValueOrEmpty(report.CTR16) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.410195
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410195_2()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.410195
            Although SBR sets a limit of 50 characters for the Line 1 of an address, the maximum allowable for this form is 38
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:AddressDetails.Line2.Text) > 38
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^CTR25) > 38
    
            Data Elements:
    
            ^CTR25 = RP:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
            */
            assertion = (Length(report.CTR25) > 38);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410195",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 2 must not exceed 38 characters",
                    LongDescription = @"Although SBR sets a limit of 50 characters for the Line 2 of an address, the maximum allowable for this form is 38",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_P_RPOccurrenceIndex) + "/tns:AddressDetails.Line2.Text",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.410195"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR25", Value = GetValueOrEmpty(report.CTR25) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.410200
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410200()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.410200
            Family name cannot contain "Exec for", "Rep for" or "Trustee for"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text = FOUND("Exec for","Rep for","Trustee for"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^CTR219, '"Exec for","Rep for","Trustee for"'))
    
            Data Elements:
    
            ^CTR219 = RP:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='DeclarationSignatory']
            */
            assertion = IsMatch(report.CTR219, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000424",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_DeclarationSignatory_RPOccurrenceIndex) + "/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.410200"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR219", Value = report.CTR219 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.410201
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410201()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.410201
            Family name cannot contain "MR", "MRS", "MISS" or "MS"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text = FOUND("MR","MRS","MISS","MS"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^CTR219, '"MR","MRS","MISS","MS"'))
    
            Data Elements:
    
            ^CTR219 = RP:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='DeclarationSignatory']
            */
            assertion = IsMatch(report.CTR219, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000426",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_DeclarationSignatory_RPOccurrenceIndex) + "/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.410201"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR219", Value = report.CTR219 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.410203
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410203()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.410203
            First name cannot contain "Exec for", "Rep for" or "Trustee for"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text = FOUND("Exec for","Rep for","Trustee for"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^CTR221, '"Exec for","Rep for","Trustee for"'))
    
            Data Elements:
    
            ^CTR221 = RP:PersonNameDetails:PersonNameDetails.GivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='DeclarationSignatory']
            */
            assertion = IsMatch(report.CTR221, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000437",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_DeclarationSignatory_RPOccurrenceIndex) + "/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.410203"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR221", Value = GetValueOrEmpty(report.CTR221) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.410204
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410204()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.410204
            First name cannot contain "MR", "MRS", "MISS" or "MS"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text = FOUND("MR","MRS","MISS","MS"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^CTR221, '"MR","MRS","MISS","MS"'))
    
            Data Elements:
    
            ^CTR221 = RP:PersonNameDetails:PersonNameDetails.GivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='DeclarationSignatory']
            */
            assertion = IsMatch(report.CTR221, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000438",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_DeclarationSignatory_RPOccurrenceIndex) + "/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.410204"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR221", Value = GetValueOrEmpty(report.CTR221) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.410205
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410205()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.410205
            Address Line 1 must not contain 'AS ABOVE'
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line1.Text = FOUND("AS ABOVE"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^CTR15, '"AS ABOVE"'))
    
            Data Elements:
    
            ^CTR15 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.CTR15, @"(^(AS ABOVE)$)|(^(.*)( AS ABOVE )(.*)$)|(^(AS ABOVE )(.*)$)|(^(.*)( AS ABOVE)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 1 must not contain 'AS ABOVE'",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.410205"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR15", Value = report.CTR15 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.410205
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410205_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.410205
            Address Line 1 must not contain 'AS ABOVE'
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line1.Text = FOUND("AS ABOVE"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^CTR24, '"AS ABOVE"'))
    
            Data Elements:
    
            ^CTR24 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
            */
            assertion = IsMatch(report.CTR24, @"(^(AS ABOVE)$)|(^(.*)( AS ABOVE )(.*)$)|(^(AS ABOVE )(.*)$)|(^(.*)( AS ABOVE)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 1 must not contain 'AS ABOVE'",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_P_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.410205"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR24", Value = report.CTR24 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.410208
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410208()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.410208
            Other given name cannot contain "Exec for", "Rep for" or "Trustee for"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text = FOUND("Exec for","Rep for","Trustee for"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^CTR222, '"Exec for","Rep for","Trustee for"'))
    
            Data Elements:
    
            ^CTR222 = RP:PersonNameDetails:PersonNameDetails.OtherGivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='DeclarationSignatory']
            */
            assertion = IsMatch(report.CTR222, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000449",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_DeclarationSignatory_RPOccurrenceIndex) + "/tns:PersonNameDetails.OtherGivenName.Text",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.410208"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR222", Value = GetValueOrEmpty(report.CTR222) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.410209
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410209()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.410209
            Other given name cannot contain "MR", "MRS", "MISS" or "MS"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text = FOUND("MR","MRS","MISS","MS"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^CTR222, '"MR","MRS","MISS","MS"'))
    
            Data Elements:
    
            ^CTR222 = RP:PersonNameDetails:PersonNameDetails.OtherGivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='DeclarationSignatory']
            */
            assertion = IsMatch(report.CTR222, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000450",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_DeclarationSignatory_RPOccurrenceIndex) + "/tns:PersonNameDetails.OtherGivenName.Text",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.410209"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR222", Value = GetValueOrEmpty(report.CTR222) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.410211
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410211()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.410211
            Overseas address indicator must not be 'TRUE' if Country Code is 'au'
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF (pyde.xx.xx:AddressDetails.Country.Code = "au") AND  (pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = TRUE) 
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR41 = 'au') AND  (^CTR40 = TRUE)
    
            Data Elements:
    
            ^CTR41 = RP:AddressDetails:AddressDetails.Country.Code WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
    
            ^CTR40 = RP:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (report.CTR41 == @"au" && report.CTR40 == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410211",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Overseas address indicator must not be 'TRUE' if Country Code is 'au'",
                    LongDescription = @"The overseas address indicator must be 'FALSE' if the Country Code is 'au' - representing Australia. If the address is an overseas address please supply the correct Country Code",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Country.Code",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.410211"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR41", Value = GetValueOrEmpty(report.CTR41) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR40", Value = GetValueOrEmpty(report.CTR40) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.410211
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410211_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.410211
            Overseas address indicator must not be 'TRUE' if Country Code is 'au'
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF (pyde.xx.xx:AddressDetails.Country.Code = "au") AND  (pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = TRUE) 
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR23 = 'au') AND  (^CTR22 = TRUE)
    
            Data Elements:
    
            ^CTR23 = RP:AddressDetails:AddressDetails.Country.Code WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
    
            ^CTR22 = RP:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (report.CTR23 == @"au" && report.CTR22 == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410211",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Overseas address indicator must not be 'TRUE' if Country Code is 'au'",
                    LongDescription = @"The overseas address indicator must be 'FALSE' if the Country Code is 'au' - representing Australia. If the address is an overseas address please supply the correct Country Code",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Country.Code",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.410211"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR23", Value = GetValueOrEmpty(report.CTR23) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR22", Value = GetValueOrEmpty(report.CTR22) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.410211
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410211_2()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.410211
            Overseas address indicator must not be 'TRUE' if Country Code is 'au'
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF (pyde.xx.xx:AddressDetails.Country.Code = "au") AND  (pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = TRUE) 
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR32 = 'au') AND  (^CTR31 = TRUE)
    
            Data Elements:
    
            ^CTR32 = RP:AddressDetails:AddressDetails.Country.Code WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
    
            ^CTR31 = RP:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
            */
            assertion = (report.CTR32 == @"au" && report.CTR31 == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410211",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Overseas address indicator must not be 'TRUE' if Country Code is 'au'",
                    LongDescription = @"The overseas address indicator must be 'FALSE' if the Country Code is 'au' - representing Australia. If the address is an overseas address please supply the correct Country Code",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_P_RPOccurrenceIndex) + "/tns:AddressDetails.Country.Code",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.410211"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR32", Value = GetValueOrEmpty(report.CTR32) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR31", Value = GetValueOrEmpty(report.CTR31) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.410212
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410212()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.410212
            If an overseas Country Code is used, the Overseas Address indicator must be 'TRUE'
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF (pyde.xx.xx:AddressDetails.Country.Code <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.Country.Code <> "au") AND  (pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = FALSE) 
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR41 <> BLANK) AND (^CTR41 <> 'au') AND  (^CTR40 = FALSE)
    
            Data Elements:
    
            ^CTR41 = RP:AddressDetails:AddressDetails.Country.Code WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
    
            ^CTR40 = RP:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='BUS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.CTR41) != true && report.CTR41 != @"au" && report.CTR40 == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410212",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"If an overseas Country Code is used, the Overseas Address indicator must be 'TRUE'",
                    LongDescription = @"If the address is an Australian address, the overseas address indicator must be set to 'FALSE' and you do not need to supply a Country Code. If the address is an overseas address, the overseas address indicator must be set to 'TRUE' and you must supply a Country Code other than 'au'",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_BUS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Country.Code",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.410212"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR41", Value = GetValueOrEmpty(report.CTR41) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR40", Value = GetValueOrEmpty(report.CTR40) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.410212
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410212_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.410212
            If an overseas Country Code is used, the Overseas Address indicator must be 'TRUE'
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF (pyde.xx.xx:AddressDetails.Country.Code <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.Country.Code <> "au") AND  (pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = FALSE) 
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR23 <> BLANK) AND (^CTR23 <> 'au') AND  (^CTR22 = FALSE)
    
            Data Elements:
    
            ^CTR23 = RP:AddressDetails:AddressDetails.Country.Code WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
    
            ^CTR22 = RP:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.CTR23) != true && report.CTR23 != @"au" && report.CTR22 == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410212",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"If an overseas Country Code is used, the Overseas Address indicator must be 'TRUE'",
                    LongDescription = @"If the address is an Australian address, the overseas address indicator must be set to 'FALSE' and you do not need to supply a Country Code. If the address is an overseas address, the overseas address indicator must be set to 'TRUE' and you must supply a Country Code other than 'au'",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Country.Code",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.410212"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR23", Value = GetValueOrEmpty(report.CTR23) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR22", Value = GetValueOrEmpty(report.CTR22) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.410212
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410212_2()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.410212
            If an overseas Country Code is used, the Overseas Address indicator must be 'TRUE'
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF (pyde.xx.xx:AddressDetails.Country.Code <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.Country.Code <> "au") AND  (pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = FALSE) 
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR32 <> BLANK) AND (^CTR32 <> 'au') AND  (^CTR31 = FALSE)
    
            Data Elements:
    
            ^CTR32 = RP:AddressDetails:AddressDetails.Country.Code WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
    
            ^CTR31 = RP:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='P']
            */
            assertion = (string.IsNullOrWhiteSpace(report.CTR32) != true && report.CTR32 != @"au" && report.CTR31 == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410212",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"If an overseas Country Code is used, the Overseas Address indicator must be 'TRUE'",
                    LongDescription = @"If the address is an Australian address, the overseas address indicator must be set to 'FALSE' and you do not need to supply a Country Code. If the address is an overseas address, the overseas address indicator must be set to 'TRUE' and you must supply a Country Code other than 'au'",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_P_RPOccurrenceIndex) + "/tns:AddressDetails.Country.Code",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.410212"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR32", Value = GetValueOrEmpty(report.CTR32) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR31", Value = GetValueOrEmpty(report.CTR31) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.410218
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410218()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.410218
            FirstName must be entered if OtherGivenNames is entered
    
            Legacy Rule Format:
            WHERE IN TUPLE(personstructuredname3.xx.xx:PersonNameDetails) 
            IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.GivenName.Text = NULLORBLANK) 
                RETURN VALIDATION MESSAGE 
             ENDIF

            Technical Business Rule Format:
            (^PersonNameDetails <> NULL) AND ((^CTR222 <> BLANK) AND (^CTR221 = BLANK))
    
            Data Elements:
    
            ^CTR221 = RP:PersonNameDetails:PersonNameDetails.GivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='DeclarationSignatory']
    
            ^CTR222 = RP:PersonNameDetails:PersonNameDetails.OtherGivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='DeclarationSignatory']
    
            ^PersonNameDetails = RP:PersonNameDetails
            */
            assertion = (report.PersonNameDetailsCollection_LGL_DeclarationSignatory_RPExists != false && (string.IsNullOrWhiteSpace(report.CTR222) != true && string.IsNullOrWhiteSpace(report.CTR221) == true));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410218",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"FirstName must be entered if OtherGivenNames is entered",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_DeclarationSignatory_RPOccurrenceIndex) + "/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.410218"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR222", Value = GetValueOrEmpty(report.CTR222) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR221", Value = GetValueOrEmpty(report.CTR221) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.428016
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428016()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.428016
            Tax File Number must pass the TFN algorithm check.
    
            Legacy Rule Format:
            IF (RP:entity.identifier.TFN <> NULLORBLANK) AND (TFNALGORITHM (RP:entity.identifier.TFN) = FALSE)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^TFN <> BLANK) AND (FailsTFNAlgorithm(^TFN))
    
            Data Elements:
    
            ^TFN = RP
            */
            assertion = (string.IsNullOrWhiteSpace(report.RPIdentifierTFN) != true && FailsTFNAlgorithm(report.RPIdentifierTFN));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.428016",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax File Number has failed the algorithm check",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.428016"
                        }
                    },
                };
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.428021
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428021()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.428021
            The Australian Company Number (ACN) has failed the algorithm check.
    
            Legacy Rule Format:
            IF (pyid.xx.xx:Identifiers.AustralianCompanyNumber.Identifier <> NULLORBLANK) AND (ACNALGORITHM(pyid.xx.xx:Identifiers.AustralianCompanyNumber.Identifier) = FALSE)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR14 <> BLANK) AND (FailsACNAlgorithm(^CTR14))
    
            Data Elements:
    
            ^CTR14 = RP.Y0-1:Identifiers.AustralianCompanyNumber.Identifier
            */
            assertion = (string.IsNullOrWhiteSpace(report.CTR14) != true && FailsACNAlgorithm(report.CTR14));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.428021",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid ACN",
                    LongDescription = @"The Australian Company Number (ACN) has failed the algorithm check",
                    Location = "/xbrli:xbrl/tns:Identifiers.AustralianCompanyNumber.Identifier",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.428021"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR14", Value = GetValueOrEmpty(report.CTR14) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.428040
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428040()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.428040
            ABN is invalid
    
            Legacy Rule Format:
            IF (pyid.xx.xx:IdentifiersAustralianBusinessNumberIdentifier <> NULLORBLANK) and (ABNALGORITHM(pyid.xx.xx:IdentifiersAustralianBusinessNumber.Identifier) = FALSE)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR11 <> BLANK) and (FailsABNAlgorithm(^CTR11))
    
            Data Elements:
    
            ^CTR11 = RP:Identifiers.AustralianBusinessNumber.Identifier
            */
            assertion = (string.IsNullOrWhiteSpace(report.CTR11) != true && FailsABNAlgorithm(report.CTR11));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000477",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"ABN is invalid",
                    Location = "/xbrli:xbrl/tns:Identifiers.AustralianBusinessNumber.Identifier",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.428040"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR11", Value = GetValueOrEmpty(report.CTR11) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.428040
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428040_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.428040
            ABN is invalid
    
            Legacy Rule Format:
            IF (pyid.xx.xx:IdentifiersAustralianBusinessNumberIdentifier <> NULLORBLANK) and (ABNALGORITHM(pyid.xx.xx:IdentifiersAustralianBusinessNumber.Identifier) = FALSE)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR47 <> BLANK) and (FailsABNAlgorithm(^CTR47))
    
            Data Elements:
    
            ^CTR47 = RP:ImmediateHoldingCompany:Identifiers.AustralianBusinessNumber.Identifier
            */
            assertion = (string.IsNullOrWhiteSpace(report.CTR47) != true && FailsABNAlgorithm(report.CTR47));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000477",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"ABN is invalid",
                    Location = "/xbrli:xbrl/tns:ImmediateHoldingCompany/tns:Identifiers.AustralianBusinessNumber.Identifier",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.428040"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR47", Value = GetValueOrEmpty(report.CTR47) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.428040
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428040_2()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.428040
            ABN is invalid
    
            Legacy Rule Format:
            IF (pyid.xx.xx:IdentifiersAustralianBusinessNumberIdentifier <> NULLORBLANK) and (ABNALGORITHM(pyid.xx.xx:IdentifiersAustralianBusinessNumber.Identifier) = FALSE)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^CTR44 <> BLANK) and (FailsABNAlgorithm(^CTR44))
    
            Data Elements:
    
            ^CTR44 = RP:UltimateHoldingCompany:Identifiers.AustralianBusinessNumber.Identifier
            */
            assertion = (string.IsNullOrWhiteSpace(report.CTR44) != true && FailsABNAlgorithm(report.CTR44));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000477",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"ABN is invalid",
                    Location = "/xbrli:xbrl/tns:UltimateHoldingCompany/tns:Identifiers.AustralianBusinessNumber.Identifier",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.428040"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR44", Value = GetValueOrEmpty(report.CTR44) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.428223
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428223()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.428223
            Reporting Party Name Cu