using Ato.CD.Inbound.Shared;
using Ato.EN.IntegrationServices.CodeGenerationCGNFT;
using Ato.EN.IntegrationServices.CodeGenerationCTR;
using Ato.EN.IntegrationServices.CodeGenerationDISTBENTRT;
using Ato.EN.IntegrationServices.CodeGenerationIDS;
using Ato.EN.IntegrationServices.CodeGenerationIEE;
using Ato.EN.IntegrationServices.CodeGenerationPSS;
using Ato.EN.IntegrationServices.CodeGenerationRDTIS;
using Ato.EN.IntegrationServices.CodeGenerationRPTTAXPOS;
using DataContracts;
using System.Collections.Generic;
using System.Linq;
using System.Runtime.CompilerServices;
using VaTS;

namespace Ato.CD.Inbound.CTR202602
{
    public class CrossFormValidator
    {
        private CTR2026 ParentDocument { get; }
        private BusinessDocumentCollection ChildDocuments { get; }
        private IEnumerable<CGNFT2018> CGNFTchildDocuments { get; }
        private IEnumerable<DISTBENTRT2024> DISTBENTRTchildDocuments { get; }
        private IEnumerable<IDS2025> IDSchildDocuments { get; }
        private IEnumerable<IEE2021> IEEchildDocuments { get; }
        private IEnumerable<RDTIS2023> RDTISchildDocuments { get; }
        private IEnumerable<RPTTAXPOS2019> RPTTAXPOSchildDocuments { get; }

        public List<ProcessMessageDocument> Response { get; private set; }

        public CrossFormValidator(CTR2026 report, BusinessDocumentCollection childDocuments)
        {
            ParentDocument = report;
            ChildDocuments = childDocuments ?? new BusinessDocumentCollection();
            Response = new List<ProcessMessageDocument>();

            CGNFTchildDocuments = ChildDocuments
                  .Where(s => s.DocumentName.Equals(Schedule.CGNFT.ToString()))
                  .Select(p => (CGNFT2018)p.ConsumedReport);

            DISTBENTRTchildDocuments = ChildDocuments
               .Where(s => s.DocumentName.Equals(Schedule.DISTBENTRT.ToString()))
               .Select(p => (DISTBENTRT2024)p.ConsumedReport);

            IDSchildDocuments = ChildDocuments
               .Where(s => s.DocumentName.Equals(Schedule.IDS.ToString()))
               .Select(p => (IDS2025)p.ConsumedReport);

            IEEchildDocuments = ChildDocuments
                 .Where(s => s.DocumentName.Equals(Schedule.IEE.ToString()))
                 .Select(p => (IEE2021)p.ConsumedReport);

            RDTISchildDocuments = ChildDocuments
                .Where(s => s.DocumentName.Equals(Schedule.RDTIS.ToString()))
                .Select(p => (RDTIS2023)p.ConsumedReport);

            RPTTAXPOSchildDocuments = ChildDocuments
               .Where(s => s.DocumentName.Equals(Schedule.RPTTAXPOS.ToString()))
               .Select(p => (RPTTAXPOS2019)p.ConsumedReport);
        }

        public List<ProcessMessageDocument> ValidateCrossFormRules()
        {
            //CTR-CGLS
            VRATOCTR428100();
            VRATOCTR428135();

            //CTR-CGNFT
            VRATOCGNFT000019();
            VRATOCGNFT000100();

            //CTR-CGTS
            VRATOCTR428096();

            //CTR-DISTBENTRT
            VRATOCTR500391();

            //CTR-IDS
            VRATOCTR430014();
            VRATOCTR430015();
            VRATOCTR430020();
            VRATOCTR448352();
            VRATOCTR448353();
            VRATOCTR500003();

            //CTR-IEE
            VRATOCTR428037();
            VRATOCTR428039();

            //CTR-LS
            VRATOCTR428034();
            VRATOCTR428099();
      
            //CTR-PSS
            VRATOCTR428040();
            VRATOCTR428101();
            VRATOCTR428109();
            VRATOCTR428278();

            //CTR-RDTIS
            VRATOCTR428354();
            VRATOCTR428355();
            VRATOCTR428398();
            VRATOCTR428399();
            VRATOCTR428404();
            VRATOCTR500006();
            VRATOCTR500304();
            VRATOCTR500305();
            VRATOCTR500307();
            VRATOCTR500382();

            //CTR-RPTTAXPOS
            VRATOCTR500002();
            VRATOCTR500004();
            VRATOCTR500005();

            return Response;
        }

        #region CTR-CGLS

        #region VR.ATO.CTR.428100

        /*  VR.ATO.CTR.428100
        Consolidated losses schedule must be attached where Tax Losses and Net Capital Losses > $100000

        Legacy Rule Format:
        IF ([CTR175] + [CTR176] > 100000) AND ([CTR53] = "1") AND (COUNT(SCHEDULE = "CGLS") = 0)
           RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:

        ^CTR175 = CTR:RP:Tax.Losses.CarriedForward.Total.Amount
        ^CTR176 = CTR:RP(Instant):Capital.Losses.CarriedForward.Net.Amount
        ^CTR53  = CTR:RP:OrganisationDetails.TaxConsolidationStatus.Code
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428100()
        {
            ProcessMessageDocument processMessage;
            bool assertion = !ChildDocuments.Any(c => c.DocumentName.Equals("CGLS")) &&
                                       (ParentDocument.CTR175.GetValueOrDefault() + ParentDocument.CTR176.GetValueOrDefault()) > 100000 &&
                                        ParentDocument.CTR53 == "1";
            if (assertion)
            {
                processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.CTR.428100",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Consolidated losses schedule must be attached where Tax Losses and Net Capital Losses greater than $100000",
                    LongDescription = "Consolidated losses schedule must be attached where total of tax losses and net capital losses exceeds $100000 and the Head company indicator is complete at Consolidated group status field",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.CarriedForward.Net.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CTR.428100" } },
                };
             
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR175", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR175) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR176", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR176) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR53", Value = ParentDocument.CTR53 });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CTR.428100

        #region VR.ATO.CTR.428135

        /* VR.ATO.CTR.428135
        Consolidated losses schedule may be required

        Legacy Rule Format:
        IF (([CTR117] > 100000) AND ([CTR53] = "1") AND (COUNT(SCHEDULE = "CGLS") = 0))
        RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:
    
        ^CTR117 = CTR:RP:IncomeTax.Deduction.TaxLossesDeducted.Amount
        ^CTR53 = CTR:RP:OrganisationDetails.TaxConsolidationStatus.Code
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428135()
        {
            ProcessMessageDocument processMessage;
            bool assertion = !ChildDocuments.Any(c => c.DocumentName.Equals("CGLS")) &&
                             ParentDocument.CTR117.GetValueOrDefault() > 100000 &&
                             ParentDocument.CTR53 == "1";
            if (assertion)
            {
                processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.CTR.428135",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Consolidated losses schedule may be required",
                    LongDescription = "If the amount of Tax losses deducted is greater than $100000 and the Consolidated group status indicator is equal to \"1\" for head company then a Consolidated losses schedule must be completed",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.TaxLossesDeducted.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CTR.428135" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR117", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR117) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR53", Value = ParentDocument.CTR53 });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CTR.428135

        #endregion //CTR-CGLS

        #region CTR-CGNFT

        #region VR.ATO.CGNFT.000019

        /* VR.ATO.CGNFT.000019
        The Consolidated group status indicator on the company return must be present and equal to '1' for Head company if the Notification of formation of a consolidated group schedule is present.

        IF (PARENT RETURN <> NULL) AND (CTR:RP:OrganisationDetails.TaxConsolidationStatus.Code <> "1")
            RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:

        ^CTR53 = CTR:RP:OrganisationDetails.TaxConsolidationStatus.Code
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCGNFT000019()
        {
            ProcessMessageDocument processMessage;
            bool assertion = (string.IsNullOrWhiteSpace(ParentDocument.CTR53) || !ParentDocument.CTR53.Equals("1")) && CGNFTchildDocuments.Count() > 0;

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGNFT.000019",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Consolidated group status absent or incorrect on Parent form",
                    LongDescription = @"The Consolidated group status indicator on the Parent company return form must be present and equal to '1' for the Head company if the Notification of formation of a consolidated group schedule is present",
                    Location = "/xbrli:xbrl/tns:OrganisationDetails.TaxConsolidationStatus.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000019" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR53", Value = ParentDocument.CTR53 });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CGNFT.000019

        #region VR.ATO.CGNFT.000100

        /* VR.ATO.CGNFT.000100
        The number of Consolidated Group Notification schedules must not exceed 1.

        Legacy Rule Format:
        IF COUNT(SCHEDULE(CGNFT)) > 1
            RETURN VALIDATION MESSAGE
        ENDIF
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCGNFT000100()
        {
            ProcessMessageDocument processMessage;
            bool assertion = CGNFTchildDocuments.Count() > 1;

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CGNFT.000100",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The number of Consolidated Group Notification schedules must not exceed 1",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CGNFT.000100" } },
                };

                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CGNFT.000100

        #endregion // CTR-CGNFT

        #region CTR-CGTS

        #region VR.ATO.CTR.428096

        /* VR.ATO.CTR.428096
        Capital Gains Tax schedule required, where net capital gains exceeds $10,000

        Legacy Rule Format:
        IF (([CTR91] > 10000) AND 
        (COUNT(SCHEDULE = "CGTS") = 0))
           RETURN VALIDATION MESSAGE
        ENDIF
        
        Data Elements:
        ^CTR91 = CTR:RP:Income.CapitalGainsNet.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428096()
        {
            ProcessMessageDocument processMessage;
            bool assertion = !ChildDocuments.Any(c => c.DocumentName.Equals("CGTS")) && ParentDocument.CTR91.HasValue && ParentDocument.CTR91.Value > 10000;

            if (assertion)
            {
                processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.CTR.428379",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Capital Gains Tax schedule required, where net capital gains exceeds $10,000",
                    LongDescription = "When Net capital gains is in excess of $10,000, a Capital Gains Tax schedule must be attached",
                    Location = "/xbrli:xbrl/tns:Income.CapitalGainsNet.Amount",
                    Parameters = new ProcessMessageParameters { new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.CTR.428096" } }
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR91", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR91) });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CTR.428096

        #endregion // CTR-CGTS

        #region CTR-DISTBENTRT

        #region VR.ATO.CTR.500391

        /*  VR.ATO.CTR.500391
        Distributions to Beneficiaries of Trust schedule must be present when Gross distribution from trusts amount is provided

        Legacy Rule Format:
        IF (COUNT(SCHEDULE= "DISTBENTRT") = 0) AND [CTR61] > 0 
            RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:
    
        ^CTR61 = RP:TrustDistributionIncome:Income.TrustDistributionGross.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR500391()
        {
            ProcessMessageDocument processMessage;
            bool assertion = DISTBENTRTchildDocuments.Count() == 0 && ParentDocument.CTR61.GetValueOrDefault() > 0;

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.500391",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Distributions to Beneficiaries of Trust schedule must be present",
                    LongDescription = @"Distributions to Beneficiaries of Trust schedule must be present when Gross distribution from trusts amount is provided",
                    Location = "/xbrli:xbrl/tns:TrustDistributionIncome/tns:Income.TrustDistributionGross.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CTR.500391" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR61", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR61) });
                Response.Add(processMessage);
            }
        }

        #endregion VR.ATO.CTR.500391

        #endregion //CTR-DISTBENTRT

        #region CTR-IDS

        #region VR.ATO.CTR.430014

        /*  VR.ATO.CTR.430014
        International dealings schedule must be attached if interest expenses overseas amount is greater than zero

        Legacy Rule Format:
        IF [CTR81] > 0 AND (COUNT(SCHEDULE = "IDS") = 0) 
            RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:

        ^CTR81 = CTR:RP:Expense.Operating.Interest.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR430014()
        {
            ProcessMessageDocument processMessage;
            bool assertion = ParentDocument.CTR81.GetValueOrDefault() > 0 && IDSchildDocuments.Count() == 0;

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.430014",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"International dealings schedule required",
                    LongDescription = @"International dealings schedule must be attached if interest expenses overseas amount is greater than zero",
                    Location = "/xbrli:xbrl/tns:Expense.Operating.Interest.Amount[@contextRef='" + ParentDocument.RPJFOREIGNId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CTR.430014" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR81", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR81) });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CTR.430014

        #region VR.ATO.CTR.430015

        /*  VR.ATO.CTR.430015
        International dealings schedule must be attached if royalty expenses overseas amount is greater than zero

        Legacy Rule Format:
        IF [CTR366] > 0 AND (COUNT(SCHEDULE = "IDS") = 0) 
           RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:

        ^CTR366 = CTR:RP:Expense.Royalties.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR430015()
        {
            ProcessMessageDocument processMessage;
            bool assertion = ParentDocument.CTR366.GetValueOrDefault() > 0 && IDSchildDocuments.Count() == 0;

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.430015",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"International dealings schedule required",
                    LongDescription = @"International dealings schedule must be attached if royalty expenses overseas amount is greater than zero",
                    Location = "/xbrli:xbrl/tns:Expense.Royalties.Amount[@contextRef='" + ParentDocument.RPJFOREIGNId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CTR.430015" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR366", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR366) });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CTR.430015

        #region VR.ATO.CTR.430020

        /*  VR.ATO.CTR.430020
        International dealings schedule must be attached if section 46FA deduction for flow-on dividends amount is greater than zero

        Legacy Rule Format:
        IF [CTR98] > 0 AND (COUNT(SCHEDULE = "IDS") = 0) 
           RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:

        ^CTR98 = CTR:RP:Expense.FlowOnDividendDeduction.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR430020()
        {
            ProcessMessageDocument processMessage;
            bool assertion = ParentDocument.CTR98.GetValueOrDefault() > 0 && IDSchildDocuments.Count() == 0;

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.430020",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"International dealings schedule required",
                    LongDescription = @"International dealings schedule must be attached if section 46FA deduction for flow-on dividends amount is greater than zero",
                    Location = "/xbrli:xbrl/tns:Expense.FlowOnDividendDeduction.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CTR.430020" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR98", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR98) });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CTR.430020

        # region VR.ATO.CTR.448352

        /*  VR.ATO.CTR.448352
        If International related party dealings/pricing question is present and equal to true or Thin capitalisation question is present and equal to true. An International Dealings Schedule is not present.

        Legacy Rule Format:
        IF ([CTR194] = TRUE OR [CTR195] = TRUE OR [CTR196] = TRUE) 
        AND
        (COUNT(SCHEDULE = "IDS") = 0)
        RETURN VALIDATION MESSAGE
        ENDIF
        
        Data Elements:
        ^CTR194 = CTR:RP:InternationalDealings.RelatedPartiesTransactionsExcessAggregateValue.Indicator
        ^CTR195 = CTR:RP:InternationalDealings.DirectOrIndirectOverseasInterest.Indicator
        ^CTR196 = CTR:RP:Liabilities.ThinCapitalisation.ProvisionsApplied.Indicator
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR448352()
        {
            ProcessMessageDocument processMessage;
            bool assertion = IDSchildDocuments.Count() == 0 &&
                             (ParentDocument.CTR194.GetValueOrDefault() || ParentDocument.CTR195.GetValueOrDefault() || ParentDocument.CTR196.GetValueOrDefault());

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.458352",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"International Dealings Schedule is required",
                    LongDescription = @"When there is a 'yes' response to any of the questions concerning whether the value of dealings with international related parties is greater than $2 million, or whether the entity had either a direct or indirect interest in a foreign trust, foreign company, controlled foreign entity or transferor trust, or a 'yes' response to the question ""Were the thin capitalisation or debt deduction creation rules applicable to you"", an International Dealings Schedule is required",
                    Location = "/xbrli:xbrl/tns:InternationalDealings.RelatedPartiesTransactionsExcessAggregateValue.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CTR.448353" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR194", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR194) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR195", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR195) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR196", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR196) });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CTR.448352

        #region VR.ATO.CTR.448353

        /* VR.ATO.CTR.448353
        When 'Interest expenses overseas', 'Royalty expenses overseas' and 'Section 46fa deduction for flowon dividends' are not present, and there is a "no" response to the questions concerning whether the value of dealings with international related parties is greater than $2 million, and whether the entity had either a direct or indirect interest in a foreign trust, foreign company, controlled foreign entity or transferor trust, and there is a "no" response to the question 'Were the thin capitalisation or debt deduction creation rules applicable to you', an International Dealings Schedule is not required
       
        Legacy Rule Format:
        IF (([CTR81] = NULL OR [CTR81] = 0) AND ([CTR366] = NULL OR [CTR366] = 0) AND ([CTR98] = NULL OR [CTR98] = 0)) AND ([CTR194] = FALSE AND [CTR195] = FALSE AND [CTR196] = FALSE) 
            AND (COUNT(SCHEDULE = "IDS") = 1)
            RETURN VALIDATION MESSAGE
        ENDIF
        
        Data Elements:
        ^CTR81 = CTR:RP:Expense.Operating.Interest.Amount
        ^CTR366 = CTR:RP:Expense.Royalties.Amount
        ^CTR98 = CTR:RP:Expense.FlowOnDividendDeduction.Amount
        ^CTR194 = CTR:RP:InternationalDealings.RelatedPartiesTransactionsExcessAggregateValue.Indicator
        ^CTR195 = CTR:RP:InternationalDealings.DirectOrIndirectOverseasInterest.Indicator
        ^CTR196 = CTR:RP:Liabilities.ThinCapitalisation.ProvisionsApplied.Indicator
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR448353()
        {
            ProcessMessageDocument processMessage;
            bool assertion = ((!ParentDocument.CTR81.HasValue || ParentDocument.CTR81.Value == 0) &&
                             (!ParentDocument.CTR366.HasValue || ParentDocument.CTR366.Value == 0) &&
                             (!ParentDocument.CTR98.HasValue || ParentDocument.CTR98.Value == 0)) &&
                              !ParentDocument.CTR194.GetValueOrDefault() &&
                              !ParentDocument.CTR195.GetValueOrDefault() &&
                              !ParentDocument.CTR196.GetValueOrDefault() && IDSchildDocuments.Count() == 1;

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.458353",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"International Dealings Schedule is not required",
                    LongDescription = @"When 'Interest expenses overseas', 'Royalty expenses overseas' and 'Section 46fa deduction for flowon dividends' are not present, and there is a ""no"" response to the questions concerning whether the value of dealings with international related parties is greater than $2 million, and whether the entity had either a direct or indirect interest in a foreign trust, foreign company, controlled foreign entity or transferor trust, and there is a ""no"" response to the question 'Were the thin capitalisation or debt deduction creation rules applicable to you', an International Dealings Schedule is not required",
                    Location = "/xbrli:xbrl/tns:InternationalDealings.RelatedPartiesTransactionsExcessAggregateValue.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CTR.448353" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR81", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR81) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR366", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR366) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR98", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR98) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR194", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR194) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR195", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR195) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR196", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR196) });
          
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CTR.448353

        #region VR.ATO.CTR.500003

        /* VR.ATO.CTR.500003
        If the entity has self-assessed as a Significant Global Entity (SGE), then 'Are you a small business entity, not a significant global entity and your international related party dealings do not exceed $5 million and 50% of your current year aggregated turnover?' must not be 'TRUE' on the International Dealings Schedule (IDS) form.

        Legacy Rule Format:
        IF [CTR370] = TRUE AND (COUNT(SCHEDULE = "IDS") = 1 AND [IDS425] = TRUE)
            RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:

        ^CTR370 = CTR:RP:OrganisationDetails.SignificantGlobalEntityStatus.Indicator
        ^IDS425 = IDS:RP:InternationalDealings.SmallBusinessConcessionsEligibility.Indicator
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR500003()
        {
            ProcessMessageDocument processMessage;
            bool assertion = ParentDocument.CTR370.GetValueOrDefault() && IDSchildDocuments.Count() == 1 && IDSchildDocuments.First().IDS425.GetValueOrDefault();

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.500003",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Significant Global Entity question on the International Dealings Schedule (IDS) form must not be 'TRUE'.",
                    LongDescription = @"If the entity has self-assessed as a Significant Global Entity (SGE), then 'Are you a small business entity, not a significant global entity and your international related party dealings do not exceed 50% of your current year aggregated turnover?' must not be 'TRUE' on the International Dealings Schedule (IDS) form.",
                    Location = "/xbrli:xbrl/tns:OrganisationDetails.SignificantGlobalEntityStatus.Indicator",

                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CTR.500003" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR370", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR370) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "IDS425", Value = CTR2026Validator.GetValueOrEmpty(IDSchildDocuments.First().IDS425) });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CTR.500003

        #endregion // CTR-IDS

        #region CTR-IEE

        #region VR.ATO.CTR.428037

        /* VR.ATO.CTR.428037
        IEE specified income year to be completed when IEE schedule is attached

        Legacy Rule Format:
        IF ([CTR55] = NULL) AND (COUNT(SCHEDULE = "IEE") > 0)
           RETURN VALIDATION MESSAGE
        ENDIF
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428037()
        {
            ProcessMessageDocument processMessage;

            bool assertion = ParentDocument.CTR55 == null && IEEchildDocuments.Count() > 0;
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428037",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"IEE specified income year to be completed when IEE schedule is attached",
                    LongDescription = @"An IEE schedule is attached: however the IEE specified income year has not been completed on the Company Income Tax Return",
                    Location = "/xbrli:xbrl/tns:Elections.InterposedEntityElectionStatus.Year",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CTR.428037" } },
                };

                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CTR.428037

        #region VR.ATO.CTR.428039

        /*  VR.ATO.CTR.428039
        Interposed Entity Election or Revocation schedule required where a revocation is being made

        Legacy Rule Format:
        IF ([CTR56] = "R") AND (COUNT(SCHEDULE = "IEE") = 0)
            RETURN VALIDATION MESSAGE
        ENDIF
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428039()
        {
            ProcessMessageDocument processMessage;
            bool assertion = !string.IsNullOrWhiteSpace(ParentDocument.CTR56) && ParentDocument.CTR56.ToUpper() == "R" && IEEchildDocuments.Count() == 0;

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428039",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Interposed Entity Election or Revocation schedule required where a revocation is being made",
                    LongDescription = @"The Interposed Entity Election code indicates a revocation is being made, however no schedule is attached",
                    Location = "/xbrli:xbrl/tns:Elections.InterposedEntityElectionOrRevocation.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CTR.428039" } },
                };

                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CTR.428039

        #endregion CTR-IEE

        #region CTR-LS

        #region VR.ATO.CTR.428034

        /*  VR.ATO.CTR.428034
        Losses schedule BP cannot be lodged with a consolidated Head company return.

        Legacy Rule Format:
        IF ([CTR53] = "1") AND (COUNT(SCHEDULE = "LS") = 1)
           RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:

        ^CTR53 = CTR:RP:OrganisationDetails.TaxConsolidationStatus.Code
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428034()
        {
            ProcessMessageDocument processMessage;
            bool assertion = ChildDocuments.Any(c => c.DocumentName.Equals("LS")) && ParentDocument.CTR53 == "1";

            if (assertion)
            {
                processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.CTR.428034",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Losses schedule BP cannot be lodged with a consolidated Head company return",
                    LongDescription = "A consolidated head company is not required to lodge a Losses schedule for unconsolidated entities",
                    Location = "/xbrli:xbrl/tns:OrganisationDetails.TaxConsolidationStatus.Code",
                    Parameters = new ProcessMessageParameters { new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.CTR.428034" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR53", Value = ParentDocument.CTR53 });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CTR.428034

        #region VR.ATO.CTR.428099

        /*  VR.ATO.CTR.428099
        Losses schedule must be attached where total of tax losses and net capital losses exceeds $100000
         
        Legacy Rule Format:
        IF ([CTR175] + [CTR176] > 100000) AND 
        ([CTR53] = NULLORBLANK) AND (COUNT(SCHEDULE = "LS") = 0)
           RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:

        ^CTR53 = CTR:RP:OrganisationDetails.TaxConsolidationStatus.Code
        ^CTR175 = CTR:RP:Tax.Losses.CarriedForward.Total.Amount
        ^CTR176 = CTR:RP(Instant):Capital.Losses.CarriedForward.Net.Amount
       */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428099()
        {
            ProcessMessageDocument processMessage;
            bool assertion = !ChildDocuments.Any(c => c.DocumentName.Equals("LS")) &&
                             string.IsNullOrWhiteSpace(ParentDocument.CTR53) &&
                             ParentDocument.CTR175.GetValueOrDefault() + ParentDocument.CTR176.GetValueOrDefault() > 100000;
            if (assertion)
            {
                processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.CTR.428099",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Losses schedule must be attached where total of tax losses and net capital losses exceeds $100000",
                    LongDescription = "Where the total of tax losses and net capital losses exceeds $100000 and the company is not part of a consolidated group, a Losses schedule must be attached",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.CarriedForward.Net.Amount",
                    Parameters = new ProcessMessageParameters { new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.CTR.428099" } }
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR53", Value = ParentDocument.CTR53 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR175", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR175) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR176", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR176) });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CTR.428099

        #endregion // CTR-LS

        #region CTR-PSS

        #region VR.ATO.CTR.428040

        /*  VR.ATO.CTR.428040
        Gross payments are required on Payment Summary Schedule, where amount entered at Gross Payments where ABN not quote

        Legacy Rule Format:
        IF (([CTR58] > 0)  AND (COUNT(SCHEDULE = "PSS") = 0)) OR ((SUM[PSS19]) <> [CTR58])
           RETURN VALIDATION MESSAGE
        ENDIF

         Data Elements:

        ^CTR58 = CTR:RP:Remuneration.ABNNotQuotedPaymentGross.Amount
        ^SUM[PSS19] = PSS:RP.Payer.{PSSeqNum}:Remuneration.ABNNotQuotedPaymentGross.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428040()
        {
            ProcessMessageDocument processMessage;
            PSS2018 PSSChildDocument = null;
            decimal? sumPSS19 = 0;

            if (ChildDocuments.Where(c => c.DocumentName.Equals("PSS")).Count() == 1)
            {
                var PSSchildDocuments = ChildDocuments.Where(c => c.DocumentName.Equals("PSS")).First();
                PSSChildDocument = (PSS2018)PSSchildDocuments.ConsumedReport;
                sumPSS19 = PSSChildDocument.RPPayerPSSeqNumContextCollection.Select(Item => Item.PSS19).Sum();
            }

            bool assertion = ((PSSChildDocument == null) && ParentDocument.CTR58.GetValueOrDefault() > 0) || (PSSChildDocument != null && ParentDocument.CTR58.GetValueOrDefault() != sumPSS19);

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428040",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Gross payments are required on Payment Summary Schedule, where amount entered at Gross Payments where ABN not quoted",
                    LongDescription = "An amount has been entered at gross payment where ABN is not quoted and there are no corresponding amounts on the Payment Summary Schedule",
                    Location = "/xbrli:xbrl/tns:Remuneration.ABNNotQuotedPaymentGross.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CTR.428040" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR58", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR58) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "SUM[PSS19]", Value = CTR2026Validator.GetValueOrEmpty(sumPSS19) });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CTR.428040

        #region VR.ATO.CTR.428101

        /*  VR.ATO.CTR.428101
        Amount withheld required on Payment Summary schedule

        Legacy Rule Format:
        IF ([CTR208] > 0) AND (((COUNT(SCHEDULE = "PSS") = 0) OR (SUM([PSS23]) <> [CTR208])))
           RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:

        ^CTR208 = CTR:RP:IncomeTax.PayAsYouGoWithholding.TaxWithheld.Amount WHERE (rvctc2.02.00:IncomeTax.PayAsYouGoWithholding.PaymentType.Code = "DNOABN") IN TUPLE(ctr.0009.lodge.req.02.00:ITPAYGWithholdingDetails)
        ^SUM[PSS23] = PSS:RP.Payer.{PSSeqNum}:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldBusinessABNNotQuoted.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428101()
        {
            ProcessMessageDocument processMessage;
            PSS2018 PSSChildDocument = null;
            CTR2026Validator validator = new CTR2026Validator();
            decimal? sumPSS23 = 0;

            if (ChildDocuments.Where(c => c.DocumentName.Equals("PSS")).Count() == 1)
            {
                var PSSchildDocuments = ChildDocuments.Where(c => c.DocumentName.Equals("PSS")).First();
                PSSChildDocument = (PSS2018)PSSchildDocuments.ConsumedReport;
                sumPSS23 = PSSChildDocument.RPPayerPSSeqNumContextCollection.Select(Item => Item.PSS23).Sum();
            }

            bool assertion = ParentDocument.CTR208.GetValueOrDefault() > 0 &&
                   (PSSChildDocument == null || ParentDocument.CTR208.GetValueOrDefault() != sumPSS23);

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428101",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Amount withheld required on Payment Summary schedule",
                    LongDescription = "An amount has been entered at tax withheld from payments where ABN no quoted in the calculation statement and on the Payment Summary schedule there are no corresponding amounts and the payment type is \"N\"",
                    Location = "/xbrli:xbrl/tns:ITPAYGWithholdingDetails" + validator.OccurrenceIndex(ParentDocument.ItpaygWithholdingDetailsCollection_DNOABN_RPOccurrenceIndex) + "/tns:IncomeTax.PayAsYouGoWithholding.TaxWithheld.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CTR.428101" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR208", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR208) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "SUM[PSS23]", Value = CTR2026Validator.GetValueOrEmpty(sumPSS23) });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CTR.428101

        #region VR.ATO.CTR.428109

        /*  VR.ATO.CTR.428109
        Credit for tax withheld-foreign resident withholding (excluding capital gains) must not be less than the amount present in the schedule

        Legacy Rule Format:
        IF (([CTR57] > 0) AND (COUNT(SCHEDULE = "PSS") = 0)) OR ([CTR207] < SUM[PSS22])
            RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:

        ^CTR57 = CTR:RP:Remuneration.PaymentToForeignResidentGross.Amount
        ^CTR207 = CTR:RP:IncomeTax.PayAsYouGoWithholding.TaxWithheld.Amount WHERE (rvctc2.02.00:IncomeTax.PayAsYouGoWithholding.PaymentType.Code = "DFRW") IN TUPLE(ctr.0009.lodge.req.02.00:ITPAYGWithholdingDetails)
        ^SUM[PSS22] = PSS:RP.Payer.{PSSeqNum}:IncomeTax.PayAsYouGoWithholding.CreditTaxWithheldBusinessForeignResident.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428109()
        {
            ProcessMessageDocument processMessage;
            PSS2018 PSSChildDocument = null;
            CTR2026Validator validator = new CTR2026Validator();
            decimal? sumPSS22 = 0;

            if (ChildDocuments.Where(c => c.DocumentName.Equals("PSS")).Count() == 1)
            {
                var PSSchildDocuments = ChildDocuments.Where(c => c.DocumentName.Equals("PSS")).First();
                PSSChildDocument = (PSS2018)PSSchildDocuments.ConsumedReport;
                sumPSS22 = PSSChildDocument.RPPayerPSSeqNumContextCollection.Select(Item => Item.PSS22).Sum();
            }

            bool assertion = ((PSSChildDocument == null) && ParentDocument.CTR57.GetValueOrDefault() > 0)
                          || (PSSChildDocument != null && ParentDocument.CTR207.GetValueOrDefault() < sumPSS22);

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.438074",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Credit for tax withheld-foreign resident withholding (excluding capital gains) must not be less than the amount present in the schedule.",
                    LongDescription = "There is an amount present at Gross payments subject to foreign resident withholding (excluding capital gains) and Credit for tax withheld - foreign resident withholding (excluding capital gains) (Calculation Statement) on the return form does not equal the sum of the amounts on the Payment Summary schedule.",
                    Location = "/xbrli:xbrl/tns:ITPAYGWithholdingDetails" + validator.OccurrenceIndex(ParentDocument.ItpaygWithholdingDetailsCollection_DFRW_RPOccurrenceIndex) + "/tns:IncomeTax.PayAsYouGoWithholding.TaxWithheld.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CTR.428109" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR57", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR57) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR207", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR207) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "SUM[PSS22]", Value = CTR2026Validator.GetValueOrEmpty(sumPSS22) });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CTR.428109

        #region VR.ATO.CTR.428278

        /*  VR.ATO.CTR.428278
        If there is an amount greater than zero at 'Gross payments subject to foreign resident withholding (excluding capital gains)' a Non-individual Payment Summary schedule must be present, and sum of the Gross payments subject to foreign resident withholding (excluding capital gains) on the schedule must equal the amount on the main form.

        Legacy Rule Format:
        IF (([CTR57] > 0) AND (COUNT(SCHEDULE = "PSS") = 0)) OR ((SUM[PSS18]) <> [CTR57])
            RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:
        ^CTR57 = CTR:RP:Remuneration.PaymentToForeignResidentGross.Amount
        ^SUM[PSS18] = PSS:RP.Payer.{PSSeqNum}:Remuneration.PaymentToForeignResidentGross.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428278()
        {
            ProcessMessageDocument processMessage;
            PSS2018 PSSChildDocument = null;
            decimal? sumPSS18 = 0;

            if (ChildDocuments.Where(c => c.DocumentName.Equals("PSS")).Count() == 1)
            {
                var PSSchildDocuments = ChildDocuments.Where(c => c.DocumentName.Equals("PSS")).First();
                PSSChildDocument = (PSS2018)PSSchildDocuments.ConsumedReport;
                sumPSS18 = PSSChildDocument.RPPayerPSSeqNumContextCollection.Select(Item => Item.PSS18).Sum();
            }

            bool assertion = ((PSSChildDocument == null) && ParentDocument.CTR57.GetValueOrDefault() > 0)
                            || (PSSChildDocument != null && ParentDocument.CTR57.GetValueOrDefault() != sumPSS18);

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.438075",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Gross payments are required on Payment Summary Schedule, where amount entered at Gross payments subject to foreign resident withholding (excluding capital gains).",
                    LongDescription = "An amount has been entered at gross payment where Gross payments are subject to foreign resident withholding (excluding capital gains) and there is no corresponding amounts on the Payment Summary Schedule.",
                    Location = "/xbrli:xbrl/tns:Remuneration.PaymentToForeignResidentGross.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CTR.428278" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR57", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR57) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "SUM[PSS18]", Value = CTR2026Validator.GetValueOrEmpty(sumPSS18) });
                Response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.CTR.428278

        #endregion CTR-PSS

        #region  CTR-RDTIS

        #region VR.ATO.CTR.428354

        /*  VR.ATO.CTR.428354
        Where R&D labels have been completed a Research and Development Tax Incentive schedule must be attached.

        Legacy Rule Format:
        IF ([CTR97] <> NULL OR [CTR328] <> NULL OR [CTR210] <> NULL) AND (COUNT(SCHEDULE = "RDTIS") = 0)
            RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:

        ^CTR97 = CTR:RP:TaxConcession.ResearchAndDevelopment.WriteBackOfAccountingExpenditure.Amount
        ^CTR328 = CTR:RP:TaxConcession.ResearchAndDevelopment.TaxOffsetNonRefundable.Amount
        ^CTR210 = CTR:RP:TaxConcession.ResearchAndDevelopment.TaxOffsetRefundable.Amount
         */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428354()
        {
            ProcessMessageDocument processMessage;
            bool assertion = (ParentDocument.CTR97.HasValue || ParentDocument.CTR328.HasValue || ParentDocument.CTR210.HasValue) && RDTISchildDocuments.Count() == 0;

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428354",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Research and Development Tax Incentive schedule must be attached",
                    LongDescription = @"Where R&D labels have been completed a Research and Development Tax Incentive schedule must be attached.",
                    Location = "/xbrli:xbrl/tns:TaxConcession.ResearchAndDevelopment.WriteBackOfAccountingExpenditure.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CTR.428354" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR97", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR97) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR328", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR328) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR210", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR210) });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CTR.428354

        #region VR.ATO.CTR.428355

        /*  VR.ATO.CTR.428355
        R&D labels do not equal their matching labels in the Research and Development Tax Incentive schedule.

        Legacy Rule Format:
        IF [CTR97] <> [RDTIS3] AND (COUNT(SCHEDULE = "RDTIS") = 1) 
            RETURN VALIDATION MESSAGE
        ENDIF
        
        Data Elements:

        ^CTR97 = CTR:RP:TaxConcession.ResearchAndDevelopment.WriteBackOfAccountingExpenditure.Amount
        ^RDTIS48 = RDTIS:RP:Expense.ResearchAndDevelopment.ResearchAndDevelopmentAccountingExpenditureAddBack.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428355()
        {
            ProcessMessageDocument processMessage;
            bool assertion = RDTISchildDocuments.Count() == 1 && (ParentDocument.CTR97.GetValueOrDefault() != RDTISchildDocuments.First().RDTIS3.GetValueOrDefault());

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428355",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"R&D label mismatch",
                    LongDescription = @"R&D labels do not equal their matching labels in the Research and Development Tax Incentive schedule.",
                    Location = "/xbrli:xbrl/tns:TaxConcession.ResearchAndDevelopment.WriteBackOfAccountingExpenditure.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CTR.428355" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR97", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR97) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "RDTIS3", Value = CTR2026Validator.GetValueOrEmpty(RDTISchildDocuments.First().RDTIS3) });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CTR.428355

        #region VR.ATO.CTR.428398

        /*  VR.ATO.CTR.428398
        R&D labels do not equal their matching labels in the Research and Development Tax Incentive schedule.

        Legacy Rule Format:
        IF [CTR328] <> [RDTIS48] AND (COUNT(SCHEDULE = "RDTIS") = 1) 
            RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:

        ^CTR328 = CTR:RP:TaxConcession.ResearchAndDevelopment.TaxOffsetNonRefundable.Amount
        ^RDTIS48 = RDTIS:RP:TaxConcession.ResearchAndDevelopment.TaxOffsetNonRefundable.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428398()
        {
            ProcessMessageDocument processMessage;

            bool assertion = RDTISchildDocuments.Count() == 1 &&
                            (ParentDocument.CTR328.GetValueOrDefault() != RDTISchildDocuments.First().RDTIS48.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428355",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"R&D label mismatch",
                    LongDescription = @"R&D labels do not equal their matching labels in the Research and Development Tax Incentive schedule.",
                    Location = "/xbrli:xbrl/tns:TaxConcession.ResearchAndDevelopment.TaxOffsetNonRefundable.Amount[@contextRef='" + ParentDocument.RPId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CTR.428398" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR328", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR328) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "RDTIS48", Value = CTR2026Validator.GetValueOrEmpty(RDTISchildDocuments.First().RDTIS48) });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CTR.428398

        #region VR.ATO.CTR.428399

        /*  VR.ATO.CTR.428399
        R&D labels do not equal their matching labels in the Research and Development Tax Incentive schedule.

        Legacy Rule Format:
        IF [CTR210] <> [RDTIS46] AND (COUNT(SCHEDULE = "RDTIS") = 1) 
           RETURN VALIDATION MESSAGE
        ENDIF

         Data Elements:

        ^CTR210 = CTR:RP:TaxConcession.ResearchAndDevelopment.TaxOffsetRefundable.Amount
        ^RDTIS46 = RDTIS:RP:TaxConcession.ResearchAndDevelopment.TaxOffsetRefundable.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428399()
        {
            ProcessMessageDocument processMessage;
            bool assertion = RDTISchildDocuments.Count() == 1 && (ParentDocument.CTR210.GetValueOrDefault() != RDTISchildDocuments.First().RDTIS46.GetValueOrDefault());

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428355",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"R&D label mismatch",
                    LongDescription = @"R&D labels do not equal their matching labels in the Research and Development Tax Incentive schedule.",
                    Location = "/xbrli:xbrl/tns:TaxConcession.ResearchAndDevelopment.TaxOffsetRefundable.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CTR.428399" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR210", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR210) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "RDTIS46", Value = CTR2026Validator.GetValueOrEmpty(RDTISchildDocuments.First().RDTIS46) });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CTR.428399 

        #region VR.ATO.CTR.428404

        /*  VR.ATO.CTR.428404
        An amended R&D tax incentive schedule may only be attached to an amended main form

        Legacy Rule Format:
        IF (COUNT(SCHEDULE = "RDTIS") = 1) AND [RDTIS1] = TRUE AND [CTR298] <> TRUE
            RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:

        ^CTR298 = CTR:RP:Report.Amendment.Indicator
        ^RDTIS1 = RDTIS:RP:Report.Amendment.Indicator
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR428404()
        {
            ProcessMessageDocument processMessage;
            bool assertion = RDTISchildDocuments.Count() == 1 && (RDTISchildDocuments.First().RDTIS1.HasValue && RDTISchildDocuments.First().RDTIS1.Value == true)
                 && (!ParentDocument.CTR298.HasValue || ParentDocument.CTR298.GetValueOrDefault() != true);

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428404",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"An amended R&D tax incentive schedule may only be attached to an amended main form",
                    Location = "/xbrli:xbrl/tns:Report.Amendment.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CTR.428404" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR298", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR298) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "RDTIS1", Value = CTR2026Validator.GetValueOrEmpty(RDTISchildDocuments.First().RDTIS1) });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CTR.428404 

        #region VR.ATO.CTR.500006

        /*  VR.ATO.CTR.500006
        AusIndustry Innovation Australia number must be provided for lodgements made on or after the year 2019.

        Legacy Rule Format:
        IF (COUNT(SCHEDULE = "RDTIS") = 1) AND [RDTIS2] = BLANK
            RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:
    
        ^RDTIS2 = RDTIS:RP:Identifiers.InnovationRegistrationNumber.Identifier
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR500006()
        {
            ProcessMessageDocument processMessage;
            bool assertion = (RDTISchildDocuments.Count() == 1 && string.IsNullOrWhiteSpace(RDTISchildDocuments.First().RDTIS2));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Mandatory field not supplied",
                    Location = "/tns:RDTIS/tns:RP/tns:InnovationRegistrationNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CTR.500006" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "RDTIS2", Value = RDTISchildDocuments.First().RDTIS2 });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CTR.500006

        #region VR.ATO.CTR.500304

        /*  VR.ATO.CTR.500304
        R&D labels do not equal their matching labels in the Research and Development Tax Incentive schedule.

        Legacy Rule Format:
        IF [CTR450] <> [RDTIS76] AND (COUNT(SCHEDULE = "RDTIS") = 1) 
            RETURN VALIDATION MESSAGE
        ENDIF

        ^CTR450  = CTR:RP:Income.ResearchAndDevelopment.Clawback.Amount
        ^RDTIS76 = RDTIS:RP:Income.ResearchAndDevelopment.Clawback.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR500304()
        {
            ProcessMessageDocument processMessage;
            bool bothParamsAreNull = false;

            if (RDTISchildDocuments.FirstOrDefault() != null)
            {
                bothParamsAreNull = RDTISchildDocuments.First().RDTIS76 == null && ParentDocument.CTR450 == null;
            }

            bool assertion = !bothParamsAreNull &&
                            ((RDTISchildDocuments.Count() == 1 && RDTISchildDocuments.First().RDTIS76.GetValueOrDefault() != ParentDocument.CTR450.GetValueOrDefault())
                            || (RDTISchildDocuments.Count() == 1 && RDTISchildDocuments.First().RDTIS76.GetValueOrDefault() == 0 && ParentDocument.CTR450 == null)
                            || (RDTISchildDocuments.Count() == 1 && RDTISchildDocuments.First().RDTIS76 == null && ParentDocument.CTR450 == 0));

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428355",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"R&D label mismatch",
                    LongDescription = @"R&D labels do not equal their matching labels in the Research and Development Tax Incentive schedule.",
                    Location = "/xbrli:xbrl/tns:Income.ResearchAndDevelopment.Clawback.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CTR.500304" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR450", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR450) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "RDTIS76", Value = CTR2026Validator.GetValueOrEmpty(RDTISchildDocuments.First().RDTIS76) });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CTR.500304

        #region VR.ATO.CTR.500305

        /*  VR.ATO.CTR.500305
        R&D labels do not equal their matching labels in the Research and Development Tax Incentive schedule.

        Legacy Rule Format:
        IF [CTR451] <> [RDTIS77] AND (COUNT(SCHEDULE = "RDTIS") = 1) 
            RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:

        ^CTR451  = CTR:RP:Expense.ResearchAndDevelopment.AdjustmentBalance.Amount
        ^RDTIS77 = RDTIS:RP:Expense.ResearchAndDevelopment.AdjustmentBalance.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR500305()
        {
            ProcessMessageDocument processMessage;
            bool bothParamsAreNull = false;

            if (RDTISchildDocuments.FirstOrDefault() != null)
            {
                bothParamsAreNull = RDTISchildDocuments.First().RDTIS77 == null && ParentDocument.CTR451 == null;
            }

            bool assertion = !bothParamsAreNull &&
                            ((RDTISchildDocuments.Count() == 1 && RDTISchildDocuments.First().RDTIS77.GetValueOrDefault() != ParentDocument.CTR451.GetValueOrDefault())
                            || (RDTISchildDocuments.Count() == 1 && RDTISchildDocuments.First().RDTIS77.GetValueOrDefault() == 0 && ParentDocument.CTR451 == null)
                            || (RDTISchildDocuments.Count() == 1 && RDTISchildDocuments.First().RDTIS77 == null && ParentDocument.CTR451 == 0));


            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.428355",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"R&D label mismatch",
                    LongDescription = @"R&D labels do not equal their matching labels in the Research and Development Tax Incentive schedule.",
                    Location = "/xbrli:xbrl/tns:Expense.ResearchAndDevelopment.AdjustmentBalance.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CTR.500305" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR451", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR451) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "RDTIS77", Value = CTR2026Validator.GetValueOrEmpty(RDTISchildDocuments.First().RDTIS77) });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CTR.500305

        #region VR.ATO.CTR.500307

        /*  VR.ATO.CTR.500307
        If the base rate entity indicator response is TRUE, the tax rate in Research and Development Tax Incentive schedule should be 25%

        Legacy Rule Format:
        IF [CTR375] = TRUE AND (COUNT(SCHEDULE = "RDTIS") = 1) AND ([RDTIS78] <> NULL AND [RDTIS78] <> SET("25", "25.00"))
            RETURN VALIDATION MESSAGE
        ENDIF

         Data Elements:

        ^CTR375 = CTR:RP:Elections.BaseRateEntityElection.Indicator
        ^RDTIS78 = RDTIS:RP:TaxRate.Designation.Rate
       */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR500307()
        {
            ProcessMessageDocument processMessage;
            bool assertion = (ParentDocument.CTR375.GetValueOrDefault() == true) && (RDTISchildDocuments.Count() == 1 &&
                (RDTISchildDocuments.First().RDTIS78.HasValue && RDTISchildDocuments.First().RDTIS78.Value != 25));

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.500307",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax rate in Research and Development Tax Incentive schedule is incorrect",

                    Location = "/xbrli:xbrl/tns:Elections.BaseRateEntityElection.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CTR.500307" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR375", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR375) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "RDTIS78", Value = CTR2026Validator.GetValueOrEmpty(RDTISchildDocuments.First().RDTIS78) });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CTR.500307

        #region VR.ATO.CTR.500382

        /*  VR.ATO.CTR.500382
        If the base rate entity indicator response is not provided, the tax rate in Research and Development Tax Incentive schedule should be 30%

        Legacy Rule Format:
        IF ([CTR375] = FALSE OR  [CTR375] = NULL) AND (COUNT(SCHEDULE = "RDTIS") = 1) AND ([RDTIS78] <> NULL AND [RDTIS78] <> SET("30", "30.00"))
            RETURN VALIDATION MESSAGE
        ENDIF 

        Data Elements:

        ^CTR375 = CTR:RP:Elections.BaseRateEntityElection.Indicator
        ^RDTIS78 = RDTIS:RP:TaxRate.Designation.Rate
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR500382()
        {
            ProcessMessageDocument processMessage;
            bool assertion = (ParentDocument.CTR375.GetValueOrDefault() == false || ParentDocument.CTR375 == null) && (RDTISchildDocuments.Count() == 1 &&
               (RDTISchildDocuments.First().RDTIS78.HasValue && RDTISchildDocuments.First().RDTIS78.Value != 30));

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.500382",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax rate in Research and Development Tax Incentive schedule is incorrect",
                    Location = "/xbrli:xbrl/tns:Elections.BaseRateEntityElection.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CTR.500382" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR375", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR375) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "RDTIS78", Value = CTR2026Validator.GetValueOrEmpty(RDTISchildDocuments.First().RDTIS78) });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CTR.500382

        #endregion //  CTR-RDTIS

        #region CTR-RPTTAXPOS

        #region VR.ATO.CTR.500002

        /*  VR.ATO.CTR.500002
        Reportable Tax Position Schedule must be attached if there is a 'Yes' response to 'Are you required to lodge a reportable tax position schedule?'

        Legacy Rule Format:
        IF ([CTR319] = TRUE) AND (COUNT(SCHEDULE = "RPTTAXPOS") = 0)

        Data Elements:

        ^CTR319  = CTR:RP:RegulatoryDisclosures.ReportableTaxPosition.Indicator
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR500002()
        {
            ProcessMessageDocument processMessage;
            bool assertion = ParentDocument.CTR319.GetValueOrDefault() && RPTTAXPOSchildDocuments.Count() == 0;

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.CTR.500002",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Reportable Tax Position Schedule must be attached",
                    LongDescription = @"Reportable Tax Position Schedule must be attached if there is a 'Yes' response to 'Are you required to lodge a reportable tax position schedule?'",
                    Location = "/xbrli:xbrl/tns:RegulatoryDisclosures.ReportableTaxPosition.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CTR.500002" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR319", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR319) });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CTR.500002 

        #region VR.ATO.CTR.500004

        /*  VR.ATO.CTR.500004
        The supplied TFN in the Reportable Tax Position Schedule does not match the TFN supplied on the form it was submitted with

        Legacy Rule Format:
        IF COUNT(SCHEDULE = "RPTTAXPOS") = 1 AND (CTR:RP:entity.identifier.TFN <> RPTTAXPOS:RP:entity.identifier.TFN)
            RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:

        ^CTR TFN  = CTR:RP:entity.identifier.TFN
        ^RPTTAXPOS TFN  = RPTTAXPOS:RP:entity.identifier.TFN
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR500004()
        {
            ProcessMessageDocument processMessage;
            bool assertion = RPTTAXPOSchildDocuments.Count() > 0 && !ParentDocument.RPIdentifierTFN.Equals(RPTTAXPOSchildDocuments.First().RPIdentifierTFN);

            if (assertion)
            {
                processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.402009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your supplied TFN does not match the TFN supplied on the form it was submitted with",
                    LongDescription = @"Your supplied TFN does not match the TFN supplied on the parent form it was submitted with",
                    Location = "/xbrli:xbrl/xbrli:context/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters { new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.CTR.500004" } }
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "CTR TFN", Value = ParentDocument.RPIdentifierTFN });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "RPTTAXPOS TFN", Value = RPTTAXPOSchildDocuments.First().RPIdentifierTFN });
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CTR.500004

        #region VR.ATO.CTR.500005

        /*  VR.ATO.CTR.500005
        The first four digits of RTP Category Number must refer to your current financial year.

        Legacy Rule Format:
        IF COUNT(SCHEDULE = "RPTTAXPOS") = 1 AND Substring(RPTTAXPOS:RP:bafot.xx.xx:RegulatoryDisclosures.ReportableTaxPositionCategoryReferenceNumber.Identifier,1,4,) <> [CTR2]
          RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:

        ^CTR2 = CTR:RP:Report.TargetFinancial.Year
        ^RPTTAXPOS4 = RPTTAXPOS:RP:ReportableTaxPositionCategoryAAndB:RegulatoryDisclosures.ReportableTaxPositionCategoryReferenceNumber.Identifier
        ^RPTTAXPOS11 = RPTTAXPOS:RP:ReportableTaxPositionCategoryC:RegulatoryDisclosures.ReportableTaxPositionCategoryReferenceNumber.Identifier
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCTR500005()
        {
            if (RPTTAXPOSchildDocuments.Count() == 1)
            {
                //Check RPTTAXPOS4
                if (RPTTAXPOSchildDocuments.First().ReportableTaxPositionCategoryAAndBCollection != null)
                    Response.AddRange(from rptCategoryAAndB in RPTTAXPOSchildDocuments.First().ReportableTaxPositionCategoryAAndBCollection
                                      where rptCategoryAAndB.RPTTAXPOS4 != null && rptCategoryAAndB.RPTTAXPOS4.Length == 8 && rptCategoryAAndB.RPTTAXPOS4.Substring(0, 4).ToString() != ParentDocument.CTR2.GetValueOrDefault().ToString()
                                      select new ProcessMessageDocument
                                      {
                                          Code = "CMN.ATO.CTR.500005",
                                          Severity = ProcessMessageSeverity.Error,
                                          Description = @"The first four digits of RTP Category Number must refer to your current financial year.",
                                          LongDescription = @"The first four digits of RTP Category Number must refer to your current financial year.",
                                          Location = "/xbrli:xbrl/tns:Report.TargetFinancial.Year",
                                          Parameters = new ProcessMessageParameters()
                                          {
                                               new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CTR.500005" },
                                               new ProcessMessageParameter() { Name = "CTR2", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR2) },
                                               new ProcessMessageParameter() { Name = "RPTTAXPOS4", Value = rptCategoryAAndB.RPTTAXPOS4 }
                                          },
                                      });

                //Check RPTTAXPOS11
                if (RPTTAXPOSchildDocuments.First().ReportableTaxPositionCategoryCCollection != null)
                    Response.AddRange(from rptCategoryC in RPTTAXPOSchildDocuments.First().ReportableTaxPositionCategoryCCollection
                                      where rptCategoryC.RPTTAXPOS11 != null && rptCategoryC.RPTTAXPOS11.Length == 8 && rptCategoryC.RPTTAXPOS11.Substring(0, 4).ToString() != ParentDocument.CTR2.GetValueOrDefault().ToString()
                                      select new ProcessMessageDocument
                                      {
                                          Code = "CMN.ATO.CTR.500005",
                                          Severity = ProcessMessageSeverity.Error,
                                          Description = @"The first four digits of RTP Category Number must refer to your current financial year.",
                                          LongDescription = @"The first four digits of RTP Category Number must refer to your current financial year.",
                                          Location = "/xbrli:xbrl/tns:Report.TargetFinancial.Year",
                                          Parameters = new ProcessMessageParameters()
                                          {
                                               new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.CTR.500005" },
                                               new ProcessMessageParameter() { Name = "CTR2", Value = CTR2026Validator.GetValueOrEmpty(ParentDocument.CTR2) },
                                               new ProcessMessageParameter() { Name = "RPTTAXPOS11", Value = rptCategoryC.RPTTAXPOS11 }
                                          },
                                      });
            }
        }
        #endregion // VR.ATO.CTR.500005

        #endregion CTR-RPTTAXPOS
    }
}
