using DataContracts;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Text.RegularExpressions;
using System.Xml;
using System.Xml.Schema;

namespace Ato.EN.IntegrationServices.CodeGenerationFITR
{
    /// <summary>
    /// XML Consumer for FITR
    /// </summary>
    public class FITR2026XmlConsumer
    {
        const string MissingMandatoryMessage = "A mandatory field has not been completed.";
        const string ContactYourProviderMsg = "The message did not pass XML validation. Please contact your software provider.";
        const string InvalidDataValidation = "A field contains invalid data (such as letters in numeric or date field).";
        private const string ErrorCode1 = "CMN.ATO.GEN.XML01";
        private const string ErrorCode3 = "CMN.ATO.GEN.XML03";
        private const string ErrorCode4 = "CMN.ATO.GEN.XML04";
        private const string ErrorCode6 = "CMN.ATO.GEN.XML06";
        private bool _found = false;
        private bool _isExiting = false;
        private bool _isValidationError = false;
        private string _lastPath = string.Empty;
        private ErrorMessageType _validationError = new ErrorMessageType(ErrorDescriptor.NoError);
#region Error Messages
        public List<ProcessMessageDocument> ErrorMessages { get; set; }

        public bool HasErrors
        {
            get
            {
                return ErrorMessages != null && ErrorMessages.Count > 0;
            }
        }

        private string GetCurrentLocation()
        {
            StringBuilder location = new StringBuilder();
            string[] paths = _currentXPath.ToArray<string>();
            for (int i = paths.Length - 1; i > -1; i--)
            {
                location.Append(paths[i]);
            }

            return location.ToString();
        }

        private void MissingElementError()
        {
            ProcessMessageDocument processMessage;
            processMessage = new ProcessMessageDocument()
            {
                Code = "CMN.ATO.GEN.XML04",
                Description = MissingMandatoryMessage,
                SeverityAsString = "Error",
                Location = GetCurrentLocation(),
            };
            this.ErrorMessages.Add(processMessage);
        }

#endregion Error Messages
#region Embedded Schema
        private static readonly XmlSchemaSet EmbeddedXmlSchemaSet;
        static FITR2026XmlConsumer()
        {
            Assembly executingAssembly = Assembly.GetExecutingAssembly();
            string[] manifestResourceNames = executingAssembly.GetManifestResourceNames();
            string resourceName;
            Stream embeddedSchemaStream;
            XmlSchema embeddedXmlSchema;
            if (manifestResourceNames != null)
            {
                resourceName = manifestResourceNames.FirstOrDefault(rn => rn.Contains("ato.fitr.0013.2026.01.00.xsd") && rn.EndsWith(".xsd"));
                if (resourceName != null)
                {
                    embeddedSchemaStream = executingAssembly.GetManifestResourceStream(resourceName);
                    embeddedXmlSchema = XmlSchema.Read(embeddedSchemaStream, SchemaCallback);
                    EmbeddedXmlSchemaSet = new XmlSchemaSet();
                    EmbeddedXmlSchemaSet.Add(embeddedXmlSchema);
                    EmbeddedXmlSchemaSet.Compile();
                }
            }
        }

        private static void SchemaCallback(object sender, ValidationEventArgs args)
        {
            if (args.Severity == XmlSeverityType.Error)
                throw new XmlSchemaException(args.Message);
        }

#endregion Embedded Schema
#region Xml Reader Settings
        private static XmlReaderSettings ReaderSettings = new XmlReaderSettings()
        {
            CloseInput = false,
            ConformanceLevel = ConformanceLevel.Fragment,
            IgnoreWhitespace = true,
            IgnoreComments = true,
            IgnoreProcessingInstructions = true,
        };
        private XmlReaderSettings GetValidatingReaderSettings()
        {
            XmlReaderSettings validatingReaderSettings = new XmlReaderSettings()
            {
                CloseInput = false,
                ConformanceLevel = ConformanceLevel.Fragment,
                IgnoreWhitespace = true,
                IgnoreComments = true,
                IgnoreProcessingInstructions = true,
                ValidationType = ValidationType.Schema,
            };
            if (EmbeddedXmlSchemaSet == null)
            {
                throw new XmlSchemaException("Embedded Schema is Null");
            }
            else
            {
                validatingReaderSettings.Schemas.Add(EmbeddedXmlSchemaSet);
                validatingReaderSettings.ValidationEventHandler += new ValidationEventHandler(ValidationCallBack);
            }

            return validatingReaderSettings;
        }

        private static Regex DataTypeFailureExpression = new Regex("'(?<Uniqueid>.*?)' element is invalid.*value '(?<Value>.*?)' is invalid.*datatype '(?<DataType>.*?)'", RegexOptions.IgnoreCase | RegexOptions.Compiled | RegexOptions.Singleline);
        private void ValidationCallBack(object sender, ValidationEventArgs args)
        {
            var msg = args.Message;
            var msgToLower = msg.ToLower();
            ProcessMessageDocument processMessage = null;
            if (msgToLower.Contains("invalid according to its datatype"))
            {
                Match match = DataTypeFailureExpression.Match(args.Message);
                if (match.Success && match.Groups.Count == 4)
                {
                    var inner = args.Exception.InnerException;
                    var hint = (inner != null && !string.IsNullOrEmpty(inner.Message)) ? " Hint: " + inner.Message : string.Empty;
                    processMessage = BuildProcessMessageDocument(InvalidDataValidation, string.Empty, ErrorCode3);
                    string uniqueID = match.Groups["Uniqueid"].Value;
                    string value = match.Groups["Value"].Value;
                    string dataType = match.Groups["DataType"].Value;
                    var longDescription = string.Format(@"The value specified for an item does not match the item type (value = ""{0}"", item type = {1}, uniqueID = {2})", value, dataType, uniqueID);
                    processMessage.LongDescription = longDescription + hint;
                    processMessage.Parameters = new ProcessMessageParameters();
                    processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "uniqueID", Value = uniqueID });
                    processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "value", Value = value });
                    processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "dataType", Value = dataType });
                }
            }
            else if (msgToLower.Contains("has incomplete content"))
            {
                _validationError = GetErrorDescriptorForHasIncompleteContent(msg);
                processMessage = BuildProcessMessageDocument(MissingMandatoryMessage, msg + _validationError.Hint, _validationError.Code);
                _isValidationError = true;
            }
            else if (msgToLower.Contains("has invalid child element"))
            {
                _validationError = GetErrorDescriptorForHasInvalidChildElement(msg);
                var shortMessage = _validationError.Code == ErrorCode4 ? MissingMandatoryMessage : ContactYourProviderMsg;
                processMessage = BuildProcessMessageDocument(shortMessage, msg + _validationError.Hint, _validationError.Code);
                _isValidationError = true;
            }
            else
            {
                processMessage = BuildProcessMessageDocument(ContactYourProviderMsg, msg, ErrorCode1);
                _isValidationError = true;
            }

            this.ErrorMessages.Add(processMessage);
        }

        private ProcessMessageDocument BuildProcessMessageDocument(string shortDescription, string longDescription, string code)
        {
            return new ProcessMessageDocument()
            {
                Code = code,
                Description = shortDescription,
                LongDescription = longDescription,
                SeverityAsString = "Error",
                Location = GetCurrentLocation()
            };
        }

        private ProcessMessageDocument BuildProcessMessageDocument(ProcessMessageDocument processMessage, string newLocation)
        {
            return new ProcessMessageDocument()
            {
                Code = processMessage.Code,
                Description = processMessage.Description,
                LongDescription = processMessage.LongDescription,
                SeverityAsString = "Error",
                Location = newLocation
            };
        }

        private enum ErrorDescriptor
        {
            NoError = 0,
            MandatoryElementError,
            NonMandatoryElementError,
            UnknownElement,
            Duplicate
        }

        private struct ErrorMessageType
        {
            public string Hint, Code, ExpectedNextValidElement;
            public ErrorDescriptor Errno;
            public ErrorMessageType(ErrorDescriptor errorDescriptor)
            {
                Errno = errorDescriptor;
                Hint = "";
                Code = "";
                ExpectedNextValidElement = "";
            }

            public ErrorMessageType(ErrorDescriptor errorDescriptor, string hint, string code, string elementName)
            {
                Errno = errorDescriptor;
                Hint = hint;
                Code = code;
                ExpectedNextValidElement = elementName;
            }
        }

        private ErrorMessageType GetErrorDescriptorForHasIncompleteContent(string validationErrorMessage)
        {
            ErrorMessageType retval = new ErrorMessageType();
            if (string.IsNullOrWhiteSpace(validationErrorMessage))
                return retval;
            validationErrorMessage = validationErrorMessage.Replace(" ", string.Empty);
            validationErrorMessage = validationErrorMessage.Replace("'", " ");
            var parts = validationErrorMessage.Split(' ');
            if (parts.Length != 9)
                return retval;
            var elementBeingProcessed = parts[1];
            var csvList = parts[5];
            retval = ParseHasIncompleteContentCVSList(csvList, elementBeingProcessed);
            return retval;
        }

        private ErrorMessageType GetErrorDescriptorForHasInvalidChildElement(string validationErrorMessage)
        {
            string msg = string.Empty;
            ErrorMessageType retval = new ErrorMessageType();
            if (string.IsNullOrWhiteSpace(validationErrorMessage))
                return retval;
            validationErrorMessage = validationErrorMessage.Replace(" ", string.Empty);
            validationErrorMessage = validationErrorMessage.Replace("'", " ");
            var parts = validationErrorMessage.Split(' ');
            //decide which message format to process
            if (validationErrorMessage.ToLower().Contains("listofpossibleelements"))
            {
                if (parts.Length == 13 || parts.Length == 12 || parts.Length == 10)
                {
                    var csvList = parts[9];
                    var elementBeingProcessed = parts[1];
                    var invalidChildElementName = parts[5];
                    retval = ParseCvsListForHasInvalidChildElementLongFormat(csvList, invalidChildElementName, elementBeingProcessed);
                }

                return retval;
            }
            else
            {
                if (parts.Length != 9)
                    return retval;
                var csvList = parts[5];
                var elementBeingProcessed = parts[1];
                var invalidChildElementName = parts[5];
                retval = ParseCvsListForHasInvalidChildElementLongFormat(csvList, invalidChildElementName, elementBeingProcessed);
            }

            return retval;
        }

        private ErrorMessageType ParseCvsListForHasInvalidChildElementShortFormat(string csvList, string invalidChildElementName, string elementBeingProcessed)
        {
            ErrorMessageType retval = new ErrorMessageType();
            if (!string.IsNullOrEmpty(csvList))
            {
                var lastElementInListOfPossibleElements = GetLastElementInCsvList(csvList);
                retval.Hint = string.Format(" Hint: while processing parent element [{0}] child element [{1}] was unexpected", elementBeingProcessed, invalidChildElementName);
                retval.Code = ErrorCode1;
                retval.Errno = ErrorDescriptor.UnknownElement;
                retval.ExpectedNextValidElement = invalidChildElementName;
            }

            return retval;
        }

        private string GetLastElementInCsvList(string csvList)
        {
            var list = (csvList.Contains(',')) ? new List<string>(csvList.Split(',')) : new List<string>
            {
                csvList
            };
            string lastElement = list.Last();
            if (lastElement.EndsWith("...."))
            {
                list.RemoveAt(list.Count - 1);
            }

            return list.Last();
        }

        private ErrorMessageType ParseCvsListForHasInvalidChildElementLongFormat(string csvList, string invalidChildElementName, string elementBeingProcessed)
        {
            ErrorMessageType retval = new ErrorMessageType();
            if (!string.IsNullOrEmpty(csvList))
            {
                var lastElementInListOfPossibleElements = GetLastElementInCsvList(csvList);
                var elementList = new List<string>(_elementCsvList.Split(','));
                if (ContainsElementForElementBeingProcessed(elementList, elementBeingProcessed, invalidChildElementName))
                {
                    var indexOfElementBeingProcessed = GetIndexOfElementBeingProcessed(elementList, elementBeingProcessed);
                    var indexOfinvalidChildElementName = GetIndexOfElementUnderElementBeingProcessed(elementList, elementBeingProcessed, invalidChildElementName);
                    var indexOflastElementInListOfPossibleElements = GetIndexOfElementUnderElementBeingProcessed(elementList, elementBeingProcessed, lastElementInListOfPossibleElements);
                    if (indexOfinvalidChildElementName > indexOflastElementInListOfPossibleElements)
                    {
                        retval.Hint = string.Format(" Hint: while processing parent element [{0}] child mandatory element [{1}] was expected but not found", elementBeingProcessed, lastElementInListOfPossibleElements);
                        retval.Code = ErrorCode4;
                        retval.Errno = ErrorDescriptor.MandatoryElementError;
                        retval.ExpectedNextValidElement = lastElementInListOfPossibleElements;
                    }
                    else
                    {
                        retval.Hint = string.Format(" Hint: while processing parent element [{0}] child non mandatory element [{1}] was unexpected", elementBeingProcessed, invalidChildElementName);
                        retval.Code = ErrorCode1;
                        retval.Errno = (indexOfinvalidChildElementName != indexOflastElementInListOfPossibleElements) ? ErrorDescriptor.NonMandatoryElementError : ErrorDescriptor.Duplicate;
                        retval.ExpectedNextValidElement = invalidChildElementName;
                    }
                }
                else
                {
                    //this element is unknown to the schema.
                    retval.Hint = string.Format(" Hint: while processing parent element [{0}] child element [{1}] was unexpected", elementBeingProcessed, invalidChildElementName);
                    retval.Code = ErrorCode1;
                    retval.Errno = ErrorDescriptor.UnknownElement;
                    retval.ExpectedNextValidElement = invalidChildElementName;
                }
            }

            return retval;
        }

        private int GetIndexOfElementUnderElementBeingProcessed(List<string> validElementList, string elementBeingProcessed, string elementName)
        {
            int result;
            string elementWithParentNameMatch = elementBeingProcessed + "#" + elementName;
            result = validElementList.IndexOf(elementWithParentNameMatch);
            if (result < 0)
            {
                result = validElementList.IndexOf(elementName);
            }

            return result;
        }

        private int GetIndexOfElementBeingProcessed(List<string> validElementList, string elementBeingProcessed)
        {
            return validElementList.IndexOf(elementBeingProcessed);
        }

        private bool ContainsElementForElementBeingProcessed(List<string> validElementList, string elementBeingProcessed, string elementName)
        {
            return validElementList.Contains(elementBeingProcessed + "#" + elementName) || validElementList.Contains(elementName);
        }

        private ErrorMessageType ParseHasIncompleteContentCVSList(string csvList, string elementBeingProcessed)
        {
            ErrorMessageType retval = new ErrorMessageType();
            if (!string.IsNullOrEmpty(csvList))
            {
                var lastElementInListOfPossibleElements = GetLastElementInCsvList(csvList);
                retval.Hint = string.Format(" Hint: while processing parent element [{0}] child mandatory element [{1}] was not found", elementBeingProcessed, lastElementInListOfPossibleElements);
                retval.Code = ErrorCode4;
                retval.Errno = ErrorDescriptor.MandatoryElementError;
                retval.ExpectedNextValidElement = lastElementInListOfPossibleElements;
            }

            return retval;
        }

        private void StartEndDateError(DateTime startDate, DateTime endDate)
        {
            ProcessMessageDocument processMessage;
            string longDescription = string.Format("End date is earlier than start date (state date = {0}, end date = {1})", startDate.ToString("yyyy-MM-dd"), endDate.ToString("yyyy-MM-dd"));
            processMessage = new ProcessMessageDocument()
            {
                Code = "CMN.ATO.GEN.XML06",
                Description = "End date is earlier than start date.",
                LongDescription = longDescription,
                SeverityAsString = "Error",
                Location = GetCurrentLocation(),
            };
            processMessage.Parameters = new ProcessMessageParameters();
            processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "startDate", Value = startDate.ToString("yyyy-MM-dd") });
            processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "endDate", Value = endDate.ToString("yyyy-MM-dd") });
            this.ErrorMessages.Add(processMessage);
        }

        private void StartEndDateError(DateTime? startDate, DateTime? endDate)
        {
            ProcessMessageDocument processMessage;
            if (startDate == null || endDate == null)
                return;
            string longDescription = string.Format("End date is earlier than start date (state date = {0}, end date = {1})", startDate.Value.ToString("yyyy-MM-dd"), endDate.Value.ToString("yyyy-MM-dd"));
            processMessage = new ProcessMessageDocument()
            {
                Code = "CMN.ATO.GEN.XML06",
                Description = "End date is earlier than start date.",
                LongDescription = longDescription,
                SeverityAsString = "Error",
                Location = GetCurrentLocation(),
            };
            processMessage.Parameters = new ProcessMessageParameters();
            processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "startDate", Value = startDate.Value.ToString("yyyy-MM-dd") });
            processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "endDate", Value = endDate.Value.ToString("yyyy-MM-dd") });
            this.ErrorMessages.Add(processMessage);
        }

#endregion Xml Reader Settings
#region IsEmptyOrNilElement
        private static bool IsEmptyOrNilElement(XmlReader reader)
        {
            bool emptyOrNil = false;
            if (reader.IsEmptyElement)
            {
                emptyOrNil = true;
            }
            else
            {
                if (reader.HasAttributes)
                {
                    string nilValue = reader.GetAttribute("nil", "http://www.w3.org/2001/XMLSchema-instance");
                    if (nilValue != null && nilValue.ToLowerInvariant() == "true" || nilValue == "1")
                    {
                        emptyOrNil = true;
                    }
                }
            }

            return emptyOrNil;
        }

#endregion IsEmptyOrNilElement
#region ToBoolean
        private bool? ToBoolean(string str)
        {
            bool returnValue;
            string value = (str ?? "").Trim();
            if (value == "0" || value == "1")
                return Convert.ToBoolean(Convert.ToInt32(value));
            if (Boolean.TryParse(value, out returnValue))
                return returnValue;
            return null;
        }

#endregion ToBoolean
#region MoveToContent
        public bool MoveToContent(XmlReader reader)
        {
            try
            {
                reader.MoveToContent();
                return true;
            }
            catch (XmlException ex)
            {
                ProcessMessageDocument processMessage;
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.XML01",
                    Description = ContactYourProviderMsg,
                    LongDescription = ex.Message,
                    SeverityAsString = "Error",
                };
                this.ErrorMessages.Add(processMessage);
                return false;
            }
        }

#endregion MoveToContent
#region ReadToNextElement
        public bool ReadToNextElement(XmlReader reader, bool isMandatory)
        {
            return ReadToNextElement(reader);
        }

        public bool ReadToNextElement(XmlReader reader)
        {
            bool retval = false;
            if ((reader.EOF && !_isValidationError) || _isExiting)
            {
                return false; //do nothing
            }

            if (!_isValidationError)
            {
                retval = SetReaderToNextElement(reader); //position the reader on the next valid element
            }

            if (_isValidationError)
            {
                //process validation errors
                var expectedNextElement = (!string.IsNullOrEmpty(_validationError.ExpectedNextValidElement)) ? _validationError.ExpectedNextValidElement : reader.LocalName;
                var actualNextValidElement = reader.LocalName;
                var location = GetCurrentLocation();
                if (_validationError.Errno == ErrorDescriptor.Duplicate)
                {
                    location = _lastPath;
                }

                var xpath = RemoveLeadingAndTrailingSlashes(location);
                var element = GetElementDescriptor(xpath);
                if (_validationError.Errno == ErrorDescriptor.UnknownElement)
                {
                    //set the xpath for elements that are not known to the schema
                    location = "/" + element.Path + "/" + element.Prefix + ":" + _validationError.ExpectedNextValidElement;
                    _isExiting = true;
                }
                else if (_validationError.Errno == ErrorDescriptor.NonMandatoryElementError)
                {
                    //set the xpath for the optional element that caused a validation
                    location = "/" + element.Path + "/" + element.Prefix + ":" + reader.LocalName;
                    _isExiting = true;
                }
                else
                {
                    //decide if we have spooled to the missing mandatory element. 
                    _isExiting = expectedNextElement == element.PathLeafNode;
                }

                if (_isExiting)
                {
                    //The location has changed since detection of the validation error - so update it.
                    UpdateErrorMessageWithLocation(location);
                }

                return false;
            }

            //Save previous path so we can detect duplicate elements
            _lastPath = GetCurrentLocation();
            return retval;
        }

        private bool ReadNext(XmlReader reader)
        {
            return !reader.EOF && reader.Read();
        }

        private void UpdateErrorMessageWithLocation(string newLocation)
        {
            var lastProcessMessage = ErrorMessages.Last();
            ErrorMessages.Remove(lastProcessMessage);
            var newProcessMessage = BuildProcessMessageDocument(lastProcessMessage, newLocation);
            ErrorMessages.Add(newProcessMessage);
        }

        private string RemoveLeadingAndTrailingSlashes(string xpath)
        {
            var xpathLength = xpath.Length;
            if (xpathLength <= 0)
                return string.Empty;
            var firstChar = xpath.Substring(0, 1);
            var lastChar = xpath.Substring(xpathLength - 1, 1);
            if (firstChar == "/")
            {
                xpath = xpath.Remove(0, 1);
                xpathLength--;
            }

            if (lastChar == "/")
            {
                xpath = xpath.Remove(xpathLength - 1, 1);
            }

            return xpath;
        }

        private struct ElementDescriptor
        {
            public string Prefix, PathLeafNode, Path;
            public ElementDescriptor(string prefix, string leaf, string path)
            {
                Prefix = prefix;
                PathLeafNode = leaf;
                Path = path;
            }
        }

        private ElementDescriptor GetElementDescriptor(string xpath)
        {
            var retval = new ElementDescriptor();
            var parts = xpath.Split('/');
            var numberOfParts = parts.Length;
            if (numberOfParts > 0)
            {
                var leafNode = parts[numberOfParts - 1];
                var p = leafNode.Split(':');
                numberOfParts = p.Length;
                if (numberOfParts == 1)
                {
                    retval.PathLeafNode = p[0];
                    retval.Prefix = string.Empty;
                }
                else if (numberOfParts == 2)
                {
                    retval.PathLeafNode = p[1];
                    retval.Prefix = p[0];
                }

                var leafNodeLength = leafNode.Length;
                var path = xpath.Remove(xpath.Length - leafNodeLength, leafNodeLength);
                retval.Path = RemoveLeadingAndTrailingSlashes(path);
            }

            return retval;
        }

        private bool SetReaderToNextElement(XmlReader reader)
        {
            try
            {
                if (reader.EOF || _isExiting)
                    return false;
                if (_found)
                {
                    reader.Read();
                    _found = false;
                }

                while (!reader.EOF && reader.NodeType != XmlNodeType.Element)
                {
                    reader.Read();
                }

                return reader.NodeType == XmlNodeType.Element && !reader.EOF;
            }
            catch (XmlException ex)
            {
                ProcessMessageDocument processMessage;
                processMessage = new ProcessMessageDocument()
                {
                    Code = ErrorCode1,
                    Description = ContactYourProviderMsg,
                    LongDescription = ex.Message,
                    SeverityAsString = "Error",
                };
                this.ErrorMessages.Add(processMessage);
                return false;
            }
        }

#endregion ReadToNextElement
        private Stack<string> _currentXPath = new Stack<string>();
        public FITR2026 Consume(Stream streamToLoad, bool validateDataTypes = false)
        {
            FITR2026 report = new FITR2026();
            // Working Variable for if can still read from the xml stream
            bool reading;
            // Working Variables for current values
            string currentValue;
            DateTime currentDateTimeValue;
            bool? currentBooleanValue;
            decimal currentDecimalValue;
            int currentIntValue;
            this.ErrorMessages = new List<ProcessMessageDocument>();
            _currentXPath.Push("/tns:FITR");
            if (streamToLoad == null || streamToLoad.Length == 0 || !streamToLoad.CanRead)
            {
                MissingElementError();
                return report;
            }

            streamToLoad.Position = 0;
            XmlReader reader;
            if (validateDataTypes)
                reader = XmlReader.Create(streamToLoad, GetValidatingReaderSettings());
            else
                reader = XmlReader.Create(streamToLoad, ReaderSettings);
            if (!MoveToContent(reader))
                return report;
            reading = !reader.EOF;
            if (ReadToNextElement(reader) && reader.LocalName == "FITR" && reader.NamespaceURI == "http://www.sbr.gov.au/ato/fitr")
            {
                _found = true;
                ReadToNextElement(reader);
            }
            else
            {
                MissingElementError();
                return report;
            }

#region RP
            _currentXPath.Push("/tns:RP");
            //3. use case
            if (ReadToNextElement(reader, true) && reader.LocalName == "RP" && reader.Depth == _currentXPath.Count - 1)
            {
                report.RpCollectionExists = true;
                report.RpCollectionCount += 1;
                _found = true;
#region TargetFinancialY
                _currentXPath.Push("/tns:TargetFinancialY");
                //6. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "TargetFinancialY" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Int32.TryParse(currentValue, out currentIntValue))
                            {
                                report.FITR1 = currentIntValue;
                            }
                        }
                    }

                    _found = true;
                }

                _currentXPath.Pop();
#endregion TargetFinancialY
#region PeriodStartD
                _currentXPath.Push("/tns:PeriodStartD");
                //6. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "PeriodStartD" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                            {
                                report.FITR300 = currentDateTimeValue.Date;
                            }
                        }
                    }

                    _found = true;
                }

                _currentXPath.Pop();
#endregion PeriodStartD
#region PeriodEndD
                _currentXPath.Push("/tns:PeriodEndD");
                //6. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "PeriodEndD" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                            {
                                report.FITR301 = currentDateTimeValue.Date;
                            }
                        }
                    }

                    _found = true;
                }

                _currentXPath.Pop();
#endregion PeriodEndD
#region TaxFileNumberId
                _currentXPath.Push("/tns:TaxFileNumberId");
                //6. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "TaxFileNumberId" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.FITR5 = currentValue;
                        }
                    }

                    _found = true;
                }

                _currentXPath.Pop();
#endregion TaxFileNumberId
#region OrganisationNameDetailsOrganisationalNameT
                _currentXPath.Push("/tns:OrganisationNameDetailsOrganisationalNameT");
                //6. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "OrganisationNameDetailsOrganisationalNameT" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.FITR6 = currentValue;
                        }
                    }

                    _found = true;
                }

                _currentXPath.Pop();
#endregion OrganisationNameDetailsOrganisationalNameT
#region AustralianBusinessNumberId
                _currentXPath.Push("/tns:AustralianBusinessNumberId");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "AustralianBusinessNumberId" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.FITR7 = currentValue;
                        }
                    }

                    _found = true;
                }

                _currentXPath.Pop();
#endregion AustralianBusinessNumberId
#region AmendmentDetails
                _currentXPath.Push("/tns:AmendmentDetails");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "AmendmentDetails" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_AmendmentDetailsCollectionExists = true;
                    report.RP_AmendmentDetailsCollectionCount += 1;
                    _found = true;
#region I
                    _currentXPath.Push("/tns:I");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "I" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.FITR171 = currentBooleanValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion I
#region AmendmentSequenceN
                    _currentXPath.Push("/tns:AmendmentSequenceN");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "AmendmentSequenceN" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR174 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion AmendmentSequenceN
#region AmendmentReasonT
                    _currentXPath.Push("/tns:AmendmentReasonT");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "AmendmentReasonT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FITR173 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion AmendmentReasonT
#region AmendmentTypeC
                    _currentXPath.Push("/tns:AmendmentTypeC");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "AmendmentTypeC" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FITR172 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion AmendmentTypeC
                } // End of if AmendmentDetails node exists

                _currentXPath.Pop();
#endregion AmendmentDetails
#region CurrentPostalAddress
                _currentXPath.Push("/tns:CurrentPostalAddress");
                //3. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "CurrentPostalAddress" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_CurrentPostalAddressCollectionExists = true;
                    report.RP_CurrentPostalAddressCollectionCount += 1;
                    _found = true;
#region AddressDetailsLine1T
                    _currentXPath.Push("/tns:AddressDetailsLine1T");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "AddressDetailsLine1T" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FITR8 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion AddressDetailsLine1T
#region AddressDetailsLine2T
                    _currentXPath.Push("/tns:AddressDetailsLine2T");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "AddressDetailsLine2T" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FITR146 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion AddressDetailsLine2T
#region AddressDetailsLocalityNameT
                    _currentXPath.Push("/tns:AddressDetailsLocalityNameT");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "AddressDetailsLocalityNameT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FITR9 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion AddressDetailsLocalityNameT
#region AddressDetailsPostcodeT
                    _currentXPath.Push("/tns:AddressDetailsPostcodeT");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "AddressDetailsPostcodeT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FITR11 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion AddressDetailsPostcodeT
#region AddressDetailsStateOrTerritoryC
                    _currentXPath.Push("/tns:AddressDetailsStateOrTerritoryC");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "AddressDetailsStateOrTerritoryC" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FITR10 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion AddressDetailsStateOrTerritoryC
#region AddressDetailsCountryC
                    _currentXPath.Push("/tns:AddressDetailsCountryC");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "AddressDetailsCountryC" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FITR150 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion AddressDetailsCountryC
                } // End of if CurrentPostalAddress node exists

                _currentXPath.Pop();
#endregion CurrentPostalAddress
#region TrusteeDetails
                _currentXPath.Push("/tns:TrusteeDetails");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "TrusteeDetails" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_TrusteeDetailsCollectionExists = true;
                    report.RP_TrusteeDetailsCollectionCount += 1;
                    _found = true;
#region OrganisationNameDetailsOrganisationalNameT
                    _currentXPath.Push("/tns:OrganisationNameDetailsOrganisationalNameT");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "OrganisationNameDetailsOrganisationalNameT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FITR12 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion OrganisationNameDetailsOrganisationalNameT
#region AustralianBusinessNumberId
                    _currentXPath.Push("/tns:AustralianBusinessNumberId");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "AustralianBusinessNumberId" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FITR13 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion AustralianBusinessNumberId
                } // End of if TrusteeDetails node exists

                _currentXPath.Pop();
#endregion TrusteeDetails
#region ElectronicFundTransfer
                _currentXPath.Push("/tns:ElectronicFundTransfer");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "ElectronicFundTransfer" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_ElectronicFundTransferCollectionExists = true;
                    report.RP_ElectronicFundTransferCollectionCount += 1;
                    _found = true;
#region FinancialInstitutionAccountBankStateBranchN
                    _currentXPath.Push("/tns:FinancialInstitutionAccountBankStateBranchN");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "FinancialInstitutionAccountBankStateBranchN" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FITR176 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion FinancialInstitutionAccountBankStateBranchN
#region FinancialInstitutionAccountFinancialInstitutionAccountN
                    _currentXPath.Push("/tns:FinancialInstitutionAccountFinancialInstitutionAccountN");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "FinancialInstitutionAccountFinancialInstitutionAccountN" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FITR177 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion FinancialInstitutionAccountFinancialInstitutionAccountN
#region FinancialInstitutionAccountFinancialInstitutionAccountNameT
                    _currentXPath.Push("/tns:FinancialInstitutionAccountFinancialInstitutionAccountNameT");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "FinancialInstitutionAccountFinancialInstitutionAccountNameT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FITR178 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion FinancialInstitutionAccountFinancialInstitutionAccountNameT
                } // End of if ElectronicFundTransfer node exists

                _currentXPath.Pop();
#endregion ElectronicFundTransfer
#region StatusOfFundOrTrust
                _currentXPath.Push("/tns:StatusOfFundOrTrust");
                //3. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "StatusOfFundOrTrust" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_StatusOfFundOrTrustCollectionExists = true;
                    report.RP_StatusOfFundOrTrustCollectionCount += 1;
                    _found = true;
#region PartyTypeC
                    _currentXPath.Push("/tns:PartyTypeC");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "PartyTypeC" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FITR14 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion PartyTypeC
#region SuperannuationFundDetailsAustralianSuperannuationFundStatusI
                    _currentXPath.Push("/tns:SuperannuationFundDetailsAustralianSuperannuationFundStatusI");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "SuperannuationFundDetailsAustralianSuperannuationFundStatusI" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.FITR15 = currentBooleanValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion SuperannuationFundDetailsAustralianSuperannuationFundStatusI
#region SuperannuationFundDetailsFundBenefitStructureC
                    _currentXPath.Push("/tns:SuperannuationFundDetailsFundBenefitStructureC");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "SuperannuationFundDetailsFundBenefitStructureC" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FITR16 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion SuperannuationFundDetailsFundBenefitStructureC
#region OrganisationControlRegisteredMemberCt
                    _currentXPath.Push("/tns:OrganisationControlRegisteredMemberCt");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "OrganisationControlRegisteredMemberCt" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR208 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion OrganisationControlRegisteredMemberCt
#region OrganisationDetailsEstablishmentD
                    _currentXPath.Push("/tns:OrganisationDetailsEstablishmentD");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "OrganisationDetailsEstablishmentD" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                                {
                                    report.FITR18 = currentDateTimeValue.Date;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion OrganisationDetailsEstablishmentD
#region OrganisationDetailsSignificantGlobalEntityStatusI
                    _currentXPath.Push("/tns:OrganisationDetailsSignificantGlobalEntityStatusI");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "OrganisationDetailsSignificantGlobalEntityStatusI" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.FITR212 = currentBooleanValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion OrganisationDetailsSignificantGlobalEntityStatusI
#region InternationalDealingsCountryByCountryReportingEntityStatusI
                    _currentXPath.Push("/tns:InternationalDealingsCountryByCountryReportingEntityStatusI");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsCountryByCountryReportingEntityStatusI" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.FITR219 = currentBooleanValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion InternationalDealingsCountryByCountryReportingEntityStatusI
#region OrganisationDetailsOperationCeaseI
                    _currentXPath.Push("/tns:OrganisationDetailsOperationCeaseI");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "OrganisationDetailsOperationCeaseI" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.FITR19 = currentBooleanValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion OrganisationDetailsOperationCeaseI
#region OrganisationDetailsOrganisationActivityStatusEndD
                    _currentXPath.Push("/tns:OrganisationDetailsOrganisationActivityStatusEndD");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "OrganisationDetailsOrganisationActivityStatusEndD" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                                {
                                    report.FITR20 = currentDateTimeValue.Date;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion OrganisationDetailsOrganisationActivityStatusEndD
                } // End of if StatusOfFundOrTrust node exists

                _currentXPath.Pop();
#endregion StatusOfFundOrTrust
#region Income
                _currentXPath.Push("/tns:Income");
                //3. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "Income" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_IncomeCollectionExists = true;
                    report.RP_IncomeCollectionCount += 1;
                    _found = true;
#region CapitalGainsTaxEventI
                    _currentXPath.Push("/tns:CapitalGainsTaxEventI");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "CapitalGainsTaxEventI" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.FITR21 = currentBooleanValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion CapitalGainsTaxEventI
#region CapitalGainsTaxExemptionOrRolloverAppliedI
                    _currentXPath.Push("/tns:CapitalGainsTaxExemptionOrRolloverAppliedI");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "CapitalGainsTaxExemptionOrRolloverAppliedI" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.FITR203 = currentBooleanValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion CapitalGainsTaxExemptionOrRolloverAppliedI
#region CapitalGainsTaxExemptionOrRolloverAppliedCCollection
                    _currentXPath.Push("/tns:CapitalGainsTaxExemptionOrRolloverAppliedCCollection");
                    if (ReadToNextElement(reader, false) && reader.LocalName == "CapitalGainsTaxExemptionOrRolloverAppliedCCollection" && reader.Depth == _currentXPath.Count - 1)
                    {
                        report.FITR204Collection = new List<string>();
                        ReadNext(reader);
                        _currentXPath.Push("/tns:CapitalGainsTaxExemptionOrRolloverAppliedC");
                        // 5. use case
                        while (ReadToNextElement(reader, false) && reader.LocalName == "CapitalGainsTaxExemptionOrRolloverAppliedC" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before the currentXPath is updated.
                                    report.FITR204Collection.Add(currentValue);
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
                    }

                    _currentXPath.Pop();
#endregion CapitalGainsTaxExemptionOrRolloverAppliedCCollection
#region CapitalGainsNetA
                    _currentXPath.Push("/tns:CapitalGainsNetA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "CapitalGainsNetA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR23 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion CapitalGainsNetA
#region RentLeasingHiringGrossA
                    _currentXPath.Push("/tns:RentLeasingHiringGrossA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "RentLeasingHiringGrossA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR24 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RentLeasingHiringGrossA
#region InterestGrossA
                    _currentXPath.Push("/tns:InterestGrossA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "InterestGrossA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR25 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion InterestGrossA
#region ForestryManagedInvestmentSchemeA
                    _currentXPath.Push("/tns:ForestryManagedInvestmentSchemeA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "ForestryManagedInvestmentSchemeA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR26 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion ForestryManagedInvestmentSchemeA
#region GrossA
                    _currentXPath.Push("/tns:GrossA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "GrossA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR27 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion GrossA
#region NetA
                    _currentXPath.Push("/tns:NetA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "NetA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR28 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion NetA
#region TaxFrankingCreditsReceivedFromNewZealandCompaniesA
                    _currentXPath.Push("/tns:TaxFrankingCreditsReceivedFromNewZealandCompaniesA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "TaxFrankingCreditsReceivedFromNewZealandCompaniesA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR29 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion TaxFrankingCreditsReceivedFromNewZealandCompaniesA
#region EquityInternationalDealingsForeignFundTransfersA
                    _currentXPath.Push("/tns:EquityInternationalDealingsForeignFundTransfersA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "EquityInternationalDealingsForeignFundTransfersA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR30 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion EquityInternationalDealingsForeignFundTransfersA
#region InternationalDealingsForeignFundTransfersCt
                    _currentXPath.Push("/tns:InternationalDealingsForeignFundTransfersCt");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsForeignFundTransfersCt" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR31 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion InternationalDealingsForeignFundTransfersCt
#region RemunerationABNNotQuotedPaymentGrossA
                    _currentXPath.Push("/tns:RemunerationABNNotQuotedPaymentGrossA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationABNNotQuotedPaymentGrossA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR32 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RemunerationABNNotQuotedPaymentGrossA
#region PartnershipDistributionGrossA
                    _currentXPath.Push("/tns:PartnershipDistributionGrossA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "PartnershipDistributionGrossA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR33 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion PartnershipDistributionGrossA
#region DividendsUnfrankedA
                    _currentXPath.Push("/tns:DividendsUnfrankedA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "DividendsUnfrankedA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR34 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion DividendsUnfrankedA
#region DividendsFrankedA
                    _currentXPath.Push("/tns:DividendsFrankedA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "DividendsFrankedA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR35 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion DividendsFrankedA
#region TaxFrankingCreditsA
                    _currentXPath.Push("/tns:TaxFrankingCreditsA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "TaxFrankingCreditsA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR36 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion TaxFrankingCreditsA
#region TrustDistributionUnfrankedA
                    _currentXPath.Push("/tns:TrustDistributionUnfrankedA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "TrustDistributionUnfrankedA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR37 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion TrustDistributionUnfrankedA
#region TrustDistributionFrankedA
                    _currentXPath.Push("/tns:TrustDistributionFrankedA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "TrustDistributionFrankedA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR38 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion TrustDistributionFrankedA
#region TaxFrankingCreditsFrankingCreditShareReceivedIndirectlyA
                    _currentXPath.Push("/tns:TaxFrankingCreditsFrankingCreditShareReceivedIndirectlyA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "TaxFrankingCreditsFrankingCreditShareReceivedIndirectlyA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR39 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion TaxFrankingCreditsFrankingCreditShareReceivedIndirectlyA
#region TrustDistributionOtherA
                    _currentXPath.Push("/tns:TrustDistributionOtherA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "TrustDistributionOtherA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR40 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion TrustDistributionOtherA
#region SuperannuationContributionEmployerAssessableA
                    _currentXPath.Push("/tns:SuperannuationContributionEmployerAssessableA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "SuperannuationContributionEmployerAssessableA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR41 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion SuperannuationContributionEmployerAssessableA
#region SuperannuationContributionPersonalAssessableA
                    _currentXPath.Push("/tns:SuperannuationContributionPersonalAssessableA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "SuperannuationContributionPersonalAssessableA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR42 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion SuperannuationContributionPersonalAssessableA
#region SuperannuationContributionEmployerContributionsTaxFileNumberNotQuotedA
                    _currentXPath.Push("/tns:SuperannuationContributionEmployerContributionsTaxFileNumberNotQuotedA");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "SuperannuationContributionEmployerContributionsTaxFileNumberNotQuotedA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR43 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion SuperannuationContributionEmployerContributionsTaxFileNumberNotQuotedA
#region SuperannuationContributionContributionsExcludedByTrusteeA
                    _currentXPath.Push("/tns:SuperannuationContributionContributionsExcludedByTrusteeA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "SuperannuationContributionContributionsExcludedByTrusteeA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR44 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion SuperannuationContributionContributionsExcludedByTrusteeA
#region SuperannuationContributionReducedContributionsA
                    _currentXPath.Push("/tns:SuperannuationContributionReducedContributionsA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "SuperannuationContributionReducedContributionsA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR45 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion SuperannuationContributionReducedContributionsA
#region SuperannuationContributionLiabilityTransferToLifeInsuranceCompanyOrPooledSuperannuationTrustAssessableA
                    _currentXPath.Push("/tns:SuperannuationContributionLiabilityTransferToLifeInsuranceCompanyOrPooledSuperannuationTrustAssessableA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "SuperannuationContributionLiabilityTransferToLifeInsuranceCompanyOrPooledSuperannuationTrustAssessableA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR46 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion SuperannuationContributionLiabilityTransferToLifeInsuranceCompanyOrPooledSuperannuationTrustAssessableA
#region ForeignExchangeGainsA
                    _currentXPath.Push("/tns:ForeignExchangeGainsA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "ForeignExchangeGainsA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR186 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion ForeignExchangeGainsA
#region OtherA
                    _currentXPath.Push("/tns:OtherA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "OtherA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR47 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion OtherA
#region OtherC
                    _currentXPath.Push("/tns:OtherC");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "OtherC" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FITR48 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion OtherC
#region AssessableIncomeDueToFundTaxStatusChangeA
                    _currentXPath.Push("/tns:AssessableIncomeDueToFundTaxStatusChangeA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "AssessableIncomeDueToFundTaxStatusChangeA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR49 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion AssessableIncomeDueToFundTaxStatusChangeA
#region NonArmsLengthOtherNetA
                    _currentXPath.Push("/tns:NonArmsLengthOtherNetA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "NonArmsLengthOtherNetA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR50 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion NonArmsLengthOtherNetA
#region TaxConcessionPensionIncomeExemptDeductionA
                    _currentXPath.Push("/tns:TaxConcessionPensionIncomeExemptDeductionA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "TaxConcessionPensionIncomeExemptDeductionA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR200 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion TaxConcessionPensionIncomeExemptDeductionA
                } // End of if Income node exists

                _currentXPath.Pop();
#endregion Income
#region Deductions
                _currentXPath.Push("/tns:Deductions");
                //3. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "Deductions" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_DeductionsCollectionExists = true;
                    report.RP_DeductionsCollectionCount += 1;
                    _found = true;
#region InterestExpensesAustraliaA
                    _currentXPath.Push("/tns:InterestExpensesAustraliaA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "InterestExpensesAustraliaA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR52 = currentDecimalValue;
                                }

                                report.RP_Deductions_InterestExpensesAustraliaCollectionExists = true;
                                report.RP_Deductions_InterestExpensesAustraliaCollectionCount += 1;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion InterestExpensesAustraliaA
#region InterestExpensesOverseasA
                    _currentXPath.Push("/tns:InterestExpensesOverseasA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "InterestExpensesOverseasA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR53 = currentDecimalValue;
                                }

                                report.RP_Deductions_InterestExpensesOverseasCollectionExists = true;
                                report.RP_Deductions_InterestExpensesOverseasCollectionCount += 1;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion InterestExpensesOverseasA
#region RemunerationWagesandSalariesPaymentsA
                    _currentXPath.Push("/tns:RemunerationWagesandSalariesPaymentsA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationWagesandSalariesPaymentsA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR54 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RemunerationWagesandSalariesPaymentsA
#region ExpenseCapitalWorksDeductionA
                    _currentXPath.Push("/tns:ExpenseCapitalWorksDeductionA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseCapitalWorksDeductionA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR55 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion ExpenseCapitalWorksDeductionA
#region ExpenseDepreciationAllowableDeductionA
                    _currentXPath.Push("/tns:ExpenseDepreciationAllowableDeductionA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseDepreciationAllowableDeductionA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR56 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion ExpenseDepreciationAllowableDeductionA
#region ExpensePremiumsDeathOrDisabilityA
                    _currentXPath.Push("/tns:ExpensePremiumsDeathOrDisabilityA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "ExpensePremiumsDeathOrDisabilityA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR57 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion ExpensePremiumsDeathOrDisabilityA
#region ExpenseInvestmentA
                    _currentXPath.Push("/tns:ExpenseInvestmentA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseInvestmentA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR59 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion ExpenseInvestmentA
#region ExpenseOperatingManagementExpenseA
                    _currentXPath.Push("/tns:ExpenseOperatingManagementExpenseA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseOperatingManagementExpenseA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR190 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion ExpenseOperatingManagementExpenseA
#region ExpenseOperatingAdministrativeExpenseA
                    _currentXPath.Push("/tns:ExpenseOperatingAdministrativeExpenseA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseOperatingAdministrativeExpenseA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR188 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion ExpenseOperatingAdministrativeExpenseA
#region ExpenseForestryManagedInvestmentSchemeDeductionA
                    _currentXPath.Push("/tns:ExpenseForestryManagedInvestmentSchemeDeductionA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseForestryManagedInvestmentSchemeDeductionA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR61 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion ExpenseForestryManagedInvestmentSchemeDeductionA
#region IncomeForeignExchangeLossesA
                    _currentXPath.Push("/tns:IncomeForeignExchangeLossesA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeForeignExchangeLossesA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR187 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion IncomeForeignExchangeLossesA
#region ExpenseDeductibleOtherA
                    _currentXPath.Push("/tns:ExpenseDeductibleOtherA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseDeductibleOtherA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR62 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion ExpenseDeductibleOtherA
#region ExpenseDeductibleOtherC
                    _currentXPath.Push("/tns:ExpenseDeductibleOtherC");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseDeductibleOtherC" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FITR63 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion ExpenseDeductibleOtherC
#region IncomeTaxTaxLossesDeductedA
                    _currentXPath.Push("/tns:IncomeTaxTaxLossesDeductedA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxTaxLossesDeductedA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR64 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion IncomeTaxTaxLossesDeductedA
#region IncomeTaxableA
                    _currentXPath.Push("/tns:IncomeTaxableA");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "IncomeTaxableA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR65 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion IncomeTaxableA
                } // End of if Deductions node exists

                _currentXPath.Pop();
#endregion Deductions
#region IncomeTaxCalculation
                _currentXPath.Push("/tns:IncomeTaxCalculation");
                //3. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "IncomeTaxCalculation" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_IncomeTaxCalculationCollectionExists = true;
                    report.RP_IncomeTaxCalculationCollectionCount += 1;
                    _found = true;
#region TotalTaxOnTaxableIncomeA
                    _currentXPath.Push("/tns:TotalTaxOnTaxableIncomeA");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "TotalTaxOnTaxableIncomeA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR182 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion TotalTaxOnTaxableIncomeA
#region TFNNotQuotedContributionsTaxA
                    _currentXPath.Push("/tns:TFNNotQuotedContributionsTaxA");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "TFNNotQuotedContributionsTaxA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR181 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion TFNNotQuotedContributionsTaxA
#region TaxableGrossA
                    _currentXPath.Push("/tns:TaxableGrossA");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "TaxableGrossA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR66 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion TaxableGrossA
#region InternationalDealingsTaxOffsetA
                    _currentXPath.Push("/tns:InternationalDealingsTaxOffsetA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsTaxOffsetA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR67 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion InternationalDealingsTaxOffsetA
#region TaxOffsetsAndRebatesTotalA
                    _currentXPath.Push("/tns:TaxOffsetsAndRebatesTotalA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "TaxOffsetsAndRebatesTotalA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR68 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion TaxOffsetsAndRebatesTotalA
#region TaxOffsetNonRefundableNonCarryForwardA
                    _currentXPath.Push("/tns:TaxOffsetNonRefundableNonCarryForwardA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "TaxOffsetNonRefundableNonCarryForwardA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR189 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion TaxOffsetNonRefundableNonCarryForwardA
#region SubtotalTaxPayableA
                    _currentXPath.Push("/tns:SubtotalTaxPayableA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "SubtotalTaxPayableA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR183 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion SubtotalTaxPayableA
#region ESVCLPTaxOffsetClaimNonRefundableOtherA
                    _currentXPath.Push("/tns:ESVCLPTaxOffsetClaimNonRefundableOtherA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "ESVCLPTaxOffsetClaimNonRefundableOtherA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR213 = currentDecimalValue;
                                }

                                report.RP_IncomeTaxCalculation_ESVCLPCollectionExists = true;
                                report.RP_IncomeTaxCalculation_ESVCLPCollectionCount += 1;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion ESVCLPTaxOffsetClaimNonRefundableOtherA
#region PreviousYearESVCLPTaxOffsetClaimNonRefundableOtherA
                    _currentXPath.Push("/tns:PreviousYearESVCLPTaxOffsetClaimNonRefundableOtherA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "PreviousYearESVCLPTaxOffsetClaimNonRefundableOtherA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR217 = currentDecimalValue;
                                }

                                report.RP_IncomeTaxCalculation_PreviousYearESVCLPCollectionExists = true;
                                report.RP_IncomeTaxCalculation_PreviousYearESVCLPCollectionCount += 1;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion PreviousYearESVCLPTaxOffsetClaimNonRefundableOtherA
#region EarlyStageInvestorTaxOffsetClaimNonRefundableOtherA
                    _currentXPath.Push("/tns:EarlyStageInvestorTaxOffsetClaimNonRefundableOtherA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "EarlyStageInvestorTaxOffsetClaimNonRefundableOtherA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR214 = currentDecimalValue;
                                }

                                report.RP_IncomeTaxCalculation_EarlyStageInvestorCollectionExists = true;
                                report.RP_IncomeTaxCalculation_EarlyStageInvestorCollectionCount += 1;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion EarlyStageInvestorTaxOffsetClaimNonRefundableOtherA
#region PreviousYearEarlyStageInvestorTaxOffsetClaimNonRefundableOtherA
                    _currentXPath.Push("/tns:PreviousYearEarlyStageInvestorTaxOffsetClaimNonRefundableOtherA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "PreviousYearEarlyStageInvestorTaxOffsetClaimNonRefundableOtherA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR218 = currentDecimalValue;
                                }

                                report.RP_IncomeTaxCalculation_PreviousYearEarlyStageInvestorCollectionExists = true;
                                report.RP_IncomeTaxCalculation_PreviousYearEarlyStageInvestorCollectionCount += 1;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion PreviousYearEarlyStageInvestorTaxOffsetClaimNonRefundableOtherA
#region TaxOffsetNonRefundableCarryForwardA
                    _currentXPath.Push("/tns:TaxOffsetNonRefundableCarryForwardA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "TaxOffsetNonRefundableCarryForwardA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR215 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion TaxOffsetNonRefundableCarryForwardA
#region TaxOffsetCarriedForwardSubTotalA
                    _currentXPath.Push("/tns:TaxOffsetCarriedForwardSubTotalA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "TaxOffsetCarriedForwardSubTotalA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR216 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion TaxOffsetCarriedForwardSubTotalA
#region FrankingCreditsRefundableA
                    _currentXPath.Push("/tns:FrankingCreditsRefundableA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "FrankingCreditsRefundableA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR73 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion FrankingCreditsRefundableA
#region TaxOffsetTFNNotQuotedA
                    _currentXPath.Push("/tns:TaxOffsetTFNNotQuotedA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "TaxOffsetTFNNotQuotedA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR131 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion TaxOffsetTFNNotQuotedA
#region TaxOffsetEntitlementForNationalRentalAffordabilitySchemeA
                    _currentXPath.Push("/tns:TaxOffsetEntitlementForNationalRentalAffordabilitySchemeA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "TaxOffsetEntitlementForNationalRentalAffordabilitySchemeA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR133 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion TaxOffsetEntitlementForNationalRentalAffordabilitySchemeA
#region TaxOffsetExplorationCreditsA
                    _currentXPath.Push("/tns:TaxOffsetExplorationCreditsA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "TaxOffsetExplorationCreditsA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR206 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion TaxOffsetExplorationCreditsA
#region TotalRefundableTaxOffsetsA
                    _currentXPath.Push("/tns:TotalRefundableTaxOffsetsA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "TotalRefundableTaxOffsetsA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR184 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion TotalRefundableTaxOffsetsA
#region PayableA
                    _currentXPath.Push("/tns:PayableA");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "PayableA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR199 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion PayableA
#region InternationalDealingsAAMInterestSection102A
                    _currentXPath.Push("/tns:InternationalDealingsAAMInterestSection102A");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsAAMInterestSection102A" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR69 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion InternationalDealingsAAMInterestSection102A
#region PayAsYouGoWithholdingCreditForAmountsWithheldFromForeignResidentsA
                    _currentXPath.Push("/tns:PayAsYouGoWithholdingCreditForAmountsWithheldFromForeignResidentsA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "PayAsYouGoWithholdingCreditForAmountsWithheldFromForeignResidentsA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR71 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion PayAsYouGoWithholdingCreditForAmountsWithheldFromForeignResidentsA
#region PayAsYouGoWithholdingCreditForAmountsWithheldTFNNotQuotedAndABNNotQuotedA
                    _currentXPath.Push("/tns:PayAsYouGoWithholdingCreditForAmountsWithheldTFNNotQuotedAndABNNotQuotedA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "PayAsYouGoWithholdingCreditForAmountsWithheldTFNNotQuotedAndABNNotQuotedA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR72 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion PayAsYouGoWithholdingCreditForAmountsWithheldTFNNotQuotedAndABNNotQuotedA
#region PayAsYouGoWithholdingCreditForTaxWithheldFromCloselyHeldTrustShareReceivedIndirectlyA
                    _currentXPath.Push("/tns:PayAsYouGoWithholdingCreditForTaxWithheldFromCloselyHeldTrustShareReceivedIndirectlyA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "PayAsYouGoWithholdingCreditForTaxWithheldFromCloselyHeldTrustShareReceivedIndirectlyA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR175 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion PayAsYouGoWithholdingCreditForTaxWithheldFromCloselyHeldTrustShareReceivedIndirectlyA
#region TaxOffsetCalculatedInterestTFNNotQuotedA
                    _currentXPath.Push("/tns:TaxOffsetCalculatedInterestTFNNotQuotedA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "TaxOffsetCalculatedInterestTFNNotQuotedA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR132 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion TaxOffsetCalculatedInterestTFNNotQuotedA
#region PayAsYouGoWithholdingCreditForCapitalGainsWithheldFromForeignResidentsA
                    _currentXPath.Push("/tns:PayAsYouGoWithholdingCreditForCapitalGainsWithheldFromForeignResidentsA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "PayAsYouGoWithholdingCreditForCapitalGainsWithheldFromForeignResidentsA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR209 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion PayAsYouGoWithholdingCreditForCapitalGainsWithheldFromForeignResidentsA
#region DeductionCreditsEligibleTotalA
                    _currentXPath.Push("/tns:DeductionCreditsEligibleTotalA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "DeductionCreditsEligibleTotalA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR201 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion DeductionCreditsEligibleTotalA
#region TotalRemainderOfRefundableTaxOffsetsA
                    _currentXPath.Push("/tns:TotalRemainderOfRefundableTaxOffsetsA");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "TotalRemainderOfRefundableTaxOffsetsA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR185 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion TotalRemainderOfRefundableTaxOffsetsA
#region LiabilityInstalmentsTotalA
                    _currentXPath.Push("/tns:LiabilityInstalmentsTotalA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilityInstalmentsTotalA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR74 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion LiabilityInstalmentsTotalA
#region PayableOrRefundableTotalA
                    _currentXPath.Push("/tns:PayableOrRefundableTotalA");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "PayableOrRefundableTotalA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR75 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion PayableOrRefundableTotalA
                } // End of if IncomeTaxCalculation node exists

                _currentXPath.Pop();
#endregion IncomeTaxCalculation
#region Losses
                _currentXPath.Push("/tns:Losses");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "Losses" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_LossesCollectionExists = true;
                    report.RP_LossesCollectionCount += 1;
                    _found = true;
#region TaxCarriedForwardLaterIncomeYearsTotalA
                    _currentXPath.Push("/tns:TaxCarriedForwardLaterIncomeYearsTotalA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "TaxCarriedForwardLaterIncomeYearsTotalA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR76 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion TaxCarriedForwardLaterIncomeYearsTotalA
#region CapitalCarriedForwardNetA
                    _currentXPath.Push("/tns:CapitalCarriedForwardNetA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "CapitalCarriedForwardNetA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR207 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion CapitalCarriedForwardNetA
                } // End of if Losses node exists

                _currentXPath.Pop();
#endregion Losses
#region OtherInformation
                _currentXPath.Push("/tns:OtherInformation");
                //3. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "OtherInformation" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_OtherInformationCollectionExists = true;
                    report.RP_OtherInformationCollectionCount += 1;
                    _found = true;
#region ListedCountriesIncomeInternationalDealingsAttributedA
                    _currentXPath.Push("/tns:ListedCountriesIncomeInternationalDealingsAttributedA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "ListedCountriesIncomeInternationalDealingsAttributedA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR78 = currentDecimalValue;
                                }

                                report.RP_OtherInformation_ListedCountriesCollectionExists = true;
                                report.RP_OtherInformation_ListedCountriesCollectionCount += 1;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion ListedCountriesIncomeInternationalDealingsAttributedA
#region UnlistedCountriesIncomeInternationalDealingsAttributedA
                    _currentXPath.Push("/tns:UnlistedCountriesIncomeInternationalDealingsAttributedA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "UnlistedCountriesIncomeInternationalDealingsAttributedA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR80 = currentDecimalValue;
                                }

                                report.RP_OtherInformation_UnlistedCountriesCollectionExists = true;
                                report.RP_OtherInformation_UnlistedCountriesCollectionCount += 1;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion UnlistedCountriesIncomeInternationalDealingsAttributedA
#region EquityNetAssetsCalculatedA
                    _currentXPath.Push("/tns:EquityNetAssetsCalculatedA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "EquityNetAssetsCalculatedA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.FITR83 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion EquityNetAssetsCalculatedA
#region TransferOfLiabilities
                    _currentXPath.Push("/tns:TransferOfLiabilities");
                    //3. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "TransferOfLiabilities" && reader.Depth == _currentXPath.Count - 1)
                    {
                        report.RP_OtherInformation_TransferOfLiabilitiesCollectionExists = true;
                        report.RP_OtherInformation_TransferOfLiabilitiesCollectionCount += 1;
                        _found = true;
#region IncomeTaxSuperannuationTransferredAssessableI
                        _currentXPath.Push("/tns:IncomeTaxSuperannuationTransferredAssessableI");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "IncomeTaxSuperannuationTransferredAssessableI" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    currentBooleanValue = ToBoolean(currentValue);
                                    if (currentBooleanValue != null)
                                    {
                                        report.FITR84 = currentBooleanValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion IncomeTaxSuperannuationTransferredAssessableI
#region Transferee
                        _currentXPath.Push("/tns:TransfereeCollection");
                        // 4. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "TransfereeCollection" && reader.Depth == _currentXPath.Count - 1)
                        {
                            report.RP_OtherInformation_TransferOfLiabilities_TransfereeCollection = new List<FITR2026.RP_OtherInformation_TransferOfLiabilities_Transferee>();
                            report.RP_OtherInformation_TransferOfLiabilities_TransfereeCollectionExists = true;
                            _found = true;
                            while (ReadToNextElement(reader, false) && reader.LocalName == "Transferee")
                            {
                                _found = true;
                                FITR2026.RP_OtherInformation_TransferOfLiabilities_Transferee transferee = new FITR2026.RP_OtherInformation_TransferOfLiabilities_Transferee();
                                report.RP_OtherInformation_TransferOfLiabilities_TransfereeCollection.Add(transferee);
                                report.RP_OtherInformation_TransferOfLiabilities_TransfereeCollectionCount += 1;
                                transferee.OccurrenceIndex = report.RP_OtherInformation_TransferOfLiabilities_TransfereeCollectionCount;
                                _currentXPath.Push("/tns:Transferee[" + report.RP_OtherInformation_TransferOfLiabilities_TransfereeCollectionCount + "]");
#region OrganisationNameDetailsOrganisationalNameT
                                _currentXPath.Push("/tns:OrganisationNameDetailsOrganisationalNameT");
                                //6. use case
                                if (ReadToNextElement(reader, true) && reader.LocalName == "OrganisationNameDetailsOrganisationalNameT" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            transferee.FITR85 = currentValue;
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion OrganisationNameDetailsOrganisationalNameT
#region AustralianBusinessNumberId
                                _currentXPath.Push("/tns:AustralianBusinessNumberId");
                                //6. use case
                                if (ReadToNextElement(reader, true) && reader.LocalName == "AustralianBusinessNumberId" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            transferee.FITR86 = currentValue;
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion AustralianBusinessNumberId
#region IncomeTaxSuperannuationTransferredAssessableA
                                _currentXPath.Push("/tns:IncomeTaxSuperannuationTransferredAssessableA");
                                //6. use case
                                if (ReadToNextElement(reader, true) && reader.LocalName == "IncomeTaxSuperannuationTransferredAssessableA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                transferee.FITR87 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion IncomeTaxSuperannuationTransferredAssessableA
#region IncomeTaxSuperannuationMarketValueTransferA
                                _currentXPath.Push("/tns:IncomeTaxSuperannuationMarketValueTransferA");
                                //6. use case
                                if (ReadToNextElement(reader, true) && reader.LocalName == "IncomeTaxSuperannuationMarketValueTransferA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                transferee.FITR88 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion IncomeTaxSuperannuationMarketValueTransferA
                                _currentXPath.Pop();
                            } // End of while loop for Transferee
                        } // End of if TransfereeCollection node exists

                        _currentXPath.Pop();
#endregion Transferee
#region IncomeTaxSuperannuationMarketValueTotalA
                        _currentXPath.Push("/tns:IncomeTaxSuperannuationMarketValueTotalA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxSuperannuationMarketValueTotalA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.FITR89 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion IncomeTaxSuperannuationMarketValueTotalA
                    } // End of if TransferOfLiabilities node exists

                    _currentXPath.Pop();
#endregion TransferOfLiabilities
#region TaxationOfFinancialArrangements
                    _currentXPath.Push("/tns:TaxationOfFinancialArrangements");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "TaxationOfFinancialArrangements" && reader.Depth == _currentXPath.Count - 1)
                    {
                        report.RP_OtherInformation_TaxationOfFinancialArrangementsCollectionExists = true;
                        report.RP_OtherInformation_TaxationOfFinancialArrangementsCollectionCount += 1;
                        _found = true;
#region IncomeGainsTotalA
                        _currentXPath.Push("/tns:IncomeGainsTotalA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeGainsTotalA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.FITR134 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion IncomeGainsTotalA
#region ExpenseLossesTotalA
                        _currentXPath.Push("/tns:ExpenseLossesTotalA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseLossesTotalA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.FITR135 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion ExpenseLossesTotalA
                    } // End of if TaxationOfFinancialArrangements node exists

                    _currentXPath.Pop();
#endregion TaxationOfFinancialArrangements
#region OverseasTransactions
                    _currentXPath.Push("/tns:OverseasTransactions");
                    //3. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "OverseasTransactions" && reader.Depth == _currentXPath.Count - 1)
                    {
                        report.RP_OtherInformation_OverseasTransactionsCollectionExists = true;
                        report.RP_OtherInformation_OverseasTransactionsCollectionCount += 1;
                        _found = true;
#region InternationalDealingsRelatedPartiesTransactionsI
                        _currentXPath.Push("/tns:InternationalDealingsRelatedPartiesTransactionsI");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "InternationalDealingsRelatedPartiesTransactionsI" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    currentBooleanValue = ToBoolean(currentValue);
                                    if (currentBooleanValue != null)
                                    {
                                        report.FITR90 = currentBooleanValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion InternationalDealingsRelatedPartiesTransactionsI
#region InternationalDealingsRelatedPartiesTransactionsExcessAggregateValueI
                        _currentXPath.Push("/tns:InternationalDealingsRelatedPartiesTransactionsExcessAggregateValueI");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "InternationalDealingsRelatedPartiesTransactionsExcessAggregateValueI" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    currentBooleanValue = ToBoolean(currentValue);
                                    if (currentBooleanValue != null)
                                    {
                                        report.FITR91 = currentBooleanValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion InternationalDealingsRelatedPartiesTransactionsExcessAggregateValueI
#region InternationalDealingsDirectOrIndirectOverseasInterestI
                        _currentXPath.Push("/tns:InternationalDealingsDirectOrIndirectOverseasInterestI");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "InternationalDealingsDirectOrIndirectOverseasInterestI" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    currentBooleanValue = ToBoolean(currentValue);
                                    if (currentBooleanValue != null)
                                    {
                                        report.FITR92 = currentBooleanValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion InternationalDealingsDirectOrIndirectOverseasInterestI
#region InternationalDealingsForeignSourceIncomeExcessAggregateValueI
                        _currentXPath.Push("/tns:InternationalDealingsForeignSourceIncomeExcessAggregateValueI");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "InternationalDealingsForeignSourceIncomeExcessAggregateValueI" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    currentBooleanValue = ToBoolean(currentValue);
                                    if (currentBooleanValue != null)
                                    {
                                        report.FITR93 = currentBooleanValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion InternationalDealingsForeignSourceIncomeExcessAggregateValueI
#region InternationalDealingsTransactionsWithTaxHavenCountriesI
                        _currentXPath.Push("/tns:InternationalDealingsTransactionsWithTaxHavenCountriesI");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "InternationalDealingsTransactionsWithTaxHavenCountriesI" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    currentBooleanValue = ToBoolean(currentValue);
                                    if (currentBooleanValue != null)
                                    {
                                        report.FITR94 = currentBooleanValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion InternationalDealingsTransactionsWithTaxHavenCountriesI
                    } // End of if OverseasTransactions node exists

                    _currentXPath.Pop();
#endregion OverseasTransactions
#region OtherTransactions
                    _currentXPath.Push("/tns:OtherTransactions");
                    //3. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "OtherTransactions" && reader.Depth == _currentXPath.Count - 1)
                    {
                        report.RP_OtherInformation_OtherTransactionsCollectionExists = true;
                        report.RP_OtherInformation_OtherTransactionsCollectionCount += 1;
                        _found = true;
#region TaxConcessionClaimedExemptCurrentPensionIncomeAndCertificateObtainedI
                        _currentXPath.Push("/tns:TaxConcessionClaimedExemptCurrentPensionIncomeAndCertificateObtainedI");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "TaxConcessionClaimedExemptCurrentPensionIncomeAndCertificateObtainedI" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    currentBooleanValue = ToBoolean(currentValue);
                                    if (currentBooleanValue != null)
                                    {
                                        report.FITR95 = currentBooleanValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion TaxConcessionClaimedExemptCurrentPensionIncomeAndCertificateObtainedI
#region IncomeTaxDeductionClaimedDeathOrDisabilityDeductionWhichRequiresCertificateI
                        _currentXPath.Push("/tns:IncomeTaxDeductionClaimedDeathOrDisabilityDeductionWhichRequiresCertificateI");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "IncomeTaxDeductionClaimedDeathOrDisabilityDeductionWhichRequiresCertificateI" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    currentBooleanValue = ToBoolean(currentValue);
                                    if (currentBooleanValue != null)
                                    {
                                        report.FITR96 = currentBooleanValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion IncomeTaxDeductionClaimedDeathOrDisabilityDeductionWhichRequiresCertificateI
#region IncomeTaxDeductionClaimedDeathOrDisabilityDeductionCertificateObtainedI
                        _currentXPath.Push("/tns:IncomeTaxDeductionClaimedDeathOrDisabilityDeductionCertificateObtainedI");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxDeductionClaimedDeathOrDisabilityDeductionCertificateObtainedI" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    currentBooleanValue = ToBoolean(currentValue);
                                    if (currentBooleanValue != null)
                                    {
                                        report.FITR97 = currentBooleanValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion IncomeTaxDeductionClaimedDeathOrDisabilityDeductionCertificateObtainedI
#region SuperannuationPaymentPaymentsToContributingEmployersAndAssociatesI
                        _currentXPath.Push("/tns:SuperannuationPaymentPaymentsToContributingEmployersAndAssociatesI");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "SuperannuationPaymentPaymentsToContributingEmployersAndAssociatesI" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    currentBooleanValue = ToBoolean(currentValue);
                                    if (currentBooleanValue != null)
                                    {
                                        report.FITR98 = currentBooleanValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion SuperannuationPaymentPaymentsToContributingEmployersAndAssociatesI
                    } // End of if OtherTransactions node exists

                    _currentXPath.Pop();
#endregion OtherTransactions
                } // End of if OtherInformation node exists

                _currentXPath.Pop();
#endregion OtherInformation
#region RegulatoryDisclosuresReportableTaxPositionI
                _currentXPath.Push("/tns:RegulatoryDisclosuresReportableTaxPositionI");
                //6. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "RegulatoryDisclosuresReportableTaxPositionI" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.FITR210 = currentBooleanValue;
                            }
                        }
                    }

                    _found = true;
                }

                _currentXPath.Pop();
#endregion RegulatoryDisclosuresReportableTaxPositionI
#region CompletionHoursN
                _currentXPath.Push("/tns:CompletionHoursN");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "CompletionHoursN" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                            {
                                report.FITR110 = currentDecimalValue;
                            }
                        }
                    }

                    _found = true;
                }

                _currentXPath.Pop();
#endregion CompletionHoursN
#region RegulatoryDisclosuresGeneralInformationAboutFinancialStatementsT
                _currentXPath.Push("/tns:RegulatoryDisclosuresGeneralInformationAboutFinancialStatementsT");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "RegulatoryDisclosuresGeneralInformationAboutFinancialStatementsT" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.FITR202 = currentValue;
                        }
                    }

                    _found = true;
                }

                _currentXPath.Pop();
#endregion RegulatoryDisclosuresGeneralInformationAboutFinancialStatementsT
#region Declaration
                _currentXPath.Push("/tns:Declaration");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "Declaration" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_DeclarationCollectionExists = true;
                    report.RP_DeclarationCollectionCount += 1;
                    _found = true;
#region StatementTypeC
                    _currentXPath.Push("/tns:StatementTypeC");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "StatementTypeC" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FITR161 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion StatementTypeC
#region StatementAcceptedI
                    _currentXPath.Push("/tns:StatementAcceptedI");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "StatementAcceptedI" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.FITR162 = currentBooleanValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion StatementAcceptedI
#region SignatureD
                    _currentXPath.Push("/tns:SignatureD");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "SignatureD" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                                {
                                    report.FITR123 = currentDateTimeValue.Date;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion SignatureD
#region SignatoryIdentifierT
                    _currentXPath.Push("/tns:SignatoryIdentifierT");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "SignatoryIdentifierT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FITR160 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion SignatoryIdentifierT
#region PersonUnstructuredNameFullNameT
                    _currentXPath.Push("/tns:PersonUnstructuredNameFullNameT");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "PersonUnstructuredNameFullNameT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FITR167 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion PersonUnstructuredNameFullNameT
                } // End of if Declaration node exists

                _currentXPath.Pop();
#endregion Declaration
#region ContactPersonDetails
                _currentXPath.Push("/tns:ContactPersonDetails");
                //3. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "ContactPersonDetails" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_ContactPersonDetailsCollectionExists = true;
                    report.RP_ContactPersonDetailsCollectionCount += 1;
                    _found = true;
#region PersonNameDetailsTitleT
                    _currentXPath.Push("/tns:PersonNameDetailsTitleT");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "PersonNameDetailsTitleT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FITR102 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion PersonNameDetailsTitleT
#region PersonNameDetailsNameSuffixT
                    _currentXPath.Push("/tns:PersonNameDetailsNameSuffixT");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "PersonNameDetailsNameSuffixT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FITR104 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion PersonNameDetailsNameSuffixT
#region PersonNameDetailsFamilyNameT
                    _currentXPath.Push("/tns:PersonNameDetailsFamilyNameT");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "PersonNameDetailsFamilyNameT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FITR103 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion PersonNameDetailsFamilyNameT
#region PersonNameDetailsGivenNameT
                    _currentXPath.Push("/tns:PersonNameDetailsGivenNameT");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "PersonNameDetailsGivenNameT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FITR105 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion PersonNameDetailsGivenNameT
#region PersonNameDetailsOtherGivenNameT
                    _currentXPath.Push("/tns:PersonNameDetailsOtherGivenNameT");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "PersonNameDetailsOtherGivenNameT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FITR106 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion PersonNameDetailsOtherGivenNameT
#region ElectronicContactTelephoneAreaC
                    _currentXPath.Push("/tns:ElectronicContactTelephoneAreaC");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "ElectronicContactTelephoneAreaC" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FITR107 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion ElectronicContactTelephoneAreaC
#region ElectronicContactTelephoneMinimalN
                    _currentXPath.Push("/tns:ElectronicContactTelephoneMinimalN");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "ElectronicContactTelephoneMinimalN" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FITR108 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion ElectronicContactTelephoneMinimalN
#region ElectronicContactElectronicMailAddressT
                    _currentXPath.Push("/tns:ElectronicContactElectronicMailAddressT");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "ElectronicContactElectronicMailAddressT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FITR109 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion ElectronicContactElectronicMailAddressT
                } // End of if ContactPersonDetails node exists

                _currentXPath.Pop();
#endregion ContactPersonDetails
            } // End of if RP node exists

            _currentXPath.Pop();
#endregion RP
#region INT
            _currentXPath.Push("/tns:INT");
            //3. use case
            if (ReadToNextElement(reader, false) && reader.LocalName == "INT" && reader.Depth == _currentXPath.Count - 1)
            {
                report.IntCollectionExists = true;
                report.IntCollectionCount += 1;
                _found = true;
#region AustralianBusinessNumberId
                _currentXPath.Push("/tns:AustralianBusinessNumberId");
                //6. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "AustralianBusinessNumberId" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.FITR302 = currentValue;
                        }
                    }

                    _found = true;
                }

                _currentXPath.Pop();
#endregion AustralianBusinessNumberId
#region TaxAgentClientReferenceT
                _currentXPath.Push("/tns:TaxAgentClientReferenceT");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "TaxAgentClientReferenceT" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.FITR119 = currentValue;
                        }
                    }

                    _found = true;
                }

                _currentXPath.Pop();
#endregion TaxAgentClientReferenceT
#region TaxAgentNumberId
                _currentXPath.Push("/tns:TaxAgentNumberId");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "TaxAgentNumberId" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.FITR2 = currentValue;
                        }
                    }

                    _found = true;
                }

                _currentXPath.Pop();
#endregion TaxAgentNumberId
#region PersonNameDetailsTitleT
                _currentXPath.Push("/tns:PersonNameDetailsTitleT");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "PersonNameDetailsTitleT" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.FITR111 = currentValue;
                        }
                    }

                    _found = true;
                }

                _currentXPath.Pop();
#endregion PersonNameDetailsTitleT
#region PersonNameDetailsNameSuffixT
                _currentXPath.Push("/tns:PersonNameDetailsNameSuffixT");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "PersonNameDetailsNameSuffixT" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.FITR113 = currentValue;
                        }
                    }

                    _found = true;
                }

                _currentXPath.Pop();
#endregion PersonNameDetailsNameSuffixT
#region PersonNameDetailsFamilyNameT
                _currentXPath.Push("/tns:PersonNameDetailsFamilyNameT");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "PersonNameDetailsFamilyNameT" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.FITR112 = currentValue;
                        }
                    }

                    _found = true;
                }

                _currentXPath.Pop();
#endregion PersonNameDetailsFamilyNameT
#region PersonNameDetailsGivenNameT
                _currentXPath.Push("/tns:PersonNameDetailsGivenNameT");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "PersonNameDetailsGivenNameT" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.FITR114 = currentValue;
                        }
                    }

                    _found = true;
                }

                _currentXPath.Pop();
#endregion PersonNameDetailsGivenNameT
#region PersonNameDetailsOtherGivenNameT
                _currentXPath.Push("/tns:PersonNameDetailsOtherGivenNameT");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "PersonNameDetailsOtherGivenNameT" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.FITR115 = currentValue;
                        }
                    }

                    _found = true;
                }

                _currentXPath.Pop();
#endregion PersonNameDetailsOtherGivenNameT
#region OrganisationNameDetailsOrganisationalNameT
                _currentXPath.Push("/tns:OrganisationNameDetailsOrganisationalNameT");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "OrganisationNameDetailsOrganisationalNameT" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.FITR116 = currentValue;
                        }
                    }

                    _found = true;
                }

                _currentXPath.Pop();
#endregion OrganisationNameDetailsOrganisationalNameT
#region ElectronicContactTelephoneAreaC
                _currentXPath.Push("/tns:ElectronicContactTelephoneAreaC");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "ElectronicContactTelephoneAreaC" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.FITR117 = currentValue;
                        }
                    }

                    _found = true;
                }

                _currentXPath.Pop();
#endregion ElectronicContactTelephoneAreaC
#region ElectronicContactTelephoneMinimalN
                _currentXPath.Push("/tns:ElectronicContactTelephoneMinimalN");
                //6. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "ElectronicContactTelephoneMinimalN" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.FITR118 = currentValue;
                        }
                    }

                    _found = true;
                }

                _currentXPath.Pop();
#endregion ElectronicContactTelephoneMinimalN
#region Declaration
                _currentXPath.Push("/tns:Declaration");
                //3. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "Declaration" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.InT_DeclarationCollectionExists = true;
                    report.InT_DeclarationCollectionCount += 1;
                    _found = true;
#region StatementTypeC
                    _currentXPath.Push("/tns:StatementTypeC");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "StatementTypeC" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FITR164 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion StatementTypeC
#region StatementAcceptedI
                    _currentXPath.Push("/tns:StatementAcceptedI");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "StatementAcceptedI" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.FITR165 = currentBooleanValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion StatementAcceptedI
#region SignatureD
                    _currentXPath.Push("/tns:SignatureD");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "SignatureD" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                                {
                                    report.FITR125 = currentDateTimeValue.Date;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion SignatureD
#region SignatoryIdentifierT
                    _currentXPath.Push("/tns:SignatoryIdentifierT");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "SignatoryIdentifierT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FITR163 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion SignatoryIdentifierT
#region PersonUnstructuredNameFullNameT
                    _currentXPath.Push("/tns:PersonUnstructuredNameFullNameT");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "PersonUnstructuredNameFullNameT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.FITR169 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion PersonUnstructuredNameFullNameT
                } // End of if Declaration node exists

                _currentXPath.Pop();
#endregion Declaration
            } // End of if INT node exists

            _currentXPath.Pop();
#endregion INT
            while (reader.EOF != true)
                reader.Read();
            return report;
        }

        private string _elementCsvList = "RP#TargetFinancialY,RP#PeriodStartD,RP#PeriodEndD,RP#TaxFileNumberId,RP#OrganisationNameDetailsOrganisationalNameT,RP#AustralianBusinessNumberId,AmendmentDetails#I,AmendmentDetails#AmendmentSequenceN,AmendmentDetails#AmendmentReasonT,AmendmentDetails#AmendmentTypeC,AmendmentDetails,CurrentPostalAddress#AddressDetailsLine1T,CurrentPostalAddress#AddressDetailsLine2T,CurrentPostalAddress#AddressDetailsLocalityNameT,CurrentPostalAddress#AddressDetailsPostcodeT,CurrentPostalAddress#AddressDetailsStateOrTerritoryC,CurrentPostalAddress#AddressDetailsCountryC,CurrentPostalAddress,TrusteeDetails#OrganisationNameDetailsOrganisationalNameT,TrusteeDetails#AustralianBusinessNumberId,TrusteeDetails,ElectronicFundTransfer#FinancialInstitutionAccountBankStateBranchN,ElectronicFundTransfer#FinancialInstitutionAccountFinancialInstitutionAccountN,ElectronicFundTransfer#FinancialInstitutionAccountFinancialInstitutionAccountNameT,ElectronicFundTransfer,StatusOfFundOrTrust#PartyTypeC,StatusOfFundOrTrust#SuperannuationFundDetailsAustralianSuperannuationFundStatusI,StatusOfFundOrTrust#SuperannuationFundDetailsFundBenefitStructureC,StatusOfFundOrTrust#OrganisationControlRegisteredMemberCt,StatusOfFundOrTrust#OrganisationDetailsEstablishmentD,StatusOfFundOrTrust#OrganisationDetailsSignificantGlobalEntityStatusI,StatusOfFundOrTrust#InternationalDealingsCountryByCountryReportingEntityStatusI,StatusOfFundOrTrust#OrganisationDetailsOperationCeaseI,StatusOfFundOrTrust#OrganisationDetailsOrganisationActivityStatusEndD,StatusOfFundOrTrust,Income#CapitalGainsTaxEventI,Income#CapitalGainsTaxExemptionOrRolloverAppliedI,CapitalGainsTaxExemptionOrRolloverAppliedC,CapitalGainsTaxExemptionOrRolloverAppliedCCollection,Income#CapitalGainsNetA,Income#RentLeasingHiringGrossA,Income#InterestGrossA,Income#ForestryManagedInvestmentSchemeA,Income#GrossA,Income#NetA,Income#TaxFrankingCreditsReceivedFromNewZealandCompaniesA,Income#EquityInternationalDealingsForeignFundTransfersA,Income#InternationalDealingsForeignFundTransfersCt,Income#RemunerationABNNotQuotedPaymentGrossA,Income#PartnershipDistributionGrossA,Income#DividendsUnfrankedA,Income#DividendsFrankedA,Income#TaxFrankingCreditsA,Income#TrustDistributionUnfrankedA,Income#TrustDistributionFrankedA,Income#TaxFrankingCreditsFrankingCreditShareReceivedIndirectlyA,Income#TrustDistributionOtherA,Income#SuperannuationContributionEmployerAssessableA,Income#SuperannuationContributionPersonalAssessableA,Income#SuperannuationContributionEmployerContributionsTaxFileNumberNotQuotedA,Income#SuperannuationContributionContributionsExcludedByTrusteeA,Income#SuperannuationContributionReducedContributionsA,Income#SuperannuationContributionLiabilityTransferToLifeInsuranceCompanyOrPooledSuperannuationTrustAssessableA,Income#ForeignExchangeGainsA,Income#OtherA,Income#OtherC,Income#AssessableIncomeDueToFundTaxStatusChangeA,Income#NonArmsLengthOtherNetA,Income#TaxConcessionPensionIncomeExemptDeductionA,Income,InterestExpensesAustraliaA,InterestExpensesOverseasA,Deductions#RemunerationWagesandSalariesPaymentsA,Deductions#ExpenseCapitalWorksDeductionA,Deductions#ExpenseDepreciationAllowableDeductionA,Deductions#ExpensePremiumsDeathOrDisabilityA,Deductions#ExpenseInvestmentA,Deductions#ExpenseOperatingManagementExpenseA,Deductions#ExpenseOperatingAdministrativeExpenseA,Deductions#ExpenseForestryManagedInvestmentSchemeDeductionA,Deductions#IncomeForeignExchangeLossesA,Deductions#ExpenseDeductibleOtherA,Deductions#ExpenseDeductibleOtherC,Deductions#IncomeTaxTaxLossesDeductedA,Deductions#IncomeTaxableA,Deductions,IncomeTaxCalculation#TotalTaxOnTaxableIncomeA,IncomeTaxCalculation#TFNNotQuotedContributionsTaxA,IncomeTaxCalculation#TaxableGrossA,IncomeTaxCalculation#InternationalDealingsTaxOffsetA,IncomeTaxCalculation#TaxOffsetsAndRebatesTotalA,IncomeTaxCalculation#TaxOffsetNonRefundableNonCarryForwardA,IncomeTaxCalculation#SubtotalTaxPayableA,ESVCLPTaxOffsetClaimNonRefundableOtherA,PreviousYearESVCLPTaxOffsetClaimNonRefundableOtherA,EarlyStageInvestorTaxOffsetClaimNonRefundableOtherA,PreviousYearEarlyStageInvestorTaxOffsetClaimNonRefundableOtherA,IncomeTaxCalculation#TaxOffsetNonRefundableCarryForwardA,IncomeTaxCalculation#TaxOffsetCarriedForwardSubTotalA,IncomeTaxCalculation#FrankingCreditsRefundableA,IncomeTaxCalculation#TaxOffsetTFNNotQuotedA,IncomeTaxCalculation#TaxOffsetEntitlementForNationalRentalAffordabilitySchemeA,IncomeTaxCalculation#TaxOffsetExplorationCreditsA,IncomeTaxCalculation#TotalRefundableTaxOffsetsA,IncomeTaxCalculation#PayableA,IncomeTaxCalculation#InternationalDealingsAAMInterestSection102A,IncomeTaxCalculation#PayAsYouGoWithholdingCreditForAmountsWithheldFromForeignResidentsA,IncomeTaxCalculation#PayAsYouGoWithholdingCreditForAmountsWithheldTFNNotQuotedAndABNNotQuotedA,IncomeTaxCalculation#PayAsYouGoWithholdingCreditForTaxWithheldFromCloselyHeldTrustShareReceivedIndirectlyA,IncomeTaxCalculation#TaxOffsetCalculatedInterestTFNNotQuotedA,IncomeTaxCalculation#PayAsYouGoWithholdingCreditForCapitalGainsWithheldFromForeignResidentsA,IncomeTaxCalculation#DeductionCreditsEligibleTotalA,IncomeTaxCalculation#TotalRemainderOfRefundableTaxOffsetsA,IncomeTaxCalculation#LiabilityInstalmentsTotalA,IncomeTaxCalculation#PayableOrRefundableTotalA,IncomeTaxCalculation,Losses#TaxCarriedForwardLaterIncomeYearsTotalA,Losses#CapitalCarriedForwardNetA,Losses,ListedCountriesIncomeInternationalDealingsAttributedA,UnlistedCountriesIncomeInternationalDealingsAttributedA,OtherInformation#EquityNetAssetsCalculatedA,TransferOfLiabilities#IncomeTaxSuperannuationTransferredAssessableI,Transferee#OrganisationNameDetailsOrganisationalNameT,Transferee#AustralianBusinessNumberId,Transferee#IncomeTaxSuperannuationTransferredAssessableA,Transferee#IncomeTaxSuperannuationMarketValueTransferA,Transferee,TransfereeCollection,TransferOfLiabilities#IncomeTaxSuperannuationMarketValueTotalA,TransferOfLiabilities,TaxationOfFinancialArrangements#IncomeGainsTotalA,TaxationOfFinancialArrangements#ExpenseLossesTotalA,TaxationOfFinancialArrangements,OverseasTransactions#InternationalDealingsRelatedPartiesTransactionsI,OverseasTransactions#InternationalDealingsRelatedPartiesTransactionsExcessAggregateValueI,OverseasTransactions#InternationalDealingsDirectOrIndirectOverseasInterestI,OverseasTransactions#InternationalDealingsForeignSourceIncomeExcessAggregateValueI,OverseasTransactions#InternationalDealingsTransactionsWithTaxHavenCountriesI,OverseasTransactions,OtherTransactions#TaxConcessionClaimedExemptCurrentPensionIncomeAndCertificateObtainedI,OtherTransactions#IncomeTaxDeductionClaimedDeathOrDisabilityDeductionWhichRequiresCertificateI,OtherTransactions#IncomeTaxDeductionClaimedDeathOrDisabilityDeductionCertificateObtainedI,OtherTransactions#SuperannuationPaymentPaymentsToContributingEmployersAndAssociatesI,OtherTransactions,OtherInformation,RP#RegulatoryDisclosuresReportableTaxPositionI,RP#CompletionHoursN,RP#RegulatoryDisclosuresGeneralInformationAboutFinancialStatementsT,Declaration#StatementTypeC,Declaration#StatementAcceptedI,Declaration#SignatureD,Declaration#SignatoryIdentifierT,Declaration#PersonUnstructuredNameFullNameT,Declaration,ContactPersonDetails#PersonNameDetailsTitleT,ContactPersonDetails#PersonNameDetailsNameSuffixT,ContactPersonDetails#PersonNameDetailsFamilyNameT,ContactPersonDetails#PersonNameDetailsGivenNameT,ContactPersonDetails#PersonNameDetailsOtherGivenNameT,ContactPersonDetails#ElectronicContactTelephoneAreaC,ContactPersonDetails#ElectronicContactTelephoneMinimalN,ContactPersonDetails#ElectronicContactElectronicMailAddressT,ContactPersonDetails,RP,INT#AustralianBusinessNumberId,INT#TaxAgentClientReferenceT,INT#TaxAgentNumberId,INT#PersonNameDetailsTitleT,INT#PersonNameDetailsNameSuffixT,INT#PersonNameDetailsFamilyNameT,INT#PersonNameDetailsGivenNameT,INT#PersonNameDetailsOtherGivenNameT,INT#OrganisationNameDetailsOrganisationalNameT,INT#ElectronicContactTelephoneAreaC,INT#ElectronicContactTelephoneMinimalN,Declaration#StatementTypeC,Declaration#StatementAcceptedI,Declaration#SignatureD,Declaration#SignatoryIdentifierT,Declaration#PersonUnstructuredNameFullNameT,Declaration,INT";
    }
}