using Ato.CD.Inbound.Shared;
using Ato.CD.Inbound.Shared201802;
using Ato.EN.IntegrationServices.CodeGenerationCGTS;
using Ato.EN.IntegrationServices.CodeGenerationFITR;
using DataContracts;
using System.Collections.Generic;
using System.Linq;
using System.Runtime.CompilerServices;
using VaTS;

namespace Ato.CD.Inbound.FITR202602
{
    internal class CrossFormValidatorCGTS : ICrossFormValidator
    {
        private FITR2026 ParentDocument { get; }
        private CGTS2018 ChildDocument { get; }
        private List<Context> Contexts { get; }
        public List<ProcessMessageDocument> Response { get; private set; }

        internal CrossFormValidatorCGTS(FITR2026 report, BusinessDocument childDocument)
        {
            ParentDocument = report;
            ChildDocument = (CGTS2018)childDocument.ConsumedReport;
            Contexts = ChildDocument.GetContexts();
            Response = new List<ProcessMessageDocument>();
        }

        public List<ProcessMessageDocument> ValidateCrossFormRules()
        {
            //Parent form cross form rules
            VRATOFITR434036();
            VRATOFITR434088();
            VRATOFITR434533();
            VRATOFITR434534();
            VRATOFITR434535();
            VRATOFITR434536();

            //Child form cross form rules
            VRATOCGTS402165();
            VRATOCGTS402171();
            VRATOCGTS402178();
            VRATOCGTS402226();
            VRATOCGTS402230();
            VRATOCGTS402240();
            VRATOGEN402009();
            VRATOGEN438000();
            VRATOGEN438001();

            return Response;
        }

        #region VR.ATO.FITR.434036

        /* VR.ATO.FITR.434036
        Net capital gain in return must be equal to the net capital gain in CGT schedule

        Legacy Rule Format:
        IF (COUNT(SCHEDULE = ""CGTS"") = 1) AND [FITR23] <> [CGTS131]
            RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:
        ^CGTS131 = CGTS:RP:Income.CapitalGainsNet.Amount
        ^FITR23 = FITR:RP:Income.CapitalGainsNet.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFITR434036( )
        {
            bool assertion = ParentDocument.FITR23.HasValue && ParentDocument.FITR23.Value != ChildDocument.CGTS131.GetValueOrDefault();

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.434036",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Net capital gains in return must equal total net capital gain in CGT schedule.",
                    Location = "/tns:FITR/tns:RP/tns:Income/tns:CapitalGainsNetA",
                    Parameters = new ProcessMessageParameters {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.FITR.434036" },
                        new ProcessMessageParameter { Name = "FITR23", Value = FITR2026Validator.GetValueOrEmpty(ParentDocument.FITR23) },
                        new ProcessMessageParameter { Name = "CGTS131", Value = FITR2026Validator.GetValueOrEmpty(ChildDocument.CGTS131) }
                    },
                    DocumentSequence = "parent_doc"
                };

                Response.Add(processMessage);
            }
        }

        #endregion //VR.ATO.FITR.434036

        #region VR.ATO.FITR.434088

        /*  VR.ATO.FITR.434088
        Net capital losses carried forward to later income years must equal amount in CGT schedule

        Technical Business Rule:
        IF (COUNT(SCHEDULE = "CGTS") = 1) AND  [FITR207] <> ([CGTS124]+[CGTS125])
            RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:
        ^FITR207 = tns:Capital.Losses.CarriedForward.Net.Amount
        ^CGTS124 = CGTS:RP.Collectables.Unapplied(Instant):Capital.Losses.CarriedForward.Net.Amount
        ^CGTS125 = CGTS:RP.CGTOther.Unapplied(Instant):Capital.Losses.CarriedForward.Net.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFITR434088( )
        {
            bool assertion = ParentDocument.FITR207.GetValueOrDefault() != ChildDocument.CGTS124.GetValueOrDefault() + ChildDocument.CGTS125.GetValueOrDefault();

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.FITR.434289",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Net capital losses carried forward to later income years must equal amount in CGT schedule",
                    LongDescription = @"Net capital losses carried forward to later income years' on the main form must equal the total of the 'Unapplied net capital losses carried forward'  in the Capital Gains Tax schedule",
                    Location = "/tns:FITR/tns:RP/tns:Losses/tns:CapitalCarriedForwardNetA",
                    Parameters = new ProcessMessageParameters() {
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.FITR.434088" },
                        new ProcessMessageParameter() { Name = "FITR207", Value = FITR2026Validator.GetValueOrEmpty(ParentDocument.FITR207) },
                        new ProcessMessageParameter() { Name = "CGTS124", Value = FITR2026Validator.GetValueOrEmpty(ChildDocument.CGTS124) },
                        new ProcessMessageParameter() { Name = "CGTS125", Value = FITR2026Validator.GetValueOrEmpty(ChildDocument.CGTS125) }
                    },
                    DocumentSequence = "parent_doc"
                };

                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.FITR.434088

        #region VR.ATO.FITR.434533

        /*  VR.ATO.FITR.434533
        If taxpayer is required to lodge a CGT schedule and CGT exemption or rollover applied code is "Small business active asset reduction (Subdivision 152-C)", then Small business active asset reduction amount on attached CGT schedule must be completed

        Technical Business Rule:
        (^FITR204 = 'A - Small business active asset reduction') AND (CountDocument('CGTS') = 1) AND ^CGTS127 = NULL
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
         public virtual void VRATOFITR434533( )
         {
            bool assertion = ParentDocument.FITR204Collection != null && ParentDocument.FITR204Collection.Any(s => s.Equals("A - Small business active asset reduction")) && !ChildDocument.CGTS127.HasValue;

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.438022",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Small business active asset reduction amount missing from CGT schedule",
                    LongDescription = @"If taxpayer is required to lodge a CGT schedule and CGT exemption or rollover applied code is ""Small business active asset reduction (Subdivision 152-C)"", then Small business active asset reduction amount on attached CGT schedule must be completed",
                    Location = "/tns:FITR/tns:RP/tns:Income/tns:CapitalGainsTaxExemptionOrRolloverAppliedCCollection/tns:CapitalGainsTaxExemptionOrRolloverAppliedC",
                    Parameters = new ProcessMessageParameters {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.FITR.434533" }
                    },
                    DocumentSequence = "parent_doc"
                };

                Response.Add(processMessage);
            }
         }
 
        #endregion // VR.ATO.FITR.434533

        #region VR.ATO.FITR.434534

        /*  VR.ATO.FITR.434534
        If taxpayer is required to lodge a CGT schedule and CGT exemption or rollover applied code is "Small business roll-over (Subdivision 152-E)", then Small business roll-over amount on attached CGT schedule must be completed

        Technical Business Rule:
        (^FITR204 = 'C - Small business roll-over') AND (CountDocument('CGTS') = 1) AND ^CGTS129 = NULL
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
         public virtual void VRATOFITR434534( )
         {
            bool assertion = ParentDocument.FITR204Collection != null && ParentDocument.FITR204Collection.Any(s => s.Equals("C - Small business roll-over")) && !ChildDocument.CGTS129.HasValue;

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.438024",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Small business roll-over amount missing from CGT schedule",
                    LongDescription = @"If taxpayer is required to lodge a CGT schedule and CGT exemption or rollover applied code is ""Small business roll-over (Subdivision 152-E)"", then Small business roll-over amount on attached CGT schedule must be completed",
                    Location = "/tns:FITR/tns:RP/tns:Income/tns:CapitalGainsTaxExemptionOrRolloverAppliedCCollection/tns:CapitalGainsTaxExemptionOrRolloverAppliedC",
                    Parameters = new ProcessMessageParameters {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.FITR.434534" }
                    },
                    DocumentSequence = "parent_doc"
                };

                Response.Add(processMessage);
            }
         }

        #endregion // VR.ATO.FITR.434534

        #region VR.ATO.FITR.434535

        /*  VR.ATO.FITR.434535
        If taxpayer is required to lodge a CGT schedule and CGT exemption or rollover applied code is "Small business 15 year exemption (Subdivision152-B)", then Small business 15 year exemption amount on attached CGT schedule must be completed

        Technical Business Rule:
        (^FITR204 = 'D - Small business 15 year exemption') AND (CountDocument('CGTS') = 1) AND ^CGTS138 = NULL
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFITR434535( )
        {
            bool assertion = ParentDocument.FITR204Collection != null && ParentDocument.FITR204Collection.Any(s => s.Equals("D - Small business 15 year exemption")) && !ChildDocument.CGTS138.HasValue;

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.438025",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Small business 15 year exemption amount missing from CGT schedule",
                    LongDescription = @"If taxpayer is required to lodge a CGT schedule and CGT exemption or rollover applied code is ""Small business 15 year exemption (Subdivision152-B)"", then Small business 15 year exemption amount on attached CGT schedule must be completed",
                    Location = "/tns:FITR/tns:RP/tns:Income/tns:CapitalGainsTaxExemptionOrRolloverAppliedCCollection/tns:CapitalGainsTaxExemptionOrRolloverAppliedC",
                    Parameters = new ProcessMessageParameters {
                        new ProcessMessageParameter {Name = "RuleIdentifier", Value = "VR.ATO.FITR.434535" }
                    },
                    DocumentSequence = "parent_doc"
                };

                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.FITR.434535

        #region VR.ATO.FITR.434536

        /*  VR.ATO.FITR.434536
        If taxpayer is required to lodge a CGT schedule and CGT exemption or rollover applied code is "Scrip for scrip roll-over (Subdivision 124-M)", then Capital gains disregarded as a result of a scrip for scrip roll-over amount on attached CGT schedule must be completed

        Technical Business Rule:
        (^FITR204 = 'F - Scrip for scrip roll-over') AND (CountDocument('CGTS') = 1) AND ^CGTS45 = NULL
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFITR434536( )
        {
            bool assertion = ParentDocument.FITR204Collection != null && ParentDocument.FITR204Collection.Any(s => s.Equals("F - Scrip for scrip roll-over")) && !ChildDocument.CGTS45.HasValue;

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.438027",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Capital gains disregarded as a result of a scrip for scrip roll-over amount missing from CGT schedule",
                    LongDescription = @"If taxpayer is required to lodge a CGT schedule and CGT exemption or rollover applied code is ""Scrip for scrip roll-over (Subdivision 124-M)"", then Capital gains disregarded as a result of a scrip for scrip roll-over amount on attached CGT schedule must be completed",
                    Location = "/tns:FITR/tns:RP/tns:Income/tns:CapitalGainsTaxExemptionOrRolloverAppliedCCollection/tns:CapitalGainsTaxExemptionOrRolloverAppliedC",
                    Parameters = new ProcessMessageParameters {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.FITR.434536" }
                    },
                    DocumentSequence = "parent_doc"
                };

                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.FITR.434536

        #region VR.ATO.CGTS.402165

        /*  VR.ATO.CGTS.402165
        Total losses transferred in applied is only valid for a company

        Legacy Rule Format:
        IF (PARENT RETURN <> "CTR") AND ([CGTS122] <> NULL)
            RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:
        ^CGTS122 = CGTS:RP.Applied:tns:Capital.Losses.TransferredTotal.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCGTS402165( )
        {
            bool assertion = ChildDocument.CGTS122.HasValue;

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.CGTS.402165",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total losses transferred in applied is only valid for a company",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.TransferredTotal.Amount",
                    Parameters = new ProcessMessageParameters {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402165" },
                        new ProcessMessageParameter { Name = "CGTS122", Value = FITR2026Validator.GetValueOrEmpty(ChildDocument.CGTS122) }
                    }
                };
                
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CGTS.402165

        #region VR.ATO.CGTS.402171

        /*  VR.ATO.CGTS.402171
        A capital loss must not be carried forward when a positive net capital gain amount has been provided 

        Legacy Rule Format:
        IF PARENT RETURN = "SMSFAR", "FITR", "TRT" AND ([CGTS131] > 0) AND ([CGTS125] > 0)
            RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:
        ^CGTS125 = CGTS:RP.CGTOther.Unapplied:Capital.Losses.CarriedForward.Net.Amount
        ^CGTS131 = CGTS:RP:Income.CapitalGainsNet.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCGTS402171( )
        {
            bool assertion = ChildDocument.CGTS131.GetValueOrDefault() > 0 && ChildDocument.CGTS125.GetValueOrDefault() > 0;

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.CGTS.402171",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"A capital loss must not be carried forward when a positive net capital gain amount has been provided",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.CarriedForward.Net.Amount",
                    Parameters = new ProcessMessageParameters {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402171" },
                        new ProcessMessageParameter { Name = "CGTS125", Value = FITR2026Validator.GetValueOrEmpty(ChildDocument.CGTS125) },
                        new ProcessMessageParameter { Name = "CGTS131", Value = FITR2026Validator.GetValueOrEmpty(ChildDocument.CGTS131) }
                    }
                };
                
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CGTS.402171

        #region VR.ATO.CGTS.402178

        /*  VR.ATO.CGTS.402178
        The small business retirement exemption is not valid for Funds or Self Managed Super Funds

        Legacy Rule Format:
        IF PARENT RETURN = "SMSFAR","FITR" AND ([CGTS128] <> NULL)
            RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:
        ^CGTS128 = CGTS:RP:TaxConcession.CapitalGains.SmallBusinessRetirementExemption.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCGTS402178( )
        {
            bool assertion = ChildDocument.CGTS128.HasValue;

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.CGTS.402178",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Small business retirement exemption is not valid for Self Managed Super Funds",
                    Location = "/xbrli:xbrl/tns:TaxConcession.CapitalGains.SmallBusinessRetirementExemption.Amount",
                    Parameters = new ProcessMessageParameters {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402178" },
                        new ProcessMessageParameter { Name = "CGTS128", Value = FITR2026Validator.GetValueOrEmpty(ChildDocument.CGTS128) }
                    }
                };

                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CGTS.402178

        #region VR.ATO.CGTS.402226

        /*  VR.ATO.CGTS.402226
        If the Amendment indicator on a parent return is set to TRUE, then the Income year earnout right created and Amount to be amended cannot be supplied.

        Legacy Rule Format:
        IF (PARENT RETURN:RP:pyin.xx.xx:Report.Amendment.Indicator = TRUE) AND ([CGTS150] <> NULL AND [CGTS151] <> NULL)
            RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:
        ^FITR171 = FITR:RP:Report.Amendment.Indicator
        ^CGTS150 = CGTS:RP:Report.TargetFinancial.Year
        ^CGTS151 = CGTS:RP:Miscellaneous.RequestedAmendment.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCGTS402226( )
        {
            bool assertion = ParentDocument.FITR171.GetValueOrDefault() && ChildDocument.CGTS150.HasValue && ChildDocument.CGTS151.HasValue;

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.CGTS.402226",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Income year earnout right created and Amount to be amended cannot be supplied when Parent return Amendment indicator is present.",
                    Location = "/xbrli:xbrl/tns:Report.TargetFinancial.Year",
                    Parameters = new ProcessMessageParameters {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402226" },
                        new ProcessMessageParameter { Name = "FITR171", Value = FITR2026Validator.GetValueOrEmpty(ParentDocument.FITR171) },
                        new ProcessMessageParameter { Name = "CGTS150", Value = FITR2026Validator.GetValueOrEmpty(ChildDocument.CGTS150) },
                        new ProcessMessageParameter { Name = "CGTS151", Value = FITR2026Validator.GetValueOrEmpty(ChildDocument.CGTS151) }
                    }
                };

                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CGTS.402226

        #region VR.ATO.CGTS.402230

        /*  VR.ATO.CGTS.402230
        A multi-class treatment can only be reported with a Trust Tax Return for Attribution Managed Investments (TRTAMI) when a multi-class election has been made

        Legacy Rule Format:
        IF PARENT RETURN  <> "TRTAMI" AND ([CGTS153] <> NULL OR [CGTS154] <> NULL OR [CGTS155] <> NULL)
            RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:
        ^CGTS153 = CGTS:RP:IncomeTax.AttributionManagedInvestmentTrustClass.Description
        ^CGTS154 = CGTS:RP:Capital.Gains.Total.Amount
        ^CGTS155 = CGTS:RP:Capital.Losses.Total.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCGTS402230( )
        {
            bool assertion = !string.IsNullOrEmpty(ChildDocument.CGTS153) || ChildDocument.CGTS154.HasValue || ChildDocument.CGTS155.HasValue;

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.CGTS.402230",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Multi-class treatment fields incorrectly supplied",
                    Location = "/xbrli:xbrl/tns:IncomeTax.AttributionManagedInvestmentTrustClass.Description",
                    Parameters = new ProcessMessageParameters {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402230" },
                        new ProcessMessageParameter { Name = "CGTS153", Value = FITR2026Validator.GetValueOrEmpty(ChildDocument.CGTS153) },
                        new ProcessMessageParameter { Name = "CGTS154", Value = FITR2026Validator.GetValueOrEmpty(ChildDocument.CGTS154) },
                        new ProcessMessageParameter { Name = "CGTS155", Value = FITR2026Validator.GetValueOrEmpty(ChildDocument.CGTS155) }
                    }
                };

                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CGTS.402230

        #region VR.ATO.CGTS.402240

        /*  VR.ATO.CGTS.402240
        The year entered at Income year earnout right created cannot be the current year.

        Legacy Rule Format:
        IF([CGTS150] <> NULL) AND ([CGTS150] = (PARENT RETURN:RP:pyin.xx.xx:Report.TargetFinancial.Year))
            RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:
        ^FITR1 = FITR:RP:Report.TargetFinancial.Year
        ^CGTS150 = CGTS:RP:Report.TargetFinancial.Year
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public void VRATOCGTS402240( )
        {
            bool assertion = ChildDocument.CGTS150.HasValue && ChildDocument.CGTS150 == ParentDocument.FITR1.GetValueOrDefault();

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.CGTS.402240",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Income year earnout right created incorrect",
                    Location = "/xbrli:xbrl/tns:Report.TargetFinancial.Year",
                    Parameters = new ProcessMessageParameters {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402240" },
                        new ProcessMessageParameter { Name = "FITR1", Value = FITR2026Validator.GetValueOrEmpty(ParentDocument.FITR1) },
                        new ProcessMessageParameter { Name = "CGTS150", Value = FITR2026Validator.GetValueOrEmpty(ChildDocument.CGTS150) }
                    }
                };

                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.CGTS.402240

        #region VR.ATO.GEN.402009

        /*  VR.ATO.GEN.402009
        Your supplied TFN does not match the TFN supplied on the form it was submitted with

        Legacy Rule Format:
        IF (RP:entity.identifier.TFN <> PARENT RETURN:RP:entity.identifier.TFN)
            RETURN VALIDATION MESSAGE
        ENDIF 

        Data Elements:
        ^FITR5 = FITR:RP:Identifiers.TaxFileNumber.Identifier
        ^CGTS TFN = CGTS:RP:Identifiers.TaxFileNumber.Identifier
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN402009()
        {
            bool assertion = !ParentDocument.FITR5.Equals(ChildDocument.RPIdentifierTFN);

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.402009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your supplied TFN does not match the TFN supplied on the form it was submitted with",
                    LongDescription = @"Your supplied TFN does not match the TFN supplied on the parent form it was submitted with",
                    Location = "/xbrli:xbrl/xbrli:context/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.GEN.402009" },
                        new ProcessMessageParameter { Name = "FITR5", Value = ParentDocument.FITR5 },
                        new ProcessMessageParameter { Name = "CGTS TFN", Value = ChildDocument.RPIdentifierTFN }
                    }
                };

                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.GEN.402009

        #region VR.ATO.GEN.438000

        /*  VR.ATO.GEN.438000
        Period start date within context on schedule doesn't match start date within PARENT RETURN:reporting party context

        Legacy Rule Format:
        IF (period.startDate WHERE CONTEXT(ALL)) <> PARENT RETURN:RP:period.startDate
            RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:
        ^FITR300 = FITR:RP:Period.Start.Date
        ^CGTS Start Date = CGTS Start Date
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438000()
        {
            IEnumerable<Context> contexts = Contexts.Where(context => context.StartDate.GetValueOrDefault() != ParentDocument.FITR300.GetValueOrDefault());

            IEnumerable<ProcessMessageDocument> processMessages = contexts.Select(context => new ProcessMessageDocument
            {
                Code = "CMN.ATO.GEN.438000",
                Severity = ProcessMessageSeverity.Error,
                Description = @"The context period start date is incorrect.",
                LongDescription = @"Period start date within context on the schedule must match period start date within the parent return reporting party context",
                Location = $"/xbrli:xbrl/xbrli:context[{context.Index}]/xbrli:period/xbrli:startDate",
                Parameters = new ProcessMessageParameters { 
                    new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438000" },
                    new ProcessMessageParameter { Name = "FITR300", Value = FITR2026Validator.GetValueOrEmptyDate(ParentDocument.FITR300) },
                    new ProcessMessageParameter { Name = "CGTS Start Date", Value = FITR2026Validator.GetValueOrEmptyDate(context.StartDate) }
                }
            });

            Response.AddRange(processMessages);
        }

        #endregion // VR.ATO.CGTS.438000

        #region VR.ATO.GEN.438001

        /*  VR.ATO.GEN.438001
        Period end date within context on schedule doesn't match end date within PARENT RETURN:reporting party context

        Legacy Rule Format:
        IF (period.endDate WHERE CONTEXT(ALL)) <> PARENT RETURN:RP:period.endDate
            RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:
        ^FITR301 = FITR:RP:Period.End.Date
        ^CGTS End Date = CGTS End Date
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438001()
        {
            IEnumerable<Context> contexts = Contexts.Where(context => context.EndDate.GetValueOrDefault() != ParentDocument.FITR301.GetValueOrDefault());

            IEnumerable<ProcessMessageDocument> processMessages = contexts.Select(context => new ProcessMessageDocument
            {
                Code = "CMN.ATO.GEN.438001",
                Severity = ProcessMessageSeverity.Error,
                Description = @"The context period end date is incorrect.",
                LongDescription = @"Period end date within context on the schedule must match period end date within the parent return reporting party context",
                Location = $"/xbrli:xbrl/xbrli:context[{context.Index}]/xbrli:period/xbrli:endDate",
                Parameters = new ProcessMessageParameters {
                    new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438001" },
                    new ProcessMessageParameter { Name = "FITR301", Value = FITR2026Validator.GetValueOrEmptyDate(ParentDocument.FITR301) },
                    new ProcessMessageParameter { Name = "CGTS End Date", Value = FITR2026Validator.GetValueOrEmptyDate(context.EndDate) }
                }
            });

            Response.AddRange(processMessages);
        }

        #endregion // VR.ATO.CGTS.438001
    }
}