using Ato.CD.Inbound.Shared;
using Ato.EN.IntegrationServices.CodeGenerationFITR;
using Ato.EN.IntegrationServices.CodeGenerationRPTTAXPOS;
using DataContracts;
using System.Collections.Generic;
using System.Linq;
using System.Runtime.CompilerServices;
using VaTS;

namespace Ato.CD.Inbound.FITR202602
{
    public class CrossFormValidatorRPTTAXPOS : ICrossFormValidator
    {
        private FITR2026 ParentDocument { get; }
        private RPTTAXPOS2019 ChildDocument { get; }
        public List<ProcessMessageDocument> Response { get; private set; }

        public CrossFormValidatorRPTTAXPOS(FITR2026 report, BusinessDocument childDocument)
        {
            ParentDocument = report;
            ChildDocument = (RPTTAXPOS2019)childDocument.ConsumedReport;
            Response = new List<ProcessMessageDocument>();
        }

        public List<ProcessMessageDocument> ValidateCrossFormRules()
        {
            //Parent form cross form rules
            VRATOFITR434620();
            VRATOFITR434621();
            VRATOFITR434624();
            VRATOFITR434625();

            return Response;
        }

        #region VR.ATO.FITR.434620

        /*  VR.ATO.FITR.434620
        The first four digits of RTP category number must refer to your current financial year

        Technical Business Rule:
        CountDocument('RPTTAXPOS') = 1 AND Substring(RPTTAXPOS:bafot.02.36:RegulatoryDisclosures.ReportableTaxPositionCategoryReferenceNumber.Identifier,1,4,) <> ^FITR1

        Data Elements:

        ^FITR1 = FITR:RP:Report.TargetFinancial.Year

        ^RPTTAXPOS4 = RPTTAXPOS:RP:ReportableTaxPositionCategoryAAndB:RegulatoryDisclosures.ReportableTaxPositionCategoryReferenceNumber.Identifier

        ^RPTTAXPOS11 = RPTTAXPOS:RP:ReportableTaxPositionCategoryC:RegulatoryDisclosures.ReportableTaxPositionCategoryReferenceNumber.Identifier
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFITR434620()
        {
            //Check RPTTAXPOS4
            if (ChildDocument.ReportableTaxPositionCategoryAAndBCollection != null)
                Response.AddRange(from rptCategoryAAndB in ChildDocument.ReportableTaxPositionCategoryAAndBCollection
                                  where rptCategoryAAndB.RPTTAXPOS4 != null && rptCategoryAAndB.RPTTAXPOS4.Length == 8 && rptCategoryAAndB.RPTTAXPOS4.Substring(0, 4).ToString() != ParentDocument.FITR1.GetValueOrDefault().ToString()
                                  select new ProcessMessageDocument
                                  {
                                      Code = "CMN.ATO.FITR.434620",
                                      Severity = ProcessMessageSeverity.Error,
                                      Description = @"The first four digits of RTP category number must refer to your current financial year",
                                      Location = "/tns:FITR/tns:RP/tns:TargetFinancialY",
                                      Parameters = new ProcessMessageParameters
                                      {
                                          new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.FITR.434620" },
                                          new ProcessMessageParameter { Name = "FITR1", Value = FITR2026Validator.GetValueOrEmpty(ParentDocument.FITR1) },
                                          new ProcessMessageParameter { Name = "RPTTAXPOS4", Value = rptCategoryAAndB.RPTTAXPOS4 }
                                      },
                                      DocumentSequence = "parent_doc"
                                  });

            //Check RPTTAXPOS11
            if (ChildDocument.ReportableTaxPositionCategoryCCollection != null)
                Response.AddRange(from rptCategoryC in ChildDocument.ReportableTaxPositionCategoryCCollection
                                  where rptCategoryC.RPTTAXPOS11 != null && rptCategoryC.RPTTAXPOS11.Length == 8 && rptCategoryC.RPTTAXPOS11.Substring(0, 4).ToString() != ParentDocument.FITR1.GetValueOrDefault().ToString()
                                  select new ProcessMessageDocument
                                  {
                                      Code = "CMN.ATO.FITR.434620",
                                      Severity = ProcessMessageSeverity.Error,
                                      Description = @"The first four digits of RTP category number must refer to your current financial year",
                                      Location = "/tns:FITR/tns:RP/tns:TargetFinancialY",
                                      Parameters = new ProcessMessageParameters
                                      {
                                          new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.FITR.434620" },
                                          new ProcessMessageParameter { Name = "FITR1", Value = FITR2026Validator.GetValueOrEmpty(ParentDocument.FITR1) },
                                          new ProcessMessageParameter { Name = "RPTTAXPOS11", Value = rptCategoryC.RPTTAXPOS11 }
                                      },
                                      DocumentSequence = "parent_doc"
                                  });
        }

        #endregion // VR.ATO.FITR.434620

        #region VR.ATO.FITR.434621

        /*  VR.ATO.FITR.434621
        The Tax File Number (TFN) provided for the reportable tax position schedule must be the same as the TFN of the parent form

        Technical Business Rule:
        CountDocument('RPTTAXPOS') = 1 AND ^FITR5 <> RPTTAXPOS:^RP:TFN

        Data Elements:

        ^FITR5 = FITR:RP:Identifiers.TaxFileNumber.Identifier

        ^RPTTAXPOS TFN  = RPTTAXPOS:RP:entity.identifier.TFN
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFITR434621()
        {
            ProcessMessageDocument processMessage;
            bool assertion = !ParentDocument.FITR5.Equals(ChildDocument.RPIdentifierTFN);

            if (assertion)
            {
                processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.402009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your supplied TFN does not match the TFN supplied on the form it was submitted with",
                    LongDescription = @"Your supplied TFN does not match the TFN supplied on the parent form it was submitted with",
                    Location = "/tns:FITR/tns:RP/tns:TaxFileNumberId",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.FITR.434621" },
                        new ProcessMessageParameter { Name = "FITR5", Value = ParentDocument.FITR5 },
                        new ProcessMessageParameter { Name = "RPTTAXPOS TFN", Value = ChildDocument.RPIdentifierTFN }
                    },
                    DocumentSequence = "parent_doc"
                };
             
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.FITR.434621

        #region VR.ATO.FITR.434624

        /*  VR.ATO.FITR.434624
        The start date in reportable tax position schedule must be the same as the start date provided in its parent form

        Technical Business Rule:
        CountDocument('RPTTAXPOS') = 1 AND ^FITR300 <> ^RPTTAXPOS:RP:period.startDate

        Data Elements:

        ^FITR300 = FITR:RP:Period.Start.Date

        ^RPTTAXPOS Start Date = RPTTAXPOS Context Start Date
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFITR434624()
        {
            ProcessMessageDocument processMessage;
            bool assertion = !ParentDocument.FITR300.Equals(ChildDocument.RPStartDate);

            if (assertion)
            {
                processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.FITR.434624",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The start date in reportable tax position schedule must be the same as the start date provided in its parent form",
                    Location = "/tns:FITR/tns:RP/tns:PeriodStartD",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.FITR.434624" },
                        new ProcessMessageParameter { Name = "FITR300", Value = FITR2026Validator.GetValueOrEmptyDate(ParentDocument.FITR300) },
                        new ProcessMessageParameter { Name = "RPTTAXPOS Start Date", Value = FITR2026Validator.GetValueOrEmptyDate(ChildDocument.RPStartDate) }
                    },
                    DocumentSequence = "parent_doc"
                };

                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.FITR.434624

        #region VR.ATO.FITR.434625

        /* VR.ATO.FITR.434625
        The end date in reportable tax position schedule must be the same as the end date provided in its parent form

        Technical Business Rule:
        CountDocument('RPTTAXPOS') = 1 AND ^FITR301 <> ^RPTTAXPOS:RP:period.endDate

        Data Elements:

        ^FITR301 = FITR:RP:Period.End.Date

        ^RPTTAXPOS End Date = RPTTAXPOS Context End Date
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOFITR434625()
        {
            ProcessMessageDocument processMessage;
            bool assertion = !ParentDocument.FITR301.Equals(ChildDocument.RPEndDate);

            if (assertion)
            {
                processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.FITR.434625",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The end date in reportable tax position schedule must be the same as the end date provided in its parent form",
                    Location = "/tns:FITR/tns:RP/tns:PeriodEndD",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.FITR.434625" },
                        new ProcessMessageParameter { Name = "FITR301", Value = FITR2026Validator.GetValueOrEmptyDate(ParentDocument.FITR301) },
                        new ProcessMessageParameter { Name = "RPTTAXPOS End Date", Value = FITR2026Validator.GetValueOrEmptyDate(ChildDocument.RPEndDate) }
                    },
                    DocumentSequence = "parent_doc"
                };

                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.FITR.434625
    }
}
