using Ato.CD.Inbound.Shared;
using Ato.EN.IntegrationServices.CodeGenerationPTR;
using Ato.EN.IntegrationServices.CodeGenerationRS;
using DataContracts;
using System;
using System.Collections.Generic;
using VaTS;
using static Ato.CD.Inbound.Shared.ValidatorHelpers;

namespace Ato.CD.Inbound.PTR202602
{
    public class CrossFormValidatorRS : ICrossFormValidator
    {
        private PTR2025 ParentDocument { get; }

        private RS2018 ChildDocument { get; }

        public List<ProcessMessageDocument> Response { get; private set; }

        public CrossFormValidatorRS(PTR2025 report, BusinessDocument childDocument)
        {
            ParentDocument = report;
            ChildDocument = (RS2018)childDocument.ConsumedReport;
            Response = new List<ProcessMessageDocument>();
        }

        public List<ProcessMessageDocument> ValidateCrossFormRules()
        {
            VRATORS000137();
            VRATORS000138();
            VRATORS000150();
            VRATOGEN402009();

            return Response;
        }

        #region VR.ATO.RS.000137
        /* 
            IF (PARENT RETURN <> "IITR") AND ([RS25] > 52)  
                RETURN VALIDATION MESSAGE
            ENDIF

            ^RS25 = /xbrli:xbrl/tns:RealEstateProperty.RentalWeeks.Count
        */
        public void VRATORS000137()
        {
            bool assertion = ChildDocument.RS25.GetValueOrDefault() > 52;

            if (assertion)
            {
                Response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.RS.000137",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Number of weeks property rented must not exceed 52",
                    Location = "/xbrli:xbrl/tns:RealEstateProperty.RentalWeeks.Count",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.RS.000137" },
                        new ProcessMessageParameter { Name = "RS25", Value = GetValueOrEmpty(ChildDocument.RS25) }
                    }
                });
            }
        }
        #endregion

        #region VR.ATO.RS.000138
        /* 
            IF (PARENT RETURN <> "IITR") AND ([RS27] > 52)  
                RETURN VALIDATION MESSAGE
            ENDIF

            ^RS27 = /xbrli:xbrl/tns:RealEstateProperty.RentalWeeksAvailable.Count
        */
        public void VRATORS000138()
        {
            bool assertion = ChildDocument.RS27.GetValueOrDefault() > 52;

            if (assertion)
            {
                Response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.RS.000138",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Number of weeks property was available for rent this year cannot exceed 52 weeks",
                    Location = "/xbrli:xbrl/tns:RealEstateProperty.RentalWeeksAvailable.Count",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.RS.000138" },
                        new ProcessMessageParameter { Name = "RS27", Value = GetValueOrEmpty(ChildDocument.RS27) }
                    }
                });
            }
        }
        #endregion

        #region VR.ATO.GEN.402009
        /* 
            IF (RP:entity.identifier.TFN <> PARENT RETURN:RP:entity.identifier.TFN)
                RETURN VALIDATION MESSAGE
            ENDIF
            
            ^PTR5 = /tns:PTR/tns:RP/tns:TaxFileNumberId
            RS:RP:entity.identifier.TFN = /xbrli:xbrl/xbrli:context/xbrli:entity/xbrli:identifier
        */
        public void VRATOGEN402009()
        {
            bool assertion = !ParentDocument.PTR5.Equals(ChildDocument.RPIdentifierTFN);

            if (assertion)
            {
                Response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.402009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Your supplied TFN does not match the TFN supplied on the form it was submitted with",
                    LongDescription = "Your supplied TFN does not match the TFN supplied on the parent form it was submitted with",
                    Location = "/xbrli:xbrl/xbrli:context/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.GEN.402009" },
                        new ProcessMessageParameter { Name = "PTR5", Value = ParentDocument.PTR5 },
                        new ProcessMessageParameter { Name = "RP:entity.identifier.TFN", Value = ChildDocument.RPIdentifierTFN }
                    }
                });
            }
        }
        #endregion

        #region VR.ATO.RS.000150
        /* 
            IF (PARENT RETURN = "IITR" AND [RS14] <> [IITR425]) OR (PARENT RETURN = "PTR" AND [RS14] <> [PTR6]) OR (PARENT RETURN = "TRT" AND [RS14] <> [TRT6]) OR (PARENT RETURN = "TRTAMI" AND [RS14] <> [TRTAMI10])
               RETURN VALIDATION MESSAGE
            ENDIF

            ^PTR6 = /tns:PTR/tns:RP/tns:CurrentPartnershipNameOrganisationNameDetailsOrganisationalNameT
            ^RS14 = /xbrli:xbrl/tns:OrganisationNameDetails.OrganisationalName.Text
        */
        public void VRATORS000150()
        {
            bool assertion = !ParentDocument.PTR6.Trim().Equals(ChildDocument.RS14.Trim(), StringComparison.CurrentCultureIgnoreCase);

            if (assertion)
            {
                Response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.RS.000150",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "The Entity name must be the same as the name on the main form",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.RS.000150" },
                        new ProcessMessageParameter { Name = "PTR6", Value = GetValueOrEmpty(ParentDocument.PTR6) },
                        new ProcessMessageParameter { Name = "RS14", Value = GetValueOrEmpty(ChildDocument.RS14) }
                    }
                });
            }
        }
        #endregion
    }
}