using Ato.CD.Inbound.Shared;
using Ato.CD.Inbound.Shared201802;
using Ato.EN.IntegrationServices.CodeGenerationCGTS;
using Ato.EN.IntegrationServices.CodeGenerationSMSFAR;
using VaTS;
using DataContracts;
using System.Collections.Generic;
using System.Linq;

namespace Ato.CD.Inbound.SMSFAR
{
    internal class CrossFormValidatorCGTS : ICrossFormValidator
    {
        private SMSFAR2026 ParentDocument { get; }

        private CGTS2018 ChildDocument { get; }

        private List<Context> Contexts { get; }

        internal CrossFormValidatorCGTS(SMSFAR2026 report, BusinessDocument childDocument)
        {
            ParentDocument = report;
            ChildDocument = (CGTS2018)childDocument.ConsumedReport;
            Contexts = ChildDocument.GetContexts();
        }

        public List<ProcessMessageDocument> ValidateCrossFormRules()
        {
            var response = new List<ProcessMessageDocument>();

            VRATOGEN402009(response);
            VRATOGEN438000(response);
            VRATOGEN438001(response);
            VRATOGEN438022(response);
            VRATOGEN438023(response);
            VRATOGEN438024(response);
            VRATOGEN438025(response);
            VRATOGEN438026(response);
            VRATOGEN438027(response);
            VRATOCGTS402165(response);
            VRATOCGTS402171(response);
            VRATOCGTS402178(response);
            VRATOCGTS402226(response);
            VRATOCGTS402230(response);
            VRATOCGTS402240(response);
            VRATOSMSFAR436574(response);
            VRATOSMSFAR436802(response);

            return response;
        }

        #region VR.ATO.GEN.402009
        /* VR.ATO.GEN.402009
            IF (RP:entity.identifier.TFN <> PARENT RETURN:RP:entity.identifier.TFN)
                RETURN VALIDATION MESSAGE
            ENDIF 
        */
        private void VRATOGEN402009(List<ProcessMessageDocument> response)
        {
            bool assertion = !ParentDocument.RPIdentifierTFN.Equals(ChildDocument.RPIdentifierTFN);

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.402009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Your supplied TFN does not match the TFN supplied on the form it was submitted with",
                    LongDescription = "Your supplied TFN does not match the TFN supplied on the parent form it was submitted with",
                    Location = "/xbrli:xbrl/xbrli:context/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters { new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.GEN.402009" } }
                });
            }
        }
        #endregion

        #region VR.ATO.GEN.438000
        /*  VR.ATO.GEN.438000
            IF (period.startDate WHERE CONTEXT(ALL)) <> PARENT RETURN:RP:period.startDate
               RETURN VALIDATION MESSAGE
            ENDIF
        */
        private void VRATOGEN438000(List<ProcessMessageDocument> response)
        {
            response.AddRange(from context in Contexts
                              where context.StartDate.GetValueOrDefault() != ParentDocument.RPStartDate.GetValueOrDefault()
                              select new ProcessMessageDocument
                              {
                                  Code = "CMN.ATO.GEN.438000",
                                  Severity = ProcessMessageSeverity.Error,
                                  Description = "The context period start date is incorrect.",
                                  LongDescription = "Period start date within context on the schedule must match period start date within the parent return reporting party context",
                                  Location = $"/xbrli:xbrl/xbrli:context[{context.Index}]/xbrli:period/xbrli:startDate",
                                  Parameters = new ProcessMessageParameters { new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438000" } }
                              });
        }
        #endregion // VR.ATO.GEN.438000

        #region VR.ATO.GEN.438001
        /*  VR.ATO.CGTS.438001
            IF (period.endDate WHERE CONTEXT(ALL)) <> PARENT RETURN:RP:period.endDate
               RETURN VALIDATION MESSAGE
            ENDIF
        */
        private void VRATOGEN438001(List<ProcessMessageDocument> response)
        {
            response.AddRange(from context in Contexts
                              where context.EndDate.GetValueOrDefault() != ParentDocument.RPEndDate.GetValueOrDefault()
                              select new ProcessMessageDocument
                              {
                                  Code = "CMN.ATO.GEN.438001",
                                  Severity = ProcessMessageSeverity.Error,
                                  Description = "The context period end date is incorrect.",
                                  LongDescription = "Period end date within context on the schedule must match period end date within the parent return reporting party context",
                                  Location = $"/xbrli:xbrl/xbrli:context[{context.Index}]/xbrli:period/xbrli:endDate",
                                  Parameters = new ProcessMessageParameters { new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438001" } }
                              });
        }
        #endregion

        #region VR.ATO.GEN.438022
        /*  VR.ATO.GEN.438022
            If taxpayer is required to lodge a CGT schedule and CGT exemption or rollover applied code is "Small business active asset reduction (Subdivision 152-C)", then Small business active asset reduction amount on attached CGT schedule must be completed

            Legacy Rule Format:
            IF (RP:rvctc3.xx.xx:CapitalGainsTax.ExemptionOrRolloverApplied.Code = "A - Small business active asset reduction") 
            AND (COUNT(SCHEDULE = "CGTS") = 1) AND [CGTS127] = NULL
                RETURN VALIDATION MESSAGE
            ENDIF
                    
            [CGTS127] = CGTS:RP:tns:TaxConcession.CapitalGains.SmallBusinessActiveAssetReduction.Amount

            Technical Business Rule Format:
            (^SMSFAR328 = 'A - Small business active asset reduction') AND (CountDocument('CGTS') = 1) AND ^CGTS127 = NULL

            Data Elements:
            
            RP:^SMSFAR328 = tns:CapitalGainsTax.ExemptionOrRolloverApplied.Code
            
            RP:^CGTS127 = CGTS:RP:tns:TaxConcession.CapitalGains.SmallBusinessActiveAssetReduction.Amount
        */
        private void VRATOGEN438022(List<ProcessMessageDocument> response)
        {
            bool assertion = ParentDocument.SMSFAR328Collection != null &&
                             ParentDocument.SMSFAR328Collection.Any(
                                 s => s.Equals("A - Small business active asset reduction")) &&
                             !ChildDocument.CGTS127.HasValue;
            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.438022",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Small business active asset reduction amount missing from CGT schedule",
                    LongDescription = @"If taxpayer is required to lodge a CGT schedule and CGT exemption or rollover applied code is ""Small business active asset reduction (Subdivision 152-C)"", then Small business active asset reduction amount on attached CGT schedule must be completed",
                    Location = "/xbrli:xbrl/tns:CapitalGainsTax.ExemptionOrRolloverApplied.Code",
                    Parameters = new ProcessMessageParameters { new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438022" } },
                    DocumentSequence = "parent_doc"
                });
            }
        }
        #endregion

        #region VR.ATO.GEN.438023
        /*  VR.ATO.GEN.438023
            If taxpayer is required to lodge a CGT schedule and CGT exemption or rollover applied code is "Small business retirement exemption (Subdivision152-D)", then Small business retirement exemption amount on attached CGT schedule must be completed

            Legacy Rule Format:
            IF (RP:rvctc3.xx.xx:CapitalGainsTax.ExemptionOrRolloverApplied.Code = "B - Small business retirement exemption") AND (COUNT(SCHEDULE = "CGTS") = 1) AND [CGTS128] = NULL
                RETURN VALIDATION MESSAGE
            ENDIF
                    
            [CGTS128] = CGTS:RP:tns:TaxConcession.CapitalGains.SmallBusinessRetirementExemption.Amount

            Technical Business Rule Format:
            (^SMSFAR328 = 'B - Small business retirement exemption') AND (CountDocument('CGTS') = 1) AND ^CGTS128 = NULL

            Data Elements:
            
            RP:^SMSFAR328 = tns:CapitalGainsTax.ExemptionOrRolloverApplied.Code
            
            RP:^CGTS128 = CGTS:RP:tns:TaxConcession.CapitalGains.SmallBusinessRetirementExemption.Amount
        */
        private void VRATOGEN438023(List<ProcessMessageDocument> response)
        {
            bool assertion = ParentDocument.SMSFAR328Collection != null &&
                             ParentDocument.SMSFAR328Collection.Any(
                                 s => s.Equals("B - Small business retirement exemption")) &&
                             !ChildDocument.CGTS128.HasValue;
            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.438023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Small business retirement exemption amount missing from CGT schedule",
                    LongDescription = @"If taxpayer is required to lodge a CGT schedule and CGT exemption or rollover applied code is ""Small business retirement exemption (Subdivision152-D)"", then Small business retirement exemption amount on attached CGT schedule must be completed",
                    Location = "/xbrli:xbrl/tns:CapitalGainsTax.ExemptionOrRolloverApplied.Code",
                    Parameters = new ProcessMessageParameters { new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438023" } },
                    DocumentSequence = "parent_doc"
                });
            }
        }
        #endregion

        #region VR.ATO.GEN.438024
        /*  VR.ATO.GEN.438024
            If taxpayer is required to lodge a CGT schedule and CGT exemption or rollover applied code is "Small business roll-over (Subdivision 152-E)", then Small business roll-over amount on attached CGT schedule must be completed

            Legacy Rule Format:
            IF (RP:rvctc3.xx.xx:CapitalGainsTax.ExemptionOrRolloverApplied.Code = "C - Small business roll-over") AND (COUNT(SCHEDULE = "CGTS") = 1) AND [CGTS129] = NULL
                RETURN VALIDATION MESSAGE
            ENDIF
                    
            [CGTS129] = CGTS:RP:tns:TaxConcession.CapitalGains.SmallBusinessRollover.Amount

            Technical Business Rule Format:
            (^SMSFAR328 = 'C - Small business roll-over') AND (CountDocument('CGTS') = 1) AND ^CGTS129 = NULL

            Data Elements:
            
            RP:^SMSFAR328 = tns:CapitalGainsTax.ExemptionOrRolloverApplied.Code
            
            RP:^CGTS129 = CGTS:RP:tns:TaxConcession.CapitalGains.SmallBusinessRollover.Amount
        */
        private void VRATOGEN438024(List<ProcessMessageDocument> response)
        {
            bool assertion = ParentDocument.SMSFAR328Collection != null &&
                             ParentDocument.SMSFAR328Collection.Any(
                                 s => s.Equals("C - Small business roll-over")) &&
                             !ChildDocument.CGTS129.HasValue;
            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.438024",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Small business roll-over amount missing from CGT schedule",
                    LongDescription = @"If taxpayer is required to lodge a CGT schedule and CGT exemption or rollover applied code is ""Small business roll-over (Subdivision 152-E)"", then Small business roll-over amount on attached CGT schedule must be completed",
                    Location = "/xbrli:xbrl/tns:CapitalGainsTax.ExemptionOrRolloverApplied.Code",
                    Parameters = new ProcessMessageParameters { new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438024" } },
                    DocumentSequence = "parent_doc"
                });
            }
        }
        #endregion

        #region VR.ATO.GEN.438025
        /*  VR.ATO.GEN.438025
            If taxpayer is required to lodge a CGT schedule and CGT exemption or rollover applied code is "Small business 15 year exemption (Subdivision152-B)", then Small business 15 year exemption amount on attached CGT schedule must be completed

            Legacy Rule Format:
            IF (RP:rvctc3.xx.xx:CapitalGainsTax.ExemptionOrRolloverApplied.Code = "D - Small business 15 year exemption") AND (COUNT(SCHEDULE = "CGTS") = 1) AND [CGTS138] = NULL
                RETURN VALIDATION MESSAGE
            ENDIF
                    
            [CGTS138] = CGTS:RP:tns:TaxConcession.CapitalGains.SmallBusinessExemptionExempt.Amount

            Technical Business Rule Format:
            (^SMSFAR328 = 'D - Small business 15 year exemption') AND (CountDocument('CGTS') = 1) AND ^CGTS138 = NULL

            Data Elements:
            
            RP:^SMSFAR328 = tns:CapitalGainsTax.ExemptionOrRolloverApplied.Code
            
            RP:^CGTS138 = CGTS:RP:tns:TaxConcession.CapitalGains.SmallBusinessExemptionExempt.Amount
        */
        private void VRATOGEN438025(List<ProcessMessageDocument> response)
        {
            bool assertion = ParentDocument.SMSFAR328Collection != null &&
                             ParentDocument.SMSFAR328Collection.Any(
                                 s => s.Equals("D - Small business 15 year exemption")) &&
                             !ChildDocument.CGTS138.HasValue;
            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.438025",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Small business 15 year exemption amount missing from CGT schedule",
                    LongDescription = @"If taxpayer is required to lodge a CGT schedule and CGT exemption or rollover applied code is ""Small business 15 year exemption (Subdivision152-B)"", then Small business 15 year exemption amount on attached CGT schedule must be completed",
                    Location = "/xbrli:xbrl/tns:CapitalGainsTax.ExemptionOrRolloverApplied.Code",
                    Parameters = new ProcessMessageParameters { new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438025" } },
                    DocumentSequence = "parent_doc"
                });
            }
        }
        #endregion

        #region VR.ATO.GEN.438026
        /*  VR.ATO.GEN.438026
            If taxpayer is required to lodge a CGT schedule and CGT exemption or rollover applied code is "Foreign resident CGT exemption (Division 855)", then Capital gains disregarded by a foreign resident amount on attached CGT schedule must be completed

            Legacy Rule Format:
            IF (RP:rvctc3.xx.xx:CapitalGainsTax.ExemptionOrRolloverApplied.Code = "E - Foreign resident CGT exemption") AND (COUNT(SCHEDULE = "CGTS") = 1) AND [CGTS140] = NULL
                RETURN VALIDATION MESSAGE
            ENDIF
                    
            [CGTS140] = CGTS:RP:tns:Capital.Gains.ForeignResidentExemption.Amount

            Technical Business Rule Format:
            (^SMSFAR328 = 'E - Foreign resident CGT exemption') AND (CountDocument('CGTS') = 1) AND ^CGTS140 = NULL

            Data Elements:
            
            RP:^SMSFAR328 = tns:CapitalGainsTax.ExemptionOrRolloverApplied.Code
            
            RP:^CGTS140 = CGTS:RP:tns:Capital.Gains.ForeignResidentExemption.Amount
        */
        private void VRATOGEN438026(List<ProcessMessageDocument> response)
        {
            bool assertion = ParentDocument.SMSFAR328Collection != null &&
                             ParentDocument.SMSFAR328Collection.Any(
                                 s => s.Equals("E - Foreign resident CGT exemption")) &&
                             !ChildDocument.CGTS140.HasValue;
            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.438026",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Capital gains disregarded by a foreign resident amount missing from CGT schedule",
                    LongDescription = @"If taxpayer is required to lodge a CGT schedule and CGT exemption or rollover applied code is ""Foreign resident CGT exemption (Division 855)"", then Capital gains disregarded by a foreign resident amount on attached CGT schedule must be completed",
                    Location = "/xbrli:xbrl/tns:CapitalGainsTax.ExemptionOrRolloverApplied.Code",
                    Parameters = new ProcessMessageParameters { new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438026" } },
                    DocumentSequence = "parent_doc"
                });
            }
        }
        #endregion

        #region VR.ATO.GEN.438027
        /*  VR.ATO.GEN.438027
            If taxpayer is required to lodge a CGT schedule and CGT exemption or rollover applied code is "Scrip for scrip roll-over (Subdivision 124-M)", then Capital gains disregarded as a result of a scrip for scrip roll-over amount on attached CGT schedule must be completed

            Legacy Rule Format:
            IF (RP:rvctc3.xx.xx:CapitalGainsTax.ExemptionOrRolloverApplied.Code = "F - Scrip for scrip roll-over") AND (COUNT(SCHEDULE = "CGTS") = 1) AND [CGTS45] = NULL
                RETURN VALIDATION MESSAGE
            ENDIF
                    
            [CGTS45] = CGTS:RP:tns:Capital.Gains.ScriptForScriptRolloverDeferral.Amount

            Technical Business Rule Format:
            (^SMSFAR328 = 'F - Scrip for scrip roll-over') AND (CountDocument('CGTS') = 1) AND ^CGTS45 = NULL

            Data Elements:
            
            RP:^SMSFAR328 = tns:CapitalGainsTax.ExemptionOrRolloverApplied.Code
            
            RP:^CGTS45 = CGTS:RP:tns:Capital.Gains.ScriptForScriptRolloverDeferral.Amount
        */
        private void VRATOGEN438027(List<ProcessMessageDocument> response)
        {
            bool assertion = ParentDocument.SMSFAR328Collection != null &&
                             ParentDocument.SMSFAR328Collection.Any(
                                 s => s.Equals("F - Scrip for scrip roll-over")) &&
                             !ChildDocument.CGTS45.HasValue;
            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.438027",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Capital gains disregarded as a result of a scrip for scrip roll-over amount missing from CGT schedule",
                    LongDescription = @"If taxpayer is required to lodge a CGT schedule and CGT exemption or rollover applied code is ""Scrip for scrip roll-over (Subdivision 124-M)"", then Capital gains disregarded as a result of a scrip for scrip roll-over amount on attached CGT schedule must be completed",
                    Location = "/xbrli:xbrl/tns:CapitalGainsTax.ExemptionOrRolloverApplied.Code",
                    Parameters = new ProcessMessageParameters { new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438027" } },
                    DocumentSequence = "parent_doc"
                });
            }
        }
        #endregion

        #region VR.ATO.CGTS.402165
        /* VR.ATO.CGTS.402165

             IF (PARENT RETURN <> "CTR") AND ([CGTS122] <> NULL)  
               RETURN VALIDATION MESSAGE
             ENDIF

            [CGTS122] = CGTS:RP.Applied:tns:Capital.Losses.TransferredTotal.Amount
    
        */
        private void VRATOCGTS402165(List<ProcessMessageDocument> response)
        {
            bool assertion = ChildDocument.CGTS122.HasValue;

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.CGTS.402165",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Total losses transferred in applied is only valid for a company",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.TransferredTotal.Amount",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402165" },
                        new ProcessMessageParameter { Name = "CGTS122", Value = ChildDocument.CGTS122.Value.ToString() }
                    }
                });
            }
        }
        #endregion // VR.ATO.CGTS.402165

        #region VR.ATO.CGTS.402171
        /* VR.ATO.CGTS.402171

            IF PARENT RETURN = "SMSFAR","FITR","TRT" AND([CGTS131] > 0) AND([CGTS125] > 0)
                RETURN VALIDATION MESSAGE
            ENDIF

            [CGTS125] = CGTS:RP.CGTOther.Unapplied:rvctc3.02.16:Capital.Losses.CarriedForward.Net.Amount
            [CGTS131] = CGTS:RP:tns:Income.CapitalGainsNet.Amount        
    
        */
        private void VRATOCGTS402171(List<ProcessMessageDocument> response)
        {
            bool assertion = ChildDocument.CGTS131.GetValueOrDefault() > 0 && ChildDocument.CGTS125.GetValueOrDefault() > 0;

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.CGTS.402171",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "A capital loss must not be carried forward when a positive net capital gain amount has been provided",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.CarriedForward.Net.Amount",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402171" },
                        new ProcessMessageParameter { Name = "CGTS131", Value = ChildDocument.CGTS131.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter { Name = "CGTS125", Value = ChildDocument.CGTS125.GetValueOrDefault().ToString() }
                    }
                });
            }
        }
        #endregion

        #region VR.ATO.CGTS.402178
        /* VR.ATO.CGTS.402178

        IF PARENT RETURN = "SMSFAR","FITR" AND
            ([CGTS128] <> NULL)
               RETURN VALIDATION MESSAGE
            ENDIF

            [CGTS128] = CGTS:RP:tns:TaxConcession.CapitalGains.SmallBusinessRetirementExemption.Amount   
        */
        private void VRATOCGTS402178(List<ProcessMessageDocument> response)
        {
            bool assertion = ChildDocument.CGTS128.HasValue;

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.CGTS.402178",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Small business retirement exemption is not valid for Self Managed Super Funds",
                    Location = "/xbrli:xbrl/tns:TaxConcession.CapitalGains.SmallBusinessRetirementExemption.Amount",
                    Parameters = new ProcessMessageParameters { new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402178" } }
                });
            }
        }
        #endregion // VR.ATO.CGTS.402178

        #region VR.ATO.CGTS.402226
        /*  VR.ATO.CGTS.402226
          
            IF (PARENT RETURN:RP:pyin.xx.xx:Report.Amendment.Indicator = TRUE) AND ([CGTS150] <> NULL AND [CGTS151] <> NULL)
               RETURN VALIDATION MESSAGE
            ENDIF

            [CGTS150] = CGTS:RP:tns:Report.TargetFinancial.Year
            [CGTS151] = CGTS:RP:tns:Miscellaneous.RequestedAmendment.Amount
           
        */
        private void VRATOCGTS402226(List<ProcessMessageDocument> response)
        {
            bool assertion = ParentDocument.SMSFAR186.GetValueOrDefault() && ChildDocument.CGTS150.HasValue && ChildDocument.CGTS151.HasValue;

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.CGTS.402226",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Income year earnout right created and Amount to be amended cannot be supplied when Parent return Amendment indicator is present.",
                    Location = "/xbrli:xbrl/tns:Report.TargetFinancial.Year",
                    Parameters = new ProcessMessageParameters { new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402226" } }
                });
            }
        }
        #endregion

        #region VR.ATO.CGTS.402230
        /*  VR.ATO.CGTS.402230
          
          IF PARENT RETURN  <> "TRTAMI" AND ([CGTS153] <> NULL OR [CGTS154] <> NULL OR [CGTS155] <> NULL)
            RETURN VALIDATION MESSAGE
          ENDIF
           
        */
        private void VRATOCGTS402230(List<ProcessMessageDocument> response)
        {
            bool assertion = !string.IsNullOrEmpty(ChildDocument.CGTS153) || ChildDocument.CGTS154.HasValue || ChildDocument.CGTS155.HasValue;

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.CGTS.402230",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Multi-class treatment fields incorrectly supplied",
                    Location = "/xbrli:xbrl/tns:IncomeTax.AttributionManagedInvestmentTrustClass.Description",
                    Parameters = new ProcessMessageParameters { new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402230" } }
                });
            }
        }
        #endregion

        #region VR.ATO.CGTS.402240
        /* VR.ATO.CGTS.402240

            IF([CGTS150] <> NULL) AND([CGTS150] = (PARENT RETURN:RP:pyin.xx.xx:Report.TargetFinancial.Year))
               RETURN VALIDATION MESSAGE
            ENDIF

            [CGTS150] = CGTS:RP:tns:Report.TargetFinancial.Year         
        */
        private void VRATOCGTS402240(List<ProcessMessageDocument> response)
        {
            bool assertion = ChildDocument.CGTS150.HasValue && ChildDocument.CGTS150 == ParentDocument.SMSFAR1.GetValueOrDefault();

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.CGTS.402240",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Income year earnout right created incorrect",
                    Location = "/xbrli:xbrl/tns:Report.TargetFinancial.Year",
                    Parameters = new ProcessMessageParameters { new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402240" } }
                });
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436574
        /*  VR.ATO.SMSFAR.436574
            Net capital gain in return must equal net capital gain in CGT schedule

            Legacy Rule Format:
            IF (COUNT(SCHEDULE = "CGTS") = 1) AND ([SMSFAR33] <> [CGTS131])
                RETURN VALIDATION MESSAGE
            ENDIF
                    
            [CGTS131] = CGTS:RP:tns:Income.CapitalGainsNet.Amount
            [SMSFAR33] = SMSFAR2019:RP:tns:Income.CapitalGainsNet.Amount

            Technical Business Rule Format:
            (CountDocument('CGTS') = 1) AND (^SMSFAR33 <> ^CGTS131)

            Data Elements:
            
            RP:^SMSFAR33 = tns:Income.CapitalGainsNet.Amount
            
            RP:^CGTS131 = CGTS:RP:tns:Income.CapitalGainsNet.Amount
        */
        private void VRATOSMSFAR436574(List<ProcessMessageDocument> response)
        {
            bool assertion = ParentDocument.SMSFAR33.HasValue && ChildDocument.CGTS131.HasValue &&
                             ParentDocument.SMSFAR33.Value != ChildDocument.CGTS131.Value;
            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.SMSFAR.436574",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Net capital gain in return must equal net capital gain in CGT schedule",
                    LongDescription = "Net capital gain amount on the company return  must equal the net capital gain amount on the CGT schedule",
                    Location = "/xbrli:xbrl/tns:Income.CapitalGainsNet.Amount",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436574" },
                        new ProcessMessageParameter { Name = "SMSFAR33", Value = ParentDocument.SMSFAR33.Value.ToString() },
                        new ProcessMessageParameter { Name = "CGTS131", Value = ChildDocument.CGTS131.Value.ToString() }
                    },
                    DocumentSequence = "parent_doc"
                });
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436802
        /*  VR.ATO.SMSFAR.436802
            The sum of unapplied net capital losses carried forward on the Capital Gains Tax schedule must equal net capital losses carried forward to later income years

            Legacy Rule Format:
            IF COUNT(SCHEDULE = "CGTS") = 1 AND ([CGTS124] + [CGTS125]) <> [SMSFAR380]  
                RETURN VALIDATION MESSAGE
            ENDIF
                    
            [CGTS124] = CGTS:RP.Collectables.Unapplied(Instant):tns:Capital.Losses.CarriedForward.Net.Amount
            [CGTS125] = CGTS:RP.CGTOther.Unapplied(Instant):tns:Capital.Losses.CarriedForward.Net.Amount
            [SMSFAR380] = SMSFAR2019:RP.Y0Plus.Closing(Instant):tns:Capital.Losses.CarriedForward.Net.Amount

            Technical Business Rule Format:
            CountDocument('CGTS') = 1 AND (^CGTS124 + ^CGTS125) <> ^SMSFAR380

            Data Elements:
            
            RP.Y0Plus.Closing(Instant):^SMSFAR380 = tns:Capital.Losses.CarriedForward.Net.Amount
            
            RP.Y0Plus.Closing(Instant):^CGTS124 = CGTS:RP.Collectables.Unapplied(Instant):tns:Capital.Losses.CarriedForward.Net.Amount
            
            RP.Y0Plus.Closing(Instant):^CGTS125 = CGTS:RP.CGTOther.Unapplied(Instant):tns:Capital.Losses.CarriedForward.Net.Amount
        */
        private void VRATOSMSFAR436802(List<ProcessMessageDocument> response)
        {
            bool assertion = ParentDocument.SMSFAR380.HasValue && ParentDocument.SMSFAR380.Value !=
                             ChildDocument.CGTS124.GetValueOrDefault() + ChildDocument.CGTS125.GetValueOrDefault();
            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.SMSFAR.436620",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Unapplied net capital losses carried forward on CGTS must equal net capital losses carried forward",
                    LongDescription = "The sum of unapplied net capital losses carried forward to later income years on the Capital Gains Tax schedule must equal net capital losses carried forward to later income years on parent return",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.CarriedForward.Net.Amount",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436802" },
                        new ProcessMessageParameter { Name = "SMSFAR380", Value = ParentDocument.SMSFAR380.Value.ToString() },
                        new ProcessMessageParameter { Name = "CGTS124", Value = ChildDocument.CGTS124.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter { Name = "CGTS125", Value = ChildDocument.CGTS125.GetValueOrDefault().ToString() }
                    },
                    DocumentSequence = "parent_doc"
                });
            }
        }
        #endregion
    }
}