using DataContracts;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Runtime.CompilerServices;
using System.Text.RegularExpressions;
using static Ato.CD.Inbound.Shared.ValidatorHelpers;

namespace Ato.EN.IntegrationServices.CodeGenerationSMSFAR
{
    public partial class SMSFAR2026Validator
    {
        /// <summary>
        /// The max parameter name length is restricted by the SBR1 and SBR2 schemas - the lowest common denominator of SBR 1 is used as the default
        /// </summary>
        private int _maxParameterNameLength;

        /// <summary>
        /// The max parameter value length is restricted by the SBR1 and SBR2 schemas
        /// </summary>
        private int _maxParameterValueLength;

        /// <summary>
        /// The SBR1 and ebms3 schemas do not allow parameter names or values to be the empty string
        /// </summary>
        private string _emptyParameterValue;

        /// <summary>
        /// The report parameter
        /// </summary>
        private SMSFAR2026 report;

        /// <summary>
        /// The response parameter
        /// </summary>
        public List<ProcessMessageDocument> response { get; private set; }

        /// <summary>
        /// Initializes a new instance of the <see cref="SMSFAR2026Validator" /> class.
        /// </summary>
        /// <param name="maxParameterNameLength">Maximum length of the parameter name.</param>
        /// <param name="maxParameterValueLength">Maximum length of the parameter value.</param>
        /// <param name="emptyParameterValue">This value will be used in place of any parameter values that result in a null or empty value.</param>
        public SMSFAR2026Validator(int maxParameterNameLength = 20, int maxParameterValueLength = 4096, string emptyParameterValue = "EMPTY")
        {
            _maxParameterNameLength = maxParameterNameLength;
            _maxParameterValueLength = maxParameterValueLength;
            _emptyParameterValue = emptyParameterValue;
            this.response = new List<ProcessMessageDocument>();
        }

        /// <summary>
        /// Initializes a new instance of the <see cref="SMSFAR2026Validator" /> class.
        /// </summary>
        /// <param name="report">Report variable </param>                
        /// <param name="maxParameterNameLength">Maximum length of the parameter name.</param>
        /// <param name="maxParameterValueLength">Maximum length of the parameter value.</param>
        /// <param name="emptyParameterValue">This value will be used in place of any parameter values that result in a null or empty value.</param>
        public SMSFAR2026Validator(SMSFAR2026 reportIn, int maxParameterNameLength = 20, int maxParameterValueLength = 4096, string emptyParameterValue = "EMPTY") : this(maxParameterNameLength, maxParameterValueLength, emptyParameterValue)
        {
            this.ConsumedReport = reportIn;
        }

        public SMSFAR2026 ConsumedReport { get { return report; } private set { report = value; } }

        private static Dictionary<string, ProcessMessageDocument> _processMessageDocuments = new Dictionary<string, ProcessMessageDocument>();

        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public List<ProcessMessageDocument> ValidateReport(SMSFAR2026 reportIn)
        {
            ProcessMessageDocument processMessage;
            ProcessMessageParameter parameter;
            bool assertion;

            this.ConsumedReport = reportIn;

            #region Manual Rules - Rules marked for manual coding

            #region VR.ATO.SMSFAR.437147

            /*  VR.ATO.SMSFAR.437147
                Subtotal 3 must equal zero.

            Legacy Rule Format:
                IF ([SMSFAR389] >= [SMSFAR304]) AND [SMSFAR390] <> 0
                   RETURN VALIDATION MESSAGE
                ENDIF

            Technical Business Rule Format:
                (^SMSFAR389 >= ^SMSFAR304) AND ^SMSFAR390 <> 0

            Data Elements:

            ^SMSFAR390 = RP:IncomeTax.TaxOffsetCarriedForwardSubTotal.Amount

            ^SMSFAR389 = RP:IncomeTax.TaxOffsetNonRefundableCarryForward.Amount

            ^SMSFAR304 = RP:IncomeTax.SubtotalTaxPayable.Amount
            */
            VRATOSMSFAR437147(response, report);
            #endregion // VR.ATO.SMSFAR.437147

            #region VR.ATO.SMSFAR.437169

            /*  VR.ATO.SMSFAR.437169
                        Member closing account balance must equal the sum of the accumulation phase account balance, the retirement phase account balance - Non-capped defined benefit income stream and retirement phase account balance - Capped defined benefit income stream

                    Legacy Rule Format:
                        WHERE IN TUPLE(smsfar.0013.lodge.req.xx.xx:MemberInformation)
                        IF ([SMSFAR223] = NULL OR [SMSFAR223] >= 0) AND ([SMSFAR223] <> ([SMSFAR393] + [SMSFAR394] + [SMSFAR395]))
                          RETURN VALIDATION MESSAGE
                        ENDIF

                Technical Business Rule Format:
                        (^SMSFAR223 = NULL OR ^SMSFAR223 >= 0) AND (^SMSFAR223 <> (^SMSFAR393 + ^SMSFAR394 + ^SMSFAR395))

            Data Elements:

                ^SMSFAR223 = RP.Closing:MemberInformation:SuperannuationBenefit.AccountBalance.Amount

                ^SMSFAR393 = RP.Closing.Accumulation:MemberInformation:SuperannuationBenefit.AccountBalance.Amount

                ^SMSFAR394 = RP.Closing.Retirement:MemberInformation:SuperannuationBenefit.NonCappedDefinedBenefitIncomeStream.Amount

                ^SMSFAR395 = RP.Closing.Retirement:MemberInformation:SuperannuationBenefit.CappedDefinedBenefitIncomeStream.Amount
                */
            VRATOSMSFAR437169(response, report);
            #endregion // VR.ATO.SMSFAR.437169
            #endregion Manual Rules - Rules marked for manual coding
            VRATOGEN000201();
            VRATOGEN000201_1();
            VRATOGEN000201_2();
            VRATOGEN000201_3();
            VRATOGEN000201_4();
            VRATOGEN000201_5();
            VRATOGEN000201_6();
            VRATOGEN000201_7();
            VRATOGEN000201_8();
            VRATOGEN000201_9();
            VRATOGEN000201_10();
            VRATOGEN000201_11();
            VRATOGEN000201_12();
            VRATOGEN000201_13();
            VRATOGEN000201_14();
            VRATOGEN000201_15();
            VRATOGEN000201_16();
            VRATOGEN000201_17();
            VRATOGEN000201_18();
            VRATOGEN000201_19();
            VRATOGEN000201_20();
            VRATOGEN000201_21();
            VRATOGEN000201_22();
            VRATOGEN000201_23();
            VRATOGEN000201_24();
            VRATOGEN000201_25();
            VRATOGEN000201_26();
            VRATOGEN000201_27();
            VRATOGEN000201_28();
            VRATOGEN000209();
            VRATOGEN000227();
            VRATOGEN000406();
            VRATOGEN000406_1();
            VRATOGEN000408();
            VRATOGEN000408_1();
            VRATOGEN000458();
            VRATOGEN000458_2();
            VRATOGEN000458_3();
            VRATOGEN000459();
            VRATOGEN000459_2();
            VRATOGEN000459_3();
            VRATOGEN001023();
            VRATOGEN001023_1();
            VRATOGEN001023_2();
            VRATOGEN001023_3();
            VRATOGEN001023_4();
            VRATOGEN001023_5();
            VRATOGEN001023_6();
            VRATOGEN001023_7();
            VRATOGEN001023_8();
            VRATOGEN001023_9();
            VRATOGEN001023_10();
            VRATOGEN001023_11();
            VRATOGEN001023_12();
            VRATOGEN001023_13();
            VRATOGEN001023_14();
            VRATOGEN001023_15();
            VRATOGEN001023_16();
            VRATOGEN001023_17();
            VRATOGEN001023_18();
            VRATOGEN001023_19();
            VRATOGEN001023_20();
            VRATOGEN001023_21();
            VRATOGEN001023_22();
            VRATOGEN001023_23();
            VRATOGEN001023_24();
            VRATOGEN001023_25();
            VRATOGEN001023_26();
            VRATOGEN001023_27();
            VRATOGEN001023_28();
            VRATOGEN001023_29();
            VRATOGEN001023_30();
            VRATOGEN010028();
            VRATOGEN010029();
            VRATOGEN010030();
            VRATOGEN010031();
            VRATOGEN408010();
            VRATOGEN408010_1();
            VRATOGEN410002();
            VRATOGEN410002_1();
            VRATOGEN410008();
            VRATOGEN410008_1();
            VRATOGEN410013();
            VRATOGEN410013_1();
            VRATOGEN410038();
            VRATOGEN410038_1();
            VRATOGEN410038_2();
            VRATOGEN410039();
            VRATOGEN410039_1();
            VRATOGEN410039_2();
            VRATOGEN410063();
            VRATOGEN410063_2();
            VRATOGEN410063_3();
            VRATOGEN410131();
            VRATOGEN410131_2();
            VRATOGEN410131_3();
            VRATOGEN410194();
            VRATOGEN410194_1();
            VRATOGEN410195();
            VRATOGEN410195_1();
            VRATOGEN410200();
            VRATOGEN410200_2();
            VRATOGEN410200_3();
            VRATOGEN410201();
            VRATOGEN410201_2();
            VRATOGEN410201_3();
            VRATOGEN410203();
            VRATOGEN410203_2();
            VRATOGEN410203_3();
            VRATOGEN410204();
            VRATOGEN410204_2();
            VRATOGEN410204_3();
            VRATOGEN410205();
            VRATOGEN410205_1();
            VRATOGEN410208();
            VRATOGEN410208_2();
            VRATOGEN410208_3();
            VRATOGEN410209();
            VRATOGEN410209_2();
            VRATOGEN410209_3();
            VRATOGEN428016();
            VRATOGEN428230();
            VRATOGEN428230_1();
            VRATOGEN428247();
            VRATOGEN428253();
            VRATOGEN428254();
            VRATOGEN428254_1();
            VRATOGEN428255();
            VRATOGEN428255_1();
            VRATOGEN428256();
            VRATOGEN428256_1();
            VRATOGEN428258();
            VRATOGEN428258_1();
            VRATOGEN428258_2();
            VRATOGEN428259();
            VRATOGEN428259_1();
            VRATOGEN428259_2();
            VRATOGEN428260();
            VRATOGEN428260_1();
            VRATOGEN428260_2();
            VRATOGEN428261();
            VRATOGEN428261_1();
            VRATOGEN428261_2();
            VRATOGEN428262();
            VRATOGEN428262_2();
            VRATOGEN428262_3();
            VRATOGEN428263();
            VRATOGEN428263_2();
            VRATOGEN428263_3();
            VRATOGEN428264();
            VRATOGEN428264_2();
            VRATOGEN428264_3();
            VRATOGEN428265();
            VRATOGEN428265_2();
            VRATOGEN428265_3();
            VRATOGEN428266();
            VRATOGEN428266_2();
            VRATOGEN428266_3();
            VRATOGEN428267();
            VRATOGEN428267_2();
            VRATOGEN428267_3();
            VRATOGEN430206();
            VRATOGEN430222();
            VRATOGEN430223();
            VRATOGEN430224();
            VRATOGEN430226();
            VRATOGEN430227();
            VRATOGEN430228();
            VRATOGEN430229();
            VRATOGEN430230();
            VRATOGEN430231();
            VRATOGEN430232();
            VRATOGEN430246();
            VRATOGEN430246_1();
            VRATOGEN430252();
            VRATOGEN430252_1();
            VRATOGEN430255();
            VRATOGEN430255_1();
            VRATOGEN430266();
            VRATOGEN430267();
            VRATOGEN430280();
            VRATOGEN430297();
            VRATOGEN430302();
            VRATOGEN430316();
            VRATOGEN432037();
            VRATOGEN432037_1();
            VRATOGEN432267();
            VRATOGEN432267_1();
            VRATOGEN432417();
            VRATOGEN434223();
            VRATOGEN434223_2();
            VRATOGEN436001();
            VRATOGEN436001_2();
            VRATOGEN436001_3();
            VRATOGEN438019();
            VRATOGEN438021();
            VRATOGEN438029();
            VRATOGEN438029_1();
            VRATOGEN438029_2();
            VRATOGEN438029_3();
            VRATOGEN438029_4();
            VRATOGEN438029_5();
            VRATOGEN438029_6();
            VRATOGEN438029_7();
            VRATOGEN438029_8();
            VRATOGEN438029_9();
            VRATOGEN438029_10();
            VRATOGEN438029_11();
            VRATOGEN438029_12();
            VRATOGEN438029_13();
            VRATOGEN438029_14();
            VRATOGEN438029_15();
            VRATOGEN438029_16();
            VRATOGEN438029_17();
            VRATOGEN438029_18();
            VRATOGEN438029_19();
            VRATOGEN438029_20();
            VRATOGEN438029_21();
            VRATOGEN438029_22();
            VRATOGEN438029_23();
            VRATOGEN438029_24();
            VRATOGEN438029_25();
            VRATOGEN438029_26();
            VRATOGEN438029_27();
            VRATOGEN438029_28();
            VRATOGEN438029_29();
            VRATOGEN438029_30();
            VRATOGEN438029_31();
            VRATOGEN438030();
            VRATOGEN438034();
            VRATOGEN438035();
            VRATOGEN438035_1();
            VRATOGEN438036();
            VRATOGEN438053();
            VRATOGEN438053_1();
            VRATOGEN438053_2();
            VRATOGEN438053_3();
            VRATOGEN438053_4();
            VRATOGEN438053_5();
            VRATOGEN438053_6();
            VRATOGEN438053_7();
            VRATOGEN438053_8();
            VRATOGEN438053_9();
            VRATOGEN438053_10();
            VRATOGEN438053_11();
            VRATOGEN438053_12();
            VRATOGEN438053_13();
            VRATOGEN438053_14();
            VRATOGEN438053_15();
            VRATOGEN438053_16();
            VRATOGEN438053_17();
            VRATOGEN438053_18();
            VRATOGEN438053_19();
            VRATOGEN438053_20();
            VRATOGEN438053_21();
            VRATOGEN438053_22();
            VRATOGEN438053_23();
            VRATOGEN438053_24();
            VRATOGEN438053_25();
            VRATOGEN438053_26();
            VRATOGEN438053_27();
            VRATOGEN438054();
            VRATOGEN438054_1();
            VRATOGEN438054_2();
            VRATOGEN438054_3();
            VRATOGEN438054_4();
            VRATOGEN438054_5();
            VRATOGEN438054_6();
            VRATOGEN438054_7();
            VRATOGEN438054_8();
            VRATOGEN438054_9();
            VRATOGEN438054_10();
            VRATOGEN438054_11();
            VRATOGEN438054_12();
            VRATOGEN438054_13();
            VRATOGEN438054_14();
            VRATOGEN438054_15();
            VRATOGEN438054_16();
            VRATOGEN438054_17();
            VRATOGEN438054_18();
            VRATOGEN438054_19();
            VRATOGEN438054_20();
            VRATOGEN438054_21();
            VRATOGEN438054_22();
            VRATOGEN438054_23();
            VRATOGEN438054_24();
            VRATOGEN438054_25();
            VRATOGEN438054_26();
            VRATOGEN438054_27();
            VRATOGEN438058();
            VRATOGEN438059();
            VRATOGEN438063();
            VRATOGEN438064();
            VRATOGEN438081();
            VRATOGEN438082();
            VRATOGEN500130();
            VRATOGEN500130_1();
            VRATOGEN500132();
            VRATOGEN500132_1();
            VRATOGEN500132_2();
            VRATOGEN500133();
            VRATOGEN500133_1();
            VRATOGEN500134();
            VRATOGEN500134_1();
            VRATOGEN500135();
            VRATOGEN500135_2();
            VRATOGEN500135_3();
            VRATOGEN500136();
            VRATOGEN500136_1();
            VRATOGEN500137();
            VRATOGEN500137_1();
            VRATOGEN500141();
            VRATOGEN500141_1();
            VRATOGEN500142();
            VRATOGEN500142_2();
            VRATOGEN500142_3();
            VRATOGEN500145();
            VRATOGEN500145_1();
            VRATOGEN500151();
            VRATOGEN500151_2();
            VRATOGEN500151_3();
            VRATOGEN500152();
            VRATOGEN500152_2();
            VRATOGEN500152_3();
            VRATOGEN500153();
            VRATOGEN500153_2();
            VRATOGEN500153_3();
            VRATOGEN500154();
            VRATOGEN500154_1();
            VRATOSMSFAR436012();
            VRATOSMSFAR436014();
            VRATOSMSFAR436016();
            VRATOSMSFAR436020();
            VRATOSMSFAR436021();
            VRATOSMSFAR436022();
            VRATOSMSFAR436023();
            VRATOSMSFAR436025();
            VRATOSMSFAR436026();
            VRATOSMSFAR436027();
            VRATOSMSFAR436029();
            VRATOSMSFAR436030();
            VRATOSMSFAR436031();
            VRATOSMSFAR436040();
            VRATOSMSFAR436041();
            VRATOSMSFAR436042();
            VRATOSMSFAR436043();
            VRATOSMSFAR436044();
            VRATOSMSFAR436046();
            VRATOSMSFAR436047();
            VRATOSMSFAR436048();
            VRATOSMSFAR436049();
            VRATOSMSFAR436050();
            VRATOSMSFAR436054();
            VRATOSMSFAR436055();
            VRATOSMSFAR436056();
            VRATOSMSFAR436057();
            VRATOSMSFAR436058();
            VRATOSMSFAR436059();
            VRATOSMSFAR436060();
            VRATOSMSFAR436061();
            VRATOSMSFAR436062();
            VRATOSMSFAR436063();
            VRATOSMSFAR436064();
            VRATOSMSFAR436066();
            VRATOSMSFAR436067();
            VRATOSMSFAR436068();
            VRATOSMSFAR436069();
            VRATOSMSFAR436071();
            VRATOSMSFAR436073();
            VRATOSMSFAR436074();
            VRATOSMSFAR436075();
            VRATOSMSFAR436076();
            VRATOSMSFAR436078();
            VRATOSMSFAR436079();
            VRATOSMSFAR436080();
            VRATOSMSFAR436081();
            VRATOSMSFAR436082();
            VRATOSMSFAR436083();
            VRATOSMSFAR436084();
            VRATOSMSFAR436086();
            VRATOSMSFAR436086_1();
            VRATOSMSFAR436086_2();
            VRATOSMSFAR436086_3();
            VRATOSMSFAR436088();
            VRATOSMSFAR436088_1();
            VRATOSMSFAR436091();
            VRATOSMSFAR436091_1();
            VRATOSMSFAR436092();
            VRATOSMSFAR436092_1();
            VRATOSMSFAR436094();
            VRATOSMSFAR436094_1();
            VRATOSMSFAR436095();
            VRATOSMSFAR436095_1();
            VRATOSMSFAR436098();
            VRATOSMSFAR436098_1();
            VRATOSMSFAR436099();
            VRATOSMSFAR436099_1();
            VRATOSMSFAR436100();
            VRATOSMSFAR436101();
            VRATOSMSFAR436103();
            VRATOSMSFAR436104();
            VRATOSMSFAR436105();
            VRATOSMSFAR436106();
            VRATOSMSFAR436107();
            VRATOSMSFAR436109();
            VRATOSMSFAR436110();
            VRATOSMSFAR436111();
            VRATOSMSFAR436113();
            VRATOSMSFAR436114();
            VRATOSMSFAR436115();
            VRATOSMSFAR436119();
            VRATOSMSFAR436120();
            VRATOSMSFAR436121();
            VRATOSMSFAR436122();
            VRATOSMSFAR436123();
            VRATOSMSFAR436124();
            VRATOSMSFAR436126();
            VRATOSMSFAR436128();
            VRATOSMSFAR436134();
            VRATOSMSFAR436135();
            VRATOSMSFAR436136();
            VRATOSMSFAR436137();
            VRATOSMSFAR436138();
            VRATOSMSFAR436139();
            VRATOSMSFAR436140();
            VRATOSMSFAR436141();
            VRATOSMSFAR436142();
            VRATOSMSFAR436144();
            VRATOSMSFAR436145();
            VRATOSMSFAR436147();
            VRATOSMSFAR436148();
            VRATOSMSFAR436149();
            VRATOSMSFAR436150();
            VRATOSMSFAR436151();
            VRATOSMSFAR436152();
            VRATOSMSFAR436153();
            VRATOSMSFAR436154();
            VRATOSMSFAR436155();
            VRATOSMSFAR436156();
            VRATOSMSFAR436157();
            VRATOSMSFAR436158();
            VRATOSMSFAR436159();
            VRATOSMSFAR436195();
            VRATOSMSFAR436198();
            VRATOSMSFAR436199();
            VRATOSMSFAR436201();
            VRATOSMSFAR436204();
            VRATOSMSFAR436205();
            VRATOSMSFAR436208();
            VRATOSMSFAR436211();
            VRATOSMSFAR436212();
            VRATOSMSFAR436213();
            VRATOSMSFAR436242();
            VRATOSMSFAR436243();
            VRATOSMSFAR436244();
            VRATOSMSFAR436247();
            VRATOSMSFAR436248();
            VRATOSMSFAR436251();
            VRATOSMSFAR436259();
            VRATOSMSFAR436263();
            VRATOSMSFAR436264();
            VRATOSMSFAR436265();
            VRATOSMSFAR436268();
            VRATOSMSFAR436274();
            VRATOSMSFAR436277();
            VRATOSMSFAR436281();
            VRATOSMSFAR436282();
            VRATOSMSFAR436283();
            VRATOSMSFAR436287();
            VRATOSMSFAR436289();
            VRATOSMSFAR436295();
            VRATOSMSFAR436295_1();
            VRATOSMSFAR436295_2();
            VRATOSMSFAR436295_3();
            VRATOSMSFAR436295_4();
            VRATOSMSFAR436295_5();
            VRATOSMSFAR436295_6();
            VRATOSMSFAR436295_7();
            VRATOSMSFAR436295_8();
            VRATOSMSFAR436295_9();
            VRATOSMSFAR436295_10();
            VRATOSMSFAR436295_11();
            VRATOSMSFAR436295_12();
            VRATOSMSFAR436295_13();
            VRATOSMSFAR436295_14();
            VRATOSMSFAR436295_15();
            VRATOSMSFAR436295_16();
            VRATOSMSFAR436295_17();
            VRATOSMSFAR436295_18();
            VRATOSMSFAR436295_19();
            VRATOSMSFAR436295_20();
            VRATOSMSFAR436295_21();
            VRATOSMSFAR436295_22();
            VRATOSMSFAR436295_23();
            VRATOSMSFAR436295_24();
            VRATOSMSFAR436295_25();
            VRATOSMSFAR436295_26();
            VRATOSMSFAR436295_27();
            VRATOSMSFAR436295_28();
            VRATOSMSFAR436295_29();
            VRATOSMSFAR436295_30();
            VRATOSMSFAR436300();
            VRATOSMSFAR436300_1();
            VRATOSMSFAR436302();
            VRATOSMSFAR436304();
            VRATOSMSFAR436309();
            VRATOSMSFAR436310();
            VRATOSMSFAR436312();
            VRATOSMSFAR436313();
            VRATOSMSFAR436314();
            VRATOSMSFAR436315();
            VRATOSMSFAR436316();
            VRATOSMSFAR436317();
            VRATOSMSFAR436318();
            VRATOSMSFAR436319();
            VRATOSMSFAR436320();
            VRATOSMSFAR436321();
            VRATOSMSFAR436322();
            VRATOSMSFAR436323();
            VRATOSMSFAR436324();
            VRATOSMSFAR436325();
            VRATOSMSFAR436326();
            VRATOSMSFAR436327();
            VRATOSMSFAR436337();
            VRATOSMSFAR436340();
            VRATOSMSFAR436341();
            VRATOSMSFAR436342();
            VRATOSMSFAR436342_1();
            VRATOSMSFAR436344();
            VRATOSMSFAR436349();
            VRATOSMSFAR436350();
            VRATOSMSFAR436351();
            VRATOSMSFAR436539();
            VRATOSMSFAR436540();
            VRATOSMSFAR436541();
            VRATOSMSFAR436542();
            VRATOSMSFAR436543();
            VRATOSMSFAR436544();
            VRATOSMSFAR436545();
            VRATOSMSFAR436546();
            VRATOSMSFAR436547();
            VRATOSMSFAR436548();
            VRATOSMSFAR436549();
            VRATOSMSFAR436550();
            VRATOSMSFAR436556();
            VRATOSMSFAR436557();
            VRATOSMSFAR436559();
            VRATOSMSFAR436560();
            VRATOSMSFAR436562();
            VRATOSMSFAR436563();
            VRATOSMSFAR436564();
            VRATOSMSFAR436565();
            VRATOSMSFAR436573();
            VRATOSMSFAR436597();
            VRATOSMSFAR436597_1();
            VRATOSMSFAR436600();
            VRATOSMSFAR436601();
            VRATOSMSFAR436610();
            VRATOSMSFAR436611();
            VRATOSMSFAR436612();
            VRATOSMSFAR436613();
            VRATOSMSFAR436614();
            VRATOSMSFAR436615();
            VRATOSMSFAR436616();
            VRATOSMSFAR436617();
            VRATOSMSFAR436618();
            VRATOSMSFAR436619();
            VRATOSMSFAR436621();
            VRATOSMSFAR436622();
            VRATOSMSFAR436630();
            VRATOSMSFAR436631();
            VRATOSMSFAR436636();
            VRATOSMSFAR436637();
            VRATOSMSFAR436638();
            VRATOSMSFAR436640();
            VRATOSMSFAR436645();
            VRATOSMSFAR436646();
            VRATOSMSFAR436653();
            VRATOSMSFAR436656();
            VRATOSMSFAR436657();
            VRATOSMSFAR436658();
            VRATOSMSFAR436661();
            VRATOSMSFAR436670();
            VRATOSMSFAR436674();
            VRATOSMSFAR436675();
            VRATOSMSFAR436676();
            VRATOSMSFAR436682();
            VRATOSMSFAR436683();
            VRATOSMSFAR436685();
            VRATOSMSFAR436686();
            VRATOSMSFAR436691();
            VRATOSMSFAR436692();
            VRATOSMSFAR436750();
            VRATOSMSFAR436760();
            VRATOSMSFAR436800();
            VRATOSMSFAR436803();
            VRATOSMSFAR436803_1();
            VRATOSMSFAR436803_2();
            VRATOSMSFAR436803_3();
            VRATOSMSFAR437100();
            VRATOSMSFAR437101();
            VRATOSMSFAR437102();
            VRATOSMSFAR437120();
            VRATOSMSFAR437121();
            VRATOSMSFAR437122();
            VRATOSMSFAR437123();
            VRATOSMSFAR437125();
            VRATOSMSFAR437126();
            VRATOSMSFAR437127();
            VRATOSMSFAR437128();
            VRATOSMSFAR437129();
            VRATOSMSFAR437130();
            VRATOSMSFAR437131();
            VRATOSMSFAR437132();
            VRATOSMSFAR437133();
            VRATOSMSFAR437134();
            VRATOSMSFAR437139();
            VRATOSMSFAR437140();
            VRATOSMSFAR437141();
            VRATOSMSFAR437142();
            VRATOSMSFAR437143();
            VRATOSMSFAR437144();
            VRATOSMSFAR437145();
            VRATOSMSFAR437146();
            VRATOSMSFAR437148();
            VRATOSMSFAR437159();
            VRATOSMSFAR437160();
            VRATOSMSFAR437161();
            VRATOSMSFAR437162();
            VRATOSMSFAR437163();
            VRATOSMSFAR437171();
            VRATOSMSFAR437182();
            VRATOSMSFAR437184();
            VRATOSMSFAR437185();
            VRATOSMSFAR437186();
            VRATOSMSFAR437189();
            VRATOSMSFAR437191();
            VRATOSMSFAR437192();
            VRATOSMSFAR437193();
            VRATOSMSFAR437198();
            VRATOSMSFAR437198_1();
            VRATOSMSFARW00001();
            VRATOSMSFARW00002();

            #region Repeating report.MemberInformationCollection  
            if (report.MemberInformationCollection != null)
            {
                int itemIndex1 = 0;
                foreach (SMSFAR2026.MemberInformation memberInformation in report.MemberInformationCollection)
                {
                    itemIndex1++;
                    VRATOGEN000458_1(memberInformation, itemIndex1);
                    VRATOGEN000459_1(memberInformation, itemIndex1);
                    VRATOGEN410031(memberInformation, itemIndex1);
                    VRATOGEN410063_1(memberInformation, itemIndex1);
                    VRATOGEN410131_1(memberInformation, itemIndex1);
                    VRATOGEN410200_1(memberInformation, itemIndex1);
                    VRATOGEN410201_1(memberInformation, itemIndex1);
                    VRATOGEN410203_1(memberInformation, itemIndex1);
                    VRATOGEN410204_1(memberInformation, itemIndex1);
                    VRATOGEN410208_1(memberInformation, itemIndex1);
                    VRATOGEN410209_1(memberInformation, itemIndex1);
                    VRATOGEN410218(memberInformation, itemIndex1);
                    VRATOGEN428262_1(memberInformation, itemIndex1);
                    VRATOGEN428263_1(memberInformation, itemIndex1);
                    VRATOGEN428264_1(memberInformation, itemIndex1);
                    VRATOGEN428265_1(memberInformation, itemIndex1);
                    VRATOGEN428266_1(memberInformation, itemIndex1);
                    VRATOGEN428267_1(memberInformation, itemIndex1);
                    VRATOGEN434223_1(memberInformation, itemIndex1);
                    VRATOGEN436001_1(memberInformation, itemIndex1);
                    VRATOGEN500135_1(memberInformation, itemIndex1);
                    VRATOGEN500142_1(memberInformation, itemIndex1);
                    VRATOGEN500151_1(memberInformation, itemIndex1);
                    VRATOGEN500152_1(memberInformation, itemIndex1);
                    VRATOGEN500153_1(memberInformation, itemIndex1);
                    VRATOSMSFAR436133(memberInformation, itemIndex1);
                    VRATOSMSFAR436210(memberInformation, itemIndex1);
                    VRATOSMSFAR436214(memberInformation, itemIndex1);
                    VRATOSMSFAR436215(memberInformation, itemIndex1);
                    VRATOSMSFAR436216(memberInformation, itemIndex1);
                    VRATOSMSFAR436217(memberInformation, itemIndex1);
                    VRATOSMSFAR436218(memberInformation, itemIndex1);
                    VRATOSMSFAR436219(memberInformation, itemIndex1);
                    VRATOSMSFAR436220(memberInformation, itemIndex1);
                    VRATOSMSFAR436221(memberInformation, itemIndex1);
                    VRATOSMSFAR436222(memberInformation, itemIndex1);
                    VRATOSMSFAR436223(memberInformation, itemIndex1);
                    VRATOSMSFAR436225(memberInformation, itemIndex1);
                    VRATOSMSFAR436226(memberInformation, itemIndex1);
                    VRATOSMSFAR436227(memberInformation, itemIndex1);
                    VRATOSMSFAR436228(memberInformation, itemIndex1);
                    VRATOSMSFAR436229(memberInformation, itemIndex1);
                    VRATOSMSFAR436230(memberInformation, itemIndex1);
                    VRATOSMSFAR436231(memberInformation, itemIndex1);
                    VRATOSMSFAR436232(memberInformation, itemIndex1);
                    VRATOSMSFAR436234(memberInformation, itemIndex1);
                    VRATOSMSFAR436239(memberInformation, itemIndex1);
                    VRATOSMSFAR436240(memberInformation, itemIndex1);
                    VRATOSMSFAR436272(memberInformation, itemIndex1);
                    VRATOSMSFAR436273(memberInformation, itemIndex1);
                    VRATOSMSFAR436291(memberInformation, itemIndex1);
                    VRATOSMSFAR436292(memberInformation, itemIndex1);
                    VRATOSMSFAR436293(memberInformation, itemIndex1);
                    VRATOSMSFAR436298(memberInformation, itemIndex1);
                    VRATOSMSFAR436299(memberInformation, itemIndex1);
                    VRATOSMSFAR436305(memberInformation, itemIndex1);
                    VRATOSMSFAR436352(memberInformation, itemIndex1);
                    VRATOSMSFAR436353(memberInformation, itemIndex1);
                    VRATOSMSFAR436603(memberInformation, itemIndex1);
                    VRATOSMSFAR436604(memberInformation, itemIndex1);
                    VRATOSMSFAR436605(memberInformation, itemIndex1);
                    VRATOSMSFAR436606(memberInformation, itemIndex1);
                    VRATOSMSFAR436607(memberInformation, itemIndex1);
                    VRATOSMSFAR436608(memberInformation, itemIndex1);
                    VRATOSMSFAR436632(memberInformation, itemIndex1);
                    VRATOSMSFAR436633(memberInformation, itemIndex1);
                    VRATOSMSFAR436634(memberInformation, itemIndex1);
                    VRATOSMSFAR436642(memberInformation, itemIndex1);
                    VRATOSMSFAR436643(memberInformation, itemIndex1);
                    VRATOSMSFAR437164(memberInformation, itemIndex1);
                    VRATOSMSFAR437165(memberInformation, itemIndex1);
                    VRATOSMSFAR437166(memberInformation, itemIndex1);
                    VRATOSMSFAR437167(memberInformation, itemIndex1);
                    VRATOSMSFAR437168(memberInformation, itemIndex1);
                    VRATOSMSFAR437170(memberInformation, itemIndex1);
                    VRATOSMSFAR437172(memberInformation, itemIndex1);
                    VRATOSMSFAR437177(memberInformation, itemIndex1);
                    VRATOSMSFAR437179(memberInformation, itemIndex1);
                    VRATOSMSFAR437180(memberInformation, itemIndex1);
                    VRATOSMSFAR437181(memberInformation, itemIndex1);
                    VRATOSMSFAR437190(memberInformation, itemIndex1);
                }
            }

            #endregion // Foreach loop

            #region Repeating SMSFAR328Collection  
            if (report.SMSFAR328Collection != null)
            {
                int itemIndex1 = 0;
                foreach (string SMSFAR328 in report.SMSFAR328Collection)
                {
                    itemIndex1++;
                    VRATOSMSFAR436572(SMSFAR328, itemIndex1);
                }
            }

            #endregion // Foreach loop

            foreach (ProcessMessageDocument currentProcessMessage in response)
            {
                if (currentProcessMessage.Parameters != null)
                {
                    foreach (ProcessMessageParameter currentParameter in currentProcessMessage.Parameters)
                    {
                        if (string.IsNullOrEmpty(currentParameter.Name))
                        {
                            currentParameter.Name = _emptyParameterValue;
                        }

                        if (currentParameter.Name.Length > _maxParameterNameLength)
                        {
                            currentParameter.Name = currentParameter.Name.Substring(0, _maxParameterNameLength - 1);
                        }

                        if (string.IsNullOrEmpty(currentParameter.Value))
                        {
                            currentParameter.Value = _emptyParameterValue;
                        }

                        if (currentParameter.Value.Length > _maxParameterValueLength)
                        {
                            currentParameter.Value = currentParameter.Value.Substring(0, _maxParameterValueLength - 1);
                        }
                    }
                }
            }

            return response;
        }



        #region VR.ATO.GEN.000201
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000201()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.000201
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate >= period.endDate WHERE CONTEXT(ALL)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate >= ^EndDate
    
            Data Elements:
    
            ^StartDate = Auditor
    
            ^EndDate = Auditor
            */
            assertion = ((report.AuditorExists == true) && (report.AuditorStartDate >= report.AuditorEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.AuditorOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.000201" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.000201
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000201_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.000201
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate >= period.endDate WHERE CONTEXT(ALL)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate >= ^EndDate
    
            Data Elements:
    
            ^StartDate = INT
    
            ^EndDate = INT
            */
            assertion = ((report.INTExists == true) && (report.INTStartDate >= report.INTEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.INTOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.000201" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.000201
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000201_2()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.000201
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate >= period.endDate WHERE CONTEXT(ALL)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate >= ^EndDate
    
            Data Elements:
    
            ^StartDate = RP
    
            ^EndDate = RP
            */
            assertion = ((report.RPExists == true) && (report.RPStartDate >= report.RPEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.000201" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.000201
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000201_3()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.000201
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate >= period.endDate WHERE CONTEXT(ALL)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate >= ^EndDate
    
            Data Elements:
    
            ^StartDate = RP:RP.Accumulation
    
            ^EndDate = RP:RP.Accumulation
            */
            assertion = ((report.RPAccumulationExists == true) && (report.RPAccumulationStartDate >= report.RPAccumulationEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPAccumulationOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.000201" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.000201
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000201_4()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.000201
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate >= period.endDate WHERE CONTEXT(ALL)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate >= ^EndDate
    
            Data Elements:
    
            ^StartDate = RP:RP.Closing
    
            ^EndDate = RP:RP.Closing
            */
            assertion = ((report.RPClosingExists == true) && (report.RPClosingStartDate >= report.RPClosingEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPClosingOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.000201" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.000201
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000201_5()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.000201
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate >= period.endDate WHERE CONTEXT(ALL)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate >= ^EndDate
    
            Data Elements:
    
            ^StartDate = RP.Closing:RP.Closing.Accumulation
    
            ^EndDate = RP.Closing:RP.Closing.Accumulation
            */
            assertion = ((report.RPClosingAccumulationExists == true) && (report.RPClosingAccumulationStartDate >= report.RPClosingAccumulationEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPClosingAccumulationOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.000201" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.000201
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000201_6()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.000201
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate >= period.endDate WHERE CONTEXT(ALL)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate >= ^EndDate
    
            Data Elements:
    
            ^StartDate = RP.Closing:RP.Closing.Retirement
    
            ^EndDate = RP.Closing:RP.Closing.Retirement
            */
            assertion = ((report.RPClosingRetirementExists == true) && (report.RPClosingRetirementStartDate >= report.RPClosingRetirementEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPClosingRetirementOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.000201" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.000201
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000201_7()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.000201
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate >= period.endDate WHERE CONTEXT(ALL)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate >= ^EndDate
    
            Data Elements:
    
            ^StartDate = RP:RP.Deductible
    
            ^EndDate = RP:RP.Deductible
            */
            assertion = ((report.RPDeductibleExists == true) && (report.RPDeductibleStartDate >= report.RPDeductibleEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPDeductibleOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.000201" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.000201
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000201_8()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.000201
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate >= period.endDate WHERE CONTEXT(ALL)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate >= ^EndDate
    
            Data Elements:
    
            ^StartDate = RP.Deductible:RP.Deductible.JAUS
    
            ^EndDate = RP.Deductible:RP.Deductible.JAUS
            */
            assertion = ((report.RPDeductibleJAUSExists == true) && (report.RPDeductibleJAUSStartDate >= report.RPDeductibleJAUSEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPDeductibleJAUSOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.000201" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.000201
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000201_9()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.000201
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate >= period.endDate WHERE CONTEXT(ALL)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate >= ^EndDate
    
            Data Elements:
    
            ^StartDate = RP.Deductible:RP.Deductible.JFOREIGN
    
            ^EndDate = RP.Deductible:RP.Deductible.JFOREIGN
            */
            assertion = ((report.RPDeductibleJFOREIGNExists == true) && (report.RPDeductibleJFOREIGNStartDate >= report.RPDeductibleJFOREIGNEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPDeductibleJFOREIGNOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.000201" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.000201
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000201_10()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.000201
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate >= period.endDate WHERE CONTEXT(ALL)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate >= ^EndDate
    
            Data Elements:
    
            ^StartDate = RP:RP.IncomeTaxRefund
    
            ^EndDate = RP:RP.IncomeTaxRefund
            */
            assertion = ((report.RPIncomeTaxRefundExists == true) && (report.RPIncomeTaxRefundStartDate >= report.RPIncomeTaxRefundEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPIncomeTaxRefundOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.000201" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.000201
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000201_11()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.000201
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate >= period.endDate WHERE CONTEXT(ALL)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate >= ^EndDate
    
            Data Elements:
    
            ^StartDate = RP:RP.Investor
    
            ^EndDate = RP:RP.Investor
            */
            assertion = ((report.RPInvestorExists == true) && (report.RPInvestorStartDate >= report.RPInvestorEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPInvestorOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.000201" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.000201
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000201_12()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.000201
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate >= period.endDate WHERE CONTEXT(ALL)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate >= ^EndDate
    
            Data Elements:
    
            ^StartDate = RP.Investor:RP.Investor.YPrevious
    
            ^EndDate = RP.Investor:RP.Investor.YPrevious
            */
            assertion = ((report.RPInvestorYPreviousExists == true) && (report.RPInvestorYPreviousStartDate >= report.RPInvestorYPreviousEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPInvestorYPreviousOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.000201" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.000201
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000201_13()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.000201
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate >= period.endDate WHERE CONTEXT(ALL)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate >= ^EndDate
    
            Data Elements:
    
            ^StartDate = RP:RP.JAUS
    
            ^EndDate = RP:RP.JAUS
            */
            assertion = ((report.RPJAUSExists == true) && (report.RPJAUSStartDate >= report.RPJAUSEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.000201" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.000201
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000201_14()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.000201
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate >= period.endDate WHERE CONTEXT(ALL)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate >= ^EndDate
    
            Data Elements:
    
            ^StartDate = RP.JAUS:RP.JAUS.AustNonResidentialRealProperty
    
            ^EndDate = RP.JAUS:RP.JAUS.AustNonResidentialRealProperty
            */
            assertion = ((report.RPJAUSAustNonResidentialRealPropertyExists == true) && (report.RPJAUSAustNonResidentialRealPropertyStartDate >= report.RPJAUSAustNonResidentialRealPropertyEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSAustNonResidentialRealPropertyOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.000201" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.000201
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000201_15()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.000201
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate >= period.endDate WHERE CONTEXT(ALL)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate >= ^EndDate
    
            Data Elements:
    
            ^StartDate = RP.JAUS:RP.JAUS.AustralianShares
    
            ^EndDate = RP.JAUS:RP.JAUS.AustralianShares
            */
            assertion = ((report.RPJAUSAustralianSharesExists == true) && (report.RPJAUSAustralianSharesStartDate >= report.RPJAUSAustralianSharesEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSAustralianSharesOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.000201" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.000201
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000201_16()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.000201
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate >= period.endDate WHERE CONTEXT(ALL)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate >= ^EndDate
    
            Data Elements:
    
            ^StartDate = RP.JAUS:RP.JAUS.AustResidentialRealProperty
    
            ^EndDate = RP.JAUS:RP.JAUS.AustResidentialRealProperty
            */
            assertion = ((report.RPJAUSAustResidentialRealPropertyExists == true) && (report.RPJAUSAustResidentialRealPropertyStartDate >= report.RPJAUSAustResidentialRealPropertyEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSAustResidentialRealPropertyOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.000201" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.000201
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000201_17()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.000201
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate >= period.endDate WHERE CONTEXT(ALL)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate >= ^EndDate
    
            Data Elements:
    
            ^StartDate = RP.JAUS:RP.JAUS.Other
    
            ^EndDate = RP.JAUS:RP.JAUS.Other
            */
            assertion = ((report.RPJAUSOtherExists == true) && (report.RPJAUSOtherStartDate >= report.RPJAUSOtherEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSOtherOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.000201" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.000201
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000201_18()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.000201
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate >= period.endDate WHERE CONTEXT(ALL)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate >= ^EndDate
    
            Data Elements:
    
            ^StartDate = RP.JAUS:RP.JAUS.OtherShares
    
            ^EndDate = RP.JAUS:RP.JAUS.OtherShares
            */
            assertion = ((report.RPJAUSOtherSharesExists == true) && (report.RPJAUSOtherSharesStartDate >= report.RPJAUSOtherSharesEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSOtherSharesOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.000201" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.000201
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000201_19()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.000201
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate >= period.endDate WHERE CONTEXT(ALL)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate >= ^EndDate
    
            Data Elements:
    
            ^StartDate = RP.JAUS:RP.JAUS.OverseasRealProperty
    
            ^EndDate = RP.JAUS:RP.JAUS.OverseasRealProperty
            */
            assertion = ((report.RPJAUSOverseasRealPropertyExists == true) && (report.RPJAUSOverseasRealPropertyStartDate >= report.RPJAUSOverseasRealPropertyEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSOverseasRealPropertyOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.000201" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.000201
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000201_20()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.000201
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate >= period.endDate WHERE CONTEXT(ALL)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate >= ^EndDate
    
            Data Elements:
    
            ^StartDate = RP:RP.JFOREIGN
    
            ^EndDate = RP:RP.JFOREIGN
            */
            assertion = ((report.RPJFOREIGNExists == true) && (report.RPJFOREIGNStartDate >= report.RPJFOREIGNEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJFOREIGNOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.000201" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.000201
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000201_21()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.000201
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate >= period.endDate WHERE CONTEXT(ALL)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate >= ^EndDate
    
            Data Elements:
    
            ^StartDate = RP:RP.LimitedPartners
    
            ^EndDate = RP:RP.LimitedPartners
            */
            assertion = ((report.RPLimitedPartnersExists == true) && (report.RPLimitedPartnersStartDate >= report.RPLimitedPartnersEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPLimitedPartnersOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.000201" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.000201
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000201_22()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.000201
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate >= period.endDate WHERE CONTEXT(ALL)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate >= ^EndDate
    
            Data Elements:
    
            ^StartDate = RP.LimitedPartners:RP.LimitedPartners.YPrevious
    
            ^EndDate = RP.LimitedPartners:RP.LimitedPartners.YPrevious
            */
            assertion = ((report.RPLimitedPartnersYPreviousExists == true) && (report.RPLimitedPartnersYPreviousStartDate >= report.RPLimitedPartnersYPreviousEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPLimitedPartnersYPreviousOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.000201" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.000201
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000201_23()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.000201
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate >= period.endDate WHERE CONTEXT(ALL)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate >= ^EndDate
    
            Data Elements:
    
            ^StartDate = RP:RP.NonDeductible
    
            ^EndDate = RP:RP.NonDeductible
            */
            assertion = ((report.RPNonDeductibleExists == true) && (report.RPNonDeductibleStartDate >= report.RPNonDeductibleEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPNonDeductibleOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.000201" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.000201
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000201_24()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.000201
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate >= period.endDate WHERE CONTEXT(ALL)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate >= ^EndDate
    
            Data Elements:
    
            ^StartDate = RP.NonDeductible:RP.NonDeductible.JAUS
    
            ^EndDate = RP.NonDeductible:RP.NonDeductible.JAUS
            */
            assertion = ((report.RPNonDeductibleJAUSExists == true) && (report.RPNonDeductibleJAUSStartDate >= report.RPNonDeductibleJAUSEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPNonDeductibleJAUSOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.000201" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.000201
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000201_25()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.000201
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate >= period.endDate WHERE CONTEXT(ALL)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate >= ^EndDate
    
            Data Elements:
    
            ^StartDate = RP.NonDeductible:RP.NonDeductible.JFOREIGN
    
            ^EndDate = RP.NonDeductible:RP.NonDeductible.JFOREIGN
            */
            assertion = ((report.RPNonDeductibleJFOREIGNExists == true) && (report.RPNonDeductibleJFOREIGNStartDate >= report.RPNonDeductibleJFOREIGNEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPNonDeductibleJFOREIGNOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.000201" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.000201
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000201_26()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.000201
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate >= period.endDate WHERE CONTEXT(ALL)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate >= ^EndDate
    
            Data Elements:
    
            ^StartDate = RP:RP.Retirement
    
            ^EndDate = RP:RP.Retirement
            */
            assertion = ((report.RPRetirementExists == true) && (report.RPRetirementStartDate >= report.RPRetirementEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPRetirementOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.000201" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.000201
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000201_27()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.000201
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate >= period.endDate WHERE CONTEXT(ALL)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate >= ^EndDate
    
            Data Elements:
    
            ^StartDate = RP:RP.Superannuation
    
            ^EndDate = RP:RP.Superannuation
            */
            assertion = ((report.RPSuperannuationExists == true) && (report.RPSuperannuationStartDate >= report.RPSuperannuationEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPSuperannuationOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.000201" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.000201
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000201_28()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.000201
            Period start date must be prior to Period end date
    
            Legacy Rule Format:
            IF period.startDate >= period.endDate WHERE CONTEXT(ALL)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^StartDate >= ^EndDate
    
            Data Elements:
    
            ^StartDate = RP:RP.TOFA
    
            ^EndDate = RP:RP.TOFA
            */
            assertion = ((report.RPTOFAExists == true) && (report.RPTOFAStartDate >= report.RPTOFAEndDate));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must be prior to Period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPTOFAOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.000201" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.000209
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000209()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.000209
            Invalid context. The number of Reporting party contexts must equal 1
    
            Legacy Rule Format:
            IF COUNT(RP) <> 1
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) <> 1
    
            Data Elements:
    
            ^Context = RP
            */
            assertion = (Count(report.RPCount) != 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430296",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Reporting party contexts must equal 1",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.000209" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.000227
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000227()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.000227
            Invalid context. The number of Intermediary contexts must not exceed 1
    
            Legacy Rule Format:
            IF COUNT(INT) > 1
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = INT
            */
            assertion = (Count(report.INTCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.436280",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Intermediary contexts must not exceed 1",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.INTLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.000227" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.000406
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000406()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.000406
            Address line 1 contains incorrect 'care of' reference.  The only acceptable 'care of' reference is C/-
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line1.Text <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.Line1.Text STARTSWITH SET("C/O ","C/ ","Care Of ","CO ")) OR (pyde.xx.xx:AddressDetails.Line1.Text CONTAINS SET(" C/O "," C/ "," Care Of "))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^SMSFAR6 <> BLANK) AND (StartsWithSet(^SMSFAR6,'"C/O ","C/ ","Care Of ","CO "')) OR (ContainsSet(^SMSFAR6, '" C/O "," C/ "," Care Of "'))
    
            Data Elements:
    
            ^SMSFAR6 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.SMSFAR6) != true && IsMatch(report.SMSFAR6, @"^(C/O |C/ |Care Of |CO )", RegexOptions.IgnoreCase) || IsMatch(report.SMSFAR6, @" C/O | C/ | Care Of ", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000406",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 contains incorrect 'care of' reference.  The only acceptable 'care of' reference is C/-",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.000406" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR6", Value = report.SMSFAR6 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.000406
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000406_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.000406
            Address line 1 contains incorrect 'care of' reference.  The only acceptable 'care of' reference is C/-
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line1.Text <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.Line1.Text STARTSWITH SET("C/O ","C/ ","Care Of ","CO ")) OR (pyde.xx.xx:AddressDetails.Line1.Text CONTAINS SET(" C/O "," C/ "," Care Of "))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^SMSFAR19 <> BLANK) AND (StartsWithSet(^SMSFAR19,'"C/O ","C/ ","Care Of ","CO "')) OR (ContainsSet(^SMSFAR19, '" C/O "," C/ "," Care Of "'))
    
            Data Elements:
    
            ^SMSFAR19 = Auditor:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.SMSFAR19) != true && IsMatch(report.SMSFAR19, @"^(C/O |C/ |Care Of |CO )", RegexOptions.IgnoreCase) || IsMatch(report.SMSFAR19, @" C/O | C/ | Care Of ", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000406",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 contains incorrect 'care of' reference.  The only acceptable 'care of' reference is C/-",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_AuditorOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.000406" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR19", Value = report.SMSFAR19 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.000408
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000408()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.000408
            Care of' is not valid for address line 2
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line2.Text <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.Line2.Text STARTSWITH SET("C/- ","C/O ","C/ ","Care Of ","CO ") OR pyde.xx.xx:AddressDetails.Line2.Text CONTAINS SET(" C/- "," C/O "," C/ "," Care Of "))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^SMSFAR239 <> BLANK) AND (StartsWithSet(^SMSFAR239,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^SMSFAR239, '" C/- "," C/O "," C/ "," Care Of "'))
    
            Data Elements:
    
            ^SMSFAR239 = RP:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.SMSFAR239) != true && (IsMatch(report.SMSFAR239, @"^(C/- |C/O |C/ |Care Of |CO )", RegexOptions.IgnoreCase) || IsMatch(report.SMSFAR239, @" C/- | C/O | C/ | Care Of ", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000408",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Care of' is not valid for address line 2",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line2.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.000408" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR239", Value = GetValueOrEmpty(report.SMSFAR239) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.000408
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000408_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.000408
            Care of' is not valid for address line 2
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line2.Text <> NULLORBLANK) AND (pyde.xx.xx:AddressDetails.Line2.Text STARTSWITH SET("C/- ","C/O ","C/ ","Care Of ","CO ") OR pyde.xx.xx:AddressDetails.Line2.Text CONTAINS SET(" C/- "," C/O "," C/ "," Care Of "))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^SMSFAR240 <> BLANK) AND (StartsWithSet(^SMSFAR240,'"C/- ","C/O ","C/ ","Care Of ","CO "') OR ContainsSet(^SMSFAR240, '" C/- "," C/O "," C/ "," Care Of "'))
    
            Data Elements:
    
            ^SMSFAR240 = Auditor:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.SMSFAR240) != true && (IsMatch(report.SMSFAR240, @"^(C/- |C/O |C/ |Care Of |CO )", RegexOptions.IgnoreCase) || IsMatch(report.SMSFAR240, @" C/- | C/O | C/ | Care Of ", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000408",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Care of' is not valid for address line 2",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_AuditorOccurrenceIndex) + "/tns:AddressDetails.Line2.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.000408" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR240", Value = GetValueOrEmpty(report.SMSFAR240) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.000458
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000458()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.000458
            Suffix is invalid
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.NameSuffix.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.NameSuffix.Text <> SET(DOMAIN(SUFFIX CODES)))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^SMSFAR179 <> BLANK) AND (NotInDomain(^SMSFAR179, 'SUFFIX CODES'))
    
            Data Elements:
    
            ^SMSFAR179 = INT:PersonNameDetails:PersonNameDetails.NameSuffix.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.SMSFAR179) != true && !(IsMatch(report.SMSFAR179, @"^(ESQ|II|III|IV|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000458",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suffix is invalid",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_Contact_C_INTOccurrenceIndex) + "/tns:PersonNameDetails.NameSuffix.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.000458" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR179", Value = GetValueOrEmpty(report.SMSFAR179) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.000458
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000458_1(SMSFAR2026.MemberInformation memberInformation, int itemIndex1)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.000458
            Suffix is invalid

            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.NameSuffix.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.NameSuffix.Text <> SET(DOMAIN(SUFFIX CODES)))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^SMSFAR198 <> BLANK) AND (NotInDomain(^SMSFAR198, 'SUFFIX CODES'))

            Data Elements:

            ^SMSFAR198 = RP:MemberInformation:PersonNameDetails:PersonNameDetails.NameSuffix.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Identifier'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(memberInformation.SMSFAR198) != true && !(IsMatch(memberInformation.SMSFAR198, @"^(ESQ|II|III|IV|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000458",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suffix is invalid",
                    Location = "/xbrli:xbrl/tns:MemberInformation" + OccurrenceIndex(memberInformation.OccurrenceIndex) + "/tns:PersonNameDetails" + OccurrenceIndex(memberInformation.MemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RPOccurrenceIndex) + "/tns:PersonNameDetails.NameSuffix.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.000458" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR198", Value = GetValueOrEmpty(memberInformation.SMSFAR198) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.000458
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000458_2()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.000458
            Suffix is invalid
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.NameSuffix.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.NameSuffix.Text <> SET(DOMAIN(SUFFIX CODES)))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^SMSFAR168 <> BLANK) AND (NotInDomain(^SMSFAR168, 'SUFFIX CODES'))
    
            Data Elements:
    
            ^SMSFAR168 = RP:TrusteeOrDirector:PersonNameDetails:PersonNameDetails.NameSuffix.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.SMSFAR168) != true && !(IsMatch(report.SMSFAR168, @"^(ESQ|II|III|IV|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000458",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suffix is invalid",
                    Location = "/xbrli:xbrl/tns:TrusteeOrDirector/tns:PersonNameDetails" + OccurrenceIndex(report.TrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RPOccurrenceIndex) + "/tns:PersonNameDetails.NameSuffix.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.000458" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR168", Value = GetValueOrEmpty(report.SMSFAR168) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.000458
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000458_3()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.000458
            Suffix is invalid
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.NameSuffix.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.NameSuffix.Text <> SET(DOMAIN(SUFFIX CODES)))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^SMSFAR12 <> BLANK) AND (NotInDomain(^SMSFAR12, 'SUFFIX CODES'))
    
            Data Elements:
    
            ^SMSFAR12 = Auditor:PersonNameDetails:PersonNameDetails.NameSuffix.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.SMSFAR12) != true && !(IsMatch(report.SMSFAR12, @"^(ESQ|II|III|IV|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000458",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suffix is invalid",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_Contact_C_AuditorOccurrenceIndex) + "/tns:PersonNameDetails.NameSuffix.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.000458" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR12", Value = GetValueOrEmpty(report.SMSFAR12) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.000459
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000459()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.000459
            Title is invalid
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.Title.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.Title.Text <> SET(DOMAIN(TITLE CODES)))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^SMSFAR177 <> BLANK) AND (NotInDomain(^SMSFAR177, 'TITLE CODES'))
    
            Data Elements:
    
            ^SMSFAR177 = INT:PersonNameDetails:PersonNameDetails.Title.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.SMSFAR177) != true && !(IsMatch(report.SMSFAR177, @"^(2LT|AB|ABBOT|AC|ACM|ADM|AIR CDRE|ALDERMAN|AM|ARCHBISHOP|ARCHDEACON|ASSOC PROF|AVM|BARON|BARONESS|BISHOP|BR|BRIG|CANON|CAPT|CARDINAL|CDRE|CDT|CHAP|CMDR|CMM|COL|CONST|COUNT|COUNTESS|CPL|CPO|DAME|DEACON|DEACONESS|DEAN|DEPUTY SUPT|DR|DUCHESS|DUKE|EARL|EF|FLGOFF|FLT LT|FR|FSGT|GEN|GNR|GP CAPT|HON JUDGE|HON JUST|HON|HRH|INSP|JUDGE|JUST|LAC|LACW|LADY|LBDR|LCPL|LORD|LS|LT CMDR|LT COL|LT GEN|LT|MADAM|MAJ GEN|MAJ|MASTER|MATRON|MAYOR|MAYORESS|MIDN|MISS|MON|MOST REV|MR|MRS|MS|PASTOR|PATRIARCH|PLT OFF|PO|PRIOR|PROF|PTE|RABBI|RADM|RECTOR|REV|RF|RT HON|RT REV MON|RT REV|RTREV BISHOP|SBLT|SEN|SGT|SIR|SMN|SNR CONST|SQN LDR|SR|SSGT|SUPR|SWAMI|TF|VADM|VERY REV|VICAR|VISCOUNT|WG CDR|WO|WO1|WO2)$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000459",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Title is invalid",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_Contact_C_INTOccurrenceIndex) + "/tns:PersonNameDetails.Title.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.000459" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR177", Value = GetValueOrEmpty(report.SMSFAR177) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.000459
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000459_1(SMSFAR2026.MemberInformation memberInformation, int itemIndex1)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.000459
            Title is invalid

            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.Title.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.Title.Text <> SET(DOMAIN(TITLE CODES)))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^SMSFAR196 <> BLANK) AND (NotInDomain(^SMSFAR196, 'TITLE CODES'))

            Data Elements:

            ^SMSFAR196 = RP:MemberInformation:PersonNameDetails:PersonNameDetails.Title.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Identifier'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(memberInformation.SMSFAR196) != true && !(IsMatch(memberInformation.SMSFAR196, @"^(2LT|AB|ABBOT|AC|ACM|ADM|AIR CDRE|ALDERMAN|AM|ARCHBISHOP|ARCHDEACON|ASSOC PROF|AVM|BARON|BARONESS|BISHOP|BR|BRIG|CANON|CAPT|CARDINAL|CDRE|CDT|CHAP|CMDR|CMM|COL|CONST|COUNT|COUNTESS|CPL|CPO|DAME|DEACON|DEACONESS|DEAN|DEPUTY SUPT|DR|DUCHESS|DUKE|EARL|EF|FLGOFF|FLT LT|FR|FSGT|GEN|GNR|GP CAPT|HON JUDGE|HON JUST|HON|HRH|INSP|JUDGE|JUST|LAC|LACW|LADY|LBDR|LCPL|LORD|LS|LT CMDR|LT COL|LT GEN|LT|MADAM|MAJ GEN|MAJ|MASTER|MATRON|MAYOR|MAYORESS|MIDN|MISS|MON|MOST REV|MR|MRS|MS|PASTOR|PATRIARCH|PLT OFF|PO|PRIOR|PROF|PTE|RABBI|RADM|RECTOR|REV|RF|RT HON|RT REV MON|RT REV|RTREV BISHOP|SBLT|SEN|SGT|SIR|SMN|SNR CONST|SQN LDR|SR|SSGT|SUPR|SWAMI|TF|VADM|VERY REV|VICAR|VISCOUNT|WG CDR|WO|WO1|WO2)$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000459",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Title is invalid",
                    Location = "/xbrli:xbrl/tns:MemberInformation" + OccurrenceIndex(memberInformation.OccurrenceIndex) + "/tns:PersonNameDetails" + OccurrenceIndex(memberInformation.MemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RPOccurrenceIndex) + "/tns:PersonNameDetails.Title.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.000459" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR196", Value = GetValueOrEmpty(memberInformation.SMSFAR196) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.000459
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000459_2()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.000459
            Title is invalid
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.Title.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.Title.Text <> SET(DOMAIN(TITLE CODES)))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^SMSFAR166 <> BLANK) AND (NotInDomain(^SMSFAR166, 'TITLE CODES'))
    
            Data Elements:
    
            ^SMSFAR166 = RP:TrusteeOrDirector:PersonNameDetails:PersonNameDetails.Title.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.SMSFAR166) != true && !(IsMatch(report.SMSFAR166, @"^(2LT|AB|ABBOT|AC|ACM|ADM|AIR CDRE|ALDERMAN|AM|ARCHBISHOP|ARCHDEACON|ASSOC PROF|AVM|BARON|BARONESS|BISHOP|BR|BRIG|CANON|CAPT|CARDINAL|CDRE|CDT|CHAP|CMDR|CMM|COL|CONST|COUNT|COUNTESS|CPL|CPO|DAME|DEACON|DEACONESS|DEAN|DEPUTY SUPT|DR|DUCHESS|DUKE|EARL|EF|FLGOFF|FLT LT|FR|FSGT|GEN|GNR|GP CAPT|HON JUDGE|HON JUST|HON|HRH|INSP|JUDGE|JUST|LAC|LACW|LADY|LBDR|LCPL|LORD|LS|LT CMDR|LT COL|LT GEN|LT|MADAM|MAJ GEN|MAJ|MASTER|MATRON|MAYOR|MAYORESS|MIDN|MISS|MON|MOST REV|MR|MRS|MS|PASTOR|PATRIARCH|PLT OFF|PO|PRIOR|PROF|PTE|RABBI|RADM|RECTOR|REV|RF|RT HON|RT REV MON|RT REV|RTREV BISHOP|SBLT|SEN|SGT|SIR|SMN|SNR CONST|SQN LDR|SR|SSGT|SUPR|SWAMI|TF|VADM|VERY REV|VICAR|VISCOUNT|WG CDR|WO|WO1|WO2)$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000459",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Title is invalid",
                    Location = "/xbrli:xbrl/tns:TrusteeOrDirector/tns:PersonNameDetails" + OccurrenceIndex(report.TrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RPOccurrenceIndex) + "/tns:PersonNameDetails.Title.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.000459" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR166", Value = GetValueOrEmpty(report.SMSFAR166) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.000459
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN000459_3()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.000459
            Title is invalid
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.Title.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.Title.Text <> SET(DOMAIN(TITLE CODES)))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^SMSFAR10 <> BLANK) AND (NotInDomain(^SMSFAR10, 'TITLE CODES'))
    
            Data Elements:
    
            ^SMSFAR10 = Auditor:PersonNameDetails:PersonNameDetails.Title.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.SMSFAR10) != true && !(IsMatch(report.SMSFAR10, @"^(2LT|AB|ABBOT|AC|ACM|ADM|AIR CDRE|ALDERMAN|AM|ARCHBISHOP|ARCHDEACON|ASSOC PROF|AVM|BARON|BARONESS|BISHOP|BR|BRIG|CANON|CAPT|CARDINAL|CDRE|CDT|CHAP|CMDR|CMM|COL|CONST|COUNT|COUNTESS|CPL|CPO|DAME|DEACON|DEACONESS|DEAN|DEPUTY SUPT|DR|DUCHESS|DUKE|EARL|EF|FLGOFF|FLT LT|FR|FSGT|GEN|GNR|GP CAPT|HON JUDGE|HON JUST|HON|HRH|INSP|JUDGE|JUST|LAC|LACW|LADY|LBDR|LCPL|LORD|LS|LT CMDR|LT COL|LT GEN|LT|MADAM|MAJ GEN|MAJ|MASTER|MATRON|MAYOR|MAYORESS|MIDN|MISS|MON|MOST REV|MR|MRS|MS|PASTOR|PATRIARCH|PLT OFF|PO|PRIOR|PROF|PTE|RABBI|RADM|RECTOR|REV|RF|RT HON|RT REV MON|RT REV|RTREV BISHOP|SBLT|SEN|SGT|SIR|SMN|SNR CONST|SQN LDR|SR|SSGT|SUPR|SWAMI|TF|VADM|VERY REV|VICAR|VISCOUNT|WG CDR|WO|WO1|WO2)$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000459",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Title is invalid",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_Contact_C_AuditorOccurrenceIndex) + "/tns:PersonNameDetails.Title.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.000459" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR10", Value = GetValueOrEmpty(report.SMSFAR10) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP(Instant)
    
            ^OtherTFN = RP(Instant)
    
            ^TFN = RP
            */
            assertion = (report.RPInstantExists != false && report.RPInstantIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPInstantOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.Accumulation
    
            ^OtherTFN = RP:RP.Accumulation
    
            ^TFN = RP
            */
            assertion = (report.RPAccumulationExists != false && report.RPAccumulationIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPAccumulationOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_2()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.Closing
    
            ^OtherTFN = RP:RP.Closing
    
            ^TFN = RP
            */
            assertion = (report.RPClosingExists != false && report.RPClosingIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPClosingOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_3()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP.Closing(Instant)
    
            ^OtherTFN = RP.Closing(Instant)
    
            ^TFN = RP
            */
            assertion = (report.RPClosingInstantExists != false && report.RPClosingInstantIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPClosingInstantOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_4()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP.Closing:RP.Closing.Accumulation
    
            ^OtherTFN = RP.Closing:RP.Closing.Accumulation
    
            ^TFN = RP
            */
            assertion = (report.RPClosingAccumulationExists != false && report.RPClosingAccumulationIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPClosingAccumulationOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_5()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP.Closing:RP.Closing.JAUS(Instant)
    
            ^OtherTFN = RP.Closing:RP.Closing.JAUS(Instant)
    
            ^TFN = RP
            */
            assertion = (report.RPClosingJAUSInstantExists != false && report.RPClosingJAUSInstantIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPClosingJAUSInstantOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_6()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP.Closing:RP.Closing.JFOREIGN(Instant)
    
            ^OtherTFN = RP.Closing:RP.Closing.JFOREIGN(Instant)
    
            ^TFN = RP
            */
            assertion = (report.RPClosingJFOREIGNInstantExists != false && report.RPClosingJFOREIGNInstantIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPClosingJFOREIGNInstantOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_7()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP.Closing:RP.Closing.Retirement
    
            ^OtherTFN = RP.Closing:RP.Closing.Retirement
    
            ^TFN = RP
            */
            assertion = (report.RPClosingRetirementExists != false && report.RPClosingRetirementIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPClosingRetirementOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_8()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.Deductible
    
            ^OtherTFN = RP:RP.Deductible
    
            ^TFN = RP
            */
            assertion = (report.RPDeductibleExists != false && report.RPDeductibleIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPDeductibleOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_9()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP.Deductible:RP.Deductible.JAUS
    
            ^OtherTFN = RP.Deductible:RP.Deductible.JAUS
    
            ^TFN = RP
            */
            assertion = (report.RPDeductibleJAUSExists != false && report.RPDeductibleJAUSIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPDeductibleJAUSOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_10()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP.Deductible:RP.Deductible.JFOREIGN
    
            ^OtherTFN = RP.Deductible:RP.Deductible.JFOREIGN
    
            ^TFN = RP
            */
            assertion = (report.RPDeductibleJFOREIGNExists != false && report.RPDeductibleJFOREIGNIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPDeductibleJFOREIGNOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_11()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.IncomeTaxRefund
    
            ^OtherTFN = RP:RP.IncomeTaxRefund
    
            ^TFN = RP
            */
            assertion = (report.RPIncomeTaxRefundExists != false && report.RPIncomeTaxRefundIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPIncomeTaxRefundOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_12()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.Investor
    
            ^OtherTFN = RP:RP.Investor
    
            ^TFN = RP
            */
            assertion = (report.RPInvestorExists != false && report.RPInvestorIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPInvestorOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_13()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP.Investor:RP.Investor.YPrevious
    
            ^OtherTFN = RP.Investor:RP.Investor.YPrevious
    
            ^TFN = RP
            */
            assertion = (report.RPInvestorYPreviousExists != false && report.RPInvestorYPreviousIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPInvestorYPreviousOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_14()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.JAUS
    
            ^OtherTFN = RP:RP.JAUS
    
            ^TFN = RP
            */
            assertion = (report.RPJAUSExists != false && report.RPJAUSIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_15()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP.JAUS:RP.JAUS.AustNonResidentialRealProperty
    
            ^OtherTFN = RP.JAUS:RP.JAUS.AustNonResidentialRealProperty
    
            ^TFN = RP
            */
            assertion = (report.RPJAUSAustNonResidentialRealPropertyExists != false && report.RPJAUSAustNonResidentialRealPropertyIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSAustNonResidentialRealPropertyOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_16()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP.JAUS:RP.JAUS.AustralianShares
    
            ^OtherTFN = RP.JAUS:RP.JAUS.AustralianShares
    
            ^TFN = RP
            */
            assertion = (report.RPJAUSAustralianSharesExists != false && report.RPJAUSAustralianSharesIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSAustralianSharesOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_17()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP.JAUS:RP.JAUS.AustResidentialRealProperty
    
            ^OtherTFN = RP.JAUS:RP.JAUS.AustResidentialRealProperty
    
            ^TFN = RP
            */
            assertion = (report.RPJAUSAustResidentialRealPropertyExists != false && report.RPJAUSAustResidentialRealPropertyIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSAustResidentialRealPropertyOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_18()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP.JAUS:RP.JAUS.Other
    
            ^OtherTFN = RP.JAUS:RP.JAUS.Other
    
            ^TFN = RP
            */
            assertion = (report.RPJAUSOtherExists != false && report.RPJAUSOtherIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSOtherOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_19()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP.JAUS:RP.JAUS.OtherShares
    
            ^OtherTFN = RP.JAUS:RP.JAUS.OtherShares
    
            ^TFN = RP
            */
            assertion = (report.RPJAUSOtherSharesExists != false && report.RPJAUSOtherSharesIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSOtherSharesOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_20()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP.JAUS:RP.JAUS.OverseasRealProperty
    
            ^OtherTFN = RP.JAUS:RP.JAUS.OverseasRealProperty
    
            ^TFN = RP
            */
            assertion = (report.RPJAUSOverseasRealPropertyExists != false && report.RPJAUSOverseasRealPropertyIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSOverseasRealPropertyOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_21()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.JFOREIGN
    
            ^OtherTFN = RP:RP.JFOREIGN
    
            ^TFN = RP
            */
            assertion = (report.RPJFOREIGNExists != false && report.RPJFOREIGNIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJFOREIGNOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_22()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.LimitedPartners
    
            ^OtherTFN = RP:RP.LimitedPartners
    
            ^TFN = RP
            */
            assertion = (report.RPLimitedPartnersExists != false && report.RPLimitedPartnersIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPLimitedPartnersOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_23()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP.LimitedPartners:RP.LimitedPartners.YPrevious
    
            ^OtherTFN = RP.LimitedPartners:RP.LimitedPartners.YPrevious
    
            ^TFN = RP
            */
            assertion = (report.RPLimitedPartnersYPreviousExists != false && report.RPLimitedPartnersYPreviousIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPLimitedPartnersYPreviousOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_24()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.NonDeductible
    
            ^OtherTFN = RP:RP.NonDeductible
    
            ^TFN = RP
            */
            assertion = (report.RPNonDeductibleExists != false && report.RPNonDeductibleIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPNonDeductibleOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_25()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP.NonDeductible:RP.NonDeductible.JAUS
    
            ^OtherTFN = RP.NonDeductible:RP.NonDeductible.JAUS
    
            ^TFN = RP
            */
            assertion = (report.RPNonDeductibleJAUSExists != false && report.RPNonDeductibleJAUSIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPNonDeductibleJAUSOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_26()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP.NonDeductible:RP.NonDeductible.JFOREIGN
    
            ^OtherTFN = RP.NonDeductible:RP.NonDeductible.JFOREIGN
    
            ^TFN = RP
            */
            assertion = (report.RPNonDeductibleJFOREIGNExists != false && report.RPNonDeductibleJFOREIGNIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPNonDeductibleJFOREIGNOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_27()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.Retirement
    
            ^OtherTFN = RP:RP.Retirement
    
            ^TFN = RP
            */
            assertion = (report.RPRetirementExists != false && report.RPRetirementIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPRetirementOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_28()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.Superannuation
    
            ^OtherTFN = RP:RP.Superannuation
    
            ^TFN = RP
            */
            assertion = (report.RPSuperannuationExists != false && report.RPSuperannuationIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPSuperannuationOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_29()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.TOFA
    
            ^OtherTFN = RP:RP.TOFA
    
            ^TFN = RP
            */
            assertion = (report.RPTOFAExists != false && report.RPTOFAIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPTOFAOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.001023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN001023_30()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.001023
            TFN in entity identifier of context does not match the TFN provided for the Reporting Party
    
            Legacy Rule Format:
            IF (entity.identifier.TFN WHERE CONTEXT(ALL)) <> RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^OtherTFN <> ^TFN
    
            Data Elements:
    
            ^Context = RP:RP.Y0Plus.Closing(Instant)
    
            ^OtherTFN = RP:RP.Y0Plus.Closing(Instant)
    
            ^TFN = RP
            */
            assertion = (report.RPY0PlusClosingInstantExists != false && report.RPY0PlusClosingInstantIdentifierTFN != report.RPIdentifierTFN);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unexpected TFN mismatch with Reporting Party",
                    LongDescription = @"TFN in entity identifier of context does not match the TFN provided for the Reporting Party",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0PlusClosingInstantOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.001023" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.010028
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN010028()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.010028
            If the return is an amendment, sequence number, type and reason must be present
    
            Legacy Rule Format:
            IF pyin.xx.xx:Report.Amendment.Indicator = TRUE AND (pyin.xx.xx:Report.AmendmentSequence.Number = NULL OR pyin.xx.xx:Report.AmendmentType.Code = NULLORBLANK OR pyin.xx.xx:Report.AmendmentReason.Text = NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^SMSFAR186 = TRUE AND (^SMSFAR291 = NULL OR ^SMSFAR289 = BLANK OR ^SMSFAR290 = BLANK)
    
            Data Elements:
    
            ^SMSFAR186 = RP:Report.Amendment.Indicator
    
            ^SMSFAR291 = RP:Report.AmendmentSequence.Number
    
            ^SMSFAR289 = RP:Report.AmendmentType.Code
    
            ^SMSFAR290 = RP:Report.AmendmentReason.Text
            */
            assertion = (report.SMSFAR186 == true && (report.SMSFAR291 == null || string.IsNullOrWhiteSpace(report.SMSFAR289) == true || string.IsNullOrWhiteSpace(report.SMSFAR290) == true));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010028",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"If the return is an amendment, sequence number, type and reason must be present",
                    LongDescription = @"If the Amendment Indicator is 'yes' (true), then the Amendment Sequence Number, Amendment Type and Amendment Reason must be present and not blank.",
                    Location = "/xbrli:xbrl/tns:Report.Amendment.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.010028" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR186", Value = GetValueOrEmpty(report.SMSFAR186) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR291", Value = report.SMSFAR291.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR289", Value = GetValueOrEmpty(report.SMSFAR289) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR290", Value = GetValueOrEmpty(report.SMSFAR290) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.010029
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN010029()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.010029
            If the return is an amendment, Amendment Indicator must be 'yes' (true)
    
            Legacy Rule Format:
            IF pyin.xx.xx:Report.Amendment.Indicator <> TRUE AND (pyin.xx.xx:Report.AmendmentSequence.Number <> NULL OR pyin.xx.xx:Report.AmendmentType.Code <> NULLORBLANK OR pyin.xx.xx:Report.AmendmentReason.Text <> NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^SMSFAR186 <> TRUE AND (^SMSFAR291 <> NULL OR ^SMSFAR289 <> BLANK OR ^SMSFAR290 <> BLANK)
    
            Data Elements:
    
            ^SMSFAR186 = RP:Report.Amendment.Indicator
    
            ^SMSFAR291 = RP:Report.AmendmentSequence.Number
    
            ^SMSFAR289 = RP:Report.AmendmentType.Code
    
            ^SMSFAR290 = RP:Report.AmendmentReason.Text
            */
            assertion = (report.SMSFAR186 != true && (report.SMSFAR291 != null || string.IsNullOrWhiteSpace(report.SMSFAR289) != true || string.IsNullOrWhiteSpace(report.SMSFAR290) != true));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"If the return is an amendment, Amendment Indicator must be 'yes' (true)",
                    LongDescription = @"If an Amendment Sequence Number, Amendment Type or Amendment Reason is present, then the Amendment Indicator must be 'yes' (true)",
                    Location = "/xbrli:xbrl/tns:Report.Amendment.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.010029" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR186", Value = GetValueOrEmpty(report.SMSFAR186) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR291", Value = report.SMSFAR291.GetValueOrDefault().ToString() });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR289", Value = GetValueOrEmpty(report.SMSFAR289) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR290", Value = GetValueOrEmpty(report.SMSFAR290) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.010030
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN010030()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.010030
            Amendment sequence number must be a number between 1 and 9
    
            Legacy Rule Format:
            IF pyin.xx.xx:Report.AmendmentSequence.Number <> NULL AND pyin.xx.xx:Report.AmendmentSequence.Number <> SET(1-9)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^SMSFAR291 <> NULL AND (^SMSFAR291 < 1 OR ^SMSFAR291 > 9)
    
            Data Elements:
    
            ^SMSFAR291 = RP:Report.AmendmentSequence.Number
            */
            assertion = (report.SMSFAR291 != null && (report.SMSFAR291.GetValueOrDefault() < 1 || report.SMSFAR291.GetValueOrDefault() > 9));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010030",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Amendment sequence number must be a number between 1 and 9",
                    Location = "/xbrli:xbrl/tns:Report.AmendmentSequence.Number",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.010030" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR291", Value = report.SMSFAR291.GetValueOrDefault().ToString() });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.010031
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN010031()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.010031
            Amendment Type must be '3' - ATO Error, or  '4' - Taxpayer Error
    
            Legacy Rule Format:
            IF pyin.xx.xx:Report.AmendmentType.Code <> NULL AND pyin.xx.xx:Report.AmendmentType.Code <> SET("3", "4")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^SMSFAR289 <> NULL AND NotInSet(^SMSFAR289, '"3", "4"')
    
            Data Elements:
    
            ^SMSFAR289 = RP:Report.AmendmentType.Code
            */
            assertion = (report.SMSFAR289 != null && !(IsMatch(report.SMSFAR289, @"^(3|4)$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010031",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Amendment Type must be '3' - Agency Error, or  '4' - Taxpayer Error",
                    Location = "/xbrli:xbrl/tns:Report.AmendmentType.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.010031" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR289", Value = GetValueOrEmpty(report.SMSFAR289) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.408010
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN408010()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.408010
            Name field cannot contain 'space hyphen space'.
    
            Legacy Rule Format:
            WHERE IN TUPLE(personunstructuredname1.xx.xx:PersonUnstructuredName)
            IF (pyde.xx.xx:PersonUnstructuredName.FullName.Text CONTAINS " - ")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^SMSFAR280, ' - ')
    
            Data Elements:
    
            ^SMSFAR280 = INT:Declaration:PersonUnstructuredName:PersonUnstructuredName.FullName.Text WHERE [PersonUnstructuredName.Usage.Code='DeclarationSignatory']
            */
            assertion = (report.SMSFAR280 == null ? false : report.SMSFAR280.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.408010",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Name field cannot contain 'space hyphen space'.",
                    LongDescription = @"Name field cannot contain 'space hyphen space'.",
                    Location = "/xbrli:xbrl/tns:Declaration" + OccurrenceIndex(report.DeclarationCollection_TrueAndCorrect_INTOccurrenceIndex) + "/tns:PersonUnstructuredName" + OccurrenceIndex(report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTOccurrenceIndex) + "/tns:PersonUnstructuredName.FullName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.408010" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR280", Value = report.SMSFAR280 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.408010
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN408010_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.408010
            Name field cannot contain 'space hyphen space'.
    
            Legacy Rule Format:
            WHERE IN TUPLE(personunstructuredname1.xx.xx:PersonUnstructuredName)
            IF (pyde.xx.xx:PersonUnstructuredName.FullName.Text CONTAINS " - ")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^SMSFAR278, ' - ')
    
            Data Elements:
    
            ^SMSFAR278 = RP:Declaration:PersonUnstructuredName:PersonUnstructuredName.FullName.Text WHERE [PersonUnstructuredName.Usage.Code='DeclarationSignatory']
            */
            assertion = (report.SMSFAR278 == null ? false : report.SMSFAR278.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.408010",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Name field cannot contain 'space hyphen space'.",
                    LongDescription = @"Name field cannot contain 'space hyphen space'.",
                    Location = "/xbrli:xbrl/tns:Declaration" + OccurrenceIndex(report.DeclarationCollection_TrueAndCorrect_RPOccurrenceIndex) + "/tns:PersonUnstructuredName" + OccurrenceIndex(report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPOccurrenceIndex) + "/tns:PersonUnstructuredName.FullName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.408010" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR278", Value = report.SMSFAR278 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410002
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410002()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410002
            Address Line 2 must be present if Address Line 1 contains 'C/-'
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF ((pyde.xx.xx:AddressDetails.Line1.Text CONTAINS "C/-") AND (pyde.xx.xx:AddressDetails.Line2.Text = NULLORBLANK))
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ((Contains(^SMSFAR6, 'C/-')) AND (^SMSFAR239 = BLANK))
    
            Data Elements:
    
            ^SMSFAR6 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
    
            ^SMSFAR239 = RP:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = ((report.SMSFAR6 == null ? false : report.SMSFAR6.ToUpperInvariant().Contains(@"C/-")) && string.IsNullOrWhiteSpace(report.SMSFAR239) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410002",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 2 must be present if Address Line 1 contains 'C/-'",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410002" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR6", Value = report.SMSFAR6 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR239", Value = GetValueOrEmpty(report.SMSFAR239) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410002
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410002_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410002
            Address Line 2 must be present if Address Line 1 contains 'C/-'
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF ((pyde.xx.xx:AddressDetails.Line1.Text CONTAINS "C/-") AND (pyde.xx.xx:AddressDetails.Line2.Text = NULLORBLANK))
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ((Contains(^SMSFAR19, 'C/-')) AND (^SMSFAR240 = BLANK))
    
            Data Elements:
    
            ^SMSFAR19 = Auditor:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
    
            ^SMSFAR240 = Auditor:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = ((report.SMSFAR19 == null ? false : report.SMSFAR19.ToUpperInvariant().Contains(@"C/-")) && string.IsNullOrWhiteSpace(report.SMSFAR240) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410002",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 2 must be present if Address Line 1 contains 'C/-'",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_AuditorOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410002" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR19", Value = report.SMSFAR19 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR240", Value = GetValueOrEmpty(report.SMSFAR240) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410008
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410008()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410008
            Address Line 3 must be left blank
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line3.Text <> NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^SMSFAR241 <> BLANK
    
            Data Elements:
    
            ^SMSFAR241 = RP:AddressDetails:AddressDetails.Line3.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.SMSFAR241) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410008",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 3 must be left blank",
                    LongDescription = @"Lines 3 and 4 of the address are not used on this form",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line3.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410008" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR241", Value = GetValueOrEmpty(report.SMSFAR241) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410008
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410008_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410008
            Address Line 3 must be left blank
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line3.Text <> NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^SMSFAR242 <> BLANK
    
            Data Elements:
    
            ^SMSFAR242 = Auditor:AddressDetails:AddressDetails.Line3.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.SMSFAR242) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410008",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 3 must be left blank",
                    LongDescription = @"Lines 3 and 4 of the address are not used on this form",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_AuditorOccurrenceIndex) + "/tns:AddressDetails.Line3.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410008" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR242", Value = GetValueOrEmpty(report.SMSFAR242) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410013
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410013()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410013
            Address Line 4 must be left blank
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line4.Text <> NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^SMSFAR243 <> BLANK
    
            Data Elements:
    
            ^SMSFAR243 = RP:AddressDetails:AddressDetails.Line4.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.SMSFAR243) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410013",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 4 must be left blank",
                    LongDescription = @"Lines 3 and 4 of the address are not used on this form",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line4.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410013" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR243", Value = GetValueOrEmpty(report.SMSFAR243) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410013
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410013_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410013
            Address Line 4 must be left blank
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line4.Text <> NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^SMSFAR244 <> BLANK
    
            Data Elements:
    
            ^SMSFAR244 = Auditor:AddressDetails:AddressDetails.Line4.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.SMSFAR244) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410013",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 4 must be left blank",
                    LongDescription = @"Lines 3 and 4 of the address are not used on this form",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_AuditorOccurrenceIndex) + "/tns:AddressDetails.Line4.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410013" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR244", Value = GetValueOrEmpty(report.SMSFAR244) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410031
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410031(SMSFAR2026.MemberInformation memberInformation, int itemIndex1)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410031
            TFN must pass TFN algorithm.

            Legacy Rule Format:
            IF (pyid.xx.xx:Identifiers.TaxFileNumber.Identifier <> NULLORBLANK) AND (TFNALGORITHM (pyid.xx.xx:Identifiers.TaxFileNumber.Identifier) = FALSE)
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^SMSFAR201 <> BLANK) AND (FailsTFNAlgorithm(^SMSFAR201))

            Data Elements:

            ^SMSFAR201 = RP:MemberInformation:Identifiers.TaxFileNumber.Identifier
            */
            assertion = (string.IsNullOrWhiteSpace(memberInformation.SMSFAR201) != true && FailsTFNAlgorithm(memberInformation.SMSFAR201));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410031",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"TFN must be a valid TFN",
                    Location = "/xbrli:xbrl/tns:MemberInformation" + OccurrenceIndex(memberInformation.OccurrenceIndex) + "/tns:Identifiers.TaxFileNumber.Identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410031" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR201", Value = GetValueOrEmpty(memberInformation.SMSFAR201) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410038
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410038()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410038
            Non-individual name field cannot contain 'Pship', 'P'ship or 'P/Ship'
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text STARTSWITH "T/A ") AND (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text ENDSWITH SET(" Pship"," P'ship"," P/ship"))
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (StartsWith(^SMSFAR182, 'T/A ')) AND (EndsWithSet(^SMSFAR182, '" Pship"," P'ship"," P/ship"'))
    
            Data Elements:
    
            ^SMSFAR182 = INT:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
            */
            assertion = (IsMatch(report.SMSFAR182, @"^(T/A )", RegexOptions.IgnoreCase) && IsMatch(report.SMSFAR182, @"( Pship| P'ship| P/ship)$", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410038",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-individual name field cannot start with 'T/A' and end with 'Pship', 'P'ship or 'P/Ship'",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_MN_C_INTOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410038" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR182", Value = report.SMSFAR182 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410038
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410038_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410038
            Non-individual name field cannot contain 'Pship', 'P'ship or 'P/Ship'
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text STARTSWITH "T/A ") AND (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text ENDSWITH SET(" Pship"," P'ship"," P/ship"))
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (StartsWith(^SMSFAR174, 'T/A ')) AND (EndsWithSet(^SMSFAR174, '" Pship"," P'ship"," P/ship"'))
    
            Data Elements:
    
            ^SMSFAR174 = RP:TrusteeOrDirector:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
            */
            assertion = (IsMatch(report.SMSFAR174, @"^(T/A )", RegexOptions.IgnoreCase) && IsMatch(report.SMSFAR174, @"( Pship| P'ship| P/ship)$", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410038",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-individual name field cannot start with 'T/A' and end with 'Pship', 'P'ship or 'P/Ship'",
                    Location = "/xbrli:xbrl/tns:TrusteeOrDirector/tns:OrganisationNameDetails" + OccurrenceIndex(report.TrusteeOrDirector_OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410038" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR174", Value = report.SMSFAR174 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410038
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410038_2()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410038
            Non-individual name field cannot contain 'Pship', 'P'ship or 'P/Ship'
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text STARTSWITH "T/A ") AND (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text ENDSWITH SET(" Pship"," P'ship"," P/ship"))
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (StartsWith(^SMSFAR4, 'T/A ')) AND (EndsWithSet(^SMSFAR4, '" Pship"," P'ship"," P/ship"'))
    
            Data Elements:
    
            ^SMSFAR4 = RP:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
            */
            assertion = (IsMatch(report.SMSFAR4, @"^(T/A )", RegexOptions.IgnoreCase) && IsMatch(report.SMSFAR4, @"( Pship| P'ship| P/ship)$", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410038",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non-individual name field cannot start with 'T/A' and end with 'Pship', 'P'ship or 'P/Ship'",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410038" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR4", Value = report.SMSFAR4 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410039
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410039()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410039
            Organisation name must contain at least one alpha or numeric character
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text  <> NULLORBLANK) AND (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text  DOES NOT CONTAIN SET("a-z", "A-Z", "0-9"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^SMSFAR182  <> BLANK) AND (NotContainsSet(^SMSFAR182, '"a-z", "A-Z", "0-9"'))
    
            Data Elements:
    
            ^SMSFAR182 = INT:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.SMSFAR182) != true && !(IsMatch(report.SMSFAR182, @"[a-z]|[A-Z]|[0-9]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410039",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Organisation name must contain at least one alpha or numeric character",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_MN_C_INTOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410039" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR182", Value = report.SMSFAR182 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410039
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410039_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410039
            Organisation name must contain at least one alpha or numeric character
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text  <> NULLORBLANK) AND (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text  DOES NOT CONTAIN SET("a-z", "A-Z", "0-9"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^SMSFAR174  <> BLANK) AND (NotContainsSet(^SMSFAR174, '"a-z", "A-Z", "0-9"'))
    
            Data Elements:
    
            ^SMSFAR174 = RP:TrusteeOrDirector:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.SMSFAR174) != true && !(IsMatch(report.SMSFAR174, @"[a-z]|[A-Z]|[0-9]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410039",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Organisation name must contain at least one alpha or numeric character",
                    Location = "/xbrli:xbrl/tns:TrusteeOrDirector/tns:OrganisationNameDetails" + OccurrenceIndex(report.TrusteeOrDirector_OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410039" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR174", Value = report.SMSFAR174 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410039
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410039_2()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410039
            Organisation name must contain at least one alpha or numeric character
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text  <> NULLORBLANK) AND (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text  DOES NOT CONTAIN SET("a-z", "A-Z", "0-9"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^SMSFAR4  <> BLANK) AND (NotContainsSet(^SMSFAR4, '"a-z", "A-Z", "0-9"'))
    
            Data Elements:
    
            ^SMSFAR4 = RP:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.SMSFAR4) != true && !(IsMatch(report.SMSFAR4, @"[a-z]|[A-Z]|[0-9]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410039",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Organisation name must contain at least one alpha or numeric character",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410039" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR4", Value = report.SMSFAR4 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410063
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410063()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410063
            Given Name must contain at least one alphabetical character
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.GivenName.Text DOES NOT CONTAIN SET("A-Z","a-z"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^SMSFAR180 <> BLANK) AND (NotContainsSet(^SMSFAR180, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^SMSFAR180 = INT:PersonNameDetails:PersonNameDetails.GivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.SMSFAR180) != true && !(IsMatch(report.SMSFAR180, @"[A-Z]|[a-z]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410063",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Given Name must contain at least one alphabetical character",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_Contact_C_INTOccurrenceIndex) + "/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410063" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR180", Value = report.SMSFAR180 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410063
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410063_1(SMSFAR2026.MemberInformation memberInformation, int itemIndex1)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410063
            Given Name must contain at least one alphabetical character

            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.GivenName.Text DOES NOT CONTAIN SET("A-Z","a-z"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^SMSFAR199 <> BLANK) AND (NotContainsSet(^SMSFAR199, '"A-Z","a-z"'))

            Data Elements:

            ^SMSFAR199 = RP:MemberInformation:PersonNameDetails:PersonNameDetails.GivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Identifier'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(memberInformation.SMSFAR199) != true && !(IsMatch(memberInformation.SMSFAR199, @"[A-Z]|[a-z]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410063",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Given Name must contain at least one alphabetical character",
                    Location = "/xbrli:xbrl/tns:MemberInformation" + OccurrenceIndex(memberInformation.OccurrenceIndex) + "/tns:PersonNameDetails" + OccurrenceIndex(memberInformation.MemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RPOccurrenceIndex) + "/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410063" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR199", Value = GetValueOrEmpty(memberInformation.SMSFAR199) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410063
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410063_2()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410063
            Given Name must contain at least one alphabetical character
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.GivenName.Text DOES NOT CONTAIN SET("A-Z","a-z"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^SMSFAR169 <> BLANK) AND (NotContainsSet(^SMSFAR169, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^SMSFAR169 = RP:TrusteeOrDirector:PersonNameDetails:PersonNameDetails.GivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.SMSFAR169) != true && !(IsMatch(report.SMSFAR169, @"[A-Z]|[a-z]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410063",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Given Name must contain at least one alphabetical character",
                    Location = "/xbrli:xbrl/tns:TrusteeOrDirector/tns:PersonNameDetails" + OccurrenceIndex(report.TrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RPOccurrenceIndex) + "/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410063" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR169", Value = report.SMSFAR169 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410063
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410063_3()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410063
            Given Name must contain at least one alphabetical character
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.GivenName.Text DOES NOT CONTAIN SET("A-Z","a-z"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^SMSFAR13 <> BLANK) AND (NotContainsSet(^SMSFAR13, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^SMSFAR13 = Auditor:PersonNameDetails:PersonNameDetails.GivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.SMSFAR13) != true && !(IsMatch(report.SMSFAR13, @"[A-Z]|[a-z]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410063",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Given Name must contain at least one alphabetical character",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_Contact_C_AuditorOccurrenceIndex) + "/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410063" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR13", Value = report.SMSFAR13 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410131
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410131()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410131
            Other Given Name must contain at least one alphabetical character
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text DOES NOT CONTAIN SET("A-Z","a-z"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^SMSFAR181 <> BLANK) AND (NotContainsSet(^SMSFAR181, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^SMSFAR181 = INT:PersonNameDetails:PersonNameDetails.OtherGivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.SMSFAR181) != true && !(IsMatch(report.SMSFAR181, @"[A-Z]|[a-z]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410131",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other Given Name must contain at least one alphabetical character",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_Contact_C_INTOccurrenceIndex) + "/tns:PersonNameDetails.OtherGivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410131" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR181", Value = GetValueOrEmpty(report.SMSFAR181) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410131
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410131_1(SMSFAR2026.MemberInformation memberInformation, int itemIndex1)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410131
            Other Given Name must contain at least one alphabetical character

            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text DOES NOT CONTAIN SET("A-Z","a-z"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^SMSFAR200 <> BLANK) AND (NotContainsSet(^SMSFAR200, '"A-Z","a-z"'))

            Data Elements:

            ^SMSFAR200 = RP:MemberInformation:PersonNameDetails:PersonNameDetails.OtherGivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Identifier'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(memberInformation.SMSFAR200) != true && !(IsMatch(memberInformation.SMSFAR200, @"[A-Z]|[a-z]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410131",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other Given Name must contain at least one alphabetical character",
                    Location = "/xbrli:xbrl/tns:MemberInformation" + OccurrenceIndex(memberInformation.OccurrenceIndex) + "/tns:PersonNameDetails" + OccurrenceIndex(memberInformation.MemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RPOccurrenceIndex) + "/tns:PersonNameDetails.OtherGivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410131" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR200", Value = GetValueOrEmpty(memberInformation.SMSFAR200) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410131
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410131_2()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410131
            Other Given Name must contain at least one alphabetical character
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text DOES NOT CONTAIN SET("A-Z","a-z"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^SMSFAR170 <> BLANK) AND (NotContainsSet(^SMSFAR170, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^SMSFAR170 = RP:TrusteeOrDirector:PersonNameDetails:PersonNameDetails.OtherGivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.SMSFAR170) != true && !(IsMatch(report.SMSFAR170, @"[A-Z]|[a-z]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410131",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other Given Name must contain at least one alphabetical character",
                    Location = "/xbrli:xbrl/tns:TrusteeOrDirector/tns:PersonNameDetails" + OccurrenceIndex(report.TrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RPOccurrenceIndex) + "/tns:PersonNameDetails.OtherGivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410131" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR170", Value = GetValueOrEmpty(report.SMSFAR170) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410131
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410131_3()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410131
            Other Given Name must contain at least one alphabetical character
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text DOES NOT CONTAIN SET("A-Z","a-z"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^SMSFAR14 <> BLANK) AND (NotContainsSet(^SMSFAR14, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^SMSFAR14 = Auditor:PersonNameDetails:PersonNameDetails.OtherGivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.SMSFAR14) != true && !(IsMatch(report.SMSFAR14, @"[A-Z]|[a-z]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410131",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other Given Name must contain at least one alphabetical character",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_Contact_C_AuditorOccurrenceIndex) + "/tns:PersonNameDetails.OtherGivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410131" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR14", Value = GetValueOrEmpty(report.SMSFAR14) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410194
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410194()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410194
            Address Line 1 must not exceed 38 characters
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:AddressDetails.Line1.Text) > 38
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^SMSFAR6) > 38
    
            Data Elements:
    
            ^SMSFAR6 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (Length(report.SMSFAR6) > 38);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410194",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 1 must not exceed 38 characters",
                    LongDescription = @"Although SBR sets a limit of 50 characters for the Line 1 of an address, the maximum allowable for this form is 38",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410194" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR6", Value = report.SMSFAR6 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410194
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410194_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410194
            Address Line 1 must not exceed 38 characters
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:AddressDetails.Line1.Text) > 38
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^SMSFAR19) > 38
    
            Data Elements:
    
            ^SMSFAR19 = Auditor:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (Length(report.SMSFAR19) > 38);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410194",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 1 must not exceed 38 characters",
                    LongDescription = @"Although SBR sets a limit of 50 characters for the Line 1 of an address, the maximum allowable for this form is 38",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_AuditorOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410194" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR19", Value = report.SMSFAR19 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410195
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410195()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410195
            Address Line 2 must not exceed 38 characters
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:AddressDetails.Line2.Text) > 38
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^SMSFAR239) > 38
    
            Data Elements:
    
            ^SMSFAR239 = RP:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (Length(report.SMSFAR239) > 38);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410195",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 2 must not exceed 38 characters",
                    LongDescription = @"Although SBR sets a limit of 50 characters for the Line 2 of an address, the maximum allowable for this form is 38",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line2.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410195" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR239", Value = GetValueOrEmpty(report.SMSFAR239) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410195
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410195_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410195
            Address Line 2 must not exceed 38 characters
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:AddressDetails.Line2.Text) > 38
                 RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^SMSFAR240) > 38
    
            Data Elements:
    
            ^SMSFAR240 = Auditor:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (Length(report.SMSFAR240) > 38);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410195",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 2 must not exceed 38 characters",
                    LongDescription = @"Although SBR sets a limit of 50 characters for the Line 2 of an address, the maximum allowable for this form is 38",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_AuditorOccurrenceIndex) + "/tns:AddressDetails.Line2.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410195" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR240", Value = GetValueOrEmpty(report.SMSFAR240) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410200
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410200()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410200
            Family name cannot contain "Exec for", "Rep for" or "Trustee for"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text = FOUND("Exec for","Rep for","Trustee for"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^SMSFAR178, '"Exec for","Rep for","Trustee for"'))
    
            Data Elements:
    
            ^SMSFAR178 = INT:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.SMSFAR178, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000424",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_Contact_C_INTOccurrenceIndex) + "/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410200" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR178", Value = report.SMSFAR178 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410200
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410200_1(SMSFAR2026.MemberInformation memberInformation, int itemIndex1)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410200
            Family name cannot contain "Exec for", "Rep for" or "Trustee for"

            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text = FOUND("Exec for","Rep for","Trustee for"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^SMSFAR197, '"Exec for","Rep for","Trustee for"'))

            Data Elements:

            ^SMSFAR197 = RP:MemberInformation:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Identifier'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(memberInformation.SMSFAR197, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000424",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/xbrli:xbrl/tns:MemberInformation" + OccurrenceIndex(memberInformation.OccurrenceIndex) + "/tns:PersonNameDetails" + OccurrenceIndex(memberInformation.MemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RPOccurrenceIndex) + "/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410200" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR197", Value = memberInformation.SMSFAR197 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410200
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410200_2()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410200
            Family name cannot contain "Exec for", "Rep for" or "Trustee for"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text = FOUND("Exec for","Rep for","Trustee for"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^SMSFAR167, '"Exec for","Rep for","Trustee for"'))
    
            Data Elements:
    
            ^SMSFAR167 = RP:TrusteeOrDirector:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.SMSFAR167, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000424",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/xbrli:xbrl/tns:TrusteeOrDirector/tns:PersonNameDetails" + OccurrenceIndex(report.TrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RPOccurrenceIndex) + "/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410200" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR167", Value = report.SMSFAR167 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410200
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410200_3()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410200
            Family name cannot contain "Exec for", "Rep for" or "Trustee for"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text = FOUND("Exec for","Rep for","Trustee for"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^SMSFAR11, '"Exec for","Rep for","Trustee for"'))
    
            Data Elements:
    
            ^SMSFAR11 = Auditor:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.SMSFAR11, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000424",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_Contact_C_AuditorOccurrenceIndex) + "/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410200" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR11", Value = report.SMSFAR11 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410201
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410201()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410201
            Family name cannot contain "MR", "MRS", "MISS" or "MS"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text = FOUND("MR","MRS","MISS","MS"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^SMSFAR178, '"MR","MRS","MISS","MS"'))
    
            Data Elements:
    
            ^SMSFAR178 = INT:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.SMSFAR178, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000426",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_Contact_C_INTOccurrenceIndex) + "/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410201" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR178", Value = report.SMSFAR178 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410201
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410201_1(SMSFAR2026.MemberInformation memberInformation, int itemIndex1)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410201
            Family name cannot contain "MR", "MRS", "MISS" or "MS"

            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text = FOUND("MR","MRS","MISS","MS"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^SMSFAR197, '"MR","MRS","MISS","MS"'))

            Data Elements:

            ^SMSFAR197 = RP:MemberInformation:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Identifier'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(memberInformation.SMSFAR197, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000426",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/xbrli:xbrl/tns:MemberInformation" + OccurrenceIndex(memberInformation.OccurrenceIndex) + "/tns:PersonNameDetails" + OccurrenceIndex(memberInformation.MemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RPOccurrenceIndex) + "/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410201" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR197", Value = memberInformation.SMSFAR197 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410201
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410201_2()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410201
            Family name cannot contain "MR", "MRS", "MISS" or "MS"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text = FOUND("MR","MRS","MISS","MS"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^SMSFAR167, '"MR","MRS","MISS","MS"'))
    
            Data Elements:
    
            ^SMSFAR167 = RP:TrusteeOrDirector:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.SMSFAR167, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000426",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/xbrli:xbrl/tns:TrusteeOrDirector/tns:PersonNameDetails" + OccurrenceIndex(report.TrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RPOccurrenceIndex) + "/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410201" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR167", Value = report.SMSFAR167 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410201
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410201_3()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410201
            Family name cannot contain "MR", "MRS", "MISS" or "MS"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text = FOUND("MR","MRS","MISS","MS"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^SMSFAR11, '"MR","MRS","MISS","MS"'))
    
            Data Elements:
    
            ^SMSFAR11 = Auditor:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.SMSFAR11, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000426",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_Contact_C_AuditorOccurrenceIndex) + "/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410201" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR11", Value = report.SMSFAR11 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410203
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410203()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410203
            First name cannot contain "Exec for", "Rep for" or "Trustee for"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text = FOUND("Exec for","Rep for","Trustee for"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^SMSFAR180, '"Exec for","Rep for","Trustee for"'))
    
            Data Elements:
    
            ^SMSFAR180 = INT:PersonNameDetails:PersonNameDetails.GivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.SMSFAR180, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000437",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_Contact_C_INTOccurrenceIndex) + "/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410203" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR180", Value = report.SMSFAR180 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410203
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410203_1(SMSFAR2026.MemberInformation memberInformation, int itemIndex1)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410203
            First name cannot contain "Exec for", "Rep for" or "Trustee for"

            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text = FOUND("Exec for","Rep for","Trustee for"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^SMSFAR199, '"Exec for","Rep for","Trustee for"'))

            Data Elements:

            ^SMSFAR199 = RP:MemberInformation:PersonNameDetails:PersonNameDetails.GivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Identifier'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(memberInformation.SMSFAR199, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000437",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/xbrli:xbrl/tns:MemberInformation" + OccurrenceIndex(memberInformation.OccurrenceIndex) + "/tns:PersonNameDetails" + OccurrenceIndex(memberInformation.MemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RPOccurrenceIndex) + "/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410203" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR199", Value = GetValueOrEmpty(memberInformation.SMSFAR199) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410203
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410203_2()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410203
            First name cannot contain "Exec for", "Rep for" or "Trustee for"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text = FOUND("Exec for","Rep for","Trustee for"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^SMSFAR169, '"Exec for","Rep for","Trustee for"'))
    
            Data Elements:
    
            ^SMSFAR169 = RP:TrusteeOrDirector:PersonNameDetails:PersonNameDetails.GivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.SMSFAR169, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000437",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/xbrli:xbrl/tns:TrusteeOrDirector/tns:PersonNameDetails" + OccurrenceIndex(report.TrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RPOccurrenceIndex) + "/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410203" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR169", Value = report.SMSFAR169 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410203
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410203_3()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410203
            First name cannot contain "Exec for", "Rep for" or "Trustee for"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text = FOUND("Exec for","Rep for","Trustee for"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^SMSFAR13, '"Exec for","Rep for","Trustee for"'))
    
            Data Elements:
    
            ^SMSFAR13 = Auditor:PersonNameDetails:PersonNameDetails.GivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.SMSFAR13, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000437",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_Contact_C_AuditorOccurrenceIndex) + "/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410203" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR13", Value = report.SMSFAR13 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410204
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410204()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410204
            First name cannot contain "MR", "MRS", "MISS" or "MS"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text = FOUND("MR","MRS","MISS","MS"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^SMSFAR180, '"MR","MRS","MISS","MS"'))
    
            Data Elements:
    
            ^SMSFAR180 = INT:PersonNameDetails:PersonNameDetails.GivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.SMSFAR180, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000438",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_Contact_C_INTOccurrenceIndex) + "/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410204" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR180", Value = report.SMSFAR180 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410204
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410204_1(SMSFAR2026.MemberInformation memberInformation, int itemIndex1)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410204
            First name cannot contain "MR", "MRS", "MISS" or "MS"

            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text = FOUND("MR","MRS","MISS","MS"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^SMSFAR199, '"MR","MRS","MISS","MS"'))

            Data Elements:

            ^SMSFAR199 = RP:MemberInformation:PersonNameDetails:PersonNameDetails.GivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Identifier'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(memberInformation.SMSFAR199, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000438",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/xbrli:xbrl/tns:MemberInformation" + OccurrenceIndex(memberInformation.OccurrenceIndex) + "/tns:PersonNameDetails" + OccurrenceIndex(memberInformation.MemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RPOccurrenceIndex) + "/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410204" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR199", Value = GetValueOrEmpty(memberInformation.SMSFAR199) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410204
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410204_2()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410204
            First name cannot contain "MR", "MRS", "MISS" or "MS"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text = FOUND("MR","MRS","MISS","MS"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^SMSFAR169, '"MR","MRS","MISS","MS"'))
    
            Data Elements:
    
            ^SMSFAR169 = RP:TrusteeOrDirector:PersonNameDetails:PersonNameDetails.GivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.SMSFAR169, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000438",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/xbrli:xbrl/tns:TrusteeOrDirector/tns:PersonNameDetails" + OccurrenceIndex(report.TrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RPOccurrenceIndex) + "/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410204" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR169", Value = report.SMSFAR169 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410204
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410204_3()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410204
            First name cannot contain "MR", "MRS", "MISS" or "MS"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text = FOUND("MR","MRS","MISS","MS"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^SMSFAR13, '"MR","MRS","MISS","MS"'))
    
            Data Elements:
    
            ^SMSFAR13 = Auditor:PersonNameDetails:PersonNameDetails.GivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.SMSFAR13, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000438",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_Contact_C_AuditorOccurrenceIndex) + "/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410204" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR13", Value = report.SMSFAR13 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410205
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410205()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410205
            Address Line 1 must not contain 'AS ABOVE'
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line1.Text = FOUND("AS ABOVE"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^SMSFAR6, '"AS ABOVE"'))
    
            Data Elements:
    
            ^SMSFAR6 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.SMSFAR6, @"(^(AS ABOVE)$)|(^(.*)( AS ABOVE )(.*)$)|(^(AS ABOVE )(.*)$)|(^(.*)( AS ABOVE)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 1 must not contain 'AS ABOVE'",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410205" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR6", Value = report.SMSFAR6 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410205
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410205_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410205
            Address Line 1 must not contain 'AS ABOVE'
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line1.Text = FOUND("AS ABOVE"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^SMSFAR19, '"AS ABOVE"'))
    
            Data Elements:
    
            ^SMSFAR19 = Auditor:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.SMSFAR19, @"(^(AS ABOVE)$)|(^(.*)( AS ABOVE )(.*)$)|(^(AS ABOVE )(.*)$)|(^(.*)( AS ABOVE)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 1 must not contain 'AS ABOVE'",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_AuditorOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410205" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR19", Value = report.SMSFAR19 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410208
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410208()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410208
            Other given name cannot contain "Exec for", "Rep for" or "Trustee for"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text = FOUND("Exec for","Rep for","Trustee for"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^SMSFAR181, '"Exec for","Rep for","Trustee for"'))
    
            Data Elements:
    
            ^SMSFAR181 = INT:PersonNameDetails:PersonNameDetails.OtherGivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.SMSFAR181, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000449",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_Contact_C_INTOccurrenceIndex) + "/tns:PersonNameDetails.OtherGivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410208" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR181", Value = GetValueOrEmpty(report.SMSFAR181) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410208
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410208_1(SMSFAR2026.MemberInformation memberInformation, int itemIndex1)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410208
            Other given name cannot contain "Exec for", "Rep for" or "Trustee for"

            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text = FOUND("Exec for","Rep for","Trustee for"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^SMSFAR200, '"Exec for","Rep for","Trustee for"'))

            Data Elements:

            ^SMSFAR200 = RP:MemberInformation:PersonNameDetails:PersonNameDetails.OtherGivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Identifier'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(memberInformation.SMSFAR200, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000449",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/xbrli:xbrl/tns:MemberInformation" + OccurrenceIndex(memberInformation.OccurrenceIndex) + "/tns:PersonNameDetails" + OccurrenceIndex(memberInformation.MemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RPOccurrenceIndex) + "/tns:PersonNameDetails.OtherGivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410208" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR200", Value = GetValueOrEmpty(memberInformation.SMSFAR200) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410208
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410208_2()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410208
            Other given name cannot contain "Exec for", "Rep for" or "Trustee for"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text = FOUND("Exec for","Rep for","Trustee for"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^SMSFAR170, '"Exec for","Rep for","Trustee for"'))
    
            Data Elements:
    
            ^SMSFAR170 = RP:TrusteeOrDirector:PersonNameDetails:PersonNameDetails.OtherGivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.SMSFAR170, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000449",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/xbrli:xbrl/tns:TrusteeOrDirector/tns:PersonNameDetails" + OccurrenceIndex(report.TrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RPOccurrenceIndex) + "/tns:PersonNameDetails.OtherGivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410208" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR170", Value = GetValueOrEmpty(report.SMSFAR170) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410208
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410208_3()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410208
            Other given name cannot contain "Exec for", "Rep for" or "Trustee for"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text = FOUND("Exec for","Rep for","Trustee for"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^SMSFAR14, '"Exec for","Rep for","Trustee for"'))
    
            Data Elements:
    
            ^SMSFAR14 = Auditor:PersonNameDetails:PersonNameDetails.OtherGivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.SMSFAR14, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000449",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_Contact_C_AuditorOccurrenceIndex) + "/tns:PersonNameDetails.OtherGivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410208" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR14", Value = GetValueOrEmpty(report.SMSFAR14) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410209
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410209()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410209
            Other given name cannot contain "MR", "MRS", "MISS" or "MS"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text = FOUND("MR","MRS","MISS","MS"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^SMSFAR181, '"MR","MRS","MISS","MS"'))
    
            Data Elements:
    
            ^SMSFAR181 = INT:PersonNameDetails:PersonNameDetails.OtherGivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.SMSFAR181, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000450",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_Contact_C_INTOccurrenceIndex) + "/tns:PersonNameDetails.OtherGivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410209" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR181", Value = GetValueOrEmpty(report.SMSFAR181) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410209
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410209_1(SMSFAR2026.MemberInformation memberInformation, int itemIndex1)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410209
            Other given name cannot contain "MR", "MRS", "MISS" or "MS"

            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text = FOUND("MR","MRS","MISS","MS"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^SMSFAR200, '"MR","MRS","MISS","MS"'))

            Data Elements:

            ^SMSFAR200 = RP:MemberInformation:PersonNameDetails:PersonNameDetails.OtherGivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Identifier'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(memberInformation.SMSFAR200, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000450",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/xbrli:xbrl/tns:MemberInformation" + OccurrenceIndex(memberInformation.OccurrenceIndex) + "/tns:PersonNameDetails" + OccurrenceIndex(memberInformation.MemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RPOccurrenceIndex) + "/tns:PersonNameDetails.OtherGivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410209" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR200", Value = GetValueOrEmpty(memberInformation.SMSFAR200) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410209
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410209_2()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410209
            Other given name cannot contain "MR", "MRS", "MISS" or "MS"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text = FOUND("MR","MRS","MISS","MS"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^SMSFAR170, '"MR","MRS","MISS","MS"'))
    
            Data Elements:
    
            ^SMSFAR170 = RP:TrusteeOrDirector:PersonNameDetails:PersonNameDetails.OtherGivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.SMSFAR170, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000450",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/xbrli:xbrl/tns:TrusteeOrDirector/tns:PersonNameDetails" + OccurrenceIndex(report.TrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RPOccurrenceIndex) + "/tns:PersonNameDetails.OtherGivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410209" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR170", Value = GetValueOrEmpty(report.SMSFAR170) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410209
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410209_3()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410209
            Other given name cannot contain "MR", "MRS", "MISS" or "MS"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text = FOUND("MR","MRS","MISS","MS"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^SMSFAR14, '"MR","MRS","MISS","MS"'))
    
            Data Elements:
    
            ^SMSFAR14 = Auditor:PersonNameDetails:PersonNameDetails.OtherGivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.SMSFAR14, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000450",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_Contact_C_AuditorOccurrenceIndex) + "/tns:PersonNameDetails.OtherGivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410209" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR14", Value = GetValueOrEmpty(report.SMSFAR14) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.410218
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN410218(SMSFAR2026.MemberInformation memberInformation, int itemIndex1)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.410218
            FirstName must be entered if OtherGivenNames is entered

            Legacy Rule Format:
            WHERE IN TUPLE(personstructuredname3.xx.xx:PersonNameDetails) 
            IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.GivenName.Text = NULLORBLANK) 
                RETURN VALIDATION MESSAGE 
             ENDIF

            Technical Business Rule Format:
            (^PersonNameDetails <> NULL) AND ((^SMSFAR200 <> BLANK) AND (^SMSFAR199 = BLANK))

            Data Elements:

            ^SMSFAR199 = RP:MemberInformation:PersonNameDetails:PersonNameDetails.GivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Identifier'].[PersonNameDetails.Currency.Code='C']

            ^SMSFAR200 = RP:MemberInformation:PersonNameDetails:PersonNameDetails.OtherGivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Identifier'].[PersonNameDetails.Currency.Code='C']

            ^PersonNameDetails = RP:MemberInformation:PersonNameDetails
            */
            assertion = (memberInformation.MemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RPExists != false && (string.IsNullOrWhiteSpace(memberInformation.SMSFAR200) != true && string.IsNullOrWhiteSpace(memberInformation.SMSFAR199) == true));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410218",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"FirstName must be entered if OtherGivenNames is entered",
                    Location = "/xbrli:xbrl/tns:MemberInformation" + OccurrenceIndex(memberInformation.OccurrenceIndex) + "/tns:PersonNameDetails" + OccurrenceIndex(memberInformation.MemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RPOccurrenceIndex) + "/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.410218" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR200", Value = GetValueOrEmpty(memberInformation.SMSFAR200) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR199", Value = GetValueOrEmpty(memberInformation.SMSFAR199) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.428016
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428016()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.428016
            Tax File Number must pass the TFN algorithm check.
    
            Legacy Rule Format:
            IF (RP:entity.identifier.TFN <> NULLORBLANK) AND (TFNALGORITHM (RP:entity.identifier.TFN) = FALSE)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^TFN <> BLANK) AND (FailsTFNAlgorithm(^TFN))
    
            Data Elements:
    
            ^TFN = RP
            */
            assertion = (string.IsNullOrWhiteSpace(report.RPIdentifierTFN) != true && FailsTFNAlgorithm(report.RPIdentifierTFN));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.428016",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax File Number has failed the algorithm check",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428016" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.428230
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428230()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.428230
            Suburb/Town/City cannot contain state code
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.LocalityName.Text = FOUND("QLD","NSW","VIC","SA","WA","NT","ACT","TAS"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^SMSFAR7, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))
    
            Data Elements:
    
            ^SMSFAR7 = RP:AddressDetails:AddressDetails.LocalityName.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.SMSFAR7, @"(^(QLD|NSW|VIC|SA|WA|NT|ACT|TAS)$)|(^(.*)( QLD | NSW | VIC | SA | WA | NT | ACT | TAS )(.*)$)|(^(QLD |NSW |VIC |SA |WA |NT |ACT |TAS )(.*)$)|(^(.*)( QLD| NSW| VIC| SA| WA| NT| ACT| TAS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000411",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain state code",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.LocalityName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428230" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR7", Value = report.SMSFAR7 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.428230
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428230_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.428230
            Suburb/Town/City cannot contain state code
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.LocalityName.Text = FOUND("QLD","NSW","VIC","SA","WA","NT","ACT","TAS"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^SMSFAR20, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))
    
            Data Elements:
    
            ^SMSFAR20 = Auditor:AddressDetails:AddressDetails.LocalityName.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.SMSFAR20, @"(^(QLD|NSW|VIC|SA|WA|NT|ACT|TAS)$)|(^(.*)( QLD | NSW | VIC | SA | WA | NT | ACT | TAS )(.*)$)|(^(QLD |NSW |VIC |SA |WA |NT |ACT |TAS )(.*)$)|(^(.*)( QLD| NSW| VIC| SA| WA| NT| ACT| TAS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000411",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain state code",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_AuditorOccurrenceIndex) + "/tns:AddressDetails.LocalityName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428230" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR20", Value = report.SMSFAR20 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.428247
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428247()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.428247
            The Tax Agent number has failed the Tax Agent number algorithm check
    
            Legacy Rule Format:
            IF ((pyid.xx.xx:Identifiers.TaxAgentNumber.Identifier) <> NULLORBLANK) AND (TANALGORITHM(pyid.xx.xx:Identifiers.TaxAgentNumber.Identifier) = FALSE)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^SMSFAR2 <> BLANK) AND (FailsTANAlgorithm(^SMSFAR2))
    
            Data Elements:
    
            ^SMSFAR2 = INT:Identifiers.TaxAgentNumber.Identifier
            */
            assertion = (string.IsNullOrWhiteSpace(report.SMSFAR2) != true && FailsTANAlgorithm(report.SMSFAR2));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The Tax Agent number has failed the Tax Agent number algorithm check",
                    Location = "/xbrli:xbrl/tns:Identifiers.TaxAgentNumber.Identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428247" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR2", Value = GetValueOrEmpty(report.SMSFAR2) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.428253
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428253()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.428253
            ABN must pass the ABN algorithm check
    
            Legacy Rule Format:
            IF (INT:entity.identifier.ABN <> NULLORBLANK) AND (ABNALGORITHM (INT:entity.identifier.ABN) = FALSE)
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^ABN <> BLANK) AND (FailsABNAlgorithm(^ABN))
    
            Data Elements:
    
            ^ABN = INT
            */
            assertion = (string.IsNullOrWhiteSpace(report.INTIdentifierABN) != true && FailsABNAlgorithm(report.INTIdentifierABN));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid ABN",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.INTOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428253" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.428254
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428254()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.428254
            Address line 1 cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line1.Text CONTAINS "UNKNOWN")
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^SMSFAR6, 'UNKNOWN')
    
            Data Elements:
    
            ^SMSFAR6 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (report.SMSFAR6 == null ? false : report.SMSFAR6.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000405",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 cannot contain ""UNKNOWN""",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428254" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR6", Value = report.SMSFAR6 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.428254
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428254_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.428254
            Address line 1 cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line1.Text CONTAINS "UNKNOWN")
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^SMSFAR19, 'UNKNOWN')
    
            Data Elements:
    
            ^SMSFAR19 = Auditor:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (report.SMSFAR19 == null ? false : report.SMSFAR19.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000405",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 1 cannot contain ""UNKNOWN""",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_AuditorOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428254" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR19", Value = report.SMSFAR19 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.428255
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428255()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.428255
            Address line 2 cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line2.Text CONTAINS "UNKNOWN")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^SMSFAR239, 'UNKNOWN')
    
            Data Elements:
    
            ^SMSFAR239 = RP:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (report.SMSFAR239 == null ? false : report.SMSFAR239.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000407",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 2 cannot contain ""UNKNOWN""",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line2.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428255" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR239", Value = GetValueOrEmpty(report.SMSFAR239) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.428255
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428255_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.428255
            Address line 2 cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.Line2.Text CONTAINS "UNKNOWN")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^SMSFAR240, 'UNKNOWN')
    
            Data Elements:
    
            ^SMSFAR240 = Auditor:AddressDetails:AddressDetails.Line2.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (report.SMSFAR240 == null ? false : report.SMSFAR240.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000407",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address line 2 cannot contain ""UNKNOWN""",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_AuditorOccurrenceIndex) + "/tns:AddressDetails.Line2.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428255" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR240", Value = GetValueOrEmpty(report.SMSFAR240) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.428256
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428256()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.428256
            Suburb/Town/City cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.LocalityName.Text CONTAINS "UNKNOWN")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^SMSFAR7, 'UNKNOWN')
    
            Data Elements:
    
            ^SMSFAR7 = RP:AddressDetails:AddressDetails.LocalityName.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (report.SMSFAR7 == null ? false : report.SMSFAR7.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000409",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain ""UNKNOWN""",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.LocalityName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428256" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR7", Value = report.SMSFAR7 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.428256
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428256_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.428256
            Suburb/Town/City cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.LocalityName.Text CONTAINS "UNKNOWN")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^SMSFAR20, 'UNKNOWN')
    
            Data Elements:
    
            ^SMSFAR20 = Auditor:AddressDetails:AddressDetails.LocalityName.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (report.SMSFAR20 == null ? false : report.SMSFAR20.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000409",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain ""UNKNOWN""",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_AuditorOccurrenceIndex) + "/tns:AddressDetails.LocalityName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428256" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR20", Value = report.SMSFAR20 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.428258
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428258()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.428258
            Non Individual Name cannot contain space hyphen space combinations
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text CONTAINS  " - " )
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^SMSFAR182, ' - ')
    
            Data Elements:
    
            ^SMSFAR182 = INT:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
            */
            assertion = (report.SMSFAR182 == null ? false : report.SMSFAR182.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000413",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain space hyphen space combinations",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_MN_C_INTOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428258" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR182", Value = report.SMSFAR182 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.428258
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428258_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.428258
            Non Individual Name cannot contain space hyphen space combinations
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text CONTAINS  " - " )
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^SMSFAR174, ' - ')
    
            Data Elements:
    
            ^SMSFAR174 = RP:TrusteeOrDirector:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
            */
            assertion = (report.SMSFAR174 == null ? false : report.SMSFAR174.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000413",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain space hyphen space combinations",
                    Location = "/xbrli:xbrl/tns:TrusteeOrDirector/tns:OrganisationNameDetails" + OccurrenceIndex(report.TrusteeOrDirector_OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428258" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR174", Value = report.SMSFAR174 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.428258
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428258_2()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.428258
            Non Individual Name cannot contain space hyphen space combinations
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text CONTAINS  " - " )
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^SMSFAR4, ' - ')
    
            Data Elements:
    
            ^SMSFAR4 = RP:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
            */
            assertion = (report.SMSFAR4 == null ? false : report.SMSFAR4.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000413",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain space hyphen space combinations",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428258" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR4", Value = report.SMSFAR4 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.428259
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428259()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.428259
            Non Individual Name cannot contain "P/L"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text CONTAINS "P/L")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^SMSFAR182, 'P/L')
    
            Data Elements:
    
            ^SMSFAR182 = INT:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
            */
            assertion = (report.SMSFAR182 == null ? false : report.SMSFAR182.ToUpperInvariant().Contains(@"P/L"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000414",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain ""P/L""",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_MN_C_INTOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428259" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR182", Value = report.SMSFAR182 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.428259
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428259_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.428259
            Non Individual Name cannot contain "P/L"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text CONTAINS "P/L")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^SMSFAR174, 'P/L')
    
            Data Elements:
    
            ^SMSFAR174 = RP:TrusteeOrDirector:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
            */
            assertion = (report.SMSFAR174 == null ? false : report.SMSFAR174.ToUpperInvariant().Contains(@"P/L"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000414",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain ""P/L""",
                    Location = "/xbrli:xbrl/tns:TrusteeOrDirector/tns:OrganisationNameDetails" + OccurrenceIndex(report.TrusteeOrDirector_OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428259" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR174", Value = report.SMSFAR174 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.428259
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428259_2()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.428259
            Non Individual Name cannot contain "P/L"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text CONTAINS "P/L")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^SMSFAR4, 'P/L')
    
            Data Elements:
    
            ^SMSFAR4 = RP:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
            */
            assertion = (report.SMSFAR4 == null ? false : report.SMSFAR4.ToUpperInvariant().Contains(@"P/L"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000414",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain ""P/L""",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428259" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR4", Value = report.SMSFAR4 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.428260
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428260()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.428260
            Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text CONTAINS SET("--","''","  ") )
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (ContainsSet(^SMSFAR182, '"--","''","  "'))
    
            Data Elements:
    
            ^SMSFAR182 = INT:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.SMSFAR182, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000416",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_MN_C_INTOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428260" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR182", Value = report.SMSFAR182 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.428260
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428260_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.428260
            Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text CONTAINS SET("--","''","  ") )
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (ContainsSet(^SMSFAR174, '"--","''","  "'))
    
            Data Elements:
    
            ^SMSFAR174 = RP:TrusteeOrDirector:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.SMSFAR174, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000416",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/xbrli:xbrl/tns:TrusteeOrDirector/tns:OrganisationNameDetails" + OccurrenceIndex(report.TrusteeOrDirector_OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428260" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR174", Value = report.SMSFAR174 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.428260
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428260_2()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.428260
            Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text CONTAINS SET("--","''","  ") )
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (ContainsSet(^SMSFAR4, '"--","''","  "'))
    
            Data Elements:
    
            ^SMSFAR4 = RP:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.SMSFAR4, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000416",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428260" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR4", Value = report.SMSFAR4 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.428261
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428261()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.428261
            Non Individual Name cannot end with "T/A", "T/A P'ship", "T/A Pship", "T/A P/Ship" or "T/A Partnership"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text ENDSWITH SET(" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"))
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (EndsWithSet(^SMSFAR182, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"'))
    
            Data Elements:
    
            ^SMSFAR182 = INT:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.SMSFAR182, @"( T/A| T/A P'ship| T/A Pship| T/A P/Ship| T/A Partnership)$", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000417",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot end with ""T/A"", ""T/A P'ship"", ""T/A Pship"", ""T/A P/Ship"" or ""T/A Partnership""",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_MN_C_INTOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428261" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR182", Value = report.SMSFAR182 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.428261
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428261_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.428261
            Non Individual Name cannot end with "T/A", "T/A P'ship", "T/A Pship", "T/A P/Ship" or "T/A Partnership"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text ENDSWITH SET(" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"))
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (EndsWithSet(^SMSFAR174, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"'))
    
            Data Elements:
    
            ^SMSFAR174 = RP:TrusteeOrDirector:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.SMSFAR174, @"( T/A| T/A P'ship| T/A Pship| T/A P/Ship| T/A Partnership)$", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000417",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot end with ""T/A"", ""T/A P'ship"", ""T/A Pship"", ""T/A P/Ship"" or ""T/A Partnership""",
                    Location = "/xbrli:xbrl/tns:TrusteeOrDirector/tns:OrganisationNameDetails" + OccurrenceIndex(report.TrusteeOrDirector_OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428261" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR174", Value = report.SMSFAR174 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.428261
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428261_2()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.428261
            Non Individual Name cannot end with "T/A", "T/A P'ship", "T/A Pship", "T/A P/Ship" or "T/A Partnership"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text ENDSWITH SET(" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"))
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (EndsWithSet(^SMSFAR4, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"'))
    
            Data Elements:
    
            ^SMSFAR4 = RP:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.SMSFAR4, @"( T/A| T/A P'ship| T/A Pship| T/A P/Ship| T/A Partnership)$", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000417",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Non Individual Name cannot end with ""T/A"", ""T/A P'ship"", ""T/A Pship"", ""T/A P/Ship"" or ""T/A Partnership""",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428261" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR4", Value = report.SMSFAR4 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.428262
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428262()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.428262
            Family Name cannot contain "space hyphen space"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text CONTAINS " - ")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^SMSFAR178, ' - ')
    
            Data Elements:
    
            ^SMSFAR178 = INT:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = (report.SMSFAR178 == null ? false : report.SMSFAR178.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000423",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family Name cannot contain ""space hyphen space""",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_Contact_C_INTOccurrenceIndex) + "/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428262" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR178", Value = report.SMSFAR178 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.428262
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428262_1(SMSFAR2026.MemberInformation memberInformation, int itemIndex1)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.428262
            Family Name cannot contain "space hyphen space"

            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text CONTAINS " - ")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^SMSFAR197, ' - ')

            Data Elements:

            ^SMSFAR197 = RP:MemberInformation:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Identifier'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = (memberInformation.SMSFAR197 == null ? false : memberInformation.SMSFAR197.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000423",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family Name cannot contain ""space hyphen space""",
                    Location = "/xbrli:xbrl/tns:MemberInformation" + OccurrenceIndex(memberInformation.OccurrenceIndex) + "/tns:PersonNameDetails" + OccurrenceIndex(memberInformation.MemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RPOccurrenceIndex) + "/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428262" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR197", Value = memberInformation.SMSFAR197 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.428262
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428262_2()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.428262
            Family Name cannot contain "space hyphen space"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text CONTAINS " - ")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^SMSFAR167, ' - ')
    
            Data Elements:
    
            ^SMSFAR167 = RP:TrusteeOrDirector:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = (report.SMSFAR167 == null ? false : report.SMSFAR167.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000423",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family Name cannot contain ""space hyphen space""",
                    Location = "/xbrli:xbrl/tns:TrusteeOrDirector/tns:PersonNameDetails" + OccurrenceIndex(report.TrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RPOccurrenceIndex) + "/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428262" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR167", Value = report.SMSFAR167 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.428262
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428262_3()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.428262
            Family Name cannot contain "space hyphen space"
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text CONTAINS " - ")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^SMSFAR11, ' - ')
    
            Data Elements:
    
            ^SMSFAR11 = Auditor:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = (report.SMSFAR11 == null ? false : report.SMSFAR11.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000423",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family Name cannot contain ""space hyphen space""",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_Contact_C_AuditorOccurrenceIndex) + "/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428262" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR11", Value = report.SMSFAR11 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.428263
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428263()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.428263
            Family name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text CONTAINS SET("--","''","  ") )
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (ContainsSet(^SMSFAR178, '"--","''","  "'))
    
            Data Elements:
    
            ^SMSFAR178 = INT:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.SMSFAR178, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000427",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_Contact_C_INTOccurrenceIndex) + "/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428263" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR178", Value = report.SMSFAR178 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.428263
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428263_1(SMSFAR2026.MemberInformation memberInformation, int itemIndex1)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.428263
            Family name cannot contain repeated hyphen, apostrophe, or space characters

            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text CONTAINS SET("--","''","  ") )
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (ContainsSet(^SMSFAR197, '"--","''","  "'))

            Data Elements:

            ^SMSFAR197 = RP:MemberInformation:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Identifier'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(memberInformation.SMSFAR197, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000427",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/xbrli:xbrl/tns:MemberInformation" + OccurrenceIndex(memberInformation.OccurrenceIndex) + "/tns:PersonNameDetails" + OccurrenceIndex(memberInformation.MemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RPOccurrenceIndex) + "/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428263" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR197", Value = memberInformation.SMSFAR197 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.428263
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428263_2()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.428263
            Family name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text CONTAINS SET("--","''","  ") )
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (ContainsSet(^SMSFAR167, '"--","''","  "'))
    
            Data Elements:
    
            ^SMSFAR167 = RP:TrusteeOrDirector:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.SMSFAR167, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000427",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/xbrli:xbrl/tns:TrusteeOrDirector/tns:PersonNameDetails" + OccurrenceIndex(report.TrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RPOccurrenceIndex) + "/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428263" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR167", Value = report.SMSFAR167 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.428263
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428263_3()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.428263
            Family name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text CONTAINS SET("--","''","  ") )
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (ContainsSet(^SMSFAR11, '"--","''","  "'))
    
            Data Elements:
    
            ^SMSFAR11 = Auditor:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.SMSFAR11, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000427",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_Contact_C_AuditorOccurrenceIndex) + "/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428263" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR11", Value = report.SMSFAR11 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.428264
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428264()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.428264
            First name cannot contain space hyphen space
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text CONTAINS  " - ")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^SMSFAR180, ' - ')
    
            Data Elements:
    
            ^SMSFAR180 = INT:PersonNameDetails:PersonNameDetails.GivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = (report.SMSFAR180 == null ? false : report.SMSFAR180.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000434",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain space hyphen space",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_Contact_C_INTOccurrenceIndex) + "/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428264" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR180", Value = report.SMSFAR180 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.428264
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428264_1(SMSFAR2026.MemberInformation memberInformation, int itemIndex1)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.428264
            First name cannot contain space hyphen space

            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text CONTAINS  " - ")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^SMSFAR199, ' - ')

            Data Elements:

            ^SMSFAR199 = RP:MemberInformation:PersonNameDetails:PersonNameDetails.GivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Identifier'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = (memberInformation.SMSFAR199 == null ? false : memberInformation.SMSFAR199.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000434",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain space hyphen space",
                    Location = "/xbrli:xbrl/tns:MemberInformation" + OccurrenceIndex(memberInformation.OccurrenceIndex) + "/tns:PersonNameDetails" + OccurrenceIndex(memberInformation.MemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RPOccurrenceIndex) + "/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428264" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR199", Value = GetValueOrEmpty(memberInformation.SMSFAR199) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.428264
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428264_2()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.428264
            First name cannot contain space hyphen space
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text CONTAINS  " - ")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^SMSFAR169, ' - ')
    
            Data Elements:
    
            ^SMSFAR169 = RP:TrusteeOrDirector:PersonNameDetails:PersonNameDetails.GivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = (report.SMSFAR169 == null ? false : report.SMSFAR169.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000434",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain space hyphen space",
                    Location = "/xbrli:xbrl/tns:TrusteeOrDirector/tns:PersonNameDetails" + OccurrenceIndex(report.TrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RPOccurrenceIndex) + "/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428264" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR169", Value = report.SMSFAR169 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.428264
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428264_3()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.428264
            First name cannot contain space hyphen space
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text CONTAINS  " - ")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^SMSFAR13, ' - ')
    
            Data Elements:
    
            ^SMSFAR13 = Auditor:PersonNameDetails:PersonNameDetails.GivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = (report.SMSFAR13 == null ? false : report.SMSFAR13.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000434",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain space hyphen space",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_Contact_C_AuditorOccurrenceIndex) + "/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428264" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR13", Value = report.SMSFAR13 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.428265
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428265()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.428265
            First name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text CONTAINS SET("--","''","  ") )
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (ContainsSet(^SMSFAR180, '"--","''","  "'))
    
            Data Elements:
    
            ^SMSFAR180 = INT:PersonNameDetails:PersonNameDetails.GivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.SMSFAR180, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000439",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_Contact_C_INTOccurrenceIndex) + "/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428265" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR180", Value = report.SMSFAR180 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.428265
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428265_1(SMSFAR2026.MemberInformation memberInformation, int itemIndex1)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.428265
            First name cannot contain repeated hyphen, apostrophe, or space characters

            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text CONTAINS SET("--","''","  ") )
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (ContainsSet(^SMSFAR199, '"--","''","  "'))

            Data Elements:

            ^SMSFAR199 = RP:MemberInformation:PersonNameDetails:PersonNameDetails.GivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Identifier'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(memberInformation.SMSFAR199, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000439",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/xbrli:xbrl/tns:MemberInformation" + OccurrenceIndex(memberInformation.OccurrenceIndex) + "/tns:PersonNameDetails" + OccurrenceIndex(memberInformation.MemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RPOccurrenceIndex) + "/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428265" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR199", Value = GetValueOrEmpty(memberInformation.SMSFAR199) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.428265
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428265_2()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.428265
            First name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text CONTAINS SET("--","''","  ") )
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (ContainsSet(^SMSFAR169, '"--","''","  "'))
    
            Data Elements:
    
            ^SMSFAR169 = RP:TrusteeOrDirector:PersonNameDetails:PersonNameDetails.GivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.SMSFAR169, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000439",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/xbrli:xbrl/tns:TrusteeOrDirector/tns:PersonNameDetails" + OccurrenceIndex(report.TrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RPOccurrenceIndex) + "/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428265" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR169", Value = report.SMSFAR169 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.428265
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428265_3()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.428265
            First name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text CONTAINS SET("--","''","  ") )
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (ContainsSet(^SMSFAR13, '"--","''","  "'))
    
            Data Elements:
    
            ^SMSFAR13 = Auditor:PersonNameDetails:PersonNameDetails.GivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.SMSFAR13, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000439",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_Contact_C_AuditorOccurrenceIndex) + "/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428265" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR13", Value = report.SMSFAR13 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.428266
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428266()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.428266
            Other given name cannot contain space hyphen space
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text CONTAINS  " - ")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^SMSFAR181, ' - ')
    
            Data Elements:
    
            ^SMSFAR181 = INT:PersonNameDetails:PersonNameDetails.OtherGivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = (report.SMSFAR181 == null ? false : report.SMSFAR181.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000446",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain space hyphen space",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_Contact_C_INTOccurrenceIndex) + "/tns:PersonNameDetails.OtherGivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428266" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR181", Value = GetValueOrEmpty(report.SMSFAR181) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.428266
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428266_1(SMSFAR2026.MemberInformation memberInformation, int itemIndex1)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.428266
            Other given name cannot contain space hyphen space

            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text CONTAINS  " - ")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^SMSFAR200, ' - ')

            Data Elements:

            ^SMSFAR200 = RP:MemberInformation:PersonNameDetails:PersonNameDetails.OtherGivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Identifier'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = (memberInformation.SMSFAR200 == null ? false : memberInformation.SMSFAR200.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000446",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain space hyphen space",
                    Location = "/xbrli:xbrl/tns:MemberInformation" + OccurrenceIndex(memberInformation.OccurrenceIndex) + "/tns:PersonNameDetails" + OccurrenceIndex(memberInformation.MemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RPOccurrenceIndex) + "/tns:PersonNameDetails.OtherGivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428266" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR200", Value = GetValueOrEmpty(memberInformation.SMSFAR200) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.428266
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428266_2()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.428266
            Other given name cannot contain space hyphen space
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text CONTAINS  " - ")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^SMSFAR170, ' - ')
    
            Data Elements:
    
            ^SMSFAR170 = RP:TrusteeOrDirector:PersonNameDetails:PersonNameDetails.OtherGivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = (report.SMSFAR170 == null ? false : report.SMSFAR170.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000446",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain space hyphen space",
                    Location = "/xbrli:xbrl/tns:TrusteeOrDirector/tns:PersonNameDetails" + OccurrenceIndex(report.TrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RPOccurrenceIndex) + "/tns:PersonNameDetails.OtherGivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428266" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR170", Value = GetValueOrEmpty(report.SMSFAR170) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.428266
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428266_3()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.428266
            Other given name cannot contain space hyphen space
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text CONTAINS  " - ")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^SMSFAR14, ' - ')
    
            Data Elements:
    
            ^SMSFAR14 = Auditor:PersonNameDetails:PersonNameDetails.OtherGivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = (report.SMSFAR14 == null ? false : report.SMSFAR14.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000446",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain space hyphen space",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_Contact_C_AuditorOccurrenceIndex) + "/tns:PersonNameDetails.OtherGivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428266" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR14", Value = GetValueOrEmpty(report.SMSFAR14) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.428267
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428267()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.428267
            Other given name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text CONTAINS SET("--","''","  ") )
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (ContainsSet(^SMSFAR181, '"--","''","  "'))
    
            Data Elements:
    
            ^SMSFAR181 = INT:PersonNameDetails:PersonNameDetails.OtherGivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.SMSFAR181, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000451",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_Contact_C_INTOccurrenceIndex) + "/tns:PersonNameDetails.OtherGivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428267" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR181", Value = GetValueOrEmpty(report.SMSFAR181) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.428267
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428267_1(SMSFAR2026.MemberInformation memberInformation, int itemIndex1)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.428267
            Other given name cannot contain repeated hyphen, apostrophe, or space characters

            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text CONTAINS SET("--","''","  ") )
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (ContainsSet(^SMSFAR200, '"--","''","  "'))

            Data Elements:

            ^SMSFAR200 = RP:MemberInformation:PersonNameDetails:PersonNameDetails.OtherGivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Identifier'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(memberInformation.SMSFAR200, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000451",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/xbrli:xbrl/tns:MemberInformation" + OccurrenceIndex(memberInformation.OccurrenceIndex) + "/tns:PersonNameDetails" + OccurrenceIndex(memberInformation.MemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RPOccurrenceIndex) + "/tns:PersonNameDetails.OtherGivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428267" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR200", Value = GetValueOrEmpty(memberInformation.SMSFAR200) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.428267
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428267_2()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.428267
            Other given name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text CONTAINS SET("--","''","  ") )
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (ContainsSet(^SMSFAR170, '"--","''","  "'))
    
            Data Elements:
    
            ^SMSFAR170 = RP:TrusteeOrDirector:PersonNameDetails:PersonNameDetails.OtherGivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.SMSFAR170, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000451",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/xbrli:xbrl/tns:TrusteeOrDirector/tns:PersonNameDetails" + OccurrenceIndex(report.TrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RPOccurrenceIndex) + "/tns:PersonNameDetails.OtherGivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428267" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR170", Value = GetValueOrEmpty(report.SMSFAR170) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.428267
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428267_3()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.428267
            Other given name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text CONTAINS SET("--","''","  ") )
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (ContainsSet(^SMSFAR14, '"--","''","  "'))
    
            Data Elements:
    
            ^SMSFAR14 = Auditor:PersonNameDetails:PersonNameDetails.OtherGivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.SMSFAR14, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000451",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_Contact_C_AuditorOccurrenceIndex) + "/tns:PersonNameDetails.OtherGivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.428267" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR14", Value = GetValueOrEmpty(report.SMSFAR14) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.430206
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN430206()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.430206
            Hours taken to prepare and complete form must be 1 to 999
    
            Legacy Rule Format:
            IF (pyin.xx.xx:Report.CompletionHours.Number <> NULL) AND (pyin.xx.xx:Report.CompletionHours.Number <> SET (1-999))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^SMSFAR176 <> NULL) AND ((^SMSFAR176 < 1 OR ^SMSFAR176 > 999))
    
            Data Elements:
    
            ^SMSFAR176 = RP:Report.CompletionHours.Number
            */
            assertion = (report.SMSFAR176 != null && (report.SMSFAR176.GetValueOrDefault() < 1 || report.SMSFAR176.GetValueOrDefault() > 999));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430206",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Hours taken to prepare and complete form must be 1 to 999",
                    Location = "/xbrli:xbrl/tns:Report.CompletionHours.Number",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.430206" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR176", Value = report.SMSFAR176.GetValueOrDefault().ToString() });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.430222
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN430222()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.430222
            Email address must not start or end with an @ symbol
    
            Legacy Rule Format:
            IF (pyde.xx.xx:ElectronicContact.ElectronicMail.Address.Text STARTSWITH "@") OR (pyde.xx.xx:ElectronicContact.ElectronicMail.Address.Text ENDSWITH "@")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (StartsWith(^SMSFAR173, '@')) OR (EndsWith(^SMSFAR173, '@'))
    
            Data Elements:
    
            ^SMSFAR173 = RP:TrusteeOrDirector:ElectronicContactElectronicMail:ElectronicContact.ElectronicMail.Address.Text WHERE [ElectronicContact.ElectronicMail.Usage.Code='03']
            */
            assertion = (IsMatch(report.SMSFAR173, @"^(@)", RegexOptions.IgnoreCase) || IsMatch(report.SMSFAR173, @"(@)$", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430222",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Email address must not start or end with an @ symbol",
                    Location = "/xbrli:xbrl/tns:TrusteeOrDirector/tns:ElectronicContactElectronicMail" + OccurrenceIndex(report.TrusteeOrDirector_ElectronicContactElectronicMailCollection_03_RPOccurrenceIndex) + "/tns:ElectronicContact.ElectronicMail.Address.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.430222" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR173", Value = report.SMSFAR173 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.430223
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN430223()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.430223
            Email address must not contain more than one @ symbol
    
            Legacy Rule Format:
            IF pyde.xx.xx:ElectronicContact.ElectronicMail.Address.Text CONTAINS MORE THAN ONE "@"
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ContainsMultiple(^SMSFAR173, '@')
    
            Data Elements:
    
            ^SMSFAR173 = RP:TrusteeOrDirector:ElectronicContactElectronicMail:ElectronicContact.ElectronicMail.Address.Text WHERE [ElectronicContact.ElectronicMail.Usage.Code='03']
            */
            assertion = IsMatch(report.SMSFAR173, @"@(.*)@", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430223",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Email address must not contain more than one @ symbol",
                    Location = "/xbrli:xbrl/tns:TrusteeOrDirector/tns:ElectronicContactElectronicMail" + OccurrenceIndex(report.TrusteeOrDirector_ElectronicContactElectronicMailCollection_03_RPOccurrenceIndex) + "/tns:ElectronicContact.ElectronicMail.Address.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.430223" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR173", Value = report.SMSFAR173 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.430224
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN430224()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.430224
            Email address must not start or end with a dot (.)
    
            Legacy Rule Format:
            IF (pyde.xx.xx:ElectronicContact.ElectronicMail.Address.Text STARTSWITH ".") OR (pyde.xx.xx:ElectronicContact.ElectronicMail.Address.Text ENDSWITH ".") 
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (StartsWith(^SMSFAR173, '.')) OR (EndsWith(^SMSFAR173, '.'))
    
            Data Elements:
    
            ^SMSFAR173 = RP:TrusteeOrDirector:ElectronicContactElectronicMail:ElectronicContact.ElectronicMail.Address.Text WHERE [ElectronicContact.ElectronicMail.Usage.Code='03']
            */
            assertion = (IsMatch(report.SMSFAR173, @"^(\.)", RegexOptions.IgnoreCase) || IsMatch(report.SMSFAR173, @"(\.)$", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430224",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Email address must not start or end with a dot (.)",
                    Location = "/xbrli:xbrl/tns:TrusteeOrDirector/tns:ElectronicContactElectronicMail" + OccurrenceIndex(report.TrusteeOrDirector_ElectronicContactElectronicMailCollection_03_RPOccurrenceIndex) + "/tns:ElectronicContact.ElectronicMail.Address.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.430224" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR173", Value = report.SMSFAR173 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.430226
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN430226()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.430226
            Email address must not contain more than one consecutive dots (..)
    
            Legacy Rule Format:
            IF pyde.xx.xx:ElectronicContact.ElectronicMail.Address.Text CONTAINS ".."
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^SMSFAR173, '..')
    
            Data Elements:
    
            ^SMSFAR173 = RP:TrusteeOrDirector:ElectronicContactElectronicMail:ElectronicContact.ElectronicMail.Address.Text WHERE [ElectronicContact.ElectronicMail.Usage.Code='03']
            */
            assertion = (report.SMSFAR173 == null ? false : report.SMSFAR173.ToUpperInvariant().Contains(@".."));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430226",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Email address must not contain more than one consecutive dots (..)",
                    Location = "/xbrli:xbrl/tns:TrusteeOrDirector/tns:ElectronicContactElectronicMail" + OccurrenceIndex(report.TrusteeOrDirector_ElectronicContactElectronicMailCollection_03_RPOccurrenceIndex) + "/tns:ElectronicContact.ElectronicMail.Address.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.430226" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR173", Value = report.SMSFAR173 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.430227
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN430227()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.430227
            Email address must not contain a dot followed by a @ symbol (.@)
    
            Legacy Rule Format:
            IF pyde.xx.xx:ElectronicContact.ElectronicMail.Address.Text CONTAINS ".@"
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^SMSFAR173, '.@')
    
            Data Elements:
    
            ^SMSFAR173 = RP:TrusteeOrDirector:ElectronicContactElectronicMail:ElectronicContact.ElectronicMail.Address.Text WHERE [ElectronicContact.ElectronicMail.Usage.Code='03']
            */
            assertion = (report.SMSFAR173 == null ? false : report.SMSFAR173.ToUpperInvariant().Contains(@".@"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430227",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Email address must not contain a dot followed by a @ symbol (.@)",
                    Location = "/xbrli:xbrl/tns:TrusteeOrDirector/tns:ElectronicContactElectronicMail" + OccurrenceIndex(report.TrusteeOrDirector_ElectronicContactElectronicMailCollection_03_RPOccurrenceIndex) + "/tns:ElectronicContact.ElectronicMail.Address.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.430227" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR173", Value = report.SMSFAR173 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.430228
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN430228()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.430228
            Email address must not contain a @ symbol followed by a dot (@.)
    
            Legacy Rule Format:
            IF pyde.xx.xx:ElectronicContact.ElectronicMail.Address.Text CONTAINS "@."
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^SMSFAR173, '@.')
    
            Data Elements:
    
            ^SMSFAR173 = RP:TrusteeOrDirector:ElectronicContactElectronicMail:ElectronicContact.ElectronicMail.Address.Text WHERE [ElectronicContact.ElectronicMail.Usage.Code='03']
            */
            assertion = (report.SMSFAR173 == null ? false : report.SMSFAR173.ToUpperInvariant().Contains(@"@."));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430228",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Email address must not contain a @ symbol followed by a dot (@.)",
                    Location = "/xbrli:xbrl/tns:TrusteeOrDirector/tns:ElectronicContactElectronicMail" + OccurrenceIndex(report.TrusteeOrDirector_ElectronicContactElectronicMailCollection_03_RPOccurrenceIndex) + "/tns:ElectronicContact.ElectronicMail.Address.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.430228" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR173", Value = report.SMSFAR173 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.430229
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN430229()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.430229
            Email address must not contain an invalid character
    
            Legacy Rule Format:
            IF pyde.xx.xx:ElectronicContact.ElectronicMail.Address.Text CONTAINS SET("(", ")", "<", ">", ",", ";", ":", "\",  """, "]", "[")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ContainsSet(^SMSFAR173, '"(", ")", "<", ">", ",", ";", ":", "\",  """, "]", "["')
    
            Data Elements:
    
            ^SMSFAR173 = RP:TrusteeOrDirector:ElectronicContactElectronicMail:ElectronicContact.ElectronicMail.Address.Text WHERE [ElectronicContact.ElectronicMail.Usage.Code='03']
            */
            assertion = IsMatch(report.SMSFAR173, @"\(|\)|<|>|,|;|:|\\|""|\]|\[", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430229",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Email address must not contain an invalid character",
                    LongDescription = @"An email address must not contain any of the following invalid characters: ( ) < > , ; : \ "" [ ]",
                    Location = "/xbrli:xbrl/tns:TrusteeOrDirector/tns:ElectronicContactElectronicMail" + OccurrenceIndex(report.TrusteeOrDirector_ElectronicContactElectronicMailCollection_03_RPOccurrenceIndex) + "/tns:ElectronicContact.ElectronicMail.Address.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.430229" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR173", Value = report.SMSFAR173 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.430230
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN430230()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.430230
            Email address must contain an @ symbol
    
            Legacy Rule Format:
            IF (pyde.xx.xx:ElectronicContact.ElectronicMail.Address.Text <> NULLORBLANK) AND (pyde.xx.xx:ElectronicContact.ElectronicMail.Address.Text DOES NOT CONTAIN "@")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^ElectronicContactElectronicMail <> NULL) AND ((^SMSFAR173 <> BLANK) AND (NotContains(^SMSFAR173, '@')))
    
            Data Elements:
    
            ^SMSFAR173 = RP:TrusteeOrDirector:ElectronicContactElectronicMail:ElectronicContact.ElectronicMail.Address.Text WHERE [ElectronicContact.ElectronicMail.Usage.Code='03']
    
            ^ElectronicContactElectronicMail = RP:TrusteeOrDirector:ElectronicContactElectronicMail
            */
            assertion = (report.TrusteeOrDirector_ElectronicContactElectronicMailCollection_03_RPExists != false && (string.IsNullOrWhiteSpace(report.SMSFAR173) != true && !(report.SMSFAR173 == null ? true : report.SMSFAR173.ToUpperInvariant().Contains(@"@"))));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430230",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Email address must contain an @ symbol",
                    Location = "/xbrli:xbrl/tns:TrusteeOrDirector/tns:ElectronicContactElectronicMail" + OccurrenceIndex(report.TrusteeOrDirector_ElectronicContactElectronicMailCollection_03_RPOccurrenceIndex) + "/tns:ElectronicContact.ElectronicMail.Address.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.430230" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR173", Value = report.SMSFAR173 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.430231
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN430231()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.430231
            An email address must not start with a space or include a space within it
    
            Legacy Rule Format:
            IF pyde.xx.xx:ElectronicContact.ElectronicMail.Address.Text = FOUND(" ")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Contains(^SMSFAR173, ' ')
    
            Data Elements:
    
            ^SMSFAR173 = RP:TrusteeOrDirector:ElectronicContactElectronicMail:ElectronicContact.ElectronicMail.Address.Text WHERE [ElectronicContact.ElectronicMail.Usage.Code='03']
            */
            assertion = (report.SMSFAR173 == null ? false : report.SMSFAR173.ToUpperInvariant().Contains(@" "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430231",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"An email address must not start with a space or include a space within it",
                    Location = "/xbrli:xbrl/tns:TrusteeOrDirector/tns:ElectronicContactElectronicMail" + OccurrenceIndex(report.TrusteeOrDirector_ElectronicContactElectronicMailCollection_03_RPOccurrenceIndex) + "/tns:ElectronicContact.ElectronicMail.Address.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.430231" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR173", Value = report.SMSFAR173 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.430232
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN430232()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.430232
            Email address must contain a dot (.)
    
            Legacy Rule Format:
            IF (pyde.xx.xx:ElectronicContact.ElectronicMail.Address.Text <> NULLORBLANK) AND (pyde.xx.xx:ElectronicContact.ElectronicMail.Address.Text DOES NOT CONTAIN ".")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^ElectronicContactElectronicMail <> NULL) AND ((^SMSFAR173 <> BLANK) AND (NotContains(^SMSFAR173, '.')))
    
            Data Elements:
    
            ^SMSFAR173 = RP:TrusteeOrDirector:ElectronicContactElectronicMail:ElectronicContact.ElectronicMail.Address.Text WHERE [ElectronicContact.ElectronicMail.Usage.Code='03']
    
            ^ElectronicContactElectronicMail = RP:TrusteeOrDirector:ElectronicContactElectronicMail
            */
            assertion = (report.TrusteeOrDirector_ElectronicContactElectronicMailCollection_03_RPExists != false && (string.IsNullOrWhiteSpace(report.SMSFAR173) != true && !(report.SMSFAR173 == null ? true : report.SMSFAR173.ToUpperInvariant().Contains(@"."))));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430232",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Email address must contain a dot (.)",
                    Location = "/xbrli:xbrl/tns:TrusteeOrDirector/tns:ElectronicContactElectronicMail" + OccurrenceIndex(report.TrusteeOrDirector_ElectronicContactElectronicMailCollection_03_RPOccurrenceIndex) + "/tns:ElectronicContact.ElectronicMail.Address.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.430232" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR173", Value = report.SMSFAR173 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.430246
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN430246()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.430246
            Suburb/Town/City cannot contain numeric characters
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF (pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = FALSE) AND (pyde.xx.xx:AddressDetails.LocalityName.Text CONTAINS NUMERICS SET(0-9))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^SMSFAR281 = FALSE) AND (ContainsSet(^SMSFAR7, '0-9'))
    
            Data Elements:
    
            ^SMSFAR7 = RP:AddressDetails:AddressDetails.LocalityName.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
    
            ^SMSFAR281 = RP:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (report.SMSFAR281 == false && IsMatch(report.SMSFAR7, @"[0-9]", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000410",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain numeric characters",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.LocalityName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.430246" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR281", Value = GetValueOrEmpty(report.SMSFAR281) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR7", Value = report.SMSFAR7 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.430246
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN430246_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.430246
            Suburb/Town/City cannot contain numeric characters
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF (pyde.xx.xx:AddressDetails.OverseasAddress.Indicator = FALSE) AND (pyde.xx.xx:AddressDetails.LocalityName.Text CONTAINS NUMERICS SET(0-9))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^SMSFAR286 = FALSE) AND (ContainsSet(^SMSFAR20, '0-9'))
    
            Data Elements:
    
            ^SMSFAR20 = Auditor:AddressDetails:AddressDetails.LocalityName.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
    
            ^SMSFAR286 = Auditor:AddressDetails:AddressDetails.OverseasAddress.Indicator WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (report.SMSFAR286 == false && IsMatch(report.SMSFAR20, @"[0-9]", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000410",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City cannot contain numeric characters",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_AuditorOccurrenceIndex) + "/tns:AddressDetails.LocalityName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.430246" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR286", Value = GetValueOrEmpty(report.SMSFAR286) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR20", Value = report.SMSFAR20 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.430252
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN430252()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.430252
            Unstructured name must not be more than 200 characters
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:PersonUnstructuredName.FullName.Text) > 200
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^SMSFAR280) > 200
    
            Data Elements:
    
            ^SMSFAR280 = INT:Declaration:PersonUnstructuredName:PersonUnstructuredName.FullName.Text WHERE [PersonUnstructuredName.Usage.Code='DeclarationSignatory']
            */
            assertion = (Length(report.SMSFAR280) > 200);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430252",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unstructured name must not be more than 200 characters",
                    Location = "/xbrli:xbrl/tns:Declaration" + OccurrenceIndex(report.DeclarationCollection_TrueAndCorrect_INTOccurrenceIndex) + "/tns:PersonUnstructuredName" + OccurrenceIndex(report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTOccurrenceIndex) + "/tns:PersonUnstructuredName.FullName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.430252" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR280", Value = report.SMSFAR280 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.430252
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN430252_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.430252
            Unstructured name must not be more than 200 characters
    
            Legacy Rule Format:
            IF LENGTH(pyde.xx.xx:PersonUnstructuredName.FullName.Text) > 200
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^SMSFAR278) > 200
    
            Data Elements:
    
            ^SMSFAR278 = RP:Declaration:PersonUnstructuredName:PersonUnstructuredName.FullName.Text WHERE [PersonUnstructuredName.Usage.Code='DeclarationSignatory']
            */
            assertion = (Length(report.SMSFAR278) > 200);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430252",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unstructured name must not be more than 200 characters",
                    Location = "/xbrli:xbrl/tns:Declaration" + OccurrenceIndex(report.DeclarationCollection_TrueAndCorrect_RPOccurrenceIndex) + "/tns:PersonUnstructuredName" + OccurrenceIndex(report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPOccurrenceIndex) + "/tns:PersonUnstructuredName.FullName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.430252" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR278", Value = report.SMSFAR278 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.430255
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN430255()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.430255
            Signature date must not be later than today
    
            Legacy Rule Format:
            IF pyin.xx.xx:Declaration.Signature.Date > DATE(TODAY)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^SMSFAR194 > Today()
    
            Data Elements:
    
            ^SMSFAR194 = INT:Declaration:Declaration.Signature.Date WHERE [Declaration.StatementType.Code='TrueAndCorrect']
            */
            assertion = (report.SMSFAR194.GetValueOrDefault() > DateTime.Now.Date);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430255",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Signature date must not be later than today",
                    Location = "/xbrli:xbrl/tns:Declaration" + OccurrenceIndex(report.DeclarationCollection_TrueAndCorrect_INTOccurrenceIndex) + "/tns:Declaration.Signature.Date",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.430255" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR194", Value = GetValueOrEmptyDate(report.SMSFAR194) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.430255
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN430255_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.430255
            Signature date must not be later than today
    
            Legacy Rule Format:
            IF pyin.xx.xx:Declaration.Signature.Date > DATE(TODAY)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^SMSFAR192 > Today()
    
            Data Elements:
    
            ^SMSFAR192 = RP:Declaration:Declaration.Signature.Date WHERE [Declaration.StatementType.Code='TrueAndCorrect']
            */
            assertion = (report.SMSFAR192.GetValueOrDefault() > DateTime.Now.Date);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430255",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Signature date must not be later than today",
                    Location = "/xbrli:xbrl/tns:Declaration" + OccurrenceIndex(report.DeclarationCollection_TrueAndCorrect_RPOccurrenceIndex) + "/tns:Declaration.Signature.Date",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.430255" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR192", Value = GetValueOrEmptyDate(report.SMSFAR192) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.430266
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN430266()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.430266
            Invalid context. The number of Reporting party: Australian contexts must not exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.JAUS)) > 1 
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.JAUS
            */
            assertion = (Count(report.RPJAUSCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430266",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Reporting party: Australian contexts must not exceed 1",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.430266" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.430267
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN430267()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.430267
            Invalid context. The number of Reporting party: Foreign contexts must not exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.JFOREIGN)) > 1 
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.JFOREIGN
            */
            assertion = (Count(report.RPJFOREIGNCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430267",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Reporting party: Foreign contexts must not exceed 1",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJFOREIGNLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.430267" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.430280
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN430280()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.430280
            Invalid context. The number of Reporting party Instant: Closing contexts must not exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.Closing(Instant))) > 1 
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP.Closing(Instant)
            */
            assertion = (Count(report.RPClosingInstantCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430280",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Reporting Party Instant: Closing contexts must not exceed 1",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPClosingInstantLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.430280" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.430297
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN430297()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.430297
            Invalid context. The number of Intermediary contexts must not exceed 1
    
            Legacy Rule Format:
            IF LENGTH(bafot.xx.xx:RegulatoryDisclosures.GeneralInformationAboutFinancialStatements.Text) > 3000
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^SMSFAR355) > 3000
    
            Data Elements:
    
            ^SMSFAR355 = RP:RegulatoryDisclosures.GeneralInformationAboutFinancialStatements.Text
            */
            assertion = (Length(report.SMSFAR355) > 3000);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430297",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The Other attachments schedule must not exceed 3000 characters",
                    LongDescription = @"The 'Other attachments' schedule is represented by the 'Regulatory Disclosures. General Information About Financial Statements.Text' element",
                    Location = "/xbrli:xbrl/tns:RegulatoryDisclosures.GeneralInformationAboutFinancialStatements.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.430297" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR355", Value = GetValueOrEmpty(report.SMSFAR355) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.430302
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN430302()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.430302
            Invalid context. The number of Reporting party: TOFA contexts must not exceed 1
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.TOFA)) > 1 
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.TOFA
            */
            assertion = (Count(report.RPTOFACount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430302",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Reporting party: TOFA contexts must not exceed 1",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPTOFALastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.430302" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.430316
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN430316()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.430316
            ABN must be present in the Intermediary context
    
            Legacy Rule Format:
            IF entity.identifier.ABN = NULLORBLANK WHERE CONTEXT = "INT"
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^ABN = BLANK
    
            Data Elements:
    
            ^ABN = INT
    
            ^Context = INT
            */
            assertion = (report.INTExists != false && string.IsNullOrWhiteSpace(report.INTIdentifierABN) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430316",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. ABN must be present in the Intermediary context",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.INTOccurrenceIndex) + "/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.430316" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.432037
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN432037()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.432037
            Bank account name must not contain invalid characters _ ! @ $ % [ ] ? = : ; " ,
    
            Legacy Rule Format:
            IF pyid.xx.xx:FinancialInstitutionAccount.FinancialInstitutionAccountName.Text CONTAINS SET("_", "!", "@", "$", "%", "]", "[", "?", "=", ":", ";", """, ",")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ContainsSet(^SMSFAR295, '"_", "!", "@", "$", "%", "]", "[", "?", "=", ":", ";", """, ","')
    
            Data Elements:
    
            ^SMSFAR295 = RP.IncomeTaxRefund:FinancialInstitutionAccount:FinancialInstitutionAccount.FinancialInstitutionAccountName.Text
            */
            assertion = IsMatch(report.SMSFAR295, @"_|!|@|\$|%|\]|\[|\?|=|:|;|""|,", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.432037",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Bank account name must not contain invalid characters _ ! @ $ % [ ] ? = : ; "" ,",
                    Location = "/xbrli:xbrl/tns:FinancialInstitutionAccount/tns:FinancialInstitutionAccount.FinancialInstitutionAccountName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.432037" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR295", Value = report.SMSFAR295 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.432037
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN432037_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.432037
            Bank account name must not contain invalid characters _ ! @ $ % [ ] ? = : ; " ,
    
            Legacy Rule Format:
            IF pyid.xx.xx:FinancialInstitutionAccount.FinancialInstitutionAccountName.Text CONTAINS SET("_", "!", "@", "$", "%", "]", "[", "?", "=", ":", ";", """, ",")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ContainsSet(^SMSFAR366, '"_", "!", "@", "$", "%", "]", "[", "?", "=", ":", ";", """, ","')
    
            Data Elements:
    
            ^SMSFAR366 = RP.Superannuation:FinancialInstitutionAccount:FinancialInstitutionAccount.FinancialInstitutionAccountName.Text
            */
            assertion = IsMatch(report.SMSFAR366, @"_|!|@|\$|%|\]|\[|\?|=|:|;|""|,", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.432037",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Bank account name must not contain invalid characters _ ! @ $ % [ ] ? = : ; "" ,",
                    Location = "/xbrli:xbrl/tns:FinancialInstitutionAccount/tns:FinancialInstitutionAccount.FinancialInstitutionAccountName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.432037" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR366", Value = report.SMSFAR366 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.432267
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN432267()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.432267
            Financial institution name and branch name must not be present
    
            Legacy Rule Format:
            IF (pyid.xx.xx:FinancialInstitutionAccount.FinancialInstitutionBranchName.Text) <> NULLORBLANK OR (pyid.xx.xx:FinancialInstitutionAccount.FinancialInstitutionName.Text <> NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^SMSFAR296 <> BLANK OR (^SMSFAR297 <> BLANK)
    
            Data Elements:
    
            ^SMSFAR297 = RP.IncomeTaxRefund:FinancialInstitutionAccount:FinancialInstitutionAccount.FinancialInstitutionName.Text
    
            ^SMSFAR296 = RP.IncomeTaxRefund:FinancialInstitutionAccount:FinancialInstitutionAccount.FinancialInstitutionBranchName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.SMSFAR296) != true || string.IsNullOrWhiteSpace(report.SMSFAR297) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.432267",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Financial institution name and branch name must not be present",
                    Location = "/xbrli:xbrl/tns:FinancialInstitutionAccount/tns:FinancialInstitutionAccount.FinancialInstitutionName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.432267" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR296", Value = GetValueOrEmpty(report.SMSFAR296) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR297", Value = GetValueOrEmpty(report.SMSFAR297) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.432267
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN432267_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.432267
            Financial institution name and branch name must not be present
    
            Legacy Rule Format:
            IF (pyid.xx.xx:FinancialInstitutionAccount.FinancialInstitutionBranchName.Text) <> NULLORBLANK OR (pyid.xx.xx:FinancialInstitutionAccount.FinancialInstitutionName.Text <> NULLORBLANK)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^SMSFAR367 <> BLANK OR (^SMSFAR368 <> BLANK)
    
            Data Elements:
    
            ^SMSFAR368 = RP.Superannuation:FinancialInstitutionAccount:FinancialInstitutionAccount.FinancialInstitutionName.Text
    
            ^SMSFAR367 = RP.Superannuation:FinancialInstitutionAccount:FinancialInstitutionAccount.FinancialInstitutionBranchName.Text
            */
            assertion = (string.IsNullOrWhiteSpace(report.SMSFAR367) != true || string.IsNullOrWhiteSpace(report.SMSFAR368) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.432267",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Financial institution name and branch name must not be present",
                    Location = "/xbrli:xbrl/tns:FinancialInstitutionAccount/tns:FinancialInstitutionAccount.FinancialInstitutionName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.432267" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR367", Value = GetValueOrEmpty(report.SMSFAR367) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR368", Value = GetValueOrEmpty(report.SMSFAR368) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.432417
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN432417()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.432417
            ABN for reporting party must be provided when there is no intermediary
    
            Legacy Rule Format:
            IF CONTEXT(INT) = NULL AND RP:pyid.xx.xx:Identifiers.AustralianBusinessNumber.Identifier = NULLORBLANK
                    RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context = NULL AND ^SMSFAR5 = BLANK
    
            Data Elements:
    
            ^SMSFAR5 = RP:Identifiers.AustralianBusinessNumber.Identifier
    
            ^Context = INT
            */
            assertion = (report.INTExists == false && string.IsNullOrWhiteSpace(report.SMSFAR5) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.432417",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"ABN for reporting party must be provided when there is no intermediary",
                    Location = "/xbrli:xbrl/tns:Identifiers.AustralianBusinessNumber.Identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.432417" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR5", Value = GetValueOrEmpty(report.SMSFAR5) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.434223
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN434223()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.434223
            ABN is invalid
    
            Legacy Rule Format:
            IF (pyid.xx.xx:Identifiers.AustralianBusinessNumber.Identifier <> NULLORBLANK) and (ABNALGORITHM(pyid.xx.xx:Identifiers.AustralianBusinessNumber.Identifier) = FALSE)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^SMSFAR175 <> BLANK) and (FailsABNAlgorithm(^SMSFAR175))
    
            Data Elements:
    
            ^SMSFAR175 = RP:TrusteeOrDirector:Identifiers.AustralianBusinessNumber.Identifier
            */
            assertion = (string.IsNullOrWhiteSpace(report.SMSFAR175) != true && FailsABNAlgorithm(report.SMSFAR175));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000477",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"ABN is invalid",
                    Location = "/xbrli:xbrl/tns:TrusteeOrDirector/tns:Identifiers.AustralianBusinessNumber.Identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.434223" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR175", Value = GetValueOrEmpty(report.SMSFAR175) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.434223
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN434223_1(SMSFAR2026.MemberInformation memberInformation, int itemIndex1)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.434223
            ABN is invalid

            Legacy Rule Format:
            IF (pyid.xx.xx:Identifiers.AustralianBusinessNumber.Identifier <> NULLORBLANK) and (ABNALGORITHM(pyid.xx.xx:Identifiers.AustralianBusinessNumber.Identifier) = FALSE)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^SMSFAR204 <> BLANK) and (FailsABNAlgorithm(^SMSFAR204))

            Data Elements:

            ^SMSFAR204 = RP:MemberInformation:Identifiers.AustralianBusinessNumber.Identifier
            */
            assertion = (string.IsNullOrWhiteSpace(memberInformation.SMSFAR204) != true && FailsABNAlgorithm(memberInformation.SMSFAR204));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000477",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"ABN is invalid",
                    Location = "/xbrli:xbrl/tns:MemberInformation" + OccurrenceIndex(memberInformation.OccurrenceIndex) + "/tns:Identifiers.AustralianBusinessNumber.Identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.434223" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR204", Value = GetValueOrEmpty(memberInformation.SMSFAR204) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.434223
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN434223_2()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.434223
            ABN is invalid
    
            Legacy Rule Format:
            IF (pyid.xx.xx:Identifiers.AustralianBusinessNumber.Identifier <> NULLORBLANK) and (ABNALGORITHM(pyid.xx.xx:Identifiers.AustralianBusinessNumber.Identifier) = FALSE)
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^SMSFAR5 <> BLANK) and (FailsABNAlgorithm(^SMSFAR5))
    
            Data Elements:
    
            ^SMSFAR5 = RP:Identifiers.AustralianBusinessNumber.Identifier
            */
            assertion = (string.IsNullOrWhiteSpace(report.SMSFAR5) != true && FailsABNAlgorithm(report.SMSFAR5));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000477",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"ABN is invalid",
                    Location = "/xbrli:xbrl/tns:Identifiers.AustralianBusinessNumber.Identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.434223" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR5", Value = GetValueOrEmpty(report.SMSFAR5) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.436001
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN436001()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.436001
            An individual's first name must begin with an alpha character in the range 'A to Z' or 'a to z'
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.GivenName.Text DOES NOT STARTWITH SET("A-Z","a-z"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^SMSFAR180 <> BLANK) AND (NOT StartsWithSet(^SMSFAR180,'"A-Z","a-z"'))
    
            Data Elements:
    
            ^SMSFAR180 = INT:PersonNameDetails:PersonNameDetails.GivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.SMSFAR180) != true && !(IsMatch(report.SMSFAR180, @"^([A-Z]|[a-z])", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.436001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"An individual's first name must begin with an alpha character in the range 'A to Z' or 'a to z'",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_Contact_C_INTOccurrenceIndex) + "/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.436001" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR180", Value = report.SMSFAR180 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.436001
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN436001_1(SMSFAR2026.MemberInformation memberInformation, int itemIndex1)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.436001
            An individual's first name must begin with an alpha character in the range 'A to Z' or 'a to z'

            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.GivenName.Text DOES NOT STARTWITH SET("A-Z","a-z"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^SMSFAR199 <> BLANK) AND (NOT StartsWithSet(^SMSFAR199,'"A-Z","a-z"'))

            Data Elements:

            ^SMSFAR199 = RP:MemberInformation:PersonNameDetails:PersonNameDetails.GivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Identifier'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(memberInformation.SMSFAR199) != true && !(IsMatch(memberInformation.SMSFAR199, @"^([A-Z]|[a-z])", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.436001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"An individual's first name must begin with an alpha character in the range 'A to Z' or 'a to z'",
                    Location = "/xbrli:xbrl/tns:MemberInformation" + OccurrenceIndex(memberInformation.OccurrenceIndex) + "/tns:PersonNameDetails" + OccurrenceIndex(memberInformation.MemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RPOccurrenceIndex) + "/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.436001" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR199", Value = GetValueOrEmpty(memberInformation.SMSFAR199) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.436001
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN436001_2()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.436001
            An individual's first name must begin with an alpha character in the range 'A to Z' or 'a to z'
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.GivenName.Text DOES NOT STARTWITH SET("A-Z","a-z"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^SMSFAR169 <> BLANK) AND (NOT StartsWithSet(^SMSFAR169,'"A-Z","a-z"'))
    
            Data Elements:
    
            ^SMSFAR169 = RP:TrusteeOrDirector:PersonNameDetails:PersonNameDetails.GivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.SMSFAR169) != true && !(IsMatch(report.SMSFAR169, @"^([A-Z]|[a-z])", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.436001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"An individual's first name must begin with an alpha character in the range 'A to Z' or 'a to z'",
                    Location = "/xbrli:xbrl/tns:TrusteeOrDirector/tns:PersonNameDetails" + OccurrenceIndex(report.TrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RPOccurrenceIndex) + "/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.436001" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR169", Value = report.SMSFAR169 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.436001
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN436001_3()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.436001
            An individual's first name must begin with an alpha character in the range 'A to Z' or 'a to z'
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text <> NULLORBLANK) AND (pyde.xx.xx:PersonNameDetails.GivenName.Text DOES NOT STARTWITH SET("A-Z","a-z"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^SMSFAR13 <> BLANK) AND (NOT StartsWithSet(^SMSFAR13,'"A-Z","a-z"'))
    
            Data Elements:
    
            ^SMSFAR13 = Auditor:PersonNameDetails:PersonNameDetails.GivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.SMSFAR13) != true && !(IsMatch(report.SMSFAR13, @"^([A-Z]|[a-z])", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.436001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"An individual's first name must begin with an alpha character in the range 'A to Z' or 'a to z'",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_Contact_C_AuditorOccurrenceIndex) + "/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.436001" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR13", Value = report.SMSFAR13 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438019
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438019()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438019
            If an exemption or rollover has been applied then the exemption or rollover code(s) must be supplied
    
            Legacy Rule Format:
            IF (RP:rvctc3.xx.xx:CapitalGainsTax.ExemptionOrRolloverApplied.Indicator = TRUE) AND (COUNT(RP:rvctc3.xx.xx:CapitalGainsTax.ExemptionOrRolloverApplied.Code <> NULL) = 0)
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^SMSFAR327 = TRUE) AND (Count(^SMSFAR328) = 0)
    
            Data Elements:
    
            ^SMSFAR327 = RP:CapitalGainsTax.ExemptionOrRolloverApplied.Indicator
    
            ^SMSFAR328 = RP:CapitalGainsTax.ExemptionOrRolloverApplied.Code
            */
            assertion = (report.SMSFAR327 == true && Count(report.SMSFAR328Collection) == 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438019",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Exemption or rollover code not supplied",
                    LongDescription = @"If an exemption or rollover has been applied then the exemption or rollover code(s) must be supplied",
                    Location = "/xbrli:xbrl/tns:CapitalGainsTax.ExemptionOrRolloverApplied.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438019" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR327", Value = GetValueOrEmpty(report.SMSFAR327) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438021
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438021()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438021
            If an exemption or rollover code(s) is provided then the exemption or rollover applied indicator must be true
    
            Legacy Rule Format:
            IF (COUNT(RP:rvctc3.xx.xx:CapitalGainsTax.ExemptionOrRolloverApplied.Code <> NULL) > 0) AND (RP:rvctc3.xx.xx:CapitalGainsTax.ExemptionOrRolloverApplied.Indicator <> TRUE)
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (Count(^SMSFAR328) > 0) AND (^SMSFAR327 <> TRUE)
    
            Data Elements:
    
            ^SMSFAR327 = RP:CapitalGainsTax.ExemptionOrRolloverApplied.Indicator
    
            ^SMSFAR328 = RP:CapitalGainsTax.ExemptionOrRolloverApplied.Code
            */
            assertion = (Count(report.SMSFAR328Collection) > 0 && report.SMSFAR327 != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438021",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Exemption or rollover applied indicator must be true",
                    LongDescription = @"If an exemption or rollover code(s) is provided then the exemption or rollover applied indicator must be true",
                    Location = "/xbrli:xbrl/tns:CapitalGainsTax.ExemptionOrRolloverApplied.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438021" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR327", Value = GetValueOrEmpty(report.SMSFAR327) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438029
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438029()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP
    
            ^Context = RP
            */
            assertion = (report.RPExists != false && report.RPIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438029
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438029_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP(Instant)
    
            ^Context = RP(Instant)
            */
            assertion = (report.RPInstantExists != false && report.RPInstantIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPInstantOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438029
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438029_2()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.Accumulation
    
            ^Context = RP:RP.Accumulation
            */
            assertion = (report.RPAccumulationExists != false && report.RPAccumulationIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPAccumulationOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438029
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438029_3()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.Closing
    
            ^Context = RP:RP.Closing
            */
            assertion = (report.RPClosingExists != false && report.RPClosingIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPClosingOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438029
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438029_4()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP.Closing(Instant)
    
            ^Context = RP.Closing(Instant)
            */
            assertion = (report.RPClosingInstantExists != false && report.RPClosingInstantIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPClosingInstantOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438029
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438029_5()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP.Closing:RP.Closing.Accumulation
    
            ^Context = RP.Closing:RP.Closing.Accumulation
            */
            assertion = (report.RPClosingAccumulationExists != false && report.RPClosingAccumulationIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPClosingAccumulationOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438029
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438029_6()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP.Closing:RP.Closing.JAUS(Instant)
    
            ^Context = RP.Closing:RP.Closing.JAUS(Instant)
            */
            assertion = (report.RPClosingJAUSInstantExists != false && report.RPClosingJAUSInstantIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPClosingJAUSInstantOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438029
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438029_7()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP.Closing:RP.Closing.JFOREIGN(Instant)
    
            ^Context = RP.Closing:RP.Closing.JFOREIGN(Instant)
            */
            assertion = (report.RPClosingJFOREIGNInstantExists != false && report.RPClosingJFOREIGNInstantIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPClosingJFOREIGNInstantOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438029
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438029_8()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP.Closing:RP.Closing.Retirement
    
            ^Context = RP.Closing:RP.Closing.Retirement
            */
            assertion = (report.RPClosingRetirementExists != false && report.RPClosingRetirementIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPClosingRetirementOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438029
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438029_9()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.Deductible
    
            ^Context = RP:RP.Deductible
            */
            assertion = (report.RPDeductibleExists != false && report.RPDeductibleIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPDeductibleOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438029
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438029_10()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP.Deductible:RP.Deductible.JAUS
    
            ^Context = RP.Deductible:RP.Deductible.JAUS
            */
            assertion = (report.RPDeductibleJAUSExists != false && report.RPDeductibleJAUSIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPDeductibleJAUSOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438029
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438029_11()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP.Deductible:RP.Deductible.JFOREIGN
    
            ^Context = RP.Deductible:RP.Deductible.JFOREIGN
            */
            assertion = (report.RPDeductibleJFOREIGNExists != false && report.RPDeductibleJFOREIGNIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPDeductibleJFOREIGNOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438029
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438029_12()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.IncomeTaxRefund
    
            ^Context = RP:RP.IncomeTaxRefund
            */
            assertion = (report.RPIncomeTaxRefundExists != false && report.RPIncomeTaxRefundIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPIncomeTaxRefundOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438029
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438029_13()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.Investor
    
            ^Context = RP:RP.Investor
            */
            assertion = (report.RPInvestorExists != false && report.RPInvestorIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPInvestorOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438029
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438029_14()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP.Investor:RP.Investor.YPrevious
    
            ^Context = RP.Investor:RP.Investor.YPrevious
            */
            assertion = (report.RPInvestorYPreviousExists != false && report.RPInvestorYPreviousIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPInvestorYPreviousOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438029
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438029_15()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.JAUS
    
            ^Context = RP:RP.JAUS
            */
            assertion = (report.RPJAUSExists != false && report.RPJAUSIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438029
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438029_16()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP.JAUS:RP.JAUS.AustNonResidentialRealProperty
    
            ^Context = RP.JAUS:RP.JAUS.AustNonResidentialRealProperty
            */
            assertion = (report.RPJAUSAustNonResidentialRealPropertyExists != false && report.RPJAUSAustNonResidentialRealPropertyIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSAustNonResidentialRealPropertyOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438029
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438029_17()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP.JAUS:RP.JAUS.AustralianShares
    
            ^Context = RP.JAUS:RP.JAUS.AustralianShares
            */
            assertion = (report.RPJAUSAustralianSharesExists != false && report.RPJAUSAustralianSharesIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSAustralianSharesOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438029
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438029_18()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP.JAUS:RP.JAUS.AustResidentialRealProperty
    
            ^Context = RP.JAUS:RP.JAUS.AustResidentialRealProperty
            */
            assertion = (report.RPJAUSAustResidentialRealPropertyExists != false && report.RPJAUSAustResidentialRealPropertyIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSAustResidentialRealPropertyOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438029
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438029_19()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP.JAUS:RP.JAUS.Other
    
            ^Context = RP.JAUS:RP.JAUS.Other
            */
            assertion = (report.RPJAUSOtherExists != false && report.RPJAUSOtherIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSOtherOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438029
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438029_20()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP.JAUS:RP.JAUS.OtherShares
    
            ^Context = RP.JAUS:RP.JAUS.OtherShares
            */
            assertion = (report.RPJAUSOtherSharesExists != false && report.RPJAUSOtherSharesIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSOtherSharesOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438029
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438029_21()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP.JAUS:RP.JAUS.OverseasRealProperty
    
            ^Context = RP.JAUS:RP.JAUS.OverseasRealProperty
            */
            assertion = (report.RPJAUSOverseasRealPropertyExists != false && report.RPJAUSOverseasRealPropertyIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSOverseasRealPropertyOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438029
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438029_22()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.JFOREIGN
    
            ^Context = RP:RP.JFOREIGN
            */
            assertion = (report.RPJFOREIGNExists != false && report.RPJFOREIGNIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJFOREIGNOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438029
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438029_23()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.LimitedPartners
    
            ^Context = RP:RP.LimitedPartners
            */
            assertion = (report.RPLimitedPartnersExists != false && report.RPLimitedPartnersIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPLimitedPartnersOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438029
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438029_24()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP.LimitedPartners:RP.LimitedPartners.YPrevious
    
            ^Context = RP.LimitedPartners:RP.LimitedPartners.YPrevious
            */
            assertion = (report.RPLimitedPartnersYPreviousExists != false && report.RPLimitedPartnersYPreviousIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPLimitedPartnersYPreviousOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438029
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438029_25()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.NonDeductible
    
            ^Context = RP:RP.NonDeductible
            */
            assertion = (report.RPNonDeductibleExists != false && report.RPNonDeductibleIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPNonDeductibleOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438029
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438029_26()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP.NonDeductible:RP.NonDeductible.JAUS
    
            ^Context = RP.NonDeductible:RP.NonDeductible.JAUS
            */
            assertion = (report.RPNonDeductibleJAUSExists != false && report.RPNonDeductibleJAUSIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPNonDeductibleJAUSOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438029
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438029_27()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP.NonDeductible:RP.NonDeductible.JFOREIGN
    
            ^Context = RP.NonDeductible:RP.NonDeductible.JFOREIGN
            */
            assertion = (report.RPNonDeductibleJFOREIGNExists != false && report.RPNonDeductibleJFOREIGNIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPNonDeductibleJFOREIGNOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438029
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438029_28()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.Retirement
    
            ^Context = RP:RP.Retirement
            */
            assertion = (report.RPRetirementExists != false && report.RPRetirementIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPRetirementOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438029
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438029_29()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.Superannuation
    
            ^Context = RP:RP.Superannuation
            */
            assertion = (report.RPSuperannuationExists != false && report.RPSuperannuationIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPSuperannuationOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438029
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438029_30()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.TOFA
    
            ^Context = RP:RP.TOFA
            */
            assertion = (report.RPTOFAExists != false && report.RPTOFAIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPTOFAOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438029
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438029_31()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438029
            Entity identifier scheme for ReportingParty must be "http://www.ato.gov.au/tfn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = "RprtPyType.xx.xx:ReportingParty") AND (entity.identifier.scheme <> "http://www.ato.gov.au/tfn")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^IdentifierScheme <> 'http://www.ato.gov.au/tfn')
    
            Data Elements:
    
            ^IdentifierScheme = RP:RP.Y0Plus.Closing(Instant)
    
            ^Context = RP:RP.Y0Plus.Closing(Instant)
            */
            assertion = (report.RPY0PlusClosingInstantExists != false && report.RPY0PlusClosingInstantIdentifierScheme != @"http://www.ato.gov.au/tfn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for ReportingParty must be ""http://www.ato.gov.au/tfn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPY0PlusClosingInstantOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438029" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438030
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438030()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438030
            Entity identifier scheme for Intermediary must be "http://www.ato.gov.au/abn"
    
            Legacy Rule Format:
            IF (RprtPyType.xx.xx:ReportPartyTypeDimension = “RprtPyType.xx.xx:Intermediary”) AND (entity.identifier.scheme <> “http://www.ato.gov.au/abn”)
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> NULL AND ^IdentifierScheme <> 'http://www.ato.gov.au/abn'
    
            Data Elements:
    
            ^IdentifierScheme = INT
    
            ^Context = INT
            */
            assertion = (report.INTExists != false && report.INTIdentifierScheme != @"http://www.ato.gov.au/abn");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438030",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Entity identifier scheme for Intermediary must be ""http://www.ato.gov.au/abn""",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.INTOccurrenceIndex) + "/xbrli:entity/xbrli:identifier/@scheme",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438030" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438034
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438034()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438034
            Both taxable income and tax losses carried forward to later income years are present.
    
            Legacy Rule Format:
            IF bafpr1.xx.xx:Income.Taxable.Amount > 0 AND
            rvctc3.xx.xx:Tax.Losses.CarriedForward.LaterIncomeYearsTotal.Amount > 0
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^SMSFAR73 > 0 AND ^SMSFAR85 > 0
    
            Data Elements:
    
            ^SMSFAR85 = RP.Closing:Tax.Losses.CarriedForward.LaterIncomeYearsTotal.Amount
    
            ^SMSFAR73 = RP:Income.Taxable.Amount
            */
            assertion = (report.SMSFAR73.GetValueOrDefault() > 0 && report.SMSFAR85.GetValueOrDefault() > 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438034",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Both taxable income and tax losses carried forward to later income years are present.",
                    LongDescription = @"Tax losses carried forward to later income years cannot be present with a positive taxable income.",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.CarriedForward.LaterIncomeYearsTotal.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438034" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR73", Value = GetValueOrEmpty(report.SMSFAR73) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR85", Value = GetValueOrEmpty(report.SMSFAR85) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438035
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438035()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438035
            Unstructured name must not  contain invalid characters ~ ^ < > + , * 
            Invalid characters ~^<>+  have been removed from the techincal business rule as they will be picked by the XBRL validator.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonUnstructuredName.FullName.Text <> NULLORBLANK AND pyde.xx.xx:PersonUnstructuredName.FullName.Text CONTAINS SET(",","*"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ContainsSet(^SMSFAR280, '",","*"')
    
            Data Elements:
    
            ^SMSFAR280 = INT:Declaration:PersonUnstructuredName:PersonUnstructuredName.FullName.Text WHERE [PersonUnstructuredName.Usage.Code='DeclarationSignatory']
            */
            assertion = IsMatch(report.SMSFAR280, @",|\*", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438035",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unstructured name must not contain invalid characters ~ ^ < > + , *",
                    Location = "/xbrli:xbrl/tns:Declaration" + OccurrenceIndex(report.DeclarationCollection_TrueAndCorrect_INTOccurrenceIndex) + "/tns:PersonUnstructuredName" + OccurrenceIndex(report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTOccurrenceIndex) + "/tns:PersonUnstructuredName.FullName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438035" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR280", Value = report.SMSFAR280 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438035
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438035_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438035
            Unstructured name must not  contain invalid characters ~ ^ < > + , * 
            Invalid characters ~^<>+  have been removed from the techincal business rule as they will be picked by the XBRL validator.
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonUnstructuredName.FullName.Text <> NULLORBLANK AND pyde.xx.xx:PersonUnstructuredName.FullName.Text CONTAINS SET(",","*"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ContainsSet(^SMSFAR278, '",","*"')
    
            Data Elements:
    
            ^SMSFAR278 = RP:Declaration:PersonUnstructuredName:PersonUnstructuredName.FullName.Text WHERE [PersonUnstructuredName.Usage.Code='DeclarationSignatory']
            */
            assertion = IsMatch(report.SMSFAR278, @",|\*", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438035",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Unstructured name must not contain invalid characters ~ ^ < > + , *",
                    Location = "/xbrli:xbrl/tns:Declaration" + OccurrenceIndex(report.DeclarationCollection_TrueAndCorrect_RPOccurrenceIndex) + "/tns:PersonUnstructuredName" + OccurrenceIndex(report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPOccurrenceIndex) + "/tns:PersonUnstructuredName.FullName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438035" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR278", Value = report.SMSFAR278 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438036
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438036()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438036
            Tax on taxable income must be supplied
    
            Legacy Rule Format:
            IF (RP:rvctc1.xx.xx:IncomeTax.TotalTaxOnTaxableIncome.Amount) = NULL
             RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^SMSFAR315 = NULL
    
            Data Elements:
    
            ^SMSFAR315 = RP:IncomeTax.TotalTaxOnTaxableIncome.Amount
            */
            assertion = (report.SMSFAR315 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438036",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax on taxable income must be supplied. A value must be included in the field, even if the value is zero.",
                    Location = "/xbrli:xbrl/tns:IncomeTax.TotalTaxOnTaxableIncome.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438036" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR315", Value = GetValueOrEmpty(report.SMSFAR315) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438053
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438053()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438053
            Period start date within context must match start date within reporting party context
    
            Legacy Rule Format:
            IF period.startDate  <> RP:period.startDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^StartDate  <> ^StartDate1)
    
            Data Elements:
    
            ^StartDate = Auditor
    
            ^StartDate1 = RP
    
            ^Context = Auditor
            */
            assertion = (report.AuditorExists != false && report.AuditorStartDate != report.RPStartDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438053",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must match Reporting Party period start date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.AuditorOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438053" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438053
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438053_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438053
            Period start date within context must match start date within reporting party context
    
            Legacy Rule Format:
            IF period.startDate  <> RP:period.startDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^StartDate  <> ^StartDate1)
    
            Data Elements:
    
            ^StartDate = INT
    
            ^StartDate1 = RP
    
            ^Context = INT
            */
            assertion = (report.INTExists != false && report.INTStartDate != report.RPStartDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438053",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must match Reporting Party period start date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.INTOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438053" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438053
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438053_2()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438053
            Period start date within context must match start date within reporting party context
    
            Legacy Rule Format:
            IF period.startDate  <> RP:period.startDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^StartDate  <> ^StartDate1)
    
            Data Elements:
    
            ^StartDate = RP:RP.Accumulation
    
            ^StartDate1 = RP
    
            ^Context = RP:RP.Accumulation
            */
            assertion = (report.RPAccumulationExists != false && report.RPAccumulationStartDate != report.RPStartDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438053",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must match Reporting Party period start date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPAccumulationOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438053" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438053
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438053_3()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438053
            Period start date within context must match start date within reporting party context
    
            Legacy Rule Format:
            IF period.startDate  <> RP:period.startDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^StartDate  <> ^StartDate1)
    
            Data Elements:
    
            ^StartDate = RP:RP.Closing
    
            ^StartDate1 = RP
    
            ^Context = RP:RP.Closing
            */
            assertion = (report.RPClosingExists != false && report.RPClosingStartDate != report.RPStartDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438053",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must match Reporting Party period start date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPClosingOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438053" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438053
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438053_4()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438053
            Period start date within context must match start date within reporting party context
    
            Legacy Rule Format:
            IF period.startDate  <> RP:period.startDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^StartDate  <> ^StartDate1)
    
            Data Elements:
    
            ^StartDate = RP.Closing:RP.Closing.Accumulation
    
            ^StartDate1 = RP
    
            ^Context = RP.Closing:RP.Closing.Accumulation
            */
            assertion = (report.RPClosingAccumulationExists != false && report.RPClosingAccumulationStartDate != report.RPStartDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438053",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must match Reporting Party period start date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPClosingAccumulationOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438053" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438053
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438053_5()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438053
            Period start date within context must match start date within reporting party context
    
            Legacy Rule Format:
            IF period.startDate  <> RP:period.startDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^StartDate  <> ^StartDate1)
    
            Data Elements:
    
            ^StartDate = RP.Closing:RP.Closing.Retirement
    
            ^StartDate1 = RP
    
            ^Context = RP.Closing:RP.Closing.Retirement
            */
            assertion = (report.RPClosingRetirementExists != false && report.RPClosingRetirementStartDate != report.RPStartDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438053",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must match Reporting Party period start date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPClosingRetirementOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438053" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438053
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438053_6()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438053
            Period start date within context must match start date within reporting party context
    
            Legacy Rule Format:
            IF period.startDate  <> RP:period.startDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^StartDate  <> ^StartDate1)
    
            Data Elements:
    
            ^StartDate = RP:RP.Deductible
    
            ^StartDate1 = RP
    
            ^Context = RP:RP.Deductible
            */
            assertion = (report.RPDeductibleExists != false && report.RPDeductibleStartDate != report.RPStartDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438053",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must match Reporting Party period start date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPDeductibleOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438053" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438053
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438053_7()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438053
            Period start date within context must match start date within reporting party context
    
            Legacy Rule Format:
            IF period.startDate  <> RP:period.startDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^StartDate  <> ^StartDate1)
    
            Data Elements:
    
            ^StartDate = RP.Deductible:RP.Deductible.JAUS
    
            ^StartDate1 = RP
    
            ^Context = RP.Deductible:RP.Deductible.JAUS
            */
            assertion = (report.RPDeductibleJAUSExists != false && report.RPDeductibleJAUSStartDate != report.RPStartDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438053",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must match Reporting Party period start date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPDeductibleJAUSOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438053" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438053
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438053_8()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438053
            Period start date within context must match start date within reporting party context
    
            Legacy Rule Format:
            IF period.startDate  <> RP:period.startDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^StartDate  <> ^StartDate1)
    
            Data Elements:
    
            ^StartDate = RP.Deductible:RP.Deductible.JFOREIGN
    
            ^StartDate1 = RP
    
            ^Context = RP.Deductible:RP.Deductible.JFOREIGN
            */
            assertion = (report.RPDeductibleJFOREIGNExists != false && report.RPDeductibleJFOREIGNStartDate != report.RPStartDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438053",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must match Reporting Party period start date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPDeductibleJFOREIGNOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438053" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438053
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438053_9()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438053
            Period start date within context must match start date within reporting party context
    
            Legacy Rule Format:
            IF period.startDate  <> RP:period.startDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^StartDate  <> ^StartDate1)
    
            Data Elements:
    
            ^StartDate = RP:RP.IncomeTaxRefund
    
            ^StartDate1 = RP
    
            ^Context = RP:RP.IncomeTaxRefund
            */
            assertion = (report.RPIncomeTaxRefundExists != false && report.RPIncomeTaxRefundStartDate != report.RPStartDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438053",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must match Reporting Party period start date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPIncomeTaxRefundOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438053" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438053
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438053_10()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438053
            Period start date within context must match start date within reporting party context
    
            Legacy Rule Format:
            IF period.startDate  <> RP:period.startDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^StartDate  <> ^StartDate1)
    
            Data Elements:
    
            ^StartDate = RP:RP.Investor
    
            ^StartDate1 = RP
    
            ^Context = RP:RP.Investor
            */
            assertion = (report.RPInvestorExists != false && report.RPInvestorStartDate != report.RPStartDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438053",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must match Reporting Party period start date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPInvestorOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438053" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438053
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438053_11()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438053
            Period start date within context must match start date within reporting party context
    
            Legacy Rule Format:
            IF period.startDate  <> RP:period.startDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^StartDate  <> ^StartDate1)
    
            Data Elements:
    
            ^StartDate = RP.Investor:RP.Investor.YPrevious
    
            ^StartDate1 = RP
    
            ^Context = RP.Investor:RP.Investor.YPrevious
            */
            assertion = (report.RPInvestorYPreviousExists != false && report.RPInvestorYPreviousStartDate != report.RPStartDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438053",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must match Reporting Party period start date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPInvestorYPreviousOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438053" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438053
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438053_12()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438053
            Period start date within context must match start date within reporting party context
    
            Legacy Rule Format:
            IF period.startDate  <> RP:period.startDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^StartDate  <> ^StartDate1)
    
            Data Elements:
    
            ^StartDate = RP:RP.JAUS
    
            ^StartDate1 = RP
    
            ^Context = RP:RP.JAUS
            */
            assertion = (report.RPJAUSExists != false && report.RPJAUSStartDate != report.RPStartDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438053",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must match Reporting Party period start date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438053" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438053
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438053_13()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438053
            Period start date within context must match start date within reporting party context
    
            Legacy Rule Format:
            IF period.startDate  <> RP:period.startDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^StartDate  <> ^StartDate1)
    
            Data Elements:
    
            ^StartDate = RP.JAUS:RP.JAUS.AustNonResidentialRealProperty
    
            ^StartDate1 = RP
    
            ^Context = RP.JAUS:RP.JAUS.AustNonResidentialRealProperty
            */
            assertion = (report.RPJAUSAustNonResidentialRealPropertyExists != false && report.RPJAUSAustNonResidentialRealPropertyStartDate != report.RPStartDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438053",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must match Reporting Party period start date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSAustNonResidentialRealPropertyOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438053" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438053
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438053_14()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438053
            Period start date within context must match start date within reporting party context
    
            Legacy Rule Format:
            IF period.startDate  <> RP:period.startDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^StartDate  <> ^StartDate1)
    
            Data Elements:
    
            ^StartDate = RP.JAUS:RP.JAUS.AustralianShares
    
            ^StartDate1 = RP
    
            ^Context = RP.JAUS:RP.JAUS.AustralianShares
            */
            assertion = (report.RPJAUSAustralianSharesExists != false && report.RPJAUSAustralianSharesStartDate != report.RPStartDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438053",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must match Reporting Party period start date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSAustralianSharesOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438053" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438053
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438053_15()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438053
            Period start date within context must match start date within reporting party context
    
            Legacy Rule Format:
            IF period.startDate  <> RP:period.startDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^StartDate  <> ^StartDate1)
    
            Data Elements:
    
            ^StartDate = RP.JAUS:RP.JAUS.AustResidentialRealProperty
    
            ^StartDate1 = RP
    
            ^Context = RP.JAUS:RP.JAUS.AustResidentialRealProperty
            */
            assertion = (report.RPJAUSAustResidentialRealPropertyExists != false && report.RPJAUSAustResidentialRealPropertyStartDate != report.RPStartDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438053",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must match Reporting Party period start date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSAustResidentialRealPropertyOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438053" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438053
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438053_16()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438053
            Period start date within context must match start date within reporting party context
    
            Legacy Rule Format:
            IF period.startDate  <> RP:period.startDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^StartDate  <> ^StartDate1)
    
            Data Elements:
    
            ^StartDate = RP.JAUS:RP.JAUS.Other
    
            ^StartDate1 = RP
    
            ^Context = RP.JAUS:RP.JAUS.Other
            */
            assertion = (report.RPJAUSOtherExists != false && report.RPJAUSOtherStartDate != report.RPStartDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438053",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must match Reporting Party period start date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSOtherOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438053" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438053
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438053_17()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438053
            Period start date within context must match start date within reporting party context
    
            Legacy Rule Format:
            IF period.startDate  <> RP:period.startDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^StartDate  <> ^StartDate1)
    
            Data Elements:
    
            ^StartDate = RP.JAUS:RP.JAUS.OtherShares
    
            ^StartDate1 = RP
    
            ^Context = RP.JAUS:RP.JAUS.OtherShares
            */
            assertion = (report.RPJAUSOtherSharesExists != false && report.RPJAUSOtherSharesStartDate != report.RPStartDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438053",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must match Reporting Party period start date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSOtherSharesOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438053" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438053
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438053_18()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438053
            Period start date within context must match start date within reporting party context
    
            Legacy Rule Format:
            IF period.startDate  <> RP:period.startDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^StartDate  <> ^StartDate1)
    
            Data Elements:
    
            ^StartDate = RP.JAUS:RP.JAUS.OverseasRealProperty
    
            ^StartDate1 = RP
    
            ^Context = RP.JAUS:RP.JAUS.OverseasRealProperty
            */
            assertion = (report.RPJAUSOverseasRealPropertyExists != false && report.RPJAUSOverseasRealPropertyStartDate != report.RPStartDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438053",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must match Reporting Party period start date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSOverseasRealPropertyOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438053" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438053
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438053_19()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438053
            Period start date within context must match start date within reporting party context
    
            Legacy Rule Format:
            IF period.startDate  <> RP:period.startDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^StartDate  <> ^StartDate1)
    
            Data Elements:
    
            ^StartDate = RP:RP.JFOREIGN
    
            ^StartDate1 = RP
    
            ^Context = RP:RP.JFOREIGN
            */
            assertion = (report.RPJFOREIGNExists != false && report.RPJFOREIGNStartDate != report.RPStartDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438053",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must match Reporting Party period start date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJFOREIGNOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438053" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438053
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438053_20()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438053
            Period start date within context must match start date within reporting party context
    
            Legacy Rule Format:
            IF period.startDate  <> RP:period.startDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^StartDate  <> ^StartDate1)
    
            Data Elements:
    
            ^StartDate = RP:RP.LimitedPartners
    
            ^StartDate1 = RP
    
            ^Context = RP:RP.LimitedPartners
            */
            assertion = (report.RPLimitedPartnersExists != false && report.RPLimitedPartnersStartDate != report.RPStartDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438053",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must match Reporting Party period start date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPLimitedPartnersOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438053" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438053
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438053_21()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438053
            Period start date within context must match start date within reporting party context
    
            Legacy Rule Format:
            IF period.startDate  <> RP:period.startDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^StartDate  <> ^StartDate1)
    
            Data Elements:
    
            ^StartDate = RP.LimitedPartners:RP.LimitedPartners.YPrevious
    
            ^StartDate1 = RP
    
            ^Context = RP.LimitedPartners:RP.LimitedPartners.YPrevious
            */
            assertion = (report.RPLimitedPartnersYPreviousExists != false && report.RPLimitedPartnersYPreviousStartDate != report.RPStartDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438053",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must match Reporting Party period start date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPLimitedPartnersYPreviousOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438053" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438053
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438053_22()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438053
            Period start date within context must match start date within reporting party context
    
            Legacy Rule Format:
            IF period.startDate  <> RP:period.startDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^StartDate  <> ^StartDate1)
    
            Data Elements:
    
            ^StartDate = RP:RP.NonDeductible
    
            ^StartDate1 = RP
    
            ^Context = RP:RP.NonDeductible
            */
            assertion = (report.RPNonDeductibleExists != false && report.RPNonDeductibleStartDate != report.RPStartDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438053",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must match Reporting Party period start date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPNonDeductibleOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438053" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438053
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438053_23()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438053
            Period start date within context must match start date within reporting party context
    
            Legacy Rule Format:
            IF period.startDate  <> RP:period.startDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^StartDate  <> ^StartDate1)
    
            Data Elements:
    
            ^StartDate = RP.NonDeductible:RP.NonDeductible.JAUS
    
            ^StartDate1 = RP
    
            ^Context = RP.NonDeductible:RP.NonDeductible.JAUS
            */
            assertion = (report.RPNonDeductibleJAUSExists != false && report.RPNonDeductibleJAUSStartDate != report.RPStartDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438053",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must match Reporting Party period start date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPNonDeductibleJAUSOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438053" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438053
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438053_24()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438053
            Period start date within context must match start date within reporting party context
    
            Legacy Rule Format:
            IF period.startDate  <> RP:period.startDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^StartDate  <> ^StartDate1)
    
            Data Elements:
    
            ^StartDate = RP.NonDeductible:RP.NonDeductible.JFOREIGN
    
            ^StartDate1 = RP
    
            ^Context = RP.NonDeductible:RP.NonDeductible.JFOREIGN
            */
            assertion = (report.RPNonDeductibleJFOREIGNExists != false && report.RPNonDeductibleJFOREIGNStartDate != report.RPStartDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438053",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must match Reporting Party period start date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPNonDeductibleJFOREIGNOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438053" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438053
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438053_25()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438053
            Period start date within context must match start date within reporting party context
    
            Legacy Rule Format:
            IF period.startDate  <> RP:period.startDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^StartDate  <> ^StartDate1)
    
            Data Elements:
    
            ^StartDate = RP:RP.Retirement
    
            ^StartDate1 = RP
    
            ^Context = RP:RP.Retirement
            */
            assertion = (report.RPRetirementExists != false && report.RPRetirementStartDate != report.RPStartDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438053",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must match Reporting Party period start date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPRetirementOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438053" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438053
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438053_26()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438053
            Period start date within context must match start date within reporting party context
    
            Legacy Rule Format:
            IF period.startDate  <> RP:period.startDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^StartDate  <> ^StartDate1)
    
            Data Elements:
    
            ^StartDate = RP:RP.Superannuation
    
            ^StartDate1 = RP
    
            ^Context = RP:RP.Superannuation
            */
            assertion = (report.RPSuperannuationExists != false && report.RPSuperannuationStartDate != report.RPStartDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438053",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must match Reporting Party period start date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPSuperannuationOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438053" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438053
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438053_27()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438053
            Period start date within context must match start date within reporting party context
    
            Legacy Rule Format:
            IF period.startDate  <> RP:period.startDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^StartDate  <> ^StartDate1)
    
            Data Elements:
    
            ^StartDate = RP:RP.TOFA
    
            ^StartDate1 = RP
    
            ^Context = RP:RP.TOFA
            */
            assertion = (report.RPTOFAExists != false && report.RPTOFAStartDate != report.RPStartDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438053",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period start date must match Reporting Party period start date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPTOFAOccurrenceIndex) + "/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438053" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438054
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438054()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438054
            Period end date within context must match end date within reporting party context
    
            Legacy Rule Format:
            IF period.endDate <> RP:period.endDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^EndDate1 <> ^EndDate)
    
            Data Elements:
    
            ^Context = Auditor
    
            ^EndDate1 = Auditor
    
            ^EndDate = RP
            */
            assertion = (report.AuditorExists != false && report.AuditorEndDate != report.RPEndDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438054",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period end date must match Reporting Party period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.AuditorOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438054" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438054
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438054_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438054
            Period end date within context must match end date within reporting party context
    
            Legacy Rule Format:
            IF period.endDate <> RP:period.endDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^EndDate1 <> ^EndDate)
    
            Data Elements:
    
            ^Context = INT
    
            ^EndDate1 = INT
    
            ^EndDate = RP
            */
            assertion = (report.INTExists != false && report.INTEndDate != report.RPEndDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438054",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period end date must match Reporting Party period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.INTOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438054" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438054
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438054_2()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438054
            Period end date within context must match end date within reporting party context
    
            Legacy Rule Format:
            IF period.endDate <> RP:period.endDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^EndDate1 <> ^EndDate)
    
            Data Elements:
    
            ^Context = RP:RP.Accumulation
    
            ^EndDate1 = RP:RP.Accumulation
    
            ^EndDate = RP
            */
            assertion = (report.RPAccumulationExists != false && report.RPAccumulationEndDate != report.RPEndDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438054",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period end date must match Reporting Party period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPAccumulationOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438054" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438054
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438054_3()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438054
            Period end date within context must match end date within reporting party context
    
            Legacy Rule Format:
            IF period.endDate <> RP:period.endDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^EndDate1 <> ^EndDate)
    
            Data Elements:
    
            ^Context = RP:RP.Closing
    
            ^EndDate1 = RP:RP.Closing
    
            ^EndDate = RP
            */
            assertion = (report.RPClosingExists != false && report.RPClosingEndDate != report.RPEndDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438054",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period end date must match Reporting Party period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPClosingOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438054" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438054
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438054_4()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438054
            Period end date within context must match end date within reporting party context
    
            Legacy Rule Format:
            IF period.endDate <> RP:period.endDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^EndDate1 <> ^EndDate)
    
            Data Elements:
    
            ^Context = RP.Closing:RP.Closing.Accumulation
    
            ^EndDate1 = RP.Closing:RP.Closing.Accumulation
    
            ^EndDate = RP
            */
            assertion = (report.RPClosingAccumulationExists != false && report.RPClosingAccumulationEndDate != report.RPEndDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438054",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period end date must match Reporting Party period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPClosingAccumulationOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438054" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438054
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438054_5()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438054
            Period end date within context must match end date within reporting party context
    
            Legacy Rule Format:
            IF period.endDate <> RP:period.endDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^EndDate1 <> ^EndDate)
    
            Data Elements:
    
            ^Context = RP.Closing:RP.Closing.Retirement
    
            ^EndDate1 = RP.Closing:RP.Closing.Retirement
    
            ^EndDate = RP
            */
            assertion = (report.RPClosingRetirementExists != false && report.RPClosingRetirementEndDate != report.RPEndDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438054",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period end date must match Reporting Party period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPClosingRetirementOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438054" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438054
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438054_6()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438054
            Period end date within context must match end date within reporting party context
    
            Legacy Rule Format:
            IF period.endDate <> RP:period.endDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^EndDate1 <> ^EndDate)
    
            Data Elements:
    
            ^Context = RP:RP.Deductible
    
            ^EndDate1 = RP:RP.Deductible
    
            ^EndDate = RP
            */
            assertion = (report.RPDeductibleExists != false && report.RPDeductibleEndDate != report.RPEndDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438054",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period end date must match Reporting Party period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPDeductibleOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438054" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438054
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438054_7()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438054
            Period end date within context must match end date within reporting party context
    
            Legacy Rule Format:
            IF period.endDate <> RP:period.endDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^EndDate1 <> ^EndDate)
    
            Data Elements:
    
            ^Context = RP.Deductible:RP.Deductible.JAUS
    
            ^EndDate1 = RP.Deductible:RP.Deductible.JAUS
    
            ^EndDate = RP
            */
            assertion = (report.RPDeductibleJAUSExists != false && report.RPDeductibleJAUSEndDate != report.RPEndDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438054",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period end date must match Reporting Party period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPDeductibleJAUSOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438054" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438054
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438054_8()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438054
            Period end date within context must match end date within reporting party context
    
            Legacy Rule Format:
            IF period.endDate <> RP:period.endDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^EndDate1 <> ^EndDate)
    
            Data Elements:
    
            ^Context = RP.Deductible:RP.Deductible.JFOREIGN
    
            ^EndDate1 = RP.Deductible:RP.Deductible.JFOREIGN
    
            ^EndDate = RP
            */
            assertion = (report.RPDeductibleJFOREIGNExists != false && report.RPDeductibleJFOREIGNEndDate != report.RPEndDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438054",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period end date must match Reporting Party period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPDeductibleJFOREIGNOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438054" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438054
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438054_9()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438054
            Period end date within context must match end date within reporting party context
    
            Legacy Rule Format:
            IF period.endDate <> RP:period.endDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^EndDate1 <> ^EndDate)
    
            Data Elements:
    
            ^Context = RP:RP.IncomeTaxRefund
    
            ^EndDate1 = RP:RP.IncomeTaxRefund
    
            ^EndDate = RP
            */
            assertion = (report.RPIncomeTaxRefundExists != false && report.RPIncomeTaxRefundEndDate != report.RPEndDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438054",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period end date must match Reporting Party period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPIncomeTaxRefundOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438054" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438054
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438054_10()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438054
            Period end date within context must match end date within reporting party context
    
            Legacy Rule Format:
            IF period.endDate <> RP:period.endDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^EndDate1 <> ^EndDate)
    
            Data Elements:
    
            ^Context = RP:RP.Investor
    
            ^EndDate1 = RP:RP.Investor
    
            ^EndDate = RP
            */
            assertion = (report.RPInvestorExists != false && report.RPInvestorEndDate != report.RPEndDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438054",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period end date must match Reporting Party period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPInvestorOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438054" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438054
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438054_11()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438054
            Period end date within context must match end date within reporting party context
    
            Legacy Rule Format:
            IF period.endDate <> RP:period.endDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^EndDate1 <> ^EndDate)
    
            Data Elements:
    
            ^Context = RP.Investor:RP.Investor.YPrevious
    
            ^EndDate1 = RP.Investor:RP.Investor.YPrevious
    
            ^EndDate = RP
            */
            assertion = (report.RPInvestorYPreviousExists != false && report.RPInvestorYPreviousEndDate != report.RPEndDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438054",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period end date must match Reporting Party period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPInvestorYPreviousOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438054" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438054
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438054_12()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438054
            Period end date within context must match end date within reporting party context
    
            Legacy Rule Format:
            IF period.endDate <> RP:period.endDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^EndDate1 <> ^EndDate)
    
            Data Elements:
    
            ^Context = RP:RP.JAUS
    
            ^EndDate1 = RP:RP.JAUS
    
            ^EndDate = RP
            */
            assertion = (report.RPJAUSExists != false && report.RPJAUSEndDate != report.RPEndDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438054",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period end date must match Reporting Party period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438054" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438054
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438054_13()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438054
            Period end date within context must match end date within reporting party context
    
            Legacy Rule Format:
            IF period.endDate <> RP:period.endDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^EndDate1 <> ^EndDate)
    
            Data Elements:
    
            ^Context = RP.JAUS:RP.JAUS.AustNonResidentialRealProperty
    
            ^EndDate1 = RP.JAUS:RP.JAUS.AustNonResidentialRealProperty
    
            ^EndDate = RP
            */
            assertion = (report.RPJAUSAustNonResidentialRealPropertyExists != false && report.RPJAUSAustNonResidentialRealPropertyEndDate != report.RPEndDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438054",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period end date must match Reporting Party period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSAustNonResidentialRealPropertyOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438054" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438054
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438054_14()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438054
            Period end date within context must match end date within reporting party context
    
            Legacy Rule Format:
            IF period.endDate <> RP:period.endDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^EndDate1 <> ^EndDate)
    
            Data Elements:
    
            ^Context = RP.JAUS:RP.JAUS.AustralianShares
    
            ^EndDate1 = RP.JAUS:RP.JAUS.AustralianShares
    
            ^EndDate = RP
            */
            assertion = (report.RPJAUSAustralianSharesExists != false && report.RPJAUSAustralianSharesEndDate != report.RPEndDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438054",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period end date must match Reporting Party period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSAustralianSharesOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438054" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438054
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438054_15()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438054
            Period end date within context must match end date within reporting party context
    
            Legacy Rule Format:
            IF period.endDate <> RP:period.endDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^EndDate1 <> ^EndDate)
    
            Data Elements:
    
            ^Context = RP.JAUS:RP.JAUS.AustResidentialRealProperty
    
            ^EndDate1 = RP.JAUS:RP.JAUS.AustResidentialRealProperty
    
            ^EndDate = RP
            */
            assertion = (report.RPJAUSAustResidentialRealPropertyExists != false && report.RPJAUSAustResidentialRealPropertyEndDate != report.RPEndDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438054",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period end date must match Reporting Party period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSAustResidentialRealPropertyOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438054" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438054
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438054_16()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438054
            Period end date within context must match end date within reporting party context
    
            Legacy Rule Format:
            IF period.endDate <> RP:period.endDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^EndDate1 <> ^EndDate)
    
            Data Elements:
    
            ^Context = RP.JAUS:RP.JAUS.Other
    
            ^EndDate1 = RP.JAUS:RP.JAUS.Other
    
            ^EndDate = RP
            */
            assertion = (report.RPJAUSOtherExists != false && report.RPJAUSOtherEndDate != report.RPEndDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438054",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period end date must match Reporting Party period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSOtherOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438054" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438054
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438054_17()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438054
            Period end date within context must match end date within reporting party context
    
            Legacy Rule Format:
            IF period.endDate <> RP:period.endDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^EndDate1 <> ^EndDate)
    
            Data Elements:
    
            ^Context = RP.JAUS:RP.JAUS.OtherShares
    
            ^EndDate1 = RP.JAUS:RP.JAUS.OtherShares
    
            ^EndDate = RP
            */
            assertion = (report.RPJAUSOtherSharesExists != false && report.RPJAUSOtherSharesEndDate != report.RPEndDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438054",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period end date must match Reporting Party period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSOtherSharesOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438054" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438054
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438054_18()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438054
            Period end date within context must match end date within reporting party context
    
            Legacy Rule Format:
            IF period.endDate <> RP:period.endDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^EndDate1 <> ^EndDate)
    
            Data Elements:
    
            ^Context = RP.JAUS:RP.JAUS.OverseasRealProperty
    
            ^EndDate1 = RP.JAUS:RP.JAUS.OverseasRealProperty
    
            ^EndDate = RP
            */
            assertion = (report.RPJAUSOverseasRealPropertyExists != false && report.RPJAUSOverseasRealPropertyEndDate != report.RPEndDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438054",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period end date must match Reporting Party period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJAUSOverseasRealPropertyOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438054" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438054
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438054_19()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438054
            Period end date within context must match end date within reporting party context
    
            Legacy Rule Format:
            IF period.endDate <> RP:period.endDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^EndDate1 <> ^EndDate)
    
            Data Elements:
    
            ^Context = RP:RP.JFOREIGN
    
            ^EndDate1 = RP:RP.JFOREIGN
    
            ^EndDate = RP
            */
            assertion = (report.RPJFOREIGNExists != false && report.RPJFOREIGNEndDate != report.RPEndDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438054",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period end date must match Reporting Party period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPJFOREIGNOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438054" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438054
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438054_20()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438054
            Period end date within context must match end date within reporting party context
    
            Legacy Rule Format:
            IF period.endDate <> RP:period.endDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^EndDate1 <> ^EndDate)
    
            Data Elements:
    
            ^Context = RP:RP.LimitedPartners
    
            ^EndDate1 = RP:RP.LimitedPartners
    
            ^EndDate = RP
            */
            assertion = (report.RPLimitedPartnersExists != false && report.RPLimitedPartnersEndDate != report.RPEndDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438054",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period end date must match Reporting Party period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPLimitedPartnersOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438054" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438054
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438054_21()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438054
            Period end date within context must match end date within reporting party context
    
            Legacy Rule Format:
            IF period.endDate <> RP:period.endDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^EndDate1 <> ^EndDate)
    
            Data Elements:
    
            ^Context = RP.LimitedPartners:RP.LimitedPartners.YPrevious
    
            ^EndDate1 = RP.LimitedPartners:RP.LimitedPartners.YPrevious
    
            ^EndDate = RP
            */
            assertion = (report.RPLimitedPartnersYPreviousExists != false && report.RPLimitedPartnersYPreviousEndDate != report.RPEndDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438054",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period end date must match Reporting Party period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPLimitedPartnersYPreviousOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438054" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438054
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438054_22()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438054
            Period end date within context must match end date within reporting party context
    
            Legacy Rule Format:
            IF period.endDate <> RP:period.endDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^EndDate1 <> ^EndDate)
    
            Data Elements:
    
            ^Context = RP:RP.NonDeductible
    
            ^EndDate1 = RP:RP.NonDeductible
    
            ^EndDate = RP
            */
            assertion = (report.RPNonDeductibleExists != false && report.RPNonDeductibleEndDate != report.RPEndDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438054",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period end date must match Reporting Party period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPNonDeductibleOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438054" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438054
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438054_23()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438054
            Period end date within context must match end date within reporting party context
    
            Legacy Rule Format:
            IF period.endDate <> RP:period.endDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^EndDate1 <> ^EndDate)
    
            Data Elements:
    
            ^Context = RP.NonDeductible:RP.NonDeductible.JAUS
    
            ^EndDate1 = RP.NonDeductible:RP.NonDeductible.JAUS
    
            ^EndDate = RP
            */
            assertion = (report.RPNonDeductibleJAUSExists != false && report.RPNonDeductibleJAUSEndDate != report.RPEndDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438054",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period end date must match Reporting Party period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPNonDeductibleJAUSOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438054" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438054
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438054_24()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438054
            Period end date within context must match end date within reporting party context
    
            Legacy Rule Format:
            IF period.endDate <> RP:period.endDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^EndDate1 <> ^EndDate)
    
            Data Elements:
    
            ^Context = RP.NonDeductible:RP.NonDeductible.JFOREIGN
    
            ^EndDate1 = RP.NonDeductible:RP.NonDeductible.JFOREIGN
    
            ^EndDate = RP
            */
            assertion = (report.RPNonDeductibleJFOREIGNExists != false && report.RPNonDeductibleJFOREIGNEndDate != report.RPEndDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438054",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period end date must match Reporting Party period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPNonDeductibleJFOREIGNOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438054" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438054
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438054_25()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438054
            Period end date within context must match end date within reporting party context
    
            Legacy Rule Format:
            IF period.endDate <> RP:period.endDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^EndDate1 <> ^EndDate)
    
            Data Elements:
    
            ^Context = RP:RP.Retirement
    
            ^EndDate1 = RP:RP.Retirement
    
            ^EndDate = RP
            */
            assertion = (report.RPRetirementExists != false && report.RPRetirementEndDate != report.RPEndDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438054",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period end date must match Reporting Party period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPRetirementOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438054" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438054
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438054_26()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438054
            Period end date within context must match end date within reporting party context
    
            Legacy Rule Format:
            IF period.endDate <> RP:period.endDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^EndDate1 <> ^EndDate)
    
            Data Elements:
    
            ^Context = RP:RP.Superannuation
    
            ^EndDate1 = RP:RP.Superannuation
    
            ^EndDate = RP
            */
            assertion = (report.RPSuperannuationExists != false && report.RPSuperannuationEndDate != report.RPEndDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438054",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period end date must match Reporting Party period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPSuperannuationOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438054" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438054
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438054_27()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438054
            Period end date within context must match end date within reporting party context
    
            Legacy Rule Format:
            IF period.endDate <> RP:period.endDate
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^Context <> NULL) AND (^EndDate1 <> ^EndDate)
    
            Data Elements:
    
            ^Context = RP:RP.TOFA
    
            ^EndDate1 = RP:RP.TOFA
    
            ^EndDate = RP
            */
            assertion = (report.RPTOFAExists != false && report.RPTOFAEndDate != report.RPEndDate);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438054",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Period end date must match Reporting Party period end date",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPTOFAOccurrenceIndex) + "/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438054" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438058
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438058()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438058
            Invalid context. The number of Reporting party: Investor contexts must not exceed 1.
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.Investor)) > 1 
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.Investor
            */
            assertion = (Count(report.RPInvestorCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438058",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Reporting party: Investor contexts must not exceed 1.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPInvestorLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438058" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438059
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438059()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438059
            Invalid context. The number of Reporting party: Limited Partners contexts must not exceed 1.
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.LimitedPartners)) > 1 
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP:RP.LimitedPartners
            */
            assertion = (Count(report.RPLimitedPartnersCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438059",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Reporting party: Limited Partners contexts must not exceed 1.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPLimitedPartnersLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438059" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438063
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438063()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438063
            If an Intermediary is present, Intermediary declaration statement accepted indicator must be supplied and must be “True”.
    
            Legacy Rule Format:
            IF INT<>NULL AND INT:pyin.xx.xx:Declaration.StatementAccepted.Indicator <> TRUE
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^INT <> NULL AND ^SMSFAR271 <> TRUE
    
            Data Elements:
    
            ^SMSFAR271 = INT:Declaration:Declaration.StatementAccepted.Indicator WHERE [Declaration.StatementType.Code='TrueAndCorrect']
    
            ^INT = INT
            */
            assertion = (report.INTExists != false && report.SMSFAR271 != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438063",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Declaration statement accepted indicator is required.",
                    LongDescription = @"If an Intermediary is present, Intermediary declaration statement accepted indicator must be supplied and must be ""True"".",
                    Location = "/xbrli:xbrl/tns:Declaration" + OccurrenceIndex(report.DeclarationCollection_TrueAndCorrect_INTOccurrenceIndex) + "/tns:Declaration.StatementAccepted.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438063" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR271", Value = GetValueOrEmpty(report.SMSFAR271) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438064
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438064()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438064
            If an Intermediary is not present, Reporting Party declaration statement accepted indicator must be supplied and must be “True”.
    
            Legacy Rule Format:
            IF INT=NULL AND RP:pyin.xx.xx:Declaration.StatementAccepted.Indicator <> TRUE
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^INT = NULL AND ^SMSFAR268 <> TRUE
    
            Data Elements:
    
            ^SMSFAR268 = RP:Declaration:Declaration.StatementAccepted.Indicator WHERE [Declaration.StatementType.Code='TrueAndCorrect']
    
            ^INT = INT
            */
            assertion = (report.INTExists == false && report.SMSFAR268 != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438064",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Declaration statement accepted indicator is required.",
                    LongDescription = @"If an Intermediary is not present, Reporting Party declaration statement accepted indicator must be supplied and must be ""True"".",
                    Location = "/xbrli:xbrl/tns:Declaration" + OccurrenceIndex(report.DeclarationCollection_TrueAndCorrect_RPOccurrenceIndex) + "/tns:Declaration.StatementAccepted.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438064" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR268", Value = GetValueOrEmpty(report.SMSFAR268) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438081
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438081()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438081
            Invalid context. The number of Reporting party:Investor:YPrevious contexts must not exceed 1.
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.Investor.YPrevious)) > 1 
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP.Investor:RP.Investor.YPrevious
            */
            assertion = (Count(report.RPInvestorYPreviousCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438081",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Reporting party:Investor:YPrevious contexts must not exceed 1.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPInvestorYPreviousLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438081" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.438082
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438082()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.438082
            Invalid context. The number of Reporting party:Limited Partners:YPrevious contexts must not exceed 1.
    
            Legacy Rule Format:
            IF COUNT(CONTEXT(RP.LimitedPartners.YPrevious)) > 1 
            RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^Context) > 1
    
            Data Elements:
    
            ^Context = RP.LimitedPartners:RP.LimitedPartners.YPrevious
            */
            assertion = (Count(report.RPLimitedPartnersYPreviousCount) > 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438082",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid context. The number of Reporting party:Limited Partners:YPrevious contexts must not exceed 1.",
                    Location = "/xbrli:xbrl/xbrli:context" + OccurrenceIndex(report.RPLimitedPartnersYPreviousLastOccurrenceIndex),
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438082" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500130
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500130()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500130
            Person unstructured name must be provided.
    
            Legacy Rule Format:
            WHERE IN TUPLE(personunstructuredname1.xx.xx:PersonUnstructuredName)
            IF pyde.xx.xx:PersonUnstructuredName.FullName.Text = BLANK
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PersonUnstructuredName <> NULL) AND (^SMSFAR280 = BLANK)
    
            Data Elements:
    
            ^SMSFAR280 = INT:Declaration:PersonUnstructuredName:PersonUnstructuredName.FullName.Text WHERE [PersonUnstructuredName.Usage.Code='DeclarationSignatory']
    
            ^PersonUnstructuredName = INT:Declaration:PersonUnstructuredName
            */
            assertion = (report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTExists != false && string.IsNullOrWhiteSpace(report.SMSFAR280) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500130",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Person unstructured name must be provided.",
                    Location = "/xbrli:xbrl/tns:Declaration" + OccurrenceIndex(report.DeclarationCollection_TrueAndCorrect_INTOccurrenceIndex) + "/tns:PersonUnstructuredName" + OccurrenceIndex(report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_INTOccurrenceIndex) + "/tns:PersonUnstructuredName.FullName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500130" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR280", Value = report.SMSFAR280 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500130
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500130_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500130
            Person unstructured name must be provided.
    
            Legacy Rule Format:
            WHERE IN TUPLE(personunstructuredname1.xx.xx:PersonUnstructuredName)
            IF pyde.xx.xx:PersonUnstructuredName.FullName.Text = BLANK
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PersonUnstructuredName <> NULL) AND (^SMSFAR278 = BLANK)
    
            Data Elements:
    
            ^SMSFAR278 = RP:Declaration:PersonUnstructuredName:PersonUnstructuredName.FullName.Text WHERE [PersonUnstructuredName.Usage.Code='DeclarationSignatory']
    
            ^PersonUnstructuredName = RP:Declaration:PersonUnstructuredName
            */
            assertion = (report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPExists != false && string.IsNullOrWhiteSpace(report.SMSFAR278) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500130",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Person unstructured name must be provided.",
                    Location = "/xbrli:xbrl/tns:Declaration" + OccurrenceIndex(report.DeclarationCollection_TrueAndCorrect_RPOccurrenceIndex) + "/tns:PersonUnstructuredName" + OccurrenceIndex(report.Declaration_PersonUnstructuredNameCollection_DeclarationSignatory_RPOccurrenceIndex) + "/tns:PersonUnstructuredName.FullName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500130" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR278", Value = report.SMSFAR278 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500132
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500132()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500132
            Organisation name must be present
    
            Legacy Rule Format:
            WHERE IN TUPLE(organisationname2.xx.xx:OrganisationNameDetails)
            IF pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text = BLANK
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^OrganisationNameDetails <> NULL) AND (^SMSFAR182 = BLANK)
    
            Data Elements:
    
            ^SMSFAR182 = INT:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
    
            ^OrganisationNameDetails = INT:OrganisationNameDetails
            */
            assertion = (report.OrganisationNameDetailsCollection_MN_C_INTExists != false && string.IsNullOrWhiteSpace(report.SMSFAR182) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500132",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Organisation name must be present.",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_MN_C_INTOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500132" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR182", Value = report.SMSFAR182 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500132
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500132_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500132
            Organisation name must be present
    
            Legacy Rule Format:
            WHERE IN TUPLE(organisationname2.xx.xx:OrganisationNameDetails)
            IF pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text = BLANK
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^OrganisationNameDetails <> NULL) AND (^SMSFAR174 = BLANK)
    
            Data Elements:
    
            ^SMSFAR174 = RP:TrusteeOrDirector:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
    
            ^OrganisationNameDetails = RP:TrusteeOrDirector:OrganisationNameDetails
            */
            assertion = (report.TrusteeOrDirector_OrganisationNameDetailsCollection_MN_C_RPExists != false && string.IsNullOrWhiteSpace(report.SMSFAR174) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500132",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Organisation name must be present.",
                    Location = "/xbrli:xbrl/tns:TrusteeOrDirector/tns:OrganisationNameDetails" + OccurrenceIndex(report.TrusteeOrDirector_OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500132" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR174", Value = report.SMSFAR174 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500132
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500132_2()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500132
            Organisation name must be present
    
            Legacy Rule Format:
            WHERE IN TUPLE(organisationname2.xx.xx:OrganisationNameDetails)
            IF pyde.xx.xx:OrganisationNameDetails.OrganisationalName.Text = BLANK
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^OrganisationNameDetails <> NULL) AND (^SMSFAR4 = BLANK)
    
            Data Elements:
    
            ^SMSFAR4 = RP:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
    
            ^OrganisationNameDetails = RP:OrganisationNameDetails
            */
            assertion = (report.OrganisationNameDetailsCollection_MN_C_RPExists != false && string.IsNullOrWhiteSpace(report.SMSFAR4) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500132",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Organisation name must be present.",
                    Location = "/xbrli:xbrl/tns:OrganisationNameDetails" + OccurrenceIndex(report.OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500132" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR4", Value = report.SMSFAR4 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500133
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500133()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500133
            Address Line 1 must be present
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF pyde.xx.xx:AddressDetails.Line1.Text = BLANK
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^AddressDetails <> NULL) AND (^SMSFAR6 = BLANK)
    
            Data Elements:
    
            ^SMSFAR6 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
    
            ^AddressDetails = RP:AddressDetails
            */
            assertion = (report.AddressDetailsCollection_POS_C_RPExists != false && string.IsNullOrWhiteSpace(report.SMSFAR6) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500133",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 1 must be present.",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500133" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR6", Value = report.SMSFAR6 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500133
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500133_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500133
            Address Line 1 must be present
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF pyde.xx.xx:AddressDetails.Line1.Text = BLANK
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^AddressDetails <> NULL) AND (^SMSFAR19 = BLANK)
    
            Data Elements:
    
            ^SMSFAR19 = Auditor:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
    
            ^AddressDetails = Auditor:AddressDetails
            */
            assertion = (report.AddressDetailsCollection_POS_C_AuditorExists != false && string.IsNullOrWhiteSpace(report.SMSFAR19) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500133",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Address Line 1 must be present.",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_AuditorOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500133" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR19", Value = report.SMSFAR19 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500134
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500134()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500134
            Suburb/Town/City must be present in an address
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF pyde.xx.xx:AddressDetails.LocalityName.Text = BLANK
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^AddressDetails <> NULL) AND (^SMSFAR7 = BLANK)
    
            Data Elements:
    
            ^SMSFAR7 = RP:AddressDetails:AddressDetails.LocalityName.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
    
            ^AddressDetails = RP:AddressDetails
            */
            assertion = (report.AddressDetailsCollection_POS_C_RPExists != false && string.IsNullOrWhiteSpace(report.SMSFAR7) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500134",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City must be present in an address.",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.LocalityName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500134" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR7", Value = report.SMSFAR7 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500134
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500134_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500134
            Suburb/Town/City must be present in an address
    
            Legacy Rule Format:
            WHERE IN TUPLE(addressdetails2.xx.xx:AddressDetails)
            IF pyde.xx.xx:AddressDetails.LocalityName.Text = BLANK
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^AddressDetails <> NULL) AND (^SMSFAR20 = BLANK)
    
            Data Elements:
    
            ^SMSFAR20 = Auditor:AddressDetails:AddressDetails.LocalityName.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
    
            ^AddressDetails = Auditor:AddressDetails
            */
            assertion = (report.AddressDetailsCollection_POS_C_AuditorExists != false && string.IsNullOrWhiteSpace(report.SMSFAR20) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500134",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Suburb/Town/City must be present in an address.",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_AuditorOccurrenceIndex) + "/tns:AddressDetails.LocalityName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500134" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR20", Value = report.SMSFAR20 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500135
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500135()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500135
            Family Name must be present
    
            Legacy Rule Format:
            WHERE IN TUPLE(personstructuredname3.xx.xx:PersonNameDetails)
            IF pyde.xx.xx:PersonNameDetails.FamilyName.Text = BLANK
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PersonNameDetails <> NULL) AND (^SMSFAR178 = BLANK)
    
            Data Elements:
    
            ^SMSFAR178 = INT:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
    
            ^PersonNameDetails = INT:PersonNameDetails
            */
            assertion = (report.PersonNameDetailsCollection_LGL_Contact_C_INTExists != false && string.IsNullOrWhiteSpace(report.SMSFAR178) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500135",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family Name must be present.",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_Contact_C_INTOccurrenceIndex) + "/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500135" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR178", Value = report.SMSFAR178 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500135
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500135_1(SMSFAR2026.MemberInformation memberInformation, int itemIndex1)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500135
            Family Name must be present

            Legacy Rule Format:
            WHERE IN TUPLE(personstructuredname3.xx.xx:PersonNameDetails)
            IF pyde.xx.xx:PersonNameDetails.FamilyName.Text = BLANK
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PersonNameDetails <> NULL) AND (^SMSFAR197 = BLANK)

            Data Elements:

            ^SMSFAR197 = RP:MemberInformation:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Identifier'].[PersonNameDetails.Currency.Code='C']

            ^PersonNameDetails = RP:MemberInformation:PersonNameDetails
            */
            assertion = (memberInformation.MemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RPExists != false && string.IsNullOrWhiteSpace(memberInformation.SMSFAR197) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500135",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family Name must be present.",
                    Location = "/xbrli:xbrl/tns:MemberInformation" + OccurrenceIndex(memberInformation.OccurrenceIndex) + "/tns:PersonNameDetails" + OccurrenceIndex(memberInformation.MemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RPOccurrenceIndex) + "/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500135" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR197", Value = memberInformation.SMSFAR197 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500135
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500135_2()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500135
            Family Name must be present
    
            Legacy Rule Format:
            WHERE IN TUPLE(personstructuredname3.xx.xx:PersonNameDetails)
            IF pyde.xx.xx:PersonNameDetails.FamilyName.Text = BLANK
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PersonNameDetails <> NULL) AND (^SMSFAR167 = BLANK)
    
            Data Elements:
    
            ^SMSFAR167 = RP:TrusteeOrDirector:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
    
            ^PersonNameDetails = RP:TrusteeOrDirector:PersonNameDetails
            */
            assertion = (report.TrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RPExists != false && string.IsNullOrWhiteSpace(report.SMSFAR167) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500135",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family Name must be present.",
                    Location = "/xbrli:xbrl/tns:TrusteeOrDirector/tns:PersonNameDetails" + OccurrenceIndex(report.TrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RPOccurrenceIndex) + "/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500135" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR167", Value = report.SMSFAR167 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500135
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500135_3()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500135
            Family Name must be present
    
            Legacy Rule Format:
            WHERE IN TUPLE(personstructuredname3.xx.xx:PersonNameDetails)
            IF pyde.xx.xx:PersonNameDetails.FamilyName.Text = BLANK
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PersonNameDetails <> NULL) AND (^SMSFAR11 = BLANK)
    
            Data Elements:
    
            ^SMSFAR11 = Auditor:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
    
            ^PersonNameDetails = Auditor:PersonNameDetails
            */
            assertion = (report.PersonNameDetailsCollection_LGL_Contact_C_AuditorExists != false && string.IsNullOrWhiteSpace(report.SMSFAR11) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500135",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family Name must be present.",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_Contact_C_AuditorOccurrenceIndex) + "/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500135" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR11", Value = report.SMSFAR11 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500136
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500136()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500136
            BSB Number must be greater than 012000
    
            Legacy Rule Format:
            WHERE IN TUPLE(financialinstitutionaccount1.xx.xx:FinancialInstitutionAccount) 
            IF pyid.xx.xx:FinancialInstitutionAccount.BankStateBranch.Number <> SET(012001- 999999)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^FinancialInstitutionAccount <> NULL) AND (^SMSFAR293 <> BLANK AND (IsNumeric(^SMSFAR293) = 'FALSE' OR AsNumeric(^SMSFAR293) < 12001 OR AsNumeric(^SMSFAR293) > 999999 OR Length(^SMSFAR293) <> 6))
    
            Data Elements:
    
            ^SMSFAR293 = RP.IncomeTaxRefund:FinancialInstitutionAccount:FinancialInstitutionAccount.BankStateBranch.Number
    
            ^FinancialInstitutionAccount = RP.IncomeTaxRefund:FinancialInstitutionAccount
            */
            assertion = (report.RpIncomeTaxRefundFinancialInstitutionAccountCollectionExists != false && (string.IsNullOrWhiteSpace(report.SMSFAR293) != true && (IsNumeric(report.SMSFAR293) == false || AsNumeric(report.SMSFAR293) < 12001 || AsNumeric(report.SMSFAR293) > 999999 || Length(report.SMSFAR293) != 6)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500136",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"BSB Number must be greater than 012000.",
                    Location = "/xbrli:xbrl/tns:FinancialInstitutionAccount/tns:FinancialInstitutionAccount.BankStateBranch.Number",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500136" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR293", Value = report.SMSFAR293 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500136
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500136_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500136
            BSB Number must be greater than 012000
    
            Legacy Rule Format:
            WHERE IN TUPLE(financialinstitutionaccount1.xx.xx:FinancialInstitutionAccount) 
            IF pyid.xx.xx:FinancialInstitutionAccount.BankStateBranch.Number <> SET(012001- 999999)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^FinancialInstitutionAccount <> NULL) AND (^SMSFAR364 <> BLANK AND (IsNumeric(^SMSFAR364) = 'FALSE' OR AsNumeric(^SMSFAR364) < 12001 OR AsNumeric(^SMSFAR364) > 999999 OR Length(^SMSFAR364) <> 6))
    
            Data Elements:
    
            ^SMSFAR364 = RP.Superannuation:FinancialInstitutionAccount:FinancialInstitutionAccount.BankStateBranch.Number
    
            ^FinancialInstitutionAccount = RP.Superannuation:FinancialInstitutionAccount
            */
            assertion = (report.RpSuperannuationFinancialInstitutionAccountCollectionExists != false && (string.IsNullOrWhiteSpace(report.SMSFAR364) != true && (IsNumeric(report.SMSFAR364) == false || AsNumeric(report.SMSFAR364) < 12001 || AsNumeric(report.SMSFAR364) > 999999 || Length(report.SMSFAR364) != 6)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500136",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"BSB Number must be greater than 012000.",
                    Location = "/xbrli:xbrl/tns:FinancialInstitutionAccount/tns:FinancialInstitutionAccount.BankStateBranch.Number",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500136" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR364", Value = report.SMSFAR364 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500137
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500137()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500137
            Account Number and Account Name must be present.
    
            Legacy Rule Format:
            WHERE IN TUPLE(financialinstitutionaccount1.xx.xx:FinancialInstitutionAccount)
            IF pyid.xx.xx:FinancialInstitutionAccount.FinancialInstitutionAccount.Number = BLANK OR pyid.xx.xx:FinancialInstitutionAccount.FinancialInstitutionAccountName.Text = BLANK
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^FinancialInstitutionAccount <> NULL) AND (^SMSFAR294 = BLANK OR ^SMSFAR295 = BLANK)
    
            Data Elements:
    
            ^SMSFAR294 = RP.IncomeTaxRefund:FinancialInstitutionAccount:FinancialInstitutionAccount.FinancialInstitutionAccount.Number
    
            ^SMSFAR295 = RP.IncomeTaxRefund:FinancialInstitutionAccount:FinancialInstitutionAccount.FinancialInstitutionAccountName.Text
    
            ^FinancialInstitutionAccount = RP.IncomeTaxRefund:FinancialInstitutionAccount
            */
            assertion = (report.RpIncomeTaxRefundFinancialInstitutionAccountCollectionExists != false && (string.IsNullOrWhiteSpace(report.SMSFAR294) == true || string.IsNullOrWhiteSpace(report.SMSFAR295) == true));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500137",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Account Number and Account Name must be present.",
                    Location = "/xbrli:xbrl/tns:FinancialInstitutionAccount/tns:FinancialInstitutionAccount.FinancialInstitutionAccount.Number",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500137" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR294", Value = report.SMSFAR294 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR295", Value = report.SMSFAR295 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500137
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500137_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500137
            Account Number and Account Name must be present.
    
            Legacy Rule Format:
            WHERE IN TUPLE(financialinstitutionaccount1.xx.xx:FinancialInstitutionAccount)
            IF pyid.xx.xx:FinancialInstitutionAccount.FinancialInstitutionAccount.Number = BLANK OR pyid.xx.xx:FinancialInstitutionAccount.FinancialInstitutionAccountName.Text = BLANK
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^FinancialInstitutionAccount <> NULL) AND (^SMSFAR365 = BLANK OR ^SMSFAR366 = BLANK)
    
            Data Elements:
    
            ^SMSFAR365 = RP.Superannuation:FinancialInstitutionAccount:FinancialInstitutionAccount.FinancialInstitutionAccount.Number
    
            ^SMSFAR366 = RP.Superannuation:FinancialInstitutionAccount:FinancialInstitutionAccount.FinancialInstitutionAccountName.Text
    
            ^FinancialInstitutionAccount = RP.Superannuation:FinancialInstitutionAccount
            */
            assertion = (report.RpSuperannuationFinancialInstitutionAccountCollectionExists != false && (string.IsNullOrWhiteSpace(report.SMSFAR365) == true || string.IsNullOrWhiteSpace(report.SMSFAR366) == true));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500137",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Account Number and Account Name must be present.",
                    Location = "/xbrli:xbrl/tns:FinancialInstitutionAccount/tns:FinancialInstitutionAccount.FinancialInstitutionAccount.Number",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500137" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR365", Value = report.SMSFAR365 });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR366", Value = report.SMSFAR366 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500141
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500141()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500141
            For an Australian address, state or territory must be one of: ACT, NSW, NT, QLD, SA, VIC, WA, TAS
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.StateOrTerritory.Code <> NULLORBLANK)
            AND (pyde.xx.xx:AddressDetails.StateOrTerritory.Code <> SET ("ACT","NSW","NT","QLD","SA","VIC","WA","TAS"))
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^SMSFAR8 <> BLANK) AND (NotInSet(^SMSFAR8, '"ACT","NSW","NT","QLD","SA","VIC","WA","TAS"'))
    
            Data Elements:
    
            ^SMSFAR8 = RP:AddressDetails:AddressDetails.StateOrTerritory.Code WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.SMSFAR8) != true && !(IsMatch(report.SMSFAR8, @"^(ACT|NSW|NT|QLD|SA|VIC|WA|TAS)$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500141",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"For an Australian address, state or territory must be one of: ACT, NSW, NT, QLD, SA, VIC, WA, TAS",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPOccurrenceIndex) + "/tns:AddressDetails.StateOrTerritory.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500141" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR8", Value = report.SMSFAR8 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500141
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500141_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500141
            For an Australian address, state or territory must be one of: ACT, NSW, NT, QLD, SA, VIC, WA, TAS
    
            Legacy Rule Format:
            IF (pyde.xx.xx:AddressDetails.StateOrTerritory.Code <> NULLORBLANK)
            AND (pyde.xx.xx:AddressDetails.StateOrTerritory.Code <> SET ("ACT","NSW","NT","QLD","SA","VIC","WA","TAS"))
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^SMSFAR21 <> BLANK) AND (NotInSet(^SMSFAR21, '"ACT","NSW","NT","QLD","SA","VIC","WA","TAS"'))
    
            Data Elements:
    
            ^SMSFAR21 = Auditor:AddressDetails:AddressDetails.StateOrTerritory.Code WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (string.IsNullOrWhiteSpace(report.SMSFAR21) != true && !(IsMatch(report.SMSFAR21, @"^(ACT|NSW|NT|QLD|SA|VIC|WA|TAS)$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500141",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"For an Australian address, state or territory must be one of: ACT, NSW, NT, QLD, SA, VIC, WA, TAS",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_AuditorOccurrenceIndex) + "/tns:AddressDetails.StateOrTerritory.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500141" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR21", Value = report.SMSFAR21 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500142
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500142()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500142
            Family name must contain at least one alphabetical character
    
            Legacy Rule Format:
            WHERE IN TUPLE(personstructuredname3.xx.xx:PersonNameDetails)
            IF pyde.xx.xx:PersonNameDetails.FamilyName.Text DOES NOT CONTAIN SET("A-Z","a-z")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PersonNameDetails <> NULL) AND (NotContainsSet(^SMSFAR178, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^SMSFAR178 = INT:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
    
            ^PersonNameDetails = INT:PersonNameDetails
            */
            assertion = (report.PersonNameDetailsCollection_LGL_Contact_C_INTExists != false && !(IsMatch(report.SMSFAR178, @"[A-Z]|[a-z]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500142",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name must contain at least one alphabetical character.",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_Contact_C_INTOccurrenceIndex) + "/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500142" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR178", Value = report.SMSFAR178 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500142
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500142_1(SMSFAR2026.MemberInformation memberInformation, int itemIndex1)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500142
            Family name must contain at least one alphabetical character

            Legacy Rule Format:
            WHERE IN TUPLE(personstructuredname3.xx.xx:PersonNameDetails)
            IF pyde.xx.xx:PersonNameDetails.FamilyName.Text DOES NOT CONTAIN SET("A-Z","a-z")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PersonNameDetails <> NULL) AND (NotContainsSet(^SMSFAR197, '"A-Z","a-z"'))

            Data Elements:

            ^SMSFAR197 = RP:MemberInformation:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Identifier'].[PersonNameDetails.Currency.Code='C']

            ^PersonNameDetails = RP:MemberInformation:PersonNameDetails
            */
            assertion = (memberInformation.MemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RPExists != false && !(IsMatch(memberInformation.SMSFAR197, @"[A-Z]|[a-z]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500142",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name must contain at least one alphabetical character.",
                    Location = "/xbrli:xbrl/tns:MemberInformation" + OccurrenceIndex(memberInformation.OccurrenceIndex) + "/tns:PersonNameDetails" + OccurrenceIndex(memberInformation.MemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RPOccurrenceIndex) + "/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500142" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR197", Value = memberInformation.SMSFAR197 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500142
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500142_2()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500142
            Family name must contain at least one alphabetical character
    
            Legacy Rule Format:
            WHERE IN TUPLE(personstructuredname3.xx.xx:PersonNameDetails)
            IF pyde.xx.xx:PersonNameDetails.FamilyName.Text DOES NOT CONTAIN SET("A-Z","a-z")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PersonNameDetails <> NULL) AND (NotContainsSet(^SMSFAR167, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^SMSFAR167 = RP:TrusteeOrDirector:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
    
            ^PersonNameDetails = RP:TrusteeOrDirector:PersonNameDetails
            */
            assertion = (report.TrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RPExists != false && !(IsMatch(report.SMSFAR167, @"[A-Z]|[a-z]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500142",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name must contain at least one alphabetical character.",
                    Location = "/xbrli:xbrl/tns:TrusteeOrDirector/tns:PersonNameDetails" + OccurrenceIndex(report.TrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RPOccurrenceIndex) + "/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500142" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR167", Value = report.SMSFAR167 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500142
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500142_3()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500142
            Family name must contain at least one alphabetical character
    
            Legacy Rule Format:
            WHERE IN TUPLE(personstructuredname3.xx.xx:PersonNameDetails)
            IF pyde.xx.xx:PersonNameDetails.FamilyName.Text DOES NOT CONTAIN SET("A-Z","a-z")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^PersonNameDetails <> NULL) AND (NotContainsSet(^SMSFAR11, '"A-Z","a-z"'))
    
            Data Elements:
    
            ^SMSFAR11 = Auditor:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
    
            ^PersonNameDetails = Auditor:PersonNameDetails
            */
            assertion = (report.PersonNameDetailsCollection_LGL_Contact_C_AuditorExists != false && !(IsMatch(report.SMSFAR11, @"[A-Z]|[a-z]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500142",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name must contain at least one alphabetical character.",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_Contact_C_AuditorOccurrenceIndex) + "/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500142" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR11", Value = report.SMSFAR11 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500145
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500145()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500145
            Account number must contain the following: A to Z a to z or 0 to 9
    
            Legacy Rule Format:
            IF pyid.xx.xx:FinancialInstitutionAccount.FinancialInstitutionAccount.Number <> NULLORBLANK AND ANY CHARACTER OF pyid.xx.xx:FinancialInstitutionAccount.FinancialInstitutionAccount.Number <> SET("a-z","A-Z","0-9")
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^SMSFAR294 <> BLANK AND NotCharacterInSet(^SMSFAR294, '"a-z","A-Z","0-9"')
    
            Data Elements:
    
            ^SMSFAR294 = RP.IncomeTaxRefund:FinancialInstitutionAccount:FinancialInstitutionAccount.FinancialInstitutionAccount.Number
            */
            assertion = (string.IsNullOrWhiteSpace(report.SMSFAR294) != true && !(IsMatch(report.SMSFAR294, @"^[a-zA-Z0-9]*$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500145",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Account number contains invalid text",
                    LongDescription = @"Account number must contain the following: A to Z a to z or 0 to 9",
                    Location = "/xbrli:xbrl/tns:FinancialInstitutionAccount/tns:FinancialInstitutionAccount.FinancialInstitutionAccount.Number",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500145" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR294", Value = report.SMSFAR294 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500145
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500145_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500145
            Account number must contain the following: A to Z a to z or 0 to 9
    
            Legacy Rule Format:
            IF pyid.xx.xx:FinancialInstitutionAccount.FinancialInstitutionAccount.Number <> NULLORBLANK AND ANY CHARACTER OF pyid.xx.xx:FinancialInstitutionAccount.FinancialInstitutionAccount.Number <> SET("a-z","A-Z","0-9")
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^SMSFAR365 <> BLANK AND NotCharacterInSet(^SMSFAR365, '"a-z","A-Z","0-9"')
    
            Data Elements:
    
            ^SMSFAR365 = RP.Superannuation:FinancialInstitutionAccount:FinancialInstitutionAccount.FinancialInstitutionAccount.Number
            */
            assertion = (string.IsNullOrWhiteSpace(report.SMSFAR365) != true && !(IsMatch(report.SMSFAR365, @"^[a-zA-Z0-9]*$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500145",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Account number contains invalid text",
                    LongDescription = @"Account number must contain the following: A to Z a to z or 0 to 9",
                    Location = "/xbrli:xbrl/tns:FinancialInstitutionAccount/tns:FinancialInstitutionAccount.FinancialInstitutionAccount.Number",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500145" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR365", Value = report.SMSFAR365 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500151
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500151()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500151
            First name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text = FOUND("ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^SMSFAR180, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
    
            Data Elements:
    
            ^SMSFAR180 = INT:PersonNameDetails:PersonNameDetails.GivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.SMSFAR180, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010436",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_Contact_C_INTOccurrenceIndex) + "/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500151" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR180", Value = report.SMSFAR180 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500151
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500151_1(SMSFAR2026.MemberInformation memberInformation, int itemIndex1)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500151
            First name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR

            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text = FOUND("ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^SMSFAR199, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))

            Data Elements:

            ^SMSFAR199 = RP:MemberInformation:PersonNameDetails:PersonNameDetails.GivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Identifier'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(memberInformation.SMSFAR199, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010436",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/xbrli:xbrl/tns:MemberInformation" + OccurrenceIndex(memberInformation.OccurrenceIndex) + "/tns:PersonNameDetails" + OccurrenceIndex(memberInformation.MemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RPOccurrenceIndex) + "/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500151" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR199", Value = GetValueOrEmpty(memberInformation.SMSFAR199) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500151
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500151_2()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500151
            First name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text = FOUND("ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^SMSFAR169, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
    
            Data Elements:
    
            ^SMSFAR169 = RP:TrusteeOrDirector:PersonNameDetails:PersonNameDetails.GivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.SMSFAR169, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010436",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/xbrli:xbrl/tns:TrusteeOrDirector/tns:PersonNameDetails" + OccurrenceIndex(report.TrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RPOccurrenceIndex) + "/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500151" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR169", Value = report.SMSFAR169 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500151
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500151_3()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500151
            First name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.GivenName.Text = FOUND("ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^SMSFAR13, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
    
            Data Elements:
    
            ^SMSFAR13 = Auditor:PersonNameDetails:PersonNameDetails.GivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.SMSFAR13, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010436",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"First name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_Contact_C_AuditorOccurrenceIndex) + "/tns:PersonNameDetails.GivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500151" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR13", Value = report.SMSFAR13 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500152
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500152()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500152
            Other given name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text = FOUND("ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^SMSFAR181, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
    
            Data Elements:
    
            ^SMSFAR181 = INT:PersonNameDetails:PersonNameDetails.OtherGivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.SMSFAR181, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010448",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_Contact_C_INTOccurrenceIndex) + "/tns:PersonNameDetails.OtherGivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500152" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR181", Value = GetValueOrEmpty(report.SMSFAR181) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500152
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500152_1(SMSFAR2026.MemberInformation memberInformation, int itemIndex1)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500152
            Other given name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR

            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text = FOUND("ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^SMSFAR200, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))

            Data Elements:

            ^SMSFAR200 = RP:MemberInformation:PersonNameDetails:PersonNameDetails.OtherGivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Identifier'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(memberInformation.SMSFAR200, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010448",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/xbrli:xbrl/tns:MemberInformation" + OccurrenceIndex(memberInformation.OccurrenceIndex) + "/tns:PersonNameDetails" + OccurrenceIndex(memberInformation.MemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RPOccurrenceIndex) + "/tns:PersonNameDetails.OtherGivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500152" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR200", Value = GetValueOrEmpty(memberInformation.SMSFAR200) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500152
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500152_2()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500152
            Other given name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text = FOUND("ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^SMSFAR170, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
    
            Data Elements:
    
            ^SMSFAR170 = RP:TrusteeOrDirector:PersonNameDetails:PersonNameDetails.OtherGivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.SMSFAR170, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010448",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/xbrli:xbrl/tns:TrusteeOrDirector/tns:PersonNameDetails" + OccurrenceIndex(report.TrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RPOccurrenceIndex) + "/tns:PersonNameDetails.OtherGivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500152" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR170", Value = GetValueOrEmpty(report.SMSFAR170) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500152
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500152_3()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500152
            Other given name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.OtherGivenName.Text = FOUND("ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^SMSFAR14, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
    
            Data Elements:
    
            ^SMSFAR14 = Auditor:PersonNameDetails:PersonNameDetails.OtherGivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.SMSFAR14, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010448",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other given name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_Contact_C_AuditorOccurrenceIndex) + "/tns:PersonNameDetails.OtherGivenName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500152" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR14", Value = GetValueOrEmpty(report.SMSFAR14) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500153
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500153()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500153
            Family name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text = FOUND("ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^SMSFAR178, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
    
            Data Elements:
    
            ^SMSFAR178 = INT:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.SMSFAR178, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500153",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_Contact_C_INTOccurrenceIndex) + "/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500153" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR178", Value = report.SMSFAR178 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500153
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500153_1(SMSFAR2026.MemberInformation memberInformation, int itemIndex1)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500153
            Family name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR

            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text = FOUND("ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^SMSFAR197, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))

            Data Elements:

            ^SMSFAR197 = RP:MemberInformation:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Identifier'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(memberInformation.SMSFAR197, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500153",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/xbrli:xbrl/tns:MemberInformation" + OccurrenceIndex(memberInformation.OccurrenceIndex) + "/tns:PersonNameDetails" + OccurrenceIndex(memberInformation.MemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RPOccurrenceIndex) + "/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500153" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR197", Value = memberInformation.SMSFAR197 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500153
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500153_2()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500153
            Family name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text = FOUND("ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^SMSFAR167, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
    
            Data Elements:
    
            ^SMSFAR167 = RP:TrusteeOrDirector:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.SMSFAR167, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500153",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/xbrli:xbrl/tns:TrusteeOrDirector/tns:PersonNameDetails" + OccurrenceIndex(report.TrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RPOccurrenceIndex) + "/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500153" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR167", Value = report.SMSFAR167 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500153
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500153_3()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500153
            Family name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
            Legacy Rule Format:
            IF (pyde.xx.xx:PersonNameDetails.FamilyName.Text = FOUND("ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (FoundSet(^SMSFAR11, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
    
            Data Elements:
    
            ^SMSFAR11 = Auditor:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = IsMatch(report.SMSFAR11, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500153",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_Contact_C_AuditorOccurrenceIndex) + "/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500153" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR11", Value = report.SMSFAR11 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500154
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500154()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500154
            Financial institution account name cannot be greater than 200 characters
    
            Legacy Rule Format:
            IF LENGTH(pyid.xx.xx:FinancialInstitutionAccount.FinancialInstitutionAccountName.Text) > 200
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^SMSFAR295) > 200
    
            Data Elements:
    
            ^SMSFAR295 = RP.IncomeTaxRefund:FinancialInstitutionAccount:FinancialInstitutionAccount.FinancialInstitutionAccountName.Text
            */
            assertion = (Length(report.SMSFAR295) > 200);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500154",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Bank account name must not be longer than 200 characters",
                    Location = "/xbrli:xbrl/tns:FinancialInstitutionAccount/tns:FinancialInstitutionAccount.FinancialInstitutionAccountName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500154" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR295", Value = report.SMSFAR295 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.GEN.500154
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500154_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.GEN.500154
            Financial institution account name cannot be greater than 200 characters
    
            Legacy Rule Format:
            IF LENGTH(pyid.xx.xx:FinancialInstitutionAccount.FinancialInstitutionAccountName.Text) > 200
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Length(^SMSFAR366) > 200
    
            Data Elements:
    
            ^SMSFAR366 = RP.Superannuation:FinancialInstitutionAccount:FinancialInstitutionAccount.FinancialInstitutionAccountName.Text
            */
            assertion = (Length(report.SMSFAR366) > 200);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500154",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Bank account name must not be longer than 200 characters",
                    Location = "/xbrli:xbrl/tns:FinancialInstitutionAccount/tns:FinancialInstitutionAccount.FinancialInstitutionAccountName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.GEN.500154" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR366", Value = report.SMSFAR366 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436012
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436012()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436012
            There must be one instance of the current postal address of the self-managed superannuation fund. This address must be an Australian address submitted with a Currency Code 'C' and a Usage of 'POS'. Line 1, Suburb/Town/City, State/Territory and Postcode must be present
    
            Legacy Rule Format:
            IF COUNT [SMSFAR6] <> 1
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^SMSFAR6) <> 1
    
            Data Elements:
    
            ^SMSFAR6 = RP:AddressDetails:AddressDetails.Line1.Text WHERE [AddressDetails.Usage.Code='POS'].[AddressDetails.Currency.Code='C']
            */
            assertion = (Count(report.AddressDetailsCollection_POS_C_RPCount) != 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.SMSFAR.436012",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Current postal address of the SMSF must be present",
                    LongDescription = @"There must be one instance of the current postal address of the self-managed superannuation fund. This address must be an Australian address submitted with a Currency Code 'C' and a Usage of 'POS'. Line 1, Suburb/Town/City, State/Territory and Postcode must be present",
                    Location = "/xbrli:xbrl/tns:AddressDetails" + OccurrenceIndex(report.AddressDetailsCollection_POS_C_RPLastOccurrenceIndex) + "/tns:AddressDetails.Line1.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436012" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436014
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436014()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436014
            There must be one instance of the current legal Name of the Fund auditor.
    
            Legacy Rule Format:
            IF COUNT([SMSFAR11]) <> 1
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^SMSFAR11) <> 1
    
            Data Elements:
    
            ^SMSFAR11 = Auditor:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Contact'].[PersonNameDetails.Currency.Code='C']
            */
            assertion = (Count(report.PersonNameDetailsCollection_LGL_Contact_C_AuditorCount) != 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.SMSFAR.436014",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Fund auditor's name must be present",
                    LongDescription = @"There must be one instance of the current legal Name of the Fund auditor.",
                    Location = "/xbrli:xbrl/tns:PersonNameDetails" + OccurrenceIndex(report.PersonNameDetailsCollection_LGL_Contact_C_AuditorLastOccurrenceIndex) + "/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436014" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436016
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436016()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436016
            Mandatory field not supplied
    
            Legacy Rule Format:
            IF [SMSFAR28] = NULL
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^SMSFAR28 = NULL
    
            Data Elements:
    
            ^SMSFAR28 = RP:OrganisationDetails.OperationCease.Indicator
            */
            assertion = (report.SMSFAR28 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Mandatory field not supplied",
                    Location = "/xbrli:xbrl/tns:OrganisationDetails.OperationCease.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436016" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR28", Value = GetValueOrEmpty(report.SMSFAR28) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436020
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436020()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436020
            Mandatory field not supplied
    
            Legacy Rule Format:
            IF [SMSFAR23] = NULL
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^SMSFAR23 = NULL
    
            Data Elements:
    
            ^SMSFAR23 = RP:AuditorReport.Signature.Date
            */
            assertion = (report.SMSFAR23 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Mandatory field not supplied",
                    Location = "/xbrli:xbrl/tns:AuditorReport.Signature.Date",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436020" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR23", Value = GetValueOrEmptyDate(report.SMSFAR23) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436021
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436021()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436021
            Audit date must not be prior to year of return or after date of lodgement.
    
            Legacy Rule Format:
            IF [SMSFAR23] <> NULL AND ([SMSFAR23] < (([SMSFAR1] - 1)&"-07-01") OR DaysInDuration([SMSFAR23] - Today()) > 1)
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^SMSFAR23 <> NULL AND (^SMSFAR23 < AsDate(Concat(^SMSFAR1 - 1, '-07-01')) OR DaysInDuration(^SMSFAR23 - Today()) > 1)
    
            Data Elements:
    
            ^SMSFAR23 = RP:AuditorReport.Signature.Date
    
            ^SMSFAR1 = RP:Report.TargetFinancial.Year
            */
            assertion = (report.SMSFAR23 != null && (report.SMSFAR23.GetValueOrDefault() < AsDate(string.Concat(report.SMSFAR1.GetValueOrDefault() - 1, @"-07-01")) || (report.SMSFAR23.GetValueOrDefault() - DateTime.Now.Date).Days > 1));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.SMSFAR.436021",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Audit date incorrect.",
                    LongDescription = @"Audit date must not be prior to year of return or after date of lodgement.",
                    Location = "/xbrli:xbrl/tns:AuditorReport.Signature.Date",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436021" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR23", Value = GetValueOrEmptyDate(report.SMSFAR23) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR1", Value = GetValueOrEmpty(report.SMSFAR1) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436022
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436022()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436022
            Mandatory field not supplied
    
            Legacy Rule Format:
            IF [SMSFAR24] = NULL
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^SMSFAR24 = NULL
    
            Data Elements:
    
            ^SMSFAR24 = RP:AuditorReport.OpinionQualified.Indicator
            */
            assertion = (report.SMSFAR24 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Mandatory field not supplied",
                    Location = "/xbrli:xbrl/tns:AuditorReport.OpinionQualified.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436022" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR24", Value = GetValueOrEmpty(report.SMSFAR24) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436023
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436023()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436023
            Mandatory field not supplied
    
            Legacy Rule Format:
            IF [SMSFAR25] = NULL
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^SMSFAR25 = NULL
    
            Data Elements:
    
            ^SMSFAR25 = RP:SuperannuationFundDetails.AustralianSuperannuationFundStatus.Indicator
            */
            assertion = (report.SMSFAR25 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Mandatory field not supplied",
                    Location = "/xbrli:xbrl/tns:SuperannuationFundDetails.AustralianSuperannuationFundStatus.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436023" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR25", Value = GetValueOrEmpty(report.SMSFAR25) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436025
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436025()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436025
            Mandatory field not supplied
    
            Legacy Rule Format:
            IF [SMSFAR26] = NULL
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^SMSFAR26 = NULL
    
            Data Elements:
    
            ^SMSFAR26 = RP:SuperannuationFundDetails.FundBenefitStructure.Code
            */
            assertion = (report.SMSFAR26 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Mandatory field not supplied",
                    Location = "/xbrli:xbrl/tns:SuperannuationFundDetails.FundBenefitStructure.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436025" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR26", Value = report.SMSFAR26 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436026
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436026()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436026
            Mandatory field not supplied
    
            Legacy Rule Format:
            IF [SMSFAR27] = NULL
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^SMSFAR27 = NULL
    
            Data Elements:
    
            ^SMSFAR27 = RP:SuperannuationFundDetails.FundAcceptGovernmentContributions.Indicator
            */
            assertion = (report.SMSFAR27 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Mandatory field not supplied",
                    Location = "/xbrli:xbrl/tns:SuperannuationFundDetails.FundAcceptGovernmentContributions.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436026" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR27", Value = GetValueOrEmpty(report.SMSFAR27) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436027
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436027()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436027
            Date wound up must be present if the fund wound up during the income year
    
            Legacy Rule Format:
            IF [SMSFAR28] = TRUE AND [SMSFAR29] = NULL  
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^SMSFAR28 = TRUE AND ^SMSFAR29 = NULL
    
            Data Elements:
    
            ^SMSFAR29 = RP:OrganisationDetails.OrganisationActivityStatusEnd.Date
    
            ^SMSFAR28 = RP:OrganisationDetails.OperationCease.Indicator
            */
            assertion = (report.SMSFAR28 == true && report.SMSFAR29 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.434026",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Date wound up must be present if the fund wound up during the income year",
                    Location = "/xbrli:xbrl/tns:OrganisationDetails.OrganisationActivityStatusEnd.Date",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436027" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR28", Value = GetValueOrEmpty(report.SMSFAR28) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR29", Value = GetValueOrEmptyDate(report.SMSFAR29) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436029
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436029()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436029
            Fund wound up date must be within year of assessment.
    
            Legacy Rule Format:
            IF [SMSFAR29] < (([SMSFAR1] - 1)&"-07-01") OR [SMSFAR29] > ([SMSFAR1]&"-06-30")
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^SMSFAR29 <> NULL AND ^SMSFAR29 < AsDate(Concat(^SMSFAR1 - 1, '-07-01')) OR ^SMSFAR29 > AsDate(Concat(^SMSFAR1, '-06-30')))
    
            Data Elements:
    
            ^SMSFAR29 = RP:OrganisationDetails.OrganisationActivityStatusEnd.Date
    
            ^SMSFAR1 = RP:Report.TargetFinancial.Year
            */
            assertion = (report.SMSFAR29 != null && report.SMSFAR29.GetValueOrDefault() < AsDate(string.Concat(report.SMSFAR1.GetValueOrDefault() - 1, @"-07-01")) || report.SMSFAR29.GetValueOrDefault() > AsDate(string.Concat(report.SMSFAR1.GetValueOrDefault(), @"-06-30")));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.434145",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Fund wound up date must be within year of assessment.",
                    LongDescription = @"The date the fund wound up must fall within the year of assessment.  For example, if the year of the return being lodged is 2010, then the date must not be before 1 July 2009 and must not be after 30 June 2010.",
                    Location = "/xbrli:xbrl/tns:OrganisationDetails.OrganisationActivityStatusEnd.Date",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436029" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR29", Value = GetValueOrEmptyDate(report.SMSFAR29) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR1", Value = GetValueOrEmpty(report.SMSFAR1) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436030
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436030()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436030
            An answer to 'Have all lodgment and payment obligations been met?' must be present
    
            Legacy Rule Format:
            IF [SMSFAR28] = TRUE AND [SMSFAR30] = NULL 
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^SMSFAR28 = TRUE AND ^SMSFAR30 = NULL
    
            Data Elements:
    
            ^SMSFAR30 = RP:SuperannuationFundDetails.TaxObligationsMet.Indicator
    
            ^SMSFAR28 = RP:OrganisationDetails.OperationCease.Indicator
            */
            assertion = (report.SMSFAR28 == true && report.SMSFAR30 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.SMSFAR.436030",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"An answer to 'Have all lodgment and payment obligations been met?' must be present",
                    LongDescription = @"If the answer to the question 'Was the fund wound up during the income year' is 'yes' (true), then an answer to the question 'Have all lodgment and payment obligations been met?' must be present",
                    Location = "/xbrli:xbrl/tns:SuperannuationFundDetails.TaxObligationsMet.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436030" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR28", Value = GetValueOrEmpty(report.SMSFAR28) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR30", Value = GetValueOrEmpty(report.SMSFAR30) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436031
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436031()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436031
            If a Net capital gain amount is entered, then the answer to the question 'Did you have a CGT event during the year' must be 'yes' (true)
    
            Legacy Rule Format:
            IF [SMSFAR33] > 0 AND [SMSFAR31] = FALSE  
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^SMSFAR33 > 0 AND ^SMSFAR31 = FALSE
    
            Data Elements:
    
            ^SMSFAR31 = RP:CapitalGainsTax.Event.Indicator
    
            ^SMSFAR33 = RP:Income.CapitalGainsNet.Amount
            */
            assertion = (report.SMSFAR33.GetValueOrDefault() > 0 && report.SMSFAR31 == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.434035",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Answer to CGT event question must be 'yes' (true)",
                    LongDescription = @"If a Net capital gain amount is entered, then the answer to the question 'Did you have a CGT event during the year' must be 'yes' (true)",
                    Location = "/xbrli:xbrl/tns:CapitalGainsTax.Event.Indicator",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436031" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR33", Value = GetValueOrEmpty(report.SMSFAR33) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR31", Value = GetValueOrEmpty(report.SMSFAR31) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436040
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436040()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436040
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [SMSFAR33] <> NULL AND [SMSFAR33] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR33, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR33 = RP:Income.CapitalGainsNet.Amount
            */
            assertion = NotMonetary(report.SMSFAR33, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.CapitalGainsNet.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436040" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR33", Value = GetValueOrEmpty(report.SMSFAR33) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436041
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436041()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436041
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [SMSFAR34] <> NULL AND [SMSFAR34] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR34, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR34 = RP:Income.RentLeasingHiringGross.Amount
            */
            assertion = NotMonetary(report.SMSFAR34, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.RentLeasingHiringGross.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436041" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR34", Value = GetValueOrEmpty(report.SMSFAR34) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436042
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436042()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436042
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [SMSFAR35] <> NULL AND [SMSFAR35] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR35, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR35 = RP:Income.Interest.Gross.Amount
            */
            assertion = NotMonetary(report.SMSFAR35, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.Interest.Gross.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436042" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR35", Value = GetValueOrEmpty(report.SMSFAR35) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436043
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436043()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436043
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [SMSFAR36] <> NULL AND [SMSFAR36] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR36, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR36 = RP:Income.ForestryManagedInvestmentScheme.Amount
            */
            assertion = NotMonetary(report.SMSFAR36, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.ForestryManagedInvestmentScheme.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436043" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR36", Value = GetValueOrEmpty(report.SMSFAR36) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436044
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436044()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436044
            Gross foreign income must be present if Net foreign income is present
    
            Legacy Rule Format:
            IF [SMSFAR38] > 0 AND [SMSFAR37] = NULL  
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^SMSFAR38 > 0 AND ^SMSFAR37 = NULL
    
            Data Elements:
    
            ^SMSFAR37 = RP.JFOREIGN:Income.Gross.Amount
    
            ^SMSFAR38 = RP.JFOREIGN:Income.Net.Amount
            */
            assertion = (report.SMSFAR38.GetValueOrDefault() > 0 && report.SMSFAR37 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.434042",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Gross foreign income must be present if Net foreign income is present",
                    Location = "/xbrli:xbrl/tns:Income.Gross.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436044" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR38", Value = GetValueOrEmpty(report.SMSFAR38) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR37", Value = GetValueOrEmpty(report.SMSFAR37) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436046
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436046()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436046
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [SMSFAR37] <> NULL AND [SMSFAR37] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR37, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR37 = RP.JFOREIGN:Income.Gross.Amount
            */
            assertion = NotMonetary(report.SMSFAR37, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.Gross.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436046" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR37", Value = GetValueOrEmpty(report.SMSFAR37) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436047
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436047()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436047
            Monetary field may be signed, maximum of 11 chars, no decimals
    
            Legacy Rule Format:
            IF [SMSFAR38] <> NULL AND [SMSFAR38] <> MONETARY(S,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR38, 'S', 11, 0)
    
            Data Elements:
    
            ^SMSFAR38 = RP.JFOREIGN:Income.Net.Amount
            */
            assertion = NotMonetary(report.SMSFAR38, @"S", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.Net.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436047" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR38", Value = GetValueOrEmpty(report.SMSFAR38) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436048
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436048()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436048
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [SMSFAR39] <> NULL AND [SMSFAR39] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR39, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR39 = RP:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
            */
            assertion = NotMonetary(report.SMSFAR39, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436048" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR39", Value = GetValueOrEmpty(report.SMSFAR39) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436049
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436049()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436049
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [SMSFAR40] <> NULL AND [SMSFAR40] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR40, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR40 = RP:Equity.InternationalDealings.ForeignFundTransfers.Amount
            */
            assertion = NotMonetary(report.SMSFAR40, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Equity.InternationalDealings.ForeignFundTransfers.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436049" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR40", Value = GetValueOrEmpty(report.SMSFAR40) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436050
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436050()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436050
            Number of transfers must be an integer between 0 and 999 (inclusive)
    
            Legacy Rule Format:
            IF [SMSFAR41] <> NULL AND [SMSFAR41] <> SET(0-999)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^SMSFAR41 <> NULL AND (^SMSFAR41 < 0 OR ^SMSFAR41 > 999)
    
            Data Elements:
    
            ^SMSFAR41 = RP:InternationalDealings.ForeignFundTransfers.Count
            */
            assertion = (report.SMSFAR41 != null && (report.SMSFAR41.GetValueOrDefault() < 0 || report.SMSFAR41.GetValueOrDefault() > 999));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.SMSFAR.436050",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Number of transfers must be an integer between 0 and 999 (inclusive)",
                    Location = "/xbrli:xbrl/tns:InternationalDealings.ForeignFundTransfers.Count",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436050" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR41", Value = report.SMSFAR41.GetValueOrDefault().ToString() });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436054
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436054()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436054
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [SMSFAR42] <> NULL AND [SMSFAR42] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR42, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR42 = RP:Remuneration.ABNNotQuotedPaymentGross.Amount
            */
            assertion = NotMonetary(report.SMSFAR42, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Remuneration.ABNNotQuotedPaymentGross.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436054" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR42", Value = GetValueOrEmpty(report.SMSFAR42) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436055
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436055()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436055
            Monetary field may be signed, maximum of 11 chars, no decimals
    
            Legacy Rule Format:
            IF [SMSFAR43] <> NULL AND [SMSFAR43] <> MONETARY(S,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR43, 'S', 11, 0)
    
            Data Elements:
    
            ^SMSFAR43 = RP:Income.PartnershipDistributionGross.Amount
            */
            assertion = NotMonetary(report.SMSFAR43, @"S", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.PartnershipDistributionGross.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436055" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR43", Value = GetValueOrEmpty(report.SMSFAR43) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436056
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436056()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436056
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [SMSFAR44] <> NULL AND [SMSFAR44] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR44, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR44 = RP:Income.DividendsUnfranked.Amount
            */
            assertion = NotMonetary(report.SMSFAR44, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.DividendsUnfranked.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436056" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR44", Value = GetValueOrEmpty(report.SMSFAR44) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436057
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436057()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436057
            Franking credit cannot be greater than 48% of dividends-franked amount.
    
            Legacy Rule Format:
            IF [SMSFAR46] > 0 AND (([SMSFAR45] * 0.48) < ([SMSFAR46] - 1)) 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^SMSFAR46 > 0 AND ((^SMSFAR45 * 0.48) < (^SMSFAR46 - 1))
    
            Data Elements:
    
            ^SMSFAR45 = RP:Income.DividendsFranked.Amount
    
            ^SMSFAR46 = RP:IncomeTax.FrankingCredits.Amount
            */
            assertion = (report.SMSFAR46.GetValueOrDefault() > 0 && report.SMSFAR45.GetValueOrDefault() * 0.48M < report.SMSFAR46.GetValueOrDefault() - 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430098",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Franking credit cannot be greater than 48% of dividends-franked amount.",
                    LongDescription = @"Franking credit must not exceed 48% of Dividends - Franked amount",
                    Location = "/xbrli:xbrl/tns:Income.DividendsFranked.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436057" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR46", Value = GetValueOrEmpty(report.SMSFAR46) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR45", Value = GetValueOrEmpty(report.SMSFAR45) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436058
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436058()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436058
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [SMSFAR45] <> NULL AND [SMSFAR45] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR45, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR45 = RP:Income.DividendsFranked.Amount
            */
            assertion = NotMonetary(report.SMSFAR45, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.DividendsFranked.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436058" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR45", Value = GetValueOrEmpty(report.SMSFAR45) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436059
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436059()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436059
            If Dividends - Franked amount is present, then Franking credit must be present
    
            Legacy Rule Format:
            IF [SMSFAR45] > 0 AND [SMSFAR46] = NULL 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^SMSFAR45 > 0 AND ^SMSFAR46 = NULL
    
            Data Elements:
    
            ^SMSFAR46 = RP:IncomeTax.FrankingCredits.Amount
    
            ^SMSFAR45 = RP:Income.DividendsFranked.Amount
            */
            assertion = (report.SMSFAR45.GetValueOrDefault() > 0 && report.SMSFAR46 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430100",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Franking credit amount must be present",
                    LongDescription = @"If Dividends - Franked amount is present, then Franking credit must be present",
                    Location = "/xbrli:xbrl/tns:IncomeTax.FrankingCredits.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436059" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR45", Value = GetValueOrEmpty(report.SMSFAR45) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR46", Value = GetValueOrEmpty(report.SMSFAR46) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436060
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436060()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436060
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [SMSFAR46] <> NULL AND [SMSFAR46] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR46, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR46 = RP:IncomeTax.FrankingCredits.Amount
            */
            assertion = NotMonetary(report.SMSFAR46, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:IncomeTax.FrankingCredits.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436060" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR46", Value = GetValueOrEmpty(report.SMSFAR46) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436061
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436061()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436061
            If 'Gross distribution from trusts - action code' is present, then a 'Gross distribution from trusts' amount must be present
    
            Legacy Rule Format:
            IF [SMSFAR48] <> NULLORBLANK AND ([SMSFAR47] = 0 OR [SMSFAR47] = NULL) 
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^SMSFAR48 <> BLANK AND (^SMSFAR47 = 0 OR ^SMSFAR47 = NULL)
    
            Data Elements:
    
            ^SMSFAR47 = RP:Income.TrustDistributionGross.Amount
    
            ^SMSFAR48 = RP:Party.TrustType.Code
            */
            assertion = (string.IsNullOrWhiteSpace(report.SMSFAR48) != true && (report.SMSFAR47 == 0 || report.SMSFAR47 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.SMSFAR.436061",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Gross distribution from trusts' amount must be present",
                    LongDescription = @"If 'Gross distribution from trusts - action code' is present, then a 'Gross distribution from trusts' amount must be present",
                    Location = "/xbrli:xbrl/tns:Income.TrustDistributionGross.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436061" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR48", Value = GetValueOrEmpty(report.SMSFAR48) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR47", Value = GetValueOrEmpty(report.SMSFAR47) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436062
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436062()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436062
            Gross foreign income must not be less than Net foreign income.
    
            Legacy Rule Format:
            IF [SMSFAR37] <> NULL AND [SMSFAR37] < [SMSFAR38]  
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^SMSFAR37 <> NULL AND ^SMSFAR37 < ^SMSFAR38
    
            Data Elements:
    
            ^SMSFAR37 = RP.JFOREIGN:Income.Gross.Amount
    
            ^SMSFAR38 = RP.JFOREIGN:Income.Net.Amount
            */
            assertion = (report.SMSFAR37 != null && report.SMSFAR37.GetValueOrDefault() < report.SMSFAR38.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.434041",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Gross foreign income must not be less than Net foreign income.",
                    Location = "/xbrli:xbrl/tns:Income.Gross.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436062" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR37", Value = GetValueOrEmpty(report.SMSFAR37) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR38", Value = GetValueOrEmpty(report.SMSFAR38) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436063
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436063()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436063
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [SMSFAR47] <> NULL AND [SMSFAR47] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR47, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR47 = RP:Income.TrustDistributionGross.Amount
            */
            assertion = NotMonetary(report.SMSFAR47, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.TrustDistributionGross.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436063" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR47", Value = GetValueOrEmpty(report.SMSFAR47) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436064
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436064()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436064
            If a 'Gross distribution from trusts' amount is present, then a 'Gross distribution from trusts - action code' must be present
    
            Legacy Rule Format:
            IF [SMSFAR47] > 0 AND [SMSFAR48] = NULLORBLANK
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^SMSFAR47 > 0 AND ^SMSFAR48 = BLANK
    
            Data Elements:
    
            ^SMSFAR48 = RP:Party.TrustType.Code
    
            ^SMSFAR47 = RP:Income.TrustDistributionGross.Amount
            */
            assertion = (report.SMSFAR47.GetValueOrDefault() > 0 && string.IsNullOrWhiteSpace(report.SMSFAR48) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.SMSFAR.436064",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Gross distribution from trusts - action code must be present",
                    LongDescription = @"If a 'Gross distribution from trusts' amount is present, then a 'Gross distribution from trusts - action code' must be present",
                    Location = "/xbrli:xbrl/tns:Party.TrustType.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436064" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR47", Value = GetValueOrEmpty(report.SMSFAR47) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR48", Value = GetValueOrEmpty(report.SMSFAR48) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436066
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436066()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436066
            If a Member information statement contains an 'Employer contribution' amount greater than zero, an 'Inward rollover amounts' greater than zero and the 'Member's TFN' is present, then there must be an 'Assessable employer contributions' amount in the Income section of the return
    
            Legacy Rule Format:
            IF ([SMSFAR49] = 0 OR [SMSFAR49] = NULL)  AND COUNT(TUPLE(smsfar.0013.lodge.req.xx.xx:MemberInformation) WHERE ([SMSFAR203] > 0 AND [SMSFAR201] <> NULLORBLANK AND ([SMSFAR219] = 0 OR [SMSFAR219] = NULL)))>0
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^SMSFAR49 = 0 OR ^SMSFAR49 = NULL) AND CountOccurrence(^MemberInformation, ^SMSFAR203 > 0 AND ^SMSFAR201 <> BLANK AND (^SMSFAR219 = 0 OR ^SMSFAR219 = NULL)) > 0
    
            Data Elements:
    
            ^SMSFAR49 = RP:SuperannuationContribution.EmployerAssessable.Amount
    
            ^SMSFAR203 = RP:MemberInformation:SuperannuationContribution.EmployerContributions.Amount
    
            ^SMSFAR201 = RP:MemberInformation:Identifiers.TaxFileNumber.Identifier
    
            ^SMSFAR219 = RP:MemberInformation:SuperannuationRollover.Received.Amount
    
            ^MemberInformation = RP:MemberInformation
            */
            assertion = ((report.SMSFAR49 == 0 || report.SMSFAR49 == null) && (report.MemberInformationCollection == null ? 0 : report.MemberInformationCollection.Count(MemberInformationRepeat => MemberInformationRepeat.SMSFAR203 > 0 && string.IsNullOrWhiteSpace(MemberInformationRepeat.SMSFAR201) != true && (MemberInformationRepeat.SMSFAR219 == 0 || MemberInformationRepeat.SMSFAR219 == null))) > 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.SMSFAR.436066",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Assessable employer contributions must be present",
                    LongDescription = @"If a Member information statement contains an 'Employer contribution' amount greater than zero, an 'Inward rollover amounts' greater than zero and the 'Member's TFN' is present, then there must be an 'Assessable employer contributions' amount in the Income section of the return",
                    Location = "/xbrli:xbrl/tns:SuperannuationContribution.EmployerAssessable.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436066" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR49", Value = GetValueOrEmpty(report.SMSFAR49) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436067
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436067()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436067
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [SMSFAR49] <> NULL AND [SMSFAR49] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR49, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR49 = RP:SuperannuationContribution.EmployerAssessable.Amount
            */
            assertion = NotMonetary(report.SMSFAR49, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:SuperannuationContribution.EmployerAssessable.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436067" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR49", Value = GetValueOrEmpty(report.SMSFAR49) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436068
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436068()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436068
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [SMSFAR50] <> NULL AND [SMSFAR50] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR50, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR50 = RP:SuperannuationContribution.PersonalAssessable.Amount
            */
            assertion = NotMonetary(report.SMSFAR50, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:SuperannuationContribution.PersonalAssessable.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436068" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR50", Value = GetValueOrEmpty(report.SMSFAR50) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436069
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436069()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436069
            If a Member information statement contains an 'Employer contribution' amount greater than zero and the 'Member's TFN' is blank, then there must be a 'No -TFN-quoted contributions'  amount in the Income section of the return
    
            Legacy Rule Format:
            IF [SMSFAR51] = 0 AND COUNT(TUPLE(smsfar.0013.lodge.req.xx.xx:MemberInformation) WHERE ([SMSFAR203] > 0 AND [SMSFAR201] = NULLORBLANK)) > 0
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^SMSFAR51 = 0 AND CountOccurrence(^MemberInformation, ^SMSFAR203 > 0 AND ^SMSFAR201 = BLANK) > 0
    
            Data Elements:
    
            ^SMSFAR51 = RP:SuperannuationContribution.EmployerContributionsTaxFileNumberNotQuoted.Amount
    
            ^SMSFAR203 = RP:MemberInformation:SuperannuationContribution.EmployerContributions.Amount
    
            ^SMSFAR201 = RP:MemberInformation:Identifiers.TaxFileNumber.Identifier
    
            ^MemberInformation = RP:MemberInformation
            */
            assertion = (report.SMSFAR51 == 0 && (report.MemberInformationCollection == null ? 0 : report.MemberInformationCollection.Count(MemberInformationRepeat => MemberInformationRepeat.SMSFAR203 > 0 && string.IsNullOrWhiteSpace(MemberInformationRepeat.SMSFAR201) == true)) > 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.SMSFAR.436671",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The 'No-TFN-quoted contributions' amount must be present",
                    LongDescription = @"If a Member information statement contains an 'Employer contribution' amount greater than zero and the 'Member's TFN' is blank, then there must be a 'No-TFN-quoted contributions'  amount in the Income section of the return",
                    Location = "/xbrli:xbrl/tns:SuperannuationContribution.EmployerContributionsTaxFileNumberNotQuoted.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436069" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR51", Value = GetValueOrEmpty(report.SMSFAR51) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436071
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436071()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436071
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [SMSFAR52] <> NULL AND [SMSFAR52] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR52, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR52 = RP:SuperannuationContribution.LiabilityTransferToLifeInsuranceCompanyOrPooledSuperannuationTrustAssessable.Amount
            */
            assertion = NotMonetary(report.SMSFAR52, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:SuperannuationContribution.LiabilityTransferToLifeInsuranceCompanyOrPooledSuperannuationTrustAssessable.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436071" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR52", Value = GetValueOrEmpty(report.SMSFAR52) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436073
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436073()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436073
            If there is an amount greater than zero at 'Credit for tax withheld - foreign resident withholding (excluding capital gains)', then there must be an amount at 'Gross distributions from partnerships', 'Gross distributions from trusts' or 'Other income'
    
            Legacy Rule Format:
            IF [SMSFAR79] > 0 AND ([SMSFAR53] = NULL AND [SMSFAR43] = NULL AND [SMSFAR47] = NULL) 
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^SMSFAR79 > 0 AND (^SMSFAR53 = NULL AND ^SMSFAR43 = NULL AND ^SMSFAR47 = NULL)
    
            Data Elements:
    
            ^SMSFAR53 = RP:Income.Other.Amount
    
            ^SMSFAR79 = RP:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromForeignResidents.Amount
    
            ^SMSFAR43 = RP:Income.PartnershipDistributionGross.Amount
    
            ^SMSFAR47 = RP:Income.TrustDistributionGross.Amount
            */
            assertion = (report.SMSFAR79.GetValueOrDefault() > 0 && (report.SMSFAR53 == null && report.SMSFAR43 == null && report.SMSFAR47 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.SMSFAR.437136",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Gross payments subject to foreign resident withholding (excluding capital gains) amount required.",
                    LongDescription = @"If there is an amount greater than zero at 'Credit for tax withheld - foreign resident withholding (excluding capital gains)', then there must be an amount at 'Gross distributions from partnerships', 'Gross distributions from trusts' or 'Other income'.",
                    Location = "/xbrli:xbrl/tns:Income.Other.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436073" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR79", Value = GetValueOrEmpty(report.SMSFAR79) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR53", Value = GetValueOrEmpty(report.SMSFAR53) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR43", Value = GetValueOrEmpty(report.SMSFAR43) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR47", Value = GetValueOrEmpty(report.SMSFAR47) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436074
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436074()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436074
            The 'Other income' amount must be greater than zero if the 'Other income - code' is present
    
            Legacy Rule Format:
            IF [SMSFAR54] <> NULL AND ([SMSFAR53] = NULL OR [SMSFAR53] = 0)   
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^SMSFAR54 <> NULL AND (^SMSFAR53 = NULL OR ^SMSFAR53 = 0)
    
            Data Elements:
    
            ^SMSFAR53 = RP:Income.Other.Amount
    
            ^SMSFAR54 = RP:Income.Other.Code
            */
            assertion = (report.SMSFAR54 != null && (report.SMSFAR53 == null || report.SMSFAR53 == 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.434055",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other income amount must be present",
                    LongDescription = @"The 'Other income' amount must be greater than zero if the 'Other income - code' is present",
                    Location = "/xbrli:xbrl/tns:Income.Other.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436074" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR54", Value = GetValueOrEmpty(report.SMSFAR54) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR53", Value = GetValueOrEmpty(report.SMSFAR53) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436075
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436075()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436075
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [SMSFAR53] <> NULL AND [SMSFAR53] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR53, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR53 = RP:Income.Other.Amount
            */
            assertion = NotMonetary(report.SMSFAR53, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.Other.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436075" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR53", Value = GetValueOrEmpty(report.SMSFAR53) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436076
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436076()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436076
            Other income code must be present if Other income amount is present.
    
            Legacy Rule Format:
            IF [SMSFAR53] > 0 AND [SMSFAR54] = NULLORBLANK
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^SMSFAR53 > 0 AND ^SMSFAR54 = BLANK
    
            Data Elements:
    
            ^SMSFAR54 = RP:Income.Other.Code
    
            ^SMSFAR53 = RP:Income.Other.Amount
            */
            assertion = (report.SMSFAR53.GetValueOrDefault() > 0 && string.IsNullOrWhiteSpace(report.SMSFAR54) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.434058",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other income code must be present if Other income amount is present.",
                    LongDescription = @"Other income codes are:    'B' - Assessable balancing adjustment amount,    'C' - LIC capital gain amount,    'F' - Forex gains,    'O' - Other income received not listed,    'R' - Rebate or refund of death or disability premiums,    'W' - Gross payments subject to foreign resident withholding, 'T' - TOFA gains or assessable TOFA balancing adjustment",
                    Location = "/xbrli:xbrl/tns:Income.Other.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436076" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR53", Value = GetValueOrEmpty(report.SMSFAR53) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR54", Value = GetValueOrEmpty(report.SMSFAR54) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436078
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436078()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436078
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [SMSFAR55] <> NULL AND [SMSFAR55] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR55, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR55 = RP:Income.AssessableIncomeDueToFundTaxStatusChange.Amount
            */
            assertion = NotMonetary(report.SMSFAR55, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.AssessableIncomeDueToFundTaxStatusChange.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436078" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR55", Value = GetValueOrEmpty(report.SMSFAR55) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436079
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436079()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436079
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [SMSFAR56] <> NULL AND [SMSFAR56] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR56, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR56 = RP:Income.NonArmsLengthPrivateCompanyDividendsNet.Amount
            */
            assertion = NotMonetary(report.SMSFAR56, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.NonArmsLengthPrivateCompanyDividendsNet.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436079" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR56", Value = GetValueOrEmpty(report.SMSFAR56) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436080
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436080()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436080
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [SMSFAR57] <> NULL AND [SMSFAR57] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR57, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR57 = RP:Income.NonArmsLengthTrustDistributionsNet.Amount
            */
            assertion = NotMonetary(report.SMSFAR57, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.NonArmsLengthTrustDistributionsNet.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436080" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR57", Value = GetValueOrEmpty(report.SMSFAR57) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436081
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436081()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436081
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [SMSFAR58] <> NULL AND [SMSFAR58] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR58, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR58 = RP:Income.NonArmsLengthOtherNet.Amount
            */
            assertion = NotMonetary(report.SMSFAR58, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.NonArmsLengthOtherNet.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436081" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR58", Value = GetValueOrEmpty(report.SMSFAR58) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436082
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436082()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436082
            If the Taxable income or loss amount is less than zero, there must be an 'Exempt current pension income' amount or a 'Tax losses carried forward to later income years' amount
    
            Legacy Rule Format:
            IF [SMSFAR73] < 0 AND ([SMSFAR301] = 0 OR [SMSFAR301] = NULL) AND ([SMSFAR85] = 0 OR [SMSFAR85] = NULL)  
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^SMSFAR73 < 0 AND (^SMSFAR301 = 0 OR ^SMSFAR301 = NULL) AND (^SMSFAR85 = 0 OR ^SMSFAR85 = NULL)
    
            Data Elements:
    
            ^SMSFAR301 = RP:TaxConcession.PensionIncomeExemptDeduction.Amount
    
            ^SMSFAR73 = RP:Income.Taxable.Amount
    
            ^SMSFAR85 = RP.Closing:Tax.Losses.CarriedForward.LaterIncomeYearsTotal.Amount
            */
            assertion = (report.SMSFAR73.GetValueOrDefault() < 0 && (report.SMSFAR301 == 0 || report.SMSFAR301 == null) && (report.SMSFAR85 == 0 || report.SMSFAR85 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.434073",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Tax losses carried forward amount must be present",
                    LongDescription = @"If the Taxable income or loss amount is less than zero, there must be an 'Exempt current pension income' amount or a 'Tax losses carried forward to later income years' amount",
                    Location = "/xbrli:xbrl/tns:TaxConcession.PensionIncomeExemptDeduction.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436082" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR73", Value = GetValueOrEmpty(report.SMSFAR73) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR301", Value = GetValueOrEmpty(report.SMSFAR301) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR85", Value = GetValueOrEmpty(report.SMSFAR85) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436083
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436083()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436083
            If the answer to questions "Did the fund pay retirement phase superannuation income stream benefits to one or more members in the income year?" is "False" or  "Did the fund have any other income that was assessable?" is "True", then exempt current pension income cannot be greater than gross income less transfers from foreign funds less assessable contributions less assessable income due to changed tax status of fund less net non-arm's length income.
    
            Legacy Rule Format:
            IF ([SMSFAR321] = FALSE OR [SMSFAR352] = TRUE) AND ([SMSFAR301]<> NULL AND [SMSFAR301] > 0) AND ([SMSFAR301] > ([SMSFAR33] + [SMSFAR34] + [SMSFAR35] + [SMSFAR36] +
            [SMSFAR38] + [SMSFAR39] + [SMSFAR42] + [SMSFAR43] + [SMSFAR44] +
            [SMSFAR45] + [SMSFAR46] + [SMSFAR47] + [SMSFAR53]))
              RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^SMSFAR321 = FALSE OR ^SMSFAR352 = TRUE) AND (^SMSFAR301 <> NULL AND ^SMSFAR301 > 0) AND (^SMSFAR301 > (^SMSFAR33 + ^SMSFAR34 + ^SMSFAR35 + ^SMSFAR36 +^SMSFAR38 + ^SMSFAR39 + ^SMSFAR42 + ^SMSFAR43 + ^SMSFAR44 +^SMSFAR45 + ^SMSFAR46 + ^SMSFAR47 + ^SMSFAR53))
    
            Data Elements:
    
            ^SMSFAR301 = RP:TaxConcession.PensionIncomeExemptDeduction.Amount
    
            ^SMSFAR321 = RP:SuperannuationFundDetails.IncomeStreamPaid.Indicator
    
            ^SMSFAR352 = RP:Income.AssessableOther.Indicator
    
            ^SMSFAR33 = RP:Income.CapitalGainsNet.Amount
    
            ^SMSFAR34 = RP:Income.RentLeasingHiringGross.Amount
    
            ^SMSFAR35 = RP:Income.Interest.Gross.Amount
    
            ^SMSFAR36 = RP:Income.ForestryManagedInvestmentScheme.Amount
    
            ^SMSFAR38 = RP.JFOREIGN:Income.Net.Amount
    
            ^SMSFAR39 = RP:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
    
            ^SMSFAR42 = RP:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^SMSFAR43 = RP:Income.PartnershipDistributionGross.Amount
    
            ^SMSFAR44 = RP:Income.DividendsUnfranked.Amount
    
            ^SMSFAR45 = RP:Income.DividendsFranked.Amount
    
            ^SMSFAR46 = RP:IncomeTax.FrankingCredits.Amount
    
            ^SMSFAR47 = RP:Income.TrustDistributionGross.Amount
    
            ^SMSFAR53 = RP:Income.Other.Amount
            */
            assertion = ((report.SMSFAR321 == false || report.SMSFAR352 == true) && (report.SMSFAR301 != null && report.SMSFAR301.GetValueOrDefault() > 0) && report.SMSFAR301.GetValueOrDefault() > report.SMSFAR33.GetValueOrDefault() + report.SMSFAR34.GetValueOrDefault() + report.SMSFAR35.GetValueOrDefault() + report.SMSFAR36.GetValueOrDefault() + report.SMSFAR38.GetValueOrDefault() + report.SMSFAR39.GetValueOrDefault() + report.SMSFAR42.GetValueOrDefault() + report.SMSFAR43.GetValueOrDefault() + report.SMSFAR44.GetValueOrDefault() + report.SMSFAR45.GetValueOrDefault() + report.SMSFAR46.GetValueOrDefault() + report.SMSFAR47.GetValueOrDefault() + report.SMSFAR53.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.SMSFAR.436352",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Exempt current pension income cannot be greater than gross income",
                    LongDescription = @"Exempt current pension income cannot be greater than Gross income less Transfers from foreign funds less Assessable contributions less Assessable income due to changed tax status of fund less Net non-arm's length income",
                    Location = "/xbrli:xbrl/tns:TaxConcession.PensionIncomeExemptDeduction.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436083" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR321", Value = GetValueOrEmpty(report.SMSFAR321) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR352", Value = GetValueOrEmpty(report.SMSFAR352) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR301", Value = GetValueOrEmpty(report.SMSFAR301) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR33", Value = GetValueOrEmpty(report.SMSFAR33) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR34", Value = GetValueOrEmpty(report.SMSFAR34) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR35", Value = GetValueOrEmpty(report.SMSFAR35) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR36", Value = GetValueOrEmpty(report.SMSFAR36) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR38", Value = GetValueOrEmpty(report.SMSFAR38) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR39", Value = GetValueOrEmpty(report.SMSFAR39) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR42", Value = GetValueOrEmpty(report.SMSFAR42) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR43", Value = GetValueOrEmpty(report.SMSFAR43) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR44", Value = GetValueOrEmpty(report.SMSFAR44) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR45", Value = GetValueOrEmpty(report.SMSFAR45) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR46", Value = GetValueOrEmpty(report.SMSFAR46) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR47", Value = GetValueOrEmpty(report.SMSFAR47) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR53", Value = GetValueOrEmpty(report.SMSFAR53) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436084
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436084()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436084
            Exempt current pension income equal to gross income and other deduction amounts are greater than zero
    
            Legacy Rule Format:
            IF ([SMSFAR301] <> NULL) AND  ([SMSFAR301] = ([SMSFAR33] + [SMSFAR34] + [SMSFAR35] + [SMSFAR36] + [SMSFAR38] + [SMSFAR39] + [SMSFAR40] + [SMSFAR42] + [SMSFAR43] + [SMSFAR44] + [SMSFAR45] + [SMSFAR46] + [SMSFAR47] + [SMSFAR49] + [SMSFAR50] + [SMSFAR51] - [SMSFAR52] + [SMSFAR53] + [SMSFAR55] + [SMSFAR56] + [SMSFAR57] + [SMSFAR58]) AND ([SMSFAR60] > 0 OR [SMSFAR61] > 0 OR [SMSFAR62] > 0 OR [SMSFAR63] > 0 OR [SMSFAR64] > 0 OR [SMSFAR66] > 0 OR [SMSFAR67] > 0 OR [SMSFAR302] > 0 OR [SMSFAR69] > 0 OR [SMSFAR70] > 0 OR [SMSFAR72] > 0)) 
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^SMSFAR301 <> NULL) AND  (^SMSFAR301 = (^SMSFAR33 + ^SMSFAR34 + ^SMSFAR35 + ^SMSFAR36 + ^SMSFAR38 + ^SMSFAR39 + ^SMSFAR40 + ^SMSFAR42 + ^SMSFAR43 + ^SMSFAR44 + ^SMSFAR45 + ^SMSFAR46 + ^SMSFAR47 + ^SMSFAR49 + ^SMSFAR50 + ^SMSFAR51 - ^SMSFAR52 + ^SMSFAR53 + ^SMSFAR55 + ^SMSFAR56 + ^SMSFAR57 + ^SMSFAR58) AND (^SMSFAR60 > 0 OR ^SMSFAR61 > 0 OR ^SMSFAR62 > 0 OR ^SMSFAR63 > 0 OR ^SMSFAR64 > 0 OR ^SMSFAR66 > 0 OR ^SMSFAR67 > 0 OR ^SMSFAR302 > 0 OR ^SMSFAR69 > 0 OR ^SMSFAR70 > 0 OR ^SMSFAR72 > 0))
    
            Data Elements:
    
            ^SMSFAR301 = RP:TaxConcession.PensionIncomeExemptDeduction.Amount
    
            ^SMSFAR33 = RP:Income.CapitalGainsNet.Amount
    
            ^SMSFAR34 = RP:Income.RentLeasingHiringGross.Amount
    
            ^SMSFAR35 = RP:Income.Interest.Gross.Amount
    
            ^SMSFAR36 = RP:Income.ForestryManagedInvestmentScheme.Amount
    
            ^SMSFAR38 = RP.JFOREIGN:Income.Net.Amount
    
            ^SMSFAR39 = RP:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
    
            ^SMSFAR40 = RP:Equity.InternationalDealings.ForeignFundTransfers.Amount
    
            ^SMSFAR42 = RP:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^SMSFAR43 = RP:Income.PartnershipDistributionGross.Amount
    
            ^SMSFAR44 = RP:Income.DividendsUnfranked.Amount
    
            ^SMSFAR45 = RP:Income.DividendsFranked.Amount
    
            ^SMSFAR46 = RP:IncomeTax.FrankingCredits.Amount
    
            ^SMSFAR47 = RP:Income.TrustDistributionGross.Amount
    
            ^SMSFAR49 = RP:SuperannuationContribution.EmployerAssessable.Amount
    
            ^SMSFAR50 = RP:SuperannuationContribution.PersonalAssessable.Amount
    
            ^SMSFAR51 = RP:SuperannuationContribution.EmployerContributionsTaxFileNumberNotQuoted.Amount
    
            ^SMSFAR52 = RP:SuperannuationContribution.LiabilityTransferToLifeInsuranceCompanyOrPooledSuperannuationTrustAssessable.Amount
    
            ^SMSFAR53 = RP:Income.Other.Amount
    
            ^SMSFAR55 = RP:Income.AssessableIncomeDueToFundTaxStatusChange.Amount
    
            ^SMSFAR56 = RP:Income.NonArmsLengthPrivateCompanyDividendsNet.Amount
    
            ^SMSFAR57 = RP:Income.NonArmsLengthTrustDistributionsNet.Amount
    
            ^SMSFAR58 = RP:Income.NonArmsLengthOtherNet.Amount
    
            ^SMSFAR60 = RP.Deductible.JAUS:Expense.Interest.Amount
    
            ^SMSFAR61 = RP.Deductible.JFOREIGN:Expense.Interest.Amount
    
            ^SMSFAR62 = RP.Deductible:Expense.CapitalWorks.Amount
    
            ^SMSFAR63 = RP.Deductible:Expense.DepreciationAndAmortisation.Amount
    
            ^SMSFAR64 = RP.Deductible:Expense.PremiumsDeathOrDisability.Amount
    
            ^SMSFAR66 = RP.Deductible:Expense.Operating.ApprovedAuditorFee.Amount
    
            ^SMSFAR67 = RP.Deductible:Expense.Investment.Amount
    
            ^SMSFAR302 = RP.Deductible:Expense.Operating.GeneralManagementAndAdministrationFees.Amount
    
            ^SMSFAR69 = RP.Deductible:Expense.ForestryManagedInvestmentScheme.Amount
    
            ^SMSFAR70 = RP.Deductible:Expense.Other.Amount
    
            ^SMSFAR72 = RP.Deductible:IncomeTax.Deduction.TaxLossesDeducted.Amount
            */
            assertion = (report.SMSFAR301 != null && (report.SMSFAR301.GetValueOrDefault() == report.SMSFAR33.GetValueOrDefault() + report.SMSFAR34.GetValueOrDefault() + report.SMSFAR35.GetValueOrDefault() + report.SMSFAR36.GetValueOrDefault() + report.SMSFAR38.GetValueOrDefault() + report.SMSFAR39.GetValueOrDefault() + report.SMSFAR40.GetValueOrDefault() + report.SMSFAR42.GetValueOrDefault() + report.SMSFAR43.GetValueOrDefault() + report.SMSFAR44.GetValueOrDefault() + report.SMSFAR45.GetValueOrDefault() + report.SMSFAR46.GetValueOrDefault() + report.SMSFAR47.GetValueOrDefault() + report.SMSFAR49.GetValueOrDefault() + report.SMSFAR50.GetValueOrDefault() + report.SMSFAR51.GetValueOrDefault() - report.SMSFAR52.GetValueOrDefault() + report.SMSFAR53.GetValueOrDefault() + report.SMSFAR55.GetValueOrDefault() + report.SMSFAR56.GetValueOrDefault() + report.SMSFAR57.GetValueOrDefault() + report.SMSFAR58.GetValueOrDefault() && (report.SMSFAR60.GetValueOrDefault() > 0 || report.SMSFAR61.GetValueOrDefault() > 0 || report.SMSFAR62.GetValueOrDefault() > 0 || report.SMSFAR63.GetValueOrDefault() > 0 || report.SMSFAR64.GetValueOrDefault() > 0 || report.SMSFAR66.GetValueOrDefault() > 0 || report.SMSFAR67.GetValueOrDefault() > 0 || report.SMSFAR302.GetValueOrDefault() > 0 || report.SMSFAR69.GetValueOrDefault() > 0 || report.SMSFAR70.GetValueOrDefault() > 0 || report.SMSFAR72.GetValueOrDefault() > 0)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.SMSFAR.436345",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Exempt current pension income equal to gross income and other deduction labels are greater than 0.",
                    Location = "/xbrli:xbrl/tns:TaxConcession.PensionIncomeExemptDeduction.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436084" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR301", Value = GetValueOrEmpty(report.SMSFAR301) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR33", Value = GetValueOrEmpty(report.SMSFAR33) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR34", Value = GetValueOrEmpty(report.SMSFAR34) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR35", Value = GetValueOrEmpty(report.SMSFAR35) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR36", Value = GetValueOrEmpty(report.SMSFAR36) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR38", Value = GetValueOrEmpty(report.SMSFAR38) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR39", Value = GetValueOrEmpty(report.SMSFAR39) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR40", Value = GetValueOrEmpty(report.SMSFAR40) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR42", Value = GetValueOrEmpty(report.SMSFAR42) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR43", Value = GetValueOrEmpty(report.SMSFAR43) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR44", Value = GetValueOrEmpty(report.SMSFAR44) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR45", Value = GetValueOrEmpty(report.SMSFAR45) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR46", Value = GetValueOrEmpty(report.SMSFAR46) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR47", Value = GetValueOrEmpty(report.SMSFAR47) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR49", Value = GetValueOrEmpty(report.SMSFAR49) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR50", Value = GetValueOrEmpty(report.SMSFAR50) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR51", Value = GetValueOrEmpty(report.SMSFAR51) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR52", Value = GetValueOrEmpty(report.SMSFAR52) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR53", Value = GetValueOrEmpty(report.SMSFAR53) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR55", Value = GetValueOrEmpty(report.SMSFAR55) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR56", Value = GetValueOrEmpty(report.SMSFAR56) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR57", Value = GetValueOrEmpty(report.SMSFAR57) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR58", Value = GetValueOrEmpty(report.SMSFAR58) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR60", Value = GetValueOrEmpty(report.SMSFAR60) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR61", Value = GetValueOrEmpty(report.SMSFAR61) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR62", Value = GetValueOrEmpty(report.SMSFAR62) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR63", Value = GetValueOrEmpty(report.SMSFAR63) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR64", Value = GetValueOrEmpty(report.SMSFAR64) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR66", Value = GetValueOrEmpty(report.SMSFAR66) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR67", Value = GetValueOrEmpty(report.SMSFAR67) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR302", Value = GetValueOrEmpty(report.SMSFAR302) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR69", Value = GetValueOrEmpty(report.SMSFAR69) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR70", Value = GetValueOrEmpty(report.SMSFAR70) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR72", Value = GetValueOrEmpty(report.SMSFAR72) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436086
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436086()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436086
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF bafpr1.xx.xx:Expense.Interest.Amount <> NULL AND bafpr1.xx.xx:Expense.Interest.Amount <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR61, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR61 = RP.Deductible.JFOREIGN:Expense.Interest.Amount
            */
            assertion = NotMonetary(report.SMSFAR61, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.Interest.Amount[@contextRef='" + report.RPDeductibleJFOREIGNId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436086" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR61", Value = GetValueOrEmpty(report.SMSFAR61) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436086
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436086_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436086
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF bafpr1.xx.xx:Expense.Interest.Amount <> NULL AND bafpr1.xx.xx:Expense.Interest.Amount <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR330, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR330 = RP.NonDeductible.JFOREIGN:Expense.Interest.Amount
            */
            assertion = NotMonetary(report.SMSFAR330, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.Interest.Amount[@contextRef='" + report.RPNonDeductibleJFOREIGNId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436086" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR330", Value = GetValueOrEmpty(report.SMSFAR330) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436086
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436086_2()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436086
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF bafpr1.xx.xx:Expense.Interest.Amount <> NULL AND bafpr1.xx.xx:Expense.Interest.Amount <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR60, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR60 = RP.Deductible.JAUS:Expense.Interest.Amount
            */
            assertion = NotMonetary(report.SMSFAR60, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.Interest.Amount[@contextRef='" + report.RPDeductibleJAUSId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436086" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR60", Value = GetValueOrEmpty(report.SMSFAR60) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436086
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436086_3()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436086
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF bafpr1.xx.xx:Expense.Interest.Amount <> NULL AND bafpr1.xx.xx:Expense.Interest.Amount <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR329, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR329 = RP.NonDeductible.JAUS:Expense.Interest.Amount
            */
            assertion = NotMonetary(report.SMSFAR329, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.Interest.Amount[@contextRef='" + report.RPNonDeductibleJAUSId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436086" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR329", Value = GetValueOrEmpty(report.SMSFAR329) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436088
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436088()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436088
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF bafpr1.xx.xx:Expense.CapitalWorks.Amount <> NULL AND bafpr1.xx.xx:Expense.CapitalWorks.Amount <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR62, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR62 = RP.Deductible:Expense.CapitalWorks.Amount
            */
            assertion = NotMonetary(report.SMSFAR62, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.CapitalWorks.Amount[@contextRef='" + report.RPDeductibleId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436088" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR62", Value = GetValueOrEmpty(report.SMSFAR62) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436088
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436088_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436088
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF bafpr1.xx.xx:Expense.CapitalWorks.Amount <> NULL AND bafpr1.xx.xx:Expense.CapitalWorks.Amount <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR331, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR331 = RP.NonDeductible:Expense.CapitalWorks.Amount
            */
            assertion = NotMonetary(report.SMSFAR331, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.CapitalWorks.Amount[@contextRef='" + report.RPNonDeductibleId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436088" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR331", Value = GetValueOrEmpty(report.SMSFAR331) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436091
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436091()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436091
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF bafpr1.xx.xx:Expense.DepreciationAndAmortisation.Amount <> NULL AND bafpr1.xx.xx:Expense.DepreciationAndAmortisation.Amount <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR63, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR63 = RP.Deductible:Expense.DepreciationAndAmortisation.Amount
            */
            assertion = NotMonetary(report.SMSFAR63, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.DepreciationAndAmortisation.Amount[@contextRef='" + report.RPDeductibleId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436091" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR63", Value = GetValueOrEmpty(report.SMSFAR63) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436091
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436091_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436091
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF bafpr1.xx.xx:Expense.DepreciationAndAmortisation.Amount <> NULL AND bafpr1.xx.xx:Expense.DepreciationAndAmortisation.Amount <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR332, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR332 = RP.NonDeductible:Expense.DepreciationAndAmortisation.Amount
            */
            assertion = NotMonetary(report.SMSFAR332, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.DepreciationAndAmortisation.Amount[@contextRef='" + report.RPNonDeductibleId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436091" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR332", Value = GetValueOrEmpty(report.SMSFAR332) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436092
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436092()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436092
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF bafpr1.xx.xx:Expense.PremiumsDeathOrDisability.Amount <> NULL AND bafpr1.xx.xx:Expense.PremiumsDeathOrDisability.Amount <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR64, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR64 = RP.Deductible:Expense.PremiumsDeathOrDisability.Amount
            */
            assertion = NotMonetary(report.SMSFAR64, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.PremiumsDeathOrDisability.Amount[@contextRef='" + report.RPDeductibleId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436092" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR64", Value = GetValueOrEmpty(report.SMSFAR64) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436092
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436092_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436092
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF bafpr1.xx.xx:Expense.PremiumsDeathOrDisability.Amount <> NULL AND bafpr1.xx.xx:Expense.PremiumsDeathOrDisability.Amount <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR353, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR353 = RP.NonDeductible:Expense.PremiumsDeathOrDisability.Amount
            */
            assertion = NotMonetary(report.SMSFAR353, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.PremiumsDeathOrDisability.Amount[@contextRef='" + report.RPNonDeductibleId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436092" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR353", Value = GetValueOrEmpty(report.SMSFAR353) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436094
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436094()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436094
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF bafpr2.xx.xx:Expense.Operating.ApprovedAuditorFee.Amount <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR66, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR66 = RP.Deductible:Expense.Operating.ApprovedAuditorFee.Amount
            */
            assertion = NotMonetary(report.SMSFAR66, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.Operating.ApprovedAuditorFee.Amount[@contextRef='" + report.RPDeductibleId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436094" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR66", Value = GetValueOrEmpty(report.SMSFAR66) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436094
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436094_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436094
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF bafpr2.xx.xx:Expense.Operating.ApprovedAuditorFee.Amount <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR333, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR333 = RP.NonDeductible:Expense.Operating.ApprovedAuditorFee.Amount
            */
            assertion = NotMonetary(report.SMSFAR333, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.Operating.ApprovedAuditorFee.Amount[@contextRef='" + report.RPNonDeductibleId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436094" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR333", Value = GetValueOrEmpty(report.SMSFAR333) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436095
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436095()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436095
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF bafpr1.xx.xx:Expense.Investment.Amount <> NULL AND bafpr1.xx.xx:Expense.Investment.Amount <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR67, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR67 = RP.Deductible:Expense.Investment.Amount
            */
            assertion = NotMonetary(report.SMSFAR67, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.Investment.Amount[@contextRef='" + report.RPDeductibleId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436095" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR67", Value = GetValueOrEmpty(report.SMSFAR67) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436095
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436095_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436095
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF bafpr1.xx.xx:Expense.Investment.Amount <> NULL AND bafpr1.xx.xx:Expense.Investment.Amount <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR334, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR334 = RP.NonDeductible:Expense.Investment.Amount
            */
            assertion = NotMonetary(report.SMSFAR334, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.Investment.Amount[@contextRef='" + report.RPNonDeductibleId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436095" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR334", Value = GetValueOrEmpty(report.SMSFAR334) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436098
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436098()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436098
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF bafpr1.xx.xx:Expense.ForestryManagedInvestmentScheme.Amount <> NULL AND bafpr1.xx.xx:Expense.ForestryManagedInvestmentScheme.Amount <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR69, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR69 = RP.Deductible:Expense.ForestryManagedInvestmentScheme.Amount
            */
            assertion = NotMonetary(report.SMSFAR69, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.ForestryManagedInvestmentScheme.Amount[@contextRef='" + report.RPDeductibleId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436098" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR69", Value = GetValueOrEmpty(report.SMSFAR69) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436098
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436098_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436098
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF bafpr1.xx.xx:Expense.ForestryManagedInvestmentScheme.Amount <> NULL AND bafpr1.xx.xx:Expense.ForestryManagedInvestmentScheme.Amount <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR335, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR335 = RP.NonDeductible:Expense.ForestryManagedInvestmentScheme.Amount
            */
            assertion = NotMonetary(report.SMSFAR335, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.ForestryManagedInvestmentScheme.Amount[@contextRef='" + report.RPNonDeductibleId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436098" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR335", Value = GetValueOrEmpty(report.SMSFAR335) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436099
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436099()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436099
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF bafpr1.xx.xx:Expense.Other.Amount <> NULL AND bafpr1.xx.xx:Expense.Other.Amount <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR70, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR70 = RP.Deductible:Expense.Other.Amount
            */
            assertion = NotMonetary(report.SMSFAR70, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.Other.Amount[@contextRef='" + report.RPDeductibleId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436099" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR70", Value = GetValueOrEmpty(report.SMSFAR70) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436099
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436099_1()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436099
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF bafpr1.xx.xx:Expense.Other.Amount <> NULL AND bafpr1.xx.xx:Expense.Other.Amount <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR336, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR336 = RP.NonDeductible:Expense.Other.Amount
            */
            assertion = NotMonetary(report.SMSFAR336, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Expense.Other.Amount[@contextRef='" + report.RPNonDeductibleId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436099" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR336", Value = GetValueOrEmpty(report.SMSFAR336) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436100
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436100()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436100
            If 'Other deductions - code' is present, 'Other deductions' amount greater than zero must be present
    
            Legacy Rule Format:
            IF ([SMSFAR70] = 0 OR [SMSFAR70] = NULL) AND [SMSFAR71] <> NULLORBLANK
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^SMSFAR70 = 0 OR ^SMSFAR70 = NULL) AND ^SMSFAR71 <> BLANK
    
            Data Elements:
    
            ^SMSFAR71 = RP.Deductible:Expense.Other.Code
    
            ^SMSFAR70 = RP.Deductible:Expense.Other.Amount
            */
            assertion = ((report.SMSFAR70 == 0 || report.SMSFAR70 == null) && string.IsNullOrWhiteSpace(report.SMSFAR71) != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.434068",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"If 'Other deductions - code' is present, 'Other deductions' amount greater than zero must be present",
                    Location = "/xbrli:xbrl/tns:Expense.Other.Code[@contextRef='" + report.RPDeductibleId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436100" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR70", Value = GetValueOrEmpty(report.SMSFAR70) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR71", Value = GetValueOrEmpty(report.SMSFAR71) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436101
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436101()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436101
            If an 'Other deductions' amount is present, then an 'Other deductions - Code' must be present
    
            Legacy Rule Format:
            IF [SMSFAR70] > 0 AND [SMSFAR71] = NULL
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^SMSFAR70 > 0 AND ^SMSFAR71 = NULL
    
            Data Elements:
    
            ^SMSFAR71 = RP.Deductible:Expense.Other.Code
    
            ^SMSFAR70 = RP.Deductible:Expense.Other.Amount
            */
            assertion = (report.SMSFAR70.GetValueOrDefault() > 0 && report.SMSFAR71 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.434069",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"If an 'Other deductions' amount is present, an 'Other deductions' code must be present",
                    Location = "/xbrli:xbrl/tns:Expense.Other.Code[@contextRef='" + report.RPDeductibleId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436101" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR70", Value = GetValueOrEmpty(report.SMSFAR70) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR71", Value = GetValueOrEmpty(report.SMSFAR71) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436103
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436103()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436103
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [SMSFAR72] <> NULL AND [SMSFAR72] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR72, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR72 = RP.Deductible:IncomeTax.Deduction.TaxLossesDeducted.Amount
            */
            assertion = NotMonetary(report.SMSFAR72, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:IncomeTax.Deduction.TaxLossesDeducted.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436103" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR72", Value = GetValueOrEmpty(report.SMSFAR72) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436104
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436104()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436104
            No 'No TFN-Quoted Contributions' amount present, therefore Gross Tax cannot be greater than the Taxable Income/Loss amount.
    
            Legacy Rule Format:
            IF ([SMSFAR51] = 0 AND [SMSFAR73] <= 0) AND [SMSFAR74] > 0  
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^SMSFAR51 = 0 AND ^SMSFAR73 <= 0) AND ^SMSFAR74 > 0
    
            Data Elements:
    
            ^SMSFAR73 = RP:Income.Taxable.Amount
    
            ^SMSFAR51 = RP:SuperannuationContribution.EmployerContributionsTaxFileNumberNotQuoted.Amount
    
            ^SMSFAR74 = RP:IncomeTax.TaxableGross.Amount
            */
            assertion = (report.SMSFAR51 == 0 && report.SMSFAR73.GetValueOrDefault() <= 0 && report.SMSFAR74.GetValueOrDefault() > 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.SMSFAR.436673",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Gross tax must not be greater than zero",
                    LongDescription = @"If a 'No-TFN-quoted contributions' amount is not present, and 'Taxable income or loss' amount is less than or equal to zero, then the Gross Tax amount cannot be greater than zero",
                    Location = "/xbrli:xbrl/tns:Income.Taxable.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436104" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR51", Value = GetValueOrEmpty(report.SMSFAR51) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR73", Value = GetValueOrEmpty(report.SMSFAR73) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR74", Value = GetValueOrEmpty(report.SMSFAR74) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436105
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436105()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436105
            If the answer to questions "Did the fund pay an income stream to one or more members in the income year?" is "False" or "Did the fund have any other income that was assessable?" is "True", then the taxable income or loss must equal the sum of income amounts minus exempt current pension income minus the sum of all deductions.
    
            Legacy Rule Format:
            IF ([SMSFAR321] = FALSE OR [SMSFAR352] = TRUE) AND [SMSFAR73] <>  (([SMSFAR33] + [SMSFAR34] + [SMSFAR35] + [SMSFAR36] + [SMSFAR38] + [SMSFAR39] + [SMSFAR40] + [SMSFAR42] + [SMSFAR43] + [SMSFAR44] + [SMSFAR45] + [SMSFAR46] + [SMSFAR47]) + ([SMSFAR49] + [SMSFAR50] + [SMSFAR51] - [SMSFAR52]) + ([SMSFAR53] + [SMSFAR55]) + ([SMSFAR56] + [SMSFAR57] + [SMSFAR58]) 
            - ([SMSFAR301] + [SMSFAR60] + [SMSFAR61] + [SMSFAR62] + [SMSFAR63] + [SMSFAR64] + [SMSFAR66] + [SMSFAR67] + [SMSFAR302] + [SMSFAR69] + [SMSFAR70] + [SMSFAR72])) 
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^SMSFAR321 = FALSE OR ^SMSFAR352 = TRUE) AND ^SMSFAR73 <>  ((^SMSFAR33 + ^SMSFAR34 + ^SMSFAR35 + ^SMSFAR36 + ^SMSFAR38 + ^SMSFAR39 + ^SMSFAR40 + ^SMSFAR42 + ^SMSFAR43 + ^SMSFAR44 + ^SMSFAR45 + ^SMSFAR46 + ^SMSFAR47) + (^SMSFAR49 + ^SMSFAR50 + ^SMSFAR51 - ^SMSFAR52) + (^SMSFAR53 + ^SMSFAR55) + (^SMSFAR56 + ^SMSFAR57 + ^SMSFAR58) - (^SMSFAR301 + ^SMSFAR60 + ^SMSFAR61 + ^SMSFAR62 + ^SMSFAR63 + ^SMSFAR64 + ^SMSFAR66 + ^SMSFAR67 + ^SMSFAR302 + ^SMSFAR69 + ^SMSFAR70 + ^SMSFAR72))
    
            Data Elements:
    
            ^SMSFAR73 = RP:Income.Taxable.Amount
    
            ^SMSFAR321 = RP:SuperannuationFundDetails.IncomeStreamPaid.Indicator
    
            ^SMSFAR352 = RP:Income.AssessableOther.Indicator
    
            ^SMSFAR33 = RP:Income.CapitalGainsNet.Amount
    
            ^SMSFAR34 = RP:Income.RentLeasingHiringGross.Amount
    
            ^SMSFAR35 = RP:Income.Interest.Gross.Amount
    
            ^SMSFAR36 = RP:Income.ForestryManagedInvestmentScheme.Amount
    
            ^SMSFAR38 = RP.JFOREIGN:Income.Net.Amount
    
            ^SMSFAR39 = RP:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
    
            ^SMSFAR40 = RP:Equity.InternationalDealings.ForeignFundTransfers.Amount
    
            ^SMSFAR42 = RP:Remuneration.ABNNotQuotedPaymentGross.Amount
    
            ^SMSFAR43 = RP:Income.PartnershipDistributionGross.Amount
    
            ^SMSFAR44 = RP:Income.DividendsUnfranked.Amount
    
            ^SMSFAR45 = RP:Income.DividendsFranked.Amount
    
            ^SMSFAR46 = RP:IncomeTax.FrankingCredits.Amount
    
            ^SMSFAR47 = RP:Income.TrustDistributionGross.Amount
    
            ^SMSFAR49 = RP:SuperannuationContribution.EmployerAssessable.Amount
    
            ^SMSFAR50 = RP:SuperannuationContribution.PersonalAssessable.Amount
    
            ^SMSFAR51 = RP:SuperannuationContribution.EmployerContributionsTaxFileNumberNotQuoted.Amount
    
            ^SMSFAR52 = RP:SuperannuationContribution.LiabilityTransferToLifeInsuranceCompanyOrPooledSuperannuationTrustAssessable.Amount
    
            ^SMSFAR53 = RP:Income.Other.Amount
    
            ^SMSFAR55 = RP:Income.AssessableIncomeDueToFundTaxStatusChange.Amount
    
            ^SMSFAR56 = RP:Income.NonArmsLengthPrivateCompanyDividendsNet.Amount
    
            ^SMSFAR57 = RP:Income.NonArmsLengthTrustDistributionsNet.Amount
    
            ^SMSFAR58 = RP:Income.NonArmsLengthOtherNet.Amount
    
            ^SMSFAR301 = RP:TaxConcession.PensionIncomeExemptDeduction.Amount
    
            ^SMSFAR60 = RP.Deductible.JAUS:Expense.Interest.Amount
    
            ^SMSFAR61 = RP.Deductible.JFOREIGN:Expense.Interest.Amount
    
            ^SMSFAR62 = RP.Deductible:Expense.CapitalWorks.Amount
    
            ^SMSFAR63 = RP.Deductible:Expense.DepreciationAndAmortisation.Amount
    
            ^SMSFAR64 = RP.Deductible:Expense.PremiumsDeathOrDisability.Amount
    
            ^SMSFAR66 = RP.Deductible:Expense.Operating.ApprovedAuditorFee.Amount
    
            ^SMSFAR67 = RP.Deductible:Expense.Investment.Amount
    
            ^SMSFAR302 = RP.Deductible:Expense.Operating.GeneralManagementAndAdministrationFees.Amount
    
            ^SMSFAR69 = RP.Deductible:Expense.ForestryManagedInvestmentScheme.Amount
    
            ^SMSFAR70 = RP.Deductible:Expense.Other.Amount
    
            ^SMSFAR72 = RP.Deductible:IncomeTax.Deduction.TaxLossesDeducted.Amount
            */
            assertion = ((report.SMSFAR321 == false || report.SMSFAR352 == true) && report.SMSFAR73.GetValueOrDefault() != report.SMSFAR33.GetValueOrDefault() + report.SMSFAR34.GetValueOrDefault() + report.SMSFAR35.GetValueOrDefault() + report.SMSFAR36.GetValueOrDefault() + report.SMSFAR38.GetValueOrDefault() + report.SMSFAR39.GetValueOrDefault() + report.SMSFAR40.GetValueOrDefault() + report.SMSFAR42.GetValueOrDefault() + report.SMSFAR43.GetValueOrDefault() + report.SMSFAR44.GetValueOrDefault() + report.SMSFAR45.GetValueOrDefault() + report.SMSFAR46.GetValueOrDefault() + report.SMSFAR47.GetValueOrDefault() + (report.SMSFAR49.GetValueOrDefault() + report.SMSFAR50.GetValueOrDefault() + report.SMSFAR51.GetValueOrDefault() - report.SMSFAR52.GetValueOrDefault()) + (report.SMSFAR53.GetValueOrDefault() + report.SMSFAR55.GetValueOrDefault()) + (report.SMSFAR56.GetValueOrDefault() + report.SMSFAR57.GetValueOrDefault() + report.SMSFAR58.GetValueOrDefault()) - (report.SMSFAR301.GetValueOrDefault() + report.SMSFAR60.GetValueOrDefault() + report.SMSFAR61.GetValueOrDefault() + report.SMSFAR62.GetValueOrDefault() + report.SMSFAR63.GetValueOrDefault() + report.SMSFAR64.GetValueOrDefault() + report.SMSFAR66.GetValueOrDefault() + report.SMSFAR67.GetValueOrDefault() + report.SMSFAR302.GetValueOrDefault() + report.SMSFAR69.GetValueOrDefault() + report.SMSFAR70.GetValueOrDefault() + report.SMSFAR72.GetValueOrDefault()));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.SMSFAR.437177",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Taxable income/loss calculation incorrect",
                    LongDescription = @"If the answer to questions ""Did the fund pay retirement phase superannuation income stream benefits to one or more members in the income year?"" is ""False"" or ""Did the fund have any other income that was assessable?"" is ""True"", then the taxable income or loss must equal the sum of income amounts minus exempt current pension income minus the sum of all deductions.",
                    Location = "/xbrli:xbrl/tns:Income.Taxable.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436105" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR321", Value = GetValueOrEmpty(report.SMSFAR321) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR352", Value = GetValueOrEmpty(report.SMSFAR352) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR73", Value = GetValueOrEmpty(report.SMSFAR73) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR33", Value = GetValueOrEmpty(report.SMSFAR33) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR34", Value = GetValueOrEmpty(report.SMSFAR34) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR35", Value = GetValueOrEmpty(report.SMSFAR35) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR36", Value = GetValueOrEmpty(report.SMSFAR36) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR38", Value = GetValueOrEmpty(report.SMSFAR38) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR39", Value = GetValueOrEmpty(report.SMSFAR39) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR40", Value = GetValueOrEmpty(report.SMSFAR40) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR42", Value = GetValueOrEmpty(report.SMSFAR42) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR43", Value = GetValueOrEmpty(report.SMSFAR43) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR44", Value = GetValueOrEmpty(report.SMSFAR44) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR45", Value = GetValueOrEmpty(report.SMSFAR45) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR46", Value = GetValueOrEmpty(report.SMSFAR46) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR47", Value = GetValueOrEmpty(report.SMSFAR47) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR49", Value = GetValueOrEmpty(report.SMSFAR49) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR50", Value = GetValueOrEmpty(report.SMSFAR50) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR51", Value = GetValueOrEmpty(report.SMSFAR51) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR52", Value = GetValueOrEmpty(report.SMSFAR52) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR53", Value = GetValueOrEmpty(report.SMSFAR53) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR55", Value = GetValueOrEmpty(report.SMSFAR55) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR56", Value = GetValueOrEmpty(report.SMSFAR56) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR57", Value = GetValueOrEmpty(report.SMSFAR57) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR58", Value = GetValueOrEmpty(report.SMSFAR58) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR301", Value = GetValueOrEmpty(report.SMSFAR301) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR60", Value = GetValueOrEmpty(report.SMSFAR60) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR61", Value = GetValueOrEmpty(report.SMSFAR61) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR62", Value = GetValueOrEmpty(report.SMSFAR62) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR63", Value = GetValueOrEmpty(report.SMSFAR63) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR64", Value = GetValueOrEmpty(report.SMSFAR64) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR66", Value = GetValueOrEmpty(report.SMSFAR66) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR67", Value = GetValueOrEmpty(report.SMSFAR67) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR302", Value = GetValueOrEmpty(report.SMSFAR302) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR69", Value = GetValueOrEmpty(report.SMSFAR69) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR70", Value = GetValueOrEmpty(report.SMSFAR70) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR72", Value = GetValueOrEmpty(report.SMSFAR72) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436106
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436106()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436106
            Mandatory field not supplied
    
            Legacy Rule Format:
            IF [SMSFAR73] = NULL
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^SMSFAR73 = NULL
    
            Data Elements:
    
            ^SMSFAR73 = RP:Income.Taxable.Amount
            */
            assertion = (report.SMSFAR73 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Mandatory field not supplied",
                    Location = "/xbrli:xbrl/tns:Income.Taxable.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436106" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR73", Value = GetValueOrEmpty(report.SMSFAR73) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436107
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436107()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436107
            Monetary field may be signed, maximum of 11 chars, no decimals
    
            Legacy Rule Format:
            IF [SMSFAR73] <> NULL AND [SMSFAR73] <> MONETARY(S,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR73, 'S', 11, 0)
    
            Data Elements:
    
            ^SMSFAR73 = RP:Income.Taxable.Amount
            */
            assertion = NotMonetary(report.SMSFAR73, @"S", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.Taxable.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436107" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR73", Value = GetValueOrEmpty(report.SMSFAR73) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436109
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436109()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436109
            If the Taxable income or loss amount is greater than zero, then the Gross tax amount must not be less than 15% of the Taxable income or loss amount
    
            Legacy Rule Format:
            IF [SMSFAR73] > 0 AND [SMSFAR74] <> NULL AND ([SMSFAR74] < ([SMSFAR73] * 0.15) - 1)  
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^SMSFAR73 > 0 AND ^SMSFAR74 <> NULL AND (^SMSFAR74 < (^SMSFAR73 * 0.15) - 1)
    
            Data Elements:
    
            ^SMSFAR74 = RP:IncomeTax.TaxableGross.Amount
    
            ^SMSFAR73 = RP:Income.Taxable.Amount
            */
            assertion = (report.SMSFAR73.GetValueOrDefault() > 0 && report.SMSFAR74 != null && report.SMSFAR74.GetValueOrDefault() < report.SMSFAR73.GetValueOrDefault() * 0.15M - 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.434074",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Gross tax must not be less than 15% of Taxable income",
                    LongDescription = @"If the Taxable income or loss amount is greater than zero, then the Gross tax amount must not be less than 15% of the Taxable income or loss amount",
                    Location = "/xbrli:xbrl/tns:IncomeTax.TaxableGross.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436109" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR73", Value = GetValueOrEmpty(report.SMSFAR73) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR74", Value = GetValueOrEmpty(report.SMSFAR74) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436110
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436110()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436110
            Mandatory field not supplied
    
            Legacy Rule Format:
            IF [SMSFAR74] = NULL
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^SMSFAR74 = NULL
    
            Data Elements:
    
            ^SMSFAR74 = RP:IncomeTax.TaxableGross.Amount
            */
            assertion = (report.SMSFAR74 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Mandatory field not supplied",
                    Location = "/xbrli:xbrl/tns:IncomeTax.TaxableGross.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436110" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR74", Value = GetValueOrEmpty(report.SMSFAR74) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436111
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436111()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436111
            Monetary field - unsigned, maximum of 13 characters, may include a decimal point and up to 2 decimals
    
            Legacy Rule Format:
            IF [SMSFAR74] <> NULL AND [SMSFAR74] <> MONETARY(U,13,2)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR74, 'U', 13, 2)
    
            Data Elements:
    
            ^SMSFAR74 = RP:IncomeTax.TaxableGross.Amount
            */
            assertion = NotMonetary(report.SMSFAR74, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:IncomeTax.TaxableGross.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436111" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR74", Value = GetValueOrEmpty(report.SMSFAR74) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436113
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436113()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436113
            Monetary field - unsigned, maximum of 13 characters, may include a decimal point and up to 2 decimals
    
            Legacy Rule Format:
            IF [SMSFAR75] <> NULL AND [SMSFAR75] <> MONETARY(U,13,2)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR75, 'U', 13, 2)
    
            Data Elements:
    
            ^SMSFAR75 = RP:Income.InternationalDealings.TaxOffset.Amount
            */
            assertion = NotMonetary(report.SMSFAR75, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Income.InternationalDealings.TaxOffset.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436113" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR75", Value = GetValueOrEmpty(report.SMSFAR75) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436114
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436114()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436114
            Monetary field - unsigned, maximum of 13 characters, may include a decimal point and up to 2 decimals
    
            Legacy Rule Format:
            IF [SMSFAR76] <> NULL AND [SMSFAR76] <> MONETARY(U,13,2)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR76, 'U', 13, 2)
    
            Data Elements:
    
            ^SMSFAR76 = RP:IncomeTax.TaxOffsetsAndRebatesTotal.Amount
            */
            assertion = NotMonetary(report.SMSFAR76, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:IncomeTax.TaxOffsetsAndRebatesTotal.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436114" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR76", Value = GetValueOrEmpty(report.SMSFAR76) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436115
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436115()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436115
            Monetary field - unsigned, maximum of 13 characters, may include a decimal point and up to 2 decimals
    
            Legacy Rule Format:
            IF [SMSFAR77] <> NULL AND [SMSFAR77] <> MONETARY(U,13,2)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR77, 'U', 13, 2)
    
            Data Elements:
    
            ^SMSFAR77 = RP:InternationalDealings.AAMInterestSection102.Amount
            */
            assertion = NotMonetary(report.SMSFAR77, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:InternationalDealings.AAMInterestSection102.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436115" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR77", Value = GetValueOrEmpty(report.SMSFAR77) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436119
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436119()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436119
            Monetary field - unsigned, maximum of 13 characters, may include a decimal point and up to 2 decimals
    
            Legacy Rule Format:
            IF [SMSFAR79] <> NULL AND [SMSFAR79] <> MONETARY(U,13,2)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR79, 'U', 13, 2)
    
            Data Elements:
    
            ^SMSFAR79 = RP:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromForeignResidents.Amount
            */
            assertion = NotMonetary(report.SMSFAR79, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromForeignResidents.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436119" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR79", Value = GetValueOrEmpty(report.SMSFAR79) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436120
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436120()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436120
            If 'Gross payments where ABN not quoted' is greater than zero, then 'Credit for tax withheld - where TFN or ABN not quoted' must be present
    
            Legacy Rule Format:
            IF [SMSFAR42] > 0 AND [SMSFAR80] = NULL  
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^SMSFAR42 > 0 AND ^SMSFAR80 = NULL
    
            Data Elements:
    
            ^SMSFAR80 = RP:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldTFNNotQuotedAndABNNotQuoted.Amount
    
            ^SMSFAR42 = RP:Remuneration.ABNNotQuotedPaymentGross.Amount
            */
            assertion = (report.SMSFAR42.GetValueOrDefault() > 0 && report.SMSFAR80 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.SMSFAR.436333",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Credit for tax withheld - where ABN or TFN not quoted must be present",
                    LongDescription = @"If 'Gross payments where ABN not quoted' is greater than zero, then 'Credit for tax withheld - where TFN or ABN not quoted' must be present",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldTFNNotQuotedAndABNNotQuoted.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436120" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR42", Value = GetValueOrEmpty(report.SMSFAR42) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR80", Value = GetValueOrEmpty(report.SMSFAR80) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436121
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436121()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436121
            Monetary field - unsigned, maximum of 13 characters, may include a decimal point and up to 2 decimals
    
            Legacy Rule Format:
            IF [SMSFAR80] <> NULL AND [SMSFAR80] <> MONETARY(U,13,2)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR80, 'U', 13, 2)
    
            Data Elements:
    
            ^SMSFAR80 = RP:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldTFNNotQuotedAndABNNotQuoted.Amount
            */
            assertion = NotMonetary(report.SMSFAR80, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldTFNNotQuotedAndABNNotQuoted.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436121" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR80", Value = GetValueOrEmpty(report.SMSFAR80) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436122
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436122()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436122
            Monetary field - unsigned, maximum of 13 characters, may include a decimal point and up to 2 decimals
    
            Legacy Rule Format:
            IF [SMSFAR81] <> NULL AND [SMSFAR81] <> MONETARY(U,13,2)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR81, 'U', 13, 2)
    
            Data Elements:
    
            ^SMSFAR81 = RP:IncomeTax.FrankingCredits.Refundable.Amount
            */
            assertion = NotMonetary(report.SMSFAR81, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:IncomeTax.FrankingCredits.Refundable.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436122" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR81", Value = GetValueOrEmpty(report.SMSFAR81) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436123
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436123()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436123
            Monetary field - unsigned, maximum of 13 characters, may include a decimal point and up to 2 decimals
    
            Legacy Rule Format:
            IF [SMSFAR82] <> NULL AND [SMSFAR82] <> MONETARY(U,13,2)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR82, 'U', 13, 2)
    
            Data Elements:
    
            ^SMSFAR82 = RP:IncomeTax.LiabilityInstalmentsTotal.Amount
            */
            assertion = NotMonetary(report.SMSFAR82, @"U", 13, 2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:IncomeTax.LiabilityInstalmentsTotal.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436123" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR82", Value = GetValueOrEmpty(report.SMSFAR82) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436124
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436124()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436124
            Supervisory levy must equal $259.
    
            Legacy Rule Format:
            IF [SMSFAR83] <> "259"
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^SMSFAR83 <> 259
    
            Data Elements:
    
            ^SMSFAR83 = RP:IncomeTax.SupervisoryLevy.Amount
            */
            assertion = (report.SMSFAR83.GetValueOrDefault() != 259);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.SMSFAR.437138",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Supervisory levy must equal $259.",
                    Location = "/xbrli:xbrl/tns:IncomeTax.SupervisoryLevy.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436124" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR83", Value = GetValueOrEmpty(report.SMSFAR83) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436126
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436126()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436126
            Mandatory field not supplied
    
            Legacy Rule Format:
            IF [SMSFAR84] = NULL
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^SMSFAR84 = NULL
    
            Data Elements:
    
            ^SMSFAR84 = RP:IncomeTax.PayableOrRefundableTotal.Amount
            */
            assertion = (report.SMSFAR84 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.001001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Mandatory field not supplied",
                    Location = "/xbrli:xbrl/tns:IncomeTax.PayableOrRefundableTotal.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436126" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR84", Value = GetValueOrEmpty(report.SMSFAR84) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436128
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436128()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436128
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [SMSFAR85] <> NULL AND [SMSFAR85] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR85, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR85 = RP.Closing:Tax.Losses.CarriedForward.LaterIncomeYearsTotal.Amount
            */
            assertion = NotMonetary(report.SMSFAR85, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Tax.Losses.CarriedForward.LaterIncomeYearsTotal.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436128" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR85", Value = GetValueOrEmpty(report.SMSFAR85) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436133
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436133(SMSFAR2026.MemberInformation memberInformation, int itemIndex1)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436133
            Date of death must be less than or equal to today's date

            Legacy Rule Format:
            IF [SMSFAR95] > DATE(TODAY)
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^SMSFAR95 > Today()

            Data Elements:

            ^SMSFAR95 = RP:MemberInformation:PersonDemographicDetails.Death.Date
            */
            assertion = (memberInformation.SMSFAR95.GetValueOrDefault() > DateTime.Now.Date);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.SMSFAR.436133",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Date of death must be less than or equal to today's date",
                    LongDescription = @"The date entered is a future date, date of death cannot be in the future.",
                    Location = "/xbrli:xbrl/tns:MemberInformation" + OccurrenceIndex(memberInformation.OccurrenceIndex) + "/tns:PersonDemographicDetails.Death.Date",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436133" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR95", Value = GetValueOrEmptyDate(memberInformation.SMSFAR95) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436134
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436134()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436134
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [SMSFAR117] <> NULL AND [SMSFAR117] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR117, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR117 = RP.JAUS:Assets.Investment.UnitTrustsListed.Amount
            */
            assertion = NotMonetary(report.SMSFAR117, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Assets.Investment.UnitTrustsListed.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436134" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR117", Value = GetValueOrEmpty(report.SMSFAR117) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436135
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436135()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436135
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [SMSFAR118] <> NULL AND [SMSFAR118] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR118, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR118 = RP.JAUS:Assets.Investment.UnitTrustsUnlisted.Amount
            */
            assertion = NotMonetary(report.SMSFAR118, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Assets.Investment.UnitTrustsUnlisted.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436135" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR118", Value = GetValueOrEmpty(report.SMSFAR118) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436136
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436136()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436136
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [SMSFAR119] <> NULL AND [SMSFAR119] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR119, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR119 = RP.Closing.JAUS(Instant):Assets.Investment.SecuritiesAndOrInsurancePoliciesHeldInLifeCompanies.Amount
            */
            assertion = NotMonetary(report.SMSFAR119, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Assets.Investment.SecuritiesAndOrInsurancePoliciesHeldInLifeCompanies.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436136" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR119", Value = GetValueOrEmpty(report.SMSFAR119) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436137
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436137()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436137
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [SMSFAR120] <> NULL AND [SMSFAR120] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR120, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR120 = RP.Closing.JAUS(Instant):Assets.ManagedInvestments.Amount
            */
            assertion = NotMonetary(report.SMSFAR120, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Assets.ManagedInvestments.Amount[@contextRef='" + report.RPClosingJAUSInstantId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436137" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR120", Value = GetValueOrEmpty(report.SMSFAR120) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436138
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436138()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436138
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [SMSFAR121] <> NULL AND [SMSFAR121] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR121, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR121 = RP.Closing.JAUS(Instant):Assets.CashAndLiquidAssets.Total.Amount
            */
            assertion = NotMonetary(report.SMSFAR121, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Assets.CashAndLiquidAssets.Total.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436138" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR121", Value = GetValueOrEmpty(report.SMSFAR121) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436139
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436139()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436139
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [SMSFAR122] <> NULL AND [SMSFAR122] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR122, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR122 = RP.Closing.JAUS(Instant):Assets.DebtSecurities.Amount
            */
            assertion = NotMonetary(report.SMSFAR122, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Assets.DebtSecurities.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436139" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR122", Value = GetValueOrEmpty(report.SMSFAR122) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436140
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436140()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436140
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [SMSFAR123] <> NULL AND [SMSFAR123] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR123, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR123 = RP.Closing.JAUS(Instant):Assets.LoansAndReceivables.Held.Amount
            */
            assertion = NotMonetary(report.SMSFAR123, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Assets.LoansAndReceivables.Held.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436140" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR123", Value = GetValueOrEmpty(report.SMSFAR123) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436141
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436141()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436141
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [SMSFAR124] <> NULL AND [SMSFAR124] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR124, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR124 = RP.JAUS:Assets.Investment.EquitySecuritiesListed.Amount
            */
            assertion = NotMonetary(report.SMSFAR124, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Assets.Investment.EquitySecuritiesListed.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436141" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR124", Value = GetValueOrEmpty(report.SMSFAR124) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436142
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436142()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436142
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [SMSFAR125] <> NULL AND [SMSFAR125] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR125, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR125 = RP.JAUS:Assets.Investment.EquitySecuritiesUnlisted.Amount
            */
            assertion = NotMonetary(report.SMSFAR125, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Assets.Investment.EquitySecuritiesUnlisted.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436142" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR125", Value = GetValueOrEmpty(report.SMSFAR125) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436144
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436144()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436144
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [SMSFAR127] <> NULL AND [SMSFAR127] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR127, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR127 = RP.JAUS:Assets.Investment.PropertyNonResidential.Amount
            */
            assertion = NotMonetary(report.SMSFAR127, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Assets.Investment.PropertyNonResidential.Amount[@contextRef='" + report.RPJAUSId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436144" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR127", Value = GetValueOrEmpty(report.SMSFAR127) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436145
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436145()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436145
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [SMSFAR128] <> NULL AND [SMSFAR128] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR128, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR128 = RP.JAUS:Assets.Investment.PropertyResidential.Amount
            */
            assertion = NotMonetary(report.SMSFAR128, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Assets.Investment.PropertyResidential.Amount[@contextRef='" + report.RPJAUSId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436145" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR128", Value = GetValueOrEmpty(report.SMSFAR128) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436147
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436147()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436147
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [SMSFAR130] <> NULL AND [SMSFAR130] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR130, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR130 = RP.Closing.JAUS(Instant):Assets.Other.Amount
            */
            assertion = NotMonetary(report.SMSFAR130, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Assets.Other.Amount[@contextRef='" + report.RPClosingJAUSInstantId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436147" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR130", Value = GetValueOrEmpty(report.SMSFAR130) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436148
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436148()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436148
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [SMSFAR131] <> NULL AND [SMSFAR131] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR131, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR131 = RP.Closing.JFOREIGN(Instant):Assets.Investment.EquitySecurities.Amount
            */
            assertion = NotMonetary(report.SMSFAR131, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Assets.Investment.EquitySecurities.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436148" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR131", Value = GetValueOrEmpty(report.SMSFAR131) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436149
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436149()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436149
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [SMSFAR132] <> NULL AND [SMSFAR132] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR132, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR132 = RP.JFOREIGN:Assets.Investment.PropertyNonResidential.Amount
            */
            assertion = NotMonetary(report.SMSFAR132, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Assets.Investment.PropertyNonResidential.Amount[@contextRef='" + report.RPJFOREIGNId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436149" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR132", Value = GetValueOrEmpty(report.SMSFAR132) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436150
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436150()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436150
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [SMSFAR133] <> NULL AND [SMSFAR133] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR133, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR133 = RP.JFOREIGN:Assets.Investment.PropertyResidential.Amount
            */
            assertion = NotMonetary(report.SMSFAR133, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Assets.Investment.PropertyResidential.Amount[@contextRef='" + report.RPJFOREIGNId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436150" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR133", Value = GetValueOrEmpty(report.SMSFAR133) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436151
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436151()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436151
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [SMSFAR134] <> NULL AND [SMSFAR134] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR134, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR134 = RP.Closing.JFOREIGN(Instant):Assets.ManagedInvestments.Amount
            */
            assertion = NotMonetary(report.SMSFAR134, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Assets.ManagedInvestments.Amount[@contextRef='" + report.RPClosingJFOREIGNInstantId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436151" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR134", Value = GetValueOrEmpty(report.SMSFAR134) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436152
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436152()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436152
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [SMSFAR135] <> NULL AND [SMSFAR135] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR135, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR135 = RP.Closing.JFOREIGN(Instant):Assets.Other.Amount
            */
            assertion = NotMonetary(report.SMSFAR135, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Assets.Other.Amount[@contextRef='" + report.RPClosingJFOREIGNInstantId + "']",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436152" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR135", Value = GetValueOrEmpty(report.SMSFAR135) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436153
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436153()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436153
            The sum of all Australian and overseas assets amounts must equal the sum of all liabilities amounts
    
            Legacy Rule Format:
            IF (OutsideRange(([SMSFAR117] + [SMSFAR118] + [SMSFAR119] + [SMSFAR120] + [SMSFAR121] + [SMSFAR122] + [SMSFAR123] + [SMSFAR124] + [SMSFAR125] + [SMSFAR311] + [SMSFAR127] + [SMSFAR128] + [SMSFAR312] + [SMSFAR130] + [SMSFAR399] + [SMSFAR131] + [SMSFAR132] + [SMSFAR133] + [SMSFAR134] + [SMSFAR135]), ([SMSFAR136] + [SMSFAR137] + [SMSFAR138]  + [SMSFAR139]),5))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (OutsideRange((^SMSFAR117 + ^SMSFAR118 + ^SMSFAR119 + ^SMSFAR120 + ^SMSFAR121 + ^SMSFAR122 + ^SMSFAR123 + ^SMSFAR124 + ^SMSFAR125 + ^SMSFAR311 + ^SMSFAR127 + ^SMSFAR128 + ^SMSFAR312 + ^SMSFAR130 + ^SMSFAR399 + ^SMSFAR131 + ^SMSFAR132 + ^SMSFAR133 + ^SMSFAR134 + ^SMSFAR135), (^SMSFAR136 + ^SMSFAR137 + ^SMSFAR138  + ^SMSFAR139),5))
    
            Data Elements:
    
            ^SMSFAR136 = RP:Liabilities.Borrowings.Amount
    
            ^SMSFAR117 = RP.JAUS:Assets.Investment.UnitTrustsListed.Amount
    
            ^SMSFAR118 = RP.JAUS:Assets.Investment.UnitTrustsUnlisted.Amount
    
            ^SMSFAR119 = RP.Closing.JAUS(Instant):Assets.Investment.SecuritiesAndOrInsurancePoliciesHeldInLifeCompanies.Amount
    
            ^SMSFAR120 = RP.Closing.JAUS(Instant):Assets.ManagedInvestments.Amount
    
            ^SMSFAR121 = RP.Closing.JAUS(Instant):Assets.CashAndLiquidAssets.Total.Amount
    
            ^SMSFAR122 = RP.Closing.JAUS(Instant):Assets.DebtSecurities.Amount
    
            ^SMSFAR123 = RP.Closing.JAUS(Instant):Assets.LoansAndReceivables.Held.Amount
    
            ^SMSFAR124 = RP.JAUS:Assets.Investment.EquitySecuritiesListed.Amount
    
            ^SMSFAR125 = RP.JAUS:Assets.Investment.EquitySecuritiesUnlisted.Amount
    
            ^SMSFAR311 = RP.JAUS:Assets.LimitedRecourseBorrowing.Amount
    
            ^SMSFAR127 = RP.JAUS:Assets.Investment.PropertyNonResidential.Amount
    
            ^SMSFAR128 = RP.JAUS:Assets.Investment.PropertyResidential.Amount
    
            ^SMSFAR312 = RP.JAUS:Assets.Investment.Collectables.Amount
    
            ^SMSFAR130 = RP.Closing.JAUS(Instant):Assets.Other.Amount
    
            ^SMSFAR399 = RP:Assets.Investment.CryptoCurrency.Amount
    
            ^SMSFAR131 = RP.Closing.JFOREIGN(Instant):Assets.Investment.EquitySecurities.Amount
    
            ^SMSFAR132 = RP.JFOREIGN:Assets.Investment.PropertyNonResidential.Amount
    
            ^SMSFAR133 = RP.JFOREIGN:Assets.Investment.PropertyResidential.Amount
    
            ^SMSFAR134 = RP.Closing.JFOREIGN(Instant):Assets.ManagedInvestments.Amount
    
            ^SMSFAR135 = RP.Closing.JFOREIGN(Instant):Assets.Other.Amount
    
            ^SMSFAR137 = RP.Closing:Equity.MemberAccountBalance.Amount
    
            ^SMSFAR138 = RP.Closing(Instant):Equity.NetAssetsNotAllocatedToMembers.Amount
    
            ^SMSFAR139 = RP(Instant):Liabilities.Other.Amount
            */
            assertion = OutsideRange(report.SMSFAR117.GetValueOrDefault() + report.SMSFAR118.GetValueOrDefault() + report.SMSFAR119.GetValueOrDefault() + report.SMSFAR120.GetValueOrDefault() + report.SMSFAR121.GetValueOrDefault() + report.SMSFAR122.GetValueOrDefault() + report.SMSFAR123.GetValueOrDefault() + report.SMSFAR124.GetValueOrDefault() + report.SMSFAR125.GetValueOrDefault() + report.SMSFAR311.GetValueOrDefault() + report.SMSFAR127.GetValueOrDefault() + report.SMSFAR128.GetValueOrDefault() + report.SMSFAR312.GetValueOrDefault() + report.SMSFAR130.GetValueOrDefault() + report.SMSFAR399.GetValueOrDefault() + report.SMSFAR131.GetValueOrDefault() + report.SMSFAR132.GetValueOrDefault() + report.SMSFAR133.GetValueOrDefault() + report.SMSFAR134.GetValueOrDefault() + report.SMSFAR135.GetValueOrDefault(), report.SMSFAR136.GetValueOrDefault() + report.SMSFAR137.GetValueOrDefault() + report.SMSFAR138.GetValueOrDefault() + report.SMSFAR139.GetValueOrDefault(), 5);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.SMSFAR.436153",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The sum of all Australian and overseas assets amounts must equal the sum of all liabilities amounts",
                    Location = "/xbrli:xbrl/tns:Liabilities.Borrowings.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436153" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR117", Value = GetValueOrEmpty(report.SMSFAR117) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR118", Value = GetValueOrEmpty(report.SMSFAR118) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR119", Value = GetValueOrEmpty(report.SMSFAR119) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR120", Value = GetValueOrEmpty(report.SMSFAR120) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR121", Value = GetValueOrEmpty(report.SMSFAR121) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR122", Value = GetValueOrEmpty(report.SMSFAR122) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR123", Value = GetValueOrEmpty(report.SMSFAR123) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR124", Value = GetValueOrEmpty(report.SMSFAR124) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR125", Value = GetValueOrEmpty(report.SMSFAR125) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR311", Value = GetValueOrEmpty(report.SMSFAR311) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR127", Value = GetValueOrEmpty(report.SMSFAR127) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR128", Value = GetValueOrEmpty(report.SMSFAR128) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR312", Value = GetValueOrEmpty(report.SMSFAR312) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR130", Value = GetValueOrEmpty(report.SMSFAR130) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR399", Value = GetValueOrEmpty(report.SMSFAR399) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR131", Value = GetValueOrEmpty(report.SMSFAR131) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR132", Value = GetValueOrEmpty(report.SMSFAR132) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR133", Value = GetValueOrEmpty(report.SMSFAR133) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR134", Value = GetValueOrEmpty(report.SMSFAR134) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR135", Value = GetValueOrEmpty(report.SMSFAR135) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR136", Value = GetValueOrEmpty(report.SMSFAR136) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR137", Value = GetValueOrEmpty(report.SMSFAR137) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR138", Value = GetValueOrEmpty(report.SMSFAR138) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR139", Value = GetValueOrEmpty(report.SMSFAR139) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436154
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436154()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436154
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [SMSFAR136] <> NULL AND [SMSFAR136] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR136, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR136 = RP:Liabilities.Borrowings.Amount
            */
            assertion = NotMonetary(report.SMSFAR136, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Liabilities.Borrowings.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436154" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR136", Value = GetValueOrEmpty(report.SMSFAR136) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436155
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436155()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436155
            If the answer to the question 'Was the fund wound up during the income year' is 'yes' (true), then the 'Total member closing account balances' amount must not be greater than zero
    
            Legacy Rule Format:
            IF [SMSFAR28] = TRUE AND [SMSFAR137] > 0 
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^SMSFAR28 = TRUE AND ^SMSFAR137 > 0
    
            Data Elements:
    
            ^SMSFAR137 = RP.Closing:Equity.MemberAccountBalance.Amount
    
            ^SMSFAR28 = RP:OrganisationDetails.OperationCease.Indicator
            */
            assertion = (report.SMSFAR28 == true && report.SMSFAR137.GetValueOrDefault() > 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.SMSFAR.436155",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"If the fund was wound up, closing account balances must be zero",
                    LongDescription = @"If the answer to the question 'Was the fund wound up during the income year' is 'yes' (true), then the 'Total member closing account balances' amount must not be greater than zero",
                    Location = "/xbrli:xbrl/tns:Equity.MemberAccountBalance.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436155" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR28", Value = GetValueOrEmpty(report.SMSFAR28) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR137", Value = GetValueOrEmpty(report.SMSFAR137) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436156
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436156()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436156
            The sum of each members closing account balance must be equal to the total member closing account balances.
    
            Legacy Rule Format:
            IF [SMSFAR137] <> (SUM([SMSFAR223]) +/- 50) 
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            OutsideRange(^SMSFAR137, Sum(^SMSFAR223), 50)
    
            Data Elements:
    
            ^SMSFAR137 = RP.Closing:Equity.MemberAccountBalance.Amount
    
            ^SMSFAR223 = RP.Closing:MemberInformation:SuperannuationBenefit.AccountBalance.Amount
            */
            assertion = OutsideRange(report.SMSFAR137.GetValueOrDefault(), (report.MemberInformationCollection == null ? 0 : report.MemberInformationCollection.Sum(f => f.SMSFAR223.GetValueOrDefault())), 50);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.SMSFAR.436156",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total member closing account balances must equal the sum of all members' closing balances",
                    Location = "/xbrli:xbrl/tns:Equity.MemberAccountBalance.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436156" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR137", Value = GetValueOrEmpty(report.SMSFAR137) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR223", Value = "SMSFAR223" });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436157
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436157()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436157
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [SMSFAR137] <> NULL AND [SMSFAR137] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR137, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR137 = RP.Closing:Equity.MemberAccountBalance.Amount
            */
            assertion = NotMonetary(report.SMSFAR137, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Equity.MemberAccountBalance.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436157" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR137", Value = GetValueOrEmpty(report.SMSFAR137) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436158
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436158()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436158
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [SMSFAR138] <> NULL AND [SMSFAR138] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR138, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR138 = RP.Closing(Instant):Equity.NetAssetsNotAllocatedToMembers.Amount
            */
            assertion = NotMonetary(report.SMSFAR138, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Equity.NetAssetsNotAllocatedToMembers.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436158" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR138", Value = GetValueOrEmpty(report.SMSFAR138) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436159
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436159()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436159
            Monetary field must be unsigned, maximum of 11 numeric characters, no decimals
    
            Legacy Rule Format:
            IF [SMSFAR139] <> NULL AND [SMSFAR139] <> MONETARY(U,11,0)
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            NotMonetary(^SMSFAR139, 'U', 11, 0)
    
            Data Elements:
    
            ^SMSFAR139 = RP(Instant):Liabilities.Other.Amount
            */
            assertion = NotMonetary(report.SMSFAR139, @"U", 11, 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.400011",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Field incorrect format. Exceeds maximum digits or decimals and/or contains illegal characters",
                    LongDescription = @"Field is in incorrect format for a monetary field. Field either exceeds maximum digits or decimals and/or contains illegal characters and/or for a signed field does not have the sign as the first character",
                    Location = "/xbrli:xbrl/tns:Liabilities.Other.Amount",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436159" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR139", Value = GetValueOrEmpty(report.SMSFAR139) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436195
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436195()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436195
            Family trust revocation code must be 'R' for revocation or 'V' for variation
    
            Legacy Rule Format:
            IF [SMSFAR163] <> NULL AND [SMSFAR163] <> SET("R","V")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^SMSFAR163 <> NULL AND NotInSet(^SMSFAR163, '"R","V"')
    
            Data Elements:
    
            ^SMSFAR163 = RP:Elections.FamilyTrustElectionRevocation.Code
            */
            assertion = (report.SMSFAR163 != null && !(IsMatch(report.SMSFAR163, @"^(R|V)$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.432413",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Family trust revocation code must be 'R' for revocation or 'V' for variation",
                    Location = "/xbrli:xbrl/tns:Elections.FamilyTrustElectionRevocation.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436195" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR163", Value = GetValueOrEmpty(report.SMSFAR163) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436198
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436198()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436198
            If revoking an interposed entity election, the Revocation code must be 'R'
    
            Legacy Rule Format:
            IF ([SMSFAR165] <> NULL) AND ([SMSFAR165] <> "R")
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^SMSFAR165 <> NULL) AND (^SMSFAR165 <> 'R')
    
            Data Elements:
    
            ^SMSFAR165 = RP:Elections.InterposedEntityElectionRevocation.Code
            */
            assertion = (report.SMSFAR165 != null && report.SMSFAR165 != @"R");
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430179",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"If revoking an interposed entity election, the Revocation code must be 'R'",
                    Location = "/xbrli:xbrl/tns:Elections.InterposedEntityElectionRevocation.Code",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436198" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR165", Value = GetValueOrEmpty(report.SMSFAR165) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436199
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436199()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436199
            There must be one name of the Preferred trustee or director contact
    
            Legacy Rule Format:
            IF COUNT (TUPLE(prsnstrcnm3.xx.xx:PersonNameDetails) IN TUPLE(smsfar.0013.lodge.req.xx.xx:TrusteeOrDirector)) <> 1
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^PersonNameDetails) <> 1
    
            Data Elements:
    
            ^PersonNameDetails = RP:TrusteeOrDirector:PersonNameDetails
            */
            assertion = (Count(report.TrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RPCount) != 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.SMSFAR.436199",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The name of the Preferred trustee or director contact must be present",
                    Location = "/xbrli:xbrl/tns:TrusteeOrDirector/tns:PersonNameDetails" + OccurrenceIndex(report.TrusteeOrDirector_PersonNameDetailsCollection_LGL_Contact_C_RPLastOccurrenceIndex) + "/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436199" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436201
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436201()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436201
            Trustee name must be present if the Trustee ABN is present
    
            Legacy Rule Format:
            IF [SMSFAR175] <> NULLORBLANK AND [SMSFAR174] = NULLORBLANK 
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^SMSFAR175 <> BLANK AND ^SMSFAR174 = BLANK
    
            Data Elements:
    
            ^SMSFAR174 = RP:TrusteeOrDirector:OrganisationNameDetails:OrganisationNameDetails.OrganisationalName.Text WHERE [OrganisationNameDetails.OrganisationalNameType.Code='MN'].[OrganisationNameDetails.Currency.Code='C']
    
            ^SMSFAR175 = RP:TrusteeOrDirector:Identifiers.AustralianBusinessNumber.Identifier
            */
            assertion = (string.IsNullOrWhiteSpace(report.SMSFAR175) != true && string.IsNullOrWhiteSpace(report.SMSFAR174) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.434015",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Trustee name must be present if the Trustee ABN is present",
                    Location = "/xbrli:xbrl/tns:TrusteeOrDirector/tns:OrganisationNameDetails" + OccurrenceIndex(report.TrusteeOrDirector_OrganisationNameDetailsCollection_MN_C_RPOccurrenceIndex) + "/tns:OrganisationNameDetails.OrganisationalName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436201" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR175", Value = GetValueOrEmpty(report.SMSFAR175) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR174", Value = report.SMSFAR174 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436204
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436204()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436204
            The phone number of the intermediary must be present
    
            Legacy Rule Format:
            IF Context(INT) <> NULLORBLANK AND [SMSFAR184] = NULL
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^Context <> BLANK AND ^SMSFAR184 = BLANK
    
            Data Elements:
    
            ^SMSFAR184 = INT:ElectronicContactTelephone:ElectronicContact.Telephone.Minimal.Number WHERE [ElectronicContact.Telephone.Usage.Code='03']
    
            ^Context = INT
            */
            assertion = (report.INTExists != false && string.IsNullOrWhiteSpace(report.SMSFAR184) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.SMSFAR.436204",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The phone number of the intermediary must be present",
                    Location = "/xbrli:xbrl/tns:ElectronicContactTelephone" + OccurrenceIndex(report.ElectronicContactTelephoneCollection_03_INTOccurrenceIndex) + "/tns:ElectronicContact.Telephone.Minimal.Number",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436204" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR184", Value = report.SMSFAR184 });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436205
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436205()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436205
            Each instance of a Member number that is allocated to Member information must be unique. The same number must not be used for more than one member.
    
            Legacy Rule Format:
            IF [SMSFAR195] = ANY OTHER OCCURRENCE OF([SMSFAR195])
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            HasDuplicateValues(^SMSFAR195)
    
            Data Elements:
    
            ^SMSFAR195 = RP:MemberInformation:Report.ItemOrder.Number
            */
            assertion = HasDuplicateValues(report.MemberInformationCollection == null ? null : report.MemberInformationCollection.Select(f => f.SMSFAR195).Cast<object>().ToArray());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.SMSFAR.436205",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member number must be unique",
                    LongDescription = @"Each instance of a Member number that is allocated to Member information must be unique. The same number must not be used for more than one member.",
                    Location = "/xbrli:xbrl/tns:MemberInformation[" + report.MemberInformationCollection[DuplicateValueIndex(report.MemberInformationCollection.Select(f => f.SMSFAR195))].OccurrenceIndex + "]/tns:Report.ItemOrder.Number",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436205" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436208
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436208()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436208
            There must be Members Information for at least one member if the fund is not wound up
    
            Legacy Rule Format:
            IF [SMSFAR28] = FALSE AND COUNT(TUPLE(smsfar.0013.lodge.req.xx.xx:MemberInformation)) = 0 
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^SMSFAR28 = FALSE AND Count(^MemberInformation) = 0
    
            Data Elements:
    
            ^MemberInformation = RP:MemberInformation
    
            ^SMSFAR28 = RP:OrganisationDetails.OperationCease.Indicator
            */
            assertion = (report.SMSFAR28 == false && Count(report.MemberInformationCollectionCount) == 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.SMSFAR.436208",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"There must be Members Information for at least one member if the fund is not wound up",
                    LongDescription = @"If the fund has not been wound up during the income year then there must be the Members information for at least one member",
                    Location = "/xbrli:xbrl/tns:MemberInformation" + OccurrenceIndex(report.MemberInformationCollection == null ? 0 : (report.MemberInformationCollection.Count() == 0 ? 0 : report.MemberInformationCollection.Last().OccurrenceIndex)) + "/tns:Report.ItemOrder.Number",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436208" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR28", Value = GetValueOrEmpty(report.SMSFAR28) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436210
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436210(SMSFAR2026.MemberInformation memberInformation, int itemIndex1)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436210
            The name of the member must be present in the member information statement

            Legacy Rule Format:
            WHERE IN TUPLE(smsfar.0013.lodge.req.xx.xx:MemberInformation)
            IF COUNT(TUPLE(prsnstrcnm3.xx.xx:PersonNameDetails)) <> 1 
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            Count(^PersonNameDetails) <> 1

            Data Elements:

            ^PersonNameDetails = RP:MemberInformation:PersonNameDetails
            */
            assertion = (Count(memberInformation.MemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RPCount) != 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.SMSFAR.436210",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The name of the member must be present in the member information statement",
                    Location = "/xbrli:xbrl/tns:MemberInformation" + OccurrenceIndex(memberInformation.OccurrenceIndex) + "/tns:PersonNameDetails" + OccurrenceIndex(memberInformation.MemberInformation_PersonNameDetailsCollection_LGL_Identifier_C_RPLastOccurrenceIndex) + "/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436210" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436211
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436211()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436211
            A member statement must not contain the same member as another member statement, where a same member is one with the same First given name, Family name, TFN and Date of birth
    
            Legacy Rule Format:
            (WHERE IN TUPLE(smsfar.0013.lodge.req.xx.xx:MemberInformation)
            IF [SMSFAR197] AND [SMSFAR199] AND [SMSFAR202] AND [SMSFAR201]) = (ANY OTHER OCCURRENCE OF([SMSFAR197] AND [SMSFAR199] AND [SMSFAR202] AND [SMSFAR201]) IN TUPLE(smsfar.0013.lodge.req.xx.xx:MemberInformation))
                RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            HasDuplicateValues(^SMSFAR197 + ^SMSFAR199 + ^SMSFAR201 + ^SMSFAR202)
    
            Data Elements:
    
            ^SMSFAR197 = RP:MemberInformation:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Identifier'].[PersonNameDetails.Currency.Code='C']
    
            ^SMSFAR199 = RP:MemberInformation:PersonNameDetails:PersonNameDetails.GivenName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Identifier'].[PersonNameDetails.Currency.Code='C']
    
            ^SMSFAR202 = RP:MemberInformation:PersonDemographicDetails.Birth.Date
    
            ^SMSFAR201 = RP:MemberInformation:Identifiers.TaxFileNumber.Identifier
            */
            assertion = HasDuplicateValues(report.MemberInformationCollection == null ? null : report.MemberInformationCollection.Select(f => f.SMSFAR197 + f.SMSFAR199 + f.SMSFAR201 + f.SMSFAR202).Cast<object>().ToArray());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.SMSFAR.436211",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"A member must not be reported more than once",
                    LongDescription = @"A member statement must not contain the same member as another member statement, where a same member is one with the same First given name, Family name, TFN and Date of birth",
                    Location = "/xbrli:xbrl/tns:MemberInformation[" + report.MemberInformationCollection[DuplicateValueIndex(report.MemberInformationCollection.Select(f => f.SMSFAR197 + f.SMSFAR199 + f.SMSFAR201 + f.SMSFAR202))].OccurrenceIndex + "]/tns:PersonNameDetails/tns:PersonNameDetails.FamilyName.Text",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436211" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436212
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436212()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436212
            Member TFN must not be the same as the Fund TFN
    
            Legacy Rule Format:
            IF ANY OCCURRENCE OF([SMSFAR201]) = RP:entity.identifier.TFN
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            AnyOccurrence(^SMSFAR201, ^SMSFAR201 = ^TFN)
    
            Data Elements:
    
            ^SMSFAR201 = RP:MemberInformation:Identifiers.TaxFileNumber.Identifier
    
            ^TFN = RP
            */
            assertion = (report.MemberInformationCollection == null ? false : report.MemberInformationCollection.Any(SMSFAR201Repeat => SMSFAR201Repeat.SMSFAR201 == report.RPIdentifierTFN));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.SMSFAR.436212",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member TFN must not be the same as the Fund TFN",
                    Location = "/xbrli:xbrl/tns:MemberInformation/tns:Identifiers.TaxFileNumber.Identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436212" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR201", Value = "SMSFAR201" });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436213
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436213()
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436213
            Member TFN must not be reported more than once.
    
            Legacy Rule Format:
            IF ([SMSFAR201] <> NULL) AND ([SMSFAR201]= ANY OTHER OCCURRENCE OF([SMSFAR201]))
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            HasDuplicateValues(^SMSFAR201)
    
            Data Elements:
    
            ^SMSFAR201 = RP:MemberInformation:Identifiers.TaxFileNumber.Identifier
            */
            assertion = HasDuplicateValues(report.MemberInformationCollection == null ? null : report.MemberInformationCollection.Select(f => f.SMSFAR201).Cast<object>().ToArray());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.SMSFAR.436213",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member TFN must not be reported more than once.",
                    Location = "/xbrli:xbrl/tns:MemberInformation[" + report.MemberInformationCollection[DuplicateValueIndex(report.MemberInformationCollection.Select(f => f.SMSFAR201))].OccurrenceIndex + "]/tns:Identifiers.TaxFileNumber.Identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436213" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436214
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436214(SMSFAR2026.MemberInformation memberInformation, int itemIndex1)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436214
            Member TFN must not have less than 9 characters

            Legacy Rule Format:
            IF LENGTH([SMSFAR201]) <  9
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            (^SMSFAR201 <> NULL AND Length(^SMSFAR201) < 9)

            Data Elements:

            ^SMSFAR201 = RP:MemberInformation:Identifiers.TaxFileNumber.Identifier
            */
            assertion = (memberInformation.SMSFAR201 != null && Length(memberInformation.SMSFAR201) < 9);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.SMSFAR.436214",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member TFN must not have less than 9 characters",
                    Location = "/xbrli:xbrl/tns:MemberInformation" + OccurrenceIndex(memberInformation.OccurrenceIndex) + "/tns:Identifiers.TaxFileNumber.Identifier",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.SMSFAR.436214" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "SMSFAR201", Value = GetValueOrEmpty(memberInformation.SMSFAR201) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.SMSFAR.436215
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOSMSFAR436215(SMSFAR2026.MemberInformation memberInformation, int itemIndex1)
        {
            ProcessMessageDocument processMessage;
            bool assertion;

            /*  VR.ATO.SMSFAR.436215
            In each Member information statement the member's date of birth or the member's TFN must be present

            Legacy Rule Format:
            WHERE IN TUPLE(smsfar.0013.lodge.req.xx.xx:MemberInformation)
            IF [SMSFAR197] <> NULLORBLANK AND [SMSFAR202] = NULL AND [SMSFAR201] = NULLORBLANK 
               RETURN VALIDATION MESSAGE
            ENDIF

            Technical Business Rule Format:
            ^SMSFAR197 <> BLANK AND ^SMSFAR202 = NULL AND ^SMSFAR201 = BLANK

            Data Elements:

            ^SMSFAR202 = RP:MemberInformation:PersonDemographicDetails.Birth.Date

            ^SMSFAR197 = RP:MemberInformation:PersonNameDetails:PersonNameDetails.FamilyName.Text WHERE [PersonNameDetails.PersonNameType.Code='LGL'].[PersonNameDetails.Usage.Code='Identifier'].[PersonNameDetails.Currency.Code='C']

            ^SMSFAR201 = RP:MemberInformation:Identifiers.TaxFileNumber.Identifier
            */
            assertion = (string.IsNullOrWhiteSpace(memberInformation.SMSFAR197) != true && memberInformation.SMSFAR202 == null && string.IsNullOrWhiteSpace(memberInformation.SMSFAR201) == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.SMSFAR.436215",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"In each Member information statement the member's date of birth or the member's TFN must be present",
                    Location = "/xbrli:xbrl/tns:MemberInformation" + OccurrenceIndex(memberInformation.OccurrenceIndex) + "/tns:PersonDemographicDetails.Birth.Date",
                    Par