using System;
using System.Xml;
using System.IO;
using System.Collections.Generic;
using DataContracts;
using System.Text.RegularExpressions;
using System.Reflection;
using System.Linq;
using System.Xml.Schema;
using System.Text;

namespace Ato.EN.IntegrationServices.CodeGenerationTRT
{
    /// <summary>
    /// XML Consumer for TRT
    /// </summary>
    public class TRT2026XmlConsumer
    {
        const string MissingMandatoryMessage = "A mandatory field has not been completed.";
        const string ContactYourProviderMsg = "The message did not pass XML validation. Please contact your software provider.";
        const string InvalidDataValidation = "A field contains invalid data (such as letters in numeric or date field).";
        private const string ErrorCode1 = "CMN.ATO.GEN.XML01";
        private const string ErrorCode3 = "CMN.ATO.GEN.XML03";
        private const string ErrorCode4 = "CMN.ATO.GEN.XML04";
        private const string ErrorCode6 = "CMN.ATO.GEN.XML06";
        private bool _found = false;
        private int _parentCollectionCount;
        private int _childCollectionCount;
        private bool _isExiting = false;
        private bool _isValidationError = false;
        private string _lastPath = string.Empty;
        private ErrorMessageType _validationError = new ErrorMessageType(ErrorDescriptor.NoError);
#region Error Messages
        public List<ProcessMessageDocument> ErrorMessages { get; set; }

        public bool HasErrors
        {
            get
            {
                return ErrorMessages != null && ErrorMessages.Count > 0;
            }
        }

        private string GetCurrentLocation()
        {
            StringBuilder location = new StringBuilder();
            string[] paths = _currentXPath.ToArray<string>();
            for (int i = paths.Length - 1; i > -1; i--)
            {
                location.Append(paths[i]);
            }

            return location.ToString();
        }

        private void MissingElementError()
        {
            ProcessMessageDocument processMessage;
            processMessage = new ProcessMessageDocument()
            {
                Code = "CMN.ATO.GEN.XML04",
                Description = MissingMandatoryMessage,
                SeverityAsString = "Error",
                Location = GetCurrentLocation(),
            };
            this.ErrorMessages.Add(processMessage);
        }

#endregion Error Messages
#region Embedded Schema
        private static readonly XmlSchemaSet EmbeddedXmlSchemaSet;
        static TRT2026XmlConsumer()
        {
            Assembly executingAssembly = Assembly.GetExecutingAssembly();
            string[] manifestResourceNames = executingAssembly.GetManifestResourceNames();
            string resourceName;
            Stream embeddedSchemaStream;
            XmlSchema embeddedXmlSchema;
            if (manifestResourceNames != null)
            {
                resourceName = manifestResourceNames.FirstOrDefault(rn => rn.Contains("ato.trt.0014.2026.01.00.xsd") && rn.EndsWith(".xsd"));
                if (resourceName != null)
                {
                    embeddedSchemaStream = executingAssembly.GetManifestResourceStream(resourceName);
                    embeddedXmlSchema = XmlSchema.Read(embeddedSchemaStream, SchemaCallback);
                    EmbeddedXmlSchemaSet = new XmlSchemaSet();
                    EmbeddedXmlSchemaSet.Add(embeddedXmlSchema);
                    EmbeddedXmlSchemaSet.Compile();
                }
            }
        }

        private static void SchemaCallback(object sender, ValidationEventArgs args)
        {
            if (args.Severity == XmlSeverityType.Error)
                throw new XmlSchemaException(args.Message);
        }

#endregion Embedded Schema
#region Xml Reader Settings
        private static XmlReaderSettings ReaderSettings = new XmlReaderSettings()
        {
            CloseInput = false,
            ConformanceLevel = ConformanceLevel.Fragment,
            IgnoreWhitespace = true,
            IgnoreComments = true,
            IgnoreProcessingInstructions = true,
        };
        private XmlReaderSettings GetValidatingReaderSettings()
        {
            XmlReaderSettings validatingReaderSettings = new XmlReaderSettings()
            {
                CloseInput = false,
                ConformanceLevel = ConformanceLevel.Fragment,
                IgnoreWhitespace = true,
                IgnoreComments = true,
                IgnoreProcessingInstructions = true,
                ValidationType = ValidationType.Schema,
            };
            if (EmbeddedXmlSchemaSet == null)
            {
                throw new XmlSchemaException("Embedded Schema is Null");
            }
            else
            {
                validatingReaderSettings.Schemas.Add(EmbeddedXmlSchemaSet);
                validatingReaderSettings.ValidationEventHandler += new ValidationEventHandler(ValidationCallBack);
            }

            return validatingReaderSettings;
        }

        private static Regex DataTypeFailureExpression = new Regex("'(?<Uniqueid>.*?)' element is invalid.*value '(?<Value>.*?)' is invalid.*datatype '(?<DataType>.*?)'", RegexOptions.IgnoreCase | RegexOptions.Compiled | RegexOptions.Singleline);
        private void ValidationCallBack(object sender, ValidationEventArgs args)
        {
            var msg = args.Message;
            var msgToLower = msg.ToLower();
            ProcessMessageDocument processMessage = null;
            if (msgToLower.Contains("invalid according to its datatype"))
            {
                Match match = DataTypeFailureExpression.Match(args.Message);
                if (match.Success && match.Groups.Count == 4)
                {
                    var inner = args.Exception.InnerException;
                    var hint = (inner != null && !string.IsNullOrEmpty(inner.Message)) ? " Hint: " + inner.Message : string.Empty;
                    processMessage = BuildProcessMessageDocument(InvalidDataValidation, string.Empty, ErrorCode3);
                    string uniqueID = match.Groups["Uniqueid"].Value;
                    string value = match.Groups["Value"].Value;
                    string dataType = match.Groups["DataType"].Value;
                    var longDescription = string.Format(@"The value specified for an item does not match the item type (value = ""{0}"", item type = {1}, uniqueID = {2})", value, dataType, uniqueID);
                    processMessage.LongDescription = longDescription + hint;
                    processMessage.Parameters = new ProcessMessageParameters();
                    processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "uniqueID", Value = uniqueID });
                    processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "value", Value = value });
                    processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "dataType", Value = dataType });
                }
            }
            else if (msgToLower.Contains("has incomplete content"))
            {
                _validationError = GetErrorDescriptorForHasIncompleteContent(msg);
                processMessage = BuildProcessMessageDocument(MissingMandatoryMessage, msg + _validationError.Hint, _validationError.Code);
                _isValidationError = true;
            }
            else if (msgToLower.Contains("has invalid child element"))
            {
                _validationError = GetErrorDescriptorForHasInvalidChildElement(msg);
                var shortMessage = _validationError.Code == ErrorCode4 ? MissingMandatoryMessage : ContactYourProviderMsg;
                processMessage = BuildProcessMessageDocument(shortMessage, msg + _validationError.Hint, _validationError.Code);
                _isValidationError = true;
            }
            else
            {
                processMessage = BuildProcessMessageDocument(ContactYourProviderMsg, msg, ErrorCode1);
                _isValidationError = true;
            }

            this.ErrorMessages.Add(processMessage);
        }

        private ProcessMessageDocument BuildProcessMessageDocument(string shortDescription, string longDescription, string code)
        {
            return new ProcessMessageDocument()
            {
                Code = code,
                Description = shortDescription,
                LongDescription = longDescription,
                SeverityAsString = "Error",
                Location = GetCurrentLocation()
            };
        }

        private ProcessMessageDocument BuildProcessMessageDocument(ProcessMessageDocument processMessage, string newLocation)
        {
            return new ProcessMessageDocument()
            {
                Code = processMessage.Code,
                Description = processMessage.Description,
                LongDescription = processMessage.LongDescription,
                SeverityAsString = "Error",
                Location = newLocation
            };
        }

        private enum ErrorDescriptor
        {
            NoError = 0,
            MandatoryElementError,
            NonMandatoryElementError,
            UnknownElement,
            Duplicate
        }

        private struct ErrorMessageType
        {
            public string Hint, Code, ExpectedNextValidElement;
            public ErrorDescriptor Errno;
            public ErrorMessageType(ErrorDescriptor errorDescriptor)
            {
                Errno = errorDescriptor;
                Hint = "";
                Code = "";
                ExpectedNextValidElement = "";
            }

            public ErrorMessageType(ErrorDescriptor errorDescriptor, string hint, string code, string elementName)
            {
                Errno = errorDescriptor;
                Hint = hint;
                Code = code;
                ExpectedNextValidElement = elementName;
            }
        }

        private ErrorMessageType GetErrorDescriptorForHasIncompleteContent(string validationErrorMessage)
        {
            ErrorMessageType retval = new ErrorMessageType();
            if (string.IsNullOrWhiteSpace(validationErrorMessage))
                return retval;
            validationErrorMessage = validationErrorMessage.Replace(" ", string.Empty);
            validationErrorMessage = validationErrorMessage.Replace("'", " ");
            var parts = validationErrorMessage.Split(' ');
            if (parts.Length != 9)
                return retval;
            var elementBeingProcessed = parts[1];
            var csvList = parts[5];
            retval = ParseHasIncompleteContentCVSList(csvList, elementBeingProcessed);
            return retval;
        }

        private ErrorMessageType GetErrorDescriptorForHasInvalidChildElement(string validationErrorMessage)
        {
            string msg = string.Empty;
            ErrorMessageType retval = new ErrorMessageType();
            if (string.IsNullOrWhiteSpace(validationErrorMessage))
                return retval;
            validationErrorMessage = validationErrorMessage.Replace(" ", string.Empty);
            validationErrorMessage = validationErrorMessage.Replace("'", " ");
            var parts = validationErrorMessage.Split(' ');
            //decide which message format to process
            if (validationErrorMessage.ToLower().Contains("listofpossibleelements"))
            {
                if (parts.Length == 13 || parts.Length == 12 || parts.Length == 10)
                {
                    var csvList = parts[9];
                    var elementBeingProcessed = parts[1];
                    var invalidChildElementName = parts[5];
                    retval = ParseCvsListForHasInvalidChildElementLongFormat(csvList, invalidChildElementName, elementBeingProcessed);
                }

                return retval;
            }
            else
            {
                if (parts.Length != 9)
                    return retval;
                var csvList = parts[5];
                var elementBeingProcessed = parts[1];
                var invalidChildElementName = parts[5];
                retval = ParseCvsListForHasInvalidChildElementLongFormat(csvList, invalidChildElementName, elementBeingProcessed);
            }

            return retval;
        }

        private ErrorMessageType ParseCvsListForHasInvalidChildElementShortFormat(string csvList, string invalidChildElementName, string elementBeingProcessed)
        {
            ErrorMessageType retval = new ErrorMessageType();
            if (!string.IsNullOrEmpty(csvList))
            {
                var lastElementInListOfPossibleElements = GetLastElementInCsvList(csvList);
                retval.Hint = string.Format(" Hint: while processing parent element [{0}] child element [{1}] was unexpected", elementBeingProcessed, invalidChildElementName);
                retval.Code = ErrorCode1;
                retval.Errno = ErrorDescriptor.UnknownElement;
                retval.ExpectedNextValidElement = invalidChildElementName;
            }

            return retval;
        }

        private string GetLastElementInCsvList(string csvList)
        {
            var list = (csvList.Contains(',')) ? new List<string>(csvList.Split(',')) : new List<string>
            {
                csvList
            };
            string lastElement = list.Last();
            if (lastElement.EndsWith("...."))
            {
                list.RemoveAt(list.Count - 1);
            }

            return list.Last();
        }

        private ErrorMessageType ParseCvsListForHasInvalidChildElementLongFormat(string csvList, string invalidChildElementName, string elementBeingProcessed)
        {
            ErrorMessageType retval = new ErrorMessageType();
            if (!string.IsNullOrEmpty(csvList))
            {
                var lastElementInListOfPossibleElements = GetLastElementInCsvList(csvList);
                var elementList = new List<string>(_elementCsvList.Split(','));
                if (ContainsElementForElementBeingProcessed(elementList, elementBeingProcessed, invalidChildElementName))
                {
                    var indexOfElementBeingProcessed = GetIndexOfElementBeingProcessed(elementList, elementBeingProcessed);
                    var indexOfinvalidChildElementName = GetIndexOfElementUnderElementBeingProcessed(elementList, elementBeingProcessed, invalidChildElementName);
                    var indexOflastElementInListOfPossibleElements = GetIndexOfElementUnderElementBeingProcessed(elementList, elementBeingProcessed, lastElementInListOfPossibleElements);
                    if (indexOfinvalidChildElementName > indexOflastElementInListOfPossibleElements)
                    {
                        retval.Hint = string.Format(" Hint: while processing parent element [{0}] child mandatory element [{1}] was expected but not found", elementBeingProcessed, lastElementInListOfPossibleElements);
                        retval.Code = ErrorCode4;
                        retval.Errno = ErrorDescriptor.MandatoryElementError;
                        retval.ExpectedNextValidElement = lastElementInListOfPossibleElements;
                    }
                    else
                    {
                        retval.Hint = string.Format(" Hint: while processing parent element [{0}] child non mandatory element [{1}] was unexpected", elementBeingProcessed, invalidChildElementName);
                        retval.Code = ErrorCode1;
                        retval.Errno = (indexOfinvalidChildElementName != indexOflastElementInListOfPossibleElements) ? ErrorDescriptor.NonMandatoryElementError : ErrorDescriptor.Duplicate;
                        retval.ExpectedNextValidElement = invalidChildElementName;
                    }
                }
                else
                {
                    //this element is unknown to the schema.
                    retval.Hint = string.Format(" Hint: while processing parent element [{0}] child element [{1}] was unexpected", elementBeingProcessed, invalidChildElementName);
                    retval.Code = ErrorCode1;
                    retval.Errno = ErrorDescriptor.UnknownElement;
                    retval.ExpectedNextValidElement = invalidChildElementName;
                }
            }

            return retval;
        }

        private int GetIndexOfElementUnderElementBeingProcessed(List<string> validElementList, string elementBeingProcessed, string elementName)
        {
            int result;
            string elementWithParentNameMatch = elementBeingProcessed + "#" + elementName;
            result = validElementList.IndexOf(elementWithParentNameMatch);
            if (result < 0)
            {
                result = validElementList.IndexOf(elementName);
            }

            return result;
        }

        private int GetIndexOfElementBeingProcessed(List<string> validElementList, string elementBeingProcessed)
        {
            return validElementList.IndexOf(elementBeingProcessed);
        }

        private bool ContainsElementForElementBeingProcessed(List<string> validElementList, string elementBeingProcessed, string elementName)
        {
            return validElementList.Contains(elementBeingProcessed + "#" + elementName) || validElementList.Contains(elementName);
        }

        private ErrorMessageType ParseHasIncompleteContentCVSList(string csvList, string elementBeingProcessed)
        {
            ErrorMessageType retval = new ErrorMessageType();
            if (!string.IsNullOrEmpty(csvList))
            {
                var lastElementInListOfPossibleElements = GetLastElementInCsvList(csvList);
                retval.Hint = string.Format(" Hint: while processing parent element [{0}] child mandatory element [{1}] was not found", elementBeingProcessed, lastElementInListOfPossibleElements);
                retval.Code = ErrorCode4;
                retval.Errno = ErrorDescriptor.MandatoryElementError;
                retval.ExpectedNextValidElement = lastElementInListOfPossibleElements;
            }

            return retval;
        }

        private void StartEndDateError(DateTime startDate, DateTime endDate)
        {
            ProcessMessageDocument processMessage;
            string longDescription = string.Format("End date is earlier than start date (state date = {0}, end date = {1})", startDate.ToString("yyyy-MM-dd"), endDate.ToString("yyyy-MM-dd"));
            processMessage = new ProcessMessageDocument()
            {
                Code = "CMN.ATO.GEN.XML06",
                Description = "End date is earlier than start date.",
                LongDescription = longDescription,
                SeverityAsString = "Error",
                Location = GetCurrentLocation(),
            };
            processMessage.Parameters = new ProcessMessageParameters();
            processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "startDate", Value = startDate.ToString("yyyy-MM-dd") });
            processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "endDate", Value = endDate.ToString("yyyy-MM-dd") });
            this.ErrorMessages.Add(processMessage);
        }

        private void StartEndDateError(DateTime? startDate, DateTime? endDate)
        {
            ProcessMessageDocument processMessage;
            if (startDate == null || endDate == null)
                return;
            string longDescription = string.Format("End date is earlier than start date (state date = {0}, end date = {1})", startDate.Value.ToString("yyyy-MM-dd"), endDate.Value.ToString("yyyy-MM-dd"));
            processMessage = new ProcessMessageDocument()
            {
                Code = "CMN.ATO.GEN.XML06",
                Description = "End date is earlier than start date.",
                LongDescription = longDescription,
                SeverityAsString = "Error",
                Location = GetCurrentLocation(),
            };
            processMessage.Parameters = new ProcessMessageParameters();
            processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "startDate", Value = startDate.Value.ToString("yyyy-MM-dd") });
            processMessage.Parameters.Add(new ProcessMessageParameter() { Name = "endDate", Value = endDate.Value.ToString("yyyy-MM-dd") });
            this.ErrorMessages.Add(processMessage);
        }

#endregion Xml Reader Settings
#region IsEmptyOrNilElement
        private static bool IsEmptyOrNilElement(XmlReader reader)
        {
            bool emptyOrNil = false;
            if (reader.IsEmptyElement)
            {
                emptyOrNil = true;
            }
            else
            {
                if (reader.HasAttributes)
                {
                    string nilValue = reader.GetAttribute("nil", "http://www.w3.org/2001/XMLSchema-instance");
                    if (nilValue != null && nilValue.ToLowerInvariant() == "true" || nilValue == "1")
                    {
                        emptyOrNil = true;
                    }
                }
            }

            return emptyOrNil;
        }

#endregion IsEmptyOrNilElement
#region ToBoolean
        private bool? ToBoolean(string str)
        {
            bool returnValue;
            string value = (str ?? "").Trim();
            if (value == "0" || value == "1")
                return Convert.ToBoolean(Convert.ToInt32(value));
            if (Boolean.TryParse(value, out returnValue))
                return returnValue;
            return null;
        }

#endregion ToBoolean
#region MoveToContent
        public bool MoveToContent(XmlReader reader)
        {
            try
            {
                reader.MoveToContent();
                return true;
            }
            catch (XmlException ex)
            {
                ProcessMessageDocument processMessage;
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.XML01",
                    Description = ContactYourProviderMsg,
                    LongDescription = ex.Message,
                    SeverityAsString = "Error",
                };
                this.ErrorMessages.Add(processMessage);
                return false;
            }
        }

#endregion MoveToContent
#region ReadToNextElement
        public bool ReadToNextElement(XmlReader reader, bool isMandatory)
        {
            return ReadToNextElement(reader);
        }

        public bool ReadToNextElement(XmlReader reader)
        {
            bool retval = false;
            if ((reader.EOF && !_isValidationError) || _isExiting)
            {
                return false; //do nothing
            }

            if (!_isValidationError)
            {
                retval = SetReaderToNextElement(reader); //position the reader on the next valid element
            }

            if (_isValidationError)
            {
                //process validation errors
                var expectedNextElement = (!string.IsNullOrEmpty(_validationError.ExpectedNextValidElement)) ? _validationError.ExpectedNextValidElement : reader.LocalName;
                var actualNextValidElement = reader.LocalName;
                var location = GetCurrentLocation();
                if (_validationError.Errno == ErrorDescriptor.Duplicate)
                {
                    location = _lastPath;
                }

                var xpath = RemoveLeadingAndTrailingSlashes(location);
                var element = GetElementDescriptor(xpath);
                if (_validationError.Errno == ErrorDescriptor.UnknownElement)
                {
                    //set the xpath for elements that are not known to the schema
                    location = "/" + element.Path + "/" + element.Prefix + ":" + _validationError.ExpectedNextValidElement;
                    _isExiting = true;
                }
                else if (_validationError.Errno == ErrorDescriptor.NonMandatoryElementError)
                {
                    //set the xpath for the optional element that caused a validation
                    location = "/" + element.Path + "/" + element.Prefix + ":" + reader.LocalName;
                    _isExiting = true;
                }
                else
                {
                    //decide if we have spooled to the missing mandatory element. 
                    _isExiting = expectedNextElement == element.PathLeafNode;
                }

                if (_isExiting)
                {
                    //The location has changed since detection of the validation error - so update it.
                    UpdateErrorMessageWithLocation(location);
                }

                return false;
            }

            //Save previous path so we can detect duplicate elements
            _lastPath = GetCurrentLocation();
            return retval;
        }

        private bool ReadNext(XmlReader reader)
        {
            return !reader.EOF && reader.Read();
        }

        private void UpdateErrorMessageWithLocation(string newLocation)
        {
            var lastProcessMessage = ErrorMessages.Last();
            ErrorMessages.Remove(lastProcessMessage);
            var newProcessMessage = BuildProcessMessageDocument(lastProcessMessage, newLocation);
            ErrorMessages.Add(newProcessMessage);
        }

        private string RemoveLeadingAndTrailingSlashes(string xpath)
        {
            var xpathLength = xpath.Length;
            if (xpathLength <= 0)
                return string.Empty;
            var firstChar = xpath.Substring(0, 1);
            var lastChar = xpath.Substring(xpathLength - 1, 1);
            if (firstChar == "/")
            {
                xpath = xpath.Remove(0, 1);
                xpathLength--;
            }

            if (lastChar == "/")
            {
                xpath = xpath.Remove(xpathLength - 1, 1);
            }

            return xpath;
        }

        private struct ElementDescriptor
        {
            public string Prefix, PathLeafNode, Path;
            public ElementDescriptor(string prefix, string leaf, string path)
            {
                Prefix = prefix;
                PathLeafNode = leaf;
                Path = path;
            }
        }

        private ElementDescriptor GetElementDescriptor(string xpath)
        {
            var retval = new ElementDescriptor();
            var parts = xpath.Split('/');
            var numberOfParts = parts.Length;
            if (numberOfParts > 0)
            {
                var leafNode = parts[numberOfParts - 1];
                var p = leafNode.Split(':');
                numberOfParts = p.Length;
                if (numberOfParts == 1)
                {
                    retval.PathLeafNode = p[0];
                    retval.Prefix = string.Empty;
                }
                else if (numberOfParts == 2)
                {
                    retval.PathLeafNode = p[1];
                    retval.Prefix = p[0];
                }

                var leafNodeLength = leafNode.Length;
                var path = xpath.Remove(xpath.Length - leafNodeLength, leafNodeLength);
                retval.Path = RemoveLeadingAndTrailingSlashes(path);
            }

            return retval;
        }

        private bool SetReaderToNextElement(XmlReader reader)
        {
            try
            {
                if (reader.EOF || _isExiting)
                    return false;
                if (_found)
                {
                    reader.Read();
                    _found = false;
                }

                while (!reader.EOF && reader.NodeType != XmlNodeType.Element)
                {
                    reader.Read();
                }

                return reader.NodeType == XmlNodeType.Element && !reader.EOF;
            }
            catch (XmlException ex)
            {
                ProcessMessageDocument processMessage;
                processMessage = new ProcessMessageDocument()
                {
                    Code = ErrorCode1,
                    Description = ContactYourProviderMsg,
                    LongDescription = ex.Message,
                    SeverityAsString = "Error",
                };
                this.ErrorMessages.Add(processMessage);
                return false;
            }
        }

#endregion ReadToNextElement
        private Stack<string> _currentXPath = new Stack<string>();
        public TRT2026 Consume(Stream streamToLoad, bool validateDataTypes = false)
        {
            TRT2026 report = new TRT2026();
            // Working Variable for if can still read from the xml stream
            bool reading;
            // Working Variables for current values
            string currentValue;
            DateTime currentDateTimeValue;
            bool? currentBooleanValue;
            decimal currentDecimalValue;
            double currentDoubleValue;
            float currentFloatValue;
            sbyte currentsByteValue;
            byte currentByteValue;
            short currentShortValue;
            ushort currentuShortValue;
            uint currentuIntValue;
            int currentIntValue;
            long currentLongValue;
            ulong currentuLongValue;
            this.ErrorMessages = new List<ProcessMessageDocument>();
            _currentXPath.Push("/tns:TRT");
            if (streamToLoad == null || streamToLoad.Length == 0 || !streamToLoad.CanRead)
            {
                MissingElementError();
                return report;
            }

            streamToLoad.Position = 0;
            XmlReader reader;
            if (validateDataTypes)
                reader = XmlReader.Create(streamToLoad, GetValidatingReaderSettings());
            else
                reader = XmlReader.Create(streamToLoad, ReaderSettings);
            if (!MoveToContent(reader))
                return report;
            reading = !reader.EOF;
            if (ReadToNextElement(reader) && reader.LocalName == "TRT" && reader.NamespaceURI == "http://www.sbr.gov.au/ato/trt")
            {
                _found = true;
                ReadToNextElement(reader);
            }
            else
            {
                MissingElementError();
                return report;
            }

#region RP
            _currentXPath.Push("/tns:RP");
            //3. use case
            if (ReadToNextElement(reader, true) && reader.LocalName == "RP" && reader.Depth == _currentXPath.Count - 1)
            {
                report.RpCollectionExists = true;
                report.RpCollectionCount += 1;
                _found = true;
#region TargetFinancialY
                _currentXPath.Push("/tns:TargetFinancialY");
                //6. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "TargetFinancialY" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Int32.TryParse(currentValue, out currentIntValue))
                            {
                                report.TRT1 = currentIntValue;
                            }
                        }
                    }

                    _found = true;
                }

                _currentXPath.Pop();
#endregion TargetFinancialY
#region PeriodStartD
                _currentXPath.Push("/tns:PeriodStartD");
                //6. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "PeriodStartD" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                            {
                                report.TRT573 = currentDateTimeValue.Date;
                            }
                        }
                    }

                    _found = true;
                }

                _currentXPath.Pop();
#endregion PeriodStartD
#region PeriodEndD
                _currentXPath.Push("/tns:PeriodEndD");
                //6. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "PeriodEndD" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                            {
                                report.TRT574 = currentDateTimeValue.Date;
                            }
                        }
                    }

                    _found = true;
                }

                _currentXPath.Pop();
#endregion PeriodEndD
#region TaxFileNumberId
                _currentXPath.Push("/tns:TaxFileNumberId");
                //6. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "TaxFileNumberId" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.TRT5 = currentValue;
                        }
                    }

                    _found = true;
                }

                _currentXPath.Pop();
#endregion TaxFileNumberId
#region AustralianBusinessNumberId
                _currentXPath.Push("/tns:AustralianBusinessNumberId");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "AustralianBusinessNumberId" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.TRT7 = currentValue;
                        }
                    }

                    _found = true;
                }

                _currentXPath.Pop();
#endregion AustralianBusinessNumberId
#region ReportAmendment
                _currentXPath.Push("/tns:ReportAmendment");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "ReportAmendment" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_ReportAmendmentCollectionExists = true;
                    report.RP_ReportAmendmentCollectionCount += 1;
                    _found = true;
#region I
                    _currentXPath.Push("/tns:I");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "I" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.TRT414 = currentBooleanValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion I
#region AmendmentTypeC
                    _currentXPath.Push("/tns:AmendmentTypeC");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "AmendmentTypeC" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRT415 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion AmendmentTypeC
#region AmendmentReasonT
                    _currentXPath.Push("/tns:AmendmentReasonT");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "AmendmentReasonT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRT416 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion AmendmentReasonT
#region AmendmentSequenceN
                    _currentXPath.Push("/tns:AmendmentSequenceN");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "AmendmentSequenceN" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRT417 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion AmendmentSequenceN
                } // End of if ReportAmendment node exists

                _currentXPath.Pop();
#endregion ReportAmendment
#region CurrentTrust
                _currentXPath.Push("/tns:CurrentTrust");
                //3. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "CurrentTrust" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_CurrentTrustCollectionExists = true;
                    report.RP_CurrentTrustCollectionCount += 1;
                    _found = true;
#region OrganisationNameDetailsOrganisationalNameT
                    _currentXPath.Push("/tns:OrganisationNameDetailsOrganisationalNameT");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "OrganisationNameDetailsOrganisationalNameT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRT6 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion OrganisationNameDetailsOrganisationalNameT
#region CurrentPostalAddressDetails
                    _currentXPath.Push("/tns:CurrentPostalAddressDetails");
                    //3. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "CurrentPostalAddressDetails" && reader.Depth == _currentXPath.Count - 1)
                    {
                        report.RP_CurrentTrust_CurrentPostalAddressDetailsCollectionExists = true;
                        report.RP_CurrentTrust_CurrentPostalAddressDetailsCollectionCount += 1;
                        _found = true;
#region OverseasAddressI
                        _currentXPath.Push("/tns:OverseasAddressI");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "OverseasAddressI" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    currentBooleanValue = ToBoolean(currentValue);
                                    if (currentBooleanValue != null)
                                    {
                                        report.TRT315 = currentBooleanValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion OverseasAddressI
#region Line1T
                        _currentXPath.Push("/tns:Line1T");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "Line1T" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.TRT9 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion Line1T
#region Line2T
                        _currentXPath.Push("/tns:Line2T");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "Line2T" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.TRT306 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion Line2T
#region LocalityNameT
                        _currentXPath.Push("/tns:LocalityNameT");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "LocalityNameT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.TRT10 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion LocalityNameT
#region PostcodeT
                        _currentXPath.Push("/tns:PostcodeT");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "PostcodeT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.TRT12 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion PostcodeT
#region StateOrTerritoryC
                        _currentXPath.Push("/tns:StateOrTerritoryC");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "StateOrTerritoryC" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.TRT11 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion StateOrTerritoryC
#region CountryC
                        _currentXPath.Push("/tns:CountryC");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "CountryC" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.TRT367 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion CountryC
                    } // End of if CurrentPostalAddressDetails node exists

                    _currentXPath.Pop();
#endregion CurrentPostalAddressDetails
                } // End of if CurrentTrust node exists

                _currentXPath.Pop();
#endregion CurrentTrust
#region PreviousTrust
                _currentXPath.Push("/tns:PreviousTrust");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "PreviousTrust" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_PreviousTrustCollectionExists = true;
                    report.RP_PreviousTrustCollectionCount += 1;
                    _found = true;
#region OrganisationNameDetailsOrganisationalNameT
                    _currentXPath.Push("/tns:OrganisationNameDetailsOrganisationalNameT");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "OrganisationNameDetailsOrganisationalNameT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRT8 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion OrganisationNameDetailsOrganisationalNameT
#region PreviousPostalAddressDetails
                    _currentXPath.Push("/tns:PreviousPostalAddressDetails");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "PreviousPostalAddressDetails" && reader.Depth == _currentXPath.Count - 1)
                    {
                        report.RP_PreviousTrust_PreviousPostalAddressDetailsCollectionExists = true;
                        report.RP_PreviousTrust_PreviousPostalAddressDetailsCollectionCount += 1;
                        _found = true;
#region OverseasAddressI
                        _currentXPath.Push("/tns:OverseasAddressI");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "OverseasAddressI" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    currentBooleanValue = ToBoolean(currentValue);
                                    if (currentBooleanValue != null)
                                    {
                                        report.TRT317 = currentBooleanValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion OverseasAddressI
#region Line1T
                        _currentXPath.Push("/tns:Line1T");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "Line1T" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.TRT14 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion Line1T
#region Line2T
                        _currentXPath.Push("/tns:Line2T");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "Line2T" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.TRT312 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion Line2T
#region LocalityNameT
                        _currentXPath.Push("/tns:LocalityNameT");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "LocalityNameT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.TRT15 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion LocalityNameT
#region PostcodeT
                        _currentXPath.Push("/tns:PostcodeT");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "PostcodeT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.TRT17 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion PostcodeT
#region StateOrTerritoryC
                        _currentXPath.Push("/tns:StateOrTerritoryC");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "StateOrTerritoryC" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.TRT16 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion StateOrTerritoryC
#region CountryC
                        _currentXPath.Push("/tns:CountryC");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "CountryC" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.TRT368 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion CountryC
                    } // End of if PreviousPostalAddressDetails node exists

                    _currentXPath.Pop();
#endregion PreviousPostalAddressDetails
                } // End of if PreviousTrust node exists

                _currentXPath.Pop();
#endregion PreviousTrust
#region MailRecipient
                _currentXPath.Push("/tns:MailRecipient");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "MailRecipient" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_MailRecipientCollectionExists = true;
                    report.RP_MailRecipientCollectionCount += 1;
                    _found = true;
#region PersonNameDetails
                    _currentXPath.Push("/tns:PersonNameDetails");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "PersonNameDetails" && reader.Depth == _currentXPath.Count - 1)
                    {
                        report.RP_MailRecipient_PersonNameDetailsCollectionExists = true;
                        report.RP_MailRecipient_PersonNameDetailsCollectionCount += 1;
                        _found = true;
#region TitleT
                        _currentXPath.Push("/tns:TitleT");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "TitleT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.TRT19 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion TitleT
#region NameSuffixT
                        _currentXPath.Push("/tns:NameSuffixT");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "NameSuffixT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.TRT21 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion NameSuffixT
#region FamilyNameT
                        _currentXPath.Push("/tns:FamilyNameT");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "FamilyNameT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.TRT20 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion FamilyNameT
#region GivenNameT
                        _currentXPath.Push("/tns:GivenNameT");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "GivenNameT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.TRT22 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion GivenNameT
#region OtherGivenNameT
                        _currentXPath.Push("/tns:OtherGivenNameT");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "OtherGivenNameT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.TRT23 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion OtherGivenNameT
                    } // End of if PersonNameDetails node exists

                    _currentXPath.Pop();
#endregion PersonNameDetails
#region OrganisationNameDetails
                    _currentXPath.Push("/tns:OrganisationNameDetails");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "OrganisationNameDetails" && reader.Depth == _currentXPath.Count - 1)
                    {
                        report.RP_MailRecipient_OrganisationNameDetailsCollectionExists = true;
                        report.RP_MailRecipient_OrganisationNameDetailsCollectionCount += 1;
                        _found = true;
#region OrganisationalNameT
                        _currentXPath.Push("/tns:OrganisationalNameT");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "OrganisationalNameT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.TRT24 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion OrganisationalNameT
#region AustralianBusinessNumberId
                        _currentXPath.Push("/tns:AustralianBusinessNumberId");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "AustralianBusinessNumberId" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.TRT25 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion AustralianBusinessNumberId
                    } // End of if OrganisationNameDetails node exists

                    _currentXPath.Pop();
#endregion OrganisationNameDetails
#region ElectronicContactTelephone
                    _currentXPath.Push("/tns:ElectronicContactTelephone");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "ElectronicContactTelephone" && reader.Depth == _currentXPath.Count - 1)
                    {
                        report.RP_MailRecipient_ElectronicContactTelephoneCollectionExists = true;
                        report.RP_MailRecipient_ElectronicContactTelephoneCollectionCount += 1;
                        _found = true;
#region ServiceLineC
                        _currentXPath.Push("/tns:ServiceLineC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "ServiceLineC" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.TRT371 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion ServiceLineC
#region AreaC
                        _currentXPath.Push("/tns:AreaC");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "AreaC" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.TRT26 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion AreaC
#region MinimalN
                        _currentXPath.Push("/tns:MinimalN");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "MinimalN" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.TRT27 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion MinimalN
                    } // End of if ElectronicContactTelephone node exists

                    _currentXPath.Pop();
#endregion ElectronicContactTelephone
                } // End of if MailRecipient node exists

                _currentXPath.Pop();
#endregion MailRecipient
#region ElectionStatus
                _currentXPath.Push("/tns:ElectionStatus");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "ElectionStatus" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_ElectionStatusCollectionExists = true;
                    report.RP_ElectionStatusCollectionCount += 1;
                    _found = true;
#region ElectionsFamilyTrustElectionStatusY
                    _currentXPath.Push("/tns:ElectionsFamilyTrustElectionStatusY");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "ElectionsFamilyTrustElectionStatusY" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Int32.TryParse(currentValue, out currentIntValue))
                                {
                                    report.TRT28 = currentIntValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion ElectionsFamilyTrustElectionStatusY
#region ElectionsFamilyTrustElectionRevocationC
                    _currentXPath.Push("/tns:ElectionsFamilyTrustElectionRevocationC");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "ElectionsFamilyTrustElectionRevocationC" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRT29 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion ElectionsFamilyTrustElectionRevocationC
                } // End of if ElectionStatus node exists

                _currentXPath.Pop();
#endregion ElectionStatus
#region InterposedEntityElectionStatus
                _currentXPath.Push("/tns:InterposedEntityElectionStatus");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "InterposedEntityElectionStatus" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_InterposedEntityElectionStatusCollectionExists = true;
                    report.RP_InterposedEntityElectionStatusCollectionCount += 1;
                    _found = true;
#region ElectionsY
                    _currentXPath.Push("/tns:ElectionsY");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "ElectionsY" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Int32.TryParse(currentValue, out currentIntValue))
                                {
                                    report.TRT30 = currentIntValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion ElectionsY
#region ElectionsInterposedEntityElectionRevocationC
                    _currentXPath.Push("/tns:ElectionsInterposedEntityElectionRevocationC");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "ElectionsInterposedEntityElectionRevocationC" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRT31 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion ElectionsInterposedEntityElectionRevocationC
                } // End of if InterposedEntityElectionStatus node exists

                _currentXPath.Pop();
#endregion InterposedEntityElectionStatus
#region TypeOfTrust
                _currentXPath.Push("/tns:TypeOfTrust");
                //3. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "TypeOfTrust" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_TypeOfTrustCollectionExists = true;
                    report.RP_TypeOfTrustCollectionCount += 1;
                    _found = true;
#region OrganisationDetailsOrganisationTypeC
                    _currentXPath.Push("/tns:OrganisationDetailsOrganisationTypeC");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "OrganisationDetailsOrganisationTypeC" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRT32 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion OrganisationDetailsOrganisationTypeC
#region OrganisationDetailsCharitableOrganisationI
                    _currentXPath.Push("/tns:OrganisationDetailsCharitableOrganisationI");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "OrganisationDetailsCharitableOrganisationI" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.TRT33 = currentBooleanValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion OrganisationDetailsCharitableOrganisationI
#region PersonDemographicDetailsDeathD
                    _currentXPath.Push("/tns:PersonDemographicDetailsDeathD");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "PersonDemographicDetailsDeathD" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                                {
                                    report.TRT34 = currentDateTimeValue.Date;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion PersonDemographicDetailsDeathD
#region AssetsInvestmentCrossOtherCCIVSubfundI
                    _currentXPath.Push("/tns:AssetsInvestmentCrossOtherCCIVSubfundI");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "AssetsInvestmentCrossOtherCCIVSubfundI" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.TRT566 = currentBooleanValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion AssetsInvestmentCrossOtherCCIVSubfundI
#region OrganisationDetailsManagedInvestmentTrustI
                    _currentXPath.Push("/tns:OrganisationDetailsManagedInvestmentTrustI");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "OrganisationDetailsManagedInvestmentTrustI" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.TRT520 = currentBooleanValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion OrganisationDetailsManagedInvestmentTrustI
#region OrganisationDetailsWithholdingManagedInvestmentTrustI
                    _currentXPath.Push("/tns:OrganisationDetailsWithholdingManagedInvestmentTrustI");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "OrganisationDetailsWithholdingManagedInvestmentTrustI" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.TRT577 = currentBooleanValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion OrganisationDetailsWithholdingManagedInvestmentTrustI
#region OrganisationDetailsStapledManagedInvestmentTrustI
                    _currentXPath.Push("/tns:OrganisationDetailsStapledManagedInvestmentTrustI");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "OrganisationDetailsStapledManagedInvestmentTrustI" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.TRT578 = currentBooleanValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion OrganisationDetailsStapledManagedInvestmentTrustI
#region OrganisationDetailsManagedInvestmentTrustTypeC
                    _currentXPath.Push("/tns:OrganisationDetailsManagedInvestmentTrustTypeC");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "OrganisationDetailsManagedInvestmentTrustTypeC" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRT521 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion OrganisationDetailsManagedInvestmentTrustTypeC
#region ElectionsManagedInvestmentTrustsI
                    _currentXPath.Push("/tns:ElectionsManagedInvestmentTrustsI");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "ElectionsManagedInvestmentTrustsI" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.TRT352 = currentBooleanValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion ElectionsManagedInvestmentTrustsI
#region StatementSummaryPaymentRefundOrNilC
                    _currentXPath.Push("/tns:StatementSummaryPaymentRefundOrNilC");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "StatementSummaryPaymentRefundOrNilC" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRT35 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion StatementSummaryPaymentRefundOrNilC
#region LodgmentFinalReturnI
                    _currentXPath.Push("/tns:LodgmentFinalReturnI");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "LodgmentFinalReturnI" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.TRT36 = currentBooleanValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion LodgmentFinalReturnI
                } // End of if TypeOfTrust node exists

                _currentXPath.Pop();
#endregion TypeOfTrust
#region FinancialInstitutionAccount
                _currentXPath.Push("/tns:FinancialInstitutionAccount");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "FinancialInstitutionAccount" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_FinancialInstitutionAccountCollectionExists = true;
                    report.RP_FinancialInstitutionAccountCollectionCount += 1;
                    _found = true;
#region BankStateBranchN
                    _currentXPath.Push("/tns:BankStateBranchN");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "BankStateBranchN" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRT38 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion BankStateBranchN
#region N
                    _currentXPath.Push("/tns:N");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "N" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRT39 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion N
#region NameT
                    _currentXPath.Push("/tns:NameT");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "NameT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRT40 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion NameT
                } // End of if FinancialInstitutionAccount node exists

                _currentXPath.Pop();
#endregion FinancialInstitutionAccount
#region BusinessDetails
                _currentXPath.Push("/tns:BusinessDetails");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "BusinessDetails" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_BusinessDetailsCollectionExists = true;
                    report.RP_BusinessDetailsCollectionCount += 1;
                    _found = true;
#region OrganisationDetailsMainIncomeActivityDe
                    _currentXPath.Push("/tns:OrganisationDetailsMainIncomeActivityDe");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "OrganisationDetailsMainIncomeActivityDe" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRT41 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion OrganisationDetailsMainIncomeActivityDe
#region OrganisationDetailsOrganisationIndustry2006ExtendedC
                    _currentXPath.Push("/tns:OrganisationDetailsOrganisationIndustry2006ExtendedC");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "OrganisationDetailsOrganisationIndustry2006ExtendedC" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRT42 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion OrganisationDetailsOrganisationIndustry2006ExtendedC
#region OrganisationDetailsActivityEventC
                    _currentXPath.Push("/tns:OrganisationDetailsActivityEventC");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "OrganisationDetailsActivityEventC" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRT43 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion OrganisationDetailsActivityEventC
#region OrganisationDetailsTaxConsolidationStatusI
                    _currentXPath.Push("/tns:OrganisationDetailsTaxConsolidationStatusI");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "OrganisationDetailsTaxConsolidationStatusI" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.TRT44 = currentBooleanValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion OrganisationDetailsTaxConsolidationStatusI
#region OrganisationDetailsSignificantGlobalEntityStatusI
                    _currentXPath.Push("/tns:OrganisationDetailsSignificantGlobalEntityStatusI");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "OrganisationDetailsSignificantGlobalEntityStatusI" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.TRT487 = currentBooleanValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion OrganisationDetailsSignificantGlobalEntityStatusI
#region InternationalDealingsCountryByCountryReportingEntityStatusI
                    _currentXPath.Push("/tns:InternationalDealingsCountryByCountryReportingEntityStatusI");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsCountryByCountryReportingEntityStatusI" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.TRT550 = currentBooleanValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion InternationalDealingsCountryByCountryReportingEntityStatusI
#region Income
                    _currentXPath.Push("/tns:Income");
                    //3. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "Income" && reader.Depth == _currentXPath.Count - 1)
                    {
                        report.RP_BusinessDetails_IncomeCollectionExists = true;
                        report.RP_BusinessDetails_IncomeCollectionCount += 1;
                        _found = true;
#region BusinessIncomeAndExpenses
                        _currentXPath.Push("/tns:BusinessIncomeAndExpenses");
                        //3. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "BusinessIncomeAndExpenses" && reader.Depth == _currentXPath.Count - 1)
                        {
                            report.RP_BusinessDetails_Income_BusinessIncomeAndExpensesCollectionExists = true;
                            report.RP_BusinessDetails_Income_BusinessIncomeAndExpensesCollectionCount += 1;
                            _found = true;
#region SmallBusinessEntityNetA
                            _currentXPath.Push("/tns:SmallBusinessEntityNetA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "SmallBusinessEntityNetA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.TRT478 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion SmallBusinessEntityNetA
#region ReconciliationAdjustmentTotalA
                            _currentXPath.Push("/tns:ReconciliationAdjustmentTotalA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "ReconciliationAdjustmentTotalA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.TRT69 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion ReconciliationAdjustmentTotalA
#region IncomePrimaryProduction
                            _currentXPath.Push("/tns:IncomePrimaryProduction");
                            //3. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomePrimaryProduction" && reader.Depth == _currentXPath.Count - 1)
                            {
                                report.RP_BusinessDetails_Income_BusinessIncomeAndExpenses_IncomePrimaryProductionCollectionExists = true;
                                report.RP_BusinessDetails_Income_BusinessIncomeAndExpenses_IncomePrimaryProductionCollectionCount += 1;
                                _found = true;
#region RemunerationABNNotQuotedPaymentGrossA
                                _currentXPath.Push("/tns:RemunerationABNNotQuotedPaymentGrossA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationABNNotQuotedPaymentGrossA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                report.TRT46 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion RemunerationABNNotQuotedPaymentGrossA
#region GovernmentFundingGovernmentIndustryPaymentsAssessableA
                                _currentXPath.Push("/tns:GovernmentFundingGovernmentIndustryPaymentsAssessableA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "GovernmentFundingGovernmentIndustryPaymentsAssessableA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                report.TRT50 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion GovernmentFundingGovernmentIndustryPaymentsAssessableA
#region GovernmentFundingGovernmentIndustryPaymentsAssessableI
                                _currentXPath.Push("/tns:GovernmentFundingGovernmentIndustryPaymentsAssessableI");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "GovernmentFundingGovernmentIndustryPaymentsAssessableI" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            currentBooleanValue = ToBoolean(currentValue);
                                            if (currentBooleanValue != null)
                                            {
                                                report.TRT361 = currentBooleanValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion GovernmentFundingGovernmentIndustryPaymentsAssessableI
#region OtherA
                                _currentXPath.Push("/tns:OtherA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "OtherA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                report.TRT54 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion OtherA
#region NetA
                                _currentXPath.Push("/tns:NetA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "NetA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                report.TRT71 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion NetA
                            } // End of if IncomePrimaryProduction node exists

                            _currentXPath.Pop();
#endregion IncomePrimaryProduction
#region IncomeNonPrimaryProduction
                            _currentXPath.Push("/tns:IncomeNonPrimaryProduction");
                            //3. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeNonPrimaryProduction" && reader.Depth == _currentXPath.Count - 1)
                            {
                                report.RP_BusinessDetails_Income_BusinessIncomeAndExpenses_IncomeNonPrimaryProductionCollectionExists = true;
                                report.RP_BusinessDetails_Income_BusinessIncomeAndExpenses_IncomeNonPrimaryProductionCollectionCount += 1;
                                _found = true;
#region RemunerationABNNotQuotedPaymentGrossA
                                _currentXPath.Push("/tns:RemunerationABNNotQuotedPaymentGrossA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationABNNotQuotedPaymentGrossA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                report.TRT47 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion RemunerationABNNotQuotedPaymentGrossA
#region RemunerationPaymentToForeignResidentGrossA
                                _currentXPath.Push("/tns:RemunerationPaymentToForeignResidentGrossA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationPaymentToForeignResidentGrossA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                report.TRT49 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion RemunerationPaymentToForeignResidentGrossA
#region GovernmentFundingGovernmentIndustryPaymentsAssessableA
                                _currentXPath.Push("/tns:GovernmentFundingGovernmentIndustryPaymentsAssessableA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "GovernmentFundingGovernmentIndustryPaymentsAssessableA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                report.TRT52 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion GovernmentFundingGovernmentIndustryPaymentsAssessableA
#region GovernmentFundingGovernmentIndustryPaymentsAssessableI
                                _currentXPath.Push("/tns:GovernmentFundingGovernmentIndustryPaymentsAssessableI");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "GovernmentFundingGovernmentIndustryPaymentsAssessableI" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            currentBooleanValue = ToBoolean(currentValue);
                                            if (currentBooleanValue != null)
                                            {
                                                report.TRT362 = currentBooleanValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion GovernmentFundingGovernmentIndustryPaymentsAssessableI
#region OtherA
                                _currentXPath.Push("/tns:OtherA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "OtherA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                report.TRT55 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion OtherA
#region NetA
                                _currentXPath.Push("/tns:NetA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "NetA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                report.TRT72 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion NetA
                            } // End of if IncomeNonPrimaryProduction node exists

                            _currentXPath.Pop();
#endregion IncomeNonPrimaryProduction
#region BusinessExpenses
                            _currentXPath.Push("/tns:BusinessExpenses");
                            //3. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "BusinessExpenses" && reader.Depth == _currentXPath.Count - 1)
                            {
                                report.RP_BusinessDetails_Income_BusinessIncomeAndExpenses_BusinessExpensesCollectionExists = true;
                                report.RP_BusinessDetails_Income_BusinessIncomeAndExpenses_BusinessExpensesCollectionCount += 1;
                                _found = true;
#region ForeignResidentWithholdingA
                                _currentXPath.Push("/tns:ForeignResidentWithholdingA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "ForeignResidentWithholdingA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                report.TRT56 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion ForeignResidentWithholdingA
#region RemunerationWagesAndSalariesPaymentsContractorAndConsultantA
                                _currentXPath.Push("/tns:RemunerationWagesAndSalariesPaymentsContractorAndConsultantA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationWagesAndSalariesPaymentsContractorAndConsultantA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                report.TRT57 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion RemunerationWagesAndSalariesPaymentsContractorAndConsultantA
#region SuperannuationContributionEmployerContributionsA
                                _currentXPath.Push("/tns:SuperannuationContributionEmployerContributionsA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "SuperannuationContributionEmployerContributionsA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                report.TRT58 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion SuperannuationContributionEmployerContributionsA
#region OperatingCostOfSalesA
                                _currentXPath.Push("/tns:OperatingCostOfSalesA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "OperatingCostOfSalesA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                report.TRT467 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion OperatingCostOfSalesA
#region OperatingBadDebtsA
                                _currentXPath.Push("/tns:OperatingBadDebtsA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "OperatingBadDebtsA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                report.TRT468 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion OperatingBadDebtsA
#region OperatingLeaseTotalA
                                _currentXPath.Push("/tns:OperatingLeaseTotalA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "OperatingLeaseTotalA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                report.TRT61 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion OperatingLeaseTotalA
#region OperatingRentA
                                _currentXPath.Push("/tns:OperatingRentA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "OperatingRentA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                report.TRT469 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion OperatingRentA
#region OperatingInterestA
                                _currentXPath.Push("/tns:OperatingInterestA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "OperatingInterestA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                report.TRT470 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion OperatingInterestA
#region RoyaltiesA
                                _currentXPath.Push("/tns:RoyaltiesA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "RoyaltiesA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                report.TRT471 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion RoyaltiesA
#region DepreciationAndAmortisationA
                                _currentXPath.Push("/tns:DepreciationAndAmortisationA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "DepreciationAndAmortisationA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                report.TRT65 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion DepreciationAndAmortisationA
#region OperatingMotorVehicleA
                                _currentXPath.Push("/tns:OperatingMotorVehicleA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "OperatingMotorVehicleA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                report.TRT472 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion OperatingMotorVehicleA
#region OperatingRepairsAndMaintenanceA
                                _currentXPath.Push("/tns:OperatingRepairsAndMaintenanceA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "OperatingRepairsAndMaintenanceA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                report.TRT473 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion OperatingRepairsAndMaintenanceA
#region DeductibleOtherTotalA
                                _currentXPath.Push("/tns:DeductibleOtherTotalA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "DeductibleOtherTotalA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                report.TRT68 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion DeductibleOtherTotalA
#region ReconciliationAdjustmentTotalA
                                _currentXPath.Push("/tns:ReconciliationAdjustmentTotalA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "ReconciliationAdjustmentTotalA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                report.TRT447 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion ReconciliationAdjustmentTotalA
                            } // End of if BusinessExpenses node exists

                            _currentXPath.Pop();
#endregion BusinessExpenses
                        } // End of if BusinessIncomeAndExpenses node exists

                        _currentXPath.Pop();
#endregion BusinessIncomeAndExpenses
#region PayAsYouGoWithholding
                        _currentXPath.Push("/tns:PayAsYouGoWithholding");
                        //3. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "PayAsYouGoWithholding" && reader.Depth == _currentXPath.Count - 1)
                        {
                            report.RP_BusinessDetails_Income_PayAsYouGoWithholdingCollectionExists = true;
                            report.RP_BusinessDetails_Income_PayAsYouGoWithholdingCollectionCount += 1;
                            _found = true;
#region TaxCreditForTaxWithheldWhereABNNotQuotedA
                            _currentXPath.Push("/tns:TaxCreditForTaxWithheldWhereABNNotQuotedA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "TaxCreditForTaxWithheldWhereABNNotQuotedA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.TRT73 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion TaxCreditForTaxWithheldWhereABNNotQuotedA
#region TaxCreditForAmountsWithheldFromForeignResidentsA
                            _currentXPath.Push("/tns:TaxCreditForAmountsWithheldFromForeignResidentsA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "TaxCreditForAmountsWithheldFromForeignResidentsA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.TRT474 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion TaxCreditForAmountsWithheldFromForeignResidentsA
                        } // End of if PayAsYouGoWithholding node exists

                        _currentXPath.Pop();
#endregion PayAsYouGoWithholding
#region PartnershipsTrustsPrimaryProduction
                        _currentXPath.Push("/tns:PartnershipsTrustsPrimaryProduction");
                        //3. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "PartnershipsTrustsPrimaryProduction" && reader.Depth == _currentXPath.Count - 1)
                        {
                            report.RP_BusinessDetails_Income_PartnershipsTrustsPrimaryProductionCollectionExists = true;
                            report.RP_BusinessDetails_Income_PartnershipsTrustsPrimaryProductionCollectionCount += 1;
                            _found = true;
#region PartnershipDistributionGrossA
                            _currentXPath.Push("/tns:PartnershipDistributionGrossA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "PartnershipDistributionGrossA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.TRT76 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion PartnershipDistributionGrossA
#region TrustShareNetA
                            _currentXPath.Push("/tns:TrustShareNetA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "TrustShareNetA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.TRT77 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion TrustShareNetA
#region PartyTrustTypeC
                            _currentXPath.Push("/tns:PartyTrustTypeC");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "PartyTrustTypeC" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        report.TRT353 = currentValue;
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion PartyTrustTypeC
#region ExpenseDistributionDeductionsOtherA
                            _currentXPath.Push("/tns:ExpenseDistributionDeductionsOtherA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseDistributionDeductionsOtherA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.TRT79 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion ExpenseDistributionDeductionsOtherA
                        } // End of if PartnershipsTrustsPrimaryProduction node exists

                        _currentXPath.Pop();
#endregion PartnershipsTrustsPrimaryProduction
#region PartnershipTrustNonPrimaryProduction
                        _currentXPath.Push("/tns:PartnershipTrustNonPrimaryProduction");
                        //3. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "PartnershipTrustNonPrimaryProduction" && reader.Depth == _currentXPath.Count - 1)
                        {
                            report.RP_BusinessDetails_Income_PartnershipTrustNonPrimaryProductionCollectionExists = true;
                            report.RP_BusinessDetails_Income_PartnershipTrustNonPrimaryProductionCollectionCount += 1;
                            _found = true;
#region PartnershipDistributionLessForeignIncomeA
                            _currentXPath.Push("/tns:PartnershipDistributionLessForeignIncomeA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "PartnershipDistributionLessForeignIncomeA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.TRT80 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion PartnershipDistributionLessForeignIncomeA
#region TrustShareNetExcludeNetCapitalGainsAndForeignIncomeAndDistributionFrankedA
                            _currentXPath.Push("/tns:TrustShareNetExcludeNetCapitalGainsAndForeignIncomeAndDistributionFrankedA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "TrustShareNetExcludeNetCapitalGainsAndForeignIncomeAndDistributionFrankedA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.TRT81 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion TrustShareNetExcludeNetCapitalGainsAndForeignIncomeAndDistributionFrankedA
#region PartyTrustTypeC
                            _currentXPath.Push("/tns:PartyTrustTypeC");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "PartyTrustTypeC" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        report.TRT436 = currentValue;
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion PartyTrustTypeC
#region ExpenseDistributionDeductionsOtherA
                            _currentXPath.Push("/tns:ExpenseDistributionDeductionsOtherA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseDistributionDeductionsOtherA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.TRT83 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion ExpenseDistributionDeductionsOtherA
#region TrustDistributionFrankedA
                            _currentXPath.Push("/tns:TrustDistributionFrankedA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "TrustDistributionFrankedA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.TRT437 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion TrustDistributionFrankedA
#region TaxDeductionTrustDistributionFrankedA
                            _currentXPath.Push("/tns:TaxDeductionTrustDistributionFrankedA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "TaxDeductionTrustDistributionFrankedA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.TRT438 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion TaxDeductionTrustDistributionFrankedA
                        } // End of if PartnershipTrustNonPrimaryProduction node exists

                        _currentXPath.Pop();
#endregion PartnershipTrustNonPrimaryProduction
#region ShareOfCredits
                        _currentXPath.Push("/tns:ShareOfCredits");
                        //3. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ShareOfCredits" && reader.Depth == _currentXPath.Count - 1)
                        {
                            report.RP_BusinessDetails_Income_ShareOfCreditsCollectionExists = true;
                            report.RP_BusinessDetails_Income_ShareOfCreditsCollectionCount += 1;
                            _found = true;
#region TaxPayAsYouGoWithholdingCreditForTaxWithheldWhereABNNotQuotedShareReceivedIndirectlyA
                            _currentXPath.Push("/tns:TaxPayAsYouGoWithholdingCreditForTaxWithheldWhereABNNotQuotedShareReceivedIndirectlyA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "TaxPayAsYouGoWithholdingCreditForTaxWithheldWhereABNNotQuotedShareReceivedIndirectlyA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.TRT84 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion TaxPayAsYouGoWithholdingCreditForTaxWithheldWhereABNNotQuotedShareReceivedIndirectlyA
#region TaxFrankingCreditsFrankingCreditShareReceivedIndirectlyA
                            _currentXPath.Push("/tns:TaxFrankingCreditsFrankingCreditShareReceivedIndirectlyA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "TaxFrankingCreditsFrankingCreditShareReceivedIndirectlyA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.TRT85 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion TaxFrankingCreditsFrankingCreditShareReceivedIndirectlyA
#region TaxPayAsYouGoWithholdingCreditForAmountsWithheldFromInterestAndDividendsWhereTFNNotQuotedShareReceivedIndirectlyA
                            _currentXPath.Push("/tns:TaxPayAsYouGoWithholdingCreditForAmountsWithheldFromInterestAndDividendsWhereTFNNotQuotedShareReceivedIndirectlyA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "TaxPayAsYouGoWithholdingCreditForAmountsWithheldFromInterestAndDividendsWhereTFNNotQuotedShareReceivedIndirectlyA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.TRT86 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion TaxPayAsYouGoWithholdingCreditForAmountsWithheldFromInterestAndDividendsWhereTFNNotQuotedShareReceivedIndirectlyA
#region TaxPayAsYouGoWithholdingCreditForTaxWithheldFromCloselyHeldTrustShareReceivedIndirectlyA
                            _currentXPath.Push("/tns:TaxPayAsYouGoWithholdingCreditForTaxWithheldFromCloselyHeldTrustShareReceivedIndirectlyA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "TaxPayAsYouGoWithholdingCreditForTaxWithheldFromCloselyHeldTrustShareReceivedIndirectlyA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.TRT418 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion TaxPayAsYouGoWithholdingCreditForTaxWithheldFromCloselyHeldTrustShareReceivedIndirectlyA
#region InternationalDealingsCreditForTaxWithheldFromForeignResidentWithholdingShareReceivedIndirectlyA
                            _currentXPath.Push("/tns:InternationalDealingsCreditForTaxWithheldFromForeignResidentWithholdingShareReceivedIndirectlyA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsCreditForTaxWithheldFromForeignResidentWithholdingShareReceivedIndirectlyA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.TRT87 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion InternationalDealingsCreditForTaxWithheldFromForeignResidentWithholdingShareReceivedIndirectlyA
                        } // End of if ShareOfCredits node exists

                        _currentXPath.Pop();
#endregion ShareOfCredits
#region Rent
                        _currentXPath.Push("/tns:Rent");
                        //3. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "Rent" && reader.Depth == _currentXPath.Count - 1)
                        {
                            report.RP_BusinessDetails_Income_RentCollectionExists = true;
                            report.RP_BusinessDetails_Income_RentCollectionCount += 1;
                            _found = true;
#region OperatingIncomeGrossA
                            _currentXPath.Push("/tns:OperatingIncomeGrossA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "OperatingIncomeGrossA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.TRT89 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion OperatingIncomeGrossA
#region ExpenseInterestA
                            _currentXPath.Push("/tns:ExpenseInterestA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseInterestA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.TRT90 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion ExpenseInterestA
#region ExpenseCapitalWorksDeductionA
                            _currentXPath.Push("/tns:ExpenseCapitalWorksDeductionA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseCapitalWorksDeductionA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.TRT91 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion ExpenseCapitalWorksDeductionA
#region TaxDeductionIncomeDeductionsOtherThanInterestAndCapitalWorksA
                            _currentXPath.Push("/tns:TaxDeductionIncomeDeductionsOtherThanInterestAndCapitalWorksA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "TaxDeductionIncomeDeductionsOtherThanInterestAndCapitalWorksA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.TRT92 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion TaxDeductionIncomeDeductionsOtherThanInterestAndCapitalWorksA
#region ExpenseBuildToRentCapitalWorksDeductionA
                            _currentXPath.Push("/tns:ExpenseBuildToRentCapitalWorksDeductionA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseBuildToRentCapitalWorksDeductionA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.TRT579 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion ExpenseBuildToRentCapitalWorksDeductionA
                        } // End of if Rent node exists

                        _currentXPath.Pop();
#endregion Rent
#region ForestryManagedInvestmentSchemeA
                        _currentXPath.Push("/tns:ForestryManagedInvestmentSchemeA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ForestryManagedInvestmentSchemeA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.TRT93 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion ForestryManagedInvestmentSchemeA
#region GrossInterest
                        _currentXPath.Push("/tns:GrossInterest");
                        //3. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "GrossInterest" && reader.Depth == _currentXPath.Count - 1)
                        {
                            report.RP_BusinessDetails_Income_GrossInterestCollectionExists = true;
                            report.RP_BusinessDetails_Income_GrossInterestCollectionCount += 1;
                            _found = true;
#region A
                            _currentXPath.Push("/tns:A");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "A" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.TRT94 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion A
#region TaxTFNAmountsWithheldFromGrossInterestA
                            _currentXPath.Push("/tns:TaxTFNAmountsWithheldFromGrossInterestA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "TaxTFNAmountsWithheldFromGrossInterestA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.TRT95 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion TaxTFNAmountsWithheldFromGrossInterestA
                        } // End of if GrossInterest node exists

                        _currentXPath.Pop();
#endregion GrossInterest
#region Dividends
                        _currentXPath.Push("/tns:Dividends");
                        //3. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "Dividends" && reader.Depth == _currentXPath.Count - 1)
                        {
                            report.RP_BusinessDetails_Income_DividendsCollectionExists = true;
                            report.RP_BusinessDetails_Income_DividendsCollectionCount += 1;
                            _found = true;
#region UnfrankedA
                            _currentXPath.Push("/tns:UnfrankedA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "UnfrankedA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.TRT96 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion UnfrankedA
#region FrankedA
                            _currentXPath.Push("/tns:FrankedA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "FrankedA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.TRT97 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion FrankedA
#region TaxFrankingCreditsA
                            _currentXPath.Push("/tns:TaxFrankingCreditsA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "TaxFrankingCreditsA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.TRT98 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion TaxFrankingCreditsA
#region TaxTFNAmountsWithheldA
                            _currentXPath.Push("/tns:TaxTFNAmountsWithheldA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "TaxTFNAmountsWithheldA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.TRT99 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion TaxTFNAmountsWithheldA
                        } // End of if Dividends node exists

                        _currentXPath.Pop();
#endregion Dividends
#region SuperannuationLumpSumWithholdingPaymentDetails
                        _currentXPath.Push("/tns:SuperannuationLumpSumWithholdingPaymentDetails");
                        //3. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "SuperannuationLumpSumWithholdingPaymentDetails" && reader.Depth == _currentXPath.Count - 1)
                        {
                            report.RP_BusinessDetails_Income_SuperannuationLumpSumWithholdingPaymentDetailsCollectionExists = true;
                            report.RP_BusinessDetails_Income_SuperannuationLumpSumWithholdingPaymentDetailsCollectionCount += 1;
                            _found = true;
#region SuperannuationBenefitDeathBenefitLumpSumPaymentTaxedElementA
                            _currentXPath.Push("/tns:SuperannuationBenefitDeathBenefitLumpSumPaymentTaxedElementA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "SuperannuationBenefitDeathBenefitLumpSumPaymentTaxedElementA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.TRT100 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion SuperannuationBenefitDeathBenefitLumpSumPaymentTaxedElementA
#region SuperannuationBenefitDeathBenefitLumpSumPaymentUntaxedElementA
                            _currentXPath.Push("/tns:SuperannuationBenefitDeathBenefitLumpSumPaymentUntaxedElementA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "SuperannuationBenefitDeathBenefitLumpSumPaymentUntaxedElementA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.TRT101 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion SuperannuationBenefitDeathBenefitLumpSumPaymentUntaxedElementA
#region InsuranceBenefitsDeathBenefitI
                            _currentXPath.Push("/tns:InsuranceBenefitsDeathBenefitI");
                            //6. use case
                            if (ReadToNextElement(reader, true) && reader.LocalName == "InsuranceBenefitsDeathBenefitI" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        currentBooleanValue = ToBoolean(currentValue);
                                        if (currentBooleanValue != null)
                                        {
                                            report.TRT408 = currentBooleanValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion InsuranceBenefitsDeathBenefitI
                        } // End of if SuperannuationLumpSumWithholdingPaymentDetails node exists

                        _currentXPath.Pop();
#endregion SuperannuationLumpSumWithholdingPaymentDetails
#region EmploymentTerminationPaymentDetails
                        _currentXPath.Push("/tns:EmploymentTerminationPaymentDetailsCollection");
                        // 4. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "EmploymentTerminationPaymentDetailsCollection" && reader.Depth == _currentXPath.Count - 1)
                        {
                            report.RP_BusinessDetails_Income_EmploymentTerminationPaymentDetailsCollection = new List<TRT2026.RP_BusinessDetails_Income_EmploymentTerminationPaymentDetails>();
                            report.RP_BusinessDetails_Income_EmploymentTerminationPaymentDetailsCollectionExists = true;
                            _found = true;
                            while (ReadToNextElement(reader, false) && reader.LocalName == "EmploymentTerminationPaymentDetails")
                            {
                                _found = true;
                                TRT2026.RP_BusinessDetails_Income_EmploymentTerminationPaymentDetails employmentTerminationPaymentDetails = new TRT2026.RP_BusinessDetails_Income_EmploymentTerminationPaymentDetails();
                                report.RP_BusinessDetails_Income_EmploymentTerminationPaymentDetailsCollection.Add(employmentTerminationPaymentDetails);
                                report.RP_BusinessDetails_Income_EmploymentTerminationPaymentDetailsCollectionCount += 1;
                                employmentTerminationPaymentDetails.OccurrenceIndex = report.RP_BusinessDetails_Income_EmploymentTerminationPaymentDetailsCollectionCount;
                                _currentXPath.Push("/tns:EmploymentTerminationPaymentDetails[" + report.RP_BusinessDetails_Income_EmploymentTerminationPaymentDetailsCollectionCount + "]");
#region SuperannuationBenefitDeathBenefitEmploymentTerminationPaymentTaxableComponentA
                                _currentXPath.Push("/tns:SuperannuationBenefitDeathBenefitEmploymentTerminationPaymentTaxableComponentA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "SuperannuationBenefitDeathBenefitEmploymentTerminationPaymentTaxableComponentA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                employmentTerminationPaymentDetails.TRT102 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion SuperannuationBenefitDeathBenefitEmploymentTerminationPaymentTaxableComponentA
#region InsuranceBenefitsDeathBenefitI
                                _currentXPath.Push("/tns:InsuranceBenefitsDeathBenefitI");
                                //6. use case
                                if (ReadToNextElement(reader, true) && reader.LocalName == "InsuranceBenefitsDeathBenefitI" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            currentBooleanValue = ToBoolean(currentValue);
                                            if (currentBooleanValue != null)
                                            {
                                                employmentTerminationPaymentDetails.TRT409 = currentBooleanValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion InsuranceBenefitsDeathBenefitI
#region InsuranceBenefitsBeneficiaryTypeC
                                _currentXPath.Push("/tns:InsuranceBenefitsBeneficiaryTypeC");
                                //6. use case
                                if (ReadToNextElement(reader, true) && reader.LocalName == "InsuranceBenefitsBeneficiaryTypeC" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            employmentTerminationPaymentDetails.TRT411 = currentValue;
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion InsuranceBenefitsBeneficiaryTypeC
                                _currentXPath.Pop();
                            } // End of while loop for EmploymentTerminationPaymentDetails
                        } // End of if EmploymentTerminationPaymentDetailsCollection node exists

                        _currentXPath.Pop();
#endregion EmploymentTerminationPaymentDetails
#region OtherAustralianIncome
                        _currentXPath.Push("/tns:OtherAustralianIncomeCollection");
                        // 4. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "OtherAustralianIncomeCollection" && reader.Depth == _currentXPath.Count - 1)
                        {
                            report.RP_BusinessDetails_Income_OtherAustralianIncomeCollection = new List<TRT2026.RP_BusinessDetails_Income_OtherAustralianIncome>();
                            report.RP_BusinessDetails_Income_OtherAustralianIncomeCollectionExists = true;
                            _found = true;
                            while (ReadToNextElement(reader, false) && reader.LocalName == "OtherAustralianIncome")
                            {
                                _found = true;
                                TRT2026.RP_BusinessDetails_Income_OtherAustralianIncome otherAustralianIncome = new TRT2026.RP_BusinessDetails_Income_OtherAustralianIncome();
                                report.RP_BusinessDetails_Income_OtherAustralianIncomeCollection.Add(otherAustralianIncome);
                                report.RP_BusinessDetails_Income_OtherAustralianIncomeCollectionCount += 1;
                                otherAustralianIncome.OccurrenceIndex = report.RP_BusinessDetails_Income_OtherAustralianIncomeCollectionCount;
                                _currentXPath.Push("/tns:OtherAustralianIncome[" + report.RP_BusinessDetails_Income_OtherAustralianIncomeCollectionCount + "]");
#region De
                                _currentXPath.Push("/tns:De");
                                //6. use case
                                if (ReadToNextElement(reader, true) && reader.LocalName == "De" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            otherAustralianIncome.TRT120 = currentValue;
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion De
#region A
                                _currentXPath.Push("/tns:A");
                                //6. use case
                                if (ReadToNextElement(reader, true) && reader.LocalName == "A" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                otherAustralianIncome.TRT121 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion A
                                _currentXPath.Pop();
                            } // End of while loop for OtherAustralianIncome
                        } // End of if OtherAustralianIncomeCollection node exists

                        _currentXPath.Pop();
#endregion OtherAustralianIncome
#region ExceptedTrustNetA
                        _currentXPath.Push("/tns:ExceptedTrustNetA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ExceptedTrustNetA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.TRT122 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion ExceptedTrustNetA
#region TaxAustralianGovernmentPensionsAndAllowancesTaxWithheldA
                        _currentXPath.Push("/tns:TaxAustralianGovernmentPensionsAndAllowancesTaxWithheldA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "TaxAustralianGovernmentPensionsAndAllowancesTaxWithheldA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.TRT104 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion TaxAustralianGovernmentPensionsAndAllowancesTaxWithheldA
#region PensionTotalA
                        _currentXPath.Push("/tns:PensionTotalA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "PensionTotalA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.TRT105 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion PensionTotalA
#region IndividualNonBusinessWithholdingPaymentDetails
                        _currentXPath.Push("/tns:IndividualNonBusinessWithholdingPaymentDetailsCollection");
                        // 4. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "IndividualNonBusinessWithholdingPaymentDetailsCollection" && reader.Depth == _currentXPath.Count - 1)
                        {
                            report.RP_BusinessDetails_Income_IndividualNonBusinessWithholdingPaymentDetailsCollection = new List<TRT2026.RP_BusinessDetails_Income_IndividualNonBusinessWithholdingPaymentDetails>();
                            report.RP_BusinessDetails_Income_IndividualNonBusinessWithholdingPaymentDetailsCollectionExists = true;
                            _found = true;
                            while (ReadToNextElement(reader, false) && reader.LocalName == "IndividualNonBusinessWithholdingPaymentDetails")
                            {
                                _found = true;
                                TRT2026.RP_BusinessDetails_Income_IndividualNonBusinessWithholdingPaymentDetails individualNonBusinessWithholdingPaymentDetails = new TRT2026.RP_BusinessDetails_Income_IndividualNonBusinessWithholdingPaymentDetails();
                                report.RP_BusinessDetails_Income_IndividualNonBusinessWithholdingPaymentDetailsCollection.Add(individualNonBusinessWithholdingPaymentDetails);
                                report.RP_BusinessDetails_Income_IndividualNonBusinessWithholdingPaymentDetailsCollectionCount += 1;
                                individualNonBusinessWithholdingPaymentDetails.OccurrenceIndex = report.RP_BusinessDetails_Income_IndividualNonBusinessWithholdingPaymentDetailsCollectionCount;
                                _currentXPath.Push("/tns:IndividualNonBusinessWithholdingPaymentDetails[" + report.RP_BusinessDetails_Income_IndividualNonBusinessWithholdingPaymentDetailsCollectionCount + "]");
#region AustralianBusinessNumberId
                                _currentXPath.Push("/tns:AustralianBusinessNumberId");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "AustralianBusinessNumberId" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            individualNonBusinessWithholdingPaymentDetails.TRT107 = currentValue;
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion AustralianBusinessNumberId
#region TaxPayAsYouGoWithholdingTaxWithheldA
                                _currentXPath.Push("/tns:TaxPayAsYouGoWithholdingTaxWithheldA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "TaxPayAsYouGoWithholdingTaxWithheldA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                individualNonBusinessWithholdingPaymentDetails.TRT108 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion TaxPayAsYouGoWithholdingTaxWithheldA
#region RemunerationIndividualNonBusinessGrossA
                                _currentXPath.Push("/tns:RemunerationIndividualNonBusinessGrossA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationIndividualNonBusinessGrossA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                individualNonBusinessWithholdingPaymentDetails.TRT109 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion RemunerationIndividualNonBusinessGrossA
#region TaxPayAsYouGoWithholdingCreditTaxWithheldUnusedAnnualOrLongServiceLeavePaymentLumpSumAA
                                _currentXPath.Push("/tns:TaxPayAsYouGoWithholdingCreditTaxWithheldUnusedAnnualOrLongServiceLeavePaymentLumpSumAA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "TaxPayAsYouGoWithholdingCreditTaxWithheldUnusedAnnualOrLongServiceLeavePaymentLumpSumAA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                individualNonBusinessWithholdingPaymentDetails.TRT110 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion TaxPayAsYouGoWithholdingCreditTaxWithheldUnusedAnnualOrLongServiceLeavePaymentLumpSumAA
#region RemunerationUnusedAnnualLongServiceLeavePaymentLumpSumAA
                                _currentXPath.Push("/tns:RemunerationUnusedAnnualLongServiceLeavePaymentLumpSumAA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationUnusedAnnualLongServiceLeavePaymentLumpSumAA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                individualNonBusinessWithholdingPaymentDetails.TRT111 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion RemunerationUnusedAnnualLongServiceLeavePaymentLumpSumAA
#region TaxPayAsYouGoWithholdingCreditTaxWithheldUnusedAnnualOrLongServiceLeavePaymentLumpSumBA
                                _currentXPath.Push("/tns:TaxPayAsYouGoWithholdingCreditTaxWithheldUnusedAnnualOrLongServiceLeavePaymentLumpSumBA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "TaxPayAsYouGoWithholdingCreditTaxWithheldUnusedAnnualOrLongServiceLeavePaymentLumpSumBA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                individualNonBusinessWithholdingPaymentDetails.TRT112 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion TaxPayAsYouGoWithholdingCreditTaxWithheldUnusedAnnualOrLongServiceLeavePaymentLumpSumBA
#region RemunerationUnusedAnnualLongServiceLeavePaymentLumpSumBA
                                _currentXPath.Push("/tns:RemunerationUnusedAnnualLongServiceLeavePaymentLumpSumBA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationUnusedAnnualLongServiceLeavePaymentLumpSumBA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                individualNonBusinessWithholdingPaymentDetails.TRT113 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion RemunerationUnusedAnnualLongServiceLeavePaymentLumpSumBA
                                _currentXPath.Pop();
                            } // End of while loop for IndividualNonBusinessWithholdingPaymentDetails
                        } // End of if IndividualNonBusinessWithholdingPaymentDetailsCollection node exists

                        _currentXPath.Pop();
#endregion IndividualNonBusinessWithholdingPaymentDetails
#region SuperannuationIncomeStreamWithholdingPaymentDetails
                        _currentXPath.Push("/tns:SuperannuationIncomeStreamWithholdingPaymentDetails");
                        //3. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "SuperannuationIncomeStreamWithholdingPaymentDetails" && reader.Depth == _currentXPath.Count - 1)
                        {
                            report.RP_BusinessDetails_Income_SuperannuationIncomeStreamWithholdingPaymentDetailsCollectionExists = true;
                            report.RP_BusinessDetails_Income_SuperannuationIncomeStreamWithholdingPaymentDetailsCollectionCount += 1;
                            _found = true;
#region TaxPayAsYouGoWithholdingTaxWithheldA
                            _currentXPath.Push("/tns:TaxPayAsYouGoWithholdingTaxWithheldA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "TaxPayAsYouGoWithholdingTaxWithheldA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.TRT114 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion TaxPayAsYouGoWithholdingTaxWithheldA
#region TaxTaxableComponentTaxedElementA
                            _currentXPath.Push("/tns:TaxTaxableComponentTaxedElementA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "TaxTaxableComponentTaxedElementA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.TRT115 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion TaxTaxableComponentTaxedElementA
#region TaxTaxableComponentUntaxedElementA
                            _currentXPath.Push("/tns:TaxTaxableComponentUntaxedElementA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "TaxTaxableComponentUntaxedElementA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.TRT116 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion TaxTaxableComponentUntaxedElementA
#region LumpSumInArrears
                            _currentXPath.Push("/tns:LumpSumInArrears");
                            //3. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "LumpSumInArrears" && reader.Depth == _currentXPath.Count - 1)
                            {
                                report.RP_BusinessDetails_Income_SuperannuationIncomeStreamWithholdingPaymentDetails_LumpSumInArrearsCollectionExists = true;
                                report.RP_BusinessDetails_Income_SuperannuationIncomeStreamWithholdingPaymentDetails_LumpSumInArrearsCollectionCount += 1;
                                _found = true;
#region TaxTaxableComponentTaxedElementA
                                _currentXPath.Push("/tns:TaxTaxableComponentTaxedElementA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "TaxTaxableComponentTaxedElementA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                report.TRT117 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion TaxTaxableComponentTaxedElementA
#region TaxTaxableComponentUntaxedElementA
                                _currentXPath.Push("/tns:TaxTaxableComponentUntaxedElementA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "TaxTaxableComponentUntaxedElementA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                report.TRT118 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion TaxTaxableComponentUntaxedElementA
                            } // End of if LumpSumInArrears node exists

                            _currentXPath.Pop();
#endregion LumpSumInArrears
                        } // End of if SuperannuationIncomeStreamWithholdingPaymentDetails node exists

                        _currentXPath.Pop();
#endregion SuperannuationIncomeStreamWithholdingPaymentDetails
#region RemunerationBonusesAndAllowancesA
                        _currentXPath.Push("/tns:RemunerationBonusesAndAllowancesA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationBonusesAndAllowancesA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.TRT119 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion RemunerationBonusesAndAllowancesA
#region Deductions
                        _currentXPath.Push("/tns:Deductions");
                        //3. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "Deductions" && reader.Depth == _currentXPath.Count - 1)
                        {
                            report.RP_BusinessDetails_Income_DeductionsCollectionExists = true;
                            report.RP_BusinessDetails_Income_DeductionsCollectionCount += 1;
                            _found = true;
#region ExpenseAustralianInvestmentAllowableDeductionA
                            _currentXPath.Push("/tns:ExpenseAustralianInvestmentAllowableDeductionA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseAustralianInvestmentAllowableDeductionA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.TRT123 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion ExpenseAustralianInvestmentAllowableDeductionA
#region TaxDistributionFrankedA
                            _currentXPath.Push("/tns:TaxDistributionFrankedA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "TaxDistributionFrankedA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.TRT439 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion TaxDistributionFrankedA
#region ExpenseForestryManagedInvestmentSchemeDeductionA
                            _currentXPath.Push("/tns:ExpenseForestryManagedInvestmentSchemeDeductionA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseForestryManagedInvestmentSchemeDeductionA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.TRT124 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion ExpenseForestryManagedInvestmentSchemeDeductionA
#region OtherDeductions
                            _currentXPath.Push("/tns:OtherDeductionsCollection");
                            // 4. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "OtherDeductionsCollection" && reader.Depth == _currentXPath.Count - 1)
                            {
                                report.RP_BusinessDetails_Income_Deductions_OtherDeductionsCollection = new List<TRT2026.RP_BusinessDetails_Income_Deductions_OtherDeductions>();
                                report.RP_BusinessDetails_Income_Deductions_OtherDeductionsCollectionExists = true;
                                _found = true;
                                while (ReadToNextElement(reader, false) && reader.LocalName == "OtherDeductions")
                                {
                                    _found = true;
                                    TRT2026.RP_BusinessDetails_Income_Deductions_OtherDeductions otherDeductions = new TRT2026.RP_BusinessDetails_Income_Deductions_OtherDeductions();
                                    report.RP_BusinessDetails_Income_Deductions_OtherDeductionsCollection.Add(otherDeductions);
                                    report.RP_BusinessDetails_Income_Deductions_OtherDeductionsCollectionCount += 1;
                                    otherDeductions.OccurrenceIndex = report.RP_BusinessDetails_Income_Deductions_OtherDeductionsCollectionCount;
                                    _currentXPath.Push("/tns:OtherDeductions[" + report.RP_BusinessDetails_Income_Deductions_OtherDeductionsCollectionCount + "]");
#region ExpenseDeductibleOtherT
                                    _currentXPath.Push("/tns:ExpenseDeductibleOtherT");
                                    //6. use case
                                    if (ReadToNextElement(reader, true) && reader.LocalName == "ExpenseDeductibleOtherT" && reader.Depth == _currentXPath.Count - 1)
                                    {
                                        if (!IsEmptyOrNilElement(reader))
                                        {
                                            ReadNext(reader);
                                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                            {
                                                currentValue = reader.Value;
                                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                                otherDeductions.TRT128 = currentValue;
                                            }
                                        }

                                        _found = true;
                                    }

                                    _currentXPath.Pop();
#endregion ExpenseDeductibleOtherT
#region ExpenseDeductibleOtherA
                                    _currentXPath.Push("/tns:ExpenseDeductibleOtherA");
                                    //6. use case
                                    if (ReadToNextElement(reader, true) && reader.LocalName == "ExpenseDeductibleOtherA" && reader.Depth == _currentXPath.Count - 1)
                                    {
                                        if (!IsEmptyOrNilElement(reader))
                                        {
                                            ReadNext(reader);
                                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                            {
                                                currentValue = reader.Value;
                                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                                {
                                                    otherDeductions.TRT129 = currentDecimalValue;
                                                }
                                            }
                                        }

                                        _found = true;
                                    }

                                    _currentXPath.Pop();
#endregion ExpenseDeductibleOtherA
                                    _currentXPath.Pop();
                                } // End of while loop for OtherDeductions
                            } // End of if OtherDeductionsCollection node exists

                            _currentXPath.Pop();
#endregion OtherDeductions
                        } // End of if Deductions node exists

                        _currentXPath.Pop();
#endregion Deductions
#region NetA
                        _currentXPath.Push("/tns:NetA");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "NetA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.TRT130 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion NetA
#region CapitalGains
                        _currentXPath.Push("/tns:CapitalGains");
                        //3. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "CapitalGains" && reader.Depth == _currentXPath.Count - 1)
                        {
                            report.RP_BusinessDetails_Income_CapitalGainsCollectionExists = true;
                            report.RP_BusinessDetails_Income_CapitalGainsCollectionCount += 1;
                            _found = true;
#region TaxEventI
                            _currentXPath.Push("/tns:TaxEventI");
                            //6. use case
                            if (ReadToNextElement(reader, true) && reader.LocalName == "TaxEventI" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        currentBooleanValue = ToBoolean(currentValue);
                                        if (currentBooleanValue != null)
                                        {
                                            report.TRT131 = currentBooleanValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion TaxEventI
#region NetA
                            _currentXPath.Push("/tns:NetA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "NetA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.TRT133 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion NetA
#region TaxPayAsYouGoWithholdingCreditForCapitalGainsWithheldFromForeignResidentsA
                            _currentXPath.Push("/tns:TaxPayAsYouGoWithholdingCreditForCapitalGainsWithheldFromForeignResidentsA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "TaxPayAsYouGoWithholdingCreditForCapitalGainsWithheldFromForeignResidentsA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.TRT480 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion TaxPayAsYouGoWithholdingCreditForCapitalGainsWithheldFromForeignResidentsA
#region TaxExemptionOrRolloverAppliedI
                            _currentXPath.Push("/tns:TaxExemptionOrRolloverAppliedI");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "TaxExemptionOrRolloverAppliedI" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        currentBooleanValue = ToBoolean(currentValue);
                                        if (currentBooleanValue != null)
                                        {
                                            report.TRT454 = currentBooleanValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion TaxExemptionOrRolloverAppliedI
#region TaxExemptionOrRolloverAppliedCCollection
                            _currentXPath.Push("/tns:TaxExemptionOrRolloverAppliedCCollection");
                            if (ReadToNextElement(reader, false) && reader.LocalName == "TaxExemptionOrRolloverAppliedCCollection" && reader.Depth == _currentXPath.Count - 1)
                            {
                                report.TRT448Collection = new List<string>();
                                ReadNext(reader);
                                _currentXPath.Push("/tns:TaxExemptionOrRolloverAppliedC");
                                // 5. use case
                                while (ReadToNextElement(reader, false) && reader.LocalName == "TaxExemptionOrRolloverAppliedC" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before the currentXPath is updated.
                                            report.TRT448Collection.Add(currentValue);
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
                            }

                            _currentXPath.Pop();
#endregion TaxExemptionOrRolloverAppliedCCollection
                        } // End of if CapitalGains node exists

                        _currentXPath.Pop();
#endregion CapitalGains
#region AttributedForeignIncome
                        _currentXPath.Push("/tns:AttributedForeignIncome");
                        //3. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "AttributedForeignIncome" && reader.Depth == _currentXPath.Count - 1)
                        {
                            report.RP_BusinessDetails_Income_AttributedForeignIncomeCollectionExists = true;
                            report.RP_BusinessDetails_Income_AttributedForeignIncomeCollectionCount += 1;
                            _found = true;
#region InternationalDealingsTrustorControlledCompanyorTransferorTrustI
                            _currentXPath.Push("/tns:InternationalDealingsTrustorControlledCompanyorTransferorTrustI");
                            //6. use case
                            if (ReadToNextElement(reader, true) && reader.LocalName == "InternationalDealingsTrustorControlledCompanyorTransferorTrustI" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        currentBooleanValue = ToBoolean(currentValue);
                                        if (currentBooleanValue != null)
                                        {
                                            report.TRT138 = currentBooleanValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion InternationalDealingsTrustorControlledCompanyorTransferorTrustI
#region ListedCountryInternationalDealingsA
                            _currentXPath.Push("/tns:ListedCountryInternationalDealingsA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "ListedCountryInternationalDealingsA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.TRT134 = currentDecimalValue;
                                        }

                                        report.RP_BusinessDetails_Income_AttributedForeignIncome_ListedCountryCollectionExists = true;
                                        report.RP_BusinessDetails_Income_AttributedForeignIncome_ListedCountryCollectionCount += 1;
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion ListedCountryInternationalDealingsA
#region UnlistedCountryInternationalDealingsA
                            _currentXPath.Push("/tns:UnlistedCountryInternationalDealingsA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "UnlistedCountryInternationalDealingsA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.TRT136 = currentDecimalValue;
                                        }

                                        report.RP_BusinessDetails_Income_AttributedForeignIncome_UnlistedCountryCollectionExists = true;
                                        report.RP_BusinessDetails_Income_AttributedForeignIncome_UnlistedCountryCollectionCount += 1;
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion UnlistedCountryInternationalDealingsA
#region OtherAssessableForeignIncome
                            _currentXPath.Push("/tns:OtherAssessableForeignIncome");
                            //3. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "OtherAssessableForeignIncome" && reader.Depth == _currentXPath.Count - 1)
                            {
                                report.RP_BusinessDetails_Income_AttributedForeignIncome_OtherAssessableForeignIncomeCollectionExists = true;
                                report.RP_BusinessDetails_Income_AttributedForeignIncome_OtherAssessableForeignIncomeCollectionCount += 1;
                                _found = true;
#region A
                                _currentXPath.Push("/tns:A");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "A" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                report.TRT140 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion A
#region NetA
                                _currentXPath.Push("/tns:NetA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "NetA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                report.TRT141 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion NetA
#region InternationalDealingsTaxOffsetA
                                _currentXPath.Push("/tns:InternationalDealingsTaxOffsetA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsTaxOffsetA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                report.TRT142 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion InternationalDealingsTaxOffsetA
#region TaxFrankingCreditsReceivedFromNewZealandCompaniesA
                                _currentXPath.Push("/tns:TaxFrankingCreditsReceivedFromNewZealandCompaniesA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "TaxFrankingCreditsReceivedFromNewZealandCompaniesA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                report.TRT143 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion TaxFrankingCreditsReceivedFromNewZealandCompaniesA
                            } // End of if OtherAssessableForeignIncome node exists

                            _currentXPath.Pop();
#endregion OtherAssessableForeignIncome
                        } // End of if AttributedForeignIncome node exists

                        _currentXPath.Pop();
#endregion AttributedForeignIncome
#region TaxLossesDeductedTaxDeductionA
                        _currentXPath.Push("/tns:TaxLossesDeductedTaxDeductionA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "TaxLossesDeductedTaxDeductionA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.TRT144 = currentDecimalValue;
                                    }

                                    report.RP_BusinessDetails_Income_TaxLossesDeductedCollectionExists = true;
                                    report.RP_BusinessDetails_Income_TaxLossesDeductedCollectionCount += 1;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion TaxLossesDeductedTaxDeductionA
#region TaxLossesCarriedForwardLaterIncomeYearsTotalA
                        _currentXPath.Push("/tns:TaxLossesCarriedForwardLaterIncomeYearsTotalA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "TaxLossesCarriedForwardLaterIncomeYearsTotalA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.TRT145 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion TaxLossesCarriedForwardLaterIncomeYearsTotalA
#region CapitalLossesCarriedForwardNetA
                        _currentXPath.Push("/tns:CapitalLossesCarriedForwardNetA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "CapitalLossesCarriedForwardNetA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.TRT475 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion CapitalLossesCarriedForwardNetA
#region TaxOffsetClaimLandcareAndWaterFacilityBroughtForwardA
                        _currentXPath.Push("/tns:TaxOffsetClaimLandcareAndWaterFacilityBroughtForwardA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "TaxOffsetClaimLandcareAndWaterFacilityBroughtForwardA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.TRT147 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion TaxOffsetClaimLandcareAndWaterFacilityBroughtForwardA
#region InternationalDealingsRelatedPartiesTransactionsExcessAggregateValueI
                        _currentXPath.Push("/tns:InternationalDealingsRelatedPartiesTransactionsExcessAggregateValueI");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "InternationalDealingsRelatedPartiesTransactionsExcessAggregateValueI" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    currentBooleanValue = ToBoolean(currentValue);
                                    if (currentBooleanValue != null)
                                    {
                                        report.TRT148 = currentBooleanValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion InternationalDealingsRelatedPartiesTransactionsExcessAggregateValueI
#region ThinCapitalisation
                        _currentXPath.Push("/tns:ThinCapitalisation");
                        //3. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "ThinCapitalisation" && reader.Depth == _currentXPath.Count - 1)
                        {
                            report.RP_BusinessDetails_Income_ThinCapitalisationCollectionExists = true;
                            report.RP_BusinessDetails_Income_ThinCapitalisationCollectionCount += 1;
                            _found = true;
#region LiabilitiesProvisionsAppliedI
                            _currentXPath.Push("/tns:LiabilitiesProvisionsAppliedI");
                            //6. use case
                            if (ReadToNextElement(reader, true) && reader.LocalName == "LiabilitiesProvisionsAppliedI" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        currentBooleanValue = ToBoolean(currentValue);
                                        if (currentBooleanValue != null)
                                        {
                                            report.TRT449 = currentBooleanValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion LiabilitiesProvisionsAppliedI
#region ExpenseInterestA
                            _currentXPath.Push("/tns:ExpenseInterestA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseInterestA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.TRT182 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion ExpenseInterestA
#region ExpenseRoyaltiesA
                            _currentXPath.Push("/tns:ExpenseRoyaltiesA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseRoyaltiesA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.TRT183 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion ExpenseRoyaltiesA
#region ResidencyTaxPurposesPersonStatusI
                            _currentXPath.Push("/tns:ResidencyTaxPurposesPersonStatusI");
                            //6. use case
                            if (ReadToNextElement(reader, true) && reader.LocalName == "ResidencyTaxPurposesPersonStatusI" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        currentBooleanValue = ToBoolean(currentValue);
                                        if (currentBooleanValue != null)
                                        {
                                            report.TRT149 = currentBooleanValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion ResidencyTaxPurposesPersonStatusI
#region InternationalDealingsTransactionsWithTaxHavenCountriesI
                            _currentXPath.Push("/tns:InternationalDealingsTransactionsWithTaxHavenCountriesI");
                            //6. use case
                            if (ReadToNextElement(reader, true) && reader.LocalName == "InternationalDealingsTransactionsWithTaxHavenCountriesI" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        currentBooleanValue = ToBoolean(currentValue);
                                        if (currentBooleanValue != null)
                                        {
                                            report.TRT151 = currentBooleanValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion InternationalDealingsTransactionsWithTaxHavenCountriesI
                        } // End of if ThinCapitalisation node exists

                        _currentXPath.Pop();
#endregion ThinCapitalisation
#region PersonalServicesIncome
                        _currentXPath.Push("/tns:PersonalServicesIncome");
                        //3. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "PersonalServicesIncome" && reader.Depth == _currentXPath.Count - 1)
                        {
                            report.RP_BusinessDetails_Income_PersonalServicesIncomeCollectionExists = true;
                            report.RP_BusinessDetails_Income_PersonalServicesIncomeCollectionCount += 1;
                            _found = true;
#region IncludedI
                            _currentXPath.Push("/tns:IncludedI");
                            //6. use case
                            if (ReadToNextElement(reader, true) && reader.LocalName == "IncludedI" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        currentBooleanValue = ToBoolean(currentValue);
                                        if (currentBooleanValue != null)
                                        {
                                            report.TRT155 = currentBooleanValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncludedI
#region TotalA
                            _currentXPath.Push("/tns:TotalA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "TotalA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.TRT450 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion TotalA
#region ExpenseDeductionA
                            _currentXPath.Push("/tns:ExpenseDeductionA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseDeductionA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.TRT451 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion ExpenseDeductionA
#region TaxIndividualResultsTestSatisfiedI
                            _currentXPath.Push("/tns:TaxIndividualResultsTestSatisfiedI");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "TaxIndividualResultsTestSatisfiedI" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        currentBooleanValue = ToBoolean(currentValue);
                                        if (currentBooleanValue != null)
                                        {
                                            report.TRT452 = currentBooleanValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion TaxIndividualResultsTestSatisfiedI
#region TaxIndividualPersonalServiceBusinessDeterminationHeldI
                            _currentXPath.Push("/tns:TaxIndividualPersonalServiceBusinessDeterminationHeldI");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "TaxIndividualPersonalServiceBusinessDeterminationHeldI" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        currentBooleanValue = ToBoolean(currentValue);
                                        if (currentBooleanValue != null)
                                        {
                                            report.TRT453 = currentBooleanValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion TaxIndividualPersonalServiceBusinessDeterminationHeldI
#region TaxUnrelatedClientsTestSatisfiedI
                            _currentXPath.Push("/tns:TaxUnrelatedClientsTestSatisfiedI");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "TaxUnrelatedClientsTestSatisfiedI" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        currentBooleanValue = ToBoolean(currentValue);
                                        if (currentBooleanValue != null)
                                        {
                                            report.TRT455 = currentBooleanValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion TaxUnrelatedClientsTestSatisfiedI
#region TaxEmploymentTestSatisfiedI
                            _currentXPath.Push("/tns:TaxEmploymentTestSatisfiedI");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "TaxEmploymentTestSatisfiedI" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        currentBooleanValue = ToBoolean(currentValue);
                                        if (currentBooleanValue != null)
                                        {
                                            report.TRT456 = currentBooleanValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion TaxEmploymentTestSatisfiedI
#region TaxBusinessPremisesTestSatisfiedI
                            _currentXPath.Push("/tns:TaxBusinessPremisesTestSatisfiedI");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "TaxBusinessPremisesTestSatisfiedI" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        currentBooleanValue = ToBoolean(currentValue);
                                        if (currentBooleanValue != null)
                                        {
                                            report.TRT457 = currentBooleanValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion TaxBusinessPremisesTestSatisfiedI
                        } // End of if PersonalServicesIncome node exists

                        _currentXPath.Pop();
#endregion PersonalServicesIncome
#region TaxationOfFinancialArrangements
                        _currentXPath.Push("/tns:TaxationOfFinancialArrangements");
                        //3. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "TaxationOfFinancialArrangements" && reader.Depth == _currentXPath.Count - 1)
                        {
                            report.RP_BusinessDetails_Income_TaxationOfFinancialArrangementsCollectionExists = true;
                            report.RP_BusinessDetails_Income_TaxationOfFinancialArrangementsCollectionCount += 1;
                            _found = true;
#region GainsTotalA
                            _currentXPath.Push("/tns:GainsTotalA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "GainsTotalA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.TRT356 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion GainsTotalA
#region ExpenseLossesTotalA
                            _currentXPath.Push("/tns:ExpenseLossesTotalA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseLossesTotalA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.TRT357 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion ExpenseLossesTotalA
                        } // End of if TaxationOfFinancialArrangements node exists

                        _currentXPath.Pop();
#endregion TaxationOfFinancialArrangements
#region NCMIBusinessIncome
                        _currentXPath.Push("/tns:NCMIBusinessIncome");
                        //3. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "NCMIBusinessIncome" && reader.Depth == _currentXPath.Count - 1)
                        {
                            report.RP_BusinessDetails_Income_NCMIBusinessIncomeCollectionExists = true;
                            report.RP_BusinessDetails_Income_NCMIBusinessIncomeCollectionCount += 1;
                            _found = true;
#region NCMIBusinessIncomePrimaryProduction
                            _currentXPath.Push("/tns:NCMIBusinessIncomePrimaryProduction");
                            //3. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "NCMIBusinessIncomePrimaryProduction" && reader.Depth == _currentXPath.Count - 1)
                            {
                                report.RP_BusinessDetails_Income_NCMIBusinessIncome_NCMIBusinessIncomePrimaryProductionCollectionExists = true;
                                report.RP_BusinessDetails_Income_NCMIBusinessIncome_NCMIBusinessIncomePrimaryProductionCollectionCount += 1;
                                _found = true;
#region NonConcessionalManagedInvestmentTrustA
                                _currentXPath.Push("/tns:NonConcessionalManagedInvestmentTrustA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "NonConcessionalManagedInvestmentTrustA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                report.TRT523 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion NonConcessionalManagedInvestmentTrustA
#region ExcludedFromNonConcessionalManagedInvestmentTrustA
                                _currentXPath.Push("/tns:ExcludedFromNonConcessionalManagedInvestmentTrustA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "ExcludedFromNonConcessionalManagedInvestmentTrustA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                report.TRT524 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion ExcludedFromNonConcessionalManagedInvestmentTrustA
                            } // End of if NCMIBusinessIncomePrimaryProduction node exists

                            _currentXPath.Pop();
#endregion NCMIBusinessIncomePrimaryProduction
#region NCMIBusinessIncomeNonPrimaryProduction
                            _currentXPath.Push("/tns:NCMIBusinessIncomeNonPrimaryProduction");
                            //3. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "NCMIBusinessIncomeNonPrimaryProduction" && reader.Depth == _currentXPath.Count - 1)
                            {
                                report.RP_BusinessDetails_Income_NCMIBusinessIncome_NCMIBusinessIncomeNonPrimaryProductionCollectionExists = true;
                                report.RP_BusinessDetails_Income_NCMIBusinessIncome_NCMIBusinessIncomeNonPrimaryProductionCollectionCount += 1;
                                _found = true;
#region NonConcessionalManagedInvestmentTrustA
                                _currentXPath.Push("/tns:NonConcessionalManagedInvestmentTrustA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "NonConcessionalManagedInvestmentTrustA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                report.TRT525 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion NonConcessionalManagedInvestmentTrustA
#region ExcludedFromNonConcessionalManagedInvestmentTrustA
                                _currentXPath.Push("/tns:ExcludedFromNonConcessionalManagedInvestmentTrustA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "ExcludedFromNonConcessionalManagedInvestmentTrustA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                report.TRT526 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion ExcludedFromNonConcessionalManagedInvestmentTrustA
                            } // End of if NCMIBusinessIncomeNonPrimaryProduction node exists

                            _currentXPath.Pop();
#endregion NCMIBusinessIncomeNonPrimaryProduction
                        } // End of if NCMIBusinessIncome node exists

                        _currentXPath.Pop();
#endregion NCMIBusinessIncome
#region NCMIPartnershipsAndTrusts
                        _currentXPath.Push("/tns:NCMIPartnershipsAndTrusts");
                        //3. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "NCMIPartnershipsAndTrusts" && reader.Depth == _currentXPath.Count - 1)
                        {
                            report.RP_BusinessDetails_Income_NCMIPartnershipsAndTrustsCollectionExists = true;
                            report.RP_BusinessDetails_Income_NCMIPartnershipsAndTrustsCollectionCount += 1;
                            _found = true;
#region NCMIPartnershipsAndTrustsPrimaryProduction
                            _currentXPath.Push("/tns:NCMIPartnershipsAndTrustsPrimaryProduction");
                            //3. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "NCMIPartnershipsAndTrustsPrimaryProduction" && reader.Depth == _currentXPath.Count - 1)
                            {
                                report.RP_BusinessDetails_Income_NCMIPartnershipsAndTrusts_NCMIPartnershipsAndTrustsPrimaryProductionCollectionExists = true;
                                report.RP_BusinessDetails_Income_NCMIPartnershipsAndTrusts_NCMIPartnershipsAndTrustsPrimaryProductionCollectionCount += 1;
                                _found = true;
#region NonConcessionalManagedInvestmentTrustA
                                _currentXPath.Push("/tns:NonConcessionalManagedInvestmentTrustA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "NonConcessionalManagedInvestmentTrustA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                report.TRT527 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion NonConcessionalManagedInvestmentTrustA
#region ExcludedFromNonConcessionalManagedInvestmentTrustA
                                _currentXPath.Push("/tns:ExcludedFromNonConcessionalManagedInvestmentTrustA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "ExcludedFromNonConcessionalManagedInvestmentTrustA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                report.TRT528 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion ExcludedFromNonConcessionalManagedInvestmentTrustA
#region PrimaryProductionShareOfNetIncome
                                _currentXPath.Push("/tns:PrimaryProductionShareOfNetIncome");
                                //3. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "PrimaryProductionShareOfNetIncome" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    report.RP_BusinessDetails_Income_NCMIPartnershipsAndTrusts_NCMIPartnershipsAndTrustsPrimaryProduction_PrimaryProductionShareOfNetIncomeCollectionExists = true;
                                    report.RP_BusinessDetails_Income_NCMIPartnershipsAndTrusts_NCMIPartnershipsAndTrustsPrimaryProduction_PrimaryProductionShareOfNetIncomeCollectionCount += 1;
                                    _found = true;
#region NonConcessionalManagedInvestmentTrustA
                                    _currentXPath.Push("/tns:NonConcessionalManagedInvestmentTrustA");
                                    //6. use case
                                    if (ReadToNextElement(reader, false) && reader.LocalName == "NonConcessionalManagedInvestmentTrustA" && reader.Depth == _currentXPath.Count - 1)
                                    {
                                        if (!IsEmptyOrNilElement(reader))
                                        {
                                            ReadNext(reader);
                                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                            {
                                                currentValue = reader.Value;
                                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                                {
                                                    report.TRT529 = currentDecimalValue;
                                                }
                                            }
                                        }

                                        _found = true;
                                    }

                                    _currentXPath.Pop();
#endregion NonConcessionalManagedInvestmentTrustA
#region ExcludedFromNonConcessionalManagedInvestmentTrustA
                                    _currentXPath.Push("/tns:ExcludedFromNonConcessionalManagedInvestmentTrustA");
                                    //6. use case
                                    if (ReadToNextElement(reader, false) && reader.LocalName == "ExcludedFromNonConcessionalManagedInvestmentTrustA" && reader.Depth == _currentXPath.Count - 1)
                                    {
                                        if (!IsEmptyOrNilElement(reader))
                                        {
                                            ReadNext(reader);
                                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                            {
                                                currentValue = reader.Value;
                                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                                {
                                                    report.TRT530 = currentDecimalValue;
                                                }
                                            }
                                        }

                                        _found = true;
                                    }

                                    _currentXPath.Pop();
#endregion ExcludedFromNonConcessionalManagedInvestmentTrustA
                                } // End of if PrimaryProductionShareOfNetIncome node exists

                                _currentXPath.Pop();
#endregion PrimaryProductionShareOfNetIncome
                            } // End of if NCMIPartnershipsAndTrustsPrimaryProduction node exists

                            _currentXPath.Pop();
#endregion NCMIPartnershipsAndTrustsPrimaryProduction
#region NCMIPartnershipsAndTrustsNonPrimaryProduction
                            _currentXPath.Push("/tns:NCMIPartnershipsAndTrustsNonPrimaryProduction");
                            //3. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "NCMIPartnershipsAndTrustsNonPrimaryProduction" && reader.Depth == _currentXPath.Count - 1)
                            {
                                report.RP_BusinessDetails_Income_NCMIPartnershipsAndTrusts_NCMIPartnershipsAndTrustsNonPrimaryProductionCollectionExists = true;
                                report.RP_BusinessDetails_Income_NCMIPartnershipsAndTrusts_NCMIPartnershipsAndTrustsNonPrimaryProductionCollectionCount += 1;
                                _found = true;
#region NonConcessionalManagedInvestmentTrustA
                                _currentXPath.Push("/tns:NonConcessionalManagedInvestmentTrustA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "NonConcessionalManagedInvestmentTrustA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                report.TRT531 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion NonConcessionalManagedInvestmentTrustA
#region ExcludedFromNonConcessionalManagedInvestmentTrustA
                                _currentXPath.Push("/tns:ExcludedFromNonConcessionalManagedInvestmentTrustA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "ExcludedFromNonConcessionalManagedInvestmentTrustA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                report.TRT532 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion ExcludedFromNonConcessionalManagedInvestmentTrustA
#region NonPrimaryProductionShareOfNetIncome
                                _currentXPath.Push("/tns:NonPrimaryProductionShareOfNetIncome");
                                //3. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "NonPrimaryProductionShareOfNetIncome" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    report.RP_BusinessDetails_Income_NCMIPartnershipsAndTrusts_NCMIPartnershipsAndTrustsNonPrimaryProduction_NonPrimaryProductionShareOfNetIncomeCollectionExists = true;
                                    report.RP_BusinessDetails_Income_NCMIPartnershipsAndTrusts_NCMIPartnershipsAndTrustsNonPrimaryProduction_NonPrimaryProductionShareOfNetIncomeCollectionCount += 1;
                                    _found = true;
#region NonConcessionalManagedInvestmentTrustA
                                    _currentXPath.Push("/tns:NonConcessionalManagedInvestmentTrustA");
                                    //6. use case
                                    if (ReadToNextElement(reader, false) && reader.LocalName == "NonConcessionalManagedInvestmentTrustA" && reader.Depth == _currentXPath.Count - 1)
                                    {
                                        if (!IsEmptyOrNilElement(reader))
                                        {
                                            ReadNext(reader);
                                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                            {
                                                currentValue = reader.Value;
                                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                                {
                                                    report.TRT533 = currentDecimalValue;
                                                }
                                            }
                                        }

                                        _found = true;
                                    }

                                    _currentXPath.Pop();
#endregion NonConcessionalManagedInvestmentTrustA
#region ExcludedFromNonConcessionalManagedInvestmentTrustA
                                    _currentXPath.Push("/tns:ExcludedFromNonConcessionalManagedInvestmentTrustA");
                                    //6. use case
                                    if (ReadToNextElement(reader, false) && reader.LocalName == "ExcludedFromNonConcessionalManagedInvestmentTrustA" && reader.Depth == _currentXPath.Count - 1)
                                    {
                                        if (!IsEmptyOrNilElement(reader))
                                        {
                                            ReadNext(reader);
                                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                            {
                                                currentValue = reader.Value;
                                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                                {
                                                    report.TRT534 = currentDecimalValue;
                                                }
                                            }
                                        }

                                        _found = true;
                                    }

                                    _currentXPath.Pop();
#endregion ExcludedFromNonConcessionalManagedInvestmentTrustA
                                } // End of if NonPrimaryProductionShareOfNetIncome node exists

                                _currentXPath.Pop();
#endregion NonPrimaryProductionShareOfNetIncome
                            } // End of if NCMIPartnershipsAndTrustsNonPrimaryProduction node exists

                            _currentXPath.Pop();
#endregion NCMIPartnershipsAndTrustsNonPrimaryProduction
                        } // End of if NCMIPartnershipsAndTrusts node exists

                        _currentXPath.Pop();
#endregion NCMIPartnershipsAndTrusts
#region NCMICapitalGains
                        _currentXPath.Push("/tns:NCMICapitalGains");
                        //3. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "NCMICapitalGains" && reader.Depth == _currentXPath.Count - 1)
                        {
                            report.RP_BusinessDetails_Income_NCMICapitalGainsCollectionExists = true;
                            report.RP_BusinessDetails_Income_NCMICapitalGainsCollectionCount += 1;
                            _found = true;
#region NonConcessionalManagedInvestmentTrustIncomeA
                            _currentXPath.Push("/tns:NonConcessionalManagedInvestmentTrustIncomeA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "NonConcessionalManagedInvestmentTrustIncomeA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.TRT535 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion NonConcessionalManagedInvestmentTrustIncomeA
#region ExcludedFromNonConcessionalManagedInvestmentTrustIncomeA
                            _currentXPath.Push("/tns:ExcludedFromNonConcessionalManagedInvestmentTrustIncomeA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "ExcludedFromNonConcessionalManagedInvestmentTrustIncomeA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.TRT536 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion ExcludedFromNonConcessionalManagedInvestmentTrustIncomeA
                        } // End of if NCMICapitalGains node exists

                        _currentXPath.Pop();
#endregion NCMICapitalGains
                    } // End of if Income node exists

                    _currentXPath.Pop();
#endregion Income
                } // End of if BusinessDetails node exists

                _currentXPath.Pop();
#endregion BusinessDetails
#region KeyFinancialInformation
                _currentXPath.Push("/tns:KeyFinancialInformation");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "KeyFinancialInformation" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_KeyFinancialInformationCollectionExists = true;
                    report.RP_KeyFinancialInformationCollectionCount += 1;
                    _found = true;
#region AssetsCurrentTotalA
                    _currentXPath.Push("/tns:AssetsCurrentTotalA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "AssetsCurrentTotalA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRT156 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion AssetsCurrentTotalA
#region AssetsTotalA
                    _currentXPath.Push("/tns:AssetsTotalA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "AssetsTotalA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRT157 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion AssetsTotalA
#region LiabilitiesCurrentTotalA
                    _currentXPath.Push("/tns:LiabilitiesCurrentTotalA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesCurrentTotalA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRT158 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion LiabilitiesCurrentTotalA
#region LiabilitiesTotalA
                    _currentXPath.Push("/tns:LiabilitiesTotalA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesTotalA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRT159 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion LiabilitiesTotalA
                } // End of if KeyFinancialInformation node exists

                _currentXPath.Pop();
#endregion KeyFinancialInformation
#region BusinessAndProfessionalInformation
                _currentXPath.Push("/tns:BusinessAndProfessionalInformation");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "BusinessAndProfessionalInformation" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_BusinessAndProfessionalInformationCollectionExists = true;
                    report.RP_BusinessAndProfessionalInformationCollectionCount += 1;
                    _found = true;
#region OrganisationNameDetailsOrganisationalNameT
                    _currentXPath.Push("/tns:OrganisationNameDetailsOrganisationalNameT");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "OrganisationNameDetailsOrganisationalNameT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRT161 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion OrganisationNameDetailsOrganisationalNameT
#region BusinessAddressDetails
                    _currentXPath.Push("/tns:BusinessAddressDetails");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "BusinessAddressDetails" && reader.Depth == _currentXPath.Count - 1)
                    {
                        report.RP_BusinessAndProfessionalInformation_BusinessAddressDetailsCollectionExists = true;
                        report.RP_BusinessAndProfessionalInformation_BusinessAddressDetailsCollectionCount += 1;
                        _found = true;
#region OverseasAddressI
                        _currentXPath.Push("/tns:OverseasAddressI");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "OverseasAddressI" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    currentBooleanValue = ToBoolean(currentValue);
                                    if (currentBooleanValue != null)
                                    {
                                        report.TRT316 = currentBooleanValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion OverseasAddressI
#region Line1T
                        _currentXPath.Push("/tns:Line1T");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "Line1T" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.TRT162 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion Line1T
#region Line2T
                        _currentXPath.Push("/tns:Line2T");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "Line2T" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.TRT309 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion Line2T
#region LocalityNameT
                        _currentXPath.Push("/tns:LocalityNameT");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "LocalityNameT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.TRT163 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion LocalityNameT
#region PostcodeT
                        _currentXPath.Push("/tns:PostcodeT");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "PostcodeT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.TRT165 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion PostcodeT
#region StateOrTerritoryC
                        _currentXPath.Push("/tns:StateOrTerritoryC");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "StateOrTerritoryC" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.TRT164 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion StateOrTerritoryC
#region CountryC
                        _currentXPath.Push("/tns:CountryC");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "CountryC" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.TRT374 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion CountryC
                    } // End of if BusinessAddressDetails node exists

                    _currentXPath.Pop();
#endregion BusinessAddressDetails
#region BusinessAndProfessionaltems
                    _currentXPath.Push("/tns:BusinessAndProfessionaltems");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "BusinessAndProfessionaltems" && reader.Depth == _currentXPath.Count - 1)
                    {
                        report.RP_BusinessAndProfessionalInformation_BusinessAndProfessionaltemsCollectionExists = true;
                        report.RP_BusinessAndProfessionalInformation_BusinessAndProfessionaltemsCollectionCount += 1;
                        _found = true;
#region AssetsInventoriesTotalA
                        _currentXPath.Push("/tns:AssetsInventoriesTotalA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "AssetsInventoriesTotalA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.TRT166 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion AssetsInventoriesTotalA
#region ExpensePurchasesAndOtherCostsA
                        _currentXPath.Push("/tns:ExpensePurchasesAndOtherCostsA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ExpensePurchasesAndOtherCostsA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.TRT167 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion ExpensePurchasesAndOtherCostsA
#region RemunerationWagesAndSalariesTotalA
                        _currentXPath.Push("/tns:RemunerationWagesAndSalariesTotalA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationWagesAndSalariesTotalA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.TRT172 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion RemunerationWagesAndSalariesTotalA
#region RemunerationWagesAndSalariesActionC
                        _currentXPath.Push("/tns:RemunerationWagesAndSalariesActionC");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationWagesAndSalariesActionC" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.TRT173 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion RemunerationWagesAndSalariesActionC
#region RemunerationPaymentToRelatedPartiesGrossA
                        _currentXPath.Push("/tns:RemunerationPaymentToRelatedPartiesGrossA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationPaymentToRelatedPartiesGrossA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.TRT174 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion RemunerationPaymentToRelatedPartiesGrossA
#region RemunerationFringeBenefitsEmployeeContributionA
                        _currentXPath.Push("/tns:RemunerationFringeBenefitsEmployeeContributionA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "RemunerationFringeBenefitsEmployeeContributionA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.TRT181 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion RemunerationFringeBenefitsEmployeeContributionA
#region IncomeTaxPrivateCompanyUnpaidPresentEntitlementA
                        _currentXPath.Push("/tns:IncomeTaxPrivateCompanyUnpaidPresentEntitlementA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxPrivateCompanyUnpaidPresentEntitlementA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.TRT186 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion IncomeTaxPrivateCompanyUnpaidPresentEntitlementA
#region IncomeTaxPrivateCompanyUnpaidPresentEntitlementC
                        _currentXPath.Push("/tns:IncomeTaxPrivateCompanyUnpaidPresentEntitlementC");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxPrivateCompanyUnpaidPresentEntitlementC" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.TRT187 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion IncomeTaxPrivateCompanyUnpaidPresentEntitlementC
#region ElectionsTradingStockI
                        _currentXPath.Push("/tns:ElectionsTradingStockI");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ElectionsTradingStockI" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    currentBooleanValue = ToBoolean(currentValue);
                                    if (currentBooleanValue != null)
                                    {
                                        report.TRT188 = currentBooleanValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion ElectionsTradingStockI
#region Closing
                        _currentXPath.Push("/tns:Closing");
                        //3. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "Closing" && reader.Depth == _currentXPath.Count - 1)
                        {
                            report.RP_BusinessAndProfessionalInformation_BusinessAndProfessionaltems_ClosingCollectionExists = true;
                            report.RP_BusinessAndProfessionalInformation_BusinessAndProfessionaltems_ClosingCollectionCount += 1;
                            _found = true;
#region AssetsInventoriesTotalA
                            _currentXPath.Push("/tns:AssetsInventoriesTotalA");
                            //6. use case
                            if (ReadToNextElement(reader, true) && reader.LocalName == "AssetsInventoriesTotalA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.TRT168 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion AssetsInventoriesTotalA
#region AssetsInventoriesValuationMethodC
                            _currentXPath.Push("/tns:AssetsInventoriesValuationMethodC");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "AssetsInventoriesValuationMethodC" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        report.TRT169 = currentValue;
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion AssetsInventoriesValuationMethodC
#region AssetsTradeandReceivablesOtherA
                            _currentXPath.Push("/tns:AssetsTradeandReceivablesOtherA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "AssetsTradeandReceivablesOtherA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.TRT170 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion AssetsTradeandReceivablesOtherA
#region LiabilitiesTradeAndOtherPayablesTotalA
                            _currentXPath.Push("/tns:LiabilitiesTradeAndOtherPayablesTotalA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "LiabilitiesTradeAndOtherPayablesTotalA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            report.TRT171 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion LiabilitiesTradeAndOtherPayablesTotalA
                        } // End of if Closing node exists

                        _currentXPath.Pop();
#endregion Closing
                    } // End of if BusinessAndProfessionaltems node exists

                    _currentXPath.Pop();
#endregion BusinessAndProfessionaltems
#region AggregatedTurnover
                    _currentXPath.Push("/tns:AggregatedTurnover");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "AggregatedTurnover" && reader.Depth == _currentXPath.Count - 1)
                    {
                        report.RP_BusinessAndProfessionalInformation_AggregatedTurnoverCollectionExists = true;
                        report.RP_BusinessAndProfessionalInformation_AggregatedTurnoverCollectionCount += 1;
                        _found = true;
#region IncomeRangeC
                        _currentXPath.Push("/tns:IncomeRangeC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "IncomeRangeC" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    report.TRT563 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion IncomeRangeC
#region IncomeA
                        _currentXPath.Push("/tns:IncomeA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.TRT564 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion IncomeA
                    } // End of if AggregatedTurnover node exists

                    _currentXPath.Pop();
#endregion AggregatedTurnover
#region CapitalAllowances
                    _currentXPath.Push("/tns:CapitalAllowances");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "CapitalAllowances" && reader.Depth == _currentXPath.Count - 1)
                    {
                        report.RP_BusinessAndProfessionalInformation_CapitalAllowancesCollectionExists = true;
                        report.RP_BusinessAndProfessionalInformation_CapitalAllowancesCollectionCount += 1;
                        _found = true;
#region ExpenseDepreciatingAssetsIntangibleFirstDeductedA
                        _currentXPath.Push("/tns:ExpenseDepreciatingAssetsIntangibleFirstDeductedA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseDepreciatingAssetsIntangibleFirstDeductedA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.TRT175 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion ExpenseDepreciatingAssetsIntangibleFirstDeductedA
#region ExpenseDepreciatingAssetsOtherFirstDeductedA
                        _currentXPath.Push("/tns:ExpenseDepreciatingAssetsOtherFirstDeductedA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseDepreciatingAssetsOtherFirstDeductedA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.TRT176 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion ExpenseDepreciatingAssetsOtherFirstDeductedA
#region AssetsPropertyPlantAndEquipmentDepreciatingAssetsFirstDeductedSelfAssessedEffectiveLifeI
                        _currentXPath.Push("/tns:AssetsPropertyPlantAndEquipmentDepreciatingAssetsFirstDeductedSelfAssessedEffectiveLifeI");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "AssetsPropertyPlantAndEquipmentDepreciatingAssetsFirstDeductedSelfAssessedEffectiveLifeI" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    currentBooleanValue = ToBoolean(currentValue);
                                    if (currentBooleanValue != null)
                                    {
                                        report.TRT458 = currentBooleanValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion AssetsPropertyPlantAndEquipmentDepreciatingAssetsFirstDeductedSelfAssessedEffectiveLifeI
#region ExpenseSubsequentYearAcceleratedDepreciationDeductionsForAssetsA
                        _currentXPath.Push("/tns:ExpenseSubsequentYearAcceleratedDepreciationDeductionsForAssetsA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseSubsequentYearAcceleratedDepreciationDeductionsForAssetsA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.TRT561 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion ExpenseSubsequentYearAcceleratedDepreciationDeductionsForAssetsA
                    } // End of if CapitalAllowances node exists

                    _currentXPath.Pop();
#endregion CapitalAllowances
#region CapitalAllowancesDepreciatingAssets
                    _currentXPath.Push("/tns:CapitalAllowancesDepreciatingAssets");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "CapitalAllowancesDepreciatingAssets" && reader.Depth == _currentXPath.Count - 1)
                    {
                        report.RP_BusinessAndProfessionalInformation_CapitalAllowancesDepreciatingAssetsCollectionExists = true;
                        report.RP_BusinessAndProfessionalInformation_CapitalAllowancesDepreciatingAssetsCollectionCount += 1;
                        _found = true;
#region PropertyPlantAndEquipmentEffectiveLifeRecalculationI
                        _currentXPath.Push("/tns:PropertyPlantAndEquipmentEffectiveLifeRecalculationI");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "PropertyPlantAndEquipmentEffectiveLifeRecalculationI" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    currentBooleanValue = ToBoolean(currentValue);
                                    if (currentBooleanValue != null)
                                    {
                                        report.TRT459 = currentBooleanValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion PropertyPlantAndEquipmentEffectiveLifeRecalculationI
#region AdjustableValuesTotalA
                        _currentXPath.Push("/tns:AdjustableValuesTotalA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "AdjustableValuesTotalA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.TRT460 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion AdjustableValuesTotalA
#region IncomeIntangibleBalancingAdjustmentA
                        _currentXPath.Push("/tns:IncomeIntangibleBalancingAdjustmentA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeIntangibleBalancingAdjustmentA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.TRT461 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion IncomeIntangibleBalancingAdjustmentA
#region ExpenseIntangibleBalancingAdjustmentA
                        _currentXPath.Push("/tns:ExpenseIntangibleBalancingAdjustmentA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseIntangibleBalancingAdjustmentA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.TRT462 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion ExpenseIntangibleBalancingAdjustmentA
#region IntangibleTerminationValueA
                        _currentXPath.Push("/tns:IntangibleTerminationValueA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "IntangibleTerminationValueA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.TRT177 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion IntangibleTerminationValueA
#region OtherTerminationValueA
                        _currentXPath.Push("/tns:OtherTerminationValueA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "OtherTerminationValueA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.TRT178 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion OtherTerminationValueA
#region ExpenseProjectPoolAllowableDeductionA
                        _currentXPath.Push("/tns:ExpenseProjectPoolAllowableDeductionA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseProjectPoolAllowableDeductionA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.TRT179 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion ExpenseProjectPoolAllowableDeductionA
#region ExpenseCapitalExpenditureSpecifiedAllowableDeductionA
                        _currentXPath.Push("/tns:ExpenseCapitalExpenditureSpecifiedAllowableDeductionA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseCapitalExpenditureSpecifiedAllowableDeductionA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.TRT180 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion ExpenseCapitalExpenditureSpecifiedAllowableDeductionA
#region ExpenseLandcareOperationsAcceleratedDepreciationA
                        _currentXPath.Push("/tns:ExpenseLandcareOperationsAcceleratedDepreciationA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseLandcareOperationsAcceleratedDepreciationA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.TRT184 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion ExpenseLandcareOperationsAcceleratedDepreciationA
                    } // End of if CapitalAllowancesDepreciatingAssets node exists

                    _currentXPath.Pop();
#endregion CapitalAllowancesDepreciatingAssets
#region LowCostDeductionExpenseDepreciatingAssetsSmallBusinessEntityA
                    _currentXPath.Push("/tns:LowCostDeductionExpenseDepreciatingAssetsSmallBusinessEntityA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "LowCostDeductionExpenseDepreciatingAssetsSmallBusinessEntityA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRT189 = currentDecimalValue;
                                }

                                report.RP_BusinessAndProfessionalInformation_LowCostDeductionCollectionExists = true;
                                report.RP_BusinessAndProfessionalInformation_LowCostDeductionCollectionCount += 1;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion LowCostDeductionExpenseDepreciatingAssetsSmallBusinessEntityA
#region GeneralPoolDeductionExpenseDepreciatingAssetsSmallBusinessEntityA
                    _currentXPath.Push("/tns:GeneralPoolDeductionExpenseDepreciatingAssetsSmallBusinessEntityA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "GeneralPoolDeductionExpenseDepreciatingAssetsSmallBusinessEntityA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRT190 = currentDecimalValue;
                                }

                                report.RP_BusinessAndProfessionalInformation_GeneralPoolDeductionCollectionExists = true;
                                report.RP_BusinessAndProfessionalInformation_GeneralPoolDeductionCollectionCount += 1;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion GeneralPoolDeductionExpenseDepreciatingAssetsSmallBusinessEntityA
                } // End of if BusinessAndProfessionalInformation node exists

                _currentXPath.Pop();
#endregion BusinessAndProfessionalInformation
#region TaxOffsets
                _currentXPath.Push("/tns:TaxOffsets");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "TaxOffsets" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_TaxOffsetsCollectionExists = true;
                    report.RP_TaxOffsetsCollectionCount += 1;
                    _found = true;
#region TaxOffsetClaimNationalRentalAffordabilitySchemeEntitlementA
                    _currentXPath.Push("/tns:TaxOffsetClaimNationalRentalAffordabilitySchemeEntitlementA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "TaxOffsetClaimNationalRentalAffordabilitySchemeEntitlementA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRT340 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion TaxOffsetClaimNationalRentalAffordabilitySchemeEntitlementA
#region IncomeTaxTaxOffsetRefundableOtherA
                    _currentXPath.Push("/tns:IncomeTaxTaxOffsetRefundableOtherA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxTaxOffsetRefundableOtherA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRT463 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion IncomeTaxTaxOffsetRefundableOtherA
#region IncomeTaxTaxOffsetRefundableOtherC
                    _currentXPath.Push("/tns:IncomeTaxTaxOffsetRefundableOtherC");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxTaxOffsetRefundableOtherC" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRT464 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion IncomeTaxTaxOffsetRefundableOtherC
#region LimitedPartnershipTaxOffsetTaxOffsetClaimNonRefundableOtherA
                    _currentXPath.Push("/tns:LimitedPartnershipTaxOffsetTaxOffsetClaimNonRefundableOtherA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "LimitedPartnershipTaxOffsetTaxOffsetClaimNonRefundableOtherA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRT500 = currentDecimalValue;
                                }

                                report.RP_TaxOffsets_LimitedPartnershipTaxOffsetCollectionExists = true;
                                report.RP_TaxOffsets_LimitedPartnershipTaxOffsetCollectionCount += 1;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion LimitedPartnershipTaxOffsetTaxOffsetClaimNonRefundableOtherA
#region InvestorTaxOffsetTaxOffsetClaimNonRefundableOtherA
                    _currentXPath.Push("/tns:InvestorTaxOffsetTaxOffsetClaimNonRefundableOtherA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "InvestorTaxOffsetTaxOffsetClaimNonRefundableOtherA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRT501 = currentDecimalValue;
                                }

                                report.RP_TaxOffsets_InvestorTaxOffsetCollectionExists = true;
                                report.RP_TaxOffsets_InvestorTaxOffsetCollectionCount += 1;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion InvestorTaxOffsetTaxOffsetClaimNonRefundableOtherA
                } // End of if TaxOffsets node exists

                _currentXPath.Pop();
#endregion TaxOffsets
#region MedicareLevy
                _currentXPath.Push("/tns:MedicareLevy");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "MedicareLevy" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_MedicareLevyCollectionExists = true;
                    report.RP_MedicareLevyCollectionCount += 1;
                    _found = true;
#region IncomeTaxBeneficiarySpouseTaxableIncomeA
                    _currentXPath.Push("/tns:IncomeTaxBeneficiarySpouseTaxableIncomeA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxBeneficiarySpouseTaxableIncomeA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRT194 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion IncomeTaxBeneficiarySpouseTaxableIncomeA
#region IncomeTaxDependentChildrenAndStudentsCt
                    _currentXPath.Push("/tns:IncomeTaxDependentChildrenAndStudentsCt");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxDependentChildrenAndStudentsCt" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRT195 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion IncomeTaxDependentChildrenAndStudentsCt
#region IncomeTaxExemptionFullDaysCt
                    _currentXPath.Push("/tns:IncomeTaxExemptionFullDaysCt");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxExemptionFullDaysCt" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRT196 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion IncomeTaxExemptionFullDaysCt
#region IncomeTaxExemptionFullDaysC
                    _currentXPath.Push("/tns:IncomeTaxExemptionFullDaysC");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxExemptionFullDaysC" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRT197 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion IncomeTaxExemptionFullDaysC
#region IncomeTaxExemptionHalfDaysCt
                    _currentXPath.Push("/tns:IncomeTaxExemptionHalfDaysCt");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxExemptionHalfDaysCt" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRT198 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion IncomeTaxExemptionHalfDaysCt
                } // End of if MedicareLevy node exists

                _currentXPath.Pop();
#endregion MedicareLevy
#region IncomeTrustEstateIncomeA
                _currentXPath.Push("/tns:IncomeTrustEstateIncomeA");
                //6. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "IncomeTrustEstateIncomeA" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                            {
                                report.TRT440 = currentDecimalValue;
                            }
                        }
                    }

                    _found = true;
                }

                _currentXPath.Pop();
#endregion IncomeTrustEstateIncomeA
#region StatementOfDistributionToBeneficiary
                _currentXPath.Push("/tns:StatementOfDistributionToBeneficiaryCollection");
                // 4. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "StatementOfDistributionToBeneficiaryCollection" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_StatementOfDistributionToBeneficiaryCollection = new List<TRT2026.RP_StatementOfDistributionToBeneficiary>();
                    report.RP_StatementOfDistributionToBeneficiaryCollectionExists = true;
                    _found = true;
                    while (ReadToNextElement(reader, false) && reader.LocalName == "StatementOfDistributionToBeneficiary")
                    {
                        _found = true;
                        TRT2026.RP_StatementOfDistributionToBeneficiary statementOfDistributionToBeneficiary = new TRT2026.RP_StatementOfDistributionToBeneficiary();
                        report.RP_StatementOfDistributionToBeneficiaryCollection.Add(statementOfDistributionToBeneficiary);
                        report.RP_StatementOfDistributionToBeneficiaryCollectionCount += 1;
                        statementOfDistributionToBeneficiary.OccurrenceIndex = report.RP_StatementOfDistributionToBeneficiaryCollectionCount;
                        _currentXPath.Push("/tns:StatementOfDistributionToBeneficiary[" + report.RP_StatementOfDistributionToBeneficiaryCollectionCount + "]");
#region TaxFileNumberId
                        _currentXPath.Push("/tns:TaxFileNumberId");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "TaxFileNumberId" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    statementOfDistributionToBeneficiary.TRT211 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion TaxFileNumberId
#region TaxFileNumberTrailingNumericId
                        _currentXPath.Push("/tns:TaxFileNumberTrailingNumericId");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "TaxFileNumberTrailingNumericId" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    statementOfDistributionToBeneficiary.TRT212 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion TaxFileNumberTrailingNumericId
#region IncomeTaxBeneficiaryEntityTypeC
                        _currentXPath.Push("/tns:IncomeTaxBeneficiaryEntityTypeC");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "IncomeTaxBeneficiaryEntityTypeC" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    statementOfDistributionToBeneficiary.TRT432 = currentValue;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion IncomeTaxBeneficiaryEntityTypeC
#region NonIndividualNameDetailsOrganisationNameDetailsOrganisationalNameT
                        _currentXPath.Push("/tns:NonIndividualNameDetailsOrganisationNameDetailsOrganisationalNameT");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "NonIndividualNameDetailsOrganisationNameDetailsOrganisationalNameT" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    statementOfDistributionToBeneficiary.TRT203 = currentValue;
                                    statementOfDistributionToBeneficiary.RP_StatementOfDistributionToBeneficiary_NonIndividualNameDetailsCollectionExists = true;
                                    statementOfDistributionToBeneficiary.RP_StatementOfDistributionToBeneficiary_NonIndividualNameDetailsCollectionCount += 1;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion NonIndividualNameDetailsOrganisationNameDetailsOrganisationalNameT
#region IndividualNameDetails
                        _currentXPath.Push("/tns:IndividualNameDetails");
                        //3. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "IndividualNameDetails" && reader.Depth == _currentXPath.Count - 1)
                        {
                            statementOfDistributionToBeneficiary.RP_StatementOfDistributionToBeneficiary_IndividualNameDetailsCollectionExists = true;
                            statementOfDistributionToBeneficiary.RP_StatementOfDistributionToBeneficiary_IndividualNameDetailsCollectionCount += 1;
                            _found = true;
#region PersonNameDetailsTitleT
                            _currentXPath.Push("/tns:PersonNameDetailsTitleT");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "PersonNameDetailsTitleT" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        statementOfDistributionToBeneficiary.TRT378 = currentValue;
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion PersonNameDetailsTitleT
#region PersonNameDetailsNameSuffixT
                            _currentXPath.Push("/tns:PersonNameDetailsNameSuffixT");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "PersonNameDetailsNameSuffixT" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        statementOfDistributionToBeneficiary.TRT379 = currentValue;
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion PersonNameDetailsNameSuffixT
#region PersonNameDetailsFamilyNameT
                            _currentXPath.Push("/tns:PersonNameDetailsFamilyNameT");
                            //6. use case
                            if (ReadToNextElement(reader, true) && reader.LocalName == "PersonNameDetailsFamilyNameT" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        statementOfDistributionToBeneficiary.TRT200 = currentValue;
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion PersonNameDetailsFamilyNameT
#region PersonNameDetailsGivenNameT
                            _currentXPath.Push("/tns:PersonNameDetailsGivenNameT");
                            //6. use case
                            if (ReadToNextElement(reader, true) && reader.LocalName == "PersonNameDetailsGivenNameT" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        statementOfDistributionToBeneficiary.TRT201 = currentValue;
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion PersonNameDetailsGivenNameT
#region PersonNameDetailsOtherGivenNameT
                            _currentXPath.Push("/tns:PersonNameDetailsOtherGivenNameT");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "PersonNameDetailsOtherGivenNameT" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        statementOfDistributionToBeneficiary.TRT202 = currentValue;
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion PersonNameDetailsOtherGivenNameT
#region PersonDemographicDetailsBirthD
                            _currentXPath.Push("/tns:PersonDemographicDetailsBirthD");
                            //6. use case
                            if (ReadToNextElement(reader, true) && reader.LocalName == "PersonDemographicDetailsBirthD" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                                        {
                                            statementOfDistributionToBeneficiary.TRT213 = currentDateTimeValue.Date;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion PersonDemographicDetailsBirthD
                        } // End of if IndividualNameDetails node exists

                        _currentXPath.Pop();
#endregion IndividualNameDetails
#region BeneficiaryAddressDetails
                        _currentXPath.Push("/tns:BeneficiaryAddressDetails");
                        //3. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "BeneficiaryAddressDetails" && reader.Depth == _currentXPath.Count - 1)
                        {
                            statementOfDistributionToBeneficiary.RP_StatementOfDistributionToBeneficiary_BeneficiaryAddressDetailsCollectionExists = true;
                            statementOfDistributionToBeneficiary.RP_StatementOfDistributionToBeneficiary_BeneficiaryAddressDetailsCollectionCount += 1;
                            _found = true;
#region OverseasAddressI
                            _currentXPath.Push("/tns:OverseasAddressI");
                            //6. use case
                            if (ReadToNextElement(reader, true) && reader.LocalName == "OverseasAddressI" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        currentBooleanValue = ToBoolean(currentValue);
                                        if (currentBooleanValue != null)
                                        {
                                            statementOfDistributionToBeneficiary.TRT320 = currentBooleanValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion OverseasAddressI
#region UsageC
                            _currentXPath.Push("/tns:UsageC");
                            //6. use case
                            if (ReadToNextElement(reader, true) && reader.LocalName == "UsageC" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        statementOfDistributionToBeneficiary.TRT327 = currentValue;
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion UsageC
#region Line1T
                            _currentXPath.Push("/tns:Line1T");
                            //6. use case
                            if (ReadToNextElement(reader, true) && reader.LocalName == "Line1T" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        statementOfDistributionToBeneficiary.TRT205 = currentValue;
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion Line1T
#region Line2T
                            _currentXPath.Push("/tns:Line2T");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "Line2T" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        statementOfDistributionToBeneficiary.TRT206 = currentValue;
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion Line2T
#region LocalityNameT
                            _currentXPath.Push("/tns:LocalityNameT");
                            //6. use case
                            if (ReadToNextElement(reader, true) && reader.LocalName == "LocalityNameT" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        statementOfDistributionToBeneficiary.TRT207 = currentValue;
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion LocalityNameT
#region PostcodeT
                            _currentXPath.Push("/tns:PostcodeT");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "PostcodeT" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        statementOfDistributionToBeneficiary.TRT209 = currentValue;
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion PostcodeT
#region StateOrTerritoryC
                            _currentXPath.Push("/tns:StateOrTerritoryC");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "StateOrTerritoryC" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        statementOfDistributionToBeneficiary.TRT208 = currentValue;
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion StateOrTerritoryC
#region CountryC
                            _currentXPath.Push("/tns:CountryC");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "CountryC" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        statementOfDistributionToBeneficiary.TRT382 = currentValue;
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion CountryC
                        } // End of if BeneficiaryAddressDetails node exists

                        _currentXPath.Pop();
#endregion BeneficiaryAddressDetails
#region Information
                        _currentXPath.Push("/tns:Information");
                        //3. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "Information" && reader.Depth == _currentXPath.Count - 1)
                        {
                            statementOfDistributionToBeneficiary.RP_StatementOfDistributionToBeneficiary_InformationCollectionExists = true;
                            statementOfDistributionToBeneficiary.RP_StatementOfDistributionToBeneficiary_InformationCollectionCount += 1;
                            _found = true;
#region IncomeTaxAssessmentCalculationC
                            _currentXPath.Push("/tns:IncomeTaxAssessmentCalculationC");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxAssessmentCalculationC" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        statementOfDistributionToBeneficiary.TRT214 = currentValue;
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxAssessmentCalculationC
#region IncomeTrustEstateIncomeShareA
                            _currentXPath.Push("/tns:IncomeTrustEstateIncomeShareA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTrustEstateIncomeShareA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            statementOfDistributionToBeneficiary.TRT441 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTrustEstateIncomeShareA
#region IncomeTaxPayAsYouGoWithholdingCreditForAmountsWithheldFromForeignResidentsA
                            _currentXPath.Push("/tns:IncomeTaxPayAsYouGoWithholdingCreditForAmountsWithheldFromForeignResidentsA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxPayAsYouGoWithholdingCreditForAmountsWithheldFromForeignResidentsA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            statementOfDistributionToBeneficiary.TRT476 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxPayAsYouGoWithholdingCreditForAmountsWithheldFromForeignResidentsA
#region IncomeTaxFrankingCreditsReceivedFromNewZealandCompaniesA
                            _currentXPath.Push("/tns:IncomeTaxFrankingCreditsReceivedFromNewZealandCompaniesA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxFrankingCreditsReceivedFromNewZealandCompaniesA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            statementOfDistributionToBeneficiary.TRT217 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxFrankingCreditsReceivedFromNewZealandCompaniesA
#region IncomeTaxPayAsYouGoWithholdingCreditForTaxWithheldWhereABNNotQuotedA
                            _currentXPath.Push("/tns:IncomeTaxPayAsYouGoWithholdingCreditForTaxWithheldWhereABNNotQuotedA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxPayAsYouGoWithholdingCreditForTaxWithheldWhereABNNotQuotedA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            statementOfDistributionToBeneficiary.TRT220 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxPayAsYouGoWithholdingCreditForTaxWithheldWhereABNNotQuotedA
#region IncomeTaxableMinorBeneficiaryA
                            _currentXPath.Push("/tns:IncomeTaxableMinorBeneficiaryA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxableMinorBeneficiaryA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            statementOfDistributionToBeneficiary.TRT549 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxableMinorBeneficiaryA
#region IncomeDistributionFrankedA
                            _currentXPath.Push("/tns:IncomeDistributionFrankedA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeDistributionFrankedA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            statementOfDistributionToBeneficiary.TRT442 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeDistributionFrankedA
#region IncomeTrustFrankedInvestmentNonPrimaryProductionShareA
                            _currentXPath.Push("/tns:IncomeTrustFrankedInvestmentNonPrimaryProductionShareA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTrustFrankedInvestmentNonPrimaryProductionShareA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            statementOfDistributionToBeneficiary.TRT580 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTrustFrankedInvestmentNonPrimaryProductionShareA
#region IncomeTaxFrankingCreditsReceivedFromAustralianCompaniesA
                            _currentXPath.Push("/tns:IncomeTaxFrankingCreditsReceivedFromAustralianCompaniesA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxFrankingCreditsReceivedFromAustralianCompaniesA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            statementOfDistributionToBeneficiary.TRT221 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxFrankingCreditsReceivedFromAustralianCompaniesA
#region IncomeTaxPayAsYouGoWithholdingCreditForAmountsWithheldWhereTFNNotQuotedA
                            _currentXPath.Push("/tns:IncomeTaxPayAsYouGoWithholdingCreditForAmountsWithheldWhereTFNNotQuotedA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxPayAsYouGoWithholdingCreditForAmountsWithheldWhereTFNNotQuotedA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            statementOfDistributionToBeneficiary.TRT222 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxPayAsYouGoWithholdingCreditForAmountsWithheldWhereTFNNotQuotedA
#region IncomeTaxPayAsYouGoWithholdingCreditForTaxWithheldFromCloselyHeldTrustA
                            _currentXPath.Push("/tns:IncomeTaxPayAsYouGoWithholdingCreditForTaxWithheldFromCloselyHeldTrustA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxPayAsYouGoWithholdingCreditForTaxWithheldFromCloselyHeldTrustA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            statementOfDistributionToBeneficiary.TRT419 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxPayAsYouGoWithholdingCreditForTaxWithheldFromCloselyHeldTrustA
#region IncomeCapitalGainsNetA
                            _currentXPath.Push("/tns:IncomeCapitalGainsNetA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeCapitalGainsNetA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            statementOfDistributionToBeneficiary.TRT223 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeCapitalGainsNetA
#region CapitalGainsTotalA
                            _currentXPath.Push("/tns:CapitalGainsTotalA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "CapitalGainsTotalA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            statementOfDistributionToBeneficiary.TRT569 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion CapitalGainsTotalA
#region CapitalLossesTotalA
                            _currentXPath.Push("/tns:CapitalLossesTotalA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "CapitalLossesTotalA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            statementOfDistributionToBeneficiary.TRT570 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion CapitalLossesTotalA
#region TaxConcessionCapitalGainsDiscountTotalA
                            _currentXPath.Push("/tns:TaxConcessionCapitalGainsDiscountTotalA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "TaxConcessionCapitalGainsDiscountTotalA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            statementOfDistributionToBeneficiary.TRT571 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion TaxConcessionCapitalGainsDiscountTotalA
#region TaxConcessionCapitalGainsDiscountAndSmallBusinessConcessionsTotalA
                            _currentXPath.Push("/tns:TaxConcessionCapitalGainsDiscountAndSmallBusinessConcessionsTotalA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "TaxConcessionCapitalGainsDiscountAndSmallBusinessConcessionsTotalA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            statementOfDistributionToBeneficiary.TRT572 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion TaxConcessionCapitalGainsDiscountAndSmallBusinessConcessionsTotalA
#region CapitalGainsNonConcessionalManagedInvestmentTrustIncomeA
                            _currentXPath.Push("/tns:CapitalGainsNonConcessionalManagedInvestmentTrustIncomeA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "CapitalGainsNonConcessionalManagedInvestmentTrustIncomeA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            statementOfDistributionToBeneficiary.TRT541 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion CapitalGainsNonConcessionalManagedInvestmentTrustIncomeA
#region CapitalGainsExcludedFromNonConcessionalManagedInvestmentTrustIncomeA
                            _currentXPath.Push("/tns:CapitalGainsExcludedFromNonConcessionalManagedInvestmentTrustIncomeA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "CapitalGainsExcludedFromNonConcessionalManagedInvestmentTrustIncomeA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            statementOfDistributionToBeneficiary.TRT542 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion CapitalGainsExcludedFromNonConcessionalManagedInvestmentTrustIncomeA
#region IncomeTaxPayAsYouGoWithholdingCreditForCapitalGainsWithheldFromForeignResidentsA
                            _currentXPath.Push("/tns:IncomeTaxPayAsYouGoWithholdingCreditForCapitalGainsWithheldFromForeignResidentsA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxPayAsYouGoWithholdingCreditForCapitalGainsWithheldFromForeignResidentsA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            statementOfDistributionToBeneficiary.TRT481 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxPayAsYouGoWithholdingCreditForCapitalGainsWithheldFromForeignResidentsA
#region IncomeTaxAttributedForeignIncomeA
                            _currentXPath.Push("/tns:IncomeTaxAttributedForeignIncomeA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxAttributedForeignIncomeA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            statementOfDistributionToBeneficiary.TRT224 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxAttributedForeignIncomeA
#region IncomeInternationalDealingsNetA
                            _currentXPath.Push("/tns:IncomeInternationalDealingsNetA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeInternationalDealingsNetA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            statementOfDistributionToBeneficiary.TRT225 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeInternationalDealingsNetA
#region IncomeInternationalDealingsFinancialInvestmentsNetA
                            _currentXPath.Push("/tns:IncomeInternationalDealingsFinancialInvestmentsNetA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeInternationalDealingsFinancialInvestmentsNetA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            statementOfDistributionToBeneficiary.TRT581 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeInternationalDealingsFinancialInvestmentsNetA
#region IncomeInternationalDealingsTaxOffsetA
                            _currentXPath.Push("/tns:IncomeInternationalDealingsTaxOffsetA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeInternationalDealingsTaxOffsetA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            statementOfDistributionToBeneficiary.TRT226 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeInternationalDealingsTaxOffsetA
#region TaxOffsetClaimNationalRentalAffordabilitySchemeEntitlementA
                            _currentXPath.Push("/tns:TaxOffsetClaimNationalRentalAffordabilitySchemeEntitlementA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "TaxOffsetClaimNationalRentalAffordabilitySchemeEntitlementA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            statementOfDistributionToBeneficiary.TRT342 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion TaxOffsetClaimNationalRentalAffordabilitySchemeEntitlementA
#region IncomeTaxExplorationCreditsDistributedA
                            _currentXPath.Push("/tns:IncomeTaxExplorationCreditsDistributedA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxExplorationCreditsDistributedA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            statementOfDistributionToBeneficiary.TRT465 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxExplorationCreditsDistributedA
#region IncomeSmallBusinessEntityNetA
                            _currentXPath.Push("/tns:IncomeSmallBusinessEntityNetA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeSmallBusinessEntityNetA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            statementOfDistributionToBeneficiary.TRT479 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeSmallBusinessEntityNetA
#region ShareOfIncomePrimaryProduction
                            _currentXPath.Push("/tns:ShareOfIncomePrimaryProduction");
                            //3. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "ShareOfIncomePrimaryProduction" && reader.Depth == _currentXPath.Count - 1)
                            {
                                statementOfDistributionToBeneficiary.RP_StatementOfDistributionToBeneficiary_Information_ShareOfIncomePrimaryProductionCollectionExists = true;
                                statementOfDistributionToBeneficiary.RP_StatementOfDistributionToBeneficiary_Information_ShareOfIncomePrimaryProductionCollectionCount += 1;
                                _found = true;
#region BeneficiaryShareA
                                _currentXPath.Push("/tns:BeneficiaryShareA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "BeneficiaryShareA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                statementOfDistributionToBeneficiary.TRT218 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion BeneficiaryShareA
#region NonConcessionalManagedInvestmentTrustA
                                _currentXPath.Push("/tns:NonConcessionalManagedInvestmentTrustA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "NonConcessionalManagedInvestmentTrustA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                statementOfDistributionToBeneficiary.TRT537 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion NonConcessionalManagedInvestmentTrustA
#region ExcludedFromNonConcessionalManagedInvestmentTrustA
                                _currentXPath.Push("/tns:ExcludedFromNonConcessionalManagedInvestmentTrustA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "ExcludedFromNonConcessionalManagedInvestmentTrustA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                statementOfDistributionToBeneficiary.TRT538 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion ExcludedFromNonConcessionalManagedInvestmentTrustA
                            } // End of if ShareOfIncomePrimaryProduction node exists

                            _currentXPath.Pop();
#endregion ShareOfIncomePrimaryProduction
#region ShareOfIncomeNonPrimaryProduction
                            _currentXPath.Push("/tns:ShareOfIncomeNonPrimaryProduction");
                            //3. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "ShareOfIncomeNonPrimaryProduction" && reader.Depth == _currentXPath.Count - 1)
                            {
                                statementOfDistributionToBeneficiary.RP_StatementOfDistributionToBeneficiary_Information_ShareOfIncomeNonPrimaryProductionCollectionExists = true;
                                statementOfDistributionToBeneficiary.RP_StatementOfDistributionToBeneficiary_Information_ShareOfIncomeNonPrimaryProductionCollectionCount += 1;
                                _found = true;
#region BeneficiaryShareA
                                _currentXPath.Push("/tns:BeneficiaryShareA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "BeneficiaryShareA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                statementOfDistributionToBeneficiary.TRT219 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion BeneficiaryShareA
#region TrustManagedInvestmentSchemeLessNetCapitalGainAndForeignIncomeNonPrimaryProductionShareA
                                _currentXPath.Push("/tns:TrustManagedInvestmentSchemeLessNetCapitalGainAndForeignIncomeNonPrimaryProductionShareA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "TrustManagedInvestmentSchemeLessNetCapitalGainAndForeignIncomeNonPrimaryProductionShareA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                statementOfDistributionToBeneficiary.TRT582 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion TrustManagedInvestmentSchemeLessNetCapitalGainAndForeignIncomeNonPrimaryProductionShareA
#region NonConcessionalManagedInvestmentTrustA
                                _currentXPath.Push("/tns:NonConcessionalManagedInvestmentTrustA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "NonConcessionalManagedInvestmentTrustA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                statementOfDistributionToBeneficiary.TRT539 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion NonConcessionalManagedInvestmentTrustA
#region ExcludedFromNonConcessionalManagedInvestmentTrustA
                                _currentXPath.Push("/tns:ExcludedFromNonConcessionalManagedInvestmentTrustA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "ExcludedFromNonConcessionalManagedInvestmentTrustA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                statementOfDistributionToBeneficiary.TRT540 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion ExcludedFromNonConcessionalManagedInvestmentTrustA
                            } // End of if ShareOfIncomeNonPrimaryProduction node exists

                            _currentXPath.Pop();
#endregion ShareOfIncomeNonPrimaryProduction
#region EarlyStageVentureTaxOffsetTaxOffsetClaimNonRefundableOtherA
                            _currentXPath.Push("/tns:EarlyStageVentureTaxOffsetTaxOffsetClaimNonRefundableOtherA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "EarlyStageVentureTaxOffsetTaxOffsetClaimNonRefundableOtherA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            statementOfDistributionToBeneficiary.TRT502 = currentDecimalValue;
                                        }

                                        statementOfDistributionToBeneficiary.RP_StatementOfDistributionToBeneficiary_Information_EarlyStageVentureTaxOffsetCollectionExists = true;
                                        statementOfDistributionToBeneficiary.RP_StatementOfDistributionToBeneficiary_Information_EarlyStageVentureTaxOffsetCollectionCount += 1;
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion EarlyStageVentureTaxOffsetTaxOffsetClaimNonRefundableOtherA
#region EarlyStageInvestorTaxOffsetTaxOffsetClaimNonRefundableOtherA
                            _currentXPath.Push("/tns:EarlyStageInvestorTaxOffsetTaxOffsetClaimNonRefundableOtherA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "EarlyStageInvestorTaxOffsetTaxOffsetClaimNonRefundableOtherA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            statementOfDistributionToBeneficiary.TRT503 = currentDecimalValue;
                                        }

                                        statementOfDistributionToBeneficiary.RP_StatementOfDistributionToBeneficiary_Information_EarlyStageInvestorTaxOffsetCollectionExists = true;
                                        statementOfDistributionToBeneficiary.RP_StatementOfDistributionToBeneficiary_Information_EarlyStageInvestorTaxOffsetCollectionCount += 1;
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion EarlyStageInvestorTaxOffsetTaxOffsetClaimNonRefundableOtherA
#region NonResidentAdditionalInformation
                            _currentXPath.Push("/tns:NonResidentAdditionalInformation");
                            //3. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "NonResidentAdditionalInformation" && reader.Depth == _currentXPath.Count - 1)
                            {
                                statementOfDistributionToBeneficiary.RP_StatementOfDistributionToBeneficiary_Information_NonResidentAdditionalInformationCollectionExists = true;
                                statementOfDistributionToBeneficiary.RP_StatementOfDistributionToBeneficiary_Information_NonResidentAdditionalInformationCollectionCount += 1;
                                _found = true;
#region InternationalDealingsAssessableIncomeNonResidentBeneficiaryNonTrusteeA
                                _currentXPath.Push("/tns:InternationalDealingsAssessableIncomeNonResidentBeneficiaryNonTrusteeA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsAssessableIncomeNonResidentBeneficiaryNonTrusteeA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                statementOfDistributionToBeneficiary.TRT228 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion InternationalDealingsAssessableIncomeNonResidentBeneficiaryNonTrusteeA
#region InternationalDealingsAssessableIncomeNonResidentBeneficiaryTrusteeA
                                _currentXPath.Push("/tns:InternationalDealingsAssessableIncomeNonResidentBeneficiaryTrusteeA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsAssessableIncomeNonResidentBeneficiaryTrusteeA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                statementOfDistributionToBeneficiary.TRT229 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion InternationalDealingsAssessableIncomeNonResidentBeneficiaryTrusteeA
                            } // End of if NonResidentAdditionalInformation node exists

                            _currentXPath.Pop();
#endregion NonResidentAdditionalInformation
#region TBStatementInformation
                            _currentXPath.Push("/tns:TBStatementInformation");
                            //3. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "TBStatementInformation" && reader.Depth == _currentXPath.Count - 1)
                            {
                                statementOfDistributionToBeneficiary.RP_StatementOfDistributionToBeneficiary_Information_TBStatementInformationCollectionExists = true;
                                statementOfDistributionToBeneficiary.RP_StatementOfDistributionToBeneficiary_Information_TBStatementInformationCollectionCount += 1;
                                _found = true;
#region IncomeTaxTrustBeneficiaryStatementI
                                _currentXPath.Push("/tns:IncomeTaxTrustBeneficiaryStatementI");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxTrustBeneficiaryStatementI" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            currentBooleanValue = ToBoolean(currentValue);
                                            if (currentBooleanValue != null)
                                            {
                                                statementOfDistributionToBeneficiary.TRT344 = currentBooleanValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion IncomeTaxTrustBeneficiaryStatementI
#region IncomeTaxSuperannuationTrusteeBeneficiaryTaxPreferredA
                                _currentXPath.Push("/tns:IncomeTaxSuperannuationTrusteeBeneficiaryTaxPreferredA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxSuperannuationTrusteeBeneficiaryTaxPreferredA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                statementOfDistributionToBeneficiary.TRT345 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion IncomeTaxSuperannuationTrusteeBeneficiaryTaxPreferredA
#region IncomeTaxSuperannuationTrusteeBeneficiaryUntaxedPartShareOfNetIncomeA
                                _currentXPath.Push("/tns:IncomeTaxSuperannuationTrusteeBeneficiaryUntaxedPartShareOfNetIncomeA");
                                //6. use case
                                if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxSuperannuationTrusteeBeneficiaryUntaxedPartShareOfNetIncomeA" && reader.Depth == _currentXPath.Count - 1)
                                {
                                    if (!IsEmptyOrNilElement(reader))
                                    {
                                        ReadNext(reader);
                                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                        {
                                            currentValue = reader.Value;
                                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                            if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                            {
                                                statementOfDistributionToBeneficiary.TRT346 = currentDecimalValue;
                                            }
                                        }
                                    }

                                    _found = true;
                                }

                                _currentXPath.Pop();
#endregion IncomeTaxSuperannuationTrusteeBeneficiaryUntaxedPartShareOfNetIncomeA
                            } // End of if TBStatementInformation node exists

                            _currentXPath.Pop();
#endregion TBStatementInformation
                        } // End of if Information node exists

                        _currentXPath.Pop();
#endregion Information
#region AnnualTrusteePaymentReportInformation
                        _currentXPath.Push("/tns:AnnualTrusteePaymentReportInformation");
                        //3. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "AnnualTrusteePaymentReportInformation" && reader.Depth == _currentXPath.Count - 1)
                        {
                            statementOfDistributionToBeneficiary.RP_StatementOfDistributionToBeneficiary_AnnualTrusteePaymentReportInformationCollectionExists = true;
                            statementOfDistributionToBeneficiary.RP_StatementOfDistributionToBeneficiary_AnnualTrusteePaymentReportInformationCollectionCount += 1;
                            _found = true;
#region ExpenseDistributionBeneficiaryCloselyHeldTrustA
                            _currentXPath.Push("/tns:ExpenseDistributionBeneficiaryCloselyHeldTrustA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "ExpenseDistributionBeneficiaryCloselyHeldTrustA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            statementOfDistributionToBeneficiary.TRT433 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion ExpenseDistributionBeneficiaryCloselyHeldTrustA
#region IncomeTaxPayAsYouGoWithholdingCreditForTaxWithheldFromCloselyHeldTrustA
                            _currentXPath.Push("/tns:IncomeTaxPayAsYouGoWithholdingCreditForTaxWithheldFromCloselyHeldTrustA");
                            //6. use case
                            if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxPayAsYouGoWithholdingCreditForTaxWithheldFromCloselyHeldTrustA" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                        {
                                            statementOfDistributionToBeneficiary.TRT434 = currentDecimalValue;
                                        }
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion IncomeTaxPayAsYouGoWithholdingCreditForTaxWithheldFromCloselyHeldTrustA
                        } // End of if AnnualTrusteePaymentReportInformation node exists

                        _currentXPath.Pop();
#endregion AnnualTrusteePaymentReportInformation
                        _currentXPath.Pop();
                    } // End of while loop for StatementOfDistributionToBeneficiary
                } // End of if StatementOfDistributionToBeneficiaryCollection node exists

                _currentXPath.Pop();
#endregion StatementOfDistributionToBeneficiary
#region IncomeNoBeneficiaryPresentlyEntitled
                _currentXPath.Push("/tns:IncomeNoBeneficiaryPresentlyEntitled");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeNoBeneficiaryPresentlyEntitled" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_IncomeNoBeneficiaryPresentlyEntitledCollectionExists = true;
                    report.RP_IncomeNoBeneficiaryPresentlyEntitledCollectionCount += 1;
                    _found = true;
#region IncomeTaxAssessmentCalculationC
                    _currentXPath.Push("/tns:IncomeTaxAssessmentCalculationC");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxAssessmentCalculationC" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRT230 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion IncomeTaxAssessmentCalculationC
#region TrustEstateIncomeShareA
                    _currentXPath.Push("/tns:TrustEstateIncomeShareA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "TrustEstateIncomeShareA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRT443 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion TrustEstateIncomeShareA
#region IncomeTaxPayAsYouGoWithholdingCreditForAmountsWithheldFromForeignResidentsA
                    _currentXPath.Push("/tns:IncomeTaxPayAsYouGoWithholdingCreditForAmountsWithheldFromForeignResidentsA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxPayAsYouGoWithholdingCreditForAmountsWithheldFromForeignResidentsA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRT477 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion IncomeTaxPayAsYouGoWithholdingCreditForAmountsWithheldFromForeignResidentsA
#region IncomeTaxFrankingCreditsReceivedFromNewZealandCompaniesA
                    _currentXPath.Push("/tns:IncomeTaxFrankingCreditsReceivedFromNewZealandCompaniesA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxFrankingCreditsReceivedFromNewZealandCompaniesA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRT233 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion IncomeTaxFrankingCreditsReceivedFromNewZealandCompaniesA
#region IncomeTaxPayAsYouGoWithholdingCreditForTaxWithheldWhereABNNotQuotedA
                    _currentXPath.Push("/tns:IncomeTaxPayAsYouGoWithholdingCreditForTaxWithheldWhereABNNotQuotedA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxPayAsYouGoWithholdingCreditForTaxWithheldWhereABNNotQuotedA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRT236 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion IncomeTaxPayAsYouGoWithholdingCreditForTaxWithheldWhereABNNotQuotedA
#region DistributionFrankedA
                    _currentXPath.Push("/tns:DistributionFrankedA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "DistributionFrankedA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRT444 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion DistributionFrankedA
#region IncomeTaxFrankingCreditsReceivedFromAustralianCompaniesA
                    _currentXPath.Push("/tns:IncomeTaxFrankingCreditsReceivedFromAustralianCompaniesA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxFrankingCreditsReceivedFromAustralianCompaniesA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRT237 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion IncomeTaxFrankingCreditsReceivedFromAustralianCompaniesA
#region IncomeTaxPayAsYouGoWithholdingCreditForAmountsWithheldWhereTFNNotQuotedA
                    _currentXPath.Push("/tns:IncomeTaxPayAsYouGoWithholdingCreditForAmountsWithheldWhereTFNNotQuotedA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxPayAsYouGoWithholdingCreditForAmountsWithheldWhereTFNNotQuotedA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRT238 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion IncomeTaxPayAsYouGoWithholdingCreditForAmountsWithheldWhereTFNNotQuotedA
#region IncomeTaxPayAsYouGoWithholdingCreditForTaxWithheldFromCloselyHeldTrustA
                    _currentXPath.Push("/tns:IncomeTaxPayAsYouGoWithholdingCreditForTaxWithheldFromCloselyHeldTrustA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxPayAsYouGoWithholdingCreditForTaxWithheldFromCloselyHeldTrustA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRT435 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion IncomeTaxPayAsYouGoWithholdingCreditForTaxWithheldFromCloselyHeldTrustA
#region CapitalGainsNetA
                    _currentXPath.Push("/tns:CapitalGainsNetA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "CapitalGainsNetA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRT239 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion CapitalGainsNetA
#region CapitalGainsNonConcessionalManagedInvestmentTrustIncomeA
                    _currentXPath.Push("/tns:CapitalGainsNonConcessionalManagedInvestmentTrustIncomeA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "CapitalGainsNonConcessionalManagedInvestmentTrustIncomeA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRT547 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion CapitalGainsNonConcessionalManagedInvestmentTrustIncomeA
#region CapitalGainsExcludedFromNonConcessionalManagedInvestmentTrustIncomeA
                    _currentXPath.Push("/tns:CapitalGainsExcludedFromNonConcessionalManagedInvestmentTrustIncomeA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "CapitalGainsExcludedFromNonConcessionalManagedInvestmentTrustIncomeA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRT548 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion CapitalGainsExcludedFromNonConcessionalManagedInvestmentTrustIncomeA
#region IncomeTaxPayAsYouGoWithholdingCreditForCapitalGainsWithheldFromForeignResidentsA
                    _currentXPath.Push("/tns:IncomeTaxPayAsYouGoWithholdingCreditForCapitalGainsWithheldFromForeignResidentsA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxPayAsYouGoWithholdingCreditForCapitalGainsWithheldFromForeignResidentsA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRT482 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion IncomeTaxPayAsYouGoWithholdingCreditForCapitalGainsWithheldFromForeignResidentsA
#region IncomeTaxAttributedForeignIncomeA
                    _currentXPath.Push("/tns:IncomeTaxAttributedForeignIncomeA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxAttributedForeignIncomeA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRT240 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion IncomeTaxAttributedForeignIncomeA
#region InternationalDealingsNetA
                    _currentXPath.Push("/tns:InternationalDealingsNetA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsNetA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRT241 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion InternationalDealingsNetA
#region InternationalDealingsTaxOffsetA
                    _currentXPath.Push("/tns:InternationalDealingsTaxOffsetA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsTaxOffsetA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRT242 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion InternationalDealingsTaxOffsetA
#region TaxOffsetClaimNationalRentalAffordabilitySchemeEntitlementA
                    _currentXPath.Push("/tns:TaxOffsetClaimNationalRentalAffordabilitySchemeEntitlementA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "TaxOffsetClaimNationalRentalAffordabilitySchemeEntitlementA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRT343 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion TaxOffsetClaimNationalRentalAffordabilitySchemeEntitlementA
#region IncomeTaxTaxOffsetRefundableOtherA
                    _currentXPath.Push("/tns:IncomeTaxTaxOffsetRefundableOtherA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxTaxOffsetRefundableOtherA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRT466 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion IncomeTaxTaxOffsetRefundableOtherA
#region PrimaryProductionNoBeneficiaryEntitled
                    _currentXPath.Push("/tns:PrimaryProductionNoBeneficiaryEntitled");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "PrimaryProductionNoBeneficiaryEntitled" && reader.Depth == _currentXPath.Count - 1)
                    {
                        report.RP_IncomeNoBeneficiaryPresentlyEntitled_PrimaryProductionNoBeneficiaryEntitledCollectionExists = true;
                        report.RP_IncomeNoBeneficiaryPresentlyEntitled_PrimaryProductionNoBeneficiaryEntitledCollectionCount += 1;
                        _found = true;
#region IncomeTaxIncomeOrLossTaxableA
                        _currentXPath.Push("/tns:IncomeTaxIncomeOrLossTaxableA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxIncomeOrLossTaxableA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.TRT234 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion IncomeTaxIncomeOrLossTaxableA
#region NonConcessionalManagedInvestmentTrustA
                        _currentXPath.Push("/tns:NonConcessionalManagedInvestmentTrustA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "NonConcessionalManagedInvestmentTrustA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.TRT543 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion NonConcessionalManagedInvestmentTrustA
#region ExcludedFromNonConcessionalManagedInvestmentTrustA
                        _currentXPath.Push("/tns:ExcludedFromNonConcessionalManagedInvestmentTrustA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ExcludedFromNonConcessionalManagedInvestmentTrustA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.TRT544 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion ExcludedFromNonConcessionalManagedInvestmentTrustA
                    } // End of if PrimaryProductionNoBeneficiaryEntitled node exists

                    _currentXPath.Pop();
#endregion PrimaryProductionNoBeneficiaryEntitled
#region NonPrimaryProductionNoBeneficiaryEntitled
                    _currentXPath.Push("/tns:NonPrimaryProductionNoBeneficiaryEntitled");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "NonPrimaryProductionNoBeneficiaryEntitled" && reader.Depth == _currentXPath.Count - 1)
                    {
                        report.RP_IncomeNoBeneficiaryPresentlyEntitled_NonPrimaryProductionNoBeneficiaryEntitledCollectionExists = true;
                        report.RP_IncomeNoBeneficiaryPresentlyEntitled_NonPrimaryProductionNoBeneficiaryEntitledCollectionCount += 1;
                        _found = true;
#region IncomeTaxIncomeOrLossTaxableA
                        _currentXPath.Push("/tns:IncomeTaxIncomeOrLossTaxableA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "IncomeTaxIncomeOrLossTaxableA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.TRT235 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion IncomeTaxIncomeOrLossTaxableA
#region NonConcessionalManagedInvestmentTrustA
                        _currentXPath.Push("/tns:NonConcessionalManagedInvestmentTrustA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "NonConcessionalManagedInvestmentTrustA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.TRT545 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion NonConcessionalManagedInvestmentTrustA
#region ExcludedFromNonConcessionalManagedInvestmentTrustA
                        _currentXPath.Push("/tns:ExcludedFromNonConcessionalManagedInvestmentTrustA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "ExcludedFromNonConcessionalManagedInvestmentTrustA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.TRT546 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion ExcludedFromNonConcessionalManagedInvestmentTrustA
                    } // End of if NonPrimaryProductionNoBeneficiaryEntitled node exists

                    _currentXPath.Pop();
#endregion NonPrimaryProductionNoBeneficiaryEntitled
#region EarlyStageVentureNoBeneficiary
                    _currentXPath.Push("/tns:EarlyStageVentureNoBeneficiary");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "EarlyStageVentureNoBeneficiary" && reader.Depth == _currentXPath.Count - 1)
                    {
                        report.RP_IncomeNoBeneficiaryPresentlyEntitled_EarlyStageVentureNoBeneficiaryCollectionExists = true;
                        report.RP_IncomeNoBeneficiaryPresentlyEntitled_EarlyStageVentureNoBeneficiaryCollectionCount += 1;
                        _found = true;
#region TaxOffsetClaimNonRefundableOtherA
                        _currentXPath.Push("/tns:TaxOffsetClaimNonRefundableOtherA");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "TaxOffsetClaimNonRefundableOtherA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.TRT504 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion TaxOffsetClaimNonRefundableOtherA
#region EarlyStageVentureNoBeneficiaryCarriedForwardTaxOffsetClaimNonRefundableOtherA
                        _currentXPath.Push("/tns:EarlyStageVentureNoBeneficiaryCarriedForwardTaxOffsetClaimNonRefundableOtherA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "EarlyStageVentureNoBeneficiaryCarriedForwardTaxOffsetClaimNonRefundableOtherA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.TRT505 = currentDecimalValue;
                                    }

                                    report.RP_IncomeNoBeneficiaryPresentlyEntitled_EarlyStageVentureNoBeneficiary_EarlyStageVentureNoBeneficiaryCarriedForwardCollectionExists = true;
                                    report.RP_IncomeNoBeneficiaryPresentlyEntitled_EarlyStageVentureNoBeneficiary_EarlyStageVentureNoBeneficiaryCarriedForwardCollectionCount += 1;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion EarlyStageVentureNoBeneficiaryCarriedForwardTaxOffsetClaimNonRefundableOtherA
                    } // End of if EarlyStageVentureNoBeneficiary node exists

                    _currentXPath.Pop();
#endregion EarlyStageVentureNoBeneficiary
#region InvestorNoBeneficiary
                    _currentXPath.Push("/tns:InvestorNoBeneficiary");
                    //3. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "InvestorNoBeneficiary" && reader.Depth == _currentXPath.Count - 1)
                    {
                        report.RP_IncomeNoBeneficiaryPresentlyEntitled_InvestorNoBeneficiaryCollectionExists = true;
                        report.RP_IncomeNoBeneficiaryPresentlyEntitled_InvestorNoBeneficiaryCollectionCount += 1;
                        _found = true;
#region TaxOffsetClaimNonRefundableOtherA
                        _currentXPath.Push("/tns:TaxOffsetClaimNonRefundableOtherA");
                        //6. use case
                        if (ReadToNextElement(reader, true) && reader.LocalName == "TaxOffsetClaimNonRefundableOtherA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.TRT506 = currentDecimalValue;
                                    }
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion TaxOffsetClaimNonRefundableOtherA
#region InvestorNoBeneficiaryCarriedForwardTaxOffsetClaimNonRefundableOtherA
                        _currentXPath.Push("/tns:InvestorNoBeneficiaryCarriedForwardTaxOffsetClaimNonRefundableOtherA");
                        //6. use case
                        if (ReadToNextElement(reader, false) && reader.LocalName == "InvestorNoBeneficiaryCarriedForwardTaxOffsetClaimNonRefundableOtherA" && reader.Depth == _currentXPath.Count - 1)
                        {
                            if (!IsEmptyOrNilElement(reader))
                            {
                                ReadNext(reader);
                                if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                {
                                    currentValue = reader.Value;
                                    ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                    if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                    {
                                        report.TRT507 = currentDecimalValue;
                                    }

                                    report.RP_IncomeNoBeneficiaryPresentlyEntitled_InvestorNoBeneficiary_InvestorNoBeneficiaryCarriedForwardCollectionExists = true;
                                    report.RP_IncomeNoBeneficiaryPresentlyEntitled_InvestorNoBeneficiary_InvestorNoBeneficiaryCarriedForwardCollectionCount += 1;
                                }
                            }

                            _found = true;
                        }

                        _currentXPath.Pop();
#endregion InvestorNoBeneficiaryCarriedForwardTaxOffsetClaimNonRefundableOtherA
                    } // End of if InvestorNoBeneficiary node exists

                    _currentXPath.Pop();
#endregion InvestorNoBeneficiary
                } // End of if IncomeNoBeneficiaryPresentlyEntitled node exists

                _currentXPath.Pop();
#endregion IncomeNoBeneficiaryPresentlyEntitled
#region CapitalGainsAssessed
                _currentXPath.Push("/tns:CapitalGainsAssessed");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "CapitalGainsAssessed" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_CapitalGainsAssessedCollectionExists = true;
                    report.RP_CapitalGainsAssessedCollectionCount += 1;
                    _found = true;
#region CapitalGainsTaxAssessmentCalculationC
                    _currentXPath.Push("/tns:CapitalGainsTaxAssessmentCalculationC");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "CapitalGainsTaxAssessmentCalculationC" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRT445 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion CapitalGainsTaxAssessmentCalculationC
#region CapitalGainsTaxTrusteeAssessedCapitalGainA
                    _currentXPath.Push("/tns:CapitalGainsTaxTrusteeAssessedCapitalGainA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "CapitalGainsTaxTrusteeAssessedCapitalGainA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRT446 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion CapitalGainsTaxTrusteeAssessedCapitalGainA
                } // End of if CapitalGainsAssessed node exists

                _currentXPath.Pop();
#endregion CapitalGainsAssessed
#region IncomeTaxBeneficiaryUnderLegalDisabilityAlsoPresentlyEntitledToIncomeFromAnotherTrustI
                _currentXPath.Push("/tns:IncomeTaxBeneficiaryUnderLegalDisabilityAlsoPresentlyEntitledToIncomeFromAnotherTrustI");
                //6. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "IncomeTaxBeneficiaryUnderLegalDisabilityAlsoPresentlyEntitledToIncomeFromAnotherTrustI" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            currentBooleanValue = ToBoolean(currentValue);
                            if (currentBooleanValue != null)
                            {
                                report.TRT244 = currentBooleanValue;
                            }
                        }
                    }

                    _found = true;
                }

                _currentXPath.Pop();
#endregion IncomeTaxBeneficiaryUnderLegalDisabilityAlsoPresentlyEntitledToIncomeFromAnotherTrustI
#region NonResidentTrust
                _currentXPath.Push("/tns:NonResidentTrust");
                //3. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "NonResidentTrust" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_NonResidentTrustCollectionExists = true;
                    report.RP_NonResidentTrustCollectionCount += 1;
                    _found = true;
#region ResidencyTaxPurposesOrganisationStatusC
                    _currentXPath.Push("/tns:ResidencyTaxPurposesOrganisationStatusC");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "ResidencyTaxPurposesOrganisationStatusC" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRT245 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion ResidencyTaxPurposesOrganisationStatusC
#region InternationalDealingsAustralianSourcedIncomeNoPresentEntitlementA
                    _currentXPath.Push("/tns:InternationalDealingsAustralianSourcedIncomeNoPresentEntitlementA");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "InternationalDealingsAustralianSourcedIncomeNoPresentEntitlementA" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRT246 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion InternationalDealingsAustralianSourcedIncomeNoPresentEntitlementA
                } // End of if NonResidentTrust node exists

                _currentXPath.Pop();
#endregion NonResidentTrust
#region OtherReportingInformation
                _currentXPath.Push("/tns:OtherReportingInformation");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "OtherReportingInformation" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_OtherReportingInformationCollectionExists = true;
                    report.RP_OtherReportingInformationCollectionCount += 1;
                    _found = true;
#region RegulatoryDisclosuresGeneralInformationAboutFinancialStatementsT
                    _currentXPath.Push("/tns:RegulatoryDisclosuresGeneralInformationAboutFinancialStatementsT");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "RegulatoryDisclosuresGeneralInformationAboutFinancialStatementsT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRT405 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion RegulatoryDisclosuresGeneralInformationAboutFinancialStatementsT
#region CompletionHoursN
                    _currentXPath.Push("/tns:CompletionHoursN");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "CompletionHoursN" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (Decimal.TryParse(currentValue, out currentDecimalValue))
                                {
                                    report.TRT247 = currentDecimalValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion CompletionHoursN
                } // End of if OtherReportingInformation node exists

                _currentXPath.Pop();
#endregion OtherReportingInformation
#region ReportingPartyDeclaration
                _currentXPath.Push("/tns:ReportingPartyDeclaration");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "ReportingPartyDeclaration" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.RP_ReportingPartyDeclarationCollectionExists = true;
                    report.RP_ReportingPartyDeclarationCollectionCount += 1;
                    _found = true;
#region StatementTypeC
                    _currentXPath.Push("/tns:StatementTypeC");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "StatementTypeC" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRT383 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion StatementTypeC
#region StatementAcceptedI
                    _currentXPath.Push("/tns:StatementAcceptedI");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "StatementAcceptedI" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.TRT384 = currentBooleanValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion StatementAcceptedI
#region StatementT
                    _currentXPath.Push("/tns:StatementT");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "StatementT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRT385 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion StatementT
#region SignatureD
                    _currentXPath.Push("/tns:SignatureD");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "SignatureD" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                                {
                                    report.TRT386 = currentDateTimeValue.Date;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion SignatureD
#region SignatoryIdentifierT
                    _currentXPath.Push("/tns:SignatoryIdentifierT");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "SignatoryIdentifierT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRT387 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion SignatoryIdentifierT
#region PersonUnstructuredNameFullNameT
                    _currentXPath.Push("/tns:PersonUnstructuredNameFullNameT");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "PersonUnstructuredNameFullNameT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRT389 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion PersonUnstructuredNameFullNameT
                } // End of if ReportingPartyDeclaration node exists

                _currentXPath.Pop();
#endregion ReportingPartyDeclaration
            } // End of if RP node exists

            _currentXPath.Pop();
#endregion RP
#region INT
            _currentXPath.Push("/tns:INT");
            //3. use case
            if (ReadToNextElement(reader, false) && reader.LocalName == "INT" && reader.Depth == _currentXPath.Count - 1)
            {
                report.IntCollectionExists = true;
                report.IntCollectionCount += 1;
                _found = true;
#region TaxAgentClientReferenceT
                _currentXPath.Push("/tns:TaxAgentClientReferenceT");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "TaxAgentClientReferenceT" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.TRT251 = currentValue;
                        }
                    }

                    _found = true;
                }

                _currentXPath.Pop();
#endregion TaxAgentClientReferenceT
#region TaxAgentNumberId
                _currentXPath.Push("/tns:TaxAgentNumberId");
                //6. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "TaxAgentNumberId" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.TRT2 = currentValue;
                        }
                    }

                    _found = true;
                }

                _currentXPath.Pop();
#endregion TaxAgentNumberId
#region AustralianBusinessNumberId
                _currentXPath.Push("/tns:AustralianBusinessNumberId");
                //6. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "AustralianBusinessNumberId" && reader.Depth == _currentXPath.Count - 1)
                {
                    if (!IsEmptyOrNilElement(reader))
                    {
                        ReadNext(reader);
                        if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                        {
                            currentValue = reader.Value;
                            ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                            report.TRT575 = currentValue;
                        }
                    }

                    _found = true;
                }

                _currentXPath.Pop();
#endregion AustralianBusinessNumberId
#region Declaration
                _currentXPath.Push("/tns:Declaration");
                //3. use case
                if (ReadToNextElement(reader, true) && reader.LocalName == "Declaration" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.InT_DeclarationCollectionExists = true;
                    report.InT_DeclarationCollectionCount += 1;
                    _found = true;
#region StatementTypeC
                    _currentXPath.Push("/tns:StatementTypeC");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "StatementTypeC" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRT390 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion StatementTypeC
#region StatementAcceptedI
                    _currentXPath.Push("/tns:StatementAcceptedI");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "StatementAcceptedI" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                currentBooleanValue = ToBoolean(currentValue);
                                if (currentBooleanValue != null)
                                {
                                    report.TRT391 = currentBooleanValue;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion StatementAcceptedI
#region StatementT
                    _currentXPath.Push("/tns:StatementT");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "StatementT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRT392 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion StatementT
#region SignatureD
                    _currentXPath.Push("/tns:SignatureD");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "SignatureD" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                if (DateTime.TryParse(currentValue, out currentDateTimeValue))
                                {
                                    report.TRT393 = currentDateTimeValue.Date;
                                }
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion SignatureD
#region SignatoryIdentifierT
                    _currentXPath.Push("/tns:SignatoryIdentifierT");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "SignatoryIdentifierT" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRT394 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion SignatoryIdentifierT
#region IntermediaryContact
                    _currentXPath.Push("/tns:IntermediaryContactCollection");
                    // 4. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "IntermediaryContactCollection" && reader.Depth == _currentXPath.Count - 1)
                    {
                        report.InT_Declaration_IntermediaryContactCollection = new List<TRT2026.InT_Declaration_IntermediaryContact>();
                        report.InT_Declaration_IntermediaryContactCollectionExists = true;
                        _found = true;
                        while (ReadToNextElement(reader, false) && reader.LocalName == "IntermediaryContact")
                        {
                            _found = true;
                            TRT2026.InT_Declaration_IntermediaryContact intermediaryContact = new TRT2026.InT_Declaration_IntermediaryContact();
                            report.InT_Declaration_IntermediaryContactCollection.Add(intermediaryContact);
                            report.InT_Declaration_IntermediaryContactCollectionCount += 1;
                            intermediaryContact.OccurrenceIndex = report.InT_Declaration_IntermediaryContactCollectionCount;
                            _currentXPath.Push("/tns:IntermediaryContact[" + report.InT_Declaration_IntermediaryContactCollectionCount + "]");
#region PersonUnstructuredNameUsageC
                            _currentXPath.Push("/tns:PersonUnstructuredNameUsageC");
                            //6. use case
                            if (ReadToNextElement(reader, true) && reader.LocalName == "PersonUnstructuredNameUsageC" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        intermediaryContact.TRT395 = currentValue;
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion PersonUnstructuredNameUsageC
#region PersonUnstructuredNameFullNameT
                            _currentXPath.Push("/tns:PersonUnstructuredNameFullNameT");
                            //6. use case
                            if (ReadToNextElement(reader, true) && reader.LocalName == "PersonUnstructuredNameFullNameT" && reader.Depth == _currentXPath.Count - 1)
                            {
                                if (!IsEmptyOrNilElement(reader))
                                {
                                    ReadNext(reader);
                                    if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                                    {
                                        currentValue = reader.Value;
                                        ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                        intermediaryContact.TRT396 = currentValue;
                                    }
                                }

                                _found = true;
                            }

                            _currentXPath.Pop();
#endregion PersonUnstructuredNameFullNameT
                            _currentXPath.Pop();
                        } // End of while loop for IntermediaryContact
                    } // End of if IntermediaryContactCollection node exists

                    _currentXPath.Pop();
#endregion IntermediaryContact
                } // End of if Declaration node exists

                _currentXPath.Pop();
#endregion Declaration
#region IntermediaryContactTelephone
                _currentXPath.Push("/tns:IntermediaryContactTelephone");
                //3. use case
                if (ReadToNextElement(reader, false) && reader.LocalName == "IntermediaryContactTelephone" && reader.Depth == _currentXPath.Count - 1)
                {
                    report.InT_IntermediaryContactTelephoneCollectionExists = true;
                    report.InT_IntermediaryContactTelephoneCollectionCount += 1;
                    _found = true;
#region ElectronicContactUsageC
                    _currentXPath.Push("/tns:ElectronicContactUsageC");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "ElectronicContactUsageC" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRT335 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion ElectronicContactUsageC
#region ElectronicContactServiceLineC
                    _currentXPath.Push("/tns:ElectronicContactServiceLineC");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "ElectronicContactServiceLineC" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRT334 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion ElectronicContactServiceLineC
#region ElectronicContactAreaC
                    _currentXPath.Push("/tns:ElectronicContactAreaC");
                    //6. use case
                    if (ReadToNextElement(reader, false) && reader.LocalName == "ElectronicContactAreaC" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRT248 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion ElectronicContactAreaC
#region ElectronicContactMinimalN
                    _currentXPath.Push("/tns:ElectronicContactMinimalN");
                    //6. use case
                    if (ReadToNextElement(reader, true) && reader.LocalName == "ElectronicContactMinimalN" && reader.Depth == _currentXPath.Count - 1)
                    {
                        if (!IsEmptyOrNilElement(reader))
                        {
                            ReadNext(reader);
                            if (reader.NodeType == XmlNodeType.Text || reader.NodeType == XmlNodeType.CDATA)
                            {
                                currentValue = reader.Value;
                                ReadNext(reader); // consume the end element so we detect any validation errors before _currentXPath is updated.
                                report.TRT249 = currentValue;
                            }
                        }

                        _found = true;
                    }

                    _currentXPath.Pop();
#endregion ElectronicContactMinimalN
                } // End of if IntermediaryContactTelephone node exists

                _currentXPath.Pop();
#endregion IntermediaryContactTelephone
            } // End of if INT node exists

            _currentXPath.Pop();
#endregion INT
            while (reader.EOF != true)
                reader.Read();
            return report;
        }

        private string _elementCsvList = "RP#TargetFinancialY,RP#PeriodStartD,RP#PeriodEndD,RP#TaxFileNumberId,RP#AustralianBusinessNumberId,ReportAmendment#I,ReportAmendment#AmendmentTypeC,ReportAmendment#AmendmentReasonT,ReportAmendment#AmendmentSequenceN,ReportAmendment,CurrentTrust#OrganisationNameDetailsOrganisationalNameT,CurrentPostalAddressDetails#OverseasAddressI,CurrentPostalAddressDetails#Line1T,CurrentPostalAddressDetails#Line2T,CurrentPostalAddressDetails#LocalityNameT,CurrentPostalAddressDetails#PostcodeT,CurrentPostalAddressDetails#StateOrTerritoryC,CurrentPostalAddressDetails#CountryC,CurrentPostalAddressDetails,CurrentTrust,PreviousTrust#OrganisationNameDetailsOrganisationalNameT,PreviousPostalAddressDetails#OverseasAddressI,PreviousPostalAddressDetails#Line1T,PreviousPostalAddressDetails#Line2T,PreviousPostalAddressDetails#LocalityNameT,PreviousPostalAddressDetails#PostcodeT,PreviousPostalAddressDetails#StateOrTerritoryC,PreviousPostalAddressDetails#CountryC,PreviousPostalAddressDetails,PreviousTrust,PersonNameDetails#TitleT,PersonNameDetails#NameSuffixT,PersonNameDetails#FamilyNameT,PersonNameDetails#GivenNameT,PersonNameDetails#OtherGivenNameT,PersonNameDetails,OrganisationNameDetails#OrganisationalNameT,OrganisationNameDetails#AustralianBusinessNumberId,OrganisationNameDetails,ElectronicContactTelephone#ServiceLineC,ElectronicContactTelephone#AreaC,ElectronicContactTelephone#MinimalN,ElectronicContactTelephone,MailRecipient,ElectionStatus#ElectionsFamilyTrustElectionStatusY,ElectionStatus#ElectionsFamilyTrustElectionRevocationC,ElectionStatus,InterposedEntityElectionStatus#ElectionsY,InterposedEntityElectionStatus#ElectionsInterposedEntityElectionRevocationC,InterposedEntityElectionStatus,TypeOfTrust#OrganisationDetailsOrganisationTypeC,TypeOfTrust#OrganisationDetailsCharitableOrganisationI,TypeOfTrust#PersonDemographicDetailsDeathD,TypeOfTrust#AssetsInvestmentCrossOtherCCIVSubfundI,TypeOfTrust#OrganisationDetailsManagedInvestmentTrustI,TypeOfTrust#OrganisationDetailsWithholdingManagedInvestmentTrustI,TypeOfTrust#OrganisationDetailsStapledManagedInvestmentTrustI,TypeOfTrust#OrganisationDetailsManagedInvestmentTrustTypeC,TypeOfTrust#ElectionsManagedInvestmentTrustsI,TypeOfTrust#StatementSummaryPaymentRefundOrNilC,TypeOfTrust#LodgmentFinalReturnI,TypeOfTrust,FinancialInstitutionAccount#BankStateBranchN,FinancialInstitutionAccount#N,FinancialInstitutionAccount#NameT,FinancialInstitutionAccount,BusinessDetails#OrganisationDetailsMainIncomeActivityDe,BusinessDetails#OrganisationDetailsOrganisationIndustry2006ExtendedC,BusinessDetails#OrganisationDetailsActivityEventC,BusinessDetails#OrganisationDetailsTaxConsolidationStatusI,BusinessDetails#OrganisationDetailsSignificantGlobalEntityStatusI,BusinessDetails#InternationalDealingsCountryByCountryReportingEntityStatusI,BusinessIncomeAndExpenses#SmallBusinessEntityNetA,BusinessIncomeAndExpenses#ReconciliationAdjustmentTotalA,IncomePrimaryProduction#RemunerationABNNotQuotedPaymentGrossA,IncomePrimaryProduction#GovernmentFundingGovernmentIndustryPaymentsAssessableA,IncomePrimaryProduction#GovernmentFundingGovernmentIndustryPaymentsAssessableI,IncomePrimaryProduction#OtherA,IncomePrimaryProduction#NetA,IncomePrimaryProduction,IncomeNonPrimaryProduction#RemunerationABNNotQuotedPaymentGrossA,IncomeNonPrimaryProduction#RemunerationPaymentToForeignResidentGrossA,IncomeNonPrimaryProduction#GovernmentFundingGovernmentIndustryPaymentsAssessableA,IncomeNonPrimaryProduction#GovernmentFundingGovernmentIndustryPaymentsAssessableI,IncomeNonPrimaryProduction#OtherA,IncomeNonPrimaryProduction#NetA,IncomeNonPrimaryProduction,BusinessExpenses#ForeignResidentWithholdingA,BusinessExpenses#RemunerationWagesAndSalariesPaymentsContractorAndConsultantA,BusinessExpenses#SuperannuationContributionEmployerContributionsA,BusinessExpenses#OperatingCostOfSalesA,BusinessExpenses#OperatingBadDebtsA,BusinessExpenses#OperatingLeaseTotalA,BusinessExpenses#OperatingRentA,BusinessExpenses#OperatingInterestA,BusinessExpenses#RoyaltiesA,BusinessExpenses#DepreciationAndAmortisationA,BusinessExpenses#OperatingMotorVehicleA,BusinessExpenses#OperatingRepairsAndMaintenanceA,BusinessExpenses#DeductibleOtherTotalA,BusinessExpenses#ReconciliationAdjustmentTotalA,BusinessExpenses,BusinessIncomeAndExpenses,PayAsYouGoWithholding#TaxCreditForTaxWithheldWhereABNNotQuotedA,PayAsYouGoWithholding#TaxCreditForAmountsWithheldFromForeignResidentsA,PayAsYouGoWithholding,PartnershipsTrustsPrimaryProduction#PartnershipDistributionGrossA,PartnershipsTrustsPrimaryProduction#TrustShareNetA,PartnershipsTrustsPrimaryProduction#PartyTrustTypeC,PartnershipsTrustsPrimaryProduction#ExpenseDistributionDeductionsOtherA,PartnershipsTrustsPrimaryProduction,PartnershipTrustNonPrimaryProduction#PartnershipDistributionLessForeignIncomeA,PartnershipTrustNonPrimaryProduction#TrustShareNetExcludeNetCapitalGainsAndForeignIncomeAndDistributionFrankedA,PartnershipTrustNonPrimaryProduction#PartyTrustTypeC,PartnershipTrustNonPrimaryProduction#ExpenseDistributionDeductionsOtherA,PartnershipTrustNonPrimaryProduction#TrustDistributionFrankedA,PartnershipTrustNonPrimaryProduction#TaxDeductionTrustDistributionFrankedA,PartnershipTrustNonPrimaryProduction,ShareOfCredits#TaxPayAsYouGoWithholdingCreditForTaxWithheldWhereABNNotQuotedShareReceivedIndirectlyA,ShareOfCredits#TaxFrankingCreditsFrankingCreditShareReceivedIndirectlyA,ShareOfCredits#TaxPayAsYouGoWithholdingCreditForAmountsWithheldFromInterestAndDividendsWhereTFNNotQuotedShareReceivedIndirectlyA,ShareOfCredits#TaxPayAsYouGoWithholdingCreditForTaxWithheldFromCloselyHeldTrustShareReceivedIndirectlyA,ShareOfCredits#InternationalDealingsCreditForTaxWithheldFromForeignResidentWithholdingShareReceivedIndirectlyA,ShareOfCredits,Rent#OperatingIncomeGrossA,Rent#ExpenseInterestA,Rent#ExpenseCapitalWorksDeductionA,Rent#TaxDeductionIncomeDeductionsOtherThanInterestAndCapitalWorksA,Rent#ExpenseBuildToRentCapitalWorksDeductionA,Rent,Income#ForestryManagedInvestmentSchemeA,GrossInterest#A,GrossInterest#TaxTFNAmountsWithheldFromGrossInterestA,GrossInterest,Dividends#UnfrankedA,Dividends#FrankedA,Dividends#TaxFrankingCreditsA,Dividends#TaxTFNAmountsWithheldA,Dividends,SuperannuationLumpSumWithholdingPaymentDetails#SuperannuationBenefitDeathBenefitLumpSumPaymentTaxedElementA,SuperannuationLumpSumWithholdingPaymentDetails#SuperannuationBenefitDeathBenefitLumpSumPaymentUntaxedElementA,SuperannuationLumpSumWithholdingPaymentDetails#InsuranceBenefitsDeathBenefitI,SuperannuationLumpSumWithholdingPaymentDetails,EmploymentTerminationPaymentDetails#SuperannuationBenefitDeathBenefitEmploymentTerminationPaymentTaxableComponentA,EmploymentTerminationPaymentDetails#InsuranceBenefitsDeathBenefitI,EmploymentTerminationPaymentDetails#InsuranceBenefitsBeneficiaryTypeC,EmploymentTerminationPaymentDetails,EmploymentTerminationPaymentDetailsCollection,OtherAustralianIncome#De,OtherAustralianIncome#A,OtherAustralianIncome,OtherAustralianIncomeCollection,Income#ExceptedTrustNetA,Income#TaxAustralianGovernmentPensionsAndAllowancesTaxWithheldA,Income#PensionTotalA,IndividualNonBusinessWithholdingPaymentDetails#AustralianBusinessNumberId,IndividualNonBusinessWithholdingPaymentDetails#TaxPayAsYouGoWithholdingTaxWithheldA,IndividualNonBusinessWithholdingPaymentDetails#RemunerationIndividualNonBusinessGrossA,IndividualNonBusinessWithholdingPaymentDetails#TaxPayAsYouGoWithholdingCreditTaxWithheldUnusedAnnualOrLongServiceLeavePaymentLumpSumAA,IndividualNonBusinessWithholdingPaymentDetails#RemunerationUnusedAnnualLongServiceLeavePaymentLumpSumAA,IndividualNonBusinessWithholdingPaymentDetails#TaxPayAsYouGoWithholdingCreditTaxWithheldUnusedAnnualOrLongServiceLeavePaymentLumpSumBA,IndividualNonBusinessWithholdingPaymentDetails#RemunerationUnusedAnnualLongServiceLeavePaymentLumpSumBA,IndividualNonBusinessWithholdingPaymentDetails,IndividualNonBusinessWithholdingPaymentDetailsCollection,SuperannuationIncomeStreamWithholdingPaymentDetails#TaxPayAsYouGoWithholdingTaxWithheldA,SuperannuationIncomeStreamWithholdingPaymentDetails#TaxTaxableComponentTaxedElementA,SuperannuationIncomeStreamWithholdingPaymentDetails#TaxTaxableComponentUntaxedElementA,LumpSumInArrears#TaxTaxableComponentTaxedElementA,LumpSumInArrears#TaxTaxableComponentUntaxedElementA,LumpSumInArrears,SuperannuationIncomeStreamWithholdingPaymentDetails,Income#RemunerationBonusesAndAllowancesA,Deductions#ExpenseAustralianInvestmentAllowableDeductionA,Deductions#TaxDistributionFrankedA,Deductions#ExpenseForestryManagedInvestmentSchemeDeductionA,OtherDeductions#ExpenseDeductibleOtherT,OtherDeductions#ExpenseDeductibleOtherA,OtherDeductions,OtherDeductionsCollection,Deductions,Income#NetA,CapitalGains#TaxEventI,CapitalGains#NetA,CapitalGains#TaxPayAsYouGoWithholdingCreditForCapitalGainsWithheldFromForeignResidentsA,CapitalGains#TaxExemptionOrRolloverAppliedI,TaxExemptionOrRolloverAppliedC,TaxExemptionOrRolloverAppliedCCollection,CapitalGains,AttributedForeignIncome#InternationalDealingsTrustorControlledCompanyorTransferorTrustI,ListedCountryInternationalDealingsA,UnlistedCountryInternationalDealingsA,OtherAssessableForeignIncome#A,OtherAssessableForeignIncome#NetA,OtherAssessableForeignIncome#InternationalDealingsTaxOffsetA,OtherAssessableForeignIncome#TaxFrankingCreditsReceivedFromNewZealandCompaniesA,OtherAssessableForeignIncome,AttributedForeignIncome,TaxLossesDeductedTaxDeductionA,Income#TaxLossesCarriedForwardLaterIncomeYearsTotalA,Income#CapitalLossesCarriedForwardNetA,Income#TaxOffsetClaimLandcareAndWaterFacilityBroughtForwardA,Income#InternationalDealingsRelatedPartiesTransactionsExcessAggregateValueI,ThinCapitalisation#LiabilitiesProvisionsAppliedI,ThinCapitalisation#ExpenseInterestA,ThinCapitalisation#ExpenseRoyaltiesA,ThinCapitalisation#ResidencyTaxPurposesPersonStatusI,ThinCapitalisation#InternationalDealingsTransactionsWithTaxHavenCountriesI,ThinCapitalisation,PersonalServicesIncome#IncludedI,PersonalServicesIncome#TotalA,PersonalServicesIncome#ExpenseDeductionA,PersonalServicesIncome#TaxIndividualResultsTestSatisfiedI,PersonalServicesIncome#TaxIndividualPersonalServiceBusinessDeterminationHeldI,PersonalServicesIncome#TaxUnrelatedClientsTestSatisfiedI,PersonalServicesIncome#TaxEmploymentTestSatisfiedI,PersonalServicesIncome#TaxBusinessPremisesTestSatisfiedI,PersonalServicesIncome,TaxationOfFinancialArrangements#GainsTotalA,TaxationOfFinancialArrangements#ExpenseLossesTotalA,TaxationOfFinancialArrangements,NCMIBusinessIncomePrimaryProduction#NonConcessionalManagedInvestmentTrustA,NCMIBusinessIncomePrimaryProduction#ExcludedFromNonConcessionalManagedInvestmentTrustA,NCMIBusinessIncomePrimaryProduction,NCMIBusinessIncomeNonPrimaryProduction#NonConcessionalManagedInvestmentTrustA,NCMIBusinessIncomeNonPrimaryProduction#ExcludedFromNonConcessionalManagedInvestmentTrustA,NCMIBusinessIncomeNonPrimaryProduction,NCMIBusinessIncome,NCMIPartnershipsAndTrustsPrimaryProduction#NonConcessionalManagedInvestmentTrustA,NCMIPartnershipsAndTrustsPrimaryProduction#ExcludedFromNonConcessionalManagedInvestmentTrustA,PrimaryProductionShareOfNetIncome#NonConcessionalManagedInvestmentTrustA,PrimaryProductionShareOfNetIncome#ExcludedFromNonConcessionalManagedInvestmentTrustA,PrimaryProductionShareOfNetIncome,NCMIPartnershipsAndTrustsPrimaryProduction,NCMIPartnershipsAndTrustsNonPrimaryProduction#NonConcessionalManagedInvestmentTrustA,NCMIPartnershipsAndTrustsNonPrimaryProduction#ExcludedFromNonConcessionalManagedInvestmentTrustA,NonPrimaryProductionShareOfNetIncome#NonConcessionalManagedInvestmentTrustA,NonPrimaryProductionShareOfNetIncome#ExcludedFromNonConcessionalManagedInvestmentTrustA,NonPrimaryProductionShareOfNetIncome,NCMIPartnershipsAndTrustsNonPrimaryProduction,NCMIPartnershipsAndTrusts,NCMICapitalGains#NonConcessionalManagedInvestmentTrustIncomeA,NCMICapitalGains#ExcludedFromNonConcessionalManagedInvestmentTrustIncomeA,NCMICapitalGains,Income,BusinessDetails,KeyFinancialInformation#AssetsCurrentTotalA,KeyFinancialInformation#AssetsTotalA,KeyFinancialInformation#LiabilitiesCurrentTotalA,KeyFinancialInformation#LiabilitiesTotalA,KeyFinancialInformation,BusinessAndProfessionalInformation#OrganisationNameDetailsOrganisationalNameT,BusinessAddressDetails#OverseasAddressI,BusinessAddressDetails#Line1T,BusinessAddressDetails#Line2T,BusinessAddressDetails#LocalityNameT,BusinessAddressDetails#PostcodeT,BusinessAddressDetails#StateOrTerritoryC,BusinessAddressDetails#CountryC,BusinessAddressDetails,BusinessAndProfessionaltems#AssetsInventoriesTotalA,BusinessAndProfessionaltems#ExpensePurchasesAndOtherCostsA,BusinessAndProfessionaltems#RemunerationWagesAndSalariesTotalA,BusinessAndProfessionaltems#RemunerationWagesAndSalariesActionC,BusinessAndProfessionaltems#RemunerationPaymentToRelatedPartiesGrossA,BusinessAndProfessionaltems#RemunerationFringeBenefitsEmployeeContributionA,BusinessAndProfessionaltems#IncomeTaxPrivateCompanyUnpaidPresentEntitlementA,BusinessAndProfessionaltems#IncomeTaxPrivateCompanyUnpaidPresentEntitlementC,BusinessAndProfessionaltems#ElectionsTradingStockI,Closing#AssetsInventoriesTotalA,Closing#AssetsInventoriesValuationMethodC,Closing#AssetsTradeandReceivablesOtherA,Closing#LiabilitiesTradeAndOtherPayablesTotalA,Closing,BusinessAndProfessionaltems,AggregatedTurnover#IncomeRangeC,AggregatedTurnover#IncomeA,AggregatedTurnover,CapitalAllowances#ExpenseDepreciatingAssetsIntangibleFirstDeductedA,CapitalAllowances#ExpenseDepreciatingAssetsOtherFirstDeductedA,CapitalAllowances#AssetsPropertyPlantAndEquipmentDepreciatingAssetsFirstDeductedSelfAssessedEffectiveLifeI,CapitalAllowances#ExpenseSubsequentYearAcceleratedDepreciationDeductionsForAssetsA,CapitalAllowances,CapitalAllowancesDepreciatingAssets#PropertyPlantAndEquipmentEffectiveLifeRecalculationI,CapitalAllowancesDepreciatingAssets#AdjustableValuesTotalA,CapitalAllowancesDepreciatingAssets#IncomeIntangibleBalancingAdjustmentA,CapitalAllowancesDepreciatingAssets#ExpenseIntangibleBalancingAdjustmentA,CapitalAllowancesDepreciatingAssets#IntangibleTerminationValueA,CapitalAllowancesDepreciatingAssets#OtherTerminationValueA,CapitalAllowancesDepreciatingAssets#ExpenseProjectPoolAllowableDeductionA,CapitalAllowancesDepreciatingAssets#ExpenseCapitalExpenditureSpecifiedAllowableDeductionA,CapitalAllowancesDepreciatingAssets#ExpenseLandcareOperationsAcceleratedDepreciationA,CapitalAllowancesDepreciatingAssets,LowCostDeductionExpenseDepreciatingAssetsSmallBusinessEntityA,GeneralPoolDeductionExpenseDepreciatingAssetsSmallBusinessEntityA,BusinessAndProfessionalInformation,TaxOffsets#TaxOffsetClaimNationalRentalAffordabilitySchemeEntitlementA,TaxOffsets#IncomeTaxTaxOffsetRefundableOtherA,TaxOffsets#IncomeTaxTaxOffsetRefundableOtherC,LimitedPartnershipTaxOffsetTaxOffsetClaimNonRefundableOtherA,InvestorTaxOffsetTaxOffsetClaimNonRefundableOtherA,TaxOffsets,MedicareLevy#IncomeTaxBeneficiarySpouseTaxableIncomeA,MedicareLevy#IncomeTaxDependentChildrenAndStudentsCt,MedicareLevy#IncomeTaxExemptionFullDaysCt,MedicareLevy#IncomeTaxExemptionFullDaysC,MedicareLevy#IncomeTaxExemptionHalfDaysCt,MedicareLevy,RP#IncomeTrustEstateIncomeA,StatementOfDistributionToBeneficiary#TaxFileNumberId,StatementOfDistributionToBeneficiary#TaxFileNumberTrailingNumericId,StatementOfDistributionToBeneficiary#IncomeTaxBeneficiaryEntityTypeC,NonIndividualNameDetailsOrganisationNameDetailsOrganisationalNameT,StatementOfDistributionToBeneficiary#PersonNameDetailsTitleT,StatementOfDistributionToBeneficiary#PersonNameDetailsNameSuffixT,StatementOfDistributionToBeneficiary#PersonNameDetailsFamilyNameT,StatementOfDistributionToBeneficiary#PersonNameDetailsGivenNameT,StatementOfDistributionToBeneficiary#PersonNameDetailsOtherGivenNameT,StatementOfDistributionToBeneficiary#PersonDemographicDetailsBirthD,IndividualNameDetails,StatementOfDistributionToBeneficiary#OverseasAddressI,StatementOfDistributionToBeneficiary#UsageC,StatementOfDistributionToBeneficiary#Line1T,StatementOfDistributionToBeneficiary#Line2T,StatementOfDistributionToBeneficiary#LocalityNameT,StatementOfDistributionToBeneficiary#PostcodeT,StatementOfDistributionToBeneficiary#StateOrTerritoryC,StatementOfDistributionToBeneficiary#CountryC,BeneficiaryAddressDetails,StatementOfDistributionToBeneficiary#IncomeTaxAssessmentCalculationC,StatementOfDistributionToBeneficiary#IncomeTrustEstateIncomeShareA,StatementOfDistributionToBeneficiary#IncomeTaxPayAsYouGoWithholdingCreditForAmountsWithheldFromForeignResidentsA,StatementOfDistributionToBeneficiary#IncomeTaxFrankingCreditsReceivedFromNewZealandCompaniesA,StatementOfDistributionToBeneficiary#IncomeTaxPayAsYouGoWithholdingCreditForTaxWithheldWhereABNNotQuotedA,StatementOfDistributionToBeneficiary#IncomeTaxableMinorBeneficiaryA,StatementOfDistributionToBeneficiary#IncomeDistributionFrankedA,StatementOfDistributionToBeneficiary#IncomeTrustFrankedInvestmentNonPrimaryProductionShareA,StatementOfDistributionToBeneficiary#IncomeTaxFrankingCreditsReceivedFromAustralianCompaniesA,StatementOfDistributionToBeneficiary#IncomeTaxPayAsYouGoWithholdingCreditForAmountsWithheldWhereTFNNotQuotedA,StatementOfDistributionToBeneficiary#IncomeTaxPayAsYouGoWithholdingCreditForTaxWithheldFromCloselyHeldTrustA,StatementOfDistributionToBeneficiary#IncomeCapitalGainsNetA,StatementOfDistributionToBeneficiary#CapitalGainsTotalA,StatementOfDistributionToBeneficiary#CapitalLossesTotalA,StatementOfDistributionToBeneficiary#TaxConcessionCapitalGainsDiscountTotalA,StatementOfDistributionToBeneficiary#TaxConcessionCapitalGainsDiscountAndSmallBusinessConcessionsTotalA,StatementOfDistributionToBeneficiary#CapitalGainsNonConcessionalManagedInvestmentTrustIncomeA,StatementOfDistributionToBeneficiary#CapitalGainsExcludedFromNonConcessionalManagedInvestmentTrustIncomeA,StatementOfDistributionToBeneficiary#IncomeTaxPayAsYouGoWithholdingCreditForCapitalGainsWithheldFromForeignResidentsA,StatementOfDistributionToBeneficiary#IncomeTaxAttributedForeignIncomeA,StatementOfDistributionToBeneficiary#IncomeInternationalDealingsNetA,StatementOfDistributionToBeneficiary#IncomeInternationalDealingsFinancialInvestmentsNetA,StatementOfDistributionToBeneficiary#IncomeInternationalDealingsTaxOffsetA,StatementOfDistributionToBeneficiary#TaxOffsetClaimNationalRentalAffordabilitySchemeEntitlementA,StatementOfDistributionToBeneficiary#IncomeTaxExplorationCreditsDistributedA,StatementOfDistributionToBeneficiary#IncomeSmallBusinessEntityNetA,StatementOfDistributionToBeneficiary#BeneficiaryShareA,StatementOfDistributionToBeneficiary#NonConcessionalManagedInvestmentTrustA,StatementOfDistributionToBeneficiary#ExcludedFromNonConcessionalManagedInvestmentTrustA,ShareOfIncomePrimaryProduction,StatementOfDistributionToBeneficiary#BeneficiaryShareA,StatementOfDistributionToBeneficiary#TrustManagedInvestmentSchemeLessNetCapitalGainAndForeignIncomeNonPrimaryProductionShareA,StatementOfDistributionToBeneficiary#NonConcessionalManagedInvestmentTrustA,StatementOfDistributionToBeneficiary#ExcludedFromNonConcessionalManagedInvestmentTrustA,ShareOfIncomeNonPrimaryProduction,EarlyStageVentureTaxOffsetTaxOffsetClaimNonRefundableOtherA,EarlyStageInvestorTaxOffsetTaxOffsetClaimNonRefundableOtherA,StatementOfDistributionToBeneficiary#InternationalDealingsAssessableIncomeNonResidentBeneficiaryNonTrusteeA,StatementOfDistributionToBeneficiary#InternationalDealingsAssessableIncomeNonResidentBeneficiaryTrusteeA,NonResidentAdditionalInformation,StatementOfDistributionToBeneficiary#IncomeTaxTrustBeneficiaryStatementI,StatementOfDistributionToBeneficiary#IncomeTaxSuperannuationTrusteeBeneficiaryTaxPreferredA,StatementOfDistributionToBeneficiary#IncomeTaxSuperannuationTrusteeBeneficiaryUntaxedPartShareOfNetIncomeA,TBStatementInformation,Information,StatementOfDistributionToBeneficiary#ExpenseDistributionBeneficiaryCloselyHeldTrustA,StatementOfDistributionToBeneficiary#IncomeTaxPayAsYouGoWithholdingCreditForTaxWithheldFromCloselyHeldTrustA,AnnualTrusteePaymentReportInformation,StatementOfDistributionToBeneficiary,StatementOfDistributionToBeneficiaryCollection,IncomeNoBeneficiaryPresentlyEntitled#IncomeTaxAssessmentCalculationC,IncomeNoBeneficiaryPresentlyEntitled#TrustEstateIncomeShareA,IncomeNoBeneficiaryPresentlyEntitled#IncomeTaxPayAsYouGoWithholdingCreditForAmountsWithheldFromForeignResidentsA,IncomeNoBeneficiaryPresentlyEntitled#IncomeTaxFrankingCreditsReceivedFromNewZealandCompaniesA,IncomeNoBeneficiaryPresentlyEntitled#IncomeTaxPayAsYouGoWithholdingCreditForTaxWithheldWhereABNNotQuotedA,IncomeNoBeneficiaryPresentlyEntitled#DistributionFrankedA,IncomeNoBeneficiaryPresentlyEntitled#IncomeTaxFrankingCreditsReceivedFromAustralianCompaniesA,IncomeNoBeneficiaryPresentlyEntitled#IncomeTaxPayAsYouGoWithholdingCreditForAmountsWithheldWhereTFNNotQuotedA,IncomeNoBeneficiaryPresentlyEntitled#IncomeTaxPayAsYouGoWithholdingCreditForTaxWithheldFromCloselyHeldTrustA,IncomeNoBeneficiaryPresentlyEntitled#CapitalGainsNetA,IncomeNoBeneficiaryPresentlyEntitled#CapitalGainsNonConcessionalManagedInvestmentTrustIncomeA,IncomeNoBeneficiaryPresentlyEntitled#CapitalGainsExcludedFromNonConcessionalManagedInvestmentTrustIncomeA,IncomeNoBeneficiaryPresentlyEntitled#IncomeTaxPayAsYouGoWithholdingCreditForCapitalGainsWithheldFromForeignResidentsA,IncomeNoBeneficiaryPresentlyEntitled#IncomeTaxAttributedForeignIncomeA,IncomeNoBeneficiaryPresentlyEntitled#InternationalDealingsNetA,IncomeNoBeneficiaryPresentlyEntitled#InternationalDealingsTaxOffsetA,IncomeNoBeneficiaryPresentlyEntitled#TaxOffsetClaimNationalRentalAffordabilitySchemeEntitlementA,IncomeNoBeneficiaryPresentlyEntitled#IncomeTaxTaxOffsetRefundableOtherA,PrimaryProductionNoBeneficiaryEntitled#IncomeTaxIncomeOrLossTaxableA,PrimaryProductionNoBeneficiaryEntitled#NonConcessionalManagedInvestmentTrustA,PrimaryProductionNoBeneficiaryEntitled#ExcludedFromNonConcessionalManagedInvestmentTrustA,PrimaryProductionNoBeneficiaryEntitled,NonPrimaryProductionNoBeneficiaryEntitled#IncomeTaxIncomeOrLossTaxableA,NonPrimaryProductionNoBeneficiaryEntitled#NonConcessionalManagedInvestmentTrustA,NonPrimaryProductionNoBeneficiaryEntitled#ExcludedFromNonConcessionalManagedInvestmentTrustA,NonPrimaryProductionNoBeneficiaryEntitled,EarlyStageVentureNoBeneficiary#TaxOffsetClaimNonRefundableOtherA,EarlyStageVentureNoBeneficiaryCarriedForwardTaxOffsetClaimNonRefundableOtherA,EarlyStageVentureNoBeneficiary,InvestorNoBeneficiary#TaxOffsetClaimNonRefundableOtherA,InvestorNoBeneficiaryCarriedForwardTaxOffsetClaimNonRefundableOtherA,InvestorNoBeneficiary,IncomeNoBeneficiaryPresentlyEntitled,CapitalGainsAssessed#CapitalGainsTaxAssessmentCalculationC,CapitalGainsAssessed#CapitalGainsTaxTrusteeAssessedCapitalGainA,CapitalGainsAssessed,RP#IncomeTaxBeneficiaryUnderLegalDisabilityAlsoPresentlyEntitledToIncomeFromAnotherTrustI,NonResidentTrust#ResidencyTaxPurposesOrganisationStatusC,NonResidentTrust#InternationalDealingsAustralianSourcedIncomeNoPresentEntitlementA,NonResidentTrust,OtherReportingInformation#RegulatoryDisclosuresGeneralInformationAboutFinancialStatementsT,OtherReportingInformation#CompletionHoursN,OtherReportingInformation,ReportingPartyDeclaration#StatementTypeC,ReportingPartyDeclaration#StatementAcceptedI,ReportingPartyDeclaration#StatementT,ReportingPartyDeclaration#SignatureD,ReportingPartyDeclaration#SignatoryIdentifierT,ReportingPartyDeclaration#PersonUnstructuredNameFullNameT,ReportingPartyDeclaration,RP,INT#TaxAgentClientReferenceT,INT#TaxAgentNumberId,INT#AustralianBusinessNumberId,Declaration#StatementTypeC,Declaration#StatementAcceptedI,Declaration#StatementT,Declaration#SignatureD,Declaration#SignatoryIdentifierT,IntermediaryContact#PersonUnstructuredNameUsageC,IntermediaryContact#PersonUnstructuredNameFullNameT,IntermediaryContact,IntermediaryContactCollection,Declaration,IntermediaryContactTelephone#ElectronicContactUsageC,IntermediaryContactTelephone#ElectronicContactServiceLineC,IntermediaryContactTelephone#ElectronicContactAreaC,IntermediaryContactTelephone#ElectronicContactMinimalN,IntermediaryContactTelephone,INT";
    }
}