﻿using Ato.CD.Inbound.Shared;
using Ato.CD.Inbound.Shared201802;
using Ato.EN.IntegrationServices.CodeGenerationCGTS;
using Ato.EN.IntegrationServices.CodeGenerationTRT;
using DataContracts;
using System.Collections.Generic;
using System.Linq;
using System.Runtime.CompilerServices;

namespace Ato.CD.Inbound.TRT
{
    public class CrossFormValidatorCGTS : ICrossFormValidator
    {
        private CGTS2018 childDocument { get; set; }

        private List<Context> contexts { get; set; }

        public List<ProcessMessageDocument> response { get; private set; }

        private TRT2026 parentDocument { get; set; }

        public CrossFormValidatorCGTS(TRT2026 parent, object child)
        {
            childDocument = (CGTS2018)child;
            contexts = childDocument.GetContexts();
            parentDocument = parent;
            response = new List<ProcessMessageDocument>();
        }

        public List<ProcessMessageDocument> ValidateCrossFormRules()
        {
            VRATOCGTS402165();
            VRATOCGTS402171();
            VRATOCGTS402174();
            VRATOCGTS402226();
            VRATOCGTS402230();
            VRATOCGTS402240();
            VRATOGEN402009();
            VRATOGEN438000();
            VRATOGEN438001();
            VRATOTRT432189();
            VRATOTRT432213();
            VRATOTRT433043();
            VRATOTRT433044();
            VRATOTRT433045();
            VRATOTRT433046();
            VRATOTRT433047();
            VRATOTRT433048();
            VRATOTRT433049();

            return response;
        }

        #region Validation Rules

        #region VR.ATO.CGTS.402165

        /*  VR.ATO.CGTS.402165
        Total losses transferred in applied is only valid for a company

        Legacy Rule Format:
        IF (PARENT RETURN <> "CTR") AND ([CGTS122] <> NULL)
          RETURN VALIDATION MESSAGE
        ENDIF

        Technical Business Rule Format:
        IF (PARENT RETURN <> "CTR") AND ([CGTS122] <> NULL)
          RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:

        ^CGTS122 = RPApplied:Capital.Losses.TransferredTotal.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCGTS402165()
        {
            if (childDocument.CGTS122.HasValue)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.CGTS.402165",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total losses transferred in applied is only valid for a company",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.TransferredTotal.Amount",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402165" },
                        new ProcessMessageParameter { Name = "CGTS122", Value = TRT2026Validator.GetValueOrEmpty(childDocument.CGTS122) }
                    }
                });
            }
        }
        #endregion // VR.ATO.CGTS.402165

        #region VR.ATO.CGTS.402171

        /*  VR.ATO.CGTS.402171
        A capital loss must not be carried forward when a positive net capital gain amount has been provided

        Legacy Rule Format:
        IF PARENT RETURN = SET("SMSFAR","FITR","TRT","TRTAMI") AND ([CGTS131] > 0) AND ([CGTS125]  > 0)
            RETURN VALIDATION MESSAGE 
        ENDIF

        Technical Business Rule Format:
        IF PARENT RETURN = SET("SMSFAR","FITR","TRT","TRTAMI") AND ([CGTS131] > 0) AND ([CGTS125]  > 0)
            RETURN VALIDATION MESSAGE 
        ENDIF

        Data Elements:

        ^CGTS131 = RP:Income.CapitalGainsNet.Amount

        ^CGTS125 = RP.CGTOther.Unapplied(Instant):Capital.Losses.CarriedForward.Net.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCGTS402171()
        {
            bool assertion = childDocument.CGTS131.GetValueOrDefault() > 0 && childDocument.CGTS125.GetValueOrDefault() > 0;

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.CGTS.402171",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"A capital loss must not be carried forward when a positive net capital gain amount has been provided",
                    Location = "/xbrli:xbrl/tns:Capital.Losses.CarriedForward.Net.Amount[@contextRef='" + childDocument.RPCGTOtherUnappliedInstantId + "']",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402171" },
                        new ProcessMessageParameter { Name = "CGTS131", Value = TRT2026Validator.GetValueOrEmpty(childDocument.CGTS131) },
                        new ProcessMessageParameter { Name = "CGTS125", Value = TRT2026Validator.GetValueOrEmpty(childDocument.CGTS125) }
                    }
                });
            }
        }
        #endregion // VR.ATO.CGTS.402171

        #region VR.ATO.CGTS.402174

        /*  VR.ATO.CGTS.402174
        The CGT discount rate available to trusts and companies is 50%.

        Legacy Rule Format:
        IF PARENT RETURN = SET("TRT", "CTR","TRTAMI") AND  [CGTS126] > ((([CGTS118] - [CGTS123]) *0.5) +1)
            RETURN VALIDATION MESSAGE
        ENDIF

        Technical Business Rule Format:
        IF PARENT RETURN = SET("TRT", "CTR","TRTAMI") AND  [CGTS126] > ((([CGTS118] - [CGTS123]) *0.5) +1)
            RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:

        ^CGTS126 = RP.Applied:TaxConcession.CapitalGains.DiscountTotal.Amount

        ^CGTS118 = RP:Capital.Gains.Total.Amount

        ^CGTS123 = RP.Applied:Capital.Losses.Total.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCGTS402174()
        {
            bool assertion = childDocument.CGTS126.GetValueOrDefault() > (childDocument.CGTS118.GetValueOrDefault() - childDocument.CGTS123.GetValueOrDefault()) * (decimal)0.5 + 1;

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.CGTS.402174",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The CGT discount rate applied to capital gains by a trust or company must not be greater than 50%",
                    Location = "/xbrli:xbrl/tns:TaxConcession.CapitalGains.DiscountTotal.Amount",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402174" },
                        new ProcessMessageParameter { Name = "CGTS126", Value = TRT2026Validator.GetValueOrEmpty(childDocument.CGTS126) },
                        new ProcessMessageParameter { Name = "CGTS118", Value = TRT2026Validator.GetValueOrEmpty(childDocument.CGTS118) },
                        new ProcessMessageParameter { Name = "CGTS123", Value = TRT2026Validator.GetValueOrEmpty(childDocument.CGTS123) }
                    }
                });
            }
        }
        #endregion // VR.ATO.CGTS.402174

        #region VR.ATO.CGTS.402226

        /*  VR.ATO.CGTS.402226
        If the Amendment indicator on a parent return is set to TRUE, then the Income year earnout right created and Amount to be amended cannot be supplied.

        Legacy Rule Format:
        IF (PARENT RETURN:RP:pyin.xx.xx:Report.Amendment.Indicator = TRUE) AND ([CGTS150] <> NULL AND [CGTS151] <> NULL)
          RETURN VALIDATION MESSAGE
        ENDIF

        Technical Business Rule Format:
           IF (PARENT RETURN:RP:pyin.xx.xx:Report.Amendment.Indicator = TRUE) AND ([CGTS150] <> NULL AND [CGTS151] <> NULL)
          RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:

        ^TRT414 = TRT:RP:ReportAmendment:Report.Amendment.Indicator

        ^CGTS150 = RP:Report.TargetFinancial.Year
            
        ^CGTS151 = RP:Miscellaneous.RequestedAmendment.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCGTS402226()
        {
            bool assertion = parentDocument.TRT414.GetValueOrDefault() && childDocument.CGTS150.HasValue && childDocument.CGTS151.HasValue;

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.CGTS.402226",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Income year earnout right created and Amount to be amended cannot be supplied when Parent return Amendment indicator is present.",
                    Location = "/xbrli:xbrl/tns:Report.TargetFinancial.Year",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402226" },
                        new ProcessMessageParameter { Name = "TRT414", Value = TRT2026Validator.GetValueOrEmpty(parentDocument.TRT414) },
                        new ProcessMessageParameter { Name = "CGTS150", Value = TRT2026Validator.GetValueOrEmpty(childDocument.CGTS150) },
                        new ProcessMessageParameter { Name = "CGTS151", Value = TRT2026Validator.GetValueOrEmpty(childDocument.CGTS151) }
                    }
                });
            }
        }

        #endregion // VR.ATO.CGTS.402226

        #region VR.ATO.CGTS.402230

        /*  VR.ATO.CGTS.402230
        A multi-class treatment can only be reported with a Trust Tax Return for Attribution Managed Investments (TRTAMI) when a multi-class election has been made

        Legacy Rule Format:
        IF PARENT RETURN  <> ""TRTAMI"" AND ([CGTS153] <> NULL OR [CGTS154] <> NULL OR [CGTS155] <> NULL)
           RETURN VALIDATION MESSAGE
        ENDIF

        Technical Business Rule Format:
         IF PARENT RETURN  <> ""TRTAMI"" AND ([CGTS153] <> NULL OR [CGTS154] <> NULL OR [CGTS155] <> NULL)
           RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:

        ^CGTS153 = RP:IncomeTax.AttributionManagedInvestmentTrustClass.Description
 
        ^CGTS154 = RP.TransfersToOtherClasses:Capital.Gains.Total.Amount

        ^CGTS155 = RP.TransfersToOtherClasses:Capital.Losses.Total.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCGTS402230()
        {
            bool assertion = !string.IsNullOrEmpty(childDocument.CGTS153) || childDocument.CGTS154.HasValue || childDocument.CGTS155.HasValue;

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.CGTS.402230",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Multi-class treatment fields incorrectly supplied",
                    Location = "/xbrli:xbrl/tns:IncomeTax.AttributionManagedInvestmentTrustClass.Description",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402230" },
                        new ProcessMessageParameter { Name = "CGTS153", Value = TRT2026Validator.GetValueOrEmpty(childDocument.CGTS153) },
                        new ProcessMessageParameter { Name = "CGTS154", Value = TRT2026Validator.GetValueOrEmpty(childDocument.CGTS154) },
                        new ProcessMessageParameter { Name = "CGTS155", Value = TRT2026Validator.GetValueOrEmpty(childDocument.CGTS155) }
                    }
                });
            }
        }
        #endregion // VR.ATO.CGTS.402230

        #region VR.ATO.CGTS.402240

        /*  VR.ATO.CGTS.402240
        The year entered at Income year earnout right created cannot be the current year.

        Legacy Rule Format:
        IF ([CGTS150] <> NULL) AND ([CGTS150] = (PARENT RETURN:RP:pyin.xx.xx:Report.TargetFinancial.Year))
            RETURN VALIDATION MESSAGE
        ENDIF

        Technical Business Rule Format:
        IF ([CGTS150] <> NULL) AND ([CGTS150] = (PARENT RETURN:RP:pyin.xx.xx:Report.TargetFinancial.Year))
            RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:

        ^TRT1 = TRT:RP:TargetFinancialY

        ^CGTS150 = RP:Report.TargetFinancial.Year
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOCGTS402240()
        {
            bool assertion = childDocument.CGTS150.HasValue && childDocument.CGTS150.GetValueOrDefault().Equals(parentDocument.TRT1.GetValueOrDefault());

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.CGTS.402240",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Income year earnout right created incorrect",
                    Location = "/xbrli:xbrl/tns:Report.TargetFinancial.Year",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.CGTS.402240" },
                        new ProcessMessageParameter { Name = "TRT1", Value = TRT2026Validator.GetValueOrEmpty(parentDocument.TRT1) },
                        new ProcessMessageParameter { Name = "CGTS150", Value = TRT2026Validator.GetValueOrEmpty(childDocument.CGTS150) }
                    }
                });
            }
        }
        #endregion // VR.ATO.CGTS.402240

        #region VR.ATO.GEN.402009

        /*  VR.ATO.GEN.402009
        Your supplied TFN does not match the TFN supplied on the form it was submitted with

        Legacy Rule Format:
        IF (RP:entity.identifier.TFN <> PARENT RETURN:RP:entity.identifier.TFN)
          RETURN VALIDATION MESSAGE
        ENDIF

        Technical Business Rule Format:
        IF (RP:entity.identifier.TFN <> PARENT RETURN:RP:entity.identifier.TFN)
          RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:

        ^TRT5 = TRT:RP:Identifiers.TaxFileNumber.Identifier

        ^CGTS_TFN = RPIdentifierTFN
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN402009()
        {
            bool assertion = childDocument.RPExists && !parentDocument.TRT5.Equals(childDocument.RPIdentifierTFN);

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.402009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your supplied TFN does not match the TFN supplied on the form it was submitted with",
                    LongDescription = @"Your supplied TFN does not match the TFN supplied on the parent form it was submitted with",
                    Location = " /xbrli:xbrl/xbrli:context/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.GEN.402009" },
                        new ProcessMessageParameter { Name = "TRT5", Value = TRT2026Validator.GetValueOrEmpty(parentDocument.TRT5) },
                        new ProcessMessageParameter { Name = "CGTS TFN", Value = TRT2026Validator.GetValueOrEmpty(childDocument.RPIdentifierTFN) },
                    }
                });
            }
        }

        #endregion // VR.ATO.GEN.402009

        #region VR.ATO.GEN.438000

        /*  VR.ATO.GEN.438000
        Period start date within context on schedule doesn't match start date within PARENT RETURN:reporting party context

        Legacy Rule Format:
        IF (period.startDate WHERE CONTEXT(ALL)) <> PARENT RETURN:RP:period.startDate

        Technical Business Rule Format:
        IF (period.startDate WHERE CONTEXT(ALL)) <> PARENT RETURN:RP:period.startDate

        Data Elements:
            
        ^TRT573 = TRT:RP:Period.Start.Date
            
        ^ContextStartDate = context.StartDate
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438000()
        {
            response.AddRange(from context in contexts
                where context.StartDate.GetValueOrDefault() != parentDocument.TRT573.GetValueOrDefault()
                select new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.438000",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The context period start date is incorrect.",
                    LongDescription = @"Period start date within context on the schedule must match period start date within the parent return reporting party context",
                    Location = $"/xbrli:xbrl/xbrli:context[{context.Index}]/xbrli:period/xbrli:startDate",
                    Parameters = new ProcessMessageParameters {
                    new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438000" },
                    new ProcessMessageParameter { Name = "TRT573", Value = TRT2026Validator.GetValueOrEmptyDate(parentDocument.TRT573) },
                    new ProcessMessageParameter { Name = "ContextStartDate", Value = TRT2026Validator.GetValueOrEmptyDate(context.StartDate) }
                    }
                }
            );
        }
        #endregion // VR.ATO.GEN.438000

        #region VR.ATO.CGTS.438001

        /*  VR.ATO.GEN.438001
        Period end date within context on schedule doesn't match end date within PARENT RETURN:reporting party context

        Legacy Rule Format:
        IF (period.endDate WHERE CONTEXT(ALL)) <> PARENT RETURN:RP:period.endDate
          RETURN VALIDATION MESSAGE
        ENDIF

        Technical Business Rule Format:
        IF (period.endDate WHERE CONTEXT(ALL)) <> PARENT RETURN:RP:period.endDate
          RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:

        ^TRT574 = TRT:RP:Period.End.Date

        ^ContextEndDate = context.EndDate
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438001()
        {
            response.AddRange(from context in contexts
                where context.EndDate.GetValueOrDefault() != parentDocument.TRT574.GetValueOrDefault()
                select new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.438001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The context period end date is incorrect.",
                    LongDescription = @"Period end date within context on the schedule must match period end date within the parent return reporting party context",
                    Location = $"/xbrli:xbrl/xbrli:context[{context.Index}]/xbrli:period/xbrli:endDate",
                    Parameters = new ProcessMessageParameters {
                    new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438001" },
                    new ProcessMessageParameter { Name = "TRT574", Value = TRT2026Validator.GetValueOrEmptyDate(parentDocument.TRT574) },
                    new ProcessMessageParameter { Name = "ContextEndDate", Value = TRT2026Validator.GetValueOrEmptyDate(context.EndDate) }
                    }
                }
            );
        }
        #endregion // VR.ATO.CGTS.438001

        #region VR.ATO.TRT.432189

        /*  VR.ATO.TRT.432189
        The Net Capital gain amount on the main return form must equal the Total net capital gain amount on the Capital Gains Tax schedule.

        Legacy Rule Format:
        CountDocument('CGTS') = 1 AND ^TRT133 <> ^CGTS131

        Technical Business Rule Format:
        CountDocument('CGTS') = 1 AND ^TRT133 <> ^CGTS131

        Data Elements:
    
        ^TRT133 = TRT:RP:BusinessDetails:Income:CapitalGains:Income.CapitalGainsNet.Amount
    
        ^CGTS131 = CGTS:RP:Income.CapitalGainsNet.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRT432189()
        {
            bool assertion = parentDocument.TRT133.GetValueOrDefault() != childDocument.CGTS131.GetValueOrDefault();

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.434036",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Net capital gains in return must equal total net capital gain in CGT schedule.",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:CapitalGains/tns:NetA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.432189" },
                        new ProcessMessageParameter() { Name = "TRT133", Value = parentDocument.TRT133.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "CGTS131", Value = childDocument.CGTS131.GetValueOrDefault().ToString() },
                    },
                    DocumentSequence = "parent_doc"
                });
            }
        }
        #endregion // VR.ATO.TRT.432189

        #region VR.ATO.TRT.432213

        /*  VR.ATO.TRT.432213
        Unapplied net capital losses carried forward on the Capital Gains Tax schedule (if lodged) must equal Net capital losses carried forward on the main income tax return.

        Legacy Rule Format:
        CountDocument('CGTS') = 1 AND (^CGTS124 + ^CGTS125) <> ^TRT475

        Technical Business Rule Format:
        CountDocument('CGTS') = 1 AND (^CGTS124 + ^CGTS125) <> ^TRT475

        Data Elements:

        ^CGTS124 = CGTS:RP.Collectables.Unapplied(Instant):rvctc3.02.16:Capital.Losses.CarriedForward.Net.Amount

        ^CGTS125 = CGTS:RP.CGTOther.Unapplied(Instant):rvctc3.02.16:Capital.Losses.CarriedForward.Net.Amount
        
        ^TRT475 = TRT:RP:BusinessDetails:Income:Capital.Losses.CarriedForward.Net.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRT432213()
        {
            bool assertion = (childDocument.CGTS124.GetValueOrDefault() + childDocument.CGTS125.GetValueOrDefault()) != parentDocument.TRT475.GetValueOrDefault();
            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.434088",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Net capital losses carried forward to later income years must equal amount in CGT schedule",
                    LongDescription = @"The total of the 'Unapplied net capital losses carried forward'  in the Capital Gains Tax schedule must equal 'Net capital losses carried forward to later income years' on the main form",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:CapitalLossesCarriedForwardNetA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.432213" },
                        new ProcessMessageParameter() { Name = "CGTS124", Value = childDocument.CGTS124.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "CGTS125", Value = childDocument.CGTS125.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "TRT475", Value = parentDocument.TRT475.GetValueOrDefault().ToString() },
                    },
                    DocumentSequence = "parent_doc"
                });
            }
        }
        #endregion // VR.ATO.TRT.432213

        #region VR.ATO.TRT.433043

        /*  VR.ATO.TRT.433043
        If taxpayer is required to lodge a CGT schedule and CGT exemption or rollover applied code is "Small business active asset reduction (Subdivision 152-C)", then Small business active asset reduction amount on attached CGT schedule must be completed.

        Legacy Rule Format:
        ^TRT448 = 'A - Small business active asset reduction' AND CountDocument('CGTS') = 1 AND ^CGTS127 = NULL

        Technical Business Rule Format:
        ^TRT448 = 'A - Small business active asset reduction' AND CountDocument('CGTS') = 1 AND ^CGTS127 = NULL

        Data Elements:

        ^TRT448 = TRT:RP:BusinessDetails:Income:CapitalGains:CapitalGainsTax.ExemptionOrRolloverApplied.Code

        ^CGTS127 = CGTS:RP:TaxConcession.CapitalGains.SmallBusinessActiveAssetReduction.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRT433043()
        {
            bool assertion = (parentDocument.TRT448Collection != null && parentDocument.TRT448Collection.Any(c => c == @"A - Small business active asset reduction")) && childDocument.CGTS127 == null;

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.438022",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Small business active asset reduction amount missing from CGT schedule",
                    LongDescription = @"If taxpayer is required to lodge a CGT schedule and CGT exemption or rollover applied code is ""Small business active asset reduction (Subdivision 152-C)"", then Small business active asset reduction amount on attached CGT schedule must be completed",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:CapitalGains/tns:TaxExemptionOrRolloverAppliedCCollection/tns:TaxExemptionOrRolloverAppliedC",
                    Parameters = new ProcessMessageParameters() {
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.433043" },
                        new ProcessMessageParameter() { Name = "TRT448", Value = "A - Small business active asset reduction" },
                        new ProcessMessageParameter() { Name = "CGTS127", Value = "null" }
                    },
                    DocumentSequence = "parent_doc"
                });
            }
        }
        #endregion // VR.ATO.TRT.433043

        #region VR.ATO.TRT.433044

        /*  VR.ATO.TRT.433044
        If taxpayer is required to lodge a CGT schedule and CGT exemption or rollover applied code is "Small business retirement exemption (Subdivision152-D)", then Small business retirement exemption amount on attached CGT schedule must be completed.

        Legacy Rule Format:
        ^TRT448 = 'B - Small business retirement exemption' AND CountDocument('CGTS') = 1 AND ^CGTS128 = NULL

        Technical Business Rule Format:
        ^TRT448 = 'B - Small business retirement exemption' AND CountDocument('CGTS') = 1 AND ^CGTS128 = NULL

        Data Elements:

        ^TRT448 = TRT:RP:BusinessDetails:Income:CapitalGains:CapitalGainsTax.ExemptionOrRolloverApplied.Code

        ^CGTS128 = CGTS:RP:TaxConcession.CapitalGains.SmallBusinessRetirementExemption.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRT433044()
        {
            bool assertion = (parentDocument.TRT448Collection != null && parentDocument.TRT448Collection.Any(c => c == @"B - Small business retirement exemption")) && childDocument.CGTS128 == null;

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.438023",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Small business retirement exemption amount missing from CGT schedule",
                    LongDescription = @"If taxpayer is required to lodge a CGT schedule and CGT exemption or rollover applied code is ""Small business retirement exemption (Subdivision152-D)"", then Small business retirement exemption amount on attached CGT schedule must be completed",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:CapitalGains/tns:TaxExemptionOrRolloverAppliedCCollection/tns:TaxExemptionOrRolloverAppliedC",
                    Parameters = new ProcessMessageParameters() {
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.433044" },
                        new ProcessMessageParameter() { Name = "TRT448", Value = "B - Small business retirement exemption" },
                        new ProcessMessageParameter() { Name = "CGTS128", Value = "null" }
                    },
                    DocumentSequence = "parent_doc"
                });
            }
        }
        #endregion // VR.ATO.TRT.433044

        #region VR.ATO.TRT.433045

        /*  VR.ATO.TRT.433045
        If taxpayer is required to lodge a CGT schedule and CGT exemption or rollover applied code is "Small business roll-over (Subdivision 152-E)", then Small business roll-over amount on attached CGT schedule must be completed.

        Legacy Rule Format:
        ^TRT448 = 'C - Small business roll-over' AND CountDocument('CGTS') = 1 AND ^CGTS129 = NULL

        Technical Business Rule Format:
        ^TRT448 = 'C - Small business roll-over' AND CountDocument('CGTS') = 1 AND ^CGTS129 = NULL

        Data Elements:
    
        ^TRT448 = TRT:RP:BusinessDetails:Income:CapitalGains:CapitalGainsTax.ExemptionOrRolloverApplied.Code

        ^CGTS129 = CGTS:RP:TaxConcession.CapitalGains.SmallBusinessRollover.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRT433045()
        {
            bool assertion = (parentDocument.TRT448Collection != null && parentDocument.TRT448Collection.Any(c => c == @"C - Small business roll-over")) && childDocument.CGTS129 == null;

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.438024",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Small business roll-over amount missing from CGT schedule",
                    LongDescription = @"If taxpayer is required to lodge a CGT schedule and CGT exemption or rollover applied code is ""Small business roll-over (Subdivision 152-E)"", then Small business roll-over amount on attached CGT schedule must be completed",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:CapitalGains/tns:TaxExemptionOrRolloverAppliedCCollection/tns:TaxExemptionOrRolloverAppliedC",
                    Parameters = new ProcessMessageParameters() {
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.433045" },
                        new ProcessMessageParameter() { Name = "TRT448", Value = "C - Small business roll-over" },
                        new ProcessMessageParameter() { Name = "CGTS129", Value = "null" }
                    },
                    DocumentSequence = "parent_doc"
                });
            }
        }
        #endregion // VR.ATO.TRT.433045

        #region VR.ATO.TRT.433046

        /*  VR.ATO.TRT.433046
        If taxpayer is required to lodge a CGT schedule and CGT exemption or rollover applied code is "Small business 15 year exemption (Subdivision152-B)", then Small business 15 year exemption amount on attached CGT schedule must be completed.

        Legacy Rule Format:
        ^TRT448 = 'D - Small business 15 year exemption' AND CountDocument('CGTS') = 1 AND ^CGTS138 = NULL

        Technical Business Rule Format:
        ^TRT448 = 'D - Small business 15 year exemption' AND CountDocument('CGTS') = 1 AND ^CGTS138 = NULL

        Data Elements:

        ^TRT448 = TRT:RP:BusinessDetails:Income:CapitalGains:CapitalGainsTax.ExemptionOrRolloverApplied.Code

        ^CGTS138 = CGTS:RP:TaxConcession.CapitalGains.SmallBusinessExemptionExempt.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRT433046()
        {
            bool assertion = (parentDocument.TRT448Collection != null && parentDocument.TRT448Collection.Any(c => c == @"D - Small business 15 year exemption")) && childDocument.CGTS138 == null;

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.438025",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Small business 15 year exemption amount missing from CGT schedule",
                    LongDescription = @"If taxpayer is required to lodge a CGT schedule and CGT exemption or rollover applied code is ""Small business 15 year exemption (Subdivision152-B)"", then Small business 15 year exemption amount on attached CGT schedule must be completed",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:CapitalGains/tns:TaxExemptionOrRolloverAppliedCCollection/tns:TaxExemptionOrRolloverAppliedC",
                    Parameters = new ProcessMessageParameters() {
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.433046" },
                        new ProcessMessageParameter() { Name = "TRT448", Value = "D - Small business 15 year exemption" },
                        new ProcessMessageParameter() { Name = "CGTS138", Value = "null" }
                    },
                    DocumentSequence = "parent_doc"
                });
            }
        }
        #endregion // VR.ATO.TRT.433046

        #region VR.ATO.TRT.433047

        /*  VR.ATO.TRT.433047
        If taxpayer is required to lodge a CGT schedule and CGT exemption or rollover applied code is "Foreign resident CGT exemption (Division 855)", then Capital gains disregarded by a foreign resident amount on attached CGT schedule must be completed.

        Legacy Rule Format:
        ^TRT448 = 'E - Foreign resident CGT exemption' AND CountDocument('CGTS') = 1 AND ^CGTS140 = NULL

        Technical Business Rule Format:
        ^TRT448 = 'E - Foreign resident CGT exemption' AND CountDocument('CGTS') = 1 AND ^CGTS140 = NULL

        Data Elements: 

        ^TRT448 = TRT:RP:BusinessDetails:Income:CapitalGains:CapitalGainsTax.ExemptionOrRolloverApplied.Code

        ^CGTS140 = CGTS:RP:Capital.Gains.ForeignResidentExemption.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRT433047()
        {
            bool assertion = (parentDocument.TRT448Collection != null && parentDocument.TRT448Collection.Any(c => c == @"E - Foreign resident CGT exemption")) && childDocument.CGTS140 == null;

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.438026",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Capital gains disregarded by a foreign resident amount missing from CGT schedule",
                    LongDescription = @"If taxpayer is required to lodge a CGT schedule and CGT exemption or rollover applied code is ""Foreign resident CGT exemption (Division 855)"", then Capital gains disregarded by a foreign resident amount on attached CGT schedule must be completed",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:CapitalGains/tns:TaxExemptionOrRolloverAppliedCCollection/tns:TaxExemptionOrRolloverAppliedC",
                    Parameters = new ProcessMessageParameters() {
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.433047" },
                        new ProcessMessageParameter() { Name = "TRT448", Value = "E - Foreign resident CGT exemption" },
                        new ProcessMessageParameter() { Name = "CGTS140", Value = "null" }
                    },
                    DocumentSequence = "parent_doc"
                });
            }
        }
        #endregion // VR.ATO.TRT.433047

        #region VR.ATO.TRT.433048

        /*  VR.ATO.TRT.433048
        If taxpayer is required to lodge a CGT schedule and CGT exemption or rollover applied code is "Scrip for scrip roll-over (Subdivision 124-M)", then Capital gains disregarded as a result of a scrip for scrip roll-over amount on attached CGT schedule must be completed.

        Legacy Rule Format:
        ^TRT448 = 'F - Scrip for scrip roll-over' AND CountDocument('CGTS') = 1 AND ^CGTS45 = NULL

        Technical Business Rule Format:
        ^TRT448 = 'F - Scrip for scrip roll-over' AND CountDocument('CGTS') = 1 AND ^CGTS45 = NULL

        Data Elements:

        ^TRT448 = TRT:RP:BusinessDetails:Income:CapitalGains:CapitalGainsTax.ExemptionOrRolloverApplied.Code

        ^CGTS45 = CGTS:RP:Capital.Gains.ScriptForScriptRolloverDeferral.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRT433048()
        {
            bool assertion = (parentDocument.TRT448Collection != null && parentDocument.TRT448Collection.Any(c => c == @"F - Scrip for scrip roll-over")) && childDocument.CGTS45 == null;

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.438027",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Capital gains disregarded as a result of a scrip for scrip roll-over amount missing from CGT schedule",
                    LongDescription = @"If taxpayer is required to lodge a CGT schedule and CGT exemption or rollover applied code is ""Scrip for scrip roll-over (Subdivision 124-M)"", then Capital gains disregarded as a result of a scrip for scrip roll-over amount on attached CGT schedule must be completed",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:CapitalGains/tns:TaxExemptionOrRolloverAppliedCCollection/tns:TaxExemptionOrRolloverAppliedC",
                    Parameters = new ProcessMessageParameters() {
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.433048" },
                        new ProcessMessageParameter() { Name = "TRT448", Value = "F - Scrip for scrip roll-over" },
                        new ProcessMessageParameter() { Name = "CGTS45", Value = "null" }
                    },
                    DocumentSequence = "parent_doc"
                });
            }
        }
        #endregion // VR.ATO.TRT.433048

        #region VR.ATO.TRT.433049

        /*  VR.ATO.TRT.433049
        If taxpayer is required to lodge a CGT schedule and CGT exemption or rollover applied code is "Demerger exemption (Subdivision 125-C)", then Capital gains disregarded by a demerging entity amount on attached CGT schedule must be completed.

        Legacy Rule Format:
        ^TRT448 = 'H - Demerger exemption' AND CountDocument('CGTS') = 1 AND ^CGTS47 = NULL

        Technical Business Rule Format:
        ^TRT448 = 'H - Demerger exemption' AND CountDocument('CGTS') = 1 AND ^CGTS47 = NULL

        Data Elements:

        ^TRT448 = TRT:RP:BusinessDetails:Income:CapitalGains:CapitalGainsTax.ExemptionOrRolloverApplied.Code

        ^CGTS47 = CGTS:RP:Capital.Gains.DemergingEntityExemption.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRT433049()
        {
            bool assertion = (parentDocument.TRT448Collection != null && parentDocument.TRT448Collection.Any(c => c == @"H - Demerger exemption")) && childDocument.CGTS47 == null;

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.438028",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Capital gains disregarded by a demerging entity amount missing from CGT schedule",
                    LongDescription = @"If taxpayer is required to lodge a CGT schedule and CGT exemption or rollover applied code is ""Demerger exemption (Subdivision 125-C)"", then Capital gains disregarded by a demerging entity amount on attached CGT schedule must be completed",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:CapitalGains/tns:TaxExemptionOrRolloverAppliedCCollection/tns:TaxExemptionOrRolloverAppliedC",
                    Parameters = new ProcessMessageParameters() {
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.433049" },
                        new ProcessMessageParameter() { Name = "TRT448", Value = "H - Demerger exemption" },
                        new ProcessMessageParameter() { Name = "CGTS47", Value = "null" }
                    },
                    DocumentSequence = "parent_doc"
                });
            }
        }
        #endregion // VR.ATO.TRT.433049

        #endregion Validation Rules
    }
}