﻿using Ato.CD.Inbound.Shared;
using Ato.EN.IntegrationServices.CodeGenerationIDS;
using Ato.EN.IntegrationServices.CodeGenerationTRT;
using DataContracts;
using System.Collections.Generic;
using System.Runtime.CompilerServices;

namespace Ato.CD.Inbound.TRT
{
    public class CrossFormValidatorIDS : ICrossFormValidator
    {
        private IDS2025 childDocument { get; set; }

        public List<ProcessMessageDocument> response { get; private set; }

        private TRT2026 parentDocument { get; set; }

        public CrossFormValidatorIDS(TRT2026 parent, object child)
        {
            childDocument = (IDS2025)child;
            parentDocument = parent;
            response = new List<ProcessMessageDocument>();
        }

        public List<ProcessMessageDocument> ValidateCrossFormRules()
        {           
            VRATOIDS440670();
            VRATOTRT432646();
            VRATOTRT432647();
            VRATOTRT432649();
            VRATOTRT432650();
            VRATOTRT432651();
            VRATOTRT432652();
            VRATOTRT432653();
            VRATOTRT432655();
            VRATOTRT432656();
            VRATOTRT432707();
            VRATOTRT432720();
            VRATOTRT432797();

            return response;
        }

        #region Validation Rules

        #region VR.ATO.IDS.440670

        /*  VR.ATO.IDS.440670
        Your supplied TFN does not match the TFN supplied on the form it was submitted with

        Legacy Rule Format:
        ^IDS500 <> ^ParentTFN

        Technical Business Rule Format:
        ^IDS500 <> ^ParentTFN

        Data Elements:
        ^TRT5 = TRT:RP:Identifiers.TaxFileNumber.Identifier

        ^IDS500 = IDS:RP:Identifiers.TaxFileNumber.Identifier
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOIDS440670()
        {
            bool assertion = !childDocument.IDS500.Equals(parentDocument.TRT5);

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.402009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your supplied TFN does not match the TFN supplied on the form it was submitted with",
                    LongDescription = @"Your supplied TFN does not match the TFN supplied on the parent form it was submitted with",
                    Location = "/tns:IDS/tns:RP/tns:TaxFileNumberId",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.IDS.440670" },
                        new ProcessMessageParameter() { Name = "TRT5", Value = parentDocument.TRT5.ToString() },
                        new ProcessMessageParameter() { Name = "IDS500", Value = childDocument.IDS500.ToString() },
                    }
                });
            }
        }
        #endregion // VR.ATO.IDS.440670

        #region VR.ATO.TRT.432646

        /*  VR.ATO.TRT.432646
        The sum of the amounts showing at attributable income of controlled foreign companies and controlled foreign trusts on IDS should be equal to or less than the sum of the amounts at Listed country and Unlisted country on the main return form.

        Legacy Rule Format:
        CountDocument('IDS') = 1 AND (^IDS191 + ^IDS192 + ^IDS193) > (^TRT134 + ^TRT136)

        Technical Business Rule Format:
        CountDocument('IDS') = 1 AND (^IDS191 + ^IDS192 + ^IDS193) > (^TRT134 + ^TRT136)

        Data Elements:
    
        ^TRT134 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:ListedCountry:Income.InternationalDealings.Attributed.Amount

        ^TRT136 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:UnlistedCountry:Income.InternationalDealings.Attributed.Amount

        ^IDS191 = IDS:RP:Income.ControlledForeignCompaniesAttributableIncomeTotal.Amount
    
        ^IDS192 = IDS:RP:Income.ControlledForeignCompaniesChangeOfResidenceIncome.Amount
    
        ^IDS193 = IDS:RP:Income.ControlledForeignCompaniesAndControlledForeignTrustsInterposedAustralianEntitiesAttributableIncomeTotal.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRT432646()
        {
            decimal totalIDS = childDocument.IDS191.GetValueOrDefault() + childDocument.IDS192.GetValueOrDefault() + childDocument.IDS193.GetValueOrDefault();
            decimal totalTRT = parentDocument.TRT134.GetValueOrDefault() + parentDocument.TRT136.GetValueOrDefault();

            bool assertion = totalIDS > totalTRT;

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.TRT.432714",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Attributed foreign income incorrect",
                    LongDescription = @"The sum of amounts at attributable income of controlled foreign companies and controlled foreign trusts on the IDS should be equal to or less than the sum of amounts at Listed country and Unlisted country on the trust return form.",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:AttributedForeignIncome/tns:UnlistedCountryInternationalDealingsA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.432646" },
                        new ProcessMessageParameter() { Name = "TRT134", Value = parentDocument.TRT134.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "TRT136", Value = parentDocument.TRT136.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "Total TRT fields", Value = totalTRT.ToString() },
                        new ProcessMessageParameter() { Name = "IDS191", Value = childDocument.IDS191.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "IDS192", Value = childDocument.IDS192.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "IDS193", Value = childDocument.IDS193.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "Total IDS fields", Value = totalIDS.ToString() },
                    },
                    DocumentSequence = "parent_doc"
                });
            }
        }
        #endregion // VR.ATO.TRT.432646

        #region VR.ATO.TRT.432647

        /*  VR.ATO.TRT.432647
        The Section 456 Controlled Foreign Companies attributable income (Listed countries) amount on the International Dealings Schedule must not exceed the Attributed foreign income (Listed country) amount on the parent Trust return

        Legacy Rule Format:
        CountDocument('IDS') = 1 AND ^TRT134 > 0 AND ^IDS188 > ^TRT134

        Technical Business Rule Format:
        CountDocument('IDS') = 1 AND ^TRT134 > 0 AND ^IDS188 > ^TRT134

        Data Elements:
    
        ^TRT134 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:ListedCountry:Income.InternationalDealings.Attributed.Amount
    
        ^IDS188 = IDS:RP:ListedCFCAmounts:Income.ControlledForeignCompaniesAttributableIncome.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRT432647()
        {
            bool assertion = parentDocument.TRT134.GetValueOrDefault() > 0 && childDocument.IDS188.GetValueOrDefault() > parentDocument.TRT134.GetValueOrDefault();

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.TRT.442647",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Attributed foreign income incorrect",
                    LongDescription = "The Section 456 Controlled Foreign Companies attributable income (Listed countries) amount on the International Dealings Schedule must not exceed the Attributed foreign income (Listed country) amount on the parent Trust return",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:AttributedForeignIncome/tns:ListedCountryInternationalDealingsA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.432647" },
                        new ProcessMessageParameter() { Name = "TRT134", Value = parentDocument.TRT134.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "IDS188", Value = childDocument.IDS188.GetValueOrDefault().ToString() },
                    },
                    DocumentSequence = "parent_doc"
                });
            }
        }
        #endregion // VR.ATO.TRT.432647

        #region VR.ATO.TRT.432649

        /*  VR.ATO.TRT.432649
        If there is a yes response to question 4 label A on the IDS, there must also be a yes response to question 29 label C (transactions with specified countries) on the trust or partnership return form.

        Legacy Rule Format:
        ^TRT151 = FALSE AND CountDocument('IDS') = 1 AND ^IDS9 = TRUE

        Technical Business Rule Format:
        ^TRT151 = FALSE AND CountDocument('IDS') = 1 AND ^IDS9 = TRUE

        Data Elements:
    
        ^TRT151 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:Income:ThinCapitalisation.InternationalDealingsTransactionsWithTaxHavenCountriesI
    
        ^IDS9 = IDS:RP:InternationalDealings.SpecifiedCountriesTransactions.Indicator
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRT432649()
        {
            bool assertion = parentDocument.TRT151.GetValueOrDefault() == false && childDocument.IDS9.GetValueOrDefault() == true;

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.TRT.432649",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Transactions with specified countries question incorrect",
                    LongDescription = "If there is a 'True' response at 'Did you have dealings with international related parties located in specified countries?' on the IDS then the response at Overseas transactions - Transactions with specified countries on the trust return must be 'True'.",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:ThinCapitalisation/tns:InternationalDealingsTransactionsWithTaxHavenCountriesI",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.432649" },
                        new ProcessMessageParameter() { Name = "TRT151", Value = parentDocument.TRT151.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "IDS9", Value = childDocument.IDS9.GetValueOrDefault().ToString() },
                    },
                    DocumentSequence = "parent_doc"
                });
            }
        }
        #endregion // VR.ATO.TRT.432649

        #region VR.ATO.TRT.432650

        /*  VR.ATO.TRT.432650
        If Debt deductions in earning non-assessable foreign income claimed under s25-90 has been completed on the IDS then Total interest expenses or Other deductions or Interest expenses overseas must be greater than zero on trust return form.

        Legacy Rule Format:
        CountDocument('IDS') = 1 AND ^IDS207 > 0 AND (^TRT470 = NULL OR ^TRT470 = 0) AND (^TRT182 = NULL OR ^TRT182 = 0) AND (^TRT129 = NULL OR ^TRT129 = 0)

        Technical Business Rule Format:
        CountDocument('IDS') = 1 AND ^IDS207 > 0 AND (^TRT470 = NULL OR ^TRT470 = 0) AND (^TRT182 = NULL OR ^TRT182 = 0) AND (^TRT129 = NULL OR ^TRT129 = 0)

        Data Elements:
    
        ^TRT129 = TRT:RP:BusinessDetails:Income:Deductions:OtherDeductions:Expense.DeductibleOther.Amount
    
        ^TRT182 = TRT:RP:BusinessDetails:Income:ThinCapitalisation:Expense.Interest.Amount
        
        ^TRT470 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:BusinessExpenses:Expense.Operating.Interest.Amount
        
        ^IDS207 = IDS:RP:IncomeTax.Deduction.DebtEarningNonAssessableNonExemptForeignIncomeTotal.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRT432650()
        {
            bool assertion = childDocument.IDS207.GetValueOrDefault() > 0 && (parentDocument.TRT470 == null || parentDocument.TRT470 == 0) && (parentDocument.TRT182 == null || parentDocument.TRT182 == 0) && (parentDocument.RP_BusinessDetails_Income_Deductions_OtherDeductionsCollection != null && parentDocument.RP_BusinessDetails_Income_Deductions_OtherDeductionsCollection.Exists(tuple => tuple.TRT129 == null || tuple.TRT129 == 0));

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.TRT.432650",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Interest expenses required",
                    LongDescription = @"If Debt deductions in earning non-assessable foreign income claimed under s25-90 has been completed on the IDS then Total interest expenses or Other deductions or Interest expenses overseas must be greater than zero on trust return form.",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:BusinessIncomeAndExpenses/tns:BusinessExpenses/tns:OperatingInterestA",
                    Parameters = new ProcessMessageParameters() {
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.432650" },
                        new ProcessMessageParameter() { Name = "TRT129", Value = parentDocument.RP_BusinessDetails_Income_Deductions_OtherDeductionsCollection.Find(tuple => tuple.TRT129 == null || tuple.TRT129 == 0).TRT129.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "TRT182", Value = parentDocument.TRT182.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "TRT470", Value = parentDocument.TRT470.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "IDS207", Value = childDocument.IDS207.GetValueOrDefault().ToString() }
                    },
                    DocumentSequence = "parent_doc"
                });
            }
        }
        #endregion // VR.ATO.TRT.432650

        #region VR.ATO.TRT.432651

        /*  VR.ATO.TRT.432651
        Where there is a positive amount present at 'Total amount of your share of FHLP/FHCs net income/profit' on the International Dealings Schedule, the 'Gross amount of Other assessable foreign source income, excluding income shown at Attributed foreign income' amount on the Trust Return must be provided

        Legacy Rule Format:
        CountDocument('IDS') = 1 AND ^TRT140 = NULL AND ^IDS219 > 0

        Technical Business Rule Format:
        CountDocument('IDS') = 1 AND ^TRT140 = NULL AND ^IDS219 > 0

        Data Elements:
    
        ^TRT140 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:Income.Other.Amount
    
        ^IDS219 = IDS:RP:Assets.Investment.ForeignHybridLimitedPartnershipOrForeignHybridCompanyNetIncomeProfitTotal.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRT432651()
        {
            bool assertion = parentDocument.TRT140 == null && childDocument.IDS219.GetValueOrDefault() > 0;

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.TRT.442651",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Gross other assessable foreign source income required",
                    LongDescription = @"Where there is a positive amount present at 'Total amount of your share of FHLP/FHCs net income/profit' on the International Dealings Schedule, the 'Gross amount of Other assessable foreign source income, excluding income shown at Attributed foreign income' amount on the Trust Return must be provided",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:AttributedForeignIncome/tns:OtherAssessableForeignIncome/tns:A",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.432651" },
                        new ProcessMessageParameter() { Name = "TRT140", Value = parentDocument.TRT140.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "IDS219", Value = childDocument.IDS219.GetValueOrDefault().ToString() },
                    },
                    DocumentSequence = "parent_doc"
                });
            }
        }
        #endregion // VR.ATO.TRT.432651

        #region VR.ATO.TRT.432652

        /*  VR.ATO.TRT.432652
        If there is a 'True' response to 'Did you have any capital gains tax (CGT) events in relation to your interest in a foreign company?' On the IDS then a 'True' response is required at 'Did you have a CGT event during the year?' on the trust return.

        Legacy Rule Format:
        ^TRT131 = FALSE AND ^IDS208 = TRUE

        Technical Business Rule Format:
        ^TRT131 = FALSE AND ^IDS208 = TRUE

        Data Elements:
    
        ^TRT131 = TRT:RP:BusinessDetails:Income:CapitalGains:CapitalGainsTax.Event.Indicator
    
        ^IDS208 = IDS:RP:CapitalGainsTax.Event.Indicator
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRT432652()
        {
            bool assertion = parentDocument.TRT131 == false && childDocument.IDS208 == true;

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.TRT.432652",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Check response to CGT event question",
                    LongDescription = @"If there is a 'True' response to 'Did you have any capital gains tax (CGT) events in relation to your interest in a foreign company?' On the IDS then a 'True' response is required at 'Did you have a CGT event during the year?' on the trust return.",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:CapitalGains/tns:TaxEventI",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.432652" },
                        new ProcessMessageParameter() { Name = "TRT131", Value = parentDocument.TRT131.ToString() },
                        new ProcessMessageParameter() { Name = "IDS208", Value = childDocument.IDS208.ToString() },
                    },
                    DocumentSequence = "parent_doc"
                });
            }
        }
        #endregion // VR.ATO.TRT.432652

        #region VR.ATO.TRT.432653

        /*  VR.ATO.TRT.432653
        If Capital gain amounts has been completed on the IDS then Net capital gain must be completed on the trust return.

        Legacy Rule Format:
        ^TRT133 = NULL AND ^IDS209 > 0

        Technical Business Rule Format:
        ^TRT133 = NULL AND ^IDS209 > 0

        Data Elements:
    
        ^TRT133 = TRT:RP:BusinessDetails:Income:CapitalGains:Income.CapitalGainsNet.Amount
    
        ^IDS209 = IDS:RP:CGTEvents:CapitalGainsTax.ActiveForeignBusinessCapitalGains.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRT432653()
        {
            bool assertion = parentDocument.TRT133 == null && childDocument.IDS209.GetValueOrDefault() > 0;

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.TRT.432653",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Net capital gain required",
                    LongDescription = @"If Capital gain amounts has been completed on the IDS then Net capital gain must be completed on the trust return.",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:CapitalGains/tns:NetA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.432653" },
                        new ProcessMessageParameter() { Name = "TRT133", Value = parentDocument.TRT133.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "IDS209", Value = childDocument.IDS209.GetValueOrDefault().ToString() },
                    },
                    DocumentSequence = "parent_doc"
                });
            }
        }
        #endregion // VR.ATO.TRT.432653

        #region VR.ATO.TRT.432655

        /*  VR.ATO.TRT.432655
        Where there is a positive Royalties deductions amount present on the International Dealings Schedule, a positive Royalty expenses overseas amount must be provided

        Legacy Rule Format:
        (^TRT183 = 0 OR ^TRT183 = NULL) AND CountDocument('IDS') = 1 AND ^IDS22 > 0

        Technical Business Rule Format:
        (^TRT183 = 0 OR ^TRT183 = NULL) AND CountDocument('IDS') = 1 AND ^IDS22 > 0

        Data Elements:
    
        ^TRT183 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:Income:ThinCapitalisation.ExpenseRoyaltiesA
    
        ^IDS22 = IDS:RP:Royalties:Expense.DeductionsTotal.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRT432655()
        {
            bool assertion = parentDocument.TRT183.GetValueOrDefault() == 0 && childDocument.IDS22.GetValueOrDefault() > 0;

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.TRT.442655",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Royalty expenses overseas required",
                    LongDescription = "Where there is a positive Royalties deductions amount present on the International Dealings Schedule, a positive Royalty expenses overseas amount must be provided",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:ThinCapitalisation/tns:ExpenseRoyaltiesA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.432655" },
                        new ProcessMessageParameter() { Name = "TRT183", Value = parentDocument.TRT183.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "IDS22", Value = childDocument.IDS22.GetValueOrDefault().ToString() },
                    },
                    DocumentSequence = "parent_doc"
                });
            }
        }
        #endregion // VR.ATO.TRT.432655

        #region VR.ATO.TRT.432656

        /*  VR.ATO.TRT.432656
        If Royalties deductions has been completed on the IDS then it must be less than or equal to Royalty expenses overseas on the trust return form

        Legacy Rule Format:
        ^TRT183 > 0 AND CountDocument('IDS') = 1 AND ^TRT183 < ^IDS22

        Technical Business Rule Format:
        ^TRT183 > 0 AND CountDocument('IDS') = 1 AND ^TRT183 < ^IDS22

        Data Elements:
    
        ^TRT183 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:Income:ThinCapitalisation.ExpenseRoyaltiesA
    
        ^IDS22 = IDS:RP:Royalties:Expense.DeductionsTotal.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRT432656()
        {
            bool assertion = parentDocument.TRT183.GetValueOrDefault() > 0 && parentDocument.TRT183.GetValueOrDefault() < childDocument.IDS22.GetValueOrDefault();

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.TRT.432656",
                    Severity = ProcessMessageSeverity.Error,
                    Description = "Royalties deductions incorrect",
                    LongDescription = "If Royalties deductions has been completed on the IDS then it must be less than or equal to Royalty expenses overseas on the trust return form",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:ThinCapitalisation/tns:ExpenseRoyaltiesA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.432656" },
                        new ProcessMessageParameter() { Name = "TRT183", Value = parentDocument.TRT183.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "IDS22", Value = childDocument.IDS22.GetValueOrDefault().ToString() },
                    },
                    DocumentSequence = "parent_doc"
                });
            }
        }
        #endregion // VR.ATO.TRT.432656

        #region VR.ATO.TRT.432707

        /*  VR.ATO.TRT.432707
        If there is a 'NO' response to the questions concerning whether the entity had either a direct or indirect interest in a foreign trust, controlled foreign company or transferor trust and whether the value of dealings with international related parties is greater than $2 million and whether the thin capitalisation provisions apply, and no amounts have been completed at Interest expenses overseas and Royalty expenses overseas, an International Dealings schedule must not be attached.

        Legacy Rule Format:
        ^TRT138 = FALSE AND ^TRT148 = FALSE AND ^TRT449 = FALSE AND (^TRT182 = NULL OR ^TRT182 = 0) AND (^TRT183 = NULL OR ^TRT183 = 0) AND CountDocument('IDS') = 1

        Technical Business Rule Format:
        ^TRT138 = FALSE AND ^TRT148 = FALSE AND ^TRT449 = FALSE AND (^TRT182 = NULL OR ^TRT182 = 0) AND (^TRT183 = NULL OR ^TRT183 = 0) AND CountDocument('IDS') = 1

        Data Elements:
    
        ^TRT138 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:Income.InternationalDealings.AttributedForeignIncomeForeignTrustorControlledForeignCompanyorTransferorTrust.Indicator
    
        ^TRT148 = TRT:RP:BusinessDetails:Income:InternationalDealings.RelatedPartiesTransactionsExcessAggregateValue.Indicator
    
        ^TRT182 = TRT:RP:BusinessDetails:Income:ThinCapitalisation:Expense.Interest.Amount
    
        ^TRT183 = TRT:RP:BusinessDetails:Income:ThinCapitalisation:Expense.Royalties.Amount
    
        ^TRT449 = TRT:RP:BusinessDetails:Income:ThinCapitalisation:Liabilities.ThinCapitalisation.ProvisionsApplied.Indicator
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRT432707()
        {
            bool assertion = parentDocument.TRT138 == false && parentDocument.TRT148 == false && parentDocument.TRT449 == false && (parentDocument.TRT182 == null || parentDocument.TRT182 == 0) && (parentDocument.TRT183 == null || parentDocument.TRT183 == 0);

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.TRT.442901",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"International Dealings Schedule is not required",
                    LongDescription = @"If there is a 'NO' response to the questions concerning whether the entity had either a direct or indirect interest in a foreign trust, controlled foreign company or transferor trust and whether the value of dealings with international related parties is greater than $2 million and whether the thin capitalisation or debt deduction creation rules apply, and no amounts have been completed at Interest expenses overseas and Royalty expenses overseas, an International Dealings schedule must not be attached.",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:AttributedForeignIncome/tns:InternationalDealingsTrustorControlledCompanyorTransferorTrustI",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.432707" },
                        new ProcessMessageParameter() { Name = "TRT138", Value = TRT2026Validator.GetValueOrEmpty(parentDocument.TRT138) },
                        new ProcessMessageParameter() { Name = "TRT148", Value = TRT2026Validator.GetValueOrEmpty(parentDocument.TRT148) },
                        new ProcessMessageParameter() { Name = "TRT182", Value = parentDocument.TRT182.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "TRT183", Value = parentDocument.TRT183.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "TRT449", Value = TRT2026Validator.GetValueOrEmpty(parentDocument.TRT449) },
                    },
                    DocumentSequence = "parent_doc"
                });
            }
        }
        #endregion // VR.ATO.TRT.432707

        #region VR.ATO.TRT.432720

        /*  VR.ATO.TRT.432720
        If there is an amount at either 'Total TOFA gains' or 'Total TOFA Losses' that indicates you made a gain or loss from a financial arrangement subject to the TOFA rules, then there must be a 'True' response to 'Are you subject to the Taxation of Financial Arrangements rules contained in Division 230 of the ITAA 1997?' on the IDS

        Legacy Rule Format:
        (^TRT356 > 0 OR ^TRT357 > 0) AND CountDocument('IDS') = 1 AND ^IDS177 <> TRUE

        Technical Business Rule Format:
        (^TRT356 > 0 OR ^TRT357 > 0) AND CountDocument('IDS') = 1 AND ^IDS177 <> TRUE

        Data Elements:

        ^TRT356 = TRT:RP:BusinessDetails:Income:TaxationOfFinancialArrangements.GainsTotalA

        ^TRT357 = TRT:RP:BusinessDetails:Income:TaxationOfFinancialArrangements.ExpenseLossesTotalA
    
        ^IDS177 = IDS:RP:InternationalDealings.TaxationOfFinancialArrangementsRulesApplied.Indicator
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRT432720()
        {
            bool assertion = (parentDocument.TRT356 > 0 || parentDocument.TRT357 > 0) && childDocument.IDS177.GetValueOrDefault() != true;

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.TRT.432762",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Response to TOFA gains or losses is required",
                    LongDescription = @"If there is an amount at either 'Total TOFA gains' or 'Total TOFA Losses' that indicates you made a gain or loss from a financial arrangement subject to the TOFA rules, then there must be a 'True' response to 'Are you subject to the Taxation of Financial Arrangements rules contained in Division 230 of the ITAA 1997?' on the IDS.",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:TaxationOfFinancialArrangements/tns:GainsTotalA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.432720" },
                        new ProcessMessageParameter() { Name = "TRT356", Value = parentDocument.TRT356.ToString() },
                        new ProcessMessageParameter() { Name = "TRT357", Value = parentDocument.TRT357.ToString() },
                        new ProcessMessageParameter() { Name = "IDS177", Value = childDocument.IDS177.ToString() },
                    },
                    DocumentSequence = "parent_doc"
                });
            }
        }
        #endregion // VR.ATO.TRT.432720

        #region VR.ATO.TRT.432797

        /*  VR.ATO.TRT.432797
        If the entity has self-assessed as a Significant Global Entity (SGE), then 'Are you a small business entity, not a significant global entity and your international related party dealings do not exceed $5 million and 50% of your current year aggregated turnover?' must not be 'TRUE' on the International Dealings Schedule (IDS) form.

        Legacy Rule Format:
        ^TRT487 = TRUE AND CountDocument('IDS') = 1 AND ^IDS425 = TRUE

        Technical Business Rule Format:
        ^TRT487 = TRUE AND CountDocument('IDS') = 1 AND ^IDS425 = TRUE

        Data Elements:
    
        ^TRT487 = TRT:RP:BusinessDetails:OrganisationDetails.SignificantGlobalEntityStatus.Indicator
    
        ^IDS425 = IDS:RP:InternationalDealings.SmallBusinessConcessionsEligibility.Indicator
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRT432797()
        {
            bool assertion = (parentDocument.TRT487 == true && childDocument.IDS425.GetValueOrDefault() == true);

            if (assertion)
            {
                response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.TRT.442797",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Significant Global Entity question on the International Dealings Schedule (IDS) form must not be 'TRUE'.",
                    LongDescription = @"If the entity has self-assessed as a Significant Global Entity (SGE), then 'Are you a small business entity, not a significant global entity and your international related party dealings do not exceed $5 million and 50% of your current year aggregated turnover?' must not be 'TRUE' on the International Dealings Schedule (IDS) form.",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:OrganisationDetailsSignificantGlobalEntityStatusI",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.432797" },
                        new ProcessMessageParameter() { Name = "TRT487", Value = parentDocument.TRT487.ToString() },
                        new ProcessMessageParameter() { Name = "IDS425", Value = childDocument.IDS425.ToString() },
                    },
                    DocumentSequence = "parent_doc"
                });
            }
        }

        #endregion // VR.ATO.TRT.432797

        #endregion Validation Rules
    }
}