﻿using DataContracts;
using System.Collections.Generic;
using System.Linq;

namespace Ato.EN.IntegrationServices.CodeGenerationTRT
{
    public partial class TRT2026Validator
    {
        #region Manual Rules - Rules marked for manual coding

        #region VR.ATO.TRT.432089

        /*  VR.ATO.TRT.432089
            The sum of 'Credit for tax withheld where ABN not quoted' recorded in the Statements of distribution must equal the sum of 'Credit for tax withheld where ABN not quoted' plus 'Share of credit for tax withheld where ABN not quoted'

            Legacy Rule Format:
            (Sum(^TRT220) + ^TRT236) <> (^TRT73 + ^TRT84)

            Technical Business Rule Format:
            (Sum(^TRT220) + ^TRT236) <> (^TRT73 + ^TRT84)

            Data Elements:
    
            ^TRT73 = TRT:RP:BusinessDetails:Income:PayAsYouGoWithholding:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount
    
            ^TRT84 = TRT:RP:BusinessDetails:Income:ShareOfCredits:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuotedShareReceivedIndirectly.Amount
    
            ^TRT220 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount
    
            ^TRT236 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldWhereABNNotQuoted.Amount
            */
        public virtual void VRATOTRT432089()
        {
            decimal? sumTRT220 = report.RP_StatementOfDistributionToBeneficiaryCollection == null ? 0 : report.RP_StatementOfDistributionToBeneficiaryCollection.Sum(f => f.TRT220.GetValueOrDefault());

            bool assertion = sumTRT220 + report.TRT236.GetValueOrDefault() != report.TRT73.GetValueOrDefault() + report.TRT84.GetValueOrDefault();

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432089",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total Tax withheld where ABN not quoted must equal beneficiaries amounts",
                    LongDescription = @"The sum of 'Credit for tax withheld where ABN not quoted' recorded in the Statements of distribution must equal the sum of 'Credit for tax withheld where ABN not quoted' plus 'Share of credit for tax withheld where ABN not quoted'",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:PayAsYouGoWithholding/tns:TaxCreditForTaxWithheldWhereABNNotQuotedA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.432089" } },
                };
                response.Add(processMessage);

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Sum of TRT220", Value = GetValueOrEmpty(sumTRT220) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT236", Value = GetValueOrEmpty(report.TRT236) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT73", Value = GetValueOrEmpty(report.TRT73) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT84", Value = GetValueOrEmpty(report.TRT84) });
            }
        }
        #endregion // VR.ATO.TRT.432089

        #region VR.ATO.TRT.432093

        /*  VR.ATO.TRT.432093
            The sum of 'Credit for tax withheld - foreign resident withholding (excluding capital gains)' recorded in the Statements of distribution must equal the sum of 'Credit for tax withheld - foreign resident withholding (excluding capital gains)' plus 'Share of credit for tax withheld - foreign resident withholding (excluding capital gains)'

            Legacy Rule Format:
            (Sum(^TRT476) + ^TRT477) <> (^TRT474 + ^TRT87)

            Technical Business Rule Format:
            (Sum(^TRT476) + ^TRT477) <> (^TRT474 + ^TRT87)

            Data Elements:
    
            ^TRT474 = TRT:RP:BusinessDetails:Income:PayAsYouGoWithholding:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromForeignResidents.Amount
    
            ^TRT87 = TRT:RP:BusinessDetails:Income:ShareOfCredits:InternationalDealings.CreditForTaxWithheldFromForeignResidentWithholdingShareReceivedIndirectly.Amount
    
            ^TRT476 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromForeignResidents.Amount
    
            ^TRT477 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromForeignResidents.Amount
            */
        public virtual void VRATOTRT432093()
        {
            decimal? sumTRT476 = report.RP_StatementOfDistributionToBeneficiaryCollection == null ? 0 : report.RP_StatementOfDistributionToBeneficiaryCollection.Sum(f => f.TRT476.GetValueOrDefault());

            bool assertion = sumTRT476 + report.TRT477.GetValueOrDefault() != report.TRT474.GetValueOrDefault() + report.TRT87.GetValueOrDefault();

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432746",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total Credit for Tax withheld - foreign resident withholding (excluding capital gains) must equal beneficiaries amounts",
                    LongDescription = @"The sum of 'Credit for tax withheld - foreign resident withholding (excluding capital gains)' recorded in the Statements of distribution must equal the sum of 'Credit for tax withheld - foreign resident withholding (excluding capital gains)' plus 'Share of credit for tax withheld - foreign resident withholding (excluding capital gains)'.",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:PayAsYouGoWithholding/tns:TaxCreditForAmountsWithheldFromForeignResidentsA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.432093" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Sum of TRT476", Value = GetValueOrEmpty(sumTRT476) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT477", Value = GetValueOrEmpty(report.TRT477) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT474", Value = GetValueOrEmpty(report.TRT474) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT87", Value = GetValueOrEmpty(report.TRT87) });

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.TRT.432093

        #region VR.ATO.TRT.432110

        /*  VR.ATO.TRT.432110
            If Total net income is greater than zero, then the sum of 'Franking credit' recorded in the Statements of distribution must equal the sum of 'Share of franking credit from franked dividends' and 'Dividends:Franking credit'

            Legacy Rule Format:
            (^TRT130 + ^TRT133 + ^TRT134 + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144) > 0 AND OutsideRange(Sum(^TRT221) + ^TRT237, ^TRT85 + ^TRT98, 1)

            Technical Business Rule Format:
            (^TRT130 + ^TRT133 + ^TRT134 + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144) > 0 AND OutsideRange(Sum(^TRT221) + ^TRT237, ^TRT85 + ^TRT98, 1)

            Data Elements:
    
            ^TRT85 = TRT:RP:BusinessDetails:Income:ShareOfCredits:IncomeTax.FrankingCredits.FrankingCreditShareReceivedIndirectly.Amount
    
            ^TRT98 = TRT:RP:BusinessDetails:Income:Dividends:IncomeTax.FrankingCredits.Amount
    
            ^TRT130 = TRT:RP:BusinessDetails:Income:Income.Net.Amount
    
            ^TRT133 = TRT:RP:BusinessDetails:Income:CapitalGains:Income.CapitalGainsNet.Amount
    
            ^TRT134 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:ListedCountry:Income.InternationalDealings.Attributed.Amount
    
            ^TRT136 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:UnlistedCountry:Income.InternationalDealings.Attributed.Amount
    
            ^TRT141 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:Income.Net.Amount
    
            ^TRT143 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
    
            ^TRT144 = TRT:RP:BusinessDetails:Income:TaxLossesDeducted:IncomeTax.Deduction.TaxLossesDeducted.Amount
    
            ^TRT221 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.FrankingCredits.ReceivedFromAustralianCompanies.Amount
    
            ^TRT237 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.FrankingCredits.ReceivedFromAustralianCompanies.Amount
            */
        public virtual void VRATOTRT432110()
        {
            decimal? sumTRT221 = report.RP_StatementOfDistributionToBeneficiaryCollection == null ? 0 : report.RP_StatementOfDistributionToBeneficiaryCollection.Sum(f => f.TRT221.GetValueOrDefault());

            bool assertion = report.TRT130.GetValueOrDefault() + report.TRT133.GetValueOrDefault() + report.TRT134.GetValueOrDefault() + report.TRT136.GetValueOrDefault() + report.TRT141.GetValueOrDefault() + report.TRT143.GetValueOrDefault() - report.TRT144.GetValueOrDefault() > 0 && OutsideRange(sumTRT221.GetValueOrDefault() + report.TRT237.GetValueOrDefault(), report.TRT85.GetValueOrDefault() + report.TRT98.GetValueOrDefault(), 1);

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432110",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Sum of Franking credit amounts must equal beneficiaries amounts",
                    LongDescription = @"If Total net income is greater than zero, then the sum of 'Franking credit' recorded in the Statements of distribution must equal the sum of 'Share of franking credit from franked dividends' and 'Dividends:Franking credit'",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:ShareOfCredits/tns:TaxFrankingCreditsFrankingCreditShareReceivedIndirectlyA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.432110" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT130", Value = GetValueOrEmpty(report.TRT130) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT133", Value = GetValueOrEmpty(report.TRT133) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT134", Value = GetValueOrEmpty(report.TRT134) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT136", Value = GetValueOrEmpty(report.TRT136) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT141", Value = GetValueOrEmpty(report.TRT141) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT143", Value = GetValueOrEmpty(report.TRT143) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT144", Value = GetValueOrEmpty(report.TRT144) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT221", Value = GetValueOrEmpty(sumTRT221) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT237", Value = GetValueOrEmpty(report.TRT237) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT85", Value = GetValueOrEmpty(report.TRT85) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT98", Value = GetValueOrEmpty(report.TRT98) });

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.TRT.432110

        #region VR.ATO.TRT.432112

        /*  VR.ATO.TRT.432112
            The sum of 'TFN amounts withheld' recorded in the Statements of distribution must equal the sum of 'Share of credit for TFN amounts withheld from interest...' plus 'TFN amounts withheld from gross interest' plus 'TFN amounts withheld from dividends'

            Legacy Rule Format:
            (Sum(^TRT222) + ^TRT238) <> (^TRT86 + ^TRT95 + ^TRT99)

            Technical Business Rule Format:
            (Sum(^TRT222) + ^TRT238) <> (^TRT86 + ^TRT95 + ^TRT99)

            Data Elements:
    
            ^TRT86 = TRT:RP:BusinessDetails:Income:ShareOfCredits:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldFromInterestAndDividendsWhereTFNNotQuotedShareReceivedIndirectly.Amount
    
            ^TRT95 = TRT:RP:BusinessDetails:Income:GrossInterest:IncomeTax.TFNAmountsWithheldFromGrossInterest.Amount
    
            ^TRT99 = TRT:RP:BusinessDetails:Income:Dividends:IncomeTax.DividendsTFNAmountsWithheld.Amount
    
            ^TRT222 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldWhereTFNNotQuoted.Amount
    
            ^TRT238 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.PayAsYouGoWithholding.CreditForAmountsWithheldWhereTFNNotQuoted.Amount
            */
        public virtual void VRATOTRT432112()
        {
            decimal? sumTRT222 = report.RP_StatementOfDistributionToBeneficiaryCollection == null ? 0 : report.RP_StatementOfDistributionToBeneficiaryCollection.Sum(f => f.TRT222.GetValueOrDefault());

            bool assertion = sumTRT222 + report.TRT238.GetValueOrDefault() != report.TRT86.GetValueOrDefault() + report.TRT95.GetValueOrDefault() + report.TRT99.GetValueOrDefault();

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432112",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"TFN amounts withheld from interest must equal beneficiaries amount",
                    LongDescription = @"The sum of 'TFN amounts withheld' recorded in the Statements of distribution must equal the sum of 'Share of credit for TFN amounts withheld from interest...' plus 'TFN amounts withheld from gross interest' plus 'TFN amounts withheld from dividends'",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:ShareOfCredits/tns:TaxPayAsYouGoWithholdingCreditForAmountsWithheldFromInterestAndDividendsWhereTFNNotQuotedShareReceivedIndirectlyA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.432112" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Sum of TRT222", Value = GetValueOrEmpty(sumTRT222) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT238", Value = GetValueOrEmpty(report.TRT238) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT86", Value = GetValueOrEmpty(report.TRT86) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT95", Value = GetValueOrEmpty(report.TRT95) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT99", Value = GetValueOrEmpty(report.TRT99) });

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.TRT.432112

        #region VR.ATO.TRT.432179

        /*  VR.ATO.TRT.432179
            The overall net trust income is greater than zero (the sum of net Australian income/loss, net capital gains, attributed foreign income other assessable foreign source income minus allowable tax losses of earlier income years deducted this year) AND the sum of the income amounts in the distribution statement is not equal to the overall net trust income.

            Legacy Rule Format:
            (^TRT130 + ^TRT133 + ^TRT134 + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144) > 0 AND (^TRT130 + ^TRT133 + ^TRT134 + ^TRT136 + ^TRT141 - ^TRT144) <> (Sum(^TRT218) + ^TRT234 + Sum(^TRT219) + ^TRT235 + Sum(^TRT442) + ^TRT444 + Sum(^TRT223) + ^TRT239 + ^TRT446 + Sum(^TRT224) + ^TRT240 + Sum(^TRT225) + ^TRT241)

            Technical Business Rule Format:
            (^TRT130 + ^TRT133 + ^TRT134 + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144) > 0 AND (^TRT130 + ^TRT133 + ^TRT134 + ^TRT136 + ^TRT141 - ^TRT144) <> (Sum(^TRT218) + ^TRT234 + Sum(^TRT219) + ^TRT235 + Sum(^TRT442) + ^TRT444 + Sum(^TRT223) + ^TRT239 + ^TRT446 + Sum(^TRT224) + ^TRT240 + Sum(^TRT225) + ^TRT241)

            Data Elements:
    
            ^TRT130 = TRT:RP:BusinessDetails:Income:Income.Net.Amount
    
            ^TRT133 = TRT:RP:BusinessDetails:Income:CapitalGains:Income.CapitalGainsNet.Amount
    
            ^TRT134 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:ListedCountry:Income.InternationalDealings.Attributed.Amount
    
            ^TRT136 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:UnlistedCountry:Income.InternationalDealings.Attributed.Amount
    
            ^TRT141 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:Income.Net.Amount
    
            ^TRT143 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
    
            ^TRT144 = TRT:RP:BusinessDetails:Income:TaxLossesDeducted:IncomeTax.Deduction.TaxLossesDeducted.Amount
    
            ^TRT218 = TRT:RP:StatementOfDistributionToBeneficiary:Information:ShareOfIncomePrimaryProduction:Income.BeneficiaryShare.Amount
    
            ^TRT219 = TRT:RP:StatementOfDistributionToBeneficiary:Information:ShareOfIncomeNonPrimaryProduction:Income.BeneficiaryShare.Amount
    
            ^TRT223 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Income.CapitalGainsNet.Amount
    
            ^TRT224 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.AttributedForeignIncome.Amount
    
            ^TRT225 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Income.InternationalDealings.Net.Amount
    
            ^TRT234 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:PrimaryProductionNoBeneficiaryEntitled:IncomeTax.IncomeOrLossTaxable.Amount
    
            ^TRT235 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:NonPrimaryProductionNoBeneficiaryEntitled:IncomeTax.IncomeOrLossTaxable.Amount
    
            ^TRT239 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:Income.CapitalGainsNet.Amount
    
            ^TRT240 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.AttributedForeignIncome.Amount
    
            ^TRT241 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:Income.InternationalDealings.Net.Amount
    
            ^TRT442 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Income.DistributionFranked.Amount
    
            ^TRT444 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:Income.DistributionFranked.Amount
    
            ^TRT446 = TRT:RP:CapitalGainsAssessed:CapitalGainsTax.TrusteeAssessedCapitalGain.Amount
            */
        public virtual void VRATOTRT432179()
        {
            decimal? sumTRT218 = report.RP_StatementOfDistributionToBeneficiaryCollection == null ? 0 : report.RP_StatementOfDistributionToBeneficiaryCollection.Sum(f => f.TRT218.GetValueOrDefault());
            decimal? sumTRT219 = report.RP_StatementOfDistributionToBeneficiaryCollection == null ? 0 : report.RP_StatementOfDistributionToBeneficiaryCollection.Sum(f => f.TRT219.GetValueOrDefault());
            decimal? sumTRT442 = report.RP_StatementOfDistributionToBeneficiaryCollection == null ? 0 : report.RP_StatementOfDistributionToBeneficiaryCollection.Sum(f => f.TRT442.GetValueOrDefault());
            decimal? sumTRT223 = report.RP_StatementOfDistributionToBeneficiaryCollection == null ? 0 : report.RP_StatementOfDistributionToBeneficiaryCollection.Sum(f => f.TRT223.GetValueOrDefault());
            decimal? sumTRT224 = report.RP_StatementOfDistributionToBeneficiaryCollection == null ? 0 : report.RP_StatementOfDistributionToBeneficiaryCollection.Sum(f => f.TRT224.GetValueOrDefault());
            decimal? sumTRT225 = report.RP_StatementOfDistributionToBeneficiaryCollection == null ? 0 : report.RP_StatementOfDistributionToBeneficiaryCollection.Sum(f => f.TRT225.GetValueOrDefault());

            decimal? sumNetTrustIncome = report.TRT130.GetValueOrDefault() + report.TRT133.GetValueOrDefault() + report.TRT134.GetValueOrDefault() + report.TRT136.GetValueOrDefault() + report.TRT141.GetValueOrDefault() - report.TRT144.GetValueOrDefault();
            decimal? sumNetTrustIncomePlusTRT143 = sumNetTrustIncome + report.TRT143.GetValueOrDefault();
            decimal? sumDistributedIncome = sumTRT218 + report.TRT234.GetValueOrDefault() + sumTRT219 + report.TRT235.GetValueOrDefault() + sumTRT442 + report.TRT444.GetValueOrDefault() + sumTRT223 + report.TRT239.GetValueOrDefault() + report.TRT446.GetValueOrDefault() + sumTRT224 + report.TRT240.GetValueOrDefault() + sumTRT225 + report.TRT241.GetValueOrDefault();

            bool assertion = sumNetTrustIncomePlusTRT143 > 0 && sumNetTrustIncome != sumDistributedIncome;

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432179",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Net trust income must equal distributed income",
                    LongDescription = @"If the overall net trust income is greater than zero (the sum of net Australian income/loss, net capital gains, attributed foreign income, other assessable foreign source income; minus allowable tax losses of earlier income years deducted this year) and the sum of the income amounts in the statements of distribution must equal the overall net trust income.",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:NetA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.432179" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Sum of Net Trust Income", Value = GetValueOrEmpty(sumNetTrustIncome) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Sum of Distributed Income", Value = GetValueOrEmpty(sumDistributedIncome) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT130", Value = GetValueOrEmpty(report.TRT130) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT133", Value = GetValueOrEmpty(report.TRT133) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT134", Value = GetValueOrEmpty(report.TRT134) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT136", Value = GetValueOrEmpty(report.TRT136) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT141", Value = GetValueOrEmpty(report.TRT141) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT143", Value = GetValueOrEmpty(report.TRT143) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT144", Value = GetValueOrEmpty(report.TRT144) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT218", Value = GetValueOrEmpty(sumTRT218) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT234", Value = GetValueOrEmpty(report.TRT234) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT219", Value = GetValueOrEmpty(sumTRT219) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT235", Value = GetValueOrEmpty(report.TRT235) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT442", Value = GetValueOrEmpty(sumTRT442) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT444", Value = GetValueOrEmpty(report.TRT444) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT223", Value = GetValueOrEmpty(sumTRT223) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT239", Value = GetValueOrEmpty(report.TRT239) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT446", Value = GetValueOrEmpty(report.TRT446) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT224", Value = GetValueOrEmpty(sumTRT224) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT240", Value = GetValueOrEmpty(report.TRT240) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT225", Value = GetValueOrEmpty(sumTRT225) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT241", Value = GetValueOrEmpty(report.TRT241) });

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.TRT.432179

        #region VR.ATO.TRT.432181

        /*  VR.ATO.TRT.432181
            There is an overall trust loss (i.e. the sum of net Australian income/loss plus net capital gains, plus attributed foreign income plus other assessable foreign source income minus allowable tax losses of earlier income years deducted this year is less than zero) AND income distribution amount/s is/are present in the distribution statement (distribution to a beneficiary or no beneficiary presently entitled). Income cannot be distributed when there is an overall trust loss. Delete the income from the distribution statement.

            Legacy Rule Format:
            (^TRT130 + ^TRT133 + ^TRT134 + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144) <= 0 AND (Sum(^TRT218) + Sum(^TRT219) + Sum(^TRT442) + Sum(^TRT223) + Sum(^TRT224) + Sum(^TRT225) + Sum(^TRT217) + ^TRT233 + ^TRT234 + ^TRT235 + ^TRT444 + ^TRT239 + ^TRT240 + ^TRT241 + ^TRT446) <> 0

            Technical Business Rule Format:
            (^TRT130 + ^TRT133 + ^TRT134 + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144) <= 0 AND (Sum(^TRT218) + Sum(^TRT219) + Sum(^TRT442) + Sum(^TRT223) + Sum(^TRT224) + Sum(^TRT225) + Sum(^TRT217) + ^TRT233 + ^TRT234 + ^TRT235 + ^TRT444 + ^TRT239 + ^TRT240 + ^TRT241 + ^TRT446) <> 0

            Data Elements:
    
            ^TRT130 = TRT:RP:BusinessDetails:Income:Income.Net.Amount
    
            ^TRT133 = TRT:RP:BusinessDetails:Income:CapitalGains:Income.CapitalGainsNet.Amount
    
            ^TRT134 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:ListedCountry:Income.InternationalDealings.Attributed.Amount
    
            ^TRT136 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:UnlistedCountry:Income.InternationalDealings.Attributed.Amount
    
            ^TRT141 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:Income.Net.Amount
    
            ^TRT143 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
    
            ^TRT144 = TRT:RP:BusinessDetails:Income:TaxLossesDeducted:IncomeTax.Deduction.TaxLossesDeducted.Amount
    
            ^TRT217 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
    
            ^TRT218 = TRT:RP:StatementOfDistributionToBeneficiary:Information:ShareOfIncomePrimaryProduction:Income.BeneficiaryShare.Amount
    
            ^TRT219 = TRT:RP:StatementOfDistributionToBeneficiary:Information:ShareOfIncomeNonPrimaryProduction:Income.BeneficiaryShare.Amount
    
            ^TRT223 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Income.CapitalGainsNet.Amount
    
            ^TRT224 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.AttributedForeignIncome.Amount
    
            ^TRT225 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Income.InternationalDealings.Net.Amount
    
            ^TRT233 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
    
            ^TRT234 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:PrimaryProductionNoBeneficiaryEntitled:IncomeTax.IncomeOrLossTaxable.Amount
    
            ^TRT235 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:NonPrimaryProductionNoBeneficiaryEntitled:IncomeTax.IncomeOrLossTaxable.Amount
    
            ^TRT239 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:Income.CapitalGainsNet.Amount
    
            ^TRT240 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.AttributedForeignIncome.Amount
    
            ^TRT241 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:Income.InternationalDealings.Net.Amount
    
            ^TRT442 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Income.DistributionFranked.Amount
    
            ^TRT444 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:Income.DistributionFranked.Amount
    
            ^TRT446 = TRT:RP:CapitalGainsAssessed:CapitalGainsTax.TrusteeAssessedCapitalGain.Amount
            */
        public virtual void VRATOTRT432181()
        {
            decimal? sumTRT218 = report.RP_StatementOfDistributionToBeneficiaryCollection == null ? 0 : report.RP_StatementOfDistributionToBeneficiaryCollection.Sum(f => f.TRT218.GetValueOrDefault());
            decimal? sumTRT219 = report.RP_StatementOfDistributionToBeneficiaryCollection == null ? 0 : report.RP_StatementOfDistributionToBeneficiaryCollection.Sum(f => f.TRT219.GetValueOrDefault());
            decimal? sumTRT442 = report.RP_StatementOfDistributionToBeneficiaryCollection == null ? 0 : report.RP_StatementOfDistributionToBeneficiaryCollection.Sum(f => f.TRT442.GetValueOrDefault());
            decimal? sumTRT223 = report.RP_StatementOfDistributionToBeneficiaryCollection == null ? 0 : report.RP_StatementOfDistributionToBeneficiaryCollection.Sum(f => f.TRT223.GetValueOrDefault());
            decimal? sumTRT224 = report.RP_StatementOfDistributionToBeneficiaryCollection == null ? 0 : report.RP_StatementOfDistributionToBeneficiaryCollection.Sum(f => f.TRT224.GetValueOrDefault());
            decimal? sumTRT225 = report.RP_StatementOfDistributionToBeneficiaryCollection == null ? 0 : report.RP_StatementOfDistributionToBeneficiaryCollection.Sum(f => f.TRT225.GetValueOrDefault());
            decimal? sumTRT217 = report.RP_StatementOfDistributionToBeneficiaryCollection == null ? 0 : report.RP_StatementOfDistributionToBeneficiaryCollection.Sum(f => f.TRT217.GetValueOrDefault());

            bool assertion = (report.TRT130.GetValueOrDefault() + report.TRT133.GetValueOrDefault() + report.TRT134.GetValueOrDefault() + report.TRT136.GetValueOrDefault() + report.TRT141.GetValueOrDefault() + report.TRT143.GetValueOrDefault() - report.TRT144.GetValueOrDefault() <= 0 && sumTRT218 + sumTRT219 + sumTRT442 + sumTRT223 + sumTRT224 + sumTRT225 + sumTRT217 + report.TRT233.GetValueOrDefault() + report.TRT234.GetValueOrDefault() + report.TRT235.GetValueOrDefault() + report.TRT444.GetValueOrDefault() + report.TRT239.GetValueOrDefault() + report.TRT240.GetValueOrDefault() + report.TRT241.GetValueOrDefault() + report.TRT446.GetValueOrDefault() != 0);

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432181",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Zero net income or an overall trust loss cannot be distributed",
                    LongDescription = @"If there is an overall trust loss (i.e. the sum of net Australian income/loss plus net capital gains, plus attributed foreign income plus other assessable foreign source income minus allowable tax losses of earlier income years deducted this year, is less than zero) there can not be income distribution amounts present in any statement of distribution",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:NetA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.432181" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT130", Value = GetValueOrEmpty(report.TRT130) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT133", Value = GetValueOrEmpty(report.TRT133) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT134", Value = GetValueOrEmpty(report.TRT134) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT136", Value = GetValueOrEmpty(report.TRT136) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT141", Value = GetValueOrEmpty(report.TRT141) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT143", Value = GetValueOrEmpty(report.TRT143) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT144", Value = GetValueOrEmpty(report.TRT144) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT218", Value = GetValueOrEmpty(sumTRT218) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT219", Value = GetValueOrEmpty(sumTRT219) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT442", Value = GetValueOrEmpty(sumTRT442) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT223", Value = GetValueOrEmpty(sumTRT223) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT224", Value = GetValueOrEmpty(sumTRT224) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT225", Value = GetValueOrEmpty(sumTRT225) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT217", Value = GetValueOrEmpty(sumTRT217) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT233", Value = GetValueOrEmpty(report.TRT233) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT234", Value = GetValueOrEmpty(report.TRT234) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT235", Value = GetValueOrEmpty(report.TRT235) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT444", Value = GetValueOrEmpty(report.TRT444) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT239", Value = GetValueOrEmpty(report.TRT239) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT240", Value = GetValueOrEmpty(report.TRT240) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT241", Value = GetValueOrEmpty(report.TRT241) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT446", Value = GetValueOrEmpty(report.TRT446) });

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.TRT.432181

        #region VR.ATO.TRT.432193

        /*  VR.ATO.TRT.432193
            The overall net trust income is greater than zero (the sum of net Australian income/loss, net capital gains, attributed foreign income other assessable foreign source income minus allowable tax losses of earlier income years deducted this year), AND the sum of Attributed Foreign Income (including no beneficiary presently entitled) in the distribution statement, is not equal to the sum of Attributed Foreign Income amounts of the trust.

            Legacy Rule Format:
            (^TRT130 + ^TRT133 + ^TRT134 + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144) > 0 AND (Sum(^TRT224) + ^TRT240) <> (^TRT134 + ^TRT136)

            Technical Business Rule Format:
            (^TRT130 + ^TRT133 + ^TRT134 + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144) > 0 AND (Sum(^TRT224) + ^TRT240) <> (^TRT134 + ^TRT136)

            Data Elements:
    
            ^TRT134 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:ListedCountry:Income.InternationalDealings.Attributed.Amount
    
            ^TRT130 = TRT:RP:BusinessDetails:Income:Income.Net.Amount
    
            ^TRT133 = TRT:RP:BusinessDetails:Income:CapitalGains:Income.CapitalGainsNet.Amount
    
            ^TRT136 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:UnlistedCountry:Income.InternationalDealings.Attributed.Amount
    
            ^TRT141 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:Income.Net.Amount
    
            ^TRT143 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
    
            ^TRT144 = TRT:RP:BusinessDetails:Income:TaxLossesDeducted:IncomeTax.Deduction.TaxLossesDeducted.Amount
    
            ^TRT224 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.AttributedForeignIncome.Amount
    
            ^TRT240 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.AttributedForeignIncome.Amount
            */
        public virtual void VRATOTRT432193()
        {
            decimal? sumTRT224 = report.RP_StatementOfDistributionToBeneficiaryCollection == null ? 0 : report.RP_StatementOfDistributionToBeneficiaryCollection.Sum(f => f.TRT224.GetValueOrDefault());

            bool assertion = (report.TRT130.GetValueOrDefault() + report.TRT133.GetValueOrDefault() + report.TRT134.GetValueOrDefault() + report.TRT136.GetValueOrDefault() + report.TRT141.GetValueOrDefault() + report.TRT143.GetValueOrDefault() - report.TRT144.GetValueOrDefault() > 0 && sumTRT224 + report.TRT240.GetValueOrDefault() != report.TRT134.GetValueOrDefault() + report.TRT136.GetValueOrDefault());

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432193",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Attributed foreign income must equal distributed amounts",
                    LongDescription = @"If the overall net trust income is greater than zero (the sum of net Australian income/loss, net capital gains, attributed foreign income, other assessable foreign source income; minus allowable tax losses), then the sum of Attributed Foreign Income in the Statements of distribution must equal to the sum of the Attributed Foreign Income amounts for the Trust",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:AttributedForeignIncome/tns:ListedCountryInternationalDealingsA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.432193" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT130", Value = GetValueOrEmpty(report.TRT130) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT133", Value = GetValueOrEmpty(report.TRT133) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT134", Value = GetValueOrEmpty(report.TRT134) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT136", Value = GetValueOrEmpty(report.TRT136) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT141", Value = GetValueOrEmpty(report.TRT141) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT143", Value = GetValueOrEmpty(report.TRT143) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT144", Value = GetValueOrEmpty(report.TRT144) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT224", Value = GetValueOrEmpty(sumTRT224) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT240", Value = GetValueOrEmpty(report.TRT240) });

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.TRT.432193

        #region VR.ATO.TRT.432298

        /*  VR.ATO.TRT.432298
            If the trust is in an income position, and Australian franking credits from a New Zealand franking company is present, then this amount must be greater than or equal to the amount of Australian franking credits from a New Zealand franking company in the distribution statement (beneficiary and non-beneficiary).

            Legacy Rule Format:
            (^TRT130 + ^TRT133 + ^TRT134 + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144) > 0 AND ^TRT143 > 0 AND ^TRT143 < (Sum(^TRT217) + ^TRT233)

            Technical Business Rule Format:
            (^TRT130 + ^TRT133 + ^TRT134 + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144) > 0 AND ^TRT143 > 0 AND ^TRT143 < (Sum(^TRT217) + ^TRT233)

            Data Elements:
    
            ^TRT217 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
    
            ^TRT130 = TRT:RP:BusinessDetails:Income:Income.Net.Amount
    
            ^TRT133 = TRT:RP:BusinessDetails:Income:CapitalGains:Income.CapitalGainsNet.Amount
    
            ^TRT134 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:ListedCountry:Income.InternationalDealings.Attributed.Amount
    
            ^TRT136 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:UnlistedCountry:Income.InternationalDealings.Attributed.Amount
    
            ^TRT141 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:Income.Net.Amount
    
            ^TRT143 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
    
            ^TRT144 = TRT:RP:BusinessDetails:Income:TaxLossesDeducted:IncomeTax.Deduction.TaxLossesDeducted.Amount
    
            ^TRT233 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
            */
        public virtual void VRATOTRT432298()
        {
            decimal? sumTRT217 = report.RP_StatementOfDistributionToBeneficiaryCollection == null ? 0 : report.RP_StatementOfDistributionToBeneficiaryCollection.Sum(f => f.TRT217.GetValueOrDefault());

            bool assertion = (report.TRT130.GetValueOrDefault() + report.TRT133.GetValueOrDefault() + report.TRT134.GetValueOrDefault() + report.TRT136.GetValueOrDefault() + report.TRT141.GetValueOrDefault() + report.TRT143.GetValueOrDefault() - report.TRT144.GetValueOrDefault()) > 0 && report.TRT143.GetValueOrDefault() > 0 && report.TRT143.GetValueOrDefault() < (sumTRT217 + report.TRT233.GetValueOrDefault());

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432667",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Australian franking credits from a NZ franking company amount must not be less than distributed amounts",
                    LongDescription = @"If the overall net trust income is greater than zero (the sum of net Australian income/loss, net capital gains, attributed foreign income, other assessable foreign source income; minus allowable tax losses), and 'Australian franking credits from a NZ franking company' is present for the trust, then this amount must be greater than or equal to the amount of 'Australian franking credits from a NZ franking company' in the Statements of distribution",
                    Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary/tns:Information/tns:IncomeTaxFrankingCreditsReceivedFromNewZealandCompaniesA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.432298" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT130", Value = GetValueOrEmpty(report.TRT130) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT133", Value = GetValueOrEmpty(report.TRT133) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT134", Value = GetValueOrEmpty(report.TRT134) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT136", Value = GetValueOrEmpty(report.TRT136) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT141", Value = GetValueOrEmpty(report.TRT141) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT143", Value = GetValueOrEmpty(report.TRT143) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT144", Value = GetValueOrEmpty(report.TRT144) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT217", Value = GetValueOrEmpty(sumTRT217) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT233", Value = GetValueOrEmpty(report.TRT233) });

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.TRT.432298

        #region VR.ATO.TRT.432315

        /*  VR.ATO.TRT.432315
            The overall net trust income is greater than zero (the sum of net Australian income/loss, net capital gains, attributed foreign income other assessable foreign source income minus allowable tax losses of earlier income years deducted this year) and the sum of Foreign income tax offsets (including no beneficiary presently entitled) is not equal to the Foreign income tax offsets.

            Legacy Rule Format:
            (^TRT130 + ^TRT133 + ^TRT134 + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144) > 0 AND (Sum(^TRT226) + ^TRT242) <> ^TRT142

            Technical Business Rule Format:
            (^TRT130 + ^TRT133 + ^TRT134 + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144) > 0 AND (Sum(^TRT226) + ^TRT242) <> ^TRT142

            Data Elements:
    
            ^TRT226 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Income.InternationalDealings.TaxOffset.Amount
    
            ^TRT130 = TRT:RP:BusinessDetails:Income:Income.Net.Amount
    
            ^TRT133 = TRT:RP:BusinessDetails:Income:CapitalGains:Income.CapitalGainsNet.Amount
    
            ^TRT134 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:ListedCountry:Income.InternationalDealings.Attributed.Amount
    
            ^TRT136 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:UnlistedCountry:Income.InternationalDealings.Attributed.Amount
    
            ^TRT141 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:Income.Net.Amount
    
            ^TRT142 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:Income.InternationalDealings.TaxOffset.Amount
    
            ^TRT143 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount
    
            ^TRT144 = TRT:RP:BusinessDetails:Income:TaxLossesDeducted:IncomeTax.Deduction.TaxLossesDeducted.Amount
    
            ^TRT242 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:Income.InternationalDealings.TaxOffset.Amount
            */
        public virtual void VRATOTRT432315()
        {
            decimal? sumTRT226 = report.RP_StatementOfDistributionToBeneficiaryCollection == null ? 0 : report.RP_StatementOfDistributionToBeneficiaryCollection.Sum(f => f.TRT226.GetValueOrDefault());

            bool assertion = (report.TRT130.GetValueOrDefault() + report.TRT133.GetValueOrDefault() + report.TRT134.GetValueOrDefault() + report.TRT136.GetValueOrDefault() + report.TRT141.GetValueOrDefault() + report.TRT143.GetValueOrDefault() - report.TRT144.GetValueOrDefault()) > 0 && (sumTRT226.GetValueOrDefault() + report.TRT242.GetValueOrDefault()) != report.TRT142.GetValueOrDefault();

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432315",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Foreign income tax offsets amount must equal distributed amounts",
                    LongDescription = @"If the overall net trust income is greater than zero (the sum of net Australian income/loss, net capital gains, attributed foreign income, other assessable foreign source income; minus allowable tax losses), then the sum of Foreign income tax offsets in the Statements of Distribution, including 'Income to which no beneficiary is presently entitled', must equal the Foreign income tax offsets' of the trust",
                    Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary/tns:Information/tns:IncomeInternationalDealingsTaxOffsetA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.432315" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT130", Value = GetValueOrEmpty(report.TRT130) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT133", Value = GetValueOrEmpty(report.TRT133) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT134", Value = GetValueOrEmpty(report.TRT134) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT136", Value = GetValueOrEmpty(report.TRT136) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT141", Value = GetValueOrEmpty(report.TRT141) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT143", Value = GetValueOrEmpty(report.TRT143) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT144", Value = GetValueOrEmpty(report.TRT144) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT226", Value = GetValueOrEmpty(sumTRT226) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT242", Value = GetValueOrEmpty(report.TRT242) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT142", Value = GetValueOrEmpty(report.TRT142) });

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.TRT.432315

        #region VR.ATO.TRT.432342

        /*  VR.ATO.TRT.432342
            The total amount at National rental affordability scheme tax offset (beneficiary and non beneficiary) must equal the amount at National rental affordability scheme tax offset entitlement.

            Legacy Rule Format:
            Sum(^TRT342) + ^TRT343 <> ^TRT340

            Technical Business Rule Format:
            Sum(^TRT342) + ^TRT343 <> ^TRT340

            Data Elements:
    
            ^TRT340 = TRT:RP:TaxOffsets:TaxOffsetClaim.NationalRentalAffordabilitySchemeEntitlement.Amount
    
            ^TRT342 = TRT:RP:StatementOfDistributionToBeneficiary:Information:TaxOffsetClaim.NationalRentalAffordabilitySchemeEntitlement.Amount
    
            ^TRT343 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:TaxOffsetClaim.NationalRentalAffordabilitySchemeEntitlement.Amount
            */
        public virtual void VRATOTRT432342()
        {
            decimal? sumTRT342 = report.RP_StatementOfDistributionToBeneficiaryCollection == null ? 0 : report.RP_StatementOfDistributionToBeneficiaryCollection.Sum(f => f.TRT342.GetValueOrDefault());

            bool assertion = sumTRT342.GetValueOrDefault() + report.TRT343.GetValueOrDefault() != report.TRT340.GetValueOrDefault();
            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432342",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"National rental affordability scheme tax offset amount must equal distributed amounts",
                    LongDescription = @"The sum of 'Share of National rental affordability scheme tax offset' amounts in the Statements of Distribution must equal the amount at 'National rental affordability scheme tax offset entitlement' for the trust",
                    Location = "/tns:TRT/tns:RP/tns:TaxOffsets/tns:TaxOffsetClaimNationalRentalAffordabilitySchemeEntitlementA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.432342" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT340", Value = GetValueOrEmpty(report.TRT340) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Sum Of TRT342", Value = GetValueOrEmpty(sumTRT342) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT343", Value = GetValueOrEmpty(report.TRT343) });

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.TRT.432342

        #region VR.ATO.TRT.432578

        /*  VR.ATO.TRT.432578
            In the distribution statement the sum of Share of credit for TFN amounts withheld from payments from closely held trusts to which beneficiaries are entitled to and to which beneficiaries are not entitled to is not equal to the Credit for TFN amounts withheld from payments from closely held trust.

            Legacy Rule Format:
            ^TRT418 <> (Sum(^TRT419) + ^TRT435)

            Technical Business Rule Format:
            ^TRT418 <> (Sum(^TRT419) + ^TRT435)

            Data Elements:
    
            ^TRT418 = TRT:RP:BusinessDetails:Income:ShareOfCredits:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldFromCloselyHeldTrustShareReceivedIndirectly.Amount
    
            ^TRT419 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldFromCloselyHeldTrust.Amount
    
            ^TRT435 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.PayAsYouGoWithholding.CreditForTaxWithheldFromCloselyHeldTrust.Amount
            */
        public virtual void VRATOTRT432578()
        {
            decimal? sumTRT419 = report.RP_StatementOfDistributionToBeneficiaryCollection == null ? 0 : report.RP_StatementOfDistributionToBeneficiaryCollection.Sum(f => f.TRT419.GetValueOrDefault());

            bool assertion = (report.TRT418.GetValueOrDefault() != sumTRT419 + report.TRT435.GetValueOrDefault());

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432624",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Sum of Share of credit for TFN amounts withheld not equal to Credit for TFN amounts withheld",
                    LongDescription = @"In the distribution statement the sum of Share of credit for TFN amounts withheld from payments from closely held trusts to which beneficiaries are entitled to and to which beneficiaries are not entitled to is not equal to the Credit for TFN amounts withheld from payments from closely held trusts.",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:ShareOfCredits/tns:TaxPayAsYouGoWithholdingCreditForTaxWithheldFromCloselyHeldTrustShareReceivedIndirectlyA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.432578" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT418", Value = GetValueOrEmpty(report.TRT418) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT419", Value = GetValueOrEmpty(sumTRT419) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT435", Value = GetValueOrEmpty(report.TRT435) });

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.TRT.432578

        #region VR.ATO.TRT.432632

        /*  VR.ATO.TRT.432632
            The sum of Share of income of the trust estate (beneficary entitled) and the Share of income of the trust estate (no beneficiary entitled) in the Statement of distribution, must equal the Income of the Trust Estate.

            Legacy Rule Format:
            (AnyOccurrence(^TRT1070, ^TRT441 > 0) OR ^TRT443 > 0) AND ((Sum(^TRT441) + ^TRT443) <> ^TRT440)

            Technical Business Rule Format:
            (AnyOccurrence(^TRT1070, ^TRT441 > 0) OR ^TRT443 > 0) AND ((Sum(^TRT441) + ^TRT443) <> ^TRT440)

            Data Elements:
    
            ^TRT440 = TRT:RP:Income.TrustEstateIncome.Amount
    
            ^TRT441 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Income.TrustEstateIncomeShare.Amount
    
            ^TRT443 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:Income.TrustEstateIncomeShare.Amount
    
            ^TRT1070 = TRT:RP:StatementOfDistributionToBeneficiary
            */
        public virtual void VRATOTRT432632()
        {
            bool anyTRT441ORTRT443 = (report.RP_StatementOfDistributionToBeneficiaryCollection == null ? false : report.RP_StatementOfDistributionToBeneficiaryCollection.Any(f => f.TRT441.GetValueOrDefault() > 0) ? true : false) || report.TRT443.GetValueOrDefault() > 0;
            decimal? sumTRT441 = report.RP_StatementOfDistributionToBeneficiaryCollection == null ? 0 : report.RP_StatementOfDistributionToBeneficiaryCollection.Sum(f => f.TRT441.GetValueOrDefault());

            bool sumEqualTRT440 = (sumTRT441.GetValueOrDefault() + report.TRT443.GetValueOrDefault()) != report.TRT440.GetValueOrDefault();

            bool assertion = anyTRT441ORTRT443 && sumEqualTRT440;
            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432632",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Share of income of the trust estate is incorrect",
                    LongDescription = @"The sum of Share of income of the trust estate (beneficiary entitled and no beneficiary entitled) does not equal the income of the trust estate amount.",
                    Location = "/tns:TRT/tns:RP/tns:IncomeTrustEstateIncomeA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.432632" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT440", Value = GetValueOrEmpty(report.TRT440) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Sum of TRT441", Value = GetValueOrEmpty(sumTRT441) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT443", Value = GetValueOrEmpty(report.TRT443) });

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.TRT.432632

        #region VR.ATO.TRT.432714

        /*  VR.ATO.TRT.432714
            The Share of Other Refundable Tax Offsets plus the sum of Exploration Credits is not equal to Other Refundable Tax Offset

            Legacy Rule Format:
            Sum(^TRT465) + ^TRT466 <> ^TRT463

            Technical Business Rule Format:
            Sum(^TRT465) + ^TRT466 <> ^TRT463

            Data Elements:
    
            ^TRT463 = TRT:RP:TaxOffsets:IncomeTax.TaxOffsetRefundableOther.Amount
    
            ^TRT465 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.ExplorationCreditsDistributed.Amount
    
            ^TRT466 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.TaxOffsetRefundableOther.Amount
            */
        public virtual void VRATOTRT432714()
        {
            decimal? sumTRT465 = report.RP_StatementOfDistributionToBeneficiaryCollection == null ? 0 : report.RP_StatementOfDistributionToBeneficiaryCollection.Sum(f => f.TRT465.GetValueOrDefault());

            bool assertion = (sumTRT465.GetValueOrDefault() + report.TRT466.GetValueOrDefault()) != report.TRT463.GetValueOrDefault();
            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432639",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The Share of Other Refundable Tax Offsets plus the sum of Exploration Credits is not equal to Other Refundable Tax Offset",
                    Location = "/tns:TRT/tns:RP/tns:TaxOffsets/tns:IncomeTaxTaxOffsetRefundableOtherA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.432714" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT463", Value = GetValueOrEmpty(report.TRT463) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Sum of TRT465", Value = GetValueOrEmpty(sumTRT465) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT466", Value = GetValueOrEmpty(report.TRT466) });

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.TRT.432714

        #region VR.ATO.TRT.432765

        /*  VR.ATO.TRT.432765
            The total amount at Credit for amounts withheld from foreign resident capital gains withholding (beneficiary and non beneficiary) must equal the amount at Credit for amounts withheld from foreign resident capital gains withholding.

            Legacy Rule Format:
            ^TRT480 <> (Sum(^TRT481) + ^TRT482)

            Technical Business Rule Format:
            ^TRT480 <> (Sum(^TRT481) + ^TRT482)

            Data Elements:
    
            ^TRT480 = TRT:RP:BusinessDetails:Income:CapitalGains:IncomeTax.PayAsYouGoWithholding.CreditForCapitalGainsWithheldFromForeignResidents.Amount
    
            ^TRT481 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.PayAsYouGoWithholding.CreditForCapitalGainsWithheldFromForeignResidents.Amount
    
            ^TRT482 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.PayAsYouGoWithholding.CreditForCapitalGainsWithheldFromForeignResidents.Amount
            */
        public virtual void VRATOTRT432765()
        {
            decimal? sumTRT481 = report.RP_StatementOfDistributionToBeneficiaryCollection == null ? 0 : report.RP_StatementOfDistributionToBeneficiaryCollection.Sum(f => f.TRT481.GetValueOrDefault());

            bool assertion = report.TRT480.GetValueOrDefault() != sumTRT481 + report.TRT482.GetValueOrDefault();

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432765",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Credit for amounts withheld from foreign resident capital gains withholding amount must equal distributed amounts",
                    LongDescription = @"The sum of income of the Credit for amounts withheld from foreign resident capital gains withholding (beneficiary and non beneficiary) does not equal the income of the trust Credit for amounts withheld from foreign resident capital gains withholding.",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:CapitalGains/tns:TaxPayAsYouGoWithholdingCreditForCapitalGainsWithheldFromForeignResidentsA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.432765" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT480", Value = GetValueOrEmpty(report.TRT480) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT481", Value = GetValueOrEmpty(sumTRT481) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT482", Value = GetValueOrEmpty(report.TRT482) });

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.TRT.432765

        #region VR.ATO.TRT.432180
		
		/*  VR.ATO.TRT.432180
            The amount of net Australian income/loss is not equal to the sum of all Australian income less all Australian source deductions.

            Legacy Rule Format:
            OutsideRange(^TRT130, ^TRT71 + ^TRT72 + ^TRT76 + ^TRT77 - ^TRT79 + ^TRT80 + ^TRT81 - ^TRT83 + ^TRT437 - ^TRT438 + ^TRT89 - ^TRT90 - ^TRT91 - ^TRT92 + ^TRT93 + ^TRT94 + ^TRT96 + ^TRT97 + ^TRT98 + ^TRT105 + Sum(^TRT109) + Sum(^TRT111) + (Sum(^TRT113) * 0.05) + ^TRT115 + ^TRT116 + ^TRT117 + ^TRT118 + ^TRT119 + ^TRT100 + ^TRT101 + Sum(^TRT102) +  Sum(^TRT121) - (^TRT123 + ^TRT439 + ^TRT124 + Sum(^TRT129)) , 1)

            Technical Business Rule Format:
            OutsideRange(^TRT130, ^TRT71 + ^TRT72 + ^TRT76 + ^TRT77 - ^TRT79 + ^TRT80 + ^TRT81 - ^TRT83 + ^TRT437 - ^TRT438 + ^TRT89 - ^TRT90 - ^TRT91 - ^TRT92 + ^TRT93 + ^TRT94 + ^TRT96 + ^TRT97 + ^TRT98 + ^TRT105 + Sum(^TRT109) + Sum(^TRT111) + (Sum(^TRT113) * 0.05) + ^TRT115 + ^TRT116 + ^TRT117 + ^TRT118 + ^TRT119 + ^TRT100 + ^TRT101 + Sum(^TRT102) +  Sum(^TRT121) - (^TRT123 + ^TRT439 + ^TRT124 + Sum(^TRT129)) , 1)

            Data Elements:

            ^TRT130 = TRT:RP:BusinessDetails:Income:Income.Net.Amount

            ^TRT71 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomePrimaryProduction:Income.Net.Amount

            ^TRT72 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:IncomeNonPrimaryProduction:Income.Net.Amount

            ^TRT76 = TRT:RP:BusinessDetails:Income:PartnershipsTrustsPrimaryProduction:Income.PartnershipDistributionGross.Amount

            ^TRT77 = TRT:RP:BusinessDetails:Income:PartnershipsTrustsPrimaryProduction:Income.TrustShareNet.Amount

            ^TRT79 = TRT:RP:BusinessDetails:Income:PartnershipsTrustsPrimaryProduction:Expense.DistributionDeductionsOther.Amount

            ^TRT80 = TRT:RP:BusinessDetails:Income:PartnershipTrustNonPrimaryProduction:Income.PartnershipDistributionLessForeignIncome.Amount

            ^TRT81 = TRT:RP:BusinessDetails:Income:PartnershipTrustNonPrimaryProduction:Income.TrustShareNetExcludeNetCapitalGainsAndForeignIncomeAndDistributionFranked.Amount

            ^TRT83 = TRT:RP:BusinessDetails:Income:PartnershipTrustNonPrimaryProduction:Expense.DistributionDeductionsOther.Amount

            ^TRT89 = TRT:RP:BusinessDetails:Income:Rent:Income.Operating.RentalIncomeGross.Amount

            ^TRT90 = TRT:RP:BusinessDetails:Income:Rent:Expense.Interest.Amount

            ^TRT91 = TRT:RP:BusinessDetails:Income:Rent:Expense.CapitalWorksDeduction.Amount

            ^TRT92 = TRT:RP:BusinessDetails:Income:Rent:IncomeTax.Deduction.RentalIncomeDeductionsOtherThanInterestAndCapitalWorks.Amount

            ^TRT93 = TRT:RP:BusinessDetails:Income:Income.ForestryManagedInvestmentScheme.Amount

            ^TRT94 = TRT:RP:BusinessDetails:Income:GrossInterest:Income.Interest.Gross.Amount

            ^TRT96 = TRT:RP:BusinessDetails:Income:Dividends:Income.DividendsUnfranked.Amount

            ^TRT97 = TRT:RP:BusinessDetails:Income:Dividends:Income.DividendsFranked.Amount

            ^TRT98 = TRT:RP:BusinessDetails:Income:Dividends:IncomeTax.FrankingCredits.Amount

            ^TRT100 = TRT:RP:BusinessDetails:Income:SuperannuationLumpSumWithholdingPaymentDetails:SuperannuationBenefit.DeathBenefitLumpSumPaymentTaxedElement.Amount

            ^TRT101 = TRT:RP:BusinessDetails:Income:SuperannuationLumpSumWithholdingPaymentDetails:SuperannuationBenefit.DeathBenefitLumpSumPaymentUntaxedElement.Amount

            ^TRT102 = TRT:RP:BusinessDetails:Income:EmploymentTerminationPaymentDetails:SuperannuationBenefit.DeathBenefitEmploymentTerminationPaymentTaxableComponent.Amount

            ^TRT105 = TRT:RP:BusinessDetails:Income:OtherAustralianIncome:Pension.Total.Amount

            ^TRT109 = TRT:RP:BusinessDetails:Income:IndividualNonBusinessWithholdingPaymentDetails:Remuneration.IndividualNonBusinessGross.Amount

            ^TRT111 = TRT:RP:BusinessDetails:Income:IndividualNonBusinessWithholdingPaymentDetails:Remuneration.UnusedAnnualLongServiceLeavePaymentLumpSumA.Amount

            ^TRT113 = TRT:RP:BusinessDetails:Income:IndividualNonBusinessWithholdingPaymentDetails:Remuneration.UnusedAnnualLongServiceLeavePaymentLumpSumB.Amount

            ^TRT115 = TRT:RP:BusinessDetails:Income:SuperannuationIncomeStreamWithholdingPaymentDetails:IncomeTax.Superannuation.TaxableComponentTaxedElement.Amount

            ^TRT116 = TRT:RP:BusinessDetails:Income:SuperannuationIncomeStreamWithholdingPaymentDetails:IncomeTax.Superannuation.TaxableComponentUntaxedElement.Amount

            ^TRT117 = TRT:RP:BusinessDetails:Income:SuperannuationIncomeStreamWithholdingPaymentDetails:LumpSumInArrears:IncomeTax.Superannuation.TaxableComponentTaxedElement.Amount

            ^TRT118 = TRT:RP:BusinessDetails:Income:SuperannuationIncomeStreamWithholdingPaymentDetails:LumpSumInArrears:IncomeTax.Superannuation.TaxableComponentUntaxedElement.Amount

            ^TRT119 = TRT:RP:BusinessDetails:Income:SuperannuationIncomeStreamWithholdingPaymentDetails:LumpSumInArrears:Remuneration.BonusesAndAllowances.Amount

            ^TRT121 = TRT:RP:BusinessDetails:Income:OtherAustralianIncome:Income.Other.Amount

            ^TRT123 = TRT:RP:BusinessDetails:Income:Deductions:Expense.AustralianInvestmentAllowableDeduction.Amount

            ^TRT124 = TRT:RP:BusinessDetails:Income:Deductions:Expense.ForestryManagedInvestmentSchemeDeduction.Amount

            ^TRT129 = TRT:RP:BusinessDetails:Income:Deductions:OtherDeductions:Expense.DeductibleOther.Amount

            ^TRT437 = TRT:RP:BusinessDetails:Income:PartnershipTrustNonPrimaryProduction:Income.TrustDistributionFranked.Amount

            ^TRT438 = TRT:RP:BusinessDetails:Income:PartnershipTrustNonPrimaryProduction:IncomeTax.Deduction.TrustDistributionFranked.Amount

            ^TRT439 = TRT:RP:BusinessDetails:Income:Deductions:IncomeTax.Deduction.DistributionFranked.Amount
            */
        public virtual void VRATOTRT432180(TRT2026.RP_BusinessDetails_Income_OtherAustralianIncome otherAustralianIncome, int itemIndex4)
        {
            decimal? sumOfAIMinusASD = report.TRT71.GetValueOrDefault() + report.TRT72.GetValueOrDefault() + report.TRT76.GetValueOrDefault() + report.TRT77.GetValueOrDefault() - report.TRT79.GetValueOrDefault() + report.TRT80.GetValueOrDefault() + report.TRT81.GetValueOrDefault() - report.TRT83.GetValueOrDefault() + report.TRT437.GetValueOrDefault() - report.TRT438.GetValueOrDefault() + report.TRT89.GetValueOrDefault() - report.TRT90.GetValueOrDefault() - report.TRT91.GetValueOrDefault() - report.TRT92.GetValueOrDefault() + report.TRT93.GetValueOrDefault() + report.TRT94.GetValueOrDefault() + report.TRT96.GetValueOrDefault() + report.TRT97.GetValueOrDefault() + report.TRT98.GetValueOrDefault() + report.TRT105.GetValueOrDefault() + (report.RP_BusinessDetails_Income_IndividualNonBusinessWithholdingPaymentDetailsCollection == null ? 0 : report.RP_BusinessDetails_Income_IndividualNonBusinessWithholdingPaymentDetailsCollection.Sum(f => f.TRT109.GetValueOrDefault())) + (report.RP_BusinessDetails_Income_IndividualNonBusinessWithholdingPaymentDetailsCollection == null ? 0 : report.RP_BusinessDetails_Income_IndividualNonBusinessWithholdingPaymentDetailsCollection.Sum(f => f.TRT111.GetValueOrDefault())) + (report.RP_BusinessDetails_Income_IndividualNonBusinessWithholdingPaymentDetailsCollection == null ? 0 : report.RP_BusinessDetails_Income_IndividualNonBusinessWithholdingPaymentDetailsCollection.Sum(f => f.TRT113.GetValueOrDefault())) * 0.05M + report.TRT115.GetValueOrDefault() + report.TRT116.GetValueOrDefault() + report.TRT117.GetValueOrDefault() + report.TRT118.GetValueOrDefault() + report.TRT119.GetValueOrDefault() + report.TRT100.GetValueOrDefault() + report.TRT101.GetValueOrDefault() + (report.RP_BusinessDetails_Income_EmploymentTerminationPaymentDetailsCollection == null ? 0 : report.RP_BusinessDetails_Income_EmploymentTerminationPaymentDetailsCollection.Sum(f => f.TRT102.GetValueOrDefault())) + (report.RP_BusinessDetails_Income_OtherAustralianIncomeCollection == null ? 0 : report.RP_BusinessDetails_Income_OtherAustralianIncomeCollection.Sum(f => f.TRT121.GetValueOrDefault())) - (report.TRT123.GetValueOrDefault() + report.TRT439.GetValueOrDefault() + report.TRT124.GetValueOrDefault() + (report.RP_BusinessDetails_Income_Deductions_OtherDeductionsCollection == null ? 0 : report.RP_BusinessDetails_Income_Deductions_OtherDeductionsCollection.Sum(f => f.TRT129.GetValueOrDefault())));
			
            bool assertion = OutsideRange(report.TRT130.GetValueOrDefault(), sumOfAIMinusASD.GetValueOrDefault(), 1);
			
            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432613",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Net Australian income must equal income less deductions",
                    LongDescription = @"The amount of net Australian income/loss is not equal to the sum of all Australian income less all Australian source deductions.",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:NetA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.432180" } },
                };
				
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT130", Value = GetValueOrEmpty(report.TRT130) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Total Calculation", Value = GetValueOrEmpty(sumOfAIMinusASD) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT71", Value = GetValueOrEmpty(report.TRT71) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT72", Value = GetValueOrEmpty(report.TRT72) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT76", Value = GetValueOrEmpty(report.TRT76) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT77", Value = GetValueOrEmpty(report.TRT77) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT79", Value = GetValueOrEmpty(report.TRT79) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT80", Value = GetValueOrEmpty(report.TRT80) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT81", Value = GetValueOrEmpty(report.TRT81) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT83", Value = GetValueOrEmpty(report.TRT83) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT437", Value = GetValueOrEmpty(report.TRT437) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT438", Value = GetValueOrEmpty(report.TRT438) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT89", Value = GetValueOrEmpty(report.TRT89) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT90", Value = GetValueOrEmpty(report.TRT90) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT91", Value = GetValueOrEmpty(report.TRT91) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT92", Value = GetValueOrEmpty(report.TRT92) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT93", Value = GetValueOrEmpty(report.TRT93) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT94", Value = GetValueOrEmpty(report.TRT94) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT96", Value = GetValueOrEmpty(report.TRT96) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT97", Value = GetValueOrEmpty(report.TRT97) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT98", Value = GetValueOrEmpty(report.TRT98) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT105", Value = GetValueOrEmpty(report.TRT105) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT109", Value = GetValueOrEmpty(report.RP_BusinessDetails_Income_IndividualNonBusinessWithholdingPaymentDetailsCollection == null ? 0 : report.RP_BusinessDetails_Income_IndividualNonBusinessWithholdingPaymentDetailsCollection.Sum(f => f.TRT109.GetValueOrDefault())) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT111", Value = GetValueOrEmpty(report.RP_BusinessDetails_Income_IndividualNonBusinessWithholdingPaymentDetailsCollection == null ? 0 : report.RP_BusinessDetails_Income_IndividualNonBusinessWithholdingPaymentDetailsCollection.Sum(f => f.TRT111.GetValueOrDefault())) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT113", Value = GetValueOrEmpty(report.RP_BusinessDetails_Income_IndividualNonBusinessWithholdingPaymentDetailsCollection == null ? 0 : report.RP_BusinessDetails_Income_IndividualNonBusinessWithholdingPaymentDetailsCollection.Sum(f => f.TRT113.GetValueOrDefault())) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT115", Value = GetValueOrEmpty(report.TRT115) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT116", Value = GetValueOrEmpty(report.TRT116) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT117", Value = GetValueOrEmpty(report.TRT117) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT118", Value = GetValueOrEmpty(report.TRT118) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT119", Value = GetValueOrEmpty(report.TRT119) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT100", Value = GetValueOrEmpty(report.TRT100) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT101", Value = GetValueOrEmpty(report.TRT101) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT102", Value = GetValueOrEmpty(report.RP_BusinessDetails_Income_EmploymentTerminationPaymentDetailsCollection == null ? 0 : report.RP_BusinessDetails_Income_EmploymentTerminationPaymentDetailsCollection.Sum(f => f.TRT102.GetValueOrDefault())) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT121", Value = GetValueOrEmpty(report.RP_BusinessDetails_Income_OtherAustralianIncomeCollection == null ? 0 : report.RP_BusinessDetails_Income_OtherAustralianIncomeCollection.Sum(f => f.TRT121.GetValueOrDefault())) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT123", Value = GetValueOrEmpty(report.TRT123) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT439", Value = GetValueOrEmpty(report.TRT439) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT124", Value = GetValueOrEmpty(report.TRT124) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT129", Value = GetValueOrEmpty(report.RP_BusinessDetails_Income_Deductions_OtherDeductionsCollection == null ? 0 : report.RP_BusinessDetails_Income_Deductions_OtherDeductionsCollection.Sum(f => f.TRT129.GetValueOrDefault())) });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.TRT.432784

        /*  VR.ATO.TRT.432784
            The total amount at Early stage venture capital limited partnership tax offset (beneficiary and no beneficiary presently entitled) must equal the amount at Early stage venture capital limited partnership tax offset.

            Legacy Rule Format:
            ^TRT500 <> (Sum(^TRT502) + ^TRT504)

            Technical Business Rule Format:
            ^TRT500 <> (Sum(^TRT502) + ^TRT504)

            Data Elements:
    
            ^TRT500 = TRT:RP:TaxOffsets:LimitedPartnershipTaxOffset:TaxOffsetClaim.NonRefundableOther.Amount
    
            ^TRT502 = TRT:RP:StatementOfDistributionToBeneficiary:Information:EarlyStageVentureTaxOffset:TaxOffsetClaim.NonRefundableOther.Amount
    
            ^TRT504 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:EarlyStageVentureNoBeneficiary:TaxOffsetClaim.NonRefundableOther.Amount
            */
        public virtual void VRATOTRT432784()
        {
            decimal? sumTRT502 = report.RP_StatementOfDistributionToBeneficiaryCollection == null ? 0 : report.RP_StatementOfDistributionToBeneficiaryCollection.Sum(f => f.TRT502.GetValueOrDefault());

            bool assertion = report.TRT500.GetValueOrDefault() != (sumTRT502.GetValueOrDefault() + report.TRT504.GetValueOrDefault());
            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432784",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Early stage venture capital limited partnership tax offset amount must equal distributed amounts.",
                    LongDescription = @"The total amount at Early stage venture capital limited partnership tax offset (beneficiary and no beneficiary presently entitled) must equal the amount at Early stage venture capital limited partnership tax offset.",
                    Location = "/tns:TRT/tns:RP/tns:TaxOffsets/tns:LimitedPartnershipTaxOffsetTaxOffsetClaimNonRefundableOtherA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.432784" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT500", Value = GetValueOrEmpty(report.TRT500) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Sum of TRT502", Value = GetValueOrEmpty(sumTRT502) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT504", Value = GetValueOrEmpty(report.TRT504) });

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.TRT.432784

        #region VR.ATO.TRT.432787

        /*  VR.ATO.TRT.432787
            The total of amounts at Early stage investor tax offset (beneficiary and no beneficiary presently entitled) must equal the amount at Early stage investor tax offset.

            Legacy Rule Format:
            ^TRT501 <> (Sum(^TRT503) + ^TRT506)

            Technical Business Rule Format:
            ^TRT501 <> (Sum(^TRT503) + ^TRT506)

            Data Elements:
    
            ^TRT501 = TRT:RP:TaxOffsets:InvestorTaxOffset:TaxOffsetClaim.NonRefundableOther.Amount
    
            ^TRT503 = TRT:RP:StatementOfDistributionToBeneficiary:Information:EarlyStageInvestorTaxOffset:TaxOffsetClaim.NonRefundableOther.Amount
    
            ^TRT506 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:InvestorNoBeneficiary:TaxOffsetClaim.NonRefundableOther.Amount
            */
        public virtual void VRATOTRT432787()
        {
            decimal? sumTRT503 = report.RP_StatementOfDistributionToBeneficiaryCollection == null ? 0 : report.RP_StatementOfDistributionToBeneficiaryCollection.Sum(f => f.TRT503.GetValueOrDefault());

            bool assertion = report.TRT501.GetValueOrDefault() != (sumTRT503.GetValueOrDefault() + report.TRT506.GetValueOrDefault());
            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.432787",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Early stage investor tax offset amount must equal distributed amounts.",
                    LongDescription = @"The total of amounts at Early stage investor tax offset (beneficiary and no beneficiary presently entitled) must equal the amount at Early stage investor tax offset.",
                    Location = "/tns:TRT/tns:RP/tns:TaxOffsets/tns:InvestorTaxOffsetTaxOffsetClaimNonRefundableOtherA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.432787" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT501", Value = GetValueOrEmpty(report.TRT501) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Sum of TRT503", Value = GetValueOrEmpty(sumTRT503) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT506", Value = GetValueOrEmpty(report.TRT506) });

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.TRT.432787

        #region VR.ATO.TRT.433083

        /*  VR.ATO.TRT.433083
            Capital gains must be equal to the sum of Gross capital gain minus Capital losses applied, CGT discounts applied and CGT small business concessions applied with a tolerance of $5

            Legacy Rule Format:
            AnyOccurrence(^TRT1070, OutsideRange(^TRT223, ^TRT569 - ^TRT570 - ^TRT571 - ^TRT572, 5))

            Technical Business Rule Format:
            AnyOccurrence(^TRT1070, OutsideRange(^TRT223, ^TRT569 - ^TRT570 - ^TRT571 - ^TRT572, 5))

            Data Elements:

            ^TRT223 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Income.CapitalGainsNet.Amount

            ^TRT569 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Capital.Gains.Total.Amount

            ^TRT570 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Capital.Losses.Total.Amount

            ^TRT571 = TRT:RP:StatementOfDistributionToBeneficiary:Information:TaxConcession.CapitalGains.DiscountTotal.Amount

            ^TRT572 = TRT:RP:StatementOfDistributionToBeneficiary:Information:TaxConcession.CapitalGains.DiscountAndSmallBusinessConcessionsTotal.Amount

            ^TRT1070 = TRT:RP:StatementOfDistributionToBeneficiary
            */
        public virtual void VRATOTRT433083()
        {
            List<TRT2026.RP_StatementOfDistributionToBeneficiary> statementOfDistributionCollection = report.RP_StatementOfDistributionToBeneficiaryCollection?.FindAll(
                tuple => OutsideRange(tuple.TRT223.GetValueOrDefault(), tuple.TRT569.GetValueOrDefault() - tuple.TRT570.GetValueOrDefault() - tuple.TRT571.GetValueOrDefault() - tuple.TRT572.GetValueOrDefault() > 0 ? tuple.TRT569.GetValueOrDefault() - tuple.TRT570.GetValueOrDefault() - tuple.TRT571.GetValueOrDefault() - tuple.TRT572.GetValueOrDefault() : 0, 5)
            );

            bool assertion = statementOfDistributionCollection != null && statementOfDistributionCollection.Any();
            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.433083",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Capital gains is incorrect",
                    LongDescription = @"Capital gains must be equal to the sum of Gross capital gain minus Capital losses applied, CGT discounts applied and CGT small business concessions applied with a tolerance of $5",
                    Location = "/tns:TRT/tns:RP/tns:StatementOfDistributionToBeneficiaryCollection/tns:StatementOfDistributionToBeneficiary/tns:Information/tns:IncomeCapitalGainsNetA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.433083" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT223", Value = GetValueOrEmpty(statementOfDistributionCollection.First().TRT223) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT569", Value = GetValueOrEmpty(statementOfDistributionCollection.First().TRT569) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT570", Value = GetValueOrEmpty(statementOfDistributionCollection.First().TRT570) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT571", Value = GetValueOrEmpty(statementOfDistributionCollection.First().TRT571) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT572", Value = GetValueOrEmpty(statementOfDistributionCollection.First().TRT572) });

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.TRT.433083

        #region VR.ATO.TRT.433084

        /*  VR.ATO.TRT.433084
            Net small business income must not be less than the sum of Share of net small business income with a tolerance of $5

            Legacy Rule Format:
            ^TRT478 + 5 <  Sum(^TRT479)

            Technical Business Rule Format:
            ^TRT478 + 5 <  Sum(^TRT479)

            Data Elements:

            ^TRT478 = TRT:RP:BusinessDetails:Income:BusinessIncomeAndExpenses:Income.SmallBusinessEntityNet.Amount

            ^TRT479 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Income.SmallBusinessEntityNet.Amount
            */
        public virtual void VRATOTRT433084()
        {
            decimal TRT479Total = report.RP_StatementOfDistributionToBeneficiaryCollection != null ? report.RP_StatementOfDistributionToBeneficiaryCollection.Sum(tuple => tuple.TRT479.GetValueOrDefault()) : 0;

            bool assertion = report.TRT478.GetValueOrDefault() + 5 < TRT479Total;

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.433084",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Net small business income is incorrect",
                    LongDescription = @"Net small business income must not be less than the sum of Share of net small business income with a tolerance of $5",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:BusinessIncomeAndExpenses/tns:SmallBusinessEntityNetA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.433084" },
                        new ProcessMessageParameter() { Name = "TRT478", Value = report.TRT478.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "Sum(TRT479)", Value = TRT479Total.ToString() }
                    }
                };

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.TRT.433084

        #region VR.ATO.TRT.433095

        /*  VR.ATO.TRT.433095
            The sum of Australian franking credits from a New Zealand franking company - Other assessable foreign source income, excluding income shown at Attributed foreign income, Share of franking credit from franked distributions, and Foreign income tax offset doesn't equal the sum of all Australian franking credits from a New Zealand company - Beneficiary share, Dividends - Franking credit, Australian franking credits from a New Zealand company - No beneficiary presently entitled, all Franking credit - Beneficiary share, Franking credit - No beneficiary presently entitled, all Foreign income tax offset - Beneficiary share, and Foreign income tax offsets - No beneficiary presently entitled

            Legacy Rule Format:
            (^TRT130 + ^TRT133 + ^TRT134 + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144) > 0 AND OutsideRange((^TRT143 + ^TRT85 + ^TRT98 + ^TRT142), (Sum(^TRT217) + ^TRT233 + Sum(^TRT221) + ^TRT237 + Sum(^TRT226) + ^TRT242), 1)

            Technical Business Rule Format:
            (^TRT130 + ^TRT133 + ^TRT134 + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144) > 0 AND OutsideRange((^TRT143 + ^TRT85 + ^TRT98 + ^TRT142), (Sum(^TRT217) + ^TRT233 + Sum(^TRT221) + ^TRT237 + Sum(^TRT226) + ^TRT242), 1)

            Data Elements:

            ^TRT130 = TRT:RP:BusinessDetails:Income:Income.Net.Amount

            ^TRT85 = TRT:RP:BusinessDetails:Income:ShareOfCredits:IncomeTax.FrankingCredits.FrankingCreditShareReceivedIndirectly.Amount

            ^TRT98 = TRT:RP:BusinessDetails:Income:Dividends:IncomeTax.FrankingCredits.Amount

            ^TRT133 = TRT:RP:BusinessDetails:Income:CapitalGains:Income.CapitalGainsNet.Amount

            ^TRT134 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:ListedCountry:Income.InternationalDealings.Attributed.Amount

            ^TRT136 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:UnlistedCountry:Income.InternationalDealings.Attributed.Amount

            ^TRT141 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:Income.Net.Amount

            ^TRT142 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:Income.InternationalDealings.TaxOffset.Amount

            ^TRT143 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount

            ^TRT144 = TRT:RP:BusinessDetails:Income:TaxLossesDeducted:IncomeTax.Deduction.TaxLossesDeducted.Amount

            ^TRT217 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount

            ^TRT221 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.FrankingCredits.ReceivedFromAustralianCompanies.Amount

            ^TRT226 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Income.InternationalDealings.TaxOffset.Amount

            ^TRT233 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount

            ^TRT237 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.FrankingCredits.ReceivedFromAustralianCompanies.Amount

            ^TRT242 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:Income.InternationalDealings.TaxOffset.Amount
            */
        public virtual void VRATOTRT433095()
        {
            decimal calculationTotal = report.TRT130.GetValueOrDefault() + report.TRT133.GetValueOrDefault() + report.TRT134.GetValueOrDefault() + report.TRT136.GetValueOrDefault() + report.TRT141.GetValueOrDefault() + report.TRT143.GetValueOrDefault() - report.TRT144.GetValueOrDefault();

            decimal TRT217Total = report.RP_StatementOfDistributionToBeneficiaryCollection != null ? report.RP_StatementOfDistributionToBeneficiaryCollection.Sum(tuple => tuple.TRT217.GetValueOrDefault()) : 0;
            decimal TRT221Total = report.RP_StatementOfDistributionToBeneficiaryCollection != null ? report.RP_StatementOfDistributionToBeneficiaryCollection.Sum(tuple => tuple.TRT221.GetValueOrDefault()) : 0;
            decimal TRT226Total = report.RP_StatementOfDistributionToBeneficiaryCollection != null ? report.RP_StatementOfDistributionToBeneficiaryCollection.Sum(tuple => tuple.TRT226.GetValueOrDefault()) : 0;

            bool assertion = calculationTotal > 0 && OutsideRange(report.TRT143.GetValueOrDefault() + report.TRT85.GetValueOrDefault() + report.TRT98.GetValueOrDefault() + report.TRT142.GetValueOrDefault(), TRT217Total + report.TRT233.GetValueOrDefault() + TRT221Total + report.TRT237.GetValueOrDefault() + TRT226Total + report.TRT242.GetValueOrDefault(), 1);

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.433095",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Distrubution of Australian franking credits from a New Zealand franking company, Franking credit, or Foreign income tax offset is incorrect",
                    LongDescription = @"The sum of Australian franking credits from a New Zealand franking company - Other assessable foreign source income, excluding income shown at Attributed foreign income, Share of franking credit from franked distributions, and Foreign income tax offset doesn't equal the sum of all Australian franking credits from a New Zealand company - Beneficiary share, Dividends - Franking credit, Australian franking credits from a New Zealand company - No beneficiary presently entitled, all Franking credit - Beneficiary share, Franking credit - No beneficiary presently entitled, all Foreign income tax offset - Beneficiary share, and Foreign income tax offsets - No beneficiary presently entitled",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:NetA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.433095" },
                        new ProcessMessageParameter() { Name = "TRT130", Value = report.TRT130.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "TRT133", Value = report.TRT133.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "TRT134", Value = report.TRT134.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "TRT136", Value = report.TRT136.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "TRT141", Value = report.TRT141.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "TRT143", Value = report.TRT143.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "TRT144", Value = report.TRT144.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "TRT85", Value = report.TRT85.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "TRT98", Value = report.TRT98.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "TRT142", Value = report.TRT142.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "Sum(TRT217)", Value = TRT217Total.ToString() },
                        new ProcessMessageParameter() { Name = "TRT233", Value = report.TRT233.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "Sum(TRT221)", Value = TRT221Total.ToString() },
                        new ProcessMessageParameter() { Name = "TRT237", Value = report.TRT237.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "Sum(TRT226)", Value = TRT226Total.ToString() },
                        new ProcessMessageParameter() { Name = "TRT242", Value = report.TRT242.GetValueOrDefault().ToString() }
                    }
                };

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.TRT.433095

        #region VR.ATO.TRT.433096

        /*  VR.ATO.TRT.433096
            When the sum of all Australian franking credits from a New Zealand company - Beneficiary share, Share of income - Primary production - Beneficiary share, Share of income - Non-primary production - Beneficiary share, Franked distributions - Beneficiary share, Capital gains - Beneficiary share, Attributed foreign income - Beneficiary share, and Other assessable foreign source income - Beneficiary share is less than 0, then Amount of capital gains on which the trustee has chosen to be assessed on behalf of beneficiaries must be provided

            Legacy Rule Format:
            (Sum(^TRT217) + Sum(^TRT218) + Sum(^TRT219) + Sum(^TRT442) + Sum(^TRT223) + Sum(^TRT224) + Sum(^TRT225)) < 0 AND ^TRT446 = NULL

            Technical Business Rule Format:
            (Sum(^TRT217) + Sum(^TRT218) + Sum(^TRT219) + Sum(^TRT442) + Sum(^TRT223) + Sum(^TRT224) + Sum(^TRT225)) < 0 AND ^TRT446 = NULL

            Data Elements:

            ^TRT446 = TRT:RP:CapitalGainsAssessed:CapitalGainsTax.TrusteeAssessedCapitalGain.Amount

            ^TRT217 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount

            ^TRT218 = TRT:RP:StatementOfDistributionToBeneficiary:Information:ShareOfIncomePrimaryProduction:Income.BeneficiaryShare.Amount

            ^TRT219 = TRT:RP:StatementOfDistributionToBeneficiary:Information:ShareOfIncomeNonPrimaryProduction:Income.BeneficiaryShare.Amount

            ^TRT223 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Income.CapitalGainsNet.Amount

            ^TRT224 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.AttributedForeignIncome.Amount

            ^TRT225 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Income.InternationalDealings.Net.Amount

            ^TRT442 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Income.DistributionFranked.Amount
            */
        public virtual void VRATOTRT433096()
        {
            decimal TRT217Total = report.RP_StatementOfDistributionToBeneficiaryCollection != null ? report.RP_StatementOfDistributionToBeneficiaryCollection.Sum(tuple => tuple.TRT217.GetValueOrDefault()) : 0;
            decimal TRT218Total = report.RP_StatementOfDistributionToBeneficiaryCollection != null ? report.RP_StatementOfDistributionToBeneficiaryCollection.Sum(tuple => tuple.TRT218.GetValueOrDefault()) : 0;
            decimal TRT219Total = report.RP_StatementOfDistributionToBeneficiaryCollection != null ? report.RP_StatementOfDistributionToBeneficiaryCollection.Sum(tuple => tuple.TRT219.GetValueOrDefault()) : 0;
            decimal TRT442Total = report.RP_StatementOfDistributionToBeneficiaryCollection != null ? report.RP_StatementOfDistributionToBeneficiaryCollection.Sum(tuple => tuple.TRT442.GetValueOrDefault()) : 0;
            decimal TRT223Total = report.RP_StatementOfDistributionToBeneficiaryCollection != null ? report.RP_StatementOfDistributionToBeneficiaryCollection.Sum(tuple => tuple.TRT223.GetValueOrDefault()) : 0;
            decimal TRT224Total = report.RP_StatementOfDistributionToBeneficiaryCollection != null ? report.RP_StatementOfDistributionToBeneficiaryCollection.Sum(tuple => tuple.TRT224.GetValueOrDefault()) : 0;
            decimal TRT225Total = report.RP_StatementOfDistributionToBeneficiaryCollection != null ? report.RP_StatementOfDistributionToBeneficiaryCollection.Sum(tuple => tuple.TRT225.GetValueOrDefault()) : 0;

            bool assertion = TRT217Total + TRT218Total + TRT219Total + TRT442Total + TRT223Total + TRT224Total + TRT225Total < 0 && report.TRT446 == null;

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.433096",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Amount of capital gains on which the trustee has chosen to be assessed on behalf of beneficiaries must be provided",
                    LongDescription = @"When the sum of all Australian franking credits from a New Zealand company - Beneficiary share, Share of income - Primary production - Beneficiary share, Share of income - Non-primary production - Beneficiary share, Franked distributions - Beneficiary share, Capital gains - Beneficiary share, Attributed foreign income - Beneficiary share, and Other assessable foreign source income - Beneficiary share is less than 0, then Amount of capital gains on which the trustee has chosen to be assessed on behalf of beneficiaries must be provided",
                    Location = "/tns:TRT/tns:RP/tns:CapitalGainsAssessed/tns:CapitalGainsTaxTrusteeAssessedCapitalGainA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.433096" },
                        new ProcessMessageParameter() { Name = "Sum(TRT217)", Value = TRT217Total.ToString() },
                        new ProcessMessageParameter() { Name = "Sum(TRT218)", Value = TRT218Total.ToString() },
                        new ProcessMessageParameter() { Name = "Sum(TRT219)", Value = TRT219Total.ToString() },
                        new ProcessMessageParameter() { Name = "Sum(TRT442)", Value = TRT442Total.ToString() },
                        new ProcessMessageParameter() { Name = "Sum(TRT223)", Value = TRT223Total.ToString() },
                        new ProcessMessageParameter() { Name = "Sum(TRT224)", Value = TRT224Total.ToString() },
                        new ProcessMessageParameter() { Name = "Sum(TRT225)", Value = TRT225Total.ToString() },
                        new ProcessMessageParameter() { Name = "TRT446", Value = "null" }
                    }
                };

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.TRT.433096

        #region VR.ATO.TRT.433098

        /*  VR.ATO.TRT.433098
            When sum of net Australian income/loss plus net capital gains, plus attributed foreign income plus other assessable foreign source income minus allowable tax losses of earlier income years deducted this year is less than or equal to zero, then the sum of all Franked distributions - Beneficiary shares plus Franked distributions - No beneficiary presently entitled must be less than or equal to zero

            Legacy Rule Format:
            ^TRT130 + ^TRT133 + ^TRT134  + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144 <= 0 AND Sum(^TRT442) + ^TRT444 > 0

            Technical Business Rule Format:
            ^TRT130 + ^TRT133 + ^TRT134  + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144 <= 0 AND Sum(^TRT442) + ^TRT444 > 0

            Data Elements:

            ^TRT444 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:Income.DistributionFranked.Amount

            ^TRT130 = TRT:RP:BusinessDetails:Income:Income.Net.Amount

            ^TRT133 = TRT:RP:BusinessDetails:Income:CapitalGains:Income.CapitalGainsNet.Amount

            ^TRT134 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:ListedCountry:Income.InternationalDealings.Attributed.Amount

            ^TRT136 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:UnlistedCountry:Income.InternationalDealings.Attributed.Amount

            ^TRT141 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:Income.Net.Amount

            ^TRT143 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount

            ^TRT144 = TRT:RP:BusinessDetails:Income:TaxLossesDeducted:IncomeTax.Deduction.TaxLossesDeducted.Amount

            ^TRT442 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Income.DistributionFranked.Amount
            */
        public virtual void VRATOTRT433098()
        {
            decimal TRT442Total = report.RP_StatementOfDistributionToBeneficiaryCollection != null ? report.RP_StatementOfDistributionToBeneficiaryCollection.Sum(tuple => tuple.TRT442.GetValueOrDefault()) : 0;
            
            bool assertion = report.TRT130.GetValueOrDefault() + report.TRT133.GetValueOrDefault() + report.TRT134.GetValueOrDefault() + report.TRT136.GetValueOrDefault() + report.TRT141.GetValueOrDefault() + report.TRT143.GetValueOrDefault() - report.TRT144.GetValueOrDefault() <= 0 && TRT442Total + report.TRT444.GetValueOrDefault() > 0;

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.433098",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Franking distributions is incorrectly distributed",
                    LongDescription = @"When sum of net Australian income/loss plus net capital gains, plus attributed foreign income plus other assessable foreign source income minus allowable tax losses of earlier income years deducted this year is less than or equal to zero, then the sum of all Franked distributions - Beneficiary shares plus Franked distributions - No beneficiary presently entitled must be less than or equal to zero",
                    Location = "/tns:TRT/tns:RP/tns:IncomeNoBeneficiaryPresentlyEntitled/tns:DistributionFrankedA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.433098" },
                        new ProcessMessageParameter() { Name = "TRT130", Value = report.TRT130.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "TRT133", Value = report.TRT133.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "TRT134", Value = report.TRT134.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "TRT136", Value = report.TRT136.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "TRT141", Value = report.TRT141.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "TRT143", Value = report.TRT143.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "TRT144", Value = report.TRT144.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "Sum(TRT442)", Value = TRT442Total.ToString() },
                        new ProcessMessageParameter() { Name = "TRT444", Value = report.TRT444.GetValueOrDefault().ToString() }
                    }
                };

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.TRT.433098
        
        #region VR.ATO.TRT.433100

        /*  VR.ATO.TRT.433100
            When sum of net Australian income/loss plus net capital gains, plus attributed foreign income plus other assessable foreign source income minus allowable tax losses of earlier income years deducted this year is less than or equal to zero, then the sum of all Capital gains - Beneficiary share, plus Net capital gain - No beneficiary presently entitled, plus Amount of capital gains on which the trustee has chosen to be assessed on behalf of beneficiaries must be less than or equal to zero

            Legacy Rule Format:
            (^TRT130 + ^TRT133 + ^TRT134 + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144) <= 0 AND (Sum(^TRT223) + ^TRT239 + ^TRT446) > 0

            Technical Business Rule Format:
            (^TRT130 + ^TRT133 + ^TRT134 + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144) <= 0 AND (Sum(^TRT223) + ^TRT239 + ^TRT446) > 0

            Data Elements:

            ^TRT239 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:Income.CapitalGainsNet.Amount

            ^TRT130 = TRT:RP:BusinessDetails:Income:Income.Net.Amount

            ^TRT133 = TRT:RP:BusinessDetails:Income:CapitalGains:Income.CapitalGainsNet.Amount

            ^TRT134 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:ListedCountry:Income.InternationalDealings.Attributed.Amount

            ^TRT136 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:UnlistedCountry:Income.InternationalDealings.Attributed.Amount

            ^TRT141 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:Income.Net.Amount

            ^TRT143 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount

            ^TRT144 = TRT:RP:BusinessDetails:Income:TaxLossesDeducted:IncomeTax.Deduction.TaxLossesDeducted.Amount

            ^TRT223 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Income.CapitalGainsNet.Amount

            ^TRT446 = TRT:RP:CapitalGainsAssessed:CapitalGainsTax.TrusteeAssessedCapitalGain.Amount
            */
        public virtual void VRATOTRT433100()
        {
            decimal TRT223Total = report.RP_StatementOfDistributionToBeneficiaryCollection != null ? report.RP_StatementOfDistributionToBeneficiaryCollection.Sum(tuple => tuple.TRT223.GetValueOrDefault()) : 0;

            bool assertion = report.TRT130.GetValueOrDefault() + report.TRT133.GetValueOrDefault() + report.TRT134.GetValueOrDefault() + report.TRT136.GetValueOrDefault() + report.TRT141.GetValueOrDefault() + report.TRT143.GetValueOrDefault() - report.TRT144.GetValueOrDefault() <= 0 && TRT223Total + report.TRT239.GetValueOrDefault() + report.TRT446.GetValueOrDefault() > 0;

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.433100",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Net capital gain is incorrectly distributed",
                    LongDescription = @"When sum of net Australian income/loss plus net capital gains, plus attributed foreign income plus other assessable foreign source income minus allowable tax losses of earlier income years deducted this year is less than or equal to zero, then the sum of all Capital gains - Beneficiary share, plus Net capital gain - No beneficiary presently entitled, plus Amount of capital gains on which the trustee has chosen to be assessed on behalf of beneficiaries must be less than or equal to zero",
                    Location = "/tns:TRT/tns:RP/tns:IncomeNoBeneficiaryPresentlyEntitled/tns:CapitalGainsNetA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.433100" },
                        new ProcessMessageParameter() { Name = "TRT130", Value = report.TRT130.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "TRT133", Value = report.TRT133.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "TRT134", Value = report.TRT134.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "TRT136", Value = report.TRT136.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "TRT141", Value = report.TRT141.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "TRT143", Value = report.TRT143.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "TRT144", Value = report.TRT144.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "Sum(TRT223)", Value = TRT223Total.ToString() },
                        new ProcessMessageParameter() { Name = "TRT239", Value = report.TRT239.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "TRT446", Value = report.TRT446.GetValueOrDefault().ToString() }
                    }
                };

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.TRT.433100

        #region VR.ATO.TRT.433105

        /*  VR.ATO.TRT.433105
            When the sum of net Australian income/loss plus net capital gains, plus attributed foreign income plus other assessable foreign source income minus allowable tax losses of earlier income years deducted this year is greater than zero, then the sum of all Attributed foreign income - Beneficiary share, plus Attributed foreign income - No beneficiary presently entitled must equal the sum of Attributed foreign income- Listed country plus Attributed foreign income- Unlisted country, with a tolerance of 1

            Legacy Rule Format:
            ^TRT130 + ^TRT133 + ^TRT134  + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144 > 0 AND OutsideRange(Sum(^TRT224) + ^TRT240, ^TRT134 + ^TRT136, 1)

            Technical Business Rule Format:
            ^TRT130 + ^TRT133 + ^TRT134  + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144 > 0 AND OutsideRange(Sum(^TRT224) + ^TRT240, ^TRT134 + ^TRT136, 1)

            Data Elements:

            ^TRT134 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:ListedCountry:Income.InternationalDealings.Attributed.Amount

            ^TRT130 = TRT:RP:BusinessDetails:Income:Income.Net.Amount

            ^TRT133 = TRT:RP:BusinessDetails:Income:CapitalGains:Income.CapitalGainsNet.Amount

            ^TRT136 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:UnlistedCountry:Income.InternationalDealings.Attributed.Amount

            ^TRT141 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:Income.Net.Amount

            ^TRT143 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount

            ^TRT144 = TRT:RP:BusinessDetails:Income:TaxLossesDeducted:IncomeTax.Deduction.TaxLossesDeducted.Amount

            ^TRT224 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.AttributedForeignIncome.Amount

            ^TRT240 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.AttributedForeignIncome.Amount
            */
        public virtual void VRATOTRT433105()
        {
            decimal TRT224Total = report.RP_StatementOfDistributionToBeneficiaryCollection != null ? report.RP_StatementOfDistributionToBeneficiaryCollection.Sum(tuple => tuple.TRT224.GetValueOrDefault()) : 0;

            bool assertion = report.TRT130.GetValueOrDefault() + report.TRT133.GetValueOrDefault() + report.TRT134.GetValueOrDefault() + report.TRT136.GetValueOrDefault() + report.TRT141.GetValueOrDefault() + report.TRT143.GetValueOrDefault() - report.TRT144.GetValueOrDefault() > 0 && OutsideRange(TRT224Total + report.TRT240.GetValueOrDefault(), report.TRT134.GetValueOrDefault() + report.TRT136.GetValueOrDefault(), 1);

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.433105",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Attributed foreign income is incorrectly distributed",
                    LongDescription = @"When the sum of net Australian income/loss plus net capital gains, plus attributed foreign income plus other assessable foreign source income minus allowable tax losses of earlier income years deducted this year is greater than zero, then the sum of all Attributed foreign income - Beneficiary share, plus Attributed foreign income - No beneficiary presently entitled must equal the sum of Attributed foreign income- Listed country plus Attributed foreign income- Unlisted country, with a tolerance of 1",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:AttributedForeignIncome/tns:ListedCountryInternationalDealingsA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.433105" },
                        new ProcessMessageParameter() { Name = "TRT130", Value = report.TRT130.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "TRT133", Value = report.TRT133.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "TRT134", Value = report.TRT134.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "TRT136", Value = report.TRT136.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "TRT141", Value = report.TRT141.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "TRT143", Value = report.TRT143.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "TRT144", Value = report.TRT144.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "Sum(TRT224)", Value = TRT224Total.ToString() },
                        new ProcessMessageParameter() { Name = "TRT240", Value = report.TRT240.GetValueOrDefault().ToString() }
                    }
                };

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.TRT.433105

        #region VR.ATO.TRT.433106

        /*  VR.ATO.TRT.433106
            When sum of net Australian income/loss plus net capital gains, plus attributed foreign income plus other assessable foreign source income minus allowable tax losses of earlier income years deducted this year is less than or equal to zero, then the sum of all Attributed foreign income - Beneficiary shares plus Attributed foreign income - No beneficiary presently entitled must be less than or equal to zero

            Legacy Rule Format:
            ^TRT130 + ^TRT133 + ^TRT134  + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144 <= 0 AND Sum(^TRT224) + ^TRT240 > 0

            Technical Business Rule Format:
            ^TRT130 + ^TRT133 + ^TRT134  + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144 <= 0 AND Sum(^TRT224) + ^TRT240 > 0

            Data Elements:

            ^TRT240 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:IncomeTax.AttributedForeignIncome.Amount

            ^TRT130 = TRT:RP:BusinessDetails:Income:Income.Net.Amount

            ^TRT133 = TRT:RP:BusinessDetails:Income:CapitalGains:Income.CapitalGainsNet.Amount

            ^TRT134 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:ListedCountry:Income.InternationalDealings.Attributed.Amount

            ^TRT136 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:UnlistedCountry:Income.InternationalDealings.Attributed.Amount

            ^TRT141 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:Income.Net.Amount

            ^TRT143 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount

            ^TRT144 = TRT:RP:BusinessDetails:Income:TaxLossesDeducted:IncomeTax.Deduction.TaxLossesDeducted.Amount

            ^TRT224 = TRT:RP:StatementOfDistributionToBeneficiary:Information:IncomeTax.AttributedForeignIncome.Amount
            */
        public virtual void VRATOTRT433106()
        {
            decimal TRT224Total = report.RP_StatementOfDistributionToBeneficiaryCollection != null ? report.RP_StatementOfDistributionToBeneficiaryCollection.Sum(tuple => tuple.TRT224.GetValueOrDefault()) : 0;

            bool assertion = report.TRT130.GetValueOrDefault() + report.TRT133.GetValueOrDefault() + report.TRT134.GetValueOrDefault() + report.TRT136.GetValueOrDefault() + report.TRT141.GetValueOrDefault() + report.TRT143.GetValueOrDefault() - report.TRT144.GetValueOrDefault() <= 0 && TRT224Total + report.TRT240.GetValueOrDefault() > 0;

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.433106",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Attributed foreign income is incorrectly distributed",
                    LongDescription = @"When sum of net Australian income/loss plus net capital gains, plus attributed foreign income plus other assessable foreign source income minus allowable tax losses of earlier income years deducted this year is less than or equal to zero, then the sum of all Attributed foreign income - Beneficiary shares plus Attributed foreign income - No beneficiary presently entitled must be less than or equal to zero",
                    Location = "/tns:TRT/tns:RP/tns:IncomeNoBeneficiaryPresentlyEntitled/tns:IncomeTaxAttributedForeignIncomeA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.433106" },
                        new ProcessMessageParameter() { Name = "TRT130", Value = report.TRT130.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "TRT133", Value = report.TRT133.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "TRT134", Value = report.TRT134.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "TRT136", Value = report.TRT136.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "TRT141", Value = report.TRT141.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "TRT143", Value = report.TRT143.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "TRT144", Value = report.TRT144.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "Sum(TRT224)", Value = TRT224Total.ToString() },
                        new ProcessMessageParameter() { Name = "TRT240", Value = report.TRT240.GetValueOrDefault().ToString() }
                    }
                };

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.TRT.433106

        #region VR.ATO.TRT.433107

        /*  VR.ATO.TRT.433107
            When sum of net Australian income/loss plus net capital gains, plus attributed foreign income plus other assessable foreign source income minus allowable tax losses of earlier income years deducted this year is less than or equal to zero, then the sum of all Other assessable foreign source income - Beneficiary shares plus Other assessable foreign source income - No beneficiary presently entitled must be less than or equal to zero

            Legacy Rule Format:
            ^TRT130 + ^TRT133 + ^TRT134  + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144 <= 0 AND Sum(^TRT225) + ^TRT241 > 0

            Technical Business Rule Format:
            ^TRT130 + ^TRT133 + ^TRT134  + ^TRT136 + ^TRT141 + ^TRT143 - ^TRT144 <= 0 AND Sum(^TRT225) + ^TRT241 > 0

            Data Elements:

            ^TRT241 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:Income.InternationalDealings.Net.Amount

            ^TRT130 = TRT:RP:BusinessDetails:Income:Income.Net.Amount

            ^TRT133 = TRT:RP:BusinessDetails:Income:CapitalGains:Income.CapitalGainsNet.Amount

            ^TRT134 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:ListedCountry:Income.InternationalDealings.Attributed.Amount

            ^TRT136 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:UnlistedCountry:Income.InternationalDealings.Attributed.Amount

            ^TRT141 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:Income.Net.Amount

            ^TRT143 = TRT:RP:BusinessDetails:Income:AttributedForeignIncome:OtherAssessableForeignIncome:IncomeTax.FrankingCredits.ReceivedFromNewZealandCompanies.Amount

            ^TRT144 = TRT:RP:BusinessDetails:Income:TaxLossesDeducted:IncomeTax.Deduction.TaxLossesDeducted.Amount

            ^TRT225 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Income.InternationalDealings.Net.Amount
            */
        public virtual void VRATOTRT433107()
        {
            decimal TRT225Total = report.RP_StatementOfDistributionToBeneficiaryCollection != null ? report.RP_StatementOfDistributionToBeneficiaryCollection.Sum(tuple => tuple.TRT225.GetValueOrDefault()) : 0;

            bool assertion = report.TRT130.GetValueOrDefault() + report.TRT133.GetValueOrDefault() + report.TRT134.GetValueOrDefault() + report.TRT136.GetValueOrDefault() + report.TRT141.GetValueOrDefault() + report.TRT143.GetValueOrDefault() - report.TRT144.GetValueOrDefault() <= 0 && TRT225Total + report.TRT241.GetValueOrDefault() > 0;

            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.433107",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Other assessable foreign source income is incorrectly distributed",
                    LongDescription = @"When sum of net Australian income/loss plus net capital gains, plus attributed foreign income plus other assessable foreign source income minus allowable tax losses of earlier income years deducted this year is less than or equal to zero, then the sum of all Other assessable foreign source income - Beneficiary shares plus Other assessable foreign source income - No beneficiary presently entitled must be less than or equal to zero",
                    Location = "/tns:TRT/tns:RP/tns:IncomeNoBeneficiaryPresentlyEntitled/tns:InternationalDealingsNetA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.433107" },
                        new ProcessMessageParameter() { Name = "TRT130", Value = report.TRT130.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "TRT133", Value = report.TRT133.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "TRT134", Value = report.TRT134.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "TRT136", Value = report.TRT136.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "TRT141", Value = report.TRT141.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "TRT143", Value = report.TRT143.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "TRT144", Value = report.TRT144.GetValueOrDefault().ToString() },
                        new ProcessMessageParameter() { Name = "Sum(TRT225)", Value = TRT225Total.ToString() },
                        new ProcessMessageParameter() { Name = "TRT241", Value = report.TRT241.GetValueOrDefault().ToString() }
                    }
                };

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.TRT.433107

        #region VR.ATO.TRT.W00009

        /*  VR.ATO.TRT.W00009
            The total amount at NCMI from primary production (beneficiary and no beneficiary presently entitled) cannot exceed the sum of NCMI from primary production - Business income, NCMI from primary production - Distribution from partnerships and NCMI from primary production - Share of net income from trusts.

            Legacy Rule Format:
            (Sum(^TRT537) + ^TRT543) > (^TRT523 + ^TRT527 + ^TRT529)

            Technical Business Rule Format:
            (Sum(^TRT537) + ^TRT543) > (^TRT523 + ^TRT527 + ^TRT529)

            Data Elements:
    
            ^TRT523 = TRT:RP:BusinessDetails:Income:NCMIBusinessIncome:NCMIBusinessIncomePrimaryProduction:Income.NonConcessionalManagedInvestmentTrust.Amount
    
            ^TRT527 = TRT:RP:BusinessDetails:Income:NCMIPartnershipsAndTrusts:NCMIPartnershipsAndTrustsPrimaryProduction:Income.NonConcessionalManagedInvestmentTrust.Amount
    
            ^TRT529 = TRT:RP:BusinessDetails:Income:NCMIPartnershipsAndTrusts:NCMIPartnershipsAndTrustsPrimaryProduction:PrimaryProductionShareOfNetIncome:Income.NonConcessionalManagedInvestmentTrust.Amount
    
            ^TRT537 = TRT:RP:StatementOfDistributionToBeneficiary:Information:ShareOfIncomePrimaryProduction:Income.NonConcessionalManagedInvestmentTrust.Amount
    
            ^TRT543 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:PrimaryProductionNoBeneficiaryEntitled:Income.NonConcessionalManagedInvestmentTrust.Amount
            */
        public virtual void VRATOTRTW00009()
        {
            decimal? sumTRT537 = report.RP_StatementOfDistributionToBeneficiaryCollection == null ? 0 : report.RP_StatementOfDistributionToBeneficiaryCollection.Sum(f => f.TRT537.GetValueOrDefault());

            bool assertion = (sumTRT537 + report.TRT543.GetValueOrDefault() )> (report.TRT523.GetValueOrDefault() + report.TRT527.GetValueOrDefault() + report.TRT529.GetValueOrDefault());
            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.W00009",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Ensure the total amount at NCMI from primary production (beneficiary and no beneficiary presently entitled) does not exceed net NCMI from primary production.",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:NCMIBusinessIncome/tns:NCMIBusinessIncomePrimaryProduction/tns:NonConcessionalManagedInvestmentTrustA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.W00009" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT537", Value = GetValueOrEmpty(sumTRT537) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT543", Value = GetValueOrEmpty(report.TRT543) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT523", Value = GetValueOrEmpty(report.TRT523) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT527", Value = GetValueOrEmpty(report.TRT527) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT529", Value = GetValueOrEmpty(report.TRT529) });

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.TRT.W00009

        #region VR.ATO.TRT.W00010

        /*  VR.ATO.TRT.W00010
            The total amount at Excluded from NCMI from primary production (beneficiary and no beneficiary presently entitled) cannot exceed the sum of excluded from NCMI from primary production - Business income, excluded from NCMI from primary production - Distribution from partnerships and excluded from NCMI from primary production - Share of net income from trusts.

            Legacy Rule Format:
            (Sum(^TRT538) + ^TRT544) > (^TRT524 + ^TRT528 + ^TRT530)

            Technical Business Rule Format:
            (Sum(^TRT538) + ^TRT544) > (^TRT524 + ^TRT528 + ^TRT530)

            Data Elements:
    
            ^TRT524 = TRT:RP:BusinessDetails:Income:NCMIBusinessIncome:NCMIBusinessIncomePrimaryProduction:Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount
    
            ^TRT528 = TRT:RP:BusinessDetails:Income:NCMIPartnershipsAndTrusts:NCMIPartnershipsAndTrustsPrimaryProduction:Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount
    
            ^TRT530 = TRT:RP:BusinessDetails:Income:NCMIPartnershipsAndTrusts:NCMIPartnershipsAndTrustsPrimaryProduction:PrimaryProductionShareOfNetIncome:Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount
    
            ^TRT538 = TRT:RP:StatementOfDistributionToBeneficiary:Information:ShareOfIncomePrimaryProduction:Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount
    
            ^TRT544 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:PrimaryProductionNoBeneficiaryEntitled:Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount
            */
        public virtual void VRATOTRTW00010()
        {
            decimal? sumTRT538 = report.RP_StatementOfDistributionToBeneficiaryCollection == null ? 0 : report.RP_StatementOfDistributionToBeneficiaryCollection.Sum(f => f.TRT538.GetValueOrDefault());

            bool assertion = (sumTRT538 + report.TRT544.GetValueOrDefault()) > (report.TRT524.GetValueOrDefault() + report.TRT528.GetValueOrDefault() + report.TRT530.GetValueOrDefault());
            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.W00010",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Ensure the total amount at Excluded from NCMI from primary production (beneficiary and no beneficiary presently entitled) does not exceed net Excluded from NCMI from primary production.",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:NCMIBusinessIncome/tns:NCMIBusinessIncomePrimaryProduction/tns:ExcludedFromNonConcessionalManagedInvestmentTrustA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.W00010" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT538", Value = GetValueOrEmpty(sumTRT538) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT543", Value = GetValueOrEmpty(report.TRT544) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT524", Value = GetValueOrEmpty(report.TRT524) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT528", Value = GetValueOrEmpty(report.TRT528) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT530", Value = GetValueOrEmpty(report.TRT530) });

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.TRT.W00010

        #region VR.ATO.TRT.W00011

        /*  VR.ATO.TRT.W00011
            The total amount at NCMI from non-primary production (beneficiary and no beneficiary presently entitled) cannot exceed the sum of NCMI from non-primary production - Business income, NCMI from non-primary production - Distribution from partnerships less foreign income and NCMI from non-primary production - Share of net income from trusts less net capital gain, foreign income and franked distributions.

            Legacy Rule Format:
            (Sum(^TRT539) + ^TRT545) > (^TRT525 + ^TRT531 + ^TRT533)

            Technical Business Rule Format:
            (Sum(^TRT539) + ^TRT545) > (^TRT525 + ^TRT531 + ^TRT533)

            Data Elements:
    
            ^TRT525 = TRT:RP:BusinessDetails:Income:NCMIBusinessIncome:NCMIBusinessIncomeNonPrimaryProduction:Income.NonConcessionalManagedInvestmentTrust.Amount
    
            ^TRT531 = TRT:RP:BusinessDetails:Income:NCMIPartnershipsAndTrusts:NCMIPartnershipsAndTrustsNonPrimaryProduction:Income.NonConcessionalManagedInvestmentTrust.Amount
    
            ^TRT533 = TRT:RP:BusinessDetails:Income:NCMIPartnershipsAndTrusts:NCMIPartnershipsAndTrustsNonPrimaryProduction:NonPrimaryProductionShareOfNetIncome:Income.NonConcessionalManagedInvestmentTrust.Amount
    
            ^TRT539 = TRT:RP:StatementOfDistributionToBeneficiary:Information:ShareOfIncomeNonPrimaryProduction:Income.NonConcessionalManagedInvestmentTrust.Amount
    
            ^TRT545 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:NonPrimaryProductionNoBeneficiaryEntitled:Income.NonConcessionalManagedInvestmentTrust.Amount
            */
        public virtual void VRATOTRTW00011()
        {
            decimal? sumTRT539 = report.RP_StatementOfDistributionToBeneficiaryCollection == null ? 0 : report.RP_StatementOfDistributionToBeneficiaryCollection.Sum(f => f.TRT539.GetValueOrDefault());

            bool assertion = (sumTRT539 + report.TRT545.GetValueOrDefault()) > (report.TRT525.GetValueOrDefault() + report.TRT531.GetValueOrDefault() + report.TRT533.GetValueOrDefault());
            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.W00011",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Ensure the total amount at NCMI from non-primary production (beneficiary and no beneficiary presently entitled) does not exceed net NCMI from non-primary production.",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:NCMIBusinessIncome/tns:NCMIBusinessIncomeNonPrimaryProduction/tns:NonConcessionalManagedInvestmentTrustA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.W00011" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT539", Value = GetValueOrEmpty(sumTRT539) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT545", Value = GetValueOrEmpty(report.TRT545) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT525", Value = GetValueOrEmpty(report.TRT525) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT531", Value = GetValueOrEmpty(report.TRT531) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT533", Value = GetValueOrEmpty(report.TRT533) });

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.TRT.W00011

        #region VR.ATO.TRT.W00012

        /*  VR.ATO.TRT.W00012
            The total amount at Excluded from NCMI from non-primary production (beneficiary and no beneficiary presently entitled) cannot exceed the sum of excluded from NCMI from non-primary production - Business income, excluded from NCMI from non-primary production - Distribution from partnerships less foreign income and excluded from NCMI from non-primary production - Share of net income from trusts less net capital gain, foreign income and franked distributions.

            Legacy Rule Format:
            (Sum(^TRT540) + ^TRT546) > (^TRT526 + ^TRT532 + ^TRT534)

            Technical Business Rule Format:
            (Sum(^TRT540) + ^TRT546) > (^TRT526 + ^TRT532 + ^TRT534)

            Data Elements:
    
            ^TRT526 = TRT:RP:BusinessDetails:Income:NCMIBusinessIncome:NCMIBusinessIncomeNonPrimaryProduction:Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount
    
            ^TRT532 = TRT:RP:BusinessDetails:Income:NCMIPartnershipsAndTrusts:NCMIPartnershipsAndTrustsNonPrimaryProduction:Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount
    
            ^TRT534 = TRT:RP:BusinessDetails:Income:NCMIPartnershipsAndTrusts:NCMIPartnershipsAndTrustsNonPrimaryProduction:NonPrimaryProductionShareOfNetIncome:Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount
    
            ^TRT540 = TRT:RP:StatementOfDistributionToBeneficiary:Information:ShareOfIncomeNonPrimaryProduction:Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount
    
            ^TRT546 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:NonPrimaryProductionNoBeneficiaryEntitled:Income.ExcludedFromNonConcessionalManagedInvestmentTrust.Amount
            */
        public virtual void VRATOTRTW00012()
        {
            decimal? sumTRT540 = report.RP_StatementOfDistributionToBeneficiaryCollection == null ? 0 : report.RP_StatementOfDistributionToBeneficiaryCollection.Sum(f => f.TRT540.GetValueOrDefault());

            bool assertion = (sumTRT540.GetValueOrDefault() + report.TRT546.GetValueOrDefault()) > (report.TRT526.GetValueOrDefault() + report.TRT532.GetValueOrDefault() + report.TRT534.GetValueOrDefault());
            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.W00012",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Ensure the total amount at Excluded from NCMI from non-primary production (beneficiary and no beneficiary presently entitled) does not exceed net Excluded from NCMI from non-primary production.",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:NCMIBusinessIncome/tns:NCMIBusinessIncomeNonPrimaryProduction/tns:ExcludedFromNonConcessionalManagedInvestmentTrustA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.W00012" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT540", Value = GetValueOrEmpty(sumTRT540) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT546", Value = GetValueOrEmpty(report.TRT546) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT526", Value = GetValueOrEmpty(report.TRT526) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT532", Value = GetValueOrEmpty(report.TRT532) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT534", Value = GetValueOrEmpty(report.TRT534) });

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.TRT.W00012

        #region VR.ATO.TRT.W00013

        /*  VR.ATO.TRT.W00013
            The total amount at capital gains NCMI (beneficiary and no beneficiary presently entitled) cannot exceed net capital gains NCMI.

            Legacy Rule Format:
            (Sum(^TRT541) + ^TRT547) > ^TRT535

            Technical Business Rule Format:
            (Sum(^TRT541) + ^TRT547) > ^TRT535

            Data Elements:
    
            ^TRT535 = TRT:RP:BusinessDetails:Income:NCMICapitalGains:Capital.Gains.NonConcessionalManagedInvestmentTrustIncome.Amount
    
            ^TRT541 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Capital.Gains.NonConcessionalManagedInvestmentTrustIncome.Amount
    
            ^TRT547 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:Capital.Gains.NonConcessionalManagedInvestmentTrustIncome.Amount
            */
        public virtual void VRATOTRTW00013()
        {
            decimal? sumTRT541 = report.RP_StatementOfDistributionToBeneficiaryCollection == null ? 0 : report.RP_StatementOfDistributionToBeneficiaryCollection.Sum(f => f.TRT541.GetValueOrDefault());

            bool assertion = (sumTRT541.GetValueOrDefault() + report.TRT547.GetValueOrDefault()) > report.TRT535.GetValueOrDefault();
            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.W00013",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Ensure the total amount at capital gains NCMI (beneficiary and no beneficiary presently entitled) does not exceed net capital gains NCMI.",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:NCMICapitalGains/tns:NonConcessionalManagedInvestmentTrustIncomeA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.W00013" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT541", Value = GetValueOrEmpty(sumTRT541) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT547", Value = GetValueOrEmpty(report.TRT547) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT535", Value = GetValueOrEmpty(report.TRT536) });

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.TRT.W00013

        #region VR.ATO.TRT.W00014

        /*  VR.ATO.TRT.W00014
            The total amount at capital gains Excluded from NCMI (beneficiary and no beneficiary presently entitled) cannot exceed net capital gains Excluded from NCMI.

            Legacy Rule Format:
            (Sum(^TRT542) + ^TRT548) > ^TRT536

            Technical Business Rule Format:
            (Sum(^TRT542) + ^TRT548) > ^TRT536

            Data Elements:
    
            ^TRT536 = TRT:RP:BusinessDetails:Income:NCMICapitalGains:Capital.Gains.ExcludedFromNonConcessionalManagedInvestmentTrustIncome.Amount
    
            ^TRT542 = TRT:RP:StatementOfDistributionToBeneficiary:Information:Capital.Gains.ExcludedFromNonConcessionalManagedInvestmentTrustIncome.Amount
    
            ^TRT548 = TRT:RP:IncomeNoBeneficiaryPresentlyEntitled:Capital.Gains.ExcludedFromNonConcessionalManagedInvestmentTrustIncome.Amount
            */
        public virtual void VRATOTRTW00014()
        {
            decimal? sumTRT542 = report.RP_StatementOfDistributionToBeneficiaryCollection == null ? 0 : report.RP_StatementOfDistributionToBeneficiaryCollection.Sum(f => f.TRT542.GetValueOrDefault());

            bool assertion = (sumTRT542.GetValueOrDefault() + report.TRT548.GetValueOrDefault()) > report.TRT536.GetValueOrDefault();
            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRT.W00014",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Ensure the total amount at capital gains Excluded from NCMI (beneficiary and no beneficiary presently entitled) does not exceed net capital gains Excluded from NCMI.",
                    Location = "/tns:TRT/tns:RP/tns:BusinessDetails/tns:Income/tns:NCMICapitalGains/tns:ExcludedFromNonConcessionalManagedInvestmentTrustIncomeA",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRT.W00014" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT542", Value = GetValueOrEmpty(sumTRT542) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT548", Value = GetValueOrEmpty(report.TRT548) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "TRT536", Value = GetValueOrEmpty(report.TRT536) });

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.TRT.W00014

        #endregion Manual Rules - Rules marked for manual coding
    }
}