using Ato.EN.IntegrationServices.CodeGenerationCGTS;
using Ato.EN.IntegrationServices.CodeGenerationIDS;
using Ato.EN.IntegrationServices.CodeGenerationPSS;
using Ato.EN.IntegrationServices.CodeGenerationRPTTAXPOS;
using Ato.EN.IntegrationServices.CodeGenerationRS;
using Ato.EN.IntegrationServices.CodeGenerationTRTAMI;
using Ato.EN.IntegrationServices.CodeGenerationTRTAMIS;
using DataContracts;
using System.Collections.Generic;
using System.Linq;
using System.Runtime.CompilerServices;
using System.Text.RegularExpressions;
using VaTS;

namespace Ato.CD.Inbound.TRTAMI202602
{
    public class CrossFormValidator
    {
        private IEnumerable<CGTS2018> CGTSChildDocuments { get; set; }

        private IEnumerable<IDS2025> IDSChildDocuments { get; set; }

        private IEnumerable<PSS2018> PSSChildDocuments { get; set; }

        private IEnumerable<RS2018> RSChildDocuments { get; set; }

        private IEnumerable<TRTAMIS2025> TRTAMISChildDocuments { get; set; }

        private IEnumerable<RPTTAXPOS2019> RPTTAXPOSChildDocuments { get; set; }

        public List<ProcessMessageDocument> Response { get; private set; }

        private TRTAMI2026 ParentDocument { get; set; }

        public CrossFormValidator(TRTAMI2026 parent, BusinessDocumentCollection childDocuments)
        {
            Response = new List<ProcessMessageDocument>();

            ParentDocument = parent;

            BusinessDocumentCollection ChildDocuments = childDocuments ?? new BusinessDocumentCollection();

            CGTSChildDocuments = ChildDocuments
                    .Where(s => s.DocumentName.Equals(Schedule.CGTS.ToString()))
                    .Select(p => (CGTS2018)p.ConsumedReport);

            IDSChildDocuments = ChildDocuments
                    .Where(s => s.DocumentName.Equals(Schedule.IDS.ToString()))
                    .Select(p => (IDS2025)p.ConsumedReport);

            PSSChildDocuments = ChildDocuments
                    .Where(s => s.DocumentName.Equals(Schedule.PSS.ToString()))
                    .Select(p => (PSS2018)p.ConsumedReport);

            RSChildDocuments = ChildDocuments
                    .Where(s => s.DocumentName.Equals(Schedule.RS.ToString()))
                    .Select(p => (RS2018)p.ConsumedReport);

            TRTAMISChildDocuments = ChildDocuments
                    .Where(s => s.DocumentName.Equals(Schedule.TRTAMIS.ToString()))
                    .Select(p => (TRTAMIS2025)p.ConsumedReport);

            RPTTAXPOSChildDocuments = ChildDocuments
                    .Where(s => s.DocumentName.Equals(Schedule.RPTTAXPOS.ToString()))
                    .Select(p => (RPTTAXPOS2019)p.ConsumedReport);
        }

        public List<ProcessMessageDocument> ValidateCrossFormRules()
        {
            VRATOTRTAMI000151();
            VRATOTRTAMI000152();
            VRATOTRTAMI000153();
            VRATOTRTAMI000154();
            VRATOTRTAMI000174();
            VRATOTRTAMI000199();
            VRATOTRTAMI000201();
            VRATOTRTAMI000203();
            VRATOTRTAMI000205();
            VRATOTRTAMI000234();
            VRATOTRTAMI000235();
            VRATOTRTAMI000236();
            VRATOTRTAMI000237();
            VRATOTRTAMI000240();
            VRATOTRTAMI000243();
            VRATOTRTAMI000244();
            VRATOTRTAMI000245();
            VRATOTRTAMI000246();
            VRATOTRTAMI000314();
            VRATOTRTAMI000315();
            VRATOTRTAMI000330();
            VRATOTRTAMI000401();
            VRATOTRTAMI000420();
            VRATOTRTAMI000425();
            VRATOTRTAMI000426();
            VRATOTRTAMI000427();

            return Response;
        }

        #region Validation Rules

        #region VR.ATO.TRTAMI.000151

        /*  VR.ATO.TRTAMI.000151
        Ensure an 'International dealings schedule' (IDS) is completed when an Interest expenses overseas amount greater than zero is provided

        Technical Business Rule Format:
        (^TRTAMI49 > 0) AND (CountDocument('IDS') = 0)

        Data Elements:
        ^TRTAMI49 = TRTAMI:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.StateOrTerritory.Code
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000151()
        {
            bool assertion = ParentDocument.TRTAMI49 > 0 && !IDSChildDocuments.Any();

            if (assertion)
            {
                Response.Add(new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000151",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"International Dealings Schedule must be attached",
                    LongDescription = @"An International Dealings Schedule (IDS) must be attached when an Interest Expenses Overseas amount above zero is provided",
                    Location = "/tns:TRTAMI/tns:RP/tns:TrustDetails/tns:ExpenseInterestA",
                    Parameters = new ProcessMessageParameters() { 
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMI.000151" },
                        new ProcessMessageParameter() { Name = "TRTAMI49", Value = TRTAMI2026Validator.GetValueOrEmpty(ParentDocument.TRTAMI49) }
                    },
                });
            }
        }

        #endregion

        #region VR.ATO.TRTAMI.000152

        /*  VR.ATO.TRTAMI.000152
        When a Royalties deduction amount is present on the International Dealings Schedule (IDS), an amount above zero must be provided for Royalty Expenses Overseas

        Technical Business Rule Format:
        (InSet(^TRTAMI50, 'NULL','0')) AND (CountDocument('IDS') = 1) AND (^IDS22 > 0)
        
        Data Elements:
        ^TRTAMI50 = TRTAMI:RP:TrustDetails: Expense.Royalties.Amount

        ^IDS22 = IDS:RP:Royalties:Expense.DeductionsTotal.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000152()
        {
            bool assertion = (ParentDocument.TRTAMI50 == 0 || ParentDocument.TRTAMI50 == null) &&
                IDSChildDocuments.Count() == 1 &&
                IDSChildDocuments.All(i => i.IDS22 > 0);

            if (assertion)
            {
                Response.Add(new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000152",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Royalty expenses overseas required",
                    LongDescription = @"When a Royalties deduction amount above zero is present on the International Dealings Schedule (IDS), a Royalty Expenses Overseas amount above zero must also be provided",
                    Location = "/tns:TRTAMI/tns:RP/tns:TrustDetails/tns:ExpenseRoyaltiesA",
                    Parameters = new ProcessMessageParameters() {
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMI.000152" },
                        new ProcessMessageParameter() { Name = "TRTAMI50", Value = TRTAMI2026Validator.GetValueOrEmpty(ParentDocument.TRTAMI50) },
                        new ProcessMessageParameter() { Name = "IDS22", Value = TRTAMI2026Validator.GetValueOrEmpty(IDSChildDocuments.FirstOrDefault()?.IDS22) },
                    },
                });
            }
        }

        #endregion

        #region VR.ATO.TRTAMI.000153

        /*  VR.ATO.TRTAMI.000153
        When a Royalties deduction amount is present on the International Dealings Schedule (IDS), it must not exceed the Royalty Expenses Overseas amount

        Technical Business Rule Format:
        (^TRTAMI50 > 0) AND (CountDocument('IDS') = 1) AND (^TRTAMI50 < ^IDS22)

        Data Elements:
        ^TRTAMI50 = TRTAMI:RP:TrustDetails: Expense.Royalties.Amount

        ^IDS22 = IDS:RP:Royalties:Expense.DeductionsTotal.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000153()
        {
            IEnumerable<decimal?> ids22 = IDSChildDocuments.Where(c => c.IDS22 != null).Select(c => c.IDS22);

            bool assertion = ParentDocument.TRTAMI50 > 0 &&
                IDSChildDocuments.Count() == 1 &&
                ParentDocument.TRTAMI50 < ids22.FirstOrDefault();

            if (assertion)
            {
                Response.Add(new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000153",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Royalties deductions incorrect",
                    LongDescription = @"When a Royalties deduction amount is present on the International Dealings Schedule (IDS), it must not exceed the Royalty Expenses Overseas amount",
                    Location = "/tns:TRTAMI/tns:RP/tns:TrustDetails/tns:ExpenseRoyaltiesA",
                    Parameters = new ProcessMessageParameters() {
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMI.000153" },
                        new ProcessMessageParameter() { Name = "TRTAMI50", Value = TRTAMI2026Validator.GetValueOrEmpty(ParentDocument.TRTAMI50) },
                        new ProcessMessageParameter() { Name = "IDS22", Value = TRTAMI2026Validator.GetValueOrEmpty(ids22.FirstOrDefault()) }
                    },
                });
            }
        }

        #endregion

        #region VR.ATO.TRTAMI.000154

        /*  VR.ATO.TRTAMI.000154
        Ensure an 'International dealings schedule' (IDS) is completed when a Royalty expenses overseas amount greater than zero is provided

        Technical Business Rule Format:
        (^TRTAMI50 > 0) AND (CountDocument('IDS') = 0)
        
        Data Elements:
        ^TRTAMI50 = TRTAMI:RP:TrustDetails: Expense.Royalties.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000154()
        {
            bool assertion = ParentDocument.TRTAMI50 > 0 && !IDSChildDocuments.Any();

            if (assertion)
            {
                Response.Add(new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000154",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"International Dealings Schedule must be attached",
                    LongDescription = @"An International Dealings Schedule (IDS) must be attached when a Royalty Expenses Overseas amount above zero is provided",
                    Location = "/tns:TRTAMI/tns:RP/tns:TrustDetails/tns:ExpenseRoyaltiesA",
                    Parameters = new ProcessMessageParameters() {
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMI.000154" },
                        new ProcessMessageParameter() { Name = "TRTAMI50", Value = TRTAMI2026Validator.GetValueOrEmpty(ParentDocument.TRTAMI50) }
                    },
                });
            }
        }

        #endregion

        #region VR.ATO.TRTAMI.000174

        /*  VR.ATO.TRTAMI.000174
        An AMIT schedule must be submitted with an AMIT return form. Some AMITs will need to submit multiple AMIT schedules with their AMIT return form.

        Technical Business Rule Format:
        CountDocument('TRTAMIS') = 0
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000174()
        {
            bool assertion = !TRTAMISChildDocuments.Any();

            if (assertion)
            {
                Response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.TRTAMI.000174",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"At least one Trust Attribution Managed Investments Schedule must be present",
                    Location = "/tns:TRTAMI/tns:RP",
                    Parameters = new ProcessMessageParameters { new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMI.000174" } }
                });
            }
        }

        #endregion

        #region VR.ATO.TRTAMI.000199

        /*  VR.ATO.TRTAMI.000199
        When the value of dealings with international related parties was not above $2 million, thin capitalisation or debt deduction creation rules did not affect you, and no amounts have been provided at Interest expenses overseas and Royalty expenses overseas, an International Dealings schedule must not be attached.

        Technical Business Rule Format:
        (^TRTAMI47 = FALSE AND ^TRTAMI48 = FALSE) AND (InSet(^TRTAMI49, "NULL","0') AND InSet(^TRTAMI50,"NULL","0')) AND (CountDocument('IDS') = 1)
        
        Data Elements:
        ^TRTAMI47 = TRTAMI:RP:TrustDetails:InternationalDealings.RelatedPartiesTransactionsExcessAggregateValue.Indicator

        ^TRTAMI48 = TRTAMI:RP:TrustDetails:Liabilities.ThinCapitalisation.ProvisionsApplied.Indicator

        ^TRTAMI49 = TRTAMI:RP:TrustDetails:Expense.Interest.Amount

        ^TRTAMI50 = TRTAMI:RP:TrustDetails:Expense.Royalties.Amount
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000199()
        {
            bool assertion = ParentDocument.TRTAMI47 == false && ParentDocument.TRTAMI48 == false
                    && (ParentDocument.TRTAMI49 == null || ParentDocument.TRTAMI49 == 0)
                    && (ParentDocument.TRTAMI50 == null || ParentDocument.TRTAMI50 == 0)
                    && IDSChildDocuments.Count() == 1;

            if (assertion)
            {
                Response.Add(new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.010199",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"International Dealings Schedule is not required",
                    LongDescription = @"When the value of dealings with international related parties was not above $2 million, thin capitalisation or debt deduction creation rules did not affect you, and no amounts have been provided at Interest expenses overseas and Royalty expenses overseas, an International Dealings schedule must not be attached",
                    Location = "/tns:TRTAMI/tns:RP/tns:TrustDetails/tns:InternationalDealingsRelatedPartiesTransactionsExcessAggregateValueI",
                    Parameters = new ProcessMessageParameters() {
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMI.000199" },
                        new ProcessMessageParameter() { Name = "TRTAMI47", Value = TRTAMI2026Validator.GetValueOrEmpty(ParentDocument.TRTAMI47) },
                        new ProcessMessageParameter() { Name = "TRTAMI48", Value = TRTAMI2026Validator.GetValueOrEmpty(ParentDocument.TRTAMI48) },
                        new ProcessMessageParameter() { Name = "TRTAMI49", Value = TRTAMI2026Validator.GetValueOrEmpty(ParentDocument.TRTAMI49) },
                        new ProcessMessageParameter() { Name = "TRTAMI50", Value = TRTAMI2026Validator.GetValueOrEmpty(ParentDocument.TRTAMI50) },
                    },
                });
            }
        }

        #endregion

        #region VR.ATO.TRTAMI.000201

        /*  VR.ATO.TRTAMI.000201
        The number of International Dealings Schedules must not exceed 1

        Technical Business Rule Format:
        CountDocument('IDS') > 1
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000201()
        {
            bool assertion = IDSChildDocuments.Count() > 1;

            if (assertion)
            {
                Response.Add(new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000201",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The number of International Dealings Schedules must not exceed 1",
                    Location = "/tns:TRTAMI/tns:RP",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMI.000201" } },
                });
            }
        }

        #endregion

        #region  VR.ATO.TRTAMI.000203

        /*  VR.ATO.TRTAMI.000203
        The number of PAYG Payment Summary Schedules must not exceed 1

        Technical Business Rule Format:
        CountDocument('PSS') > 1
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000203()
        {
            bool assertion = PSSChildDocuments.Count() > 1;

            if (assertion)
            {
                Response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.TRTAMI.000203",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The number of PAYG Payment Summary Schedules must not exceed 1",
                    Location = "/tns:TRTAMI/tns:RP",
                    Parameters = new ProcessMessageParameters { new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMI.000203" } }
                });
            }
        }
        #endregion

        #region VR.ATO.TRTAMI.000205

        /*  VR.ATO.TRTAMI.000205
        When the response to 'Did you have dealings with international related parties located in specified countries?' on the International Dealings Schedule (IDS) is 'True' (yes), the Response at either 'Funds or property, whether directly or indirectly, have been sent to, or received from, any of the specified countries', or 'Trustee has the ability or expectation to control, whether directly or indirectly, the disposition of any funds, property, assets or investments located in, or located elsewhere but controlled or managed from any of the specified countries', must also be 'True' (yes)

        Technical Business Rule Format:
        ^TRTAMI137 = FALSE AND ^TRTAMI138 = FALSE AND CountDocument('IDS') = 1 AND ^IDS9 = TRUE
        
        Data Elements:
        ^TRTAMI137 = TRTAMI:RP:TrustDetails:InternationalDealings.TransactionsWithTaxHavenCountries.Indicator

        ^TRTAMI138 = TRTAMI:RP:TrustDetails:InternationalDealings.TransactionControlTaxHavenCountries.Indicator

        ^IDS9 = IDS:RP:InternationalDealings.SpecifiedCountriesTransactions.Indicator
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000205()
        {
            IEnumerable<bool?> ids9 = IDSChildDocuments.Where(c => (c.IDS9) != null).Select(c => c.IDS9);

            bool assertion = ParentDocument.TRTAMI137 == false && ParentDocument.TRTAMI138 == false
                && IDSChildDocuments.Count() == 1
                && ids9.FirstOrDefault() == true;

            if (assertion)
            {
                Response.Add(new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000205",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Transactions with specified countries question incorrect",
                    LongDescription = @"When the response to 'Did you have dealings with international related parties located in specified countries?' on the International Dealings Schedule (IDS) is 'True' (yes), the response at either 'Funds or property, whether directly or indirectly, have been sent to, or received from, any of the specified countries', or 'Trustee has the ability or expectation to control, whether directly or indirectly, the disposition of any funds, property, assets or investments located in, or located elsewhere but controlled or managed from any of the specified countries', must also be 'True' (yes)",
                    Location = "/tns:TRTAMI/tns:RP/tns:TrustDetails/tns:InternationalDealingsTransactionsWithTaxHavenCountriesI",
                    Parameters = new ProcessMessageParameters() {
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMI.000205" },
                        new ProcessMessageParameter() { Name = "TRTAMI137", Value = TRTAMI2026Validator.GetValueOrEmpty(ParentDocument.TRTAMI137) },
                        new ProcessMessageParameter() { Name = "TRTAMI138", Value = TRTAMI2026Validator.GetValueOrEmpty(ParentDocument.TRTAMI138) },
                        new ProcessMessageParameter() { Name = "IDS9", Value = TRTAMI2026Validator.GetValueOrEmpty(ids9.FirstOrDefault()) }
                    },
                });
            }
        }

        #endregion

        #region VR.ATO.TRTAMI.000234

        /*  VR.ATO.TRTAMI.000234
        When multiple Trust Attribution Managed Investments Schedules (TRTAMIS) are provided, a unique Name of AMIT Class must be provided for each.

        Technical Business Rule Format:
        (CountDocument('TRTAMIS') > 1 AND ^TRTAMIS313 <> NULL AND HasDuplicateValues(^TRTAMIS313))
        
        Data Elements:

        ^TRTAMIS313 = TRTAMIS:RP:IncomeTax.AttributionManagedInvestmentTrustClass.Description
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000234()
        {
            IEnumerable<TRTAMIS2025> duplicates = TRTAMISChildDocuments.Where(s => !string.IsNullOrEmpty(s.TRTAMIS313))
                .GroupBy(s => s.TRTAMIS313).Where(s => s.Skip(1).Any()).SelectMany(c => c);

            Response.AddRange(from duplicate in duplicates
                              select new ProcessMessageDocument
                              {
                                  Code = "CMN.ATO.TRTAMI.000234",
                                  Severity = ProcessMessageSeverity.Error,
                                  Description = @"Name of AMIT Class for TRTAMIS schedule is invalid",
                                  LongDescription = @"When multiple Trust Attribution Managed Investments Schedules (TRTAMIS) are provided, a unique Name of AMIT Class must be provided for each",
                                  Location = "/tns:TRTAMI/tns:RP",
                                  Parameters = new ProcessMessageParameters {
                                      new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMI.000234" },
                                      new ProcessMessageParameter { Name = "TRTAMIS313", Value = TRTAMI2026Validator.GetValueOrEmpty(duplicate.TRTAMIS313) }
                                  }
                              });
        }

        #endregion

        #region VR.ATO.TRTAMI.000235

        /*  VR.ATO.TRTAMI.000235
        When multiple Capital Gains Tax (CGT) schedules are provided, a unique Name of AMIT Class must be provided for each.

        Technical Business Rule Format:
        (CountDocument('CGTS') > 1 AND HasDuplicateValues(^CGTS153))
        
        Data Elements:

        ^CGTS153 = CGTS:RP:IncomeTax.AttributionManagedInvestmentTrustClass.Description
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000235()
        {
            IEnumerable<CGTS2018> duplicates = CGTSChildDocuments.GroupBy(s => s.CGTS153).Where(s => s.Skip(1).Any()).SelectMany(c => c);

            Response.AddRange(from duplicate in duplicates
                              select new ProcessMessageDocument
                              {
                                  Code = "CMN.ATO.TRTAMI.000235",
                                  Severity = ProcessMessageSeverity.Error,
                                  Description = @"Name of AMIT Class for CGT schedule is invalid",
                                  LongDescription = @"When multiple Capital Gains Tax (CGT) schedules are provided, a unique Name of AMIT Class must be provided for each",
                                  Location = "/tns:TRTAMI/tns:RP",
                                  Parameters = new ProcessMessageParameters {
                                      new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMI.000235" },
                                      new ProcessMessageParameter { Name = "CGTS153", Value = TRTAMI2026Validator.GetValueOrEmpty(duplicate.CGTS153) }
                                  }
                              });
        }

        #endregion // VR.ATO.TRTAMI.000235

        #region VR.ATO.TRTAMI.000236

        /*  VR.ATO.TRTAMI.000236
        When multiple Trust Attribution Managed Investments Schedules (TRTAMIS) are provided, the response to 'Have you chosen to treat separate classes in the trust as a separate AMIT?' must be 'True' (yes).

        Technical Business Rule Format:
        ^TRTAMI102 = FALSE AND CountDocument('TRTAMIS') > 1

        Data Elements:
    
        ^TRTAMI102 = TRTAMI:RP:MultiClassTreatment:IncomeTax.MultiClassTreatment.Indicator
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000236()
        {
            bool assertion = !ParentDocument.TRTAMI102.GetValueOrDefault() && TRTAMISChildDocuments.Count() > 1;

            if (assertion)
            {
                Response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.TRTAMI.000236",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"'Have you chosen to treat separate classes in the trust as a separate AMIT?' response is incorrect",
                    LongDescription = @"When multiple Trust Attribution Managed Investments Schedules (TRTAMIS) are provided, the response to 'Have you chosen to treat separate classes in the trust as a separate AMIT?' must be 'True' (yes)",
                    Location = "/tns:TRTAMI/tns:RP/tns:MultiClassTreatment/tns:IncomeTaxI",
                    Parameters = new ProcessMessageParameters {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMI.000236" },
                        new ProcessMessageParameter { Name = "TRTAMI102", Value = TRTAMI2026Validator.GetValueOrEmpty(ParentDocument.TRTAMI102) }
                    }
                });
            }
        }

        #endregion

        #region VR.ATO.TRTAMI.000237

        /*  VR.ATO.TRTAMI.000237
        When the response to 'Have you chosen to treat separate classes in the trust as a separate AMIT?' is 'True' (yes), the Name of AMIT Class must be provided for each Trust Attribution Managed Investments Schedule (TRTAMIS).

        Technical Business Rule Format:
        ^TRTAMI102 = TRUE AND AnyOccurrence(^TRTAMIS313, ^TRTAMIS313 = NULL)

        Data Elements:
    
        ^TRTAMI102 = TRTAMI:RP:MultiClassTreatment:IncomeTax.MultiClassTreatment.Indicator
    
        ^TRTAMIS313 = TRTAMIS:RP:IncomeTax.AttributionManagedInvestmentTrustClass.Description
         */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000237()
        {
            bool anyTRTAMIS313NotProvided = TRTAMISChildDocuments != null && TRTAMISChildDocuments.Count() > 0 ? TRTAMISChildDocuments.Any(trtamis => string.IsNullOrWhiteSpace(trtamis.TRTAMIS313)) : true;

            bool assertion = ParentDocument.TRTAMI102.GetValueOrDefault() && anyTRTAMIS313NotProvided;

            if (assertion)
            {
                Response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.TRTAMI.000237",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Name of AMIT Class must be provided for each TRTAMIS schedule",
                    LongDescription = @"When the response to 'Have you chosen to treat separate classes in the trust as a separate AMIT?' is 'True' (yes), the Name of AMIT Class must be provided for each Trust Attribution Managed Investments Schedule (TRTAMIS)",
                    Location = "/tns:TRTAMI/tns:RP/tns:MultiClassTreatment/tns:IncomeTaxI",
                    Parameters = new ProcessMessageParameters {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMI.000237" },
                        new ProcessMessageParameter { Name = "TRTAMI102", Value = TRTAMI2026Validator.GetValueOrEmpty(ParentDocument.TRTAMI102) },
                        new ProcessMessageParameter { Name = "TRTAMIS313 is empty", Value = TRTAMI2026Validator.GetValueOrEmpty(anyTRTAMIS313NotProvided) }
                    }
                });
            }
        }

        #endregion

        #region VR.ATO.TRTAMI.000240

        /*  VR.ATO.TRTAMI.000240
        Name of AMIT Class provided on each Capital Gains Tax (CGT) schedule, must have a matching Trust Attribution Managed Investments Schedule (TRTAMIS).

        Technical Business Rule Format:
        AnyOccurrence(CountDocument('CGTS') > 0, ^CGTS153 <> NULL AND AnyOccurrence(^TRTAMIS313, ^TRTAMIS313 <> ^CGTS153))
        
        Data Elements:

        ^CGTS153 = CGTS:RP:IncomeTax.AttributionManagedInvestmentTrustClass.Description
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000240()
        {
            IEnumerable<string> cgts153 = CGTSChildDocuments.Where(c => !string.IsNullOrWhiteSpace(c.CGTS153)).Select(c => c.CGTS153);

            if (cgts153.Any())
            {
                IEnumerable<string> trtamis313 = TRTAMISChildDocuments.Select(t => t.TRTAMIS313);

                Response.AddRange(cgts153.Where(c => !trtamis313.Any(c.Equals)).Select(c => new ProcessMessageDocument
                {
                    Code = "CMN.ATO.TRTAMI.000240",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Name of AMIT Class for CGT schedule is invalid",
                    LongDescription = @"Name of AMIT Class provided on each Capital Gains Tax (CGT) schedule, must have a matching Trust Attribution Managed Investments Schedule (TRTAMIS)",
                    Location = "/tns:TRTAMI/tns:RP",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter {Name = "RuleIdentifier", Value = "VR.ATO.TRTAMI.000240"},
                        new ProcessMessageParameter {Name = "CGTS153", Value = TRTAMI2026Validator.GetValueOrEmpty(c)}
                    }
                }));
            }
        }

        #endregion

        #region VR.ATO.TRTAMI.000243

        /*  VR.ATO.TRTAMI.000243
        When the response to 'Have you chosen to treat separate classes in the trust as a separate AMIT?' is 'True' (yes), the Name of AMIT Class must be provided for each Capital Gains Tax (CGT) schedule.

        Technical Business Rule Format:
        ^TRTAMI102 = TRUE AND CountDocument('CGTS') > 0 AND AnyOccurrence(^CGTS153, ^CGTS153 = NULL)

        Data Elements:

        ^TRTAMI102 = TRTAMI:RP:MultiClassTreatment:IncomeTax.MultiClassTreatment.Indicator

        ^CGTS153 = CGTS:RP:IncomeTax.AttributionManagedInvestmentTrustClass.Description
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000243()
        {
            bool anyCGTS153NotProvided = CGTSChildDocuments != null && CGTSChildDocuments.Count() > 0 ? CGTSChildDocuments.Any(cgts => string.IsNullOrWhiteSpace(cgts.CGTS153)) : false;

            bool assertion = ParentDocument.TRTAMI102.GetValueOrDefault() && anyCGTS153NotProvided;

            if (assertion)
            {
                Response.Add(new ProcessMessageDocument
                {

                    Code = "CMN.ATO.TRTAMI.000243",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Name of AMIT Class must be provided for each CGT schedule",
                    LongDescription = @"When the response to 'Have you chosen to treat separate classes in the trust as a separate AMIT?' is 'True' (yes), the Name of AMIT Class must be provided for each Capital Gains Tax (CGT) schedule",
                    Location = "/tns:TRTAMI/tns:RP/tns:MultiClassTreatment/tns:IncomeTaxI",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMI.000243" },
                        new ProcessMessageParameter { Name = "TRTAMI102", Value = TRTAMI2026Validator.GetValueOrEmpty(ParentDocument.TRTAMI102) },
                        new ProcessMessageParameter { Name = "CGTS153 is empty", Value = TRTAMI2026Validator.GetValueOrEmpty(anyCGTS153NotProvided) }
                    }
                });
            }
        }

        #endregion // VR.ATO.TRTAMI.000243

        #region VR.ATO.TRTAMI.000244

        /*  VR.ATO.TRTAMI.000244
        When the Name of AMIT Class has been provided for any Trust Attribution Managed Investments Schedule (TRTAMIS), the response to 'Have you chosen to treat separate classes in the trust as a separate AMIT?' must be 'True' (yes).

        Technical Business Rule Format:
        ^TRTAMI102 = FALSE AND AnyOccurrence(^TRTAMIS313, ^TRTAMIS313 <> NULL)

        Data Elements:
    
        ^TRTAMI102 = TRTAMI:RP:MultiClassTreatment:IncomeTax.MultiClassTreatment.Indicator
    
        ^TRTAMIS313 = TRTAMIS:RP:IncomeTax.AttributionManagedInvestmentTrustClass.Description
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000244()
        {
            bool anyTRTAMIS313Provided = TRTAMISChildDocuments != null && TRTAMISChildDocuments.Count() > 0 ? TRTAMISChildDocuments.Any(trtamis => !string.IsNullOrWhiteSpace(trtamis.TRTAMIS313)) : false;

            bool assertion = ParentDocument.TRTAMI102.GetValueOrDefault() == false && anyTRTAMIS313Provided;

            if (assertion)
            {
                Response.Add(new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000244",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"'Have you chosen to treat separate classes in the trust as a separate AMIT?' response is incorrect",
                    LongDescription = @"When the Name of AMIT Class has been provided for any Trust Attribution Managed Investments Schedule (TRTAMIS), the response to 'Have you chosen to treat separate classes in the trust as a separate AMIT?' must be 'True' (yes)",
                    Location = "/tns:TRTAMI/tns:RP/tns:MultiClassTreatment/tns:IncomeTaxI",
                    Parameters = new ProcessMessageParameters {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMI.000244" },
                        new ProcessMessageParameter { Name = "TRTAMI102", Value = TRTAMI2026Validator.GetValueOrEmpty(ParentDocument.TRTAMI102) },
                        new ProcessMessageParameter { Name = "TRTAMIS313 Provided", Value = TRTAMI2026Validator.GetValueOrEmpty(anyTRTAMIS313Provided) }
                    }
                });
            }
        }

        #endregion

        #region VR.ATO.TRTAMI.000245

        /*  VR.ATO.TRTAMI.000245
        When the response to 'Have you chosen to treat separate classes in the trust as a separate AMIT?' is 'False' (no), the Total capital gains/losses from transfers to other classes amounts for the Capital Gains Tax (CGT) schedule must not be provided.

        Technical Business Rule Format:
        ^TRTAMI102 = FALSE AND CountDocument('CGTS') > 0 AND (AnyOccurrence(^CGTS154, NotInSet(^CGTS154, 'NULL','0')) OR AnyOccurrence(^CGTS155, NotInSet(^CGTS155, 'NULL','0')))

        Data Elements:
    
        ^TRTAMI102 = TRTAMI:RP:MultiClassTreatment:IncomeTax.MultiClassTreatment.Indicator
    
        ^CGTS154 = CGTS154
    
        ^CGTS155 = CGTS155
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000245()
        {
            bool anyCGTS154Provided = CGTSChildDocuments != null && CGTSChildDocuments.Count() > 0 ? CGTSChildDocuments.Any(cgts => !Regex.IsMatch(cgts.CGTS154.GetValueOrDefault().ToString(), @"^(NULL|0)$", RegexOptions.IgnoreCase)) : false;
            bool anyCGTS155Provided = CGTSChildDocuments != null && CGTSChildDocuments.Count() > 0 ? CGTSChildDocuments.Any(cgts => !Regex.IsMatch(cgts.CGTS155.GetValueOrDefault().ToString(), @"^(NULL|0)$", RegexOptions.IgnoreCase)) : false;

            bool assertion = ParentDocument.TRTAMI102.GetValueOrDefault() == false && (anyCGTS154Provided || anyCGTS155Provided);

            if (assertion)
            {
                Response.Add(new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000245",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total capital gains/losses from transfers to other classes amounts must not be provided in CGT schedule",
                    LongDescription = @"When the response to 'Have you chosen to treat separate classes in the trust as a separate AMIT?' is 'False' (no), the Total capital gains/losses from transfers to other classes amounts for the Capital Gains Tax (CGT) schedule must not be provided",
                    Location = "/tns:TRTAMI/tns:RP/tns:MultiClassTreatment/tns:IncomeTaxI",
                    Parameters = new ProcessMessageParameters {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMI.000245" },
                        new ProcessMessageParameter { Name = "TRTAMI102", Value = TRTAMI2026Validator.GetValueOrEmpty(ParentDocument.TRTAMI102) },
                        new ProcessMessageParameter { Name = "CGTS154 Provided", Value = TRTAMI2026Validator.GetValueOrEmpty(anyCGTS154Provided) },
                        new ProcessMessageParameter { Name = "CGTS155 Provided", Value = TRTAMI2026Validator.GetValueOrEmpty(anyCGTS155Provided) }
                    }
                });
            }
        }

        #endregion // VR.ATO.TRTAMI.000245

        #region VR.ATO.TRTAMI.000246

        /*  VR.ATO.TRTAMI.000246
        When the value of dealings with international related parties was above $2 million, the International Dealings Schedule (IDS) must be attached

        Technical Business Rule Format:
        ^TRTAMI47 = TRUE AND CountDocument('IDS') = 0

        Data Elements:
    
        ^TRTAMI47 = TRTAMI:RP:TrustDetails:InternationalDealings.RelatedPartiesTransactionsExcessAggregateValue.Indicator
       */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000246()
        {
            bool assertion = ParentDocument.TRTAMI47 == true && !IDSChildDocuments.Any();

            if (assertion)
            {
                Response.Add(new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000246",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"International Dealings Schedule (IDS) must be attached",
                    LongDescription = @"When the value of dealings with international related parties was above $2 million, the International Dealings Schedule (IDS) must be attached",
                    Location = "/tns:TRTAMI/tns:RP/tns:TrustDetails/tns:InternationalDealingsRelatedPartiesTransactionsExcessAggregateValueI",
                    Parameters = new ProcessMessageParameters {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMI.000246" },
                        new ProcessMessageParameter { Name = "TRTAMI47", Value = TRTAMI2026Validator.GetValueOrEmpty(ParentDocument.TRTAMI47) }
                    }
                });
            }
        }

        #endregion

        #region VR.ATO.TRTAMI.000314

        /*  VR.ATO.TRTAMI.000314
        Where Total Liabilities on the main AMIT tax return has a value of zero or greater and there is no occurrence (zero or greater) of the corresponding Total Liabilities labels in at least one (1) AMIT schedule, the system shall disallow lodgements.

        Technical Business Rule Format:
        ^TRTAMI60 <> NULL AND CountDocument('TRTAMIS') > 0 AND AllOccurrences(^TRTAMIS334, ^TRTAMIS334 = NULL)

        Data Elements:
    
        ^TRTAMI60 = TRTAMI:RP:KeyFinancialInformation:Liabilities.Total.Amount
    
        ^TRTAMIS334 = TRTAMIS334
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000314()
        {
            bool allTRTAMIS334NotProvided = TRTAMISChildDocuments != null && TRTAMISChildDocuments.Count() > 0 ? TRTAMISChildDocuments.All(t => t.TRTAMIS334 == null) : false;

            bool assertion = ParentDocument.TRTAMI60 != null && allTRTAMIS334NotProvided;

            if (assertion)
            {
                Response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.TRTAMI.000314",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"At least one occurrence of Total Liabilities should be present if the corresponding label in AMIT tax return has a value",
                    LongDescription = @"Where Total Liabilities on the main AMIT tax return has a value of zero or greater and there is no occurrence (zero or greater) of the corresponding Total Liabilities labels in at least one (1) AMIT schedule, the system shall disallow lodgements.",
                    Location = "/tns:TRTAMI/tns:RP/tns:KeyFinancialInformation/tns:LiabilitiesTotalA",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMI.000314" },
                        new ProcessMessageParameter { Name = "TRTAMI60", Value = TRTAMI2026Validator.GetValueOrEmpty(ParentDocument.TRTAMI60) },
                        new ProcessMessageParameter { Name = "TRTAMIS334 is empty", Value = TRTAMI2026Validator.GetValueOrEmpty(allTRTAMIS334NotProvided) }
                    }
                });
            }
        }

        #endregion

        #region VR.ATO.TRTAMI.000315

        /*  VR.ATO.TRTAMI.000315
        Where Total Assets on the main AMIT tax return has a value of zero or greater and there is no occurrence (zero or greater) of the corresponding Total Assets labels in at least one (1) AMIT schedule, the system shall disallow lodgements.

        Technical Business Rule Format:
        ^TRTAMI58 <> NULL AND CountDocument('TRTAMIS') > 0 AND AllOccurrences(^TRTAMIS333, ^TRTAMIS333 = NULL)

        Data Elements:
    
        ^TRTAMI58 = TRTAMI:RP:KeyFinancialInformation:Assets.Total.Amount
    
        ^TRTAMIS333 = TRTAMIS333
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000315()
        {
            bool allTRTAMIS333NotProvided = TRTAMISChildDocuments != null && TRTAMISChildDocuments.Count() > 0 ? TRTAMISChildDocuments.All(t => t.TRTAMIS333 == null) : false;

            bool assertion = ParentDocument.TRTAMI58 != null && allTRTAMIS333NotProvided;

            if (assertion)
            {
                Response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.TRTAMI.000315",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"At least one occurrence of Total Assets should be present if the corresponding label in AMIT tax return has a value",
                    LongDescription = @"Where Total Assets on the main AMIT tax return has a value of zero or greater and there is no occurrence (zero or greater) of the corresponding Total Assets labels in at least one (1) AMIT schedule, the system shall disallow lodgements.",
                    Location = "/tns:TRTAMI/tns:RP/tns:KeyFinancialInformation/tns:AssetsTotalA",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMI.000315" },
                        new ProcessMessageParameter { Name = "TRTAMI58", Value = TRTAMI2026Validator.GetValueOrEmpty(ParentDocument.TRTAMI58) },
                        new ProcessMessageParameter { Name = "TRTAMIS333 is empty", Value = TRTAMI2026Validator.GetValueOrEmpty(allTRTAMIS333NotProvided) }
                    }
                });
            }
        }

        #endregion

        #region VR.ATO.TRTAMI.000330

        /*  VR.ATO.TRTAMI.000330
        If the trust has self-assessed that they are a Significant Global Entity (SGE), then 'Are you a small business entity, not a significant global entity and your international related party dealings do not exceed $5 million and 50% of your current year aggregated turnover?' must not be 'TRUE' on the International Dealings Schedule (IDS) form.

        Technical Business Rule Format:
        ^TRTAMI162 = TRUE AND (CountDocument('IDS') = 1 AND ^IDS425 = TRUE)

        Data Elements:
    
        ^TRTAMI162 = TRTAMI:RP:TrustDetails:OrganisationDetails.SignificantGlobalEntityStatus.Indicator
    
        ^IDS425 = IDS:RP:InternationalDealings.SmallBusinessConcessionsEligibility.Indicator
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000330()
        {
            bool assertion = ParentDocument.TRTAMI162.GetValueOrDefault() == true &&
               IDSChildDocuments.Count() == 1 &&
               IDSChildDocuments.FirstOrDefault().IDS425.GetValueOrDefault() == true;

            if (assertion)
            {
                Response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.TRTAMI.000330",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Small business entity question on the International Dealings Schedule (IDS) form must not be 'TRUE'.",
                    LongDescription = @"If the trust has self-assessed that they are a Significant Global Entity (SGE), then 'Are you a small business entity, not a significant global entity and your international related party dealings do not exceed $5 million and 50% of your current year aggregated turnover?' must not be 'TRUE' on the International Dealings Schedule (IDS) form.",
                    Location = $"/tns:TRTAMI/tns:RP/tns:TrustDetails/tns:OrganisationDetailsSignificantGlobalEntityStatusI",
                    Parameters = new ProcessMessageParameters {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMI.000330" },
                        new ProcessMessageParameter { Name = "TRTAMI162", Value = TRTAMI2026Validator.GetValueOrEmpty(ParentDocument.TRTAMI162) },
                        new ProcessMessageParameter { Name = "IDS425", Value = TRTAMI2026Validator.GetValueOrEmpty(IDSChildDocuments.FirstOrDefault().IDS425) },
                    }
                });
            }
        }

        #endregion

        #region VR.ATO.TRTAMI.000401

        /*  VR.ATO.TRTAMI.000401
        When Is this lodgment for an attribution CCIV sub-fund? is true, more than one Trust Attribution Managed Investments Schedules (TRTAMIS) must not be provided

        Technical Business Rule Format:
        ^TRTAMI192 = TRUE AND CountDocument('TRTAMIS') > 1

        Data Elements:
    
        ^TRTAMI192 = TRTAMI:RP:Report.CorporateCollectiveInvestmentVehicleSubFund.Indicator
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000401()
        {
            bool assertion = ParentDocument.TRTAMI192.GetValueOrDefault() && TRTAMISChildDocuments.Count() > 1;

            if (assertion)
            {
                Response.Add(new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000401",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"More than one Trust Attribution Managed Investments Schedules (TRTAMIS) must not be provided",
                    LongDescription = @"When Is this lodgment for an attribution CCIV sub-fund? is true, more than one Trust Attribution Managed Investments Schedules (TRTAMIS) must not be provided",
                    Location = "/tns:TRTAMI/tns:RP/tns:CorporateCollectiveInvestmentVehicleSubFundI",
                    Parameters = new ProcessMessageParameters {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMI.000401" },
                        new ProcessMessageParameter { Name = "TRTAMI192", Value = TRTAMI2026Validator.GetValueOrEmpty(ParentDocument.TRTAMI192) }
                    }
                });
            }
        }
        #endregion // VR.ATO.TRTAMI.000401

        #region VR.ATO.TRTAMI.000420

        /*  VR.ATO.TRTAMI.000420
        When you were affected by the Thin capitalisation or debt deduction creation rules, the International Dealings Schedule (IDS) must be attached

        Technical Business Rule Format:
        (^TRTAMI48 = TRUE) AND (CountDocument('IDS') = 0)

        Data Elements:
    
        ^TRTAMI48 = TRTAMI:RP:TrustDetails:Liabilities.ThinCapitalisation.ProvisionsApplied.Indicator
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000420()
        {
            bool assertion = ParentDocument.TRTAMI48 == true && !IDSChildDocuments.Any();

            if (assertion)
            {
                Response.Add(new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000420",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"International Dealings Schedule not attached",
                    LongDescription = @"If the thin capitalisation or debt deduction creation rules apply then International Dealings Schedule must be attached",
                    Location = "/tns:TRTAMI/tns:RP/tns:TrustDetails/tns:LiabilitiesThinCapitalisationProvisionsAppliedI",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMI.000420" },
                        new ProcessMessageParameter() { Name = "TRTAMI48", Value = TRTAMI2026Validator.GetValueOrEmpty(ParentDocument.TRTAMI48) }
                    },
                });
            }
        }

        #endregion

        #region VR.ATO.TRTAMI.000425

        /*  VR.ATO.TRTAMI.000425
        The reportable tax position schedule must be attached if there is a 'yes' (true) response to 'Are you required to lodge a reportable tax position schedule?'

        Technical Business Rule:
        ^TRTAMI212 = TRUE AND CountDocument('RPTTAXPOS') = 0

        Data Elements:

        ^TRTAMI212  = TRTAMI:RP:RegulatoryDisclosures.ReportableTaxPosition.Indicator
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000425()
        {
            ProcessMessageDocument processMessage;
            bool assertion = ParentDocument.TRTAMI212.GetValueOrDefault() && RPTTAXPOSChildDocuments.Count() == 0;

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000425",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The reportable tax position schedule must be attached",
                    LongDescription = @"The reportable tax position schedule must be attached if there is a 'yes' (true) response to 'Are you required to lodge a reportable tax position schedule?'",
                    Location = "/tns:TRTAMI/tns:RP/tns:RegulatoryDisclosuresReportableTaxPositionI",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMI.000425" } ,
                        new ProcessMessageParameter { Name = "TRTAMI212", Value = TRTAMI2026Validator.GetValueOrEmpty(ParentDocument.TRTAMI212) }
                    }
                };

                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.TRTAMI.000425

        #region VR.ATO.TRTAMI.000426

        /*  VR.ATO.TRTAMI.000426
        The reportable tax position schedule must not be attached if there is a 'no' (false) response to 'Are you required to lodge a reportable tax position schedule?'

        Technical Business Rule:
        ^TRTAMI212 = FALSE AND CountDocument('RPTTAXPOS') = 1

        Data Elements:

        ^TRTAMI212  = TRTAMI:RP:RegulatoryDisclosures.ReportableTaxPosition.Indicator
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000426()
        {
            ProcessMessageDocument processMessage;
            bool assertion = !ParentDocument.TRTAMI212.GetValueOrDefault() && RPTTAXPOSChildDocuments.Count() == 1;

            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000426",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The reportable tax position schedule must not be attached",
                    LongDescription = @"The reportable tax position schedule must not be attached if there is a 'no' (false) response to 'Are you required to lodge a reportable tax position schedule?'",
                    Location = "/tns:TRTAMI/tns:RP/tns:RegulatoryDisclosuresReportableTaxPositionI",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMI.000426" } ,
                        new ProcessMessageParameter { Name = "TRTAMI212", Value = TRTAMI2026Validator.GetValueOrEmpty(ParentDocument.TRTAMI212) }
                    }
                };

                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.TRTAMI.000426

        #region VR.ATO.TRTAMI.000427

        /*  VR.ATO.TRTAMI.000427
        The number of Reportable Tax Position Schedule  must not exceed 1

        Technical Business Rule:
        CountDocument('RPTTAXPOS') > 1
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000427()
        {
            bool assertion = RPTTAXPOSChildDocuments.Count() > 1;

            if (assertion)
            {
                Response.Add(new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000427",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The number of Reportable Tax Position Schedule must not exceed 1",
                    Location = "/tns:TRTAMI/tns:RP",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMI.000427" } },
                });
            }
        }

        #endregion

        #endregion
    }
}
