using Ato.CD.Inbound.Shared;
using Ato.EN.IntegrationServices.CodeGenerationRPTTAXPOS;
using Ato.EN.IntegrationServices.CodeGenerationTRTAMI;
using DataContracts;
using System.Collections.Generic;
using System.Linq;
using System.Runtime.CompilerServices;

namespace Ato.CD.Inbound.TRTAMI202602
{
    public class CrossFormValidatorRPTTAXPOS : ICrossFormValidator
    {
        private TRTAMI2026 ParentDocument { get; }
        private RPTTAXPOS2019 ChildDocument { get; }
        public List<ProcessMessageDocument> Response { get; private set; }

        public CrossFormValidatorRPTTAXPOS(TRTAMI2026 report, object childDocument)
        {
            ParentDocument = report;
            ChildDocument = (RPTTAXPOS2019)childDocument;
            Response = new List<ProcessMessageDocument>();
        }

        public List<ProcessMessageDocument> ValidateCrossFormRules()
        {
            //Parent form cross form rules
            VRATOTRTAMI000423();
            VRATOTRTAMI000424();
            VRATOTRTAMI000428();
            VRATOTRTAMI000429();

            return Response;
        }

        #region VR.ATO.TRTAMI.000423

        /*  VR.ATO.TRTAMI.000423
        The first four digits of RTP category number must refer to your current financial year

        Technical Business Rule:
        CountDocument('RPTTAXPOS') = 1 AND Substring(RPTTAXPOS:bafot.02.36:RegulatoryDisclosures.ReportableTaxPositionCategoryReferenceNumber.Identifier,1,4,) <> ^TRTAMI1

        Data Elements:

        ^TRTAMI1 = TRTAMI:RP:Report.TargetFinancial.Year

        ^RPTTAXPOS4 = RPTTAXPOS:RP:ReportableTaxPositionCategoryAAndB:RegulatoryDisclosures.ReportableTaxPositionCategoryReferenceNumber.Identifier

        ^RPTTAXPOS11 = RPTTAXPOS:RP:ReportableTaxPositionCategoryC:RegulatoryDisclosures.ReportableTaxPositionCategoryReferenceNumber.Identifier
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000423()
        {
            //Check RPTTAXPOS4
            if (ChildDocument.ReportableTaxPositionCategoryAAndBCollection != null)
                Response.AddRange(from rptCategoryAAndB in ChildDocument.ReportableTaxPositionCategoryAAndBCollection
                                  where rptCategoryAAndB.RPTTAXPOS4 != null && rptCategoryAAndB.RPTTAXPOS4.Length == 8 && rptCategoryAAndB.RPTTAXPOS4.Substring(0, 4).ToString() != ParentDocument.TRTAMI1.GetValueOrDefault().ToString()
                                  select new ProcessMessageDocument
                                  {
                                      Code = "CMN.ATO.TRTAMI.000423",
                                      Severity = ProcessMessageSeverity.Error,
                                      Description = @"The first four digits of RTP category number must refer to your current financial year",
                                      Location = "/tns:TRTAMI/tns:RP/tns:TargetFinancialY",
                                      Parameters = new ProcessMessageParameters
                                      {
                                          new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMI.000423" },
                                          new ProcessMessageParameter { Name = "TRTAMI1", Value = TRTAMI2026Validator.GetValueOrEmpty(ParentDocument.TRTAMI1) },
                                          new ProcessMessageParameter { Name = "RPTTAXPOS4", Value = rptCategoryAAndB.RPTTAXPOS4 }
                                      },
                                      DocumentSequence = "parent_doc"
                                  });

            //Check RPTTAXPOS11
            if (ChildDocument.ReportableTaxPositionCategoryCCollection != null)
                Response.AddRange(from rptCategoryC in ChildDocument.ReportableTaxPositionCategoryCCollection
                                  where rptCategoryC.RPTTAXPOS11 != null && rptCategoryC.RPTTAXPOS11.Length == 8 && rptCategoryC.RPTTAXPOS11.Substring(0, 4).ToString() != ParentDocument.TRTAMI1.GetValueOrDefault().ToString()
                                  select new ProcessMessageDocument
                                  {
                                      Code = "CMN.ATO.TRTAMI.000423",
                                      Severity = ProcessMessageSeverity.Error,
                                      Description = @"The first four digits of RTP category number must refer to your current financial year",
                                      Location = "/tns:TRTAMI/tns:RP/tns:TargetFinancialY",
                                      Parameters = new ProcessMessageParameters
                                      {
                                          new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMI.000423" },
                                          new ProcessMessageParameter { Name = "TRTAMI1", Value = TRTAMI2026Validator.GetValueOrEmpty(ParentDocument.TRTAMI1) },
                                          new ProcessMessageParameter { Name = "RPTTAXPOS11", Value = rptCategoryC.RPTTAXPOS11 }
                                      },
                                      DocumentSequence = "parent_doc"
                                  });
        }

        #endregion // VR.ATO.TRTAMI.000423

        #region VR.ATO.TRTAMI.000424

        /*  VR.ATO.TRTAMI.000424
        The Tax File Number (TFN) provided for the reportable tax position schedule must be the same as the TFN of the parent form

        Technical Business Rule:
        CountDocument('RPTTAXPOS') = 1 AND ^TRTAMI4 <> RPTTAXPOS:^RP:TFN

        Data Elements:

        ^TRTAMI4 = TRTAMI:RP:Identifiers.TaxFileNumber.Identifier

        ^RPTTAXPOS TFN  = RPTTAXPOS:RP:entity.identifier.TFN
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000424()
        {
            ProcessMessageDocument processMessage;
            bool assertion = !ParentDocument.TRTAMI4.Equals(ChildDocument.RPIdentifierTFN);

            if (assertion)
            {
                processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.402009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your supplied TFN does not match the TFN supplied on the form it was submitted with",
                    LongDescription = @"Your supplied TFN does not match the TFN supplied on the parent form it was submitted with",
                    Location = "/tns:TRTAMI/tns:RP/tns:TaxFileNumberId",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMI.000424" },
                        new ProcessMessageParameter { Name = "TRTAMI4", Value = ParentDocument.TRTAMI4 },
                        new ProcessMessageParameter { Name = "RPTTAXPOS TFN", Value = ChildDocument.RPIdentifierTFN }
                    },
                    DocumentSequence = "parent_doc"
                };
             
                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.TRTAMI.000424

        #region VR.ATO.TRTAMI.000428

        /*  VR.ATO.TRTAMI.000428
        The start date in reportable tax position schedule must be the same as the start date provided in its parent form

        Technical Business Rule:
        CountDocument('RPTTAXPOS') = 1 AND ^TRTAMI2 <> ^RPTTAXPOS:RP:period.startDate

        Data Elements:

        ^TRTAMI2 = TRTAMI:RP:Period.Start.Date

        ^RPTTAXPOS Start Date = RPTTAXPOS Context Start Date
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000428()
        {
            ProcessMessageDocument processMessage;
            bool assertion = !ParentDocument.TRTAMI2.Equals(ChildDocument.RPStartDate);

            if (assertion)
            {
                processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.TRTAMI.000428",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The start date in reportable tax position schedule must be the same as the start date provided in its parent form",
                    Location = "/tns:TRTAMI/tns:RP/tns:PeriodStartD",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMI.000428" },
                        new ProcessMessageParameter { Name = "TRTAMI2", Value = TRTAMI2026Validator.GetValueOrEmptyDate(ParentDocument.TRTAMI2) },
                        new ProcessMessageParameter { Name = "RPTTAXPOS Start Date", Value = TRTAMI2026Validator.GetValueOrEmptyDate(ChildDocument.RPStartDate) }
                    },
                    DocumentSequence = "parent_doc"
                };

                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.TRTAMI.000428

        #region VR.ATO.TRTAMI.000429

        /*  VR.ATO.TRTAMI.000429
        The end date in reportable tax position schedule must be the same as the end date provided in its parent form

        Technical Business Rule:
        CountDocument('RPTTAXPOS') = 1 AND ^TRTAMI3 <> ^RPTTAXPOS:RP:period.endDate

        Data Elements:

        ^TRTAMI3 = TRTAMI:RP:Period.End.Date

        ^RPTTAXPOS End Date = RPTTAXPOS Context End Date
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000429()
        {
            ProcessMessageDocument processMessage;
            bool assertion = !ParentDocument.TRTAMI3.Equals(ChildDocument.RPEndDate);

            if (assertion)
            {
                processMessage = new ProcessMessageDocument
                {
                    Code = "CMN.ATO.TRTAMI.000429",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The end date in reportable tax position schedule must be the same as the end date provided in its parent form",
                    Location = "/tns:TRTAMI/tns:RP/tns:PeriodEndD",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMI.000429" },
                        new ProcessMessageParameter { Name = "TRTAMI3", Value = TRTAMI2026Validator.GetValueOrEmptyDate(ParentDocument.TRTAMI3) },
                        new ProcessMessageParameter { Name = "RPTTAXPOS End Date", Value = TRTAMI2026Validator.GetValueOrEmptyDate(ChildDocument.RPEndDate) }
                    },
                    DocumentSequence = "parent_doc"
                };

                Response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.TRTAMI.000429
    }
}
