using Ato.CD.Inbound.Shared;
using Ato.EN.IntegrationServices.CodeGenerationTRTAMI;
using Ato.EN.IntegrationServices.CodeGenerationTRTAMIS;
using DataContracts;
using System.Collections.Generic;
using System.Linq;
using System.Runtime.CompilerServices;

namespace Ato.CD.Inbound.TRTAMI202602
{
    public class CrossFormValidatorTRTAMIS : ICrossFormValidator
    {
        private TRTAMIS2025 ChildDocument { get; set; }

        private List<Context> Contexts { get; set; }

        public List<ProcessMessageDocument> Response { get; private set; }

        private TRTAMI2026 ParentDocument { get; set; }

        public CrossFormValidatorTRTAMIS(TRTAMI2026 parent, object child)
        {
            ChildDocument = (TRTAMIS2025)child;
            Contexts = ChildDocument.GetContexts();
            ParentDocument = parent;
            Response = new List<ProcessMessageDocument>();
        }

        public List<ProcessMessageDocument> ValidateCrossFormRules()
        {
            VRATOGEN402009();
            VRATOGEN402010();
            VRATOGEN438000();
            VRATOGEN438001();
            VRATOTRTAMIS000001();

            return Response;
        }

        #region Validation Rules

        #region VR.ATO.GEN.402009

        /*  VR.ATO.GEN.402009
        Your supplied TFN does not match the TFN supplied on the form it was submitted with

        Legacy Rule Format:
        IF (RP:entity.identifier.TFN <> PARENT RETURN:RP:entity.identifier.TFN)
            RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:

        ^TRTAMI4 = TRTAMI:RP:Identifiers.TaxFileNumber.Identifier

        ^TRTAMIS TFN = TRTAMIS TFN
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN402009()
        {
            bool assertion = !ParentDocument.TRTAMI4.Equals(ChildDocument.RPIdentifierTFN);

            if (assertion)
            {
                Response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.402009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your supplied TFN does not match the TFN supplied on the form it was submitted with",
                    LongDescription = @"Your supplied TFN does not match the TFN supplied on the parent form it was submitted with",
                    Location = "/xbrli:xbrl/xbrli:context/xbrli:entity/xbrli:identifier",
                    Parameters = new ProcessMessageParameters {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.GEN.402009" },
                        new ProcessMessageParameter { Name = "TRTAMI4", Value = TRTAMI2026Validator.GetValueOrEmpty(ParentDocument.TRTAMI4) },
                        new ProcessMessageParameter { Name = "TRTAMIS TFN", Value = TRTAMI2026Validator.GetValueOrEmpty(ChildDocument.RPIdentifierTFN) }
                    }
                });
            }
        }

        #endregion

        #region VR.ATO.GEN.402010

        /*  VR.ATO.GEN.402010
        Your supplied ABN does not match the ABN supplied on the form it was submitted with

        Legacy Rule Format:
        IF (RP:pyid.xx.xx:Identifiers.AustralianBusinessNumber.Identifier <> NULLORBLANK) AND (PARENT RETURN:RP:pyid.xx.xx:Identifiers.AustralianBusinessNumber.Identifier <> NULLORBLANK) AND (RP:pyid.xx.xx:Identifiers.AustralianBusinessNumber.Identifier <> PARENT RETURN:RP:pyid.xx.xx:Identifiers.AustralianBusinessNumber.Identifier)
            RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:

        ^TRTAMI5 = TRTAMI:RP:Identifiers.AustralianBusinessNumber.Identifier

        ^TRTAMIS5 = TRTAMIS:RP:Identifiers.AustralianBusinessNumber.Identifier
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN402010()
        {
            bool assertion = !string.IsNullOrWhiteSpace(ParentDocument.TRTAMI5) &&
                             !string.IsNullOrWhiteSpace(ChildDocument.TRTAMIS5) &&
                             !ParentDocument.TRTAMI5.Equals(ChildDocument.TRTAMIS5);
            if (assertion)
            {
                Response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.GEN.402010",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Your supplied ABN does not match the ABN supplied on the form it was submitted with",
                    LongDescription = @"Your supplied ABN does not match the ABN supplied on the parent form it was submitted with",
                    Location = "/xbrli:xbrl/tns:Identifiers.AustralianBusinessNumber.Identifier",
                    Parameters = new ProcessMessageParameters
                    {
                        new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.GEN.402010" },
                        new ProcessMessageParameter { Name = "TRTAMI5", Value = TRTAMI2026Validator.GetValueOrEmpty(ParentDocument.TRTAMI5) },
                        new ProcessMessageParameter { Name = "TRTAMIS5", Value = TRTAMI2026Validator.GetValueOrEmpty(ChildDocument.TRTAMIS5) }
                    }
                });
            }
        }

        #endregion

        #region VR.ATO.GEN.438000

        /*  VR.ATO.GEN.438000
        Period start date within context on schedule doesn't match start date within PARENT RETURN:reporting party context

        Legacy Rule Format:
        IF (period.startDate WHERE CONTEXT(ALL)) <> PARENT RETURN:RP:period.startDate
            RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:

        ^TRTAMI2 = TRTAMI:RP:Period.Start.Date

        ^TRTAMIS Context Start Date = TRTAMIS Context Start Date
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438000()
        {
            Response.AddRange(from context in Contexts
                              where context.StartDate.GetValueOrDefault() != ParentDocument.TRTAMI2.GetValueOrDefault()
                              select new ProcessMessageDocument
                              {
                                  Code = "CMN.ATO.GEN.438000",
                                  Severity = ProcessMessageSeverity.Error,
                                  Description = @"The context period start date is incorrect.",
                                  LongDescription = @"Period start date within context on the schedule must match period start date within the parent return reporting party context",
                                  Location = $"/xbrli:xbrl/xbrli:context[{context.Id}]/xbrli:period/xbrli:startDate",
                                  Parameters = new ProcessMessageParameters {
                                      new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438000" },
                                      new ProcessMessageParameter { Name = "TRTAMI2", Value = TRTAMI2026Validator.GetValueOrEmptyDate(ParentDocument.TRTAMI2) },
                                      new ProcessMessageParameter { Name = "Context Start Date", Value = TRTAMI2026Validator.GetValueOrEmptyDate(context.StartDate) }
                                  }
                              });
        }

        #endregion

        #region VR.ATO.GEN.438001

        /*  VR.ATO.GEN.438001
        Period end date within context on schedule doesn't match end date within PARENT RETURN:reporting party context

        Legacy Rule Format:
        IF (period.endDate WHERE CONTEXT(ALL)) <> PARENT RETURN:RP:period.endDate
            RETURN VALIDATION MESSAGE
        ENDIF

        Data Elements:

        ^TRTAMI3 = TRTAMI:RP:Period.End.Date

        ^TRTAMIS Context End Date = TRTAMIS Context End Date
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438001()
        {
            Response.AddRange(from context in Contexts
                              where context.EndDate.GetValueOrDefault() != ParentDocument.TRTAMI3.GetValueOrDefault()
                              select new ProcessMessageDocument
                              {
                                  Code = "CMN.ATO.GEN.438001",
                                  Severity = ProcessMessageSeverity.Error,
                                  Description = @"The context period end date is incorrect.",
                                  LongDescription = @"Period end date within context on the schedule must match period end date within the parent return reporting party context",
                                  Location = $"/xbrli:xbrl/xbrli:context[{context.Id}]/xbrli:period/xbrli:endDate",
                                  Parameters = new ProcessMessageParameters {
                                      new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.GEN.438001" },
                                      new ProcessMessageParameter { Name = "TRTAMI3", Value = TRTAMI2026Validator.GetValueOrEmptyDate(ParentDocument.TRTAMI3) },
                                      new ProcessMessageParameter { Name = "Context End Date", Value = TRTAMI2026Validator.GetValueOrEmptyDate(context.EndDate) }
                                  }
                              });
        }

        #endregion

        #region VR.ATO.TRTAMIS.000001

        /*  VR.ATO.TRTAMIS.000001
        Invalid context. Target Financial Year must match the target year of the parent form.

        Legacy Rule Format:
        IF [TRTAMIS4] <> NULL AND [TRTAMIS4] <> PARENT RETURN:pyin.xx.xx:Report.TargetFinancial.Year
            RETURN VALIDATION MESSAGE
        ENDIF
   
        Data Elements:

        ^TRTAMI1 = TRTAMI:RP:Report.TargetFinancial.Year

        ^TRTAMIS4 = TRTAMIS:RP:Report.TargetFinancial.Year
        */
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMIS000001()
        {
            bool assertion = ChildDocument.TRTAMIS4 != null && ChildDocument.TRTAMIS4 != ParentDocument.TRTAMI1;

            if (assertion)
            {
                Response.Add(new ProcessMessageDocument
                {
                    Code = "CMN.ATO.TRTAMIS.000001",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The Target Financial Year must match the Year shown on the parent form.",
                    LongDescription = @"The date present in the Target Financial Year must match the date that is shown in the same field in the parent form",
                    Location = "/xbrli:xbrl/tns:Report.TargetFinancial.Year",
                    Parameters = new ProcessMessageParameters {
                                      new ProcessMessageParameter { Name = "RuleIdentifier", Value = "VR.ATO.TRTAMIS.000001" },
                                      new ProcessMessageParameter { Name = "TRTAMI1", Value = TRTAMI2026Validator.GetValueOrEmpty(ParentDocument.TRTAMI1) },
                                      new ProcessMessageParameter { Name = "TRTAMIS4", Value = TRTAMI2026Validator.GetValueOrEmpty(ChildDocument.TRTAMIS4) }
                                  }
                });
            }
        }

        #endregion

        #endregion
    }
}
