using DataContracts;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Runtime.CompilerServices;
using System.Text.RegularExpressions;

namespace Ato.EN.IntegrationServices.CodeGenerationTRTAMI
{
    public partial class TRTAMI2026Validator
    {
        /// <summary>
        /// The max parameter name length is restricted by the SBR1 and SBR2 schemas - the lowest common denominator of SBR 1 is used as the default
        /// </summary>
        private int _maxParameterNameLength;
        /// <summary>
        /// The max parameter value length is restricted by the SBR1 and SBR2 schemas
        /// </summary>
        private int _maxParameterValueLength;
        /// <summary>
        /// The SBR1 and ebms3 schemas do not allow parameter names or values to be the empty string
        /// </summary>
        private string _emptyParameterValue;
        /// <summary>
        /// The report parameter
        /// </summary>
        private TRTAMI2026 report;
        /// <summary>
        /// The response parameter
        /// </summary>
        public List<ProcessMessageDocument> response { get; private set; }

        /// <summary>
        /// Initializes a new instance of the <see cref = "TRTAMI2026Validator"/> class.
        /// </summary>
        /// <param name = "maxParameterNameLength">Maximum length of the parameter name.</param>
        /// <param name = "maxParameterValueLength">Maximum length of the parameter value.</param>
        /// <param name = "emptyParameterValue">This value will be used in place of any parameter values that result in a null or empty value.</param>
        public TRTAMI2026Validator(int maxParameterNameLength = 20, int maxParameterValueLength = 4096, string emptyParameterValue = "EMPTY")
        {
            _maxParameterNameLength = maxParameterNameLength;
            _maxParameterValueLength = maxParameterValueLength;
            _emptyParameterValue = emptyParameterValue;
            this.response = new List<ProcessMessageDocument>();
        }

        /// <summary>
        /// Initializes a new instance of the <see cref = "TRTAMI2026Validator"/> class.
        /// </summary>
        /// <param name = "report">Report variable </param>                
        /// <param name = "maxParameterNameLength">Maximum length of the parameter name.</param>
        /// <param name = "maxParameterValueLength">Maximum length of the parameter value.</param>
        /// <param name = "emptyParameterValue">This value will be used in place of any parameter values that result in a null or empty value.</param>
        public TRTAMI2026Validator(TRTAMI2026 reportIn, int maxParameterNameLength = 20, int maxParameterValueLength = 4096, string emptyParameterValue = "EMPTY") : this(maxParameterNameLength, maxParameterValueLength, emptyParameterValue)
        {
            this.ConsumedReport = reportIn;
        }

#region Functions
        private static IEnumerable<string> Union(IEnumerable<string> list1, IEnumerable<string> list2)
        {
            IEnumerable<string> response;
            if (list1 == null && list2 == null)
            {
                response = null;
            }
            else if (list1 == null)
            {
                response = list2.Distinct();
            }
            else if (list2 == null)
            {
                response = list1.Distinct();
            }
            else
            {
                response = list1.Union(list2);
            }

            return response;
        }

        private static bool IsMatch(int? field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(Convert.ToString(field.Value), expression, options);
        }

        private static bool IsMatch(string field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(field, expression, options);
        }

        // This is just context and tuple counts where they are integer values - easier that changing the parsing logic to just return the value
        private static int Count(int count)
        {
            return count;
        }

        private static int Count<T>(IEnumerable<T> values)
        {
            return values == null ? 0 : values.Where(f => f != null).Count();
        }

        private static int Count<T>(ICollection<T> values)
        {
            return values == null ? 0 : values.Where(f => f != null).Count();
        }

        private static bool exists(bool value)
        {
            return value;
        }

        public static string GetValueOrEmpty(bool? val)
        {
            return (val.HasValue) ? val.ToString().ToLower() : string.Empty;
        }

        public static string GetValueOrEmptyDate(DateTime? val)
        {
            return (val.HasValue) ? val.Value.ToString("yyyy-MM-dd") : string.Empty;
        }

        private static string GetValueOrEmptyDate(string val)
        {
            DateTime dateTimeOut;
            if (DateTime.TryParse(val, out dateTimeOut))
            {
                return dateTimeOut.ToString("yyyy-MM-dd");
            }

            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmptyDateTime(string val)
        {
            DateTime dateTimeOut;
            if (DateTime.TryParse(val, out dateTimeOut))
            {
                return dateTimeOut.ToString("yyyy-MM-ddTHH:mm:ssZ");
            }

            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        private static string GetValueOrEmptyDateTime(DateTime? val)
        {
            return (val.HasValue) ? val.Value.ToString("yyyy-MM-ddTHH:mm:ssZ") : string.Empty;
        }

        public static string GetValueOrEmpty(string val)
        {
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        public static string GetValueOrEmpty(decimal? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        public static string GetValueOrEmpty(int? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        /// <summary>
        /// Get string value between [first] a and [last] b.
        /// </summary>
        public static string Between(string value, string a, string b)
        {
            int posA = value.IndexOf(a);
            int posB = value.LastIndexOf(b);
            if (posA == -1)
            {
                return "";
            }

            if (posB == -1)
            {
                return "";
            }

            int adjustedPosA = posA + a.Length;
            if (adjustedPosA >= posB)
            {
                return "";
            }

            return value.Substring(adjustedPosA, posB - adjustedPosA);
        }

        /// <summary>
        /// Get string value after [first] a.
        /// </summary>
        public static string Before(string value, string a)
        {
            int posA = value.IndexOf(a);
            if (posA == -1)
            {
                return "";
            }

            return value.Substring(0, posA);
        }

        /// <summary>
        /// Get string value after [last] a.
        /// </summary>
        public static string After(string value, string a)
        {
            int posA = value.LastIndexOf(a);
            if (posA == -1)
            {
                return "";
            }

            int adjustedPosA = posA + a.Length;
            if (adjustedPosA >= value.Length)
            {
                return "";
            }

            return value.Substring(adjustedPosA);
        }

        private static int Length(object field)
        {
            if (field == null)
                return 0;
            else
                return field.ToString().Trim().Length;
        }

        private static bool NotSameValues(IEnumerable<object> nodes)
        {
            if (nodes == null)
                return false;
            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool NotSameValues(params object[] nodes)
        {
            if (nodes == null)
                return false;
            return (
                from x in nodes
                select x).Distinct().Count() != 1;
        }

        private static bool HasDuplicateValues(IEnumerable<DateTime> nodes)
        {
            if (nodes == null)
                return false;
            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool HasDuplicateValues(IEnumerable<DateTime?> nodes)
        {
            if (nodes == null)
                return false;
            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool NotSameValues(IEnumerable<DateTime> nodes)
        {
            if (nodes == null)
                return false;
            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool NotSameValues(IEnumerable<DateTime?> nodes)
        {
            if (nodes == null)
                return false;
            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        private static bool HasDuplicateValues(IEnumerable<object> nodes)
        {
            if (nodes == null)
                return false;
            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        private static bool HasDuplicateValues(params object[] nodes)
        {
            if (nodes == null)
                return false;
            nodes = nodes.Where(x => x != null).ToArray();
            return !((
                from x in nodes
                select x).Distinct().Count() == nodes.Count());
        }

        private int DuplicateValueIndex(IEnumerable<object> values)
        {
            int response = 0;
            var hashset = new HashSet<object>();
            foreach (var value in values)
            {
                if (!hashset.Add(value))
                {
                    return response;
                }

                response++;
            }

            return response;
        }

        private int DuplicateValueIndex<T>(IEnumerable<T?> values)
            where T : struct
        {
            int response = 0;
            var hashset = new HashSet<T?>();
            foreach (var value in values)
            {
                if (!hashset.Add(value))
                {
                    return response;
                }

                response++;
            }

            return response;
        }

        private static bool IsDate(object value)
        {
            DateTime dateValue;
            return (value != null && DateTime.TryParse(value.ToString(), out dateValue));
        }

        public DateTime AsDate(string dateAsString)
        {
            DateTime response = DateTime.MinValue;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date;
            }

            return response;
        }

        public DateTime Date(DateTime? datetime)
        {
            return datetime.GetValueOrDefault().Date;
        }

        //The logic in After function expects "---" for day and "--" for month. 
        //Since hyphen was missing the date always returned null
        public DateTime? ConvertToDate(int day, int month, int year)
        {
            return ConvertToDate(day == 0 ? null : "---" + day.ToString(), month == 0 ? null : "--" + month.ToString(), year == 0 ? null : year.ToString());
        }

        public DateTime? ConvertToDate(string day, string month, string year)
        {
            DateTime? response;
            DateTime result;
            if (year == null || month == null || day == null)
            {
                return null;
            }

            string dateAsString = year + "-" + After(month, "--") + "-" + After(day, "---");
            if (DateTime.TryParse(dateAsString, out result))
            {
                response = result;
            }
            else
            {
                response = null;
            }

            return response;
        }

        public DateTime? ConvertToDate(string day, string month, int year)
        {
            DateTime? response;
            DateTime result;
            if (year == 0 || month == null || day == null)
            {
                return null;
            }

            string dateAsString = year.ToString() + "-" + After(month, "--") + "-" + After(day, "---");
            if (DateTime.TryParse(dateAsString, out result))
            {
                response = result;
            }
            else
            {
                response = null;
            }

            return response;
        }

        private static int Day(string dateAsString)
        {
            int response = 0;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Day;
            }

            return response;
        }

        private static int Day(DateTime? date)
        {
            if (date == null)
                return 0;
            else
                return date.Value.Day;
        }

        private static string Month(string dateAsString)
        {
            string response = null;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.ToString("MMMM");
            }
            else
            {
                return "NotAMonth";
            }

            return response;
        }

        private static string Month(DateTime? date)
        {
            if (date == null)
                return "NotAMonth";
            else
                return date.Value.ToString("MMMM");
        }

        private static int MonthAsInt(string dateAsString)
        {
            int response = 0;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Month;
            }

            return response;
        }

        private static int MonthAsInt(DateTime? date)
        {
            if (date == null)
            {
                return 0;
            }

            return date.Value.Month;
        }

        private static int Year(string dateAsString)
        {
            int response = 0;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Year;
            }

            return response;
        }

        private static int Year(DateTime? date)
        {
            if (date == null)
                return 0;
            else
                return date.Value.Year;
        }

        private static int CurrentFinancialYear()
        {
            return DateToFinancialYear(DateTime.Now, 7);
        }

        private static int FinancialYear(string dateAsString)
        {
            return DateToFinancialYear(dateAsString, 7);
        }

        private static int FinancialYear(DateTime? date)
        {
            return DateToFinancialYear(date, 7);
        }

        private static int DateToFinancialYear(string dateAsString, int startingMonth)
        {
            int response = 0;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
            {
                response = DateToFinancialYear(date, startingMonth);
            }

            return response;
        }

        private static int DateToFinancialYear(DateTime? date, int startingMonth)
        {
            int response;
            if (date == null)
            {
                response = 0;
            }
            else
            {
                int year = date.Value.Year;
                int month = date.Value.Month;
                if (startingMonth > month)
                    response = year;
                else
                    response = year + 1;
            }

            return response;
        }

        private static int FBTYear(string dateAsString)
        {
            int response = 0;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
                response = FBTYear(date);
            return response;
        }

        private static int FBTYear(DateTime? date)
        {
            if (date == null)
            {
                return 0;
            }
            else
            {
                if (date.Value.Month > 3)
                    return date.Value.Year + 1;
                else
                    return date.Value.Year;
            }
        }

        private static DateTime? AddMonthsToDate(DateTime? dateTime, int months)
        {
            return dateTime == null ? null : (DateTime? )dateTime.Value.AddMonths(months);
        }

        private static bool IsNumeric(object value)
        {
            decimal numbervalue;
            return (value != null && decimal.TryParse(value.ToString(), out numbervalue));
        }

        private static bool NotMonetary(decimal? field, string sign, int digits, int decimals)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                string decimalExpression;
                int digitsToUse = digits - decimals;
                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";
                if (decimals > 0)
                    decimalExpression = @"(\.\d{1," + decimals + "})?$";
                else
                    decimalExpression = @"$";
                return !(Regex.IsMatch(field.Value.ToString("0.#########################"), signExpression + @"\d{1," + digitsToUse + "}" + decimalExpression));
            }
        }

        private static bool NotNumeric(decimal? field, string sign, int digits, int decimals)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                string decimalExpression;
                int digitsToUse = digits - decimals;
                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";
                if (decimals > 0)
                    decimalExpression = @"(\.\d{1," + decimals + "})?$";
                else
                    decimalExpression = @"$";
                return !(Regex.IsMatch(field.Value.ToString("0.#########################"), signExpression + @"\d{1," + digitsToUse + "}" + decimalExpression));
            }
        }

        private static bool NotNumeric(int? field, string sign, int digits, int decimals = 0)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";
                return !(Regex.IsMatch(field.Value.ToString(), signExpression + @"\d{1," + digits + "}$"));
            }
        }

        private static bool NotNumeric(long? field, string sign, int digits, int decimals = 0)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";
                return !(Regex.IsMatch(field.Value.ToString(), signExpression + @"\d{1," + digits + "}$"));
            }
        }

        private static bool OutsideRange(decimal field, decimal expression, int range)
        {
            bool response;
            response = (field < (expression - range)) || (field > (expression + range));
            return response;
        }

        private static bool FailsUSIAlgorithm(string usi, string abn)
        {
            bool response = true;
            if (usi == null || abn == null)
            {
                response = false;
            }
            else
            {
                usi = usi.Trim();
                abn = abn.Trim();
                if (usi.Length == 14)
                {
                    int numeric;
                    if (usi.Substring(0, 11) == abn && int.TryParse(usi.Substring(11, 3), out numeric))
                        response = false;
                }
                else if (usi.Length == 9)
                {
                    if (Regex.IsMatch(usi, @"^([a-zA-Z]{3}\d{4}[a-zA-Z]{2})$"))
                        response = false;
                }
            }

            return response;
        }

        private static bool FailsTANAlgorithm(string tan)
        {
            bool response;
            decimal decimalTan;
            if (tan == null)
                return false;
            tan = tan.Trim();
            if (!decimal.TryParse(tan, out decimalTan))
                return true;
            if (tan.Length != 8)
                return true;
            decimal tanSum = 7 * int.Parse(tan.Substring(0, 1)) + 9 * int.Parse(tan.Substring(1, 1)) + 8 * int.Parse(tan.Substring(2, 1)) + 4 * int.Parse(tan.Substring(3, 1)) + 6 * int.Parse(tan.Substring(4, 1)) + 3 * int.Parse(tan.Substring(5, 1)) + 5 * int.Parse(tan.Substring(6, 1)) + 1 * int.Parse(tan.Substring(7, 1));
            if ((tanSum % 11) == 0)
                response = false;
            else
                response = true;
            return response;
        }

        private static bool FailsABNAlgorithm(string abn)
        {
            bool response;
            decimal decimalAbn;
            if (abn == null)
                return false;
            abn = abn.Trim();
            if (!decimal.TryParse(abn, out decimalAbn))
                return true;
            if (abn.Length != 11)
                return true;
            decimal abnSum = 10 * (int.Parse(abn.Substring(0, 1)) - 1) + 1 * int.Parse(abn.Substring(1, 1)) + 3 * int.Parse(abn.Substring(2, 1)) + 5 * int.Parse(abn.Substring(3, 1)) + 7 * int.Parse(abn.Substring(4, 1)) + 9 * int.Parse(abn.Substring(5, 1)) + 11 * int.Parse(abn.Substring(6, 1)) + 13 * int.Parse(abn.Substring(7, 1)) + 15 * int.Parse(abn.Substring(8, 1)) + 17 * int.Parse(abn.Substring(9, 1)) + 19 * int.Parse(abn.Substring(10, 1));
            if ((abnSum % 89) == 0)
                response = false;
            else
                response = true;
            return response;
        }

        private static bool FailsACNAlgorithm(string acn)
        {
            bool response;
            decimal decimalAbn;
            if (acn == null)
                return false;
            acn = acn.Trim();
            if (!decimal.TryParse(acn, out decimalAbn))
                return true;
            if (acn.Length != 9)
                return true;
            decimal abnSum = 8 * int.Parse(acn.Substring(0, 1)) + 7 * int.Parse(acn.Substring(1, 1)) + 6 * int.Parse(acn.Substring(2, 1)) + 5 * int.Parse(acn.Substring(3, 1)) + 4 * int.Parse(acn.Substring(4, 1)) + 3 * int.Parse(acn.Substring(5, 1)) + 2 * int.Parse(acn.Substring(6, 1)) + 1 * int.Parse(acn.Substring(7, 1));
            decimal checkDigit = int.Parse(acn.Substring(8, 1));
            decimal acnRemainder = abnSum % 10;
            if (((10 - acnRemainder) % 10) == checkDigit)
                response = false;
            else
                response = true;
            return response;
        }

        private static bool FailsTFNAlgorithm(string tfn)
        {
            bool response;
            decimal decimalTfn;
            if (tfn == null)
                return false;
            tfn = tfn.Trim();
            tfn = Regex.Replace(tfn, "^0+", "");
            if (!decimal.TryParse(tfn, out decimalTfn))
                return true;
            if (tfn.Length < 8)
                return true;
            decimal tfn1To7Sum = 1 * int.Parse(tfn.Substring(0, 1)) + 4 * int.Parse(tfn.Substring(1, 1)) + 3 * int.Parse(tfn.Substring(2, 1)) + 7 * int.Parse(tfn.Substring(3, 1)) + 5 * int.Parse(tfn.Substring(4, 1)) + 8 * int.Parse(tfn.Substring(5, 1)) + 6 * int.Parse(tfn.Substring(6, 1));
            decimal tfn8 = 9 * int.Parse(tfn.Substring(7, 1));
            if (tfn.Length == 8)
            {
                decimal tFNLg8WSum9 = 10 * int.Parse(tfn.Substring(7, 1));
                decimal tFNLg8WSum = tfn1To7Sum + tFNLg8WSum9;
                if ((tFNLg8WSum % 11) == 0)
                    response = false;
                else
                    response = true;
            }
            else if (tfn.Length == 9)
            {
                decimal tfn9 = 10 * int.Parse(tfn.Substring(8, 1));
                decimal tFNLg9WSum = tfn1To7Sum + tfn8 + tfn9;
                if ((tFNLg9WSum % 11) == 0)
                    response = false;
                else
                    response = true;
            }
            else
            {
                response = true;
            }

            return response;
        }

        private static decimal ConditionalValue(bool expression, decimal? trueVal, decimal? falseVal)
        {
            return expression ? trueVal.GetValueOrDefault() : falseVal.GetValueOrDefault();
        }

        private static decimal AsNumeric(string value)
        {
            decimal numberValue;
            decimal.TryParse(value, out numberValue);
            return numberValue;
        }

        private static bool RegexMatch(int? field, string expression, string flags = "")
        {
            return IsMatch(field, expression, GetRegexOption(flags));
        }

        private static bool RegexMatch(string field, string expression, string flags = "")
        {
            return IsMatch(field, expression, GetRegexOption(flags));
        }

        private static RegexOptions GetRegexOption(string flags)
        {
            RegexOptions options = RegexOptions.None;
            char[] characters = flags.ToCharArray();
            foreach (char character in characters)
            {
                switch (character)
                {
                    case 'i':
                        options = options | RegexOptions.IgnoreCase;
                        break;
                    case 'm':
                        options = options | RegexOptions.Multiline;
                        break;
                    case 's':
                        options = options | RegexOptions.Singleline;
                        break;
                    case 'n':
                        options = options | RegexOptions.ExplicitCapture;
                        break;
                    case 'x':
                        options = options | RegexOptions.IgnorePatternWhitespace;
                        break;
                }
            }

            return options;
        }

        private static string Substring(object field, int start, int length)
        {
            return field.ToString().Substring(start, length);
        }

        /// <summary>
        /// Returns an occurrence index as [occurrenceIndex] of occurrenceIndex > 0, otherwise the empty string
        /// </summary>
        /// <param name = "occurrenceIndex">Index of the occurrence.</param>
        /// <returns>Occurrence in XPath [#] format</returns>
        public string OccurrenceIndex(int occurrenceIndex)
        {
            return occurrenceIndex > 0 ? "[" + occurrenceIndex + "]" : "";
        }

#endregion // Functions

        public TRTAMI2026 ConsumedReport
        {
            get
            {
                return report;
            }

            private set
            {
                report = value;
            }
        }

        private static Dictionary<string, ProcessMessageDocument> _processMessageDocuments = new Dictionary<string, ProcessMessageDocument>();
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public List<ProcessMessageDocument> ValidateReport(TRTAMI2026 reportIn)
        {
            //ProcessMessageDocument processMessage;
            //ProcessMessageParameter parameter;
            //bool assertion;
            this.ConsumedReport = reportIn;
            // ------------------------------------------------------------------------------
            // Validations are now done in code.
            // This version supports full validation of a report - for those rules that have successfully parsed and could be generated.
            // Generated Validations include:
            //   - Validation logic in C#.
            //   - Production of errors in EBMS, SBR and SWS format.
            // 
            // The generation at this stage does not support the following - and has to be completed manually (later versions of the generation will do this for you)
            //   - Business Rules that did not parse.  The ESR team will help support you where this happens as an effort is being made to rectify these
            //   - You will get TODO tasks for those that the parser could not cope with
            // ------------------------------------------------------------------------------
#region Manual Rules - Rules marked for manual coding
#region VR.ATO.TRTAMI.000189
            /*  VR.ATO.TRTAMI.000189
            If any Trustee Liabilities amounts, or any Statement of Attribution (SA) Taxable Determined Member Component amounts have a positive value, then 'Is any tax payable by the trustee?' must be 'yes'.

        Legacy Rule Format:
            ^TRTAMI33 <> '1' AND (^TRTAMI35 > 0 OR ^TRTAMI36 > 0 OR ^TRTAMI39 > 0 OR ^TRTAMI40 > 0 OR AnyOccurrence(^TRTAMI82, ^TRTAMI82 > 0))
    
        Technical Business Rule Format:
            ^TRTAMI33 <> '1' AND (^TRTAMI35 > 0 OR ^TRTAMI36 > 0 OR ^TRTAMI39 > 0 OR ^TRTAMI40 > 0 OR AnyOccurrence(^TRTAMI82, ^TRTAMI82 > 0))

        Data Elements:

        ^TRTAMI33 = TRTAMI:RP:OtherReportingInformation:Report.Statement.Summary.PaymentRefundOrNil.Code

        ^TRTAMI35 = TRTAMI:RP:TrusteeLiabilities:IncomeTax.TrustComponentDeficitOffset.Amount

        ^TRTAMI36 = TRTAMI:RP:TrusteeLiabilities:IncomeTax.TrustShortfallIncome.Amount

        ^TRTAMI39 = TRTAMI:RP:TrusteeLiabilities:IncomeTax.TrustExcessOffset.Amount

        ^TRTAMI40 = TRTAMI:RP:TrusteeLiabilities:IncomeTax.TrustComponentNotReflected.Amount

        ^TRTAMI82 = TRTAMI:RP:StatementOfAttributionDetails:AssessmentDetails:InternationalDealings.AssessableIncomeNonResidentBeneficiary.Amount
        */
            VRATOTRTAMI000189();
#endregion // VR.ATO.TRTAMI.000189

#region VR.ATO.TRTAMI.000190
            /*  VR.ATO.TRTAMI.000190
            If all Trustee Liabilities amounts and all Statement of Attribution (SA) Taxable Determined Member Component amounts are either NULL or zero, then 'Is any tax payable by the trustee?' must be 'no'.

        Legacy Rule Format:
            ^TRTAMI33 <> '0' AND ((^TRTAMI35 = NULL OR ^TRTAMI35 = 0) AND (^TRTAMI36 = NULL OR ^TRTAMI36 = 0) AND (^TRTAMI39 = NULL OR ^TRTAMI39 = 0) AND (^TRTAMI40 = NULL OR ^TRTAMI40 = 0) AND (AllOccurrences(^TRTAMI82, ^TRTAMI82 = NULL OR ^TRTAMI82 = 0)))

    Technical Business Rule Format:
            ^TRTAMI33 <> '0' AND ((^TRTAMI35 = NULL OR ^TRTAMI35 = 0) AND (^TRTAMI36 = NULL OR ^TRTAMI36 = 0) AND (^TRTAMI39 = NULL OR ^TRTAMI39 = 0) AND (^TRTAMI40 = NULL OR ^TRTAMI40 = 0) AND (AllOccurrences(^TRTAMI82, ^TRTAMI82 = NULL OR ^TRTAMI82 = 0)))

Data Elements:
    
    ^TRTAMI33 = TRTAMI:RP:OtherReportingInformation:Report.Statement.Summary.PaymentRefundOrNil.Code
    
    ^TRTAMI35 = TRTAMI:RP:TrusteeLiabilities:IncomeTax.TrustComponentDeficitOffset.Amount
    
    ^TRTAMI36 = TRTAMI:RP:TrusteeLiabilities:IncomeTax.TrustShortfallIncome.Amount
    
    ^TRTAMI39 = TRTAMI:RP:TrusteeLiabilities:IncomeTax.TrustExcessOffset.Amount
    
    ^TRTAMI40 = TRTAMI:RP:TrusteeLiabilities:IncomeTax.TrustComponentNotReflected.Amount
    
    ^TRTAMI82 = TRTAMI:RP:StatementOfAttributionDetails:AssessmentDetails:InternationalDealings.AssessableIncomeNonResidentBeneficiary.Amount
    */
            VRATOTRTAMI000190();
#endregion // VR.ATO.TRTAMI.000190

#region VR.ATO.TRTAMI.000291
            /*  VR.ATO.TRTAMI.000291
            When the response to Is the Trust Stapled to another Entity? is 'True' (yes), at least one Stapled Entity Australian Business Number must be provided

        Legacy Rule Format:
            ^TRTAMI112 = TRUE AND AllOccurrences(^TRTAMI139, ^TRTAMI139 = NULL)

    Technical Business Rule Format:
            ^TRTAMI112 = TRUE AND AllOccurrences(^TRTAMI139, ^TRTAMI139 = NULL)

Data Elements:
    
    ^TRTAMI139 = TRTAMI:RP:IncomeTaxStapledEntities:Identifiers.AustralianBusinessNumber.Identifier
    
    ^TRTAMI112 = TRTAMI:RP:IncomeTaxStapledEntities:IncomeTax.AttributionManagedInvestmentTrustStapledStructure.Indicator
    */
            VRATOTRTAMI000291();
#endregion // VR.ATO.TRTAMI.000291

#endregion Manual Rules - Rules marked for manual coding
            VRATOGEN010028();
            VRATOGEN010029();
            VRATOGEN428247();
            VRATOGEN438071();
            VRATOGEN438072();
            VRATOGEN438073();
            VRATOGEN438074();
            VRATOGEN438075();
            VRATOGEN438077();
            VRATOGEN438079();
            VRATOGEN500054();
            VRATOGEN500092();
            VRATOGEN500093();
            VRATOGEN500094();
            VRATOGEN500100();
            VRATOGEN500102();
            VRATOGEN500128();
            VRATOTRTAMI000006();
            VRATOTRTAMI000007();
            VRATOTRTAMI000009();
            VRATOTRTAMI000010();
            VRATOTRTAMI000012();
            VRATOTRTAMI000015();
            VRATOTRTAMI000017();
            VRATOTRTAMI000018();
            VRATOTRTAMI000028();
            VRATOTRTAMI000030();
            VRATOTRTAMI000031();
            VRATOTRTAMI000033();
            VRATOTRTAMI000034();
            VRATOTRTAMI000035();
            VRATOTRTAMI000037();
            VRATOTRTAMI000038();
            VRATOTRTAMI000040();
            VRATOTRTAMI000042();
            VRATOTRTAMI000043();
            VRATOTRTAMI000044();
            VRATOTRTAMI000045();
            VRATOTRTAMI000046();
            VRATOTRTAMI000048();
            VRATOTRTAMI000049();
            VRATOTRTAMI000050();
            VRATOTRTAMI000051();
            VRATOTRTAMI000053();
            VRATOTRTAMI000054();
            VRATOTRTAMI000055();
            VRATOTRTAMI000057();
            VRATOTRTAMI000058();
            VRATOTRTAMI000059();
            VRATOTRTAMI000061();
            VRATOTRTAMI000062();
            VRATOTRTAMI000065();
            VRATOTRTAMI000066();
            VRATOTRTAMI000067();
            VRATOTRTAMI000068();
            VRATOTRTAMI000069();
            VRATOTRTAMI000071();
            VRATOTRTAMI000072();
            VRATOTRTAMI000073();
            VRATOTRTAMI000074();
            VRATOTRTAMI000075();
            VRATOTRTAMI000076();
            VRATOTRTAMI000078();
            VRATOTRTAMI000079();
            VRATOTRTAMI000080();
            VRATOTRTAMI000081();
            VRATOTRTAMI000082();
            VRATOTRTAMI000083();
            VRATOTRTAMI000084();
            VRATOTRTAMI000086();
            VRATOTRTAMI000087();
            VRATOTRTAMI000088();
            VRATOTRTAMI000089();
            VRATOTRTAMI000091();
            VRATOTRTAMI000102();
            VRATOTRTAMI000113();
            VRATOTRTAMI000116();
            VRATOTRTAMI000150();
            VRATOTRTAMI000167();
            VRATOTRTAMI000209();
            VRATOTRTAMI000210();
            VRATOTRTAMI000238();
            VRATOTRTAMI000290();
            VRATOTRTAMI000301();
            VRATOTRTAMI000303();
            VRATOTRTAMI000312();
            VRATOTRTAMI000331();
            VRATOTRTAMI000347();
            VRATOTRTAMI000350();
            VRATOTRTAMI000351();
            VRATOTRTAMI000352();
            VRATOTRTAMI000353();
            VRATOTRTAMI000354();
            VRATOTRTAMI000355();
            VRATOTRTAMI000356();
            VRATOTRTAMI000357();
            VRATOTRTAMI000359();
            VRATOTRTAMI000400();
            VRATOTRTAMI000402();
            VRATOTRTAMI000403();
            VRATOTRTAMI000404();
            VRATOTRTAMI000405();
            VRATOTRTAMI000406();
            VRATOTRTAMI000407();
            VRATOTRTAMI000408();
            VRATOTRTAMI000409();
            VRATOTRTAMI000410();
            VRATOTRTAMI000416();
            VRATOTRTAMI000419();
            VRATOTRTAMI000421();
            VRATOTRTAMIW00001();
            VRATOTRTAMIW00003();
            VRATOTRTAMIW00004();
#region Repeating report.RP_StatementOfAttributionDetailsCollection  
            if (report.RP_StatementOfAttributionDetailsCollection != null)
            {
                int itemIndex2 = 0;
                foreach (TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails in report.RP_StatementOfAttributionDetailsCollection)
                {
                    itemIndex2++;
                    VRATOGEN500012(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000123(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000124(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000125(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000126(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000127(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000129(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000130(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000131(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000132(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000133(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000134(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000136(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000137(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000138(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000139(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000140(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000141(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000142(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000143(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000144(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000145(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000146(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000147(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000149(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000166(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000168(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000171(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000197(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000241(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000242(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000248(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000249(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000250(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000251(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000253(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000254(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000255(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000256(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000257(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000258(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000259(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000267(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000269(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000270(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000272(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000273(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000274(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000275(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000276(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000277(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000278(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000305(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000306(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000308(statementOfAttributionDetails, itemIndex2);
                    VRATOTRTAMI000309(statementOfAttributionDetails, itemIndex2);
                }
            }

#endregion // Foreach loop

#region Repeating TRTAMI139Collection  
            if (report.TRTAMI139Collection != null)
            {
                int itemIndex3 = 0;
                foreach (string TRTAMI139 in report.TRTAMI139Collection)
                {
                    itemIndex3++;
                    VRATOTRTAMI000293(TRTAMI139, itemIndex3);
                    VRATOTRTAMI000294(TRTAMI139, itemIndex3);
                    VRATOTRTAMI000295(TRTAMI139, itemIndex3);
                }
            }

#endregion // Foreach loop

            foreach (ProcessMessageDocument currentProcessMessage in response)
            {
                if (currentProcessMessage.Parameters != null)
                {
                    foreach (ProcessMessageParameter currentParameter in currentProcessMessage.Parameters)
                    {
                        if (string.IsNullOrEmpty(currentParameter.Name))
                        {
                            currentParameter.Name = _emptyParameterValue;
                        }

                        if (currentParameter.Name.Length > _maxParameterNameLength)
                        {
                            currentParameter.Name = currentParameter.Name.Substring(0, _maxParameterNameLength - 1);
                        }

                        if (string.IsNullOrEmpty(currentParameter.Value))
                        {
                            currentParameter.Value = _emptyParameterValue;
                        }

                        if (currentParameter.Value.Length > _maxParameterValueLength)
                        {
                            currentParameter.Value = currentParameter.Value.Substring(0, _maxParameterValueLength - 1);
                        }
                    }
                }
            }

            return response;
        }

#region VR.ATO.GEN.010028
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN010028()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.010028
            If the return is an amendment, sequence number, type and reason must be present
    
            Legacy Rule Format:
            ^TRTAMI6 = TRUE AND (^TRTAMI9 = NULL OR ^TRTAMI7 = NULL OR ^TRTAMI8 = NULL)

            Technical Business Rule Format:
            ^TRTAMI6 = TRUE AND (^TRTAMI9 = NULL OR ^TRTAMI7 = NULL OR ^TRTAMI8 = NULL)
    
            Data Elements:
    
            ^TRTAMI6 = TRTAMI:RP:ReportAmendmentDetails:Report.Amendment.Indicator
    
            ^TRTAMI7 = TRTAMI:RP:ReportAmendmentDetails:Report.AmendmentType.Code
    
            ^TRTAMI8 = TRTAMI:RP:ReportAmendmentDetails:Report.AmendmentReason.Text
    
            ^TRTAMI9 = TRTAMI:RP:ReportAmendmentDetails:Report.AmendmentSequence.Number
            */
            assertion = (report.TRTAMI6 == true && (report.TRTAMI9 == null || report.TRTAMI7 == null || report.TRTAMI8 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010028",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"If the return is an amendment, sequence number, type and reason must be present",
                    LongDescription = @"If the Amendment Indicator is 'yes' (true), then the Amendment Sequence Number, Amendment Type and Amendment Reason must be present and not blank.",
                    Location = "/tns:TRTAMI/tns:RP/tns:ReportAmendmentDetails/tns:I",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.010028"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI6", Value = GetValueOrEmpty(report.TRTAMI6) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI9", Value = report.TRTAMI9.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI7", Value = GetValueOrEmpty(report.TRTAMI7) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI8", Value = GetValueOrEmpty(report.TRTAMI8) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.010029
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN010029()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.010029
            If the return is an amendment, Amendment Indicator must be 'yes' (true)
    
            Legacy Rule Format:
            ^TRTAMI6 <> TRUE AND (^TRTAMI9 <> NULL OR ^TRTAMI7 <> NULL OR ^TRTAMI8 <> NULL)

            Technical Business Rule Format:
            ^TRTAMI6 <> TRUE AND (^TRTAMI9 <> NULL OR ^TRTAMI7 <> NULL OR ^TRTAMI8 <> NULL)
    
            Data Elements:
    
            ^TRTAMI6 = TRTAMI:RP:ReportAmendmentDetails:Report.Amendment.Indicator
    
            ^TRTAMI7 = TRTAMI:RP:ReportAmendmentDetails:Report.AmendmentType.Code
    
            ^TRTAMI8 = TRTAMI:RP:ReportAmendmentDetails:Report.AmendmentReason.Text
    
            ^TRTAMI9 = TRTAMI:RP:ReportAmendmentDetails:Report.AmendmentSequence.Number
            */
            assertion = (report.TRTAMI6 != true && (report.TRTAMI9 != null || report.TRTAMI7 != null || report.TRTAMI8 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.010029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"If the return is an amendment, Amendment Indicator must be 'yes' (true)",
                    LongDescription = @"If an Amendment Sequence Number, Amendment Type or Amendment Reason is present, then the Amendment Indicator must be 'yes' (true)",
                    Location = "/tns:TRTAMI/tns:RP/tns:ReportAmendmentDetails/tns:I",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.010029"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI6", Value = GetValueOrEmpty(report.TRTAMI6) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI9", Value = report.TRTAMI9.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI7", Value = GetValueOrEmpty(report.TRTAMI7) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI8", Value = GetValueOrEmpty(report.TRTAMI8) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.428247
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN428247()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.428247
            Your Tax Agent number has failed the Tax Agent registration algorithm check. Input the correct Tax Agent registration number.
    
            Legacy Rule Format:
            (^TRTAMI91 <> NULL) AND (FailsTANAlgorithm(^TRTAMI91))

            Technical Business Rule Format:
            (^TRTAMI91 <> NULL) AND (FailsTANAlgorithm(^TRTAMI91))
    
            Data Elements:
    
            ^TRTAMI91 = TRTAMI:INT:Identifiers.TaxAgentNumber.Identifier
            */
            assertion = (report.TRTAMI91 != null && FailsTANAlgorithm(report.TRTAMI91));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.410009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The Tax Agent number has failed the Tax Agent number algorithm check",
                    Location = "/tns:TRTAMI/tns:INT/tns:TaxAgentNumberId",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.428247"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI91", Value = GetValueOrEmpty(report.TRTAMI91) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.438071
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438071()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.438071
            If an Intermediary is not present, Reporting Party Declarer contact name must be provided.
    
            Legacy Rule Format:
            (^TRTAMI1030 = NULL) AND (^TRTAMI94 = NULL)

            Technical Business Rule Format:
            (^TRTAMI1030 = NULL) AND (^TRTAMI94 = NULL)
    
            Data Elements:
    
            ^TRTAMI94 = TRTAMI:RP:ContactDetails:PersonUnstructuredName.FullName.Text
    
            ^TRTAMI1030 = TRTAMI:INT
            */
            assertion = (report.IntCollectionExists == false && report.TRTAMI94 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438071",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer contact name must be provided",
                    LongDescription = @"If an Intermediary is not present, Reporting Party Declarer contact name must be provided.",
                    Location = "/tns:TRTAMI/tns:RP/tns:ContactDetails/tns:PersonUnstructuredNameFullNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.438071"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI94", Value = report.TRTAMI94 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.438072
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438072()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.438072
            If an area code is supplied and Intermediary is present, Intermediary Telephone number is required.
    
            Legacy Rule Format:
            (^TRTAMI1030 <> NULL) AND (^TRTAMI86 = NULL) AND (^TRTAMI85 <> NULL)

            Technical Business Rule Format:
            (^TRTAMI1030 <> NULL) AND (^TRTAMI86 = NULL) AND (^TRTAMI85 <> NULL)
    
            Data Elements:
    
            ^TRTAMI85 = TRTAMI:INT:ElectronicContact.Telephone.Area.Code
    
            ^TRTAMI86 = TRTAMI:INT:ElectronicContact.Telephone.Minimal.Number
    
            ^TRTAMI1030 = TRTAMI:INT
            */
            assertion = (report.IntCollectionExists != false && report.TRTAMI86 == null && report.TRTAMI85 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438072",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Telephone number is required, if an area code is provided",
                    LongDescription = @"If an area code is supplied and Intermediary is present, Intermediary Telephone number is required.",
                    Location = "/tns:TRTAMI/tns:INT/tns:ElectronicContactTelephoneAreaC",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.438072"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI86", Value = report.TRTAMI86 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI85", Value = report.TRTAMI85 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.438073
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438073()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.438073
            If an area code is supplied and Intermediary is not present, Reporting Party Telephone number is required.
    
            Legacy Rule Format:
            (^TRTAMI1030 = NULL) AND (^TRTAMI97 = NULL) AND (^TRTAMI96 <> NULL)

            Technical Business Rule Format:
            (^TRTAMI1030 = NULL) AND (^TRTAMI97 = NULL) AND (^TRTAMI96 <> NULL)
    
            Data Elements:
    
            ^TRTAMI96 = TRTAMI:RP:ContactDetails:ElectronicContact.Telephone.Area.Code
    
            ^TRTAMI97 = TRTAMI:RP:ContactDetails:ElectronicContact.Telephone.Minimal.Number
    
            ^TRTAMI1030 = TRTAMI:INT
            */
            assertion = (report.IntCollectionExists == false && report.TRTAMI97 == null && report.TRTAMI96 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438073",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Telephone number is required, if an area code is provided",
                    LongDescription = @"If an area code is supplied and Intermediary is not present, Reporting Party Telephone number is required.",
                    Location = "/tns:TRTAMI/tns:RP/tns:ContactDetails/tns:ElectronicContactTelephoneAreaC",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.438073"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI97", Value = report.TRTAMI97 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI96", Value = report.TRTAMI96 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.438074
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438074()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.438074
            If a telephone number is supplied and Intermediary is present, Intermediary area code is required.
    
            Legacy Rule Format:
            (^TRTAMI1030 <> NULL) AND (^TRTAMI86 <> NULL) AND (^TRTAMI85 = NULL)

            Technical Business Rule Format:
            (^TRTAMI1030 <> NULL) AND (^TRTAMI86 <> NULL) AND (^TRTAMI85 = NULL)
    
            Data Elements:
    
            ^TRTAMI85 = TRTAMI:INT:ElectronicContact.Telephone.Area.Code
    
            ^TRTAMI86 = TRTAMI:INT:ElectronicContact.Telephone.Minimal.Number
    
            ^TRTAMI1030 = TRTAMI:INT
            */
            assertion = (report.IntCollectionExists != false && report.TRTAMI86 != null && report.TRTAMI85 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438074",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Area code is required, if a Telephone number is provided",
                    LongDescription = @"If a telephone number is supplied and Intermediary is present, Intermediary area code is required.",
                    Location = "/tns:TRTAMI/tns:INT/tns:ElectronicContactTelephoneAreaC",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.438074"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI86", Value = report.TRTAMI86 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI85", Value = report.TRTAMI85 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.438075
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438075()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.438075
            If a telephone number is supplied and Intermediary is not present, Reporting Party area code is required.
    
            Legacy Rule Format:
            (^TRTAMI1030 = NULL) AND (^TRTAMI97 <> NULL) AND (^TRTAMI96 = NULL)

            Technical Business Rule Format:
            (^TRTAMI1030 = NULL) AND (^TRTAMI97 <> NULL) AND (^TRTAMI96 = NULL)
    
            Data Elements:
    
            ^TRTAMI96 = TRTAMI:RP:ContactDetails:ElectronicContact.Telephone.Area.Code
    
            ^TRTAMI97 = TRTAMI:RP:ContactDetails:ElectronicContact.Telephone.Minimal.Number
    
            ^TRTAMI1030 = TRTAMI:INT
            */
            assertion = (report.IntCollectionExists == false && report.TRTAMI97 != null && report.TRTAMI96 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438075",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Area code is required, if a Telephone number is provided",
                    LongDescription = @"If a telephone number is supplied and Intermediary is not present, Reporting Party area code is required.",
                    Location = "/tns:TRTAMI/tns:RP/tns:ContactDetails/tns:ElectronicContactTelephoneAreaC",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.438075"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI97", Value = report.TRTAMI97 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI96", Value = report.TRTAMI96 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.438077
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438077()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.438077
            If an Intermediary is not present, Reporting Party Declarer Telephone number must be provided.
    
            Legacy Rule Format:
            (^TRTAMI1030 = NULL) AND (^TRTAMI97 = NULL)

            Technical Business Rule Format:
            (^TRTAMI1030 = NULL) AND (^TRTAMI97 = NULL)
    
            Data Elements:
    
            ^TRTAMI97 = TRTAMI:RP:ContactDetails:ElectronicContact.Telephone.Minimal.Number
    
            ^TRTAMI1030 = TRTAMI:INT
            */
            assertion = (report.IntCollectionExists == false && report.TRTAMI97 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438077",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Contact telephone details must be provided",
                    LongDescription = @"If an Intermediary is not present, Reporting Party Declarer Telephone number must be provided.",
                    Location = "/tns:TRTAMI/tns:RP/tns:ContactDetails/tns:ElectronicContactTelephoneMinimalN",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.438077"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI97", Value = report.TRTAMI97 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.438079
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN438079()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.438079
            If an Intermediary is not present, Reporting Party Declarer Signature Date must be present.
    
            Legacy Rule Format:
            (^TRTAMI1030 = NULL) AND (^TRTAMI98 = NULL)

            Technical Business Rule Format:
            (^TRTAMI1030 = NULL) AND (^TRTAMI98 = NULL)
    
            Data Elements:
    
            ^TRTAMI98 = TRTAMI:RP:ContactDetails:Declaration:Declaration.Signature.Date
    
            ^TRTAMI1030 = TRTAMI:INT
            */
            assertion = (report.IntCollectionExists == false && report.TRTAMI98 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.438079",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Declaration Signature Date is required",
                    LongDescription = @"If an Intermediary is not present, Reporting Party Declarer Signature Date must be present.",
                    Location = "/tns:TRTAMI/tns:RP/tns:ContactDetails/tns:Declaration/tns:SignatureD",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.438079"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI98", Value = GetValueOrEmptyDate(report.TRTAMI98) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.500012
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500012(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.500012
                    The date entered is a future date. Date of birth cannot be any day in the future.
    
                    Legacy Rule Format:
                    (^TRTAMI124 <> NULL) AND (^TRTAMI124 > ^TRTAMI2)
        
                    Technical Business Rule Format:
                    (^TRTAMI124 <> NULL) AND (^TRTAMI124 > ^TRTAMI2)
            
                    Data Elements:
            
                    ^TRTAMI124 = TRTAMI:RP:StatementOfAttributionDetails:IndividualPersonNameDetails:PersonDemographicDetails.Birth.Date
            
                    ^TRTAMI2 = TRTAMI:RP:Period.Start.Date
                    */
            assertion = (statementOfAttributionDetails.TRTAMI124 != null && statementOfAttributionDetails.TRTAMI124.GetValueOrDefault() > report.TRTAMI2.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500012",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Date of birth must be prior to the form submission start date",
                    LongDescription = @"The date entered is a future date. Date of birth cannot be any day in advance of the form submission start date.",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:IndividualPersonNameDetails/tns:PersonDemographicDetailsBirthD",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.500012"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI124", Value = GetValueOrEmptyDate(statementOfAttributionDetails.TRTAMI124) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI2", Value = GetValueOrEmptyDate(report.TRTAMI2) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.500054
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500054()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.500054
            Intermediary ABN must pass the ABN algorithm check
    
            Legacy Rule Format:
            ^TRTAMI92 <> NULL AND FailsABNAlgorithm(^TRTAMI92)

            Technical Business Rule Format:
            ^TRTAMI92 <> NULL AND FailsABNAlgorithm(^TRTAMI92)
    
            Data Elements:
    
            ^TRTAMI92 = TRTAMI:INT:Identifiers.AustralianBusinessNumber.Identifier
            */
            assertion = (report.TRTAMI92 != null && FailsABNAlgorithm(report.TRTAMI92));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000477",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"ABN is invalid",
                    Location = "/tns:TRTAMI/tns:INT/tns:AustralianBusinessNumberId",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.500054"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI92", Value = report.TRTAMI92 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.500092
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500092()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.500092
            This is an annual accounting submission. The dates provided should be for your annual accounting period, commencing on the 1st day of the month. For normal balancers, this will be the period 1 July - 30 June.
    
            Legacy Rule Format:
            ^TRTAMI3 > AddMonthsToDate(^TRTAMI2, 23)

            Technical Business Rule Format:
            ^TRTAMI3 > AddMonthsToDate(^TRTAMI2, 23)
    
            Data Elements:
    
            ^TRTAMI2 = TRTAMI:RP:Period.Start.Date
    
            ^TRTAMI3 = TRTAMI:RP:Period.End.Date
            */
            assertion = (report.TRTAMI3.GetValueOrDefault() > AddMonthsToDate(report.TRTAMI2.GetValueOrDefault(), 23));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500092",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The account period dates must not exceed 23 months for an approved transitional period",
                    Location = "/tns:TRTAMI/tns:RP/tns:PeriodStartD",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.500092"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI3", Value = GetValueOrEmptyDate(report.TRTAMI3) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI2", Value = GetValueOrEmptyDate(report.TRTAMI2) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.500093
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500093()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.500093
            Accounting period start date must commence on or after 1st January in the previous year
    
            Legacy Rule Format:
            ^TRTAMI2 <  ConvertToDate(1, 1, ^TRTAMI1 - 1)

            Technical Business Rule Format:
            ^TRTAMI2 <  ConvertToDate(1, 1, ^TRTAMI1 - 1)
    
            Data Elements:
    
            ^TRTAMI2 = TRTAMI:RP:Period.Start.Date
    
            ^TRTAMI1 = TRTAMI:RP:Report.TargetFinancial.Year
            */
            assertion = (report.TRTAMI2.GetValueOrDefault() < ConvertToDate(1, 1, report.TRTAMI1.GetValueOrDefault() - 1));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500093",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Date financial year from date must commence on or after 1st January in the previous year",
                    Location = "/tns:TRTAMI/tns:RP/tns:PeriodStartD",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.500093"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI2", Value = GetValueOrEmptyDate(report.TRTAMI2) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI1", Value = GetValueOrEmpty(report.TRTAMI1) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.500094
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500094()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.500094
            Intermediary Declaration Statement Accepted Indicator must be 'TRUE'
    
            Legacy Rule Format:
            (^TRTAMI1031 <> NULL) AND (^TRTAMI88 <> TRUE)

            Technical Business Rule Format:
            (^TRTAMI1031 <> NULL) AND (^TRTAMI88 <> TRUE)
    
            Data Elements:
    
            ^TRTAMI88 = TRTAMI:INT:Declaration:Declaration.StatementAccepted.Indicator
    
            ^TRTAMI1031 = TRTAMI:INT:Declaration
            */
            assertion = (report.InT_DeclarationCollectionExists != false && report.TRTAMI88 != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500094",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Intermediary Declaration Statement Accepted Indicator must be 'TRUE'",
                    Location = "/tns:TRTAMI/tns:INT/tns:Declaration/tns:StatementAcceptedI",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.500094"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI88", Value = GetValueOrEmpty(report.TRTAMI88) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.500100
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500100()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.500100
            Reporting Party Declaration Statement Accepted Indicator must be 'TRUE'
    
            Legacy Rule Format:
            (^TRTAMI1028 <> NULL) AND (^TRTAMI99 <> TRUE)

            Technical Business Rule Format:
            (^TRTAMI1028 <> NULL) AND (^TRTAMI99 <> TRUE)
    
            Data Elements:
    
            ^TRTAMI99 = TRTAMI:RP:ContactDetails:Declaration:Declaration.StatementAccepted.Indicator
    
            ^TRTAMI1028 = TRTAMI:RP:ContactDetails
            */
            assertion = (report.RP_ContactDetailsCollectionExists != false && report.TRTAMI99 != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500100",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Reporting Party Declaration Statement Accepted Indicator must be 'TRUE'",
                    Location = "/tns:TRTAMI/tns:RP/tns:ContactDetails/tns:Declaration/tns:StatementAcceptedI",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.500100"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI99", Value = GetValueOrEmpty(report.TRTAMI99) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.500102
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500102()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.500102
            The accounting period should commence from the 1st of the month.
    
            Legacy Rule Format:
            (Day(^TRTAMI2) <> 1)

            Technical Business Rule Format:
            (Day(^TRTAMI2) <> 1)
    
            Data Elements:
    
            ^TRTAMI2 = TRTAMI:RP:Period.Start.Date
            */
            assertion = (Day(report.TRTAMI2.GetValueOrDefault()) != 1);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500102",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The period start date must commence from the first of the month",
                    LongDescription = @"This is an annual accounting submission. The dates provided should be for your annual accounting period, commencing on the 1st day of the month. For normal balancers, this will be the period 1 July - 30 June.",
                    Location = "/tns:TRTAMI/tns:RP/tns:PeriodStartD",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.500102"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI2", Value = GetValueOrEmptyDate(report.TRTAMI2) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.GEN.500128
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOGEN500128()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.GEN.500128
            Where either BSB number, account number or account name is provided, all must be present
    
            Legacy Rule Format:
            (^TRTAMI41 <> NULL OR ^TRTAMI42 <> NULL OR ^TRTAMI43 <> NULL) AND (^TRTAMI41 = NULL OR ^TRTAMI42 = NULL OR ^TRTAMI43 = NULL)

            Technical Business Rule Format:
            (^TRTAMI41 <> NULL OR ^TRTAMI42 <> NULL OR ^TRTAMI43 <> NULL) AND (^TRTAMI41 = NULL OR ^TRTAMI42 = NULL OR ^TRTAMI43 = NULL)
    
            Data Elements:
    
            ^TRTAMI42 = TRTAMI:RP:FinancialInstitutionAccount:FinancialInstitutionAccount.FinancialInstitutionAccount.Number
    
            ^TRTAMI41 = TRTAMI:RP:FinancialInstitutionAccount:FinancialInstitutionAccount.BankStateBranch.Number
    
            ^TRTAMI43 = TRTAMI:RP:FinancialInstitutionAccount:FinancialInstitutionAccount.FinancialInstitutionAccountName.Text
            */
            assertion = ((report.TRTAMI41 != null || report.TRTAMI42 != null || report.TRTAMI43 != null) && (report.TRTAMI41 == null || report.TRTAMI42 == null || report.TRTAMI43 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500128",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Where either BSB number, account number or account name is provided, all must be present",
                    LongDescription = @"If any EFT details are present, then a BSB number, account number and an account name must all be present. If EFT is not required, then all fields must be blank or not present.",
                    Location = "/tns:TRTAMI/tns:RP/tns:FinancialInstitutionAccount/tns:N",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.GEN.500128"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI41", Value = report.TRTAMI41 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI42", Value = report.TRTAMI42 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI43", Value = report.TRTAMI43 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000006
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000006()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000006
            The Trust Australian Business Number has failed the ABN algorithm check.
    
            Legacy Rule Format:
            (^TRTAMI5 <> NULL) AND (FailsABNAlgorithm(^TRTAMI5))

            Technical Business Rule Format:
            (^TRTAMI5 <> NULL) AND (FailsABNAlgorithm(^TRTAMI5))
    
            Data Elements:
    
            ^TRTAMI5 = TRTAMI:RP:Identifiers.AustralianBusinessNumber.Identifier
            */
            assertion = (report.TRTAMI5 != null && FailsABNAlgorithm(report.TRTAMI5));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000006",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Trust Australian Business Number is invalid",
                    LongDescription = @"The Trust Australian Business Number has failed the ABN algorithm check",
                    Location = "/tns:TRTAMI/tns:RP/tns:AustralianBusinessNumberId",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000006"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI5", Value = report.TRTAMI5 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000007
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000007()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000007
            The Trustee Australian Business Number has failed the ABN algorithm check.
    
            Legacy Rule Format:
            (^TRTAMI29 <> NULL) AND (FailsABNAlgorithm(^TRTAMI29))

            Technical Business Rule Format:
            (^TRTAMI29 <> NULL) AND (FailsABNAlgorithm(^TRTAMI29))
    
            Data Elements:
    
            ^TRTAMI29 = TRTAMI:RP:TrusteeDetails:Identifiers.AustralianBusinessNumber.Identifier
            */
            assertion = (report.TRTAMI29 != null && FailsABNAlgorithm(report.TRTAMI29));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000007",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Trustee Australian Business Number is invalid",
                    LongDescription = @"The Trustee Australian Business Number has failed the ABN algorithm check",
                    Location = "/tns:TRTAMI/tns:RP/tns:TrusteeDetails/tns:AustralianBusinessNumberId",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000007"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI29", Value = GetValueOrEmpty(report.TRTAMI29) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000009
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000009()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000009
            Non Individual Name cannot contain space hyphen space combinations
    
            Legacy Rule Format:
            Contains(^TRTAMI10, ' - ')

            Technical Business Rule Format:
            Contains(^TRTAMI10, ' - ')
    
            Data Elements:
    
            ^TRTAMI10 = TRTAMI:RP:CurrentTrust:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.TRTAMI10 == null ? false : report.TRTAMI10.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Name of Trust cannot contain space hyphen space combinations",
                    Location = "/tns:TRTAMI/tns:RP/tns:CurrentTrust/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000009"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI10", Value = report.TRTAMI10 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000010
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000010()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000010
            Non Individual Name cannot contain "P/L"
    
            Legacy Rule Format:
            Contains(^TRTAMI10, 'P/L')

            Technical Business Rule Format:
            Contains(^TRTAMI10, 'P/L')
    
            Data Elements:
    
            ^TRTAMI10 = TRTAMI:RP:CurrentTrust:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.TRTAMI10 == null ? false : report.TRTAMI10.ToUpperInvariant().Contains(@"P/L"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000010",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Name of Trust cannot contain ""P/L""",
                    Location = "/tns:TRTAMI/tns:RP/tns:CurrentTrust/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000010"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI10", Value = report.TRTAMI10 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000012
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000012()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000012
            Non Individual Name cannot start with "T/A" and end with "Pship", "P'ship" or "P/Ship"
    
            Legacy Rule Format:
            (StartsWith(^TRTAMI10, 'T/A ')) AND (EndsWithSet(^TRTAMI10, '" Pship"," P'ship"," P/ship"'))

            Technical Business Rule Format:
            (StartsWith(^TRTAMI10, 'T/A ')) AND (EndsWithSet(^TRTAMI10, '" Pship"," P'ship"," P/ship"'))
    
            Data Elements:
    
            ^TRTAMI10 = TRTAMI:RP:CurrentTrust:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (IsMatch(report.TRTAMI10, @"^(T/A )", RegexOptions.IgnoreCase) && IsMatch(report.TRTAMI10, @"( Pship| P'ship| P/ship)$", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000012",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Name of Trust cannot start with ""T/A"" and end with ""Pship"", ""P'ship"" or ""P/Ship""",
                    Location = "/tns:TRTAMI/tns:RP/tns:CurrentTrust/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000012"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI10", Value = report.TRTAMI10 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000015
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000015()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000015
            Non Individual Name cannot end with "T/A", "T/A P'ship", "T/A Pship", "T/A P/Ship" or "T/A Partnership"
    
            Legacy Rule Format:
            (EndsWithSet(^TRTAMI10, '" T/A", " T/A P'ship", " T/A Pship", " T/A P/Ship", " T/A Partnership"'))

            Technical Business Rule Format:
            (EndsWithSet(^TRTAMI10, '" T/A", " T/A P'ship", " T/A Pship", " T/A P/Ship", " T/A Partnership"'))
    
            Data Elements:
    
            ^TRTAMI10 = TRTAMI:RP:CurrentTrust:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.TRTAMI10, @"( T/A| T/A P'ship| T/A Pship| T/A P/Ship| T/A Partnership)$", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000015",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Name of Trust cannot end with ""T/A"", ""T/A P'ship"", ""T/A Pship"", ""T/A P/Ship"" or ""T/A Partnership""",
                    Location = "/tns:TRTAMI/tns:RP/tns:CurrentTrust/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000015"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI10", Value = report.TRTAMI10 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000017
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000017()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000017
            Non Individual Name cannot contain space hyphen space combinations
    
            Legacy Rule Format:
            Contains(^TRTAMI11, ' - ')

            Technical Business Rule Format:
            Contains(^TRTAMI11, ' - ')
    
            Data Elements:
    
            ^TRTAMI11 = TRTAMI:RP:PreviousTrust:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.TRTAMI11 == null ? false : report.TRTAMI11.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000017",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Previous Name of Trust cannot contain space hyphen space combinations",
                    Location = "/tns:TRTAMI/tns:RP/tns:PreviousTrust/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000017"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI11", Value = GetValueOrEmpty(report.TRTAMI11) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000018
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000018()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000018
            Non Individual Name cannot contain "P/L"
    
            Legacy Rule Format:
            Contains(^TRTAMI11, 'P/L')

            Technical Business Rule Format:
            Contains(^TRTAMI11, 'P/L')
    
            Data Elements:
    
            ^TRTAMI11 = TRTAMI:RP:PreviousTrust:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.TRTAMI11 == null ? false : report.TRTAMI11.ToUpperInvariant().Contains(@"P/L"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000018",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Previous Name of Trust cannot contain ""P/L""",
                    Location = "/tns:TRTAMI/tns:RP/tns:PreviousTrust/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000018"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI11", Value = GetValueOrEmpty(report.TRTAMI11) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000028
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000028()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000028
            Address Line 1 must not contain 'AS ABOVE'
    
            Legacy Rule Format:
            Contains(^TRTAMI12, 'AS ABOVE')

            Technical Business Rule Format:
            Contains(^TRTAMI12, 'AS ABOVE')
    
            Data Elements:
    
            ^TRTAMI12 = TRTAMI:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.Line1.Text
            */
            assertion = (report.TRTAMI12 == null ? false : report.TRTAMI12.ToUpperInvariant().Contains(@"AS ABOVE"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000028",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Current Postal Address of Trust - Address Line 1 must not contain 'AS ABOVE'",
                    Location = "/tns:TRTAMI/tns:RP/tns:CurrentTrust/tns:CurrentPostalAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000028"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI12", Value = report.TRTAMI12 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000030
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000030()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000030
            Address line 1 contains incorrect 'care of' reference. The only acceptable 'care of' reference is C/-.
    
            Legacy Rule Format:
            FoundSet(^TRTAMI12, '"C/O","C/","Care Of","CO"')

            Technical Business Rule Format:
            FoundSet(^TRTAMI12, '"C/O","C/","Care Of","CO"')
    
            Data Elements:
    
            ^TRTAMI12 = TRTAMI:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.Line1.Text
            */
            assertion = IsMatch(report.TRTAMI12, @"(^(C/O|C/|Care Of|CO)$)|(^(.*)( C/O | C/ | Care Of | CO )(.*)$)|(^(C/O |C/ |Care Of |CO )(.*)$)|(^(.*)( C/O| C/| Care Of| CO)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000030",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Current Postal Address of Trust - Address Line 1 is invalid",
                    LongDescription = @"Current Postal Address of Trust - Address Line 1 contains incorrect 'care of' reference. The only acceptable 'care of' reference is C/-.",
                    Location = "/tns:TRTAMI/tns:RP/tns:CurrentTrust/tns:CurrentPostalAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000030"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI12", Value = report.TRTAMI12 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000031
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000031()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000031
            Address line 1 cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            Contains(^TRTAMI12, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^TRTAMI12, 'UNKNOWN')
    
            Data Elements:
    
            ^TRTAMI12 = TRTAMI:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.Line1.Text
            */
            assertion = (report.TRTAMI12 == null ? false : report.TRTAMI12.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000031",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Current Postal Address of Trust - Address Line 1 cannot contain ""UNKNOWN""",
                    Location = "/tns:TRTAMI/tns:RP/tns:CurrentTrust/tns:CurrentPostalAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000031"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI12", Value = report.TRTAMI12 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000033
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000033()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000033
            Care of' is not valid for address line 2
    
            Legacy Rule Format:
            FoundSet(^TRTAMI13, '"C/-","C/O","C/","Care Of","CO"')

            Technical Business Rule Format:
            FoundSet(^TRTAMI13, '"C/-","C/O","C/","Care Of","CO"')
    
            Data Elements:
    
            ^TRTAMI13 = TRTAMI:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.Line2.Text
            */
            assertion = IsMatch(report.TRTAMI13, @"(^(C/-|C/O|C/|Care Of|CO)$)|(^(.*)( C/- | C/O | C/ | Care Of | CO )(.*)$)|(^(C/- |C/O |C/ |Care Of |CO )(.*)$)|(^(.*)( C/-| C/O| C/| Care Of| CO)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000033",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Care of' is not valid for Current Postal Address of Trust - Address Line 2",
                    Location = "/tns:TRTAMI/tns:RP/tns:CurrentTrust/tns:CurrentPostalAddressDetails/tns:Line2T",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000033"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI13", Value = GetValueOrEmpty(report.TRTAMI13) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000034
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000034()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000034
            Address line 2 cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            Contains(^TRTAMI13, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^TRTAMI13, 'UNKNOWN')
    
            Data Elements:
    
            ^TRTAMI13 = TRTAMI:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.Line2.Text
            */
            assertion = (report.TRTAMI13 == null ? false : report.TRTAMI13.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000034",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Current Postal Address of Trust - Address Line 2 cannot contain ""UNKNOWN""",
                    Location = "/tns:TRTAMI/tns:RP/tns:CurrentTrust/tns:CurrentPostalAddressDetails/tns:Line2T",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000034"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI13", Value = GetValueOrEmpty(report.TRTAMI13) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000035
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000035()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000035
            Address Line 2 must be present if Address Line 1 contains 'C/-'
    
            Legacy Rule Format:
            (Contains(^TRTAMI12, 'C/-')) AND (^TRTAMI13 = NULL)

            Technical Business Rule Format:
            (Contains(^TRTAMI12, 'C/-')) AND (^TRTAMI13 = NULL)
    
            Data Elements:
    
            ^TRTAMI13 = TRTAMI:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.Line2.Text
    
            ^TRTAMI12 = TRTAMI:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.Line1.Text
            */
            assertion = ((report.TRTAMI12 == null ? false : report.TRTAMI12.ToUpperInvariant().Contains(@"C/-")) && report.TRTAMI13 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000035",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Current Postal Address of Trust - Address Line 2 must be provided",
                    LongDescription = @"When Current Postal Address of Trust - Address Line 1 contains 'C/-', Current Postal Address of Trust - Address Line 2 must be provided",
                    Location = "/tns:TRTAMI/tns:RP/tns:CurrentTrust/tns:CurrentPostalAddressDetails/tns:Line2T",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000035"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI12", Value = report.TRTAMI12 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI13", Value = GetValueOrEmpty(report.TRTAMI13) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000037
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000037()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000037
            State code [QLD,NSW,VIC,SA,WA,NT,ACT,TAS] or a leading space or combination of spaces with a State code is present in the suburb/town field.
    
            Legacy Rule Format:
            (^TRTAMI17 = NULL OR ^TRTAMI17 = 'au') AND (FoundSet(^TRTAMI14, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))

            Technical Business Rule Format:
            (^TRTAMI17 = NULL OR ^TRTAMI17 = 'au') AND (FoundSet(^TRTAMI14, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))
    
            Data Elements:
    
            ^TRTAMI14 = TRTAMI:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.LocalityName.Text
    
            ^TRTAMI17 = TRTAMI:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.Country.Code
            */
            assertion = ((report.TRTAMI17 == null || report.TRTAMI17 == @"au") && IsMatch(report.TRTAMI14, @"(^(QLD|NSW|VIC|SA|WA|NT|ACT|TAS)$)|(^(.*)( QLD | NSW | VIC | SA | WA | NT | ACT | TAS )(.*)$)|(^(QLD |NSW |VIC |SA |WA |NT |ACT |TAS )(.*)$)|(^(.*)( QLD| NSW| VIC| SA| WA| NT| ACT| TAS)$)", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000037",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Current Postal Address of Trust - Suburb/Town cannot contain state code",
                    Location = "/tns:TRTAMI/tns:RP/tns:CurrentTrust/tns:CurrentPostalAddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000037"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI17", Value = GetValueOrEmpty(report.TRTAMI17) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI14", Value = report.TRTAMI14 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000038
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000038()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000038
            Suburb/Town/City cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            Contains(^TRTAMI14, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^TRTAMI14, 'UNKNOWN')
    
            Data Elements:
    
            ^TRTAMI14 = TRTAMI:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.LocalityName.Text
            */
            assertion = (report.TRTAMI14 == null ? false : report.TRTAMI14.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000038",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Current Postal Address of Trust - Suburb/Town cannot contain ""UNKNOWN""",
                    Location = "/tns:TRTAMI/tns:RP/tns:CurrentTrust/tns:CurrentPostalAddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000038"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI14", Value = report.TRTAMI14 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000040
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000040()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000040
            Numerics [0 to 9] are not allowed in the suburb/town field.
    
            Legacy Rule Format:
            (^TRTAMI17 = NULL OR ^TRTAMI17 = 'au') AND (ContainsSet(^TRTAMI14, '0-9'))

            Technical Business Rule Format:
            (^TRTAMI17 = NULL OR ^TRTAMI17 = 'au') AND (ContainsSet(^TRTAMI14, '0-9'))
    
            Data Elements:
    
            ^TRTAMI14 = TRTAMI:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.LocalityName.Text
    
            ^TRTAMI17 = TRTAMI:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.Country.Code
            */
            assertion = ((report.TRTAMI17 == null || report.TRTAMI17 == @"au") && IsMatch(report.TRTAMI14, @"[0-9]", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000040",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Current Postal Address of Trust - Suburb/Town is invalid",
                    LongDescription = @"When an Australian address is provided, Current Postal Address of Trust - Suburb/Town must not contain numeric characters",
                    Location = "/tns:TRTAMI/tns:RP/tns:CurrentTrust/tns:CurrentPostalAddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000040"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI17", Value = GetValueOrEmpty(report.TRTAMI17) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI14", Value = report.TRTAMI14 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000042
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000042()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000042
            State is present in the address field without a postcode.
    
            Legacy Rule Format:
            (^TRTAMI15 <> NULL) AND (^TRTAMI16 = NULL)

            Technical Business Rule Format:
            (^TRTAMI15 <> NULL) AND (^TRTAMI16 = NULL)
    
            Data Elements:
    
            ^TRTAMI16 = TRTAMI:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.Postcode.Text
    
            ^TRTAMI15 = TRTAMI:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.StateOrTerritory.Code
            */
            assertion = (report.TRTAMI15 != null && report.TRTAMI16 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000042",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Current Postal Address of Trust - Postcode must be present",
                    LongDescription = @"If the Current Postal Address of Trust is an Australian address, then state and postcode must be present. If the address is an overseas address, then country must be present.",
                    Location = "/tns:TRTAMI/tns:RP/tns:CurrentTrust/tns:CurrentPostalAddressDetails/tns:PostcodeT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000042"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI15", Value = GetValueOrEmpty(report.TRTAMI15) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI16", Value = GetValueOrEmpty(report.TRTAMI16) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000043
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000043()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000043
            State, postcode, or country are missing from the address field. Determine whether the address is Australian or overseas and correct accordingly.
    
            Legacy Rule Format:
            (^TRTAMI15 = NULL) AND (^TRTAMI16 = NULL) AND (^TRTAMI17 = NULL OR ^TRTAMI17 = 'au')

            Technical Business Rule Format:
            (^TRTAMI15 = NULL) AND (^TRTAMI16 = NULL) AND (^TRTAMI17 = NULL OR ^TRTAMI17 = 'au')
    
            Data Elements:
    
            ^TRTAMI15 = TRTAMI:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.StateOrTerritory.Code
    
            ^TRTAMI16 = TRTAMI:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.Postcode.Text
    
            ^TRTAMI17 = TRTAMI:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.Country.Code
            */
            assertion = (report.TRTAMI15 == null && report.TRTAMI16 == null && (report.TRTAMI17 == null || report.TRTAMI17 == @"au"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000043",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Current Postal Address of Trust - State and postcode are required for Australian addresses",
                    LongDescription = @"Current Postal Address of Trust - State and postcode, or country are required. Determine whether the address is Australian or overseas and correct accordingly.",
                    Location = "/tns:TRTAMI/tns:RP/tns:CurrentTrust/tns:CurrentPostalAddressDetails/tns:StateOrTerritoryC",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000043"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI15", Value = GetValueOrEmpty(report.TRTAMI15) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI16", Value = GetValueOrEmpty(report.TRTAMI16) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI17", Value = GetValueOrEmpty(report.TRTAMI17) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000044
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000044()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000044
            Postcode is present in the address field without a State code.
    
            Legacy Rule Format:
            (^TRTAMI16 <> NULL) AND (^TRTAMI15 = NULL)

            Technical Business Rule Format:
            (^TRTAMI16 <> NULL) AND (^TRTAMI15 = NULL)
    
            Data Elements:
    
            ^TRTAMI15 = TRTAMI:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.StateOrTerritory.Code
    
            ^TRTAMI16 = TRTAMI:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.Postcode.Text
            */
            assertion = (report.TRTAMI16 != null && report.TRTAMI15 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000044",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Current Postal Address of Trust - State must be present",
                    LongDescription = @"If the Current Postal Address of Trust is an Australian address, then state and postcode must be present. If the address is an overseas address, then country must be present.",
                    Location = "/tns:TRTAMI/tns:RP/tns:CurrentTrust/tns:CurrentPostalAddressDetails/tns:StateOrTerritoryC",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000044"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI16", Value = GetValueOrEmpty(report.TRTAMI16) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI15", Value = GetValueOrEmpty(report.TRTAMI15) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000045
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000045()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000045
            Country name is present with State or postcode in the address field.  Determine whether the address is Australian or overseas and correct accordingly.
    
            Legacy Rule Format:
            (^TRTAMI17 <> NULL AND ^TRTAMI17 <> 'au') AND (^TRTAMI15 <> NULL OR ^TRTAMI16 <> NULL)

            Technical Business Rule Format:
            (^TRTAMI17 <> NULL AND ^TRTAMI17 <> 'au') AND (^TRTAMI15 <> NULL OR ^TRTAMI16 <> NULL)
    
            Data Elements:
    
            ^TRTAMI17 = TRTAMI:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.Country.Code
    
            ^TRTAMI15 = TRTAMI:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.StateOrTerritory.Code
    
            ^TRTAMI16 = TRTAMI:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.Postcode.Text
            */
            assertion = (report.TRTAMI17 != null && report.TRTAMI17 != @"au" && (report.TRTAMI15 != null || report.TRTAMI16 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000045",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Australia must be entered as the country where an Australian state or territory has been entered",
                    Location = "/tns:TRTAMI/tns:RP/tns:CurrentTrust/tns:CurrentPostalAddressDetails/tns:CountryC",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000045"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI17", Value = GetValueOrEmpty(report.TRTAMI17) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI15", Value = GetValueOrEmpty(report.TRTAMI15) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI16", Value = GetValueOrEmpty(report.TRTAMI16) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000046
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000046()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000046
            Previous postal address is only required when an address change has occurred since the last return.  Delete previous postal address or input the old address if a change of address has occurred.
    
            Legacy Rule Format:
            (^TRTAMI18 = ^TRTAMI12) AND (^TRTAMI19 = ^TRTAMI13) AND (^TRTAMI20 = ^TRTAMI14) AND (^TRTAMI21 = ^TRTAMI15) AND (^TRTAMI22 = ^TRTAMI16)

            Technical Business Rule Format:
            (^TRTAMI18 = ^TRTAMI12) AND (^TRTAMI19 = ^TRTAMI13) AND (^TRTAMI20 = ^TRTAMI14) AND (^TRTAMI21 = ^TRTAMI15) AND (^TRTAMI22 = ^TRTAMI16)
    
            Data Elements:
    
            ^TRTAMI18 = TRTAMI:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.Line1.Text
    
            ^TRTAMI12 = TRTAMI:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.Line1.Text
    
            ^TRTAMI13 = TRTAMI:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.Line2.Text
    
            ^TRTAMI14 = TRTAMI:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.LocalityName.Text
    
            ^TRTAMI15 = TRTAMI:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.StateOrTerritory.Code
    
            ^TRTAMI16 = TRTAMI:RP:CurrentTrust:CurrentPostalAddressDetails:AddressDetails.Postcode.Text
    
            ^TRTAMI19 = TRTAMI:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.Line2.Text
    
            ^TRTAMI20 = TRTAMI:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.LocalityName.Text
    
            ^TRTAMI21 = TRTAMI:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.StateOrTerritory.Code
    
            ^TRTAMI22 = TRTAMI:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.Postcode.Text
            */
            assertion = (report.TRTAMI18 == report.TRTAMI12 && report.TRTAMI19 == report.TRTAMI13 && report.TRTAMI20 == report.TRTAMI14 && report.TRTAMI21 == report.TRTAMI15 && report.TRTAMI22 == report.TRTAMI16);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Previous postal address must not be the same as current postal address",
                    LongDescription = @"Previous postal address is only required when an address change has occurred since the last return.  Delete previous postal address or input the old address if a change of address has occurred.",
                    Location = "/tns:TRTAMI/tns:RP/tns:PreviousTrust/tns:PreviousPostalAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000046"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI18", Value = report.TRTAMI18 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI12", Value = report.TRTAMI12 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI19", Value = GetValueOrEmpty(report.TRTAMI19) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI13", Value = GetValueOrEmpty(report.TRTAMI13) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI20", Value = report.TRTAMI20 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI14", Value = report.TRTAMI14 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI21", Value = GetValueOrEmpty(report.TRTAMI21) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI15", Value = GetValueOrEmpty(report.TRTAMI15) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI22", Value = GetValueOrEmpty(report.TRTAMI22) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI16", Value = GetValueOrEmpty(report.TRTAMI16) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000048
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000048()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000048
            Address Line 1 must not contain 'AS ABOVE'
    
            Legacy Rule Format:
            Contains(^TRTAMI18, 'AS ABOVE')

            Technical Business Rule Format:
            Contains(^TRTAMI18, 'AS ABOVE')
    
            Data Elements:
    
            ^TRTAMI18 = TRTAMI:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.Line1.Text
            */
            assertion = (report.TRTAMI18 == null ? false : report.TRTAMI18.ToUpperInvariant().Contains(@"AS ABOVE"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000048",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Previous Postal Address of Trust - Address Line 1 must not contain 'AS ABOVE'",
                    Location = "/tns:TRTAMI/tns:RP/tns:PreviousTrust/tns:PreviousPostalAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000048"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI18", Value = report.TRTAMI18 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000049
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000049()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000049
            Address line 1 contains incorrect 'care of' reference. The only acceptable 'care of' reference is C/-.
    
            Legacy Rule Format:
            FoundSet(^TRTAMI18, '"C/O","C/","Care Of","CO"')

            Technical Business Rule Format:
            FoundSet(^TRTAMI18, '"C/O","C/","Care Of","CO"')
    
            Data Elements:
    
            ^TRTAMI18 = TRTAMI:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.Line1.Text
            */
            assertion = IsMatch(report.TRTAMI18, @"(^(C/O|C/|Care Of|CO)$)|(^(.*)( C/O | C/ | Care Of | CO )(.*)$)|(^(C/O |C/ |Care Of |CO )(.*)$)|(^(.*)( C/O| C/| Care Of| CO)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000049",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Previous Postal Address of Trust - Address Line 1 is invalid",
                    LongDescription = @"Previous Postal Address of Trust - Address Line 1 contains incorrect 'care of' reference. The only acceptable 'care of' reference is C/-.",
                    Location = "/tns:TRTAMI/tns:RP/tns:PreviousTrust/tns:PreviousPostalAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000049"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI18", Value = report.TRTAMI18 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000050
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000050()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000050
            Address line 1 cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            Contains(^TRTAMI18, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^TRTAMI18, 'UNKNOWN')
    
            Data Elements:
    
            ^TRTAMI18 = TRTAMI:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.Line1.Text
            */
            assertion = (report.TRTAMI18 == null ? false : report.TRTAMI18.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000050",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Previous Postal Address of Trust - Address Line 1 cannot contain ""UNKNOWN""",
                    Location = "/tns:TRTAMI/tns:RP/tns:PreviousTrust/tns:PreviousPostalAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000050"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI18", Value = report.TRTAMI18 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000051
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000051()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000051
            When any Previous Postal Address for Trust details are provided, Address Line 1 and Suburb/Town must be provided, including State and Postcode if address is Australian
    
            Legacy Rule Format:
            (^TRTAMI18 <> NULL OR ^TRTAMI19 <> NULL OR ^TRTAMI20 <> NULL OR ^TRTAMI21 <> NULL OR ^TRTAMI22 <> NULL OR (^TRTAMI23 <> NULL AND ^TRTAMI23 <> 'au')) AND (^TRTAMI18 = NULL OR ^TRTAMI20 = NULL OR (^TRTAMI21 = NULL AND ^TRTAMI22 = NULL AND (^TRTAMI23 = NULL OR ^TRTAMI23 = 'au')))

            Technical Business Rule Format:
            (^TRTAMI18 <> NULL OR ^TRTAMI19 <> NULL OR ^TRTAMI20 <> NULL OR ^TRTAMI21 <> NULL OR ^TRTAMI22 <> NULL OR (^TRTAMI23 <> NULL AND ^TRTAMI23 <> 'au')) AND (^TRTAMI18 = NULL OR ^TRTAMI20 = NULL OR (^TRTAMI21 = NULL AND ^TRTAMI22 = NULL AND (^TRTAMI23 = NULL OR ^TRTAMI23 = 'au')))
    
            Data Elements:
    
            ^TRTAMI18 = TRTAMI:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.Line1.Text
    
            ^TRTAMI19 = TRTAMI:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.Line2.Text
    
            ^TRTAMI20 = TRTAMI:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.LocalityName.Text
    
            ^TRTAMI21 = TRTAMI:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.StateOrTerritory.Code
    
            ^TRTAMI22 = TRTAMI:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.Postcode.Text
    
            ^TRTAMI23 = TRTAMI:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.Country.Code
            */
            assertion = ((report.TRTAMI18 != null || report.TRTAMI19 != null || report.TRTAMI20 != null || report.TRTAMI21 != null || report.TRTAMI22 != null || report.TRTAMI23 != null && report.TRTAMI23 != @"au") && (report.TRTAMI18 == null || report.TRTAMI20 == null || report.TRTAMI21 == null && report.TRTAMI22 == null && (report.TRTAMI23 == null || report.TRTAMI23 == @"au")));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000051",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Full Previous Postal Address of Trust details must be provided",
                    LongDescription = @"When any Previous Postal Address for Trust details are provided, Address Line 1 and Suburb/Town must be provided, including State and Postcode if address is Australian",
                    Location = "/tns:TRTAMI/tns:RP/tns:PreviousTrust/tns:PreviousPostalAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000051"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI18", Value = report.TRTAMI18 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI19", Value = GetValueOrEmpty(report.TRTAMI19) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI20", Value = report.TRTAMI20 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI21", Value = GetValueOrEmpty(report.TRTAMI21) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI22", Value = GetValueOrEmpty(report.TRTAMI22) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI23", Value = GetValueOrEmpty(report.TRTAMI23) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000053
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000053()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000053
            Care of' is not valid for address line 2
    
            Legacy Rule Format:
            FoundSet(^TRTAMI19, '"C/-","C/O","C/","Care Of","CO"')

            Technical Business Rule Format:
            FoundSet(^TRTAMI19, '"C/-","C/O","C/","Care Of","CO"')
    
            Data Elements:
    
            ^TRTAMI19 = TRTAMI:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.Line2.Text
            */
            assertion = IsMatch(report.TRTAMI19, @"(^(C/-|C/O|C/|Care Of|CO)$)|(^(.*)( C/- | C/O | C/ | Care Of | CO )(.*)$)|(^(C/- |C/O |C/ |Care Of |CO )(.*)$)|(^(.*)( C/-| C/O| C/| Care Of| CO)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000053",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Care of' is not valid for Previous Postal Address of Trust - Address Line 2",
                    Location = "/tns:TRTAMI/tns:RP/tns:PreviousTrust/tns:PreviousPostalAddressDetails/tns:Line2T",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000053"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI19", Value = GetValueOrEmpty(report.TRTAMI19) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000054
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000054()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000054
            Address line 2 cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            Contains(^TRTAMI19, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^TRTAMI19, 'UNKNOWN')
    
            Data Elements:
    
            ^TRTAMI19 = TRTAMI:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.Line2.Text
            */
            assertion = (report.TRTAMI19 == null ? false : report.TRTAMI19.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000054",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Previous Postal Address of Trust - Address Line 2 cannot contain ""UNKNOWN""",
                    Location = "/tns:TRTAMI/tns:RP/tns:PreviousTrust/tns:PreviousPostalAddressDetails/tns:Line2T",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000054"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI19", Value = GetValueOrEmpty(report.TRTAMI19) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000055
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000055()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000055
            Address Line 2 must be present if Address Line 1 contains 'C/-'
    
            Legacy Rule Format:
            (Contains(^TRTAMI18, 'C/-')) AND (^TRTAMI19 = NULL)

            Technical Business Rule Format:
            (Contains(^TRTAMI18, 'C/-')) AND (^TRTAMI19 = NULL)
    
            Data Elements:
    
            ^TRTAMI19 = TRTAMI:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.Line2.Text
    
            ^TRTAMI18 = TRTAMI:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.Line1.Text
            */
            assertion = ((report.TRTAMI18 == null ? false : report.TRTAMI18.ToUpperInvariant().Contains(@"C/-")) && report.TRTAMI19 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000055",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Previous Postal Address of Trust - Address Line 2 must be provided",
                    LongDescription = @"When Previous Postal Address of Trust - Address Line 1 contains 'C/-', Current Previous Address of Trust - Address Line 2 must be provided",
                    Location = "/tns:TRTAMI/tns:RP/tns:PreviousTrust/tns:PreviousPostalAddressDetails/tns:Line2T",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000055"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI18", Value = report.TRTAMI18 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI19", Value = GetValueOrEmpty(report.TRTAMI19) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000057
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000057()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000057
            State code [QLD,NSW,VIC,SA,WA,NT,ACT,TAS] or a leading space or combination of spaces with a State code is present in the suburb/town field.
    
            Legacy Rule Format:
            (^TRTAMI23 = NULL OR ^TRTAMI23 = 'au') AND (FoundSet(^TRTAMI20, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))

            Technical Business Rule Format:
            (^TRTAMI23 = NULL OR ^TRTAMI23 = 'au') AND (FoundSet(^TRTAMI20, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))
    
            Data Elements:
    
            ^TRTAMI20 = TRTAMI:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.LocalityName.Text
    
            ^TRTAMI23 = TRTAMI:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.Country.Code
            */
            assertion = ((report.TRTAMI23 == null || report.TRTAMI23 == @"au") && IsMatch(report.TRTAMI20, @"(^(QLD|NSW|VIC|SA|WA|NT|ACT|TAS)$)|(^(.*)( QLD | NSW | VIC | SA | WA | NT | ACT | TAS )(.*)$)|(^(QLD |NSW |VIC |SA |WA |NT |ACT |TAS )(.*)$)|(^(.*)( QLD| NSW| VIC| SA| WA| NT| ACT| TAS)$)", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000057",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Previous Postal Address of Trust - Suburb/Town cannot contain state code",
                    Location = "/tns:TRTAMI/tns:RP/tns:PreviousTrust/tns:PreviousPostalAddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000057"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI23", Value = GetValueOrEmpty(report.TRTAMI23) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI20", Value = report.TRTAMI20 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000058
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000058()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000058
            Suburb/Town/City cannot contain "UNKNOWN"
    
            Legacy Rule Format:
            Contains(^TRTAMI20, 'UNKNOWN')

            Technical Business Rule Format:
            Contains(^TRTAMI20, 'UNKNOWN')
    
            Data Elements:
    
            ^TRTAMI20 = TRTAMI:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.LocalityName.Text
            */
            assertion = (report.TRTAMI20 == null ? false : report.TRTAMI20.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000058",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Previous Postal Address of Trust - Suburb/Town cannot contain ""UNKNOWN""",
                    Location = "/tns:TRTAMI/tns:RP/tns:PreviousTrust/tns:PreviousPostalAddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000058"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI20", Value = report.TRTAMI20 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000059
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000059()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000059
            Numerics [0 to 9] are not allowed in the suburb/town field.
    
            Legacy Rule Format:
            (^TRTAMI23 = NULL OR ^TRTAMI23 = 'au') AND (ContainsSet(^TRTAMI20, '0-9'))

            Technical Business Rule Format:
            (^TRTAMI23 = NULL OR ^TRTAMI23 = 'au') AND (ContainsSet(^TRTAMI20, '0-9'))
    
            Data Elements:
    
            ^TRTAMI20 = TRTAMI:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.LocalityName.Text
    
            ^TRTAMI23 = TRTAMI:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.Country.Code
            */
            assertion = ((report.TRTAMI23 == null || report.TRTAMI23 == @"au") && IsMatch(report.TRTAMI20, @"[0-9]", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000059",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Previous Postal Address of Trust - Suburb/Town is invalid",
                    LongDescription = @"When an Australian address is provided, Previous Postal Address of Trust - Suburb/Town must not contain numeric characters",
                    Location = "/tns:TRTAMI/tns:RP/tns:PreviousTrust/tns:PreviousPostalAddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000059"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI23", Value = GetValueOrEmpty(report.TRTAMI23) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI20", Value = report.TRTAMI20 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000061
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000061()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000061
            State is present in the address field without a postcode.
    
            Legacy Rule Format:
            (^TRTAMI21 <> NULL) AND (^TRTAMI22 = NULL)

            Technical Business Rule Format:
            (^TRTAMI21 <> NULL) AND (^TRTAMI22 = NULL)
    
            Data Elements:
    
            ^TRTAMI22 = TRTAMI:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.Postcode.Text
    
            ^TRTAMI21 = TRTAMI:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.StateOrTerritory.Code
            */
            assertion = (report.TRTAMI21 != null && report.TRTAMI22 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000061",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Previous Postal Address of Trust - Postcode must be present",
                    LongDescription = @"If the Previous Postal Address of Trust is an Australian address, then state and postcode must be present. If the address is an overseas address, then country must be present.",
                    Location = "/tns:TRTAMI/tns:RP/tns:PreviousTrust/tns:PreviousPostalAddressDetails/tns:PostcodeT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000061"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI21", Value = GetValueOrEmpty(report.TRTAMI21) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI22", Value = GetValueOrEmpty(report.TRTAMI22) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000062
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000062()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000062
            Postcode is present in the address field without a State code.
    
            Legacy Rule Format:
            (^TRTAMI22 <> NULL) AND (^TRTAMI21 = NULL)

            Technical Business Rule Format:
            (^TRTAMI22 <> NULL) AND (^TRTAMI21 = NULL)
    
            Data Elements:
    
            ^TRTAMI21 = TRTAMI:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.StateOrTerritory.Code
    
            ^TRTAMI22 = TRTAMI:RP:PreviousTrust:PreviousPostalAddressDetails:AddressDetails.Postcode.Text
            */
            assertion = (report.TRTAMI22 != null && report.TRTAMI21 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000062",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Previous Postal Address of Trust - State must be present",
                    LongDescription = @"If the Previous Postal Address of Trust is an Australian address, then state and postcode must be present. If the address is an overseas address, then country must be present.",
                    Location = "/tns:TRTAMI/tns:RP/tns:PreviousTrust/tns:PreviousPostalAddressDetails/tns:StateOrTerritoryC",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000062"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI22", Value = GetValueOrEmpty(report.TRTAMI22) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI21", Value = GetValueOrEmpty(report.TRTAMI21) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000065
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000065()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000065
            Family name cannot contain "Exec for", "Rep for" or "Trustee for"
    
            Legacy Rule Format:
            (FoundSet(^TRTAMI25, '"Exec for","Rep for","Trustee for"'))

            Technical Business Rule Format:
            (FoundSet(^TRTAMI25, '"Exec for","Rep for","Trustee for"'))
    
            Data Elements:
    
            ^TRTAMI25 = TRTAMI:RP:TrusteeDetails:PersonNameDetails:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.TRTAMI25, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000065",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Trustee Family Name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/tns:TRTAMI/tns:RP/tns:TrusteeDetails/tns:PersonNameDetails/tns:FamilyNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000065"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI25", Value = report.TRTAMI25 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000066
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000066()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000066
            Family name cannot contain "MR", "MRS", "MISS" or "MS"
    
            Legacy Rule Format:
            (FoundSet(^TRTAMI25, '"MR","MRS","MISS","MS"'))

            Technical Business Rule Format:
            (FoundSet(^TRTAMI25, '"MR","MRS","MISS","MS"'))
    
            Data Elements:
    
            ^TRTAMI25 = TRTAMI:RP:TrusteeDetails:PersonNameDetails:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.TRTAMI25, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000066",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Trustee Family Name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/tns:TRTAMI/tns:RP/tns:TrusteeDetails/tns:PersonNameDetails/tns:FamilyNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000066"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI25", Value = report.TRTAMI25 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000067
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000067()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000067
            Family name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
            Legacy Rule Format:
            (FoundSet(^TRTAMI25, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))

            Technical Business Rule Format:
            (FoundSet(^TRTAMI25, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
    
            Data Elements:
    
            ^TRTAMI25 = TRTAMI:RP:TrusteeDetails:PersonNameDetails:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.TRTAMI25, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000319",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Trustee Family Name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/tns:TRTAMI/tns:RP/tns:TrusteeDetails/tns:PersonNameDetails/tns:FamilyNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000067"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI25", Value = report.TRTAMI25 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000068
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000068()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000068
            Family Name cannot contain "space hyphen space"
    
            Legacy Rule Format:
            Contains(^TRTAMI25, ' - ')

            Technical Business Rule Format:
            Contains(^TRTAMI25, ' - ')
    
            Data Elements:
    
            ^TRTAMI25 = TRTAMI:RP:TrusteeDetails:PersonNameDetails:PersonNameDetails.FamilyName.Text
            */
            assertion = (report.TRTAMI25 == null ? false : report.TRTAMI25.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000068",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Trustee Family Name cannot contain ""space hyphen space""",
                    Location = "/tns:TRTAMI/tns:RP/tns:TrusteeDetails/tns:PersonNameDetails/tns:FamilyNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000068"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI25", Value = report.TRTAMI25 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000069
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000069()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000069
            Family name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            (ContainsSet(^TRTAMI25, '"--","''","  "'))

            Technical Business Rule Format:
            (ContainsSet(^TRTAMI25, '"--","''","  "'))
    
            Data Elements:
    
            ^TRTAMI25 = TRTAMI:RP:TrusteeDetails:PersonNameDetails:PersonNameDetails.FamilyName.Text
            */
            assertion = IsMatch(report.TRTAMI25, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000069",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Trustee Family Name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:TRTAMI/tns:RP/tns:TrusteeDetails/tns:PersonNameDetails/tns:FamilyNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000069"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI25", Value = report.TRTAMI25 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000071
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000071()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000071
            First name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
            Legacy Rule Format:
            (FoundSet(^TRTAMI26, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))

            Technical Business Rule Format:
            (FoundSet(^TRTAMI26, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
    
            Data Elements:
    
            ^TRTAMI26 = TRTAMI:RP:TrusteeDetails:PersonNameDetails:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.TRTAMI26, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000320",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Trustee First Name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/tns:TRTAMI/tns:RP/tns:TrusteeDetails/tns:PersonNameDetails/tns:GivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000071"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI26", Value = GetValueOrEmpty(report.TRTAMI26) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000072
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000072()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000072
            First name cannot contain "Exec for", "Rep for" or "Trustee for"
    
            Legacy Rule Format:
            (FoundSet(^TRTAMI26, '"Exec for","Rep for","Trustee for"'))

            Technical Business Rule Format:
            (FoundSet(^TRTAMI26, '"Exec for","Rep for","Trustee for"'))
    
            Data Elements:
    
            ^TRTAMI26 = TRTAMI:RP:TrusteeDetails:PersonNameDetails:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.TRTAMI26, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000072",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Trustee First Given Name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/tns:TRTAMI/tns:RP/tns:TrusteeDetails/tns:PersonNameDetails/tns:GivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000072"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI26", Value = GetValueOrEmpty(report.TRTAMI26) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000073
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000073()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000073
            First name cannot contain "MR", "MRS", "MISS" or "MS"
    
            Legacy Rule Format:
            (FoundSet(^TRTAMI26, '"MR","MRS","MISS","MS"'))

            Technical Business Rule Format:
            (FoundSet(^TRTAMI26, '"MR","MRS","MISS","MS"'))
    
            Data Elements:
    
            ^TRTAMI26 = TRTAMI:RP:TrusteeDetails:PersonNameDetails:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.TRTAMI26, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000073",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Trustee First Given Name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/tns:TRTAMI/tns:RP/tns:TrusteeDetails/tns:PersonNameDetails/tns:GivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000073"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI26", Value = GetValueOrEmpty(report.TRTAMI26) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000074
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000074()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000074
            FirstName must be entered if OtherGivenNames is entered
    
            Legacy Rule Format:
            (^TRTAMI27 <> NULL) AND (^TRTAMI26 = NULL)

            Technical Business Rule Format:
            (^TRTAMI27 <> NULL) AND (^TRTAMI26 = NULL)
    
            Data Elements:
    
            ^TRTAMI26 = TRTAMI:RP:TrusteeDetails:PersonNameDetails:PersonNameDetails.GivenName.Text
    
            ^TRTAMI27 = TRTAMI:RP:TrusteeDetails:PersonNameDetails:PersonNameDetails.OtherGivenName.Text
            */
            assertion = (report.TRTAMI27 != null && report.TRTAMI26 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000074",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Trustee First Given Name must be entered if Trustee Other Given Names is entered",
                    Location = "/tns:TRTAMI/tns:RP/tns:TrusteeDetails/tns:PersonNameDetails/tns:GivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000074"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI27", Value = GetValueOrEmpty(report.TRTAMI27) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI26", Value = GetValueOrEmpty(report.TRTAMI26) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000075
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000075()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000075
            First name cannot contain space hyphen space
    
            Legacy Rule Format:
            Contains(^TRTAMI26, ' - ')

            Technical Business Rule Format:
            Contains(^TRTAMI26, ' - ')
    
            Data Elements:
    
            ^TRTAMI26 = TRTAMI:RP:TrusteeDetails:PersonNameDetails:PersonNameDetails.GivenName.Text
            */
            assertion = (report.TRTAMI26 == null ? false : report.TRTAMI26.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000075",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Trustee First Given Name cannot contain space hyphen space",
                    Location = "/tns:TRTAMI/tns:RP/tns:TrusteeDetails/tns:PersonNameDetails/tns:GivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000075"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI26", Value = GetValueOrEmpty(report.TRTAMI26) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000076
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000076()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000076
            First name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            (ContainsSet(^TRTAMI26, '"--","''","  "'))

            Technical Business Rule Format:
            (ContainsSet(^TRTAMI26, '"--","''","  "'))
    
            Data Elements:
    
            ^TRTAMI26 = TRTAMI:RP:TrusteeDetails:PersonNameDetails:PersonNameDetails.GivenName.Text
            */
            assertion = IsMatch(report.TRTAMI26, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000076",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Trustee First Given Name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:TRTAMI/tns:RP/tns:TrusteeDetails/tns:PersonNameDetails/tns:GivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000076"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI26", Value = GetValueOrEmpty(report.TRTAMI26) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000078
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000078()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000078
            Other given name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
            Legacy Rule Format:
            (FoundSet(^TRTAMI27, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))

            Technical Business Rule Format:
            (FoundSet(^TRTAMI27, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
    
            Data Elements:
    
            ^TRTAMI27 = TRTAMI:RP:TrusteeDetails:PersonNameDetails:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.TRTAMI27, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000321",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Trustee Other Given Name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/tns:TRTAMI/tns:RP/tns:TrusteeDetails/tns:PersonNameDetails/tns:OtherGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000078"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI27", Value = GetValueOrEmpty(report.TRTAMI27) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000079
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000079()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000079
            Other given name cannot contain "Exec for", "Rep for" or "Trustee for"
    
            Legacy Rule Format:
            (FoundSet(^TRTAMI27, '"Exec for","Rep for","Trustee for"'))

            Technical Business Rule Format:
            (FoundSet(^TRTAMI27, '"Exec for","Rep for","Trustee for"'))
    
            Data Elements:
    
            ^TRTAMI27 = TRTAMI:RP:TrusteeDetails:PersonNameDetails:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.TRTAMI27, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000079",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Trustee Other Given Names cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/tns:TRTAMI/tns:RP/tns:TrusteeDetails/tns:PersonNameDetails/tns:OtherGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000079"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI27", Value = GetValueOrEmpty(report.TRTAMI27) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000080
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000080()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000080
            Other given name cannot contain "MR", "MRS", "MISS" or "MS"
    
            Legacy Rule Format:
            (FoundSet(^TRTAMI27, '"MR","MRS","MISS","MS"'))

            Technical Business Rule Format:
            (FoundSet(^TRTAMI27, '"MR","MRS","MISS","MS"'))
    
            Data Elements:
    
            ^TRTAMI27 = TRTAMI:RP:TrusteeDetails:PersonNameDetails:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.TRTAMI27, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000080",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Trustee Other Given Names cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/tns:TRTAMI/tns:RP/tns:TrusteeDetails/tns:PersonNameDetails/tns:OtherGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000080"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI27", Value = GetValueOrEmpty(report.TRTAMI27) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000081
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000081()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000081
            Other given name cannot contain space hyphen space
    
            Legacy Rule Format:
            Contains(^TRTAMI27, ' - ')

            Technical Business Rule Format:
            Contains(^TRTAMI27, ' - ')
    
            Data Elements:
    
            ^TRTAMI27 = TRTAMI:RP:TrusteeDetails:PersonNameDetails:PersonNameDetails.OtherGivenName.Text
            */
            assertion = (report.TRTAMI27 == null ? false : report.TRTAMI27.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000081",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Trustee Other Given Names cannot contain space hyphen space",
                    Location = "/tns:TRTAMI/tns:RP/tns:TrusteeDetails/tns:PersonNameDetails/tns:OtherGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000081"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI27", Value = GetValueOrEmpty(report.TRTAMI27) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000082
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000082()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000082
            Other given name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            (ContainsSet(^TRTAMI27, '"--","''","  "'))

            Technical Business Rule Format:
            (ContainsSet(^TRTAMI27, '"--","''","  "'))
    
            Data Elements:
    
            ^TRTAMI27 = TRTAMI:RP:TrusteeDetails:PersonNameDetails:PersonNameDetails.OtherGivenName.Text
            */
            assertion = IsMatch(report.TRTAMI27, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000082",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Trustee Other Given Names cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:TRTAMI/tns:RP/tns:TrusteeDetails/tns:PersonNameDetails/tns:OtherGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000082"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI27", Value = GetValueOrEmpty(report.TRTAMI27) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000083
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000083()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000083
            Family name or non-individual name must be present.
    
            Legacy Rule Format:
            ^TRTAMI25 = NULL AND ^TRTAMI28 = NULL

            Technical Business Rule Format:
            ^TRTAMI25 = NULL AND ^TRTAMI28 = NULL
    
            Data Elements:
    
            ^TRTAMI28 = TRTAMI:RP:TrusteeDetails:OrganisationNameDetails.OrganisationalName.Text
    
            ^TRTAMI25 = TRTAMI:RP:TrusteeDetails:PersonNameDetails:PersonNameDetails.FamilyName.Text
            */
            assertion = (report.TRTAMI25 == null && report.TRTAMI28 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000083",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Name of the trustee to whom notices should be sent must be present",
                    LongDescription = @"If the trustee is an individual, provide the name of the individual. If the trustee is a company, provide the name of the company.",
                    Location = "/tns:TRTAMI/tns:RP/tns:TrusteeDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000083"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI25", Value = report.TRTAMI25 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI28", Value = GetValueOrEmpty(report.TRTAMI28) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000084
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000084()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000084
            Non-individual name and Family name details are both present. Only one name can be present.
    
            Legacy Rule Format:
            (^TRTAMI28 <> NULL OR ^TRTAMI29 <> NULL) AND (^TRTAMI24 <> NULL OR ^TRTAMI25 <> NULL OR ^TRTAMI26 <> NULL OR ^TRTAMI27 <> NULL)

            Technical Business Rule Format:
            (^TRTAMI28 <> NULL OR ^TRTAMI29 <> NULL) AND (^TRTAMI24 <> NULL OR ^TRTAMI25 <> NULL OR ^TRTAMI26 <> NULL OR ^TRTAMI27 <> NULL)
    
            Data Elements:
    
            ^TRTAMI28 = TRTAMI:RP:TrusteeDetails:OrganisationNameDetails.OrganisationalName.Text
    
            ^TRTAMI24 = TRTAMI:RP:TrusteeDetails:PersonNameDetails:PersonNameDetails.Title.Text
    
            ^TRTAMI25 = TRTAMI:RP:TrusteeDetails:PersonNameDetails:PersonNameDetails.FamilyName.Text
    
            ^TRTAMI26 = TRTAMI:RP:TrusteeDetails:PersonNameDetails:PersonNameDetails.GivenName.Text
    
            ^TRTAMI27 = TRTAMI:RP:TrusteeDetails:PersonNameDetails:PersonNameDetails.OtherGivenName.Text
    
            ^TRTAMI29 = TRTAMI:RP:TrusteeDetails:Identifiers.AustralianBusinessNumber.Identifier
            */
            assertion = ((report.TRTAMI28 != null || report.TRTAMI29 != null) && (report.TRTAMI24 != null || report.TRTAMI25 != null || report.TRTAMI26 != null || report.TRTAMI27 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000084",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Trustee Non-Individual Name and Trustee Family Name details must not both be present",
                    Location = "/tns:TRTAMI/tns:RP/tns:TrusteeDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000084"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI28", Value = GetValueOrEmpty(report.TRTAMI28) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI29", Value = GetValueOrEmpty(report.TRTAMI29) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI24", Value = GetValueOrEmpty(report.TRTAMI24) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI25", Value = report.TRTAMI25 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI26", Value = GetValueOrEmpty(report.TRTAMI26) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI27", Value = GetValueOrEmpty(report.TRTAMI27) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000086
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000086()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000086
            Non Individual Name cannot contain space hyphen space combinations
    
            Legacy Rule Format:
            Contains(^TRTAMI28, ' - ')

            Technical Business Rule Format:
            Contains(^TRTAMI28, ' - ')
    
            Data Elements:
    
            ^TRTAMI28 = TRTAMI:RP:TrusteeDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.TRTAMI28 == null ? false : report.TRTAMI28.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000086",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Trustee Non-Individual Name cannot contain space hyphen space combinations",
                    Location = "/tns:TRTAMI/tns:RP/tns:TrusteeDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000086"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI28", Value = GetValueOrEmpty(report.TRTAMI28) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000087
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000087()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000087
            Non Individual Name cannot contain "P/L"
    
            Legacy Rule Format:
            Contains(^TRTAMI28, 'P/L')

            Technical Business Rule Format:
            Contains(^TRTAMI28, 'P/L')
    
            Data Elements:
    
            ^TRTAMI28 = TRTAMI:RP:TrusteeDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.TRTAMI28 == null ? false : report.TRTAMI28.ToUpperInvariant().Contains(@"P/L"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000087",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Trustee Non-Individual Name cannot contain ""P/L""",
                    Location = "/tns:TRTAMI/tns:RP/tns:TrusteeDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000087"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI28", Value = GetValueOrEmpty(report.TRTAMI28) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000088
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000088()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000088
            Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            ContainsSet(^TRTAMI28, '"--","''","  "')

            Technical Business Rule Format:
            ContainsSet(^TRTAMI28, '"--","''","  "')
    
            Data Elements:
    
            ^TRTAMI28 = TRTAMI:RP:TrusteeDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.TRTAMI28, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000088",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Trustee Non-Individual Name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:TRTAMI/tns:RP/tns:TrusteeDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000088"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI28", Value = GetValueOrEmpty(report.TRTAMI28) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000089
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000089()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000089
            Non Individual Name cannot start with "T/A" and end with "Pship", "P'ship" or "P/Ship"
    
            Legacy Rule Format:
            (StartsWith(^TRTAMI28, 'T/A ')) AND (EndsWithSet(^TRTAMI28, '" Pship"," P'ship"," P/ship"'))

            Technical Business Rule Format:
            (StartsWith(^TRTAMI28, 'T/A ')) AND (EndsWithSet(^TRTAMI28, '" Pship"," P'ship"," P/ship"'))
    
            Data Elements:
    
            ^TRTAMI28 = TRTAMI:RP:TrusteeDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (IsMatch(report.TRTAMI28, @"^(T/A )", RegexOptions.IgnoreCase) && IsMatch(report.TRTAMI28, @"( Pship| P'ship| P/ship)$", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000089",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Trustee Non-Individual Name cannot start with ""T/A"" and end with ""Pship"", ""P'ship"" or ""P/Ship""",
                    Location = "/tns:TRTAMI/tns:RP/tns:TrusteeDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000089"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI28", Value = GetValueOrEmpty(report.TRTAMI28) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000091
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000091()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000091
            Non Individual Name cannot end with "T/A", "T/A P'ship", "T/A Pship", "T/A P/Ship" or "T/A Partnership"
    
            Legacy Rule Format:
            (EndsWithSet(^TRTAMI28, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"'))

            Technical Business Rule Format:
            (EndsWithSet(^TRTAMI28, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"'))
    
            Data Elements:
    
            ^TRTAMI28 = TRTAMI:RP:TrusteeDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.TRTAMI28, @"( T/A| T/A P'ship| T/A Pship| T/A P/Ship| T/A Partnership)$", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000091",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Trustee Non-Individual Name cannot end with ""T/A"", ""T/A P'ship"", ""T/A Pship"", ""T/A P/Ship"" or ""T/A Partnership""",
                    Location = "/tns:TRTAMI/tns:RP/tns:TrusteeDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000091"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI28", Value = GetValueOrEmpty(report.TRTAMI28) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000102
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000102()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000102
            When Industry Code has been provided, Description of Main Business Activity must be present
    
            Legacy Rule Format:
            (^TRTAMI45 <> NULL) AND (^TRTAMI44 = NULL)

            Technical Business Rule Format:
            (^TRTAMI45 <> NULL) AND (^TRTAMI44 = NULL)
    
            Data Elements:
    
            ^TRTAMI44 = TRTAMI:RP:TrustDetails:OrganisationDetails.MainIncomeActivity.Description
    
            ^TRTAMI45 = TRTAMI:RP:TrustDetails:OrganisationDetails.OrganisationIndustry2006Extended.Code
            */
            assertion = (report.TRTAMI45 != null && report.TRTAMI44 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000415",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Description of Main Business Activity must be present",
                    LongDescription = @"When Industry Code has been provided, Description of Main Business Activity must be present",
                    Location = "/tns:TRTAMI/tns:RP/tns:TrustDetails/tns:OrganisationDetailsMainIncomeActivityDe",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000102"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI45", Value = GetValueOrEmpty(report.TRTAMI45) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI44", Value = GetValueOrEmpty(report.TRTAMI44) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000113
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000113()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000113
            All current assets is greater than total assets.
    
            Legacy Rule Format:
            ^TRTAMI57 <> NULL AND  ^TRTAMI58 < ^TRTAMI57

            Technical Business Rule Format:
            ^TRTAMI57 <> NULL AND  ^TRTAMI58 < ^TRTAMI57
    
            Data Elements:
    
            ^TRTAMI57 = TRTAMI:RP:KeyFinancialInformation:Assets.CurrentTotal.Amount
    
            ^TRTAMI58 = TRTAMI:RP:KeyFinancialInformation:Assets.Total.Amount
            */
            assertion = (report.TRTAMI57 != null && report.TRTAMI58.GetValueOrDefault() < report.TRTAMI57.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430194",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"All current assets must not exceed Total assets",
                    Location = "/tns:TRTAMI/tns:RP/tns:KeyFinancialInformation/tns:AssetsCurrentTotalA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000113"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI57", Value = GetValueOrEmpty(report.TRTAMI57) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI58", Value = GetValueOrEmpty(report.TRTAMI58) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000116
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000116()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000116
            All current liabilities is greater than total liabilities.
    
            Legacy Rule Format:
            ^TRTAMI59 <> NULL AND ^TRTAMI60 < ^TRTAMI59

            Technical Business Rule Format:
            ^TRTAMI59 <> NULL AND ^TRTAMI60 < ^TRTAMI59
    
            Data Elements:
    
            ^TRTAMI59 = TRTAMI:RP:KeyFinancialInformation:Liabilities.CurrentTotal.Amount
    
            ^TRTAMI60 = TRTAMI:RP:KeyFinancialInformation:Liabilities.Total.Amount
            */
            assertion = (report.TRTAMI59 != null && report.TRTAMI60.GetValueOrDefault() < report.TRTAMI59.GetValueOrDefault());
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430195",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"All current liabilities must not exceed Total liabilities",
                    Location = "/tns:TRTAMI/tns:RP/tns:KeyFinancialInformation/tns:LiabilitiesCurrentTotalA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000116"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI59", Value = GetValueOrEmpty(report.TRTAMI59) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI60", Value = GetValueOrEmpty(report.TRTAMI60) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000123
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000123(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000123
                    Family name cannot contain "Exec for", "Rep for" or "Trustee for"
    
                    Legacy Rule Format:
                    (FoundSet(^TRTAMI77, '"Exec for","Rep for","Trustee for"'))
        
                    Technical Business Rule Format:
                    (FoundSet(^TRTAMI77, '"Exec for","Rep for","Trustee for"'))
            
                    Data Elements:
            
                    ^TRTAMI77 = TRTAMI:RP:StatementOfAttributionDetails:IndividualPersonNameDetails:PersonNameDetails.FamilyName.Text
                    */
            assertion = IsMatch(statementOfAttributionDetails.TRTAMI77, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000123",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member Family Name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:IndividualPersonNameDetails/tns:FamilyNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000123"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI77", Value = statementOfAttributionDetails.TRTAMI77 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000124
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000124(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000124
                    Family name cannot contain "MR", "MRS", "MISS" or "MS"
    
                    Legacy Rule Format:
                    (FoundSet(^TRTAMI77, '"MR","MRS","MISS","MS"'))
        
                    Technical Business Rule Format:
                    (FoundSet(^TRTAMI77, '"MR","MRS","MISS","MS"'))
            
                    Data Elements:
            
                    ^TRTAMI77 = TRTAMI:RP:StatementOfAttributionDetails:IndividualPersonNameDetails:PersonNameDetails.FamilyName.Text
                    */
            assertion = IsMatch(statementOfAttributionDetails.TRTAMI77, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000124",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member Family Name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:IndividualPersonNameDetails/tns:FamilyNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000124"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI77", Value = statementOfAttributionDetails.TRTAMI77 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000125
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000125(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000125
                    Family name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
                    Legacy Rule Format:
                    (FoundSet(^TRTAMI77, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
        
                    Technical Business Rule Format:
                    (FoundSet(^TRTAMI77, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
            
                    Data Elements:
            
                    ^TRTAMI77 = TRTAMI:RP:StatementOfAttributionDetails:IndividualPersonNameDetails:PersonNameDetails.FamilyName.Text
                    */
            assertion = IsMatch(statementOfAttributionDetails.TRTAMI77, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000316",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member Family Name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:IndividualPersonNameDetails/tns:FamilyNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000125"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI77", Value = statementOfAttributionDetails.TRTAMI77 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000126
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000126(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000126
                    Family Name cannot contain "space hyphen space"
    
                    Legacy Rule Format:
                    Contains(^TRTAMI77, ' - ')
        
                    Technical Business Rule Format:
                    Contains(^TRTAMI77, ' - ')
            
                    Data Elements:
            
                    ^TRTAMI77 = TRTAMI:RP:StatementOfAttributionDetails:IndividualPersonNameDetails:PersonNameDetails.FamilyName.Text
                    */
            assertion = (statementOfAttributionDetails.TRTAMI77 == null ? false : statementOfAttributionDetails.TRTAMI77.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000126",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member Family Name cannot contain ""space hyphen space""",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:IndividualPersonNameDetails/tns:FamilyNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000126"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI77", Value = statementOfAttributionDetails.TRTAMI77 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000127
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000127(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000127
                    Family name cannot contain repeated hyphen, apostrophe, or space characters
    
                    Legacy Rule Format:
                    (ContainsSet(^TRTAMI77, '"--","''","  "'))
        
                    Technical Business Rule Format:
                    (ContainsSet(^TRTAMI77, '"--","''","  "'))
            
                    Data Elements:
            
                    ^TRTAMI77 = TRTAMI:RP:StatementOfAttributionDetails:IndividualPersonNameDetails:PersonNameDetails.FamilyName.Text
                    */
            assertion = IsMatch(statementOfAttributionDetails.TRTAMI77, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000127",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member Family Name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:IndividualPersonNameDetails/tns:FamilyNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000127"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI77", Value = statementOfAttributionDetails.TRTAMI77 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000129
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000129(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000129
                    First name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
                    Legacy Rule Format:
                    FoundSet(^TRTAMI78, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')
        
                    Technical Business Rule Format:
                    FoundSet(^TRTAMI78, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"')
            
                    Data Elements:
            
                    ^TRTAMI78 = TRTAMI:RP:StatementOfAttributionDetails:IndividualPersonNameDetails:PersonNameDetails.GivenName.Text
                    */
            assertion = IsMatch(statementOfAttributionDetails.TRTAMI78, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000317",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member First Name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:IndividualPersonNameDetails/tns:GivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000129"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI78", Value = GetValueOrEmpty(statementOfAttributionDetails.TRTAMI78) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000130
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000130(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000130
                    First name cannot contain "Exec for", "Rep for" or "Trustee for"
    
                    Legacy Rule Format:
                    FoundSet(^TRTAMI78, '"Exec for","Rep for","Trustee for"')
        
                    Technical Business Rule Format:
                    FoundSet(^TRTAMI78, '"Exec for","Rep for","Trustee for"')
            
                    Data Elements:
            
                    ^TRTAMI78 = TRTAMI:RP:StatementOfAttributionDetails:IndividualPersonNameDetails:PersonNameDetails.GivenName.Text
                    */
            assertion = IsMatch(statementOfAttributionDetails.TRTAMI78, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000130",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member First Given Name cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:IndividualPersonNameDetails/tns:GivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000130"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI78", Value = GetValueOrEmpty(statementOfAttributionDetails.TRTAMI78) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000131
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000131(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000131
                    First name cannot contain "MR", "MRS", "MISS" or "MS"
    
                    Legacy Rule Format:
                    FoundSet(^TRTAMI78, '"MR","MRS","MISS","MS"')
        
                    Technical Business Rule Format:
                    FoundSet(^TRTAMI78, '"MR","MRS","MISS","MS"')
            
                    Data Elements:
            
                    ^TRTAMI78 = TRTAMI:RP:StatementOfAttributionDetails:IndividualPersonNameDetails:PersonNameDetails.GivenName.Text
                    */
            assertion = IsMatch(statementOfAttributionDetails.TRTAMI78, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000131",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member First Given Name cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:IndividualPersonNameDetails/tns:GivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000131"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI78", Value = GetValueOrEmpty(statementOfAttributionDetails.TRTAMI78) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000132
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000132(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000132
                    FirstName must be entered if OtherGivenNames is entered
    
                    Legacy Rule Format:
                    (^TRTAMI79 <> NULL) AND (^TRTAMI78 = NULL)
        
                    Technical Business Rule Format:
                    (^TRTAMI79 <> NULL) AND (^TRTAMI78 = NULL)
            
                    Data Elements:
            
                    ^TRTAMI78 = TRTAMI:RP:StatementOfAttributionDetails:IndividualPersonNameDetails:PersonNameDetails.GivenName.Text
            
                    ^TRTAMI79 = TRTAMI:RP:StatementOfAttributionDetails:IndividualPersonNameDetails:PersonNameDetails.OtherGivenName.Text
                    */
            assertion = (statementOfAttributionDetails.TRTAMI79 != null && statementOfAttributionDetails.TRTAMI78 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000132",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member First Given Name must be entered if Member Other Given Names is entered",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:IndividualPersonNameDetails/tns:GivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000132"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI79", Value = GetValueOrEmpty(statementOfAttributionDetails.TRTAMI79) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI78", Value = GetValueOrEmpty(statementOfAttributionDetails.TRTAMI78) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000133
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000133(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000133
                    First name cannot contain space hyphen space
    
                    Legacy Rule Format:
                    Contains(^TRTAMI78, ' - ')
        
                    Technical Business Rule Format:
                    Contains(^TRTAMI78, ' - ')
            
                    Data Elements:
            
                    ^TRTAMI78 = TRTAMI:RP:StatementOfAttributionDetails:IndividualPersonNameDetails:PersonNameDetails.GivenName.Text
                    */
            assertion = (statementOfAttributionDetails.TRTAMI78 == null ? false : statementOfAttributionDetails.TRTAMI78.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000133",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member First Given Name cannot contain space hyphen space",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:IndividualPersonNameDetails/tns:GivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000133"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI78", Value = GetValueOrEmpty(statementOfAttributionDetails.TRTAMI78) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000134
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000134(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000134
                    First name cannot contain repeated hyphen, apostrophe, or space characters
    
                    Legacy Rule Format:
                    ContainsSet(^TRTAMI78, '"--","''","  "')
        
                    Technical Business Rule Format:
                    ContainsSet(^TRTAMI78, '"--","''","  "')
            
                    Data Elements:
            
                    ^TRTAMI78 = TRTAMI:RP:StatementOfAttributionDetails:IndividualPersonNameDetails:PersonNameDetails.GivenName.Text
                    */
            assertion = IsMatch(statementOfAttributionDetails.TRTAMI78, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000134",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member First Given Name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:IndividualPersonNameDetails/tns:GivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000134"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI78", Value = GetValueOrEmpty(statementOfAttributionDetails.TRTAMI78) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000136
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000136(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000136
                    Other given name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR
    
                    Legacy Rule Format:
                    (FoundSet(^TRTAMI79, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
        
                    Technical Business Rule Format:
                    (FoundSet(^TRTAMI79, '"ESQ","JNR","JP","MHA","MHR","MLA","MLC","MP","QC","SNR"'))
            
                    Data Elements:
            
                    ^TRTAMI79 = TRTAMI:RP:StatementOfAttributionDetails:IndividualPersonNameDetails:PersonNameDetails.OtherGivenName.Text
                    */
            assertion = IsMatch(statementOfAttributionDetails.TRTAMI79, @"(^(ESQ|JNR|JP|MHA|MHR|MLA|MLC|MP|QC|SNR)$)|(^(.*)( ESQ | JNR | JP | MHA | MHR | MLA | MLC | MP | QC | SNR )(.*)$)|(^(ESQ |JNR |JP |MHA |MHR |MLA |MLC |MP |QC |SNR )(.*)$)|(^(.*)( ESQ| JNR| JP| MHA| MHR| MLA| MLC| MP| QC| SNR)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000318",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member Other Given Name cannot contain suffix ESQ, JNR, JP, MHA, MHR, MLA, MLC, MP, QC or SNR",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:IndividualPersonNameDetails/tns:OtherGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000136"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI79", Value = GetValueOrEmpty(statementOfAttributionDetails.TRTAMI79) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000137
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000137(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000137
                    Other given name cannot contain "Exec for", "Rep for" or "Trustee for"
    
                    Legacy Rule Format:
                    (FoundSet(^TRTAMI79, '"Exec for","Rep for","Trustee for"'))
        
                    Technical Business Rule Format:
                    (FoundSet(^TRTAMI79, '"Exec for","Rep for","Trustee for"'))
            
                    Data Elements:
            
                    ^TRTAMI79 = TRTAMI:RP:StatementOfAttributionDetails:IndividualPersonNameDetails:PersonNameDetails.OtherGivenName.Text
                    */
            assertion = IsMatch(statementOfAttributionDetails.TRTAMI79, @"(^(Exec for|Rep for|Trustee for)$)|(^(.*)( Exec for | Rep for | Trustee for )(.*)$)|(^(Exec for |Rep for |Trustee for )(.*)$)|(^(.*)( Exec for| Rep for| Trustee for)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000137",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member Other Given Names cannot contain ""Exec for"", ""Rep for"" or ""Trustee for""",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:IndividualPersonNameDetails/tns:OtherGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000137"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI79", Value = GetValueOrEmpty(statementOfAttributionDetails.TRTAMI79) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000138
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000138(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000138
                    Other given name cannot contain "MR", "MRS", "MISS" or "MS"
    
                    Legacy Rule Format:
                    (FoundSet(^TRTAMI79, '"MR","MRS","MISS","MS"'))
        
                    Technical Business Rule Format:
                    (FoundSet(^TRTAMI79, '"MR","MRS","MISS","MS"'))
            
                    Data Elements:
            
                    ^TRTAMI79 = TRTAMI:RP:StatementOfAttributionDetails:IndividualPersonNameDetails:PersonNameDetails.OtherGivenName.Text
                    */
            assertion = IsMatch(statementOfAttributionDetails.TRTAMI79, @"(^(MR|MRS|MISS|MS)$)|(^(.*)( MR | MRS | MISS | MS )(.*)$)|(^(MR |MRS |MISS |MS )(.*)$)|(^(.*)( MR| MRS| MISS| MS)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000138",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member Other Given Names cannot contain ""MR"", ""MRS"", ""MISS"" or ""MS""",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:IndividualPersonNameDetails/tns:OtherGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000138"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI79", Value = GetValueOrEmpty(statementOfAttributionDetails.TRTAMI79) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000139
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000139(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000139
                    Other given name cannot contain space hyphen space
    
                    Legacy Rule Format:
                    Contains(^TRTAMI79, ' - ')
        
                    Technical Business Rule Format:
                    Contains(^TRTAMI79, ' - ')
            
                    Data Elements:
            
                    ^TRTAMI79 = TRTAMI:RP:StatementOfAttributionDetails:IndividualPersonNameDetails:PersonNameDetails.OtherGivenName.Text
                    */
            assertion = (statementOfAttributionDetails.TRTAMI79 == null ? false : statementOfAttributionDetails.TRTAMI79.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000139",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member Other Given Names cannot contain space hyphen space",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:IndividualPersonNameDetails/tns:OtherGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000139"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI79", Value = GetValueOrEmpty(statementOfAttributionDetails.TRTAMI79) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000140
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000140(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000140
                    Other given name cannot contain repeated hyphen, apostrophe, or space characters
    
                    Legacy Rule Format:
                    ContainsSet(^TRTAMI79, '"--","''","  "')
        
                    Technical Business Rule Format:
                    ContainsSet(^TRTAMI79, '"--","''","  "')
            
                    Data Elements:
            
                    ^TRTAMI79 = TRTAMI:RP:StatementOfAttributionDetails:IndividualPersonNameDetails:PersonNameDetails.OtherGivenName.Text
                    */
            assertion = IsMatch(statementOfAttributionDetails.TRTAMI79, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000140",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member Other Given Names cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:IndividualPersonNameDetails/tns:OtherGivenNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000140"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI79", Value = GetValueOrEmpty(statementOfAttributionDetails.TRTAMI79) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000141
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000141(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000141
                    Family name or non-individual name must be present.
    
                    Legacy Rule Format:
                    (^TRTAMI81 <> NULL) AND (^TRTAMI77 = NULL) AND (^TRTAMI80 = NULL)
        
                    Technical Business Rule Format:
                    (^TRTAMI81 <> NULL) AND (^TRTAMI77 = NULL) AND (^TRTAMI80 = NULL)
            
                    Data Elements:
            
                    ^TRTAMI80 = TRTAMI:RP:StatementOfAttributionDetails:NonIndividualDetails:OrganisationNameDetails.OrganisationalName.Text
            
                    ^TRTAMI77 = TRTAMI:RP:StatementOfAttributionDetails:IndividualPersonNameDetails:PersonNameDetails.FamilyName.Text
            
                    ^TRTAMI81 = TRTAMI:RP:StatementOfAttributionDetails:AssessmentDetails:IncomeTax.AssessmentCalculation.Code
                    */
            assertion = (statementOfAttributionDetails.TRTAMI81 != null && statementOfAttributionDetails.TRTAMI77 == null && statementOfAttributionDetails.TRTAMI80 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000141",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Name of the Statement of Attribution member must be present",
                    LongDescription = @"If the member is an individual, provide the name of the individual. If the member is a company, provide the name of the company.",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:NonIndividualDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000141"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI81", Value = statementOfAttributionDetails.TRTAMI81 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI77", Value = statementOfAttributionDetails.TRTAMI77 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI80", Value = statementOfAttributionDetails.TRTAMI80 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000142
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000142(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000142
                    Non-individual name and Family name details are both present. Only one name can be present.
    
                    Legacy Rule Format:
                    (^TRTAMI80 <> NULL) AND (^TRTAMI76 <> NULL OR ^TRTAMI77 <> NULL OR ^TRTAMI78 <> NULL OR ^TRTAMI79 <> NULL OR ^TRTAMI124 <> NULL)
        
                    Technical Business Rule Format:
                    (^TRTAMI80 <> NULL) AND (^TRTAMI76 <> NULL OR ^TRTAMI77 <> NULL OR ^TRTAMI78 <> NULL OR ^TRTAMI79 <> NULL OR ^TRTAMI124 <> NULL)
            
                    Data Elements:
            
                    ^TRTAMI80 = TRTAMI:RP:StatementOfAttributionDetails:NonIndividualDetails:OrganisationNameDetails.OrganisationalName.Text
            
                    ^TRTAMI76 = TRTAMI:RP:StatementOfAttributionDetails:IndividualPersonNameDetails:PersonNameDetails.Title.Text
            
                    ^TRTAMI77 = TRTAMI:RP:StatementOfAttributionDetails:IndividualPersonNameDetails:PersonNameDetails.FamilyName.Text
            
                    ^TRTAMI78 = TRTAMI:RP:StatementOfAttributionDetails:IndividualPersonNameDetails:PersonNameDetails.GivenName.Text
            
                    ^TRTAMI79 = TRTAMI:RP:StatementOfAttributionDetails:IndividualPersonNameDetails:PersonNameDetails.OtherGivenName.Text
            
                    ^TRTAMI124 = TRTAMI:RP:StatementOfAttributionDetails:IndividualPersonNameDetails:PersonDemographicDetails.Birth.Date
                    */
            assertion = (statementOfAttributionDetails.TRTAMI80 != null && (statementOfAttributionDetails.TRTAMI76 != null || statementOfAttributionDetails.TRTAMI77 != null || statementOfAttributionDetails.TRTAMI78 != null || statementOfAttributionDetails.TRTAMI79 != null || statementOfAttributionDetails.TRTAMI124 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000142",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member Non-Individual Name and Member Family Name details must not both be present",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:NonIndividualDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000142"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI80", Value = statementOfAttributionDetails.TRTAMI80 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI76", Value = GetValueOrEmpty(statementOfAttributionDetails.TRTAMI76) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI77", Value = statementOfAttributionDetails.TRTAMI77 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI78", Value = GetValueOrEmpty(statementOfAttributionDetails.TRTAMI78) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI79", Value = GetValueOrEmpty(statementOfAttributionDetails.TRTAMI79) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI124", Value = GetValueOrEmptyDate(statementOfAttributionDetails.TRTAMI124) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000143
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000143(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000143
                    Organisation name must contain at least one alpha or numeric character
    
                    Legacy Rule Format:
                    (^TRTAMI80 <> NULL) AND (NotContainsSet(^TRTAMI80, '"a-z", "A-Z", "0-9"'))
        
                    Technical Business Rule Format:
                    (^TRTAMI80 <> NULL) AND (NotContainsSet(^TRTAMI80, '"a-z", "A-Z", "0-9"'))
            
                    Data Elements:
            
                    ^TRTAMI80 = TRTAMI:RP:StatementOfAttributionDetails:NonIndividualDetails:OrganisationNameDetails.OrganisationalName.Text
                    */
            assertion = (statementOfAttributionDetails.TRTAMI80 != null && !(IsMatch(statementOfAttributionDetails.TRTAMI80, @"[a-z]|[A-Z]|[0-9]", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000143",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member Non-Individual Name must contain at least one alpha or numeric character",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:NonIndividualDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000143"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI80", Value = statementOfAttributionDetails.TRTAMI80 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000144
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000144(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000144
                    Non Individual Name cannot contain space hyphen space combinations
    
                    Legacy Rule Format:
                    Contains(^TRTAMI80, ' - ')
        
                    Technical Business Rule Format:
                    Contains(^TRTAMI80, ' - ')
            
                    Data Elements:
            
                    ^TRTAMI80 = TRTAMI:RP:StatementOfAttributionDetails:NonIndividualDetails:OrganisationNameDetails.OrganisationalName.Text
                    */
            assertion = (statementOfAttributionDetails.TRTAMI80 == null ? false : statementOfAttributionDetails.TRTAMI80.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000144",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member Non-Individual Name cannot contain space hyphen space combinations",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:NonIndividualDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000144"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI80", Value = statementOfAttributionDetails.TRTAMI80 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000145
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000145(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000145
                    Non Individual Name cannot contain "P/L"
    
                    Legacy Rule Format:
                    Contains(^TRTAMI80, 'P/L')
        
                    Technical Business Rule Format:
                    Contains(^TRTAMI80, 'P/L')
            
                    Data Elements:
            
                    ^TRTAMI80 = TRTAMI:RP:StatementOfAttributionDetails:NonIndividualDetails:OrganisationNameDetails.OrganisationalName.Text
                    */
            assertion = (statementOfAttributionDetails.TRTAMI80 == null ? false : statementOfAttributionDetails.TRTAMI80.ToUpperInvariant().Contains(@"P/L"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000145",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member Non-Individual Name cannot contain ""P/L""",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:NonIndividualDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000145"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI80", Value = statementOfAttributionDetails.TRTAMI80 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000146
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000146(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000146
                    Non Individual Name cannot contain repeated hyphen, apostrophe, or space characters
    
                    Legacy Rule Format:
                    (ContainsSet(^TRTAMI80, '"--","''","  "'))
        
                    Technical Business Rule Format:
                    (ContainsSet(^TRTAMI80, '"--","''","  "'))
            
                    Data Elements:
            
                    ^TRTAMI80 = TRTAMI:RP:StatementOfAttributionDetails:NonIndividualDetails:OrganisationNameDetails.OrganisationalName.Text
                    */
            assertion = IsMatch(statementOfAttributionDetails.TRTAMI80, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000146",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member Non-Individual Name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:NonIndividualDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000146"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI80", Value = statementOfAttributionDetails.TRTAMI80 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000147
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000147(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000147
                    Non Individual Name cannot start with "T/A" and end with "Pship", "P'ship" or "P/Ship"
    
                    Legacy Rule Format:
                    (StartsWith(^TRTAMI80, 'T/A ')) AND (EndsWithSet(^TRTAMI80, '" Pship"," P'ship"," P/ship"'))
        
                    Technical Business Rule Format:
                    (StartsWith(^TRTAMI80, 'T/A ')) AND (EndsWithSet(^TRTAMI80, '" Pship"," P'ship"," P/ship"'))
            
                    Data Elements:
            
                    ^TRTAMI80 = TRTAMI:RP:StatementOfAttributionDetails:NonIndividualDetails:OrganisationNameDetails.OrganisationalName.Text
                    */
            assertion = (IsMatch(statementOfAttributionDetails.TRTAMI80, @"^(T/A )", RegexOptions.IgnoreCase) && IsMatch(statementOfAttributionDetails.TRTAMI80, @"( Pship| P'ship| P/ship)$", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000147",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member Non-Individual Name cannot start with ""T/A"" and end with ""Pship"", ""P'ship"" or ""P/Ship""",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:NonIndividualDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000147"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI80", Value = statementOfAttributionDetails.TRTAMI80 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000149
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000149(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000149
                    Non Individual Name cannot end with "T/A", "T/A P'ship", "T/A Pship", "T/A P/Ship" or "T/A Partnership"
    
                    Legacy Rule Format:
                    (EndsWithSet(^TRTAMI80, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"'))
        
                    Technical Business Rule Format:
                    (EndsWithSet(^TRTAMI80, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"'))
            
                    Data Elements:
            
                    ^TRTAMI80 = TRTAMI:RP:StatementOfAttributionDetails:NonIndividualDetails:OrganisationNameDetails.OrganisationalName.Text
                    */
            assertion = IsMatch(statementOfAttributionDetails.TRTAMI80, @"( T/A| T/A P'ship| T/A Pship| T/A P/Ship| T/A Partnership)$", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000149",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member Non-Individual Name cannot end with ""T/A"", ""T/A P'ship"", ""T/A Pship"", ""T/A P/Ship"" or ""T/A Partnership""",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:NonIndividualDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000149"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI80", Value = statementOfAttributionDetails.TRTAMI80 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000150
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000150()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000150
            Tax File Number has failed the algorithm check.
    
            Legacy Rule Format:
            (^TRTAMI4 <> NULL) AND (FailsTFNAlgorithm(^TRTAMI4))

            Technical Business Rule Format:
            (^TRTAMI4 <> NULL) AND (FailsTFNAlgorithm(^TRTAMI4))
    
            Data Elements:
    
            ^TRTAMI4 = TRTAMI:RP:Identifiers.TaxFileNumber.Identifier
            */
            assertion = (report.TRTAMI4 != null && FailsTFNAlgorithm(report.TRTAMI4));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000150",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Trust Tax File Number has failed the algorithm check",
                    Location = "/tns:TRTAMI/tns:RP/tns:TaxFileNumberId",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000150"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI4", Value = report.TRTAMI4 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000166
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000166(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000166
                    The Tax File Number has failed the Tax File Number algorithm check.
    
                    Legacy Rule Format:
                    (^TRTAMI75 <> NULL) AND (FailsTFNAlgorithm(^TRTAMI75))
        
                    Technical Business Rule Format:
                    (^TRTAMI75 <> NULL) AND (FailsTFNAlgorithm(^TRTAMI75))
            
                    Data Elements:
            
                    ^TRTAMI75 = TRTAMI:RP:StatementOfAttributionDetails:Identifiers.TaxFileNumber.Identifier
                    */
            assertion = (statementOfAttributionDetails.TRTAMI75 != null && FailsTFNAlgorithm(statementOfAttributionDetails.TRTAMI75));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000166",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member Tax File Number has failed the algorithm check",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:TaxFileNumberId",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000166"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI75", Value = GetValueOrEmpty(statementOfAttributionDetails.TRTAMI75) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000167
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000167()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000167
            The TFN provided for the Trust member must not be the same as the TFN of the Trust.
    
            Legacy Rule Format:
            AnyOccurrence(^TRTAMI75, ^TRTAMI75 = ^TRTAMI4)

            Technical Business Rule Format:
            AnyOccurrence(^TRTAMI75, ^TRTAMI75 = ^TRTAMI4)
    
            Data Elements:
    
            ^TRTAMI75 = TRTAMI:RP:StatementOfAttributionDetails:Identifiers.TaxFileNumber.Identifier
    
            ^TRTAMI4 = TRTAMI:RP:Identifiers.TaxFileNumber.Identifier
            */
            assertion = (report.RP_StatementOfAttributionDetailsCollection == null ? false : report.RP_StatementOfAttributionDetailsCollection.Any(TRTAMI75Repeat => TRTAMI75Repeat.TRTAMI75 == report.TRTAMI4));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000167",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member Tax File Number must not be the same as the Trust Tax File Number",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails/tns:TaxFileNumberId",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000167"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI75", Value = "TRTAMI75" });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI4", Value = "TRTAMI4" });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000168
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000168(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000168
                    The member TFN reported must not be reported more than once.
    
                    Legacy Rule Format:
                    ^TRTAMI75 <> NULL AND HasDuplicateValues(^TRTAMI75)
        
                    Technical Business Rule Format:
                    ^TRTAMI75 <> NULL AND HasDuplicateValues(^TRTAMI75)
            
                    Data Elements:
            
                    ^TRTAMI75 = TRTAMI:RP:StatementOfAttributionDetails:Identifiers.TaxFileNumber.Identifier
                    */
            assertion = (statementOfAttributionDetails.TRTAMI75 != null && HasDuplicateValues(report.RP_StatementOfAttributionDetailsCollection == null ? null : report.RP_StatementOfAttributionDetailsCollection.Select(f => f.TRTAMI75).Cast<object>().ToArray()));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000168",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member Tax File Number must not be repeated",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails[" + report.RP_StatementOfAttributionDetailsCollection[DuplicateValueIndex(report.RP_StatementOfAttributionDetailsCollection.Select(f => f.TRTAMI75))].OccurrenceIndex + "]/tns:TaxFileNumberId",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000168"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI75", Value = GetValueOrEmpty(statementOfAttributionDetails.TRTAMI75) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000171
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000171(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000171
                    In the Statement of Attribution, if the assessment calculation code is 138, 139 or 140, the S98 assessable amounts must be provided.
    
                    Legacy Rule Format:
                    ^TRTAMI81 <> NULL AND ^TRTAMI82 = NULL
        
                    Technical Business Rule Format:
                    ^TRTAMI81 <> NULL AND ^TRTAMI82 = NULL
            
                    Data Elements:
            
                    ^TRTAMI82 = TRTAMI:RP:StatementOfAttributionDetails:AssessmentDetails:InternationalDealings.AssessableIncomeNonResidentBeneficiary.Amount
            
                    ^TRTAMI81 = TRTAMI:RP:StatementOfAttributionDetails:AssessmentDetails:IncomeTax.AssessmentCalculation.Code
                    */
            assertion = (statementOfAttributionDetails.TRTAMI81 != null && statementOfAttributionDetails.TRTAMI82 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000171",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Taxable Determined Member Component amount must be present",
                    LongDescription = @"When the Assessment Calculation Code of Member is present, the Taxable Determined Member Component amount must be provided",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:AssessmentDetails/tns:InternationalDealingsAssessableIncomeNonResidentBeneficiaryA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000171"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI81", Value = statementOfAttributionDetails.TRTAMI81 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI82", Value = GetValueOrEmpty(statementOfAttributionDetails.TRTAMI82) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000197
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000197(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000197
                    In the Statement of Attribution, if the S98 assessable amount is positive, the assessment calculation code must be provided.
    
                    Legacy Rule Format:
                    ^TRTAMI82 > 0 AND ^TRTAMI81 = NULL
        
                    Technical Business Rule Format:
                    ^TRTAMI82 > 0 AND ^TRTAMI81 = NULL
            
                    Data Elements:
            
                    ^TRTAMI81 = TRTAMI:RP:StatementOfAttributionDetails:AssessmentDetails:IncomeTax.AssessmentCalculation.Code
            
                    ^TRTAMI82 = TRTAMI:RP:StatementOfAttributionDetails:AssessmentDetails:InternationalDealings.AssessableIncomeNonResidentBeneficiary.Amount
                    */
            assertion = (statementOfAttributionDetails.TRTAMI82.GetValueOrDefault() > 0 && statementOfAttributionDetails.TRTAMI81 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000197",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Assessment Calculation Code of Member must be present",
                    LongDescription = @"When a positive value for Trustee Liability for Non-Resident Amounts is present, the Assessment Calculation Code of Member must be provided",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:AssessmentDetails/tns:IncomeTaxAssessmentCalculationC",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000197"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI82", Value = GetValueOrEmpty(statementOfAttributionDetails.TRTAMI82) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI81", Value = statementOfAttributionDetails.TRTAMI81 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000209
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000209()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000209
            When the response to 'Have you chosen to treat separate classes in the trust as a separate AMIT?' is 'True' (yes), the minimum Number of Separate Classes for Trust must be at least 1.
    
            Legacy Rule Format:
            ^TRTAMI102 = TRUE AND (^TRTAMI103 = NULL OR ^TRTAMI103 = 0)

            Technical Business Rule Format:
            ^TRTAMI102 = TRUE AND (^TRTAMI103 = NULL OR ^TRTAMI103 = 0)
    
            Data Elements:
    
            ^TRTAMI103 = TRTAMI:RP:MultiClassTreatment:IncomeTax.MultiClassTreatment.Count
    
            ^TRTAMI102 = TRTAMI:RP:MultiClassTreatment:IncomeTax.MultiClassTreatment.Indicator
            */
            assertion = (report.TRTAMI102 == true && (report.TRTAMI103 == null || report.TRTAMI103 == 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000209",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Number of Separate Classes for Trust is invalid",
                    LongDescription = @"When the response to 'Have you chosen to treat separate classes in the trust as a separate AMIT?' is 'True' (yes), the minimum Number of Separate Classes for Trust must be at least 1",
                    Location = "/tns:TRTAMI/tns:RP/tns:MultiClassTreatment/tns:IncomeTaxCt",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000209"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI102", Value = GetValueOrEmpty(report.TRTAMI102) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI103", Value = report.TRTAMI103.GetValueOrDefault().ToString() });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000210
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000210()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000210
            When the response to 'Have you chosen to treat separate classes in the trust as a separate AMIT?' is 'True' (yes), a response to 'Have you transferred assets between classes during this income year?' must be provided
    
            Legacy Rule Format:
            ^TRTAMI102 = TRUE AND ^TRTAMI104 = NULL

            Technical Business Rule Format:
            ^TRTAMI102 = TRUE AND ^TRTAMI104 = NULL
    
            Data Elements:
    
            ^TRTAMI104 = TRTAMI:RP:MultiClassTreatment:IncomeTax.MultiClassTransfer.Indicator
    
            ^TRTAMI102 = TRTAMI:RP:MultiClassTreatment:IncomeTax.MultiClassTreatment.Indicator
            */
            assertion = (report.TRTAMI102 == true && report.TRTAMI104 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000210",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Have you transferred assets between classes during this income year?' must be provided",
                    LongDescription = @"When the response to 'Have you chosen to treat separate classes in the trust as a separate AMIT?' is 'True' (yes), a response to 'Have you transferred assets between classes during this income year?' must be provided",
                    Location = "/tns:TRTAMI/tns:RP/tns:MultiClassTreatment/tns:IncomeTaxMultiClassTransferI",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000210"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI102", Value = GetValueOrEmpty(report.TRTAMI102) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI104", Value = GetValueOrEmpty(report.TRTAMI104) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000238
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000238()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000238
            When the response to 'Have you chosen to treat separate classes in the trust as a separate AMIT?' is 'False' (no), the Number of Separate Classes for Trust must be left NULL or set to zero.
    
            Legacy Rule Format:
            ^TRTAMI102 = FALSE AND (^TRTAMI103 <> NULL AND ^TRTAMI103 <> 0)

            Technical Business Rule Format:
            ^TRTAMI102 = FALSE AND (^TRTAMI103 <> NULL AND ^TRTAMI103 <> 0)
    
            Data Elements:
    
            ^TRTAMI103 = TRTAMI:RP:MultiClassTreatment:IncomeTax.MultiClassTreatment.Count
    
            ^TRTAMI102 = TRTAMI:RP:MultiClassTreatment:IncomeTax.MultiClassTreatment.Indicator
            */
            assertion = (report.TRTAMI102 == false && (report.TRTAMI103 != null && report.TRTAMI103 != 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000238",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Number of Separate Classes for Trust is incorrect",
                    LongDescription = @"When the response to 'Have you chosen to treat separate classes in the trust as a separate AMIT?' is 'False' (no), the Number of Separate Classes for Trust must be left blank or set to zero",
                    Location = "/tns:TRTAMI/tns:RP/tns:MultiClassTreatment/tns:IncomeTaxCt",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000238"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI102", Value = GetValueOrEmpty(report.TRTAMI102) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI103", Value = report.TRTAMI103.GetValueOrDefault().ToString() });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000241
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000241(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000241
                    For individuals, the only valid Assessment Calculation Code of Member value is "138".
    
                    Legacy Rule Format:
                    ^TRTAMI81 <> '138' AND ^TRTAMI77 <> NULL
        
                    Technical Business Rule Format:
                    ^TRTAMI81 <> '138' AND ^TRTAMI77 <> NULL
            
                    Data Elements:
            
                    ^TRTAMI81 = TRTAMI:RP:StatementOfAttributionDetails:AssessmentDetails:IncomeTax.AssessmentCalculation.Code
            
                    ^TRTAMI77 = TRTAMI:RP:StatementOfAttributionDetails:IndividualPersonNameDetails:PersonNameDetails.FamilyName.Text
                    */
            assertion = (statementOfAttributionDetails.TRTAMI81 != @"138" && statementOfAttributionDetails.TRTAMI77 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000241",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Assessment Calculation Code of Member is invalid",
                    LongDescription = @"For individuals, the only valid Assessment Calculation Code of Member value is ""138""",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:AssessmentDetails/tns:IncomeTaxAssessmentCalculationC",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000241"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI81", Value = statementOfAttributionDetails.TRTAMI81 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI77", Value = statementOfAttributionDetails.TRTAMI77 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000242
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000242(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000242
                    For non-individuals, the valid Assessment Calculation Code of Member values are "139" or "140".
    
                    Legacy Rule Format:
                    NotInSet(^TRTAMI81, '"139","140"') AND ^TRTAMI80 <> NULL
        
                    Technical Business Rule Format:
                    NotInSet(^TRTAMI81, '"139","140"') AND ^TRTAMI80 <> NULL
            
                    Data Elements:
            
                    ^TRTAMI81 = TRTAMI:RP:StatementOfAttributionDetails:AssessmentDetails:IncomeTax.AssessmentCalculation.Code
            
                    ^TRTAMI80 = TRTAMI:RP:StatementOfAttributionDetails:NonIndividualDetails:OrganisationNameDetails.OrganisationalName.Text
                    */
            assertion = (!(IsMatch(statementOfAttributionDetails.TRTAMI81, @"^(139|140)$", RegexOptions.IgnoreCase)) && statementOfAttributionDetails.TRTAMI80 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000242",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Assessment Calculation Code of Member is invalid",
                    LongDescription = @"For non-individuals, the valid Assessment Calculation Code of Member values are ""139"" or ""140""",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:AssessmentDetails/tns:IncomeTaxAssessmentCalculationC",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000242"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI81", Value = statementOfAttributionDetails.TRTAMI81 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI80", Value = statementOfAttributionDetails.TRTAMI80 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000248
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000248(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000248
                    Address Line 1 must not contain 'AS ABOVE'
    
                    Legacy Rule Format:
                    Contains(^TRTAMI125, 'AS ABOVE')
        
                    Technical Business Rule Format:
                    Contains(^TRTAMI125, 'AS ABOVE')
            
                    Data Elements:
            
                    ^TRTAMI125 = TRTAMI:RP:StatementOfAttributionDetails:IndividualPersonNameDetails:IndividualAddressDetails:AddressDetails.Line1.Text
                    */
            assertion = (statementOfAttributionDetails.TRTAMI125 == null ? false : statementOfAttributionDetails.TRTAMI125.ToUpperInvariant().Contains(@"AS ABOVE"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000248",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member Residential Address - Address Line 1 must not contain 'AS ABOVE'",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:IndividualPersonNameDetails/tns:IndividualAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000248"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI125", Value = statementOfAttributionDetails.TRTAMI125 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000249
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000249(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000249
                    Address Line 1 must be present
    
                    Legacy Rule Format:
                    ^TRTAMI125 = NULL AND ^TRTAMI77 <> NULL
        
                    Technical Business Rule Format:
                    ^TRTAMI125 = NULL AND ^TRTAMI77 <> NULL
            
                    Data Elements:
            
                    ^TRTAMI125 = TRTAMI:RP:StatementOfAttributionDetails:IndividualPersonNameDetails:IndividualAddressDetails:AddressDetails.Line1.Text
            
                    ^TRTAMI77 = TRTAMI:RP:StatementOfAttributionDetails:IndividualPersonNameDetails:PersonNameDetails.FamilyName.Text
                    */
            assertion = (statementOfAttributionDetails.TRTAMI125 == null && statementOfAttributionDetails.TRTAMI77 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000249",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member Residential Address - Address Line 1 must be present for an individual",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:IndividualPersonNameDetails/tns:IndividualAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000249"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI125", Value = statementOfAttributionDetails.TRTAMI125 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI77", Value = statementOfAttributionDetails.TRTAMI77 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000250
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000250(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000250
                    Address line 1 contains incorrect 'care of' reference. The only acceptable 'care of' reference is C/-.
    
                    Legacy Rule Format:
                    FoundSet(^TRTAMI125, '"C/O","C/","Care Of","CO"')
        
                    Technical Business Rule Format:
                    FoundSet(^TRTAMI125, '"C/O","C/","Care Of","CO"')
            
                    Data Elements:
            
                    ^TRTAMI125 = TRTAMI:RP:StatementOfAttributionDetails:IndividualPersonNameDetails:IndividualAddressDetails:AddressDetails.Line1.Text
                    */
            assertion = IsMatch(statementOfAttributionDetails.TRTAMI125, @"(^(C/O|C/|Care Of|CO)$)|(^(.*)( C/O | C/ | Care Of | CO )(.*)$)|(^(C/O |C/ |Care Of |CO )(.*)$)|(^(.*)( C/O| C/| Care Of| CO)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000250",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member Residential Address - Address Line 1 is invalid",
                    LongDescription = @"Member Residential Address - Address Line 1 contains incorrect 'care of' reference. The only acceptable 'care of' reference is C/-.",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:IndividualPersonNameDetails/tns:IndividualAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000250"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI125", Value = statementOfAttributionDetails.TRTAMI125 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000251
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000251(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000251
                    Address line 1 cannot contain "UNKNOWN"
    
                    Legacy Rule Format:
                    Contains(^TRTAMI125, 'UNKNOWN')
        
                    Technical Business Rule Format:
                    Contains(^TRTAMI125, 'UNKNOWN')
            
                    Data Elements:
            
                    ^TRTAMI125 = TRTAMI:RP:StatementOfAttributionDetails:IndividualPersonNameDetails:IndividualAddressDetails:AddressDetails.Line1.Text
                    */
            assertion = (statementOfAttributionDetails.TRTAMI125 == null ? false : statementOfAttributionDetails.TRTAMI125.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000251",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member Residential Address - Address Line 1 cannot contain ""UNKNOWN""",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:IndividualPersonNameDetails/tns:IndividualAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000251"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI125", Value = statementOfAttributionDetails.TRTAMI125 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000253
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000253(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000253
                    Care of' is not valid for address line 2
    
                    Legacy Rule Format:
                    FoundSet(^TRTAMI126, '"C/-","C/O","C/","Care Of","CO"')
        
                    Technical Business Rule Format:
                    FoundSet(^TRTAMI126, '"C/-","C/O","C/","Care Of","CO"')
            
                    Data Elements:
            
                    ^TRTAMI126 = TRTAMI:RP:StatementOfAttributionDetails:IndividualPersonNameDetails:IndividualAddressDetails:AddressDetails.Line2.Text
                    */
            assertion = IsMatch(statementOfAttributionDetails.TRTAMI126, @"(^(C/-|C/O|C/|Care Of|CO)$)|(^(.*)( C/- | C/O | C/ | Care Of | CO )(.*)$)|(^(C/- |C/O |C/ |Care Of |CO )(.*)$)|(^(.*)( C/-| C/O| C/| Care Of| CO)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000253",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Care of' is not valid for Member Residential Address - Address Line 2",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:IndividualPersonNameDetails/tns:IndividualAddressDetails/tns:Line2T",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000253"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI126", Value = GetValueOrEmpty(statementOfAttributionDetails.TRTAMI126) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000254
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000254(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000254
                    Address line 2 cannot contain "UNKNOWN"
    
                    Legacy Rule Format:
                    Contains(^TRTAMI126, 'UNKNOWN')
        
                    Technical Business Rule Format:
                    Contains(^TRTAMI126, 'UNKNOWN')
            
                    Data Elements:
            
                    ^TRTAMI126 = TRTAMI:RP:StatementOfAttributionDetails:IndividualPersonNameDetails:IndividualAddressDetails:AddressDetails.Line2.Text
                    */
            assertion = (statementOfAttributionDetails.TRTAMI126 == null ? false : statementOfAttributionDetails.TRTAMI126.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000254",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member Residential Address - Address Line 2 cannot contain ""UNKNOWN""",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:IndividualPersonNameDetails/tns:IndividualAddressDetails/tns:Line2T",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000254"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI126", Value = GetValueOrEmpty(statementOfAttributionDetails.TRTAMI126) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000255
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000255(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000255
                    Address Line 2 must be present if Address Line 1 contains 'C/-'
    
                    Legacy Rule Format:
                    (Contains(^TRTAMI125, 'C/-')) AND (^TRTAMI126 = NULL)
        
                    Technical Business Rule Format:
                    (Contains(^TRTAMI125, 'C/-')) AND (^TRTAMI126 = NULL)
            
                    Data Elements:
            
                    ^TRTAMI126 = TRTAMI:RP:StatementOfAttributionDetails:IndividualPersonNameDetails:IndividualAddressDetails:AddressDetails.Line2.Text
            
                    ^TRTAMI125 = TRTAMI:RP:StatementOfAttributionDetails:IndividualPersonNameDetails:IndividualAddressDetails:AddressDetails.Line1.Text
                    */
            assertion = ((statementOfAttributionDetails.TRTAMI125 == null ? false : statementOfAttributionDetails.TRTAMI125.ToUpperInvariant().Contains(@"C/-")) && statementOfAttributionDetails.TRTAMI126 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000255",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member Residential Address - Address Line 2 must be provided",
                    LongDescription = @"When Member Residential Address - Address Line 1 contains 'C/-', Member Residential Address - Address Line 2 must be provided",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:IndividualPersonNameDetails/tns:IndividualAddressDetails/tns:Line2T",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000255"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI125", Value = statementOfAttributionDetails.TRTAMI125 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI126", Value = GetValueOrEmpty(statementOfAttributionDetails.TRTAMI126) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000256
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000256(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000256
                    State code [QLD,NSW,VIC,SA,WA,NT,ACT,TAS] or a leading space or combination of spaces with a State code is present in the suburb/town field.
    
                    Legacy Rule Format:
                    (^TRTAMI130 = NULL OR ^TRTAMI130 = 'au') AND (FoundSet(^TRTAMI127, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))
        
                    Technical Business Rule Format:
                    (^TRTAMI130 = NULL OR ^TRTAMI130 = 'au') AND (FoundSet(^TRTAMI127, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))
            
                    Data Elements:
            
                    ^TRTAMI127 = TRTAMI:RP:StatementOfAttributionDetails:IndividualPersonNameDetails:IndividualAddressDetails:AddressDetails.LocalityName.Text
            
                    ^TRTAMI130 = TRTAMI:RP:StatementOfAttributionDetails:IndividualPersonNameDetails:IndividualAddressDetails:AddressDetails.Country.Code
                    */
            assertion = ((statementOfAttributionDetails.TRTAMI130 == null || statementOfAttributionDetails.TRTAMI130 == @"au") && IsMatch(statementOfAttributionDetails.TRTAMI127, @"(^(QLD|NSW|VIC|SA|WA|NT|ACT|TAS)$)|(^(.*)( QLD | NSW | VIC | SA | WA | NT | ACT | TAS )(.*)$)|(^(QLD |NSW |VIC |SA |WA |NT |ACT |TAS )(.*)$)|(^(.*)( QLD| NSW| VIC| SA| WA| NT| ACT| TAS)$)", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000256",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member Residential Address - City/Town cannot contain state code",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:IndividualPersonNameDetails/tns:IndividualAddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000256"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI130", Value = GetValueOrEmpty(statementOfAttributionDetails.TRTAMI130) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI127", Value = statementOfAttributionDetails.TRTAMI127 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000257
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000257(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000257
                    Suburb/Town/City cannot contain "UNKNOWN"
    
                    Legacy Rule Format:
                    Contains(^TRTAMI127, 'UNKNOWN')
        
                    Technical Business Rule Format:
                    Contains(^TRTAMI127, 'UNKNOWN')
            
                    Data Elements:
            
                    ^TRTAMI127 = TRTAMI:RP:StatementOfAttributionDetails:IndividualPersonNameDetails:IndividualAddressDetails:AddressDetails.LocalityName.Text
                    */
            assertion = (statementOfAttributionDetails.TRTAMI127 == null ? false : statementOfAttributionDetails.TRTAMI127.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000257",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member Residential Address - City/Town cannot contain ""UNKNOWN""",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:IndividualPersonNameDetails/tns:IndividualAddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000257"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI127", Value = statementOfAttributionDetails.TRTAMI127 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000258
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000258(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000258
                    Suburb/Town/City is mandatory
    
                    Legacy Rule Format:
                    ^TRTAMI127 = NULL AND ^TRTAMI77 <> NULL
        
                    Technical Business Rule Format:
                    ^TRTAMI127 = NULL AND ^TRTAMI77 <> NULL
            
                    Data Elements:
            
                    ^TRTAMI127 = TRTAMI:RP:StatementOfAttributionDetails:IndividualPersonNameDetails:IndividualAddressDetails:AddressDetails.LocalityName.Text
            
                    ^TRTAMI77 = TRTAMI:RP:StatementOfAttributionDetails:IndividualPersonNameDetails:PersonNameDetails.FamilyName.Text
                    */
            assertion = (statementOfAttributionDetails.TRTAMI127 == null && statementOfAttributionDetails.TRTAMI77 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000258",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member Residential Address - City/Town must be provided for an individual",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:IndividualPersonNameDetails/tns:IndividualAddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000258"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI127", Value = statementOfAttributionDetails.TRTAMI127 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI77", Value = statementOfAttributionDetails.TRTAMI77 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000259
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000259(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000259
                    Numerics [0 to 9] are not allowed in the suburb/town field.
    
                    Legacy Rule Format:
                    (^TRTAMI130 = NULL OR ^TRTAMI130 = 'au') AND (ContainsSet(^TRTAMI127, '0-9'))
        
                    Technical Business Rule Format:
                    (^TRTAMI130 = NULL OR ^TRTAMI130 = 'au') AND (ContainsSet(^TRTAMI127, '0-9'))
            
                    Data Elements:
            
                    ^TRTAMI127 = TRTAMI:RP:StatementOfAttributionDetails:IndividualPersonNameDetails:IndividualAddressDetails:AddressDetails.LocalityName.Text
            
                    ^TRTAMI130 = TRTAMI:RP:StatementOfAttributionDetails:IndividualPersonNameDetails:IndividualAddressDetails:AddressDetails.Country.Code
                    */
            assertion = ((statementOfAttributionDetails.TRTAMI130 == null || statementOfAttributionDetails.TRTAMI130 == @"au") && IsMatch(statementOfAttributionDetails.TRTAMI127, @"[0-9]", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000259",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member Residential Address - City/Town is invalid",
                    LongDescription = @"When an Australian address is provided, Member Residential Address - City/Town must not contain numeric characters",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:IndividualPersonNameDetails/tns:IndividualAddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000259"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI130", Value = GetValueOrEmpty(statementOfAttributionDetails.TRTAMI130) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI127", Value = statementOfAttributionDetails.TRTAMI127 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000267
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000267(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000267
                    Address Line 1 must not contain 'AS ABOVE'
    
                    Legacy Rule Format:
                    Contains(^TRTAMI131, 'AS ABOVE')
        
                    Technical Business Rule Format:
                    Contains(^TRTAMI131, 'AS ABOVE')
            
                    Data Elements:
            
                    ^TRTAMI131 = TRTAMI:RP:StatementOfAttributionDetails:NonIndividualDetails:NonIndividualAddressDetails:AddressDetails.Line1.Text
                    */
            assertion = (statementOfAttributionDetails.TRTAMI131 == null ? false : statementOfAttributionDetails.TRTAMI131.ToUpperInvariant().Contains(@"AS ABOVE"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000267",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member Business Address - Address Line 1 must not contain 'AS ABOVE'",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:NonIndividualDetails/tns:NonIndividualAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000267"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI131", Value = statementOfAttributionDetails.TRTAMI131 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000269
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000269(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000269
                    Address line 1 contains incorrect 'care of' reference. The only acceptable 'care of' reference is C/-.
    
                    Legacy Rule Format:
                    FoundSet(^TRTAMI131, '"C/O","C/","Care Of","CO"')
        
                    Technical Business Rule Format:
                    FoundSet(^TRTAMI131, '"C/O","C/","Care Of","CO"')
            
                    Data Elements:
            
                    ^TRTAMI131 = TRTAMI:RP:StatementOfAttributionDetails:NonIndividualDetails:NonIndividualAddressDetails:AddressDetails.Line1.Text
                    */
            assertion = IsMatch(statementOfAttributionDetails.TRTAMI131, @"(^(C/O|C/|Care Of|CO)$)|(^(.*)( C/O | C/ | Care Of | CO )(.*)$)|(^(C/O |C/ |Care Of |CO )(.*)$)|(^(.*)( C/O| C/| Care Of| CO)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000269",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member Business Address - Address Line 1 is invalid",
                    LongDescription = @"Member Business Address - Address Line 1 contains incorrect 'care of' reference. The only acceptable 'care of' reference is C/-.",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:NonIndividualDetails/tns:NonIndividualAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000269"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI131", Value = statementOfAttributionDetails.TRTAMI131 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000270
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000270(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000270
                    Address line 1 cannot contain "UNKNOWN"
    
                    Legacy Rule Format:
                    Contains(^TRTAMI131, 'UNKNOWN')
        
                    Technical Business Rule Format:
                    Contains(^TRTAMI131, 'UNKNOWN')
            
                    Data Elements:
            
                    ^TRTAMI131 = TRTAMI:RP:StatementOfAttributionDetails:NonIndividualDetails:NonIndividualAddressDetails:AddressDetails.Line1.Text
                    */
            assertion = (statementOfAttributionDetails.TRTAMI131 == null ? false : statementOfAttributionDetails.TRTAMI131.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000270",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member Business Address - Address Line 1 cannot contain ""UNKNOWN""",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:NonIndividualDetails/tns:NonIndividualAddressDetails/tns:Line1T",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000270"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI131", Value = statementOfAttributionDetails.TRTAMI131 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000272
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000272(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000272
                    Care of' is not valid for address line 2
    
                    Legacy Rule Format:
                    FoundSet(^TRTAMI132, '"C/-","C/O","C/","Care Of","CO"')
        
                    Technical Business Rule Format:
                    FoundSet(^TRTAMI132, '"C/-","C/O","C/","Care Of","CO"')
            
                    Data Elements:
            
                    ^TRTAMI132 = TRTAMI:RP:StatementOfAttributionDetails:NonIndividualDetails:NonIndividualAddressDetails:AddressDetails.Line2.Text
                    */
            assertion = IsMatch(statementOfAttributionDetails.TRTAMI132, @"(^(C/-|C/O|C/|Care Of|CO)$)|(^(.*)( C/- | C/O | C/ | Care Of | CO )(.*)$)|(^(C/- |C/O |C/ |Care Of |CO )(.*)$)|(^(.*)( C/-| C/O| C/| Care Of| CO)$)", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000272",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Care of' is not valid for Member Business Address - Address Line 2",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:NonIndividualDetails/tns:NonIndividualAddressDetails/tns:Line2T",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000272"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI132", Value = GetValueOrEmpty(statementOfAttributionDetails.TRTAMI132) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000273
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000273(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000273
                    Address line 2 cannot contain "UNKNOWN"
    
                    Legacy Rule Format:
                    Contains(^TRTAMI132, 'UNKNOWN')
        
                    Technical Business Rule Format:
                    Contains(^TRTAMI132, 'UNKNOWN')
            
                    Data Elements:
            
                    ^TRTAMI132 = TRTAMI:RP:StatementOfAttributionDetails:NonIndividualDetails:NonIndividualAddressDetails:AddressDetails.Line2.Text
                    */
            assertion = (statementOfAttributionDetails.TRTAMI132 == null ? false : statementOfAttributionDetails.TRTAMI132.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000273",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member Business Address - Address Line 2 cannot contain ""UNKNOWN""",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:NonIndividualDetails/tns:NonIndividualAddressDetails/tns:Line2T",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000273"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI132", Value = GetValueOrEmpty(statementOfAttributionDetails.TRTAMI132) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000274
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000274(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000274
                    Address Line 2 must be present if Address Line 1 contains 'C/-'
    
                    Legacy Rule Format:
                    (Contains(^TRTAMI131, 'C/-')) AND (^TRTAMI132 = NULL)
        
                    Technical Business Rule Format:
                    (Contains(^TRTAMI131, 'C/-')) AND (^TRTAMI132 = NULL)
            
                    Data Elements:
            
                    ^TRTAMI132 = TRTAMI:RP:StatementOfAttributionDetails:NonIndividualDetails:NonIndividualAddressDetails:AddressDetails.Line2.Text
            
                    ^TRTAMI131 = TRTAMI:RP:StatementOfAttributionDetails:NonIndividualDetails:NonIndividualAddressDetails:AddressDetails.Line1.Text
                    */
            assertion = ((statementOfAttributionDetails.TRTAMI131 == null ? false : statementOfAttributionDetails.TRTAMI131.ToUpperInvariant().Contains(@"C/-")) && statementOfAttributionDetails.TRTAMI132 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000274",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member Business Address - Address Line 2 must be provided",
                    LongDescription = @"When Member Business Address - Address Line 1 contains 'C/-', Member Business Address - Address Line 2 must be provided",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:NonIndividualDetails/tns:NonIndividualAddressDetails/tns:Line2T",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000274"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI131", Value = statementOfAttributionDetails.TRTAMI131 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI132", Value = GetValueOrEmpty(statementOfAttributionDetails.TRTAMI132) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000275
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000275(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000275
                    State code [QLD,NSW,VIC,SA,WA,NT,ACT,TAS] or a leading space or combination of spaces with a State code is present in the suburb/town field.
    
                    Legacy Rule Format:
                    (^TRTAMI136 = NULL OR ^TRTAMI136 = 'au') AND (FoundSet(^TRTAMI133, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))
        
                    Technical Business Rule Format:
                    (^TRTAMI136 = NULL OR ^TRTAMI136 = 'au') AND (FoundSet(^TRTAMI133, '"QLD","NSW","VIC","SA","WA","NT","ACT","TAS"'))
            
                    Data Elements:
            
                    ^TRTAMI133 = TRTAMI:RP:StatementOfAttributionDetails:NonIndividualDetails:NonIndividualAddressDetails:AddressDetails.LocalityName.Text
            
                    ^TRTAMI136 = TRTAMI:RP:StatementOfAttributionDetails:NonIndividualDetails:NonIndividualAddressDetails:AddressDetails.Country.Code
                    */
            assertion = ((statementOfAttributionDetails.TRTAMI136 == null || statementOfAttributionDetails.TRTAMI136 == @"au") && IsMatch(statementOfAttributionDetails.TRTAMI133, @"(^(QLD|NSW|VIC|SA|WA|NT|ACT|TAS)$)|(^(.*)( QLD | NSW | VIC | SA | WA | NT | ACT | TAS )(.*)$)|(^(QLD |NSW |VIC |SA |WA |NT |ACT |TAS )(.*)$)|(^(.*)( QLD| NSW| VIC| SA| WA| NT| ACT| TAS)$)", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000275",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member Business Address - City/Town cannot contain state code",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:NonIndividualDetails/tns:NonIndividualAddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000275"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI136", Value = GetValueOrEmpty(statementOfAttributionDetails.TRTAMI136) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI133", Value = statementOfAttributionDetails.TRTAMI133 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000276
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000276(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000276
                    Suburb/Town/City cannot contain "UNKNOWN"
    
                    Legacy Rule Format:
                    Contains(^TRTAMI133, 'UNKNOWN')
        
                    Technical Business Rule Format:
                    Contains(^TRTAMI133, 'UNKNOWN')
            
                    Data Elements:
            
                    ^TRTAMI133 = TRTAMI:RP:StatementOfAttributionDetails:NonIndividualDetails:NonIndividualAddressDetails:AddressDetails.LocalityName.Text
                    */
            assertion = (statementOfAttributionDetails.TRTAMI133 == null ? false : statementOfAttributionDetails.TRTAMI133.ToUpperInvariant().Contains(@"UNKNOWN"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000276",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member Business Address - City/Town cannot contain ""UNKNOWN""",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:NonIndividualDetails/tns:NonIndividualAddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000276"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI133", Value = statementOfAttributionDetails.TRTAMI133 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000277
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000277(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000277
                    Suburb/Town/City is mandatory
    
                    Legacy Rule Format:
                    ^TRTAMI133 = NULL AND ^TRTAMI80 <> NULL
        
                    Technical Business Rule Format:
                    ^TRTAMI133 = NULL AND ^TRTAMI80 <> NULL
            
                    Data Elements:
            
                    ^TRTAMI133 = TRTAMI:RP:StatementOfAttributionDetails:NonIndividualDetails:NonIndividualAddressDetails:AddressDetails.LocalityName.Text
            
                    ^TRTAMI80 = TRTAMI:RP:StatementOfAttributionDetails:NonIndividualDetails:OrganisationNameDetails.OrganisationalName.Text
                    */
            assertion = (statementOfAttributionDetails.TRTAMI133 == null && statementOfAttributionDetails.TRTAMI80 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000277",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member Business Address - City/Town must be provided for a non-individual",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:NonIndividualDetails/tns:NonIndividualAddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000277"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI133", Value = statementOfAttributionDetails.TRTAMI133 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI80", Value = statementOfAttributionDetails.TRTAMI80 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000278
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000278(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000278
                    Numerics [0 to 9] are not allowed in the suburb/town field.
    
                    Legacy Rule Format:
                    (^TRTAMI136 = NULL OR ^TRTAMI136 = 'au') AND (ContainsSet(^TRTAMI133, '0-9'))
        
                    Technical Business Rule Format:
                    (^TRTAMI136 = NULL OR ^TRTAMI136 = 'au') AND (ContainsSet(^TRTAMI133, '0-9'))
            
                    Data Elements:
            
                    ^TRTAMI133 = TRTAMI:RP:StatementOfAttributionDetails:NonIndividualDetails:NonIndividualAddressDetails:AddressDetails.LocalityName.Text
            
                    ^TRTAMI136 = TRTAMI:RP:StatementOfAttributionDetails:NonIndividualDetails:NonIndividualAddressDetails:AddressDetails.Country.Code
                    */
            assertion = ((statementOfAttributionDetails.TRTAMI136 == null || statementOfAttributionDetails.TRTAMI136 == @"au") && IsMatch(statementOfAttributionDetails.TRTAMI133, @"[0-9]", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000278",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member Business Address - City/Town is invalid",
                    LongDescription = @"When an Australian address is provided, Member Business Address - City/Town must not contain numeric characters",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:NonIndividualDetails/tns:NonIndividualAddressDetails/tns:LocalityNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000278"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI136", Value = GetValueOrEmpty(statementOfAttributionDetails.TRTAMI136) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI133", Value = statementOfAttributionDetails.TRTAMI133 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000290
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000290()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000290
            When any Stapled Entity Australian Business Numbers are provided, the response to 'Is the Attribution Managed Investment Trust (AMIT) Stapled to another Entity?' must be set to 'True' (yes).
    
            Legacy Rule Format:
            ^TRTAMI112 <> TRUE AND AnyOccurrence(^TRTAMI139, ^TRTAMI139 <> NULL)

            Technical Business Rule Format:
            ^TRTAMI112 <> TRUE AND AnyOccurrence(^TRTAMI139, ^TRTAMI139 <> NULL)
    
            Data Elements:
    
            ^TRTAMI112 = TRTAMI:RP:IncomeTaxStapledEntities:IncomeTax.AttributionManagedInvestmentTrustStapledStructure.Indicator
    
            ^TRTAMI139 = TRTAMI:RP:IncomeTaxStapledEntities:Identifiers.AustralianBusinessNumber.Identifier
            */
            assertion = (report.TRTAMI112 != true && (report.TRTAMI139Collection == null ? false : report.TRTAMI139Collection.Any(TRTAMI139Repeat => TRTAMI139Repeat != null)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000411",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Is the Trust Stapled to another Entity? is incorrect",
                    LongDescription = @"When any Stapled Entity Australian Business Numbers are provided, the response to Is the Trust Stapled to another Entity? must be set to 'True' (yes)",
                    Location = "/tns:TRTAMI/tns:RP/tns:IncomeTaxStapledEntities/tns:AttributionManagedInvestmentTrustStapledStructureI",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000290"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI112", Value = GetValueOrEmpty(report.TRTAMI112) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI139", Value = "TRTAMI139" });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000293
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000293(string TRTAMI139, int itemIndex3)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000293
                    The Stapled Entity Australian Business Number has failed the ABN algorithm check.
    
                    Legacy Rule Format:
                    AnyOccurrence(^TRTAMI139, ^TRTAMI139 <> NULL) AND FailsABNAlgorithm(^TRTAMI139)
        
                    Technical Business Rule Format:
                    AnyOccurrence(^TRTAMI139, ^TRTAMI139 <> NULL) AND FailsABNAlgorithm(^TRTAMI139)
            
                    Data Elements:
            
                    ^TRTAMI139 = TRTAMI:RP:IncomeTaxStapledEntities:Identifiers.AustralianBusinessNumber.Identifier
                    */
            assertion = ((report.TRTAMI139Collection == null ? false : report.TRTAMI139Collection.Any(TRTAMI139Repeat => TRTAMI139Repeat != null)) && FailsABNAlgorithm(TRTAMI139));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000293",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Stapled Entity Australian Business Number is invalid",
                    LongDescription = @"The Stapled Entity Australian Business Number has failed the ABN algorithm check",
                    Location = "/tns:TRTAMI/tns:RP/tns:IncomeTaxStapledEntities/tns:AustralianBusinessNumberIdCollection/tns:AustralianBusinessNumberId",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000293"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI139", Value = "TRTAMI139" });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000294
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000294(string TRTAMI139, int itemIndex3)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000294
                    Stapled Entity Australian Business Number must differ from the Trust Australian Business Number.
    
                    Legacy Rule Format:
                    AnyOccurrence(^TRTAMI139, ^TRTAMI139 <> NULL) AND ^TRTAMI139 = ^TRTAMI5
        
                    Technical Business Rule Format:
                    AnyOccurrence(^TRTAMI139, ^TRTAMI139 <> NULL) AND ^TRTAMI139 = ^TRTAMI5
            
                    Data Elements:
            
                    ^TRTAMI139 = TRTAMI:RP:IncomeTaxStapledEntities:Identifiers.AustralianBusinessNumber.Identifier
            
                    ^TRTAMI5 = TRTAMI:RP:Identifiers.AustralianBusinessNumber.Identifier
                    */
            assertion = ((report.TRTAMI139Collection == null ? false : report.TRTAMI139Collection.Any(TRTAMI139Repeat => TRTAMI139Repeat != null)) && TRTAMI139 == report.TRTAMI5);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000294",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Stapled Entity Australian Business Number must differ from the Trust Australian Business Number",
                    Location = "/tns:TRTAMI/tns:RP/tns:IncomeTaxStapledEntities/tns:AustralianBusinessNumberIdCollection/tns:AustralianBusinessNumberId",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000294"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI139", Value = "TRTAMI139" });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI5", Value = report.TRTAMI5 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000295
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000295(string TRTAMI139, int itemIndex3)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000295
                    Stapled Entity Australian Business Number must differ from the Trustee Australian Business Number.
    
                    Legacy Rule Format:
                    AnyOccurrence(^TRTAMI139, ^TRTAMI139 <> NULL) AND ^TRTAMI139 = ^TRTAMI29
        
                    Technical Business Rule Format:
                    AnyOccurrence(^TRTAMI139, ^TRTAMI139 <> NULL) AND ^TRTAMI139 = ^TRTAMI29
            
                    Data Elements:
            
                    ^TRTAMI139 = TRTAMI:RP:IncomeTaxStapledEntities:Identifiers.AustralianBusinessNumber.Identifier
            
                    ^TRTAMI29 = TRTAMI:RP:TrusteeDetails:Identifiers.AustralianBusinessNumber.Identifier
                    */
            assertion = ((report.TRTAMI139Collection == null ? false : report.TRTAMI139Collection.Any(TRTAMI139Repeat => TRTAMI139Repeat != null)) && TRTAMI139 == report.TRTAMI29);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000295",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Stapled Entity Australian Business Number must differ from the Trustee Australian Business Number",
                    Location = "/tns:TRTAMI/tns:RP/tns:IncomeTaxStapledEntities/tns:AustralianBusinessNumberIdCollection/tns:AustralianBusinessNumberId",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000295"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI139", Value = "TRTAMI139" });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI29", Value = GetValueOrEmpty(report.TRTAMI29) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000301
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000301()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000301
            When a positive amount is provided for Total Payments Received from Related Entities during the Income Year, the response to Did the Trust receive Payments from Related Entities during the Income Year? must be set to 'True' (yes)
    
            Legacy Rule Format:
            ^TRTAMI117 <> TRUE AND ^TRTAMI142 > 0

            Technical Business Rule Format:
            ^TRTAMI117 <> TRUE AND ^TRTAMI142 > 0
    
            Data Elements:
    
            ^TRTAMI117 = TRTAMI:RP:RelatedEntitiesPayments:IncomeTax.AttributionManagedInvestmentTrustRelatedEntitiesPayment.Indicator
    
            ^TRTAMI142 = TRTAMI:RP:RelatedEntitiesPayments:IncomeTax.RelatedEntitiesPaymentsReceived.Amount
            */
            assertion = (report.TRTAMI117 != true && report.TRTAMI142.GetValueOrDefault() > 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000413",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Did the Trust receive Payments from Related Entities during the Income Year? is incorrect",
                    LongDescription = @"When a positive amount is provided for Total Payments Received from Related Entities during the Income Year, the response to Did the Trust receive Payments from Related Entities during the Income Year? must be set to 'True' (yes)",
                    Location = "/tns:TRTAMI/tns:RP/tns:RelatedEntitiesPayments/tns:IncomeTaxAttributionManagedInvestmentTrustRelatedEntitiesPaymentI",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000301"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI117", Value = GetValueOrEmpty(report.TRTAMI117) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI142", Value = GetValueOrEmpty(report.TRTAMI142) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000303
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000303()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000303
            When the response to Did the Trust receive Payments from Related Entities during the Income Year? is 'True' (yes), a positive amount for Total Payments Received from Related Entities during the Income Year must be provided.
    
            Legacy Rule Format:
            ^TRTAMI117 = TRUE AND (^TRTAMI142 = NULL OR ^TRTAMI142 = 0)

            Technical Business Rule Format:
            ^TRTAMI117 = TRUE AND (^TRTAMI142 = NULL OR ^TRTAMI142 = 0)
    
            Data Elements:
    
            ^TRTAMI142 = TRTAMI:RP:RelatedEntitiesPayments:IncomeTax.RelatedEntitiesPaymentsReceived.Amount
    
            ^TRTAMI117 = TRTAMI:RP:RelatedEntitiesPayments:IncomeTax.AttributionManagedInvestmentTrustRelatedEntitiesPayment.Indicator
            */
            assertion = (report.TRTAMI117 == true && (report.TRTAMI142 == null || report.TRTAMI142 == 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000414",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Total Payments Received from Related Entities during the Income Year is invalid",
                    LongDescription = @"When the response to Did the Trust receive Payments from Related Entities during the Income Year? is 'True' (yes), a positive amount for Total Payments Received from Related Entities during the Income Year must be provided.",
                    Location = "/tns:TRTAMI/tns:RP/tns:RelatedEntitiesPayments/tns:IncomeTaxReceivedA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000303"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI117", Value = GetValueOrEmpty(report.TRTAMI117) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI142", Value = GetValueOrEmpty(report.TRTAMI142) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000305
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000305(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000305
                    For an Australian address, Postal/Zip Code is mandatory
    
                    Legacy Rule Format:
                    ^TRTAMI143 = NULL AND ^TRTAMI77 <> NULL AND (^TRTAMI130 = NULL OR ^TRTAMI130 = 'au')
        
                    Technical Business Rule Format:
                    ^TRTAMI143 = NULL AND ^TRTAMI77 <> NULL AND (^TRTAMI130 = NULL OR ^TRTAMI130 = 'au')
            
                    Data Elements:
            
                    ^TRTAMI143 = TRTAMI:RP:StatementOfAttributionDetails:IndividualPersonNameDetails:IndividualAddressDetails:AddressDetails.PostalCode.Text
            
                    ^TRTAMI77 = TRTAMI:RP:StatementOfAttributionDetails:IndividualPersonNameDetails:PersonNameDetails.FamilyName.Text
            
                    ^TRTAMI130 = TRTAMI:RP:StatementOfAttributionDetails:IndividualPersonNameDetails:IndividualAddressDetails:AddressDetails.Country.Code
                    */
            assertion = (statementOfAttributionDetails.TRTAMI143 == null && statementOfAttributionDetails.TRTAMI77 != null && (statementOfAttributionDetails.TRTAMI130 == null || statementOfAttributionDetails.TRTAMI130 == @"au"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000305",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member Residential Address - Postal/Zip Code is required for an Australian address",
                    LongDescription = @"For an Australian address, Member Residential Address - Postal/Zip Code is required for an individual. Determine whether the address is Australian or overseas and correct accordingly.",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:IndividualPersonNameDetails/tns:IndividualAddressDetails/tns:PostalCodeT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000305"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI143", Value = GetValueOrEmpty(statementOfAttributionDetails.TRTAMI143) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI77", Value = statementOfAttributionDetails.TRTAMI77 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI130", Value = GetValueOrEmpty(statementOfAttributionDetails.TRTAMI130) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000306
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000306(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000306
                    For an Australian address, Member Residential Address - Postal/Zip Code is invalid. Valid range is 0200 - 9999.
    
                    Legacy Rule Format:
                    ^TRTAMI77 <> NULL AND (^TRTAMI130 = NULL OR ^TRTAMI130 = 'au') AND ^TRTAMI143 <> NULL AND (Length(^TRTAMI143) <> 4 OR (^TRTAMI143 < 0200 OR ^TRTAMI143 > 9999))
        
                    Technical Business Rule Format:
                    ^TRTAMI77 <> NULL AND (^TRTAMI130 = NULL OR ^TRTAMI130 = 'au') AND ^TRTAMI143 <> NULL AND (Length(^TRTAMI143) <> 4 OR (^TRTAMI143 < 0200 OR ^TRTAMI143 > 9999))
            
                    Data Elements:
            
                    ^TRTAMI143 = TRTAMI:RP:StatementOfAttributionDetails:IndividualPersonNameDetails:IndividualAddressDetails:AddressDetails.PostalCode.Text
            
                    ^TRTAMI77 = TRTAMI:RP:StatementOfAttributionDetails:IndividualPersonNameDetails:PersonNameDetails.FamilyName.Text
            
                    ^TRTAMI130 = TRTAMI:RP:StatementOfAttributionDetails:IndividualPersonNameDetails:IndividualAddressDetails:AddressDetails.Country.Code
                    */
            assertion = (statementOfAttributionDetails.TRTAMI77 != null && (statementOfAttributionDetails.TRTAMI130 == null || statementOfAttributionDetails.TRTAMI130 == @"au") && statementOfAttributionDetails.TRTAMI143 != null && (Length(statementOfAttributionDetails.TRTAMI143) != 4 || (AsNumeric(statementOfAttributionDetails.TRTAMI143) < 200 || AsNumeric(statementOfAttributionDetails.TRTAMI143) > 9999)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000306",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member Residential Address - Postal/Zip Code is invalid",
                    LongDescription = @"For an Australian address, Member Residential Address - Postal/Zip Code is invalid. Valid range is 0200 - 9999.",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:IndividualPersonNameDetails/tns:IndividualAddressDetails/tns:PostalCodeT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000306"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI77", Value = statementOfAttributionDetails.TRTAMI77 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI130", Value = GetValueOrEmpty(statementOfAttributionDetails.TRTAMI130) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI143", Value = GetValueOrEmpty(statementOfAttributionDetails.TRTAMI143) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000308
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000308(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000308
                    For an Australian address, Postal/Zip Code is mandatory
    
                    Legacy Rule Format:
                    ^TRTAMI144 = NULL AND ^TRTAMI80 <> NULL AND (^TRTAMI136 = NULL OR ^TRTAMI136 = 'au')
        
                    Technical Business Rule Format:
                    ^TRTAMI144 = NULL AND ^TRTAMI80 <> NULL AND (^TRTAMI136 = NULL OR ^TRTAMI136 = 'au')
            
                    Data Elements:
            
                    ^TRTAMI144 = TRTAMI:RP:StatementOfAttributionDetails:NonIndividualDetails:NonIndividualAddressDetails:AddressDetails.PostalCode.Text
            
                    ^TRTAMI80 = TRTAMI:RP:StatementOfAttributionDetails:NonIndividualDetails:OrganisationNameDetails.OrganisationalName.Text
            
                    ^TRTAMI136 = TRTAMI:RP:StatementOfAttributionDetails:NonIndividualDetails:NonIndividualAddressDetails:AddressDetails.Country.Code
                    */
            assertion = (statementOfAttributionDetails.TRTAMI144 == null && statementOfAttributionDetails.TRTAMI80 != null && (statementOfAttributionDetails.TRTAMI136 == null || statementOfAttributionDetails.TRTAMI136 == @"au"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000308",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member Business Address - Postal/Zip Code is required for an Australian address",
                    LongDescription = @"For an Australian address, Member Business Address - Postal/Zip Code is required for a non-individual. Determine whether the address is Australian or overseas and correct accordingly.",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:NonIndividualDetails/tns:NonIndividualAddressDetails/tns:PostalCodeT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000308"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI144", Value = GetValueOrEmpty(statementOfAttributionDetails.TRTAMI144) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI80", Value = statementOfAttributionDetails.TRTAMI80 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI136", Value = GetValueOrEmpty(statementOfAttributionDetails.TRTAMI136) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000309
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000309(TRTAMI2026.RP_StatementOfAttributionDetails statementOfAttributionDetails, int itemIndex2)
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000309
                    For an Australian address, Member Business Address - Postal/Zip Code is invalid. Valid range is 0200 - 9999.
    
                    Legacy Rule Format:
                    ^TRTAMI80 <> NULL AND (^TRTAMI136 = NULL OR ^TRTAMI136 = 'au') AND ^TRTAMI144 <> NULL AND (Length(^TRTAMI144) <> 4 OR (^TRTAMI144 < 0200 OR ^TRTAMI144 > 9999))
        
                    Technical Business Rule Format:
                    ^TRTAMI80 <> NULL AND (^TRTAMI136 = NULL OR ^TRTAMI136 = 'au') AND ^TRTAMI144 <> NULL AND (Length(^TRTAMI144) <> 4 OR (^TRTAMI144 < 0200 OR ^TRTAMI144 > 9999))
            
                    Data Elements:
            
                    ^TRTAMI144 = TRTAMI:RP:StatementOfAttributionDetails:NonIndividualDetails:NonIndividualAddressDetails:AddressDetails.PostalCode.Text
            
                    ^TRTAMI80 = TRTAMI:RP:StatementOfAttributionDetails:NonIndividualDetails:OrganisationNameDetails.OrganisationalName.Text
            
                    ^TRTAMI136 = TRTAMI:RP:StatementOfAttributionDetails:NonIndividualDetails:NonIndividualAddressDetails:AddressDetails.Country.Code
                    */
            assertion = (statementOfAttributionDetails.TRTAMI80 != null && (statementOfAttributionDetails.TRTAMI136 == null || statementOfAttributionDetails.TRTAMI136 == @"au") && statementOfAttributionDetails.TRTAMI144 != null && (Length(statementOfAttributionDetails.TRTAMI144) != 4 || (AsNumeric(statementOfAttributionDetails.TRTAMI144) < 200 || AsNumeric(statementOfAttributionDetails.TRTAMI144) > 9999)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000309",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Member Business Address - Postal/Zip Code is invalid",
                    LongDescription = @"For an Australian address, Member Business Address - Postal/Zip Code is invalid. Valid range is 0200 - 9999.",
                    Location = "/tns:TRTAMI/tns:RP/tns:StatementOfAttributionDetailsCollection/tns:StatementOfAttributionDetails" + OccurrenceIndex(statementOfAttributionDetails.OccurrenceIndex) + "/tns:NonIndividualDetails/tns:NonIndividualAddressDetails/tns:PostalCodeT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000309"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI80", Value = statementOfAttributionDetails.TRTAMI80 });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI136", Value = GetValueOrEmpty(statementOfAttributionDetails.TRTAMI136) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI144", Value = GetValueOrEmpty(statementOfAttributionDetails.TRTAMI144) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000312
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000312()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000312
            Year of return must be either the current or next (future) lodgment year associated with this service. If a future year lodgment is required, then it must be lodged no later than the 15th of June of the current year, unless it is an amendment. After the 15th of June, a return for that lodgment year must be submitted from the 1st of July, using the new service associated to that specific lodgment year.
    
            Legacy Rule Format:
            (NotInSet(^TRTAMI1, '"2026","2027"')) OR (^TRTAMI1 = 2027 AND Today() > AsDate('2027-06-15') AND ^TRTAMI6 <> TRUE)

            Technical Business Rule Format:
            (NotInSet(^TRTAMI1, '"2026","2027"')) OR (^TRTAMI1 = 2027 AND Today() > AsDate('2027-06-15') AND ^TRTAMI6 <> TRUE)
    
            Data Elements:
    
            ^TRTAMI1 = TRTAMI:RP:Report.TargetFinancial.Year
    
            ^TRTAMI6 = TRTAMI:RP:ReportAmendmentDetails:Report.Amendment.Indicator
            */
            assertion = (!(IsMatch(report.TRTAMI1.GetValueOrDefault().ToString(), @"^(2026|2027)$", RegexOptions.IgnoreCase)) || report.TRTAMI1.GetValueOrDefault() == 2027 && DateTime.Now.Date > AsDate(@"2027-06-15") && report.TRTAMI6 != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500127",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Year of return is invalid",
                    LongDescription = @"Year of return must be either the current or next (future) lodgment year associated with this service. If a future year lodgment is required, then it must be lodged no later than the 15th of June of the current year, unless it is an amendment. After the 15th of June, a return for that lodgment year must be submitted from the 1st of July, using the new service associated to that specific lodgment year.",
                    Location = "/tns:TRTAMI/tns:RP/tns:TargetFinancialY",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000312"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI1", Value = GetValueOrEmpty(report.TRTAMI1) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI6", Value = GetValueOrEmpty(report.TRTAMI6) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000331
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000331()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000331
            If the trust has self-assessed that they are a Country by Country Reporting Entity (CbCRE), then the Significant Global Entity (SGE) Indicator must be true.
    
            Legacy Rule Format:
            ^TRTAMI163 = TRUE AND (^TRTAMI162 = NULL OR ^TRTAMI162 = FALSE)

            Technical Business Rule Format:
            ^TRTAMI163 = TRUE AND (^TRTAMI162 = NULL OR ^TRTAMI162 = FALSE)
    
            Data Elements:
    
            ^TRTAMI163 = TRTAMI:RP:TrustDetails:InternationalDealings.CountryByCountryReportingEntityStatus.Indicator
    
            ^TRTAMI162 = TRTAMI:RP:TrustDetails:OrganisationDetails.SignificantGlobalEntityStatus.Indicator
            */
            assertion = (report.TRTAMI163 == true && (report.TRTAMI162 == null || report.TRTAMI162 == false));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000331",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Significant Global Entity (SGE)  must be 'TRUE'.",
                    LongDescription = @"If the trust has self-assessed that they are a Country by Country Reporting Entity (CbCRE), then the Significant Global Entity (SGE) Indicator must be true.",
                    Location = "/tns:TRTAMI/tns:RP/tns:TrustDetails/tns:InternationalDealingsCountryByCountryReportingEntityStatusI",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000331"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI163", Value = GetValueOrEmpty(report.TRTAMI163) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI162", Value = GetValueOrEmpty(report.TRTAMI162) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000347
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000347()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000347
            Aggregated turnover must be provided if aggregated turnover range of 'P' is supplied.
    
            Legacy Rule Format:
            ^TRTAMI160 = 'P' AND (^TRTAMI161 = NULL OR ^TRTAMI161 = 0)

            Technical Business Rule Format:
            ^TRTAMI160 = 'P' AND (^TRTAMI161 = NULL OR ^TRTAMI161 = 0)
    
            Data Elements:
    
            ^TRTAMI160 = TRTAMI:RP:KeyFinancialInformation:Income.AggregatedTurnoverRange.Code
    
            ^TRTAMI161 = TRTAMI:RP:KeyFinancialInformation:Income.AggregatedTurnover.Amount
            */
            assertion = (report.TRTAMI160 == @"P" && (report.TRTAMI161 == null || report.TRTAMI161 == 0));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000347",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Aggregated turnover must be provided if aggregated turnover range of 'P' is supplied.",
                    Location = "/tns:TRTAMI/tns:RP/tns:KeyFinancialInformation/tns:IncomeAggregatedTurnoverRangeC",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000347"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI160", Value = GetValueOrEmpty(report.TRTAMI160) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI161", Value = GetValueOrEmpty(report.TRTAMI161) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000350
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000350()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000350
            Name field cannot contain 'space hyphen space'.
    
            Legacy Rule Format:
            Contains(^TRTAMI94, ' - ')

            Technical Business Rule Format:
            Contains(^TRTAMI94, ' - ')
    
            Data Elements:
    
            ^TRTAMI94 = TRTAMI:RP:ContactDetails:PersonUnstructuredName.FullName.Text
            */
            assertion = (report.TRTAMI94 == null ? false : report.TRTAMI94.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500057",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer Contact Name cannot contain space hyphen space combinations",
                    Location = "/tns:TRTAMI/tns:RP/tns:ContactDetails/tns:PersonUnstructuredNameFullNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000350"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI94", Value = report.TRTAMI94 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000351
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000351()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000351
            Name field cannot contain 'space hyphen space'.
    
            Legacy Rule Format:
            Contains(^TRTAMI83, ' - ')

            Technical Business Rule Format:
            Contains(^TRTAMI83, ' - ')
    
            Data Elements:
    
            ^TRTAMI83 = TRTAMI:INT:PersonUnstructuredName.FullName.Text
            */
            assertion = (report.TRTAMI83 == null ? false : report.TRTAMI83.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500057",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Declarer Contact Name cannot contain space hyphen space combinations",
                    Location = "/tns:TRTAMI/tns:INT/tns:PersonUnstructuredNameFullNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000351"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI83", Value = report.TRTAMI83 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000352
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000352()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000352
            Signature date must not be later than today
    
            Legacy Rule Format:
            (^TRTAMI98 > Today())

            Technical Business Rule Format:
            (^TRTAMI98 > Today())
    
            Data Elements:
    
            ^TRTAMI98 = TRTAMI:RP:ContactDetails:Declaration:Declaration.Signature.Date
            */
            assertion = (report.TRTAMI98.GetValueOrDefault() > DateTime.Now.Date);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430255",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Signature date must not be later than today",
                    Location = "/tns:TRTAMI/tns:RP/tns:ContactDetails/tns:Declaration/tns:SignatureD",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000352"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI98", Value = GetValueOrEmptyDate(report.TRTAMI98) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000353
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000353()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000353
            Signature date must not be later than today
    
            Legacy Rule Format:
            (^TRTAMI87 > Today())

            Technical Business Rule Format:
            (^TRTAMI87 > Today())
    
            Data Elements:
    
            ^TRTAMI87 = TRTAMI:INT:Declaration:Declaration.Signature.Date
            */
            assertion = (report.TRTAMI87.GetValueOrDefault() > DateTime.Now.Date);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.430255",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Signature date must not be later than today",
                    Location = "/tns:TRTAMI/tns:INT/tns:Declaration/tns:SignatureD",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000353"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI87", Value = GetValueOrEmptyDate(report.TRTAMI87) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000354
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000354()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000354
            The CCIV Sub-fund Australian Registered Fund Number has failed the ACN algorithm check
    
            Legacy Rule Format:
            (^TRTAMI180 <> NULL) AND (FailsACNAlgorithm(^TRTAMI180))

            Technical Business Rule Format:
            (^TRTAMI180 <> NULL) AND (FailsACNAlgorithm(^TRTAMI180))
    
            Data Elements:
    
            ^TRTAMI180 = TRTAMI:RP:Identifiers.AustralianRegisteredFundNumber.Identifier
            */
            assertion = (report.TRTAMI180 != null && FailsACNAlgorithm(report.TRTAMI180));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000354",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The CCIV Sub-fund Australian Registered Fund Number is invalid",
                    LongDescription = @"The CCIV Sub-fund Australian Registered Fund Number has failed the ACN algorithm check",
                    Location = "/tns:TRTAMI/tns:RP/tns:AustralianRegisteredFundNumberId",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000354"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI180", Value = GetValueOrEmpty(report.TRTAMI180) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000355
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000355()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000355
            The CCIV Australian Company Number has failed the ACN algorithm check
    
            Legacy Rule Format:
            (^TRTAMI191 <> NULL) AND (FailsACNAlgorithm(^TRTAMI191))

            Technical Business Rule Format:
            (^TRTAMI191 <> NULL) AND (FailsACNAlgorithm(^TRTAMI191))
    
            Data Elements:
    
            ^TRTAMI191 = TRTAMI:RP:TrusteeDetails:Identifiers.AustralianCompanyNumber.Identifier
            */
            assertion = (report.TRTAMI191 != null && FailsACNAlgorithm(report.TRTAMI191));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000355",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"CCIV Australian Company Number is invalid",
                    LongDescription = @"The CCIV Australian Company Number has failed the ACN algorithm check",
                    Location = "/tns:TRTAMI/tns:RP/tns:TrusteeDetails/tns:AustralianCompanyNumberId",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000355"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI191", Value = GetValueOrEmpty(report.TRTAMI191) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000356
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000356()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000356
            When Is this lodgment for an attribution CCIV sub-fund? is false or not provided,CCIV Sub-fund Australian Registered Fund Number, Are you relying on a safe harbour provision in determining AMIT eligibility for the period covered by this income tax return, Safe harbour provision period code, Do you own shares that are referable to a sub-fund within the same CCIV as you, Corporate Director Details or Have you determined that the CCIV sub-fund meets the AMIT eligibility requirements for the period covered by this income tax return? must not be provided
    
            Legacy Rule Format:
            (^TRTAMI192 = FALSE OR ^TRTAMI192 = NULL) AND (^TRTAMI180 <> NULL OR ^TRTAMI181 <> NULL OR ^TRTAMI182 <> NULL OR ^TRTAMI189 <> NULL OR ^TRTAMI1036 <> NULL OR ^TRTAMI201 <> NULL)

            Technical Business Rule Format:
            (^TRTAMI192 = FALSE OR ^TRTAMI192 = NULL) AND (^TRTAMI180 <> NULL OR ^TRTAMI181 <> NULL OR ^TRTAMI182 <> NULL OR ^TRTAMI189 <> NULL OR ^TRTAMI1036 <> NULL OR ^TRTAMI201 <> NULL)
    
            Data Elements:
    
            ^TRTAMI192 = TRTAMI:RP:Report.CorporateCollectiveInvestmentVehicleSubFund.Indicator
    
            ^TRTAMI180 = TRTAMI:RP:Identifiers.AustralianRegisteredFundNumber.Identifier
    
            ^TRTAMI181 = TRTAMI:RP:AMITEligibility:IncomeTax.EligibleInvestmentBusinessSafeHarbour.Indicator
    
            ^TRTAMI182 = TRTAMI:RP:AMITEligibility:IncomeTax.SafeHarbourPeriod.Code
    
            ^TRTAMI189 = TRTAMI:RP:Assets.Investment.CrossOtherCCIVSubfund.Indicator
    
            ^TRTAMI201 = TRTAMI:RP:AMITEligibility:IncomeTax.AttributionManagedInvestmentEligibility.Indicator
    
            ^TRTAMI1036 = TRTAMI:RP:TrusteeDetails:DirectorDetails
            */
            assertion = ((report.TRTAMI192 == false || report.TRTAMI192 == null) && (report.TRTAMI180 != null || report.TRTAMI181 != null || report.TRTAMI182 != null || report.TRTAMI189 != null || report.RP_TrusteeDetails_DirectorDetailsCollectionExists != false || report.TRTAMI201 != null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000356",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Corporate Collective Investment Vehicle fields must not be provided",
                    LongDescription = @"When Is this lodgment for an attribution CCIV sub-fund? is false or not provided,CCIV Sub-fund Australian Registered Fund Number, Are you relying on a safe harbour provision in determining AMIT eligibility for the period covered by this income tax return, Safe harbour provision period code, Do you own shares that are referable to a sub-fund within the same CCIV as you, Corporate Director Details or Have you determined that the CCIV sub-fund meets the AMIT eligibility requirements for the period covered by this income tax return? must not be provided",
                    Location = "/tns:TRTAMI/tns:RP/tns:CorporateCollectiveInvestmentVehicleSubFundI",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000356"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI192", Value = GetValueOrEmpty(report.TRTAMI192) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI180", Value = GetValueOrEmpty(report.TRTAMI180) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI181", Value = GetValueOrEmpty(report.TRTAMI181) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI182", Value = GetValueOrEmpty(report.TRTAMI182) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI189", Value = GetValueOrEmpty(report.TRTAMI189) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI201", Value = GetValueOrEmpty(report.TRTAMI201) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000357
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000357()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000357
            When Are you relying on a safe harbour provision in determining AMIT eligibility for the period covered by this income tax return is true, Safe harbour provision period code must be provided
    
            Legacy Rule Format:
            ^TRTAMI181 = TRUE AND ^TRTAMI182 = NULL

            Technical Business Rule Format:
            ^TRTAMI181 = TRUE AND ^TRTAMI182 = NULL
    
            Data Elements:
    
            ^TRTAMI182 = TRTAMI:RP:AMITEligibility:IncomeTax.SafeHarbourPeriod.Code
    
            ^TRTAMI181 = TRTAMI:RP:AMITEligibility:IncomeTax.EligibleInvestmentBusinessSafeHarbour.Indicator
            */
            assertion = (report.TRTAMI181 == true && report.TRTAMI182 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000357",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Safe harbour provision period code must be provided",
                    LongDescription = @"When Are you relying on a safe harbour provision in determining AMIT eligibility for the period covered by this income tax return is true, Safe harbour provision period code must be provided",
                    Location = "/tns:TRTAMI/tns:RP/tns:AMITEligibility/tns:IncomeTaxSafeHarbourPeriodC",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000357"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI181", Value = GetValueOrEmpty(report.TRTAMI181) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI182", Value = GetValueOrEmpty(report.TRTAMI182) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000359
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000359()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000359
            Number of Members in the CCIV sub-fund at the end of the income year must not be less than the count of Statement Of Attribution Details
    
            Legacy Rule Format:
            ^TRTAMI193 <> NULL AND ^TRTAMI193 < Count(^TRTAMI1022)

            Technical Business Rule Format:
            ^TRTAMI193 <> NULL AND ^TRTAMI193 < Count(^TRTAMI1022)
    
            Data Elements:
    
            ^TRTAMI193 = TRTAMI:RP:TrustDetails:OrganisationControl.RegisteredMember.Count
    
            ^TRTAMI1022 = TRTAMI:RP:StatementOfAttributionDetails
            */
            assertion = (report.TRTAMI193 != null && report.TRTAMI193.GetValueOrDefault() < Count(report.RP_StatementOfAttributionDetailsCollectionCount));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000359",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Number of Members in the CCIV sub-fund at the end of the income year must not be less than the count of Statement Of Attribution Details",
                    Location = "/tns:TRTAMI/tns:RP/tns:TrustDetails/tns:OrganisationControlRegisteredMemberCt",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000359"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI193", Value = report.TRTAMI193.GetValueOrDefault().ToString() });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000400
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000400()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000400
            When Is this lodgment for an attribution CCIV sub-fund? is true, Separate Attribution Managed Investment Trust Treatment must not be provided
    
            Legacy Rule Format:
            ^TRTAMI192 = TRUE AND ^TRTAMI1014 <> NULL

            Technical Business Rule Format:
            ^TRTAMI192 = TRUE AND ^TRTAMI1014 <> NULL
    
            Data Elements:
    
            ^TRTAMI192 = TRTAMI:RP:Report.CorporateCollectiveInvestmentVehicleSubFund.Indicator
    
            ^TRTAMI1014 = TRTAMI:RP:MultiClassTreatment
            */
            assertion = (report.TRTAMI192 == true && report.RP_MultiClassTreatmentCollectionExists != false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.020400",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Fields not applicable for a Corporate Collective Investment Vehicle tax return must not be provided",
                    LongDescription = @"When Is this lodgment for an attribution CCIV sub-fund? is true, Separate Attribution Managed Investment Trust Treatment must not be provided",
                    Location = "/tns:TRTAMI/tns:RP/tns:CorporateCollectiveInvestmentVehicleSubFundI",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000400"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI192", Value = GetValueOrEmpty(report.TRTAMI192) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000402
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000402()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000402
            When Is this lodgment for an attribution CCIV sub-fund? is true,CCIV Sub-fund Australian Registered Fund Number, CCIV Australian Company Number,  Are you relying on a safe harbour provision in determining AMIT eligibility for the period covered by this income tax return, Description of Main Business, Number of Members in the CCIV sub-fund at the end of the income year, Do you own shares that are referable to a sub-fund within the same CCIV as you, Corporate Director Details, and Have you determined that the CCIV sub-fund meets the AMIT eligibility requirements for the period covered by this income tax return? must be provided
    
            Legacy Rule Format:
            ^TRTAMI192 = TRUE AND (^TRTAMI180 = NULL OR ^TRTAMI191 = NULL OR ^TRTAMI181 = NULL OR ^TRTAMI44 = NULL OR ^TRTAMI193 = NULL OR ^TRTAMI189 = NULL OR ^TRTAMI1036 = NULL or ^TRTAMI201 = NULL)

            Technical Business Rule Format:
            ^TRTAMI192 = TRUE AND (^TRTAMI180 = NULL OR ^TRTAMI191 = NULL OR ^TRTAMI181 = NULL OR ^TRTAMI44 = NULL OR ^TRTAMI193 = NULL OR ^TRTAMI189 = NULL OR ^TRTAMI1036 = NULL or ^TRTAMI201 = NULL)
    
            Data Elements:
    
            ^TRTAMI192 = TRTAMI:RP:Report.CorporateCollectiveInvestmentVehicleSubFund.Indicator
    
            ^TRTAMI44 = TRTAMI:RP:TrustDetails:OrganisationDetails.MainIncomeActivity.Description
    
            ^TRTAMI180 = TRTAMI:RP:Identifiers.AustralianRegisteredFundNumber.Identifier
    
            ^TRTAMI181 = TRTAMI:RP:AMITEligibility:IncomeTax.EligibleInvestmentBusinessSafeHarbour.Indicator
    
            ^TRTAMI189 = TRTAMI:RP:Assets.Investment.CrossOtherCCIVSubfund.Indicator
    
            ^TRTAMI191 = TRTAMI:RP:TrusteeDetails:Identifiers.AustralianCompanyNumber.Identifier
    
            ^TRTAMI193 = TRTAMI:RP:TrustDetails:OrganisationControl.RegisteredMember.Count
    
            ^TRTAMI201 = TRTAMI:RP:AMITEligibility:IncomeTax.AttributionManagedInvestmentEligibility.Indicator
    
            ^TRTAMI1036 = TRTAMI:RP:TrusteeDetails:DirectorDetails
            */
            assertion = (report.TRTAMI192 == true && (report.TRTAMI180 == null || report.TRTAMI191 == null || report.TRTAMI181 == null || report.TRTAMI44 == null || report.TRTAMI193 == null || report.TRTAMI189 == null || report.RP_TrusteeDetails_DirectorDetailsCollectionExists == false || report.TRTAMI201 == null));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000402",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Corporate Collective Investment Vehicle fields must be provided",
                    LongDescription = @"When Is this lodgment for an attribution CCIV sub-fund? is true,CCIV Sub-fund Australian Registered Fund Number, CCIV Australian Company Number,  Are you relying on a safe harbour provision in determining AMIT eligibility for the period covered by this income tax return, Description of Main Business, Number of Members in the CCIV sub-fund at the end of the income year, Do you own shares that are referable to a sub-fund within the same CCIV as you, Corporate Director Details, and Have you determined that the CCIV sub-fund meets the AMIT eligibility requirements for the period covered by this income tax return? must be provided",
                    Location = "/tns:TRTAMI/tns:RP/tns:CorporateCollectiveInvestmentVehicleSubFundI",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000402"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI192", Value = GetValueOrEmpty(report.TRTAMI192) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI180", Value = GetValueOrEmpty(report.TRTAMI180) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI191", Value = GetValueOrEmpty(report.TRTAMI191) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI181", Value = GetValueOrEmpty(report.TRTAMI181) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI44", Value = GetValueOrEmpty(report.TRTAMI44) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI193", Value = report.TRTAMI193.GetValueOrDefault().ToString() });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI189", Value = GetValueOrEmpty(report.TRTAMI189) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI201", Value = GetValueOrEmpty(report.TRTAMI201) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000403
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000403()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000403
            When Is this lodgment for an attribution CCIV sub-fund? is false, Separate Attribution Managed Investment Trust Treatment must be provided
    
            Legacy Rule Format:
            ^TRTAMI192 = FALSE AND ^TRTAMI1014 = NULL

            Technical Business Rule Format:
            ^TRTAMI192 = FALSE AND ^TRTAMI1014 = NULL
    
            Data Elements:
    
            ^TRTAMI1014 = TRTAMI:RP:MultiClassTreatment
    
            ^TRTAMI192 = TRTAMI:RP:Report.CorporateCollectiveInvestmentVehicleSubFund.Indicator
            */
            assertion = (report.TRTAMI192 == false && report.RP_MultiClassTreatmentCollectionExists == false);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000403",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Separate Attribution Managed Investment Trust Treatment must be provided",
                    LongDescription = @"When Is this lodgment for an attribution CCIV sub-fund? is false, Separate Attribution Managed Investment Trust Treatment must be provided",
                    Location = "/tns:TRTAMI/tns:RP/tns:MultiClassTreatment",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000403"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI192", Value = GetValueOrEmpty(report.TRTAMI192) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000404
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000404()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000404
            When Are you relying on a safe harbour provision in determining AMIT eligibility for the period covered by this income tax return is false, Safe harbour provision period code must not be provided
    
            Legacy Rule Format:
            ^TRTAMI181 = FALSE AND ^TRTAMI182 <> NULL

            Technical Business Rule Format:
            ^TRTAMI181 = FALSE AND ^TRTAMI182 <> NULL
    
            Data Elements:
    
            ^TRTAMI182 = TRTAMI:RP:AMITEligibility:IncomeTax.SafeHarbourPeriod.Code
    
            ^TRTAMI181 = TRTAMI:RP:AMITEligibility:IncomeTax.EligibleInvestmentBusinessSafeHarbour.Indicator
            */
            assertion = (report.TRTAMI181 == false && report.TRTAMI182 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000404",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Safe harbour provision period code must not be provided",
                    LongDescription = @"When Are you relying on a safe harbour provision in determining AMIT eligibility for the period covered by this income tax return is false, Safe harbour provision period code must not be provided",
                    Location = "/tns:TRTAMI/tns:RP/tns:AMITEligibility/tns:IncomeTaxSafeHarbourPeriodC",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000404"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI181", Value = GetValueOrEmpty(report.TRTAMI181) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI182", Value = GetValueOrEmpty(report.TRTAMI182) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000405
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000405()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000405
            Corporate Director Name cannot contain space hyphen space combinations
    
            Legacy Rule Format:
            Contains(^TRTAMI197, ' - ')

            Technical Business Rule Format:
            Contains(^TRTAMI197, ' - ')
    
            Data Elements:
    
            ^TRTAMI197 = TRTAMI:RP:TrusteeDetails:DirectorDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.TRTAMI197 == null ? false : report.TRTAMI197.ToUpperInvariant().Contains(@" - "));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000405",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Corporate Director Name cannot contain space hyphen space combinations",
                    Location = "/tns:TRTAMI/tns:RP/tns:TrusteeDetails/tns:DirectorDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000405"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI197", Value = report.TRTAMI197 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000406
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000406()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000406
            Corporate Director Name cannot contain "P/L"
    
            Legacy Rule Format:
            Contains(^TRTAMI197, 'P/L')

            Technical Business Rule Format:
            Contains(^TRTAMI197, 'P/L')
    
            Data Elements:
    
            ^TRTAMI197 = TRTAMI:RP:TrusteeDetails:DirectorDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (report.TRTAMI197 == null ? false : report.TRTAMI197.ToUpperInvariant().Contains(@"P/L"));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000406",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Corporate Director Name cannot contain ""P/L""",
                    Location = "/tns:TRTAMI/tns:RP/tns:TrusteeDetails/tns:DirectorDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000406"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI197", Value = report.TRTAMI197 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000407
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000407()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000407
            Corporate Director Name cannot contain repeated hyphen, apostrophe, or space characters
    
            Legacy Rule Format:
            ContainsSet(^TRTAMI197, '"--","''","  "')

            Technical Business Rule Format:
            ContainsSet(^TRTAMI197, '"--","''","  "')
    
            Data Elements:
    
            ^TRTAMI197 = TRTAMI:RP:TrusteeDetails:DirectorDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.TRTAMI197, @"--|''|  ", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000407",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Corporate Director Name cannot contain repeated hyphen, apostrophe, or space characters",
                    Location = "/tns:TRTAMI/tns:RP/tns:TrusteeDetails/tns:DirectorDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000407"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI197", Value = report.TRTAMI197 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000408
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000408()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000408
            Corporate Director Name cannot start with "T/A" and end with "Pship", "P'ship" or "P/Ship"
    
            Legacy Rule Format:
            (StartsWith(^TRTAMI197, 'T/A ')) AND (EndsWithSet(^TRTAMI197, '" Pship"," P'ship"," P/ship"'))

            Technical Business Rule Format:
            (StartsWith(^TRTAMI197, 'T/A ')) AND (EndsWithSet(^TRTAMI197, '" Pship"," P'ship"," P/ship"'))
    
            Data Elements:
    
            ^TRTAMI197 = TRTAMI:RP:TrusteeDetails:DirectorDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = (IsMatch(report.TRTAMI197, @"^(T/A )", RegexOptions.IgnoreCase) && IsMatch(report.TRTAMI197, @"( Pship| P'ship| P/ship)$", RegexOptions.IgnoreCase));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000408",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Corporate Director Name cannot start with ""T/A"" and end with ""Pship"", ""P'ship"" or ""P/Ship""",
                    Location = "/tns:TRTAMI/tns:RP/tns:TrusteeDetails/tns:DirectorDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000408"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI197", Value = report.TRTAMI197 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000409
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000409()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000409
            Corporate Director Name cannot end with "T/A", "T/A P'ship", "T/A Pship", "T/A P/Ship" or "T/A Partnership"
    
            Legacy Rule Format:
            (EndsWithSet(^TRTAMI197, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"'))

            Technical Business Rule Format:
            (EndsWithSet(^TRTAMI197, '" T/A"," T/A P'ship"," T/A Pship"," T/A P/Ship"," T/A Partnership"'))
    
            Data Elements:
    
            ^TRTAMI197 = TRTAMI:RP:TrusteeDetails:DirectorDetails:OrganisationNameDetails.OrganisationalName.Text
            */
            assertion = IsMatch(report.TRTAMI197, @"( T/A| T/A P'ship| T/A Pship| T/A P/Ship| T/A Partnership)$", RegexOptions.IgnoreCase);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000409",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Corporate Director Name cannot end with ""T/A"", ""T/A P'ship"", ""T/A Pship"", ""T/A P/Ship"" or ""T/A Partnership""",
                    Location = "/tns:TRTAMI/tns:RP/tns:TrusteeDetails/tns:DirectorDetails/tns:OrganisationNameDetailsOrganisationalNameT",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000409"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI197", Value = report.TRTAMI197 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000410
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000410()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000410
            Corporate Director Australian Business Number has failed the ABN algorithm check.
    
            Legacy Rule Format:
            (^TRTAMI198 <> NULL) AND (FailsABNAlgorithm(^TRTAMI198))

            Technical Business Rule Format:
            (^TRTAMI198 <> NULL) AND (FailsABNAlgorithm(^TRTAMI198))
    
            Data Elements:
    
            ^TRTAMI198 = TRTAMI:RP:TrusteeDetails:DirectorDetails:Identifiers.AustralianBusinessNumber.Identifier
            */
            assertion = (report.TRTAMI198 != null && FailsABNAlgorithm(report.TRTAMI198));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000410",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Corporate Director Australian Business Number is invalid",
                    LongDescription = @"Corporate Director Australian Business Number has failed the ABN algorithm check.",
                    Location = "/tns:TRTAMI/tns:RP/tns:TrusteeDetails/tns:DirectorDetails/tns:AustralianBusinessNumberId",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000410"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI198", Value = report.TRTAMI198 });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000416
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000416()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000416
            Have you determined that the CCIV sub-fund meets the AMIT eligibility requirements for the period covered by this income tax return? must be true
    
            Legacy Rule Format:
            ^TRTAMI201 <> NULL AND ^TRTAMI201 <> TRUE

            Technical Business Rule Format:
            ^TRTAMI201 <> NULL AND ^TRTAMI201 <> TRUE
    
            Data Elements:
    
            ^TRTAMI201 = TRTAMI:RP:AMITEligibility:IncomeTax.AttributionManagedInvestmentEligibility.Indicator
            */
            assertion = (report.TRTAMI201 != null && report.TRTAMI201 != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000416",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Have you determined that the CCIV sub-fund meets the AMIT eligibility requirements for the period covered by this income tax return? Must be true",
                    LongDescription = @"Where a CCIV sub-fund does not meet the AMIT eligibility requirements, an Attribution Corporate Collective Investment Vehicle (CCIV) sub-fund tax return should not be lodged. Please refer to the CCIV tax return instructions for further information.",
                    Location = "/tns:TRTAMI/tns:RP/tns:AMITEligibility/tns:IncomeTaxAttributionManagedInvestmentEligibilityI",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000416"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI201", Value = GetValueOrEmpty(report.TRTAMI201) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000419
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000419()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000419
            Business Industry code is not a valid code
    
            Legacy Rule Format:
            ^TRTAMI45 <> NULL AND NotInDomain(^TRTAMI45, 'BUSINESS INDUSTRY CODES')

            Technical Business Rule Format:
            ^TRTAMI45 <> NULL AND NotInDomain(^TRTAMI45, 'BUSINESS INDUSTRY CODES')
    
            Data Elements:
    
            ^TRTAMI45 = TRTAMI:RP:TrustDetails:OrganisationDetails.OrganisationIndustry2006Extended.Code
            */
            assertion = (report.TRTAMI45 != null && !(IsMatch(report.TRTAMI45, @"^(01110|01120|01130|01140|01150|01210|01220|01230|01310|01320|01330|01340|01350|01360|01370|01390|01410|01420|01430|01440|01450|01460|01490|01510|01520|01590|01600|01710|01720|01800|01910|01920|01930|01990|02011|02019|02020|02031|02039|03010|03020|04111|04112|04120|04130|04140|04191|04192|04199|04200|05100|05210|05220|05290|06000|07000|08010|08020|08030|08040|08050|08060|08070|08090|09110|09190|09901|09902|09903|09909|10111|10112|10121|10122|10900|11110|11120|11130|11200|11310|11320|11330|11400|11500|11610|11620|11710|11720|11730|11740|11810|11820|11910|11920|11990|12110|12120|12130|12140|12200|13110|13120|13130|13200|13310|13320|13330|13340|13400|13510|13520|14110|14120|14130|14910|14920|14930|14940|14990|15100|15210|15220|15230|15240|15290|16110|16120|16200|17010|17090|18110|18120|18130|18210|18290|18310|18320|18410|18420|18510|18520|18910|18920|18990|19110|19120|19130|19140|19150|19160|19190|19200|20100|20210|20290|20310|20320|20330|20340|20900|21100|21210|21220|21310|21320|21330|21390|21410|21420|21490|22100|22210|22220|22230|22240|22290|22310|22390|22400|22910|22920|22930|22990|23110|23120|23130|23190|23910|23920|23930|23940|23990|24110|24120|24190|24211|24219|24220|24290|24310|24320|24390|24410|24490|24510|24520|24610|24620|24630|24690|24910|24990|25110|25120|25130|25190|25910|25920|25990|26110|26120|26190|26200|26300|26400|27000|28110|28120|29110|29190|29210|29220|30110|30190|30200|31010|31091|31099|32110|32120|32210|32220|32230|32240|32310|32320|32330|32340|32390|32410|32420|32430|32440|32450|32910|32920|32990|33110|33120|33190|33210|33221|33229|33230|33310|33320|33390|34110|34190|34910|34920|34930|34940|34990|35010|35020|35030|35040|35050|36010|36020|36030|36040|36050|36061|36062|36090|37110|37120|37200|37310|37320|37330|37340|37350|37360|37390|38000|39110|39120|39130|39211|39219|39220|40000|41100|41211|41212|41213|41220|41230|41290|42110|42120|42130|42140|42210|42220|42290|42310|42320|42410|42420|42430|42441|42442|42450|42510|42520|42530|42590|42600|42711|42712|42720|42731|42732|42733|42740|42791|42792|42793|42794|42799|43101|43102|43109|43201|43202|43209|44000|45110|45120|45130|45200|45301|45302|46100|46210|46220|46231|46239|47100|47200|48100|48200|49001|49009|50100|50210|50290|51010|51020|52110|52120|52190|52200|52910|52920|52991|52999|53010|53090|54110|54120|54130|54140|54190|54200|55110|55120|55130|55140|55210|55220|56100|56210|56220|57000|58010|58020|58090|59100|59210|59220|60100|60200|62100|62210|62220|62230|62290|62300|62400|63100|63210|63220|63300|64110|64190|64200|66110|66190|66200|66310|66320|66390|66400|67110|67120|67200|69100|69210|69220|69230|69240|69250|69310|69320|69400|69500|69610|69621|69622|69629|69700|69910|69990|70000|72110|72120|72200|72910|72920|72930|72940|72990|73110|73120|73130|73200|75100|75200|75300|75400|75510|75520|76000|77110|77120|77130|77140|77190|77200|80100|80210|80220|80230|80240|81011|81019|81020|82110|82120|82191|82199|82200|84010|84020|85110|85121|85122|85123|85124|85125|85129|85201|85202|85203|85209|85310|85320|85330|85340|85391|85399|85910|85990|86011|86012|86090|87100|87900|89100|89210|89220|90010|90021|90029|90030|91110|91121|91122|91123|91124|91125|91126|91129|91131|91139|91140|91210|91291|91292|91299|91310|91390|92010|92020|92091|92092|92099|94110|94121|94129|94191|94192|94193|94199|94210|94220|94290|94910|94990|95110|95120|95200|95310|95320|95330|95340|95391|95392|95393|95399|95400|95510|95520|95591|95599|96010|96020|96030|98000|99010|99020|99030|99040|99050|99060|99070)$", RegexOptions.IgnoreCase)));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000419",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Business Industry code is not a valid code",
                    Location = "/tns:TRTAMI/tns:RP/tns:TrustDetails/tns:OrganisationDetailsOrganisationIndustry2006ExtendedC",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000419"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI45", Value = GetValueOrEmpty(report.TRTAMI45) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.000421
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMI000421()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.000421
            You must enter a valid amount of $0 or more in the 'Build to Rent Capital Works Deductions at 4%' when Capital Works Deductions is $250,000 or more or Business Entity Industry Code indicates a residential property-related industry
    
            Legacy Rule Format:
            ^TRTAMI210 = NULL AND (^TRTAMI140 >= 250000 OR InSet(^TRTAMI45, '"67110","30190"'))

            Technical Business Rule Format:
            ^TRTAMI210 = NULL AND (^TRTAMI140 >= 250000 OR InSet(^TRTAMI45, '"67110","30190"'))
    
            Data Elements:
    
            ^TRTAMI210 = TRTAMI:RP:CapitalAllowances:Expense.BuildToRentCapitalWorksDeduction.Amount
    
            ^TRTAMI45 = TRTAMI:RP:TrustDetails:OrganisationDetails.OrganisationIndustry2006Extended.Code
    
            ^TRTAMI140 = TRTAMI:RP:CapitalAllowances:Expense.CapitalWorksDeduction.Amount
            */
            assertion = (report.TRTAMI210 == null && (report.TRTAMI140.GetValueOrDefault() >= 250000 || IsMatch(report.TRTAMI45, @"^(67110|30190)$")));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.000421",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Build to rent capital works deductions at 4% amount must be provided",
                    LongDescription = @"You must enter a valid amount of $0 or more in the 'Build to Rent Capital Works Deduction at 4%' when Capital Works Deductions is $250,000 or more or Business Entity Industry Code indicates a residential property-related industry",
                    Location = "/tns:TRTAMI/tns:RP/tns:CapitalAllowances/tns:ExpenseBuildToRentCapitalWorksDeductionA",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.000421"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI210", Value = GetValueOrEmpty(report.TRTAMI210) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI140", Value = GetValueOrEmpty(report.TRTAMI140) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI45", Value = GetValueOrEmpty(report.TRTAMI45) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.W00001
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMIW00001()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.W00001
            If the trust has self-assessed that they are a Significant Global Entity (SGE) then please confirm if they are a Country by Country Reporting Entity (CbCRE).
    
            Legacy Rule Format:
            ^TRTAMI163 = NULL AND ^TRTAMI162 = TRUE

            Technical Business Rule Format:
            ^TRTAMI163 = NULL AND ^TRTAMI162 = TRUE
    
            Data Elements:
    
            ^TRTAMI163 = TRTAMI:RP:TrustDetails:InternationalDealings.CountryByCountryReportingEntityStatus.Indicator
    
            ^TRTAMI162 = TRTAMI:RP:TrustDetails:OrganisationDetails.SignificantGlobalEntityStatus.Indicator
            */
            assertion = (report.TRTAMI163 == null && report.TRTAMI162 == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.W00001",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Country by Country Reporting Entity (CbCRE) indicator not supplied.",
                    LongDescription = @"If the trust has self-assessed that they are a Significant Global Entity (SGE) then please confirm if they are a Country by Country Reporting Entity (CbCRE).",
                    Location = "/tns:TRTAMI/tns:RP/tns:TrustDetails/tns:InternationalDealingsCountryByCountryReportingEntityStatusI",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.W00001"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI163", Value = GetValueOrEmpty(report.TRTAMI163) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI162", Value = GetValueOrEmpty(report.TRTAMI162) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.W00003
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMIW00003()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.W00003
            If 'P' is selected at 'Aggregated turnover range', then the aggregated turnover amount should be $1 billion or over.
    
            Legacy Rule Format:
            ^TRTAMI160 = 'P' AND ^TRTAMI161 < 1000000000

            Technical Business Rule Format:
            ^TRTAMI160 = 'P' AND ^TRTAMI161 < 1000000000
    
            Data Elements:
    
            ^TRTAMI160 = TRTAMI:RP:KeyFinancialInformation:Income.AggregatedTurnoverRange.Code
    
            ^TRTAMI161 = TRTAMI:RP:KeyFinancialInformation:Income.AggregatedTurnover.Amount
            */
            assertion = (report.TRTAMI160 == @"P" && report.TRTAMI161.GetValueOrDefault() < 1000000000);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.W00003",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Aggregated turnover amount is incorrect.",
                    LongDescription = @"If 'P' is selected at 'Aggregated turnover range', then the aggregated turnover amount should be $1 billion or over.",
                    Location = "/tns:TRTAMI/tns:RP/tns:KeyFinancialInformation/tns:IncomeAggregatedTurnoverRangeC",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.W00003"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI160", Value = GetValueOrEmpty(report.TRTAMI160) });
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI161", Value = GetValueOrEmpty(report.TRTAMI161) });
                response.Add(processMessage);
            }
        }

#endregion
#region VR.ATO.TRTAMI.W00004
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public virtual void VRATOTRTAMIW00004()
        {
            ProcessMessageDocument processMessage;
            bool assertion;
            /*  VR.ATO.TRTAMI.W00004
            Consider if you are a Significant global entity (SGE). An entity is an SGE for a period if it is any of the following:
            - a global parent entity (GPE) with an annual global income of A$1 billion or more
            - a member of a group of entities consolidated for accounting purposes as a single group and one of the other group members is a GPE with an annual global income of A$1 billion or more
            - a member of a notional listed company group and one of the other group members is a GPE with an annual global income of A$1 billion or more.
            
            If you are an SGE, you also need to consider whether you are a country-by-country (CBC) reporting entity. CBC reporting entities must complete the Country by country reporting entity label, and may have additional reporting obligations. 
            
            For more information on SGEs and CBC reporting, visit our website at ato.gov.au.
    
            Legacy Rule Format:
            ^TRTAMI162 = TRUE

            Technical Business Rule Format:
            ^TRTAMI162 = TRUE
    
            Data Elements:
    
            ^TRTAMI162 = TRTAMI:RP:TrustDetails:OrganisationDetails.SignificantGlobalEntityStatus.Indicator
            */
            assertion = (report.TRTAMI162 == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.TRTAMI.W00004",
                    Severity = ProcessMessageSeverity.Warning,
                    Description = @"Confirm whether taxpayer is a Significant Global Entity (SGE)",
                    LongDescription = @"Consider if you are a Significant global entity (SGE). An entity is an SGE for a period if it is any of the following:
- a global parent entity (GPE) with an annual global income of A$1 billion or more
- a member of a group of entities consolidated for accounting purposes as a single group and one of the other group members is a GPE with an annual global income of A$1 billion or more
- a member of a notional listed company group and one of the other group members is a GPE with an annual global income of A$1 billion or more.

If you are an SGE, you also need to consider whether you are a country-by-country (CBC) reporting entity. CBC reporting entities must complete the Country by country reporting entity label, and may have additional reporting obligations. 

For more information on SGEs and CBC reporting, visit our website at ato.gov.au.",
                    Location = "/tns:TRTAMI/tns:RP/tns:TrustDetails/tns:OrganisationDetailsSignificantGlobalEntityStatusI",
                    Parameters = new ProcessMessageParameters()
                    {
                        new ProcessMessageParameter()
                        {
                            Name = "RuleIdentifier",
                            Value = "VR.ATO.TRTAMI.W00004"
                        }
                    },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "TRTAMI162", Value = GetValueOrEmpty(report.TRTAMI162) });
                response.Add(processMessage);
            }
        }
#endregion
    }
}