﻿using Ato.EN.IntegrationServices.CodeGenerationPAYEVNTEMPRequest2020;
using DataContracts;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text.RegularExpressions;

namespace Ato.EN.IntegrationServices.CodeGenerationPAYEVNTRequest2020
{
    public abstract class PAYEVNT2020RequestValidatorBase
    {
        public abstract List<ProcessMessageDocument> ValidateReport(PAYEVNT2020 report, DateTime? CreatedAt = null);

        public abstract List<ProcessMessageDocument> ValidateCrossformReport(PAYEVNT2020 report, PAYEVNTEMP2020 childReport, List<PAYEVNTEMP2020> childDocuments, int childcount = 0);

        #region Functions
        public static IEnumerable<string> Union(IEnumerable<string> list1, IEnumerable<string> list2)
        {
            IEnumerable<string> response;

            if (list1 == null && list2 == null)
            {
                response = null;
            }
            else if (list1 == null)
            {
                response = list2.Distinct();
            }
            else if (list2 == null)
            {
                response = list1.Distinct();
            }
            else
            {
                response = list1.Union(list2);
            }

            return response;
        }

        public static bool IsMatch(int? field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(Convert.ToString(field.Value), expression, options);
        }

        public static bool IsMatch(string field, string expression, RegexOptions options = RegexOptions.None)
        {
            if (field == null)
                return false;
            else
                return Regex.IsMatch(field, expression, options);
        }

        // This is just context and tuple counts where they are integer values - easier that changing the parsing logic to just return the value
        public static int Count(int count)
        {
            return count;
        }

        public static int Count<T>(IEnumerable<T> values)
        {
            return values == null ? 0 : values.Where(f => f != null).Count();
        }


        public static int Count<T>(ICollection<T> values)
        {
            return values == null ? 0 : values.Where(f => f != null).Count();
        }


        public static bool exists(bool value)
        {
            return value;
        }

        public static string GetValueOrEmpty(bool? val)
        {
            return (val.HasValue) ? val.ToString().ToLower() : string.Empty;
        }

        public static string GetValueOrEmpty(DateTime? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        public static string GetValueOrEmpty(string val)
        {
            return !string.IsNullOrWhiteSpace(val) ? val : string.Empty;
        }

        public static string GetValueOrEmpty(decimal? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        public static string GetValueOrEmpty(int? val)
        {
            return (val.HasValue) ? val.ToString() : string.Empty;
        }

        /// <summary>
        /// Get string value between [first] a and [last] b.
        /// </summary>
        public static string Between(string value, string a, string b)
        {
            int posA = value.IndexOf(a);
            int posB = value.LastIndexOf(b);
            if (posA == -1)
            {
                return "";
            }
            if (posB == -1)
            {
                return "";
            }
            int adjustedPosA = posA + a.Length;
            if (adjustedPosA >= posB)
            {
                return "";
            }
            return value.Substring(adjustedPosA, posB - adjustedPosA);
        }

        /// <summary>
        /// Get string value after [first] a.
        /// </summary>
        public static string Before(string value, string a)
        {
            int posA = value.IndexOf(a);
            if (posA == -1)
            {
                return "";
            }
            return value.Substring(0, posA);
        }

        /// <summary>
        /// Get string value after [last] a.
        /// </summary>
        public static string After(string value, string a)
        {
            int posA = value.LastIndexOf(a);
            if (posA == -1)
            {
                return "";
            }
            int adjustedPosA = posA + a.Length;
            if (adjustedPosA >= value.Length)
            {
                return "";
            }
            return value.Substring(adjustedPosA);
        }

        public static int Length(object field)
        {
            if (field == null)
                return 0;
            else
                return field.ToString().Trim().Length;
        }

        public static bool NotSameValues(IEnumerable<object> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return NotSameValues(nodesArray);
        }

        public static bool NotSameValues(params object[] nodes)
        {
            if (nodes == null)
                return false;

            return ((from x in nodes select x).Distinct().Count() == nodes.Count());
        }

        public static bool HasDuplicateValues(IEnumerable<object> nodes)
        {
            if (nodes == null)
                return false;

            object[] nodesArray = nodes.Cast<object>().ToArray();
            return HasDuplicateValues(nodesArray);
        }

        public static bool HasDuplicateValues(params object[] nodes)
        {
            if (nodes == null)
                return false;

            return !((from x in nodes select x).Distinct().Count() == nodes.Count());
        }

        public static int DuplicateValueIndex(IEnumerable<object> values)
        {
            int response = 0;
            var hashset = new HashSet<object>();
            foreach (var value in values)
            {
                if (!hashset.Add(value))
                {
                    return response;
                }
                response++;
            }
            return response;
        }

        public static int DuplicateValueIndex<T>(IEnumerable<T?> values) where T : struct
        {
            int response = 0;
            var hashset = new HashSet<T?>();
            foreach (var value in values)
            {
                if (!hashset.Add(value))
                {
                    return response;
                }
                response++;
            }
            return response;
        }

        public static bool IsDate(object value)
        {
            DateTime dateValue;
            return (value != null && DateTime.TryParse(value.ToString(), out dateValue));
        }

        public static DateTime AsDate(string dateAsString)
        {
            DateTime response = DateTime.MinValue;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date;
            }

            return response;
        }
        //The logic in After function expects "---" for day and "--" for month. 
        //Since hyphen was missing the date always returned null
        public static DateTime? ConvertToDate(int day, int month, int year)
        {
            return ConvertToDate(day == 0 ? null : "---" + day.ToString(), month == 0 ? null : "--" + month.ToString(), year == 0 ? null : year.ToString());
        }

        public static DateTime? ConvertToDate(string day, string month, string year)
        {
            DateTime? response;
            DateTime result;

            if (year == null || month == null || day == null)
            {
                return null;
            }
            string dateAsString = year + "-" + After(month, "--") + "-" + After(day, "---");
            if (DateTime.TryParse(dateAsString, out result))
            {
                response = result;
            }
            else
            {
                response = null;
            }
            return response;
        }


        public static DateTime? ConvertToDate(string day, string month, int year)
        {
            DateTime? response;
            DateTime result;

            if (year == 0 || month == null || day == null)
            {
                return null;
            }
            string dateAsString = year.ToString() + "-" + After(month, "--") + "-" + After(day, "---");
            if (DateTime.TryParse(dateAsString, out result))
            {
                response = result;
            }
            else
            {
                response = null;
            }
            return response;
        }


        public static int Day(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Day;
            }

            return response;
        }


        public static int Day(DateTime? date)
        {
            if (date == null)
                return 0;
            else
                return date.Value.Day;

        }


        public static string Month(string dateAsString)
        {
            string response = null;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.ToString("MMMM");
            }
            else
            {
                return "NotAMonth";
            }

            return response;
        }

        public static string Month(DateTime? date)
        {
            if (date == null)
                return "NotAMonth";
            else
                return date.Value.ToString("MMMM");
        }


        public static int MonthAsInt(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Month;
            }

            return response;
        }

        public static int MonthAsInt(DateTime? date)
        {
            if (date == null)
            {
                return 0;
            }
            return date.Value.Month;
        }


        public static int Year(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
            {
                response = date.Year;
            }

            return response;
        }

        public static int Year(DateTime? date)
        {
            if (date == null)
                return 0;
            else
                return date.Value.Year;
        }


        public static int CurrentFinancialYear()
        {
            return DateToFinancialYear(DateTime.Now, 7);
        }

        public static int FinancialYear(string dateAsString)
        {
            return DateToFinancialYear(dateAsString, 7);
        }

        public static int FinancialYear(DateTime? date)
        {
            return DateToFinancialYear(date, 7);
        }

        public static int DateToFinancialYear(string dateAsString, int startingMonth)
        {
            int response = 0;
            DateTime date;
            if (DateTime.TryParse(dateAsString, out date))
            {
                response = DateToFinancialYear(date, startingMonth);
            }

            return response;
        }
        public static int DateToFinancialYear(DateTime? date, int startingMonth)
        {
            int response;
            if (date == null)
            {
                response = 0;
            }
            else
            {
                int year = date.Value.Year;
                int month = date.Value.Month;

                if (startingMonth > month)
                    response = year;
                else
                    response = year + 1;
            }
            return response;
        }


        public static int FBTYear(string dateAsString)
        {
            int response = 0;
            DateTime date;

            if (DateTime.TryParse(dateAsString, out date))
                response = FBTYear(date);
            return response;
        }

        public static int FBTYear(DateTime? date)
        {
            if (date == null)
            {
                return 0;
            }
            else
            {
                if (date.Value.Month > 3)
                    return date.Value.Year + 1;
                else
                    return date.Value.Year;
            }
        }


        public static bool IsNumeric(object value)
        {
            decimal numbervalue;
            return (value != null && decimal.TryParse(value.ToString(), out numbervalue));
        }

        public static bool NotMonetary(decimal? field, string sign, int digits, int decimals)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                string decimalExpression;
                int digitsToUse = digits - decimals;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                if (decimals > 0)
                    decimalExpression = @"(\.\d{1," + decimals + "})?$";
                else
                    decimalExpression = @"$";

                return !(Regex.IsMatch(field.Value.ToString("0.#########################"), signExpression + @"\d{1," + digitsToUse + "}" + decimalExpression));
            }
        }

        public static bool NotNumeric(decimal? field, string sign, int digits, int decimals)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;
                string decimalExpression;
                int digitsToUse = digits - decimals;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                if (decimals > 0)
                    decimalExpression = @"(\.\d{1," + decimals + "})?$";
                else
                    decimalExpression = @"$";

                return !(Regex.IsMatch(field.Value.ToString("0.#########################"), signExpression + @"\d{1," + digitsToUse + "}" + decimalExpression));
            }
        }

        public static bool NotNumeric(int? field, string sign, int digits, int decimals = 0)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                return !(Regex.IsMatch(field.Value.ToString(), signExpression + @"\d{1," + digits + "}$"));
            }
        }

        public static bool NotNumeric(long? field, string sign, int digits, int decimals = 0)
        {
            if (field == null)
            {
                return false;
            }
            else
            {
                string signExpression;

                if (sign == "U")
                    signExpression = "^";
                else
                    signExpression = "^-?";

                return !(Regex.IsMatch(field.Value.ToString(), signExpression + @"\d{1," + digits + "}$"));
            }
        }


        public static bool OutsideRange(decimal field, decimal expression, int range)
        {
            bool response;

            response = (field < (expression - range)) || (field > (expression + range));

            return response;
        }


        public static bool FailsUSIAlgorithm(string usi, string abn)
        {
            bool response;
            if (usi == null || abn == null)
            {
                response = false;
            }
            else
            {
                usi = usi.Trim();
                abn = abn.Trim();
                if (usi.Length < 13 || abn.Length < 11)
                {
                    response = false;
                }
                else
                {
                    int numeric;
                    if (usi.Substring(0, 11) == abn && int.TryParse(usi.Substring(11, 2), out numeric))
                        response = false;
                    else if (Regex.IsMatch(usi, @"^[a-zA-Z]{3}\d{4}[a-zA-Z]{2}"))
                        response = false;
                    else
                        response = true;
                }
            }
            return response;
        }


        public static bool FailsTANAlgorithm(string tan)
        {
            bool response;
            decimal decimalTan;

            if (tan == null)
                return false;

            tan = tan.Trim();

            if (!decimal.TryParse(tan, out decimalTan))
                return true;

            if (tan.Length != 8)
                return true;

            decimal tanSum =
                7 * int.Parse(tan.Substring(0, 1)) +
                9 * int.Parse(tan.Substring(1, 1)) +
                8 * int.Parse(tan.Substring(2, 1)) +
                4 * int.Parse(tan.Substring(3, 1)) +
                6 * int.Parse(tan.Substring(4, 1)) +
                3 * int.Parse(tan.Substring(5, 1)) +
                5 * int.Parse(tan.Substring(6, 1)) +
                1 * int.Parse(tan.Substring(7, 1));

            if ((tanSum % 11) == 0)
                response = false;
            else
                response = true;

            return response;
        }


        public static bool FailsABNAlgorithm(string abn)
        {
            bool response;
            decimal decimalAbn;

            if (abn == null)
                return false;

            abn = abn.Trim();

            if (!decimal.TryParse(abn, out decimalAbn))
                return true;

            if (abn.Length != 11)
                return true;

            decimal abnSum =
                10 * (int.Parse(abn.Substring(0, 1)) - 1) +
                1 * int.Parse(abn.Substring(1, 1)) +
                3 * int.Parse(abn.Substring(2, 1)) +
                5 * int.Parse(abn.Substring(3, 1)) +
                7 * int.Parse(abn.Substring(4, 1)) +
                9 * int.Parse(abn.Substring(5, 1)) +
                11 * int.Parse(abn.Substring(6, 1)) +
                13 * int.Parse(abn.Substring(7, 1)) +
                15 * int.Parse(abn.Substring(8, 1)) +
                17 * int.Parse(abn.Substring(9, 1)) +
                19 * int.Parse(abn.Substring(10, 1));

            if ((abnSum % 89) == 0)
                response = false;
            else
                response = true;

            return response;
        }
        public static bool FailsACNAlgorithm(string acn)
        {
            bool response;
            decimal decimalAbn;
            if (acn == null)
                return false;

            acn = acn.Trim();

            if (!decimal.TryParse(acn, out decimalAbn))
                return true;

            if (acn.Length != 9)
                return true;

            decimal abnSum =
                8 * int.Parse(acn.Substring(0, 1)) +
                7 * int.Parse(acn.Substring(1, 1)) +
                6 * int.Parse(acn.Substring(2, 1)) +
                5 * int.Parse(acn.Substring(3, 1)) +
                4 * int.Parse(acn.Substring(4, 1)) +
                3 * int.Parse(acn.Substring(5, 1)) +
                2 * int.Parse(acn.Substring(6, 1)) +
                1 * int.Parse(acn.Substring(7, 1));

            decimal checkDigit = int.Parse(acn.Substring(8, 1));
            decimal acnRemainder = abnSum % 10;

            if (((10 - acnRemainder) % 10) == checkDigit)
                response = false;
            else
                response = true;

            return response;
        }


        public static bool FailsTFNAlgorithm(string tfn)
        {
            bool response;
            decimal decimalTfn;

            if (tfn == null)
                return false;

            tfn = tfn.Trim();
            tfn = Regex.Replace(tfn, "^0+", "");

            if (!decimal.TryParse(tfn, out decimalTfn))
                return true;

            if (tfn.Length < 8)
                return true;


            decimal tfn1To7Sum =
                1 * int.Parse(tfn.Substring(0, 1)) +
                4 * int.Parse(tfn.Substring(1, 1)) +
                3 * int.Parse(tfn.Substring(2, 1)) +
                7 * int.Parse(tfn.Substring(3, 1)) +
                5 * int.Parse(tfn.Substring(4, 1)) +
                8 * int.Parse(tfn.Substring(5, 1)) +
                6 * int.Parse(tfn.Substring(6, 1));

            decimal tfn8 = 9 * int.Parse(tfn.Substring(7, 1));

            if (tfn.Length == 8)
            {
                decimal tFNLg8WSum9 = 10 * int.Parse(tfn.Substring(7, 1));
                decimal tFNLg8WSum = tfn1To7Sum + tFNLg8WSum9;

                if ((tFNLg8WSum % 11) == 0)
                    response = false;
                else
                    response = true;
            }
            else if (tfn.Length == 9)
            {
                decimal tfn9 = 10 * int.Parse(tfn.Substring(8, 1));
                decimal tFNLg9WSum = tfn1To7Sum + tfn8 + tfn9;

                if ((tFNLg9WSum % 11) == 0)
                    response = false;
                else
                    response = true;
            }
            else
            {
                response = true;
            }

            return response;
        }


        public static decimal ConditionalValue(bool expression, decimal? trueVal, decimal? falseVal)
        {
            return expression ? trueVal.GetValueOrDefault() : falseVal.GetValueOrDefault();
        }

        public static decimal AsNumeric(string value)
        {
            decimal numberValue;
            decimal.TryParse(value, out numberValue);
            return numberValue;
        }

        public static bool RegexMatch(int? field, string expression, string flags = "")
        {
            return IsMatch(field, expression, GetRegexOption(flags));
        }

        public static bool RegexMatch(string field, string expression, string flags = "")
        {
            return IsMatch(field, expression, GetRegexOption(flags));
        }

        public static RegexOptions GetRegexOption(string flags)
        {
            RegexOptions options = RegexOptions.None;

            char[] characters = flags.ToCharArray();

            foreach (char character in characters)
            {
                switch (character)
                {
                    case 'i':
                        options = options | RegexOptions.IgnoreCase;
                        break;
                    case 'm':
                        options = options | RegexOptions.Multiline;
                        break;
                    case 's':
                        options = options | RegexOptions.Singleline;
                        break;
                    case 'n':
                        options = options | RegexOptions.ExplicitCapture;
                        break;
                    case 'x':
                        options = options | RegexOptions.IgnorePatternWhitespace;
                        break;
                }
            }

            return options;
        }

        /// <summary>
        /// Returns an occurrence index as [occurrenceIndex] of occurrenceIndex > 0, otherwise the empty string
        /// </summary>
        /// <param name="occurrenceIndex">Index of the occurrence.</param>
        /// <returns>Occurrence in XPath [#] format</returns>
        public static string OccurrenceIndex(int occurrenceIndex)
        {
            return occurrenceIndex > 0 ? "[" + occurrenceIndex + "]" : "";
        }
        #endregion

        #region Shared Rules

        #region VR.ATO.PAYEVNT.000015

        /*  VR.ATO.PAYEVNT.000015
        The Australian Business Number (ABN) has failed the ABN algorithm check.

        Legacy Rule Format:
        FailsABNAlgorithm(^PAYEVNT2)

        Technical Business Rule Format:
        FailsABNAlgorithm(^PAYEVNT2)

        Data Elements:

        ^PAYEVNT2 = PAYEVNT:Rp:Identifiers.AustralianBusinessNumber.Identifier
        */
        public static void VRATOPAYEVNT000015(PAYEVNT2020 report, List<ProcessMessageDocument> response)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = FailsABNAlgorithm(report.PAYEVNT2);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.434223",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"ABN is invalid.",
                    Location = "/tns:PAYEVNT/tns:Rp/tns:AustralianBusinessNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNT.000015" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PAYEVNT2", Value = GetValueOrEmpty(report.PAYEVNT2) });

                response.Add(processMessage);
            }
        }

        #endregion

        #region VR.ATO.PAYEVNT.000110

        /*  VR.ATO.PAYEVNT.000110
        If provided, must be a valid format.

        Legacy Rule Format:
        (^PAYEVNT15 <> NULLORBLANK AND NOT IsValidEmail(^PAYEVNT15))

        Technical Business Rule Format:
        (^PAYEVNT15 <> BLANK AND NOT IsValidEmail(^PAYEVNT15))

        Data Elements:

        ^PAYEVNT15 = PAYEVNT:Rp:ElectronicContact:ElectronicContact.ElectronicMail.Address.Text
        */
        public static void VRATOPAYEVNT000110(PAYEVNT2020 report, List<ProcessMessageDocument> response)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (string.IsNullOrWhiteSpace(report.PAYEVNT15) != true && !(IsMatch(report.PAYEVNT15, @"^\S.*@.+\.\S+$")));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Email Address contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - _ = [ ] ; : ' "" , . ? / or a space character.",
                    Location = "/tns:PAYEVNT/tns:Rp/tns:ElectronicContact/tns:ElectronicMailAddressT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNT.000110" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PAYEVNT15", Value = GetValueOrEmpty(report.PAYEVNT15) });

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.PAYEVNT.000110

        #region VR.ATO.PAYEVNT.000170

        /*  VR.ATO.PAYEVNT.000170
        The Payer Declaration Date cannot be later than today.

        Legacy Rule Format:
        ^PAYEVNT38 > Today()

        Technical Business Rule Format:
        ^PAYEVNT38 > Today()

        Data Elements:

        ^PAYEVNT38 = PAYEVNT:Rp:Declaration:Declaration.Signature.Date
        */
        public static void VRATOPAYEVNT000170(PAYEVNT2020 report, List<ProcessMessageDocument> response)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (report.PAYEVNT38.GetValueOrDefault() > DateTime.Now.Date);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PAYEVNT.000193",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The date entered for Payer Declaration Date must not be later than today's date",
                    Location = "/tns:PAYEVNT/tns:Rp/tns:Declaration/tns:SignatureD",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNT.000170" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PAYEVNT38", Value = GetValueOrEmpty(report.PAYEVNT38) });

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.PAYEVNT.000170
        
        #region VR.ATO.PAYEVNT.000172

        /*  VR.ATO.PAYEVNT.000172
        The Australian Business Number (ABN) has failed the ABN algorithm check.

        Legacy Rule Format:
        FailsABNAlgorithm(^PAYEVNT64)

        Technical Business Rule Format:
        FailsABNAlgorithm(^PAYEVNT64)

        Data Elements:

        ^PAYEVNT64 = PAYEVNT:Int:Identifiers.AustralianBusinessNumber.Identifier
        */
        public static void VRATOPAYEVNT000172(PAYEVNT2020 report, List<ProcessMessageDocument> response)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = FailsABNAlgorithm(report.PAYEVNT64);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PAYEVNT.000206",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Intermediary ABN is invalid",
                    Location = "/tns:PAYEVNT/tns:Int/tns:AustralianBusinessNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNT.000172" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PAYEVNT64", Value = report.PAYEVNT64 });

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.PAYEVNT.000172

        #region VR.ATO.PAYEVNT.000173

        /*  VR.ATO.PAYEVNT.000173
        If provided, must be a valid format.

        Legacy Rule Format:
        (^PAYEVNT66 <> NULLORBLANK AND NOT IsValidEmail(^PAYEVNT66))

        Technical Business Rule Format:
        (^PAYEVNT66 <> BLANK AND NOT IsValidEmail(^PAYEVNT66))

        Data Elements:

        ^PAYEVNT66 = PAYEVNT:Int:ElectronicContact:ElectronicContact.ElectronicMail.Address.Text
        */
        public static void VRATOPAYEVNT000173(PAYEVNT2020 report, List<ProcessMessageDocument> response)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (string.IsNullOrWhiteSpace(report.PAYEVNT66) != true && !(IsMatch(report.PAYEVNT66, @"^\S.*@.+\.\S+$")));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.500029",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Email Address contains invalid text",
                    LongDescription = @"A text character must be one of the following: A to Z a to z 0 to 9 ! @ $ % & * ( ) - _ = [ ] ; : ' "" , . ? / or a space character.",
                    Location = "/tns:PAYEVNT/tns:Int/tns:ElectronicContact/tns:ElectronicMailAddressT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNT.000173" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PAYEVNT66", Value = GetValueOrEmpty(report.PAYEVNT66) });

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.PAYEVNT.000173

        #region VR.ATO.PAYEVNT.000177

        /*  VR.ATO.PAYEVNT.000177
        Either the Payer Australian Business Number or the Payer Withholding Payer Number must be provided, but not both.

        Legacy Rule Format:
        ((^PAYEVNT2 = NULL AND ^PAYEVNT3 = NULL) OR (^PAYEVNT2 <> NULL AND ^PAYEVNT3 <> NULL))

        Technical Business Rule Format:
        ((^PAYEVNT2 = NULL AND ^PAYEVNT3 = NULL) OR (^PAYEVNT2 <> NULL AND ^PAYEVNT3 <> NULL))

        Data Elements:

        ^PAYEVNT2 = PAYEVNT:Rp:Identifiers.AustralianBusinessNumber.Identifier

        ^PAYEVNT3 = PAYEVNT:Rp:Identifiers.WithholdingPayerNumber.Identifier
        */
        public static void VRATOPAYEVNT000177(PAYEVNT2020 report, List<ProcessMessageDocument> response)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (report.PAYEVNT2 == null && report.PAYEVNT3 == null || report.PAYEVNT2 != null && report.PAYEVNT3 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200010",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Either ABN or WPN must be present",
                    Location = "/tns:PAYEVNT/tns:Rp/tns:AustralianBusinessNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNT.000177" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PAYEVNT2", Value = GetValueOrEmpty(report.PAYEVNT2) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PAYEVNT3", Value = GetValueOrEmpty(report.PAYEVNT3) });

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.PAYEVNT.000177
        
        #region VR.ATO.PAYEVNT.000179

        /*  VR.ATO.PAYEVNT.000179
        The Postcode must be in the range of '0200-9999'.

        Legacy Rule Format:
        (^PAYEVNT12 <> NULL  AND (AsNumeric(^PAYEVNT12) < 200 OR AsNumeric(^PAYEVNT12) > 9999))

        Technical Business Rule Format:
        (^PAYEVNT12 <> NULL  AND (AsNumeric(^PAYEVNT12) < 200 OR AsNumeric(^PAYEVNT12) > 9999))

        Data Elements:

        ^PAYEVNT12 = PAYEVNT:Rp:AddressDetailsPostal:AddressDetails.Postcode.Text
        */
        public static void VRATOPAYEVNT000179(PAYEVNT2020 report, List<ProcessMessageDocument> response)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (report.PAYEVNT12 != null && (AsNumeric(report.PAYEVNT12) < 200 || AsNumeric(report.PAYEVNT12) > 9999));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PAYEVNT.000179",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"A valid postcode must be provided",
                    LongDescription = @"The Postcode must be in the range of '0200-9999'",
                    Location = "/tns:PAYEVNT/tns:Rp/tns:AddressDetailsPostal/tns:PostcodeT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNT.000179" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PAYEVNT12", Value = GetValueOrEmpty(report.PAYEVNT12) });

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.PAYEVNT.000179

        #region VR.ATO.PAYEVNT.000180

        /*  VR.ATO.PAYEVNT.000180
        When the address is overseas, the Postcode must be null.

        Legacy Rule Format:
        ((^PAYEVNT14 <> NULL AND ^PAYEVNT14 <> "au") AND (^PAYEVNT12 <> NULL))

        Technical Business Rule Format:
        ((^PAYEVNT14 <> NULL AND ^PAYEVNT14 <> 'au') AND (^PAYEVNT12 <> NULL))

        Data Elements:

        ^PAYEVNT12 = PAYEVNT:Rp:AddressDetailsPostal:AddressDetails.Postcode.Text

        ^PAYEVNT14 = PAYEVNT:Rp:AddressDetailsPostal:AddressDetails.Country.Code
        */
        public static void VRATOPAYEVNT000180(PAYEVNT2020 report, List<ProcessMessageDocument> response)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (report.PAYEVNT14 != null && report.PAYEVNT14 != @"au" && report.PAYEVNT12 != null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000480",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Country code must be blank or set to Australia",
                    Location = "/tns:PAYEVNT/tns:Rp/tns:AddressDetailsPostal/tns:PostcodeT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNT.000180" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PAYEVNT14", Value = GetValueOrEmpty(report.PAYEVNT14) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PAYEVNT12", Value = GetValueOrEmpty(report.PAYEVNT12) });

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.PAYEVNT.000180

        #region VR.ATO.PAYEVNT.000183

        /*  VR.ATO.PAYEVNT.000183
        The Payer Withholding Payer Number(WPN) has failed the WPN algorithm check.

        Legacy Rule Format:
        FailsWPNAlgorithm(^PAYEVNT3)

        Technical Business Rule Format:
        FailsWPNAlgorithm(^PAYEVNT3)

        Data Elements:

        ^PAYEVNT3 = PAYEVNT:Rp:Identifiers.WithholdingPayerNumber.Identifier
        */
        public static void VRATOPAYEVNT000183(PAYEVNT2020 report, List<ProcessMessageDocument> response)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = FailsTFNAlgorithm(report.PAYEVNT3);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.200012",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Invalid WPN",
                    Location = "/tns:PAYEVNT/tns:Rp/tns:WithholdingPayerNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNT.000183" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PAYEVNT3", Value = GetValueOrEmpty(report.PAYEVNT3) });

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.PAYEVNT.000183

        #region VR.ATO.PAYEVNT.000184

        /*  VR.ATO.PAYEVNT.000184
        When address is within Australia, the Postcode must be supplied.

        Legacy Rule Format:
        ((^PAYEVNT14  = "au" OR ^PAYEVNT14 = NULL) AND ^PAYEVNT12 = NULL)

        Technical Business Rule Format:
        ((^PAYEVNT14  = 'au' OR ^PAYEVNT14 = NULL) AND ^PAYEVNT12 = NULL)

        Data Elements:

        ^PAYEVNT12 = PAYEVNT:Rp:AddressDetailsPostal:AddressDetails.Postcode.Text

        ^PAYEVNT14 = PAYEVNT:Rp:AddressDetailsPostal:AddressDetails.Country.Code
        */
        public static void VRATOPAYEVNT000184(PAYEVNT2020 report, List<ProcessMessageDocument> response)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = ((report.PAYEVNT14 == @"au" || report.PAYEVNT14 == null) && report.PAYEVNT12 == null);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.GEN.000009",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Postcode is required for an Australian address",
                    Location = "/tns:PAYEVNT/tns:Rp/tns:AddressDetailsPostal/tns:PostcodeT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNT.000184" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PAYEVNT14", Value = GetValueOrEmpty(report.PAYEVNT14) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PAYEVNT12", Value = GetValueOrEmpty(report.PAYEVNT12) });

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.PAYEVNT.000184

        #region VR.ATO.PAYEVNT.000185

        /*  VR.ATO.PAYEVNT.000185
        Payer Declaration Acceptance Indicator must be 'TRUE'.

        Legacy Rule Format:
        (^PAYEVNT39 <> TRUE)

        Technical Business Rule Format:
        ^PAYEVNT39 <> TRUE

        Data Elements:

        ^PAYEVNT39 = PAYEVNT:Rp:Declaration:Declaration.StatementAccepted.Indicator
        */
        public static void VRATOPAYEVNT000185(PAYEVNT2020 report, List<ProcessMessageDocument> response)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (report.PAYEVNT39 != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PAYEVNT.000194",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Payer Declaration Acceptance Indicator must be 'TRUE'",
                    Location = "/tns:PAYEVNT/tns:Rp/tns:Declaration/tns:StatementAcceptedI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNT.000185" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PAYEVNT39", Value = GetValueOrEmpty(report.PAYEVNT39) });

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.PAYEVNT.000185
		
		#region VR.ATO.PAYEVNT.000195

        /*  VR.ATO.PAYEVNT.000195
        Full File Replacement Indicator must be set to false.

        Legacy Rule Format:
        ^PAYEVNT19 = TRUE

        Technical Business Rule Format:
        ^PAYEVNT19 = TRUE

        Data Elements:

        ^PAYEVNT19 = PAYEVNT:Rp:Payroll:Report.Amendment.Indicator
        */
        public static void VRATOPAYEVNT000195(PAYEVNT2020 report, List<ProcessMessageDocument> response)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (report.PAYEVNT19 == true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PAYEVNT.000201",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The Full File Replacement Indicator must be set to false",
                    Location = "/tns:PAYEVNT/tns:Rp/tns:Payroll/tns:AmendmentI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNT.000195" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PAYEVNT19", Value = GetValueOrEmpty(report.PAYEVNT19) });

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.PAYEVNT.000195

        #region VR.ATO.PAYEVNT.000199

        /*  VR.ATO.PAYEVNT.000199
        Payer Period Totals must be provided.

        Legacy Rule Format:
        Count(^PAYEVNT79) = 0

        Technical Business Rule Format:
        Count(^PAYEVNT79) = 0

        Data Elements:

        ^PAYEVNT79 = PAYEVNT:Rp:Payroll:IncomeTaxAndRemuneration
        */
        public static void VRATOPAYEVNT000199(PAYEVNT2020 report, List<ProcessMessageDocument> response)
        {
            bool assertion = (Count(report.Rp_Payroll_IncomeTaxAndRemunerationCollectionCount) == 0);
            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PAYEVNT.000203",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The Payer Period Totals tuple must be provided",
                    Location = "/tns:PAYEVNT/tns:Rp/tns:Payroll/tns:IncomeTaxAndRemuneration",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNT.000199" } },
                };

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.PAYEVNT.000199

        #region VR.ATO.PAYEVNT.000200

        /*  VR.ATO.PAYEVNT.000200
        The Pay/Update Date must be on/after the 1st July 2017.

        Legacy Rule Format:
        (^PAYEVNT69 < ConvertToDate(1, 7, 2017))

        Technical Business Rule Format:
        (^PAYEVNT69 < ConvertToDate(1, 7, 2017))

        Data Elements:

        ^PAYEVNT69 = PAYEVNT:Rp:Payroll:PaymentRecord.Transaction.Date
        */
        public static void VRATOPAYEVNT000200(PAYEVNT2020 report, List<ProcessMessageDocument> response)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (report.PAYEVNT69.GetValueOrDefault() < ConvertToDate(1, 7, 2017));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PAYEVNT.000205",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The Pay/Update Date must be on/after the 1st July 2017",
                    Location = "/tns:PAYEVNT/tns:Rp/tns:Payroll/tns:PaymentRecordTransactionD",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNT.000200" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PAYEVNT69", Value = GetValueOrEmpty(report.PAYEVNT69) });

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.PAYEVNT.000200

        #region VR.ATO.PAYEVNT.000201

        /*  VR.ATO.PAYEVNT.000201
        The Intermediary Declaration Date cannot be later than today.

        Legacy Rule Format:
        ^PAYEVNT42 > Today()

        Technical Business Rule Format:
        ^PAYEVNT42 > Today()

        Data Elements:

        ^PAYEVNT42 = PAYEVNT:Int:Declaration:Declaration.Signature.Date
        */
        public static void VRATOPAYEVNT000201(PAYEVNT2020 report, List<ProcessMessageDocument> response)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (report.PAYEVNT42.GetValueOrDefault() > DateTime.Now.Date);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PAYEVNT.000207",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The date entered for Intermediary Declaration Date must not be later than today's date",
                    Location = "/tns:PAYEVNT/tns:Int/tns:Declaration/tns:SignatureD",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNT.000201" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PAYEVNT42", Value = GetValueOrEmpty(report.PAYEVNT42) });

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.PAYEVNT.000201

        #region VR.ATO.PAYEVNT.000202

        /*  VR.ATO.PAYEVNT.000202
        Intermediary Declaration Acceptance Indicator must be 'TRUE'.

        Legacy Rule Format:
        (Count(^PAYEVNT81) > 0 AND (^PAYEVNT43 <> TRUE))

        Technical Business Rule Format:
        (Count(^PAYEVNT81) > 0 AND (^PAYEVNT43 <> TRUE))

        Data Elements:

        ^PAYEVNT43 = PAYEVNT:Int:Declaration:Declaration.StatementAccepted.Indicator

        ^PAYEVNT81 = PAYEVNT:Int
        */
        public static void VRATOPAYEVNT000202(PAYEVNT2020 report, List<ProcessMessageDocument> response)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (Count(report.IntCollectionCount) > 0 && report.PAYEVNT43 != true);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PAYEVNT.000208",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Intermediary Declaration Acceptance Indicator must be 'TRUE'",
                    Location = "/tns:PAYEVNT/tns:Int/tns:Declaration/tns:StatementAcceptedI",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNT.000202" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PAYEVNT43", Value = GetValueOrEmpty(report.PAYEVNT43) });

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.PAYEVNT.000202

        #region VR.ATO.PAYEVNT.000203

        /*  VR.ATO.PAYEVNT.000203
        The Registered Agent Number has failed the TAN algorithm check.

        Legacy Rule Format:
        (^PAYEVNT65 <> NULL AND FailsTANAlgorithm(^PAYEVNT65))

        Technical Business Rule Format:
        (^PAYEVNT65 <> NULL AND FailsTANAlgorithm(^PAYEVNT65))

        Data Elements:

        ^PAYEVNT65 = PAYEVNT:Int:Identifiers.TaxAgentNumber.Identifier
        */
        public static void VRATOPAYEVNT000203(PAYEVNT2020 report, List<ProcessMessageDocument> response)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (report.PAYEVNT65 != null && FailsTANAlgorithm(report.PAYEVNT65));
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PAYEVNT.000196",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Registered Agent Number is invalid",
                    Location = "/tns:PAYEVNT/tns:Int/tns:TaxAgentNumberId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNT.000203" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PAYEVNT65", Value = GetValueOrEmpty(report.PAYEVNT65) });

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.PAYEVNT.000203
		
		#region VR.ATO.PAYEVNT.000210

        /*  VR.ATO.PAYEVNT.000210
        Previous BMS Identifier must not be provided
    
        Legacy Rule Format:
        ^PAYEVNT101 <> NULL

        Technical Business Rule Format:
        ^PAYEVNT101 <> NULL
    
        Data Elements:
    
        ^PAYEVNT101 = PAYEVNT:Rp:Previous:SoftwareInformation.BusinessManagementSystem.Identifier
        */
        public static void VRATOPAYEVNT000210(PAYEVNT2020 report, List<ProcessMessageDocument> response)
        {
            bool assertion = report.PAYEVNT101 != null;
            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PAYEVNT.000210",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Previous BMS Identifier must not be provided",
                    Location = "/tns:PAYEVNT/tns:Rp/tns:PreviousSoftwareInformationBusinessManagementSystemId",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNT.000210" } },
                };
                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.PAYEVNT.000210

        #region VR.ATO.PAYEVNT.000194

        /*  VR.ATO.PAYEVNT.000194
        The Run Date/Time Stamp cannot be later than the time message received in the channel.

        Legacy Rule Format:
        (^PAYEVNT71 > AddHoursToDateTime(^Sent_TS timestamp in the Business Document object, 1))

        Technical Business Rule Format:
        (^PAYEVNT71 > AddHoursToDateTime(^Sent_TS timestamp in the Business Document object, 1))

        Data Elements:

        ^PAYEVNT71 = PAYEVNT:Rp:Payroll:Message.Timestamp.Generation.Datetime

        ^Sent_TS = Sent_TS
        */
        public static void VRATOPAYEVNT000194(PAYEVNT2020 report, List<ProcessMessageDocument> response, DateTime? CreatedAt)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (report.PAYEVNT71.Value.AddHours(-1).ToUniversalTime() > CreatedAt);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PAYEVNT.000200",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"The Run Date/Time Stamp cannot be later than the time message received in the channel",
                    Location = "/tns:PAYEVNT/tns:Rp/tns:Payroll/tns:MessageTimestampGenerationDt",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNT.000194" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PAYEVNT71", Value = (report.PAYEVNT71.HasValue) ? report.PAYEVNT71.Value.ToUniversalTime().ToString("u") : string.Empty });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "Sent_TS", Value = CreatedAt.Value.ToString("u") });

                response.Add(processMessage);
            }
        }
        #endregion // VR.ATO.PAYEVNT.000194

        #region VR.ATO.PAYEVNT.000192

        /*  VR.ATO.PAYEVNT.000192
        At least one PAYEVNTEMP child must be provided

        Legacy Rule Format:
        Count(^PAYEVNTEMP) = 0

        Technical Business Rule Format:
        Count(^PAYEVNTEMP) = 0
        */
        public static void VRATOPAYEVNT000192(List<ProcessMessageDocument> response, int childcount)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (childcount == 0);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PAYEVNT.000192",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"At least one PAYEVNTEMP child must be provided",
                    Location = null,
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNT.000192" } },
                };
                
                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "PAYEVNTEMP", Value = "PAYEVNTEMP" });

                response.Add(processMessage);
            }
        }
        #endregion VR.ATO.PAYEVNT.000192

        #region VR.ATO.PAYEVNT.000191

        /*  VR.ATO.PAYEVNT.000191
        Payee Record Count must be equal to the number of children (PAYEVNTEMP)

        Legacy Rule Format:
        (^PAYEVNT70 <> Count(^PAYEVNTEMP))

        Technical Business Rule Format:
        (^PAYEVNT70 <> Count(^PAYEVNTEMP))
        */
        public static void VRATOPAYEVNT000191(PAYEVNT2020 report, List<ProcessMessageDocument> response, int childcount)
        {
            ProcessMessageDocument processMessage = null;

            bool assertion = (report.PAYEVNT70.GetValueOrDefault() != childcount);
            if (assertion)
            {
                processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PAYEVNT.000214",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Payee Record Count must be equal to the number of children (PAYEVNTEMP)",
                    Location = "/tns:PAYEVNT/tns:Rp/tns:Payroll/tns:InteractionRecordCt",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNT.000191" } },
                };

                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "PAYEVNT70", Value = report.PAYEVNT70.ToString() });

                response.Add(processMessage);
            }
        }
        #endregion

        #region VR.ATO.PAYEVNT.000212

        /*  VR.ATO.PAYEVNT.000212
        Branch code must be provided when an ABN is provided

        Legacy Rule Format:
        ^PAYEVNT4 = NULLORBLANK AND ^PAYEVNT2 <> NULLORBLANK

        Technical Business Rule Format:
        ^PAYEVNT4 = BLANK AND ^PAYEVNT2 <> BLANK

        Data Elements:

        ^PAYEVNT4 = PAYEVNT:Rp:OrganisationDetails.OrganisationBranch.Code

        ^PAYEVNT2 = PAYEVNT:Rp:Identifiers.AustralianBusinessNumber.Identifier
        */

        public static void VRATOPAYEVNT000212(PAYEVNT2020 report, List<ProcessMessageDocument> response)
        {
            bool assertion = (string.IsNullOrWhiteSpace(report.PAYEVNT4) == true && string.IsNullOrWhiteSpace(report.PAYEVNT2) != true);
            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PAYEVNT.000212",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Branch code must be provided when an ABN is provided",
                    Location = "/tns:PAYEVNT/tns:Rp/tns:OrganisationDetailsOrganisationBranchC",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNT.000212" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PAYEVNT4", Value = GetValueOrEmpty(report.PAYEVNT4) });

                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PAYEVNT2", Value = GetValueOrEmpty(report.PAYEVNT2) });

                response.Add(processMessage);
            }
        }

        #endregion // VR.ATO.PAYEVNT.000212
		
	    #region VR.ATO.PAYEVNT.000219

        /*  VR.ATO.PAYEVNT.000219
        Payer Contact Name must have a length of at least 1 character.

        Legacy Rule Format:
        Length(^PAYEVNT6) < 1

        Technical Business Rule Format:
        Length(^PAYEVNT6) < 1

        Data Elements:

        ^PAYEVNT6 = PAYEVNT:Rp:OrganisationName:PersonUnstructuredName.FullName.Text
        */
        public static void VRATOPAYEVNT000219(PAYEVNT2020 report, List<ProcessMessageDocument> response)
        {

            bool assertion = (Length(report.PAYEVNT6) < 1);
            if (assertion)
            {
                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                {
                    Code = "CMN.ATO.PAYEVNT.000219",
                    Severity = ProcessMessageSeverity.Error,
                    Description = @"Payer Contact Name must be at least 1 character long",
                    LongDescription = @"Payer Contact Name must have a length of at least 1 character.",
                    Location = "/tns:PAYEVNT/tns:Rp/tns:OrganisationName/tns:PersonUnstructuredNameFullNameT",
                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNT.000219" } },
                };
                processMessage.Parameters.Add(new ProcessMessageParameter
                { Name = "PAYEVNT6", Value = GetValueOrEmpty(report.PAYEVNT6) });

                response.Add(processMessage);
            }
        }
        #endregion

        //Child Cross Form

        #region VR.ATO.PAYEVNTEMP.000180

        /*  VR.ATO.PAYEVNTEMP.000180
        Both ETP Payment Date as well as Pay/Update Date must be in the same financial year.

        Technical Business Rule Format:
        (FinancialYear(^PAYEVNTEMP123) <> FinancialYear(^PAYEVNT69))
        */
        public static void VRATOPAYEVNTEMP000180(PAYEVNT2020 report, List<ProcessMessageDocument> response, PAYEVNTEMP2020 childReport)
        {
            if (childReport != null)
            {
                if (childReport.Payee_PayrollPeriod_RemunerationCollection != null)
                {
                    foreach (PAYEVNTEMP2020.Payee_PayrollPeriod_Remuneration remuneration in childReport.Payee_PayrollPeriod_RemunerationCollection)
                    {
                        if (remuneration.Payee_PayrollPeriod_Remuneration_EmploymentTerminationPaymentCollection != null)
                        {
                            PAYEVNTEMP2020.Payee_PayrollPeriod_Remuneration.Payee_PayrollPeriod_Remuneration_EmploymentTerminationPayment employmentTerminationPayment = remuneration.Payee_PayrollPeriod_Remuneration_EmploymentTerminationPaymentCollection.Find(f => DateToFinancialYear(report.PAYEVNT69, 7) != DateToFinancialYear(f.PAYEVNTEMP123.Value, 7));

                            bool assertion = (employmentTerminationPayment != null);
                            if (assertion)
                            {
                                ProcessMessageDocument processMessage = new ProcessMessageDocument()
                                {
                                    Code = "CMN.ATO.PAYEVNTEMP.000180",
                                    Severity = ProcessMessageSeverity.Error,
                                    Description = @"ETP payment date must be in the same financial year as the Pay/Update Date",
                                    Location = "/tns:PAYEVNTEMP/tns:Payee/tns:PayrollPeriod/tns:RemunerationCollection/tns:Remuneration" + OccurrenceIndex(remuneration.OccurrenceIndex) + "/tns:EmploymentTerminationPaymentCollection/tns:EmploymentTerminationPayment" + OccurrenceIndex(employmentTerminationPayment.OccurrenceIndex) + "/tns:IncomeD",
                                    Parameters = new ProcessMessageParameters() { new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNTEMP.000180" } },
                                };

                                processMessage.Parameters.Add(new ProcessMessageParameter { Name = "PAYEVNTEMP123", Value = employmentTerminationPayment.PAYEVNTEMP123.ToString() });

                                response.Add(processMessage);

                                break;
                            }
                        }
                    }
                }
            }
        }

        #endregion

        #region VR.ATO.PAYEVNTEMP.000267

        /*  VR.ATO.PAYEVNTEMP.000267
        Super Entitlement Type Code must be one of the following values: L (Superannuation Liability), O (Ordinary Time Earnings), Q (Qualifying Earnings - only valid from 01/07/2026) or R (Reportable Employer Superannuation Contribution)

        Legacy Rule Format:
        RegexMatch(^PAYEVNTEMP283, '^[LOQR]$', 'i') = FALSE OR (RegexMatch(^PAYEVNTEMP283, 'Q', 'i') AND ^PAYEVNT69 < ConvertToDate(1, 7, 2026))

        Technical Business Rule Format:
        RegexMatch(^PAYEVNTEMP283, '^[LOQR]$', 'i') = FALSE OR (RegexMatch(^PAYEVNTEMP283, 'Q', 'i') AND ^PAYEVNT69 < ConvertToDate(1, 7, 2026))

        Data Elements:

        ^PAYEVNTEMP283 = PAYEVNTEMP:Payee:PayrollPeriod:SuperannuationContribution:SuperannuationContribution.EntitlementType.Code

        ^PAYEVNT69 = PAYEVNT:Rp:Payroll:PaymentRecord.Transaction.Date
        */
        public static void VRATOPAYEVNTEMP000267(PAYEVNT2020 report, List<ProcessMessageDocument> response, PAYEVNTEMP2020 childReport)
        {
            if (childReport != null)
            {
                if (childReport.Payee_PayrollPeriod_SuperannuationContributionCollection != null)
                {
                    foreach (PAYEVNTEMP2020.Payee_PayrollPeriod_SuperannuationContribution superannuationContribution in childReport.Payee_PayrollPeriod_SuperannuationContributionCollection)
                    {
                        bool assertion = (RegexMatch(superannuationContribution.PAYEVNTEMP283, @"^[LOQR]$", @"i") == false || RegexMatch(superannuationContribution.PAYEVNTEMP283, @"Q", @"i") && report.PAYEVNT69.GetValueOrDefault() < ConvertToDate(1, 7, 2026));
                        if (assertion)
                        {
                            ProcessMessageDocument processMessage = new ProcessMessageDocument()
                            {
                                Code = "CMN.ATO.PAYEVNTEMP.000267",
                                Severity = ProcessMessageSeverity.Error,
                                Description = @"Super Entitlement Type Code is Invalid",
                                LongDescription = @"Super Entitlement Type Code must be one of the following values: L (Superannuation Liability), O (Ordinary Time Earnings), Q (Qualifying Earnings - only valid from 01/07/2026) or R (Reportable Employer Superannuation Contribution)",
                                Location = "/tns:PAYEVNTEMP/tns:Payee/tns:PayrollPeriod/tns:SuperannuationContributionCollection/tns:SuperannuationContribution" + OccurrenceIndex(superannuationContribution.OccurrenceIndex) + "/tns:EntitlementTypeC",
                                Parameters = new ProcessMessageParameters() {
                                    new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNTEMP.000267" },
                                    new ProcessMessageParameter() { Name = "PAYEVNTEMP283", Value = superannuationContribution.PAYEVNTEMP283 },
                                    new ProcessMessageParameter() { Name = "PAYEVNT69", Value = GetValueOrEmpty(report.PAYEVNT69) }
                                },
                            };

                            response.Add(processMessage);
                        }
                    }
                }
            }
        }

        #endregion // VR.ATO.PAYEVNTEMP.000267

        #region VR.ATO.PAYEVNTEMP.000269

        /*  VR.ATO.PAYEVNTEMP.000269
        Contractor ABN cannot be the same as Payer Australian Business Number

        Technical Business Rule Format:
        ^PAYEVNTEMP91 <> NULL AND ^PAYEVNT2 <> NULL AND ^PAYEVNTEMP91 = ^PAYEVNT2
        */
        public static void VRATOPAYEVNTEMP000269(PAYEVNT2020 report, List<ProcessMessageDocument> response, PAYEVNTEMP2020 childReport)
        {
            if (childReport != null)
            {
                bool assertion = (!string.IsNullOrWhiteSpace(childReport.PAYEVNTEMP91) && !string.IsNullOrWhiteSpace(report.PAYEVNT2) && childReport.PAYEVNTEMP91 == report.PAYEVNT2);

                if (assertion)
                {
                    ProcessMessageDocument processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.PAYEVNTEMP.000269",
                        Severity = ProcessMessageSeverity.Error,
                        Description = @"Contractor ABN cannot be the same as Payer ABN",
                        Location = "/tns:PAYEVNTEMP/tns:Payee/tns:Identifiers/tns:AustralianBusinessNumberId",
                        Parameters = new ProcessMessageParameters() {
                            new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNTEMP.000269" },
                            new ProcessMessageParameter() { Name = "PAYEVNT2", Value = report.PAYEVNT2.ToString() },
                            new ProcessMessageParameter() { Name = "PAYEVNTEMP91", Value = childReport.PAYEVNTEMP91.ToString() }
                        }
                    };

                    response.Add(processMessage);
                }
            }
        }

        #endregion

        #region VR.ATO.PAYEVNTEMP.000271

        /*  VR.ATO.PAYEVNTEMP.000271
        Payee Commencement Date must be on or after 1/1/1950, or the default of 1/1/1800, when the Pay/Update Date is after 30/6/2020

        Technical Business Rule Format:
        (^PAYEVNT69 > 30/6/2020 AND (^PAYEVNTEMP92 <> 1/1/1800 AND ^PAYEVNTEMP92 < 1/1/1950))
        */
        public static void VRATOPAYEVNTEMP000271(PAYEVNT2020 report, List<ProcessMessageDocument> response, PAYEVNTEMP2020 childReport)
        {
            if (childReport != null)
            {
                bool assertion = report.PAYEVNT69.GetValueOrDefault() > DateTime.Parse("2020-06-30") && childReport.PAYEVNTEMP92.GetValueOrDefault() != DateTime.Parse("1800-01-01") && childReport.PAYEVNTEMP92.GetValueOrDefault() < DateTime.Parse("1950-01-01");

                if (assertion)
                {
                    ProcessMessageDocument processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.PAYEVNTEMP.000271",
                        Severity = ProcessMessageSeverity.Error,
                        Description = @"Payee Commencement Date must be after 1/1/1950 or the default of 1/1/1800",
                        LongDescription = @"Payee Commencement Date must be on or after 1/1/1950, or the default of 1/1/1800, when the Pay/Update Date is after 30/6/2020",
                        Location = "/tns:PAYEVNTEMP/tns:Payee/tns:EmployerConditions/tns:EmploymentStartD",
                        Parameters = new ProcessMessageParameters() {
                            new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNTEMP.000271" },
                            new ProcessMessageParameter() { Name = "PAYEVNT69", Value = report.PAYEVNT69.GetValueOrDefault().ToString("yyyy-MM-dd") },
                            new ProcessMessageParameter() { Name = "PAYEVNTEMP92", Value = childReport.PAYEVNTEMP92.GetValueOrDefault().ToString("yyyy-MM-dd") }
                        }
                    };

                    response.Add(processMessage);
                }
            }
        }

        #endregion

        #region VR.ATO.PAYEVNTEMP.000282

        /*  VR.ATO.PAYEVNTEMP.000282
        Payee Cessation Date cannot be before the Payee Commencement Date, when the Pay/Update Date is after 30/6/2020

        Technical Business Rule Format:
        (^PAYEVNTEMP93 <> NULL) AND (^PAYEVNT69 > 30/6/2020 AND ^PAYEVNTEMP92 > ^PAYEVNTEMP93)
        */
        public static void VRATOPAYEVNTEMP000282(PAYEVNT2020 report, List<ProcessMessageDocument> response, PAYEVNTEMP2020 childReport)
        {
            if (childReport != null)
            {
                bool assertion = childReport.PAYEVNTEMP93 != null && report.PAYEVNT69 > DateTime.Parse("2020-06-30") && childReport.PAYEVNTEMP92 > childReport.PAYEVNTEMP93;

                if (assertion)
                {
                    ProcessMessageDocument processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.PAYEVNTEMP.000282",
                        Severity = ProcessMessageSeverity.Error,
                        Description = @"Cessation date cannot be before the Commencement Date",
                        LongDescription = @"Payee Cessation Date cannot be before the Payee Commencement Date, when the Pay/Update Date is after 30/6/2020",
                        Location = "/tns:PAYEVNTEMP/tns:Payee/tns:EmployerConditions/tns:EmploymentEndD",
                        Parameters = new ProcessMessageParameters() {
                            new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNTEMP.000282" },
                            new ProcessMessageParameter() { Name = "PAYEVNT69", Value = report.PAYEVNT69.GetValueOrDefault().ToString("yyyy-MM-dd") },
                            new ProcessMessageParameter() { Name = "PAYEVNTEMP92", Value = childReport.PAYEVNTEMP92.GetValueOrDefault().ToString("yyyy-MM-dd") },
                            new ProcessMessageParameter() { Name = "PAYEVNTEMP93", Value = childReport.PAYEVNTEMP93.GetValueOrDefault().ToString("yyyy-MM-dd") }
                        }
                    };

                    response.Add(processMessage);
                }
            }
        }

        #endregion

        #region VR.ATO.PAYEVNTEMP.000286

        /*  VR.ATO.PAYEVNTEMP.000286
        When the Pay/Update Date is after 30/6/2020, Tax Treatment Code must be provided
    
        Technical Business Rule Format:
        (^PAYEVNTEMP254 = NULL AND ^PAYEVNT69 > 30/6/2020)
        */
        public static void VRATOPAYEVNTEMP000286(PAYEVNT2020 report, List<ProcessMessageDocument> response, PAYEVNTEMP2020 childReport)
        {
            if (childReport != null)
            {
                bool assertion = (childReport.PAYEVNTEMP254 == null && report.PAYEVNT69.GetValueOrDefault() > DateTime.Parse("2020-06-30"));
                if (assertion)
                {
                    ProcessMessageDocument processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.PAYEVNTEMP.000286",
                        Severity = ProcessMessageSeverity.Error,
                        Description = @"Tax Treatment Code must be provided",
                        LongDescription = @"When the Pay/Update Date is after 30/6/2020, Tax Treatment Code must be provided",
                        Location = "/tns:PAYEVNTEMP/tns:Payee/tns:EmployerConditions/tns:TaxTreatmentC",
                        Parameters = new ProcessMessageParameters() {
                            new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNTEMP.000286" },
                            new ProcessMessageParameter() { Name = "PAYEVNT69", Value = report.PAYEVNT69.GetValueOrDefault().ToString("yyyy-MM-dd") },
                            new ProcessMessageParameter() { Name = "PAYEVNTEMP254", Value = GetValueOrEmpty(childReport.PAYEVNTEMP254) }
                        }
                    };

                    response.Add(processMessage);
                }
            }
        }

        #endregion // VR.ATO.PAYEVNTEMP.000286

        #region VR.ATO.PAYEVNTEMP.000350

        /*  VR.ATO.PAYEVNTEMP.000350
        An Income Stream Type Code of; CHP, SWP, or IAA cannot be provided when the Pay/Update Date is before 1/7/2020

        Technical Business Rule Format:
        (RegexMatch(^PAYEVNTEMP257, 'CHP|SWP|IAA', 'i') AND ^PAYEVNT69 < 1/7/2020)
        */
        public static void VRATOPAYEVNTEMP000350(PAYEVNT2020 report, List<ProcessMessageDocument> response, PAYEVNTEMP2020 childReport)
        {
            if (childReport != null)
            {
                if (childReport.Payee_PayrollPeriod_RemunerationCollection != null)
                {
                    foreach (PAYEVNTEMP2020.Payee_PayrollPeriod_Remuneration remuneration in childReport.Payee_PayrollPeriod_RemunerationCollection)
                    {
                        bool assertion = RegexMatch(remuneration.PAYEVNTEMP257, "CHP|SWP|IAA", @"i") && report.PAYEVNT69 < DateTime.Parse("2020-07-01");

                        if (assertion)
                        {
                            ProcessMessageDocument processMessage = new ProcessMessageDocument()
                            {
                                Code = "CMN.ATO.PAYEVNTEMP.000350",
                                Severity = ProcessMessageSeverity.Error,
                                Description = @"CHP, SWP and IAA cannot be provided before 1/7/2020",
                                LongDescription = @"An Income Stream Type Code of; CHP, SWP, or IAA cannot be provided when the Pay/Update Date is before 1/7/2020",
                                Location = "/tns:PAYEVNTEMP/tns:Payee/tns:PayrollPeriod/tns:RemunerationCollection/tns:Remuneration" + OccurrenceIndex(remuneration.OccurrenceIndex) + "/tns:IncomeStreamTypeC",
                                Parameters = new ProcessMessageParameters() {
                                    new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNTEMP.000350" },
                                    new ProcessMessageParameter() { Name = "PAYEVNT69", Value = report.PAYEVNT69.GetValueOrDefault().ToString("yyyy-MM-dd") },
                                    new ProcessMessageParameter() { Name = "PAYEVNTEMP257", Value = remuneration.PAYEVNTEMP257 }
                                }
                            };

                            response.Add(processMessage);
                        }
                    }
                }
            }
        }

        #endregion

        #region VR.ATO.PAYEVNTEMP.000351

        /*  VR.ATO.PAYEVNTEMP.000351
        An Income Stream Type Code of JPD cannot be provided when the Pay/Update Date is after 30/6/2020

        Technical Business Rule Format:
        (RegexMatch(^PAYEVNTEMP257, 'JPD', 'i') AND ^PAYEVNT69 > 30/6/2020)
        */
        public static void VRATOPAYEVNTEMP000351(PAYEVNT2020 report, List<ProcessMessageDocument> response, PAYEVNTEMP2020 childReport)
        {
            if (childReport != null)
            {
                if (childReport.Payee_PayrollPeriod_RemunerationCollection != null)
                {
                    foreach (PAYEVNTEMP2020.Payee_PayrollPeriod_Remuneration remuneration in childReport.Payee_PayrollPeriod_RemunerationCollection)
                    {
                        bool assertion = RegexMatch(remuneration.PAYEVNTEMP257, "JPD", @"i") && report.PAYEVNT69 > DateTime.Parse("2020-06-30");

                        if (assertion)
                        {
                            ProcessMessageDocument processMessage = new ProcessMessageDocument()
                            {
                                Code = "CMN.ATO.PAYEVNTEMP.000351",
                                Severity = ProcessMessageSeverity.Error,
                                Description = @"JPD cannot be provided after 30/06/2020",
                                LongDescription = @"An Income Stream Type Code of JPD cannot be provided when the Pay/Update Date is after 30/6/2020",
                                Location = "/tns:PAYEVNTEMP/tns:Payee/tns:PayrollPeriod/tns:RemunerationCollection/tns:Remuneration" + OccurrenceIndex(remuneration.OccurrenceIndex) + "/tns:IncomeStreamTypeC",
                                Parameters = new ProcessMessageParameters() {
                                    new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNTEMP.000351" },
                                    new ProcessMessageParameter() { Name = "PAYEVNT69", Value = report.PAYEVNT69.GetValueOrDefault().ToString("yyyy-MM-dd") },
                                    new ProcessMessageParameter() { Name = "PAYEVNTEMP257", Value = remuneration.PAYEVNTEMP257 }
                                }
                            };

                            response.Add(processMessage);
                        }
                    }
                }
            }
        }

        #endregion

        #region VR.ATO.PAYEVNTEMP.000354

        /*  VR.ATO.PAYEVNTEMP.000354
        Only 10 Income Stream Type Codes with a value of IAA or FEI can be provided when the Pay/Update Date is after 30/06/2020

        Technical Business Rule Format:
        CountOccurrence(^PAYEVNTEMP256, RegexMatch(^PAYEVNTEMP257, 'FEI|IAA', 'i')) > 10 AND ^PAYEVNT69 > 30/6/2020
        */
        public static void VRATOPAYEVNTEMP000354(PAYEVNT2020 report, List<ProcessMessageDocument> response, PAYEVNTEMP2020 childReport)
        {
            if (childReport != null)
            {
                if (childReport.Payee_PayrollPeriod_RemunerationCollection != null)
                {
                    bool assertion = childReport.Payee_PayrollPeriod_RemunerationCollection.Count(remuneration => RegexMatch(remuneration.PAYEVNTEMP257, "FEI|IAA", @"i")) > 10 && report.PAYEVNT69 > DateTime.Parse("2020-06-30");

                    if (assertion)
                    {
                        ProcessMessageDocument processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.PAYEVNTEMP.000354",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Limit of 10 IAA and FEI codes reached",
                            LongDescription = @"Only 10 Income Stream Type Codes with a value of IAA or FEI can be provided when the Pay/Update Date is after 30/06/2020",
                            Location = "/tns:PAYEVNTEMP/tns:Payee/tns:PayrollPeriod/tns:RemunerationCollection/tns:Remuneration" + OccurrenceIndex(childReport.Payee_PayrollPeriod_RemunerationCollection.FindAll(remuneration => RegexMatch(remuneration.PAYEVNTEMP257, "FEI|IAA", @"i"))[10].OccurrenceIndex) + "/tns:IncomeStreamTypeC",
                            Parameters = new ProcessMessageParameters() {
                                new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNTEMP.000354" },
                                new ProcessMessageParameter() { Name = "PAYEVNT69", Value = report.PAYEVNT69.GetValueOrDefault().ToString("yyyy-MM-dd") },
                                new ProcessMessageParameter() { Name = "CountOccurrence(PAYEVNTEMP257 = FEI|IAA)", Value = childReport.Payee_PayrollPeriod_RemunerationCollection.Count(remuneration => RegexMatch(remuneration.PAYEVNTEMP257, "FEI|IAA", @"i")).ToString() }
                            }
                        };

                        response.Add(processMessage);
                    }
                }
            }
        }

        #endregion

        #region VR.ATO.PAYEVNTEMP.000355

        /*  VR.ATO.PAYEVNTEMP.000355
        Only 1 Income Stream Type Code with a value of FEI can be provided when the Pay/Update Date is before 1/7/2020

        Technical Business Rule Format:
        CountOccurrence(^PAYEVNTEMP256, RegexMatch(^PAYEVNTEMP257, 'FEI', 'i') > 1 AND ^PAYEVNT69 < 1/7/2020)
        */
        public static void VRATOPAYEVNTEMP000355(PAYEVNT2020 report, List<ProcessMessageDocument> response, PAYEVNTEMP2020 childReport)
        {
            if (childReport != null)
            {
                if (childReport.Payee_PayrollPeriod_RemunerationCollection != null)
                {
                    bool assertion = childReport.Payee_PayrollPeriod_RemunerationCollection.Count(remuneration => RegexMatch(remuneration.PAYEVNTEMP257, "FEI", @"i")) > 1 && report.PAYEVNT69 < DateTime.Parse("2020-07-01");

                    if (assertion)
                    {
                        ProcessMessageDocument processMessage = new ProcessMessageDocument()
                        {
                            Code = "CMN.ATO.PAYEVNTEMP.000355",
                            Severity = ProcessMessageSeverity.Error,
                            Description = @"Only one FEI income type can be provided before 1/7/2020",
                            LongDescription = @"Only 1 Income Stream Type Codes with a value of FEI can be provided when the Pay/Update Date is before 1/7/2020",
                            Location = "/tns:PAYEVNTEMP/tns:Payee/tns:PayrollPeriod/tns:RemunerationCollection/tns:Remuneration" + OccurrenceIndex(childReport.Payee_PayrollPeriod_RemunerationCollection.FindAll(remuneration => RegexMatch(remuneration.PAYEVNTEMP257, "FEI", @"i"))[1].OccurrenceIndex) + "/tns:IncomeStreamTypeC",
                            Parameters = new ProcessMessageParameters() {
                                new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNTEMP.000355" },
                                new ProcessMessageParameter() { Name = "PAYEVNT69", Value = report.PAYEVNT69.GetValueOrDefault().ToString("yyyy-MM-dd") },
                                new ProcessMessageParameter() { Name = "CountOccurrence(PAYEVNTEMP257 = FEI)", Value = childReport.Payee_PayrollPeriod_RemunerationCollection.Count(remuneration => RegexMatch(remuneration.PAYEVNTEMP257, "FEI", @"i")).ToString() }
                            }
                        };

                        response.Add(processMessage);
                    }
                }
            }
        }

        #endregion

        #region VR.ATO.PAYEVNTEMP.000358

        /*  VR.ATO.PAYEVNTEMP.000358
        When the Pay/Update Date is after 30/6/2020, Country Code must be provided when the Income Stream Type Code is one of the following: FEI, WHM, or IAA

        Technical Business Rule Format:
        (^PAYEVNT69 > 30/6/2020 AND ^PAYEVNTEMP260 = NULL AND RegexMatch(^PAYEVNTEMP257, 'FEI|WHM|IAA', 'i'))
        */
        public static void VRATOPAYEVNTEMP000358(PAYEVNT2020 report, List<ProcessMessageDocument> response, PAYEVNTEMP2020 childReport)
        {
            if (childReport != null)
            {
                if (childReport.Payee_PayrollPeriod_RemunerationCollection != null)
                {
                    foreach (PAYEVNTEMP2020.Payee_PayrollPeriod_Remuneration remuneration in childReport.Payee_PayrollPeriod_RemunerationCollection)
                    {
                        bool assertion = report.PAYEVNT69 > DateTime.Parse("2020-06-30") && remuneration.PAYEVNTEMP260 == null && RegexMatch(remuneration.PAYEVNTEMP257, "FEI|WHM|IAA", @"i");

                        if (assertion)
                        {
                            ProcessMessageDocument processMessage = new ProcessMessageDocument()
                            {
                                Code = "CMN.ATO.PAYEVNTEMP.000358",
                                Severity = ProcessMessageSeverity.Error,
                                Description = @"Country Code must be provided",
                                LongDescription = @"When the Pay/Update Date is after 30/6/2020, Country Code must be provided when the Income Stream Type Code is one of the following: FEI, WHM, or IAA",
                                Location = "/tns:PAYEVNTEMP/tns:Payee/tns:PayrollPeriod/tns:RemunerationCollection/tns:Remuneration" + OccurrenceIndex(remuneration.OccurrenceIndex) + "/tns:AddressDetailsCountryC",
                                Parameters = new ProcessMessageParameters() {
                                    new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNTEMP.000358" },
                                    new ProcessMessageParameter() { Name = "PAYEVNT69", Value = report.PAYEVNT69.GetValueOrDefault().ToString("yyyy-MM-dd") },
                                    new ProcessMessageParameter() { Name = "PAYEVNTEMP257", Value = remuneration.PAYEVNTEMP257 }
                                }
                            };

                            response.Add(processMessage);
                        }
                    }
                }
            }
        }

        #endregion

        #region VR.ATO.PAYEVNTEMP.000365

        /*  VR.ATO.PAYEVNTEMP.000365
        When Deduction Type Code is D, a Payer Withholding Payer Number cannot be provided

        Technical Business Rule Format:
        RegexMatch(^PAYEVNTEMP10, 'D', 'i') AND ^PAYEVNT3 <> NULL
        */
        public static void VRATOPAYEVNTEMP000365(PAYEVNT2020 report, List<ProcessMessageDocument> response, PAYEVNTEMP2020 childReport)
        {
            if (childReport != null)
            {
                if (childReport.Payee_PayrollPeriod_DeductionCollection != null)
                {
                    foreach (PAYEVNTEMP2020.Payee_PayrollPeriod_Deduction deduction in childReport.Payee_PayrollPeriod_DeductionCollection)
                    {
                        bool assertion = RegexMatch(deduction.PAYEVNTEMP10, "D", @"i") && report.PAYEVNT3 != null;

                        if (assertion)
                        {
                            ProcessMessageDocument processMessage = new ProcessMessageDocument()
                            {
                                Code = "CMN.ATO.PAYEVNTEMP.000365",
                                Severity = ProcessMessageSeverity.Error,
                                Description = @"When Deduction Type Code is D, a Payer Withholding Payer Number cannot be provided",
                                Location = "/tns:PAYEVNTEMP/tns:Payee/tns:PayrollPeriod/tns:DeductionCollection/tns:Deduction" + OccurrenceIndex(deduction.OccurrenceIndex) + "/tns:RemunerationTypeC",
                                Parameters = new ProcessMessageParameters() {
                                    new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNTEMP.000365" },
                                    new ProcessMessageParameter() { Name = "PAYEVNT3", Value = report.PAYEVNT3 },
                                    new ProcessMessageParameter() { Name = "PAYEVNTEMP10", Value = deduction.PAYEVNTEMP10 }
                                }
                            };

                            response.Add(processMessage);
                        }
                    }
                }
            }
        }

        #endregion

        #region VR.ATO.PAYEVNTEMP.000366

        /*  VR.ATO.PAYEVNTEMP.000366
            When the Pay/Update Date is before 1/10/2020, Deduction Type must not be D or G

            Legacy Rule Format:
                ^PAYEVNT69 < 1/10/2020 AND RegexMatch(^PAYEVNTEMP10, '[DG]', 'i')

            Technical Business Rule Format:
                    ^PAYEVNT69 < 1/10/2020 AND RegexMatch(^PAYEVNTEMP10, '[DG]', 'i')

            Data Elements:
    
            ^PAYEVNTEMP10 = PAYEVNTEMP:Payee:PayrollPeriod:Deduction:Remuneration.DeductionType.Code
    
            ^PAYEVNT69 = PAYEVNT69
        */
        public static void VRATOPAYEVNTEMP000366(PAYEVNT2020 report, List<ProcessMessageDocument> response, PAYEVNTEMP2020 childReport)
        {
            if (childReport != null)
            {
                if (childReport.Payee_PayrollPeriod_DeductionCollection != null)
                {
                    foreach (PAYEVNTEMP2020.Payee_PayrollPeriod_Deduction deduction in childReport.Payee_PayrollPeriod_DeductionCollection)
                    {
                        bool assertion = report.PAYEVNT69 < DateTime.Parse("2020-10-01") && RegexMatch(deduction.PAYEVNTEMP10, "[DG]", @"i");

                        if (assertion)
                        {
                            ProcessMessageDocument processMessage = new ProcessMessageDocument()
                            {
                                Code = "CMN.ATO.PAYEVNTEMP.000366",
                                Severity = ProcessMessageSeverity.Error,
                                Description = @"Child Support deductions not allowed prior to 1/10/2020",
                                LongDescription = @"When the Pay/Update Date is before 1/10/2020, Deduction Type must not be D or G",
                                Location = "/tns:PAYEVNTEMP/tns:Payee/tns:PayrollPeriod/tns:DeductionCollection/tns:Deduction" + OccurrenceIndex(deduction.OccurrenceIndex) + "/tns:RemunerationTypeC",
                                Parameters = new ProcessMessageParameters() {
                                    new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNTEMP.000366" },
                                    new ProcessMessageParameter() { Name = "PAYEVNT69", Value = report.PAYEVNT69.GetValueOrDefault().ToString("yy-MM-dd") },
                                    new ProcessMessageParameter() { Name = "PAYEVNTEMP10", Value = deduction.PAYEVNTEMP10 }
                                }
                            };

                            response.Add(processMessage);
                        }
                    }
                }
            }
        }

        #endregion

        #region VR.ATO.PAYEVNTEMP.000371

        /*  VR.ATO.PAYEVNTEMP.000371
        Lump Sum Type Code of W must not be provided when Pay/Update Date is before 1/7/2020

        Technical Business Rule Format:
        ^PAYEVNT69 < 1/7/2020 AND RegexMatch(^PAYEVNTEMP271, 'W', 'i')
        */
        public static void VRATOPAYEVNTEMP000371(PAYEVNT2020 report, List<ProcessMessageDocument> response, PAYEVNTEMP2020 childReport)
        {
            if (childReport != null)
            {
                if (childReport.Payee_PayrollPeriod_RemunerationCollection != null)
                {
                    foreach (PAYEVNTEMP2020.Payee_PayrollPeriod_Remuneration remuneration in childReport.Payee_PayrollPeriod_RemunerationCollection)
                    {
                        if (remuneration.Payee_PayrollPeriod_Remuneration_LumpSumCollection != null)
                        {
                            foreach (PAYEVNTEMP2020.Payee_PayrollPeriod_Remuneration.Payee_PayrollPeriod_Remuneration_LumpSum lumpSum in remuneration.Payee_PayrollPeriod_Remuneration_LumpSumCollection)
                            {
                                bool assertion = report.PAYEVNT69 < DateTime.Parse("2020-07-01") && RegexMatch(lumpSum.PAYEVNTEMP271, "W", @"i");

                                if (assertion)
                                {
                                    ProcessMessageDocument processMessage = new ProcessMessageDocument()
                                    {
                                        Code = "CMN.ATO.PAYEVNTEMP.000371",
                                        Severity = ProcessMessageSeverity.Error,
                                        Description = @"Lump Sum Type Code of W must not be provided",
                                        LongDescription = @"Lump Sum Type Code of W must not be provided when Pay/Update Date is before 1/7/2020",
                                        Location = "/tns:PAYEVNTEMP/tns:Payee/tns:PayrollPeriod/tns:RemunerationCollection/tns:Remuneration" + OccurrenceIndex(remuneration.OccurrenceIndex) + "/tns:LumpSumCollection/tns:LumpSum" + OccurrenceIndex(lumpSum.OccurrenceIndex) + "/tns:TypeC",
                                        Parameters = new ProcessMessageParameters() {
                                            new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNTEMP.000371" },
                                            new ProcessMessageParameter() { Name = "PAYEVNT69", Value = report.PAYEVNT69.GetValueOrDefault().ToString("yyyy-MM-dd") },
                                            new ProcessMessageParameter() { Name = "PAYEVNTEMP271", Value = "W" }
                                        }
                                    };

                                    response.Add(processMessage);
                                }
                            }
                        }
                    }
                }
            }
        }

        #endregion

        #region VR.ATO.PAYEVNTEMP.000372

        /*  VR.ATO.PAYEVNTEMP.000372
        Lump Sum Financial Year must be within 10 years of the financial year of the Pay/Update Date and prior to the current financial year

        Technical Business Rule Format:
        ^PAYEVNTEMP272 < (FinancialYear(^PAYEVNT69) -10) OR ^PAYEVNTEMP272 >= CurrentFinancialYear()
        */
        public static void VRATOPAYEVNTEMP000372(PAYEVNT2020 report, List<ProcessMessageDocument> response, PAYEVNTEMP2020 childReport)
        {
            if (childReport != null)
            {
                if (childReport.Payee_PayrollPeriod_RemunerationCollection != null)
                {
                    foreach (PAYEVNTEMP2020.Payee_PayrollPeriod_Remuneration remuneration in childReport.Payee_PayrollPeriod_RemunerationCollection)
                    {
                        if (remuneration.Payee_PayrollPeriod_Remuneration_LumpSumCollection != null)
                        {
                            foreach (PAYEVNTEMP2020.Payee_PayrollPeriod_Remuneration.Payee_PayrollPeriod_Remuneration_LumpSum lumpSum in remuneration.Payee_PayrollPeriod_Remuneration_LumpSumCollection)
                            {
                                bool assertion = lumpSum.PAYEVNTEMP272 < (FinancialYear(report.PAYEVNT69) - 10) || lumpSum.PAYEVNTEMP272 >= CurrentFinancialYear();

                                if (assertion)
                                {
                                    ProcessMessageDocument processMessage = new ProcessMessageDocument()
                                    {
                                        Code = "CMN.ATO.PAYEVNTEMP.000372",
                                        Severity = ProcessMessageSeverity.Error,
                                        Description = @"Lump Sum Financial Year is invalid",
                                        LongDescription = @"Lump Sum Financial Year must be within 10 years of the financial year of the Pay/Update Date and prior to the current financial year",
                                        Location = "/tns:PAYEVNTEMP/tns:Payee/tns:PayrollPeriod/tns:RemunerationCollection/tns:Remuneration" + OccurrenceIndex(remuneration.OccurrenceIndex) + "/tns:LumpSumCollection/tns:LumpSum" + OccurrenceIndex(lumpSum.OccurrenceIndex) + "/tns:FinancialY",
                                        Parameters = new ProcessMessageParameters() {
                                            new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNTEMP.000372" },
                                            new ProcessMessageParameter() { Name = "PAYEVNT69", Value = report.PAYEVNT69.GetValueOrDefault().ToString("yyyy-MM-dd") },
                                            new ProcessMessageParameter() { Name = "PAYEVNTEMP272", Value = lumpSum.PAYEVNTEMP272.GetValueOrDefault().ToString("yyyy-MM-dd") }
                                        }
                                    };

                                    response.Add(processMessage);
                                }
                            }
                        }
                    }
                }
            }
        }

        #endregion

        #region VR.ATO.PAYEVNTEMP.000378

        /*  VR.ATO.PAYEVNTEMP.000378
        Employment Basis Code must not be N when the Pay/Update Date is before 1/7/2020

        Technical Business Rule Format:
        ^PAYEVNT69 < 1/7/2020 AND RegexMatch(^PAYEVNTEMP282, 'N', 'i')
        */
        public static void VRATOPAYEVNTEMP000378(PAYEVNT2020 report, List<ProcessMessageDocument> response, PAYEVNTEMP2020 childReport)
        {
            if (childReport != null)
            {
                bool assertion = report.PAYEVNT69 < DateTime.Parse("2020-07-01") && RegexMatch(childReport.PAYEVNTEMP282, "N", @"i");

                if (assertion)
                {
                    ProcessMessageDocument processMessage = new ProcessMessageDocument()
                    {
                        Code = "CMN.ATO.PAYEVNTEMP.000378",
                        Severity = ProcessMessageSeverity.Error,
                        Description = @"Employment Basis Code must not be N",
                        LongDescription = @"Employment Basis Code must not be N when the Pay/Update Date is before 1/7/2020",
                        Location = "/tns:PAYEVNTEMP/tns:Payee/tns:EmployerConditions/tns:PaymentBasisC",
                        Parameters = new ProcessMessageParameters() {
                            new ProcessMessageParameter() { Name = "RuleIdentifier", Value = "VR.ATO.PAYEVNTEMP.000378" },
                            new ProcessMessageParameter() { Name = "PAYEVNT69", Value = report.PAYEVNT69.GetValueOrDefault().ToString("yyyy-MM-dd") },
                            new ProcessMessageParameter() { Name = "PAYEVNTEMP282", Value = GetValueOrEmpty(childReport.PAYEVNTEMP282) }
                        }
                    };

                    response.Add(processMessage);
                }
            }
        }

        #endregion

        #endregion
    }
}